/***************************************************************************

  psg.c


  Emulation of the AY-3-8910 sound chip.

  Based on various code snippets by Ville Hallik, Michael Cuddy,
  Tatsuyuki Satoh, Fabrice Frances, Nicola Salmoria.

***************************************************************************/

//#include "driver.h"
#include "psg.h"



#define AUDIO_CONV(A) (A)

#define MAX_OUTPUT 0xcfff

#define STEP 0x10000

struct AY8910
{
  unsigned char Regs[16];
  SAMPLE *Buf;      /* sound buffer */
  int bufp;       /* update buffer point */
  unsigned int UpdateStep;
  int PeriodA,PeriodB,PeriodC,PeriodN,PeriodE;
  int CountA,CountB,CountC,CountN,CountE;
  unsigned int VolA,VolB,VolC,VolE;
  unsigned char EnvelopeA,EnvelopeB,EnvelopeC;
  unsigned char OutputA,OutputB,OutputC,OutputN;
  signed char CountEnv;
  unsigned char Hold,Alternate,Attack,Holding;
  int RNG;
  unsigned int VolTable[32];
};

/* register id's */
#define AY_AFINE  (0)
#define AY_ACOARSE  (1)
#define AY_BFINE  (2)
#define AY_BCOARSE  (3)
#define AY_CFINE  (4)
#define AY_CCOARSE  (5)
#define AY_NOISEPER (6)
#define AY_ENABLE (7)
#define AY_AVOL   (8)
#define AY_BVOL   (9)
#define AY_CVOL   (10)
#define AY_EFINE  (11)
#define AY_ECOARSE  (12)
#define AY_ESHAPE (13)

#define AY_PORTA  (14)
#define AY_PORTB  (15)


/*
** some globals ...
*/
static int AYBufSize;   /* size of sound buffer, in samples */
static int AYNumChips;    /* total # of PSG's emulated */

static struct AY8910 AYPSG[MAX_PSG];    /* array of PSG's */


/*
** Initialize AY8910 emulator(s).
**
** 'num'      is the number of virtual AY8910's to allocate
** 'clock'    is master clock rate (Hz)
** 'rate'     is sampling rate and 'bufsiz' is the size of the
** buffer that should be updated at each interval
*/
int AYInit(int num, int clock, int rate, int bufsiz, SAMPLE **buffer )
{
  int i;


  if (num > MAX_PSG) return -1;

  AYNumChips = num;
  AYBufSize = bufsiz;

  /* init chip state */
  for ( i = 0 ; i < AYNumChips; i++ )
  {
    memset(&AYPSG[i],0,sizeof(struct AY8910));
    AYSetClock(i,clock,rate);
    AYPSG[i].Buf = buffer[i];
    AYSetGain(i,0x00);
    AYResetChip(i);
  }

  return 0;
}

void AYShutdown()
{
  AYBufSize = 0;
}

/*
** reset all chip registers.
*/
void AYResetChip(int num)
{
  int i;
  struct AY8910 *PSG = &AYPSG[num];


  PSG->RNG = 1;
  PSG->OutputA = 0;
  PSG->OutputB = 0;
  PSG->OutputC = 0;
  PSG->OutputN = 0xff;

  PSG->bufp = 0;
  for (i = 0;i < AY_PORTA;i++)
    AYWriteReg(num,i,0);
}

/* write a register on AY8910 chip number 'n' */
void AYWriteReg(int n, int r, int v)
{
  struct AY8910 *PSG = &AYPSG[n];
  int old;


  if (n >= AYNumChips)
  {
//if (errorlog) fprintf(errorlog,"error: write to 8910 #%d, allocated only %d\n",n,AYNumChips);
    return;
  }

  if (r > 15) return;

  PSG->Regs[r] = v;

  /* A note about the period of tones, noise and envelope: for speed reasons,*/
  /* we count down from the period to 0, but careful studies of the chip     */
  /* output prove that it instead counts up from 0 until the counter becomes */
  /* greater or equal to the period. This is an important difference when the*/
  /* program is rapidly changing the period to modulate the sound.           */
  /* To compensate for the difference, when the period is changed we adjust  */
  /* our internal counter.                                                   */
  /* Also, note that period = 0 is the same as period = 1. This is mentioned */
  /* in the YM2203 data sheets. However, this does NOT apply to the Envelope */
  /* period. In that case, period = 0 is half as period = 1. */
  switch( r )
  {
  case AY_AFINE:
  case AY_ACOARSE:
    PSG->Regs[AY_ACOARSE] &= 0x0f;
    old = PSG->PeriodA;
    PSG->PeriodA = (PSG->Regs[AY_AFINE] + 256 * PSG->Regs[AY_ACOARSE]) * PSG->UpdateStep;
    if (PSG->PeriodA == 0) PSG->PeriodA = PSG->UpdateStep;
    PSG->CountA += PSG->PeriodA - old;
    if (PSG->CountA <= 0) PSG->CountA = 1;
    break;
  case AY_BFINE:
  case AY_BCOARSE:
    PSG->Regs[AY_BCOARSE] &= 0x0f;
    old = PSG->PeriodB;
    PSG->PeriodB = (PSG->Regs[AY_BFINE] + 256 * PSG->Regs[AY_BCOARSE]) * PSG->UpdateStep;
    if (PSG->PeriodB == 0) PSG->PeriodB = PSG->UpdateStep;
    PSG->CountB += PSG->PeriodB - old;
    if (PSG->CountB <= 0) PSG->CountB = 1;
    break;
  case AY_CFINE:
  case AY_CCOARSE:
    PSG->Regs[AY_CCOARSE] &= 0x0f;
    old = PSG->PeriodC;
    PSG->PeriodC = (PSG->Regs[AY_CFINE] + 256 * PSG->Regs[AY_CCOARSE]) * PSG->UpdateStep;
    if (PSG->PeriodC == 0) PSG->PeriodC = PSG->UpdateStep;
    PSG->CountC += PSG->PeriodC - old;
    if (PSG->CountC <= 0) PSG->CountC = 1;
    break;
  case AY_NOISEPER:
    PSG->Regs[AY_NOISEPER] &= 0x1f;
    old = PSG->PeriodN;
    PSG->PeriodN = PSG->Regs[AY_NOISEPER] * PSG->UpdateStep;
    if (PSG->PeriodN == 0) PSG->PeriodN = PSG->UpdateStep;
    PSG->CountN += PSG->PeriodN - old;
    if (PSG->CountN <= 0) PSG->CountN = 1;
    break;
  case AY_AVOL:
    PSG->Regs[AY_AVOL] &= 0x1f;
    PSG->EnvelopeA = PSG->Regs[AY_AVOL] & 0x10;
    PSG->VolA = PSG->EnvelopeA ? PSG->VolE : PSG->VolTable[PSG->Regs[AY_AVOL] ? PSG->Regs[AY_AVOL]*2+1 : 0];
    break;
  case AY_BVOL:
    PSG->Regs[AY_BVOL] &= 0x1f;
    PSG->EnvelopeB = PSG->Regs[AY_BVOL] & 0x10;
    PSG->VolB = PSG->EnvelopeB ? PSG->VolE : PSG->VolTable[PSG->Regs[AY_BVOL] ? PSG->Regs[AY_BVOL]*2+1 : 0];
    break;
  case AY_CVOL:
    PSG->Regs[AY_CVOL] &= 0x1f;
    PSG->EnvelopeC = PSG->Regs[AY_CVOL] & 0x10;
    PSG->VolC = PSG->EnvelopeC ? PSG->VolE : PSG->VolTable[PSG->Regs[AY_CVOL] ? PSG->Regs[AY_CVOL]*2+1 : 0];
    break;
  case AY_EFINE:
  case AY_ECOARSE:
    old = PSG->PeriodE;
    PSG->PeriodE = ((PSG->Regs[AY_EFINE] + 256 * PSG->Regs[AY_ECOARSE])) * PSG->UpdateStep;
    if (PSG->PeriodE == 0) PSG->PeriodE = PSG->UpdateStep / 2;
    PSG->CountE += PSG->PeriodE - old;
    if (PSG->CountE <= 0) PSG->CountE = 1;
    break;
  case AY_ESHAPE:
    /* envelope shapes:
    C AtAlH
    0 0 x x  \___

    0 1 x x  /___

    1 0 0 0  \\\\

    1 0 0 1  \___

    1 0 1 0  \/\/
              ___
    1 0 1 1  \

    1 1 0 0  ////
              ___
    1 1 0 1  /

    1 1 1 0  /\/\

    1 1 1 1  /___

    The envelope counter on the AY-3-8910 has 16 steps. On the YM2203 it
    has twice the steps, happening twice as fast. Since the end result is
    just a smoother curve, we always use the YM2203 behaviour.
    */
    PSG->Regs[AY_ESHAPE] &= 0x0f;
    PSG->Attack = (PSG->Regs[AY_ESHAPE] & 0x04) ? 0x1f : 0x00;
    if ((PSG->Regs[AY_ESHAPE] & 0x08) == 0)
    {
      /* if Continue = 0, map the shape to the equivalent one which has Continue = 1 */
      PSG->Hold = 1;
      PSG->Alternate = PSG->Attack;
    }
    else
    {
      PSG->Hold = PSG->Regs[AY_ESHAPE] & 0x01;
      PSG->Alternate = PSG->Regs[AY_ESHAPE] & 0x02;
    }
    PSG->CountE = PSG->PeriodE;
    PSG->CountEnv = 0x1f;
    PSG->Holding = 0;
    PSG->VolE = PSG->VolTable[PSG->CountEnv ^ PSG->Attack];
    if (PSG->EnvelopeA) PSG->VolA = PSG->VolE;
    if (PSG->EnvelopeB) PSG->VolB = PSG->VolE;
    if (PSG->EnvelopeC) PSG->VolC = PSG->VolE;
    break;
  case AY_PORTA:
    if ((PSG->Regs[AY_ENABLE] & 0x40) == 0)
//if (errorlog) fprintf(errorlog,"warning: write to 8910 #%d Port A set as input\n",n);
//    AYPortHandler(n,0,1,v);
    break;
  case AY_PORTB:
    if ((PSG->Regs[AY_ENABLE] & 0x80) == 0)
//if (errorlog) fprintf(errorlog,"warning: write to 8910 #%d Port B set as input\n",n);
//    AYPortHandler(n,1,1,v);
    break;
  }
}



unsigned char AYReadReg(int n, int r)
{
  struct AY8910 *PSG = &AYPSG[n];


  if (n >= AYNumChips)
  {
//if (errorlog) fprintf(errorlog,"error: read from 8910 #%d, allocated only %d\n",n,AYNumChips);
    return 0;
  }

  if (r > 15) return 0;

  switch (r)
  {
  case AY_PORTA:
    if ((PSG->Regs[AY_ENABLE] & 0x40) != 0)
//if (errorlog) fprintf(errorlog,"warning: read from 8910 #%d Port A set as output\n",n);
//    PSG->Regs[AY_PORTA] = AYPortHandler(n,0,0,0);
    break;
  case AY_PORTB:
    if ((PSG->Regs[AY_ENABLE] & 0x80) != 0)
//if (errorlog) fprintf(errorlog,"warning: read from 8910 #%d Port B set as output\n",n);
//    PSG->Regs[AY_PORTB] = AYPortHandler(n,1,0,0);
    break;
  }
  return PSG->Regs[r];
}



void AYUpdateOne(int chip,int endp)
{
  struct AY8910 *PSG = &AYPSG[chip];
  SAMPLE *buffer = &PSG->Buf[PSG->bufp];
  int length;
  int outn;


  if( endp > AYBufSize ) endp = AYBufSize;
  length = endp - PSG->bufp;

  /* The 8910 has three outputs, each output is the mix of one of the three */
  /* tone generators and of the (single) noise generator. The two are mixed */
  /* BEFORE going into the DAC. The formula to mix each channel is this one: */
  /* (ToneOn | ToneDisable) & (NoiseOn | NoiseDisable). */
  /* Note that his means that if both tone and noise are disabled, the output */
  /* is 1, not 0, and can be modulated changing the volume. */


  /* if the channels are disabled, set their output to 1, and set */
  /* the counter so they will not be inverted during this update. */
  if (PSG->Regs[AY_ENABLE] & 0x01)
  {
    PSG->CountA = length*STEP + 1;
    PSG->OutputA = 1;
  }
  if (PSG->Regs[AY_ENABLE] & 0x02)
  {
    PSG->CountB = length*STEP + 1;
    PSG->OutputB = 1;
  }
  if (PSG->Regs[AY_ENABLE] & 0x04)
  {
    PSG->CountC = length*STEP + 1;
    PSG->OutputC = 1;
  }
  /* for the noise channel we must not touch OutputN - it's also not necessary */
  /* since we use outn. */
  if ((PSG->Regs[AY_ENABLE] & 0x38) == 0x38)  /* all off */
    PSG->CountN = length*STEP + 1;
  outn = (PSG->OutputN | PSG->Regs[AY_ENABLE]);


  /* buffering loop */
  while (length)
  {
    int vola,volb,volc;
    int left;


    /* vola, volb and volc keep track of how long each square wave stays */
    /* in the 1 position during the sample period. */
    vola = volb = volc = 0;

    left = STEP;
    do
    {
      int nextevent;


      if (PSG->CountN < left) nextevent = PSG->CountN;
      else nextevent = left;

      if (outn & 0x08)
      {
        if (PSG->OutputA) vola += PSG->CountA;
        PSG->CountA -= nextevent;
        /* PeriodA is the half period of the square wave. Here, in each */
        /* loop I add PeriodA twice, so that at the end of the loop the */
        /* square wave is in the same status (0 or 1) it was at the start. */
        /* vola is also incremented by PeriodA, since the wave has been 1 */
        /* exactly half of the time, regardless of the initial position. */
        /* If we exit the loop in the middle, OutputA has to be inverted */
        /* and vola incremented only if the exit status of the square */
        /* wave is 1. */
        while (PSG->CountA <= 0)
        {
          PSG->CountA += PSG->PeriodA;
          if (PSG->CountA > 0)
          {
            PSG->OutputA ^= 1;
            if (PSG->OutputA) vola += PSG->PeriodA;
            break;
          }
          PSG->CountA += PSG->PeriodA;
          vola += PSG->PeriodA;
        }
        if (PSG->OutputA) vola -= PSG->CountA;
      }
      else
      {
        PSG->CountA -= nextevent;
        while (PSG->CountA <= 0)
        {
          PSG->CountA += PSG->PeriodA;
          if (PSG->CountA > 0)
          {
            PSG->OutputA ^= 1;
            break;
          }
          PSG->CountA += PSG->PeriodA;
        }
      }

      if (outn & 0x10)
      {
        if (PSG->OutputB) volb += PSG->CountB;
        PSG->CountB -= nextevent;
        while (PSG->CountB <= 0)
        {
          PSG->CountB += PSG->PeriodB;
          if (PSG->CountB > 0)
          {
            PSG->OutputB ^= 1;
            if (PSG->OutputB) volb += PSG->PeriodB;
            break;
          }
          PSG->CountB += PSG->PeriodB;
          volb += PSG->PeriodB;
        }
        if (PSG->OutputB) volb -= PSG->CountB;
      }
      else
      {
        PSG->CountB -= nextevent;
        while (PSG->CountB <= 0)
        {
          PSG->CountB += PSG->PeriodB;
          if (PSG->CountB > 0)
          {
            PSG->OutputB ^= 1;
            break;
          }
          PSG->CountB += PSG->PeriodB;
        }
      }

      if (outn & 0x20)
      {
        if (PSG->OutputC) volc += PSG->CountC;
        PSG->CountC -= nextevent;
        while (PSG->CountC <= 0)
        {
          PSG->CountC += PSG->PeriodC;
          if (PSG->CountC > 0)
          {
            PSG->OutputC ^= 1;
            if (PSG->OutputC) volc += PSG->PeriodC;
            break;
          }
          PSG->CountC += PSG->PeriodC;
          volc += PSG->PeriodC;
        }
        if (PSG->OutputC) volc -= PSG->CountC;
      }
      else
      {
        PSG->CountC -= nextevent;
        while (PSG->CountC <= 0)
        {
          PSG->CountC += PSG->PeriodC;
          if (PSG->CountC > 0)
          {
            PSG->OutputC ^= 1;
            break;
          }
          PSG->CountC += PSG->PeriodC;
        }
      }

      PSG->CountN -= nextevent;
      if (PSG->CountN <= 0)
      {
        /* Is noise output going to change? */
        if ((PSG->RNG + 1) & 2) /* (bit0^bit1)? */
        {
          PSG->OutputN = ~PSG->OutputN;
          outn = (PSG->OutputN | PSG->Regs[AY_ENABLE]);
        }

        /* The Random Number Generator of the 8910 is a 17-bit shift */
        /* register. The input to the shift register is bit0 XOR bit2 */
        /* (bit0 is the output). */

        /* The following is a fast way to compute bit 17 = bit0^bit2. */
        /* Instead of doing all the logic operations, we only check */
        /* bit 0, relying on the fact that after two shifts of the */
        /* register, what now is bit 2 will become bit 0, and will */
        /* invert, if necessary, bit 16, which previously was bit 18. */
        if (PSG->RNG & 1) PSG->RNG ^= 0x28000;
        PSG->RNG >>= 1;
        PSG->CountN += PSG->PeriodN;
      }

      left -= nextevent;
    } while (left > 0);

    /* update envelope */
    if (PSG->Holding == 0)
    {
      PSG->CountE -= STEP;
      if (PSG->CountE <= 0)
      {
        do
        {
          PSG->CountEnv--;
          if (PSG->CountEnv < 0)
          {
            PSG->CountEnv = 0x1f;

            if (PSG->Alternate)
              PSG->Attack ^= 0x1f;

            if (PSG->Hold != 0)
            {
              PSG->Holding = 1;
              PSG->CountEnv = 0;
            }
          }

          PSG->VolE = PSG->VolTable[PSG->CountEnv ^ PSG->Attack];
          if (PSG->EnvelopeA) PSG->VolA = PSG->VolE;
          if (PSG->EnvelopeB) PSG->VolB = PSG->VolE;
          if (PSG->EnvelopeC) PSG->VolC = PSG->VolE;

          PSG->CountE += PSG->PeriodE;
        } while (PSG->CountE <= 0 && PSG->Holding == 0);
      }
    }

    vola *= PSG->VolA;
    volb *= PSG->VolB;
    volc *= PSG->VolC;

#ifdef SAMPLE_16BIT
    *buffer = (vola + volb + volc) / STEP;
#else
//    *buffer = AUDIO_CONV((vola + volb + volc) / (STEP * 256));
    {
      int sam=(vola + volb + volc) / (STEP * 128);
      if ( sam > 127 )
        *buffer = 127;
      else if ( sam < -128 )
        *buffer = -128;
      else *buffer = sam;
    }
#endif

    buffer++;
    length--;
  }
//  PSG->bufp  = endp;
}



/*
** called to update all chips
*/
/* ASG 971010 -- modified to pass arbitrary buffer length and buffer pointers */
void AYUpdate(void)
{
  int i;


  for (i = 0;i < AYNumChips;i++)
  {
    AYUpdateOne( i , AYBufSize );
    AYPSG[i].bufp = 0;
  }
}



void AYSetClock(int n,int clock,int rate)
{
  /* the step clock for the tone and noise generators is the chip clock */
  /* divided by 8; for the envelope generator of the AY-3-8910, it is half */
  /* that much (clock/16), but the envelope of the YM2203 goes twice as */
  /* fast, therefore again clock/8. */
  /* Here we calculate the number of steps which happen during one sample */
  /* at the given sample rate. No. of events = sample rate / (clock/8). */
  /* STEP is a multiplier used to turn the fraction into a fixed point */
  /* number. */
  AYPSG[n].UpdateStep = ((double)STEP * rate * 8) / clock;
};



/*
** set output gain
**
** The gain is expressed in 0.2dB increments, e.g. a gain of 10 is an increase
** of 2dB. Note that the gain aonly affects sounds not playing at full volume,
** since the ones at full volume are already played at the maximum intensity
** allowed by the sound card.
** 0x00 is the default.
** 0xff is the maximum allowed value.
*/
void AYSetGain(int n,int gain)
{
  struct AY8910 *PSG = &AYPSG[n];
  int i;
  double out;


  gain &= 0xff;

  /* increase max output basing on gain (0.2 dB per step) */
  out = MAX_OUTPUT/3;
  while (gain-- > 0)
    out *= 1.023292992; /* = (10 ^ (0.2/20)) */

  /* calculate the volume->voltage conversion table */
  /* The AY-3-8910 has 16 levels, in a logarithmic scale (3dB per step) */
  /* The YM2203 still has 16 levels for the tone generators, but 32 for */
  /* the envelope generator (1.5dB per step). */
  for (i = 31;i > 0;i--)
  {
    /* limit volume to avoid clipping */
    if (out > MAX_OUTPUT/3) PSG->VolTable[i] = MAX_OUTPUT/3;
    else PSG->VolTable[i] = out;

    out /= 1.188502227; /* = 10 ^ (1.5/20) */
  }
  PSG->VolTable[0] = 0;
}
