
/************************************************\
***                                            ***
***                 STARSCREAM                 ***
***             680x0 Code Builder             ***
***                                            ***
***          Written by Neill Corlett          ***
***                                            ***
\************************************************/

#define VERSION "0.12 (8/2/97)"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* NOTE: Ignore all comments in this file.  (except this one ;)  They may be
** outdated, or they may even contradict each other.
*/

/*
** EAX: Primary value (destination)
** EBX: CPU_PCPTR
** ECX: Secondary value (source)
** EDX: Temp. register
** EBP: Current address "bank"
** ESI: Current address
** EDI: Current reg. number
**
** Within a read routine:
** - EBX, EDX, EBP, ESI, EDI are preserved.
** - ESI is the address, EAX will hold the read value.
** - EAX is only guaranteed up to the instruction size:
**   when using 8-bit instructions, rely ONLY on AL (not AX or EAX)
**   when using 16-bit instructions, rely ONLY on AX (not EAX)
** - Unused bits of EAX may be destroyed at will.
*/

typedef unsigned char byte;
typedef unsigned short int word;
typedef unsigned int dword;

static char sizes[5]={'?','b','w','?','l'};
static char*sizeinc[5]={"?","incl ","addl $2,","?","addl $4,"};
static char*sizedec[5]={"?","decl ","subl $2,","?","subl $4,"};
static char*x86ax[5]={"?","%al","%ax","?","%eax"};
static char*x86cx[5]={"?","%cl","%cx","?","%ecx"};
static char*x86dx[5]={"?","%dl","%dx","?","%edx"};
static int quickvalue[8]={8,1,2,3,4,5,6,7};
static char direction[2]={'r','l'};

static FILE*asf;
static int linenum;
static int apu2x;

static char sysname[100];
static int bankbits;

typedef enum eam{dreg,areg,aind,ainc,adec,adsp,axdp,absw,absl,pcdp,pcxd,immd} eamode;

static eamode main_eamode,main_destmode;
static int main_size,sizedef,main_reg;
static int main_cc,main_dr,main_ir,main_qv;

static void banner(void){
	fprintf(asf,
		"# Generated by STARSCREAM Version "VERSION"\n#\n"
		"# Machine name: %s\n# Bank bits: %d\n#\n"
		".file \"starcpu.s\"\n",sysname,bankbits
	);
}

static void gen_variables(void){
	fprintf(asf,
		".data\n"
		".align 4\n"
		".globl _cpu_r\n"
		".globl _cpu_pc\n"
		".globl _cpu_asp\n"
		".globl _cpu_tmpf\n"
		".globl _cpu_intmask\n"
		".globl _cpu_odometer\n"
		".globl _cpu_pcptrstart\n"
		".globl APUPTR\n"
		"_cpu_r:\n"
		".long 0\n"
		".long 0\n"
		".long 0\n"
		".long 0\n"
		".long 0\n"
		".long 0\n"
		".long 0\n"
		".long 0\n"
		".long 0\n"
		".long 0\n"
		".long 0\n"
		".long 0\n"
		".long 0\n"
		".long 0\n"
		".long 0\n"
		".long 0\n"
		"_cpu_tmpf:\n"
		".long 0\n"
		/*
		** Format of cpu_tmpf:
		** Bit  0:  Carry
		** Bit  6:  Zero
		** Bit  7:  Negative
		** Bit 11:  Overflow
		** Bit 16:  "X"
		** Bit 29:  Supervisor mode
		*/
		"_cpu_intmask:\n"
		".long 0\n"
		"_cpu_pc:\n"
		".long 0\n"
		"_cpu_instcount:\n"
		".long 0\n"
		"_cpu_pcptrstart:\n"
		".long 0\n"
		"_cpu_odometer:\n"
		".long 0\n"
		"_cpu_intf:\n"
		".long 0\n"
		"_cpu_stopped:\n"
		".long 0\n"
		"_cpu_asp:\n"
		".long 0\n"
		"_cpu_live:\n"
		".long 0\n"
		".globl _%s_ptr\n"
		".globl _%s_rpb\n"
		".globl _%s_rpw\n"
		".globl _%s_rpl\n"
		".globl _%s_wpb\n"
		".globl _%s_wpw\n"
                ".globl _%s_wpl\n"
                ".globl _cpu_checker\n"
                "_cpu_checker: .long 0\n",
		sysname,sysname,sysname,sysname,sysname,sysname,sysname
	);
}

/* Handy routine to decode brief extension words */
static void gen_decode_bx(void){
	fprintf(asf,".align 4\n");
	fprintf(asf,"decode_bx:\n");
	fprintf(asf,"movzwl (%%ebx),%%edx\n");
	fprintf(asf,"movsbl %%dl,%%esi\n");
	fprintf(asf,"addl $2,%%ebx\n");
	fprintf(asf,"shrl $12,%%edx\n");
	fprintf(asf,"movl _cpu_r(,%%edx,4),%%edx\n");
	fprintf(asf,"jc decode_bx_long\n");
	fprintf(asf,"movswl %%dx,%%edx\n");
	fprintf(asf,"decode_bx_long:\n");
	fprintf(asf,"addl %%edx,%%esi\n");
	fprintf(asf,"ret\n");
}

static void gen_unimplemented(void){
	fprintf(asf,".text\n");
	fprintf(asf,".align 4\n");
	fprintf(asf,"A:\n");
	fprintf(asf,"subl _cpu_pcptrstart,%%ebx\n");
	fprintf(asf,"subl $2,%%ebx\n");
	fprintf(asf,"addl %%ebx,_cpu_pc\n");
	fprintf(asf,"movl %%edi,_cpu_odometer\n");
	fprintf(asf,"call _%s_uni\n",sysname);
	fprintf(asf,"ret\n");
}

static void prep_address(void){
	fprintf(asf,"movl %%esi,%%ebp\n");
	fprintf(asf,"shrl $%d,%%ebp\n",bankbits);
	/* Cut address off to 24 bits. */
/*	fprintf(asf,"andl $%d,%%ebp\n",0xFFFFFF>>bankbits);*/
}

static void newpc(char*newpcsrc){
	fprintf(asf,"movl %s,%%ebx\n",newpcsrc);
	if(strcmp(newpcsrc,"_cpu_pc")){
		fprintf(asf,"movl %s,_cpu_pc\n",newpcsrc);
	}
	fprintf(asf,"movl %%ebx,%%ebp\n");
	fprintf(asf,"shrl $%d,%%ebp\n",bankbits);
	fprintf(asf,"andl $0x%X,%%ebx\n",(1<<bankbits)-1);
	fprintf(asf,"addl _%s_ptr(,%%ebp,4),%%ebx\n",sysname);
	fprintf(asf,"movl %%ebx,_cpu_pcptrstart\n");
}

static void al2ccr(void){
	fprintf(asf,
		"movb %%al,%%ah\n"
		"shlb $4,%%ah\n"
		"andw $0xC01F,%%ax\n"
		"orb %%ah,%%al\n"
		"movb %%al,_cpu_tmpf\n"
		"shlb $2,%%al\n"
		"movb %%al,_cpu_tmpf+1\n"
		"shrb $6,%%al\n"
		"movb %%al,_cpu_tmpf+2\n"
	);
}

static void ax2sr(void){
	int myline;
	myline=linenum;
	linenum++;
	fprintf(asf,"pushl %%eax\n");
	fprintf(asf,"andb $7,%%ah\n");
	fprintf(asf,"movb %%ah,_cpu_intmask\n");
	fprintf(asf,"popl %%eax\n");
	fprintf(asf,"andb $0x20,%%ah\n");
	fprintf(asf,"pushl %%eax\n");
	fprintf(asf,"xorb _cpu_tmpf+3,%%ah\n");
	fprintf(asf,"jz l%d\n",myline);
	fprintf(asf,"movl _cpu_r+%d,%%eax\n",32+28);
	fprintf(asf,"xchgl _cpu_asp,%%eax\n");
	fprintf(asf,"movl %%eax,_cpu_r+%d\n",32+28);
	fprintf(asf,"l%d:\n",myline);
	fprintf(asf,"popl %%eax\n");
	fprintf(asf,"movb %%ah,_cpu_tmpf+3\n");
	al2ccr();
}

static void ccr2al(void){
	fprintf(asf,
		"movb _cpu_tmpf,%%al\n"
		"movb %%al,%%ah\n"
		"shrb $5,%%al\n"
		"shrb $1,%%ah\n"
		"rclb $1,%%al\n"
		"andb $0x0D,%%al\n"
		"movb _cpu_tmpf+1,%%ah\n"
		"shrb $2,%%ah\n"
		"andb $2,%%ah\n"
		"orb %%ah,%%al\n"
		"shlb $4,%%al\n"
		"movb _cpu_tmpf+2,%%ah\n"
		"rcrb $1,%%ah\n"
		"rcrb $4,%%al\n"
	);
}

static void sr2ax(void){
	ccr2al();
	fprintf(asf,
		"movb _cpu_intmask,%%ah\n"
		"orb _cpu_tmpf+3,%%ah\n"
	);
}

static void suffixes(void){
	fprintf(asf,".text\n");
	fprintf(asf,".globl _cpu_readb\n");
	fprintf(asf,".globl _cpu_readw\n");
	fprintf(asf,".globl _cpu_readl\n");
	fprintf(asf,".globl _cpu_writeb\n");
	fprintf(asf,".globl _cpu_writew\n");
	fprintf(asf,".globl _cpu_writel\n");
	fprintf(asf,".align 4\n_cpu_readb:\npushl %%ebp\npushl %%esi\nmovl 12(%%esp),%%esi\n");prep_address();fprintf(asf,"call *_%s_rpb(,%%ebp,4)\npopl %%esi\npopl %%ebp\nret\n",sysname);
	fprintf(asf,".align 4\n_cpu_readw:\npushl %%ebp\npushl %%esi\nmovl 12(%%esp),%%esi\n");prep_address();fprintf(asf,"call *_%s_rpw(,%%ebp,4)\npopl %%esi\npopl %%ebp\nret\n",sysname);
	fprintf(asf,".align 4\n_cpu_readl:\npushl %%ebp\npushl %%esi\nmovl 12(%%esp),%%esi\n");prep_address();fprintf(asf,"call *_%s_rpl(,%%ebp,4)\npopl %%esi\npopl %%ebp\nret\n",sysname);
	fprintf(asf,".align 4\n_cpu_writeb:\npushl %%ebp\npushl %%esi\npushl %%eax\nmovl 16(%%esp),%%esi\nmovl 20(%%esp),%%eax\n");prep_address();fprintf(asf,"call *_%s_wpb(,%%ebp,4)\npopl %%eax\npopl %%esi\npopl %%ebp\nret\n",sysname);
	fprintf(asf,".align 4\n_cpu_writew:\npushl %%ebp\npushl %%esi\npushl %%eax\nmovl 16(%%esp),%%esi\nmovl 20(%%esp),%%eax\n");prep_address();fprintf(asf,"call *_%s_wpw(,%%ebp,4)\npopl %%eax\npopl %%esi\npopl %%ebp\nret\n",sysname);
	fprintf(asf,".align 4\n_cpu_writel:\npushl %%ebp\npushl %%esi\npushl %%eax\nmovl 16(%%esp),%%esi\nmovl 20(%%esp),%%eax\n");prep_address();fprintf(asf,"call *_%s_wpl(,%%ebp,4)\npopl %%eax\npopl %%esi\npopl %%ebp\nret\n",sysname);
	fprintf(asf,".globl _cpu_init\n");
	fprintf(asf,".align 4\n_cpu_init:\npushal\n");
	fprintf(asf,"leal A,%%ebx\n");
	fprintf(asf,"leal SPTR,%%edi\n");
	fprintf(asf,"leal SPCK,%%esi\n");
	fprintf(asf,"leal 262144(%%edi),%%ebp\n");
	fprintf(asf,"_cpu_init_0:\n");
	fprintf(asf,"lodsl\n");
	fprintf(asf,"movl %%eax,%%ecx\n");
	fprintf(asf,"andl $0xFFFFFF,%%eax\n");
	fprintf(asf,"shrl $24,%%ecx\n");
	fprintf(asf,"addl %%ebx,%%eax\n");
	fprintf(asf,"incl %%ecx\n");
	fprintf(asf,"repe\nstosl\n");
	fprintf(asf,"cmpl %%ebp,%%edi\n");
	fprintf(asf,"jne _cpu_init_0\n");
	fprintf(asf,"popal\nret\n");
	fprintf(asf,".globl _cpu_reset\n");
	fprintf(asf,".align 4\n_cpu_reset:\npushl %%eax\npushl %%esi\n");
	fprintf(asf,"xorl %%eax,%%eax\n");
	fprintf(asf,"movl %%eax,_cpu_odometer\n");
	fprintf(asf,"movl %%eax,_cpu_intmask\n");
	fprintf(asf,"movl $0x20000000,_cpu_tmpf\n");
	fprintf(asf,"movl %%eax,%%esi\n");
	fprintf(asf,"call *_%s_rpl\n",sysname);
	fprintf(asf,"movl %%eax,_cpu_r+32+28\n");
	fprintf(asf,"movl %%eax,_cpu_asp\n");
	fprintf(asf,"addl $4,%%esi\n");
	fprintf(asf,"call *_%s_rpl\n",sysname);
	fprintf(asf,"movl %%eax,_cpu_pc\n");
	fprintf(asf,"popl %%esi\npopl %%eax\nret\n");
	fprintf(asf,".globl _cpu_dispatch\n");
	fprintf(asf,".align 4\n_cpu_dispatch:\n");
	fprintf(asf,"movb $1,_cpu_live\n");
	fprintf(asf,"pushl %%eax\n");
	fprintf(asf,"pushl %%ebx\n");
	fprintf(asf,"pushl %%ecx\n");
	fprintf(asf,"pushl %%edx\n");
	fprintf(asf,"pushl %%ebp\n");
	fprintf(asf,"pushl %%esi\n");
	fprintf(asf,"pushl %%edi\n");
	fprintf(asf,"movl 32(%%esp),%%eax\n");
	fprintf(asf,"movl %%eax,_cpu_instcount\n");
	fprintf(asf,"addl %%eax,_cpu_odometer\n");
	newpc("_cpu_pc");
	fprintf(asf,"xorl %%edi,%%edi\n");
	fprintf(asf,"shrl $1,_cpu_instcount\n");
	fprintf(asf,"jnc _cpu_dispatch_0\n");
	fprintf(asf,"incl _cpu_instcount\n");
	fprintf(asf,"jmp _cpu_dispatch_1\n");
	fprintf(asf,"_cpu_dispatch_0:\n");
	fprintf(asf,"movw (%%ebx),%%di\naddl $2,%%ebx\ncall *SPTR(,%%edi,4)\n");
	fprintf(asf,"_cpu_dispatch_1:\n");
	fprintf(asf,"movw (%%ebx),%%di\naddl $2,%%ebx\ncall *SPTR(,%%edi,4)\n");
	fprintf(asf,"decl _cpu_instcount\n");
	fprintf(asf,"jnz _cpu_dispatch_0\n");
	fprintf(asf,"subl _cpu_pcptrstart,%%ebx\n");
	fprintf(asf,"addl %%ebx,_cpu_pc\n");
	fprintf(asf,"popl %%edi\n");
	fprintf(asf,"popl %%esi\n");
	fprintf(asf,"popl %%ebp\n");
	fprintf(asf,"popl %%edx\n");
	fprintf(asf,"popl %%ecx\n");
	fprintf(asf,"popl %%ebx\n");
	fprintf(asf,"popl %%eax\n");
	fprintf(asf,"movb $0,_cpu_live\n");
	fprintf(asf,"ret\n");
	if(apu2x){
	fprintf(asf,".globl _cpu_timeslice\n");
	fprintf(asf,".align 4\n_cpu_timeslice:\n");
	fprintf(asf,"movb $1,_cpu_live\n");
	fprintf(asf,"pushl %%eax\n");
	fprintf(asf,"pushl %%ebx\n");
	fprintf(asf,"pushl %%ecx\n");
	fprintf(asf,"pushl %%edx\n");
	fprintf(asf,"pushl %%ebp\n");
	fprintf(asf,"pushl %%esi\n");
	fprintf(asf,"pushl %%edi\n");
	fprintf(asf,"movl 32(%%esp),%%eax\n");
	fprintf(asf,"movl %%eax,_cpu_instcount\n");
	fprintf(asf,"shll $2,%%eax\n");
	fprintf(asf,"addl %%eax,_cpu_odometer\n");
	newpc("_cpu_pc");
	fprintf(asf,"xorl %%edi,%%edi\n");
	fprintf(asf,"_cpu_timeslice_0:\n");

	fprintf(asf,"movw (%%ebx),%%di\n");
	fprintf(asf,"addl $2,%%ebx\n");
	fprintf(asf,"call *SPTR(,%%edi,4)\n");

	fprintf(asf,"movw _apu_pc,%%di\n");
	fprintf(asf,"xorl %%edx,%%edx\n");
	fprintf(asf,"movb _%s_apumem(%%edi),%%dl\n",sysname);
	fprintf(asf,"call *APUPTR(,%%edx,4)\n");

	fprintf(asf,"movw (%%ebx),%%di\n");
	fprintf(asf,"addl $2,%%ebx\n");
	fprintf(asf,"call *SPTR(,%%edi,4)\n");

	fprintf(asf,"movw _apu_pc,%%di\n");
	fprintf(asf,"xorl %%edx,%%edx\n");
	fprintf(asf,"movb _%s_apumem(%%edi),%%dl\n",sysname);
	fprintf(asf,"call *APUPTR(,%%edx,4)\n");

	fprintf(asf,"movw _apu_pc,%%di\n");
	fprintf(asf,"xorl %%edx,%%edx\n");
	fprintf(asf,"movb _%s_apumem(%%edi),%%dl\n",sysname);
	fprintf(asf,"call *APUPTR(,%%edx,4)\n");

	fprintf(asf,"movw (%%ebx),%%di\n");
	fprintf(asf,"addl $2,%%ebx\n");
	fprintf(asf,"call *SPTR(,%%edi,4)\n");

	fprintf(asf,"movw _apu_pc,%%di\n");
	fprintf(asf,"xorl %%edx,%%edx\n");
	fprintf(asf,"movb _%s_apumem(%%edi),%%dl\n",sysname);
	fprintf(asf,"call *APUPTR(,%%edx,4)\n");


	fprintf(asf,"movw (%%ebx),%%di\n");
	fprintf(asf,"addl $2,%%ebx\n");
	fprintf(asf,"call *SPTR(,%%edi,4)\n");

	fprintf(asf,"movw _apu_pc,%%di\n");
	fprintf(asf,"xorl %%edx,%%edx\n");
	fprintf(asf,"movb _%s_apumem(%%edi),%%dl\n",sysname);
	fprintf(asf,"call *APUPTR(,%%edx,4)\n");

	fprintf(asf,"movw _apu_pc,%%di\n");
	fprintf(asf,"xorl %%edx,%%edx\n");
	fprintf(asf,"movb _%s_apumem(%%edi),%%dl\n",sysname);
	fprintf(asf,"call *APUPTR(,%%edx,4)\n");

	fprintf(asf,"decl _cpu_instcount\n");
	fprintf(asf,"jnz _cpu_timeslice_0\n");
	fprintf(asf,"subl _cpu_pcptrstart,%%ebx\n");
	fprintf(asf,"addl %%ebx,_cpu_pc\n");
	fprintf(asf,"popl %%edi\n");
	fprintf(asf,"popl %%esi\n");
	fprintf(asf,"popl %%ebp\n");
	fprintf(asf,"popl %%edx\n");
	fprintf(asf,"popl %%ecx\n");
	fprintf(asf,"popl %%ebx\n");
	fprintf(asf,"popl %%eax\n");
	fprintf(asf,"movb $0,_cpu_live\n");
	fprintf(asf,"ret\n");
	}
	fprintf(asf,".globl _cpu_interrupt\n");
	fprintf(asf,".align 4\n_cpu_interrupt:\n");
	fprintf(asf,"pushl %%eax\n");
	fprintf(asf,"movb _cpu_intmask,%%al\n");
	fprintf(asf,"cmpb 8(%%esp),%%al\n");
	fprintf(asf,"jae _cpu_interrupt_x\n");
	fprintf(asf,"testb $1,_cpu_live\n");
	fprintf(asf,"jz _cpu_interrupt_notlive1\n");
	fprintf(asf,"subl _cpu_pcptrstart,%%ebx\n");
	fprintf(asf,"addl %%ebx,_cpu_pc\n");
	fprintf(asf,"_cpu_interrupt_notlive1:\n");
	fprintf(asf,"pushl %%ebp\n");
	fprintf(asf,"pushl %%esi\n");
	fprintf(asf,"movb $1,_cpu_intf\n");
	sr2ax();
	fprintf(asf,"pushl %%eax\n");
	fprintf(asf,"orb $0x20,%%ah\n");
	ax2sr();
	fprintf(asf,"movl _cpu_r+60,%%esi\n");
	fprintf(asf,"subl $4,%%esi\n");
	prep_address();
	fprintf(asf,"movl _cpu_pc,%%eax\n");
	fprintf(asf,"call *_%s_wpl(,%%ebp,4)\n",sysname);
	fprintf(asf,"subl $2,%%esi\n");
	prep_address();
	fprintf(asf,"popl %%eax\n");/* status reg */
	fprintf(asf,"call *_%s_wpw(,%%ebp,4)\n",sysname);
	fprintf(asf,"movl %%esi,_cpu_r+60\n");
	fprintf(asf,"movl 16(%%esp),%%esi\n");
	fprintf(asf,"shll $2,%%esi\n");
	fprintf(asf,"addl $0x60,%%esi\n");
	prep_address();
	fprintf(asf,"call *_%s_rpl(,%%ebp,4)\n",sysname);
	fprintf(asf,"movl %%eax,_cpu_pc\n");
	fprintf(asf,"popl %%esi\n");
	fprintf(asf,"testb $1,_cpu_live\n");
	fprintf(asf,"jz _cpu_interrupt_notlive2\n");
	newpc("_cpu_pc");
	fprintf(asf,"_cpu_interrupt_notlive2:\n");
	fprintf(asf,"popl %%ebp\n");
	fprintf(asf,"_cpu_interrupt_x:\n");
	fprintf(asf,"popl %%eax\n");
	fprintf(asf,"ret\n");
	fprintf(asf,".data\n.align 4\n");
	fprintf(asf,".end\n");
}

static void usereg(void){
	fprintf(asf,"andl $7,%%edi\n");
}

/* usereg only where applicable */
static void selective_usereg(void){
	switch(main_eamode){
	case dreg:
	case areg:
	case aind:
	case ainc:
	case adec:
	case adsp:
	case axdp:
		usereg();
		break;
	default:
		break;
	}
}

static void selftest(int size){
	fprintf(asf,"test%c %s,%s\n",sizes[size],x86ax[size],x86ax[size]);
}

static void getflags(void){
	fprintf(asf,
		"movl _cpu_tmpf,%%edx\n"
		"pushfw\n"
		"popw %%bp\n"
		"andl $0x8C1,%%edx\n"
		"andl $0xF73E,%%ebp\n"
		"orl %%edx,%%ebp\n"
		"pushw %%bp\n"
		"popfw\n"
	);
}

static void flags(void){fprintf(asf,"pushfw\npopw _cpu_tmpf\n");}
static void flagz(void){
	fprintf(asf,
		"movb _cpu_tmpf,%%dh\n"
		"setz %%dl\n"
		"rorb $2,%%dl\n"
		"andb $0xBF,%%dh\n"
		"orb %%dh,%%dl\n"
		"movb %%dl,_cpu_tmpf\n"
	);
}
/* carry to X flag */
static void c2x(void){fprintf(asf,"setc _cpu_tmpf+2\n");}
/* with previous flags in EDX, adjust for non-changing zero */
static void adjzero_edxr(void){
	int myline;
	myline=linenum;
	linenum++;
	fprintf(asf,"jnz l%d\n",myline);
	fprintf(asf,"roll $17,%%edx\n");
	fprintf(asf,"orb $0xBF,%%dl\n");
	fprintf(asf,"andb %%dl,_cpu_tmpf\n");
	fprintf(asf,"l%d:\n",myline);
}

static void ret(void){fprintf(asf,"ret\n");}

/****************************************************************************
** EFFECTIVE ADDRESS GENERATION
****************************************************************************/

/* There are five types of EA activity:
**
** 1. Read:      gnea -> ldea -> pgea
** 2. Write:     gnea -> stea -> pgea
** 3. R-M-W:     gnea -> ldea -> stea -> pgea
** 4. Move:      gnea -> ldea -> pgea -> gnea2 -> stea2 -> pgea2
** 5. Control:   ccea
**
** NOTE:
** All 32-bit values must be word-swapped after loading and before storing!
*/

static void io_ccea(int size,eamode mode,int reg){
	char regs[100];
	if(reg==-1)sprintf(regs,"(,%%edi,4)");
	else sprintf(regs,"+%d",reg*4);
	switch(mode){
	case dreg:
	case areg:
		break;
	case aind:
	case ainc:
	case adec:
		fprintf(asf,"movl _cpu_r+32%s,%%esi\n",regs);
		if(mode==adec){
			if(size==1){
				if(reg==-1){
					fprintf(asf,"cmpl $7,%%edi\n");
					fprintf(asf,"adcl $-2,%%esi\n");
				}else if(reg==7){
					fprintf(asf,"subl $2,%%esi\n");
				}else{
					fprintf(asf,"decl %%esi\n");
				}
			}else{
				fprintf(asf,"%s%%esi\n",sizedec[size]);
			}
			fprintf(asf,"movl %%esi,_cpu_r+32%s\n",regs);
		}
		if((mode==adec)||(mode==ainc)){
			if(size==1){
				if(reg==-1){
					fprintf(asf,"cmpl $7,%%edi\n");
					fprintf(asf,"sbbl $-1,%%esi\n");
				}else if(reg==7){
					fprintf(asf,"incl %%esi\n");
				}
			}
		}
		break;
	case adsp:
		fprintf(asf,"movswl (%%ebx),%%esi\n");
		fprintf(asf,"addl $2,%%ebx\n");
		fprintf(asf,"addl _cpu_r+32%s,%%esi\n",regs);
		break;
	case axdp:
		fprintf(asf,"call decode_bx\n");
		fprintf(asf,"addl _cpu_r+32%s,%%esi\n",regs);
		break;
	case absw:
		fprintf(asf,"movswl (%%ebx),%%esi\n");
		fprintf(asf,"addl $2,%%ebx\n");
		break;
	case absl:
		fprintf(asf,"movl (%%ebx),%%esi\n");
		fprintf(asf,"addl $4,%%ebx\n");
		fprintf(asf,"roll $16,%%esi\n");
		break;
	case pcdp:
		fprintf(asf,"movswl (%%ebx),%%esi\n");
		fprintf(asf,"movl %%ebx,%%edx\n");
		fprintf(asf,"subl _cpu_pcptrstart,%%edx\n");
		fprintf(asf,"addl _cpu_pc,%%esi\n");
		fprintf(asf,"addl $2,%%ebx\n");
		fprintf(asf,"addl %%edx,%%esi\n");
		break;
	case pcxd:
		fprintf(asf,"call decode_bx\n");
		fprintf(asf,"movl %%ebx,%%edx\n");
		fprintf(asf,"subl _cpu_pcptrstart,%%edx\n");
		fprintf(asf,"addl _cpu_pc,%%esi\n");
		fprintf(asf,"addl %%edx,%%esi\n");
		fprintf(asf,"subl $2,%%esi\n");
		break;
	case immd:
		break;
	default:
		fprintf(asf,"#error io_ccea\n");
		break;
	}
}

static void io_gnea(int size,eamode mode){
	switch(mode){
	case dreg:
	case areg:
	case immd:
		break;
	case aind:
	case ainc:
	case adec:
	case adsp:
	case axdp:
	case absw:
	case absl:
	case pcdp:
	case pcxd:
		prep_address();
		break;
	default:
		fprintf(asf,"#error io_gnea\n");
		break;
	}
}

static void io_ldea(int size,eamode mode,int reg){
	char regs[100];
	if(reg==-1)sprintf(regs,"(,%%edi,4)");
	else sprintf(regs,"+%d",reg*4);
	switch(mode){
	case dreg:
		fprintf(asf,"mov%c _cpu_r%s,%s\n",sizes[size],regs,x86ax[size]);
		break;
	case areg:
		fprintf(asf,"mov%c _cpu_r+32%s,%s\n",sizes[size],regs,x86ax[size]);
		break;
	case aind:
	case ainc:
	case adec:
	case adsp:
	case axdp:
	case absw:
	case absl:
	case pcdp:
	case pcxd:
		fprintf(asf,"call *_%s_rp%c(,%%ebp,4)\n",sysname,sizes[size]);
		break;
	case immd:
		switch(size){
		case 1:
		case 2:
			fprintf(asf,"movw (%%ebx),%%ax\n");
			fprintf(asf,"addl $2,%%ebx\n");
			break;
		case 4:
			fprintf(asf,"movl (%%ebx),%%eax\n");
			fprintf(asf,"addl $4,%%ebx\n");
			fprintf(asf,"roll $16,%%eax\n");
			break;
		default:
			fprintf(asf,"#error io_ldea\n");
			break;
		}
		break;
	default:
		fprintf(asf,"#error io_ldea\n");
		break;
	}
}

static void io_stea(int size,eamode mode,int reg){
	char regs[100];
	if(reg==-1)sprintf(regs,"(,%%edi,4)");
	else sprintf(regs,"+%d",reg*4);
	switch(mode){
	case dreg:
		fprintf(asf,"mov%c %s,_cpu_r%s\n",sizes[size],x86ax[size],regs);
		break;
	case areg:
		fprintf(asf,"mov%c %s,_cpu_r+32%s\n",sizes[size],x86ax[size],regs);
		break;
	case aind:
	case ainc:
	case adec:
	case adsp:
	case axdp:
	case absw:
	case absl:
		fprintf(asf,"call *_%s_wp%c(,%%ebp,4)\n",sysname,sizes[size]);
		break;
	default:
		fprintf(asf,"#error io_stea\n");
		break;
	}
}

static void io_pgea(int size,eamode mode,int reg){
	char regs[100];
	if(reg==-1)sprintf(regs,"(,%%edi,4)");
	else sprintf(regs,"+%d",reg*4);
	switch(mode){
	case ainc:
		if(size==1){
			if(reg==-1){
				fprintf(asf,"cmpl $7,%%edi\n");
				fprintf(asf,"sbbl $-2,_cpu_r+32%s\n",regs);
			}else if(reg==7){
				fprintf(asf,"%s_cpu_r+32%s\n",sizeinc[2],regs);
			}else{
				fprintf(asf,"%s_cpu_r+32%s\n",sizeinc[1],regs);
			}
		}else{
			fprintf(asf,"%s_cpu_r+32%s\n",sizeinc[size],regs);
		}
		break;
	case dreg:
	case areg:
	case aind:
	case adec:
	case adsp:
	case axdp:
	case absw:
	case absl:
	case pcdp:
	case pcxd:
	case immd:
		break;
	default:
		fprintf(asf,"#error io_pgea\n");
		break;
	}
}

/* Combined EA routines */

static void ea_load(int size,eamode mode,int reg){
	io_ccea(size,mode,reg);
	io_gnea(size,mode);
	io_ldea(size,mode,reg);
	io_pgea(size,mode,reg);
}

static void ea_store(int size,eamode mode,int reg){
	io_ccea(size,mode,reg);
	io_gnea(size,mode);
	io_stea(size,mode,reg);
	io_pgea(size,mode,reg);
}

static void ea_rmw_load(int size,eamode mode,int reg){
	io_ccea(size,mode,reg);
	io_gnea(size,mode);
	io_ldea(size,mode,reg);
}

static void ea_rmw_store(int size,eamode mode,int reg){
	io_stea(size,mode,reg);
	io_pgea(size,mode,reg);
}

static void ea_control(eamode mode,int reg){
	io_ccea(0,mode,reg);
}

static void main_ea_load(void){
	ea_load(main_size,main_eamode,-1);
}

static void main_ea_store(void){
	ea_store(main_size,main_eamode,-1);
}

static void main_ea_rmw_load(void){
	ea_rmw_load(main_size,main_eamode,-1);
}

static void main_ea_rmw_store(void){
	ea_rmw_store(main_size,main_eamode,-1);
}

static void main_ea_control(void){
	ea_control(main_eamode,-1);
}

/***************************************************************************/

static void gen_exception(void){
	fprintf(asf,"exception:\n");
	fprintf(asf,"movl %%eax,%%edx\n");
	sr2ax();
	fprintf(asf,"pushl %%eax\n");
	fprintf(asf,"orb $0x20,%%ah\n");
	ax2sr();
	fprintf(asf,"subl _cpu_pcptrstart,%%ebx\n");
	fprintf(asf,"addl %%ebx,_cpu_pc\n");
	fprintf(asf,"xchgl %%edx,_cpu_pc\n");
	fprintf(asf,"movl %%edx,%%eax\n");
	ea_store(4,adec,7);
	fprintf(asf,"popl %%eax\n");
	ea_store(2,adec,7);
	newpc("_cpu_pc");
	ret();
}

static void gen_selftests(void){
	fprintf(asf,"selftest1:\n");
	selftest(1);
	flags();
	ret();
	fprintf(asf,"selftest2:\n");
	selftest(2);
	flags();
	ret();
	fprintf(asf,"selftest4:\n");
	selftest(4);
	flags();
	ret();
}

static void jmp_selftest(int s){fprintf(asf,"jmp selftest%d\n",s);}

static void prefixes(void){
	banner();
	gen_variables();
	gen_unimplemented();
	gen_decode_bx();
	gen_exception();
	gen_selftests();
}

/****************************************************************************
** INSTRUCTION ROUTINES
****************************************************************************/

/* called 600 times (!) */
static void i_move(void){
	selective_usereg();
	main_ea_load();
	ea_store(main_size,main_destmode,main_reg);
	jmp_selftest(main_size);
}

static void i_moveq(void){
	fprintf(asf,"movl %%edi,%%eax\n");
	fprintf(asf,"movsbl %%al,%%eax\n");
	fprintf(asf,"movl %%eax,_cpu_r+%d\n",main_reg*4);
	fprintf(asf,"testb %%al,%%al\n");
	flags();
	ret();
}

static void op_to_areg(char*s){
	selective_usereg();
	main_ea_load();
	if(main_size==2)fprintf(asf,"cwde\n");
	fprintf(asf,"%sl %%eax,_cpu_r+%d\n",s,32+(main_reg*4));
}

static void i_movea(void){op_to_areg("mov");ret();}
static void i_adda (void){op_to_areg("add");ret();}
static void i_suba (void){op_to_areg("sub");ret();}
static void i_cmpa (void){op_to_areg("cmp");flags();ret();}

static void i_move_to_sr(void){
	selective_usereg();
	main_size=2;
	main_ea_load();
	ax2sr();
	ret();
}

static void i_move_to_ccr(void){
	selective_usereg();
	main_size=1;
	main_ea_load();
	al2ccr();
	ret();
}

static void i_move_from_sr(void){
	selective_usereg();
	main_size=2;
	sr2ax();
	main_ea_store();
	ret();
}

static void i_move_from_ccr(void){
	selective_usereg();
	main_size=1;
	ccr2al();
	main_ea_store();
	ret();
}

static void op_to_ccr(char*op){
	ccr2al();
	fprintf(asf,"%sb (%%ebx),%%al\n",op);
	fprintf(asf,"addl $2,%%ebx\n");
	al2ccr();
	ret();
}

static void op_to_sr(char*op){
	sr2ax();
	fprintf(asf,"%sw (%%ebx),%%ax\n",op);
	fprintf(asf,"addl $2,%%ebx\n");
	ax2sr();
	ret();
}

static void  i_ori_ccr(void){op_to_ccr("or" );}
static void i_andi_ccr(void){op_to_ccr("and");}
static void i_eori_ccr(void){op_to_ccr("xor");}
static void  i_ori_sr (void){op_to_sr ("or" );}
static void i_andi_sr (void){op_to_sr ("and");}
static void i_eori_sr (void){op_to_sr ("xor");}

static void i_clr(void){
	selective_usereg();
	fprintf(asf,"xorl %%eax,%%eax\n");
	main_ea_store();
	fprintf(asf,"movw $0x40,_cpu_tmpf\n");
	ret();
}

static void i_tst(void){
	selective_usereg();
	main_ea_load();
	jmp_selftest(main_size);
}

/* Always affects X flag */
static void op_quick(char*op){
	selective_usereg();
	if(main_eamode==dreg){
		fprintf(asf,
			"%s%c $%d,_cpu_r(,%%edi,4)\n",
			op,sizes[main_size],quickvalue[main_qv]
		);
		flags();
		c2x();
	}else if(main_eamode==areg){
		fprintf(asf,
			"%sl $%d,_cpu_r+32(,%%edi,4)\n",
			op,quickvalue[main_qv]
		);
	}else{
		main_ea_rmw_load();
		fprintf(asf,
			"%s%c $%d,%s\n",op,sizes[main_size],
			quickvalue[main_qv],x86ax[main_size]
		);
		flags();
		c2x();
		main_ea_rmw_store();
	}
	ret();
}

static void i_addq(void){op_quick("add");}
static void i_subq(void){op_quick("sub");}

static void op_to_dn(char*op,int affectx){
	selective_usereg();
	main_ea_load();
	fprintf(asf,
		"%s%c %s,_cpu_r+%d\n",op,sizes[main_size],
		x86ax[main_size],main_reg*4
	);
	flags();
	if(affectx)c2x();
	ret();
}

static void i_cmp_dn(void){op_to_dn("cmp",0);}
static void i_add_dn(void){op_to_dn("add",1);}
static void i_sub_dn(void){op_to_dn("sub",1);}
static void i_and_dn(void){op_to_dn("and",0);}
static void i_or_dn (void){op_to_dn("or" ,0);}

static void op_to_ea(char*op,int affectx){
	selective_usereg();
	main_ea_rmw_load();
	fprintf(asf,
		"%s%c _cpu_r+%d,%s\n",op,sizes[main_size],
		main_reg*4,x86ax[main_size]
	);
	flags();
	if(affectx)c2x();
	main_ea_rmw_store();
	ret();
}

static void i_eor_ea(void){op_to_ea("xor",0);}
static void i_add_ea(void){op_to_ea("add",1);}
static void i_sub_ea(void){op_to_ea("sub",1);}
static void i_and_ea(void){op_to_ea("and",0);}
static void i_or_ea (void){op_to_ea("or" ,0);}

static void save_ecx(void){/*fprintf(asf,"pushl %%ecx\n");*/}
static void restore_ecx(void){/*fprintf(asf,"popl %%ecx\n");*/}

/* called 144 times */
static void im_to_ea(char*op,int wback,int affectx){
	save_ecx();
	selective_usereg();
	switch(main_size){
	case 1:
	case 2:
		fprintf(asf,"movw (%%ebx),%%cx\n");
		fprintf(asf,"addl $2,%%ebx\n");
		break;
	case 4:
		fprintf(asf,"movl (%%ebx),%%ecx\n");
		fprintf(asf,"roll $16,%%ecx\n");
		fprintf(asf,"addl $4,%%ebx\n");
		break;
	default:
		break;
	}
	if(main_eamode==dreg){
		fprintf(asf,
			"%s%c %s,_cpu_r(,%%edi,4)\n",
			op,sizes[main_size],x86cx[main_size]
		);
		flags();
		if(affectx)c2x();
	}else{
		if(wback)main_ea_rmw_load();else main_ea_load();
		fprintf(asf,
			"%s%c %s,%s\n",op,sizes[main_size],
			x86cx[main_size],x86ax[main_size]
		);
		flags();
		if(affectx)c2x();
		if(wback)main_ea_rmw_store();
	}
	restore_ecx();
	ret();
}

/* each called 24 times */
static void i_addi(void){im_to_ea("add",1,1);}
static void i_subi(void){im_to_ea("sub",1,1);}
static void i_cmpi(void){im_to_ea("cmp",0,0);}
static void i_andi(void){im_to_ea("and",1,0);}
static void i_ori (void){im_to_ea("or" ,1,0);}
static void i_eori(void){im_to_ea("xor",1,0);}

static void flick_reg(char*op,int needxf,int affectx,int zerov){
	char tmps[5];
	save_ecx();
	usereg();
	if(main_ir==1){
		fprintf(asf,"movb _cpu_r+%d,%%cl\n",main_reg*4);
		fprintf(asf,"andb $63,%%cl\n");
		sprintf(tmps,"%%cl");
	}else{
		sprintf(tmps,"$%d",quickvalue[main_reg]);
	}
	if(needxf)fprintf(asf,"shrb $1,_cpu_tmpf+2\n");
	fprintf(asf,
		"%s%c%c %s,_cpu_r(,%%edi,4)\n",
		op,direction[main_dr],sizes[main_size],tmps
	);
	flags();
	if(affectx)c2x();
	if(zerov)fprintf(asf,"movb $0,_cpu_tmpf+1\n");
	restore_ecx();
	ret();
}

static void i_lsx_reg(void){flick_reg("sh",0,1,1);}
static void i_asx_reg(void){flick_reg("sa",0,1,0);}
static void i_rox_reg(void){flick_reg("ro",0,0,1);}
static void i_rxx_reg(void){flick_reg("rc",1,1,1);}

static void flick_mem(char*op,int needxf,int affectx,int zerov){
	main_size=2;
	selective_usereg();
	main_ea_rmw_load();
	if(needxf)fprintf(asf,"shrb $1,_cpu_tmpf+2\n");
	fprintf(asf,"%s%cw $1,%%ax\n",op,direction[main_dr]);
	flags();
	if(affectx)c2x();
	if(zerov)fprintf(asf,"movb $0,_cpu_tmpf+1\n");
	main_ea_rmw_store();
	ret();
}

static void i_lsx_mem(void){flick_mem("sh",0,1,1);}
static void i_asx_mem(void){flick_mem("sa",0,1,0);}
static void i_rox_mem(void){flick_mem("ro",0,0,1);}
static void i_rxx_mem(void){flick_mem("rc",1,1,1);}

static int created_bra_b=0;
static void i_bra_b(void){
	if(!created_bra_b){fprintf(asf,"r_bra_b:\n");created_bra_b=1;}
	fprintf(asf,
		"movl %%edi,%%eax\n"
		"movsbl %%al,%%eax\n"
		"addl %%eax,%%ebx\n"
	);
	ret();
}

static int created_bra_w=0;
static void i_bra_w(void){
	if(!created_bra_w){fprintf(asf,"r_bra_w:\n");created_bra_w=1;}
	fprintf(asf,
		"movswl (%%ebx),%%eax\n"
		"addl %%eax,%%ebx\n"
	);
	ret();
}

static void i_bsr_b(void){
	fprintf(asf,
		"movl %%edi,%%eax\n"
		"movsbl %%al,%%eax\n"
		"movl %%ebx,%%edx\n"
		"subl _cpu_pcptrstart,%%edx\n"
		"addl _cpu_pc,%%edx\n"
		"addl %%eax,%%ebx\n"
		"movl %%edx,%%eax\n"
	);
	ea_store(4,adec,7);
	ret();
}

static void i_bsr_w(void){
	fprintf(asf,
		"movswl (%%ebx),%%eax\n"
		"movl %%ebx,%%edx\n"
		"subl _cpu_pcptrstart,%%edx\n"
		"addl _cpu_pc,%%edx\n"
		"addl $2,%%edx\n"
		"addl %%eax,%%ebx\n"
		"movl %%edx,%%eax\n"
	);
	ea_store(4,adec,7);
	ret();
}

static char optcc[10],optrc[10];

static void main_getcondition(void){
	switch(main_cc){
	case 0x0:
	case 0x1:
		break;
	case 0x2:/* a */
		fprintf(asf,"testb $0x41,_cpu_tmpf\n");
		sprintf(optcc,"z");
		sprintf(optrc,"nz");
		break;
	case 0x3:/* be */
		fprintf(asf,"testb $0x41,_cpu_tmpf\n");
		sprintf(optcc,"nz");
		sprintf(optrc,"z");
		break;
	case 0x4:/* nc */
		fprintf(asf,"testb $1,_cpu_tmpf\n");
		sprintf(optcc,"z");
		sprintf(optrc,"nz");
		break;
	case 0x5:/* c */
		fprintf(asf,"testb $1,_cpu_tmpf\n");
		sprintf(optcc,"nz");
		sprintf(optrc,"z");
		break;
	case 0x6:/* ne */
		fprintf(asf,"testb $0x40,_cpu_tmpf\n");
		sprintf(optcc,"z");
		sprintf(optrc,"nz");
		break;
	case 0x7:/* e */
		fprintf(asf,"testb $0x40,_cpu_tmpf\n");
		sprintf(optcc,"nz");
		sprintf(optrc,"z");
		break;
	case 0x8:/* no */
		fprintf(asf,"testb $8,_cpu_tmpf+1\n");
		sprintf(optcc,"z");
		sprintf(optrc,"nz");
		break;
	case 0x9:/* o */
		fprintf(asf,"testb $8,_cpu_tmpf+1\n");
		sprintf(optcc,"nz");
		sprintf(optrc,"z");
		break;
	case 0xA:/* ns */
		fprintf(asf,"testb $0x80,_cpu_tmpf\n");
		sprintf(optcc,"z");
		sprintf(optrc,"nz");
		break;
	case 0xB:/* s */
		fprintf(asf,"testb $0x80,_cpu_tmpf\n");
		sprintf(optcc,"nz");
		sprintf(optrc,"z");
		break;
	case 0xC:/* ge */
		getflags();
		sprintf(optcc,"ge");
		sprintf(optrc,"l");
		break;
	case 0xD:/* l */
		getflags();
		sprintf(optcc,"l");
		sprintf(optrc,"ge");
		break;
	case 0xE:/* g */
		getflags();
		sprintf(optcc,"g");
		sprintf(optrc,"le");
		break;
	case 0xF:/* le */
		getflags();
		sprintf(optcc,"le");
		sprintf(optrc,"g");
		break;
	default:break;
	}
}

static void i_bcc_b(void){
	main_getcondition();
	fprintf(asf,"j%s r_bra_b\n",optcc);
	ret();
}

static void i_bcc_w(void){
	main_getcondition();
	fprintf(asf,"j%s r_bra_w\n",optcc);
	fprintf(asf,"addl $2,%%ebx\n");
	ret();
}

static int created_dbra=0;
static void i_dbra(void){
	if(!created_dbra){fprintf(asf,"r_dbra:\n");created_dbra=1;}
	usereg();
	fprintf(asf,
		"subw $1,_cpu_r(,%%edi,4)\n"
		"jnc r_bra_w\n");
	fprintf(asf,"addl $2,%%ebx\n");
	ret();
}

static void i_dbcc(void){
	main_getcondition();
	fprintf(asf,"j%s r_dbra\n",optrc);
	fprintf(asf,"addl $2,%%ebx\n");
	ret();
}

static void i_scc(void){
	main_size=1;
	selective_usereg();
	if(main_cc>1){
		main_getcondition();
		fprintf(asf,"set%s %%al\n",optcc);
		fprintf(asf,"negb %%al\n");
	}else{
		fprintf(asf,"movb $%d,%%al\n",(main_cc^1)*0xFF);
	}
	main_ea_store();
	ret();
}

/* called 315 times */
/* 0=btst,1=bchg,2=bclr,3=bset */
static void bitop(void){
	selective_usereg();
	fprintf(asf,"andl $%d,%%ecx\n",(main_eamode==dreg)?31:7);
	if(main_eamode==dreg){
		main_size=4;
		if(!main_cc){
			fprintf(asf,"movl _cpu_r(,%%edi,4),%%eax\n");
			fprintf(asf,"shrl %%cl,%%eax\n");
			fprintf(asf,"testb $1,%%al\n");
			flagz();
		}else{
			fprintf(asf,"movl $1,%%eax\n");
			fprintf(asf,"shll %%cl,%%eax\n");
			fprintf(asf,"testl %%eax,_cpu_r(,%%edi,4)\n");
			flagz();
			switch(main_cc){
			case 1:fprintf(asf,"xorl %%eax,_cpu_r(,%%edi,4)\n");break;
			case 2:fprintf(asf,"notl %%eax\nandl %%eax,_cpu_r(,%%edi,4)\n");break;
			case 3:fprintf(asf,"orl %%eax,_cpu_r(,%%edi,4)\n");break;
			default:break;
			}
		}
	}else{
		main_size=1;
		if(!main_cc){
			main_ea_load();
			fprintf(asf,"shrl %%cl,%%eax\n");
			fprintf(asf,"testb $1,%%al\n");
			flagz();
		}else{
			fprintf(asf,"movl $1,%%eax\n");
			fprintf(asf,"shll %%cl,%%eax\n");
			fprintf(asf,"movl %%eax,%%ecx\n");
			main_ea_rmw_load();
			fprintf(asf,"testl %%ecx,%%eax\n");
			flagz();
			switch(main_cc){
			case 1:fprintf(asf,"xorl %%ecx,%%eax\n");break;
			case 2:fprintf(asf,"notl %%ecx\nandl %%ecx,%%eax\n");break;
			case 3:fprintf(asf,"orl %%ecx,%%eax\n");break;
			default:break;
			}
			main_ea_rmw_store();
		}
	}
	restore_ecx();
	ret();
}

/* called 35 times */
static void i_bitop_imm(void){
	save_ecx();
	fprintf(asf,"movw (%%ebx),%%cx\n");
	fprintf(asf,"addl $2,%%ebx\n");
	bitop();
}

/* called 280 times */
static void i_bitop_reg(void){
	save_ecx();
	fprintf(asf,"movb _cpu_r+%d,%%cl\n",main_reg*4);
	bitop();
}

static void i_jmp(void){
	selective_usereg();
	main_ea_control();
	newpc("%esi");
	ret();
}

static void i_jsr(void){
	selective_usereg();
	main_ea_control();
	fprintf(asf,"subl _cpu_pcptrstart,%%ebx\n");
	fprintf(asf,"addl %%ebx,_cpu_pc\n");
	fprintf(asf,"xchgl %%esi,_cpu_pc\n");
	fprintf(asf,"movl %%esi,%%eax\n");
	ea_store(4,adec,7);
	newpc("_cpu_pc");
	ret();
}

static void i_rts(void){
	ea_load(4,ainc,7);
	newpc("%eax");
	ret();
}

static void i_rte(void){
	ea_load(2,ainc,7);
        fprintf(asf,"xorl $1,_cpu_checker\n");
        fprintf(asf,"pushl %%eax\n");
	ea_load(4,ainc,7);
	newpc("%eax");
	fprintf(asf,"popl %%eax\n");
	ax2sr();
	ret();
}

static void i_wait(int cmd){
        int myline=linenum;
        linenum+=1;
        switch (cmd) {
        case 0: // wait until bit1 is 1
          fprintf(asf, "testl $0x00000002, _cpu_checker\n");
          fprintf(asf, "jnz l%d\n",myline);
          fprintf(asf, "subl $2,%%ebx\n");
          fprintf(asf, "l%d:\n",myline);
          fprintf(asf, "andl  $0xfffffffd, _cpu_checker\n");
          break;
        case 1: // set bit1 to 1
          fprintf(asf, "orl   $0x00000002, _cpu_checker\n");
          break;

        }
        ret();
}

static void i_wait0(void) {
        i_wait(0);
}
static void i_wait1(void) {
        i_wait(1);
}

static void i_lea(void){
	selective_usereg();
	main_ea_control();
	fprintf(asf,"movl %%esi,_cpu_r+%d\n",32+(main_reg*4));
	ret();
}

static void i_pea(void){
	selective_usereg();
	main_ea_control();
	fprintf(asf,"movl %%esi,%%eax\n");
	ea_store(4,adec,7);
	ret();
}

static void i_reset(void) {
        fprintf(asf,"pushal\n");
        fprintf(asf,"call _reset_external_device\n");
        fprintf(asf,"popal\n");
        ret();
}

static void i_nop(void){
	ret();
}

static void i_movem_control(void){
	int myline;
	myline=linenum;
	linenum+=2;
	save_ecx();
	selective_usereg();
	fprintf(asf,"movw (%%ebx),%%cx\n");
	fprintf(asf,"addl $2,%%ebx\n");
	main_ea_control();
	prep_address();
	fprintf(asf,"xorl %%edx,%%edx\n");
	fprintf(asf,"l%d:\n",myline);
	fprintf(asf,"shrl $1,%%ecx\n");
	fprintf(asf,"jnc l%d\n",myline+1);
	if(main_dr==0){/*register->memory*/
		fprintf(asf,"movl _cpu_r(,%%edx,4),%%eax\n");
		fprintf(asf,
			"call *_%s_wp%c(,%%ebp,4)\n",
			sysname,sizes[main_size]
		);
		fprintf(asf,"addl $%d,%%esi\n",main_size);
	}else{/*memory->register*/
		fprintf(asf,
			"call *_%s_rp%c(,%%ebp,4)\n",
			sysname,sizes[main_size]
		);
		fprintf(asf,"addl $%d,%%esi\n",main_size);
                if (main_size==2) {
                 fprintf(asf,"cwde\n");
                 fprintf(asf,"movl %%eax,_cpu_r(,%%edx,4)\n");

                } else
                fprintf(asf,
			"mov%c %s,_cpu_r(,%%edx,4)\n",
			sizes[main_size],x86ax[main_size]
		);
	}
	fprintf(asf,"l%d:\n",myline+1);
	fprintf(asf,"incl %%edx\n");
	fprintf(asf,"cmpl $0x10,%%edx\n");
	fprintf(asf,"jne l%d\n",myline);
	restore_ecx();
	ret();
}

static void i_movem_postinc(void){
	int myline;
	myline=linenum;
	linenum+=2;
	save_ecx();
	usereg();
	fprintf(asf,"movw (%%ebx),%%cx\n");
	fprintf(asf,"addl $2,%%ebx\n");
	fprintf(asf,"movl _cpu_r+32(,%%edi,4),%%esi\n");
	prep_address();
	fprintf(asf,"xorl %%edx,%%edx\n");
	fprintf(asf,"l%d:\n",myline);
	fprintf(asf,"shrl $1,%%ecx\n");
	fprintf(asf,"jnc l%d\n",myline+1);
	fprintf(asf,
		"call *_%s_rp%c(,%%ebp,4)\n",
		sysname,sizes[main_size]
	);
	fprintf(asf,"addl $%d,%%esi\n",main_size);
	fprintf(asf,"addl $%d,_cpu_r+32(,%%edi,4)\n",main_size);
        if (main_size==2) {
                 fprintf(asf,"cwde\n");
                 fprintf(asf,"movl %%eax,_cpu_r(,%%edx,4)\n");
        } else
        fprintf(asf,
		"mov%c %s,_cpu_r(,%%edx,4)\n",
		sizes[main_size],x86ax[main_size]
	);
	fprintf(asf,"l%d:\n",myline+1);
	fprintf(asf,"incl %%edx\n");
	fprintf(asf,"cmpl $0x10,%%edx\n");
	fprintf(asf,"jne l%d\n",myline);
	restore_ecx();
	ret();
}

static void i_movem_predec(void){
	int myline;
	myline=linenum;
	linenum+=2;
	save_ecx();
	usereg();
	fprintf(asf,"movw (%%ebx),%%cx\n");
	fprintf(asf,"addl $2,%%ebx\n");
	fprintf(asf,"movl _cpu_r+32(,%%edi,4),%%esi\n");
	prep_address();
	fprintf(asf,"movl $0xF,%%edx\n");
	fprintf(asf,"l%d:\n",myline);
	fprintf(asf,"shrl $1,%%ecx\n");
	fprintf(asf,"jnc l%d\n",myline+1);
	fprintf(asf,"movl _cpu_r(,%%edx,4),%%eax\n");
	fprintf(asf,"subl $%d,%%esi\n",main_size);
	fprintf(asf,"subl $%d,_cpu_r+32(,%%edi,4)\n",main_size);
	fprintf(asf,
		"call *_%s_wp%c(,%%ebp,4)\n",
		sysname,sizes[main_size]
	);
	fprintf(asf,"l%d:\n",myline+1);
	fprintf(asf,"decl %%edx\n");
	fprintf(asf,"jns l%d\n",myline);
	restore_ecx();
	ret();
}

static void i_link(void){
	usereg();
	fprintf(asf,"movl _cpu_r+32(,%%edi,4),%%eax\n");
	ea_store(4,adec,7);
	fprintf(asf,"movl _cpu_r+60,%%eax\n");
	fprintf(asf,"movl %%eax,_cpu_r+32(,%%edi,4)\n");
	fprintf(asf,"movswl (%%ebx),%%eax\n");
	fprintf(asf,"addl %%eax,_cpu_r+60\n");
	fprintf(asf,"addl $2,%%ebx\n");
	ret();
}

static void i_unlk(void){
	usereg();
	fprintf(asf,"movl _cpu_r+32(,%%edi,4),%%eax\n");
	fprintf(asf,"movl %%eax,_cpu_r+60\n");
	ea_load(4,ainc,7);
	fprintf(asf,"movl %%eax,_cpu_r+32(,%%edi,4)\n");
	ret();
}

static void i_movefusp(void){
	int myline;
	myline=linenum;
	linenum++;
	usereg();
	fprintf(asf,"testb $0x20,_cpu_tmpf+3\n");
	fprintf(asf,"jz l%d\n",myline);
	fprintf(asf,"movl _cpu_asp,%%eax\n");
	fprintf(asf,"movl %%eax,_cpu_r+32(,%%edi,4)\n");
	ret();
	fprintf(asf,"l%d:\n",myline);
	fprintf(asf,"movl _cpu_r+%d,%%eax\n",32+28);
	fprintf(asf,"movl %%eax,_cpu_r+32(,%%edi,4)\n");
	ret();
}

static void i_move2usp(void){
	int myline;
	myline=linenum;
	linenum++;
	usereg();
	fprintf(asf,"movl _cpu_r+32(,%%edi,4),%%eax\n");
	fprintf(asf,"testb $0x20,_cpu_tmpf+3\n");
	fprintf(asf,"jz l%d\n",myline);
	fprintf(asf,"movl %%eax,_cpu_asp\n");
	ret();
	fprintf(asf,"l%d:\n",myline);
	fprintf(asf,"movl %%eax,_cpu_r+%d\n",32+28);
	ret();
}

static void i_trap(void){
	sr2ax();
	fprintf(asf,"pushl %%eax\n");
	fprintf(asf,"orb $0x20,%%ah\n");
	ax2sr();
	fprintf(asf,"movl %%edi,%%esi\n");
	fprintf(asf,"andl $0xF,%%esi\n");
	fprintf(asf,"shll $2,%%esi\n");
	fprintf(asf,"addl $0x80,%%esi\n");
	prep_address();
	fprintf(asf,"call *_%s_rpl(,%%ebp,4)\n",sysname);
	fprintf(asf,"subl _cpu_pcptrstart,%%ebx\n");
	fprintf(asf,"addl %%ebx,_cpu_pc\n");
	fprintf(asf,"xchgl %%eax,_cpu_pc\n");
	ea_store(4,adec,7);
	fprintf(asf,"popl %%eax\n");
	ea_store(2,adec,7);
	newpc("_cpu_pc");
	ret();
}

static void i_stop(void){
	int myline;
	myline=linenum;
	linenum+=2;
	fprintf(asf,"cmpb $1,_cpu_stopped\n");
	fprintf(asf,"je l%d\n",myline);
	fprintf(asf,"movw (%%ebx),%%ax\n");
	ax2sr();
	fprintf(asf,"movb $1,_cpu_stopped\n");
	fprintf(asf,"l%d:\n",myline);
	fprintf(asf,"subl $2,%%ebx\n");
	fprintf(asf,"cmpb $0,_cpu_intf\n");
	fprintf(asf,"je l%d\n",myline+1);
	fprintf(asf,"movb $0,_cpu_intf\n");
	fprintf(asf,"movb $0,_cpu_stopped\n");
	fprintf(asf,"addl $4,%%ebx\n");
	fprintf(asf,"l%d:\n",myline+1);
	ret();
}

static void i_extbw(void){
	usereg();
	fprintf(asf,"movsbw _cpu_r(,%%edi,4),%%ax\n");
	fprintf(asf,"movw %%ax,_cpu_r(,%%edi,4)\n");
	jmp_selftest(2);
}

static void i_extwl(void){
	usereg();
	fprintf(asf,"movswl _cpu_r(,%%edi,4),%%eax\n");
	fprintf(asf,"movl %%eax,_cpu_r(,%%edi,4)\n");
	jmp_selftest(4);
}

static void i_swap(void){
	usereg();
	fprintf(asf,"movl _cpu_r(,%%edi,4),%%eax\n");
	fprintf(asf,"roll $16,%%eax\n");
	fprintf(asf,"movl %%eax,_cpu_r(,%%edi,4)\n");
	jmp_selftest(4);
}

/* if main_cc=1 then it's signed */
static void i_mul(void){
	selective_usereg();
	main_size=2;
	main_ea_load();
	fprintf(asf,"%smulw _cpu_r+%d\n",(main_cc==1)?"i":"",main_reg*4);
	fprintf(asf,"andl $0xFFFF,%%eax\n");
	fprintf(asf,"shll $16,%%edx\n");
	fprintf(asf,"orl %%edx,%%eax\n");
	flags();
	fprintf(asf,"movl %%eax,_cpu_r+%d\n",main_reg*4);
	ret();
}

static void exception(int e){
	fprintf(asf,"movl $%d,%%esi\n",e<<2);
	fprintf(asf,"call *_%s_rpl+%d\n",sysname,(e<<2)>>bankbits);
	fprintf(asf,"call exception\n");
}

/* if main_cc=1 then it's signed */
static void i_div(void){
	int myline;
	myline=linenum;
	linenum+=2;
	selective_usereg();
	main_size=2;
	main_ea_load();
	fprintf(asf,"testw %%ax,%%ax\n");
	fprintf(asf,"jnz l%d\n",myline);
	exception(5);
	ret();
	fprintf(asf,"l%d:\n",myline);myline++;
	save_ecx();
	fprintf(asf,"mov%cwl %%ax,%%ecx\n",main_cc?'s':'z');
	fprintf(asf,"movl _cpu_r+%d,%%eax\n",main_reg*4);
	if(main_cc)fprintf(asf,"cdq\n");
	else fprintf(asf,"xorl %%edx,%%edx\n");
	fprintf(asf,"%sdivl %%ecx\n",main_cc?"i":"");
//        fprintf(asf,"testl $0xFFFF0000,%%eax\n");
//        fprintf(asf,"jnz l%d\n",myline);
	fprintf(asf,"shll $16,%%edx\n");
	fprintf(asf,"orl %%edx,%%eax\n");
	flags();
	fprintf(asf,"movl %%eax,_cpu_r+%d\n",main_reg*4);
	restore_ecx();
	ret();
	fprintf(asf,"l%d:\n",myline);myline++;
	fprintf(asf,"movw $0x800,_cpu_tmpf\n");
	restore_ecx();
	ret();
}

static void i_neg(void){
	selective_usereg();
	if(main_eamode==dreg){
		fprintf(asf,"neg%c _cpu_r(,%%edi,4)\n",sizes[main_size]);
		flags();
		c2x();
	}else{
		main_ea_rmw_load();
		fprintf(asf,"neg%c %s\n",sizes[main_size],x86ax[main_size]);
		flags();
		c2x();
		main_ea_rmw_store();
	}
	ret();
}

static void i_negx(void){
	selective_usereg();
	if(main_eamode==dreg){
		fprintf(asf,"movl _cpu_tmpf,%%eax\n");
		fprintf(asf,"shrl $17,%%eax\n");
		if(main_size==1)fprintf(asf,"movb $0,%%al\n");
		else fprintf(asf,"movl $0,%%eax\n");
		fprintf(asf,
			"sbb%c _cpu_r(,%%edi,4),%s\n",
			sizes[main_size],x86ax[main_size]
		);
		flags();
		c2x();
		fprintf(asf,"mov%c %s,_cpu_r(,%%edi,4)\n",
			sizes[main_size],x86ax[main_size]
		);
	}else{
		main_ea_rmw_load();

		fprintf(asf,"movl _cpu_tmpf,%%edx\n");
		fprintf(asf,"shrl $17,%%edx\n");
		if(main_size==1)fprintf(asf,"movb $0,%%dl\n");
		else fprintf(asf,"movl $0,%%edx\n");
		fprintf(asf,
			"sbb%c %s,%s\n",sizes[main_size],
			x86ax[main_size],x86dx[main_size]
		);
		fprintf(asf,"movl %%edx,%%eax\n");
		flags();
		c2x();
		main_ea_rmw_store();
	}
	ret();
}

static void i_nbcd(void){
	main_size=1;
	selective_usereg();
	if(main_eamode==dreg){
		fprintf(asf,"movl _cpu_tmpf,%%eax\n");
		fprintf(asf,"shrl $17,%%eax\n");
		fprintf(asf,"movb $0,%%al\n");
		fprintf(asf,"sbbb _cpu_r(,%%edi,4),%%al\n");
		fprintf(asf,"das\n");
		flags();
		c2x();
		fprintf(asf,"movb %%al,_cpu_r(,%%edi,4)\n");
	}else{
		main_ea_rmw_load();
		fprintf(asf,"movb %%al,%%dl\n");
		fprintf(asf,"movl _cpu_tmpf,%%eax\n");
		fprintf(asf,"shrl $17,%%eax\n");
		fprintf(asf,"movb $0,%%al\n");
		fprintf(asf,"sbbb %%dl,%%al\n");
		fprintf(asf,"das\n");
		flags();
		c2x();
		main_ea_rmw_store();
	}
	ret();
}

static void i_tas(void){
	main_size=1;
	selective_usereg();
	if(main_eamode==dreg){
		fprintf(asf,"movb _cpu_r(,%%edi,4),%%al\n");
		fprintf(asf,"testb %%al,%%al\n");
		flags();
		fprintf(asf,"orb $0x80,%%al\n");
		fprintf(asf,"movb %%al,_cpu_r(,%%edi,4)\n");
	}else{
		main_ea_rmw_load();
		fprintf(asf,"testb %%al,%%al\n");
		flags();
		fprintf(asf,"orb $0x80,%%al\n");
		main_ea_rmw_store();
	}
	ret();
}

static void i_not(void){
	selective_usereg();
	if(main_eamode==dreg){
		fprintf(asf,"xor%c $-1,_cpu_r(,%%edi,4)\n",sizes[main_size]);
		flags();
	}else{
		main_ea_rmw_load();
		fprintf(asf,"xor%c $-1,%s\n",sizes[main_size],x86ax[main_size]);
		flags();
		main_ea_rmw_store();
	}
	ret();
}

/* main_reg=rx, main_dr is 0 or 32 (rx a/d), main_ir is 0 or 32 (ry a/d) */
static void i_exg(void){
	usereg();
	fprintf(asf,"movl _cpu_r+%d,%%eax\n",(main_reg*4)+main_dr);
	fprintf(asf,"xchgl %%eax,_cpu_r+%d(,%%edi,4)\n",main_ir);
	fprintf(asf,"movl %%eax,_cpu_r+%d\n",(main_reg*4)+main_dr);
	ret();
}

static void i_cmpm(void){
	int myline;
	myline=linenum;
	linenum++;
	save_ecx();
	usereg();
	fprintf(asf,"movl _cpu_r+32(,%%edi,4),%%esi\n");
	prep_address();
	fprintf(asf,"call *_%s_rp%c(,%%ebp,4)\n",sysname,sizes[main_size]);
	fprintf(asf,"movl %%eax,%%ecx\n");
	fprintf(asf,"movl _cpu_r+%d,%%esi\n",32+(main_reg*4));
	prep_address();
	fprintf(asf,"call *_%s_rp%c(,%%ebp,4)\n",sysname,sizes[main_size]);
	fprintf(asf,
		"cmp%c %s,%s\n",sizes[main_size],
		x86cx[main_size],x86ax[main_size]
	);
	flags();
	fprintf(asf,"cmpl $%d,%%edi\n",main_reg);
	fprintf(asf,"je l%d\n",myline);
	fprintf(asf,"%s_cpu_r+32(,%%edi,4)\n",sizeinc[main_size]);
	fprintf(asf,"l%d:\n",myline);myline++;
	fprintf(asf,"%s_cpu_r+%d\n",sizeinc[main_size],32+(main_reg*4));
	restore_ecx();
	ret();
}

static void opx_dreg(char*op){
	usereg();
	fprintf(asf,"movl _cpu_r+%d,%%eax\n",main_reg*4);
	fprintf(asf,"movl _cpu_tmpf,%%edx\n");
	fprintf(asf,"rorl $17,%%edx\n");
	fprintf(asf,"%s%c _cpu_r(,%%edi,4),%s\n",op,
		sizes[main_size],x86ax[main_size]
	);
	flags();
	c2x();
	adjzero_edxr();
	fprintf(asf,"mov%c %s,_cpu_r+%d\n",sizes[main_size],
		x86ax[main_size],main_reg*4
	);
	ret();
}

static void opx_adec(char*op){
	save_ecx();
	usereg();
	fprintf(asf,"movl _cpu_r+32(,%%edi,4),%%esi\n");
	fprintf(asf,"%s%%esi\n",sizedec[main_size]);
	prep_address();
	fprintf(asf,"call *_%s_rp%c(,%%ebp,4)\n",sysname,sizes[main_size]);
	fprintf(asf,"movl %%eax,%%ecx\n");
	fprintf(asf,"movl %%esi,%%edx\n");
	fprintf(asf,"movl _cpu_r+%d,%%esi\n",(main_reg*4)+32);
	fprintf(asf,"%s%%esi\n",sizedec[main_size]);
	prep_address();
	fprintf(asf,"movl %%esi,_cpu_r+%d\n",(main_reg*4)+32);
	fprintf(asf,"movl %%edx,_cpu_r+32(,%%edi,4)\n");
	fprintf(asf,"call *_%s_rp%c(,%%ebp,4)\n",sysname,sizes[main_size]);
	fprintf(asf,"movl _cpu_tmpf,%%edx\n");
	fprintf(asf,"rorl $17,%%edx\n");
	fprintf(asf,"%s%c %s,%s\n",op,sizes[main_size],
		x86cx[main_size],x86ax[main_size]
	);
	flags();
	c2x();
	adjzero_edxr();
	fprintf(asf,"call *_%s_wp%c(,%%ebp,4)\n",sysname,sizes[main_size]);
	restore_ecx();
	ret();
}

static void i_addx_dreg(void){opx_dreg("adc");}
static void i_addx_adec(void){opx_adec("adc");}
static void i_subx_dreg(void){opx_dreg("sbb");}
static void i_subx_adec(void){opx_adec("sbb");}

static void opbcd_dreg(char*op,char*adjust){
	usereg();
	fprintf(asf,"movl _cpu_r+%d,%%eax\n",main_reg*4);
	fprintf(asf,"movl _cpu_tmpf,%%edx\n");
	fprintf(asf,"rorl $17,%%edx\n");
	fprintf(asf,"%sb _cpu_r(,%%edi,4),%%al\n",op);
	fprintf(asf,"%s\n",adjust);
	flags();
	c2x();
	adjzero_edxr();
	fprintf(asf,"movb %%al,_cpu_r+%d\n",main_reg*4);
	ret();
}

static void opbcd_adec(char*op,char*adjust){
	save_ecx();
	usereg();
	fprintf(asf,"movl _cpu_r+32(,%%edi,4),%%esi\n");
	fprintf(asf,"decl %%esi\n");
	prep_address();
	fprintf(asf,"call *_%s_rpb(,%%ebp,4)\n",sysname);
	fprintf(asf,"movl %%eax,%%ecx\n");
	fprintf(asf,"movl %%esi,%%edx\n");
	fprintf(asf,"movl _cpu_r+%d,%%esi\n",(main_reg*4)+32);
	fprintf(asf,"decl %%esi\n");
	prep_address();
	fprintf(asf,"movl %%esi,_cpu_r+%d\n",(main_reg*4)+32);
	fprintf(asf,"movl %%edx,_cpu_r+32(,%%edi,4)\n");
	fprintf(asf,"call *_%s_rpb(,%%ebp,4)\n",sysname);
	fprintf(asf,"movl _cpu_tmpf,%%edx\n");
	fprintf(asf,"rorl $17,%%edx\n");
	fprintf(asf,"%sb %%cl,%%al\n",op);
	fprintf(asf,"%s\n",adjust);
	flags();
	c2x();
	adjzero_edxr();
	fprintf(asf,"call *_%s_wpb(,%%ebp,4)\n",sysname);
	restore_ecx();
	ret();
}

/* each called 8 times */
static void i_abcd_dreg(void){opbcd_dreg("adc","daa");}
static void i_abcd_adec(void){opbcd_adec("adc","daa");}
static void i_sbcd_dreg(void){opbcd_dreg("sbb","das");}
static void i_sbcd_adec(void){opbcd_adec("sbb","das");}

static void i_movep_mem2reg(void){
	usereg();
	fprintf(asf,"movswl (%%ebx),%%esi\n");
	fprintf(asf,"addl $2,%%ebx\n");
	fprintf(asf,"addl _cpu_r+32(,%%edi,4),%%esi\n");
	prep_address();
	fprintf(asf,"call *_%s_rpb(,%%ebp,4)\n",sysname);
	fprintf(asf,"movb %%al,%%dh\n");
	fprintf(asf,"addl $2,%%esi\n");
	fprintf(asf,"call *_%s_rpb(,%%ebp,4)\n",sysname);
	fprintf(asf,"movb %%al,%%dl\n");
	if(main_size==2){
		fprintf(asf,"movw %%dx,_cpu_r+%d\n",main_reg*4);
	}else{
		fprintf(asf,"addl $2,%%esi\n");
		fprintf(asf,"shll $16,%%edx\n");
		fprintf(asf,"call *_%s_rpb(,%%ebp,4)\n",sysname);
		fprintf(asf,"movb %%al,%%dh\n");
		fprintf(asf,"addl $2,%%esi\n");
		fprintf(asf,"call *_%s_rpb(,%%ebp,4)\n",sysname);
		fprintf(asf,"movb %%al,%%dl\n");
		fprintf(asf,"movl %%edx,_cpu_r+%d\n",main_reg*4);
	}
	ret();
}

static void i_movep_reg2mem(void){
	usereg();
	fprintf(asf,"movswl (%%ebx),%%esi\n");
	fprintf(asf,"addl $2,%%ebx\n");
	fprintf(asf,"addl _cpu_r+32(,%%edi,4),%%esi\n");
	prep_address();
	fprintf(asf,"movl _cpu_r+%d,%%edx\n",main_reg*4);
	if(main_size==4)fprintf(asf,"roll $16,%%edx\n");
	fprintf(asf,"movb %%dh,%%al\n");
	fprintf(asf,"call *_%s_wpb(,%%ebp,4)\n",sysname);
	fprintf(asf,"addl $2,%%esi\n");
	fprintf(asf,"movb %%dl,%%al\n");
	fprintf(asf,"call *_%s_wpb(,%%ebp,4)\n",sysname);
	if(main_size==4){
		fprintf(asf,"addl $2,%%esi\n");
		fprintf(asf,"roll $16,%%edx\n");
		fprintf(asf,"movb %%dh,%%al\n");
		fprintf(asf,"call *_%s_wpb(,%%ebp,4)\n",sysname);
		fprintf(asf,"addl $2,%%esi\n");
		fprintf(asf,"movb %%dl,%%al\n");
		fprintf(asf,"call *_%s_wpb(,%%ebp,4)\n",sysname);
	}
	ret();
}

static void i_chk(void){
	int myline;
	myline=linenum;
	linenum++;
	selective_usereg();
	main_ea_load();
	fprintf(asf,"movl _cpu_r+%d,%%edx\n",main_reg);
	fprintf(asf,"cmp%c $0,%s\n",sizes[main_size],x86dx[main_size]);
	fprintf(asf,"movw $0x080,_cpu_tmpf\n");
	fprintf(asf,"jl l%d\n",myline);
	fprintf(asf,"cmp%c %s,%s\n",sizes[main_size],
		x86ax[main_size],x86dx[main_size]
	);
	fprintf(asf,"movw $0x000,_cpu_tmpf\n");
	fprintf(asf,"jg l%d\n",myline);
	ret();
	fprintf(asf,"l%d:",myline);myline++;
	exception(6);
	ret();
}

static void i_illegal(void){
	exception(4);
	ret();
}

static void i_aline(void){
	fprintf(asf,"subl $2,%%ebx\n");
	exception(10);
	ret();
}

static void i_fline(void){
	fprintf(asf,"subl $2,%%ebx\n");
	exception(11);
	ret();
}

/****************************************************************************
** DECODE ROUTINES
****************************************************************************/

static int rproc[0x10000];
static byte unique[0x10000];
int c;

static int test(int n,int m,int op){
	int t;
	if((n&m)!=op)return 0;
	for(t=op&0xF000;t<n;t++){
		if((!unique[t])&&((t&m)==(n&m))){
			rproc[n]=t;
			return 2;
		}
	}
	unique[n]=(m>>16)&1;
	rproc[n]=n;
/*	fprintf(asf,".align 4\n%c%03X:\n",((n>>12)&0xF)+'K',n&0xFFF);*/
	fprintf(asf,"%c%03X:\n",((n>>12)&0xF)+'K',n&0xFFF);
	return 1;
}

static void idef(int n,int mask,int op,void(*(proc))(void)){
	if(c)return;
	c=test(n,mask,op);
	if(c==1)proc();
}

static void eadef_all(int n,int m,int op,void(*(proc))(void)){
	if(c)return;
	main_eamode=dreg;idef(n,m|0x38,op|0x00,proc);
	main_eamode=areg;idef(n,m|0x38,op|0x08,proc);
	main_eamode=aind;idef(n,m|0x38,op|0x10,proc);
	main_eamode=ainc;idef(n,m|0x38,op|0x18,proc);
	main_eamode=adec;idef(n,m|0x38,op|0x20,proc);
	main_eamode=adsp;idef(n,m|0x38,op|0x28,proc);
	main_eamode=axdp;idef(n,m|0x38,op|0x30,proc);
	main_eamode=absw;idef(n,m|0x3F,op|0x38,proc);
	main_eamode=absl;idef(n,m|0x3F,op|0x39,proc);
	main_eamode=pcdp;idef(n,m|0x3F,op|0x3A,proc);
	main_eamode=pcxd;idef(n,m|0x3F,op|0x3B,proc);
	main_eamode=immd;idef(n,m|0x3F,op|0x3C,proc);
}

static void eadef_all_nobyteaddress(int n,int m,int op,void(*(proc))(void)){
	if(c)return;
	main_eamode=dreg;idef(n,m|0x38,op|0x00,proc);
	if(main_size!=1){main_eamode=areg;idef(n,m|0x38,op|0x08,proc);}
	main_eamode=aind;idef(n,m|0x38,op|0x10,proc);
	main_eamode=ainc;idef(n,m|0x38,op|0x18,proc);
	main_eamode=adec;idef(n,m|0x38,op|0x20,proc);
	main_eamode=adsp;idef(n,m|0x38,op|0x28,proc);
	main_eamode=axdp;idef(n,m|0x38,op|0x30,proc);
	main_eamode=absw;idef(n,m|0x3F,op|0x38,proc);
	main_eamode=absl;idef(n,m|0x3F,op|0x39,proc);
	main_eamode=pcdp;idef(n,m|0x3F,op|0x3A,proc);
	main_eamode=pcxd;idef(n,m|0x3F,op|0x3B,proc);
	main_eamode=immd;idef(n,m|0x3F,op|0x3C,proc);
}

static void eadef_data(int n,int m,int op,void(*(proc))(void)){
	if(c)return;
	main_eamode=dreg;idef(n,m|0x38,op|0x00,proc);
	main_eamode=aind;idef(n,m|0x38,op|0x10,proc);
	main_eamode=ainc;idef(n,m|0x38,op|0x18,proc);
	main_eamode=adec;idef(n,m|0x38,op|0x20,proc);
	main_eamode=adsp;idef(n,m|0x38,op|0x28,proc);
	main_eamode=axdp;idef(n,m|0x38,op|0x30,proc);
	main_eamode=absw;idef(n,m|0x3F,op|0x38,proc);
	main_eamode=absl;idef(n,m|0x3F,op|0x39,proc);
	main_eamode=pcdp;idef(n,m|0x3F,op|0x3A,proc);
	main_eamode=pcxd;idef(n,m|0x3F,op|0x3B,proc);
	main_eamode=immd;idef(n,m|0x3F,op|0x3C,proc);
}

static void eadef_alterable_nobyteaddress(int n,int m,int op,void(*(proc))(void)){
	if(c)return;
	main_eamode=dreg;idef(n,m|0x38,op|0x00,proc);
	if(main_size!=1){main_eamode=areg;idef(n,m|0x38,op|0x08,proc);}
	main_eamode=aind;idef(n,m|0x38,op|0x10,proc);
	main_eamode=ainc;idef(n,m|0x38,op|0x18,proc);
	main_eamode=adec;idef(n,m|0x38,op|0x20,proc);
	main_eamode=adsp;idef(n,m|0x38,op|0x28,proc);
	main_eamode=axdp;idef(n,m|0x38,op|0x30,proc);
	main_eamode=absw;idef(n,m|0x3F,op|0x38,proc);
	main_eamode=absl;idef(n,m|0x3F,op|0x39,proc);
}

static void eadef_data_alterable(int n,int m,int op,void(*(proc))(void)){
	if(c)return;
	main_eamode=dreg;idef(n,m|0x38,op|0x00,proc);
	main_eamode=aind;idef(n,m|0x38,op|0x10,proc);
	main_eamode=ainc;idef(n,m|0x38,op|0x18,proc);
	main_eamode=adec;idef(n,m|0x38,op|0x20,proc);
	main_eamode=adsp;idef(n,m|0x38,op|0x28,proc);
	main_eamode=axdp;idef(n,m|0x38,op|0x30,proc);
	main_eamode=absw;idef(n,m|0x3F,op|0x38,proc);
	main_eamode=absl;idef(n,m|0x3F,op|0x39,proc);
}

static void eadef_memory_alterable(int n,int m,int op,void(*(proc))(void)){
	if(c)return;
	main_eamode=aind;idef(n,m|0x38,op|0x10,proc);
	main_eamode=ainc;idef(n,m|0x38,op|0x18,proc);
	main_eamode=adec;idef(n,m|0x38,op|0x20,proc);
	main_eamode=adsp;idef(n,m|0x38,op|0x28,proc);
	main_eamode=axdp;idef(n,m|0x38,op|0x30,proc);
	main_eamode=absw;idef(n,m|0x3F,op|0x38,proc);
	main_eamode=absl;idef(n,m|0x3F,op|0x39,proc);
}

static void eadef_control(int n,int m,int op,void(*(proc))(void)){
	if(c)return;
	main_eamode=aind;idef(n,m|0x38,op|0x10,proc);
	main_eamode=adsp;idef(n,m|0x38,op|0x28,proc);
	main_eamode=axdp;idef(n,m|0x38,op|0x30,proc);
	main_eamode=absw;idef(n,m|0x3F,op|0x38,proc);
	main_eamode=absl;idef(n,m|0x3F,op|0x39,proc);
	main_eamode=pcdp;idef(n,m|0x3F,op|0x3A,proc);
	main_eamode=pcxd;idef(n,m|0x3F,op|0x3B,proc);
}

static void eadef_control_alterable(int n,int m,int op,void(*(proc))(void)){
	if(c)return;
	main_eamode=aind;idef(n,m|0x38,op|0x10,proc);
	main_eamode=adsp;idef(n,m|0x38,op|0x28,proc);
	main_eamode=axdp;idef(n,m|0x38,op|0x30,proc);
	main_eamode=absw;idef(n,m|0x3F,op|0x38,proc);
	main_eamode=absl;idef(n,m|0x3F,op|0x39,proc);
}

static void defmove(int majorop,int n){
	if(c)return;
	main_destmode=dreg;for(main_reg=0;main_reg<8;main_reg++){eadef_all(n,0xFFC0,(0x000|majorop)|(main_reg<<9),i_move);}
	main_destmode=aind;for(main_reg=0;main_reg<8;main_reg++){eadef_all(n,0xFFC0,(0x080|majorop)|(main_reg<<9),i_move);}
	main_destmode=ainc;for(main_reg=0;main_reg<8;main_reg++){eadef_all(n,0xFFC0,(0x0C0|majorop)|(main_reg<<9),i_move);}
	main_destmode=adec;for(main_reg=0;main_reg<8;main_reg++){eadef_all(n,0xFFC0,(0x100|majorop)|(main_reg<<9),i_move);}
	main_destmode=adsp;for(main_reg=0;main_reg<8;main_reg++){eadef_all(n,0xFFC0,(0x140|majorop)|(main_reg<<9),i_move);}
	main_destmode=axdp;for(main_reg=0;main_reg<8;main_reg++){eadef_all(n,0xFFC0,(0x180|majorop)|(main_reg<<9),i_move);}
	main_destmode=absw;eadef_all(n,0xFFC0,(0x1C0|majorop),i_move);
	main_destmode=absl;eadef_all(n,0xFFC0,(0x3C0|majorop),i_move);
}

static void decode0(int n){c=0;
	for(sizedef=0;sizedef<3;sizedef++){main_size=1<<sizedef;eadef_data_alterable(n,0xFFC0,0x0000|(sizedef<<6),i_ori);}
	for(sizedef=0;sizedef<3;sizedef++){main_size=1<<sizedef;eadef_data_alterable(n,0xFFC0,0x0200|(sizedef<<6),i_andi);}
	for(sizedef=0;sizedef<3;sizedef++){main_size=1<<sizedef;eadef_data_alterable(n,0xFFC0,0x0400|(sizedef<<6),i_subi);}
	for(sizedef=0;sizedef<3;sizedef++){main_size=1<<sizedef;eadef_data_alterable(n,0xFFC0,0x0600|(sizedef<<6),i_addi);}
	for(sizedef=0;sizedef<3;sizedef++){main_size=1<<sizedef;eadef_data_alterable(n,0xFFC0,0x0A00|(sizedef<<6),i_eori);}
	for(sizedef=0;sizedef<3;sizedef++){main_size=1<<sizedef;eadef_data_alterable(n,0xFFC0,0x0C00|(sizedef<<6),i_cmpi);}
	for(main_cc=0;main_cc<4;main_cc++){
		if(!main_cc)   eadef_data(n,0xFFC0,0x0800|(main_cc<<6),i_bitop_imm);
		else eadef_data_alterable(n,0xFFC0,0x0800|(main_cc<<6),i_bitop_imm);
	}
	for(main_reg=0;main_reg<8;main_reg++)for(main_cc=0;main_cc<4;main_cc++){
		if(!main_cc)   eadef_data(n,0xFFC0,0x0100|(main_cc<<6)|(main_reg<<9),i_bitop_reg);
		else eadef_data_alterable(n,0xFFC0,0x0100|(main_cc<<6)|(main_reg<<9),i_bitop_reg);
	}
	idef(n,0xFFFF,0x003C,i_ori_ccr);
	idef(n,0xFFFF,0x023C,i_andi_ccr);
	idef(n,0xFFFF,0x0A3C,i_eori_ccr);
	idef(n,0xFFFF,0x007C,i_ori_sr);
	idef(n,0xFFFF,0x027C,i_andi_sr);
	idef(n,0xFFFF,0x0A7C,i_eori_sr);
	for(main_reg=0;main_reg<8;main_reg++)for(main_size=2;main_size<=4;main_size+=2){
		idef(n,0xFFF8,0x0108|((main_size&4)<<4)|(main_reg<<9),i_movep_mem2reg);
		idef(n,0xFFF8,0x0188|((main_size&4)<<4)|(main_reg<<9),i_movep_reg2mem);
	}
}
static void decode1(int n){c=0;
	main_size=1;
	defmove(0x1000,n);
}
static void decode2(int n){c=0;
	main_size=4;
	defmove(0x2000,n);
	for(main_reg=0;main_reg<8;main_reg++)eadef_all(n,0xFFC0,0x2040|(main_reg<<9),i_movea);
}
static void decode3(int n){c=0;
	main_size=2;
	defmove(0x3000,n);
	for(main_reg=0;main_reg<8;main_reg++)eadef_all(n,0xFFC0,0x3040|(main_reg<<9),i_movea);
}
static void decode4(int n){c=0;
	eadef_data_alterable(n,0xFFC0,0x40C0,i_move_from_sr);
	eadef_data_alterable(n,0xFFC0,0x42C0,i_move_from_ccr);
	eadef_data(n,0xFFC0,0x44C0,i_move_to_ccr);
	eadef_data(n,0xFFC0,0x46C0,i_move_to_sr);
	eadef_control(n,0xFFC0,0x4EC0,i_jmp);
	eadef_control(n,0xFFC0,0x4E80,i_jsr);
	for(main_reg=0;main_reg<8;main_reg++)eadef_control(n,0xFFC0,0x41C0|(main_reg<<9),i_lea);
	main_size=2;for(main_reg=0;main_reg<8;main_reg++)eadef_data(n,0xFFC0,0x4100|(main_reg<<9),i_chk);
	eadef_control(n,0xFFC0,0x4840,i_pea);
	for(sizedef=0;sizedef<3;sizedef++){main_size=1<<sizedef;eadef_data_alterable(n,0xFFC0,0x4200|(sizedef<<6),i_clr);}
	for(sizedef=0;sizedef<3;sizedef++){main_size=1<<sizedef;eadef_data_alterable(n,0xFFC0,0x4A00|(sizedef<<6),i_tst);}
        idef(n,0xFFFF,0x4E70,i_reset);
	idef(n,0xFFFF,0x4E71,i_nop);
	idef(n,0xFFFF,0x4E72,i_stop);
	idef(n,0xFFFF,0x4E73,i_rte);
	idef(n,0xFFFF,0x4E75,i_rts);
        idef(n,0xffff,0x4e76,i_wait0);
        idef(n,0xffff,0x4e78,i_wait1);
	main_dr=0;for(sizedef=0;sizedef<2;sizedef++){main_size=1<<(sizedef+1);eadef_control_alterable(n,0xFFC0,0x4880|(main_dr<<10)|(sizedef<<6),i_movem_control);}
	main_dr=1;for(sizedef=0;sizedef<2;sizedef++){main_size=1<<(sizedef+1);eadef_control          (n,0xFFC0,0x4880|(main_dr<<10)|(sizedef<<6),i_movem_control);}
	for(sizedef=0;sizedef<2;sizedef++){main_size=1<<(sizedef+1);idef(n,0xFFF8,0x4C98|(sizedef<<6),i_movem_postinc);}
	for(sizedef=0;sizedef<2;sizedef++){main_size=1<<(sizedef+1);idef(n,0xFFF8,0x48A0|(sizedef<<6),i_movem_predec);}
	idef(n,0xFFF8,0x4E50,i_link);
	idef(n,0xFFF8,0x4E58,i_unlk);
	idef(n,0xFFF0,0x4E40,i_trap);
	idef(n,0xFFF8,0x4E60,i_move2usp);
	idef(n,0xFFF8,0x4E68,i_movefusp);
	idef(n,0xFFF8,0x4840,i_swap);
	idef(n,0xFFF8,0x4880,i_extbw);
	idef(n,0xFFF8,0x48C0,i_extwl);
	for(sizedef=0;sizedef<3;sizedef++){main_size=1<<sizedef;eadef_data_alterable(n,0xFFC0,0x4000|(sizedef<<6),i_negx);}
	for(sizedef=0;sizedef<3;sizedef++){main_size=1<<sizedef;eadef_data_alterable(n,0xFFC0,0x4400|(sizedef<<6),i_neg);}
	for(sizedef=0;sizedef<3;sizedef++){main_size=1<<sizedef;eadef_data_alterable(n,0xFFC0,0x4600|(sizedef<<6),i_not);}
	eadef_data_alterable(n,0xFFC0,0x4800,i_nbcd);
	eadef_data_alterable(n,0xFFC0,0x4AC0,i_tas);
	idef(n,0xFFFF,0x4AFC,i_illegal);
}
static void decode5(int n){c=0;
	for(sizedef=0;sizedef<3;sizedef++){main_size=1<<sizedef;for(main_qv=0;main_qv<8;main_qv++){eadef_alterable_nobyteaddress(n,0xFFC0,0x5000|(sizedef<<6)|(main_qv<<9),i_addq);}}
	for(sizedef=0;sizedef<3;sizedef++){main_size=1<<sizedef;for(main_qv=0;main_qv<8;main_qv++){eadef_alterable_nobyteaddress(n,0xFFC0,0x5100|(sizedef<<6)|(main_qv<<9),i_subq);}}
	for(main_cc=0x2;main_cc<=0xF;main_cc++)idef(n,0xFFF8,0x50C8|(main_cc<<8),i_dbcc);
	idef(n,0xFFF8,0x51C8,i_dbra);
	main_size=1;for(main_cc=0x0;main_cc<=0xF;main_cc++)eadef_data_alterable(n,0xFFC0,0x50C0|(main_cc<<8),i_scc);
}
static void decode6(int n){c=0;
	idef(n,0x1FFFF,0x6000,i_bra_w);
	idef(n,0x1FFFF,0x6100,i_bsr_w);
	for(main_cc=0x2;main_cc<=0xF;main_cc++){
		idef(n,0x1FFFF,0x6000|(main_cc<<8),i_bcc_w);
	}
	idef(n,0x0FF00,0x6000,i_bra_b);
	idef(n,0x0FF00,0x6100,i_bsr_b);
	for(main_cc=0x2;main_cc<=0xF;main_cc++){
		idef(n,0x0FF00,0x6000|(main_cc<<8),i_bcc_b);
	}
}
static void decode7(int n){c=0;
	for(main_reg=0;main_reg<8;main_reg++)idef(n,0xFF00,0x7000|(main_reg<<9),i_moveq);
}
static void decode8(int n){c=0;
	for(sizedef=0;sizedef<3;sizedef++){main_size=1<<sizedef;for(main_reg=0;main_reg<8;main_reg++){eadef_data             (n,0xFFC0,0x8000|(main_reg<<9)|(sizedef<<6),i_or_dn );}}
	for(sizedef=0;sizedef<3;sizedef++){main_size=1<<sizedef;for(main_reg=0;main_reg<8;main_reg++){eadef_memory_alterable (n,0xFFC0,0x8100|(main_reg<<9)|(sizedef<<6),i_or_ea );}}
	for(main_cc=0;main_cc<2;main_cc++)for(main_reg=0;main_reg<8;main_reg++)eadef_data(n,0xFFC0,0x80C0|(main_reg<<9)|(main_cc<<8),i_div);
	main_size=1;
	for(main_reg=0;main_reg<8;main_reg++)idef(n,0xFFF8,0x8100|(main_reg<<9),i_sbcd_dreg);
	for(main_reg=0;main_reg<8;main_reg++)idef(n,0xFFF8,0x8108|(main_reg<<9),i_sbcd_adec);
}
static void decode9(int n){c=0;
	for(sizedef=0;sizedef<3;sizedef++){main_size=1<<sizedef;for(main_reg=0;main_reg<8;main_reg++){eadef_all_nobyteaddress(n,0xFFC0,0x9000|(main_reg<<9)|(sizedef<<6),i_sub_dn);}}
	for(sizedef=0;sizedef<3;sizedef++){main_size=1<<sizedef;for(main_reg=0;main_reg<8;main_reg++){eadef_memory_alterable (n,0xFFC0,0x9100|(main_reg<<9)|(sizedef<<6),i_sub_ea);}}
	main_size=2;for(main_reg=0;main_reg<8;main_reg++)eadef_all(n,0xFFC0,0x90C0|(main_reg<<9),i_suba);
	main_size=4;for(main_reg=0;main_reg<8;main_reg++)eadef_all(n,0xFFC0,0x91C0|(main_reg<<9),i_suba);
	for(sizedef=0;sizedef<3;sizedef++){main_size=1<<sizedef;for(main_reg=0;main_reg<8;main_reg++)idef(n,0xFFF8,0x9100|(main_reg<<9)|(sizedef<<6),i_subx_dreg);}
	for(sizedef=0;sizedef<3;sizedef++){main_size=1<<sizedef;for(main_reg=0;main_reg<8;main_reg++)idef(n,0xFFF8,0x9108|(main_reg<<9)|(sizedef<<6),i_subx_adec);}
}
static void decodeA(int n){c=0;
	idef(n,0xF000,0xA000,i_aline);
}
static void decodeB(int n){c=0;
	for(sizedef=0;sizedef<3;sizedef++){main_size=1<<sizedef;for(main_reg=0;main_reg<8;main_reg++){eadef_all_nobyteaddress(n,0xFFC0,0xB000|(main_reg<<9)|(sizedef<<6),i_cmp_dn);}}
	for(sizedef=0;sizedef<3;sizedef++){main_size=1<<sizedef;for(main_reg=0;main_reg<8;main_reg++){eadef_data_alterable   (n,0xFFC0,0xB100|(main_reg<<9)|(sizedef<<6),i_eor_ea);}}
	main_size=2;for(main_reg=0;main_reg<8;main_reg++)eadef_all(n,0xFFC0,0xB0C0|(main_reg<<9),i_cmpa);
	main_size=4;for(main_reg=0;main_reg<8;main_reg++)eadef_all(n,0xFFC0,0xB1C0|(main_reg<<9),i_cmpa);
	for(sizedef=0;sizedef<3;sizedef++){main_size=1<<sizedef;for(main_reg=0;main_reg<8;main_reg++)idef(n,0xFFF8,0xB108|(sizedef<<6)|(main_reg<<9),i_cmpm);}
}
static void decodeC(int n){c=0;
	for(sizedef=0;sizedef<3;sizedef++){main_size=1<<sizedef;for(main_reg=0;main_reg<8;main_reg++){eadef_data             (n,0xFFC0,0xC000|(main_reg<<9)|(sizedef<<6),i_and_dn);}}
	for(sizedef=0;sizedef<3;sizedef++){main_size=1<<sizedef;for(main_reg=0;main_reg<8;main_reg++){eadef_memory_alterable (n,0xFFC0,0xC100|(main_reg<<9)|(sizedef<<6),i_and_ea);}}
	for(main_cc=0;main_cc<2;main_cc++)for(main_reg=0;main_reg<8;main_reg++)eadef_data(n,0xFFC0,0xC0C0|(main_reg<<9)|(main_cc<<8),i_mul);
	main_dr=0 ;main_ir=0 ;for(main_reg=0;main_reg<8;main_reg++)idef(n,0xFFF8,0xC140|(main_reg<<9),i_exg);
	main_dr=32;main_ir=32;for(main_reg=0;main_reg<8;main_reg++)idef(n,0xFFF8,0xC148|(main_reg<<9),i_exg);
	main_dr=0 ;main_ir=32;for(main_reg=0;main_reg<8;main_reg++)idef(n,0xFFF8,0xC188|(main_reg<<9),i_exg);
	main_size=1;
	for(main_reg=0;main_reg<8;main_reg++)idef(n,0xFFF8,0xC100|(main_reg<<9),i_abcd_dreg);
	for(main_reg=0;main_reg<8;main_reg++)idef(n,0xFFF8,0xC108|(main_reg<<9),i_abcd_adec);
}
static void decodeD(int n){c=0;
	for(sizedef=0;sizedef<3;sizedef++){main_size=1<<sizedef;for(main_reg=0;main_reg<8;main_reg++){eadef_all_nobyteaddress(n,0xFFC0,0xD000|(main_reg<<9)|(sizedef<<6),i_add_dn);}}
	for(sizedef=0;sizedef<3;sizedef++){main_size=1<<sizedef;for(main_reg=0;main_reg<8;main_reg++){eadef_memory_alterable (n,0xFFC0,0xD100|(main_reg<<9)|(sizedef<<6),i_add_ea);}}
	main_size=2;for(main_reg=0;main_reg<8;main_reg++)eadef_all(n,0xFFC0,0xD0C0|(main_reg<<9),i_adda);
	main_size=4;for(main_reg=0;main_reg<8;main_reg++)eadef_all(n,0xFFC0,0xD1C0|(main_reg<<9),i_adda);
	for(sizedef=0;sizedef<3;sizedef++){main_size=1<<sizedef;for(main_reg=0;main_reg<8;main_reg++)idef(n,0xFFF8,0xD100|(main_reg<<9)|(sizedef<<6),i_addx_dreg);}
	for(sizedef=0;sizedef<3;sizedef++){main_size=1<<sizedef;for(main_reg=0;main_reg<8;main_reg++)idef(n,0xFFF8,0xD108|(main_reg<<9)|(sizedef<<6),i_addx_adec);}
}
static void decodeE(int n){c=0;
	for(main_reg=0;main_reg<8;main_reg++)
	for(main_dr=0;main_dr<2;main_dr++)
	for(main_ir=0;main_ir<2;main_ir++)
	for(sizedef=0;sizedef<3;sizedef++){
		main_size=1<<sizedef;
		idef(n,0xFFF8,0xE000|(main_reg<<9)|(main_dr<<8)|(sizedef<<6)|(main_ir<<5),i_asx_reg);
		idef(n,0xFFF8,0xE008|(main_reg<<9)|(main_dr<<8)|(sizedef<<6)|(main_ir<<5),i_lsx_reg);
		idef(n,0xFFF8,0xE010|(main_reg<<9)|(main_dr<<8)|(sizedef<<6)|(main_ir<<5),i_rxx_reg);
		idef(n,0xFFF8,0xE018|(main_reg<<9)|(main_dr<<8)|(sizedef<<6)|(main_ir<<5),i_rox_reg);
	}
	for(main_dr=0;main_dr<2;main_dr++){
		eadef_memory_alterable(n,0xFFC0,0xE0C0|(main_dr<<8),i_asx_mem);
		eadef_memory_alterable(n,0xFFC0,0xE2C0|(main_dr<<8),i_lsx_mem);
		eadef_memory_alterable(n,0xFFC0,0xE4C0|(main_dr<<8),i_rxx_mem);
		eadef_memory_alterable(n,0xFFC0,0xE6C0|(main_dr<<8),i_rox_mem);
	}
}
static void decodeF(int n){c=0;
        idef(n,0xF000,0xF000,i_fline);
}

static void(*(decodetable[16]))(int n)={
decode0,decode1,decode2,decode3,decode4,decode5,decode6,decode7,
decode8,decode9,decodeA,decodeB,decodeC,decodeD,decodeE,decodeF};

int main(int argc,char**argv){
	int i,j,last,rl,bank,bankend;
	void(*(decoderoutine))(int n);
#ifdef APU2X
	apu2x=1;
#else
	apu2x=0;
#endif
	fprintf(stderr,"STARSCREAM Version "VERSION"\n");
	if(argc!=4){
		fprintf(stderr,"usage: %s outputfile machinename bankbits\n",argv[0]);
		return 1;
	}
	strcpy(sysname,argv[2]);
	bankbits=atoi(argv[3]);
	if((bankbits>24)||(bankbits==0)){
		fprintf(stderr,"can't use '%s' bank bits!\n",argv[3]);
		return 1;
	}
	linenum=0;
	fflush(stdout);
	fflush(stderr);
	asf=fopen(argv[1],"w");
	if(asf==NULL){
		fprintf(stderr,"Bad news: couldn't create '%s'\n",argv[1]);
		fflush(stderr);
		return 1;
	}
	prefixes();
	for(i=0;i<0x10000;i++)rproc[i]=-1;
	fprintf(stderr,"Bank ");
	fflush(stderr);
	for(bank=0;bank<=0xF;bank++){
		fprintf(stderr,"%X",bank);
		fflush(stderr);
		bankend=(bank+1)<<12;
		decoderoutine=decodetable[bank];
		for(i=bank<<12;i<bankend;i++)decoderoutine(i);
	}
	fprintf(stderr," done\nBuilding table...");
	fprintf(asf,".data\n.align 4\n.lcomm SPTR,262144\n");
	fprintf(asf,".data\n.align 4\nSPCK:\n");
	last=-2;
	rl=0;
	for(i=0;i<0x10000;i++){
		j=rproc[i];
		if(j==last){
			if(rl==256){
				if(last==-1)fprintf(asf,".long %d\n",(rl-1)<<24);
				else fprintf(asf,".long %c%03X-A+%d\n",((last>>12)&0xF)+'K',last&0xFFF,(rl-1)<<24);
				rl=1;
			}else{
				rl++;
			}
		}else{
			if(rl){
				if(last==-1)fprintf(asf,".long %d\n",(rl-1)<<24);
				else fprintf(asf,".long %c%03X-A+%d\n",((last>>12)&0xF)+'K',last&0xFFF,(rl-1)<<24);
			}
			rl=1;
			last=j;
		}
	}
	if(last==-1)fprintf(asf,".long %d\n",(rl-1)<<24);
	else fprintf(asf,".long %c%03X-A+%d\n",((last>>12)&0xF)+'K',last&0xFFF,(rl-1)<<24);
	suffixes();
	fprintf(stderr," done\n");
	fclose(asf);
	return 0;
}
