//----------------------------------------------------------------------------//
// RbPSe - Sony Play Station Emulator                                         //
// Copyright (C) notes:                                                       //
//   Code: RomikB, 2002-2003                                                  //
//        <romikb@mail.ru>                                                    //
//                                                                            //
// This program is free software; you can redistribute it and/or              //
// modify it under the terms of the GNU General Public License                //
// as published by the Free Software Foundation; either version 2             //
// of the License, or (at your option) any later version.                     //
//                                                                            //
// This program is distributed in the hope that it will be useful,            //
// but WITHOUT ANY WARRANTY; without even the implied warranty of             //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the              //
// GNU General Public License for more details.                               //
//                                                                            //
// You should have received a copy of the GNU General Public License          //
// along with this program; if not, write to the Free Software                //
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.//
//----------------------------------------------------------------------------//
#include "emulator.h"
#include "interface.h"
#include "plugin.h"
#include <stdio.h>
#include <string.h>

RbPSe_Emulator Emulator;
RbPSe_Interface *Interface;

/******************************************************************************/
#define		STORAGE_SYSTEM		( 0x0020 )
#define		STORAGE_REGISTERS	( 0x0020 )
#define		STORAGE_PRIVATE		( 0x00e0 )
#define		STORAGE_INTERFACE	( 0x0300 )
#define		STORAGE_ALL			( STORAGE_SYSTEM + STORAGE_REGISTERS + STORAGE_PRIVATE + STORAGE_INTERFACE )
// | 0x00000000 - 0x0000001f | -> | 0x80 - 0xfc | -> System ( Fixed )
// | 0x00000020 - 0x0000003f | -> | 0x00 - 0x7c | -> Registers ( Fixed )
// | 0x00000040 - 0x0000011f | -> | 0x00000080 - 0x000003fc | -> Private ( Float )
// | 0x00000120 - 0x0000041f | -> | 0x00000400 - 0x00000ffc | -> Interface ( Float && Info )
/******************************************************************************/
static void *StorageData[ STORAGE_ALL ];
static char *StorageName[ STORAGE_ALL ];
static void *StorageCopy[ STORAGE_ALL ];
static long StoragePrivate;
static long StorageInterface;
/******************************************************************************/
static void *section;
/******************************************************************************/
/* Storage Functions */
static long StorageGet(RbPSe_Storage *storage)
{
	long i;
	for( i = 0 ; i < STORAGE_ALL ; i++ )
	{
		if( StorageName[ i ] == 0 ) continue;
		if( strcmp( storage -> name , StorageName[ i ] ) ) continue;
		printf("%li: %s\n", i - STORAGE_SYSTEM , StorageName[ i ] );//?
		storage -> func = StorageData[ i ];
		storage -> place = i - STORAGE_SYSTEM;
		return 0;
	}
	return -1;
}

static long StorageSet(RbPSe_Storage *storage)
{
	switch( storage->place )
	{
	case 0:
		storage->place = STORAGE_REGISTERS + STORAGE_PRIVATE + StorageInterface++;
		break;
	case 1:
		storage->place = STORAGE_REGISTERS + StoragePrivate++;
		break;
	default:
		break;
	}
	StorageData[ storage -> place + STORAGE_SYSTEM ] = storage -> func;
	StorageName[ storage -> place + STORAGE_SYSTEM ] = storage -> name;
	StorageCopy[ storage -> place + STORAGE_SYSTEM ] = storage -> func;
	return 0;
}

static void StorageInit()
{
	long i;
	for( i = 0 ; i < STORAGE_ALL ; i++ )
		StorageData[ i ] = StorageName[ i ] = StorageCopy[ i ] = 0;
	Emulator.Storage = StorageData + 0x20;
	Emulator.Storage2 = StorageCopy + 0x20;
	Emulator.StorageSet = StorageSet;
	Emulator.StorageGet = StorageGet;
	StoragePrivate = 0;
	StorageInterface = 0;
}

/******************************************************************************/
/* Periodic Function */
//void UserSync(void)
//{
//	Interface -> LeaveCritical(&section);
//	printf("Periodic Function\n");
//	Interface -> EnterCritical(&section);
//	if(...)...
//}

/******************************************************************************/

void EmulatorFunction(void *data)
{
	RbPSe_Storage Storage;
	printf("Entering thread...\n");//?
	Interface -> EnterCritical(&section);
	printf("Executing thread...\n");//?
	Storage.name = "___boot";
	if( StorageGet( &Storage ) )
	{
		printf("[ RBPSE ]Error: Can't find \"___boot\" function\n");
	}
	else
	{
        Storage.func();//Exit only on error//
	}
	Interface -> LeaveCritical(&section);
	printf("Ending thread...\n");//?
	Interface -> Fatal();
	Interface -> ExitTh(0);
}

/******************************************************************************/

static long EmulatorQuit()
{
	return 0;
}

static long EmulatorClose()
{
	return 0;
}

static long EmulatorStop()
{
	long i;
	Interface -> EnterCritical(&section);
	for( i = 0 ; i < Emulator.PluginsCount ; i++ )
	{
		if( Emulator.Plugins[i] -> Stop() )
		{
			return -1;
		}
	}
	return 0;
}

static long EmulatorStart()
{
	long i;
	for(i=0;i<Emulator.PluginsCount;i++)
	{
		if(Emulator.Plugins[i] -> Start())
		{
			return -1;
		}
	}
	Interface -> LeaveCritical(&section);
	return 0;
}

static long EmulatorOpen()
{
	long i;
	for(i=0;i<Emulator.PluginsCount;i++)
	{
		if(Emulator.Plugins[i] -> Open())
		{
			return -1;
		}
	}
	Interface -> InitCritical(&section);
	Interface -> EnterCritical(&section);
	Interface -> CreateTh( EmulatorFunction , 0 );
	return 0;
}

static char *Digits[] = {
	"00" , "01" , "02" , "03" , "04" , "05" , "06" , "07" , "08" , "09" ,
	"10" , "11" , "12" , "13" , "14" , "15" , "16" , "17" , "18" , "19" ,
	"20" , "21" , "22" , "23" , "24" , "25" , "26" , "27" , "28" , "29" ,
	"30" , "31"
};

static long EmulatorInit()
{
	char tmpstr[256];
	int i,j;
	void *hplugin,*hfunc;
	RbPSe_Plugin *plugin;
//	RbPSe_Storage Storage;
//
	StorageInit();
//
//	Storage.name = "___sync";
//	Storage.func = UserSync;
//	Storage.place = 0;
//	StorageSet(&Storage);
//
	for( i=0 , j=0 ; i< MAX_PLUGINS ; i++ )
	{
		if(!strcmp(Interface->GetConfigStr("RbPSe.cfg","plugins",Digits[i],"",tmpstr),""))
		{
			continue;
		}
		hplugin = Interface -> PluginLoad(tmpstr);
		if(!hplugin)
		{
			printf("Error loading plugin: %s\n",tmpstr);
			continue;
		}
		hfunc = Interface -> PluginData(hplugin,RbPSePluginLoadName);
		if(!hfunc)
		{
			printf("Error plugin damaged or incorrect: %s\n",tmpstr);
			continue;
		}
		plugin = ((RbPSePluginLoadType)hfunc)(&Emulator);
		if(!plugin)
		{
			printf("Error loadtime in: %s\n",tmpstr);
			continue;
		}
		if( plugin -> Init() )
		{
			printf("Error plugin init: %s\n",tmpstr);
			continue;
		}
		Emulator.hPlugins[j] = hplugin;
		Emulator.Plugins[j++] = plugin;
	}
	Emulator.PluginsCount = j;
	return 0;
}

/******************************************************************************/

int main(int argc, char **argv)
{
	long CurEmulatorState = EMULATOR_STATE_QUIT;
	if(!(Interface = RbPSeInterfaceLoad(&Emulator))) return -1;
	if(Interface -> Init(argc, argv)) return -2;
	Emulator.Interface = Interface;
	for(;;)
	{
		switch(Interface -> Exec(&Emulator, CurEmulatorState))
		{
        case EMULATOR_STATE_ERROR:
            break;
		case EMULATOR_STATE_INIT:
			if(EmulatorInit())
			{
				CurEmulatorState = EMULATOR_STATE_ERROR;
			}
			else
			{
				CurEmulatorState = EMULATOR_STATE_INIT;
			}
			continue;
		case EMULATOR_STATE_OPEN:
			if(EmulatorOpen())
			{
				CurEmulatorState = EMULATOR_STATE_ERROR;
			}
			else
			{
				CurEmulatorState = EMULATOR_STATE_OPEN;
			}
			continue;
		case EMULATOR_STATE_START:
			if(EmulatorStart())
			{
				CurEmulatorState = EMULATOR_STATE_ERROR;
			}
			else
			{
				CurEmulatorState = EMULATOR_STATE_START;
			}
			continue;
		case EMULATOR_STATE_STOP:
			if(EmulatorStop())
			{
				CurEmulatorState = EMULATOR_STATE_ERROR;
			}
			else
			{
				CurEmulatorState = EMULATOR_STATE_STOP;
			}
			continue;
		case EMULATOR_STATE_CLOSE:
			if(EmulatorClose())
			{
				CurEmulatorState = EMULATOR_STATE_ERROR;
			}
			else
			{
				CurEmulatorState = EMULATOR_STATE_CLOSE;
			}
			continue;
		case EMULATOR_STATE_QUIT:
			if(EmulatorQuit())
			{
				CurEmulatorState = EMULATOR_STATE_ERROR;
			}
			else
			{
				CurEmulatorState = EMULATOR_STATE_QUIT;
			}
			continue;
		default:
			break;
		}
		break;
	}
	Interface -> Quit();
	return 0;
}
