#include "emulator.h"
#include "interface.h"
#include <windows.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

//data
static RbPSe_Interface Interface;
static RbPSe_Emulator *Emulator;
static char ProfileDir[256];
static char PluginsDir[256];
static char MapsDir[256];

//code
static void print_error_info(void)
{
	printf("RbPSe - Sony Play Station Emulator (www.rbpse.sf.net)\n"
		"Syntax Error! Use \"--help\" for usage information\n"
	);
}

static void print_help_info(void)
{
	printf("RbPSe - Sony Play Station Emulator (www.rbpse.sf.net)\n"
		"Version 0.45 Usage:\n\tRbPSe [Option] [Advanced Options]\n"
		"Options:\n"
		"        --help                  Print Help Screen\n"
		"        --about                 Print About Information\n"
		"        --cd device             Execute CD device\n"
		"        --iso filename          Execute ISO image\n"
		"        --file filename         Execute PSX-EXE file\n"
		"        --cdfile                Execute PSX-EXE file on CD device\n"
		"        --isofile               Execute PSX-EXE file on ISO image\n"
		"Advanced Options:\n"
		"        --maps dir              Use \"dir\" as Maps dirextory\n"
		"        --profile dir           Use \"dir\" as Profile directory\n"
		"        --plugins dir           Use \"dir\" as Plugins directory\n"
//		"        --start type            One of { init , open , start }\n"
	);
}

static void print_about_info(void)
{
    printf("RbPSe - Sony Play Station Emulator (www.rbpse.sf.net)\n"
		"Version:\n"
		"        0.50(.0) 2004/02/01\n"
		"Developers:\n"
		"        RomikB aka Roman Zakharchuk\n"
		"Testers:\n"
		"        Organic\n"
		"        Lynx\n"
		"Thanks to:\n"
		"        Organic,Lynx,Jeksik and of cause Ilka\n"
	);
}

static long MapOpen( char *name, long size )
{
	char map_name[256];
	strcat( strcpy( map_name, MapsDir ), ( strrchr( name, '\\' ) ? : name ) );
	sprintf( map_name + strlen(map_name), ".%li.map", size );
	return (long)fopen( map_name, "r" );
}
//
//static char GetDirSymbol()
//{
//	return '\\';
//}

static inline long long __rdtsc()
{
	long long tmp;
	__asm__  __volatile__ ( "rdtsc" : "=A" (tmp) );
	return tmp;
}

static long long GetStepClock(long val)
{
	long long tmp = __rdtsc();
	Sleep(val * 1000);
	return __rdtsc() - tmp;
}

static void QuitCritical(void **section)
{
	DeleteCriticalSection(*section);
	free(*section);
}

static void LeaveCritical(void **section)
{
	LeaveCriticalSection(*section);
}

static void EnterCritical(void **section)
{
	EnterCriticalSection(*section);
}

static void InitCritical(void **section)
{
	*section = malloc(sizeof(CRITICAL_SECTION));
	InitializeCriticalSection(*section);
}

static void ExitTh(long exitcode)
{
	ExitThread(exitcode);
}

static void *CreateTh(void *func, void *data)
{
	long id;
	return CreateThread(NULL, 0, func, data, 0, &id);
}

static char *GetConfigStr(char *file, char *section, char *key, char *defvalue, char *buffer)
{
	char tmpstr[256];
	GetPrivateProfileString(section,key,defvalue,buffer,256,
 			strcat(strcat(strcpy(tmpstr,ProfileDir),"\\"),file));
	return buffer;
}

static void *PluginData(void *plugin, char *func_name)
{
	return GetProcAddress(plugin,func_name);
}

static void *PluginLoad(char *plugin_name)
{
	char tmpstr[1024];
	void *hlib;
	hlib = LoadLibrary(strcat(strcat(strcpy(tmpstr,PluginsDir),"\\"),plugin_name));
    if(!hlib) printf(".%li.",GetLastError());
	return hlib;
}

static long NeedEmulatorState = EMULATOR_STATE_QUIT;
static long ReqEmulatorState = EMULATOR_STATE_QUIT;
static long MessageHandling = 1;

static char szClassName[] = "RbPSeWindowClass";
static HWND hWindow;

static LRESULT CALLBACK WindowProcedure(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch(message)
	{
		case WM_DESTROY:
			NeedEmulatorState = ReqEmulatorState = EMULATOR_STATE_ERROR;
			PostQuitMessage(0);
			break;
		case WM_KEYDOWN:
			if( (lParam >> 16) == 1 )
			{
				NeedEmulatorState = ReqEmulatorState = EMULATOR_STATE_ERROR;
				PostQuitMessage(0);
			}
			break;
		default:
			return DefWindowProc(hWnd,message,wParam,lParam);
	}
	return 0;
}

static void IntFatal(void)//need to complete
{
	MessageBox(hWindow, "Fatal Error: Exiting" , "Fatal Error!" , MB_OK );
	NeedEmulatorState = EMULATOR_STATE_ERROR;
	ReqEmulatorState = EMULATOR_STATE_ERROR;
	MessageHandling = 0;
	PostMessage(hWindow,WM_MOUSEMOVE,0,0);
}

static void IntQuit()
{
	DestroyWindow(hWindow);
}

static long IntExec(RbPSe_Emulator *Emulator, long CurEmulatorState)
{
	MSG msg;
//Error Handling
	if( CurEmulatorState == EMULATOR_STATE_ERROR ) return EMULATOR_STATE_ERROR;
//Selfdecision Handling
	if( CurEmulatorState != ReqEmulatorState ) return EMULATOR_STATE_ERROR;
//Chain Handling -- End
	if( ReqEmulatorState == NeedEmulatorState )
	{
		while( MessageHandling )
		{
			if( GetMessage(&msg, NULL, 0, 0) ) DispatchMessage(&msg);
			else MessageHandling = 0;
		}
		MessageHandling = 1;
	}
//Chain Handling -- Next
	else
	{
		if( ReqEmulatorState == EMULATOR_STATE_QUIT ) ReqEmulatorState = EMULATOR_STATE_INIT;
		else ReqEmulatorState++;
	}
	return ReqEmulatorState;
}

static long IntInit(int argc, char **argv)
{
	int i;
	WNDCLASSEX wincl;
////Proccess Args
//Options
	Emulator -> SourceType = SOURCE_TYPE_NONE;
	Emulator -> SourceName = 0;
	Emulator -> SourceDevice = 0;
	ProfileDir[0] = 0;
	PluginsDir[0] = 0;
	MapsDir[0] = 0;
	if( argc == 1 )
	{
		;
	}
	else
	if( argc == 2 )
	{
		if(!strcmp(argv[1],"--about"))
		{
			print_about_info();
			return -1;
		}
		else
		if(!strcmp(argv[1],"--help"))
		{
			print_help_info();
			return -1;
		}
		else
		{
			print_error_info();
			return -2;
		}
	}
	else
	{
		i = 1;
		while( i < argc )
		{
			if((i + 1) == argc )
			{
				print_help_info();
				return -2;
			}
			if(!strcmp(argv[i],"--cd"))
            {
				Emulator -> SourceType = SOURCE_TYPE_CD;
				Emulator -> SourceDevice = malloc(strlen(argv[i+1])+1);
				strcpy(Emulator -> SourceDevice, argv[i+1]);
				NeedEmulatorState = EMULATOR_STATE_START;
			}
			else
			if(!strcmp(argv[i],"--iso"))
			{
				Emulator -> SourceType = SOURCE_TYPE_ISO;
				Emulator -> SourceDevice = malloc(strlen(argv[i+1])+1);
				strcpy(Emulator -> SourceDevice, argv[i+1]);
				NeedEmulatorState = EMULATOR_STATE_START;
			}
			else
			if(!strcmp(argv[i],"--file"))
			{
				Emulator -> SourceType = SOURCE_TYPE_FILE;
				if( argv[i+1][1] == ':' )
				{
					Emulator -> SourceName = malloc(strlen(argv[i+1])+1);
					strcpy(Emulator -> SourceName, argv[i+1]);
				}
				else
				{
					Emulator -> SourceName = malloc(256);
					getcwd( Emulator -> SourceName , 256 );
					strcat( strcat( Emulator -> SourceName , "\\" ) , argv[i+1] );
				}
				NeedEmulatorState = EMULATOR_STATE_START;
			}
			else
			if(!strcmp(argv[i],"--profile"))
			{
				strcpy(ProfileDir, argv[i+1]);
			}
			else
			if(!strcmp(argv[i],"--plugins"))
			{
				strcpy(PluginsDir, argv[i+1]);
			}
			else
//			if(!strcmp(argv[i],"--start"))
//			{
//				if(!strcmp(argv[i+1],"init"))
//				{
//					NeedEmulatorState = EMULATOR_STATE_INIT;
//				}
//				else
//				if(!strcmp(argv[i+1],"open"))
//				{
//					NeedEmulatorState = EMULATOR_STATE_OPEN;
//				}
//				else
//				if(!strcmp(argv[i+1],"start"))
//				{
//					NeedEmulatorState = EMULATOR_STATE_START;
//				}
//				else
//				{
//					print_help_info();
//					return -2;
//				}
//			}
//			else
			{
				print_error_info();
				return -2;
			}
			i+=2;
		}
	}
//Default Advanced Options
	if(!ProfileDir[0])
	{
		if(!chdir("Profiles\\default"))
		{
			getcwd(ProfileDir,256);
			chdir("..\\..");
		}
		else
		{
			printf("Default Profile directory not found. Use \"--profile\".\n");
			return -3;
		}
	}
	if(!PluginsDir[0])
	{
		if(!chdir("Plugins"))
		{
			getcwd(PluginsDir,256);
			chdir("..");
		}
		else
		{
			printf("Default Plugins directory not found. Use \"--plugins\".\n");
			return -3;
		}
	}
	if(!MapsDir[0])
	{
		if(!chdir("Maps"))
		{
			getcwd(MapsDir,256);
			chdir("..");
		}
		else
		{
			printf("Default Maps directory not found. Use \"--maps\".\n");
			return -3;
		}
	}
//Print Info
	switch(Emulator -> SourceType)
	{
	case SOURCE_TYPE_NONE:
		printf("GUI Loading...");
		break;
	case SOURCE_TYPE_CD:
		printf("CD-ROM\t\t: %s", Emulator -> SourceDevice );
		if( Emulator -> SourceName ) printf("\\%s" , Emulator -> SourceName );
		break;
	case SOURCE_TYPE_ISO:
		printf("ISO image\t: %s", Emulator -> SourceDevice );
		if( Emulator -> SourceName ) printf("\\%s" , Emulator -> SourceName );
		break;
	case SOURCE_TYPE_FILE:
		printf("PSX-EXE\t\t: %s",Emulator -> SourceName);
		break;
	}
	printf("\n");
	printf("Plugins at\t: %s\nProfile at\t: %s\n", PluginsDir, ProfileDir);
////Window Menagement
    wincl.hInstance = 0;//hThisInstance;
    wincl.lpszClassName = szClassName;
    wincl.lpfnWndProc = WindowProcedure;
    wincl.style = CS_DBLCLKS;
    wincl.cbSize = sizeof(WNDCLASSEX);
    /* Use default icon and mouse-pointer */
    wincl.hIcon = LoadIcon(NULL, IDI_APPLICATION);
    wincl.hIconSm = LoadIcon(NULL, IDI_APPLICATION);
    wincl.hCursor = LoadCursor(NULL, IDC_ARROW);
    wincl.lpszMenuName = NULL;
    wincl.cbClsExtra = 0;
    wincl.cbWndExtra = 0;
    wincl.hbrBackground = (HBRUSH)GetStockObject(BLACK_BRUSH);//(HBRUSH)(COLOR_WINDOW + 1);
    if (!RegisterClassEx(&wincl)) return -4;
//
	hWindow = CreateWindowEx(0, szClassName, "RbPSe Emulator Windows",
		WS_OVERLAPPEDWINDOW, CW_USEDEFAULT, CW_USEDEFAULT,
		544, 375, HWND_DESKTOP, NULL, 0/*hThisInstance*/, NULL);
//
	ShowWindow(hWindow, SW_SHOWNORMAL);
	Emulator -> Display = hWindow;
	return 0;
}

RbPSe_Interface *RbPSeInterfaceLoad(RbPSe_Emulator *emulator)
{
	Emulator = emulator;
	Interface.Init = IntInit;
	Interface.Exec = IntExec;
	Interface.Quit = IntQuit;
	Interface.Fatal = IntFatal;
	Interface.PluginLoad = PluginLoad;
	Interface.PluginData = PluginData;
	Interface.GetConfigStr = GetConfigStr;
	Interface.CreateTh = CreateTh;
	Interface.ExitTh = ExitTh;
	Interface.InitCritical = InitCritical;
	Interface.EnterCritical = EnterCritical;
	Interface.LeaveCritical = LeaveCritical;
	Interface.QuitCritical = QuitCritical;
	Interface.GetStepClock = GetStepClock;
	Interface.MapOpen = MapOpen;
//	Interface.GetDirSymbol = GetDirSymbol;
	return &Interface;
}

BOOL APIENTRY DllMain (HINSTANCE hInst     /* Library instance handle. */ ,
                       DWORD reason        /* Reason this function is being called. */ ,
                       LPVOID reserved     /* Not used. */ )
{
	switch (reason)
	{
	case DLL_PROCESS_ATTACH:
		break;
	case DLL_PROCESS_DETACH:
		break;
	case DLL_THREAD_ATTACH:
		break;
	case DLL_THREAD_DETACH:
		break;
	}
	/* Returns TRUE on success, FALSE on failure */
	return TRUE;
}
