/*
 * A windows version of the RPN calculator
 */

#include <owl\owlpch.h>
#include <owl\applicat.h>
#include <owl\dialog.h>
#include <owl\framewin.h>
#include <iostream.h>
#include <iomanip.h>
#include <strstrea.h>
#include <stdlib.h>
#include <signal.h>
#include "button.h"
#include "wrpn.h"

const char* HELP_FILE = "wrpn.hlp";

					/* the application object */
class wrpn_app : public TApplication {
	public:
		wrpn_app(const char far* name) : TApplication(name) {}
		void InitMainWindow();
};

void
wrpn_app::InitMainWindow()
{
	TWindow* wrpn_win = new wrpn;
	wrpn_win->Attr.AccelTable = "wrpn";

	MainWindow = new TFrameWindow(0, Name, wrpn_win, TRUE);
	MainWindow->SetIcon(this, "wrpn");
	MainWindow->Attr.Style &= ~(WS_MAXIMIZEBOX | WS_THICKFRAME);
	MainWindow->AssignMenu("wrpn");
	EnableBWCC();// because of the graphics in wrpn.rc
}

int
OwlMain(int, char **)
{
	/*
	 * I'll take care of Floating Point Exceptions
	 */
	signal(SIGFPE, SIG_IGN);

	return wrpn_app("Reverse Polish Notation Calculator").Run();
}

					/* the dialog constructor */
wrpn::wrpn()
  : TWindow((TWindow*)0), TDialog(0, "wrpn")
{
	mode = FLOAT_MODE;
	complement = TWOS;
	decimal_places = 4;
	word_size = 32;
	carry_bit = 0;
	last_x = 0.0;

	for (int i=0; i<STACK_SIZE; i++)
		stack[i] = 0.0;
	for (i=0; i<REGISTER_SIZE; i++)
		sto[i] = 0.0;

	alt_bn = new alt_button(this, PB_ALT);
	ctrl_bn = new ctrl_button(this, PB_CTRL);

	bg = new TBitmap(*GetApplication(), BACKGRND);
	return;
}

DEFINE_RESPONSE_TABLE1(wrpn, TDialog)
	EV_COMMAND(CM_FILE_EXIT, cm_file_exit),
	EV_COMMAND(CM_EDIT_COPY, cm_edit_copy),
	EV_COMMAND(CM_EDIT_PASTE, cm_edit_paste),
	EV_COMMAND_ENABLE(CM_EDIT_PASTE, ce_edit_paste),
	EV_COMMAND(CM_VIEW_FLOAT, cm_view_float),
	EV_COMMAND(CM_VIEW_DECIMAL, cm_view_decimal),
	EV_COMMAND(CM_VIEW_HEX, cm_view_hex),
	EV_COMMAND(CM_VIEW_OCTAL, cm_view_octal),
	EV_COMMAND(CM_VIEW_BINARY, cm_view_binary),
	EV_COMMAND(CM_OPTIONS_8BIT, cm_options_8bit),
	EV_COMMAND(CM_OPTIONS_16BIT, cm_options_16bit),
	EV_COMMAND(CM_OPTIONS_32BIT, cm_options_32bit),
	EV_COMMAND(CM_OPTIONS_1S, cm_options_1s),
	EV_COMMAND(CM_OPTIONS_2S, cm_options_2s),
	EV_COMMAND(CM_OPTIONS_UNSGN, cm_options_unsgn),
	EV_COMMAND(CM_HELP_CONTENTS, cm_help_contents),
	EV_COMMAND(CM_HELP_SEARCH, cm_help_search),
	EV_COMMAND(CM_HELP_USING, cm_help_using),
	EV_COMMAND(CM_HELP_ABOUT, cm_help_about),
	EV_WM_PAINT,
	EV_WM_CTLCOLOR,
END_RESPONSE_TABLE;

/*
 * Handle all of the mouse and keyboard events here at one location.
 * Much nicer than a zillon response tables
 */
LRESULT
wrpn::EvCommand(UINT id, HWND ctrl, UINT notify)
{
	int key;

	if (ctrl != 0 && notify == BN_CLICKED) {
		key = id;
		/*
		 * This trick only works if you also have VK_CONTROL
		 * and VK_MENU defined in the Accelerator table
		 */
		if (GetKeyState(VK_CONTROL) & 0x8000)
			key = id + CTRL;
		if (GetKeyState(VK_MENU) & 0x8000)
			key = id + ALT;
		calc_key(key);
	}
					/* from the keyboard */
	else if (ctrl == 0 && notify == 1) {
		flash_button(id);
		calc_key(id);
	}

	return TDialog::EvCommand(id, ctrl, notify);
}

/*
 * The file menu... (such that it is)
 */

void
wrpn::cm_file_exit()
{
	CloseWindow(0);
}

/*
 * The edit menu...
 */

void
wrpn::cm_edit_copy()
{
	char *p, screen[SCREEN_SIZE+3];

	TClipboard& cb = OpenClipboard();
	if (!cb) {
		cerr << '\a';
		return;
	}
	cb.EmptyClipboard();

	HANDLE data = GlobalAlloc(GMEM_SHARE, SCREEN_SIZE+3);
	char far* data_p = (char far*)GlobalLock(data);

	::GetWindowText(GetDlgItem(DISPLAY), screen, SCREEN_SIZE+2);
	// zap leading spaces
	p = screen;
	while (*p++ == ' ')
		;
	strcpy(data_p, --p);

	GlobalUnlock(data);
	cb.SetClipboardData(CF_TEXT, data);

	cb.CloseClipboard();
	return;
}

void
wrpn::cm_edit_paste()
{
	int odd;
	char input[SCREEN_SIZE+1];
	double f;

	TClipboard& cb = OpenClipboard();
	if (!cb || !cb.IsClipboardFormatAvailable(CF_TEXT)) {
		cerr << '\a';
		return;
	}

	HANDLE data = GetClipboardData(CF_TEXT);
	char far* data_p = (char far*)GlobalLock(data);
	strncpy(input, data_p, SCREEN_SIZE);
	GlobalUnlock(data);
	/*
	 * since convert() doesn't do a sanity check,
	 * we'll do that here.  But, just as a warning.
	 */
	odd = 0;
	for (int i=0; i<strlen(input); i++) {
		switch (mode) {
			case DEC_MODE:
				if (!strchr(" \t\n-0123456789", input[i]))
					odd++;
				break;
			case FLOAT_MODE:
				if (!strchr(" \t\n-0123456789eE.", input[i]))
					odd++;
				break;
			case OCT_MODE:
				if (!strchr(" \t\n01234567", input[i]))
					odd++;
				break;
			case HEX_MODE:
				if (!strchr(" \t\n0123456789abcdefABCDEF", input[i]))
					odd++;
				break;
			case BIN_MODE:
				if (!strchr(" \t\n01", input[i]))
					odd++;
				break;
		}
	}
	if (odd)
		cerr << '\a';

	f = convert(input, mode);
	push(f);
	display(f, mode);

	cb.CloseClipboard();
	return;
}

void
wrpn::ce_edit_paste(TCommandEnabler& ce)
{
	TClipboard& cb = OpenClipboard();
	if (cb && cb.IsClipboardFormatAvailable(CF_TEXT))
		ce.Enable(1);
	else
		ce.Enable(0);
	cb.CloseClipboard();
	return;
}

/*
 * The view menu...
 */

void
wrpn::cm_view_float()
{
	HMENU hwnd = Parent->GetMenu();
	/*
	 * I'm using the check mark like a radio button
	 * each option is mutually exclusive
	 */
	CheckMenuItem(hwnd, CM_VIEW_FLOAT, MF_CHECKED);
	CheckMenuItem(hwnd, CM_VIEW_DECIMAL, MF_UNCHECKED);
	CheckMenuItem(hwnd, CM_VIEW_HEX, MF_UNCHECKED);
	CheckMenuItem(hwnd, CM_VIEW_OCTAL, MF_UNCHECKED);
	CheckMenuItem(hwnd, CM_VIEW_BINARY, MF_UNCHECKED);
	/*
	 * The first calc_key(0) terminates any pending input,
	 * the second one updates the display
	 */
	calc_key(0);
	mode = FLOAT_MODE;
	calc_key(0);
	return;
}

void
wrpn::cm_view_decimal()
{
	HMENU hwnd = Parent->GetMenu();
	CheckMenuItem(hwnd, CM_VIEW_FLOAT, MF_UNCHECKED);
	CheckMenuItem(hwnd, CM_VIEW_DECIMAL, MF_CHECKED);
	CheckMenuItem(hwnd, CM_VIEW_HEX, MF_UNCHECKED);
	CheckMenuItem(hwnd, CM_VIEW_OCTAL, MF_UNCHECKED);
	CheckMenuItem(hwnd, CM_VIEW_BINARY, MF_UNCHECKED);

	calc_key(0);
	mode = DEC_MODE;
	calc_key(0);
	return;
}

void
wrpn::cm_view_hex()
{
	HMENU hwnd = Parent->GetMenu();
	CheckMenuItem(hwnd, CM_VIEW_FLOAT, MF_UNCHECKED);
	CheckMenuItem(hwnd, CM_VIEW_DECIMAL, MF_UNCHECKED);
	CheckMenuItem(hwnd, CM_VIEW_HEX, MF_CHECKED);
	CheckMenuItem(hwnd, CM_VIEW_OCTAL, MF_UNCHECKED);
	CheckMenuItem(hwnd, CM_VIEW_BINARY, MF_UNCHECKED);

	calc_key(0);
	mode = HEX_MODE;
	calc_key(0);
	return;
}

void
wrpn::cm_view_octal()
{
	HMENU hwnd = Parent->GetMenu();
	CheckMenuItem(hwnd, CM_VIEW_FLOAT, MF_UNCHECKED);
	CheckMenuItem(hwnd, CM_VIEW_DECIMAL, MF_UNCHECKED);
	CheckMenuItem(hwnd, CM_VIEW_HEX, MF_UNCHECKED);
	CheckMenuItem(hwnd, CM_VIEW_OCTAL, MF_CHECKED);
	CheckMenuItem(hwnd, CM_VIEW_BINARY, MF_UNCHECKED);

	calc_key(0);
	mode = OCT_MODE;
	calc_key(0);
	return;
}

void
wrpn::cm_view_binary()
{
	HMENU hwnd = Parent->GetMenu();
	CheckMenuItem(hwnd, CM_VIEW_FLOAT, MF_UNCHECKED);
	CheckMenuItem(hwnd, CM_VIEW_DECIMAL, MF_UNCHECKED);
	CheckMenuItem(hwnd, CM_VIEW_HEX, MF_UNCHECKED);
	CheckMenuItem(hwnd, CM_VIEW_OCTAL, MF_UNCHECKED);
	CheckMenuItem(hwnd, CM_VIEW_BINARY, MF_CHECKED);

	calc_key(0);
	mode = BIN_MODE;
	calc_key(0);
	return;
}

/*
 * The options menu....
 */

void
wrpn::cm_options_8bit()
{
	HMENU hwnd = Parent->GetMenu();
	CheckMenuItem(hwnd, CM_OPTIONS_8BIT, MF_CHECKED);
	CheckMenuItem(hwnd, CM_OPTIONS_16BIT, MF_UNCHECKED);
	CheckMenuItem(hwnd, CM_OPTIONS_32BIT, MF_UNCHECKED);

	calc_key(0);
	word_size = 8;
	calc_key(0);
	return;
}

void
wrpn::cm_options_16bit()
{
	HMENU hwnd = Parent->GetMenu();
	CheckMenuItem(hwnd, CM_OPTIONS_8BIT, MF_UNCHECKED);
	CheckMenuItem(hwnd, CM_OPTIONS_16BIT, MF_CHECKED);
	CheckMenuItem(hwnd, CM_OPTIONS_32BIT, MF_UNCHECKED);

	calc_key(0);
	word_size = 16;
	calc_key(0);
	return;
}

void
wrpn::cm_options_32bit()
{
	HMENU hwnd = Parent->GetMenu();
	CheckMenuItem(hwnd, CM_OPTIONS_8BIT, MF_UNCHECKED);
	CheckMenuItem(hwnd, CM_OPTIONS_16BIT, MF_UNCHECKED);
	CheckMenuItem(hwnd, CM_OPTIONS_32BIT, MF_CHECKED);

	calc_key(0);
	word_size = 32;
	calc_key(0);
	return;
}

void
wrpn::cm_options_1s()
{
	HMENU hwnd = Parent->GetMenu();
	CheckMenuItem(hwnd, CM_OPTIONS_1S, MF_CHECKED);
	CheckMenuItem(hwnd, CM_OPTIONS_2S, MF_UNCHECKED);
	CheckMenuItem(hwnd, CM_OPTIONS_UNSGN, MF_UNCHECKED);

	calc_key(0);
	complement = ONES;
	calc_key(0);
	return;
}

void
wrpn::cm_options_2s()
{
	HMENU hwnd = Parent->GetMenu();
	CheckMenuItem(hwnd, CM_OPTIONS_1S, MF_UNCHECKED);
	CheckMenuItem(hwnd, CM_OPTIONS_2S, MF_CHECKED);
	CheckMenuItem(hwnd, CM_OPTIONS_UNSGN, MF_UNCHECKED);

	calc_key(0);
	complement = TWOS;
	calc_key(0);
	return;
}

void
wrpn::cm_options_unsgn()
{
	HMENU hwnd = Parent->GetMenu();
	CheckMenuItem(hwnd, CM_OPTIONS_1S, MF_UNCHECKED);
	CheckMenuItem(hwnd, CM_OPTIONS_2S, MF_UNCHECKED);
	CheckMenuItem(hwnd, CM_OPTIONS_UNSGN, MF_CHECKED);

	calc_key(0);
	complement = UNSGN;
	calc_key(0);
	return;
}

/*
 * And last, the help menu...
 */

void
wrpn::cm_help_contents()
{
	WinHelp(HELP_FILE, HELP_CONTENTS, 0);
	return;
}

void
wrpn::cm_help_search()
{
	WinHelp(HELP_FILE, HELP_PARTIALKEY, 0);
	return;
}

void
wrpn::cm_help_using()
{
	WinHelp(HELP_FILE, HELP_HELPONHELP, 0);
	return;
}

void
wrpn::cm_help_about()
{
	TDialog(this, "about").Execute();
	return;
}

					/* the message response functions */
void
wrpn::EvPaint()
{
	DefaultProcessing();

	int x = TWindow::Attr.W -2;
	int y = TWindow::Attr.H -2;

	TClientDC winDC(*this);
	TMemoryDC memDC(winDC);
	memDC.SelectObject(*bg);
	/*
	 * Since I want the bitmap background to exactly overlay the
	 * dialog box, I have to use the "stretch" function since the
	 * size of the dialog is dependant on the font.
	 */
	winDC.StretchBlt(0, 0, x, y, memDC, 0, 0, bg->Width(), bg->Height(), SRCCOPY);
	calc_key(0);
	return;
}

/*
 * Change the way buttons are painted.  Otherwise white dots appear a the
 * corners of each button.
 */

HBRUSH
wrpn::EvCtlColor(HDC hDC, HWND hWndChild, UINT ctlType)
{
	if (ctlType == CTLCOLOR_BTN) {
			SetBkMode(hDC, TRANSPARENT);
			return (HBRUSH)GetStockObject(NULL_BRUSH);
	}
	return TDialog::EvCtlColor(hDC, hWndChild, ctlType);
}

/*
 * Display an important message in the "screen"
 */

void
wrpn::message(char *buf, int mode)
{
	char *flag = " hdob", screen[SCREEN_SIZE+3];

	ostrstream scr(screen, sizeof screen);
	scr << setw(SCREEN_SIZE) << buf << ' ' << flag[mode] << ends;

	::SetWindowText(GetDlgItem(DISPLAY), screen);
	return;
}

					/* flash a button */
void
wrpn::flash_button(char key)
{
	HWND button = GetDlgItem(key);

	if (button) {
		::SendMessage(button, BM_SETSTATE, 1, 0);
		SetDefaultId(UINT(key));

		for (long delay = 1; delay <= 50000L; ++delay)
			;

		::SendMessage(button, BM_SETSTATE, 0, 0);
	}
	return;
}

