/*********************************************************** {{{1 ***********
 *  Copyright (C) 2005 Kai G. Schwebke, upn.schwebke.com
 *  Copyright (C) 2005,2008  Martin Krischik (extensions for FX-602P)
 ****************************************************************************
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ****************************************************************************
 *  $Author: krischik $
 *
 *  $Revision: 461 $
 *  $Date: 2008-08-20 19:11:49 +0200 (Mi, 20 Aug 2008) $
 *
 *  $Id: BCDFloat.java 461 2008-08-20 17:11:49Z krischik $
 *  $HeadURL: https://uiq3.svn.sourceforge.net/svnroot/uiq3/tags/HP45-CLDC-2.0.1/Calculator/src/net/sourceforge/uiq3/bcd/BCDFloat.java $
 ************************************************************ }}}1 **********/

/**
 * @author "Martin Krischik" <krischik@users.sourceforge.net>
 * @model name="Uiq3.Calculator.Bcd"
 */
package net.sourceforge.uiq3.bcd;

/**
 * Arbitrary Precision Unnormalized Floating Point Arithmetic.
 * <p>
 * For the use in an FX-602 P Simulatort the precision is fixed at 14 decimal
 * digits.
 * </p>
 */
public final class BCDFloat
   {
   /**
    * String represenatiion for Numeric Type NAN
    * <dl>
    * <dt>Finite</dt>
    * <dd>Normal Real Number</dd>
    * <dt>NAN</dt>
    * <dd>Not a Number</dd>
    * <dt>Type_Positive_Infinity</dt>
    * <dd>+∞</dd>
    * <dt>Type_Negative_Infinity</dt>
    * <dd>-∞</dd>
    * </dl>
    */
   public static final String Type_NaN_Text = "NaN";

   /**
    * String representation for Numeric Type +∞
    * <dl>
    * <dt>Finite</dt>
    * <dd>Normal Real Number</dd>
    * <dt>NAN</dt>
    * <dd>Not a Number</dd>
    * <dt>Type_Positive_Infinity</dt>
    * <dd>+∞</dd>
    * <dt>Type_Negative_Infinity</dt>
    * <dd>-∞</dd>
    * </dl>
    */
   public static final String Type_Positive_Infinity_Text = "+Inf";

   /**
    * String representation for Numeric Type -∞
    * <dl>
    * <dt>Finite</dt>
    * <dd>Normal Real Number</dd>
    * <dt>NAN</dt>
    * <dd>Not a Number</dd>
    * <dt>Type_Positive_Infinity</dt>
    * <dd>+∞</dd>
    * <dt>Type_Negative_Infinity</dt>
    * <dd>-∞</dd>
    * </dl>
    */
   public static final String Type_Negative_Infinity_Text = "-Inf";

   /**
    * degree half circle constant
    */
   final static public BCDFloat Num_180 = new BCDFloat ("180");

   /**
    * radiant half circle constant
    */
   final static public BCDFloat Num_200 = new BCDFloat ("200");

   /**
    * 1 × 10<sup>100</sup>
    */
   final static public BCDFloat Num_1e100 = new BCDFloat ("1e100");

   /**
    * 1 × 10<sup>-100</sup>
    */
   final static public BCDFloat Num_1em100 = new BCDFloat ("1e-100");

   /**
    * 0.5
    */
   final static public BCDFloat Num_0_5 = new BCDFloat ("0.5");
   /**
    * 0
    */
   final static public BCDFloat Num_0 = new BCDFloat ("0");
   /**
    * 2
    */
   final static public BCDFloat Num_2 = new BCDFloat ("2");
   /**
    * 10
    */
   final static public BCDFloat Num_10 = new BCDFloat ("10");
   /**
    * 1
    */
   final static public BCDFloat Num_1 = new BCDFloat ("1");
   /**
    * Not a number
    */
   final static public BCDFloat NaN = new BCDFloat (BCDFloat.Type_NaN_Text);
   /**
    * π (18 digits)
    * <p>
    * Unlike the original BCDFloat from Kai G. Schwebke I use a fixed maximum
    * precision defined at compile time - so there is no need for truly
    * arbitrary irrational constants.
    * </p>
    */
   final static public BCDFloat Num_PI = new BCDFloat ("3.14159265358979324");
   /**
    * 0.5 π (18 digits)
    * <p>
    * Unlike the original BCDFloat from Kai G. Schwebke I use a fixed maximum
    * precision defined at compile time - so there is no need for truly
    * arbitrary irrational constants.
    * </p>
    */
   final static public BCDFloat Num_0_5_PI =
      new BCDFloat ("1.57079632679489662");
   /**
    * 2 π (18 digits)
    * <p>
    * Unlike the original BCDFloat from Kai G. Schwebke I use a fixed maximum
    * precision defined at compile time - so there is no need for truly
    * arbitrary irrational constants.
    * </p>
    */
   final static public BCDFloat Num_2_PI =
      new BCDFloat ("6.28318530717958648");
   /**
    * Euler's number (18 digits)
    * <p>
    * Unlike the original BCDFloat from Kai G. Schwebke I use a fixed maximum
    * precision defined at compile time - so there is no need for truly
    * arbitrary irrational constants.
    * </p>
    */
   final static public BCDFloat Num_e = new BCDFloat ("2.71828182845904526");

   /**
    * 1 divided by Euler's number (18 digits)
    * <p>
    * Unlike the original BCDFloat from Kai G. Schwebke I use a fixed maximum
    * precision defined at compile time - so there is no need for truly
    * arbitrary irrational constants.
    * </p>
    */
   final static public BCDFloat Num_1_div_e =
      new BCDFloat ("0.367879441171442322");
   /**
    * Log<sub>e</sub>10 (18 digits)
    * <p>
    * Unlike the original BCDFloat from Kai G. Schwebke I use a fixed maximum
    * precision defined at compile time - so there is no need for truly
    * arbitrary irrational constants.
    * </p>
    */
   final static public BCDFloat Num_log_e_10 =
      new BCDFloat ("2.30258509299404567");

   /**
    * Numeric Type
    * <dl>
    * <dt>Finite</dt>
    * <dd>Normal Real Number</dd>
    * <dt>NAN</dt>
    * <dd>Not a Number</dd>
    * <dt>Type_Positive_Infinity</dt>
    * <dd>+∞i</dd>
    * <dt>Type_Negative_Infinity</dt>
    * <dd>-∞</dd>
    * </dl>
    */
   static final int Type_Finite = 0;
   /**
    * Numeric Type
    * <dl>
    * <dt>Finite</dt>
    * <dd>Normal Real Number</dd>
    * <dt>NaN</dt>
    * <dd>Not a Number</dd>
    * <dt>Type_Positive_Infinity</dt>
    * <dd>+∞</dd>
    * <dt>Type_Negative_Infinity</dt>
    * <dd>-∞</dd>
    * </dl>
    */
   static final int Type_NaN = 1;
   /**
    * Numeric Type
    * <dl>
    * <dt>Finite</dt>
    * <dd>Normal Real Number</dd>
    * <dt>NaN</dt>
    * <dd>Not a Number</dd>
    * <dt>Type_Positive_Infinity</dt>
    * <dd>+∞</dd>
    * <dt>Type_Negative_Infinity</dt>
    * <dd>-∞</dd>
    * </dl>
    */
   static final int Type_Positive_Infinity = 2;
   /**
    * Numeric Type
    * <dl>
    * <dt>Finite</dt>
    * <dd>Normal Real Number</dd>
    * <dt>NaN</dt>
    * <dd>Not a Number</dd>
    * <dt>Type_Positive_Infinity</dt>
    * <dd>+∞</dd>
    * <dt>Type_Negative_Infinity</dt>
    * <dd>-∞</dd>
    * </dl>
    */
   static final int Type_Negative_Infinity = 3;

   /**
    * Max supported Exponent + 1
    */
   static final int Max_Exponent = 1000;

   /**
    * Max supported precision. Fixed at 18 digits because:
    * <ol>
    * <li>18 digits is more the enough for simulated calculators - most of
    * which have only 12 digits precision</li>
    * <li>18 digits mantissa can also be represented as a long.</li>
    * <ol>
    */
   static final int Max_Precision = 18;

   /**
    * coefficient - while a 18 digits number can also be represented as an long
    * an 18digits multiplication need a 36digits temporary and that is more then
    * a long can do.
    */
   private BCDInteger Coefficient;

   /**
    * current exponent - note that the exponent is based on the internal
    * representation where the radix point is right most. For example π is
    * 314159265358979324×10<sup>-17</sup>.
    */
   private int Exponent;

   /**
    * Numeric Type
    * <dl>
    * <dt>Finite</dt>
    * <dd>Normal Real Number</dd>
    * <dt>NaN</dt>
    * <dd>Not a Number</dd>
    * <dt>Type_Positive_Infinity</dt>
    * <dd>+∞dd>
    * <dt>Type_Negative_Infinity</dt>
    * <dd>-∞</dd>
    * </dl>
    */
   private int Type;

   /**
    * Create empty value
    * 
    * @model name="BCDFloat"
    */
   public BCDFloat ()
      {
      init ();

      return;
      } // BCDFloat

   /**
    * create new value from existing one.
    * 
    * @param Value
    * @model name="BCDFloat"
    */
   public BCDFloat (final BCDFloat Value)
      {
      Coefficient = new BCDInteger (Value.Coefficient);
      Exponent = Value.Exponent;
      Type = Value.Type;

      return;
      } // BCDFloat

   /**
    * create from java long
    * 
    * @param Value
    * @model name="BCDFloat"
    */
   public BCDFloat (final long Value)
      {
      init ();
      Coefficient.initFromInteger (Value);

      return;
      } // BCDFloat

   /**
    * create from string
    * 
    * @param Value
    * @model name="BCDFloat"
    */
   public BCDFloat (final String Value)
      {
      init ();
      init (Value);

      return;
      } // BCDFloat

   /**
    * |x|
    * 
    * @return abs value
    * @model name="abs"
    */
   public final BCDFloat abs ()
      {
      final BCDFloat Retval = new BCDFloat (this);

      Retval.Coefficient.Negative = false;

      return Retval;
      } // abs

   /**
    * x + y
    * 
    * @param y
    * @return
    * @model name="add"
    */
   public final BCDFloat add (final BCDFloat y)
      {
      final BCDFloat Retval = new BCDFloat ();

      if ((Type == BCDFloat.Type_NaN)
         || (y.Type == BCDFloat.Type_NaN)
         || ((Type == BCDFloat.Type_Positive_Infinity) && (y.Type == BCDFloat.Type_Negative_Infinity))
         || ((Type == BCDFloat.Type_Negative_Infinity) && (y.Type == BCDFloat.Type_Positive_Infinity)))
         {
         Retval.Type = BCDFloat.Type_NaN;
         }
      else if ((Type == BCDFloat.Type_Positive_Infinity)
         || (y.Type == BCDFloat.Type_Positive_Infinity))
         {
         Retval.Type = BCDFloat.Type_Positive_Infinity;
         }
      else if ((Type == BCDFloat.Type_Negative_Infinity)
         || (y.Type == BCDFloat.Type_Negative_Infinity))
         {
         Retval.Type = BCDFloat.Type_Negative_Infinity;
         } // if
      else
         {
         final int de = Math.abs (Exponent - y.Exponent);

         if (de >= 2 * BCDFloat.Max_Precision + 2)
            {
            if (Exponent > y.Exponent)
               {
               Retval.Coefficient = new BCDInteger (Coefficient);
               Retval.Exponent = Exponent;
               }
            else
               {
               Retval.Coefficient = new BCDInteger (y.Coefficient);
               Retval.Exponent = y.Exponent;
               } // if
            }
         else
            {
            BCDInteger coeff = Coefficient;
            BCDInteger coeffV = y.Coefficient;
            Retval.Exponent = y.Exponent;
            if (Exponent > y.Exponent)
               {
               coeff = Coefficient.shiftRight (de);
               } // if
            if (y.Exponent > Exponent)
               {
               coeffV = y.Coefficient.shiftRight (de);
               Retval.Exponent = Exponent;
               } // if
            Retval.Coefficient = coeff.add (coeffV);
            Retval.Round ();
            } // if
         }// if
      return Retval;
      } // add

   /**
    * arc cosine
    * 
    * @return arc cosine
    * @model name="arc_cos"
    */
   public final BCDFloat arc_cos ()
      {
      //System.out.println ("+ BCDFloat.arc_cos (" + To_Debug_String () + ")");

      final BCDFloat x = new BCDFloat (this);
      x.Coefficient.Negative = !x.Coefficient.Negative;
      BCDFloat Retval = x.arc_sin ();
      Retval = Retval.add (BCDFloat.Num_0_5_PI);

      //System.out.println ("- BCDFloat.arc_cos (" + Retval.To_Debug_String ()
      //+ ")");
      return Retval;
      } // arc_cos

   /**
    * arc cosine
    * 
    * @param Half_Circle
    *                value for a half circle
    * @return arc cosine
    * @model name="arc_cos"
    */
   public final BCDFloat arc_cos (final BCDFloat Half_Circle)
      {
      //System.out.println ("+ BCDFloat.arc_cos (" + To_Debug_String () + ","
      //+ Half_Circle.To_Debug_String () + ")");

      final BCDFloat Retval =
         arc_cos ().div (BCDFloat.Num_PI).mul (Half_Circle);

      //System.out.println ("- BCDFloat.arc_cos (" + Retval.To_Debug_String ()
      //+ ")");

      return Retval;
      } // arc_cos

   /**
    * area hyperbolic cosine: log<sub>e</sub> (x + √(x-1) × √(x+1))
    * 
    * @return Hyperbolic sine.
    * @model name="arc_cos_hyp"
    */
   public final BCDFloat arc_cos_hyp ()
      {
      final BCDFloat x_m_1 = sub (BCDFloat.Num_1);
      final BCDFloat x_p_1 = add (BCDFloat.Num_1);
      final BCDFloat root_x_m_1 = x_m_1.square_root ();
      final BCDFloat root_x_p_1 = x_p_1.square_root ();
      final BCDFloat mul = root_x_m_1.mul (root_x_p_1);
      final BCDFloat x_mul = add (mul);
      final BCDFloat retval = x_mul.log_e ();

      return retval;
      } // arc_cos_hyp

   /**
    * arc sine
    * 
    * @return arc sine
    * @model name="arc_sin"
    */
   public final BCDFloat arc_sin ()
      {
      //System.out.println ("+ BCDFloat.arc_sin (" + To_Debug_String () + ")");

      final BCDFloat x = new BCDFloat (this);
      boolean xNeg = false;

      if (x.Coefficient.Negative)
         {
         x.Coefficient.Negative = false;
         xNeg = true;
         } // if

      final BCDFloat xTest = x.sub (BCDFloat.Num_1);
      BCDFloat Retval = BCDFloat.Num_0_5_PI;

      if (!xTest.Is_Zero ())
         {
         if (!xTest.Coefficient.Negative) // x > 1
            {
            Retval = new BCDFloat ();
            Retval.Type = BCDFloat.Type_NaN;
            }
         else
            {
            Retval =
               x.div (BCDFloat.Num_1.sub (x.mul (x)).square_root ())
                  .arc_tan ();
            } // if
         } // if

      if (xNeg)
         {
         Retval.Coefficient.Negative = !Retval.Coefficient.Negative;
         } // if

      //System.out.println ("- BCDFloat.arc_sin (" + To_Debug_String () + ")");
      return Retval;
      }

   /**
    * arc sine
    * 
    * @param Half_Circle
    *                value for a half circle
    * @return arc sine
    * @model name="arc_sin"
    */
   public final BCDFloat arc_sin (final BCDFloat Half_Circle)
      {
      //System.out.println ("+ BCDFloat.arc_sin (" + To_Debug_String () + ","
      //+ Half_Circle.To_Debug_String () + ")");

      final BCDFloat Retval =
         arc_sin ().div (BCDFloat.Num_PI).mul (Half_Circle);

      //System.out.println ("- BCDFloat.arc_sin (" + Retval.To_Debug_String ()
      //+ ")");
      return Retval;
      } // arc_sin

   /**
    * area hyperbolic sine: log<sub>e</sub> (x + √(x² + 1))
    * 
    * @return Hyperbolic sine.
    * @model name="arc_sin_hyp"
    */
   public final BCDFloat arc_sin_hyp ()
      {
      final BCDFloat x_x = mul (this);
      final BCDFloat x_x_1 = x_x.add (BCDFloat.Num_1);
      final BCDFloat root = x_x_1.square_root ();
      final BCDFloat x_root = add (root);
      final BCDFloat retval = x_root.log_e ();

      return retval;
      } // arc_sin_hyp

   /**
    * arc tangent
    * 
    * @return arc tangent
    * @model name="arc_tan"
    */
   public final BCDFloat arc_tan ()
      {
      //System.out.println ("+ BCDFloat.arc_tan (" + To_Debug_String () + ")");

      BCDFloat x = new BCDFloat (this);
      boolean xNeg = false;

      if (x.Coefficient.Negative)
         {
         x.Coefficient.Negative = false;
         xNeg = true;
         } // if

      boolean gtOne = false;

      if (!x.sub (BCDFloat.Num_1).Coefficient.Negative) // x > 1
         {
         gtOne = true;
         x = BCDFloat.Num_1.div (x);
         } // if

      BCDFloat Retval = x;
      BCDFloat atn;

      for (int i = 1; i < BCDFloat.Max_Precision; ++i)
         {
         final BCDFloat atTan = Retval.tan ();
         atn =
            Retval.sub (atTan.sub (x).div (
               BCDFloat.Num_1.add (atTan.mul (atTan))));

         if (atn.sub (Retval).Is_Zero ()
            || (atn.Type != BCDFloat.Type_Finite)) // stop if increment is zero
            {
            atn = Retval;
            break;
            } // if

         Retval = atn;
         } // for

      if (gtOne)
         {
         Retval = BCDFloat.Num_0_5_PI.sub (Retval);
         }

      if (xNeg)
         {
         Retval.Coefficient.Negative = !Retval.Coefficient.Negative;
         } // if

      //System.out.println ("- BCDFloat.arc_tan (" + Retval.To_Debug_String ()
      //+ ")");

      return Retval;
      } // arc_tan

   /**
    * arc tangent
    * 
    * @param Half_Circle
    *                value for a half circle
    * @return arc tangent
    * @model name="arc_tan"
    */
   public final BCDFloat arc_tan (final BCDFloat Half_Circle)
      {
      //System.out.println ("+ BCDFloat.arc_tan (" + To_Debug_String () + ","
      //+ Half_Circle.To_Debug_String () + ")");

      final BCDFloat Retval =
         arc_tan ().div (BCDFloat.Num_PI).mul (Half_Circle);

      //System.out.println ("- BCDFloat.arc_tan (" + Retval.To_Debug_String ()
      //+ ")");

      return Retval;
      } // arc_tan

   /**
    * area hyperbolic tangent: log<sub>e</sub> ((1 + x) ÷ (1 - x)) / 2
    * 
    * @return Hyperbolic sine.
    * @model name="arc_tan_hyp"
    */
   public final BCDFloat arc_tan_hyp ()
      {
      final BCDFloat x_p_1 = BCDFloat.Num_1.add (this);
      final BCDFloat x_m_1 = BCDFloat.Num_1.sub (this);
      final BCDFloat diff = x_p_1.div (x_m_1);
      final BCDFloat ln_diff = diff.log_e ();
      final BCDFloat retval = ln_diff.div (BCDFloat.Num_2);

      return retval;
      } // arc_tan_hyp

   /**
    * @param Value
    * @return
    * @model name="Compare"
    */
   public final int Compare (final BCDFloat Value)
      {
      int retval;
      final BCDFloat Left = new BCDFloat (this);
      final BCDFloat Right = new BCDFloat (Value);

      Left.normalize ();
      Right.normalize ();

      if (Left.Exponent > Right.Exponent)
         {
         retval = +1;
         }
      else if (Left.Exponent < Right.Exponent)
         {
         retval = -1;
         }
      else
         {
         retval = Left.Coefficient.cmp (Right.Coefficient);
         } // if

      return retval;
      } // Compare

   /**
    * @return
    * @model name="cos"
    */
   public final BCDFloat cos ()
      {
      //System.out.println ("+ BCDFloat.cos (" + To_Debug_String () + ")");

      final BCDFloat Retval = add (BCDFloat.Num_0_5_PI).sin ();

      //System.out.println ("- BCDFloat.cos (" + Retval.To_Debug_String ()
      //+ ")");
      return Retval;
      } // cos

   /**
    * cosine
    * 
    * @param Half_Circle
    *                value for a half circle
    * @return cosine
    * @model name="cos"
    */
   public final BCDFloat cos (final BCDFloat Half_Circle)
      {
      //System.out.println ("+ BCDFloat.cos (" + To_Debug_String () + ","
      //+ Half_Circle.To_Debug_String () + ")");

      final BCDFloat Retval = mul (BCDFloat.Num_PI).div (Half_Circle).cos ();

      //System.out.println ("- BCDFloat.cos (" + Retval.To_Debug_String ()
      //+ ")");
      return Retval;
      } // cos

   /**
    * Hyperbolic cosine: (Exponent^x + Exponent^-x)/2
    * 
    * @return Hyperbolic cosine.
    * @model name="cos_hyp"
    */
   public final BCDFloat cos_hyp ()
      {
      final BCDFloat e_x = exp_e ();
      final BCDFloat e_m_x = BCDFloat.Num_0.sub (this).exp_e ();
      final BCDFloat sub = e_x.add (e_m_x);
      final BCDFloat retval = sub.div (BCDFloat.Num_2);

      return retval;
      } // cos_hyp

   /**
    * @param v
    * @return
    * @model name="div"
    */
   public final BCDFloat div (final BCDFloat v)
      {
      final BCDFloat q = new BCDFloat ();

      if ((Type == BCDFloat.Type_NaN)
         || (v.Type == BCDFloat.Type_NaN)
         || (Is_Zero () && v.Is_Zero ())
         || (((Type == BCDFloat.Type_Positive_Infinity) || (Type == BCDFloat.Type_Negative_Infinity)) && ((v.Type == BCDFloat.Type_Positive_Infinity) || (v.Type == BCDFloat.Type_Negative_Infinity))))
         {
         q.Type = BCDFloat.Type_NaN;
         return q;
         } // if

      if (((Type == BCDFloat.Type_Positive_Infinity) && (!v.Coefficient.Negative))
         || ((Type == BCDFloat.Type_Negative_Infinity) && (v.Coefficient.Negative)))
         {
         q.Type = BCDFloat.Type_Positive_Infinity;
         return q;
         } // if

      if (((Type == BCDFloat.Type_Negative_Infinity) && (!v.Coefficient.Negative))
         || ((Type == BCDFloat.Type_Positive_Infinity) && (v.Coefficient.Negative)))
         {
         q.Type = BCDFloat.Type_Negative_Infinity;
         return q;
         } // if

      if ((v.Type == BCDFloat.Type_Positive_Infinity)
         || (v.Type == BCDFloat.Type_Negative_Infinity))
         {
         return q;
         } // if

      if (v.Is_Zero ())
         {
         if (Coefficient.Negative)
            {
            q.Type = BCDFloat.Type_Negative_Infinity;
            }
         else
            {
            q.Type = BCDFloat.Type_Positive_Infinity;
            } // if
         return q;
         } // if

      final int idealExp = Exponent - v.Exponent;
      q.Exponent = idealExp;

      BCDInteger coeffU = new BCDInteger (Coefficient);
      coeffU.shorten ();
      final BCDInteger coeffV = new BCDInteger (v.Coefficient);
      coeffV.shorten ();

      if (coeffU.Num_Digits < BCDFloat.Max_Precision + 1 + coeffV.Num_Digits)
         {
         final int corr =
            BCDFloat.Max_Precision + 1 + coeffV.Num_Digits
               - coeffU.Num_Digits;
         coeffU = coeffU.shiftRight (corr);
         q.Exponent -= corr;
         } // if

      q.Coefficient = coeffU.div (coeffV);
      q.Round ();

      if (q.Exponent < idealExp)
         {
         q.normalize (idealExp);
         } // if

      if (q.Exponent > idealExp)
         {
         q.Coefficient = q.Coefficient.shiftRight (q.Exponent - idealExp);
         q.Exponent -= q.Exponent - idealExp;
         } // if

      return q;
      }

   /**
    * @return
    * @model name="exp_10"
    */
   public final BCDFloat exp_10 ()
      {
      return BCDFloat.Num_10.pow (this);
      } // exp_10

   /**
    * e<sup>x</sub>
    * 
    * @return
    * @model name="exp"
    */
   public final BCDFloat exp_e () // e^x based on expI using transformation
      {
      // to interval [0, ^ln 10] using:
      // x - k * ln 10 = r
      // 10^k * Exponent^r = Exponent^x

      final BCDFloat ln10 = BCDFloat.Num_log_e_10;
      final long k = div (ln10).To_Long ();

      if (k == 0)
         {
         return expI ();
         } // if

      BCDFloat r = sub ((new BCDFloat (k)).mul (ln10));

      r = r.expI ();

      r.Exponent += k;

      return r;
      }

   /**
    * Exponent^x using series expansion, usable around 0
    * 
    * @return
    * @model name="expI"
    */
   public final BCDFloat expI ()
      {
      BCDFloat rn;
      BCDFloat a = new BCDFloat (this);
      BCDFloat b = BCDFloat.Num_1;
      BCDFloat Retval = b.add (this); // 1+x

      // Taylor Series Expansion
      for (int i = 2; i < 10 * BCDFloat.Max_Precision; ++i) // max. number of iterations
         {
         a = a.mul (this);
         b = b.mul (new BCDFloat (i));
         rn = Retval.add (a.div (b));

         if (rn.sub (Retval).Is_Zero () || (rn.Type != BCDFloat.Type_Finite)) // stop if increment is zero
            {
            Retval = rn;
            break;
            } // if

         Retval = rn;
         } // for

      return Retval;
      }

   /**
    * Recompose a frac double from the given data.
    * 
    * @return a double
    * @model name="Compose"
    */
   public final BCDFloat Frac ()
      {
      //System.out.println ("+ BCDFloat.Compose_Frac ()");

      BCDFloat retval = new BCDFloat (this);

      retval.normalize ();

      final int Int_Digits =
         retval.Exponent + retval.Coefficient.Num_Digits >= 0
            ? retval.Exponent + retval.Coefficient.Num_Digits + 1 : 0;
      final int Frac_Digits =
         retval.Coefficient.Num_Digits > Int_Digits
            ? retval.Coefficient.Num_Digits - Int_Digits : 0;

      if (Frac_Digits <= 0)
         {
         retval = net.sourceforge.uiq3.bcd.BCDFloat.Num_0;
         }
      else if (Int_Digits > 0)
         {
         retval = retval.sub (retval.Integer ());
         } // if

      //System.out.println ("- BCDFloat.Compose_Frac ()");
      return retval;
      } // Compose

   /**
    * coefficient - while a 18 digits number can also be represented as an long
    * an 18digits multiplication need a 36digits temporary and that is more then
    * a long can do.
    */
   public long Get_Coefficient ()
      throws BCDError
      {
      if (Type == BCDFloat.Type_Finite)
         {
         return Coefficient.To_Long_Integer ();
         }
      else
         {
         throw new BCDError ("Not a finite number.");
         } // if
      } // Get_Coefficient

   /**
    * current exponent - note that the exponent is based on the internal
    * representation where the radix point is right most. For example π is
    * 314159265358979324×10<sup>-17</sup>.
    * 
    * @return Current exponent.
    * @throws BCDError
    *                 when the value is not finite
    * @model name="Get_Exponent"
    */
   public final int Get_Exponent ()
      throws BCDError
      {
      if (Type == BCDFloat.Type_Finite)
         {
         return Exponent;
         }
      else
         {
         throw new BCDError ("Not a finite number.");
         } // if
      } // Get_Exponent

   /**
    * coefficient - while a 18 digits number can also be represented as an long
    * an 18digits multiplication need a 36digits temporary and that is more then
    * a long can do.
    */
   public int Get_Num_Digits ()
      throws BCDError
      {
      if (Type == BCDFloat.Type_Finite)
         {
         Coefficient.shorten ();
         return Coefficient.Get_Num_Digits ();
         }
      else
         {
         throw new BCDError ("Not a finite number.");
         } // if
      } // Get_Num_Digits

   /**
    * @model name="init"
    */
   public final void init ()
      {
      Exponent = 0;
      Coefficient = new BCDInteger ();
      Type = BCDFloat.Type_Finite;

      return;
      } // init

   /**
    * @param s
    * @model name="init"
    */
   public final void init (String s)
      {
      if (s.equals (BCDFloat.Type_NaN_Text))
         {
         Type = BCDFloat.Type_NaN;
         return;
         }
      else if (s.equals (BCDFloat.Type_Positive_Infinity_Text))
         {
         Type = BCDFloat.Type_Positive_Infinity;
         return;
         }
      else if (s.equals (BCDFloat.Type_Negative_Infinity_Text))
         {
         Type = BCDFloat.Type_Negative_Infinity;
         return;
         } // if

      // search for exponent, parse, cut
      int idx = s.indexOf ('e');

      if (idx == (-1))
         {
         idx = s.indexOf ('E');
         } // if

      if (idx != (-1))
         {
         final String es = s.substring (idx + 1);
         try
            {
            Exponent = Integer.parseInt (es);
            }
         catch (final NumberFormatException Exception)
            {
            //#ifdef DEBUG
            //# Exception.printStackTrace ();
            //#endif
            } // try
         s = s.substring (0, idx);
         }

      // search for decimal point, cut
      idx = s.indexOf ('.');
      if (idx != (-1))
         {
         Exponent -= s.length () - idx - 1;
         s = s.substring (0, idx) + s.substring (idx + 1);
         }

      Coefficient.initFromString (s);
      } // int

   /**
    * integer part of the BCDFloat
    * </p>
    * Do not mix up with To_Integer - This version returns another BCDFloat!
    * </p>
    * 
    * @return integer part of this
    * @model name="Integer"
    */
   public BCDFloat Integer ()
      {
      // System.out.println ("+ BCDFloat.Integer ()");

      BCDFloat Retval = new BCDFloat (this);

      Retval.normalize ();

      // System.out.println ("> Exponent  = " + Exponent);

      final int Int_Digits =
         Retval.Exponent + Retval.Coefficient.Num_Digits >= 0
            ? Retval.Exponent + Retval.Coefficient.Num_Digits + 1 : 0;
      final int Frac_Digits =
         Retval.Coefficient.Num_Digits > Int_Digits
            ? Retval.Coefficient.Num_Digits - Int_Digits : 0;

      // System.out.println ("> Int_Digits  = " + Int_Digits);
      // System.out.println ("> Frac_Digits = " + Frac_Digits);

      if (Int_Digits <= 0)
         {
         Retval = BCDFloat.Num_0;
         }
      else if (Frac_Digits > 0)
         {
         Retval = new BCDFloat (Retval.To_Long ());
         } // if

      // System.out.println ("- BCDFloat.Integer ()");
      return Retval;
      } // Integer

   /**
    * current value is Infinity.
    * 
    * @return
    * @model name="Is_Infinite"
    */
   public final boolean Is_Infinite ()
      {
      return (Type == BCDFloat.Type_Negative_Infinity)
         || (Type == BCDFloat.Type_Positive_Infinity);
      } // Is_Infinite

   /**
    * Current value is integer value.
    * 
    * @return
    * @model name="Is_Integer"
    */
   public boolean Is_Integer ()
      {
      final BCDFloat x = new BCDFloat (this);

      x.normalize ();

      final int Int_Digits =
         x.Exponent + x.Coefficient.Num_Digits >= 0 ? x.Exponent
            + x.Coefficient.Num_Digits + 1 : 0;
      final int Frac_Digits =
         x.Coefficient.Num_Digits > Int_Digits ? x.Coefficient.Num_Digits
            - Int_Digits : 0;

      return Frac_Digits == 0;
      } // Is_Integer

   /**
    * current value is not a number
    * 
    * @return
    * @model name="Is_NaN"
    */
   public final boolean Is_NaN ()
      {
      return Type == BCDFloat.Type_NaN;
      } // Is_NaN

   /**
    * Current value us negative
    * 
    * @return
    * @model name="Is_Negative"
    */
   public final boolean Is_Negative ()
      {
      return (Type == BCDFloat.Type_Negative_Infinity)
         || (Coefficient.Negative);
      } // Is_Negative

   public final boolean Is_Zero ()
      {
      return ((Type == BCDFloat.Type_Finite) && (Coefficient.Num_Digits == 0));
      } // Is_Zero

   /**
    * @return log<sub>10</sub> (x)
    * @model name="log_10"
    */
   public final BCDFloat log_10 ()
      {
      return new BCDFloat (log_e ().div (BCDFloat.Num_log_e_10));
      } // log_10

   /**
    * @return log<sub>e</sub> (x)
    * @model name="log_e"
    */
   public final BCDFloat log_e ()
      {
      BCDFloat Retval = new BCDFloat ();

      if (Is_Zero ())
         {
         Retval.Type = BCDFloat.Type_Negative_Infinity;
         } // if
      else if (Coefficient.Negative)
         {
         Retval.Type = BCDFloat.Type_NaN;
         } // if
      else
         {
         BCDFloat x = new BCDFloat (this);

         boolean cInv = false;
         if (!x.sub (BCDFloat.Num_1).Coefficient.Negative) // x > 1 use x' = 1/x, ln x' = - ln x
            {
            cInv = true;
            x = BCDFloat.Num_1.div (x);
            } // if

         // determine k (min, integer) such that Exponent^k*x = r > 1/Exponent
         Retval = x;
         BCDFloat ep = BCDFloat.Num_e;

         //    calculate max n (step) required with k = n�
         int p = 1;
         while (Retval.sub (BCDFloat.Num_1_div_e).Coefficient.Negative)
            {
            p *= 2;
            ep = ep.mul (ep);

            Retval = x.mul (ep);
            } // while

         //    using max step approximate k with bisection
         //    (decreasing number of multiplications for large k)
         int k = 0;
         Retval = x;
         for (p /= 2; p >= 1; p /= 2)
            {
            // prepare step
            int pt = 1;
            ep = BCDFloat.Num_e;
            while (pt != p)
               {
               pt *= 2;
               ep = ep.mul (ep);
               } // while

            int kp = k;
            BCDFloat rp = Retval;

            // add step, rollback on overflow
            while (Retval.sub (BCDFloat.Num_1_div_e).Coefficient.Negative)
               {
               kp = k;
               rp = Retval;

               Retval = Retval.mul (ep);
               k += p;
               } // while
            Retval = rp;
            k = kp;
            } // for

         // ln x = ln (r * Exponent^(-k) ) = ln r - k
         Retval = Retval.lnI ().sub (new BCDFloat (k));

         if (cInv) // x > 1
            {
            Retval.Coefficient.Negative = !Retval.Coefficient.Negative;
            } // if
         } // if
      return Retval;
      } // log_e

   /**
    * @param v
    * @return
    * @model name="mul"
    */
   public final BCDFloat mul (final BCDFloat v)
      {
      final BCDFloat p = new BCDFloat ();

      if ((Type == BCDFloat.Type_NaN) || (v.Type == BCDFloat.Type_NaN))
         {
         p.Type = BCDFloat.Type_NaN;
         return p;
         }
      if (((Type == BCDFloat.Type_Positive_Infinity) && (v.Type == BCDFloat.Type_Positive_Infinity))
         || ((Type == BCDFloat.Type_Negative_Infinity) && (v.Type == BCDFloat.Type_Negative_Infinity)))
         {
         p.Type = BCDFloat.Type_Positive_Infinity;
         return p;
         }
      if (((Type == BCDFloat.Type_Positive_Infinity) && (v.Type == BCDFloat.Type_Negative_Infinity))
         || ((Type == BCDFloat.Type_Negative_Infinity) && (v.Type == BCDFloat.Type_Positive_Infinity)))
         {
         p.Type = BCDFloat.Type_Negative_Infinity;
         return p;
         }
      if (((Type == BCDFloat.Type_Negative_Infinity) && (!v.Coefficient.Negative))
         || ((Type == BCDFloat.Type_Positive_Infinity) && (v.Coefficient.Negative))
         || ((v.Type == BCDFloat.Type_Negative_Infinity) && (!Coefficient.Negative))
         || ((v.Type == BCDFloat.Type_Positive_Infinity) && (Coefficient.Negative)))
         {
         p.Type = BCDFloat.Type_Negative_Infinity;
         return p;
         }
      if (((Type == BCDFloat.Type_Positive_Infinity) && (!v.Coefficient.Negative))
         || ((Type == BCDFloat.Type_Negative_Infinity) && (v.Coefficient.Negative))
         || ((v.Type == BCDFloat.Type_Positive_Infinity) && (!Coefficient.Negative))
         || ((v.Type == BCDFloat.Type_Negative_Infinity) && (Coefficient.Negative)))
         {
         p.Type = BCDFloat.Type_Negative_Infinity;
         return p;
         }

      p.Coefficient = Coefficient.mul (v.Coefficient);
      p.Exponent = Exponent + v.Exponent;

      p.Round ();

      return p;
      }

   /**
    * @return negative value
    * @model name="neg"
    */
   public final BCDFloat neg ()
      {
      final BCDFloat retval = new BCDFloat (this);

      retval.Coefficient.Negative = !retval.Coefficient.Negative;

      return retval;
      } // neeg

   /**
    * @model name="normalize"
    */
   public final void normalize ()
      {
      if (Type != BCDFloat.Type_Finite)
         {
         return;
         }

      Coefficient.shorten ();
      int l = Coefficient.Num_Digits;

      while ((l > 1) && (Coefficient.Digits [0] == 0))
         {
         Coefficient = Coefficient.Shift_Left (1);
         ++Exponent;
         --l;
         }
      if (l == 0)
         {
         Exponent = 0;
         }
      }

   /**
    * @param maxE
    * @model name="normalize"
    */
   public final void normalize (final int maxE)
      {
      if (Type != BCDFloat.Type_Finite)
         {
         return;
         }

      Coefficient.shorten ();
      int l = Coefficient.Num_Digits;

      while ((l > 1) && (Coefficient.Digits [0] == 0) && (Exponent < maxE))
         {
         Coefficient = Coefficient.Shift_Left (1);
         ++Exponent;
         --l;
         }
      if (l == 0)
         {
         Exponent = 0;
         }
      }

   /**
    * Convert Polar to Rectangle Coordinates
    * 
    * @param r
    *                radius
    * @param w
    *                angle
    * @return x = cos (w) × r
    * @model name="P_To_X"
    */
   public final BCDFloat P_To_X (final BCDFloat w)
      {
      //System.out.println ("+ BCDFloat.P_To_X ()");

      final BCDFloat retval = w.cos ().mul (this);

      //System.out.println ("- BCDFloat.P_To_X ()");
      return retval;
      } // P_To_X

   /**
    * Convert Polar to Rectangle Coordinates
    * 
    * @param r
    *                radius
    * @param w
    *                angle
    * @return x = cos (w) × r
    * @model name="P_To_X"
    */
   public final BCDFloat P_To_X (final BCDFloat w, final BCDFloat Half_Circle)
      {
      //System.out.println ("+ BCDFloat.P_To_X ()");

      final BCDFloat retval = w.cos (Half_Circle).mul (this);

      //System.out.println ("- BCDFloat.P_To_X ()");
      return retval;
      } // P_To_X

   /**
    * Convert Polar to Rectangle Coordinates
    * 
    * @param r
    *                radius
    * @param w
    *                angle
    * @return y = sin (w) × r
    * @model name="P_To_Y"
    */
   public final BCDFloat P_To_Y (final BCDFloat w)
      {
      //System.out.println ("+ BCDFloat.P_To_Y ()");

      final BCDFloat retval = w.sin ().mul (this);

      //System.out.println ("- BCDFloat.P_To_Y ()");
      return retval;
      } // P_To_Y

   /**
    * Convert Polar to Rectangle Coordinates
    * 
    * @param r
    *                radius
    * @param w
    *                angle
    * @return y = sin (w) × r
    * @model name="P_To_Y"
    */
   public final BCDFloat P_To_Y (final BCDFloat w, final BCDFloat Half_Circle)
      {
      //System.out.println ("+ BCDFloat.P_To_Y ()");

      final BCDFloat retval = w.sin (Half_Circle).mul (this);

      //System.out.println ("- BCDFloat.P_To_Y ()");
      return retval;
      } // P_To_Y

   /**
    * @param y
    * @return
    * @model name="pow"
    */
   public final BCDFloat pow (final BCDFloat y)
      {
      BCDFloat retval = null;

      if (y.equals (BCDFloat.Num_0))
         {
         retval = BCDFloat.Num_1;
         }
      else if (y.equals (BCDFloat.Num_1))
         {
         retval = new BCDFloat (this);
         }
      else if (equals (BCDFloat.Num_0))
         {
         retval = BCDFloat.Num_0;
         }
      else if (equals (BCDFloat.Num_1))
         {
         retval = BCDFloat.Num_1;
         }
      else
         {
         if (y.Is_Integer ())
            {
            long y_long = y.To_Long ();
            boolean neg = false;

            if (y_long < 0)
               {
               neg = true;
               } // if

            retval = this;

            for (int i = 1; i < (neg ? -y_long : y_long); i++)
               {
               retval = retval.mul (this);
               }

            if (neg)
               {
               retval = BCDFloat.Num_1.div (retval);
               } // if
            }
         else
            {
            if (To_Long () >= 0)
               {
               // exp(y*log(x));
               final BCDFloat ln_x = log_e ();
               final BCDFloat ln_pow = y.mul (ln_x);
               retval = ln_pow.exp_e ();
               }
            else
               {
               retval = new BCDFloat ("Type_NaN");
               } // if
            } // if
         } // if
      return retval;
      }// pow

   /**
    * Convert Rectangle to Polar Coordinates
    * 
    * @param x
    *                x position
    * @param y
    *                y position
    * @return r = √(x²+y²)
    * @model name="R_To_R"
    */
   public final BCDFloat R_To_R (final BCDFloat y)
      {
      //System.out.println ("+ BCDFloat.R_To_R ()");

      final BCDFloat x2_y2 = square ().add (y.square ());
      final BCDFloat retval = x2_y2.square_root ();

      //System.out.println ("- BCDFloat.R_To_R ()");
      return retval;
      } // R_To_R

   /**
    * Convert Rectangle to Polar Coordinates
    * 
    * @param x
    *                x position
    * @param y
    *                y position
    * @return w = arctan (y÷x)
    * @model name="R_To_W"
    */
   public final BCDFloat R_To_W (final BCDFloat y)
      {
      //System.out.println ("+ BCDFloat.R_To_W ()");

      BCDFloat retval;

      if (Is_Zero ())
         {
         retval = BCDFloat.Num_PI.div (BCDFloat.Num_2);
         }
      else
         {
         final BCDFloat Div = y.div (this);
         retval = Div.arc_tan ();
         } // if

      //System.out.println ("- BCDFloat.R_To_W ()");
      return retval;
      } // R_To_W

   /**
    * Convert Rectangle to Polar Coordinates
    * 
    * @param x
    *                x position
    * @param y
    *                y position
    * @return w = arctan (y÷x)
    * @model name="R_To_W"
    */
   public final BCDFloat R_To_W (final BCDFloat y, final BCDFloat Half_Circle)
      {
      //System.out.println ("+ BCDFloat.R_To_W ()");

      BCDFloat retval;

      if (Is_Zero ())
         {
         retval = BCDFloat.Num_PI.div (BCDFloat.Num_2);
         }
      else
         {
         final BCDFloat Div = y.div (this);
         retval = Div.arc_tan (Half_Circle);
         } // if

      //System.out.println ("- BCDFloat.R_To_W ()");
      return retval;
      } // R_To_W

   /**
    * <sup>r</sup>√this
    * 
    * @return e<sup>(log<sub>e</sub> this ÷ r)</sup>
    * @model name="root"
    */
   public final BCDFloat root (final BCDFloat r)
      {
      final BCDFloat log = log_e ();
      final BCDFloat div = log.div (r);
      final BCDFloat retval = div.exp_e ();

      return retval;
      } // root

   /**
    * Round to Max_Precision and Max_Exponent.
    * 
    * @model name="Round"
    */
   public final void Round ()
      {
      if (Type == BCDFloat.Type_Finite)
         {
         Coefficient.shorten ();
         final int Length = Coefficient.Num_Digits;

         if (Length > BCDFloat.Max_Precision)
            {
            boolean Round_Up = false;

            if (Coefficient.Digits [Length - BCDFloat.Max_Precision - 1] >= 5)
               {
               Round_Up = true;
               } // if

            Coefficient =
               Coefficient.Shift_Left (Length - BCDFloat.Max_Precision);
            Exponent += Length - BCDFloat.Max_Precision;

            if (Round_Up)
               {
               Coefficient = Coefficient.addUnsigned (1);
               } // if
            } // if

         if (Exponent > BCDFloat.Max_Exponent)
            {
            if (Coefficient.Negative)
               {
               Type = BCDFloat.Type_Negative_Infinity;
               }
            else
               {
               Type = BCDFloat.Type_Positive_Infinity;
               } // if
            } // if
         if (Exponent < -BCDFloat.Max_Exponent)
            {
            Coefficient = new BCDInteger ();
            Exponent = 0;
            } // if
         } // if
      return;
      } // Round

   /**
    * Round to a given Precision and Exponent.
    * 
    * @param Precision
    * @param Exponent
    * @model name="Round"
    */
   public final BCDFloat Round (final int Precision, int Exponent)
      {
      final BCDFloat Retval = new BCDFloat (this);

      if (Retval.Type == BCDFloat.Type_Finite)
         {
         Retval.Coefficient.shorten ();
         final int Length = Retval.Coefficient.Num_Digits;

         if (Length > Precision)
            {
            boolean Round_Up = false;
            if (Retval.Coefficient.Digits [Length - Precision - 1] >= 5)
               {
               Round_Up = true;
               } // if

            Retval.Coefficient =
               Retval.Coefficient.Shift_Left (Length - Precision);
            Retval.Exponent += Length - Precision;

            if (Round_Up)
               {
               Retval.Coefficient = Retval.Coefficient.addUnsigned (1);
               } // if
            } // if

         if (Retval.Exponent > Exponent)
            {
            if (Retval.Coefficient.Negative)
               {
               Retval.Type = BCDFloat.Type_Negative_Infinity;
               }
            else
               {
               Retval.Type = BCDFloat.Type_Positive_Infinity;
               } // if
            } // if
         if (Retval.Exponent < -Exponent)
            {
            Retval.Coefficient = new BCDInteger ();
            Retval.Exponent = 0;
            } // if
         } // if

      Retval.normalize ();

      return Retval;
      } // round

   /**
    * sine
    * 
    * @return sine
    * @model name="sin"
    */
   public final BCDFloat sin ()
      {
      //System.out.println ("+ BCDFloat.sin (" + To_Debug_String () + ")");

      BCDFloat x = new BCDFloat (this);
      boolean xNeg = false;

      // sin x = - sin(-x)
      if (x.Coefficient.Negative)
         {
         xNeg = true;
         x.Coefficient.Negative = false;
         }

      if (!x.sub (BCDFloat.Num_2_PI).Coefficient.Negative) // x > 2 pi
         {
         final long k = x.div (BCDFloat.Num_2_PI).To_Long ();
         x = x.sub (BCDFloat.Num_2_PI.mul (new BCDFloat (k)));
         }

      if (!x.sub (BCDFloat.Num_PI).Coefficient.Negative) // π < x < 2 π
         {
         xNeg = !xNeg;
         x = x.sub (BCDFloat.Num_PI);
         }

      final BCDFloat Retval = x.sinI ();

      if (xNeg)
         {
         Retval.Coefficient.Negative = !Retval.Coefficient.Negative;
         }

      //System.out.println ("- BCDFloat.sin (" + Retval.To_Debug_String ()
      //+ ")");
      return Retval;
      } // sin

   /**
    * sine
    * 
    * @param Half_Circle
    *                value for a half circle
    * @return sine
    * @model name="sin"
    */
   public final BCDFloat sin (final BCDFloat Half_Circle)
      {
      //System.out.println ("+ BCDFloat.sin (" + To_Debug_String () + ","
      //+ Half_Circle.To_Debug_String () + ")");

      final BCDFloat Retval = mul (BCDFloat.Num_PI).div (Half_Circle).sin ();

      //System.out.println ("- BCDFloat.sin (" + Retval.To_Debug_String ()
      //+ ")");
      return Retval;
      } // sin

   /**
    * hyperbolic sine: (e<sup>x</sup> - e<sup>-x</sup>) ÷ 2
    * 
    * @return Hyperbolic sine.
    * @model name="sin_hyp"
    */
   public final BCDFloat sin_hyp ()
      {
      final BCDFloat e_x = exp_e ();
      final BCDFloat e_m_x = BCDFloat.Num_0.sub (this).exp_e ();
      final BCDFloat sub = e_x.sub (e_m_x);
      final BCDFloat Retval = sub.div (BCDFloat.Num_2);

      return Retval;
      } // sin_hyp

   /**
    * X Square
    * 
    * @return x<sup>2</sup>
    * @model name="sqr"
    */
   public final BCDFloat square ()
      {
      return mul (this);
      } // square

   /**
    * <sup>2</sup>√x
    * 
    * @return
    * @model name="sqrt"
    */
   public final BCDFloat square_root ()
      {
      // start value: input value with half of effective exponent
      BCDFloat r = new BCDFloat ();

      if (Coefficient.Negative)
         {
         r.Type = BCDFloat.Type_NaN;
         return r;
         }

      r.Coefficient = new BCDInteger (Coefficient);
      r.Exponent = (Exponent - Coefficient.Num_Digits) / 2;

      final BCDFloat oneHalf = BCDFloat.Num_0_5;

      BCDFloat rn;
      // Newton iteration
      for (int i = 0; i < 10 * BCDFloat.Max_Precision; ++i) // max. number of iterations
         {
         rn = r.add (div (r)).mul (oneHalf);
         if (rn.sub (r).Is_Zero () || (rn.Type != BCDFloat.Type_Finite)) // stop if increment is zero
            {
            r = rn;
            break;
            } // if
         r = rn;
         } // for

      r.normalize ((Exponent + 1) / 2);

      return r;
      } // square_root

   /**
    * x - y
    * 
    * @param vt
    * @return
    * @model name="sub"
    */
   public BCDFloat sub (final BCDFloat vt)
      {
      final BCDFloat v = new BCDFloat (vt);
      v.Coefficient.Negative = !v.Coefficient.Negative;
      if (v.Type == BCDFloat.Type_Positive_Infinity)
         {
         v.Type = BCDFloat.Type_Negative_Infinity;
         }
      else if (v.Type == BCDFloat.Type_Negative_Infinity)
         {
         v.Type = BCDFloat.Type_Positive_Infinity;
         } // if

      final BCDFloat r = add (v);
      return r;
      } // sub

   /**
    * tan x
    * 
    * @return
    * @model name="tan"
    */
   public final BCDFloat tan ()
      {
      //System.out.println ("+ BCDFloat.tan (" + To_Debug_String () + ")");

      final BCDFloat Sin = this.sin ();
      final BCDFloat Cos = this.cos ();
      final BCDFloat Retval = Sin.div (Cos);

      //System.out.println ("- BCDFloat.tan (" + Retval.To_Debug_String ()
      //+ ")");

      return Retval;
      } // tan

   /**
    * tangent
    * 
    * @param Half_Circle
    *                value for a half circle
    * @return tangent
    * @model name="tan"
    */
   public final BCDFloat tan (final BCDFloat Half_Circle)
      {
      //System.out.println ("+ BCDFloat.tan (" + To_Debug_String () + ","
      //+ Half_Circle.To_Debug_String () + ")");

      final BCDFloat Retval = mul (BCDFloat.Num_PI).div (Half_Circle).tan ();

      //System.out.println ("- BCDFloat.tan (" + Retval.To_Debug_String ()
      //+ ")");

      return Retval;
      } // tan

   /**
    * Hyperbolic tangent: (e<sup>2x</sup> - 1) ÷ (e<sup>2x</sup> + 1)
    * 
    * @return Hyperbolic tangent.
    * @model name="tan_hyp"
    */
   public final BCDFloat tan_hyp ()
      {
      final BCDFloat x_2 = mul (BCDFloat.Num_2);
      final BCDFloat exp_x_2 = x_2.exp_e ();
      final BCDFloat exp_x_2_m_1 = exp_x_2.sub (BCDFloat.Num_1);
      final BCDFloat exp_x_2_p_1 = exp_x_2.add (BCDFloat.Num_1);
      final BCDFloat retval = exp_x_2_m_1.div (exp_x_2_p_1);

      return retval;
      } // tan_hyp

   /**
    * convert to native/debug string representation
    * 
    * @return
    * @model name="To_Debug_String"
    */
   public String To_Debug_String ()
      {
      String s = "";

      if (Type == BCDFloat.Type_Finite)
         {
         s = "(";
         s += Coefficient.toString ();
         s += ",";
         s += Exponent;
         s += ")";
         }

      if (Type == BCDFloat.Type_NaN)
         {
         s = "(NaN)";
         }

      if (Type == BCDFloat.Type_Positive_Infinity)
         {
         s = "(+Inf)";
         }

      if (Type == BCDFloat.Type_Negative_Infinity)
         {
         s = "(-Inf)";
         }

      return s;
      } // To_Debug_String

   /**
    * Convert to integer
    * </p>
    * Do not mix up with Integer - This version returns a Java int!
    * </p>
    * 
    * @return value as java int
    * @model name="To_Integer"
    */
   public int To_Integer ()
      {
      return (int) To_Long ();
      } // To_Integer

   /**
    * Convert to long integer
    * </p>
    * Do not mix up with Integer - This version returns a Java int!
    * </p>
    * 
    * @return value as java int
    * @model name="To_Integer"
    */
   public long To_Long ()
      {
      BCDInteger retval;

      if (Exponent > 0)
         {
         retval = Coefficient.shiftRight (Exponent);
         }
      else if (Exponent < 0)
         {
         if (Coefficient.Num_Digits > (-Exponent))
            {
            retval = Coefficient.Shift_Left (-Exponent);
            }
         else
            {
            retval = new BCDInteger (0);
            } // if
         }
      else
         {
         retval = new BCDInteger (Coefficient);
         } // if
      return retval.To_Long_Integer ();
      } // To_Integer

   /**
    * convert to scientific string representation
    * 
    * @see java.lang.Object#toString()
    * @model name="toString"
    */
   public String toString ()
      {
      String Retval;

      if (Type == BCDFloat.Type_NaN)
         {
         Retval = BCDFloat.Type_NaN_Text;
         }
      else if (Type == BCDFloat.Type_Positive_Infinity)
         {
         Retval = BCDFloat.Type_Positive_Infinity_Text;
         }
      else if (Type == BCDFloat.Type_Negative_Infinity)
         {
         Retval = BCDFloat.Type_Negative_Infinity_Text;
         }
      else
         {
         String s = Coefficient.toString ();
         String sign = "";
         if (Coefficient.Negative)
            {
            s = s.substring (1);
            sign = "-";
            }

         final int adjExp = Exponent + s.length () - 1;

         if ((Exponent <= 0) && (adjExp >= (-6))) // no E-notation
            {
            if (Exponent == 0) // no decimal point
               {
               Retval = sign + s;
               }
            else
               { // decimal point
               int l = s.length ();
               while (l < (-Exponent + 1))
                  {
                  s = "0" + s;
                  ++l;
                  }
               Retval =
                  sign + s.substring (0, l + Exponent) + "."
                     + s.substring (l + Exponent);
               }
            }
         else
            { // E-notation
            final int l = s.length ();
            if (l > 1) // decimal point
               {
               s = s.substring (0, 1) + "." + s.substring (1);
               }
            Retval = sign + s + "E" + adjExp;
            }
         } // if
      return Retval;
      } // toString

   /**
    * log<sub>e</sub> x using series expansion, usable around 1
    * 
    * @return
    * @model name="lnI"
    */
   private final BCDFloat lnI ()
      {
      BCDFloat b;
      final BCDFloat d = sub (BCDFloat.Num_1); // 1-x
      BCDFloat a = new BCDFloat (d);
      BCDFloat r = new BCDFloat (d);
      BCDFloat rn;

      // Taylor Series Expansion
      for (int i = 2; i < 10 * BCDFloat.Max_Precision; ++i) // max. number of iterations
         {
         a = a.mul (d);
         b = new BCDFloat (i);
         if (i % 2 == 0)
            {
            rn = r.sub (a.div (b));
            }
         else
            {
            rn = r.add (a.div (b));
            } // if

         if (rn.sub (r).Is_Zero () || (rn.Type != BCDFloat.Type_Finite)) // stop if increment is zero
            {
            r = rn;
            break;
            } // if

         r = rn;
         } // for

      return r;
      } // lnI

   /**
    * sin using series expansion, usable around 0 (-π .. π)
    * 
    * @return
    * @model name="sinI"
    */
   private final BCDFloat sinI () // 
      {
      BCDFloat x = this;
      final BCDFloat xsq = x.mul (x);
      BCDFloat f = BCDFloat.Num_1;

      BCDFloat s = x;
      BCDFloat sn;

      int i;
      for (i = 1; i < 10 * BCDFloat.Max_Precision; ++i)
         {
         x = x.mul (xsq);
         f = f.mul (new BCDFloat (2 * i)).mul (new BCDFloat (2 * i + 1));

         if (i % 2 != 0)
            {
            sn = s.sub (x.div (f));
            }
         else
            {
            sn = s.add (x.div (f));
            }

         if (sn.sub (s).Is_Zero () || (sn.Type != BCDFloat.Type_Finite)) // stop if increment is zero
            {
            s = sn;
            break;
            }

         s = sn;
         }

      return s;
      } // sinI
   } // BCDFloat

//vim: set nowrap tabstop=8 shiftwidth=3 softtabstop=3 expandtab :
//vim: set textwi dth=0 filetype=java foldmethod=marker nospell :
