/*********************************************************** {{{1 ***********
 *  Copyright (C) 2005 Kai G. Schwebke, upn.schwebke.com
 *  Copyright (C) 2005,2008  Martin Krischik (extensions for FX-602P)
 ****************************************************************************
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ****************************************************************************
 *  $Author: krischik $
 *
 *  $Revision: 460 $
 *  $Date: 2008-08-16 09:08:31 +0200 (Sa, 16 Aug 2008) $
 *
 *  $Id: BCDInteger.java 460 2008-08-16 07:08:31Z krischik $
 *  $HeadURL: https://uiq3.svn.sourceforge.net/svnroot/uiq3/tags/HP45-CLDC-2.0.1/Calculator/src/net/sourceforge/uiq3/bcd/BCDInteger.java $
 ************************************************************ }}}1 **********/

/**
 * @author "Martin Krischik" <krischik@users.sourceforge.net>
 * @model name="Uiq3.Calculator.Bcd"
 */
package net.sourceforge.uiq3.bcd;

/**
 * Arbitrary Max_Precision BCD Integer Arithmetic
 */
final class BCDInteger
   {
   /**
    * remainder of last long division
    */
   private BCDInteger Last_Remainder;
   /**
    * remainder of last short division
    */
   private byte Last_Remainder_Short;
   /**
    * number: sum(i = 0 ... Num_Digits-1; Digits[i]*10^i)
    */
   protected byte [] Digits;
   /**
    * sign
    */
   protected boolean Negative;
   /**
    * number of digits
    */
   protected int Num_Digits;

   /**
    * new Number, val = 0, length = 0
    * 
    * @model name="BCDInteger"
    */
   public BCDInteger ()
      {
      initDigits (0);

      return;
      } // BCDInteger

   /**
    * new Number from existing number
    * 
    * @param Value
    * @model name="BCDInteger"
    */
   public BCDInteger (final BCDInteger Value)
      {
      initDigits (Value.Num_Digits);
      Negative = Value.Negative;

      for (int i = 0; i < Num_Digits; ++i)
         {
         Digits [i] = Value.Digits [i];
         } // for

      return;
      } // BCDInteger

   /**
    * new Number, val = 0, length = n
    * 
    * @param n
    * @model name="BCDInteger"
    */
   public BCDInteger (final int n)
      {
      initDigits (n);

      return;
      } // BCDInteger

   /**
    * new Number, val and length from string
    * 
    * @param String
    * @model name="BCDInteger"
    */
   public BCDInteger (final String String)
      {
      initFromString (String);
      return;
      } // BCDInteger

   /**
    * signed addition
    * 
    * @param a
    * @return
    * @model name="add"
    */
   public BCDInteger add (final BCDInteger a)
      {
      BCDInteger Retval;

      if (Negative == a.Negative)
         {
         Retval = addUnsigned (a);
         }
      else
         {
         final int c = cmpUnsigned (a);

         if (c > 0)
            {
            Retval = subUnsigned (a);
            }
         else
            {
            Retval = a.subUnsigned (this);
            } // if

         if (((c > 0) && (Negative)) || ((c < 0) && (a.Negative)))
            {
            Retval.Negative = true;
            } // if
         } // if
      return Retval;
      }

   /**
    * unsigned (absolute) addition
    * 
    * @param a
    * @return
    * @model name="addUnsigned"
    */
   public BCDInteger addUnsigned (final BCDInteger a)
      {
      final int newDigits = Math.max (Num_Digits, a.Num_Digits) + 1;
      final BCDInteger newInt = new BCDInteger (newDigits);

      int i;
      for (i = 0; i < (newDigits - 1); ++i)
         {
         if (i < Num_Digits)
            {
            newInt.Digits [i] += Digits [i];
            }
         if (i < a.Num_Digits)
            {
            newInt.Digits [i] += a.Digits [i];
            }
         if (newInt.Digits [i] >= 10)
            {
            newInt.Digits [i] -= 10;
            ++newInt.Digits [i + 1];
            }
         }

      newInt.Negative = Negative;

      return newInt;
      }

   /**
    * unsigned (absolute) addition of one digit
    * 
    * @param a
    * @return
    * @model name="addUnsigned"
    */
   public BCDInteger addUnsigned (final int a)
      {
      final int newDigits = Num_Digits + 1;
      final BCDInteger newInt = new BCDInteger (newDigits);

      newInt.Digits [0] = (byte) a;

      int i;
      for (i = 0; i < (newDigits - 1); ++i)
         {
         if (i < Num_Digits)
            {
            newInt.Digits [i] += Digits [i];
            }
         if (newInt.Digits [i] >= 10)
            {
            newInt.Digits [i] -= 10;
            ++newInt.Digits [i + 1];
            }
         }

      newInt.Negative = Negative;

      return newInt;
      }

   /**
    * signed compare
    * 
    * @param a
    * @return
    * @model name="cmp"
    */
   public int cmp (final BCDInteger a)
      {
      if ((!Negative) && (!a.Negative))
         {
         return cmpUnsigned (a);
         }
      else if (Negative && a.Negative)
         {
         return -cmpUnsigned (a);
         }
      else if (Negative && (!a.Negative))
         {
         return -1;
         }
      return 1;
      }

   /**
    * unsigned (absolute) compare
    * 
    * @param a
    * @return
    * @model name="cmpUnsigned"
    */
   public int cmpUnsigned (final BCDInteger a)
      {
      final int cmpDigits = Math.max (Num_Digits, a.Num_Digits);
      int i;

      for (i = cmpDigits - 1; i >= 0; --i)
         {
         int digit, digita;
         digit = digita = 0;

         if (i < Num_Digits)
            {
            digit = Digits [i];
            }
         if (i < a.Num_Digits)
            {
            digita = a.Digits [i];
            }
         if (digit > digita)
            {
            return 1;
            }
         else if (digit < digita)
            {
            return -1;
            }
         }
      return 0;
      }

   /**
    * return Num_Digits
    * 
    * @return
    * @model name="digits"
    */
   public int Get_Num_Digits ()
      {
      return Num_Digits;
      } // Get_Num_Digits

   /**
    * signed division
    * 
    * @param v
    * @return
    * @model name="div"
    */
   public BCDInteger div (final BCDInteger v)
      {
      BCDInteger Retval;

      v.shorten ();
      if (v.Num_Digits > 1)
         {
         Retval = divUnsigned (v);
         }
      else
         {
         Retval = divUnsigned (v.Digits [0]);
         Last_Remainder = new BCDInteger (1);
         Last_Remainder.Digits [0] = Last_Remainder_Short;
         }

      if (Negative != v.Negative)
         {
         Retval.Negative = true;
         }

      return Retval;
      }

   /**
    * <code>
    *   predicate: Num_Digits > v.numDigits, 
    *      v = v.shorten(), 
    *      v.numDigits > 1
    * </code>
    * 
    * @param v
    * @return
    * @model name="divUnsigned"
    */
   public BCDInteger divUnsigned (BCDInteger v)
      {
      final int n = v.Num_Digits;
      final int m = Num_Digits - n;

      final BCDInteger q = new BCDInteger (m + 1);

      final byte d = (byte) (10 / (v.Digits [n - 1] + 1));
      final BCDInteger u = mulUnsigned (d);
      v = v.mulUnsigned (d);

      int j;
      byte qc, rc;
      for (j = m; j >= 0; --j)
         {
         qc =
            (byte) ((u.Digits [j + n] * 10 + u.Digits [j + n - 1]) / v.Digits [n - 1]);
         rc =
            (byte) ((u.Digits [j + n] * 10 + u.Digits [j + n - 1]) % v.Digits [n - 1]);
         if ((qc == 10)
            || (qc * v.Digits [n - 2] > 10 * rc + u.Digits [j + n - 2]))
            {
            --qc;
            rc += v.Digits [n - 1];

            if (rc < 10)
               {
               if ((qc == 10)
                  || (qc * v.Digits [n - 2] > 10 * rc + u.Digits [j + n - 2]))
                  {
                  --qc;
                  rc += v.Digits [n - 1];
                  }
               }
            }

         BCDInteger ut = u.mid (j, n + 1);
         ut = ut.sub (v.mulUnsigned (qc));

         q.Digits [j] = qc;
         if (ut.Negative)
            {
            --q.Digits [j];
            ut = ut.add (v);
            }

         for (int k = 0; k <= n; ++k)
            {
            u.Digits [k + j] = ut.Digits [k];
            }
         }

      Last_Remainder = u.divUnsigned (d);

      return q;
      }

   public BCDInteger divUnsigned (final int v)
      {
      final BCDInteger u = this.mulUnsigned (1);
      final BCDInteger q = new BCDInteger (Num_Digits);

      for (int i = Num_Digits - 1; i >= 0; --i)
         {
         final int ut = u.Digits [i + 1] * 10 + u.Digits [i];
         q.Digits [i] = (byte) (ut / v);
         u.Digits [i + 1] -= (byte) (q.Digits [i] * v / 10);
         u.Digits [i] -= (byte) (q.Digits [i] * v % 10);
         if (u.Digits [i] < 0)
            {
            u.Digits [i] += 10;
            --u.Digits [i + 1];
            } // if
         } // for

      Last_Remainder_Short = u.Digits [0];

      return q;
      }

   /**
    * set val = 0 and Num_Digits = n
    * 
    * @param n
    * @model name="initDigits"
    */
   public void initDigits (final int n)
      {
      Num_Digits = n;
      if (Num_Digits > 0)
         {
         Digits = new byte [Num_Digits];
         }
      else
         {
         Num_Digits = 0;
         Digits = null;
         }
      Negative = false;
      }

   /**
    * set val and length from integer
    * 
    * @param i
    * @model name="initFromInteger"
    */
   public void initFromInteger (final long i)
      {
      initFromString (Long.toString (i));
      }

   /**
    * set val and length from string
    * 
    * @param s
    * @model name="initFromString"
    */
   public void initFromString (final String s)
      {
      Num_Digits = s.length ();

      int i = 0;
      Negative = false;

      if (s.charAt (0) == '-')
         {
         Negative = true;
         }

      while ((i < s.length ())
         && ((s.charAt (i) == '+') || (s.charAt (i) == '-') || (s.charAt (i) == '0')))
         {
         ++i;
         --Num_Digits;
         }

      if (Num_Digits <= 0)
         {
         Num_Digits = 0;
         Digits = null;
         return;
         }

      int j = Num_Digits - 1;
      Digits = new byte [Num_Digits];
      while (i < s.length ())
         {
         Digits [j] = (byte) Character.digit (s.charAt (i), 10);
         --j;
         ++i;
         }
      }

   /**
    * get part of number
    * 
    * @param start
    * @param len
    * @return
    * @model name="mid"
    */
   public BCDInteger mid (final int start, final int len)
      {
      final BCDInteger newInt = new BCDInteger (len);

      int i;
      for (i = start; i < start + len; ++i)
         {
         newInt.Digits [i - start] = Digits [i];
         }

      return newInt;
      }

   /**
    * signed multiplication
    * 
    * @param a
    * @return
    * @model name="mul"
    */
   public BCDInteger mul (final BCDInteger a)
      {
      final BCDInteger p = mulUnsigned (a);

      if (Negative != a.Negative)
         {
         p.Negative = true;
         }

      return p;
      }

   /**
    * unsigned (absolute) multiplication
    * 
    * @param a
    * @return
    * @model name="mulUnsigned"
    */
   public BCDInteger mulUnsigned (final BCDInteger a)
      {
      final int newDigits = Num_Digits + a.Num_Digits;
      BCDInteger newInt = new BCDInteger (newDigits);

      for (int i = 0; i < a.Num_Digits; ++i)
         {
         newInt =
            newInt.addUnsigned (shiftRight (i).mulUnsigned (a.Digits [i]));
         } // for

      return newInt;
      } // mulUnsigned

   /**
    * unsigned (absolute) multiplication by one digit
    * 
    * @param digit
    * @return
    * @model name="mulUnsigned"
    */
   public BCDInteger mulUnsigned (final int digit)
      {
      final int newDigits = Num_Digits + 1;
      final BCDInteger newInt = new BCDInteger (newDigits);

      for (int i = 0; i < Num_Digits; ++i)
         {
         final int t = newInt.Digits [i] + digit * Digits [i];
         newInt.Digits [i] = (byte) (t % 10);
         newInt.Digits [i + 1] = (byte) (t / 10);
         } // for

      return newInt;
      } // mulUnsigned

   /**
    * shift left by n digits (= divide by 10^n)
    * 
    * @param n
    * @return
    * @model name="shiftLeft"
    */
   protected BCDInteger Shift_Left (final int n)
      {
      final int newDigits = Num_Digits - n;
      final BCDInteger Retval = new BCDInteger (newDigits);
      Retval.Negative = Negative;

      int i;
      for (i = 0; i < newDigits; ++i)
         {
         Retval.Digits [i] = Digits [i + n];
         }

      return Retval;
      }

   /**
    * shift right by n digits (= multiply by 10^n)
    * 
    * @param n
    * @return
    * @model name="shiftRight"
    */
   public BCDInteger shiftRight (final int n)
      {
      final int newDigits = Num_Digits + n;
      final BCDInteger newInt = new BCDInteger (newDigits);
      newInt.Negative = Negative;

      for (int i = n; i < newDigits; ++i)
         {
         newInt.Digits [i] = Digits [i - n];
         }

      return newInt;
      }

   /**
    * shorten (remove leading zero digits)
    * 
    * @model name="shorten"
    */
   public void shorten ()
      {
      int sh = Num_Digits - 1;

      while ((sh >= 0) && (Digits [sh] == 0))
         {
         --sh;
         } // while

      if (sh < (Num_Digits - 1))
         {
         if (sh == (-1))
            {
            initDigits (0);
            }
         else
            {
            final byte [] nd = new byte [sh + 1];

            for (int i = 0; i <= sh; ++i)
               {
               nd [i] = Digits [i];
               } // for

            Digits = nd;
            Num_Digits = sh + 1;
            } // if
         } //if
      } // shorten

   /**
    * signed subtraction
    * 
    * @param at
    * @return
    * @model name="sub"
    */
   public BCDInteger sub (final BCDInteger at)
      {
      final BCDInteger a = new BCDInteger (at);
      a.Negative = !a.Negative;
      final BCDInteger Retval = add (a);
      a.Negative = !a.Negative;
      return Retval;
      }

   // unsigned (absolute) subtraction
   // predicate: this > a
   //            (or this < a if Num_Digits == a.numDigits and
   //             underflow in most significant digit is allowed)
   public BCDInteger subUnsigned (final BCDInteger a)
      {
      final int newDigits = Num_Digits;
      final BCDInteger newInt = new BCDInteger (newDigits);

      int i;
      for (i = 0; i < Num_Digits; ++i)
         {
         newInt.Digits [i] += Digits [i];
         if (i < a.Num_Digits)
            {
            newInt.Digits [i] -= a.Digits [i];
            }
         if ((newInt.Digits [i] < 0) && (i < (Num_Digits - 1)))
            {
            newInt.Digits [i] += 10;
            --newInt.Digits [i + 1];
            }
         }

      return newInt;
      }

   public long To_Long_Integer ()
      {
      try
         {
         return Long.parseLong (toString ());
         }
      catch (final NumberFormatException Exception)
         {
         //#ifdef DEBUG
         Exception.printStackTrace ();
         //#endif
         }
      return 0;
      }

   /**
    * return string representation of number
    * 
    * @see java.lang.Object#toString()
    * @model name="toString"
    */
   public String toString ()
      {
      String Retval = "";

      if (Negative)
         {
         Retval = "-";
         } // if
      if (Num_Digits == 0)
         {
         Retval = Retval + "0";
         }
      else
         {
         boolean leadingZeros = true;
         int i;
         for (i = Num_Digits - 1; i >= 0; --i)
            {
            if ((Digits [i] != 0) || (i == 0))
               {
               leadingZeros = false;
               } // if
            if (!leadingZeros)
               {
               Retval = Retval + Integer.toString (Digits [i], 10);
               } // if
            } // for
         } // if

      return Retval;
      } // toString
   } // BCDInteger
