/*********************************************************** {{{1 ***********
 *  Copyright (C) 2005,2006  Martin Krischik (JavaME port)
 *  Copyright 2000 David G. Hicks            (Java port)
 *  Copyright 1995 Eric L. Smith             (Original C/X Windows)
 ****************************************************************************
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ****************************************************************************
 *  $Author: krischik $
 *
 *  $Revision: 461 $
 *  $Date: 2008-08-20 19:11:49 +0200 (Mi, 20 Aug 2008) $
 *
 *  $Id: Calculator.java 461 2008-08-20 17:11:49Z krischik $
 *  $HeadURL: https://uiq3.svn.sourceforge.net/svnroot/uiq3/tags/HP45-CLDC-2.0.1/HP45/src/net/sourceforge/uiq3/hp45/Calculator.java $
 ************************************************************ }}}1 **********/

/**
 * @author "Martin Krischik" <krischik@users.sourceforge.net>
 * @model name="Uiq3.HP45"
 */
package net.sourceforge.uiq3.hp45;

/**
 * HP-45 Calculator Engine
 * 
 * @author "Martin Krischik" <krischik@users.sourceforge.net>
 * @model name="Calculator"
 */
public abstract class Calculator
   implements
      net.sourceforge.uiq3.calculator.engine.ICalculator
   {
   /**
    * ROM Groups
    * 
    * @model name="Max_Group"
    */
   private static final int Max_Group = 2;

   /**
    * ROMs per group
    * 
    * @model name="Max_Rom"
    */
   private static final int Max_Rom = 8;

   /**
    * Size of each ROM
    * 
    * @model name="Rom_Size"
    */
   private static final int Rom_Size = 256;
   /**
    * Size of one register.
    * 
    * @model name="Word_Size"
    */
   protected static final int Word_Size = 14;
   /**
    * s register size
    * 
    * @model name="S_Size"
    */
   protected static final int S_Size = 12;
   /**
    * RAM Size
    * 
    * @model name=""
    */
   protected static final int Max_Ram = 100;
   /**
    * All Rom modules
    * 
    * @model name="U_Code"
    */
   private final short ucode[][][] =
      new short [Calculator.Max_Group] [Calculator.Max_Rom] [Calculator.Rom_Size];
   /**
    * Calculator is running - needed for power saving which was not there in the
    * original
    * 
    * @model
    */
   private boolean Calculator_Running = false;

   /**
    * array of all operator functions
    * 
    * @model
    */
   private net.sourceforge.uiq3.calculator.engine.IOperation op_fcn[] = null;

   /**
    * clear register operations
    * 
    * @model
    */
   private net.sourceforge.uiq3.calculator.engine.IOperation do_op_clear_reg =
      null;

   /**
    * clear s register operation
    * 
    * @model
    */
   private net.sourceforge.uiq3.calculator.engine.IOperation do_op_clear_s =
      null;

   /**
    * Group address read from file
    * 
    * @model
    */
   private int Group_Read;

   /**
    * Rom address read from file
    * 
    * @model
    */
   private int Rom_Read;

   /**
    * address read from file
    * 
    * @model
    */
   private int Pointer_Read;

   /**
    * opcode read from file
    * 
    * @model
    */
   private int Opcode_Read;

   /**
    * a key was pressed
    * 
    * @model
    */
   protected boolean Key_Flag = false;

   /**
    * Canvas class where we display our result or get our keycodes
    */
   protected net.sourceforge.uiq3.calculator.ui.ICanvas Form = null;

   /**
    * @model name="p"
    */
   protected int p = 0;

   /**
    * carry
    * 
    * @model
    */
   protected byte carry;

   /**
    * previous carry
    * 
    * @model
    */
   protected byte Previous_Carry;

   /**
    * register a
    * 
    * @model
    */
   protected final byte a[] = new byte [Calculator.Word_Size];

   /**
    * register b
    * 
    * @model
    */
   protected final byte b[] = new byte [Calculator.Word_Size];

   /**
    * register Coefficient
    * 
    * @model
    */
   protected final byte c[] = new byte [Calculator.Word_Size];

   /**
    * register Digits
    * 
    * @model
    */
   protected final byte d[] = new byte [Calculator.Word_Size];

   /**
    * register e
    * 
    * @model
    */
   protected final byte e[] = new byte [Calculator.Word_Size];

   /**
    * register f
    * 
    * @model
    */
   protected final byte f[] = new byte [Calculator.Word_Size];

   /**
    * register m
    * 
    * @model
    */
   protected final byte m[] = new byte [Calculator.Word_Size];

   /**
    * register s
    * 
    * @model
    */
   protected final byte s[] = new byte [Calculator.S_Size];

   /**
    * ram address
    * 
    * @model
    */
   protected int ram_addr;

   /**
    * Calculator RAM.
    * 
    * @model
    */
   protected final byte ram[][] =
      new byte [Calculator.Max_Ram] [Calculator.Word_Size];

   /**
    * Display enable - used to switch off the display so that intermediate
    * result are not shown to the user.
    * 
    * @model
    */
   protected boolean Display_Enable = false;

   /**
    * last key pressed
    * 
    * @model
    */
   protected int Key_Buffer;

   /**
    * System time when last Key was clicked.
    * 
    * @model
    */
   protected long Key_Time;

   /**
    * Number of virtual CPU cycles until next IO run.
    * 
    * @model
    */
   protected int IO_Count = 0;

   /**
    * @model
    */
   protected int Programm_Counter = 0;

   /**
    * @model
    */
   protected int Programm_Rom = 0;

   /**
    * @model
    */
   protected int Programm_Group = 0;

   /**
    * @model
    */
   protected int del_rom = 0;

   /**
    * @model
    */
   protected int del_grp = 0;

   /**
    * @model
    */
   protected int Return_Programm_Counter;

   //#ifdef DEBUG
   /**
    * used to store complete five-digit octal address of instruction. Used for
    * debugging only.
    * 
    * @model name="Previous_Programm_Counter"
    */
   protected int Previous_Programm_Counter;

   //#endif

   /**
    * Initialize Opcodes
    * 
    * @model
    */
   public void Init_Opcodes ()
      {
      Form.Set_Text ("Init Operations...");

      op_fcn = new net.sourceforge.uiq3.calculator.engine.IOperation [1024];

      for (int i = 0; i < 1024; i += 4)
         {
         op_fcn [i + 0] = new bad_op (this);
         op_fcn [i + 1] = new op_jsb (this);
         op_fcn [i + 2] = new op_arith (this);
         op_fcn [i + 3] = new op_goto (this);
         } // for

      op_fcn [0x000] = new op_nop (this);
      op_fcn [0x01c] = new op_dec_p (this);
      op_fcn [0x028] = new op_display_toggle (this);
      op_fcn [0x030] = new op_return (this);
      op_fcn [0x034] = do_op_clear_s = new op_clear_s (this);
      op_fcn [0x03c] = new op_inc_p (this);
      op_fcn [0x0d0] = new op_keys_to_rom_addr (this);
      op_fcn [0x0a8] = new op_c_exch_m (this);
      op_fcn [0x128] = new op_c_to_stack (this);
      op_fcn [0x1a8] = new op_stack_to_a (this);
      op_fcn [0x200] = new op_rom_addr_to_buf (this);
      op_fcn [0x228] = new op_display_off (this);
      op_fcn [0x270] = new op_c_to_addr (this);
      op_fcn [0x2a8] = new op_m_to_c (this);
      op_fcn [0x2f0] = new op_c_to_data (this);
      op_fcn [0x2f8] = new op_data_to_c (this);
      op_fcn [0x328] = new op_down_rotate (this);
      op_fcn [0x3a8] = do_op_clear_reg = new op_clear_reg (this);

      op_fcn [0x234] = new op_del_sel_grp (this);
      op_fcn [0x2b4] = new op_del_sel_grp (this);

      for (int i = 0; i < 1024; i += 128)
         {
         op_fcn [i | 0x010] = new op_sel_rom (this);
         op_fcn [i | 0x074] = new op_del_sel_rom (this);
         } // for

      for (int i = 0; i < 1024; i += 64)
         {
         op_fcn [i | 0x018] = new op_load_constant (this);
         op_fcn [i | 0x004] = new op_set_s (this);
         op_fcn [i | 0x024] = new op_clr_s (this);
         op_fcn [i | 0x014] = new op_test_s (this);
         op_fcn [i | 0x00c] = new op_set_p (this);
         op_fcn [i | 0x02c] = new op_test_p (this);
         } // for

      return;
      } // Init_Opcodes

   /**
    * @return true when calculator is running.
    * @model
    */
   public synchronized boolean Is_Running ()
      {
      return Calculator_Running;
      } // Is_Running

   /**
    * This reads a zip file that is expected to contain one entry which is the
    * HP-45 firmware listing. The instructions are parsed for later execution.
    * 
    * @model name="Value"
    */
   public void Read_Listing_File (final String File_Name)
      throws java.io.IOException
      {
      String Buffer;
      int count = 0;
      final Class self = getClass ();
      final java.io.InputStream Listing_File =
         self.getResourceAsStream (File_Name);
      final java.io.InputStreamReader Listing_Reader =
         new java.io.InputStreamReader (Listing_File);

      while ((Buffer = net.sourceforge.uiq3.Utils.Read_Line (Listing_Reader)) != null)
         {
         if ((Buffer.length () >= 25) && (Buffer.charAt (7) == 'L')
            && (Buffer.charAt (13) == ':')
            && Parse_Address (Buffer.substring (8))
            && Parse_Opcode (Buffer.substring (16)))
            {
            if ((Group_Read >= Calculator.Max_Group)
               || (Rom_Read >= Calculator.Max_Rom)
               || (Pointer_Read >= Calculator.Rom_Size))
               {
               //#ifdef DEBUG
               //# System.out.println ("bad address");
               //#endif
               }
            else
               {
               ucode [Group_Read] [Rom_Read] [Pointer_Read] =
                  (short) Opcode_Read;
               count++;
               } // if
            } // if
         } // while

      //#ifdef DEBUG
      //# System.out.println ("Read " + count + " words from " + File_Name);
      //#endif
      return;
      } // Read_Listing_File

   /**
    * Reset the calculator
    * 
    * @model
    */
   public void Reset ()
      {
      Form.Set_Text ("Reset Calculator...");

      Display_Enable = false;
      Key_Flag = false;
      Programm_Counter = 0;
      Programm_Rom = 0;
      Programm_Group = 0;
      del_rom = 0;
      del_grp = 0;

      try
         {
         do_op_clear_reg.Execute ((short) (0));
         do_op_clear_s.Execute ((short) (0));
         }
      catch (final net.sourceforge.uiq3.calculator.engine.Error Exception)
         {
         //#ifdef DEBUG
         Exception.printStackTrace ();
         //#endif
         } // try

      p = 0;
      IO_Count = 0;

      return;
      } // Reset

   /**
    * Start virtual CPU
    * 
    * @see net.sourceforge.uiq3.calculator.engine.ICalculator#run()
    * @model "Run"
    */
   public void run ()
      {
      //System.out.println ("+ Calculator.run ()");

      Form.Set_Text ("Start CPU...");

      Key_Time = System.currentTimeMillis ();
      Calculator_Running = true;
      Load ();

      while (Calculator_Running)
         {
         //#ifdef DEBUG
         Previous_Programm_Counter =
            (Programm_Group << 12) | (Programm_Rom << 9) | Programm_Counter;
         //#endif
         final short opcode =
            ucode [Programm_Group] [Programm_Rom] [Programm_Counter];
         Previous_Carry = carry;
         carry = 0;
         if (Key_Flag)
            {
            s [0] = 1;
            } // if

         Programm_Counter++;
         try
            {
            op_fcn [opcode].Execute (opcode);
            }
         catch (final net.sourceforge.uiq3.calculator.engine.Error Exception)
            {
            //#ifdef DEBUG
            Exception.printStackTrace ();
            //#endif
            } // try

         if (Programm_Counter > 0377)
            {
            // wrap back to beginning of the ROM
            Programm_Counter -= 0400;
            } // if

         IO_Count--;
         if (IO_Count <= 0)
            {
            Handle_IO ();
            // make keyboard more responsive in slow
            // case - handle_io slow down full speed runs
            IO_Count = 35;
            } // if
         } // while

      Save ();
      Form.Set_Text ("- Off -");

      //System.out.println ("- Calculator.run ()");
      return;
      }// run

   /**
    * Set Canvas to display result read keyboard
    * 
    * @model
    */
   public void Set_Form (final net.sourceforge.uiq3.calculator.ui.ICanvas Form)
      {
      this.Form = Form;
      return;
      } // setForm

   /**
    * Stop virtual CPU
    * 
    * @see net.sourceforge.uiq3.calculator.engine.ICalculator#stop()
    * @model "Stop"
    */
   public synchronized void stop ()
      {
      //System.out.println ("+ Calculator.stop ()");

      Calculator_Running = false;

      //System.out.println ("- Calculator.stop ()");
      return;
      } // stop

   /**
    * Display result and read keyboard
    * 
    * @model
    */
   private final void Handle_IO ()
      {
      //System.out.println ("+ Calculator.Handle_IO ()");
      String Display_String = new String ("");

      if (Display_Enable)
         {
         for (int i = Calculator.Word_Size - 1; i >= 0; i--)
            {
            if (b [i] >= 8)
               {
               Display_String += " ";
               }
            else if (i == 2)
               {
               if (a [i] >= 8)
                  {
                  Display_String += " -";
                  }
               else
                  {
                  Display_String += "  ";
                  } // if
               }
            else if (i == 13)
               {
               if (a [i] >= 8)
                  {
                  Display_String += "-";
                  }
               else
                  {
                  Display_String += " ";
                  } // if
               }
            else
               {
               Display_String += Integer.toString (a [i]);
               } // if
            if (b [i] == 2)
               {
               Display_String += ".";
               } // if
            } // for
         } // if

      Form.Set_Text (Display_String);

      final int Key = Form.Get_Keycode ();

      if (Key >= 0)
         {
         Key_Flag = true;
         Key_Buffer = Key;
         Key_Time = System.currentTimeMillis ();
         }
      else
         {
         Key_Flag = false;

         if (System.currentTimeMillis () > Key_Time + 180000)
            {
            // after 3 minutes without key click - switch off
            stop ();
            }
         else if (System.currentTimeMillis () > Key_Time + 1000)
            {
            // after 1 second without key click - reduce virtual CPU speed.
            try
               {
               synchronized (this)
                  {
                  wait (500);
                  } // synchronized
               }
            catch (final InterruptedException Exception)
               {
               //#ifdef DEBUG
               Exception.printStackTrace ();
               //#endif
               } // try
            } // if
         } //if

      //System.out.println ("- Calculator.Handle_IO ()");
      return;
      } // Handle_IO

   /**
    * Parse address field
    * 
    * @model
    */
   private boolean Parse_Address (final String oct)
      {
      String s;

      try
         {
         s = oct.substring (0, 1);
         Group_Read = Integer.parseInt (s, 8);
         s = oct.substring (1, 2);
         Rom_Read = Integer.parseInt (s, 8);
         s = oct.substring (2, 5);
         Pointer_Read = Integer.parseInt (s, 8);
         }
      catch (final NumberFormatException Exception)
         {
         //#ifdef DEBUG
         Exception.printStackTrace ();
         //#endif
         return false;
         } // try

      return true;
      } // Parse_Address

   /**
    * Parse
    * 
    * @model
    */
   private boolean Parse_Opcode (final String bin)
      {
      int i;

      Opcode_Read = 0;
      for (i = 0; i < 10; i++)
         {
         Opcode_Read <<= 1;
         if (bin.charAt (i) == '1')
            {
            Opcode_Read += 1;
            }
         else if (bin.charAt (i) == '.')
            {
            Opcode_Read += 0;
            }
         else
            {
            return false;
            } // if
         } // for
      return true;
      } // Parse_Opcode

   /**
    * Load Calculator State
    * 
    * @model name="Load"
    */
   protected abstract void Load ();

   /**
    * Save Calculator state
    * 
    * @model name="Save"
    */
   protected abstract void Save ();

   /**
    * add support operation
    * 
    * @model
    */
   byte do_add (final byte x, final byte y)
      {
      int res;

      res = x + y + carry;
      if (res > 9)
         {
         res -= 10;
         carry = 1;
         }
      else
         {
         carry = 0;
         } // if
      return (byte) res;
      } // do_add

   /**
    * subtract support operation
    * 
    * @model
    */
   byte do_sub (final byte x, final byte y)
      {
      int res;

      res = (x - y) - carry;
      if (res < 0)
         {
         res += 10;
         carry = 1;
         }
      else
         {
         carry = 0;
         } //if
      return (byte) res;
      } // do_sub
   } // end Calculator

//vim: set nowrap tabstop=8 shiftwidth=3 softtabstop=3 expandtab :
//vim: set textwidth=0 filetype=java foldmethod=marker nospell :