/*
 *   Serial.c
 *
 *   This file is part of Emu48
 *
 *   Copyright (C) 1998 Christoph Gieelink
 *
 */
#include "pch.h"
#include "Emu48.h"
#include "Serial.h"

#define INTERRUPT ((void)(Chipset.SoftInt=TRUE,bInterrupt=TRUE))

static OVERLAPPED os = { 0 };
static HANDLE     hComm = NULL;
static HANDLE     hCThread = NULL;
static DWORD      lSerialThreadId = 0;
static BOOL       bReading = TRUE;
static WORD       wPort = PORT_CLOSE;

static CRITICAL_SECTION csRecv;				// 01.06.98 cg, new, critical section for receive byte

static DWORD WINAPI SerialThread(LPVOID pParam)
{
	DWORD      dwEvent;
	SetCommMask(hComm,EV_RXCHAR);			// event on RX
	while (bReading)
	{
		_ASSERT(hComm != NULL);
		WaitCommEvent(hComm,&dwEvent,NULL);	// wait for serial event
		if (dwEvent & EV_RXCHAR)			// signal char received
		{
			CommReceive();					// get data
			// interrupt request and emulation thread down
			if (Chipset.SoftInt && Chipset.Shutdn)
				ResumeThread(hThread);		// wake up emulation thread
		}
	}
	lSerialThreadId = 0;					// signal serial thread is down
	return 0;
	UNREFERENCED_PARAMETER(pParam);
}


WORD CommConnect(VOID)
{
	return wPort;
}

VOID CommOpen(LPSTR strWirePort,LPSTR strIrPort)
{
	COMMTIMEOUTS CommTimeouts = { MAXDWORD, 0L, 0L, 0L, 0L };

	LPSTR strPort = (Chipset.IORam[IR_CTRL] & EIRU) ? strIrPort : strWirePort;

	if (hComm != NULL)						// port already open
		CloseHandle(hComm);

	if (strcmp(strPort,NO_SERIAL))			// port defined
	{
		hComm = CreateFile(strPort,
			               GENERIC_READ | GENERIC_WRITE,
						   0,
						   NULL,
						   OPEN_EXISTING,
			               FILE_ATTRIBUTE_NORMAL | FILE_FLAG_OVERLAPPED,
						   NULL);

		if(hComm != INVALID_HANDLE_VALUE)
		{
			// 01.06.98 cg, bugfix, initialize critical section
			InitializeCriticalSection(&csRecv);	

			wPort = (Chipset.IORam[IR_CTRL] & EIRU) ? PORT_IR : PORT_WIRE;
			SetCommTimeouts(hComm,&CommTimeouts);
			CommSetBaud();

			// set event RXD handler
			bReading = TRUE;
			hCThread = CreateThread(NULL,0,(LPTHREAD_START_ROUTINE)&SerialThread,NULL,0,&lSerialThreadId);
			_ASSERT(lSerialThreadId);
		}
		else
			hComm = NULL;
	}

	#if defined DEBUG_SERIAL
	{
		char buffer[256];
		wsprintf(buffer,"COM port %s.\n",hComm ? "opened": "open error");
		OutputDebugString(buffer);
	}
	#endif
}

VOID CommClose(VOID)
{
	if (hComm != NULL)						// port open
	{
		bReading = FALSE;					// kill read thread
		SetCommMask(hComm,0L);				// clear all events and force WaitCommEvent to return
		while (lSerialThreadId != 0) Sleep(0); // wait for termination
		CloseHandle(hComm);					// close port
		hComm = NULL;
		#if defined DEBUG_SERIAL
			OutputDebugString("COM port closed.\n");
		#endif
		DeleteCriticalSection(&csRecv);		// 01.06.98 cg, bugfix, release critical section
		wPort = PORT_CLOSE;
	}
}

VOID CommSetBaud(VOID)
{
	if (hComm != NULL)
	{
		const DWORD dwBaudrates[] = { 1200, 1920, 2400, 3840, 4800, 7680, 9600, 15360 };

		DCB dcb;

		FillMemory(&dcb,sizeof(dcb),0);
		dcb.DCBlength = sizeof(dcb); 
		dcb.BaudRate = dwBaudrates[Chipset.IORam[BAUD] & 0x7];
		dcb.fBinary = TRUE;
		dcb.fParity = TRUE;
		dcb.fOutxCtsFlow = FALSE;
		dcb.fOutxDsrFlow = FALSE;
		dcb.fDtrControl = DTR_CONTROL_DISABLE;
		dcb.fDsrSensitivity = FALSE;
		dcb.fOutX = FALSE;
		dcb.fErrorChar = FALSE;
		dcb.fNull = FALSE;
		dcb.fRtsControl = RTS_CONTROL_DISABLE;
		dcb.fAbortOnError = FALSE;			// may changed in further implementations
		dcb.ByteSize = 8;
		dcb.Parity = NOPARITY;				// no parity check, emulated by software
		dcb.StopBits = ONESTOPBIT;

		#if defined DEBUG_SERIAL
		{
			char buffer[256];
			wsprintf(buffer,"CommsetBaud: %ld\n",dcb.BaudRate);
			OutputDebugString(buffer);
		}
		#endif

		SetCommState(hComm,&dcb);
	}
}

VOID CommTransmit(VOID)
{
	DWORD dwWritten;
	
	BYTE tbr = (Chipset.IORam[TBR_MSB] << 4) | Chipset.IORam[TBR_LSB];

	#if defined DEBUG_SERIAL
	{
		char buffer[256];
		if (isprint(tbr))
			wsprintf(buffer,"-> '%c'\n",tbr);
		else
			wsprintf(buffer,"-> %02X\n",tbr);
		OutputDebugString(buffer);
	}
	#endif

	if (hComm != NULL)						// wire com port open
		WriteFile(hComm,(LPCVOID) &tbr,1,&dwWritten,&os);

	Chipset.IORam[TCS] &= (~TBF);			// clear transmit buffer
	if (Chipset.IORam[IO_CTRL] & ETBE)		// interrupt on transmit buffer empty
		INTERRUPT;
}

VOID CommReceive(VOID)
{
	static BYTE  cBuffer[128];
	static WORD  nRp;
	static DWORD dwBytesRead = 0L;

	if (hComm == NULL)						// not open
		return;

	if (!(Chipset.IORam[IO_CTRL] & SON))	// UART off
	{
		dwBytesRead = 0L;					// no bytes received
		return;
	}

	EnterCriticalSection(&csRecv);			// 01.06.98 cg, bugfix, synchronize
	do
	{
		if (Chipset.IORam[RCS] & RBF)		// receive buffer full
			break;

		if (dwBytesRead == 0L)				// buffer empty
		{
			if(ReadFile(hComm,&cBuffer,sizeof(cBuffer),&dwBytesRead,&os) == FALSE)
				dwBytesRead = 0L;
			else							// bytes received
				nRp = 0;					// reset read pointer
		}

		if(dwBytesRead == 0L)				// receive buffer empty
			break;

		#if defined DEBUG_SERIAL
		{
			char buffer[256];
			if (isprint(cBuffer[nRp]))
				wsprintf(buffer,"<- '%c'\n",cBuffer[nRp]);
			else
				wsprintf(buffer,"<- %02X\n",cBuffer[nRp]);
			OutputDebugString(buffer);
		}
		#endif

		Chipset.IORam[RBR_MSB] = (cBuffer[nRp] >> 4);
		Chipset.IORam[RBR_LSB] = (cBuffer[nRp] & 0x0f);
		++nRp;
		--dwBytesRead;

		Chipset.IORam[RCS] |= RBF;			// receive buffer full
		if (Chipset.IORam[IO_CTRL] & ERBF)	// interrupt on recv buffer full
			INTERRUPT;
	}
	while(0);
	LeaveCriticalSection(&csRecv);			// 01.06.98 cg
}
