// configdlg.c
#include <windows.h>
#include <stdio.h>
#include <crtdbg.h>
#include <commctrl.h>
#include "resource.h"
#include "config.h"
#include "rominfo.h"
#include "bootmsx.h"
#include "error.h"
#include "mem.h"

extern HINSTANCE	g_hInstance ;
extern const char	g_szAppName[] ;

static int Validate (HWND, int) ;
BOOL CALLBACK SetupGeneralDlgProc (HWND, UINT, WPARAM, LPARAM) ;
BOOL CALLBACK SetupCartridgesDlgProc (HWND, UINT, WPARAM, LPARAM) ;
BOOL CALLBACK SetupDiskDrivesDlgProc (HWND, UINT, WPARAM, LPARAM) ;
BOOL CALLBACK SetupColoursDlgProc (HWND, UINT, WPARAM, LPARAM) ;
BOOL CALLBACK SetupJoysticksDlgProc (HWND, UINT, WPARAM, LPARAM) ;
BOOL CALLBACK SetupMemoryDlgProc (HWND, UINT, WPARAM, LPARAM) ;
BOOL CALLBACK SetupSoundDlgProc (HWND, UINT, WPARAM, LPARAM) ;

#define C_TABS		7

static CONFIG			cf ;
static char				szName[CONFIG_MAX_NAMELEN] ;
static const DLGPROC	TabProcs[C_TABS] = { SetupGeneralDlgProc, SetupCartridgesDlgProc,
	SetupDiskDrivesDlgProc, SetupColoursDlgProc, SetupJoysticksDlgProc, 
	SetupMemoryDlgProc, SetupSoundDlgProc } ;

/////////////////////////////////////////////////////////////////////////////////////////
// save config dialog

BOOL CALLBACK SaveConfigDlgProc (HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
	{
	static char		*szName ;
	char			szBuf[256] ;
	
	switch (message)
		{
		case WM_INITDIALOG:
			szName = (char*)lParam ;

			SetWindowText (GetDlgItem (hDlg, IDC_MEMSAVE), szName) ;
			EnableWindow (GetDlgItem (hDlg, IDOK), szName[0] != 0) ;

			return TRUE ;

		case WM_COMMAND:
			switch (LOWORD (wParam) )
				{
				case IDOK:
					GetWindowText (GetDlgItem (hDlg, IDC_MEMSAVE), szBuf, sizeof (szBuf) ) ;
					if (strlen (szBuf) >= 64)
						{
						MessageBox (hDlg, "Please do not type more than 63 characters here.", 
							NULL, MB_ICONINFORMATION) ;
						
						return TRUE ;
						}

					strcpy (szName, szBuf) ;

				case IDCANCEL:
					EndDialog (hDlg, (LOWORD (wParam) == IDOK) ) ;

					return TRUE ;

				case IDC_MEMSAVE:
					if (EN_CHANGE == HIWORD (wParam) )
						{
						GetWindowText (GetDlgItem (hDlg, IDC_MEMSAVE), szBuf, sizeof (szBuf) ) ;

						EnableWindow (GetDlgItem (hDlg, IDOK), szBuf[0] != 0) ;
						
						return TRUE ;
						}

					break ;

				}

			break ;
		}

	return FALSE ;
	}

/////////////////////////////////////////////////////////////////////////////////////////
// Main dialog

static int			iLastSelect ;
static HWND			hwndConfig ;

static struct
	{
	HWND		hwndTab, hwndDisplay ;
	DLGTEMPLATE	*apRes[C_TABS] ;
	} tabctrl ;

static void InitTabControl (HWND hDlg)
	{
	TC_ITEM		tie ;
	int			i ;
	HRSRC		hrsrc ;
	HGLOBAL		hglb ;

	tabctrl.hwndTab = GetDlgItem (hDlg, IDC_MAINTAB) ;
	tabctrl.hwndDisplay = NULL ;
	tie.mask = TCIF_TEXT | TCIF_IMAGE ;
	tie.iImage = -1 ;
	tie.pszText = "General" ;
	TabCtrl_InsertItem (GetDlgItem (hDlg, IDC_MAINTAB), 0, &tie) ;
	tie.pszText = "Cartridges" ;
	TabCtrl_InsertItem (GetDlgItem (hDlg, IDC_MAINTAB), 1, &tie) ;
	tie.pszText = "Disk Drives" ;
	TabCtrl_InsertItem (GetDlgItem (hDlg, IDC_MAINTAB), 2, &tie) ;
	tie.pszText = "Colours" ;
	TabCtrl_InsertItem (GetDlgItem (hDlg, IDC_MAINTAB), 3, &tie) ;
	tie.pszText = "Joysticks" ;
	TabCtrl_InsertItem (GetDlgItem (hDlg, IDC_MAINTAB), 4, &tie) ;
	tie.pszText = "Memory" ;
	TabCtrl_InsertItem (GetDlgItem (hDlg, IDC_MAINTAB), 5, &tie) ;
	tie.pszText = "Sound" ;
	TabCtrl_InsertItem (GetDlgItem (hDlg, IDC_MAINTAB), 6, &tie) ;

	for (i=0;i<C_TABS;i++)
		{
		hrsrc = FindResource (NULL, MAKEINTRESOURCE (IDD_SETUPGENERAL + i), RT_DIALOG) ;
		hglb = LoadResource (g_hInstance, hrsrc) ;
		tabctrl.apRes[i] = LockResource (hglb) ;
		}
	}

static void OnSelChanged (HWND hDlg, int iSel)
	{
	if (tabctrl.hwndDisplay != NULL)
		DestroyWindow (tabctrl.hwndDisplay) ;

	tabctrl.hwndDisplay = CreateDialogIndirect (g_hInstance, 
		tabctrl.apRes[iSel], hDlg, TabProcs[iSel]) ;
	}

BOOL CALLBACK SetupMainDlgProc (HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
	{
	static int			iLastPage ;
	LPNMHDR				pnmh ;
	static FILECONFIG	*pcf ;
	static const char	sID[4] = "SVCF" ;
	int					i, n ;
	static BOOL			fFileChanged ;
	char				szBuf[64] ;
	BOOL				fFound ;

	switch (message)
		{
		case WM_INITDIALOG:
			hwndConfig = hDlg ;

			InitTabControl (hDlg) ;
			TabCtrl_SetCurSel (tabctrl.hwndTab, iLastPage) ;
			OnSelChanged (hDlg, iLastPage) ;

			// saved memory layouts
			LoadConfigs (hDlg, &pcf, &i) ;

			fFileChanged = FALSE ;

			i = pcf->iCount ;

			while (i--)
				{
				SendDlgItemMessage (hDlg, IDC_MAINCONFIGCMB, CB_ADDSTRING, 0, 
					(LPARAM)(LPCSTR) pcf->scf[i].szName) ;
				}
			
			iLastSelect = -1 ;

			return TRUE ;
		
		case WM_NOTIFY:
			pnmh = (LPNMHDR) lParam ;

			switch (pnmh->code)
				{
				case TCN_SELCHANGE:
					iLastPage = Validate (hDlg, iLastPage) ;
					if (iLastPage == -1)
						iLastPage = TabCtrl_GetCurSel (tabctrl.hwndTab) ;

					OnSelChanged (hDlg, iLastPage) ;
					TabCtrl_SetCurSel (tabctrl.hwndTab, iLastPage) ;

					return TRUE ;
				}

			break ;
		
		case WM_COMMAND:
			switch (LOWORD (wParam) )
				{
				case IDOK:
					i = Validate (hDlg, iLastPage) ;
					if (i != -1)
						{
						TabCtrl_SetCurSel (tabctrl.hwndTab, i) ;
						OnSelChanged (hDlg, i) ;
						iLastPage = i ;

						return TRUE ;
						}

					pcf->scfCurrent.cf = cf ;
					i = SendDlgItemMessage (hDlg, IDC_MAINCONFIGCMB, CB_GETCURSEL, 0, 0) ;
					SendDlgItemMessage (hDlg, IDC_MAINCONFIGCMB, CB_GETLBTEXT, i, 
						(LPARAM)(LPSTR) pcf->scfCurrent.szName) ;

				case IDCANCEL:
					EndDialog (hDlg, (LOWORD (wParam) == IDOK) ) ;

					return TRUE ;

				case IDC_MAINCONFIGCMB:
					if (CBN_SELCHANGE == HIWORD (wParam) )
						{
						i = (int)SendMessage ((HWND)lParam, CB_GETCURSEL, 0, 0) ;
						if (i == -1)
							break ;
						SendMessage ((HWND)lParam, CB_GETLBTEXT, i, 
							(LPARAM)(LPSTR) szBuf) ;

						for (i=0;i<pcf->iCount;i++)
							{
							if (!strcmp (szBuf, pcf->scf[i].szName) )
								{
								cf = pcf->scf[i].cf ;
								
								OnSelChanged (hDlg, iLastPage) ;

								return TRUE ;
								}
							}

						MessageBox (hDlg, "Name not found!", NULL, MB_ICONINFORMATION) ;
						}
					
					break ;

				case IDC_MAINSAVEAS:
					i = SendDlgItemMessage (hDlg, IDC_MAINCONFIGCMB, CB_GETCURSEL, 0, 0) ;
					if (i == -1)
						i = iLastSelect ;

					if (i != -1)
						SendDlgItemMessage (hDlg, IDC_MAINCONFIGCMB, CB_GETLBTEXT, i, (LPARAM)szBuf) ;
					else
						szBuf[0] = 0 ;

					if (!DialogBoxParam (g_hInstance, MAKEINTRESOURCE (IDD_SAVECONFIG), 
						hDlg, SaveConfigDlgProc, (LPARAM) szBuf) )
							return TRUE ;

					_ASSERT (szBuf[0] != 0) ;

					fFound = FALSE ;
					for (i=0;i<pcf->iCount;i++)
						{
						if (!strcmp (szBuf, pcf->scf[i].szName) )
							{
							n = i ;
							fFound = TRUE ;
							break ;
							}
						}

					if (!fFound)
						{
						n = pcf->iCount ;
						SendDlgItemMessage (hDlg, IDC_MAINCONFIGCMB, CB_ADDSTRING, 0, (LPARAM)(LPSTR)szBuf) ;

						pcf->iCount++ ;
						pcf = realloc (pcf, pcf->iCount * sizeof (SAVECONFIG) + sizeof (FILECONFIG) ) ;
						strcpy (pcf->scf[n].szName, szBuf) ;
						}

					SendDlgItemMessage (hDlg, IDC_MAINCONFIGCMB, CB_SELECTSTRING, (WPARAM)-1, (LPARAM)(LPSTR)szBuf) ;

					pcf->scf[n].cf = cf ;

					return TRUE ;
				
				case IDC_MAINDELETE:
					i = (int)SendDlgItemMessage (hDlg, IDC_MAINCONFIGCMB, CB_GETCURSEL, 0, 0) ;
					if (i == -1)
						{
						MessageBox (hDlg, "Select memory block to delete.", 
							NULL, MB_ICONINFORMATION) ;

						return TRUE ;
						}

					SendDlgItemMessage (hDlg, IDC_MAINCONFIGCMB, CB_GETLBTEXT, i, 
						(LPARAM)(LPSTR) szBuf) ;

					for (i=0;i<pcf->iCount;i++)
						{
						if (!strcmp (szBuf, pcf->scf[i].szName) )
							{
							n = i ;
							pcf->iCount-- ;
							
							while (n < pcf->iCount)
								{
								pcf->scf[n] = pcf->scf[n + 1] ;
								n++ ;
								}
							
							i = SendDlgItemMessage (hDlg, IDC_MAINCONFIGCMB, CB_FINDSTRINGEXACT, 
								(WPARAM) -1, (LPARAM)(LPSTR) szBuf) ;
							SendDlgItemMessage (hDlg, IDC_MAINCONFIGCMB, CB_DELETESTRING, i, 0) ;

							iLastSelect = -1 ;
							
							return TRUE ;
							}
						}

					MessageBox (hDlg, "Name not found!", NULL, MB_ICONINFORMATION) ;
					
					return TRUE ;
				}

			break ;

		case WM_DESTROY:
			DestroyWindow (tabctrl.hwndDisplay) ;
			
			SaveConfigs (hDlg, pcf, ( (pcf->iCount - 1) * sizeof (SAVECONFIG) + sizeof (FILECONFIG) ) ) ;

			free (pcf) ;

			return TRUE ;
		}

	return FALSE ;
	}

void OnConfigDialog (HWND hwndMain)
	{
	GetCurrentConfig (&cf, szName) ;
	
	if (DialogBox (g_hInstance, MAKEINTRESOURCE (IDD_SETUPMAIN), hwndMain, SetupMainDlgProc) )
		SetCurrentConfig (&cf, szName) ;
	}

static void OnChildDialogInit (HWND hDlg)
	{
	HWND		hwndParent ;
	RECT		rc ;

	hwndParent = GetParent (hDlg) ;
	GetWindowRect (tabctrl.hwndTab, &rc) ;
	TabCtrl_AdjustRect (tabctrl.hwndTab, FALSE, &rc) ;
	ScreenToClient (hwndParent, (LPPOINT)&rc.left) ;
	ScreenToClient (hwndParent, (LPPOINT)&rc.right) ;

	SetWindowPos (hDlg, HWND_TOP, rc.left, rc.top, 
		0, 0, SWP_NOSIZE) ;
	}

static void ConfigChanged ()
	{
	int		i ;
	
	i = SendDlgItemMessage (hwndConfig, IDC_MAINCONFIGCMB, CB_GETCURSEL, 0, 0) ;
	if (i != -1)
		iLastSelect = i ;

	SendDlgItemMessage (hwndConfig, IDC_MAINCONFIGCMB, CB_SETCURSEL, (WPARAM)-1, 0) ;
	}

static BOOL GetCartridge (HWND) ;
static BOOL CheckMemoryLayout (HWND) ;

static int Validate (HWND hDlg, int iPage)
	{
	switch (iPage)
		{
		case 1: // cartridges page
			if (!GetCartridge (tabctrl.hwndDisplay) )
				return 1 ;

			break ;
		}

	if (!CheckMemoryLayout (hDlg) )
		return 5 ;
	
	return -1 ;
	}

/////////////////////////////////////////////////////////////////////////////////////////
// general tab

static void SetVRAMsize (HWND hDlg)
	{
	static const char	*pszVRAMsizeNot1[] = { "64 Kb", "128 Kb *", "196 Kb" } ;
	int		i ;
	
	if (cf.iMSXversion == CONFIG_VERSION_MSX1)
		{
		if (SendDlgItemMessage (hDlg, IDC_VRAMCMB, CB_GETCOUNT, 0, 0) == 3)
			cf.iVRAMsize = (char)SendDlgItemMessage (hDlg, IDC_VRAMCMB, CB_GETCURSEL, 0, 0) ;
		
		SendDlgItemMessage (hDlg, IDC_VRAMCMB, CB_RESETCONTENT, 0, 0) ;
		
		SendDlgItemMessage (hDlg, IDC_VRAMCMB, CB_ADDSTRING, 0, 
			(LPARAM) (LPCSTR) "16 Kb") ;
		SendDlgItemMessage (hDlg, IDC_VRAMCMB, CB_SETCURSEL, 0, 0) ;
	
		EnableWindow (GetDlgItem (hDlg, IDC_VRAMCMB), FALSE) ;
		}
	else
		{
		if (SendDlgItemMessage (hDlg, IDC_VRAMCMB, CB_GETCOUNT, 0, 0) != 3)
			{
			SendDlgItemMessage (hDlg, IDC_VRAMCMB, CB_RESETCONTENT, 0, 0) ;
			
			for (i=0;i<3;i++)
				SendDlgItemMessage (hDlg, IDC_VRAMCMB, CB_ADDSTRING, 0, 
					(LPARAM) (LPCSTR) pszVRAMsizeNot1[i]) ;
			
			SendDlgItemMessage (hDlg, IDC_VRAMCMB, CB_SETCURSEL, cf.iVRAMsize, 0) ;
			
			EnableWindow (GetDlgItem (hDlg, IDC_VRAMCMB), TRUE) ;
			}
		}
	}

static void SetFPSText (HWND hDlg, int iPos)
	{
	char		szBuf[256] ;
	
	wsprintf (szBuf, "Frames per Second: %d", 60 / (iPos + 1) ) ;
	SetWindowText (GetDlgItem (hDlg, IDC_VIDEOFPS), szBuf) ;
	}

BOOL CALLBACK SetupGeneralDlgProc (HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
	{
	static const char	*pszMSXversion[] = { "MSX 1", "MSX 2", "MSX 2+", "MSX Turbo R" } ;
	static const char	*pszMSXlanguage[] = { "International", "Japanese" } ;
	static const char	*pszIntSpeed[] = { "50 Hz (PAL)", "60 Hz (NTSC)", "Set by MSX" } ;
	static const char	*pszSync[] = { "Every Interrupt", "Every Frame Update", "Never (Max Speed)" } ;
	int			i ;

	switch (message)
		{
		case WM_INITDIALOG:
			OnChildDialogInit (hDlg) ;
			// version
			for (i=0;i<4;i++)
				SendDlgItemMessage (hDlg, IDC_COMPUTERCMB, CB_ADDSTRING, 0, 
					(LPARAM) (LPCSTR) pszMSXversion[i]) ;
			
			SendDlgItemMessage (hDlg, IDC_COMPUTERCMB, CB_SETCURSEL, 
				cf.iMSXversion, 0) ;
			
			// language
			for (i=0;i<2;i++)
				SendDlgItemMessage (hDlg, IDC_COMPLANGAUGECMB, CB_ADDSTRING, 0, 
					(LPARAM) (LPCSTR) pszMSXlanguage[i]) ;
			
			SendDlgItemMessage (hDlg, IDC_COMPLANGAUGECMB, CB_SETCURSEL, 
				cf.iMSXlanguage, 0) ;
			
			
			// interrupt speed
			for (i=0;i<3;i++)
				SendDlgItemMessage (hDlg, IDC_INTSPEEDCMB, CB_ADDSTRING, 0, 
					(LPARAM)(LPCSTR) pszIntSpeed[i]) ;
			
			SendDlgItemMessage (hDlg, IDC_INTSPEEDCMB, CB_SETCURSEL, 
				cf.iIntSpeed, 0) ;

			// synchronise
			for (i=0;i<3;i++)
				SendDlgItemMessage (hDlg, IDC_SYNCCMB, CB_ADDSTRING, 0, 
					(LPARAM)(LPCSTR) pszSync[i]) ;

			SendDlgItemMessage (hDlg, IDC_SYNCCMB, CB_SETCURSEL, 
				cf.iSync, 0) ;
			
			// VRAM
			SetVRAMsize (hDlg) ;
			
			// FPS
			SendDlgItemMessage (hDlg, IDC_VIDEOFRAMESSLIDER, TBM_SETRANGE, 
				FALSE, MAKELONG (CONFIG_FPS_60, CONFIG_FPS_10) ) ;
			
			SendDlgItemMessage (hDlg, IDC_VIDEOFRAMESSLIDER, TBM_SETPOS, 
				TRUE, 5 - cf.iFPS) ;
			
			SetFPSText (hDlg, cf.iFPS) ;
			
			// Illegal Sprites
			SendDlgItemMessage (hDlg, IDC_VIDEODISPLAYCHK, BM_SETCHECK, cf.fIllegalSprites, 0) ;
			
			
			return FALSE ;
		
		case WM_COMMAND:
			switch ((int)LOWORD (wParam) )
				{
				case IDC_COMPUTERCMB:
					if (HIWORD (wParam) == CBN_SELCHANGE)
						{
						i = SendDlgItemMessage (hDlg, IDC_COMPUTERCMB, CB_GETCURSEL, 0, 0) ;
						if ( (char)i != cf.iMSXversion)
							{
							cf.iMSXversion = (char)i ;

							ConfigChanged () ;

							SetVRAMsize (hDlg) ;
							}
						}
					
					return TRUE ;

				case IDC_COMPLANGAUGECMB:
					if (HIWORD (wParam) == CBN_SELCHANGE)
						{
						i = SendDlgItemMessage (hDlg, IDC_COMPLANGAUGECMB, CB_GETCURSEL, 0, 0) ;
						if ( (char)i != cf.iMSXlanguage)
							{
							cf.iMSXlanguage = (char)i ;

							ConfigChanged () ;
							}

						}

					return TRUE ;

				case IDC_VRAMCMB:
					if (HIWORD (wParam) == CBN_SELCHANGE)
						{
						i = SendDlgItemMessage (hDlg, IDC_VRAMCMB, CB_GETCURSEL, 0, 0) ;
						if ( (char)i != cf.iVRAMsize)
							{
							cf.iVRAMsize = (char)i ;

							ConfigChanged () ;
							}
						}

					return TRUE ;
				
				case IDC_INTSPEEDCMB:
					if (HIWORD (wParam) == CBN_SELCHANGE)
						{
						i = SendDlgItemMessage (hDlg, IDC_INTSPEEDCMB, CB_GETCURSEL, 0, 0) ;
						if ( (char)i != cf.iIntSpeed)
							{
							cf.iIntSpeed = (char)i ;

							ConfigChanged () ;
							}

						}

					return TRUE ;

				case IDC_SYNCCMB:
					if (HIWORD (wParam) == CBN_SELCHANGE)
						{
						i = SendDlgItemMessage (hDlg, IDC_SYNCCMB, CB_GETCURSEL, 0, 0) ;
						if ( (char)i != cf.iSync)
							{
							cf.iSync = (char)i ;

							ConfigChanged () ;
							}

						}

					return TRUE ;

				case IDC_VIDEODISPLAYCHK:
					i = SendDlgItemMessage (hDlg, IDC_VIDEODISPLAYCHK, BM_GETCHECK, 0, 0) ;
					if ( (BOOL)i != cf.fIllegalSprites)
						{
						cf.fIllegalSprites = (BOOL)i ;

						ConfigChanged () ;
						}

					return TRUE ;
				}
			
			break ;
		
		case WM_HSCROLL:
			if ((HWND)lParam == GetDlgItem (hDlg, IDC_VIDEOFRAMESSLIDER) )
				{
				i = 5 - SendDlgItemMessage (hDlg, IDC_VIDEOFRAMESSLIDER, TBM_GETPOS, 0, 0) ;
				if ( (char)i != cf.iFPS)
					{
					ConfigChanged () ;
					SetFPSText (hDlg, i) ;
					cf.iFPS = i ;
					}
				
				return TRUE ;
				}
			
			break ;
		}
	
	return FALSE ;
	}

/////////////////////////////////////////////////////////////////////////////////////////
// rominfo dialog

static void DoubleAmpersand (char *szString)
	{
	int			i, n ;

	i = 0 ;
	while (szString[i])
		{
		if (szString[i] == '&')
			{
			n = strlen (szString) ;
			while (n >= i)
				{
				szString[n + 1] = szString[n] ;
				n-- ;
				}
			i++ ;
			}
		i++ ;
		}
	}

BOOL CALLBACK ROMInfoDlgProc (HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
	{
	ROMINFO		*prominfo ;
	char		szBuf[256] ;
	BOOL		fFirst ;
	
	switch (message)
		{
		case WM_INITDIALOG:
			prominfo = (ROMINFO*)lParam ;

			// title and producer
			strcpy (szBuf, prominfo->szTitle) ;
			DoubleAmpersand (szBuf) ;
			SetWindowText (GetDlgItem (hDlg, IDC_TITLE), szBuf) ;
			strcpy (szBuf, prominfo->szProducer) ;
			DoubleAmpersand (szBuf) ;
			SetWindowText (GetDlgItem (hDlg, IDC_COMPANY), szBuf) ;
			
			// computer
			fFirst = TRUE ;
			if (prominfo->iFeatures & RI_MSX1)
				{
				strcpy (szBuf, "MSX 1") ;
				fFirst = FALSE ;
				}
			
			if (prominfo->iFeatures & RI_MSX2)
				{
				if (fFirst)
					{
					strcpy (szBuf, "MSX 2") ;
					fFirst = FALSE ;
					}
				else
					strcat (szBuf, " or MSX 2") ;
				}

			if (prominfo->iFeatures & RI_MSX2P)
				{
				if (fFirst)
					{
					strcpy (szBuf, "MSX 2+") ;
					fFirst = FALSE ;
					}
				else
					strcat (szBuf, " or MSX 2+") ;
				}
			
			SetWindowText (GetDlgItem (hDlg, IDC_COMPUTER), szBuf) ;

			// memory
			wsprintf (szBuf, "%d Kb ROM", prominfo->iSize) ;
			
			if (prominfo->iFeatures & RI_SRAM)
				strcat (szBuf, " and 8 Kb SRAM") ;

			SetWindowText (GetDlgItem (hDlg, IDC_MEMORY), szBuf) ;

			// sound
			SetWindowText (GetDlgItem (hDlg, IDC_SOUNDCHIP), 
				(prominfo->iFeatures & RI_SCC ? "SCC" : "None") ) ;

			// RC
			wsprintf (szBuf, "RC %3d", prominfo->nRC) ;
			SetWindowText (GetDlgItem (hDlg, IDC_RC), szBuf) ;
			
			return TRUE ;

		case WM_COMMAND:
			switch (LOWORD (wParam) )
				{
				case IDOK:
				case IDCANCEL:
					EndDialog (hDlg, TRUE) ;

					return TRUE ;
				}

			break ;
		}

	return FALSE ;
	}

static BOOL OnROMInfo (HWND hDlg, char *szFile)
	{
	BYTE	*p ;
	int		iSize ;
	ROMINFO	rominf ;
	
	if (LoadROMfile (hDlg, szFile, &p, &iSize, ROM_CART) )
		{
		if (!GetROMInfo (p, &rominf) )
			{
			MessageBoxID  (hDlg, IDS_NOINFORMATIONONROM, MB_ICONEXCLAMATION) ;

			MemoryFree (&p) ;

			return FALSE ;
			}

		DialogBoxParam (g_hInstance, MAKEINTRESOURCE (IDD_ROMINFO), hDlg, 
			ROMInfoDlgProc, (LPARAM) &rominf) ;
		
		MemoryFree (&p) ;

		return TRUE ;
		}

	return FALSE ;
	}

/////////////////////////////////////////////////////////////////////////////////////////
// cartridges tab

static int		iCurCart ;

static void EnableOptions (HWND hDlg, int iCart)
	{
	BOOL	fEnable, fEnable2 ;
	int		i ;
	
	fEnable = FALSE ;
	for (i=0;i<4;i++)
		if (cf.cart[i].fEnabled)
			{
			fEnable = TRUE ;
			break ;
			}

	EnableWindow (GetDlgItem (hDlg, IDC_CARTRIDGETYPE), fEnable) ;
	EnableWindow (GetDlgItem (hDlg, IDC_CARTRIDGETYPECMB), fEnable) ;

	if (fEnable)
		fEnable2 = (cf.cart[iCart].iType < CONFIG_CARTRIDGE_THSCCP) ;
	else
		fEnable2 = FALSE ;

	for (i=IDC_CARTRIDGEROMFILE;i<=IDC_CARTRIDGESUGGESTBTN;i++)
		EnableWindow (GetDlgItem (hDlg, i), fEnable2) ;
	
	if (fEnable)
		{
		switch (cf.cart[iCart].iType)
			{
			case CONFIG_CARTRIDGE_GM2:
			case CONFIG_CARTRIDGE_FM_PAC:
			case CONFIG_CARTRIDGE_ASCII_8KB_SRAM:
				fEnable2 = TRUE ;
				break ;
			default:
				fEnable2 = FALSE ;
			}
		}
	else
		fEnable2 = FALSE ;
	
	for (i=IDC_CARTRIDGESRAMFILE;i<=IDC_CARTRIDGESRAMBROWSE;i++)
		EnableWindow (GetDlgItem (hDlg, i), fEnable2) ;
	}

static void SetCartridge (HWND hDlg, int iCart)
	{
	SetWindowText (GetDlgItem (hDlg, IDC_CARTRIDGEROMEDIT), cf.cart[iCart].szROMfile) ;
	SetWindowText (GetDlgItem (hDlg, IDC_CARTRIDGESRAMEDIT), cf.cart[iCart].szSRAMfile) ;

	SendDlgItemMessage (hDlg, IDC_CARTRIDGETYPECMB, CB_SETCURSEL, 
		cf.cart[iCart].iType, 0) ;
	}

static BOOL GetCartridge (HWND hDlg)
	{
	char		szBuf[_MAX_PATH] ;

	GetWindowText (GetDlgItem (hDlg, IDC_CARTRIDGESRAMEDIT), szBuf, sizeof (szBuf) ) ;
	if (strcmp (szBuf, cf.cart[iCurCart].szSRAMfile) )
		{
		strcpy (cf.cart[iCurCart].szSRAMfile, szBuf) ;
		ConfigChanged () ;
		}
	
	GetWindowText (GetDlgItem (hDlg, IDC_CARTRIDGEROMEDIT), szBuf, sizeof (szBuf) ) ;
	if (strcmp (szBuf, cf.cart[iCurCart].szROMfile) )
		{
		strcpy (cf.cart[iCurCart].szROMfile, szBuf) ;
		ConfigChanged () ;
		}
	
	return TRUE ;
	}

static BOOL OnSuggest (HWND hDlg, char *szFile, int iCart)
	{
	BYTE	*p ;
	int		i, iSize ;
	ROMINFO	rominf ;
	
	if (LoadROMfile (hDlg, szFile, &p, &iSize, ROM_CART) )
		{
		if (!GetROMInfo (p, &rominf) )
			{
			MemoryFree (&p) ;

			return FALSE ;
			}

		MemoryFree (&p) ;

		if (iSize <= 0x8000)
			i = CONFIG_CARTRIDGE_STANDARD ;
		else if (rominf.iFeatures & RI_SRAM)
			i = CONFIG_CARTRIDGE_GM2 ;
		else if (rominf.iFeatures & RI_SCC)
			i = CONFIG_CARTRIDGE_KONAMI5 ;
		else
			i = CONFIG_CARTRIDGE_KONAMI4 ;

		SendDlgItemMessage (hDlg, IDC_CARTRIDGETYPECMB, CB_SETCURSEL, i, 0) ;
		cf.cart[iCart].iType = i ;
		ConfigChanged () ;

		EnableOptions (hDlg, iCart) ;

		return TRUE ;
		}

	return FALSE ;
	}

static BOOL BrowseSRAM (HWND hDlg, char *szFile)
	{
	OPENFILENAME		ofn ;
	char				szBuf[_MAX_PATH] ;

	strcpy (szBuf, szFile) ;
	
	memset (&ofn, 0, sizeof (ofn) ) ;

	ofn.lStructSize = sizeof (ofn) ;
	ofn.hwndOwner   = hDlg ;
	ofn.lpstrFilter = "SRAM images (*.sram)\0*.SRAM\0"
		"Virtual MSX 1.x SRAM (*.ram)\0*.RAM\0FM-PAC Files (*.pac)\0*.pac\0"
		"All Files\0*\0" ;
	ofn.lpstrFile   = szBuf ;
	ofn.nMaxFile    = sizeof (szBuf) ;
	ofn.lpstrTitle  = "SRAM Browse" ;
	ofn.Flags       = OFN_HIDEREADONLY | OFN_LONGNAMES | OFN_PATHMUSTEXIST ;

	if (!GetOpenFileName (&ofn) )
		{
		if (CommDlgExtendedError () == FNERR_INVALIDFILENAME)
			{
			szFile[0] = '\0' ;
			if (!GetOpenFileName (&ofn) )
				return FALSE ;
			}
		else
			return FALSE ;
		}
	
	strcpy (szFile, szBuf) ;

	return TRUE ;
	}

static BOOL BrowseROM (HWND hDlg, char *szFile)
	{
	OPENFILENAME		ofn ;
	char				szBuf[_MAX_PATH] ;

	strcpy (szBuf, szFile) ;
	
	memset (&ofn, 0, sizeof (ofn) ) ;

	ofn.lStructSize = sizeof (ofn) ;
	ofn.hwndOwner   = hDlg ;
	ofn.lpstrFilter = "ROM Images (*.rom)\0*.ROM\0MSX Binaries (*)\0*\0All Files\0*\0" ;
	ofn.lpstrFile   = szBuf ;
	ofn.nMaxFile    = sizeof (szBuf) ;
	ofn.lpstrTitle  = "ROM Browse" ;
	ofn.Flags       = OFN_HIDEREADONLY | OFN_LONGNAMES | OFN_FILEMUSTEXIST ;
	ofn.lpstrDefExt = "ROM" ;

	if (!GetOpenFileName (&ofn) )
		{
		if (CommDlgExtendedError () == FNERR_INVALIDFILENAME)
			{
			szFile[0] = '\0' ;
			if (!GetOpenFileName (&ofn) )
				return FALSE ;
			}
		else
			return FALSE ;
		}
	
	strcpy (szFile, szBuf) ;

	return TRUE ;
	}

BOOL CALLBACK SetupCartridgesDlgProc (HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
	{
	char			szBuf[_MAX_PATH] ;
	int				i ;
	static const char	*szTypes[] = { 
		"Standard ROM", "FM-PAC with SRAM", "MSX DOS 2", 
		"Konami megaROM without SCC (konami4/8Kb)", "Konami megaROM with SCC (konami5/8Kb)", 
		"ASCII 8Kb megaROM", "ASCII 8Kb megaROM with SRAM", "ASCII 16Kb megaROM", 
		"Konami's The Game Master 2 with SRAM", "Irem's R-Type", 
		"Snatcher Sound Cartridge (SCC+)", "Super Deform Snatcher Sound Cartridge (SCC+)",
		"Sound Cartridge with 128Kb RAM (SCC+)",
		"Brazilian MegaRAM 256Kb (original Ademir Carchano)", "Brazilian MegaRAM 512Kb (DDX)", 
		"Brazilian MegaRAM 768Kb (DDX)" } ;
	
	switch (message)
		{
		case WM_INITDIALOG:
			OnChildDialogInit (hDlg) ;

			// enable the stuff
			EnableWindow (GetDlgItem (hDlg, IDC_CARTRIDGEA), cf.cart[0].fEnabled) ;
			EnableWindow (GetDlgItem (hDlg, IDC_CARTRIDGEB), cf.cart[1].fEnabled) ;
			EnableWindow (GetDlgItem (hDlg, IDC_CARTRIDGEC), cf.cart[2].fEnabled) ;
			EnableWindow (GetDlgItem (hDlg, IDC_CARTRIDGED), cf.cart[3].fEnabled) ;
			
			for (i=0;i<16;i++)
				SendDlgItemMessage (hDlg, IDC_CARTRIDGETYPECMB, CB_ADDSTRING, 0, 
					(LPARAM)(LPCSTR)szTypes[i]) ;

			iCurCart = 0 ;
			for (i=0;i<4;i++)
				if (cf.cart[i].fEnabled)
					{
					iCurCart = i ;
					
					break ;
					}

			SetCartridge (hDlg, iCurCart) ;
			EnableOptions (hDlg, iCurCart) ;

			CheckRadioButton (hDlg, IDC_CARTRIDGEA, IDC_CARTRIDGED, IDC_CARTRIDGEA + i) ;

			return FALSE ;
		
		case WM_COMMAND:
			switch (LOWORD (wParam) )
				{
				case IDC_CARTRIDGEA:
				case IDC_CARTRIDGEB:
				case IDC_CARTRIDGEC:
				case IDC_CARTRIDGED:
					if ((int)LOWORD (wParam) != iCurCart)
						{
						if (GetCartridge (hDlg) )
							{
							iCurCart = (int)LOWORD (wParam) - IDC_CARTRIDGEA ;
							SetCartridge (hDlg, iCurCart) ;
							EnableOptions (hDlg, iCurCart) ;
							}
						else
							{
							CheckRadioButton (hDlg, IDC_CARTRIDGEA, IDC_CARTRIDGED, 
								IDC_CARTRIDGEA + iCurCart) ;
							}
						}

					return TRUE ;
				
				case IDC_CARTRIDGETYPECMB:
					if (HIWORD (wParam) == CBN_SELCHANGE)
						{
						i = SendDlgItemMessage (hDlg, IDC_CARTRIDGETYPECMB, CB_GETCURSEL, 0, 0) ;
						
						cf.cart[iCurCart].iType = (char)i ;

						ConfigChanged () ;
						EnableOptions (hDlg, iCurCart) ;
						}

					return TRUE ;
				
				case IDC_CARTRIDGESRAMBROWSE:
					GetWindowText (GetDlgItem (hDlg, IDC_CARTRIDGESRAMEDIT), szBuf, sizeof (szBuf) ) ;
					if (BrowseSRAM (hDlg, szBuf) )
						{
						ConfigChanged () ;

						SetWindowText (GetDlgItem (hDlg, IDC_CARTRIDGESRAMEDIT), szBuf) ;
						}

					return TRUE ;
				
				case IDC_CARTRIDGEROMBROWSE:
					GetWindowText (GetDlgItem (hDlg, IDC_CARTRIDGEROMEDIT), szBuf, sizeof (szBuf) ) ;
					if (BrowseROM (hDlg, szBuf) )
						{
						ConfigChanged () ;
						SetWindowText (GetDlgItem (hDlg, IDC_CARTRIDGEROMEDIT), szBuf) ;
						OnSuggest (hDlg, szBuf, iCurCart) ;
						}
					
					return TRUE ;
				
				case IDC_CARTRIDGEINFOBTN:
					GetWindowText (GetDlgItem (hDlg, IDC_CARTRIDGEROMEDIT), szBuf, sizeof (szBuf) ) ;
					if (strlen (szBuf) == 0)
						MessageBoxID (hDlg, IDS_FILEERRNOFILE, MB_ICONEXCLAMATION) ;
					else
						OnROMInfo (hDlg, szBuf) ;

					return TRUE ;

				case IDC_CARTRIDGESUGGESTBTN:
					GetWindowText (GetDlgItem (hDlg, IDC_CARTRIDGEROMEDIT), szBuf, sizeof (szBuf) ) ;
					if (strlen (szBuf) == 0)
						MessageBoxID (hDlg, IDS_FILEERRNOFILE, MB_ICONEXCLAMATION) ;
					else
						if (!OnSuggest (hDlg, szBuf, iCurCart) )
							MessageBoxID  (hDlg, IDS_NOINFORMATIONONROM, MB_ICONEXCLAMATION) ;

					return TRUE ;
				}
			
			break ;
		}

	return FALSE ;
	}

BOOL CALLBACK SetupDiskDrivesDlgProc (HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
	{
	switch (message)
		{
		case WM_INITDIALOG:
			OnChildDialogInit (hDlg) ;

			return FALSE ;
		}

	return FALSE ;
	}

BOOL CALLBACK SetupColoursDlgProc (HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
	{
	switch (message)
		{
		case WM_INITDIALOG:
			OnChildDialogInit (hDlg) ;

			return FALSE ;
		}

	return FALSE ;
	}

BOOL CALLBACK SetupJoysticksDlgProc (HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
	{
	switch (message)
		{
		case WM_INITDIALOG:
			OnChildDialogInit (hDlg) ;

			return FALSE ;
		}

	return FALSE ;
	}

/////////////////////////////////////////////////////////////////////////////////////////
// memory
#define BLOCKS_DISPLAY	8
#define BLOCKS_LINE		2
#define BLOCKS_GRID		20
#define BLOCKS_SIZE		16
#define LAYOUT_X		400
#define LAYOUT_Y		110
#define SLOTOFFSET		14
#define SLOTSLOTOFFSET	99
#define MEMORY_GRID		10

typedef struct tagLOCATION
	{
	int			iID ;
	short		iBlock ;
	short		iPrim, iSec, iPage ;
	} LOCATION ;

static struct
	{
	HCURSOR		hcurMemory, hcurBomb ;
	LOCATION	locSelect, locOrg ;
	BOOL		fSelect ;
	} mouse ;

static struct
	{
	HBITMAP		hbmpLayout, hbmpBlocks ;
	int			ixOffsetLayout, iyOffsetLayout ;
	HBRUSH		hbrBack ;
	} paint ;

static void DrawBackground (HBITMAP hbmp)
	{
	BITMAP		bmp ;
	HBITMAP		hbmpNew, hbmpOld1, hbmpOld2, hbmpMask ;
	HDC			hdc, hdc1, hdc2 ;
	RECT		rc ;

	GetObject (hbmp, sizeof (bmp), &bmp) ;
	hdc = GetDC (NULL) ;
	hdc1 = CreateCompatibleDC (hdc) ;
	hdc2 = CreateCompatibleDC (hdc) ;
	
	rc.left = rc.top = 0 ;
	rc.right = bmp.bmWidth ;
	rc.bottom = bmp.bmHeight ;

	hbmpOld1 = SelectObject (hdc1, hbmp) ;
	
	hbmpMask = CreateBitmap (bmp.bmWidth, bmp.bmHeight, 1, 1, NULL) ;
	hbmpOld2 = SelectObject (hdc2, hbmpMask) ;
	SetBkColor (hdc1, RGB (255, 0, 255) ) ;
	
	BitBlt (hdc2, 0, 0, bmp.bmWidth, bmp.bmHeight, 
		hdc1, 0, 0, SRCCOPY) ;
	InvertRect (hdc2, &rc) ;
	SetBkColor (hdc1, RGB (255, 255, 255) ) ;

	BitBlt (hdc1, 0, 0, bmp.bmWidth, bmp.bmHeight, 
		hdc2, 0, 0, SRCAND) ;

	InvertRect (hdc2, &rc) ;
	hbmpNew = CreateBitmap (bmp.bmWidth, bmp.bmHeight, 
		bmp.bmPlanes, bmp.bmBitsPixel, NULL) ;
	SelectObject (hdc1, hbmpNew) ;
	FillRect (hdc1, &rc, paint.hbrBack) ;

	BitBlt (hdc1, 0, 0, bmp.bmWidth, bmp.bmHeight, 
		hdc2, 0, 0, SRCAND) ;
	
	SelectObject (hdc2, hbmp) ;

	BitBlt (hdc2, 0, 0, bmp.bmWidth, bmp.bmHeight, 
		hdc1, 0, 0, SRCPAINT) ;

	SelectObject (hdc1, hbmpOld1) ;
	SelectObject (hdc2, hbmpOld2) ;
	DeleteDC (hdc1) ;
	DeleteDC (hdc2) ;
	ReleaseDC (NULL, hdc) ;

	DeleteObject (hbmpNew) ;
	DeleteObject (hbmpMask) ;
	}

static void DoExpanded (HWND hDlg)
	{
	int		iPrim, iSec, iPage ;
	BOOL	fEnable ;
	
	for (iPrim=0;iPrim<4;iPrim++)
		{
		if (!cf.mem.fSlotExpanded[iPrim])
			{
			EnableWindow (GetDlgItem (hDlg, IDC_MEMSLOT0EXPANDCHK + iPrim), TRUE) ;
			SendDlgItemMessage (hDlg, IDC_MEMSLOT0EXPANDCHK + iPrim, BM_SETCHECK, BST_UNCHECKED, 0) ;
			}
		else
			{
			SendDlgItemMessage (hDlg, IDC_MEMSLOT0EXPANDCHK + iPrim, BM_SETCHECK, BST_CHECKED, 0) ;
			fEnable = TRUE ;
			for (iSec=1;iSec<4;iSec++)
				for (iPage=0;iPage<4;iPage++)
					{
					if (cf.mem.iMem[iPrim][iSec][iPage] != CONFIG_MEMORY_EMPTY)
						fEnable = FALSE ;
					}
			EnableWindow (GetDlgItem (hDlg, IDC_MEMSLOT0EXPANDCHK + iPrim), fEnable) ;
			}
		}
	}

static void DrawSlot (HWND hDlg, int iPrim)
	{
	HBITMAP			hbmpOld ;
	int				iSec, iPage, i ;
	HDC				hdc, hdcMem ;

	hdc = GetDC (hDlg) ;
	hdcMem = CreateCompatibleDC (hdc) ;
	hbmpOld = SelectObject (hdcMem, paint.hbmpBlocks) ;

	for (iSec=0;iSec<4;iSec++)
		for (iPage=0;iPage<4;iPage++)
			{
			if (!iSec || cf.mem.iMem[iPrim][iSec][iPage] != CONFIG_MEMORY_EMPTY || cf.mem.fSlotExpanded[iPrim])
				i = cf.mem.iMem[iPrim][iSec][iPage] ;
			else
				i = 14 ;

			BitBlt (hdc, 
				paint.ixOffsetLayout + 2 * BLOCKS_GRID + BLOCKS_LINE + 
					(9 * MEMORY_GRID) * iPrim + BLOCKS_GRID * iSec, 
				paint.iyOffsetLayout + BLOCKS_GRID + BLOCKS_LINE + 
					BLOCKS_GRID * (3 - iPage), 
				BLOCKS_SIZE, BLOCKS_SIZE, hdcMem, 
				i * BLOCKS_SIZE, 0, SRCCOPY) ;
			}

	SelectObject (hdcMem, hbmpOld) ;
	DeleteDC (hdcMem) ;
	DeleteDC (hdc) ;
	}

static void DrawSelection (HWND hDlg, LOCATION *ploc, COLORREF clr)
	{
	HDC		hdc ;
	HBRUSH	hbr ;
	RECT	rc ;

	hdc = GetDC (hDlg) ;
	hbr = CreateSolidBrush (clr) ;

	switch (ploc->iID)
		{
		case IDC_MEMPLURGEBTN:
			GetWindowRect (GetDlgItem (hDlg, IDC_MEMPLURGEBTN), &rc) ;
			ScreenToClient (hDlg, (LPPOINT)&rc.left) ;
			ScreenToClient (hDlg, (LPPOINT)&rc.right) ;
			rc.left = rc.right - BLOCKS_GRID ;
			rc.bottom = rc.top + BLOCKS_GRID ;
			FrameRect (hdc, &rc, hbr) ;
			break ;
			
		case IDC_MEMBLOCKS:
			GetWindowRect (GetDlgItem (hDlg, IDC_MEMBLOCKS), &rc) ;
			ScreenToClient (hDlg, (LPPOINT)&rc.left) ;
			ScreenToClient (hDlg, (LPPOINT)&rc.right) ;
			rc.left += ploc->iBlock * BLOCKS_GRID ;
			rc.right = rc.left + BLOCKS_GRID ;
			rc.bottom = rc.top + BLOCKS_GRID ;
			FrameRect (hdc, &rc, hbr) ;
			break ;
			
		case IDC_MEMORYLAYOUT:
			switch (cf.mem.iMem[ploc->iPrim][ploc->iSec][ploc->iPage])
				{
				case CONFIG_MEMORY_BIOS:
					rc.top = 1 + paint.iyOffsetLayout + 3 * BLOCKS_GRID ;
					rc.left = 1 + paint.ixOffsetLayout + 2 * BLOCKS_GRID ;
					rc.right = rc.left + BLOCKS_GRID - 2 ;
					rc.bottom = rc.top + 2 * BLOCKS_GRID - 2 ;
					FrameRect (hdc, &rc, hbr) ;
					break ;

				case CONFIG_MEMORY_RAMMM:
				case CONFIG_MEMORY_CARTA:
				case CONFIG_MEMORY_CARTB:
				case CONFIG_MEMORY_CARTC:
				case CONFIG_MEMORY_CARTD:
					rc.top = 1 + paint.iyOffsetLayout + BLOCKS_GRID  ;
					rc.left = 1 + paint.ixOffsetLayout + 2 * BLOCKS_GRID + ploc->iPrim * 9 * MEMORY_GRID + ploc->iSec * BLOCKS_GRID ;
					rc.right = rc.left + BLOCKS_GRID - 2 ;
					rc.bottom = rc.top + 4 * BLOCKS_GRID - 2 ;
					FrameRect (hdc, &rc, hbr) ;
					break ;

				default:
					rc.top = 1 + paint.iyOffsetLayout + BLOCKS_GRID + (3 - ploc->iPage) * BLOCKS_GRID ;
					rc.left = 1 + paint.ixOffsetLayout + 2 * BLOCKS_GRID + ploc->iPrim * 9 * MEMORY_GRID + ploc->iSec * BLOCKS_GRID ;
					rc.right = rc.left + BLOCKS_GRID - 2 ;
					rc.bottom = rc.top + BLOCKS_GRID - 2 ;
					FrameRect (hdc, &rc, hbr) ;
					break ;
				}

			break ;
		}

	DeleteObject (hbr) ;
	ReleaseDC (hDlg, hdc) ;
	}

static BOOL PointInRect (RECT *prc, POINT *ppt)
	{
	if (ppt->x < prc->left || ppt->x > prc->right)
		return FALSE ;

	return !(ppt->y < prc->top || ppt->y > prc->bottom) ;
	}

static int GetLocation (HWND hDlg, POINT *ppt, LOCATION *ploc)
	{
	RECT		rc ;
	int			i, x, y ;
	POINT		pt ;

	GetWindowRect (GetDlgItem (hDlg, IDC_MEMPLURGEBTN), &rc) ;
	if (PointInRect (&rc, ppt) )
		{
		ploc->iID = IDC_MEMPLURGEBTN ;

		return IDC_MEMPLURGEBTN ;
		}

	GetWindowRect (GetDlgItem (hDlg, IDC_MEMBLOCKS), &rc) ;
	if (PointInRect (&rc, ppt) )
		{
		i = (ppt->x - rc.left) / BLOCKS_GRID ;
		
		if (i < BLOCKS_DISPLAY)
			{
			ploc->iID = IDC_MEMBLOCKS ;
			ploc->iBlock = i ;

			return IDC_MEMBLOCKS ;
			}

		return 0 ;
		}

	GetWindowRect (GetDlgItem (hDlg, IDC_MEMORYLAYOUT), &rc) ;
	if (PointInRect (&rc, ppt) )
		{
		pt.x = paint.ixOffsetLayout ;
		pt.y = paint.iyOffsetLayout ;
		ClientToScreen (hDlg, &pt) ;
		
		x = (ppt->x - pt.x) / MEMORY_GRID ;
		y = (ppt->y - pt.y) / MEMORY_GRID ;

		if ( !(x > 3 && x < 40 && y > 1 && y < 10) )
			return 0 ;

		ploc->iPage = 3 - ( (y - 2) / 2) ;
		ploc->iPrim = (x - 4) / 9 ;
		
		i = (x - 4) % 9 ;
		if (i == 8)
			return 0 ;
		
		ploc->iSec = i / 2 ;
		ploc->iID = IDC_MEMORYLAYOUT ;
		
		return IDC_MEMORYLAYOUT ;
		}
	
	ploc->iID = 0 ;
	
	return 0 ;
	}

static void SetInformation (HWND hDlg, LOCATION *ploc)
	{
	int		i ;
	char	szBuf[256] ;
	
	if (ploc != NULL)
		{
		switch (ploc->iID)
			{
			case IDC_MEMBLOCKS:
				i = IDS_MEMNEW ;
				break ;

			case IDC_MEMPLURGEBTN:
				i = IDS_MEMDEL ;
				break ;

			case IDC_MEMORYLAYOUT:
				if (ploc->iSec && !cf.mem.fSlotExpanded[ploc->iPrim])
					i = IDS_MEMUNUSED ;
				else
					i = IDS_MEMEMPTY + cf.mem.iMem[ploc->iPrim][ploc->iSec][ploc->iPage] ;
				break ;

			default:
				i = IDS_MEMNOTHINGSELECTED ;
			}
		}
	else 
		i = IDS_MEMNOTHINGSELECTED ;
	
	LoadString (g_hInstance, i, szBuf, sizeof (szBuf) ) ;
	SetWindowText (GetDlgItem (hDlg, IDC_MEMSELECTTEXT), szBuf) ;
	
	if (i == IDS_MEMRAMMM && ploc != NULL)
		{
		EnableWindow (GetDlgItem (hDlg, IDC_MEMSIZECMB), TRUE) ;
		SendDlgItemMessage (hDlg, IDC_MEMSIZECMB, CB_SETCURSEL, 
			cf.mem.iMapperSize[ploc->iPrim * 4 + ploc->iSec], 0) ;
		}
	else
		{
		EnableWindow (GetDlgItem (hDlg, IDC_MEMSIZECMB), FALSE) ;
		SendDlgItemMessage (hDlg, IDC_MEMSIZECMB, CB_SETCURSEL, (UINT)-1, 0) ;
		}
	}

static BOOL IsBlockExclusive (short iBlock)
	{
	switch (iBlock)
		{
		case CONFIG_MEMORY_BIOS:
		case CONFIG_MEMORY_SUBBIOS:
		case CONFIG_MEMORY_DISKROM:
		case CONFIG_MEMORY_CARTA:
		case CONFIG_MEMORY_CARTB:
		case CONFIG_MEMORY_CARTC:
		case CONFIG_MEMORY_CARTD:
			return TRUE ;
		}

	return FALSE ;
	}

static BOOL FindBlock (short iBlock, LOCATION *ploc)
	{
	int		iPrim, iSec, iPage ;

	for (iPrim=0;iPrim<4;iPrim++)
		for (iSec=0;iSec<4;iSec++)
			for (iPage=0;iPage<4;iPage++)
				{
				if (cf.mem.iMem[iPrim][iSec][iPage] == iBlock)
					{
					if (ploc != NULL)
						{
						ploc->iPrim = iPrim ;
						ploc->iSec  = iSec ;
						ploc->iPage = iPage ;
						}

					return TRUE ;
					}
				}

	return FALSE ;
	}

static BOOL AddBlock (HWND hDlg, LOCATION *ploc, short iBlock)
	{
	int			i ;

	if (cf.mem.iMem[ploc->iPrim][ploc->iSec][ploc->iPage] != CONFIG_MEMORY_EMPTY)
		{
		MessageBox (hDlg, "Destination must be empty.", NULL, MB_ICONINFORMATION) ;

		return FALSE ;
		}
	
	switch (iBlock)
		{
		case CONFIG_MEMORY_RAMMM:
		case CONFIG_MEMORY_CARTA:
		case CONFIG_MEMORY_CARTB:
		case CONFIG_MEMORY_CARTC:
		case CONFIG_MEMORY_CARTD:
			for (i=0;i<4;i++)
				if (cf.mem.iMem[ploc->iPrim][ploc->iSec][i] != CONFIG_MEMORY_EMPTY)
					{
					MessageBox (hDlg, "Entire subslot must be empty.", NULL, MB_ICONINFORMATION) ;

					return FALSE ;
					}

			for (i=0;i<4;i++)
				cf.mem.iMem[ploc->iPrim][ploc->iSec][i] = (char)iBlock ;

			if (iBlock == CONFIG_MEMORY_RAMMM)
				cf.mem.iMapperSize[ploc->iPrim * 4 + ploc->iSec] = 2 ;

			break ;

		case CONFIG_MEMORY_BIOS:
			MessageBox (hDlg, "BIOS cannot be moved.", NULL, MB_ICONINFORMATION) ;

			return FALSE ;

		case CONFIG_MEMORY_SUBBIOS:
			if (ploc->iPage != 0)
				{
				MessageBox (hDlg, "SUB BIOS must reside in page 0.", NULL, MB_ICONINFORMATION) ;

				return FALSE ;
				}

			cf.mem.iMem[ploc->iPrim][ploc->iSec][ploc->iPage] = (char)iBlock ;
			
			break ;

		case CONFIG_MEMORY_DISKROM:
			if (ploc->iPage != 1)
				{
				MessageBox (hDlg, "This ROM must reside in page 1.", NULL, MB_ICONINFORMATION) ;

				return FALSE ;
				}

			cf.mem.iMem[ploc->iPrim][ploc->iSec][ploc->iPage] = (char)iBlock ;
			
			break ;

		case CONFIG_MEMORY_EMPTY:
			MessageBox (hDlg, "Cannot move empty memory.", NULL, MB_ICONINFORMATION) ;

			return FALSE ;
		
		default:
			cf.mem.iMem[ploc->iPrim][ploc->iSec][ploc->iPage] = (char)iBlock ;

			break ;
		}
	
	if (ploc->iSec && !cf.mem.fSlotExpanded[ploc->iPrim])
		cf.mem.fSlotExpanded[ploc->iPrim] = TRUE ;
	ConfigChanged () ;

	return TRUE ;
	}

static void DeleteBlock (HWND hDlg, LOCATION *ploc)
	{
	int		i, iSec, iPrim, iPage ;
	BOOL	fFound ;

	switch (cf.mem.iMem[ploc->iPrim][ploc->iSec][ploc->iPage])
		{
		case CONFIG_MEMORY_EMPTY:
		case CONFIG_MEMORY_BIOS:
			MessageBox (hDlg, "Memory cannot be deleted.", NULL, MB_ICONINFORMATION) ;
			return ;

		case CONFIG_MEMORY_RAMMM:
			fFound = FALSE ;
			
			for (iPrim=0;iPrim<4;iPrim++)
				for (iSec=0;iSec<4;iSec++)
					{
					if ( (iSec != ploc->iSec || ploc->iPrim != iPrim) && 
						(cf.mem.iMem[iPrim][iSec][3] == CONFIG_MEMORY_RAM || 
						cf.mem.iMem[iPrim][iSec][3] == CONFIG_MEMORY_RAMMM) )
						{
						fFound = TRUE ;
						}
					}

			if (!fFound)
				{
				if (IDOK != MessageBox (hDlg, "Without RAM in page 3, the MSX cannot operate.", 
					NULL, MB_ICONEXCLAMATION | MB_OKCANCEL) )
					return ;
				}

		case CONFIG_MEMORY_CARTA:
		case CONFIG_MEMORY_CARTB:
		case CONFIG_MEMORY_CARTC:
		case CONFIG_MEMORY_CARTD:
			for (i=0;i<4;i++)
				cf.mem.iMem[ploc->iPrim][ploc->iSec][i] = CONFIG_MEMORY_EMPTY ;

			break ;

		case CONFIG_MEMORY_RAM:
			if (ploc->iPage == 3)
				{
				fFound = FALSE ;
				
				for (iPrim=0;iPrim<4;iPrim++)
					for (iSec=0;iSec<4;iSec++)
						{
						if ( (iSec != ploc->iSec || ploc->iPrim != iPrim) && 
							(cf.mem.iMem[iPrim][iSec][3] == CONFIG_MEMORY_RAM || 
							cf.mem.iMem[iPrim][iSec][3] == CONFIG_MEMORY_RAMMM) )
							{
							fFound = TRUE ;
							}
						}

				if (!fFound)
					{
					if (IDOK != MessageBox (hDlg, "Without RAM in page 3, the MSX cannot operate.", 
						NULL, MB_ICONEXCLAMATION | MB_OKCANCEL) )
						return ;
					}
				}
		
		default:
			cf.mem.iMem[ploc->iPrim][ploc->iSec][ploc->iPage] = CONFIG_MEMORY_EMPTY ;

			break ;
		}

	ConfigChanged () ;
	for (iSec=1;iSec<4;iSec++)
		for (iPage=0;iPage<4;iPage++)
			{
			if (cf.mem.iMem[ploc->iPrim][iSec][iPage] != CONFIG_MEMORY_EMPTY)
				return ;
			}

	cf.mem.fSlotExpanded[ploc->iPrim] = FALSE ;
	}

static void EndSelection (HWND hDlg)
	{
	if (mouse.fSelect)
		{
		DrawSelection (hDlg, &mouse.locSelect, GetSysColor (COLOR_BTNFACE) ) ;
		mouse.fSelect = FALSE ;
		}
	}

static void ProcessDrag (HWND hDlg, LOCATION *ploc)
	{
	switch (ploc->iID)
		{
		case IDC_MEMPLURGEBTN:
			switch (mouse.locOrg.iID)
				{
				case IDC_MEMPLURGEBTN:
					if (mouse.fSelect)
						{
						EndSelection (hDlg) ;
						
						DeleteBlock (hDlg, &mouse.locSelect) ;

						DrawSlot (hDlg, mouse.locSelect.iPrim) ;
						DoExpanded (hDlg) ;
						}
					else
						{
						// error: select block to delete
						}

					SetInformation (hDlg, NULL) ;

					return ;

				case IDC_MEMBLOCKS:
					// error: new memory block deleted
					SetInformation (hDlg, NULL) ;
					EndSelection (hDlg) ;
					
					return ;

				case IDC_MEMORYLAYOUT:
					// delete memory block
					EndSelection (hDlg) ;

					DeleteBlock (hDlg, &mouse.locOrg) ;

					DrawSlot (hDlg, mouse.locOrg.iPrim) ;
					DoExpanded (hDlg) ;
					SetInformation (hDlg, NULL) ;
					
					return ;

				default:
					_ASSERT (0) ;
				}

			break ;

		case IDC_MEMBLOCKS:
			// error : cannot drag shit to mem blocks
		default:
			SetInformation (hDlg, NULL) ;
			EndSelection (hDlg) ;
			
			return ;

		case IDC_MEMORYLAYOUT:
			switch (mouse.locOrg.iID)
				{
				case IDC_MEMORYLAYOUT:
					if (mouse.locOrg.iPrim == ploc->iPrim && 
						mouse.locOrg.iSec  == ploc->iSec  && 
						mouse.locOrg.iPage == ploc->iPage)
						{
						// des = org, select it
						if (mouse.fSelect)
							DrawSelection (hDlg, &mouse.locSelect, GetSysColor (COLOR_BTNFACE) ) ;
						
						mouse.locSelect = *ploc ;
						mouse.fSelect = TRUE ;
						DrawSelection (hDlg, &mouse.locSelect, RGB (0, 0, 255) ) ;

						return ;
						}

					// move it
					SetInformation (hDlg, NULL) ;
					EndSelection (hDlg) ;
					
					if (AddBlock (hDlg, ploc, cf.mem.iMem[mouse.locOrg.iPrim][mouse.locOrg.iSec][mouse.locOrg.iPage]) )
						{
						DeleteBlock (hDlg, &mouse.locOrg) ;

						DoExpanded (hDlg) ;
						DrawSlot (hDlg, mouse.locOrg.iPrim) ;
						if (mouse.locOrg.iPrim != ploc->iPrim)
							DrawSlot (hDlg, ploc->iPrim) ;
						}

					return ;
				
				case IDC_MEMBLOCKS:
					if (IsBlockExclusive ((short) (mouse.locOrg.iBlock + CONFIG_MEMORY_SUBBIOS) ) && 
						FindBlock ((short) (mouse.locOrg.iBlock + CONFIG_MEMORY_SUBBIOS), NULL) )
							{
							MessageBox (hDlg, "There can be only one of this type in memory.", NULL, MB_ICONINFORMATION) ;

							SetInformation (hDlg, NULL) ;
							EndSelection (hDlg) ;

							return ;
							}

					AddBlock (hDlg, ploc, (short) (mouse.locOrg.iBlock + CONFIG_MEMORY_SUBBIOS) ) ;

					DoExpanded (hDlg) ;
					DrawSlot (hDlg, ploc->iPrim) ;
					SetInformation (hDlg, NULL) ;
					EndSelection (hDlg) ;
					
					return ;
				
				case IDC_MEMPLURGEBTN:
					EndSelection (hDlg) ;
					SetInformation (hDlg, NULL) ;

					DeleteBlock (hDlg, ploc) ;

					DoExpanded (hDlg) ;
					DrawSlot (hDlg, ploc->iPrim) ;
					
					return ;

				default:
					_ASSERT (0) ;
				}

			break ;
		}
	}

static BOOL CheckMemoryLayout (HWND hDlg)
	{
	if (cf.iMSXversion != CONFIG_VERSION_MSX1)
		{
		if (!FindBlock (CONFIG_MEMORY_SUBBIOS, NULL) )
			{
			MessageBox (hDlg, "A MSX2 or higher requires a Sub BIOS.", NULL, MB_ICONINFORMATION) ;

			return FALSE ;
			}
		}
	
	return TRUE ;
	}

BOOL CALLBACK SetupMemoryDlgProc (HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
	{
	HBITMAP				hbmpOld ;
	HBRUSH				hbrOld ;
	RECT				rc ;
	int					i, iPrim, iSec ;
	LPDRAWITEMSTRUCT	lpdis ;
	HDC					hdc ;
	POINT				pt ;
	LOCATION			loc ;
	char				szBuf[64] ;
	
	switch (message)
		{
		case WM_WININICHANGE:
			DeleteObject (paint.hbrBack) ;
			DeleteObject (paint.hbmpLayout) ;
			DeleteObject (paint.hbmpBlocks) ;
			paint.hbrBack    = CreateSolidBrush (GetSysColor (COLOR_BTNFACE) ) ;
			paint.hbmpLayout = LoadBitmap (g_hInstance, MAKEINTRESOURCE (IDBITMAP_MEMORYLAYOUT) ) ;
			paint.hbmpBlocks = LoadBitmap (g_hInstance, MAKEINTRESOURCE (IDBITMAP_MEMORYBLOCKS) ) ;
			DrawBackground (paint.hbmpLayout) ;
			DrawBackground (paint.hbmpBlocks) ;
			
			return 0 ;
		
		case WM_INITDIALOG:
			OnChildDialogInit (hDlg) ;
			mouse.fSelect = FALSE ;

			// mapper sizes
			for (i=0;i<7;i++)
				{
				wsprintf (szBuf, "%d Kb %c", 64 << i, (i == 2 ? '*' : ' ') ) ;
				
				SendDlgItemMessage (hDlg, IDC_MEMSIZECMB, CB_ADDSTRING, 0, (LPARAM) (LPCSTR)szBuf) ;
				}
			
			paint.hbrBack    = CreateSolidBrush (GetSysColor (COLOR_BTNFACE) ) ;
			paint.hbmpLayout = LoadBitmap (g_hInstance, MAKEINTRESOURCE (IDBITMAP_MEMORYLAYOUT) ) ;
			paint.hbmpBlocks = LoadBitmap (g_hInstance, MAKEINTRESOURCE (IDBITMAP_MEMORYBLOCKS) ) ;
			mouse.hcurMemory = LoadCursor (g_hInstance, MAKEINTRESOURCE (IDC_MEMORY) ) ;
			mouse.hcurBomb   = LoadCursor (g_hInstance, MAKEINTRESOURCE (IDC_BOMB) ) ;
			DrawBackground (paint.hbmpLayout) ;
			DrawBackground (paint.hbmpBlocks) ;

			DoExpanded (hDlg) ;
			SetInformation (hDlg, NULL) ;

			return FALSE ;
		
		case WM_DRAWITEM :
			lpdis = (LPDRAWITEMSTRUCT) lParam ;
			
			switch (wParam)
				{
				case IDC_MEMPLURGEBTN:
					FillRect (lpdis->hDC, &lpdis->rcItem, paint.hbrBack) ;
					
					hdc = CreateCompatibleDC (lpdis->hDC) ;
					hbmpOld = SelectObject (hdc, paint.hbmpBlocks) ;
					hbrOld  = SelectObject (lpdis->hDC, paint.hbrBack) ;
					
					BitBlt (lpdis->hDC, 
						lpdis->rcItem.right - BLOCKS_GRID + BLOCKS_LINE, 
						lpdis->rcItem.top + BLOCKS_LINE, 
						BLOCKS_SIZE, BLOCKS_SIZE, hdc, BLOCKS_SIZE * 15, 0, SRCCOPY) ;
					
					SelectObject (lpdis->hDC, hbrOld) ;
					SelectObject (hdc, hbmpOld) ;
					DeleteDC (hdc) ;
				
					return TRUE ;
				
				case IDC_MEMBLOCKS:
					FillRect (lpdis->hDC, &lpdis->rcItem, paint.hbrBack) ;
					
					hdc = CreateCompatibleDC (lpdis->hDC) ;
					hbmpOld = SelectObject (hdc, paint.hbmpBlocks) ;
					hbrOld  = SelectObject (lpdis->hDC, paint.hbrBack) ;

					for (i=0;i<BLOCKS_DISPLAY;i++)
						{
						BitBlt (lpdis->hDC, 
							lpdis->rcItem.left + BLOCKS_GRID * i + BLOCKS_LINE, 
							lpdis->rcItem.top + BLOCKS_LINE, 
							BLOCKS_SIZE, BLOCKS_SIZE, hdc, 
							BLOCKS_SIZE * (i + 2), 0, SRCCOPY) ;
						}

					SelectObject (lpdis->hDC, hbrOld) ;
					SelectObject (hdc, hbmpOld) ;
					DeleteDC (hdc) ;
				
					return TRUE ;
				
				case IDC_MEMORYLAYOUT:
					paint.ixOffsetLayout = (lpdis->rcItem.right - LAYOUT_X) / 2 ;
					paint.iyOffsetLayout = (lpdis->rcItem.bottom - LAYOUT_Y) / 2 ;

					FillRect (lpdis->hDC, &lpdis->rcItem, paint.hbrBack) ;
					
					hdc = CreateCompatibleDC (lpdis->hDC) ;
					hbmpOld = SelectObject (hdc, paint.hbmpLayout) ;
					
					BitBlt (lpdis->hDC, 
						paint.ixOffsetLayout, 
						paint.iyOffsetLayout, 
						LAYOUT_X, LAYOUT_Y, hdc, 0, 0, SRCCOPY) ;
					
					SelectObject (hdc, hbmpOld) ;
					DeleteDC (hdc) ;
					
					GetWindowRect (GetDlgItem (hDlg, IDC_MEMORYLAYOUT), &rc) ;
					ScreenToClient (hDlg, (LPPOINT)&rc.left) ;
					ScreenToClient (hDlg, (LPPOINT)&rc.right) ;					

					paint.ixOffsetLayout += rc.left ;
					paint.iyOffsetLayout += rc.top ;

					for (i=0;i<4;i++)
						DrawSlot (hDlg, i) ;

					if (mouse.fSelect)
						DrawSelection (hDlg, &mouse.locSelect, RGB (0, 0, 255) ) ;

					return TRUE ;
				}
			
			break ;
		
		case WM_DESTROY:
			for (i=0;i<4;i++)
				cf.cart[i].fEnabled = FALSE ;

			for (iPrim=0;iPrim<4;iPrim++)
				for (iSec=0;iSec<4;iSec++)
					{
					switch (cf.mem.iMem[iPrim][iSec][0])
						{
						case CONFIG_MEMORY_CARTA:
							cf.cart[0].fEnabled = TRUE ;
							break ;
						case CONFIG_MEMORY_CARTB:
							cf.cart[1].fEnabled = TRUE ;
							break ;
						case CONFIG_MEMORY_CARTC:
							cf.cart[2].fEnabled = TRUE ;
							break ;
						case CONFIG_MEMORY_CARTD:
							cf.cart[3].fEnabled = TRUE ;
							break ;
						}
					}
			
			DeleteObject (paint.hbrBack) ;
			DeleteObject (paint.hbmpLayout) ;
			DeleteObject (paint.hbmpBlocks) ;
			
			return TRUE ;

		case WM_LBUTTONDOWN:
			pt.x = LOWORD (lParam) ;
			pt.y = HIWORD (lParam) ;
			ClientToScreen (hDlg, &pt) ;
			i = GetLocation (hDlg, &pt, &loc) ;
			if (i)
				{
				mouse.locOrg = loc ;
				SetInformation (hDlg, &loc) ;
				SetCapture (hDlg) ;
				SetCursor (mouse.locOrg.iID == IDC_MEMPLURGEBTN ? mouse.hcurBomb : mouse.hcurMemory) ;
				DrawSelection (hDlg, &mouse.locOrg, RGB (127, 0, 0) ) ;

				return TRUE ;
				}

			break ;
		
		case WM_MOUSEMOVE:
			if (hDlg == GetCapture  () )
				{
				pt.x = LOWORD (lParam) ;
				pt.y = HIWORD (lParam) ;
				ClientToScreen (hDlg, &pt) ;
				i = GetLocation (hDlg, &pt, &loc) ;

				switch (i)
					{
					case IDC_MEMBLOCKS:
						SetCursor (LoadCursor (NULL, IDC_NO) ) ;
						break ;
				
					case IDC_MEMPLURGEBTN:
						SetCursor ( (mouse.locOrg.iID == IDC_MEMORYLAYOUT) ? 
							mouse.hcurMemory : LoadCursor (NULL, IDC_NO) ) ;
						break ;
					
					case IDC_MEMORYLAYOUT:
						SetCursor ( (mouse.locOrg.iID == IDC_MEMPLURGEBTN ? 
							mouse.hcurBomb : mouse.hcurMemory) ) ;
						break ;
					
					default:
						SetCursor (LoadCursor (NULL, IDC_NO) ) ;
						break ;
					}

				return TRUE ;
				}

			break ;
		
		case WM_LBUTTONUP:
			if (GetCapture () == hDlg)
				{
				pt.x = LOWORD (lParam) ;
				pt.y = HIWORD (lParam) ;
				ClientToScreen (hDlg, &pt) ;
				GetLocation (hDlg, &pt, &loc) ;
				DrawSelection (hDlg, &mouse.locOrg, GetSysColor (COLOR_BTNFACE) ) ;

				ProcessDrag (hDlg, &loc) ;

				ReleaseCapture () ;
				SetCursor (LoadCursor (NULL, IDC_ARROW) ) ;

				return TRUE ;
				}

			break ;
		
		case WM_COMMAND:
			switch (LOWORD (wParam) )
				{
				case IDC_MEMSLOT0EXPANDCHK:
				case IDC_MEMSLOT1EXPANDCHK:
				case IDC_MEMSLOT2EXPANDCHK:
				case IDC_MEMSLOT3EXPANDCHK:
					ConfigChanged () ;
					cf.mem.fSlotExpanded[(int)LOWORD (wParam) - IDC_MEMSLOT0EXPANDCHK] = 
						(char)SendDlgItemMessage (hDlg, (int)LOWORD (wParam), BM_GETCHECK, 0, 0) ;
					DoExpanded (hDlg) ;
					DrawSlot (hDlg, (int)LOWORD (wParam) - IDC_MEMSLOT0EXPANDCHK) ;

					return TRUE ;

				case IDC_MEMSIZECMB:
					if (CBN_SELCHANGE == HIWORD (wParam) )
						{
						_ASSERT (mouse.fSelect) ;
						i = SendMessage ((HWND)lParam, CB_GETCURSEL, 0, 0) ;
						if (cf.mem.iMapperSize[mouse.locSelect.iPrim * 4 + mouse.locSelect.iSec] != (char)i)
							{
							ConfigChanged () ;

							cf.mem.iMapperSize[mouse.locSelect.iPrim * 4 + mouse.locSelect.iSec] = (char)i ;
							}

						return TRUE ;
						}

					break ;
				}
			
			break ;
		}

	return FALSE ;
	}


// Sound Dailog
static void SetVolume (HWND hDlg, VOLUME *vol)
	{
	SendDlgItemMessage (hDlg, IDC_SOUNDENABLECHK, BM_SETCHECK, vol->fEnable, 0) ;

	SendDlgItemMessage (hDlg, IDC_SOUNDVOLSLIDER, TBM_SETPOS, TRUE, 20 - vol->lVol) ;
	SendDlgItemMessage (hDlg, IDC_SOUNDBALSLIDER, TBM_SETPOS, TRUE, 10 + vol->lPan) ;
	}

BOOL CALLBACK SetupSoundDlgProc (HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
	{
	static char		*pszDevices[] = {
		"PPI (key click)", "PSG (Standard sound chip)", 
		"Konami SCC/SCC+ cartridge A", "Konami SCC/SCC+ cartridge B", 
		"Konami SCC/SCC+ cartridge C", "Konami SCC/SCC+ cartridge D", 
		"FM PAC (MSX Music)" } ;
	static			iCurrent ;
	static VOLUME	*vol ;
	int				i ;

	switch (message)
		{
		case WM_INITDIALOG:
			OnChildDialogInit (hDlg) ;
			
			for (i=0;i<7;i++)
				SendDlgItemMessage (hDlg, IDC_SOUNDDEVCMB, CB_ADDSTRING, 0, 
					(LPARAM) (LPCSTR) pszDevices[i]) ;

			SendDlgItemMessage (hDlg, IDC_SOUNDVOLSLIDER, TBM_SETRANGE, 
				FALSE, MAKELONG (0, 20) ) ;

			SendDlgItemMessage (hDlg, IDC_SOUNDBALSLIDER, TBM_SETRANGE, 
				FALSE, MAKELONG (0, 20) ) ;

			SendDlgItemMessage (hDlg, IDC_SOUNDDEVCMB, CB_SETCURSEL, 0, 0) ;
			vol = &cf.vol.PPI ;
			iCurrent = 0 ;
			SetVolume (hDlg, vol) ;

			return FALSE ;

		case WM_COMMAND:
			switch (LOWORD (wParam) )
				{
				case IDC_SOUNDDEVCMB:
					if (CBN_SELCHANGE == HIWORD (wParam) )
						{
						iCurrent = SendDlgItemMessage (hDlg, IDC_SOUNDDEVCMB, CB_GETCURSEL, 0, 0) ;

						switch (iCurrent)
							{
							case 0:
								vol = &cf.vol.PPI ; break ;
							case 1:
								vol = &cf.vol.PSG ; break ;
							case 6:
								vol = &cf.vol.FM ; break ;
							default:
								vol = &cf.vol.SCC[iCurrent - 2] ; break ;
							}
						
						// set volume settings
						SetVolume (hDlg, vol) ;
						}

					return TRUE ;

				case IDC_SOUNDENABLECHK:
					vol->fEnable = !vol->fEnable ;
					ConfigChanged () ;

					return TRUE ;
				}

			break ;

		case WM_HSCROLL:
			if ((HWND)lParam == GetDlgItem (hDlg, IDC_SOUNDVOLSLIDER) )
				{
				i = 20 - SendDlgItemMessage (hDlg, IDC_SOUNDVOLSLIDER, TBM_GETPOS, 0, 0) ;
				if (i != vol->lVol)
					{
					ConfigChanged () ;
					vol->lVol = i ;
					}
				
				return TRUE ;
				}

			if ((HWND)lParam == GetDlgItem (hDlg, IDC_SOUNDBALSLIDER) )
				{
				i = SendDlgItemMessage (hDlg, IDC_SOUNDBALSLIDER, TBM_GETPOS, 0, 0) - 10 ;
				if (i != vol->lPan)
					{
					ConfigChanged () ;
					vol->lPan = i ;
					}
				
				return TRUE ;
				}
			
			break ;
		}

	return FALSE ;
	}

