// main.c : main file
#include <windows.h>
#include <stdio.h>
#include <commctrl.h>
#include <ddraw.h>
#include <mmsystem.h>
#include <assert.h>
#include "msxcore.h"
#include "resource.h"
#include "config.h"
#include "configdlg.h"
#include "main.h"
#include "keyboard.h"

LRESULT CALLBACK MainWndProc (HWND, UINT, WPARAM, LPARAM) ;
const char	g_szAppName[] = "Virtual MSX" ;
HWND		g_hwndMain ;
HINSTANCE	g_hInstance ;
static struct
	{
	LPDIRECTDRAW			lpDD ;
	LPDIRECTDRAWSURFACE		lpDDSPrimary, lpDDSBack ;
	LPDIRECTDRAWPALETTE		lpPalette ;
	int				iWidth, iHeight ;
	PBYTE			pSurface ;
	int				iClear ;
	BOOL			fNowFullScreen, fScanlines, fBlitting ;
	WINDOWPLACEMENT	wp ;
	} dd ;

BOOL CreateMainWindow (int iCmdShow)
	{
	WNDCLASSEX  wc;
	
    wc.cbSize        = sizeof (WNDCLASSEX) ;
	wc.style         = 0 ;
    wc.lpfnWndProc   = MainWndProc ;
    wc.cbClsExtra    = 0 ;
    wc.cbWndExtra    = 0 ;
    wc.hInstance     = g_hInstance ;
    wc.hIcon         = LoadIcon (g_hInstance, MAKEINTRESOURCE (IDICON_MAIN) ) ;
    wc.hCursor       = LoadCursor (NULL, IDC_ARROW) ;
    wc.hbrBackground = NULL ; 
    wc.lpszMenuName  = MAKEINTRESOURCE (IDMENU_MAIN) ;
    wc.lpszClassName = g_szAppName ;
	wc.hIconSm 		 = LoadIcon (g_hInstance, MAKEINTRESOURCE (IDICON_MAIN) ) ;

	if (!RegisterClassEx (&wc) )
		return FALSE ;

	g_hwndMain = CreateWindow (
		g_szAppName, g_szAppName,
		WS_OVERLAPPEDWINDOW, 
		CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, 
		NULL, NULL, g_hInstance, NULL) ;
	
	if (g_hwndMain == NULL)
		return FALSE ;

    ShowWindow (g_hwndMain, iCmdShow) ;
    UpdateWindow (g_hwndMain) ;
    
	return TRUE ;
	}

int WINAPI WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance, 
					PSTR szCmdLine, int iCmdShow)
	{
	MSG		msg ;
	HACCEL	haccel ;
	
	g_hInstance = hInstance ;
	InitCommonControls () ;
	
	if (DirectDrawCreate (NULL, &dd.lpDD, NULL) != DD_OK)
		{
		MessageBox (NULL, "Unable to create DirectDraw object.", g_szAppName, MB_ICONSTOP) ;

		return FALSE ;
		}
	
	if (!CreateMainWindow (iCmdShow) )
		{
		MessageBox (NULL, "Unable to create main window.", g_szAppName, MB_ICONSTOP) ;

		return FALSE ;
		}

	LoadDefaultConfig (g_hwndMain) ;
	haccel = LoadAccelerators (g_hInstance, MAKEINTRESOURCE (IDACCEL_MAIN) ) ;
	
	while (GetMessage (&msg, NULL, 0, 0) )
		{
		if (!TranslateAccelerator (g_hwndMain, haccel, &msg) )
			{
			TranslateMessage (&msg) ;
			DispatchMessage (&msg) ;
			}
		}

	dd.lpDD->lpVtbl->Release (dd.lpDD) ;

	return msg.wParam ;
	}

static void DoCaption ()
	{
	static char *state[] = { "%s", "%s - Running", "%s - Paused", "%s - Crashed" } ;
	char	szBuf[256] ;

	wsprintf (szBuf, state[GetEmulatorState ()], g_szAppName) ;
	SetWindowText (g_hwndMain, szBuf) ;
	}

LRESULT CALLBACK MainWndProc (HWND hwnd, UINT message, UINT wParam, LONG lParam)
	{
	PAINTSTRUCT		ps ;
	HBRUSH			hbr ;
	UINT			n ;
	
	switch (message)
		{
        case WM_CREATE:
			
			return 0 ;
		
		case WM_COMMAND:
			switch (LOWORD (wParam) )
				{
				case IDM_SETUP_MSXCONFIG:
					OnConfigDialog (hwnd) ;

					return 0 ;
	
				case IDM_SETUP_SCANLINES:
					dd.fScanlines = !dd.fScanlines ;

					if (dd.fNowFullScreen)
						{
						n = GetEmulatorState () ;
						SetEmulatorState (MS_PAUSED) ;
						
						while (dd.fBlitting) ;
						
						dd.lpDD->lpVtbl->RestoreDisplayMode (dd.lpDD) ;
						dd.lpDD->lpVtbl->SetCooperativeLevel (dd.lpDD, g_hwndMain, DDSCL_NORMAL) ;
						dd.lpDDSPrimary->lpVtbl->Release (dd.lpDDSPrimary) ;

						InitDisplay (0) ;

						SetEmulatorState (n) ;
						}

					return 0 ;

				case IDM_RUN_START:
					if (GetEmulatorState () == MS_NOTRUNNING)
						StartEmulator () ;
					else
						StopEmulator () ;
					
					DoCaption () ;

					return 0 ;

				case IDM_RUN_PAUSE:
					PauseEmulator () ;

					DoCaption () ;

					return 0 ;
				
				case IDM_SETUP_EXIT:
					PostQuitMessage (0) ;

					return 0 ;
				}

			break ;

		case WM_KEYDOWN:
		case WM_KEYUP:
			OnKey (wParam, lParam) ;
			
			return 0 ;
		
		case WM_CHAR:
		case WM_DEADCHAR:
			OnChar (wParam, lParam) ;
			
			return 0 ;
		
		case WM_PAINT:
			BeginPaint (hwnd, &ps) ;		
			
			if (!dd.fNowFullScreen)
				{
				hbr = CreateSolidBrush (GetSysColor (COLOR_WINDOW) ) ;

				FillRect (ps.hdc, &ps.rcPaint, hbr) ;

				DeleteObject (hbr) ;
				}
			else dd.iClear = 4 ;

			EndPaint (hwnd, &ps) ;
			
			return 0 ;

		case WM_SETCURSOR:
			if (dd.fNowFullScreen && GetEmulatorState () == MS_RUNNING)
				{
				SetCursor (NULL) ;

				return 0 ;
				}
			
			break ;
		
		case WM_ENTERMENULOOP:
			if (dd.fNowFullScreen)
				{
				RECT	rc ;

				dd.lpDD->lpVtbl->FlipToGDISurface (dd.lpDD) ;
				SetRect (&rc, 0,	
					GetSystemMetrics (SM_CYCAPTION) + GetSystemMetrics (SM_CYMENU), 
					dd.iWidth, dd.iHeight) ;
				AdjustWindowRectEx (&rc, WS_POPUP | WS_SYSMENU | WS_CAPTION, TRUE, 0) ;
				SetWindowPos (g_hwndMain, HWND_TOPMOST, rc.left, rc.top, 
					rc.right-rc.left, rc.bottom-rc.top, SWP_NOCOPYBITS) ;
				}

			return 0 ;
		
		case WM_EXITMENULOOP:
			if (dd.fNowFullScreen)
				{
				RECT	rc ;

				SetRect (&rc, 0, 0, 
					dd.iWidth, dd.iHeight) ;
				AdjustWindowRectEx (&rc, WS_POPUP | WS_SYSMENU | WS_CAPTION, TRUE, 0) ;
				SetWindowPos (g_hwndMain, HWND_TOPMOST, rc.left, rc.top, 
					rc.right-rc.left, rc.bottom-rc.top, SWP_NOCOPYBITS) ;
				dd.iClear = 10 ;
				}

			return 0 ;
		
		case WM_DESTROY:
			PostQuitMessage(0) ;
            
			return 0 ;
		}

	return DefWindowProc (hwnd, message, wParam, lParam) ;
	}

BOOL InitDisplay (int iType)
	{
    DDSURFACEDESC	ddsd ;
    DDSCAPS			caps ;
	PALETTEENTRY	aRGB[256] ;
	RECT			rc ;
	
    dd.wp.length = sizeof (WINDOWPLACEMENT) ;
	GetWindowPlacement (g_hwndMain, &dd.wp) ;

	if (dd.lpDD->lpVtbl->SetCooperativeLevel (dd.lpDD, g_hwndMain, 
		DDSCL_ALLOWMODEX | DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN) != DD_OK)
		{
		MessageBox (g_hwndMain, "Could not initialize DirectDraw object.", NULL, 
			MB_ICONEXCLAMATION) ;

		return FALSE ;
		}

	if (dd.fScanlines)
		{
		dd.iWidth  = 640 ;
		dd.iHeight = 480 ;
		}
	else
		{
		dd.iWidth  = 320 ;
		dd.iHeight = 240 ;
		}

	if (dd.lpDD->lpVtbl->SetDisplayMode (dd.lpDD, dd.iWidth, dd.iHeight, 8) != DD_OK)
		{
		MessageBox (g_hwndMain, "Could not set display mode.", NULL, 
			MB_ICONEXCLAMATION) ;

		return FALSE ;
		}

	SetWindowLong (g_hwndMain, GWL_STYLE, 
		WS_POPUP | WS_SYSMENU | WS_CAPTION | WS_MINIMIZEBOX) ;
	SetRect (&rc, 0, 0, dd.iWidth, dd.iHeight) ;
	SetWindowPos (g_hwndMain, HWND_TOPMOST, rc.left, rc.top, 
		rc.right-rc.left, rc.bottom-rc.top, SWP_SHOWWINDOW) ;
	memset (&ddsd, 0, sizeof (ddsd) ) ;
	ddsd.dwSize = sizeof (ddsd) ;
    ddsd.dwFlags = DDSD_CAPS | DDSD_BACKBUFFERCOUNT ;
    ddsd.dwBackBufferCount = 1 ;
    ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE | 
                          DDSCAPS_FLIP | 
                          DDSCAPS_COMPLEX ;

	if (dd.lpDD->lpVtbl->CreateSurface (dd.lpDD, &ddsd, &dd.lpDDSPrimary, NULL) != DD_OK)
		{
		MessageBox (g_hwndMain, "DirectDraw failed to create primary surface.", NULL, 
			MB_ICONEXCLAMATION) ;

		return FALSE ;
		}
	
	caps.dwCaps = DDSCAPS_BACKBUFFER ;
	if (dd.lpDDSPrimary->lpVtbl->GetAttachedSurface (dd.lpDDSPrimary, &caps, &dd.lpDDSBack) != DD_OK)
		{
		MessageBox (g_hwndMain, "DirectDraw failed to create secundairy surface.", NULL, 
			MB_ICONEXCLAMATION) ;

		return FALSE ;
		}
	
	if (!dd.lpPalette)
		{
		memset (aRGB, 0, sizeof (aRGB) ) ;
		if (dd.lpDD->lpVtbl->CreatePalette (dd.lpDD, DDPCAPS_8BIT, aRGB, &dd.lpPalette, NULL) != DD_OK)
			{
			MessageBox (g_hwndMain, "DirectDraw failed to create palette.", NULL, 
				MB_ICONEXCLAMATION) ;

			return FALSE ;
			}
		}

	if (dd.lpDDSPrimary->lpVtbl->SetPalette (dd.lpDDSPrimary, dd.lpPalette) != DD_OK)
		{
		MessageBox (g_hwndMain, "SetPalette failed.", NULL, MB_ICONEXCLAMATION) ;

		return FALSE ;
		}
	
	dd.iClear = 10 ;
	
	dd.fNowFullScreen = TRUE ;

	return TRUE ;
	}

void ReleaseDisplay ()
	{
	dd.fNowFullScreen = FALSE ;

	dd.lpDD->lpVtbl->RestoreDisplayMode (dd.lpDD) ;
	dd.lpDD->lpVtbl->SetCooperativeLevel (dd.lpDD, g_hwndMain, DDSCL_NORMAL) ;
	dd.lpDDSPrimary->lpVtbl->Release (dd.lpDDSPrimary) ;

	SetWindowLong (g_hwndMain, GWL_STYLE, WS_OVERLAPPEDWINDOW) ;
	SetWindowPlacement (g_hwndMain, &dd.wp) ;
	InvalidateRect (NULL, NULL, FALSE) ;
	}

BOOL LockSurface (BYTE **ppdes, DWORD *pPitch)
	{
	DDSURFACEDESC	ddsd ;
	HRESULT			ddRet ;

	memset (&ddsd, 0, sizeof (ddsd) ) ;
	ddsd.dwSize = sizeof (ddsd) ;
	
	ddRet = dd.lpDDSBack->lpVtbl->Lock (dd.lpDDSBack, NULL, &ddsd, 
		DDLOCK_WRITEONLY  | DDLOCK_WAIT | DDLOCK_SURFACEMEMORYPTR ,NULL) ;

	if (ddRet == DDERR_SURFACELOST)
		{
		if (dd.lpDDSBack->lpVtbl->Restore (dd.lpDDSBack) != DD_OK)
			return FALSE ;
		
		ddRet = dd.lpDDSBack->lpVtbl->Lock (dd.lpDDSBack, NULL, &ddsd, 
			DDLOCK_WRITEONLY | DDLOCK_WAIT | DDLOCK_SURFACEMEMORYPTR ,NULL) ;

		dd.iClear = 10 ;
		}
	else if (ddRet != DD_OK)
		return FALSE ;

	*pPitch = ddsd.lPitch ;
	*ppdes  = ddsd.lpSurface ;

	return TRUE ;
	}

void UnlockSurface ()
	{
	dd.lpDDSBack->lpVtbl->Unlock (dd.lpDDSBack, dd.pSurface);
	}

void SetPalette (int iFirst, int iNumb, PALETTEENTRY *ape)
	{
	dd.lpPalette->lpVtbl->SetEntries (dd.lpPalette, 0, iFirst, iNumb, ape) ;
	}

void DisplayScreen (BYTE *pScreen)
	{
	int		x, y, yy, iPitch ;

extern void Move256Enlarge () ;
extern void Line256x8 () ;
	
	assert ( !(((DWORD)pScreen) & 3) ) ;	// 4 byte aligned
	
	dd.fBlitting = TRUE ;

	if (!LockSurface (&dd.pSurface, &iPitch) )
		{
		dd.fBlitting = FALSE ;
		return ;
		}

	if (dd.fScanlines)
		{
		y = (dd.iHeight - 192 * 2) / 2 ;
		x = (dd.iWidth  - 256 * 2) / 2 ;
		x &= ~3 ;
		
		if (dd.iClear)
			{
			dd.iClear-- ;

			if (!(y % 2) )
				{
				for (yy=0;yy<dd.iHeight;yy++)
					memset (dd.pSurface + yy * iPitch, (yy % 2 ? 0 : 0x80), iPitch) ;
				}
			else
				{
				for (yy=0;yy<dd.iHeight;yy++)
					memset (dd.pSurface + yy * iPitch, (yy % 2 ? 0x80 : 0), iPitch) ;
				}
			}

		__asm
			{
			mov		esi, pScreen
			mov		edi, dd.pSurface
			mov		ax, ds
			mov		es, ax
			mov		eax, iPitch
			mov		ecx, eax
			mul		y
			add		edi, eax
			add		edi, x
			lea		edx, [ecx*2 - 512]
			mov		ecx, 192
	do_loop:
			call	Move256Enlarge
			add		edi, edx
			dec		ecx
			jne		do_loop
			}
		}
	else
		{
		y = (dd.iHeight - 192) / 2 ;
		x = (dd.iWidth  - 256) / 2 ;

		x &= ~3 ;

		if (dd.iClear)
			{
			dd.iClear-- ;

			memset (dd.pSurface, 0x80, iPitch * dd.iHeight) ;
			}

		__asm
			{
			mov		esi, pScreen
			mov		edi, dd.pSurface
			mov		eax, iPitch
			mul		y
			add		edi, eax
			add		edi, x
			mov		edx, iPitch
			sub		edx, 256
			mov		ecx, 192
	do_loop2:
			call	Line256x8
			add		edi, edx
			dec		ecx
			jne		do_loop2
			}
		}
	
	UnlockSurface () ;

	dd.lpDDSPrimary->lpVtbl->Flip (dd.lpDDSPrimary, NULL, 0) ;

	dd.fBlitting = FALSE ;
	}
