; ****************************************************************************
; z80.asm : z80 emulation
; ****************************************************************************

.386
.model flat, c

include z80.inc
include memory.inc
include io.inc

PUBLIC	z80

COMMENT $

 - to calculate F: z80.rF OR (rFb35 AND 028H)
 - to calculate R: z80.rR OR (rRb7 AND 080H)

In this source we assume that after lahf bit 1 will be set

Register Registers: :)
 di = PC
esi = Tstates
 al = A
 ah = F
ebp = R

Each instruction emulation code ends with sub esi, t
 t equals the number of T states the instruction takes to execute.

	esi > t		CF=0 ZF=0
	esi = t		CF=0 ZF=1
	esi < t		CF=1 ZF=0
	check for INT	CF=1 ZF=1
$

.const

include z80tbl.inc

.data?

z80		Z80STATE { }

ALIGN 4

dwReturn	DWORD	?
rFb35		BYTE	?	; bits 3 and 5 are of reg F, rest undefined
rRb7		BYTE	?	; bit 7 is bit of reg R, rest undefined

.code

LoadRegs MACRO
	mov	di, z80.rPC
	mov	esi, z80.dwTstates
	;; R register
	mov	al, z80.rR
	mov	rRb7, al
	mov	ebp, eax
	;; AF
	mov	al, z80.rA
	mov	ah, z80.rF
	;; F correction
	mov	rFb35, ah
	ENDM

SaveRegs MACRO
	mov	z80.rPC, di
	mov	z80.dwTstates, esi
	;; A register
	mov	z80.rA, al
	;; F register
	mov	al, rFb35
	and	ax, FLAG_5 OR FLAG_3 OR 256 * (NOT (FLAG_3 OR FLAG_5) AND 255)
	or	al, ah
	mov	z80.rF, al
	;; R register
	mov	al, rRb7
	and	al,  BIT_7
	and	ebp, NOT BIT_7 AND 255
	or	eax, ebp
	mov	z80.rR, al
	ENDM

ReadMemW MACRO
	push	cx
	call	ReadMem_dl
	pop	cx
	inc	cx
	call	ReadMem_dh
	ENDM

WriteMemW MACRO
	push	cx
	call	WriteMem_dl
	pop	cx
	inc	cx
	call	WriteMem_dh
	ENDM

DoPush MACRO
	mov	cx, z80.rSP
	dec	cx
	call	WriteMem_dh
	mov	cx, z80.rSP
	sub	cx, 2
	mov	z80.rSP, cx
	call	WriteMem_dl
	ENDM

DoPop MACRO
	mov	cx, z80.rSP
	call	ReadMem_dl
	mov	cx, z80.rSP
	add	cx, 2
	mov	z80.rSP, cx
	dec	cx
	call	ReadMem_dh
	ENDM

DoZ80 PROC C
	push	ebx
	push	esi
	push	edi
	push	ebp

	mov	dwReturn, 1

	LoadRegs

dz_fde_int:
	call 	Interrupt
	jc	dz_end
	
;	xor	edx, edx
	call	ReadPC_dl
	inc	ebp
	call	JumpTbl[edx*4]
	jnc	dz_fastloop
	jz	dz_fde_int
	jmp	dz_end
	
dz_fastloop:
	call 	Interrupt
	jc	dz_end

dz_fde:
	xor	edx, edx
	call	ReadPC_dl
	inc	ebp
	call	JumpTbl[edx*4]
	jnc	dz_fde
	jz	dz_fde_int

dz_end:
	xor	esi, esi
	SaveRegs
	
	mov	eax, dwReturn
	pop	ebp
	pop	edi
	pop	esi
	pop	ebx
	ret

Interrupt::
	sub	bl, bl
	cmp	z80.fNoINT, bl
	jne	ResNoINT
	cmp	z80.fINT, bl
	je	NoInt
	IF IFF_RESET NE 0
		mov	bl, IFF_RESET
	ENDIF
	cmp	z80.fIFF, bl
	je	NoInt
	
	; ok, generate int.
	mov	z80.fIFF, bl
	inc	ebp	; int generation increases R by 1. always.
	mov	edx, edi
	DoPush
	cmp	z80.wIM, 2
	je	DoIM2
	mov	edi, 038h
	mov	edx, 1
	sub	esi, 13 + 1
	retn

DoIM2:
	mov	cl, 0FFh
	mov	ch, z80.rI
	ReadMemW
	mov	edi, edx
	mov	edx, 1
	sub	esi, 19 + 1
	retn

ResNoINT:
	mov	z80.fNoINT, bl
NoInt:
	xor	edx, edx
	retn


Opcodeed::
	call	ReadPC_dl
	inc	ebp
	jmp	JumpTblED[edx*4]

Opcodecb::
	call	ReadPC_dl
	inc	ebp
	jmp	JumpTblCB[edx*4]

Opcodedd::
	call	ReadPC_dl
	inc	ebp
	jmp	JumpTblDD[edx*4]

Opcodefd::
	call	ReadPC_dl
	inc	ebp
	jmp	JumpTblFD[edx*4]

Opcodeddcb::
	call	ReadPC_dh
	call	ReadPC_dl

	movsx	ecx, dh
	and	edx, 0ffh
	add	cx, z80.rIX

	jmp	JumpTblDDCB[edx*4]

Opcodefdcb::
	call	ReadPC_dh
	call	ReadPC_dl

	movsx	ecx, dh
	and	edx, 0ffh
	add	cx, z80.rIY

	jmp	JumpTblDDCB[edx*4]

Opcodeedxx::
	sub	esi, 8 + 2
	retn

Opcodefdxx::
Opcodeddxx::
	dec	edi
	dec	ebp

	sub	esi, 4 + 1
	retn

; ****************************************************************************
; LD r,r

LD_R_R MACRO r,p
	;; mem/mem not allowed; optimize code
	IFIDNI <r>, <p>
		;; no effect; no code to generate
	ELSEIFIDNI <r>, <al>
		mov	al, p		;; 1
	ELSEIFIDNI <p>, <al>
		mov	r, al		;; 1
	ELSE
		mov	dl, p		;; 1
		mov	r, dl		;; 1
	ENDIF
						;; total: 1/2
	
	sub	esi, 4 + 1
	retn
	ENDM

Opcode40::
	LD_R_R z80.rB, z80.rB

Opcode41::
	LD_R_R z80.rB, z80.rC

Opcode42::
	LD_R_R z80.rB, z80.rD

Opcode43::
	LD_R_R z80.rB, z80.rE

Opcode44::
	LD_R_R z80.rB, z80.rH

Opcode45::
	LD_R_R z80.rB, z80.rL

Opcode47::
	LD_R_R z80.rB, al

Opcode48::
	LD_R_R z80.rC, z80.rB

Opcode49::
	LD_R_R z80.rC, z80.rC

Opcode4a::
	LD_R_R z80.rC, z80.rD

Opcode4b::
	LD_R_R z80.rC, z80.rE

Opcode4c::
	LD_R_R z80.rC, z80.rH

Opcode4d::
	LD_R_R z80.rC, z80.rL

Opcode4f::
	LD_R_R z80.rC, al

Opcode50::
	LD_R_R z80.rD, z80.rB

Opcode51::
	LD_R_R z80.rD, z80.rC

Opcode52::
	LD_R_R z80.rD, z80.rD

Opcode53::
	LD_R_R z80.rD, z80.rE

Opcode54::
	LD_R_R z80.rD, z80.rH

Opcode55::
	LD_R_R z80.rD, z80.rL

Opcode57::
	LD_R_R z80.rD, al

Opcode58::
	LD_R_R z80.rE, z80.rB

Opcode59::
	LD_R_R z80.rE, z80.rC

Opcode5a::
	LD_R_R z80.rE, z80.rD

Opcode5b::
	LD_R_R z80.rE, z80.rE

Opcode5c::
	LD_R_R z80.rE, z80.rH

Opcode5d::
	LD_R_R z80.rE, z80.rL

Opcode5f::
	LD_R_R z80.rE, al

Opcode60::
	LD_R_R z80.rH, z80.rB

Opcode61::
	LD_R_R z80.rH, z80.rC

Opcode62::
	LD_R_R z80.rH, z80.rD

Opcode63::
	LD_R_R z80.rH, z80.rE

Opcode64::
	LD_R_R z80.rH, z80.rH

Opcode65::
	LD_R_R z80.rH, z80.rL

Opcode67::
	LD_R_R z80.rH, al

Opcode68::
	LD_R_R z80.rL, z80.rB

Opcode69::
	LD_R_R z80.rL, z80.rC

Opcode6a::
	LD_R_R z80.rL, z80.rD

Opcode6b::
	LD_R_R z80.rL, z80.rE

Opcode6c::
	LD_R_R z80.rL, z80.rH

Opcode6d::
	LD_R_R z80.rL, z80.rL

Opcode6f::
	LD_R_R z80.rL, al

Opcode78::
	LD_R_R al, z80.rB

Opcode79::
	LD_R_R al, z80.rC

Opcode7a::
	LD_R_R al, z80.rD

Opcode7b::
	LD_R_R al, z80.rE

Opcode7c::
	LD_R_R al, z80.rH

Opcode7d::
	LD_R_R al, z80.rL

Opcode7f::
	LD_R_R al, al

; ****************************************************************************
; LD r,r (r=IXl,IXh,IYl,IYh)

LD_XY_R_R MACRO r,p
	IFIDNI <r>, <al>
		mov	al, p		;; 1
	ELSEIFIDNI <p>, <al>
		mov	r, al		;; 1
	ELSE
		mov	dl, p		;; 1
		mov	r, dl		;; 1
	ENDIF
						;; total: 1/2
	
	sub	esi, 8 + 2
	retn
	ENDM

Opcodedd44::
	LD_XY_R_R z80.rB, z80.rIXh

Opcodedd45::
	LD_XY_R_R z80.rB, z80.rIXl

Opcodefd44::
	LD_XY_R_R z80.rB, z80.rIYh

Opcodefd45::
	LD_XY_R_R z80.rB, z80.rIYl

Opcodedd4c::
	LD_XY_R_R z80.rC, z80.rIXh

Opcodedd4d::
	LD_XY_R_R z80.rC, z80.rIXl

Opcodefd4c::
	LD_XY_R_R z80.rC, z80.rIYh

Opcodefd4d::
	LD_XY_R_R z80.rC, z80.rIYl

Opcodedd54::
	LD_XY_R_R z80.rD, z80.rIXh

Opcodedd55::
	LD_XY_R_R z80.rD, z80.rIXl

Opcodefd54::
	LD_XY_R_R z80.rD, z80.rIYh

Opcodefd55::
	LD_XY_R_R z80.rD, z80.rIYl

Opcodedd5c::
	LD_XY_R_R z80.rE, z80.rIXh

Opcodedd5d::
	LD_XY_R_R z80.rE, z80.rIXl

Opcodefd5c::
	LD_XY_R_R z80.rE, z80.rIYh

Opcodefd5d::
	LD_XY_R_R z80.rE, z80.rIYl

Opcodedd7c::
	LD_XY_R_R al, z80.rIXh

Opcodedd7d::
	LD_XY_R_R al, z80.rIXl

Opcodefd7c::
	LD_XY_R_R al, z80.rIYh

Opcodefd7d::
	LD_XY_R_R al, z80.rIYl

Opcodedd60::
	LD_XY_R_R z80.rIXh, z80.rB

Opcodedd61::
	LD_XY_R_R z80.rIXh, z80.rC

Opcodedd62::
	LD_XY_R_R z80.rIXh, z80.rD

Opcodedd63::
	LD_XY_R_R z80.rIXh, z80.rE

;Opcodedd64::
;	LD_XY_R_R z80.rIXh, z80.rIXh

Opcodedd65::
	LD_XY_R_R z80.rIXh, z80.rIXl

Opcodedd67::
	LD_XY_R_R z80.rIXh, al

Opcodedd68::
	LD_XY_R_R z80.rIXl, z80.rB

Opcodedd69::
	LD_XY_R_R z80.rIXl, z80.rC

Opcodedd6a::
	LD_XY_R_R z80.rIXl, z80.rD

Opcodedd6b::
	LD_XY_R_R z80.rIXl, z80.rE

Opcodedd6c::
	LD_XY_R_R z80.rIXl, z80.rIXh

;Opcodedd6d::
;	LD_XY_R_R z80.rIXl, z80.rIXl

Opcodedd6f::
	LD_XY_R_R z80.rIXl, al

Opcodefd60::
	LD_XY_R_R z80.rIYh, z80.rB

Opcodefd61::
	LD_XY_R_R z80.rIYh, z80.rC

Opcodefd62::
	LD_XY_R_R z80.rIYh, z80.rD

Opcodefd63::
	LD_XY_R_R z80.rIYh, z80.rE

;Opcodefd64::
;	LD_XY_R_R z80.rIYh, z80.rIYh

Opcodefd65::
	LD_XY_R_R z80.rIYh, z80.rIYl

Opcodefd67::
	LD_XY_R_R z80.rIYh, al

Opcodefd68::
	LD_XY_R_R z80.rIYl, z80.rB

Opcodefd69::
	LD_XY_R_R z80.rIYl, z80.rC

Opcodefd6a::
	LD_XY_R_R z80.rIYl, z80.rD

Opcodefd6b::
	LD_XY_R_R z80.rIYl, z80.rE

Opcodefd6c::
	LD_XY_R_R z80.rIYl, z80.rIYh

;Opcodefd6d::
;	LD_XY_R_R z80.rIYl, z80.rIYl

Opcodefd6f::
	LD_XY_R_R z80.rIYl, al

; ****************************************************************************
; LD r,n

LD_R_N	MACRO r
	call	ReadPC_dl
	mov	r, dl

	sub	esi, 7 + 1
	retn
	ENDM

Opcode06::
	LD_R_N z80.rB

Opcode0E::
	LD_R_N z80.rC
	
Opcode16::
	LD_R_N z80.rD

Opcode1E::
	LD_R_N z80.rE
	
Opcode26::
	LD_R_N z80.rH

Opcode2E::
	LD_R_N z80.rL
	
Opcode3E::
	LD_R_N al
	
; ****************************************************************************
; LD r,n (r=IXl,IXh,IYl,IYh)

LD_XY_R_N MACRO r
	call	ReadPC_dl
	mov	r, dl

	sub	esi, 11 + 2
	retn
	ENDM

Opcodedd26::
	LD_XY_R_N z80.rIXh
	
Opcodedd2e::
	LD_XY_R_N z80.rIXl

Opcodefd26::
	LD_XY_R_N z80.rIYh

Opcodefd2e::
	LD_XY_R_N z80.rIYl

; ****************************************************************************
; LD r,(HL)

LD_R_HL MACRO r
	mov	cx, z80.rHL
	call	ReadMem_dh
	mov	r, dh

	sub	esi, 7 + 1
	retn
	ENDM

Opcode46::
	LD_R_HL z80.rB

Opcode4e::
	LD_R_HL z80.rC

Opcode56::
	LD_R_HL z80.rD

Opcode5e::
	LD_R_HL z80.rE

Opcode66::
	LD_R_HL z80.rH

Opcode6e::
	LD_R_HL z80.rL

Opcode7e::
	LD_R_HL al

; ****************************************************************************
; LD r,(XY+d)

LD_R_XY MACRO xy, r
	call	ReadPC_dl
	movsx	cx, dl
	add	cx, xy
	call	ReadMem_dl
	mov	r, dl
	
	sub	esi, 19 + 2
	retn
	ENDM

Opcodedd46::
	LD_R_XY z80.rIX, z80.rB

Opcodefd46::
	LD_R_XY z80.rIY, z80.rB

Opcodedd4e::
	LD_R_XY z80.rIX, z80.rC

Opcodefd4e::
	LD_R_XY z80.rIY, z80.rC

Opcodedd56::
	LD_R_XY z80.rIX, z80.rD

Opcodefd56::
	LD_R_XY z80.rIY, z80.rD

Opcodedd5e::
	LD_R_XY z80.rIX, z80.rE

Opcodefd5e::
	LD_R_XY z80.rIY, z80.rE

Opcodedd66::
	LD_R_XY z80.rIX, z80.rH

Opcodefd66::
	LD_R_XY z80.rIY, z80.rH

Opcodedd6e::
	LD_R_XY z80.rIX, z80.rL

Opcodefd6e::
	LD_R_XY z80.rIY, z80.rL

Opcodedd7e::
	LD_R_XY z80.rIX, al

Opcodefd7e::
	LD_R_XY z80.rIY, al

; ****************************************************************************
; LD (HL),r

LD_HL_R MACRO r
	mov	cx, z80.rHL
	mov	dl, r
	call	WriteMem_dl

	sub	esi, 7 + 1
	retn
	ENDM

Opcode70::
	LD_HL_R z80.rB

Opcode71::
	LD_HL_R z80.rC

Opcode72::
	LD_HL_R z80.rD

Opcode73::
	LD_HL_R z80.rE

Opcode74::
	LD_HL_R z80.rH

Opcode75::
	LD_HL_R z80.rL

Opcode77::
	LD_HL_R al

; ****************************************************************************
; LD (XY+d),r

LD_XY_R MACRO xy, r
	call	ReadPC_dl
	movsx	cx, dl
	add	cx, xy
	mov	dl, r
	call	WriteMem_dl

	sub	esi, 19 + 2
	retn
	ENDM

Opcodedd70::
	LD_XY_R z80.rIX, z80.rB

Opcodefd70::
	LD_XY_R z80.rIY, z80.rB

Opcodedd71::
	LD_XY_R z80.rIX, z80.rC

Opcodefd71::
	LD_XY_R z80.rIY, z80.rC

Opcodedd72::
	LD_XY_R z80.rIX, z80.rD

Opcodefd72::
	LD_XY_R z80.rIY, z80.rD

Opcodedd73::
	LD_XY_R z80.rIX, z80.rE

Opcodefd73::
	LD_XY_R z80.rIY, z80.rE

Opcodedd74::
	LD_XY_R z80.rIX, z80.rH

Opcodefd74::
	LD_XY_R z80.rIY, z80.rH

Opcodedd75::
	LD_XY_R z80.rIX, z80.rL

Opcodefd75::
	LD_XY_R z80.rIY, z80.rL

Opcodedd77::
	LD_XY_R z80.rIX, al

Opcodefd77::
	LD_XY_R z80.rIY, al

; ****************************************************************************
; LD (HL),n / LD (XY+d),n

Opcode36::
	call	ReadPC_dl
	mov	cx, z80.rHL
	call	WriteMem_dl

	sub	esi, 10 + 1
	retn

Opcodedd36::
	call	ReadPC_dh
	call	ReadPC_dl
	movsx	cx, dh
	add	cx, z80.rIX
	call	WriteMem_dl

	sub	esi, 19 + 2
	retn

Opcodefd36::
	call	ReadPC_dh
	call	ReadPC_dl
	movsx	cx, dh
	add	cx, z80.rIY
	call	WriteMem_dl

	sub	esi, 19 + 2
	retn

; ****************************************************************************
; LD A,(BC) / LD A,(DE) / LD A,(nn)

Opcode0a::
	mov	cx, z80.rBC
	call	ReadMem_dl
	mov	al, dl

	sub	esi, 7 + 1
	retn

Opcode1a::
	mov	cx, z80.rDE
	call	ReadMem_dl
	mov	al, dl

	sub	esi, 7 + 1
	retn

Opcode3a::
	call	ReadPC_dl
	call	ReadPC_dh
	mov	cx, dx
	call	ReadMem_dl
	mov	al, dl

	sub	esi, 13 + 1
	retn

; ****************************************************************************
; LD (BC),A / LD (DE),A / LD (nn),A

Opcode02::
	mov	cx, z80.rBC
	mov	dl, al
	call	WriteMem_dl

	sub	esi, 7 + 1
	retn

Opcode12::
	mov	cx, z80.rDE
	mov	dl, al
	call	WriteMem_dl

	sub	esi, 7 + 1
	retn

Opcode32::
	call	ReadPC_dl
	call	ReadPC_dh
	mov	cx, dx
	mov	dl, al
	call	WriteMem_dl

	sub	esi, 13 + 1
	retn

; ****************************************************************************
; LD [ARI],[ARI]

Opcodeed57:: ; LD A,I
	mov	al, z80.rI

	and	ah, FLAG_C
	mov	dl, ah

	test	al, al
	lahf
	and	ah, FLAG_S OR FLAG_Z
	or	ah, dl
	or	ah, z80.fIFF
	mov	rFb35, al

	sub	esi, 9 + 2
	retn

Opcodeed5f:: ; LD A,R
	mov	dl, ah
	and	dl, FLAG_C

	mov	al, rRb7
	and	al, BIT_7
	and	ebp, NOT BIT_7 AND 255
	or	eax, ebp

	test	al, al
	lahf
	and	ah, FLAG_S OR FLAG_Z
	or	ah, dl
	or	ah, z80.fIFF
	mov	rFb35, al

	sub	esi, 9 + 2
	retn

Opcodeed47:: ; LD I,A
	mov	z80.rI, al

	sub	esi, 9 + 2
	retn

Opcodeed4f:: ; LD R,A
	mov	rRb7, al
	mov	ebp, eax

	sub	esi, 9 + 2
	retn

; ****************************************************************************
; LD dd,nn

LD_DD_NN MACRO r
	call	ReadPC_dl
	call	ReadPC_dh
	mov	r, dx
	sub	esi, 10+1
	retn
	ENDM

Opcode01::
	LD_DD_NN z80.rBC

Opcode11::
	LD_DD_NN z80.rDE

Opcode21::
	LD_DD_NN z80.rHL

Opcode31::
	LD_DD_NN z80.rSP

LD_XY_NN MACRO r
	call	ReadPC_dl
	call	ReadPC_dh
	mov	r, dx

	sub	esi, 14 + 2
	retn
	ENDM

Opcodedd21::
	LD_XY_NN z80.rIX

Opcodefd21::
	LD_XY_NN z80.rIY

; ****************************************************************************
; LD dd,(nn)

Opcode2a::
	call	ReadPC_dl
	call	ReadPC_dh
	mov	cx, dx
	ReadMemW
	mov	z80.rHL, dx

	sub	esi, 16 + 1
	retn

LD_DD_MM MACRO r
	call	ReadPC_dl
	call	ReadPC_dh
	mov	cx, dx
	ReadMemW
	mov	r, dx

	sub	esi, 20 + 2
	retn
	ENDM

Opcodeed4b::
	LD_DD_MM z80.rBC

Opcodeed5b::
	LD_DD_MM z80.rDE

Opcodeed6b::
	LD_DD_MM z80.rHL

Opcodeed7b::
	LD_DD_MM z80.rSP

Opcodedd2a::
	LD_DD_MM z80.rIX

Opcodefd2a::
	LD_DD_MM z80.rIY

; ****************************************************************************
; LD (mm),dd

Opcode22::
	call	ReadPC_dl
	call	ReadPC_dh
	mov	cx, dx
	mov	dx, z80.rHL
	WriteMemW

	sub	esi, 16 + 1
	retn

LD_MM_DD MACRO r
	call	ReadPC_dl
	call	ReadPC_dh
	mov	cx, dx
	mov	dx, r
	WriteMemW

	sub	esi, 20 + 2
	retn
	ENDM

Opcodeed43::
	LD_MM_DD z80.rBC

Opcodeed53::
	LD_MM_DD z80.rDE

Opcodeed63::
	LD_MM_DD z80.rHL

Opcodeed73::
	LD_MM_DD z80.rSP

Opcodedd22::
	LD_MM_DD z80.rIX

Opcodefd22::
	LD_MM_DD z80.rIY

; ****************************************************************************
; LD SP,dd

Opcodef9::
	mov	dx, z80.rHL
	mov	z80.rSP, dx

	sub	esi, 6 + 1
	retn

Opcodeddf9::
	mov	dx, z80.rIX
	mov	z80.rSP, dx

	sub	esi, 10 + 2
	retn

Opcodefdf9::
	mov	dx, z80.rIY
	mov	z80.rSP, dx

	sub	esi, 10 + 2
	retn

; ****************************************************************************
; PUSH dd

PUSH_DD MACRO r
	mov	dx, r
	DoPush

	sub	esi, 11 + 1
	retn
	ENDM

Opcodec5::
	PUSH_DD z80.rBC

Opcoded5::
	PUSH_DD z80.rDE

Opcodee5::
	PUSH_DD z80.rHL

Opcodef5::
	mov	dh, rFb35
	mov	dl, ah
	and	dx, 028d7h
	or	dl, dh
	mov	dh, al
	DoPush
	
	sub	esi, 11 + 1
	retn

PUSH_XY MACRO r
	mov	dx, r
	DoPush
	sub	esi, 15 + 2
	retn
	ENDM

Opcodedde5::
	PUSH_XY z80.rIX

Opcodefde5::
	PUSH_XY z80.rIY

; ****************************************************************************
; POP dd

POP_DD MACRO r
	DoPop
	mov	r, dx
	sub	esi, 11 + 1
	retn
	ENDM

Opcodec1::
	POP_DD z80.rBC

Opcoded1::
	POP_DD z80.rDE

Opcodee1::
	POP_DD z80.rHL

Opcodef1::
	DoPop
	mov	al, dh
	mov	rFb35, dl
	mov	ah, dl

	sub	esi, 11 + 1
	retn

POP_XY MACRO r
	DoPop
	mov	r, dx

	sub	esi, 14 + 2
	retn
	ENDM

Opcodedde1::
	POP_XY z80.rIX

Opcodefde1::
	POP_XY z80.rIY

; ****************************************************************************
; EX group

Opcodeeb::
	mov	dx, z80.rHL
	mov	cx, z80.rDE
	mov	z80.rHL, cx
	mov	z80.rDE, dx
	
	sub	esi, 4 + 1
	retn

Opcode08::
	mov	dh, rFb35
	mov	dl, ah
	and	dx, 028d7h
	or	dl, dh
	mov	dh, al
	mov	ah, BYTE PTR z80.rAF_
	mov	al, BYTE PTR z80.rAF_ + 1
	mov	rFb35, ah
	mov	z80.rAF_, dx
	
	sub	esi, 4 + 1
	retn

Opcoded9::
	mov	dx, z80.rBC
	mov	cx, z80.rBC_
	mov	z80.rBC, cx
	mov	z80.rBC_, dx

	mov	dx, z80.rDE
	mov	cx, z80.rDE_
	mov	z80.rDE, cx
	mov	z80.rDE_, dx

	mov	dx, z80.rHL
	mov	cx, z80.rHL_
	mov	z80.rHL, cx
	mov	z80.rHL_, dx

	sub	esi, 4 + 1
	retn

Opcodee3::
	mov	cx, z80.rSP
	call	ReadMem_dl
	mov	cx, z80.rSP
	inc	cx
	call	ReadMem_dh
	xchg	z80.rHL, dx
	mov	cx, z80.rSP
	inc	cx
	call	WriteMem_dh
	mov	cx, z80.rSP
	call	WriteMem_dl

	sub	esi, 19 + 1
	retn

Opcodedde3::
	mov	cx, z80.rSP
	call	ReadMem_dl
	mov	cx, z80.rSP
	inc	cx
	call	ReadMem_dh
	xchg	z80.rIX, dx
	mov	cx, z80.rSP
	inc	cx
	call	WriteMem_dh
	mov	cx, z80.rSP
	call	WriteMem_dl

	sub	esi, 23 + 2
	retn

Opcodefde3::
	mov	cx, z80.rSP
	call	ReadMem_dl
	mov	cx, z80.rSP
	inc	cx
	call	ReadMem_dh
	xchg	z80.rIY, dx
	mov	cx, z80.rSP
	inc	cx
	call	WriteMem_dh
	mov	cx, z80.rSP
	call	WriteMem_dl

	sub	esi, 23 + 2
	retn

; ****************************************************************************
; Block Transfer (LD[ID][R ])

Opcodeeda0::	; ldi
	mov	cx, z80.rHL
	call	ReadMem_dh
	mov	cx, z80.rDE
	call	WriteMem_dh
	add	dh, al
	mov	dl, dh
	shl	dl, 4
	and	dx, FLAG_5 OR (FLAG_3 * 256)
	or	dl, dh
	mov	rFb35, dl
	and	ah, FLAG_S OR FLAG_Z OR FLAG_C
	inc	z80.rHL
	inc	z80.rDE
	dec	z80.rBC
	jz	ldi_zero
	or	ah, FLAG_P
ldi_zero:
	sub	esi, 16 + 2
	retn

Opcodeeda8::	; ldd
	mov	cx, z80.rHL
	call	ReadMem_dh
	mov	cx, z80.rDE
	call	WriteMem_dh
	add	dh, al
	mov	dl, dh
	shl	dl, 4
	and	dx, FLAG_5 OR (FLAG_3 * 256)
	or	dl, dh
	mov	rFb35, dl
	and	ah, FLAG_S OR FLAG_Z OR FLAG_C
	dec	z80.rHL
	dec	z80.rDE
	dec	z80.rBC
	jz	ldd_zero
	or	ah, FLAG_P

ldd_zero:
	sub	esi, 16 + 2
	retn

ldir_loop:
	add	ebp, 2
Opcodeedb0::
	mov	cx, z80.rHL
	call	ReadMem_dh
	mov	cx, z80.rDE
	call	WriteMem_dh
	inc	z80.rHL
	inc	z80.rDE
	dec	z80.rBC
	jz	ldir_zero
	
	sub	esi, 21 + 2
	jnc	ldir_loop
	
	sub	edi, 2
	add	dh, al
	mov	dl, dh
	shl	dl, 4
	and	dx, FLAG_5 OR (FLAG_3 * 256)
	or	dl, dh
	mov	rFb35, dl
	and	ah, FLAG_S OR FLAG_Z OR FLAG_C
	or	ah, FLAG_P
	add	esi, 255	; ZF = 0, CF=1
	retn

ldir_zero:
	and	ah, FLAG_S OR FLAG_Z OR FLAG_C
	add	dh, al
	mov	dl, dh
	shl	dl, 4
	and	dx, FLAG_5 OR (FLAG_3 * 256)
	or	dl, dh
	mov	rFb35, dl
	sub	esi, 16 + 2
	retn


lddr_loop:
	add	ebp, 2
Opcodeedb8::
	mov	cx, z80.rHL
	call	ReadMem_dh
	mov	cx, z80.rDE
	call	WriteMem_dh
	dec	z80.rHL
	dec	z80.rDE
	dec	z80.rBC
	jz	lddr_zero
	
	sub	esi, 21 + 2
	jnc	lddr_loop
	
	sub	edi, 2
	add	dh, al
	mov	dl, dh
	shl	dl, 4
	and	dx, FLAG_5 OR (FLAG_3 * 256)
	or	dl, dh
	mov	rFb35, dl
	and	ah, FLAG_S OR FLAG_Z OR FLAG_C
	or	ah, FLAG_P
	add	esi, 255	; ZF = 0, CF=1
	retn

lddr_zero:
	and	ah, FLAG_S OR FLAG_Z OR FLAG_C
	add	dh, al
	mov	dl, dh
	shl	dl, 4
	and	dx, FLAG_5 OR (FLAG_3 * 256)
	or	dl, dh
	mov	rFb35, dl
	sub	esi, 16 + 2
	retn

; ****************************************************************************
; search group CP[ID][R ]

Opcodeeda1::
	mov	cx, z80.rHL
	call	ReadMem_dh
	mov	cx, ax
	and	ch, FLAG_C
	sub	cl, dh
	lahf
	and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_N
	or	ah, ch
	inc	z80.rHL
	dec	z80.rBC
	jz	cpi_zero
	or	ah, FLAG_P
cpi_zero:
	test	ah, FLAG_H
	jz	cpi_hzero
	dec	cl
cpi_hzero:
	mov	ch, cl
	shl	cl, 4
	and	cx, FLAG_5 OR (FLAG_3 * 256)
	or	cl, ch
	mov	rFb35, cl
	
	sub	esi, 16 + 2
	retn

Opcodeeda9::
	mov	cx, z80.rHL
	call	ReadMem_dh
	mov	cx, ax
	and	ch, FLAG_C
	sub	cl, dh
	lahf
	and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_N
	or	ah, ch
	dec	z80.rHL
	dec	z80.rBC
	jz	cpd_zero
	or	ah, FLAG_P
cpd_zero:
	test	ah, FLAG_H
	jz	cpd_hzero
	dec	cl
cpd_hzero:
	mov	ch, cl
	shl	cl, 4
	and	cx, FLAG_5 OR (FLAG_3 * 256)
	or	cl, ch
	mov	rFb35, cl
	
	sub	esi, 16 + 2
	retn

Opcodeedb1::	; cpir
	and	ah, FLAG_C
	sub	ebp, 2
cpir_loop:
	add	ebp, 2
	mov	cx, z80.rHL
	call	ReadMem_dh
	inc	z80.rHL
	dec	z80.rBC
	jz	cpir_endbc
	cmp	al, dh
	jz	cpir_end
	sub	esi, 21 + 2
	jnc	cpir_loop

	mov	cx, ax
	sub	cl, dh
	lahf
	and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_N
	or	ah, ch
	test	ah, FLAG_H
	jz	cpir_hzero
	dec	cl
cpir_hzero:
	mov	ch, cl
	shl	cl, 4
	and	cx, FLAG_5 OR (FLAG_3 * 256)
	or	cl, ch
	mov	rFb35, cl

	sub	edi, 2
	or	ah, FLAG_P
	add	esi, 255	; ZF = 0, CF=1
	retn

cpir_end:
	or	ah, FLAG_P
cpir_endbc:
	mov	cx, ax
	sub	cl, dh
	lahf
	and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_N
	or	ah, ch
	test	ah, FLAG_H
	jz	cpir_hzero2
	dec	cl
cpir_hzero2:
	mov	ch, cl
	shl	cl, 4
	and	cx, FLAG_5 OR (FLAG_3 * 256)
	or	cl, ch
	mov	rFb35, cl
	
	sub	esi, 16 + 2
	retn

Opcodeedb9::	; cpdr
	and	ah, FLAG_C
	sub	ebp, 2
cpdr_loop:
	add	ebp, 2
	mov	cx, z80.rHL
	call	ReadMem_dh
	dec	z80.rHL
	dec	z80.rBC
	jz	cpdr_endbc
	cmp	al, dh
	jz	cpdr_end
	sub	esi, 21 + 2
	jnc	cpdr_loop

	mov	cx, ax
	sub	cl, dh
	lahf
	and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_N
	or	ah, ch
	test	ah, FLAG_H
	jz	cpdr_hzero
	dec	cl
cpdr_hzero:
	mov	ch, cl
	shl	cl, 4
	and	cx, FLAG_5 OR (FLAG_3 * 256)
	or	cl, ch
	mov	rFb35, cl

	sub	edi, 2
	or	ah, FLAG_P
	add	esi, 255	; ZF = 0, CF=1
	retn

cpdr_end:
	or	ah, FLAG_P
cpdr_endbc:
	mov	cx, ax
	sub	cl, dh
	lahf
	and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_N
	or	ah, ch
	test	ah, FLAG_H
	jz	cpdr_hzero2
	dec	cl
cpdr_hzero2:
	mov	ch, cl
	shl	cl, 4
	and	cx, FLAG_5 OR (FLAG_3 * 256)
	or	cl, ch
	mov	rFb35, cl
	
	sub	esi, 16 + 2
	retn

; ****************************************************************************
; 8 bit Arithmatic

ART_R	MACRO i, r
	LOCAL	overflow

	IFIDN <i>, <adc>
		sahf
	ELSEIFIDN <i>, <sbb>
		sahf
	ENDIF
	i	al, r
	mov	rFb35, al
	lahf
	jo	overflow
	IFIDN <i>, <adc>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_C
	ELSEIFIDN <i>, <add>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_C
	ELSEIFIDN <i>, <sub>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_N OR FLAG_C
	ELSEIFIDN <i>, <sbb>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_N OR FLAG_C
	ELSE
		.ERR <i parm wrong>
	ENDIF
	
	sub	esi, 4 + 1
	retn

overflow:
	IFIDN <i>, <adc>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_C
		or	ah, FLAG_P
	ELSEIFIDN <i>, <add>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_C
		or	ah, FLAG_P
	ELSEIFIDN <i>, <sub>
		or	ah, FLAG_P
	ELSEIFIDN <i>, <sbb>
		or	ah, FLAG_P
	ELSE
		.ERR <i parm wrong>
	ENDIF
	
	sub	esi, 4 + 1
	retn
	ENDM

ART_XY_R MACRO i, r
	LOCAL	overflow

	IFIDN <i>, <adc>
		sahf
	ELSEIFIDN <i>, <sbb>
		sahf
	ENDIF
	i	al, r
	mov	rFb35, al
	lahf
	jo	overflow
	IFIDN <i>, <adc>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_C
	ELSEIFIDN <i>, <add>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_C
	ELSEIFIDN <i>, <sub>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_N OR FLAG_C
	ELSEIFIDN <i>, <sbb>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_N OR FLAG_C
	ELSE
		.ERR <i parm wrong>
	ENDIF
	
	sub	esi, 8 + 2
	retn

overflow:
	IFIDN <i>, <adc>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_C
		or	ah, FLAG_P
	ELSEIFIDN <i>, <add>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_C
		or	ah, FLAG_P
	ELSEIFIDN <i>, <sub>
		or	ah, FLAG_P
	ELSEIFIDN <i>, <sbb>
		or	ah, FLAG_P
	ELSE
		.ERR <i parm wrong>
	ENDIF
	
	sub	esi, 8 + 2
	retn
	ENDM

ART_HL	MACRO i
	LOCAL	overflow

	mov	cx, z80.rHL
	call	ReadMem_dl
	IFIDN <i>, <adc>
		sahf
	ELSEIFIDN <i>, <sbb>
		sahf
	ENDIF
	i	al, dl
	mov	rFb35, al
	lahf
	jo	overflow
	IFIDN <i>, <adc>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_C
	ELSEIFIDN <i>, <add>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_C
	ELSEIFIDN <i>, <sub>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_N OR FLAG_C
	ELSEIFIDN <i>, <sbb>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_N OR FLAG_C
	ELSE
		.ERR <i parm wrong>
	ENDIF
	
	sub	esi, 7 + 1
	retn

overflow:
	IFIDN <i>, <adc>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_C
		or	ah, FLAG_P
	ELSEIFIDN <i>, <add>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_C
		or	ah, FLAG_P
	ELSEIFIDN <i>, <sub>
		or	ah, FLAG_P
	ELSEIFIDN <i>, <sbb>
		or	ah, FLAG_P
	ELSE
		.ERR <i parm wrong>
	ENDIF
	
	sub	esi, 7 + 1
	retn
	ENDM

ART_N	MACRO i
	LOCAL	overflow

	call	ReadPC_dl
	IFIDN <i>, <adc>
		sahf
	ELSEIFIDN <i>, <sbb>
		sahf
	ENDIF
	i	al, dl
	mov	rFb35, al
	lahf
	jo	overflow
	IFIDN <i>, <adc>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_C
	ELSEIFIDN <i>, <add>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_C
	ELSEIFIDN <i>, <sub>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_N OR FLAG_C
	ELSEIFIDN <i>, <sbb>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_N OR FLAG_C
	ELSE
		.ERR <i parm wrong>
	ENDIF
	
	sub	esi, 7 + 1
	retn

overflow:
	IFIDN <i>, <adc>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_C
		or	ah, FLAG_P
	ELSEIFIDN <i>, <add>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_C
		or	ah, FLAG_P
	ELSEIFIDN <i>, <sub>
		or	ah, FLAG_P
	ELSEIFIDN <i>, <sbb>
		or	ah, FLAG_P
	ELSE
		.ERR <i parm wrong>
	ENDIF
	
	sub	esi, 7 + 1
	retn
	ENDM

ART_XY	MACRO i, xy
	LOCAL	overflow

	call	ReadPC_dl
	movsx	cx, dl
	add	cx, xy
	call	ReadMem_dl

	IFIDN <i>, <adc>
		sahf
	ELSEIFIDN <i>, <sbb>
		sahf
	ENDIF
	i	al, dl
	mov	rFb35, al
	lahf
	jo	overflow
	IFIDN <i>, <adc>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_C
	ELSEIFIDN <i>, <add>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_C
	ELSEIFIDN <i>, <sub>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_N OR FLAG_C
	ELSEIFIDN <i>, <sbb>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_N OR FLAG_C
	ELSE
		.ERR <i parm wrong>
	ENDIF
	
	sub	esi, 19 + 2
	retn

overflow:
	IFIDN <i>, <adc>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_C
		or	ah, FLAG_P
	ELSEIFIDN <i>, <add>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_C
		or	ah, FLAG_P
	ELSEIFIDN <i>, <sub>
		or	ah, FLAG_P
	ELSEIFIDN <i>, <sbb>
		or	ah, FLAG_P
	ELSE
		.ERR <i parm wrong>
	ENDIF
	
	sub	esi, 19 + 2
	retn
	ENDM

Opcode80::	; ADD A,B
	ART_R add, z80.rB

Opcode81::	; ADD A,C
	ART_R add, z80.rC

Opcode82::	; ADD A,D
	ART_R add, z80.rD

Opcode83::	; ADD A,E
	ART_R add, z80.rE

Opcode84::	; ADD A,H
	ART_R add, z80.rH

Opcode85::	; ADD A,L
	ART_R add, z80.rL

Opcode87::	; ADD A,A
	ART_R add, al

Opcodedd84::	; ADD A,IXh
	ART_XY_R add, z80.rIXh

Opcodedd85::	; ADD A,IXl
	ART_XY_R add, z80.rIXl

Opcodefd84::	; ADD A,IYl
	ART_XY_R add, z80.rIYh

Opcodefd85::	; ADD A,IYh
	ART_XY_R add, z80.rIYl

Opcode86::	; ADD A,(HL)
	ART_HL add

Opcodec6::	; ADD A,n
	ART_N add

Opcodedd86::	; ADD A,(IX+d)
	ART_XY add, z80.rIX

Opcodefd86::	; ADD A,(IY+d)
	ART_XY add, z80.rIY

Opcode88::	; ADC A,B
	ART_R adc, z80.rB

Opcode89::	; ADC A,C
	ART_R adc, z80.rC

Opcode8a::	; ADC A,D
	ART_R adc, z80.rD

Opcode8b::	; ADC A,E
	ART_R adc, z80.rE

Opcode8c::	; ADC A,H
	ART_R adc, z80.rH

Opcode8d::	; ADC A,L
	ART_R adc, z80.rL

Opcode8f::	; ADC A,A
	ART_R adc, al

Opcodedd8c::	; ADC A,IXh
	ART_XY_R adc, z80.rIXh

Opcodedd8d::	; ADC A,IXl
	ART_XY_R adc, z80.rIXl

Opcodefd8c::	; ADC A,IYl
	ART_XY_R adc, z80.rIYh

Opcodefd8d::	; ADC A,IYh
	ART_XY_R adc, z80.rIYl

Opcode8e::	; ADC A,(HL)
	ART_HL adc

Opcodece::	; ADC A,n
	ART_N adc

Opcodedd8e::	; ADC A,(IX+d)
	ART_XY adc, z80.rIX

Opcodefd8e::	; ADC A,(IY+d)
	ART_XY adc, z80.rIY

Opcode90::	; SUB A,B
	ART_R sub, z80.rB

Opcode91::	; SUB A,C
	ART_R sub, z80.rC

Opcode92::	; SUB A,D
	ART_R sub, z80.rD

Opcode93::	; SUB A,E
	ART_R sub, z80.rE

Opcode94::	; SUB A,H
	ART_R sub, z80.rH

Opcode95::	; SUB A,L
	ART_R sub, z80.rL

Opcode97::	; SUB A,A
	mov	ax, (FLAG_Z OR FLAG_N) * 256
	mov	rFb35, al

	sub	esi, 4 + 1
	retn

Opcodedd94::	; SUB A,IXh
	ART_XY_R sub, z80.rIXh

Opcodedd95::	; SUB A,IXl
	ART_XY_R sub, z80.rIXl

Opcodefd94::	; SUB A,IYl
	ART_XY_R sub, z80.rIYh

Opcodefd95::	; SUB A,IYh
	ART_XY_R sub, z80.rIYl

Opcode96::	; SUB A,(HL)
	ART_HL sub

Opcoded6::	; SUB A,n
	ART_N sub

Opcodedd96::	; SUB A,(IX+d)
	ART_XY sub, z80.rIX

Opcodefd96::	; SUB A,(IY+d)
	ART_XY sub, z80.rIY

Opcode98::	; SBC A,B
	ART_R sbb, z80.rB

Opcode99::	; SBC A,C
	ART_R sbb, z80.rC

Opcode9a::	; SBC A,D
	ART_R sbb, z80.rD

Opcode9b::	; SBC A,E
	ART_R sbb, z80.rE

Opcode9c::	; SBC A,H
	ART_R sbb, z80.rH

Opcode9d::	; SBC A,L
	ART_R sbb, z80.rL

Opcode9f::	; SBC A,A
	test	ah, FLAG_C
	jz	sbca_reset
	mov	ax, (FLAG_S OR FLAG_H OR FLAG_N OR FLAG_C) * 256 + 255
	mov	rFb35, al

	sub	esi, 4 + 1
	retn

sbca_reset:
	mov	ax, (FLAG_Z OR FLAG_N) * 256
	mov	rFb35, al

	sub	esi, 4 + 1
	retn

Opcodedd9c::	; SBC A,IXh
	ART_XY_R sbb, z80.rIXh

Opcodedd9d::	; SBC A,IXl
	ART_XY_R sbb, z80.rIXl

Opcodefd9c::	; SBC A,IYl
	ART_XY_R sbb, z80.rIYh

Opcodefd9d::	; SBC A,IYh
	ART_XY_R sbb, z80.rIYl

Opcode9e::	; SBC A,(HL)
	ART_HL sbb

Opcodede::	; SBC A,n
	ART_N sbb

Opcodedd9e::	; SBC A,(IX+d)
	ART_XY sbb, z80.rIX

Opcodefd9e::	; SBC A,(IY+d)
	ART_XY sbb, z80.rIY

; ****************************************************************************
; 8 bit logical group

LOG_R MACRO i, r
	IFDIFI <r>, <al>
		i	al, r
	ELSEIFIDNI <i>, <xor>
		mov	eax, (FLAG_Z OR FLAG_P) * 256
		mov	rFb35, ah

		sub	esi, 4 + 1
		retn
		EXITM
	ENDIF
	mov	dl, al
	mov	rFb35, al
	IFIDNI	<i>, <and>
		mov	ah, ANDTbl[edx]
	ELSE
		mov	ah, ORTbl[edx]
	ENDIF

	sub	esi, 4 + 1
	retn
	ENDM

LOG_XY_R MACRO i, r
	i	al, r
	mov	dl, al
	mov	rFb35, al
	IFIDNI	<i>, <and>
		mov	ah, ANDTbl[edx]
	ELSE
		mov	ah, ORTbl[edx]
	ENDIF

	sub	esi, 8 + 2
	retn
	ENDM

LOG_N MACRO	i
	call	ReadPC_dl
	i	al, dl
	mov	dl, al
	mov	rFb35, al
	IFIDNI	<i>, <and>
		mov	ah, ANDTbl[edx]
	ELSE
		mov	ah, ORTbl[edx]
	ENDIF

	sub	esi, 7 + 1
	retn
	ENDM

LOG_HL	MACRO i
	mov	cx, z80.rHL
	call	ReadMem_dl
	i	al, dl
	mov	dl, al
	mov	rFb35, al
	IFIDNI	<i>, <and>
		mov	ah, ANDTbl[edx]
	ELSE
		mov	ah, ORTbl[edx]
	ENDIF
	
	sub	esi, 7 + 1
	retn
	ENDM

LOG_XY	MACRO i, xy
	call	ReadPC_dl
	movsx	cx, dl
	add	cx, xy
	call	ReadMem_dl
	i	al, dl
	mov	dl, al
	mov	rFb35, al
	IFIDNI	<i>, <and>
		mov	ah, ANDTbl[edx]
	ELSE
		mov	ah, ORTbl[edx]
	ENDIF

	sub	esi, 19 + 2
	retn
	ENDM

Opcodea0::	; AND B
	LOG_R and, z80.rB

Opcodea1::	; AND C
	LOG_R and, z80.rC

Opcodea2::	; AND D
	LOG_R and, z80.rD

Opcodea3::	; AND E
	LOG_R and, z80.rE

Opcodea4::	; AND H
	LOG_R and, z80.rH

Opcodea5::	; AND L
	LOG_R and, z80.rL

Opcodea7::	; AND A
	LOG_R and, al

Opcodedda4::	; AND IXh
	LOG_XY_R and, z80.rIXh

Opcodedda5::	; AND IXl
	LOG_XY_R and, z80.rIXl

Opcodefda4::	; AND IYh
	LOG_XY_R and, z80.rIYh

Opcodefda5::	; AND IYl
	LOG_XY_R and, z80.rIYl

Opcodee6::	; AND n
	LOG_N and

Opcodea6::	; AND (HL)
	LOG_HL and

Opcodedda6::	; AND (IX+d)
	LOG_XY and, z80.rIX

Opcodefda6::	; AND (IY+d)
	LOG_XY and, z80.rIY

Opcodea8::	; XOR B
	LOG_R xor, z80.rB

Opcodea9::	; XOR C
	LOG_R xor, z80.rC

Opcodeaa::	; XOR D
	LOG_R xor, z80.rD

Opcodeab::	; XOR E
	LOG_R xor, z80.rE

Opcodeac::	; XOR H
	LOG_R xor, z80.rH

Opcodead::	; XOR L
	LOG_R xor, z80.rL

Opcodeaf::	; XOR A
	LOG_R xor, al

Opcodeddac::	; XOR IXh
	LOG_XY_R xor, z80.rIXh

Opcodeddad::	; XOR IXl
	LOG_XY_R xor, z80.rIXl

Opcodefdac::	; XOR IYh
	LOG_XY_R xor, z80.rIYh

Opcodefdad::	; XOR IYl
	LOG_XY_R xor, z80.rIYl

Opcodeee::	; XOR n
	LOG_N xor

Opcodeae::	; XOR (HL)
	LOG_HL xor

Opcodeddae::	; XOR (IX+d)
	LOG_XY xor, z80.rIX

Opcodefdae::	; XOR (IY+d)
	LOG_XY xor, z80.rIY

Opcodeb0::	; OR B
	LOG_R or, z80.rB

Opcodeb1::	; OR C
	LOG_R or, z80.rC

Opcodeb2::	; OR D
	LOG_R or, z80.rD

Opcodeb3::	; OR E
	LOG_R or, z80.rE

Opcodeb4::	; OR H
	LOG_R or, z80.rH

Opcodeb5::	; OR L
	LOG_R or, z80.rL

Opcodeb7::	; OR A
	LOG_R or, al

Opcodeddb4::	; OR IXh
	LOG_XY_R or, z80.rIXh

Opcodeddb5::	; OR IXl
	LOG_XY_R or, z80.rIXl

Opcodefdb4::	; OR IYh
	LOG_XY_R or, z80.rIYh

Opcodefdb5::	; OR IYl
	LOG_XY_R or, z80.rIYl

Opcodef6::	; OR n
	LOG_N or

Opcodeb6::	; OR (HL)
	LOG_HL or

Opcodeddb6::	; OR (IX+d)
	LOG_XY or, z80.rIX

Opcodefdb6::	; OR (IY+d)
	LOG_XY or, z80.rIY

; ****************************************************************************
; CP group

CP_R MACRO r
	LOCAL	overflow
	
	mov	dl, r
	mov	rFb35, dl
	sahf
	cmp	al, dl
	lahf
	jo	overflow
	and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_N OR FLAG_C

	sub	esi, 4 + 1
	retn

overflow:
	or	ah, FLAG_V

	sub	esi, 4 + 1
	retn
	ENDM

Opcodeb8::
	CP_R z80.rB

Opcodeb9::
	CP_R z80.rC

Opcodeba::
	CP_R z80.rD

Opcodebb::
	CP_R z80.rE

Opcodebc::
	CP_R z80.rH

Opcodebd::
	CP_R z80.rL

Opcodebf::
	mov	rFb35, al
	mov	ah, FLAG_Z OR FLAG_N

	sub	esi, 4 + 1
	retn

CP_XY_R MACRO r
	LOCAL	@overflow

	mov	dl, r
	mov	rFb35, dl
	sahf
	cmp	al, dl
	lahf
	jo	@overflow
	and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_N OR FLAG_C

	sub	esi, 8 + 2
	retn


@overflow:
	or	ah, FLAG_V

	sub	esi, 8 + 2
	retn
	ENDM

Opcodeddbc::
	CP_XY_R z80.rIXh

Opcodeddbd::
	CP_XY_R z80.rIXl

Opcodefdbc::
	CP_XY_R z80.rIYh

Opcodefdbd::
	CP_XY_R z80.rIYl

Opcodebe::
	mov	cx, z80.rHL
	call	ReadMem_dl
	
	mov	rFb35, dl

	sahf
	cmp	al, dl
	lahf

	jo	cp_hl_overflow
	and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_N OR FLAG_C

	sub	esi, 7 + 1
	retn

cp_hl_overflow:
	or	ah, FLAG_V

	sub	esi, 7 + 1
	retn

Opcodefe::
	call	ReadPC_dl
	mov	rFb35, dl
	sahf
	cmp	al, dl
	lahf
	jo	ofe_overflow
	and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_N OR FLAG_C

	sub	esi, 7 + 1
	retn

ofe_overflow:
	or	ah, FLAG_V

	sub	esi, 7 + 1
	retn


Opcodeddbe::
	call	ReadPC_dl
	movsx	cx, dl
	add	cx, z80.rIX
	call	ReadMem_dl

	mov	rFb35, dl
	sahf
	cmp	al, dl
	lahf

	jo	cp_ix_overflow
	and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_N OR FLAG_C

	sub	esi, 19 + 2
	retn

cp_ix_overflow:
	or	ah, FLAG_V

	sub	esi, 19 + 2
	retn

Opcodefdbe::
	call	ReadPC_dl
	movsx	cx, dl
	add	cx, z80.rIY
	call	ReadMem_dl

	mov	rFb35, dl
	sahf
	cmp	al, dl
	lahf

	jo	cp_iy_overflow
	and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_N OR FLAG_C 

	sub	esi, 19 + 2
	retn

cp_iy_overflow:
	or	ah, FLAG_V

	sub	esi, 19 + 2
	retn

; ****************************************************************************
; DEC/INC 8 bit group

DECINC_R MACRO d, r
	LOCAL	overflow

	IFIDNI <r>, <al>
		sahf
		d	al
		lahf
		mov	rFb35, al
	ELSE
		mov	dl, r
		sahf
		d	dl
		lahf
		mov	rFb35, dl
		mov	r, dl
	ENDIF
	jo	overflow
	IFIDNI <d>, <dec>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_N OR FLAG_C
	ELSE
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_C
	ENDIF

	sub	esi, 4 + 1
	retn

overflow:
	IFIDNI <d>, <dec>
		or	ah, FLAG_V
	ELSE
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_C
		or	ah, FLAG_V
	ENDIF

	sub	esi, 4 + 1
	retn
	ENDM

DECINC_XY_R MACRO d, r
	LOCAL	overflow

	mov	dl, r
	sahf
	d	dl
	lahf
	mov	rFb35, dl
	mov	r, dl
	jo	overflow
	IFIDNI <d>, <dec>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_N OR FLAG_C
	ELSE
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_C
	ENDIF

	sub	esi, 8 + 2
	retn

overflow:
	IFIDNI <d>, <dec>
		or	ah, FLAG_V
	ELSE
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_C
		or	ah, FLAG_V
	ENDIF

	sub	esi, 8 + 2
	retn
	ENDM

DECINC_HL MACRO d
	LOCAL	overflow

	mov	cx, z80.rHL
	call	ReadMem_dl
	sahf
	d	dl
	lahf
	mov	rFb35, dl
	jo	overflow
	IFIDNI <d>, <dec>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_N OR FLAG_C
	ELSE
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_C
	ENDIF
	mov	cx, z80.rHL
	call	WriteMem_dl

	sub	esi, 11 + 1
	retn

overflow:
	IFIDNI <d>, <dec>
		or	ah, FLAG_V
	ELSE
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_C
		or	ah, FLAG_V
	ENDIF
	mov	cx, z80.rHL
	call	WriteMem_dl

	sub	esi, 11 + 1
	retn
	ENDM

DECINC_XY MACRO d, xy
	LOCAL	overflow

	call	ReadPC_dl
	movsx	cx, dl
	add	cx, xy
	push	cx
	call	ReadMem_dl
	sahf
	d	dl
	lahf
	mov	rFb35, dl
	jo	overflow
	IFIDNI <d>, <dec>
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_N OR FLAG_C
	ELSE
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_C
	ENDIF
	pop	cx
	call	WriteMem_dl

	sub	esi, 23 + 2
	retn

overflow:
	IFIDNI <d>, <dec>
		or	ah, FLAG_V
	ELSE
		and	ah, FLAG_S OR FLAG_Z OR FLAG_H OR FLAG_C
		or	ah, FLAG_V
	ENDIF
	pop	cx
	call	WriteMem_dl

	sub	esi, 23 + 2
	retn
	ENDM

Opcode04::
	DECINC_R inc, z80.rB

Opcode0c::
	DECINC_R inc, z80.rC

Opcode14::
	DECINC_R inc, z80.rD

Opcode1c::
	DECINC_R inc, z80.rE

Opcode24::
	DECINC_R inc, z80.rH

Opcode2c::
	DECINC_R inc, z80.rL

Opcode3c::
	DECINC_R inc, al

Opcodedd24::
	DECINC_XY_R inc, z80.rIXh

Opcodedd2c::
	DECINC_XY_R inc, z80.rIXl

Opcodefd24::
	DECINC_XY_R inc, z80.rIYh

Opcodefd2c::
	DECINC_XY_R inc, z80.rIYl

Opcode34::
	DECINC_HL inc

Opcodedd34::
	DECINC_XY inc, z80.rIX

Opcodefd34::
	DECINC_XY inc, z80.rIY

Opcode05::
	DECINC_R dec, z80.rB

Opcode0d::
	DECINC_R dec, z80.rC

Opcode15::
	DECINC_R dec, z80.rD

Opcode1d::
	DECINC_R dec, z80.rE

Opcode25::
	DECINC_R dec, z80.rH

Opcode2d::
	DECINC_R dec, z80.rL

Opcode3d::
	DECINC_R dec, al

Opcodedd25::
	DECINC_XY_R dec, z80.rIXh

Opcodedd2d::
	DECINC_XY_R dec, z80.rIXl

Opcodefd25::
	DECINC_XY_R dec, z80.rIYh

Opcodefd2d::
	DECINC_XY_R dec, z80.rIYl

Opcode35::
	DECINC_HL dec

Opcodedd35::
	DECINC_XY dec, z80.rIX

Opcodefd35::
	DECINC_XY dec, z80.rIY

; ****************************************************************************
; ADD HL,ss

COMMENT $

Another possibility:

	mov	cx, rr
	mov	dx, ss
	and	ah, FLAG_S OR FLAG_Z OR FLAG_P
	mov	bl, ah
	add	cl, dl
	adc	ch, dh
	lahf
	and	ah, FLAG_H OR FLAG_C
	or	ah, dl
	mov	rr, cx
	mov	rFb35, ch

13 cycles, below 12 cycles.
$

ADD_16 MACRO	rr, ss, t
	and	ah, FLAG_S OR FLAG_Z OR FLAG_P
	mov	cx, rr
	mov	dx, cx
	mov	bx, ss
	add	cx, bx
	adc	ah, 0
	xor	dx, cx
	xor	dx, bx
	and	dh, FLAG_H
	or	ah, dh
	mov	rFb35, ch
	mov	rr, cx

	sub	esi, t
	retn
	ENDM

ADD_16_S MACRO	rr, t
	mov	cx, rr
	and	ah, FLAG_S OR FLAG_Z OR FLAG_P
	add	cx, cx
	adc	ah, 0
	mov	rr, cx
	mov	rFb35, ch
	and	ch, FLAG_H
	or	ah, ch

	sub	esi, t
	retn
	ENDM

Opcode09::
	ADD_16 z80.rHL, z80.rBC, (11 + 1)

Opcode19::
	ADD_16 z80.rHL, z80.rDE, (11 + 1)

Opcode29::
	ADD_16_S z80.rHL, (11 + 1)

Opcode39::
	ADD_16 z80.rHL, z80.rSP, (11 + 1)

Opcodedd09::
	ADD_16 z80.rIX, z80.rBC, (15 + 2)

Opcodedd19::
	ADD_16 z80.rIX, z80.rDE, (15 + 2)

Opcodedd29::
	ADD_16_S z80.rIX, (15 + 2)

Opcodedd39::
	ADD_16 z80.rIX, z80.rSP, (15 + 2)

Opcodefd09::
	ADD_16 z80.rIY, z80.rBC, (15 + 2)

Opcodefd19::
	ADD_16 z80.rIY, z80.rDE, (15 + 2)

Opcodefd29::
	ADD_16_S z80.rIY, (15 + 2)

Opcodefd39::
	ADD_16 z80.rIY, z80.rSP, (15 + 2)

ADC_HL MACRO ss
	LOCAL zero, overflow, zero2

	mov	cx, z80.rHL
	mov	dx, ss
	sahf
	adc	cl, dl
	adc	ch, dh
	lahf
	mov	z80.rHL, cx
	mov	rFb35, ch
	jo	overflow
	and	ah, FLAG_S OR FLAG_H OR FLAG_C
	test	cx, cx
	jz	zero

	sub	esi, 15 + 2
	retn

zero:
	or	ah, FLAG_Z

	sub	esi, 15 + 2
	retn

overflow:
	and	ah, FLAG_S OR FLAG_H OR FLAG_C
	test	cx, cx
	jz	zero2
	or	ah, FLAG_V

	sub	esi, 15 + 2
	retn

zero2:
	or	ah, FLAG_V OR FLAG_Z

	sub	esi, 15 + 2
	retn
	ENDM

SBC_HL	MACRO	ss
	LOCAL	zero, overflow, zero2

	mov	cx, z80.rHL
	mov	dx, ss
	sahf
	sbb	cl, dl
	sbb	ch, dh
	lahf
	mov	z80.rHL, cx
	mov	rFb35, ch
	jo	overflow
	and	ah, FLAG_S OR FLAG_H OR FLAG_C OR FLAG_N
	test	cx, cx
	jz	zero

	sub	esi, 15 + 2
	retn

zero:
	or	ah, FLAG_Z

	sub	esi, 15 + 2
	retn

overflow:
	and	ah, FLAG_S OR FLAG_H OR FLAG_C OR FLAG_N
	test	cx, cx
	jz	zero2
	or	ah, FLAG_V

	sub	esi, 15 + 2
	retn

zero2:
	or	ah, FLAG_V OR FLAG_Z

	sub	esi, 15 + 2
	retn
	ENDM

Opcodeed4a::
	ADC_HL z80.rBC

Opcodeed5a::
	ADC_HL z80.rDE

Opcodeed6a::
	ADC_HL z80.rHL

Opcodeed7a::
	ADC_HL z80.rSP

Opcodeed42::
	SBC_HL z80.rBC

Opcodeed52::
	SBC_HL z80.rDE

Opcodeed62::
	test	ah, FLAG_C
	jz	oed63_nocarry
	mov	ah, FLAG_S OR FLAG_H OR FLAG_N OR FLAG_C
	mov	cx, -1
	mov	z80.rHL, cx
	mov	rFb35, ch

	sub	esi, 15 + 2
	retn

oed63_nocarry:
	mov	ah, FLAG_Z OR FLAG_N
	sub	cx, cx
	mov	z80.rHL, cx
	mov	rFb35, ch

	sub	esi, 15 + 2
	retn

Opcodeed72::
	SBC_HL z80.rSP

DECINC_16 MACRO d, ss, t
	d	ss

	sub	esi, t
	retn
	ENDM

Opcode03::
	DECINC_16 inc, z80.rBC, (1 + 6)

Opcode13::
	DECINC_16 inc, z80.rDE, (1 + 6)

Opcode23::
	DECINC_16 inc, z80.rHL, (1 + 6)

Opcode33::
	DECINC_16 inc, z80.rSP, (1 + 6)

Opcodedd23::
	DECINC_16 inc, z80.rIX, (2 + 10)

Opcodefd23::
	DECINC_16 inc, z80.rIY, (2 + 10)

Opcode0b::
	DECINC_16 dec, z80.rBC, (1 + 6)

Opcode1b::
	DECINC_16 dec, z80.rDE, (1 + 6)

Opcode2b::
	DECINC_16 dec, z80.rHL, (1 + 6)

Opcode3b::
	DECINC_16 dec, z80.rSP, (1 + 6)

Opcodedd2b::
	DECINC_16 dec, z80.rIX, (2 + 10)

Opcodefd2b::
	DECINC_16 dec, z80.rIY, (2 + 10)

; ****************************************************************************
; rotate/shift

R_SHORT MACRO i
	IFIDNI <i>, <rcl>
		mov	dl, ah
		and	dl, FLAG_S OR FLAG_Z OR FLAG_P
		sahf
		i	al, 33
		adc	dl, 0
		mov	ah, dl
	ELSEIFIDNI <i>, <rcr>
		mov	dl, ah
		and	dl, FLAG_S OR FLAG_Z OR FLAG_P
		sahf
		i	al, 33
		adc	dl, 0
		mov	ah, dl
	ELSE
		and	ah, FLAG_S OR FLAG_Z OR FLAG_P
		i	al, 33
		adc	ah, 0
	ENDIF
	mov	rFb35, al

	sub	esi, 4 + 1
	retn
	ENDM

Opcode07::	; RLCA
	R_SHORT rol

Opcode0f::	; RRCA
	R_SHORT ror

Opcode17::	; RLA
	R_SHORT rcl

Opcode1f::	; RRA
	R_SHORT rcr

ROT_R	MACRO i, r
	IFIDN <r>, <al>
		xor	dl, dl
		IFIDNI <i>, <rcl>
			sahf
		ELSEIFIDNI <i>, <rcr>
			sahf
		ENDIF
		i	al, 33
		adc	dl, dl
		test	al, al
		lahf
		and	ah, FLAG_S OR FLAG_Z OR FLAG_P
		or	ah, dl
		mov	rFb35, al
	ELSE
		mov	dl, r
		xor	dh, dh
		IFIDNI <i>, <rcl>
			sahf
		ELSEIFIDNI <i>, <rcr>
			sahf
		ENDIF
		i	dl, 33
		adc	dh, dh
		test	dl, dl
		lahf
		and	ah, FLAG_S OR FLAG_Z OR FLAG_P
		or	ah, dh
		mov	r, dl
		mov	rFb35, dl
	ENDIF

	sub	esi, 8 + 2
	retn
	ENDM

ROT_HL	MACRO i
	mov	cx, z80.rHL
	call	ReadMem_dl
	xor	dh, dh
	IFIDNI <i>, <rcl>
		sahf
	ELSEIFIDNI <i>, <rcr>
		sahf
	ENDIF
	i	dl, 33
	adc	dh, dh
	test	dl, dl
	lahf
	and	ah, FLAG_S OR FLAG_Z OR FLAG_P
	or	ah, dh
	mov	rFb35, dl
	mov	cx, z80.rHL
	call	WriteMem_dl

	sub	esi, 15 + 2
	retn
	ENDM

ROT_XY MACRO i, r
	push	cx
	call	ReadMem_dl
	xor	dh, dh
	IFIDNI <i>, <rcl>
		sahf
	ELSEIFIDNI <i>, <rcr>
		sahf
	ENDIF
	i	dl, 33
	adc	dh, dh
	test	dl, dl
	lahf
	and	ah, FLAG_S OR FLAG_Z OR FLAG_P
	or	ah, dh
	mov	rFb35, dl
	IFNB	<r>
		mov	r, dl
	ENDIF
	pop	cx
	call	WriteMem_dl

	sub	esi, 23 + 2
	retn
	ENDM

Opcodecb00::
	ROT_R rol, z80.rB

Opcodecb01::
	ROT_R rol, z80.rC

Opcodecb02::
	ROT_R rol, z80.rD

Opcodecb03::
	ROT_R rol, z80.rE

Opcodecb04::
	ROT_R rol, z80.rH

Opcodecb05::
	ROT_R rol, z80.rL

Opcodecb07::
	ROT_R rol, al

Opcodecb06::
	ROT_HL rol

Opcodeddcb00::
	ROT_XY rol, z80.rB

Opcodeddcb01::
	ROT_XY rol, z80.rC

Opcodeddcb02::
	ROT_XY rol, z80.rD

Opcodeddcb03::
	ROT_XY rol, z80.rE

Opcodeddcb04::
	ROT_XY rol, z80.rH

Opcodeddcb05::
	ROT_XY rol, z80.rL

Opcodeddcb06::
	ROT_XY rol

Opcodeddcb07::
	ROT_XY rol, al

Opcodecb08::
	ROT_R ror, z80.rB

Opcodecb09::
	ROT_R ror, z80.rC

Opcodecb0a::
	ROT_R ror, z80.rD

Opcodecb0b::
	ROT_R ror, z80.rE

Opcodecb0c::
	ROT_R ror, z80.rH

Opcodecb0d::
	ROT_R ror, z80.rL

Opcodecb0f::
	ROT_R ror, al

Opcodecb0e::
	ROT_HL ror

Opcodeddcb08::
	ROT_XY ror, z80.rB

Opcodeddcb09::
	ROT_XY ror, z80.rC

Opcodeddcb0a::
	ROT_XY ror, z80.rD

Opcodeddcb0b::
	ROT_XY ror, z80.rE

Opcodeddcb0c::
	ROT_XY ror, z80.rH

Opcodeddcb0d::
	ROT_XY ror, z80.rL

Opcodeddcb0e::
	ROT_XY ror

Opcodeddcb0f::
	ROT_XY ror, al

Opcodecb10::
	ROT_R rcl, z80.rB

Opcodecb11::
	ROT_R rcl, z80.rC

Opcodecb12::
	ROT_R rcl, z80.rD

Opcodecb13::
	ROT_R rcl, z80.rE

Opcodecb14::
	ROT_R rcl, z80.rH

Opcodecb15::
	ROT_R rcl, z80.rL

Opcodecb17::
	ROT_R rcl, al

Opcodecb16::
	ROT_HL rcl

Opcodeddcb10::
	ROT_XY rcl, z80.rB

Opcodeddcb11::
	ROT_XY rcl, z80.rC

Opcodeddcb12::
	ROT_XY rcl, z80.rD

Opcodeddcb13::
	ROT_XY rcl, z80.rE

Opcodeddcb14::
	ROT_XY rcl, z80.rH

Opcodeddcb15::
	ROT_XY rcl, z80.rL

Opcodeddcb16::
	ROT_XY rcl

Opcodeddcb17::
	ROT_XY rcl, al

Opcodecb18::
	ROT_R rcr, z80.rB

Opcodecb19::
	ROT_R rcr, z80.rC

Opcodecb1a::
	ROT_R rcr, z80.rD

Opcodecb1b::
	ROT_R rcr, z80.rE

Opcodecb1c::
	ROT_R rcr, z80.rH

Opcodecb1d::
	ROT_R rcr, z80.rL

Opcodecb1f::
	ROT_R rcr, al

Opcodecb1e::
	ROT_HL rcr

Opcodeddcb18::
	ROT_XY rcr, z80.rB

Opcodeddcb19::
	ROT_XY rcr, z80.rC

Opcodeddcb1a::
	ROT_XY rcr, z80.rD

Opcodeddcb1b::
	ROT_XY rcr, z80.rE

Opcodeddcb1c::
	ROT_XY rcr, z80.rH

Opcodeddcb1d::
	ROT_XY rcr, z80.rL

Opcodeddcb1e::
	ROT_XY rcr

Opcodeddcb1f::
	ROT_XY rcr, al

SHF_R	MACRO i, r
	IFIDNI	<r>, <al>
		IFIDNI <i>, <sal>
			add	al, al
			inc	al
		ELSEIFIDNI <i>, <shl>
			add	al, al
		ELSE
			i	al, 33
		ENDIF
		lahf
		and	ah, FLAG_S OR FLAG_Z OR FLAG_P OR FLAG_C
		mov	rFb35, al
	ELSE
		mov	dl, r
		IFIDNI <i>, <sal>
			add	dl, dl
			inc	dl
		ELSEIFIDNI <i>, <shl>
			add	dl, dl
		ELSE
			i	dl, 33
		ENDIF
		lahf
		and	ah, FLAG_S OR FLAG_Z OR FLAG_P OR FLAG_C
		mov	rFb35, dl
		mov	r, dl
	ENDIF

	sub	esi, 8 + 2
	retn
	ENDM

SHF_HL	MACRO i
	mov	cx, z80.rHL
	call	ReadMem_dl
	IFIDNI <i>, <sal>
		add	dl, dl
		inc	dl
	ELSEIFIDNI <i>, <shl>
		add	dl, dl
	ELSE
		i	dl, 33
	ENDIF
	lahf
	and	ah, FLAG_S OR FLAG_Z OR FLAG_P OR FLAG_C
	mov	rFb35, dl
	mov	cx, z80.rHL
	call	WriteMem_dl

	sub	esi, 15 + 2
	retn
	ENDM

SHF_XY MACRO i, r
	push	cx
	call	ReadMem_dl
	IFIDNI <i>, <sal>
		add	dl, dl
		inc	dl
	ELSEIFIDNI <i>, <shl>
		add	dl, dl
	ELSE
		i	dl, 33
	ENDIF
	lahf
	and	ah, FLAG_S OR FLAG_Z OR FLAG_P OR FLAG_C
	mov	rFb35, dl
	IFNB	<r>
		mov	r, dl
	ENDIF
	pop	cx
	call	WriteMem_dl

	sub	esi, 23 + 2
	retn
	ENDM

Opcodecb20::
	SHF_R shl, z80.rB

Opcodecb21::
	SHF_R shl, z80.rC

Opcodecb22::
	SHF_R shl, z80.rD

Opcodecb23::
	SHF_R shl, z80.rE

Opcodecb24::
	SHF_R shl, z80.rH

Opcodecb25::
	SHF_R shl, z80.rL

Opcodecb27::
	SHF_R shl, al

Opcodecb26::
	SHF_HL shl

Opcodeddcb20::
	SHF_XY shl, z80.rB

Opcodeddcb21::
	SHF_XY shl, z80.rC

Opcodeddcb22::
	SHF_XY shl, z80.rD

Opcodeddcb23::
	SHF_XY shl, z80.rE

Opcodeddcb24::
	SHF_XY shl, z80.rH

Opcodeddcb25::
	SHF_XY shl, z80.rL

Opcodeddcb26::
	SHF_XY shl

Opcodeddcb27::
	SHF_XY shl, al

Opcodecb28::
	SHF_R sar, z80.rB

Opcodecb29::
	SHF_R sar, z80.rC

Opcodecb2a::
	SHF_R sar, z80.rD

Opcodecb2b::
	SHF_R sar, z80.rE

Opcodecb2c::
	SHF_R sar, z80.rH

Opcodecb2d::
	SHF_R sar, z80.rL

Opcodecb2f::
	SHF_R sar, al

Opcodecb2e::
	SHF_HL sar

Opcodeddcb28::
	SHF_XY sar, z80.rB

Opcodeddcb29::
	SHF_XY sar, z80.rC

Opcodeddcb2a::
	SHF_XY sar, z80.rD

Opcodeddcb2b::
	SHF_XY sar, z80.rE

Opcodeddcb2c::
	SHF_XY sar, z80.rH

Opcodeddcb2d::
	SHF_XY sar, z80.rL

Opcodeddcb2e::
	SHF_XY sar

Opcodeddcb2f::
	SHF_XY sar, al

Opcodecb30::
	SHF_R sal, z80.rB

Opcodecb31::
	SHF_R sal, z80.rC

Opcodecb32::
	SHF_R sal, z80.rD

Opcodecb33::
	SHF_R sal, z80.rE

Opcodecb34::
	SHF_R sal, z80.rH

Opcodecb35::
	SHF_R sal, z80.rL

Opcodecb37::
	SHF_R sal, al

Opcodecb36::
	SHF_HL sal

Opcodeddcb30::
	SHF_XY sal, z80.rB

Opcodeddcb31::
	SHF_XY sal, z80.rC

Opcodeddcb32::
	SHF_XY sal, z80.rD

Opcodeddcb33::
	SHF_XY sal, z80.rE

Opcodeddcb34::
	SHF_XY sal, z80.rH

Opcodeddcb35::
	SHF_XY sal, z80.rL

Opcodeddcb36::
	SHF_XY sal

Opcodeddcb37::
	SHF_XY sal, al

Opcodecb38::
	SHF_R shr, z80.rB

Opcodecb39::
	SHF_R shr, z80.rC

Opcodecb3a::
	SHF_R shr, z80.rD

Opcodecb3b::
	SHF_R shr, z80.rE

Opcodecb3c::
	SHF_R shr, z80.rH

Opcodecb3d::
	SHF_R shr, z80.rL

Opcodecb3f::
	SHF_R shr, al

Opcodecb3e::
	SHF_HL shr

Opcodeddcb38::
	SHF_XY shr, z80.rB

Opcodeddcb39::
	SHF_XY shr, z80.rC

Opcodeddcb3a::
	SHF_XY shr, z80.rD

Opcodeddcb3b::
	SHF_XY shr, z80.rE

Opcodeddcb3c::
	SHF_XY shr, z80.rH

Opcodeddcb3d::
	SHF_XY shr, z80.rL

Opcodeddcb3e::
	SHF_XY shr

Opcodeddcb3f::
	SHF_XY shr, al

Opcodeed67::	; RRD
	mov	cx, z80.rHL
	call	ReadMem_dl
	mov	dh, al
	and	ah, FLAG_C
	mov	cl, ah
	ror	dx, 4
	shr	dh, 4
	and	al, 0f0h
	or	al, dh
	lahf
	and	ah, FLAG_S OR FLAG_Z OR FLAG_P
	or	ah, cl
	mov	rFb35, al
	mov	cx, z80.rHL
	call	WriteMem_dl

	sub	esi, 18 + 2
	retn

Opcodeed6f::	; RLD
	mov	cx, z80.rHL
	call	ReadMem_dl
	mov	dh, al
	and	ah, FLAG_C
	mov	cl, ah
	shl	dh, 4
	rol	dx, 4
	and	al, 0f0h
	or	al, dh
	lahf
	and	ah, FLAG_S OR FLAG_Z OR FLAG_P
	or	ah, cl
	mov	rFb35, al
	mov	cx, z80.rHL
	call	WriteMem_dl

	sub	esi, 18 + 2
	retn

; ****************************************************************************
; BIT group

BIT_R MACRO r,b
	LOCAL	nzero

	and	ah, FLAG_C
	IFIDNI <r>, <al>
		mov	rFb35, al
		test	al, b
		jnz	nzero
		or	ah, FLAG_Z OR FLAG_H OR FLAG_P

		sub	esi, 8 + 2
		retn
	ELSE
		mov	dl, r
		mov	rFb35, dl
		test	dl, b
		jnz	nzero
		or	ah, FLAG_Z OR FLAG_H OR FLAG_P

		sub	esi, 8 + 2
		retn
	ENDIF

nzero:
	IFIDNI <b>, <BIT_7>
		or	ah, FLAG_S OR FLAG_H
	ELSE
		or	ah, FLAG_H
	ENDIF

	sub	esi, 8 + 2
	retn
	ENDM

BIT_HL MACRO b
	LOCAL	nzero

	mov	cx, z80.rHL
	call	ReadMem_dl
	and	ah, FLAG_C
	mov	rFb35, ah	; not a bug!
	test	dl, b
	jnz	nzero
	or	ah, FLAG_Z OR FLAG_H OR FLAG_P

	sub	esi, 12 + 2
	retn

nzero:
	IFIDNI <b>, <BIT_7>
		or	ah, FLAG_S OR FLAG_H
	ELSE
		or	ah, FLAG_H
	ENDIF

	sub	esi, 12 + 2
	retn
	ENDM

BIT_XY MACRO b
	LOCAL	nzero

	call	ReadMem_dl
	and	ah, FLAG_C
	mov	rFb35, ah	; not a bug!
	test	dl, b
	jnz	nzero
	or	ah, FLAG_Z OR FLAG_H OR FLAG_P

	sub	esi, 20 + 2
	retn
nzero:
	IFIDNI <b>, <BIT_7>
		or	ah, FLAG_S OR FLAG_H
	ELSE
		or	ah, FLAG_H
	ENDIF

	sub	esi, 20 + 2
	retn
	ENDM

Opcodecb40::
	BIT_R z80.rB, BIT_0

Opcodecb41::
	BIT_R z80.rC, BIT_0

Opcodecb42::
	BIT_R z80.rD, BIT_0

Opcodecb43::
	BIT_R z80.rE, BIT_0

Opcodecb44::
	BIT_R z80.rH, BIT_0

Opcodecb45::
	BIT_R z80.rL, BIT_0

Opcodecb47::
	BIT_R al, BIT_0

Opcodecb46::
	BIT_HL BIT_0

Opcodeddcb40::
Opcodeddcb41::
Opcodeddcb42::
Opcodeddcb43::
Opcodeddcb44::
Opcodeddcb45::
Opcodeddcb46::
Opcodeddcb47::
	BIT_XY BIT_0
	
Opcodecb48::
	BIT_R z80.rB, BIT_1

Opcodecb49::
	BIT_R z80.rC, BIT_1

Opcodecb4a::
	BIT_R z80.rD, BIT_1

Opcodecb4b::
	BIT_R z80.rE, BIT_1

Opcodecb4c::
	BIT_R z80.rH, BIT_1

Opcodecb4d::
	BIT_R z80.rL, BIT_1

Opcodecb4f::
	BIT_R al, BIT_1

Opcodecb4e::
	BIT_HL BIT_1

Opcodeddcb48::
Opcodeddcb49::
Opcodeddcb4a::
Opcodeddcb4b::
Opcodeddcb4c::
Opcodeddcb4d::
Opcodeddcb4e::
Opcodeddcb4f::
	BIT_XY BIT_1

Opcodecb50::
	BIT_R z80.rB, BIT_2

Opcodecb51::
	BIT_R z80.rC, BIT_2

Opcodecb52::
	BIT_R z80.rD, BIT_2

Opcodecb53::
	BIT_R z80.rE, BIT_2

Opcodecb54::
	BIT_R z80.rH, BIT_2

Opcodecb55::
	BIT_R z80.rL, BIT_2

Opcodecb57::
	BIT_R al, BIT_2

Opcodecb56::
	BIT_HL BIT_2

Opcodeddcb50::
Opcodeddcb51::
Opcodeddcb52::
Opcodeddcb53::
Opcodeddcb54::
Opcodeddcb55::
Opcodeddcb56::
Opcodeddcb57::
	BIT_XY BIT_2
	
Opcodecb58::
	BIT_R z80.rB, BIT_3

Opcodecb59::
	BIT_R z80.rC, BIT_3

Opcodecb5a::
	BIT_R z80.rD, BIT_3

Opcodecb5b::
	BIT_R z80.rE, BIT_3

Opcodecb5c::
	BIT_R z80.rH, BIT_3

Opcodecb5d::
	BIT_R z80.rL, BIT_3

Opcodecb5f::
	BIT_R al, BIT_3

Opcodecb5e::
	BIT_HL BIT_3

Opcodeddcb58::
Opcodeddcb59::
Opcodeddcb5a::
Opcodeddcb5b::
Opcodeddcb5c::
Opcodeddcb5d::
Opcodeddcb5e::
Opcodeddcb5f::
	BIT_XY BIT_3

Opcodecb60::
	BIT_R z80.rB, BIT_4

Opcodecb61::
	BIT_R z80.rC, BIT_4

Opcodecb62::
	BIT_R z80.rD, BIT_4

Opcodecb63::
	BIT_R z80.rE, BIT_4

Opcodecb64::
	BIT_R z80.rH, BIT_4

Opcodecb65::
	BIT_R z80.rL, BIT_4

Opcodecb67::
	BIT_R al, BIT_4

Opcodecb66::
	BIT_HL BIT_4

Opcodeddcb60::
Opcodeddcb61::
Opcodeddcb62::
Opcodeddcb63::
Opcodeddcb64::
Opcodeddcb65::
Opcodeddcb66::
Opcodeddcb67::
	BIT_XY BIT_4
	
Opcodecb68::
	BIT_R z80.rB, BIT_5

Opcodecb69::
	BIT_R z80.rC, BIT_5

Opcodecb6a::
	BIT_R z80.rD, BIT_5

Opcodecb6b::
	BIT_R z80.rE, BIT_5

Opcodecb6c::
	BIT_R z80.rH, BIT_5

Opcodecb6d::
	BIT_R z80.rL, BIT_5

Opcodecb6f::
	BIT_R al, BIT_5

Opcodecb6e::
	BIT_HL BIT_5

Opcodeddcb68::
Opcodeddcb69::
Opcodeddcb6a::
Opcodeddcb6b::
Opcodeddcb6c::
Opcodeddcb6d::
Opcodeddcb6e::
Opcodeddcb6f::
	BIT_XY BIT_5

Opcodecb70::
	BIT_R z80.rB, BIT_6

Opcodecb71::
	BIT_R z80.rC, BIT_6

Opcodecb72::
	BIT_R z80.rD, BIT_6

Opcodecb73::
	BIT_R z80.rE, BIT_6

Opcodecb74::
	BIT_R z80.rH, BIT_6

Opcodecb75::
	BIT_R z80.rL, BIT_6

Opcodecb77::
	BIT_R al, BIT_6

Opcodecb76::
	BIT_HL BIT_6

Opcodeddcb70::
Opcodeddcb71::
Opcodeddcb72::
Opcodeddcb73::
Opcodeddcb74::
Opcodeddcb75::
Opcodeddcb76::
Opcodeddcb77::
	BIT_XY BIT_6
	
Opcodecb78::
	BIT_R z80.rB, BIT_7

Opcodecb79::
	BIT_R z80.rC, BIT_7

Opcodecb7a::
	BIT_R z80.rD, BIT_7

Opcodecb7b::
	BIT_R z80.rE, BIT_7

Opcodecb7c::
	BIT_R z80.rH, BIT_7

Opcodecb7d::
	BIT_R z80.rL, BIT_7

Opcodecb7f::
	BIT_R al, BIT_7

Opcodecb7e::
	BIT_HL BIT_7

Opcodeddcb78::
Opcodeddcb79::
Opcodeddcb7a::
Opcodeddcb7b::
Opcodeddcb7c::
Opcodeddcb7d::
Opcodeddcb7e::
Opcodeddcb7f::
	BIT_XY BIT_7

RES_R MACRO r,b
	and	r, NOT b

	sub	esi, 8 + 2
	retn
	ENDM

RES_HL MACRO b
	mov	cx, z80.rHL
	call	ReadMem_dl
	and	dl, NOT b
	mov	cx, z80.rHL
	call	WriteMem_dl

	sub	esi, 15 + 2
	retn
	ENDM

RES_XY MACRO b, r
	push	cx
	call	ReadMem_dl
	and	dl, NOT b
	IFNB <r>
		mov	r, dl
	ENDIF
	pop	cx
	call	WriteMem_dl

	sub	esi, 23 + 2
	retn
	ENDM

Opcodecb80::
	RES_R z80.rB, BIT_0

Opcodecb81::
	RES_R z80.rC, BIT_0

Opcodecb82::
	RES_R z80.rD, BIT_0

Opcodecb83::
	RES_R z80.rE, BIT_0

Opcodecb84::
	RES_R z80.rH, BIT_0

Opcodecb85::
	RES_R z80.rL, BIT_0

Opcodecb87::
	RES_R al, BIT_0

Opcodecb86::
	RES_HL BIT_0

Opcodeddcb80::
	RES_XY BIT_0, z80.rB

Opcodeddcb81::
	RES_XY BIT_0, z80.rC

Opcodeddcb82::
	RES_XY BIT_0, z80.rD

Opcodeddcb83::
	RES_XY BIT_0, z80.rE

Opcodeddcb84::
	RES_XY BIT_0, z80.rH

Opcodeddcb85::
	RES_XY BIT_0, z80.rL

Opcodeddcb86::
	RES_XY BIT_0

Opcodeddcb87::
	RES_XY BIT_0, al

Opcodecb88::
	RES_R z80.rB, BIT_1

Opcodecb89::
	RES_R z80.rC, BIT_1

Opcodecb8a::
	RES_R z80.rD, BIT_1

Opcodecb8b::
	RES_R z80.rE, BIT_1

Opcodecb8c::
	RES_R z80.rH, BIT_1

Opcodecb8d::
	RES_R z80.rL, BIT_1

Opcodecb8f::
	RES_R al, BIT_1

Opcodecb8e::
	RES_HL BIT_1

Opcodeddcb88::
	RES_XY BIT_1, z80.rB

Opcodeddcb89::
	RES_XY BIT_1, z80.rC

Opcodeddcb8a::
	RES_XY BIT_1, z80.rD

Opcodeddcb8b::
	RES_XY BIT_1, z80.rE

Opcodeddcb8c::
	RES_XY BIT_1, z80.rH

Opcodeddcb8d::
	RES_XY BIT_1, z80.rL

Opcodeddcb8e::
	RES_XY BIT_1

Opcodeddcb8f::
	RES_XY BIT_1, al

Opcodecb90::
	RES_R z80.rB, BIT_2

Opcodecb91::
	RES_R z80.rC, BIT_2

Opcodecb92::
	RES_R z80.rD, BIT_2

Opcodecb93::
	RES_R z80.rE, BIT_2

Opcodecb94::
	RES_R z80.rH, BIT_2

Opcodecb95::
	RES_R z80.rL, BIT_2

Opcodecb97::
	RES_R al, BIT_2

Opcodecb96::
	RES_HL BIT_2

Opcodeddcb90::
	RES_XY BIT_2, z80.rB

Opcodeddcb91::
	RES_XY BIT_2, z80.rC

Opcodeddcb92::
	RES_XY BIT_2, z80.rD

Opcodeddcb93::
	RES_XY BIT_2, z80.rE

Opcodeddcb94::
	RES_XY BIT_2, z80.rH

Opcodeddcb95::
	RES_XY BIT_2, z80.rL

Opcodeddcb96::
	RES_XY BIT_2

Opcodeddcb97::
	RES_XY BIT_2, al
	
Opcodecb98::
	RES_R z80.rB, BIT_3

Opcodecb99::
	RES_R z80.rC, BIT_3

Opcodecb9a::
	RES_R z80.rD, BIT_3

Opcodecb9b::
	RES_R z80.rE, BIT_3

Opcodecb9c::
	RES_R z80.rH, BIT_3

Opcodecb9d::
	RES_R z80.rL, BIT_3

Opcodecb9f::
	RES_R al, BIT_3

Opcodecb9e::
	RES_HL BIT_3

Opcodeddcb98::
	RES_XY BIT_3, z80.rB

Opcodeddcb99::
	RES_XY BIT_3, z80.rC

Opcodeddcb9a::
	RES_XY BIT_3, z80.rD

Opcodeddcb9b::
	RES_XY BIT_3, z80.rE

Opcodeddcb9c::
	RES_XY BIT_3, z80.rH

Opcodeddcb9d::
	RES_XY BIT_3, z80.rL

Opcodeddcb9e::
	RES_XY BIT_3

Opcodeddcb9f::
	RES_XY BIT_3, al

Opcodecba0::
	RES_R z80.rB, BIT_4

Opcodecba1::
	RES_R z80.rC, BIT_4

Opcodecba2::
	RES_R z80.rD, BIT_4

Opcodecba3::
	RES_R z80.rE, BIT_4

Opcodecba4::
	RES_R z80.rH, BIT_4

Opcodecba5::
	RES_R z80.rL, BIT_4

Opcodecba7::
	RES_R al, BIT_4

Opcodecba6::
	RES_HL BIT_4

Opcodeddcba0::
	RES_XY BIT_4, z80.rB

Opcodeddcba1::
	RES_XY BIT_4, z80.rC

Opcodeddcba2::
	RES_XY BIT_4, z80.rD

Opcodeddcba3::
	RES_XY BIT_4, z80.rE

Opcodeddcba4::
	RES_XY BIT_4, z80.rH

Opcodeddcba5::
	RES_XY BIT_4, z80.rL

Opcodeddcba6::
	RES_XY BIT_4

Opcodeddcba7::
	RES_XY BIT_4, al

Opcodecba8::
	RES_R z80.rB, BIT_5

Opcodecba9::
	RES_R z80.rC, BIT_5

Opcodecbaa::
	RES_R z80.rD, BIT_5

Opcodecbab::
	RES_R z80.rE, BIT_5

Opcodecbac::
	RES_R z80.rH, BIT_5

Opcodecbad::
	RES_R z80.rL, BIT_5

Opcodecbaf::
	RES_R al, BIT_5

Opcodecbae::
	RES_HL BIT_5

Opcodeddcba8::
	RES_XY BIT_5, z80.rB

Opcodeddcba9::
	RES_XY BIT_5, z80.rC

Opcodeddcbaa::
	RES_XY BIT_5, z80.rD

Opcodeddcbab::
	RES_XY BIT_5, z80.rE

Opcodeddcbac::
	RES_XY BIT_5, z80.rH

Opcodeddcbad::
	RES_XY BIT_5, z80.rL

Opcodeddcbae::
	RES_XY BIT_5

Opcodeddcbaf::
	RES_XY BIT_5, al

Opcodecbb0::
	RES_R z80.rB, BIT_6

Opcodecbb1::
	RES_R z80.rC, BIT_6

Opcodecbb2::
	RES_R z80.rD, BIT_6

Opcodecbb3::
	RES_R z80.rE, BIT_6

Opcodecbb4::
	RES_R z80.rH, BIT_6

Opcodecbb5::
	RES_R z80.rL, BIT_6

Opcodecbb7::
	RES_R al, BIT_6

Opcodecbb6::
	RES_HL BIT_6

Opcodeddcbb0::
	RES_XY BIT_6, z80.rB

Opcodeddcbb1::
	RES_XY BIT_6, z80.rC

Opcodeddcbb2::
	RES_XY BIT_6, z80.rD

Opcodeddcbb3::
	RES_XY BIT_6, z80.rE

Opcodeddcbb4::
	RES_XY BIT_6, z80.rH

Opcodeddcbb5::
	RES_XY BIT_6, z80.rL

Opcodeddcbb6::
	RES_XY BIT_6

Opcodeddcbb7::
	RES_XY BIT_6, al

Opcodecbb8::
	RES_R z80.rB, BIT_7

Opcodecbb9::
	RES_R z80.rC, BIT_7

Opcodecbba::
	RES_R z80.rD, BIT_7

Opcodecbbb::
	RES_R z80.rE, BIT_7

Opcodecbbc::
	RES_R z80.rH, BIT_7

Opcodecbbd::
	RES_R z80.rL, BIT_7

Opcodecbbf::
	RES_R al, BIT_7

Opcodecbbe::
	RES_HL BIT_7

Opcodeddcbb8::
	RES_XY BIT_7, z80.rB

Opcodeddcbb9::
	RES_XY BIT_7, z80.rC

Opcodeddcbba::
	RES_XY BIT_7, z80.rD

Opcodeddcbbb::
	RES_XY BIT_7, z80.rE

Opcodeddcbbc::
	RES_XY BIT_7, z80.rH

Opcodeddcbbd::
	RES_XY BIT_7, z80.rL

Opcodeddcbbe::
	RES_XY BIT_7

Opcodeddcbbf::
	RES_XY BIT_7, al

SET_R MACRO r,b
	or	r, b

	sub	esi, 8 + 2
	retn
	ENDM

SET_HL MACRO b
	mov	cx, z80.rHL
	call	ReadMem_dl
	or	dl, b
	mov	cx, z80.rHL
	call	WriteMem_dl

	sub	esi, 15 + 2
	retn
	ENDM

SET_XY MACRO b, r
	push	cx
	call	ReadMem_dl
	or	dl, b
	IFNB <r>
		mov	r, dl
	ENDIF
	pop	cx
	call	WriteMem_dl

	sub	esi, 23 + 2
	retn
	ENDM

Opcodecbc0::
	SET_R z80.rB, BIT_0

Opcodecbc1::
	SET_R z80.rC, BIT_0

Opcodecbc2::
	SET_R z80.rD, BIT_0

Opcodecbc3::
	SET_R z80.rE, BIT_0

Opcodecbc4::
	SET_R z80.rH, BIT_0

Opcodecbc5::
	SET_R z80.rL, BIT_0

Opcodecbc7::
	SET_R al, BIT_0

Opcodecbc6::
	SET_HL BIT_0

Opcodeddcbc0::
	SET_XY BIT_0, z80.rB

Opcodeddcbc1::
	SET_XY BIT_0, z80.rC

Opcodeddcbc2::
	SET_XY BIT_0, z80.rD

Opcodeddcbc3::
	SET_XY BIT_0, z80.rE

Opcodeddcbc4::
	SET_XY BIT_0, z80.rH

Opcodeddcbc5::
	SET_XY BIT_0, z80.rL

Opcodeddcbc6::
	SET_XY BIT_0

Opcodeddcbc7::
	SET_XY BIT_0, al

Opcodecbc8::
	SET_R z80.rB, BIT_1

Opcodecbc9::
	SET_R z80.rC, BIT_1

Opcodecbca::
	SET_R z80.rD, BIT_1

Opcodecbcb::
	SET_R z80.rE, BIT_1

Opcodecbcc::
	SET_R z80.rH, BIT_1

Opcodecbcd::
	SET_R z80.rL, BIT_1

Opcodecbcf::
	SET_R al, BIT_1

Opcodecbce::
	SET_HL BIT_1

Opcodeddcbc8::
	SET_XY BIT_1, z80.rB

Opcodeddcbc9::
	SET_XY BIT_1, z80.rC

Opcodeddcbca::
	SET_XY BIT_1, z80.rD

Opcodeddcbcb::
	SET_XY BIT_1, z80.rE

Opcodeddcbcc::
	SET_XY BIT_1, z80.rH

Opcodeddcbcd::
	SET_XY BIT_1, z80.rL

Opcodeddcbce::
	SET_XY BIT_1

Opcodeddcbcf::
	SET_XY BIT_1, al

Opcodecbd0::
	SET_R z80.rB, BIT_2

Opcodecbd1::
	SET_R z80.rC, BIT_2

Opcodecbd2::
	SET_R z80.rD, BIT_2

Opcodecbd3::
	SET_R z80.rE, BIT_2

Opcodecbd4::
	SET_R z80.rH, BIT_2

Opcodecbd5::
	SET_R z80.rL, BIT_2

Opcodecbd7::
	SET_R al, BIT_2

Opcodecbd6::
	SET_HL BIT_2

Opcodeddcbd0::
	SET_XY BIT_2, z80.rB

Opcodeddcbd1::
	SET_XY BIT_2, z80.rC

Opcodeddcbd2::
	SET_XY BIT_2, z80.rD

Opcodeddcbd3::
	SET_XY BIT_2, z80.rE

Opcodeddcbd4::
	SET_XY BIT_2, z80.rH

Opcodeddcbd5::
	SET_XY BIT_2, z80.rL

Opcodeddcbd6::
	SET_XY BIT_2

Opcodeddcbd7::
	SET_XY BIT_2, al

Opcodecbd8::
	SET_R z80.rB, BIT_3

Opcodecbd9::
	SET_R z80.rC, BIT_3

Opcodecbda::
	SET_R z80.rD, BIT_3

Opcodecbdb::
	SET_R z80.rE, BIT_3

Opcodecbdc::
	SET_R z80.rH, BIT_3

Opcodecbdd::
	SET_R z80.rL, BIT_3

Opcodecbdf::
	SET_R al, BIT_3

Opcodecbde::
	SET_HL BIT_3

Opcodeddcbd8::
	SET_XY BIT_3, z80.rB

Opcodeddcbd9::
	SET_XY BIT_3, z80.rC

Opcodeddcbda::
	SET_XY BIT_3, z80.rD

Opcodeddcbdb::
	SET_XY BIT_3, z80.rE

Opcodeddcbdc::
	SET_XY BIT_3, z80.rH

Opcodeddcbdd::
	SET_XY BIT_3, z80.rL

Opcodeddcbde::
	SET_XY BIT_3

Opcodeddcbdf::
	SET_XY BIT_3, al

Opcodecbe0::
	SET_R z80.rB, BIT_4

Opcodecbe1::
	SET_R z80.rC, BIT_4

Opcodecbe2::
	SET_R z80.rD, BIT_4

Opcodecbe3::
	SET_R z80.rE, BIT_4

Opcodecbe4::
	SET_R z80.rH, BIT_4

Opcodecbe5::
	SET_R z80.rL, BIT_4

Opcodecbe7::
	SET_R al, BIT_4

Opcodecbe6::
	SET_HL BIT_4

Opcodeddcbe0::
	SET_XY BIT_4, z80.rB

Opcodeddcbe1::
	SET_XY BIT_4, z80.rC

Opcodeddcbe2::
	SET_XY BIT_4, z80.rD

Opcodeddcbe3::
	SET_XY BIT_4, z80.rE

Opcodeddcbe4::
	SET_XY BIT_4, z80.rH

Opcodeddcbe5::
	SET_XY BIT_4, z80.rL

Opcodeddcbe6::
	SET_XY BIT_4

Opcodeddcbe7::
	SET_XY BIT_4, al
	
Opcodecbe8::
	SET_R z80.rB, BIT_5

Opcodecbe9::
	SET_R z80.rC, BIT_5

Opcodecbea::
	SET_R z80.rD, BIT_5

Opcodecbeb::
	SET_R z80.rE, BIT_5

Opcodecbec::
	SET_R z80.rH, BIT_5

Opcodecbed::
	SET_R z80.rL, BIT_5

Opcodecbef::
	SET_R al, BIT_5

Opcodecbee::
	SET_HL BIT_5

Opcodeddcbe8::
	SET_XY BIT_5, z80.rB

Opcodeddcbe9::
	SET_XY BIT_5, z80.rC

Opcodeddcbea::
	SET_XY BIT_5, z80.rD

Opcodeddcbeb::
	SET_XY BIT_5, z80.rE

Opcodeddcbec::
	SET_XY BIT_5, z80.rH

Opcodeddcbed::
	SET_XY BIT_5, z80.rL

Opcodeddcbee::
	SET_XY BIT_5

Opcodeddcbef::
	SET_XY BIT_5, al

Opcodecbf0::
	SET_R z80.rB, BIT_6

Opcodecbf1::
	SET_R z80.rC, BIT_6

Opcodecbf2::
	SET_R z80.rD, BIT_6

Opcodecbf3::
	SET_R z80.rE, BIT_6

Opcodecbf4::
	SET_R z80.rH, BIT_6

Opcodecbf5::
	SET_R z80.rL, BIT_6

Opcodecbf7::
	SET_R al, BIT_6

Opcodecbf6::
	SET_HL BIT_6

Opcodeddcbf0::
	SET_XY BIT_6, z80.rB

Opcodeddcbf1::
	SET_XY BIT_6, z80.rC

Opcodeddcbf2::
	SET_XY BIT_6, z80.rD

Opcodeddcbf3::
	SET_XY BIT_6, z80.rE

Opcodeddcbf4::
	SET_XY BIT_6, z80.rH

Opcodeddcbf5::
	SET_XY BIT_6, z80.rL

Opcodeddcbf6::
	SET_XY BIT_6

Opcodeddcbf7::
	SET_XY BIT_6, al

Opcodecbf8::
	SET_R z80.rB, BIT_7

Opcodecbf9::
	SET_R z80.rC, BIT_7

Opcodecbfa::
	SET_R z80.rD, BIT_7

Opcodecbfb::
	SET_R z80.rE, BIT_7

Opcodecbfc::
	SET_R z80.rH, BIT_7

Opcodecbfd::
	SET_R z80.rL, BIT_7

Opcodecbff::
	SET_R al, BIT_7

Opcodecbfe::
	SET_HL BIT_7

Opcodeddcbf8::
	SET_XY BIT_7, z80.rB

Opcodeddcbf9::
	SET_XY BIT_7, z80.rC

Opcodeddcbfa::
	SET_XY BIT_7, z80.rD

Opcodeddcbfb::
	SET_XY BIT_7, z80.rE

Opcodeddcbfc::
	SET_XY BIT_7, z80.rH

Opcodeddcbfd::
	SET_XY BIT_7, z80.rL

Opcodeddcbfe::
	SET_XY BIT_7

Opcodeddcbff::
	SET_XY BIT_7, al

; ****************************************************************************
; jump group

Opcodec3::
	call	ReadPC_dl
	call	ReadPC_dh
	mov	di, dx

	sub	esi, 10 + 1
	retn

Opcodeda::	; jp c,nn
	test	ah, FLAG_C
	jnz	Opcodec3
	add	edi, 2

	sub	esi, 10 + 1
	retn

Opcoded2::	; jp nc,nn
	test	ah, FLAG_C
	jz	Opcodec3
	add	edi, 2

	sub	esi, 10 + 1
	retn

Opcodeca::	; jp z,nn
	test	ah, FLAG_Z
	jnz	Opcodec3
	add	edi, 2

	sub	esi, 10 + 1
	retn

Opcodec2::	; jp nz,nn
	test	ah, FLAG_Z
	jz	Opcodec3
	add	edi, 2

	sub	esi, 10 + 1
	retn

Opcodeea::	; jp pe,nn
	test	ah, FLAG_P
	jnz	Opcodec3
	add	edi, 2

	sub	esi, 10 + 1
	retn

Opcodee2::	; jp po,nn
	test	ah, FLAG_P
	jz	Opcodec3
	add	edi, 2

	sub	esi, 10 + 1
	retn

Opcodefa::	; jp m,nn
	test	ah, FLAG_S
	jnz	Opcodec3
	add	edi, 2

	sub	esi, 10 + 1
	retn

Opcodef2::	; jp p,nn
	test	ah, FLAG_S
	jz	Opcodec3
	add	edi, 2

	sub	esi, 10 + 1
	retn

Opcode18::	; jr n
	call	ReadPC_dl
	movsx	ebx, dl
	add	edi, ebx
	cmp	dl, -2
	je	EndProcess
	sub	esi, 12 + 1
	retn

EndProcess:
	push	eax
	mov	eax, esi
	mov	ecx, 12 + 1
	xor	edx, edx
	div	ecx
	add	ebp, eax
	pop	eax
	mov	dl, z80.fIFF
	mov	dwReturn, edx
	cmp	dh, cl ; CF=1, ZF=0
	retn

Opcode38::	; jr c,n
	test	ah, FLAG_C
	jnz	Opcode18
	inc	edi
	sub	esi, 7 + 1
	retn

Opcode30::	; jr nc,n
	test	ah, FLAG_C
	jz	Opcode18
	inc	edi
	sub	esi, 7 + 1
	retn

Opcode28::	; jr z,n
	test	ah, FLAG_Z
	jnz	Opcode18
	inc	edi
	sub	esi, 7 + 1
	retn

Opcode20::	; jr nz,n
	test	ah, FLAG_Z
	jz	Opcode18
	inc	edi
	sub	esi, 7 + 1
	retn

Opcodee9::	; jp hl
	mov	di, z80.rHL
	sub	esi, 4 + 1
	retn

Opcodedde9::	; jp ix
	mov	di, z80.rIX
	sub	esi, 8 + 2
	retn

Opcodefde9::	; jp iy
	mov	di, z80.rIY
	sub	esi, 8 + 2
	retn

Opcode10::	; djnz n
	dec	z80.rB
	je	short NoJr
	call	ReadPC_dl
	movsx	ebx, dl
	add	edi, ebx
	sub	esi, 13 + 1
	retn

NoJr:
	inc	edi
	sub	esi, 8 + 1
	retn

; ****************************************************************************
; call and return group


Opcodecd::
	call	ReadPC_dl
	call	ReadPC_dh
	xchg	di, dx
	DoPush

	sub	esi, 17 + 1
	retn

Opcodec4::
	test	ah, FLAG_Z
	jz	Opcodecd
	add	edi, 2

	sub	esi, 10 + 1
	retn

Opcodecc::
	test	ah, FLAG_Z
	jnz	Opcodecd
	add	edi, 2

	sub	esi, 10 + 1
	retn

Opcoded4::
	test	ah, FLAG_C
	jz	Opcodecd
	add	edi, 2

	sub	esi, 10 + 1
	retn

Opcodedc::
	test	ah, FLAG_C
	jnz	Opcodecd
	add	edi, 2

	sub	esi, 10 + 1
	retn

Opcodee4::
	test	ah, FLAG_P
	jz	Opcodecd
	add	edi, 2

	sub	esi, 10 + 1
	retn

Opcodeec::
	test	ah, FLAG_P
	jnz	Opcodecd
	add	edi, 2

	sub	esi, 10 + 1
	retn

Opcodef4::
	test	ah, FLAG_S
	jz	Opcodecd
	add	edi, 2

	sub	esi, 10 + 1
	retn

Opcodefc::
	test	ah, FLAG_S
	jnz	Opcodecd
	add	edi, 2

	sub	esi, 10 + 1
	retn

Opcodec9::
	DoPop
	mov	di, dx

	sub	esi, 10 + 1
	retn

Opcodec0::
	test	ah, FLAG_Z
	jz	Opcodec9

	sub	esi, 5 + 1
	retn

Opcodec8::
	test	ah, FLAG_Z
	jnz	Opcodec9

	sub	esi, 5 + 1
	retn

Opcoded0::
	test	ah, FLAG_C
	jz	Opcodec9

	sub	esi, 5 + 1
	retn

Opcoded8::
	test	ah, FLAG_C
	jnz	Opcodec9

	sub	esi, 5 + 1
	retn

Opcodee0::
	test	ah, FLAG_P
	jz	Opcodec9

	sub	esi, 5 + 1
	retn

Opcodee8::
	test	ah, FLAG_P
	jnz	Opcodec9

	sub	esi, 5 + 1
	retn

Opcodef0::
	test	ah, FLAG_S
	jz	Opcodec9

	sub	esi, 5 + 1
	retn

Opcodef8::
	test	ah, FLAG_S
	jnz	Opcodec9

	sub	esi, 5 + 1
	retn

Opcodeed4d::	; reti
Opcodeed45::	; retn
Opcodeed5d::	; retn*
Opcodeed55::	; retn*
Opcodeed6d::	; retn*
Opcodeed65::	; retn*
Opcodeed7d::	; retn*
Opcodeed75::	; retn*
	DoPop
	mov	di, dx
	sub	esi, 14 + 2
	retn

RST	MACRO p
	mov	dx, di
	DoPush
	mov	edi, p

	sub	esi, 11 + 1
	retn
	ENDM

Opcodec7::
	RST 0h

Opcodecf::
	RST 8h

Opcoded7::
	RST 10h

Opcodedf::
	RST 18h

Opcodee7::
	RST 20h

Opcodeef::
	RST 28h

Opcodef7::
	RST 30h

Opcodeff::
	RST 38h

; ****************************************************************************
; I/O

Opcoded3::	; OUT (n),A
	call	ReadPC_dl
	mov	dh, al
	call	WriteIO

	sub	esi, 11 + 1
	retn

Opcodedb::	; IN A,(n)
	call	ReadPC_dl
	call	ReadIO
	mov	al, dh

	sub	esi, 11 + 1
	retn

IN_R	MACRO r
	mov	dl, z80.rC
	call	ReadIO
	mov	dl, ah
	and	dl, FLAG_C
	test	dh, dh
	lahf
	and	ah, FLAG_S OR FLAG_Z OR FLAG_P
	or	ah, dl
	mov	rFb35, dh
	IFNB	<r>
		mov	r, dh
	ENDIF

	sub	esi, 12 + 2
	retn
	ENDM

Opcodeed40::	; IN B,(C)
	IN_R z80.rB

Opcodeed48::	; IN C,(C)
	IN_R z80.rC

Opcodeed50::	; IN D,(C)
	IN_R z80.rD

Opcodeed58::	; IN E,(C)
	IN_R z80.rE

Opcodeed60::	; IN H,(C)
	IN_R z80.rH

Opcodeed68::	; IN L,(C)
	IN_R z80.rL

Opcodeed70::	; IN F,(C) / IN (C)
	IN_R 

Opcodeed78::	; IN A,(C)
	IN_R al

OUT_R	MACRO r
	IFB <r>
		mov	dl, z80.rC
		sub	dh, dh
	ELSEIFIDNI <r>, <z80.rB>
		mov	dx, z80.rBC
	ELSEIFIDNI <r>, <z80.rC>
		mov	dl, z80.rC
		mov	dh, dl
	ELSE
		mov	dl, z80.rC
		mov	dh, r
	ENDIF
	call	WriteIO

	sub	esi, 12 + 2
	retn
	ENDM

Opcodeed41::	; OUT (C),B
	OUT_R z80.rB

Opcodeed49::	; OUT (C),C
	OUT_R z80.rC

Opcodeed51::	; OUT (C),D
	OUT_R z80.rD

Opcodeed59::	; OUT (C),E
	OUT_R z80.rE

Opcodeed61::	; OUT (C),H
	OUT_R z80.rH

Opcodeed69::	; OUT (C),L
	OUT_R z80.rL

Opcodeed71::	; OUT (C),0
	OUT_R

Opcodeed79::	; OUT (C),A
	OUT_R al

Opcodeeda2::	; INI
	mov	dl, z80.rC
	call	ReadIO
	mov	cx, z80.rHL
	call	WriteMem_dh
	mov	cx, z80.rBC
	dec	ch
	mov	z80.rB, ch
	mov	rFb35, ch
	lahf
	and	ah, FLAG_S OR FLAG_Z OR FLAG_P
	inc	cl
	add	cl, dh
	jnc	ini_noc
	or	ah, FLAG_H OR FLAG_C
ini_noc:
	inc	z80.rHL
	shr	dh, 6
	and	dh, FLAG_N
	or	ah, dh

	sub	esi, 16 + 2
	retn

Opcodeedaa::	; IND
	mov	dl, z80.rC
	call	ReadIO
	mov	cx, z80.rHL
	call	WriteMem_dh
	mov	cx, z80.rBC
	dec	ch
	mov	z80.rB, ch
	mov	rFb35, ch
	lahf
	and	ah, FLAG_S OR FLAG_Z OR FLAG_P
	dec	cl
	add	cl, dh
	jnc	ind_noc
	or	ah, FLAG_H OR FLAG_C
ind_noc:
	dec	z80.rHL
	shr	dh, 6
	and	dh, FLAG_N
	or	ah, dh

	sub	esi, 16 + 2
	retn

inir_loop:
	add	ebp, 2
Opcodeedb2::	; INIR
	mov	dl, z80.rC
	call	ReadIO
	mov	cx, z80.rHL
	call	WriteMem_dh
	inc	z80.rHL
	dec	z80.rB
	jz	inir_zero
	sub	esi, 21 + 2
	jnc	inir_loop
	sub	edi, 2
	mov	ah, z80.rB
	mov	rFb35, ah
	and	ah, FLAG_S
	mov	dl, z80.rC
	inc	dl
	add	dl, dh
	jnc	inir_noc
	or	ah, FLAG_H OR FLAG_C
inir_noc:
	shr	dh, 6
	and	dh, FLAG_N
	or	ah, dh
	add	esi, 255 ; CF=1, ZF=0
	retn

inir_zero:
	mov	ah, FLAG_Z OR FLAG_P
	mov	rFb35, ah
	mov	dl, z80.rC
	inc	dl
	add	dl, dh
	jnc	inir_noc2
	or	ah, FLAG_H OR FLAG_C
inir_noc2:
	shr	dh, 6
	and	dh, FLAG_N
	or	ah, dh
	sub	esi, 16 + 2
	retn

indr_loop:
	add	ebp, 2
Opcodeedba::	; INDR
	mov	dl, z80.rC
	call	ReadIO
	mov	cx, z80.rHL
	call	WriteMem_dh
	dec	z80.rHL
	dec	z80.rB
	jz	indr_zero
	sub	esi, 21 + 2
	jnc	indr_loop
	sub	edi, 2
	mov	ah, z80.rB
	mov	rFb35, ah
	and	ah, FLAG_S
	mov	dl, z80.rC
	inc	dl
	add	dl, dh
	jnc	indr_noc
	or	ah, FLAG_H OR FLAG_C
indr_noc:
	shr	dh, 6
	and	dh, FLAG_N
	or	ah, dh
	add	esi, 255 ; CF=1, ZF=0
	retn

indr_zero:
	mov	ah, FLAG_Z OR FLAG_P
	mov	rFb35, ah
	mov	dl, z80.rC
	dec	dl
	add	dl, dh
	jnc	indr_noc2
	or	ah, FLAG_H OR FLAG_C
indr_noc2:
	shr	dh, 6
	and	dh, FLAG_N
	or	ah, dh
	sub	esi, 16 + 2
	retn

Opcodeeda3::	; OUTI
	mov	dx, z80.rBC
	dec	dh
	mov	z80.rB, dh
	mov	rFb35, dh
	lahf
	mov	cx, z80.rHL
	and	ah, FLAG_S OR FLAG_Z OR FLAG_P
	lea	ebx, [ecx+1]
	mov	z80.rHL, bx
	call	ReadMem_dh
	mov	bx, dx
	inc	bl
	add	bl, bh
	jnc	outi_noc
	or	ah, FLAG_H OR FLAG_C
outi_noc:
	shr	bh, 6
	and	bh, FLAG_N
	or	ah, bh
	call	WriteIO

	sub	esi, 16
	retn

Opcodeedab::	; OUTD
	mov	dx, z80.rBC
	dec	dh
	mov	z80.rB, dh
	mov	rFb35, dh
	lahf
	mov	cx, z80.rHL
	and	ah, FLAG_S OR FLAG_Z OR FLAG_P
	lea	ebx, [ecx-1]
	mov	z80.rHL, bx
	call	ReadMem_dh
	mov	bx, dx
	inc	bl
	add	bl, bh
	jnc	outd_noc
	or	ah, FLAG_H OR FLAG_C
outd_noc:
	shr	bh, 6
	and	bh, FLAG_N
	or	ah, bh
	call	WriteIO

	sub	esi, 16
	retn

otir_loop:
	add	ebp, 2
Opcodeedb3::	; OTIR
	mov	dl, z80.rC
	mov	cx, z80.rHL
	call	ReadMem_dh
	push	edx
	call	WriteIO
	pop	edx
	inc	z80.rHL
	dec	z80.rB
	jz	otir_zero
	sub	esi, 21 + 2
	jnc	otir_loop
	mov	ah, z80.rB
	mov	rFb35, ah
	and	ah, FLAG_S
	inc	dl
	add	dl, dh
	jnc	otir_noc
	or	ah, FLAG_H OR FLAG_C
otir_noc:
	shr	dh, 6
	and	dh, FLAG_N
	or	ah, dh
	add	esi, 255 ; CF=1, ZF=0
	retn

otir_zero:
	mov	ah, FLAG_Z OR FLAG_P
	mov	rFb35, ah
	inc	dl
	add	dl, dh
	jnc	otir_noc2
	or	ah, FLAG_H OR FLAG_C
otir_noc2:
	shr	dh, 6
	and	dh, FLAG_N
	or	ah, dh
	sub	esi, 16 + 2
	retn
	
otdr_loop:
	add	ebp, 2
Opcodeedbb::	; OTDR
	mov	cx, z80.rHL
	call	ReadMem_dh
	mov	dl, z80.rC
	push	dx
	call	WriteIO
	pop	dx
	dec	z80.rHL
	dec	z80.rB
	jz	otdr_zero
	sub	esi, 21 + 2
	jnc	otdr_loop
	mov	ah, z80.rB
	mov	rFb35, ah
	and	ah, FLAG_S
	dec	dl
	add	dl, dh
	jnc	otdr_noc
	or	ah, FLAG_H OR FLAG_C
otdr_noc:
	shr	dh, 6
	and	dh, FLAG_N
	or	ah, dh
	add	esi, 255 ; CF=1, ZF=0
	retn

otdr_zero:
	mov	ah, FLAG_Z OR FLAG_P
	mov	rFb35, ah
	dec	dl
	add	dl, dh
	jnc	otdr_noc2
	or	ah, FLAG_H OR FLAG_C
otdr_noc2:
	shr	dh, 6
	and	dh, FLAG_N
	or	ah, dh
	sub	esi, 16 + 2
	retn

; ****************************************************************************
; misc

Opcode27::	; DAA
	test	ah, FLAG_H
	jnz	daa_hset
	and	eax, 03ffh
	sub	esi, 4 + 1
	mov	ax, DAATbl[eax*2]
	mov	rFb35, ah
	retn

daa_hset:
	and	eax, 03ffh
	sub	esi, 4 + 1
	mov	ax, DAATbl[eax*2+4*512]
	mov	rFb35, ah
	retn

Opcode2f::	; CPL
	not	al
	or	ah, FLAG_H OR FLAG_N
	mov	rFb35, al

	sub	esi, 4 + 1
	retn

Opcodeed44::	;  NEG
Opcodeed4c::	;  NEG*
Opcodeed54::	;  NEG*
Opcodeed5c::	;  NEG*
Opcodeed64::	;  NEG*
Opcodeed6c::	;  NEG*
Opcodeed74::	;  NEG*
Opcodeed7c::	;  NEG*
	neg	al
	lahf
	mov	rFb35, al
	jo	neg_overflow
	and	ah, NOT FLAG_V

	sub	esi, 8 + 2
	retn

neg_overflow:
	or	ah, FLAG_V

	sub	esi, 8 + 2
	retn

Opcode3f::
	test	ah, FLAG_C
	jz	Opcode37
	and	ah, FLAG_S OR FLAG_Z OR FLAG_P
	or	ah, FLAG_H
	mov	rFb35, al

	sub	esi, 4 + 1
	retn

Opcode37::	; SCF
	and	ah, FLAG_S OR FLAG_Z OR FLAG_P
	or	ah, FLAG_C
	mov	rFb35, al
Opcode00::	; NOP
	sub	esi, 4 + 1
	retn

Opcode76:: ; halt
	push	eax
	mov	eax, esi
	mov	ecx, 4 + 1
	xor	edx, edx
	div	ecx
	add	ebp, eax
	pop	eax
	mov	dl, z80.fIFF
	mov	dwReturn, edx
	cmp	dh, cl ; CF=1, ZF=0
	retn

Opcodefb::	; EI
	mov	ebx, 8000h + IFF_SET
	mov	z80.fIFF, bl
	mov	z80.fNoINT, bh
	sub	esi, 4 + 1
	adc	bh, bh
	; if sub esi -> CF=1, then ZF=0, CF=1
	; if sub esi -> CF=0, then ZF=1, CF=1
	retn

Opcodef3::
	mov	z80.fIFF, IFF_RESET

	sub	esi, 4 + 1
	retn

Opcodeed46::	; IM 0
Opcodeed66::	; IM 0*
	mov	z80.wIM, 0

	sub	esi, 8 + 2
	retn

Opcodeed56::	; IM 1
Opcodeed76::	; IM 1*
Opcodeed4e::	; IM 0/1*
Opcodeed6e::	; IM 0/1*
	mov	z80.wIM, 1

	sub	esi, 8 + 2
	retn

Opcodeed5e::	; IM 2
Opcodeed7e::	; IM 2*
	mov	z80.wIM, 2

	sub	esi, 8 + 2
	retn

DoZ80 ENDP


DoZ80Step PROC C
	push	ebx
	push	esi
	push	edi
	push	ebp

	mov	dwReturn, 1

	LoadRegs

	call	Interrupt
	jc	dzs_end
	test	edx, edx
	jnz	dzs_notend

;	xor	edx, edx
	call	ReadPC_dl
	inc	ebp
	call	JumpTbl[edx*4]

	jnc	dzs_notend
dzs_end:
	xor	esi, esi
dzs_notend:
	SaveRegs

	mov	eax, dwReturn
	pop	ebp
	pop	edi
	pop	esi
	pop	ebx
	ret
DoZ80Step ENDP

END

