// bootmsx.c
#include <windows.h>
#include <memory.h>
#include <assert.h>
#include "error.h"
#include "resource.h"
#include "mem.h"
#include "config.h"
#include "bootmsx.h"

MSXMEM			msxmem ;
extern HWND		g_hwndMain ;

///////////////////////////////////////////////////////////////////////////////
// ROM/BIN file loading...

static BOOL IsHexNumber (char ch)
	{
	return	(ch >= '0' && ch <= '9') || 
			(ch >= 'a' && ch <= 'f') || 
			(ch >= 'A' && ch <= 'F') ;
	}

BOOL LoadROMfile (HWND hwnd, char *szFile, BYTE **ppDestination, int *piSize, WORD type)
	{
	HANDLE	hFile ;
	BYTE	*p ;
	int		i, iSize, iSizeBlock, iCount, iLen ;
	char	szCountFileFormat[_MAX_PATH], szCountFile[_MAX_PATH] ;
	HCURSOR	hcurOld ;
	DWORD	dwBytesRead ;
	WORD	wHead ;

	hcurOld = SetCursor (LoadCursor (NULL, IDC_WAIT) ) ;
	ShowCursor (TRUE) ;
	
	p = NULL ;
	hFile = CreateFile (szFile, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, 
		FILE_ATTRIBUTE_NORMAL | FILE_FLAG_SEQUENTIAL_SCAN, 0) ;

	if (hFile == INVALID_HANDLE_VALUE)
		goto error ;

	if (!ReadFile (hFile, &wHead, 2, &dwBytesRead, NULL) || dwBytesRead != 2)
		goto error ;
	
	iSize = GetFileSize (hFile, NULL) ;
	
	if ((wHead & 0xff) == 0xfe)
		{
		if (iSize >= 0x4007)
			iSizeBlock = 0x4000 ;
		else if (iSize >= 0x2007)
			iSizeBlock = 0x2000 ;
		else
			{
			CloseHandle (hFile) ;

			ShowCursor (FALSE) ;
			SetCursor (hcurOld) ;

			ErrorFileID (hwnd, szFile, IDS_FILEERRBIN, MB_ICONEXCLAMATION) ;

			return FALSE ;
			}

		CloseHandle (hFile) ;

		iLen = strlen (szFile) ;
		if (IsHexNumber (szFile[iLen - 1]) )
			{
			if (iLen > 2 && (szFile[iLen - 2] >= '0' && szFile[iLen - 2] <= '3') )
				iCount = 2 ;
			else
				iCount = 1 ;
			}
		else
			iCount = 0 ;
		
		while (iCount)
			{
			strcpy (szCountFile, szFile) ;
			strcpy (szCountFile + iLen - iCount, "%%0%dx") ;
			wsprintf (szCountFileFormat, szCountFile, iCount) ;

			for (i=0;i<64;i++)
				{
				wsprintf (szCountFile, szCountFileFormat, i) ;

				hFile = CreateFile (szCountFile, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, 
					FILE_ATTRIBUTE_NORMAL | FILE_FLAG_SEQUENTIAL_SCAN, 0) ;

				if (hFile == INVALID_HANDLE_VALUE)
					{
					if (GetLastError () == ERROR_FILE_NOT_FOUND)
						{
						if (!i)
							iCount-- ;
						
						break ;
						}
					
					MemoryFree (&p) ;

					ShowCursor (FALSE) ;
					SetCursor (hcurOld) ;

					ErrorFile (hwnd, szCountFile) ;
				
					return FALSE ;
					}
				
				if (!ReadFile (hFile, &wHead, 2, &dwBytesRead, NULL) || dwBytesRead != 2)
					{
					MemoryFree (&p) ;

					CloseHandle (hFile) ;

					ShowCursor (FALSE) ;
					SetCursor (hcurOld) ;

					ErrorFile (hwnd, szCountFile) ;
				
					return FALSE ;
					}
				
				SetFilePointer (hFile, 7, NULL, FILE_BEGIN) ;
				if (!MemoryReSize (&p, (i + 1) * iSizeBlock) )
					{
					MemoryFree (&p) ;

					CloseHandle (hFile) ;

					ShowCursor (FALSE) ;
					SetCursor (hcurOld) ;

					return FALSE ;
					}
				
				if (!ReadFile (hFile, p + i * iSizeBlock, iSizeBlock, &dwBytesRead, NULL) || (int)dwBytesRead != iSizeBlock)
					{
					MemoryFree (&p) ;

					CloseHandle (hFile) ;

					ShowCursor (FALSE) ;
					SetCursor (hcurOld) ;

					ErrorFile (hwnd, szCountFile) ;
				
					return FALSE ;
					}
				
				if ((wHead & 0xff) != 0xfe)
					{
					MemoryFree (&p) ;

					CloseHandle (hFile) ;

					ShowCursor (FALSE) ;
					SetCursor (hcurOld) ;

					ErrorFileID (hwnd, szCountFile, IDS_FILEERRBIN, MB_ICONEXCLAMATION) ;

					return FALSE ;
					}
				
				CloseHandle (hFile) ;
				}
			
			if (i)
				{
				*piSize = i * iSizeBlock ;
				*ppDestination = p ;
				
				ShowCursor (FALSE) ;
				SetCursor (hcurOld) ;

				return TRUE ;
				}
			}
		
		// just load the user file name; one block
		hFile = CreateFile (szFile, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, 
			FILE_ATTRIBUTE_NORMAL | FILE_FLAG_SEQUENTIAL_SCAN, 0) ;
		if (hFile == INVALID_HANDLE_VALUE)
			goto error ;
		
		SetFilePointer (hFile, 7, NULL, FILE_BEGIN) ;
		if (!MemoryReSize (&p, iSizeBlock) )
			{
			MemoryFree (&p) ;

			if (hFile != INVALID_HANDLE_VALUE)
				CloseHandle (hFile) ;

			ShowCursor (FALSE) ;
			SetCursor (hcurOld) ;

			return FALSE ;
			}

		if (!ReadFile (hFile, p, iSizeBlock, &dwBytesRead, NULL) || (int)dwBytesRead != iSizeBlock)
			goto error ;
		
		if (*((WORD*)p) != type)
			{
			MemoryFree (&p) ;

			CloseHandle (hFile) ;

			ShowCursor (FALSE) ;
			SetCursor (hcurOld) ;

			ErrorFileID (hwnd, szFile, IDS_FILEERRBIN, MB_ICONEXCLAMATION) ;

			return FALSE ;
			}
		
		CloseHandle (hFile) ;

		*piSize = iSizeBlock ;
		*ppDestination = p ;

		ShowCursor (FALSE) ;
		SetCursor (hcurOld) ;

		return TRUE ;
		}
	else if (wHead == type)
		{
		if (!MemoryNew (&p, iSize) )
			{
			MemoryFree (&p) ;

			if (hFile != INVALID_HANDLE_VALUE)
				CloseHandle (hFile) ;

			ShowCursor (FALSE) ;
			SetCursor (hcurOld) ;

			return FALSE ;
			}
		
		SetFilePointer (hFile, 0, NULL, FILE_BEGIN) ;
		if (!ReadFile (hFile, p, iSize, &dwBytesRead, NULL) || (int)dwBytesRead != iSize)
			goto error ;
		
		CloseHandle (hFile) ;

		*piSize = iSize ;
		*ppDestination = p ;
		
		ShowCursor (FALSE) ;
		SetCursor (hcurOld) ;

		return TRUE ;
		}
	else
		{
		CloseHandle (hFile) ;

		ShowCursor (FALSE) ;
		SetCursor (hcurOld) ;

		ErrorFileID (hwnd, szFile, IDS_FILEERRROMBIN, MB_ICONEXCLAMATION) ;

		return FALSE ;
		}

error:
	MemoryFree (&p) ;

	if (hFile != INVALID_HANDLE_VALUE)
		CloseHandle (hFile) ;

	ShowCursor (FALSE) ;
	SetCursor (hcurOld) ;

	ErrorFile (hwnd, szFile) ;

	return FALSE ;
	}

enum BiosType /* oh no! */ { MSX, MSX2, MSX2P, MSXTR, MSX_JP, MSX2_JP, MSX2P_JP, MSXTR_JP, 
	MSX2EXT, MSX2PEXT, MSXTREXT, MSX2EXT_JP, MSX2PEXT_JP, MSXTREXT_JP, DISK, MSXDOS2} ;

static BOOL LoadBIOS (BYTE **pdes, enum BiosType bios)
	{
	static const char	*szBios[] = { 
		"MSX.ROM", "MSX2.ROM", "MSX2P.ROM", "MSXTR.ROM", 
		"MSX.ROM.JP", "MSX2.ROM.JP", "MSX2P.ROM.JP", "MSXTR.ROM.JP", 
		"MSX2EXT.ROM", "MSX2PEXT.ROM", "MSXTREXT.ROM", 
		"MSX2EXT.ROM.JP", "MSX2PEXT.ROM.JP", "MSXTREXT.ROM.JP", 
		"DISK.ROM", "MSXDOS2.ROM" } ;
	char		szBuf[_MAX_PATH] ;
	BYTE		*p ;
	int			n, i ;
	WORD		wHead ;

	GetModuleFileName (NULL, szBuf, sizeof (szBuf) ) ;
	strcpy (strrchr (szBuf, '\\'), "\\MSX Bin\\") ;
	strcat (szBuf, szBios[bios]) ;

	if (bios <= MSXTR_JP)
		wHead = ROM_BIOS ;
	else if (bios <= MSXTREXT_JP)
		wHead = ROM_SUBBIOS ;
	else
		wHead = ROM_CART ;

	if (!LoadROMfile (g_hwndMain, szBuf, &p, &i, wHead) )
		return FALSE ;
	
	if (bios <= MSXTR_JP)
		n =  0x8000 ;
	else if (bios <= DISK)
		n =  0x4000 ;
	else
		n = 0x10000 ;

	if (i < n)
		{
		ErrorFileID (g_hwndMain, szBuf, IDS_FILEINCORRECT, MB_ICONEXCLAMATION) ;
		return FALSE ;
		}
	
	*pdes = p ;

	return TRUE ;
	}

BOOL LoadSRAM (BYTE *pb, char *szFile)
	{
	HANDLE		hFile ;
	int			i ;
	DWORD		dwBytesRead ;

	if (!szFile[0])
		return TRUE ;

	hFile = CreateFile (szFile, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, 
		FILE_ATTRIBUTE_NORMAL | FILE_FLAG_SEQUENTIAL_SCAN, 0) ;

	if (hFile == INVALID_HANDLE_VALUE)
		goto error ;

	i = GetFileSize (hFile, NULL) ;
	if (i < 0x2000)
		{
		ErrorFileID (g_hwndMain, szFile, IDS_FILEERRSRAM, MB_ICONEXCLAMATION) ;

		CloseHandle (hFile) ;

		return FALSE ;
		}
	else if (i < 0x4000)
		{
		if (!ReadFile (hFile, pb, 0x2000, &dwBytesRead, NULL) || dwBytesRead != 0x2000)
			goto error ;
		}
	else
		{
		SetFilePointer (hFile, 0x1000, NULL, FILE_BEGIN) ;
		if (!ReadFile (hFile, pb, 0x1000, &dwBytesRead, NULL) || dwBytesRead != 0x1000)
			goto error ;
		SetFilePointer (hFile, 0x3000, NULL, FILE_BEGIN) ;
		if (!ReadFile (hFile, pb + 0x1000, 0x1000, &dwBytesRead, NULL) || dwBytesRead != 0x1000)
			goto error ;
		}
	
	CloseHandle (hFile) ;
	
	return TRUE ;

error:
	if (hFile != INVALID_HANDLE_VALUE)
		CloseHandle (hFile) ;

	ErrorFile (g_hwndMain, szFile) ;

	return FALSE ;
	}

BOOL LoadCartridge (CONFIG *pcf, int iCart)
	{
	int		ps, ss, p, i, n, iSize,type ;
	BOOL	fFound ;
	BYTE	*pb ;
	
	switch (iCart)
		{
		case 0:
			i = CONFIG_MEMORY_CARTA ;
			type = MAP_CARTA ;
			break ;
		case 1:
			i = CONFIG_MEMORY_CARTB ;
			type = MAP_CARTB ;
			break ;
		case 2:
			i = CONFIG_MEMORY_CARTC ;
			type = MAP_CARTC ;
			break ;
		case 3:
			i = CONFIG_MEMORY_CARTD ;
			type = MAP_CARTD ;
			break ;
		default: assert (0);
		}

	fFound = FALSE ;
	for (ps=0;ps<4;ps++) for (ss=0;ss<4;ss++) for (p=0;p<4;p++)
		{
		if (pcf->mem.iMem[ps][ss][p] == i)
			{
			fFound = TRUE ;
			goto doend ;
			}
		}

doend:
	if (!fFound) return TRUE ; /* do nothing was completed succesfully */

	if (pcf->cart[iCart].iType < CONFIG_CARTRIDGE_THSCCP)
		{
		if (pcf->cart[iCart].szROMfile[0])
			{
			if (!LoadROMfile (g_hwndMain, pcf->cart[iCart].szROMfile, &pb, &iSize, ROM_CART) )
				return FALSE ;

			// round it to upper 2^n
			n = 0x4000 ;
			while (n < iSize) n *= 2 ;

			if (n != iSize)
				{
				if (!MemoryReSize (&pb, n) )
					{
					MemoryFree (&pb) ;
					return FALSE ;
					}

				memset (pb + iSize, 0xff, n - iSize) ;
				iSize = n ;
				}
			}
		else
			iSize = 0 ;
		}
	else
		iSize = 0 ;

	switch (pcf->cart[iCart].iType)
		{
		case CONFIG_CARTRIDGE_STANDARD:
			if (!iSize) break ;
			
			p = 1 ;
			for (i=1;i<5;i++)
				{
				n = ((PWORD)pb)[i] ;
				if (n)
					{
					p = n / 0x4000 ;
					break ;
					}
				}

			if (iSize == 0x4000)
				{
				if (p == 0 || p == 3)
					{
					for (p=0;p<4;p++)
						msxmem.pMap[ps][ss][p] = pb ;
					}
				else
					{
					msxmem.pMap[ps][ss][p] = pb ; 
					}
				}
			else if (iSize == 0x8000)
				{
				if (2 == p) p = 1 ;
				
				msxmem.pMap[ps][ss][p] = pb ;
				msxmem.pMap[ps][ss][(p + 1) & 3] = pb + 0x4000 ;
				}
			else
				{
				msxmem.pMap[ps][ss][p] = pb ;
				msxmem.pMap[ps][ss][(p + 1) & 3] = pb + 0x4000 ;
				msxmem.pMap[ps][ss][(p + 2) & 3] = pb + 0x8000 ;
				msxmem.pMap[ps][ss][(p + 3) & 3] = pb + 0xc000 ;
				}

			break ;

		case CONFIG_CARTRIDGE_KONAMI4:
		case CONFIG_CARTRIDGE_ASCII_8KB:
		case CONFIG_CARTRIDGE_ASCII_16KB:
		case CONFIG_CARTRIDGE_R_TYPE:
			if (!iSize) break ;

			for (p=1;p<=2;p++)
				msxmem.bMapType[ps][ss][p] = type ;

			msxmem.bLastCartPage[iCart] = (iSize / 0x2000) - 1 ;
			msxmem.bCartType[iCart] = pcf->cart[iCart].iType ;

			break ;

		case CONFIG_CARTRIDGE_ASCII_8KB_SRAM:
			if (!iSize)
				{
				if (!MemoryNew (&pb, 0x4000) )
					return FALSE ;

				memset (pb, 0xff, 0x2000) ;
				iSize = 0x2000 ;
				}
			else if (!MemoryReSize (&pb, iSize + 0x2000) )
				{
				MemoryFree (&pb) ;

				return FALSE ;
				}

			LoadSRAM (pb + iSize, pcf->cart[iCart].szSRAMfile) ;
			
			for (p=1;p<=2;p++)
				msxmem.bMapType[ps][ss][p] = type ;

			msxmem.bLastCartPage[iCart] = (iSize / 0x2000) - 1 ;
			msxmem.bCartType[iCart] = pcf->cart[iCart].iType ;

			break ;
		
		case CONFIG_CARTRIDGE_MSXDOS2:
			if (!iSize) break ;

			msxmem.bMapType[ps][ss][1] = type ;

			msxmem.bLastCartPage[iCart] = (iSize / 0x2000) - 1 ;
			msxmem.bCartType[iCart] = pcf->cart[iCart].iType ;

			break ;

		case CONFIG_CARTRIDGE_KONAMI5:
			for (p=1;p<=2;p++)
				msxmem.bMapType[ps][ss][p] = type ;

			if (!iSize)
				{
				iSize = 0x2000 ;
				if (!MemoryNew (&pb, 0x4000) )
					return FALSE ;

				memset (pb, 0xff, 0x4000) ;
				}
			else if (!MemoryReSize (&pb, iSize + 0x2000) )
				{
				MemoryFree (&pb) ;

				return FALSE ;
				}

			memcpy (pb + iSize, pb + iSize - 0x2000, 0x1800) ;
			for (i=0;i<8;i++)
				{
				memset (pb + iSize + 0x1800 + i * 0x100, 0x00, 0x80) ;
				memset (pb + iSize + 0x1880 + i * 0x100, 0xff, 0x80) ;
				}

			msxmem.bLastCartPage[iCart] = (iSize / 0x2000) - 1 ;
			msxmem.bCartType[iCart] = pcf->cart[iCart].iType ;

			break ;

		case CONFIG_CARTRIDGE_FM_PAC:
			msxmem.bMapType[ps][ss][1] = type ;

			if (!iSize)
				{
				iSize = 0x4000 ;

				if (!MemoryNew (&pb, 0x8000) )
					return FALSE ;

				memset (pb, 0xff, 0x8000) ;
				}
			else if (!MemoryReSize (&pb, iSize + 0x4000) )
				{
				MemoryFree (&pb) ;

				return FALSE ;
				}

			memset (pb + iSize, 0, 0x2000) ;
			memset (pb + iSize + 0x2000, 0xff, 0x2000) ;
			LoadSRAM (pb + iSize, pcf->cart[iCart].szSRAMfile) ;
			pb[iSize + 0x1ffe] = 0x4d ;
			pb[iSize + 0x1fff] = 0x69 ;
			pb[0x3ff7] = 0 ;

			msxmem.bLastCartPage[iCart] = (iSize / 0x2000) - 1 ;
			msxmem.bCartType[iCart] = pcf->cart[iCart].iType ;

			break ;
		
		case CONFIG_CARTRIDGE_GM2:
			if (!MemoryReSize (&pb, 0x24000) )
				{
				if (iSize) MemoryFree (&pb) ;

				return FALSE ;
				}

			if (iSize < 0x20000)
				{
				memset (pb + iSize, 0xff, 0x20000 - iSize) ;
				}

			iSize = 0x20000 ;
			LoadSRAM (pb + 0x21000, pcf->cart[iCart].szSRAMfile) ;
			memcpy (pb + 0x20000, pb + 0x21000, 0x1000) ;
			memcpy (pb + 0x23000, pb + 0x22000, 0x1000) ;

			for (p=1;p<=2;p++)
				msxmem.bMapType[ps][ss][p] = type ;
				
			msxmem.bLastCartPage[iCart] = (iSize / 0x2000) - 1 ;
			msxmem.bCartType[iCart] = pcf->cart[iCart].iType ;

			break ;		
	
		case CONFIG_CARTRIDGE_THSCCP:
		case CONFIG_CARTRIDGE_SDSCCP:
		case CONFIG_CARTRIDGE_SCCP:
			if (!MemoryNew (&pb, 0x24000) )
				return FALSE ;

			memset (pb + 0x20000, 0xff, 0x4000) ;
			for (i=0;i<8;i++)
				{
				memset (pb + 0x21800 + i * 0x100, 0, 0x80) ;
				memset (pb + 0x218A0 + i * 0x100, 0, 0x20) ;
				memset (pb + 0x23800 + i * 0x100, 0, 0xA0) ;
				}

			i = pcf->cart[iCart].iType ;
			if (i == CONFIG_CARTRIDGE_THSCCP)
				{
				memset (pb, 0, 0x10000) ;
				memset (pb + 0x10000, 0xff, 0x10000) ;
				}
			else if (i == CONFIG_CARTRIDGE_SDSCCP)
				{
				memset (pb, 0xff, 0x10000) ;
				memset (pb + 0x10000, 0, 0x10000) ;
				}
			else
				{
				memset (pb, 0, 0x20000) ;
				}

			iSize = 0x20000 ;
			for (p=1;p<=2;p++)
				msxmem.bMapType[ps][ss][p] = type ;
			
			msxmem.bLastCartPage[iCart] = (iSize / 0x2000) - 1 ;
			msxmem.bCartType[iCart] = pcf->cart[iCart].iType ;

			break ;		
	
		case CONFIG_CARTRIDGE_RAM256:
		case CONFIG_CARTRIDGE_RAM512:
		case CONFIG_CARTRIDGE_RAM768:
			switch (pcf->cart[iCart].iType)
				{
				case CONFIG_CARTRIDGE_RAM256:
					iSize = 0x40000 ; break ;
				case CONFIG_CARTRIDGE_RAM512:
					iSize = 0x80000 ; break ;
				case CONFIG_CARTRIDGE_RAM768:
					iSize = 0xc0000 ; break ;
				}

			if (!MemoryNew (&pb, iSize + 0x2000) )
				return FALSE ;

			memset (pb, 0, iSize) ;
			memset (pb + iSize, 0xff, 0x2000) ;

			for (p=1;p<=2;p++)
				msxmem.bMapType[ps][ss][p] = type ;
			
			msxmem.bLastCartPage[iCart] = (iSize / 0x2000) - 1 ;
			msxmem.bCartType[iCart] = pcf->cart[iCart].iType ;

			break ;
		}

	msxmem.pCarts[iCart] = pb ;

	return TRUE ;
	}

BOOL LoadMSXMemory (CONFIG *pcf)
	{
	int		ps, ss, p, i ;

	memset (&msxmem, 0, sizeof (MSXMEM) ) ;

	// the void
	if (!MemoryNew (&msxmem.pVoid, 0x4000) )
		goto error ;

	memset (msxmem.pVoid, 0xff, 0x4000) ;

	for (ps=0;ps<4;ps++) for (ss=0;ss<4;ss++) for (p=0;p<4;p++)
		{
		msxmem.pMap[ps][ss][p] = msxmem.pVoid ;
		msxmem.bMapType[ps][ss][p] = MAP_ROM ;
		}

	// main bios
	if (!LoadBIOS (&msxmem.pBIOS, (pcf->iMSXlanguage == CONFIG_LANGUAGE_INT ? 
		MSX + pcf->iMSXversion : MSX_JP + pcf->iMSXversion) ) )
			goto error ;

	msxmem.pMap[0][0][0] = msxmem.pBIOS ;
	msxmem.pMap[0][0][1] = msxmem.pBIOS + 0x4000 ;
	
	// extended bios
	if (pcf->iMSXversion != CONFIG_VERSION_MSX1)
		{
		if (!LoadBIOS (&msxmem.pSubBIOS, (pcf->iMSXlanguage == CONFIG_LANGUAGE_INT ? 
			MSX2EXT + pcf->iMSXversion - CONFIG_VERSION_MSX2 : 
			MSX2EXT_JP + pcf->iMSXversion - CONFIG_VERSION_MSX2) ) )
				goto error ;

		for (ps=0;ps<4;ps++) for (ss=0;ss<4;ss++) for (p=0;p<4;p++)
			{
			if (pcf->mem.iMem[ps][ss][p] == CONFIG_MEMORY_SUBBIOS)
				{
				msxmem.pMap[ps][ss][p] = msxmem.pSubBIOS ;
				break ;
				}
			}
		}
	
	// disk rom -- if present (only one)
	for (ps=0;ps<4;ps++) for (ss=0;ss<4;ss++) for (p=0;p<4;p++)
		{
		if (pcf->mem.iMem[ps][ss][p] == CONFIG_MEMORY_DISKROM)
			{
			if (!LoadBIOS (&msxmem.pDiskBIOS, DISK) )
				goto error ;
			
			msxmem.pMap[ps][ss][p] = msxmem.pDiskBIOS ;
			}
		}

	// cartridges
	for (i=0;i<4;i++)
		if (!LoadCartridge (pcf, i) ) return FALSE ;
	
	// ram memory mapper
	for (ps=0;ps<4;ps++) for (ss=0;ss<4;ss++)
		{
		if (pcf->mem.iMem[ps][ss][0] != CONFIG_MEMORY_RAMMM)
			continue ;
		
		if (!MemoryNewZero (&msxmem.pMMRAM[ps*4+ss], 2 << (pcf->mem.iMapperSize[ps*4+ss] + 15) ) )
			goto error ;
		
		msxmem.bLastMMpage[ps * 4 + ss] = (2 << (pcf->mem.iMapperSize[ps * 4 + ss] + 1) ) - 1 ;
		for (p=0;p<4;p++)
			msxmem.bMapType[ps][ss][p] = MAP_RAMMM ;
		}

	p = 0 ;
	for (i=0;i<16;i++)
		{
		if (msxmem.bLastMMpage[i] > p)
			p = msxmem.bLastMMpage[i] ;
		}

	msxmem.bMaxMMpage = p ;

	// standard ram
	for (ps=0;ps<4;ps++) for (ss=0;ss<4;ss++) for (p=0;p<4;p++)
		{
		if (pcf->mem.iMem[ps][ss][p] != CONFIG_MEMORY_RAM)
			continue ;

		if (!MemoryNewZero (&msxmem.pRAM[ps][ss][p], 0x4000) )
			goto error ;
		///*******
		if (p == 3)
			{
			HANDLE	hFile ;

			hFile = CreateFile ("C:\\DEV32\\VMSX\\MSX Bin\\MONMSX.BIN",
				GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, 
				FILE_ATTRIBUTE_NORMAL | FILE_FLAG_SEQUENTIAL_SCAN, 0) ;

			SetFilePointer (hFile, 6, NULL, FILE_BEGIN) ;
			ReadFile (hFile, msxmem.pRAM[ps][ss][p], 0x1800, &i, NULL) ;
			CloseHandle (hFile) ;
			}

		///*******

		msxmem.pMap[ps][ss][p] = msxmem.pRAM[ps][ss][p] ;
		msxmem.bMapType[ps][ss][p] = MAP_RAM ;
		}

	// expanded flags
	for (i=0;i<4;i++)
		msxmem.bSlotExpanded[i] = pcf->mem.fSlotExpanded[i] ;

	// video ram
	if (pcf->iMSXversion == CONFIG_VERSION_MSX1)
		i = 0x4000 ;
	else
		{
		switch (pcf->iVRAMsize)
			{
			case CONFIG_VRAM_64KB:
				i = 0x10000 ;
				break ;
			case CONFIG_VRAM_128KB:
				i = 0x20000 ;
				break ;
			case CONFIG_VRAM_196KB:
				i = 0x30000 ;
				break ;
			}
		}
	
	if (!MemoryNewZero (&msxmem.pVideo, i) )
		goto error ;

	if (!MemoryNew (&msxmem.pSurface, 256 * 200) )
		goto error ;

	if (!MemoryNew (&msxmem.pSurfaceBack, 256 * 200) )
		goto error ;

	msxmem.dwLastVideoAddress = i - 1 ;

	return TRUE ;

error:
	MemoryFree (&msxmem.pBIOS) ;
	MemoryFree (&msxmem.pSubBIOS) ;
	MemoryFree (&msxmem.pDiskBIOS) ;

	for (i=0;i<16;i++)
		MemoryFree (&msxmem.pMMRAM[i]) ;

	for (ps=0;ps<4;ps++) for (ss=0;ss<4;ss++) for (p=0;p<4;p++)
		MemoryFree (&msxmem.pRAM[ps][ss][p]) ;

	for (i=0;i<3;i++)
		MemoryFree (&msxmem.pCarts[i]) ;
	
	MemoryFree (&msxmem.pVideo) ;
	MemoryFree (&msxmem.pSurface) ;
	MemoryFree (&msxmem.pSurfaceBack) ;

	return FALSE ;
	}

BOOL SaveSRAM (BYTE *pb, char *szFile)
	{
	HANDLE	hFile ;
	DWORD	dw ;

	if (!szFile[0]) return TRUE ;

	hFile = CreateFile (szFile, GENERIC_WRITE, 0, NULL, CREATE_ALWAYS, 
		FILE_ATTRIBUTE_NORMAL | FILE_FLAG_SEQUENTIAL_SCAN, 0) ;
	
	if (hFile == INVALID_HANDLE_VALUE)
		goto error ;

	if (!WriteFile (hFile, pb, 0x2000, &dw, NULL) || dw != 0x2000)
		goto error ;

	CloseHandle (hFile) ;

	return TRUE ;

error:
	if (hFile != INVALID_HANDLE_VALUE)
		CloseHandle (hFile) ;

	ErrorFile (g_hwndMain, szFile) ;

	return FALSE ;
	}

void FreeMSXMemory (CONFIG *pcf)
	{
	int		p, s, i ;

	MemoryFree (&msxmem.pBIOS) ;
	MemoryFree (&msxmem.pSubBIOS) ;
	MemoryFree (&msxmem.pDiskBIOS) ;

	for (i=0;i<16;i++)
		MemoryFree (&msxmem.pMMRAM[i]) ;

	for (p=0;p<4;p++) for (s=0;s<4;s++) for (i=0;i<4;i++)
		MemoryFree (&msxmem.pRAM[p][s][i]) ;

	for (i=0;i<4;i++)
		{
		switch (pcf->cart[i].iType)
			{
			case CONFIG_CARTRIDGE_GM2:
				SaveSRAM (msxmem.pCarts[i] + 0x21000, pcf->cart[i].szSRAMfile) ;
				break ;
			case CONFIG_CARTRIDGE_ASCII_8KB_SRAM:
				SaveSRAM (msxmem.pCarts[i] + (msxmem.bLastCartPage[i] + 1) * 0x2000,
					pcf->cart[i].szSRAMfile) ;
				break ;
			case CONFIG_CARTRIDGE_FM_PAC:
				SaveSRAM (msxmem.pCarts[i] + (msxmem.bLastCartPage[i] + 1) * 0x2000,
					pcf->cart[i].szSRAMfile) ;
				break ;
			}
		
		MemoryFree (&msxmem.pCarts[i]) ;
		}
	
	MemoryFree (&msxmem.pVideo) ;
	MemoryFree (&msxmem.pSurface) ;
	MemoryFree (&msxmem.pSurfaceBack) ;

	memset (&msxmem, 0, sizeof (MSXMEM) ) ;
	}
