/*
**
** File: scc.c -- software implementation of the Konami SCC
** emulator. More than one SCC supported, and SCC+ as well as
** normal megarom SCC.
*/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <assert.h>
#include "scc.h"

static int SCCSoundRate; 	/* Output rate (8 / 16 bit) */
static int SCCSoundFreq ;	/* Output freq (Hz) */
static int SCCBufSize;		/* size of sound buffer, in samples */
static int SCCNumChips;		/* total # of SCC's emulated */

static SCC *SCClist;		/* array of SCC's */

/* forward declarations */
static int _SCCInitChip (int, void*) ;
static void _SCCFreeChip (int) ;
static void _SCCUpdateChip (int, int) ;

/*
** Initialize SCC emulator(s).
**
** 'num' is the number of virtual SCC's to allocate
** 'rate' is sampling rate and 'bufsiz' is the size of the
** buffer that should be updated at each interval
*/

int SCCInit(int num, int freq, int rate, int bufsiz, ... )
{
    int		i ;
    va_list	ap ;
    void	*userbuffer ;
    int		moreargs = 1 ;

    assert (rate == 8 || rate == 16) ;

    va_start (ap,bufsiz) ;

    if (SCClist) return (-1) ;	/* duplicate init. */

    SCCNumChips = num ;
    SCCSoundRate = rate ;
    SCCSoundFreq = freq ;
    SCCBufSize = bufsiz ;

    SCClist = (SCC*)malloc (sizeof(SCC) * SCCNumChips) ;
    if (SCClist == NULL) return (0) ;
    for ( i = 0 ; i < SCCNumChips; i++ ) {
	if (moreargs) userbuffer = va_arg(ap,void*);
	if (userbuffer == NULL) moreargs = 0;
	if (_SCCInitChip (i,userbuffer) < 0) {
	    int j;
	    for ( j = 0 ; j < i ; j ++ ) {
		_SCCFreeChip(j);
	    }
	    return -1 ;
	}
    }
    return 0 ;
}

void SCCShutdown()
{
    int i;

    if (!SCClist) return ;

    for ( i = 0 ; i < SCCNumChips ; i++ ) {
	_SCCFreeChip(i);
    }
    free(SCClist); SCClist = NULL;
    SCCNumChips = SCCSoundRate = SCCSoundFreq = SCCBufSize = 0;
}

/*
** reset all chip registers.
*/
void SCCResetChip(int num)
{
    int i;
    SCC	*scc ;
    word *pw  ;
    
    assert (num < SCCNumChips) ;

    scc = &(SCClist[num]) ;

    if (SCCSoundRate == 8) {
	memset (scc->Buf, SCC_AUDIO_CONV(0), SCCBufSize);
    } else {
	pw = scc->Buf;
	i = SCCBufSize;
	while (i--) *pw++ = SCC_AUDIO_CONV16(0);
    }

    /* initialize hardware registers */
    memset (scc->Regs, 0, 17) ;
    for( i=10; i<=15; i++ ) scc->Regs[i] = 10;

    scc->Counter0 = scc->Counter1 = scc->Counter2 = scc->Counter3 = scc->Counter4 = 0;

    /* waves memory is not reset */
}

/*
** allocate buffers and clear registers for one of the emulated
** SCC chips.
*/
static int _SCCInitChip (int num, void *buf)
{
    SCC *scc ;
    
    assert (num < SCCNumChips) ;

    scc = &(SCClist[num]);

    scc->UserBuffer = 0;
    if (buf) {
	scc->Buf = buf;
	scc->UserBuffer = 1;
    } else {
	if ((scc->Buf=(void*)malloc(
		SCCSoundRate==8?SCCBufSize:SCCBufSize*2))==NULL)
	    return -1;
    }

    SCCResetChip (num) ;

    return 0;
}

/*
** release storage for a chip
*/
static void _SCCFreeChip(int num)
{
    SCC *scc ;
    
    assert (num < SCCNumChips) ;

    scc = &(SCClist[num]);

    if (scc->Buf && !scc->UserBuffer) free(scc->Buf); scc->Buf = NULL;
}

void SCCWriteReg (int n, int r, byte v, int t)
{
    static byte Mask[16] = { 0xff, 0xf, 0xff, 0xf, 0xff, 0xf, 
	   0xff, 0xf, 0xff, 0xf, 0xf, 0xf, 0xf, 0xf, 0xf, 0x1f } ;
    SCC	   *scc ;

    assert (n < SCCNumChips) ;
    r &= 0xff ;
    
    scc = &(SCClist[n]) ;

    switch (t) {
	case SCC_MEGAROM:
	case SCC_PLUSCOMP:
	    if (r < 0x80) {
		/* 0 - 0x7f is wave forms, 0x60-0x7f for channel 4 and 5 */
		scc->Waves[r] = v ;
		if (r >= 0x60) {
		    scc->Waves[r + 0x20] = v ;
		}
	    } else if (r < 0xa0) {
		/* 9880-988f = 9890-989f */
		scc->Regs[r & 15] = (v & Mask[r & 15]) ;
	    } else switch (t) {
		case SCC_MEGAROM:
		    if (r >= 0xe0) {
			scc->Regs[SCC_DEFORM] = v ;
		    } 
		    break ;
		case SCC_PLUSCOMP:
		    if ( (r < 0xe0) && (r >= 0xc0) ) {
			scc->Regs[SCC_DEFORM] = v ;
		    }
		    break ;
		default:
		    assert (0) ; /* t invalid */
		    break ;
		}
	    break ;
	case SCC_PLUSEXT:
	    if (r < 0xa0) {
		scc->Waves[r] = v ;
	    } else if (r < 0xc0) {
		scc->Regs[r & 15] = (v & Mask[r & 15]) ;
	    } else if (r < 0xe0) {
		scc->Regs[SCC_DEFORM] = v ;
	    }
	    break ;
	default:
	    assert (0) ; /* t parameter invalid */
	    break ;
    }
}

static void _SCCUpdateChip (int num, int part)
{
    int	x, i, c0, c1, l ;
    int	Incr0, Incr1, Incr2, Incr3, Incr4, Vol0, Vol1, Vol2, Vol3, Vol4 ;	
    SCC	*scc ;
    word *pw ;
    byte *pb ;
#ifdef SCC_INTERPOLATE
    int	c2 ;
#endif

    scc = &(SCClist[num]) ;

    x = ((int)scc->Regs[SCC_1FINE] + (((int)scc->Regs[SCC_1COARSE])<<8)) ;
    Incr0 = x ? 512 * SCC_CLOCK / SCCSoundFreq * 4 / x : 0 ;

    x = ((int)scc->Regs[SCC_2FINE] + (((int)scc->Regs[SCC_2COARSE])<<8)) ;
    Incr1 = x ? 512 * SCC_CLOCK / SCCSoundFreq * 4 / x : 0 ;

    x = ((int)scc->Regs[SCC_3FINE] + (((int)scc->Regs[SCC_3COARSE])<<8)) ;
    Incr2 = x ? 512 * SCC_CLOCK / SCCSoundFreq * 4 / x : 0 ;

    x = ((int)scc->Regs[SCC_4FINE] + (((int)scc->Regs[SCC_4COARSE])<<8)) ;
    Incr3 = x ? 512 * SCC_CLOCK / SCCSoundFreq * 4 / x : 0 ;

    x = ((int)scc->Regs[SCC_5FINE] + (((int)scc->Regs[SCC_5COARSE])<<8)) ;
    Incr4 = x ? 512 * SCC_CLOCK / SCCSoundFreq * 4 / x : 0 ;

    Vol0 = (scc->Regs[SCC_ENABLE] & 0x01) ? scc->Regs[SCC_1VOL] : 0 ;
    Vol1 = (scc->Regs[SCC_ENABLE] & 0x02) ? scc->Regs[SCC_2VOL] : 0 ;
    Vol2 = (scc->Regs[SCC_ENABLE] & 0x04) ? scc->Regs[SCC_3VOL] : 0 ;
    Vol3 = (scc->Regs[SCC_ENABLE] & 0x08) ? scc->Regs[SCC_4VOL] : 0 ;
    Vol4 = (scc->Regs[SCC_ENABLE] & 0x10) ? scc->Regs[SCC_5VOL] : 0 ;

    pw = (word*)scc->Buf ;
    pb = (byte*)scc->Buf ;
    
    i = part ? part : SCCBufSize ;

    if ( !(Vol0 || Vol1 || Vol2 || Vol3 || Vol4) ) {
	if (SCCSoundRate == 8) {
	    while (i--) *pb++ = SCC_AUDIO_CONV(0) ;
	} else {
	    while (i--) *pw++ = SCC_AUDIO_CONV16(0) ;
	}
	return ;
    }
    
    while (i--)
    {
	l = 0 ;

	if (Vol0 && Incr0) {
	    c0 = (scc->Counter0 + Incr0) & 0xffff ;
	    c1 = (sbyte)scc->Waves[0x00 + (c0>>11)] ;
#ifdef SCC_INTERPOLATE
	    c2 = (sbyte)scc->Waves[0x00 + ((c0>>11)+1)&0x1f] ;
	    l += Vol0*(c1+(c2-c1)*(c0&0x7ff)/0x800);
#else
	    l += Vol0*c1 ;
#endif
	    scc->Counter0 = c0 ;
	}

	if (Vol1 && Incr1) {
	    c0 = (scc->Counter1 + Incr1) & 0xffff ;
	    c1 = (sbyte)scc->Waves[0x20 + (c0>>11)] ;
#ifdef SCC_INTERPOLATE
	    c2 = (sbyte)scc->Waves[0x20 + ((c0>>11)+1)&0x1f] ;
	    l += Vol1*(c1+(c2-c1)*(c0&0x7ff)/0x800);
#else
	    l += Vol1*c1 ;
#endif
	    scc->Counter1 = c0 ;
	}

	if (Vol2 && Incr2) {
	    c0 = (scc->Counter2 + Incr2) & 0xffff ;
	    c1 = (sbyte)scc->Waves[0x40 + (c0>>11)] ;
#ifdef SCC_INTERPOLATE
	    c2 = (sbyte)scc->Waves[0x40 + ((c0>>11)+1)&0x1f] ;
	    l += Vol2*(c1+(c2-c1)*(c0&0x7ff)/0x800);
#else
	    l += Vol2*c1 ;
#endif
	    scc->Counter2 = c0 ;
	}

	if (Vol3 && Incr3) {
	    c0 = (scc->Counter3 + Incr3) & 0xffff ;
	    c1 = (sbyte)scc->Waves[0x60 + (c0>>11)] ;
#ifdef SCC_INTERPOLATE
	    c2 = (sbyte)scc->Waves[0x60 + ((c0>>11)+1)&0x1f] ;
	    l += Vol3*(c1+(c2-c1)*(c0&0x7ff)/0x800);
#else
	    l += Vol3*c1 ;
#endif
	    scc->Counter3 = c0 ;
	}

	if (Vol4 && Incr4) {
	    c0 = (scc->Counter4 + Incr4) & 0xffff ;
	    c1 = (sbyte)scc->Waves[0x80 + (c0>>11)] ;
#ifdef SCC_INTERPOLATE
	    c2 = (sbyte)scc->Waves[0x80 + ((c0>>11)+1)&0x1f] ;
	    l += Vol4*(c1+(c2-c1)*(c0&0x7ff)/0x800);
#else
	    l += Vol4*c1 ;
#endif
	    scc->Counter4 = c0 ;
	}

	if (SCCSoundRate == 8) {
	    *pb++ = SCC_AUDIO_CONV(l / 75) ;
	} else {
	    *pw++ = SCC_AUDIO_CONV16(l) ;
	}
    }
}

/*
** called to update all chips
*/
void SCCUpdate (int part)
{
    int i;

    for ( i = 0 ; i < SCCNumChips; i++ ) {
	_SCCUpdateChip(i, part) ;
    }
}

/*
** return the buffer into which SCCUpdate() has just written it's sample
** data
*/
void *SCCBuffer(int n)
{
    assert (n < SCCNumChips) ;

    return SCClist[n].Buf ;
}

