unit Comms;

interface

Uses Classes,SyncObjs;

Type
  TConnectionParameters = Class
  Protected
    FList : TStringList;
  Public
    Constructor Create;
    Destructor  Destroy; Override;
    Procedure   Put(Key,Value: String);
    Function    GetAsInteger(Key: String): Integer;
    Function    GetAsFloat(Key: String): Extended;
    Function    GetAsBoolean(Key: String): Boolean;
    Function    GetAsString(Key: String): String;
    Function    HasParameter(Key: String): Boolean;
  End;

  TConnectorResult = (crOk, crFail, crDisconnected);

  PDataDescriptor = ^TDataDescriptor;
  TDataDescriptor = Packed Record
    Data : Pointer;
    Size : Integer;
  End;

  TConnector = Class
  Protected
    FConnected : Boolean;
  Public
    Constructor Create;
    Function    Connnect(Const ConnectionParameters: TConnectionParameters): TConnectorResult; Dynamic; Abstract;
    Procedure   Disconnect; Dynamic; Abstract;
    Function    Transmit(Const Descriptor: TDataDescriptor): TConnectorResult; Dynamic; Abstract;
    Function    Receive(Out Descriptor: TDataDescriptor): TConnectorResult; Dynamic; Abstract;
    Function    IsConnected: Boolean; Dynamic;
  End;

  TSocketConnector = Class(TConnector)
  Protected
    FSocket       : Integer; // Handle to socket
    FListenSocket : Integer; // Handle to socket
    FHost         : Integer;
    FPort         : Integer;
    FMutex        : TCriticalSection;
    FBuffer       : Pointer;
    Function    StopReceiving: Boolean;
    Function    StopSending: Boolean;
    Function    Close: Boolean;
  Public
    Constructor Create;
    Destructor  Destroy; Override;
    Function    Connnect(Const ConnectionParameters: TConnectionParameters): TConnectorResult; Override;
    Procedure   Disconnect; Override;
    Function    Transmit(Const Descriptor: TDataDescriptor): TConnectorResult; Override;
    Function    Receive(Out Descriptor: TDataDescriptor): TConnectorResult; Override;
  End;

  TExecutingThreadProc = Procedure Of Object;

  TExecutingThread = Class(TThread)
  Protected
    FExecuteProc : TExecutingThreadProc;
  Public
    Constructor Create(CreateSuspended: Boolean; ExecuteProc: TExecutingThreadProc);
    Procedure   Execute; Override;
  End;

  TCommPipe = Class
  Protected
    FQueue        : TList;               // List of TDataDescriptor
    FThread       : TExecutingThread;
    FMutex        : TCriticalSection;
    Function    IsThreadRunning: Boolean;
  Public
    ThreadRunning : Boolean;
    Constructor Create(ExecuteProc: TExecutingThreadProc);
    Destructor  Destroy; Override;
    Procedure   Enqueue(Const Descriptor: TDataDescriptor);
    Procedure   Dequeue(Out Descriptor: TDataDescriptor);
    Function    ThreadTerminated: Boolean;
    Procedure   Flush;
  End;

  TRepeatFunction = Function: Boolean Of Object;

  TCommDisconnectedEvent = Procedure Of Object;

  TCommReceiveEvent      = Procedure(Data: Pointer; Size: Integer) Of Object;

  TComm = Class
  Protected
    FConnector      : TConnector;
    FTransmitPipe   : TCommPipe;
    FReceivePipe    : TCommPipe;
    FOnDisconnected : TCommDisconnectedEvent;
    Procedure   HandleLostConnection;
    Procedure   TransmitThreadExecute;
    Procedure   ReceiveThreadExecute;
  Public
    Constructor Create(AConnector: TConnector);
    Destructor  Destroy; Override;
    Function    Connnect(Const ConnectionParameters: TConnectionParameters): TConnectorResult;
    Procedure   Disconnect;
    Procedure   Transmit(Data: Pointer; Size: Integer);
    Function    Receive(Out Data: Pointer): Integer;
    Property    OnDisconnected : TCommDisconnectedEvent Read FOnDisconnected Write FOnDisconnected;
  End;

implementation

Uses SysUtils,WinSock,Windows;

Const BufferSize = 64 * 1024;

Var
  Version  : Word;
  _WSAData : WSADATA;

Procedure RepeatWithTimeout(Func: TRepeatFunction; TryAgainWait,Timeout: Integer);
Var
  T0   : Integer;
  T1   : Integer;
  Done : Boolean;

Begin
  T0   := GetTickCount;
  Done := False;
  While Not Done Do
  Begin
    If Not Func Then
    Begin
      T1 := GetTickCount;
      If (T1 < T0) Or (T1 > T0 + Timeout) Then Done := True;
    End
    Else Done := True;
    If Not Done Then Sleep(TryAgainWait);
  End;
End;

// TConnectionParameters

Constructor TConnectionParameters.Create;
Begin
  Inherited;
  FList := TStringList.Create;
  FList.Sorted := True;
End;

Destructor TConnectionParameters.Destroy;
Var I: Integer;
Begin
  For I := 0 To FList.Count - 1 Do FreeMem(Pointer(FList.Objects[I]));
  FList.Free;
  Inherited;
End;

Function TConnectionParameters.HasParameter(Key: String): Boolean;
Begin
  Result := (FList.IndexOf(Key) >= 0);
End;

Procedure TConnectionParameters.Put(Key,Value: String);
Var
  I : Integer;
  P : PChar;

Begin
  I := FList.IndexOf(Key);
  If I >= 0 Then
  Begin
    FreeMem(Pointer(FList.Objects[I]));
    GetMem(P,Length(Value) + 1);
    StrPCopy(P,Value);
    FList.Objects[I] := Pointer(P);
  End
  Else
  Begin
    GetMem(P,Length(Value) + 1);
    StrPCopy(P,Value);
    FList.AddObject(Key,Pointer(P));
  End;
End;

Function TConnectionParameters.GetAsInteger(Key: String): Integer;
Var
  St  : String;
  I,J : Integer;

Begin
  St := Trim(GetAsString(Key));
  If St <> '' Then
  Begin
    Val(St,I,J);
    If J <> 0 Then I := 0;
    Result := I;
  End
  Else Result := 0;
End;

Function TConnectionParameters.GetAsFloat(Key: String): Extended;
Var
  St : String;
  E  : Extended;
  I  : Integer;

Begin
  St := Trim(GetAsString(Key));
  If St <> '' Then
  Begin
    Val(St,E,I);
    If I <> 0 Then E := 0;
    Result := E;
  End
  Else Result := 0;
End;

Function TConnectionParameters.GetAsBoolean(Key: String): Boolean;
Var St: String;
Begin
  St := Trim(GetAsString(Key));
  If St <> ''
   Then Result := (St = '1')
   Else Result := False;
End;

Function TConnectionParameters.GetAsString(Key: String): String;
Var I: Integer;
Begin
  I := FList.IndexOf(Key);
  If I >= 0
   Then Result := StrPas(PChar(FList.Objects[I]))
   Else Result := '';
End;

// TConnector

Constructor TConnector.Create;
Begin
  FConnected := False;
End;

Function TConnector.IsConnected: Boolean;
Begin
  Result := FConnected;
End;

// TSocketConnector

Constructor TSocketConnector.Create;
Begin
  Inherited;
  FMutex  := TCriticalSection.Create;
  FHost   := INADDR_LOOPBACK;
  FPort   := 0;
  FSocket := -1;
  GetMem(FBuffer,BufferSize + 1);
End;

Destructor TSocketConnector.Destroy;
Begin
  Disconnect;
  FMutex.Free;
  FreeMem(FBuffer);
  Inherited;
End;

Function TSocketConnector.Connnect(Const ConnectionParameters: TConnectionParameters): TConnectorResult;
Var
  Host         : Integer;
  Port         : Integer;
  Protocol     : String;
  Address      : SockAddr_In;
  InAddress    : SockAddr_In;
  InLen        : Integer;
  ReuseAddress : Integer;
  BufSize      : Integer;
  NonBlocking  : LongInt;
  Listen       : Boolean;

Begin
  Try
    FMutex.Enter;
    Result := crFail;
    If (ConnectionParameters <> Nil) And
       ConnectionParameters.HasParameter('host') And
       ConnectionParameters.HasParameter('port') And
       ConnectionParameters.HasParameter('protocol') And
       ConnectionParameters.HasParameter('listen') And
       Not IsConnected Then
    Begin
      Host     := ConnectionParameters.GetAsInteger('host');
      Port     := ConnectionParameters.GetAsInteger('port');
      Protocol := UpperCase(ConnectionParameters.GetAsString('protocol'));
      Listen   := ConnectionParameters.GetAsBoolean('listen');
      If Listen Then
      Begin
        FillChar(Address,SizeOf(Address),0);
        Address.sin_family      := AF_INET;
        Address.sin_port        := htons(Port);
        Address.sin_addr.s_addr := htonl(INADDR_ANY);
        If Protocol = 'TCP'
         Then FListenSocket := socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)
         Else FListenSocket := socket(AF_INET, SOCK_DGRAM,  IPPROTO_UDP);
        If FListenSocket <> INVALID_SOCKET Then
        Begin
          ReuseAddress := 1;
          If SetSockOpt(FListenSocket, SOL_SOCKET, SO_REUSEADDR, @ReuseAddress, SizeOf(ReuseAddress)) = 0 Then
          Begin
            If Bind(FListenSocket, Address, SizeOf(Address)) = 0 Then
            Begin
              BufSize := BufferSize;
              If setsockopt(FListenSocket, SOL_SOCKET, SO_RCVBUF, @BufSize, Sizeof(BufSize)) = 0 Then
              Begin
                If setsockopt(FListenSocket, SOL_SOCKET, SO_SNDBUF, @BufSize, Sizeof(BufSize)) = 0 Then
                Begin
                  NonBlocking := 1; // Important
                  If ioctlsocket(FListenSocket, FIONBIO, NonBlocking) = 0 Then
                  Begin
                    If WinSock.listen(FListenSocket, SOMAXCONN) = 0 Then
                    Begin
                      FillChar(InAddress,SizeOf(InAddress),0);
                      InAddress.sin_family := AF_INET;
                      InLen   := SizeOf(InAddress);
                      FSocket := accept(FListenSocket,@InAddress,@InLen);
                      If FSocket <> INVALID_SOCKET Then
                      Begin


                        Result := crOk;
                      End
                      Else CloseSocket(FListenSocket);
                    End
                    Else CloseSocket(FListenSocket);
                  End
                  Else CloseSocket(FListenSocket);
                End
                Else CloseSocket(FListenSocket);
              End
              Else CloseSocket(FListenSocket);
            End
            Else CloseSocket(FListenSocket);
          End
          Else CloseSocket(FListenSocket);
        End;
      End
      Else
      Begin
        FListenSocket := INVALID_SOCKET;
        FillChar(Address,SizeOf(Address),0);
        Address.sin_family      := AF_INET;
        Address.sin_port        := htons(Port);
        Address.sin_addr.s_addr := htonl(Host);
        If Protocol = 'TCP'
         Then FSocket := socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)
         Else FSocket := socket(AF_INET, SOCK_DGRAM,  IPPROTO_UDP);
        If FSocket <> INVALID_SOCKET Then
        Begin
          If WinSock.connect(FSocket,Address,SizeOf(Address)) = 0 Then
          Begin
            BufSize := BufferSize;
            If setsockopt(FSocket, SOL_SOCKET, SO_RCVBUF, @BufSize, Sizeof(BufSize)) = 0 Then
            Begin
              If setsockopt(FSocket, SOL_SOCKET, SO_SNDBUF, @BufSize, Sizeof(BufSize)) = 0 Then
              Begin
                NonBlocking := 1; // Important
                If ioctlsocket(FSocket, FIONBIO, NonBlocking) = 0 Then
                Begin
                  FHost      := Host;
                  FPort      := Port;
                  FConnected := True;
                  Result     := crOk;
                End
                Else CloseSocket(FSocket);
              End
              Else CloseSocket(FSocket);
            End
            Else CloseSocket(FSocket);
          End
          Else CloseSocket(FSocket);
        End;
      End;
    End;
  Finally
    FMutex.Leave;
  End;
End;

Function TSocketConnector.StopReceiving: Boolean;
Var Code: Integer;
Begin
  Code := shutdown(FSocket, SD_RECEIVE);
  If Code = SOCKET_ERROR Then
  Begin
    Code := WSAGetLastError;
    Result := (Code <> WSAEINPROGRESS);
  End
  Else Result := True;
End;

Function TSocketConnector.StopSending: Boolean;
Var Code: Integer;
Begin
  Code := shutdown(FSocket, SD_SEND);
  If Code = SOCKET_ERROR Then
  Begin
    Code := WSAGetLastError;
    Result := (Code <> WSAEINPROGRESS);
  End
  Else Result := True;
End;

Function TSocketConnector.Close: Boolean;
Var Code: Integer;
Begin
  Code := CloseSocket(FSocket);
  If Code = SOCKET_ERROR Then
  Begin
    Code := WSAGetLastError;
    Result := (Code <> WSAEINPROGRESS);
  End
  Else Result := True;
End;

Procedure TSocketConnector.Disconnect;
Const
  Timeout      = 5000;
  TryAgainWait = 10;

Var Code: Integer;  
Begin
  Try
    FMutex.Enter;
    If FConnected Then
    Begin
      FConnected := False;

      RepeatWithTimeout(StopReceiving, TryAgainWait, Timeout);
      RepeatWithTimeout(StopSending, TryAgainWait, Timeout);

      // Clear the socket buffer(s)

      Repeat
        Code := recv(FSocket,FBuffer^,BufferSize,0);
        Sleep(1);
      Until Code <= 0;

      RepeatWithTimeout(Close, TryAgainWait, Timeout);
    End;
  Finally
    FMutex.Leave;
  End;
End;

Function TSocketConnector.Transmit(Const Descriptor: TDataDescriptor): TConnectorResult;
Var Code: Integer;
Begin
  Try
    FMutex.Enter;
    Result := crFail;
    If FConnected Then
    Begin
      Code := send(FSocket,Descriptor.Data^,Descriptor.Size,0);
      If Code = SOCKET_ERROR Then
      Begin
        Code := WSAGetLastError;
        If Code <> WSAEWOULDBLOCK Then Result := crDisconnected;
      End
      Else Result := crOk;
    End;
  Finally
    FMutex.Leave;
  End;
End;

Function TSocketConnector.Receive(Out Descriptor: TDataDescriptor): TConnectorResult;
Var Code: Integer;
Begin
  Try
    FMutex.Enter;
    If FConnected
     Then Descriptor.Size := recv(FSocket,FBuffer^,BufferSize,0)
     Else Descriptor.Size := 0;
    If Descriptor.Size > 0 Then
    Begin
      GetMem(Descriptor.Data,Descriptor.Size);
      Move(FBuffer^,Descriptor.Data^,Descriptor.Size);
      Result := crOk;
    End
    Else
    Begin
      Result := crFail;
      Descriptor.Data := Nil;
      If Descriptor.Size = SOCKET_ERROR Then
      Begin
        Descriptor.Size := 0;
        Code := WSAGetLastError;
        If Code <> WSAEWOULDBLOCK Then Result := crDisconnected;
      End;
    End;
  Finally
    FMutex.Leave;
  End;
End;

// TExecutingThread

Constructor TExecutingThread.Create(CreateSuspended: Boolean; ExecuteProc: TExecutingThreadProc);
Begin
  FExecuteProc := ExecuteProc;
  Inherited Create(CreateSuspended);
End;

Procedure TExecutingThread.Execute;
Begin
  If @FExecuteProc <> Nil Then FExecuteProc;
End;

// TCommPipe

Constructor TCommPipe.Create(ExecuteProc: TExecutingThreadProc);
Begin
  Inherited Create;
  FQueue        := TList.Create;
  FMutex        := TCriticalSection.Create;
  ThreadRunning := False;
  FThread       := TExecutingThread.Create(True, ExecuteProc);
End;

Destructor TCommPipe.Destroy;
Const
  Timeout      = 5000;
  TryAgainWait = 10;
  
Begin
  FThread.Terminate;
  RepeatWithTimeout(IsThreadRunning, TryAgainWait, Timeout);
  FQueue.Free;
  FMutex.Free;
  Inherited;
End;

Function TCommPipe.IsThreadRunning: Boolean;
Begin
  Result := ThreadRunning;
End;

Procedure TCommPipe.Enqueue(Const Descriptor: TDataDescriptor);
Var P: PDataDescriptor;
Begin
  If (Descriptor.Data <> Nil) And (Descriptor.Size > 0) Then
  Begin
    GetMem(P,SizeOf(TDataDescriptor));
    P^ := Descriptor;
    Try
      FMutex.Enter;
      FQueue.Add(P);
    Finally
      FMutex.Leave;
    End;
  End;
End;

Procedure TCommPipe.Dequeue(Out Descriptor: TDataDescriptor);
Begin
  Try
    FMutex.Enter;
    If FQueue.Count > 0 Then
    Begin
      Descriptor := PDataDescriptor(FQueue.Items[0])^;
      FreeMem(PDataDescriptor(FQueue.Items[0]));
      FQueue.Delete(0);
    End
    Else
    Begin
      Descriptor.Data := Nil;
      Descriptor.Size := 0;
    End;
  Finally
    FMutex.Leave;
  End;
End;

Function TCommPipe.ThreadTerminated: Boolean;
Begin
  Result := FThread.Terminated;
End;

Procedure TCommPipe.Flush;
Var
  I : Integer;
  P : PDataDescriptor;

Begin
  Try
    FMutex.Enter;
    For I := 0 To FQueue.Count - 1 Do
    Begin
      P := PDataDescriptor(FQueue.Items[I]);
      FreeMem(P.Data,P.Size);
      FreeMem(PDataDescriptor(FQueue.Items[I]));
    End;
    FQueue.Clear;
  Finally
    FMutex.Leave;
  End;
End;

// TComm

Constructor TComm.Create(AConnector: TConnector);
Begin
  Inherited Create;
  FConnector      := AConnector;
  FTransmitPipe   := TCommPipe.Create(TransmitThreadExecute);
  FReceivePipe    := TCommPipe.Create(ReceiveThreadExecute);
  FOnDisconnected := Nil;
End;

Destructor TComm.Destroy;
Begin
  FConnector.Disconnect;
  FTransmitPipe.Free;
  FReceivePipe.Free;
  Inherited;
End;

Procedure TComm.TransmitThreadExecute;
Const WaitAmount = 5;
Var Descriptor: TDataDescriptor;
Begin
  Try
    While Not FTransmitPipe.ThreadTerminated Do
    Begin
      FTransmitPipe.ThreadRunning := True;
      FTransmitPipe.Dequeue(Descriptor);
      If (Descriptor.Data <> Nil) And (Descriptor.Size > 0) Then
      Begin
        Try
          If FConnector.IsConnected Then
          Begin
            If FConnector.Transmit(Descriptor) = crDisconnected Then HandleLostConnection;
          End
          Else Sleep(WaitAmount);
        Finally
          FreeMem(Descriptor.Data,Descriptor.Size);
        End;
      End
      Else Sleep(WaitAmount);
    End;
  Except
  End;
  FTransmitPipe.ThreadRunning := False;
End;

Procedure TComm.ReceiveThreadExecute;
Const WaitAmount = 5;
Var
  Descriptor      : TDataDescriptor;
  ConnectorResult : TConnectorResult;

Begin
  Try
    While Not FReceivePipe.ThreadTerminated Do
    Begin
      FReceivePipe.ThreadRunning := True;
      If FConnector.IsConnected Then
      Begin
        ConnectorResult := FConnector.Receive(Descriptor);
        Case ConnectorResult Of
          crOk:
          Begin
            If (Descriptor.Data <> Nil) And (Descriptor.Size > 0)
             Then FReceivePipe.Enqueue(Descriptor)
             Else Sleep(WaitAmount);
          End;
          crFail: Sleep(WaitAmount); // Happens if there is no data to read
          crDisconnected: HandleLostConnection;
        End;
      End
      Else Sleep(WaitAmount);
    End;
  Except
  End;
  FReceivePipe.ThreadRunning := False;
End;

Procedure TComm.HandleLostConnection;
Begin
  Disconnect;
  If @FOnDisconnected <> Nil Then FOnDisconnected;
End;

Function TComm.Connnect(Const ConnectionParameters: TConnectionParameters): TConnectorResult;
Begin
  Result := FConnector.Connnect(ConnectionParameters);
End;

Procedure TComm.Disconnect;
Begin
  FConnector.Disconnect;
  FTransmitPipe.Flush;
  FReceivePipe.Flush;
End;

Procedure TComm.Transmit(Data: Pointer; Size: Integer);
Var Descriptor: TDataDescriptor;
Begin
  If FConnector.IsConnected And (Data <> Nil) And (Size > 0) Then
  Begin
    // Make a local copy of the data to be sent and queue it for transmission
    
    GetMem(Descriptor.Data,Size);
    Descriptor.Size := Size;
    Move(Data^,Descriptor.Data^,Size);
    FTransmitPipe.Enqueue(Descriptor);
  End;
End;

Function TComm.Receive(Out Data: Pointer): Integer;
Var Descriptor: TDataDescriptor;
Begin
  FReceivePipe.Dequeue(Descriptor);
  Data   := Descriptor.Data;
  Result := Descriptor.Size;
End;

Initialization
  Version := MAKEWORD(1,1);
  WSAStartup(Version, _WSAData);
Finalization
  WSACleanup;
end.
