Unit WriteAVI;
{ ---------------------------------------------------------------- }
{ Unit to handle saving to .AVI files.  Only handles 8-bit         }
{ uncompressed and RLE-compressed 8-bit .AVI files.                }
{ ---------------------------------------------------------------- }

Interface

Uses AVI,HandTIA;

Const
  AVISampleRate      = 31440; // I know, I know...but this is divisible by 60.  Will have to interpolate...
  AVIFPS             = 60;
  AVISamplesPerFrame = AVISampleRate Div AVIFPS;

Type
  TAVIExpand      = (exNone,exHorz2x);
  TAVICompression = (acNone,acRLE);
  TAVIFile = Object
  Private
    F         : File;
    Width     : LongInt;
    Height    : LongInt;
    Comp      : TAVICompression;
    FileName  : String;
    NumFrames : LongInt;
    Expand    : TAVIExpand;
    Palette   : TPalette;
    Sound     : Boolean;
    SoundBuf  : Pointer;
    { ------------- Internal WriteFrame variables -------------- }
    RIFF      : TRIFF;
    StrLV     : TStreamListV;
    StrLA     : TStreamListA;
    MovI      : TMovI;
    DV        : TChunk;
    DA        : TChunk;
    RLE       : Pointer;
    SoundLast : Integer;
  Public
    Opened    : Boolean;
    Procedure   Init;
    Procedure   Open(Const AFileName: String; AWidth,AHeight: LongInt;
                     AComp: TAVICompression; AExpand: TAVIExpand;
                     Var Pal: TPalette; ASound: Boolean);
    Procedure   WriteFrame(VideoBuf,AudioBuf: Pointer; AudioBufLast, AudioBufPos, AudioBufEnd: Integer; VFlip: Boolean);
    Procedure   Close;
  End;

Implementation

Uses Windows,Math;

Type
  PRLERec = ^TRLERec;
  TRLERec = Packed Record
    Count : Byte;
    Num   : Byte;
  End;

Procedure TAVIFile.Init;
Begin
  FileName  := '';
  NumFrames := 0;
  Width     := 320;
  Height    := 200;
  Comp      := acNone;
  Expand    := exNone;
  Opened    := False;
  SoundBuf  := Nil;
  FillChar(Palette,SizeOf(Palette),#0);
End; { TAVIFile.Init }

Procedure TAVIFile.Open(Const AFileName: String; AWidth,AHeight: LongInt;
                        AComp: TAVICompression; AExpand: TAVIExpand;
                        Var Pal: TPalette; ASound: Boolean);
Var I: Integer;
Begin
  If Not Opened Then
  Begin
    GetMem(SoundBuf, AVISamplesPerFrame);
    FillChar(SoundBuf^, AVISamplesPerFrame, 0);
    FileName  := AFileName;
    NumFrames := 0;
    SoundLast := 0;
    Width     := AWidth;
    Height    := AHeight;
    Comp      := AComp;
    Expand    := AExpand;
    Opened    := True;
    Palette   := Pal;
    Sound     := ASound;
    Assign(F,FileName);
    ReWrite(F,1);
    GetMem(RLE,Width * 2 + 2);   { Leave room for the end-of-line specifier }
    RIFF.Chunk.FourCC                                      := CC_RIFF;
    RIFF.Chunk.dwSize                                      := 0;           { Will fix on close }
    RIFF.AVI                                               := CC_AVI;
    RIFF.HeaderList.Chunk.FourCC                           := CC_LIST;

    If Sound
     Then RIFF.HeaderList.Chunk.dwSize                     := SizeOf(RIFF.HeaderList) + SizeOf(StrLV) + SizeOf(StrLA) - SizeOf(TChunk)
     Else RIFF.HeaderList.Chunk.dwSize                     := SizeOf(RIFF.HeaderList) + SizeOf(StrLV) - SizeOf(TChunk);

    RIFF.HeaderList.hdrl                                   := CC_hdrl;
    RIFF.HeaderList.AVIHeader.Chunk.FourCC                 := CC_avih;
    RIFF.HeaderList.AVIHeader.Chunk.dwSize                 := SizeOf(RIFF.HeaderList.AVIHeader.Header);
    RIFF.HeaderList.AVIHeader.Header.dwMicroSecPerFrame    := 60000;//33367;       // ~29.97 fps (NTSC)
    RIFF.HeaderList.AVIHeader.Header.dwMaxBytesPerSec      := 6720000;//3728000;
    RIFF.HeaderList.AVIHeader.Header.dwPaddingGranularity  := 0;
    RIFF.HeaderList.AVIHeader.Header.dwFlags               := $110;//$810;
    RIFF.HeaderList.AVIHeader.Header.dwTotalFrames         := NumFrames;        { Will fix on close }
    RIFF.HeaderList.AVIHeader.Header.dwInitialFrames       := 0;

    If Sound
     Then RIFF.HeaderList.AVIHeader.Header.dwStreams       := 2
     Else RIFF.HeaderList.AVIHeader.Header.dwStreams       := 1;
     
    RIFF.HeaderList.AVIHeader.Header.dwSuggestedBufferSize := 120000;

    If Expand = exHorz2x
     Then RIFF.HeaderList.AVIHeader.Header.dwWidth         := Width * 2
     Else RIFF.HeaderList.AVIHeader.Header.dwWidth         := Width;
     
    RIFF.HeaderList.AVIHeader.Header.dwHeight              := Height;
    RIFF.HeaderList.AVIHeader.Header.dwReserved[0]         := 0;
    RIFF.HeaderList.AVIHeader.Header.dwReserved[1]         := 0;
    RIFF.HeaderList.AVIHeader.Header.dwReserved[2]         := 0;
    RIFF.HeaderList.AVIHeader.Header.dwReserved[3]         := 0;

    // Build the video stream list

    StrLV.Chunk.FourCC                                     := CC_LIST;
    StrLV.Chunk.dwSize                                     := SizeOf(StrLV) - SizeOf(TChunk);
    StrLV.strl                                             := CC_strl;
    
    StrLV.StreamHeader.Chunk.FourCC                        := CC_strh;
    StrLV.StreamHeader.Chunk.dwSize                        := SizeOf(StrLV.StreamHeader) - SizeOf(TChunk);
    StrLV.StreamHeader.StreamHeader.fccType                := CC_vids;
    If Comp = acNone
     Then StrLV.StreamHeader.StreamHeader.fccHandler       := CC_DIB
     Else StrLV.StreamHeader.StreamHeader.fccHandler       := CC_RLE;
    StrLV.StreamHeader.StreamHeader.dwFlags                := 0;
    StrLV.StreamHeader.StreamHeader.wPriority              := 0;
    StrLV.StreamHeader.StreamHeader.wLanguage              := 0;
    StrLV.StreamHeader.StreamHeader.dwInitialFrames        := 0;
    StrLV.StreamHeader.StreamHeader.dwScale                := 100;  { 60 frames/sec }
    StrLV.StreamHeader.StreamHeader.dwRate                 := 6000;
    StrLV.StreamHeader.StreamHeader.dwStart                := 0;
    StrLV.StreamHeader.StreamHeader.dwLength               := NumFrames;      { Will fix on close }
    StrLV.StreamHeader.StreamHeader.dwSuggestedBufferSize  := 120000;
    StrLV.StreamHeader.StreamHeader.dwQuality              := 0;
    StrLV.StreamHeader.StreamHeader.dwSampleSize           := 0;
    StrLV.StreamHeader.StreamHeader.rcFrame.Left           := 0;
    StrLV.StreamHeader.StreamHeader.rcFrame.Top            := 0;
    If Expand = exHorz2x
     Then StrLV.StreamHeader.StreamHeader.rcFrame.Right    := Width * 2
     Else StrLV.StreamHeader.StreamHeader.rcFrame.Right    := Width;
    StrLV.StreamHeader.StreamHeader.rcFrame.Bottom         := Height;

    StrLV.StreamFormat.Chunk.FourCC                        := CC_strf;
    StrLV.StreamFormat.Chunk.dwSize                        := SizeOf(StrLV.StreamFormat) - SizeOf(TChunk);
    StrLV.StreamFormat.BI.bmiHeader.biSize                 := SizeOf(BITMAPINFOHEADER);
    If Expand = exHorz2x
     Then StrLV.StreamFormat.BI.bmiHeader.biWidth          := Width * 2
     Else StrLV.StreamFormat.BI.bmiHeader.biWidth          := Width;
    StrLV.StreamFormat.BI.bmiHeader.biHeight               := Height;
    StrLV.StreamFormat.BI.bmiHeader.biPlanes               := 1;
    StrLV.StreamFormat.BI.bmiHeader.biBitCount             := 8;

    If Comp = acNone
     Then StrLV.StreamFormat.BI.bmiHeader.biCompression          := BI_RGB
     Else StrLV.StreamFormat.BI.bmiHeader.biCompression          := BI_RLE8;
    StrLV.StreamFormat.BI.bmiHeader.biSizeImage            := 0;
    StrLV.StreamFormat.BI.bmiHeader.biXPelsPerMeter        := 0;
    StrLV.StreamFormat.BI.bmiHeader.biYPelsPerMeter        := 0;
    StrLV.StreamFormat.BI.bmiHeader.biClrUsed              := 256;
    StrLV.StreamFormat.BI.bmiHeader.biClrImportant         := 256;
    StrLV.StreamFormat.DVInfo.dwDVAAuxSrc                  := 0;
    StrLV.StreamFormat.DVInfo.dwDVAAuxCtl                  := 0;
    StrLV.StreamFormat.DVInfo.dwDVAAuxSrc1                 := 0;
    StrLV.StreamFormat.DVInfo.dwDVAAuxCtl1                 := 0;
    StrLV.StreamFormat.DVInfo.dwDVVAuxSrc                  := 0;
    StrLV.StreamFormat.DVInfo.dwDVVAuxCtl                  := 0;
    StrLV.StreamFormat.DVInfo.dwDVReserved[0]              := 0;
    StrLV.StreamFormat.DVInfo.dwDVReserved[1]              := 0;
    FillChar(StrLV.StreamFormat.BI.bmiColors,SizeOf(StrLV.StreamFormat.BI.bmiColors),#0);
    For I := 0 To 127 Do
    Begin
      StrLV.StreamFormat.BI.bmiColors[I * 2 + 0].R  := Palette[I].R;
      StrLV.StreamFormat.BI.bmiColors[I * 2 + 0].G  := Palette[I].G;
      StrLV.StreamFormat.BI.bmiColors[I * 2 + 0].B  := Palette[I].B;
      StrLV.StreamFormat.BI.bmiColors[I * 2 + 1].R  := Palette[I].R;
      StrLV.StreamFormat.BI.bmiColors[I * 2 + 1].G  := Palette[I].G;
      StrLV.StreamFormat.BI.bmiColors[I * 2 + 1].B  := Palette[I].B;
    End; { For I }

    BlockWrite(F,RIFF,SizeOf(RIFF));
    BlockWrite(F,StrLV,SizeOf(StrLV));

    // Build the audio stream list

    If Sound Then
    Begin
      StrLA.Chunk.FourCC                                     := CC_LIST;
      StrLA.Chunk.dwSize                                     := SizeOf(StrLA) - SizeOf(TChunk);
      StrLA.strl                                             := CC_strl;
      
      StrLA.StreamHeader.Chunk.FourCC                        := CC_strh;
      StrLA.StreamHeader.Chunk.dwSize                        := SizeOf(StrLA.StreamHeader) - SizeOf(TChunk);
      StrLA.StreamHeader.StreamHeader.fccType                := CC_auds;
      StrLA.StreamHeader.StreamHeader.fccHandler             := AVI.CC_none;
      StrLA.StreamHeader.StreamHeader.dwFlags                := 0;
      StrLA.StreamHeader.StreamHeader.wPriority              := 0;
      StrLA.StreamHeader.StreamHeader.wLanguage              := 0;
      StrLA.StreamHeader.StreamHeader.dwInitialFrames        := 0;
      StrLA.StreamHeader.StreamHeader.dwScale                := 1;     
      StrLA.StreamHeader.StreamHeader.dwRate                 := AVISampleRate;
      StrLA.StreamHeader.StreamHeader.dwStart                := 0;
      StrLA.StreamHeader.StreamHeader.dwLength               := NumFrames;      { Will fix on close }
      StrLA.StreamHeader.StreamHeader.dwSuggestedBufferSize  := 4272;
      StrLA.StreamHeader.StreamHeader.dwQuality              := 0;
      StrLA.StreamHeader.StreamHeader.dwSampleSize           := 1;
      StrLA.StreamHeader.StreamHeader.rcFrame.Left           := 0;
      StrLA.StreamHeader.StreamHeader.rcFrame.Top            := 0;
      StrLA.StreamHeader.StreamHeader.rcFrame.Right          := 0;
      StrLA.StreamHeader.StreamHeader.rcFrame.Bottom         := 0;

      StrLA.StreamFormat.Chunk.FourCC                        := CC_strf;
      StrLA.StreamFormat.Chunk.dwSize                        := SizeOf(StrLA.StreamFormat) - SizeOf(TChunk);
      StrLA.StreamFormat.waveFmt.wFormatTag                  := 1; // 8-bit
      StrLA.StreamFormat.waveFmt.nChannels                   := 1; // Mono
      StrLA.StreamFormat.waveFmt.nSamplesPerSec              := AVISampleRate;
      StrLA.StreamFormat.waveFmt.nAvgBytesPerSec             := AVISampleRate;
      StrLA.StreamFormat.waveFmt.nBlockAlign                 := 1;
      StrLA.StreamFormat.waveFmt.wBitsPerSample              := 8;
      StrLA.StreamFormat.waveFmt.cbSize                      := 0;

      BlockWrite(F,StrLA,SizeOf(StrLA));
    End;

    MovI.Chunk.FourCC := CC_List;
    MovI.Chunk.dwSize := 0;            { Will fix on close }
    MovI.movi         := CC_movi;
    BlockWrite(F,MovI,SizeOf(MovI));
  End;
End; { TAVIFile.Open }

Procedure TAVIFile.Close;
Var
  FPIndex : LongInt;
  Posn    : LongInt;
  Frame   : Integer;
  FP0     : LongInt;
  FP1     : LongInt;
  Index   : TIndex;
  Blocks  : Integer;

Begin
  If Opened Then
  Begin
    // Write frame index header

    DV.FourCC := CC_idx1;
    If Sound
     Then DV.dwSize := NumFrames * SizeOf(TIndex) * 2
     Else DV.dwSize := NumFrames * SizeOf(TIndex);
    BlockWrite(F,DV,SizeOf(DV));

    // Get the current file position

    FPIndex := FilePos(F);

    // Now go back to the top of the file and correct the header values

    Seek(F,0);

    // We now know the frame count and the size of the RIFF block

    If Sound
     Then RIFF.Chunk.dwSize := FPIndex + 2 * NumFrames * SizeOf(TIndex) - SizeOf(TChunk)
     Else RIFF.Chunk.dwSize := FPIndex +     NumFrames * SizeOf(TIndex) - SizeOf(TChunk);

    RIFF.HeaderList.AVIHeader.Header.dwTotalFrames := NumFrames;
    BlockWrite(F,RIFF,SizeOf(RIFF));

    // We now know how many video frames were written

    StrLV.StreamHeader.StreamHeader.dwLength := NumFrames;
    BlockWrite(F,StrLV,SizeOf(StrLV));

    If Sound Then
    Begin
      // We now know how many audio samples were written

      StrLA.StreamHeader.StreamHeader.dwLength := NumFrames * AVISamplesPerFrame;
      BlockWrite(F,StrLA,SizeOf(StrLA));
    End;

    // We now know the size of the "movi" block

    FP1 := FilePos(F);
    MovI.Chunk.dwSize := FPIndex - FP1 - 2 * SizeOf(TChunk);
    BlockWrite(F,MovI,SizeOf(MovI));

    // Walk through the frames and write the index entries

    Posn   := 4;
    Blocks := NumFrames;
    If Sound Then Blocks := Blocks * 2;
    For Frame := 1 To Blocks Do
    Begin
      // Get the file position of the frame

      FP1 := FilePos(F);
      FP0 := FP1;

      // Read the frame's video chunk

      BlockRead(F,DV,SizeOf(DV));

      // Calculate the file position of the next frame

      Inc(FP1,DV.dwSize + SizeOf(DV));

      // Seek down to the frame's index entry

      Seek(F,FPIndex);

      // Populate the index record and write it

      If Sound And ((Frame And 1) = 0) Then Index.FourCC := CC_01wb
      Else If Comp = acNone Then Index.FourCC := CC_00db
      Else Index.FourCC := CC_00dc;

      Index.L1     := AVIIF_KEYFRAME;//0;
      Index.L2     := FP0;//Posn;
      Index.L3     := DV.dwSize;
      BlockWrite(F,Index,SizeOf(Index));

      // Calculate the next frame's index position

      Inc(Posn,DV.dwSize + SizeOf(DV));

      // Save the file position of the next frame's index entry

      FPIndex := FilePos(F);

      // Seek up to the next frame

      Seek(F,FP1);
    End; { For Frame }

    System.Close(F);
    FreeMem(RLE,Width * 2 + 2);
    Opened := False;
    FreeMem(SoundBuf);
    SoundBuf := Nil;
  End;
End; { TAVIFile.Close }

Procedure TAVIFile.WriteFrame(VideoBuf,AudioBuf: Pointer; AudioBufLast, AudioBufPos, AudioBufEnd: Integer; VFlip: Boolean);
Type BPtr = ^Byte;
Var
  Line     : LongInt;
  RLERec   : TRLERec;
  RLELen   : LongInt;
  FP1      : LongInt;
  FP2      : LongInt;
  P1       : Pointer;

  Procedure WriteSoundFrame;
  Var
    Available : Integer;
    Block1    : Integer;
    Block2    : Integer;

  Begin
    DA.FourCC := CC_01wb;
    DA.dwSize := AVISamplesPerFrame;

    If AudioBufLast > AudioBufPos Then
    Begin
      // We wrapped the write buffer

      Available := AudioBufPos + AudioBufEnd - AudioBufLast;
      If Available > 0 Then
      Begin
        Block1 := Min(AudioBufEnd - AudioBufLast, AVISamplesPerFrame);
        Block2 := Min(AudioBufPos, AVISamplesPerFrame - Block1);
        If Block1 > 0 Then
        Begin
          Move(Pointer(LongWord(AudioBuf) + AudioBufLast)^, SoundBuf^, Block1);
          If Block2 > 0 Then
          Begin
            Move(AudioBuf^, Pointer(LongWord(SoundBuf) + Block1)^, Block2);
          End;
        End;
      End;
    End
    Else
    Begin
      Available := AudioBufPos - AudioBufLast;
      If Available > 0 Then
      Begin
        Move(Pointer(LongWord(AudioBuf) + AudioBufLast)^, SoundBuf^, Min(Available, AVISamplesPerFrame));
      End;
    End;

    BlockWrite(F,DA,SizeOf(DA));
    BlockWrite(F, SoundBuf^, AVISamplesPerFrame);
  End;

  Function CompressLine(Line: LongInt): LongInt;
  { ------------------------------------------------------------------------------- }
  { Note that this procedure is written specifically for the Atari 2600 screen and  }
  { (1) takes advantage of the fact that odd and even pixels contain equal values   }
  {     for 320-column modes, and                                                   }
  { (2) doubles the count for 160-column modes.                                     }
  { ------------------------------------------------------------------------------- }
  Var
    Addr  : LongInt;
    Dest  : LongInt;
    P1    : BPtr;
    P2    : PRLERec;
    Wid   : LongInt;
    Expd  : TAVIExpand;

  Begin
    Dest := 0;
    Addr := Line * Width;
    P1   := VideoBuf;               { Make local copies to the stack }
    P2   := RLE;
    Wid  := Width;
    Expd := Expand;

    Asm
      CLD
      PUSHAD
      MOV   EDX,DWORD PTR Wid       { Copy the line width }
      MOV   EDI,DWORD PTR P1        { Copy the buffer address }
      ADD   EDI,DWORD PTR Addr
      MOV   ESI,DWORD PTR P2        { Copy the address of the destination buffer }
      CMP   BYTE PTR Expd,exHorz2x  { Do we need to double up the pixels? }
      JE    @Expand2x

{ =============================================================================== }
{ Handle 320-width modes                                                          }
{ =============================================================================== }

@L1:
      MOV   AX,WORD PTR [EDI]
      MOV   ECX,127
      CMP   ECX,EDX
      JBE   @NoCopy
      MOV   ECX,EDX
@NoCopy:
      MOV   EBX,EDI
      REPE  SCASW
      JE    @NoDec
      SUB   EDI,2
@NoDec:
      MOV   ECX,EDI
      SUB   ECX,EBX
      SUB   EDX,ECX
      MOV   AL,CL
      MOV   WORD PTR [ESI],AX
      ADD   ESI,2
      ADD   DWORD PTR Dest,2
      CMP   EDX,0
      JG    @L1
      JMP   @GetOut

{ =============================================================================== }
{ Handle 160-width modes: look at every pixel but count each twice                }
{ =============================================================================== }

@Expand2x:
      MOV   AL,BYTE PTR [EDI]
      MOV   AH,AL
      MOV   ECX,127
      CMP   ECX,EDX
      JBE   @NoCopy1
      MOV   ECX,EDX
@NoCopy1:
      MOV   EBX,EDI
      REPE  SCASB
      JE    @NoDec1
      DEC   EDI
@NoDec1:
      MOV   ECX,EDI
      SUB   ECX,EBX
      SUB   EDX,ECX
      ADD   ECX,ECX
      MOV   AL,CL
      MOV   WORD PTR [ESI],AX
      ADD   ESI,2
      ADD   DWORD PTR Dest,2
      CMP   EDX,0
      JG    @Expand2x
@GetOut:
      MOV   WORD PTR [ESI],0
      POPAD
    End; { Asm }
    CompressLine := Dest;
  End; { CompressLine }

  Procedure ExpandLine(Line: LongInt);
  Var
    Addr  : LongInt;
    P1    : BPtr;
    P2    : BPtr;
    Wid   : LongInt;

  Begin
    Addr := Line * Width;
    P1   := VideoBuf;               { Make local copies to the stack }
    P2   := RLE;
    Wid  := Width;
    Asm
      CLD
      PUSHAD
      MOV   ECX,DWORD PTR Wid       { Copy the line width }
      MOV   EDI,DWORD PTR P2        { Copy the address of the destination buffer }
      MOV   ESI,DWORD PTR P1        { Copy the buffer address }
      ADD   ESI,DWORD PTR Addr
      TEST  ECX,1
      JZ    @Even
@L1:
      LODSB
      MOV   AH,AL
      STOSW
      DEC   ECX
      JNZ   @L1
      JMP   @GetOut
@Even:
      SHR   ECX,1
@L2:
      SUB   EAX,EAX
      LODSW
      MOV   EBX,EAX
      MOV   AH,AL
      MOV   BL,BH
      SHL   EBX,16
      OR    EAX,EBX
      STOSD
      DEC   ECX
      JNZ   @L2
@GetOut:
      POPAD
    End; { Asm }
  End; { ExpandLine }

Begin
  { Write screen data }

  FP1 := FilePos(F);
  If Comp = acNone
   Then DV.FourCC := CC_00db
   Else DV.FourCC := CC_00dc;
  If Comp = acRLE Then
  Begin
    // RLE compression
    
    DV.dwSize := 0;                         { Will change afterwards }
    BlockWrite(F,DV,SizeOf(DV));
    If VFlip Then
    Begin
      For Line := Height - 1 DownTo 0 Do
      Begin
        RLELen := CompressLine(Line);         { Compression routine automatically places }
        BlockWrite(F,RLE^,RLELen + 2);        {  an end-of-line specifier afterward }
      End; { For Line }
    End
    Else
    Begin
      For Line := 0 To Height - 1 Do
      Begin
        RLELen := CompressLine(Line);         { Compression routine automatically places }
        BlockWrite(F,RLE^,RLELen + 2);        {  an end-of-line specifier afterward }
      End; { For Line }
    End;
    RLERec.Count := 0;
    RLERec.Num   := 1;                      { End-of-frame specifier }
    BlockWrite(F,RLERec,SizeOf(RLERec));

    { Correct the frame length specifier }

    FP2 := FilePos(F);
    Seek(F,FP1);
    DV.dwSize := FP2 - FP1 - SizeOf(DV);
    BlockWrite(F,DV,SizeOf(DV));
    Seek(F,FP2);
  End
  Else
  Begin
    // Uncompressed

    DV.dwSize := Height * Width;            { Will change afterwards }
    If Expand = exHorz2x Then DV.dwSize := DV.dwSize * 2;
    BlockWrite(F,DV,SizeOf(DV));
    If VFlip Then
    Begin
      For Line := Height - 1 DownTo 0 Do
      Begin
        If Expand = exHorz2x Then
        Begin
          ExpandLine(Line);
          BlockWrite(F,RLE^,Width * 2);
        End
        Else
        Begin
          P1 := VideoBuf;
          Inc(LongInt(P1),Line * Width);
          BlockWrite(F,P1^,Width);
        End;
      End;
    End
    Else
    Begin
      For Line := 0 To Height - 1 Do
      Begin
        If Expand = exHorz2x Then
        Begin
          ExpandLine(Line);
          BlockWrite(F,RLE^,Width * 2);
        End
        Else
        Begin
          P1 := VideoBuf;
          Inc(LongInt(P1),Line * Width);
          BlockWrite(F,P1^,Width);
        End;
      End;
    End;
  End;

  If Sound Then WriteSoundFrame;

  Inc(NumFrames);
End; { TAVIFile.WriteFrame }

End.

// ----------------------------------------------------------------------
// PCAE and PCAEWin - PC Atari Emulator - Atari 2600 emulator
// Copyright (C) 2000 John Dullea
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
// ----------------------------------------------------------------------