unit DXInput;

interface

uses
  Windows, Messages, SysUtils, Classes, Controls, Forms, MMSystem,
  DirectX, DXClass;

type

  {  EDXInputError  }

  EDXInputError = class(Exception);

  {  EForceFeedbackEffectError  }

  EForceFeedbackEffectError = class(Exception);

  {  TForceFeedbackEffect  }

  TForceFeedbackEffectType = (etNone, etConstantForce, etPeriodic, etCondition);

  TForceFeedbackEffect = class;
  TForceFeedbackEffects = class;

  TForceFeedbackEffectObject = class
  private
    FAxes: array[0..1] of DWORD;
    FAxesCount: Integer;
    Feff: DIEFFECT;
    FDirections: array[0..1] of DWORD;
    FEnvelope: DIENVELOPE;
    FConstantForce: DICONSTANTFORCE;
    FCondition: DICONDITION;
    FPeriodic: DIPERIODIC;
    FEffect: IDirectInputEffect;
    procedure Clear;
    procedure Init(Effect: TForceFeedbackEffect);
    procedure Release;
  public
    destructor Destroy; override;
  End;

  TForceFeedbackEffect = class(TPersistent)
  private
    FRoot: TForceFeedbackEffects;                 
    FParent: TForceFeedbackEffect;
    FList: TList;
    FAttackLevel: Integer;
    FAttackTime: Integer;
    FCondition: TPoint;
    FConstant: TPoint;
    FEffectType: TForceFeedbackEffectType;
    FFadeLevel: Integer;
    FFadeTime: Integer;
    FName: string;
    FPeriod: Integer;
    FPlaying: Boolean;
    FPower: Integer;
    FTime: Integer;
    FObject: TForceFeedbackEffectObject;
    FObject2: TForceFeedbackEffectObject;
    FFindEffectFlag: Boolean;
    FFindEffectGUID: TGUID;
    procedure Acquire;
    procedure Finalize;
    procedure Initialize;
    procedure ChangeEffect;
    procedure MakeEff;
    procedure CreateEffect;
    function GetCount: Integer;
    function GetEffect(Index: Integer): TForceFeedbackEffect;
    function GetIndex: Integer;
    function GetPlaying: Boolean;
    procedure SetAttackLevel(Value: Integer);
    procedure SetAttackTime(Value: Integer);
    procedure SetCondition(Value: TPoint);
    procedure SetConstant(Value: TPoint);
    procedure SetEffectType(Value: TForceFeedbackEffectType);
    procedure SetFadeLevel(Value: Integer);
    procedure SetFadeTime(Value: Integer);
    procedure SetIndex(Value: Integer);
    procedure SetPeriod(Value: Integer);
    procedure SetParent(Value: TForceFeedbackEffect);
    procedure SetPower(Value: Integer);
    procedure SetTime(Value: Integer);
    function HasInterface: Boolean;
  protected
    function GetOwner: TPersistent; override;
  public
    constructor Create(AParent: TForceFeedbackEffect);
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
    procedure Clear;
    function Find(const Name: string): TForceFeedbackEffect;
    function IndexOf(const Name: string): Integer;
    procedure LoadFromFile(const FileName: string);
    procedure LoadFromStream(Stream: TStream);
    procedure SaveToFile(const FileName: string);
    procedure SaveToStream(Stream: TStream);
    procedure Start;
    procedure Stop;
    procedure Unload(Recurse: Boolean);
    property Count: Integer read GetCount;
    property Effects[Index: Integer]: TForceFeedbackEffect read GetEffect; default;
    property Index: Integer read GetIndex write SetIndex;
    property Playing: Boolean read GetPlaying;
    property Parent: TForceFeedbackEffect read FParent write SetParent;
    property Name: string read FName write FName;
    property EffectType: TForceFeedbackEffectType read FEffectType write SetEffectType;
    property AttackLevel: Integer read FAttackLevel write SetAttackLevel;
    property AttackTime: Integer read FAttackTime write SetAttackTime;
    property Condition: TPoint read FCondition write SetCondition;
    property Constant: TPoint read FConstant write SetConstant;
    property FadeLevel: Integer read FFadeLevel write SetFadeLevel;
    property FadeTime: Integer read FFadeTime write SetFadeTime;
    property Period: Integer read FPeriod write SetPeriod;
    property Power: Integer read FPower write SetPower;
    property Time: Integer read FTime write SetTime;
  End;

  {  TForceFeedbackEffects  }

  TCustomInput = class;

  TForceFeedbackEffects = class(TForceFeedbackEffect)
  private
    FComponent: TComponent;
    FInput: TCustomInput;
  protected
    procedure DefineProperties(Filer: TFiler); override;
  public
    constructor Create(Input: TCustomInput);
    destructor Destroy; override;
    property Input: TCustomInput read FInput;
  End;

  {  TCustomInput  }

  TDXInputState = (isUp, isDown, isLeft, isRight, isButton1, isButton2, isButton3,
    isButton4, isButton5, isButton6, isButton7, isButton8, isButton9, isButton10, isButton11,
    isButton12, isButton13, isButton14, isButton15, isButton16, isButton17, isButton18,
    isButton19, isButton20, isButton21, isButton22, isButton23, isButton24, isButton25,
    isButton26, isButton27, isButton28, isButton29, isButton30, isButton31, isButton32);

  TDXInputStates = set of TDXInputState;

  TCustomDXInputBase = Class;

  TCustomInput = class(TPersistent)
  private
    FButtonCount: Integer;
    FDataFormat: DIDATAFORMAT;
    FDataFormatObjects: array[0..255] of DIOBJECTDATAFORMAT;
    FDataFormatGUIDs: array[0..255] of TGUID;
    FDevice: IDirectInputDevice;
    FDevice2: IDirectInputDevice2;
    FDXInput: TCustomDXInputBase;
    FEffects: TForceFeedbackEffects;
    FEnabled: Boolean;
    FForceFeedback: Boolean;
    FForceFeedbackDevice: Boolean;
    FStates: TDXInputStates;
    procedure Acquire;
    procedure Finalize; virtual;
    procedure Initialize; virtual;
    function GetButton(Index: Integer): Boolean;
    function GetCooperativeLevel: Integer; virtual;
    function GetDeviceState(dwSize: Integer; var Data): Boolean;
    function SetDataFormat: Boolean;
    procedure SetEffects(Value: TForceFeedbackEffects);
    procedure SetEnabled(Value: Boolean);
    procedure SetForceFeedback(Value: Boolean);
    procedure SetWindowHandle(Value: Integer);
  public
    constructor Create(DXInput: TCustomDXInputBase); virtual;
    destructor Destroy; override;
    procedure Update; virtual; abstract;
    property ButtonCount: Integer read FButtonCount;
    property Buttons[Index: Integer]: Boolean read GetButton;
    property States: TDXInputStates read FStates;
  published
    property Effects: TForceFeedbackEffects read FEffects write SetEffects;
    property Enabled: Boolean read FEnabled write SetEnabled;
    property ForceFeedback: Boolean read FForceFeedback write SetForceFeedback;
  End;

  {  TKeyboard  }

  PKeyAssign = ^TKeyAssign;
  TKeyAssign = array[0..2] of Integer;

  TKeyAssignList = array[TDXInputState] of TKeyAssign;

  TKeyboard = class(TCustomInput)
  private
    FKeyStates : TKeyboardState;
    dikb       : DIKeyboardState;
    procedure   Initialize; override;
    function    GetKey(Key: Integer): Boolean;
    function    GetDIKey(Key: Integer): Boolean;
    procedure   ReadAssigns(Stream: TStream);
    procedure   WriteAssigns(Stream: TStream);
  protected
    procedure   DefineProperties(Filer: TFiler); override;
  public
    KeyAssigns: TKeyAssignList;
    constructor Create(DXInput: TCustomDXInputBase); override;
    procedure   Update; override;
    property    Keys[Key: Integer]: Boolean read GetKey;
    property    DIKeys[Key: Integer]: Boolean read GetDIKey;
  End;

  // TJoystickBase

  TJoystickBase = class(TCustomInput)
  private
    Fdijs       : DIJOYSTATE2;
    FAutoCenter : Boolean;
    FDeviceGUID : TGUID;
    FEnumFlag   : Boolean;
    FID         : Integer;
    FID2        : Integer;
    FJoyCaps    : TJoyCaps;
    FDeadZone   : Array[0..SizeOf(DIJOYSTATE2) - 1] Of DWORD;
    FRange      : Array[0..SizeOf(DIJOYSTATE2) - 1] Of DWORD;
    function  GetCooperativeLevel: Integer; Override;
    function  GetDeadZone(Obj: Integer): Integer;
    function  GetRange(Obj: Integer): Integer;
    function  GetX: Integer;
    function  GetY: Integer;
    function  GetZ: Integer;
    procedure SetAutoCenter(Value: Boolean);
    procedure SetDeadZone(Obj: Integer; Value: Integer);
    procedure SetRange(Obj: Integer; Value: Integer);
  public
    procedure Update; Override;
    property  DeadZone[Obj: Integer]: Integer read GetDeadZone write SetDeadZone;
    property  Joystate: DIJOYSTATE2 read Fdijs;
    property  Range[Obj: Integer]: Integer read GetRange write SetRange;
    property  X: Integer read GetX;
    property  Y: Integer read GetY;
    property  Z: Integer read GetZ;
  published
    property  AutoCenter: Boolean read FAutoCenter write SetAutoCenter;
    property  DeadZoneX: Integer index DIJOFS_X read GetDeadZone write SetDeadZone;
    property  DeadZoneY: Integer index DIJOFS_Y read GetDeadZone write SetDeadZone;
    property  DeadZoneZ: Integer index DIJOFS_Z read GetDeadZone write SetDeadZone;
    property  RangeX: Integer index DIJOFS_X read GetRange write SetRange;
    property  RangeY: Integer index DIJOFS_Y read GetRange write SetRange;
    property  RangeZ: Integer index DIJOFS_Z read GetRange write SetRange;
  End;

  // TJoystick

  TJoystick = class(TJoystickBase)
  private
    FEnumIndex : Integer;
    procedure   Finalize; Override;
    procedure   Initialize; Override;
    procedure   SetID(Value: Integer);
  public
    constructor Create(DXInput: TCustomDXInputBase); Override;
  published
    property    ID: Integer read FID write SetID;
  end;

  // TJoystick2

  TJoystick2 = class(TJoystickBase)
  private
    FProductName : String;
    FNumAxes     : Integer;
    FNumPOVs     : Integer;
    procedure   Finalize; Override;
    procedure   Initialize; Override;
    function    GetU: Integer;
    function    GetV: Integer;
    function    GetW: Integer;
    Function    GetPOV(Obj: Integer): Integer;
  public
    constructor Create(DXInput: TCustomDXInputBase); Override;
    property    U: Integer read GetU;
    property    V: Integer read GetV;
    property    W: Integer read GetW;
    property    POV[Obj: Integer]: Integer read GetPOV;
    property    ProductName: String read FProductName;
    property    AxisCount: Integer read FNumAxes;
    property    POVCount: Integer read FNumPOVs;
  End;

  // TCustomDXInputBase

  TCustomDXInputBase = Class(TComponent)
  private
    FActiveOnly     : Boolean;
    FDevice         : TList;
    FDInput         : IDirectInput;
    FForm           : TCustomForm;
    FKeyboard       : TKeyboard;
    FOldStates      : TDXInputStates;
    FStates         : TDXInputStates;
    FSubClass       : TControlSubClass;
    FUseDirectInput : Boolean;
    procedure Initialize; Dynamic; Abstract;
    procedure Finalize;
    procedure FormWndProc(var Message: TMessage; DefWindowProc: TWndMethod);
    procedure SetActiveOnly(Value: Boolean);
    procedure SetKeyboard(Value: TKeyboard);
    procedure SetUseDirectInput(Value: Boolean);
    procedure SetWindowHandle; 
  protected
    procedure Loaded; Override;
  public
    procedure Update;
    property  ActiveOnly     : Boolean read FActiveOnly write SetActiveOnly;
    property  Keyboard       : TKeyboard read FKeyboard write SetKeyboard;
    property  States         : TDXInputStates read FStates write FStates;
    property  UseDirectInput : Boolean read FUseDirectInput write SetUseDirectInput;
  End;

  // TCustomDXInput

  TCustomDXInput = class(TCustomDXInputBase)
  private
    FJoystick : TJoystick;
    procedure   Initialize; Override;
    procedure   SetJoystick(Value: TJoystick);
  public
    constructor Create(AOwner: TComponent); Override;
    destructor  Destroy; Override;
    property    Joystick: TJoystick read FJoystick write SetJoystick;
  end;

  // TCustomDXInput2

  TCustomDXInput2 = class(TCustomDXInputBase)
  private
    FJoysticks    : TList;
    FNumJoysticks : Integer;
    procedure   Initialize; Override;
    function    GetJoystick(Index: Integer): TJoystick2;
  public
    constructor Create(AOwner: TComponent); Override;
    destructor  Destroy; Override;
    property    Joystick[Index: Integer]: TJoystick2 read GetJoystick;
    property    NumJoysticks: Integer read FNumJoysticks;
  End;

  // TDXInput

  TDXInput = class(TCustomDXInput)
  published
    property ActiveOnly;
    property Joystick;
    property Keyboard;
    property UseDirectInput;
  end;

  // TDXInput2

  TDXInput2 = class(TCustomDXInput2)
  published
    property ActiveOnly;
    property Keyboard;
    property UseDirectInput;
  End;

function DefKeyAssign: TKeyAssignList;
function DefKeyAssign2_1: TKeyAssignList;
function DefKeyAssign2_2: TKeyAssignList;

implementation

uses DXConsts;

procedure AssignKey(var KeyAssignList: TKeyAssignList; State: TDXInputState;
  const Keys: array of Integer);
var
  i, i2: Integer;
  KeyAssign: PKeyAssign;
begin
  KeyAssign := @KeyAssignList[State];
  FillChar(KeyAssign^, SizeOf(TKeyAssign), 0);

  i2 := 0;
  for i:=LOW(Keys) to HIGH(Keys) do
  begin
    if i2<3 then
      KeyAssign^[i2] := Keys[i]
    else
      Exit;
    Inc(i2);
  End;
End;

function DefKeyAssign: TKeyAssignList;
begin
  FillChar(Result, SizeOf(Result), 0);

  AssignKey(Result, isUp,      [Ord('K'), VK_UP, VK_NUMPAD8]);
  AssignKey(Result, isDown,    [Ord('J'), VK_DOWN, VK_NUMPAD2]);
  AssignKey(Result, isLeft,    [Ord('H'), VK_LEFT, VK_NUMPAD4]);
  AssignKey(Result, isRight,   [Ord('L'), VK_RIGHT, VK_NUMPAD6]);
  AssignKey(Result, isButton1, [Ord('Z'), VK_SPACE]);
  AssignKey(Result, isButton2, [Ord('X'), VK_RETURN]);
  AssignKey(Result, isButton9, [VK_F2]);
End;

function DefKeyAssign2_1: TKeyAssignList;
begin
  FillChar(Result, SizeOf(Result), 0);

  AssignKey(Result, isUp,      [Ord('K'), VK_UP, VK_NUMPAD8]);
  AssignKey(Result, isDown,    [Ord('J'), VK_DOWN, VK_NUMPAD2]);
  AssignKey(Result, isLeft,    [Ord('H'), VK_LEFT, VK_NUMPAD4]);
  AssignKey(Result, isRight,   [Ord('L'), VK_RIGHT, VK_NUMPAD6]);
  AssignKey(Result, isButton1, [VK_SPACE , VK_NUMPAD0]);
  AssignKey(Result, isButton2, [VK_RETURN, VK_NUMPAD5]);
  AssignKey(Result, isButton9, [VK_F2]);
End;

function DefKeyAssign2_2: TKeyAssignList;
begin
  FillChar(Result, SizeOf(Result), 0);

  AssignKey(Result, isUp,      [Ord('E')]);
  AssignKey(Result, isDown,    [Ord('C')]);
  AssignKey(Result, isLeft,    [Ord('S')]);
  AssignKey(Result, isRight,   [Ord('F')]);
  AssignKey(Result, isButton1, [Ord('Z')]);
  AssignKey(Result, isButton2, [Ord('X')]);
  AssignKey(Result, isButton9, [VK_F2]);
End;

// ---------------------------------
// TForceFeedbackEffectObject
// ---------------------------------

destructor TForceFeedbackEffectObject.Destroy;
begin
  Release;
  inherited Destroy;
End;

function ConvertTime(i: Integer): Integer;
begin
  if i=-1 then Result := Integer(INFINITE) else Result := i*1000;
End;

procedure TForceFeedbackEffectObject.Clear;
begin
  FillChar(Feff, SizeOf(Feff), 0);
End;

procedure TForceFeedbackEffectObject.Init(Effect: TForceFeedbackEffect);
begin
  with FEnvelope do
  begin
    dwSize := SizeOf(FEnvelope);
    dwAttackLevel := Effect.FAttackLevel;
    dwAttackTime := Min(Effect.FAttackTime, Max(Effect.FTime, 0))*1000;
    dwFadeLevel := Effect.FFadeLevel;
    dwFadeTime := Min(Effect.FFadeTime, Max(Effect.FTime, 0))*1000;
  End;

  FillChar(Feff, SizeOf(Feff), 0);
  with Feff do
  begin
    dwSize := SizeOf(Feff);
    dwFlags := DIEFF_CARTESIAN or DIEFF_OBJECTOFFSETS;
    dwDuration := ConvertTime(Effect.FTime);
    dwSamplePeriod := 0;
    dwGain := Effect.FPower;
    dwTriggerButton := DIEB_NOTRIGGER;
    dwTriggerRepeatInterval := 0;
    cAxes := FAxesCount;
    rgdwAxes := @FAxes;
    rglDirection := @FDirections;
    lpEnvelope := @FEnvelope;
  End;
End;

procedure TForceFeedbackEffectObject.Release;
begin
  FEffect := nil;
End;

// ---------------------------------
// TForceFeedbackEffect
// ---------------------------------

constructor TForceFeedbackEffect.Create(AParent: TForceFeedbackEffect);
begin
  inherited Create;
  FParent := AParent;
  FList := TList.Create;

  if FParent<>nil then
  begin
    FParent.FList.Add(Self);
    FRoot := FParent.FRoot;
  end else
  begin
    FName := 'Effects';
    FRoot := Self as TForceFeedbackEffects;
  End;

  FObject := TForceFeedbackEffectObject.Create;
  FObject2 := TForceFeedbackEffectObject.Create;

  AttackTime := 0;
  Constant := Point(0, 0);
  EffectType := etNone;
  FadeTime := 0;
  Period := 50;
  Power := 10000;
  Time := 1000;
End;

destructor TForceFeedbackEffect.Destroy;
begin
  Clear;
  FObject.Free;
  FObject2.Free;
  FList.Free;
  if FParent<>nil then
    FParent.FList.Remove(Self);
  inherited Destroy;
End;

function TForceFeedbackEffect.GetOwner: TPersistent;
begin
  Result := Parent;
End;

procedure TForceFeedbackEffect.Assign(Source: TPersistent);
var
  i: Integer;
begin
  if Source is TForceFeedbackEffect then
  begin
    if Source<>Self then
    begin
      Clear;

      EffectType := etNone;

      Name := TForceFeedbackEffect(Source).Name;

      AttackLevel := TForceFeedbackEffect(Source).AttackLevel;
      AttackTime := TForceFeedbackEffect(Source).AttackTime;
      Constant := TForceFeedbackEffect(Source).Constant;
      Condition := TForceFeedbackEffect(Source).Condition;
      EffectType := TForceFeedbackEffect(Source).EffectType;
      FadeLevel := TForceFeedbackEffect(Source).FadeLevel;
      FadeTime := TForceFeedbackEffect(Source).FadeTime;
      Period := TForceFeedbackEffect(Source).Period;
      Power := TForceFeedbackEffect(Source).Power;
      Time := TForceFeedbackEffect(Source).Time;

      EffectType := TForceFeedbackEffect(Source).EffectType;

      for i:=0 to TForceFeedbackEffect(Source).Count-1 do
        TForceFeedbackEffect.Create(Self).Assign(TForceFeedbackEffect(Source)[i]);
    End;
  end else
    inherited Assign(Source);
End;

procedure TForceFeedbackEffect.Acquire;
var
  i: Integer;
begin
  if Playing and (Time=-1) then
    Start;

  for i:=0 to Count-1 do
    Effects[i].Initialize;
End;

procedure TForceFeedbackEffect.Clear;
begin
  while Count>0 do
    Effects[Count-1].Free;
End;

procedure TForceFeedbackEffect.Initialize;
var
  i: Integer;
begin
  CreateEffect;
  for i:=0 to Count-1 do
    Effects[i].Initialize;
End;

procedure TForceFeedbackEffect.Finalize;
var
  i: Integer;
begin
  try
    Stop;
    FObject.Release;
    FObject2.Release;
  finally
    for i:=0 to Count-1 do
      Effects[i].Finalize;
  End;
End;

function TForceFeedbackEffect.Find(const Name: string): TForceFeedbackEffect;
var
  i, p: Integer;
  Effect: TForceFeedbackEffect;
  AName: string;
begin
  AName := Name;
  Effect := Self;

  p := AnsiPos('.', AName);
  while p<>0 do
  begin
    i := Effect.IndexOf(AName);
    if i<>-1 then
    begin
      Result := Effect[i];
      Exit;
    end else
    begin
      i := Effect.IndexOf(Copy(Name, 1, p-1));
      if i=-1 then
        raise EForceFeedbackEffectError.CreateFmt(SEffectNotFound, [Name]);
      Effect := Effect[i];
      AName := Copy(Name, p+1, MaxInt);
      p := AnsiPos('.', AName);
    End;
  End;

  i := Effect.IndexOf(Name);
  if i=-1 then
    raise EForceFeedbackEffectError.CreateFmt(SEffectNotFound, [Name]);
  Result := Effect[i];
End;

function TForceFeedbackEffect.IndexOf(const Name: string): Integer;
var
  i: Integer;
begin
  Result := -1;
  for i:=0 to Count-1 do
    if Effects[i].Name=Name then
    begin
      Result := i;
      Break;
    End;
End;

function TForceFeedbackEffect.HasInterface: Boolean;
begin
  Result := (FEffectType<>etNone) and ((FObject.FEffect<>nil) or (FObject2.FEffect<>nil));
End;

procedure TForceFeedbackEffect.MakeEff;
var
  Constant2: TPoint;
begin
  FObject.Clear;
  FObject2.Clear;

  with Constant2 do
  begin
    X := -FConstant.X;
    Y := -FConstant.Y;
  End;

  case FEffectType of
    etConstantForce:  { etConstantForce }
        begin
          with FObject do
          begin
            FDirections[0] := Constant2.X;
            FDirections[1] := Constant2.Y;

            FAxesCount := 2;
            FAxes[0] := DIJOFS_X;
            FAxes[1] := DIJOFS_Y;

            with Constant2 do
              FConstantForce.lMagnitude := Trunc(Sqrt(X*X+Y*Y));

            Init(Self);
            with Feff do
            begin
              cbTypeSpecificParams := SizeOf(FConstantForce);
              lpvTypeSpecificParams := @FConstantForce;
            End;
          End;
        End;
    etPeriodic:       { etPeriodic }
        begin
          with FObject do
          begin
            FDirections[0] := Constant2.X;
            FDirections[1] := Constant2.Y;

            FAxesCount := 2;
            FAxes[0] := DIJOFS_X;
            FAxes[1] := DIJOFS_Y;

            with FPeriodic do
            begin
              with Constant2 do
                dwMagnitude := Trunc(Sqrt(X*X+Y*Y));
              lOffset := 0;
              dwPhase := 0;
              dwPeriod := ConvertTime(FPeriod);
            End;

            Init(Self);
            with Feff do
            begin
              cbTypeSpecificParams := SizeOf(FPeriodic);
              lpvTypeSpecificParams := @FPeriodic;
            End;
          End;
        End;
    etCondition:      { etCondition }
        begin
          with FObject do
          begin
            FillChar(FDirections, SizeOf(FDirections), 0);

            FAxesCount := 1;
            FAxes[0] := DIJOFS_X;

            with FCondition do
            begin
              lOffset := -Constant2.X;
              lPositiveCoefficient := Self.FCondition.X;
              lNegativeCoefficient := -Self.FCondition.X;
              dwPositiveSaturation := 0;
              dwNegativeSaturation := 0;
              lDeadBand := 0;
            End;

            Init(Self);
            with Feff do
            begin
              cbTypeSpecificParams := SizeOf(FCondition);
              lpvTypeSpecificParams := @FCondition;
            End;
          End;

          with FObject2 do
          begin
            FillChar(FDirections, SizeOf(FDirections), 0);

            FAxesCount := 1;
            FAxes[0] := DIJOFS_Y;

            with FCondition do
            begin
              lOffset := -Constant2.Y;
              lPositiveCoefficient := Self.FCondition.Y;
              lNegativeCoefficient := -Self.FCondition.Y;
              dwPositiveSaturation := 0;
              dwNegativeSaturation := 0;
              lDeadBand := 0;
            End;

            Init(Self);
            with Feff do
            begin
              cbTypeSpecificParams := SizeOf(FCondition);
              lpvTypeSpecificParams := @FCondition;
            End;
          End;
        End;
  End;
End;

procedure TForceFeedbackEffect.CreateEffect;

  function FindEffectCallBack(const pdei: DIEFFECTINFOA;
    pvRef: Pointer): HRESULT; stdcall;
  begin
    with TForceFeedbackEffect(pvRef) do
    begin
      FFindEffectFlag := True;
      FFindEffectGUID := pdei.guid;
    End;

    Result := DIENUM_STOP;
  End;

  procedure CreateIEffectGuid(const GUID: TGUID;
    EffectObject: TForceFeedbackEffectObject);
  begin
    if EffectObject.Feff.dwSize=0 then Exit;

    if FRoot.FInput.FDevice2<>nil then
      FRoot.FInput.FDevice2.CreateEffect(GUID, EffectObject.Feff, EffectObject.FEffect, nil);
  End;

  procedure CreateIEffect(dwFlags: DWORD;
    EffectObject: TForceFeedbackEffectObject);
  begin
    if EffectObject.Feff.dwSize=0 then Exit;

    if FRoot.FInput.FDevice2<>nil then
    begin
      FFindEffectFlag := False;
      FRoot.FInput.FDevice2.EnumEffects(@FindEffectCallBack,
        Self, dwFlags);
      if FFindEffectFlag then
        CreateIEffectGuid(FFindEffectGUID, EffectObject);
    End;
  End;

begin
  FObject.Release;
  FObject2.Release;

  if (FRoot.FInput=nil) or (FRoot.FInput.FDevice2=nil) or
    (not FRoot.FInput.FForceFeedbackDevice) or
    (not FRoot.FInput.FForceFeedback) then Exit;

  if FEffectType=etNone then Exit;

  MakeEff;
  case FEffectType of
    etConstantForce:
        begin
          CreateIEffectGUID(GUID_ConstantForce, FObject);
        End;
    etPeriodic:
        begin
          CreateIEffect(DIEFT_PERIODIC, FObject);
        End;
    etCondition:
        begin
          CreateIEffect(DIEFT_CONDITION, FObject);
          CreateIEffect(DIEFT_CONDITION, FObject2);
        End;
  End;

  if Playing and (Time=-1) then
    Start;
End;

procedure TForceFeedbackEffect.ChangeEffect;
var
  dwFlags: DWORD;
begin
  if HasInterface then
  begin
    MakeEff;

    dwFlags := DIEP_DIRECTION or DIEP_DURATION or DIEP_ENVELOPE or
      DIEP_GAIN or DIEP_SAMPLEPERIOD or DIEP_TRIGGERBUTTON or
      DIEP_TRIGGERREPEATINTERVAL or DIEP_TYPESPECIFICPARAMS;

    if Playing then
      dwFlags := dwFlags or DIEP_START;

    if FObject.FEffect<>nil then FObject.FEffect.SetParameters(FObject.Feff, dwFlags);
    if FObject2.FEffect<>nil then FObject2.FEffect.SetParameters(FObject2.Feff, dwFlags);
  End;
End;

function TForceFeedbackEffect.GetPlaying: Boolean;
var
  dwFlags: DWORD;
begin
  Result := False;

  if not FPlaying then Exit;

  if FPlaying and (FTime=-1) then
  begin
    Result := True;
    Exit;
  End;

  if FObject.FEffect<>nil then
  begin
    dwFlags := 0;
    FObject.FEffect.GetEffectStatus(dwFlags);
    if dwFlags and DIEGES_PLAYING<>0 then
    begin
      Result := True;
      Exit;
    End;
  End;

  if FObject2.FEffect<>nil then
  begin
    dwFlags := 0;
    FObject2.FEffect.GetEffectStatus(dwFlags);
    if dwFlags and DIEGES_PLAYING<>0 then
    begin
      Result := True;
      Exit;
    End;
  End;

  if not Result then
    FPlaying := False;
End;

function TForceFeedbackEffect.GetCount: Integer;
begin
  Result := FList.Count;
End;

function TForceFeedbackEffect.GetEffect(Index: Integer): TForceFeedbackEffect;
begin
  Result :=FList[Index];
End;

function TForceFeedbackEffect.GetIndex: Integer;
begin
  if FParent<>nil then
    Result := FParent.FList.IndexOf(Self)
  else
    Result := 0;
End;

procedure TForceFeedbackEffect.SetIndex(Value: Integer);
begin
  if FParent<>nil then
  begin
    FParent.FList.Remove(Self);
    FParent.FList.Insert(Value, Self);
  End;
End;

procedure TForceFeedbackEffect.SetParent(Value: TForceFeedbackEffect);
begin
  if Parent<>Value then
  begin
    if (Value=nil) or (FRoot<>Value.FRoot) then
      raise EForceFeedbackEffectError.CreateFmt(SCannotChanged, ['Parent']);

    FParent.FList.Remove(Self);
    FParent := Value;
    FParent.FList.Add(Self);
  End;
End;

procedure TForceFeedbackEffect.SetAttackLevel(Value: Integer);
begin
  if Value<0 then Value := 0;
  if Value>10000 then Value := 10000;

  if FAttackLevel<>Value then
  begin
    FAttackLevel := Value;
    ChangeEffect;
  End;
End;

procedure TForceFeedbackEffect.SetAttackTime(Value: Integer);
begin
  if Value<0 then Value := 0;

  if FAttackTime<>Value then
  begin
    FAttackTime := Value;
    ChangeEffect;
  End;
End;

procedure TForceFeedbackEffect.SetCondition(Value: TPoint);
begin
  with Value do
  begin
    if X<-10000 then X := -10000;
    if X>+10000 then X := +10000;

    if Y<-10000 then Y := -10000;
    if Y>+10000 then Y := +10000;
  End;

  if not CompareMem(@FCondition, @Value, SizeOf(FCondition)) then
  begin
    FCondition := Value;

    if HasInterface then
      ChangeEffect;
  End;
End;

procedure TForceFeedbackEffect.SetConstant(Value: TPoint);
begin
  with Value do
  begin
    if X<-10000 then X := -10000;
    if X>+10000 then X := +10000;

    if Y<-10000 then Y := -10000;
    if Y>+10000 then Y := +10000;
  End;

  if not CompareMem(@FConstant, @Value, SizeOf(FConstant)) then
  begin
    FConstant := Value;

    if HasInterface then
      ChangeEffect;
  End;
End;

procedure TForceFeedbackEffect.SetEffectType(Value: TForceFeedbackEffectType);
begin
  if FEffectType<>Value then
  begin
    FEffectType := Value;
    Stop;
    CreateEffect;
  End;
End;

procedure TForceFeedbackEffect.SetFadeLevel(Value: Integer);
begin
  if Value<0 then Value := 0;
  if Value>10000 then Value := 10000;

  if FFadeLevel<>Value then
  begin
    FFadeLevel := Value;
    ChangeEffect;
  End;
End;

procedure TForceFeedbackEffect.SetFadeTime(Value: Integer);
begin
  if Value<0 then Value := 0;

  if FFadeTime<>Value then
  begin
    FFadeTime := Value;
    ChangeEffect;
  End;
End;

procedure TForceFeedbackEffect.SetPeriod(Value: Integer);
begin
  if Value<0 then Value := 0;

  if FPeriod<>Value then
  begin
    FPeriod := Value;
    ChangeEffect;
  End;
End;

procedure TForceFeedbackEffect.SetPower(Value: Integer);
begin
  if Value<0 then Value := 0;
  if Value>10000 then Value := 10000;

  if FPower<>Value then
  begin
    FPower := Value;
    ChangeEffect;
  End;
End;

procedure TForceFeedbackEffect.SetTime(Value: Integer);
begin
  if (Value<>-1) and (Value<0) then Value := 0;

  if FTime<>Value then
  begin
    FTime := Value;
    Stop;
    ChangeEffect;
  End;
End;

procedure TForceFeedbackEffect.Start;

  procedure StartEffect(Effect: IDirectInputEffect);
  var
    hr: HRESULT;
  begin
    if Effect<>nil then
    begin
      hr := Effect.Start(1, 0);
      if (hr=DIERR_INPUTLOST) or (hr=DIERR_NOTACQUIRED) then
      begin
        FRoot.FInput.Acquire;
        Effect.Start(1, 0);
      End;
    End;
  End;

var
  i: Integer;
begin
  for i:=0 to Count-1 do
    Effects[i].Start;

  if not HasInterface then
  begin
    CreateEffect;
    if not HasInterface then Exit;
  End;

  StartEffect(FObject.FEffect);
  StartEffect(FObject2.FEffect);

  FPlaying := True;
End;

procedure TForceFeedbackEffect.Stop;
var
  i: Integer;
begin
  if Playing then
  begin
    FPlaying := False;
    if FObject.FEffect<>nil then FObject.FEffect.Stop;
    if FObject2.FEffect<>nil then FObject2.FEffect.Stop;
  End;

  for i:=0 to Count-1 do
    Effects[i].Stop;
End;

procedure TForceFeedbackEffect.Unload(Recurse: Boolean);
var
  i: Integer;
begin
  if Playing then
  begin
    if FObject.FEffect<>nil then FObject.FEffect.Stop;
    if FObject2.FEffect<>nil then FObject2.FEffect.Stop;
  End;

  if FObject.FEffect<>nil then FObject.FEffect.Unload;
  if FObject2.FEffect<>nil then FObject2.FEffect.Unload;

  if Recurse then
  begin
    for i:=0 to Count-1 do
      Effects[i].Unload(True);
  End;
End;

type
  TForceFeedbackEffectItem = class(TCollectionItem)
  private
    FName: string;
    FEffectType: TForceFeedbackEffectType;
    FAttackLevel: Integer;
    FAttackTime: Integer;
    FConditionX: Integer;
    FConditionY: Integer;
    FConstantX: Integer;
    FConstantY: Integer;
    FFadeLevel: Integer;
    FFadeTime: Integer;
    FPeriod: Integer;
    FPower: Integer;
    FTime: Integer;
    FEffects: TCollection;
    function GetStoredEffects: Boolean;
  public
    constructor Create(Collection: TCollection); override;
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
    procedure AssignTo(Dest: TPersistent); override;
  published
    property Name: string read FName write FName;
    property EffectType: TForceFeedbackEffectType read FEffectType write FEffectType;
    property AttackLevel: Integer read FAttackLevel write FAttackLevel default 0;
    property AttackTime: Integer read FAttackTime write FAttackTime default 0;
    property ConditionX: Integer read FConditionX write FConditionX default 0;
    property ConditionY: Integer read FConditionY write FConditionY default 0;
    property ConstantX: Integer read FConstantX write FConstantX default 0;
    property ConstantY: Integer read FConstantY write FConstantY default 0;
    property FadeLevel: Integer read FFadeLevel write FFadeLevel default 0;
    property FadeTime: Integer read FFadeTime write FFadeTime default 0;
    property Period: Integer read FPeriod write FPeriod;
    property Power: Integer read FPower write FPower;
    property Time: Integer read FTime write FTime;
    property Effects: TCollection read FEffects write FEffects stored GetStoredEffects;
  End;

  TForceFeedbackEffectComponent = class(TComponent)
  private
    FEffects: TCollection;
  published
    property Effects: TCollection read FEffects write FEffects;
  End;

constructor TForceFeedbackEffectItem.Create(Collection: TCollection);
begin
  inherited Create(Collection);
  FEffects := TCollection.Create(TForceFeedbackEffectItem);
End;

destructor TForceFeedbackEffectItem.Destroy;
begin
  FEffects.Free;
  inherited Destroy;
End;

procedure TForceFeedbackEffectItem.Assign(Source: TPersistent);
var
  Effect: TForceFeedbackEffect;
  i: Integer;
begin
  Effect := Source as TForceFeedbackEffect;

  FName := Effect.Name;
  FEffectType := Effect.EffectType;
  FAttackLevel := Effect.AttackLevel;
  FAttackTime := Effect.AttackTime;
  FConditionX := Effect.Condition.X;
  FConditionY := Effect.Condition.Y;
  FConstantX := Effect.Constant.X;
  FConstantY := Effect.Constant.Y;
  FFadeLevel := Effect.FadeLevel;
  FFadeTime := Effect.FadeTime;
  FPeriod := Effect.Period;
  FPower := Effect.Power;
  FTime := Effect.Time;

  for i:=0 to Effect.Count-1 do
    TForceFeedbackEffectItem.Create(FEffects).Assign(Effect[i]);
End;

procedure TForceFeedbackEffectItem.AssignTo(Dest: TPersistent);
var
  Effect: TForceFeedbackEffect;
  i: Integer;
begin
  Effect := Dest as TForceFeedbackEffect;

  Effect.EffectType := etNone;

  Effect.Name := FName;
  Effect.AttackLevel := FAttackLevel;
  Effect.AttackTime := FAttackTime;
  Effect.Condition := Point(FConditionX, FConditionY);
  Effect.Constant := Point(FConstantX, FConstantY);
  Effect.FadeLevel := FFadeLevel;
  Effect.FadeTime := FFadeTime;
  Effect.Period := FPeriod;
  Effect.Power := FPower;
  Effect.Time := FTime;

  Effect.EffectType := FEffectType;

  for i:=0 to FEffects.Count-1 do
    TForceFeedbackEffectItem(FEffects.Items[i]).AssignTo(TForceFeedbackEffect.Create(Effect));
End;

function TForceFeedbackEffectItem.GetStoredEffects: Boolean;
begin
  Result := FEffects.Count>0;
End;

procedure TForceFeedbackEffect.LoadFromFile(const FileName: string);
var
  Stream: TFileStream;
begin
  Stream := TFileStream.Create(FileName, fmOpenRead or fmShareDenyWrite);
  try
    LoadFromStream(Stream);
  finally
    Stream.Free;
  End;
End;

procedure TForceFeedbackEffect.LoadFromStream(Stream: TStream);
var
  Component: TForceFeedbackEffectComponent;
begin
  Clear;

  Component := TForceFeedbackEffectComponent(FRoot.FComponent);
  try
    Component.FEffects := TCollection.Create(TForceFeedbackEffectItem);
    Stream.ReadComponentRes(Component);
    TForceFeedbackEffectItem(Component.FEffects.Items[0]).AssignTo(Self);
  finally
    Component.FEffects.Free;
    Component.FEffects := nil;
  End;
End;

procedure TForceFeedbackEffect.SaveToFile(const FileName: string);
var
  Stream: TFileStream;
begin
  Stream := TFileStream.Create(FileName, fmCreate);
  try
    SaveToStream(Stream);
  finally
    Stream.Free;
  End;
End;

procedure TForceFeedbackEffect.SaveToStream(Stream: TStream);
var
  Component: TForceFeedbackEffectComponent;
begin
  Component := TForceFeedbackEffectComponent(FRoot.FComponent);
  try
    Component.FEffects := TCollection.Create(TForceFeedbackEffectItem);
    TForceFeedbackEffectItem.Create(Component.FEffects).Assign(Self);
    Stream.WriteComponentRes('DelphiXForceFeedbackEffect', Component);
  finally
    Component.FEffects.Free;
    Component.FEffects := nil;
  End;
End;

// ---------------------------------
// TForceFeedbackEffects
// ---------------------------------

constructor TForceFeedbackEffects.Create(Input: TCustomInput);
begin
  inherited Create(nil);
  FInput := Input;
  FComponent := TForceFeedbackEffectComponent.Create(nil);
End;

destructor TForceFeedbackEffects.Destroy;
begin
  FComponent.Free;
  inherited Destroy;
End;

procedure TForceFeedbackEffects.DefineProperties(Filer: TFiler);
begin
  inherited DefineProperties(Filer);
  Filer.DefineBinaryProperty('Effects', LoadFromStream, SaveToStream, True);
End;

// ---------------------------------
// TCustomInput
// ---------------------------------

Constructor TCustomInput.Create(DXInput: TCustomDXInputBase);
Begin
  Inherited Create;
  FDXInput := DXInput;
  FDXInput.FDevice.Add(Self);
  FEffects := TForceFeedbackEffects.Create(Self);
  FEnabled := True;
End; // TCustomInput.Create

Destructor TCustomInput.Destroy;
Begin
  Finalize;
  FEffects.Free;
  FDXInput.FDevice.Remove(Self);
  Inherited Destroy;
End; // TCustomInput.Destroy

Procedure TCustomInput.Acquire;
Begin
  If FDXInput.FActiveOnly And (GetForegroundWindow <> FDXInput.FForm.Handle) Then Exit;

  If FDevice <> Nil Then FDevice.Acquire;

  FEffects.Acquire;
End; // TCustomInput.Acquire

Procedure TCustomInput.Finalize;
Begin
  If FDevice <> Nil Then FDevice.Unacquire;
  FButtonCount := 0;
  FEffects.Finalize;
  FDevice  := Nil;
  FDevice2 := Nil;
  FForceFeedbackDevice := False;
  FStates := [];
End; // TCustomInput.Finalize

Procedure TCustomInput.Initialize;
Begin
  FEffects.Initialize;
End; // TCustomInput.Initialize

Function TCustomInput.GetButton(Index: Integer): Boolean;
Begin
  If Index In [0..31]
   Then Result := TDXInputState(Integer(isButton1) + Index) in FStates
   Else Result := False;
End; // TCustomInput.GetButton

Function TCustomInput.GetCooperativeLevel: Integer;
Const
  Levels  : Array[Boolean] Of Integer = (DISCL_NONEXCLUSIVE, DISCL_EXCLUSIVE);
  Levels2 : Array[Boolean] Of Integer = (DISCL_BACKGROUND, DISCL_FOREGROUND);

Begin
  Result := Levels[FForceFeedbackDevice And FForceFeedback] Or Levels2[FDXInput.ActiveOnly];
End; // TCustomInput.GetCooperativeLevel

Function TCustomInput.GetDeviceState(dwSize: Integer; Var Data): Boolean;
Var hr: HRESULT;
Begin
  FillChar(Data, dwSize, 0);

  hr := Not 0;
  If FDevice <> Nil Then
  Begin
    hr := FDevice.GetDeviceState(dwSize, Data);
    If (hr = DIERR_INPUTLOST) Or (hr = DIERR_NOTACQUIRED) Then
    Begin
      FDevice.Acquire;
      hr := FDevice.GetDeviceState(dwSize, Data);
    End;
  End;

  Result := (hr = DI_OK);
End; // TCustomInput.GetDeviceState

Var
  AxisBase : Integer;
  POVBase  : Integer;
  BtnBase  : Integer;

Function TCustomInput.SetDataFormat: Boolean;
Var HR : HRESULT;

  Function DIEnumDeviceObjectsProc(Const peff: DIDEVICEOBJECTINSTANCEA;
                                   pvRef: Pointer): HRESULT; Stdcall;
  Begin
    Result := DIENUM_CONTINUE;
    With TCustomInput(pvRef) Do
    Begin
      If (peff.dwOfs < FDataFormat.dwDataSize) And
         ((peff.dwType And DIDFT_NODATA) = 0) Then
      Begin
        FDataFormatGUIDs[FDataFormat.dwNumObjs]           := peff.guidType;

        // Fill in the DIOBJECTDATAFORMAT structure

        FDataFormatObjects[FDataFormat.dwNumObjs].pguid   := @FDataFormatGUIDs[FDataFormat.dwNumObjs];


        If (peff.dwType And DIDFT_AXIS) <> 0 Then
        Begin
          FDataFormatObjects[FDataFormat.dwNumObjs].dwOfs := AxisBase;
          Inc(AxisBase,4);
        End;
        If (peff.dwType And DIDFT_BUTTON) <> 0 Then
        Begin
          FDataFormatObjects[FDataFormat.dwNumObjs].dwOfs := BtnBase;
          Inc(BtnBase);
        End;
        If (peff.dwType And DIDFT_POV) <> 0 Then
        Begin
          FDataFormatObjects[FDataFormat.dwNumObjs].dwOfs := POVBase;
          Inc(POVBase,4);
        End;
        FDataFormatObjects[FDataFormat.dwNumObjs].dwType  := peff.dwType;
        Case peff.dwType Of
          DIDFT_ABSAXIS: FDataFormatObjects[FDataFormat.dwNumObjs].dwFlags := DIDF_ABSAXIS;
          DIDFT_RELAXIS: FDataFormatObjects[FDataFormat.dwNumObjs].dwFlags := DIDF_RELAXIS;
        Else
          FDataFormatObjects[FDataFormat.dwNumObjs].dwFlags := 0;
        End; // Case
        Inc(FDataFormat.dwNumObjs);
      End;
    End;
  End; // DIEnumDeviceObjectsProc

Begin
  Result := False;
  If FDevice <> Nil Then
  Begin
    AxisBase := 0;
    POVBase  := 32;
    BtnBase  := 48;

    FDataFormat.dwSize    := SizeOf(FDataFormat);
    FDataFormat.dwObjSize := SizeOf(DIOBJECTDATAFORMAT);
    FDataFormat.dwNumObjs := 0;
    FDataFormat.rgodf     := @FDataFormatObjects;
    FDevice.EnumObjects(@DIEnumDeviceObjectsProc, Self, DIDFT_ALL);
    HR := FDevice.SetDataFormat(FDataFormat);
    If HR <> DI_OK Then Exit;
  End;
  Result := True;
End; // TCustomInput.SetDataFormat

Procedure TCustomInput.SetEffects(Value: TForceFeedbackEffects);
Begin
  FEffects.Assign(Value);
End; // TCustomInput.SetEffects

Procedure TCustomInput.SetEnabled(Value: Boolean);
Begin
  If FEnabled <> Value Then
  Begin
    FEnabled := Value;
    Initialize;
  End;
End; // TCustomInput.SetEnabled

Procedure TCustomInput.SetForceFeedback(Value: Boolean);
Begin
  If FForceFeedback <> Value Then
  Begin
    FForceFeedback := Value;
    Initialize;
  End;
End; // TCustomInput.SetForceFeedback

Procedure TCustomInput.SetWindowHandle(Value: Integer);
Begin
  If FDevice <> Nil Then FDevice.SetCooperativeLevel(Value, GetCooperativeLevel);
End; // TCustomInput.SetWindowHandle

// ---------------------------------
// TKeyboard
// ---------------------------------

constructor TKeyboard.Create(DXInput: TCustomDXInputBase);
begin
  inherited Create(DXInput);
  KeyAssigns := DefKeyAssign;
End;

procedure TKeyboard.DefineProperties(Filer: TFiler);
begin
  inherited DefineProperties(Filer);
  Filer.DefineBinaryProperty('Aissgns', ReadAssigns, WriteAssigns, False);
  Filer.DefineBinaryProperty('Assigns', ReadAssigns, WriteAssigns, True);
End;

function TKeyboard.GetKey(Key: Integer): Boolean;
begin
  if Key in [1..255] then
    Result := FKeyStates[Key] and $80<>0
  else
    Result := False;
End;

function TKeyboard.GetDIKey(Key: Integer): Boolean;
begin
  if Key in [0..255]
   then Result := ((dikb[Key] and $80) <> 0)
   else Result := False;
End;

procedure TKeyboard.Initialize;
begin
  Finalize;

  if (not FEnabled) or (csDesigning in FDXInput.ComponentState) then Exit;

  if FDXInput.FDInput<>nil then
  begin
    if FDXInput.FDInput.CreateDevice(GUID_SysKeyboard, FDevice, nil)<>DI_OK then Exit;
    FDevice.SetDataFormat(c_dfDIKeyboard);
  End;

  FButtonCount := 32;
End;

procedure TKeyboard.Update;

  function DIKEYtoVK(Key: Byte): Integer;
  begin
    Result := 0;
    case Key of
      DIK_ESCAPE       : Result := VK_ESCAPE;
      DIK_1            : Result := Ord('1');
      DIK_2            : Result := Ord('2');
      DIK_3            : Result := Ord('3');
      DIK_4            : Result := Ord('4');
      DIK_5            : Result := Ord('5');
      DIK_6            : Result := Ord('6');
      DIK_7            : Result := Ord('7');
      DIK_8            : Result := Ord('8');
      DIK_9            : Result := Ord('9');
      DIK_0            : Result := Ord('0');
      DIK_EQUALS       : Result := Ord('=');
      DIK_BACK         : Result := VK_BACK;
      DIK_TAB          : Result := VK_TAB;
      DIK_Q            : Result := Ord('Q');
      DIK_W            : Result := Ord('W');
      DIK_E            : Result := Ord('E');
      DIK_R            : Result := Ord('R');
      DIK_T            : Result := Ord('T');
      DIK_Y            : Result := Ord('Y');
      DIK_U            : Result := Ord('U');
      DIK_I            : Result := Ord('I');
      DIK_O            : Result := Ord('O');
      DIK_P            : Result := Ord('P');
      DIK_LBRACKET     : Result := Ord('[');
      DIK_RBRACKET     : Result := Ord(']');
      DIK_RETURN       : Result := VK_RETURN;
      DIK_LCONTROL     : Result := VK_CONTROL;
      DIK_A            : Result := Ord('A');
      DIK_S            : Result := Ord('S');
      DIK_D            : Result := Ord('D');
      DIK_F            : Result := Ord('F');
      DIK_G            : Result := Ord('G');
      DIK_H            : Result := Ord('H');
      DIK_J            : Result := Ord('J');
      DIK_K            : Result := Ord('K');
      DIK_L            : Result := Ord('L');
      DIK_SEMICOLON    : Result := Ord(';');
      DIK_APOSTROPHE   : Result := Ord('''');
      DIK_LSHIFT       : Result := VK_SHIFT;
      DIK_BACKSLASH    : Result := Ord('\');
      DIK_Z            : Result := Ord('Z');
      DIK_X            : Result := Ord('X');
      DIK_C            : Result := Ord('C');
      DIK_V            : Result := Ord('V');
      DIK_B            : Result := Ord('B');
      DIK_N            : Result := Ord('N');
      DIK_M            : Result := Ord('M');
      DIK_COMMA        : Result := Ord(',');
      DIK_PERIOD       : Result := Ord('.');
      DIK_SLASH        : Result := Ord('/');
      DIK_RSHIFT       : Result := VK_SHIFT;
      DIK_MULTIPLY     : Result := Ord('*');
      DIK_LMENU        : Result := VK_MENU;
      DIK_SPACE        : Result := VK_SPACE;
      DIK_CAPITAL      : Result := VK_CAPITAL;
      DIK_F1           : Result := VK_F1;
      DIK_F2           : Result := VK_F2;
      DIK_F3           : Result := VK_F3;
      DIK_F4           : Result := VK_F4;
      DIK_F5           : Result := VK_F5;
      DIK_F6           : Result := VK_F6;
      DIK_F7           : Result := VK_F7;
      DIK_F8           : Result := VK_F8;
      DIK_F9           : Result := VK_F9;
      DIK_F10          : Result := VK_F10;
      DIK_NUMLOCK      : Result := VK_NUMLOCK;
      DIK_SCROLL       : Result := VK_SCROLL;
      DIK_NUMPAD7      : Result := VK_NUMPAD7;
      DIK_NUMPAD8      : Result := VK_NUMPAD8;
      DIK_NUMPAD9      : Result := VK_NUMPAD9;
      DIK_SUBTRACT     : Result := VK_SUBTRACT;
      DIK_NUMPAD4      : Result := VK_NUMPAD4;
      DIK_NUMPAD5      : Result := VK_NUMPAD5;
      DIK_NUMPAD6      : Result := VK_NUMPAD6;
      DIK_ADD          : Result := VK_ADD;
      DIK_NUMPAD1      : Result := VK_NUMPAD1;
      DIK_NUMPAD2      : Result := VK_NUMPAD2;
      DIK_NUMPAD3      : Result := VK_NUMPAD3;
      DIK_NUMPAD0      : Result := VK_NUMPAD0;
      DIK_DECIMAL      : Result := VK_DECIMAL;
      DIK_F11          : Result := VK_F11;
      DIK_F12          : Result := VK_F12;
      DIK_NUMPADENTER  : Result := VK_RETURN;
      DIK_RCONTROL     : Result := VK_CONTROL;
      DIK_DIVIDE       : Result := VK_DIVIDE;
      DIK_RMENU        : Result := VK_MENU;
      DIK_HOME         : Result := VK_HOME;
      DIK_UP           : Result := VK_UP;
      DIK_PRIOR        : Result := VK_PRIOR;
      DIK_LEFT         : Result := VK_LEFT;
      DIK_RIGHT        : Result := VK_RIGHT;
      DIK_END          : Result := VK_End;
      DIK_DOWN         : Result := VK_DOWN;
      DIK_NEXT         : Result := VK_NEXT;
      DIK_INSERT       : Result := VK_INSERT;
      DIK_DELETE       : Result := VK_DELETE;
      DIK_LWIN         : Result := VK_LWIN;
      DIK_RWIN         : Result := VK_RWIN;
      DIK_APPS         : Result := VK_APPS;
    End;
  End;

var
  j: Integer;
  i: TDXInputState;
//  dikb: DIKEYBOARDSTATE;

begin
  FillChar(FKeyStates, SizeOf(FKeyStates), 0);
  FStates := [];

  if FDXInput.FActiveOnly and (GetForegroundWindow<>FDXInput.FForm.Handle) then
    Exit;

  If FDevice <> Nil Then
  begin
    FillChar(dikb, SizeOf(dikb), 0);

    if GetDeviceState(SizeOf(dikb), dikb) then
    begin
      {  The DirectInput key code is converted into the Windows virtual key code.  }
      for j:=Low(dikb) to High(dikb) do
        if dikb[j] and $80<>0 then
          FKeyStates[Byte(DIKEYtoVK(j))] := $80;
    End;
  end else
  begin           
    GetKeyboardState(FKeyStates);
  End;

  FStates := [];
  for i:=LOW(TDXInputState) to HIGH(TDXInputState) do
  begin
    for j:=0 to 2 do
      if Keys[KeyAssigns[i, j]] then
      begin
        FStates := FStates + [i];
        Break;
      End;
  End;
End;

procedure TKeyboard.ReadAssigns(Stream: TStream);
begin
  Stream.ReadBuffer(KeyAssigns, SizeOf(KeyAssigns));
End;

procedure TKeyboard.WriteAssigns(Stream: TStream);
begin
  Stream.WriteBuffer(KeyAssigns, SizeOf(KeyAssigns));
End;

// ---------------------------------
// TJoystick and TJoystick2
// ---------------------------------

Function SetDIDwordProperty(pdev: IDirectInputDevice; guidProperty: PGUID;
  dwObject, dwHow, dwValue: DWORD): HRESULT;
Var dipdw: DIPROPDWORD;
Begin
  dipdw.diph.dwSize       := SizeOf(dipdw);
  dipdw.diph.dwHeaderSize := SizeOf(dipdw.diph);
  dipdw.diph.dwObj        := dwObject;
  dipdw.diph.dwHow        := dwHow;
  dipdw.dwData            := dwValue;
  Result                  := pdev.SetProperty(guidProperty, dipdw.diph);
End; // SetDIDwordProperty

Function SetDIRangeProperty(pdev: IDirectInputDevice; guidProperty: PGUID;
  dwObject, dwHow, Value: DWORD): HRESULT;
Var diprg: DIPROPRANGE;
Begin
  diprg.diph.dwSize       := SizeOf(diprg);
  diprg.diph.dwHeaderSize := SizeOf(diprg.diph);
  diprg.diph.dwObj        := dwObject;
  diprg.diph.dwHow        := dwHow;
  diprg.lMin              := -Integer(Value);
  diprg.lMax              := +Integer(Value);
  Result                  := pdev.SetProperty(guidProperty, diprg.diph);
End; // SetDIRangeProperty

Function GetDIRangeProperty(pdev: IDirectInputDevice; guidProperty: PGUID;
  dwObject, dwHow: DWORD; Var Value: DWORD): HRESULT;
Var diprg: DIPROPRANGE;
Begin
  diprg.diph.dwSize       := SizeOf(diprg);
  diprg.diph.dwHeaderSize := SizeOf(diprg.diph);
  diprg.diph.dwObj        := dwObject;
  diprg.diph.dwHow        := dwHow;
  Result                  := pdev.GetProperty(guidProperty, diprg.diph);
  Value                   := diprg.lMax;
End; // GetDIRangeProperty

// ---------------------------------
// TJoystickBase
// ---------------------------------

Function TJoystickBase.GetCooperativeLevel: Integer;
Begin
  If Not FAutoCenter
   Then Result := DISCL_EXCLUSIVE Or DISCL_FOREGROUND
   Else Result := Inherited GetCooperativeLevel;
End; // TJoystickBase.GetCooperativeLevel

Function TJoystickBase.GetDeadZone(Obj: Integer): Integer;
Begin
  Result := 0;
  If (Obj >= Low(FDeadZone)) And (Obj < High(FDeadZone)) Then Result := FDeadZone[Obj];
End; // TJoystickBase.GetDeadZone

Function TJoystickBase.GetRange(Obj: Integer): Integer;
Begin
  Result := 0;
  If (Obj >= Low(FRange)) And (Obj < High(FRange)) Then Result := FRange[Obj];
End; // TJoystickBase.GetRange

Procedure TJoystickBase.SetDeadZone(Obj: Integer; Value: Integer);
Begin
  If (Obj < Low(FDeadZone)) Or (Obj >= High(FDeadZone)) Then Exit;

  If Value < 0   Then Value := 0;
  If Value > 100 Then Value := 100;

  If Obj = Integer(@PDIJOYSTATE2(Nil).rgdwPOV[0]) Then
  Begin
    FDeadZone[Obj] := DWORD(-1);
    Exit;
  End;

  FDeadZone[Obj] := Value;

  If FDevice <> Nil Then
  Begin
    If SetDIDwordProperty(FDevice, DIPROP_DEADZONE, Obj, DIPH_BYOFFSET, Value * 100) <> -1 Then
      FDeadZone[Obj] := DWORD(-1);
  End;
End; // TJoystickBase.SetDeadZone

Procedure TJoystickBase.SetRange(Obj: Integer; Value: Integer);
Begin
  If (Obj<Low(FRange)) Or (Obj>=High(FRange)) Then Exit;

  If Value < 0 Then Value := 0;

  If Obj >= DIJOFS_POV Then
  Begin
    FRange[Obj] := DWORD(-1);
    Exit;
  End;

  FRange[Obj] := Value;

  If FDevice <> Nil Then
  Begin
    If SetDIRangeProperty(FDevice, DIPROP_RANGE, Obj, DIPH_BYOFFSET, Value) <> -1 Then
    Begin
      If SetDIRangeProperty(FDevice, DIPROP_RANGE, Obj, DIPH_BYOFFSET, Value) <> -1
       then FRange[Obj] := Value
       Else FRange[Obj] := DWORD(-1);
    End;
  End;
End; // TJoystickBase.SetRange

Procedure TJoystickBase.SetAutoCenter(Value: Boolean);
Begin
  FAutoCenter := Value;
  If FDevice <> Nil Then
    SetDIDwordProperty(FDevice, DIPROP_AUTOCENTER, 0, DIPH_DEVICE, Ord(Value));
End; // TJoystickBase.SetAutoCenter

Function TJoystickBase.GetX: Integer;
Begin
  Result := Fdijs.lX;
End; // TJoystickBase.GetX

Function TJoystickBase.GetY: Integer;
Begin
  Result := Fdijs.lY;
End; // TJoystickBase.GetY

Function TJoystickBase.GetZ: Integer;
Begin
  Result := Fdijs.lZ;
End; // TJoystickBase.GetZ

Procedure TJoystickBase.Update;
Var
  I       : Integer;
  JoyInfo : TJoyInfoEx;

  Function ConvertValue(Value, wXmax, wXmin, DeadZone, Range: Integer): Integer;
  Var C,W: Integer;
  Begin
    Result := 0;
    C      := (wXmax - wXmin) Div 2;
    Value  := Value - C;
    W      := C * DeadZone Div 100;
    C      := C - W;
    If C <> 0 Then
    Begin
      If Abs(Value) > W Then
      Begin
        If Value > 0
         Then Result := MulDiv(Value - w, Range, C)
         Else Result := MulDiv(Value + w, Range, C);
      End;
    End;
  End; // ConvertValue

begin
  FillChar(Fdijs, SizeOf(Fdijs), 0);
  FStates := [];

  If FDXInput.FActiveOnly And (GetForegroundWindow <> FDXInput.FForm.Handle) Then Exit;

  If FDevice <> Nil Then
  Begin
    FDevice2.Poll;
    GetDeviceState(SizeOf(Fdijs), Fdijs);
  End
  Else
  Begin
    If FID2 <> -1 Then
    Begin
      JoyInfo.dwSize  := SizeOf(JoyInfo);
      JoyInfo.dwFlags := JOY_RETURNX or JOY_RETURNY or JOY_RETURNZ or JOY_RETURNPOV or
                         JOY_RETURNBUTTONS or JOY_RETURNCENTERED;

      joyGetPosEx(FID2, @JoyInfo);

      with FJoyCaps do
        Fdijs.lX := ConvertValue(JoyInfo.wXpos, wXmax, wXmin, FDeadZone[DIJOFS_X], FRange[DIJOFS_X]);

      with FJoyCaps do
        Fdijs.lY := ConvertValue(JoyInfo.wYpos, wYmax, wYmin, FDeadZone[DIJOFS_Y], FRange[DIJOFS_Y]);

      with FJoyCaps do
        Fdijs.lZ := ConvertValue(JoyInfo.wZpos, wZmax, wZmin, FDeadZone[DIJOFS_Z], FRange[DIJOFS_Z]);

      Fdijs.rgdwPOV[0] := JoyInfo.dwPOV;

      If FJoyCaps.wNumButtons > 0 Then
       For I := 0 To FJoyCaps.wNumButtons - 1 Do
        If JoyInfo.wButtons And (1 Shl I) <> 0 Then Fdijs.rgbButtons[I] := $80;
    End;
  End;

  For I := 0 To 31 Do
   If Fdijs.rgbButtons[I] And $80 <> 0 Then
    FStates := FStates + [TDXInputState(Ord(isButton1) + I)];

  If Fdijs.lX < 0 Then FStates := FStates + [isLeft];
  If Fdijs.lX > 0 Then FStates := FStates + [isRight];
  If Fdijs.lY < 0 Then FStates := FStates + [isUp];
  If Fdijs.lY > 0 Then FStates := FStates + [isDown];
End; // TJoystickBase.Update

// ---------------------------------
// TJoystick
// ---------------------------------

constructor TJoystick.Create(DXInput: TCustomDXInputBase);
begin
  inherited Create(DXInput);
  FAutoCenter := True;

  FID := 0;

  DeadZoneX := 50;
  DeadZoneY := 50;
  DeadZoneZ := 50;

  RangeX := 1000;
  RangeY := 1000;
  RangeZ := 1000;
end; // TJoystick.Create

procedure TJoystick.Finalize;
begin
  FID2 := -1;
  FillChar(Fdijs, SizeOf(Fdijs), 0);
  FillChar(FJoyCaps, SizeOf(FJoyCaps), 0);
  inherited Finalize;
end; // TJoystick.Finalize

function TJoystick_EnumJoysticksCallback(const lpddi: DIDEVICEINSTANCEA;
  pvRef: Pointer): HRESULT; stdcall;
begin
  Result := DIENUM_CONTINUE;

  with TJoystick(pvRef) do
  begin
    if FEnumIndex=FID then
    begin
      FDeviceGUID := lpddi.guidInstance;
      FEnumFlag := True;
      Result := DIENUM_STOP;
      Exit;
    end;
    Inc(FEnumIndex);
  end;
end; // TJoystick_EnumJoysticksCallback

procedure TJoystick.Initialize;
var
  i, j: Integer;
  devcaps: DIDEVCAPS;
begin
  Finalize;

  if (not FEnabled) or (FID<0) or (csDesigning in FDXInput.ComponentState) then Exit;

  try
    try
      if FDXInput.FDInput<>nil then
      begin
        {  Device search.  }
        FEnumFlag := False;
        FEnumIndex := 0;

        FDXInput.FDInput.EnumDevices(DIDEVTYPE_JOYSTICK, @TJoystick_EnumJoysticksCallback,
          Self, DIEDFL_ATTACHEDONLY);

        if not FEnumFlag then Exit;

        {  Device making.  }
        if FDXInput.FDInput.CreateDevice(FDeviceGUID, FDevice, nil)<>DI_OK then Exit;

        devcaps.dwSize := SizeOf(devcaps);
        if FDevice.GetCapabilities(devcaps)=DI_OK then
        begin
          FButtonCount := devcaps.dwButtons;
          if devcaps.dwFlags and DIDC_FORCEFEEDBACK<>0 then
            FForceFeedbackDevice := True;
        end;

        if FDXInput.FDInput.CreateDevice(GUID_Joystick, FDevice, nil)<>DI_OK then Exit;

        {  Device data format (DIDATAFORMAT) making.  }
        with FDataFormat do
        begin
          dwFlags := DIDF_ABSAXIS;
          dwDataSize := SizeOf(Fdijs);
        end;

        if not SetDataFormat then
        begin
          FDevice := nil;
          Exit;
        end;

        AutoCenter := FAutoCenter;

        for i:=Low(FDeadZone) to High(FDeadZone) do
          SetDeadZone(i, FDeadZone[i]);

        for i:=Low(FRange) to High(FRange) do
          SetRange(i, FRange[i]);

        FDevice2 := FDevice as IDirectInputDevice2;
      end;
    except
      Finalize;
      raise;
    end;
  finally
    if FDevice=nil then
    begin
      {  Because DirectInput cannot be used,  the GetJoyPosEx function is used.  }
      FID2 := -1;

      j := 0;
      for i:=0 to 255 do
      begin
        FillChar(FJoyCaps, SizeOf(FJoyCaps), 0);
        if joyGetDevCaps(i, @FJoyCaps, SizeOf(FJoyCaps))=JOYERR_NOERROR then
        begin
          if FID=j then
          begin
            FID2 := i;
            Break;
          end;
          Inc(j);
        end;
      end;

      if FID2<>-1 then
      begin
        if joyGetDevCaps(FID2, @FJoyCaps, SizeOf(FJoyCaps))=JOYERR_NOERROR then
        begin
          FButtonCount := FJoyCaps.wNumButtons;
        end else
        begin
          FID2 := -1;
        end;
      end;
    end;
  end;
  inherited Initialize;
end; // TJoystick.Initialize

procedure TJoystick.SetID(Value: Integer);
begin
  if Value<>FID then
  begin
    FID := Value;
    Initialize;
  end;
end; // TJoystick.SetID

// ---------------------------------
// TJoystick2
// ---------------------------------

Constructor TJoystick2.Create(DXInput: TCustomDXInputBase);
Begin
  Inherited Create(DXInput);
  FAutoCenter  := True;
  FID          := 0;
  DeadZoneX    := 50;
  DeadZoneY    := 50;
  DeadZoneZ    := 50;
  RangeX       := 1000;
  RangeY       := 1000;
  RangeZ       := 1000;
  FNumAxes     := 0;
  FNumPOVs     := 0;
  FProductName := '';
  FEnumFlag    := False;
End; // TJoystick2.Create

Function TJoystick2.GetU: Integer;
Begin
  Result := Fdijs.lRx;
End; // TJoystick2.GetU

Function TJoystick2.GetV: Integer;
Begin
  Result := Fdijs.lRy;
End; // TJoystick2.GetV

Function TJoystick2.GetW: Integer;
Begin
  Result := Fdijs.lRz;
End; // TJoystick2.GetW

Function TJoystick2.GetPOV(Obj: Integer): Integer;
Begin
  If (Obj >= 0) And (Obj < 4)
   Then Result := Fdijs.rgdwPOV[Obj]
   Else Result := 0;
End; // TJoystick2.GetPOV

Procedure TJoystick2.Finalize;
Begin
  FillChar(Fdijs, SizeOf(Fdijs), 0);
  FillChar(FJoyCaps, SizeOf(FJoyCaps), 0);
  FID2     := -1;
  FNumAxes := 0;
  FNumPOVs := 0;
  Inherited Finalize;
End; // TJoystick2.Finalize

Procedure TJoystick2.Initialize;
Var
  I,J     : Integer;
  DevCaps : DIDEVCAPS;

Begin
  Finalize;
  If (Not FEnabled) Or (FID < 0) Or (csDesigning In FDXInput.ComponentState) Then Exit;
  Try
    Try
      If FDXInput.FDInput <> Nil Then
      Begin

        {  Device search.  }

        If Not FEnumFlag Then Exit;

        {  Device making.  }

        If FDXInput.FDInput.CreateDevice(FDeviceGUID, FDevice, Nil) <> DI_OK Then Exit;
        DevCaps.dwSize := SizeOf(DevCaps);
        If FDevice.GetCapabilities(DevCaps) = DI_OK Then
        Begin
          FButtonCount := DevCaps.dwButtons;
          FNumAxes     := DevCaps.dwAxes;
          FNumPOVs     := DevCaps.dwPOVs;
          If (DevCaps.dwFlags And DIDC_FORCEFEEDBACK) <> 0 Then FForceFeedbackDevice := True;
        End;

        // Set the device data format (DIDATAFORMAT)

        FDataFormat.dwFlags    := DIDF_ABSAXIS;
        FDataFormat.dwDataSize := SizeOf(Fdijs);
        If Not SetDataFormat Then
        Begin
          FDevice := Nil;
          Exit;
        End;
        AutoCenter := FAutoCenter;

        // Set the dead zone and ranges

        For I := Low(FDeadZone) To High(FDeadZone) Do SetDeadZone(I, FDeadZone[I]);
        For I := Low(FRange) To High(FRange) Do SetRange(I, FRange[I]);
        FDevice2 := FDevice As IDirectInputDevice2;
      End;
    Except
      Finalize;
      Raise;
    End;
  Finally
    If FDevice = Nil Then
    begin

      {  Because DirectInput cannot be used,  the GetJoyPosEx function is used.  }
      
      FID2 := -1;

      j := 0;
      for i:=0 to 255 do
      begin
        FillChar(FJoyCaps, SizeOf(FJoyCaps), 0);
        if joyGetDevCaps(i, @FJoyCaps, SizeOf(FJoyCaps))=JOYERR_NOERROR then
        begin
          if FID=j then
          begin
            FID2 := i;
            Break;
          End;
          Inc(j);
        End;
      End;

      if FID2<>-1 then
      begin
        if joyGetDevCaps(FID2, @FJoyCaps, SizeOf(FJoyCaps))=JOYERR_NOERROR then
        begin
          FButtonCount := FJoyCaps.wNumButtons;
        end else
        begin
          FID2 := -1;
        End;
      End;
    End;
  End;
  Inherited Initialize;
End; // TJoystick2.Initialize

// ---------------------------------
// Used by TCustomDXInputBase
// ---------------------------------

Var
  FDirectInput      : IDirectInput;
  FDirectInputCount : Integer;        // Reference count for FDirectInput

Procedure InitDirectInput(Out DI: IDirectInput);
Type
  TDirectInputCreate = Function(hinst: THandle; dwVersion: DWORD;
           Out ppDI: IDirectInputA; punkOuter: IUnknown): HRESULT; Stdcall;

Begin
  If FDirectInput = Nil Then
  Begin
    Try
      TDirectInputCreate(DXLoadLibrary('DInput.dll', 'DirectInputCreateA'))
        (HInstance, DIRECTINPUT_VERSION, FDirectInput, Nil);
    Except
      FDirectInput := Nil;
    End;
  End;
  DI := FDirectInput;
  If FDirectInput <> Nil Then Inc(FDirectInputCount);
End; // InitDirectInput

Procedure FinDirectInput(Var DI: IDirectInput);
Begin
  If DI <> Nil Then
  Begin
    DI := Nil;
    Dec(FDirectInputCount);
    If FDirectInputCount <= 0 Then
    Begin
      FDirectInputCount := 0;
      FDirectInput      := Nil;
    End;
  End;
End; // FinDirectInput

// ---------------------------------
// TCustomDXInputBase
// ---------------------------------

procedure TCustomDXInputBase.Loaded;
begin
  Initialize;
end; // TCustomDXInputBase.Loaded

procedure TCustomDXInputBase.Finalize;
Var I: Integer;
Begin
  For I := 0 To FDevice.Count - 1 Do TCustomInput(FDevice[I]).Finalize;
  FinDirectInput(FDInput);
End; // TCustomDXInputBase.Finalize

Procedure TCustomDXInputBase.SetActiveOnly(Value: Boolean);
Begin
  If Value <> FActiveOnly Then
  Begin
    FActiveOnly := Value;
    If [csLoading, csReading] * ComponentState = [] Then SetWindowHandle;
  End;
End; // TCustomDXInputBase.SetActiveOnly

Procedure TCustomDXInputBase.SetKeyboard(Value: TKeyboard);
Begin
  FKeyboard.Assign(Value);
End; // TCustomDXInputBase.SetKeyboard

Procedure TCustomDXInputBase.SetWindowHandle;
Var I: Integer;
Begin
  For I := 0 To FDevice.Count - 1 Do TCustomInput(FDevice[I]).SetWindowHandle(FForm.Handle);
End; // TCustomDXInputBase.SetWindowHandle

Procedure TCustomDXInputBase.SetUseDirectInput(Value: Boolean);
begin
  If FUseDirectInput <> Value Then
  Begin
    FUseDirectInput := Value;
    Initialize;
  End;
End; // TCustomDXInputBase.SetUseDirectInput

Procedure TCustomDXInputBase.FormWndProc(Var Message: TMessage; DefWindowProc: TWndMethod);

  Procedure AcquireDevice;
  Var I: Integer;
  Begin
    For I := 0 To FDevice.Count - 1 Do TCustomInput(FDevice[I]).Acquire;
  End; // AcquireDevice

Begin
  Case Message.Msg Of
    WM_CREATE:
        Begin
          {  Window handle of Form changed.  }
          DefWindowProc(Message);
          SetWindowHandle;
          Exit;
        End;
    WM_ACTIVATEAPP:
        Begin
          DefWindowProc(Message);
          If TWMActivateApp(Message).Active Then AcquireDevice;
          Exit;
        End;
    WM_ACTIVATE:
        Begin
          DefWindowProc(Message);
          If TWMActivate(Message).Active <> WA_INACTIVE Then AcquireDevice;
          Exit;
        End;
  End; // Case
  DefWindowProc(Message);
End; // TCustomDXInputBase.FormWndProc

Procedure TCustomDXInputBase.Update;
Var
  J : Integer;
  I : TDXInputState;
  S : TDXInputStates;

Begin
  S := [];

  For J := 0 To FDevice.Count - 1 Do
  Begin
    TCustomInput(FDevice[J]).Update;
    S := S + TCustomInput(FDevice[J]).States;
  End;

  For I := Low(TDXInputState) To High(TDXInputState) Do
  Begin
    If (I In S) And (Not (I In FOldStates)) Then FStates := FStates + [i];
    If (Not (I In S)) And (I In FOldStates) Then FStates := FStates - [i];
  End;

  FOldStates := s;
End; // TCustomDXInputBase.Update

// ---------------------------------
// TCustomDXInput
// ---------------------------------

constructor TCustomDXInput.Create(AOwner: TComponent);
var
  Component: TComponent;
begin
  inherited Create(AOwner);

  FDevice := TList.Create;

  FActiveOnly := True;
  FJoystick := TJoystick.Create(Self);
  FKeyboard := TKeyboard.Create(Self);
  FUseDirectInput := True;

  Component := Owner;
  while (Component<>nil) and (not (Component is TCustomForm)) do
    Component := Component.Owner;
  if Component=nil then
    raise EDXInputError.CreateFmt(SNoForm, ['Owner']);
  FForm := TCustomForm(Component);

  FSubClass := TControlSubClass.Create(FForm, FormWndProc);
end; // TCustomDXInput.Create

destructor TCustomDXInput.Destroy;
begin
  Finalize;
  FJoystick.Free;
  FKeyboard.Free;
  FSubClass.Free;
  while FDevice.Count>0 do
    TCustomInput(FDevice[FDevice.Count-1]).Free;
  FDevice.Free;
  inherited Destroy;
end; // TCustomDXInput.Destroy

procedure TCustomDXInput.Initialize;
var
  i: Integer;
begin
  Finalize;
  if not (csDesigning in ComponentState) then
  begin
    if FUseDirectInput then InitDirectInput(FDInput);

    for i:=0 to FDevice.Count-1 do
      TCustomInput(FDevice[i]).Initialize;

    SetWindowHandle;

    Update;
  end;
end; // TCustomDXInput.Initialize

procedure TCustomDXInput.SetJoystick(Value: TJoystick);
begin
  FJoystick.Assign(Value);
end; // TCustomDXInput.SetJoystick

// ---------------------------------
// TCustomDXInput2
// ---------------------------------

Constructor TCustomDXInput2.Create(AOwner: TComponent);
Var Component: TComponent;
Begin
  Inherited Create(AOwner);
  FDevice         := TList.Create;
  FActiveOnly     := True;
  FJoysticks      := TList.Create;
  FKeyboard       := TKeyboard.Create(Self);
  FUseDirectInput := True;
  Component       := Owner;
  While (Component <> Nil) And (Not (Component Is TCustomForm)) Do Component := Component.Owner;
  If Component = Nil Then Raise EDXInputError.CreateFmt(SNoForm,['Owner']);
  FForm           := TCustomForm(Component);
  FSubClass       := TControlSubClass.Create(FForm, FormWndProc);
  FNumJoysticks   := 0;
End; // TCustomDXInput2.Create

Destructor TCustomDXInput2.Destroy;
Begin
  Finalize;
  FJoysticks.Free;
  FKeyboard.Free;
  FSubClass.Free;
  While FDevice.Count > 0 Do TCustomInput(FDevice[FDevice.Count - 1]).Free;
  FDevice.Free;
  Inherited Destroy;
End; // TCustomDXInput2.Destroy

Function TCustomDXInput2_EnumJoysticksCallback(Const lpddi: DIDEVICEINSTANCEA;
                                              pvRef: Pointer): HRESULT; Stdcall;
Var Joy: TJoystick2;
Begin
  Result           := DIENUM_CONTINUE;
  Joy              := TJoystick2.Create(TCustomDXInput2(pvRef));
  Joy.FID          := TCustomDXInput2(pvRef).FNumJoysticks;
  Joy.FDeviceGUID  := lpddi.guidInstance;
  Joy.FEnumFlag    := True;
  Joy.FProductName := StrPas(lpddi.tszProductName);
  TCustomDXInput2(pvRef).FJoysticks.Add(Joy);
  Inc(TCustomDXInput2(pvRef).FNumJoysticks);
End; // TCustomDXInput2_EnumJoysticksCallback

Procedure TCustomDXInput2.Initialize;
Var I: Integer;
Begin
  Finalize;
  If Not (csDesigning In ComponentState) Then
  Begin
    If FUseDirectInput Then InitDirectInput(FDInput);

    // Find out how many joysticks there are and create an instance for each one

    FNumJoysticks := 0;

    FJoysticks.Free;
    FJoysticks := TList.Create;

    FDInput.EnumDevices(DIDEVTYPE_JOYSTICK, @TCustomDXInput2_EnumJoysticksCallback,
      Self, DIEDFL_ALLDEVICES);

    // Initialize all of the devices

    For I := 0 To FDevice.Count - 1 Do TCustomInput(FDevice[I]).Initialize;

    SetWindowHandle;
    Update;
  End;
End; // TCustomDXInput2.Initialize

Function TCustomDXInput2.GetJoystick(Index: Integer): TJoystick2;
Begin
  If (Index >= 0) And (Index < FJoysticks.Count)
   Then Result := TJoystick2(FJoysticks.Items[Index])
   Else Result := Nil;
End; // TCustomDXInput2.GetJoystick

End.
