////////////////////////////////////////////////////////////////////////
//
// Attract-Mode Frontend - Emulator detection script
//
// Copyright (c) 2017 Andrew Mickelson
//
// This program comes with ABSOLUTELY NO WARRANTY.  It is licensed under
// the terms of the GNU General Public License, version 3 or later.
//
////////////////////////////////////////////////////////////////////////

fe.load_module( "file" );

////////////////////////////////////////////////////////////////////////
//
// This is the array of emulators we have generated that we think are
// actually in use by the user.  The plan is for Attract-Mode to read
// this array after running this script, and prompt the user for some
// automatic setup (when teh frontend is first run)
//
////////////////////////////////////////////////////////////////////////
emulators_to_generate <- [];

////////////////////////////////////////////////////////////////////////
//
// Write contents of "emu" to specified "filename".  Don't overwrite
// existing files unless force=true.
//
// "emu" is a table in the form of one of the entries in db.nut
//
// Returns true if file written, false otherwise
//
////////////////////////////////////////////////////////////////////////
function write_config( emu, filename, force=false )
{
	if ( !force && fe.path_test( filename, PathTest.IsFile ) )
		return false;

	local new_cfg = WriteTextFile( filename );

	new_cfg.write_line( "# Generated by Attract-Mode " + FeVersion + "\n" );
	new_cfg.write_line( "#\n" );
	new_cfg.write_line( "executable   " + emu["exe"] + "\n" );

	if ( emu.rawin( "args" ) )
		new_cfg.write_line( "args         " + emu["args"] + "\n" );

	if ( emu.rawin( "workdir" ) )
		new_cfg.write_line( "workdir      " + emu["workdir"] + "\n" );

	if ( emu.rawin( "rompath" ) && ( emu["rompath"].len() > 0 ))
		new_cfg.write_line( "rompath      " + emu["rompath"] + "\n" );

	if ( emu.rawin( "exts" ) )
		new_cfg.write_line( "romext       " + emu["exts"] + "\n" );

	local is_arcade = false;
	if ( emu.rawin( "system" ) )
	{
		new_cfg.write_line( "system       " + emu["system"] + "\n" );

		if ( emu["system"] == "Arcade" )
			is_arcade = true;
	}

	if ( emu.rawin( "source" ) )
		new_cfg.write_line( "info_source  " + emu["source"] + "\n" );

	if ( emu.rawin( "import_extras" ) )
		new_cfg.write_line( "import_extras    " + emu["import_extras"] + "\n" );

	if ( emu.rawin( "min_run" ) )
		new_cfg.write_line( "minimum_run_time " + emu["min_run"] + "\n" );

	if ( emu.rawin( "hotkey" ) )
		new_cfg.write_line( "exit_hotkey  " + emu["hotkey"] + "\n" );

	if ( is_arcade )
		new_cfg.write_line( "artwork      marquee   $HOME/" + emu["name"] + "/marquee\n" );
	else
		new_cfg.write_line( "artwork      flyer   $HOME/" + emu["name"] + "/boxart\n" );

	new_cfg.write_line( "artwork      snap    $HOME/" + emu["name"]
		+ "/video;$HOME/" + emu["name"] + "/snap\n" );

	return true;
}

////////////////////////////////////////////////////////////////////////
//
// Search for the first file with a name in the "names" array, within
// the specified paths in the "paths" array, returning the first match
//
// return value:  an array [ "path", "name" ] if file found, null otherwise
//
////////////////////////////////////////////////////////////////////////
function search_for_file( paths, names )
{
	foreach ( p in paths )
	{
		local ep = fe.path_expand( p );
		foreach ( e in names )
		{
			local temp = ep + e;
			if ( fe.path_test( temp, PathTest.IsFile ) )
				return [ ep, e ];
		}
	}

	return null;
}

////////////////////////////////////////////////////////////////////////
// return true if the specified paths (array) doesn't contain any files with
// the specified extension(s) (array)
////////////////////////////////////////////////////////////////////////
function path_is_empty( paths, exts )
{
	if (( paths == null ) || ( exts == null ))
		return true;

	foreach ( p in paths )
	{
		local dl = DirectoryListing( fe.path_expand( p ) );

		foreach ( f in dl.results )
		{
			foreach ( e in exts )
			{
				local tlen = f.len() - e.len();
				if ( tlen < 0 )
					continue;

				if ( f.slice( tlen ) == e )
					return false;
			}
		}
	}
	return true;
}

////////////////////////////////////////////////////////////////////////
//
// Write a line to the console reporting what we have done
//
////////////////////////////////////////////////////////////////////////
function console_report( name, found, wrote_cfg )
{
	//
	// Report status to the command line
	//
	local report = " * " + format( "%-13s", name ) + "[" + ( ( found.len() > 0 ) ? found : "Not found" ) + "]";

	if ( wrote_cfg )
		report += " [Wrote emulators/" + name + ".cfg]";

	print( report + "\n" );
}

////////////////////////////////////////////////////////////////////////
//
// Routine for generating emulator config and template from
// a db.nut entry
//
// General idea:
//   1. Try to find the specified emulator
//   2. If found, create a config for it in "<config_dir>/emulators/"
//   3. Separately generate a config template in "<config_dir>/emulators/templates"
//      This template will be available to the user when they create a new emulator
//      in the frontend.  The idea is to provide sensible default settings.
//
////////////////////////////////////////////////////////////////////////
function generate_emulator_config( emu )
{
	local my_OS = OS;
	if ( OS != "Windows" )
		my_OS = "Linux";

	// Nothing to do if this is an emulator for another OS
	//
	if ( emu.rawin( "OS" ) && ( my_OS != emu["OS"] ) )
		return;

	// Put all our OS-specific values in the base emu table now (if any)
	//
	if ( emu.rawin( my_OS ) )
	{
		foreach ( i,v in emu[ my_OS ] )
			emu[i] <- v;
	}

	// If a custom setup script is specified, use it instead
	// of our generic routine
	//
	if ( emu.rawin( "custom" ) )
		return dofile( fe.script_dir + emu["custom"], true );

	////////////////////////////////////////////////////////////////
	//
	// Generic routine for generating emulator config:
	//
	////////////////////////////////////////////////////////////////

	// Default paths to check for the executable
	//
	local default_search_paths = {

		"Linux" : [
			"/usr/games",
			"/usr/bin",
			"/usr/local/games",
			"/usr/local/bin",
			"$HOME/" + emu["name"],
			"./" + emu["name"],
			"../" + emu["name"],
			],

		"Windows" : [
			"$HOME/" + emu["name"] + "/",
			"./" + emu["name"] + "/",
			"../" + emu["name"] + "/",
			],
	};

	local search_paths;

	// Override search_paths if emu["path"] already provided
	//
	if ( emu.rawin( "path" ) )
	{
		search_paths = [];

		if ( type( emu[ "path" ] ) == "array" )
			foreach ( p in emu[ "path" ] )
				search_paths.push( p );
		else
			search_paths.push( emu["path"] );
	}
	else
		search_paths = default_search_paths[ my_OS ];


	local search_names = [];

	// Override search_names if emu["exe"] already provided
	//
	if ( emu.rawin( "exe" ) )
	{
		if ( type( emu[ "exe" ] ) == "array" )
			foreach ( p in emu[ "exe" ] )
				search_names.push( p );
		else
			search_names.push( emu["exe"] );
	}
	else
		search_names = [ emu["name"] ];

	if ( my_OS == "Windows" )
	{
		for ( local i=0; i<search_names.len(); i++ )
			search_names[i] += ".exe";
	}

	local res = search_for_file( search_paths, search_names );

	// Fill in emu rompath with sensible default value (if not provided)
	//
	if ( !emu.rawin( "rompath" ) )
		emu["rompath"] <- "$HOME/" + emu["name"] + "/roms/";

	local wrote_cfg = false;
	if ( res == null )
	{
		// Did not find emulator executable
		//
		if ( !emu.rawin( "exe" ) )
		{
			// Set emu["exe"] now so there is something there for the template
			//
			emu["exe"] <- search_paths[0] + "/" + search_names[0];
		}
	}
	else
	{
		// Found emulator, write config to emulators/
		//
		emu["exe"] <- res[0] + res[1];
		wrote_cfg = write_config( emu, FeConfigDirectory + "emulators/" + emu["name"] + ".cfg" );

		//
		// Add emulator to our list of emulators to prompt the user to generate a list for
		// if we find any roms in any of the rompaths for the emulator
		//
		local pp = split( emu["rompath"], ";" );
		if ( emu.rawin( "exts" ) && ( !path_is_empty( pp, split( emu["exts"], ";" ))))
			emulators_to_generate.push( emu["name"] );
	}

	// Write template to emulators/templates/
	//
	write_config( emu, FeConfigDirectory + "emulators/templates/" + emu["name"] + ".cfg", true );

	console_report( emu["name"], ( res ) ? emu["exe"] : "", wrote_cfg );
}

////////////////////////////////////////////////////////////////////////
//
// Load db.nut and generate emulator configs and templates based on its
// contents...
//
////////////////////////////////////////////////////////////////////////
emus <- dofile( fe.script_dir + "db.nut", true );

foreach ( e in emus )
	generate_emulator_config( e );
