/***************************************************************************

  machine.c

  Functions to emulate general aspects of the machine (RAM, ROM, interrupts,
  I/O ports)

***************************************************************************/

#include <stdio.h>
#include <string.h>
#include <time.h>
#include <stdlib.h>
#include "Z80.h"
#include "machine.h"
#include "vidhrdw.h"
#include "roms.h"
#include "memmap.h"
#include "osdepend.h"

#ifdef UNIX
#define uclock_t clock_t
#define	uclock clock
#define UCLOCKS_PER_SEC CLOCKS_PER_SEC
#endif

/* CPU_SPEED is the speed of the CPU in Hz. It is used together with */
/* FRAMES_PER_SECOND to calculate how much CPU cycles must pass between */
/* interrupts. */
#define CPU_SPEED 3000000       /* 3 Mhz */
#define FRAMES_PER_SECOND 60

unsigned char   RAM[0x10000];   /* 64k of RAM */
int             Sound;
int             Joystick;
int             Sprite = 0;
int             soundsource;
int             gameemulated;
int             base;

/* default dip switches settings */
int             dsw1 = 0;
int             interrupt_set = 0;

/***************************************************************************

  Initialize the emulated machine (load the roms, initialize the various
  subsystems...). Returns 0 if successful.

***************************************************************************/
int             init_machine(const char *gamename)
{

    if (readroms(RAM, gameinfo[gameemulated].rom, gamename) != 0)
        return 1;

    if (vh_init(gamename))
    {
        printf("Cannot initialize video emulation\n");
        return 1;
    }

    return 0;
}



/***************************************************************************

  Run the emulation. Start the various subsystems and the CPU emulation.
  Returns non zero in case of error.

***************************************************************************/
int             run_machine(void)
{
    srand(1);

    if (vh_start() == 0)        /* start the video hardware */
    {
        reg             StartRegs;


        if ((gameemulated == PHOENIX) || (gameemulated == PLEIADS))
            IPeriod = 12000;
        else
            IPeriod = CPU_SPEED / FRAMES_PER_SECOND;    /* Number of T-states
                                                         * per interrupt */
        ResetZ80(&StartRegs);
        Z80(&StartRegs);        /* start the CPU emulation */

        vh_stop();

        return 0;
    }
    else
        printf("Unable to setup display\n");

    return 1;
}



/***************************************************************************

  Perform a memory read. This function is called by the CPU emulation.

***************************************************************************/
byte            M_RDMEM(dword A)
{
    byte            rv = 0;
    byte            pom = 0;

    if (gameemulated == AMIDAR)
    {
        switch (A)
        {

        case AMIDAR_IN0_PORT:   // Switch - left, right, jump, credit
            osd_poll_joystick();
            rv = ALL_BITS_1;
            if ((osd_key_pressed(OSD_KEY_LEFT)) || (osd_joy_left))
                rv &= ~AMIDAR_IN0_P1_LEFT;
            if ((osd_key_pressed(OSD_KEY_RIGHT)) || (osd_joy_right))
                rv &= ~AMIDAR_IN0_P1_RIGHT;
            if ((osd_key_pressed(OSD_KEY_CONTROL)) ||
                (osd_joy_b1 || osd_joy_b2))
                rv &= ~AMIDAR_IN0_P1_JUMP;
            if (osd_key_pressed(OSD_KEY_3))
                rv &= ~AMIDAR_IN0_COIN1;
            break;
        case AMIDAR_IN1_PORT:   // Switch - P1, P2
            rv = ALL_BITS_1;
            if (osd_key_pressed(OSD_KEY_1))
                rv &= ~AMIDAR_IN1_1P_START;
            break;
        case AMIDAR_IN2_PORT:   // Switch - Up, Down
            osd_poll_joystick();
            rv = ALL_BITS_1;
            if ((osd_key_pressed(OSD_KEY_UP)) || (osd_joy_up))
                rv &= ~AMIDAR_IN1_P1_UP;
            if ((osd_key_pressed(OSD_KEY_DOWN)) || (osd_joy_down))
                rv &= ~AMIDAR_IN1_P1_DOWN;
            break;
        case AMIDAR_DSW_PORT:   // Dip Switches
            rv = ALL_BITS_1;
            break;
        default:
            rv = RAM[A];
        }
        return (rv);

    }
    else if (gameemulated == DKONG)
    {
        switch (A)
        {

        case DKONG_IN0_PORT:    // Switch - left, right, up, down, jump
            osd_poll_joystick();
            rv = ALL_BITS_0;
            if ((osd_key_pressed(OSD_KEY_LEFT)) || (osd_joy_left))
                rv |= DKONG_IN0_P1_LEFT;
            if ((osd_key_pressed(OSD_KEY_RIGHT)) || (osd_joy_right))
                rv |= DKONG_IN0_P1_RIGHT;
            if ((osd_key_pressed(OSD_KEY_UP)) || (osd_joy_up))
                rv |= DKONG_IN0_P1_UP;
            if ((osd_key_pressed(OSD_KEY_DOWN)) || (osd_joy_down))
                rv |= DKONG_IN0_P1_DOWN;
            if ((osd_key_pressed(OSD_KEY_CONTROL)) ||
                (osd_joy_b1 || osd_joy_b2))
                rv |= DKONG_IN0_P1_JUMP;
            break;
        case DKONG_IN1_PORT:    // Switch - P1, P2
            rv = ALL_BITS_0;
            if (osd_key_pressed(OSD_KEY_1))
                rv |= DKONG_IN1_1P_START;
            if (osd_key_pressed(OSD_KEY_2))
                rv |= DKONG_IN1_2P_START;
            if (osd_key_pressed(OSD_KEY_3))
                rv |= DKONG_IN1_COIN;
            break;
        default:
            rv = RAM[A];
        }
        return (rv);

    }
    else if (gameemulated == TURTLES)
    {
        switch (A)
        {

        case TURTLES_IN0_PORT:  // Switch - left, right, jump, credit
            osd_poll_joystick();
            rv = ALL_BITS_1;
            if ((osd_key_pressed(OSD_KEY_LEFT)) || (osd_joy_left))
                rv &= ~TURTLES_IN0_P1_LEFT;
            if ((osd_key_pressed(OSD_KEY_RIGHT)) || (osd_joy_right))
                rv &= ~TURTLES_IN0_P1_RIGHT;
            if ((osd_key_pressed(OSD_KEY_CONTROL)) ||
                (osd_joy_b1 || osd_joy_b2))
                rv &= ~TURTLES_IN0_P1_BOMB;
            if (osd_key_pressed(OSD_KEY_3))
                rv &= ~TURTLES_IN0_COIN1;
            break;
        case TURTLES_IN1_PORT:  // Switch - P1, P2
            rv = ALL_BITS_1;
            if (osd_key_pressed(OSD_KEY_1))
                rv &= ~TURTLES_IN1_1P_START;
            break;
        case TURTLES_IN2_PORT:  // Switch - Up, Down
            osd_poll_joystick();
            rv = ALL_BITS_1;
            if ((osd_key_pressed(OSD_KEY_UP)) || (osd_joy_up))
                rv &= ~TURTLES_IN1_P1_UP;
            if ((osd_key_pressed(OSD_KEY_DOWN)) || (osd_joy_down))
                rv &= ~TURTLES_IN1_P1_DOWN;
            break;
        case TURTLES_DSW_PORT:  // Dip Switches
            rv = ALL_BITS_1;
            break;
        default:
            rv = RAM[A];
        }
        return (rv);

    }
    else if (gameemulated == FROGGER)
    {
        switch (A)
        {
        case FROGGER_IN0_PORT:  // Switch 0
            osd_poll_joystick();
            rv = ALL_BITS_1;
            if ((osd_key_pressed(OSD_KEY_LEFT)) || (osd_joy_left))
                rv &= ~FROGGER_IN0_P1_LEFT;
            if ((osd_key_pressed(OSD_KEY_RIGHT)) || (osd_joy_right))
                rv &= ~FROGGER_IN0_P1_RIGHT;
            if (osd_key_pressed(OSD_KEY_3))
                rv &= ~FROGGER_IN0_COIN1;
            break;

        case FROGGER_IN1_PORT:  // Switch 1
            rv = ALL_BITS_1;
            if (osd_key_pressed(OSD_KEY_1))
                rv &= ~FROGGER_IN1_1P_START;
            if (osd_key_pressed(OSD_KEY_2))
                rv &= ~FROGGER_IN1_2P_START;
            break;
        case FROGGER_IN2_PORT:
            osd_poll_joystick();
            rv = ALL_BITS_1;
            if ((osd_key_pressed(OSD_KEY_UP)) || (osd_joy_up))
                rv &= ~FROGGER_IN1_P1_UP;
            if ((osd_key_pressed(OSD_KEY_DOWN)) || (osd_joy_down))
                rv &= ~FROGGER_IN1_P1_DOWN;
            break;

        default:
            rv = RAM[A];
        }
        return (rv);

    }
    else if (gameemulated == PHOENIX)
    {
        if (A >= PHOENIX_IN0_PORT_START && A <= PHOENIX_IN0_PORT_END)   // are we reading dip
                                                                        // switch memory?
        {
            osd_poll_joystick();
            rv = ALL_BITS_1;
            if ((osd_key_pressed(OSD_KEY_LEFT)) || (osd_joy_left))
                rv &= ~PHOENIX_IN0_P1_LEFT;
            if ((osd_key_pressed(OSD_KEY_RIGHT)) || (osd_joy_right))
                rv &= ~PHOENIX_IN0_P1_RIGHT;
            if (osd_key_pressed(OSD_KEY_3))
                rv &= ~PHOENIX_IN0_COIN1;
            if (osd_key_pressed(OSD_KEY_1))
                rv &= ~PHOENIX_IN0_1P_START;
            if (osd_key_pressed(OSD_KEY_2))
                rv &= ~PHOENIX_IN0_2P_START;
            if (osd_key_pressed(OSD_KEY_CONTROL) ||
                osd_joy_b1)
                rv &= ~PHOENIX_IN0_P1_FIRE;
            if (osd_key_pressed(OSD_KEY_ALT) ||
                osd_joy_b2)
                rv &= ~PHOENIX_IN0_P1_BARRIER;
        }
        else if (A >= PHOENIX_DSW_PORT_START && A <= PHOENIX_DSW_PORT_END)  // are we reading dip
                                                                            // switch memory?
        {
            rv = ALL_BITS_1;
            rv &= ~1;
            rv &= ~2;
            rv &= ~16;
            if (interrupt_set == 1)
                interrupt_set = 0;
            else
                rv &= ~128;
        }
        else
            rv = RAM[A];
        return (rv);

    }
    else if (gameemulated == PLEIADS)
    {
        if (A >= PLEIADS_IN0_PORT_START && A <= PLEIADS_IN0_PORT_END)   // are we reading dip
                                                                        // switch memory?
        {
            osd_poll_joystick();
            rv = ALL_BITS_1;
            if ((osd_key_pressed(OSD_KEY_LEFT)) || (osd_joy_left))
                rv &= ~PLEIADS_IN0_P1_LEFT;
            if ((osd_key_pressed(OSD_KEY_RIGHT)) || (osd_joy_right))
                rv &= ~PLEIADS_IN0_P1_RIGHT;
            if (osd_key_pressed(OSD_KEY_3))
                rv &= ~PLEIADS_IN0_COIN1;
            if (osd_key_pressed(OSD_KEY_1))
                rv &= ~PLEIADS_IN0_1P_START;
            if (osd_key_pressed(OSD_KEY_2))
                rv &= ~PLEIADS_IN0_2P_START;
            if (osd_key_pressed(OSD_KEY_CONTROL) ||
                osd_joy_b1)
                rv &= ~PLEIADS_IN0_P1_FIRE;
            if (osd_key_pressed(OSD_KEY_ALT) ||
                osd_joy_b2)
                rv &= ~PLEIADS_IN0_P1_WARP;
        }
        else if (A >= PLEIADS_DSW_PORT_START && A <= PLEIADS_DSW_PORT_END)  // are we reading dip
                                                                            // switch memory?
        {
            rv = ALL_BITS_1;
            rv &= ~1;
            rv &= ~2;
            rv &= ~16;
            if (interrupt_set == 1)
                interrupt_set = 0;
            else
                rv &= ~128;
        }
        else
            rv = RAM[A];
        return (rv);

    }
    else if (gameemulated == MOONCRST)
    {
        switch (A)
        {
        case MOONCRST_IN0_PORT: // Switch 0
            osd_poll_joystick();
            rv = ALL_BITS_0;
            if (osd_key_pressed(OSD_KEY_3))
                rv |= MOONCRST_IN0_COIN1;
            if ((osd_key_pressed(OSD_KEY_LEFT)) || (osd_joy_left))
                rv |= MOONCRST_IN0_P1_LEFT;
            if ((osd_key_pressed(OSD_KEY_RIGHT)) || (osd_joy_right))
                rv |= MOONCRST_IN0_P1_RIGHT;
            if ((osd_key_pressed(OSD_KEY_CONTROL)) ||
                (osd_joy_b1 || osd_joy_b2))
                rv |= MOONCRST_IN0_P1_SHOOT;
            break;
        case MOONCRST_IN1_PORT: // Switch 1
            rv = ALL_BITS_0;
            if (dsw1 & 1)
                rv |= MOONCRST_IN1_SW1;
            if (dsw1 & 2)
                rv |= MOONCRST_IN1_SW2;
            if (osd_key_pressed(OSD_KEY_1))
                rv |= MOONCRST_IN1_1P_START;
            if (osd_key_pressed(OSD_KEY_2))
                rv |= MOONCRST_IN1_2P_START;
            break;
        case MOONCRST_DSW_PORT: // Dip Switch
            rv = ALL_BITS_0;
            rv |= dsw1 >> 2;
            break;

        default:
            rv = RAM[A];
        }
        return (rv);
    }
    else if (gameemulated == BAGMAN)
    {
        switch (A)
        {
        case BAGMAN_IN0_PORT:
            pom = 0;
            rv = RAM[A];

            if (RAM[BAGMAN_IN0_PORT_ON] == 1)
            {
                osd_poll_joystick();
                if ((osd_key_pressed(OSD_KEY_LEFT)) || (osd_joy_left))
                    pom |= BAGMAN_IN0_LEFT;
                if ((osd_key_pressed(OSD_KEY_RIGHT)) || (osd_joy_right))
                    pom |= BAGMAN_IN0_RIGHT;
                if ((osd_key_pressed(OSD_KEY_UP)) || (osd_joy_up))
                    pom |= BAGMAN_IN0_UP;
                if ((osd_key_pressed(OSD_KEY_DOWN)) || (osd_joy_down))
                    pom |= BAGMAN_IN0_DOWN;
                if ((osd_key_pressed(OSD_KEY_CONTROL)) ||
                    (osd_joy_b1 || osd_joy_b2))
                    pom |= BAGMAN_IN0_SHOOT;
                rv &= (~RAM[BAGMAN_IN0_PORT_MASK]);
            }
            pom &= RAM[BAGMAN_IN0_PORT_MASK];
            rv |= pom;
            rv &= (0xff - 0x07);
            if (osd_key_pressed(OSD_KEY_1))
                rv |= BAGMAN_1P_START;
            break;

        case BAGMAN_IN1_PORT:
            if (osd_key_pressed(OSD_KEY_3))
                rv = BAGMAN_COIN1;
            break;

        case BAGMAN_IN0_PORT_MASK:
            rv = 0x00;
            if (osd_key_pressed(OSD_KEY_2))
                rv |= BAGMAN_2P_START;
            break;

        case BAGMAN_RANDOM_PORT:
            rv = 1 + (rand() & 254);
            break;

        case BAGMAN_DSW_PORT:   // Dip Switch
            rv = BAGMAN_SWITCH_VALUE;
            break;

        default:
            rv = RAM[A];
        }
        return (rv);
    }
    else if (gameemulated == THEEND)
    {
        switch (A)
        {
        case THEEND_IN0_PORT:   // Switch 0
            osd_poll_joystick();
            rv = ALL_BITS_0;
            if (osd_key_pressed(OSD_KEY_3))
                rv |= 4;
            if ((osd_key_pressed(OSD_KEY_LEFT)) || (osd_joy_left))
                rv |= 16;
            if ((osd_key_pressed(OSD_KEY_RIGHT)) || (osd_joy_right))
                rv |= 32;
            if ((osd_key_pressed(OSD_KEY_CONTROL)) ||
                (osd_joy_b1 || osd_joy_b2))
                rv |= 8;
            break;
        case THEEND_IN1_PORT:   // Switch 1
            rv = ALL_BITS_0;
            if (osd_key_pressed(OSD_KEY_1))
                rv |= THEEND_IN1_1P_START;
            else
                rv |= THEEND_IN1_MASK;
            break;
        case THEEND_DSW_PORT:   // Dip Switch
            rv = ALL_BITS_0;
            rv |= THEEND_DSW_MASK;
            break;

        default:
            rv = RAM[A];
        }
        return (rv);

    }
    else
    {                           // Galaxian and the rest
        switch (A)
        {
        case GALAXIAN_IN0_PORT: // Switch 0
            osd_poll_joystick();
            rv = ALL_BITS_0;
            if (osd_key_pressed(OSD_KEY_3))
                rv |= GALAXIAN_IN0_COIN1;
            if ((osd_key_pressed(OSD_KEY_LEFT)) || (osd_joy_left))
                rv |= GALAXIAN_IN0_P1_LEFT;
            if ((osd_key_pressed(OSD_KEY_RIGHT)) || (osd_joy_right))
                rv |= GALAXIAN_IN0_P1_RIGHT;
            if ((osd_key_pressed(OSD_KEY_CONTROL)) ||
                (osd_joy_b1 || osd_joy_b2))
                rv |= GALAXIAN_IN0_P1_SHOOT;
            if ((osd_key_pressed(OSD_KEY_F9)) && (gameemulated != WAROFBUG) && (gameemulated != PISCES))
                rv |= GALAXIAN_IN0_TEST;
            if (gameemulated == WAROFBUG)
            {
                if ((osd_key_pressed(OSD_KEY_UP)) || (osd_joy_up))
                    rv |= WAROFBUG_IN0_UP;
                if ((osd_key_pressed(OSD_KEY_DOWN)) || (osd_joy_down))
                    rv |= WAROFBUG_IN0_DOWN;
            }
            break;
        case GALAXIAN_IN1_PORT: // Switch 1
            rv = ALL_BITS_0;
            if (dsw1 & 1)
                rv |= GALAXIAN_IN1_SW1;
            if (dsw1 & 2)
                rv |= GALAXIAN_IN1_SW2;
            if (osd_key_pressed(OSD_KEY_1))
                rv |= GALAXIAN_IN1_1P_START;
            if (osd_key_pressed(OSD_KEY_2))
                rv |= GALAXIAN_IN1_2P_START;
            break;
        case GALAXIAN_DSW_PORT: // Dip Switch
            rv = ALL_BITS_0;
            rv |= dsw1 >> 2;
            break;

        default:
            rv = RAM[A];
        }
        return (rv);

    }
}


/***************************************************************************

  Perform a memory read. This function is called by the CPU emulation.

***************************************************************************/
void            M_WRMEM(dword A, byte V)
{

    if ((gameemulated == AMIDAR) ||
        (gameemulated == TURTLES) ||
        (gameemulated == FROGGER) ||
        (gameemulated == THEEND))
    {
        if (A < THEEND_END_OF_ROM)
        {
            return;
        }
        else
        {
            RAM[A] = V;
        }
    }
    else if (gameemulated == DKONG)
    {
        if (A < DKONG_END_OF_ROM)
            return;
        RAM[A] = V;
    }
    else if (gameemulated == BAGMAN)
    {
        if (A < BAGMAN_END_OF_ROM)
            return;
        RAM[A] = V;
    }
    else if ((gameemulated == PHOENIX) || (gameemulated == PLEIADS))
    {
        if (A < PHOENIX_END_OF_ROM)
        {
            return;
        }
        else if (A == PHOENIX_SOUND_PORT_A)
        {
            if (V & 48)
            {
                V &= 15;
                if (V && V != 15)
                {
                    osd_sound(V * 16);
                }
                else
                {
                    osd_sound_stop();
                }
            }
            else
                osd_sound_stop();
        }
        else if (A == PHOENIX_SOUND_PORT_B)
        {
            if (V & 48)
            {
                V &= 15;
                if (V && V != 15)
                {
                    osd_sound(V * 16);
                }
                else
                {
                    osd_sound_stop();
                }
            }
            else
                osd_sound_stop();
        }
        else
        {
            RAM[A] = V;
        }
    }
    else if (gameemulated == MOONCRST)
    {
        if (A < MOONCRST_END_OF_ROM)
            return;
        switch (A)
        {
// Writes to 0xA000 to 0xA7FF decode to /LAMP
// This selects the extra character ROMs.
        case MOONCRST_LAMP_1:
            if (V)
                Sprite |= 0x10;
            else
                Sprite &= ~0x10;
            break;
        case MOONCRST_LAMP_2:
            if (V)
                Sprite |= 0x20;
            else
                Sprite &= ~0x20;
            break;
        case MOONCRST_LAMP_3:
            if (V)
                Sprite |= 0x40;
            else
                Sprite &= ~0x40;
            break;

// Sound to be added here (/SOUND)
        case MOONCRST_SOUND_RESET_F1:   // FS1
            break;
        case MOONCRST_SOUND_RESET_F2:   // FS2
            break;
        case MOONCRST_SOUND_RESET_F3:   // FS3
            break;
        case MOONCRST_SOUND_NOISE:  // Noise
            break;
        case MOONCRST_SOUND_NOT_USED:
            break;
        case MOONCRST_SOUND_SHOOT:  // MSHOOT
            break;
        case MOONCRST_SOUND_VOLUME_F1:  // VOL 1
            break;
        case MOONCRST_SOUND_VOLUME_F2:  // VOL 2
            break;
        case MOONCRST_NMI:      // NMION
            break;
        case MOONCRST_STARS:    // STARS ON
            break;
        case MOONCRST_HFLIP:    // HFLIP
            break;
        case MOONCRST_VFLIP:    // VFLIP
            break;
        case MOONCRST_SOUND_FREQUENCY:  // /PITCH
            if (V && (V != 255))
            {
                osd_sound(V);
            }
            else
            {
                osd_sound_stop();
            }
            break;
        }
        RAM[A] = V;
    }
    else
    {                           // Galaxian and the rest
// Ignore writes to ROM
        if (A < GALAXIAN_END_OF_ROM)
            return;

        switch (A)
        {
// Writes to 0x6000 to 0x67FF decode to /LAMP

        case GALAXIAN_LAMP_1:
            break;
        case GALAXIAN_LAMP_2:
            break;
        case GALAXIAN_LAMP_3:
            break;

// Sound to be added here (/SOUND)
        case GALAXIAN_SOUND_RESET_F1:   // Reset background F1
            break;
        case GALAXIAN_SOUND_RESET_F2:   // Reset background F2
            break;
        case GALAXIAN_SOUND_RESET_F3:   // Reset background F3
            break;
        case GALAXIAN_SOUND_NOISE:  // Noise on/off
            break;
        case GALAXIAN_SOUND_NOT_USED:   // Not used
            break;
        case GALAXIAN_SOUND_SHOOT:  // SHOOT on/off
            break;
        case GALAXIAN_SOUND_VOLUME_F1:  // VOL f1
            break;
        case GALAXIAN_SOUND_VOLUME_F2:  // VOL f2
            break;
        case GALAXIAN_NMI:      // NMION
            break;
        case GALAXIAN_STARS:    // STARS ON
            break;
        case GALAXIAN_HFLIP:    // HFLIP
            break;
        case GALAXIAN_VFLIP:    // VFLIP
            break;
        case GALAXIAN_SOUND_FREQUENCY:  // SOUND BASE FREQUENCY
            if (V && (V != 255))
            {
                osd_sound(V);
            }
            else
            {
                osd_sound_stop();
            }
            break;
        }
        RAM[A] = V;
    }
}

/***************************************************************************

  Interrupt handler. This function is called at regular intervals
  (determined by IPeriod) by the CPU emulation.

***************************************************************************/

int             Interrupt(void)
{
    int             IntVector = 0;  /* Here we store the interrupt vector, if
                                     * the code performs */
    /* an OUT to port $0. Not all games do it: many use */
    /* Interrupt Mode 1, which doesn't use an interrupt vector */
    /* (see Z80.c for details). */

    static uclock_t prev;
    uclock_t        curr;

    /* first of all, refresh the screen */
    vh_screenrefresh();

// Check for keypad inputs

    /* if the user pressed ESC, stop the emulation */
    if (osd_key_pressed(OSD_KEY_ESC))
        CPURunning = 0;

    /* if the user hits F10, toggle pause */
    if (osd_key_pressed(OSD_KEY_F10))   /* pause the game */
    {
        osd_sound_stop();
        while (osd_key_pressed(OSD_KEY_F10));
        while (!osd_key_pressed(OSD_KEY_F10));
        while (osd_key_pressed(OSD_KEY_F10));
    }

    /* if the user hits F7, toggle joystick */
    if (osd_key_pressed(OSD_KEY_F7))    /* toggle joystick */
    {
        while (osd_key_pressed(OSD_KEY_F7));    // wait until key is
        // depressed
        if (Joystick)
            Joystick = 0;
        else
            Joystick = 1;
    }

    /* if the user hits F8, toggle sound */
    if (osd_key_pressed(OSD_KEY_F8))    /* toggle sound */
    {
        while (osd_key_pressed(OSD_KEY_F8));    // wait until key is
        // depressed
        if (Sound)
        {
            osd_sound_stop();
            Sound = 0;
        }
        else
            Sound = 1;
    }

    /* now wait until it's time to trigger the interrupt */
    do
    {
        curr = uclock();
    } while ((curr - prev) < UCLOCKS_PER_SEC / FRAMES_PER_SECOND);

    prev = curr;

    if ((gameemulated == THEEND) || (gameemulated == AMIDAR) ||
        (gameemulated == TURTLES) || (gameemulated == FROGGER))
        IntVector = NMI_INT;
    else if ((gameemulated == PHOENIX) || (gameemulated == PLEIADS))
    {
        IntVector = IGNORE_INT;
        interrupt_set = 1;
    }
    else if (gameemulated == BAGMAN)
    {
        IntVector = 0;
    }
    else if (gameemulated == DKONG)
    {
        if (RAM[DKONG_NMI])
            IntVector = NMI_INT;
        else
            IntVector = IGNORE_INT;
    }
    else if (gameemulated == MOONCRST)
    {
        if (RAM[MOONCRST_NMI])
            IntVector = NMI_INT;
        else
            IntVector = IGNORE_INT;
    }
    else
    {                           // Galaxian and the rest
        if (RAM[GALAXIAN_NMI])
            IntVector = NMI_INT;
        else
            IntVector = IGNORE_INT;
    }


    return IntVector;
}



/***************************************************************************

  This function is called by the CPU emulation when the EI instruction is
  executed. We don't need to do anything fancy.

***************************************************************************/
int             InterruptsEnabled(void)
{
    return IGNORE_INT;
}



/***************************************************************************

  Execute an OUT instruction. This function is called by the CPU emulation.

  Some machines uses OUT to port $0 to set the interrupt vector, so
  we have to remember the value passed.

***************************************************************************/
void            DoOut(byte A, byte V)
{

}



byte            DoIn(byte A)
{
    return 0;
}



void            Patch(reg * R)
{
}
