/***************************************************************************

  gfxdecod.c

  Functions to convert the graphic information stored in the roms to a more
  usable format.

  Back in the early '80s, arcade machines didn't have the memory or the
  speed to handle bitmaps like we do today. They used "character maps",
  instead: they had one or more sets of characters (usually 8x8 pixels),
  which would be placed on the screen in order to form a picture. This was
  very fast: updating a character mapped display is, rougly speaking, 64
  times faster than updating an equivalent bitmap display, since you only
  modify groups of 8x8 pixels and not the single pixels. However, it was
  also much less versatile than a bitmap screen, since with only 256
  characters you had to do all of your graphics. To overcome this
  limitation, some special hardware graphics were used: "sprites". A sprite
  is essentially a bitmap, usually larger than a character, which can be
  placed anywhere on the screen (not limited to character boundaries) by
  just telling the hardware the coordinates. Moreover, sprites can be
  flipped along the major axis just by setting the appropriate bit. This
  saves precious memory, since you need only one copy of the image instead
  of four.

  What about colors? Well, the early machines had a limited palette (let's
  say 16 colors) and each character or sprite could not use all of them at
  the same time. Characters and sprites data would use 2 bits per pixel,
  allowing them to address four different colors. You then needed some way
  to tell to the hardware which, among the available colors, were the four
  colors. This was done using a "color attribute", which typically was an
  index for a lookup table.

  OK, after this brief and incomplete introduction, let's come to the
  purpose of this section: how to interpret the data which was stored in
  the graphic roms. Unfortunately, there is no easy answer: it depends on
  the hardware. The easiest way to find how data is encoded, is to start by
  making a bit by bit dump of the rom. You will usually be able to
  immediately recognize some pattern: if you are lucky, you will see
  letters and numbers right away, otherwise you will see something which
  looks like letters and numbers, but with halves switched, dilated, or
  something like that. You'll then have to find a way to put it all
  together to obtain our standard one byte per pixel representation. Two
  things to remember:
  - keep in mind that every pixel has typically two (or more) bits
    associated with it, and they are not necessarily near to each other.
  - characters might be rotated 90 degrees. That's because many games used a
    tube rotated 90 degrees. Think how your monitor would look like if you
    put it on its side ;-)

  After you have successfully decoded the characters, you have to decode
  the sprites. This is usually more difficult, because sprites are larger,
  maybe have more colors, and are more difficult to recognize when they are
  messed up, since they are pure graphics without letters and numbers.
  However, with some work you'll hopefully be able to work them out as
  well. As a rule of thumb, the sprites should be encoded in a way not too
  dissimilar from the characters.

***************************************************************************/

#include <string.h>
#include "Z80.h"
#include "gfxdecod.h"
#include "machine.h"
#include "osdepend.h"
#include "roms.h"

int             graphicsmode = VESA;
int             MAXCHARS;
int             MAXSPRITES;
int             MAXPALETTES;
unsigned char   pal[3 * 256];
BITMAP         *chars[1024][16];// All the characters - maximum possible
BITMAP         *sprites[256][16];   // All the sprites - maximum possible
BITMAP         *piccie;

void            decodechars(unsigned char *CHARRAM)
{

    byte            byte1 = 0;
    byte            byte2 = 0;
    byte            tempbyte;
    word            offset;
    word            y;
    int MAXTELLER;
    int             teller,
                    no;
    int             charsets;
    int             count1,
                    count2,
                    count3;
    // Reserve room for characters

    for (count2 = 0; count2 < MAXPALETTES; count2++)
        for (count1 = 0; count1 < MAXCHARS; count1++)
        {                       // Allocate memory for
            chars[count1][count2] = create_bitmap(8, 8);    // charset
            clear(chars[count1][count2]);
        }

    if (gameemulated == DKONG)
        charsets = 3;
    else if ((gameemulated == MOONCRST) || (gameemulated == BAGMAN))
        charsets = 2;
    else
        charsets = 1;

    if (gameemulated == BAGMAN)
       MAXTELLER = 512;
    else 
       MAXTELLER = 256;

    for (count3 = 0; count3 < charsets; count3++)
        for (count2 = 0; count2 < MAXPALETTES; count2++)
            for (teller = 0; teller < MAXTELLER; teller++)
            {

               offset = teller * 8;

                if (gameemulated == BAGMAN)

                    no = teller + count3 * 512;
                else
                    no = teller + count3 * 256;

                for (y = 0; y < 8; y++)
                {

                    if (gameemulated != BAGMAN)
                    {
                        byte1 = CHARRAM[count3 * 0x1000 + offset + y];
                        byte2 = CHARRAM[count3 * 0x1000 + offset + y + 0x800];
                    }
                    else
                    {
                        byte1 = CHARRAM[count3 * 0x2000 + offset + y];
                        byte2 = CHARRAM[count3 * 0x2000 + offset + y + 0x1000];
                    };

                    tempbyte = byte1 & 0x80;
                    tempbyte = tempbyte >> 6;
                    tempbyte = tempbyte + ((byte2 & 0x80) >> 7);
                    if (tempbyte)
                        chars[no][count2]->line[0][7 - y] = count2 * 4 + tempbyte;

                    tempbyte = byte1 & 0x40;
                    tempbyte = tempbyte >> 5;
                    tempbyte = tempbyte + ((byte2 & 0x40) >> 6);
                    if (tempbyte)
                        chars[no][count2]->line[1][7 - y] = count2 * 4 + tempbyte;

                    tempbyte = byte1 & 0x20;
                    tempbyte = tempbyte >> 4;
                    tempbyte = tempbyte + ((byte2 & 0x20) >> 5);
                    if (tempbyte)
                        chars[no][count2]->line[2][7 - y] = count2 * 4 + tempbyte;

                    tempbyte = byte1 & 0x10;
                    tempbyte = tempbyte >> 3;
                    tempbyte = tempbyte + ((byte2 & 0x10) >> 4);
                    if (tempbyte)
                        chars[no][count2]->line[3][7 - y] = count2 * 4 + tempbyte;


                    tempbyte = byte1 & 0x08;
                    tempbyte = tempbyte >> 2;
                    tempbyte = tempbyte + ((byte2 & 0x8) >> 3);
                    if (tempbyte)
                        chars[no][count2]->line[4][7 - y] = count2 * 4 + tempbyte;

                    tempbyte = byte1 & 0x04;
                    tempbyte = tempbyte >> 1;
                    tempbyte = tempbyte + ((byte2 & 0x4) >> 2);
                    if (tempbyte)
                        chars[no][count2]->line[5][7 - y] = count2 * 4 + tempbyte;

                    tempbyte = byte1 & 0x02;
                    tempbyte = tempbyte + ((byte2 & 0x2) >> 1);
                    if (tempbyte)
                        chars[no][count2]->line[6][7 - y] = count2 * 4 + tempbyte;

                    tempbyte = byte1 & 0x01;
                    tempbyte = tempbyte << 1;
                    tempbyte = tempbyte + (byte2 & 0x1);
                    if (tempbyte)
                        chars[no][count2]->line[7][7 - y] = count2 * 4 + tempbyte;

                }
            }

}

void            decodesprites(unsigned char *CHARRAM)
{
    int             count1,
                    count2;

    // Reserve room for sprites

    for (count2 = 0; count2 < MAXPALETTES; count2++)
        for (count1 = 0; count1 < MAXSPRITES; count1++)
        {
            sprites[count1][count2] = create_bitmap(16, 16);    // Allocate memory for
            // sprites
        }

    // Build sprites from the characters
    // Games use 256 characters and 64 sprites.

    // But BAGMAN uses 512 characters and 128 sprites

  if (gameemulated != DKONG)
  {
    for (count2 = 0; count2 < MAXPALETTES; count2++)
        for (count1 = 0; count1 < MAXSPRITES; count1++)
        {
            blit(chars[count1 * 4][count2], sprites[count1][count2], 0, 0, 8, 0, 8, 8);
            blit(chars[count1 * 4 + 1][count2], sprites[count1][count2], 0, 0, 8, 8, 8, 8);
            blit(chars[count1 * 4 + 2][count2], sprites[count1][count2], 0, 0, 0, 0, 8, 8);
            blit(chars[count1 * 4 + 3][count2], sprites[count1][count2], 0, 0, 0, 8, 8, 8);
        }
   }
   else
   {
    for (count2 = 0; count2 < MAXPALETTES; count2++)
        for (count1 = 0; count1 < MAXSPRITES; count1++)
        {
            blit(chars[256+(count1 * 2)][count2], sprites[count1][count2], 0, 0, 8, 0, 8, 8);
            blit(chars[512+(count1 * 2)][count2], sprites[count1][count2], 0, 0, 8, 8, 8, 8);
            blit(chars[257+(count1 * 2)][count2], sprites[count1][count2], 0, 0, 0, 0, 8, 8);
            blit(chars[513+(count1 * 2)][count2], sprites[count1][count2], 0, 0, 0, 8, 8, 8);
        }
    }
}
