require 'json'
require 'fileutils'
require 'pathname'
require_relative 'archive.rb'

ARCHIVE_TIMEOUT_THRESHOLD = 30 * 86400 #seconds

def initialize
  @data_path = File.join(__dir__, "unpack.json")
  @archives = File.exist?(@data_path) ? load(@data_path) : Array.new
  @archives_directory = "cache/scripts/unpack"
  @i18n = JSON.parse(File.read(File.join(__dir__, "i18n.json"), :encoding => 'UTF-8'))

  @on_post_remove = $romstation.database.game_file.on_post_remove
  $romstation.database.game_file.on_post_remove = -> (game_file) {
    archive = @archives.find { |obj| obj.id == game_file.id }
    delete_archive(archive) if archive

    @on_post_remove&.call(game_file)
  }

  @on_init = $romstation.runtime.on_init
  $romstation.runtime.on_init = -> {
    timestamp = Time.now.to_i
    @archives.each { |archive|
      delete_archive(archive) if timestamp - archive.timestamp >= ARCHIVE_TIMEOUT_THRESHOLD
    }

    @on_init&.call
  }

  @on_stop = $romstation.runtime.on_stop
  $romstation.runtime.on_stop = -> {
    save(@data_path)

    @on_stop&.call
  }

  @game_profile_path = $romstation.system.get_property "game.profile.path"
  $romstation.system.set_property("game.profile.path", -> (ctx) {
    if ctx.game_profile
      archive = extract(ctx)
      return File.expand_path(archive.output_full_path) if archive
    end

    @game_profile_path&.call(ctx)
  })

  @game_profile_path_parent = $romstation.system.get_property "game.profile.path.parent"
  $romstation.system.set_property("game.profile.path.parent", -> (ctx) {
    if ctx.game_profile
      archive = extract(ctx)
      return File.dirname(archive.output_full_path) if archive
    end

    @game_profile_path_parent&.call(ctx)
  })
end

def save(path)
  File.write(path, JSON.pretty_generate(@archives))
end

def load(path)
  begin
    data = JSON.parse(File.read(path))
    data.map { |archive| Archive.new(archive["id"], archive["source"], archive["output_dir"], archive["output_path"], archive["timestamp"]) }
  rescue
    Array.new
  end
end

def translate(key)
  @i18n[$romstation.i18n.locale][key]
end

def delete_archive(archive)
  FileUtils.remove_dir(archive.output_dir, true) if File.exist?(archive.output_dir)
  @archives.delete(archive)
end

def fix_file_metas(path)
  dir = File.expand_path("..", path)
  basename = File.basename(path)

  Dir.chdir(dir) do
    file_content = File.read(basename)
    begin
      if /FILE "(?<file>[^"]+)" BINARY/i =~ file_content
        unless File.exist?(file)
          files = Dir.glob("*.{bin,iso}")
          file_content.gsub!(file, files.first) if files
          File.write(basename, file_content)
        end
      end
    rescue => exception
      puts exception
    end
  end
end

def unpacked?(game_file)
  archive = @archives.find { |obj| obj.id == game_file.id }

  if archive
    if File.exist?(archive.output_dir)
      return true
    else
      @archives.delete(archive)
    end
  end

  false
end

def unzip(source, target)
  result = nil
  $romstation.runtime.run_and_wait {
    result = $romstation.dialog.unzip(source, target)
  }

  result
end

def unrar(source, target)
  result = nil
  $romstation.runtime.run_and_wait {
    result = $romstation.dialog.task { |task|
      task.update_title translate("task.unpacking")
      FileUtils.mkdir_p target
      IO.popen([File.join(__dir__, "bin/win/unrar/UnRAR.exe"), "x", source, target]) { |io|
        begin
          while (string = io.readpartial(2048)) != nil || !task.cancelled?
            case string
            when /(?<value>\d+)%/
              task.update_progress($LAST_MATCH_INFO['value'].to_i, 100)
            when /Extracting\s{2}(?<filename>.+\.\w{3})/
              task.update_message Pathname.new($LAST_MATCH_INFO['filename']).relative_path_from(Pathname.new(target)).to_s
            end
          end
        rescue => exception
          puts exception
        end
      }
    }
  }

  result
end

def extract(ctx)
  meta = ctx.emulator_profile.metas["unpack"]
  json = JSON.parse(meta) if meta
  return unless json && json["formats"].include?(File.extname(ctx.game_profile.path).downcase)

  unless unpacked?(ctx.game_profile.game_file)
    id = ctx.game_profile.game_file.id
    source = ctx.game_profile.path
    target = File.expand_path(id.to_s, @archives_directory)
    case json["formats"].find { |i| i == File.extname(ctx.game_profile.path).downcase }
    when ".zip"
      return unless unzip(source, target)
    when ".rar"
      return unless unrar(source, target)
    else
      return
    end
    @archives << Archive.new(id, source, target)
  end

  archive = @archives.find { |obj| obj.id == ctx.game_profile.game_file.id }
  archive.update_timestamp
  files = []
  Dir.chdir(archive.output_dir) do
    files = Dir["**/*"].reject { |file| !json["extensions"].include?(".*") && !json["extensions"].include?(File.extname(file).downcase) }
  end

  path = nil
  case files.length
  when 0
    $romstation.runtime.run_and_wait {
      $romstation.dialog.error_alert(translate("file.choice.invalid.alert.header"), translate("file.choice.invalid.alert.content").gsub("${0}", json["extensions"].to_s))
    }
  when 1
    path = files.first
  else
    $romstation.runtime.run_and_wait {
      path = $romstation.dialog.combo_box_choice(translate("file.choice.dialog.header"), translate("file.choice.dialog.content"), archive.output_path, files.to_java)
    }
  end

  archive.output_path = path
  if archive.output_path
    fix_file_metas(archive.output_full_path) if File.extname(archive.output_path) == ".cue"
    archive
  end
end