;========================================================================================
;  fputest.s
;  Copyright (C) 2003-2019 Makoto Kamada
;
;  This file is part of the XEiJ (X68000 Emulator in Java).
;  You can use, modify and redistribute the XEiJ if the conditions are met.
;  Read the XEiJ License for more details.
;  https://stdkmd.net/xeij/
;========================================================================================

	.include	doscall.mac
	.include	iocscall.mac

	.cpu	68000


;--------------------------------------------------------------------------------
;	定数
;--------------------------------------------------------------------------------

MI	equ	1<<27
ZE	equ	1<<26
IN	equ	1<<25
NA	equ	1<<24
BS	equ	1<<15
SN	equ	1<<14
OE	equ	1<<13
OF	equ	1<<12
UF	equ	1<<11
DZ	equ	1<<10
X2	equ	1<<9
X1	equ	1<<8
AV	equ	1<<7
AO	equ	1<<6
AU	equ	1<<5
AZ	equ	1<<4
AX	equ	1<<3


;--------------------------------------------------------------------------------
;	マクロ
;--------------------------------------------------------------------------------

leamsg	.macro	p0,an
	.data
@msg:
	.dc.b	p0,0
	.text
	lea.l	@msg,an
	.endm

peamsg	.macro	p0,p1,p2,p3,p4,p5,p6,p7,p8,p9,q0,q1,q2,q3,q4,q5,q6,q7,q8,q9
	.data
@msg:
	.dc.b	p0,p1,p2,p3,p4,p5,p6,p7,p8,p9,q0,q1,q2,q3,q4,q5,q6,q7,q8,q9,0
	.text
	pea.l	@msg
	.endm

putmsg	.macro	p0,p1,p2,p3,p4,p5,p6,p7,p8,p9,q0,q1,q2,q3,q4,q5,q6,q7,q8,q9
	peamsg	p0,p1,p2,p3,p4,p5,p6,p7,p8,p9,q0,q1,q2,q3,q4,q5,q6,q7,q8,q9
	jbsr	printstr
	addq.l	#4,sp
	.endm

putstr	.macro	p0
	move.l	p0,-(sp)
	jbsr	printstr
	addq.l	#4,sp
	.endm

putchr	.macro	p0
	move.b	p0,-(sp)
	jbsr	printchr
	addq.l	#2,sp
	.endm

putlong	.macro	p0
	move.l	p0,-(sp)
	jbsr	printlong
	addq.l	#4,sp
	.endm

putdec	.macro	p0
	move.l	p0,-(sp)
	jbsr	printdec
	addq.l	#4,sp
	.endm

putdecz2	.macro	p0
	pea.l	2.w
	move.l	p0,-(sp)
	jbsr	printdecz
	addq.l	#8,sp
	.endm

putdecz4	.macro	p0
	pea.l	4.w
	move.l	p0,-(sp)
	jbsr	printdecz
	addq.l	#8,sp
	.endm

putfix	.macro	p0,p1
	move.b	p1,-(sp)
	move.l	p0,-(sp)
	jbsr	printfix
	addq.l	#6,sp
	.endm

puthex2	.macro	p0
	move.b	p0,-(sp)
	jbsr	printhex2
	addq.l	#2,sp
	.endm

puthex4	.macro	p0
	move.w	p0,-(sp)
	jbsr	printhex4
	addq.l	#2,sp
	.endm

puthex8	.macro	p0
	move.l	p0,-(sp)
	jbsr	printhex8
	addq.l	#4,sp
	.endm

puthex16	.macro	p0,p1
	puthex8	p0
	putchr	#','
	puthex8	p1
	.endm

puthex24	.macro	p0,p1,p2
	puthex8	p0
	putchr	#','
	puthex8	p1
	putchr	#','
	puthex8	p2
	.endm

putcrlf	.macro
	jbsr	printcrlf
	.endm

putdate	.macro
	jbsr	printdate
	.endm


;--------------------------------------------------------------------------------
;	メイン
;--------------------------------------------------------------------------------
	.text
	.even
main::
;------------------------------------------------
;bssが確保されているか確認する
	lea.l	(16,a0),a0
	suba.l	a0,a1
	movem.l	a0-a1,-(sp)
	DOS	_SETBLOCK
	addq.l	#8,sp
	tst.l	d0
	bmi	exit
;------------------------------------------------
;スタックエリアを設定する
	lea.l	stack_area_end,sp	;スタックエリアの末尾
;------------------------------------------------
;FPUの種類を確認する
	jbsr	fpu_check
	beq	exit
;------------------------------------------------
;コマンドラインをコピーする
	pea.l	1(a2)
	jbsr	option_start
	addq.l	#4,sp
	beq	exit
;------------------------------------------------
;アボートの準備をする
	move.l	sp,abort_sp
	move.w	#_CTRLVC,-(sp)
	DOS	_INTVCG
	addq.l	#2,sp
	move.l	d0,abort_ctrlvc		;元の_CTRLVC
	move.w	#_ERRJVC,-(sp)
	DOS	_INTVCG
	addq.l	#2,sp
	move.l	d0,abort_errjvc		;元の_ERRJVC
	pea.l	abort
	move.w	#_CTRLVC,-(sp)
	DOS	_INTVCS
	addq.l	#6,sp
	pea.l	abort
	move.w	#_ERRJVC,-(sp)
	DOS	_INTVCS
	addq.l	#6,sp
;------------------------------------------------
;ベクタを変更する
	pea.l	trapv_routine		;TRAPV/TRAPcc/FTRAPccルーチン
	move.w	#7,-(sp)
	DOS	_INTVCS
	addq.l	#6,sp
	move.l	d0,trapv_vector		;元のTRAPV/TRAPcc/FTRAPccベクタ
;------------------------------------------------
;ロギングを開始する
	jbsr	logging_start
	beq	exit
;------------------------------------------------
;統計を開始する
	jbsr	statistics_start
;------------------------------------------------
;開始メッセージ
	putmsg	'begin: '
	putdate
	putcrlf
;------------------------------------------------
;圧縮された間接データを展開する
	jbsr	indirect_start
;------------------------------------------------
;全体ループ開始
loop::
;------------------------------------------------
;引数を確認する
	jbsr	option_loop
	beq	loopend
;------------------------------------------------
;テストを実行する
	jbsr	fabs_test
	jbsr	facos_test
	jbsr	fadd_test
	jbsr	fasin_test
	jbsr	fatan_test
	jbsr	fatanh_test
	jbsr	fbccl060_test
	jbsr	fbccl88x_test
	jbsr	fbccw060_test
	jbsr	fbccw88x_test
	jbsr	fcmp_test
	jbsr	fcos_test
	jbsr	fcosh_test
	jbsr	fdabs_test
	jbsr	fdadd_test
	jbsr	fdbcc060_test
	jbsr	fdbcc88x_test
	jbsr	fddiv_test
	jbsr	fdiv_test
	jbsr	fdmove_test
	jbsr	fdmul_test
	jbsr	fdneg_test
	jbsr	fdsqrt_test
	jbsr	fdsub_test
	jbsr	fetox_test
	jbsr	fetoxm1_test
	jbsr	fgetexp_test
	jbsr	fgetman_test
	jbsr	fint_test
	jbsr	fintrz_test
	jbsr	flog10_test
	jbsr	flog2_test
	jbsr	flogn_test
	jbsr	flognp1_test
	jbsr	fmod_test
	jbsr	fmovebregto_test
	jbsr	fmovebtoreg_test
	jbsr	fmovedregto_test
	jbsr	fmovedtoreg_test
	jbsr	fmovelregto_test
	jbsr	fmoveltoreg_test
	jbsr	fmovepregto_test
	jbsr	fmoveptoreg_test
	jbsr	fmovesregto_test
	jbsr	fmovestoreg_test
	jbsr	fmovewregto_test
	jbsr	fmovewtoreg_test
	jbsr	fmovexregto_test
	jbsr	fmovextoreg_test
	jbsr	fmovecr881_test
	jbsr	fmovecr882_test
	jbsr	fmul_test
	jbsr	fneg_test
	jbsr	frem_test
	jbsr	fsabs_test
	jbsr	fsadd_test
	jbsr	fscale_test
	jbsr	fscc060_test
	jbsr	fscc88x_test
	jbsr	fsdiv_test
	jbsr	fsgldiv060_test
	jbsr	fsgldiv88x_test
	jbsr	fsglmul060_test
	jbsr	fsglmul88x_test
	jbsr	fsin_test
	jbsr	fsincos_test
	jbsr	fsinh_test
	jbsr	fsmove_test
	jbsr	fsmul_test
	jbsr	fsneg_test
	jbsr	fsqrt_test
	jbsr	fssqrt_test
	jbsr	fssub_test
	jbsr	fsub_test
	jbsr	ftan_test
	jbsr	ftanh_test
	jbsr	ftentox_test
	jbsr	ftrapcc060_test
	jbsr	ftrapcc88x_test
	jbsr	ftrapccl060_test
	jbsr	ftrapccl88x_test
	jbsr	ftrapccw060_test
	jbsr	ftrapccw88x_test
	jbsr	ftst_test
	jbsr	ftwotox_test

;------------------------------------------------
;次の引数に進む
	jbsr	option_next
	jbra	loop

;------------------------------------------------
;全体ループ終了
loopend::
;------------------------------------------------
;アボート処理
abort::
	movea.l	abort_sp,sp
	move.l	abort_ctrlvc,-(sp)	;元の_CTRLVC
	move.w	#_CTRLVC,-(sp)
	DOS	_INTVCS
	addq.l	#6,sp
	move.l	abort_errjvc,-(sp)	;元の_ERRJVC
	move.w	#_ERRJVC,-(sp)
	DOS	_INTVCS
	addq.l	#6,sp
;------------------------------------------------
;ベクタを復元する
	move.l	trapv_vector,d0		;元のTRAPV/TRAPcc/FTRAPccベクタ
	beq	@f
	move.l	d0,-(sp)
	move.w	#7,-(sp)
	DOS	_INTVCS
	addq.l	#6,sp
@@:
;------------------------------------------------
;統計を終了する
	jbsr	statistics_end
;------------------------------------------------
;終了メッセージ
	putmsg	'end: '
	putdate
	putcrlf
	putcrlf
;------------------------------------------------
;ロギングを終了する
	jbsr	logging_end
;------------------------------------------------
;プログラムを終了する
exit::
	DOS	_EXIT

;------------------------------------------------
;アボート処理
	.data
	.align	4
abort_sp::		.dc.l	0	;アボートするときのスタックポインタ
abort_ctrlvc::		.dc.l	0	;元の_CTRLVC
abort_errjvc::		.dc.l	0	;元の_ERRJVC

;------------------------------------------------
;スタックエリア
	.bss
	.align	4
stack_area::	.ds.b	1024*64		;スタックエリア
stack_area_end::			;スタックエリアの末尾


;--------------------------------------------------------------------------------
;	TRAPV/TRAPcc/FTRAPcc
;--------------------------------------------------------------------------------

;TRAPV/TRAPcc/FTRAPccルーチン
	.text
	.even
trapv_routine::
	addq.l	#1,trapv_occurred	;TRAPV/TRAPcc/FTRAPccが発生した
	rte

	.data
	.align	4
trapv_vector::		.dc.l	0	;元のTRAPV/TRAPcc/FTRAPccベクタ
trapv_occurred::	.dc.l	0	;0=TRAPV/TRAPcc/FTRAPccは発生していない,1=TRAPV/TRAPcc/FTRAPccが発生した


;--------------------------------------------------------------------------------
;	FPUの種類
;--------------------------------------------------------------------------------

MC68881		equ	1<<0
MC68882		equ	1<<1
MC68040		equ	1<<2
FPSP040		equ	1<<3
MC68060		equ	1<<4
FPSP060		equ	1<<5

;----------------------------------------------------------------
;successful=fpu_check()
;	FPUの種類を確認する
;>d0.l:0=failed,1=successful
;------------------------------------------------
	.cpu	68030
	.text
	.even
fpu_check::
	movem.l	d1-d2/a1,-(sp)
	lea.l	$0CBD.w,a1		;FPUの有無
	IOCS	_B_BPEEK
	tst.b	d0
	beq	80f			;FPUなし
	lea.l	$0CBC.w,a1		;MPUの種類
	IOCS	_B_BPEEK
	move.l	#MC68060,d2
	cmp.b	#6,d0
	beq	@f
	move.l	#MC68040,d2
	cmp.b	#4,d0
	beq	@f
	move.l	#MC68882,d2
	fmove.l	#0,fpcr
	fmovecr.x	#1,fp0
	fmove.x	fp0,-(sp)
	move.l	(sp)+,d0
	or.l	(sp)+,d0
	or.l	(sp)+,d0
	bne	@f
	move.l	#MC68881,d2
@@:
	move.l	d2,fpu_type		;FPUの種類
	move.l	#-1,fpu_last		;前回のFPUの種類
	moveq.l	#1,d0
99:
	movem.l	(sp)+,d1-d2/a1
	rts

80:
	putmsg	'no floating point unit',13,10
	moveq.l	#0,d0
	bra	99b
	.cpu	68000

	.bss
	.align	4
fpu_type::		.ds.l	1	;FPUの種類
fpu_last::		.ds.l	1	;前回のFPUの種類


;--------------------------------------------------------------------------------
;	動作モード
;--------------------------------------------------------------------------------

OPTION_MARGIN_DEFAULT	equ	1	;超越関数の許容誤差
OPTION_MARGIN_LIMIT	equ	11
OPTION_MARGIN_MASK	equ	$F
OPTION_MAXIMUM_DEFAULT	equ	100	;テスト毎の出力される結果の最大数
OPTION_MAXIMUM_LIMIT	equ	1000000
OPTION_MAXIMUM_MASK	equ	$FFFFF
OPTION_MAXIMUM_SHIFT	equ	4
OPTION_DESTINATION	equ	1<<24	;デスティネーションオペランドを調べる
OPTION_FAILED		equ	1<<25	;失敗したテストの結果を出力する
OPTION_NANS		equ	1<<26	;NaNの仮数部を調べる
OPTION_FPSP		equ	1<<27	;浮動小数点ソフトウェアパッケージで処理される命令をテストする
OPTION_STATUS		equ	1<<28	;ステータスレジスタを調べる
OPTION_SUCCESSFUL	equ	1<<29	;成功したテストの結果を出力する
OPTION_DEFAULT		equ	OPTION_STATUS+OPTION_FAILED+OPTION_DESTINATION+(OPTION_MAXIMUM_DEFAULT<<OPTION_MAXIMUM_SHIFT)+OPTION_MARGIN_DEFAULT

;----------------------------------------------------------------
;successful=option_start(cmdl)
;	全体ループの開始前に呼び出す
;	コマンドラインをコピーする
;	引数がひとつもないとき使用法を表示する
;	全体ループの準備をする
;<4(sp).l:コマンドライン
;>d0.l:0=failed,1=successful
;------------------------------------------------
	.offsym	0,_a6
_size:
regs	reg	d1-d2/a0-a2
_regs:	.ds.l	.sizeof.(regs)
_a6:	.ds.l	1
_pc:	.ds.l	1
_cmdl:	.ds.l	1
;------------------------------------------------
	.text
	.even
option_start::
	link.w	a6,#_size
	movem.l	regs,(_regs,a6)
;------------------------------------------------
;コマンドラインをコピーする
	movea.l	(_cmdl,a6),a0		;コマンドライン
	lea.l	option_args,a1		;単語列バッファ
	lea.l	option_argsend,a2	;単語列バッファの末尾
	bra	2f			;単語の間の空白を読み飛ばす

;単語が終わった
1:
	cmpa.l	a2,a1
	bcc	11f			;バッファが溢れた
	clr.b	(a1)+			;単語の末尾の0
;単語の間の空白を読み飛ばす
2:
	move.b	(a0)+,d0
	beq	10f			;単語が始まる前にコマンドラインが終わった
	cmp.b	#' ',d0
	bls	2b			;単語の間の空白を読み飛ばす
	cmp.b	#$22,d0
	beq	6f			;$22～$22の中の次の文字を読み出す
	cmp.b	#$27,d0
	beq	8f			;'～'の中の次の文字を読み出す
	cmp.b	#'a',d0
	blo	3f			;文字をバッファに書き込む
	cmp.b	#'z',d0
	bhi	3f			;文字をバッファに書き込む
	and.b	#$DF,d0			;大文字にする
;文字をバッファに書き込む
3:
	cmpa.l	a2,a1
	bcc	11f			;バッファが溢れた
	move.b	d0,(a1)+
;次の文字を読み出す
4:
	move.b	(a0)+,d0
	beq	9f			;単語の途中でコマンドラインが終わった
	cmp.b	#' ',d0
	bls	1b			;単語が終わった
	cmp.b	#$22,d0
	beq	6f			;$22～$22の中の次の文字を読み出す
	cmp.b	#$27,d0
	beq	8f			;'～'の中の次の文字を読み出す
	cmp.b	#'a',d0
	blo	3b			;文字をバッファに書き込む
	cmp.b	#'z',d0
	bhi	3b			;文字をバッファに書き込む
	and.b	#$DF,d0			;大文字にする
	bra	3b			;文字をバッファに書き込む

;$22～$22の中の文字をバッファに書き込む
5:
	cmpa.l	a2,a1
	bcc	11f			;バッファが溢れた
	move.b	d0,(a1)+
;$22～$22の中の次の文字を読み出す
6:
	move.b	(a0)+,d0
	beq	9f			;単語の途中でコマンドラインが終わった
	cmp.b	#$22,d0
	beq	4b			;次の文字を読み出す
	cmp.b	#'a',d0
	blo	5b			;$22～$22の中の文字をバッファに書き込む
	cmp.b	#'z',d0
	bhi	5b			;$22～$22の中の文字をバッファに書き込む
	and.b	#$DF,d0			;大文字にする
	bra	5b			;$22～$22の中の文字をバッファに書き込む

;'～'の中の文字をバッファに書き込む
7:
	cmpa.l	a2,a1
	bcc	11f			;バッファが溢れた
	move.b	d0,(a1)+
;'～'の中の次の文字を読み出す
8:
	move.b	(a0)+,d0
	beq	9f			;単語の途中でコマンドラインが終わった
	cmp.b	#$27,d0
	beq	4b			;次の文字を読み出す
	cmp.b	#'a',d0
	blo	7b			;'～'の中の文字をバッファに書き込む
	cmp.b	#'z',d0
	bhi	7b			;'～'の中の文字をバッファに書き込む
	and.b	#$DF,d0			;大文字にする
	bra	7b			;'～'の中の文字をバッファに書き込む

;単語の途中でコマンドラインが終わった
9:
	cmpa.l	a2,a1
	bcc	11f			;バッファが溢れた
	clr.b	(a1)+			;単語の末尾の0
;単語が始まる前にコマンドラインが終わった
10:
	cmpa.l	a2,a1
	bcc	11f			;バッファが溢れた
	clr.b	(a1)			;バッファの末尾の0
	bra	12f

;バッファが溢れた
11:
	putmsg	'too long command line',13,10
	moveq.l	#0,d0
	bra	99f

12:
;------------------------------------------------
;引数がひとつもないとき使用法を表示する
	tst.b	option_args
	bne	@f
	putstr	#option_usage
	moveq.l	#0,d0
	bra	99f

@@:
;------------------------------------------------
;全体ループの準備をする
	lea.l	option_args,a1		;単語列バッファ
	move.l	a1,option_word		;次の単語
	move.l	#OPTION_DEFAULT,option_mode	;動作モード
	move.l	#-1,option_lastmode	;前回の動作モード
	lea.l	option_donestart,a1	;実行済みのテストのIDと動作モードのリストの先頭
	move.l	a1,option_donepointer	;実行済みのテストのIDと動作モードのリストの末尾
;------------------------------------------------
98:
	moveq.l	#1,d0
99:
	movem.l	(_regs,a6),regs
	unlk	a6
	rts

;------------------------------------------------
;使用法
	.text
option_usage::
	.dc.b	'usage: fputest.x <mnemonic or option> ...',13,10
	.dc.b	'  FABS FACOS FADD ... -- Run tests selected by the first few letters of a mnemonic.',13,10
	.dc.b	'  all -- Run all tests.',13,10
	.dc.b	'  destination=0..[1] -- Check the destination operand.',13,10
	.dc.b	'  easy -- It',39,'s the same as margin=1 nans=0.',13,10
	.dc.b	'  failed=0..[1] -- Output the results of failed tests.',13,10
	.dc.b	'  fpsp=[0]..1 -- Test the instructions processed by the software package.',13,10
	.dc.b	'  hard -- It',39,'s the same as margin=0 nans=1.',13,10
	.dc.b	'  logging=0..[1] -- Output the results to ',34,'fputest.log',34,'.',13,10
	.dc.b	'  margin=0..[1]..11 -- The acceptable margin of error in transcendental functions.',13,10
	.dc.b	'  maximum=0..[100]..1000000 -- The maximum number of results outputted for each test.',13,10
	.dc.b	'  nans=[0]..1 -- Check the mantissa of NaNs.',13,10
	.dc.b	'  status=0..[1] -- Check the status register.',13,10
	.dc.b	'  stdout=0..[1] -- Output the results to the standard output.',13,10
	.dc.b	'  successful=[0]..1 -- Output the results of successful tests.',13,10
	.dc.b	0

;----------------------------------------------------------------
;option_loop()
;	全体ループの先頭で呼び出す
;	次の単語がなければ全体ループを終了する
;	次の単語によって動作モードを変更する
	.text
	.even
option_loop::
	movea.l	option_word,a2		;次の単語
	move.l	option_mode,d2		;動作モード
	bra	20f

;次の単語に進む
10:
	tst.b	(a2)+
	bne	10b
;------------------------------------------------
;次の単語がなければ全体ループを終了する
20:
	tst.b	(a2)
	beq	80f			;全体ループを終了する
;------------------------------------------------
;次の単語によって動作モードを更新する
;--------------------------------
;destination=0..1
	leamsg	'DESTINATION',a0
	movea.l	a2,a1
	bsr	option_stringstartwith
	bne	2f
	lea.l	(.sizeof.('DESTINATION'),a2),a0
	moveq.l	#1,d0
	jbsr	option_equal_n
	bmi	2f
	and.l	#.not.(OPTION_DESTINATION),d2
	neg.l	d0
	and.l	#OPTION_DESTINATION,d0
	or.l	d0,d2			;destination
	bra	10b			;次の単語に進む

2:
;--------------------------------
;easy=1
	leamsg	'EASY',a0
	movea.l	a2,a1
	bsr	option_stringequals
	bne	2f
	and.l	#.not.(OPTION_NANS+OPTION_MARGIN_MASK),d2
	or.l	#1,d2			;margin=1 nans=0
	bra	10b			;次の単語に進む

2:
;--------------------------------
;failed=0..1
	leamsg	'FAILED',a0
	movea.l	a2,a1
	bsr	option_stringstartwith
	bne	2f
	lea.l	(.sizeof.('FAILED'),a2),a0
	moveq.l	#1,d0
	jbsr	option_equal_n
	bmi	2f
	and.l	#.not.(OPTION_FAILED),d2
	neg.l	d0
	and.l	#OPTION_FAILED,d0
	or.l	d0,d2			;failed
	bra	10b			;次の単語に進む

2:
;--------------------------------
;fpsp=0..1
	leamsg	'FPSP',a0
	movea.l	a2,a1
	bsr	option_stringstartwith
	bne	2f
	lea.l	(.sizeof.('FPSP'),a2),a0
	moveq.l	#1,d0
	jbsr	option_equal_n
	bmi	2f
	bne	1f
	and.l	#.not.(OPTION_FPSP),d2	;fpsp=0
	cmpi.l	#FPSP040,fpu_type
	bne	@f
	move.l	#MC68040,fpu_type
	bra	10b			;次の単語に進む

@@:
	cmpi.l	#FPSP060,fpu_type
	bne	10b			;次の単語に進む
	move.l	#MC68060,fpu_type
	bra	10b			;次の単語に進む

1:
	or.l	#OPTION_FPSP,d2		;fpsp=1
	cmpi.l	#MC68040,fpu_type
	bne	@f
	move.l	#FPSP040,fpu_type
	bra	10b			;次の単語に進む

@@:
	cmpi.l	#MC68060,fpu_type
	bne	10b			;次の単語に進む
	move.l	#FPSP060,fpu_type
	bra	10b			;次の単語に進む

2:
;--------------------------------
;hard=1
	leamsg	'HARD',a0
	movea.l	a2,a1
	bsr	option_stringequals
	bne	2f
	and.l	#.not.(OPTION_NANS+OPTION_MARGIN_MASK),d2
	or.l	#OPTION_NANS+0,d2	;margin=0 nans=1
	bra	10b			;次の単語に進む

2:
;--------------------------------
;logging=0..1
	leamsg	'LOGGING',a0
	movea.l	a2,a1
	bsr	option_stringstartwith
	bne	2f
	lea.l	(.sizeof.('LOGGING'),a2),a0
	moveq.l	#1,d0
	jbsr	option_equal_n
	bmi	2f
	sne.b	logging_logging
	bra	10b			;次の単語に進む

2:
;--------------------------------
;margin=0..OPTION_MARGIN_LIMIT
	leamsg	'MARGIN=',a0
	movea.l	a2,a1
	bsr	option_stringstartwith
	bne	2f
	lea.l	(.sizeof.('MARGIN'),a2),a0
	moveq.l	#OPTION_MARGIN_LIMIT,d0
	jbsr	option_equal_n
	bmi	2f
	and.l	#.not.(OPTION_MARGIN_MASK),d2
	or.l	d0,d2			;margin
	bra	10b			;次の単語に進む

2:
;--------------------------------
;maxmum=0..OPTION_MAXIMUM_LIMIT
	leamsg	'MAXIMUM=',a0
	movea.l	a2,a1
	bsr	option_stringstartwith
	bne	2f
	lea.l	(.sizeof.('MAXIMUM'),a2),a0
	move.l	#OPTION_MAXIMUM_LIMIT,d0
	jbsr	option_equal_n
	bmi	2f
	and.l	#.not.(OPTION_MAXIMUM_MASK<<OPTION_MAXIMUM_SHIFT),d2
	lsl.l	#OPTION_MAXIMUM_SHIFT,d0
	or.l	d0,d2			;maximum
	bra	10b			;次の単語に進む

2:
;--------------------------------
;nans=0..1
	leamsg	'NANS',a0
	movea.l	a2,a1
	bsr	option_stringstartwith
	bne	2f
	lea.l	(.sizeof.('NANS'),a2),a0
	moveq.l	#1,d0
	jbsr	option_equal_n
	bmi	2f
	and.l	#.not.(OPTION_NANS),d2
	neg.l	d0
	and.l	#OPTION_NANS,d0
	or.l	d0,d2			;nans
	bra	10b			;次の単語に進む

2:
;--------------------------------
;successful=0..1
	leamsg	'SUCCESSFUL',a0
	movea.l	a2,a1
	bsr	option_stringstartwith
	bne	2f
	lea.l	(.sizeof.('SUCCESSFUL'),a2),a0
	moveq.l	#1,d0
	jbsr	option_equal_n
	bmi	2f
	and.l	#.not.(OPTION_SUCCESSFUL),d2
	neg.l	d0
	and.l	#OPTION_SUCCESSFUL,d0
	or.l	d0,d2			;successful
	bra	10b			;次の単語に進む

2:
;--------------------------------
;status=0..1
	leamsg	'STATUS',a0
	movea.l	a2,a1
	bsr	option_stringstartwith
	bne	2f
	lea.l	(.sizeof.('STATUS'),a2),a0
	moveq.l	#1,d0
	jbsr	option_equal_n
	bmi	2f
	and.l	#.not.(OPTION_STATUS),d2
	neg.l	d0
	and.l	#OPTION_STATUS,d0
	or.l	d0,d2			;status
	bra	10b			;次の単語に進む

2:
;--------------------------------
;stdout=0..1
	leamsg	'STDOUT',a0
	movea.l	a2,a1
	bsr	option_stringstartwith
	bne	2f
	lea.l	(.sizeof.('STDOUT'),a2),a0
	moveq.l	#1,d0
	jbsr	option_equal_n
	bmi	2f
	sne.b	logging_stdout
	bra	10b			;次の単語に進む

2:
;--------------------------------
	move.l	d2,option_mode		;動作モード
	move.l	a2,option_word		;次の単語
;
	bra	98f

;全体ループを終了する
80:
	moveq.l	#0,d0
	bra	99f

98:
	moveq.l	#1,d0
99:
	rts

;----------------------------------------------------------------
;mnemonic_start(mnemonic,fpu)
;	個々のテストの開始前に呼び出す
;	ニモニックを確認する
;	FPUを確認する
;	実行済みかどうか確認する
;	ニモニックのカウンタをクリアする
;	初回または動作モードが変更されたとき動作モードを表示する
;	バックグラウンドスレッドを回す
;<(sp).l:復帰アドレス。テストのIDとして使う
;<4(sp).l:テストのニモニックの条件
;<8(sp).l:テストのFPUの条件
;>d0.l:0=失敗,1=成功
;>eq=失敗,ne=成功
;------------------------------------------------
	.offsym	0,_a6
_size:
regs	reg	d1-d2/a0-a2
_regs:	.ds.l	.sizeof.(regs)
_a6:	.ds.l	1
_pc:	.ds.l	1
_mnem:	.ds.l	1
_fpu:	.ds.l	1
;------------------------------------------------
	.text
	.even
mnemonic_start::
	link.w	a6,#_size
	movem.l	regs,(_regs,a6)
;------------------------------------------------
;ニモニックを確認する
	leamsg	'ALL',a0
	movea.l	option_word,a1		;次の単語
	bsr	option_stringequals
	beq	@f			;allはすべてのニモニックと一致する
	movea.l	option_word,a0		;次の単語
	movea.l	(_mnem,a6),a1		;テストのニモニックの条件
	bsr	option_stringstartwith
	bne	97f			;ニモニックが一致しなかった。失敗
@@:
;------------------------------------------------
;FPUを確認する
	move.l	(_fpu,a6),d0		;テストのFPUの条件
	and.l	fpu_type,d0	;FPUの種類
	beq	97f			;FPUが一致しなかった。失敗
;------------------------------------------------
;実行済みかどうか確認する
	move.l	(_pc,a6),d0		;テストのID
	move.l	option_mode,d1		;動作モード
	lea.l	option_donestart,a0	;実行済みのテストのIDと動作モードのリストの先頭
	movea.l	option_donepointer,a1	;実行済みのテストのIDと動作モードのリストの末尾
	bra	3f

1:
	cmp.l	(a0),d0			;テストのID
	bne	2f
	cmp.l	4(a0),d1		;動作モード
	beq	97f			;実行済みだった。失敗
2:
	addq.l	#8,a0
3:
	cmpa.l	a1,a0
	blo	1b
;実行済みではなかった
	move.l	d0,(a1)+		;実行済みのテストのID
	move.l	d1,(a1)+		;実行済みの動作モード
	move.l	a1,option_donepointer
;------------------------------------------------
;ニモニックのカウンタをクリアする
	clr.l	mnemonic_counter
	clr.l	statistics_tested
	clr.l	statistics_failed
;------------------------------------------------
;初回はFPUの種類を表示する
	move.l	fpu_type,d2	;FPUの種類
	cmp.l	fpu_last,d2	;前回のFPUの種類
	beq	3f
	move.l	d2,fpu_last
	putmsg	'fpu: '
;
	move.l	#MC68881,d0
	and.l	d2,d0
	beq	1f
	leamsg	'MC68881',a0
	bra	2f

1:
	move.l	#MC68882,d0
	and.l	d2,d0
	beq	1f
	leamsg	'MC68882',a0
	bra	2f

1:
	move.l	#MC68040,d0
	and.l	d2,d0
	beq	1f
	leamsg	'MC68040',a0
	bra	2f

1:
	move.l	#FPSP040,d0
	and.l	d2,d0
	beq	1f
	leamsg	'040FPSP',a0
	bra	2f

1:
	move.l	#MC68060,d0
	and.l	d2,d0
	beq	1f
	leamsg	'MC68060',a0
	bra	2f

1:
	move.l	#FPSP060,d0
	and.l	d2,d0
	beq	1f
	leamsg	'060FPSP',a0
	bra	2f

1:
	leamsg	'???',a0
2:
	putstr	a0
	putcrlf
3:
;------------------------------------------------
;初回または動作モードが変更されたとき動作モードを表示する
	move.l	option_mode,d2		;動作モード
	cmp.l	option_lastmode,d2	;前回の動作モード
	beq	3f
	move.l	d2,option_lastmode
	putmsg	'option:'
;--------------------------------
	putmsg	' destination'
	move.l	d2,d0
	and.l	#OPTION_DESTINATION,d0
	bne	@f
	putmsg	'=0'
@@:
;--------------------------------
	move.l	d2,d0
	and.l	#OPTION_NANS+OPTION_MARGIN_MASK,d0
	cmp.l	#1,d0			;margin=1 nans=0
	bne	@f
	putmsg	' easy'
@@:
;--------------------------------
	putmsg	' failed'
	move.l	d2,d0
	and.l	#OPTION_FAILED,d0
	bne	@f
	putmsg	'=0'
@@:
;--------------------------------
	putmsg	' fpsp'
	move.l	d2,d0
	and.l	#OPTION_FPSP,d0
	bne	@f
	putmsg	'=0'
@@:
;--------------------------------
	move.l	d2,d0
	and.l	#OPTION_NANS+OPTION_MARGIN_MASK,d0
	cmp.l	#OPTION_NANS+0,d0	;margin=0 nans=1
	bne	@f
	putmsg	' hard'
@@:
;--------------------------------
	putmsg	' margin='
	move.l	d2,d0
	and.l	#OPTION_MARGIN_MASK,d0
	putdec	d0
;--------------------------------
	putmsg	' maximum='
	move.l	d2,d0
	lsr.l	#OPTION_MAXIMUM_SHIFT,d0
	and.l	#OPTION_MAXIMUM_MASK,d0
	putdec	d0
;--------------------------------
	putmsg	' nans'
	move.l	d2,d0
	and.l	#OPTION_NANS,d0
	bne	@f
	putmsg	'=0'
@@:
;--------------------------------
	putmsg	' status'
	move.l	d2,d0
	and.l	#OPTION_STATUS,d0
	bne	@f
	putmsg	'=0'
@@:
;--------------------------------
	putmsg	' successful'
	move.l	d2,d0
	and.l	#OPTION_SUCCESSFUL,d0
	bne	@f
	putmsg	'=0'
@@:
;------------------------------------------------
	putcrlf
3:
;------------------------------------------------
;バックグラウンドスレッドを回す
	move.l	fpu_type,d0	;FPUの種類
	and.l	#MC68881+MC68882+MC68040+FPSP040+MC68060+FPSP060,d0
	beq	@f
	.cpu	68030
	fnop
	.cpu	68000
@@:
	DOS	_CHANGE_PR
;------------------------------------------------
;成功
98:
	moveq.l	#1,d0
99:
	movem.l	(_regs,a6),regs
	unlk	a6
	rts

;失敗
97:
	moveq.l	#0,d0
	bra	99b

	.data
	.align	4
mnemonic_counter::	.dc.l	1	;ニモニックのカウンタ

;----------------------------------------------------------------
;mnemonic_end()
;	ニモニックの成績を表示する
;	ニモニックの成績を総合成績に加える
	.text
	.even
mnemonic_end::
	movem.l	d0-d1,-(sp)
	move.l	statistics_tested,d0
	move.l	statistics_failed,d1
	putmsg	'score: '
	movem.l	d0-d1,-(sp)
	jbsr	statistics_output
	addq.l	#8,sp
	add.l	d0,statistics_ttl_tested
	add.l	d1,statistics_ttl_failed
	movem.l	(sp)+,d0-d1
	rts

;----------------------------------------------------------------
;option_next()
;	全体ループの末尾で呼び出す
;	次の単語に進む
	.text
	.even
option_next::
	move.l	a0,-(sp)
;------------------------------------------------
;次の単語に進む
	movea.l	option_word,a0		;次の単語
@@:
	tst.b	(a0)+
	bne	@b
	move.l	a0,option_word		;次の単語
;------------------------------------------------
	movea.l	(sp)+,a0
	rts

;----------------------------------------------------------------
;	文字列比較
;<a0.l:文字列
;<a1.l:文字列
;>eq=一致,ne=不一致
	.text
	.even
option_stringequals::
1:
	tst.b	(a0)
	beq	2f
	cmpm.b	(a1)+,(a0)+
	beq	1b
	rts

2:
	tst.b	(a1)
	rts

;----------------------------------------------------------------
;	先頭文字列比較
;<a0.l:先頭文字列
;<a1.l:文字列
;>eq=一致,ne=不一致
	.text
	.even
option_stringstartwith::
1:
	tst.b	(a0)
	beq	2f
	cmpm.b	(a1)+,(a0)+
	beq	1b
2:
	rts

;----------------------------------------------------------------
;	'=N'を読み取る
;<d0.l:上限
;<a0.l:'='の位置
;>d0.l:N。'='がないとき1。エラーのとき-1
	.text
	.even
option_equal_n::
	movem.l	d1-d2/a0,-(sp)
	move.l	d0,d2
	moveq.l	#1,d0
	tst.b	(a0)
	beq	9f			;最初に文字がない
	cmpi.b	#'=',(a0)+
	bne	8f			;最初の文字が'='でない
	move.b	(a0)+,d0
	beq	8f			;'='の後に文字がない
	sub.b	#'0',d0
	blo	8f			;'='の後の文字が数字でない
	cmp.b	#9,d0
	bhi	8f			;'='の後の文字が数字でない
1:
	moveq.l	#0,d1
	move.b	(a0)+,d1
	beq	2f			;次の文字がない
	sub.b	#'0',d1
	blo	8f			;次の文字が数字でない
	cmp.b	#9,d1
	bhi	8f			;次の文字が数字でない
	add.l	d0,d0
	add.l	d0,d1
	lsl.l	#2,d0
	add.l	d1,d0
	bra	1b

2:
	cmp.l	d2,d0
	bls	9f
8:
	moveq.l	#-1,d0
9:
	movem.l	(sp)+,d1-d2/a0
	tst.l	d0
	rts

;----------------------------------------------------------------

	.bss
option_args::		.ds.b	4096	;単語列バッファ。単語,0,単語,0,…,単語,0,0
option_argsend::

	.bss
	.align	4
option_word::		.ds.l	1	;次の単語
option_mode::		.ds.l	1	;動作モード
option_lastmode::	.ds.l	1	;動作モード

	.bss
	.align	4
option_donepointer::	.ds.l	1	;実行済みのテストのIDと動作モードの書き込み位置
option_donestart::	.ds.l	2*1024*32	;実行済みのテストのIDと動作モードのリスト
option_doneend::


;--------------------------------------------------------------------------------
;	logging
;--------------------------------------------------------------------------------

;------------------------------------------------
;logging_start()
;>d0.l:0=failed,1=successful
	.text
	.even
logging_start::
	move.l	d1,-(sp)
;
	st.b	logging_logging
	st.b	logging_stdout
;open log file
	move.w	#2,-(sp)		;read and write
	peamsg	'fputest.log'		;log file name
	DOS	_OPEN
	addq.l	#6,sp
	tst.l	d0
	bpl	@f
	move.w	#$0020,-(sp)		;file
	peamsg	'fputest.log'		;log file name
	DOS	_CREATE
	addq.l	#6,sp
	tst.l	d0
	bpl	@f
	putmsg	'cannot open fputest.log to write'
	moveq.l	#0,d0
	bra	99f

@@:
	move.w	d0,d1			;log file handle
	move.w	d1,logging_file_handle
;check device type
	move.w	d1,-(sp)		;log file handle
	clr.w	-(sp)			;device information
	DOS	_IOCTRL
	addq.l	#4,sp
	tst.b	d0
	bmi	1f			;character device
;block device or remote device
;seek end
	move.w	#2,-(sp)		;SEEK_END
	clr.l	-(sp)
	move.w	d1,-(sp)		;log file handle
	DOS	_SEEK
	addq.l	#8,sp
	bra	2f

;character device
1:
;check output status
	move.w	d1,-(sp)		;log file handle
	move.w	#7,-(sp)		;output status
	DOS	_IOCTRL
	addq.l	#4,sp
	tst.l	d0
	bne	@f
	putmsg	'cannot output to fputest.log'
	moveq.l	#0,d0
	bra	99f

@@:
2:
	move.l	#logging_cache_start,logging_cache_pointer
;
98:
	moveq.l	#1,d0
99:
	move.l	(sp)+,d1
	tst.l	d0
	rts

;------------------------------------------------
;logging_write(buffer,length)
;<4(sp).l:buffer
;<8(sp).l:length
regs		reg	d0-d3/a0-a3
	.offsym	0,_a6
_size:
_regs:	.ds.b	.sizeof.(regs)
_a6:	.ds.l	1
_pc:	.ds.l	1
_buff:	.ds.l	1
_leng:	.ds.l	1
	.text
	.even
logging_write::
	link.w	a6,#_size
	movem.l	regs,(_regs,a6)
;ログファイルに出力する
	tst.b	logging_logging
	beq	19f
	move.w	logging_file_handle,d1	;log file handle
	bmi	19f
;出力データをキャッシュにコピーする
	move.l	(_leng,a6),d2		;出力データの残りのバイト数
	movea.l	(_buff,a6),a0		;出力データの読み出し位置
	movea.l	logging_cache_pointer,a1	;キャッシュの書き込み位置
	lea.l	logging_cache_end,a3	;キャッシュの終了アドレス。固定
	bra	17f

10:
	move.l	a3,d3			;キャッシュの終了アドレス
	sub.l	a1,d3			;キャッシュの残りのバイト数。キャッシュの終了アドレス-キャッシュの書き込み位置
	cmp.l	d2,d3			;キャッシュの残りのバイト数<=>出力データの残りのバイト数
	blo	11f
	move.l	d2,d3			;出力データの残りのバイト数
11:
;<d3.l:出力データをキャッシュにコピーするバイト数。出力データの残りのバイト数とキャッシュの残りのバイト数の少ない方。0ではない
	move.l	d3,d0			;出力データをキャッシュにコピーするバイト数
	subq.l	#1,d0
	swap.w	d0
12:
	swap.w	d0
13:
	move.b	(a0)+,(a1)+		;出力データの読み出し位置からキャッシュの書き込み位置へ
	dbra	d0,13b
	swap.w	d0
	dbra	d0,12b
	cmpa.l	a3,a1			;キャッシュの書き込み位置<=>キャッシュの終了アドレス
	blo	14f
;キャッシュが一杯になった
	lea.l	logging_cache_start,a1	;キャッシュの開始アドレス
	movea.l	a3,a2			;キャッシュの終了アドレス
	suba.l	a1,a2			;キャッシュの全体のバイト数
	movem.l	a1-a2,-(sp)
	move.w	d1,-(sp)		;log file handle
	DOS	_WRITE
	lea.l	10(sp),sp
14:
	move.l	a1,logging_cache_pointer	;キャッシュの書き込み位置
	sub.l	d3,d2			;出力データの残りのバイト数
17:
	tst.l	d2			;出力データの残りのバイト数
	bne	10b
19:
;標準出力に出力する
	tst.b	logging_stdout
	beq	29f
	move.l	(_leng,a6),-(sp)	;length
	move.l	(_buff,a6),-(sp)	;buffer
	move.w	#1,-(sp)		;stdout
	DOS	_WRITE
	lea.l	10(sp),sp
29:
	movem.l	(_regs,a6),regs
	unlk	a6
	rts

;------------------------------------------------
;logging_end()
logging_end::
	movem.l	d0-d1/a1-a2,-(sp)
;close log file
	move.w	logging_file_handle,d1	;log file handle
	bmi	19f
	lea.l	logging_cache_start,a1	;キャッシュの開始アドレス
	movea.l	logging_cache_pointer,a2	;キャッシュの書き込み位置
	suba.l	a1,a2			;キャッシュの現在のバイト数
	move.l	a2,d0
	beq	11f
	movem.l	a1-a2,-(sp)
	move.w	d1,-(sp)		;log file handle
	DOS	_WRITE
	lea.l	10(sp),sp
11:
	move.l	a1,logging_cache_pointer	;キャッシュの書き込み位置
	move.w	d1,-(sp)		;log file handle
	DOS	_CLOSE
	addq.l	#2,sp
	move.w	#-1,logging_file_handle
19:
;
	sf.b	logging_logging
	st.b	logging_stdout
;
	movem.l	(sp)+,d0-d1/a1-a2
	rts

;------------------------------------------------

	.data
logging_logging::	.dc.b	0	;ファイルに出力する
logging_stdout::	.dc.b	-1	;標準出力に出力する
	.even
logging_file_handle::	.dc.w	-1	;log file handle

	.bss
	.align	4
logging_cache_pointer::	.ds.l	1	;キャッシュの書き込み位置
logging_cache_start::	.ds.b	1024*64	;キャッシュ
logging_cache_end::


;--------------------------------------------------------------------------------
;	statistics
;--------------------------------------------------------------------------------

;--------------------------------------------------------------------------------
	.text
	.even
statistics_start::
	clr.l	statistics_ttl_tested
	clr.l	statistics_ttl_failed
	rts

;--------------------------------------------------------------------------------
;statistics_update()
;<(4,sp).l:0=failed,1=successful
;>d0.l:0=not output,1=output
	.text
	.even
statistics_update::
	addq.l	#1,statistics_tested
	tst.l	(4,sp)
	bne	1f			;successful
;failed
	addq.l	#1,statistics_failed
	move.l	#OPTION_FAILED,d0
	bra	2f

;successful
1:
	move.l	#OPTION_SUCCESSFUL,d0
2:
	and.l	option_mode,d0
	beq	8f			;not output
	addq.l	#1,mnemonic_counter
	move.l	option_mode,d0
	lsr.l	#OPTION_MAXIMUM_SHIFT,d0
	and.l	#OPTION_MAXIMUM_MASK,d0
	cmp.l	mnemonic_counter,d0
	blo	8f			;not output
;output
	move.l	mnemonic_counter,d0
	putdec	d0
	putmsg	': ',
	moveq.l	#1,d0
9:
	rts

;not output
8:
	moveq.l	#0,d0
	bra	9b

;--------------------------------------------------------------------------------
	.text
	.even
statistics_end::
	putmsg	'total: '
	move.l	statistics_ttl_failed,-(sp)
	move.l	statistics_ttl_tested,-(sp)
	jbsr	statistics_output
	addq.l	#8,sp
	rts

;--------------------------------------------------------------------------------
;<(4,sp).l:tested
;<(8,sp).l:failed
	.text
	.even
statistics_output::
	movem.l	d0-d6,-(sp)
	move.l	(4*7+4,sp),d6		;d6=tested
	move.l	(4*7+8,sp),d4		;d4=failed
	putmsg	'tested='
	move.l	d6,d5			;tested
	sub.l	d4,d5			;d5=successful=tested-failed
	putdec	d6			;tested
	tst.l	d6			;tested
	beq	8f			;no tests were performed
	move.l	#10000,d0
	move.l	d5,d1			;successful
	jbsr	mull			;d0:d1=10000*successful
	moveq.l	#0,d2
	move.l	d6,d3			;d2:d3=tested
	jbsr	divq			;d1=10000*successful/tested
	move.l	#10000,d0
	sub.l	d1,d0			;d0=10000-10000*successful/tested
	putmsg	' failed='
	putdec	d4			;failed
	putchr	#'('
	putfix	d0,#2			;10000-10000*successful/tested
	putmsg	'%) successful='
	putdec	d5			;successful
	putchr	#'('
	putfix	d1,#2			;10000*successful/tested
	putmsg	'%)'
8:
	putcrlf
	movem.l	(sp)+,d0-d6
	rts

	.bss

	.align	4
statistics_tested::	.ds.l	1	;number of tests
statistics_failed::	.ds.l	1	;number of failed tests
statistics_ttl_tested::	.ds.l	1	;number of total tests
statistics_ttl_failed::	.ds.l	1	;number of total failed tests


;--------------------------------------------------------------------------------
;	結果を比較する
;--------------------------------------------------------------------------------

;--------------------------------------------------------------------------------
;successful=test_status()
;	statusを比較する
;<(4,sp).l:actual status
;<(8,sp).l:expected status
;>d0.l:0=failed,1=successful
	.cpu	68030
	.offsym	0,_a6
_size:
_a6:	.ds.l	1
_pc:	.ds.l	1
_asta:	.ds.l	1	;actual status
_esta:	.ds.l	1	;expected status
	.text
	.even
test_status::
	link.w	a6,#_size
;------------------------------------------------
;status
	move.l	option_mode,d0
	and.l	#OPTION_STATUS,d0
	beq	19f			;statusをテストしない。statusが一致
;statusをテストする
	move.l	(_asta,a6),d0		;actual status
	cmp.l	(_esta,a6),d0		;expected status
	bne	97f			;statusが一致しない。失敗
;statusが一致
19:
;------------------------------------------------
;成功
98:
	moveq.l	#1,d0
99:
	unlk	a6
	rts

;失敗
97:
	moveq.l	#0,d0
	bra	99b
	.cpu	68000

;--------------------------------------------------------------------------------
;successful=test_single()
;	singleの結果を比較する。誤差は許容しない
;<(4,sp).s:actual result
;<(8,sp).l:actual status
;<(12,sp).s:expected result
;<(16,sp).l:expected status
;>d0.l:0=failed,1=successful
	.cpu	68030
	.offsym	0,_a6
_size:
_a6:	.ds.l	1
_pc:	.ds.l	1
_ares:	.ds.s	1	;actual result
_asta:	.ds.l	1	;actual status
_eres:	.ds.s	1	;expected result
_esta:	.ds.l	1	;expected status
	.text
	.even
test_single::
	link.w	a6,#_size
;------------------------------------------------
;status
	move.l	option_mode,d0
	and.l	#OPTION_STATUS,d0
	beq	19f			;statusをテストしない。statusが一致
;statusをテストする
	move.l	(_asta,a6),d0		;actual status
	cmp.l	(_esta,a6),d0		;expected status
	bne	97f			;statusが一致しない。失敗
;statusが一致
19:
;------------------------------------------------
;result
	move.l	option_mode,d0
	and.l	#OPTION_DESTINATION,d0
	beq	39f			;resultをテストしない。resultが一致
;resultをテストする
	move.l	(_ares,a6),d0		;actual result
	cmp.l	(_eres,a6),d0		;expected result
	bne	97f			;resultが一致しない。失敗
;resultが一致
39:
;------------------------------------------------
;成功
98:
	moveq.l	#1,d0
99:
	unlk	a6
	rts

;失敗
97:
	moveq.l	#0,d0
	bra	99b
	.cpu	68000

;--------------------------------------------------------------------------------
;successful=test_double()
;	doubleの結果を比較する。誤差は許容しない
;<(4,sp).d:actual result
;<(12,sp).l:actual status
;<(16,sp).d:expected result
;<(24,sp).l:expected status
;>d0.l:0=failed,1=successful
	.cpu	68030
	.offsym	0,_a6
_size:
_a6:	.ds.l	1
_pc:	.ds.l	1
_ares:	.ds.d	1	;actual result
_asta:	.ds.l	1	;actual status
_eres:	.ds.d	1	;expected result
_esta:	.ds.l	1	;expected status
	.text
	.even
test_double::
	link.w	a6,#_size
;------------------------------------------------
;status
	move.l	option_mode,d0
	and.l	#OPTION_STATUS,d0
	beq	19f			;statusをテストしない。statusが一致
;statusをテストする
	move.l	(_asta,a6),d0		;actual status
	cmp.l	(_esta,a6),d0		;expected status
	bne	97f			;statusが一致しない。失敗
;statusが一致
19:
;------------------------------------------------
;result
	move.l	option_mode,d0
	and.l	#OPTION_DESTINATION,d0
	beq	39f			;resultをテストしない。resultが一致
;resultをテストする
	move.l	(_ares,a6),d0		;actual result 1st
	cmp.l	(_eres,a6),d0		;expected result 1st
	bne	97f			;resultが一致しない。失敗
	move.l	(4+_ares,a6),d0		;actual result 2nd
	cmp.l	(4+_eres,a6),d0		;expected result 2nd
	bne	97f			;resultが一致しない。失敗
;resultが一致
39:
;------------------------------------------------
;成功
98:
	moveq.l	#1,d0
99:
	unlk	a6
	rts

;失敗
97:
	moveq.l	#0,d0
	bra	99b
	.cpu	68000

;--------------------------------------------------------------------------------
;successful=test_packed()
;	packedの結果を比較する。誤差は許容しない
;<(4,sp).p:actual result
;<(12,sp).l:actual status
;<(16,sp).p:expected result
;<(24,sp).l:expected status
;>d0.l:0=failed,1=successful
	.cpu	68030
	.offsym	0,_a6
_size:
_a6:	.ds.l	1
_pc:	.ds.l	1
_ares:	.ds.p	1	;actual result
_asta:	.ds.l	1	;actual status
_eres:	.ds.p	1	;expected result
_esta:	.ds.l	1	;expected status
	.text
	.even
test_packed::
	link.w	a6,#_size
;------------------------------------------------
;status
	move.l	option_mode,d0
	and.l	#OPTION_STATUS,d0
	beq	19f			;statusをテストしない。statusが一致
;statusをテストする
	move.l	(_asta,a6),d0		;actual status
	cmp.l	(_esta,a6),d0		;expected status
	bne	97f			;statusが一致しない。失敗
;statusが一致
19:
;------------------------------------------------
;result
	move.l	option_mode,d0
	and.l	#OPTION_DESTINATION,d0
	beq	39f			;resultをテストしない。resultが一致
;resultをテストする
	move.l	(_ares,a6),d0		;actual result 1st
	cmp.l	(_eres,a6),d0		;expected result 1st
	bne	97f			;resultが一致しない。失敗
	move.l	(4+_ares,a6),d0		;actual result 2nd
	cmp.l	(4+_eres,a6),d0		;expected result 2nd
	bne	97f			;resultが一致しない。失敗
	move.l	(8+_ares,a6),d0		;actual result 3rd
	cmp.l	(8+_eres,a6),d0		;expected result 3rd
	bne	97f			;resultが一致しない。失敗
;resultが一致
39:
;------------------------------------------------
;成功
98:
	moveq.l	#1,d0
99:
	unlk	a6
	rts

;失敗
97:
	moveq.l	#0,d0
	bra	99b
	.cpu	68000

;--------------------------------------------------------------------------------
;successful=test_extended()
;	extendedの結果を比較する。誤差を許容する
;<(4,sp).x:actual result
;<(16,sp).l:actual status
;<(20,sp).x:expected result
;<(32,sp).l:expected status
;<(36,sp).l:fpcr(rp<<6,-1=strict)
;>d0.l:0=failed,1=successful
	.cpu	68030
regs		reg	d1-d6
cregs		reg	fpcr/fpsr/fpiar
fregs		reg	fp0-fp2
	.offsym	0,_a6
_size:
_regs:	.ds.b	.sizeof.(regs)
_fregs:	.ds.b	.sizeof.(fregs)
_cregs:	.ds.b	.sizeof.(cregs)
_a6:	.ds.l	1
_pc:	.ds.l	1
_ares:	.ds.x	1	;actual result
_asta:	.ds.l	1	;actual status
_eres:	.ds.x	1	;expected result
_esta:	.ds.l	1	;expected status
_fpcr:	.ds.l	1	;fpcr(rp<<6,-1=strict)
	.text
	.even
test_extended::
	link.w	a6,#_size
	movem.l	regs,(_regs,a6)
	fmovem.l	cregs,(_cregs,a6)
	fmovem.x	fregs,(_fregs,a6)
;----------------------------------------------------------------
;status
	move.l	option_mode,d0
	and.l	#OPTION_STATUS,d0
	beq	19f			;statusをテストしない。statusが一致
;statusをテストする
	move.l	(_asta,a6),d0		;actual status
	cmp.l	(_esta,a6),d0		;expected status
	bne	97f			;statusが一致しない。失敗
;statusが一致
19:
;----------------------------------------------------------------
;result
	move.l	option_mode,d0
	and.l	#OPTION_DESTINATION,d0
	beq	39f			;resultをテストしない。resultが一致
;resultをテストする
	fmove.l	#0,fpcr
;------------------------------------------------
;expected resultが正規化数または非正規化数か調べる
	move.l	(_eres,a6),d0		;expected result 1st
	and.l	#$7FFF0000,d0
	cmp.l	#$7FFF0000,d0
	beq	30f			;expected resultが±InfまたはNaN。完全に一致しなければならない
	or.l	(4+_eres,a6),d0		;expected result 2nd
	or.l	(8+_eres,a6),d0		;expected result 3rd
	beq	30f			;expected resultが±0。完全に一致しなければならない
;expected resultは正規化数または非正規化数
;------------------------------------------------
;1ulpを求める
	move.l	(_fpcr,a6),d0		;fpcr(rp<<6,-1=strict)
	cmp.l	#-1,d0
	beq	30f			;strict。完全に一致しなければならない
	move.l	#$00000000,d4		;extendedの1ulp
	move.l	#$00000001,d5
	lsr.b	#6,d0			;rounding precision
	beq	49f			;extended
;singleまたはdouble
	subq.b	#2,d0
	bcc	43f			;double
;--------------------------------
;single
	move.l	#$00000100,d4		;singleの1ulp
	move.l	#$00000000,d5
	movem.l	(_ares,a6),d1-d3	;actual result
	and.l	#$7FFF0000,d1
	cmp.l	#$7FFF0000,d1
	beq	49f			;actual resultが±InfまたはNaN
;actual resultが±0または非正規化数または正規化数
	move.l	d2,d0
	or.l	d3,d0
	beq	49f			;actual resultが±0
;actual resultが非正規化数または正規化数
;actual resultがsingleに丸められていることを確認する
	tst.l	d2
	bmi	42f			;actual resultが正規化数
;actual resultが非正規化数
41:
	add.l	d3,d3
	addx.l	d2,d2
	bpl	41b			;actual resultが非正規化数
;actual resultが正規化数
42:
	and.l	#$000000FF,d2
	or.l	d3,d2
	bne	97f			;actual resultがsingleに丸められていない。失敗
	bra	49f

;--------------------------------
;double
43:
	move.l	#$00000000,d4		;doubleの1ulp
	move.l	#$00000800,d5
	movem.l	(_ares,a6),d1-d3	;actual result
	and.l	#$7FFF0000,d1
	cmp.l	#$7FFF0000,d1
	beq	49f			;actual resultが±InfまたはNaN
;actual resultが±0または非正規化数または正規化数
	move.l	d2,d0
	or.l	d3,d0
	beq	49f			;actual resultが±0
;actual resultが非正規化数または正規化数
;actual resultがdoubleに丸められていることを確認する
	tst.l	d2
	bmi	45f			;actual resultが正規化数
;actual resultが非正規化数
44:
	add.l	d3,d3
	addx.l	d2,d2
	bpl	44b			;actual resultが非正規化数
;actual resultが正規化数
45:
	and.l	#$000007FF,d3
	bne	97f			;actual resultがdoubleに丸められていない。失敗
49:
;<d4-d5:1ulp
;------------------------------------------------
;マージンを求める
	move.l	option_mode,d0
	and.l	#OPTION_MARGIN_MASK,d0	;margin
	bne	1f
	moveq.l	#0,d4
	moveq.l	#0,d5
	bra	2f

1:
	subq.w	#1,d0
	lsl.l	d0,d4			;extended,single,doubleのいずれも下位ワードが1bit立っているだけなのでこれだけでよい
	lsl.l	d0,d5
2:
;<d4-d5:マージン
;------------------------------------------------
;許容範囲の絶対値の下限を求める
	movem.l	(_eres,a6),d1-d3	;expected result
	tst.l	d2
	bpl	22f			;非正規化数
;正規化数
	sub.l	d5,d3
	subx.l	d4,d2
	bmi	23f			;指数部が変わらない
	move.l	d1,d0
	and.l	#$7FFF0000,d0
	beq	23f			;指数部が0のときは正規化数が非正規化数になる
	sub.l	#$00010000,d1		;指数部を1減らす
	add.l	d3,d3			;仮数部を左にずらす
	addx.l	d2,d2
	bra	23f

;非正規化数
22:
	sub.l	d5,d3
	subx.l	d4,d2
	bpl	23f
	and.l	#$80000000,d1		;引けなかったとき許容範囲の絶対値の下限は0
	move.l	#$00000000,d2
	move.l	#$00000000,d3
23:
	movem.l	d1-d3,-(sp)
	fmove.x	(sp)+,fp1		;許容範囲の絶対値の下限
;<fp1.x:許容範囲の絶対値の下限
;------------------------------------------------
;許容範囲の絶対値の上限を求める
	movem.l	(_eres,a6),d1-d3	;expected result
	tst.l	d2
	bpl	25f			;非正規化数
;正規化数
	add.l	d5,d3
	addx.l	d4,d2
	bmi	26f			;指数部が変わらない
	move.l	d1,d0
	and.l	#$7FFF0000,d0
	cmp.l	#$7FFE0000,d0
	beq	24f			;指数部が$7FFEのときは±Infになる
	add.l	#$00010000,d1		;指数部を1増やす
	roxr.l	#1,d2			;仮数部を右にずらす
	roxr.l	#1,d3
	or.l	#$80000000,d2
	bra	26f

24:
	or.l	#$7FFF0000,d1
	moveq.l	#0,d2
	moveq.l	#0,d3
	bra	26f

;非正規化数
25:
	add.l	d5,d3			;仮数部にマージンを加える。非正規化数が正規化数になる場合がある
	addx.l	d4,d2
26:
	movem.l	d1-d3,-(sp)
	fmove.x	(sp)+,fp2		;許容範囲の絶対値の上限
;<fp2.x:許容範囲の絶対値の上限
;------------------------------------------------
;下限<=上限にする。マージンが0のときは下限==上限になる。下限と上限は範囲に含まれる
	fcmp.x	fp1,fp2
	fbge	27f
	fmove.x	fp2,fp0
	fmove.x	fp1,fp2
	fmove.x	fp0,fp1
27:
;<fp1.x:許容範囲の下限
;<fp2.x:許容範囲の上限
;------------------------------------------------
;比較する
	fmove.x	(_ares,a6),fp0		;actual result
	fcmp.x	fp1,fp0
	fbult	97f			;actual resultがNaNまたは小さすぎる。失敗
	fcmp.x	fp2,fp0
	fbugt	97f			;actual resultがNaNまたは大きすぎる。失敗
	bra	98f			;成功

;------------------------------------------------
;完全に一致しなければならない
30:
	movem.l	(_eres,a6),d1-d3	;expected result
	movem.l	(_ares,a6),d4-d6	;actual result
	move.l	#OPTION_NANS,d0
	and.l	option_mode,d0
	bne	35f			;NaNの仮数部を比較する
;NaNの仮数部を無視する
;------------------------------------------------
;expected resultがNaNのとき仮数をすべて1にする
	move.l	d1,d0
	and.l	#$7FFF0000,d0
	cmp.l	#$7FFF0000,d0
	bne	32f			;±InfまたはNaNではない
	tst.l	d2
	bne	31f			;NaN
	tst.l	d3
	beq	32f			;±Inf
;NaN
31:
	move.l	#$FFFFFFFF,d2		;NaNの仮数部をすべて1にする
	move.l	#$FFFFFFFF,d3
32:
;------------------------------------------------
;actual resultがNaNのとき仮数をすべて1にする
	move.l	d4,d0
	and.l	#$7FFF0000,d0
	cmp.l	#$7FFF0000,d0
	bne	34f			;±InfまたはNaNではない
	tst.l	d5
	bne	33f			;NaN
	tst.l	d6
	beq	34f			;±Inf
;NaN
33:
	move.l	#$FFFFFFFF,d5		;NaNの仮数部をすべて1にする
	move.l	#$FFFFFFFF,d6
34:
;------------------------------------------------
;比較する
35:
	cmp.l	d1,d4
	bne	97f			;resultが一致しない。失敗
	cmp.l	d2,d5
	bne	97f			;resultが一致しない。失敗
	cmp.l	d3,d6
	bne	97f			;resultが一致しない。失敗
;resultが一致
39:
;------------------------------------------------
;成功
98:
	moveq.l	#1,d0
99:
	fmovem.x	(_fregs,a6),fregs
	fmovem.l	(_cregs,a6),cregs
	movem.l	(_regs,a6),regs
	unlk	a6
	rts

;失敗
97:
	moveq.l	#0,d0
	bra	99b
	.cpu	68000


;--------------------------------------------------------------------------------
;	結果を出力する
;--------------------------------------------------------------------------------

;--------------------------------------------------------------------------------
;output_status()
;	output status
;<(4,sp).l:actual status
;<(8,sp).l:expected status
;<(12,sp).l:0=failed,1=successful
	.cpu	68030
regs		reg	d3-d4/a3
	.offsym	0,_a6
_size:
_regs:	.ds.b	.sizeof.(regs)
_a6:	.ds.l	1
_pc:	.ds.l	1
_asta:	.ds.l	1	;actual status
_esta:	.ds.l	1	;expected status
_succ:	.ds.l	1	;0=failed,1=successful
	.text
	.even
output_status::
	link.w	a6,#_size
	movem.l	regs,(_regs,a6)
	movea.l	(_esta,a6),a3		;expected status
	move.l	(_asta,a6),d3		;actual status
;successfulかどうかに関わらず完全に一致しているときだけexpectedを省略する
	moveq.l	#1,d4			;0=完全に一致してはいない,1=完全に一致している
	cmp.l	a3,d3
	beq	2f
	moveq.l	#0,d4			;0=完全に一致してはいない,1=完全に一致している
;expected
	putchr	#9
	move.l	a3,-(sp)
	jbsr	printfpsr
	addq.l	#4,sp
	putmsg	9,';expected',13,10
2:
;actual
	putchr	#9
	move.l	d3,-(sp)
	jbsr	printfpsr
	addq.l	#4,sp
	tst.l	d4
	beq	3f			;完全に一致してはいない
;完全に一致している
	putmsg	9,';'
	bra	4f

3:
;完全に一致してはいない
	putmsg	9,';actual ... '
4:
	tst.l	(_succ,a6)
	beq	5f			;failed
;successful
	putmsg	'OK',13,10
	bra	6f

5:
;failed
	putmsg	'ERROR',13,10
6:
;
99:
	movem.l	(_regs,a6),regs
	unlk	a6
	rts
	.cpu	68000

;--------------------------------------------------------------------------------
;output_single()
;	output result and status
;<(4,sp).s:actual result
;<(8,sp).l:actual status
;<(12,sp).s:expected result
;<(16,sp).l:expected status
;<(20,sp).l:0=failed,1=successful
	.cpu	68030
regs		reg	d0/d3-d4/a0/a3
	.offsym	0,_a6
_size:
_regs:	.ds.b	.sizeof.(regs)
_a6:	.ds.l	1
_pc:	.ds.l	1
_ares:	.ds.s	1	;actual result
_asta:	.ds.l	1	;actual status
_eres:	.ds.s	1	;expected result
_esta:	.ds.l	1	;expected status
_succ:	.ds.l	1	;0=failed,1=successful
	.text
	.even
output_single::
	link.w	a6,#_size
	movem.l	regs,(_regs,a6)
	movea.l	(_eres,a6),a0		;expected result
	movea.l	(_esta,a6),a3		;expected status
	move.l	(_ares,a6),d0		;actual result
	move.l	(_asta,a6),d3		;actual status
;successfulかどうかに関わらず完全に一致しているときだけexpectedを省略する
	moveq.l	#1,d4			;0=完全に一致してはいない,1=完全に一致している
	cmp.l	a0,d0
	bne	1f
	cmp.l	a3,d3
	beq	2f
1:
	moveq.l	#0,d4			;0=完全に一致してはいない,1=完全に一致している
;expected
	putchr	#9
	puthex8	a0
	putchr	#','
	move.l	a3,-(sp)
	jbsr	printfpsr
	addq.l	#4,sp
	putmsg	9,';expected',13,10
2:
;actual
	putchr	#9
	puthex8	d0
	putchr	#','
	move.l	d3,-(sp)
	jbsr	printfpsr
	addq.l	#4,sp
	tst.l	d4
	beq	3f			;完全に一致してはいない
;完全に一致している
	putmsg	9,';'
	bra	4f

3:
;完全に一致してはいない
	putmsg	9,';actual ... '
4:
	tst.l	(_succ,a6)
	beq	5f			;failed
;successful
	putmsg	'OK',13,10
	bra	6f

5:
;failed
	putmsg	'ERROR',13,10
6:
;
99:
	movem.l	(_regs,a6),regs
	unlk	a6
	rts
	.cpu	68000

;--------------------------------------------------------------------------------
;output_double()
;	output result and status
;<(4,sp).d:actual result
;<(12,sp).l:actual status
;<(16,sp).d:expected result
;<(24,sp).l:expected status
;<(28,sp).l:0=failed,1=successful
	.cpu	68030
regs		reg	d0-d1/d3-d4/a0-a1/a3
	.offsym	0,_a6
_size:
_regs:	.ds.b	.sizeof.(regs)
_a6:	.ds.l	1
_pc:	.ds.l	1
_ares:	.ds.d	1	;actual result
_asta:	.ds.l	1	;actual status
_eres:	.ds.d	1	;expected result
_esta:	.ds.l	1	;expected status
_succ:	.ds.l	1	;0=failed,1=successful
	.text
	.even
output_double::
	link.w	a6,#_size
	movem.l	regs,(_regs,a6)
	movem.l	(_eres,a6),a0-a1	;expected result
	movea.l	(_esta,a6),a3		;expected status
	movem.l	(_ares,a6),d0-d1	;actual result
	move.l	(_asta,a6),d3		;actual status
;successfulかどうかに関わらず完全に一致しているときだけexpectedを省略する
	moveq.l	#1,d4			;0=完全に一致してはいない,1=完全に一致している
	cmp.l	a0,d0
	bne	1f
	cmp.l	a1,d1
	bne	1f
	cmp.l	a3,d3
	beq	2f
1:
	moveq.l	#0,d4			;0=完全に一致してはいない,1=完全に一致している
;expected
	putchr	#9
	puthex16	a0,a1
	putchr	#','
	move.l	a3,-(sp)
	jbsr	printfpsr
	addq.l	#4,sp
	putmsg	9,';expected',13,10
2:
;actual
	putchr	#9
	puthex16	d0,d1
	putchr	#','
	move.l	d3,-(sp)
	jbsr	printfpsr
	addq.l	#4,sp
	tst.l	d4
	beq	3f			;完全に一致してはいない
;完全に一致している
	putmsg	9,';'
	bra	4f

3:
;完全に一致してはいない
	putmsg	9,';actual ... '
4:
	tst.l	(_succ,a6)
	beq	5f			;failed
;successful
	putmsg	'OK',13,10
	bra	6f

5:
;failed
	putmsg	'ERROR',13,10
6:
;
	movem.l	(_regs,a6),regs
	unlk	a6
	rts
	.cpu	68000

;--------------------------------------------------------------------------------
;output_packed()
;	output result and status
;<(4,sp).p:actual result
;<(16,sp).l:actual status
;<(20,sp).p:expected result
;<(32,sp).l:expected status
;<(36,sp).l:0=failed,1=successful
	.cpu	68030
regs		reg	d0-d4/a0-a3
	.offsym	0,_a6
_size:
_regs:	.ds.b	.sizeof.(regs)
_a6:	.ds.l	1
_pc:	.ds.l	1
_ares:	.ds.p	1	;actual result
_asta:	.ds.l	1	;actual status
_eres:	.ds.p	1	;expected result
_esta:	.ds.l	1	;expected status
_succ:	.ds.l	1	;0=failed,1=successful
	.text
	.even
output_packed::
	link.w	a6,#_size
	movem.l	regs,(_regs,a6)
	movem.l	(_eres,a6),a0-a2	;expected result
	movea.l	(_esta,a6),a3		;expected status
	movem.l	(_ares,a6),d0-d2	;actual result
	move.l	(_asta,a6),d3		;actual status
;successfulかどうかに関わらず完全に一致しているときだけexpectedを省略する
	moveq.l	#1,d4			;0=完全に一致してはいない,1=完全に一致している
	cmp.l	a0,d0
	bne	1f
	cmp.l	a1,d1
	bne	1f
	cmp.l	a2,d2
	bne	1f
	cmp.l	a3,d3
	beq	2f
1:
	moveq.l	#0,d4			;0=完全に一致してはいない,1=完全に一致している
;expected
	putchr	#9
	puthex24	a0,a1,a2
	putchr	#','
	move.l	a3,-(sp)
	jbsr	printfpsr
	addq.l	#4,sp
	putmsg	9,';expected',13,10
2:
;actual
	putchr	#9
	puthex24	d0,d1,d2
	putchr	#','
	move.l	d3,-(sp)
	jbsr	printfpsr
	addq.l	#4,sp
	tst.l	d4
	beq	3f			;完全に一致してはいない
;完全に一致している
	putmsg	9,';'
	bra	4f

3:
;完全に一致してはいない
	putmsg	9,';actual ... '
4:
	tst.l	(_succ,a6)
	beq	5f			;failed
;successful
	putmsg	'OK',13,10
	bra	6f

5:
;failed
	putmsg	'ERROR',13,10
6:
;
	movem.l	(_regs,a6),regs
	unlk	a6
	rts
	.cpu	68000

;--------------------------------------------------------------------------------
;output_extended()
;	output result and status
;<(4,sp).x:actual result
;<(16,sp).l:actual status
;<(20,sp).x:expected result
;<(32,sp).l:expected status
;<(36,sp).l:0=failed,1=successful
	.cpu	68030
regs		reg	d0-d4/a0-a3
	.offsym	0,_a6
_size:
_regs:	.ds.b	.sizeof.(regs)
_a6:	.ds.l	1
_pc:	.ds.l	1
_ares:	.ds.x	1	;actual result
_asta:	.ds.l	1	;actual status
_eres:	.ds.x	1	;expected result
_esta:	.ds.l	1	;expected status
_succ:	.ds.l	1	;0=failed,1=successful
	.text
	.even
output_extended::
	link.w	a6,#_size
	movem.l	regs,(_regs,a6)
	movem.l	(_eres,a6),a0-a2	;expected result
	movea.l	(_esta,a6),a3		;expected status
	movem.l	(_ares,a6),d0-d2	;actual result
	move.l	(_asta,a6),d3		;actual status
;successfulかどうかに関わらず完全に一致しているときだけexpectedを省略する
	moveq.l	#1,d4			;0=完全に一致してはいない,1=完全に一致している
	cmp.l	a0,d0
	bne	1f
	cmp.l	a1,d1
	bne	1f
	cmp.l	a2,d2
	bne	1f
	cmp.l	a3,d3
	beq	2f
1:
	moveq.l	#0,d4			;0=完全に一致してはいない,1=完全に一致している
;expected
	putchr	#9
	puthex24	a0,a1,a2
	putchr	#','
	move.l	a3,-(sp)
	jbsr	printfpsr
	addq.l	#4,sp
	putmsg	9,';expected',13,10
2:
;actual
	putchr	#9
	puthex24	d0,d1,d2
	putchr	#','
	move.l	d3,-(sp)
	jbsr	printfpsr
	addq.l	#4,sp
	tst.l	d4
	beq	3f			;完全に一致してはいない
;完全に一致している
	putmsg	9,';'
	bra	4f

3:
;完全に一致してはいない
	putmsg	9,';actual ... '
4:
	tst.l	(_succ,a6)
	beq	5f			;failed
;successful
	putmsg	'OK',13,10
	bra	6f

5:
;failed
	putmsg	'ERROR',13,10
6:
;
	movem.l	(_regs,a6),regs
	unlk	a6
	rts
	.cpu	68000


;--------------------------------------------------------------------------------
;	出力サブルーチン
;--------------------------------------------------------------------------------

;----------------------------------------------------------------
;printfpcrrprm(fpcr)
;<(4,sp).l:fpcr
	.text
	.even
printfpcrrprm::
	movem.l	d0/a0,-(sp)
	move.l	(4*2+4,sp),d0		;fpcr
	and.w	#$00F0,d0		;rp,rm
	lsr.w	#4-2,d0
	lea.l	(10f,pc,d0.w),a0
	putstr	a0
	movem.l	(sp)+,d0/a0
	rts

	.align	4
10:
	.dc.l	'XRN'<<8|0
	.dc.l	'XRZ'<<8|0
	.dc.l	'XRM'<<8|0
	.dc.l	'XRP'<<8|0
	.dc.l	'SRN'<<8|0
	.dc.l	'SRZ'<<8|0
	.dc.l	'SRM'<<8|0
	.dc.l	'SRP'<<8|0
	.dc.l	'DRN'<<8|0
	.dc.l	'DRZ'<<8|0
	.dc.l	'DRM'<<8|0
	.dc.l	'DRP'<<8|0
	.dc.l	'DRN'<<8|0
	.dc.l	'DRZ'<<8|0
	.dc.l	'DRM'<<8|0
	.dc.l	'DRP'<<8|0

;----------------------------------------------------------------
;printfpsr
;<4(sp).l:fpsr
	.text
	.even
printfpsr::
	movem.l	d0/a0,-(sp)
	move.l	(4*2+4,sp),d0		;fpsr
	lea.l	(-128,sp),sp
	movea.l	sp,a0
	jbsr	fpsrstr
	putstr	sp
	lea.l	(128,sp),sp
	movem.l	(sp)+,d0/a0
	rts

;----------------------------------------------------------------
;fpsrstr
;<d0.l:fpsr
;<a0.l:buffer
;>a0.l:buffer
	.text
	.even
fpsrstr::
	movem.l	d0-d5/a1,-(sp)
	move.l	d0,d5			;fpsr
	sf.b	d4
;condition code byte
	lea.l	(10f,pc),a1
	moveq.l	#27,d1
1:
	move.b	(a1)+,d2
	move.b	(a1)+,d3
	btst.l	d1,d5
	beq	2f
	move.b	d2,(a0)+
	move.b	d3,(a0)+
	move.b	#'+',(a0)+
	st.b	d4
2:
	subq.w	#1,d1
	cmp.w	#24,d1
	bhs	1b
;quotient byte
	btst.l	#23,d5
	beq	1f
	move.b	#'(',(a0)+
	move.b	#'1',(a0)+
	move.b	#'<',(a0)+
	move.b	#'<',(a0)+
	move.b	#'2',(a0)+
	move.b	#'3',(a0)+
	move.b	#')',(a0)+
	move.b	#'+',(a0)+
	st.b	d4
1:
	move.l	d5,d0
	and.l	#$007F0000,d0
	beq	2f
	swap.w	d0
	move.b	#'(',(a0)+
	jbsr	decstr
	move.b	#'<',(a0)+
	move.b	#'<',(a0)+
	move.b	#'1',(a0)+
	move.b	#'6',(a0)+
	move.b	#')',(a0)+
	move.b	#'+',(a0)+
	st.b	d4
2:
;exception byte, accrued exception byte
	lea.l	(11f,pc),a1
	moveq.l	#15,d1
1:
	move.b	(a1)+,d2
	move.b	(a1)+,d3
	btst.l	d1,d5
	beq	2f
	move.b	d2,(a0)+
	move.b	d3,(a0)+
	move.b	#'+',(a0)+
	st.b	d4
2:
	subq.w	#1,d1
	cmp.w	#3,d1
	bhs	1b
	tst.b	d4
	beq	3f
	clr.b	-(a0)			;remove unnecessary '+'
	bra	8f

3:
	move.b	#'0',(a0)+
	clr.b	(a0)
8:
	movem.l	(sp)+,d0-d5/a1
	rts

10:
;		 27   26   25   24
	.dc.b	'MI','ZE','IN','NA'
11:
;		 15   14   13   12   11   10    9    8    7    6    5    4    3
	.dc.b	'BS','SN','OE','OF','UF','DZ','X2','X1','AV','AO','AU','AZ','AX'

;----------------------------------------------------------------
;printdate
;	yyyy-mm-ddThh:mm:ss+09:00
	.text
	.even
printdate::
	move.l	a0,-(sp)
	lea.l	(-28,sp),sp
	movea.l	sp,a0
	jbsr	datestr
	putstr	sp
	lea.l	(28,sp),sp
	movea.l	(sp)+,a0
	rts

;----------------------------------------------------------------
;datestr
;	yyyy-mm-ddThh:mm:ss+09:00
	.text
	.even
datestr::
	movem.l	d0-d3,-(sp)
;get date and time
	DOS	_GETDATE
	move.l	d0,d2
;<d2.l:date((dayofweek(0=sunday)<<16)+((year-1980)<<9)+(month<<5)+dayofmonth)
	DOS	_GETTIM2
	move.l	d0,d3
;<d3.l:time((hour<<16)+(minute<<8)+second)
;year
					;........ .....www yyyyyyym mmmddddd
	rol.w	#7,d2			;........ .....www mmmmdddd dyyyyyyy
	moveq.l	#$7F,d0			;________ ________ ________ _1111111
	and.w	d2,d0			;________ ________ ________ _yyyyyyy
	add.w	#1980,d0
	move.l	#4,d1
	jbsr	deczstr
	move.b	#'-',(a0)+
;month
	rol.w	#4,d2			;........ .....www dddddyyy yyyymmmm
	moveq.l	#$0F,d0			;________ ________ ________ ____1111
	and.w	d2,d0			;________ ________ ________ ____mmmm
	move.l	#2,d1
	jbsr	deczstr
	move.b	#'-',(a0)+
;dayofmonth
	rol.w	#5,d2			;........ .....www yyyyyyym mmmddddd
	moveq.l	#$1F,d0			;________ ________ ________ ___11111
	and.w	d2,d0			;________ ________ ________ ___ddddd
;	move.l	#2,d1
	jbsr	deczstr
	move.b	#'T',(a0)+
;hour
					;........ ...hhhhh ..mmmmmm ..ssssss
	swap.w	d3			;..mmmmmm ..ssssss ........ ...hhhhh
	moveq.l	#$1F,d0			;________ ________ ________ ___11111
	and.w	d3,d0			;________ ________ ________ ___hhhhh
;	move.l	#2,d1
	jbsr	deczstr
	move.b	#':',(a0)+
;minute
	rol.l	#8,d3			;..ssssss ........ ...hhhhh ..mmmmmm
	moveq.l	#$3F,d0			;________ ________ ________ __111111
	and.w	d3,d0			;________ ________ ________ __mmmmmm
;	move.l	#2,d1
	jbsr	deczstr
	move.b	#':',(a0)+
;second
	rol.l	#8,d3			;........ ...hhhhh ..mmmmmm ..ssssss
	moveq.l	#$3F,d0			;________ ________ ________ __111111
	and.w	d3,d0			;________ ________ ________ __ssssss
;	move.l	#2,d1
	jbsr	deczstr
	move.b	#'+',(a0)+
	move.b	#'0',(a0)+
	move.b	#'9',(a0)+
	move.b	#':',(a0)+
	move.b	#'0',(a0)+
	move.b	#'0',(a0)+
	clr.b	(a0)
	movem.l	(sp)+,d0-d3
	rts

;----------------------------------------------------------------
;printlong(number)
;	%d
;<(4,sp).l:number
	.text
	.even
printlong::
	move.l	(4,sp),-(sp)
	bpl	@f
	putchr	#'-'
	neg.l	(sp)
@@:
	jbsr	printdec
	addq.l	#4,sp
	rts

;----------------------------------------------------------------
;printdec(number)
;	%u
;<(4,sp).l:number
	.text
	.even
printdec::
	movem.l	d0/a0,-(sp)
	move.l	(4*2+4,sp),d0		;number
	lea.l	(-12,sp),sp
	movea.l	sp,a0
	jbsr	decstr
	putstr	sp
	lea.l	(12,sp),sp
	movem.l	(sp)+,d0/a0
	rts

;----------------------------------------------------------------
;printdecs(number,digits)
;	%*u
;<(4,sp).l:number
;<(8,sp).l:digits
	.text
	.even
printdecs::
	movem.l	d0-d2/a0,-(sp)
	movem.l	(4*4+4,sp),d0-d1	;number,digits
	moveq.l	#4,d2
	add.l	d1,d2
	and.w	#-4,d2
	suba.l	d2,sp
	movea.l	sp,a0
	jbsr	decsstr
	putstr	sp
	adda.l	d2,sp
	movem.l	(sp)+,d0-d2
	rts

;----------------------------------------------------------------
;printdecz(number,digits)
;	%0*u
;<(4,sp).l:number
;<(8,sp).l:digits
	.text
	.even
printdecz::
	movem.l	d0-d2/a0,-(sp)
	movem.l	(4*4+4,sp),d0-d1	;number,digits
	moveq.l	#4,d2
	add.l	d1,d2
	and.w	#-4,d2
	suba.l	d2,sp
	movea.l	sp,a0
	jbsr	deczstr
	putstr	sp
	adda.l	d2,sp
	movem.l	(sp)+,d0-d2
	rts

;----------------------------------------------------------------
;decstr
;	%u
;<d0.l:number
;<a0.l:buffer
;>a0.l:buffer
	.text
	.even
decstr::
	movem.l	d0-d2/a1,-(sp)
	tst.l	d0
	bne	1f
	move.b	#'0',(a0)+
	bra	5f

1:
	lea.l	baseten,a1
2:
	move.l	(a1)+,d1
	cmp.l	d1,d0
	blo	2b
3:
	moveq.l	#'0'-1,d2
4:
	addq.b	#1,d2
	sub.l	d1,d0
	bhs	4b
	add.l	d1,d0
	move.b	d2,(a0)+
	move.l	(a1)+,d1
	bne	3b
5:
	clr.b	(a0)
	movem.l	(sp)+,d0-d2/a1
	rts

;----------------------------------------------------------------
;decsstr
;	%*u
;<d0.l:number
;<d1.l:digits
;<a0.l:buffer
;>a0.l:buffer
	.text
	.even
decsstr::
	movem.l	d0-d2/a1-a2,-(sp)
	lea.l	baseten+4*10,a1
	movea.l	a1,a2
;remove zero digits
	tst.l	d1
	bne	@f
	moveq.l	#1,d1
@@:
;remove 11 or more digits
	cmp.l	#11,d1
	blo	3f
	sub.l	#11,d1
	swap.w	d1
1:
	swap.w	d1
2:
	move.b	#' ',(a0)+
	dbra	d1,2b
	swap.w	d1
	dbra	d1,1b
	moveq.l	#10,d1
3:
	lsl.w	#2,d1
	suba.w	d1,a2			;specified start point
;calculate minimum digits
	tst.l	d0
	beq	2f
	lea.l	(-4*10,a1),a1
1:
	cmp.l	(a1)+,d0
	blo	1b
2:
	subq.l	#4,a1			;required start point
;fill
	cmpa.l	a1,a2
	bhs	2f
1:
	move.b	#' ',(a0)+
	addq.l	#4,a2
	cmpa.l	a1,a2
	blo	1b
2:
;print decimal number
	move.l	(a1)+,d1
1:
	moveq.l	#'0'-1,d2
2:
	addq.b	#1,d2
	sub.l	d1,d0
	bhs	2b
	add.l	d1,d0
	move.b	d2,(a0)+
	move.l	(a1)+,d1
	bne	1b
	move.b	d1,(a0)			;0
	movem.l	(sp)+,d0-d2/a1-a2
	rts

;----------------------------------------------------------------
;deczstr
;	%0*u
;<d0.l:number
;<d1.l:digits
;<a0.l:buffer
;>a0.l:buffer
	.text
	.even
deczstr::
	movem.l	d0-d2/a1-a2,-(sp)
	lea.l	baseten+4*10,a1
	movea.l	a1,a2
;remove zero digits
	tst.l	d1
	bne	@f
	moveq.l	#1,d1
@@:
;remove 11 or more digits
	cmp.l	#11,d1
	blo	3f
	sub.l	#11,d1
	swap.w	d1
1:
	swap.w	d1
2:
	move.b	#'0',(a0)+
	dbra	d1,2b
	swap.w	d1
	dbra	d1,1b
	moveq.l	#10,d1
3:
	lsl.w	#2,d1
	suba.w	d1,a2			;specified start point
;calculate minimum digits
	tst.l	d0
	beq	2f
	lea.l	(-4*10,a1),a1
1:
	cmp.l	(a1)+,d0
	blo	1b
2:
	subq.l	#4,a1			;required start point
;fill
	cmpa.l	a1,a2
	bhs	2f
1:
	move.b	#'0',(a0)+
	addq.l	#4,a2
	cmpa.l	a1,a2
	blo	1b
2:
;print decimal number
	move.l	(a1)+,d1
1:
	moveq.l	#'0'-1,d2
2:
	addq.b	#1,d2
	sub.l	d1,d0
	bhs	2b
	add.l	d1,d0
	move.b	d2,(a0)+
	move.l	(a1)+,d1
	bne	1b
	move.b	d1,(a0)			;0
	movem.l	(sp)+,d0-d2/a1-a2
	rts

;----------------------------------------------------------------
;printfix(number,digits)
;	print fixed point decimal number
;<4(sp).l:number. fixed point decimal number * 10^(number of digits after decimal point)
;<8(sp).b:digits. number of digits after decimal point
regs		reg	d0-d4/a0-a1
	.offsym	0,_a6
_size:
_regs:	.ds.b	.sizeof.(regs)
_a6:	.ds.l	1
_pc:	.ds.l	1
_numb:	.ds.l	1
_digi:	.ds.w	1
	.text
	.even
printfix::
	link.w	a6,#_size
	movem.l	regs,(_regs,a6)
	moveq.l	#0,d3
	move.b	(_digi,a6),d3
;<d3.l:number of digits after decimal point
	move.l	d3,d4
	addq.w	#3,d4
	and.w	#-4,d4			;round up to multiples of four
	add.w	#12,d4			;12 bytes for integer part and decimal point
;<d4.l:buffer size
	suba.l	d4,sp
;<sp.l:buffer
	movea.l	sp,a0
	move.l	(_numb,a6),d0		;fixed point decimal number * 10^(number of digits after decimal point)
	bne	20f			;non-zero
;zero
	move.b	#'0',(a0)+
	move.w	d3,d2			;number of digits after decimal point
	beq	13f			;omit decimal point
	move.b	#'.',(a0)+
	bra	12f

11:
	move.b	#'0',(a0)+
12:
	dbra	d2,11b
13:
	bra	80f			;print

20:
;non-zero
;<d0.l:fixed point decimal number * 10^(number of digits after decimal point)
	lea.l	baseten,a1
;zero suppression
21:
	move.l	(a1)+,d1
	cmp.l	d1,d0
	blo	21b
;convert to decimal number
22:
	moveq.l	#'0'-1,d2
23:
	addq.b	#1,d2
	sub.l	d1,d0
	bcc	23b
	add.l	d1,d0
	move.b	d2,(a0)+
	move.l	(a1)+,d1
	bne	22b
;
	move.l	a0,d2
	sub.l	sp,d2
;<d2.l:actual number of digits
	cmp.w	d3,d2
	bls	40f
;actual number of digits > number of digits after decimal point
;integer part exists
;insert '.'
	move.w	d3,d2			;number of digits after decimal point
	beq	33f			;omit decimal point
	movea.l	a0,a1
	addq.l	#1,a0
	bra	32f

31:
	move.b	-(a1),1(a1)
32:
	dbra	d2,31b
	move.b	#'.',(a1)
33:
	bra	80f			;print

40:
;actual number of digits <= number of digits after decimal point
;no integer part exists
;insert '0.00...'
	move.w	d3,d0			;number of digits after decimal point
	sub.w	d2,d0			;number of zeros after decimal point
	movea.l	a0,a1
	lea.l	2(a0,d0.w),a0
	bra	42f

41:
	move.b	-(a1),2(a1,d0.w)
42:
	dbra	d2,41b
	movea.l	sp,a1
	move.b	#'0',(a1)+
	move.b	#'.',(a1)+
	bra	44f

43:
	move.b	#'0',(a1)+
44:
	dbra	d0,43b

;print
80:
	suba.l	sp,a0
	move.l	a0,-(sp)		;length
	pea.l	4(sp)			;buffer
	jbsr	logging_write
;	addq.l	#8,sp
;	adda.l	d4,sp
	movem.l	(_regs,a6),regs
	unlk	a6
	rts

;----------------------------------------------------------------
	.text
	.align	4
baseten::
	.dc.l	1000000000
	.dc.l	100000000
	.dc.l	10000000
	.dc.l	1000000
	.dc.l	100000
	.dc.l	10000
	.dc.l	1000
	.dc.l	100
	.dc.l	10
	.dc.l	1
	.dc.l	0

;----------------------------------------------------------------
;printhex2(number)
;	print hexadecimal number $XX
;<4(sp).b:number
	.text
	.even
printhex2::
	movem.l	d0-d2/a0,-(sp)
	move.b	4*4+4(sp),d0		;number
	subq.l	#4,sp			;buffer
	movea.l	sp,a0
	move.b	#'$',(a0)+
	moveq.l	#2-1,d2
2:
	rol.b	#4,d0
	moveq.l	#15,d1
	and.w	d0,d1
	move.b	10f(pc,d1.w),(a0)+
	dbra	d2,2b
	pea.l	1+2.w			;length
	pea.l	4(sp)			;buffer
	jbsr	logging_write
	lea.l	8+4(sp),sp
	movem.l	(sp)+,d0-d2/a0
	rts

10:
	.dc.b	'0123456789ABCDEF'

;----------------------------------------------------------------
;printhex4(number)
;	print hexadecimal number $XXXX
;<4(sp).w:number
	.text
	.even
printhex4::
	movem.l	d0-d2/a0,-(sp)
	move.w	4*4+4(sp),d0		;number
	subq.l	#6,sp			;buffer
	movea.l	sp,a0
	move.b	#'$',(a0)+
	moveq.l	#4-1,d2
2:
	rol.w	#4,d0
	moveq.l	#15,d1
	and.w	d0,d1
	move.b	10f(pc,d1.w),(a0)+
	dbra	d2,2b
	pea.l	1+4.w			;length
	pea.l	4(sp)			;buffer
	jbsr	logging_write
	lea.l	8+6(sp),sp
	movem.l	(sp)+,d0-d2/a0
	rts

10:
	.dc.b	'0123456789ABCDEF'

;----------------------------------------------------------------
;printhex8(number)
;	print hexadecimal number $XXXXXXXX
;<4(sp).l:number
	.text
	.even
printhex8::
	movem.l	d0-d2/a0,-(sp)
	move.l	4*4+4(sp),d0		;number
	lea.l	-10(sp),sp		;buffer
	movea.l	sp,a0
	move.b	#'$',(a0)+
	moveq.l	#8-1,d2
2:
	rol.l	#4,d0
	moveq.l	#15,d1
	and.w	d0,d1
	move.b	10f(pc,d1.w),(a0)+
	dbra	d2,2b
	pea.l	1+8.w			;length
	pea.l	4(sp)			;buffer
	jbsr	logging_write
	lea.l	8+10(sp),sp
	movem.l	(sp)+,d0-d2/a0
	rts

10:
	.dc.b	'0123456789ABCDEF'

;----------------------------------------------------------------
;printcrlf()
;	print CR and LF
	.text
	.even
printcrlf::
	move.l	d0,-(sp)
	pea.l	2.w			;length
	pea.l	10f(pc)			;buffer
	jbsr	logging_write
	addq.l	#8,sp
	move.l	(sp)+,d0
	rts

10:
	.dc.b	13,10

;----------------------------------------------------------------
;printchr(character)
;	print character
;<4(sp).b:character
	.text
	.even
printchr::
	move.l	d0,-(sp)
	pea.l	1.w			;length
	pea.l	4+4+4(sp)		;character
	jbsr	logging_write
	addq.l	#8,sp
	move.l	(sp)+,d0
	rts

;----------------------------------------------------------------
;printstr(string)
;	print string
;<4(sp).l:string
	.text
	.even
printstr::
	movem.l	d0/a0-a1,-(sp)
	movea.l	4*3+4(sp),a0		;string
	movea.l	a0,a1
@@:
	tst.b	(a1)+
	bne	@b
	subq.l	#1,a1
	suba.l	a0,a1			;length
	movem.l	a0-a1,-(sp)		;length, string
	jbsr	logging_write
	addq.l	#8,sp
	movem.l	(sp)+,d0/a0-a1
	rts


;--------------------------------------------------------------------------------
;	計算サブルーチン
;--------------------------------------------------------------------------------

;----------------------------------------------------------------
;mull
;	unsigned multiplication. long*long
;<d0.l:multiplicand
;<d1.l:multiplier
;>d0:d1.q:product
	.text
	.even
mull::
	cmp.l	#$0000FFFF,d0
	bhi	10f
	cmp.l	#$0000FFFF,d1
	bhi	10f
	mulu.w	d0,d1
	moveq.l	#0,d0
	rts

10:
	movem.l	d2-d4,-(sp)
					;    d0      d1      d2      d3      d4
					;   A   B   C   D   .   .   .   .   .   .
	move.l	d1,d2			;                   C   D
	move.l	d1,d3			;                           C   D
	swap.w	d2			;                   D   C
	move.l	d2,d4			;                                   D   C
	mulu.w	d0,d1			;            B*D
	mulu.w	d0,d4			;                                    B*C
	swap.w	d0			;   B   A
	mulu.w	d0,d3			;                            A*D
	mulu.w	d2,d0			;    A*C
					;  ACh ACl BDh BDl
	move.w	d1,d2			;                  --- BDl
	move.w	d0,d1			;      ---     ACl
	swap.w	d1			;          ACl BDh
	swap.w	d0			;  --- ACh
	add.l	d3,d1			;            +AD
	clr.w	d3
	addx.w	d3,d0
	add.l	d4,d1			;            +BC
	addx.w	d3,d0
	swap.w	d0			;  ACh ---
	swap.w	d1			;          BDh ACl
	move.w	d1,d0			;      ACl     ---
	move.w	d2,d1			;              BDl
	movem.l	(sp)+,d2-d4
	rts

;----------------------------------------------------------------
;divq
;	unsigned division. quad/quad
;<d0:d1.q:dividend
;<d2:d3.q:divisor
;>d0:d1.q:quotient
;>d2:d3.q:remainder
;>x:0
;>n:1=quotient is negative
;>z:1=quotient is zero
;>v:0
;>c:1=divide by zero. d0:d1 and d2:d3 are not changed. z=0,n=0
	.text
	.even
divq::
	tst.l	d2
	bne	20f			;$FFFFFFFF<divisor
;divisor<=$FFFFFFFF
	tst.l	d3
	beq	40f			;divisor==0
	tst.l	d0
	bne	50f			;$FFFFFFFF<dividend && divisor<=$FFFFFFFF
;dividend<=$FFFFFFFF && divisor<=$FFFFFFFF
	cmp.l	d3,d1
	bls	60f			;dividend<=divisor
;divisor<dividend
10:
	movem.l	d5-d6,-(sp)
	move.l	d3,d5
	moveq.l	#0,d3
	moveq.l	#31,d6
1:
	add.l	d1,d1
	addx.l	d3,d3
	cmp.l	d5,d3
	blo	2f
	addq.b	#1,d1
	sub.l	d5,d3
2:
	dbra	d6,1b
	subq.w	#1,d6			;$0000FFFF->$0000FFFE. x=0
	or.l	d0,d6			;n=*,z=0,v=0,c=0
	movem.l	(sp)+,d5-d6
	rts

;$FFFFFFFF<divisor
20:
	cmp.l	d2,d0
	bhi	30f			;divisor<dividend
	blo	70f			;dividend<divisor
	cmp.l	d3,d1
	bls	60f			;dividend<=divisor
;divisor<dividend
30:
	movem.l	d4-d6,-(sp)
	move.l	d2,d4
	move.l	d3,d5
	moveq.l	#0,d2
	move.l	d0,d3
	move.l	d1,d0
	moveq.l	#0,d1
	moveq.l	#31,d6
1:
	add.l	d1,d1
	addx.l	d0,d0
	addx.l	d3,d3
	addx.l	d2,d2
  .if 1
	cmp.l	d4,d2
	blo	3f
	bhi	2f
	cmp.l	d5,d3
	blo	3f
2:
	sub.l	d5,d3
	subx.l	d4,d2
	addq.b	#1,d1
3:
  .else
	addq.b	#1,d1
	sub.l	d5,d3
	subx.l	d4,d2
	bhs	2f
	subq.b	#1,d1
	add.l	d5,d3
	addx.l	d4,d2
2:
  .endif
	dbra	d6,1b
	subq.w	#1,d6			;$0000FFFF->$0000FFFE. x=0
	or.l	d0,d6			;n=*,z=0,v=0,c=0
	movem.l	(sp)+,d4-d6
	rts

;divisor==0
40:
	move.w	#%00001,ccr		;x=0,n=0,z=0,v=0,c=1
	rts

;$FFFFFFFF<dividend && divisor<=$FFFFFFFF
50:
	movem.l	d5-d6,-(sp)
	move.l	d3,d5
	moveq.l	#0,d3
	moveq.l	#63,d6
1:
	add.l	d1,d1
	addx.l	d0,d0
	addx.l	d3,d3
	cmp.l	d5,d3
	blo	2f
	addq.b	#1,d1
	sub.l	d5,d3
2:
	dbra	d6,1b
	subq.w	#1,d6			;$0000FFFF->$0000FFFE. x=0
	or.l	d0,d6			;n=*,z=0,v=0,c=0
	movem.l	(sp)+,d5-d6
	rts

;dividend<=divisor
60:
	beq	80f			;dividend==divisor
;dividend<divisor
70:
	move.l	d0,d2			;remainder=dividend
	move.l	d1,d3
	sub.l	d0,d0			;quotient=0. x=0
	moveq.l	#0,d1			;n=0,z=1,v=0,c=0
	rts

;dividend==divisor
80:
	moveq.l	#0,d2			;remainder=0
	moveq.l	#0,d3
	sub.l	d0,d0			;quotient=1. x=0
	moveq.l	#1,d1			;n=0,z=0,v=0,c=0
	rts

  .if 0
	movem.l	d4-d6,-(sp)
	move.l	d2,d4
	move.l	d3,d5
	moveq.l	#0,d2
	moveq.l	#0,d3
	moveq.l	#63,d6
1:
	add.l	d1,d1
	addx.l	d0,d0
	addx.l	d3,d3
	addx.l	d2,d2
	addq.b	#1,d1
	sub.l	d5,d3
	subx.l	d4,d2
	bcc	2f
	subq.b	#1,d1
	add.l	d5,d3
	addx.l	d4,d2
2:
	dbra	d6,1b
	movem.l	(sp)+,d4-d6
	rts
  .endif


;--------------------------------------------------------------------------------
;	解凍
;--------------------------------------------------------------------------------

DICTIONARY_BITS	equ	9
DICTIONARY_SIZE	equ	512
COMPRESS_PAGE	equ	1		;1=単語辞書のページ番号を圧縮する
COMPRESS_CHAR	equ	1		;1=文字を圧縮する

;----------------------------------------------------------------
;brcode n
;	n bit取り出す
;	n	取り出すbit数。0..25bit
;<d1.l:入力プール。右寄せ
;<d2.w:入力プールの残りbit数
;<a0.l:入力アドレス
;>d0.l:n bitのデータ。0拡張
;>d1.l:入力プール。右寄せ
;>d2.w:入力プールの残りbit数
;>a0.l:入力アドレス
brcode	.macro	n
	sub.w	n,d2
	bpl	@skip
@loop:
	lsl.l	#8,d1
	move.b	(a0)+,d1
	addq.w	#8,d2
	bmi	@loop
@skip:
	move.l	d1,d0
	lsr.l	d2,d0
	lsl.l	d2,d0
	eor.l	d0,d1
	lsr.l	d2,d0
	.endm

;----------------------------------------------------------------
;decompress(in,out)
;	解凍する
;	汎用ではない。エラーチェックを行っていないのでデータが壊れているとクラッシュする
regs		reg	d0-d7/a0-a5
	.offsym	0,_a6
_size:
_regs:	.ds.b	.sizeof.(regs)
_a6:	.ds.l	1
_pc:	.ds.l	1
_in:	.ds.l	1			;入力アドレス
_out:	.ds.l	1			;出力アドレス
	.text
	.even
decompress::
	link.w	a6,#_size
	movem.l	regs,(_regs,a6)
;単語辞書を初期化する
	lea.l	decompress_wdic,a3	;単語辞書
	movea.l	a3,a0
	move.w	#DICTIONARY_SIZE-1,d0
@@:
	clr.l	(a0)+
	clr.l	(a0)+
	dbra	d0,@b
;文字辞書を初期化する
	lea.l	decompress_cdic,a5	;文字辞書
	movea.l	a5,a0
	move.l	#$00010203,d3
	move.l	#$04040404,d4
	moveq.l	#256/4-1,d0
@@:
	move.l	d3,(a0)+
	add.l	d4,d3
	dbra	d0,@b
;
	moveq.l	#0,d1			;入力プール。右寄せ
	moveq.l	#0,d2			;入力プールの残りbit数
	moveq.l	#0,d5			;単語辞書のアドレスのゲタ
	movea.l	(_in,a6),a0		;入力アドレス
	movea.l	(_out,a6),a1		;出力アドレス
	movea.l	a1,a2
	adda.l	(a0)+,a2		;出力バッファの末尾
;解凍ループ
	cmpa.l	a2,a1
	bhs	199f
100:
	brcode	#1			;1bit取り出す
	bne	20f			;単語辞書にある
;単語辞書にない
	moveq.l	#1,d3			;length
	subq.w	#8,d5			;単語辞書のアドレスのゲタをずらす
	and.w	#(DICTIONARY_SIZE-1)<<3,d5
	movem.l	d3/a1,(a3,d5.l)		;単語辞書の先頭に登録する。length,address
	bra	30f

;単語辞書にある
20:
  .if COMPRESS_PAGE
;単語辞書のページ番号が圧縮されている
;単語辞書のページ番号を求める
	moveq.l	#0,d4			;wid。単語辞書のページ番号に2を加えた値の先頭の1を除いたbit数
@@:
	addq.w	#1,d4
	brcode	#1			;brcode(1)
	bne	@b
	brcode	d4			;brcode(wid)
	bset.l	d4,d0			;(1<<wid)+brcode(wid)
	subq.w	#2,d0			;(1<<wid)+brcode(wid)-2。単語辞書のページ番号
  .else
;単語辞書のページ番号が圧縮されていない
	brcode	#DICTIONARY_BITS	;brcode(DICTIONARY_BITS)。単語辞書のページ番号
  .endif
;単語辞書から取り出す
	lsl.l	#3,d0			;ページオフセット
	add.w	d5,d0			;ページオフセットに単語辞書のアドレスのゲタを加える
	and.w	#(DICTIONARY_SIZE-1)<<3,d0
	movem.l	(a3,d0.l),d3/a4		;length,address
;単語辞書のアドレスのゲタをずらす
	subq.l	#8,d5
	and.l	#(DICTIONARY_SIZE-1)<<3,d5
;1文字伸ばす
	addq.l	#1,d3
;単語辞書の先頭に登録する
	movem.l	d3/a1,(a3,d5.l)		;length,address
;単語辞書から取り出した文字を出力する
	subq.l	#1+1,d3
@@:
	move.b	(a4)+,(a1)+
	dbra	d3,@b
30:
  .if COMPRESS_CHAR
;文字が圧縮されている
;文字辞書のページ番号を求める
	moveq.l	#0,d4			;wid。文字辞書のページ番号に2を加えた値の先頭の1を除いたbit数
@@:
	addq.w	#1,d4
	brcode	#1			;brcode(1)
	bne	@b
	brcode	d4			;brcode(wid)
	bset.l	d4,d0			;(1<<wid)+brcode(wid)
	subq.w	#2,d0			;(1<<wid)+brcode(wid)-2。文字辞書のページ番号
;文字辞書から取り出す
	lea.l	(a5,d0.w),a4		;文字辞書のページアドレス
	move.b	(a4),d3
;文字辞書から取り出した文字を出力する
	move.b	d3,(a1)+
;取り出した文字を文字辞書の先頭に移動させる
	bra	2f

1:
	move.b	-(a4),1(a4)
2:
	dbra	d0,1b
	move.b	d3,(a5)
  .else
;文字が圧縮されていない
;入力された文字を出力する
	move.b	(a0)+,(a1)+
  .endif
	cmpa.l	a2,a1
	blo	100b
199:
	movem.l	(_regs,a6),regs
	unlk	a6
	rts

	.bss
	.align	4
decompress_wdic::	.ds.l	2*DICTIONARY_SIZE	;単語辞書。length,address,...
decompress_cdic::	.ds.b	256			;文字辞書。0..255を最近使われた順に並べる

;--------------------------------------------------------------------------------
;	FABS.X FPm,FPn
;--------------------------------------------------------------------------------
	.cpu	68030
regs		reg	d0-d7/a0-a5
cregs		reg	fpcr/fpsr/fpiar
fregs		reg	fp0-fp7
	.offsym	0,_a6
_size:
_regs:	.ds.b	.sizeof.(regs)
_fregs:	.ds.b	.sizeof.(fregs)
_cregs:	.ds.b	.sizeof.(cregs)
_a6:	.ds.l	1
_pc:	.ds.l	1
	.text
	.even
fabs_test::
	link.w	a6,#_size
	movem.l	regs,(_regs,a6)
	fmovem.l	cregs,(_cregs,a6)
	fmovem.x	fregs,(_fregs,a6)
;
	move.l	#63,-(sp)
	peamsg	'FABS.X FPM,FPN'
	jbsr	mnemonic_start
	addq.l	#8,sp
	beq	99f
	putmsg	'test: FABS.X FPm,FPn',13,10
;------------------------------------------------
;	d1	fpcr=(XRN..DRP)<<4
;	d3	0=failed,1=successful
;	d5	expected status
;	d7	actual status
;	a2	source handle,...
;	a4	expected result handle,expected status,...
;	fp2	source
;	fp5	expected result
;	fp7	actual result
;------------------------------------------------
	lea.l	push_decompressed,a0
;decompress data
	move.l	a0,-(sp)
	pea.l	fabs_data_compressed
	jbsr	decompress
	addq.l	#8,sp
;relocate decompressed handle
	move.l	#indirect_decompressed,d0
@@:
	add.l	d0,(a0)+		;source handle
	tst.l	(a0)
	bpl	@b
	addq.l	#4,a0			;-1
@@:
	add.l	d0,(a0)+		;expected result handle
	addq.l	#4,a0			;expected status
	tst.l	(a0)
	bpl	@b
;	addq.l	#4,a0			;-1
;
	lea.l	push_decompressed,a2	;source handle,...
	lea.l	(4*1589+4,a2),a4	;expected result handle,expected status,...
22:
	move.l	#0,d1		;fpcr=(XRN..DRP)<<4
11:
;FPn,FPn
	fmove.l	#0,fpcr
	fmove.x	([a2]),fp2		;source
	fmove.x	fp2,fp7			;source
	fmove.l	d1,fpcr			;fpcr
	fmove.l	#0,fpsr
					;source
	fabs.x	fp7,fp7		;EXECUTE
					;actual result
	fmove.l	fpsr,d7			;actual status
	fmove.l	#0,fpcr
;
	fmove.x	([a4]),fp5		;expected result
	move.l	(4,a4),d5		;expected status
;
	move.l	#-1,-(sp)	;fpcr(rp<<6,-1=strict)
	move.l	d5,-(sp)		;expected status
	fmove.x	fp5,-(sp)		;expected result
	move.l	d7,-(sp)		;actual status
	fmove.x	fp7,-(sp)		;actual result
	jbsr	test_extended
	lea.l	36(sp),sp
	move.l	d0,d3			;0=failed,1=successful
;
	move.l	d3,-(sp)		;0=failed,1=successful
	jbsr	statistics_update
	addq.l	#4,sp
	beq	@f			;not output
;output
	putmsg	'FABS.X FPn='
	puthex24	([a2]),(4,[a2]),(8,[a2])	;source
	putmsg	',FPn @'
	move.l	d1,-(sp)
	jbsr	printfpcrrprm
	addq.l	#4,sp
	putcrlf
	move.l	d3,-(sp)		;0=failed,1=successful
	move.l	d5,-(sp)		;expected status
	fmove.x	fp5,-(sp)		;expected result
	move.l	d7,-(sp)		;actual status
	fmove.x	fp7,-(sp)		;actual result
	jbsr	output_extended
	lea.l	36(sp),sp
@@:
;
	addq.l	#8,a4			;expected result handle,expected status,...
;
	add.w	#1<<4,d1		;rprm++
	cmp.w	#176,d1		;fpcr=(XRN..DRP)<<4
	bls	11b
;
	addq.l	#4,a2			;source handle,...
	tst.l	(a2)			;source handle,...
	bpl	22b
;
	lea.l	push_decompressed,a2	;source handle,...
	lea.l	(4*1589+4,a2),a4	;expected result handle,expected status,...
22:
	move.l	#0,d1		;fpcr=(XRN..DRP)<<4
11:
;FPm,FPn
	fmove.l	#0,fpcr
	fmove.x	([a2]),fp2		;source
	fmove.s	#$7FFFFFFF,fp7		;fill=NaN
	fmove.l	d1,fpcr			;fpcr
	fmove.l	#0,fpsr
					;source
	fabs.x	fp2,fp7		;EXECUTE
					;actual result
	fmove.l	fpsr,d7			;actual status
	fmove.l	#0,fpcr
;
	fmove.x	([a4]),fp5		;expected result
	move.l	(4,a4),d5		;expected status
;
	move.l	#-1,-(sp)	;fpcr(rp<<6,-1=strict)
	move.l	d5,-(sp)		;expected status
	fmove.x	fp5,-(sp)		;expected result
	move.l	d7,-(sp)		;actual status
	fmove.x	fp7,-(sp)		;actual result
	jbsr	test_extended
	lea.l	36(sp),sp
	move.l	d0,d3			;0=failed,1=successful
;
	move.l	d3,-(sp)		;0=failed,1=successful
	jbsr	statistics_update
	addq.l	#4,sp
	beq	@f			;not output
;output
	putmsg	'FABS.X FPm='
	puthex24	([a2]),(4,[a2]),(8,[a2])	;source
	putmsg	',FPn @'
	move.l	d1,-(sp)
	jbsr	printfpcrrprm
	addq.l	#4,sp
	putcrlf
	move.l	d3,-(sp)		;0=failed,1=successful
	move.l	d5,-(sp)		;expected status
	fmove.x	fp5,-(sp)		;expected result
	move.l	d7,-(sp)		;actual status
	fmove.x	fp7,-(sp)		;actual result
	jbsr	output_extended
	lea.l	36(sp),sp
@@:
;
	addq.l	#8,a4			;expected result handle,expected status,...
;
	add.w	#1<<4,d1		;rprm++
	cmp.w	#176,d1		;fpcr=(XRN..DRP)<<4
	bls	11b
;
	addq.l	#4,a2			;source handle,...
	tst.l	(a2)			;source handle,...
	bpl	22b
;
	jbsr	mnemonic_end
99:
	fmovem.x	(_fregs,a6),fregs
	fmovem.l	(_cregs,a6),cregs
	movem.l	(_regs,a6),regs
	unlk	a6
	rts
	.cpu	68000

	.align	4
fabs_data_compressed::
	.dc.b	$00,$02,$6C,$BC,$10,$86,$36,$BD,$58,$F1,$B3,$E9,$6B,$DE,$DF,$C5
	.dc.b	$70,$F9,$6E,$3F,$45,$CB,$EB,$B9,$FD,$D7,$4F,$C1,$BA,$FE,$25,$DB
	.dc.b	$F1,$EE,$FF,$95,$78,$3F,$36,$F0,$FE,$85,$E2,$FD,$3B,$C7,$FA,$D7
	.dc.b	$93,$F6,$6F,$2F,$EE,$5E,$6F,$DF,$BD,$1D,$0E,$FB,$E5,$63,$D5,$CF
	.dc.b	$C0,$6B,$E2,$F7,$F2,$DC,$3E,$87,$8F,$D6,$72,$FB,$7E,$7F,$81,$74
	.dc.b	$FC,$37,$AF,$E3,$1D,$BF,$23,$EF,$F9,$77,$83,$F3,$9F,$0F,$E8,$9E
	.dc.b	$2F,$D3,$FC,$7F,$AD,$79,$3F,$65,$F2,$FE,$E1,$E6,$FD,$EF,$E7,$79
	.dc.b	$CF,$42,$FB,$58,$F8,$4C,$FC,$76,$BE,$63,$7F,$45,$C3,$EB,$38,$FD
	.dc.b	$B7,$2F,$C0,$39,$FE,$15,$D3,$F1,$4E,$BF,$8F,$76,$FC,$A3,$BF,$E6
	.dc.b	$5E,$0F,$CF,$3C,$3F,$A3,$78,$BF,$50,$F1,$FE,$B5,$E4,$FD,$93,$CB
	.dc.b	$FB,$77,$9B,$F7,$8F,$C2,$7C,$27,$E1,$07,$C1,$E3,$E0,$F3,$F0,$7A
	.dc.b	$F8,$3D,$FC,$1F,$0F,$83,$E3,$F0,$7C,$BE,$0F,$9F,$C1,$F4,$F8,$3E
	.dc.b	$BF,$07,$DB,$E0,$FB,$FC,$1F,$83,$E0,$FC,$3F,$07,$E2,$F8,$3F,$1F
	.dc.b	$C1,$F9,$3E,$0F,$CB,$F0,$7E,$6F,$83,$F3,$FC,$1F,$E6,$FC,$CF,$C2
	.dc.b	$7C,$27,$E7,$97,$C1,$67,$E0,$B5,$F0,$5B,$F8,$2E,$1F,$05,$C7,$E0
	.dc.b	$B9,$7C,$17,$3F,$82,$E9,$F0,$5D,$7E,$0B,$B7,$C1,$77,$F8,$2F,$07
	.dc.b	$C1,$78,$7E,$0B,$C5,$F0,$5E,$3F,$82,$F2,$7C,$17,$97,$E0,$BC,$DF
	.dc.b	$05,$E7,$F8,$2F,$D9,$7C,$2F,$E1,$7A,$3F,$6A,$BE,$0B,$3F,$05,$AF
	.dc.b	$82,$DF,$C1,$70,$F8,$2E,$3F,$05,$CB,$E0,$B9,$FC,$17,$4F,$82,$EB
	.dc.b	$F0,$5D,$BE,$0B,$BF,$C1,$78,$3E,$0B,$C3,$F0,$5E,$2F,$82,$F1,$FC
	.dc.b	$17,$93,$E0,$BC,$BF,$05,$E6,$F8,$2F,$3F,$C1,$7F,$09,$F8,$6F,$E1
	.dc.b	$27,$E1,$BA,$3F,$86,$AF,$81,$D7,$C0,$EF,$E0,$78,$7C,$0F,$1F,$81
	.dc.b	$E5,$F0,$3C,$FE,$07,$A7,$C0,$F5,$F8,$1E,$DF,$03,$DF,$E0,$7C,$1F
	.dc.b	$03,$E1,$F8,$1F,$17,$C0,$F8,$FE,$07,$C9,$F0,$3E,$5F,$81,$F3,$7C
	.dc.b	$0F,$9F,$E0,$7D,$1F,$03,$FC,$87,$E1,$FF,$0F,$D0,$7F,$0E,$FF,$26
	.dc.b	$5F,$01,$BF,$80,$E1,$F0,$1C,$7E,$03,$97,$C0,$73,$F8,$0E,$9F,$01
	.dc.b	$D7,$E0,$3B,$7C,$07,$7F,$80,$F0,$7C,$07,$87,$E0,$3C,$5F,$01,$E3
	.dc.b	$F8,$0F,$27,$C0,$79,$7E,$03,$CD,$F0,$1E,$7F,$80,$F4,$7C,$07,$F3
	.dc.b	$9F,$86,$FC,$47,$3F,$F2,$3F,$80,$C7,$C0,$67,$E0,$35,$F0,$1B,$F8
	.dc.b	$0E,$1F,$01,$C7,$E0,$39,$7C,$07,$3F,$80,$E9,$F0,$1D,$7E,$03,$B7
	.dc.b	$C0,$77,$F8,$0F,$07,$C0,$78,$7E,$03,$C5,$F0,$1E,$3F,$80,$F2,$7C
	.dc.b	$07,$97,$E0,$3C,$DF,$01,$FD,$3F,$E2,$3F,$A7,$9F,$87,$E7,$3F,$87
	.dc.b	$7F,$AA,$AF,$7F,$87,$BF,$C7,$DF,$E5,$EF,$F3,$F7,$FA,$7B,$FD,$7D
	.dc.b	$FE,$DE,$FF,$7F,$7F,$C1,$EF,$F8,$7D,$FF,$17,$BF,$E3,$F7,$FC,$9E
	.dc.b	$FF,$97,$DF,$F3,$7B,$FE,$7F,$7F,$D1,$EF,$FA,$7D,$FF,$ED,$1F,$13
	.dc.b	$F8,$83,$F0,$FC,$E7,$F0,$EF,$F6,$E5,$EF,$70,$F7,$B8,$FB,$DC,$BD
	.dc.b	$EE,$7E,$F7,$4F,$7B,$AF,$BD,$DB,$DE,$EF,$EF,$78,$3D,$EF,$0F,$BD
	.dc.b	$E2,$F7,$BC,$7E,$F7,$93,$DE,$F2,$FB,$DE,$6F,$7B,$CF,$EF,$7A,$3D
	.dc.b	$EF,$F0,$07,$C5,$7E,$20,$FE,$1D,$CD,$FE,$04,$5E,$EE,$BD,$DD,$FB
	.dc.b	$BC,$3D,$DE,$3E,$EF,$2F,$77,$9F,$BB,$D3,$DD,$EB,$EE,$F6,$F7,$7B
	.dc.b	$FB,$BE,$0F,$77,$C3,$EE,$F8,$BD,$DF,$1F,$BB,$E4,$F7,$7C,$BE,$EF
	.dc.b	$9B,$DD,$F3,$FB,$BF,$E1,$7F,$87,$FF,$0B,$9F,$89,$3F,$11,$CF,$FD
	.dc.b	$5B,$DD,$D7,$BB,$BF,$77,$87,$BB,$C7,$DD,$E5,$EE,$F3,$F7,$7A,$7B
	.dc.b	$BD,$7D,$DE,$DE,$EF,$7F,$77,$C1,$EE,$F8,$7D,$DF,$17,$BB,$E3,$F7
	.dc.b	$7C,$9E,$EF,$97,$DD,$F3,$7B,$BE,$7F,$77,$FC,$5D,$F1,$7F,$C3,$8F
	.dc.b	$C3,$9F,$86,$E6,$3F,$86,$7F,$C6,$AB,$DC,$E3,$EE,$72,$F7,$39,$FB
	.dc.b	$9D,$3D,$CE,$BE,$E7,$6F,$73,$BF,$B9,$E0,$F7,$3C,$3E,$E7,$8B,$DC
	.dc.b	$F1,$FB,$9E,$4F,$73,$CB,$EE,$79,$BD,$CF,$3F,$B9,$E8,$F7,$3F,$C8
	.dc.b	$DF,$19,$FC,$38,$FC,$3F,$29,$FC,$3B,$FE,$4C,$5E,$DF,$0F,$6F,$8F
	.dc.b	$B7,$CB,$DB,$E7,$ED,$F4,$F6,$FA,$FB,$7D,$BD,$BE,$FE,$DF,$83,$DB
	.dc.b	$F0,$FB,$7E,$2F,$6F,$C7,$ED,$F9,$3D,$BF,$2F,$B7,$E6,$F6,$FC,$FE
	.dc.b	$DF,$A3,$DB,$FF,$2F,$7C,$6F,$F9,$78,$FE,$20,$7E,$20,$FE,$1D,$CD
	.dc.b	$FE,$64,$5E,$D7,$0F,$6B,$8F,$B5,$CB,$DA,$E7,$ED,$74,$F6,$BA,$FB
	.dc.b	$5D,$BD,$AE,$FE,$D7,$83,$DA,$F0,$FB,$5E,$2F,$6B,$C7,$ED,$79,$3D
	.dc.b	$AF,$2F,$B5,$E6,$F6,$BC,$FE,$D7,$A3,$DA,$FF,$3B,$7C,$17,$F5,$63
	.dc.b	$F8,$79,$F8,$93,$F1,$1C,$DF,$E0,$7F,$6B,$7E,$D7,$0F,$6B,$8F,$B5
	.dc.b	$CB,$DA,$E7,$ED,$74,$F6,$BA,$FB,$5D,$BD,$AE,$FE,$D7,$83,$DA,$F0
	.dc.b	$FB,$5E,$2F,$6B,$C7,$ED,$79,$3D,$AF,$2F,$B5,$E6,$F6,$BF,$D1,$9F
	.dc.b	$1D,$FD,$50,$FC,$39,$F8,$6E,$43,$F8,$67,$FD,$26,$BD,$9E,$3E,$CF
	.dc.b	$2F,$67,$9F,$B3,$D3,$D9,$EB,$EC,$F6,$F6,$7B,$FB,$3E,$0F,$67,$C3
	.dc.b	$EC,$F8,$BD,$9F,$1F,$B3,$E4,$F6,$7C,$BE,$CF,$9B,$D9,$F3,$FB,$3E
	.dc.b	$8F,$67,$FD,$49,$F1,$FF,$EA,$43,$FA,$B1,$FC,$38,$FC,$3F,$29,$FC
	.dc.b	$3B,$FE,$AC,$5E,$C7,$2F,$63,$9F,$B1,$D3,$D8,$EB,$EC,$76,$F6,$3B
	.dc.b	$FB,$1E,$0F,$63,$C3,$EC,$78,$BD,$8F,$1F,$B1,$E4,$F6,$3C,$BE,$C7
	.dc.b	$9B,$D8,$F3,$FB,$1E,$8F,$63,$D3,$EC,$7F,$AF,$7E,$43,$FC,$3A,$7F
	.dc.b	$56,$3F,$88,$1F,$88,$3F,$87,$72,$FF,$B1,$17,$AF,$C7,$D7,$E5,$EB
	.dc.b	$F3,$F5,$FA,$7A,$FD,$7D,$7E,$DE,$BF,$7F,$5F,$C1,$EB,$F8,$7D,$7F
	.dc.b	$17,$AF,$E3,$F5,$FC,$9E,$BF,$97,$D7,$F3,$7A,$FE,$7F,$5F,$FD,$A9
	.dc.b	$F1,$9F,$E1,$C3,$F8,$79,$F8,$93,$F1,$1C,$9F,$E7,$CF,$5F,$7E,$BF
	.dc.b	$0F,$5F,$8F,$AF,$CB,$D7,$E7,$EB,$F4,$F5,$FA,$FA,$FD,$BD,$7E,$FE
	.dc.b	$BF,$83,$D7,$F0,$FA,$FE,$2F,$5F,$C7,$EB,$F9,$3D,$7F,$2F,$AF,$E6
	.dc.b	$F5,$FF,$DD,$5F,$23,$FE,$EA,$3F,$C3,$87,$F5,$43,$F0,$E7,$E1,$B9
	.dc.b	$0F,$E1,$9F,$F7,$9A,$F5,$B9,$FA,$DD,$3D,$6E,$BE,$B7,$6F,$5B,$BF
	.dc.b	$AD,$E0,$F5,$BC,$3E,$B7,$8B,$D6,$F1,$FA,$DE,$4F,$5B,$CB,$EB,$79
	.dc.b	$BD,$6F,$3F,$AD,$E8,$F5,$BD,$3E,$B3,$8E,$7C,$97,$F9,$90,$FF,$0E
	.dc.b	$9F,$D5,$CF,$E1,$E7,$E2,$13,$F8,$87,$8F,$AB,$CD,$CF,$AB,$65,$EA
	.dc.b	$DF,$57,$1E,$AE,$7D,$5D,$7A,$BB,$F5,$78,$7A,$BC,$7D,$5E,$5E,$AF
	.dc.b	$3F,$57,$A7,$AB,$D7,$D5,$ED,$EA,$F8,$7E,$4F,$FC,$CA,$7F,$58,$3F
	.dc.b	$89,$1F,$8A,$3F,$88,$71,$7C,$15,$EA,$71,$F5,$39,$7A,$9C,$FD,$4E
	.dc.b	$9E,$A7,$5F,$53,$B7,$A9,$DF,$D4,$F0,$7A,$9E,$1F,$53,$C5,$EA,$78
	.dc.b	$FD,$4F,$27,$A9,$E5,$F5,$3C,$DE,$A7,$9F,$D4,$F9,$1F,$1B,$F2,$79
	.dc.b	$3F,$D5,$5F,$0B,$88,$CF,$C2,$EB,$E0,$77,$EE,$F0,$F6,$38,$FB,$1C
	.dc.b	$BD,$8E,$7E,$C7,$4F,$63,$AF,$B1,$DB,$D8,$EF,$EC,$78,$3D,$8F,$0F
	.dc.b	$B1,$E2,$F6,$3C,$7E,$C7,$93,$D8,$F2,$FB,$1E,$6F,$63,$CF,$EC,$7D
	.dc.b	$FF,$94,$FF,$55,$9F,$E6,$43,$FC,$3C,$7F,$56,$3F,$10,$7E,$1F,$A0
	.dc.b	$BD,$9E,$5E,$CF,$3F,$63,$A7,$B1,$D7,$D8,$ED,$EC,$77,$F6,$3C,$1E
	.dc.b	$C7,$87,$D8,$F1,$7B,$1E,$3F,$63,$C9,$EC,$79,$7D,$8F,$37,$B1,$E7
	.dc.b	$F6,$3F,$2D,$F2,$BF,$EA,$A3,$FC,$3A,$7F,$57,$3F,$87,$9F,$86,$E3
	.dc.b	$3F,$86,$7F,$50,$BD,$7E,$7E,$BF,$4F,$5F,$AF,$AF,$DB,$D7,$EF,$EB
	.dc.b	$F8,$3D,$7F,$0F,$AF,$E2,$F5,$FC,$7E,$BF,$93,$D7,$F2,$FA,$FE,$6F
	.dc.b	$5F,$CF,$EB,$FA,$3D,$7F,$D7,$7C,$B7,$E1,$CF,$F5,$61,$FE,$64,$3F
	.dc.b	$AB,$9F,$C4,$4F,$E2,$1C,$67,$F8,$83,$FC,$05,$7A,$5D,$3D,$2E,$BE
	.dc.b	$97,$6F,$4B,$BF,$A5,$E0,$F4,$BC,$3E,$97,$8B,$D2,$F1,$FA,$5E,$4F
	.dc.b	$4B,$CB,$E9,$79,$BD,$2F,$3F,$A5,$E8,$F4,$BD,$3E,$97,$F0,$7F,$89
	.dc.b	$FF,$79,$9F,$EA,$C3,$FC,$CC,$7F,$87,$8F,$E2,$07,$E2,$4F,$C4,$71
	.dc.b	$FF,$19,$5E,$8F,$3F,$47,$A7,$A3,$D7,$D1,$ED,$E8,$F7,$F4,$7C,$1E
	.dc.b	$8F,$87,$D1,$F1,$7A,$3E,$3F,$47,$C9,$E8,$F9,$7D,$1F,$37,$A3,$E7
	.dc.b	$F4,$7F,$8F,$7C,$BF,$FB,$C8,$FF,$32,$9F,$E1,$F3,$FA,$B9,$F8,$93
	.dc.b	$F1,$1C,$47,$F1,$0F,$F3,$15,$E8,$74,$F4,$3A,$FA,$1D,$BD,$0E,$FE
	.dc.b	$87,$83,$D0,$F0,$FA,$1E,$2F,$43,$C7,$E8,$79,$3D,$0F,$2F,$A1,$E6
	.dc.b	$F4,$3C,$FE,$87,$A3,$D0,$FE,$6D,$F3,$1F,$C3,$CF,$F7,$A1,$FE,$AC
	.dc.b	$3F,$C3,$E7,$F5,$93,$F1,$27,$E2,$38,$8F,$E2,$1F,$E9,$4B,$CF,$EB
	.dc.b	$E7,$F6,$F3,$FB,$F9,$FE,$0F,$3F,$C3,$E7,$F8,$BC,$FF,$1F,$9F,$E4
	.dc.b	$F3,$FC,$BE,$7F,$9B,$CF,$F3,$F9,$FE,$8F,$3F,$D3,$E7,$FF,$4F,$F9
	.dc.b	$97,$C8,$DF,$83,$C7,$C9,$67,$E4,$35,$F1,$5B,$F8,$7E,$1E,$DF,$1F
	.dc.b	$5B,$97,$AD,$CF,$D6,$E9,$EB,$75,$F5,$BB,$7A,$DD,$FD,$6F,$07,$AD
	.dc.b	$E1,$F5,$BC,$5E,$B7,$8F,$D6,$F2,$7A,$DE,$5F,$5B,$CD,$EB,$7F,$65
	.dc.b	$E4,$7C,$B5,$F8,$CC,$7B,$39,$F8,$ED,$7C,$56,$FE,$1F,$87,$BF,$C7
	.dc.b	$DC,$E5,$EE,$73,$F7,$3A,$7B,$9D,$7D,$CE,$DE,$E7,$7F,$73,$C1,$EE
	.dc.b	$78,$7D,$CF,$17,$B9,$E3,$F7,$3C,$9E,$E7,$97,$DC,$F3,$7B,$9F,$DE
	.dc.b	$FE,$6B,$FA,$A0,$F8,$8C,$7C,$46,$7E,$23,$5F,$11,$BF,$88,$E1,$F1
	.dc.b	$1C,$7E,$17,$97,$C1,$73,$F8,$2E,$9F,$05,$D7,$E0,$BB,$7C,$17,$7F
	.dc.b	$82,$F0,$7C,$17,$87,$E0,$BC,$5F,$05,$E3,$F8,$2F,$27,$C1,$79,$7E
	.dc.b	$0B,$CD,$F0,$5E,$7F,$82,$FF,$09,$7C,$DF,$E1,$3E,$17,$FC,$3C,$BE
	.dc.b	$0B,$3E,$86,$BE,$0B,$7F,$05,$C3,$E0,$B8,$FC,$17,$2F,$82,$E7,$F0
	.dc.b	$5D,$3E,$0B,$AF,$C1,$76,$F8,$2E,$FF,$05,$E0,$F8,$2F,$0F,$C1,$78
	.dc.b	$BE,$0B,$C7,$F0,$5E,$4F,$82,$F2,$FC,$17,$9B,$E0,$BC,$FF,$05,$FE
	.dc.b	$29,$F9,$CF,$C2,$F4,$7F,$8D,$5F,$F7,$89,$FE,$AB,$3F,$CC,$67,$F5
	.dc.b	$53,$F8,$69,$FC,$31,$3F,$C3,$1E,$5E,$DF,$5F,$6F,$B7,$B7,$DF,$DB
	.dc.b	$F0,$7B,$7E,$1F,$6F,$C5,$ED,$F8,$FD,$BF,$27,$B7,$E5,$F6,$FC,$DE
	.dc.b	$DF,$9F,$DB,$F4,$7B,$7F,$E4,$1F,$63,$FC,$34,$7E,$1F,$83,$FC,$9C
	.dc.b	$BD,$9D,$3B,$F6,$78,$7B,$3C,$7D,$9E,$5E,$CF,$3F,$67,$A7,$B3,$D7
	.dc.b	$D9,$ED,$EC,$F7,$F6,$7C,$1E,$CF,$87,$D9,$F1,$7B,$3E,$3F,$67,$C9
	.dc.b	$EC,$F9,$7D,$9F,$37,$B3,$E7,$F6,$7D,$1E,$CF,$F9,$67,$E7,$7F,$11
	.dc.b	$CE,$7F,$10,$FF,$99,$D7,$B1,$BF,$63,$87,$B1,$C7,$D8,$E5,$EC,$73
	.dc.b	$F6,$3A,$7B,$1D,$7D,$8E,$DE,$C7,$7F,$63,$C1,$EC,$78,$7D,$8F,$17
	.dc.b	$B1,$E3,$F6,$3C,$9E,$C7,$97,$D8,$F3,$7B,$1E,$7F,$63,$D1,$EC,$7F
	.dc.b	$9C,$7E,$7B,$F1,$3C,$E5,$F0,$99,$F8,$4D,$7C,$26,$FE,$13,$87,$C2
	.dc.b	$71,$F8,$4E,$5F,$09,$CF,$DF,$E9,$EF,$F5,$F7,$FB,$7B,$FD,$FD,$FF
	.dc.b	$07,$BF,$E1,$F7,$FC,$5E,$FF,$8F,$DF,$F2,$7B,$FE,$5F,$7F,$CD,$EF
	.dc.b	$F9,$FD,$FF,$F4,$47,$CF,$FF,$98,$4F,$C4,$74,$1F,$C4,$3F,$E9,$77
	.dc.b	$FD,$E6,$7F,$AB,$4F,$F0,$E9,$FD,$54,$FE,$A8,$F1,$F6,$BA,$FB,$5D
	.dc.b	$BD,$AE,$FE,$D7,$83,$DA,$F0,$FB,$5E,$2F,$6B,$C7,$ED,$79,$3D,$AF
	.dc.b	$2F,$B5,$E6,$F6,$BC,$FE,$D7,$A3,$DA,$F4,$FB,$5F,$EA,$1F,$0F,$F1
	.dc.b	$27,$E2,$38,$4F,$E2,$1F,$F5,$82,$F6,$38,$7B,$1C,$7D,$8E,$5E,$C7
	.dc.b	$3F,$63,$A7,$B1,$D7,$D8,$ED,$EC,$77,$F6,$3C,$1E,$C7,$87,$D8,$F1
	.dc.b	$7B,$1E,$3F,$63,$C9,$EC,$79,$7D,$8F,$37,$B1,$E7,$F6,$3D,$1E,$C7
	.dc.b	$FA,$E7,$E8,$3F,$12,$7E,$23,$9B,$FD,$8C,$BD,$7D,$73,$EF,$D7,$E1
	.dc.b	$EB,$F1,$F5,$F9,$7A,$FC,$FD,$7E,$9E,$BF,$5F,$5F,$B7,$AF,$DF,$D7
	.dc.b	$F0,$7A,$FE,$1F,$5F,$C5,$EB,$F8,$FD,$7F,$27,$AF,$E5,$F5,$FC,$DE
	.dc.b	$BF,$9F,$D7,$FF,$68,$7D,$0F,$FA,$A4,$FC,$49,$F8,$8E,$72,$F8,$1D
	.dc.b	$FC,$0F,$0F,$81,$E3,$F0,$3C,$BE,$07,$9F,$B9,$D3,$DC,$EB,$EE,$76
	.dc.b	$F7,$3B,$FB,$9E,$0F,$73,$C3,$EE,$78,$BD,$CF,$1F,$B9,$E4,$F7,$3C
	.dc.b	$BE,$E7,$9B,$DC,$F3,$FB,$9E,$8F,$73,$FD,$D1,$F4,$5F,$C4,$0F,$C4
	.dc.b	$1F,$87,$E7,$3F,$87,$7F,$DF,$0F,$FB,$CC,$FF,$56,$1F,$E6,$47,$7E
	.dc.b	$CF,$4F,$67,$AF,$B3,$DB,$D9,$EF,$EC,$F8,$3D,$9F,$0F,$B3,$E2,$F6
	.dc.b	$7C,$7E,$CF,$93,$D9,$F2,$FB,$3E,$6F,$67,$CF,$EC,$FA,$3D,$9F,$F8
	.dc.b	$01,$EE,$FF,$10,$3F,$11,$C3,$F9,$DF,$67,$3E,$CE,$BD,$9D,$FB,$3C
	.dc.b	$3D,$9E,$3E,$CF,$2F,$67,$9F,$B3,$D3,$D9,$EB,$EC,$F6,$F6,$7B,$FB
	.dc.b	$3E,$0F,$67,$C3,$EC,$F8,$BD,$9F,$1F,$B3,$E4,$F6,$7C,$BE,$CF,$9B
	.dc.b	$D9,$72,$CF,$A3,$FF,$78,$1F,$C4,$0F,$C4,$1F,$87,$75,$EC,$70,$51
	.dc.b	$FE,$AC,$3F,$CC,$87,$F1,$03,$FC,$C4,$F5,$F5,$38,$43,$D4,$BE,$A6
	.dc.b	$3D,$4C,$FA,$9A,$F5,$37,$EA,$70,$F5,$38,$FA,$9C,$BD,$4E,$7E,$A7
	.dc.b	$4F,$53,$AF,$A9,$E5,$FA,$4F,$EB,$27,$F1,$13,$F1,$47,$E2,$78,$4B
	.dc.b	$DF,$E1,$EF,$F1,$F7,$F9,$7A,$FC,$FD,$7E,$9E,$BF,$5F,$5F,$B7,$AF
	.dc.b	$DF,$D7,$F0,$7A,$FE,$1F,$5F,$C5,$EB,$F8,$FD,$7F,$27,$AF,$E5,$F5
	.dc.b	$FC,$DE,$BF,$9F,$D7,$F9,$9F,$4B,$FD,$60,$FC,$49,$FC,$3F,$97,$EB
	.dc.b	$BF,$CA,$9B,$EA,$F0,$F5,$78,$FA,$BC,$BD,$5E,$7E,$AF,$4F,$57,$AF
	.dc.b	$AB,$DB,$D5,$EF,$EA,$F8,$3D,$5F,$0F,$AB,$E2,$F5,$7C,$7E,$AF,$93
	.dc.b	$D5,$F2,$FA,$BE,$6F,$57,$CF,$EA,$FE,$0F,$D5,$FC,$27,$21,$FE,$B2
	.dc.b	$7F,$11,$3F,$14,$E5,$E1,$EE,$F1,$F7,$79,$7B,$BC,$FD,$DE,$9E,$CF
	.dc.b	$5F,$63,$B7,$B1,$DF,$D8,$F0,$7B,$1E,$1F,$63,$C5,$EC,$78,$FD,$8F
	.dc.b	$27,$B1,$E5,$F6,$3C,$DE,$C7,$9F,$D8,$F4,$7B,$1F,$9B,$FA,$6F,$F0
	.dc.b	$F9,$FD,$5C,$FE,$20,$7E,$1C,$FC,$37,$2F,$EA,$17,$AF,$C7,$D7,$E5
	.dc.b	$EB,$F3,$F5,$FA,$7A,$FD,$7D,$7E,$DE,$BF,$7F,$5F,$C1,$EB,$F8,$7D
	.dc.b	$7F,$17,$AF,$E3,$F5,$FC,$9E,$BF,$97,$D7,$F3,$7A,$FE,$7F,$5F,$F6
	.dc.b	$5F,$4F,$FE,$1E,$3F,$87,$9F,$87,$E4,$3F,$87,$6F,$AF,$C3,$D7,$E3
	.dc.b	$EB,$F2,$F5,$F9,$FA,$FD,$3D,$7E,$BE,$BF,$6F,$5F,$BF,$AF,$E0,$F5
	.dc.b	$FC,$3E,$BF,$8B,$D7,$F1,$FA,$FE,$4F,$5F,$CB,$EB,$F9,$BD,$7F,$3F
	.dc.b	$AF,$FC,$27,$EA,$3F,$0E,$7F,$87,$8F,$EA,$C7,$E1,$8F,$C5,$72,$7F
	.dc.b	$16,$5E,$BF,$1F,$5F,$97,$AF,$CF,$D7,$E9,$EB,$F5,$F5,$FB,$7A,$FD
	.dc.b	$FD,$7F,$07,$AF,$E1,$F5,$FC,$5E,$BF,$8F,$D7,$F2,$7A,$FE,$5F,$5F
	.dc.b	$CD,$EB,$F9,$FD,$7F,$47,$AF,$FC,$87,$C5,$FF,$32,$9F,$E1,$D3,$FA
	.dc.b	$B1,$FC,$34,$FC,$3F,$29,$FC,$3B,$FC,$C1,$FE,$AC,$3D,$4E,$9E,$A7
	.dc.b	$5F,$53,$B7,$A9,$DF,$D4,$F0,$7A,$9E,$1F,$53,$C5,$EA,$78,$FD,$4F
	.dc.b	$27,$A9,$E5,$F5,$3C,$DE,$A7,$9F,$D4,$F4,$7A,$9F,$CE,$7E,$A7,$FC
	.dc.b	$CA,$7F,$56,$3F,$10,$7E,$1C,$F4,$71,$7F,$46,$5E,$A7,$1F,$53,$97
	.dc.b	$A9,$CF,$D4,$E9,$EA,$75,$F5,$3B,$7A,$9D,$FD,$4F,$07,$A9,$E1,$F5
	.dc.b	$3C,$5E,$A7,$8F,$D4,$F2,$7A,$9E,$5F,$53,$CD,$EA,$79,$FD,$4F,$EA
	.dc.b	$1F,$55,$FC,$38,$FF,$32,$9F,$E1,$D3,$F8,$69,$F8,$8E,$43,$F8,$87
	.dc.b	$FA,$FA,$F4,$F9,$FA,$7D,$3D,$3E,$BE,$9F,$6F,$4F,$BF,$A7,$E0,$F4
	.dc.b	$FC,$3E,$9F,$8B,$D3,$F1,$FA,$7E,$4F,$4F,$CB,$E9,$F9,$BD,$3F,$3F
	.dc.b	$A7,$E8,$F4,$FD,$3E,$9F,$F6,$9F,$AB,$FF,$56,$9F,$E6,$53,$FC,$3C
	.dc.b	$7F,$55,$3F,$12,$7E,$23,$90,$FF,$87,$4F,$C4,$3F,$DE,$97,$A5,$D7
	.dc.b	$D2,$ED,$E9,$77,$F4,$BC,$1E,$97,$87,$D2,$F1,$7A,$5E,$3F,$4B,$C9
	.dc.b	$E9,$79,$7D,$2F,$37,$A5,$E7,$F4,$BD,$1E,$97,$F8,$07,$DD,$FF,$56
	.dc.b	$9F,$E1,$E3,$F8,$81,$FC,$3C,$FC,$4F,$0F,$F6,$DF,$4B,$87,$A5,$C7
	.dc.b	$D2,$E5,$E9,$73,$F4,$BA,$7A,$5D,$7D,$2E,$DE,$97,$7F,$4B,$C1,$E9
	.dc.b	$78,$7D,$2F,$17,$A5,$E3,$F4,$BC,$9E,$97,$97,$D2,$F3,$7A,$5F,$E1
	.dc.b	$7F,$AC,$FE,$AA,$7F,$AB,$0F,$F3,$21,$FD,$50,$FC,$41,$F8,$7E,$33
	.dc.b	$F8,$77,$FC,$4E,$FF,$AB,$0F,$F0,$EB,$AF,$3F,$B7,$9F,$DF,$CF,$F0
	.dc.b	$79,$FE,$1F,$3F,$C5,$E7,$F8,$FC,$FF,$27,$9F,$E5,$F3,$FC,$DE,$7F
	.dc.b	$9F,$CF,$F4,$79,$FE,$9F,$3F,$FC,$61,$F5,$BF,$EF,$43,$FD,$58,$7F
	.dc.b	$99,$4F,$F0,$E1,$FC,$40,$FC,$41,$FE,$63,$3F,$0F,$C0,$7F,$0E,$FF
	.dc.b	$90,$97,$9D,$DB,$CE,$EF,$E7,$78,$3C,$EF,$0F,$9D,$E2,$F3,$BC,$7E
	.dc.b	$77,$93,$CE,$F2,$F9,$DE,$6F,$3B,$CF,$E7,$7A,$3C,$EF,$F2,$47,$D7
	.dc.b	$7F,$BD,$0F,$F3,$29,$FD,$5C,$FE,$AC,$7E,$20,$FE,$1F,$C1,$FE,$56
	.dc.b	$5E,$6F,$2F,$37,$9F,$9B,$D3,$CD,$EB,$E6,$F6,$F3,$7B,$F9,$BE,$0F
	.dc.b	$37,$C3,$E6,$F8,$BC,$DF,$1F,$9B,$E4,$F3,$7C,$BE,$6F,$9B,$CD,$F3
	.dc.b	$F9,$BF,$E5,$FF,$67,$FC,$3A,$7F,$BD,$0F,$F5,$61,$FE,$1C,$3F,$88
	.dc.b	$1F,$89,$3F,$11,$C7,$FD,$63,$D4,$E5,$EA,$73,$F3,$FA,$79,$FD,$7C
	.dc.b	$FE,$DE,$7F,$7F,$3F,$C1,$E7,$F8,$7C,$FF,$17,$9F,$E3,$F3,$FC,$9E
	.dc.b	$7F,$97,$CF,$F3,$79,$FE,$7F,$3F,$FC,$ED,$F5,$EF,$97,$BF,$27,$8F
	.dc.b	$90,$CF,$C4,$EB,$D7,$DF,$C1,$70,$F8,$2E,$3E,$FF,$2F,$6F,$9F,$B1
	.dc.b	$D3,$D8,$EB,$EC,$76,$F6,$3B,$FB,$1E,$0F,$63,$C3,$EC,$78,$BD,$8F
	.dc.b	$1F,$B1,$E4,$F6,$3C,$BE,$C7,$9B,$D8,$FF,$46,$7D,$83,$E6,$2F,$C8
	.dc.b	$E3,$E3,$33,$F1,$9A,$F8,$5D,$FC,$0F,$0F,$7B,$8F,$BD,$CB,$DE,$E7
	.dc.b	$EF,$74,$F7,$BA,$FB,$DD,$BD,$EE,$FE,$F7,$83,$DE,$F0,$FB,$DE,$2F
	.dc.b	$7B,$C7,$EF,$79,$3D,$EF,$2F,$BD,$E6,$F7,$BF,$D4,$5F,$63,$FE,$60
	.dc.b	$1F,$31,$8F,$96,$CF,$C8,$EB,$E3,$77,$F0,$7C,$3E,$23,$8F,$C1,$F2
	.dc.b	$F8,$3E,$7F,$07,$D3,$E0,$FA,$FC,$1F,$6F,$83,$EF,$F0,$7E,$0F,$83
	.dc.b	$F0,$FC,$1F,$8B,$E0,$FC,$7F,$07,$E4,$F8,$3F,$2F,$C1,$F9,$BE,$0F
	.dc.b	$CF,$F0,$7F,$EB,$9F,$1F,$F0,$9F,$09,$FE,$C2,$5F,$05,$9F,$82,$D7
	.dc.b	$C1,$6F,$E0,$B8,$7C,$17,$1F,$82,$E5,$F0,$5C,$FE,$0B,$A7,$C1,$75
	.dc.b	$F8,$2E,$DF,$05,$DF,$E0,$BC,$1F,$05,$E1,$F8,$2F,$17,$C1,$78,$FE
	.dc.b	$0B,$C9,$F0,$5E,$5F,$82,$F3,$7C,$17,$9F,$E0,$BF,$DA,$1F,$65,$F8
	.dc.b	$5E,$8F,$F6,$CB,$FE,$EE,$3F,$CC,$07,$F9,$7C,$FE,$A2,$7F,$0A,$4F
	.dc.b	$F0,$A7,$8F,$B9,$D3,$DC,$EB,$EE,$76,$F7,$3B,$FB,$9E,$0F,$73,$C3
	.dc.b	$EE,$78,$BD,$CF,$1F,$B9,$E4,$F7,$3C,$BE,$E7,$9B,$DC,$F3,$FB,$9E
	.dc.b	$8F,$73,$FD,$D1,$F6,$7F,$EA,$63,$F0,$FC,$3F,$EF,$25,$ED,$EB,$DB
	.dc.b	$DF,$B7,$C3,$DB,$E3,$ED,$F2,$F6,$F9,$FB,$7D,$3D,$BE,$BE,$DF,$6F
	.dc.b	$6F,$BF,$B7,$E0,$F6,$FC,$3E,$DF,$8B,$DB,$F1,$FB,$7E,$4F,$6F,$CB
	.dc.b	$ED,$F9,$BD,$BF,$3F,$B7,$E8,$F6,$FF,$E0,$07,$DA,$7E,$23,$A0,$FE
	.dc.b	$21,$18,$F6,$B7,$ED,$70,$F6,$B8,$FB,$5C,$BD,$AE,$7E,$D7,$4F,$6B
	.dc.b	$AF,$B5,$DB,$DA,$EF,$ED,$78,$3D,$AF,$0F,$B5,$E2,$F6,$BC,$7E,$D7
	.dc.b	$93,$DA,$F2,$FB,$5E,$6F,$6B,$CF,$ED,$7A,$3D,$AF,$2F,$2F,$FA,$E3
	.dc.b	$E1,$2F,$1B,$E1,$2F,$C2,$63,$E1,$33,$F0,$9A,$F8,$1D,$FC,$27,$0F
	.dc.b	$80,$E3,$F0,$1C,$BE,$03,$9F,$C0,$74,$F8,$0E,$BF,$01,$DB,$E0,$3B
	.dc.b	$FC,$07,$83,$E0,$3C,$3F,$01,$E2,$F8,$0F,$1F,$C0,$79,$3E,$03,$CB
	.dc.b	$F0,$1E,$8F,$B5,$FF,$76,$9F,$88,$E9,$3F,$88,$7E,$8B,$FE,$EE,$3F
	.dc.b	$CC,$47,$F8,$6C,$FE,$A6,$3D,$BE,$9E,$DF,$5F,$6F,$B7,$B7,$DF,$DB
	.dc.b	$F0,$7B,$7E,$1F,$6F,$C5,$ED,$F8,$FD,$BF,$27,$B7,$E5,$F6,$FC,$DE
	.dc.b	$DF,$9F,$DB,$F4,$7B,$7E,$9F,$6F,$E8,$FD,$B7,$E2,$4F,$C4,$71,$97
	.dc.b	$B7,$AF,$6F,$7E,$DF,$0F,$2F,$8F,$B7,$CB,$DB,$E7,$ED,$F4,$F6,$FA
	.dc.b	$FB,$7D,$BD,$BE,$FE,$DF,$83,$DB,$F0,$FB,$7E,$2F,$6F,$C7,$ED,$F9
	.dc.b	$3D,$BF,$2F,$B7,$E6,$F6,$FC,$FE,$DF,$E2,$3E,$DF,$F8,$87,$39,$FC
	.dc.b	$43,$FA,$07,$FD,$E0,$7F,$BB,$CF,$F3,$09,$FC,$44,$FE,$20,$F1,$F5
	.dc.b	$7A,$7A,$BD,$7D,$5E,$DE,$AF,$7F,$57,$C1,$EA,$F8,$7D,$5F,$17,$AB
	.dc.b	$E3,$F5,$7C,$9E,$AF,$97,$D5,$F3,$7A,$BE,$7F,$57,$D1,$EA,$FE,$83
	.dc.b	$DE,$FD,$0F,$0F,$FA,$87,$E1,$B1,$F0,$79,$F8,$3D,$7C,$1E,$FE,$0F
	.dc.b	$86,$1C,$7D,$AE,$5E,$D7,$3F,$6B,$A7,$B5,$D7,$DA,$ED,$ED,$77,$F6
	.dc.b	$BC,$1E,$D7,$87,$DA,$F1,$7B,$5E,$3F,$6B,$C9,$ED,$79,$7D,$AF,$37
	.dc.b	$B5,$E7,$F6,$BF,$6F,$F7,$1F,$C3,$CF,$C3,$9F,$86,$E8,$FE,$1E,$BD
	.dc.b	$9D,$FB,$3C,$3D,$9E,$3E,$CF,$2F,$67,$9F,$B3,$D3,$D9,$EB,$EC,$F6
	.dc.b	$F6,$7B,$FB,$3E,$0F,$67,$C3,$EC,$F8,$BD,$9F,$1F,$B3,$E4,$F6,$7C
	.dc.b	$BE,$CF,$9B,$D9,$F3,$FB,$3F,$C3,$BE,$E7,$F0,$E7,$E1,$B9,$8B,$E0
	.dc.b	$B5,$F0,$5B,$F8,$2E,$1F,$05,$C7,$E0,$B9,$7B,$DC,$FD,$EE,$9E,$F3
	.dc.b	$F8,$06,$17,$4F,$E6,$3A,$29,$F7,$3F,$80,$9C,$C6,$1C,$4E,$E7,$51
	.dc.b	$E6,$9D,$3F,$88,$9D,$3C,$23,$2F,$96,$78,$CE,$A3,$84,$E9,$F0,$CE
	.dc.b	$A3,$11,$DE,$38,$E1,$E2,$8F,$8E,$DA,$34,$C5,$7A,$FE,$BB,$FC,$DF
	.dc.b	$F0,$5F,$AE,$6E,$7F,$D0,$1F,$F3,$9D,$FA,$9B,$5C,$7D,$2F,$D1,$E3
	.dc.b	$FD,$6A,$7F,$CF,$31,$FC,$B6,$11,$C3,$2D,$4F,$D7,$9F,$83,$87,$9C
	.dc.b	$8F,$BE,$76,$FF,$07,$8F,$24,$FF,$0D,$6C,$7C,$63,$BC,$7C,$D6,$9F
	.dc.b	$DB,$9F,$25,$79,$BD,$6E,$33,$F2,$CD,$9F,$90,$DB,$E5,$11,$FA,$EF
	.dc.b	$CA,$C3,$FB,$CF,$C3,$7C,$89,$B8,$79,$5A,$79,$63,$E1,$3F,$DA,$63
	.dc.b	$E4,$B4,$F3,$5F,$F7,$09,$D9,$F4,$B8,$3E,$E1,$27,$F1,$2F,$8D,$F4
	.dc.b	$EB,$F6,$EF,$C5,$1E,$CD,$7E,$D4,$7C,$5C,$FE,$14,$FC,$E1,$1F,$18
	.dc.b	$D1,$FB,$07,$F2,$FC,$3F,$8D,$23,$F5,$CF,$0E,$BF,$33,$0F,$14,$7D
	.dc.b	$5B,$F8,$27,$81,$F9,$F7,$DD,$D3,$FC,$0F,$8B,$F3,$44,$8F,$D3,$BE
	.dc.b	$F9,$FA,$11,$E0,$3D,$58,$FB,$9E,$67,$CC,$3D,$D5,$DD,$FC,$04,$FE
	.dc.b	$7B,$3F,$D4,$12,$3F,$6A,$F7,$9F,$97,$3E,$03,$D6,$FC,$69,$EC,$7B
	.dc.b	$34,$FC,$6D,$F8,$E3,$F8,$DF,$C3,$FE,$D4,$8F,$F0,$27,$B3,$F9,$F3
	.dc.b	$EA,$65,$F2,$9F,$3D,$F7,$7E,$BD,$FE,$6A,$3D,$97,$DF,$C6,$72,$FF
	.dc.b	$05,$6D,$FF,$18,$6B,$FC,$7D,$AF,$F1,$2E,$0F,$0E,$DF,$6C,$F5,$F1
	.dc.b	$7F,$01,$9F,$9A,$78,$8F,$9B,$96,$AB,$F1,$96,$E6,$F1,$12,$BC,$4F
	.dc.b	$C0,$7F,$C0,$D8,$77,$7E,$DE,$1D,$DF,$82,$7C,$ED,$BF,$14,$7D,$27
	.dc.b	$FA,$EF,$3F,$BC,$8F,$BD,$ED,$B7,$CB,$3E,$66,$BF,$54,$FA,$DB,$7E
	.dc.b	$14,$F9,$78,$78,$73,$74,$7C,$B2,$63,$F2,$DE,$B9,$FA,$63,$F6,$15
	.dc.b	$E9,$C3,$AE,$DF,$5C,$F8,$D3,$FC,$43,$07,$4F,$D7,$24,$7D,$77,$AB
	.dc.b	$F5,$0F,$8B,$BC,$7F,$20,$7D,$23,$1B,$7E,$F1,$FF,$00,$F5,$7E,$0C
	.dc.b	$8F,$E0,$DE,$AF,$EA,$CF,$4C,$F2,$D7,$F8,$BD,$7D,$07,$F2,$58,$3A
	.dc.b	$FF,$1C,$4B,$FC,$73,$D2,$BF,$51,$5F,$99,$CF,$E1,$D7,$9C,$EE,$7C
	.dc.b	$3E,$CF,$CC,$11,$FC,$C3,$CE,$7F,$66,$73,$D7,$F0,$CF,$D9,$D7,$E6
	.dc.b	$E1,$F0,$6D,$93,$B7,$F4,$84,$8F,$D2,$3D,$5F,$D9,$1E,$49,$FC,$73
	.dc.b	$F9,$F3,$E7,$99,$BF,$F1,$23,$FA,$DE,$EF,$D8,$11,$FE,$4D,$3F,$09
	.dc.b	$EA,$F4,$CF,$E1,$C7,$48,$EF,$83,$BF,$F7,$44,$9F,$DD,$3E,$77,$F5
	.dc.b	$27,$D7,$3F,$6B,$87,$C3,$70,$7C,$2F,$03,$F8,$21,$1F,$E0,$8F,$3E
	.dc.b	$4F,$D2,$EB,$F9,$93,$E6,$EB,$F9,$9A,$E7,$8B,$C1,$FE,$1C,$48,$FE
	.dc.b	$1C,$F3,$E4,$FA,$79,$7F,$19,$3C,$93,$F2,$4F,$F2,$AF,$85,$FC,$5C
	.dc.b	$8F,$F1,$77,$93,$EA,$DB,$E3,$ED,$F9,$38,$FE,$27,$8B,$C3,$FE,$40
	.dc.b	$48,$FE,$40,$F2,$7C,$07,$F0,$C3,$EE,$4F,$37,$12,$F1,$3F,$95,$12
	.dc.b	$3F,$95,$3C,$E7,$ED,$6B,$F1,$A7,$F3,$91,$F6,$BC,$5F,$E6,$44,$8F
	.dc.b	$E6,$4F,$1C,$FF,$35,$CD,$C1,$FD,$79,$FA,$EF,$1B,$F9,$D1,$2B,$F9
	.dc.b	$D3,$C5,$FE,$67,$A7,$C4,$9E,$31,$F1,$3E,$17,$FA,$11,$1F,$E8,$4F
	.dc.b	$17,$E1,$4F,$17,$F9,$74,$FE,$FB,$E2,$FF,$A5,$12,$3F,$A5,$3C,$67
	.dc.b	$F9,$56,$7F,$06,$7F,$35,$5F,$C1,$BE,$3B,$FA,$91,$23,$FA,$93,$C6
	.dc.b	$7F,$1D,$97,$C1,$9F,$09,$F9,$AF,$2B,$FC,$BD,$66,$92,$3F,$5B,$1F
	.dc.b	$C1,$1F,$D6,$BF,$31,$F1,$9D,$0F,$C0,$2F,$2F,$A9,$23,$FC,$FB,$19
	.dc.b	$7A,$9D,$08,$7C,$8B,$F8,$1B,$7E,$27,$2F,$E9,$7E,$67,$A0,$95,$D1
	.dc.b	$F8,$19,$7E,$D3,$0F,$D1,$DF,$F0,$E5,$E6,$FC,$02,$5F,$91,$E2,$FE
	.dc.b	$6A,$38,$9F,$AD,$F3,$BE,$39,$23,$D6,$F0,$AE,$DD,$1F,$BB,$9E,$5E
	.dc.b	$7F,$CE,$23,$F0,$9E,$17,$FB,$99,$E1,$5E,$7F,$43,$EA,$92,$FE,$4B
	.dc.b	$C1,$FE,$27,$3C,$2B,$F8,$17,$A3,$F7,$09,$3F,$41,$E0,$FD,$69,$E0
	.dc.b	$FF,$1C,$F4,$BF,$06,$47,$EB,$BC,$36,$FC,$D4,$7E,$F2,$FF,$88,$7A
	.dc.b	$7F,$8A,$24,$7D,$E7,$87,$07,$DA,$9F,$D4,$CF,$E1,$BD,$4F,$C8,$91
	.dc.b	$FC,$2B,$C3,$83,$F3,$11,$DC,$FE,$63,$EA,$FE,$60,$8F,$E3,$1E,$0F
	.dc.b	$E5,$0E,$07,$E5,$3D,$6F,$CF,$91,$FC,$9B,$C3,$7F,$E2,$D8,$7F,$85
	.dc.b	$AF,$FC,$BB,$EB,$FE,$98,$97,$F9,$A7,$84,$FB,$0E,$11,$FE,$72,$F6
	.dc.b	$3F,$5A,$47,$F4,$2F,$07,$F5,$11,$C2,$FF,$D7,$EF,$ED,$09,$1F,$A8
	.dc.b	$78,$3E,$C7,$80,$7F,$D6,$9E,$CF,$EF,$08,$FE,$C1,$E3,$1F,$9C,$8F
	.dc.b	$EB,$7E,$D7,$F0,$22,$3F,$B6,$77,$C3,$F8,$68,$FD,$EF,$B7,$FC,$20
	.dc.b	$8F,$EF,$1F,$C6,$D7,$FB,$5F,$37,$5F,$FA,$D1,$B5,$FE,$7C,$6B,$F9
	.dc.b	$33,$F8,$2D,$FF,$14,$23,$FC,$18,$FE,$E6,$7F,$8A,$F2,$FD,$7B,$C0
	.dc.b	$87,$C3,$3F,$D8,$65,$3F,$D6,$7E,$E7,$F2,$02,$3F,$C5,$1F,$49,$87
	.dc.b	$81,$C0,$D1,$FC,$68,$FE,$52,$43,$07,$FA,$E7,$D5,$FF,$55,$93,$1F
	.dc.b	$F7,$53,$FE,$FC,$8E,$07,$68,$EE,$FA,$5A,$FD,$DF,$21,$EE,$E1,$B1
	.dc.b	$C1,$AF,$A1,$AF,$FA,$44,$F9,$13,$FC,$ED,$FE,$ED,$1A,$7E,$43,$5F
	.dc.b	$04,$F7,$3C,$1E,$F4,$8F,$03,$5F,$8F,$CF,$FE,$B5,$AF,$F4,$27,$93
	.dc.b	$FE,$16,$9F,$55,$6D,$FA,$92,$8C,$47,$8B,$DF,$F7,$11,$F7,$3F,$C4
	.dc.b	$FE,$1F,$EF,$6F,$F8,$AC,$5F,$0F,$8E,$1F,$C1,$46,$30,$FD,$35,$6E
	.dc.b	$F5,$1F,$37,$E0,$BE,$61,$1F,$98,$F7,$78,$DF,$4C,$38,$C3,$0E,$CB
	.dc.b	$0E,$CF,$03,$ED,$7C,$1F,$D9,$23,$F6,$5F,$E5,$DF,$0F,$D2,$8D,$31
	.dc.b	$F0,$FE,$0D,$9F,$0D,$F8,$32,$61,$F8,$37,$9B,$E1,$3C,$D5,$CD,$C5
	.dc.b	$AF,$A7,$C3,$FE,$34,$93,$F8,$D7,$FA,$11,$E5,$B9,$1C,$A3,$C9,$3F
	.dc.b	$95,$24,$7E,$55,$E2,$F8,$27,$8B,$F1,$F4,$FA,$F5,$ED,$C3,$F3,$C4
	.dc.b	$9F,$CF,$3F,$D4,$F2,$F8,$DA,$C9,$F5,$C7,$C5,$D7,$F4,$C4,$8F,$D3
	.dc.b	$3C,$5F,$83,$3C,$5F,$11,$FC,$AF,$35,$7C,$FD,$BF,$60,$49,$FD,$83
	.dc.b	$FD,$97,$7F,$B0,$48,$F9,$BC,$1F,$5F,$E1,$FF,$70,$48,$FD,$C3,$C5
	.dc.b	$FA,$73,$C5,$F1,$9F,$8D,$9F,$C0,$78,$3F,$C0,$49,$1F,$C0,$5F,$E0
	.dc.b	$9C,$7F,$85,$24,$7D,$E8,$FF,$80,$1C,$47,$8C,$7F,$07,$6B,$FB,$4F
	.dc.b	$86,$FE,$1C,$47,$F8,$73,$D9,$FC,$64,$AB,$BC,$7A,$34,$FE,$05,$18
	.dc.b	$E4,$FE,$7F,$5B,$AE,$38,$70,$7F,$81,$D7,$57,$F8,$D3,$5C,$9F,$C7
	.dc.b	$DC,$3F,$C6,$1A,$FF,$B3,$8D,$9F,$C8,$FE,$1B,$F9,$41,$1F,$E5,$0D
	.dc.b	$74,$60,$86,$08,$7F,$18,$78,$3F,$C6,$BE,$1B,$F9,$A1,$1F,$E6,$8D
	.dc.b	$7F,$9A,$35,$E0,$68,$86,$8A,$7F,$26,$F8,$6F,$E8,$04,$7F,$A0,$35
	.dc.b	$FE,$80,$D7,$81,$A2,$1A,$29,$FC,$CB,$E1,$BF,$A6,$11,$FE,$98,$D7
	.dc.b	$FA,$63,$5E,$06,$88,$68,$A7,$F3,$EE,$BF,$EB,$04,$7F,$AC,$3F,$D7
	.dc.b	$B4,$FE,$B3,$A7,$F8,$13,$EE,$FF,$BC,$36,$BF,$92,$75,$FF,$30,$97
	.dc.b	$6D,$EF,$95,$FC,$7E,$72,$FE,$5B,$E1,$FF,$B4,$11,$FE,$D0,$F8,$6F
	.dc.b	$A4,$43,$F9,$2B,$DE,$FA,$C7,$ED,$2B,$FC,$3D,$D9,$FE,$0F,$7B,$C7
	.dc.b	$EE,$3E,$25,$F1,$1F,$DE,$9B,$1F,$DF,$9A,$FF,$83,$9A,$FF,$85,$D7
	.dc.b	$C4,$7F,$4E,$6C,$7F,$3B,$6B,$E3,$F1,$35,$32,$34,$FC,$7F,$17,$11
	.dc.b	$DE,$3C,$84,$7C,$8D,$74,$60,$86,$08,$79,$DF,$77,$C4,$48,$F7,$35
	.dc.b	$E1,$BF,$DD,$F1,$FE,$D1,$E4,$7F,$C8,$78,$FB,$1F,$C1,$78,$DF,$7C
	.dc.b	$D6,$BF,$E3,$D9,$7C,$47,$CD,$26,$1F,$35,$F5,$5F,$78,$FC,$26,$1C
	.dc.b	$9F,$F3,$45,$77,$BF,$B5,$87,$79,$F9,$27,$CF,$F8,$8F,$B4,$47,$ED
	.dc.b	$3F,$77,$E1,$FE,$B1,$23,$F2,$5E,$5D,$7F,$09,$17,$2F,$CF,$79,$9F
	.dc.b	$77,$E2,$3F,$0C,$4A,$FC,$33,$D5,$FC,$05,$71,$CB,$F2,$18,$59,$F5
	.dc.b	$CD,$5C,$6C,$3C,$51,$F8,$D9,$E8,$FE,$45,$AE,$4F,$CB,$38,$7E,$83
	.dc.b	$5E,$E6,$8F,$CE,$7C,$47,$E7,$C8,$E0,$D1,$FA,$9F,$3F,$F7,$47,$9F
	.dc.b	$65,$7E,$F7,$D0,$95,$F0,$0F,$D0,$7C,$47,$EA,$C8,$FE,$AD,$F5,$ED
	.dc.b	$FC,$27,$0E,$B3,$C4,$FA,$27,$F0,$5A,$EA,$E0,$FD,$C7,$C4,$7E,$DC
	.dc.b	$8F,$ED,$DE,$BB,$7E,$FE,$BA,$CF,$1B,$FC,$CE,$3D,$1F,$E3,$58,$7B
	.dc.b	$7E,$23,$F8,$11,$2B,$F8,$13,$5F,$E0,$4D,$7F,$81,$35,$FD,$03,$5F
	.dc.b	$CC,$BF,$C3,$7E,$22,$47,$F0,$D6,$BF,$C3,$5A,$FF,$0D,$6B,$FA,$B6
	.dc.b	$BF,$A4,$6A,$D1,$F1,$1F,$C6,$C9,$1F,$C6,$DA,$FF,$1B,$6B,$FC,$6D
	.dc.b	$AF,$EE,$5A,$FE,$C9,$FE,$4D,$F1,$12,$3F,$93,$35,$FE,$4C,$D7,$F9
	.dc.b	$33,$5F,$E0,$8D,$7F,$7A,$D7,$D5,$F1,$1F,$CC,$C8,$FF,$33,$6B,$FC
	.dc.b	$CD,$AF,$F3,$36,$BF,$C4,$5A,$FF,$0A,$7F,$9F,$7C,$3C,$8F,$E7,$CD
	.dc.b	$7F,$9F,$35,$FE,$7C,$D7,$F8,$E3,$5F,$E2,$AD,$7E,$97,$C3,$FF,$4B
	.dc.b	$23,$FD,$2D,$AF,$F4,$B6,$BF,$D2,$DA,$FF,$29,$6B,$FC,$89,$FE,$AD
	.dc.b	$F0,$F2,$3F,$AB,$35,$FE,$AC,$D7,$FA,$B3,$5F,$E6,$8D,$7F,$96,$B5
	.dc.b	$FC,$57,$C3,$FF,$63,$23,$FD,$8D,$AF,$F6,$36,$BF,$D8,$DA,$FF,$41
	.dc.b	$6B,$FC,$E9,$FE,$DD,$F0,$F2,$3F,$B7,$35,$FF,$7D,$FC,$53,$7F,$F5
	.dc.b	$CB,$63,$FD,$4C,$D7,$FB,$F7,$BD,$FE,$0F,$F7,$F7,$52,$7F,$C1,$D1
	.dc.b	$FD,$D7,$97,$FB,$F1,$23,$FB,$F3,$FE,$FA,$BE,$9C,$68,$4C,$9C,$C6
	.dc.b	$21,$18,$77,$D4,$C7,$C5,$1F,$E8,$69,$D3,$C3,$1F,$43,$EC,$F8,$8F
	.dc.b	$11,$1E,$C7,$79,$47,$9E,$7A,$E3,$E7,$AF,$34,$7F,$8C,$6F,$38,$F1
	.dc.b	$8B,$3E,$6C,$BE,$6F,$C4,$7C,$12,$47,$80,$F8,$0F,$77,$E8,$CF,$27
	.dc.b	$FA,$CD,$DC,$BD,$A7,$BB,$36,$1F,$6B,$E2,$3E,$71,$2F,$B1,$F2,$9E
	.dc.b	$CF,$C7,$C3,$B5,$7C,$B8,$F0,$90,$D1,$F7,$8F,$C2,$FC,$47,$DB,$23
	.dc.b	$DB,$04,$76,$D3,$E4,$4F,$D3,$3A,$3F,$E5,$DC,$BD,$98,$78,$B2,$FC
	.dc.b	$9F,$C4,$7E,$1C,$97,$D5,$E9,$AF,$CF,$8E,$6F,$F9,$46,$F9,$C6,$0F
	.dc.b	$A4,$7E,$73,$E2,$3F,$24,$47,$D0,$F4,$7E,$E9,$C4,$FA,$25,$CF,$E8
	.dc.b	$E3,$F8,$C8,$FC,$87,$C4,$7E,$74,$8F,$C3,$7A,$3F,$04,$F9,$67,$EB
	.dc.b	$0E,$91,$F0,$5F,$D8,$FC,$47,$E9,$C8,$FC,$D7,$9D,$78,$A3,$FC,$FD
	.dc.b	$87,$C0,$7A,$E9,$FE,$89,$D3,$F7,$BF,$11,$FB,$12,$4F,$D7,$79,$E3
	.dc.b	$F8,$2B,$E5,$7F,$63,$0F,$C7,$61,$FC,$18,$FD,$D1,$30,$FC,$03,$C9
	.dc.b	$F9,$1B,$FE,$2D,$F1,$EB,$F5,$DA,$7D,$43,$F7,$18,$7F,$05,$23,$F8
	.dc.b	$B7,$A3,$C0,$FA,$65,$BF,$84,$FF,$A4,$EF,$F8,$1B,$FB,$C8,$FE,$34
	.dc.b	$DC,$7F,$42,$D3,$F3,$0E,$1F,$31,$93,$F9,$3B,$97,$F0,$A7,$F2,$AE
	.dc.b	$FF,$E3,$84,$7F,$44,$FE,$03,$16,$7F,$EB,$73,$E6,$E5,$FC,$77,$A1
	.dc.b	$EE,$FF,$1F,$D7,$F4,$F1,$8C,$7F,$21,$E2,$7F,$29,$23,$FA,$F7,$B6
	.dc.b	$1D,$63,$F1,$67,$F2,$CA,$E7,$7D,$70,$F0,$3F,$C2,$FC,$5F,$E6,$84
	.dc.b	$AF,$DD,$3D,$1F,$F6,$55,$FA,$1F,$E6,$F8,$F9,$56,$1D,$9F,$D3,$79
	.dc.b	$1F,$CF,$C8,$FF,$05,$7A,$3F,$EC,$9A,$E8,$AB,$A5,$73,$8F,$BD,$E4
	.dc.b	$FF,$49,$23,$FC,$3D,$E7,$1F,$EE,$7A,$E7,$5C,$B2,$F7,$DF,$EE,$7C
	.dc.b	$DF,$EA,$44,$7F,$8C,$3C,$A3,$FD,$B5,$5C,$B3,$F9,$15,$F0,$1F,$E6
	.dc.b	$5E,$77,$F5,$B2,$47,$F2,$17,$93,$FD,$5A,$7F,$9D,$61,$E3,$8E,$47
	.dc.b	$D5,$BF,$2B,$F9,$BF,$4F,$FB,$31,$1F,$E5,$8F,$47,$FD,$37,$5D,$30
	.dc.b	$F2,$C7,$C9,$6D,$F7,$8F,$E2,$19,$7E,$EF,$D1,$FE,$E4,$48,$FE,$70
	.dc.b	$F5,$E0,$FC,$BC,$7F,$21,$AE,$44,$BF,$EA,$51,$FD,$EC,$87,$F4,$13
	.dc.b	$96,$1E,$57,$DC,$7C,$03,$F0,$F9,$7C,$0C,$7F,$87,$71,$7A,$3D,$2F
	.dc.b	$EA,$F5,$F8,$36,$E2,$69,$DA,$3C,$D8,$FC,$02,$38,$3F,$B0,$CF,$E5
	.dc.b	$DF,$EF,$ED,$7E,$89,$FB,$FB,$FC,$53,$FA,$03,$91,$FD,$DE,$38,$3E
	.dc.b	$B3,$9C,$F9,$BD,$E9,$1E,$06,$BF,$C2,$8E,$87,$E8,$4E,$B5,$FD,$9B
	.dc.b	$37,$47,$DC,$6C,$FC,$A3,$EF,$17,$07,$C1,$7C,$62,$47,$C0,$7E,$4F
	.dc.b	$B5,$F8,$42,$47,$3A,$75,$FE,$23,$E0,$A4,$F5,$7C,$5D,$B8,$8F,$85
	.dc.b	$1F,$50,$8F,$CE,$BE,$AF,$DE,$3F,$83,$BE,$BA,$E5,$1F,$4A,$78,$BD
	.dc.b	$0F,$AF,$F0,$DF,$80,$23,$83,$B1,$F0,$6B,$BE,$9F,$26,$FC,$CF,$F2
	.dc.b	$A8,$CF,$0F,$F2,$FC,$3F,$4B,$F0,$DF,$8C,$26,$1F,$86,$7E,$3B,$D4
	.dc.b	$87,$AC,$67,$F2,$1F,$8C,$6B,$0F,$03,$83,$86,$AF,$87,$FC,$D9,$23
	.dc.b	$F2,$AF,$81,$DE,$5E,$87,$FC,$BF,$9F,$B3,$3F,$E9,$13,$99,$FE,$51
	.dc.b	$9F,$F0,$0F,$87,$FD,$39,$23,$F4,$2F,$5C,$BC,$B1,$D8,$FC,$16,$1D
	.dc.b	$5F,$AE,$7A,$F0,$F9,$0F,$E3,$FE,$1F,$F6,$64,$7F,$5A,$F5,$8F,$2C
	.dc.b	$71,$3D,$27,$C3,$AF,$09,$23,$AC,$7C,$6F,$87,$FD,$F9,$1F,$DC,$BD
	.dc.b	$5F,$29,$E9,$3E,$A6,$BF,$4D,$C5,$75,$76,$FF,$6C,$CB,$F9,$2F,$C3
	.dc.b	$FF,$0C,$24,$FF,$06,$7B,$3F,$D4,$F4,$F9,$D7,$D6,$FA,$4E,$4F,$71
	.dc.b	$F4,$8F,$DA,$7C,$2F,$F1,$82,$3F,$C4,$9E,$BB,$AF,$AE,$BF,$6E,$31
	.dc.b	$5D,$16,$5F,$CF,$3E,$1F,$F9,$19,$27,$F8,$EB,$D2,$F9,$C7,$17,$A4
	.dc.b	$FF,$44,$F8,$6F,$E5,$64,$9F,$E4,$AF,$17,$D7,$9F,$BF,$6F,$B7,$1F
	.dc.b	$31,$BA,$7C,$B8,$FE,$25,$F0,$DF,$CD,$88,$FF,$30,$7A,$3C,$4F,$C1
	.dc.b	$94,$F9,$45,$CF,$DE,$FE,$C0,$FE,$6F,$97,$E9,$5A,$7F,$02,$3F,$9A
	.dc.b	$39,$7C,$E6,$8C,$9B,$30,$7F,$42,$F8,$6F,$E9,$A4,$7F,$A3,$BF,$82
	.dc.b	$C5,$E5,$7F,$33,$3E,$B5,$7F,$70,$F8,$B7,$C1,$53,$ED,$7C,$3F,$F5
	.dc.b	$82,$63,$FD,$49,$E9,$7C,$E3,$EB,$1F,$CF,$6B,$9B,$6B,$F0,$9F,$0F
	.dc.b	$FD,$88,$91,$FD,$71,$E6,$FF,$3B,$AF,$2B,$C4,$87,$CD,$F8,$6F,$ED
	.dc.b	$64,$7F,$B2,$BC,$9F,$D5,$C7,$F0,$DB,$F1,$C7,$F2,$16,$ED,$FE,$BC
	.dc.b	$6E,$FF,$84,$3F,$DF,$0B,$E1,$BF,$BD,$B6,$BF,$DA,$6D,$7F,$DC,$4D
	.dc.b	$7E,$C5,$7F,$63,$BF,$8D,$1F,$87,$9E,$E8,$38,$6E,$5E,$93,$C2,$47
	.dc.b	$C2,$FB,$EB,$B4,$7E,$EA,$7D,$EF,$53,$B3,$87,$DB,$3C,$DF,$4F,$C3
	.dc.b	$7B,$88,$FB,$9E,$D8,$F3,$DB,$F1,$73,$DE,$BF,$15,$F0,$72,$3E,$BE
	.dc.b	$3F,$37,$E1,$BE,$49,$23,$E4,$BD,$1F,$85,$7D,$1F,$8D,$7C,$DF,$B1
	.dc.b	$7F,$95,$7F,$1D,$F0,$DF,$50,$8F,$D4,$7A,$3F,$D7,$B4,$FC,$14,$F3
	.dc.b	$C7,$F0,$0F,$39,$FC,$27,$C3,$7D,$E2,$3F,$79,$E7,$8F,$96,$BF,$94
	.dc.b	$EA,$BE,$9A,$7E,$21,$E9,$3F,$8D,$F8,$4F,$C4,$93,$1F,$C4,$BD,$32
	.dc.b	$F2,$C7,$EA,$6B,$9B,$F6,$34,$F8,$4F,$DB,$3F,$0F,$1F,$94,$23,$FC
	.dc.b	$2E,$FF,$06,$FE,$88,$FC,$24,$69,$FC,$BB,$5E,$98,$BE,$1B,$F3,$E4
	.dc.b	$8F,$CF,$BE,$77,$99,$FE,$31,$F6,$3F,$C7,$C7,$A1,$FC,$DF,$C3,$7E
	.dc.b	$A0,$8F,$EA,$1F,$F3,$2E,$AF,$6B,$FC,$8C,$49,$FF,$1E,$7B,$7F,$E4
	.dc.b	$9F,$72,$47,$F8,$FD,$FF,$43,$F6,$B8,$7F,$8D,$7C,$37,$ED,$89,$5F
	.dc.b	$B6,$7B,$E1,$D6,$3F,$B7,$BF,$E5,$5C,$3A,$C7,$F7,$2C,$2E,$FD,$63
	.dc.b	$F6,$D8,$D7,$F8,$57,$C3,$7F,$04,$23,$FC,$11,$AF,$F0,$46,$BF,$39
	.dc.b	$AF,$F0,$E6,$BF,$B6,$7F,$87,$FC,$2C,$8F,$E1,$ED,$7F,$87,$B5,$FB
	.dc.b	$2D,$7F,$8C,$B5,$FE,$02,$D7,$F8,$7F,$C2,$FF,$1C,$23,$FC,$71,$AF
	.dc.b	$F1,$C6,$BF,$86,$6B,$FC,$99,$AF,$F0,$E7,$F9,$3F,$C2,$C8,$FE,$4E
	.dc.b	$D7,$F9,$3B,$5F,$C8,$B5,$FE,$62,$D7,$F8,$CB,$5F,$E4,$FE,$0F,$F3
	.dc.b	$42,$3F,$CD,$1A,$FF,$34,$6B,$F9,$E6,$BF,$CF,$9A,$FF,$26,$7F,$9F
	.dc.b	$FC,$2C,$8F,$E7,$ED,$7F,$9F,$B5,$FD,$4B,$5F,$E9,$2D,$7F,$98,$B5
	.dc.b	$FE,$7F,$F0,$BF,$D3,$08,$FF,$4C,$6B,$FD,$31,$AF,$ED,$9A,$FF,$56
	.dc.b	$6B,$FC,$F9,$FE,$AF,$F0,$B2,$3F,$AB,$B5,$FE,$AE,$D7,$F8,$0B,$5F
	.dc.b	$EC,$2D,$7F,$A4,$B5,$FE,$AF,$F0,$BF,$D9,$08,$FF,$64,$6B,$FD,$91
	.dc.b	$AF,$F0,$E6,$BF,$DB,$9A,$FF,$56,$7F,$B7,$F1,$48,$FE,$DE,$D7,$FB
	.dc.b	$7B,$5F,$E3,$2D,$7F,$BC,$B5,$FE,$C2,$D5,$F1,$8E,$0F,$F7,$BB,$63
	.dc.b	$F9,$71,$FD,$D5,$C3,$86,$BF,$CC,$DA,$FF,$6F,$6B,$C6,$3A,$91,$EA
	.dc.b	$78,$9C,$3D,$5F,$1F,$0C,$8D,$CA,$31,$1E,$36,$BE,$36,$BE,$36,$BE
	.dc.b	$B6,$BF,$0B,$8B,$E0,$91,$EE,$D7,$E0,$35,$F8,$0D,$7E,$03,$5F,$90
	.dc.b	$FC,$CE,$69,$1E,$A6,$BF,$2D,$AF,$CB,$6B,$F2,$DA,$FD,$36,$BF,$6B
	.dc.b	$AF,$EC,$91,$F8,$ED,$7E,$C3,$5F,$B0,$D7,$EC,$35,$FC,$03,$F8,$4F
	.dc.b	$4E,$47,$D2,$6B,$F8,$36,$BF,$83,$6B,$F8,$36,$BF,$8B,$6B,$F9,$2F
	.dc.b	$2F,$F2,$24,$7E,$FB,$5F,$C8,$35,$FC,$83,$5F,$C8,$35,$FC,$C3,$F9
	.dc.b	$CF,$56,$47,$E2,$9A,$FE,$6D,$AF,$E6,$DA,$FE,$6D,$AF,$E8,$DA,$FE
	.dc.b	$AB,$DA,$FD,$49,$1F,$CB,$B5,$FD,$43,$5F,$D4,$35,$FD,$43,$5F,$D8
	.dc.b	$3F,$B4,$F8,$59,$1F,$A2,$6B,$FB,$6F,$8D,$7B,$7F,$E5,$FE,$BF,$EE
	.dc.b	$D1,$89,$FF,$37,$46,$27,$FC,$13,$C5,$BE,$2F,$DC,$7C,$3D,$BF,$E0
	.dc.b	$0D,$7F,$83,$BF,$E1,$0E,$4C,$71,$A3,$F8,$49,$27,$F6,$EF,$EF,$96
	.dc.b	$5E,$35,$97,$8D,$F2,$E1,$FE,$77,$9D,$4F,$F8,$3F,$B1,$97,$83,$0E
	.dc.b	$D8,$7F,$13,$6B,$FC,$4F,$DD,$FE,$36,$47,$F8,$4B,$E3,$C3,$AE,$3D
	.dc.b	$F4,$F7,$46,$2B,$FC,$3F,$87,$77,$CA,$7F,$88,$B9,$FF,$C6,$26,$5F
	.dc.b	$5D,$FF,$21,$F8,$6F,$E5,$64,$8F,$E3,$2F,$87,$0E,$E4,$8E,$A3,$3F
	.dc.b	$7B,$8D,$5F,$0D,$F4,$09,$1C,$25,$D5,$88,$FE,$6D,$3F,$CF,$08,$FF
	.dc.b	$30,$D3,$DC,$FD,$4C,$3C,$4F,$A8,$F8,$27,$86,$78,$BD,$0F,$E6,$6F
	.dc.b	$F4,$8F,$86,$91,$FC,$D9,$AF,$E5,$67,$AF,$07,$DF,$8E,$C7,$BF,$6F
	.dc.b	$44,$7D,$FE,$CF,$F2,$E9,$8D,$BF,$A6,$7C,$37,$F5,$62,$5F,$F4,$47
	.dc.b	$E6,$3F,$2B,$3F,$69,$EE,$D7,$E5,$C7,$83,$67,$C3,$7E,$68,$91,$F2
	.dc.b	$FB,$EB,$FD,$7E,$3F,$B2,$91,$F9,$5A,$FB,$32,$FC,$54,$7D,$3D,$78
	.dc.b	$BF,$D9,$6B,$83,$CC,$FE,$B7,$F0,$DF,$DC,$08,$E0,$EA,$7A,$EF,$B3
	.dc.b	$F5,$0F,$54,$FF,$A0,$BC,$19,$1F,$87,$8F,$DA,$FC,$37,$F7,$C2,$47
	.dc.b	$F6,$A7,$E3,$BF,$4B,$5F,$8F,$7F,$C1,$23,$F2,$1F,$0D,$FB,$E2,$3F
	.dc.b	$8F,$F1,$2B,$FE,$03,$8F,$F0,$49,$1F,$49,$D6,$7D,$2F,$DF,$3A,$9F
	.dc.b	$DF,$23,$83,$CC,$FE,$FF,$F0,$DF,$E1,$42,$38,$3A,$F1,$FE,$2C,$9B
	.dc.b	$7F,$BF,$F8,$7E,$FC,$7F,$A5,$F8,$58,$7F,$AA,$3C,$69,$87,$88,$FC
	.dc.b	$21,$FE,$22,$21,$FE,$10,$3E,$3D,$76,$47,$EA,$4F,$C0,$1F,$5C,$87
	.dc.b	$C1,$3F,$6D,$E3,$FE,$B9,$FF,$6D,$6B,$FC,$8D,$BA,$7D,$46,$9F,$28
	.dc.b	$A7,$EA,$CC,$1F,$90,$30,$43,$AF,$C3,$7F,$1F,$23,$FC,$05,$F5,$E9
	.dc.b	$F1,$FC,$AF,$F6,$59,$F5,$AF,$F5,$1E,$5F,$FB,$38,$C7,$C3,$7F,$29
	.dc.b	$26,$1F,$C2,$DE,$6F,$FB,$1A,$B9,$BF,$0B,$3F,$EA,$7F,$0D,$FC,$B8
	.dc.b	$91,$FC,$67,$CF,$AF,$F5,$B8,$FE,$6C,$47,$EF,$E5,$E7,$7F,$90,$5F
	.dc.b	$9E,$75,$1C,$4F,$E3,$CF,$F3,$FF,$76,$47,$F2,$16,$BF,$23,$0F,$E9
	.dc.b	$D5,$E8,$8F,$B0,$7E,$96,$BF,$DD,$FE,$94,$8F,$E1,$D1,$FD,$23,$E1
	.dc.b	$BF,$A6,$12,$3F,$97,$3F,$11,$FE,$79,$5D,$6F,$F1,$51,$FD,$93,$E1
	.dc.b	$BF,$A8,$91,$FE,$79,$EA,$D7,$FB,$F4,$7F,$59,$23,$F1,$0F,$A0,$7C
	.dc.b	$43,$E8,$67,$FD,$AE,$37,$1C,$8F,$E8,$4F,$F6,$3F,$86,$91,$FD,$15
	.dc.b	$AF,$C5,$9F,$A0,$FF,$38,$E7,$63,$F8,$7B,$FF,$03,$43,$D7,$A7,$A7
	.dc.b	$2F,$F0,$9F,$C3,$7F,$71,$26,$1F,$D4,$DF,$8D,$5D,$23,$D4,$7D,$29
	.dc.b	$FF,$0D,$FC,$37,$F7,$62,$47,$F6,$1F,$6A,$BF,$E2,$38,$FE,$FC,$47
	.dc.b	$E3,$1F,$48,$EB,$3F,$4B,$95,$3B,$9E,$47,$F6,$77,$FC,$11,$F0,$D2
	.dc.b	$3F,$B5,$35,$F8,$C7,$D2,$3E,$36,$9F,$AB,$9F,$8B,$EE,$49,$FE,$AD
	.dc.b	$1F,$E1,$0F,$86,$FF,$0A,$12,$3F,$BB,$3F,$12,$BF,$11,$BF,$EF,$F2
	.dc.b	$FF,$1E,$FC,$37,$F8,$60,$97,$FE,$03,$F7,$AB,$FE,$45,$8F,$F1,$01
	.dc.b	$1F,$87,$B7,$CF,$7C,$E7,$E2,$0F,$E3,$BA,$A7,$84,$73,$3F,$C1,$4F
	.dc.b	$F8,$AF,$E1,$A4,$7F,$83,$5A,$FC,$43,$E8,$1D,$8F,$B9,$8F,$F8,$FE
	.dc.b	$7F,$13,$F0,$12,$7F,$BD,$47,$F8,$D3,$0F,$F1,$C1,$23,$FC,$34,$FF
	.dc.b	$93,$7E,$19,$D3,$FE,$57,$6C,$FF,$8B,$1A,$FF,$84,$3E,$F1,$F0,$7F
	.dc.b	$E4,$09,$93,$FE,$09,$7F,$CC,$12,$F8,$4C,$CF,$F9,$BE,$7F,$C9,$44
	.dc.b	$7F,$DB,$6F,$6D,$DA,$7F,$46,$94,$E6,$30,$D8,$FE,$97,$7E,$47,$C4
	.dc.b	$7F,$8D,$E3,$B3,$E1,$3F,$C7,$5F,$0D,$FE,$5B,$23,$83,$C7,$97,$C3
	.dc.b	$23,$FE,$4E,$AE,$CF,$F8,$06,$BD,$FF,$0D,$23,$C2,$7F,$81,$7E,$1F
	.dc.b	$FC,$D0,$47,$FC,$94,$FD,$1B,$EB,$1F,$E3,$CA,$F6,$2B,$EF,$97,$FA
	.dc.b	$53,$0F,$F3,$79,$27,$C9,$85,$7F,$D3,$91,$FE,$77,$23,$EC,$3E,$E6
	.dc.b	$1E,$0A,$F9,$F1,$F2,$B4,$E4,$FF,$6E,$AE,$2F,$43,$FC,$C5,$A7,$FA
	.dc.b	$0C,$8E,$0E,$C7,$AE,$FB,$69,$F6,$F2,$F5,$BF,$E9,$7B,$FC,$3E,$92
	.dc.b	$FF,$CA,$51,$FE,$8C,$C7,$FD,$20,$48,$FF,$39,$3F,$25,$F5,$90,$FC
	.dc.b	$B3,$F9,$BD,$E4,$7E,$6D,$FC,$AE,$B8,$8F,$C7,$1F,$E1,$45,$1F,$A5
	.dc.b	$24,$7B,$34,$F8,$EF,$E5,$33,$F6,$BF,$24,$EC,$7E,$72,$78,$BD,$8F
	.dc.b	$CD,$BF,$B1,$E2,$91,$FA,$26,$BB,$BF,$C1,$92,$BF,$5B,$87,$A7,$0F
	.dc.b	$1E,$9F,$EA,$1B,$FC,$B6,$F9,$D7,$F4,$FC,$DF,$BF,$24,$FE,$C1,$F9
	.dc.b	$4F,$D7,$3E,$55,$F4,$C3,$F8,$27,$44,$9F,$E0,$8F,$EF,$B8,$F1,$1E
	.dc.b	$78,$FE,$18,$47,$E4,$61,$FB,$8C,$FE,$3C,$7E,$76,$F8,$BF,$4B,$45
	.dc.b	$F1,$8E,$87,$F0,$6E,$AF,$E3,$04,$70,$76,$3E,$D5,$F6,$D7,$F1,$F1
	.dc.b	$F7,$6B,$F8,$AF,$24,$8F,$59,$FC,$7F,$B7,$F9,$21,$1F,$E2,$AF,$C6
	.dc.b	$D7,$F0,$39,$7D,$39,$FE,$09,$9F,$B2,$31,$1F,$C9,$FA,$3F,$7A,$FB
	.dc.b	$73,$F9,$ED,$3F,$4E,$53,$F0,$0E,$9F,$90,$68,$A6,$8F,$E5,$3E,$0F
	.dc.b	$F3,$B2,$38,$3E,$21,$FC,$2F,$AF,$F9,$A9,$E7,$9F,$E6,$FD,$72,$7F
	.dc.b	$03,$C1,$FC,$BB,$C3,$FE,$8C,$48,$FE,$6E,$FC,$38,$FD,$E5,$F4,$9E
	.dc.b	$27,$EA,$FB,$3F,$A6,$11,$FE,$55,$DB,$5F,$CF,$47,$F5,$32,$3E,$8E
	.dc.b	$25,$75,$8F,$CB,$1F,$C9,$E7,$6F,$23,$FA,$5B,$FD,$77,$C6,$91,$FD
	.dc.b	$39,$AF,$C9,$3F,$AE,$4F,$5E,$0F,$BB,$A7,$95,$FD,$6F,$81,$FD,$70
	.dc.b	$C6,$7F,$D4,$BC,$7F,$ED,$44,$C3,$FA,$DB,$F1,$5F,$EA,$B8,$75,$9F
	.dc.b	$B3,$7F,$8A,$F2,$FF,$B8,$12,$BF,$A7,$78,$55,$FE,$05,$1F,$DD,$C8
	.dc.b	$FC,$5B,$E8,$FE,$80,$FE,$F1,$1A,$78,$9F,$DB,$9F,$F0,$07,$9D,$23
	.dc.b	$FB,$8B,$5F,$89,$7F,$C3,$2B,$AC,$7F,$1F,$3A,$9F,$D8,$FC,$4F,$EF
	.dc.b	$E7,$9E,$7F,$4B,$E8,$FF,$83,$49,$3F,$DF,$9F,$88,$FE,$4C,$FE,$D9
	.dc.b	$5F,$A9,$F4,$FF,$C2,$44,$8F,$EE,$1E,$45,$7F,$92,$47,$F8,$64,$8F
	.dc.b	$C3,$39,$9F,$92,$3F,$C1,$2E,$9E,$27,$F8,$41,$FF,$11,$7A,$D2,$3F
	.dc.b	$C2,$6D,$7E,$1E,$BF,$9E,$7A,$9F,$92,$C7,$F8,$1D,$7F,$88,$3C,$99
	.dc.b	$3F,$E2,$98,$FF,$16,$7B,$1F,$E2,$F2,$47,$F8,$81,$F8,$7C,$BF,$9E
	.dc.b	$8E,$B7,$F9,$27,$FC,$65,$ED,$FF,$8D,$48,$FF,$84,$FC,$CA,$FF,$47
	.dc.b	$8F,$F1,$E1,$1F,$88,$7E,$78,$F8,$9D,$A8,$D4,$71,$3F,$C6,$6F,$F9
	.dc.b	$17,$D5,$91,$FE,$36,$6B,$F1,$0F,$CE,$9E,$8D,$3F,$3A,$7F,$8D,$A7
	.dc.b	$FC,$87,$E7,$49,$F5,$9F,$E4,$D9,$FF,$29,$11,$FF,$21,$B5,$FE,$C4
	.dc.b	$D7,$FA,$33,$5F,$F2,$1B,$5F,$EF,$4F,$F6,$F4,$3E,$3D,$F0,$D9,$8C
	.dc.b	$36,$33,$1C,$5A,$FF,$79,$6B,$FE,$08,$6B,$FE,$57,$6B,$E0,$6B,$D8
	.dc.b	$E9,$1D,$C8,$F7,$F8,$6F,$11,$1C,$10,$F1,$39,$69,$93,$C8,$E5,$A7
	.dc.b	$AF,$E1,$BD,$C4,$70,$43,$DC,$E5,$A6,$4F,$80,$E5,$A7,$C7,$F8,$6F
	.dc.b	$94,$47,$04,$3E,$53,$96,$99,$3E,$63,$96,$9F,$4F,$E1,$BE,$B1,$1C
	.dc.b	$10,$FA,$CE,$5A,$64,$FB,$0E,$5A,$7D,$FF,$86,$FC,$11,$1C,$1F,$8B
	.dc.b	$E1,$7A,$DF,$E9,$5E,$2F,$F2,$1E,$8F,$6B,$1B,$7E,$0D,$FF,$51,$D6
	.dc.b	$30,$FC,$27,$C3,$57,$F2,$8D,$B3,$9F,$BD,$FF,$2A,$E3,$DA,$3F,$2C
	.dc.b	$4A,$FC,$B3,$F9,$9C,$39,$12,$3F,$D3,$18,$75,$9E,$11,$F9,$B6,$BF
	.dc.b	$9B,$F8,$7F,$D1,$11,$FD,$13,$D5,$FD,$5D,$FC,$0B,$ED,$1F,$EC,$5B
	.dc.b	$EC,$EB,$2F,$F5,$D6,$2F,$77,$D8,$6A,$58,$7C,$AF,$87,$FD,$91,$23
	.dc.b	$F6,$4F,$85,$DD,$FC,$D7,$B1,$FA,$FC,$7E,$7B,$83,$F8,$07,$C3,$FC
	.dc.b	$D2,$39,$C5,$EF,$62,$3F,$7D,$3F,$C0,$C8,$FF,$03,$AF,$7E,$3F,$06
	.dc.b	$3E,$69,$DF,$65,$70,$8E,$67,$F0,$B7,$F8,$77,$C3,$C8,$FE,$1C,$D7
	.dc.b	$F8,$75,$F6,$CB,$DD,$1F,$34,$ED,$5F,$85,$99,$18,$3C,$5F,$0F,$FC
	.dc.b	$6C,$8F,$F1,$B7,$E5,$63,$E9,$8F,$96,$7C,$93,$D6,$60,$FE,$51,$F0
	.dc.b	$FF,$95,$23,$F2,$7E,$12,$BF,$C9,$63,$F9,$51,$1F,$90,$7C,$5B,$F6
	.dc.b	$3F,$EF,$8E,$0E,$2F,$F2,$B9,$E0,$F3,$3F,$99,$7C,$3F,$F3,$52,$38
	.dc.b	$3A,$9E,$7D,$7F,$64,$F7,$3D,$78,$FE,$DA,$BF,$59,$F0,$B2,$7E,$C4
	.dc.b	$7E,$C7,$E1,$FF,$A1,$12,$3F,$A1,$3F,$23,$6F,$A7,$7F,$24,$FF,$58
	.dc.b	$E5,$FB,$9F,$87,$97,$F8,$37,$F7,$46,$1F,$E9,$4F,$F4,$F2,$3E,$A9
	.dc.b	$E9,$5D,$8F,$F7,$45,$71,$9F,$EA,$15,$C1,$E6,$7F,$57,$C7,$FA,$D1
	.dc.b	$1C,$1D,$6B,$A3,$F7,$CF,$C6,$1F,$26,$7F,$88,$61,$23,$D4,$7E,$77
	.dc.b	$5F,$EC,$C4,$7F,$B3,$3F,$15,$E9,$8F,$A9,$FB,$C7,$D3,$3E,$B6,$BF
	.dc.b	$82,$5C,$4D,$BF,$85,$C7,$F6,$D9,$FE,$00,$D3,$EF,$94,$FD,$39,$83
	.dc.b	$F4,$E6,$48,$60,$FE,$ED,$C5,$FD,$FC,$8F,$F7,$F7,$D4,$BE,$45,$EA
	.dc.b	$7F,$37,$6C,$7A,$F1,$F7,$3E,$AE,$0F,$E7,$07,$B7,$87,$F9,$04,$7F
	.dc.b	$82,$F9,$BF,$C1,$C4,$7E,$A1,$FE,$16,$BE,$AF,$EC,$5B,$7D,$AB,$FA
	.dc.b	$1F,$07,$F8,$3E,$BF,$A4,$1F,$D1,$F8,$EB,$FE,$19,$3F,$C3,$84,$3B
	.dc.b	$9E,$DC,$BD,$87,$9D,$D1,$FE,$1F,$29,$FE,$2A,$3C,$DC,$7F,$E2,$9C
	.dc.b	$7F,$B1,$C7,$F8,$BD,$AE,$CD,$9E,$07,$D7,$7C,$DB,$1F,$88,$8F,$63
	.dc.b	$FC,$D7,$97,$FB,$11,$EC,$7E,$27,$81,$E1,$23,$E1,$7E,$CD,$74,$8F
	.dc.b	$71,$F5,$EB,$C1,$E2,$7D,$B2,$47,$F4,$3E,$7A,$FA,$4F,$C1,$47,$BC
	.dc.b	$8F,$AF,$2F,$92,$F5,$3E,$D4,$69,$E5,$68,$C4,$7C,$58,$78,$FF,$2C
	.dc.b	$91,$F2,$DF,$9D,$5D,$A3,$F0,$8F,$EA,$67,$A4,$7F,$5B,$E9,$FF,$04
	.dc.b	$1E,$AD,$D9,$7C,$FF,$27,$EC,$12,$BE,$C3,$D9,$FE,$D1,$7F,$24,$F9
	.dc.b	$5C,$1F,$CB,$23,$F2,$9E,$67,$E8,$C8,$FF,$6B,$EC,$AF,$C0,$8F,$C3
	.dc.b	$11,$FC,$36,$7E,$BC,$BC,$B1,$FA,$18,$F6,$4E,$E3,$91,$F8,$D7,$F2
	.dc.b	$1E,$7C,$8F,$C8,$35,$F9,$F3,$D7,$3F,$C7,$3D,$23,$FC,$01,$DD,$FE
	.dc.b	$2D,$3C,$F8,$78,$A3,$F8,$07,$A3,$F9,$D2,$3F,$9D,$7E,$33,$F4,$8F
	.dc.b	$8F,$5F,$4C,$F0,$3E,$B3,$FC,$1D,$E0,$FD,$97,$F0,$11,$C9,$A7,$C4
	.dc.b	$21,$EA,$29,$A3,$04,$30,$7E,$2F,$D6,$FD,$A1,$1F,$DA,$3E,$8B,$F7
	.dc.b	$76,$BC,$3F,$F2,$71,$E8,$AF,$D3,$9F,$E0,$18,$78,$9F,$E5,$33,$5C
	.dc.b	$3F,$ED,$79,$FC,$EF,$9D,$FC,$0C,$8F,$F0,$3F,$63,$F9,$41,$1F,$F1
	.dc.b	$87,$8F,$5F,$E3,$D1,$72,$F2,$F9,$19,$9F,$D3,$4F,$F0,$F2,$3D,$48
	.dc.b	$7E,$B9,$CB,$4C,$9C,$8A,$7F,$11,$F6,$FF,$8D,$11,$FE,$34,$F2,$AF
	.dc.b	$C2,$7C,$93,$E0,$3F,$CF,$0D,$AF,$D3,$DF,$A7,$93,$FC,$BF,$E6,$7F
	.dc.b	$9E,$4C,$4F,$F3,$DF,$80,$FE,$50,$47,$F9,$43,$DE,$7F,$97,$5F,$59
	.dc.b	$FF,$19,$37,$1F,$CB,$6C,$C7,$FA,$A7,$C0,$CB,$FD,$CB,$FE,$8D,$FB
	.dc.b	$EF,$CE,$5F,$08,$FE,$70,$49,$FE,$70,$F8,$5F,$F7,$3E,$1D,$F1,$F1
	.dc.b	$E9,$EA,$9F,$EC,$95,$C5,$FE,$3A,$D7,$F8,$E7,$A5,$FD,$14,$8F,$F4
	.dc.b	$57,$B2,$BE,$CF,$82,$BB,$66,$BF,$57,$A5,$2B,$DA,$7F,$16,$F8,$6F
	.dc.b	$EA,$44,$7F,$A9,$3D,$B0,$EB,$1D,$F0,$EB,$F1,$1F,$DE,$C9,$1F,$E8
	.dc.b	$7D,$5E,$A6,$23,$F9,$CC,$FF,$5F,$23,$FD,$14,$E8,$7F,$A3,$EF,$B9
	.dc.b	$E0,$CF,$EB,$47,$F2,$C9,$E2,$F4,$3F,$B4,$3F,$DA,$FE,$1E,$47,$F6
	.dc.b	$B6,$BF,$D4,$EB,$AE,$3F,$92,$3D,$71,$E9,$F5,$BF,$00,$7A,$78,$3F
	.dc.b	$0F,$F1,$1F,$DD,$C9,$3F,$DD,$DF,$53,$EA,$3F,$1F,$5D,$5F,$CE,$9F
	.dc.b	$D1,$3E,$0F,$FC,$44,$47,$E4,$7B,$55,$FE,$BF,$1F,$E0,$52,$3F,$1F
	.dc.b	$1F,$4B,$F1,$EB,$EF,$CE,$A3,$89,$FE,$0B,$7C,$B2,$F8,$8A,$49,$E8
	.dc.b	$EA,$BD,$11,$EA,$3D,$27,$F8,$AE,$BF,$49,$ED,$C8,$F6,$1F,$67,$E2
	.dc.b	$3B,$91,$F4,$3D,$5F,$41,$0F,$E3,$25,$CF,$F8,$3D,$F8,$9F,$E9,$9C
	.dc.b	$2F,$77,$CB,$5F,$E2,$5C,$FF,$7C,$DB,$EC,$FD,$F6,$B9,$3E,$53,$87
	.dc.b	$F6,$A6,$08,$7E,$1F,$E1,$FE,$49,$1F,$A4,$F6,$C3,$F8,$EF,$C2,$FF
	.dc.b	$1C,$3D,$1D,$3D,$2B,$FD,$85,$93,$E0,$3F,$86,$1A,$8F,$E0,$FF,$11
	.dc.b	$F5,$88,$FD,$F7,$B5,$7F,$1C,$BE,$D3,$F1,$E3,$F8,$FF,$C4,$7F,$2C
	.dc.b	$24,$7E,$CF,$E0,$EB,$FA,$18,$FC,$19,$1F,$7E,$5E,$A7,$F6,$57,$E5
	.dc.b	$7F,$87,$4E,$DE,$47,$E4,$DF,$C6,$FC,$2C,$8F,$CB,$35,$FD,$84,$FA
	.dc.b	$1F,$E6,$6F,$5D,$95,$FC,$BF,$E1,$65,$7F,$03,$8F,$E8,$FF,$11,$F9
	.dc.b	$A2,$47,$E8,$DF,$CE,$DF,$5C,$7D,$53,$F9,$93,$B9,$EC,$21,$F5,$3E
	.dc.b	$1F,$C8,$FE,$2E,$79,$BF,$AF,$6B,$EA,$30,$43,$04,$39,$19,$3F,$2D
	.dc.b	$F1,$1F,$B2,$23,$FB,$B7,$AD,$7F,$66,$E3,$FE,$CA,$7A,$F6,$FC,$05
	.dc.b	$7F,$2D,$AF,$EA,$86,$5F,$E9,$F8,$7E,$F8,$8F,$F0,$77,$AC,$FF,$64
	.dc.b	$AE,$B3,$F0,$AF,$FB,$AE,$7F,$DD,$09,$1F,$CE,$F1,$AF,$F2,$98,$FE
	.dc.b	$16,$47,$F8,$A5,$74,$7F,$B7,$BE,$57,$FA,$B4,$6D,$E4,$7F,$1B,$7F
	.dc.b	$8A,$ED,$23,$F8,$F3,$5F,$E7,$13,$D6,$FE,$34,$7A,$70,$FE,$FF,$7F
	.dc.b	$DD,$F3,$93,$FD,$2E,$3F,$C1,$FC,$1F,$C8,$89,$1F,$CA,$DF,$E4,$B8
	.dc.b	$75,$BF,$8D,$3D,$4F,$EF,$71,$96,$9F,$D7,$F7,$F8,$0F,$F1,$38,$E6
	.dc.b	$FC,$42,$1E,$A2,$9B,$29,$E0,$34,$43,$F8,$F7,$2F,$F3,$A2,$3F,$D0
	.dc.b	$DE,$EF,$F0,$ED,$FF,$C3,$07,$9D,$FF,$03,$4F,$F8,$6F,$86,$47,$F6
	.dc.b	$F8,$FF,$19,$F4,$7F,$46,$24,$7F,$4F,$6B,$FD,$A5,$AF,$F3,$16,$BF
	.dc.b	$E3,$06,$BF,$E2,$27,$FA,$8F,$54,$8F,$EB,$4D,$7F,$BA,$35,$FE,$78
	.dc.b	$D7,$FC,$76,$D7,$FC,$5A,$D7,$FC,$45,$DD,$FD,$78,$8F,$F6,$76,$BF
	.dc.b	$E0,$16,$BF,$D2,$5A,$FF,$92,$1A,$FF,$8E,$9F,$ED,$3E,$04,$8F,$EE
	.dc.b	$4D,$7F,$C1,$8D,$7F,$AA,$35,$FF,$29,$B5,$FF,$22,$B5,$FF,$1D,$69
	.dc.b	$FD,$D8,$8F,$F7,$F7,$FC,$1B,$CC,$E7,$FF,$2A,$74,$4A,$FF,$18,$47
	.dc.b	$F9,$5F,$A6,$47,$F8,$CD,$FF,$2D,$75,$A2,$72,$FF,$AD,$FE,$1D,$E3
	.dc.b	$7F,$86,$9B,$3F,$E3,$56,$BF,$EF,$B6,$BC,$CE,$99,$79,$5F,$F2,$16
	.dc.b	$5E,$57,$FB,$0E,$5E,$47,$FB,$1E,$57,$1F,$EF,$B9,$73,$7C,$9E,$47
	.dc.b	$8C,$95,$E3,$6B,$E3,$6B,$E3,$19,$23,$31,$E7,$8C,$3C,$C8,$7B,$7C
	.dc.b	$9F,$84,$47,$E1,$35,$D1,$82,$18,$21,$E3,$84,$71,$8C,$79,$7F,$3C
	.dc.b	$8F,$CF,$7E,$97,$9B,$F0,$F7,$F4,$36,$7E,$D5,$FF,$B4,$5F,$B7,$3F
	.dc.b	$4F,$91,$E0,$EE,$7E,$1D,$DB,$E1,$0C,$FF,$05,$3B,$B7,$85,$B8,$F1
	.dc.b	$D7,$E0,$DA,$FE,$0D,$AF,$BD,$AF,$A0,$E0,$43,$81,$F8,$DF,$43,$F2
	.dc.b	$04,$70,$6E,$D9,$7B,$7C,$4F,$31,$E5,$9F,$82,$7C,$4B,$F5,$78,$B2
	.dc.b	$32,$7C,$22,$E6,$F4,$BF,$40,$48,$FD,$03,$FA,$3F,$57,$E4,$12,$39
	.dc.b	$EA,$F2,$71,$1F,$2F,$3F,$C7,$56,$34,$F4,$CF,$EC,$6D,$E5,$F0,$8F
	.dc.b	$C3,$57,$77,$F6,$0D,$7F,$60,$D7,$66,$88,$68,$A7,$E4,$7D,$8F,$DE
	.dc.b	$11,$FD,$E3,$D6,$FC,$B1,$FE,$6D,$F3,$3F,$04,$7A,$71,$F5,$C7,$C0
	.dc.b	$F3,$BF,$0C,$65,$FC,$9F,$B7,$FC,$24,$8F,$F0,$97,$F8,$67,$BB,$F9
	.dc.b	$12,$47,$CA,$F4,$6B,$F2,$4F,$CE,$C6,$31,$F2,$EB,$FE,$C2,$9D,$1F
	.dc.b	$BE,$9E,$AF,$F1,$76,$BF,$C5,$DA,$FE,$CD,$AF,$F0,$36,$BE,$03,$91
	.dc.b	$FC,$87,$DF,$FE,$48,$47,$06,$8F,$1C,$BD,$3F,$D5,$1E,$9C,$FB,$4F
	.dc.b	$E2,$3D,$5F,$D7,$19,$7F,$77,$F0,$3F,$CC,$48,$FF,$31,$7F,$9A,$7C
	.dc.b	$27,$EE,$49,$1F,$92,$F6,$2B,$F1,$39,$93,$8D,$7E,$4F,$B3,$AA,$FE
	.dc.b	$29,$3D,$1F,$E7,$CD,$7F,$9F,$35,$D9,$A2,$1A,$29,$FC,$7F,$E1,$BF
	.dc.b	$A5,$11,$FE,$94,$F5,$AF,$B7,$ED,$FF,$0F,$3C,$DA,$7F,$35,$BF,$E2
	.dc.b	$5E,$E4,$C3,$F6,$F1,$FC,$67,$E1,$FF,$AB,$12,$3F,$AB,$3F,$D6,$73
	.dc.b	$FE,$3C,$48,$FD,$BF,$BF,$5F,$8D,$17,$2F,$C6,$FC,$06,$67,$F9,$C4
	.dc.b	$F3,$7F,$B1,$35,$FE,$C4,$D7,$66,$88,$68,$A7,$F4,$3F,$87,$FE,$DC
	.dc.b	$47,$FB,$73,$D6,$BF,$3F,$F0,$5F,$CC,$CF,$36,$9F,$D6,$6F,$F9,$B7
	.dc.b	$C1,$CC,$3F,$8D,$C7,$F3,$DF,$87,$FE,$F4,$48,$FE,$F4,$FF,$7D,$F8
	.dc.b	$7F,$E8,$44,$8F,$E3,$7F,$0D,$5F,$C5,$45,$CB,$F8,$2F,$DF,$FF,$6D
	.dc.b	$7F,$C1,$11,$CE,$7F,$C0,$CD,$7F,$C0,$CD,$7F,$B1,$35,$FE,$E4,$D7
	.dc.b	$FA,$DB,$5F,$EF,$3F,$11,$FE,$13,$23,$FE,$13,$7B,$4F,$91,$FE,$BD
	.dc.b	$47,$D1,$9F,$6A,$FE,$83,$2B,$D9,$A8,$FE,$D6,$7F,$88,$48,$7F,$88
	.dc.b	$4F,$F0,$61,$FE,$1F,$F9,$56,$3F,$DB,$5B,$1F,$D3,$9A,$FF,$16,$A6
	.dc.b	$5B,$A9,$3B,$7D,$3B,$62,$3B,$DF,$27,$FC,$38,$D7,$C5,$3E,$47,$FC
	.dc.b	$79,$19,$6C,$65,$46,$5B,$19,$8C,$11,$C1,$9D,$64,$F7,$94,$62,$3C
	.dc.b	$F3,$F4,$B5,$91,$ED,$21,$F0,$F8,$7E,$B9,$EF,$DB,$FA,$8E,$0D,$FC
	.dc.b	$84,$AF,$03,$E7,$E0,$FA,$04,$8F,$BB,$CB,$5F,$E8,$B4,$9C,$5F,$E6
	.dc.b	$B8,$F5,$5F,$32,$7A,$3E,$D6,$BF,$49,$AE,$CD,$10,$D1,$4F,$AF,$C7
	.dc.b	$F3,$48,$FC,$97,$AD,$7E,$B3,$A3,$F3,$27,$9B,$8F,$F1,$D7,$F9,$CE
	.dc.b	$C9,$87,$E3,$63,$F2,$1C,$DF,$64,$91,$F5,$5F,$B9,$CF,$F9,$52,$47
	.dc.b	$E8,$BB,$6B,$F1,$A2,$E5,$FB,$EE,$FC,$CF,$E4,$27,$9B,$F8,$16,$BF
	.dc.b	$96,$6B,$B3,$44,$34,$53,$F3,$7D,$5F,$91,$23,$F8,$D7,$AD,$7F,$10
	.dc.b	$F0,$BF,$7A,$79,$B4,$FD,$95,$FF,$00,$F0,$E6,$1F,$B1,$8F,$DA,$76
	.dc.b	$FE,$74,$91,$F9,$97,$F4,$1D,$DF,$B9,$24,$7F,$06,$F1,$AB,$F8,$A8
	.dc.b	$B9,$7F,$22,$F2,$B3,$3F,$B4,$9E,$6F,$E9,$5A,$FE,$E9,$AE,$CD,$10
	.dc.b	$D1,$4F,$DF,$F8,$3F,$B5,$23,$FB,$17,$AD,$7F,$34,$F2,$7F,$90,$9E
	.dc.b	$6D,$3F,$8B,$5F,$F2,$4F,$36,$61,$FC,$56,$3F,$8C,$7C,$3F,$F0,$12
	.dc.b	$47,$EF,$5F,$E0,$9E,$37,$F1,$D2,$47,$F2,$AF,$3E,$BF,$AE,$8B,$97
	.dc.b	$F4,$6F,$43,$33,$FC,$62,$79,$BF,$C2,$5A,$FF,$1E,$6B,$B3,$44,$34
	.dc.b	$53,$F9,$1F,$93,$FC,$64,$8F,$F1,$57,$AD,$7F,$58,$F5,$BF,$A2,$9E
	.dc.b	$6D,$3F,$9D,$5F,$FA,$97,$F0,$0F,$F3,$0D,$7F,$98,$9B,$3A,$C3,$F9
	.dc.b	$E7,$99,$FC,$9C,$8F,$F2,$37,$F9,$57,$9B,$FD,$0C,$91,$FE,$90,$A7
	.dc.b	$B1,$88,$FE,$2B,$37,$0F,$07,$B3,$99,$FE,$7F,$3D,$1F,$E6,$0D,$7F
	.dc.b	$A2,$B5,$D9,$A2,$1A,$29,$FD,$2B,$D0,$FE,$80,$47,$F9,$E3,$D6,$BE
	.dc.b	$17,$B5,$F1,$4F,$36,$7F,$D8,$6F,$E4,$7B,$73,$0F,$EC,$11,$FD,$8F
	.dc.b	$D3,$FE,$98,$48,$FE,$90,$FF,$4F,$F5,$3F,$B4,$12,$3E,$37,$BD,$5F
	.dc.b	$E7,$31,$72,$FA,$9E,$AE,$67,$FB,$1C,$F3,$7F,$AA,$35,$FE,$D2,$D7
	.dc.b	$66,$88,$68,$A7,$F6,$EF,$5B,$FB,$21,$1F,$EC,$0F,$5A,$FC,$37,$C0
	.dc.b	$7E,$24,$F3,$69,$FE,$05,$BF,$C6,$7C,$1C,$C3,$FC,$09,$1F,$E0,$7F
	.dc.b	$67,$FB,$81,$23,$FB,$63,$FD,$DF,$E5,$DE,$DF,$FB,$E5,$B3,$FE,$DA
	.dc.b	$6B,$FE,$A2,$E5,$7C,$1E,$EA,$4E,$DF,$CF,$7C,$36,$23,$FC,$37,$7C
	.dc.b	$91,$03,$97,$0D,$76,$D7,$73,$E9,$23,$FE,$15,$8D,$12,$33,$1E,$2F
	.dc.b	$7B,$E0,$11,$E0,$EB,$6F,$DA,$7C,$3F,$ED,$CF,$53,$FE,$D8,$BF,$86
	.dc.b	$FE,$E7,$E1,$FF,$78,$6B,$7F,$4F,$C1,$7C,$D2,$63,$E5,$7E,$87,$C1
	.dc.b	$7C,$F2,$47,$EE,$7E,$1E,$BF,$ED,$0C,$53,$8C,$7F,$84,$7C,$3E,$AB
	.dc.b	$E5,$1F,$2A,$7A,$BF,$0D,$AF,$D2,$6B,$C2,$BA,$BB,$21,$B2,$9B,$F8
	.dc.b	$6F,$C4,$11,$FA,$0F,$6B,$FE,$39,$F0,$FF,$C7,$CF,$46,$7F,$73,$0F
	.dc.b	$E4,$55,$2F,$F1,$D1,$F9,$1A,$FC,$A1,$23,$EE,$3F,$97,$C7,$F2,$C4
	.dc.b	$8F,$E4,$13,$5F,$91,$17,$0F,$E5,$D9,$E6,$7E,$04,$F3,$7F,$10,$D7
	.dc.b	$F2,$ED,$76,$68,$86,$8A,$7C,$BC,$FF,$58,$47,$F2,$8F,$5A,$FE,$81
	.dc.b	$B7,$F4,$33,$CD,$9F,$E8,$2F,$FA,$36,$F3,$0F,$D9,$47,$ED,$76,$FD
	.dc.b	$C1,$23,$F4,$0F,$EE,$F7,$FD,$D1,$23,$F5,$F8,$B0,$FD,$87,$E0,$2E
	.dc.b	$4D,$57,$E3,$EE,$DE,$E3,$F6,$F4,$E8,$DC,$7E,$26,$BC,$0F,$EC,$5A
	.dc.b	$FF,$00,$6B,$B3,$44,$34,$53,$F2,$9C,$7F,$C5,$48,$FE,$F5,$EB,$7E
	.dc.b	$58,$5B,$9B,$F8,$64,$E6,$31,$3F,$C6,$EF,$DB,$97,$C2,$EB,$DB,$67
	.dc.b	$47,$F2,$1E,$6F,$E4,$C4,$7F,$87,$3F,$CA,$7A,$3F,$94,$12,$3E,$7C
	.dc.b	$FC,$EE,$FC,$47,$EE,$E6,$E5,$E5,$ED,$CC,$FE,$CE,$7A,$3F,$C6,$DA
	.dc.b	$FF,$2A,$6B,$B3,$44,$34,$53,$F7,$DD,$7F,$CF,$C8,$FF,$27,$7A,$D7
	.dc.b	$C7,$F0,$BE,$1C,$FE,$01,$F3,$E3,$FC,$D2,$F3,$3F,$43,$C5,$BB,$1D
	.dc.b	$C7,$F3,$8E,$EF,$E9,$C4,$7F,$9B,$35,$FE,$8A,$D7,$F8,$A3,$5F,$E8
	.dc.b	$6D,$7A,$1F,$CF,$FB,$FF,$AD,$11,$C1,$0F,$EA,$2E,$5A,$64,$FE,$44
	.dc.b	$E5,$A7,$F4,$BF,$03,$FB,$31,$1C,$10,$FE,$BA,$E5,$A6,$4F,$E5,$CE
	.dc.b	$5A,$7F,$57,$F0,$7F,$B9,$11,$C1,$0F,$ED,$2E,$5A,$64,$FE,$74,$E5
	.dc.b	$A7,$F6,$3F,$0B,$FB,$F1,$1C,$10,$FE,$EA,$E5,$A6,$4F,$E8,$CE,$5A
	.dc.b	$7F,$6F,$F0,$FF,$C1,$44,$70,$43,$FC,$02,$E5,$A6,$4F,$EA,$4E,$5A
	.dc.b	$7F,$7B,$D3,$FC,$2C,$47,$07,$F6,$03,$FC,$17,$F3,$2F,$1B,$F2,$AD
	.dc.b	$8F,$D9,$9A,$8F,$CB,$F8,$DA,$64,$68,$4F,$15,$C3,$FC,$2F,$C9,$DC
	.dc.b	$78,$67,$4F,$F7,$26,$BF,$E1,$D6,$BE,$13,$D0,$76,$AA,$D3,$31,$C1
	.dc.b	$C1,$DE,$37,$19,$27,$91,$82,$4F,$27,$73,$FB,$6F,$36,$4F,$BE,$BD
	.dc.b	$D8,$7F,$2B,$6E,$5F,$D1,$BC,$BF,$DE,$9F,$02,$75,$E5,$F8,$89,$7E
	.dc.b	$87,$CB,$E6,$7D,$22,$47,$CD,$E5,$7A,$18,$8F,$9B,$8F,$C7,$AC,$61
	.dc.b	$FC,$E7,$D4,$D5,$7C,$D9,$EA,$FC,$66,$BF,$4D,$AE,$CD,$10,$D1,$4F
	.dc.b	$B1,$E7,$FC,$C2,$3F,$49,$EB,$5F,$D2,$FD,$2F,$E3,$E7,$9B,$0F,$45
	.dc.b	$7B,$7D,$7F,$E4,$66,$5F,$CA,$FA,$5F,$60,$8F,$DE,$7E,$DF,$A7,$F9
	.dc.b	$82,$47,$C7,$F6,$EB,$F1,$8F,$CF,$C6,$31,$FE,$BD,$EC,$EA,$7F,$25
	.dc.b	$3D,$1F,$C5,$B5,$FC,$C3,$5D,$9A,$21,$A2,$9F,$9D,$F5,$BF,$22,$47
	.dc.b	$F2,$EF,$5A,$FE,$E1,$EF,$FF,$44,$3C,$D9,$FE,$8E,$FF,$A3,$FC,$1C
	.dc.b	$C3,$F2,$51,$FB,$8F,$67,$F3,$A4,$8F,$D1,$BF,$A0,$F6,$BF,$7A,$48
	.dc.b	$FC,$67,$C1,$D7,$E3,$1F,$BA,$8C,$65,$FE,$07,$F8,$1F,$F3,$3B,$F9
	.dc.b	$F8,$E9,$3F,$B0,$6B,$FB,$D6,$BF,$BD,$6B,$FB,$D6,$BF,$C4,$9A,$FF
	.dc.b	$0A,$F7,$7F,$6E,$47,$F7,$AF,$69,$F2,$7C,$47,$F6,$CB,$3E,$8D,$3B
	.dc.b	$5F,$F1,$59,$7C,$46,$CD,$47,$F2,$2F,$80,$FE,$0A,$47,$F8,$5B,$FC
	.dc.b	$23,$E0,$7F,$92,$92,$3A,$7C,$3D,$7E,$4F,$7A,$71,$9F,$C9,$F8,$7D
	.dc.b	$57,$F1,$F9,$E8,$FF,$18,$6B,$FC,$95,$AE,$CD,$10,$D1,$4F,$E5,$5F
	.dc.b	$09,$FC,$78,$8F,$F2,$47,$AD,$7D,$BF,$87,$F9,$07,$9B,$5F,$E6,$17
	.dc.b	$F2,$FE,$1E,$61,$FC,$7E,$3F,$A2,$ED,$FC,$A8,$91,$FC,$C1,$FE,$5B
	.dc.b	$3F,$D2,$89,$1F,$46,$2B,$F1,$A2,$E1,$F8,$DC,$73,$3F,$D0,$E7,$9B
	.dc.b	$FC,$F1,$AF,$F4,$96,$BB,$34,$43,$45,$3F,$A7,$63,$FD,$08,$8F,$F4
	.dc.b	$87,$AD,$7E,$7F,$0F,$C6,$1E,$6C,$7F,$AA,$5F,$E3,$F5,$98,$7F,$43
	.dc.b	$8F,$ED,$39,$FF,$4E,$24,$7F,$54,$7F,$A8,$EB,$FD,$B8,$91,$F9,$5E
	.dc.b	$0A,$FE,$2A,$2E,$5F,$B1,$E1,$CC,$FF,$67,$9E,$6F,$F6,$06,$BF,$DB
	.dc.b	$5A,$EC,$D1,$0D,$14,$FE,$E5,$C1,$FD,$98,$8F,$F6,$C7,$AD,$7F,$03
	.dc.b	$E3,$FD,$81,$E6,$D3,$FB,$C5,$FE,$CF,$9E,$61,$FD,$9E,$3F,$C1,$BC
	.dc.b	$7F,$DC,$89,$1F,$DE,$1F,$EE,$AF,$F8,$48,$87,$EE,$79,$6B,$FA,$E2
	.dc.b	$E3,$FC,$57,$AB,$31,$FE,$0B,$39,$9F,$E0,$42,$9F,$E1,$12,$9B,$34
	.dc.b	$43,$45,$04,$1E,$63,$9F,$FC,$26,$D8,$FF,$0B,$B5,$FF,$18,$35,$F0
	.dc.b	$1F,$CC,$39,$9D,$DF,$C6,$CF,$6E,$BF,$E2,$39,$FE,$3D,$DF,$1F,$F0
	.dc.b	$91,$FE,$8C,$B7,$57,$88,$91,$E2,$7C,$BD,$7B,$24,$7F,$25,$F0,$EB
	.dc.b	$FD,$0A,$93,$8D,$7F,$A1,$78,$FA,$AF,$84,$7C,$29,$EA,$FC,$06,$BF
	.dc.b	$01,$AF,$0A,$EA,$EC,$86,$CA,$6F,$BB,$E6,$91,$F9,$AF,$6B,$FE,$AD
	.dc.b	$E3,$FF,$43,$3D,$1A,$7D,$7B,$EF,$7F,$D7,$BF,$02,$FF,$71,$5F,$FB
	.dc.b	$8D,$B5,$BC,$7E,$67,$83,$F7,$09,$7F,$71,$FB,$FE,$17,$D1,$24,$7F
	.dc.b	$98,$ED,$E5,$E2,$3E,$64,$DC,$7C,$1E,$66,$67,$E2,$4F,$47,$F1,$2D
	.dc.b	$7F,$12,$D7,$66,$88,$68,$A7,$CF,$F1,$7F,$2A,$47,$F2,$AF,$5A,$F8
	.dc.b	$5E,$6F,$FB,$5B,$13,$E7,$D3,$E5,$61,$9D,$66,$3F,$E8,$15,$1F,$93
	.dc.b	$F1,$FF,$46,$49,$FD,$1B,$FA,$6F,$23,$F3,$04,$8F,$91,$E7,$57,$F2
	.dc.b	$51,$73,$FA,$DE,$7E,$67,$F1,$53,$CD,$FD,$7B,$5F,$D7,$B5,$D9,$A2
	.dc.b	$1A,$29,$F9,$5F,$33,$F7,$64,$7F,$76,$F5,$AF,$C4,$7A,$7F,$1C,$F3
	.dc.b	$69,$F8,$FB,$F9,$5E,$94,$C3,$E3,$47,$ED,$FC,$FF,$E0,$E4,$8F,$E0
	.dc.b	$EF,$F0,$AF,$43,$F7,$84,$8F,$C6,$7A,$D5,$FD,$B4,$5C,$BF,$35,$ED
	.dc.b	$66,$7F,$5D,$3C,$DF,$E2,$6D,$7F,$89,$B5,$D9,$A2,$1A,$29,$FB,$9F
	.dc.b	$4F,$F8,$F9,$1F,$E3,$EF,$5A,$FD,$67,$B3,$F8,$B3,$CD,$A7,$EC,$EF
	.dc.b	$F1,$DE,$EC,$C3,$F1,$51,$FC,$6F,$D6,$FE,$56,$48,$FE,$56,$FF,$2E
	.dc.b	$F5,$FF,$90,$12,$3F,$61,$F0,$35,$FE,$35,$17,$2F,$DF,$7B,$F9,$9F
	.dc.b	$E2,$53,$CD,$FE,$6E,$D7,$F9,$BB,$5D,$9A,$21,$A2,$9F,$C7,$7D,$AF
	.dc.b	$E8,$64,$7F,$A1,$BD,$6B,$F8,$87,$C2,$7E,$BC,$F3,$69,$FC,$5E,$FF
	.dc.b	$65,$F0,$B3,$0F,$D7,$47,$F3,$FF,$77,$FA,$79,$23,$FA,$7B,$FD,$4B
	.dc.b	$DE,$FE,$88,$48,$FE,$29,$F0,$B5,$FE,$7D,$17,$2F,$E4,$5F,$0B,$99
	.dc.b	$FE,$6D,$3C,$DF,$EB,$6D,$7F,$AD,$B5,$D9,$A2,$1A,$29,$FD,$07,$E0
	.dc.b	$7F,$B3,$91,$FE,$CE,$F5,$AF,$E6,$9F,$0B,$FC,$4C,$F3,$69,$FC,$EE
	.dc.b	$FF,$8B,$7C,$2C,$C3,$F8,$94,$7F,$63,$F8,$4F,$EE,$64,$8F,$EE,$6F
	.dc.b	$F7,$6F,$85,$FE,$D0,$48,$FE,$FB,$8B,$0F,$EF,$DF,$81,$AB,$55,$FD
	.dc.b	$95,$B5,$B8,$FF,$02,$CA,$F7,$1F,$D7,$EB,$C0,$FF,$81,$1A,$FF,$81
	.dc.b	$1A,$EC,$D1,$0D,$14,$FE,$E3,$F3,$8C,$BF,$B9,$C7,$F8,$56,$30,$E0
	.dc.b	$38,$76,$D7,$C0,$7F,$B0,$AD,$A7,$F8,$7F,$1F,$74,$A7,$2F,$F4,$1C
	.dc.b	$3E,$1E,$5F,$23,$3E,$0D,$9D,$1F,$39,$F3,$69,$E5,$8F,$2B,$72,$34
	.dc.b	$C6,$BD,$27,$FC,$67,$28,$C4,$7B,$9B,$8B,$7C,$D6,$EB,$C4,$72,$AC
	.dc.b	$9C,$38,$EC,$7C,$99,$E0,$FC,$89,$F1,$B9,$6B,$93,$04,$30,$43,$EC
	.dc.b	$BE,$FE,$09,$1D,$DA,$F0,$D7,$E8,$E2,$E1,$F6,$9E,$69,$FF,$7C,$E1
	.dc.b	$E6,$AD,$F3,$7C,$23,$4F,$DA,$87,$1F,$CF,$25,$7B,$DF,$A5,$C9,$F2
	.dc.b	$09,$1F,$20,$FC,$17,$56,$1F,$0C,$5B,$F3,$F4,$66,$7F,$2F,$3D,$1F
	.dc.b	$98,$D7,$F2,$6D,$76,$68,$86,$8A,$7E,$8B,$9F,$F1,$04,$7E,$C3,$D6
	.dc.b	$BE,$4F,$6F,$E1,$0F,$37,$1F,$E9,$EF,$F0,$FE,$0C,$C3,$DD,$A2,$7F
	.dc.b	$13,$D5,$F9,$52,$4F,$E1,$5F,$CC,$75,$FE,$38,$91,$F1,$F8,$5D,$F8
	.dc.b	$8F,$AF,$37,$4F,$BB,$E2,$66,$7F,$7B,$3D,$1F,$C9,$35,$FD,$CB,$5D
	.dc.b	$9A,$21,$A2,$9F,$C2,$3B,$BF,$5C,$47,$F3,$CF,$5A,$FC,$7F,$93,$FA
	.dc.b	$A3,$CD,$B7,$F0,$DB,$FD,$6F,$8D,$30,$F8,$F1,$FA,$EF,$07,$F7,$44
	.dc.b	$8F,$D5,$35,$FE,$22,$D7,$F8,$DB,$5F,$E3,$4D,$7F,$7E,$FF,$05,$F0
	.dc.b	$A4,$7E,$D5,$AF,$F1,$C6,$BF,$C9,$9A,$FF,$25,$6B,$FC,$29,$AF,$F2
	.dc.b	$6F,$0F,$F8,$91,$1F,$E0,$4D,$7F,$94,$B5,$FE,$66,$D7,$F9,$93,$5F
	.dc.b	$E2,$EF,$F1,$DF,$12,$47,$F0,$D6,$BF,$CD,$3C,$5F,$E2,$E4,$7F,$15
	.dc.b	$E6,$57,$F7,$F1,$75,$FE,$0B,$E7,$E6,$7F,$A2,$CF,$F2,$B2,$3F,$C6
	.dc.b	$DA,$F7,$30,$43,$04,$39,$9F,$C8,$FC,$8F,$E6,$C4,$70,$71,$AF,$E2
	.dc.b	$FE,$77,$F3,$23,$CD,$BF,$F5,$5B,$FE,$6B,$E9,$4B,$FD,$7C,$7F,$36
	.dc.b	$F2,$FF,$A1,$12,$3F,$99,$3F,$D1,$F9,$BF,$DD,$98,$FF,$54,$FC,$13
	.dc.b	$FD,$A6,$DA,$FE,$83,$EB,$FF,$44,$97,$B5,$49,$3F,$C6,$E1,$EF,$7F
	.dc.b	$46,$F4,$3F,$A9,$12,$7F,$A1,$3E,$1B,$E9,$10,$FD,$F3,$D1,$61,$E5
	.dc.b	$C3,$0F,$9E,$FF,$A9,$FA,$5F,$D7,$C9,$5F,$D3,$DF,$EC,$98,$BD,$CF
	.dc.b	$51,$27,$D1,$F0,$3E,$DF,$85,$F7,$7C,$1C,$9F,$64,$7F,$60,$F5,$FF
	.dc.b	$B6,$12,$3F,$AE,$3D,$AF,$A4,$7A,$38,$CC,$C7,$C4,$F8,$0E,$A7,$C9
	.dc.b	$C3,$C0,$F1,$38,$61,$FD,$C2,$BF,$BE,$12,$3F,$B6,$3F,$E0,$1F,$9E
	.dc.b	$7B,$3F,$DE,$9B,$3F,$D3,$DA,$F5,$18,$3D,$EE,$15,$27,$83,$F8,$1F
	.dc.b	$87,$C4,$7F,$89,$AF,$9B,$EE,$3D,$EE,$1C,$35,$CC,$F9,$C8,$FF,$8B
	.dc.b	$63,$49,$A4,$8D,$C7,$87,$DC,$F8,$04,$79,$3A,$E1,$FC,$E7,$C3,$FE
	.dc.b	$64,$F4,$BF,$E3,$2B,$F8,$6F,$E8,$7E,$1F,$F3,$C6,$B4,$F4,$FB,$FF
	.dc.b	$34,$98,$F9,$DF,$C6,$FC,$3F,$E9,$7E,$1E,$4F,$E8,$E3,$F5,$1F,$0F
	.dc.b	$23,$F4,$CF,$EA,$FE,$1F,$EB,$7C,$27,$D9,$24,$FC,$37,$AB,$FA,$DB
	.dc.b	$EA,$FD,$9E,$26,$1D,$63,$18,$5D,$3F,$79,$8F,$E0,$FE,$17,$F0,$C4
	.dc.b	$AF,$A0,$FE,$9B,$E1,$FF,$7C,$48,$FD,$E7,$C3,$FF,$01,$9F,$E0,$5A
	.dc.b	$49,$FE,$01,$1F,$8F,$9F,$C9,$12,$3E,$E3,$D5,$FE,$0D,$5D,$5F,$54
	.dc.b	$75,$7F,$85,$57,$57,$DD,$5F,$9B,$AF,$CF,$12,$3F,$10,$FE,$86,$FF
	.dc.b	$40,$48,$FE,$29,$AD,$7F,$41,$C0,$9C,$65,$FC,$87,$8B,$55,$F9,$B3
	.dc.b	$F2,$53,$D5,$FC,$B3,$5F,$D1,$35,$E1,$5D,$5D,$90,$D9,$4D,$E9,$FB
	.dc.b	$92,$3F,$A3,$7B,$5F,$F3,$5E,$3F,$E6,$47,$A3,$87,$F5,$B8,$7F,$9E
	.dc.b	$3D,$BF,$F4,$6D,$7F,$A3,$9B,$3A,$C3,$F8,$17,$17,$F0,$72,$3F,$B1
	.dc.b	$7F,$D1,$52,$EA,$E9,$D9,$2B,$FC,$B1,$39,$6F,$81,$E1,$23,$AF,$0B
	.dc.b	$F8,$97,$2F,$F1,$52,$47,$EF,$DE,$F8,$75,$8F,$E2,$96,$C3,$D9,$3D
	.dc.b	$27,$38,$5C,$FD,$D8,$FF,$1F,$E7,$FE,$44,$4A,$FE,$18,$FB,$3C,$6F
	.dc.b	$8C,$48,$F5,$F8,$9F,$27,$C9,$F9,$5E,$54,$9F,$7C,$7F,$2B,$EC,$FE
	.dc.b	$5C,$48,$FE,$30,$F5,$7E,$2D,$F5,$7D,$77,$D5,$F9,$37,$D5,$F7,$DF
	.dc.b	$F3,$7E,$DF,$E7,$44,$8F,$E4,$8F,$F3,$DE,$EF,$E7,$84,$8F,$A7,$E5
	.dc.b	$D7,$F3,$B1,$75,$FC,$17,$99,$99,$FC,$AC,$F3,$7F,$98,$35,$FE,$78
	.dc.b	$D7,$66,$88,$68,$A7,$E8,$3C,$2F,$EA,$44,$7F,$9E,$3D,$6B,$F2,$9E
	.dc.b	$87,$E5,$CF,$37,$3F,$F4,$6B,$FC,$D7,$9D,$30,$FE,$B1,$1F,$D6,$FC
	.dc.b	$4F,$EB,$C4,$8F,$E9,$0F,$E6,$3D,$3F,$D1,$7B,$12,$7F,$2F,$1F,$A5
	.dc.b	$F5,$64,$7E,$69,$FD,$47,$B7,$FD,$9F,$CA,$FE,$D4,$49,$FE,$A8,$F5
	.dc.b	$7F,$43,$7D,$5F,$CB,$DF,$47,$F3,$37,$75,$FD,$96,$1F,$DC,$FC,$AF
	.dc.b	$EE,$C4,$AF,$EC,$0F,$EC,$3D,$DF,$DD,$12,$3F,$5F,$EF,$7E,$F7,$E0
	.dc.b	$7F,$7D,$F0,$B2,$7F,$67,$1F,$DF,$FC,$FF,$F0,$11,$23,$FB,$63,$D5
	.dc.b	$FD,$CD,$F5,$7F,$5F,$7D,$5F,$DE,$DF,$57,$F6,$77,$FE,$0B,$F4,$3F
	.dc.b	$C1,$C4,$8F,$EF,$0F,$F8,$45,$FF,$08,$10,$FF,$6F,$7E,$0A,$F8,$5C
	.dc.b	$47,$F3,$22,$EA,$E5,$FF,$05,$53,$E1,$F7,$5F,$CB,$0E,$C7,$F8,$1C
	.dc.b	$A7,$F8,$4C,$A6,$CD,$10,$D1,$4F,$E7,$DF,$40,$F5,$7F,$A0,$47,$F8
	.dc.b	$6E,$30,$E0,$38,$76,$D7,$C0,$7F,$8B,$69,$F1,$1F,$E2,$5C,$7D,$D3
	.dc.b	$FE,$44,$9C,$BF,$E3,$53,$E1,$E3,$F0,$BE,$23,$83,$63,$47,$F9,$32
	.dc.b	$B9,$BE,$4F,$67,$CC,$D3,$D1,$8A,$32,$D8,$F6,$60,$F8,$8E,$F1,$E5
	.dc.b	$F8,$8A,$A5,$18,$78,$47,$DE,$F8,$8C,$11,$FC,$0F,$9D,$CB,$07,$BB
	.dc.b	$E2,$24,$F9,$9F,$4D,$F4,$8F,$95,$87,$73,$C1,$8F,$58,$F9,$D8,$DC
	.dc.b	$FE,$1D,$B2,$F0,$7B,$DF,$00,$97,$F1,$1E,$FF,$11,$EB,$24,$7E,$63
	.dc.b	$E2,$3D,$DF,$11,$EF,$F8,$89,$3F,$9C,$8F,$67,$C1,$FC,$C2,$47,$D0
	.dc.b	$7A,$BF,$A2,$BE,$AF,$E1,$6F,$AB,$FA,$6B,$EA,$FE,$26,$FE,$47,$C2
	.dc.b	$7D,$82,$47,$DC,$7E,$DF,$C2,$FC,$D2,$47,$EC,$3E,$1E,$BF,$E7,$FE
	.dc.b	$34,$E3,$7F,$8F,$F0,$FA,$AF,$D1,$CF,$47,$F1,$2D,$7F,$3E,$D7,$66
	.dc.b	$88,$68,$A7,$EB,$A3,$F2,$24,$7F,$2A,$F5,$9F,$B3,$3F,$8D,$3C,$DC
	.dc.b	$DF,$B3,$7F,$21,$8C,$BF,$D1,$4E,$5A,$7F,$AC,$30,$63,$FA,$02,$47
	.dc.b	$E9,$1F,$C7,$ED,$F9,$BD,$E4,$FF,$18,$8F,$CF,$72,$C8,$FE,$36,$FE
	.dc.b	$87,$9B,$F3,$9B,$FE,$B0,$93,$FB,$07,$AB,$FC,$8A,$FA,$BF,$C1,$6F
	.dc.b	$A3,$FC,$1E,$EE,$BF,$9A,$C3,$F5,$1B,$FE,$E0,$95,$FB,$C7,$F4,$FD
	.dc.b	$3F,$B3,$24,$7C,$21,$F8,$37,$FB,$F3,$6F,$F5,$B3,$FE,$FA,$8D,$3F
	.dc.b	$EF,$86,$F8,$5F,$C0,$23,$51,$F8,$9E,$3F,$E1,$24,$7F,$86,$BE,$27
	.dc.b	$F9,$FD,$F8,$9F,$F6,$74,$F8,$5F,$F7,$1B,$D3,$2F,$E3,$2F,$86,$37
	.dc.b	$7E,$1E,$EF,$E1,$1C,$BF,$C6,$09,$3F,$C7,$1F,$E3,$7C,$DF,$C3,$49
	.dc.b	$1E,$CF,$0E,$BF,$A1,$8B,$8F,$83,$C8,$CC,$FF,$42,$9E,$6F,$F2,$86
	.dc.b	$BF,$CF,$9A,$EC,$D1,$0D,$14,$FE,$9B,$D5,$FC,$C0,$8F,$F3,$47,$AD
	.dc.b	$7C,$2F,$27,$E8,$1E,$6D,$FF,$A9,$5F,$D3,$F3,$66,$1F,$77,$A3,$F8
	.dc.b	$F3,$7B,$BF,$9E,$91,$FE,$82,$FF,$44,$E0,$FF,$7F,$47,$E0,$3D,$0F
	.dc.b	$BC,$7D,$FF,$33,$F0,$1E,$AC,$9F,$C2,$BF,$83,$F5,$BF,$9D,$78,$5F
	.dc.b	$D2,$89,$3F,$D3,$9E,$D5,$D2,$3B,$1F,$8E,$7A,$3F,$84,$AB,$BF,$E0
	.dc.b	$AF,$FA,$37,$87,$FD,$58,$95,$FD,$69,$F5,$7B,$7F,$95,$24,$7E,$6B
	.dc.b	$DC,$FC,$C7,$BF,$F9,$9F,$7A,$4F,$E7,$A3,$FA,$97,$91,$FD,$88,$91
	.dc.b	$FD,$99,$EA,$FE,$92,$FA,$BF,$9A,$BE,$AF,$EA,$2F,$AB,$F9,$EB,$FE
	.dc.b	$BD,$E4,$FF,$6E,$24,$7F,$72,$7F,$B8,$F9,$5F,$D8,$C9,$1F,$B2,$F8
	.dc.b	$3A,$FF,$37,$8B,$AF,$E9,$BE,$1B,$33,$FE,$18,$9E,$6F,$F7,$E6,$BF
	.dc.b	$E1,$46,$BB,$34,$43,$45,$3F,$DC,$1C,$2F,$3B,$FC,$16,$D8,$FF,$11
	.dc.b	$35,$FF,$13,$B5,$EC,$7F,$00,$F8,$5F,$E1,$47,$AB,$AB,$FC,$59,$7F
	.dc.b	$C3,$7E,$1A,$57,$E6,$E3,$FC,$1F,$6F,$43,$C2,$49,$F0,$BF,$9B,$F8
	.dc.b	$6F,$E2,$DF,$0D,$27,$F8,$DC,$7F,$19,$F8,$59,$1F,$C7,$9F,$F0,$DF
	.dc.b	$E0,$FF,$EE,$48,$1F,$E2,$1F,$57,$DE,$47,$DE,$F7,$7F,$93,$DF,$77
	.dc.b	$F8,$ED,$F6,$7F,$8F,$DD,$E0,$FF,$1A,$E1,$DE,$7F,$C5,$DE,$B7,$CC
	.dc.b	$23,$F3,$1F,$F1,$CC,$7F,$BB,$61,$3C,$BE,$17,$D2,$7A,$BE,$1B,$D7
	.dc.b	$F0,$DE,$CF,$86,$97,$FD,$07,$04,$FC,$CF,$6F,$ED,$92,$7E,$DB,$E0
	.dc.b	$C3,$AC,$71,$7B,$3C,$E7,$AB,$E4,$E3,$F9,$2A,$D8,$79,$23,$FA,$97
	.dc.b	$0B,$DC,$FC,$59,$23,$F1,$6F,$E3,$BD,$EF,$C0,$92,$3D,$5F,$0D,$5F
	.dc.b	$01,$F6,$29,$39,$DB,$E7,$FC,$36,$EB,$F3,$73,$D5,$FC,$CB,$5F,$CC
	.dc.b	$B5,$D9,$A2,$1A,$29,$FA,$6F,$80,$FD,$29,$1F,$D2,$BD,$6B,$F0,$3F
	.dc.b	$0B,$F9,$03,$CD,$AF,$C8,$C7,$D3,$7F,$5F,$E1,$BF,$2A,$69,$F7,$7C
	.dc.b	$1F,$EC,$88,$FE,$C9,$FC,$3F,$C3,$7E,$70,$91,$F8,$1F,$86,$FC,$FF
	.dc.b	$C3,$7E,$82,$A4,$FE,$16,$3F,$5E,$7E,$F8,$8F,$EF,$9E,$AF,$E2,$EB
	.dc.b	$AB,$F4,$8F,$C0,$57,$67,$F2,$15,$D9,$D7,$12,$FF,$7B,$5F,$C3,$09
	.dc.b	$3F,$C3,$1F,$D0,$EB,$FB,$6D,$A4,$FE,$6E,$3F,$73,$C3,$23,$F3,$CF
	.dc.b	$EF,$39,$7F,$84,$E7,$FC,$60,$93,$FC,$61,$EA,$FE,$96,$FA,$BE,$6B
	.dc.b	$E8,$FA,$2E,$F0,$7E,$EB,$0F,$E2,$BA,$FF,$24,$25,$7F,$24,$7F,$93
	.dc.b	$F0,$7F,$1A,$24,$7E,$CF,$9E,$BF,$27,$81,$38,$C3,$F8,$3F,$5E,$AB
	.dc.b	$F9,$9C,$F4,$7F,$98,$B5,$FE,$62,$D7,$66,$88,$68,$A7,$F3,$EE,$3F
	.dc.b	$E7,$A4,$7F,$9E,$BD,$6B,$F8,$CF,$67,$F2,$83,$CD,$C9,$F9,$9B,$FE
	.dc.b	$57,$E0,$4C,$3F,$7F,$1F,$CE,$B9,$BF,$A4,$92,$3F,$A4,$BF,$ED,$68
	.dc.b	$77,$7F,$B3,$9B,$3F,$E5,$96,$BF,$8F,$5E,$1E,$FC,$8E,$1E,$4C,$AE
	.dc.b	$11,$FD,$2B,$AF,$FA,$C9,$23,$FA,$CB,$E1,$C3,$C0,$48,$F0,$5F,$57
	.dc.b	$C5,$7D,$5F,$15,$FF,$56,$EC,$FE,$CA,$48,$FE,$CA,$FB,$78,$7D,$9E
	.dc.b	$7C,$9F,$64,$7B,$FD,$49,$1E,$F7,$E1,$7A,$7F,$D8,$BC,$1F,$EE,$24
	.dc.b	$9F,$EE,$2F,$57,$E3,$5F,$57,$E3,$5F,$47,$E4,$5D,$E3,$71,$7F,$38
	.dc.b	$E0,$FF,$24,$D7,$E4,$FC,$1F,$EF,$E4,$7F,$BF,$BF,$E0,$29,$FE,$EA
	.dc.b	$48,$FA,$DE,$BD,$7F,$39,$16,$FB,$C7,$F6,$EB,$7B,$7B,$8F,$F0,$7D
	.dc.b	$75,$7F,$C1,$AD,$7F,$C1,$AD,$76,$68,$86,$8A,$7F,$54,$1F,$46,$F1
	.dc.b	$71,$1F,$D6,$23,$0E,$0D,$C5,$69,$DB,$0F,$5B,$FB,$8F,$77,$F3,$07
	.dc.b	$BE,$FF,$9E,$1F,$9B,$F8,$29,$3F,$D8,$A3,$FB,$3D,$72,$C1,$E4,$F7
	.dc.b	$6D,$78,$DA,$FA,$1A,$FB,$73,$FF,$6E,$BF,$A7,$F8,$3F,$D5,$1F,$AB
	.dc.b	$F8,$6F,$D6,$7C,$37,$EB,$7E,$1A,$57,$EB,$63,$C7,$9F,$C3,$F3,$7C
	.dc.b	$44,$9F,$13,$FB,$7C,$3A,$C7,$73,$F6,$D8,$76,$7F,$71,$87,$67,$F7
	.dc.b	$18,$76,$7F,$C5,$F9,$3C,$EF,$80,$48,$F8,$0F,$F0,$5F,$85,$FE,$09
	.dc.b	$F0,$D2,$7F,$82,$47,$F0,$7F,$86,$91,$FC,$1D,$FE,$15,$F0,$DD,$E3
	.dc.b	$F0,$3E,$9F,$CD,$23,$F3,$5E,$CF,$F0,$FB,$EC,$FF,$0F,$BE,$AF,$F1
	.dc.b	$1B,$BC,$5F,$82,$E1,$63,$DF,$37,$A9,$F6,$C9,$87,$DB,$7E,$F7,$AB
	.dc.b	$B2,$47,$F1,$FF,$86,$AF,$F6,$DA,$4E,$31,$FC,$8F,$E1,$1F,$EE,$BA
	.dc.b	$FC,$EC,$F4,$7F,$10,$D7,$F1,$0D,$7F,$49,$5D,$1D,$90,$D9,$4D,$C7
	.dc.b	$E5,$7D,$99,$1F,$95,$6B,$C0,$FF,$02,$57,$FB,$F7,$9B,$FC,$8A,$D8
	.dc.b	$FC,$0D,$7A,$E7,$3F,$0F,$F5,$E8,$E6,$BF,$0F,$ED,$7E,$94,$8F,$E9
	.dc.b	$5F,$DB,$4B,$E1,$B9,$12,$7F,$CF,$B9,$FC,$8F,$86,$ED,$F0,$DD,$FE
	.dc.b	$1A,$56,$B3,$F3,$CF,$DB,$F7,$FF,$68,$49,$FD,$A3,$E0,$C7,$B4,$7C
	.dc.b	$07,$D5,$87,$68,$E8,$7F,$A3,$B2,$63,$E0,$8F,$8F,$87,$E2,$7E,$03
	.dc.b	$F8,$11,$23,$F8,$13,$EE,$F8,$3F,$8D,$F0,$92,$7A,$8C,$3F,$DF,$AF
	.dc.b	$D2,$F8,$4F,$98,$47,$70,$F8,$4F,$CC,$1F,$C4,$49,$1F,$C4,$5F,$03
	.dc.b	$C2,$BC,$0F,$DE,$BE,$8F,$73,$AD,$FD,$BA,$FD,$23,$FC,$74,$91,$FC
	.dc.b	$75,$FE,$41,$5F,$A9,$24,$79,$B0,$AF,$E8,$E2,$EF,$FA,$2D,$B3,$3F
	.dc.b	$9C,$9E,$6F,$F2,$96,$BF,$CA,$5A,$EC,$D1,$0D,$14,$FD,$3E,$3F,$CD
	.dc.b	$48,$FF,$35,$7A,$D7,$ED,$38,$3F,$34,$79,$B8,$3F,$A7,$5F,$E7,$78
	.dc.b	$E6,$1F,$63,$91,$3F,$C2,$78,$3F,$A1,$12,$7F,$A1,$3F,$9E,$E4,$FD
	.dc.b	$31,$23,$ED,$F3,$7E,$A7,$B3,$F5,$5D,$B2,$7E,$FC,$7F,$15,$E4,$FE
	.dc.b	$9C,$48,$FE,$9C,$F5,$7F,$0B,$7D,$5F,$DD,$DF,$57,$F1,$37,$D5,$FD
	.dc.b	$FD,$FF,$21,$E5,$FE,$B4,$48,$FE,$B4,$FE,$DB,$C2,$FD,$F7,$87,$27
	.dc.b	$F2,$71,$FC,$07,$C6,$91,$F9,$67,$F9,$D7,$E1,$3F,$EE,$9D,$FF,$7D
	.dc.b	$3F,$D9,$7A,$BF,$B4,$11,$FE,$F5,$D4,$C3,$B4,$7F,$14,$C7,$F8,$74
	.dc.b	$76,$8F,$E2,$F8,$DE,$5F,$EF,$B9,$77,$EB,$FE,$E8,$4A,$FE,$E8,$FF
	.dc.b	$77,$F8,$8F,$E8,$04,$8F,$D9,$CF,$FB,$E6,$BF,$BF,$4E,$67,$F9,$E3
	.dc.b	$EC,$F3,$F1,$1F,$CD,$23,$A3,$FE,$01,$6B,$FE,$01,$6B,$B3,$44,$34
	.dc.b	$53,$F9,$F9,$FE,$0D,$21,$FE,$0D,$3A,$CF,$C0,$F4,$7E,$79,$E7,$7D
	.dc.b	$31,$EC,$F5,$BE,$99,$9E,$9F,$F8,$03,$F4,$CF,$07,$FC,$50,$DA,$FF
	.dc.b	$16,$35,$FF,$18,$B5,$FF,$1B,$66,$F6,$3E,$F9,$23,$E5,$FB,$7F,$82
	.dc.b	$F1,$7F,$07,$EF,$49,$F9,$F1,$FD,$FF,$0F,$EE,$71,$E0,$F1,$BC,$64
	.dc.b	$7F,$C7,$D8,$F7,$7E,$BE,$3D,$DF,$F6,$E4,$31,$F0,$47,$DC,$C7,$C0
	.dc.b	$EB,$1F,$07,$8F,$F0,$09,$1F,$01,$FF,$25,$5B,$E0,$FF,$37,$F0,$92
	.dc.b	$BF,$0F,$1F,$9E,$F8,$59,$1F,$8A,$7F,$43,$F0,$FE,$DF,$37,$E6,$92
	.dc.b	$7E,$6B,$DA,$BA,$47,$AA,$FF,$53,$1D,$23,$DB,$77,$8F,$F3,$F8,$7C
	.dc.b	$9F,$37,$EC,$92,$BE,$CB,$F7,$3C,$FF,$9C,$48,$FC,$FF,$C2,$D5,$7F
	.dc.b	$E6,$6E,$35,$66,$3F,$5B,$F0,$DB,$AF,$F3,$B5,$69,$FC,$3D,$77,$69
	.dc.b	$0F,$C4,$B9,$69,$93,$9B,$97,$AF,$A3,$F9,$42,$3F,$94,$79,$5F,$F0
	.dc.b	$0F,$86,$FE,$1E,$79,$FA,$3F,$3B,$7F,$C4,$BE,$1A,$61,$FD,$C3,$1F
	.dc.b	$43,$F0,$2B,$F4,$7E,$A7,$E8,$89,$1F,$A2,$7F,$07,$F0,$DF,$C7,$09
	.dc.b	$1F,$C2,$3E,$1B,$F8,$FF,$C3,$7F,$20,$F8,$69,$3F,$C3,$23,$F3,$DE
	.dc.b	$CF,$EB,$89,$1F,$AE,$7A,$BF,$C4,$AF,$AB,$F9,$CB,$EA,$FF,$16,$BE
	.dc.b	$AF,$E8,$2F,$F5,$5E,$D7,$EE,$89,$1F,$BA,$7F,$D4,$9F,$85,$7F,$BD
	.dc.b	$38,$3F,$D4,$6F,$FB,$D2,$7F,$C2,$6D,$7E,$20,$0F,$1D,$31,$18,$F8
	.dc.b	$4F,$F3,$C4,$9A,$63,$E1,$7F,$7D,$EF,$7F,$0E,$25,$7F,$0E,$7C,$BF
	.dc.b	$0F,$CB,$E1,$E4,$F9,$23,$9B,$E6,$BA,$F9,$8F,$2D,$FF,$0B,$F8,$1F
	.dc.b	$E3,$64,$8F,$E3,$6F,$F1,$EF,$82,$FE,$20,$48,$F8,$3F,$0B,$5E,$9C
	.dc.b	$A9,$C6,$DF,$23,$E1,$B5,$5F,$A2,$9E,$8F,$F2,$86,$BF,$CA,$1A,$EC
	.dc.b	$D1,$0D,$14,$FD,$6F,$C3,$FF,$34,$23,$FC,$D1,$EB,$5F,$6B,$E1,$FE
	.dc.b	$89,$E6,$E1,$FE,$7B,$7F,$52,$A5,$FE,$86,$3F,$98,$57,$F4,$02,$47
	.dc.b	$F4,$07,$F0,$F1,$F7,$09,$1F,$86,$CF,$EF,$F0,$7E,$03,$8A,$4F,$E2
	.dc.b	$A3,$F9,$E6,$7F,$D3,$09,$1F,$D3,$1E,$AF,$E4,$2F,$AB,$F6,$F8,$98
	.dc.b	$76,$8F,$CA,$E1,$D9,$D6,$1F,$D2,$76,$FE,$B2,$48,$FE,$B2,$FE,$83
	.dc.b	$9F,$F2,$BD,$12,$7F,$3F,$1F,$98,$EC,$91,$FA,$27,$F3,$7D,$7F,$D5
	.dc.b	$78,$FF,$B2,$92,$7F,$B2,$BD,$5F,$D4,$DF,$57,$D5,$7D,$1F,$65,$DE
	.dc.b	$2F,$D8,$E1,$FD,$87,$93,$FB,$89,$2B,$FB,$8B,$FD,$D3,$9B,$FB,$31
	.dc.b	$23,$F6,$FD,$F5,$FC,$BC,$5B,$FD,$EF,$89,$99,$FE,$6F,$3C,$DF,$EF
	.dc.b	$AD,$7F,$BE,$B5,$D9,$A2,$1A,$29,$FD,$0B,$A3,$FC,$12,$47,$FC,$12
	.dc.b	$F5,$AF,$E1,$FE,$37,$F0,$63,$CD,$D1,$FE,$12,$BF,$E1,$3E,$5C,$C3
	.dc.b	$F9,$B4,$7F,$0E,$B5,$7F,$10,$FA,$87,$6F,$F1,$2B,$91,$87,$07,$F1
	.dc.b	$97,$0E,$1A,$F1,$E7,$FF,$0F,$5F,$F2,$4F,$37,$F8,$D9,$E1,$C3,$F8
	.dc.b	$FF,$C0,$7F,$BE,$4F,$F7,$25,$FF,$A0,$BD,$3D,$3F,$C9,$E3,$FD,$C6
	.dc.b	$FF,$AF,$9D,$E2,$F0,$3C,$B3,$E5,$6A,$8D,$3A,$6B,$EC,$CB,$CF,$87
	.dc.b	$F3,$26,$C7,$F3,$76,$BF,$E9,$63,$CA,$7F,$A6,$3D,$2B,$87,$9A,$3F
	.dc.b	$95,$37,$3F,$9F,$1E,$B9,$E3,$83,$C2,$F3,$B6,$7C,$D1,$83,$D7,$7F
	.dc.b	$E8,$08,$EB,$ED,$78,$0F,$0F,$B3,$E2,$F7,$E4,$FC,$77,$CB,$F0,$1E
	.dc.b	$0F,$1F,$B9,$27,$E2,$3D,$AB,$A4,$7C,$FB,$EA,$F3,$21,$EE,$BE,$AF
	.dc.b	$FB,$CB,$27,$91,$EC,$24,$7D,$17,$DF,$E4,$FC,$22,$47,$DD,$F8,$3A
	.dc.b	$FF,$53,$A4,$E3,$7F,$97,$3F,$EC,$8D,$76,$FE,$62,$7A,$BF,$7D,$AF
	.dc.b	$E5,$DA,$EC,$D1,$0D,$14,$FD,$1F,$97,$F5,$88,$FE,$2D,$EB,$5E,$4A
	.dc.b	$7C,$37,$E1,$0F,$47,$1F,$EB,$F0,$FC,$3F,$C3,$CC,$7E,$44,$7E,$13
	.dc.b	$CE,$FC,$19,$23,$F3,$0F,$E0,$BE,$1F,$F1,$E4,$8F,$D2,$7C,$3F,$E4
	.dc.b	$BE,$1F,$F2,$7F,$0F,$27,$F5,$11,$F9,$0F,$4F,$F1,$E4,$8F,$D2,$3D
	.dc.b	$5F,$D7,$5F,$57,$F3,$77,$D5,$FD,$95,$F5,$7F,$3F,$7F,$9C,$F5,$3F
	.dc.b	$36,$48,$FD,$83,$F9,$AF,$87,$FD,$3F,$C3,$C9,$FD,$E4,$7E,$AB,$E1
	.dc.b	$A4,$7E,$FD,$FD,$6F,$C3,$FE,$A3,$DA,$FD,$39,$27,$F7,$8F,$57,$F8
	.dc.b	$35,$F5,$7F,$67,$7D,$1F,$DA,$DD,$E4,$FB,$18,$7E,$D3,$DA,$FD,$99
	.dc.b	$2B,$F8,$43,$FB,$6F,$77,$F7,$24,$8F,$E2,$9F,$09,$5F,$CC,$45,$BF
	.dc.b	$C2,$7C,$2E,$67,$F9,$6C,$F3,$7F,$8A,$35,$FE,$58,$D7,$66,$88,$68
	.dc.b	$A7,$F3,$8F,$7F,$F8,$59,$1F,$E4,$0F,$5A,$FC,$B7,$C2,$FF,$18,$3C
	.dc.b	$DD,$1F,$D0,$EF,$F8,$EF,$E1,$7F,$AE,$AF,$D7,$B6,$75,$87,$EF,$FE
	.dc.b	$0B,$F8,$C9,$1F,$E5,$CF,$F3,$29,$7C,$4F,$42,$4F,$F9,$8E,$FA,$FC
	.dc.b	$3F,$83,$E1,$FC,$3F,$0F,$2B,$59,$7F,$99,$B1,$57,$F2,$AF,$87,$FE
	.dc.b	$52,$4A,$FE,$7C,$F8,$72,$EF,$1F,$E5,$DA,$F1,$BD,$0F,$F3,$A2,$C7
	.dc.b	$C5,$19,$10,$C7,$C9,$1C,$F0,$FE,$77,$1F,$CE,$89,$1F,$D3,$DF,$85
	.dc.b	$87,$CC,$CE,$4F,$5B,$FF,$79,$3E,$5D,$BE,$89,$1D,$6B,$FD,$23,$1F
	.dc.b	$E8,$E4,$8F,$EB,$8F,$67,$A5,$F6,$7E,$06,$1C,$DE,$C7,$86,$FE,$FD
	.dc.b	$FF,$54,$CB,$FA,$99,$27,$FB,$43,$FD,$5B,$3F,$EB,$24,$8F,$57,$0D
	.dc.b	$7F,$9C,$45,$DB,$F1,$3C,$D9,$9F,$F0,$6C,$F3,$7F,$BA,$35,$FF,$06
	.dc.b	$35,$D9,$A2,$1A,$29,$FE,$18,$DF,$FB,$59,$1F,$F0,$03,$D6,$BF,$37
	.dc.b	$CF,$F9,$E3,$CD,$1F,$E2,$3A,$FD,$0F,$64,$BF,$EB,$91,$FE,$00,$F1
	.dc.b	$DC,$7F,$E0,$E6,$CF,$F8,$49,$AF,$F8,$A1,$AF,$F8,$B7,$89,$DD,$FA
	.dc.b	$E2,$47,$E0,$FC,$3F,$D8,$F8,$1F,$B2,$F1,$A4,$FE,$1E,$3C,$5D,$1E
	.dc.b	$12,$47,$85,$FD,$85,$F4,$8F,$D0,$3F,$BC,$BE,$B1,$F9,$1B,$EA,$FE
	.dc.b	$96,$FD,$DD,$3E,$D2,$47,$B5,$FD,$E7,$93,$FC,$1B,$CB,$93,$F9,$B8
	.dc.b	$FE,$13,$E7,$48,$FC,$F3,$FC,$33,$D1,$F9,$5D,$DF,$24,$93,$F2,$5E
	.dc.b	$AF,$E9,$6F,$AB,$FB,$9B,$E8,$FE,$EE,$EF,$2F,$F1,$6C,$3E,$B7,$7F
	.dc.b	$D5,$25,$7D,$57,$EC,$78,$3F,$6C,$91,$FB,$FF,$C3,$3F,$DE,$8F,$FB
	.dc.b	$DF,$8D,$3A,$8F,$F7,$34,$61,$E1,$E9,$E5,$FF,$35,$46,$5F,$C1,$4F
	.dc.b	$81,$FC,$33,$5C,$98,$21,$82,$1D,$1F,$C8,$78,$72,$3F,$20,$D7,$86
	.dc.b	$5F,$E2,$6C,$1E,$CF,$F9,$61,$B3,$FC,$77,$0F,$35,$7F,$8B,$1B,$D4
	.dc.b	$99,$4F,$7B,$91,$F0,$EF,$87,$8F,$FA,$32,$57,$E8,$DF,$F3,$9D,$3D
	.dc.b	$FF,$11,$27,$E8,$7C,$27,$97,$E1,$7C,$DF,$0D,$27,$EA,$47,$EC,$7C
	.dc.b	$BF,$D8,$12,$3F,$60,$F6,$BE,$90,$CB,$E1,$47,$C1,$C3,$B4,$FD,$FC
	.dc.b	$3B,$3B,$D3,$E0,$2C,$99,$7F,$01,$F3,$3F,$80,$12,$7F,$80,$3E,$EF
	.dc.b	$86,$F9,$DF,$0D,$27,$F1,$91,$F4,$7E,$16,$47,$E3,$DF,$A9,$F0,$DF
	.dc.b	$C3,$7D,$1F,$E1,$84,$9F,$E1,$8F,$57,$F2,$D7,$D5,$F4,$1F,$5A,$FA
	.dc.b	$B8,$3D,$57,$8D,$FF,$09,$87,$F1,$BF,$4B,$F8,$D1,$2B,$F8,$D3,$FC
	.dc.b	$77,$D3,$FE,$40,$48,$FD,$37,$C2,$D7,$F4,$71,$6F,$F2,$1F,$0D,$99
	.dc.b	$FD,$1C,$F3,$7F,$93,$35,$FE,$4C,$D7,$66,$88,$68,$A7,$EB,$BD,$6F
	.dc.b	$E6,$44,$7F,$99,$3D,$6B,$F4,$5F,$0D,$FA,$13,$CD,$C5,$F7,$BA,$7F
	.dc.b	$CE,$F7,$FC,$07,$E1,$BF,$4C,$69,$FE,$05,$ED,$7F,$3F,$23,$FC,$FD
	.dc.b	$FC,$EF,$C3,$7E,$B4,$91,$FC,$2F,$E1,$BF,$61,$F0,$BF,$B1,$F8,$59
	.dc.b	$3F,$C3,$E3,$FA,$67,$C0,$FF,$4B,$24,$7F,$4B,$7A,$BF,$C5,$6F,$AB
	.dc.b	$F7,$7A,$98,$76,$8F,$E3,$58,$76,$75,$87,$F5,$9F,$80,$FE,$B0,$48
	.dc.b	$FE,$B0,$FE,$DB,$E1,$7F,$DB,$BF,$86,$FE,$8A,$7F,$14,$DE,$1F,$E1
	.dc.b	$93,$AF,$84,$95,$FE,$27,$AF,$F7,$D9,$FE,$E1,$E1,$F9,$DC,$3F,$D9
	.dc.b	$C9,$3F,$D9,$DF,$03,$C4,$F0,$9F,$C5,$EF,$AB,$E2,$BB,$87,$FB,$0B
	.dc.b	$0E,$D3,$FC,$DF,$E1,$3F,$BA,$11,$FE,$E8,$FF,$77,$F8,$6F,$EF,$44
	.dc.b	$8F,$44,$57,$F9,$EC,$5A,$ED,$8E,$67,$F9,$EC,$F3,$7F,$C0,$0D,$7F
	.dc.b	$C0,$0D,$76,$68,$86,$8A,$7F,$48,$C3,$FC,$18,$47,$FC,$18,$F5,$AF
	.dc.b	$81,$9F,$D6,$3C,$D9,$FE,$6E,$FE,$CF,$0C,$C3,$F9,$F4,$7F,$86,$5F
	.dc.b	$F0,$C1,$0F,$F0,$C1,$F7,$CF,$E4,$39,$B8,$BF,$06,$48,$FB,$DC,$BF
	.dc.b	$86,$E6,$FC,$3F,$4C,$9F,$C1,$47,$F2,$AD,$3F,$DE,$FF,$58,$E1,$ED
	.dc.b	$5F,$D3,$63,$0E,$1A,$F8,$DA,$FE,$4F,$2F,$23,$F8,$FC,$BC,$8F,$E5
	.dc.b	$B2,$F2,$3F,$93,$CB,$C8,$FE,$6B,$2F,$26,$BF,$CF,$B1,$71,$F9,$29
	.dc.b	$18,$7C,$91,$98,$C4,$7C,$7C,$9D,$FF,$A8,$F0,$24,$FE,$8E,$3F,$57
	.dc.b	$E2,$C8,$FD,$33,$FA,$EF,$1B,$C7,$1F,$0F,$9F,$B1,$1F,$88,$F9,$70
	.dc.b	$EF,$1F,$AF,$C3,$BB,$FB,$6C,$3B,$3F,$B8,$C2,$F0,$FE,$9F,$1E,$F3
	.dc.b	$EE,$E8,$F6,$11,$FA,$4F,$BF,$AB,$7A,$1A,$7F,$06,$F2,$6D,$7F,$A9
	.dc.b	$E7,$4D,$9F,$E1,$7E,$56,$A3,$EE,$47,$57,$F0,$0D,$72,$60,$86,$08
	.dc.b	$7F,$15,$9E,$EE,$0F,$B9,$DB,$F5,$C8,$FE,$35,$ED,$97,$F0,$9F,$33
	.dc.b	$F8,$F9,$E5,$7F,$2F,$7F,$C8,$BD,$19,$7E,$B3,$DD,$E0,$7E,$0C,$8F
	.dc.b	$E6,$5F,$AD,$E9,$7F,$2A,$24,$7F,$25,$F5,$FF,$96,$FB,$3F,$4B,$C8
	.dc.b	$FF,$76,$4F,$F8,$BD,$AB,$97,$FD,$17,$9F,$EA,$BC,$5F,$C9,$12,$7F
	.dc.b	$50,$F8,$1F,$E6,$58,$F8,$1F,$E7,$38,$F8,$1F,$F6,$E6,$1D,$8E,$F8
	.dc.b	$76,$E9,$FD,$27,$8D,$F9,$F2,$57,$ED,$5F,$D3,$57,$FB,$AA,$3F,$C1
	.dc.b	$CF,$FB,$A5,$EB,$EF,$FB,$7D,$BF,$77,$C2,$49,$F0,$BF,$07,$E1,$7F
	.dc.b	$1B,$E5,$FE,$B4,$93,$FC,$0D,$EF,$5F,$1E,$E4,$FA,$A3,$A3,$E2,$BB
	.dc.b	$BB,$3F,$F1,$DE,$5F,$EF,$9A,$FC,$B7,$99,$FB,$B2,$3F,$C4,$5F,$DF
	.dc.b	$79,$BF,$A0,$24,$7C,$8F,$84,$AF,$F8,$B7,$A5,$38,$C7,$C1,$3F,$6F
	.dc.b	$E1,$76,$FE,$5E,$7A,$BF,$C7,$DA,$FF,$1F,$6B,$B3,$44,$34,$53,$F4
	.dc.b	$58,$FF,$15,$23,$FC,$AD,$EB,$5E,$48,$43,$E1,$7F,$24,$7A,$73,$FE
	.dc.b	$49,$7F,$95,$F8,$59,$7F,$BD,$8F,$E0,$7E,$9F,$F2,$32,$47,$F3,$97
	.dc.b	$F0,$BF,$0B,$F9,$DF,$85,$93,$F9,$08,$FD,$07,$C2,$C8,$FC,$9B,$FA
	.dc.b	$3F,$85,$FE,$1F,$EB,$FF,$2F,$24,$FF,$45,$7A,$BF,$9A,$BE,$AF,$E4
	.dc.b	$AF,$A3,$F9,$4B,$BB,$7C,$1D,$98,$FF,$1C,$F6,$3F,$9E,$12,$FF,$A9
	.dc.b	$3F,$9E,$F8,$4F,$DB,$12,$3F,$53,$F0,$9F,$B9,$F8,$3F,$DD,$7C,$24
	.dc.b	$9F,$D6,$C7,$F2,$8F,$77,$FA,$41,$23,$FA,$F3,$D5,$FD,$9D,$F5,$7F
	.dc.b	$57,$7D,$5F,$DB,$DF,$57,$F5,$F7,$FC,$D3,$DE,$FE,$A8,$48,$FE,$D4
	.dc.b	$FF,$57,$F7,$FF,$9C,$92,$3F,$80,$FC,$1D,$7F,$35,$17,$6F,$8D,$F0
	.dc.b	$99,$9F,$E5,$D3,$CD,$FE,$EC,$D7,$FB,$B3,$5D,$9A,$21,$A2,$9F,$CE
	.dc.b	$7E,$0B,$FB,$61,$1F,$F0,$13,$D6,$BE,$C7,$C2,$7F,$27,$3C,$DD,$3F
	.dc.b	$DE,$6F,$F9,$57,$C2,$4C,$3F,$A9,$47,$F5,$8F,$88,$FE,$F0,$48,$FF
	.dc.b	$07,$3F,$DF,$FF,$0E,$FF,$7F,$C1,$65,$FD,$99,$B1,$FE,$E3,$CB,$F5
	.dc.b	$51,$C0,$FF,$75,$C7,$FA,$BD,$B1,$AB,$E5,$AF,$F7,$0F,$AE,$65,$FE
	.dc.b	$0F,$6E,$1F,$E2,$46,$BF,$E3,$36,$BF,$E7,$3C,$3C,$18,$F9,$4F,$36
	.dc.b	$5E,$7C,$65,$79,$F3,$FF,$6D,$DF,$91,$B9,$78,$F4,$F1,$12,$BC,$4F
	.dc.b	$C5,$E3,$F8,$44,$8E,$80,$F9,$1C,$9F,$1F,$AB,$E4,$75,$C9,$D9,$F3
	.dc.b	$63,$E1,$70,$FC,$12,$47,$C1,$7C,$0F,$0C,$D8,$78,$63,$E7,$61,$DD
	.dc.b	$D6,$1D,$DF,$A5,$87,$CF,$E2,$F9,$C4,$8F,$9C,$FD,$1E,$4F,$A8,$48
	.dc.b	$F0,$74,$57,$FD,$D5,$C4,$9C,$72,$7E,$0B,$B3,$55,$FE,$A7,$AD,$3F
	.dc.b	$6E,$BB,$34,$87,$DD,$72,$D3,$27,$37,$2F,$5E,$6F,$C4,$11,$FC,$43
	.dc.b	$CA,$FF,$2B,$E1,$7E,$64,$F3,$F1,$FD,$7A,$61,$E6,$7F,$31,$E3,$7E
	.dc.b	$7C,$DB,$F7,$61,$D3,$F9,$72,$47,$E5,$DF,$D5,$79,$1F,$A7,$F3,$24
	.dc.b	$FD,$98,$FD,$57,$9D,$23,$EE,$3F,$AD,$F4,$BF,$49,$DD,$FA,$32,$4F
	.dc.b	$E8,$DE,$AF,$E0,$EF,$AB,$FA,$FB,$E8,$FE,$C6,$EE,$BF,$B6,$C3,$F6
	.dc.b	$1D,$FF,$AF,$25,$7E,$BD,$FE,$05,$E8,$FE,$FC,$91,$F9,$1F,$57,$F8
	.dc.b	$17,$B5,$FC,$0F,$DB,$93,$F9,$58,$FD,$E7,$89,$FB,$B2,$47,$EE,$DE
	.dc.b	$AF,$E6,$EF,$AB,$FC,$1E,$FA,$BF,$9F,$BE,$AF,$F0,$BB,$FE,$11,$E3
	.dc.b	$7F,$07,$24,$7F,$07,$7F,$85,$78,$DF,$C3,$49,$1F,$A9,$F8,$0A,$FE
	.dc.b	$6A,$2E,$BF,$E0,$3F,$C3,$FF,$DF,$9C,$DF,$85,$E5,$57,$4B,$FE,$28
	.dc.b	$D7,$F8,$A3,$5F,$D0,$35,$E4,$D7,$F1,$4D,$7F,$23,$E4,$FF,$21,$23
	.dc.b	$FC,$85,$ED,$87,$97,$E1,$7F,$D2,$F8,$9F,$4E,$3D,$AF,$F8,$6C,$BE
	.dc.b	$0F,$66,$A3,$F3,$5E,$67,$F2,$F2,$3F,$CB,$DF,$F6,$04,$BE,$1F,$BF
	.dc.b	$C3,$CA,$EB,$1E,$2F,$87,$91,$E0,$7C,$BF,$0F,$FC,$F7,$D1,$FE,$78
	.dc.b	$49,$FE,$78,$F6,$AE,$90,$D7,$E9,$61,$EE,$9E,$B3,$9D,$3F,$71,$93
	.dc.b	$2C,$E1,$F6,$72,$FE,$99,$E9,$FF,$4B,$26,$1F,$D2,$DF,$6F,$C3,$FC
	.dc.b	$A2,$47,$C8,$F8,$7F,$9B,$F0,$DF,$3B,$E1,$E4,$FC,$C8,$FE,$B1,$ED
	.dc.b	$FF,$57,$24,$7F,$57,$7A,$BF,$4A,$FA,$BE,$93,$FA,$36,$8C,$3B,$C7
	.dc.b	$D8,$C3,$BB,$B1,$2C,$3F,$B4,$7B,$3F,$D9,$C9,$3F,$D9,$DF,$ED,$5E
	.dc.b	$DF,$F6,$D2,$47,$E1,$3E,$1A,$BF,$CE,$A2,$ED,$FA,$1F,$87,$CC,$FF
	.dc.b	$39,$9E,$6F,$F7,$36,$BF,$DC,$DA,$EC,$D1,$0D,$14,$FE,$89,$EE,$FF
	.dc.b	$7F,$23,$FD,$FD,$EB,$5F,$B3,$F8,$7F,$CF,$9E,$6E,$3F,$DA,$5F,$E8
	.dc.b	$BE,$1E,$61,$FD,$B3,$B7,$FD,$60,$FF,$56,$AF,$F0,$61,$1F,$F0,$63
	.dc.b	$F9,$EF,$87,$FD,$5F,$C3,$C9,$FD,$4C,$7E,$BB,$E1,$E4,$7E,$B1,$FD
	.dc.b	$8F,$C3,$FF,$49,$D3,$FD,$15,$F6,$0F,$88,$FE,$99,$72,$30,$E0,$FF
	.dc.b	$03,$38,$70,$D7,$8F,$23,$1F,$24,$7E,$0E,$19,$7F,$02,$9F,$2C,$E7
	.dc.b	$2B,$8F,$F1,$BC,$BC,$D9,$FF,$5F,$C9,$F0,$FE,$3C,$3C,$6D,$51,$A7
	.dc.b	$4D,$7D,$59,$BE,$23,$F8,$99,$23,$F8,$8D,$7F,$16,$D7,$C3,$1F,$E2
	.dc.b	$5F,$C4,$3F,$DF,$67,$FB,$F2,$FF,$DD,$D1,$F1,$2F,$C5,$87,$58,$F0
	.dc.b	$C6,$0F,$E2,$A7,$77,$E2,$3F,$C9,$30,$F0,$47,$F8,$BA,$FB,$E3,$88
	.dc.b	$FD,$65,$B2,$EE,$48,$F7,$BE,$2D,$3D,$31,$F1,$1B,$27,$37,$FF,$00
	.dc.b	$74,$FC,$1C,$68,$95,$E3,$E1,$AF,$E0,$23,$B3,$F3,$9A,$E4,$C1,$0C
	.dc.b	$10,$FC,$73,$C1,$FC,$86,$FF,$28,$8F,$DA,$7A,$D7,$FB,$86,$DC,$DE
	.dc.b	$A3,$CB,$1E,$AA,$F9,$5D,$7E,$E3,$3A,$7E,$2F,$87,$EB,$92,$7F,$0E
	.dc.b	$FE,$2F,$07,$67,$C6,$F0,$65,$7D,$58,$F9,$3E,$14,$8F,$B0,$FC,$CF
	.dc.b	$17,$E7,$F3,$FE,$10,$93,$F9,$47,$B5,$74,$8F,$E5,$18,$30,$FA,$B3
	.dc.b	$D6,$73,$85,$D3,$E1,$63,$F7,$3A,$3F,$22,$4A,$FD,$0B,$EB,$F1,$BE
	.dc.b	$F9,$23,$F1,$DE,$47,$E0,$BC,$DF,$C1,$F9,$D2,$7F,$25,$1F,$88,$EE
	.dc.b	$FC,$E9,$23,$F5,$AF,$57,$F3,$17,$D5,$F5,$DF,$57,$F3,$97,$D5,$F7
	.dc.b	$DF,$E5,$3B,$FF,$52,$48,$FD,$CB,$FA,$CF,$07,$F3,$24,$8F,$D3,$7A
	.dc.b	$75,$FD,$2C,$5D,$7F,$0D,$EA,$66,$7F,$8E,$4F,$37,$F8,$2B,$5F,$E3
	.dc.b	$6D,$76,$68,$86,$8A,$7F,$27,$F0,$FF,$80,$91,$FE,$22,$F5,$AF,$CC
	.dc.b	$7A,$FF,$B0,$3C,$DC,$FF,$C7,$6F,$F6,$7E,$C4,$C3,$FD,$5F,$DC,$9F
	.dc.b	$DC,$F8,$DF,$C3,$89,$3F,$C7,$9F,$CC,$7B,$7F,$BC,$F8,$19,$3F,$C2
	.dc.b	$E3,$F7,$FF,$05,$23,$F8,$73,$FC,$0B,$E1,$FF,$82,$F9,$7F,$C6,$89
	.dc.b	$3F,$CA,$9E,$AF,$F1,$5B,$EA,$FE,$62,$FA,$3F,$9A,$BB,$B7,$ED,$F0
	.dc.b	$FE,$23,$E6,$7F,$26,$25,$7F,$36,$7F,$9D,$75,$7F,$BE,$89,$F0,$DF
	.dc.b	$C6,$9F,$F7,$CC,$FF,$83,$5F,$F7,$13,$FE,$D9,$5F,$0B,$B3,$87,$C3
	.dc.b	$7F,$20,$F4,$3F,$9A,$92,$BF,$A2,$BE,$0B,$EC,$48,$FF,$6F,$DF,$83
	.dc.b	$27,$1F,$FA,$C6,$3C,$32,$7C,$2F,$0C,$7C,$2F,$DA,$F4,$BF,$A2,$91
	.dc.b	$FE,$AA,$FF,$48,$F4,$BF,$9A,$92,$3C,$3F,$0B,$5F,$E8,$31,$71,$F1
	.dc.b	$C7,$C1,$F8,$5D,$3F,$DE,$67,$A3,$FD,$89,$AF,$F7,$86,$BB,$34,$43
	.dc.b	$45,$3F,$C0,$9E,$AF,$F5,$E2,$3F,$DB,$9E,$B5,$E4,$8F,$89,$F0,$9F
	.dc.b	$64,$F4,$69,$FD,$EE,$FE,$E7,$C2,$CA,$FC,$FC,$7F,$56,$F6,$3F,$B5
	.dc.b	$92,$3F,$BD,$BF,$86,$F8,$5F,$C2,$7C,$2C,$9F,$AF,$1F,$87,$F8,$59
	.dc.b	$1F,$69,$FC,$57,$C2,$FF,$62,$8F,$EE,$E4,$3F,$C0,$E7,$53,$F0,$35
	.dc.b	$D5,$FC,$85,$74,$7F,$25,$57,$4F,$81,$C4,$C3,$FB,$8F,$A4,$F7,$7F
	.dc.b	$C2,$CD,$C3,$FC,$34,$D7,$FC,$5C,$D7,$FC,$67,$4F,$84,$FD,$11,$23
	.dc.b	$F2,$BF,$0B,$FA,$5F,$85,$FD,$37,$C2,$C9,$FC,$CC,$78,$BE,$0F,$C2
	.dc.b	$48,$F0,$BF,$B0,$BE,$91,$FA,$D7,$F6,$17,$D6,$3F,$47,$7D,$5F,$D9
	.dc.b	$5F,$BB,$E0,$FD,$A4,$8F,$6B,$F0,$3E,$13,$E1,$92,$3F,$5B,$F0,$B5
	.dc.b	$FF,$44,$68,$9C,$6F,$F3,$1D,$4F,$F9,$5A,$74,$FC,$A8,$EC,$D2,$1F
	.dc.b	$31,$CB,$4C,$9C,$DC,$BD,$6B,$EB,$91,$FA,$EF,$2A,$FB,$F7,$FC,$54
	.dc.b	$F3,$F3,$2A,$61,$E6,$7F,$8C,$6B,$FC,$6C,$DB,$AC,$FF,$0A,$47,$F0
	.dc.b	$AF,$F2,$4D,$FF,$91,$12,$3B,$63,$FA,$FF,$C4,$7F,$E0,$0C,$78,$5F
	.dc.b	$F7,$73,$AC,$7C,$8A,$3F,$DD,$CE,$4E,$1D,$3F,$E4,$88,$E3,$1F,$53
	.dc.b	$6F,$CC,$91,$FC,$CB,$E5,$7F,$93,$E1,$E5,$7F,$D8,$53,$E4,$7C,$1C
	.dc.b	$DD,$6E,$47,$84,$C9,$74,$70,$7E,$A0,$91,$FA,$87,$FD,$01,$7F,$EF
	.dc.b	$F8,$E4,$BB,$3D,$47,$AF,$A3,$D9,$DF,$2B,$6F,$BF,$C1,$FD,$BF,$2F
	.dc.b	$ED,$89,$3F,$B6,$7B,$D7,$48,$EF,$7D,$5E,$64,$3E,$0D,$F5,$7D,$9C
	.dc.b	$FF,$C0,$88,$FF,$02,$7F,$82,$F4,$7F,$08,$24,$7B,$3C,$6A,$FE,$9A
	.dc.b	$2F,$1F,$E0,$7C,$7C,$CF,$E9,$67,$9B,$FC,$39,$AF,$F0,$E6,$BB,$34
	.dc.b	$43,$45,$3F,$61,$D9,$FC,$68,$8F,$F1,$A7,$AD,$7E,$4F,$CC,$FC,$51
	.dc.b	$E6,$E6,$FC,$8F,$13,$0F,$C7,$79,$33,$1F,$C0,$47,$F2,$8E,$EF,$E4
	.dc.b	$E4,$8F,$E4,$EF,$E0,$3C,$FF,$CB,$12,$3F,$03,$EA,$7E,$67,$D5,$FC
	.dc.b	$D7,$B5,$27,$F0,$B1,$FC,$D3,$C3,$FE,$66,$48,$FE,$66,$F5,$7F,$17
	.dc.b	$7D,$5F,$CD,$DF,$57,$F1,$F7,$D5,$FC,$FD,$FF,$40,$F1,$3F,$9F,$92
	.dc.b	$3F,$9F,$BF,$98,$F6,$FF,$5D,$EE,$C9,$FC,$CC,$7E,$C7,$DE,$91,$F9
	.dc.b	$C7,$F6,$9F,$01,$FD,$33,$C9,$FE,$96,$49,$FE,$96,$F5,$7F,$47,$7D
	.dc.b	$5F,$D9,$DF,$47,$F6,$B7,$79,$3F,$85,$61,$FD,$63,$CA,$FE,$AE,$4A
	.dc.b	$FE,$AE,$FF,$5A,$F2,$FF,$AE,$92,$3F,$63,$F0,$55,$FE,$77,$16,$FF
	.dc.b	$8B,$7C,$16,$67,$F9,$D4,$F3,$7F,$B1,$B5,$FE,$C6,$D7,$66,$88,$68
	.dc.b	$A7,$F4,$5F,$3F,$FB,$79,$1F,$ED,$EF,$5A,$FE,$51,$F0,$9F,$C7,$CF
	.dc.b	$37,$4F,$E5,$AF,$FD,$D3,$F8,$97,$F8,$6E,$BF,$C3,$8D,$9D,$61,$FC
	.dc.b	$5F,$E2,$7F,$BE,$91,$FE,$FA,$FF,$30,$64,$FF,$BB,$E7,$F9,$9B,$63
	.dc.b	$FA,$43,$52,$F8,$5F,$1F,$67,$93,$E1,$65,$7F,$29,$AF,$F7,$1B,$FC
	.dc.b	$A7,$D4,$FF,$07,$91,$FF,$07,$BE,$37,$C3,$F1,$1E,$E2,$3C,$F0,$EE
	.dc.b	$F5,$BE,$EB,$3F,$87,$8F,$F4,$C7,$FC,$81,$F6,$8F,$5F,$B5,$7F,$63
	.dc.b	$8C,$38,$6B,$E3,$6B,$F1,$BE,$17,$E8,$FC,$34,$8F,$74,$7D,$4F,$86
	.dc.b	$91,$F0,$5F,$AF,$ED,$78,$F8,$3F,$AF,$E0,$F6,$FC,$76,$8C,$3E,$38
	.dc.b	$CC,$62,$3E,$37,$0B,$1F,$14,$7C,$1B,$FB,$39,$7E,$1A,$7C,$73,$93
	.dc.b	$ED,$E5,$8D,$3F,$13,$9F,$96,$7E,$27,$BB,$E0,$23,$F1,$5F,$1F,$BB
	.dc.b	$B8,$F4,$37,$43,$F9,$6F,$86,$C3,$27,$FB,$64,$E9,$FC,$37,$C3,$65
	.dc.b	$FA,$13,$D9,$FA,$8D,$72,$60,$86,$08,$7E,$76,$7B,$38,$3E,$8F,$C0
	.dc.b	$FC,$C2,$3F,$81,$7B,$65,$FA,$CF,$85,$FD,$79,$E5,$7F,$17,$7F,$B2
	.dc.b	$F8,$69,$7E,$C8,$F8,$5F,$09,$F6,$09,$1F,$8D,$7E,$FF,$13,$E1,$BF
	.dc.b	$76,$49,$FD,$47,$C3,$7E,$FA,$BF,$7F,$9C,$9F,$D6,$47,$CF,$9F,$C2
	.dc.b	$92,$3F,$34,$F6,$AE,$91,$F8,$13,$F7,$2F,$F0,$A8,$ED,$1F,$B7,$AE
	.dc.b	$CE,$ED,$7F,$76,$FF,$26,$49,$FD,$43,$EC,$C3,$F8,$AE,$F2,$7F,$81
	.dc.b	$47,$F1,$8E,$59,$1F,$C1,$5F,$E3,$7C,$DF,$89,$D3,$F3,$E4,$9F,$DA
	.dc.b	$3D,$5F,$E1,$97,$D5,$F3,$5F,$47,$D1,$77,$5F,$E4,$B8,$7E,$57,$6F
	.dc.b	$D5,$92,$BF,$80,$3F,$AE,$E0,$FC,$D1,$23,$FC,$85,$F8,$9B,$B3,$11
	.dc.b	$F2,$F8,$95,$8C,$7F,$D9,$D7,$FE,$F5,$E8,$FD,$FC,$F5,$BF,$E1,$CD
	.dc.b	$7F,$87,$35,$FC,$C3,$5E,$27,$02,$1C,$0F,$E3,$1C,$9F,$C1,$88,$E0
	.dc.b	$DD,$F9,$3C,$0F,$DC,$60,$7D,$18,$76,$AF,$C3,$4B,$BF,$66,$A3,$F9
	.dc.b	$47,$3F,$F1,$42,$3F,$CA,$5F,$E3,$72,$F0,$FB,$F9,$12,$BE,$5C,$78
	.dc.b	$BC,$99,$1F,$39,$F2,$F9,$DF,$C2,$FB,$3F,$90,$92,$7F,$9B,$3D,$AB
	.dc.b	$A4,$7A,$2F,$DB,$1D,$23,$D7,$77,$5F,$AB,$87,$F1,$7E,$EF,$E5,$A4
	.dc.b	$AF,$E8,$4F,$AB,$CF,$F8,$E4,$8F,$C1,$FA,$7F,$2B,$D1,$F9,$7E,$B4
	.dc.b	$9F,$C3,$C7,$F2,$3F,$0B,$F9,$C9,$23,$FA,$73,$D5,$FC,$6D,$F5,$7E
	.dc.b	$8D,$F5,$7F,$23,$7D,$5F,$AB,$7F,$CB,$FC,$4F,$E8,$A4,$8F,$EB,$4F
	.dc.b	$F4,$8F,$13,$F9,$A1,$23,$F3,$7E,$D5,$7F,$35,$17,$5F,$CC,$FB,$B9
	.dc.b	$9F,$E5,$53,$CD,$FE,$CC,$D7,$FB,$33,$5D,$9A,$21,$A2,$9F,$CD,$7C
	.dc.b	$7F,$EB,$A4,$7F,$B9,$3D,$6B,$F5,$7F,$03,$F9,$83,$CD,$D1,$FD,$AA
	.dc.b	$FF,$37,$F0,$53,$0F,$E9,$B1,$FD,$4F,$CA,$FE,$D2,$48,$FE,$FC,$FE
	.dc.b	$5B,$E1,$7F,$47,$F0,$D2,$7F,$7F,$1F,$A6,$F8,$69,$1F,$C0,$9F,$D4
	.dc.b	$FC,$37,$F5,$FC,$BF,$BA,$92,$7F,$C1,$4F,$57,$F8,$4D,$F5,$7F,$5B
	.dc.b	$7D,$1F,$D7,$DD,$D3,$F8,$4E,$1F,$DB,$7E,$D9,$E8,$FF,$81,$9B,$7F
	.dc.b	$E1,$A6,$BF,$E2,$96,$BF,$E3,$DE,$E7,$C3,$FF,$00,$24,$7F,$B7,$7D
	.dc.b	$EF,$F7,$11,$FE,$EC,$C5,$AF,$FB,$D2,$3F,$DE,$73,$FD,$25,$B3,$A8
	.dc.b	$08,$E5,$3F,$EF,$CC,$1E,$A7,$A4,$95,$E9,$7F,$83,$E1,$E4,$8F,$E2
	.dc.b	$2F,$F8,$36,$FC,$BF,$0F,$7E,$1E,$57,$A6,$3E,$37,$AF,$F1,$49,$1F
	.dc.b	$15,$F9,$1E,$C7,$CB,$24,$7C,$1F,$85,$AF,$F6,$5C,$D3,$8D,$7D,$D3
	.dc.b	$ED,$F8,$5D,$BF,$E6,$9A,$DB,$F4,$EB,$BB,$48,$7D,$57,$2D,$32,$73
	.dc.b	$72,$F5,$F6,$FF,$00,$47,$F0,$0F,$2B,$F2,$E8,$F8,$5F,$90,$7A,$78
	.dc.b	$3E,$96,$1F,$2F,$E1,$A6,$3F,$63,$0F,$6E,$5E,$A8,$F8,$5E,$FF,$E3
	.dc.b	$C8,$FE,$3D,$FF,$2B,$E4,$F8,$6F,$AC,$49,$FC,$67,$C3,$7D,$99,$FB
	.dc.b	$5F,$0D,$27,$F2,$11,$F9,$DF,$82,$FC,$E1,$23,$F3,$8F,$6B,$E9,$1F
	.dc.b	$91,$7F,$09,$7D,$63,$F3,$57,$D5,$FC,$B5,$FE,$A7,$E1,$3F,$50,$48
	.dc.b	$FD,$43,$EA,$F8,$6F,$C7,$7C,$34,$9F,$D2,$47,$E4,$7E,$1A,$47,$E9
	.dc.b	$DF,$CA,$57,$ED,$5F,$DA,$12,$3F,$68,$F5,$7F,$5D,$5D,$5F,$D5,$57
	.dc.b	$47,$F5,$93,$76,$FB,$77,$FC,$06,$FF,$80,$12,$BF,$80,$3F,$C0,$EF
	.dc.b	$F8,$31,$23,$F7,$9A,$57,$F3,$91,$6F,$F0,$FA,$E6,$7F,$35,$3C,$DF
	.dc.b	$E1,$8D,$7F,$86,$35,$D9,$A2,$1A,$29,$FA,$5C,$FF,$8C,$11,$FE,$30
	.dc.b	$F5,$AF,$CC,$F1,$FE,$FC,$F3,$71,$7F,$09,$BF,$E0,$5C,$53,$0F,$CC
	.dc.b	$4E,$5A,$7F,$12,$E1,$FE,$24,$FF,$25,$DF,$F9,$41,$1F,$F6,$0A,$E6
	.dc.b	$FE,$1D,$D5,$27,$F9,$24,$7F,$11,$EC,$91,$FC,$9D,$FF,$50,$7E,$29
	.dc.b	$FE,$EF,$8F,$07,$27,$F3,$52,$3F,$CD,$5F,$D3,$D7,$58,$FE,$55,$7F
	.dc.b	$C7,$63,$AC,$7F,$2D,$BB,$B7,$F1,$0C,$3B,$4B,$2F,$D8,$3F,$D0,$79
	.dc.b	$7F,$A2,$11,$C1,$0F,$E8,$8E,$5A,$64,$FE,$34,$E5,$FE,$39,$CD,$FD
	.dc.b	$3C,$8F,$F4,$F7,$FD,$AF,$4F,$0B,$E1,$F8,$D2,$BE,$98,$9D,$38,$5E
	.dc.b	$27,$CA,$24,$67,$C9,$FE,$C1,$D7,$FD,$7C,$91,$FD,$7D,$F0,$E1,$DE
	.dc.b	$3F,$D3,$96,$C3,$EA,$CF,$49,$CE,$17,$4F,$99,$8F,$F6,$DE,$CF,$ED
	.dc.b	$84,$AF,$ED,$8D,$7F,$B6,$35,$FE,$80,$D7,$FA,$F3,$5F,$C9,$3F,$DD
	.dc.b	$FB,$64,$7F,$77,$6B,$FD,$DD,$AF,$F4,$B6,$BF,$DA,$5A,$FE,$75,$AF
	.dc.b	$E0,$FB,$FF,$C0,$84,$7F,$C0,$8D,$7F,$C0,$8D,$7F,$AC,$35,$FE,$EC
	.dc.b	$D7,$F5,$4F,$F8,$3F,$BE,$47,$F8,$3D,$AF,$F8,$3D,$AF,$F6,$36,$BF
	.dc.b	$E0,$16,$BF,$B5,$6B,$F9,$B3,$FC,$38,$43,$FC,$38,$53,$FC,$38,$53
	.dc.b	$FB,$81,$4F,$F0,$71,$4F,$E0,$47,$F8,$50,$87,$F8,$5B,$EE,$3F,$C4
	.dc.b	$6E,$0C,$78,$35,$A5,$8D,$BB,$6B,$D1,$AF,$76,$BC,$8E,$11,$D5,$46
	.dc.b	$9B,$19,$8C,$0F,$0F,$04,$9F,$23,$5F,$23,$5F,$23,$5F,$23,$5F,$84
	.dc.b	$F7,$F1,$24,$7B,$DA,$FB,$DA,$FB,$DA,$FB,$DA,$FC,$D6,$BE,$EF,$1B
	.dc.b	$D8,$47,$E6,$35,$F9,$8D,$7E,$63,$5F,$98,$D7,$ED,$3F,$1F,$C6,$91
	.dc.b	$F5,$DA,$FD,$76,$BF,$5D,$AF,$D7,$6B,$F8,$56,$BF,$5B,$C7,$FA,$84
	.dc.b	$7F,$08,$D7,$F0,$8D,$7F,$08,$D7,$F0,$8D,$7F,$24,$FD,$FF,$22,$47
	.dc.b	$E3,$DA,$FE,$3D,$AF,$E3,$DA,$FE,$3D,$AF,$E7,$5A,$FE,$3B,$C9,$FC
	.dc.b	$61,$1F,$CE,$35,$FC,$E3,$5F,$CE,$35,$FC,$E3,$5F,$D5,$3F,$97,$F2
	.dc.b	$A4,$7E,$9D,$AF,$E9,$DA,$FE,$9D,$AF,$E9,$DA,$FE,$D5,$AF,$E9,$BC
	.dc.b	$CF,$D2,$11,$FD,$A3,$5F,$DA,$35,$FD,$A3,$5F,$DA,$35,$FE,$04,$FE
	.dc.b	$BF,$CC,$91,$FB,$F6,$BF,$BF,$6B,$FB,$F6,$BF,$BF,$6B,$FC,$35,$AF
	.dc.b	$EF,$BC,$DF,$DE,$11,$FE,$18,$D7,$F8,$63,$5F,$E1,$8D,$7F,$86,$35
	.dc.b	$FE,$34,$FF,$07,$F3,$A4,$7F,$17,$6B,$FC,$5D,$AF,$F1,$76,$BF,$C5
	.dc.b	$DA,$FF,$25,$6B,$FC,$5B,$D1,$FE,$28,$47,$F9,$23,$5F,$E4,$8D,$7F
	.dc.b	$92,$35,$FE,$48,$D7,$F9,$93,$FC,$7F,$D2,$91,$FC,$BD,$AF,$F2,$F6
	.dc.b	$BF,$CB,$DA,$FF,$2F,$6B,$FC,$F5,$AF,$F2,$EF,$53,$F9,$61,$1F,$E7
	.dc.b	$8D,$7F,$9E,$35,$FE,$78,$D7,$F9,$E3,$5F,$E9,$4F,$F3,$7F,$52,$47
	.dc.b	$F4,$76,$BF,$D1,$DA,$FF,$47,$6B,$FD,$1D,$AF,$F5,$56,$BF,$D1,$BD
	.dc.b	$7F,$E8,$84,$7F,$AA,$35,$FE,$A8,$D7,$FA,$A3,$5F,$EA,$8D,$7F,$B1
	.dc.b	$3F,$D3,$FD,$89,$1F,$D7,$DA,$FF,$5F,$6B,$FD,$7D,$AF,$F5,$F6,$BF
	.dc.b	$DB,$5A,$FF,$5E,$F8,$8F,$EB,$84,$7F,$B6,$35,$FE,$D8,$D7,$FB,$63
	.dc.b	$5F,$ED,$8D,$7F,$BD,$3F,$D9,$FD,$D9,$1F,$DD,$DA,$FF,$77,$6B,$FD
	.dc.b	$DD,$AF,$F7,$76,$BF,$E0,$56,$AA,$7B,$FF,$DE,$5B,$1F,$E0,$76,$BF
	.dc.b	$E0,$B6,$BF,$E1,$06,$BF,$E1,$36,$A3,$B4,$75,$23,$D5,$AF,$57,$C5
	.dc.b	$F0,$12,$3C,$4D,$7C,$4D,$7C,$4D,$7C,$4D,$7C,$52,$8C,$47,$C1,$F8
	.dc.b	$2F,$80,$47,$E0,$35,$F8,$0D,$7E,$03,$5F,$80,$D7,$BB,$F2,$FE,$0A
	.dc.b	$47,$CB,$6B,$F2,$DA,$FC,$B6,$BF,$2D,$AF,$A9,$AF,$D9,$F8,$7F,$B0
	.dc.b	$47,$EC,$35,$FB,$0D,$7E,$C3,$5F,$B0,$D7,$E3,$BF,$83,$F8,$69,$1F
	.dc.b	$83,$6B,$F8,$36,$BF,$83,$6B,$F8,$36,$BF,$49,$AF,$E4,$7E,$23,$F2
	.dc.b	$04,$7F,$20,$D7,$F2,$0D,$7F,$20,$D7,$F2,$0D,$7E,$FB,$F9,$BF,$87
	.dc.b	$91,$F9,$B6,$BF,$9B,$6B,$F9,$B6,$BF,$9B,$6B,$F8,$A6,$BF,$A9,$F8
	.dc.b	$7F,$D4,$11,$FD,$43,$5F,$D4,$35,$FD,$43,$5F,$D4,$35,$FC,$BB,$FB
	.dc.b	$3F,$87,$91,$FB,$36,$BF,$B3,$6B,$FB,$36,$BF,$B3,$6B,$FA,$26,$BF
	.dc.b	$C0,$7E,$1B,$F8,$01,$1F,$E0,$0D,$7F,$80,$35,$FE,$00,$D7,$F8,$03
	.dc.b	$5F,$D7,$BF,$C2,$FE,$1A,$47,$F0,$B6,$BF,$C2,$DA,$FF,$0B,$6B,$FC
	.dc.b	$2D,$AF,$EE,$9A,$FF,$19,$F8,$6F,$E3,$04,$7F,$8C,$35,$FE,$30,$D7
	.dc.b	$F8,$C3,$5F,$E3,$0D,$7F,$83,$BF,$C8,$FE,$1A,$47,$F2,$36,$BF,$C8
	.dc.b	$DA,$FF,$23,$6B,$FC,$8D,$AF,$F1,$26,$BF,$CC,$7E,$1B,$F9,$81,$1F
	.dc.b	$E6,$0D,$7F,$98,$35,$FE,$60,$D7,$F9,$83,$5F,$E3,$EF,$F3,$BF,$86
	.dc.b	$91,$FC,$ED,$AF,$F3,$B6,$BF,$CE,$DA,$FF,$3B,$6B,$FC,$A9,$AF,$F4
	.dc.b	$9F,$37,$FA,$41,$1F,$E9,$0D,$7F,$A4,$35,$FE,$90,$D7,$FA,$43,$5F
	.dc.b	$E6,$EF,$F5,$3F,$3E,$47,$F5,$36,$BF,$D4,$DA,$FF,$53,$6B,$FD,$4D
	.dc.b	$AF,$F4,$26,$BF,$D8,$7D,$1F,$EC,$04,$7F,$B0,$35,$FE,$C0,$D7,$FB
	.dc.b	$03,$5F,$EC,$0D,$7F,$A7,$BF,$DA,$FD,$19,$1F,$DA,$DA,$FF,$6B,$6B
	.dc.b	$FD,$AD,$AF,$F6,$B6,$BF,$D6,$9A,$FF,$79,$F7,$3F,$BC,$11,$FE,$F0
	.dc.b	$D7,$FB,$C3,$5F,$EF,$0D,$7F,$BC,$35,$FE,$CE,$FF,$BE,$FE,$E5,$EC
	.dc.b	$D2,$4F,$F7,$D2,$08,$DB,$83,$FC,$04,$E1,$C3,$5E,$2D,$7A,$35,$E3
	.dc.b	$1E,$07,$8B,$5E,$3E,$DF,$9C,$8F,$84,$F2,$38,$7C,$2D,$78,$9B,$94
	.dc.b	$66,$3C,$6D,$7D,$7E,$DF,$C5,$23,$C5,$AF,$B9,$AF,$B9,$AF,$B9,$AF
	.dc.b	$43,$C5,$EF,$7D,$12,$38,$21,$F2,$9C,$B4,$C9,$F3,$1C,$B4,$F7,$7C
	.dc.b	$07,$DD,$23,$82,$1F,$59,$CB,$4C,$9F,$61,$CB,$4F,$95,$F0,$5F,$89
	.dc.b	$23,$82,$1F,$82,$72,$D3,$27,$E1,$1C,$B4,$FA,$DF,$07,$F9,$52,$38
	.dc.b	$21,$F8,$E7,$2D,$32,$7E,$41,$CB,$4F,$C1,$7C,$2F,$E8,$48,$E0,$87
	.dc.b	$E6,$9C,$B4,$C9,$F9,$C7,$2D,$3F,$1D,$F0,$DF,$AD,$23,$82,$1F,$A6
	.dc.b	$72,$D3,$27,$EA,$1C,$B4,$FC,$D7,$C3,$7E,$E4,$8E,$08,$7E,$C9,$CB
	.dc.b	$4C,$9F,$B4,$72,$D3,$F4,$DF,$0B,$FC,$14,$8E,$08,$7E,$F9,$CB,$4C
	.dc.b	$9F,$C0,$1C,$B4,$FD,$97,$C3,$7F,$11,$23,$82,$1F,$C2,$9C,$B4,$C9
	.dc.b	$FC,$31,$CB,$4F,$DF,$7C,$37,$F1,$D2,$38,$21,$FC,$59,$CB,$4C,$9F
	.dc.b	$C6,$1C,$B4,$FE,$15,$F0,$DF,$CA,$48,$E0,$87,$F2,$27,$2D,$32,$7F
	.dc.b	$24,$72,$D3,$F8,$B7,$C2,$FF,$35,$23,$82,$1F,$CB,$9C,$B4,$C9,$FC
	.dc.b	$C1,$CB,$4F,$E4,$5F,$0B,$FD,$04,$8E,$08,$7F,$3A,$72,$D3,$27,$F3
	.dc.b	$C7,$2D,$3F,$97,$7C,$2F,$F4,$D2,$38,$21,$FD,$19,$CB,$4C,$9F,$D2
	.dc.b	$1C,$B4,$FE,$75,$F0,$7F,$D6,$48,$E0,$87,$F5,$27,$2D,$32,$7F,$54
	.dc.b	$72,$D3,$FA,$37,$C1,$FF,$65,$23,$82,$1F,$D7,$9C,$B4,$C9,$FD,$81
	.dc.b	$CB,$4F,$EA,$5F,$07,$FD,$C4,$8E,$0E,$BF,$87,$7F,$BE,$5F,$F7,$77
	.dc.b	$DD,$BE,$0B,$FB,$A4,$E6,$53,$97,$FD,$E9,$19,$E9,$61,$C3,$CF,$FE
	.dc.b	$E2,$FF,$7D,$F8,$67,$6F,$F7,$4F,$86,$C3,$71,$FF,$7F,$4E,$9F,$EF
	.dc.b	$F5,$A7,$FC,$03,$5F,$E0,$B2,$3F,$DB,$DF,$8B,$8F,$9F,$2F,$34,$A3
	.dc.b	$19,$F9,$63,$A9,$EA,$CB,$C4,$FF,$83,$2F,$AB,$4F,$EF,$87,$F8,$6C
	.dc.b	$86,$0F,$F0,$35,$F8,$2F,$BC,$75,$3E,$6D,$F8,$5F,$F0,$F6,$17,$3F
	.dc.b	$F0,$74,$7F,$87,$8E,$A7,$F8,$78,$CD,$BE,$1A,$92,$3C,$2E,$AF,$B4
	.dc.b	$F8,$F2,$FF,$17,$36,$BE,$09,$F1,$63,$33,$F3,$E3,$EF,$B6,$7E,$5E
	.dc.b	$7F,$56,$7F,$C6,$C6,$67,$C0,$FC,$26,$BF,$E3,$99,$FA,$7A,$7F,$8F
	.dc.b	$9B,$1E,$00,$75,$3C,$26,$23,$84,$F4,$8A,$77,$3E,$8F,$C2,$FC,$62
	.dc.b	$38,$21,$D6,$91,$FE,$51,$AF,$23,$67,$E3,$B5,$FD,$2B,$D1,$F6,$35
	.dc.b	$F6,$7C,$2F,$D6,$23,$F8,$46,$BE,$17,$A3,$FA,$C6,$D1,$A7,$08,$EA
	.dc.b	$73,$3F,$1F,$F0,$BF,$84,$23,$82,$1E,$F7,$A9,$1F,$98,$7E,$11,$E2
	.dc.b	$FD,$76,$BF,$5F,$E1,$7F,$22,$47,$F3,$ED,$7E,$63,$EA,$23,$D8,$FE
	.dc.b	$1A,$ED,$E8,$72,$3F,$53,$F0,$BF,$9E,$23,$82,$1F,$65,$F8,$E4,$7B
	.dc.b	$1F,$C5,$9E,$2F,$E4,$5A,$FE,$47,$E1,$7F,$56,$47,$F7,$2D,$7F,$0C
	.dc.b	$FD,$42,$3D,$B0,$46,$E3,$A1,$C8,$FE,$07,$F0,$BF,$B8,$23,$83,$47
	.dc.b	$EC,$4F,$D0,$FD,$E3,$CB,$7B,$9F,$C2,$65,$39,$8C,$61,$E2,$9B,$87
	.dc.b	$ED,$FE,$1A,$47,$F2,$CD,$FF,$80,$CE,$23,$F6,$93,$FC,$30,$8F,$93
	.dc.b	$9B,$FA,$06,$3D,$E3,$CC,$78,$B2,$F2,$B8,$C7,$C4,$DB,$E6,$F6,$3F
	.dc.b	$92,$BF,$C6,$BE,$1A,$47,$F2,$86,$BF,$EB,$FB,$F0,$61,$E2,$C3,$BB
	.dc.b	$73,$F7,$D7,$C2,$BE,$2F,$F0,$D6,$BF,$C8,$3E,$1B,$F9,$49,$1F,$E6
	.dc.b	$EF,$F3,$0C,$3B,$F2,$FF,$B4,$DB,$3F,$53,$1F,$0C,$FC,$A8,$FA,$35
	.dc.b	$CD,$FE,$3C,$D7,$F8,$F7,$C3,$7F,$38,$23,$FD,$19,$F0,$57,$48,$ED
	.dc.b	$8F,$DF,$C3,$C3,$1F,$DC,$35,$FE,$77,$7C,$A3,$F9,$63,$5F,$E5,$9F
	.dc.b	$0D,$FD,$18,$8F,$F5,$47,$BD,$F5,$C3,$C1,$3F,$6E,$3E,$5B,$F6,$E7
	.dc.b	$8B,$FC,$E5,$AF,$F3,$9F,$86,$FE,$A6,$47,$FB,$0B,$D9,$FC,$3E,$3E
	.dc.b	$A3,$D7,$BF,$E1,$B6,$FE,$ED,$5C,$6B,$FA,$33,$5F,$E8,$DF,$0D,$FD
	.dc.b	$80,$8F,$F6,$E7,$B4,$FE,$0E,$3B,$E5,$EC,$BE,$DB,$7F,$60,$C3,$FB
	.dc.b	$1B,$2B,$F5,$31,$FD,$9B,$E1,$BF,$B6,$92,$3F,$BD,$BD,$A7,$F0,$97
	.dc.b	$F2,$32,$F5,$ED,$CE,$FF,$45,$57,$2F,$6C,$7E,$DA,$3F,$BC,$91,$FF
	.dc.b	$03,$BD,$5F,$C0,$9E,$AD,$7E,$9C,$FF,$86,$E7,$FA,$29,$27,$F8,$93
	.dc.b	$FD,$23,$E1,$BF,$C0,$A4,$8F,$F0,$9B,$D5,$FB,$C7,$53,$F7,$17,$FE
	.dc.b	$06,$8E,$51,$FC,$75,$AF,$F6,$7F,$86,$FF,$08,$91,$FF,$0F,$BD,$71
	.dc.b	$FB,$D9,$7D,$1B,$FF,$08,$CF,$F5,$C2,$57,$F2,$97,$FA,$FE,$5F,$E1
	.dc.b	$C2,$47,$F8,$A9,$E9,$5F,$3C,$E9,$1F,$C8,$23,$8B,$FC,$CD,$AF,$F8
	.dc.b	$26,$DF,$0D,$FB,$D2,$47,$F8,$38,$87,$F2,$C7,$BC,$FE,$94,$FA,$39
	.dc.b	$7F,$03,$8F,$EE,$F5,$D2,$BF,$A7,$B5,$F0,$46,$BE,$1B,$F8,$61,$23
	.dc.b	$B9,$FC,$F9,$EE,$FD,$3D,$3F,$CB,$19,$FF,$91,$6D,$5C,$EB,$BB,$5F
	.dc.b	$77,$C3,$7F,$19,$23,$D8,$FE,$8E,$F6,$BF,$D3,$1D,$A3,$F8,$74,$7B
	.dc.b	$08,$FB,$1F,$7F,$C3,$7F,$25,$24,$75,$B4,$75,$CB,$F4,$B9,$FC,$8A
	.dc.b	$FE,$B7,$7E,$BD,$3E,$D4,$F8,$27,$F8,$F7,$C3,$7F,$31,$23,$D5,$E8
	.dc.b	$FE,$93,$2F,$43,$F1,$F0,$EE,$D3,$B9,$FC,$A7,$E1,$BF,$9E,$11,$F4
	.dc.b	$3D,$1F,$D1,$CF,$43,$E2,$CF,$46,$C7,$43,$F8,$DF,$C3,$7F,$47,$23
	.dc.b	$F0,$DE,$9A,$7F,$04,$8F,$C5,$1E,$53,$BD,$7F,$4D,$CF,$EF,$7C,$37
	.dc.b	$F5,$12,$4F,$CD,$79,$E9,$FA,$08,$E5,$7F,$AE,$D3,$E5,$47,$F2,$EF
	.dc.b	$86,$FE,$B4,$47,$EB,$3C,$A3,$EA,$47,$88,$F9,$98,$FB,$CF,$23,$FD
	.dc.b	$47,$E1,$BF,$3E,$47,$F3,$EF,$E8,$B0,$FB,$71,$ED,$E7,$FE,$5E,$F4
	.dc.b	$21,$DC,$FC,$99,$FD,$77,$E1,$BF,$56,$47,$F5,$6F,$5C,$3E,$4E,$DF
	.dc.b	$4A,$7F,$B1,$18,$C7,$D0,$F6,$CD,$AF,$EA,$7E,$1B,$F6,$E4,$9F,$DB
	.dc.b	$BD,$5F,$97,$9F,$CD,$C3,$B5,$FC,$D9,$FE,$B3,$7F,$B3,$F8,$6F,$E0
	.dc.b	$24,$8F,$E0,$2F,$3A,$E3,$7C,$F1,$F2,$D7,$C1,$C3,$FB,$16,$1F,$BD
	.dc.b	$F8,$6F,$E1,$64,$9F,$E1,$6F,$3A,$E3,$5F,$8B,$9F,$3E,$7F,$05,$FC
	.dc.b	$A6,$3F,$E3,$6F,$86,$FE,$2A,$48,$FE,$2A,$F3,$7E,$36,$5E,$56,$D7
	.dc.b	$37,$EA,$9F,$DE,$3E,$1B,$F8,$F9,$1F,$E3,$EF,$37,$D9,$3C,$DB,$3C
	.dc.b	$F0,$FA,$71,$FA,$23,$F9,$7B,$FE,$4B,$6C,$FF,$92,$9B,$3F,$C3,$63
	.dc.b	$11,$FB,$E3,$27,$F9,$15,$CB,$FD,$A0,$C9,$0E,$04,$34,$53,$FB,$19
	.dc.b	$B3,$24,$32,$6E,$7D,$0D,$8F,$0B,$FE,$74,$AF,$8C,$FF,$51,$6F,$89
	.dc.b	$E8,$C7,$F0,$B8,$7E,$CB,$0F,$C7,$47,$E4,$34,$FE,$81,$97,$D9,$C7
	.dc.b	$FA,$D3,$70,$E5,$9F,$E1,$33,$FA,$18,$F2,$8F,$C3,$E1,$F8,$DC,$3F
	.dc.b	$16,$43,$FB,$26,$9F,$66,$B8,$47,$FB,$C3,$83,$F7,$2F,$DA,$3F,$87
	.dc.b	$D7,$E8,$5F,$CF,$90,$FE,$DD,$B7,$3C,$7F,$81,$E9,$FC,$AF,$2F,$36
	.dc.b	$BF,$83,$C3,$F6,$98,$7E,$C8,$87,$F7,$7D,$F9,$4F,$E4,$F2,$F1,$E7
	.dc.b	$E6,$AF,$D5,$5F,$F6,$5E,$0F,$E1,$11,$FD,$FD,$BB,$7C,$68,$FC,$E9
	.dc.b	$FE,$1E,$3F,$29,$A7,$9B,$3F,$E3,$F1,$FE,$56,$E1,$FE,$3C,$FF,$82
	.dc.b	$5B,$5C,$E3,$E2,$6B,$ED,$7A,$BF,$65,$FF,$3C,$71,$7F,$31,$7F,$D3
	.dc.b	$4D,$79,$67,$EB,$D3,$DB,$97,$8E,$7F,$8D,$69,$FD,$11,$FE,$84,$43
	.dc.b	$FD,$49,$C7,$FC,$ED,$F6,$1F,$C1,$0F,$AE,$7F,$A2,$79,$3F,$AA,$BF
	.dc.b	$EA,$B6,$BE,$2A,$F0,$3F,$CB,$0E,$59,$7A,$DF,$F3,$CF,$2F,$F6,$97
	.dc.b	$FD,$68,$D7,$97,$07,$D7,$E9,$FE,$FC,$FC,$E3,$FA,$AD,$7F,$B2,$B9
	.dc.b	$FF,$BE,$D7,$FA,$F5,$AF,$23,$FA,$55,$72,$7C,$7C,$3F,$5A,$BF,$DA
	.dc.b	$7D,$1F,$E1,$29,$FF,$63,$35,$E5,$1F,$D1,$F6,$FC,$4E,$DC,$E3,$D8
	.dc.b	$7F,$B7,$1F,$F1,$3C,$FF,$B3,$DA,$F2,$BF,$E7,$35,$CA,$3F,$C4,$84
	.dc.b	$3F,$C6,$27,$F8,$D0,$FF,$12,$74,$FF,$8F,$DF,$F6,$D3,$F5,$EF,$F1
	.dc.b	$11,$FE,$FA,$8F,$F1,$D7,$17,$F2,$46,$C7,$F9,$3C,$FF,$17,$75,$7F
	.dc.b	$96,$1F,$F7,$23,$CF,$0F,$F6,$94,$7F,$AB,$A3,$FC,$AA,$47,$FC,$AA
	.dc.b	$FF,$99,$1F,$F2,$FE,$1F,$C6,$13,$4C,$71,$7E,$BC,$FE,$A1,$8A,$FF
	.dc.b	$CD,$13,$FC,$DF,$1E,$7B,$7F,$9C,$1B,$87,$F9,$F4,$FF,$42,$90,$FF
	.dc.b	$30,$CB,$AF,$D5,$1F,$C4,$76,$FF,$43,$BC,$4F,$F4,$81,$6B,$D3,$D9
	.dc.b	$FE,$9D,$8E,$6D,$7E,$C1,$FE,$10,$BF,$E3,$62,$3F,$D3,$73,$CA,$7F
	.dc.b	$D3,$C7,$FA,$A8,$FF,$59,$90,$F3,$E3,$E6,$7F,$4D,$97,$FA,$CD,$E0
	.dc.b	$27,$E6,$CF,$FA,$D6,$30,$7C,$8E,$DF,$F6,$63,$F0,$9A,$FA,$0F,$F1
	.dc.b	$36,$1E,$89,$FF,$65,$1F,$40,$FF,$4C,$5F,$FB,$32,$4F,$D5,$7F,$9D
	.dc.b	$77,$FF,$B9,$9F,$9E,$D7,$B6,$BF,$72,$BF,$DC,$91,$FE,$D5,$21,$D1
	.dc.b	$CB,$4F,$E8,$DD,$FF,$EF,$57,$27,$E3,$1F,$8D,$87,$D0,$75,$1C,$0F
	.dc.b	$F4,$A4,$23,$FD,$2D,$1F,$92,$F0,$2C,$9A,$48,$FF,$02,$6F,$FC,$35
	.dc.b	$FE,$43,$97,$C1,$79,$1D,$A7,$83,$F9,$07,$AF,$83,$23,$C6,$D7,$FC
	.dc.b	$19,$5D,$6B,$F6,$F6,$AD,$57,$EA,$9A,$FE,$93,$C2,$F4,$91,$F7,$BF
	.dc.b	$13,$83,$EC,$E7,$B9,$C9,$0F,$D2,$CF,$C2,$F0,$E4,$7C,$86,$BE,$63
	.dc.b	$F9,$29,$FA,$90,$7D,$73,$F7,$AF,$07,$F8,$0B,$5F,$DD,$F8,$9F,$40
	.dc.b	$8F,$D6,$7E,$9D,$FF,$29,$BF,$E4,$6F,$AA,$7F,$C1,$13,$C1,$FD,$EB
	.dc.b	$5F,$E1,$FE,$2F,$DC,$23,$F8,$27,$AB,$F3,$EB,$F8,$DE,$7F,$EB,$EF
	.dc.b	$33,$F5,$33,$FC,$5A,$FF,$DF,$11,$8D,$11,$8E,$3F,$E4,$73,$9C,$BA
	.dc.b	$DF,$F1,$06,$BF,$C8,$9F,$8E,$F6,$3F,$B1,$4E,$5F,$96,$D7,$56,$9E
	.dc.b	$71,$FC,$C8,$D1,$FC,$BB,$C7,$FC,$D9,$1F,$D0,$B5,$E6,$F8,$AB,$84
	.dc.b	$78,$0F,$E5,$F1,$96,$9C,$88,$7F,$2C,$7F,$53,$E4,$48,$FD,$7B,$5F
	.dc.b	$E6,$2F,$37,$FC,$80,$D7,$FA,$1B,$6C,$D3,$94,$7F,$4A,$6B,$FD,$13
	.dc.b	$C9,$FD,$C9,$1F,$DF,$B5,$F9,$2F,$52,$3E,$F3,$FC,$A4,$ED,$FE,$AE
	.dc.b	$D7,$FA,$8F,$95,$FC,$18,$8F,$F0,$C6,$BF,$59,$F5,$11,$EC,$7F,$60
	.dc.b	$76,$FF,$64,$6B,$FD,$7B,$CB,$FE,$26,$47,$F8,$CB,$5F,$C1,$BF,$1C
	.dc.b	$8F,$6B,$46,$E3,$FB,$8B,$5F,$ED,$7E,$67,$F2,$02,$3F,$C9,$9A,$FE
	.dc.b	$41,$FA,$84,$7B,$3A,$7F,$BE,$B5,$FE,$EF,$E6,$FF,$2C,$23,$FC,$C9
	.dc.b	$EB,$BF,$F2,$9B,$57,$EF,$6B,$FC,$0A,$DF,$39,$E7,$E6,$B7,$5F,$C4
	.dc.b	$38,$3F,$69,$7F,$B3,$8F,$F4,$63,$55,$FE,$C1,$4F,$EF,$A3,$FA,$1C
	.dc.b	$68,$FE,$8A,$E0,$CD,$78,$23,$F8,$F3,$70,$F5,$5F,$9F,$0F,$45,$72
	.dc.b	$21,$E1,$3F,$93,$DF,$5F,$43,$B9,$23,$47,$A1,$F1,$CF,$C1,$A9,$3F
	.dc.b	$CC,$2F,$FA,$95,$FC,$68,$C7,$1F,$F0,$E8,$61,$D6,$7C,$51,$87,$BB
	.dc.b	$5C,$FA,$3F,$08,$8E,$9F,$1B,$D8,$F9,$E5,$3E,$C9,$63,$C8,$FF,$45
	.dc.b	$6E,$7F,$46,$BE,$B4,$78,$63,$EA,$65,$83,$BD,$7A,$1A,$FD,$B9,$EA
	.dc.b	$D8,$CB,$F0,$5A,$FD,$06,$BF,$84,$7E,$FF,$A5,$23,$E1,$B5,$F8,$ED
	.dc.b	$7C,$0D,$7E,$73,$5F,$BA,$D7,$F2,$3E,$9F,$E3,$08,$FD,$06,$BF,$51
	.dc.b	$AF,$B5,$AF,$DB,$6B,$F8,$A7,$F2,$FE,$A4,$8F,$B6,$D7,$EF,$B5,$F9
	.dc.b	$0D,$7F,$0C,$D7,$F2,$AD,$7F,$53,$EA,$FE,$90,$8F,$E2,$1A,$FE,$31
	.dc.b	$AF,$D5,$6B,$F9,$36,$BF,$A2,$7F,$5F,$EB,$48,$FC,$9B,$5F,$CB,$B5
	.dc.b	$FC,$03,$5F,$CF,$35,$FD,$6B,$5F,$E0,$3E,$BF,$EF,$08,$FE,$81,$AF
	.dc.b	$E9,$1A,$FE,$35,$AF,$EA,$DA,$FE,$E9,$FE,$0F,$EC,$48,$FD,$5B,$5F
	.dc.b	$D7,$B5,$FC,$C3,$5F,$DB,$35,$FE,$0A,$D7,$F8,$CF,$B3,$FC,$50,$8F
	.dc.b	$EE,$1A,$FE,$F1,$AF,$E9,$5A,$FF,$03,$6B,$FC,$49,$FE,$3F,$ED,$48
	.dc.b	$FE,$06,$D7,$F8,$3B,$5F,$D8,$35,$FE,$1C,$D7,$F8,$EB,$5F,$E6,$3E
	.dc.b	$DF,$F2,$C2,$3F,$C4,$1A,$FF,$14,$6B,$FB,$D6,$BF,$C6,$DA,$FF,$2A
	.dc.b	$7F,$9B,$FB,$92,$3F,$8D,$B5,$FE,$3E,$D7,$F8,$43,$5F,$E4,$CD,$7F
	.dc.b	$9A,$B5,$FE,$93,$EE,$FF,$44,$23,$FC,$A1,$AF,$F2,$C6,$BF,$C5,$5A
	.dc.b	$FF,$33,$6B,$FD,$09,$FE,$9F,$EF,$48,$FE,$66,$D7,$F9,$BB,$5F,$E4
	.dc.b	$0D,$7F,$9F,$35,$FE,$9A,$D7,$FB,$0F,$BF,$FD,$70,$8F,$F4,$06,$BF
	.dc.b	$D1,$1A,$FF,$2D,$6B,$FD,$2D,$AF,$F5,$A7,$FB,$3F,$C0,$48,$FE,$96
	.dc.b	$D7,$FA,$7B,$5F,$E7,$0D,$7F,$AB,$35,$FE,$CA,$D7,$FB,$CF,$C0,$FF
	.dc.b	$74,$23,$FD,$61,$AF,$F5,$C6,$BF,$D1,$5A,$FF,$63,$6B,$FD,$C9,$FE
	.dc.b	$FF,$F0,$52,$3F,$B1,$B5,$FE,$CE,$D7,$FA,$83,$5F,$ED,$CD,$7F,$BE
	.dc.b	$B5,$FF,$08,$FC,$1F,$F8,$30,$8F,$F7,$06,$BF,$DD,$1A,$FF,$5D,$6B
	.dc.b	$FD,$ED,$AF,$F8,$29,$FF,$6D,$F5,$7F,$A2,$E3,$FD,$1A,$D7,$FD,$6A
	.dc.b	$D4,$D2,$78,$9C,$89,$1F,$E1,$28,$CB,$63,$2F,$F8,$6E,$1F,$09,$E2
	.dc.b	$24,$71,$6B,$B6,$BF,$E3,$38,$E2,$EC,$86,$CA,$6F,$E1,$7D,$E4,$7C
	.dc.b	$CD,$7C,$8D,$78,$1A,$21,$A2,$9E,$3F,$86,$F9,$64,$7E,$13,$5F,$80
	.dc.b	$D7,$81,$A2,$1A,$29,$EF,$F7,$3E,$B9,$1F,$9C,$D7,$E6,$35,$E0,$68
	.dc.b	$86,$8A,$7C,$BF,$86,$FC,$19,$1F,$B4,$D7,$EC,$35,$E0,$68,$86,$8A
	.dc.b	$7D,$7F,$86,$FC,$79,$1F,$C3,$35,$FC,$23,$5E,$06,$88,$68,$A7,$E0
	.dc.b	$FE,$1B,$F3,$64,$7F,$24,$D7,$F2,$0D,$78,$1A,$21,$A2,$9F,$8F,$F8
	.dc.b	$6F,$D3,$91,$FC,$F3,$5F,$CE,$35,$E0,$68,$86,$8A,$7E,$6F,$E1,$BF
	.dc.b	$66,$47,$F5,$4D,$7F,$50,$D7,$81,$A2,$1A,$29,$FA,$7F,$86,$FD,$F9
	.dc.b	$1F,$DB,$35,$FD,$A3,$5E,$06,$88,$68,$A7,$EC,$FE,$1B,$F8,$59,$1F
	.dc.b	$E0,$4D,$7F,$80,$35,$E0,$68,$86,$8A,$7E,$FF,$E1,$BF,$8B,$91,$FE
	.dc.b	$1C,$D7,$F8,$63,$5E,$06,$88,$68,$A7,$F0,$BF,$86,$FE,$46,$47,$F8
	.dc.b	$D3,$5F,$E3,$0D,$78,$1A,$21,$A2,$9F,$C5,$FE,$1B,$F9,$79,$1F,$E4
	.dc.b	$CD,$7F,$92,$35,$E0,$68,$86,$8A,$7F,$23,$F8,$6F,$E7,$64,$7F,$99
	.dc.b	$35,$FE,$60,$D7,$81,$A2,$1A,$29,$FC,$BF,$E1,$BF,$A3,$91,$FE,$7C
	.dc.b	$D7,$F9,$E3,$5E,$06,$88,$68,$A7,$F3,$BF,$86,$FE,$A6,$47,$FA,$53
	.dc.b	$5F,$E9,$0D,$78,$1A,$21,$A2,$9F,$D1,$FE,$1B,$FA,$F9,$1F,$EA,$CD
	.dc.b	$7F,$AA,$35,$E0,$68,$86,$8A,$7F,$53,$F8,$6F,$ED,$64,$7F,$B1,$35
	.dc.b	$FE,$C0,$D7,$81,$A2,$1A,$29,$FD,$7F,$E1,$BF,$BB,$91,$FE,$DC,$D7
	.dc.b	$FB,$63,$5E,$06,$88,$68,$A0,$83,$E1,$DF,$01,$88,$91,$87,$07,$17
	.dc.b	$0E,$1A,$F0,$8D,$BB,$6B,$DD,$B1,$C2,$3C,$6D,$78,$1C,$9C,$3C,$4C
	.dc.b	$7C,$3F,$22,$38,$21,$E5,$72,$D3,$27,$B9,$CB,$4F,$07,$C3,$F9,$88
	.dc.b	$E0,$87,$C1,$72,$D3,$27,$CA,$72,$D3,$D9,$D5,$F0,$88,$E0,$87,$CD
	.dc.b	$72,$D3,$27,$D6,$72,$D3,$E4,$7C,$3F,$CE,$23,$82,$1F,$65,$CB,$4C
	.dc.b	$9F,$82,$72,$D3,$EA,$7C,$3F,$DA,$23,$82,$1F,$85,$72,$D3,$27,$E3
	.dc.b	$9C,$B4,$FC,$07,$C3,$7E,$18,$8E,$08,$7E,$45,$CB,$4C,$9F,$9A,$72
	.dc.b	$D3,$F1,$9F,$0D,$F9,$22,$38,$21,$F9,$D7,$2D,$32,$7E,$99,$CB,$4F
	.dc.b	$CC,$7C,$37,$E7,$88,$E0,$87,$EA,$5C,$B4,$C9,$FB,$27,$2D,$3F,$49
	.dc.b	$F0,$DF,$AA,$23,$82,$1F,$B5,$72,$D3,$27,$EF,$9C,$B4,$FD,$87,$C3
	.dc.b	$7E,$D8,$8E,$08,$7F,$01,$72,$D3,$27,$F0,$A7,$2D,$3F,$79,$F0,$BF
	.dc.b	$C0,$88,$E0,$87,$F0,$D7,$2D,$32,$7F,$16,$72,$D3,$F8,$47,$C3,$7F
	.dc.b	$0E,$23,$82,$1F,$C6,$5C,$B4,$C9,$FC,$89,$CB,$4F,$E2,$9F,$0D,$FC
	.dc.b	$68,$8E,$08,$7F,$25,$72,$D3,$27,$F2,$E7,$2D,$3F,$90,$7C,$37,$F2
	.dc.b	$62,$38,$21,$FC,$C5,$CB,$4C,$9F,$CE,$9C,$B4,$FE,$59,$F0,$DF,$CC
	.dc.b	$88,$E0,$87,$F3,$D7,$2D,$32,$7F,$46,$72,$D3,$F9,$C7,$C3,$7F,$3E
	.dc.b	$23,$82,$1F,$D2,$5C,$B4,$C9,$FD,$49,$CB,$4F,$E8,$9F,$0D,$FD,$28
	.dc.b	$8E,$08,$7F,$55,$72,$D3,$27,$F5,$E7,$2D,$3F,$A8,$7C,$37,$F5,$62
	.dc.b	$38,$21,$FD,$85,$CB,$4C,$9F,$DA,$9C,$B4,$FE,$B9,$F0,$9F,$D8,$88
	.dc.b	$E0,$87,$F6,$D7,$2D,$32,$7F,$76,$72,$D3,$FB,$47,$C3,$7F,$6E,$23
	.dc.b	$82,$1F,$DE,$5C,$B4,$C9,$FE,$02,$72,$D3,$FB,$EF,$C4,$FF,$7C,$4D
	.dc.b	$23,$FE,$00,$6B,$C0,$C1,$0C,$10,$FF,$07,$BC,$FE,$13,$91,$23,$C2
	.dc.b	$D7,$6D,$7F,$C3,$50,$8C,$47,$F8,$82,$30,$F2,$21,$E0,$F8,$5F,$41
	.dc.b	$1F,$7B,$5D,$18,$21,$82,$1E,$27,$83,$E4,$F8,$3F,$88,$47,$E5,$B5
	.dc.b	$F8,$2D,$78,$1A,$21,$A2,$9F,$03,$E0,$FE,$81,$1F,$AE,$D7,$E6,$B5
	.dc.b	$E0,$68,$86,$8A,$7C,$CF,$83,$FB,$84,$7F,$06,$FF,$8F,$BE,$25,$E9
	.dc.b	$FF,$93,$BE,$19,$EA,$FF,$94,$25,$EA,$E6,$30,$DC,$3F,$CA,$D3,$C9
	.dc.b	$FF,$2C,$57,$27,$3E,$AE,$D4,$A3,$0D,$8E,$37,$E1,$F5,$78,$6C,$F8
	.dc.b	$5F,$CB,$12,$BF,$3A,$FA,$F3,$E7,$9C,$9E,$67,$9A,$3B,$E5,$23,$27
	.dc.b	$91,$F1,$E1,$E0,$F8,$5F,$D2,$12,$7F,$56,$D7,$F4,$2D,$7F,$36,$D7
	.dc.b	$F3,$4D,$7F,$CF,$98,$A3,$F6,$1F,$0B,$27,$F6,$ED,$7F,$CF,$7F,$0A
	.dc.b	$F7,$3E,$52,$F7,$29,$88,$CF,$B9,$2B,$DB,$48,$F9,$FB,$FC,$C8,$FE
	.dc.b	$00,$49,$FE,$0E,$FF,$A0,$8F,$03,$CA,$FB,$BD,$2F,$B2,$E1,$FC,$02
	.dc.b	$AF,$03,$E2,$C3,$C2,$FD,$0C,$DF,$0B,$FC,$44,$91,$FC,$61,$FE,$29
	.dc.b	$F0,$BF,$03,$E0,$24,$F1,$8F,$C2,$9F,$13,$E0,$24,$64,$87,$F1,$68
	.dc.b	$E6,$FF,$24,$6B,$93,$04,$30,$43,$F8,$BB,$C1,$FE,$33,$F0,$BF,$CA
	.dc.b	$48,$FF,$30,$7A,$E7,$F1,$31,$57,$A2,$BE,$AF,$C1,$D7,$E5,$6F,$FC
	.dc.b	$86,$71,$7F,$33,$57,$C1,$FE,$93,$E1,$7F,$9C,$13,$0F,$E7,$EF,$D9
	.dc.b	$BF,$C3,$E7,$27,$F0,$71,$F8,$A9,$91,$F8,$67,$F1,$BF,$0B,$FB,$03
	.dc.b	$FA,$21,$23,$FA,$5B,$D5,$FC,$6D,$75,$7D,$27,$FB,$67,$0F,$7B,$D4
	.dc.b	$86,$8F,$C3,$57,$81,$FC,$26,$7F,$D4,$C8,$FF,$5A,$7F,$29,$C9,$FA
	.dc.b	$22,$47,$E7,$B9,$3F,$4B,$7F,$A6,$E1,$93,$FA,$28,$FE,$17,$A7,$F5
	.dc.b	$F2,$47,$F6,$67,$AB,$FA,$8B,$EA,$F9,$E5,$7D,$A3,$F5,$B7,$D9,$D4
	.dc.b	$FF,$18,$E1,$FE,$D8,$48,$FE,$E6,$FF,$6F,$E1,$FD,$DF,$5C,$9F,$D2
	.dc.b	$47,$F2,$03,$F7,$FD,$52,$32,$43,$FB,$8C,$73,$7F,$BF,$B5,$C9,$82
	.dc.b	$18,$21,$FD,$C9,$E0,$FF,$74,$E5,$FF,$02,$11,$FF,$05,$BD,$78,$BF
	.dc.b	$39,$3E,$79,$FE,$1F,$DB,$5F,$BD,$CA,$9C,$57,$E6,$FB,$7F,$9E,$F4
	.dc.b	$7F,$84,$49,$7F,$E1,$84,$C2,$0F,$8A,$78,$3F,$D3,$5B,$1F,$CF,$1A
	.dc.b	$FD,$93,$F9,$4F,$83,$FC,$AB,$C1,$91,$FC,$9A,$3F,$A7,$F6,$7F,$88
	.dc.b	$C9,$1F,$E2,$A7,$C2,$F8,$EB,$AE,$5F,$C8,$1B,$D9,$9C,$7F,$9B,$D7
	.dc.b	$80,$FE,$67,$8F,$81,$FC,$66,$3F,$D9,$7E,$2F,$F9,$4C,$7B,$9A,$FC
	.dc.b	$16,$BF,$11,$AF,$C3,$F1,$BE,$17,$8D,$23,$E1,$47,$C5,$F1,$A4,$7C
	.dc.b	$57,$E4,$78,$DE,$1F,$13,$DB,$3D,$5A,$B0,$FE,$FC,$FC,$26,$BF,$53
	.dc.b	$56,$1E,$18,$FD,$F4,$31,$FA,$D3,$E2,$9C,$E3,$76,$F9,$59,$7C,$3F
	.dc.b	$17,$C3,$97,$F8,$4A,$FC,$AD,$7C,$51,$AF,$17,$F0,$5E,$44,$9F,$EF
	.dc.b	$B5,$FE,$0D,$7F,$88,$37,$0F,$B1,$E4,$7E,$24,$FA,$57,$2B,$BB,$D5
	.dc.b	$A4,$3E,$13,$96,$99,$3E,$D3,$96,$9E,$AF,$1F,$DA,$47,$07,$5A,$FE
	.dc.b	$D7,$7E,$7E,$3F,$C1,$5F,$E7,$FC,$AF,$35,$FB,$22,$DF,$E4,$FC,$AF
	.dc.b	$C6,$7C,$3F,$CA,$25,$7C,$A7,$EE,$F1,$3C,$FF,$D5,$12,$7F,$55,$E7
	.dc.b	$FE,$B7,$CF,$FD,$77,$9F,$27,$F5,$D1,$F9,$9F,$87,$FA,$E4,$8F,$AE
	.dc.b	$F6,$BE,$91,$F8,$47,$F7,$17,$D6,$3F,$71,$7D,$5F,$C4,$DF,$E9,$7E
	.dc.b	$1F,$F0,$64,$8F,$C1,$BE,$AF,$57,$F8,$17,$AB,$27,$F8,$14,$7F,$05
	.dc.b	$F5,$64,$7F,$05,$7F,$84,$7A,$BF,$B1,$F8,$7F,$C7,$92,$7F,$1E,$F5
	.dc.b	$7F,$86,$5F,$57,$F3,$B7,$D1,$FC,$FD,$DE,$1F,$E0,$98,$7E,$F7,$E1
	.dc.b	$FF,$36,$4A,$FC,$DB,$F9,$EF,$87,$FE,$35,$ED,$49,$FC,$CC,$7F,$05
	.dc.b	$3F,$8F,$7B,$52,$32,$43,$F6,$91,$CD,$FD,$3B,$5C,$98,$21,$82,$1F
	.dc.b	$C4,$5E,$0F,$F1,$3F,$87,$FD,$99,$1F,$D9,$BD,$71,$FC,$BC,$F9,$E7
	.dc.b	$F9,$97,$BD,$5F,$D8,$C5,$9F,$E5,$1E,$EF,$F2,$1F,$87,$FD,$F9,$2B
	.dc.b	$F7,$EF,$F1,$2F,$8C,$7C,$27,$FA,$2D,$7C,$25,$31,$19,$F8,$49,$7F
	.dc.b	$E7,$EA,$47,$1F,$84,$FE,$61,$F1,$1F,$C3,$89,$5F,$C3,$9F,$05,$F5
	.dc.b	$8E,$58,$76,$7A,$5F,$67,$FC,$39,$87,$67,$C1,$87,$F3,$DF,$88,$FE
	.dc.b	$36,$48,$FE,$36,$FB,$71,$F6,$12,$3D,$11,$EF,$F8,$8F,$81,$F1,$12
	.dc.b	$7D,$91,$FD,$24,$FE,$4E,$47,$F9,$3B,$D5,$F8,$55,$D5,$F3,$D7,$57
	.dc.b	$E3,$57,$57,$D7,$5F,$D5,$63,$F9,$99,$23,$F9,$9B,$FC,$DB,$3F,$A9
	.dc.b	$C3,$27,$F9,$8C,$7F,$5D,$3E,$C6,$B2,$32,$43,$FA,$64,$73,$7F,$9F
	.dc.b	$B5,$C9,$82,$18,$21,$FD,$A5,$E0,$FF,$6B,$D7,$FA,$59,$1F,$E9,$6F
	.dc.b	$5D,$FF,$97,$CF,$9E,$7F,$11,$C9,$5F,$E9,$31,$67,$FD,$3F,$B3,$87
	.dc.b	$FB,$D7,$07,$F5,$82,$5F,$F5,$87,$F3,$5E,$0F,$E6,$7C,$09,$3F,$95
	.dc.b	$8F,$CE,$76,$48,$FC,$C3,$F9,$FE,$CF,$F0,$37,$27,$F6,$42,$4F,$F6
	.dc.b	$47,$AB,$F9,$FB,$EA,$FE,$52,$FA,$3F,$96,$BB,$9F,$AB,$0F,$F7,$B7
	.dc.b	$86,$E7,$FF,$00,$B6,$FF,$C0,$AD,$7F,$C4,$6D,$7F,$C6,$BA,$FF,$81
	.dc.b	$AF,$E2,$F9,$FF,$B7,$3F,$71,$E5,$FE,$E6,$FF,$75,$E4,$CC,$3F,$6D
	.dc.b	$1E,$2E,$AF,$09,$23,$C2,$FF,$03,$BE,$91,$FB,$D7,$AB,$C8,$FE,$01
	.dc.b	$5D,$5F,$DA,$D7,$BB,$B3,$DA,$48,$F6,$BF,$03,$B3,$FC,$A1,$97,$FB
	.dc.b	$73,$E3,$7F,$DB,$75,$FE,$99,$6C,$7F,$B8,$2D,$5F,$28,$FE,$2D,$E4
	.dc.b	$CB,$FF,$2D,$D6,$E7,$51,$F3,$1F,$03,$48,$7C,$E7,$2D,$32,$7F,$99
	.dc.b	$2B,$C2,$E0,$FF,$34,$38,$7E,$BF,$83,$F7,$08,$FD,$C7,$AD,$79,$63
	.dc.b	$CF,$7C,$AF,$E1,$BF,$E7,$3C,$1E,$D7,$AA,$BF,$D7,$98,$F7,$C3,$94
	.dc.b	$7C,$7F,$5B,$F1,$BE,$27,$E3,$09,$1F,$8C,$7F,$C9,$F8,$3E,$0F,$BC
	.dc.b	$FF,$A0,$A3,$FD,$C2,$F9,$0C,$7C,$1E,$94,$71,$74,$63,$DF,$D4,$1A
	.dc.b	$63,$DB,$FD,$0F,$91,$FA,$02,$57,$E8,$1F,$2E,$1C,$B0,$93,$DB,$0F
	.dc.b	$03,$F8,$5C,$3A,$B9,$7B,$3E,$4D,$1E,$47,$EB,$89,$1F,$AE,$7E,$1F
	.dc.b	$C3,$FA,$09,$1D,$FE,$1B,$D7,$F0,$DE,$CF,$87,$93,$E3,$8F,$DD,$FA
	.dc.b	$1F,$BA,$24,$7E,$E9,$EA,$FA,$6F,$AB,$F9,$DB,$EA,$FB,$6F,$AB,$FA
	.dc.b	$1B,$FE,$0F,$E8,$7F,$06,$24,$7F,$06,$7F,$84,$F9,$FF,$43,$E1,$24
	.dc.b	$FC,$18,$FE,$1E,$7D,$4F,$85,$91,$92,$1F,$C3,$A3,$9B,$FC,$49,$AE
	.dc.b	$4C,$10,$C1,$0F,$D3,$3C,$1F,$D4,$7A,$5F,$C7,$88,$FF,$1E,$7A,$F3
	.dc.b	$FE,$F6,$7C,$F3,$F7,$FE,$1E,$BF,$A0,$8B,$3F,$C1,$FE,$1F,$F9,$5F
	.dc.b	$AF,$FC,$A8,$95,$FC,$A9,$FC,$FF,$C2,$FE,$53,$E1,$24,$FE,$2E,$3F
	.dc.b	$2F,$F0,$F2,$3F,$1C,$FE,$6B,$E1,$FF,$9B,$FB,$7F,$CD,$89,$3F,$CD
	.dc.b	$9E,$AF,$E5,$6F,$AB,$FC,$66,$FA,$3F,$C6,$EE,$E5,$FC,$7F,$0F,$E8
	.dc.b	$7E,$F7,$F4,$22,$57,$F4,$27,$FD,$8B,$E6,$7F,$B8,$23,$FD,$EE,$D4
	.dc.b	$DE,$BF,$F7,$CD,$FF,$2B,$F8,$3F,$D9,$FC,$24,$BF,$D5,$47,$F5,$2F
	.dc.b	$80,$FE,$A2,$48,$FE,$A2,$F8,$3E,$17,$F8,$06,$3D,$CE,$D3,$FB,$EA
	.dc.b	$E8,$FF,$37,$BC,$3F,$B1,$F8,$4F,$EB,$A4,$7F,$AE,$BF,$D8,$3E,$17
	.dc.b	$AF,$C2,$C9,$FD,$C4,$7F,$66,$3C,$3F,$0B,$23,$24,$3F,$B2,$C7,$37
	.dc.b	$FB,$4B,$5C,$98,$21,$82,$1F,$CF,$5E,$0F,$F3,$F9,$FE,$EA,$47,$FB
	.dc.b	$AB,$D7,$4F,$EA,$93,$E7,$9F,$E4,$79,$7F,$BD,$9F,$34,$E9,$BE,$59
	.dc.b	$FB,$75,$FE,$04,$C3,$FC,$06,$48,$FF,$01,$BF,$82,$E5,$FA,$64,$8E
	.dc.b	$FC,$BF,$5B,$87,$EB,$F0,$49,$F1,$C7,$F8,$43,$83,$FC,$1E,$48,$FF
	.dc.b	$07,$BD,$5F,$4D,$F5,$59,$B3,$F4,$47,$E4,$31,$F0,$3F,$03,$1F,$03
	.dc.b	$C2,$77,$A3,$2F,$E9,$5F,$1D,$CA,$A0,$6B,$D9,$AF,$C7,$6B,$FA,$0F
	.dc.b	$0B,$F2,$BD,$D2,$3E,$8C,$7E,$63,$AA,$47,$D4,$7F,$37,$D9,$E2,$E5
	.dc.b	$EB,$81,$A4,$AF,$1E,$5F,$EF,$97,$EE,$E3,$DD,$F7,$1F,$B4,$C7,$BB
	.dc.b	$83,$F6,$B8,$E3,$6F,$95,$1F,$BA,$CB,$E1,$F2,$F8,$89,$1F,$0B,$34
	.dc.b	$79,$BA,$7F,$67,$E4,$4A,$FE,$E3,$49,$F9,$A7,$EE,$3C,$49,$3A,$8C
	.dc.b	$BD,$5E,$8D,$21,$F5,$27,$A3,$83,$EC,$38,$75,$1C,$9A,$6C,$FB,$5D
	.dc.b	$9F,$38,$8F,$B5,$E1,$97,$C4,$8F,$C1,$4F,$99,$F9,$5B,$3C,$EC,$56
	.dc.b	$F0,$F1,$36,$FF,$8B,$F9,$7F,$8E,$EE,$FB,$A4,$C7,$E6,$3F,$CB,$3D
	.dc.b	$5F,$E3,$A4,$8F,$DC,$FA,$7F,$C8,$3D,$1F,$E4,$3E,$8C,$9F,$DE,$C7
	.dc.b	$E6,$BC,$2F,$C4,$92,$3E,$C3,$D5,$FE,$07,$7D,$5F,$5D,$F5,$7F,$83
	.dc.b	$DF,$57,$DF,$7F,$A6,$F1,$3F,$2A,$48,$FC,$23,$FA,$4F,$8F,$FF,$7B
	.dc.b	$C7,$F8,$C9,$FF,$7B,$B7,$DF,$FF,$1C,$46,$A3,$F2,$8F,$FB,$A5,$BE
	.dc.b	$E6,$8D,$FB,$9F,$B6,$F2,$3F,$4A,$49,$FC,$A3,$E1,$F7,$BA,$E1,$27
	.dc.b	$C5,$1C,$DF,$25,$D7,$C3,$7F,$C0,$BC,$BF,$D8,$92,$3F,$40,$FE,$D3
	.dc.b	$CB,$F4,$FC,$1C,$9F,$CF,$C7,$F0,$A3,$DB,$F0,$72,$32,$43,$F3,$51
	.dc.b	$CD,$FD,$63,$5C,$98,$21,$82,$1F,$A0,$78,$3F,$A2,$F4,$7F,$84,$91
	.dc.b	$FD,$C3,$D7,$9F,$F2,$D3,$E7,$9F,$AD,$F0,$B5,$FE,$0D,$D2,$9C,$57
	.dc.b	$E8,$F8,$1F,$0B,$FC,$A3,$D3,$FE,$2E,$4C,$3F,$83,$3F,$83,$F8,$3F
	.dc.b	$BE,$48,$FC,$27,$C1,$FE,$0B,$E1,$7F,$07,$F0,$B2,$7F,$11,$1F,$CD
	.dc.b	$3D,$6F,$E4,$64,$8F,$E2,$4F,$57,$F1,$D7,$D5,$FC,$5D,$F5,$7F,$25
	.dc.b	$7D,$5F,$C7,$DF,$F4,$0F,$5F,$F9,$79,$23,$F8,$F3,$F9,$BF,$84,$FC
	.dc.b	$BF,$C2,$C9,$FC,$E4,$7E,$6B,$E1,$64,$7E,$7D,$FC,$EF,$8D,$FD,$33
	.dc.b	$DE,$FE,$76,$49,$FE,$54,$F5,$7F,$4D,$7D,$5F,$D1,$DF,$47,$F4,$B7
	.dc.b	$78,$7E,$96,$1F,$D6,$3D,$EF,$E8,$E4,$AF,$E6,$CF,$F4,$AF,$7B,$F5
	.dc.b	$FE,$F4,$9F,$E6,$B1,$FD,$80,$FD,$9F,$C1,$48,$C9,$0F,$E6,$11,$CD
	.dc.b	$FE,$84,$D7,$26,$08,$60,$87,$F3,$67,$83,$FC,$E3,$E0,$3F,$AF,$91
	.dc.b	$FE,$9C,$F5,$C7,$F9,$64,$F9,$E7,$F8,$67,$C1,$D7,$F2,$F1,$67,$F2
	.dc.b	$DF,$09,$FD,$F3,$E1,$3F,$B5,$92,$BF,$AD,$3F,$DE,$BE,$43,$FD,$B7
	.dc.b	$1F,$EC,$86,$BF,$D9,$DB,$5F,$D0,$D7,$C2,$FF,$B4,$57,$C4,$FF,$1C
	.dc.b	$F7,$65,$F3,$8F,$F0,$6D,$7F,$7C,$24,$7F,$6B,$7C,$58,$73,$24,$78
	.dc.b	$E3,$B4,$F4,$8F,$E4,$F5,$87,$FB,$7C,$7F,$82,$08,$FF,$77,$7D,$DC
	.dc.b	$9E,$7E,$09,$3E,$F8,$F5,$69,$23,$E1,$3E,$DD,$BF,$DD,$DF,$23,$FD
	.dc.b	$8E,$7F,$C4,$CD,$7F,$C6,$2D,$7F,$C9,$4D,$5A,$7F,$B9,$B5,$F9,$58
	.dc.b	$F8,$A7,$E7,$63,$E1,$7E,$86,$36,$BE,$86,$3E,$2D,$FF,$C6,$39,$1A
	.dc.b	$4B,$FF,$1B,$C3,$7F,$AD,$D7,$27,$FC,$BD,$87,$F7,$DA,$D4,$63,$6F
	.dc.b	$5D,$FD,$DE,$79,$39,$3D,$B1,$F1,$0F,$19,$1F,$0B,$5F,$F2,$5D,$76
	.dc.b	$6C,$65,$F9,$6D,$7C,$27,$23,$D7,$C7,$F0,$08,$E0,$D1,$FE,$3D,$9E
	.dc.b	$D8,$7B,$6B,$F2,$DE,$17,$9E,$BD,$42,$B7,$8F,$82,$3F,$41,$DB,$F8
	.dc.b	$0E,$5F,$9E,$48,$F9,$AF,$EA,$7C,$BF,$D1,$12,$3F,$55,$E5,$FE,$97
	.dc.b	$C7,$FD,$37,$87,$27,$F5,$D1,$F8,$CE,$CF,$B6,$48,$FB,$2F,$57,$F6
	.dc.b	$97,$D5,$FD,$A5,$F5,$7F,$71,$7D,$5F,$DC,$5F,$E6,$3B,$7F,$0E,$48
	.dc.b	$FC,$2B,$FC,$07,$D3,$FD,$D7,$AB,$27,$F8,$14,$7E,$F7,$D3,$91,$FC
	.dc.b	$15,$FE,$01,$E9,$FE,$93,$C1,$FC,$99,$27,$F2,$2F,$57,$F8,$65,$F5
	.dc.b	$7F,$86,$5F,$47,$F8,$75,$DE,$4F,$E1,$D8,$7E,$C3,$C1,$FC,$F9,$2B
	.dc.b	$F3,$AF,$E8,$BC,$5F,$E2,$5E,$EC,$9F,$BB,$CA,$9F,$DC,$9F,$C5,$FD
	.dc.b	$99,$3A,$8C,$BF,$A8,$7A,$34,$87,$E6,$9C,$B4,$C9,$FC,$11,$CB,$4F
	.dc.b	$D9,$F8,$FF,$B8,$23,$83,$AE,$5F,$B3,$9F,$1C,$7F,$A5,$FE,$49,$F0
	.dc.b	$78,$AD,$4F,$A4,$FE,$70,$7F,$16,$F2,$3F,$82,$91,$FE,$06,$FF,$AE
	.dc.b	$2B,$FD,$C5,$1D,$7E,$1F,$FD,$93,$5F,$E7,$F9,$CB,$7E,$1B,$3D,$9F
	.dc.b	$EC,$E9,$7C,$3E,$30,$FF,$74,$43,$E1,$F1,$08,$C3,$FB,$07,$8B,$F5
	.dc.b	$BC,$DF,$E3,$64,$7F,$8C,$BE,$7C,$B9,$CF,$93,$0F,$14,$63,$1C,$E3
	.dc.b	$AC,$66,$3D,$E3,$F9,$CF,$A3,$FC,$A0,$91,$FC,$99,$F8,$9F,$0F,$DB
	.dc.b	$E1,$E4,$F9,$E3,$C3,$F0,$D2,$3D,$4F,$93,$E1,$FF,$A2,$FA,$7F,$CD
	.dc.b	$09,$3F,$CC,$9E,$AF,$C1,$BE,$AF,$DD,$BE,$8F,$DF,$BB,$C1,$F4,$F0
	.dc.b	$FE,$A3,$EB,$FF,$40,$25,$7F,$3E,$7F,$A1,$FA,$FF,$1B,$E1,$64,$FE
	.dc.b	$8A,$3F,$AC,$9F,$2B,$E1,$64,$64,$87,$F3,$68,$E6,$FF,$4A,$6B,$93
	.dc.b	$04,$30,$43,$FB,$2B,$C1,$FE,$CF,$EE,$7F,$58,$23,$FD,$59,$EB,$8F
	.dc.b	$F5,$39,$F3,$CF,$E1,$FE,$1A,$BF,$CF,$62,$CF,$E5,$FE,$1F,$FB,$AF
	.dc.b	$BF,$FD,$90,$95,$FD,$89,$FD,$07,$C2,$FE,$28,$91,$F9,$3F,$85,$FC
	.dc.b	$6F,$C3,$FE,$3B,$E1,$E4,$FE,$5E,$3F,$C0,$3F,$07,$FD,$C0,$91,$FD
	.dc.b	$B9,$EA,$FE,$76,$FA,$BF,$B9,$BE,$AF,$E8,$6F,$AB,$FB,$DB,$FF,$06
	.dc.b	$FC,$1F,$F7,$C2,$47,$F7,$A7,$F7,$1F,$0F,$FA,$2F,$87,$93,$FA,$B8
	.dc.b	$FD,$2F,$C3,$48,$FD,$73,$FA,$8F,$87,$FD,$9F,$C9,$A9,$50,$82,$34
	.dc.b	$E0,$DB,$87,$0D,$78,$F8,$59,$E1,$FE,$15,$8F,$9A,$7F,$8D,$5F,$95
	.dc.b	$FE,$39,$8D,$AF,$E2,$18,$F9,$AF,$C1,$5E,$06,$BE,$56,$E2,$6D,$FF
	.dc.b	$10,$E3,$FF,$75,$60,$F0,$BD,$D1,$87,$E0,$5A,$BE,$29,$C7,$49,$8F
	.dc.b	$F8,$1A,$74,$D9,$EE,$F9,$A3,$2F,$C4,$8C,$B4,$D3,$87,$E7,$C7,$06
	.dc.b	$C6,$5F,$2B,$EE,$CA,$47,$76,$BD,$46,$3E,$F7,$EB,$CF,$B1,$FF,$0F
	.dc.b	$E2,$E5,$C4,$70,$CB,$BB,$ED,$37,$C7,$F8,$8E,$2F,$9E,$48,$F8,$0F
	.dc.b	$D4,$B7,$77,$D1,$24,$F8,$3B,$BE,$A7,$57,$D5,$E8,$93,$E4,$8F,$CA
	.dc.b	$F3,$7D,$C2,$47,$CD,$7B,$5F,$48,$F4,$1F,$9A,$C3,$C2,$F7,$3A,$1F
	.dc.b	$02,$FB,$BC,$04,$2F,$F4,$BC,$BF,$8C,$24,$FD,$D7,$DD,$E3,$7E,$37
	.dc.b	$C7,$93,$F3,$23,$F2,$1E,$34,$8F,$9E,$FE,$4F,$C6,$FD,$8F,$57,$E6
	.dc.b	$09,$3F,$89,$7A,$BF,$5A,$FA,$BE,$7C,$BF,$C3,$4F,$42,$19,$BE,$CF
	.dc.b	$E5,$E3,$F4,$A4,$7F,$2C,$FE,$A3,$B3,$F4,$DE,$84,$9F,$CF,$47,$F0
	.dc.b	$63,$F5,$5E,$64,$8C,$90,$FE,$03,$1C,$DF,$D1,$35,$C9,$82,$18,$21
	.dc.b	$FA,$27,$83,$FA,$4F,$03,$F7,$A4,$7F,$5C,$F5,$E8,$FC,$F4,$7F,$17
	.dc.b	$9F,$43,$F9,$EF,$5A,$BF,$7F,$AD,$38,$AE,$39,$3D,$2F,$E4,$BE,$17
	.dc.b	$F0,$C2,$61,$FB,$D7,$F3,$DE,$F7,$F0,$DF,$6E,$4F,$EB,$23,$F8,$87
	.dc.b	$B9,$23,$F5,$EF,$F1,$3F,$6B,$F9,$8F,$93,$FC,$60,$93,$FC,$25,$EA
	.dc.b	$FE,$DA,$FA,$BF,$81,$BE,$8F,$E0,$EE,$E7,$EA,$C3,$F9,$EF,$95,$FC
	.dc.b	$90,$95,$FC,$55,$FE,$53,$F2,$9F,$EF,$AF,$07,$FA,$03,$67,$FA,$1B
	.dc.b	$53,$FE,$FE,$7F,$D4,$EF,$FB,$F9,$FE,$17,$1F,$CB,$CF,$F1,$FC,$66
	.dc.b	$3E,$87,$97,$FC,$D8,$8F,$F2,$77,$C7,$F0,$FC,$09,$1D,$BE,$1F,$9F
	.dc.b	$C3,$F5,$8E,$E7,$89,$FE,$B7,$E7,$7F,$42,$24,$7F,$33,$7F,$A2,$FA
	.dc.b	$5E,$4F,$84,$93,$FC,$EE,$3F,$B1,$9E,$8F,$84,$91,$92,$1F,$D7,$A3
	.dc.b	$9B,$FC,$FD,$AE,$4C,$10,$C1,$0F,$E7,$EF,$07,$FA,$0F,$A9,$FD,$68
	.dc.b	$8F,$F4,$B7,$AF,$07,$E7,$BA,$7F,$CC,$13,$FD,$CA,$BE,$7F,$C2,$FA
	.dc.b	$A3,$CF,$5B,$9F,$A7,$F0,$DF,$E0,$0F,$57,$FB,$39,$2B,$FA,$C3,$F8
	.dc.b	$4F,$84,$FB,$3F,$09,$27,$EF,$C7,$DC,$F8,$69,$1F,$82,$7E,$FF,$C3
	.dc.b	$7F,$83,$3D,$AF,$EE,$64,$9F,$EC,$8F,$57,$F1,$37,$D5,$FE,$65,$CC
	.dc.b	$C3,$AC,$63,$0B,$A7,$EF,$A9,$8F,$F8,$6B,$DB,$FF,$00,$92,$FF,$B9
	.dc.b	$3F,$9E,$F8,$6F,$C9,$92,$3F,$33,$F0,$DF,$96,$F8,$5F,$CB,$FC,$34
	.dc.b	$9F,$CE,$C7,$FA,$63,$E1,$BF,$C1,$F0,$69,$1E,$C7,$E9,$2F,$B3,$EC
	.dc.b	$BE,$CF,$EA,$2F,$B3,$F0,$2F,$B7,$C1,$FF,$87,$30,$34,$93,$FE,$22
	.dc.b	$87,$C1,$7E,$BB,$E1,$64,$FF,$8E,$B6,$FE,$47,$5A,$8C,$78,$1F,$E3
	.dc.b	$D8,$C4,$7D,$0F,$85,$FD,$B9,$EF,$C2,$47,$77,$C8,$47,$04,$3F,$C8
	.dc.b	$93,$D1,$FF,$24,$B5,$F0,$47,$17,$6D,$77,$F0,$DF,$04,$8F,$BD,$E1
	.dc.b	$87,$B2,$7C,$B3,$F8,$89,$B3,$98,$F7,$9F,$E5,$CA,$F7,$C7,$E2,$5F
	.dc.b	$B8,$7C,$F2,$47,$CD,$7F,$C9,$BF,$2B,$FE,$EE,$8F,$E3,$E7,$F1,$C3
	.dc.b	$FD,$B6,$E5,$FC,$CE,$7F,$C8,$08,$FF,$26,$C3,$F1,$59,$FD,$D2,$47
	.dc.b	$DB,$7B,$F0,$75,$8E,$E7,$A8,$5F,$7C,$7F,$45,$87,$69,$F0,$9D,$CE
	.dc.b	$18,$B3,$FC,$61,$23,$F1,$4F,$BB,$A7,$D6,$48,$F0,$60,$EC,$F7,$F4
	.dc.b	$FC,$0E,$99,$5A,$8F,$D3,$72,$7E,$64,$91,$F9,$77,$B3,$D3,$0E,$CF
	.dc.b	$F8,$C7,$0E,$CF,$33,$C3,$7D,$9D,$5F,$EC,$F9,$3F,$4C,$48,$FD,$23
	.dc.b	$FA,$9E,$3F,$AB,$E1,$C9,$F8,$71,$FB,$B3,$EC,$F8,$12,$32,$43,$F3
	.dc.b	$71,$CD,$FD,$83,$5C,$98,$21,$82,$1F,$A2,$78,$3F,$A4,$E6,$FD,$F1
	.dc.b	$1F,$DE,$3D,$78,$7F,$73,$3E,$79,$FA,$7E,$4D,$7F,$CB,$1C,$29,$C5
	.dc.b	$7F,$88,$B5,$78,$BF,$C6,$7B,$3F,$86,$13,$0F,$E1,$4F,$E7,$BC,$CF
	.dc.b	$CD,$F9,$B2,$7F,$03,$1F,$9E,$F3,$E4,$7E,$11,$FD,$0F,$9B,$FC,$97
	.dc.b	$C1,$FE,$30,$49,$FE,$2C,$F5,$7F,$17,$7D,$5F,$C0,$5F,$47,$F0,$57
	.dc.b	$73,$F5,$61,$FC,$C7,$C1,$FE,$48,$4A,$FE,$44,$FE,$DB,$DD,$FD,$99
	.dc.b	$23,$F3,$3E,$E7,$ED,$BD,$9F,$DB,$FA,$F2,$7F,$3B,$1F,$CF,$7C,$4F
	.dc.b	$E6,$04,$8F,$E5,$CF,$57,$F4,$77,$D5,$FC,$C5,$F5,$7F,$4F,$7D,$5F
	.dc.b	$CE,$5F,$F4,$9F,$1F,$F9,$E1,$23,$F9,$D3,$FC,$FF,$C7,$FE,$17,$F0
	.dc.b	$12,$7F,$73,$1F,$D4,$4F,$E1,$FF,$03,$23,$24,$3F,$98,$C7,$37,$FA
	.dc.b	$33,$5C,$98,$21,$82,$1F,$CE,$1E,$0F,$F3,$AF,$33,$FA,$A1,$1F,$EA
	.dc.b	$4F,$5D,$FF,$A7,$CF,$9E,$7F,$84,$FC,$25,$7F,$2F,$16,$7F,$2D,$F0
	.dc.b	$9F,$DB,$7C,$EF,$EC,$04,$AF,$EB,$CF,$FB,$FB,$E5,$BF,$DA,$D1,$FC
	.dc.b	$8D,$7C,$1A,$30,$D7,$FD,$79,$1C,$1B,$F0,$F9,$C3,$FD,$0F,$3F,$F0
	.dc.b	$06,$79,$35,$2F,$86,$FF,$01,$7A,$BF,$DD,$49,$3F,$DC,$DF,$37,$C3
	.dc.b	$75,$F4,$BC,$57,$E6,$AF,$29,$E9,$9D,$3E,$0C,$6B,$E0,$6C,$F8,$CF
	.dc.b	$14,$7F,$3C,$F5,$BF,$C0,$C4,$7F,$C0,$8F,$C7,$F8,$5E,$24,$8F,$91
	.dc.b	$F0,$DD,$3E,$17,$AF,$C3,$49,$F9,$91,$FE,$1F,$F6,$BF,$C2,$44,$8F
	.dc.b	$F0,$83,$D5,$FA,$57,$D5,$F9,$F7,$D5,$FA,$D7,$D5,$FA,$77,$EE,$F9
	.dc.b	$7F,$F5,$BC,$7D,$70,$E1,$FB,$4D,$7E,$F3,$5D,$FB,$DF,$1B,$E1,$24
	.dc.b	$7F,$67,$94,$F7,$3E,$5F,$C3,$49,$D4,$65,$F1,$1D,$9A,$D2,$CF,$46
	.dc.b	$C6,$9F,$19,$82,$1D,$23,$83,$63,$84,$62,$3E,$13,$5F,$83,$F0,$3A
	.dc.b	$23,$E2,$78,$CF,$A6,$1C,$9F,$EE,$2A,$EE,$FE,$7B,$E1,$F4,$E6,$AF
	.dc.b	$1F,$E5,$BE,$1F,$EA,$7C,$1F,$90,$98,$FC,$07,$F6,$5F,$0B,$F9,$4F
	.dc.b	$85,$93,$FB,$38,$FC,$BF,$C3,$C8,$FD,$B3,$F9,$AF,$87,$FC,$07,$C3
	.dc.b	$FC,$02,$4F,$CC,$7A,$BF,$BD,$BE,$AF,$EE,$AF,$A3,$FB,$CB,$BA,$7E
	.dc.b	$EB,$0F,$C6,$57,$CC,$25,$7D,$87,$F8,$56,$9F,$AC,$24,$7F,$0B,$DF
	.dc.b	$F5,$F9,$FE,$C3,$39,$3F,$C3,$E3,$F3,$19,$FD,$82,$47,$E1,$1E,$AF
	.dc.b	$F1,$5B,$EA,$FF,$12,$BE,$AF,$F1,$9B,$EA,$FF,$16,$BF,$D2,$67,$F8
	.dc.b	$42,$47,$E4,$1F,$C3,$E5,$FC,$1B,$DA,$FF,$76,$68,$DB,$F9,$04,$61
	.dc.b	$FD,$4C,$FE,$BC,$FE,$16,$CB,$FD,$34,$65,$FC,$EB,$D9,$A4,$3F,$34
	.dc.b	$E5,$A6,$4F,$CC,$B9,$69,$FA,$8D,$FF,$3E,$47,$07,$5B,$FC,$C4,$F8
	.dc.b	$E3,$F3,$75,$EC,$EC,$C5,$6A,$7D,$27,$F9,$97,$07,$4F,$F0,$DE,$1F
	.dc.b	$D6,$92,$7F,$72,$FE,$DE,$5E,$37,$CB,$F0,$A5,$73,$8F,$9D,$E2,$48
	.dc.b	$EC,$FD,$1F,$0B,$F8,$D7,$47,$EE,$89,$3F,$C1,$9E,$D5,$D2,$3D,$67
	.dc.b	$F5,$7C,$18,$7D,$B9,$ED,$3A,$12,$C3,$39,$FD,$F9,$E5,$8F,$F2,$BE
	.dc.b	$8F,$E1,$64,$8F,$E2,$EF,$BF,$D2,$FC,$41,$23,$E4,$7A,$5F,$8B,$F3
	.dc.b	$BF,$19,$E7,$49,$F9,$91,$FC,$DF,$C0,$FE,$2E,$48,$FE,$46,$F5,$7E
	.dc.b	$95,$F5,$7D,$5A,$30,$ED,$1F,$5F,$0E,$CE,$B0,$FE,$89,$E0,$7F,$24
	.dc.b	$24,$7F,$30,$7F,$93,$F7,$FE,$87,$D5,$93,$FC,$FE,$3F,$A6,$1F,$A5
	.dc.b	$F6,$24,$64,$87,$F3,$38,$E6,$FF,$3C,$6B,$93,$04,$30,$43,$F9,$BB
	.dc.b	$C1,$FE,$73,$E1,$7F,$3C,$23,$FD,$21,$EB,$C1,$FC,$D6,$7C,$F3,$F9
	.dc.b	$9F,$7A,$BF,$81,$F0,$D3,$8A,$FC,$04,$FE,$13,$6F,$ED,$5E,$3F,$F4
	.dc.b	$A2,$47,$F5,$67,$F3,$DF,$0B,$FC,$1F,$D9,$93,$FA,$78,$FE,$15,$F0
	.dc.b	$92,$3F,$54,$FF,$0D,$F8,$2F,$EE,$DE,$6F,$F5,$62,$4F,$F6,$27,$AB
	.dc.b	$FB,$1B,$EA,$FE,$02,$FA,$3F,$82,$BB,$97,$AB,$2F,$F6,$A4,$FF,$77
	.dc.b	$F3,$3F,$B1,$91,$FE,$DE,$FF,$6D,$F9,$8F,$F7,$07,$85,$FE,$3D,$6C
	.dc.b	$FF,$90,$5A,$FE,$A0,$FE,$4D,$F0,$9F,$C9,$FE,$1A,$47,$F0,$78,$FF
	.dc.b	$09,$7A,$1F,$DC,$89,$1F,$DF,$9F,$07,$C3,$7F,$8D,$2F,$A9,$DB,$2F
	.dc.b	$F0,$C6,$7E,$BC,$3A,$E1,$FA,$0C,$7C,$2F,$FB,$E7,$E6,$5E,$A7,$F8
	.dc.b	$1D,$B1,$FE,$0E,$6B,$FE,$12,$6B,$FE,$25,$5E,$A7,$8F,$E1,$24,$F5
	.dc.b	$8F,$F2,$4C,$EA,$3F,$CC,$98,$FE,$23,$37,$C2,$FA,$CD,$DC,$AF,$13
	.dc.b	$E5,$23,$82,$1E,$57,$2D,$32,$79,$DC,$BE,$9F,$63,$E0,$11,$F8,$0F
	.dc.b	$2B,$FC,$EE,$1E,$99,$FE,$44,$D8,$F5,$7C,$37,$9E,$73,$76,$7D,$DF
	.dc.b	$0F,$F0,$BD,$AF,$9C,$4A,$F9,$CF,$E2,$FE,$17,$EE,$7C,$24,$9F,$8F
	.dc.b	$CB,$F8,$D9,$FC,$07,$C3,$C9,$C9,$F8,$F7,$F0,$9F,$0F,$F4,$3E,$0B
	.dc.b	$EE,$12,$7E,$E3,$DD,$E2,$38,$7F,$DE,$AE,$1A,$FF,$41,$57,$E1,$8C
	.dc.b	$5D,$D3,$EE,$61,$F8,$0F,$82,$FC,$61,$2B,$F1,$8F,$EA,$7E,$17,$F3
	.dc.b	$44,$8F,$07,$C3,$7E,$77,$E1,$BF,$3D,$F0,$D2,$7C,$91,$F8,$CF,$85
	.dc.b	$FC,$C1,$23,$F3,$0F,$57,$D5,$7D,$5F,$55,$F5,$7D,$D7,$D5,$F7,$5F
	.dc.b	$E6,$3E,$1B,$F4,$84,$8F,$D2,$3F,$A7,$BF,$D9,$61,$27,$F5,$91,$FA
	.dc.b	$03,$F6,$D3,$23,$24,$3F,$59,$1C,$DF,$D8,$35,$C9,$82,$18,$21,$FB
	.dc.b	$37,$83,$FB,$5C,$3F,$78,$47,$F7,$8F,$5C,$FF,$3B,$3E,$79,$FC,$06
	.dc.b	$F5,$FC,$FC,$59,$FD,$C6,$7F,$BC,$CB,$F8,$41,$2B,$F8,$43,$FC,$97
	.dc.b	$AB,$F8,$B7,$2C,$9F,$C6,$47,$F1,$9E,$89,$1F,$8F,$7F,$8E,$72,$FF
	.dc.b	$08,$DF,$F8,$A1,$27,$F8,$A3,$D5,$FC,$B5,$F5,$7F,$2D,$7D,$1F,$CC
	.dc.b	$5D,$CB,$F8,$86,$1F,$C5,$38,$BF,$90,$12,$BF,$90,$3F,$EA,$9F,$9A
	.dc.b	$FF,$7B,$57,$F5,$E6,$F2,$FF,$BE,$DB,$5F,$EF,$F7,$FD,$D8,$FF,$89
	.dc.b	$DF,$F7,$C1,$FE,$E6,$71,$E0,$7F,$BB,$23,$FD,$F9,$D5,$FA,$E9,$FE
	.dc.b	$59,$D1,$FC,$CC,$8F,$FB,$02,$7B,$A6,$93,$C5,$E9,$E1,$75,$C2,$57
	.dc.b	$EF,$AB,$BF,$4F,$F3,$F2,$47,$F3,$F7,$FA,$17,$4F,$97,$C8,$93,$FD
	.dc.b	$1E,$3F,$9D,$9E,$9F,$2E,$46,$48,$7F,$47,$8E,$6F,$F4,$B6,$B9,$30
	.dc.b	$43,$04,$3F,$A7,$3C,$1F,$EA,$1D,$DF,$D5,$C8,$FF,$57,$7A,$EF,$FC
	.dc.b	$E2,$7C,$F3,$F0,$7D,$3A,$FF,$3A,$8B,$3F,$77,$CF,$FE,$AF,$E0,$7F
	.dc.b	$63,$25,$7F,$63,$7F,$09,$EF,$7D,$9F,$6E,$4F,$CD,$8F,$B9,$ED,$C8
	.dc.b	$FA,$0F,$DF,$F5,$FF,$B1,$F8,$DF,$DB,$C9,$3F,$DB,$DE,$AF,$D7,$BE
	.dc.b	$AF,$E2,$EF,$A3,$F8,$DB,$B9,$7E,$57,$0F,$ED,$FE,$37,$F7,$B2,$57
	.dc.b	$F7,$B7,$F3,$9F,$0B,$F9,$12,$47,$E1,$7E,$17,$F2,$9F,$03,$F9,$5F
	.dc.b	$81,$93,$F8,$98,$FE,$F7,$E7,$7F,$81,$C9,$1F,$E0,$77,$AB,$F8,$FB
	.dc.b	$EA,$FE,$8E,$FA,$BF,$93,$BE,$AF,$E9,$EC,$FF,$54,$F9,$CA,$C2,$B5
	.dc.b	$C3,$5C,$94,$2F,$3B,$F6,$1F,$07,$27,$B4,$76,$D7,$FC,$B7,$5B,$9F
	.dc.b	$D4,$7C,$27,$ED,$CD,$78,$3E,$2B,$B3,$E8,$AE,$0F,$95,$AF,$9A,$3C
	.dc.b	$91,$88,$E4,$7A,$5C,$3C,$E3,$64,$74,$D7,$B3,$D3,$D0,$91,$CD,$AE
	.dc.b	$8F,$79,$0F,$E2,$DF,$0D,$EA,$7E,$3C,$DA,$F5,$E3,$F1,$23,$F8,$C7
	.dc.b	$C3,$67,$D5,$F5,$92,$FD,$4F,$D1,$F9,$C7,$C3,$F5,$7F,$DC,$73,$FE
	.dc.b	$10,$6B,$FE,$CD,$76,$E7,$E2,$38,$11,$FE,$35,$AF,$97,$D9,$F9,$64
	.dc.b	$8F,$94,$F8,$6F,$C1,$F1,$12,$7C,$11,$CD,$F0,$5D,$D3,$F9,$F6,$1F
	.dc.b	$07,$DA,$FA,$E4,$AF,$AC,$FB,$7E,$1F,$D8,$48,$F6,$7C,$47,$BF,$E2
	.dc.b	$3E,$07,$C4,$49,$F8,$11,$F3,$7D,$FF,$C1,$92,$3F,$04,$F5,$7E,$35
	.dc.b	$F5,$7E,$35,$F5,$7E,$55,$F5,$7E,$55,$FD,$9F,$82,$FC,$79,$23,$F1
	.dc.b	$CF,$E4,$BE,$03,$EA,$7C,$2C,$9F,$D8,$47,$E0,$4F,$B1,$F0,$D2,$32
	.dc.b	$43,$F3,$D1,$CD,$FC,$D3,$5C,$98,$21,$82,$1F,$BA,$78,$3F,$BC,$F8
	.dc.b	$6F,$D3,$91,$FD,$33,$D7,$7F,$DB,$CF,$9E,$7F,$19,$F1,$15,$4E,$9F
	.dc.b	$43,$FE,$D8,$E5,$7C,$47,$E7,$FE,$23,$F6,$A4,$AF,$DA,$3F,$9B,$D3
	.dc.b	$F3,$58,$C9,$FC,$D4,$7E,$77,$29,$1F,$9D,$7F,$41,$5F,$AB,$CB,$F8
	.dc.b	$09,$27,$F8,$03,$D5,$FD,$25,$F5,$7F,$49,$7D,$1F,$D3,$5D,$CB,$D5
	.dc.b	$87,$ED,$F1,$FE,$1A,$4A,$FE,$18,$FE,$CF,$A3,$F6,$44,$8F,$D9,$74
	.dc.b	$7E,$D7,$93,$F6,$DC,$92,$7F,$6D,$1F,$C0,$F8,$3F,$8C,$92,$3F,$8C
	.dc.b	$3D,$5F,$DE,$5F,$57,$F7,$97,$D5,$FE,$01,$7D,$5F,$E0,$17,$FC,$3F
	.dc.b	$83,$F9,$29,$23,$F9,$23,$FC,$A3,$93,$F8,$57,$54,$9F,$E8,$D1,$FC
	.dc.b	$5C,$FE,$1D,$DB,$23,$24,$3F,$9A,$47,$37,$F9,$83,$5C,$98,$21,$82
	.dc.b	$1F,$D4,$1E,$0F,$F5,$2E,$6F,$E7,$A4,$7F,$9E,$3D,$77,$FE,$9B,$3E
	.dc.b	$79,$FF,$58,$F8,$0F,$13,$11,$F9,$8E,$9F,$EB,$F5,$8B,$FC,$C4,$FF
	.dc.b	$35,$E8,$FE,$96,$47,$FA,$53,$FD,$87,$07,$9F,$FC,$F5,$79,$94,$C4
	.dc.b	$67,$CC,$97,$E4,$78,$79,$1F,$D0,$FB,$7F,$AC,$92,$7F,$AC,$3D,$EB
	.dc.b	$A4,$7B,$2F,$AB,$CC,$87,$B4,$FE,$FF,$83,$0E,$F1,$FE,$01,$EF,$FE
	.dc.b	$CE,$47,$FB,$33,$ED,$F6,$7D,$84,$8F,$97,$EC,$7B,$FD,$4F,$81,$E8
	.dc.b	$C9,$F9,$F1,$FD,$77,$C4,$FE,$E6,$48,$FE,$E4,$F5,$7E,$AD,$F5,$7D
	.dc.b	$03,$2F,$F7,$2F,$03,$8B,$FC,$2F,$0C,$3C,$15,$C2,$18,$F9,$63,$FC
	.dc.b	$3D,$E2,$7F,$81,$08,$FF,$80,$DF,$F0,$D7,$C1,$75,$8F,$C3,$FC,$07
	.dc.b	$DB,$3F,$BC,$46,$63,$CA,$DC,$B1,$83,$E0,$3F,$04,$70,$CE,$5F,$F8
	.dc.b	$A6,$F9,$3F,$E2,$F6,$BF,$E3,$26,$BF,$E3,$66,$A6,$92,$3D,$24,$8E
	.dc.b	$CD,$7F,$C7,$50,$F2,$BD,$E4,$8F,$23,$BC,$3D,$D3,$ED,$AF,$21,$F9
	.dc.b	$DF,$82,$F5,$DF,$C2,$8B,$5F,$0B,$E0,$FE,$3F,$97,$F3,$09,$5F,$05
	.dc.b	$FD,$46,$1F,$A7,$F8,$39,$3F,$B6,$8F,$D5,$7C,$2C,$8F,$DC,$BF,$AD
	.dc.b	$F8,$5F,$A7,$E8,$FD,$82,$4F,$CD,$7A,$BF,$C0,$2F,$AB,$F8,$5B,$E8
	.dc.b	$FE,$1E,$EE,$5F,$6B,$0F,$BF,$E8,$FE,$10,$95,$F6,$5F,$E0,$1F,$09
	.dc.b	$FB,$F2,$47,$F0,$EF,$85,$FE,$05,$F0,$BF,$C0,$FE,$16,$4F,$F1,$28
	.dc.b	$FC,$5F,$B1,$F9,$02,$47,$E1,$5E,$AF,$F1,$8B,$EA,$FE,$76,$FA,$BF
	.dc.b	$C7,$2F,$AB,$FA,$1B,$FC,$BF,$B1,$F9,$C2,$47,$E4,$5F,$CF,$F3,$7F
	.dc.b	$94,$FE,$75,$F0,$94,$FC,$FC,$4A,$FF,$43,$ED,$FA,$1A,$FF,$77,$37
	.dc.b	$E1,$35,$3F,$A4,$AE,$AF,$E7,$DA,$FE,$49,$AF,$E8,$5A,$FE,$85,$AF
	.dc.b	$E8,$5A,$FE,$87,$DB,$FD,$B9,$1F,$D6,$3D,$AB,$F3,$33,$E4,$7D,$1F
	.dc.b	$11,$7A,$3B,$5F,$B1,$AF,$F1,$0C,$5F,$0D,$FC,$03,$E0,$3F,$82,$92
	.dc.b	$7F,$74,$FF,$A8,$65,$F0,$7F,$2F,$E0,$E5,$75,$3F,$D5,$F3,$A7,$01
	.dc.b	$7C,$2F,$D3,$24,$6A,$1F,$0B,$FC,$4B,$E1,$7F,$8B,$92,$7F,$86,$3E
	.dc.b	$3C,$3C,$31,$EB,$C3,$EF,$47,$48,$F7,$61,$75,$FB,$98,$7F,$1E,$F8
	.dc.b	$5F,$E4,$64,$AF,$E3,$0F,$BF,$3F,$C4,$12,$3C,$D9,$7E,$2E,$7F,$19
	.dc.b	$F0,$B2,$7D,$51,$FC,$AA,$3F,$97,$92,$3F,$92,$3D,$5F,$81,$5D,$5F
	.dc.b	$A9,$5D,$5F,$89,$3D,$5F,$B1,$5F,$CD,$B2,$FE,$76,$48,$FE,$60,$FF
	.dc.b	$3E,$CF,$F4,$1C,$12,$7E,$9F,$6A,$7F,$9F,$9F,$A5,$E2,$93,$A8,$CB
	.dc.b	$FC,$ED,$E8,$D2,$1F,$CC,$1C,$B4,$C9,$FD,$05,$CB,$4F,$E8,$DB,$7F
	.dc.b	$54,$23,$83,$AE,$DF,$CD,$A7,$C7,$1F,$99,$EA,$B3,$98,$F4,$1F,$8B
	.dc.b	$E6,$FE,$B7,$C1,$FD,$80,$91,$FD,$55,$FC,$E7,$87,$FC,$14,$91,$F9
	.dc.b	$3F,$07,$F8,$47,$83,$FC,$27,$BA,$4F,$E5,$E3,$FB,$3F,$47,$F6,$C2
	.dc.b	$47,$F6,$17,$AB,$F9,$DB,$EA,$FE,$C6,$FA,$BF,$A1,$BE,$AF,$ED,$6F
	.dc.b	$FB,$9F,$3F,$F7,$82,$47,$F6,$D4,$C2,$0F,$9E,$FF,$70,$BF,$E0,$E7
	.dc.b	$FD,$CB,$A7,$F7,$D6,$F4,$F1,$77,$19,$F1,$FF,$DF,$50,$69,$FC,$AE
	.dc.b	$7F,$C0,$FD,$9F,$E0,$C2,$5F,$F8,$05,$F1,$BD,$7C,$DE,$7E,$64,$8F
	.dc.b	$E0,$B8,$78,$2F,$C5,$87,$69,$CB,$DD,$FC,$CF,$77,$F8,$68,$8F,$F8
	.dc.b	$3D,$EE,$FC,$4F,$9F,$78,$7E,$AF,$52,$4F,$F1,$49,$CB,$67,$DF,$E9
	.dc.b	$C8,$C1,$0F,$5C,$F0,$6C,$F8,$14,$65,$FA,$6D,$7E,$B3,$5F,$34,$71
	.dc.b	$8C,$47,$28,$C1,$1F,$6C,$68,$78,$BC,$49,$1A,$78,$3F,$E4,$E8,$F4
	.dc.b	$6F,$FE,$3D,$6E,$7F,$81,$F5,$7F,$15,$39,$C2,$CF,$A1,$B7,$E8,$3E
	.dc.b	$CF,$AF,$87,$E0,$F8,$DE,$B2,$3E,$87,$E3,$6E,$F8,$5F,$C6,$92,$7F
	.dc.b	$07,$F0,$3F,$90,$F8,$1F,$C8,$FB,$B2,$7F,$0F,$1D,$F6,$79,$7F,$24
	.dc.b	$93,$F1,$9E,$F8,$75,$8F,$D1,$3F,$9C,$C3,$B4,$7E,$4B,$0E,$CF,$E9
	.dc.b	$F0,$ED,$E2,$7D,$52,$47,$D2,$7D,$7F,$09,$FA,$6F,$85,$93,$F9,$C8
	.dc.b	$FD,$4F,$C2,$C8,$FC,$FB,$FA,$CF,$85,$F5,$7A,$5F,$81,$24,$FD,$E7
	.dc.b	$AB,$FA,$6B,$EA,$FE,$EE,$FA,$3F,$BD,$BB,$C9,$FB,$DC,$3E,$37,$A5
	.dc.b	$F8,$D2,$57,$E2,$9F,$C8,$7A,$5F,$BE,$F8,$29,$3F,$A5,$8F,$9C,$7F
	.dc.b	$02,$F8,$39,$19,$21,$FA,$48,$E6,$FE,$59,$AE,$4C,$10,$C1,$0F,$D3
	.dc.b	$BC,$1F,$D4,$FA,$DF,$A5,$23,$FA,$27,$AE,$3F,$9B,$9F,$3C,$FF,$0C
	.dc.b	$F8,$5A,$FE,$86,$2C,$FF,$21,$F8,$5F,$C5,$7B,$1F,$B1,$25,$7E,$B9
	.dc.b	$FD,$E7,$D0,$7F,$BA,$E3,$FC,$36,$D7,$FD,$72,$DA,$FE,$59,$1F,$CA
	.dc.b	$3E,$1B,$F9,$4F,$C3,$48,$FE,$3F,$1F,$99,$F7,$7F,$80,$12,$3F,$7A
	.dc.b	$F8,$3E,$1B,$F5,$98,$75,$F0,$7F,$9F,$36,$BF,$A0,$F7,$7F,$40,$CB
	.dc.b	$F9,$E6,$3D,$E7,$FA,$06,$5E,$37,$F9,$07,$BD,$FC,$3C,$8F,$F0,$C7
	.dc.b	$DF,$F0,$BE,$0F,$85,$93,$F0,$23,$C7,$F0,$B2,$3E,$1B,$E6,$F8,$6F
	.dc.b	$DA,$7C,$2F,$F1,$B2,$4F,$F1,$87,$AB,$F2,$AF,$AB,$F2,$AF,$A3,$F3
	.dc.b	$2E,$E7,$FE,$D3,$D5,$8F,$F0,$1F,$86,$FE,$50,$4B,$FE,$4A,$FF,$2B
	.dc.b	$F8,$6F,$92,$48,$FE,$80,$FF,$09,$3E,$6F,$C3,$48,$C9,$0F,$E7,$F1
	.dc.b	$CD,$FE,$62,$D7,$26,$08,$60,$87,$F4,$27,$83,$FD,$12,$3F,$A0,$11
	.dc.b	$FE,$7A,$F5,$C3,$F9,$B4,$F9,$E7,$F2,$3A,$57,$F9,$E4,$59,$F8,$D7
	.dc.b	$FC,$97,$3F,$E9,$84,$AF,$E9,$2F,$E7,$3A,$3F,$1A,$48,$FC,$EF,$2F
	.dc.b	$E4,$39,$7F,$23,$C1,$27,$F4,$31,$FC,$C7,$83,$FA,$C1,$23,$FA,$AB
	.dc.b	$D5,$FD,$3D,$F5,$7F,$4F,$7D,$5F,$D5,$DF,$57,$F5,$77,$FC,$F7,$83
	.dc.b	$FB,$21,$23,$FB,$0B,$FB,$4F,$07,$F4,$BE,$0C,$9F,$DA,$C7,$EA,$3B
	.dc.b	$64,$7E,$E1,$FD,$5F,$57,$F4,$9E,$3F,$EE,$04,$9F,$ED,$AF,$57,$F7
	.dc.b	$F7,$D5,$FD,$FD,$F4,$7F,$80,$DD,$E0,$FC,$B6,$1F,$D5,$7A,$7F,$BE
	.dc.b	$12,$BF,$BC,$BF,$E0,$7F,$A1,$FD,$C7,$57,$EF,$3C,$9F,$DF,$9F,$DA
	.dc.b	$E3,$35,$FE,$3E,$6A,$E8,$FE,$C2,$FF,$14,$F2,$38,$30,$FF,$11,$CF
	.dc.b	$F8,$55,$34,$8F,$8C,$87,$02,$9F,$E3,$A7,$6E,$4E,$BD,$7E,$B2,$38
	.dc.b	$34,$33,$63,$FE,$57,$AF,$59,$EC,$C3,$FC,$27,$5E,$F3,$D4,$71,$9E
	.dc.b	$0F,$F2,$4F,$37,$E3,$76,$7C,$B2,$47,$C0,$7F,$C6,$9A,$BD,$AF,$F0
	.dc.b	$B3,$67,$FD,$AA,$D7,$B1,$FE,$6C,$7F,$DC,$AE,$BD,$5E,$3E,$94,$8C
	.dc.b	$1F,$ED,$EC,$DE,$17,$DD,$24,$7D,$27,$C7,$87,$88,$91,$E2,$F5,$BC
	.dc.b	$B8,$75,$8F,$18,$C5,$8E,$A3,$C1,$E1,$FE,$2C,$8F,$E0,$1F,$83,$F0
	.dc.b	$5F,$03,$DF,$93,$F0,$23,$E1,$FC,$04,$8F,$86,$FC,$6F,$7B,$F2,$5E
	.dc.b	$4F,$E5,$C9,$3F,$8C,$7A,$BF,$2A,$FA,$BE,$8C,$98,$75,$8C,$61,$79
	.dc.b	$3F,$07,$1F,$3F,$1F,$D0,$79,$3F,$A5,$24,$7E,$69,$FD,$47,$99,$F6
	.dc.b	$BE,$0E,$4F,$D1,$8F,$D4,$1F,$7B,$E1,$24,$64,$87,$E6,$E3,$9B,$FA
	.dc.b	$66,$B9,$30,$43,$04,$3F,$D5,$FC,$88,$E3,$18,$F3,$FF,$7C,$47,$F6
	.dc.b	$6F,$6C,$BE,$0C,$7E,$D2,$7D,$2F,$E5,$3E,$1A,$BF,$82,$BE,$CF,$AA
	.dc.b	$B8,$EC,$F8,$6F,$E1,$1E,$8F,$F0,$D2,$57,$F0,$27,$F4,$7F,$09,$FA
	.dc.b	$22,$47,$E8,$BE,$13,$F4,$BF,$0D,$FA,$6F,$86,$93,$FA,$68,$FE,$29
	.dc.b	$EA,$FF,$19,$24,$7F,$0E,$7A,$BF,$AC,$BE,$AF,$E0,$EF,$AB,$FB,$0B
	.dc.b	$EA,$FE,$1E,$FF,$90,$7B,$3F,$C9,$49,$1F,$C6,$9F,$DD,$FC,$37,$EE
	.dc.b	$BE,$1A,$4F,$EE,$A3,$F7,$BF,$0B,$23,$F7,$AF,$F0,$0F,$86,$FE,$59
	.dc.b	$EE,$FF,$31,$24,$FF,$26,$7A,$BF,$C1,$2F,$AB,$F9,$BB,$E8,$FE,$76
	.dc.b	$EF,$0F,$D2,$C3,$F9,$C7,$B9,$FC,$F4,$95,$FC,$C9,$FE,$81,$EE,$FF
	.dc.b	$A9,$FE,$8B,$FD,$D9,$3F,$EF,$AB,$FC,$1C,$62,$3F,$01,$3F,$D0,$CF
	.dc.b	$E3,$3E,$4C,$8F,$E2,$11,$97,$F9,$FB,$D9,$A4,$3F,$9D,$39,$69,$93
	.dc.b	$F3,$6E,$5A,$7F,$4A,$F8,$2F,$EB,$04,$70,$75,$BF,$CD,$F3,$AB,$C9
	.dc.b	$3F,$C6,$B0,$FF,$7F,$3F,$D6,$E3,$35,$C1,$F8,$9F,$0D,$FD,$87,$E0
	.dc.b	$BF,$B3,$12,$BF,$AD,$BF,$EE,$FB,$63,$F2,$C9,$3F,$2F,$23,$25,$5F
	.dc.b	$47,$E1,$7E,$97,$C2,$CA,$E0,$6E,$3F,$B9,$1F,$DD,$C8,$FF,$6C,$7C
	.dc.b	$58,$78,$23,$DC,$FD,$DA,$ED,$1D,$05,$AF,$C1,$1F,$16,$FF,$C0,$11
	.dc.b	$FE,$05,$24,$7F,$7A,$7E,$07,$1F,$E2,$78,$E4,$F5,$BF,$F7,$63,$77
	.dc.b	$FC,$71,$1D,$6B,$FE,$0D,$F8,$7F,$F0,$91,$23,$FC,$0E,$F6,$7A,$5F
	.dc.b	$67,$ED,$61,$CD,$FB,$97,$77,$FC,$CE,$1F,$E1,$9D,$FF,$C3,$C4,$AF
	.dc.b	$F0,$9B,$FE,$21,$F8,$7F,$D1,$75,$C9,$FE,$D5,$1F,$E2,$13,$F4,$DD
	.dc.b	$72,$32,$43,$FC,$29,$1C,$DF,$F0,$FB,$5C,$98,$21,$82,$1F,$CE,$9E
	.dc.b	$0F,$E2,$0F,$C4,$FD,$27,$F3,$67,$01,$C3,$B6,$BC,$38,$FC,$59,$7E
	.dc.b	$92,$BD,$93,$F5,$FC,$0A,$FF,$3D,$8B,$3F,$C0,$BC,$1F,$13,$FD,$1B
	.dc.b	$07,$4F,$8A,$FC,$0D,$7C,$ED,$7C,$CF,$E9,$FC,$FF,$E2,$04,$8F,$C4
	.dc.b	$79,$7F,$C4,$FC,$BF,$E2,$9E,$3C,$9F,$C6,$47,$8B,$5F,$77,$77,$5C
	.dc.b	$8D,$25,$78,$F9,$FF,$C4,$6F,$E5,$72,$F0,$3F,$C0,$F2,$F0,$3F,$99
	.dc.b	$CB,$C0,$FF,$07,$CB,$C0,$FD,$2E,$EF,$01,$1F,$81,$92,$3E,$07,$D2
	.dc.b	$7F,$B9,$67,$F5,$E7,$F2,$F3,$98,$75,$93,$D4,$FE,$68,$49,$FD,$67
	.dc.b	$A5,$C3,$07,$83,$F0,$49,$3E,$27,$C3,$EB,$F7,$8F,$DA,$1E,$0C,$7B
	.dc.b	$E1,$D6,$3F,$92,$3F,$D0,$F0,$ED,$E3,$7C,$D2,$4F,$B9,$FA,$1E,$37
	.dc.b	$3F,$72,$4F,$F0,$DC,$D3,$E7,$3C,$1E,$E4,$9D,$46,$5F,$92,$F4,$69
	.dc.b	$0F,$C7,$4F,$47,$07,$E8,$9C,$3A,$8E,$4D,$36,$7E,$97,$C7,$FC,$41
	.dc.b	$1F,$B2,$F0,$E7,$FC,$F4,$F9,$67,$FC,$C3,$A3,$E0,$B1,$5A,$9F,$59
	.dc.b	$FC,$B6,$5F,$05,$F5,$FC,$AF,$CB,$12,$7F,$0E,$D7,$DA,$D7,$D8,$D7
	.dc.b	$F7,$8D,$7F,$81,$BF,$A1,$F3,$64,$7E,$49,$AF,$C8,$6B,$F1,$DA,$FF
	.dc.b	$07,$6B,$FC,$39,$AF,$F0,$DF,$33,$F5,$C4,$7F,$3E,$D7,$EA,$B5,$FA
	.dc.b	$8D,$7F,$8A,$35,$FE,$36,$FE,$E7,$CF,$91,$FA,$A6,$BF,$80,$F3,$FF
	.dc.b	$31,$F0,$92,$3F,$1D,$1F,$A3,$3F,$39,$F0,$D2,$32,$43,$91,$FB,$68
	.dc.b	$FE,$10,$47,$04,$36,$43,$F1,$CE,$DC,$1F,$CB,$FD,$4F,$E2,$44,$7F
	.dc.b	$81,$BB,$D7,$F9,$44,$F9,$E7,$F0,$DF,$0D,$5F,$F5,$6F,$12,$71,$5F
	.dc.b	$E7,$4D,$DF,$0D,$FC,$07,$D7,$FE,$40,$4C,$3F,$88,$BF,$C8,$FD,$7F
	.dc.b	$DF,$7C,$1C,$9F,$DA,$47,$F0,$93,$F8,$17,$C2,$48,$C9,$0F,$D7,$C7
	.dc.b	$37,$F8,$EB,$5C,$98,$21,$82,$1F,$B5,$78,$3F,$B7,$F6,$7F,$9C,$11
	.dc.b	$FE,$52,$F5,$CB,$FA,$74,$F9,$E7,$F5,$9F,$0D,$5F,$8F,$16,$7E,$37
	.dc.b	$C3,$7F,$25,$F7,$FF,$A2,$12,$BF,$9A,$BF,$D1,$FD,$FF,$E4,$5F,$09
	.dc.b	$27,$F8,$C4,$7F,$2D,$3F,$93,$7B,$D2,$32,$43,$F8,$9C,$73,$7F,$A0
	.dc.b	$B5,$C9,$82,$18,$21,$FC,$65,$E0,$FF,$1B,$F8,$0F,$EB,$84,$7F,$A6
	.dc.b	$BD,$71,$FE,$E5,$3E,$79,$FE,$21,$F0,$B5,$FC,$5C,$59,$FC,$57,$C3
	.dc.b	$7F,$49,$F8,$2F,$ED,$04,$AF,$EB,$2F,$F7,$7F,$25,$F1,$3F,$D1,$FE
	.dc.b	$1E,$57,$F9,$26,$72,$D9,$FE,$99,$F0,$F2,$30,$43,$FC,$0D,$3C,$1B
	.dc.b	$3D,$C8,$FF,$82,$63,$4F,$F8,$79,$AF,$F8,$8D,$AF,$F9,$B1,$AF,$F6
	.dc.b	$48,$E6,$F6,$3A,$7C,$37,$B4,$8E,$0F,$04,$7F,$74,$9F,$36,$5F,$EE
	.dc.b	$46,$FA,$BF,$E3,$3E,$AB,$7E,$1B,$E0,$47,$FA,$16,$1C,$F2,$F9,$84
	.dc.b	$8F,$73,$F3,$F2,$FF,$01,$FD,$2B,$82,$9E,$38,$95,$FE,$79,$96,$BC
	.dc.b	$0C,$C3,$5F,$45,$75,$8F,$98,$D7,$E9,$B5,$F9,$CD,$7E,$73,$5F,$9C
	.dc.b	$D7,$FC,$23,$C1,$F8,$72,$3F,$65,$ED,$3E,$28,$FC,$44,$FB,$B8,$FC
	.dc.b	$F3,$99,$F4,$6D,$E0,$7D,$DC,$3F,$4F,$87,$F2,$84,$8F,$C3,$3F,$97
	.dc.b	$E3,$F9,$3C,$B2,$7F,$2D,$1F,$6C,$F9,$BC,$F2,$32,$43,$F2,$B1,$CD
	.dc.b	$FC,$93,$5C,$98,$21,$82,$1F,$97,$78,$3F,$99,$E5,$FD,$61,$1F,$CF
	.dc.b	$3D,$72,$FC,$74,$F9,$E7,$EF,$77,$D7,$E2,$9F,$9F,$8C,$4B,$3F,$F4
	.dc.b	$37,$03,$AF,$F3,$5D,$3F,$BB,$26,$1F,$AD,$7F,$7D,$D3,$F9,$0F,$0A
	.dc.b	$4F,$EF,$63,$F4,$47,$E5,$3B,$A4,$64,$87,$EF,$23,$9B,$FB,$96,$B9
	.dc.b	$30,$43,$04,$3F,$7C,$F0,$7F,$80,$76,$FF,$13,$23,$FC,$15,$EB,$9F
	.dc.b	$ED,$67,$CF,$3F,$A5,$F1,$2B,$F2,$0F,$F3,$3C,$62,$7E,$44,$7F,$BD
	.dc.b	$33,$78,$9F,$C0,$7B,$FF,$91,$12,$7F,$8A,$3F,$C9,$7C,$2F,$DB,$F9
	.dc.b	$32,$7F,$92,$47,$F0,$93,$F7,$7E,$34,$8C,$90,$FE,$47,$1C,$DF,$E4
	.dc.b	$0D,$72,$60,$86,$08,$7F,$25,$78,$3F,$C9,$FC,$3F,$E7,$44,$7F,$96
	.dc.b	$3D,$72,$FE,$39,$3E,$79,$FE,$19,$E6,$57,$E4,$78,$89,$C5,$7C,$89
	.dc.b	$69,$FE,$70,$8F,$E4,$DE,$3F,$F4,$92,$4F,$F3,$B7,$FA,$67,$8F,$FC
	.dc.b	$7B,$D1,$93,$FD,$2E,$3F,$98,$1F,$C8,$B1,$91,$92,$1F,$D2,$A3,$9B
	.dc.b	$FD,$1D,$AE,$4C,$10,$C1,$0F,$E9,$8F,$07,$FA,$77,$95,$FD,$84,$8F
	.dc.b	$F5,$37,$AE,$1F,$D0,$E7,$CF,$3F,$CC,$FD,$7A,$FC,$88,$B3,$F1,$F2
	.dc.b	$FF,$6A,$CF,$F6,$9F,$2F,$FB,$71,$1F,$EC,$0F,$F7,$1F,$3B,$FD,$8B
	.dc.b	$ED,$3D,$AA,$7F,$55,$89,$5F,$D0,$FB,$7F,$A4,$98,$7E,$CC,$73,$7F
	.dc.b	$B6,$35,$FE,$E6,$D7,$FB,$73,$5F,$ED,$CD,$7F,$B7,$35,$FF,$68,$FA
	.dc.b	$1F,$E0,$72,$3F,$DE,$5E,$D9,$7F,$69,$9F,$23,$F8,$9C,$1F,$05,$8C
	.dc.b	$3B,$E1,$FE,$0D,$BF,$83,$EF,$7F,$73,$F5,$7F,$C2,$84,$9F,$F0,$32
	.dc.b	$61,$07,$D3,$5C,$BC,$1F,$0B,$F3,$0F,$F7,$8C,$67,$1E,$2D,$AE,$99
	.dc.b	$BE,$17,$E9,$9C,$22,$1D,$CF,$14,$D7,$99,$4F,$19,$5C,$38,$38,$38
	.dc.b	$23,$B2,$1A,$39,$7B,$11,$E2,$D7,$C9,$1E,$B7,$B1,$FC,$8D,$C5,$3E
	.dc.b	$1F,$D7,$39,$2E,$5F,$5F,$E1,$FE,$3F,$B5,$E9,$24,$FA,$1F,$67,$BF
	.dc.b	$F9,$8F,$85,$93,$F5,$6F,$F8,$46,$4A,$FA,$27,$E7,$BE,$16,$56,$F6
	.dc.b	$FA,$73,$A8,$F8,$AF,$66,$90,$F7,$B9,$69,$93,$F0,$95,$E1,$70,$76
	.dc.b	$3E,$8F,$C0,$FD,$52,$38,$3B,$61,$EA,$AF,$24,$7F,$4D,$D5,$F0,$D8
	.dc.b	$AD,$4F,$AC,$FC,$2D,$7E,$F3,$E1,$FF,$23,$F0,$5F,$83,$24,$FE,$09
	.dc.b	$FC,$37,$C1,$FF,$02,$F8,$79,$3C,$CF,$CE,$C7,$E6,$8F,$E0,$FF,$0F
	.dc.b	$23,$5A,$A7,$51,$F8,$E7,$AB,$48,$7E,$21,$CB,$4C,$9F,$29,$CB,$4F
	.dc.b	$CC,$E9,$F9,$D2,$38,$3A,$E7,$F8,$29,$F1,$C7,$C9,$9B,$39,$8F,$4C
	.dc.b	$FF,$22,$F8,$7F,$DB,$9F,$A9,$24,$7E,$A1,$FD,$66,$3F,$CA,$32,$93
	.dc.b	$F0,$A3,$F7,$E7,$F2,$CA,$91,$92,$1F,$A8,$8E,$6F,$ED,$1A,$E4,$C1
	.dc.b	$0C,$10,$FC,$7B,$C1,$FC,$8E,$5F,$C0,$48,$FF,$00,$7A,$E1,$FA,$69
	.dc.b	$F3,$CF,$DF,$C7,$F3,$0F,$EE,$63,$2D,$35,$B7,$F1,$CD,$BF,$87,$12
	.dc.b	$3F,$86,$BF,$C4,$75,$FE,$9B,$CB,$27,$F0,$F1,$FC,$90,$FE,$A3,$C7
	.dc.b	$23,$24,$3F,$86,$C7,$37,$F8,$CB,$5C,$98,$21,$82,$1F,$B4,$78,$3F
	.dc.b	$B6,$E3,$FE,$4C,$47,$F9,$2B,$D7,$3F,$E1,$73,$E7,$9F,$D2,$7D,$3B
	.dc.b	$A7,$11,$FC,$7E,$6D,$7F,$51,$8F,$E8,$3C,$BF,$CC,$C8,$FF,$32,$7F
	.dc.b	$9B,$73,$FF,$4C,$9F,$F1,$06,$0E,$BC,$73,$FF,$50,$8C,$61,$FB,$0B
	.dc.b	$FE,$9A,$69,$FE,$AD,$0E,$DC,$42,$30,$F1,$21,$FC,$EA,$7C,$6F,$F4
	.dc.b	$66,$B9,$30,$43,$04,$3F,$90,$BC,$1F,$E4,$7D,$3F,$D4,$C8,$FF,$52
	.dc.b	$7A,$E3,$FC,$E2,$7D,$73,$F3,$BC,$3A,$FF,$4B,$8B,$3F,$1B,$C0,$FE
	.dc.b	$DB,$DB,$FD,$7C,$95,$FD,$79,$FE,$C5,$DB,$F0,$B2,$78,$D4,$FE,$EB
	.dc.b	$12,$BE,$EF,$87,$B3,$0F,$F5,$B8,$E6,$FF,$6A,$6B,$FD,$89,$AF,$F4
	.dc.b	$26,$BF,$E0,$06,$BF,$DE,$9A,$FF,$82,$BC,$1F,$EF,$04,$7F,$BB,$BD
	.dc.b	$B2,$FE,$BB,$3E,$47,$F0,$FE,$3D,$E0,$ED,$7F,$CD,$2B,$E0,$79,$3F
	.dc.b	$E1,$2F,$0F,$FC,$08,$49,$FF,$01,$BF,$E0,$7F,$13,$E3,$7A,$12,$7F
	.dc.b	$9E,$C7,$F8,$68,$F9,$5E,$6C,$8C,$90,$FF,$01,$C7,$37,$FC,$1E,$D7
	.dc.b	$26,$08,$60,$87,$F6,$A7,$83,$FE,$2F,$FA,$8F,$F4,$23,$72,$FF,$10
	.dc.b	$46,$23,$FC,$6A,$D7,$FC,$4F,$3E,$97,$CD,$A7,$EB,$BD,$4C,$47,$F8
	.dc.b	$5E,$6C,$FE,$CB,$D1,$FD,$4D,$BC,$BE,$E4,$BF,$F1,$94,$65,$B1,$97
	.dc.b	$FC,$71,$19,$6C,$65,$F6,$35,$F4,$BE,$CF,$89,$91,$D9,$AF,$46,$BF
	.dc.b	$15,$AF,$C2,$6B,$FE,$59,$9F,$F4,$2B,$C4,$F4,$F9,$DF,$28,$8E,$08
	.dc.b	$7A,$9C,$B4,$C9,$F5,$5C,$B4,$F8,$BE,$77,$D6,$23,$82,$1F,$19,$CB
	.dc.b	$4C,$9F,$81,$72,$D3,$E8,$F9,$FF,$82,$23,$82,$1F,$49,$CB,$4C,$9F
	.dc.b	$8D,$72,$D3,$EE,$FC,$47,$E3,$88,$E0,$87,$DE,$72,$D3,$27,$E6,$5C
	.dc.b	$B4,$FC,$4F,$A5,$F9,$A2,$38,$3F,$1B,$E9,$7F,$19,$F6,$24,$7F,$8D
	.dc.b	$72,$4F,$E5,$8F,$E3,$DE,$D4,$9D,$46,$5E,$64,$3F,$2A,$FE,$A8,$8E
	.dc.b	$4F,$D8,$38,$76,$64,$FD,$7B,$FB,$2F,$5A,$47,$E9,$1A,$F0,$1C,$1F
	.dc.b	$EF,$DB,$F4,$C7,$98,$FE,$DB,$F0,$3E,$77,$7C,$9D,$9D,$18,$3F,$AF
	.dc.b	$FB,$DF,$B6,$F6,$3F,$81,$92,$3F,$6A,$FF,$06,$F6,$BF,$A1,$7C,$24
	.dc.b	$9F,$95,$1F,$BE,$3F,$A3,$7C,$14,$8C,$90,$FD,$DC,$73,$7F,$80,$B5
	.dc.b	$C9,$82,$18,$21,$FC,$6D,$E0,$FF,$1D,$F7,$3F,$8D,$91,$FE,$1A,$F5
	.dc.b	$D3,$E4,$4F,$9E,$7F,$C2,$7F,$0D,$5F,$88,$35,$4E,$6B,$FC,$1B,$F0
	.dc.b	$DF,$C7,$3E,$03,$F9,$49,$2F,$F8,$DB,$FC,$B3,$E0,$3F,$C1,$3F,$52
	.dc.b	$F8,$7A,$7F,$24,$89,$5F,$E8,$79,$7C,$3F,$03,$31,$E4,$CD,$5D,$63
	.dc.b	$F9,$4B,$5F,$E4,$ED,$7F,$A0,$35,$FE,$7A,$D7,$F8,$E3,$5F,$81,$F0
	.dc.b	$BF,$D1,$08,$FF,$36,$7B,$67,$F9,$19,$F2,$AE,$3F,$F7,$EE,$2F,$87
	.dc.b	$CC,$78,$30,$FD,$85,$FC,$4F,$87,$FE,$8B,$F0,$FF,$D4,$89,$3F,$D1
	.dc.b	$1F,$EA,$B7,$F2,$B2,$93,$FB,$98,$FE,$9A,$7C,$E9,$91,$92,$1F,$D2
	.dc.b	$A3,$9B,$FD,$41,$AE,$4C,$10,$C1,$0F,$ED,$4F,$07,$FB,$65,$7F,$6A
	.dc.b	$23,$FD,$71,$EB,$97,$ED,$E7,$CF,$3F,$7F,$6A,$FD,$D8,$B3,$F2,$E0
	.dc.b	$33,$FC,$0B,$FC,$67,$6F,$EF,$24,$7F,$B5,$BF,$DF,$36,$FC,$8F,$04
	.dc.b	$9F,$E4,$11,$FD,$D0,$FC,$AF,$14,$8C,$90,$FE,$E3,$1C,$DF,$EE,$ED
	.dc.b	$72,$60,$86,$08,$7F,$84,$5E,$0F,$B0,$F6,$FD,$53,$8F,$11,$23,$0E
	.dc.b	$0D,$C5,$69,$E0,$CB,$F9,$25,$7B,$63,$F5,$9D,$35,$FD,$54,$59,$F9
	.dc.b	$D1,$F3,$79,$FF,$CF,$F8,$39,$7B,$4A,$30,$F8,$5A,$F9,$5A,$FA,$79
	.dc.b	$FF,$80,$F6,$C8,$FB,$78,$78,$29,$5E,$03,$F8,$3F,$5C,$AD,$F3,$FC
	.dc.b	$7A,$DC,$60,$F9,$35,$E6,$8C,$C7,$8E,$30,$47,$D0,$D7,$46,$08,$60
	.dc.b	$87,$D3,$E9,$EC,$47,$E1,$3C,$F0,$F6,$4F,$AE,$7F,$8E,$77,$D7,$F9
	.dc.b	$6D,$F9,$AB,$71,$FC,$87,$BF,$E8,$F6,$FA,$C9,$1F,$3D,$F7,$76,$FF
	.dc.b	$2C,$F0,$64,$FC,$33,$EB,$47,$DA,$3F,$98,$F8,$32,$35,$8A,$75,$1F
	.dc.b	$6D,$EA,$D2,$1F,$75,$CB,$4C,$9F,$97,$72,$D3,$F0,$DE,$0F,$D5,$23
	.dc.b	$83,$AE,$5F,$83,$9F,$1C,$7F,$52,$E9,$FF,$30,$4E,$63,$D3,$3A,$8F
	.dc.b	$CC,$CF,$F9,$51,$FC,$67,$83,$F8,$32,$57,$E5,$DF,$C3,$78,$DF,$ED
	.dc.b	$CF,$AB,$78,$F4,$FD,$0C,$4A,$FE,$A9,$C7,$FD,$68,$C3,$F6,$A3,$9B
	.dc.b	$FA,$36,$BF,$A3,$6B,$FB,$B6,$BF,$9B,$6B,$FC,$09,$AF,$EB,$FC,$7F
	.dc.b	$CE,$11,$FD,$83,$DB,$1F,$D5,$4F,$91,$FD,$DE,$0F,$3B,$18,$77,$C3
	.dc.b	$F0,$37,$F0,$7C,$BF,$DF,$F9,$7F,$A9,$24,$FE,$F5,$FD,$67,$97,$F1
	.dc.b	$FD,$29,$3F,$8A,$8F,$E0,$E7,$CB,$F3,$A4,$64,$87,$F0,$58,$E6,$FF
	.dc.b	$09,$6B,$93,$04,$30,$43,$F9,$23,$C1,$FE,$4D,$E7,$FF,$01,$23,$FC
	.dc.b	$55,$EB,$AF,$F0,$F9,$F3,$CF,$E9,$F5,$7A,$98,$8F,$DA,$CD,$AF,$97
	.dc.b	$EB,$7F,$24,$F4,$7F,$87,$12,$BF,$91,$3F,$C4,$7D,$3F,$C6,$7B,$92
	.dc.b	$7F,$5F,$1F,$CB,$0F,$C8,$7B,$52,$32,$43,$F9,$54,$73,$7F,$97,$35
	.dc.b	$C9,$82,$18,$21,$FD,$25,$E0,$FF,$4B,$F5,$7F,$93,$11,$FE,$74,$F5
	.dc.b	$C7,$F9,$A4,$F9,$E7,$E3,$7B,$F5,$FE,$33,$16,$7F,$1F,$EE,$7F,$48
	.dc.b	$F6,$BF,$99,$12,$BF,$A3,$3F,$CD,$7D,$AF,$D8,$7C,$0C,$9F,$E2,$71
	.dc.b	$FD,$40,$FD,$A7,$C1,$C8,$C9,$0F,$E9,$D1,$CD,$FE,$A4,$D7,$26,$08
	.dc.b	$60,$87,$F6,$D7,$83,$FD,$BF,$DE,$FE,$94,$47,$FA,$F3,$D7,$1F,$EB
	.dc.b	$13,$E7,$9F,$C5,$7C,$1D,$7F,$9E,$C5,$9F,$D9,$FC,$37,$F6,$CF,$80
	.dc.b	$FE,$AC,$4A,$FE,$D4,$FF,$59,$F8,$1F,$E2,$9F,$0D,$27,$F9,$BC,$7F
	.dc.b	$74,$3F,$8C,$7C,$34,$8C,$90,$FE,$E5,$1C,$DF,$EE,$CD,$72,$60,$86
	.dc.b	$08,$7F,$84,$5E,$0F,$F8,$52,$7F,$C0,$69,$A4,$7B,$61,$FD,$F6,$7D
	.dc.b	$11,$FA,$FF,$88,$AF,$F6,$28,$B3,$FC,$63,$E1,$FF,$C4,$8F,$F8,$52
	.dc.b	$DF,$0F,$FE,$12,$6D,$FF,$88,$5A,$FF,$89,$5A,$F9,$FE,$1F,$F9,$F4
	.dc.b	$C8,$FE,$CD,$29,$EF,$E2,$7F,$8C,$E3,$13,$FC,$16,$7F,$A3,$1A,$7F
	.dc.b	$CA,$71,$A7,$FD,$4E,$D7,$C0,$79,$63,$D0,$47,$04,$3D,$0E,$5A,$64
	.dc.b	$E6,$7B,$72,$F8,$44,$7E,$13,$C7,$0F,$91,$3E,$79,$FA,$CD,$F4,$19
	.dc.b	$E7,$0F,$55,$F0,$BF,$8B,$7F,$D9,$5D,$67,$F4,$48,$FD,$17,$EA,$57
	.dc.b	$5E,$19,$3F,$3A,$3B,$1E,$1D,$A4,$64,$87,$D9,$8E,$6F,$DD,$6B,$93
	.dc.b	$04,$30,$43,$F2,$CF,$07,$F3,$1B,$7E,$24,$8F,$E2,$5E,$B8,$FE,$3E
	.dc.b	$7C,$F3,$FE,$2B,$CD,$C9,$88,$FB,$F3,$6B,$FC,$D3,$4E,$1F,$99,$C5
	.dc.b	$F9,$72,$5F,$E5,$DF,$CD,$71,$7D,$3E,$69,$3F,$94,$8F,$A8,$7D,$7E
	.dc.b	$89,$19,$21,$F9,$F8,$E6,$FE,$8D,$AE,$4C,$10,$C1,$0F,$DE,$3C,$1F
	.dc.b	$DF,$73,$7E,$BC,$8F,$EB,$DE,$B9,$7E,$D6,$7C,$F3,$F1,$FC,$0A,$FE
	.dc.b	$92,$2C,$FC,$6E,$CF,$C8,$74,$7E,$EC,$95,$FB,$B7,$F7,$DD,$5F,$97
	.dc.b	$F0,$E4,$FE,$E2,$3F,$30,$7E,$6F,$BE,$46,$48,$7F,$03,$8E,$6F,$F0
	.dc.b	$76,$B9,$30,$43,$04,$3F,$90,$3C,$1F,$E4,$5D,$9F,$C4,$C8,$FF,$13
	.dc.b	$7A,$E3,$FC,$66,$7C,$F3,$F8,$BF,$16,$BF,$C2,$22,$CF,$E2,$BC,$7F
	.dc.b	$DA,$78,$1F,$C7,$C9,$5F,$C7,$DF,$E4,$5E,$07,$EE,$FC,$C9,$3F,$C7
	.dc.b	$23,$F7,$87,$EF,$FC,$A9,$19,$21,$FC,$9E,$39,$BF,$CA,$DA,$E4,$C1
	.dc.b	$0C,$10,$FE,$88,$F0,$7F,$A3,$78,$9F,$CD,$C8,$FF,$37,$7A,$E3,$FC
	.dc.b	$F6,$7C,$F3,$FA,$FF,$42,$BF,$CB,$22,$CF,$EB,$BC,$DF,$E3,$1E,$37
	.dc.b	$F4,$32,$57,$F4,$37,$FA,$37,$93,$FC,$7F,$D2,$93,$FD,$02,$3F,$90
	.dc.b	$1F,$C8,$FD,$39,$19,$21,$FD,$2E,$39,$BF,$D3,$DA,$E4,$C1,$0C,$10
	.dc.b	$FE,$D0,$F0,$7F,$B5,$79,$7F,$D6,$C8,$FF,$5B,$7A,$E3,$FD,$86,$7C
	.dc.b	$F3,$FC,$4F,$DA,$AF,$F5,$08,$B3,$FC,$4B,$D7,$FE,$79,$E7,$FF,$67
	.dc.b	$25,$7F,$67,$6B,$FD,$9D,$AF,$F7,$D6,$BF,$D8,$DA,$FF,$60,$7F,$B9
	.dc.b	$79,$F2,$3F,$B9,$35,$FE,$E4,$D7,$FC,$10,$D7,$FB,$73,$5F,$ED,$6D
	.dc.b	$7F,$C1,$9E,$87,$F7,$F2,$3F,$DF,$D3,$08,$3E,$B1,$EF,$FF,$61,$5F
	.dc.b	$03,$4C,$46,$7E,$02,$57,$FB,$A6,$91,$C7,$DD,$FE,$D5,$F5,$AF,$53
	.dc.b	$BD,$F8,$63,$0E,$03,$87,$6D,$7C,$F7,$E6,$7C,$18,$F9,$5F,$16,$3E
	.dc.b	$45,$B7,$3A,$F2,$C7,$9F,$2F,$33,$FF,$00,$70,$7C,$47,$8E,$3C,$6D
	.dc.b	$8C,$C7,$9D,$AF,$9A,$34,$47,$4D,$74,$60,$87,$BB,$E2,$3B,$11,$EC
	.dc.b	$D7,$46,$08,$60,$87,$DC,$94,$71,$8C,$7B,$9E,$B2,$3E,$B6,$BF,$41
	.dc.b	$AF,$03,$44,$34,$53,$C1,$EE,$7C,$72,$3F,$1D,$AF,$DC,$6B,$C0,$D1
	.dc.b	$0D,$14,$F6,$7B,$BF,$4C,$8F,$D3,$7E,$B7,$A1,$F9,$9F,$86,$93,$F7
	.dc.b	$63,$F2,$67,$E7,$7E,$1A,$46,$48,$7E,$42,$39,$BF,$7D,$AE,$4C,$10
	.dc.b	$C1,$0F,$A4,$F0,$7E,$A7,$C0,$FE,$2C,$8F,$E2,$DE,$BB,$FE,$87,$45
	.dc.b	$7A,$2B,$F2,$DF,$09,$5F,$CE,$F1,$FE,$AE,$71,$7F,$9F,$F8,$7F,$D8
	.dc.b	$FC,$1F,$E6,$49,$7F,$99,$7B,$DF,$18,$F8,$35,$E1,$C3,$F7,$3F,$0B
	.dc.b	$FC,$0F,$E1,$E5,$7E,$9E,$32,$FE,$8D,$FD,$31,$1F,$89,$5E,$BC,$BC
	.dc.b	$B8,$7F,$01,$C3,$8D,$7E,$FA,$F8,$34,$FD,$77,$C3,$7E,$C4,$8E,$0F
	.dc.b	$D9,$4F,$47,$F8,$EE,$9F,$21,$F8,$E7,$9C,$87,$C8,$37,$5F,$C7,$E7
	.dc.b	$F8,$BF,$C3,$FE,$FC,$93,$FB,$F7,$F8,$14,$77,$8F,$E2,$F8,$FF,$1E
	.dc.b	$89,$1F,$C8,$E7,$F9,$51,$87,$C0,$7F,$1E,$8E,$6F,$A8,$87,$F2,$07
	.dc.b	$2D,$32,$7E,$F5,$CB,$4C,$9F,$CB,$73,$FE,$30,$47,$F8,$C3,$D3,$3F
	.dc.b	$BD,$1F,$18,$F2,$E3,$FD,$01,$C4,$69,$CD,$7F,$45,$CB,$F9,$EE,$BF
	.dc.b	$C9,$49,$3F,$C9,$5E,$DA,$7D,$F8,$E6,$FF,$41,$E9,$FE,$8B,$C3,$FD
	.dc.b	$3F,$69,$5F,$CC,$E3,$2F,$F2,$F7,$F9,$91,$1F,$E6,$53,$F3,$72,$F3
	.dc.b	$1F,$A9,$78,$C7,$F4,$A9,$E0,$D3,$F9,$D7,$0F,$F3,$D2,$38,$3A,$9E
	.dc.b	$83,$F1,$B9,$FF,$5C,$AE,$B7,$FB,$7B,$B8,$FF,$6D,$E8,$70,$FF,$62
	.dc.b	$E9,$FE,$94,$4C,$BF,$A5,$3F,$D3,$71,$F5,$5F,$C7,$9F,$A7,$1D,$DF
	.dc.b	$ED,$DC,$7F,$DD,$4C,$3A,$3C,$E7,$F6,$88,$E8,$E0,$FE,$88,$E1,$D1
	.dc.b	$93,$EA,$14,$FD,$5B,$FD,$87,$A6,$47,$F6,$16,$BC,$8F,$11,$33,$FE
	.dc.b	$F5,$F5,$CF,$37,$CF,$1F,$56,$7F,$B8,$5F,$F8,$3F,$9B,$FC,$0F,$D9
	.dc.b	$FD,$BC,$93,$FD,$BD,$F4,$57,$18,$ED,$9F,$F8,$6F,$1E,$D1,$FE,$1E
	.dc.b	$9F,$0F,$85,$FE,$1B,$E9,$93,$9B,$E1,$3F,$EF,$FB,$3F,$E0,$16,$CF
	.dc.b	$F8,$55,$AF,$F8,$69,$AF,$CC,$3D,$64,$3F,$7E,$F9,$1B,$1D,$48,$F5
	.dc.b	$7C,$1E,$0C,$8F,$03,$5F,$9D,$87,$C1,$25,$FC,$1D,$3D,$57,$2B,$DC
	.dc.b	$7F,$90,$30,$F4,$DF,$F8,$8F,$AF,$E0,$78,$BE,$F2,$57,$BD,$F9,$F8
	.dc.b	$FE,$12,$3E,$79,$CF,$1F,$91,$E2,$FF,$91,$FB,$A4,$F8,$06,$49,$DC
	.dc.b	$7C,$B7,$E7,$11,$FC,$CE,$5E,$C7,$F2,$27,$39,$F9,$51,$F8,$C9,$E1
	.dc.b	$3F,$4D,$AF,$F9,$72,$5E,$4F,$DB,$24,$7D,$B7,$EF,$61,$F9,$4C,$7D
	.dc.b	$D8,$7E,$62,$3C,$F5,$F6,$B0,$B3,$FE,$4B,$F0,$3F,$11,$E4,$7E,$1C
	.dc.b	$95,$F8,$77,$AB,$F8,$39,$F6,$3F,$12,$FF,$21,$E6,$7F,$9B,$FC,$39
	.dc.b	$3E,$F8,$CB,$F9,$17,$F2,$84,$7F,$29,$3E,$97,$E3,$9F,$90,$78,$BF
	.dc.b	$97,$8E,$0D,$3F,$39,$E6,$7E,$7C,$8E,$0E,$A7,$A7,$3F,$8A,$7E,$3D
	.dc.b	$F3,$10,$FC,$FC,$FF,$A2,$63,$F5,$9E,$7F,$EA,$C9,$3F,$AB,$7E,$2B
	.dc.b	$F8,$C3,$FD,$03,$CF,$FB,$3C,$24,$7E,$2F,$EB,$DE,$3F,$FA,$67,$4F
	.dc.b	$F4,$34,$62,$7F,$0D,$5C,$1F,$DB,$3D,$9F,$C2,$35,$F5,$1A,$3D,$E5
	.dc.b	$21,$C0,$C9,$F8,$DF,$4F,$F8,$31,$1F,$E0,$CF,$B2,$BF,$15,$1E,$43
	.dc.b	$D5,$E5,$79,$63,$31,$F5,$CF,$77,$95,$FC,$4F,$D3,$FE,$24,$48,$FE
	.dc.b	$24,$FF,$15,$F5,$FE,$16,$DF,$EB,$BC,$DE,$6E,$2F,$FD,$7F,$18,$7F
	.dc.b	$55,$5F,$C7,$CD,$28,$79,$D8,$6C,$70,$21,$FC,$86,$7C,$4F,$F2,$56
	.dc.b	$B9,$30,$43,$04,$3F,$6C,$F0,$7F,$71,$EB,$7F,$31,$23,$FC,$C5,$EB
	.dc.b	$9F,$F4,$C9,$F5,$4F,$F0,$FF,$B0,$7C,$36,$23,$F9,$8C,$DA,$FC,$3F
	.dc.b	$A5,$FC,$FF,$DC,$FE,$7C,$4A,$FE,$7C,$F6,$D7,$FA,$65,$F1,$7F,$DD
	.dc.b	$18,$F6,$AF,$E2,$9E,$E7,$C7,$F5,$A4,$60,$87,$F4,$78,$FE,$98,$47
	.dc.b	$E2,$65,$EB,$7E,$6B,$FE,$E1,$AE,$D1,$FD,$2A,$78,$3C,$CF,$EA,$8F
	.dc.b	$F5,$7F,$80,$91,$FD,$5D,$AF,$D0,$AE,$CF,$C4,$3E,$46,$9F,$D8,$6A
	.dc.b	$47,$A2,$7F,$5B,$EB,$FF,$65,$F7,$FF,$B2,$12,$7F,$B2,$3F,$1D,$FE
	.dc.b	$9D,$A7,$C7,$43,$2F,$5D,$FF,$81,$3E,$0F,$F1,$DE,$D4,$AF,$E7,$D1
	.dc.b	$FD,$CC,$FE,$E4,$47,$FB,$93,$FE,$F5,$AE,$7B,$7D,$69,$F9,$2F,$D6
	.dc.b	$D3,$94,$70,$21,$FD,$E6,$BF,$BF,$7C,$1C,$8F,$EF,$CD,$7D,$27,$33
	.dc.b	$A9,$EE,$3E,$49,$E8,$21,$FE,$00,$8F,$D5,$7B,$BF,$E0,$DF,$86,$FF
	.dc.b	$06,$12,$7F,$C1,$8F,$AE,$FE,$44,$7F,$68,$D3,$F6,$B9,$7B,$A3,$E5
	.dc.b	$BE,$47,$FD,$17,$CA,$F7,$BF,$70,$70,$88,$6C,$E2,$7C,$92,$19,$3E
	.dc.b	$B5,$F8,$BC,$86,$1E,$28,$E7,$3C,$F0,$E0,$F4,$39,$38,$7F,$01,$F0
	.dc.b	$FF,$5C,$8E,$9F,$80,$FC,$C3,$E0,$1F,$30,$FF,$53,$C7,$BF,$83,$F4
	.dc.b	$91,$E2,$7F,$A0,$FC,$07,$C6,$F8,$7F,$C2,$12,$7C,$6F,$67,$F4,$04
	.dc.b	$9F,$D7,$61,$ED,$8F,$E9,$DF,$07,$FC,$84,$C3,$E2,$8E,$4F,$D1,$C3
	.dc.b	$F6,$AD,$8C,$BF,$87,$3E,$33,$87,$EC,$B5,$F9,$E7,$C8,$7F,$39,$F0
	.dc.b	$F2,$3E,$5B,$5E,$67,$C5,$BF,$4B,$FE,$2F,$FB,$17,$07,$A2,$73,$3E
	.dc.b	$93,$FB,$57,$C2,$FE,$33,$E1,$FF,$54,$48,$FB,$2F,$C7,$9E,$24,$E6
	.dc.b	$FD,$1E,$1D,$AB,$F9,$87,$C4,$7F,$47,$F8,$89,$18,$21,$F8,$78,$FD
	.dc.b	$C1,$1F,$80,$7A,$EB,$E4,$59,$EF,$97,$E8,$AB,$85,$73,$3F,$1E,$FF
	.dc.b	$03,$B9,$1F,$92,$6B,$F2,$0F,$84,$4B,$EC,$FF,$16,$C3,$D8,$FE,$42
	.dc.b	$BC,$0F,$AA,$7F,$C1,$F1,$FB,$4C,$BF,$89,$12,$7F,$42,$FC,$9C,$7D
	.dc.b	$64,$8E,$D8,$FE,$F3,$6F,$ED,$D9,$C9,$FE,$B5,$19,$7F,$56,$FF,$1F
	.dc.b	$23,$F2,$0F,$55,$75,$59,$7A,$B2,$FE,$95,$5C,$27,$F7,$0D,$7F,$35
	.dc.b	$C1,$FC,$AC,$8F,$EE,$9F,$53,$F9,$83,$EB,$1E,$A3,$CE,$43,$F6,$F1
	.dc.b	$FE,$09,$D7,$F8,$CF,$27,$F3,$72,$4F,$F0,$67,$AB,$FA,$23,$D6,$75
	.dc.b	$3F,$80,$F3,$7F,$84,$F8,$24,$7F,$7F,$8F,$E4,$A7,$F4,$32,$3F,$C4
	.dc.b	$9E,$BC,$3F,$CA,$63,$B6,$5F,$80,$7E,$3E,$DC,$9E,$04,$3F,$8F,$57
	.dc.b	$F4,$FE,$99,$1F,$C7,$9A,$FE,$36,$BF,$4C,$FC,$70,$F7,$D7,$EA,$6D
	.dc.b	$FE,$0A,$BF,$05,$7B,$8F,$F2,$F7,$1F,$F4,$0E,$BF,$EB,$C4,$8F,$E5
	.dc.b	$AF,$B3,$4F,$DF,$5F,$E9,$A7,$FC,$45,$DD,$FE,$39,$E6,$91,$82,$1F
	.dc.b	$CE,$63,$FB,$51,$1F,$C6,$67,$EA,$7E,$3D,$75,$7F,$99,$CE,$DE,$47
	.dc.b	$F4,$07,$FB,$A7,$81,$23,$FA,$1B,$5F,$EA,$5A,$7C,$27,$E3,$5F,$78
	.dc.b	$C4,$7F,$81,$EB,$FB,$09,$1F,$F4,$11,$FD,$93,$C3,$FF,$01,$11,$FE
	.dc.b	$A2,$D7,$FA,$B3,$5F,$E7,$0D,$7F,$C7,$6D,$7F,$B6,$BF,$E0,$DF,$0E
	.dc.b	$47,$F5,$C6,$BF,$D8,$5A,$FF,$43,$6B,$FE,$46,$6B,$FD,$E1,$AF,$F8
	.dc.b	$8F,$C6,$FF,$0D,$11,$FE,$D2,$D7,$FB,$73,$5F,$EA,$0D,$7F,$CA,$6D
	.dc.b	$7F,$C0,$AF,$F8,$97,$C6,$91,$FD,$D1,$AF,$F7,$96,$BF,$D6,$DA,$FF
	.dc.b	$97,$9A,$FF,$84,$1A,$FF,$8E,$FC,$9F,$F1,$91,$1F,$F0,$0B,$5F,$F0
	.dc.b	$33,$5F,$ED,$0D,$7F,$CD,$6D,$7F,$C3,$AF,$F8,$33,$CF,$FD,$B4,$7F
	.dc.b	$87,$DA,$FF,$8D,$1A,$FF,$93,$9A,$9A,$4D,$7A,$12,$3F,$C4,$11,$96
	.dc.b	$C7,$87,$CF,$91,$A6,$BF,$E2,$B9,$7C,$27,$FA,$AB,$57,$97,$4F,$F5
	.dc.b	$6C,$EA,$BF,$CC,$7B,$FF,$11,$87,$A3,$C7,$C6,$95,$AC,$FD,$D3,$C8
	.dc.b	$F8,$84,$8F,$4B,$F1,$F2,$F0,$47,$F8,$0F,$1F,$0C,$71,$8D,$90,$F5
	.dc.b	$4F,$CF,$F5,$24,$7C,$46,$BE,$F9,$EA,$FF,$1C,$BF,$F0,$1D,$F6,$8F
	.dc.b	$41,$F4,$EF,$C0,$EC,$C1,$FE,$17,$F1,$FE,$2F,$AD,$F7,$C9,$1F,$51
	.dc.b	$F6,$3E,$B2,$1C,$CF,$BD,$87,$CE,$F6,$7E,$97,$95,$27,$C6,$7E,$0E
	.dc.b	$34,$FE,$01,$FC,$71,$1F,$C1,$CF,$BB,$6F,$E7,$19,$7B,$67,$C1,$B2
	.dc.b	$F8,$47,$E0,$DA,$FE,$5F,$DB,$FC,$D1,$1F,$C7,$BE,$C7,$D5,$C1,$F8
	.dc.b	$87,$DB,$87,$E1,$71,$F3,$C7,$82,$EC,$FF,$4B,$F3,$7F,$1B,$EE,$FE
	.dc.b	$9C,$95,$F9,$C7,$B4,$FA,$A3,$AF,$07,$EA,$70,$FC,$AF,$BF,$F9,$9F
	.dc.b	$3E,$57,$C8,$D5,$3A,$8F,$D3,$BF,$B5,$23,$F2,$0E,$A7,$F8,$AF,$6F
	.dc.b	$91,$8F,$E7,$E7,$73,$FA,$96,$BF,$BD,$F8,$4F,$E0,$04,$7F,$6A,$FA
	.dc.b	$B2,$FC,$7C,$76,$3F,$02,$79,$E7,$CC,$FD,$67,$F6,$9E,$8F,$EC,$BE
	.dc.b	$17,$F8,$61,$27,$F8,$0B,$D5,$F4,$1D,$72,$FD,$C3,$F8,$0F,$88,$FD
	.dc.b	$F7,$A9,$23,$E3,$47,$EF,$8F,$E3,$04,$7F,$86,$BD,$5F,$F1,$AE,$9F
	.dc.b	$8D,$CB,$E9,$47,$23,$64,$3F,$87,$4F,$F2,$3F,$85,$91,$FC,$61,$AF
	.dc.b	$ED,$4F,$89,$C7,$F8,$B7,$F2,$FC,$1E,$57,$B1,$EB,$3F,$D1,$1E,$9F
	.dc.b	$F1,$6F,$88,$FE,$60,$48,$FE,$4A,$FA,$6B,$A7,$0F,$E2,$E7,$A9,$F8
	.dc.b	$7B,$FE,$45,$EB,$C8,$C1,$0F,$31,$0F,$E7,$91,$FA,$36,$9F,$8B,$34
	.dc.b	$7C,$62,$9F,$CF,$DE,$0F,$F4,$3F,$86,$FE,$90,$47,$F9,$EB,$F4,$77
	.dc.b	$FB,$77,$FA,$D9,$FF,$54,$7C,$0F,$E5,$B6,$ED,$39,$9F,$DA,$FB,$5F
	.dc.b	$CE,$BE,$1F,$FA,$A1,$23,$FA,$4B,$D7,$2E,$51,$FE,$8B,$C3,$FA,$0F
	.dc.b	$C3,$7F,$45,$F7,$A4,$FE,$B6,$32,$FF,$52,$7F,$B0,$11,$F8,$58,$F8
	.dc.b	$C9,$97,$FA,$CA,$F8,$CF,$F5,$3A,$E0,$D3,$FA,$EF,$BD,$FD,$AC,$8E
	.dc.b	$0E,$B5,$F0,$08,$FA,$4F,$31,$0F,$E4,$33,$FD,$7F,$E0,$3F,$AE,$FC
	.dc.b	$37,$F7,$72,$4F,$F6,$C7,$E2,$3F,$81,$C7,$FC,$C5,$8F,$F6,$4F,$86
	.dc.b	$FE,$D1,$F0,$52,$BF,$88,$46,$5F,$EE,$AF,$F8,$0C,$8F,$A0,$FC,$11
	.dc.b	$D7,$2F,$88,$F2,$38,$90,$FE,$F1,$06,$10,$7D,$92,$F1,$1A,$8C,$38
	.dc.b	$37,$15,$A7,$94,$FE,$2A,$7E,$98,$FE,$7E,$FB,$5F,$E9,$73,$70,$FF
	.dc.b	$43,$70,$F9,$1F,$F4,$B5,$BE,$1F,$B3,$6F,$C4,$D7,$CE,$D7,$F5,$67
	.dc.b	$86,$7F,$12,$FD,$D3,$FB,$54,$3E,$1F,$FC,$35,$F0,$F2,$7F,$B4,$CE
	.dc.b	$5B,$3E,$22,$3E,$27,$EF,$BD,$F3,$FA,$6F,$68,$DB,$E6,$6B,$F4,$65
	.dc.b	$F0,$FE,$E2,$47,$B9,$EB,$87,$AA,$3F,$DB,$FA,$7C,$7C,$3E,$B6,$16
	.dc.b	$7E,$F6,$1E,$97,$C4,$7F,$81,$3A,$3B,$BE,$EC,$BE,$69,$1F,$9A,$F8
	.dc.b	$2F,$F4,$B1,$E0,$3E,$2E,$9F,$E4,$1C,$24,$60,$87,$D4,$8F,$B2,$47
	.dc.b	$F0,$CF,$F1,$49,$EB,$7F,$8E,$9F,$AB,$3B,$79,$1F,$80,$7F,$07,$B4
	.dc.b	$8F,$C1,$B5,$FE,$5F,$9F,$C7,$9E,$D1,$F8,$18,$FC,$6C,$7C,$78,$FD
	.dc.b	$0B,$72,$FF,$35,$47,$C7,$D7,$1C,$1F,$92,$24,$7E,$49,$FC,$AF,$B9
	.dc.b	$CF,$17,$6D,$3A,$47,$5E,$5D,$71,$4C,$3F,$1F,$19,$7F,$3A,$FE,$80
	.dc.b	$8F,$E8,$27,$A3,$FE,$3E,$BE,$EF,$E6,$23,$93,$FA,$49,$E2,$D3,$F5
	.dc.b	$3D,$1F,$AC,$23,$83,$B1,$D7,$0E,$EF,$EA,$2B,$AE,$BF,$B6,$BB,$BF
	.dc.b	$F0,$CC,$5C,$9F,$2F,$BB,$F7,$24,$C7,$F7,$2F,$C8,$7B,$10,$EC,$7E
	.dc.b	$CB,$AF,$E6,$74,$48,$C1,$0F,$E0,$31,$FC,$18,$8F,$A8,$FF,$34,$5F
	.dc.b	$C6,$6E,$9F,$51,$E4,$7E,$CE,$78,$BD,$0F,$E1,$FD,$BF,$C4,$88,$E0
	.dc.b	$ED,$A3,$3F,$85,$7F,$22,$3F,$8D,$EB,$F8,$DC,$25,$FA,$67,$83,$FC
	.dc.b	$D7,$17,$4F,$E4,$FB,$FF,$90,$92,$7F,$90,$BF,$25,$F5,$9F,$26,$BB
	.dc.b	$BF,$C6,$BC,$1F,$CA,$76,$C8,$C1,$0F,$E5,$31,$FC,$B8,$8F,$B3,$4F
	.dc.b	$5B,$D5,$FE,$89,$9F,$E9,$E7,$73,$C8,$FE,$6A,$FF,$38,$F1,$24,$7F
	.dc.b	$38,$6B,$FC,$E2,$7E,$35,$76,$8F,$C0,$9F,$CD,$AB,$B3,$E7,$C3,$E3
	.dc.b	$AC,$DD,$FF,$B9,$F1,$BF,$A3,$92,$7F,$A3,$BF,$1E,$B8,$D7,$E0,$DB
	.dc.b	$BF,$E0,$EB,$FA,$2F,$93,$FB,$BF,$06,$47,$F4,$48,$FE,$0C,$7F,$54
	.dc.b	$23,$FD,$51,$ED,$9F,$AE,$FA,$CF,$ED,$4E,$46,$C8,$7F,$5A,$9F,$EB
	.dc.b	$FE,$5C,$8F,$EB,$ED,$7E,$41,$F4,$CF,$C4,$61,$E9,$AF,$ED,$35,$E6
	.dc.b	$9E,$E7,$E8,$BC,$3F,$E3,$DE,$77,$F6,$C2,$47,$F6,$C7,$D2,$F4,$3B
	.dc.b	$1F,$E0,$6D,$FB,$65,$FB,$FF,$17,$F9,$01,$89,$FE,$A3,$3F,$95,$7F
	.dc.b	$AF,$35,$FC,$49,$FB,$53,$25,$32,$7B,$88,$68,$FF,$0A,$FD,$9F,$F8
	.dc.b	$45,$34,$8F,$A8,$F4,$1F,$44,$FE,$07,$E8,$79,$DC,$9F,$90,$CF,$F0
	.dc.b	$BE,$47,$F4,$5A,$7B,$1F,$E1,$F6,$D7,$F8,$8D,$AF,$92,$3C,$1E,$1B
	.dc.b	$1E,$DC,$4C,$BA,$7B,$7F,$D1,$FC,$C9,$8F,$FB,$E3,$24,$EA,$3B,$3E
	.dc.b	$12,$3F,$B5,$3E,$79,$ED,$D3,$F3,$B7,$C6,$3E,$3C,$F0,$69,$E9,$F6
	.dc.b	$BD,$84,$70,$75,$3E,$2D,$7F,$0C,$7E,$49,$E7,$9E,$D1,$FB,$4B,$F4
	.dc.b	$CF,$F5,$5F,$37,$E5,$7C,$07,$C9,$24,$7C,$97,$E3,$E9,$F1,$B2,$FA
	.dc.b	$1F,$07,$FD,$AF,$D0,$97,$F1,$63,$2F,$D1,$7E,$A1,$1F,$46,$FF,$45
	.dc.b	$F8,$9C,$7F,$D5,$6F,$8C,$7E,$2A,$B8,$34,$FB,$9F,$0B,$F7,$C8,$E0
	.dc.b	$EB,$8F,$9D,$F2,$10,$6B,$F0,$EB,$FC,$09,$E9,$FE,$2F,$E1,$BF,$14
	.dc.b	$4A,$FC,$53,$F0,$9F,$39,$FD,$0F,$1F,$C9,$7C,$37,$F8,$37,$D5,$93
	.dc.b	$F8,$78,$CB,$F9,$37,$F2,$C4,$7C,$E7,$43,$F9,$44,$F0,$7F,$57,$1B
	.dc.b	$69,$F9,$DF,$85,$FD,$01,$1C,$1D,$0F,$D4,$6B,$F3,$5F,$21,$0F,$86
	.dc.b	$FF,$A4,$38,$95,$FE,$27,$F5,$FF,$5D,$F0,$DF,$AD,$25,$FE,$B5,$F8
	.dc.b	$8F,$63,$E2,$1F,$A7,$F8,$6F,$F1,$87,$B7,$23,$04,$3F,$83,$C7,$EE
	.dc.b	$48,$FA,$78,$3F,$38,$7E,$A1,$E0,$7F,$A9,$71,$4F,$08,$E6,$7F,$01
	.dc.b	$F8,$4F,$E0,$84,$70,$75,$CF,$F8,$DD,$F5,$8F,$86,$7F,$AC,$EF,$AB
	.dc.b	$E7,$8F,$D5,$5F,$F9,$2F,$DC,$FE,$25,$F0,$BF,$C4,$49,$5F,$C4,$5A
	.dc.b	$FF,$24,$6B,$E9,$6B,$F4,$5A,$F8,$5F,$E3,$9F,$0B,$23,$F8,$E3,$5F
	.dc.b	$E5,$ED,$7E,$23,$5F,$B8,$D7,$D8,$D7,$F9,$AF,$C2,$7F,$29,$23,$FC
	.dc.b	$A5,$AF,$F3,$C6,$BF,$45,$AF,$E2,$5A,$FC,$97,$F9,$A7,$C2,$C8,$FE
	.dc.b	$68,$D7,$FA,$3B,$5F,$B8,$D7,$F2,$8D,$7E,$A3,$5F,$E9,$BF,$0B,$FD
	.dc.b	$04,$8F,$F4,$16,$BF,$D5,$1A,$FE,$25,$AF,$E8,$5A,$FE,$05,$FE,$99
	.dc.b	$F0,$B2,$3F,$A6,$35,$FE,$BE,$D7,$F2,$8D,$7F,$58,$D7,$F1,$8D,$7F
	.dc.b	$B2,$FC,$2F,$F5,$92,$3F,$D6,$5A,$FF,$6C,$6B,$FA,$16,$BF,$B9,$6B
	.dc.b	$F9,$97,$FB,$27,$C2,$C8,$FE,$C8,$D7,$FB,$BB,$5F,$D6,$35,$FE,$08
	.dc.b	$D7,$F4,$8D,$7F,$BE,$FC,$2F,$F7,$12,$3F,$DC,$5A,$FF,$81,$1A,$FE
	.dc.b	$E5,$AF,$F1,$16,$BF,$B1,$4C,$20,$FB,$5F,$F2,$7B,$86,$BB,$6B,$C9
	.dc.b	$AF,$1E,$1E,$6E,$DA,$ED,$AE,$CC,$C7,$22,$46,$9C,$74,$52,$4F,$06
	.dc.b	$BE,$08,$FF,$08,$39,$21,$98,$DB,$ED,$6B,$ED,$DF,$C0,$47,$CA,$D7
	.dc.b	$B3,$5F,$8C,$D7,$91,$C0,$87,$97,$93,$D8,$47,$E0,$B5,$D1,$82,$18
	.dc.b	$21,$F5,$5E,$0F,$D7,$E6,$F9,$04,$7E,$6B,$5F,$8C,$D7,$81,$A2,$1A
	.dc.b	$29,$F8,$3E,$9F,$A8,$47,$EC,$B5,$FA,$4D,$78,$1A,$21,$A2,$9F,$8F
	.dc.b	$EB,$FC,$01,$1F,$C2,$B5,$FB,$CD,$78,$1A,$21,$A2,$9F,$9B,$F0,$3F
	.dc.b	$18,$47,$F2,$2D,$7F,$14,$D7,$81,$A2,$1A,$29,$FA,$7F,$0B,$F3,$04
	.dc.b	$7F,$3A,$D7,$F2,$CD,$78,$1A,$21,$A2,$9F,$B3,$F1,$7F,$48,$47,$F5
	.dc.b	$2D,$7F,$44,$D7,$81,$A2,$1A,$29,$FB,$FF,$23,$F6,$04,$7F,$6A,$FE
	.dc.b	$E3,$ED,$5C,$5F,$C2,$7A,$BF,$DA,$B1,$8A,$FE,$1B,$18,$9D,$70,$FF
	.dc.b	$0F,$A7,$1E,$2F,$F8,$9C,$38,$F1,$08,$C3,$D2,$B9,$4E,$BC,$8F,$F7
	.dc.b	$2E,$3F,$EE,$6E,$47,$9D,$FC,$24,$98,$7F,$0E,$7D,$59,$F6,$CB,$D1
	.dc.b	$3F,$E1,$63,$C1,$19,$CF,$59,$49,$F1,$1E,$87,$C5,$9F,$A3,$17,$97
	.dc.b	$FC,$64,$95,$FC,$79,$FE,$43,$C4,$F8,$5F,$17,$4C,$AE,$A2,$74,$E2
	.dc.b	$DE,$77,$A7,$AA,$4E,$63,$C4,$7F,$29,$23,$FC,$B9,$F0,$E1,$DE,$3E
	.dc.b	$06,$CC,$7E,$8E,$3E,$09,$F8,$98,$F8,$1E,$44,$3F,$98,$61,$FC,$DF
	.dc.b	$D1,$91,$FC,$F1,$AF,$F4,$4A,$E9,$C3,$F8,$8C,$3C,$4B,$5F,$EA,$93
	.dc.b	$A7,$FA,$86,$1E,$CB,$F7,$1F,$0F,$DD,$FA,$1E,$A7,$F4,$72,$47,$F4
	.dc.b	$C7,$DE,$FC,$CD,$3D,$CD,$CB,$DF,$AF,$58,$F7,$E1,$73,$FC,$66,$3F
	.dc.b	$77,$D7,$FE,$A8,$4A,$FE,$B2,$F6,$7E,$5E,$1E,$CC,$FD,$78,$F3,$7D
	.dc.b	$73,$EA,$3E,$6D,$FE,$27,$D9,$FE,$C0,$48,$FE,$CA,$F5,$C7,$E4,$47
	.dc.b	$A4,$FC,$3E,$BF,$1D,$FE,$E9,$5D,$DF,$EE,$D9,$FE,$56,$3F,$2D,$EE
	.dc.b	$7F,$6D,$24,$FF,$72,$7B,$63,$FB,$7C,$FD,$53,$FE,$08,$C3,$B3,$FE
	.dc.b	$07,$BE,$CF,$F8,$27,$0E,$CF,$D5,$3F,$BE,$C7,$F7,$B2,$3F,$E0,$49
	.dc.b	$E7,$AF,$D4,$8F,$92,$7B,$34,$E6,$FF,$7D,$AE,$2D,$3F,$C1,$3F,$01
	.dc.b	$FE,$07,$23,$83,$B4,$FA,$9F,$E1,$98,$FE,$01,$F3,$1F,$48,$BA,$7E
	.dc.b	$4D,$F8,$13,$FA,$4F,$86,$FD,$C9,$23,$D4,$FC,$87,$F8,$DD,$77,$CB
	.dc.b	$E3,$F1,$32,$F5,$E3,$F8,$CB,$F7,$1F,$6B,$E0,$FF,$82,$91,$F8,$CF
	.dc.b	$57,$D2,$7E,$04,$FE,$31,$7F,$50,$EF,$5D,$B6,$FC,$8C,$FE,$1B,$E1
	.dc.b	$BF,$88,$91,$FA,$4F,$5D,$BE,$96,$3F,$8E,$9E,$BA,$FC,$77,$E3,$12
	.dc.b	$3E,$33,$FC,$5E,$3F,$84,$FC,$3F,$F1,$E2,$47,$DF,$7B,$6B,$F1,$67
	.dc.b	$A5,$7C,$3A,$E9,$8F,$A1,$F0,$9F,$68,$87,$F0,$B9,$FD,$83,$FE,$5A
	.dc.b	$CF,$F8,$C3,$63,$2F,$33,$07,$22,$9F,$C5,$7E,$1B,$F9,$A9,$1F,$CB
	.dc.b	$3F,$46,$7E,$74,$FF,$49,$F2,$3D,$37,$E7,$C3,$E1,$D7,$D3,$F7,$7F
	.dc.b	$95,$F7,$FF,$40,$26,$5F,$A1,$7A,$63,$CA,$3F,$B1,$E3,$5F,$98,$7E
	.dc.b	$18,$FB,$F5,$FC,$DB,$E1,$BF,$A5,$92,$3F,$58,$F4,$78,$63,$FD,$DA
	.dc.b	$BA,$63,$EA,$9F,$E1,$37,$FD,$03,$E1,$7F,$AA,$92,$3F,$6C,$F3,$7E
	.dc.b	$99,$FD,$22,$B9,$E3,$D2,$BF,$A4,$FC,$37,$F5,$E2,$57,$EF,$DE,$58
	.dc.b	$FC,$6A,$F9,$53,$E6,$D3,$E5,$11,$FE,$21,$7F,$D5,$3E,$1B,$FB,$49
	.dc.b	$23,$F8,$53,$D2,$78,$47,$F8,$27,$1F,$2B,$D4,$E6,$7F,$28,$F8,$5F
	.dc.b	$EE,$64,$7F,$8A,$3C,$DF,$E0,$27,$F4,$98,$FE,$CB,$7D,$23,$E9,$9F
	.dc.b	$CC,$BE,$1B,$FB,$E9,$1F,$E3,$CF,$37,$ED,$9F,$24,$FA,$27,$94,$CE
	.dc.b	$5F,$8A,$9F,$EE,$7F,$0D,$FE,$08,$24,$FF,$29,$7A,$6D,$FC,$AB,$0F
	.dc.b	$2C,$F3,$8F,$C9,$3F,$D1,$BE,$17,$FC,$22,$47,$F9,$93,$CB,$5F,$A7
	.dc.b	$1E,$C3,$EF,$E3,$F5,$DF,$39,$FC,$43,$E1,$BF,$C3,$64,$7F,$9E,$3C
	.dc.b	$DF,$90,$70,$8E,$4F,$4D,$7F,$BF,$E7,$E8,$8F,$F0,$33,$FE,$26,$23
	.dc.b	$FC,$02,$78,$92,$7F,$C9,$99,$71,$9F,$E0,$55,$C1,$E6,$7F,$4F,$F8
	.dc.b	$5F,$F1,$89,$1C,$1D,$6B,$F1,$39,$FC,$B8,$F4,$47,$99,$A7,$F9,$63
	.dc.b	$D3,$FF,$13,$FC,$37,$F8,$F0,$91,$FD,$75,$FF,$6F,$FD,$B3,$DB,$FF
	.dc.b	$28,$7B,$12,$BF,$C7,$F3,$97,$FB,$DF,$C3,$FF,$95,$7D,$99,$18,$21
	.dc.b	$FE,$2B,$8F,$F2,$59,$1F,$F7,$7C,$F3,$7F,$C7,$B5,$DD,$FF,$2F,$46
	.dc.b	$77,$FE,$2F,$5C,$67,$A1,$FD,$D5,$FF,$2B,$FC,$0C,$8F,$EF,$0D,$7F
	.dc.b	$DF,$F5,$CF,$4F,$F3,$F6,$1D,$F0,$FE,$8B,$39,$AF,$F0,$56,$1F,$E4
	.dc.b	$E2,$3F,$0B,$2F,$F4,$0C,$A3,$E4,$90,$F6,$9F,$32,$3B,$61,$E0,$8E
	.dc.b	$47,$C6,$BF,$6B,$63,$A2,$AF,$23,$7E,$1F,$EB,$91,$F9,$6F,$77,$FC
	.dc.b	$95,$8F,$81,$F3,$4F,$2D,$99,$7B,$27,$FB,$AB,$EC,$CB,$DB,$F0,$DF
	.dc.b	$84,$24,$FD,$87,$B3,$F6,$0E,$E7,$B0,$E6,$7A,$CF,$96,$67,$3F,$9F
	.dc.b	$F0,$FF,$90,$24,$7E,$11,$EB,$1F,$EB,$8C,$BD,$53,$F0,$EF,$D1,$3F
	.dc.b	$22,$3B,$9F,$88,$8F,$AB,$F0,$FF,$9C,$24,$7E,$41,$EB,$AF,$D3,$8F
	.dc.b	$51,$C4,$F3,$3F,$E7,$EC,$BB,$BF,$53,$4F,$A6,$3E,$17,$F5,$24,$AF
	.dc.b	$CE,$BD,$9F,$F6,$4E,$5E,$A9,$E3,$8F,$49,$F8,$A7,$84,$FD,$0F,$C3
	.dc.b	$7E,$D0,$8F,$EA,$1E,$8F,$FB,$46,$BA,$4F,$D9,$8F,$A0,$7F,$AD,$71
	.dc.b	$C3,$FA,$CF,$86,$FD,$F9,$1F,$D9,$BD,$3C,$65,$F4,$9F,$86,$79,$B5
	.dc.b	$FA,$07,$EA,$30,$FD,$1F,$C2,$FF,$0A,$24,$FE,$F9,$E9,$5C,$E3,$DD
	.dc.b	$5C,$B5,$F9,$D1,$D8,$FD,$07,$C3,$7F,$14,$23,$FC,$21,$E6,$FE,$40
	.dc.b	$E7,$1F,$57,$0F,$99,$1F,$04,$E9,$AF,$F0,$F7,$F5,$A7,$E2,$CF,$E2
	.dc.b	$71,$F9,$C7,$31,$B7,$A1,$A2,$1F,$C3,$23,$83,$A3,$07,$F1,$EF,$84
	.dc.b	$FE,$5A,$47,$07,$2C,$3E,$53,$F0,$73,$4F,$D8,$9C,$CF,$B8,$FE,$21
	.dc.b	$2F,$86,$FE,$26,$7F,$3A,$24,$7F,$2E,$7F,$0F,$87,$E4,$23,$FD,$9D
	.dc.b	$7D,$71,$F4,$61,$D8,$FF,$33,$E1,$D6,$BF,$A0,$E7,$FD,$18,$8F,$F3
	.dc.b	$A7,$AE,$5C,$E3,$CB,$85,$7E,$71,$E9,$15,$D5,$FE,$7F,$CF,$FD,$48
	.dc.b	$8F,$F4,$67,$AB,$F1,$67,$A6,$5E,$B8,$63,$E8,$CF,$F9,$AF,$67,$F5
	.dc.b	$C2,$4F,$F5,$07,$9B,$E3,$C7,$E7,$BF,$86,$CB,$D1,$1E,$5B,$FE,$8B
	.dc.b	$D5,$FD,$98,$95,$FD,$69,$E7,$1F,$27,$5F,$E7,$35,$D2,$BE,$06,$5E
	.dc.b	$7A,$F3,$47,$F6,$CF,$27,$FB,$89,$1F,$EC,$AF,$4B,$FA,$A7,$4D,$3E
	.dc.b	$0E,$5E,$6D,$3F,$A9,$FA,$FF,$DE,$89,$87,$F6,$E7,$93,$FD,$0A,$7F
	.dc.b	$A6,$E3,$F7,$AF,$9C,$FE,$46,$FF,$AE,$FB,$7F,$E0,$42,$47,$F7,$87
	.dc.b	$9B,$F7,$2B,$93,$F2,$4E,$47,$F8,$23,$DB,$FF,$06,$91,$FF,$00,$BC
	.dc.b	$B4,$FE,$BB,$AF,$C7,$AF,$95,$5E,$73,$99,$FE,$04,$F6,$FF,$C2,$E4
	.dc.b	$7F,$C1,$6F,$3C,$FD,$BA,$FC,$79,$F2,$1F,$7F,$3C,$BF,$DE,$7E,$DF
	.dc.b	$FD,$8A,$FF,$04,$6B,$FD,$C9,$AE,$E7,$B3,$FA,$A3,$AE,$1E,$87,$F3
	.dc.b	$B5,$D9,$53,$E1,$BF,$88,$C7,$56,$BE,$36,$BC,$1E,$EF,$E7,$AF,$D8
	.dc.b	$D3,$F8,$1D,$7D,$A7,$07,$88,$FF,$0B,$5F,$FB,$8A,$DF,$0D,$D9,$B3
	.dc.b	$E2,$6B,$E6,$6B,$F1,$63,$81,$1E,$0D,$78,$35,$E0,$75,$F8,$6F,$01
	.dc.b	$1C,$10,$F0,$39,$69,$93,$EB,$39,$69,$E9,$F8,$6F,$61,$1C,$10,$F6
	.dc.b	$39,$69,$93,$F0,$4E,$5A,$7C,$5F,$07,$E4,$11,$C1,$0F,$90,$E5,$A6
	.dc.b	$4F,$C7,$39,$69,$F4,$7E,$1B,$EA,$11,$C1,$0F,$A8,$E5,$A6,$4F,$CD
	.dc.b	$39,$69,$F7,$7E,$1B,$F0,$04,$70,$43,$F0,$0E,$5A,$64,$FD,$33,$96
	.dc.b	$9F,$89,$F8,$6F,$C6,$11,$C1,$0F,$C6,$39,$69,$93,$F6,$4E,$5A,$7E
	.dc.b	$57,$E1,$BF,$30,$47,$04,$3F,$30,$E5,$A6,$4F,$DF,$39,$69,$FA,$1F
	.dc.b	$86,$FD,$21,$1C,$10,$FD,$23,$96,$99,$3F,$85,$39,$69,$FA,$DF,$86
	.dc.b	$FD,$81,$1C,$1F,$B1,$F8,$5F,$E1,$1C,$AF,$02,$9F,$C2,$F8,$13,$4D
	.dc.b	$F2,$CB,$FD,$FF,$27,$F1,$99,$E1,$F0,$DF,$BF,$24,$FE,$FD,$FE,$05
	.dc.b	$1D,$A3,$89,$FE,$0E,$C3,$B3,$83,$F8,$DC,$77,$7F,$87,$E3,$FE,$96
	.dc.b	$C7,$CB,$1B,$5F,$05,$FC,$3C,$91,$FC,$3D,$F0,$BD,$67,$E0,$36,$FC
	.dc.b	$2F,$F1,$CA,$EA,$F8,$32,$F1,$72,$FF,$1D,$C7,$73,$FC,$ED,$EA,$FF
	.dc.b	$1B,$6B,$93,$E8,$10,$D1,$4D,$18,$3F,$92,$7C,$37,$F2,$92,$3F,$CA
	.dc.b	$5E,$F3,$E3,$8F,$85,$3F,$73,$C5,$F5,$CE,$67,$D9,$3F,$80,$F1,$7E
	.dc.b	$F7,$C3,$7F,$35,$24,$FF,$35,$7A,$EF,$F7,$63,$A9,$F2,$70,$EB,$19
	.dc.b	$8E,$A7,$CD,$AF,$C5,$7C,$37,$F4,$12,$47,$F4,$17,$FA,$27,$C0,$FE
	.dc.b	$4B,$CB,$93,$F6,$E3,$F2,$47,$E5,$BC,$B9,$19,$3F,$01,$F0,$DF,$D3
	.dc.b	$48,$FF,$4D,$7A,$BF,$7E,$BA,$BF,$76,$BA,$BF,$83,$AE,$AE,$4E,$C7
	.dc.b	$F5,$58,$E6,$D3,$FA,$D3,$87,$FB,$E3,$5F,$EF,$8D,$7F,$C0,$4D,$7F
	.dc.b	$BB,$FC,$37,$F6,$52,$3F,$D9,$5E,$FB,$7F,$6B,$9F,$3C,$FE,$8B,$CF
	.dc.b	$B3,$98,$F3,$BF,$E6,$CF,$3F,$FC,$09,$F7,$2F,$87,$FE,$FE,$DA,$FF
	.dc.b	$01,$B5,$FF,$03,$B5,$FF,$4C,$FD,$CB,$D5,$FE,$0F,$10,$FD,$B4,$65
	.dc.b	$FD,$67,$AB,$FC,$33,$D5,$91,$83,$F7,$96,$F8,$7F,$19,$23,$C6,$FF
	.dc.b	$85,$2F,$A4,$7F,$00,$3C,$07,$F8,$41,$CE,$7F,$E8,$AC,$3B,$CF,$F0
	.dc.b	$6C,$3F,$8E,$C7,$C2,$F8,$7F,$82,$49,$F8,$2D,$7E,$0B,$5F,$F5,$CE
	.dc.b	$28,$C4,$09,$19,$79,$90,$F9,$7F,$0F,$F4,$08,$FD,$06,$BA,$30,$43
	.dc.b	$04,$3C,$6F,$07,$CB,$EE,$FD,$C2,$3F,$71,$AF,$DC,$6B,$C0,$D1,$0D
	.dc.b	$14,$F8,$3F,$0F,$F8,$82,$3F,$88,$6B,$F8,$86,$BC,$0D,$10,$D1,$4F
	.dc.b	$9B,$F0,$FF,$94,$23,$F9,$46,$BF,$94,$6B,$C0,$D1,$0D,$14,$FB,$3F
	.dc.b	$0F,$FA,$02,$3F,$A0,$7A,$E8,$EA,$FB,$92,$C9,$7F,$AA,$3F,$CD,$DB
	.dc.b	$3E,$1F,$FD,$87,$4F,$BA,$74,$72,$C3,$8C,$FE,$83,$0E,$4F,$E8,$AF
	.dc.b	$F6,$A7,$ED,$08,$FE,$D1,$F2,$E3,$CE,$78,$CF,$E8,$67,$C3,$87,$82
	.dc.b	$3F,$27,$1C,$9F,$DD,$B5,$FC,$DF,$C4,$7F,$02,$23,$FC,$09,$FD,$84
	.dc.b	$74,$C7,$C1,$3F,$9F,$AE,$CF,$C5,$BF,$8F,$18,$CB,$E2,$67,$BB,$F9
	.dc.b	$70,$F8,$0F,$92,$4C,$3F,$84,$FC,$07,$F1,$6F,$88,$FE,$2A,$48,$FE
	.dc.b	$2A,$F9,$30,$E9,$1F,$B0,$CB,$C3,$8F,$9A,$3E,$93,$E0,$CF,$D5,$1F
	.dc.b	$B7,$CB,$C4,$FF,$18,$CB,$C4,$FF,$0B,$CB,$F9,$3F,$C4,$7F,$26,$24
	.dc.b	$7F,$26,$7C,$4F,$63,$C4,$79,$0F,$88,$5A,$F8,$6E,$C2,$F8,$4F,$C5
	.dc.b	$FC,$24,$9F,$E5,$71,$FC,$E3,$E2,$3F,$9B,$92,$3F,$9B,$BE,$2D,$FE
	.dc.b	$9F,$17,$E3,$A7,$B3,$63,$EA,$69,$F0,$A7,$C3,$7D,$DF,$E5,$58,$7E
	.dc.b	$72,$3F,$A3,$FC,$47,$F4,$62,$4F,$F4,$67,$C1,$B7,$CF,$C3,$C3,$BF
	.dc.b	$D2,$AF,$9D,$8F,$7D,$1F,$11,$FA,$8F,$88,$95,$FD,$22,$3F,$AA,$9F
	.dc.b	$D5,$08,$FF,$54,$7B,$BF,$38,$F6,$DF,$77,$E5,$DF,$79,$EB,$1F,$D1
	.dc.b	$6B,$F6,$F1,$FD,$89,$FE,$C2,$48,$FE,$C2,$F6,$7E,$C9,$D8,$FB,$53
	.dc.b	$EB,$DB,$F8,$09,$23,$FB,$0E,$7F,$DB,$B4,$FE,$DA,$48,$FE,$DA,$F5
	.dc.b	$DB,$EA,$6B,$F2,$4F,$CB,$71,$7E,$1F,$0E,$97,$FD,$75,$FE,$1F,$87
	.dc.b	$F7,$AD,$FF,$BC,$92,$7F,$BC,$BD,$75,$F8,$98,$F6,$9F,$A7,$87,$E4
	.dc.b	$F8,$FF,$8C,$F0,$C9,$FE,$F3,$1F,$E0,$6E,$3F,$F0,$29,$23,$FC,$0A
	.dc.b	$F5,$7F,$3D,$7D,$5F,$A3,$7D,$5F,$F0,$1D,$F5,$7F,$BB,$5F,$F8,$4B
	.dc.b	$8F,$FC,$22,$48,$FF,$08,$BD,$5F,$A0,$7E,$9F,$8B,$F5,$3C,$7F,$9E
	.dc.b	$9E,$13,$FE,$16,$6B,$FD,$DB,$BB,$FC,$3A,$47,$FC,$3A,$F5,$9F,$CB
	.dc.b	$C7,$DC,$C7,$D7,$3B,$8F,$F1,$0B,$5F,$E8,$67,$F2,$02,$94,$CF,$06
	.dc.b	$1F,$E9,$2D,$7E,$89,$FE,$1F,$7F,$8B,$77,$78,$A3,$06,$9F,$E5,$B1
	.dc.b	$C6,$3B,$1D,$0E,$8E,$47,$6E,$09,$1E,$38,$CB,$E5,$CB,$E5,$C7,$E4
	.dc.b	$63,$F5,$98,$FF,$06,$C7,$F8,$D4,$F0,$9F,$7B,$5F,$7D,$F6,$73,$DD
	.dc.b	$E4,$23,$E1,$7E,$84,$F7,$8F,$C3,$9F,$C8,$D4,$70,$8E,$67,$13,$D5
	.dc.b	$E0,$FC,$02,$38,$3D,$9C,$DF,$C7,$78,$3F,$28,$7E,$57,$3F,$7C,$FF
	.dc.b	$8D,$F0,$EE,$FF,$B3,$5A,$B2,$FF,$01,$3B,$C7,$FB,$FE,$1F,$CA,$69
	.dc.b	$E0,$FD,$22,$57,$CF,$7D,$EF,$CC,$8F,$DE,$BF,$9B,$9E,$55,$FC,$66
	.dc.b	$7C,$3A,$7D,$1A,$F8,$9E,$27,$DE,$23,$F6,$DE,$AF,$F1,$68,$E0,$49
	.dc.b	$FC,$8D,$75,$3F,$93,$65,$F0,$AB,$FC,$2B,$3F,$8A,$23,$F8,$77,$AE
	.dc.b	$7F,$27,$1F,$E7,$47,$A2,$7F,$0B,$7D,$23,$E2,$79,$1F,$94,$24,$FE
	.dc.b	$45,$E7,$CB,$BD,$7F,$61,$3F,$81,$BF,$33,$8C,$FF,$43,$E4,$7E,$78
	.dc.b	$91,$F9,$B7,$9B,$BC,$FE,$6C,$FE,$A2,$B9,$CF,$CF,$BF,$CD,$F9,$DF
	.dc.b	$A8,$24,$7E,$95,$E6,$EC,$95,$CD,$F7,$9D,$CF,$C5,$79,$BF,$B2,$23
	.dc.b	$FA,$F7,$9C,$7D,$38,$FE,$02,$F3,$7E,$01,$CC,$FF,$23,$FC,$17,$EF
	.dc.b	$08,$FE,$E5,$E7,$8F,$B8,$91,$F9,$53,$CA,$7A,$1F,$E2,$3E,$EF,$F0
	.dc.b	$62,$47,$F0,$37,$9B,$E4,$8E,$6F,$F9,$63,$2F,$93,$7E,$98,$FF,$2D
	.dc.b	$FC,$47,$F1,$02,$3F,$C3,$5E,$6F,$EF,$23,$85,$7E,$70,$FC,$9C,$7B
	.dc.b	$D7,$C4,$7F,$1A,$24,$7F,$17,$79,$F1,$7E,$5C,$FC,$C4,$F3,$9F,$80
	.dc.b	$7B,$CF,$DE,$7C,$47,$F2,$42,$3F,$C8,$5E,$7C,$5F,$9A,$DF,$E3,$E3
	.dc.b	$F4,$CF,$3F,$C4,$7F,$2C,$25,$FF,$29,$78,$CF,$A7,$4F,$8B,$8F,$C8
	.dc.b	$1F,$11,$FC,$D0,$97,$FC,$C5,$E2,$F9,$CF,$9C,$78,$A3,$97,$C4,$7F
	.dc.b	$3C,$24,$7F,$39,$78,$E3,$F3,$2F,$C3,$3F,$64,$FE,$39,$F1,$1F,$D1
	.dc.b	$08,$FF,$41,$78,$DF,$9E,$78,$CF,$D7,$8F,$EA,$FF,$11,$FD,$30,$91
	.dc.b	$FD,$25,$E2,$F7,$3D,$27,$E6,$8F,$64,$3E,$1F,$FA,$A9,$23,$FA,$93
	.dc.b	$C9,$F0,$1F,$6B,$0F,$A5,$3F,$92,$FB,$BF,$E6,$33,$FD,$B9,$AF,$F7
	.dc.b	$26,$BF,$EB,$66,$BE,$5E,$AF,$E8,$3D,$3F,$C3,$E3,$CB,$97,$C0,$E2
	.dc.b	$FE,$45,$6F,$88,$FE,$EC,$DA,$FE,$F4,$D7,$C5,$1E,$99,$FC,$9D,$73
	.dc.b	$E2,$FF,$36,$5B,$E2,$39,$93,$0E,$6F,$19,$FE,$55,$1F,$27,$4F,$D7
	.dc.b	$53,$E2,$3C,$64,$9F,$1B,$C7,$8F,$F1,$11,$F2,$76,$F9,$1F,$11,$EA
	.dc.b	$24,$7A,$9E,$0F,$78,$FD,$9B,$F9,$3B,$7C,$47,$C2,$24,$7C,$27,$8E
	.dc.b	$3F,$5A,$3C,$07,$E0,$E5,$F1,$1F,$28,$91,$F2,$9E,$37,$E0,$AE,$53
	.dc.b	$F8,$21,$4F,$88,$FA,$64,$8F,$A6,$F2,$7C,$27,$E4,$A7,$E4,$FC,$47
	.dc.b	$DA,$24,$7D,$A7,$83,$BB,$F1,$1F,$8F,$F8,$8F,$C0,$92,$3F,$02,$F0
	.dc.b	$8F,$E2,$11,$C4,$F8,$1F,$11,$F8,$82,$3F,$88,$78,$70,$FE,$17,$6F
	.dc.b	$F6,$26,$6F,$88,$FC,$79,$2F,$F1,$EF,$07,$EE,$9F,$16,$1F,$11,$F9
	.dc.b	$62,$47,$E5,$9E,$0F,$DD,$3E,$A7,$C4,$7E,$70,$8F,$E7,$1D,$EB,$F4
	.dc.b	$B9,$7F,$47,$F0,$BF,$A2,$25,$7E,$89,$DE,$BF,$22,$3F,$01,$F0,$DF
	.dc.b	$A8,$23,$FA,$87,$6F,$D6,$AF,$D7,$FC,$37,$EB,$89,$1F,$AE,$76,$FC
	.dc.b	$D3,$E0,$7C,$37,$ED,$08,$FE,$D1,$DD,$FD,$4A,$FD,$DF,$C3,$7E,$E8
	.dc.b	$95,$FB,$A7,$7C,$1F,$C0,$F3,$5F,$F0,$9F,$85,$FE,$02,$4B,$FE,$02
	.dc.b	$F0,$7F,$BB,$6E,$F8,$6F,$E0,$E4,$8F,$E0,$EE,$DF,$E7,$D9,$3E,$1B
	.dc.b	$FD,$04,$D8,$FF,$39,$46,$0E,$18,$FF,$6F,$97,$A9,$C0,$95,$FE,$79
	.dc.b	$77,$97,$F7,$89,$7C,$37,$62,$57,$F9,$FD,$DB,$FC,$E6,$1F,$0D,$E4
	.dc.b	$24,$7F,$A1,$5D,$BF,$B5,$87,$C3,$7A,$89,$1F,$E8,$B7,$71,$FD,$DF
	.dc.b	$E1,$BD,$E4,$9F,$F4,$83,$F0,$BC,$1F,$F3,$74,$7F,$A8,$0F,$F7,$7E
	.dc.b	$5F,$EA,$33,$FC,$C4,$4C,$BF,$8B,$BF,$E2,$93,$10,$16,$F1,$3E,$79
	.dc.b	$23,$FD,$40,$F9,$AF,$C3,$A7,$E0,$A7,$95,$3E,$1B,$EB,$92,$3F,$D5
	.dc.b	$0F,$1E,$BF,$F2,$CC,$7F,$B1,$25,$7E,$C8,$C3,$5F,$F4,$09,$89,$FF
	.dc.b	$4B,$91,$FF,$65,$CF,$28,$C4,$6C,$CC,$F3,$7F,$D9,$67,$87,$E1,$5F
	.dc.b	$8A,$7F,$AF,$5B,$1D,$1A,$EE,$34,$47,$4D,$74,$60,$86,$0F,$A1,$F7
	.dc.b	$A2,$00

;--------------------------------------------------------------------------------
;	FACOS.X FPm,FPn
;--------------------------------------------------------------------------------
	.cpu	68030
regs		reg	d0-d7/a0-a5
cregs		reg	fpcr/fpsr/fpiar
fregs		reg	fp0-fp7
	.offsym	0,_a6
_size:
_regs:	.ds.b	.sizeof.(regs)
_fregs:	.ds.b	.sizeof.(fregs)
_cregs:	.ds.b	.sizeof.(cregs)
_a6:	.ds.l	1
_pc:	.ds.l	1
	.text
	.even
facos_test::
	link.w	a6,#_size
	movem.l	regs,(_regs,a6)
	fmovem.l	cregs,(_cregs,a6)
	fmovem.x	fregs,(_fregs,a6)
;
	move.l	#43,-(sp)
	peamsg	'FACOS.X FPM,FPN'
	jbsr	mnemonic_start
	addq.l	#8,sp
	beq	99f
	putmsg	'test: FACOS.X FPm,FPn',13,10
;------------------------------------------------
;	d1	fpcr=(XRN..DRP)<<4
;	d3	0=failed,1=successful
;	d5	expected status
;	d7	actual status
;	a2	source handle,...
;	a4	expected result handle,expected status,...
;	fp2	source
;	fp5	expected result
;	fp7	actual result
;------------------------------------------------
	lea.l	push_decompressed,a0
;decompress data
	move.l	a0,-(sp)
	pea.l	facos_data_compressed
	jbsr	decompress
	addq.l	#8,sp
;relocate decompressed handle
	move.l	#indirect_decompressed,d0
@@:
	add.l	d0,(a0)+		;source handle
	tst.l	(a0)
	bpl	@b
	addq.l	#4,a0			;-1
@@:
	add.l	d0,(a0)+		;expected result handle
	addq.l	#4,a0			;expected status
	tst.l	(a0)
	bpl	@b
;	addq.l	#4,a0			;-1
;
	lea.l	push_decompressed,a2	;source handle,...
	lea.l	(4*1687+4,a2),a4	;expected result handle,expected status,...
22:
	move.l	#0,d1		;fpcr=(XRN..DRP)<<4
11:
;FPn,FPn
	fmove.l	#0,fpcr
	fmove.x	([a2]),fp2		;source
	fmove.x	fp2,fp7			;source
	fmove.l	d1,fpcr			;fpcr
	fmove.l	#0,fpsr
					;source
	facos.x	fp7,fp7		;EXECUTE
					;actual result
	fmove.l	fpsr,d7			;actual status
	fmove.l	#0,fpcr
;
	fmove.x	([a4]),fp5		;expected result
	move.l	(4,a4),d5		;expected status
;
	move.l	d1,-(sp)	;fpcr(rp<<6,-1=strict)
	move.l	d5,-(sp)		;expected status
	fmove.x	fp5,-(sp)		;expected result
	move.l	d7,-(sp)		;actual status
	fmove.x	fp7,-(sp)		;actual result
	jbsr	test_extended
	lea.l	36(sp),sp
	move.l	d0,d3			;0=failed,1=successful
;
	move.l	d3,-(sp)		;0=failed,1=successful
	jbsr	statistics_update
	addq.l	#4,sp
	beq	@f			;not output
;output
	putmsg	'FACOS.X FPn='
	puthex24	([a2]),(4,[a2]),(8,[a2])	;source
	putmsg	',FPn @'
	move.l	d1,-(sp)
	jbsr	printfpcrrprm
	addq.l	#4,sp
	putcrlf
	move.l	d3,-(sp)		;0=failed,1=successful
	move.l	d5,-(sp)		;expected status
	fmove.x	fp5,-(sp)		;expected result
	move.l	d7,-(sp)		;actual status
	fmove.x	fp7,-(sp)		;actual result
	jbsr	output_extended
	lea.l	36(sp),sp
@@:
;
	addq.l	#8,a4			;expected result handle,expected status,...
;
	add.w	#1<<4,d1		;rprm++
	cmp.w	#176,d1		;fpcr=(XRN..DRP)<<4
	bls	11b
;
	addq.l	#4,a2			;source handle,...
	tst.l	(a2)			;source handle,...
	bpl	22b
;
	lea.l	push_decompressed,a2	;source handle,...
	lea.l	(4*1687+4,a2),a4	;expected result handle,expected status,...
22:
	move.l	#0,d1		;fpcr=(XRN..DRP)<<4
11:
;FPm,FPn
	fmove.l	#0,fpcr
	fmove.x	([a2]),fp2		;source
	fmove.s	#$7FFFFFFF,fp7		;fill=NaN
	fmove.l	d1,fpcr			;fpcr
	fmove.l	#0,fpsr
					;source
	facos.x	fp2,fp7		;EXECUTE
					;actual result
	fmove.l	fpsr,d7			;actual status
	fmove.l	#0,fpcr
;
	fmove.x	([a4]),fp5		;expected result
	move.l	(4,a4),d5		;expected status
;
	move.l	d1,-(sp)	;fpcr(rp<<6,-1=strict)
	move.l	d5,-(sp)		;expected status
	fmove.x	fp5,-(sp)		;expected result
	move.l	d7,-(sp)		;actual status
	fmove.x	fp7,-(sp)		;actual result
	jbsr	test_extended
	lea.l	36(sp),sp
	move.l	d0,d3			;0=failed,1=successful
;
	move.l	d3,-(sp)		;0=failed,1=successful
	jbsr	statistics_update
	addq.l	#4,sp
	beq	@f			;not output
;output
	putmsg	'FACOS.X FPm='
	puthex24	([a2]),(4,[a2]),(8,[a2])	;source
	putmsg	',FPn @'
	move.l	d1,-(sp)
	jbsr	printfpcrrprm
	addq.l	#4,sp
	putcrlf
	move.l	d3,-(sp)		;0=failed,1=successful
	move.l	d5,-(sp)		;expected status
	fmove.x	fp5,-(sp)		;expected result
	move.l	d7,-(sp)		;actual status
	fmove.x	fp7,-(sp)		;actual result
	jbsr	output_extended
	lea.l	36(sp),sp
@@:
;
	addq.l	#8,a4			;expected result handle,expected status,...
;
	add.w	#1<<4,d1		;rprm++
	cmp.w	#176,d1		;fpcr=(XRN..DRP)<<4
	bls	11b
;
	addq.l	#4,a2			;source handle,...
	tst.l	(a2)			;source handle,...
	bpl	22b
;
	jbsr	mnemonic_end
99:
	fmovem.x	(_fregs,a6),fregs
	fmovem.l	(_cregs,a6),cregs
	movem.l	(_regs,a6),regs
	unlk	a6
	rts
	.cpu	68000

	.align	4
facos_data_compressed::
	.dc.b	$00,$02,$93,$04,$10,$86,$36,$BD,$58,$F1,$B3,$E9,$6B,$DE,$DF,$C5
	.dc.b	$70,$F9,$6E,$3F,$45,$CB,$EB,$B9,$FD,$D7,$4F,$C1,$BA,$FE,$25,$DB
	.dc.b	$F1,$EE,$FF,$95,$78,$3F,$36,$F0,$FE,$85,$E2,$FD,$3B,$C7,$FA,$D7
	.dc.b	$93,$F6,$6F,$2F,$EE,$5E,$6F,$DF,$BD,$1D,$0E,$FB,$E5,$63,$D5,$CF
	.dc.b	$C0,$6B,$E2,$F7,$F2,$DC,$3E,$87,$8F,$D6,$72,$FB,$7E,$7F,$81,$74
	.dc.b	$FC,$37,$AF,$E3,$1D,$BF,$23,$EF,$F9,$77,$83,$F3,$9F,$0F,$E8,$9E
	.dc.b	$2F,$D3,$FC,$7F,$AD,$79,$3F,$65,$F2,$FE,$E1,$E6,$FD,$EF,$E7,$79
	.dc.b	$CF,$42,$FB,$58,$F8,$4C,$FC,$76,$BE,$63,$7F,$45,$C3,$EB,$38,$FD
	.dc.b	$B7,$2F,$C0,$39,$FE,$15,$D3,$F1,$4E,$BF,$8F,$76,$FC,$A3,$BF,$E6
	.dc.b	$5E,$0F,$CF,$3C,$3F,$A3,$78,$BF,$50,$F1,$FE,$B5,$E4,$FD,$93,$CB
	.dc.b	$FB,$77,$9B,$F7,$8F,$C2,$7C,$27,$E1,$07,$C1,$E3,$E0,$F3,$F0,$7A
	.dc.b	$F8,$3D,$FC,$1F,$0F,$83,$E3,$F0,$7C,$BE,$0F,$9F,$C1,$F4,$F8,$3E
	.dc.b	$BF,$07,$DB,$E0,$FB,$FC,$1F,$83,$E0,$FC,$3F,$07,$E2,$F8,$3F,$1F
	.dc.b	$C1,$F9,$3E,$0F,$CB,$F0,$7E,$6F,$83,$F3,$FC,$1F,$E6,$FC,$CF,$C2
	.dc.b	$7C,$27,$E7,$97,$C1,$67,$E0,$B5,$F0,$5B,$F8,$2E,$1F,$05,$C7,$E0
	.dc.b	$B9,$7C,$17,$3F,$82,$E9,$F0,$5D,$7E,$0B,$B7,$C1,$77,$F8,$2F,$07
	.dc.b	$C1,$78,$7E,$0B,$C5,$F0,$5E,$3F,$82,$F2,$7C,$17,$97,$E0,$BC,$DF
	.dc.b	$05,$E7,$F8,$2F,$D9,$7C,$2F,$E1,$7A,$3F,$6A,$BE,$0B,$3F,$05,$AF
	.dc.b	$82,$DF,$C1,$70,$F8,$2E,$3F,$05,$CB,$E0,$B9,$FC,$17,$4F,$82,$EB
	.dc.b	$F0,$5D,$BE,$0B,$BF,$C1,$78,$3E,$0B,$C3,$F0,$5E,$2F,$82,$F1,$FC
	.dc.b	$17,$93,$E0,$BC,$BF,$05,$E6,$F8,$2F,$3F,$C1,$7F,$09,$F8,$6F,$E1
	.dc.b	$27,$E1,$BA,$3F,$86,$AF,$81,$D7,$C0,$EF,$E0,$78,$7C,$0F,$1F,$81
	.dc.b	$E5,$F0,$3C,$FE,$07,$A7,$C0,$F5,$F8,$1E,$DF,$03,$DF,$E0,$7C,$1F
	.dc.b	$03,$E1,$F8,$1F,$17,$C0,$F8,$FE,$07,$C9,$F0,$3E,$5F,$81,$F3,$7C
	.dc.b	$0F,$9F,$E0,$7D,$1F,$03,$FC,$87,$E1,$FF,$0F,$D0,$7F,$0E,$FF,$26
	.dc.b	$5F,$01,$BF,$80,$E1,$F0,$1C,$7E,$03,$97,$C0,$73,$F8,$0E,$9F,$01
	.dc.b	$D7,$E0,$3B,$7C,$07,$7F,$80,$F0,$7C,$07,$87,$E0,$3C,$5F,$01,$E3
	.dc.b	$F8,$0F,$27,$C0,$79,$7E,$03,$CD,$F0,$1E,$7F,$80,$F4,$7C,$07,$F3
	.dc.b	$9F,$86,$FC,$47,$3F,$F2,$3F,$80,$C7,$C0,$67,$E0,$35,$F0,$1B,$F8
	.dc.b	$0E,$1F,$01,$C7,$E0,$39,$7C,$07,$3F,$80,$E9,$F0,$1D,$7E,$03,$B7
	.dc.b	$C0,$77,$F8,$0F,$07,$C0,$78,$7E,$03,$C5,$F0,$1E,$3F,$80,$F2,$7C
	.dc.b	$07,$97,$E0,$3C,$DF,$01,$FD,$3F,$E2,$3F,$A7,$9F,$87,$E7,$3F,$87
	.dc.b	$7F,$AA,$AF,$7F,$87,$BF,$C7,$DF,$E5,$EF,$F3,$F7,$FA,$7B,$FD,$7D
	.dc.b	$FE,$DE,$FF,$7F,$7F,$C1,$EF,$F8,$7D,$FF,$17,$BF,$E3,$F7,$FC,$9E
	.dc.b	$FF,$97,$DF,$F3,$7B,$FE,$7F,$7F,$D1,$EF,$FA,$7D,$FF,$ED,$1F,$13
	.dc.b	$F8,$83,$F0,$FC,$E7,$F0,$EF,$F6,$E5,$EF,$70,$F7,$B8,$FB,$DC,$BD
	.dc.b	$EE,$7E,$F7,$4F,$7B,$AF,$BD,$DB,$DE,$EF,$EF,$78,$3D,$EF,$0F,$BD
	.dc.b	$E2,$F7,$BC,$7E,$F7,$93,$DE,$F2,$FB,$DE,$6F,$7B,$CF,$EF,$7A,$3D
	.dc.b	$EF,$F0,$07,$C5,$7E,$20,$FE,$1D,$CD,$FE,$04,$5E,$EE,$BD,$DD,$FB
	.dc.b	$BC,$3D,$DE,$3E,$EF,$2F,$77,$9F,$BB,$D3,$DD,$EB,$EE,$F6,$F7,$7B
	.dc.b	$FB,$BE,$0F,$77,$C3,$EE,$F8,$BD,$DF,$1F,$BB,$E4,$F7,$7C,$BE,$EF
	.dc.b	$9B,$DD,$F3,$FB,$BF,$E1,$7F,$87,$FF,$0B,$9F,$89,$3F,$11,$CF,$FD
	.dc.b	$5B,$DD,$D7,$BB,$BF,$77,$87,$BB,$C7,$DD,$E5,$EE,$F3,$F7,$7A,$7B
	.dc.b	$BD,$7D,$DE,$DE,$EF,$7F,$77,$C1,$EE,$F8,$7D,$DF,$17,$BB,$E3,$F7
	.dc.b	$7C,$9E,$EF,$97,$DD,$F3,$7B,$BE,$7F,$77,$FC,$5D,$F1,$7F,$C3,$8F
	.dc.b	$C3,$9F,$86,$E6,$3F,$86,$7F,$C6,$AB,$DC,$E3,$EE,$72,$F7,$39,$FB
	.dc.b	$9D,$3D,$CE,$BE,$E7,$6F,$73,$BF,$B9,$E0,$F7,$3C,$3E,$E7,$8B,$DC
	.dc.b	$F1,$FB,$9E,$4F,$73,$CB,$EE,$79,$BD,$CF,$3F,$B9,$E8,$F7,$3F,$C8
	.dc.b	$DF,$19,$FC,$38,$FC,$3F,$29,$FC,$3B,$FE,$4C,$5E,$DF,$0F,$6F,$8F
	.dc.b	$B7,$CB,$DB,$E7,$ED,$F4,$F6,$FA,$FB,$7D,$BD,$BE,$FE,$DF,$83,$DB
	.dc.b	$F0,$FB,$7E,$2F,$6F,$C7,$ED,$F9,$3D,$BF,$2F,$B7,$E6,$F6,$FC,$FE
	.dc.b	$DF,$A3,$DB,$FF,$2F,$7C,$6F,$F9,$78,$FE,$20,$7E,$20,$FE,$1D,$CD
	.dc.b	$FE,$64,$5E,$D7,$0F,$6B,$8F,$B5,$CB,$DA,$E7,$ED,$74,$F6,$BA,$FB
	.dc.b	$5D,$BD,$AE,$FE,$D7,$83,$DA,$F0,$FB,$5E,$2F,$6B,$C7,$ED,$79,$3D
	.dc.b	$AF,$2F,$B5,$E6,$F6,$BC,$FE,$D7,$A3,$DA,$FF,$3B,$7C,$17,$F5,$63
	.dc.b	$F8,$79,$F8,$93,$F1,$1C,$DF,$E0,$7F,$6B,$7E,$D7,$0F,$6B,$8F,$B5
	.dc.b	$CB,$DA,$E7,$ED,$74,$F6,$BA,$FB,$5D,$BD,$AE,$FE,$D7,$83,$DA,$F0
	.dc.b	$FB,$5E,$2F,$6B,$C7,$ED,$79,$3D,$AF,$2F,$B5,$E6,$F6,$BF,$D1,$9F
	.dc.b	$1D,$FD,$50,$FC,$39,$F8,$6E,$43,$F8,$67,$FD,$26,$BD,$9E,$3E,$CF
	.dc.b	$2F,$67,$9F,$B3,$D3,$D9,$EB,$EC,$F6,$F6,$7B,$FB,$3E,$0F,$67,$C3
	.dc.b	$EC,$F8,$BD,$9F,$1F,$B3,$E4,$F6,$7C,$BE,$CF,$9B,$D9,$F3,$FB,$3E
	.dc.b	$8F,$67,$FD,$49,$F1,$FF,$EA,$43,$FA,$B1,$FC,$38,$FC,$3F,$29,$FC
	.dc.b	$3B,$FE,$AC,$5E,$C7,$2F,$63,$9F,$B1,$D3,$D8,$EB,$EC,$76,$F6,$3B
	.dc.b	$FB,$1E,$0F,$63,$C3,$EC,$78,$BD,$8F,$1F,$B1,$E4,$F6,$3C,$BE,$C7
	.dc.b	$9B,$D8,$F3,$FB,$1E,$8F,$63,$D3,$EC,$7F,$AF,$7E,$43,$FC,$3A,$7F
	.dc.b	$56,$3F,$88,$1F,$88,$3F,$87,$72,$FF,$B1,$17,$AF,$C7,$D7,$E5,$EB
	.dc.b	$F3,$F5,$FA,$7A,$FD,$7D,$7E,$DE,$BF,$7F,$5F,$C1,$EB,$F8,$7D,$7F
	.dc.b	$17,$AF,$E3,$F5,$FC,$9E,$BF,$97,$D7,$F3,$7A,$FE,$7F,$5F,$FD,$A9
	.dc.b	$F1,$9F,$E1,$C3,$F8,$79,$F8,$93,$F1,$1C,$9F,$E7,$CF,$5F,$7E,$BF
	.dc.b	$0F,$5F,$8F,$AF,$CB,$D7,$E7,$EB,$F4,$F5,$FA,$FA,$FD,$BD,$7E,$FE
	.dc.b	$BF,$83,$D7,$F0,$FA,$FE,$2F,$5F,$C7,$EB,$F9,$3D,$7F,$2F,$AF,$E6
	.dc.b	$F5,$FF,$DD,$5F,$23,$FE,$EA,$3F,$C3,$87,$F5,$43,$F0,$E7,$E1,$B9
	.dc.b	$0F,$E1,$9F,$F7,$9A,$F5,$B9,$FA,$DD,$3D,$6E,$BE,$B7,$6F,$5B,$BF
	.dc.b	$AD,$E0,$F5,$BC,$3E,$B7,$8B,$D6,$F1,$FA,$DE,$4F,$5B,$CB,$EB,$79
	.dc.b	$BD,$6F,$3F,$AD,$E8,$F5,$BD,$3E,$B3,$8E,$7C,$97,$F9,$90,$FF,$0E
	.dc.b	$9F,$D5,$CF,$E1,$E7,$E2,$13,$F8,$87,$8F,$AB,$CD,$CF,$AB,$65,$EA
	.dc.b	$DF,$57,$1E,$AE,$7D,$5D,$7A,$BB,$F5,$78,$7A,$BC,$7D,$5E,$5E,$AF
	.dc.b	$3F,$57,$A7,$AB,$D7,$D5,$ED,$EA,$F8,$7E,$4F,$FC,$CA,$7F,$58,$3F
	.dc.b	$89,$1F,$8A,$3F,$88,$71,$7C,$15,$EA,$71,$F5,$39,$7A,$9C,$FD,$4E
	.dc.b	$9E,$A7,$5F,$53,$B7,$A9,$DF,$D4,$F0,$7A,$9E,$1F,$53,$C5,$EA,$78
	.dc.b	$FD,$4F,$27,$A9,$E5,$F5,$3C,$DE,$A7,$9F,$D4,$F9,$1F,$1B,$F2,$79
	.dc.b	$3F,$D5,$5F,$0B,$88,$CF,$C2,$EB,$E0,$77,$EE,$F0,$F6,$38,$FB,$1C
	.dc.b	$BD,$8E,$7E,$C7,$4F,$63,$AF,$B1,$DB,$D8,$EF,$EC,$78,$3D,$8F,$0F
	.dc.b	$B1,$E2,$F6,$3C,$7E,$C7,$93,$D8,$F2,$FB,$1E,$6F,$63,$CF,$EC,$7D
	.dc.b	$FF,$94,$FF,$55,$9F,$E6,$43,$FC,$3C,$7F,$56,$3F,$10,$7E,$1F,$A0
	.dc.b	$BD,$9E,$5E,$CF,$3F,$63,$A7,$B1,$D7,$D8,$ED,$EC,$77,$F6,$3C,$1E
	.dc.b	$C7,$87,$D8,$F1,$7B,$1E,$3F,$63,$C9,$EC,$79,$7D,$8F,$37,$B1,$E7
	.dc.b	$F6,$3F,$2D,$F2,$BF,$EA,$A3,$FC,$3A,$7F,$57,$3F,$87,$9F,$86,$E3
	.dc.b	$3F,$86,$7F,$50,$BD,$7E,$7E,$BF,$4F,$5F,$AF,$AF,$DB,$D7,$EF,$EB
	.dc.b	$F8,$3D,$7F,$0F,$AF,$E2,$F5,$FC,$7E,$BF,$93,$D7,$F2,$FA,$FE,$6F
	.dc.b	$5F,$CF,$EB,$FA,$3D,$7F,$D7,$7C,$B7,$E1,$CF,$F5,$61,$FE,$64,$3F
	.dc.b	$AB,$9F,$C4,$4F,$E2,$1C,$67,$F8,$83,$FC,$05,$7A,$5D,$3D,$2E,$BE
	.dc.b	$97,$6F,$4B,$BF,$A5,$E0,$F4,$BC,$3E,$97,$8B,$D2,$F1,$FA,$5E,$4F
	.dc.b	$4B,$CB,$E9,$79,$BD,$2F,$3F,$A5,$E8,$F4,$BD,$3E,$97,$F0,$7F,$89
	.dc.b	$FF,$79,$9F,$EA,$C3,$FC,$CC,$7F,$87,$8F,$E2,$07,$E2,$4F,$C4,$71
	.dc.b	$FF,$19,$5E,$8F,$3F,$47,$A7,$A3,$D7,$D1,$ED,$E8,$F7,$F4,$7C,$1E
	.dc.b	$8F,$87,$D1,$F1,$7A,$3E,$3F,$47,$C9,$E8,$F9,$7D,$1F,$37,$A3,$E7
	.dc.b	$F4,$7F,$8F,$7C,$BF,$FB,$C8,$FF,$32,$9F,$E1,$F3,$FA,$B9,$F8,$93
	.dc.b	$F1,$1C,$47,$F1,$0F,$F3,$15,$E8,$74,$F4,$3A,$FA,$1D,$BD,$0E,$FE
	.dc.b	$87,$83,$D0,$F0,$FA,$1E,$2F,$43,$C7,$E8,$79,$3D,$0F,$2F,$A1,$E6
	.dc.b	$F4,$3C,$FE,$87,$A3,$D0,$FE,$6D,$F3,$1F,$C3,$CF,$F7,$A1,$FE,$AC
	.dc.b	$3F,$C3,$E7,$F5,$93,$F1,$27,$E2,$38,$8F,$E2,$1F,$E9,$4B,$CF,$EB
	.dc.b	$E7,$F6,$F3,$FB,$F9,$FE,$0F,$3F,$C3,$E7,$F8,$BC,$FF,$1F,$9F,$E4
	.dc.b	$F3,$FC,$BE,$7F,$9B,$CF,$F3,$F9,$FE,$8F,$3F,$FA,$77,$E2,$7F,$C2
	.dc.b	$CF,$C1,$F0,$9F,$C1,$BF,$D7,$5F,$EA,$2D,$F3,$38,$F9,$9C,$BC,$CE
	.dc.b	$7E,$67,$4F,$33,$AF,$99,$DB,$CC,$EF,$E6,$78,$3C,$CF,$0F,$99,$E2
	.dc.b	$F3,$3C,$7E,$67,$F6,$1F,$C5,$1F,$2F,$7E,$2B,$E7,$FA,$7F,$D9,$7E
	.dc.b	$6D,$F2,$9B,$F5,$71,$F0,$F9,$F9,$5D,$7C,$9E,$FE,$3B,$87,$C5,$F1
	.dc.b	$F7,$F9,$7B,$7C,$FD,$BE,$9D,$9D,$7B,$3B,$76,$77,$EC,$F0,$76,$78
	.dc.b	$7B,$3C,$5D,$9E,$3E,$CF,$27,$67,$97,$B3,$CD,$D9,$E7,$EC,$FE,$F9
	.dc.b	$C9,$EC,$ED,$FF,$04,$3F,$EC,$4E,$DF,$F1,$AB,$FE,$7F,$2F,$C5,$F0
	.dc.b	$F8,$8E,$3F,$01,$CB,$DD,$E7,$EE,$F4,$F7,$7A,$FB,$BD,$BD,$DE,$FE
	.dc.b	$EF,$83,$DD,$F0,$FB,$BE,$2F,$77,$C7,$EE,$F9,$3D,$DF,$2F,$BB,$F4
	.dc.b	$78,$3C,$FE,$EF,$A3,$DD,$FF,$0A,$7C,$E7,$F6,$D1,$F1,$38,$F8,$9C
	.dc.b	$FC,$4E,$BE,$27,$7F,$13,$C3,$E2,$78,$FC,$37,$2F,$84,$E7,$F0,$9D
	.dc.b	$3E,$13,$AF,$C2,$76,$F8,$4E,$FF,$09,$E0,$F8,$4F,$0F,$C2,$78,$BE
	.dc.b	$13,$C7,$F0,$9E,$4F,$84,$F2,$FC,$27,$9B,$E1,$3C,$FF,$03,$FE,$2B
	.dc.b	$F9,$DF,$C3,$F9,$9F,$E3,$75,$EF,$E7,$CC,$D7,$BF,$BF,$7F,$87,$BF
	.dc.b	$C7,$DF,$E5,$EF,$F3,$F7,$FF,$2D,$E9,$75,$F7,$FB,$7B,$FD,$FD,$FF
	.dc.b	$07,$BF,$E1,$F7,$FC,$5E,$FF,$8F,$DF,$F2,$7B,$FE,$5F,$7F,$CD,$EF
	.dc.b	$F9,$FD,$FF,$47,$BF,$FE,$43,$F9,$EF,$C4,$74,$BE,$03,$F9,$0A,$87
	.dc.b	$C2,$6F,$E0,$B8,$7C,$9F,$1F,$5B,$97,$2F,$3F,$83,$E9,$F0,$7D,$7E
	.dc.b	$0F,$B7,$C1,$F7,$F8,$3F,$07,$C1,$F8,$7E,$0F,$C5,$F0,$7E,$3F,$83
	.dc.b	$F2,$7C,$1F,$97,$E0,$FC,$DF,$07,$E7,$F8,$3F,$DD,$6B,$FB,$BD,$7F
	.dc.b	$79,$AF,$EF,$75,$FD,$F6,$BE,$DF,$83,$F7,$7C,$1F,$F9,$87,$E0,$3F
	.dc.b	$C7,$47,$E3,$BA,$3F,$CD,$CB,$E0,$74,$EF,$E0,$78,$7C,$0F,$1F,$81
	.dc.b	$E5,$F0,$3C,$FE,$07,$A7,$C0,$FF,$0D,$E8,$ED,$F0,$3D,$FE,$07,$C1
	.dc.b	$F0,$3F,$C4,$7A,$FF,$89,$75,$FF,$13,$EB,$FE,$29,$D7,$FC,$57,$AF
	.dc.b	$F8,$B7,$5F,$F1,$7E,$6F,$E3,$1C,$DF,$E7,$9F,$C5,$7F,$CA,$47,$A7
	.dc.b	$9C,$FA,$5F,$F4,$43,$FE,$E4,$3F,$D3,$C7,$F8,$38,$FE,$8C,$7F,$45
	.dc.b	$78,$F0,$F5,$E1,$ED,$C1,$DF,$83,$E4,$7C,$17,$C9,$F8,$2F,$95,$F0
	.dc.b	$5F,$2F,$E0,$BE,$67,$C1,$7C,$DF,$82,$F9,$DF,$05,$F3,$FE,$0B,$E8
	.dc.b	$7C,$17,$FA,$47,$E8,$3F,$4F,$C0,$7F,$4E,$FF,$A7,$D7,$C0,$EF,$E0
	.dc.b	$78,$7C,$0F,$1F,$81,$E5,$F0,$3C,$FE,$07,$A7,$C0,$F5,$F8,$1E,$DF
	.dc.b	$03,$DF,$E0,$7C,$1F,$03,$E1,$F8,$1F,$17,$C0,$F8,$FE,$07,$C9,$F0
	.dc.b	$3F,$37,$C9,$F9,$DD,$9F,$3F,$B3,$E8,$76,$7D,$1E,$CF,$A5,$D9,$F4
	.dc.b	$FB,$3E,$A7,$67,$D5,$EC,$FA,$DD,$9F,$EA,$FF,$4F,$DE,$7B,$BB,$4F
	.dc.b	$B9,$FF,$5F,$2E,$BE,$1D,$7C,$7A,$F9,$75,$F3,$EB,$E9,$D7,$D7,$AF
	.dc.b	$E6,$7C,$0F,$CD,$F8,$1F,$9D,$F0,$3E,$1E,$EF,$A1,$F0,$3F,$EC,$1F
	.dc.b	$A1,$FD,$F9,$FB,$CE,$0F,$F6,$72,$F7,$F5,$EF,$EF,$DF,$E1,$EF,$F1
	.dc.b	$F7,$F9,$7B,$FC,$FD,$FE,$9C,$FD,$7D,$FF,$81,$E4,$7C,$1F,$23,$E1
	.dc.b	$79,$1F,$0F,$C8,$F8,$9E,$47,$8F,$DF,$F2,$7B,$3E,$5F,$67,$CD,$EC
	.dc.b	$F9,$FD,$9F,$47,$B3,$E9,$F6,$7D,$5E,$CF,$AF,$D9,$F6,$7B,$3F,$ED
	.dc.b	$DF,$A2,$FF,$5B,$1F,$A5,$3F,$49,$D8,$5E,$FE,$FD,$FE,$1E,$FF,$1F
	.dc.b	$7F,$97,$BF,$F6,$7D,$8E,$9E,$F7,$5F,$7B,$B7,$BD,$DF,$DE,$F0,$7B
	.dc.b	$9E,$1F,$73,$C5,$EE,$78,$FD,$CF,$27,$B9,$E5,$F7,$3C,$DE,$E7,$9F
	.dc.b	$DC,$F4,$7B,$9F,$87,$F5,$3D,$5E,$E7,$FB,$DF,$CF,$FD,$39,$F8,$93
	.dc.b	$F1,$1D,$27,$F1,$08,$C7,$B7,$C7,$DB,$E5,$ED,$F3,$F6,$FA,$7B,$7D
	.dc.b	$7D,$BE,$DE,$CF,$7F,$3F,$C1,$E7,$F8,$7C,$FF,$17,$9F,$E3,$F3,$FC
	.dc.b	$9D,$FE,$5E,$2F,$37,$B7,$E7,$F6,$FD,$1E,$DF,$8F,$8D,$F4,$7F,$D3
	.dc.b	$8F,$C8,$38,$F6,$F5,$1E,$B8,$F5,$3F,$EB,$D3,$FC,$EA,$7F,$54,$3F
	.dc.b	$CE,$6E,$BD,$7E,$3E,$BF,$2F,$5F,$F6,$19,$74,$F5,$FA,$FA,$FD,$BD
	.dc.b	$7E,$FE,$BF,$83,$D7,$F0,$FA,$FE,$2F,$5F,$C7,$EB,$F9,$3D,$7F,$2F
	.dc.b	$AF,$EB,$FA,$4F,$67,$09,$FE,$64,$7E,$38,$FC,$6B,$F4,$D7,$AD,$C7
	.dc.b	$D6,$E5,$EB,$7F,$05,$E8,$E9,$EB,$75,$F5,$BB,$7A,$DD,$FD,$6F,$07
	.dc.b	$AD,$E1,$F5,$BC,$5E,$B7,$8F,$D6,$F2,$7A,$DE,$5F,$5B,$CD,$EB,$79
	.dc.b	$FD,$6F,$47,$AD,$E9,$F5,$BD,$5E,$B7,$D5,$FA,$5F,$E6,$67,$F1,$53
	.dc.b	$F1,$A7,$E3,$39,$CB,$D6,$E1,$EB,$71,$F5,$BF,$8F,$79,$7C,$FD,$6E
	.dc.b	$9E,$B7,$E0,$7F,$16,$FF,$06,$9F,$80,$3F,$87,$65,$F9,$77,$F0,$09
	.dc.b	$FF,$7E,$1F,$EB,$43,$FC,$26,$7E,$01,$D7,$99,$D7,$CC,$ED,$E6,$77
	.dc.b	$F2,$FC,$1E,$5F,$87,$CB,$F1,$79,$7E,$3F,$2F,$C9,$E5,$F9,$7C,$BF
	.dc.b	$85,$EB,$FC,$3F,$5F,$E2,$7A,$FF,$17,$D7,$F8,$DE,$BF,$C7,$F5,$FE
	.dc.b	$47,$AF,$F2,$7D,$7F,$95,$EB,$FC,$BF,$57,$E6,$79,$DF,$37,$AF,$F4
	.dc.b	$5F,$01,$FE,$2E,$3F,$42,$7C,$2E,$9F,$E8,$1E,$CE,$BD,$FD,$FB,$FC
	.dc.b	$3D,$FE,$3E,$FF,$2E,$EE,$7D,$DD,$3B,$BA,$F7,$76,$EE,$EF,$DD,$E0
	.dc.b	$EE,$F0,$F7,$78,$BB,$BC,$7D,$DE,$4F,$6F,$CB,$ED,$F9,$BD,$BF,$DC
	.dc.b	$7D,$3F,$E7,$8F,$F0,$71,$FC,$A8,$FC,$EF,$29,$FC,$EB,$FC,$45,$7B
	.dc.b	$5C,$BD,$AE,$7E,$D7,$4F,$6B,$AF,$B5,$DB,$DA,$EF,$ED,$78,$3D,$AF
	.dc.b	$0F,$B5,$E2,$F6,$BC,$7E,$D7,$93,$DA,$F2,$FB,$5E,$6F,$6B,$CF,$ED
	.dc.b	$7A,$3D,$AF,$4F,$B5,$FC,$47,$EA,$3F,$C1,$E7,$F7,$13,$F9,$49,$F9
	.dc.b	$D3,$F3,$9C,$BF,$CA,$57,$B3,$C7,$D9,$E5,$EC,$F3,$F6,$7A,$7B,$1D
	.dc.b	$7D,$8E,$DE,$C7,$7F,$63,$C1,$EC,$78,$7D,$8F,$17,$B1,$E3,$F6,$3C
	.dc.b	$9E,$C7,$97,$D8,$F3,$7B,$1E,$7F,$63,$F9,$47,$D4,$FF,$9D,$0F,$E4
	.dc.b	$E7,$E1,$CF,$D3,$8F,$5F,$7E,$BF,$0F,$5F,$8F,$AF,$CB,$D7,$E7,$EB
	.dc.b	$F4,$F5,$FA,$FA,$FD,$BD,$7E,$FE,$BF,$83,$D7,$F0,$FA,$FE,$2F,$5F
	.dc.b	$C7,$EB,$F9,$3D,$7F,$2F,$AF,$E6,$F5,$FF,$9F,$7B,$BF,$C9,$0F,$F2
	.dc.b	$A1,$FD,$C0,$FC,$39,$F8,$6F,$1B,$F9,$1F,$AF,$C3,$D7,$E3,$EB,$F2
	.dc.b	$F5,$F9,$FA,$FD,$3D,$7E,$BE,$BF,$6F,$5F,$BF,$AF,$E0,$F5,$FC,$3E
	.dc.b	$BF,$8B,$D7,$F1,$FA,$FE,$4F,$5F,$CB,$EB,$F9,$BD,$7F,$3F,$AF,$FD
	.dc.b	$57,$EA,$BF,$CA,$67,$F8,$D4,$FE,$DA,$7F,$21,$3F,$0B,$C8,$7F,$0A
	.dc.b	$FF,$6F,$5E,$B7,$3F,$5B,$A7,$AD,$D7,$D6,$ED,$EB,$77,$F5,$BC,$1E
	.dc.b	$B7,$87,$D6,$F1,$7A,$DE,$3F,$5B,$C9,$EB,$79,$7D,$6F,$37,$AD,$E7
	.dc.b	$F5,$BD,$1E,$B7,$F6,$DF,$AB,$FF,$5C,$9F,$DB,$0F,$E1,$87,$F2,$C3
	.dc.b	$F0,$DC,$7F,$E0,$55,$EA,$F1,$F5,$79,$7A,$BC,$FD,$5E,$9E,$AF,$5F
	.dc.b	$57,$B7,$AB,$DF,$D5,$F0,$7A,$BE,$1F,$57,$C5,$EA,$F8,$FD,$5F,$27
	.dc.b	$AB,$E5,$F5,$7C,$DE,$AF,$9F,$D5,$FF,$02,$7D,$67,$F6,$A3,$FD,$34
	.dc.b	$7F,$8D,$8F,$E1,$A7,$E1,$CF,$E1,$7C,$9F,$E1,$D5,$EA,$72,$F5,$39
	.dc.b	$FA,$9D,$3D,$4E,$BE,$A7,$6F,$53,$BF,$A9,$E0,$F5,$3C,$3E,$A7,$8B
	.dc.b	$D4,$F1,$FA,$9E,$4F,$53,$CB,$EA,$79,$BD,$4F,$3F,$A9,$E8,$F5,$3F
	.dc.b	$C3,$9E,$BF,$FA,$68,$FF,$3E,$1F,$E3,$43,$FB,$41,$F8,$83,$F0,$FC
	.dc.b	$9F,$B7,$F5,$38,$FA,$9C,$BD,$4E,$7E,$A7,$4F,$53,$AF,$A9,$DB,$D4
	.dc.b	$EF,$EA,$78,$3D,$4F,$0F,$A9,$E2,$F5,$3C,$7E,$A7,$93,$D4,$F2,$FA
	.dc.b	$9E,$6F,$53,$FC,$65,$F5,$BF,$EF,$E3,$FC,$62,$7F,$51,$3F,$B8,$9F
	.dc.b	$86,$E2,$3F,$86,$7F,$C9,$8B,$D3,$E7,$E9,$F4,$F4,$FA,$FA,$7D,$BD
	.dc.b	$3E,$FE,$9F,$83,$D3,$F0,$FA,$7E,$2F,$4F,$C7,$E9,$F9,$3D,$3F,$2F
	.dc.b	$A7,$E6,$F4,$FC,$FE,$9F,$A3,$D3,$FF,$25,$7D,$77,$F8,$C0,$FF,$70
	.dc.b	$1F,$E7,$C3,$FA,$99,$FC,$34,$FC,$3F,$19,$FC,$3B,$FE,$65,$7F,$DC
	.dc.b	$03,$D1,$EB,$E8,$F6,$F4,$7B,$FA,$3E,$0F,$47,$C3,$E8,$F8,$BD,$1F
	.dc.b	$1F,$A3,$E4,$F4,$7C,$BE,$8F,$9B,$D1,$F3,$FA,$3E,$8F,$47,$D3,$E8
	.dc.b	$FF,$98,$FE,$BF,$FD,$C2,$7F,$B0,$CF,$F3,$D9,$FE,$2F,$3F,$87,$9F
	.dc.b	$88,$3F,$86,$F1,$7F,$9F,$57,$A1,$CF,$D0,$E9,$E8,$75,$F4,$3B,$7A
	.dc.b	$1D,$FD,$0F,$07,$A1,$E1,$F4,$3C,$5E,$87,$8F,$D0,$F2,$7A,$1E,$5F
	.dc.b	$43,$CD,$E8,$79,$FD,$0F,$F3,$D7,$C1,$3E,$5E,$FC,$7E,$3E,$23,$3F
	.dc.b	$07,$AF,$5F,$7E,$DF,$0F,$5B,$8F,$AD,$CB,$D6,$E7,$EB,$74,$F5,$BA
	.dc.b	$FA,$DD,$BD,$6E,$FE,$B7,$83,$D6,$F0,$FA,$DE,$2F,$5B,$C7,$EB,$79
	.dc.b	$3D,$6F,$2F,$AD,$E6,$F5,$BF,$D2,$1F,$61,$FE,$76,$1F,$2B,$8F,$91
	.dc.b	$CF,$C6,$6B,$E0,$F7,$F0,$7C,$3E,$03,$8F,$C0,$72,$F7,$79,$FB,$BD
	.dc.b	$3D,$DE,$BE,$EF,$6F,$77,$BF,$BB,$E0,$F7,$7C,$3E,$EF,$8B,$DD,$F1
	.dc.b	$FB,$BE,$4F,$77,$CB,$EE,$F9,$BD,$DF,$3F,$BB,$FE,$A7,$FB,$17,$CC
	.dc.b	$5F,$96,$C7,$C9,$67,$E3,$35,$E9,$6F,$E2,$B8,$7C,$37,$1F,$82,$E5
	.dc.b	$F0,$5C,$FE,$0B,$A7,$C1,$75,$F8,$2E,$DF,$05,$DF,$E0,$BC,$1F,$05
	.dc.b	$E1,$F8,$2F,$17,$C1,$78,$FE,$0B,$C9,$F0,$5E,$5F,$82,$F3,$7C,$17
	.dc.b	$FA,$F7,$EC,$BF,$09,$F0,$BF,$ED,$17,$FD,$7A,$7F,$97,$4F,$F3,$E9
	.dc.b	$FC,$34,$FE,$18,$9F,$E1,$8F,$1F,$77,$A7,$BB,$D7,$DD,$ED,$EE,$F7
	.dc.b	$F7,$7C,$1E,$EF,$87,$DD,$F1,$7B,$BE,$3F,$77,$C9,$EE,$F9,$7D,$DF
	.dc.b	$37,$BB,$E7,$F7,$7D,$1E,$EF,$FB,$57,$D8,$FF,$5D,$9F,$86,$E1,$FF
	.dc.b	$75,$2F,$6F,$5E,$DE,$FD,$BE,$13,$C7,$DB,$E5,$ED,$F3,$F6,$FA,$7B
	.dc.b	$7D,$7D,$BE,$DE,$DF,$7F,$6F,$C1,$ED,$F8,$7D,$BF,$17,$B7,$E3,$F6
	.dc.b	$FC,$9E,$DF,$97,$DB,$F3,$7B,$7E,$7F,$6F,$D1,$ED,$FF,$BB,$7E,$CF
	.dc.b	$F0,$FC,$FF,$F0,$05,$7B,$59,$F6,$B5,$ED,$6F,$DA,$E1,$ED,$71,$F6
	.dc.b	$B9,$7B,$5C,$FD,$AE,$9E,$D7,$5F,$6B,$B7,$B5,$DF,$DA,$F0,$7B,$5E
	.dc.b	$1F,$6B,$C5,$ED,$78,$FD,$AF,$27,$B5,$E5,$F6,$BC,$DE,$D7,$9F,$DA
	.dc.b	$73,$CF,$B4,$FC,$4A,$5F,$0B,$A8,$C7,$C2,$D9,$FF,$63,$9F,$D5,$CF
	.dc.b	$C5,$9F,$8A,$8E,$5E,$E6,$FD,$CE,$1E,$E7,$1F,$73,$97,$B9,$CF,$DC
	.dc.b	$E9,$EE,$75,$F7,$3B,$7B,$9D,$FD,$CF,$07,$B9,$E1,$F7,$3C,$5E,$E7
	.dc.b	$8F,$DC,$F3,$7D,$AF,$9F,$84,$FE,$2D,$BE,$EE,$BD,$DD,$FB,$BC,$3D
	.dc.b	$DE,$3E,$EF,$2F,$77,$9F,$BB,$D3,$DD,$EB,$EE,$F6,$F7,$7B,$FB,$BE
	.dc.b	$0F,$77,$C3,$EE,$F8,$BD,$DF,$1F,$BB,$E4,$F7,$7C,$BE,$EF,$9B,$DD
	.dc.b	$F3,$FB,$BE,$8F,$77,$E7,$7C,$27,$E2,$B9,$FF,$C8,$BE,$EE,$3D,$DC
	.dc.b	$FB,$BA,$F7,$77,$EE,$F0,$F7,$78,$FB,$BC,$BD,$DE,$7E,$EF,$4F,$77
	.dc.b	$AF,$BB,$DB,$DD,$EF,$EE,$F8,$3D,$DF,$0F,$BB,$E2,$F7,$7C,$7E,$EF
	.dc.b	$93,$DD,$F2,$FB,$BE,$6F,$77,$F0,$9F,$6D,$F8,$83,$F0,$FC,$FF,$8C
	.dc.b	$7F,$DD,$83,$E1,$F7,$F0,$FC,$3A,$B8,$FC,$3F,$2F,$7B,$9F,$BD,$D3
	.dc.b	$DE,$EB,$EF,$76,$F7,$BB,$FB,$DE,$0F,$7B,$C3,$EF,$78,$BD,$EF,$1F
	.dc.b	$BD,$E4,$F7,$BC,$BE,$F7,$9B,$DE,$F3,$FB,$DF,$9B,$FB,$7F,$C3,$1F
	.dc.b	$85,$3F,$87,$73,$7E,$91,$FF,$76,$9F,$EA,$73,$FC,$36,$7F,$86,$92
	.dc.b	$F6,$7A,$7B,$3D,$7D,$9E,$DE,$CF,$7F,$67,$C1,$EC,$F8,$7D,$9F,$17
	.dc.b	$B3,$E3,$F6,$7C,$9E,$CF,$97,$D9,$F3,$7B,$3E,$7F,$67,$D1,$EC,$FA
	.dc.b	$7D,$9F,$DA,$7D,$C7,$E1,$CF,$E1,$FC,$7F,$BC,$5E,$CE,$BD,$9D,$FB
	.dc.b	$3C,$3D,$9E,$3E,$CF,$2F,$67,$9F,$B3,$D3,$D9,$EB,$EC,$F6,$F6,$7B
	.dc.b	$FB,$3E,$0F,$67,$C3,$EC,$F8,$BD,$9F,$1F,$B3,$E4,$F6,$7C,$BE,$CF
	.dc.b	$9B,$D9,$F3,$FB,$3F,$C2,$F8,$7F,$87,$9F,$88,$E6,$FF,$7F,$FC,$16
	.dc.b	$7D,$9D,$7B,$3B,$F6,$78,$7B,$3C,$7D,$9E,$5E,$CF,$3F,$67,$A7,$B3
	.dc.b	$D7,$D9,$ED,$EC,$F7,$F6,$7C,$1E,$CF,$87,$D9,$F1,$7B,$3E,$3F,$67
	.dc.b	$C9,$EC,$F9,$7D,$9F,$37,$B3,$FC,$87,$EE,$7F,$86,$1F,$C2,$CF,$C3
	.dc.b	$F3,$17,$C0,$6F,$E0,$38,$7C,$07,$1F,$80,$E5,$EE,$F3,$F7,$7A,$7B
	.dc.b	$BD,$7D,$DE,$DE,$EF,$7F,$77,$C1,$EE,$F8,$7D,$DF,$17,$BB,$E3,$F7
	.dc.b	$7C,$9E,$EF,$97,$DD,$F3,$7B,$BE,$7F,$77,$D1,$EE,$FF,$39,$FB,$AF
	.dc.b	$E1,$87,$E1,$B9,$CF,$E1,$9F,$E8,$6B,$DC,$E1,$EE,$71,$F3,$79,$7B
	.dc.b	$9C,$FD,$CE,$9E,$E7,$5F,$73,$B7,$B9,$DF,$DC,$F0,$7B,$9E,$1F,$73
	.dc.b	$C5,$EE,$78,$FD,$CF,$27,$B9,$E5,$F7,$3C,$DE,$E7,$9F,$DC,$F4,$7B
	.dc.b	$9F,$D4,$7E,$EF,$FA,$A9,$F8,$71,$EE,$E7,$DD,$D7,$BB,$BF,$77,$87
	.dc.b	$BB,$C7,$DD,$E5,$EE,$F3,$F7,$7A,$7B,$BD,$7D,$DE,$DE,$EF,$7F,$77
	.dc.b	$C1,$EE,$F8,$7D,$DF,$17,$BB,$E3,$F7,$7C,$9E,$EF,$97,$DD,$F3,$7B
	.dc.b	$BF,$D9,$FD,$9F,$EA,$47,$F5,$13,$F0,$E7,$E1,$BC,$AF,$EE,$2B,$DB
	.dc.b	$E1,$ED,$F1,$8E,$5E,$DF,$3F,$6F,$A7,$B7,$D7,$DB,$ED,$ED,$F7,$F6
	.dc.b	$FC,$1E,$DF,$87,$DB,$F1,$7B,$7E,$3F,$6F,$C9,$ED,$F9,$7D,$BF,$37
	.dc.b	$B7,$E7,$F6,$FD,$1E,$DF,$F7,$FF,$BC,$FE,$A4,$7E,$1C,$FC,$37,$29
	.dc.b	$FC,$33,$FE,$0A,$5E,$D7,$1F,$6B,$97,$B5,$CF,$DA,$E9,$ED,$75,$F6
	.dc.b	$BB,$7B,$5D,$FD,$AF,$07,$B5,$E1,$F6,$BC,$5E,$D7,$8F,$DA,$F2,$7B
	.dc.b	$5E,$5F,$6B,$CD,$ED,$79,$FD,$AF,$47,$B5,$FE,$17,$FB,$DF,$F0,$D9
	.dc.b	$FC,$34,$FC,$3F,$2F,$F8,$81,$FE,$A8,$DF,$63,$87,$B1,$C7,$D8,$E5
	.dc.b	$EC,$73,$F6,$3A,$7B,$0F,$E0,$18,$5C,$7A,$43,$EF,$BF,$C5,$BF,$04
	.dc.b	$E7,$E3,$18,$6D,$7F,$36,$6C,$EE,$39,$CE,$DA,$6C,$B1,$B5,$AA,$FC
	.dc.b	$51,$B3,$2E,$27,$84,$EE,$BB,$56,$9B,$19,$8E,$4D,$9E,$35,$CE,$FD
	.dc.b	$24,$AF,$2B,$5E,$87,$28,$E0,$E8,$86,$A7,$AD,$F8,$A7,$73,$9A,$E9
	.dc.b	$7E,$36,$CE,$A3,$13,$D1,$AF,$13,$11,$B2,$57,$72,$47,$9D,$B7,$E4
	.dc.b	$8C,$11,$C1,$2F,$C7,$3C,$1B,$3B,$6C,$EE,$3C,$73,$B6,$9B,$2C,$6D
	.dc.b	$F5,$12,$FD,$A4,$BF,$5C,$71,$34,$E2,$79,$4E,$EB,$C1,$5A,$6C,$66
	.dc.b	$39,$36,$78,$D7,$3B,$F5,$92,$BC,$ED,$7A,$1C,$A3,$83,$A2,$1A,$9E
	.dc.b	$B7,$E2,$9D,$CE,$6B,$A5,$F8,$DB,$3A,$8C,$4F,$46,$BC,$4C,$46,$C9
	.dc.b	$5D,$C9,$1E,$76,$DF,$92,$30,$47,$04,$BF,$1C,$F0,$6C,$ED,$B3,$B8
	.dc.b	$F1,$CE,$DA,$6C,$B1,$B7,$D4,$4B,$F6,$92,$FD,$71,$C4,$D3,$89,$E5
	.dc.b	$3B,$AF,$05,$69,$B1,$98,$E4,$D9,$E3,$5C,$EF,$D6,$4A,$F3,$B5,$E8
	.dc.b	$72,$8E,$0E,$88,$6A,$7A,$DF,$8A,$77,$39,$AE,$97,$E3,$6C,$EA,$31
	.dc.b	$3D,$1A,$F1,$31,$1B,$25,$77,$24,$79,$DB,$7E,$48,$C1,$1C,$12,$FC
	.dc.b	$73,$C1,$B3,$B6,$CE,$E3,$C7,$3B,$69,$B2,$C6,$DF,$51,$2F,$DA,$4B
	.dc.b	$F5,$C7,$13,$4E,$27,$94,$EE,$BC,$15,$A6,$C6,$63,$93,$67,$8D,$73
	.dc.b	$BF,$59,$2B,$CE,$D7,$A1,$CA,$38,$3A,$21,$A9,$EB,$7E,$29,$DC,$E6
	.dc.b	$BA,$5F,$8D,$B3,$A8,$C4,$F4,$6B,$C4,$C4,$6C,$95,$DC,$91,$E7,$6D
	.dc.b	$F9,$23,$04,$70,$4B,$F1,$CF,$06,$CE,$DB,$3B,$8F,$1C,$ED,$A6,$CB
	.dc.b	$1B,$7D,$44,$BF,$69,$2F,$D7,$1C,$4D,$38,$9E,$53,$BA,$F0,$56,$9B
	.dc.b	$19,$8E,$4D,$9E,$35,$CE,$FD,$64,$AF,$3B,$5E,$87,$28,$E0,$E8,$86
	.dc.b	$A7,$AD,$F8,$A7,$73,$9A,$E9,$7E,$36,$CE,$A3,$13,$D1,$AF,$13,$11
	.dc.b	$B2,$57,$72,$47,$9D,$B7,$E4,$8C,$11,$C1,$2F,$C7,$3C,$1B,$3B,$6C
	.dc.b	$EE,$3C,$73,$B6,$9B,$2C,$6D,$F5,$12,$FD,$A4,$BF,$5C,$71,$34,$E2
	.dc.b	$79,$4E,$EB,$C1,$5A,$6C,$66,$39,$36,$78,$D7,$3B,$F5,$92,$BC,$ED
	.dc.b	$7A,$1C,$A3,$83,$A2,$1A,$9E,$B7,$E2,$9D,$CE,$6B,$A5,$F8,$DB,$3A
	.dc.b	$8C,$4F,$46,$BC,$4C,$46,$C9,$5D,$C9,$1E,$76,$DF,$92,$30,$47,$04
	.dc.b	$BF,$1C,$F0,$6C,$ED,$B3,$B8,$F1,$CE,$DA,$6C,$B1,$B7,$D4,$4B,$F6
	.dc.b	$92,$FD,$71,$C4,$D3,$89,$E5,$3B,$AF,$05,$69,$B1,$98,$E4,$D9,$E3
	.dc.b	$5C,$EF,$D6,$4A,$F3,$B5,$E8,$72,$8E,$0E,$88,$6A,$7A,$DF,$8A,$77
	.dc.b	$39,$AE,$97,$E3,$6C,$EA,$31,$3D,$1A,$F1,$31,$1B,$25,$77,$24,$79
	.dc.b	$DB,$7E,$48,$C1,$1C,$12,$FC,$73,$C1,$B3,$B6,$CE,$E3,$C7,$3B,$69
	.dc.b	$B2,$C6,$DF,$51,$2F,$DA,$4B,$F5,$C7,$13,$4E,$27,$94,$EE,$BC,$15
	.dc.b	$A6,$C6,$63,$93,$67,$8D,$73,$BF,$59,$2B,$CE,$D7,$A1,$CA,$38,$3A
	.dc.b	$21,$A9,$EB,$7E,$29,$DC,$E6,$BA,$5F,$8D,$B3,$A8,$C4,$F4,$6B,$C4
	.dc.b	$C4,$6C,$95,$DC,$91,$E7,$6D,$F9,$23,$04,$70,$4B,$F1,$CF,$06,$CE
	.dc.b	$DB,$3B,$8F,$1C,$ED,$A6,$CB,$1B,$7D,$44,$BF,$69,$2F,$D7,$1C,$4D
	.dc.b	$38,$9E,$53,$BA,$F0,$56,$9B,$19,$8E,$4D,$9E,$35,$CE,$FD,$64,$AF
	.dc.b	$3B,$5E,$87,$28,$E0,$E8,$86,$A7,$AD,$F8,$A7,$73,$9A,$E9,$7E,$36
	.dc.b	$CE,$A3,$13,$D1,$AF,$13,$11,$B2,$57,$72,$47,$9D,$B7,$E4,$8C,$11
	.dc.b	$C1,$2F,$C7,$3C,$1B,$3B,$6C,$EE,$3C,$73,$B6,$9B,$2C,$6D,$F5,$12
	.dc.b	$FD,$A4,$BF,$5C,$71,$34,$E2,$79,$4E,$EB,$C1,$5A,$6C,$66,$39,$36
	.dc.b	$78,$D7,$3B,$F5,$92,$BC,$ED,$7A,$1C,$A3,$83,$A2,$1A,$9E,$B7,$E2
	.dc.b	$9D,$CE,$6B,$A5,$F8,$DB,$3A,$8C,$4F,$46,$BC,$4C,$46,$C9,$5D,$C9
	.dc.b	$1E,$76,$DF,$92,$30,$47,$04,$BF,$1C,$F0,$6C,$ED,$B3,$B8,$F1,$CE
	.dc.b	$DA,$6C,$B1,$B7,$D4,$4B,$F6,$92,$FD,$71,$C4,$D3,$89,$E5,$3B,$AF
	.dc.b	$05,$69,$B1,$98,$E4,$D9,$E3,$5C,$EF,$D6,$4A,$F3,$B5,$E8,$72,$8E
	.dc.b	$0E,$88,$6A,$7A,$DF,$8A,$77,$39,$AE,$97,$E3,$6C,$EA,$31,$3D,$1A
	.dc.b	$F1,$31,$1B,$25,$77,$24,$79,$DB,$7E,$48,$C1,$1C,$12,$FC,$73,$C1
	.dc.b	$B3,$B6,$CE,$E3,$C7,$3B,$69,$B2,$C6,$DF,$51,$2F,$DA,$4B,$F5,$C7
	.dc.b	$13,$4E,$27,$94,$EE,$BC,$15,$A6,$C6,$63,$93,$67,$8D,$73,$BF,$59
	.dc.b	$2B,$CE,$D7,$A1,$CA,$38,$3A,$21,$A9,$EB,$7E,$29,$DC,$E6,$BA,$5F
	.dc.b	$8D,$B3,$A8,$C4,$F4,$6B,$C4,$C4,$6C,$95,$DC,$91,$E7,$6D,$F9,$23
	.dc.b	$04,$70,$4B,$F1,$CF,$06,$CE,$DB,$3B,$8F,$1C,$ED,$A6,$CB,$1B,$7D
	.dc.b	$44,$BF,$69,$2F,$D7,$1C,$4D,$38,$9E,$53,$BA,$F0,$56,$9B,$19,$8E
	.dc.b	$4D,$9E,$35,$CE,$FD,$64,$AF,$3B,$5E,$87,$28,$E0,$E8,$86,$A7,$AD
	.dc.b	$F8,$A7,$73,$9A,$E9,$7E,$36,$CE,$A3,$13,$D1,$AF,$13,$11,$B2,$57
	.dc.b	$72,$47,$9D,$B7,$E4,$8C,$11,$C1,$2F,$C7,$3C,$1B,$3B,$6C,$EE,$3C
	.dc.b	$73,$B6,$9B,$2C,$6D,$F5,$12,$FD,$A4,$BF,$5C,$71,$34,$E2,$79,$4E
	.dc.b	$EB,$C1,$5A,$6C,$66,$39,$36,$78,$D7,$3B,$F5,$92,$BC,$ED,$7A,$1C
	.dc.b	$A3,$83,$A2,$1A,$9E,$B7,$E2,$9D,$CE,$6B,$A5,$F8,$DB,$3A,$8C,$4F
	.dc.b	$46,$BC,$4C,$46,$C9,$5D,$C9,$1E,$76,$DF,$92,$30,$47,$04,$BF,$1C
	.dc.b	$F0,$6C,$ED,$B3,$B8,$F1,$CE,$DA,$6C,$B1,$B7,$D4,$4B,$F6,$92,$FD
	.dc.b	$71,$C4,$D3,$89,$E5,$3B,$AF,$05,$69,$B1,$98,$E4,$D9,$E3,$5C,$EF
	.dc.b	$D6,$4A,$F3,$B5,$E8,$72,$8E,$0E,$88,$6A,$7A,$DF,$8A,$77,$39,$AE
	.dc.b	$97,$E3,$6C,$EA,$31,$3D,$1A,$F1,$31,$1B,$25,$77,$24,$79,$DB,$7E
	.dc.b	$48,$C1,$1C,$12,$FC,$73,$C1,$B3,$B6,$CE,$E3,$C7,$3B,$69,$B2,$C6
	.dc.b	$DF,$51,$2F,$DA,$4B,$F5,$C7,$13,$4E,$27,$94,$EE,$BC,$15,$A6,$C6
	.dc.b	$63,$93,$67,$8D,$73,$BF,$59,$28,$29,$2B,$CD,$AF,$77,$BA,$D2,$9D
	.dc.b	$3D,$3E,$6D,$EA,$70,$FB,$E7,$C7,$F2,$97,$AD,$98,$C3,$6F,$95,$D5
	.dc.b	$4F,$5B,$1F,$8C,$79,$25,$8F,$67,$DB,$19,$8C,$47,$29,$E3,$F2,$52
	.dc.b	$3E,$1C,$EA,$93,$DB,$E3,$3B,$FC,$3F,$83,$DA,$98,$70,$D7,$E2,$E0
	.dc.b	$AF,$46,$7E,$7A,$D1,$E4,$3D,$26,$B3,$F5,$FB,$52,$36,$2D,$1F,$03
	.dc.b	$DA,$F8,$3E,$D7,$C2,$F6,$A5,$F0,$B4,$F5,$8C,$1E,$03,$AE,$3E,$57
	.dc.b	$E4,$E7,$EA,$F6,$BE,$69,$23,$A7,$B5,$F4,$3D,$AF,$A3,$ED,$49,$F0
	.dc.b	$47,$77,$99,$EF,$3C,$27,$83,$6F,$B3,$ED,$48,$F4,$3F,$73,$DA,$FB
	.dc.b	$BE,$D4,$9F,$63,$F8,$0F,$6B,$B7,$3F,$D7,$9F,$61,$E0,$3E,$F1,$F0
	.dc.b	$FD,$AF,$C3,$FB,$52,$30,$7C,$7F,$6B,$F1,$64,$7E,$57,$B5,$DB,$87
	.dc.b	$EE,$C7,$6C,$3F,$17,$C5,$F2,$5F,$C4,$7B,$5F,$94,$24,$FD,$AF,$C6
	.dc.b	$7F,$23,$3F,$92,$5F,$1B,$F9,$D7,$F5,$53,$BF,$07,$13,$DF,$1F,$0C
	.dc.b	$FC,$A3,$DD,$8F,$86,$3F,$6B,$6F,$88,$E9,$EE,$4B,$F1,$47,$7F,$72
	.dc.b	$47,$95,$F1,$7B,$9D,$E3,$F8,$2D,$BD,$CF,$31,$98,$43,$DC,$F5,$7B
	.dc.b	$92,$7E,$04,$7B,$7C,$A9,$1F,$0D,$F8,$1E,$E7,$48,$F8,$DE,$E7,$C3
	.dc.b	$30,$FD,$BC,$1E,$E7,$C7,$F7,$24,$FC,$D8,$F9,$5E,$E7,$CB,$F7,$24
	.dc.b	$FD,$18,$F9,$DE,$E4,$8F,$A8,$FD,$1F,$73,$E9,$7B,$9F,$4F,$DC,$95
	.dc.b	$F6,$63,$F8,$97,$03,$E5,$6F,$C8,$CA,$FD,$BC,$E5,$BF,$13,$C3,$E2
	.dc.b	$64,$60,$DF,$C0,$74,$23,$CB,$E0,$3B,$FC,$07,$83,$E0,$24,$F6,$8F
	.dc.b	$1F,$C0,$79,$3D,$D9,$3E,$28,$F3,$FC,$4C,$8F,$2B,$EA,$F8,$9F,$5F
	.dc.b	$C4,$FB,$3E,$26,$57,$AA,$3D,$FF,$13,$F0,$3E,$26,$4F,$BA,$3E,$1F
	.dc.b	$C4,$FC,$4F,$89,$93,$F0,$A3,$E3,$FC,$27,$C8,$F8,$59,$3F,$1A,$3E
	.dc.b	$5F,$B1,$FC,$F7,$F1,$AF,$E6,$DE,$67,$F4,$E6,$D7,$F5,$6E,$8F,$CF
	.dc.b	$3F,$D6,$23,$F9,$AB,$FC,$99,$AF,$F0,$16,$FA,$1F,$D0,$17,$C2,$F7
	.dc.b	$F8,$49,$5C,$5F,$17,$C2,$F8,$FE,$16,$4F,$57,$CD,$F0,$BE,$7F,$83
	.dc.b	$F4,$7C,$2C,$AF,$14,$7A,$FE,$13,$D9,$F0,$92,$7C,$D1,$EF,$F8,$39
	.dc.b	$1E,$97,$E1,$7C,$27,$C3,$F8,$4F,$89,$F0,$92,$BD,$D1,$F1,$FE,$12
	.dc.b	$47,$C1,$7E,$57,$C2,$7C,$BF,$84,$F9,$9F,$09,$2B,$E3,$47,$CF,$F8
	.dc.b	$4F,$A1,$F0,$92,$7E,$54,$7D,$3F,$84,$91,$F3,$5F,$EE,$5F,$8D,$FF
	.dc.b	$1A,$EB,$FF,$01,$BF,$C6,$A7,$FA,$BB,$5F,$E6,$4D,$EA,$FE,$E2,$FF
	.dc.b	$1E,$8C,$FC,$0F,$42,$3C,$BE,$0B,$BF,$C1,$78,$3E,$0A,$4F,$68,$F1
	.dc.b	$FC,$17,$93,$E0,$A4,$F8,$A3,$CF,$E0,$C8,$F2,$BE,$AF,$82,$F5,$FC
	.dc.b	$17,$B3,$E0,$A5,$7A,$A3,$DF,$F0,$12,$3D,$AF,$C2,$F8,$1F,$87,$F0
	.dc.b	$1F,$13,$E0,$65,$7C,$28,$F8,$FF,$03,$F2,$3E,$06,$4F,$C6,$8F,$97
	.dc.b	$F0,$12,$3E,$4B,$F3,$BE,$07,$FC,$41,$F8,$E7,$F0,$B9,$FE,$B8,$BE
	.dc.b	$07,$26,$BE,$0B,$AC,$FD,$0C,$6F,$4F,$D0,$CB,$B4,$FD,$4C,$AC,$FD
	.dc.b	$4C,$B9,$4F,$F8,$43,$2E,$CD,$CB,$2F,$8B,$E0,$BC,$BF,$05,$23,$84
	.dc.b	$7A,$3E,$0B,$D3,$F0,$52,$7A,$47,$B3,$E0,$7D,$BF,$05,$27,$C1,$1F
	.dc.b	$03,$E0,$7E,$0F,$C1,$49,$F2,$47,$C4,$F8,$29,$1E,$77,$E3,$FC,$17
	.dc.b	$C8,$F8,$1F,$93,$F0,$52,$BD,$91,$CD,$F9,$78,$5E,$2F,$F2,$2F,$0B
	.dc.b	$2E,$B5,$F4,$32,$B5,$AC,$BB,$4F,$C7,$CB,$B3,$F5,$B2,$FB,$3F,$03
	.dc.b	$F6,$BE,$0A,$4F,$CD,$8F,$BB,$F0,$32,$3E,$83,$F8,$0F,$82,$FC,$0F
	.dc.b	$C0,$C9,$FA,$8F,$E1,$3E,$0B,$AC,$7D,$7C,$3A,$BF,$86,$C3,$A3,$F8
	.dc.b	$8C,$2E,$9E,$EC,$7A,$4F,$E3,$31,$B3,$F0,$B1,$E6,$FC,$4C,$6F,$99
	.dc.b	$FE,$82,$E0,$67,$D2,$31,$9D,$CB,$FD,$1B,$4C,$FA,$D7,$1C,$ED,$6B
	.dc.b	$3F,$CF,$FC,$17,$F9,$FF,$DA,$FE,$83,$26,$E3,$FD,$0A,$31,$F1,$1B
	.dc.b	$F8,$29,$1F,$E6,$8A,$FE,$83,$97,$FA,$61,$FF,$3A,$46,$7E,$07,$B1
	.dc.b	$DF,$E0,$BC,$31,$C4,$F1,$18,$3F,$C9,$93,$D1,$FF,$54,$BF,$D3,$1F
	.dc.b	$05,$FA,$70,$E9,$1D,$C8,$72,$AF,$77,$C1,$48,$F4,$3F,$07,$E0,$BE
	.dc.b	$17,$C0,$C9,$F6,$3F,$17,$E0,$BE,$37,$C1,$7C,$7C,$E5,$7C,$18,$E8
	.dc.b	$FA,$3E,$0B,$E6,$18,$E7,$FF,$51,$D7,$FA,$1A,$7A,$90,$F7,$DF,$67
	.dc.b	$E6,$47,$67,$E1,$DF,$D7,$F8,$29,$1F,$45,$FB,$5F,$03,$F6,$FE,$0B
	.dc.b	$EE,$7C,$14,$AF,$AD,$1F,$7F,$E0,$BF,$01,$F0,$32,$7E,$D4,$74,$7E
	.dc.b	$8E,$17,$4F,$4E,$3D,$67,$F0,$18,$F5,$7E,$BE,$3D,$5F,$C2,$63,$D5
	.dc.b	$FB,$78,$F3,$E5,$FC,$36,$56,$7E,$EE,$3D,$5F,$C0,$E3,$D7,$2F,$93
	.dc.b	$97,$59,$FC,$2E,$5D,$67,$E6,$E5,$F9,$BF,$82,$95,$F9,$57,$F3,$DF
	.dc.b	$03,$F9,$FF,$82,$93,$F9,$97,$F4,$5F,$05,$FE,$E1,$FC,$77,$FB,$84
	.dc.b	$FF,$BE,$1F,$ED,$9B,$FF,$BD,$1B,$19,$8E,$EF,$E6,$30,$B9,$7E,$03
	.dc.b	$1E,$F3,$F9,$CC,$6C,$FE,$13,$1F,$06,$1E,$1F,$87,$97,$FE,$D4,$CB
	.dc.b	$C6,$9F,$EF,$97,$FE,$EE,$6D,$79,$7E,$23,$D5,$F0,$9E,$BF,$84,$98
	.dc.b	$73,$7D,$DF,$07,$EF,$F8,$4F,$81,$F0,$72,$BC,$11,$F0,$F0,$91,$E3
	.dc.b	$7A,$BC,$C8,$7F,$B0,$F2,$61,$DA,$39,$9E,$BB,$EC,$EE,$FE,$87,$C2
	.dc.b	$7D,$1F,$84,$93,$F0,$63,$EA,$7C,$24,$8F,$88,$FD,$7F,$83,$FB,$1F
	.dc.b	$09,$F6,$7E,$12,$57,$C9,$8E,$6F,$97,$0B,$C3,$F2,$71,$E9,$3E,$9C
	.dc.b	$6C,$FC,$DC,$7A,$CF,$D3,$C7,$AB,$EF,$C7,$9F,$0F,$E1,$72,$B3,$F8
	.dc.b	$9C,$B9,$CF,$E2,$72,$B3,$F8,$DC,$7A,$BF,$27,$1E,$B9,$7D,$DC,$BF
	.dc.b	$29,$F0,$92,$BF,$08,$FE,$5F,$E0,$FF,$31,$F0,$9F,$99,$F8,$29,$5F
	.dc.b	$89,$8F,$CE,$7C,$1C,$8F,$C6,$3F,$9F,$F8,$3E,$91,$F6,$7E,$0F,$F4
	.dc.b	$46,$33,$FC,$75,$F4,$9F,$BD,$87,$81,$E6,$43,$F0,$58,$75,$42,$9B
	.dc.b	$E2,$1F,$8F,$7F,$8C,$67,$31,$C2,$3F,$C4,$AF,$19,$E1,$18,$9D,$CF
	.dc.b	$2F,$85,$E9,$18,$97,$C1,$E2,$F8,$FC,$25,$C3,$B3,$5F,$1C,$DA,$7C
	.dc.b	$1D,$24,$FB,$30,$ED,$F0,$9B,$F8,$3E,$1F,$09,$2B,$59,$FA,$27,$A7
	.dc.b	$C1,$F5,$F8,$49,$5A,$3E,$04,$78,$7E,$13,$C5,$F0,$92,$74,$7C,$88
	.dc.b	$F3,$7C,$1C,$8C,$EC,$8F,$57,$C2,$7A,$FE,$0F,$D9,$F0,$92,$F7,$1E
	.dc.b	$FF,$84,$91,$C9,$F8,$5F,$09,$D6,$3E,$91,$F5,$70,$C7,$17,$CF,$C7
	.dc.b	$B4,$F8,$B1,$EC,$F0,$3E,$FE,$3D,$9C,$64,$CB,$18,$7D,$9C,$BE,$97
	.dc.b	$C2,$7D,$3F,$84,$98,$7B,$A3,$EB,$7C,$27,$D9,$FC,$7F,$FC,$36,$7F
	.dc.b	$87,$32,$FF,$0D,$3F,$E0,$BA,$3F,$E0,$B7,$C8,$DC,$7F,$16,$EC,$FF
	.dc.b	$0D,$C6,$BE,$07,$B1,$DF,$E1,$3C,$1F,$05,$27,$EA,$D2,$3C,$9F,$05
	.dc.b	$E5,$F8,$2F,$37,$C0,$CB,$DC,$7A,$7E,$03,$D5,$F0,$32,$79,$C7,$B7
	.dc.b	$E0,$BD,$DF,$05,$27,$BC,$7C,$1F,$82,$91,$E2,$7E,$27,$C0,$FC,$5F
	.dc.b	$82,$93,$E6,$7E,$47,$C0,$FC,$9F,$82,$93,$EA,$7E,$67,$C1,$7C,$DF
	.dc.b	$82,$93,$EE,$7E,$87,$C1,$7D,$1F,$82,$93,$F0,$9F,$D6,$78,$5F,$D6
	.dc.b	$8F,$ED,$30,$7F,$AC,$BF,$C1,$9A,$E7,$E0,$B8,$FC,$14,$8F,$D0,$CA
	.dc.b	$3A,$FB,$FD,$BE,$0A,$56,$9F,$0F,$BF,$E2,$F8,$0F,$1F,$BF,$2B,$9C
	.dc.b	$79,$BE,$02,$47,$67,$D3,$F0,$1E,$AF,$80,$F5,$FB,$F2,$BC,$71,$EE
	.dc.b	$F8,$0F,$7F,$C0,$49,$F3,$C7,$C2,$F8,$09,$1E,$A7,$E2,$FB,$DF,$1B
	.dc.b	$DE,$F8,$FE,$F4,$AF,$7C,$7C,$AF,$80,$F9,$7F,$01,$27,$E1,$C7,$F1
	.dc.b	$9F,$C8,$3F,$95,$7B,$7F,$CB,$5F,$E4,$CD,$F8,$7D,$7C,$16,$FE,$07
	.dc.b	$87,$C1,$4C,$7F,$86,$E1,$FC,$A1,$BF,$05,$D7,$E0,$A4,$60,$DF,$C0
	.dc.b	$F8,$7E,$0A,$46,$0E,$7F,$03,$E5,$23,$DB,$E0,$BD,$1E,$0F,$A7,$E0
	.dc.b	$A4,$F8,$A3,$D9,$F0,$5E,$DF,$82,$93,$E6,$8F,$81,$F0,$32,$3D,$2F
	.dc.b	$C3,$F8,$2F,$89,$F0,$5F,$17,$E0,$A5,$7B,$A3,$E4,$7C,$0F,$C9,$F8
	.dc.b	$29,$3F,$0A,$3E,$67,$C1,$48,$F8,$AF,$F4,$7F,$C8,$7F,$D5,$27,$F0
	.dc.b	$D3,$F8,$57,$57,$F5,$D6,$CF,$F6,$27,$5F,$03,$C7,$E0,$64,$FF,$3C
	.dc.b	$9F,$E1,$FF,$05,$FD,$71,$FE,$15,$19,$F8,$1F,$01,$E1,$F8,$1F,$17
	.dc.b	$C0,$78,$FE,$06,$57,$38,$F3,$7C,$07,$9F,$E0,$64,$F7,$8F,$57,$C0
	.dc.b	$48,$F1,$3E,$DF,$81,$F7,$7C,$0F,$BF,$E0,$65,$79,$E3,$E1,$7C,$0F
	.dc.b	$C3,$F8,$19,$3E,$B8,$F8,$DF,$03,$23,$DC,$FC,$9F,$81,$F9,$5F,$03
	.dc.b	$F2,$FE,$06,$57,$C3,$8F,$9D,$F0,$32,$3E,$33,$FD,$DF,$F2,$2F,$F3
	.dc.b	$29,$FC,$30,$FE,$12,$4F,$82,$D7,$C1,$4A,$FE,$D3,$3F,$C2,$5B,$F0
	.dc.b	$5C,$FE,$0A,$46,$9E,$DF,$05,$DF,$E0,$64,$F2,$7C,$5F,$05,$E3,$F8
	.dc.b	$2F,$27,$C1,$4A,$EF,$1E,$7F,$81,$91,$E2,$7D,$5F,$05,$EB,$F8,$19
	.dc.b	$3E,$67,$DD,$F0,$5E,$FF,$82,$F8,$1F,$05,$2B,$D7,$1F,$0F,$E0,$64
	.dc.b	$7B,$9F,$8D,$F0,$5F,$1F,$E0,$7E,$47,$C1,$4A,$F8,$71,$FE,$21,$DB
	.dc.b	$F8,$4C,$7F,$8E,$1F,$E1,$2D,$F8,$2D,$7C,$16,$FE,$0A,$4F,$F8,$52
	.dc.b	$BF,$84,$3F,$DF,$D7,$C1,$75,$F8,$1E,$DF,$03,$2B,$84,$78,$7E,$07
	.dc.b	$C5,$F0,$32,$7A,$47,$97,$E0,$64,$77,$7D,$1F,$03,$E9,$F8,$0F,$57
	.dc.b	$C0,$CA,$F2,$47,$37,$C9,$F0,$1E,$F3,$1E,$B7,$F9,$2B,$67,$C0,$FC
	.dc.b	$32,$57,$B7,$E0,$7E,$37,$C0,$FC,$7F,$80,$93,$F0,$63,$A3,$F0,$F1
	.dc.b	$AF,$C3,$DF,$E7,$7C,$0C,$8F,$90,$FD,$1F,$80,$FA,$5F,$03,$F4,$FE
	.dc.b	$06,$57,$CD,$8E,$7B,$7F,$98,$3F,$23,$FF,$35,$4F,$F8,$B5,$B1,$EB
	.dc.b	$CB,$FC,$B9,$1F,$E2,$12,$7C,$46,$3E,$27,$3F,$07,$AF,$83,$98,$FF
	.dc.b	$93,$AF,$F8,$9F,$C2,$7F,$9D,$F9,$3F,$C9,$93,$93,$A9,$E4,$8F,$F3
	.dc.b	$E4,$FF,$18,$8F,$27,$C1,$F8,$8F,$1F,$C1,$F9,$3E,$0B,$CB,$F0,$72
	.dc.b	$BC,$11,$CD,$FF,$44,$DF,$B3,$0E,$B1,$91,$0F,$82,$F7,$92,$3D,$5F
	.dc.b	$07,$F0,$BE,$0F,$E1,$FC,$1C,$9F,$74,$73,$BE,$D1,$E0,$31,$F0,$5F
	.dc.b	$28,$8F,$C5,$F8,$3F,$9B,$F0,$5F,$3B,$E0,$E4,$FC,$98,$E8,$FC,$63
	.dc.b	$F0,$87,$CD,$F8,$3F,$AA,$47,$E8,$7C,$1F,$D8,$F8,$2F,$B3,$F0,$72
	.dc.b	$7E,$A4,$74,$7E,$91,$D0,$FC,$17,$C1,$FE,$00,$8F,$FA,$23,$F2,$4F
	.dc.b	$F5,$3F,$6F,$FA,$A1,$FF,$19,$9F,$E3,$02,$7C,$27,$59,$FC,$61,$F3
	.dc.b	$E3,$F9,$43,$FE,$7B,$6C,$F3,$F8,$4F,$F5,$54,$3E,$0E,$FC,$1E,$3E
	.dc.b	$0E,$61,$E3,$8E,$D8,$7E,$6E,$3B,$7C,$1F,$22,$3E,$9F,$83,$EB,$F0
	.dc.b	$7D,$BC,$09,$3E,$D8,$E7,$D9,$F8,$68,$F8,$98,$7B,$FE,$0F,$CA,$48
	.dc.b	$F8,$9F,$07,$E8,$F8,$3F,$4F,$C1,$C9,$F9,$11,$D1,$FC,$21,$D3,$0F
	.dc.b	$F6,$D7,$03,$E0,$BE,$09,$27,$E8,$7C,$1F,$C4,$F8,$3F,$8B,$F0,$72
	.dc.b	$7E,$A4,$75,$7F,$36,$7C,$FF,$82,$F9,$64,$7E,$D7,$C1,$FF,$B5,$FF
	.dc.b	$25,$FE,$6F,$07,$F9,$BB,$7E,$13,$5F,$09,$D7,$C5,$FC,$EC,$B9,$7F
	.dc.b	$76,$FF,$AD,$6F,$F9,$CF,$27,$FB,$B1,$BC,$1F,$EF,$76,$CF,$FB,$F1
	.dc.b	$FE,$7A,$FF,$B5,$1F,$F7,$3B,$7E,$0B,$CC,$79,$FE,$0B,$C5,$1F,$B9
	.dc.b	$97,$C1,$7A,$C9,$1C,$FD,$AF,$77,$C1,$7B,$FE,$0A,$4F,$78,$E6,$FD
	.dc.b	$33,$C9,$F0,$3F,$14,$8F,$97,$E0,$BE,$47,$B5,$F2,$7E,$0A,$4F,$A6
	.dc.b	$39,$BF,$D1,$8F,$67,$C1,$7C,$F2,$3E,$FF,$81,$FA,$5F,$05,$F4,$FE
	.dc.b	$0A,$4F,$C3,$8E,$7C,$7F,$CC,$E3,$E3,$7C,$17,$D9,$23,$F2,$7E,$07
	.dc.b	$EE,$7C,$12,$3F,$DF,$FE,$BF,$D8,$3E,$CF,$22,$7A,$F3,$FF,$35,$DB
	.dc.b	$85,$FF,$BB,$EF,$FD,$C0,$4F,$82,$CE,$6B,$DC,$FF,$95,$A8,$FF,$40
	.dc.b	$7C,$4B,$E0,$38,$CF,$06,$CE,$BE,$07,$C6,$FF,$82,$71,$CB,$D3,$E0
	.dc.b	$6E,$1F,$D1,$71,$F1,$4F,$69,$CF,$C0,$EB,$E0,$37,$F0,$32,$BE,$1C
	.dc.b	$76,$7F,$35,$8F,$4F,$81,$91,$F2,$1E,$FF,$03,$E0,$CF,$C3,$F0,$32
	.dc.b	$BE,$6C,$73,$F0,$BF,$47,$AF,$92,$7E,$1F,$C0,$FA,$08,$FD,$4F,$81
	.dc.b	$F5,$FC,$07,$B3,$E0,$64,$FD,$88,$E9,$AF,$C0,$F8,$19,$3F,$71,$F8
	.dc.b	$7F,$03,$F1,$3E,$03,$E2,$FC,$0C,$AF,$C0,$C7,$27,$F0,$7F,$03,$F2
	.dc.b	$88,$FE,$1F,$5F,$47,$E4,$DF,$CF,$BA,$BF,$06,$FF,$BC,$0F,$F7,$69
	.dc.b	$3E,$0B,$A5,$2B,$8F,$C1,$7D,$8A,$FE,$7B,$7C,$A3,$1F,$05,$9F,$81
	.dc.b	$D7,$C1,$4C,$7C,$57,$E0,$9E,$B8,$F9,$3E,$0B,$A1,$2B,$67,$E7,$F2
	.dc.b	$7C,$17,$87,$E0,$BC,$5D,$12,$B6,$29,$3D,$B2,$F7,$FC,$17,$A4,$95
	.dc.b	$C2,$9F,$05,$ED,$F8,$1F,$77,$C1,$4A,$D4,$73,$7C,$9F,$03,$F0,$C8
	.dc.b	$F3,$F8,$1F,$8D,$F0,$1F,$1F,$E0,$64,$F7,$8E,$4F,$9F,$E0,$7E,$61
	.dc.b	$1F,$27,$C0,$FE,$9B,$F2,$7F,$E7,$67,$F3,$A2,$7C,$0F,$27,$F9,$8C
	.dc.b	$9A,$63,$65,$F3,$F8,$5E,$9F,$01,$2F,$E4,$DF,$F3,$87,$F8,$F1,$FD
	.dc.b	$F8,$FF,$05,$98,$E7,$FD,$14,$76,$F8,$6F,$31,$E9,$9F,$C4,$72,$7F
	.dc.b	$10,$97,$C1,$7A,$FE,$0A,$57,$82,$3D,$57,$EF,$F8,$29,$3E,$48,$F8
	.dc.b	$5F,$03,$F0,$FE,$0A,$4F,$A2,$38,$BE,$AA,$E2,$F8,$8F,$E5,$18,$FC
	.dc.b	$4F,$82,$F9,$9F,$05,$27,$E0,$C7,$CB,$AE,$3A,$FB,$9F,$6C,$FC,$6F
	.dc.b	$2B,$EA,$7C,$14,$8C,$1F,$D3,$CF,$EE,$E7,$C8,$AF,$E7,$CF,$8C,$FA
	.dc.b	$D5,$DB,$7F,$8C,$F6,$8D,$64,$F8,$2F,$C1,$7C,$14,$9F,$AF,$1E,$77
	.dc.b	$E0,$10,$E3,$BF,$E2,$7E,$07,$F1,$5F,$05,$27,$EF,$C7,$17,$BE,$3E
	.dc.b	$37,$E3,$D7,$E4,$7E,$0B,$F2,$5F,$05,$27,$F0,$F1,$C7,$5F,$8D,$1E
	.dc.b	$17,$F0,$B5,$F9,$9F,$81,$FE,$7F,$F9,$47,$F8,$38,$FF,$07,$ED,$FD
	.dc.b	$5E,$FE,$47,$C2,$7F,$84,$8F,$F0,$9F,$91,$FC,$C3,$87,$E2,$CF,$56
	.dc.b	$BE,$E3,$F8,$0D,$BE,$0F,$BF,$C3,$C9,$CC,$79,$EB,$8E,$9E,$89,$E4
	.dc.b	$7D,$5B,$FE,$BC,$9F,$EB,$CF,$6F,$84,$F5,$FC,$27,$38,$FC,$6F,$0F
	.dc.b	$C9,$7F,$65,$BF,$C0,$AF,$83,$F0,$92,$79,$1F,$00,$FF,$41,$1F,$AD
	.dc.b	$CB,$CA,$47,$A9,$FA,$33,$DF,$1C,$DF,$43,$DD,$E8,$7E,$46,$3F,$C0
	.dc.b	$4F,$C9,$39,$9F,$80,$3A,$1C,$8F,$97,$F0,$9F,$60,$ED,$9F,$E6,$8F
	.dc.b	$F3,$A4,$75,$3F,$BE,$1D,$4F,$CE,$97,$83,$F8,$AE,$9F,$DF,$6F,$FC
	.dc.b	$26,$AF,$B5,$7B,$0F,$49,$FC,$44,$F8,$87,$A2,$FE,$C1,$2B,$F8,$76
	.dc.b	$5E,$67,$AD,$74,$AF,$61,$C8,$FD,$A9,$D6,$BD,$93,$F3,$E3,$D2,$7C
	.dc.b	$53,$F0,$B1,$FE,$01,$BF,$54,$7F,$8C,$DF,$53,$F2,$0F,$51,$DF,$83
	.dc.b	$F3,$8F,$FB,$57,$16,$9F,$87,$6C,$7D,$2B,$ED,$9F,$A5,$F0,$65,$ED
	.dc.b	$9F,$9E,$7A,$72,$FF,$4E,$3E,$DB,$E6,$FA,$6B,$E3,$3F,$BC,$3C,$AF
	.dc.b	$47,$D0,$7C,$67,$E1,$BF,$E9,$C3,$B1,$E0,$79,$BF,$19,$B1,$F7,$AF
	.dc.b	$D1,$5F,$4A,$BB,$BF,$92,$79,$9F,$4A,$BB,$BC,$E3,$E9,$57,$AA,$7F
	.dc.b	$85,$CF,$23,$E8,$5F,$67,$E9,$5F,$EC,$DE,$47,$E1,$EF,$F0,$AF,$13
	.dc.b	$F7,$93,$C5,$FD,$F4,$F1,$BF,$05,$78,$B3,$FA,$15,$C0,$F8,$F8,$7D
	.dc.b	$67,$F2,$EF,$E9,$1F,$A5,$8F,$DD,$3F,$9E,$46,$E3,$F8,$76,$3E,$1C
	.dc.b	$FF,$17,$1F,$E5,$83,$F5,$87,$C5,$C3,$E4,$57,$CE,$BF,$C4,$61,$E4
	.dc.b	$BF,$C8,$DF,$43,$DD,$C1,$FC,$A5,$FB,$57,$E5,$3E,$7E,$5E,$E8,$F8
	.dc.b	$C4,$3F,$8C,$9F,$1E,$BA,$CF,$E1,$E3,$F0,$D8,$F9,$0F,$C8,$C7,$57
	.dc.b	$11,$CE,$BB,$3F,$50,$EA,$72,$3E,$D9,$F3,$23,$F8,$2E,$7F,$4F,$2F
	.dc.b	$7C,$F8,$F3,$F9,$13,$98,$FE,$1D,$9F,$C3,$9C,$C7,$D4,$BF,$29,$E4
	.dc.b	$7C,$58,$79,$09,$5C,$63,$C4,$7E,$30,$E4,$7C,$A9,$E5,$3C,$70,$F2
	.dc.b	$CF,$AC,$FC,$31,$F4,$CC,$67,$F4,$DF,$B8,$73,$3F,$59,$5F,$A0,$9E
	.dc.b	$79,$7E,$1C,$FA,$04,$8F,$8B,$3E,$43,$99,$F8,$93,$F8,$03,$CD,$FE
	.dc.b	$18,$F0,$7C,$91,$CD,$F7,$1F,$C5,$CF,$EA,$6F,$E1,$A7,$F3,$51,$C8
	.dc.b	$DC,$71,$7D,$EF,$9E,$7E,$76,$9F,$C8,$27,$FC,$49,$87,$C7,$7E,$B9
	.dc.b	$F6,$CF,$9E,$7E,$C8,$F6,$47,$F8,$0B,$A7,$C3,$1E,$C3,$F8,$D1,$F3
	.dc.b	$F5,$FC,$AD,$FF,$0F,$8F,$9F,$3E,$A8,$FC,$0C,$7C,$E3,$F2,$98,$FC
	.dc.b	$E8,$F8,$98,$FA,$EF,$9D,$7B,$A7,$D6,$F3,$39,$1F,$8A,$AE,$4E,$CF
	.dc.b	$CD,$9F,$3A,$3A,$9C,$9F,$DE,$E5,$F8,$3C,$BE,$24,$7B,$4F,$D2,$E1
	.dc.b	$F2,$DE,$6F,$97,$5F,$D0,$46,$E7,$F9,$83,$F2,$B1,$F4,$BF,$2C,$FB
	.dc.b	$87,$83,$2F,$97,$7F,$CE,$34,$FD,$04,$F9,$CF,$51,$F8,$B8,$FD,$E3
	.dc.b	$F9,$73,$19,$7E,$35,$F1,$4F,$29,$F3,$47,$DA,$3F,$B0,$67,$E0,$AF
	.dc.b	$E6,$71,$F6,$75,$FD,$BB,$E6,$3F,$A2,$E9,$F1,$6B,$84,$7E,$02,$FE
	.dc.b	$23,$F9,$0B,$FC,$04,$62,$BF,$5D,$AF,$E4,$6B,$F9,$99,$FE,$13,$BF
	.dc.b	$E6,$77,$F5,$23,$E6,$BC,$8C,$3E,$E3,$E0,$4F,$E4,$DE,$47,$E6,$8F
	.dc.b	$51,$F4,$75,$FA,$B8,$7C,$D8,$FC,$F1,$F3,$4F,$A8,$FF,$60,$9F,$33
	.dc.b	$FA,$F8,$EA,$7F,$84,$2B,$C7,$A7,$DC,$25,$F6,$8F,$EA,$D3,$DA,$BE
	.dc.b	$8B,$F2,$DF,$B2,$FA,$CF,$CC,$1E,$97,$E5,$D7,$5C,$BF,$AF,$E5,$F2
	.dc.b	$AB,$F4,$07,$1D,$7F,$2F,$7E,$D8,$F8,$27,$B6,$FF,$AD,$D7,$18,$FC
	.dc.b	$81,$EC,$9E,$07,$82,$FF,$02,$FC,$38,$EF,$3F,$81,$78,$61,$F8,$89
	.dc.b	$E1,$1F,$3C,$FC,$71,$FC,$3B,$3F,$C3,$BF,$AA,$9F,$93,$7D,$9F,$D9
	.dc.b	$47,$E3,$B5,$DE,$5F,$D0,$A3,$E5,$4F,$E0,$DF,$D1,$E3,$EB,$78,$1E
	.dc.b	$43,$FA,$F6,$BF,$A1,$9E,$11,$F7,$9E,$F9,$7D,$2A,$F6,$4F,$EE,$72
	.dc.b	$FC,$9B,$FB,$0C,$FE,$93,$FD,$EB,$2F,$A3,$AF,$E4,$63,$81,$F6,$71
	.dc.b	$F6,$BF,$0C,$FC,$03,$FC,$81,$FB,$17,$EE,$CB,$CD,$1F,$A0,$3D,$B7
	.dc.b	$D9,$FD,$B9,$ED,$3E,$51,$E5,$D3,$E5,$3F,$C8,$B0,$FC,$CE,$1F,$D9
	.dc.b	$30,$ED,$3F,$28,$EC,$7C,$33,$81,$F7,$AF,$E8,$C7,$C2,$8D,$1F,$BD
	.dc.b	$38,$E3,$F8,$C9,$F3,$47,$DF,$3C,$DA,$7F,$22,$7F,$25,$8F,$E6,$A7
	.dc.b	$71,$FC,$08,$F9,$27,$EB,$4F,$61,$F2,$74,$FC,$93,$F9,$83,$6F,$E6
	.dc.b	$31,$F9,$F1,$F9,$FD,$7F,$64,$FE,$08,$F8,$07,$A4,$F5,$EB,$F8,$9A
	.dc.b	$FC,$15,$FC,$09,$FE,$D8,$7C,$0C,$BF,$02,$FF,$5A,$3F,$C8,$7A,$7E
	.dc.b	$06,$BF,$C8,$51,$F8,$13,$D2,$78,$CF,$D1,$9E,$F7,$6F,$BC,$ED,$97
	.dc.b	$DA,$7F,$82,$4F,$DA,$7F,$AE,$E7,$FB,$EC,$FF,$86,$C7,$EF,$AF,$F8
	.dc.b	$69,$F1,$9F,$D1,$9F,$1A,$37,$1F,$B6,$8F,$21,$F9,$43,$FA,$56,$5F
	.dc.b	$A2,$8F,$29,$FE,$7E,$8E,$D3,$FA,$AC,$7F,$01,$1F,$08,$FC,$A9,$DC
	.dc.b	$FE,$89,$1F,$52,$7C,$AF,$EC,$0F,$D1,$57,$77,$FA,$C4,$77,$3C,$C7
	.dc.b	$09,$FC,$A9,$F8,$17,$CB,$1E,$C3,$B1,$E5,$CF,$F0,$33,$FE,$4C,$D7
	.dc.b	$F5,$AF,$E0,$70,$F4,$BF,$EA,$0C,$BE,$D5,$F8,$8E,$91,$F6,$B0,$F1
	.dc.b	$3F,$50,$FD,$31,$F9,$ED,$7F,$4D,$7F,$E8,$83,$F0,$F3,$FD,$87,$5F
	.dc.b	$CF,$BF,$D2,$EB,$E3,$1F,$BA,$3F,$02,$7E,$7E,$FE,$33,$F5,$27,$F2
	.dc.b	$8F,$E1,$EB,$B6,$BF,$87,$D3,$F3,$57,$F3,$2F,$78,$7C,$B8,$F8,$B8
	.dc.b	$7E,$E3,$4F,$8B,$AF,$F2,$D8,$FB,$A6,$4F,$CB,$D7,$9C,$F9,$EF,$DC
	.dc.b	$3E,$25,$7C,$F7,$6F,$F0,$F7,$F8,$0E,$BF,$43,$3F,$EC,$B1,$A3,$F3
	.dc.b	$7A,$7F,$14,$7C,$27,$D5,$C7,$CB,$1F,$31,$FF,$5A,$69,$FC,$26,$7F
	.dc.b	$20,$7F,$0F,$3E,$5E,$5F,$C3,$F2,$FF,$24,$4E,$E7,$EF,$1F,$96,$8F
	.dc.b	$5B,$F4,$70,$F9,$6F,$D1,$3F,$1E,$7E,$AA,$37,$5F,$1D,$FC,$AE,$5F
	.dc.b	$06,$7F,$54,$7F,$A0,$34,$FC,$E3,$FE,$5C,$3F,$85,$DF,$67,$F6,$35
	.dc.b	$FB,$73,$F8,$8B,$F8,$B3,$78,$FC,$07,$FA,$81,$F0,$0F,$E1,$57,$F9
	.dc.b	$77,$EA,$1F,$8A,$3A,$BF,$8A,$7C,$9A,$7E,$12,$3F,$BC,$1F,$84,$8F
	.dc.b	$A0,$7D,$53,$FA,$A9,$F6,$8F,$E5,$37,$F6,$9F,$7E,$BF,$A8,$7D,$C7
	.dc.b	$F9,$40,$DE,$7F,$47,$1F,$F0,$FC,$EE,$7F,$C3,$E6,$CF,$C0,$D7,$59
	.dc.b	$F8,$97,$F9,$B9,$F8,$87,$E8,$71,$F8,$8F,$F3,$33,$EE,$9F,$83,$9F
	.dc.b	$AF,$AF,$E4,$71,$F1,$5F,$F7,$1C,$7C,$5A,$7F,$2B,$C3,$FA,$31,$E2
	.dc.b	$AD,$BF,$E4,$E3,$F1,$15,$F3,$DF,$C4,$63,$E2,$C7,$F7,$D1,$F5,$5F
	.dc.b	$D4,$4E,$EF,$F2,$8F,$F2,$9B,$EB,$5F,$EC,$6C,$3C,$7A,$FE,$8F,$0F
	.dc.b	$C6,$CF,$EC,$CF,$9E,$78,$8F,$9E,$FE,$F5,$FD,$99,$B7,$E4,$BF,$C2
	.dc.b	$4F,$92,$ED,$FC,$33,$FC,$73,$5F,$E1,$31,$B3,$F5,$73,$FB,$17,$FD
	.dc.b	$35,$7E,$8B,$FE,$A9,$1F,$86,$3F,$63,$9F,$CF,$7F,$2E,$7D,$43,$F2
	.dc.b	$E7,$A3,$1F,$C8,$BE,$83,$F2,$27,$B8,$FB,$4F,$C9,$8E,$91,$F7,$F1
	.dc.b	$FC,$5B,$F0,$CF,$C5,$BE,$DD,$3F,$5C,$FB,$75,$FE,$63,$7F,$C5,$6B
	.dc.b	$CE,$7E,$58,$F3,$CF,$47,$C2,$79,$CF,$D6,$D7,$49,$FE,$01,$3F,$3A
	.dc.b	$7F,$5A,$7C,$E8,$F6,$BF,$39,$D9,$FB,$3C,$3E,$FC,$7E,$B4,$FC,$81
	.dc.b	$E2,$8F,$CB,$3E,$DD,$3F,$2C,$FE,$40,$FA,$66,$CF,$CB,$1E,$83,$E1
	.dc.b	$C7,$8A,$FD,$11,$F8,$58,$F1,$3D,$73,$FF,$73,$BE,$73,$47,$F2,$2C
	.dc.b	$3F,$8A,$BF,$84,$3F,$8A,$9E,$23,$EA,$4F,$A1,$F1,$5F,$A0,$FD,$0B
	.dc.b	$FE,$48,$CF,$F1,$4F,$F6,$A3,$A1,$FB,$D3,$A5,$7E,$B1,$FC,$53,$FA
	.dc.b	$C7,$EA,$1F,$0C,$FF,$26,$9F,$0C,$FD,$61,$ED,$3F,$C6,$87,$B4,$FC
	.dc.b	$7B,$F6,$71,$FA,$71,$F8,$43,$E9,$9E,$C3,$E6,$9F,$9A,$3E,$69,$F8
	.dc.b	$4D,$7F,$81,$61,$FE,$D6,$D7,$F5,$73,$F8,$43,$DB,$7F,$95,$AD,$C7
	.dc.b	$F9,$F8,$FE,$11,$5F,$95,$9F,$E3,$31,$F9,$53,$C4,$7B,$0F,$9B,$97
	.dc.b	$D8,$CF,$EE,$C7,$B2,$BE,$EB,$A3,$E0,$C6,$8F,$E5,$D1,$A8,$F6,$67
	.dc.b	$F5,$F5,$FD,$7E,$1F,$44,$FD,$24,$EB,$0F,$06,$3E,$A8,$FF,$35,$EB
	.dc.b	$FA,$E7,$FB,$E9,$F9,$DA,$FE,$36,$F9,$4F,$C6,$63,$FC,$F5,$F8,$AF
	.dc.b	$E5,$EB,$A6,$5F,$E3,$6A,$F0,$9F,$81,$AF,$A2,$7E,$05,$FE,$FE,$FE
	.dc.b	$07,$4F,$E2,$BA,$7F,$72,$8D,$E9,$F9,$27,$FC,$85,$AF,$E4,$9F,$F2
	.dc.b	$66,$5F,$CD,$F4,$FE,$41,$1F,$8C,$3E,$8E,$1D,$34,$FE,$D5,$1F,$81
	.dc.b	$3F,$2F,$5F,$81,$8F,$85,$1F,$C9,$72,$FB,$6F,$E3,$1F,$99,$3F,$A7
	.dc.b	$8F,$98,$7D,$6D,$3E,$09,$FE,$22,$7F,$5E,$78,$E7,$8B,$F9,$47,$E2
	.dc.b	$65,$F6,$63,$F1,$11,$A7,$F1,$07,$63,$FB,$83,$F5,$34,$FC,$1C,$7D
	.dc.b	$43,$FA,$14,$7F,$88,$32,$F8,$6F,$CE,$8F,$86,$7E,$E5,$D3,$E6,$74
	.dc.b	$78,$72,$F9,$11,$FC,$94,$F9,$07,$AF,$1F,$90,$FE,$88,$FC,$49,$FB
	.dc.b	$39,$F3,$67,$F5,$E3,$FC,$69,$3F,$5D,$F0,$3F,$DB,$5F,$C1,$1E,$57
	.dc.b	$4F,$F0,$D3,$C0,$7E,$C4,$FE,$1A,$FE,$C4,$FE,$A0,$FC,$A7,$CA,$E8
	.dc.b	$F2,$CF,$DC,$3C,$AF,$80,$FC,$59,$FC,$5B,$1F,$F2,$44,$FF,$8B,$F1
	.dc.b	$F9,$F3,$FE,$8B,$36,$7E,$5B,$1F,$9F,$3E,$B8,$F9,$E6,$AB,$F9,$64
	.dc.b	$EA,$3F,$96,$1E,$13,$E2,$9E,$17,$4F,$E8,$0D,$5F,$F3,$C3,$EC,$BF
	.dc.b	$15,$FB,$35,$F0,$1F,$D8,$BF,$16,$3D,$9A,$FF,$06,$8D,$1F,$9B,$9F
	.dc.b	$C9,$D7,$F0,$6A,$FD,$29,$F3,$E3,$9B,$F2,$A7,$E7,$BF,$A5,$79,$BF
	.dc.b	$9D,$7C,$0F,$EE,$AB,$C0,$F3,$8F,$01,$AB,$FD,$14,$73,$34,$73,$3C
	.dc.b	$07,$A8,$FB,$C6,$8F,$D3,$D7,$A9,$FE,$53,$8F,$ED,$1F,$F3,$36,$BF
	.dc.b	$C3,$6B,$BD,$2F,$FD,$CD,$F0,$6B,$93,$97,$19,$A7,$50,$9D,$29,$63
	.dc.b	$FC,$0E,$3F,$D8,$55,$C2,$B7,$4F,$84,$C6,$1D,$23,$17,$E2,$8C,$5F
	.dc.b	$3B,$FF,$14,$7C,$27,$0A,$F2,$FC,$25,$7C,$4D,$C1,$3C,$70,$FF,$07
	.dc.b	$7C,$1E,$7E,$12,$61,$DC,$F7,$C6,$DF,$F1,$5F,$C2,$73,$F8,$49,$1B
	.dc.b	$3E,$04,$6D,$FF,$5B,$7C,$1F,$87,$E1,$24,$6F,$34,$EF,$85,$C7,$E7
	.dc.b	$F8,$4F,$47,$C2,$4C,$38,$46,$E7,$D9,$F0,$7E,$DF,$84,$95,$D6,$34
	.dc.b	$F7,$F8,$4F,$89,$E5,$7D,$6E,$5F,$AE,$DE,$3F,$A1,$3C,$2B,$EE,$47
	.dc.b	$DE,$97,$C2,$72,$F8,$E9,$5F,$06,$72,$FE,$3B,$BF,$FA,$7A,$DF,$DA
	.dc.b	$EF,$E1,$7C,$7F,$0B,$23,$07,$13,$E6,$4B,$BF,$D3,$F0,$92,$73,$1E
	.dc.b	$CF,$85,$F6,$FC,$2C,$9F,$24,$7E,$0F,$C2,$FF,$0E,$F8,$7F,$EB,$BD
	.dc.b	$58,$F3,$BF,$79,$FA,$18,$CB,$D8,$F7,$7C,$2F,$CA,$F8,$59,$18,$3D
	.dc.b	$07,$21,$3C,$9C,$37,$93,$E8,$DF,$E5,$F5,$F7,$57,$CB,$CB,$EB,$FC
	.dc.b	$2C,$8C,$1D,$AF,$EC,$FC,$2F,$DC,$24,$7D,$3C,$3E,$FF,$C2,$7E,$03
	.dc.b	$E1,$64,$FD,$78,$F8,$9C,$9F,$E0,$0C,$8F,$E1,$BE,$16,$57,$DD,$7F
	.dc.b	$13,$F0,$BF,$8A,$F8,$4F,$C5,$FC,$2C,$AF,$C1,$47,$47,$FB,$20,$C3
	.dc.b	$AB,$F8,$7F,$82,$FC,$99,$1F,$C5,$7C,$2F,$EE,$7F,$29,$7C,$35,$F8
	.dc.b	$89,$5F,$AF,$9C,$BF,$0F,$C0,$FE,$27,$1F,$8E,$8F,$A4,$DE,$3F,$E2
	.dc.b	$AD,$9D,$FC,$37,$7F,$86,$F0,$7C,$2C,$C3,$B4,$7A,$DF,$ED,$47,$AF
	.dc.b	$8B,$CD,$F0,$D2,$3C,$8F,$A7,$C0,$F5,$7C,$2F,$AF,$E1,$A5,$7A,$63
	.dc.b	$A6,$BF,$D3,$23,$A1,$ED,$F8,$6F,$84,$47,$E0,$77,$7C,$5C,$BE,$37
	.dc.b	$C3,$49,$F8,$91,$D3,$7F,$E9,$B1,$FC,$EF,$97,$E8,$BF,$13,$E1,$7E
	.dc.b	$71,$1F,$99,$F0,$DF,$47,$E1,$7E,$97,$C3,$49,$FA,$11,$D5,$FE,$A8
	.dc.b	$7A,$0F,$9F,$F0,$BF,$64,$8F,$D7,$F8,$6F,$B9,$F0,$DF,$CD,$7F,$2A
	.dc.b	$FE,$93,$3F,$A7,$6F,$0F,$D6,$9E,$D3,$FD,$73,$5F,$E9,$FA,$FA,$AB
	.dc.b	$FA,$7D,$BE,$1B,$A1,$2F,$F9,$A4,$3E,$1F,$C1,$F0,$FE,$1F,$87,$95
	.dc.b	$A8,$EF,$E7,$7F,$90,$23,$C9,$F0,$FE,$72,$3D,$3E,$1F,$D5,$F0,$DE
	.dc.b	$BF,$87,$93,$E0,$8E,$6F,$F8,$EC,$FE,$D9,$9F,$F5,$27,$CB,$F0,$DF
	.dc.b	$0C,$F8,$9F,$0F,$F1,$7E,$1B,$E3,$7C,$3C,$AF,$5C,$74,$7F,$C6,$36
	.dc.b	$F8,$6F,$98,$48,$F8,$3F,$0F,$F3,$FE,$17,$E8,$7C,$3C,$9F,$8B,$1C
	.dc.b	$DF,$F1,$88,$B7,$C2,$FD,$72,$47,$CC,$F8,$6F,$B5,$F0,$9F,$6F,$E1
	.dc.b	$64,$FD,$08,$E9,$D3,$FE,$37,$EB,$FF,$03,$E2,$F8,$5F,$C1,$12,$FF
	.dc.b	$B5,$7E,$57,$FD,$4D,$FF,$09,$3F,$D3,$1F,$E5,$8B,$E1,$7F,$BF,$36
	.dc.b	$38,$FC,$37,$83,$0F,$F5,$B6,$3F,$E1,$4A,$7C,$37,$02,$5F,$83,$E1
	.dc.b	$B9,$FB,$FD,$3E,$16,$4F,$92,$3A,$3F,$E8,$EC,$7F,$C4,$38,$3E,$1B
	.dc.b	$C6,$49,$F5,$FC,$37,$9B,$E1,$BC,$FE,$34,$9F,$7C,$74,$7F,$DE,$1A
	.dc.b	$F9,$FE,$17,$DA,$48,$F8,$BF,$0B,$F0,$3E,$13,$E0,$FC,$2C,$9F,$93
	.dc.b	$1C,$FC,$6F,$D5,$ED,$FA,$8F,$4B,$E3,$FC,$2C,$BF,$9E,$FC,$AF,$85
	.dc.b	$F9,$7F,$09,$F3,$3E,$16,$57,$D4,$8E,$8F,$ED,$38,$DF,$0B,$F4,$89
	.dc.b	$1F,$6B,$E1,$7F,$C6,$5C,$1F,$CD,$A3,$FB,$22,$F8,$4C,$9A,$F8,$5E
	.dc.b	$B3,$FE,$2B,$E5,$FD,$F6,$FF,$E4,$EB,$FE,$71,$3F,$E5,$56,$A6,$8F
	.dc.b	$84,$F0,$FC,$1F,$8B,$E1,$25,$7F,$8D,$E9,$3E,$08,$FE,$F1,$87,$81
	.dc.b	$ED,$F0,$9E,$A2,$3C,$7E,$13,$DB,$F0,$7E,$EF,$84,$93,$D6,$3A,$75
	.dc.b	$FF,$99,$30,$E9,$3E,$7F,$84,$F8,$A4,$7C,$9F,$09,$F2,$3A,$7E,$4F
	.dc.b	$C2,$49,$F4,$47,$4E,$1F,$B9,$1E,$BF,$84,$F9,$E4,$7D,$DF,$09,$F4
	.dc.b	$BE,$0F,$E9,$FC,$24,$9F,$85,$1C,$DF,$E8,$D8,$7D,$8F,$84,$91,$F1
	.dc.b	$DF,$B7,$F0,$9F,$73,$E1,$3F,$CD,$DF,$96,$7E,$3A,$BF,$D1,$0D,$7E
	.dc.b	$A4,$75,$7F,$A3,$70,$FF,$A1,$A7,$F9,$C4,$7F,$9A,$5A,$FF,$8A,$5B
	.dc.b	$1B,$F8,$4E,$FF,$07,$E0,$F8,$49,$7C,$63,$B7,$27,$CE,$C1,$F0,$9E
	.dc.b	$62,$57,$7F,$84,$F4,$FC,$1F,$AB,$E1,$24,$F8,$E3,$FD,$15,$E5,$7A
	.dc.b	$CF,$E8,$9A,$FA,$7E,$13,$E0,$92,$BD,$5F,$09,$F1,$3E,$0F,$E2,$FC
	.dc.b	$24,$9F,$74,$74,$7F,$C6,$1C,$BF,$2B,$E1,$24,$7C,$37,$E6,$FC,$27
	.dc.b	$CE,$F8,$3F,$9F,$F0,$92,$BE,$44,$73,$7E,$57,$C2,$7D,$43,$0F,$CB
	.dc.b	$F8,$4F,$AE,$47,$E8,$7C,$1F,$DA,$F8,$4F,$B7,$F0,$72,$7E,$A4,$73
	.dc.b	$7F,$D3,$DF,$96,$FF,$35,$4D,$E8,$FE,$8F,$1A,$F8,$3F,$F5,$94,$FF
	.dc.b	$39,$E4,$FF,$62,$B7,$C5,$FF,$64,$BF,$CE,$6B,$FC,$D0,$FF,$36,$7B
	.dc.b	$FC,$17,$23,$9F,$C1,$F4,$F8,$2E,$BF,$07,$2B,$CB,$1E,$0F,$83,$91
	.dc.b	$E8,$7C,$7F,$07,$E4,$F8,$39,$3E,$C7,$CF,$E1,$7A,$3D,$5F,$4F,$C1
	.dc.b	$CA,$F8,$31,$EC,$F8,$29,$1F,$11,$F7,$FC,$1F,$C0,$F8,$39,$3F,$21
	.dc.b	$F8,$7F,$07,$F1,$3E,$0B,$E2,$FC,$1C,$AF,$9B,$1F,$23,$E0,$A4,$7D
	.dc.b	$07,$E5,$FC,$17,$FB,$77,$F2,$EF,$E5,$9D,$1F,$EF,$07,$F9,$65,$7F
	.dc.b	$B4,$DA,$FF,$A4,$DD,$F4,$7F,$BE,$1F,$E5,$24,$F8,$AE,$9F,$05,$D7
	.dc.b	$E0,$65,$F3,$7C,$1F,$05,$E1,$F8,$29,$3D,$DF,$27,$C1,$79,$7E,$07
	.dc.b	$CD,$F0,$52,$BC,$91,$E9,$F8,$29,$1E,$77,$D9,$F0,$5E,$DF,$3A,$4F
	.dc.b	$AD,$F8,$1F,$05,$F0,$7E,$0B,$E1,$7C,$14,$AF,$81,$1F,$17,$E0,$7E
	.dc.b	$37,$C1,$49,$F8,$91,$F2,$7E,$0A,$47,$C7,$7E,$67,$C1,$7C,$DF,$82
	.dc.b	$4F,$FB,$F7,$D8,$FE,$38,$7F,$1D,$E7,$4E,$A7,$FD,$81,$3F,$C6,$3A
	.dc.b	$30,$FF,$94,$63,$97,$C1,$6F,$A9,$5A,$F8,$8C,$3C,$3E,$0A,$DF,$8B
	.dc.b	$04,$62,$11,$87,$CD,$F0,$79,$8F,$E4,$31,$E9,$F8,$2C,$19,$F8,$3D
	.dc.b	$7C,$16,$FE,$0E,$57,$BE,$39,$7C,$14,$8F,$84,$F5,$F8,$2E,$DF,$05
	.dc.b	$DF,$29,$5F,$1E,$3C,$5F,$05,$E3,$F8,$29,$3F,$2E,$3C,$DF,$03,$23
	.dc.b	$E7,$3E,$9F,$82,$F5,$7C,$17,$AF,$E0,$A5,$7D,$38,$F7,$7C,$17,$BF
	.dc.b	$E0,$64,$FD,$78,$F8,$5F,$05,$F1,$FF,$2F,$FE,$30,$7F,$19,$DB,$FC
	.dc.b	$A5,$5F,$7B,$A7,$E9,$BF,$80,$9F,$E2,$CD,$F8,$8E,$47,$3F,$83,$E9
	.dc.b	$F0,$3D,$7E,$0A,$57,$3E,$CF,$C2,$61,$FC,$6E,$18,$F6,$50,$F8,$1F
	.dc.b	$37,$C1,$49,$E4,$78,$E3,$D5,$F0,$5E,$BF,$82,$93,$A3,$9C,$7B,$FE
	.dc.b	$0B,$E0,$7C,$14,$9D,$62,$9F,$89,$F0,$32,$72,$FC,$7F,$82,$F9,$1F
	.dc.b	$05,$F2,$7E,$0A,$57,$28,$F9,$9F,$05,$23,$AB,$F3,$FE,$0B,$E8,$7C
	.dc.b	$0C,$9F,$0B,$F4,$FE,$0B,$EA,$7C,$17,$D5,$F8,$29,$5E,$68,$FC,$F7
	.dc.b	$E6,$1F,$C6,$23,$F6,$6D,$4D,$3F,$8F,$F2,$FC,$D7,$FC,$8A,$4F,$83
	.dc.b	$E7,$F0,$7D,$3E,$0F,$AF,$C1,$CC,$3E,$55,$FF,$91,$1B,$F0,$7E,$22
	.dc.b	$3A,$F8,$3F,$2F,$C1,$48,$E2,$FA,$3E,$0F,$D3,$F0,$7E,$AF,$83,$95
	.dc.b	$DA,$3D,$BF,$07,$23,$C2,$FC,$0F,$83,$F8,$3F,$05,$F0,$BE,$0E,$57
	.dc.b	$9A,$3E,$2F,$C1,$48,$F4,$BF,$23,$E0,$FE,$4F,$C1,$C9,$F6,$BF,$33
	.dc.b	$E0,$FE,$6F,$C1,$49,$F8,$2F,$F0,$DF,$CC,$7F,$C7,$47,$F1,$B8,$FE
	.dc.b	$46,$FF,$1B,$7F,$37,$E5,$7F,$21,$8C,$74,$FE,$75,$E1,$27,$F8,$B3
	.dc.b	$C0,$E7,$F0,$3D,$CF,$07,$C0,$F8,$7E,$03,$C5,$F0,$32,$BF,$3D,$7F
	.dc.b	$C7,$1E,$47,$5F,$80,$F4,$7C,$0C,$8D,$C7,$AF,$E0,$7D,$9F,$03,$27
	.dc.b	$9C,$7B,$FE,$06,$47,$67,$E1,$7B,$1F,$0F,$E0,$24,$F8,$9F,$8D,$F0
	.dc.b	$3F,$1F,$E0,$64,$F9,$9F,$95,$F0,$3D,$23,$CF,$AB,$1C,$71,$7C,$7C
	.dc.b	$BA,$D6,$B2,$B3,$F2,$F2,$EB,$3C,$B2,$B3,$F3,$F2,$FB,$1F,$01,$F6
	.dc.b	$7E,$06,$5F,$C8,$8F,$B9,$F0,$12,$3E,$5B,$F7,$FE,$07,$F0,$1F,$03
	.dc.b	$27,$E7,$BF,$D3,$FC,$1F,$F2,$01,$FC,$92,$0F,$F8,$E7,$2F,$E3,$8D
	.dc.b	$E9,$FE,$C8,$EC,$FE,$45,$1A,$F7,$BA,$1D,$7D,$FE,$DE,$F7,$7F,$7E
	.dc.b	$63,$FC,$77,$0F,$E4,$CD,$F7,$BC,$9E,$FC,$8C,$1B,$F7,$7D,$04,$79
	.dc.b	$7B,$FE,$BF,$7A,$47,$57,$DD,$EF,$73,$8F,$C6,$E2,$C3,$1B,$7B,$B2
	.dc.b	$E9,$5A,$CA,$CF,$C2,$CB,$AC,$F2,$CA,$CF,$C6,$CB,$E6,$7B,$B2,$BD
	.dc.b	$CF,$CF,$F7,$BE,$87,$BD,$27,$E1,$3F,$4F,$DE,$FA,$9E,$F7,$D5,$F7
	.dc.b	$A5,$7C,$78,$E8,$FA,$B0,$B9,$7D,$6C,$7A,$4F,$BB,$1B,$3F,$6B,$1E
	.dc.b	$B3,$F4,$71,$EA,$FC,$4C,$79,$F2,$FD,$EF,$77,$F0,$C6,$27,$FC,$53
	.dc.b	$C4,$CB,$AD,$7E,$11,$F5,$4F,$9C,$EC,$43,$B1,$F8,$1F,$7B,$FC,$41
	.dc.b	$F9,$97,$F4,$13,$FA,$15,$A3,$3F,$15,$2B,$FB,$25,$7F,$44,$C3,$FC
	.dc.b	$6C,$BE,$03,$9F,$C0,$74,$F8,$09,$7C,$23,$BF,$C5,$48,$E6,$FE,$8F
	.dc.b	$2E,$D1,$FC,$4B,$0E,$CF,$F9,$21,$A9,$F0,$3D,$4F,$2D,$F8,$1E,$57
	.dc.b	$EE,$F8,$19,$1E,$A7,$E0,$FC,$07,$C2,$F8,$09,$3E,$E7,$E2,$FC,$07
	.dc.b	$C6,$F8,$09,$3F,$09,$F9,$3E,$FF,$48,$F4,$7C,$07,$CC,$31,$D1,$FE
	.dc.b	$15,$D5,$8F,$6A,$F7,$3D,$9E,$84,$3E,$16,$1D,$9F,$A3,$C7,$F6,$08
	.dc.b	$FD,$4F,$80,$FB,$7E,$FF,$DC,$F8,$09,$3F,$62,$3E,$FF,$C0,$7E,$03
	.dc.b	$E0,$24,$FD,$C8,$E6,$FD,$1C,$2E,$9E,$9C,$7A,$CF,$E0,$B1,$EA,$FD
	.dc.b	$7C,$7A,$BF,$86,$C7,$AB,$F6,$F1,$E7,$CF,$F8,$8F,$80,$FC,$89,$89
	.dc.b	$FB,$D7,$D3,$2F,$91,$8F,$82,$79,$1D,$48,$60,$FC,$47,$C0,$7E,$73
	.dc.b	$E0,$24,$7E,$62,$3F,$CF,$DF,$99,$FF,$A3,$63,$FA,$FB,$6F,$FC,$94
	.dc.b	$DC,$BF,$D2,$4F,$F7,$03,$FB,$9B,$8F,$81,$E6,$74,$F8,$7E,$BF,$05
	.dc.b	$30,$FF,$25,$5A,$3C,$71,$E0,$3C,$64,$37,$F0,$FC,$DE,$37,$D9,$E7
	.dc.b	$7E,$AF,$83,$F5,$FC,$1C,$9E,$F1,$EE,$F8,$2F,$7F,$C1,$C9,$F1,$C7
	.dc.b	$C2,$F8,$29,$1E,$67,$E2,$FC,$1F,$38,$F8,$BF,$05,$F2,$0C,$70,$7C
	.dc.b	$6B,$E9,$3F,$2B,$0F,$03,$CC,$87,$CE,$C3,$AB,$F1,$3E,$0B,$E9,$91
	.dc.b	$F8,$FF,$05,$F5,$BE,$0B,$EB,$FC,$14,$9F,$97,$1F,$6B,$E0,$BE,$DF
	.dc.b	$C0,$C9,$F9,$F1,$F7,$BE,$0B,$EF,$FC,$14,$9F,$A7,$1D,$5F,$AB,$87
	.dc.b	$57,$F0,$78,$75,$7E,$CE,$1D,$5F,$C3,$E1,$F8,$AF,$82,$91,$F7,$9E
	.dc.b	$CF,$23,$A9,$0C,$1C,$8E,$A4,$30,$7E,$53,$E0,$BF,$2E,$47,$FD,$29
	.dc.b	$F9,$A7,$FA,$7F,$87,$FD,$40,$FF,$82,$4F,$F0,$34,$7F,$BB,$9A,$9D
	.dc.b	$7C,$17,$FB,$7D,$FF,$04,$F6,$7F,$BB,$DB,$3C,$7E,$0F,$5F,$07,$23
	.dc.b	$C2,$F1,$F8,$3E,$5F,$0F,$CF,$E2,$25,$79,$A3,$B6,$52,$3D,$2F,$87
	.dc.b	$0F,$17,$C2,$49,$F6,$BE,$5F,$85,$F3,$7C,$2F,$9F,$E1,$65,$7C,$28
	.dc.b	$F5,$7C,$2C,$8F,$8A,$FB,$7E,$17,$DD,$F0,$B2,$7E,$4B,$F0,$7E,$17
	.dc.b	$E1,$7C,$0C,$9F,$9A,$FC,$5F,$85,$F8,$DF,$09,$F1,$FE,$16,$57,$D2
	.dc.b	$8F,$95,$F0,$B2,$3E,$AB,$F3,$7E,$0B,$A4,$7F,$9B,$39,$FF,$BC,$3D
	.dc.b	$8F,$B7,$87,$67,$EE,$E1,$D9,$E1,$81,$FF,$6A,$69,$FE,$B7,$27,$C1
	.dc.b	$E6,$3F,$CF,$31,$FD,$98,$FE,$CA,$E3,$E1,$35,$D7,$FE,$E7,$9C,$B7
	.dc.b	$E0,$F3,$D9,$CE,$7F,$D7,$17,$D3,$E1,$32,$F8,$E3,$37,$CF,$E1,$33
	.dc.b	$29,$7C,$26,$1B,$1E,$A8,$C7,$C0,$67,$E1,$35,$F0,$32,$BD,$F1,$C7
	.dc.b	$E1,$39,$7C,$24,$9F,$87,$1D,$7E,$0E,$47,$C6,$7C,$1F,$07,$E1,$F8
	.dc.b	$09,$3F,$29,$F2,$7C,$07,$97,$E0,$BC,$DF,$07,$2B,$E7,$C7,$A7,$E0
	.dc.b	$E4,$7D,$27,$D9,$F0,$5E,$DF,$83,$F8,$3F,$9A,$FF,$49,$AF,$B0,$D7
	.dc.b	$E2,$46,$BE,$0A,$47,$DD,$E6,$74,$7D,$BB,$FF,$5F,$9F,$D4,$27,$9F
	.dc.b	$C1,$76,$23,$C6,$78,$7C,$1F,$8B,$E0,$7C,$7F,$05,$2B,$47,$92,$3C
	.dc.b	$FF,$01,$E8,$F8,$09,$3A,$19,$27,$DB,$F0,$1E,$EF,$82,$95,$BA,$4F
	.dc.b	$C2,$F8,$29,$39,$7E,$2F,$C1,$7C,$6F,$82,$F8,$FE,$FC,$AE,$51,$F2
	.dc.b	$BE,$03,$E5,$FB,$F2,$7B,$47,$CE,$F8,$0F,$9F,$F0,$32,$7C,$51,$F4
	.dc.b	$BE,$0B,$E9,$FC,$14,$9F,$34,$7D,$6F,$82,$91,$E9,$7F,$33,$F9,$B7
	.dc.b	$F2,$DE,$7F,$D6,$3F,$EB,$52,$7C,$06,$BE,$07,$7F,$03,$C3,$E0,$66
	.dc.b	$3F,$26,$FF,$95,$A7,$FD,$5E,$FF,$BB,$97,$C1,$78,$0F,$0F,$C1,$F8
	.dc.b	$BE,$0E,$57,$37,$CB,$F0,$7E,$6F,$82,$F3,$FC,$1C,$AF,$04,$7A,$BE
	.dc.b	$0F,$D7,$EA,$C9,$F2,$47,$BB,$E0,$FD,$FF,$07,$27,$D1,$1F,$0B,$E0
	.dc.b	$FE,$1F,$C0,$49,$F6,$47,$C6,$F7,$FE,$3F,$C0,$49,$F8,$11,$F2,$BE
	.dc.b	$0E,$47,$C3,$7E,$6F,$C1,$FF,$06,$FC,$DD,$F0,$77,$E2,$25,$FE,$46
	.dc.b	$72,$DF,$84,$E0,$47,$5F,$09,$CF,$E1,$3A,$7C,$24,$9E,$51,$DF,$E1
	.dc.b	$24,$75,$7C,$5F,$09,$E3,$F8,$39,$3E,$17,$CD,$F0,$9E,$7F,$83,$93
	.dc.b	$E5,$7D,$5F,$09,$EB,$F8,$49,$3E,$97,$DD,$F0,$9E,$FF,$84,$93,$ED
	.dc.b	$7E,$17,$C2,$7C,$3F,$83,$F8,$9F,$09,$2B,$E1,$47,$C7,$F8,$49,$1F
	.dc.b	$15,$F9,$5F,$09,$F2,$FE,$13,$F9,$8F,$AD,$FC,$3C,$FE,$21,$CE,$8C
	.dc.b	$FC,$26,$BE,$1A,$63,$FB,$FB,$4F,$2F,$84,$E7,$F0,$92,$B5,$1D,$BE
	.dc.b	$0F,$BF,$C2,$49,$E5,$1E,$2F,$83,$F1,$FC,$24,$9E,$D1,$E6,$F8,$2F
	.dc.b	$3F,$C2,$49,$F1,$47,$AB,$E0,$E4,$79,$5F,$6F,$C1,$FB,$B6,$F7,$FC
	.dc.b	$1C,$AF,$54,$7C,$2F,$82,$91,$ED,$7E,$2F,$C1,$FF,$56,$F1,$5E,$55
	.dc.b	$25,$7C,$3F,$82,$AB,$E1,$7F,$9A,$BF,$DC,$EB,$8B,$FC,$9D,$AE,$FE
	.dc.b	$03,$FB,$A6,$4F,$84,$E7,$7C,$A9,$7C,$23,$0D,$BE,$3F,$07,$F7,$3E
	.dc.b	$12,$5F,$D7,$7E,$FF,$C2,$7F,$70,$FC,$E3,$F8,$C4,$FF,$81,$DA,$9A
	.dc.b	$3E,$17,$87,$C2,$F1,$F8,$59,$5F,$D0,$6B,$FC,$93,$F8,$1B,$E1,$BF
	.dc.b	$67,$3F,$EA,$98,$C7,$AD,$FE,$AB,$8F,$F0,$B4,$61,$C3,$5E,$13,$C9
	.dc.b	$FF,$0C,$46,$5B,$19,$79,$B5,$46,$DC,$CF,$85,$CC,$EE,$75,$0F,$A7
	.dc.b	$7A,$0F,$A7,$FF,$16,$66,$F8,$3F,$F1,$55,$6A,$73,$18,$AE,$16,$BE
	.dc.b	$0F,$F8,$D6,$F8,$39,$AE,$17,$CA,$35,$7C,$AF,$8C,$79,$63,$15,$C2
	.dc.b	$39,$1C,$2B,$93,$B6,$DF,$82,$F9,$B6,$B9,$46,$2F,$AD,$F3,$3A,$9C
	.dc.b	$CB,$5C,$5D,$4F,$1B,$EF,$5C,$A7,$84,$E5,$E3,$19,$AE,$51,$AB,$F1
	.dc.b	$91,$EA,$D7,$46,$08,$60,$91,$D0,$93,$DC,$97,$E6,$24,$F9,$1B,$7E
	.dc.b	$72,$5F,$86,$F9,$CE,$A3,$17,$D2,$F9,$12,$F8,$BE,$6B,$E2,$E6,$B8
	.dc.b	$DF,$38,$D5,$F3,$BE,$31,$EC,$8C,$57,$08,$E4,$70,$AE,$4E,$DB,$7E
	.dc.b	$0B,$E6,$DA,$E5,$18,$BE,$B7,$CC,$EA,$73,$2D,$71,$75,$3C,$6F,$BD
	.dc.b	$72,$9E,$13,$97,$8C,$66,$B9,$46,$AF,$C6,$47,$AB,$5D,$18,$21,$82
	.dc.b	$47,$42,$4F,$72,$5F,$98,$93,$E4,$6D,$F9,$C9,$7E,$1B,$E7,$3A,$8C
	.dc.b	$5F,$4B,$E4,$4B,$E2,$F9,$AF,$8B,$9A,$E3,$7C,$E3,$57,$CE,$F8,$C7
	.dc.b	$B2,$31,$5C,$23,$91,$C2,$B9,$3B,$6D,$F8,$2F,$9B,$6B,$94,$62,$FA
	.dc.b	$DF,$33,$A9,$CC,$B5,$C5,$D4,$F1,$BE,$F5,$CA,$78,$4E,$5E,$31,$9A
	.dc.b	$E5,$1A,$BF,$19,$1E,$AD,$74,$60,$86,$09,$1D,$09,$3D,$C9,$7E,$62
	.dc.b	$4F,$91,$B7,$E7,$25,$F8,$6F,$9C,$EA,$31,$7D,$2F,$91,$2F,$8B,$E6
	.dc.b	$BE,$2E,$6B,$8D,$F3,$8D,$5F,$3B,$E3,$1E,$C8,$C5,$70,$8E,$47,$0A
	.dc.b	$E4,$ED,$B7,$E0,$BE,$6D,$AE,$51,$8B,$EB,$7C,$CE,$A7,$32,$D7,$17
	.dc.b	$53,$C6,$FB,$D7,$29,$E1,$39,$78,$C6,$6B,$94,$6A,$FC,$64,$7A,$B5
	.dc.b	$D1,$82,$18,$24,$74,$24,$F7,$25,$F9,$89,$3E,$46,$DF,$9C,$97,$E1
	.dc.b	$BE,$73,$A8,$C5,$F4,$BE,$44,$BE,$2F,$9A,$F8,$B9,$AE,$37,$CE,$35
	.dc.b	$7C,$EF,$8C,$7B,$23,$15,$C2,$39,$1C,$2B,$93,$B6,$DF,$82,$F9,$B6
	.dc.b	$B9,$46,$2F,$AD,$F3,$3A,$9C,$CB,$5C,$5D,$4F,$1B,$EF,$5C,$A7,$84
	.dc.b	$E5,$E3,$19,$AE,$51,$AB,$F1,$91,$EA,$D7,$46,$08,$60,$91,$D0,$93
	.dc.b	$DC,$97,$E6,$24,$F9,$1B,$7E,$72,$5F,$86,$F9,$CE,$A3,$17,$D2,$F9
	.dc.b	$12,$F8,$BE,$6B,$E2,$E6,$B8,$DF,$38,$D5,$F3,$BE,$31,$EC,$8C,$57
	.dc.b	$08,$E4,$70,$AE,$4E,$DB,$7E,$0B,$E6,$DA,$E5,$18,$BE,$B7,$CC,$EA
	.dc.b	$73,$2D,$71,$75,$3C,$6F,$BD,$72,$9E,$13,$97,$8C,$66,$B9,$46,$AF
	.dc.b	$C6,$47,$AB,$5D,$18,$21,$82,$47,$42,$4F,$72,$5F,$98,$93,$E4,$6D
	.dc.b	$F9,$C9,$7E,$1B,$E7,$3A,$8C,$5F,$4B,$E4,$4B,$E2,$F9,$AF,$8B,$9A
	.dc.b	$E3,$7C,$E3,$57,$CE,$F8,$C7,$B2,$31,$5C,$23,$91,$C2,$B9,$3B,$6D
	.dc.b	$F8,$2F,$9B,$6B,$94,$62,$FA,$DF,$33,$A9,$CC,$B5,$C5,$D4,$F1,$BE
	.dc.b	$F5,$CA,$78,$4E,$5E,$31,$9A,$E5,$1A,$BF,$19,$1E,$AD,$74,$60,$86
	.dc.b	$09,$1D,$09,$3D,$C9,$7E,$62,$4F,$91,$B7,$E7,$25,$F8,$6F,$9C,$EA
	.dc.b	$31,$7D,$2F,$91,$2F,$8B,$E6,$BE,$2E,$6B,$8D,$F3,$8D,$5F,$3B,$E3
	.dc.b	$1E,$C8,$C5,$70,$8E,$47,$0A,$E4,$ED,$B7,$E0,$BE,$6D,$AE,$51,$8B
	.dc.b	$EB,$7C,$CE,$A7,$32,$D7,$17,$53,$C6,$FB,$D7,$29,$E1,$39,$78,$C6
	.dc.b	$6B,$94,$6A,$FC,$64,$7A,$B5,$D1,$82,$18,$24,$74,$24,$F7,$25,$F9
	.dc.b	$89,$3E,$46,$DF,$9C,$97,$E1,$BE,$73,$A8,$C5,$F4,$BE,$44,$BE,$2F
	.dc.b	$9A,$F8,$B9,$AE,$37,$CE,$35,$7C,$EF,$8C,$7B,$23,$15,$C2,$39,$1C
	.dc.b	$2B,$93,$B6,$DF,$82,$F9,$B6,$B9,$46,$2F,$AD,$F3,$3A,$9C,$CB,$5C
	.dc.b	$5D,$4F,$1B,$EF,$5C,$A7,$84,$E5,$E3,$19,$AE,$51,$AB,$F1,$91,$EA
	.dc.b	$D7,$46,$08,$60,$91,$D0,$93,$DC,$97,$E6,$24,$F9,$1B,$7E,$72,$5F
	.dc.b	$86,$F9,$CE,$A3,$17,$D2,$F9,$12,$F8,$BE,$6B,$E2,$E6,$B8,$DF,$38
	.dc.b	$D5,$F3,$BE,$31,$EC,$8C,$57,$08,$E4,$70,$AE,$4E,$DB,$7E,$0B,$E6
	.dc.b	$DA,$E5,$18,$BE,$B7,$CC,$EA,$73,$2D,$71,$75,$3C,$6F,$BD,$72,$9E
	.dc.b	$13,$97,$8C,$66,$B9,$46,$AF,$C6,$47,$AB,$5D,$18,$21,$82,$47,$42
	.dc.b	$4F,$72,$5F,$98,$93,$E4,$6D,$F9,$C9,$7E,$1B,$E7,$3A,$8C,$5F,$4B
	.dc.b	$E4,$4B,$E2,$F9,$AF,$8B,$9A,$E3,$7C,$E3,$57,$CE,$F8,$C7,$B2,$31
	.dc.b	$5C,$23,$91,$C2,$B9,$3B,$6D,$F8,$2F,$9B,$6B,$94,$62,$FA,$DF,$33
	.dc.b	$A9,$CC,$B5,$C5,$D4,$F1,$BE,$F5,$CA,$78,$4E,$5E,$31,$9A,$E5,$1A
	.dc.b	$BF,$19,$1E,$AD,$74,$60,$86,$09,$1D,$09,$3D,$C9,$7E,$62,$4F,$91
	.dc.b	$B7,$E7,$25,$F8,$6F,$9C,$EA,$31,$7D,$2F,$91,$2F,$8B,$E6,$BE,$2E
	.dc.b	$6B,$8D,$F3,$8D,$5F,$3B,$E3,$1E,$C8,$C5,$70,$8E,$47,$0A,$E4,$ED
	.dc.b	$B7,$E0,$BE,$6D,$AE,$51,$8B,$EB,$7C,$CE,$A7,$32,$D7,$17,$53,$C6
	.dc.b	$FB,$D7,$29,$E1,$39,$78,$C6,$6B,$94,$6A,$FC,$64,$7A,$B5,$D1,$82
	.dc.b	$18,$24,$74,$24,$F7,$25,$F9,$89,$3E,$46,$DF,$9C,$97,$E1,$BE,$73
	.dc.b	$A8,$C5,$F4,$BE,$44,$BE,$2F,$9A,$F8,$B9,$AE,$37,$CE,$35,$7C,$EF
	.dc.b	$8C,$7B,$23,$15,$C2,$39,$1C,$2B,$93,$B6,$DF,$82,$F9,$B6,$B9,$46
	.dc.b	$2F,$AD,$F3,$3A,$9C,$CB,$5C,$5D,$4F,$1B,$EF,$5C,$A7,$84,$E5,$E3
	.dc.b	$19,$AE,$51,$AB,$F1,$91,$EA,$D7,$46,$08,$60,$91,$D0,$93,$DC,$97
	.dc.b	$E6,$24,$F9,$1B,$7E,$72,$5F,$86,$F9,$CE,$A3,$17,$D2,$F9,$12,$F8
	.dc.b	$BE,$6B,$E2,$E6,$B8,$DF,$38,$D5,$F3,$BE,$31,$EC,$8C,$57,$08,$E4
	.dc.b	$70,$AE,$4E,$DB,$7E,$0B,$E6,$DA,$E5,$18,$BE,$B7,$CC,$EA,$73,$2D
	.dc.b	$71,$75,$3C,$6F,$BD,$72,$9E,$13,$97,$8C,$66,$B9,$46,$AF,$C6,$47
	.dc.b	$AB,$5D,$18,$21,$82,$47,$42,$4F,$72,$5F,$98,$93,$E4,$6D,$F9,$C9
	.dc.b	$7E,$1B,$E7,$3A,$8C,$5F,$4B,$E4,$4B,$E2,$F9,$AF,$8B,$9A,$E3,$7C
	.dc.b	$E3,$57,$CE,$F8,$C7,$B2,$31,$5C,$23,$91,$C2,$B9,$3B,$6D,$F8,$2F
	.dc.b	$9B,$6B,$94,$62,$FA,$DF,$33,$A9,$CC,$B5,$C5,$D4,$F1,$BE,$F5,$CA
	.dc.b	$78,$4E,$5E,$31,$9A,$E5,$1A,$BF,$19,$1E,$AD,$74,$60,$86,$09,$1D
	.dc.b	$09,$3D,$C9,$7E,$62,$4F,$91,$B7,$E7,$25,$F8,$6F,$9C,$EA,$31,$7D
	.dc.b	$2F,$91,$2F,$8B,$E6,$BE,$2E,$6B,$8D,$F3,$8D,$5F,$3B,$E3,$1E,$C8
	.dc.b	$C5,$70,$8E,$47,$0A,$E4,$ED,$B7,$E0,$BE,$6D,$AE,$51,$8B,$EB,$7C
	.dc.b	$CE,$A7,$32,$D2,$16,$95,$2F,$10,$BE,$15,$BA,$D4,$F1,$9C,$BC,$0C
	.dc.b	$5F,$67,$07,$37,$0E,$E3,$8B,$63,$2F,$8A,$32,$D8,$CB,$DA,$BA,$BD
	.dc.b	$5B,$3C,$23,$47,$89,$E7,$F5,$02,$00

;--------------------------------------------------------------------------------
;	FADD.X FPm,FPn
;--------------------------------------------------------------------------------
	.cpu	68030
regs		reg	d0-d7/a0-a5
cregs		reg	fpcr/fpsr/fpiar
fregs		reg	fp0-fp7
	.offsym	0,_a6
_size:
_regs:	.ds.b	.sizeof.(regs)
_fregs:	.ds.b	.sizeof.(fregs)
_cregs:	.ds.b	.sizeof.(cregs)
_a6:	.ds.l	1
_pc:	.ds.l	1
	.text
	.even
fadd_test::
	link.w	a6,#_size
	movem.l	regs,(_regs,a6)
	fmovem.l	cregs,(_cregs,a6)
	fmovem.x	fregs,(_fregs,a6)
;
	move.l	#63,-(sp)
	peamsg	'FADD.X FPM,FPN'
	jbsr	mnemonic_start
	addq.l	#8,sp
	beq	99f
	putmsg	'test: FADD.X FPm,FPn',13,10
;------------------------------------------------
;	d1	fpcr=(XRN..XRP)<<4
;	d3	0=failed,1=successful
;	d5	expected status
;	d7	actual status
;	a2	source handle,...
;	a3	destination handle,...
;	a4	expected result handle,expected status,...
;	fp2	source
;	fp3	destination
;	fp5	expected result
;	fp7	actual result
;------------------------------------------------
	lea.l	push_decompressed,a0
;decompress data
	move.l	a0,-(sp)
	pea.l	fadd_data_compressed
	jbsr	decompress
	addq.l	#8,sp
;relocate decompressed handle
	move.l	#indirect_decompressed,d0
@@:
	add.l	d0,(a0)+		;destination handle
	tst.l	(a0)
	bpl	@b
	addq.l	#4,a0			;-1
	movea.l	a0,a4			;expected result handle,expected status,...
@@:
	add.l	d0,(a0)+		;expected result handle
	addq.l	#4,a0			;expected status
	tst.l	(a0)
	bpl	@b
;	addq.l	#4,a0			;-1
;
	lea.l	push_decompressed,a3	;destination handle,...
33:
	move.l	#0,d1		;fpcr=(XRN..XRP)<<4
11:
;FPn,FPn
	fmove.l	#0,fpcr
	fmove.x	([a3]),fp3		;destination
	fmove.x	fp3,fp7			;actual result=destination
	fmove.l	d1,fpcr			;fpcr
	fmove.l	#0,fpsr
					;source
	fadd.x	fp7,fp7		;EXECUTE
					;actual result
	fmove.l	fpsr,d7			;actual status
	fmove.l	#0,fpcr
;
	fmove.x	([a4]),fp5		;expected result
	move.l	(4,a4),d5		;expected status
;
	move.l	#-1,-(sp)	;fpcr(rp<<6,-1=strict)
	move.l	d5,-(sp)		;expected status
	fmove.x	fp5,-(sp)		;expected result
	move.l	d7,-(sp)		;actual status
	fmove.x	fp7,-(sp)		;actual result
	jbsr	test_extended
	lea.l	36(sp),sp
	move.l	d0,d3			;0=failed,1=successful
;
	move.l	d3,-(sp)		;0=failed,1=successful
	jbsr	statistics_update
	addq.l	#4,sp
	beq	@f			;not output
;output
	putmsg	'FADD.X FPn='
	puthex24	([a3]),(4,[a3]),(8,[a3])	;source
	putmsg	',FPn='
	puthex24	([a3]),(4,[a3]),(8,[a3])	;destination
	putmsg	' @'
	move.l	d1,-(sp)
	jbsr	printfpcrrprm
	addq.l	#4,sp
	putcrlf
	move.l	d3,-(sp)		;0=failed,1=successful
	move.l	d5,-(sp)		;expected status
	fmove.x	fp5,-(sp)		;expected result
	move.l	d7,-(sp)		;actual status
	fmove.x	fp7,-(sp)		;actual result
	jbsr	output_extended
	lea.l	36(sp),sp
@@:
;
	addq.l	#8,a4			;expected result handle,expected status,...
;
	add.w	#1<<4,d1		;rprm++
	cmp.w	#48,d1		;fpcr=(XRN..XRP)<<4
	bls	11b
;
	addq.l	#4,a3			;destination handle,...
	tst.l	(a3)			;destination handle,...
	bpl	33b
;
	lea.l	push_decompressed,a3	;destination handle,...
33:
	lea.l	push_decompressed,a2	;source handle,...
22:
	move.l	#0,d1		;fpcr=(XRN..XRP)<<4
11:
;FPm,FPn
	fmove.l	#0,fpcr
	fmove.x	([a2]),fp2		;source
	fmove.x	([a3]),fp3		;destination
	fmove.x	fp3,fp7			;actual result=destination
	fmove.l	d1,fpcr			;fpcr
	fmove.l	#0,fpsr
					;source
	fadd.x	fp2,fp7		;EXECUTE
					;actual result
	fmove.l	fpsr,d7			;actual status
	fmove.l	#0,fpcr
;
	fmove.x	([a4]),fp5		;expected result
	move.l	(4,a4),d5		;expected status
;
	move.l	#-1,-(sp)	;fpcr(rp<<6,-1=strict)
	move.l	d5,-(sp)		;expected status
	fmove.x	fp5,-(sp)		;expected result
	move.l	d7,-(sp)		;actual status
	fmove.x	fp7,-(sp)		;actual result
	jbsr	test_extended
	lea.l	36(sp),sp
	move.l	d0,d3			;0=failed,1=successful
;
	move.l	d3,-(sp)		;0=failed,1=successful
	jbsr	statistics_update
	addq.l	#4,sp
	beq	@f			;not output
;output
	putmsg	'FADD.X FPm='
	puthex24	([a2]),(4,[a2]),(8,[a2])	;source
	putmsg	',FPn='
	puthex24	([a3]),(4,[a3]),(8,[a3])	;destination
	putmsg	' @'
	move.l	d1,-(sp)
	jbsr	printfpcrrprm
	addq.l	#4,sp
	putcrlf
	move.l	d3,-(sp)		;0=failed,1=successful
	move.l	d5,-(sp)		;expected status
	fmove.x	fp5,-(sp)		;expected result
	move.l	d7,-(sp)		;actual status
	fmove.x	fp7,-(sp)		;actual result
	jbsr	output_extended
	lea.l	36(sp),sp
@@:
;
	addq.l	#8,a4			;expected result handle,expected status,...
;
	add.w	#1<<4,d1		;rprm++
	cmp.w	#48,d1		;fpcr=(XRN..XRP)<<4
	bls	11b
;
	addq.l	#4,a2			;source handle,...
	tst.l	(a2)			;source handle,...
	bpl	22b
;
	addq.l	#4,a3			;destination handle,...
	tst.l	(a3)			;destination handle,...
	bpl	33b
;
	jbsr	mnemonic_end
99:
	fmovem.x	(_fregs,a6),fregs
	fmovem.l	(_cregs,a6),cregs
	movem.l	(_regs,a6),regs
	unlk	a6
	rts
	.cpu	68000

	.align	4
fadd_data_compressed::
	.dc.b	$00,$0E,$B5,$7C,$10,$86,$36,$BD,$58,$F1,$B3,$E9,$6B,$DE,$E1,$8B
	.dc.b	$E0,$AF,$C7,$73,$D1,$95,$CB,$1F,$75,$9F,$C1,$B5,$F9,$56,$FF,$36
	.dc.b	$E1,$FA,$77,$8F,$89,$F4,$77,$F6,$EF,$37,$2B,$F0,$3B,$F9,$37,$A7
	.dc.b	$A1,$CB,$7F,$44,$F6,$75,$3F,$63,$F7,$76,$3E,$F3,$E0,$77,$7C,$01
	.dc.b	$D9,$8E,$2F,$89,$DE,$F9,$3B,$F9,$BF,$C7,$F0,$BE,$07,$83,$F2,$97
	.dc.b	$EA,$99,$FD,$DB,$E6,$77,$3B,$3E,$77,$8A,$F3,$3E,$87,$A8,$FC,$52
	.dc.b	$FE,$41,$8E,$AC,$FE,$97,$AD,$3E,$B7,$B1,$F5,$8F,$9B,$BF,$DA,$7E
	.dc.b	$0B,$7C,$AD,$FC,$EF,$0F,$BF,$E3,$E3,$F2,$F1,$F9,$F4,$74,$FD,$77
	.dc.b	$AF,$ED,$1D,$BF,$73,$FC,$27,$BD,$E7,$D3,$F0,$EB,$DF,$CF,$D4,$6B
	.dc.b	$CB,$DF,$E2,$7C,$3F,$1D,$E3,$F9,$47,$2F,$D2,$39,$FE,$A5,$D3,$F7
	.dc.b	$AF,$C8,$64,$F4,$BF,$25,$F1,$3F,$33,$62,$F5,$F3,$E5,$EB,$D2,$DF
	.dc.b	$6F,$0E,$DF,$CD,$79,$6F,$5A,$B8,$F4,$B3,$DF,$AF,$1F,$7F,$92,$F0
	.dc.b	$FC,$BF,$8F,$7F,$2E,$FE,$7E,$A7,$4F,$63,$AF,$83,$DB,$C1,$EF,$E0
	.dc.b	$F8,$3F,$7E,$FD,$57,$C6,$BD,$5B,$E6,$E3,$B3,$3F,$29,$AF,$9F,$DF
	.dc.b	$AB,$C3,$C7,$E3,$E3,$F2,$FC,$23,$9F,$E2,$5D,$3C,$9E,$BE,$4F,$6F
	.dc.b	$27,$BF,$E6,$FE,$0F,$63,$C3,$E5,$F8,$BC,$BF,$1F,$EB,$DE,$4F,$73
	.dc.b	$F7,$DF,$28,$F9,$6B,$F4,$18,$F8,$5C,$F9,$BA,$F6,$37,$EC,$70,$FD
	.dc.b	$BF,$8F,$BD,$FC,$1F,$E6,$3E,$77,$A9,$FC,$29,$FD,$4B,$FC,$33,$E6
	.dc.b	$9F,$82,$FF,$0D,$5E,$DD,$F3,$31,$E6,$67,$CA,$D7,$95,$BF,$2B,$F8
	.dc.b	$A7,$CE,$7C,$BC,$FF,$8B,$BF,$93,$4F,$ED,$5C,$F8,$5C,$3C,$0E,$3E
	.dc.b	$07,$2F,$03,$9F,$77,$4E,$EE,$BF,$13,$DB,$DB,$FE,$45,$E3,$BE,$66
	.dc.b	$F7,$63,$BB,$3F,$7F,$AF,$C2,$77,$DD,$C3,$BB,$8F,$B5,$CB,$DA,$E7
	.dc.b	$FB,$37,$F2,$DF,$AC,$72,$DF,$1B,$F9,$87,$D6,$FF,$13,$E8,$FE,$66
	.dc.b	$B3,$FE,$69,$F5,$DF,$32,$4B,$D8,$FE,$71,$E4,$7E,$CE,$7F,$9D,$AE
	.dc.b	$8F,$E7,$9F,$5F,$FA,$B1,$C3,$8F,$1F,$FA,$07,$D8,$7F,$15,$C3,$FA
	.dc.b	$1F,$D8,$BC,$FF,$E8,$9E,$9B,$A2,$F3,$63,$CC,$FE,$91,$F6,$4F,$88
	.dc.b	$FE,$95,$F6,$7F,$5F,$32,$F5,$33,$C3,$AF,$0F,$7D,$1C,$3C,$3F,$EA
	.dc.b	$5F,$69,$FB,$FC,$7F,$AA,$AF,$03,$FA,$B7,$C5,$FE,$42,$4F,$C5,$FB
	.dc.b	$6F,$CE,$47,$F5,$C7,$D0,$DE,$9F,$EB,$D1,$D6,$0F,$57,$87,$46,$BE
	.dc.b	$25,$FC,$03,$0B,$3F,$D9,$BE,$53,$FB,$3B,$4A,$7F,$6C,$6B,$4A,$FF
	.dc.b	$70,$F9,$EF,$4C,$7F,$46,$F8,$1C,$67,$D2,$AC,$E4,$57,$07,$FA,$A9
	.dc.b	$89,$FE,$F3,$F2,$9F,$A7,$9D,$4F,$85,$B1,$99,$ED,$3C,$4E,$D3,$C1
	.dc.b	$57,$89,$FF,$02,$5D,$7F,$BE,$BE,$EB,$FC,$04,$78,$A3,$07,$C2,$F8
	.dc.b	$CC,$C7,$F8,$3D,$AF,$F8,$49,$AF,$F3,$98,$F9,$7F,$9B,$3E,$0B,$6D
	.dc.b	$8C,$C6,$0C,$F6,$C8,$E2,$DA,$FA,$F5,$FB,$29,$EB,$E3,$F4,$23,$E0
	.dc.b	$F8,$2E,$E4,$70,$79,$3E,$0B,$C6,$47,$07,$A3,$D4,$F3,$91,$C1,$F8
	.dc.b	$6F,$9B,$FE,$46,$DF,$73,$E1,$C6,$32,$FC,$6B,$5F,$F1,$AB,$FA,$F9
	.dc.b	$E5,$DF,$C0,$8F,$5F,$8A,$E6,$47,$07,$CC,$F5,$FE,$59,$1C,$1F,$9A
	.dc.b	$F9,$1F,$F1,$03,$77,$FA,$F1,$8C,$7F,$42,$D7,$FC,$93,$BB,$D8,$FA
	.dc.b	$E4,$BF,$B6,$FD,$AF,$0A,$47,$DE,$6D,$74,$EF,$EE,$49,$F2,$3F,$82
	.dc.b	$F2,$24,$7E,$19,$B3,$F7,$BC,$3F,$C3,$92,$7F,$16,$FE,$43,$E6,$5F
	.dc.b	$0D,$49,$3F,$BA,$8C,$BF,$92,$F1,$7F,$22,$47,$07,$E5,$A7,$F2,$A4
	.dc.b	$70,$7F,$04,$3F,$CA,$05,$EF,$FD,$0B,$8C,$3F,$85,$B5,$F9,$7F,$0D
	.dc.b	$FA,$02,$47,$E9,$1F,$D1,$F8,$D2,$3F,$4E,$D9,$FE,$15,$AF,$EB,$BB
	.dc.b	$BF,$99,$57,$F1,$66,$CE,$A3,$F5,$FB,$7E,$B8,$8E,$0F,$D9,$FB,$7F
	.dc.b	$B2,$23,$83,$F9,$0F,$CC,$7F,$A0,$DB,$E3,$7E,$FA,$31,$7F,$CA,$1A
	.dc.b	$FF,$A4,$3F,$37,$FF,$21,$57,$F2,$A6,$C7,$F2,$E6,$BB,$E5,$FE,$13
	.dc.b	$F5,$0F,$36,$92,$FF,$99,$C6,$5B,$E9,$70,$23,$83,$F9,$CF,$D4,$FF
	.dc.b	$A9,$1B,$87,$F1,$68,$C3,$FD,$01,$AF,$FA,$A1,$FF,$0F,$C7,$2F,$87
	.dc.b	$E0,$47,$AF,$81,$CC,$8E,$0F,$0F,$C2,$77,$23,$83,$CB,$F0,$9E,$32
	.dc.b	$38,$3F,$A8,$FC,$0F,$F2,$5C,$7F,$AA,$B5,$FF,$5E,$B5,$FF,$60,$35
	.dc.b	$FF,$61,$BF,$BB,$73,$DF,$FC,$DA,$31,$1A,$8C,$78,$F2,$BC,$2D,$9F
	.dc.b	$0F,$6F,$12,$4F,$99,$E9,$D9,$23,$D2,$D9,$FE,$DB,$F3,$5F,$E3,$39
	.dc.b	$FE,$DA,$D8,$CE,$3F,$D2,$A3,$11,$FE,$DA,$7F,$CB,$D3,$C3,$E0,$B4
	.dc.b	$47,$9F,$CD,$71,$23,$83,$BF,$BD,$D4,$8E,$0F,$1F,$C1,$F8,$48,$E0
	.dc.b	$FF,$01,$FC,$D7,$E1,$B1,$FF,$5C,$9F,$D8,$1C,$3F,$EB,$93,$1E,$84
	.dc.b	$9E,$2D,$8F,$F0,$7C,$FF,$43,$7F,$D7,$93,$DB,$8B,$A1,$D7,$CA,$91
	.dc.b	$E2,$6C,$F2,$E2,$F1,$12,$7C,$EF,$97,$E1,$E4,$7A,$9B,$3E,$2F,$92
	.dc.b	$F5,$12,$7D,$EF,$F7,$CF,$9A,$FE,$DD,$3F,$E2,$66,$A3,$FD,$93,$3F
	.dc.b	$E2,$88,$CC,$7F,$B2,$E3,$1E,$8F,$32,$3B,$E3,$EC,$47,$07,$3F,$23
	.dc.b	$C4,$47,$07,$F8,$DB,$E6,$BF,$0F,$7F,$ED,$0E,$5F,$F0,$9C,$62,$3C
	.dc.b	$CD,$5E,$07,$F8,$CA,$BF,$8B,$7A,$BF,$E1,$A8,$C4,$FF,$17,$8C,$5C
	.dc.b	$AF,$E3,$8D,$9F,$E3,$9C,$DC,$49,$3F,$C8,$5F,$1F,$E9,$92,$3D,$0D
	.dc.b	$9F,$E4,$9F,$01,$E1,$24,$FF,$29,$7D,$7E,$04,$8F,$80,$D9,$FE,$59
	.dc.b	$E2,$FA,$49,$3F,$CC,$5F,$67,$C8,$48,$FE,$68,$D9,$FE,$69,$EE,$FC
	.dc.b	$12,$4F,$F3,$97,$FC,$7B,$F3,$5F,$E1,$79,$FF,$33,$B5,$FC,$11,$F8
	.dc.b	$37,$0F,$FB,$EB,$9F,$E6,$7B,$B2,$FF,$A2,$36,$7F,$A2,$7A,$1F,$44
	.dc.b	$93,$FD,$25,$FA,$9F,$37,$23,$FA,$63,$67,$FA,$67,$81,$F6,$49,$3F
	.dc.b	$D4,$5F,$4F,$C8,$48,$FF,$3E,$E4,$8C,$57,$B3,$E4,$3D,$E4,$AD,$BF
	.dc.b	$85,$F0,$A4,$7F,$5D,$6C,$FF,$5D,$F0,$BF,$14,$49,$FE,$C4,$FE,$37
	.dc.b	$F4,$39,$1F,$D9,$5B,$3F,$33,$C8,$F9,$E4,$9F,$3B,$F4,$BC,$0F,$F7
	.dc.b	$54,$7D,$09,$66,$9C,$C7,$D4,$8C,$4F,$F1,$96,$C7,$F9,$FB,$A9,$CF
	.dc.b	$C2,$3F,$79,$18,$AF,$E4,$0D,$8E,$13,$B5,$F2,$EF,$97,$C3,$6F,$F9
	.dc.b	$43,$67,$FD,$1A,$D8,$F5,$2F,$92,$78,$FF,$EB,$28,$C3,$63,$35,$FE
	.dc.b	$B8,$9C,$B7,$E0,$5E,$8F,$16,$C6,$63,$07,$5F,$33,$F3,$38,$F8,$9B
	.dc.b	$3F,$9C,$BF,$F7,$FB,$EA,$8C,$BF,$1B,$E2,$6B,$FC,$31,$BA,$9A,$6A
	.dc.b	$9E,$F5,$25,$7C,$67,$5E,$C7,$FB,$42,$7F,$DE,$13,$F0,$E3,$07,$FB
	.dc.b	$52,$DF,$25,$49,$3F,$E3,$17,$5F,$25,$23,$FC,$68,$D7,$FC,$69,$EB
	.dc.b	$73,$23,$FE,$39,$7C,$5E,$54,$8F,$A8,$D7,$E9,$79,$5E,$72,$3F,$65
	.dc.b	$F3,$78,$B2,$3F,$C8,$8D,$7F,$DE,$3F,$9D,$7F,$BB,$93,$48,$FF,$BC
	.dc.b	$9F,$C0,$BC,$39,$30,$47,$FD,$F3,$0F,$86,$E0,$48,$CB,$F3,$7D,$B9
	.dc.b	$1F,$8F,$6B,$E4,$E9,$F0,$11,$EA,$F8,$FE,$76,$47,$81,$AF,$B3,$E7
	.dc.b	$7D,$04,$7C,$AF,$EE,$5F,$E5,$3F,$32,$F1,$B1,$1F,$DB,$63,$0E,$1A
	.dc.b	$F1,$6B,$FC,$0F,$2F,$F8,$00,$FF,$75,$6B,$F3,$3E,$2B,$E5,$1D,$94
	.dc.b	$BC,$1C,$36,$3C,$11,$83,$E9,$7E,$09,$23,$E7,$B5,$FB,$7E,$E7,$D7
	.dc.b	$23,$F5,$1F,$B5,$EC,$48,$FA,$ED,$7E,$27,$CD,$7A,$09,$8F,$9D,$F5
	.dc.b	$FC,$D7,$F3,$99,$F8,$ED,$7E,$63,$5F,$E4,$CD,$7F,$AB,$AF,$27,$E1
	.dc.b	$36,$39,$C6,$0F,$8D,$F3,$4F,$06,$92,$7E,$BC,$65,$BE,$2F,$02,$38
	.dc.b	$37,$F1,$BD,$08,$E0,$E7,$E0,$F8,$08,$E0,$EF,$F1,$5E,$42,$38,$3C
	.dc.b	$7C,$FE,$82,$38,$3C,$FF,$0B,$EC,$23,$83,$F2,$3E,$43,$D3,$FC,$33
	.dc.b	$63,$31,$83,$3E,$94,$8E,$2D,$7F,$3D,$3F,$C8,$DE,$BE,$3F,$42,$3E
	.dc.b	$0F,$13,$B9,$1C,$1E,$4F,$13,$C6,$47,$07,$EA,$3E,$C5,$DB,$F9,$D6
	.dc.b	$C6,$63,$06,$79,$A4,$71,$6B,$FB,$59,$FC,$23,$79,$3A,$11,$F0,$7C
	.dc.b	$C7,$72,$38,$3F,$77,$F6,$5F,$D9,$5B,$C9,$FB,$18,$C4,$7F,$03,$6B
	.dc.b	$FD,$F5,$7C,$EE,$C9,$1D,$1E,$5B,$48,$EE,$D7,$A7,$1F,$72,$3E,$47
	.dc.b	$C5,$E5,$48,$F3,$B5,$F2,$71,$F9,$C8,$FB,$1F,$E1,$5F,$67,$FD,$BA
	.dc.b	$3F,$8E,$35,$FF,$0B,$35,$FF,$23,$B5,$FF,$45,$2E,$3E,$64,$8F,$E4
	.dc.b	$71,$96,$F8,$5E,$02,$38,$37,$EB,$F9,$08,$E0,$FE,$5B,$F8,$03,$D4
	.dc.b	$FE,$44,$D8,$CC,$60,$FE,$4D,$F3,$BF,$E2,$E8,$FE,$6A,$D7,$F9,$D3
	.dc.b	$5F,$F1,$A3,$5F,$F3,$2F,$E0,$0F,$33,$F9,$83,$63,$F9,$FB,$5F,$E8
	.dc.b	$8D,$7F,$9E,$7B,$7F,$D8,$48,$FF,$5E,$7C,$B7,$23,$CA,$D7,$FB,$6F
	.dc.b	$D1,$FF,$69,$23,$FD,$99,$FE,$D9,$F2,$B2,$3F,$B4,$B5,$FE,$AD,$F3
	.dc.b	$FE,$37,$FC,$FE,$D7,$FD,$84,$D4,$DF,$33,$F9,$84,$7F,$53,$F7,$7F
	.dc.b	$BD,$11,$FE,$EF,$D5,$FD,$F4,$8E,$0F,$EC,$7F,$3F,$C9,$34,$93,$FE
	.dc.b	$05,$EB,$93,$FE,$02,$6B,$FD,$BF,$F0,$0F,$DD,$BF,$E6,$96,$BF,$E9
	.dc.b	$A6,$BF,$ED,$D6,$AF,$9F,$6D,$FD,$A5,$B1,$FD,$E1,$AF,$F7,$D6,$BD
	.dc.b	$F4,$FE,$E1,$46,$92,$3F,$C3,$3F,$49,$FE,$1F,$23,$83,$FC,$0B,$F4
	.dc.b	$1F,$C1,$1B,$DB,$FD,$EE,$31,$1F,$E0,$D6,$BF,$E0,$5E,$1F,$F1,$51
	.dc.b	$1F,$F1,$3B,$C7,$E6,$24,$76,$6B,$CF,$C8,$EC,$47,$C6,$F8,$7B,$24
	.dc.b	$79,$9A,$FF,$8F,$BE,$3F,$FC,$74,$47,$FC,$6E,$FF,$8F,$7E,$0A,$47
	.dc.b	$F8,$E9,$AF,$F9,$1B,$E3,$FF,$C8,$44,$7F,$C7,$EF,$F9,$17,$E9,$24
	.dc.b	$7F,$90,$9A,$FF,$8B,$FE,$17,$FB,$B3,$7D,$1F,$F1,$14,$62,$3F,$D5
	.dc.b	$8D,$7F,$DB,$AF,$F3,$46,$F8,$FC,$08,$FF,$8E,$3E,$81,$F1,$3F,$E2
	.dc.b	$D6,$C6,$63,$06,$7D,$09,$1C,$5A,$FF,$91,$67,$F9,$27,$D4,$A1,$F1
	.dc.b	$EF,$3B,$FD,$E3,$5F,$DD,$67,$31,$8B,$FF,$7A,$D6,$E7,$95,$74,$76
	.dc.b	$63,$C9,$61,$DE,$B8,$4A,$BB,$46,$1B,$87,$8A,$F8,$D7,$43,$83,$8A
	.dc.b	$E1,$87,$92,$BC,$D5,$DD,$E5,$85,$8F,$23,$F3,$6F,$AC,$7F,$B8,$70
	.dc.b	$F5,$36,$32,$FC,$4F,$21,$F5,$5F,$55,$B1,$F3,$5A,$F3,$6B,$FE,$75
	.dc.b	$F9,$BF,$F4,$A1,$1F,$F3,$43,$5F,$F3,$BB,$5E,$06,$88,$68,$A6,$8C
	.dc.b	$10,$C1,$0F,$F7,$F3,$D5,$A7,$22,$9C,$4A,$71,$3F,$10,$1E,$4E,$1A
	.dc.b	$FE,$35,$AE,$CE,$C6,$CA,$6C,$A6,$CF,$11,$B2,$9B,$29,$B3,$CC,$64
	.dc.b	$A6,$4F,$59,$0E,$86,$9F,$39,$1F,$1B,$5E,$ED,$78,$1A,$21,$A2,$9A
	.dc.b	$30,$43,$04,$3C,$87,$A4,$A7,$A4,$F3,$98,$7C,$4D,$3B,$94,$EC,$53
	.dc.b	$B1,$C8,$87,$72,$9D,$CC,$10,$C1,$1F,$0B,$C1,$AE,$DA,$EC,$F2,$9B
	.dc.b	$29,$B2,$9B,$3D,$26,$CA,$6C,$A6,$CF,$69,$92,$99,$3E,$11,$0E,$86
	.dc.b	$9F,$39,$1F,$1B,$5E,$ED,$78,$1A,$21,$A2,$9A,$30,$43,$04,$3C,$87
	.dc.b	$A4,$A7,$A4,$F3,$98,$7C,$4D,$3B,$94,$EC,$53,$B1,$C8,$87,$72,$9D
	.dc.b	$CC,$10,$C1,$1F,$0B,$C1,$AE,$DA,$EC,$F2,$9B,$29,$B2,$9B,$3D,$26
	.dc.b	$CA,$6C,$A6,$CF,$69,$92,$99,$3E,$11,$0E,$86,$9F,$39,$1F,$1B,$5E
	.dc.b	$ED,$78,$1A,$21,$A2,$9A,$30,$43,$04,$3C,$87,$A4,$A7,$A4,$F3,$98
	.dc.b	$7C,$4D,$3B,$94,$EC,$53,$B1,$C8,$87,$72,$9D,$CC,$10,$C1,$1F,$0B
	.dc.b	$C1,$AE,$DA,$EC,$F2,$9B,$29,$B2,$9B,$3D,$26,$CA,$6C,$A6,$CF,$69
	.dc.b	$92,$99,$3E,$11,$0E,$86,$9F,$39,$1F,$1B,$5E,$ED,$78,$1A,$21,$A2
	.dc.b	$9A,$30,$43,$04,$3C,$87,$A4,$FC,$B5,$A7,$FC,$1F,$D2,$F5,$7F,$C2
	.dc.b	$98,$7F,$8E,$E3,$17,$FE,$34,$AD,$D6,$A3,$FE,$00,$B5,$FF,$0B,$57
	.dc.b	$F8,$6C,$8F,$AF,$5F,$F1,$BA,$F8,$27,$17,$F8,$AE,$3F,$BD,$4E,$A7
	.dc.b	$31,$FE,$37,$3F,$BE,$7D,$76,$6B,$FC,$A0,$F4,$8F,$F0,$23,$94,$E7
	.dc.b	$4D,$EF,$FE,$10,$9F,$15,$61,$FF,$0C,$D7,$13,$83,$CA,$78,$5F,$92
	.dc.b	$72,$6E,$3F,$C5,$B8,$78,$4E,$25,$78,$DF,$F9,$9F,$7F,$F3,$E5,$FB
	.dc.b	$9F,$F3,$E6,$3E,$3C,$7F,$CD,$93,$9A,$E3,$1F,$E6,$26,$D7,$08,$F3
	.dc.b	$1F,$E2,$C9,$E0,$D9,$F0,$1C,$0F,$F2,$1E,$3F,$1E,$BC,$71,$A9,$E0
	.dc.b	$77,$3E,$2E,$1C,$5C,$CF,$07,$47,$F8,$CE,$BA,$56,$E3,$D6,$F5,$75
	.dc.b	$59,$7E,$99,$C8,$B5,$ED,$AE,$71,$E0,$AE,$0F,$E0,$2B,$95,$FC,$08
	.dc.b	$EE,$73,$34,$71,$21,$AB,$ED,$3C,$AB,$CD,$3E,$27,$E3,$C6,$63,$CD
	.dc.b	$5D,$DF,$CA,$C6,$6B,$E2,$1D,$23,$B9,$CC,$A7,$90,$F1,$57,$13,$BB
	.dc.b	$C2,$BB,$1D,$DE,$07,$33,$89,$D3,$0F,$C5,$39,$24,$71,$8C,$BD,$27
	.dc.b	$DE,$E6,$7F,$16,$74,$21,$CC,$F0,$BC,$5F,$87,$1F,$21,$CB,$D0,$8F
	.dc.b	$D3,$9F,$0C,$62,$BE,$3B,$F0,$27,$BD,$75,$9F,$95,$18,$9E,$06,$DF
	.dc.b	$6B,$87,$F2,$E7,$C9,$70,$FE,$40,$C1,$E8,$3C,$87,$C0,$39,$99,$8F
	.dc.b	$0B,$87,$CE,$F6,$3C,$86,$E3,$C9,$3E,$97,$F2,$F5,$F0,$27,$47,$D7
	.dc.b	$3C,$13,$B7,$18,$7E,$81,$EB,$7E,$29,$D4,$7D,$43,$E7,$9D,$0C,$57
	.dc.b	$79,$F3,$C6,$0E,$F7,$F1,$1F,$B9,$3B,$8F,$C9,$47,$B4,$F8,$0F,$A9
	.dc.b	$E2,$77,$38,$98,$9E,$EE,$2F,$D4,$F2,$3A,$5F,$A9,$D6,$1E,$07,$CD
	.dc.b	$3C,$27,$51,$EE,$AE,$B1,$F9,$79,$D5,$FA,$EB,$4F,$37,$F6,$A7,$E0
	.dc.b	$2B,$CA,$E8,$C4,$7D,$83,$E2,$57,$5B,$E4,$D9,$D9,$F5,$A3,$33,$FA
	.dc.b	$1C,$3E,$54,$E8,$FC,$81,$F6,$A3,$E5,$61,$E8,$8F,$94,$78,$CD,$1F
	.dc.b	$1A,$3C,$67,$57,$E7,$3E,$D7,$81,$F3,$4E,$E5,$3E,$69,$E3,$29,$D8
	.dc.b	$F9,$F5,$EA,$7F,$47,$3E,$5C,$3E,$3D,$FB,$27,$81,$F3,$0B,$87,$C5
	.dc.b	$6E,$1E,$7B,$E4,$FC,$0C,$3B,$61,$F9,$B8,$F6,$D7,$82,$73,$1F,$4E
	.dc.b	$3F,$1D,$7D,$5F,$63,$4F,$C5,$16,$39,$D7,$59,$FD,$F1,$FE,$01,$3F
	.dc.b	$A7,$96,$7F,$C1,$ED,$9D,$3F,$D2,$8C,$18,$8D,$BF,$DD,$0C,$CE,$3E
	.dc.b	$0F,$F3,$84,$AF,$D7,$36,$7F,$50,$D9,$FD,$3B,$67,$FB,$23,$67,$FA
	.dc.b	$4B,$5F,$E6,$6D,$7F,$9F,$36,$BF,$AA,$36,$7F,$AD,$B6,$7F,$B7,$36
	.dc.b	$BD,$13,$E6,$24,$F9,$9B,$1E,$66,$9E,$62,$C7,$99,$AF,$99,$AF,$99
	.dc.b	$B5,$E6,$69,$E6,$2C,$79,$9B,$3E,$68,$F2,$11,$F2,$36,$3C,$8D,$3C
	.dc.b	$85,$3C,$85,$3C,$85,$9F,$23,$63,$C8,$D3,$C8,$59,$F2,$1E,$22,$47
	.dc.b	$89,$B1,$E2,$6B,$E2,$6B,$E2,$6C,$F8,$9B,$3E,$26,$CF,$89,$A7,$8A
	.dc.b	$3C,$04,$8F,$03,$5F,$03,$5F,$03,$67,$C0,$D3,$C0,$58,$F0,$36,$7C
	.dc.b	$11,$D8,$87,$62,$9D,$8B,$3D,$9B,$3D,$9A,$76,$2C,$F6,$7A,$10,$E8
	.dc.b	$59,$E8,$D9,$E8,$D3,$A1,$67,$A3,$C8,$93,$C9,$B3,$C9,$A7,$22,$CF
	.dc.b	$29,$E0,$49,$E0,$D9,$E0,$D9,$E1,$3A,$24,$E9,$B1,$A8,$C1,$1C,$12
	.dc.b	$3F,$D3,$D5,$FB,$D6,$9F,$AA,$2C,$FE,$65,$B3,$F8,$A6,$9F,$74,$B1
	.dc.b	$F4,$9A,$87,$C1,$FF,$24,$70,$62,$3F,$95,$B4,$FE,$5B,$1B,$79,$7C
	.dc.b	$1D,$4D,$24,$F2,$F5,$F4,$47,$07,$F3,$8E,$77,$17,$56,$C6,$63,$07
	.dc.b	$87,$E2,$1F,$16,$F8,$BF,$E8,$31,$86,$FB,$1F,$ED,$9C,$FF,$A5,$4E
	.dc.b	$5F,$E9,$9C,$EE,$86,$FF,$D2,$A3,$0D,$8C,$CF,$06,$BF,$EE,$A3,$9F
	.dc.b	$43,$9F,$FD,$D7,$19,$3F,$AA,$F3,$7C,$C4,$D2,$3F,$41,$FF,$28,$73
	.dc.b	$7F,$B6,$39,$BF,$D9,$6F,$FB,$59,$FF,$66,$F3,$DA,$FF,$60,$B5,$FF
	.dc.b	$55,$B5,$FF,$5B,$BC,$DF,$F6,$3C,$D7,$FD,$62,$D7,$FD,$7A,$DE,$7F
	.dc.b	$F7,$55,$7E,$2A,$5C,$FF,$D6,$CF,$D2,$71,$E5,$B1,$F8,$C7,$F6,$86
	.dc.b	$E3,$F4,$A6,$CB,$1B,$7F,$71,$1B,$8F,$D4,$46,$DA,$6C,$FD,$E4,$6F
	.dc.b	$5F,$D6,$46,$5F,$E0,$F1,$99,$F5,$5F,$91,$F0,$D7,$E9,$67,$0F,$F4
	.dc.b	$F7,$D6,$E6,$3C,$A7,$F6,$68,$CC,$7C,$48,$CB,$C5,$EA,$E4,$F3,$BC
	.dc.b	$4C,$C7,$82,$BF,$A6,$BD,$0C,$47,$F4,$17,$0F,$68,$DB,$C8,$C1,$B3
	.dc.b	$13,$E0,$3C,$87,$CA,$3D,$86,$63,$9B,$93,$D0,$FC,$A8,$F4,$46,$67
	.dc.b	$CA,$FF,$48,$7C,$AF,$38,$F1,$3B,$7C,$AE,$DF,$2C,$F5,$8F,$2B,$F6
	.dc.b	$9F,$2B,$93,$CA,$65,$F2,$C7,$E1,$63,$2F,$DE,$3B,$C7,$27,$C7,$1C
	.dc.b	$67,$D3,$3F,$D5,$5F,$4C,$77,$9D,$9C,$E7,$D8,$7E,$35,$CB,$4F,$61
	.dc.b	$EF,$36,$75,$7D,$E7,$9C,$E2,$F9,$4E,$31,$E5,$8F,$4C,$7F,$5F,$9F
	.dc.b	$2C,$71,$8D,$BE,$59,$F4,$47,$8C,$F0,$1E,$37,$2E,$1E,$CF,$98,$ED
	.dc.b	$1E,$67,$91,$F3,$8E,$E6,$2B,$E4,$B9,$3A,$47,$8C,$F3,$3E,$D3,$27
	.dc.b	$17,$DB,$5F,$BF,$8F,$F4,$03,$FD,$34,$91,$FD,$51,$B1,$FE,$80,$6B
	.dc.b	$C2,$3F,$AF,$57,$FA,$1B,$91,$C3,$FE,$0E,$6C,$FF,$83,$9B,$5F,$E0
	.dc.b	$E6,$D7,$F8,$65,$B3,$FD,$A9,$B3,$F3,$5A,$FB,$5A,$FC,$36,$DA,$7C
	.dc.b	$A7,$F7,$08,$CB,$89,$FE,$ED,$18,$7F,$AA,$C6,$EF,$D4,$47,$D4,$D8
	.dc.b	$F5,$36,$FD,$4D,$9F,$53,$67,$D4,$D7,$D4,$D7,$D4,$DB,$F5,$47,$90
	.dc.b	$F3,$C6,$67,$D5,$0B,$F5,$5F,$D3,$AD,$4F,$13,$4E,$23,$55,$EC,$24
	.dc.b	$FB,$1B,$1E,$C6,$9E,$C2,$C7,$B1,$AF,$B1,$AF,$B1,$B7,$EC,$9F,$0C
	.dc.b	$F7,$23,$DD,$B3,$DD,$A7,$72,$9D,$CA,$77,$2C,$F7,$8F,$B8,$78,$8F
	.dc.b	$93,$7F,$21,$CC,$F0,$73,$7F,$1D,$CD,$F9,$48,$F9,$5B,$3E,$56,$BE
	.dc.b	$56,$BE,$56,$DF,$96,$3E,$EC,$F9,$5F,$B1,$7E,$57,$27,$94,$CD,$F9
	.dc.b	$5C,$DF,$4A,$F4,$9C,$DF,$9E,$66,$38,$C6,$4F,$69,$EB,$23,$EB,$6B
	.dc.b	$EB,$6B,$EB,$6D,$FA,$E3,$F1,$F1,$D5,$F2,$BA,$3C,$AF,$AE,$7C,$B1
	.dc.b	$98,$F2,$99,$7C,$AF,$8C,$87,$8C,$A7,$8C,$B3,$E3,$BF,$95,$1E,$43
	.dc.b	$A1,$E4,$37,$3E,$4A,$F8,$C7,$92,$32,$79,$1C,$C7,$92,$3C,$44,$3C
	.dc.b	$45,$AF,$15,$FD,$78,$F1,$57,$27,$C4,$6E,$BC,$47,$C3,$8F,$14,$64
	.dc.b	$F1,$46,$4F,$13,$E0,$25,$78,$23,$E9,$47,$81,$E0,$F8,$07,$2F,$FA
	.dc.b	$25,$FF,$4B,$9F,$E9,$87,$0E,$30,$FF,$42,$CF,$FA,$37,$93,$FD,$18
	.dc.b	$47,$FD,$21,$F3,$3F,$E8,$F2,$38,$3F,$D1,$71,$FE,$8D,$AF,$F4,$4B
	.dc.b	$5F,$0F,$27,$F4,$CD,$37,$CB,$FE,$9B,$F8,$CF,$F5,$4E,$46,$92,$FF
	.dc.b	$D1,$9A,$C7,$FD,$18,$D7,$FD,$10,$7F,$AD,$39,$9C,$BF,$E8,$D8,$FF
	.dc.b	$47,$F2,$FF,$A3,$9F,$86,$7F,$A3,$0F,$F4,$51,$0F,$F4,$89,$8E,$37
	.dc.b	$1F,$FA,$30,$E3,$CB,$FE,$90,$6C,$B9,$76,$FF,$A3,$A3,$31,$88,$FF
	.dc.b	$47,$C6,$0F,$F4,$A9,$5E,$07,$D8,$9F,$F6,$95,$71,$86,$DF,$B5,$7F
	.dc.b	$DA,$D3,$AA,$FB,$F3,$93,$FD,$A0,$4E,$1F,$F6,$BB,$FE,$D3,$E9,$E4
	.dc.b	$7E,$B7,$E5,$6C,$FF,$54,$C3,$FD,$B9,$B7,$F5,$A7,$FD,$C9,$5B,$9D
	.dc.b	$3C,$23,$FA,$B1,$C2,$3F,$BC,$56,$6B,$94,$E4,$FF,$75,$D7,$37,$8B
	.dc.b	$BA,$FC,$CB,$99,$E2,$E6,$78,$CF,$F4,$2A,$C4,$79,$A3,$CA,$64,$AF
	.dc.b	$66,$D7,$F8,$86,$78,$CF,$94,$E4,$7F,$3A,$73,$5D,$B8,$11,$8B,$E6
	.dc.b	$7C,$02,$D7,$98,$F0,$1C,$DC,$BD,$AF,$DB,$1E,$78,$DC,$61,$B1,$D5
	.dc.b	$A7,$58,$D1,$EB,$8E,$B1,$EB,$3D,$95,$E8,$8C,$DF,$AE,$FD,$55,$D0
	.dc.b	$FF,$0A,$38,$BE,$B5,$E1,$6C,$F8,$63,$EB,$9E,$98,$CC,$7B,$63,$2F
	.dc.b	$C1,$3B,$1E,$5A,$FB,$D1,$A9,$E3,$3E,$8A,$F1,$4E,$4F,$11,$63,$C4
	.dc.b	$43,$D2,$FB,$8F,$9F,$7F,$06,$3C,$37,$F0,$67,$51,$F0,$67,$57,$E2
	.dc.b	$AD,$4F,$0A,$E7,$7E,$48,$F0,$5F,$09,$F2,$47,$C1,$33,$3B,$70,$79
	.dc.b	$8E,$87,$85,$D3,$F0,$63,$CB,$7F,$32,$BB,$4F,$BC,$F9,$53,$F4,$4F
	.dc.b	$0B,$83,$E8,$4F,$AE,$3D,$27,$43,$CF,$3D,$27,$CA,$71,$9F,$C6,$3E
	.dc.b	$67,$C5,$5E,$6B,$FB,$33,$E3,$7E,$3C,$F7,$8F,$65,$F7,$7E,$1B,$5F
	.dc.b	$87,$7E,$17,$C1,$1E,$4B,$EC,$FB,$63,$B3,$EA,$7C,$8E,$CE,$8F,$C9
	.dc.b	$32,$78,$0A,$78,$23,$AB,$F1,$A3,$E4,$DF,$A9,$A7,$7B,$F6,$1F,$09
	.dc.b	$EF,$1F,$0A,$7E,$4B,$C5,$F3,$9F,$08,$F0,$9E,$59,$E7,$3E,$98,$CC
	.dc.b	$F3,$7E,$19,$E9,$9D,$B6,$7C,$AF,$58,$CB,$ED,$8F,$29,$F2,$5F,$2B
	.dc.b	$EB,$38,$C7,$AE,$3C,$31,$B7,$C2,$73,$32,$7C,$73,$DB,$1C,$4F,$0B
	.dc.b	$ED,$8E,$A7,$B6,$33,$3E,$D3,$F4,$97,$F8,$17,$FA,$65,$7F,$78,$7F
	.dc.b	$AC,$57,$F7,$A8,$FE,$6A,$65,$C2,$8D,$BB,$6D,$7F,$52,$EA,$FE,$7E
	.dc.b	$4A,$FE,$4E,$DA,$FE,$2A,$D9,$FE,$26,$D9,$FD,$6B,$67,$F6,$4D,$7F
	.dc.b	$52,$D7,$F9,$93,$6B,$FB,$D3,$67,$FB,$83,$67,$FB,$5B,$6B,$D1,$3E
	.dc.b	$62,$4F,$99,$B1,$E6,$69,$E6,$2C,$79,$9A,$F9,$9A,$F9,$9B,$5E,$66
	.dc.b	$9E,$62,$C7,$99,$B3,$E6,$8F,$21,$1F,$23,$63,$C8,$D3,$C8,$53,$C8
	.dc.b	$53,$C8,$59,$F2,$36,$3C,$8D,$3C,$85,$9F,$21,$E2,$24,$78,$9B,$1E
	.dc.b	$26,$BE,$26,$BE,$26,$CF,$89,$B3,$E2,$6C,$F8,$9A,$78,$A3,$C0,$48
	.dc.b	$F0,$35,$F0,$35,$F0,$36,$7C,$0D,$3C,$05,$8F,$03,$67,$C1,$1D,$88
	.dc.b	$76,$29,$D8,$B3,$D9,$B3,$D9,$A7,$62,$CF,$67,$A1,$0E,$85,$9E,$8D
	.dc.b	$9E,$8D,$3A,$16,$7A,$3C,$89,$3C,$9B,$3C,$9A,$72,$2C,$F2,$9E,$04
	.dc.b	$9E,$0D,$9E,$0D,$9E,$13,$A2,$4E,$9B,$1A,$8C,$11,$C1,$23,$FD,$6B
	.dc.b	$5F,$BD,$69,$FA,$A2,$CF,$E6,$5B,$3F,$8A,$69,$F7,$4B,$1F,$49,$AA
	.dc.b	$DD,$5F,$E7,$28,$19,$70,$D9,$FE,$5D,$3C,$1B,$F4,$5A,$83,$49,$1F
	.dc.b	$EB,$E9,$FF,$61,$F2,$7F,$37,$E5,$FF,$61,$46,$0F,$F5,$9D,$BE,$0B
	.dc.b	$FD,$86,$48,$F0,$BF,$CF,$39,$5F,$29,$49,$3F,$D0,$E3,$2F,$FB,$14
	.dc.b	$FF,$62,$76,$B5,$C4,$7C,$09,$4E,$63,$FD,$8E,$43,$FD,$9A,$6F,$9F
	.dc.b	$FD,$98,$FF,$B1,$88,$7F,$B0,$8A,$7F,$AE,$47,$2F,$FA,$E7,$8D,$B7
	.dc.b	$FA,$DE,$75,$39,$8C,$47,$D4,$3F,$DA,$D1,$97,$9F,$27,$FB,$42,$30
	.dc.b	$7F,$B1,$63,$71,$FE,$C2,$9F,$F5,$D4,$60,$E9,$5F,$EC,$58,$FF,$61
	.dc.b	$92,$3F,$B6,$47,$E0,$97,$67,$FB,$3C,$E0,$FF,$AF,$9F,$D8,$71,$FF
	.dc.b	$5D,$3F,$4D,$5F,$90,$94,$61,$F6,$3C,$23,$FA,$7B,$98,$FC,$9B,$97
	.dc.b	$FA,$03,$97,$9C,$7E,$F6,$3C,$0F,$EC,$0F,$03,$B3,$96,$9F,$CE,$1C
	.dc.b	$47,$13,$C2,$71,$9F,$2B,$97,$C2,$E5,$F2,$B9,$9E,$8E,$63,$CA,$7F
	.dc.b	$0F,$9F,$2B,$FC,$06,$3C,$A6,$E3,$CA,$FF,$3D,$7C,$B1,$C6,$3C,$AF
	.dc.b	$13,$CA,$64,$F2,$99,$3C,$A6,$67,$CA,$E4,$F2,$BF,$CA,$A7,$CA,$FF
	.dc.b	$13,$8E,$AE,$67,$C2,$E6,$3A,$B9,$7C,$2E,$5E,$B1,$EE,$72,$FE,$0F
	.dc.b	$93,$DF,$1D,$A3,$E0,$1F,$0C,$F9,$86,$1F,$45,$70,$7F,$A5,$3F,$1A
	.dc.b	$BC,$47,$B8,$F1,$3B,$8F,$E8,$CE,$E7,$C9,$3B,$7C,$87,$82,$7C,$87
	.dc.b	$F6,$29,$E0,$E0,$F8,$31,$E5,$9E,$EF,$18,$F2,$BE,$B7,$E9,$3B,$3B
	.dc.b	$B8,$3A,$1E,$67,$84,$7B,$1E,$11,$AA,$F6,$57,$D8,$9E,$75,$BA,$D4
	.dc.b	$78,$DF,$A6,$7A,$DE,$A7,$C2,$3A,$BE,$C3,$47,$D0,$79,$57,$81,$FC
	.dc.b	$19,$E0,$39,$47,$80,$F6,$1C,$CF,$86,$73,$32,$7B,$CC,$4F,$A5,$DB
	.dc.b	$F3,$1F,$F1,$C9,$FC,$8C,$FC,$E9,$4F,$D7,$96,$BF,$8C,$12,$7F,$D0
	.dc.b	$6D,$8F,$F4,$43,$FC,$C9,$FE,$89,$1D,$7B,$3F,$C1,$44,$7F,$C1,$4D
	.dc.b	$8F,$F0,$53,$6B,$FC,$26,$D9,$FE,$40,$D9,$FB,$AD,$7E,$20,$70,$FD
	.dc.b	$63,$FB,$9B,$8A,$F3,$57,$9E,$73,$5E,$42,$3E,$46,$C7,$91,$A7,$90
	.dc.b	$B1,$E4,$6B,$E4,$AF,$04,$7F,$81,$27,$C1,$3E,$78,$DB,$E7,$30,$78
	.dc.b	$49,$1E,$16,$C7,$85,$A7,$84,$A7,$85,$EB,$5F,$01,$FA,$86,$0F,$21
	.dc.b	$88,$E2,$78,$EB,$D2,$FD,$C3,$FB,$53,$8A,$F5,$B6,$79,$47,$9C,$8F
	.dc.b	$9D,$B3,$E7,$6B,$E7,$3C,$35,$E7,$7B,$CF,$9D,$F1,$1F,$60,$DB,$EA
	.dc.b	$76,$60,$D9,$F3,$27,$BB,$ED,$8E,$91,$EA,$69,$EA,$3D,$04,$8F,$43
	.dc.b	$5F,$44,$74,$9F,$44,$79,$63,$13,$E5,$8C,$3C,$5F,$11,$E7,$8F,$13
	.dc.b	$63,$C4,$F8,$08,$78,$1E,$53,$E0,$8F,$74,$60,$F0,$38,$38,$9E,$09
	.dc.b	$F2,$BF,$1A,$BE,$D3,$F2,$A3,$27,$90,$B3,$E4,$8F,$11,$D2,$BC,$55
	.dc.b	$CA,$3D,$E6,$67,$C7,$19,$8F,$21,$C8,$87,$96,$3D,$4F,$95,$FA,$AF
	.dc.b	$95,$CD,$79,$5A,$79,$63,$91,$E1,$39,$1F,$34,$E4,$66,$39,$36,$39
	.dc.b	$57,$B1,$F0,$C7,$B0,$F3,$C6,$5E,$86,$4F,$09,$A8,$F8,$27,$27,$E0
	.dc.b	$9A,$24,$73,$9F,$9F,$1E,$63,$91,$EA,$72,$F7,$8E,$13,$FE,$5F,$E1
	.dc.b	$FF,$45,$BF,$E8,$E2,$4F,$FA,$1B,$93,$FD,$2B,$C6,$EB,$72,$FF,$A3
	.dc.b	$EB,$FA,$34,$FF,$A3,$9B,$3A,$7F,$D1,$FD,$DF,$D2,$5F,$F5,$0E,$28
	.dc.b	$C4,$7F,$A9,$DB,$19,$8F,$F5,$1F,$BD,$82,$38,$3F,$D2,$5F,$29,$FE
	.dc.b	$97,$23,$83,$FD,$22,$39,$BF,$D2,$3B,$61,$B1,$FE,$91,$3F,$D0,$4B
	.dc.b	$97,$FD,$23,$18,$6B,$FE,$95,$ED,$F4,$AE,$CA,$47,$DA,$DE,$DF,$F4
	.dc.b	$61,$FE,$89,$7F,$D8,$6E,$39,$FF,$D1,$C7,$FA,$2E,$7F,$D2,$27,$FA
	.dc.b	$61,$C7,$47,$FA,$56,$3F,$BA,$74,$7F,$A2,$49,$C9,$F3,$3A,$3F,$D1
	.dc.b	$4F,$EF,$A2,$CF,$FA,$2E,$34,$47,$EB,$4F,$FA,$4F,$9F,$F0,$F1,$FC
	.dc.b	$11,$AF,$EC,$9A,$FE,$55,$BC,$F8,$6D,$7E,$72,$7F,$28,$FE,$A0,$C6
	.dc.b	$FF,$AC,$8F,$CB,$46,$0F,$DB,$4F,$F7,$83,$9C,$72,$3F,$40,$7E,$9A
	.dc.b	$BC,$13,$C6,$3D,$0D,$7D,$0D,$7D,$0D,$8E,$31,$D4,$D9,$4D,$95,$DD
	.dc.b	$7A,$63,$2D,$8C,$CF,$BE,$79,$57,$AA,$B8,$36,$7D,$E7,$8C,$F3,$39
	.dc.b	$9F,$7B,$5F,$E1,$4D,$7F,$83,$B4,$F0,$BF,$08,$F4,$47,$43,$B3,$E4
	.dc.b	$6C,$F9,$1A,$79,$23,$89,$B2,$C6,$DB,$1B,$8F,$84,$F7,$8C,$9E,$A7
	.dc.b	$AB,$4C,$9E,$C7,$2D,$32,$7B,$9C,$B4,$CC,$FC,$07,$EC,$CF,$89,$F4
	.dc.b	$34,$F4,$16,$7D,$07,$D7,$72,$D3,$31,$F9,$0A,$EC,$FE,$46,$3D,$D3
	.dc.b	$CE,$3D,$8D,$7D,$4D,$7D,$0D,$AE,$75,$D1,$FD,$05,$7B,$9C,$BD,$8F
	.dc.b	$25,$76,$6B,$D9,$AF,$66,$C7,$28,$F8,$71,$83,$A3,$F8,$7A,$F2,$46
	.dc.b	$23,$E5,$BB,$78,$9B,$71,$1F,$22,$30,$7C,$78,$E4,$FC,$73,$89,$FC
	.dc.b	$00,$A6,$63,$93,$96,$B9,$8F,$A6,$E5,$FC,$B3,$FE,$97,$9F,$E3,$51
	.dc.b	$FD,$39,$B1,$FD,$4D,$AE,$E3,$47,$F8,$CF,$8B,$FB,$03,$FC,$25,$B1
	.dc.b	$FC,$15,$B5,$FC,$65,$B3,$FE,$21,$6C,$FF,$75,$6B,$FE,$05,$6B,$FE
	.dc.b	$1E,$6D,$7F,$91,$9B,$3E,$3E,$4F,$E6,$31,$8A,$FF,$2F,$B6,$FF,$D3
	.dc.b	$0D,$AF,$EF,$D5,$C0,$8F,$AD,$B5,$EB,$6D,$7A,$DB,$3E,$B6,$CF,$AD
	.dc.b	$AF,$AD,$AF,$AD,$B7,$EB,$6C,$FA,$CF,$11,$27,$C4,$DA,$F1,$34,$F1
	.dc.b	$16,$3C,$4D,$7C,$4D,$7C,$4D,$AF,$13,$4F,$14,$78,$09,$3E,$06,$C7
	.dc.b	$81,$A7,$80,$A7,$80,$A7,$80,$B3,$E0,$6C,$78,$27,$B1,$1E,$CD,$8E
	.dc.b	$CD,$7B,$35,$EC,$D9,$EC,$D9,$EC,$74,$24,$F4,$6B,$D1,$AF,$46,$CF
	.dc.b	$46,$9D,$23,$91,$0E,$45,$39,$16,$39,$36,$39,$3C,$08,$70,$2C,$70
	.dc.b	$6C,$70,$74,$48,$D3,$63,$51,$82,$38,$24,$7E,$AA,$BF,$2A,$D8,$FC
	.dc.b	$33,$4F,$B2,$5A,$F9,$CD,$8F,$8A,$D3,$DC,$53,$D2,$53,$CC,$5A,$F2
	.dc.b	$B6,$3C,$B3,$FB,$CA,$F2,$B6,$3C,$AD,$9F,$2B,$4F,$29,$63,$CA,$D8
	.dc.b	$F2,$B5,$F2,$B5,$F2,$B6,$BC,$AD,$9F,$29,$FC,$38,$F1,$96,$7F,$4B
	.dc.b	$CB,$FE,$89,$E6,$FF,$44,$40,$91,$FE,$89,$70,$1E,$0F,$FA,$28,$FF
	.dc.b	$45,$63,$FE,$85,$70,$43,$88,$5D,$DF,$E8,$C2,$47,$27,$FD,$23,$C8
	.dc.b	$EB,$A4,$9F,$F4,$C4,$65,$BF,$29,$C0,$8E,$0F,$F4,$67,$63,$4A,$72
	.dc.b	$E7,$FF,$44,$CF,$FA,$4D,$B7,$FE,$8F,$E7,$D1,$FE,$89,$9F,$F5,$47
	.dc.b	$3D,$7D,$A7,$FA,$3C,$FF,$56,$72,$7F,$BF,$4F,$F7,$F3,$9E,$57,$37
	.dc.b	$FA,$2F,$27,$17,$C6,$E6,$FF,$41,$C6,$2F,$E8,$E1,$8A,$F9,$7D,$99
	.dc.b	$AD,$57,$D9,$8F,$EA,$DD,$F9,$AD,$9F,$76,$B7,$4E,$8D,$3F,$79,$AF
	.dc.b	$39,$E0,$7D,$A8,$FF,$0B,$4F,$FA,$17,$E8,$7F,$B8,$D7,$DE,$6B,$F8
	.dc.b	$86,$BF,$8D,$6A,$5E,$0F,$81,$B7,$E0,$9E,$1D,$59,$AF,$2D,$B9,$29
	.dc.b	$FD,$D7,$E0,$6E,$1C,$70,$FC,$04,$E1,$FE,$EB,$36,$BE,$49,$CE,$B8
	.dc.b	$36,$77,$18,$AE,$0F,$E9,$27,$47,$F7,$EA,$CD,$7C,$FA,$AF,$E8,$62
	.dc.b	$CF,$0A,$E5,$5E,$97,$B1,$2B,$F3,$0E,$8F,$F1,$15,$72,$9E,$BD,$08
	.dc.b	$C4,$F1,$BF,$23,$4E,$57,$D0,$F3,$3D,$0D,$38,$34,$7C,$A7,$4E,$23
	.dc.b	$D4,$ED,$F9,$77,$EB,$8D,$3E,$47,$E5,$98,$9E,$EF,$53,$33,$CA,$32
	.dc.b	$F6,$8F,$95,$18,$9F,$83,$5D,$6B,$85,$72,$AF,$7C,$60,$8F,$F1,$CB
	.dc.b	$F9,$0F,$33,$C6,$E2,$7C,$C4,$9E,$2F,$F2,$78,$D4,$62,$3D,$4D,$8C
	.dc.b	$CF,$88,$93,$F1,$DF,$15,$7A,$5C,$C7,$B8,$F0,$38,$AF,$85,$5D,$27
	.dc.b	$91,$0F,$0C,$F4,$3D,$8F,$23,$D1,$5D,$DE,$13,$CD,$E2,$73,$AE,$F3
	.dc.b	$B8,$C9,$E5,$8F,$80,$7D,$53,$75,$E0,$7E,$14,$F1,$73,$3E,$33,$9B
	.dc.b	$83,$9D,$FD,$CA,$E4,$7C,$4B,$F3,$4F,$79,$D5,$73,$8C,$57,$E5,$9E
	.dc.b	$6F,$DC,$BF,$15,$7E,$56,$73,$5E,$3B,$EB,$3E,$F6,$DF,$8A,$7E,$4B
	.dc.b	$5F,$87,$1A,$7C,$53,$D6,$7E,$C3,$E6,$AE,$E7,$C4,$8E,$4F,$C1,$76
	.dc.b	$79,$CC,$C7,$39,$FA,$8F,$38,$DC,$7A,$6F,$DB,$1F,$04,$CD,$77,$9F
	.dc.b	$2C,$75,$69,$C5,$EE,$71,$7B,$BB,$8F,$4B,$F4,$E7,$BC,$FE,$BC,$D9
	.dc.b	$CC,$EB,$3E,$13,$8B,$F9,$08,$E3,$1F,$C5,$A3,$73,$D4,$F1,$BD,$4F
	.dc.b	$C3,$D7,$C1,$7B,$CF,$59,$F7,$9C,$63,$99,$DD,$E7,$1D,$CF,$1C,$7D
	.dc.b	$AA,$F1,$CF,$EA,$63,$89,$97,$89,$99,$E2,$7D,$98,$F1,$C7,$86,$33
	.dc.b	$5F,$66,$7D,$33,$FE,$08,$7F,$90,$1F,$E2,$69,$FE,$A0,$D9,$FE,$EE
	.dc.b	$FF,$33,$31,$3C,$67,$71,$FC,$F0,$C9,$FC,$D7,$C5,$FE,$9E,$49,$FE
	.dc.b	$56,$D8,$FD,$43,$67,$F7,$6D,$9F,$E2,$0D,$9F,$E0,$8D,$7F,$72,$D9
	.dc.b	$FE,$04,$DA,$FE,$EE,$DA,$FE,$DA,$DA,$F3,$47,$90,$8F,$91,$B5,$E4
	.dc.b	$69,$E4,$2C,$79,$1A,$F9,$1B,$1E,$46,$D7,$91,$A7,$90,$B3,$E4,$3C
	.dc.b	$44,$9F,$13,$63,$C4,$D3,$C4,$53,$C4,$58,$F1,$36,$7C,$4D,$9F,$13
	.dc.b	$4F,$14,$F8,$09,$1E,$06,$C7,$81,$AF,$81,$A7,$80,$B3,$E0,$6C,$F8
	.dc.b	$1B,$3E,$08,$EC,$47,$B3,$5E,$CD,$8E,$CD,$9E,$CD,$3B,$16,$7B,$1D
	.dc.b	$08,$74,$2C,$F4,$6C,$F4,$6C,$F4,$69,$D0,$E4,$49,$E4,$D9,$E4,$D9
	.dc.b	$E4,$D3,$94,$F0,$24,$F0,$6C,$F0,$6C,$F0,$9D,$12,$74,$D8,$D4,$60
	.dc.b	$8E,$09,$3F,$E5,$6A,$FD,$43,$67,$F2,$ED,$9F,$C4,$34,$FB,$65,$8F
	.dc.b	$A0,$D8,$F8,$ED,$7E,$03,$4F,$59,$6B,$D0,$D9,$F3,$B6,$7C,$F1,$FE
	.dc.b	$65,$AF,$3B,$4F,$39,$63,$CE,$D9,$F3,$B4,$FC,$76,$0F,$A1,$FE,$47
	.dc.b	$5F,$EC,$B6,$C6,$67,$FD,$A7,$1A,$EB,$FF,$6B,$D7,$FB,$49,$AF,$FB
	.dc.b	$71,$FF,$6F,$3F,$EE,$0D,$1D,$9F,$EC,$EE,$BF,$F3,$F6,$1F,$E4,$7F
	.dc.b	$02,$DF,$F9,$05,$AB,$AB,$FD,$A8,$FF,$9A,$17,$81,$A7,$3A,$FF,$AE
	.dc.b	$1B,$7F,$EE,$E7,$75,$FE,$99,$7F,$D3,$CD,$AE,$47,$3D,$FA,$6F,$E2
	.dc.b	$DD,$F3,$94,$CD,$F3,$F1,$EB,$E5,$AF,$24,$79,$7B,$FF,$DC,$D3,$FE
	.dc.b	$DC,$C9,$18,$8F,$F7,$DC,$63,$4F,$F7,$9C,$E6,$7D,$5D,$FF,$EE,$A2
	.dc.b	$77,$FF,$BA,$A3,$FB,$01,$FE,$E7,$3F,$DD,$66,$3B,$FF,$DD,$8F,$FB
	.dc.b	$83,$7F,$F5,$D2,$EC,$FF,$75,$CA,$1D,$FF,$EE,$C7,$81,$FE,$98,$1D
	.dc.b	$DF,$EE,$97,$39,$FF,$B8,$A3,$FD,$BF,$0E,$EA,$7F,$BA,$B1,$F0,$BA
	.dc.b	$ED,$91,$83,$CA,$FF,$B9,$1F,$24,$F9,$7C,$1F,$F6,$C4,$7F,$B4,$FB
	.dc.b	$BF,$DB,$B3,$FE,$AC,$5D,$5A,$8C,$77,$7E,$7A,$77,$1F,$EE,$D8,$C1
	.dc.b	$FA,$2C,$DC,$9F,$B4,$C3,$F5,$15,$FE,$FF,$8C,$36,$3F,$8B,$47,$F7
	.dc.b	$73,$30,$9E,$73,$A3,$FB,$4C,$EA,$33,$3F,$C9,$23,$94,$FF,$02,$3C
	.dc.b	$31,$CE,$38,$1F,$C9,$CD,$BF,$C1,$8F,$21,$CE,$3B,$9C,$DC,$CF,$F1
	.dc.b	$39,$E2,$FF,$0F,$31,$3B,$9F,$03,$E9,$7F,$B8,$4F,$8A,$31,$3F,$D9
	.dc.b	$23,$13,$E1,$37,$3C,$E7,$DE,$76,$8C,$9E,$93,$B9,$C5,$E9,$1C,$67
	.dc.b	$E0,$3E,$C9,$E1,$3D,$23,$E3,$B6,$3F,$AF,$CF,$92,$3C,$07,$48,$E3
	.dc.b	$1D,$1F,$1C,$7C,$39,$E7,$3C,$5A,$78,$63,$53,$F3,$23,$2F,$27,$C7
	.dc.b	$CF,$D6,$77,$3E,$E7,$C7,$3F,$DF,$EB,$B4,$F3,$21,$FD,$BD,$CB,$C6
	.dc.b	$3A,$9F,$48,$AF,$C9,$6C,$F6,$8E,$6F,$F7,$19,$E6,$FC,$78,$E8,$7C
	.dc.b	$89,$FA,$07,$57,$07,$C8,$70,$F9,$5D,$BE,$13,$71,$F0,$0E,$C4,$9F
	.dc.b	$C8,$CF,$98,$F9,$31,$B8,$F9,$11,$F1,$4E,$B1,$EB,$9F,$01,$E3,$3C
	.dc.b	$06,$1E,$51,$E6,$7E,$03,$C8,$EE,$75,$8E,$51,$D4,$DB,$88,$DB,$EC
	.dc.b	$7D,$0F,$CC,$9F,$96,$48,$ED,$1F,$B4,$3F,$D2,$91,$FC,$3C,$91,$FE
	.dc.b	$91,$6C,$7F,$20,$38,$1F,$DF,$4E,$31,$FC,$DE,$4F,$F2,$6E,$77,$C9
	.dc.b	$7F,$8C,$9B,$1F,$E3,$26,$D7,$F8,$ED,$B5,$FE,$17,$6C,$FF,$10,$6C
	.dc.b	$FE,$49,$AF,$E1,$83,$89,$FC,$53,$6F,$CB,$3F,$C8,$EB,$CD,$7E,$52
	.dc.b	$3E,$56,$C7,$95,$B7,$E5,$6C,$F9,$5B,$3E,$56,$BE,$5B,$F1,$4F,$C7
	.dc.b	$3C,$27,$72,$47,$76,$D7,$76,$9D,$CB,$1D,$DA,$F7,$AF,$0C,$FC,$26
	.dc.b	$DF,$C2,$8F,$87,$3F,$22,$73,$5E,$58,$FE,$E5,$7E,$87,$E4,$18,$9D
	.dc.b	$B8,$39,$4F,$8A,$BC,$0D,$BF,$03,$F0,$27,$DC,$49,$F7,$36,$3D,$CD
	.dc.b	$3D,$C5,$3D,$CF,$53,$D0,$F8,$8D,$47,$89,$B3,$E2,$7D,$53,$C5,$F5
	.dc.b	$5F,$BE,$BE,$2B,$67,$D1,$1E,$62,$3E,$66,$CF,$99,$AF,$98,$F0,$C7
	.dc.b	$9A,$7A,$47,$88,$C9,$E2,$BF,$9B,$5E,$26,$D7,$88,$F0,$12,$7C,$0D
	.dc.b	$7C,$13,$E2,$3E,$11,$6F,$E1,$47,$C7,$8C,$4F,$96,$BB,$CF,$96,$72
	.dc.b	$F9,$67,$E4,$47,$95,$A7,$96,$3C,$64,$3C,$6F,$96,$3C,$6D,$AF,$1D
	.dc.b	$FC,$4A,$F2,$4E,$4F,$21,$D9,$F2,$39,$7C,$87,$CA,$AF,$23,$63,$C8
	.dc.b	$F8,$8F,$24,$78,$9B,$5E,$2B,$F8,$31,$E2,$33,$5E,$27,$A1,$E2,$32
	.dc.b	$78,$AB,$E3,$9E,$22,$C7,$89,$F0,$47,$8C,$8E,$09,$5E,$39,$F8,$13
	.dc.b	$E3,$37,$19,$7C,$66,$67,$8C,$F9,$DF,$0C,$79,$CF,$07,$6F,$FA,$A1
	.dc.b	$FF,$53,$76,$FF,$3F,$7F,$CE,$51,$FE,$AA,$4E,$DF,$E5,$FF,$25,$FE
	.dc.b	$80,$C8,$D2,$47,$FA,$BC,$FD,$07,$6F,$FA,$6C,$FF,$54,$77,$7F,$A8
	.dc.b	$23,$FD,$48,$E1,$FF,$52,$1F,$EA,$7D,$74,$FF,$AC,$7B,$BF,$D4,$CF
	.dc.b	$FA,$AF,$5F,$F5,$23,$FA,$DE,$EF,$F5,$1E,$BF,$EB,$3E,$EB,$BF,$FA
	.dc.b	$CE,$FF,$D4,$B7,$FE,$B3,$ED,$FF,$4F,$3F,$E9,$1E,$EF,$F5,$13,$FE
	.dc.b	$A7,$8C,$BF,$EC,$36,$C3,$BB,$FD,$47,$3F,$EA,$D6,$BF,$EA,$22,$1F
	.dc.b	$37,$B3,$FD,$46,$FF,$A7,$FB,$7F,$D2,$E7,$FA,$89,$FF,$50,$F8,$5F
	.dc.b	$DC,$63,$FD,$3D,$1F,$EA,$5E,$DD,$E7,$FE,$AA,$7F,$D3,$AF,$FA,$66
	.dc.b	$3F,$8D,$F8,$58,$7F,$D5,$1D,$BF,$E9,$E7,$FD,$55,$DD,$78,$3F,$A1
	.dc.b	$C3,$BB,$FD,$52,$67,$0F,$F0,$13,$FC,$22,$5C,$94,$FD,$CF,$85,$FE
	.dc.b	$2A,$BF,$F0,$24,$6F,$8B,$F8,$7F,$73,$C3,$FF,$51,$13,$2F,$C8,$5A
	.dc.b	$75,$18,$5A,$F2,$39,$D5,$9F,$E5,$9E,$8F,$AA,$FF,$AC,$B5,$FF,$02
	.dc.b	$B7,$B0,$D3,$F9,$1F,$D1,$7F,$B4,$EB,$99,$FD,$31,$C5,$7F,$46,$9D
	.dc.b	$F1,$7F,$18,$6D,$70,$9F,$EC,$95,$C3,$6F,$E9,$75,$A9,$CD,$D9,$CD
	.dc.b	$F7,$8C,$3F,$ED,$0B,$E4,$FF,$8B,$6F,$37,$FE,$E7,$9C,$B6,$BC,$95
	.dc.b	$CC,$91,$E0,$7C,$D7,$FE,$02,$AE,$32,$69,$8C,$3E,$3D,$72,$AE,$17
	.dc.b	$E2,$BF,$9B,$39,$3D,$11,$9A,$E3,$3E,$12,$57,$2A,$F1,$9C,$0E,$51
	.dc.b	$EB,$72,$D3,$8B,$B7,$EA,$3C,$1F,$55,$78,$70,$F3,$61,$EC,$8F,$8B
	.dc.b	$3C,$8E,$97,$E3,$79,$99,$39,$DF,$5C,$3D,$73,$B2,$47,$8D,$FB,$73
	.dc.b	$C6,$FC,$13,$EA,$8E,$66,$6F,$C7,$5C,$A3,$C7,$3C,$CF,$C0,$1E,$DA
	.dc.b	$E7,$1E,$69,$E0,$F4,$6D,$F9,$AB,$94,$F4,$21,$F6,$0F,$13,$ED,$C3
	.dc.b	$E3,$E1,$F2,$63,$C0,$D9,$D9,$F1,$6B,$B3,$D3,$0F,$2E,$1F,$08,$F2
	.dc.b	$B8,$BF,$5C,$F2,$9F,$3E,$1E,$09,$F1,$1E,$02,$CE,$DF,$5B,$E9,$9E
	.dc.b	$B8,$79,$5B,$3C,$1B,$3E,$7A,$EC,$D9,$EC,$D3,$B1,$F1,$B0,$F6,$BF
	.dc.b	$36,$3C,$38,$7E,$96,$B9,$DF,$AA,$79,$B6,$B9,$9C,$4F,$E2,$B1,$A3
	.dc.b	$D7,$7D,$AF,$E3,$4F,$86,$FA,$D6,$9B,$19,$8E,$4D,$3B,$CF,$96,$33
	.dc.b	$1E,$13,$E4,$3E,$AC,$3E,$43,$4F,$90,$58,$F8,$98,$7C,$A3,$D8,$58
	.dc.b	$F2,$1D,$AB,$C1,$3E,$2A,$FC,$EC,$F3,$AE,$2D,$38,$CE,$DF,$D3,$57
	.dc.b	$C2,$8F,$46,$1F,$16,$FE,$0C,$7A,$8F,$C8,$3E,$A6,$D7,$A9,$A7,$A9
	.dc.b	$F0,$3F,$1E,$FD,$4D,$8C,$BE,$0C,$3B,$E1,$EC,$AD,$CE,$6B,$E0,$5F
	.dc.b	$BA,$3B,$CF,$B9,$B3,$EE,$8F,$C9,$C7,$FB,$B6,$BF,$A6,$B6,$BF,$AC
	.dc.b	$46,$23,$F8,$F1,$9A,$FE,$CF,$19,$8F,$E4,$86,$4F,$EA,$3C,$5F,$C8
	.dc.b	$09,$3F,$C2,$5B,$1F,$C5,$5B,$7F,$B6,$6C,$FE,$79,$B3,$FB,$D6,$BF
	.dc.b	$8B,$A4,$61,$FE,$92,$DB,$FE,$DA,$DA,$FE,$D8,$DA,$F3,$C7,$94,$8F
	.dc.b	$95,$B7,$E5,$69,$E5,$2C,$79,$5A,$F9,$67,$B4,$FF,$5F,$6C,$FF,$6D
	.dc.b	$6C,$FF,$6B,$6B,$FD,$8D,$B7,$EE,$9E,$AF,$B9,$B7,$EE,$69,$EE,$2D
	.dc.b	$7B,$8F,$61,$2F,$D8,$D8,$F6,$34,$F6,$14,$F6,$3D,$23,$D8,$D3,$D8
	.dc.b	$58,$F6,$35,$F6,$36,$BD,$91,$D5,$F6,$36,$BD,$8D,$9F,$63,$4F,$65
	.dc.b	$7A,$89,$1E,$A6,$C7,$A9,$AF,$A8,$E4,$FA,$9B,$1E,$A6,$9E,$A2,$9E
	.dc.b	$A2,$D7,$AA,$7A,$BE,$A6,$D7,$A9,$B5,$EA,$6D,$7A,$A3,$D0,$47,$D0
	.dc.b	$D7,$D1,$1C,$0F,$41,$63,$D0,$D8,$F4,$35,$F4,$36,$BD,$13,$D5,$F4
	.dc.b	$36,$BD,$0D,$3D,$05,$AF,$41,$E6,$21,$E6,$75,$5E,$66,$9E,$62,$C7
	.dc.b	$99,$AF,$99,$B5,$E6,$8E,$AF,$99,$B7,$E6,$6C,$F9,$9A,$79,$8F,$21
	.dc.b	$F6,$CF,$E4,$51,$FC,$80,$91,$FC,$7B,$D9,$FE,$46,$47,$07,$FA,$23
	.dc.b	$BF,$FD,$A8,$4F,$31,$96,$7C,$47,$85,$FE,$8E,$E8,$FE,$2F,$98,$6A
	.dc.b	$CB,$10,$DD,$F4,$38,$AD,$4F,$19,$CB,$CE,$32,$AF,$BB,$DA,$FA,$C3
	.dc.b	$A3,$A4,$63,$7B,$5E,$5A,$E3,$7D,$1E,$11,$FE,$E8,$3F,$DD,$A4,$3B
	.dc.b	$1F,$ED,$BF,$07,$FD,$D3,$3E,$4E,$D4,$7F,$BB,$E7,$FD,$D8,$7F,$BB
	.dc.b	$CF,$F7,$89,$83,$98,$8F,$F6,$0B,$FE,$F1,$3F,$DE,$04,$36,$B9,$FF
	.dc.b	$DD,$F1,$87,$FD,$E6,$FF,$BB,$FB,$FD,$EF,$FB,$B6,$FF,$D8,$75,$FE
	.dc.b	$F2,$8F,$A6,$64,$FF,$7A,$46,$BB,$BF,$DD,$F1,$F6,$27,$FD,$E9,$E3
	.dc.b	$E4,$FF,$77,$C7,$DF,$EF,$A9,$FF,$68,$BF,$EA,$22,$19,$3F,$DE,$67
	.dc.b	$E1,$38,$FF,$DE,$6F,$E8,$CF,$F7,$D6,$DF,$EF,$33,$FD,$E5,$8F,$FB
	.dc.b	$B3,$7F,$C3,$BF,$84,$E0,$FF,$6D,$CE,$8F,$6F,$7F,$FB,$BE,$FF,$23
	.dc.b	$75,$FC,$B7,$7F,$FB,$D2,$BF,$1C,$70,$2C,$7E,$62,$3F,$DE,$91,$88
	.dc.b	$FC,$EF,$77,$FB,$DA,$31,$8F,$FB,$E9,$BC,$1F,$EF,$77,$33,$6B,$F0
	.dc.b	$3D,$DE,$86,$F7,$7F,$BC,$AB,$B4,$E7,$0F,$D0,$BB,$AF,$B2,$D7,$F5
	.dc.b	$AD,$7F,$64,$D7,$FD,$4A,$7F,$86,$C8,$7F,$88,$8F,$E2,$7C,$BF,$DD
	.dc.b	$0F,$E2,$0F,$F1,$73,$F8,$7C,$FF,$1F,$6D,$7F,$1F,$69,$FC,$7C,$A7
	.dc.b	$13,$F8,$A9,$7B,$3E,$F1,$FC,$7D,$C5,$7F,$87,$CF,$0C,$64,$DB,$99
	.dc.b	$FE,$45,$3D,$27,$31,$FE,$0A,$9E,$3B,$F2,$6C,$70,$9D,$D7,$F7,$59
	.dc.b	$CB,$FC,$AE,$33,$7F,$D8,$27,$2F,$C1,$9F,$3D,$77,$3A,$5F,$76,$9D
	.dc.b	$EB,$8B,$F1,$E7,$89,$E1,$76,$F8,$63,$6F,$CB,$37,$5E,$39,$DB,$E3
	.dc.b	$BF,$3C,$7F,$5F,$AE,$35,$C2,$3C,$13,$F2,$E7,$2F,$09,$EC,$E6,$7A
	.dc.b	$9C,$27,$67,$B6,$7D,$07,$C7,$BC,$5F,$8D,$B5,$E0,$6C,$F6,$6B,$F8
	.dc.b	$36,$DF,$E1,$1B,$7F,$7E,$7B,$57,$CB,$3C,$AF,$A2,$3C,$B1,$98,$F2
	.dc.b	$DF,$8E,$30,$D8,$F9,$D1,$C9,$F8,$C7,$95,$D9,$E9,$2B,$93,$89,$E4
	.dc.b	$2C,$75,$3E,$A4,$F9,$23,$33,$F5,$4F,$05,$FD,$18,$F8,$51,$D9,$A7
	.dc.b	$03,$C1,$5D,$23,$8D,$7C,$8B,$F8,$F1,$F9,$3A,$F2,$56,$DB,$1F,$6C
	.dc.b	$E2,$77,$BF,$C0,$46,$8F,$99,$1A,$7C,$47,$EB,$EF,$D9,$5D,$4F,$CA
	.dc.b	$47,$13,$F3,$A7,$F8,$4A,$7F,$D5,$B3,$FE,$12,$8F,$EA,$27,$F9,$3C
	.dc.b	$D1,$8A,$FF,$34,$B8,$33,$5F,$CF,$FD,$4F,$E2,$BC,$06,$93,$0F,$E0
	.dc.b	$B1,$96,$C6,$6B,$FC,$51,$39,$6C,$66,$BF,$7D,$39,$6C,$66,$BF,$75
	.dc.b	$39,$6C,$66,$BF,$71,$39,$6C,$65,$FE,$49,$19,$6C,$66,$FF,$B1,$4E
	.dc.b	$5B,$19,$C3,$FB,$0D,$FF,$8B,$2B,$51,$8B,$FF,$15,$56,$EB,$FA,$C5
	.dc.b	$69,$B1,$99,$FF,$09,$CE,$5B,$19,$C3,$E8,$57,$CA,$24,$FC,$86,$C7
	.dc.b	$C6,$69,$F1,$0B,$1F,$09,$AF,$C0,$6E,$1E,$E6,$E1,$EB,$9F,$F2,$45
	.dc.b	$F8,$E7,$E0,$36,$3D,$F1,$C0,$F6,$95,$F5,$B6,$FD,$71,$E9,$23,$E9
	.dc.b	$6C,$7A,$5A,$7A,$4A,$7A,$4B,$7E,$96,$DF,$A7,$0F,$99,$7E,$03,$B1
	.dc.b	$2F,$B3,$6B,$B3,$5E,$CD,$BE,$CD,$BE,$D8,$7C,$3A,$F8,$F7,$EF,$3E
	.dc.b	$39,$5F,$8E,$79,$9F,$8E,$D3,$E3,$96,$BD,$11,$E6,$24,$79,$9A,$F9
	.dc.b	$9B,$7E,$66,$DF,$99,$F1,$4F,$98,$EC,$F9,$9A,$79,$AB,$DD,$1E,$66
	.dc.b	$C7,$99,$A7,$99,$F2,$10,$F2,$16,$BC,$8D,$AF,$23,$E0,$3C,$93,$D2
	.dc.b	$BC,$8D,$8F,$24,$7B,$0F,$21,$5F,$23,$63,$C9,$1E,$22,$57,$89,$B5
	.dc.b	$E2,$9E,$D1,$E2,$AF,$09,$1F,$66,$1F,$C2,$4F,$C6,$C6,$63,$E6,$1F
	.dc.b	$E7,$E8,$FE,$3C,$DC,$3F,$D5,$6F,$CF,$3F,$D0,$7D,$9F,$EC,$0E,$BF
	.dc.b	$F5,$EC,$7F,$36,$87,$67,$FB,$0A,$31,$3A,$7F,$D1,$7E,$B7,$F3,$37
	.dc.b	$F9,$E8,$70,$FF,$40,$6B,$C2,$FF,$D0,$FD,$7F,$EB,$4C,$BF,$8E,$76
	.dc.b	$7F,$A6,$B3,$FF,$61,$DF,$FA,$A2,$B4,$FF,$AE,$F9,$7F,$D3,$F1,$88
	.dc.b	$FF,$63,$6D,$89,$FF,$5A,$6D,$FE,$87,$F1,$7A,$18,$CB,$FD,$3D,$E2
	.dc.b	$DC,$3F,$D7,$C7,$FA,$E2,$BF,$D7,$DC,$BF,$EC,$0A,$DC,$FF,$AF,$4F
	.dc.b	$F5,$91,$FE,$B7,$9F,$F4,$A7,$8B,$FE,$B4,$8F,$F5,$F7,$75,$ED,$FF
	.dc.b	$5D,$47,$FA,$2D,$47,$FA,$E1,$FF,$5B,$47,$FA,$C7,$C5,$FF,$75,$BF
	.dc.b	$EB,$62,$1F,$EB,$A8,$FF,$5A,$BF,$DB,$1A,$E7,$BF,$FD,$76,$FF,$B0
	.dc.b	$3B,$BF,$D6,$E7,$FA,$EB,$A7,$F9,$74,$7F,$AD,$A3,$FD,$F5,$E4,$FF
	.dc.b	$BB,$E3,$FD,$6D,$A7,$FA,$CD,$FF,$6C,$9F,$EB,$73,$FD,$7D,$C3,$CB
	.dc.b	$86,$4F,$F5,$0E,$CB,$5C,$3B,$FF,$D6,$4F,$FB,$DE,$7F,$D5,$8A,$3F
	.dc.b	$D6,$8F,$6E,$FD,$9F,$EB,$27,$FD,$D3,$C3,$FE,$AF,$7F,$AC,$F8,$D9
	.dc.b	$85,$A6,$47,$FB,$A6,$3F,$D6,$A7,$FA,$D0,$B1,$FD,$8B,$C5,$FD,$2B
	.dc.b	$C7,$A1,$18,$84,$62,$7F,$3E,$D8,$FF,$0D,$6F,$BE,$F3,$49,$5F,$A6
	.dc.b	$F0,$1E,$6F,$56,$C6,$63,$04,$7F,$D8,$BE,$77,$F6,$59,$DE,$1F,$EC
	.dc.b	$C8,$FE,$A1,$EA,$CA,$FF,$6C,$7E,$76,$F2,$31,$7F,$E0,$0A,$F6,$68
	.dc.b	$69,$2F,$DA,$FF,$73,$B7,$9F,$CB,$7F,$9F,$87,$CF,$9C,$DF,$83,$E6
	.dc.b	$F8,$1C,$7C,$AB,$19,$3F,$83,$CA,$7E,$73,$63,$31,$83,$C5,$E0,$72
	.dc.b	$B6,$1E,$25,$69,$F4,$78,$13,$0F,$87,$E4,$DB,$E3,$1F,$87,$78,$CF
	.dc.b	$88,$91,$E6,$6C,$70,$3E,$07,$81,$CF,$1F,$16,$3F,$BC,$BF,$A3,$E0
	.dc.b	$7C,$93,$E5,$77,$D8,$C9,$F0,$A3,$E0,$91,$C1,$F6,$BE,$23,$86,$3E
	.dc.b	$13,$DE,$FD,$5F,$9B,$93,$F8,$0F,$9B,$B5,$BC,$55,$F3,$89,$5F,$49
	.dc.b	$B3,$BE,$3E,$39,$FB,$1F,$C1,$7C,$BC,$BF,$C6,$FC,$DD,$BE,$D3,$F6
	.dc.b	$89,$3F,$79,$B1,$E3,$F9,$2E,$18,$7C,$87,$F1,$DF,$25,$2F,$F3,$3F
	.dc.b	$25,$6F,$D5,$3F,$86,$24,$FE,$29,$B1,$EF,$F9,$AE,$18,$7D,$47,$FC
	.dc.b	$99,$DE,$F9,$7A,$7E,$77,$E3,$EE,$3F,$C2,$0F,$C2,$4E,$DF,$EB,$AD
	.dc.b	$C3,$FD,$31,$5F,$BF,$24,$F1,$8F,$E2,$52,$F5,$BA,$C7,$D9,$8C,$4E
	.dc.b	$E1,$1D,$71,$FC,$E3,$A8,$E1,$7E,$AB,$E7,$5A,$6D,$FA,$1B,$5E,$26
	.dc.b	$BD,$9B,$7D,$6B,$BD,$F5,$BF,$24,$F8,$DE,$86,$23,$57,$E4,$8D,$DF
	.dc.b	$42,$3B,$32,$58,$F8,$51,$A7,$91,$88,$D5,$FB,$5A,$78,$A3,$71,$96
	.dc.b	$C6,$63,$D6,$F8,$23,$27,$C3,$79,$B4,$F0,$12,$38,$B6,$3A,$11,$FB
	.dc.b	$B5,$F0,$A7,$C4,$49,$FC,$4C,$65,$E1,$3F,$45,$F0,$B4,$EC,$78,$E7
	.dc.b	$C2,$73,$71,$1C,$AF,$EA,$46,$08,$73,$9E,$33,$BB,$F1,$4F,$A2,$3E
	.dc.b	$24,$F5,$8E,$11,$E8,$32,$71,$3C,$F5,$F8,$77,$11,$F3,$0E,$27,$8A
	.dc.b	$34,$F7,$3D,$A6,$88,$FA,$9B,$5E,$67,$F4,$77,$67,$9C,$70,$8C,$4E
	.dc.b	$5B,$5E,$89,$F0,$5F,$B6,$F8,$92,$B7,$3E,$48,$F1,$1A,$BE,$97,$CE
	.dc.b	$7C,$AF,$C2,$8E,$37,$F3,$A7,$E0,$47,$07,$A3,$63,$C4,$D9,$F9,$31
	.dc.b	$E5,$BE,$71,$F7,$5B,$5F,$2D,$AF,$C1,$8F,$25,$F9,$DF,$21,$8B,$FD
	.dc.b	$65,$F8,$23,$35,$E4,$6C,$66,$3E,$D5,$73,$7E,$25,$F8,$23,$E4,$57
	.dc.b	$53,$C3,$1F,$2A,$7F,$B5,$1F,$9B,$2C,$FF,$19,$6D,$FF,$89,$49,$5F
	.dc.b	$CB,$1B,$1C,$8F,$C2,$FC,$77,$E0,$49,$3F,$01,$B1,$F6,$5B,$87,$DF
	.dc.b	$6C,$FE,$9D,$B3,$F9,$56,$BD,$A3,$A1,$1E,$8D,$9E,$8D,$3A,$16,$3A
	.dc.b	$35,$E8,$72,$24,$72,$6C,$72,$69,$C8,$A7,$28,$E0,$48,$E0,$D8,$E0
	.dc.b	$D7,$84,$68,$8E,$9A,$E8,$C1,$0C,$12,$FF,$53,$5F,$21,$B3,$EF,$6C
	.dc.b	$FA,$1B,$5E,$36,$CF,$81,$57,$F7,$43,$A1,$63,$A3,$6B,$A3,$6B,$A3
	.dc.b	$67,$A3,$67,$D5,$5D,$A7,$FB,$CC,$FF,$9D,$48,$E0,$FE,$BF,$F0,$DC
	.dc.b	$6B,$FB,$9B,$68,$D2,$4F,$F6,$6A,$FF,$06,$C6,$2B,$FB,$F4,$62,$74
	.dc.b	$76,$2C,$7F,$80,$E7,$85,$FF,$80,$2B,$84,$E6,$B8,$3D,$1B,$3F,$E0
	.dc.b	$6A,$DB,$C6,$32,$7F,$86,$A3,$2F,$F8,$72,$AD,$7F,$85,$A6,$BF,$E1
	.dc.b	$28,$B7,$F0,$2B,$9C,$FF,$87,$27,$35,$A6,$CE,$A7,$93,$6B,$94,$7E
	.dc.b	$F5,$AF,$F0,$6A,$FF,$54,$12,$7F,$C4,$9F,$9E,$7F,$AE,$DE,$8D,$9F
	.dc.b	$F3,$13,$FE,$AE,$8F,$E1,$10,$8C,$3F,$C2,$DA,$FF,$16,$6B,$FE,$81
	.dc.b	$5E,$67,$FA,$FD,$B3,$D1,$B3,$CE,$75,$E6,$4A,$D3,$67,$C1,$E5,$F3
	.dc.b	$24,$F3,$7B,$79,$72,$3B,$36,$7D,$1E,$97,$8C,$93,$E3,$7C,$DF,$35
	.dc.b	$23,$CC,$D9,$F8,$1F,$35,$EB,$24,$FA,$DF,$77,$0C,$8F,$73,$67,$E4
	.dc.b	$70,$FC,$32,$4F,$C3,$7E,$37,$04,$8F,$8C,$D9,$FA,$1B,$FC,$B2,$4F
	.dc.b	$CB,$7E,$77,$C6,$C8,$F9,$CD,$9F,$B1,$F1,$FF,$4C,$93,$F4,$DF,$AD
	.dc.b	$F1,$D2,$3E,$B3,$67,$F0,$1F,$19,$F6,$C9,$3F,$6D,$FB,$DF,$19,$23
	.dc.b	$EF,$36,$7F,$11,$F1,$9F,$83,$24,$FE,$0D,$FF,$7F,$61,$23,$FD,$0A
	.dc.b	$D9,$FC,$F6,$1F,$ED,$57,$FD,$16,$D7,$FD,$0A,$D7,$FD,$02,$DE,$66
	.dc.b	$14,$97,$FE,$92,$78,$7C,$FB,$E3,$69,$27,$F4,$71,$97,$F4,$BF,$04
	.dc.b	$F8,$17,$A5,$86,$E7,$FD,$C6,$5F,$04,$F5,$F5,$7F,$A3,$CB,$F5,$F7
	.dc.b	$FA,$F6,$F6,$F8,$E3,$47,$ED,$3E,$99,$E8,$70,$6C,$66,$30,$74,$EB
	.dc.b	$FD,$9C,$7F,$A9,$7C,$87,$C0,$FF,$BD,$FC,$2F,$F8,$01,$18,$9F,$F5
	.dc.b	$55,$70,$7F,$DC,$CB,$C1,$E1,$C3,$FE,$FF,$8D,$5A,$31,$E1,$6A,$C2
	.dc.b	$B9,$3F,$EB,$7F,$7D,$EA,$EE,$19,$3E,$07,$0D,$8E,$13,$BA,$7A,$B6
	.dc.b	$24,$F5,$8D,$7A,$BB,$64,$78,$63,$96,$FD,$BE,$06,$4F,$96,$3B,$7A
	.dc.b	$DA,$8C,$1E,$97,$C5,$EA,$B8,$3F,$94,$E6,$8C,$57,$F8,$CE,$31,$3A
	.dc.b	$9C,$CF,$F9,$02,$5E,$B3,$E1,$39,$B6,$79,$35,$E0,$D7,$FC,$81,$F0
	.dc.b	$5F,$8C,$4D,$23,$FC,$FB,$AB,$EC,$3E,$9E,$4F,$F2,$8C,$63,$2F,$D7
	.dc.b	$57,$DA,$F8,$2F,$B5,$7F,$70,$FE,$8B,$3F,$B5,$E5,$E1,$87,$F9,$69
	.dc.b	$B3,$BC,$2B,$F8,$48,$AF,$E3,$67,$FD,$F4,$D3,$C8,$7E,$2B,$8D,$F0
	.dc.b	$5C,$39,$F8,$F9,$F2,$F8,$C5,$BE,$27,$E4,$BC,$FE,$E6,$73,$FE,$18
	.dc.b	$F9,$17,$2D,$3C,$F9,$3E,$0B,$18,$FE,$72,$BE,$4E,$1F,$81,$E5,$E4
	.dc.b	$69,$F0,$C7,$EE,$27,$F6,$5F,$05,$F0,$BC,$6F,$93,$1F,$2F,$14,$E7
	.dc.b	$82,$4F,$73,$E8,$57,$F6,$AB,$F0,$BC,$B0,$FD,$7B,$99,$FA,$DF,$02
	.dc.b	$F8,$2F,$A2,$BE,$0A,$C6,$19,$3F,$E0,$29,$D3,$63,$F9,$74,$62,$FF
	.dc.b	$D1,$D9,$BE,$0B,$F9,$93,$6F,$FC,$16,$D9,$DC,$E8,$FC,$1F,$BC,$F8
	.dc.b	$2F,$9F,$18,$65,$FF,$85,$27,$4D,$8F,$E8,$11,$8A,$FF,$50,$2F,$82
	.dc.b	$C9,$2B,$93,$67,$F0,$B3,$FC,$2A,$7E,$74,$7F,$18,$8E,$51,$88,$46
	.dc.b	$DE,$3E,$07,$88,$93,$E7,$7C,$BC,$72,$3D,$4D,$9F,$3B,$F0,$BB,$FE
	.dc.b	$1B,$67,$C1,$69,$F7,$92,$3E,$23,$F0,$BB,$E4,$7C,$76,$DF,$C4,$7E
	.dc.b	$7F,$C1,$7D,$06,$CF,$68,$F9,$64,$7E,$83,$F3,$BE,$06,$47,$D3,$6D
	.dc.b	$7D,$07,$ED,$ED,$F7,$1B,$3E,$A8,$FA,$E4,$7E,$E3,$F6,$BE,$02,$47
	.dc.b	$DF,$6D,$7D,$C7,$F0,$FF,$01,$F8,$86,$CF,$C6,$8F,$C1,$91,$FC,$43
	.dc.b	$F8,$6F,$7E,$47,$E2,$DB,$5F,$88,$7F,$27,$EF,$FE,$51,$B3,$F4,$A3
	.dc.b	$F1,$E4,$7F,$28,$FF,$83,$FD,$CF,$DE,$7B,$9F,$BD,$7F,$E0,$03,$67
	.dc.b	$FE,$00,$5F,$F1,$4E,$84,$E6,$30,$F2,$9F,$C8,$4E,$9B,$5D,$1A,$74
	.dc.b	$E9,$FE,$07,$5C,$DB,$19,$3F,$84,$CF,$1B,$F1,$36,$7F,$85,$B6,$76
	.dc.b	$71,$9F,$E1,$B5,$D6,$3F,$89,$BE,$78,$F4,$C7,$91,$A4,$AF,$1B,$E0
	.dc.b	$8E,$8A,$FD,$CE,$4E,$11,$D8,$F8,$75,$E1,$8F,$1C,$60,$FF,$4C,$CF
	.dc.b	$28,$FE,$AC,$75,$9F,$F2,$21,$27,$FD,$93,$5F,$EE,$D8,$FF,$6B,$12
	.dc.b	$32,$7F,$BB,$DC,$4F,$FB,$24,$FF,$7B,$38,$9F,$F6,$44,$62,$78,$BF
	.dc.b	$EE,$AF,$73,$51,$FE,$C9,$33,$1F,$EB,$E3,$FD,$EA,$E2,$FF,$DF,$F1
	.dc.b	$FE,$C7,$8C,$5F,$FB,$AE,$B9,$CE,$4F,$25,$F8,$8F,$F7,$ED,$70,$3A
	.dc.b	$4F,$09,$F0,$4E,$E7,$C9,$1E,$23,$9B,$F4,$5A,$77,$8E,$0F,$B6,$3B
	.dc.b	$3A,$3A,$BD,$23,$C7,$19,$BF,$6B,$A3,$A4,$F7,$9E,$B1,$92,$4F,$01
	.dc.b	$5E,$B6,$D7,$B8,$FC,$24,$F7,$9D,$1F,$26,$33,$5D,$AB,$8D,$79,$6F
	.dc.b	$4D,$AE,$8D,$3A,$3F,$02,$B9,$5F,$68,$DC,$7C,$03,$B5,$74,$6C,$68
	.dc.b	$F8,$C7,$C1,$2D,$7D,$D3,$F3,$77,$E6,$6D,$FF,$A7,$C9,$3F,$A2,$6C
	.dc.b	$7E,$88,$FC,$71,$C2,$3F,$86,$FB,$9F,$E1,$39,$FF,$0A,$36,$32,$7E
	.dc.b	$4D,$C4,$FF,$19,$6D,$FF,$84,$9B,$3F,$E1,$66,$CF,$F8,$51,$AF,$EF
	.dc.b	$DB,$66,$92,$7C,$04,$9F,$C4,$46,$5B,$19,$AF,$B5,$39,$6C,$64,$F9
	.dc.b	$73,$E1,$70,$FE,$40,$F3,$38,$7F,$1F,$3C,$E3,$8D,$79,$DD,$3F,$88
	.dc.b	$AF,$85,$18,$BF,$96,$D3,$E5,$DF,$A0,$93,$E6,$6C,$76,$BF,$04,$79
	.dc.b	$4F,$41,$A3,$17,$E3,$70,$7A,$27,$C0,$FD,$76,$D7,$9A,$37,$3E,$99
	.dc.b	$FB,$06,$8B,$1A,$BF,$5B,$63,$D7,$5C,$C8,$FA,$4C,$9C,$9F,$71,$A8
	.dc.b	$F9,$E7,$8A,$FA,$C6,$A7,$81,$27,$DF,$3A,$8C,$47,$BC,$F7,$12,$79
	.dc.b	$3D,$8F,$D4,$CF,$F9,$E1,$F6,$98,$9F,$E9,$31,$FE,$96,$F4,$3F,$DB
	.dc.b	$31,$EA,$9F,$F6,$DC,$61,$ED,$1F,$EA,$38,$C5,$7B,$65,$18,$9F,$3B
	.dc.b	$63,$E0,$CF,$FA,$2A,$31,$1F,$7A,$30,$FB,$CF,$09,$4E,$A5,$8E,$38
	.dc.b	$7E,$0E,$78,$D7,$A6,$B6,$DA,$DD,$FB,$E6,$4F,$F9,$7E,$72,$D8,$C9
	.dc.b	$FE,$AC,$8C,$B4,$CD,$78,$1F,$8A,$7A,$27,$91,$1C,$9E,$43,$93,$98
	.dc.b	$F9,$53,$F1,$09,$3E,$F3,$C6,$7F,$7E,$7E,$A3,$4F,$BC,$78,$4F,$E4
	.dc.b	$4F,$3F,$8E,$F3,$D7,$D9,$7F,$01,$E3,$70,$AF,$B2,$DA,$34,$C7,$89
	.dc.b	$2F,$F9,$75,$7E,$02,$30,$F8,$3E,$3B,$91,$27,$76,$8E,$DE,$2C,$9C
	.dc.b	$B6,$BD,$3E,$87,$8C,$93,$C9,$F4,$7E,$7A,$8A,$43,$CA,$E5,$F0,$FC
	.dc.b	$07,$B4,$8E,$0F,$2F,$17,$C1,$23,$83,$9F,$C7,$F8,$08,$E0,$F6,$F1
	.dc.b	$7C,$92,$38,$3E,$0F,$0F,$CD,$23,$83,$CF,$F1,$FE,$C2,$38,$3E,$4F
	.dc.b	$BF,$F5,$48,$E0,$F9,$BE,$FF,$D9,$23,$83,$E1,$FB,$7F,$20,$8E,$0F
	.dc.b	$AB,$EF,$FE,$04,$8E,$0F,$B3,$F0,$1F,$85,$23,$83,$E7,$FB,$FF,$50
	.dc.b	$8E,$0F,$F2,$BD,$FE,$3F,$A4,$D2,$47,$F9,$72,$F4,$47,$07,$DC,$F8
	.dc.b	$DF,$C0,$91,$C1,$F9,$4F,$80,$60,$F8,$0C,$37,$97,$FC,$0D,$2D,$1E
	.dc.b	$86,$AF,$F3,$BF,$36,$F9,$0F,$CF,$E0,$8C,$3A,$8C,$C6,$23,$F6,$3F
	.dc.b	$04,$7F,$87,$35,$FE,$18,$DF,$03,$F3,$51,$BF,$82,$7A,$D9,$DB,$84
	.dc.b	$62,$7F,$DE,$CD,$55,$C9,$46,$63,$0F,$FB,$B6,$3F,$DF,$DD,$9F,$BE
	.dc.b	$7F,$E0,$07,$90,$F8,$2F,$04,$7A,$63,$0F,$0F,$0F,$2A,$9E,$1E,$1E
	.dc.b	$91,$E4,$CB,$E0,$46,$32,$F3,$3C,$27,$E3,$47,$A7,$DF,$7C,$0F,$6C
	.dc.b	$9F,$05,$4F,$0C,$EA,$F8,$E3,$F3,$61,$E9,$EB,$09,$5A,$3E,$A4,$E9
	.dc.b	$C0,$5E,$9F,$6A,$93,$A9,$CB,$78,$3C,$DF,$05,$23,$06,$FD,$3F,$3C
	.dc.b	$C8,$C1,$CF,$D3,$FB,$2F,$F9,$66,$D1,$88,$FF,$5E,$C6,$36,$FE,$62
	.dc.b	$D8,$E1,$F0,$2F,$82,$CB,$63,$31,$83,$F3,$6B,$E0,$A9,$1F,$CF,$BF
	.dc.b	$A0,$EB,$FC,$AE,$1F,$E0,$B6,$F0,$7F,$9B,$E7,$F3,$DF,$05,$F8,$3B
	.dc.b	$FC,$F5,$69,$FD,$07,$35,$7F,$48,$D7,$F6,$37,$F4,$9F,$EA,$2D,$E6
	.dc.b	$D1,$C7,$95,$F0,$58,$9E,$1F,$05,$CC,$C5,$FE,$44,$FD,$83,$DF,$CF
	.dc.b	$93,$C0,$F0,$3E,$5D,$7F,$03,$1F,$84,$6D,$73,$18,$3E,$0B,$87,$3F
	.dc.b	$1E,$49,$87,$18,$B7,$C6,$DF,$05,$F0,$FC,$8F,$83,$39,$8C,$70,$4C
	.dc.b	$3F,$2B,$5F,$12,$BC,$4F,$0C,$FF,$80,$39,$7F,$81,$60,$F7,$FE,$41
	.dc.b	$F4,$7D,$DF,$E4,$98,$7F,$00,$8C,$BF,$DF,$27,$83,$63,$FC,$0D,$18
	.dc.b	$C3,$FD,$5D,$83,$D1,$FE,$78,$4B,$FF,$02,$C6,$5B,$EB,$FB,$CF,$81
	.dc.b	$36,$3F,$DB,$35,$FE,$C0,$9F,$E8,$C4,$8F,$EF,$7F,$1B,$FE,$00,$23
	.dc.b	$83,$FB,$FF,$1F,$F8,$10,$8E,$0F,$F0,$1F,$C1,$FF,$82,$08,$E0,$FF
	.dc.b	$03,$FC,$1F,$F8,$30,$8E,$0F,$F0,$5F,$91,$FE,$10,$23,$83,$FC,$1F
	.dc.b	$E1,$FF,$85,$08,$E0,$FF,$09,$F2,$FF,$86,$08,$E0,$FF,$0B,$FC,$87
	.dc.b	$F8,$70,$8E,$0F,$F0,$DF,$85,$FE,$20,$23,$83,$FC,$3F,$F2,$1F,$E2
	.dc.b	$42,$38,$3F,$C4,$7F,$21,$FE,$28,$23,$83,$FC,$4F,$CF,$FE,$2C,$23
	.dc.b	$83,$FC,$57,$E4,$7F,$8C,$08,$E0,$FF,$17,$FC,$7F,$F8,$D0,$8E,$0F
	.dc.b	$F1,$9F,$37,$F8,$E0,$8E,$0F,$F2,$5E,$5F,$94,$E3,$FF,$7D,$35,$FE
	.dc.b	$D0,$1C,$3F,$DA,$DA,$FF,$6F,$5F,$0D,$FE,$5D,$6C,$F6,$8C,$19,$F8
	.dc.b	$69,$1E,$26,$D7,$8B,$CA,$FE,$31,$7F,$D5,$2B,$FA,$3B,$FD,$39,$B1
	.dc.b	$A3,$C0,$47,$DA,$D9,$FD,$CF,$C5,$6E,$7F,$7A,$D8,$CC,$74,$32,$57
	.dc.b	$27,$F1,$3A,$F2,$38,$EC,$FF,$01,$46,$1F,$E0,$B5,$D5,$B5,$CD,$B5
	.dc.b	$CE,$FF,$8A,$56,$EB,$FC,$31,$D4,$8C,$57,$F8,$76,$31,$3E,$68,$F1
	.dc.b	$3F,$56,$B9,$B9,$BE,$E7,$9A,$33,$3C,$0E,$97,$D9,$5F,$67,$46,$23
	.dc.b	$51,$9A,$EB,$7E,$89,$C9,$DA,$33,$5F,$87,$BC,$56,$CF,$31,$E9,$2C
	.dc.b	$79,$1F,$E3,$B5,$FE,$5B,$23,$83,$F9,$BF,$B1,$D2,$7E,$33,$5F,$5B
	.dc.b	$70,$34,$93,$E0,$9D,$B6,$76,$D9,$DC,$FC,$49,$CB,$63,$38,$7C,$08
	.dc.b	$E6,$7F,$4A,$8F,$EA,$46,$AB,$EF,$B8,$7B,$1E,$C2,$47,$56,$CF,$F5
	.dc.b	$29,$E5,$3F,$02,$72,$7C,$D7,$C0,$E1,$D1,$CC,$C9,$27,$D1,$5E,$C7
	.dc.b	$B4,$F1,$9F,$C1,$36,$74,$FA,$8F,$4C,$73,$70,$F7,$9E,$6E,$4F,$BC
	.dc.b	$53,$E3,$BA,$3C,$75,$F1,$63,$55,$F9,$1A,$EB,$5B,$69,$F0,$5C,$4F
	.dc.b	$C0,$6B,$EA,$39,$D7,$58,$EF,$3E,$D8,$F9,$B3,$E4,$8E,$AF,$A2,$79
	.dc.b	$D7,$C2,$8E,$D3,$E1,$36,$F1,$32,$71,$73,$1C,$63,$47,$96,$7A,$D7
	.dc.b	$96,$BA,$4F,$23,$75,$C4,$F8,$8E,$5C,$4F,$A9,$B3,$F5,$AB,$AB,$4F
	.dc.b	$1C,$EC,$F0,$57,$9C,$E8,$E1,$F3,$47,$03,$FC,$27,$1F,$ED,$12,$38
	.dc.b	$24,$7F,$87,$23,$2D,$8F,$F1,$8F,$AF,$D6,$FF,$DB,$31,$F8,$F8,$C1
	.dc.b	$1C,$12,$7F,$0B,$19,$6C,$66,$FE,$BC,$E5,$B1,$93,$E9,$C6,$5A,$66
	.dc.b	$7E,$64,$65,$B1,$97,$E9,$C6,$5B,$19,$AF,$AD,$39,$6C,$66,$FF,$CA
	.dc.b	$D3,$96,$C6,$4F,$8A,$5A,$FF,$2A,$CE,$E3,$31,$F0,$09,$7E,$E6,$C7
	.dc.b	$B1,$A7,$A8,$A7,$A0,$B7,$E6,$6D,$F9,$1B,$3E,$0A,$EC,$47,$B3,$FE
	.dc.b	$68,$BE,$0E,$67,$FA,$E4,$62,$76,$7F,$62,$71,$1E,$59,$E5,$3D,$CD
	.dc.b	$C7,$27,$0E,$DE,$46,$1F,$64,$72,$7E,$4C,$7D,$59,$FF,$7E,$47,$03
	.dc.b	$FC,$E2,$E1,$EC,$62,$3F,$D0,$FF,$15,$E6,$8F,$ED,$2D,$C0,$D2,$4F
	.dc.b	$8E,$32,$F2,$24,$F9,$0F,$F4,$8C,$FF,$A6,$49,$1C,$5B,$1A,$8C,$1C
	.dc.b	$9F,$F1,$DC,$F6,$7F,$D1,$73,$EB,$7C,$F3,$FE,$8A,$8F,$F5,$37,$B1
	.dc.b	$D3,$0F,$F4,$FB,$67,$FC,$2A,$DA,$FF,$0A,$35,$FF,$04,$B6,$CF,$49
	.dc.b	$FF,$56,$B9,$31,$3F,$EA,$E7,$11,$A2,$47,$B4,$D3,$87,$DB,$1E,$B9
	.dc.b	$EF,$3F,$EA,$48,$C1,$FE,$C1,$F6,$7B,$DF,$FA,$E9,$FF,$27,$F8,$7F
	.dc.b	$E5,$92,$57,$F9,$99,$FF,$2E,$78,$92,$3F,$CD,$2D,$C7,$FC,$D3,$E7
	.dc.b	$FF,$98,$49,$3F,$E6,$E7,$FC,$C9,$E7,$C8,$FF,$38,$B6,$7F,$CE,$3E
	.dc.b	$77,$F9,$A4,$93,$FE,$76,$7F,$CD,$9F,$11,$23,$FC,$F2,$D9,$FF,$3C
	.dc.b	$FC,$47,$F9,$C4,$93,$FE,$7E,$7F,$CE,$9F,$1F,$23,$FD,$02,$D9,$FF
	.dc.b	$40,$F9,$1F,$E7,$92,$4F,$FA,$19,$FF,$3E,$7C,$4C,$8F,$F4,$4B,$67
	.dc.b	$FD,$13,$F1,$FF,$E8,$12,$4F,$FA,$39,$FF,$42,$7C,$54,$8F,$F4,$8B
	.dc.b	$67,$FD,$23,$F1,$5F,$E8,$92,$4F,$FA,$59,$FF,$46,$7C,$7C,$8F,$F4
	.dc.b	$CB,$67,$FD,$33,$F1,$7F,$E9,$12,$4F,$FA,$79,$FF,$4A,$7C,$7C,$8F
	.dc.b	$F5,$0B,$67,$FD,$3E,$3F,$3E,$3F,$A5,$6C,$EA,$34,$64,$D7,$A5,$23
	.dc.b	$52,$8C,$4F,$6F,$A3,$6F,$FB,$16,$DF,$56,$CE,$E7,$F6,$DF,$20,$F8
	.dc.b	$0A,$4B,$F2,$46,$5F,$37,$C2,$3E,$15,$F0,$98,$6E,$3F,$02,$5F,$0B
	.dc.b	$E7,$8C,$4E,$9F,$F6,$1E,$AF,$77,$E1,$47,$C7,$6C,$66,$3E,$0C,$6B
	.dc.b	$E4,$9E,$95,$24,$FC,$B8,$CB,$F3,$7E,$3F,$DC,$FD,$2F,$09,$AF,$D4
	.dc.b	$84,$E9,$4E,$9F,$B3,$19,$8C,$43,$E1,$9E,$0F,$39,$CC,$62,$7C,$17
	.dc.b	$DE,$31,$1D,$A3,$15,$C5,$D7,$83,$67,$F1,$3F,$0D,$F0,$DE,$07,$E3
	.dc.b	$3B,$9E,$8E,$E1,$18,$7C,$27,$2D,$B5,$4F,$81,$B1,$27,$62,$74,$E3
	.dc.b	$E1,$5E,$77,$7C,$1E,$C5,$3C,$53,$AA,$E1,$F0,$D8,$B9,$18,$3A,$79
	.dc.b	$3C,$26,$46,$0F,$06,$FD,$FD,$99,$18,$3C,$9B,$F8,$26,$46,$0F,$47
	.dc.b	$95,$F7,$1F,$DA,$E4,$8C,$47,$E0,$C4,$67,$8F,$F6,$CD,$8E,$3E,$53
	.dc.b	$E1,$B2,$D8,$CC,$60,$FC,$EA,$F8,$6A,$47,$F4,$2F,$E8,$BC,$3F,$C8
	.dc.b	$61,$FE,$08,$6F,$37,$F0,$C9,$FE,$17,$F0,$DF,$56,$FF,$35,$5A,$7F
	.dc.b	$39,$CB,$5F,$E2,$2D,$7F,$85,$5F,$DE,$7F,$64,$DE,$6D,$1C,$79,$5F
	.dc.b	$01,$89,$E1,$F0,$DC,$CC,$5F,$E5,$0F,$E3,$EF,$7F,$86,$93,$C0,$F0
	.dc.b	$3E,$5D,$5C,$D4,$F0,$77,$5A,$FC,$9C,$FE,$B9,$BC,$99,$C7,$C1,$39
	.dc.b	$7C,$59,$BD,$DF,$85,$EA,$7C,$09,$CC,$63,$83,$3C,$13,$2F,$D0,$DE
	.dc.b	$F1,$FD,$FB,$F1,$B0,$EB,$EE,$7F,$87,$F2,$34,$97,$FC,$F6,$32,$FE
	.dc.b	$B3,$E1,$9E,$EF,$CD,$AF,$AD,$18,$9F,$0F,$C2,$FA,$0F,$4F,$C3,$58
	.dc.b	$C9,$FD,$27,$24,$FF,$85,$C9,$19,$F8,$BF,$F0,$D1,$1C,$1F,$D4,$7E
	.dc.b	$3B,$EE,$B7,$9B,$FC,$75,$18,$D7,$FA,$DB,$5F,$D3,$2F,$87,$D9,$2B
	.dc.b	$A3,$FD,$7F,$A2,$47,$F5,$A6,$D7,$4F,$86,$EE,$49,$F2,$3E,$2F,$12
	.dc.b	$47,$9D,$B3,$FD,$BB,$CA,$75,$FF,$6D,$25,$7F,$68,$7D,$7F,$05,$23
	.dc.b	$E0,$36,$BD,$BE,$27,$C0,$24,$FC,$57,$FB,$C7,$6C,$8F,$EE,$6D,$9F
	.dc.b	$8B,$F0,$BF,$20,$93,$F3,$5F,$97,$D7,$23,$E8,$36,$7D,$5E,$47,$F8
	.dc.b	$14,$93,$FE,$00,$7E,$9F,$C2,$C8,$FB,$0D,$9F,$AB,$D1,$F6,$09,$3F
	.dc.b	$75,$FF,$08,$7C,$94,$8F,$F0,$5B,$67,$FC,$2F,$A7,$E4,$5B,$D5,$FE
	.dc.b	$7B,$8C,$47,$F2,$56,$BF,$DA,$57,$C3,$6C,$97,$FE,$1D,$8C,$BF,$E1
	.dc.b	$AF,$8F,$FF,$11,$11,$C1,$F9,$BF,$87,$FD,$B9,$FC,$66,$7F,$A7,$C7
	.dc.b	$ED,$23,$2E,$C9,$7F,$E3,$0E,$B4,$FF,$9E,$4F,$F8,$00,$49,$FD,$EF
	.dc.b	$C3,$F2,$AF,$E3,$14,$8C,$78,$1F,$C0,$1B,$5F,$E0,$79,$D3,$E3,$39
	.dc.b	$96,$B9,$CF,$07,$FC,$83,$3C,$1E,$F7,$E3,$77,$5C,$6F,$C0,$D3,$C1
	.dc.b	$7C,$E5,$7C,$EB,$71,$C8,$EB,$5E,$47,$11,$A3,$F8,$DD,$FF,$24,$B9
	.dc.b	$5B,$9C,$B4,$F0,$96,$BD,$75,$C1,$EB,$1A,$7C,$26,$0F,$83,$5E,$FA
	.dc.b	$F1,$4E,$E7,$FD,$97,$3F,$EA,$F2,$47,$F1,$D7,$FD,$C0,$47,$FD,$BC
	.dc.b	$D8,$FF,$6E,$4F,$FC,$01,$7F,$DE,$73,$FE,$FA,$69,$FE,$F9,$21,$83
	.dc.b	$E5,$FC,$3F,$39,$FF,$7D,$12,$7F,$DF,$0F,$5C,$13,$B9,$C9,$2B,$FD
	.dc.b	$F7,$5B,$9D,$46,$27,$4F,$81,$F9,$8D,$3D,$6E,$AB,$EF,$36,$79,$CE
	.dc.b	$EB,$CC,$F5,$3D,$B5,$E6,$AF,$8E,$7F,$5C,$AF,$EC,$67,$28,$D4,$F8
	.dc.b	$0E,$47,$C5,$73,$5D,$A3,$13,$C4,$E7,$5D,$A3,$91,$E5,$7D,$57,$A8
	.dc.b	$CD,$F4,$9F,$AC,$D8,$F0,$BF,$01,$B5,$DE,$7D,$47,$7B,$F2,$57,$1A
	.dc.b	$F0,$1F,$0A,$B8,$3E,$07,$F3,$13,$C1,$CB,$6B,$E9,$D7,$32,$4F,$08
	.dc.b	$FA,$47,$84,$B5,$EC,$BF,$75,$F3,$8F,$D1,$1E,$42,$5F,$59,$DD,$7F
	.dc.b	$0D,$BF,$F4,$6D,$58,$FE,$FD,$18,$8F,$E1,$6D,$7F,$10,$7F,$96,$6B
	.dc.b	$ED,$B6,$35,$18,$8F,$C6,$7C,$3F,$81,$FE,$2A,$D7,$F8,$6B,$60,$D2
	.dc.b	$57,$D2,$9F,$E5,$31,$87,$EE,$B7,$0F,$AA,$D9,$F0,$9C,$63,$FC,$5B
	.dc.b	$19,$9F,$C0,$B5,$F8,$ED,$AF,$A8,$DB,$FF,$1E,$B6,$FF,$C7,$2D,$AF
	.dc.b	$2C,$77,$3F,$9D,$46,$6F,$C3,$19,$9F,$86,$D7,$E1,$B6,$FE,$1B,$6B
	.dc.b	$93,$D0,$C1,$27,$E1,$B6,$BE,$1B,$4F,$87,$1A,$24,$69,$B1,$A8,$C1
	.dc.b	$1C,$12,$3E,$4B,$F0,$88,$F2,$3D,$A7,$C4,$30,$7B,$8C,$47,$57,$67
	.dc.b	$E3,$67,$F1,$64,$9F,$F2,$93,$FE,$6E,$F8,$7F,$A7,$1F,$E6,$A6,$CF
	.dc.b	$F6,$E6,$D7,$F9,$B1,$B3,$FD,$B5,$B3,$FD,$A1,$E9,$1C,$CF,$5C,$F1
	.dc.b	$71,$3D,$5C,$9C,$CC,$C6,$E3,$DB,$19,$78,$1F,$E0,$99,$FD,$39,$23
	.dc.b	$FC,$DC,$FF,$A2,$BE,$1F,$C7,$3F,$E0,$86,$D7,$FA,$19,$B5,$FE,$07
	.dc.b	$6C,$FF,$80,$9A,$FF,$7A,$7A,$47,$FA,$41,$CE,$07,$33,$DE,$7D,$67
	.dc.b	$53,$81,$1F,$5D,$78,$E3,$11,$FC,$16,$7F,$81,$92,$3F,$D1,$4F,$FB
	.dc.b	$1F,$B9,$F1,$9E,$0B,$52,$E5,$6A,$3C,$13,$A9,$CD,$71,$9E,$10,$AE
	.dc.b	$0F,$AC,$FF,$64,$38,$AF,$44,$62,$BC,$84,$AE,$E7,$4A,$E0,$D8,$D4
	.dc.b	$F5,$AF,$23,$E1,$8C,$C6,$2B,$8C,$FD,$5A,$F7,$4E,$6B,$9C,$FF,$A8
	.dc.b	$1F,$F5,$29,$27,$FD,$54,$FF,$BB,$FA,$DE,$1F,$69,$F8,$31,$87,$A9
	.dc.b	$FE,$69,$F2,$3F,$CD,$C4,$7F,$CD,$0F,$F9,$C7,$D1,$91,$FE,$6B,$6E
	.dc.b	$1F,$EA,$1F,$17,$FA,$15,$FF,$B0,$AB,$E7,$CF,$82,$72,$7F,$9E,$FC
	.dc.b	$F9,$3F,$E7,$56,$CF,$FA,$0B,$E2,$3F,$D0,$04,$9F,$F3,$D3,$DB,$C7
	.dc.b	$91,$DA,$7F,$09,$82,$73,$1F,$E8,$CF,$7F,$FD,$14,$4A,$FF,$42,$3F
	.dc.b	$E8,$DF,$2A,$47,$FA,$1D,$B3,$FE,$BD,$F8,$EF,$69,$27,$C1,$0C,$13
	.dc.b	$FE,$97,$F8,$79,$5F,$E9,$16,$D7,$FA,$7B,$E1,$FF,$D3,$84,$9F,$F4
	.dc.b	$B3,$F2,$BA,$A4,$76,$C3,$FD,$16,$D8,$FF,$52,$7C,$34,$8F,$F4,$F3
	.dc.b	$6F,$FD,$57,$F0,$DF,$EA,$92,$4F,$FA,$8D,$FA,$FD,$92,$3B,$36,$7F
	.dc.b	$DB,$87,$DC,$24,$78,$DF,$BF,$E8,$C8,$F3,$36,$3F,$DC,$9D,$BF,$84
	.dc.b	$24,$FA,$DF,$F0,$47,$C1,$BE,$12,$92,$7E,$6C,$65,$F9,$FF,$06,$F8
	.dc.b	$27,$C2,$61,$B8,$FE,$0E,$5F,$08,$CF,$57,$F6,$3E,$35,$E9,$FD,$AC
	.dc.b	$11,$87,$51,$98,$C4,$7E,$3F,$E1,$5E,$D7,$16,$C7,$E1,$1A,$EE,$35
	.dc.b	$E9,$C8,$D3,$5F,$CD,$7C,$23,$C1,$FE,$2F,$E2,$7F,$08,$F8,$5F,$8D
	.dc.b	$5F,$B8,$3F,$91,$38,$51,$FC,$3A,$B9,$46,$C3,$87,$7C,$3F,$C9,$23
	.dc.b	$13,$C6,$FC,$33,$9F,$0E,$DF,$F3,$1F,$49,$EE,$62,$1E,$03,$E1,$70
	.dc.b	$D8,$F1,$57,$06,$FB,$5C,$63,$11,$E4,$3B,$5E,$67,$B7,$C1,$3E,$17
	.dc.b	$AF,$B9,$2F,$DD,$39,$6F,$C2,$F0,$A9,$18,$37,$F0,$3D,$26,$46,$0E
	.dc.b	$7B,$FA,$B8,$E4,$60,$EF,$BF,$92,$64,$60,$F1,$FC,$2F,$D4,$7F,$8B
	.dc.b	$64,$8C,$47,$F2,$68,$C7,$17,$F0,$C6,$C7,$0F,$31,$F0,$B9,$6C,$66
	.dc.b	$30,$7E,$6D,$7C,$2D,$23,$F9,$F7,$F4,$1D,$BF,$D9,$F0,$FF,$3F,$B7
	.dc.b	$9B,$F9,$64,$FF,$6F,$F8,$57,$2F,$F3,$CA,$D6,$1F,$DB,$39,$6B,$FD
	.dc.b	$D5,$AF,$EC,$70,$FC,$93,$F8,$86,$F2,$E8,$E3,$CC,$F8,$4C,$4F,$0F
	.dc.b	$83,$E6,$62,$FF,$0C,$7F,$80,$DE,$FF,$07,$27,$81,$E0,$7C,$BB,$39
	.dc.b	$E9,$E0,$ED,$B5,$FA,$79,$FD,$93,$79,$73,$8F,$82,$72,$F8,$BD,$8F
	.dc.b	$73,$77,$F5,$FC,$17,$C4,$9C,$D7,$F8,$09,$F8,$55,$C7,$83,$57,$B7
	.dc.b	$E1,$FC,$0F,$CC,$F8,$57,$C0,$E3,$4F,$DE,$46,$1B,$1F,$EB,$38,$C6
	.dc.b	$1F,$D9,$5A,$FF,$74,$6B,$FE,$24,$C7,$FB,$C3,$FD,$AD,$AF,$FA,$EF
	.dc.b	$D8,$FF,$5C,$9F,$EB,$AF,$5E,$61,$FD,$8A,$BF,$B8,$36,$3F,$D7,$FE
	.dc.b	$B4,$8D,$36,$FF,$BC,$7B,$6F,$85,$FF,$65,$36,$BF,$B6,$B6,$77,$3F
	.dc.b	$E0,$EE,$89,$5F,$D9,$5B,$3F,$E0,$3F,$5B,$FD,$4A,$DF,$7F,$FC,$2A
	.dc.b	$4A,$EB,$E3,$78,$08,$E0,$FE,$E1,$D3,$FE,$1C,$23,$83,$FB,$A7,$BF
	.dc.b	$FE,$20,$23,$83,$D3,$E2,$FB,$08,$E0,$FE,$F9,$D7,$FE,$28,$23,$83
	.dc.b	$FC,$01,$EF,$FF,$8B,$08,$E0,$F8,$BD,$3F,$20,$8E,$0F,$F0,$47,$7F
	.dc.b	$F8,$D0,$8E,$0F,$F0,$67,$A9,$FE,$38,$23,$83,$E8,$F4,$7D,$42,$38
	.dc.b	$3F,$C2,$9F,$07,$FE,$40,$23,$83,$FC,$31,$F2,$3F,$E4,$42,$38,$3F
	.dc.b	$C5,$3F,$04,$F8,$4F,$F3,$2F,$17,$F9,$DA,$31,$3A,$9C,$CF,$FA,$06
	.dc.b	$72,$FD,$FF,$84,$FF,$28,$92,$BF,$C5,$0F,$F9,$53,$E4,$64,$7F,$8A
	.dc.b	$DB,$3F,$9F,$F7,$3F,$6C,$7E,$DF,$DF,$B5,$FD,$A7,$B5,$3E,$2A,$46
	.dc.b	$27,$F4,$53,$98,$C4,$F5,$9F,$F7,$C4,$FF,$03,$F7,$7A,$CF,$F0,$96
	.dc.b	$F8,$9F,$C1,$5B,$5F,$B5,$6C,$FF,$A8,$6D,$18,$8F,$1B,$63,$33,$E9
	.dc.b	$BF,$03,$4F,$19,$2B,$C4,$FB,$EB,$71,$FC,$76,$6D,$F9,$6B,$83,$67
	.dc.b	$89,$1F,$E3,$D3,$C2,$7D,$47,$02,$CE,$DB,$3C,$1F,$11,$23,$47,$C1
	.dc.b	$9E,$8F,$C3,$AA,$FD,$48,$B7,$F0,$A3,$F8,$E4,$FD,$52,$57,$F1,$97
	.dc.b	$FD,$FF,$1F,$2A,$3F,$DF,$2F,$56,$B9,$3F,$DF,$26,$4A,$66,$FF,$9F
	.dc.b	$74,$F4,$BF,$E2,$71,$F1,$CC,$D7,$48,$CB,$F0,$8F,$E6,$93,$C4,$F5
	.dc.b	$9F,$D2,$CF,$C7,$11,$FE,$64,$DA,$FF,$78,$C7,$23,$FD,$EE,$FE,$26
	.dc.b	$3F,$DF,$A4,$8C,$C7,$E7,$23,$FE,$00,$F3,$E6,$39,$9F,$91,$71,$7F
	.dc.b	$D6,$DD,$D7,$AD,$B3,$F1,$2D,$18,$AF,$83,$3D,$0F,$D5,$C7,$4A,$FE
	.dc.b	$A8,$ED,$F5,$46,$EB,$9C,$7A,$1F,$93,$1E,$58,$C4,$FD,$13,$94,$7C
	.dc.b	$18,$C9,$C6,$F3,$1E,$28,$F5,$1A,$7D,$53,$AA,$EF,$1B,$3C,$91,$DE
	.dc.b	$78,$5F,$B8,$F8,$CE,$0F,$5B,$89,$E3,$1E,$06,$9E,$07,$83,$E9,$33
	.dc.b	$1C,$4E,$D1,$F2,$0D,$C7,$DB,$6D,$7C,$88,$E8,$72,$8F,$19,$C5,$C1
	.dc.b	$DE,$7B,$46,$5B,$19,$8C,$1C,$5F,$F8,$03,$3D,$DB,$5F,$C9,$49,$3F
	.dc.b	$E0,$96,$C7,$F8,$45,$B5,$FE,$27,$E5,$FF,$15,$47,$F8,$AD,$B5,$FC
	.dc.b	$D5,$B5,$FE,$33,$6D,$71,$9F,$C3,$46,$1F,$F1,$03,$67,$F9,$63,$5F
	.dc.b	$E7,$AD,$B3,$49,$3D,$09,$3F,$95,$8C,$B6,$33,$5F,$22,$3A,$C6,$0F
	.dc.b	$92,$F8,$9C,$3F,$9B,$3D,$8E,$1F,$C8,$47,$B5,$EC,$47,$B5,$7A,$27
	.dc.b	$83,$F8,$18,$E0,$6A,$BC,$ED,$9D,$DF,$CF,$69,$F3,$CB,$7E,$43,$C0
	.dc.b	$FA,$9F,$69,$83,$D5,$3E,$08,$D5,$F9,$67,$47,$B2,$3E,$A9,$DA,$78
	.dc.b	$47,$28,$CB,$C2,$7D,$91,$89,$F5,$4F,$95,$E2,$F5,$3E,$71,$98,$E0
	.dc.b	$79,$2F,$E0,$46,$4F,$01,$C2,$7E,$83,$F9,$E8,$FF,$00,$12,$7F,$CC
	.dc.b	$8F,$FA,$93,$B5,$F0,$BD,$2D,$4B,$95,$A8,$F9,$D1,$FE,$9E,$8C,$3E
	.dc.b	$30,$E2,$7D,$AD,$9F,$5C,$F4,$AF,$11,$23,$AD,$78,$A7,$51,$E3,$8C
	.dc.b	$47,$69,$E0,$F8,$6B,$B3,$C0,$F3,$47,$B6,$BD,$91,$C5,$E8,$62,$7D
	.dc.b	$71,$A3,$47,$FA,$12,$7F,$D0,$64,$9E,$31,$F1,$FB,$1F,$0B,$E4,$C3
	.dc.b	$CF,$18,$AD,$D7,$49,$CD,$6E,$7C,$B5,$BA,$F4,$C7,$08,$C5,$7C,$2C
	.dc.b	$3D,$F4,$9C,$D7,$2B,$FF,$70,$C6,$27,$E2,$5F,$82,$BA,$56,$8F,$14
	.dc.b	$FA,$DC,$46,$8F,$4D,$F8,$2F,$A5,$78,$6B,$FC,$A5,$1E,$98,$F9,$4D
	.dc.b	$7C,$95,$F3,$5E,$CF,$C4,$FC,$FF,$F2,$2F,$78,$FF,$75,$C6,$1F,$F7
	.dc.b	$6C,$F8,$23,$18,$7A,$E7,$75,$E8,$6B,$CE,$3F,$13,$3C,$23,$F0,$6D
	.dc.b	$9F,$D0,$35,$F6,$47,$94,$FC,$A4,$F2,$69,$C8,$F0,$39,$78,$12,$3D
	.dc.b	$37,$C1,$EF,$15,$57,$BD,$FC,$93,$5E,$D1,$F2,$5D,$BF,$E8,$18,$F4
	.dc.b	$0F,$92,$E8,$E1,$ED,$C8,$F5,$3F,$D1,$0D,$AF,$F5,$0C,$60,$FB,$7E
	.dc.b	$2C,$8F,$06,$7F,$EF,$2C,$7F,$2F,$3E,$1F,$92,$95,$E5,$C9,$18,$AF
	.dc.b	$C0,$62,$F8,$8F,$09,$2F,$FD,$62,$FE,$2B,$C7,$91,$E0,$6D,$7E,$27
	.dc.b	$07,$C0,$7C,$12,$57,$85,$F6,$FA,$32,$3F,$D7,$AD,$AF,$F7,$A7,$93
	.dc.b	$F9,$82,$4F,$B9,$F9,$9F,$05,$23,$D8,$D9,$F5,$FC,$37,$CA,$24,$FF
	.dc.b	$B2,$9F,$D1,$F6,$48,$F9,$2D,$9F,$57,$37,$D9,$24,$FC,$97,$EA,$FC
	.dc.b	$2C,$8F,$F6,$AB,$67,$F5,$5C,$5F,$EB,$C7,$F5,$FD,$BF,$B2,$25,$7D
	.dc.b	$3F,$92,$FC,$31,$1C,$1F,$73,$D1,$FD,$C9,$1C,$1F,$80,$EF,$FD,$E9
	.dc.b	$1C,$1F,$7F,$D5,$FC,$91,$1C,$1F,$97,$F8,$56,$EF,$85,$C3,$73,$FE
	.dc.b	$09,$3F,$73,$1C,$D7,$E7,$FE,$11,$F0,$BF,$A1,$B4,$61,$D4,$66,$31
	.dc.b	$1F,$C3,$F9,$5F,$1D,$C5,$B1,$FA,$B6,$BB,$8E,$DF,$0C,$F4,$69,$27
	.dc.b	$F6,$51,$96,$FC,$8F,$02,$38,$3F,$71,$F0,$CF,$6F,$FA,$9F,$8B,$FD
	.dc.b	$73,$E1,$DE,$96,$38,$FF,$AB,$C6,$30,$FE,$93,$87,$16,$F8,$BB,$7F
	.dc.b	$BC,$46,$56,$1F,$DA,$23,$0D,$9E,$12,$F4,$DE,$E6,$3C,$17,$C3,$D3
	.dc.b	$73,$AA,$FE,$B6,$BD,$AC,$DC,$9C,$C7,$6F,$84,$7C,$3F,$5F,$72,$57
	.dc.b	$F6,$49,$CB,$7E,$1F,$85,$48,$C1,$BF,$82,$E9,$32,$30,$73,$E3,$F5
	.dc.b	$73,$48,$C1,$DF,$7F,$24,$C8,$C1,$E3,$F8,$7F,$92,$FF,$57,$C9,$18
	.dc.b	$8F,$F0,$34,$63,$87,$F9,$A3,$63,$87,$C3,$BE,$1F,$2D,$8C,$C6,$0F
	.dc.b	$CE,$2F,$87,$A4,$7F,$40,$FE,$87,$C0,$FE,$91,$06,$BC,$47,$C3,$FD
	.dc.b	$D7,$FC,$43,$1C,$38,$E7,$4F,$F8,$52,$79,$36,$3A,$F2,$48,$FF,$0C
	.dc.b	$F3,$BC,$4C,$5F,$E3,$A7,$FC,$51,$F0,$7E,$43,$3B,$FF,$8E,$25,$1E
	.dc.b	$8F,$83,$95,$E1,$8B,$3C,$CF,$F2,$07,$37,$F6,$5B,$3F,$D9,$F1,$7C
	.dc.b	$3E,$63,$5C,$93,$1F,$F1,$7D,$F0,$6C,$7F,$6F,$F6,$FD,$F8,$F7,$0E
	.dc.b	$2B,$FC,$9F,$8B,$E1,$38,$70,$71,$E0,$99,$7F,$8F,$30,$E0,$FF,$93
	.dc.b	$9F,$D5,$BF,$E4,$0F,$86,$7C,$17,$F7,$F6,$C6,$7B,$31,$32,$3F,$BF
	.dc.b	$4E,$B4,$FF,$14,$35,$FF,$2B,$35,$FF,$2D,$79,$BF,$3D,$FF,$3C,$B5
	.dc.b	$FF,$3A,$B7,$DD,$37,$92,$5F,$F8,$36,$7E,$F3,$63,$87,$C3,$48,$D3
	.dc.b	$6F,$FC,$79,$F2,$7F,$CD,$A7,$FC,$2C,$D8,$CF,$99,$E3,$8C,$47,$F8
	.dc.b	$7A,$BF,$C3,$44,$AF,$F0,$53,$E9,$F3,$7F,$CE,$B3,$FE,$7A,$6B,$FE
	.dc.b	$97,$6B,$FD,$37,$D4,$FF,$12,$7C,$9C,$BF,$F0,$A3,$67,$FC,$15,$8F
	.dc.b	$F8,$A4,$93,$FE,$19,$7F,$C5,$9E,$2C,$8F,$F0,$E3,$67,$FC,$25,$D3
	.dc.b	$FE,$31,$24,$FF,$88,$5F,$F1,$A7,$B9,$23,$FC,$48,$D9,$FF,$0D,$78
	.dc.b	$BF,$E3,$92,$4F,$F8,$A5,$FF,$1E,$79,$92,$3F,$C5,$8D,$9F,$F1,$17
	.dc.b	$C7,$FF,$90,$49,$3F,$E3,$17,$FC,$89,$D3,$23,$FC,$68,$D9,$FF,$15
	.dc.b	$79,$5F,$E4,$92,$4F,$F8,$E5,$FF,$26,$7C,$7C,$8F,$F1,$E3,$67,$FC
	.dc.b	$65,$F1,$5F,$E5,$12,$4F,$F9,$05,$FC,$87,$E8,$1F,$DF,$E3,$FC,$B0
	.dc.b	$D9,$FF,$41,$35,$FF,$58,$35,$FF,$5B,$AF,$8C,$E6,$49,$E6,$F6,$EA
	.dc.b	$91,$D9,$B5,$DB,$AF,$FC,$DD,$7F,$D4,$AB,$FD,$66,$BC,$DC,$CC,$9F
	.dc.b	$3B,$EA,$23,$83,$8F,$87,$BB,$FE,$15,$7F,$C3,$DB,$E1,$F8,$0F,$0F
	.dc.b	$9B,$6F,$C3,$16,$78,$CD,$F1,$3F,$DB,$F7,$FD,$42,$FC,$FE,$5F,$AB
	.dc.b	$CD,$F5,$B6,$32,$7B,$23,$2F,$BB,$E0,$7E,$09,$F0,$BE,$52,$C6,$4F
	.dc.b	$AB,$1F,$60,$8E,$0F,$8F,$F2,$3C,$33,$F7,$BF,$33,$C9,$93,$F3,$3E
	.dc.b	$52,$D7,$C4,$9B,$A7,$B2,$6D,$7D,$EF,$92,$FA,$BF,$05,$F5,$9B,$1F
	.dc.b	$59,$B8,$7C,$E6,$DA,$7F,$DF,$BE,$2F,$F4,$47,$FD,$DE,$DA,$FA,$BE
	.dc.b	$CF,$FA,$BE,$31,$5C,$64,$23,$83,$83,$2E,$1E,$FE,$07,$F1,$B2,$5F
	.dc.b	$E3,$3E,$4B,$B4,$7E,$41,$FC,$8F,$C1,$49,$FC,$8F,$CA,$5B,$FC,$4C
	.dc.b	$DC,$7F,$03,$36,$BC,$5F,$05,$E6,$7C,$F3,$E9,$B4,$FF,$52,$7D,$6E
	.dc.b	$DF,$EB,$CE,$5D,$9F,$ED,$17,$06,$E3,$A3,$8F,$43,$46,$FD,$4B,$7E
	.dc.b	$23,$CB,$1F,$D3,$E7,$FB,$0D,$FF,$BF,$DA,$FC,$B3,$CF,$1B,$3C,$5E
	.dc.b	$9F,$48,$F3,$BE,$5F,$52,$5F,$79,$F6,$FB,$38,$7D,$B3,$72,$F4,$CD
	.dc.b	$AF,$93,$F2,$5E,$FF,$91,$F8,$0D,$8E,$98,$7C,$A8,$C3,$F2,$1B,$7F
	.dc.b	$3F,$D0,$FE,$CB,$3F,$E0,$96,$C6,$7D,$4F,$F7,$E4,$62,$FF,$AC,$7A
	.dc.b	$78,$33,$36,$3C,$33,$BB,$FE,$B9,$5A,$9C,$BB,$8F,$03,$96,$C6,$88
	.dc.b	$F9,$6B,$85,$FA,$6A,$D7,$F6,$B9,$B5,$E0,$8E,$35,$EB,$69,$EA,$2C
	.dc.b	$7F,$71,$9E,$2F,$FA,$F6,$AD,$FF,$83,$AB,$73,$96,$C6,$63,$D2,$4B
	.dc.b	$F3,$4F,$C4,$BE,$77,$D1,$B1,$C9,$B5,$C6,$3D,$87,$69,$E0,$D3,$81
	.dc.b	$63,$81,$EB,$71,$5E,$28,$F9,$51,$83,$E1,$38,$38,$9E,$D2,$DF,$B6
	.dc.b	$79,$12,$78,$BF,$36,$77,$5E,$66,$CF,$99,$A7,$4B,$F7,$BF,$AB,$BF
	.dc.b	$C9,$B6,$BF,$C5,$04,$AF,$D8,$47,$DD,$2C,$F0,$7D,$67,$18,$FC,$9F
	.dc.b	$A7,$F6,$88,$FD,$86,$C7,$B9,$B8,$7B,$63,$FC,$69,$18,$9F,$79,$FE
	.dc.b	$39,$71,$1E,$39,$FD,$64,$62,$3E,$54,$62,$7C,$04,$8F,$03,$63,$C1
	.dc.b	$0A,$57,$37,$F5,$91,$98,$C5,$FE,$D9,$B7,$F0,$0E,$87,$A2,$7D,$6E
	.dc.b	$1F,$43,$EB,$30,$FA,$1C,$3E,$62,$5F,$9A,$FC,$B3,$C1,$F2,$98,$9D
	.dc.b	$C7,$85,$F4,$3B,$BF,$43,$4F,$44,$70,$3C,$A7,$A0,$C1,$E5,$31,$5E
	.dc.b	$3B,$FA,$1B,$BE,$37,$FA,$21,$FD,$17,$E3,$2E,$1F,$C9,$A7,$4D,$9D
	.dc.b	$34,$D7,$C5,$FF,$4D,$F8,$AF,$E9,$CD,$8C,$9C,$E3,$2D,$3F,$91,$FA
	.dc.b	$7F,$CA,$8F,$E5,$7D,$36,$32,$7E,$73,$4F,$C0,$4F,$FA,$1C,$91,$9D
	.dc.b	$91,$FC,$D3,$D3,$E5,$5F,$CE,$5F,$EC,$5F,$15,$2F,$DD,$A5,$AF,$ED
	.dc.b	$33,$77,$FE,$CB,$36,$BF,$BA,$FC,$4F,$F6,$BF,$90,$FE,$D8,$D8,$F9
	.dc.b	$0D,$C3,$FB,$9B,$6F,$FA,$67,$A5,$FD,$20,$FE,$93,$E9,$DB,$FE,$8B
	.dc.b	$1F,$EA,$32,$3E,$D6,$DF,$F5,$3F,$4F,$83,$67,$EC,$7C,$87,$F7,$FF
	.dc.b	$90,$FF,$00,$36,$32,$7D,$D8,$CB,$4F,$C0,$7C,$3F,$F8,$1C,$FF,$04
	.dc.b	$7C,$95,$8C,$92,$FF,$C1,$F1,$97,$FD,$DD,$F6,$3F,$E6,$F9,$FE,$DE
	.dc.b	$D8,$CF,$AB,$FE,$33,$D7,$FC,$39,$39,$6F,$AD,$FE,$DB,$8E,$15,$FE
	.dc.b	$0B,$6B,$FE,$37,$6B,$FE,$3C,$79,$5E,$7D,$6E,$CE,$EA,$D6,$E7,$FC
	.dc.b	$49,$F2,$78,$87,$E8,$3F,$E4,$FB,$FD,$14,$E6,$31,$3F,$E2,$E8,$C4
	.dc.b	$F2,$3F,$C1,$B3,$AC,$7F,$C7,$0B,$E5,$39,$1C,$FE,$52,$D7,$08,$B8
	.dc.b	$2F,$8D,$61,$F8,$FE,$A7,$9F,$43,$7D,$7C,$33,$1F,$D4,$4E,$9B,$1F
	.dc.b	$34,$8F,$D6,$F5,$B8,$63,$F6,$9F,$81,$F1,$FF,$E5,$18,$FF,$33,$68
	.dc.b	$8C,$47,$F9,$B2,$31,$5F,$B7,$6C,$7F,$54,$5F,$25,$C2,$3E,$7B,$63
	.dc.b	$33,$C6,$31,$F0,$B2,$3A,$36,$BA,$79,$1C,$49,$3E,$17,$A7,$91,$23
	.dc.b	$C8,$D9,$F2,$79,$1E,$12,$4F,$A5,$F2,$7C,$2C,$8F,$63,$67,$D9,$F0
	.dc.b	$BE,$92,$4F,$C1,$7D,$9F,$0B,$23,$E2,$36,$7E,$27,$8F,$F0,$49,$3F
	.dc.b	$25,$F8,$9E,$34,$8F,$98,$D9,$F9,$9E,$37,$C9,$24,$FD,$17,$E6,$7C
	.dc.b	$2C,$8F,$A8,$D9,$FF,$03,$F2,$3D,$CF,$D8,$12,$BF,$12,$FD,$7F,$B2
	.dc.b	$7A,$74,$93,$FC,$D6,$32,$FF,$37,$F9,$77,$BF,$F7,$DB,$19,$8C,$1F
	.dc.b	$83,$F9,$77,$89,$49,$3F,$D0,$63,$2F,$F4,$3F,$95,$FC,$5F,$9F,$7A
	.dc.b	$7F,$C0,$9B,$7F,$88,$A7,$F2,$DF,$1B,$9A,$FC,$A7,$C5,$7F,$4B,$AF
	.dc.b	$DF,$B7,$B0,$D3,$1F,$15,$27,$F3,$8D,$7F,$39,$E7,$71,$23,$FA,$17
	.dc.b	$A7,$9D,$23,$F4,$8D,$7F,$49,$E7,$78,$48,$FE,$A5,$F3,$7C,$93,$E2
	.dc.b	$7F,$DA,$DE,$3F,$FB,$0F,$E4,$FE,$3D,$FF,$A7,$1A,$FF,$09,$E1,$FF
	.dc.b	$6E,$5F,$18,$FF,$50,$AF,$8A,$E7,$3A,$C2,$D7,$FB,$9A,$B8,$3C,$CC
	.dc.b	$56,$97,$BA,$F7,$F1,$D6,$DA,$9B,$9D,$57,$FA,$C9,$7C,$06,$6E,$4E
	.dc.b	$63,$B7,$C9,$BD,$3E,$BE,$24,$AF,$F5,$DC,$E5,$BE,$9F,$0A,$91,$83
	.dc.b	$7E,$9F,$49,$91,$83,$9E,$FE,$AF,$12,$46,$0E,$FB,$F9,$26,$46,$0F
	.dc.b	$1F,$C5,$7F,$39,$7F,$CE,$79,$23,$11,$FE,$E5,$8C,$72,$7F,$87,$5B
	.dc.b	$1C,$3D,$17,$C9,$E5,$B1,$98,$C1,$F9,$B5,$F2,$74,$8F,$E7,$DF,$D0
	.dc.b	$77,$7F,$5F,$83,$5E,$23,$E4,$FE,$EA,$24,$71,$E3,$9D,$1F,$EE,$C9
	.dc.b	$E6,$D8,$ED,$C9,$23,$FD,$E3,$CF,$FE,$76,$AF,$C0,$C7,$1F,$1F,$C8
	.dc.b	$6E,$3F,$20,$78,$5F,$47,$C2,$C9,$E0,$78,$5F,$EF,$5B,$B9,$69,$FD
	.dc.b	$F7,$07,$99,$8B,$F6,$51,$FD,$6F,$1F,$23,$59,$78,$A7,$F6,$53,$C0
	.dc.b	$F8,$90,$F9,$3F,$82,$FA,$1B,$3D,$63,$E5,$60,$F0,$3F,$C2,$15,$AB
	.dc.b	$E6,$FE,$AD,$ED,$5F,$46,$30,$F5,$DF,$8E,$FF,$E1,$9F,$93,$7C,$8F
	.dc.b	$F6,$D6,$E1,$FE,$69,$6D,$FF,$B1,$9A,$FA,$83,$AC,$FD,$11,$8E,$AE
	.dc.b	$D7,$DD,$FF,$14,$7C,$9F,$FB,$36,$7F,$0C,$D4,$D2,$7A,$7F,$E0,$17
	.dc.b	$FD,$45,$1E,$16,$BF,$31,$AF,$E4,$7E,$5B,$F0,$EF,$FB,$79,$AF,$A5
	.dc.b	$AF,$D4,$6B,$F9,$9F,$97,$E6,$9A,$4A,$FF,$0A,$79,$2F,$62,$93,$5F
	.dc.b	$F4,$73,$FE,$19,$FD,$0B,$FD,$A5,$1F,$E9,$56,$E1,$FE,$F2,$6B,$F3
	.dc.b	$5A,$FE,$23,$07,$AD,$FE,$23,$6D,$7F,$A7,$DB,$3F,$EF,$C6,$BF,$5D
	.dc.b	$78,$9C,$49,$3C,$5F,$37,$CB,$48,$F3,$36,$BA,$7A,$FE,$12,$4F,$85
	.dc.b	$F2,$78,$B2,$3C,$8D,$9F,$77,$0F,$C3,$24,$FC,$37,$D9,$E6,$C8,$F6
	.dc.b	$36,$7D,$9D,$1F,$04,$93,$F0,$5F,$9D,$E3,$C8,$F9,$CD,$9F,$89,$E6
	.dc.b	$FC,$92,$4F,$C9,$7E,$67,$9B,$23,$E6,$36,$7E,$B7,$37,$DB,$24,$FD
	.dc.b	$B7,$FC,$AD,$F2,$CD,$A9,$27,$FD,$A9,$19,$6E,$9C,$08,$E0,$FC,$37
	.dc.b	$81,$F8,$92,$38,$3F,$DB,$DF,$25,$FC,$59,$FE,$03,$8A,$30,$FF,$24
	.dc.b	$B7,$C9,$FF,$A6,$67,$3D,$7F,$B8,$77,$5F,$BD,$6B,$FC,$61,$AF,$F3
	.dc.b	$D6,$BD,$FC,$2F,$F5,$25,$BE,$4F,$B4,$61,$CF,$C9,$CC,$BC,$F4,$F9
	.dc.b	$3C,$61,$E7,$AB,$97,$F0,$CA,$D5,$F0,$9F,$F5,$97,$C6,$7F,$4F,$8F
	.dc.b	$82,$FF,$25,$BF,$EA,$4E,$5F,$E8,$AE,$53,$9B,$FE,$32,$E4,$C4,$FF
	.dc.b	$38,$71,$E1,$78,$0F,$0F,$93,$67,$C3,$1F,$E9,$79,$FE,$63,$7F,$C9
	.dc.b	$DA,$F3,$8D,$1E,$AF,$27,$94,$79,$9F,$6F,$93,$2F,$DB,$E8,$DA,$F5
	.dc.b	$CD,$CB,$D3,$36,$BD,$1F,$25,$F1,$BE,$4F,$E3,$B6,$3E,$3B,$70,$F8
	.dc.b	$AD,$BF,$E7,$7E,$23,$E0,$BF,$DC,$0D,$BF,$C7,$AF,$83,$CC,$C9,$FE
	.dc.b	$8D,$87,$F8,$0E,$BF,$9F,$12,$7F,$94,$9D,$3C,$7E,$13,$FD,$EA,$51
	.dc.b	$F7,$BE,$4A,$5F,$DE,$F9,$3B,$7F,$72,$6E,$5F,$6A,$6D,$7D,$9F,$93
	.dc.b	$FC,$3F,$C9,$FE,$21,$B1,$F8,$86,$E1,$F8,$66,$DF,$C2,$F1,$BE,$09
	.dc.b	$F0,$BA,$AD,$FC,$28,$FE,$86,$47,$E0,$DB,$44,$E6,$3D,$DE,$37,$26
	.dc.b	$D7,$E6,$BE,$4F,$F3,$A7,$E7,$B9,$EC,$64,$93,$F9,$C8,$CB,$F9,$EF
	.dc.b	$92,$FD,$2F,$C9,$FE,$99,$B5,$FA,$66,$CE,$E3,$F5,$33,$F7,$BC,$69
	.dc.b	$1F,$7B,$DD,$B5,$F5,$66,$ED,$F8,$69,$B5,$FE,$1E,$D5,$58,$BF,$C1
	.dc.b	$55,$7E,$24,$59,$FF,$41,$57,$E3,$23,$07,$E4,$2A,$C6,$4F,$21,$6B
	.dc.b	$C8,$DB,$F2,$34,$F2,$57,$7A,$E7,$5C,$0F,$C8,$38,$AF,$D0,$CF,$86
	.dc.b	$B9,$57,$E6,$E3,$94,$62,$3D,$6D,$BF,$5C,$75,$AE,$73,$9B,$EC,$F9
	.dc.b	$0F,$5C,$ED,$C3,$4F,$92,$F5,$BF,$0C,$64,$F0,$96,$FC,$37,$E0,$8C
	.dc.b	$4F,$AD,$B5,$E3,$3A,$15,$C9,$CE,$FC,$EF,$C1,$8F,$3D,$75,$23,$E8
	.dc.b	$7E,$8C,$72,$72,$FA,$8E,$A7,$F9,$2E,$FF,$0F,$3F,$A8,$6C,$7F,$0A
	.dc.b	$6B,$BB,$FF,$18,$92,$78,$36,$3F,$85,$78,$FF,$8D,$24,$FD,$56,$C7
	.dc.b	$C7,$6D,$F9,$A3,$F7,$11,$89,$F6,$1F,$9D,$71,$1F,$0C,$EF,$1D,$48
	.dc.b	$F5,$6C,$F5,$B5,$72,$AF,$B3,$3F,$CC,$23,$13,$F0,$A3,$15,$C8,$F1
	.dc.b	$D7,$F2,$DA,$DD,$FF,$36,$6C,$FF,$36,$6C,$F9,$8F,$21,$2B,$C9,$1A
	.dc.b	$7B,$C6,$67,$BB,$F1,$1F,$13,$F0,$63,$67,$28,$CC,$7B,$1F,$90,$62
	.dc.b	$FD,$8E,$23,$B1,$CD,$C4,$F3,$8C,$9C,$CF,$81,$1F,$E0,$0F,$93,$FF
	.dc.b	$01,$FC,$9F,$F8,$11,$B1,$93,$FC,$0B,$19,$69,$FE,$08,$F8,$6F,$F0
	.dc.b	$59,$FE,$0C,$F9,$3B,$19,$25,$FF,$82,$A3,$2F,$FB,$43,$E2,$DE,$67
	.dc.b	$FA,$E7,$2F,$F5,$1C,$62,$FF,$C6,$AB,$CD,$D4,$7F,$B1,$23,$57,$CE
	.dc.b	$FF,$D5,$6E,$5A,$FF,$BB,$63,$6F,$FA,$CD,$AE,$E7,$BF,$97,$DA,$38
	.dc.b	$55,$55,$FE,$E1,$AF,$F1,$4F,$C9,$E7,$E1,$BF,$DB,$D1,$83,$FC,$9A
	.dc.b	$E2,$BF,$C5,$73,$B9,$FF,$19,$B5,$DF,$C9,$F3,$3A,$7C,$9D,$8C,$92
	.dc.b	$BF,$DF,$36,$9D,$47,$CD,$F9,$3F,$46,$26,$FC,$26,$19,$87,$F7,$09
	.dc.b	$D3,$63,$E4,$D7,$C0,$6C,$EA,$3E,$D7,$C9,$F2,$6D,$7F,$9B,$3D,$AF
	.dc.b	$F7,$DC,$38,$3F,$DB,$F0,$F8,$8F,$98,$EA,$33,$18,$F8,$3C,$B6,$FA
	.dc.b	$3C,$3E,$22,$47,$7D,$2E,$3D,$E0,$CC,$55,$AC,$7C,$77,$6F,$76,$BF
	.dc.b	$75,$7A,$34,$F3,$E7,$E8,$65,$FF,$B5,$5F,$F6,$C1,$1C,$1F,$0F,$E3
	.dc.b	$B7,$87,$94,$F2,$3F,$0B,$E1,$7E,$1F,$CB,$5A,$F9,$4D,$AE,$07,$95
	.dc.b	$C6,$3F,$31,$F9,$7F,$2B,$2B,$E9,$7C,$B5,$BE,$3A,$2B,$18,$7D,$4A
	.dc.b	$B7,$FE,$99,$F1,$BF,$D4,$07,$D6,$F6,$EC,$7F,$AA,$3D,$BF,$F1,$AB
	.dc.b	$FE,$AC,$9F,$C3,$C6,$32,$FF,$2D,$B5,$FF,$53,$47,$E0,$3D,$BE,$71
	.dc.b	$E4,$7F,$55,$F2,$1F,$90,$E8,$34,$98,$7F,$91,$A1,$F0,$9B,$24,$65
	.dc.b	$FC,$CE,$CF,$6E,$92,$7F,$C9,$F1,$97,$A7,$A7,$E1,$23,$83,$C1,$E9
	.dc.b	$F9,$48,$E0,$E7,$EE,$F8,$08,$E0,$F4,$7C,$27,$B4,$8E,$0F,$67,$C1
	.dc.b	$FC,$12,$38,$3F,$CC,$BF,$A1,$FF,$AD,$77,$FC,$B8,$70,$FE,$69,$AF
	.dc.b	$E8,$9A,$B5,$FA,$8F,$E8,$1A,$FE,$A1,$AE,$F8,$3E,$A7,$A5,$2F,$E9
	.dc.b	$36,$7F,$D3,$36,$F8,$9F,$E1,$2D,$AF,$4C,$60,$FE,$17,$D3,$23,$F8
	.dc.b	$93,$6B,$FD,$47,$F2,$8F,$7F,$F8,$9B,$6B,$FD,$0E,$D9,$DC,$EB,$D2
	.dc.b	$95,$A6,$CF,$FA,$BB,$E4,$DE,$6F,$F1,$E6,$D7,$FA,$49,$B3,$B9,$FE
	.dc.b	$47,$F2,$7F,$94,$F4,$7F,$D2,$ED,$9F,$F5,$CA,$F9,$7F,$85,$18,$EA
	.dc.b	$FF,$4E,$3F,$EB,$EF,$99,$72,$FF,$A8,$21,$18,$7F,$D4,$6D,$78,$46
	.dc.b	$F9,$1F,$13,$49,$3F,$A5,$75,$EB,$C8,$FD,$43,$5F,$AF,$F0,$BE,$1E
	.dc.b	$53,$49,$1F,$AF,$F5,$FC,$04,$70,$7E,$53,$E1,$B9,$11,$C1,$FE,$BC
	.dc.b	$F9,$47,$9F,$FC,$BB,$C7,$72,$FF,$30,$F9,$5F,$A7,$7F,$CE,$27,$2F
	.dc.b	$F3,$A8,$CB,$FC,$8A,$F9,$BB,$AF,$E7,$31,$FD,$36,$30,$7F,$42,$76
	.dc.b	$F1,$BF,$14,$F0,$F4,$2C,$FF,$48,$F8,$7F,$86,$F2,$1D,$8D,$F7,$18
	.dc.b	$6C,$F0,$8F,$E6,$EB,$4C,$ED,$27,$A4,$E5,$ED,$F2,$BF,$E1,$F7,$C1
	.dc.b	$83,$C1,$EE,$D8,$DF,$B3,$E2,$AF,$F1,$5C,$60,$87,$F4,$87,$2D,$F6
	.dc.b	$78,$7B,$32,$3C,$31,$CF,$7E,$FE,$1C,$9F,$2C,$77,$DF,$51,$83,$D2
	.dc.b	$F8,$FD,$5F,$F0,$04,$7F,$16,$18,$FF,$72,$E4,$FF,$7F,$B6,$34,$D7
	.dc.b	$F8,$EA,$F5,$5F,$2B,$C1,$B3,$C1,$AF,$06,$BE,$1F,$95,$FD,$72,$69
	.dc.b	$1F,$31,$FC,$5C,$FE,$DE,$5B,$FF,$25,$C7,$F8,$33,$E5,$5C,$5F,$CC
	.dc.b	$BC,$8F,$F2,$35,$FE,$AB,$9F,$F1,$0D,$39,$C6,$B9,$77,$CD,$2B,$EF
	.dc.b	$F9,$96,$FC,$3E,$47,$43,$1A,$7F,$97,$9E,$FF,$19,$2F,$BC,$5A,$FF
	.dc.b	$03,$D7,$F9,$5D,$BC,$BF,$E5,$BE,$87,$C2,$FF,$97,$6B,$47,$EC,$2B
	.dc.b	$FC,$71,$CD,$93,$57,$71,$FD,$27,$CA,$FC,$57,$FC,$52,$DA,$FF,$18
	.dc.b	$E3,$FC,$5A,$77,$BF,$F9,$AE,$BE,$26,$3D,$DB,$3C,$A7,$5B,$FF,$9C
	.dc.b	$7E,$55,$F2,$17,$AF,$0D,$BF,$DB,$4F,$F9,$F3,$3F,$F2,$0B,$53,$4C
	.dc.b	$7C,$AB,$D2,$FF,$2F,$B6,$BF,$06,$DA,$FC,$D3,$5F,$EE,$3F,$2B,$FB
	.dc.b	$D9,$FC,$4B,$63,$F4,$0D,$77,$CB,$FE,$6C,$87,$A1,$49,$5C,$9F,$F3
	.dc.b	$8F,$9D,$FE,$1E,$8F,$F7,$F7,$AD,$82,$3F,$E1,$A8,$FF,$0F,$91,$C1
	.dc.b	$F9,$CF,$63,$F8,$53,$7C,$4F,$F1,$B9,$1F,$D0,$FC,$AE,$B3,$FD,$C3
	.dc.b	$5F,$F0,$7B,$5F,$F3,$3B,$5F,$F3,$77,$CA,$7F,$8C,$09,$3F,$E2,$E7
	.dc.b	$FC,$07,$D5,$23,$FC,$36,$D9,$EF,$E2,$FF,$A4,$2C,$D2,$57,$F8,$E3
	.dc.b	$8B,$FC,$7A,$47,$07,$F8,$9B,$B3,$FC,$22,$47,$07,$BB,$CC,$EC,$47
	.dc.b	$07,$F9,$13,$C7,$FF,$24,$91,$C1,$FE,$32,$F0,$3F,$C3,$A4,$70,$7C
	.dc.b	$AF,$2F,$D4,$47,$07,$F9,$43,$A3,$FC,$AA,$47,$07,$F8,$FB,$E4,$BF
	.dc.b	$C5,$24,$70,$7F,$16,$F9,$4F,$D1,$B7,$3F,$F5,$BC,$63,$6F,$E6,$0D
	.dc.b	$7F,$C8,$FE,$D7,$F0,$36,$F8,$3F,$E6,$62,$5F,$F9,$3B,$E5,$3F,$C7
	.dc.b	$24,$70,$7F,$26,$E6,$78,$BF,$EE,$56,$C7,$F6,$75,$F2,$99,$99,$3C
	.dc.b	$5F,$CE,$91,$C1,$C7,$BB,$78,$FF,$A8,$2F,$F6,$5E,$1F,$F1,$5F,$8F
	.dc.b	$7C,$A7,$FB,$D5,$B8,$7F,$77,$5F,$29,$99,$93,$97,$84,$63,$0F,$F5
	.dc.b	$6D,$3E,$4F,$A7,$CA,$75,$6E,$1D,$5B,$5C,$1B,$8F,$F2,$9C,$BF,$D4
	.dc.b	$4D,$F8,$0F,$8D,$E6,$7C,$76,$61,$F2,$DF,$DF,$91,$C1,$F2,$FB,$37
	.dc.b	$8F,$C3,$7E,$07,$CA,$49,$F8,$1A,$5A,$F5,$CD,$C7,$E4,$4D,$AF,$F6
	.dc.b	$FA,$F9,$4F,$95,$E2,$FC,$B6,$CF,$CB,$6E,$1F,$19,$B7,$EF,$F9,$4F
	.dc.b	$C1,$1F,$83,$F5,$ED,$FE,$0E,$3F,$8D,$91,$FC,$53,$6F,$EF,$76,$F0
	.dc.b	$6C,$FD,$BF,$94,$FB,$DF,$29,$F7,$DB,$19,$3F,$01,$19,$69,$F8,$3F
	.dc.b	$83,$FC,$31,$F8,$7F,$94,$B1,$92,$5F,$E0,$A3,$2F,$FB,$C3,$E4,$5E
	.dc.b	$17,$F9,$53,$5F,$EC,$D1,$8B,$FF,$23,$2F,$13,$50,$B9,$3B,$7A,$5F
	.dc.b	$F6,$F7,$42,$3F,$AC,$12,$35,$1C,$9E,$AF,$4E,$FA,$F1,$8B,$3F,$DF
	.dc.b	$2B,$EF,$FC,$A6,$BE,$07,$FC,$E3,$1F,$A7,$8C,$3F,$AB,$6B,$93,$FC
	.dc.b	$C5,$1C,$2D,$97,$F8,$1E,$1F,$29,$D4,$ED,$F2,$96,$F8,$C5,$BE,$31
	.dc.b	$FE,$D1,$9C,$5F,$CC,$F9,$3F,$49,$EA,$F7,$6D,$7C,$E8,$B7,$F5,$66
	.dc.b	$D7,$1A,$D1,$F5,$23,$2F,$CE,$CD,$78,$BF,$AB,$55,$FA,$51,$67,$FC
	.dc.b	$A5,$5D,$27,$F0,$93,$9A,$D3,$67,$57,$E3,$AD,$C7,$8C,$C9,$E3,$9E
	.dc.b	$11,$A2,$4F,$13,$F1,$F5,$DA,$B8,$CF,$A1,$AF,$A1,$B7,$E8,$8E,$71
	.dc.b	$DC,$CC,$F3,$8C,$9C,$CA,$73,$2C,$F3,$AF,$25,$73,$9C,$D7,$39,$CC
	.dc.b	$73,$6B,$CD,$B5,$CE,$B9,$46,$2F,$F0,$11,$EA,$7E,$2C,$66,$3D,$26
	.dc.b	$4B,$5E,$69,$DC,$74,$24,$7E,$CA,$3A,$BB,$8F,$F4,$0D,$7E,$11,$B3
	.dc.b	$FB,$09,$F8,$47,$ED,$8A,$7C,$13,$13,$C5,$FF,$50,$35,$CC,$7E,$D3
	.dc.b	$E5,$3F,$18,$4A,$FA,$ED,$9F,$9E,$DB,$F8,$ED,$9F,$5C,$FF,$07,$8C
	.dc.b	$3E,$D8,$FE,$27,$18,$8F,$A1,$1F,$A1,$8C,$5F,$9E,$BF,$8D,$C6,$2F
	.dc.b	$CD,$5E,$42,$5F,$91,$B5,$E4,$69,$E4,$B5,$F3,$8F,$19,$AA,$F3,$1F
	.dc.b	$CB,$63,$4D,$8D,$3E,$83,$F9,$4C,$6A,$BC,$F3,$AA,$F5,$3E,$D3,$17
	.dc.b	$EA,$BF,$41,$2B,$D0,$D9,$F4,$5F,$57,$94,$64,$F3,$CF,$81,$FE,$32
	.dc.b	$7F,$3D,$F1,$DE,$E7,$F3,$25,$F0,$56,$30,$DB,$FE,$89,$7F,$EA,$FA
	.dc.b	$DC,$60,$FE,$86,$F4,$71,$7D,$EB,$B7,$C1,$49,$CC,$AB,$F5,$8F,$96
	.dc.b	$74,$FA,$0D,$3E,$7D,$33,$87,$8F,$3F,$51,$EB,$CE,$E3,$DB,$1F,$CE
	.dc.b	$63,$E6,$AF,$D3,$46,$23,$E2,$46,$0F,$89,$F2,$9C,$A3,$C5,$28,$F9
	.dc.b	$1E,$FF,$C9,$F9,$4B,$87,$76,$DF,$06,$E5,$F2,$5F,$A1,$F2,$92,$BC
	.dc.b	$DF,$29,$6F,$BC,$DB,$FA,$13,$6B,$FB,$8F,$B5,$FE,$38,$6F,$8D,$F6
	.dc.b	$FE,$53,$FB,$6C,$E7,$1F,$EF,$8A,$7F,$83,$36,$BF,$D6,$11,$82,$3F
	.dc.b	$27,$E5,$39,$C7,$E1,$DF,$EF,$3F,$17,$FE,$5A,$BF,$AC,$FF,$96,$AB
	.dc.b	$EE,$B9,$CB,$FD,$EC,$E6,$77,$E7,$F2,$F3,$F9,$B6,$F8,$F9,$DD,$8E
	.dc.b	$FE,$7D,$8C,$92,$FC,$31,$97,$C7,$E6,$79,$8F,$3F,$99,$6B,$CB,$1F
	.dc.b	$C9,$F2,$34,$91,$FC,$A6,$BF,$C4,$93,$A8,$F6,$F9,$7C,$E7,$E0,$BF
	.dc.b	$13,$CC,$F8,$BE,$4D,$AF,$86,$DA,$F8,$ED,$DB,$E2,$BF,$33,$C9,$95
	.dc.b	$F2,$BC,$9B,$7F,$3A,$6D,$FC,$C9,$B5,$FE,$34,$C9,$E4,$FD,$63,$EB
	.dc.b	$F8,$D6,$7E,$AC,$7C,$12,$3E,$FC,$55,$CA,$30,$D8,$FD,$A7,$4F,$F8
	.dc.b	$B7,$1F,$BB,$7F,$E3,$C8,$C5,$FF,$91,$BC,$67,$BD,$FD,$55,$B1,$DD
	.dc.b	$B8,$7C,$F6,$BF,$EC,$2F,$94,$FD,$B9,$27,$CD,$DE,$8F,$EC,$18,$7F
	.dc.b	$8F,$27,$D4,$DA,$FB,$4D,$7F,$B2,$35,$F3,$7A,$5E,$52,$4F,$95,$FE
	.dc.b	$0F,$F2,$92,$3C,$2D,$9E,$BE,$4F,$88,$93,$E2,$7E,$07,$C0,$48,$F8
	.dc.b	$0D,$9F,$C7,$6F,$FE,$71,$9F,$9F,$1F,$95,$23,$EC,$F9,$1F,$79,$1C
	.dc.b	$1F,$2F,$D3,$F9,$C4,$70,$7D,$AF,$87,$FD,$01,$1C,$1F,$23,$E4,$7E
	.dc.b	$59,$1C,$1F,$85,$F9,$47,$C1,$FF,$83,$DB,$19,$8C,$19,$EA,$91,$C5
	.dc.b	$B9,$7D,$4E,$4F,$B2,$49,$FB,$2F,$F8,$6F,$E4,$D8,$BD,$AF,$CB,$46
	.dc.b	$1B,$AB,$4C,$D6,$F1,$FD,$D7,$C9,$BE,$4F,$F3,$D0,$8C,$3F,$A1,$6B
	.dc.b	$C2,$37,$6F,$26,$92,$7F,$51,$19,$7F,$39,$ED,$F6,$EA,$34,$91,$FA
	.dc.b	$CF,$8C,$F1,$F2,$1A,$48,$FC,$DF,$BF,$E8,$EA,$34,$91,$FB,$6F,$8A
	.dc.b	$E6,$47,$07,$E9,$3E,$03,$91,$1C,$1F,$C0,$7E,$4B,$FC,$0A,$FF,$76
	.dc.b	$F9,$37,$89,$FE,$C0,$F0,$DD,$1F,$EC,$29,$CD,$7F,$7C,$BE,$2F,$F7
	.dc.b	$97,$FC,$1A,$E7,$DA,$E5,$5F,$EC,$A9,$DD,$F4,$AF,$F6,$94,$70,$7F
	.dc.b	$C0,$5F,$0F,$F1,$5F,$F0,$4F,$AB,$F3,$1B,$C3,$D1,$FF,$6D,$4E,$DE
	.dc.b	$3A,$E5,$46,$70,$F0,$46,$38,$35,$2A,$E1,$F2,$7F,$EE,$7A,$F0,$F8
	.dc.b	$3C,$F2,$3A,$F8,$1D,$FC,$05,$AF,$F0,$FD,$F1,$7F,$C3,$8F,$FB,$D9
	.dc.b	$CF,$C0,$F0,$77,$3E,$08,$F0,$F0,$77,$F1,$25,$74,$8F,$2F,$BF,$A8
	.dc.b	$C1,$E0,$7D,$3E,$FF,$FB,$8A,$3F,$AE,$62,$8C,$47,$FA,$1E,$31,$C1
	.dc.b	$FD,$0D,$B1,$C3,$DF,$7C,$9E,$5B,$19,$8C,$1F,$AE,$5F,$27,$48,$FE
	.dc.b	$C9,$FD,$9F,$87,$F8,$1C,$3E,$9C,$FE,$EF,$9F,$FC,$8E,$8E,$63,$E4
	.dc.b	$FE,$DB,$93,$81,$FE,$49,$3F,$59,$1C,$B9,$BB,$3F,$EA,$5B,$B3,$C6
	.dc.b	$6B,$DF,$96,$BF,$83,$AF,$C8,$F5,$E2,$73,$E4,$C9,$F3,$C5,$AF,$17
	.dc.b	$C6,$F2,$31,$9F,$B9,$EE,$B9,$29,$E2,$E8,$7C,$46,$31,$FF,$34,$D7
	.dc.b	$F9,$BF,$0F,$F3,$33,$4F,$CD,$9C,$E3,$7C,$BE,$CF,$93,$F8,$B3,$FA
	.dc.b	$06,$CF,$C9,$C1,$F1,$DB,$E0,$E1,$C1,$2B,$FC,$F3,$7C,$32,$FA,$8F
	.dc.b	$CE,$7F,$CF,$1F,$27,$E1,$6E,$BF,$32,$3C,$D5,$F5,$DD,$67,$FB,$27
	.dc.b	$85,$7F,$B6,$5A,$EE,$BF,$DB,$D1,$88,$FC,$27,$C9,$BE,$63,$FD,$C4
	.dc.b	$DC,$3F,$C3,$8D,$9F,$F1,$E3,$5F,$F7,$33,$FD,$16,$37,$E5,$72,$23
	.dc.b	$CB,$BF,$A9,$1C,$1F,$E2,$9F,$92,$E9,$7F,$93,$6B,$F9,$96,$A6,$9E
	.dc.b	$17,$FC,$A0,$D7,$FD,$F4,$D7,$A5,$7F,$BF,$E3,$F8,$8C,$F8,$6B,$04
	.dc.b	$7F,$C7,$9F,$0B,$F8,$A6,$F1,$FF,$A0,$88,$FF,$90,$7C,$F7,$47,$EC
	.dc.b	$B2,$FD,$84,$62,$75,$39,$9F,$EE,$8D,$97,$C7,$52,$57,$57,$FC,$6F
	.dc.b	$EB,$C8,$FF,$1B,$B6,$FF,$C6,$FB,$7F,$8F,$09,$3F,$E3,$C7,$FC,$7F
	.dc.b	$AC,$8F,$F1,$FB,$67,$FC,$7F,$A7,$F9,$10,$93,$FE,$44,$7F,$C8,$F9
	.dc.b	$C8,$FF,$23,$B6,$7F,$C8,$FF,$0D,$FE,$4C,$24,$FF,$93,$1F,$F2,$7F
	.dc.b	$87,$23,$FC,$9E,$D9,$FF,$27,$FC,$C7,$F9,$50,$93,$FE,$54,$7F,$CA
	.dc.b	$FF,$2F,$23,$FC,$AE,$D9,$FF,$2B,$F4,$FF,$97,$09,$3F,$E5,$C7,$FC
	.dc.b	$BF,$E4,$C8,$FF,$2F,$B6,$7F,$CB,$FB,$FF,$99,$09,$3F,$E6,$47,$FC
	.dc.b	$CF,$F2,$1F,$E7,$73,$F8,$FF,$E8,$9F,$E2,$D9,$FF,$7A,$FA,$09,$D4
	.dc.b	$7F,$9B,$1E,$2D,$25,$FF,$BD,$E9,$F2,$DC,$23,$0F,$6F,$95,$EF,$E6
	.dc.b	$DB,$FE,$79,$DB,$FD,$CB,$04,$E7,$0E,$8D,$9D,$3E,$8F,$36,$5F,$23
	.dc.b	$FA,$15,$FF,$0C,$6C,$6A,$FC,$CF,$C0,$EE,$C6,$1E,$1F,$0F,$E1,$9F
	.dc.b	$13,$CD,$B8,$75,$C3,$FA,$96,$3F,$CC,$5F,$EA,$01,$C3,$E2,$AF,$D3
	.dc.b	$46,$23,$8C,$7D,$0F,$37,$AC,$7C,$97,$E9,$F9,$9F,$53,$E4,$2E,$1E
	.dc.b	$1A,$ED,$18,$7E,$83,$72,$F9,$4F,$DC,$F9,$19,$5C,$72,$7C,$8E,$30
	.dc.b	$FB,$15,$70,$FA,$53,$6F,$FD,$75,$D1,$FD,$93,$D7,$3F,$88,$F2,$B0
	.dc.b	$C9,$FF,$5E,$3F,$EB,$F2,$38,$3C,$5C,$DB,$CB,$F1,$0F,$E4,$BE,$47
	.dc.b	$F2,$7F,$23,$6B,$D0,$DA,$FC,$73,$73,$FC,$43,$F9,$AF,$6E,$57,$BF
	.dc.b	$E4,$6D,$FE,$56,$6D,$FE,$3E,$6D,$7E,$23,$E1,$7F,$46,$7E,$93,$E1
	.dc.b	$6C,$7C,$98,$FB,$64,$7F,$DD,$1C,$3F,$ED,$36,$C7,$F8,$2B,$F4,$5F
	.dc.b	$F5,$E6,$5F,$A5,$8C,$39,$F4,$75,$E8,$DB,$FB,$D7,$FD,$AF,$3F,$F2
	.dc.b	$AB,$FD,$BE,$CE,$67,$FC,$40,$E4,$C4,$7F,$89,$DF,$F0,$C2,$9E,$7E
	.dc.b	$9F,$97,$D4,$F3,$32,$FC,$2E,$5A,$7E,$47,$E4,$DC,$9F,$E2,$66,$C6
	.dc.b	$79,$B1,$32,$38,$3B,$8C,$63,$E3,$9F,$F1,$C5,$72,$8C,$1F,$E3,$F7
	.dc.b	$18,$FF,$9D,$9F,$CC,$CF,$42,$4F,$01,$1C,$23,$AD,$78,$2B,$95,$D9
	.dc.b	$C9,$CA,$32,$D3,$37,$D6,$FA,$3E,$5A,$E4,$F9,$EE,$BE,$28,$B3,$E8
	.dc.b	$8E,$51,$83,$E2,$57,$78,$E8,$D7,$D7,$5B,$8C,$57,$3B,$F3,$D7,$49
	.dc.b	$CD,$6E,$30,$7A,$5C,$4F,$03,$E6,$38,$9F,$A7,$7E,$4A,$E3,$5B,$9C
	.dc.b	$9F,$7C,$F1,$CF,$BA,$B9,$57,$A8,$F0,$4E,$8F,$29,$5E,$AF,$CE,$AE
	.dc.b	$D1,$C5,$AF,$C2,$8E,$11,$8B,$F8,$51,$E4,$9E,$67,$17,$A1,$99,$E1
	.dc.b	$7E,$D6,$C7,$F3,$A9,$F4,$1F,$CF,$2B,$F2,$8D,$AF,$F1,$04,$62,$3C
	.dc.b	$06,$67,$93,$F8,$16,$C7,$E2,$23,$F2,$BC,$9F,$84,$24,$FD,$C6,$C7
	.dc.b	$A8,$DC,$64,$8E,$65,$3C,$2F,$E2,$36,$BE,$01,$FA,$E7,$15,$F1,$4F
	.dc.b	$D7,$38,$7D,$07,$E7,$DC,$57,$A4,$EF,$3E,$2A,$EE,$D3,$BD,$74,$70
	.dc.b	$F7,$3D,$67,$A6,$BA,$DF,$F0,$B6,$CF,$F0,$F6,$CF,$8E,$39,$BE,$6A
	.dc.b	$FE,$09,$3A,$6C,$EA,$FD,$13,$98,$F8,$46,$6F,$C0,$D3,$C0,$5B,$F0
	.dc.b	$1E,$77,$C3,$9B,$D5,$FE,$7D,$ED,$7F,$3F,$6C,$FF,$3A,$6D,$7F,$15
	.dc.b	$AE,$13,$FC,$FA,$BF,$A3,$7C,$BC,$8F,$E8,$3F,$2D,$6F,$84,$DD,$3F
	.dc.b	$A1,$CD,$AF,$E5,$3C,$5F,$CC,$CF,$E6,$9C,$B6,$3F,$9C,$47,$E6,$08
	.dc.b	$FE,$51,$B8,$7F,$1D,$E5,$E0,$D9,$FE,$A7,$F2,$9F,$D6,$FE,$57,$FA
	.dc.b	$E3,$63,$27,$F5,$88,$CB,$4F,$EB,$7F,$2B,$FD,$8C,$FE,$C9,$F2,$B6
	.dc.b	$32,$4B,$F8,$D1,$97,$F9,$E7,$C6,$39,$FF,$DE,$AD,$AF,$F1,$23,$FD
	.dc.b	$7B,$64,$2A,$4E,$DE,$75,$FE,$99,$8C,$47,$F8,$E9,$AF,$10,$E1,$E7
	.dc.b	$1E,$2E,$2F,$1B,$6B,$FB,$CF,$CA,$7F,$80,$4F,$F0,$17,$CA,$D8,$C9
	.dc.b	$2F,$F0,$91,$97,$F0,$DF,$2B,$FE,$09,$F8,$FF,$F0,$53,$6B,$FC,$0A
	.dc.b	$D9,$DC,$7F,$82,$27,$E2,$ED,$23,$E2,$EF,$6B,$E5,$CF,$C1,$24,$FF
	.dc.b	$18,$D1,$18,$9F,$A3,$BF,$1D,$BE,$6C,$7C,$C8,$FF,$0F,$7C,$AC,$C3
	.dc.b	$FC,$33,$F2,$B6,$BF,$41,$36,$FF,$C3,$73,$6B,$FC,$33,$F1,$5F,$E2
	.dc.b	$7F,$94,$FF,$14,$36,$3F,$C4,$6D,$C3,$F5,$4D,$BF,$1C,$7D,$6D,$7F
	.dc.b	$0A,$7E,$1B,$6B,$7F,$86,$8F,$C0,$11,$F8,$0D,$9F,$C7,$E5,$C1,$B3
	.dc.b	$FE,$34,$F9,$5F,$F7,$07,$59,$A4,$8F,$E8,$3F,$A3,$7E,$49,$A7,$FB
	.dc.b	$9D,$C5,$7F,$4F,$6B,$FE,$8A,$5F,$2F,$B2,$4F,$47,$97,$CB,$48,$EE
	.dc.b	$DA,$E9,$F2,$5D,$C9,$3E,$47,$C5,$F2,$92,$3C,$ED,$9F,$27,$CB,$79
	.dc.b	$C9,$3E,$C7,$D5,$F2,$72,$3D,$ED,$9F,$67,$C4,$7B,$C9,$3F,$11,$F8
	.dc.b	$5F,$0F,$23,$E3,$B6,$7E,$27,$C3,$FC,$72,$4F,$CC,$7E,$57,$C3,$C8
	.dc.b	$F9,$ED,$9F,$EF,$5F,$37,$FE,$2F,$7F,$D6,$AD,$7F,$78,$D7,$F8,$33
	.dc.b	$57,$CD,$B9,$FF,$1B,$1F,$C0,$E3,$17,$FD,$F9,$B1,$FE,$09,$6B,$FE
	.dc.b	$C4,$F9,$86,$CF,$8B,$FF,$02,$C6,$1B,$C0,$F4,$73,$5B,$BF,$C3,$7C
	.dc.b	$5F,$F8,$31,$72,$A3,$0D,$8F,$C8,$C6,$FE,$CD,$F1,$74,$93,$FE,$16
	.dc.b	$8C,$BF,$E1,$8E,$57,$C1,$77,$6C,$66,$30,$78,$3E,$32,$47,$E8,$1A
	.dc.b	$FE,$76,$FC,$A4,$7F,$4A,$FA,$3E,$32,$47,$EA,$1A,$FE,$97,$DC,$F6
	.dc.b	$91,$FD,$6B,$F0,$3E,$32,$47,$EC,$1A,$FF,$8E,$FE,$59,$F0,$9F,$CE
	.dc.b	$FC,$4F,$E0,$FF,$2F,$F7,$6B,$FD,$D8,$D7,$FD,$15,$5C,$1F,$F7,$82
	.dc.b	$F8,$6E,$5D,$1F,$D1,$63,$53,$C3,$0C,$BF,$D4,$63,$83,$FE,$F9,$F5
	.dc.b	$DE,$EE,$A3,$FD,$57,$F1,$B7,$DA,$39,$AD,$CF,$1F,$7B,$17,$23,$07
	.dc.b	$F5,$FD,$7F,$91,$FC,$BB,$E2,$BA,$74,$4A,$D6,$DF,$D7,$6B,$51,$83
	.dc.b	$FB,$43,$F8,$07,$FD,$58,$E9,$B5,$E9,$8D,$5F,$83,$E2,$B1,$CF,$2F
	.dc.b	$A0,$A4,$F1,$E0,$F8,$1C,$B2,$B7,$5F,$33,$0F,$5F,$16,$E7,$37,$E1
	.dc.b	$8F,$17,$C5,$7F,$B7,$67,$FC,$F5,$68,$C4,$7F,$7B,$3C,$4E,$1F,$84
	.dc.b	$D7,$E0,$FC,$53,$E5,$FF,$CF,$74,$8C,$4F,$66,$BD,$63,$53,$FB,$78
	.dc.b	$7C,$BD,$23,$FB,$B7,$F7,$9E,$07,$F6,$CC,$FF,$87,$B7,$83,$FD,$29
	.dc.b	$3F,$E1,$BF,$97,$72,$FA,$EB,$58,$7F,$85,$F9,$2B,$FE,$21,$6B,$FC
	.dc.b	$37,$0F,$A6,$FE,$49,$BC,$BA,$38,$F3,$BC,$BC,$4F,$0F,$73,$99,$8B
	.dc.b	$FF,$78,$1F,$E2,$DD,$7F,$93,$FB,$BE,$03,$75,$E0,$7C,$B1,$F8,$28
	.dc.b	$FC,$2B,$67,$FD,$F5,$83,$C0,$DF,$3F,$0E,$69,$87,$08,$B7,$FF,$00
	.dc.b	$7E,$5F,$DC,$DD,$FD,$7F,$27,$F1,$27,$39,$FF,$82,$1F,$85,$5C,$78
	.dc.b	$35,$7B,$7E,$5F,$CB,$F9,$67,$F1,$4D,$7F,$C8,$CD,$7F,$CA,$0D,$4F
	.dc.b	$B6,$FF,$C2,$EF,$F0,$BC,$BF,$4A,$E5,$E7,$D3,$F6,$BE,$5D,$E7,$FF
	.dc.b	$52,$C3,$FC,$B9,$18,$AF,$F6,$7B,$63,$FD,$B4,$D7,$F4,$0F,$F4,$78
	.dc.b	$DF,$7F,$39,$4E,$5E,$31,$9F,$9C,$FF,$30,$41,$A4,$E4,$E5,$F3,$1D
	.dc.b	$E7,$FC,$D0,$D7,$FC,$E4,$D4,$D3,$C0,$FE,$D5,$AF,$2A,$EA,$79,$FE
	.dc.b	$6D,$F2,$D4,$99,$7F,$BC,$23,$2F,$FB,$CB,$CF,$FC,$D3,$7A,$7E,$04
	.dc.b	$63,$0F,$F8,$00,$D7,$F8,$32,$F9,$8D,$92,$BA,$3F,$27,$E3,$9F,$2D
	.dc.b	$49,$3F,$F0,$01,$F9,$DF,$1E,$F6,$29,$27,$F8,$A6,$3F,$E9,$79,$D4
	.dc.b	$62,$3F,$DF,$BC,$5C,$89,$19,$9F,$99,$E9,$3A,$BF,$C9,$C4,$9F,$F2
	.dc.b	$9B,$E2,$ED,$91,$C5,$B8,$78,$AF,$FC,$0F,$3D,$7D,$0F,$49,$1F,$F2
	.dc.b	$EF,$99,$FE,$5B,$23,$83,$C9,$E8,$7C,$02,$38,$3D,$1E,$87,$C4,$23
	.dc.b	$83,$FC,$D3,$C1,$FE,$67,$23,$83,$E0,$57,$CC,$23,$83,$E2,$47,$D0
	.dc.b	$23,$83,$FC,$EB,$F3,$1F,$E7,$32,$38,$3E,$87,$CB,$30,$FE,$56,$D8
	.dc.b	$FE,$B7,$F2,$6F,$17,$35,$27,$8B,$FA,$DA,$E1,$AF,$DE,$6B,$FD,$4D
	.dc.b	$CF,$81,$FE,$9A,$C1,$B7,$3A,$4E,$63,$F9,$EC,$BE,$5A,$FC,$B6,$1B
	.dc.b	$97,$99,$B8,$72,$8F,$F6,$E5,$71,$7D,$75,$D3,$E5,$A4,$7B,$BE,$4E
	.dc.b	$DF,$13,$FD,$D9,$6A,$CE,$5E,$9A,$B7,$DB,$E4,$FC,$E7,$A3,$E4,$EC
	.dc.b	$7C,$A8,$FE,$25,$81,$A4,$8F,$D1,$46,$5F,$F0,$5F,$C3,$FF,$BF,$5E
	.dc.b	$46,$32,$FE,$C3,$1F,$CF,$A3,$FD,$F4,$FF,$3E,$C7,$FD,$8E,$E6,$41
	.dc.b	$D4,$F0,$F2,$79,$F9,$3D,$1B,$1E,$2F,$3B,$B9,$E0,$F1,$AC,$64,$98
	.dc.b	$7C,$93,$B4,$69,$C7,$81,$F8,$B9,$FF,$13,$B6,$33,$F2,$DF,$DF,$23
	.dc.b	$19,$FF,$B8,$1D,$4F,$F8,$75,$AF,$B0,$38,$7D,$ED,$73,$CD,$D7,$93
	.dc.b	$B3,$70,$F9,$FE,$7F,$CC,$F4,$3E,$6B,$63,$27,$D4,$8C,$B4,$FA,$FF
	.dc.b	$2D,$F5,$0F,$AB,$F2,$D6,$32,$4C,$7E,$37,$0A,$75,$1F,$1B,$8F,$E1
	.dc.b	$1F,$0F,$96,$DF,$C8,$8F,$E6,$84,$7F,$95,$6B,$F8,$9A,$FE,$5E,$DD
	.dc.b	$BE,$57,$2F,$29,$FA,$6F,$E2,$BE,$4F,$F1,$7F,$25,$70,$FC,$8B,$6B
	.dc.b	$CE,$DB,$FC,$6B,$F9,$2F,$92,$95,$F9,$8F,$96,$B7,$EE,$9B,$7F,$94
	.dc.b	$9B,$5F,$67,$E5,$3F,$C5,$8D,$E6,$FF,$1E,$C3,$9E,$C4,$9F,$F3,$7C
	.dc.b	$F0,$C7,$F3,$AE,$E3,$33,$B9,$FC,$A4,$ED,$3B,$73,$1B,$C1,$5E,$33
	.dc.b	$BD,$E5,$C5,$70,$72,$78,$08,$E4,$F0,$C7,$17,$CF,$55,$F1,$C5,$BF
	.dc.b	$31,$1C,$4E,$5B,$5D,$AF,$CF,$7F,$0A,$72,$F8,$63,$2F,$69,$E0,$7A
	.dc.b	$A3,$37,$DE,$33,$3E,$66,$BC,$4E,$8F,$12,$3E,$17,$71,$EA,$BF,$B7
	.dc.b	$7E,$4B,$EF,$18,$AD,$BE,$D6,$9C,$A7,$83,$DA,$FC,$55,$CA,$37,$1F
	.dc.b	$39,$B1,$98,$C1,$E8,$3D,$85,$39,$C7,$5B,$F4,$3F,$72,$35,$18,$79
	.dc.b	$CF,$53,$B3,$FE,$B1,$8F,$F2,$59,$2F,$FC,$A4,$DA,$FF,$10,$36,$7F
	.dc.b	$C4,$AF,$E0,$39,$3F,$04,$49,$F9,$6D,$9F,$1B,$6F,$C5,$3F,$AB,$8C
	.dc.b	$3E,$B6,$CF,$C0,$9F,$BF,$18,$8F,$D1,$DF,$EC,$23,$17,$E3,$BF,$09
	.dc.b	$2B,$C2,$DB,$F0,$C7,$EC,$AB,$8D,$F3,$AD,$1E,$32,$CF,$8D,$F4,$18
	.dc.b	$7C,$6F,$EF,$A7,$9B,$4D,$5F,$96,$7C,$64,$BF,$1B,$FB,$EA,$E2,$FF
	.dc.b	$02,$6C,$7C,$2A,$E7,$7E,$7A,$E5,$1E,$76,$9E,$73,$DC,$F0,$3C,$8F
	.dc.b	$F3,$CF,$96,$FE,$EA,$7F,$76,$F9,$6B,$3F,$DF,$22,$CF,$F4,$29,$B7
	.dc.b	$FD,$E3,$E5,$3F,$BE,$FC,$A7,$F7,$E6,$E1,$FE,$04,$6D,$FF,$4A,$6D
	.dc.b	$7E,$EB,$7F,$E5,$27,$F2,$AE,$0B,$7F,$CA,$23,$F2,$64,$7F,$26,$DB
	.dc.b	$FE,$4F,$C1,$C1,$B3,$FE,$14,$F9,$4F,$F0,$89,$FE,$12,$F9,$6B,$19
	.dc.b	$24,$FF,$5E,$8C,$BF,$D8,$3E,$3F,$FC,$33,$F2,$DF,$E1,$A6,$D7,$F8
	.dc.b	$81,$B3,$B8,$FF,$11,$CF,$F4,$4D,$64,$7F,$42,$D6,$D7,$F3,$E9,$FD
	.dc.b	$79,$27,$F5,$ED,$8F,$E8,$78,$F0,$D7,$F9,$0B,$FC,$A9,$FE,$4F,$F0
	.dc.b	$CF,$53,$18,$7F,$83,$E3,$0F,$F8,$B2,$77,$EB,$EB,$0F,$F1,$7D,$69
	.dc.b	$FF,$12,$36,$7A,$DE,$E5,$3F,$EB,$C7,$71,$98,$7C,$C5,$F8,$9C,$37
	.dc.b	$1F,$11,$E5,$71,$7E,$06,$E1,$EC,$F8,$9E,$87,$5F,$82,$B7,$CA,$FD
	.dc.b	$33,$FE,$E9,$C4,$D2,$4F,$FA,$E6,$32,$F1,$9F,$5F,$C1,$F4,$7E,$0B
	.dc.b	$EB,$F8,$19,$3B,$A7,$C4,$E2,$FE,$3D,$5C,$BE,$9C,$DB,$FA,$5F,$13
	.dc.b	$F1,$7D,$DF,$8C,$D8,$E8,$DC,$3E,$8B,$6F,$FB,$1F,$94,$F8,$CF,$A4
	.dc.b	$DB,$F2,$7C,$C6,$26,$47,$FB,$4D,$FF,$6F,$91,$C1,$EB,$F3,$77,$8F
	.dc.b	$E0,$1F,$BD,$EE,$FF,$83,$27,$FD,$C0,$D9,$FF,$72,$B5,$FF,$2E,$07
	.dc.b	$0F,$F9,$7D,$7C,$BF,$F7,$C8,$C3,$5F,$F3,$FC,$6A,$7B,$CF,$DB,$F3
	.dc.b	$E4,$7D,$96,$D7,$8B,$E4,$79,$3E,$16,$D9,$A6,$3E,$46,$5F,$B1,$B5
	.dc.b	$FF,$00,$38,$BF,$D7,$F3,$F8,$7D,$7F,$18,$47,$E1,$FC,$B7,$62,$38
	.dc.b	$3E,$3F,$C2,$78,$88,$E0,$FC,$97,$B3,$F9,$72,$38,$3E,$7F,$C2,$7A
	.dc.b	$88,$E0,$FA,$7F,$09,$EE,$23,$83,$F3,$DE,$DF,$E8,$C8,$E0,$FA,$F9
	.dc.b	$FF,$AB,$BE,$51,$E2,$FC,$A6,$CE,$A5,$18,$78,$B5,$FC,$0F,$CA,$37
	.dc.b	$FE,$25,$1D,$A3,$1E,$0F,$37,$EA,$7B,$56,$B7,$5F,$8A,$F9,$B6,$9E
	.dc.b	$18,$46,$1F,$23,$5F,$5B,$5F,$C9,$FC,$AB,$E1,$B9,$36,$3D,$8D,$7E
	.dc.b	$1B,$5F,$CD,$F5,$7F,$0A,$7E,$13,$5F,$92,$D7,$7E,$17,$9E,$1C,$14
	.dc.b	$93,$C9,$F6,$74,$77,$8F,$D3,$61,$82,$3F,$B0,$E7,$D9,$1C,$1F,$C6
	.dc.b	$FE,$03,$E3,$47,$FC,$01,$8C,$10,$FD,$BF,$C3,$78,$08,$FE,$F1,$E3
	.dc.b	$ED,$C8,$FE,$4C,$D7,$F0,$FF,$28,$F1,$BF,$DB,$5E,$17,$F9,$DB,$E5
	.dc.b	$7F,$13,$5F,$CA,$DA,$FF,$BB,$6B,$83,$FC,$BD,$78,$FC,$BA,$7F,$DC
	.dc.b	$91,$A9,$E1,$86,$50,$8E,$2F,$F3,$9F,$81,$7C,$16,$E1,$E1,$BE,$23
	.dc.b	$0D,$8E,$13,$BA,$FF,$5C,$46,$08,$34,$FF,$49,$C6,$A3,$19,$7F,$BE
	.dc.b	$FE,$55,$E7,$76,$F8,$D9,$5F,$D3,$67,$2D,$E8,$F3,$46,$23,$99,$EE
	.dc.b	$71,$8F,$FA,$89,$FC,$83,$7C,$EE,$35,$2B,$B4,$74,$DF,$DF,$C9,$27
	.dc.b	$C5,$1E,$0E,$1D,$46,$0F,$33,$E4,$F3,$BE,$04,$7F,$06,$CB,$FA,$FE
	.dc.b	$5F,$C0,$DB,$19,$9D,$BF,$D0,$A1,$E6,$BE,$57,$83,$67,$A3,$5E,$0D
	.dc.b	$7F,$8A,$FC,$AF,$F0,$14,$D2,$3F,$C5,$7C,$2F,$F0,$A2,$6B,$C0,$7C
	.dc.b	$AB,$8F,$FA,$BC,$F0,$E4,$9C,$9F,$E0,$3B,$E4,$D8,$EB,$F0,$32,$3F
	.dc.b	$C1,$1C,$CF,$2F,$17,$F6,$8F,$CB,$FB,$FE,$43,$3C,$3F,$D8,$5F,$3F
	.dc.b	$C0,$4C,$3C,$11,$6B,$95,$39,$BF,$C6,$0D,$AF,$0F,$C7,$62,$73,$CD
	.dc.b	$27,$FC,$39,$5B,$BE,$73,$FE,$39,$F9,$5F,$6C,$79,$DB,$3F,$EF,$0D
	.dc.b	$1F,$21,$BE,$0E,$1C,$13,$2F,$F1,$46,$1C,$1F,$DA,$E2,$8F,$D9,$C7
	.dc.b	$F8,$B7,$E5,$7C,$59,$7F,$BF,$1F,$E0,$31,$9D,$3F,$C9,$91,$8B,$F9
	.dc.b	$4D,$7F,$C1,$AD,$7B,$B5,$FE,$0A,$D5,$F2,$AE,$DF,$F0,$2B,$6F,$FA
	.dc.b	$E3,$67,$FB,$1B,$5F,$F0,$DF,$CA,$B8,$7F,$C1,$AD,$AF,$EC,$8D,$9F
	.dc.b	$ED,$6D,$78,$FB,$BC,$09,$3C,$1F,$F0,$BF,$CA,$F7,$9F,$92,$D7,$F8
	.dc.b	$FB,$53,$7E,$57,$FC,$E1,$1F,$E3,$06,$BF,$49,$AF,$F2,$86,$AF,$9B
	.dc.b	$F3,$3F,$61,$AF,$F2,$E6,$BB,$C7,$FC,$5B,$E9,$BE,$5A,$93,$2F,$F0
	.dc.b	$34,$65,$FF,$04,$7C,$73,$E0,$3F,$C6,$AD,$8C,$C6,$0F,$F1,$CF,$CD
	.dc.b	$FF,$77,$8F,$F0,$83,$71,$FF,$0B,$B5,$FF,$27,$B5,$FC,$8F,$BB,$FE
	.dc.b	$08,$9F,$F0,$C3,$63,$FC,$3E,$D7,$FC,$AE,$D7,$FC,$AD,$D1,$FE,$48
	.dc.b	$24,$FF,$98,$1F,$F4,$67,$C8,$48,$FF,$45,$36,$7A,$FA,$3F,$E5,$2B
	.dc.b	$F2,$47,$F5,$86,$A6,$FA,$5F,$D5,$6F,$FC,$DB,$E6,$FF,$96,$08,$FF
	.dc.b	$A5,$FE,$43,$FD,$38,$47,$07,$BB,$D2,$F0,$11,$C1,$FE,$78,$E0,$FF
	.dc.b	$31,$91,$C1,$FE,$3D,$FD,$1F,$FB,$6B,$72,$FF,$35,$C6,$1F,$F2,$3B
	.dc.b	$5F,$93,$3E,$F2,$4F,$D0,$7F,$CE,$3F,$31,$23,$FD,$12,$D9,$E7,$E0
	.dc.b	$76,$24,$F8,$DF,$C3,$F8,$3F,$EC,$FE,$3F,$EF,$73,$FE,$B7,$5F,$09
	.dc.b	$99,$93,$F9,$07,$8B,$49,$7F,$CD,$6B,$FC,$FA,$D9,$ED,$BF,$FB,$77
	.dc.b	$E5,$1E,$35,$F2,$70,$DC,$7F,$D9,$13,$A6,$CE,$9A,$6B,$E5,$7A,$1D
	.dc.b	$7E,$56,$C6,$49,$97,$F4,$9B,$4E,$A3,$D1,$CD,$F0,$CF,$89,$EC,$5B
	.dc.b	$F8,$91,$FA,$E2,$3F,$B2,$6C,$FC,$1E,$4E,$0D,$9F,$5F,$C8,$FC,$1F
	.dc.b	$91,$F8,$4D,$8C,$9F,$02,$32,$D3,$E1,$FC,$8F,$C9,$3E,$57,$CA,$D8
	.dc.b	$C9,$2F,$DD,$19,$7F,$B4,$7C,$AF,$F2,$BF,$2D,$5F,$D3,$23,$0D,$F4
	.dc.b	$73,$5C,$3C,$3B,$78,$85,$FF,$1A,$24,$FF,$B1,$9B,$7F,$F0,$05,$FB
	.dc.b	$53,$CE,$31,$C7,$F8,$3F,$95,$91,$F8,$0F,$95,$B5,$F5,$E6,$E3,$E1
	.dc.b	$E3,$56,$2F,$FB,$2C,$3E,$4F,$F1,$DF,$27,$F8,$F6,$CF,$E3,$1B,$8F
	.dc.b	$E0,$DB,$87,$B7,$37,$07,$C0,$BF,$8C,$DB,$FF,$33,$CB,$87,$11,$F1
	.dc.b	$26,$CF,$2A,$FF,$5F,$36,$7F,$D8,$AD,$7F,$DF,$AD,$7B,$C7,$C9,$E1
	.dc.b	$E7,$5D,$1F,$D5,$79,$72,$7F,$4F,$F2,$B6,$FF,$3D,$37,$2F,$91,$36
	.dc.b	$BE,$3F,$C6,$7E,$CF,$E5,$7F,$68,$D8,$FD,$8B,$70,$FD,$43,$6F,$E3
	.dc.b	$6D,$EE,$3D,$FB,$5B,$FC,$04,$5B,$FC,$8C,$DA,$E3,$5A,$3F,$0D,$19
	.dc.b	$7F,$29,$56,$BE,$7C,$D7,$F2,$31,$75,$FC,$05,$73,$9F,$CF,$CE,$6B
	.dc.b	$4D,$9D,$5F,$8A,$B7,$1E,$23,$27,$88,$AF,$89,$AF,$89,$B7,$E2,$8E
	.dc.b	$64,$9E,$C7,$E3,$6F,$C7,$5C,$1B,$3B,$8F,$DA,$46,$27,$8C,$60,$EF
	.dc.b	$7E,$BA,$E0,$7E,$51,$C4,$F0,$8C,$4F,$73,$89,$5E,$27,$AC,$B7,$EB
	.dc.b	$9E,$51,$96,$C6,$67,$E3,$4F,$0A,$FC,$1C,$76,$8E,$77,$EB,$8D,$DF
	.dc.b	$C3,$74,$F4,$8F,$87,$3F,$A4,$AF,$E1,$84,$AF,$D5,$B6,$3F,$56,$D7
	.dc.b	$84,$7F,$11,$CF,$F0,$C4,$7E,$BB,$63,$DE,$DB,$F6,$B6,$7C,$E7,$F2
	.dc.b	$77,$13,$F3,$0F,$E5,$AE,$1F,$8D,$1B,$3B,$91,$EE,$D9,$EE,$D3,$BC
	.dc.b	$7F,$0E,$9E,$98,$2F,$F8,$6C,$61,$B5,$C6,$BC,$97,$FC,$6E,$B5,$3F
	.dc.b	$0C,$D4,$F9,$8E,$CE,$2F,$B3,$B8,$F3,$92,$BC,$ED,$8F,$3B,$FD,$12
	.dc.b	$BB,$BF,$12,$3A,$5F,$9A,$38,$1E,$68,$F8,$CE,$0F,$21,$CD,$E2,$78
	.dc.b	$EB,$FA,$97,$C0,$3C,$CF,$E6,$2D,$95,$7F,$B1,$E1,$F2,$BA,$A9,$5A
	.dc.b	$BE,$71,$CA,$73,$7C,$7E,$57,$AF,$C2,$76,$6E,$1C,$4F,$F5,$AE,$28
	.dc.b	$CD,$71,$9D,$3E,$49,$57,$A3,$E1,$25,$6A,$95,$FE,$BC,$8E,$B3,$A9
	.dc.b	$FF,$61,$9C,$DF,$EE,$8E,$A3,$2E,$5A,$F4,$8F,$8F,$ED,$F3,$CF,$E5
	.dc.b	$BF,$2B,$E1,$25,$F8,$F0,$7C,$26,$31,$F9,$95,$70,$FA,$73,$6F,$E1
	.dc.b	$FC,$17,$D4,$F5,$3E,$AB,$63,$A3,$70,$FA,$CD,$BF,$F0,$77,$8E,$F8
	.dc.b	$9F,$ED,$AD,$BF,$27,$CA,$E2,$64,$7F,$A5,$1F,$F4,$31,$1C,$1E,$BF
	.dc.b	$33,$78,$FE,$15,$FE,$F3,$E8,$7F,$BB,$21,$8F,$F8,$7E,$30,$DF,$13
	.dc.b	$7F,$01,$30,$FB,$15,$A7,$FD,$EE,$A7,$57,$D2,$73,$E2,$F8,$3C,$5F
	.dc.b	$0B,$63,$9B,$67,$81,$F9,$CB,$FF,$1B,$B6,$BD,$3E,$64,$BF,$1F,$99
	.dc.b	$6B,$88,$B5,$7F,$A4,$89,$5F,$E9,$76,$CF,$09,$FF,$33,$CF,$BA,$7E
	.dc.b	$3F,$99,$F2,$3C,$59,$3F,$0B,$C5,$B8,$71,$9B,$B7,$CB,$9B,$5F,$1B
	.dc.b	$C5,$FA,$3E,$4F,$D2,$6C,$7C,$C6,$E1,$E3,$6D,$FD,$5F,$27,$EC,$9F
	.dc.b	$6B,$C9,$B7,$F5,$23,$FD,$5E,$47,$FD,$68,$DB,$F7,$78,$BC,$1B,$3F
	.dc.b	$B3,$F5,$1F,$2B,$FA,$E9,$FE,$1B,$18,$9F,$D6,$B6,$3F,$62,$D7,$F6
	.dc.b	$FD,$DF,$EA,$E7,$F6,$AD,$73,$E7,$F5,$7F,$80,$BF,$EC,$6F,$1A,$5F
	.dc.b	$FB,$2D,$B3,$DB,$E3,$F8,$60,$69,$2B,$F8,$0E,$1F,$EB,$4A,$F1,$F6
	.dc.b	$F7,$8C,$47,$F1,$A8,$FF,$6A,$F9,$12,$BF,$DB,$4D,$9F,$6F,$C2,$78
	.dc.b	$09,$3F,$09,$E9,$F2,$72,$3C,$2D,$9F,$F6,$FF,$9D,$FE,$E2,$24,$FF
	.dc.b	$B9,$DF,$5F,$C8,$48,$F9,$AD,$9F,$93,$F1,$DE,$92,$4F,$B9,$FF,$76
	.dc.b	$F9,$F2,$3F,$DE,$4D,$9F,$E4,$BF,$23,$F1,$FC,$FF,$E4,$AD,$8C,$FC
	.dc.b	$AB,$D8,$CF,$0F,$F4,$F9,$FE,$5B,$F2,$4D,$3F,$96,$C2,$30,$FF,$30
	.dc.b	$6B,$C2,$37,$6F,$91,$A4,$9F,$E6,$F1,$97,$F9,$CF,$92,$F4,$3B,$B6
	.dc.b	$33,$18,$33,$BC,$8E,$2D,$7F,$A2,$FC,$FF,$F8,$2D,$FE,$8A,$D7,$3D
	.dc.b	$1E,$C8,$C4,$7F,$50,$E8,$FE,$AA,$FE,$83,$A3,$E0,$C6,$23,$F4,$31
	.dc.b	$F1,$0F,$53,$FA,$0C,$F0,$47,$FA,$AF,$8A,$F5,$BE,$5B,$63,$31,$83
	.dc.b	$A7,$B5,$23,$F3,$AD,$7F,$B0,$CB,$E7,$79,$92,$3B,$BF,$63,$E4,$9E
	.dc.b	$AF,$F8,$22,$7F,$93,$FC,$9B,$A7,$1E,$2F,$FB,$86,$31,$3F,$DD,$EF
	.dc.b	$8B,$7D,$6D,$F6,$73,$9C,$CF,$F3,$26,$DF,$FB,$AD,$57,$09,$FF,$0C
	.dc.b	$F9,$8F,$47,$10,$F1,$1F,$05,$86,$C7,$F8,$26,$B8,$37,$D2,$E3,$18
	.dc.b	$8F,$E8,$6D,$CB,$FD,$F8,$FF,$57,$8F,$F0,$97,$C9,$FF,$85,$9F,$F0
	.dc.b	$A5,$BE,$33,$C0,$79,$27,$53,$FD,$39,$FC,$A3,$7D,$AD,$1B,$EC,$E1
	.dc.b	$F1,$D6,$7B,$C6,$1F,$0F,$1F,$B7,$A6,$46,$0F,$2F,$07,$86,$64,$60
	.dc.b	$F4,$FB,$7F,$DE,$9F,$F1,$1E,$5F,$E1,$AD,$BF,$C4,$6D,$8C,$CE,$DF
	.dc.b	$F1,$6C,$3D,$A7,$C9,$F0,$6C,$F4,$6B,$C1,$AF,$F8,$CB,$E4,$FF,$82
	.dc.b	$A6,$91,$FF,$1A,$F8,$EE,$7F,$F2,$14,$7F,$11,$6B,$C4,$7C,$9F,$F7
	.dc.b	$A7,$F3,$13,$C5,$FE,$FE,$7F,$84,$CF,$CE,$73,$E7,$4F,$F2,$3C,$DB
	.dc.b	$FF,$25,$CD,$AF,$D1,$73,$57,$FC,$D6,$DE,$57,$9B,$99,$FF,$0F,$5F
	.dc.b	$A3,$DB,$C1,$9B,$B1,$EC,$F6,$F8,$18,$AF,$3B,$DF,$3F,$F3,$04,$7F
	.dc.b	$9D,$79,$BC,$1F,$27,$97,$5C,$93,$0F,$F3,$45,$F0,$BF,$F1,$CC,$FA
	.dc.b	$7E,$4F,$E5,$47,$F8,$89,$B3,$F1,$E3,$F5,$9F,$19,$BE,$0E,$1C,$13
	.dc.b	$1F,$F3,$A6,$1C,$1F,$E0,$CF,$CD,$B7,$C9,$FF,$99,$A7,$FC,$F4,$D8
	.dc.b	$FF,$18,$BF,$8B,$8C,$F3,$FC,$D8,$D4,$FF,$9F,$1A,$FF,$13,$6B,$FE
	.dc.b	$9C,$6A,$F9,$36,$DF,$E0,$96,$DF,$FA,$01,$B3,$FE,$84,$6B,$FE,$8A
	.dc.b	$5F,$1D,$C4,$93,$C5,$E9,$D7,$23,$A3,$6B,$FD,$1F,$F2,$5D,$1F,$E5
	.dc.b	$4D,$7F,$D6,$6D,$4D,$3C,$33,$FE,$96,$6C,$74,$AF,$F1,$24,$62,$3F
	.dc.b	$D4,$2F,$FA,$9A,$78,$56,$88,$FF,$A8,$3D,$87,$CE,$7F,$8B,$5B,$19
	.dc.b	$8C,$19,$F9,$79,$1C,$5B,$87,$FA,$B2,$BF,$C0,$13,$D7,$D3,$E8,$47
	.dc.b	$FD,$5F,$F2,$5E,$5A,$FF,$59,$B5,$FF,$5C,$B5,$FE,$A0,$D7,$FD,$B3
	.dc.b	$FA,$47,$F0,$39,$FF,$5C,$B6,$3F,$D7,$AD,$7F,$D8,$6D,$7F,$AD,$7D
	.dc.b	$A3,$E5,$3F,$C9,$8D,$AF,$F6,$1B,$67,$FD,$8E,$D7,$9F,$BF,$FE,$53
	.dc.b	$BF,$F2,$D4,$7F,$65,$6B,$E6,$8D,$7C,$FC,$BF,$3B,$67,$CF,$EC,$F3
	.dc.b	$24,$FB,$1E,$DE,$DC,$8F,$7B,$67,$DF,$ED,$78,$C9,$3F,$11,$F3,$7C
	.dc.b	$DC,$8F,$8E,$D9,$F8,$FE,$1F,$AC,$93,$F3,$1F,$77,$14,$8F,$9E,$D9
	.dc.b	$F9,$FC,$5F,$0C,$93,$F5,$1F,$8D,$C3,$FC,$9F,$6F,$C2,$D7,$FA,$C1
	.dc.b	$FE,$D1,$5F,$EF,$D7,$3D,$DF,$C4,$5D,$CF,$DA,$AE,$75,$F8,$06,$C7
	.dc.b	$3E,$8E,$D1,$DD,$F0,$73,$F8,$79,$AD,$F7,$6D,$F8,$DB,$8F,$85,$F4
	.dc.b	$7C,$F4,$AF,$37,$2D,$BF,$54,$DB,$F4,$4D,$AF,$C2,$77,$FC,$13,$E1
	.dc.b	$7A,$16,$3E,$04,$7E,$2C,$8F,$E7,$1B,$87,$C7,$F9,$7E,$0D,$9F,$91
	.dc.b	$F2,$FF,$D4,$FF,$49,$7C,$17,$CF,$8C,$36,$55,$FE,$E0,$AD,$C7,$23
	.dc.b	$39,$FF,$41,$8C,$CE,$FE,$4F,$B1,$DF,$E4,$ED,$F0,$3F,$C8,$D4,$9C
	.dc.b	$D7,$7A,$B7,$FD,$37,$BB,$F5,$CD,$DB,$D5,$EA,$7F,$6D,$9C,$E5,$FC
	.dc.b	$0A,$7F,$D8,$51,$FB,$3C,$31,$3F,$C3,$27,$27,$C3,$F5,$39,$36,$3C
	.dc.b	$D3,$F0,$B3,$FE,$C8,$7F,$97,$A3,$30,$8C,$F9,$3F,$37,$CA,$F9,$CD
	.dc.b	$AD,$CF,$1A,$F9,$9E,$A7,$D3,$3E,$A7,$91,$6B,$71,$67,$E9,$4D,$D7
	.dc.b	$E8,$7A,$5F,$6C,$FB,$9F,$27,$70,$EF,$1F,$EE,$3C,$7F,$B4,$61,$FE
	.dc.b	$75,$3D,$91,$92,$4F,$9B,$E4,$F9,$4F,$E0,$5F,$C4,$7C,$9F,$E2,$7D
	.dc.b	$3B,$5E,$C6,$D7,$E1,$DB,$97,$E1,$5F,$C8,$79,$52,$BE,$1F,$C4,$DB
	.dc.b	$FC,$74,$DB,$FC,$64,$DA,$FE,$67,$D7,$FE,$2B,$F2,$FF,$CF,$31,$F4
	.dc.b	$97,$66,$66,$4E,$5E,$11,$8C,$58,$7F,$86,$6F,$9C,$62,$FB,$9D,$6B
	.dc.b	$85,$6E,$EC,$FF,$46,$9B,$5F,$A8,$BE,$77,$BA,$E4,$78,$C9,$3D,$CE
	.dc.b	$17,$E6,$8C,$CF,$19,$D0,$AE,$B3,$E1,$36,$7A,$23,$71,$A7,$C9,$7E
	.dc.b	$67,$95,$D9,$F9,$35,$CA,$7C,$6D,$7C,$15,$BA,$E0,$D9,$D5,$72,$AF
	.dc.b	$0D,$65,$F1,$D5,$BF,$61,$F4,$5C,$5F,$A6,$30,$F4,$AE,$ED,$3B,$CF
	.dc.b	$0A,$F3,$9D,$A3,$4F,$89,$FA,$2D,$25,$78,$A3,$E5,$46,$1F,$11,$1F
	.dc.b	$92,$F4,$8E,$11,$83,$81,$E2,$72,$FC,$88,$E0,$73,$70,$70,$3E,$9B
	.dc.b	$FE,$0C,$AF,$F0,$79,$1C,$12,$3F,$C1,$F1,$96,$C7,$E8,$FB,$3E,$E1
	.dc.b	$1F,$AA,$D8,$F0,$60,$8C,$5F,$A2,$31,$5C,$09,$5C,$2B,$71,$EB,$9D
	.dc.b	$CE,$63,$75,$CE,$B7,$3F,$16,$7F,$C5,$B1,$88,$F8,$27,$E6,$1C,$57
	.dc.b	$E6,$63,$F4,$11,$8B,$FE,$04,$DA,$F8,$07,$80,$F2,$39,$9E,$91,$97
	.dc.b	$C4,$7E,$DE,$35,$7F,$C0,$DB,$3E,$63,$84,$F9,$A7,$51,$8B,$F3,$1F
	.dc.b	$0A,$3E,$04,$66,$FC,$ED,$3C,$F5,$F1,$C9,$3C,$1F,$EE,$1F,$0D,$FD
	.dc.b	$DF,$E0,$FF,$BC,$36,$BF,$A5,$B6,$77,$1F,$D3,$A7,$FB,$F7,$C1,$48
	.dc.b	$FE,$A3,$F2,$F6,$BF,$BE,$4D,$C3,$FB,$D4,$DA,$FF,$00,$D7,$FB,$AF
	.dc.b	$B3,$FD,$B5,$06,$F6,$E1,$91,$FD,$36,$7F,$C1,$2E,$CC,$3C,$DA,$65
	.dc.b	$E0,$DB,$FF,$66,$2E,$CE,$B3,$FC,$8A,$30,$7F,$6C,$F9,$7F,$F7,$77
	.dc.b	$C6,$F8,$27,$FC,$31,$18,$7F,$C4,$0D,$9E,$07,$F7,$A9,$E1,$7F,$ED
	.dc.b	$B7,$FD,$AD,$5F,$E2,$26,$A7,$FD,$B0,$60,$9E,$BF,$8B,$D7,$F1,$BF
	.dc.b	$23,$A6,$61,$FE,$08,$3F,$DB,$98,$7B,$A3,$38,$7C,$27,$FD,$D8,$FD
	.dc.b	$29,$FE,$F2,$FF,$79,$6B,$98,$FA,$BD,$7C,$AB,$E4,$BF,$13,$D9,$97
	.dc.b	$EF,$F6,$AE,$1F,$D3,$33,$56,$30,$F8,$55,$6F,$F0,$74,$F6,$BE,$7F
	.dc.b	$C0,$7D,$06,$CF,$CB,$6E,$3D,$5B,$87,$DC,$EA,$FC,$59,$F8,$CE,$BB
	.dc.b	$7F,$16,$3E,$0D,$FE,$29,$AA,$11,$86,$9F,$8C,$7F,$2F,$CF,$D2,$7F
	.dc.b	$10,$FE,$1B,$E5,$65,$7E,$07,$E5,$6D,$7C,$39,$B8,$FE,$0E,$6D,$7C
	.dc.b	$2F,$95,$FC,$7F,$CA,$FE,$41,$B1,$F8,$B6,$E1,$F3,$5B,$7E,$98,$FF
	.dc.b	$36,$FD,$D7,$E4,$2B,$F5,$ED,$8F,$D4,$F3,$6E,$B8,$42,$74,$D3,$E2
	.dc.b	$BF,$A1,$7F,$C7,$CD,$7A,$9A,$3A,$57,$83,$96,$C7,$90,$A7,$77,$F5
	.dc.b	$BF,$2B,$F4,$B8,$8D,$25,$7E,$FA,$1F,$2D,$B2,$46,$5E,$5F,$2B,$23
	.dc.b	$83,$70,$FE,$15,$4F,$96,$F0,$12,$BB,$3E,$38,$91,$E1,$6C,$F2,$F9
	.dc.b	$2F,$41,$27,$CC,$FA,$FE,$5A,$47,$A5,$B3,$FC,$6F,$4F,$D2,$CF,$F1
	.dc.b	$A6,$C6,$7A,$7E,$89,$F4,$B0,$95,$B6,$CF,$F2,$5F,$A1,$79,$3F,$9C
	.dc.b	$6D,$7F,$26,$6C,$EE,$7F,$43,$F3,$EF,$90,$A4,$BF,$E5,$B1,$97,$F9
	.dc.b	$7F,$CB,$BE,$0F,$FD,$7F,$87,$F3,$58,$C7,$AF,$AF,$84,$72,$F0,$BD
	.dc.b	$DF,$FA,$1F,$57,$CC,$7F,$3A,$8F,$E4,$4D,$8C,$C7,$E0,$63,$1A,$48
	.dc.b	$FC,$23,$5F,$BB,$F2,$BC,$48,$FE,$25,$E9,$F2,$92,$3F,$18,$D7,$FA
	.dc.b	$77,$E9,$5F,$C1,$13,$49,$B7,$9D,$7C,$55,$24,$FF,$54,$8C,$B7,$E2
	.dc.b	$B8,$11,$C1,$BD,$BA,$11,$C1,$CF,$5F,$01,$1C,$1D,$FE,$2B,$C8,$47
	.dc.b	$07,$8F,$E2,$BD,$04,$70,$7F,$6A,$F9,$47,$BB,$FD,$DB,$C0,$FE,$0B
	.dc.b	$F2,$BF,$85,$AF,$F6,$AB,$5F,$F4,$4D,$70,$7F,$DB,$2B,$DE,$E5,$BF
	.dc.b	$F8,$02,$35,$3C,$30,$CA,$11,$C5,$FF,$71,$F9,$8F,$43,$70,$F6,$DE
	.dc.b	$C6,$1B,$1C,$27,$75,$FD,$4A,$30,$43,$FD,$42,$E5,$FF,$51,$FC,$A7
	.dc.b	$F8,$4D,$FF,$79,$D3,$E5,$7B,$1E,$29,$D4,$F0,$F5,$71,$EB,$48,$FF
	.dc.b	$7D,$CF,$F8,$3D,$BF,$03,$C6,$A4,$60,$DE,$BE,$BF,$87,$91,$83,$9F
	.dc.b	$07,$86,$64,$60,$EF,$F0,$5F,$ED,$27,$FC,$51,$8A,$31,$1F,$EF,$78
	.dc.b	$C7,$1F,$F8,$A9,$B1,$C3,$D9,$7C,$AE,$5B,$19,$8C,$1F,$B9,$5F,$2B
	.dc.b	$48,$FE,$F5,$FD,$F7,$87,$FD,$0A,$3F,$C4,$27,$E0,$63,$F0,$5F,$2A
	.dc.b	$E3,$FF,$23,$CE,$B9,$38,$1C,$6E,$F3,$FE,$27,$4F,$F0,$EF,$95,$D4
	.dc.b	$FE,$9F,$CC,$F9,$1C,$56,$CE,$BE,$E7,$8C,$CE,$5F,$EE,$40,$F9,$FD
	.dc.b	$C9,$3C,$67,$FC,$B7,$38,$7F,$DD,$78,$B9,$FF,$AF,$B6,$BF,$17,$F1
	.dc.b	$B8,$9C,$F2,$C9,$E9,$6B,$E0,$F4,$9F,$ED,$3F,$2B,$FE,$48,$3F,$C9
	.dc.b	$3A,$7F,$B9,$6B,$FB,$DC,$67,$7D,$9C,$37,$B8,$F6,$F9,$5F,$C2,$3F
	.dc.b	$E6,$A6,$E5,$FE,$6F,$6B,$F4,$9A,$FF,$9E,$9F,$F2,$7C,$5F,$80,$FE
	.dc.b	$FB,$18,$9F,$DE,$28,$FF,$00,$FC,$AB,$E4,$3F,$C6,$2D,$BF,$F3,$DB
	.dc.b	$6F,$FD,$00,$D7,$F0,$6B,$B3,$89,$27,$8B,$FE,$3C,$F9,$FF,$F2,$64
	.dc.b	$7F,$A1,$9B,$5F,$E8,$B6,$BF,$8E,$6B,$FE,$96,$F9,$6F,$03,$70,$FF
	.dc.b	$23,$C6,$23,$F8,$C3,$53,$E5,$AF,$F4,$AB,$63,$FD,$34,$D7,$F3,$AD
	.dc.b	$7F,$D4,$BF,$3E,$F8,$6F,$F2,$AB,$6B,$FD,$3E,$D9,$FF,$52,$35,$FD
	.dc.b	$3F,$C3,$3E,$5B,$FC,$BA,$DA,$FF,$53,$B6,$7F,$D5,$8D,$78,$FC,$AF
	.dc.b	$02,$4F,$07,$FC,$CF,$F3,$2F,$8F,$A4,$9F,$F5,$9C,$65,$FF,$5A,$FC
	.dc.b	$BF,$F9,$B5,$A7,$F9,$C1,$C5,$FF,$AF,$5A,$FF,$00,$F8,$F7,$C9,$FF
	.dc.b	$9D,$9B,$5F,$EC,$06,$CF,$FB,$15,$AF,$F0,$8F,$53,$F8,$04,$FF,$73
	.dc.b	$F0,$3F,$CF,$F1,$89,$FE,$FF,$1E,$3F,$52,$57,$A1,$B3,$E5,$F5,$3D
	.dc.b	$04,$9F,$6B,$D3,$E6,$24,$7F,$80,$5B,$3E,$DF,$53,$E0,$12,$7E,$2B
	.dc.b	$F0,$FD,$49,$1F,$21,$B3,$FE,$14,$E2,$F4,$92,$7F,$C1,$CF,$CB,$B9
	.dc.b	$1F,$41,$B3,$F3,$67,$E8,$12,$7E,$AB,$F1,$38,$64,$7F,$86,$5B,$3F
	.dc.b	$EF,$1F,$95,$F9,$FE,$07,$FA,$21,$B5,$FE,$FC,$F9,$5F,$F7,$73,$FE
	.dc.b	$FE,$8C,$D7,$F7,$D7,$EA,$B9,$E1,$FF,$58,$C6,$23,$51,$83,$FD,$0F
	.dc.b	$5F,$EF,$48,$FE,$49,$5C,$4F,$E4,$F1,$9B,$FF,$4D,$BF,$CA,$1B,$E1
	.dc.b	$F2,$F1,$39,$B6,$B8,$B6,$FA,$9F,$E1,$3B,$FF,$11,$CF,$6F,$43,$C6
	.dc.b	$79,$3D,$0B,$19,$25,$72,$15,$FC,$8D,$B5,$EC,$F3,$E5,$7A,$BC,$2B
	.dc.b	$5C,$6F,$FD,$E6,$F8,$EB,$FD,$AF,$18,$8E,$B1,$83,$E3,$FC,$AF,$2D
	.dc.b	$3E,$1B,$F2,$BE,$0B,$E5,$F9,$37,$2F,$92,$DC,$3C,$0D,$BF,$96,$FD
	.dc.b	$2E,$E9,$5F,$43,$E4,$6D,$F9,$E6,$DF,$D2,$9B,$5F,$E3,$DA,$76,$7F
	.dc.b	$8E,$23,$FA,$FB,$63,$E4,$3F,$D6,$E3,$83,$9A,$FF,$6B,$BB,$9F,$E0
	.dc.b	$DE,$BF,$F0,$52,$57,$F0,$57,$FD,$9F,$2F,$95,$B1,$87,$F1,$BF,$25
	.dc.b	$F8,$EF,$95,$B7,$F8,$C6,$E1,$F4,$DB,$8F,$E3,$9F,$CA,$FC,$AC,$AF
	.dc.b	$C9,$FC,$AD,$BF,$B5,$36,$FF,$2B,$36,$BE,$21,$FA,$8A,$FF,$31,$35
	.dc.b	$3F,$CF,$E7,$FD,$EE,$E4,$7B,$7F,$EF,$A9,$E6,$D7,$9B,$5E,$6D,$8E
	.dc.b	$5E,$DF,$FB,$D1,$A7,$EB,$3E,$47,$FD,$EF,$F0,$DE,$4B,$FC,$A4,$61
	.dc.b	$FD,$AB,$67,$81,$FE,$1A,$9E,$18,$7A,$AB,$FC,$FD,$1F,$EC,$49,$D7
	.dc.b	$A5,$DB,$D3,$EE,$C8,$C9,$CE,$31,$5F,$AD,$87,$3F,$A4,$F5,$75,$D8
	.dc.b	$FF,$15,$9A,$2C,$69,$A4,$9E,$91,$96,$C6,$71,$FA,$33,$96,$C6,$4F
	.dc.b	$B7,$19,$69,$99,$FB,$B1,$96,$C6,$5F,$BF,$19,$6C,$64,$F5,$3F,$06
	.dc.b	$BD,$CF,$98,$95,$F8,$68,$CB,$E6,$8C,$CF,$E4,$27,$2F,$9A,$32,$7E
	.dc.b	$4A,$32,$F9,$8C,$9F,$94,$72,$F9,$8C,$C7,$F1,$F3,$CF,$3E,$12,$3E
	.dc.b	$13,$D9,$5C,$9E,$0D,$3E,$13,$C8,$F3,$47,$29,$F3,$47,$23,$E3,$57
	.dc.b	$16,$99,$8F,$2C,$65,$B1,$99,$E9,$1E,$D8,$C1,$E0,$3D,$0F,$BA,$BD
	.dc.b	$15,$CE,$3F,$D2,$F3,$FC,$D4,$93,$FA,$F2,$3F,$E9,$C8,$F9,$87,$08
	.dc.b	$CB,$F9,$DE,$AF,$88,$49,$FC,$13,$63,$E7,$B6,$FD,$33,$FE,$6F,$36
	.dc.b	$49,$DB,$63,$73,$C6,$3D,$D3,$FC,$4A,$30,$FE,$06,$7F,$88,$C6,$23
	.dc.b	$F0,$11,$89,$E8,$72,$8F,$E3,$33,$B8,$FB,$51,$D5,$DB,$D5,$FE,$47
	.dc.b	$1B,$AF,$E2,$8D,$8F,$DD,$B6,$7E,$34,$72,$7C,$6F,$23,$07,$81,$F1
	.dc.b	$B8,$9E,$AD,$3A,$96,$3A,$9F,$20,$95,$F2,$23,$4F,$C1,$8C,$C7,$C1
	.dc.b	$3E,$93,$FE,$03,$C9,$EC,$7F,$2C,$3F,$96,$FA,$F6,$BF,$94,$45,$AE
	.dc.b	$13,$6F,$F9,$54,$2F,$FD,$37,$18,$7F,$DD,$17,$FC,$C7,$D8,$FE,$73
	.dc.b	$EC,$FF,$3A,$6C,$64,$FE,$6D,$1F,$D0,$7A,$64,$7F,$28,$EA,$B8,$FF
	.dc.b	$2F,$9F,$E9,$04,$9F,$CD,$47,$F5,$68,$C3,$FC,$D7,$9F,$8E,$5F,$D6
	.dc.b	$1F,$E9,$7F,$2B,$2F,$FA,$37,$CA,$DB,$F8,$D3,$6F,$D9,$36,$BF,$C1
	.dc.b	$5A,$3E,$53,$FA,$AF,$CA,$FF,$56,$6C,$FF,$51,$6E,$3F,$45,$B8,$7C
	.dc.b	$AF,$8F,$FE,$AB,$5F,$E1,$56,$D7,$F7,$2E,$7F,$EB,$15,$B8,$B5,$B7
	.dc.b	$FB,$65,$7F,$65,$7F,$C3,$C1,$C3,$B8,$F8,$F1,$DA,$35,$CF,$D1,$1C
	.dc.b	$4C,$1A,$9C,$D7,$E0,$BD,$17,$C1,$E9,$B3,$A3,$FC,$60,$BD,$FD,$4C
	.dc.b	$9D,$5F,$82,$B8,$BF,$E5,$97,$84,$6A,$11,$AB,$F1,$7B,$5E,$1F,$73
	.dc.b	$C4,$DB,$E1,$5F,$EF,$BA,$F4,$56,$A7,$2F,$A7,$E6,$24,$70,$F3,$6E
	.dc.b	$3E,$39,$FF,$5C,$92,$7F,$23,$82,$31,$3E,$9F,$67,$8E,$3F,$1D,$F8
	.dc.b	$DE,$34,$C3,$CB,$E3,$5B,$F8,$13,$6F,$E7,$4D,$AF,$97,$F3,$1F,$53
	.dc.b	$F4,$B7,$C7,$FD,$18,$C3,$67,$E4,$4E,$9F,$87,$3A,$7E,$67,$A5,$FE
	.dc.b	$5F,$CB,$FD,$4D,$9F,$5F,$99,$FC,$0C,$E6,$4F,$FB,$B5,$FF,$5A,$4F
	.dc.b	$FA,$A8,$95,$FB,$49,$E1,$87,$F9,$26,$3F,$D6,$93,$E9,$F7,$FF,$C7
	.dc.b	$BF,$31,$FE,$45,$EB,$34,$95,$FB,$E8,$7C,$C6,$C9,$19,$7F,$C9,$D8
	.dc.b	$7F,$95,$23,$F8,$2B,$72,$FE,$C0,$D7,$F4,$ED,$7F,$7B,$5E,$32,$4F
	.dc.b	$85,$F3,$7C,$BC,$8F,$23,$67,$AF,$97,$E2,$24,$F8,$9F,$DB,$F8,$F2
	.dc.b	$3F,$66,$D9,$FE,$E9,$D9,$FC,$B6,$7F,$6F,$87,$EF,$88,$FF,$20,$F8
	.dc.b	$97,$BB,$FE,$6D,$6C,$66,$30,$7F,$9C,$7E,$55,$F2,$74,$93,$FC,$AA
	.dc.b	$32,$FF,$2C,$F9,$57,$C8,$FE,$3B,$3F,$F0,$44,$63,$9F,$5F,$04,$F0
	.dc.b	$F8,$5E,$EF,$F8,$9F,$CE,$3E,$03,$F9,$BC,$23,$0F,$F3,$A6,$BF,$E1
	.dc.b	$56,$BF,$C7,$37,$F4,$BF,$83,$D7,$9C,$62,$3F,$C3,$8D,$78,$7C,$AE
	.dc.b	$88,$FE,$2D,$F1,$F2,$BE,$06,$92,$7F,$12,$FA,$3C,$2F,$F4,$D4,$7F
	.dc.b	$12,$E0,$C1,$1F,$E0,$7F,$29,$B2,$38,$3F,$27,$F0,$5E,$02,$38,$3F
	.dc.b	$85,$F0,$77,$23,$83,$F8,$7E,$FE,$32,$38,$3F,$3F,$F0,$5E,$C2,$38
	.dc.b	$3F,$8B,$FC,$2F,$AC,$8E,$0F,$D3,$FB,$3F,$10,$8E,$0F,$ED,$FF,$26
	.dc.b	$C7,$FD,$D9,$E3,$7F,$81,$3E,$53,$E9,$37,$8F,$3B,$7F,$BC,$6F,$86
	.dc.b	$1F,$CE,$D6,$7A,$8F,$EA,$31,$A5,$7F,$E1,$08,$C3,$67,$84,$BD,$77
	.dc.b	$B9,$8F,$82,$79,$74,$DC,$EA,$BF,$A6,$3F,$E2,$46,$D6,$8D,$F8,$3D
	.dc.b	$BE,$53,$FD,$BF,$3F,$D4,$F0,$FE,$C2,$DF,$47,$5E,$FE,$FC,$5B,$3F
	.dc.b	$D6,$6F,$8B,$EC,$DF,$FB,$7A,$84,$F7,$E9,$E5,$3C,$63,$0F,$3E,$3F
	.dc.b	$77,$A3,$23,$07,$7E,$0F,$44,$C8,$C1,$E3,$E5,$F9,$AF,$FA,$0F,$14
	.dc.b	$62,$3F,$C1,$71,$8E,$2F,$F1,$7B,$63,$87,$BC,$F9,$4C,$B6,$33,$18
	.dc.b	$3F,$80,$AF,$1E,$91,$FE,$0A,$FF,$06,$F0,$FF,$C2,$31,$FE,$F7,$3E
	.dc.b	$41,$FE,$1E,$8F,$F0,$BF,$CA,$7E,$06,$3E,$5C,$F0,$E5,$9C,$3F,$E0
	.dc.b	$99,$E5,$9F,$F9,$69,$FC,$3C,$67,$97,$B9,$FE,$20,$E5,$7C,$66,$27
	.dc.b	$97,$C0,$F9,$0C,$5F,$90,$FF,$14,$BE,$8F,$81,$93,$C0,$F1,$3F,$E2
	.dc.b	$CC,$7F,$88,$47,$F8,$75,$B5,$CF,$0F,$F1,$0B,$D6,$32,$7F,$9C,$8F
	.dc.b	$D9,$4F,$1E,$2E,$9C,$DD,$5F,$F1,$CF,$CA,$7C,$09,$FC,$7B,$6F,$FC
	.dc.b	$7F,$87,$F7,$D8,$C4,$1F,$8A,$FB,$5B,$87,$FA,$09,$D9,$8A,$FC,$DF
	.dc.b	$CA,$7E,$29,$FF,$1C,$3F,$E9,$29,$FF,$44,$B9,$AF,$90,$E6,$78,$FC
	.dc.b	$27,$D5,$9F,$F2,$9C,$FF,$A6,$9F,$F2,$EB,$5D,$CF,$FA,$0B,$E5,$1D
	.dc.b	$3C,$65,$18,$9D,$4E,$63,$13,$FE,$E3,$F9,$46,$DC,$5B,$7F,$E8,$E6
	.dc.b	$CE,$E7,$5F,$2D,$2B,$4D,$9F,$F7,$77,$CA,$F6,$7F,$CE,$AD,$7F,$80
	.dc.b	$B5,$FF,$38,$B5,$7C,$AB,$D8,$F6,$36,$FF,$D4,$2D,$9F,$F7,$E3,$5F
	.dc.b	$F4,$1F,$D0,$7A,$1B,$8F,$C5,$8C,$47,$F1,$26,$BF,$E8,$2F,$8A,$FE
	.dc.b	$F1,$7F,$EA,$E6,$C7,$FA,$38,$38,$7F,$D2,$2D,$7F,$8E,$FC,$B7,$F9
	.dc.b	$1D,$FF,$4A,$B5,$CF,$B9,$F5,$E3,$11,$FE,$92,$DD,$F1,$9F,$6D,$B8
	.dc.b	$7F,$B0,$1B,$3E,$36,$BF,$EA,$2F,$8F,$7C,$27,$E0,$DB,$5F,$EC,$86
	.dc.b	$CF,$AD,$AF,$FA,$AB,$D2,$FE,$03,$3F,$E7,$EE,$CF,$C6,$46,$27,$E1
	.dc.b	$35,$FF,$58,$FC,$07,$FA,$0E,$BF,$CC,$5F,$09,$F9,$38,$C4,$FC,$96
	.dc.b	$BF,$EB,$8F,$96,$7A,$9F,$99,$6D,$FF,$B7,$9B,$3F,$45,$AF,$9F,$E5
	.dc.b	$7C,$A4,$9F,$F4,$AB,$FE,$64,$F2,$24,$7F,$99,$1B,$3C,$FD,$4F,$01
	.dc.b	$27,$C0,$FC,$0F,$95,$91,$FE,$A0,$6C,$FF,$9B,$3B,$BF,$CE,$24,$9F
	.dc.b	$F3,$8B,$EB,$99,$1E,$B6,$CF,$CA,$F7,$3E,$69,$27,$FD,$5A,$FF,$9F
	.dc.b	$3E,$52,$47,$F9,$F1,$B3,$F6,$7D,$4F,$F6,$BF,$17,$F8,$02,$BF,$D9
	.dc.b	$0F,$FA,$64,$C4,$FF,$37,$6B,$F8,$79,$FC,$31,$27,$F0,$CF,$4F,$88
	.dc.b	$EA,$D8,$FF,$6A,$7C,$AB,$E4,$7F,$DC,$EB,$E5,$2C,$61,$B3,$C1,$B8
	.dc.b	$7F,$49,$AE,$13,$D3,$E5,$39,$9D,$3A,$AC,$64,$95,$C6,$32,$FF,$53
	.dc.b	$8F,$F5,$07,$8E,$7F,$C1,$E6,$0B,$E1,$7F,$BF,$9C,$48,$E1,$EB,$FF
	.dc.b	$2B,$8F,$F5,$34,$65,$AE,$9F,$E0,$8D,$3C,$5E,$BF,$59,$EC,$FC,$6F
	.dc.b	$8F,$F2,$4F,$F6,$9F,$82,$E8,$FC,$38,$CE,$3A,$84,$EA,$31,$A2,$B7
	.dc.b	$8F,$F8,$B2,$B4,$FF,$8B,$A7,$4F,$B6,$BA,$4E,$AF,$E0,$3F,$D8,$89
	.dc.b	$E9,$74,$8C,$FA,$75,$F1,$B6,$77,$7D,$4F,$14,$7C,$DE,$99,$19,$AF
	.dc.b	$95,$EC,$61,$FB,$13,$F3,$C9,$3F,$30,$38,$8F,$BD,$D3,$C7,$2F,$8D
	.dc.b	$5F,$DA,$E7,$D7,$EA,$79,$4F,$37,$A9,$67,$C5,$3F,$32,$70,$FD,$39
	.dc.b	$B5,$F1,$BC,$AF,$77,$B7,$EF,$6E,$1C,$B5,$FE,$CB,$5F,$32,$B5,$39
	.dc.b	$7F,$1D,$F2,$92,$3C,$27,$F4,$A2,$B9,$C7,$FA,$74,$23,$F2,$44,$7E
	.dc.b	$23,$F9,$9E,$8E,$17,$F0,$9F,$A9,$F2,$92,$BC,$DF,$29,$70,$FC,$84
	.dc.b	$DB,$FC,$1C,$DA,$FC,$57,$C9,$FD,$DF,$94,$FB,$CD,$8F,$80,$DC,$3F
	.dc.b	$34,$DB,$FC,$C7,$C6,$FE,$36,$4D,$AF,$C9,$C7,$F9,$26,$34,$47,$F6
	.dc.b	$9C,$FF,$8F,$8C,$4F,$D8,$79,$46,$66,$4E,$5E,$2D,$23,$DC,$C9,$5C
	.dc.b	$D7,$A2,$32,$D8,$C9,$FE,$98,$AF,$0B,$89,$FE,$11,$1E,$67,$F8,$95
	.dc.b	$79,$1E,$C4,$AF,$CB,$C6,$5E,$D1,$9A,$FE,$2F,$39,$7C,$11,$4F,$E2
	.dc.b	$51,$5F,$9A,$58,$F9,$B3,$D0,$8F,$23,$F2,$33,$C6,$BD,$13,$E4,$9F
	.dc.b	$65,$74,$69,$D0,$E4,$F9,$CF,$82,$7C,$87,$31,$E0,$8C,$C7,$80,$EB
	.dc.b	$3E,$07,$DE,$62,$3D,$55,$E5,$6C,$64,$F1,$C6,$5A,$66,$3C,$87,$F4
	.dc.b	$18,$D3,$88,$F5,$3E,$43,$EA,$9F,$ED,$43,$E0,$1F,$E3,$F9,$FE,$6A
	.dc.b	$E5,$AF,$C2,$30,$7F,$9C,$5C,$BD,$4C,$C7,$E6,$79,$7E,$81,$1F,$86
	.dc.b	$D8,$F4,$CF,$FA,$50,$FD,$2C,$66,$FE,$64,$FE,$B2,$31,$5F,$40,$FC
	.dc.b	$B3,$88,$F7,$4F,$ED,$23,$11,$EE,$8C,$4F,$88,$91,$E2,$9D,$D7,$8A
	.dc.b	$73,$5E,$28,$F3,$57,$ED,$EB,$6D,$AD,$9E,$57,$F4,$F1,$A8,$F8,$86
	.dc.b	$A3,$CE,$FA,$8C,$3E,$77,$0F,$95,$CC,$F9,$63,$04,$AF,$35,$77,$8F
	.dc.b	$E2,$0D,$8F,$83,$1B,$AF,$33,$F3,$5E,$AF,$AC,$D1,$FE,$07,$F2,$9F
	.dc.b	$03,$FD,$95,$B3,$FD,$39,$FF,$6D,$D7,$F7,$B7,$35,$FE,$E9,$6C,$66
	.dc.b	$B9,$7C,$A7,$37,$A7,$CA,$75,$F4,$2E,$3D,$5B,$8F,$36,$E1,$DD,$F2
	.dc.b	$79,$D2,$BC,$9E,$85,$BF,$14,$FF,$9F,$89,$3F,$E1,$CC,$FF,$CE,$F3
	.dc.b	$FE,$25,$0F,$F9,$F3,$0F,$67,$A5,$CD,$B5,$F0,$7C,$6F,$88,$7C,$5F
	.dc.b	$1E,$C6,$49,$87,$C3,$8C,$BF,$17,$E5,$3F,$C2,$1F,$14,$F8,$DF,$9D
	.dc.b	$18,$6D,$FF,$7C,$9D,$3F,$46,$74,$FF,$80,$3D,$36,$7F,$E1,$B6,$C7
	.dc.b	$1F,$94,$C4,$C8,$FF,$1E,$CF,$FA,$DB,$13,$49,$3E,$6F,$27,$84,$B5
	.dc.b	$FF,$1D,$65,$FE,$D7,$6F,$CA,$FB,$0F,$6F,$C0,$DC,$BD,$51,$6F,$94
	.dc.b	$7D,$1B,$56,$6F,$AF,$B3,$F1,$BC,$0F,$8E,$DB,$F8,$8D,$C3,$99,$FE
	.dc.b	$5F,$C3,$FC,$24,$E0,$3F,$E1,$19,$F9,$9E,$07,$D2,$3E,$9F,$9F,$63
	.dc.b	$27,$F9,$8A,$7F,$D0,$8D,$9F,$F2,$D5,$7A,$5A,$F8,$5F,$BB,$E7,$F3
	.dc.b	$8F,$15,$23,$F0,$3E,$7C,$BF,$BD,$DF,$6F,$73,$74,$F9,$F3,$6B,$F3
	.dc.b	$1F,$A6,$7F,$84,$C7,$7F,$F9,$6D,$C3,$70,$C3,$67,$FC,$8F,$3C,$1B
	.dc.b	$E6,$F1,$39,$79,$56,$32,$7F,$A1,$25,$3F,$02,$D1,$89,$D4,$E6,$31
	.dc.b	$3F,$4D,$79,$5D,$63,$0F,$F9,$C7,$E5,$7A,$79,$46,$93,$0F,$A9,$E8
	.dc.b	$F8,$78,$3F,$D3,$24,$7C,$1E,$64,$9F,$43,$71,$FE,$C8,$7F,$A4,$BE
	.dc.b	$57,$B9,$27,$FD,$43,$8A,$3B,$7A,$52,$7F,$08,$DA,$E7,$A3,$CE,$F8
	.dc.b	$04,$AF,$90,$FF,$6B,$C2,$47,$FA,$BB,$CA,$FF,$51,$B6,$3E,$67,$C4
	.dc.b	$3E,$56,$92,$7F,$D6,$B1,$97,$EA,$7A,$1F,$3C,$8E,$0F,$F5,$DF,$CA
	.dc.b	$3E,$47,$F8,$86,$9F,$0A,$31,$E3,$6B,$DE,$FC,$5C,$E6,$B6,$FF,$82
	.dc.b	$FE,$51,$ED,$FF,$B1,$A1,$18,$7F,$D9,$2D,$7E,$7B,$5F,$F0,$A5,$7F
	.dc.b	$BB,$5F,$F6,$7B,$5F,$AA,$D7,$7E,$07,$7D,$9F,$29,$49,$3F,$61,$F2
	.dc.b	$73,$BE,$32,$92,$7F,$78,$FA,$79,$1E,$BD,$24,$FE,$7D,$F7,$79,$EF
	.dc.b	$94,$A4,$9F,$E9,$8F,$83,$E3,$64,$7F,$09,$6B,$F8,$6F,$63,$C0,$47
	.dc.b	$F5,$8F,$6E,$19,$1F,$D5,$DA,$FE,$63,$37,$C8,$FB,$89,$1F,$C5,$DF
	.dc.b	$67,$B5,$23,$F7,$0D,$79,$7B,$BF,$18,$8F,$F1,$F7,$E2,$7C,$E4,$8F
	.dc.b	$E0,$0D,$7F,$53,$F2,$6C,$FF,$B5,$79,$3F,$EF,$DF,$95,$7C,$AE,$3B
	.dc.b	$3F,$B7,$46,$30,$FF,$6B,$E1,$C5,$BA,$6D,$08,$D3,$FC,$ED,$B8,$2F
	.dc.b	$FB,$D4,$7C,$49,$FF,$71,$F8,$3F,$C5,$9F,$F7,$2F,$A2,$F6,$7B,$47
	.dc.b	$F5,$06,$CE,$E3,$FA,$42,$F2,$F3,$E5,$C9,$EB,$1C,$BE,$55,$E5,$74
	.dc.b	$C7,$FC,$53,$5E,$0F,$7F,$53,$BF,$26,$CF,$FB,$E2,$F8,$B6,$3A,$76
	.dc.b	$EE,$30,$6D,$F0,$71,$F7,$F7,$A4,$F3,$8F,$27,$16,$A3,$07,$77,$D1
	.dc.b	$CD,$FD,$EE,$3F,$91,$E2,$8C,$47,$FA,$26,$31,$C7,$FC,$A2,$11,$87
	.dc.b	$8F,$C1,$3E,$57,$52,$8C,$4E,$A7,$31,$88,$FE,$36,$7F,$0B,$F2,$BF
	.dc.b	$86,$26,$9F,$C3,$FD,$1F,$C0,$47,$F7,$D3,$FA,$D1,$4E,$03,$E5,$7F
	.dc.b	$9A,$3F,$E3,$F8,$E3,$8F,$F7,$AE,$4B,$D1,$FE,$A7,$6D,$6E,$3F,$27
	.dc.b	$3B,$69,$FE,$AD,$E5,$DC,$FE,$61,$BC,$AF,$8C,$CC,$FE,$2E,$B8,$BF
	.dc.b	$E5,$1F,$4F,$26,$AE,$CE,$A7,$F4,$7E,$97,$23,$2F,$C1,$7C,$38,$FF
	.dc.b	$1F,$9E,$0D,$9F,$F1,$16,$3F,$E7,$89,$FD,$BF,$2F,$0E,$49,$1C,$22
	.dc.b	$D7,$F8,$A7,$E5,$7E,$6E,$1F,$42,$11,$8A,$FF,$35,$61,$FE,$50,$9E
	.dc.b	$1B,$7C,$4A,$FF,$00,$DF,$16,$DF,$F9,$C5,$E5,$5F,$C0,$1C,$BF,$C0
	.dc.b	$BE,$55,$F1,$5F,$39,$B5,$F1,$57,$BB,$99,$93,$F9,$79,$DD,$7F,$80
	.dc.b	$1A,$FF,$B9,$5A,$FF,$A0,$FE,$57,$FD,$0A,$FF,$12,$6A,$69,$3E,$1B
	.dc.b	$FC,$44,$9A,$4A,$FD,$44,$7F,$9B,$C9,$F0,$3C,$0F,$F1,$57,$C9,$F6
	.dc.b	$27,$C3,$E0,$CD,$CB,$FD,$94,$F7,$6C,$72,$A9,$1A,$6E,$1F,$E2,$0F
	.dc.b	$81,$FF,$5B,$BF,$EA,$10,$E1,$FF,$52,$35,$FE,$66,$D7,$7F,$13,$FE
	.dc.b	$41,$83,$49,$5F,$C1,$E1,$F1,$3B,$24,$65,$FF,$25,$7C,$87,$F6,$D8
	.dc.b	$FE,$1C,$DB,$FF,$1B,$B5,$F4,$35,$FF,$5D,$2F,$98,$E6,$49,$E6,$F6
	.dc.b	$F6,$24,$76,$6D,$7F,$91,$3E,$23,$E6,$37,$4F,$F2,$F4,$62,$3E,$43
	.dc.b	$5F,$F6,$57,$C7,$7F,$4D,$BF,$E4,$CD,$8F,$F2,$7B,$5F,$A0,$D7,$FD
	.dc.b	$A5,$1F,$C9,$CF,$E5,$C5,$7F,$CA,$ED,$7E,$C3,$5F,$F4,$1F,$CC,$7F
	.dc.b	$9B,$C9,$1F,$E7,$77,$FC,$E5,$F0,$92,$3F,$CF,$4D,$9F,$F2,$CF,$CE
	.dc.b	$7F,$9D,$C9,$3F,$E7,$F7,$FC,$F5,$EE,$48,$FF,$41,$36,$7F,$CC,$3F
	.dc.b	$0B,$FE,$7F,$24,$FF,$A1,$DF,$F4,$17,$CE,$48,$FF,$45,$36,$7F,$CD
	.dc.b	$3F,$31,$FE,$99,$3F,$D3,$5E,$AD,$AF,$F4,$BC,$7F,$A2,$C8,$FF,$A4
	.dc.b	$DB,$3F,$E8,$DF,$7F,$83,$67,$FD,$3F,$F3,$7F,$EA,$53,$F9,$BF,$C6
	.dc.b	$FF,$82,$63,$24,$9F,$F4,$B4,$EA,$38,$1F,$CC,$E7,$F2,$9E,$07,$F2
	.dc.b	$37,$F2,$11,$9F,$7B,$AF,$2F,$66,$C6,$4E,$51,$E2,$E5,$91,$D7,$DD
	.dc.b	$B9,$7F,$B6,$70,$57,$FA,$BD,$B5,$FD,$CD,$B3,$FE,$9C,$6B,$FB,$D7
	.dc.b	$38,$F9,$FE,$23,$AB,$6B,$D9,$F2,$BF,$10,$F8,$BE,$65,$8C,$93,$0F
	.dc.b	$24,$65,$F3,$7C,$B7,$CC,$F9,$6F,$9A,$DA,$F9,$0D,$9D,$C7,$CB,$9F
	.dc.b	$ED,$3C,$AE,$4A,$7D,$6F,$88,$B5,$FD,$8E,$74,$FF,$AD,$A7,$FD,$8E
	.dc.b	$47,$07,$D7,$F6,$F8,$E1,$F2,$DF,$C0,$F8,$32,$7E,$E7,$81,$6B,$E8
	.dc.b	$4D,$DB,$EF,$4D,$AF,$F2,$AD,$38,$1E,$CF,$E3,$3C,$5F,$C6,$B6,$BF
	.dc.b	$10,$DA,$FB,$8D,$C3,$F1,$5E,$0F,$E5,$0F,$CA,$F8,$96,$FF,$21,$1F
	.dc.b	$81,$23,$F8,$16,$DF,$E1,$FD,$DE,$0D,$9F,$CC,$7C,$A7,$E8,$0F,$D0
	.dc.b	$FC,$A5,$8C,$92,$7F,$21,$19,$58,$FF,$80,$7F,$4D,$7C,$67,$F8,$81
	.dc.b	$7C,$AD,$8C,$37,$0F,$F0,$D4,$E9,$B3,$A7,$FC,$3D,$F2,$1F,$E2,$5C
	.dc.b	$3B,$C7,$CE,$8C,$61,$FE,$11,$7F,$C0,$B2,$73,$3B,$F5,$FF,$C6,$71
	.dc.b	$89,$FF,$45,$BC,$0C,$3C,$63,$67,$0F,$66,$D7,$92,$2C,$F9,$0E,$73
	.dc.b	$8B,$F2,$4D,$AF,$44,$DA,$F0,$CD,$7D,$11,$6F,$C6,$47,$13,$96,$D7
	.dc.b	$8C,$F8,$EE,$2B,$C8,$7A,$E3,$26,$DC,$D7,$6A,$F7,$4E,$6F,$D2,$D3
	.dc.b	$D2,$5A,$F4,$B5,$F4,$B6,$BA,$3E,$08,$EF,$5E,$5A,$FB,$17,$6B,$EB
	.dc.b	$1B,$8F,$0D,$70,$8E,$93,$B8,$F0,$9E,$A7,$0F,$BC,$C4,$78,$EB,$E4
	.dc.b	$34,$E6,$58,$EC,$70,$2D,$78,$89,$1F,$0A,$32,$FA,$1A,$7E,$09,$E8
	.dc.b	$FF,$8A,$EF,$F3,$4D,$AF,$F3,$43,$C4,$C4,$FF,$AA,$E7,$71,$92,$1F
	.dc.b	$02,$BF,$41,$EC,$FE,$1C,$95,$F4,$5B,$3E,$43,$F4,$8E,$27,$D7,$1F
	.dc.b	$A2,$8C,$4F,$AE,$7A,$BC,$C9,$3C,$E9,$5C,$23,$C9,$3F,$B5,$8C,$47
	.dc.b	$D3,$AE,$6F,$18,$CD,$7E,$79,$B7,$FB,$66,$DF,$F0,$26,$D7,$E6,$1B
	.dc.b	$7D,$0F,$5C,$7B,$A7,$CF,$18,$BF,$3C,$60,$F7,$1E,$73,$13,$EA,$3E
	.dc.b	$23,$87,$D4,$ED,$F5,$39,$8F,$53,$6F,$D4,$D3,$D4,$59,$F5,$36,$BD
	.dc.b	$4F,$F4,$AC,$9F,$2B,$FC,$AC,$FE,$59,$F0,$76,$7F,$94,$C5,$BF,$DF
	.dc.b	$D2,$B1,$7F,$CD,$7E,$0F,$F9,$A7,$B5,$FC,$D5,$B8,$FF,$32,$6E,$1C
	.dc.b	$9B,$5F,$CB,$7E,$2B,$F9,$49,$FC,$AB,$E3,$2D,$FF,$36,$8F,$CD,$DF
	.dc.b	$F2,$C6,$BF,$CC,$DB,$A9,$A7,$F3,$4F,$8A,$E7,$3F,$C5,$1F,$E9,$5E
	.dc.b	$F4,$BF,$E8,$FF,$0D,$6B,$DF,$37,$2F,$EA,$53,$6B,$E5,$7C,$37,$F5
	.dc.b	$3F,$84,$FE,$A8,$D8,$FE,$A2,$DC,$3E,$4B,$6F,$FA,$67,$87,$FD,$20
	.dc.b	$FE,$93,$E9,$DB,$FE,$AB,$1E,$C2,$3F,$B9,$6D,$FF,$2E,$F4,$F8,$36
	.dc.b	$7F,$B0,$FC,$B7,$F6,$7F,$7B,$FB,$43,$63,$27,$F6,$68,$CB,$4F,$ED
	.dc.b	$3E,$FF,$F6,$F3,$FB,$87,$C1,$58,$C9,$2F,$F0,$B1,$97,$FA,$0F,$A9
	.dc.b	$FD,$90,$FE,$CB,$EA,$DA,$FE,$E3,$1F,$60,$8F,$F1,$56,$CF,$F4,$EF
	.dc.b	$85,$FF,$34,$3D,$24,$E5,$FF,$02,$7B,$DF,$E0,$5F,$26,$C7,$F8,$09
	.dc.b	$B3,$F9,$A6,$F0,$7F,$83,$5F,$F0,$67,$CB,$4A,$FF,$04,$FC,$85,$BF
	.dc.b	$D0,$CD,$BF,$F0,$A4,$DA,$FF,$1A,$E2,$F5,$51,$E6,$AB,$87,$A2,$B5
	.dc.b	$39,$38,$FA,$FA,$AD,$D2,$B6,$FC,$3B,$FD,$0C,$62,$3F,$DD,$8D,$7C
	.dc.b	$71,$A3,$A5,$F9,$BD,$6B,$3E,$39,$AF,$78,$FF,$17,$7C,$B7,$EB,$BC
	.dc.b	$23,$49,$7F,$EF,$BA,$FA,$72,$F9,$6E,$04,$9D,$4A,$75,$18,$5F,$2D
	.dc.b	$DC,$91,$97,$F7,$FF,$A7,$7E,$02,$39,$B6,$BC,$4D,$7F,$CC,$E1,$C3
	.dc.b	$FE,$6A,$5F,$13,$D0,$93,$D1,$EF,$E5,$48,$EE,$DA,$F8,$1F,$3C,$E8
	.dc.b	$FE,$26,$DA,$F7,$B6,$77,$3F,$C6,$3E,$79,$F0,$54,$97,$F1,$A3,$2F
	.dc.b	$C8,$F9,$87,$27,$F9,$8B,$0F,$A1,$18,$F3,$F5,$F0,$1E,$C9,$CD,$6D
	.dc.b	$F8,$7B,$3C,$FF,$A7,$1F,$15,$B1,$98,$F9,$B1,$AF,$35,$F0,$34,$93
	.dc.b	$F7,$23,$2D,$F6,$38,$11,$C1,$F6,$3D,$2E,$E4,$70,$7D,$CF,$8D,$F1
	.dc.b	$91,$C1,$F8,$0F,$91,$F3,$91,$C1,$F8,$4F,$99,$F5,$91,$C1,$F8,$8F
	.dc.b	$4B,$DE,$47,$07,$E3,$3D,$1F,$86,$47,$07,$E4,$3D,$1F,$8E,$47,$07
	.dc.b	$E5,$3E,$57,$E5,$91,$C1,$F9,$8F,$89,$F9,$E4,$70,$7E,$73,$E2,$7E
	.dc.b	$99,$1C,$1F,$A0,$F4,$3E,$B9,$1C,$1F,$AE,$F9,$47,$AB,$FE,$C8,$F1
	.dc.b	$FF,$85,$FC,$AF,$E4,$2B,$F8,$81,$FC,$A9,$C3,$FE,$7C,$AE,$2E,$82
	.dc.b	$F5,$BA,$76,$7F,$B5,$23,$73,$C7,$0C,$BF,$CD,$A3,$83,$CF,$85,$E0
	.dc.b	$EA,$1E,$83,$DE,$C3,$63,$84,$EE,$BF,$2D,$1F,$E9,$27,$9B,$A3,$B7
	.dc.b	$57,$5F,$95,$FF,$0C,$47,$AE,$DF,$0B,$D8,$D7,$B9,$BF,$72,$D7,$C1
	.dc.b	$BE,$2F,$FA,$81,$47,$5E,$DE,$13,$B8,$C3,$C7,$6F,$5F,$6C,$8C,$1D
	.dc.b	$78,$7C,$D3,$23,$07,$87,$DA,$FF,$71,$BF,$CB,$F1,$46,$23,$F8,$FC
	.dc.b	$63,$93,$F9,$7B,$63,$87,$B6,$F9,$5C,$B6,$33,$18,$3F,$82,$BF,$D4
	.dc.b	$4F,$E0,$FE,$96,$08,$FF,$3C,$F3,$3F,$A4,$BF,$CB,$39,$FF,$10,$FF
	.dc.b	$5F,$1F,$2A,$E5,$FE,$6B,$5C,$39,$A5,$F0,$8B,$7F,$EC,$E3,$FD,$D2
	.dc.b	$7F,$BA,$B9,$5F,$19,$E0,$E7,$F0,$F8,$93,$0F,$0C,$5B,$E5,$E2,$79
	.dc.b	$8C,$70,$FF,$A9,$9F,$EC,$2B,$9E,$9F,$D9,$32,$7C,$46,$31,$FD,$25
	.dc.b	$7E,$B9,$BC,$B9,$C7,$FD,$CB,$39,$7F,$DC,$FF,$2B,$F1,$5B,$B7,$5F
	.dc.b	$87,$FE,$E3,$39,$AF,$F0,$0B,$FA,$6C,$3F,$64,$DB,$E3,$18,$E0,$FE
	.dc.b	$F3,$F2,$AF,$7F,$FC,$CA,$D9,$FE,$E2,$DA,$FE,$F0,$D7,$FB,$3B,$9C
	.dc.b	$3F,$C2,$2F,$ED,$27,$F8,$13,$97,$FC,$63,$28,$D2,$9F,$F0,$CF,$CA
	.dc.b	$BE,$27,$FC,$EE,$DB,$FF,$01,$B6,$BF,$C1,$4D,$7F,$BC,$AC,$F8,$92
	.dc.b	$78,$BF,$E8,$2F,$9E,$FC,$64,$7F,$83,$9B,$5F,$E1,$56,$BF,$E0,$46
	.dc.b	$BE,$AF,$96,$F0,$37,$0F,$F4,$6C,$62,$3F,$C9,$ED,$4F,$96,$BF,$C3
	.dc.b	$AD,$8F,$F1,$23,$5F,$F0,$9B,$5F,$93,$F3,$DF,$EA,$B9,$FF,$12,$B6
	.dc.b	$3F,$C5,$8D,$7F,$C3,$6D,$7E,$8F,$C1,$7E,$52,$7F,$C5,$AD,$8F,$F1
	.dc.b	$A3,$5F,$F1,$1B,$5E,$3F,$1D,$FE,$AA,$83,$49,$3F,$E3,$8F,$9D,$FE
	.dc.b	$68,$DF,$1F,$FD,$63,$18,$BF,$F2,$13,$5F,$F1,$8F,$CC,$7F,$82,$2B
	.dc.b	$FC,$82,$D8,$FF,$24,$35,$DF,$A7,$FE,$BC,$87,$AD,$49,$5C,$9F,$F6
	.dc.b	$0F,$CD,$3E,$56,$92,$7F,$C9,$F1,$97,$FC,$A3,$F3,$1E,$5C,$7F,$32
	.dc.b	$D7,$FD,$16,$D4,$D8,$78,$DF,$ED,$06,$9F,$E5,$C2,$CF,$F9,$8D,$AF
	.dc.b	$F9,$4B,$E3,$BF,$CA,$2F,$FA,$61,$A9,$A6,$3E,$6D,$E9,$FF,$B7,$9B
	.dc.b	$5F,$E6,$A6,$D7,$F9,$C5,$AF,$F9,$81,$7A,$7C,$49,$3C,$5F,$F3,$5F
	.dc.b	$C7,$C8,$FF,$35,$B6,$BA,$7C,$7F,$84,$93,$E1,$7C,$9F,$2B,$23,$C8
	.dc.b	$D9,$FF,$39,$CF,$F9,$E0,$93,$FE,$78,$7F,$DF,$34,$AF,$F4,$8B,$67
	.dc.b	$FC,$21,$F1,$AF,$96,$CD,$49,$FF,$45,$D6,$F0,$FF,$49,$35,$FF,$40
	.dc.b	$B5,$FE,$2E,$EF,$BB,$FD,$5B,$87,$F8,$0A,$BF,$C3,$D3,$97,$F8,$EC
	.dc.b	$66,$BF,$8E,$46,$3C,$5F,$F5,$3C,$FE,$BE,$F9,$F4,$62,$65,$7A,$9D
	.dc.b	$C6,$2F,$99,$FE,$DA,$B7,$A7,$D3,$E5,$BA,$B7,$0F,$86,$DA,$E2,$2B
	.dc.b	$8C,$FC,$8A,$F2,$F7,$48,$F9,$7E,$05,$BE,$37,$FC,$6C,$95,$FE,$D0
	.dc.b	$8F,$82,$D7,$CC,$FC,$0E,$CE,$5A,$7C,$A7,$E1,$FC,$B4,$C7,$EC,$FC
	.dc.b	$B5,$C3,$BC,$DB,$FA,$33,$6B,$FA,$5F,$C5,$BE,$17,$FD,$A6,$BE,$5A
	.dc.b	$C6,$1B,$3F,$E5,$59,$D3,$67,$4F,$F9,$67,$A7,$FD,$40,$DF,$43,$FD
	.dc.b	$BF,$1F,$DA,$F1,$FF,$77,$BA,$CF,$E6,$B5,$FD,$D4,$FC,$37,$F7,$0D
	.dc.b	$8D,$47,$F6,$47,$94,$F4,$F4,$7B,$D7,$81,$F6,$FC,$B4,$C3,$D5,$EE
	.dc.b	$5C,$7E,$EE,$6A,$C6,$2B,$F8,$F7,$87,$E4,$FB,$9F,$23,$D3,$F9,$2D
	.dc.b	$8F,$8C,$DC,$7A,$B6,$FE,$0F,$C1,$FC,$43,$E2,$FC,$25,$BF,$95,$1E
	.dc.b	$F2,$3F,$EF,$CD,$91,$8C,$3E,$4F,$C2,$71,$6D,$7D,$7F,$57,$EE,$9F
	.dc.b	$7B,$DF,$B1,$92,$4F,$C2,$8C,$BF,$13,$DF,$FC,$2F,$8D,$F8,$66,$D7
	.dc.b	$E0,$DB,$3B,$8F,$C3,$4F,$E0,$3C,$09,$1F,$85,$F1,$2D,$7E,$02,$6E
	.dc.b	$DF,$84,$9B,$5F,$76,$6D,$7F,$3B,$9A,$FF,$3B,$8B,$AF,$F3,$2A,$FC
	.dc.b	$94,$60,$FC,$8D,$58,$C9,$E3,$2D,$78,$DB,$7E,$36,$9E,$32,$C7,$8D
	.dc.b	$AF,$8D,$B7,$E3,$8E,$84,$9F,$D0,$DF,$92,$B6,$FA,$AF,$CD,$5B,$9F
	.dc.b	$CF,$46,$2B,$8C,$7A,$67,$47,$F5,$FA,$E5,$1D,$0F,$1D,$71,$7D,$53
	.dc.b	$FB,$08,$C3,$C6,$3A,$57,$68,$EA,$7B,$4E,$11,$D5,$CD,$FB,$E7,$81
	.dc.b	$27,$BB,$F2,$E7,$77,$E6,$6C,$F9,$8E,$93,$C6,$3E,$71,$A3,$BD,$FC
	.dc.b	$67,$FC,$01,$5F,$C1,$49,$5F,$A7,$6C,$7F,$9A,$8F,$71,$4F,$61,$CA
	.dc.b	$3F,$8D,$78,$9F,$8C,$23,$F7,$1B,$1F,$01,$B7,$E7,$8F,$F0,$74,$62
	.dc.b	$7E,$09,$FC,$5D,$C4,$78,$A7,$F9,$1C,$62,$3E,$74,$62,$7C,$04,$8F
	.dc.b	$03,$63,$C1,$82,$B9,$2B,$FE,$63,$18,$6D,$71,$3C,$B7,$FE,$27,$9D
	.dc.b	$47,$C2,$35,$1E,$77,$D4,$61,$F3,$B8,$7C,$A4,$BF,$2C,$F4,$8F,$E7
	.dc.b	$ED,$8F,$74,$6E,$7C,$6F,$CA,$7B,$5F,$AD,$F2,$46,$CF,$21,$E9,$77
	.dc.b	$7E,$E8,$D1,$E6,$34,$7F,$40,$F9,$6F,$EC,$3E,$AF,$F6,$26,$D7,$F5
	.dc.b	$F6,$CE,$E3,$FB,$14,$FF,$68,$F8,$C9,$1F,$D9,$7D,$AB,$5F,$D3,$26
	.dc.b	$E5,$FD,$2E,$03,$A3,$FD,$77,$19,$8C,$4F,$F6,$8F,$95,$98,$7F,$6D
	.dc.b	$F9,$6B,$5F,$DA,$27,$FB,$21,$27,$FA,$33,$63,$FC,$33,$F0,$3F,$D6
	.dc.b	$E7,$FB,$B4,$F8,$F0,$F0,$4E,$6B,$FB,$1F,$AB,$FE,$04,$F1,$7F,$C0
	.dc.b	$AD,$C3,$FC,$04,$D9,$DC,$7F,$81,$67,$FC,$17,$F2,$D2,$3F,$C1,$1F
	.dc.b	$2D,$6B,$FB,$7C,$DC,$7E,$06,$AA,$C5,$FF,$B8,$32,$7C,$B7,$9E,$30
	.dc.b	$DC,$7E,$5C,$BE,$3B,$10,$8E,$35,$F4,$31,$F7,$DF,$F8,$36,$73,$5F
	.dc.b	$C0,$DA,$FF,$07,$6B,$E0,$AE,$47,$97,$E5,$BB,$CF,$2B,$F9,$71,$E2
	.dc.b	$F8,$FF,$F2,$F5,$FE,$01,$7C,$CE,$7D,$2D,$37,$1E,$B5,$B8,$C3,$FE
	.dc.b	$52,$AE,$33,$A8,$43,$E4,$FC,$07,$87,$8E,$D6,$E7,$4D,$9D,$34,$D7
	.dc.b	$1F,$A4,$F9,$7F,$A7,$FE,$17,$4F,$CB,$5A,$75,$5F,$CA,$F1,$F1,$C7
	.dc.b	$F8,$E6,$75,$39,$6C,$FF,$31,$6D,$7E,$86,$DF,$35,$D2,$BF,$CE,$31
	.dc.b	$8B,$F8,$0F,$FB,$76,$B8,$7B,$9E,$6F,$7B,$CE,$DA,$DB,$6F,$9B,$EC
	.dc.b	$E2,$93,$D3,$8A,$D7,$82,$6E,$1E,$59,$B5,$FB,$AA,$7B,$2F,$93,$FC
	.dc.b	$8B,$67,$CF,$C7,$89,$91,$FE,$CD,$7F,$DA,$04,$70,$7B,$BE,$2B,$77
	.dc.b	$F0,$9F,$F3,$BF,$CC,$7F,$99,$7A,$0D,$25,$7D,$38,$7C,$C6,$C9,$19
	.dc.b	$7F,$CD,$DC,$1F,$EF,$9C,$3F,$B2,$35,$FE,$D0,$D7,$E0,$35,$FC,$E7
	.dc.b	$C5,$FD,$F2,$57,$DF,$7F,$05,$F1,$52,$3F,$04,$D9,$EF,$F3,$5E,$22
	.dc.b	$4F,$E2,$ED,$1F,$E8,$3F,$98,$FC,$CD,$7F,$79,$6B,$FD,$F5,$AF,$D5
	.dc.b	$6B,$FC,$C3,$E6,$3F,$83,$7A,$FF,$97,$6C,$7F,$80,$D7,$2B,$3D,$72
	.dc.b	$FF,$A9,$EB,$FC,$11,$F3,$BF,$EE,$77,$F4,$0D,$73,$F0,$7F,$A3,$51
	.dc.b	$97,$FC,$25,$EA,$BA,$74,$D8,$FD,$53,$5F,$F0,$B3,$5E,$3F,$03,$C0
	.dc.b	$8F,$07,$C1,$F5,$A9,$A4,$9F,$DB,$46,$5F,$DA,$7C,$8F,$9F,$B8,$D2
	.dc.b	$47,$EE,$7E,$4B,$44,$70,$7D,$7C,$FE,$07,$1F,$BE,$3F,$0F,$18,$3B
	.dc.b	$7B,$12,$3F,$82,$35,$FE,$09,$EC,$78,$C8,$FF,$09,$7A,$FC,$B4,$8F
	.dc.b	$C3,$35,$FE,$19,$F3,$5E,$B2,$3F,$C4,$5F,$77,$CD,$48,$FE,$28,$D7
	.dc.b	$F4,$3F,$13,$EA,$23,$F9,$37,$E3,$7A,$F2,$3F,$8E,$35,$FD,$4F,$C4
	.dc.b	$FC,$22,$3F,$9B,$7F,$09,$F3,$2F,$77,$F8,$57,$91,$FE,$01,$F9,$AF
	.dc.b	$E7,$57,$FC,$F5,$AA,$3F,$C0,$D5,$C6,$3F,$A1,$2F,$7B,$9F,$83,$FC
	.dc.b	$56,$35,$3C,$63,$18,$6A,$3F,$C2,$73,$C5,$FE,$99,$C6,$F0,$B5,$0F
	.dc.b	$5D,$F1,$18,$6C,$70,$9D,$D3,$E1,$6C,$49,$EB,$1D,$3E,$69,$F0,$9C
	.dc.b	$F9,$65,$7F,$5D,$9C,$B7,$E1,$38,$54,$8C,$1B,$EE,$E9,$32,$30,$73
	.dc.b	$DB,$D1,$DF,$23,$07,$7E,$1F,$24,$C8,$C1,$E3,$F8,$3F,$A4,$FF,$A2
	.dc.b	$72,$46,$23,$FD,$1D,$18,$E3,$FE,$C8,$D8,$E1,$F0,$6F,$9A,$CB,$63
	.dc.b	$31,$83,$F8,$53,$FE,$1B,$3F,$86,$7B,$58,$23,$FD,$C7,$C7,$FF,$10
	.dc.b	$BF,$C6,$79,$FF,$C0,$CF,$F8,$98,$7C,$D3,$97,$FD,$49,$5C,$39,$65
	.dc.b	$F0,$8B,$7F,$89,$3F,$B0,$72,$F6,$3F,$C5,$FC,$CF,$47,$17,$E0,$F8
	.dc.b	$8F,$19,$8B,$E4,$7F,$98,$9F,$3F,$8B,$27,$81,$E8,$7F,$C7,$7C,$3F
	.dc.b	$C6,$E3,$FC,$40,$DA,$E7,$83,$BF,$7D,$1C,$39,$66,$1C,$22,$DF,$8B
	.dc.b	$E6,$BE,$33,$76,$F5,$FC,$77,$F9,$2A,$73,$9F,$F9,$D5,$FD,$26,$1F
	.dc.b	$E3,$16,$DF,$18,$C7,$07,$F9,$4B,$E6,$9F,$0B,$76,$C3,$6B,$FC,$75
	.dc.b	$3F,$AD,$7F,$C5,$4D,$BF,$F6,$B3,$5F,$F6,$D3,$5F,$F0,$97,$CD,$33
	.dc.b	$FF,$45,$46,$1A,$FF,$A4,$23,$55,$FE,$37,$6C,$7F,$B8,$FE,$6B,$F0
	.dc.b	$93,$FE,$3B,$6C,$67,$97,$9A,$8C,$BF,$EE,$D7,$FB,$7B,$9F,$7F,$71
	.dc.b	$83,$3F,$35,$E0,$9F,$F7,$A3,$5F,$F1,$63,$5F,$F5,$3B,$57,$CD,$7F
	.dc.b	$08,$C3,$FC,$A0,$D8,$FF,$7E,$35,$FF,$80,$0D,$7F,$C7,$1F,$41,$E8
	.dc.b	$6E,$3E,$C8,$C4,$7F,$AD,$9A,$BE,$25,$E4,$FC,$26,$E1,$FE,$61,$6C
	.dc.b	$FF,$91,$4F,$F5,$FB,$87,$1F,$37,$FC,$AA,$7F,$CD,$0D,$8E,$01,$C3
	.dc.b	$C3,$C8,$FA,$3E,$2F,$F8,$F2,$BF,$CE,$0D,$9E,$2D,$7C,$4D,$7F,$CB
	.dc.b	$1F,$37,$FE,$1E,$9F,$F3,$C3,$63,$C2,$D7,$D4,$D7,$FC,$C1,$F3,$5F
	.dc.b	$E4,$14,$D2,$7C,$DF,$E1,$BE,$7D,$E3,$52,$61,$FE,$87,$8C,$BF,$E8
	.dc.b	$92,$DF,$C2,$6B,$F2,$DA,$FF,$9C,$5A,$FF,$BB,$3E,$4B,$FD,$DB,$1F
	.dc.b	$E9,$36,$C7,$CC,$6B,$F5,$5A,$FF,$9E,$FE,$7F,$F1,$C4,$E7,$FC,$FB
	.dc.b	$F9,$1F,$87,$7A,$5F,$A2,$6E,$1F,$EA,$16,$DF,$DA,$6B,$F8,$3F,$65
	.dc.b	$F2,$7F,$EA,$52,$57,$FA,$95,$FF,$39,$F0,$48,$FF,$31,$36,$BA,$7C
	.dc.b	$77,$84,$93,$E1,$7F,$D6,$1F,$29,$23,$FD,$60,$D9,$FF,$3E,$FC,$DF
	.dc.b	$FA,$00,$93,$FE,$6F,$7F,$7B,$B7,$F7,$98,$FF,$67,$36,$7F,$9C,$BF
	.dc.b	$C8,$3C,$44,$FF,$A8,$E3,$51,$89,$FA,$9E,$67,$42,$57,$D5,$6C,$74
	.dc.b	$F2,$FF,$DF,$37,$CA,$BA,$47,$FB,$F3,$E2,$1F,$2F,$4C,$7A,$D6,$BF
	.dc.b	$DC,$93,$A6,$CE,$9A,$6B,$D6,$E9,$E6,$75,$6C,$64,$E7,$19,$69,$F1
	.dc.b	$3D,$CF,$39,$E8,$F7,$6C,$64,$FC,$94,$7C,$C2,$38,$3E,$6F,$CB,$F0
	.dc.b	$D7,$E6,$BE,$FF,$42,$4F,$B3,$D9,$B5,$FD,$AB,$4F,$F2,$54,$FC,$0A
	.dc.b	$F8,$D5,$AE,$1F,$A5,$6B,$FF,$0B,$DF,$C6,$ED,$F9,$9E,$CF,$CD,$6C
	.dc.b	$64,$F9,$71,$97,$F0,$7D,$1F,$58,$FA,$FD,$76,$32,$7E,$C6,$3F,$16
	.dc.b	$47,$07,$E3,$39,$F8,$63,$F8,$C7,$EF,$7C,$BF,$DF,$F3,$ED,$7D,$C6
	.dc.b	$D7,$C1,$6E,$9E,$ED,$3F,$B7,$D7,$E0,$BE,$5F,$F1,$07,$E2,$7D,$1B
	.dc.b	$19,$26,$1F,$2E,$32,$FE,$7B,$E4,$BF,$26,$7E,$53,$E4,$ED,$7E,$92
	.dc.b	$3F,$4E,$47,$F8,$83,$67,$FC,$3F,$ED,$FE,$9A,$78,$08,$CB,$F9,$DF
	.dc.b	$2F,$F3,$DF,$2F,$63,$F3,$6D,$9F,$C0,$37,$4F,$80,$FE,$97,$E5,$E5
	.dc.b	$7E,$8B,$E5,$ED,$FE,$1E,$6D,$FC,$79,$B5,$E3,$A6,$2F,$95,$FD,$D5
	.dc.b	$FE,$2A,$31,$5E,$98,$C7,$C6,$6A,$39,$4E,$AF,$E0,$DE,$5F,$81,$19
	.dc.b	$BE,$35,$CC,$E3,$5B,$AF,$1B,$6B,$84,$F6,$BC,$4C,$8E,$55,$DA,$38
	.dc.b	$4E,$63,$C7,$3A,$7A,$B6,$93,$BB,$F8,$17,$6B,$C0,$6D,$C5,$F7,$8E
	.dc.b	$55,$88,$EA,$78,$0F,$C1,$B9,$BF,$5C,$71,$33,$3D,$48,$F1,$BF,$8E
	.dc.b	$D8,$F8,$E6,$A3,$27,$A1,$C1,$A8,$EA,$71,$3C,$11,$B7,$2D,$24,$71
	.dc.b	$7A,$B4,$91,$C9,$F8,$AF,$53,$E0,$C7,$27,$F0,$35,$F0,$6F,$E1,$36
	.dc.b	$BC,$F3,$A6,$BC,$9E,$EF,$E5,$A3,$83,$DE,$FE,$CC,$F0,$7E,$6C,$71
	.dc.b	$3A,$1B,$AF,$0C,$ED,$B1,$98,$E5,$5D,$A3,$FC,$79,$5F,$9D,$6D,$7F
	.dc.b	$99,$23,$11,$F3,$CC,$D7,$F8,$F1,$FC,$19,$83,$FC,$7F,$1E,$61,$F2
	.dc.b	$BF,$85,$24,$FA,$DB,$3E,$F6,$E1,$F3,$63,$F7,$31,$89,$F8,$F5,$D1
	.dc.b	$E4,$49,$E4,$DA,$E5,$3B,$9C,$92,$33,$5D,$63,$F0,$07,$EF,$1C,$3F
	.dc.b	$3C,$F1,$D7,$53,$C7,$1A,$AF,$80,$DB,$FD,$F3,$67,$F6,$8D,$AF,$E5
	.dc.b	$6D,$BF,$E2,$8D,$9F,$E2,$2D,$9F,$24,$78,$5F,$39,$E4,$3D,$8F,$90
	.dc.b	$D3,$E9,$BF,$85,$5F,$02,$73,$1E,$87,$7E,$43,$C7,$FE,$30,$BE,$5E
	.dc.b	$C6,$1B,$3F,$C4,$67,$4D,$9D,$39,$F6,$79,$9D,$38,$AC,$64,$99,$7F
	.dc.b	$BB,$2D,$3A,$8C,$71,$79,$0F,$E5,$9F,$17,$FD,$4E,$33,$7E,$39,$FF
	.dc.b	$5E,$D9,$A4,$8F,$CD,$46,$5F,$E5,$54,$F9,$7E,$D1,$87,$BF,$C7,$78
	.dc.b	$3E,$42,$CF,$06,$E2,$DF,$FC,$2F,$87,$FA,$4D,$B7,$C6,$FB,$F1,$7A
	.dc.b	$8F,$5F,$15,$8C,$93,$0E,$8F,$F5,$DC,$3F,$81,$9F,$C1,$A3,$27,$F7
	.dc.b	$3F,$3D,$F2,$FF,$D4,$5B,$5E,$EE,$4C,$4C,$8F,$F4,$03,$FE,$83,$23
	.dc.b	$83,$E2,$FC,$36,$EF,$B8,$C1,$1F,$53,$E6,$FE,$AF,$A9,$6F,$E6,$B6
	.dc.b	$F8,$DF,$FB,$E3,$6F,$8F,$7C,$2F,$E9,$FB,$3F,$78,$FB,$FF,$37,$63
	.dc.b	$24,$C3,$9C,$65,$EB,$F3,$7F,$86,$3F,$B9,$7A,$AF,$96,$C5,$FE,$02
	.dc.b	$7F,$C9,$B5,$F3,$5A,$FF,$87,$73,$46,$23,$E0,$36,$33,$3F,$DC,$ED
	.dc.b	$F3,$7D,$AF,$E3,$C6,$27,$5E,$9F,$93,$D4,$F2,$B6,$32,$73,$8C,$B4
	.dc.b	$ED,$EA,$7B,$0F,$6F,$A9,$63,$24,$C3,$C3,$19,$7C,$7E,$8F,$C4,$3E
	.dc.b	$2F,$A3,$6B,$D5,$1E,$FC,$BF,$CE,$B5,$FE,$FC,$6B,$FE,$2F,$0E,$1F
	.dc.b	$F1,$93,$C3,$8B,$FC,$F2,$FF,$9E,$23,$C3,$5F,$1F,$D0,$FF,$52,$7C
	.dc.b	$D7,$CD,$F3,$3F,$DF,$D5,$FE,$4D,$6B,$FE,$81,$8F,$A1,$EE,$4B,$FE
	.dc.b	$A9,$C3,$FE,$FB,$3F,$DF,$98,$FF,$BB,$2F,$CB,$E9,$FA,$C9,$5F,$C0
	.dc.b	$BE,$23,$F8,$39,$1C,$1D,$3E,$6B,$EF,$11,$C1,$FD,$8E,$DE,$97,$C7
	.dc.b	$24,$65,$FC,$47,$CC,$FF,$8D,$3A,$3F,$B5,$37,$2F,$94,$17,$88,$F8
	.dc.b	$CD,$F8,$9F,$99,$AD,$FC,$CB,$E6,$3F,$BA,$42,$30,$FF,$77,$6B,$C2
	.dc.b	$37,$6F,$42,$92,$7F,$BF,$46,$5F,$F0,$07,$4B,$E0,$3B,$B6,$33,$18
	.dc.b	$3C,$7F,$32,$9A,$49,$FF,$05,$46,$5F,$95,$EC,$7A,$B8,$4D,$24,$7F
	.dc.b	$38,$F8,$9F,$7F,$9C,$69,$23,$F0,$7A,$7C,$4E,$7F,$83,$1F,$97,$6B
	.dc.b	$FC,$AE,$3C,$1E,$CC,$8F,$B8,$D7,$EA,$DB,$DE,$F0,$92,$3F,$A5,$BD
	.dc.b	$FE,$56,$47,$E5,$1A,$FE,$A7,$DB,$F7,$11,$FC,$53,$EC,$F9,$89,$1F
	.dc.b	$D5,$9A,$FE,$1B,$27,$C9,$7B,$49,$1F,$A2,$7E,$57,$BB,$23,$F2,$CD
	.dc.b	$79,$7C,$5F,$C5,$23,$FA,$A7,$E9,$7B,$D2,$3F,$3C,$D7,$F2,$FF,$2E
	.dc.b	$F1,$FF,$D6,$1E,$3F,$FB,$DB,$E6,$1E,$E6,$3C,$2F,$F5,$A4,$63,$0F
	.dc.b	$F4,$C6,$1C,$5B,$E5,$6D,$FF,$66,$46,$5F,$E9,$ED,$C1,$7F,$EB,$F8
	.dc.b	$57,$1A,$FC,$27,$8B,$78,$5E,$8D,$35,$D0,$9B,$F1,$18,$65,$FF,$AB
	.dc.b	$65,$3D,$7E,$61,$E7,$F5,$E7,$97,$FE,$B3,$9C,$B7,$CF,$E1,$52,$30
	.dc.b	$6F,$C0,$E9,$32,$30,$73,$E3,$F5,$78,$32,$30,$77,$DF,$C9,$32,$30
	.dc.b	$78,$FC,$FF,$F0,$A3,$FC,$3B,$24,$62,$3F,$9D,$46,$38,$FF,$CF,$CD
	.dc.b	$8E,$1E,$63,$E6,$32,$D8,$CC,$60,$FE,$26,$FF,$BC,$8F,$E2,$BF,$05
	.dc.b	$82,$3F,$E8,$BF,$2B,$FD,$CC,$FF,$1F,$E7,$FF,$6E,$3F,$EE,$E1,$F3
	.dc.b	$1F,$BF,$7F,$DF,$D3,$C3,$96,$57,$08,$B3,$FE,$16,$3E,$CC,$74,$E4
	.dc.b	$FF,$7A,$73,$FF,$86,$23,$07,$83,$E2,$BC,$67,$92,$6C,$79,$3E,$63
	.dc.b	$CE,$62,$FB,$3F,$EF,$FD,$FF,$90,$4F,$FB,$D5,$B3,$FE,$28,$C3,$F7
	.dc.b	$33,$9E,$5E,$1C,$52,$30,$71,$7F,$3F,$5F,$E3,$0F,$98,$F8,$CD,$D3
	.dc.b	$D7,$ED,$EB,$2D,$CA,$B7,$18,$7F,$41,$82,$7A,$6D,$C6,$31,$1C,$70
	.dc.b	$C3,$BA,$E2,$FF,$90,$BE,$61,$E1,$6A,$11,$8B,$ED,$8F,$92,$1E,$0E
	.dc.b	$DD,$57,$3B,$FF,$17,$46,$23,$FC,$1C,$D7,$CA,$D7,$D5,$F3,$1F,$BE
	.dc.b	$C3,$FD,$AC,$D8,$FE,$DA,$D7,$FC,$30,$D7,$DE,$BD,$CF,$F1,$F3,$67
	.dc.b	$9C,$60,$CF,$B9,$23,$BB,$6B,$FB,$DF,$CB,$F5,$7E,$73,$53,$49,$5F
	.dc.b	$E4,$D7,$C1,$1C,$A3,$06,$66,$57,$FB,$CE,$BF,$CA,$35,$E0,$9F,$F2
	.dc.b	$B6,$3F,$DD,$EB,$FD,$F0,$D9,$FF,$07,$B5,$FF,$1D,$35,$FC,$0B,$FE
	.dc.b	$3C,$8E,$3E,$7F,$42,$3D,$3D,$EE,$E4,$70,$7F,$87,$29,$F2,$9F,$E9
	.dc.b	$77,$FC,$90,$D7,$F1,$ED,$7F,$28,$D7,$8F,$C2,$7F,$9C,$60,$D2,$4F
	.dc.b	$78,$FF,$80,$13,$DB,$E0,$F8,$12,$7C,$9F,$08,$F3,$FF,$CF,$8D,$8C
	.dc.b	$C6,$0C,$FC,$E4,$8E,$2D,$BF,$F1,$CD,$3E,$3F,$99,$2B,$BB,$FE,$89
	.dc.b	$F9,$3F,$C1,$13,$CC,$C1,$FE,$6F,$F8,$E9,$5F,$E4,$D6,$E1,$FE,$46
	.dc.b	$F0,$FF,$84,$B7,$D2,$FF,$4A,$46,$23,$FC,$E8,$D7,$FC,$E7,$ED,$FF
	.dc.b	$9E,$C9,$7F,$E5,$97,$FC,$FD,$F3,$F2,$3F,$CB,$8D,$9F,$F4,$07,$CE
	.dc.b	$FF,$A0,$C9,$3F,$E6,$17,$FD,$0D,$F0,$BF,$EA,$E3,$F8,$17,$EA,$1F
	.dc.b	$C3,$67,$FD,$49,$19,$7F,$CD,$0F,$06,$92,$BF,$D9,$B3,$FC,$62,$3F
	.dc.b	$D1,$73,$D3,$C0,$EB,$E0,$F6,$F1,$ED,$7F,$0E,$3F,$8C,$DF,$F4,$0C
	.dc.b	$39,$D6,$E7,$51,$FC,$86,$7F,$8D,$F8,$3F,$CA,$1E,$0D,$F2,$3D,$67
	.dc.b	$B3,$C8,$B7,$CA,$D6,$AC,$C7,$AA,$AD,$F5,$F4,$FE,$29,$F1,$BE,$2E
	.dc.b	$E3,$CF,$1F,$F2,$E3,$FA,$96,$E1,$FE,$86,$6B,$FE,$B1,$6B,$F0,$2B
	.dc.b	$E8,$FC,$AF,$48,$F9,$6F,$D4,$F6,$FE,$AF,$CA,$DC,$3C,$0D,$AF,$A2
	.dc.b	$DC,$BE,$23,$F7,$3E,$76,$57,$9F,$B6,$DF,$D8,$9B,$7F,$2E,$6D,$7F
	.dc.b	$4C,$EA,$FE,$2F,$3F,$D5,$9B,$1F,$63,$B7,$FA,$54,$63,$0F,$EA,$CF
	.dc.b	$F1,$E7,$FD,$A0,$D7,$A7,$89,$FA,$D2,$57,$CB,$ED,$E7,$1F,$94,$7F
	.dc.b	$2B,$F2,$DF,$96,$F9,$6B,$5F,$45,$B7,$F9,$26,$E5,$F8,$57,$F3,$BF
	.dc.b	$2B,$2B,$EC,$7C,$B5,$BF,$CC,$CD,$BF,$C6,$4D,$AF,$CA,$FB,$FF,$A6
	.dc.b	$3F,$4F,$F2,$D6,$3F,$05,$1F,$DE,$B0,$DC,$7E,$0E,$31,$1F,$99,$6B
	.dc.b	$F9,$27,$F6,$3F,$2D,$CE,$3F,$56,$FF,$75,$FD,$47,$FC,$D3,$5F,$99
	.dc.b	$5F,$2F,$98,$D3,$5D,$36,$99,$7F,$83,$AB,$8D,$73,$F9,$7E,$C7,$7F
	.dc.b	$8C,$B1,$92,$5F,$27,$FC,$1F,$5F,$EE,$07,$37,$FE,$1A,$F1,$7F,$AC
	.dc.b	$37,$E0,$3F,$C1,$D0,$F8,$6B,$12,$BF,$69,$3C,$1F,$F0,$DB,$B7,$31
	.dc.b	$BC,$3C,$47,$F8,$8E,$78,$36,$B5,$7E,$1B,$E2,$76,$8C,$BD,$AE,$D7
	.dc.b	$18,$DB,$D8,$5F,$66,$CF,$B2,$B8,$C6,$2B,$8D,$FC,$39,$CD,$F1,$23
	.dc.b	$8B,$CB,$6B,$89,$F0,$9C,$5F,$8E,$B8,$CE,$A3,$E1,$57,$28,$C4,$78
	.dc.b	$CE,$11,$E7,$8C,$CF,$8E,$31,$3A,$9C,$C6,$2F,$E5,$4E,$48,$FB,$67
	.dc.b	$67,$4B,$F3,$C7,$03,$53,$C2,$7E,$81,$B3,$8B,$93,$E9,$14,$F5,$47
	.dc.b	$38,$F8,$87,$07,$11,$C1,$AF,$08,$EE,$FC,$99,$F2,$C6,$27,$CA,$70
	.dc.b	$BF,$F1,$65,$7C,$C9,$FF,$12,$46,$23,$E6,$1F,$DE,$2F,$FB,$D9,$27
	.dc.b	$6D,$8F,$B1,$E8,$7E,$08,$93,$F4,$9B,$1E,$FA,$FF,$1D,$36,$33,$7E
	.dc.b	$56,$D7,$A4,$FD,$3B,$89,$F4,$1E,$08,$EC,$47,$A3,$67,$A3,$4E,$91
	.dc.b	$FA,$D9,$E8,$FC,$B3,$11,$B7,$BC,$FF,$01,$9D,$D7,$E7,$DB,$7F,$C0
	.dc.b	$9B,$7F,$C0,$1B,$7E,$73,$CA,$4B,$F2,$B6,$7C,$AF,$ED,$27,$8C,$F8
	.dc.b	$67,$C9,$3E,$73,$4E,$2F,$C8,$DB,$F2,$34,$F2,$1C,$B1,$7C,$77,$F3
	.dc.b	$3F,$43,$F9,$A3,$67,$F9,$A3,$70,$FE,$11,$5C,$27,$F9,$C5,$7F,$3B
	.dc.b	$F3,$E4,$7F,$3B,$F9,$7B,$7C,$26,$E7,$FC,$DE,$6D,$7F,$24,$E9,$FE
	.dc.b	$56,$7F,$2C,$F5,$6C,$7F,$2A,$8F,$F2,$69,$1F,$CC,$B7,$0F,$E3,$9F
	.dc.b	$7C,$F9,$2E,$31,$87,$FA,$87,$CA,$FF,$51,$F9,$5B,$87,$F5,$16,$D7
	.dc.b	$B9,$B9,$FF,$4D,$7F,$AC,$7C,$84,$AF,$EB,$1F,$17,$6F,$E2,$CD,$BF
	.dc.b	$EA,$93,$6B,$FD,$B7,$97,$FB,$CA,$7F,$DC,$AD,$8C,$FC,$57,$FB,$66
	.dc.b	$31,$87,$FB,$95,$FF,$7C,$BF,$EE,$40,$E1,$EB,$F0,$7F,$DC,$A3,$11
	.dc.b	$C6,$3A,$7C,$1F,$56,$DF,$F7,$1F,$5F,$FB,$A1,$FD,$D7,$A6,$C6,$49
	.dc.b	$7F,$82,$8C,$BF,$84,$E7,$FE,$F9,$D3,$FD,$F5,$B5,$FD,$F5,$B3,$B8
	.dc.b	$FF,$00,$4F,$BF,$E4,$64,$7F,$96,$F3,$7C,$AE,$2F,$EE,$D7,$B4,$95
	.dc.b	$FE,$BD,$A4,$62,$7F,$07,$F2,$5C,$77,$F9,$EF,$F8,$47,$DF,$98,$7F
	.dc.b	$84,$7E,$1A,$DF,$E7,$A6,$DF,$F8,$36,$6D,$7F,$81,$BD,$EF,$F0,$D7
	.dc.b	$4F,$F8,$6D,$B1,$FE,$1B,$6E,$1F,$A8,$6D,$FE,$03,$DD,$FB,$C7,$DF
	.dc.b	$F7,$AD,$FC,$18,$FB,$44,$7D,$CD,$BF,$CF,$FB,$DC,$1B,$3F,$E2,$BE
	.dc.b	$CF,$F1,$9F,$81,$FE,$55,$90,$70,$D8,$FC,$47,$BB,$2B,$E6,$36,$7F
	.dc.b	$20,$3D,$D7,$A7,$FE,$40,$6D,$7F,$BB,$DB,$3C,$1A,$FE,$67,$E8,$9C
	.dc.b	$5F,$E4,$86,$D7,$FB,$DD,$B3,$E0,$6B,$FA,$1F,$A1,$79,$9F,$E5,$06
	.dc.b	$D7,$FB,$FD,$B3,$E8,$6B,$FA,$9F,$97,$FE,$43,$E2,$7B,$21,$18,$7D
	.dc.b	$EB,$CA,$F8,$11,$8F,$27,$F8,$73,$F1,$3F,$52,$AD,$BE,$8A,$32,$FE
	.dc.b	$E5,$AF,$D4,$35,$F3,$7C,$20,$D2,$47,$9E,$1F,$37,$B2,$46,$5E,$5F
	.dc.b	$29,$23,$83,$5F,$47,$8D,$DC,$8F,$57,$C5,$E2,$C8,$F0,$35,$F8,$1E
	.dc.b	$37,$9C,$8F,$95,$F5,$7C,$A4,$8F,$43,$5F,$91,$F2,$9E,$F2,$3E,$D7
	.dc.b	$E1,$7C,$9C,$8F,$80,$D7,$E8,$78,$BF,$1C,$8F,$C5,$7E,$57,$89,$23
	.dc.b	$E4,$35,$FB,$1C,$FF,$3C,$8F,$CD,$7E,$97,$C8,$C8,$FA,$0D,$7F,$01
	.dc.b	$F1,$9F,$5C,$8F,$D5,$7E,$D7,$C8,$48,$FB,$0D,$7F,$11,$E1,$7D,$F2
	.dc.b	$3F,$75,$FC,$4F,$CA,$B6,$FC,$37,$88,$EB,$FC,$47,$CB,$7E,$8B,$0F
	.dc.b	$DE,$CE,$5F,$DF,$C6,$5F,$F4,$05,$F3,$77,$5F,$91,$87,$0F,$68,$C6
	.dc.b	$16,$BF,$84,$D7,$17,$9D,$65,$7C,$33,$E2,$F1,$B3,$C8,$A6,$E7,$55
	.dc.b	$E2,$3F,$8A,$AF,$8C,$D5,$C9,$D1,$DA,$3C,$1F,$2C,$E9,$EF,$E9,$CA
	.dc.b	$DC,$E5,$BD,$5C,$2A,$46,$0D,$F5,$74,$99,$18,$39,$EF,$EC,$F4,$64
	.dc.b	$60,$EF,$BF,$92,$64,$60,$F1,$F5,$FF,$B6,$1F,$E5,$79,$23,$11,$FD
	.dc.b	$F6,$31,$C9,$FC,$55,$B1,$C3,$AD,$F2,$D9,$6C,$66,$30,$7F,$12,$7F
	.dc.b	$D7,$E7,$F1,$4F,$6F,$04,$7F,$9C,$73,$7C,$E7,$F9,$F3,$5E,$3F,$2D
	.dc.b	$F7,$9F,$E3,$1E,$4F,$E3,$F2,$FC,$84,$7F,$4E,$E4,$C5,$FF,$85,$DB
	.dc.b	$1C,$23,$B4,$62,$3F,$AA,$F2,$57,$F0,$53,$F8,$0E,$77,$B5,$9A,$D7
	.dc.b	$C7,$CA,$F4,$45,$BF,$F6,$F0,$F8,$FE,$86,$72,$F3,$C6,$6B,$F1,$B1
	.dc.b	$FE,$30,$83,$71,$7B,$3E,$78,$D1,$FB,$5A,$E3,$C7,$9E,$7D,$3E,$DF
	.dc.b	$96,$F9,$78,$FF,$BC,$5B,$7F,$23,$17,$AD,$BE,$0E,$1C,$13,$3F,$D0
	.dc.b	$E1,$C1,$FE,$90,$FE,$BD,$FD,$2F,$CB,$3C,$3F,$6E,$7F,$E0,$08,$C4
	.dc.b	$FF,$5F,$6C,$7F,$7C,$6B,$FE,$4D,$73,$9F,$DC,$5E,$17,$D7,$73,$5F
	.dc.b	$D6,$5C,$BF,$B6,$F9,$67,$B1,$E2,$C1,$18,$9F,$EE,$0D,$8F,$F0,$6B
	.dc.b	$5F,$F2,$F3,$FE,$16,$27,$AB,$CD,$FF,$0C,$FD,$13,$E4,$FB,$36,$FF
	.dc.b	$BE,$36,$FF,$C3,$2D,$7F,$CD,$5F,$2F,$DD,$FE,$E4,$D7,$FC,$4E,$D4
	.dc.b	$D3,$FC,$52,$4C,$3D,$CF,$F8,$B3,$E8,$9E,$1F,$C1,$6E,$1F,$E0,$F6
	.dc.b	$DF,$F8,$B1,$AF,$F9,$F7,$7F,$F7,$3B,$7D,$FF,$95,$18,$8F,$F0,$43
	.dc.b	$5F,$F1,$EA,$F7,$B6,$4B,$FF,$0F,$46,$5F,$F1,$07,$D0,$FB,$49,$BF
	.dc.b	$D7,$7E,$07,$CC,$BE,$1F,$ED,$36,$BF,$C5,$0D,$C7,$FC,$8A,$D7,$FD
	.dc.b	$2C,$BE,$1F,$89,$27,$8B,$F8,$5F,$A1,$7C,$3D,$24,$FF,$8D,$63,$2F
	.dc.b	$F8,$DF,$E6,$9F,$29,$F8,$D6,$C6,$63,$06,$7E,$4E,$47,$16,$E5,$FE
	.dc.b	$5C,$F9,$C7,$CA,$7E,$5D,$B5,$FE,$45,$6C,$EE,$7F,$39,$C5,$FA,$08
	.dc.b	$FF,$32,$B5,$FF,$5B,$35,$FF,$1F,$35,$FF,$39,$F9,$5F,$C3,$2F,$FC
	.dc.b	$A4,$D8,$FF,$37,$B5,$FF,$60,$35,$FF,$24,$7B,$8F,$89,$FD,$73,$6B
	.dc.b	$FC,$8D,$18,$3C,$FF,$13,$23,$FC,$C1,$82,$31,$7E,$AF,$9B,$F6,$92
	.dc.b	$B6,$F6,$99,$1F,$E5,$36,$CF,$FA,$1B,$E5,$9D,$9F,$CF,$5B,$5F,$DE
	.dc.b	$BE,$6F,$EB,$D7,$F2,$E8,$FF,$31,$4E,$A3,$07,$FA,$61,$FF,$4B,$92
	.dc.b	$BF,$86,$36,$7A,$6F,$DE,$7F,$A6,$5A,$2E,$D8,$F9,$7B,$7E,$36,$E3
	.dc.b	$E0,$16,$8C,$F8,$9F,$D7,$A3,$15,$EB,$F9,$7E,$E7,$83,$E5,$EC,$64
	.dc.b	$98,$75,$A4,$EA,$31,$F1,$7E,$83,$D3,$F2,$F6,$FE,$44,$7F,$2B,$BF
	.dc.b	$F4,$DD,$7F,$A2,$5A,$FF,$AF,$9A,$FA,$A3,$E1,$FC,$BF,$38,$F1,$3F
	.dc.b	$DC,$7E,$47,$FC,$99,$7F,$E2,$E5,$F2,$F9,$8D,$35,$D3,$6B,$E4,$4F
	.dc.b	$09,$E5,$F2,$FD,$4E,$DF,$1D,$63,$24,$AE,$31,$97,$E9,$74,$BE,$03
	.dc.b	$FC,$0C,$DA,$FF,$21,$AF,$95,$CC,$C9,$FD,$5B,$FD,$46,$78,$57,$EB
	.dc.b	$23,$07,$4E,$9E,$11,$E0,$7E,$27,$C7,$FC,$5F,$12,$E1,$F1,$5B,$5F
	.dc.b	$01,$BC,$9F,$E2,$2D,$7F,$92,$37,$C3,$F9,$C7,$CF,$F7,$2E,$1F,$3E
	.dc.b	$2E,$1F,$2A,$6D,$7F,$8B,$34,$FD,$23,$7A,$BE,$1F,$8F,$F1,$19,$8F
	.dc.b	$F0,$A7,$E0,$11,$C1,$FE,$6A,$F9,$57,$C8,$70,$CB,$F4,$11,$89,$FC
	.dc.b	$17,$CB,$FE,$17,$C8,$FC,$33,$63,$27,$E1,$E3,$2D,$3F,$15,$F2,$FF
	.dc.b	$8D,$3F,$1D,$F1,$16,$32,$4C,$3F,$13,$19,$7C,$7E,$EF,$F9,$17,$1F
	.dc.b	$F1,$A1,$FD,$A3,$5F,$F2,$7B,$88,$D1,$BF,$46,$E1,$EF,$8B,$7E,$E9
	.dc.b	$B7,$CA,$6D,$7F,$BC,$EF,$B3,$63,$36,$13,$E4,$BC,$C6,$26,$CF,$28
	.dc.b	$E0,$F9,$EB,$C8,$7D,$3A,$DD,$70,$6C,$6A,$BD,$0D,$BF,$43,$4F,$44
	.dc.b	$F0,$9F,$93,$1D,$4D,$3E,$2A,$ED,$18,$8F,$43,$6F,$D1,$18,$39,$92
	.dc.b	$FE,$8D,$F7,$7C,$71,$E6,$35,$5F,$24,$E9,$3B,$71,$1B,$9E,$24,$8C
	.dc.b	$BD,$9F,$25,$FA,$27,$A9,$EB,$8E,$A6,$8A,$E8,$F5,$4F,$57,$E9,$D7
	.dc.b	$8D,$E4,$78,$E7,$C2,$47,$33,$E3,$7A,$5A,$FF,$CB,$CD,$AF,$C3,$4C
	.dc.b	$3E,$51,$63,$4E,$4C,$9F,$10,$D4,$F4,$24,$FE,$E7,$DD,$FC,$69,$1F
	.dc.b	$B0,$D9,$F2,$B6,$FE,$04,$7E,$62,$31,$3E,$D8,$C3,$C8,$8F,$26,$CF
	.dc.b	$2B,$FF,$50,$4E,$49,$19,$AE,$B1,$F2,$2B,$F5,$91,$88,$FA,$71,$83
	.dc.b	$A9,$E3,$AE,$B5,$CE,$33,$3C,$CF,$D5,$D7,$3A,$FE,$0A,$DB,$FE,$10
	.dc.b	$DA,$F9,$07,$2A,$F0,$9C,$9C,$57,$26,$D7,$26,$9C,$A7,$DE,$F9,$EB
	.dc.b	$B4,$7F,$2A,$6D,$FF,$16,$7E,$07,$1F,$F8,$F6,$3F,$A3,$7B,$DF,$DC
	.dc.b	$7A,$7F,$B9,$33,$0F,$EE,$4E,$5A,$7F,$75,$E9,$FE,$F1,$D3,$FD,$E5
	.dc.b	$B1,$93,$FB,$D4,$65,$A7,$F4,$1F,$8F,$FE,$58,$7F,$2D,$F9,$0B,$19
	.dc.b	$3F,$29,$A7,$F1,$E9,$FC,$E9,$23,$3B,$23,$F9,$B7,$C8,$72,$AF,$E9
	.dc.b	$6F,$F8,$37,$D5,$97,$FE,$0D,$F8,$0B,$5F,$E0,$89,$BB,$7F,$69,$9B
	.dc.b	$5F,$AA,$DD,$F1,$BF,$E1,$7E,$AF,$F0,$C3,$67,$FC,$30,$DC,$7F,$C2
	.dc.b	$6D,$C3,$FA,$3F,$9F,$FD,$2C,$FE,$99,$EF,$5B,$FE,$C9,$1F,$BA,$23
	.dc.b	$EE,$6D,$FF,$66,$F3,$F8,$36,$7F,$A2,$59,$FF,$3A,$FA,$16,$7F,$C9
	.dc.b	$9E,$8A,$31,$59,$94,$BE,$5F,$37,$C8,$EA,$59,$E7,$7A,$7F,$AA,$57
	.dc.b	$19,$DC,$62,$74,$FF,$96,$CB,$E7,$BE,$0A,$C6,$1B,$5C,$A3,$FC,$C5
	.dc.b	$18,$7F,$CC,$D3,$C6,$74,$BE,$0B,$A9,$DB,$E1,$6C,$EA,$3E,$05,$7F
	.dc.b	$26,$AE,$13,$F9,$06,$C7,$F2,$56,$BF,$CA,$DC,$E7,$F2,$57,$C3,$76
	.dc.b	$9F,$2B,$EE,$F8,$69,$7E,$6F,$8E,$B5,$EA,$9B,$87,$39,$B5,$E5,$F8
	.dc.b	$3F,$8F,$F3,$7F,$21,$B1,$F0,$1B,$87,$C4,$6D,$FD,$DF,$BB,$FF,$16
	.dc.b	$4F,$FA,$35,$B1,$E9,$F9,$CF,$AB,$18,$BF,$E9,$2F,$0A,$FE,$94,$D7
	.dc.b	$3E,$57,$CA,$7F,$C2,$91,$96,$C7,$DE,$F9,$CE,$AD,$AF,$F2,$17,$27
	.dc.b	$F5,$82,$3F,$BC,$7A,$57,$E1,$7E,$6E,$F1,$9A,$4A,$FE,$0F,$F8,$1F
	.dc.b	$FA,$06,$BF,$AE,$35,$FE,$C4,$D7,$FB,$2B,$5F,$F1,$67,$CD,$FF,$A6
	.dc.b	$09,$EC,$F8,$9F,$F4,$DF,$CD,$BD,$6F,$F2,$CD,$7F,$69,$8C,$69,$FC
	.dc.b	$61,$7A,$FF,$9D,$9D,$57,$07,$FB,$A7,$D0,$3D,$1F,$E3,$F0,$8C,$3F
	.dc.b	$C8,$9A,$FF,$77,$6B,$FD,$F7,$B5,$F3,$9C,$9B,$1F,$CA,$9A,$FF,$7F
	.dc.b	$6B,$C7,$E7,$38,$11,$E0,$F9,$3B,$3E,$84,$7E,$97,$87,$04,$7B,$7A
	.dc.b	$5E,$12,$38,$3C,$5E,$97,$94,$8E,$0F,$D8,$7C,$F7,$72,$38,$3D,$5F
	.dc.b	$33,$ED,$23,$83,$DD,$F3,$3F,$04,$8E,$0F,$DE,$7C,$4F,$AC,$8E,$0F
	.dc.b	$8D,$E8,$FC,$92,$38,$3E,$57,$A1,$F3,$48,$E0,$FE,$11,$F0,$FF,$1C
	.dc.b	$8E,$0F,$A5,$F2,$BF,$54,$8E,$0F,$AD,$F1,$FF,$64,$8E,$0F,$E2,$9F
	.dc.b	$13,$F4,$C8,$E0,$FB,$DE,$77,$E0,$48,$E0,$FE,$39,$C3,$F6,$C8,$E0
	.dc.b	$FE,$DF,$F3,$6F,$6F,$FC,$93,$E3,$7F,$45,$F9,$C7,$81,$8F,$0B,$FC
	.dc.b	$9B,$18,$C3,$F9,$B6,$1C,$5B,$C7,$B7,$FD,$31,$19,$58,$7F,$51,$8C
	.dc.b	$36,$78,$4B,$CA,$FE,$78,$FF,$43,$DF,$FB,$9B,$7D,$2D,$CE,$A3,$AE
	.dc.b	$5F,$E6,$28,$79,$FA,$F3,$E5,$7F,$4D,$9C,$BF,$D3,$FE,$6D,$F3,$9E
	.dc.b	$2C,$9E,$FD,$3C,$B3,$AA,$E1,$DB,$8B,$91,$83,$A7,$6F,$09,$91,$83
	.dc.b	$C1,$C7,$DF,$DD,$91,$83,$C9,$C1,$E0,$99,$18,$3D,$1D,$BF,$31,$FF
	.dc.b	$3E,$64,$8C,$47,$FA,$0E,$31,$C5,$FE,$2E,$6C,$70,$F9,$C7,$CE,$65
	.dc.b	$B1,$98,$C1,$FC,$61,$7C,$2D,$23,$FC,$71,$FE,$3B,$E7,$7F,$80,$A3
	.dc.b	$FB,$C1,$F5,$8F,$F4,$B4,$7F,$B7,$7E,$73,$F0,$11,$FE,$9A,$9E,$1C
	.dc.b	$93,$9B,$FC,$09,$3C,$B3,$FE,$F7,$3D,$79,$A5,$7E,$1F,$E1,$ED,$F7
	.dc.b	$3F,$DD,$5E,$C7,$8C,$CE,$1F,$E0,$47,$CD,$EC,$4B,$EF,$16,$B9,$57
	.dc.b	$FA,$A8,$FF,$15,$16,$7F,$D6,$3D,$1F,$E6,$F7,$F5,$7C,$DF,$E2,$29
	.dc.b	$FF,$46,$C7,$EB,$B8,$F1,$39,$7F,$C4,$FF,$39,$ED,$9F,$F5,$DB,$70
	.dc.b	$0C,$1E,$FF,$0D,$F8,$EF,$30,$FF,$15,$61,$C7,$03,$FE,$1A,$8F,$F6
	.dc.b	$3F,$CE,$3C,$4F,$F1,$BA,$F0,$EC,$61,$93,$FE,$76,$9D,$36,$3F,$BC
	.dc.b	$C6,$2F,$FD,$9D,$F3,$8F,$8A,$FE,$FA,$DB,$FF,$3F,$B6,$77,$3F,$E0
	.dc.b	$2F,$9C,$7B,$B4,$97,$FE,$86,$8C,$B7,$E4,$B8,$11,$C1,$FE,$8C,$F9
	.dc.b	$CE,$D8,$7F,$B9,$5A,$FF,$BA,$1A,$FC,$E6,$AF,$9C,$78,$5F,$E1,$66
	.dc.b	$D7,$FA,$61,$B3,$FE,$ED,$6B,$FE,$F3,$FA,$4F,$3B,$FE,$F1,$6B,$9C
	.dc.b	$BF,$C4,$31,$88,$FF,$7D,$EE,$F8,$8F,$F1,$2B,$70,$FF,$53,$B6,$7F
	.dc.b	$DF,$8D,$7F,$0E,$7E,$23,$E7,$5F,$11,$FE,$2E,$6D,$7F,$AC,$1B,$3C
	.dc.b	$0E,$13,$FE,$34,$FA,$7F,$C8,$D7,$FA,$D9,$B3,$B0,$E1,$E4,$D7,$C7
	.dc.b	$F3,$CF,$93,$FF,$1F,$B6,$BF,$D8,$2D,$9E,$8D,$7C,$9F,$3C,$F9,$3F
	.dc.b	$F2,$3B,$6B,$FD,$92,$D9,$F2,$35,$F6,$7D,$2B,$D8,$FF,$27,$B6,$BF
	.dc.b	$DA,$2D,$9F,$63,$5F,$89,$F3,$EF,$63,$FC,$AE,$DA,$FF,$6C,$B6,$7E
	.dc.b	$23,$5F,$99,$F3,$EF,$67,$FC,$BE,$DA,$FF,$70,$B6,$7E,$63,$5F,$A9
	.dc.b	$F4,$4F,$4B,$FC,$CE,$DA,$FF,$74,$B6,$7E,$A3,$5F,$E5,$FF,$2D,$FE
	.dc.b	$6E,$F3,$8D,$25,$7F,$92,$7E,$4B,$FC,$E7,$F0,$46,$92,$3F,$DE,$FF
	.dc.b	$41,$F8,$8C,$3F,$10,$D7,$F2,$2D,$7F,$8C,$B5,$7C,$BF,$FB,$56,$7F
	.dc.b	$E0,$0B,$63,$F1,$8D,$7F,$28,$D7,$F3,$B6,$F9,$4F,$39,$2B,$FC,$C4
	.dc.b	$FF,$A3,$3E,$01,$F4,$14,$93,$FE,$C1,$7E,$26,$DF,$ED,$8D,$7E,$57
	.dc.b	$A1,$FE,$E7,$5F,$33,$99,$97,$FA,$1C,$FF,$B1,$C7,$28,$C5,$7D,$19
	.dc.b	$FD,$9B,$D2,$2E,$9C,$A7,$FA,$B4,$FC,$49,$FF,$71,$7C,$BF,$FA,$45
	.dc.b	$BF,$37,$FE,$EC,$8C,$57,$F5,$D5,$19,$F8,$DD,$36,$B8,$9D,$3B,$79
	.dc.b	$1C,$FD,$7B,$19,$26,$1D,$63,$2F,$9B,$CB,$F8,$C7,$C7,$F0,$6D,$7F
	.dc.b	$BE,$73,$FE,$52,$49,$FC,$81,$23,$FD,$FD,$F3,$5F,$ED,$1B,$FE,$BB
	.dc.b	$5F,$D0,$F8,$15,$FC,$6A,$B5,$4D,$5F,$37,$F0,$FC,$1F,$88,$DB,$F9
	.dc.b	$2D,$AF,$8C,$DC,$3A,$C7,$F6,$CB,$FF,$5A,$57,$D0,$F8,$FF,$9D,$F0
	.dc.b	$7F,$3D,$91,$93,$EA,$C6,$2B,$FC,$1D,$97,$F8,$0D,$FE,$EE,$D4,$FF
	.dc.b	$46,$C3,$C3,$18,$F6,$7C,$71,$89,$47,$FB,$91,$B7,$F8,$F6,$D7,$F3
	.dc.b	$16,$BF,$CE,$5F,$01,$FE,$10,$7C,$73,$DA,$7E,$67,$B3,$F8,$BE,$D9
	.dc.b	$1F,$90,$E7,$B7,$F8,$D9,$B8,$FA,$34,$FD,$1D,$FE,$5A,$7F,$29,$CF
	.dc.b	$F9,$5E,$7B,$3F,$9A,$6E,$1F,$97,$6E,$1C,$DF,$24,$7C,$8F,$89,$AF
	.dc.b	$C8,$6E,$BF,$CD,$6B,$E3,$46,$0F,$8C,$4B,$F8,$CD,$AF,$AB,$F3,$7B
	.dc.b	$9F,$C8,$BF,$AB,$F3,$24,$7E,$C3,$DA,$B5,$FA,$D9,$B8,$7C,$19,$B5
	.dc.b	$FD,$BB,$57,$C3,$FE,$D7,$A3,$F6,$CD,$9F,$DD,$B7,$1F,$DC,$37,$0F
	.dc.b	$C0,$78,$5E,$E3,$DF,$EC,$5B,$FD,$3C,$5B,$FC,$7C,$DA,$E3,$5A,$3F
	.dc.b	$5B,$19,$7F,$55,$56,$BF,$2B,$35,$FC,$AC,$5D,$BF,$C7,$57,$D2,$7F
	.dc.b	$3B,$39,$AD,$36,$FC,$13,$A6,$D7,$E6,$EB,$71,$83,$F7,$8E,$27,$81
	.dc.b	$EC,$8D,$3F,$AA,$AE,$CE,$AB,$C8,$49,$E6,$70,$8F,$5B,$6F,$D6,$FC
	.dc.b	$23,$15,$F3,$9A,$6E,$FC,$97,$D6,$FE,$3C,$E9,$F1,$3E,$06,$9E,$77
	.dc.b	$91,$B8,$E5,$1F,$C0,$A7,$AC,$F8,$E3,$E0,$1C,$8F,$49,$6F,$D2,$F5
	.dc.b	$31,$7E,$96,$CF,$81,$A7,$1A,$F9,$CF,$F0,$59,$FE,$18,$49,$FE,$18
	.dc.b	$D8,$FE,$18,$D7,$84,$7F,$38,$F0,$7F,$0C,$47,$E7,$36,$3D,$0D,$C3
	.dc.b	$E0,$E1,$FD,$DE,$B6,$4A,$DB,$67,$75,$C6,$3E,$34,$7F,$25,$8C,$4F
	.dc.b	$C3,$8C,$4F,$13,$67,$F4,$99,$DC,$FC,$C9,$DB,$E0,$23,$E0,$9E,$51
	.dc.b	$D5,$E4,$61,$E4,$E1,$E2,$F8,$1F,$E8,$33,$E0,$AF,$F1,$54,$FC,$93
	.dc.b	$93,$89,$FE,$93,$3B,$BF,$E3,$ED,$AE,$13,$D8,$F2,$1D,$9E,$73,$C1
	.dc.b	$A7,$0B,$F6,$4F,$13,$D2,$71,$31,$E1,$7F,$8C,$63,$FA,$AB,$67,$FC
	.dc.b	$5F,$F2,$DF,$CD,$23,$15,$FE,$D4,$6A,$68,$8E,$1F,$37,$FC,$F3,$C4
	.dc.b	$FF,$44,$E1,$FD,$89,$FF,$6B,$18,$C3,$FD,$32,$E3,$0E,$1E,$EF,$16
	.dc.b	$CF,$FB,$92,$B9,$7B,$BD,$4E,$DF,$03,$67,$47,$48,$FF,$22,$11,$FF
	.dc.b	$1D,$60,$8C,$5A,$FC,$DF,$01,$CE,$31,$7E,$68,$F6,$7C,$0C,$8E,$F3
	.dc.b	$FE,$F8,$C1,$F1,$39,$8E,$26,$6B,$18,$7C,$39,$B7,$F0,$BD,$9F,$91
	.dc.b	$F3,$7F,$25,$B1,$DB,$06,$3F,$EC,$EC,$3E,$65,$EE,$B5,$28,$FE,$DB
	.dc.b	$F3,$5F,$E2,$C3,$FC,$5B,$0F,$9B,$CF,$CC,$4B,$FF,$2F,$52,$3F,$C3
	.dc.b	$38,$FF,$B3,$A3,$13,$A9,$CC,$84,$F6,$F9,$BE,$91,$F0,$8F,$4B,$F8
	.dc.b	$5F,$A0,$97,$E8,$F9,$2B,$5B,$C9,$58,$D3,$F1,$55,$6F,$F1,$3F,$09
	.dc.b	$F8,$FF,$9F,$FC,$83,$63,$E1,$B7,$1E,$8D,$BF,$F1,$AF,$EA,$7F,$E5
	.dc.b	$2A,$F9,$CD,$AF,$42,$FA,$0F,$F0,$BB,$9A,$E3,$5F,$0A,$93,$C5,$FF
	.dc.b	$77,$35,$FF,$7B,$B5,$F9,$0F,$38,$CF,$D0,$76,$7B,$BE,$0F,$9D,$91
	.dc.b	$E6,$F5,$ED,$F9,$A6,$E1,$EC,$9B,$5E,$0F,$5F,$D3,$EB,$FA,$9B,$1F
	.dc.b	$01,$B8,$7C,$06,$DF,$C6,$F5,$FE,$09,$F0,$BD,$4B,$7F,$22,$3E,$16
	.dc.b	$86,$92,$3F,$D5,$51,$97,$E7,$FA,$BC,$B1,$F9,$EF,$F9,$13,$BD,$F3
	.dc.b	$5F,$35,$B5,$FE,$B3,$6D,$7E,$44,$8E,$1C,$0F,$A0,$FB,$53,$FE,$BB
	.dc.b	$6C,$68,$D7,$0F,$DB,$8C,$C7,$DA,$F5,$FD,$A4,$AF,$5B,$F8,$1F,$9F
	.dc.b	$70,$FE,$A6,$BA,$46,$3A,$FF,$D9,$4B,$8D,$9E,$2F,$85,$FA,$3E,$7D
	.dc.b	$8F,$FB,$46,$11,$87,$FD,$A8,$D7,$D4,$D7,$E0,$DF,$FB,$F5,$FF,$6D
	.dc.b	$35,$F8,$0D,$77,$E4,$77,$F2,$1F,$3D,$49,$3F,$EE,$38,$CB,$FC,$B3
	.dc.b	$83,$CD,$DB,$F6,$23,$04,$3E,$77,$1B,$DA,$F6,$36,$3F,$0B,$18,$3C
	.dc.b	$5E,$D4,$8F,$03,$5F,$B1,$4F,$9E,$F0,$92,$3F,$9F,$BD,$BD,$99,$1F
	.dc.b	$92,$6B,$F8,$1C,$9F,$33,$F0,$49,$1E,$F7,$DB,$F3,$D2,$3F,$A5,$B5
	.dc.b	$F6,$FB,$9E,$C2,$3F,$A0,$7E,$67,$B5,$23,$E5,$35,$F5,$FC,$37,$CA
	.dc.b	$23,$FD,$61,$F8,$FE,$EC,$8F,$D5,$B5,$F5,$7C,$6F,$D9,$23,$F5,$DF
	.dc.b	$AB,$F0,$D2,$3F,$B1,$B5,$FA,$BE,$FF,$D4,$23,$FB,$87,$EE,$EF,$23
	.dc.b	$FB,$5B,$5F,$BB,$EF,$7D,$C2,$3F,$C0,$1F,$CD,$7C,$EB,$E0,$7F,$71
	.dc.b	$E3,$7F,$B9,$BE,$79,$E6,$E3,$C3,$FD,$E4,$63,$1F,$EC,$58,$71,$6F
	.dc.b	$CE,$ED,$FE,$A5,$19,$7F,$A8,$37,$0F,$F7,$9A,$9E,$13,$FD,$E3,$D8
	.dc.b	$BF,$33,$FE,$BE,$8F,$EE,$0D,$7F,$AD,$7A,$FF,$E0,$97,$FB,$A2,$F6
	.dc.b	$B9,$B9,$8D,$D6,$27,$BF,$CE,$BE,$7B,$BF,$7C,$BF,$EF,$D3,$96,$FC
	.dc.b	$B7,$0A,$91,$83,$7E,$07,$49,$91,$83,$9F,$17,$B3,$A2,$46,$0E,$FB
	.dc.b	$79,$26,$46,$0F,$1F,$6F,$F8,$95,$FE,$4D,$97,$E9,$B8,$7F,$94,$42
	.dc.b	$30,$EA,$78,$3F,$CC,$A1,$F3,$CF,$9E,$E3,$18,$6C,$66,$7B,$B5,$EF
	.dc.b	$F1,$5F,$C8,$D3,$48,$FF,$45,$E6,$FF,$14,$3F,$CB,$9A,$F0,$1F,$3D
	.dc.b	$FE,$38,$7F,$A4,$C7,$0E,$33,$4E,$7E,$6E,$27,$FC,$38,$7F,$92,$E3
	.dc.b	$BF,$26,$75,$FF,$14,$B6,$3F,$AB,$73,$7F,$61,$AE,$33,$97,$FA,$D7
	.dc.b	$C0,$D7,$FC,$58,$D7,$F1,$D5,$EC,$8C,$7C,$0C,$8C,$1F,$01,$E5,$83
	.dc.b	$96,$9D,$71,$78,$D8,$BF,$F2,$75,$7E,$D5,$BC,$79,$CB,$FC,$75,$39
	.dc.b	$7F,$C7,$9F,$3D,$F3,$5B,$AF,$5F,$8B,$F5,$4E,$6B,$E4,$52,$3E,$43
	.dc.b	$CB,$7D,$DF,$07,$E1,$7C,$F7,$FA,$A6,$F8,$B7,$2F,$F0,$03,$FE,$CF
	.dc.b	$BF,$F2,$73,$97,$FC,$A8,$FF,$AC,$5F,$F0,$1B,$5E,$3E,$77,$F8,$E6
	.dc.b	$31,$1F,$E5,$BF,$23,$FC,$BE,$FF,$81,$5A,$FF,$83,$1A,$FF,$84,$5A
	.dc.b	$F1,$F9,$FF,$F2,$24,$1A,$4B,$FD,$4C,$7F,$6E,$6F,$9F,$C0,$8F,$EB
	.dc.b	$BE,$C3,$AD,$9A,$4F,$5B,$FC,$A4,$F7,$9E,$18,$60,$8F,$3B,$D9,$1C
	.dc.b	$1F,$BC,$F2,$9C,$1F,$E5,$D6,$C6,$63,$06,$7E,$32,$47,$16,$E5,$FE
	.dc.b	$2D,$A7,$C6,$F3,$25,$77,$7F,$CD,$5F,$33,$FA,$5A,$FF,$1C,$B5,$FF
	.dc.b	$1F,$35,$FF,$48,$35,$2F,$7B,$99,$2B,$F8,$BC,$65,$BE,$EF,$80,$8E
	.dc.b	$0F,$E3,$BF,$2C,$F4,$3F,$CF,$CD,$8C,$C6,$0C,$F9,$F2,$38,$B7,$1F
	.dc.b	$F2,$75,$3E,$4F,$99,$2B,$BB,$FE,$8B,$F4,$7F,$27,$46,$93,$9B,$5F
	.dc.b	$1D,$2F,$F9,$9C,$FE,$8E,$BF,$CC,$2D,$7F,$CC,$CB,$E3,$7B,$12,$78
	.dc.b	$BF,$E9,$BF,$94,$7C,$A5,$24,$FF,$40,$8C,$BF,$CD,$3C,$DF,$F5,$17
	.dc.b	$90,$69,$23,$FA,$3F,$C8,$FF,$13,$6F,$8B,$FE,$A9,$8C,$67,$FE,$73
	.dc.b	$6B,$FE,$AA,$E8,$FF,$58,$9F,$EB,$2E,$8B,$7F,$EB,$88,$FF,$48,$91
	.dc.b	$FF,$47,$B6,$7F,$D7,$7C,$FC,$1B,$3F,$EC,$0D,$3F,$D7,$A7,$F3,$1F
	.dc.b	$D5,$3F,$80,$F8,$1F,$EC,$68,$CC,$EC,$FF,$7A,$4E,$9F,$F5,$EC,$71
	.dc.b	$EC,$E5,$D9,$67,$6D,$C3,$FD,$C9,$7F,$C0,$5B,$5C,$2B,$AF,$A3,$E3
	.dc.b	$3C,$9E,$8D,$8C,$9F,$DC,$AF,$FC,$18,$D9,$FF,$6A,$46,$2F,$FD,$3C
	.dc.b	$DC,$BF,$99,$BF,$41,$B3,$F0,$77,$F8,$5E,$34,$C3,$DF,$E8,$DC,$7D
	.dc.b	$33,$70,$E7,$36,$BF,$D6,$98,$FF,$9B,$1F,$8F,$E7,$7C,$DF,$47,$E7
	.dc.b	$36,$32,$7C,$C8,$CA,$C5,$87,$F4,$AF,$94,$FC,$8C,$62,$7F,$D8,$51
	.dc.b	$89,$F6,$FC,$CF,$F6,$B8,$C4,$EB,$87,$FD,$F1,$1C,$AF,$FD,$8E,$D7
	.dc.b	$53,$FE,$1E,$8E,$31,$E3,$C9,$ED,$F5,$BF,$C4,$3F,$89,$F9,$69,$87
	.dc.b	$E1,$BC,$FB,$87,$E0,$26,$E7,$F4,$A6,$D7,$C8,$F1,$FF,$25,$F4,$9F
	.dc.b	$93,$6C,$7E,$41,$B8,$7E,$29,$B7,$F7,$3E,$5F,$F3,$47,$E6,$FC,$2B
	.dc.b	$7F,$98,$8F,$C2,$91,$FC,$53,$6F,$F2,$BE,$17,$06,$CF,$E8,$3C,$1F
	.dc.b	$D3,$1F,$A7,$F9,$7B,$19,$24,$FE,$7A,$32,$FD,$73,$EE,$7E,$AB,$F7
	.dc.b	$67,$EF,$65,$FE,$06,$8F,$F0,$44,$7F,$BF,$5D,$4F,$10,$E9,$E6,$7F
	.dc.b	$8A,$5C,$3A,$31,$E8,$FF,$8B,$7C,$16,$14,$C6,$37,$0F,$19,$F9,$1B
	.dc.b	$57,$08,$CC,$7E,$56,$30,$F9,$5F,$E5,$D1,$A8,$E3,$5D,$5D,$3E,$08
	.dc.b	$D4,$F2,$C7,$FC,$4E,$D8,$F1,$36,$BC,$17,$CE,$B7,$76,$F7,$36,$BC
	.dc.b	$D3,$5F,$5C,$5B,$E9,$1E,$6A,$F1,$CE,$6B,$4D,$8F,$3C,$7A,$67,$D7
	.dc.b	$18,$23,$F0,$E7,$83,$A3,$11,$AB,$F4,$B4,$F4,$96,$3D,$2D,$7D,$2D
	.dc.b	$BF,$49,$2B,$BD,$74,$AE,$E7,$85,$C1,$2B,$BC,$65,$F9,$F7,$F0,$1E
	.dc.b	$CD,$8D,$1E,$42,$9E,$42,$DF,$92,$3E,$35,$77,$9E,$0D,$78,$36,$78
	.dc.b	$57,$D2,$3C,$27,$BE,$FD,$AF,$98,$D5,$78,$2F,$FB,$61,$23,$FB,$8B
	.dc.b	$63,$FB,$73,$5F,$CF,$1F,$81,$33,$1F,$99,$C3,$F6,$38,$79,$DB,$1E
	.dc.b	$19,$FD,$A4,$61,$F6,$C7,$EA,$63,$11,$D6,$7F,$D0,$71,$8C,$3F,$22
	.dc.b	$DB,$FF,$51,$36,$FC,$57,$F9,$98,$C3,$E0,$30,$48,$FC,$34,$6A,$FB
	.dc.b	$CE,$4F,$AF,$19,$9F,$34,$66,$7D,$CD,$8F,$74,$72,$7A,$C6,$4F,$79
	.dc.b	$FD,$5E,$7A,$34,$D1,$F0,$63,$B9,$C8,$8F,$90,$DD,$FC,$46,$DF,$C4
	.dc.b	$6C,$FC,$43,$8D,$FA,$1A,$7A,$0E,$F1,$D5,$CB,$E8,$17,$E8,$69,$D8
	.dc.b	$F9,$D1,$C1,$F5,$1E,$D7,$F8,$F6,$3F,$EB,$CA,$FE,$75,$E8,$FF,$33
	.dc.b	$F5,$BF,$9A,$32,$B6,$E5,$A7,$2F,$5B,$F9,$D7,$AD,$FC,$ED,$B1,$93
	.dc.b	$BC,$65,$A7,$E5,$DF,$C4,$9F,$8A,$8B,$19,$3F,$3F,$8F,$D7,$9F,$F6
	.dc.b	$AB,$F9,$76,$B9,$8D,$BF,$1A,$3F,$2B,$ED,$F4,$8F,$D0,$BF,$D4,$49
	.dc.b	$3F,$13,$E6,$2C,$FF,$59,$CD,$58,$CF,$FB,$05,$5B,$FE,$C1,$F3,$1F
	.dc.b	$D6,$FE,$63,$FA,$E3,$63,$E7,$B7,$1E,$8D,$BF,$D6,$FC,$47,$EA,$8F
	.dc.b	$D5,$FC,$BD,$BF,$DE,$47,$BA,$FF,$02,$DE,$03,$49,$3F,$C0,$FE,$23
	.dc.b	$95,$7F,$00,$7F,$B8,$5F,$F7,$1A,$B5,$F8,$46,$D7,$C1,$6E,$1F,$DF
	.dc.b	$DF,$EF,$17,$2B,$F1,$7F,$0F,$6F,$E4,$4D,$BF,$F0,$34,$DA,$FE,$1D
	.dc.b	$E9,$7F,$08,$3F,$84,$FA,$96,$3F,$8D,$47,$B4,$8F,$DC,$CF,$FD,$68
	.dc.b	$D8,$FE,$23,$F2,$FC,$72,$FE,$3A,$FF,$83,$7E,$23,$FC,$1D,$EB,$DB
	.dc.b	$FD,$0B,$6B,$EF,$37,$3F,$F0,$E3,$FE,$15,$CA,$57,$EA,$31,$B7,$F8
	.dc.b	$59,$B7,$FE,$23,$9B,$5F,$E1,$3F,$90,$7C,$8A,$3F,$C1,$D1,$86,$99
	.dc.b	$C3,$FC,$51,$19,$6C,$FE,$06,$31,$1F,$D0,$9A,$F7,$8D,$1C,$EF,$A4
	.dc.b	$25,$F2,$B9,$BE,$07,$91,$C4,$FF,$9E,$FE,$99,$E2,$FE,$45,$B5,$FA
	.dc.b	$E6,$CF,$ED,$9A,$FF,$56,$FA,$57,$C6,$7E,$65,$B5,$FB,$66,$CF,$EF
	.dc.b	$9A,$FF,$5E,$F9,$47,$CA,$3E,$13,$F7,$B1,$86,$FC,$97,$F2,$2A,$E1
	.dc.b	$5B,$7F,$84,$F7,$39,$7F,$82,$C7,$F9,$51,$B1,$98,$FE,$12,$D7,$F8
	.dc.b	$8F,$E0,$CF,$98,$E2,$D8,$FE,$1E,$D7,$71,$AF,$7A,$46,$9A,$F9,$36
	.dc.b	$F0,$91,$F0,$BE,$4F,$99,$91,$E4,$6B,$EC,$F9,$AF,$49,$1F,$4B,$EE
	.dc.b	$F3,$3F,$84,$41,$A4,$E1,$D7,$3C,$9F,$E5,$33,$F9,$86,$C7,$3E,$59
	.dc.b	$1A,$6B,$FC,$DA,$5C,$1E,$12,$47,$47,$C9,$C1,$23,$BB,$5E,$5F,$35
	.dc.b	$E9,$23,$E4,$7D,$9B,$48,$F3,$B5,$F1,$7C,$D7,$C1,$23,$EC,$7E,$27
	.dc.b	$C5,$48,$F7,$B5,$F5,$7C,$4F,$C9,$23,$F1,$1F,$99,$F3,$52,$3E,$3B
	.dc.b	$5F,$85,$F2,$9F,$44,$8F,$CC,$7E,$A7,$C8,$C8,$F9,$ED,$7E,$57,$C7
	.dc.b	$FD,$92,$3F,$51,$FC,$CF,$CA,$BE,$0B,$FD,$41,$E0,$FE,$FB,$E5,$BF
	.dc.b	$5F,$7F,$E9,$B6,$BF,$EB,$3A,$E0,$FF,$A7,$D7,$C1,$F2,$E6,$FF,$55
	.dc.b	$46,$A7,$86,$19,$7F,$DF,$D1,$C1,$FF,$54,$7A,$0F,$4F,$50,$F7,$1D
	.dc.b	$B8,$6C,$70,$9D,$D3,$D5,$B1,$27,$AC,$74,$F9,$67,$C3,$73,$E2,$95
	.dc.b	$FE,$B9,$9C,$B7,$E1,$F8,$54,$8C,$1B,$F8,$7E,$93,$23,$07,$3D,$BD
	.dc.b	$1F,$19,$23,$07,$7E,$1F,$24,$C8,$C1,$E3,$F8,$77,$27,$F8,$53,$34
	.dc.b	$62,$7F,$C3,$B1,$89,$D4,$E6,$7F,$CF,$F2,$F8,$87,$CB,$73,$6C,$F2
	.dc.b	$6B,$C1,$AF,$F8,$AB,$E3,$BF,$90,$A6,$91,$FE,$DD,$CD,$FE,$E3,$7F
	.dc.b	$DE,$A0,$E0,$3E,$5B,$FC,$4A,$FF,$A2,$63,$87,$1F,$F4,$A8,$E0,$D7
	.dc.b	$9B,$78,$E4,$FF,$BC,$79,$5D,$D8,$BF,$93,$3F,$EA,$0F,$7F,$91,$9D
	.dc.b	$FF,$C8,$6F,$5F,$7E,$4F,$A6,$2D,$72,$3D,$54,$E5,$F3,$36,$BC,$5D
	.dc.b	$B8,$9C,$F1,$C9,$E2,$7E,$C2,$31,$7D,$47,$CB,$7F,$9D,$A7,$71,$F2
	.dc.b	$63,$0E,$F4,$7C,$87,$C7,$9C,$E5,$FD,$E1,$FF,$3B,$5F,$D1,$BE,$BB
	.dc.b	$EF,$DA,$E0,$FC,$FF,$96,$7C,$3F,$F9,$1D,$B7,$FE,$64,$6D,$FF,$9B
	.dc.b	$1A,$FF,$B3,$1C,$E3,$FE,$08,$5F,$0D,$D6,$FB,$3F,$77,$C3,$7C,$7F
	.dc.b	$F9,$55,$B8,$FF,$9C,$9B,$8F,$F9,$E9,$AF,$FB,$69,$7C,$87,$12,$4F
	.dc.b	$17,$FC,$C1,$F4,$AF,$85,$A4,$9F,$F3,$FC,$65,$FF,$40,$FC,$BF,$6C
	.dc.b	$3E,$43,$5F,$CD,$35,$34,$F1,$C7,$2B,$FF,$38,$46,$27,$FD,$26,$D7
	.dc.b	$FD,$E7,$F4,$9F,$89,$7F,$D2,$ED,$73,$C9,$FE,$79,$8C,$47,$DB,$F8
	.dc.b	$DF,$E4,$B8,$7F,$A6,$1B,$1F,$EA,$06,$BF,$F0,$01,$AF,$1F,$94,$FF
	.dc.b	$42,$41,$A4,$9F,$F5,$1F,$D1,$BE,$4B,$FD,$14,$D8,$CC,$60,$FF,$47
	.dc.b	$7C,$CB,$D1,$A4,$9F,$F5,$6C,$65,$BE,$8F,$02,$38,$3F,$D6,$7F,$44
	.dc.b	$F4,$BF,$D3,$4D,$8C,$C6,$0F,$F4,$F7,$CD,$3E,$22,$92,$7F,$D7,$B1
	.dc.b	$96,$FC,$47,$02,$38,$3F,$D8,$7F,$43,$BE,$0F,$F6,$5B,$5F,$D6,$07
	.dc.b	$0F,$EB,$DA,$FF,$21,$F9,$B7,$CA,$FF,$AC,$DB,$5F,$ED,$06,$CF,$FB
	.dc.b	$61,$AF,$67,$F5,$71,$BE,$BE,$44,$7F,$DB,$3F,$3B,$C2,$BF,$80,$35
	.dc.b	$FE,$64,$D4,$DF,$31,$E9,$7F,$B1,$DB,$5F,$EE,$46,$D7,$F4,$39,$FF
	.dc.b	$66,$7C,$B7,$E5,$23,$F9,$F3,$53,$49,$E0,$70,$25,$FF,$BC,$23,$F8
	.dc.b	$5C,$FF,$17,$F9,$AF,$C7,$3F,$C9,$5B,$E1,$FF,$A5,$3E,$69,$F3,$7F
	.dc.b	$D1,$E7,$38,$71,$E9,$FF,$70,$47,$2C,$3E,$BB,$5D,$4F,$FC,$00,$8E
	.dc.b	$31,$FD,$5E,$DF,$37,$D6,$FF,$AA,$46,$2F,$F9,$AD,$23,$FD,$37,$8F
	.dc.b	$97,$E4,$37,$E6,$70,$65,$F0,$AD,$BF,$CD,$5F,$F1,$03,$98,$EF,$DF
	.dc.b	$E0,$F9,$BB,$3C,$5B,$8F,$F4,$9B,$46,$31,$F8,$CF,$A3,$E1,$65,$F8
	.dc.b	$78,$2E,$1C,$67,$FD,$7E,$F8,$DA,$FC,$36,$BE,$F6,$E1,$FE,$85,$AF
	.dc.b	$67,$37,$4C,$7E,$B3,$F2,$39,$64,$FC,$1F,$9B,$B8,$FA,$66,$DF,$DD
	.dc.b	$9B,$5F,$D6,$7E,$37,$FC,$14,$FD,$07,$FA,$FB,$9B,$FF,$04,$46,$1F
	.dc.b	$F0,$83,$C1,$FE,$DC,$E5,$39,$AF,$F6,$FB,$9E,$45,$FF,$84,$DB,$7F
	.dc.b	$80,$7F,$67,$7F,$DE,$5C,$B6,$7F,$4A,$49,$FC,$F3,$63,$FC,$01,$E7
	.dc.b	$78,$67,$FB,$B5,$72,$C3,$FD,$E3,$39,$AF,$5F,$9D,$EE,$3D,$FE,$B5
	.dc.b	$C3,$E2,$45,$9F,$7C,$DB,$F6,$FA,$9F,$1B,$D5,$F8,$ED,$C3,$E6,$36
	.dc.b	$FE,$3B,$6B,$C7,$5F,$E3,$5C,$4D,$9C,$32,$FF,$92,$DF,$F9,$06,$B7
	.dc.b	$18,$84,$61,$DB,$5F,$E5,$AE,$6F,$E1,$FC,$07,$59,$F9,$2F,$E0,$09
	.dc.b	$5F,$85,$F9,$BB,$3F,$80,$9B,$5F,$7A,$6D,$7E,$37,$D0,$FC,$4F,$CD
	.dc.b	$FE,$29,$B1,$F8,$F6,$E1,$F8,$A6,$DF,$D6,$F9,$AF,$88,$7C,$5F,$9B
	.dc.b	$B7,$FC,$1B,$34,$E3,$0F,$C0,$D5,$BE,$57,$BA,$E4,$D2,$57,$E2,$23
	.dc.b	$2D,$8C,$CF,$28,$EB,$18,$3F,$C8,$57,$E2,$71,$1F,$56,$FC,$51,$AB
	.dc.b	$B3,$98,$DB,$F6,$23,$2F,$DA,$9B,$5D,$63,$C8,$78,$8B,$7E,$52,$5F
	.dc.b	$92,$B8,$DF,$26,$C7,$82,$D1,$8B,$E7,$3C,$4D,$57,$16,$9C,$67,$85
	.dc.b	$F8,$2B,$D8,$72,$3F,$2D,$5C,$5C,$1E,$FA,$E4,$74,$73,$7E,$48,$C5
	.dc.b	$75,$9F,$89,$18,$3E,$61,$F8,$3B,$F1,$B8,$8F,$8D,$3C,$6B,$E5,$36
	.dc.b	$B7,$3E,$6A,$E4,$78,$DF,$D3,$DF,$F3,$6B,$FA,$51,$89,$FD,$84,$62
	.dc.b	$3C,$06,$4F,$D8,$CF,$13,$F7,$DF,$37,$F7,$09,$5F,$4D,$B3,$F0,$9B
	.dc.b	$87,$A5,$B3,$ED,$8F,$E3,$71,$89,$F5,$1F,$C8,$9C,$3F,$3C,$C4,$F7
	.dc.b	$23,$DD,$B3,$DD,$A7,$73,$F3,$F3,$D2,$FE,$5C,$60,$DD,$78,$AB,$CB
	.dc.b	$18,$8F,$14,$61,$F0,$11,$F0,$36,$3C,$0F,$F1,$FA,$F0,$5F,$CA,$9E
	.dc.b	$97,$E2,$AD,$46,$27,$51,$CC,$F2,$47,$9D,$C1,$E4,$71,$7E,$22,$3E
	.dc.b	$28,$FE,$5B,$7E,$28,$FE,$24,$FC,$EE,$3F,$CC,$4F,$F6,$4B,$FE,$D3
	.dc.b	$8F,$F6,$A6,$DF,$F6,$F6,$CE,$E3,$FB,$95,$FF,$70,$3F,$B8,$FC,$DD
	.dc.b	$8C,$92,$7F,$B8,$C6,$5F,$EC,$3E,$3F,$F4,$B3,$FA,$67,$93,$6B,$F9
	.dc.b	$44,$7E,$6C,$8F,$E6,$9B,$3F,$DA,$BE,$53,$83,$67,$FC,$0D,$F3,$7F
	.dc.b	$E0,$4F,$9B,$FF,$02,$B6,$32,$7F,$82,$E3,$2D,$3F,$C1,$D8,$7F,$83
	.dc.b	$0F,$F0,$6E,$16,$32,$4B,$FF,$06,$C6,$5F,$F7,$E7,$CD,$BE,$4D,$5F
	.dc.b	$E4,$88,$C3,$4C,$CF,$09,$BC,$18,$85,$70,$7F,$B6,$B5,$FF,$49,$B5
	.dc.b	$FF,$80,$0D,$47,$29,$EB,$5A,$F9,$3B,$3A,$AE,$D7,$E1,$53,$A9,$78
	.dc.b	$1F,$E8,$4A,$F3,$2F,$37,$4E,$70,$F0,$D7,$09,$CF,$97,$A6,$CF,$F9
	.dc.b	$B2,$B9,$37,$E7,$79,$9D,$3E,$06,$C6,$49,$83,$1F,$1C,$EA,$31,$3E
	.dc.b	$2B,$FF,$73,$D7,$A3,$E3,$BC,$FE,$CF,$A1,$95,$FE,$17,$AF,$F7,$4D
	.dc.b	$FC,$A6,$BF,$0A,$7F,$CC,$B1,$D2,$7B,$79,$9C,$AB,$C2,$71,$7E,$57
	.dc.b	$EA,$CF,$9A,$BF,$05,$86,$DF,$FA,$3A,$74,$D9,$D3,$4D,$7C,$07,$43
	.dc.b	$AF,$C7,$D8,$C9,$31,$F1,$8B,$4E,$E3,$3F,$3F,$FE,$EF,$6F,$B5,$E8
	.dc.b	$F9,$0F,$C1,$4E,$78,$3F,$2C,$EE,$BF,$DC,$AD,$7F,$02,$D7,$F2,$6D
	.dc.b	$4F,$3C,$7F,$0F,$8B,$E0,$BB,$E3,$F2,$1F,$C9,$7C,$EB,$E6,$FF,$D2
	.dc.b	$AD,$BF,$F2,$73,$71,$FF,$29,$B5,$FF,$7C,$FC,$EB,$E6,$7E,$94,$7F
	.dc.b	$96,$23,$1C,$3F,$E5,$75,$F3,$5F,$E5,$89,$D5,$70,$7F,$3A,$3E,$81
	.dc.b	$A7,$F9,$7E,$30,$D8,$CC,$7F,$98,$DA,$FF,$9A,$3E,$79,$F3,$7C,$5B
	.dc.b	$1F,$E6,$96,$BB,$8D,$74,$48,$D3,$5F,$F3,$9F,$E0,$EF,$95,$F2,$36
	.dc.b	$3F,$CE,$8D,$77,$1E,$9F,$31,$F3,$F4,$93,$FE,$7B,$8C,$B7,$E7,$F8
	.dc.b	$11,$C1,$FE,$81,$E5,$7C,$A7,$C5,$6C,$66,$30,$7C,$9E,$EF,$E5,$51
	.dc.b	$FB,$3F,$1B,$04,$7F,$6D,$E2,$EC,$8E,$0F,$07,$CD,$F7,$23,$83,$F7
	.dc.b	$DC,$7D,$C8,$E0,$FE,$05,$F3,$FE,$32,$38,$3D,$9F,$39,$EB,$23,$83
	.dc.b	$F8,$57,$CC,$FA,$C8,$E0,$FE,$1D,$F2,$3E,$F2,$38,$3E,$47,$0F,$C7
	.dc.b	$23,$83,$F8,$B7,$3F,$C7,$23,$83,$F8,$D7,$CE,$FC,$B2,$38,$3E,$A7
	.dc.b	$C5,$FD,$32,$38,$3F,$91,$7C,$BF,$D3,$23,$83,$EE,$7C,$5F,$DB,$23
	.dc.b	$83,$FD,$81,$F3,$DF,$79,$FE,$23,$F3,$FF,$B9,$7F,$AA,$F8,$DF,$D7
	.dc.b	$E3,$11,$FC,$5A,$78,$4F,$F6,$35,$C3,$FD,$32,$31,$CB,$7C,$3F,$E9
	.dc.b	$F5,$C1,$B5,$9B,$FE,$C7,$EF,$BE,$0F,$10,$F8,$D7,$7E,$1B,$1E,$6C
	.dc.b	$3F,$AD,$DF,$18,$C1,$D3,$E1,$39,$C6,$63,$F9,$7B,$74,$FE,$F8,$A7
	.dc.b	$C3,$F3,$FF,$CC,$AB,$99,$E3,$E2,$F1,$1A,$F3,$77,$E6,$D9,$DD,$71
	.dc.b	$7F,$9E,$A8,$EB,$E6,$70,$9D,$C6,$1E,$3C,$3E,$FF,$56,$46,$0E,$BC
	.dc.b	$3E,$69,$91,$83,$C3,$D7,$FD,$C5,$FF,$80,$38,$A3,$11,$FC,$9A,$31
	.dc.b	$B7,$F2,$88,$FF,$0A,$C6,$1E,$3D,$2F,$9F,$D4,$27,$F8,$D4,$61,$B1
	.dc.b	$C2,$3F,$98,$2F,$91,$A4,$7F,$9A,$3F,$CD,$7E,$0B,$FC,$39,$06,$BC
	.dc.b	$47,$CF,$B8,$FF,$9F,$CF,$0E,$79,$CB,$F6,$2F,$93,$63,$AF,$2C,$8F
	.dc.b	$BB,$CC,$EF,$C5,$FF,$74,$1F,$11,$E4,$33,$C1,$FE,$43,$8C,$DF,$7F
	.dc.b	$88,$F4,$9B,$70,$7E,$2A,$7F,$B5,$37,$9F,$FC,$75,$8B,$E7,$FF,$C7
	.dc.b	$95,$AC,$3F,$C0,$AF,$F3,$6B,$E5,$CB,$AC,$36,$FF,$90,$FE,$7F,$DF
	.dc.b	$7F,$E0,$C6,$DF,$F9,$67,$17,$B7,$BE,$0E,$1C,$13,$3F,$CF,$E1,$C1
	.dc.b	$FF,$23,$BF,$B5,$7F,$47,$F3,$EF,$91,$FF,$29,$2F,$8F,$B1,$86,$4F
	.dc.b	$E2,$27,$4D,$8F,$95,$A7,$F9,$92,$72,$FE,$4F,$C4,$7B,$7A,$84,$62
	.dc.b	$FF,$26,$D8,$E1,$3B,$F9,$F7,$C6,$52,$5F,$E6,$A3,$2D,$F4,$78,$11
	.dc.b	$C1,$FA,$0F,$9F,$ED,$87,$F8,$39,$AF,$F9,$B1,$AF,$F9,$F9,$AB,$E7
	.dc.b	$FE,$E3,$FE,$16,$6B,$FE,$74,$6B,$FE,$86,$6A,$FA,$6F,$3B,$FB,$76
	.dc.b	$BF,$E2,$06,$BF,$E7,$E6,$BF,$E9,$0F,$97,$7C,$27,$CB,$6E,$1F,$BC
	.dc.b	$6C,$FF,$03,$6B,$FE,$2C,$FA,$07,$C2,$FD,$36,$D7,$F0,$46,$CF,$F0
	.dc.b	$B6,$BF,$E3,$4F,$A6,$FC,$AB,$79,$BE,$E4,$62,$3F,$D2,$AD,$7F,$D4
	.dc.b	$FF,$42,$FA,$1F,$C1,$37,$0F,$E2,$6D,$9F,$E3,$4D,$7F,$C8,$7F,$43
	.dc.b	$FE,$5D,$27,$AB,$F8,$B7,$FC,$C1,$F4,$CF,$57,$F1,$CD,$C3,$F8,$FB
	.dc.b	$6F,$F9,$33,$5F,$F2,$7F,$D1,$3E,$3F,$F2,$CD,$AF,$E4,$ED,$9F,$E5
	.dc.b	$CD,$7F,$CA,$FF,$44,$F9,$1F,$CF,$36,$BF,$97,$B6,$7F,$9B,$35,$FF
	.dc.b	$2F,$FD,$37,$FA,$D9,$FE,$74,$D7,$3F,$23,$FA,$98,$C4,$7F,$B0,$A3
	.dc.b	$F5,$EF,$F9,$A1,$AE,$7B,$FF,$61,$18,$8F,$F6,$CF,$D1,$BE,$7F,$F6
	.dc.b	$8D,$C7,$FA,$2B,$67,$FA,$63,$5F,$F3,$AE,$5F,$05,$FF,$69,$B5,$FF
	.dc.b	$72,$B5,$F8,$9A,$7F,$01,$F9,$EF,$D1,$93,$CA,$C1,$9F,$12,$63,$FD
	.dc.b	$5A,$7F,$8B,$36,$3F,$85,$7D,$37,$FA,$EB,$0F,$EC,$0D,$7F,$D1,$2D
	.dc.b	$7F,$DC,$ED,$7F,$DF,$34,$FA,$0F,$F7,$93,$6F,$FC,$0E,$FF,$1A,$F2
	.dc.b	$D1,$FE,$F0,$8D,$57,$FB,$01,$AF,$FA,$61,$AE,$EB,$E0,$12,$7F,$C2
	.dc.b	$7D,$49,$F0,$CF,$67,$FD,$97,$19,$9F,$F0,$CF,$C4,$FF,$39,$9F,$F6
	.dc.b	$7B,$70,$E1,$08,$FF,$11,$4F,$0F,$2B,$A5,$6A,$78,$4F,$86,$B8,$56
	.dc.b	$A3,$13,$CA,$7B,$CE,$5F,$44,$7F,$68,$7C,$7E,$7E,$5F,$17,$A1,$65
	.dc.b	$36,$30,$DB,$F6,$BC,$71,$DC,$7C,$68,$C4,$E5,$91,$FE,$1A,$9D,$DF
	.dc.b	$F8,$85,$AF,$F8,$32,$7F,$A8,$46,$B1,$F8,$EF,$F8,$46,$BB,$1F,$53
	.dc.b	$D3,$F4,$11,$FA,$3F,$41,$70,$FA,$33,$6F,$E6,$4D,$AF,$93,$E5,$7C
	.dc.b	$1F,$A0,$F8,$4D,$8F,$B6,$DC,$3E,$DB,$6F,$CB,$F3,$FF,$0C,$F8,$9F
	.dc.b	$41,$6F,$F1,$31,$EC,$23,$F0,$DB,$7F,$8A,$FA,$0E,$0D,$9F,$C5,$D7
	.dc.b	$D5,$AF,$AC,$D8,$C9,$F9,$08,$CB,$4F,$C9,$D7,$DD,$3E,$F7,$D0,$58
	.dc.b	$C9,$2F,$F2,$F1,$97,$F3,$BC,$FF,$7C,$FC,$07,$45,$AF,$D2,$C7,$D8
	.dc.b	$23,$F7,$DB,$3F,$A6,$FA,$0E,$0D,$9F,$D3,$FD,$07,$E4,$7E,$83,$F2
	.dc.b	$4D,$8C,$9F,$AC,$8C,$B4,$FD,$7E,$1F,$99,$3F,$35,$85,$8C,$92,$FF
	.dc.b	$67,$19,$5A,$FE,$D3,$E8,$1F,$23,$FA,$56,$DF,$FA,$51,$FE,$1A,$D8
	.dc.b	$D1,$1D,$16,$BA,$BF,$B9,$9E,$71,$83,$F8,$63,$8A,$F1,$4F,$17,$C9
	.dc.b	$56,$39,$16,$B8,$C5,$55,$E6,$7B,$18,$8D,$47,$79,$E6,$7A,$C9,$3C
	.dc.b	$23,$4F,$9A,$38,$CF,$26,$9C,$2B,$D2,$D3,$E0,$46,$DC,$4F,$A1,$AF
	.dc.b	$A1,$E6,$F8,$63,$E9,$46,$27,$E1,$BF,$38,$E3,$5E,$6A,$F7,$46,$5E
	.dc.b	$4E,$AB,$9B,$4E,$27,$53,$CE,$71,$AE,$F3,$C8,$8F,$B9,$B1,$C1,$A7
	.dc.b	$C7,$7B,$38,$3D,$CE,$23,$A9,$CC,$AF,$36,$C7,$33,$95,$7B,$C9,$3E
	.dc.b	$98,$F9,$87,$D9,$7E,$5B,$FD,$24,$F8,$A7,$FB,$46,$3F,$26,$E5,$AF
	.dc.b	$AC,$C5,$7F,$A0,$E3,$31,$D8,$CC,$FE,$97,$E4,$7F,$1A,$47,$EA,$B6
	.dc.b	$3D,$2D,$BF,$6C,$7E,$62,$31,$3F,$04,$FD,$43,$88,$F6,$CF,$ED,$E3
	.dc.b	$11,$EC,$8C,$4F,$80,$91,$E0,$6C,$78,$29,$5C,$9F,$98,$DB,$FE,$04
	.dc.b	$DA,$F7,$1C,$8F,$35,$FE,$EA,$34,$FE,$A5,$B7,$E4,$8F,$11,$1F,$14
	.dc.b	$6A,$7D,$E7,$59,$F7,$B8,$78,$3E,$07,$E4,$3B,$7E,$13,$B9,$EA,$E6
	.dc.b	$FD,$AD,$BF,$6B,$C7,$E4,$9F,$11,$FE,$05,$7F,$CB,$D7,$FE,$18,$73
	.dc.b	$3F,$E2,$38,$C7,$87,$BA,$E0,$D9,$D1,$FE,$19,$AE,$6D,$FA,$0E,$C7
	.dc.b	$7F,$76,$C7,$0C,$FF,$C6,$14,$AC,$C7,$F1,$C8,$FF,$76,$CE,$E1,$1B
	.dc.b	$8C,$2F,$8D,$F6,$1E,$DF,$56,$CE,$A3,$FD,$C1,$3B,$8F,$D4,$57,$8A
	.dc.b	$72,$48,$EB,$EA,$72,$9F,$23,$FE,$2D,$F9,$4F,$F3,$06,$BF,$E4,$05
	.dc.b	$F4,$19,$F9,$DD,$37,$0F,$54,$E9,$B3,$A7,$3E,$F7,$53,$B7,$C7,$D8
	.dc.b	$C9,$32,$FE,$83,$69,$D4,$7F,$44,$FA,$07,$AD,$FE,$43,$6D,$F8,$BE
	.dc.b	$43,$13,$23,$E1,$D2,$B6,$FC,$66,$BF,$C3,$DA,$FF,$96,$5D,$E9,$F0
	.dc.b	$FC,$EE,$93,$E6,$7E,$47,$D1,$FC,$9F,$86,$B8,$7C,$46,$D7,$B1,$B7
	.dc.b	$F3,$5F,$A1,$EF,$4A,$F9,$7F,$47,$6F,$E1,$CD,$BF,$A9,$36,$BF,$CD
	.dc.b	$DB,$7E,$29,$FA,$7C,$9F,$6F,$D2,$FB,$8C,$8F,$87,$3B,$C7,$F0,$ED
	.dc.b	$7F,$98,$35,$EB,$3F,$85,$F5,$39,$47,$D6,$7F,$11,$EA,$FE,$27,$D2
	.dc.b	$B8,$7E,$11,$B5,$F7,$1B,$9F,$E3,$5F,$F3,$E7,$EA,$FF,$E9,$39,$FF
	.dc.b	$00,$B6,$7F,$D2,$2B,$CF,$DC,$CB,$FD,$54,$ED,$FF,$48,$3B,$73,$1D
	.dc.b	$BD,$39,$1C,$3E,$32,$CF,$82,$7F,$D3,$EF,$26,$BF,$EA,$BA,$FF,$53
	.dc.b	$3F,$D5,$5A,$EE,$BC,$7E,$9F,$4C,$7D,$2F,$FA,$67,$E8,$5F,$36,$EC
	.dc.b	$FC,$AC,$61,$BF,$43,$EF,$C7,$F7,$73,$97,$F7,$BF,$43,$FA,$27,$F3
	.dc.b	$AD,$73,$D7,$FA,$05,$19,$7F,$B6,$53,$97,$4D,$8F,$D3,$B5,$FE,$16
	.dc.b	$D7,$FB,$A7,$D0,$FF,$15,$7F,$5A,$D7,$F8,$8B,$5D,$F8,$5E,$3F,$A0
	.dc.b	$7C,$BD,$24,$FE,$DA,$32,$FE,$E3,$EF,$5F,$49,$EB,$6C,$66,$30,$7B
	.dc.b	$FD,$D7,$C7,$52,$4F,$F0,$28,$CB,$FA,$2F,$93,$F8,$DD,$C6,$92,$3F
	.dc.b	$84,$F3,$FF,$B3,$9B,$EF,$FC,$D8,$C4,$7F,$30,$6B,$FE,$19,$EF,$FE
	.dc.b	$EE,$FF,$12,$6B,$FC,$D9,$AE,$FD,$2F,$1F,$CD,$C8,$FD,$BB,$5E,$DF
	.dc.b	$01,$E1,$23,$E6,$7F,$01,$4F,$A1,$A4,$9F,$13,$F0,$3E,$72,$47,$F0
	.dc.b	$46,$BE,$BF,$82,$F7,$11,$F8,$6F,$6F,$A1,$91,$ED,$6B,$FC,$33,$93
	.dc.b	$E6,$91,$FE,$22,$FC,$9F,$82,$91,$F3,$9A,$FC,$5F,$9E,$F5,$91,$F9
	.dc.b	$4F,$D8,$FA,$19,$1F,$C7,$1A,$FD,$3F,$85,$FA,$C4,$7E,$DB,$F8,$0F
	.dc.b	$8D,$91,$FC,$91,$AF,$DB,$F8,$6F,$BC,$47,$F0,$6F,$EB,$7E,$83,$F1
	.dc.b	$11,$FD,$EF,$E8,$7F,$2A,$FF,$A6,$9B,$E3,$BC,$84,$CA,$E2,$FF,$A8
	.dc.b	$17,$26,$A3,$1D,$16,$F9,$47,$18,$C5,$EE,$F8,$CF,$F8,$37,$95,$E4
	.dc.b	$EA,$3C,$3E,$8F,$ED,$F1,$E7,$08,$C6,$1E,$1B,$E3,$3C,$FC,$BE,$4A
	.dc.b	$32,$FF,$AE,$1B,$7D,$14,$F0,$FA,$1F,$EF,$95,$FE,$21,$C1,$C5,$D1
	.dc.b	$B1,$FE,$25,$9D,$BF,$EC,$45,$EB,$73,$F5,$E4,$EE,$3B,$7A,$DD,$D9
	.dc.b	$1C,$E3,$C5,$BF,$7F,$7E,$4F,$78,$F3,$70,$EA,$30,$78,$DF,$57,$5F
	.dc.b	$FB,$CA,$3F,$C5,$F9,$23,$11,$FE,$99,$8C,$6F,$FE,$12,$6C,$70,$E9
	.dc.b	$7D,$0E,$5B,$19,$8C,$1F,$CE,$D7,$D0,$D2,$3F,$CF,$DF,$E8,$1E,$9F
	.dc.b	$E1,$B0,$FE,$B6,$DE,$4F,$F2,$2C,$FE,$4B,$E8,$7E,$D5,$FE,$3E,$B4
	.dc.b	$FE,$47,$9A,$BF,$98,$6B,$FD,$B4,$FF,$7B,$9F,$8E,$BF,$F2,$E3,$F8
	.dc.b	$06,$F2,$F2,$E4,$7D,$0F,$28,$E7,$EA,$CA,$D9,$C9,$EF,$E3,$48,$C1
	.dc.b	$D9,$F2,$EA,$E5,$A7,$7F,$A1,$B5,$FE,$6F,$9F,$E7,$CD,$E5,$CE,$89
	.dc.b	$FD,$19,$A7,$13,$FB,$3F,$A1,$F8,$13,$FA,$86,$C7,$ED,$F5,$FF,$3E
	.dc.b	$D7,$03,$F7,$F3,$A6,$ED,$F3,$23,$E4,$46,$C3,$B6,$C6,$FC,$09,$7F
	.dc.b	$E8,$9F,$A1,$7B,$38,$BE,$58,$2F,$B7,$83,$9C,$FD,$0F,$DC,$FA,$1F
	.dc.b	$F3,$76,$5F,$E9,$36,$E1,$E3,$7F,$5F,$19,$EE,$F4,$7C,$7C,$8C,$1F
	.dc.b	$E9,$BC,$91,$ED,$F8,$E9,$39,$8B,$87,$BF,$E8,$3B,$E7,$FC,$49,$B5
	.dc.b	$FC,$85,$F0,$CE,$F1,$FC,$9F,$D4,$38,$E9,$F9,$6A,$FF,$0E,$5F,$FA
	.dc.b	$B6,$74,$FF,$AB,$FD,$DF,$C8,$B7,$A7,$F3,$B0,$FA,$4B,$1F,$A0,$F8
	.dc.b	$37,$C4,$53,$F4,$7F,$6D,$FE,$A6,$C3,$F9,$B4,$E9,$FF,$5E,$7D,$BF
	.dc.b	$FB,$23,$6F,$E7,$ED,$55,$FC,$BE,$77,$F1,$DF,$AE,$FB,$6C,$57,$E0
	.dc.b	$DB,$C3,$FC,$D9,$7C,$4E,$FB,$3F,$6B,$F4,$1F,$E9,$6C,$7F,$A4,$37
	.dc.b	$0F,$E9,$EF,$F2,$E8,$FF,$68,$39,$87,$0F,$FB,$45,$FC,$93,$71,$FE
	.dc.b	$8B,$F6,$7F,$52,$7F,$AC,$CE,$6F,$FA,$B7,$D9,$FF,$B4,$67,$F3,$4D
	.dc.b	$BF,$E9,$AF,$E9,$1B,$F0,$5F,$C3,$3E,$81,$F3,$34,$C7,$C1,$5C,$7F
	.dc.b	$DC,$DF,$32,$F9,$8F,$E2,$6D,$8F,$F7,$57,$C8,$7A,$E3,$1A,$7E,$AF
	.dc.b	$EC,$FF,$89,$37,$CC,$FE,$37,$1F,$B0,$D7,$FB,$0B,$A7,$FB,$1F,$2F
	.dc.b	$EB,$31,$FF,$7B,$36,$FF,$6D,$F6,$3F,$E8,$F7,$FB,$DC,$66,$BF,$B5
	.dc.b	$F0,$FE,$0E,$7E,$5F,$57,$F2,$AB,$FD,$2E,$1F,$95,$8F,$E5,$BE,$83
	.dc.b	$D6,$A6,$3C,$6B,$5F,$E0,$8C,$BF,$BB,$FC,$9F,$CB,$8F,$F6,$B7,$CC
	.dc.b	$7F,$B8,$F9,$BF,$DC,$8F,$F3,$8F,$AE,$F1,$5F,$F7,$E6,$D7,$F8,$37
	.dc.b	$EB,$9F,$03,$BD,$7F,$9F,$FD,$07,$AE,$BF,$C2,$4D,$E0,$FF,$0C,$AF
	.dc.b	$96,$DF,$A1,$FD,$1B,$F0,$7C,$5F,$FB,$C1,$B7,$FE,$1B,$3F,$C4,$2E
	.dc.b	$1C,$35,$FF,$10,$3D,$D3,$4E,$7A,$7F,$86,$1C,$57,$84,$F1,$5D,$BF
	.dc.b	$90,$7F,$8A,$FF,$07,$FB,$CD,$E8,$FF,$68,$FD,$17,$F5,$E9,$CE,$5F
	.dc.b	$E3,$67,$F1,$2F,$A5,$AF,$1A,$F6,$3B,$AF,$F6,$3A,$F8,$D7,$D1,$61
	.dc.b	$B5,$FE,$8C,$8C,$11,$F3,$CF,$87,$E2,$7F,$DC,$B3,$E5,$7F,$DC,$FE
	.dc.b	$97,$9E,$2D,$70,$A6,$3D,$9F,$E9,$8D,$7D,$53,$E6,$8C,$47,$C5,$8C
	.dc.b	$57,$C3,$3B,$57,$5A,$FC,$CD,$74,$F0,$3E,$41,$FE,$FA,$F4,$ED,$6E
	.dc.b	$3B,$11,$F8,$F3,$E6,$8F,$89,$E0,$49,$FF,$57,$B6,$7F,$D5,$8F,$F8
	.dc.b	$4B,$E7,$3F,$CD,$53,$FF,$00,$23,$0F,$F8,$5B,$47,$D1,$6F,$E7,$30
	.dc.b	$D9,$F8,$D7,$FE,$F2,$7C,$6D,$C3,$ED,$B5,$FC,$C3,$5F,$F3,$A3,$BE
	.dc.b	$4F,$F3,$4B,$FE,$AF,$9E,$B7,$E7,$F8,$FF,$47,$D4,$C9,$FA,$D9,$7F
	.dc.b	$16,$C3,$8C,$E7,$1F,$F0,$16,$28,$F6,$69,$9A,$FF,$3B,$DF,$F2,$8C
	.dc.b	$3D,$7E,$07,$C1,$F2,$3E,$33,$63,$27,$82,$32,$FC,$2E,$FF,$98,$7C
	.dc.b	$9F,$22,$C6,$4F,$8B,$1F,$9C,$CF,$F9,$F6,$9F,$12,$30,$7D,$5F,$A0
	.dc.b	$E3,$1E,$A7,$FC,$93,$F6,$2A,$A7,$D7,$FB,$0B,$87,$E6,$67,$4D,$9D
	.dc.b	$34,$F9,$3F,$31,$D3,$E3,$3F,$00,$D8,$C9,$CE,$32,$D3,$B7,$C9,$79
	.dc.b	$0F,$F2,$EF,$EB,$1F,$8C,$CF,$FC,$65,$8A,$75,$5C,$0F,$D7,$56,$F8
	.dc.b	$7F,$D5,$35,$B9,$D5,$F2,$BE,$B1,$CA,$30,$EA,$BF,$D6,$37,$CE,$3F
	.dc.b	$D8,$17,$CE,$B6,$4A,$DC,$F3,$7C,$37,$63,$FD,$91,$7C,$EB,$FD,$9F
	.dc.b	$7C,$EF,$75,$E1,$36,$71,$9D,$DD,$57,$CD,$5F,$3A,$F3,$D6,$EB,$53
	.dc.b	$9A,$E7,$5D,$DF,$F7,$0C,$EA,$BB,$56,$E7,$BD,$62,$7A,$34,$E5,$1A
	.dc.b	$6C,$F1,$39,$57,$58,$CD,$7D,$26,$9C,$A3,$55,$F3,$63,$D6,$7D,$E3
	.dc.b	$C0,$E4,$F4,$B8,$BF,$97,$1F,$E7,$99,$E7,$18,$24,$FC,$E3,$C3,$3C
	.dc.b	$CF,$51,$6F,$D5,$7E,$78,$C3,$F0,$A7,$8D,$74,$9F,$1C,$74,$7F,$B7
	.dc.b	$57,$E2,$5B,$5F,$D5,$23,$11,$EE,$3F,$AC,$B8,$7E,$B9,$88,$E0,$7E
	.dc.b	$EB,$D5,$FA,$E4,$AF,$A2,$D9,$F5,$36,$FC,$D1,$F8,$88,$C4,$F8,$67
	.dc.b	$F5,$31,$89,$F0,$C6,$27,$A9,$23,$AB,$63,$AD,$FF,$99,$23,$07,$53
	.dc.b	$F5,$75,$D6,$BF,$33,$7E,$B9,$DC,$78,$1D,$98,$76,$FF,$15,$9E,$55
	.dc.b	$F8,$F6,$D7,$3A,$F2,$92,$FC,$B3,$AB,$E4,$75,$BE,$4D,$BE,$55,$C2
	.dc.b	$30,$70,$BE,$8F,$B5,$FD,$ED,$F5,$7F,$8C,$B6,$7E,$54,$70,$3F,$63
	.dc.b	$F4,$BF,$C8,$3E,$63,$F9,$0B,$6F,$F8,$EB,$67,$F3,$8D,$AF,$E2,$FF
	.dc.b	$3D,$FC,$A3,$A7,$F9,$4B,$6F,$F9,$2C,$5B,$FD,$24,$DA,$FE,$3F,$D3
	.dc.b	$FC,$C0,$FE,$63,$EF,$5C,$3F,$96,$C7,$F8,$AB,$6F,$ED,$98,$7F,$8B
	.dc.b	$9A,$FF,$92,$DA,$FF,$20,$84,$FF,$88,$FE,$C9,$F1,$9D,$6B,$FD,$DD
	.dc.b	$18,$9F,$E7,$7F,$01,$DC,$8F,$F8,$5B,$0F,$F6,$3D,$78,$7E,$63,$FA
	.dc.b	$39,$FD,$22,$AD,$7F,$44,$EE,$FE,$97,$F4,$5F,$D3,$1B,$1F,$D2,$23
	.dc.b	$D8,$47,$B3,$EF,$F9,$6F,$EA,$67,$F5,$4F,$86,$B3,$FD,$42,$34,$F9
	.dc.b	$5B,$8F,$F9,$4B,$3F,$F0,$23,$7B,$BF,$CA,$71,$88,$FF,$39,$35,$FE
	.dc.b	$8F,$EA,$FC,$F2,$5F,$C1,$7E,$97,$81,$23,$E2,$36,$7E,$A7,$B3,$F5
	.dc.b	$C9,$3F,$25,$FB,$5F,$4D,$23,$E6,$36,$7E,$E7,$AD,$F7,$C9,$3F,$45
	.dc.b	$FC,$17,$85,$23,$EA,$36,$7F,$09,$E1,$7E,$1C,$93,$F6,$5F,$C5,$7A
	.dc.b	$72,$3E,$E3,$67,$FC,$F3,$83,$E1,$BF,$CF,$2D,$AF,$F2,$BB,$67,$73
	.dc.b	$AF,$A6,$95,$A6,$CF,$4F,$13,$99,$27,$9B,$DB,$E2,$64,$76,$6C,$FF
	.dc.b	$A3,$3F,$13,$67,$FE,$8C,$6D,$7F,$9A,$9B,$3B,$9F,$F4,$8F,$D3,$3C
	.dc.b	$0F,$64,$FF,$A5,$A3,$19,$FF,$9C,$97,$80,$F9,$4C,$5F,$0B,$FF,$5C
	.dc.b	$6C,$F9,$A4,$7F,$A3,$E1,$F2,$98,$87,$87,$FB,$27,$73,$A7,$85,$6E
	.dc.b	$75,$0F,$A1,$75,$67,$83,$A6,$0F,$95,$C5,$F4,$6B,$C6,$AF,$D2,$3C
	.dc.b	$6B,$E3,$4A,$FF,$53,$4E,$5F,$57,$99,$CB,$17,$CE,$53,$11,$DB,$E8
	.dc.b	$DE,$9E,$FE,$72,$61,$FE,$B1,$9C,$BF,$EB,$3F,$A4,$79,$FE,$67,$FC
	.dc.b	$99,$19,$F4,$31,$3E,$1F,$83,$F1,$44,$9F,$9D,$1E,$EF,$AC,$7B,$FE
	.dc.b	$7B,$95,$FE,$C1,$9C,$BF,$EC,$3E,$87,$A7,$F1,$97,$A7,$4C,$47,$C1
	.dc.b	$F9,$1F,$85,$F2,$B2,$FE,$F4,$7C,$EF,$BF,$FF,$6D,$C7,$E0,$7E,$3F
	.dc.b	$E4,$19,$F8,$2F,$A7,$F6,$FF,$2A,$7F,$DA,$8B,$97,$26,$B9,$BE,$87
	.dc.b	$D3,$7D,$1F,$A8,$97,$F8,$D8,$FB,$BF,$64,$F5,$7E,$B7,$D5,$CA,$FF
	.dc.b	$70,$4E,$5F,$F7,$17,$DE,$3D,$6F,$C2,$AF,$62,$98,$8F,$C0,$7C,$5F
	.dc.b	$E0,$7E,$32,$5F,$E7,$63,$F1,$BF,$74,$F8,$9F,$C7,$FE,$B3,$F9,$FA
	.dc.b	$FC,$FF,$95,$FA,$18,$C7,$99,$F9,$CB,$E1,$5F,$EF,$5F,$21,$8F,$E6
	.dc.b	$16,$F4,$C4,$76,$F3,$FB,$FD,$24,$BF,$F7,$F6,$5F,$B7,$7F,$E0,$07
	.dc.b	$DC,$27,$CB,$E1,$C9,$CC,$7E,$9B,$EF,$BE,$14,$7E,$EB,$5F,$D8,$C7
	.dc.b	$C9,$8F,$59,$FB,$E7,$11,$AE,$2F,$83,$F0,$DF,$0B,$E9,$E5,$F9,$23
	.dc.b	$B7,$BF,$A8,$C1,$DF,$8B,$F8,$36,$29,$FD,$CF,$D6,$7E,$F6,$B4,$7F
	.dc.b	$0E,$FA,$D7,$C6,$FE,$F9,$B1,$A1,$3C,$5F,$49,$8E,$FE,$BE,$AC,$8E
	.dc.b	$13,$F9,$37,$97,$D6,$FC,$97,$A7,$8C,$F8,$3F,$E1,$AD,$8D,$F1,$7F
	.dc.b	$10,$AF,$D7,$C7,$81,$E4,$D8,$D7,$1F,$F1,$6F,$AD,$7C,$0F,$4F,$0A
	.dc.b	$5F,$AE,$72,$DF,$6F,$85,$48,$C1,$BF,$3B,$A4,$C8,$C1,$CF,$6F,$4F
	.dc.b	$C3,$48,$C1,$DF,$83,$C9,$32,$30,$78,$FC,$8F,$E7,$6F,$E7,$C6,$5F
	.dc.b	$C9,$BA,$B1,$18,$9D,$BF,$A5,$9D,$3C,$57,$AC,$FA,$DE,$31,$86,$C6
	.dc.b	$67,$83,$5E,$FF,$3B,$E0,$7F,$A9,$B5,$34,$D7,$81,$FD,$4F,$1F,$E9
	.dc.b	$F3,$FD,$7F,$AF,$F1,$4F,$21,$F5,$AE,$5F,$DF,$5F,$1A,$FE,$01,$CB
	.dc.b	$6F,$FB,$03,$5F,$EC,$B3,$5F,$DA,$4F,$6E,$5C,$1B,$E6,$FF,$26,$E5
	.dc.b	$E4,$8E,$1D,$B2,$B8,$1E,$67,$A7,$74,$8C,$1E,$A7,$C5,$5F,$DD,$23
	.dc.b	$F4,$CD,$C3,$F4,$B8,$7F,$23,$9C,$F3,$70,$E3,$91,$83,$8B,$ED,$FA
	.dc.b	$AF,$91,$7F,$AE,$6E,$1F,$DE,$B0,$FD,$E5,$FE,$D6,$72,$FF,$81,$5F
	.dc.b	$87,$5C,$75,$E9,$AF,$57,$B7,$D6,$7D,$69,$FF,$07,$D2,$31,$08,$C5
	.dc.b	$FE,$49,$B1,$F9,$75,$F5,$9E,$93,$AC,$F9,$5F,$B1,$E5,$FE,$D2,$3F
	.dc.b	$36,$DA,$FD,$03,$5F,$E8,$0D,$7D,$9F,$35,$E4,$B3,$49,$5E,$5F,$2F
	.dc.b	$44,$70,$7E,$AB,$EB,$3A,$5F,$F8,$B9,$A9,$A6,$3E,$93,$ED,$47,$ED
	.dc.b	$5A,$FF,$55,$6B,$FE,$35,$6B,$FE,$3A,$F8,$9F,$F4,$EB,$FD,$6D,$AF
	.dc.b	$F8,$ED,$AE,$FE,$4F,$E8,$FB,$2F,$8F,$A4,$C7,$F8,$1C,$65,$FE,$0B
	.dc.b	$F5,$8F,$91,$FB,$2D,$8C,$C6,$0F,$BB,$F5,$6E,$5A,$49,$FE,$1F,$19
	.dc.b	$7F,$88,$FD,$5B,$E5,$7F,$0A,$D8,$CC,$60,$FC,$4E,$CF,$89,$A4,$9F
	.dc.b	$E3,$71,$97,$F8,$EF,$D5,$7F,$43,$6F,$99,$F9,$28,$C7,$07,$F2,$56
	.dc.b	$BF,$E0,$5F,$AA,$FF,$57,$3F,$E6,$86,$A6,$98,$FA,$0F,$F2,$6D,$FF
	.dc.b	$2C,$6C,$FF,$31,$6B,$FE,$11,$6B,$FE,$71,$FA,$AF,$F2,$2C,$FF,$34
	.dc.b	$6C,$7F,$39,$6B,$FE,$19,$6B,$FE,$79,$DD,$EE,$7E,$B9,$B5,$FC,$F9
	.dc.b	$B3,$FD,$0D,$AF,$E3,$F8,$FE,$69,$27,$E5,$BF,$B8,$F1,$DE,$6D,$24
	.dc.b	$FF,$3C,$7F,$7B,$F5,$3F,$C2,$AF,$FA,$83,$5F,$F1,$83,$5F,$F4,$83
	.dc.b	$5F,$F4,$AF,$D4,$FE,$45,$FE,$B0,$D7,$FC,$70,$D7,$FD,$30,$D7,$FD
	.dc.b	$3B,$F5,$1E,$C7,$FD,$38,$D7,$39,$FF,$12,$8C,$43,$EA,$3E,$2F,$8B
	.dc.b	$FE,$9C,$6E,$5F,$EA,$57,$FC,$82,$FF,$86,$5C,$CF,$FA,$B1,$E6,$4B
	.dc.b	$FE,$3E,$D9,$FF,$10,$7D,$37,$FA,$C9,$E5,$3F,$D4,$63,$15,$FE,$22
	.dc.b	$7F,$03,$7F,$EA,$E5,$E5,$6F,$CB,$E0,$CC,$3F,$BD,$CE,$9B,$1F,$CC
	.dc.b	$23,$17,$FE,$03,$5F,$4F,$92,$57,$27,$87,$8F,$E5,$3C,$CD,$8F,$5F
	.dc.b	$89,$E9,$3D,$5F,$11,$63,$27,$92,$3C,$44,$70,$7C,$4F,$17,$86,$3F
	.dc.b	$05,$F8,$9F,$27,$F1,$7E,$5A,$D7,$CA,$70,$FC,$07,$DC,$47,$FC,$3B
	.dc.b	$F3,$A7,$FA,$8B,$5C,$98,$33,$F1,$FF,$E0,$38,$FF,$57,$C7,$F9,$5D
	.dc.b	$E1,$87,$F9,$59,$CB,$7E,$1B,$81,$C7,$E6,$6D,$71,$8F,$09,$1F,$2B
	.dc.b	$67,$A7,$CC,$70,$6C,$F8,$7E,$83,$C9,$E2,$F9,$5B,$1E,$58,$B3,$E5
	.dc.b	$9B,$CB,$FE,$F6,$CB,$F7,$37,$EB,$EE,$F7,$76,$7B,$D9,$19,$3E,$04
	.dc.b	$63,$0F,$81,$F3,$BF,$14,$F8,$DE,$25,$AF,$8D,$1F,$38,$8F,$D2,$6C
	.dc.b	$FC,$9F,$03,$83,$67,$FD,$09,$F4,$7F,$DE,$73,$FB,$12,$11,$99,$C4
	.dc.b	$63,$3F,$F1,$D4,$E6,$38,$35,$E3,$F0,$1A,$9D,$BC,$3D,$FE,$3E,$FD
	.dc.b	$C3,$C6,$DB,$FF,$4A,$E0,$8C,$63,$E6,$9F,$2C,$78,$BE,$5A,$5F,$B3
	.dc.b	$E6,$2D,$72,$9B,$7C,$70,$56,$2F,$DB,$0A,$C5,$FB,$2A,$BF,$12,$2C
	.dc.b	$F2,$AF,$97,$18,$3E,$2D,$58,$C9,$E5,$2D,$79,$5B,$7E,$59,$E1,$39
	.dc.b	$3E,$1B,$88,$F5,$57,$6A,$E2,$7A,$9F,$2C,$EE,$30,$78,$6B,$A5,$72
	.dc.b	$8E,$31,$EF,$6B,$EF,$6D,$FB,$E3,$8C,$74,$69,$C4,$A7,$12,$C7,$19
	.dc.b	$F0,$46,$2F,$D4,$DA,$EF,$7E,$98,$FC,$6C,$EA,$B8,$92,$BC,$4F,$C9
	.dc.b	$7C,$F1,$F0,$9A,$74,$2C,$F9,$2B,$B3,$E7,$AE,$4F,$8C,$D4,$F0,$7F
	.dc.b	$46,$7B,$4B,$7F,$89,$8F,$F5,$FB,$63,$FD,$88,$D7,$77,$FE,$76,$8C
	.dc.b	$47,$CB,$33,$3F,$B6,$F3,$FF,$1E,$47,$E9,$B6,$3E,$6B,$6F,$E2,$47
	.dc.b	$ED,$23,$13,$F1,$8F,$CF,$B8,$8F,$9B,$1D,$CE,$A4,$7A,$B6,$7A,$C7
	.dc.b	$F0,$69,$E4,$FC,$99,$EA,$6C,$F0,$3E,$37,$0F,$80,$FD,$FC,$72,$AF
	.dc.b	$91,$5F,$B7,$8C,$5F,$F1,$06,$DF,$9E,$FC,$A4,$AF,$2D,$FF,$98,$63
	.dc.b	$13,$E2,$7D,$0F,$89,$F4,$1E,$53,$67,$96,$7C,$51,$B9,$F2,$BE,$E9
	.dc.b	$D9,$C8,$EA,$62,$36,$F5,$9F,$2D,$7A,$A3,$E0,$D7,$5F,$91,$FE,$5B
	.dc.b	$F2,$5F,$CB,$9B,$1F,$CE,$1B,$3F,$C9,$9B,$7F,$C9,$7E,$87,$FC,$3A
	.dc.b	$DF,$73,$FD,$E1,$18,$BF,$E7,$6F,$FB,$42,$FF,$A0,$35,$34,$E0,$EA
	.dc.b	$79,$FB,$F2,$BF,$A4,$5F,$87,$C5,$C5,$A3,$C9,$5F,$ED,$4C,$0D,$25
	.dc.b	$FE,$B2,$BC,$71,$E8,$F8,$0E,$77,$ED,$7D,$7E,$2F,$32,$5F,$1B,$F4
	.dc.b	$C7,$7F,$77,$E0,$9C,$27,$CE,$7C,$27,$E2,$F8,$7F,$1B,$D0,$B3,$AB
	.dc.b	$F8,$B1,$98,$ED,$9A,$3D,$84,$8C,$FC,$4F,$D0,$3E,$8F,$89,$63,$A5
	.dc.b	$FC,$E9,$F8,$84,$8E,$27,$AF,$F3,$67,$87,$FE,$FC,$6C,$66,$30,$7F
	.dc.b	$C0,$0F,$C2,$FF,$C4,$F7,$FC,$59,$AF,$F7,$A6,$BF,$DF,$DA,$BE,$41
	.dc.b	$BF,$F6,$F8,$FF,$00,$C6,$2F,$E7,$36,$3E,$C3,$5F,$E4,$AB,$EC,$39
	.dc.b	$47,$F8,$36,$31,$3C,$A7,$33,$E1,$7D,$5E,$F4,$9F,$B9,$87,$FB,$BA
	.dc.b	$8D,$FA,$67,$85,$C9,$B5,$F8,$36,$D7,$E3,$5A,$FC,$5F,$8C,$F9,$04
	.dc.b	$9E,$EF,$93,$2F,$F3,$CC,$7C,$0F,$3F,$04,$7E,$1F,$2E,$C8,$E0,$F8
	.dc.b	$FC,$BC,$C8,$E0,$F9,$7F,$57,$DC,$8E,$0F,$9F,$E8,$78,$C8,$E0,$FD
	.dc.b	$07,$D2,$FF,$AD,$73,$34,$9E,$3F,$CE,$FA,$57,$9E,$F8,$4F,$D4,$46
	.dc.b	$1B,$F4,$AF,$07,$35,$BC,$7F,$6D,$F8,$4B,$6F,$D8,$42,$30,$FE,$CD
	.dc.b	$AF,$08,$DF,$C1,$3E,$46,$92,$7F,$75,$19,$7F,$79,$F4,$CE,$FE,$ED
	.dc.b	$8C,$C6,$0F,$1F,$D3,$3D,$6A,$49,$FE,$0D,$19,$7B,$F9,$FE,$42,$38
	.dc.b	$3F,$86,$7D,$33,$D9,$F7,$B6,$33,$18,$3E,$1F,$D3,$7F,$92,$E3,$F8
	.dc.b	$A3,$5F,$E3,$CD,$7F,$C1,$6D,$7F,$CD,$F9,$BE,$9B,$E3,$92,$3E,$2B
	.dc.b	$F4,$3E,$99,$EF,$D2,$4F,$F2,$38,$CB,$FC,$97,$E4,$DF,$4D,$F6,$1B
	.dc.b	$19,$8C,$1F,$5B,$D3,$91,$F5,$1A,$FF,$37,$FA,$67,$77,$E0,$9B,$1F
	.dc.b	$CC,$DA,$EE,$3F,$0F,$F4,$BF,$E7,$D8,$FE,$70,$D7,$FA,$13,$5F,$F1
	.dc.b	$7B,$5F,$8B,$EA,$7E,$28,$8F,$E2,$1F,$C9,$FD,$2B,$C6,$A4,$9F,$E8
	.dc.b	$D1,$97,$FA,$47,$D2,$BE,$5B,$F3,$6D,$8C,$C6,$0F,$CC,$FD,$1C,$8F
	.dc.b	$CB,$B5,$FE,$B5,$F4,$AF,$5B,$F4,$AD,$8F,$EA,$CD,$77,$1F,$AA,$F4
	.dc.b	$95,$49,$3F,$D7,$63,$2F,$F3,$2F,$0B,$F6,$3E,$67,$ED,$E3,$04,$3F
	.dc.b	$B6,$7D,$23,$E6,$BF,$6E,$D8,$FE,$D0,$D7,$71,$FB,$CF,$A3,$7D,$25
	.dc.b	$24,$FF,$6F,$8C,$BF,$D1,$3D,$DF,$19,$1C,$1B,$FA,$4E,$84,$70,$7F
	.dc.b	$79,$FA,$37,$CF,$FF,$B4,$3D,$FF,$F6,$07,$D2,$3D,$EC,$7A,$9F,$ED
	.dc.b	$48,$C6,$1F,$DB,$30,$E2,$DF,$1F,$6F,$FB,$76,$32,$FF,$85,$6B,$FD
	.dc.b	$BB,$18,$C3,$FD,$9C,$AB,$8D,$7F,$64,$DA,$F7,$3E,$26,$9A,$F7,$BE
	.dc.b	$56,$3F,$ED,$77,$F6,$37,$DC,$FF,$77,$3A,$CB,$FD,$DA,$D8,$CF,$89
	.dc.b	$E0,$FA,$47,$C4,$F8,$3B,$E5,$7F,$82,$A7,$2D,$F4,$38,$54,$8C,$1B
	.dc.b	$F1,$BA,$4C,$8C,$1C,$F8,$FD,$BF,$17,$23,$07,$7E,$1F,$24,$C8,$C1
	.dc.b	$E3,$F8,$AF,$F2,$03,$FE,$3B,$CB,$FD,$F5,$BF,$F8,$D5,$B1,$99,$DB
	.dc.b	$FE,$AC,$87,$D1,$BE,$93,$83,$67,$A3,$5E,$0D,$7F,$C9,$5F,$33,$FD
	.dc.b	$51,$34,$8F,$F9,$1F,$BD,$C7,$FE,$34,$8F,$F3,$10,$38,$8F,$A4,$FF
	.dc.b	$12,$B9,$37,$CF,$C9,$FB,$C7,$17,$1A,$7F,$8A,$EE,$DF,$FB,$2A,$6D
	.dc.b	$73,$E4,$AF,$85,$BC,$EF,$85,$CC,$FF,$40,$BF,$47,$6E,$0C,$DD,$8F
	.dc.b	$67,$C5,$70,$31,$5E,$63,$FC,$E7,$1F,$E5,$4C,$DC,$DE,$26,$C7,$F9
	.dc.b	$63,$2F,$E0,$6E,$E7,$5C,$72,$30,$73,$7D,$7F,$49,$FE,$8D,$3F,$D1
	.dc.b	$D4,$EF,$CE,$FA,$DE,$61,$FE,$66,$BE,$11,$CB,$E0,$F9,$FB,$76,$B4
	.dc.b	$E1,$A7,$18,$C6,$BF,$53,$CA,$FD,$5C,$F5,$F9,$5F,$9F,$18,$9F,$B3
	.dc.b	$8A,$7D,$27,$AB,$CA,$B7,$E5,$70,$F9,$FE,$2E,$DF,$27,$ED,$FD,$27
	.dc.b	$5E,$2F,$C3,$4A,$3E,$6B,$F8,$87,$55,$F8,$D7,$33,$C3,$E8,$3E,$8D
	.dc.b	$6F,$E9,$7F,$0F,$18,$9E,$B4,$9F,$C5,$FC,$27,$E2,$70,$FF,$48,$AF
	.dc.b	$A8,$CF,$56,$9F,$C8,$FD,$33,$E2,$3F,$27,$F6,$0F,$A9,$CC,$61,$FC
	.dc.b	$BF,$D4,$3E,$6B,$77,$86,$E7,$FE,$A0,$FA,$8F,$F7,$A6,$FF,$09,$57
	.dc.b	$E8,$1C,$CF,$EB,$5C,$BF,$EA,$8F,$A8,$7C,$F7,$E9,$9F,$81,$1F,$B3
	.dc.b	$72,$EF,$D5,$F8,$BF,$09,$FA,$CF,$B0,$FF,$28,$4F,$FA,$CB,$95,$39
	.dc.b	$F3,$FF,$63,$F5,$3F,$CF,$6B,$FD,$6E,$FF,$A2,$B5,$FB,$2E,$7A,$3F
	.dc.b	$7E,$E6,$7F,$82,$7D,$4F,$F9,$5D,$FF,$5F,$AF,$AC,$FD,$EB,$9A,$FE
	.dc.b	$0E,$E6,$7F,$D8,$BD,$7F,$E4,$F7,$FD,$8F,$F5,$7F,$E1,$B7,$3F,$17
	.dc.b	$FC,$1A,$31,$7F,$C4,$3E,$AF,$F8,$23,$7C,$DF,$E1,$71,$8D,$3F,$22
	.dc.b	$A3,$F8,$7F,$33,$E9,$FF,$88,$FD,$87,$FA,$07,$3F,$F6,$BC,$E5,$FF
	.dc.b	$6C,$C6,$71,$FE,$3F,$C5,$F8,$79,$CF,$85,$F9,$4E,$CE,$6F,$F1,$CF
	.dc.b	$45,$EB,$DC,$B0,$DB,$FF,$72,$FD,$67,$F0,$56,$FC,$EF,$F2,$48,$C6
	.dc.b	$7F,$CB,$9F,$A0,$FF,$37,$FA,$CF,$13,$FC,$CB,$EC,$1F,$4F,$FC,$B2
	.dc.b	$73,$C7,$FB,$29,$CE,$3F,$EF,$4F,$AD,$F7,$61,$FC,$EB,$D1,$FE,$85
	.dc.b	$39,$E3,$FE,$6B,$18,$9F,$E7,$FF,$5B,$C3,$1F,$DE,$B6,$FF,$A3,$3F
	.dc.b	$E3,$C7,$FE,$00,$B9,$9F,$E9,$CD,$5F,$5B,$D6,$B9,$E1,$FD,$12,$31
	.dc.b	$7F,$D4,$5F,$15,$7F,$53,$38,$9F,$EE,$C7,$2E,$CC,$73,$49,$E1,$1F
	.dc.b	$D7,$E7,$F0,$91,$88,$F1,$CF,$FB,$8F,$CE,$FE,$AB,$F1,$3F,$C9,$6B
	.dc.b	$FB,$36,$3F,$C7,$30,$E3,$87,$F6,$2A,$D3,$FD,$92,$1E,$97,$AC,$F6
	.dc.b	$7D,$15,$C3,$99,$ED,$8F,$51,$1F,$24,$F2,$6E,$3E,$F2,$38,$25,$F8
	.dc.b	$79,$FE,$54,$FC,$B6,$4F,$47,$E4,$11,$EF,$F3,$7F,$2C,$8E,$0F,$8D
	.dc.b	$97,$F9,$FD,$F4,$DF,$F8,$26,$30,$FC,$19,$F9,$35,$FD,$FB,$EA,$DF
	.dc.b	$5B,$4C,$61,$4E,$76,$9D,$C7,$D5,$6C,$7D,$A6,$BF,$83,$6B,$FE,$17
	.dc.b	$5F,$1D,$FD,$4A,$30,$D7,$FC,$41,$1A,$9E,$D3,$99,$F1,$3E,$9F,$9D
	.dc.b	$F5,$36,$7C,$F5,$FE,$F1,$6C,$E9,$A6,$BC,$CE,$93,$FC,$E2,$FD,$51
	.dc.b	$82,$4F,$C2,$CD,$E2,$7C,$99,$FC,$9B,$F2,$FB,$65,$78,$BB,$6E,$1F
	.dc.b	$E3,$4C,$7F,$AB,$4F,$C9,$AF,$09,$1F,$99,$DB,$F5,$7B,$FE,$B3,$63
	.dc.b	$DA,$DB,$E8,$DE,$5F,$EF,$B8,$7F,$0E,$AF,$A1,$7C,$9F,$8A,$FF,$8F
	.dc.b	$BE,$AF,$FB,$FE,$1F,$87,$FB,$0C,$7C,$4F,$EB,$9B,$7F,$05,$B3,$FA
	.dc.b	$D7,$97,$C4,$49,$FC,$3D,$FF,$A6,$5B,$19,$AF,$F2,$AC,$A3,$C5,$F1
	.dc.b	$39,$CF,$FC,$2B,$3F,$DC,$AB,$FC,$C4,$D4,$BE,$B7,$D5,$F0,$53,$1F
	.dc.b	$0B,$ED,$8C,$57,$82,$2D,$F6,$9B,$5F,$E6,$BB,$EC,$48,$CD,$F8,$AE
	.dc.b	$BE,$28,$B3,$C2,$7C,$15,$E2,$25,$78,$5F,$8B,$5C,$27,$8B,$63,$77
	.dc.b	$E5,$69,$E5,$9D,$B8,$AF,$23,$5F,$23,$6F,$C8,$F8,$0C,$4F,$29,$F2
	.dc.b	$12,$BE,$8C,$7A,$A3,$13,$E1,$3A,$5F,$9E,$32,$7D,$67,$15,$CA,$7C
	.dc.b	$13,$A3,$EC,$D7,$86,$75,$19,$BF,$A1,$5E,$63,$B9,$63,$57,$DA,$3C
	.dc.b	$91,$F8,$C8,$C3,$F1,$0C,$47,$1A,$EE,$7C,$D8,$F0,$5F,$F0,$DB,$34
	.dc.b	$95,$FD,$4E,$32,$D8,$CC,$FF,$5A,$9C,$B6,$3F,$AB,$F9,$7F,$88,$E4
	.dc.b	$FF,$16,$4F,$C3,$6E,$1E,$4C,$3F,$D7,$91,$87,$DB,$1F,$E7,$88,$C4
	.dc.b	$FB,$63,$15,$D4,$95,$F8,$3A,$DD,$7D,$08,$FE,$35,$3A,$8F,$13,$6B
	.dc.b	$C5,$1F,$D2,$63,$13,$E2,$3C,$AE,$23,$C5,$18,$3C,$11,$98,$F0,$1F
	.dc.b	$DC,$DC,$61,$FB,$BB,$F2,$B6,$33,$87,$C8,$BE,$64,$BF,$3B,$70,$F3
	.dc.b	$C3,$0E,$75,$C6,$36,$49,$DC,$EA,$BE,$19,$CC,$F7,$4F,$C1,$70,$FB
	.dc.b	$8C,$1C,$DD,$BE,$1B,$F4,$C7,$48,$D1,$C4,$D3,$D1,$F8,$B5,$F6,$A3
	.dc.b	$F9,$4F,$D6,$FF,$87,$BA,$BF,$9B,$36,$32,$7F,$53,$8C,$B4,$FE,$AD
	.dc.b	$F2,$3F,$E2,$63,$FD,$F7,$F5,$BF,$E6,$7B,$FC,$54,$66,$76,$7F,$7E
	.dc.b	$9D,$3F,$92,$8E,$3F,$1F,$23,$5F,$13,$72,$61,$FE,$CC,$C1,$7F,$DD
	.dc.b	$DB,$7F,$E3,$B6,$D7,$F9,$01,$AF,$F9,$27,$4F,$F4,$93,$FE,$39,$F4
	.dc.b	$BB,$61,$FA,$17,$D7,$F0,$BD,$49,$87,$56,$D7,$F9,$07,$EB,$F7,$3F
	.dc.b	$DE,$53,$FE,$1C,$FD,$6A,$B4,$F8,$FD,$BA,$AF,$6C,$19,$3C,$1C,$98
	.dc.b	$8F,$F0,$B4,$66,$3F,$CC,$8D,$79,$07,$0F,$5C,$FE,$0B,$FE,$6B,$6B
	.dc.b	$C1,$AE,$F1,$FF,$12,$7D,$7F,$F8,$9E,$3F,$CD,$6D,$7F,$CE,$8D,$7C
	.dc.b	$2D,$7D,$7E,$17,$A0,$98,$7F,$9C,$70,$FF,$1C,$CF,$F9,$2F,$6F,$F1
	.dc.b	$96,$DF,$62,$3F,$8B,$C6,$0F,$83,$F4,$12,$F9,$36,$7F,$CF,$FF,$5C
	.dc.b	$F8,$5F,$F1,$E3,$6B,$FD,$00,$D9,$DC,$FC,$CF,$AA,$95,$E5,$6C,$F4
	.dc.b	$F9,$5E,$64,$9E,$6F,$A3,$CC,$91,$FE,$68,$6C,$FF,$A4,$B4,$7D,$57
	.dc.b	$DA,$25,$7C,$37,$E0,$F2,$48,$FF,$38,$B6,$B9,$79,$3F,$82,$24,$FC
	.dc.b	$B7,$E4,$FD,$14,$8F,$F3,$CB,$67,$C5,$E5,$7E,$28,$93,$F4,$DF,$F3
	.dc.b	$17,$D5,$F9,$BE,$2F,$FD,$4C,$DA,$FF,$55,$AD,$9E,$A6,$B4,$5F,$FA
	.dc.b	$B7,$EA,$FF,$D4,$78,$5F,$4B,$FD,$60,$A3,$2F,$FA,$D9,$AF,$E7,$38
	.dc.b	$B7,$C1,$F6,$E3,$F1,$11,$83,$9F,$C1,$BE,$0F,$F4,$3D,$FF,$AD,$A4
	.dc.b	$E5,$FF,$47,$B5,$FD,$69,$FA,$7F,$8C,$E3,$C3,$FA,$AF,$AB,$78,$98
	.dc.b	$AF,$DB,$46,$1F,$F4,$CB,$5E,$79,$BE,$C3,$84,$3E,$B1,$CB,$86,$C7
	.dc.b	$FA,$89,$AF,$91,$F8,$0F,$9B,$37,$99,$D1,$B1,$EC,$8E,$F8,$3D,$5F
	.dc.b	$0F,$D6,$3C,$BA,$4B,$FE,$2D,$DD,$E6,$FA,$CF,$F3,$04,$7F,$AD,$1A
	.dc.b	$FC,$66,$F1,$7C,$D7,$FC,$37,$1F,$33,$EB,$3C,$84,$7F,$91,$7C,$67
	.dc.b	$C2,$FA,$C7,$CE,$52,$4F,$F2,$7F,$98,$F8,$FF,$38,$F1,$E9,$27,$FD
	.dc.b	$8F,$1F,$91,$7E,$DF,$A7,$F0,$C8,$FF,$31,$F7,$7E,$8F,$D6,$3C,$6A
	.dc.b	$49,$FE,$6D,$EF,$7D,$6F,$AC,$FF,$44,$47,$FB,$51,$AF,$E1,$9B,$C3
	.dc.b	$F8,$D5,$F4,$1F,$4F,$EB,$25,$7E,$39,$FA,$7D,$5F,$82,$FA,$C7,$35
	.dc.b	$25,$7F,$47,$FA,$0F,$C3,$FD,$63,$8E,$92,$7F,$DC,$F0,$E0,$FC,$17
	.dc.b	$D5,$C9,$FC,$EB,$F8,$2F,$A7,$FC,$8F,$D6,$3E,$22,$92,$BF,$AB,$7C
	.dc.b	$47,$E5,$BD,$36,$14,$93,$FE,$F6,$87,$C6,$7E,$73,$D9,$FE,$03,$3F
	.dc.b	$ED,$9F,$4F,$06,$7B,$7F,$43,$F5,$8F,$A5,$A4,$AF,$F8,$03,$F5,$9F
	.dc.b	$11,$BE,$17,$EA,$20,$D3,$1C,$7E,$1F,$8F,$95,$FE,$E9,$78,$72,$C8
	.dc.b	$EA,$D7,$F7,$F6,$E3,$FD,$AF,$D5,$BE,$A3,$F6,$FF,$58,$F8,$8A,$6A
	.dc.b	$37,$E7,$5C,$7F,$BC,$78,$3F,$C3,$F0,$E5,$18,$F5,$3F,$B9,$C7,$ED
	.dc.b	$A3,$F8,$7F,$C0,$FC,$37,$47,$F8,$07,$C7,$F2,$38,$3F,$D0,$FA,$7F
	.dc.b	$0A,$9E,$11,$CB,$93,$1C,$72,$38,$4F,$EB,$9E,$5F,$58,$F8,$FF,$E2
	.dc.b	$78,$FC,$F9,$FE,$42,$DD,$3F,$94,$56,$E7,$8F,$B3,$FC,$6A,$31,$1F
	.dc.b	$CA,$1B,$3C,$27,$5E,$5E,$D9,$5E,$28,$E5,$C1,$E2,$F0,$64,$F9,$A3
	.dc.b	$B7,$0E,$A3,$07,$A9,$F1,$7C,$77,$F9,$12,$3F,$C3,$E7,$F8,$A7,$2F
	.dc.b	$F8,$01,$96,$23,$13,$B7,$73,$A7,$4F,$EE,$BE,$B3,$84,$23,$0E,$BC
	.dc.b	$94,$EA,$73,$3C,$3E,$6F,$FA,$EA,$69,$1E,$BD,$DF,$E4,$1C,$7F,$AB
	.dc.b	$47,$F9,$90,$F5,$CF,$F9,$0F,$EB,$3F,$CB,$F1,$FE,$3C,$AE,$1C,$FF
	.dc.b	$03,$92,$CF,$F9,$01,$AF,$F9,$2E,$6B,$FB,$19,$FC,$63,$FC,$C9,$BC
	.dc.b	$BC,$38,$FF,$C4,$99,$79,$23,$8F,$64,$AD,$9C,$9E,$BF,$1D,$23,$07
	.dc.b	$67,$C7,$4E,$4A,$77,$EF,$B5,$F9,$F9,$FD,$93,$79,$33,$9F,$F7,$69
	.dc.b	$CB,$FD,$E3,$EB,$3E,$42,$6E,$8E,$FF,$87,$1A,$3E,$14,$F1,$DF,$3B
	.dc.b	$E9,$DF,$C1,$70,$F8,$7B,$A7,$EC,$F0,$46,$63,$0E,$B0,$F2,$3F,$4B
	.dc.b	$C2,$FD,$84,$F1,$F9,$6C,$76,$F9,$19,$1D,$7C,$AB,$87,$B5,$F4,$9E
	.dc.b	$AF,$8F,$B3,$E3,$7E,$FF,$D6,$7C,$4E,$1F,$F1,$03,$67,$FD,$06,$FF
	.dc.b	$82,$1F,$E1,$8E,$6F,$FD,$20,$FF,$40,$8C,$F9,$7D,$3C,$AE,$AF,$E3
	.dc.b	$31,$6F,$4C,$6D,$6F,$F8,$C7,$D5,$BE,$63,$F2,$6F,$C2,$CF,$FC,$5E
	.dc.b	$E6,$77,$BE,$BE,$8F,$7E,$AD,$9E,$6F,$E7,$3F,$02,$76,$5F,$23,$0D
	.dc.b	$AF,$E5,$1F,$51,$FD,$A5,$BE,$47,$E9,$23,$1A,$7F,$90,$1F,$ED,$53
	.dc.b	$FC,$BF,$EA,$3F,$2C,$DE,$EF,$D5,$C6,$2B,$FC,$8E,$FC,$94,$B3,$DF
	.dc.b	$CC,$70,$6D,$FF,$3B,$FA,$77,$CC,$FE,$D9,$B1,$FC,$FF,$7F,$F6,$07
	.dc.b	$4F,$F9,$66,$73,$97,$F9,$5A,$7F,$B1,$D6,$FC,$2F,$DF,$C7,$B2,$FF
	.dc.b	$CC,$0E,$AB,$FC,$BB,$F8,$0F,$CF,$6F,$2F,$F0,$68,$FA,$95,$FE,$67
	.dc.b	$74,$FF,$9A,$7E,$9F,$F4,$ED,$F1,$FF,$86,$C6,$32,$FF,$67,$BF,$C2
	.dc.b	$F0,$FF,$35,$AE,$ED,$FC,$F7,$06,$D7,$F5,$DF,$A4,$FF,$51,$B7,$D9
	.dc.b	$FE,$33,$18,$C3,$FC,$EA,$FF,$B4,$72,$FF,$3E,$7D,$33,$E8,$7F,$8F
	.dc.b	$B7,$0F,$ED,$1F,$4B,$FE,$92,$7F,$B5,$46,$70,$FF,$3F,$CB,$E8,$BF
	.dc.b	$93,$B6,$B8,$F9,$98,$9F,$E5,$7F,$27,$EF,$BF,$F4,$0F,$97,$8E,$1C
	.dc.b	$BF,$CC,$3E,$8F,$E1,$6B,$FE,$91,$6E,$5F,$EF,$4F,$B3,$7D,$8E,$F0
	.dc.b	$93,$FD,$FE,$9E,$96,$32,$FF,$4A,$C6,$27,$53,$FD,$22,$42,$74,$FF
	.dc.b	$61,$6B,$FD,$99,$AF,$03,$87,$CF,$F3,$6D,$7F,$62,$94,$E5,$B1,$9F
	.dc.b	$27,$E0,$1F,$06,$6C,$7A,$FC,$EF,$88,$7C,$5F,$8B,$B1,$93,$B4,$75
	.dc.b	$23,$E1,$F9,$1F,$01,$1C,$13,$1F,$8B,$F3,$1F,$46,$7E,$93,$27,$D4
	.dc.b	$FA,$48,$FF,$7E,$ED,$3F,$05,$AE,$4C,$19,$FA,$1F,$ED,$B0,$6C,$61
	.dc.b	$CF,$54,$8F,$F1,$D4,$FE,$6F,$8B,$FD,$A8,$FE,$6E,$3E,$92,$9F,$11
	.dc.b	$23,$CC,$F9,$7E,$66,$47,$A5,$B9,$7F,$87,$EB,$FD,$A1,$3E,$0F,$96
	.dc.b	$F3,$D7,$A1,$91,$93,$CB,$18,$AF,$F7,$1D,$7F,$24,$7A,$79,$5F,$14
	.dc.b	$95,$F2,$5F,$87,$F2,$1F,$11,$B1,$F0,$3E,$2B,$E3,$9F,$23,$E1,$2C
	.dc.b	$64,$FA,$51,$F4,$48,$FF,$8C,$BE,$97,$F8,$9B,$F9,$77,$F8,$BB,$71
	.dc.b	$FF,$0F,$C6,$31,$FD,$1B,$C1,$BE,$96,$A7,$6C,$AF,$F3,$75,$71,$A3
	.dc.b	$C5,$B3,$DB,$E1,$3B,$FC,$25,$BE,$2D,$A7,$07,$E3,$1F,$1C,$66,$31
	.dc.b	$1D,$A3,$D3,$DF,$2B,$C7,$F5,$16,$B9,$C7,$C1,$9C,$5F,$2C,$97,$8A
	.dc.b	$F8,$DE,$77,$C4,$3F,$C8,$7F,$65,$FE,$EB,$8F,$7C,$65,$A7,$FA,$3E
	.dc.b	$35,$8F,$80,$F9,$73,$B7,$07,$C6,$7A,$FC,$AE,$71,$F2,$45,$BF,$E1
	.dc.b	$D7,$D2,$73,$1E,$0B,$E7,$8F,$F6,$98,$F1,$61,$C6,$7C,$31,$BA,$F9
	.dc.b	$37,$CE,$3D,$E6,$4B,$3E,$68,$D3,$C8,$EF,$87,$55,$1A,$7D,$93,$CD
	.dc.b	$E1,$5E,$0B,$E1,$87,$9E,$79,$D7,$A0,$E0,$F2,$32,$78,$E7,$8E,$1E
	.dc.b	$1C,$3C,$17,$CF,$0F,$8B,$7C,$1F,$06,$1D,$27,$4F,$BA,$33,$3B,$8C
	.dc.b	$11,$FE,$8B,$5D,$AF,$C3,$3E,$79,$E0,$7B,$9E,$0F,$C4,$C3,$6E,$8C
	.dc.b	$47,$8E,$7D,$53,$AB,$F4,$9E,$42,$CF,$4C,$3D,$37,$E4,$AE,$13,$E0
	.dc.b	$C3,$CC,$F5,$3C,$D7,$CA,$BC,$C4,$3C,$67,$96,$FE,$A3,$FE,$C4,$6B
	.dc.b	$F8,$43,$18,$7F,$5F,$8C,$C7,$E1,$4C,$D7,$F1,$19,$D1,$FC,$3B,$D1
	.dc.b	$FC,$09,$23,$EA,$36,$7C,$4D,$BF,$44,$7F,$0C,$8C,$4F,$C3,$6C,$7A
	.dc.b	$ED,$18,$8F,$C5,$57,$EF,$E3,$13,$F2,$A7,$A5,$78,$88,$F8,$9B,$5E
	.dc.b	$29,$EB,$18,$3C,$45,$9F,$15,$76,$BF,$81,$39,$9F,$1B,$FC,$CE,$74
	.dc.b	$FF,$2A,$6C,$7E,$F1,$B7,$E9,$9E,$57,$E9,$3C,$75,$E7,$24,$F9,$EB
	.dc.b	$8D,$77,$38,$DF,$92,$B7,$1E,$A6,$9E,$A7,$FC,$4B,$5F,$CC,$3E,$9F
	.dc.b	$FC,$4F,$18,$3F,$C4,$4F,$FB,$28,$CC,$FF,$1E,$70,$FF,$BB,$1B,$1C
	.dc.b	$63,$FC,$63,$1C,$7D,$6D,$1B,$F0,$6D,$6E,$95,$FE,$9A,$8C,$36,$3C
	.dc.b	$13,$C1,$F1,$78,$1E,$13,$F9,$DF,$CF,$5B,$E3,$1F,$E3,$9B,$FF,$5A
	.dc.b	$C7,$F8,$F6,$BD,$B3,$93,$5E,$DF,$AC,$E5,$1E,$F7,$DD,$EE,$73,$25
	.dc.b	$73,$9F,$7C,$61,$F8,$DE,$7F,$09,$F5,$BF,$1F,$E3,$24,$7B,$BC,$1B
	.dc.b	$5F,$36,$7D,$04,$9E,$D1,$F1,$A3,$FC,$A9,$F3,$9F,$E7,$98,$F6,$35
	.dc.b	$FF,$2D,$35,$FF,$31,$B5,$7D,$2F,$F8,$DE,$BA,$B6,$3D,$8D,$7E,$43
	.dc.b	$5F,$F3,$37,$C9,$3D,$EF,$F3,$33,$6B,$CE,$D9,$F8,$CD,$7E,$92,$FB
	.dc.b	$3E,$24,$9E,$2F,$F9,$C6,$DF,$43,$49,$3E,$87,$FC,$ED,$D0,$F0,$29
	.dc.b	$27,$E4,$C6,$5F,$7F,$CB,$F5,$23,$83,$E7,$FC,$CF,$FA,$9F,$3F,$BC
	.dc.b	$D7,$F1,$4D,$7F,$D1,$0D,$7F,$D2,$39,$FE,$41,$F8,$3C,$7F,$18,$95
	.dc.b	$F1,$B8,$FE,$49,$1C,$1F,$2B,$D0,$F9,$A4,$70,$7C,$EF,$37,$E8,$91
	.dc.b	$C1,$F8,$8F,$B3,$FC,$01,$3D,$8F,$F4,$E3,$F8,$3F,$A7,$7D,$33,$E9
	.dc.b	$BF,$1F,$18,$6F,$D3,$BE,$3F,$35,$BD,$BF,$45,$FA,$DD,$6D,$FE,$61
	.dc.b	$46,$5F,$D9,$35,$FF,$58,$2F,$B3,$E0,$D8,$FD,$0B,$5E,$F1,$AF,$36
	.dc.b	$46,$9A,$F9,$3C,$DF,$F5,$E7,$8B,$FE,$4E,$B4,$63,$7F,$0B,$6B,$FD
	.dc.b	$93,$2F,$99,$C4,$7A,$BE,$52,$C9,$A6,$3E,$2F,$3E,$84,$AF,$E3,$D4
	.dc.b	$8E,$3E,$87,$2F,$99,$E7,$F4,$92,$F7,$1D,$BD,$1E,$FE,$94,$9E,$71
	.dc.b	$E2,$F8,$2F,$1F,$D9,$49,$EF,$1E,$6F,$8A,$91,$E2,$7D,$3E,$97,$AB
	.dc.b	$D3,$93,$E6,$7D,$BF,$69,$EE,$F9,$39,$3E,$A7,$E0,$FC,$A7,$C4,$F8
	.dc.b	$0F,$E5,$9D,$1F,$E7,$A7,$F0,$D5,$FC,$F5,$AF,$F9,$99,$DF,$47,$FB
	.dc.b	$BD,$7C,$37,$3F,$85,$95,$C6,$3B,$7B,$1D,$FD,$99,$3D,$63,$C5,$E9
	.dc.b	$78,$FD,$29,$3E,$18,$F3,$7C,$27,$9F,$CC,$93,$E5,$8F,$57,$99,$EB
	.dc.b	$F3,$24,$FA,$63,$DD,$E6,$48,$F6,$3F,$07,$E1,$3E,$17,$B3,$27,$E0
	.dc.b	$3F,$17,$D9,$F8,$DF,$17,$27,$E2,$3F,$27,$8B,$F1,$BE,$8F,$C6,$E5
	.dc.b	$FF,$51,$B7,$97,$F9,$7F,$9B,$AE,$6F,$C9,$FD,$07,$C8,$CB,$FB,$53
	.dc.b	$5F,$F4,$9B,$70,$FF,$56,$FD,$03,$D5,$FC,$DA,$F9,$0A,$4B,$FE,$E9
	.dc.b	$0F,$98,$F2,$F4,$C9,$FE,$F1,$AA,$39,$FC,$D4,$9E,$2D,$7F,$D4,$B6
	.dc.b	$F8,$4F,$D5,$FC,$FB,$CE,$FD,$6F,$D0,$3E,$06,$9A,$8D,$F9,$F7,$1F
	.dc.b	$F0,$45,$F1,$6C,$7E,$DB,$BF,$E0,$72,$A6,$79,$3E,$17,$FC,$20,$D7
	.dc.b	$FD,$73,$3C,$1E,$0B,$97,$A7,$87,$27,$FC,$31,$39,$7F,$C3,$5F,$3F
	.dc.b	$FC,$31,$EB,$CB,$FC,$23,$0F,$A1,$1A,$F5,$3F,$85,$D7,$F0,$D8,$C1
	.dc.b	$0F,$F6,$5B,$96,$FA,$7C,$3D,$49,$1E,$08,$E7,$BF,$7F,$88,$93,$E4
	.dc.b	$8E,$FB,$EA,$30,$7A,$1F,$1F,$A9,$FE,$F6,$8F,$F0,$E6,$28,$C4,$7F
	.dc.b	$8A,$63,$1D,$7F,$E1,$46,$C7,$0F,$4D,$F4,$19,$6C,$66,$30,$7F,$45
	.dc.b	$5F,$41,$48,$FF,$49,$7F,$A5,$78,$0E,$5F,$F2,$44,$FF,$53,$6B,$C8
	.dc.b	$7D,$07,$F8,$55,$FC,$11,$B3,$FD,$EE,$7F,$10,$8E,$7C,$BD,$63,$35
	.dc.b	$73,$C4,$7E,$2E,$71,$5F,$E4,$BE,$6B,$3D,$9A,$B9,$3F,$C3,$33,$E8
	.dc.b	$9C,$CF,$EA,$63,$D7,$DB,$89,$CB,$AF,$36,$4E,$8F,$82,$F6,$C7,$FA
	.dc.b	$94,$7F,$9A,$DB,$67,$D3,$87,$EF,$E7,$5C,$9C,$78,$E4,$60,$E4,$FA
	.dc.b	$BE,$83,$E4,$5F,$C2,$6C,$FC,$CB,$77,$EF,$6E,$1B,$4A,$FF,$3F,$5F
	.dc.b	$0C,$FF,$68,$FD,$17,$CD,$1F,$5A,$3F,$C5,$18,$F7,$F2,$BC,$5E,$6F
	.dc.b	$8D,$95,$D1,$CB,$4F,$F4,$67,$87,$FD,$F9,$F0,$F9,$5F,$52,$31,$8F
	.dc.b	$80,$F1,$C7,$BC,$F8,$1E,$55,$AF,$4B,$87,$D7,$E5,$5A,$E4,$33,$47
	.dc.b	$E0,$BE,$83,$F8,$F5,$7F,$A7,$DB,$5C,$1F,$F0,$24,$67,$87,$87,$A3
	.dc.b	$23,$87,$A3,$6F,$BE,$5F,$1A,$7F,$3A,$D8,$FE,$40,$B5,$FC,$9B,$98
	.dc.b	$7D,$03,$DC,$FC,$BB,$F1,$38,$7F,$3E,$E7,$0F,$F5,$B3,$97,$F5,$6B
	.dc.b	$D7,$E3,$EB,$F2,$6C,$FF,$AF,$7E,$83,$FD,$75,$7F,$AA,$7F,$D0,$8D
	.dc.b	$EA,$FD,$3C,$62,$7F,$6F,$F4,$1F,$D4,$5F,$F6,$32,$F1,$7F,$5A,$E7
	.dc.b	$4F,$A2,$E6,$7F,$D9,$9F,$41,$FA,$2C,$7F,$72,$FD,$06,$F6,$FE,$DA
	.dc.b	$31,$3F,$C1,$96,$99,$F6,$B4,$D9,$FF,$6C,$7C,$FB,$E8,$3F,$7E,$D8
	.dc.b	$FF,$6E,$6F,$FC,$5D,$DC,$FF,$04,$8F,$D9,$69,$F8,$86,$E7,$FC,$39
	.dc.b	$F6,$37,$D4,$FE,$17,$F4,$1F,$4E,$BF,$1E,$D9,$FE,$2A,$FD,$58,$FE
	.dc.b	$C0,$E7,$0F,$E4,$3F,$41,$FA,$FA,$FE,$36,$FF,$5D,$8F,$F7,$93,$97
	.dc.b	$7E,$2F,$F1,$A8,$FC,$DC,$6B,$D9,$C7,$6E,$5F,$E3,$FF,$41,$FE,$8A
	.dc.b	$9F,$F7,$E2,$F9,$AC,$FC,$46,$9F,$E4,$BF,$40,$F8,$DA,$7F,$28,$FA
	.dc.b	$0F,$F5,$86,$BF,$CC,$27,$58,$FF,$7B,$8F,$E6,$72,$F8,$7E,$07,$1F
	.dc.b	$32,$D6,$CF,$E8,$3F,$41,$ED,$E2,$FF,$02,$35,$3F,$0D,$FD,$DB,$9F
	.dc.b	$9F,$C4,$7F,$4A,$0E,$1F,$E9,$AD,$7F,$C2,$0D,$77,$E7,$FF,$4A,$76
	.dc.b	$D7,$6D,$77,$5F,$EE,$AF,$2B,$FA,$5F,$C0,$1F,$E9,$B7,$FD,$26,$BF
	.dc.b	$88,$CE,$6B,$E0,$47,$8B,$CD,$D9,$C3,$E1,$6C,$EA,$7D,$73,$E1,$BF
	.dc.b	$F6,$C4,$7B,$DA,$FB,$A3,$58,$4A,$F8,$93,$F0,$23,$0E,$2B,$C8,$FF
	.dc.b	$26,$5E,$3F,$AA,$7D,$6F,$B3,$E0,$FD,$BF,$2F,$70,$F3,$5A,$33,$1E
	.dc.b	$0C,$3E,$9C,$7A,$49,$19,$CF,$BB,$FE,$58,$C3,$8D,$7C,$87,$FB,$BF
	.dc.b	$92,$F9,$FA,$62,$A9,$F1,$E3,$4F,$85,$B5,$F0,$9A,$FE,$1D,$AF,$F8
	.dc.b	$49,$FD,$24,$71,$E6,$FC,$04,$62,$79,$46,$0F,$2F,$D5,$79,$9B,$5E
	.dc.b	$3F,$9F,$F4,$FD,$07,$A9,$B1,$E7,$8E,$A4,$7D,$0D,$BF,$C4,$F6,$F0
	.dc.b	$6C,$FB,$FE,$83,$E2,$9F,$1B,$CB,$B1,$92,$4F,$F6,$2C,$FF,$99,$36
	.dc.b	$BE,$6F,$CC,$FC,$EE,$DB,$7F,$2D,$B5,$C5,$BD,$9F,$E3,$0C,$3F,$80
	.dc.b	$CF,$F6,$EF,$9F,$FE,$3C,$9A,$7F,$1F,$7F,$D0,$0E,$3E,$0F,$F5,$8F
	.dc.b	$F8,$02,$3F,$74,$D7,$FC,$8A,$D7,$8C,$7E,$C3,$C2,$FF,$1D,$CF,$83
	.dc.b	$91,$18,$F8,$19,$9F,$EA,$EB,$FB,$DE,$1E,$79,$CB,$67,$5F,$05,$8C
	.dc.b	$FF,$C9,$B7,$FE,$47,$9E,$5D,$FE,$0F,$03,$C2,$C8,$C9,$DE,$31,$87
	.dc.b	$C5,$F3,$7C,$E7,$A3,$CD,$B5,$E5,$8B,$3F,$2E,$6E,$3E,$29,$B5,$F1
	.dc.b	$26,$D7,$CC,$9A,$FD,$18,$B8,$A6,$C6,$1B,$5E,$12,$57,$9D,$F9,$13
	.dc.b	$B7,$37,$F1,$23,$13,$B3,$D1,$3A,$3E,$CD,$76,$8D,$57,$AE,$75,$1F
	.dc.b	$84,$AE,$73,$98,$E6,$70,$9E,$67,$07,$DC,$DB,$F7,$36,$B9,$BC,$27
	.dc.b	$9D,$75,$24,$65,$F6,$3C,$EF,$D1,$1A,$9F,$9D,$3D,$4D,$15,$F3,$1A
	.dc.b	$29,$E6,$2D,$F9,$9B,$7E,$67,$85,$78,$9D,$DF,$8A,$FC,$C4,$8D,$BF
	.dc.b	$83,$3C,$6F,$F4,$0B,$EC,$FE,$B2,$7F,$5E,$49,$FE,$6D,$3A,$9F,$8C
	.dc.b	$6E,$33,$3F,$D7,$BA,$FF,$1B,$B7,$F3,$F9,$FA,$4D,$AF,$7B,$70,$F9
	.dc.b	$10,$8C,$4F,$A6,$31,$5D,$09,$1D,$1B,$1D,$1B,$5D,$2B,$9C,$FC,$20
	.dc.b	$74,$AE,$44,$79,$36,$79,$57,$48,$F2,$57,$4A,$FE,$35,$7F,$06,$78
	.dc.b	$1A,$AE,$13,$E1,$3C,$07,$62,$5F,$69,$E3,$1B,$BE,$D8,$7E,$0E,$BF
	.dc.b	$DB,$71,$8C,$3C,$15,$CA,$FC,$07,$27,$CA,$FF,$81,$E7,$83,$70,$F2
	.dc.b	$CE,$8F,$F0,$9B,$8C,$3C,$13,$FE,$F2,$AE,$35,$A8,$ED,$3A,$8F,$9B
	.dc.b	$3A,$C3,$CF,$7C,$F0,$F3,$C7,$FB,$D3,$E7,$BF,$D9,$97,$F8,$1F,$88
	.dc.b	$FF,$78,$C6,$30,$46,$3D,$EF,$F1,$5B,$73,$F6,$3C,$3D,$5E,$3F,$0F
	.dc.b	$67,$FC,$4D,$87,$FA,$57,$1F,$C0,$B5,$34,$FF,$4F,$C7,$E1,$A3,$33
	.dc.b	$E4,$F6,$24,$F4,$F1,$6D,$7F,$B2,$AF,$FD,$71,$3F,$E1,$D6,$C7,$F8
	.dc.b	$B9,$AF,$F9,$4D,$AF,$77,$7A,$7D,$EF,$4B,$AB,$6B,$D7,$DB,$FE,$4F
	.dc.b	$AF,$F3,$ED,$F5,$8C,$1F,$E5,$3F,$9E,$FF,$4D,$47,$FB,$79,$B7,$F4
	.dc.b	$65,$7F,$E4,$B9,$CB,$E3,$FA,$DE,$C4,$7D,$8F,$F9,$7F,$EB,$7F,$C4
	.dc.b	$18,$7F,$9B,$5A,$FC,$96,$BF,$B0,$6A,$FA,$07,$D1,$7F,$9A,$5B,$7F
	.dc.b	$E4,$56,$CF,$F9,$59,$AF,$F9,$EF,$8D,$C1,$FE,$71,$6D,$7F,$92,$63
	.dc.b	$07,$C3,$B9,$1F,$E5,$1C,$B9,$DF,$DF,$6B,$FC,$1B,$07,$D0,$F9,$49
	.dc.b	$5E,$57,$E7,$7C,$EC,$8E,$8D,$AF,$47,$C7,$7B,$49,$3E,$D7,$D1,$E7
	.dc.b	$C8,$FF,$30,$36,$7F,$CE,$BA,$7F,$9E,$9F,$D6,$F1,$FB,$C9,$5F,$E6
	.dc.b	$7E,$4F,$84,$47,$07,$BF,$C9,$FC,$21,$1C,$1F,$E6,$FF,$94,$F9,$44
	.dc.b	$70,$7C,$7F,$2F,$F1,$84,$70,$7F,$9E,$FE,$79,$EB,$3E,$7B,$0D,$CF
	.dc.b	$FD,$1F,$2D,$DD,$BA,$BF,$F4,$27,$D6,$B6,$FF,$43,$60,$8C,$3A,$8C
	.dc.b	$C6,$23,$FD,$37,$F3,$CF,$9F,$E2,$D8,$FF,$48,$35,$DC,$76,$F5,$7F
	.dc.b	$D2,$D1,$FE,$95,$6B,$FE,$A6,$6B,$FE,$CB,$6B,$FC,$1D,$FE,$2E,$F1
	.dc.b	$F0,$7A,$11,$F0,$F9,$7F,$C3,$7B,$FF,$63,$1F,$D5,$30,$4E,$78,$3F
	.dc.b	$89,$46,$27,$FD,$80,$BE,$53,$51,$C7,$E1,$2A,$94,$61,$B1,$CB,$E2
	.dc.b	$B1,$E6,$48,$EA,$EF,$CD,$E1,$F2,$7C,$7E,$7E,$57,$8A,$3A,$7C,$FF
	.dc.b	$5F,$3E,$4F,$9A,$3C,$1E,$87,$87,$E7,$E4,$FA,$A3,$C9,$F0,$F2,$3D
	.dc.b	$AF,$9F,$D2,$F4,$7A,$12,$7E,$0B,$EB,$F9,$FF,$67,$C7,$C9,$F8,$AF
	.dc.b	$BF,$E3,$FE,$1F,$4F,$FC,$01,$E8,$F9,$F4,$8F,$C8,$BF,$EE,$B6,$A8
	.dc.b	$DC,$71,$E7,$FB,$DF,$27,$D3,$E2,$65,$F1,$8E,$FF,$09,$E0,$F5,$A4
	.dc.b	$F5,$8F,$1F,$A3,$E4,$F9,$F9,$3E,$18,$F3,$F9,$BE,$8F,$0E,$4F,$96
	.dc.b	$3D,$7E,$67,$B3,$E7,$E4,$FA,$63,$DF,$E6,$48,$F6,$3F,$0B,$E1,$BE
	.dc.b	$1F,$C2,$C9,$F8,$0F,$C6,$F6,$3E,$3F,$BB,$27,$E2,$3F,$2B,$8F,$F1
	.dc.b	$BE,$97,$C6,$E5,$FF,$09,$F9,$B8,$33,$CB,$F9,$2F,$9F,$73,$52,$65
	.dc.b	$FE,$32,$BF,$A7,$F3,$EF,$57,$F3,$2B,$E3,$E9,$2B,$87,$57,$F7,$59
	.dc.b	$7C,$BF,$93,$AA,$57,$F8,$7D,$E7,$F3,$12,$38,$9F,$E4,$2A,$46,$63
	.dc.b	$F0,$17,$D7,$E7,$FF,$57,$F3,$CF,$37,$F5,$BF,$3F,$FB,$EB,$D9,$FE
	.dc.b	$4C,$71,$B7,$E3,$AB,$84,$F8,$02,$F3,$BA,$DC,$9D,$4E,$5E,$1D,$FF
	.dc.b	$29,$E5,$E4,$32,$FD,$FB,$63,$7C,$5F,$C0,$A9,$C9,$62,$4F,$48,$FE
	.dc.b	$0F,$F3,$FF,$C3,$63,$D6,$2F,$EA,$3E,$0E,$4D,$FA,$7C,$3D,$3B,$3E
	.dc.b	$DB,$E5,$BF,$EB,$9F,$E2,$D1,$9F,$4F,$83,$BA,$F0,$C7,$8B,$7F,$5F
	.dc.b	$C5,$CA,$E9,$1E,$6E,$1D,$46,$0F,$03,$EA,$F4,$FF,$71,$1F,$E3,$93
	.dc.b	$FC,$FF,$8A,$33,$19,$8C,$65,$B8,$DC,$E9,$FF,$21,$4B,$D2,$7C,$FE
	.dc.b	$E5,$39,$8C,$36,$3A,$35,$E1,$F3,$FF,$D7,$13,$48,$F5,$EE,$FE,$8B
	.dc.b	$8F,$F1,$88,$FF,$33,$9F,$E9,$A9,$FF,$4D,$FC,$FF,$F8,$8A,$3F,$CB
	.dc.b	$15,$C3,$A7,$E1,$72,$D9,$FF,$2E,$35,$FF,$51,$CD,$7F,$6F,$3F,$97
	.dc.b	$7F,$12,$DE,$5E,$1C,$BF,$E2,$48,$C1,$BE,$CE,$47,$39,$B1,$CF,$CC
	.dc.b	$EC,$62,$FF,$1E,$7F,$9C,$0F,$E0,$9C,$9E,$5D,$7F,$AC,$D7,$E8,$3B
	.dc.b	$F1,$39,$E4,$93,$FA,$DA,$DD,$FE,$69,$FD,$C4,$7E,$C7,$E7,$FF,$D1
	.dc.b	$E7,$FA,$43,$4F,$E0,$31,$9D,$7E,$24,$FC,$2B,$FD,$1B,$6F,$FA,$4B
	.dc.b	$CB,$6D,$F9,$9C,$3C,$AB,$8F,$77,$0D,$3C,$51,$8C,$3F,$5F,$E1,$7F
	.dc.b	$80,$27,$CD,$E4,$FD,$58,$C4,$F8,$A3,$CA,$79,$BC,$9B,$5E,$D7,$0F
	.dc.b	$BF,$C9,$B5,$EA,$7F,$01,$F3,$FF,$A3,$9F,$E2,$8F,$E7,$5B,$D7,$AE
	.dc.b	$BF,$C3,$61,$FE,$E8,$9F,$E3,$4F,$FA,$E0,$C6,$1F,$C2,$1C,$79,$9C
	.dc.b	$3C,$CE,$2D,$AF,$E4,$5F,$3F,$FE,$19,$6F,$83,$F9,$38,$FB,$3C,$7F
	.dc.b	$DD,$9D,$63,$FC,$51,$FF,$67,$26,$8F,$4B,$97,$A7,$F9,$DF,$9F,$7A
	.dc.b	$54,$C7,$A5,$73,$FE,$69,$F3,$EF,$93,$FD,$23,$63,$3F,$43,$89,$FD
	.dc.b	$47,$D0,$7F,$39,$9F,$E7,$8D,$D7,$F9,$4B,$F3,$5B,$E4,$F0,$D6,$57
	.dc.b	$0F,$62,$D7,$F3,$0F,$69,$F3,$1F,$B5,$7F,$D4,$3C,$5F,$E1,$87,$37
	.dc.b	$FD,$2D,$CB,$FE,$D7,$5E,$6F,$1D,$39,$36,$7F,$A9,$7C,$FF,$CC,$BF
	.dc.b	$F1,$1B,$F4,$23,$FD,$C0,$E5,$BE,$37,$F0,$7F,$9F,$79,$74,$C7,$95
	.dc.b	$6B,$FA,$F4,$FF,$8B,$5B,$ED,$71,$F9,$4E,$4F,$F1,$1F,$9F,$7C,$65
	.dc.b	$31,$F1,$D6,$FF,$B4,$7C,$F7,$F9,$ED,$BF,$31,$FC,$66,$31,$C9,$FD
	.dc.b	$61,$46,$7E,$13,$5C,$96,$F9,$BF,$C8,$7E,$7B,$D7,$1F,$DD,$5F,$E2
	.dc.b	$58,$1C,$BF,$D9,$1C,$FD,$23,$C5,$FE,$69,$21,$19,$AF,$E8,$CD,$8F
	.dc.b	$E9,$CD,$7F,$AB,$7C,$FB,$C0,$E7,$08,$C5,$7F,$4F,$6C,$7F,$57,$6B
	.dc.b	$CB,$CE,$FE,$79,$1F,$EA,$99,$FF,$08,$3C,$3A,$3F,$CC,$2E,$6B,$FB
	.dc.b	$FA,$F3,$78,$9C,$B2,$B7,$CA,$3C,$B3,$B8,$F2,$46,$0C,$F1,$C8,$F4
	.dc.b	$36,$FD,$0F,$9B,$93,$CE,$D9,$FF,$13,$68,$F3,$7F,$AB,$70,$A7,$FA
	.dc.b	$C4,$E5,$FF,$16,$46,$5A,$7C,$A7,$18,$7F,$7D,$6B,$FE,$05,$6B,$F6
	.dc.b	$4F,$C4,$F6,$F8,$0F,$0C,$D3,$BF,$C9,$F9,$0F,$2F,$C9,$D8,$C9,$FE
	.dc.b	$05,$B4,$F8,$C9,$19,$F9,$FF,$31,$1C,$13,$0F,$F1,$9D,$3E,$7F,$E1
	.dc.b	$D7,$C4,$7E,$1E,$BF,$1B,$C0,$B8,$7C,$47,$0F,$8D,$F8,$84,$70,$7C
	.dc.b	$AF,$07,$78,$F8,$1F,$A3,$F3,$F2,$7E,$7F,$C1,$DA,$F2,$CD,$E5,$FF
	.dc.b	$56,$E9,$FE,$55,$BF,$AB,$3F,$E0,$FF,$9E,$FD,$AC,$7F,$98,$DF,$EB
	.dc.b	$B5,$FE,$B4,$73,$7F,$E5,$27,$33,$F5,$5C,$CE,$FE,$0B,$A6,$1D,$5E
	.dc.b	$DE,$44,$BF,$C5,$FC,$F5,$C7,$A4,$FE,$BB,$1F,$E7,$11,$FE,$47,$6B
	.dc.b	$DE,$34,$7A,$7D,$5E,$59,$79,$DF,$67,$C8,$4C,$7D,$9F,$21,$70,$F5
	.dc.b	$4D,$BF,$74,$DA,$F6,$7C,$87,$C5,$3F,$C6,$9F,$47,$FC,$F3,$0F,$E1
	.dc.b	$99,$A7,$55,$F2,$5D,$FC,$E6,$2B,$FC,$F3,$F3,$FF,$DD,$32,$FF,$6F
	.dc.b	$B6,$FF,$3D,$8B,$A7,$FC,$85,$39,$BF,$E3,$4D,$4D,$9E,$15,$FE,$9B
	.dc.b	$8D,$12,$7D,$35,$D1,$93,$B7,$2D,$23,$E8,$32,$57,$37,$EA,$8C,$B6
	.dc.b	$32,$7E,$E6,$BC,$D5,$E0,$AE,$44,$AF,$F8,$03,$7E,$A8,$C5,$F1,$78
	.dc.b	$4E,$AA,$CE,$4E,$F3,$FC,$FE,$B9,$57,$8E,$39,$9F,$28,$F3,$CF,$17
	.dc.b	$27,$CC,$32,$6D,$EA,$E5,$FF,$64,$4F,$67,$94,$F7,$23,$F7,$2B,$CF
	.dc.b	$5C,$CE,$F5,$C5,$C5,$70,$6D,$71,$3C,$37,$F6,$DF,$0D,$F4,$9F,$21
	.dc.b	$D1,$C4,$EE,$3F,$D9,$37,$DD,$DD,$F7,$6D,$F7,$7E,$1D,$71,$9F,$A1
	.dc.b	$3E,$29,$F0,$57,$3A,$D4,$FC,$23,$95,$75,$8C,$D7,$D5,$BF,$2B,$FD
	.dc.b	$B2,$3F,$CA,$EF,$FB,$15,$AF,$D5,$30,$6C,$95,$FE,$BF,$8D,$C7,$C8
	.dc.b	$3F,$4F,$F0,$3F,$08,$93,$F2,$9B,$1E,$E6,$DF,$9D,$B3,$EB,$9E,$6F
	.dc.b	$12,$3C,$5B,$3C,$5A,$71,$8F,$DB,$CF,$47,$CC,$62,$36,$E2,$93,$B8
	.dc.b	$FA,$A7,$88,$F0,$12,$BC,$0D,$AF,$05,$72,$7A,$C7,$23,$C0,$6C,$F0
	.dc.b	$57,$79,$FE,$12,$D8,$FE,$26,$D9,$FE,$1C,$D8,$FE,$3A,$DA,$FE,$3A
	.dc.b	$D9,$FE,$04,$D9,$F5,$4F,$A0,$93,$FC,$1F,$CD,$FE,$43,$E6,$FF,$22
	.dc.b	$6C,$7F,$1D,$6D,$7F,$2C,$6D,$7F,$0A,$FA,$4F,$E5,$3E,$2F,$F3,$86
	.dc.b	$DF,$F2,$58,$B7,$FA,$49,$B5,$FC,$D3,$DC,$FE,$62,$7F,$40,$ED,$B8
	.dc.b	$7F,$2D,$8F,$CC,$11,$FC,$93,$6F,$F9,$FF,$85,$C1,$B3,$FC,$D7,$BB
	.dc.b	$F4,$04,$7F,$34,$FE,$8F,$E5,$FF,$A0,$9F,$D0,$9B,$1F,$E5,$BC,$7F
	.dc.b	$C6,$15,$FE,$5D,$F9,$E7,$D0,$FF,$95,$9B,$5F,$E2,$99,$FF,$15,$46
	.dc.b	$1C,$07,$0E,$FE,$7F,$FD,$36,$E9,$AE,$7C,$5F,$F3,$1C,$62,$3F,$C6
	.dc.b	$DE,$3B,$E8,$7F,$CC,$ED,$C3,$BC,$A3,$0F,$89,$AF,$9F,$E5,$FF,$D8
	.dc.b	$B3,$C1,$B1,$E8,$6B,$BF,$77,$C7,$E9,$CA,$EF,$8A,$31,$5F,$03,$A9
	.dc.b	$E2,$FF,$9F,$1B,$7E,$66,$CE,$E7,$DF,$F2,$32,$BA,$B6,$7E,$5F,$BA
	.dc.b	$F9,$EF,$F4,$3B,$6B,$E0,$B6,$77,$3F,$E8,$CD,$3F,$D0,$11,$F0,$7A
	.dc.b	$70,$4B,$E3,$E4,$74,$23,$83,$E3,$7C,$27,$32,$38,$3E,$57,$A5,$DC
	.dc.b	$8E,$0F,$AB,$F4,$9F,$82,$C3,$FC,$E0,$D7,$FD,$54,$D4,$DF,$A1,$FF
	.dc.b	$0C,$67,$F7,$9B,$3F,$8E,$5F,$43,$FC,$72,$31,$E2,$7C,$77,$F2,$7F
	.dc.b	$3F,$5B,$7F,$88,$51,$97,$F3,$8D,$7F,$D0,$CB,$E8,$38,$36,$3F,$22
	.dc.b	$D7,$BC,$75,$E7,$79,$D4,$93,$F9,$88,$CB,$F9,$AF,$D7,$3F,$D8,$69
	.dc.b	$A4,$D7,$D0,$BA,$E9,$27,$93,$AF,$27,$FD,$3D,$DF,$FB,$C9,$E9,$B6
	.dc.b	$30,$FF,$51,$61,$F2,$A7,$F8,$14,$62,$F5,$1F,$ED,$F9,$FF,$25,$3A
	.dc.b	$31,$EA,$E9,$F4,$36,$37,$0C,$3C,$50,$F7,$B5,$D9,$2B,$A1,$F0,$63
	.dc.b	$97,$8F,$23,$34,$8E,$DE,$B7,$7F,$22,$56,$9F,$17,$6F,$8F,$E4,$E4
	.dc.b	$F2,$7C,$DF,$1D,$E7,$F6,$3D,$1E,$C4,$AE,$F1,$EB,$F0,$7D,$9F,$39
	.dc.b	$27,$C7,$1E,$FF,$67,$E0,$78,$92,$7C,$F1,$F0,$FC,$A9,$1E,$A7,$E4
	.dc.b	$7E,$BB,$F9,$5E,$4F,$AC,$FE,$2D,$BD,$9A,$37,$E2,$70,$F6,$A6,$1F
	.dc.b	$22,$D1,$D3,$E3,$BA,$FC,$34,$AD,$3E,$0F,$95,$F0,$FB,$B2,$79,$3E
	.dc.b	$4F,$77,$CB,$F2,$12,$7B,$3E,$8F,$A2,$F4,$FB,$DE,$AF,$7E,$57,$8E
	.dc.b	$3D,$BF,$2D,$EE,$F8,$79,$3E,$78,$F8,$3F,$0F,$F0,$BE,$06,$4F,$AE
	.dc.b	$3F,$19,$EA,$3E,$5E,$FC,$0C,$AF,$C3,$CE,$5F,$C4,$FC,$FB,$C2,$FC
	.dc.b	$A7,$D0,$3E,$62,$9F,$97,$9D,$57,$0F,$0F,$1E,$24,$8C,$1D,$BE,$0F
	.dc.b	$C6,$47,$BF,$87,$CC,$8E,$0F,$1F,$83,$FA,$5F,$9F,$75,$7E,$9F,$E8
	.dc.b	$1F,$07,$4D,$46,$FB,$2E,$1F,$A4,$BE,$2D,$8F,$D8,$7D,$07,$B6,$3F
	.dc.b	$53,$CE,$CB,$F6,$8D,$8F,$D6,$CE,$DF,$DA,$7A,$46,$47,$CB,$72,$63
	.dc.b	$87,$B9,$E0,$E6,$FE,$01,$F3,$FF,$90,$9F,$0F,$2F,$80,$CF,$85,$AF
	.dc.b	$0A,$CF,$EF,$AB,$83,$D8,$28,$EB,$E1,$71,$9E,$11,$87,$97,$07,$BB
	.dc.b	$CA,$91,$83,$B7,$07,$9A,$64,$60,$F1,$78,$7F,$C5,$5F,$F0,$CE,$28
	.dc.b	$C4,$7F,$87,$E3,$1E,$3F,$F8,$41,$B1,$C3,$C4,$7D,$06,$5B,$19,$8C
	.dc.b	$1F,$D4,$57,$D0,$52,$3F,$D5,$5F,$EA,$DE,$07,$F8,$D3,$0F,$E7,$53
	.dc.b	$FE,$71,$3E,$41,$F6,$67,$ED,$7D,$03,$93,$FC,$6D,$7C,$79,$7F,$C7
	.dc.b	$1C,$B7,$0F,$C0,$1F,$A5,$8F,$E6,$2F,$5C,$7B,$CE,$E3,$F0,$DC,$B5
	.dc.b	$FD,$A5,$7E,$43,$D4,$C4,$E7,$B6,$4F,$48,$B5,$E8,$F8,$BE,$46,$31
	.dc.b	$F6,$BD,$D7,$35,$3C,$5D,$2E,$FC,$63,$FD,$22,$BF,$5A,$DE,$5C,$E3
	.dc.b	$F8,$D9,$CB,$F8,$FF,$A0,$F7,$57,$C2,$6C,$7E,$8F,$07,$81,$F0,$67
	.dc.b	$32,$7E,$43,$CF,$7D,$70,$6D,$E1,$F0,$9C,$7E,$82,$E5,$FD,$7B,$04
	.dc.b	$66,$30,$EB,$0F,$33,$F4,$7C,$3F,$E7,$B3,$C7,$E4,$71,$DF,$E5,$64
	.dc.b	$75,$F8,$EB,$87,$BD,$F5,$1E,$BF,$8E,$B3,$E3,$7E,$F7,$D0,$7E,$86
	.dc.b	$3F,$B9,$AE,$DC,$F7,$E9,$FC,$25,$FF,$09,$AF,$EF,$2F,$C6,$31,$87
	.dc.b	$EF,$5C,$7C,$4F,$0F,$89,$E2,$DA,$FF,$00,$FD,$07,$F0,$16,$F8,$BF
	.dc.b	$91,$8F,$C1,$C6,$0D,$FD,$06,$4D,$7C,$ED,$8F,$F0,$5F,$D0,$7F,$9F
	.dc.b	$BA,$BF,$C4,$4D,$8F,$D0,$AF,$98,$FC,$F3,$9D,$7F,$92,$7D,$03,$DF
	.dc.b	$FD,$13,$6F,$FC,$2E,$BD,$FC,$CF,$E9,$FE,$81,$E6,$53,$1C,$97,$0F
	.dc.b	$F0,$FC,$B5,$E0,$71,$FA,$0B,$3F,$E2,$4F,$5F,$FD,$53,$BF,$F8,$ED
	.dc.b	$7C,$47,$ED,$5C,$CF,$EF,$1C,$BC,$7D,$0D,$69,$B6,$C7,$F8,$C3,$E8
	.dc.b	$3E,$87,$07,$F3,$37,$E7,$B7,$97,$F8,$14,$63,$2F,$F2,$67,$D0,$3C
	.dc.b	$CF,$E0,$CD,$C3,$FC,$76,$BC,$BC,$CE,$BE,$36,$56,$BE,$66,$E7,$FD
	.dc.b	$0F,$E8,$3F,$C8,$D3,$FE,$44,$6C,$67,$D0,$FE,$29,$18,$CF,$FC,$BD
	.dc.b	$F4,$1F,$E7,$7B,$FF,$25,$AF,$85,$FE,$34,$E7,$0F,$EA,$0E,$67,$7F
	.dc.b	$03,$AE,$3D,$B6,$3F,$CA,$BF,$41,$ED,$BF,$EB,$0F,$B6,$33,$B7,$F2
	.dc.b	$78,$C4,$FF,$5D,$F9,$F7,$87,$FC,$AF,$E8,$3F,$21,$87,$F9,$82,$72
	.dc.b	$FF,$98,$A3,$35,$FD,$95,$79,$DC,$4E,$5E,$15,$AE,$11,$FD,$1A,$BF
	.dc.b	$BE,$57,$FB,$09,$FE,$7E,$D7,$31,$B5,$0F,$2B,$0D,$AF,$F7,$7B,$67
	.dc.b	$FD,$FE,$D7,$FA,$59,$CA,$7D,$DE,$3F,$BD,$B5,$ED,$E5,$F8,$5C,$BF
	.dc.b	$0D,$B1,$F0,$63,$C2,$47,$C2,$F9,$F8,$91,$48,$78,$B0,$FF,$0E,$46
	.dc.b	$A3,$11,$F8,$3F,$93,$FE,$B9,$3F,$A8,$8D,$CE,$26,$3C,$27,$FD,$60
	.dc.b	$D8,$F1,$1E,$3F,$92,$91,$B6,$D7,$8E,$91,$E0,$F9,$2F,$0B,$6B,$FD
	.dc.b	$43,$2F,$90,$F2,$FC,$87,$99,$B3,$A8,$F8,$44,$7C,$CF,$C5,$EF,$91
	.dc.b	$E9,$6E,$3E,$3F,$03,$E0,$9F,$0A,$2C,$F8,$A2,$CF,$F8,$D7,$05,$63
	.dc.b	$AB,$FD,$99,$0F,$90,$FA,$84,$C3,$E4,$36,$F9,$7C,$87,$D0,$8F,$A2
	.dc.b	$FF,$83,$7E,$83,$FA,$EC,$FF,$90,$DB,$87,$FB,$65,$FD,$B4,$67,$D3
	.dc.b	$E0,$7E,$66,$38,$C7,$E4,$9A,$FF,$87,$9A,$FF,$97,$DF,$E6,$B8,$7F
	.dc.b	$B4,$1F,$EC,$11,$E1,$F2,$BC,$9E,$54,$AD,$3E,$78,$C4,$F9,$E2,$E1
	.dc.b	$E7,$9B,$5E,$1F,$8E,$F7,$7B,$7E,$F6,$C7,$85,$B8,$7C,$16,$DF,$C1
	.dc.b	$F4,$3E,$31,$F1,$FE,$36,$DF,$A6,$2D,$FC,$99,$B5,$FF,$00,$7E,$83
	.dc.b	$FC,$32,$49,$FF,$1E,$38,$F7,$29,$88,$B8,$7F,$BD,$27,$4D,$8C,$9F
	.dc.b	$E7,$98,$CB,$4C,$D7,$86,$AD,$7F,$94,$2B,$FC,$0B,$39,$8E,$D3,$AA
	.dc.b	$FF,$4A,$56,$A7,$84,$E6,$53,$EB,$9E,$24,$8F,$24,$FA,$E3,$15,$DE
	.dc.b	$73,$1E,$B3,$27,$62,$BE,$23,$C1,$5E,$E9,$F9,$91,$8A,$CB,$6B,$8B
	.dc.b	$E6,$30,$78,$0F,$8C,$EA,$BA,$3F,$16,$37,$5E,$73,$BC,$6E,$BB,$B4
	.dc.b	$E5,$3C,$1F,$53,$D8,$F0,$4F,$38,$CB,$D2,$33,$18,$23,$F0,$5E,$CE
	.dc.b	$8C,$14,$FA,$73,$E4,$71,$5E,$C7,$75,$E1,$7E,$EC,$FF,$51,$9F,$F4
	.dc.b	$E4,$7F,$4A,$6B,$EA,$31,$1B,$25,$7F,$3C,$9D,$CF,$48,$FF,$6E,$57
	.dc.b	$F4,$68,$FF,$6F,$7C,$77,$FB,$C2,$7E,$AB,$63,$D4,$D9,$F3,$42,$31
	.dc.b	$3F,$12,$31,$F1,$7F,$93,$9C,$C7,$84,$FF,$6E,$38,$7E,$2C,$F9,$0F
	.dc.b	$F5,$8C,$66,$FF,$D1,$AD,$C3,$FD,$60,$D9,$E1,$5E,$AB,$F0,$12,$3D
	.dc.b	$6D,$8F,$5B,$70,$F5,$D7,$39,$F2,$CE,$6B,$D9,$39,$AF,$91,$5E,$E9
	.dc.b	$EA,$47,$AB,$63,$AD,$73,$9F,$85,$1C,$AB,$A5,$7C,$0A,$D5,$7C,$43
	.dc.b	$C7,$1C,$CF,$1C,$6B,$0F,$8A,$DC,$3E,$2B,$67,$E2,$D7,$9C,$F2,$93
	.dc.b	$0F,$2C,$75,$8F,$2B,$E3,$30,$F8,$E3,$D1,$3E,$1F,$2F,$F9,$3F,$C2
	.dc.b	$7F,$28,$6D,$FF,$25,$6D,$FF,$23,$6D,$FF,$31,$F7,$7F,$97,$9F,$CC
	.dc.b	$3C,$3B,$7F,$CB,$62,$DF,$F2,$B9,$B5,$FC,$E7,$DC,$FE,$6E,$7F,$38
	.dc.b	$F0,$EE,$1F,$CD,$63,$F5,$44,$7F,$48,$FF,$8F,$7E,$83,$FC,$45,$82
	.dc.b	$FE,$EC,$61,$ED,$5F,$2A,$30,$FF,$37,$D3,$F6,$C4,$7F,$60,$FE,$E7
	.dc.b	$E8,$24,$7E,$CD,$B7,$FE,$33,$F9,$FF,$F4,$9B,$7D,$5F,$F2,$84,$62
	.dc.b	$3F,$C9,$2D,$7F,$4B,$BF,$F9,$57,$88,$D2,$61,$FE,$3C,$AF,$F4,$2D
	.dc.b	$72,$F0,$FA,$92,$75,$E9,$78,$08,$E0,$FF,$22,$E8,$F9,$8F,$90,$48
	.dc.b	$CB,$E8,$F8,$D9,$1E,$16,$E1,$FE,$51,$CF,$FC,$F2,$FF,$99,$1A,$FF
	.dc.b	$00,$F3,$3D,$87,$B7,$8E,$5F,$F9,$5A,$BF,$CE,$D5,$FC,$46,$7E,$17
	.dc.b	$24,$AD,$36,$7C,$1E,$FF,$C7,$24,$F3,$7E,$57,$9F,$23,$B3,$67,$FC
	.dc.b	$D1,$A7,$CA,$44,$70,$DD,$BF,$18,$7F,$A2,$7E,$75,$F3,$AF,$9D,$FF
	.dc.b	$37,$C6,$1B,$C0,$EB,$CD,$6F,$0F,$F3,$C7,$D1,$FF,$A7,$9F,$F3,$BB
	.dc.b	$5C,$EF,$FE,$7A,$51,$97,$FD,$23,$F3,$CF,$9B,$D3,$63,$FD,$02,$D7
	.dc.b	$FD,$0C,$D7,$FD,$33,$E9,$3D,$2E,$CD,$8F,$F4,$4B,$5F,$F4,$73,$5F
	.dc.b	$F5,$0F,$CE,$FF,$AA,$D3,$49,$C5,$E9,$F9,$C7,$8B,$49,$5F,$E9,$68
	.dc.b	$CB,$7E,$0B,$81,$1C,$1F,$E9,$CB,$77,$7D,$88,$FD,$F6,$16,$7F,$39
	.dc.b	$1F,$C9,$9A,$E8,$FB,$EF,$27,$5B,$F3,$F0,$7F,$A1,$CF,$FA,$6A,$79
	.dc.b	$7A,$99,$BE,$D7,$67,$9F,$35,$7E,$1F,$75,$C0,$D9,$C7,$CE,$CA,$F3
	.dc.b	$3B,$F5,$78,$7A,$B2,$7D,$4F,$3F,$07,$A7,$C8,$49,$F7,$3D,$FE,$33
	.dc.b	$C1,$EB,$78,$7D,$79,$5F,$0E,$3C,$9F,$0F,$E5,$F8,$F9,$3F,$1E,$3D
	.dc.b	$1F,$2F,$E9,$F6,$64,$FC,$B8,$F6,$7B,$32,$3E,$73,$F0,$7A,$7F,$3D
	.dc.b	$1F,$3C,$3F,$93,$E7,$FE,$DE,$DE,$6F,$B4,$D9,$FD,$9A,$F6,$BA,$7B
	.dc.b	$52,$7F,$B0,$CA,$3C,$1F,$0F,$E1,$F9,$29,$5A,$7C,$9F,$25,$E5,$F6
	.dc.b	$E4,$F2,$7D,$1E,$E7,$A7,$E3,$E4,$F6,$7D,$9F,$3B,$ED,$F9,$DF,$77
	.dc.b	$BD,$2B,$C7,$1F,$07,$DF,$F8,$5F,$0F,$27,$CF,$1F,$17,$E2,$3E,$37
	.dc.b	$C0,$C9,$F5,$C7,$E2,$7D,$87,$CA,$DF,$4A,$57,$F8,$02,$72,$FF,$80
	.dc.b	$BE,$71,$E1,$7E,$47,$E7,$5F,$2F,$4F,$CA,$4E,$AB,$87,$87,$8F,$12
	.dc.b	$46,$0E,$DF,$19,$E3,$23,$DF,$C3,$E6,$47,$07,$8F,$C1,$FD,$17,$CE
	.dc.b	$3A,$BF,$49,$F3,$AF,$84,$A6,$A3,$7D,$97,$0F,$F0,$ED,$F1,$6C,$7E
	.dc.b	$B7,$E0,$FD,$B1,$FE,$22,$E7,$65,$FB,$16,$C7,$F8,$9A,$76,$FE,$4D
	.dc.b	$F9,$2D,$E0,$E8,$75,$E5,$FD,$D7,$CE,$FE,$36,$7B,$72,$75,$33,$E0
	.dc.b	$EB,$C1,$B3,$FE,$34,$AE,$0F,$E8,$54,$74,$F0,$78,$4E,$E3,$0F,$1D
	.dc.b	$FD,$5E,$4C,$8C,$1D,$77,$F2,$CC,$8C,$1E,$1F,$0B,$FC,$40,$FF,$25
	.dc.b	$3F,$AE,$E2,$8C,$C6,$63,$1E,$36,$E3,$73,$A7,$F9,$74,$BC,$37,$CE
	.dc.b	$EE,$53,$98,$C3,$63,$A3,$5E,$1F,$3B,$FD,$75,$34,$8F,$5E,$FF,$F4
	.dc.b	$E6,$3F,$E7,$68,$FE,$D9,$E4,$7F,$98,$9F,$EF,$35,$FD,$EB,$E7,$7F
	.dc.b	$C5,$31,$FD,$16,$F8,$D7,$C3,$E4,$B3,$FD,$25,$AF,$F8,$0A,$6B,$FA
	.dc.b	$A9,$ED,$C9,$83,$7C,$DF,$CF,$23,$27,$8F,$B3,$89,$B7,$07,$9F,$E2
	.dc.b	$FA,$9A,$70,$7F,$9F,$34,$72,$F9,$1B,$1F,$9F,$EE,$C4,$E7,$8E,$4F
	.dc.b	$18,$B9,$FF,$31,$F9,$DF,$8E,$DD,$7D,$3D,$FF,$02,$73,$97,$FA,$89
	.dc.b	$F8,$15,$C7,$7D,$5E,$DE,$1F,$01,$C7,$E1,$EE,$7F,$E9,$4C,$11,$98
	.dc.b	$C3,$AC,$3C,$8F,$D0,$F0,$BF,$9D,$4F,$1F,$90,$C7,$6F,$95,$91,$D7
	.dc.b	$E3,$6E,$1E,$D7,$D4,$7A,$FE,$36,$CF,$8D,$FB,$BF,$3A,$F3,$E9,$8F
	.dc.b	$3A,$D7,$FA,$B2,$FE,$0F,$0F,$FB,$11,$B1,$FE,$0F,$7F,$9A,$BB,$E0
	.dc.b	$FC,$4F,$C3,$B6,$A6,$39,$AD,$7F,$AE,$3E,$77,$F9,$23,$79,$FF,$23
	.dc.b	$0F,$0A,$C6,$3E,$6A,$57,$1F,$A3,$BC,$5F,$DB,$7E,$D5,$5F,$9A,$6D
	.dc.b	$7F,$B1,$97,$C2,$66,$7F,$3F,$F3,$CE,$CA,$63,$CC,$B8,$7F,$B3,$BE
	.dc.b	$7B,$F8,$03,$7C,$8F,$D3,$C7,$CB,$8C,$1F,$ED,$43,$ED,$E7,$FE,$01
	.dc.b	$6F,$2F,$FB,$B9,$FE,$F8,$6F,$AF,$F6,$3E,$A7,$FA,$4E,$3F,$DB,$AD
	.dc.b	$AF,$F0,$53,$FE,$70,$8C,$FC,$6F,$0F,$76,$47,$0D,$2D,$FF,$84,$7E
	.dc.b	$D1,$E1,$7E,$F9,$B5,$FE,$EA,$5F,$27,$99,$FE,$07,$F3,$DF,$EB,$2A
	.dc.b	$FF,$77,$B6,$FF,$C3,$8B,$E9,$37,$EA,$70,$F5,$A5,$F0,$F9,$7B,$87
	.dc.b	$F8,$8F,$D0,$7C,$0F,$F1,$06,$D7,$FB,$F1,$7C,$16,$67,$F8,$AF,$CF
	.dc.b	$7C,$AA,$FF,$30,$36,$FF,$D3,$2F,$F4,$E8,$FF,$4D,$F2,$9D,$F2,$F2
	.dc.b	$F7,$E4,$68,$E7,$F1,$D8,$C3,$FC,$71,$E7,$BD,$1F,$E4,$AD,$AE,$13
	.dc.b	$FC,$9F,$D2,$CD,$7F,$2A,$F9,$D7,$A1,$FC,$B3,$E7,$5F,$31,$98,$C3
	.dc.b	$96,$63,$D3,$15,$70,$8C,$3D,$1B,$3F,$D0,$E7,$FD,$73,$39,$7F,$AD
	.dc.b	$35,$FE,$C4,$D7,$FD,$10,$BE,$4B,$94,$78,$1B,$19,$AF,$EB,$CD,$9E
	.dc.b	$B3,$ED,$F9,$3F,$73,$6B,$D5,$8B,$D4,$FE,$9B,$F1,$3F,$4E,$73,$18
	.dc.b	$61,$FD,$AE,$35,$87,$F7,$36,$BF,$DE,$DA,$FF,$A8,$5F,$D2,$3B,$F9
	.dc.b	$1F,$5C,$62,$38,$46,$3E,$4B,$C6,$79,$1B,$5F,$46,$93,$B8,$F2,$36
	.dc.b	$BC,$9F,$1D,$D8,$93,$E9,$6C,$7F,$A0,$69,$F1,$DF,$0A,$7E,$1B,$F8
	.dc.b	$7D,$3E,$2F,$C1,$DC,$3C,$4E,$1F,$88,$FB,$08,$FC,$7F,$2B,$DE,$47
	.dc.b	$07,$CB,$EB,$FA,$27,$D2,$8B,$1E,$F8,$B9,$79,$E6,$F4,$FE,$A6,$03
	.dc.b	$E7,$3F,$8C,$BF,$E2,$76,$BF,$E2,$F6,$BF,$ED,$56,$BC,$9E,$31,$D6
	.dc.b	$3A,$75,$7F,$84,$A7,$38,$71,$8C,$7A,$52,$3F,$1B,$F4,$56,$FF,$D5
	.dc.b	$D7,$FD,$26,$7F,$C6,$8D,$8F,$35,$7E,$7A,$31,$1E,$7F,$6B,$C1,$F1
	.dc.b	$DE,$16,$DF,$EC,$E9,$38,$AE,$F5,$72,$F5,$56,$AF,$85,$F3,$69,$FB
	.dc.b	$DF,$A4,$FF,$7C,$BD,$7B,$FF,$C6,$31,$8C,$7D,$71,$8F,$96,$F8,$8D
	.dc.b	$AE,$07,$F0,$7F,$9E,$FA,$AD,$AE,$7C,$FF,$E3,$C9,$CC,$7F,$B4,$DB
	.dc.b	$9F,$D1,$6D,$7D,$E9,$63,$FA,$CA,$D4,$73,$EF,$95,$EB,$E8,$B3,$D6
	.dc.b	$6D,$7D,$DB,$FD,$5D,$F9,$E7,$2F,$7C,$11,$F0,$B0,$CE,$1F,$90,$9C
	.dc.b	$D7,$13,$C0,$58,$F0,$37,$0F,$05,$72,$95,$72,$76,$7F,$BD,$70,$F2
	.dc.b	$B8,$AF,$89,$87,$19,$CC,$63,$0E,$C7,$A6,$7D,$0E,$2B,$D1,$1C,$70
	.dc.b	$F1,$61,$E1,$BE,$55,$E1,$9C,$D6,$EF,$94,$FA,$23,$0F,$96,$3C,$06
	.dc.b	$B0,$F2,$B8,$8D,$1E,$27,$E3,$B8,$3A,$3C,$6B,$C2,$D3,$C3,$87,$9D
	.dc.b	$E9,$87,$CC,$73,$87,$92,$3D,$64,$8D,$9F,$8B,$C3,$CB,$87,$8C,$F1
	.dc.b	$5F,$57,$F8,$6C,$69,$F2,$9A,$BF,$9A,$DA,$F3,$34,$F6,$4F,$1A,$E9
	.dc.b	$1C,$AF,$C8,$F6,$C3,$DE,$FF,$AF,$E7,$F0,$8F,$FB,$1D,$AF,$CB,$31
	.dc.b	$87,$F6,$68,$CC,$7A,$CC,$E1,$FE,$C3,$8F,$09,$FE,$07,$8F,$7F,$7F
	.dc.b	$E0,$89,$3F,$39,$B1,$A8,$C1,$1C,$13,$0F,$5B,$F9,$C9,$E3,$18,$AF
	.dc.b	$03,$63,$35,$F5,$2B,$F5,$91,$87,$EA,$4F,$F0,$38,$C4,$7C,$99,$FD
	.dc.b	$B4,$62,$BD,$AD,$BF,$E1,$CD,$9F,$E2,$8D,$AF,$E2,$0D,$BE,$D3,$ED
	.dc.b	$7C,$CD,$3C,$D3,$ED,$70,$F9,$9F,$19,$87,$CC,$7B,$1C,$5F,$99,$B7
	.dc.b	$E6,$69,$E6,$2C,$F9,$9E,$98,$BE,$53,$F9,$57,$97,$FC,$D9,$B3,$FC
	.dc.b	$ED,$B5,$FC,$2D,$B8,$FF,$2D,$F7,$FF,$99,$1F,$CF,$BA,$AD,$FF,$39
	.dc.b	$8B,$7F,$C4,$E6,$D7,$F3,$5E,$AF,$E7,$47,$F4,$6F,$8C,$B8,$7F,$41
	.dc.b	$8F,$CA,$11,$FC,$93,$F9,$7F,$2B,$FA,$11,$C3,$0F,$E8,$4F,$F5,$0E
	.dc.b	$CF,$F2,$0E,$A6,$92,$FF,$A2,$57,$F4,$5F,$9C,$FF,$3B,$B8,$0E,$1D
	.dc.b	$B5,$E1,$E4,$FF,$93,$BC,$37,$97,$49,$7C,$EB,$FC,$A9,$1F,$E5,$4F
	.dc.b	$92,$FF,$47,$4E,$5B,$3E,$16,$BE,$66,$BF,$D6,$FE,$53,$FD,$86,$F9
	.dc.b	$DA,$FB,$1A,$EF,$DA,$F8,$7E,$1C,$BF,$26,$BF,$E9,$46,$C7,$F9,$A3
	.dc.b	$E7,$9C,$34,$93,$EB,$8C,$BE,$DF,$59,$F0,$7F,$E7,$06,$C6,$63,$07
	.dc.b	$C1,$E8,$91,$F0,$5B,$9F,$D1,$AF,$F1,$6A,$69,$3C,$8F,$95,$F3,$D2
	.dc.b	$FE,$53,$6B,$ED,$7D,$27,$B5,$BA,$7F,$A1,$23,$11,$F8,$26,$BF,$E0
	.dc.b	$77,$EF,$3F,$E4,$66,$A6,$98,$F9,$F7,$83,$F0,$A7,$F1,$51,$8D,$3F
	.dc.b	$04,$BD,$B7,$A5,$8B,$E1,$7F,$94,$F9,$DA,$DC,$3F,$14,$A3,$2F,$F8
	.dc.b	$75,$AF,$F9,$71,$7C,$F7,$06,$C7,$E4,$DA,$FE,$7D,$AF,$E9,$3A,$1E
	.dc.b	$67,$81,$B1,$F9,$B6,$BF,$A7,$6B,$FA,$CF,$9E,$FF,$58,$A6,$93,$83
	.dc.b	$D7,$D0,$F8,$5A,$4A,$FD,$44,$65,$FD,$57,$EB,$DF,$09,$34,$8F,$C6
	.dc.b	$7E,$6B,$C3,$6C,$11,$E7,$F1,$7F,$E4,$5F,$0F,$FC,$F1,$4F,$92,$C6
	.dc.b	$5F,$E4,$98,$C5,$FF,$B6,$D5,$EA,$3F,$D1,$7F,$93,$3E,$7B,$11,$E6
	.dc.b	$F2,$AD,$FF,$91,$23,$FD,$C9,$5B,$7F,$DC,$DF,$66,$F8,$EE,$2F,$EC
	.dc.b	$A3,$3E,$06,$27,$9F,$9D,$D3,$E0,$65,$79,$63,$BF,$C1,$78,$3E,$12
	.dc.b	$4F,$A6,$3C,$BF,$64,$F8,$CB,$F0,$72,$BF,$DF,$13,$97,$E0,$78,$DE
	.dc.b	$AF,$93,$91,$83,$E2,$78,$BE,$EF,$84,$91,$83,$FD,$5F,$D1,$FE,$A5
	.dc.b	$F9,$5F,$EA,$51,$8E,$8F,$F3,$B4,$3E,$8B,$E4,$B6,$74,$7F,$AE,$1E
	.dc.b	$2A,$BE,$CF,$C3,$7C,$EF,$5E,$4F,$D7,$8F,$A3,$F0,$9F,$4B,$EA,$64
	.dc.b	$FD,$B8,$FA,$FF,$60,$FA,$3B,$F3,$52,$BC,$22,$74,$FE,$0F,$C6,$FB
	.dc.b	$BE,$DC,$8C,$1C,$3E,$B9,$1F,$82,$F7,$24,$E6,$3F,$0F,$F5,$EF,$4E
	.dc.b	$FC,$4C,$AF,$0C,$E5,$FC,$97,$B7,$F8,$CF,$3A,$46,$0F,$3F,$67,$FB
	.dc.b	$4F,$93,$FD,$A8,$BE,$A3,$F2,$AD,$9D,$F4,$7E,$63,$F0,$8F,$CB,$47
	.dc.b	$CD,$8F,$F5,$54,$7E,$8A,$30,$7F,$BA,$9C,$5F,$FB,$85,$E2,$FF,$B8
	.dc.b	$FD,$DC,$D7,$FA,$CD,$FD,$AD,$72,$0E,$27,$87,$9B,$FA,$AF,$A6,$FD
	.dc.b	$15,$6F,$E9,$FF,$62,$FD,$46,$BC,$E3,$75,$FE,$F3,$5F,$3B,$CF,$C9
	.dc.b	$95,$F7,$27,$2F,$DE,$C7,$D3,$97,$9F,$CE,$FD,$DC,$62,$42,$9F,$2D
	.dc.b	$E7,$24,$75,$7D,$5C,$7F,$C1,$3E,$99,$F3,$BF,$C1,$BE,$9D,$D3,$4D
	.dc.b	$46,$FE,$46,$E3,$F9,$2B,$E2,$D8,$FE,$1F,$E7,$FC,$2E,$16,$FF,$EE
	.dc.b	$1E,$F6,$9F,$C5,$1B,$87,$E7,$2F,$85,$7F,$B9,$DF,$AF,$1E,$E7,$2D
	.dc.b	$F5,$7B,$F2,$FF,$1E,$FA,$47,$C9,$F5,$F4,$E5,$FE,$A2,$72,$DF,$A7
	.dc.b	$E1,$52,$30,$6F,$CA,$E9,$32,$30,$73,$E1,$F5,$7C,$5C,$8C,$1D,$F8
	.dc.b	$7C,$93,$23,$07,$8F,$E9,$FF,$84,$3F,$E3,$FC,$91,$88,$FF,$22,$C6
	.dc.b	$38,$BF,$C7,$4D,$8E,$1F,$3E,$FA,$7C,$B6,$33,$18,$3F,$AC,$7C,$E4
	.dc.b	$8F,$EF,$6D,$7F,$CA,$1E,$F7,$D8,$7F,$48,$7F,$81,$B9,$3F,$88,$BF
	.dc.b	$90,$9F,$C8,$FD,$3B,$97,$FC,$B7,$7C,$73,$E5,$7F,$96,$E5,$C6,$3F
	.dc.b	$E6,$06,$C7,$E6,$8E,$C7,$81,$CB,$F9,$EE,$5A,$FF,$65,$9F,$E4,$7C
	.dc.b	$DF,$CD,$DF,$2C,$EB,$BA,$46,$0F,$4B,$CB,$DB,$91,$83,$DA,$F8,$31
	.dc.b	$73,$53,$C7,$87,$F6,$5A,$FE,$67,$3A,$6F,$1F,$13,$83,$83,$F9,$CF
	.dc.b	$D3,$FB,$A7,$E2,$B6,$3F,$4D,$DD,$F0,$23,$1A,$7E,$F9,$FE,$FF,$3C
	.dc.b	$B5,$D6,$DB,$78,$7B,$1C,$7D,$8B,$5E,$07,$0D,$3C,$71,$8C,$3F,$91
	.dc.b	$60,$F5,$3C,$47,$D2,$F1,$7F,$5B,$7E,$A8,$CB,$EC,$F5,$6D,$F2,$7D
	.dc.b	$5E,$A4,$9D,$1F,$09,$FB,$BF,$4E,$FA,$7F,$C3,$DF,$E0,$E3,$15,$FE
	.dc.b	$99,$6C,$7F,$A9,$5A,$FF,$AA,$AF,$E2,$BF,$EA,$96,$B9,$D7,$AB,$FC
	.dc.b	$A2,$32,$FE,$47,$E2,$3F,$9A,$DF,$FA,$9D,$B1,$FE,$B3,$6B,$FE,$B8
	.dc.b	$6B,$F9,$9F,$A7,$FE,$57,$3F,$EA,$F6,$C7,$FA,$ED,$AF,$FB,$01,$AF
	.dc.b	$1F,$92,$F2,$A8,$FF,$54,$BF,$CF,$9A,$E6,$37,$F4,$F8,$23,$83,$9F
	.dc.b	$D3,$FF,$44,$BF,$F6,$13,$6B,$FD,$9C,$D7,$FD,$A6,$D7,$F6,$9F,$4E
	.dc.b	$F0,$7C,$AD,$AF,$F6,$63,$67,$FD,$B0,$D7,$FD,$BB,$1F,$88,$7F,$DB
	.dc.b	$8D,$73,$E3,$FC,$08,$C4,$7F,$03,$FA,$7F,$F4,$B4,$7F,$B6,$5B,$1F
	.dc.b	$EE,$56,$BF,$EE,$A6,$BC,$7E,$53,$E5,$C7,$FA,$E6,$79,$46,$0F,$9F
	.dc.b	$F4,$EE,$7A,$49,$FF,$74,$46,$5F,$F7,$57,$D3,$FF,$1F,$6F,$87,$F6
	.dc.b	$23,$18,$FF,$BE,$5A,$FF,$BF,$9F,$ED,$B3,$CB,$CF,$E0,$47,$FD,$ED
	.dc.b	$E3,$7F,$AF,$89,$F3,$3F,$84,$7F,$D8,$3F,$4F,$F1,$B0,$FF,$00,$B5
	.dc.b	$34,$9F,$41,$F8,$C4,$D2,$5F,$F3,$19,$FE,$65,$E0,$3C,$6F,$C9,$B6
	.dc.b	$75,$3F,$E1,$08,$C3,$FD,$22,$7F,$33,$F4,$EF,$87,$A4,$BE,$34,$9D
	.dc.b	$46,$1F,$E2,$0D,$F0,$39,$91,$C1,$FE,$62,$87,$67,$F4,$67,$8D,$7F
	.dc.b	$8E,$DC,$CE,$FE,$0F,$5F,$05,$B6,$C7,$38,$F4,$91,$F2,$3F,$B4,$F9
	.dc.b	$AF,$F0,$DC,$7A,$9B,$7F,$11,$AF,$CB,$6B,$FD,$75,$FF,$30,$47,$F9
	.dc.b	$E9,$B1,$FE,$51,$79,$F9,$1F,$DA,$1C,$B7,$E8,$7F,$B1,$C1,$BF,$31
	.dc.b	$86,$5F,$95,$F4,$11,$C1,$E1,$FA,$8D,$E1,$E1,$7A,$FD,$3F,$6F,$A1
	.dc.b	$B5,$E7,$70,$FC,$17,$E2,$11,$F8,$9E,$C7,$C7,$23,$83,$E4,$7C,$B7
	.dc.b	$A8,$F5,$C5,$8F,$89,$F1,$3E,$E3,$DF,$EE,$D8,$C9,$F4,$63,$EA,$11
	.dc.b	$C1,$F3,$7A,$B8,$67,$F3,$5F,$F0,$5F,$D3,$37,$BE,$1E,$1B,$7F,$EA
	.dc.b	$E9,$FE,$31,$C3,$FE,$1A,$6C,$7F,$85,$DE,$4E,$EB,$AF,$B9,$DB,$DC
	.dc.b	$B3,$CD,$C3,$4E,$D1,$8C,$BF,$D2,$56,$FA,$6F,$39,$E8,$FA,$7B,$7E
	.dc.b	$58,$FE,$8F,$3B,$8F,$D6,$B6,$BF,$6E,$D7,$F7,$CE,$72,$F6,$7B,$1D
	.dc.b	$27,$C2,$FC,$6E,$FF,$8F,$F1,$17,$0F,$88,$E1,$A7,$C7,$8C,$57,$FA
	.dc.b	$AF,$E9,$BF,$C3,$29,$A7,$FA,$62,$7F,$DB,$6E,$3C,$1F,$F1,$FB,$A9
	.dc.b	$B1,$8F,$06,$57,$0F,$72,$D7,$79,$B9,$FF,$94,$30,$56,$2F,$FC,$9D
	.dc.b	$1F,$C6,$27,$55,$E0,$AD,$5F,$FA,$B2,$B5,$7C,$2B,$4F,$A1,$DC,$FB
	.dc.b	$5C,$BE,$38,$D1,$1F,$89,$5C,$2B,$AC,$77,$AE,$37,$E4,$BE,$0D,$9E
	.dc.b	$31,$83,$C8,$57,$D1,$7D,$6B,$B3,$67,$53,$F4,$E3,$0F,$9C,$EA,$5A
	.dc.b	$F0,$12,$7E,$4C,$F0,$8C,$4F,$82,$7A,$DF,$B2,$3E,$24,$7D,$73,$37
	.dc.b	$E4,$69,$C2,$FA,$3E,$C9,$F1,$DF,$67,$99,$E5,$8F,$19,$C9,$C1,$E3
	.dc.b	$29,$C2,$39,$5F,$E1,$27,$84,$62,$BC,$35,$F4,$E7,$D0,$F2,$AF,$61
	.dc.b	$27,$E4,$C7,$F6,$98,$FE,$C6,$DB,$FF,$1D,$C6,$23,$D8,$66,$BF,$B1
	.dc.b	$C6,$63,$C0,$7F,$6E,$7B,$F8,$FF,$87,$24,$FC,$E6,$CF,$95,$B8,$7B
	.dc.b	$9B,$3F,$2E,$3F,$49,$18,$9E,$F1,$89,$E8,$48,$E8,$D8,$E8,$D3,$A1
	.dc.b	$FA,$C9,$E9,$3F,$3E,$30,$6D,$EE,$E1,$EA,$49,$EA,$D8,$EA,$FE,$CE
	.dc.b	$7A,$D7,$C4,$9E,$93,$DC,$D3,$88,$D4,$73,$3C,$26,$8C,$1C,$4F,$E2
	.dc.b	$D3,$DA,$FF,$89,$B6,$BF,$8B,$B6,$7F,$8F,$36,$FE,$54,$74,$BF,$0C
	.dc.b	$7F,$13,$F2,$7F,$8E,$FB,$3F,$C7,$9B,$87,$F2,$26,$DF,$F2,$96,$D7
	.dc.b	$F1,$BF,$0F,$F9,$2F,$AF,$FC,$99,$B7,$FC,$AA,$2D,$FF,$31,$9B,$5F
	.dc.b	$C8,$FC,$9F,$E5,$A7,$F2,$EF,$0E,$E1,$FC,$CA,$3F,$2E,$47,$F2,$AF
	.dc.b	$E6,$BE,$32,$47,$E6,$1B,$87,$F9,$13,$E8,$FF,$C3,$6F,$F9,$2D,$AF
	.dc.b	$FB,$F5,$A8,$38,$6F,$D3,$FF,$A8,$A0,$D2,$57,$F9,$2A,$1E,$26,$C9
	.dc.b	$19,$79,$7C,$A4,$8E,$0D,$BF,$47,$89,$DC,$93,$D5,$F1,$7C,$34,$8F
	.dc.b	$03,$67,$FC,$C7,$F1,$4F,$99,$FF,$5C,$B6,$BF,$CB,$ED,$9D,$CE,$B8
	.dc.b	$A5,$69,$B3,$D3,$8F,$99,$27,$9B,$DB,$E5,$A4,$76,$6C,$F9,$3C,$FF
	.dc.b	$19,$27,$C6,$FF,$B3,$BE,$91,$E3,$52,$4F,$F9,$DA,$32,$FF,$9E,$3E
	.dc.b	$91,$E7,$3D,$EC,$37,$1F,$F4,$2C,$BE,$91,$E0,$6A,$FF,$D0,$3F,$53
	.dc.b	$FE,$AF,$BB,$BF,$FA,$11,$46,$5F,$F4,$7B,$5F,$F4,$B7,$D2,$3E,$97
	.dc.b	$83,$63,$FD,$1A,$D7,$FD,$2E,$D7,$FD,$3D,$EA,$3B,$7C,$0D,$8F,$F4
	.dc.b	$AB,$5F,$F4,$FB,$5F,$F5,$37,$D2,$FF,$BD,$93,$49,$BF,$AF,$E9,$1E
	.dc.b	$45,$25,$7F,$A8,$A3,$2F,$FA,$93,$E9,$3E,$12,$69,$1F,$8D,$F8,$7F
	.dc.b	$8D,$FD,$50,$70,$FE,$B5,$AE,$72,$F9,$D7,$FA,$C8,$FF,$59,$B5,$FF
	.dc.b	$5E,$B5,$FF,$62,$35,$EF,$4F,$87,$FB,$0D,$8F,$54,$60,$FB,$9F,$4A
	.dc.b	$F6,$29,$27,$FD,$87,$19,$7F,$D8,$BF,$4A,$F9,$9F,$C2,$36,$33,$18
	.dc.b	$3C,$3E,$6C,$8F,$94,$D7,$FD,$AF,$8B,$E9,$7C,$E4,$8F,$A0,$FE,$47
	.dc.b	$E9,$1F,$4B,$49,$3F,$ED,$88,$CB,$F5,$F9,$FE,$09,$1C,$1F,$6F,$C0
	.dc.b	$F8,$A4,$70,$7F,$B8,$BE,$51,$F5,$3F,$A1,$6C,$66,$30,$7C,$EF,$83
	.dc.b	$91,$F8,$76,$BE,$EF,$47,$E9,$91,$FC,$63,$FA,$DF,$A4,$7D,$2D,$24
	.dc.b	$FF,$BD,$23,$2F,$E4,$FA,$FE,$E9,$1C,$1F,$EF,$AF,$A5,$FC,$69,$3A
	.dc.b	$3F,$74,$FE,$43,$E9,$5F,$05,$FB,$D6,$D7,$E6,$0F,$CC,$B8,$70,$1C
	.dc.b	$3B,$E3,$7A,$7F,$90,$24,$7E,$A5,$FC,$EF,$74,$8F,$CE,$B5,$EB,$F4
	.dc.b	$7F,$E8,$47,$B8,$70,$EB,$DA,$FE,$29,$18,$8F,$47,$9E,$F9,$BF,$E3
	.dc.b	$0D,$8F,$E7,$F1,$83,$F8,$E7,$D1,$3D,$2A,$49,$F0,$C6,$5B,$F3,$3C
	.dc.b	$08,$E0,$DF,$CD,$FF,$00,$F6,$DA,$7F,$02,$CB,$D3,$18,$AF,$ED,$D1
	.dc.b	$89,$FE,$0F,$82,$F9,$46,$1B,$5F,$DD,$67,$4F,$F7,$7B,$E8,$A3,$2A
	.dc.b	$33,$0F,$88,$E7,$2F,$A1,$7C,$EE,$3E,$89,$EF,$D3,$73,$AA,$F0,$67
	.dc.b	$FC,$7A,$1F,$11,$AC,$25,$68,$FF,$06,$4E,$9C,$0F,$83,$FE,$2C,$F9
	.dc.b	$EF,$F9,$4F,$C2,$F9,$E3,$6D,$74,$7F,$85,$E7,$8E,$BF,$09,$FC,$04
	.dc.b	$68,$ED,$E6,$7C,$08,$DD,$FF,$36,$77,$5F,$2B,$E8,$5F,$17,$F1,$B1
	.dc.b	$FC,$16,$3C,$DB,$5B,$9F,$97,$4F,$A2,$B1,$2F,$DF,$39,$6F,$8B,$C2
	.dc.b	$A4,$60,$DE,$BE,$1F,$8E,$91,$83,$9E,$FE,$09,$91,$83,$BF,$C6,$FE
	.dc.b	$05,$FD,$CE,$28,$C4,$7E,$FE,$31,$9F,$EC,$1B,$1C,$3E,$65,$F4,$59
	.dc.b	$6C,$66,$30,$7F,$66,$F9,$59,$1F,$E0,$96,$BF,$C2,$BD,$DF,$D1,$61
	.dc.b	$F9,$88,$FF,$09,$F1,$7E,$11,$FF,$29,$57,$F9,$4F,$E8,$BF,$0D,$1F
	.dc.b	$DB,$2F,$8D,$7F,$17,$E4,$B3,$FD,$C1,$AF,$F9,$6E,$6B,$FD,$96,$7B
	.dc.b	$77,$60,$DF,$23,$E2,$31,$3E,$3F,$07,$89,$8B,$FC,$61,$FE,$D7,$2F
	.dc.b	$A7,$D8,$DC,$FA,$DF,$1E,$AE,$6A,$78,$3D,$4B,$5F,$A4,$9F,$D9,$B7
	.dc.b	$93,$38,$7E,$6E,$72,$FE,$77,$E8,$BE,$4B,$76,$F5,$F8,$1F,$0A,$73
	.dc.b	$5F,$C1,$9F,$85,$5C,$78,$35,$7B,$78,$7A,$5C,$7E,$1E,$E7,$FB,$8C
	.dc.b	$11,$98,$C3,$AC,$3C,$8F,$87,$D1,$93,$FC,$03,$C5,$FD,$6B,$CE,$3C
	.dc.b	$DF,$1B,$23,$07,$77,$D5,$E9,$48,$C1,$E3,$7E,$FF,$D0,$BE,$8A,$FC
	.dc.b	$9E,$1B,$5F,$C4,$2F,$E0,$F0,$7F,$28,$7E,$D3,$77,$FC,$4C,$62,$7F
	.dc.b	$C7,$7F,$0A,$D3,$F1,$AB,$C2,$B1,$87,$F2,$3F,$45,$F5,$B1,$FE,$6A
	.dc.b	$FE,$C6,$FF,$90,$B9,$7F,$C8,$EE,$78,$3F,$8C,$2F,$A1,$FC,$DF,$D7
	.dc.b	$7F,$AD,$6B,$8C,$62,$BF,$3F,$F4,$7B,$9F,$E5,$AB,$EA,$33,$5A,$7F
	.dc.b	$4B,$F4,$9F,$D3,$EB,$F9,$9B,$FE,$04,$6F,$4E,$BD,$7F,$D5,$FD,$23
	.dc.b	$C7,$BF,$37,$86,$DF,$F3,$C3,$EE,$E9,$FC,$C5,$FE,$48,$67,$CD,$FD
	.dc.b	$AB,$88,$FE,$6D,$F4,$9F,$E9,$58,$FE,$8A,$BE,$7F,$F7,$6E,$67,$71
	.dc.b	$8F,$A4,$CF,$AB,$6F,$8B,$FC,$0B,$E9,$1E,$15,$F0,$70,$DA,$FE,$A7
	.dc.b	$F4,$9F,$D4,$DB,$F3,$3F,$C2,$A3,$19,$FF,$4B,$7F,$D9,$1D,$3F,$DC
	.dc.b	$1F,$EC,$F5,$CB,$E9,$F8,$7B,$1C,$5F,$E2,$7E,$8F,$FA,$E3,$0F,$EA
	.dc.b	$9F,$6F,$FE,$B7,$BF,$EC,$73,$9C,$7F,$BD,$CE,$6B,$FA,$D7,$D3,$7C
	.dc.b	$C9,$FE,$FE,$FF,$0B,$BF,$ED,$6E,$5F,$F4,$8B,$99,$E3,$F4,$5F,$C9
	.dc.b	$23,$FC,$D7,$3C,$23,$1F,$19,$FC,$A3,$CC,$7A,$17,$DB,$C3,$6F,$FB
	.dc.b	$CF,$D3,$3C,$EF,$E5,$ED,$8C,$FC,$F6,$26,$47,$0F,$2A,$EB,$FD,$D6
	.dc.b	$D1,$C8,$FE,$89,$01,$1A,$9F,$F2,$2B,$63,$FC,$AA,$D7,$FC,$B4,$FF
	.dc.b	$B2,$E3,$7F,$11,$D2,$79,$B6,$75,$1F,$E5,$EF,$82,$E9,$C1,$FE,$2C
	.dc.b	$6D,$FF,$AF,$2B,$73,$9C,$3F,$A7,$FD,$2B,$EA,$29,$8B,$A7,$F8,$82
	.dc.b	$34,$FF,$97,$9B,$8F,$F9,$B9,$AF,$F9,$CD,$AF,$FB,$31,$FD,$24,$71
	.dc.b	$EE,$F6,$46,$27,$94,$60,$F2,$FD,$17,$99,$B5,$E3,$8E,$24,$7C,$8F
	.dc.b	$4F,$A2,$91,$E7,$6D,$7C,$7F,$57,$FB,$7E,$A7,$E1,$CE,$1F,$8B,$F2
	.dc.b	$D2,$FE,$17,$A1,$70,$FF,$1B,$67,$FE,$D3,$9F,$89,$5E,$C2,$3F,$1F
	.dc.b	$C9,$F7,$91,$C1,$2B,$9F,$D1,$7D,$49,$FA,$AC,$9F,$A4,$E5,$A7,$F9
	.dc.b	$53,$E9,$BF,$8C,$D7,$FA,$8D,$AF,$FB,$F1,$AF,$E5,$5A,$9F,$D9,$3F
	.dc.b	$E7,$EF,$A9,$FF,$09,$C6,$21,$AA,$73,$F0,$B2,$FC,$5F,$0D,$6F,$FC
	.dc.b	$B5,$87,$F4,$57,$C9,$5F,$9B,$8C,$47,$96,$31,$ED,$78,$3E,$0B,$C2
	.dc.b	$DC,$3D,$D1,$C4,$8F,$FA,$C2,$D1,$8B,$FF,$35,$64,$F9,$1E,$51,$87
	.dc.b	$DB,$F3,$5F,$82,$EA,$B8,$FF,$9F,$AF,$F4,$EF,$C9,$9D,$3F,$3D,$B7
	.dc.b	$DB,$C6,$FC,$59,$FE,$3A,$FA,$7F,$F0,$CD,$7F,$AA,$E3,$2F,$57,$5E
	.dc.b	$0E,$2B,$FD,$01,$5F,$CE,$3B,$3F,$D4,$AD,$BF,$E1,$92,$8F,$E3,$F1
	.dc.b	$FE,$94,$75,$1F,$C4,$5A,$EF,$C1,$F2,$78,$3E,$56,$57,$17,$2D,$24
	.dc.b	$F9,$23,$2D,$8C,$DF,$9A,$72,$D8,$C9,$E8,$BF,$19,$E8,$AE,$44,$9E
	.dc.b	$0F,$C2,$BF,$2B,$9A,$E5,$7D,$9A,$76,$8F,$81,$19,$6C,$65,$F8,$4F
	.dc.b	$C7,$9E,$B5,$E0,$24,$79,$E3,$2F,$69,$F1,$1F,$36,$32,$F6,$32,$7D
	.dc.b	$57,$2F,$80,$CD,$FC,$1A,$EF,$1C,$5A,$F1,$AF,$44,$FC,$8B,$F6,$4F
	.dc.b	$3A,$DC,$FC,$CB,$EE,$EE,$FD,$C7,$2B,$CC,$E9,$B1,$93,$BC,$65,$A6
	.dc.b	$6F,$D3,$19,$6C,$65,$FC,$A9,$EF,$2D,$7F,$84,$9B,$5F,$D6,$E3,$11
	.dc.b	$F3,$4C,$CF,$27,$E6,$18,$3F,$77,$F2,$1F,$33,$43,$49,$3F,$7E,$32
	.dc.b	$D8,$CC,$FD,$69,$CB,$63,$38,$FD,$29,$CB,$63,$35,$F8,$38,$FE,$DD
	.dc.b	$18,$AF,$B5,$3F,$B6,$AE,$2E,$CC,$4E,$E7,$ED,$46,$0F,$31,$1F,$23
	.dc.b	$6B,$C4,$D3,$A4,$7F,$83,$EB,$BB,$E2,$31,$5D,$EF,$F8,$1C,$EA,$7C
	.dc.b	$26,$AB,$A3,$C4,$FE,$7F,$5C,$70,$FE,$CE,$D9,$FE,$4C,$D9,$FA,$B1
	.dc.b	$C9,$EA,$72,$71,$7E,$29,$E1,$87,$CA,$9C,$DF,$66,$9D,$8F,$E3,$DE
	.dc.b	$67,$F7,$8F,$27,$F9,$73,$63,$27,$F3,$98,$CB,$4F,$F1,$A5,$1F,$E5
	.dc.b	$7F,$57,$5C,$98,$3F,$9D,$76,$FF,$BD,$23,$59,$AB,$93,$FE,$3F,$8C
	.dc.b	$C6,$21,$18,$AF,$E4,$55,$CD,$FF,$7F,$FD,$87,$F8,$86,$F8,$36,$3F
	.dc.b	$94,$67,$AF,$06,$47,$E9,$71,$57,$F8,$A5,$FD,$8B,$5F,$E8,$ED,$7F
	.dc.b	$CA,$9C,$5F,$E9,$37,$F6,$ED,$7F,$A6,$B5,$DF,$D2,$7F,$94,$78,$3F
	.dc.b	$6F,$1F,$8F,$DF,$04,$C3,$C9,$6F,$A6,$FF,$2C,$36,$33,$18,$33,$94
	.dc.b	$8E,$2D,$BF,$E0,$5B,$78,$DF,$97,$97,$F9,$8E,$30,$67,$07,$87,$E0
	.dc.b	$26,$1E,$47,$D5,$E7,$C8,$F9,$5A,$23,$18,$F8,$33,$EC,$4A,$FA,$18
	.dc.b	$FA,$E7,$FC,$E1,$C5,$F1,$49,$3D,$1F,$36,$92,$38,$B6,$7F,$CF,$1E
	.dc.b	$4F,$CD,$24,$F9,$1F,$76,$72,$3C,$2D,$9F,$E4,$DF,$4D,$F3,$FE,$0B
	.dc.b	$F0,$8D,$8C,$EE,$E8,$CE,$AB,$FC,$57,$D3,$7F,$B3,$AE,$FA,$7F,$8D
	.dc.b	$51,$97,$F9,$A3,$5F,$E7,$EB,$6E,$0D,$8E,$B1,$83,$A7,$6B,$B6,$92
	.dc.b	$7F,$39,$19,$7F,$3D,$F4,$AF,$23,$C8,$D8,$CC,$60,$F4,$7D,$23,$9E
	.dc.b	$92,$7F,$51,$19,$7F,$55,$F4,$BF,$EA,$16,$F9,$BF,$06,$31,$1F,$D4
	.dc.b	$DA,$FF,$5E,$FA,$5F,$33,$FD,$5D,$AE,$73,$F9,$51,$88,$FF,$1C,$FC
	.dc.b	$DF,$D3,$7F,$74,$D7,$FB,$03,$5F,$ED,$6D,$7F,$6B,$F2,$3F,$57,$23
	.dc.b	$49,$1D,$3E,$2B,$FC,$8F,$E0,$FF,$13,$B4,$63,$A7,$C6,$DA,$FE,$0B
	.dc.b	$F4,$15,$7D,$2B,$D1,$C3,$63,$F9,$39,$FE,$DF,$70,$FF,$25,$BF,$E1
	.dc.b	$7F,$55,$CF,$E9,$9F,$27,$49,$5C,$84,$3E,$3F,$C5,$A7,$FB,$F2,$74
	.dc.b	$D7,$B1,$FE,$EF,$71,$AF,$F3,$1D,$DE,$BF,$2F,$A6,$97,$C9,$F7,$4B
	.dc.b	$E7,$FE,$07,$D3,$3D,$EA,$4B,$F3,$43,$E3,$FC,$BE,$04,$9F,$2B,$D3
	.dc.b	$7F,$4F,$D3,$49,$F4,$BE,$0D,$FE,$77,$CC,$39,$A9,$2B,$E2,$43,$8F
	.dc.b	$E1,$FC,$0C,$9F,$86,$FA,$BD,$FF,$8F,$E9,$C9,$F8,$EF,$BB,$A7,$EE
	.dc.b	$7D,$2B,$E9,$A9,$2B,$E9,$43,$E3,$3E,$8F,$8D,$27,$E8,$BF,$23,$E3
	.dc.b	$3F,$0D,$F4,$DF,$A2,$EB,$FE,$C0,$DE,$EF,$F4,$BB,$6F,$FA,$8F,$C9
	.dc.b	$7E,$37,$E9,$9F,$0D,$49,$3F,$83,$87,$C3,$7E,$0B,$DC,$93,$F8,$27
	.dc.b	$EC,$FC,$7F,$DA,$CA,$4F,$E2,$FC,$0F,$EE,$5A,$7E,$0F,$2F,$CE,$7D
	.dc.b	$33,$E7,$A9,$2F,$F2,$51,$F8,$AC,$3F,$C7,$5F,$3F,$FA,$3F,$78,$D3
	.dc.b	$F4,$DF,$57,$FA,$CB,$FC,$D7,$D3,$FE,$2D,$BE,$37,$EA,$E3,$11,$F9
	.dc.b	$80,$FE,$91,$E3,$E0,$6C,$8F,$4E,$CE,$5C,$9E,$3D,$64,$F8,$23,$B7
	.dc.b	$0E,$A3,$07,$91,$FD,$E7,$D3,$3E,$8B,$F7,$DF,$4E,$F8,$9A,$6A,$37
	.dc.b	$F1,$D7,$0F,$DB,$5F,$16,$C7,$F0,$7F,$86,$F8,$51,$ED,$F8,$B6,$5F
	.dc.b	$C2,$DB,$1F,$BF,$9D,$F3,$FF,$89,$DF,$9B,$19,$F4,$FA,$1D,$79,$3F
	.dc.b	$8A,$7D,$33,$E6,$7A,$72,$CB,$FE,$15,$39,$6F,$D3,$F0,$A9,$18,$37
	.dc.b	$E5,$74,$99,$18,$39,$F0,$7A,$79,$64,$60,$EF,$C1,$E4,$99,$18,$3C
	.dc.b	$7F,$23,$FE,$47,$7F,$D3,$99,$23,$11,$FE,$A8,$8C,$6F,$FE,$3D,$6C
	.dc.b	$70,$F9,$B7,$D3,$E5,$B1,$98,$C1,$FD,$E5,$7C,$D5,$23,$FD,$F5,$FE
	.dc.b	$FD,$F0,$FF,$E5,$AC,$3F,$CB,$13,$FE,$07,$EF,$FE,$68,$8E,$63,$E9
	.dc.b	$DC,$BF,$EC,$0B,$E5,$5F,$EC,$2E,$5B,$7C,$4F,$EA,$31,$D2,$73,$1D
	.dc.b	$66,$BF,$AE,$9F,$0F,$2E,$B9,$BF,$AF,$C6,$0F,$2F,$67,$03,$8C,$D8
	.dc.b	$F4,$FC,$9F,$43,$17,$FB,$13,$FD,$E5,$1F,$D9,$B9,$BC,$7A,$9F,$07
	.dc.b	$A9,$87,$3C,$92,$BF,$AC,$56,$DB,$1E,$CF,$A7,$F9,$58,$7E,$FD,$B5
	.dc.b	$FE,$85,$D1,$DF,$BD,$F8,$6F,$32,$FE,$C9,$87,$07,$E5,$3F,$2B,$1E
	.dc.b	$9E,$B7,$7F,$87,$F0,$36,$B8,$C5,$9D,$C6,$0E,$BE,$A7,$9C,$FA,$7E
	.dc.b	$1F,$EB,$63,$27,$8B,$E6,$3D,$86,$DC,$1E,$6F,$57,$E0,$1A,$70,$7F
	.dc.b	$7D,$FA,$76,$FF,$80,$6C,$67,$D5,$C4,$FE,$12,$FE,$AE,$5F,$91,$6A
	.dc.b	$7F,$23,$5F,$E0,$97,$23,$2E,$47,$36,$C7,$F8,$3F,$E9,$9D,$BF,$92
	.dc.b	$5F,$4F,$63,$0F,$E5,$BE,$9F,$E9,$63,$FA,$37,$E9,$72,$FF,$AF,$DC
	.dc.b	$C1,$CE,$1F,$A1,$5C,$3C,$F8,$7A,$36,$7F,$C4,$5F,$4B,$FD,$81,$FF
	.dc.b	$12,$7D,$2F,$F9,$DD,$CF,$A9,$FA,$88,$C6,$3F,$D2,$D7,$CE,$E7,$87
	.dc.b	$4D,$BE,$7F,$4D,$FB,$0E,$5F,$DE,$46,$0F,$F1,$93,$8C,$3F,$AB,$7B
	.dc.b	$7F,$EA,$C9,$FF,$1B,$36,$33,$F1,$3F,$BA,$8C,$70,$7F,$B8,$5F,$EF
	.dc.b	$B5,$C3,$D6,$D6,$BB,$7F,$80,$FD,$87,$F5,$39,$FF,$21,$37,$1F,$EC
	.dc.b	$CB,$CB,$DF,$81,$FC,$23,$E9,$7E,$5D,$FF,$92,$5F,$F4,$93,$78,$F5
	.dc.b	$C9,$FC,$3B,$EC,$3F,$DE,$59,$7F,$1B,$5F,$21,$9B,$D3,$BF,$84,$E1
	.dc.b	$F0,$96,$7F,$CA,$D5,$FE,$33,$7F,$CB,$1F,$4B,$E7,$E2,$FF,$80,$0D
	.dc.b	$8F,$E4,$0F,$A5,$DE,$BF,$C8,$23,$FC,$95,$1C,$BE,$1F,$1C,$D9,$7F
	.dc.b	$92,$FD,$87,$E7,$B5,$FF,$04,$37,$15,$FF,$2D,$87,$C5,$F0,$F3,$FF
	.dc.b	$96,$FD,$2B,$E1,$E9,$FC,$C3,$EC,$3F,$55,$87,$F9,$C6,$74,$D9,$E3
	.dc.b	$F2,$7C,$23,$4E,$1F,$E7,$99,$FF,$3D,$AF,$F5,$E4,$62,$3F,$A4,$46
	.dc.b	$1C,$35,$E3,$F2,$DE,$A9,$E1,$1F,$EB,$F7,$FA,$9B,$FD,$46,$34,$F5
	.dc.b	$6B,$CB,$93,$FA,$5C,$7D,$C9,$FF,$46,$3F,$80,$9F,$F6,$54,$EB,$E7
	.dc.b	$E5,$6B,$E6,$ED,$7F,$8D,$6D,$5E,$28,$FF,$69,$36,$33,$5E,$86,$CF
	.dc.b	$97,$97,$44,$9D,$36,$3B,$7C,$97,$A2,$3D,$2C,$9E,$8F,$72,$3D,$FD
	.dc.b	$DF,$11,$1C,$1E,$3F,$8F,$F8,$67,$C4,$8B,$1F,$13,$BF,$E3,$9F,$23
	.dc.b	$E0,$AC,$64,$F6,$47,$BC,$8F,$D6,$FA,$5F,$AD,$97,$DF,$6B,$F8,$46
	.dc.b	$BF,$E0,$16,$BF,$E0,$B7,$8F,$37,$E6,$A3,$1F,$05,$FE,$04,$9C,$D7
	.dc.b	$18,$C1,$1F,$F6,$44,$6A,$78,$C6,$3C,$1F,$B1,$3F,$B1,$9F,$34,$60
	.dc.b	$93,$CB,$E0,$7C,$11,$E1,$7C,$5E,$44,$9E,$FF,$55,$6F,$FC,$F5,$87
	.dc.b	$F3,$09,$F8,$95,$E0,$23,$F9,$CF,$B0,$7D,$87,$B7,$EA,$BF,$C4,$31
	.dc.b	$89,$F5,$C6,$5F,$9C,$FA,$88,$E0,$F8,$3F,$1F,$BC,$7C,$6F,$F8,$B3
	.dc.b	$EA,$7F,$9D,$6F,$F3,$65,$1F,$D5,$8C,$D7,$F8,$D5,$C4,$71,$F7,$79
	.dc.b	$36,$7F,$DE,$D3,$FD,$C8,$9F,$07,$DF,$E0,$A4,$E9,$F1,$46,$2B,$FD
	.dc.b	$41,$5F,$CE,$1B,$E5,$FC,$2F,$2B,$D0,$CC,$3B,$B9,$69,$27,$8B,$FE
	.dc.b	$55,$CB,$DD,$AF,$FB,$AE,$35,$1F,$07,$0F,$03,$6F,$A5,$F2,$24,$66
	.dc.b	$BA,$1F,$E5,$9B,$E9,$3A,$8F,$91,$5E,$78,$FF,$5E,$4E,$AB,$94,$6C
	.dc.b	$FF,$35,$5F,$8A,$FA,$61,$C6,$B6,$48,$C9,$EA,$3B,$96,$BF,$CE,$97
	.dc.b	$DE,$33,$1D,$CF,$BF,$87,$BA,$BA,$47,$8A,$3C,$13,$DA,$38,$61,$E5
	.dc.b	$75,$1F,$09,$EF,$87,$07,$27,$7C,$3C,$4E,$30,$EE,$F8,$AB,$92,$9D
	.dc.b	$B9,$3C,$38,$79,$E7,$D7,$5C,$B0,$F2,$57,$1C,$3C,$4F,$19,$EF,$87
	.dc.b	$4A,$D9,$30,$E9,$5E,$5C,$3A,$91,$F6,$BF,$35,$E8,$E5,$F0,$39,$9E
	.dc.b	$8F,$B2,$BC,$15,$E4,$9F,$3C,$62,$78,$4F,$8A,$BA,$9E,$32,$1D,$8F
	.dc.b	$87,$1F,$D9,$2B,$F5,$11,$FD,$C1,$AF,$E0,$0C,$4F,$F5,$78,$CC,$7D
	.dc.b	$63,$33,$CE,$7F,$83,$79,$7E,$72,$47,$B5,$B3,$F8,$46,$E1,$F4,$DB
	.dc.b	$3F,$16,$FF,$C6,$35,$B8,$E4,$47,$04,$AE,$51,$96,$C6,$63,$A5,$76
	.dc.b	$8F,$8B,$5D,$AB,$64,$AD,$9C,$CB,$3C,$CE,$35,$CD,$FA,$27,$F2,$47
	.dc.b	$15,$F8,$86,$DF,$C6,$AE,$95,$C9,$AF,$2A,$F2,$47,$82,$FC,$93,$B3
	.dc.b	$C4,$6E,$FA,$CF,$F2,$C6,$C7,$F2,$67,$F9,$D7,$B3,$FC,$EF,$D2,$B5
	.dc.b	$FC,$C9,$B5,$FC,$B1,$B7,$FC,$7D,$FE,$85,$F1,$72,$BF,$9C,$F8,$96
	.dc.b	$FF,$99,$CD,$BF,$E4,$D3,$6B,$FD,$F1,$F6,$1F,$E2,$BB,$57,$F4,$B8
	.dc.b	$C3,$FD,$37,$E9,$FF,$D3,$13,$AA,$E3,$E2,$FF,$BF,$AF,$94,$FF,$92
	.dc.b	$1A,$EA,$7F,$C8,$31,$88,$07,$FB,$33,$FE,$4D,$6B,$CF,$9B,$FC,$6D
	.dc.b	$18,$8E,$23,$E4,$7F,$CC,$B5,$FE,$4E,$6C,$7F,$95,$DA,$FF,$97,$1A
	.dc.b	$FC,$2F,$A6,$EA,$49,$F3,$BE,$0F,$0E,$47,$A9,$B3,$E7,$F8,$EF,$29
	.dc.b	$27,$DE,$FF,$93,$7A,$7E,$4C,$FF,$9B,$1A,$FA,$9A,$FC,$46,$BE,$DE
	.dc.b	$4F,$F2,$B6,$3F,$C0,$23,$17,$EF,$9C,$F1,$C8,$F8,$4D,$9F,$85,$F3
	.dc.b	$3C,$89,$3F,$1D,$EB,$F4,$F2,$3E,$53,$67,$FD,$01,$F6,$1E,$42,$7B
	.dc.b	$3F,$E6,$97,$CF,$F5,$0F,$35,$F4,$FF,$E8,$08,$C3,$7E,$A1,$E0,$E6
	.dc.b	$B7,$8F,$FA,$37,$E9,$EB,$6F,$FD,$12,$A3,$2F,$FA,$59,$AF,$E1,$17
	.dc.b	$D8,$70,$6C,$7F,$A4,$9A,$F7,$8E,$BE,$9B,$B2,$92,$7F,$D3,$31,$97
	.dc.b	$FD,$37,$F4,$FF,$E8,$94,$D2,$70,$7A,$3E,$A1,$CD,$49,$5F,$EA,$48
	.dc.b	$CB,$FE,$A6,$FA,$5F,$7A,$69,$1F,$87,$F5,$1E,$37,$FD,$68,$D7,$FD
	.dc.b	$72,$D7,$F4,$ED,$7F,$63,$FA,$FF,$C8,$6E,$DF,$3A,$31,$06,$FD,$F7
	.dc.b	$BC,$99,$7D,$47,$E1,$47,$CC,$9F,$F6,$4B,$5F,$DB,$B5,$F0,$ED,$DF
	.dc.b	$D7,$FD,$DF,$87,$FE,$5B,$A4,$63,$0E,$2D,$BF,$4F,$A5,$5F,$37,$CE
	.dc.b	$5B,$34,$C7,$CB,$67,$E0,$E5,$7F,$86,$29,$1C,$7D,$5E,$5E,$AF,$3F
	.dc.b	$8E,$97,$B8,$ED,$F1,$DD,$FD,$79,$3C,$E3,$C5,$EB,$78,$FE,$2E,$4F
	.dc.b	$78,$F3,$7C,$8C,$8F,$13,$E9,$F8,$BF,$57,$B3,$27,$CC,$FB,$7A,$FD
	.dc.b	$DF,$51,$27,$D4,$FC,$1F,$87,$F8,$5F,$11,$F1,$7F,$60,$FF,$7D,$F4
	.dc.b	$FF,$A1,$DB,$D3,$F5,$27,$5E,$2E,$FE,$5E,$65,$FE,$3C,$BF,$DB,$B7
	.dc.b	$E6,$BA,$7B,$D2,$30,$6F,$DF,$F0,$7C,$4C,$8C,$1C,$FE,$2B,$C9,$F5
	.dc.b	$B2,$30,$77,$F5,$3D,$1E,$A4,$8C,$1E,$3F,$53,$D8,$47,$CD,$EA,$7B
	.dc.b	$FE,$5A,$47,$A5,$F8,$5F,$01,$F0,$FE,$02,$4F,$B5,$F8,$DF,$61,$F8
	.dc.b	$BF,$65,$C3,$49,$5F,$E5,$EF,$A4,$72,$7E,$47,$E9,$1E,$E5,$25,$7F
	.dc.b	$99,$21,$F3,$DB,$F9,$C9,$39,$79,$7C,$67,$3F,$9E,$F1,$74,$CA,$E5
	.dc.b	$1D,$FE,$37,$51,$83,$B3,$FA,$3F,$A3,$FC,$DC,$7F,$9D,$3E,$91,$F1
	.dc.b	$5F,$A7,$6C,$7F,$9E,$27,$6F,$F7,$D3,$FB,$AA,$F3,$FA,$7A,$12,$78
	.dc.b	$47,$EC,$7B,$BE,$1C,$7F,$A0,$BC,$C6,$5F,$B5,$6C,$7F,$A1,$A7,$6F
	.dc.b	$70,$FD,$07,$5C,$5D,$4E,$DC,$9F,$BF,$FA,$47,$AB,$D7,$96,$5F,$FA
	.dc.b	$4A,$72,$DF,$57,$85,$48,$C1,$BF,$57,$A4,$C8,$C1,$CF,$6F,$57,$24
	.dc.b	$8C,$1D,$F7,$F2,$4C,$8C,$1E,$3F,$4F,$FC,$5A,$FF,$86,$0F,$F1,$0E
	.dc.b	$48,$CC,$66,$31,$E5,$6E,$37,$3A,$7F,$C4,$32,$F5,$1F,$49,$B9,$4E
	.dc.b	$63,$0D,$8E,$8D,$78,$7D,$27,$F5,$02,$3F,$DC,$1F,$EE,$3E,$5B,$CD
	.dc.b	$FF,$5F,$CF,$F7,$A6,$A9,$FF,$1E,$FD,$27,$DC,$8F,$F1,$BD,$71,$7F
	.dc.b	$00,$7E,$AA,$B9,$F2,$67,$4D,$4F,$F9,$1A,$B1,$5F,$26,$AC,$FF,$90
	.dc.b	$B9,$2B,$FE,$4C,$6F,$33,$E7,$B3,$3F,$BC,$BF,$4F,$6E,$0C,$DD,$8F
	.dc.b	$6F,$9B,$C0,$C5,$7A,$1E,$F9,$B9,$A9,$E0,$F0,$2D,$FF,$B9,$27,$FA
	.dc.b	$6B,$79,$33,$87,$E8,$67,$2F,$E8,$FE,$93,$E6,$37,$7F,$5F,$81,$EE
	.dc.b	$9C,$D7,$C6,$7E,$55,$71,$E0,$D5,$ED,$E1,$E6,$71,$F3,$2E,$7F,$EF
	.dc.b	$8C,$3F,$D3,$B8,$7F,$9D,$1A,$9A,$7F,$BD,$8B,$E5,$78,$CF,$A3,$E2
	.dc.b	$7E,$D2,$BA,$47,$9F,$CA,$91,$83,$C0,$FA,$FC,$A9,$18,$3C,$8F,$DC
	.dc.b	$FA,$47,$3D,$31,$EB,$DA,$FF,$45,$F0,$7E,$23,$07,$2F,$E1,$1B,$3A
	.dc.b	$3F,$D1,$2F,$F1,$07,$1A,$FF,$A5,$23,$6F,$F1,$55,$9F,$33,$A3,$67
	.dc.b	$8C,$BE,$93,$F6,$91,$85,$ED,$7E,$59,$CC,$FF,$21,$73,$3C,$BE,$93
	.dc.b	$EE,$BF,$E9,$E7,$EE,$E5,$FA,$47,$35,$FE,$AA,$73,$3B,$E2,$FD,$1C
	.dc.b	$7D,$A0,$E3,$8B,$F4,$FF,$49,$FE,$BC,$C3,$F5,$FF,$49,$FE,$78,$C3
	.dc.b	$D7,$39,$7F,$D5,$F1,$9C,$7F,$9C,$2F,$77,$F6,$2F,$D6,$9F,$86,$E5
	.dc.b	$BC,$3A,$F6,$3F,$6F,$C9,$FE,$C2,$9D,$C6,$30,$FD,$DF,$BA,$F9,$1A
	.dc.b	$63,$C1,$B7,$F4,$25,$F3,$5C,$0E,$3F,$23,$67,$EA,$FD,$27,$EB,$B8
	.dc.b	$3F,$86,$B6,$3F,$D9,$6F,$F5,$F7,$7E,$07,$F0,$CF,$A4,$F9,$31,$F7
	.dc.b	$9F,$D2,$98,$D3,$FA,$D3,$FA,$8F,$A4,$FE,$4D,$3F,$84,$7F,$DD,$6F
	.dc.b	$FB,$61,$CC,$FF,$61,$72,$FF,$B9,$17,$CE,$7F,$1C,$5F,$3F,$63,$0F
	.dc.b	$F2,$0F,$A4,$F5,$57,$FB,$B1,$A9,$F6,$3F,$DB,$DC,$8F,$83,$E4,$73
	.dc.b	$E2,$B8,$FE,$73,$E9,$3F,$4D,$7F,$DD,$DB,$B7,$F3,$67,$FD,$DE,$DF
	.dc.b	$5F,$F9,$9F,$D1,$BE,$4F,$F9,$DD,$FF,$BE,$23,$18,$7F,$57,$6C,$7F
	.dc.b	$5F,$6B,$FD,$9D,$FF,$5A,$46,$FE,$53,$9C,$84,$E9,$E5,$1F,$EE,$8F
	.dc.b	$0B,$FA,$2F,$C0,$9F,$2C,$61,$FE,$93,$F1,$BF,$4A,$BF,$D2,$3F,$3B
	.dc.b	$8C,$72,$C7,$FB,$74,$A3,$C7,$3C,$27,$31,$89,$7D,$1D,$24,$F2,$6D
	.dc.b	$7F,$88,$E7,$F4,$13,$FC,$31,$E2,$DE,$CF,$3F,$1F,$A3,$E7,$2C,$ED
	.dc.b	$C3,$E6,$7C,$44,$7F,$C0,$DA,$3B,$BE,$29,$C3,$1F,$F1,$A4,$66,$7C
	.dc.b	$5F,$47,$F1,$8F,$8F,$EE,$D8,$C9,$F1,$23,$E0,$11,$F8,$FD,$DF,$0C
	.dc.b	$8E,$09,$7F,$07,$BB,$E9,$CF,$D4,$7F,$BE,$FD,$1F,$FB,$12,$BF,$97
	.dc.b	$BF,$D3,$EB,$FC,$94,$E5,$BF,$01,$C0,$FA,$B1,$B8,$FF,$19,$35,$FC
	.dc.b	$50,$70,$FE,$35,$ED,$CF,$FE,$14,$CD,$F1,$1D,$70,$FE,$83,$18,$7C
	.dc.b	$BE,$FF,$A0,$F4,$FC,$05,$8C,$93,$1F,$4C,$65,$F5,$F9,$1F,$03,$C7
	.dc.b	$F8,$2D,$AF,$83,$1E,$C2,$3F,$E4,$1C,$FF,$99,$57,$F9,$25,$7C,$7F
	.dc.b	$CA,$3E,$5F,$C8,$5A,$F9,$71,$F1,$88,$F3,$6D,$7E,$13,$AF,$83,$67
	.dc.b	$FB,$0F,$D1,$7F,$54,$86,$BF,$E9,$09,$CF,$B3,$FB,$2F,$89,$91,$F2
	.dc.b	$1F,$F1,$CD,$F1,$AE,$17,$FE,$85,$8F,$E1,$F5,$F5,$9B,$1F,$C0,$5F
	.dc.b	$EE,$FC,$3F,$E4,$87,$30,$69,$3D,$5E,$FE,$17,$83,$D0,$98,$7A,$BD
	.dc.b	$1B,$87,$86,$6E,$3D,$66,$D7,$8E,$6D,$78,$E6,$BE,$18,$B8,$FF,$47
	.dc.b	$BF,$34,$60,$F8,$35,$6F,$E5,$C6,$1A,$4B,$F2,$3E,$73,$15,$D1,$A7
	.dc.b	$C7,$AE,$93,$C0,$F9,$8E,$27,$84,$62,$78,$BF,$36,$BD,$35,$B9,$F8
	.dc.b	$07,$86,$7A,$57,$22,$47,$84,$E7,$1F,$0D,$B7,$F0,$DE,$71,$C9,$B3
	.dc.b	$F3,$DD,$DF,$89,$A7,$2B,$F2,$57,$9E,$3C,$95,$E2,$8C,$1D,$4F,$AD
	.dc.b	$1C,$9C,$1F,$87,$9E,$6F,$9C,$F8,$86,$AF,$FC,$CA,$48,$DB,$67,$A5
	.dc.b	$78,$6B,$A4,$77,$7F,$D8,$52,$BF,$E9,$2D,$9F,$F5,$D3,$67,$FD,$84
	.dc.b	$D7,$8C,$F0,$24,$F0,$7F,$AF,$FD,$87,$E4,$B3,$34,$93,$F5,$23,$2D
	.dc.b	$8C,$D7,$C4,$9C,$B6,$33,$87,$B6,$72,$D8,$CD,$7D,$A3,$F9,$6C,$E9
	.dc.b	$C5,$7F,$79,$8C,$3F,$5A,$BF,$97,$57,$28,$C5,$72,$8C,$47,$D5,$9D
	.dc.b	$CF,$A4,$91,$E7,$6C,$79,$5A,$78,$A3,$F8,$BC,$F4,$C3,$FD,$43,$7E
	.dc.b	$49,$ED,$5E,$13,$8B,$E4,$3D,$55,$C7,$0F,$A3,$3A,$9E,$26,$9C,$5F
	.dc.b	$6A,$E1,$1F,$D2,$9B,$3F,$DC,$9B,$87,$BA,$3D,$84,$7D,$8D,$9F,$E4
	.dc.b	$DD,$9F,$EE,$4F,$A3,$FF,$27,$D7,$F2,$58,$C4,$FF,$30,$6B,$FC,$E9
	.dc.b	$B5,$FC,$C7,$E9,$FF,$2A,$49,$FC,$AB,$FE,$20,$FB,$0F,$F0,$AC,$AF
	.dc.b	$FD,$F3,$18,$7F,$9A,$4F,$FB,$DA,$31,$5F,$84,$DB,$FC,$6A,$7F,$BE
	.dc.b	$BE,$91,$F4,$B8,$9F,$ED,$B3,$FE,$2E,$DF,$FC,$A7,$97,$F7,$83,$FA
	.dc.b	$7C,$64,$FF,$19,$F2,$BE,$8A,$92,$F8,$57,$F9,$26,$75,$18,$86,$2F
	.dc.b	$5B,$FC,$7A,$D9,$D3,$67,$9B,$5F,$25,$7C,$46,$F9,$1E,$42,$57,$79
	.dc.b	$FF,$1F,$CF,$91,$AF,$2F,$1F,$D6,$7C,$1F,$95,$95,$FE,$3A,$6C,$F8
	.dc.b	$BE,$AF,$CE,$49,$F3,$BF,$27,$9A,$47,$F9,$09,$B3,$F1,$39,$3D,$0F
	.dc.b	$F8,$91,$AF,$F9,$BD,$A9,$BB,$BE,$57,$EB,$12,$FF,$C9,$EF,$F9,$9A
	.dc.b	$7D,$B5,$D7,$2C,$12,$BE,$97,$D2,$FD,$9F,$9F,$BC,$27,$3F,$4A,$EE
	.dc.b	$CC,$7D,$9F,$B0,$FF,$6F,$E5,$7B,$7E,$E2,$8C,$BF,$82,$6B,$F8,$BF
	.dc.b	$A6,$7D,$0F,$06,$C7,$E1,$5A,$FE,$29,$AF,$E4,$FD,$57,$87,$E0,$6C
	.dc.b	$7E,$35,$AF,$E4,$9A,$F8,$7D,$2F,$01,$1F,$03,$EA,$FA,$5F,$F1,$DC
	.dc.b	$7E,$5D,$AF,$E7,$1A,$FE,$95,$AF,$F9,$7B,$E9,$BF,$D3,$0F,$E7,$DA
	.dc.b	$FE,$91,$AF,$EB,$5A,$FF,$99,$BE,$97,$CE,$DC,$3E,$6C,$62,$3F,$D6
	.dc.b	$8D,$5F,$4D,$F5,$DF,$D7,$35,$FD,$9B,$5F,$DE,$35,$FF,$3A,$FD,$37
	.dc.b	$FA,$B9,$34,$9C,$9F,$7B,$E8,$FF,$DA,$13,$FB,$B6,$BF,$C0,$1A,$FF
	.dc.b	$09,$6B,$FE,$82,$7D,$04,$C3,$A1,$F8,$AF,$A6,$69,$FE,$88,$F1,$98
	.dc.b	$7F,$A2,$A7,$38,$7F,$B5,$67,$2F,$FB,$5E,$32,$FF,$27,$9E,$1E,$9F
	.dc.b	$37,$97,$CE,$59,$E0,$D5,$48,$C2,$F9,$5B,$F0,$D2,$BC,$57,$E5,$8D
	.dc.b	$FA,$9C,$3D,$5E,$3F,$39,$2F,$67,$9E,$3A,$FC,$77,$6F,$A6,$93,$A3
	.dc.b	$C7,$1E,$2F,$63,$C7,$EC,$49,$D6,$A9,$F3,$FC,$8C,$9C,$BE,$AF,$8C
	.dc.b	$F5,$FB,$32,$78,$BE,$EF,$37,$DF,$F4,$D2,$7A,$BF,$0B,$E2,$3E,$1F
	.dc.b	$C4,$7C,$7E,$AF,$E8,$DD,$1F,$4A,$D1,$8E,$5F,$B1,$5B,$F1,$B8,$7C
	.dc.b	$B4,$BF,$75,$FE,$5F,$0E,$4B,$E6,$7B,$7C,$D4,$AD,$C7,$87,$DE,$F1
	.dc.b	$7B,$12,$79,$C7,$97,$E9,$BC,$DE,$B4,$9E,$F1,$E9,$F3,$3D,$5E,$AC
	.dc.b	$9F,$1C,$7B,$7E,$9A,$47,$99,$F8,$1E,$AF,$C1,$F9,$79,$3E,$A7,$E2
	.dc.b	$7C,$67,$C5,$F8,$09,$3E,$E7,$E4,$7C,$2F,$E3,$FD,$AF,$74,$FF,$B7
	.dc.b	$B8,$30,$7E,$4F,$E9,$9C,$BF,$95,$FA,$67,$D1,$D2,$5F,$E1,$E1,$F3
	.dc.b	$FB,$F9,$D9,$39,$79,$7C,$77,$3F,$9F,$F1,$74,$CA,$E5,$1D,$FE,$93
	.dc.b	$51,$83,$B3,$FA,$7F,$A5,$79,$FF,$A9,$FA,$67,$C7,$53,$51,$BF,$46
	.dc.b	$E1,$F9,$DB,$E2,$D8,$FD,$97,$7F,$C0,$8F,$D1,$79,$AC,$BF,$6C,$D8
	.dc.b	$FD,$34,$ED,$FE,$00,$03,$F3,$9C,$98,$E2,$EE,$78,$39,$3F,$81,$7D
	.dc.b	$2B,$D5,$EF,$CB,$2F,$F6,$93,$96,$FA,$BC,$2A,$46,$0D,$FA,$BD,$26
	.dc.b	$46,$0E,$7B,$FB,$39,$24,$60,$EF,$BF,$92,$64,$60,$F1,$FA,$9F,$E4
	.dc.b	$57,$FC,$63,$92,$31,$1F,$E3,$B8,$C7,$8F,$FE,$33,$6C,$70,$F5,$5F
	.dc.b	$4D,$96,$C6,$63,$07,$F8,$01,$7D,$35,$23,$FE,$04,$7F,$C0,$BE,$5F
	.dc.b	$F3,$6C,$3F,$83,$4F,$F8,$37,$CD,$FE,$3C,$FD,$7A,$FB,$1F,$4D,$F6
	.dc.b	$23,$ED,$1B,$3E,$C9,$FA,$49,$E3,$C9,$D5,$FF,$2C,$61,$67,$8C,$D7
	.dc.b	$F0,$9C,$95,$FE,$1D,$5F,$8E,$F8,$DC,$4E,$7B,$64,$F9,$A2,$D7,$8B
	.dc.b	$CD,$E4,$63,$4F,$6B,$DD,$FE,$91,$5F,$CE,$DB,$3F,$92,$E9,$FD,$BC
	.dc.b	$E7,$9B,$87,$1C,$8C,$1C,$5F,$57,$D3,$7C,$3C,$3F,$98,$36,$7E,$3D
	.dc.b	$BB,$F7,$BF,$0D,$E5,$7F,$4D,$BE,$19,$FE,$9D,$F9,$8F,$E9,$97,$97
	.dc.b	$E0,$F3,$3C,$2D,$AE,$51,$6B,$84,$DC,$FC,$BE,$57,$A0,$FA,$5E,$1F
	.dc.b	$EB,$70,$F1,$47,$B3,$CA,$91,$83,$CC,$FC,$0F,$2A,$46,$0F,$53,$F7
	.dc.b	$7E,$99,$EB,$5F,$5B,$0D,$AF,$ED,$96,$F4,$3F,$08,$BD,$1B,$18,$73
	.dc.b	$8E,$BC,$FB,$7C,$9F,$C6,$7D,$33,$DC,$A6,$3D,$AB,$5F,$DE,$FE,$9B
	.dc.b	$F1,$FC,$7F,$C4,$9B,$C5,$FD,$A5,$77,$7E,$5D,$CD,$7E,$89,$7A,$B9
	.dc.b	$35,$EA,$D9,$FF,$04,$7D,$2F,$F9,$B9,$BF,$11,$FA,$18,$FF,$16,$C6
	.dc.b	$0D,$FB,$B9,$35,$C9,$63,$FC,$25,$C3,$F8,$6D,$4D,$3F,$0F,$AF,$F8
	.dc.b	$05,$D5,$3E,$8B,$F6,$0B,$E7,$AC,$61,$FD,$A7,$D2,$FF,$A4,$71,$FF
	.dc.b	$10,$3E,$9C,$7F,$9B,$B9,$6E,$9C,$3C,$DE,$3E,$65,$BE,$2F,$EF,$FE
	.dc.b	$97,$E3,$C7,$F8,$AD,$B7,$FD,$09,$FF,$56,$46,$78,$7F,$83,$7D,$2F
	.dc.b	$FB,$9A,$3F,$C6,$4D,$AF,$E9,$2B,$E4,$37,$F3,$1C,$3E,$86,$5F,$0F
	.dc.b	$5E,$E1,$FD,$43,$E9,$7F,$9A,$4F,$F8,$F5,$69,$FC,$55,$CC,$EE,$3F
	.dc.b	$8C,$7D,$2F,$F5,$BA,$FF,$22,$3F,$4C,$C4,$FF,$5C,$71,$DF,$C3,$C6
	.dc.b	$E2,$DA,$FF,$26,$7D,$4F,$DA,$7F,$DE,$8D,$BF,$F7,$CB,$5F,$E6,$21
	.dc.b	$C3,$FC,$D3,$E2,$FF,$CE,$B3,$FE,$FD,$6C,$70,$3F,$1F,$19,$3F,$A0
	.dc.b	$BF,$DD,$61,$F4,$BF,$CD,$5B,$3F,$E6,$17,$9F,$57,$F9,$35,$CD,$7F
	.dc.b	$7A,$5F,$39,$C4,$E5,$F3,$16,$F9,$47,$82,$BF,$D1,$AD,$AF,$F7,$79
	.dc.b	$3A,$70,$4A,$F6,$36,$BC,$1C,$3E,$78,$F4,$3F,$D3,$73,$FF,$6A,$D7
	.dc.b	$A7,$DC,$C7,$C7,$FB,$59,$1F,$1D,$F1,$11,$F9,$5F,$4B,$F0,$CD,$E3
	.dc.b	$EF,$7E,$37,$D2,$C9,$F8,$DF,$07,$6B,$E2,$4F,$B4,$93,$F5,$5F,$85
	.dc.b	$F4,$9F,$C0,$A7,$EB,$35,$FB,$ED,$7F,$B8,$B5,$FF,$16,$3C,$78,$3F
	.dc.b	$D0,$0F,$F3,$B8,$FF,$16,$B9,$AE,$3F,$07,$F1,$7E,$8F,$FB,$94,$BE
	.dc.b	$0F,$15,$FE,$9F,$8C,$36,$3C,$E4,$7F,$19,$69,$DB,$EA,$9E,$BE,$07
	.dc.b	$66,$D7,$5F,$2F,$C2,$78,$BC,$CB,$19,$25,$FA,$A3,$2F,$F9,$23,$EA
	.dc.b	$7F,$81,$69,$F0,$3D,$2F,$F0,$64,$62,$FE,$3C,$7B,$8F,$9E,$47,$D3
	.dc.b	$9A,$3F,$29,$F4,$BF,$BA,$BF,$F5,$53,$6B,$FC,$43,$6E,$9F,$F0,$D4
	.dc.b	$E7,$1F,$F2,$F3,$FC,$09,$FD,$BB,$5F,$F1,$4B,$5D,$9E,$03,$9F,$B5
	.dc.b	$D1,$B7,$FE,$D1,$9F,$EA,$6D,$F7,$FC,$3E,$8F,$89,$91,$93,$84,$62
	.dc.b	$BF,$CD,$F5,$FE,$0A,$9E,$9D,$FF,$23,$CB,$F5,$B2,$32,$7F,$B8,$73
	.dc.b	$FE,$94,$E2,$F9,$FA,$BF,$08,$FA,$3D,$96,$7D,$11,$67,$EC,$4D,$DB
	.dc.b	$E6,$CD,$AF,$9B,$36,$BE,$5C,$D5,$69,$C6,$4A,$D4,$E6,$FC,$24,$BF
	.dc.b	$3B,$EB,$BE,$35,$C1,$B1,$BC,$3F,$DD,$57,$CE,$F8,$92,$3D,$33,$CC
	.dc.b	$E0,$7D,$EB,$E8,$E2,$7D,$6D,$7D,$77,$E0,$7A,$47,$C2,$35,$87,$B5
	.dc.b	$E1,$87,$8F,$0F,$0D,$70,$3A,$9F,$3A,$78,$E1,$E2,$8D,$0B,$E8,$E2
	.dc.b	$FA,$57,$A4,$E2,$7B,$63,$8C,$F0,$BF,$09,$F1,$B0,$F5,$38,$AF,$55
	.dc.b	$70,$C3,$C5,$39,$8F,$3B,$5F,$87,$3E,$07,$C9,$5D,$23,$C9,$1E,$D3
	.dc.b	$A3,$F1,$23,$E5,$47,$EC,$2B,$F7,$64,$9F,$E2,$F7,$F0,$27,$51,$89
	.dc.b	$D4,$7F,$67,$9F,$74,$FE,$E3,$CE,$FC,$79,$1F,$AA,$D9,$F8,$AD,$C3
	.dc.b	$D9,$5F,$DE,$DB,$19,$AF,$5C,$7F,$0E,$8C,$57,$C4,$9F,$01,$D8,$8F
	.dc.b	$66,$CF,$46,$9D,$0E,$31,$B2,$46,$DB,$1B,$7F,$8C,$CF,$89,$FC,$35
	.dc.b	$FF,$1E,$8C,$5F,$C4,$9D,$D7,$81,$F4,$18,$3A,$98,$3F,$21,$3C,$AB
	.dc.b	$A4,$F1,$9E,$EF,$F1,$F9,$D3,$4D,$57,$86,$79,$C6,$2B,$9C,$78,$0E
	.dc.b	$67,$F2,$0A,$E6,$D9,$E7,$D9,$FC,$EB,$E2,$7F,$9D,$B6,$3F,$98,$36
	.dc.b	$7F,$94,$B7,$14,$FF,$B9,$3E,$85,$F4,$DF,$E1,$D6,$C7,$F6,$66,$DF
	.dc.b	$F8,$CD,$AF,$F8,$F3,$E9,$3F,$C5,$77,$D2,$02,$33,$5D,$EF,$FD,$D5
	.dc.b	$18,$9D,$FC,$8F,$FA,$AE,$B8,$DF,$39,$4E,$54,$67,$E7,$9F,$4D,$FE
	.dc.b	$F5,$AD,$4F,$17,$85,$7F,$94,$9A,$F9,$8E,$BE,$E7,$F8,$F6,$33,$1E
	.dc.b	$AC,$5F,$49,$FE,$41,$6E,$5F,$E0,$A6,$CF,$F9,$5D,$AF,$F9,$83,$8D
	.dc.b	$E1,$7F,$92,$5B,$5F,$E1,$26,$CF,$F9,$7D,$AF,$CD,$E4,$F0,$12,$7C
	.dc.b	$0F,$F9,$4F,$E6,$7E,$9C,$FA,$DA,$FD,$26,$A6,$E5,$FD,$AE,$3F,$CD
	.dc.b	$CD,$7E,$37,$37,$C3,$3F,$CC,$1F,$2C,$E4,$A4,$C7,$FC,$57,$19,$7F
	.dc.b	$C5,$B7,$FE,$A6,$BF,$A1,$EA,$FC,$D2,$4E,$BD,$CF,$A0,$47,$07,$F8
	.dc.b	$DB,$EA,$7D,$24,$E5,$FF,$39,$3E,$CF,$A5,$FF,$0E,$7D,$2F,$F8,$F1
	.dc.b	$B8,$FF,$A2,$5F,$F5,$77,$81,$9F,$05,$5F,$E4,$2F,$A4,$B7,$63,$FC
	.dc.b	$8A,$A3,$2F,$FA,$4D,$AF,$FA,$71,$7B,$9C,$1B,$1D,$63,$07,$4E,$B7
	.dc.b	$95,$49,$3F,$E5,$48,$CB,$FE,$56,$FA,$3F,$F4,$82,$69,$36,$F3,$FD
	.dc.b	$27,$DD,$9F,$F2,$EB,$5F,$F5,$53,$5F,$F5,$9B,$5F,$D1,$7D,$1F,$C0
	.dc.b	$4D,$23,$F1,$3E,$91,$E1,$52,$47,$F9,$B2,$32,$FF,$9B,$BE,$8B,$E3
	.dc.b	$BF,$AF,$6B,$FC,$19,$A9,$BF,$49,$F0,$09,$8F,$D4,$7E,$1F,$47,$F0
	.dc.b	$0A,$FC,$8F,$87,$F6,$23,$11,$FE,$C9,$6B,$FE,$D2,$7E,$6B,$9D,$B8
	.dc.b	$11,$FF,$42,$FE,$C3,$F4,$09,$87,$E1,$1C,$7B,$5F,$C2,$7C,$5F,$E2
	.dc.b	$D0,$F8,$0C,$57,$F0,$D8,$C5,$FF,$1C,$8C,$5E,$A3,$F9,$3C,$BE,$07
	.dc.b	$10,$8F,$F3,$C3,$C4,$DF,$3E,$A3,$84,$64,$E1,$EE,$67,$C7,$91,$99
	.dc.b	$FF,$6F,$B7,$E5,$B9,$11,$D7,$9D,$D7,$DC,$91,$C5,$F0,$78,$FE,$1F
	.dc.b	$82,$93,$D5,$F2,$7C,$FF,$97,$E3,$FC,$DE,$F4,$AF,$14,$7A,$7D,$EF
	.dc.b	$57,$D3,$C9,$F3,$47,$B7,$E5,$3D,$DF,$01,$27,$D5,$1F,$07,$D1,$91
	.dc.b	$ED,$7E,$27,$95,$F2,$A7,$F7,$3D,$3F,$D2,$DB,$D1,$FD,$05,$B7,$FD
	.dc.b	$61,$FD,$CB,$FD,$3A,$35,$F0,$5C,$C9,$1F,$3E,$5F,$03,$DF,$E7,$64
	.dc.b	$E5,$F1,$7C,$C7,$8F,$E6,$24,$F1,$7C,$DF,$07,$E7,$F8,$49,$3D,$5F
	.dc.b	$57,$C4,$7A,$FE,$9B,$D9,$F0,$B2,$BC,$51,$EF,$F8,$6F,$81,$F3,$32
	.dc.b	$7C,$D1,$F0,$FE,$43,$E2,$7C,$84,$9F,$54,$7E,$43,$E8,$DF,$35,$F9
	.dc.b	$2F,$75,$F3,$74,$D4,$7E,$5B,$E8,$DE,$05,$F0,$66,$5F,$93,$9C,$B7
	.dc.b	$C1,$E1,$E0,$C8,$C1,$BE,$9F,$07,$C5,$48,$C1,$CF,$C0,$F0,$4C,$8C
	.dc.b	$1F,$A1,$FA,$27,$4F,$EA,$BE,$8F,$FA,$C4,$FE,$95,$AF,$DD,$3F,$C2
	.dc.b	$F3,$C1,$E5,$D5,$FB,$28,$C4,$6C,$38,$F0,$3F,$C2,$CA,$7F,$73,$F4
	.dc.b	$7F,$2A,$BF,$69,$D0,$CF,$F7,$AD,$8F,$DC,$4E,$DE,$CF,$D0,$73,$C5
	.dc.b	$D0,$EB,$C7,$FC,$1B,$E8,$FF,$23,$3D,$B8,$FA,$99,$F0,$35,$E0,$59
	.dc.b	$FE,$0D,$5C,$1F,$62,$8E,$9E,$07,$09,$DC,$61,$E3,$C3,$EA,$E5,$91
	.dc.b	$83,$AE,$FE,$59,$91,$83,$C3,$E0,$7F,$8E,$9F,$F1,$91,$FE,$48,$C5
	.dc.b	$19,$8C,$C6,$38,$B7,$1B,$9D,$3F,$E3,$B9,$78,$0F,$A3,$DC,$A7,$31
	.dc.b	$86,$C7,$46,$BC,$3E,$8F,$FB,$21,$1F,$F0,$2B,$FE,$06,$F8,$2F,$F3
	.dc.b	$06,$3F,$E5,$79,$46,$3C,$5F,$3D,$7F,$99,$3E,$8D,$CB,$F0,$2B,$57
	.dc.b	$FE,$53,$E5,$AF,$F9,$A9,$AF,$4C,$3F,$9F,$3F,$B6,$6F,$2E,$8E,$3C
	.dc.b	$CF,$3B,$13,$C3,$B3,$99,$8B,$FC,$E0,$7B,$FC,$94,$9E,$07,$81,$F2
	.dc.b	$EC,$E7,$A7,$83,$C0,$B5,$FD,$4E,$7F,$66,$DE,$5C,$E3,$FA,$B9,$CB
	.dc.b	$FA,$DF,$A3,$F7,$37,$7F,$5F,$7F,$C4,$9C,$D7,$FA,$15,$F8,$55,$C7
	.dc.b	$83,$57,$B7,$87,$C5,$F1,$FA,$3B,$9F,$F6,$7C,$11,$98,$C3,$AC,$3C
	.dc.b	$8F,$87,$E8,$E4,$FF,$6F,$F0,$FF,$A0,$BC,$E3,$CD,$F2,$92,$30,$77
	.dc.b	$7D,$5F,$29,$23,$07,$8D,$FC,$0F,$D1,$BC,$BB,$F2,$F8,$6D,$7F,$7F
	.dc.b	$BF,$F4,$FB,$FE,$9B,$5F,$35,$F8,$A7,$3A,$7F,$AD,$1C,$CF,$1C,$75
	.dc.b	$F4,$7B,$6C,$7F,$82,$FE,$8D,$E9,$7E,$51,$B1,$9F,$47,$13,$F9,$8F
	.dc.b	$A3,$FE,$8F,$3F,$E1,$36,$E5,$FC,$A5,$FC,$74,$67,$BF,$87,$CA,$48
	.dc.b	$E1,$F2,$96,$FF,$98,$7D,$1F,$F9,$BA,$7F,$C3,$CD,$8C,$FA,$FF,$A8
	.dc.b	$8C,$75,$FF,$B2,$57,$AD,$93,$5C,$77,$0F,$F1,$47,$07,$E3,$1F,$F6
	.dc.b	$23,$71,$FF,$6D,$BF,$8E,$C3,$FD,$A8,$E6,$3E,$B7,$BD,$FB,$8B,$7B
	.dc.b	$D6,$30,$FE,$F3,$E8,$FF,$D2,$38,$FF,$8D,$DF,$3B,$FE,$D3,$73,$7F
	.dc.b	$D3,$DC,$BF,$EE,$C7,$F9,$2C,$71,$F3,$7A,$79,$DD,$5F,$E1,$3F,$46
	.dc.b	$E1,$A6,$38,$6D,$FF,$91,$FE,$8F,$FC,$B0,$DF,$88,$FE,$23,$1F,$E5
	.dc.b	$38,$C1,$BF,$82,$C9,$AF,$5E,$C7,$F9,$47,$E8,$FC,$BB,$7F,$67,$7F
	.dc.b	$99,$C6,$75,$FE,$3B,$18,$9F,$ED,$9F,$46,$7F,$90,$BF,$34,$C5,$7F
	.dc.b	$97,$5C,$77,$EB,$C6,$DB,$67,$FC,$C5,$F4,$6E,$6F,$E5,$7D,$49,$FE
	.dc.b	$5B,$39,$7F,$CD,$11,$96,$FC,$5F,$03,$8B,$4C,$9F,$D3,$21,$1F,$D0
	.dc.b	$72,$F5,$46,$27,$53,$99,$4B,$E2,$70,$DA,$E9,$18,$23,$E6,$F8,$9F
	.dc.b	$5C,$7B,$1F,$6F,$47,$BB,$86,$D7,$B1,$C3,$E7,$7B,$91,$F5,$7B,$7E
	.dc.b	$22,$38,$3D,$DF,$0F,$F2,$8F,$97,$16,$3E,$4C,$7A,$88,$F0,$3E,$37
	.dc.b	$C3,$FB,$88,$E0,$99,$FF,$8B,$73,$7C,$3F,$D8,$AF,$B2,$FF,$64,$FA
	.dc.b	$3F,$93,$7F,$E9,$F7,$F9,$8D,$FF,$8D,$5C,$B7,$E0,$78,$1F,$46,$37
	.dc.b	$1F,$E1,$06,$BF,$91,$0E,$1F,$CA,$3F,$E3,$D9,$FF,$05,$B6,$3F,$C2
	.dc.b	$6D,$7A,$7C,$07,$7E,$7F,$41,$E9,$6D,$7A,$62,$CF,$A6,$6F,$85,$F9
	.dc.b	$78,$F0,$F9,$1E,$92,$5F,$89,$B1,$F0,$7D,$BF,$8D,$1F,$1D,$F9,$1F
	.dc.b	$3D,$2B,$E4,$79,$D6,$FE,$44,$FC,$52,$4F,$B9,$FC,$C7,$D1,$FE,$CA
	.dc.b	$7F,$CC,$4F,$F2,$13,$18,$F9,$E1,$19,$F5,$B8,$92,$FF,$C6,$71,$FD
	.dc.b	$FA,$7F,$C8,$2F,$F2,$BF,$88,$7C,$FF,$F8,$8A,$30,$DF,$77,$1F,$01
	.dc.b	$70,$FF,$70,$57,$F8,$B1,$B3,$A6,$9A,$9E,$9E,$C4,$8E,$3E,$0D,$CB
	.dc.b	$DB,$37,$4F,$66,$2A,$C5,$FE,$1E,$3F,$85,$5F,$C4,$F0,$3D,$1E,$AF
	.dc.b	$A5,$95,$E6,$72,$D2,$4F,$17,$D7,$3C,$23,$04,$C3,$C7,$19,$7C,$B5
	.dc.b	$6B,$C9,$36,$BF,$96,$5F,$9A,$F7,$5D,$8F,$0B,$89,$ED,$1C,$8C,$9E
	.dc.b	$FB,$E9,$1F,$7A,$ED,$F5,$AD,$1D,$67,$99,$E2,$AD,$D7,$06,$CF,$69
	.dc.b	$D3,$5E,$CD,$AE,$D1,$A3,$F2,$75,$CE,$74,$D8,$F0,$9C,$8B,$3C,$9F
	.dc.b	$CB,$D7,$D4,$72,$F9,$1B,$5D,$0F,$14,$66,$FE,$41,$23,$E0,$CF,$91
	.dc.b	$D9,$C5,$E6,$78,$AF,$CD,$3C,$63,$EB,$C6,$27,$9C,$FE,$01,$E7,$3F
	.dc.b	$EA,$2B,$FC,$8B,$FA,$AB,$FD,$B9,$27,$F5,$ED,$8E,$0D,$AF,$DC,$7D
	.dc.b	$1F,$F6,$19,$F8,$ED,$8F,$24,$23,$15,$E4,$3F,$94,$38,$8F,$A9,$3D
	.dc.b	$27,$F0,$F3,$99,$F9,$F1,$FD,$FE,$74,$FD,$F9,$D5,$FD,$56,$DF,$FA
	.dc.b	$91,$B3,$C4,$F0,$9D,$88,$73,$2D,$FB,$1B,$7E,$C9,$E7,$3F,$EF,$D6
	.dc.b	$C7,$92,$38,$4F,$32,$38,$24,$73,$8E,$A7,$C5,$8F,$F0,$E4,$F4,$8F
	.dc.b	$39,$CA,$78,$34,$E1,$3F,$3D,$E6,$FC,$C7,$8B,$E0,$77,$3D,$63,$BC
	.dc.b	$70,$7E,$79,$ED,$71,$7F,$35,$B7,$F3,$5A,$7C,$D9,$E6,$F9,$5F,$11
	.dc.b	$83,$FA,$1E,$BF,$CD,$FE,$4F,$F9,$C3,$67,$F9,$AC,$59,$DC,$60,$EC
	.dc.b	$BE,$8B,$FC,$A6,$FF,$BF,$9A,$FF,$75,$35,$E5,$7F,$8A,$E3,$31,$FD
	.dc.b	$E7,$87,$FA,$12,$D3,$FA,$A4,$61,$FE,$AD,$3D,$9C,$F0,$7F,$BC,$23
	.dc.b	$FD,$D1,$87,$91,$B7,$CE,$3F,$C0,$D0,$F0,$B8,$7C,$1C,$9F,$35,$7F
	.dc.b	$96,$EB,$8B,$E9,$6B,$CA,$7F,$C8,$9C,$BF,$E4,$18,$E2,$D7,$E0,$35
	.dc.b	$FF,$0B,$35,$ED,$E5,$7C,$32,$5F,$8A,$3E,$64,$76,$C2,$4E,$6B,$D4
	.dc.b	$D8,$F9,$9F,$1F,$23,$96,$28,$C6,$1E,$ED,$3F,$DD,$35,$DD,$B1,$9E
	.dc.b	$5F,$F3,$04,$62,$3E,$D7,$C0,$3A,$7F,$CC,$8D,$BF,$43,$67,$E5,$B5
	.dc.b	$FC,$15,$3C,$EE,$A4,$AF,$13,$FE,$6F,$E6,$F6,$D7,$D4,$6B,$F8,$B6
	.dc.b	$BF,$C6,$B4,$FF,$3B,$7D,$03,$E4,$9F,$41,$F2,$E3,$0D,$F9,$47,$76
	.dc.b	$6B,$78,$7E,$0B,$E9,$FF,$C1,$0F,$D3,$6B,$9E,$EF,$AC,$A3,$2F,$E7
	.dc.b	$FE,$85,$F4,$3A,$6C,$7D,$D6,$BF,$90,$6B,$FA,$7F,$A1,$7C,$F7,$66
	.dc.b	$C7,$E1,$5A,$FE,$61,$AF,$EB,$FE,$87,$FD,$30,$9A,$4E,$2F,$4F,$D0
	.dc.b	$BA,$A9,$2B,$F2,$51,$97,$F2,$95,$FE,$A8,$6F,$A9,$F0,$A3,$11,$FA
	.dc.b	$A6,$BF,$C0,$7E,$83,$CA,$FF,$9D,$DA,$FF,$B2,$5A,$9B,$F4,$3F,$5A
	.dc.b	$3F,$48,$D7,$F6,$ED,$7F,$87,$35,$FF,$42,$7D,$07,$C2,$26,$5F,$61
	.dc.b	$F8,$BF,$40,$D3,$ED,$B6,$BF,$64,$D7,$F8,$2B,$5F,$E3,$8B,$E8,$38
	.dc.b	$92,$38,$BD,$FA,$24,$7E,$F6,$FF,$89,$36,$3F,$17,$D9,$F3,$E3,$F5
	.dc.b	$3B,$E0,$8F,$F0,$5F,$A0,$F9,$FE,$2F,$F9,$E5,$78,$DF,$EA,$38,$C3
	.dc.b	$75,$FF,$53,$4F,$06,$F5,$66,$FF,$DF,$53,$FA,$97,$F9,$EB,$5F,$E5
	.dc.b	$46,$A3,$82,$8F,$F7,$23,$C3,$2C,$C7,$BA,$38,$9B,$F6,$B3,$E3,$48
	.dc.b	$EE,$F0,$F8,$4E,$3E,$DC,$9F,$1B,$D3,$DB,$EB,$E4,$C9,$F3,$BE,$0F
	.dc.b	$93,$F0,$FC,$87,$8B,$DD,$95,$EC,$8F,$2F,$BB,$E6,$F3,$24,$FC,$08
	.dc.b	$F4,$F2,$FA,$BE,$6E,$4F,$C4,$8F,$6F,$BB,$23,$E3,$BF,$03,$DD,$F8
	.dc.b	$B3,$FC,$32,$7E,$0B,$79,$FF,$D6,$0D,$E6,$FF,$6D,$3F,$C1,$23,$3E
	.dc.b	$F7,$22,$57,$F7,$09,$7B,$DD,$BE,$6E,$4E,$5F,$0F,$CB,$78,$BE,$5A
	.dc.b	$4F,$17,$CB,$EF,$F9,$BE,$02,$4F,$57,$D3,$F1,$3E,$AF,$A0,$F5,$FC
	.dc.b	$8C,$AF,$14,$7B,$BE,$0F,$DF,$F0,$72,$7C,$D1,$F0,$BE,$3B,$E1,$FC
	.dc.b	$74,$9F,$54,$7E,$2B,$E8,$1F,$31,$F8,$CF,$75,$F2,$54,$D4,$7E,$43
	.dc.b	$E8,$1E,$05,$F0,$66,$5F,$E1,$49,$CB,$7C,$1E,$1E,$0C,$8C,$1B,$E9
	.dc.b	$F0,$7C,$D4,$8C,$1C,$FC,$0F,$04,$C8,$C1,$FE,$22,$F9,$F7,$4F,$E9
	.dc.b	$3E,$83,$FD,$C7,$3F,$E2,$66,$BF,$A2,$9D,$BC,$7A,$BF,$57,$18,$8D
	.dc.b	$56,$27,$F5,$FF,$0D,$F1,$A7,$FC,$63,$CE,$CB,$F6,$6D,$8F,$F1,$AC
	.dc.b	$ED,$FD,$8B,$F3,$1C,$F0,$F4,$3A,$F1,$7E,$F3,$E7,$FF,$1B,$3D,$B8
	.dc.b	$BA,$99,$EF,$D7,$7D,$9F,$F2,$1D,$70,$7F,$7E,$A3,$A7,$7F,$09,$DC
	.dc.b	$61,$E3,$B7,$AB,$92,$46,$0E,$BB,$F9,$66,$46,$0F,$0F,$7F,$F9,$25
	.dc.b	$FF,$1D,$E2,$8C,$47,$F9,$4E,$31,$E0,$FF,$8B,$DB,$1C,$3C,$07,$D0
	.dc.b	$65,$B1,$98,$C1,$FE,$02,$5F,$41,$48,$FF,$81,$9F,$F0,$3F,$C1,$3C
	.dc.b	$5F,$F3,$4C,$FF,$84,$9A,$F2,$1F,$40,$E5,$FF,$35,$57,$19,$FE,$56
	.dc.b	$7E,$A2,$B9,$F2,$E7,$4D,$48,$EC,$70,$8D,$4F,$F9,$D7,$96,$CE,$CF
	.dc.b	$D8,$C7,$E3,$BC,$EC,$C7,$39,$AB,$B2,$98,$8B,$5E,$EF,$90,$D9,$8A
	.dc.b	$F4,$BD,$31,$73,$53,$BF,$4F,$EF,$AF,$FD,$1F,$3A,$6F,$27,$13,$83
	.dc.b	$83,$FD,$2B,$F4,$1F,$02,$7E,$1B,$63,$DD,$4E,$EF,$6C,$E7,$2F,$90
	.dc.b	$FC,$D7,$FA,$CB,$6F,$9E,$FC,$37,$E3,$F0,$9C,$BE,$2E,$E7,$E2,$70
	.dc.b	$D3,$CB,$18,$C7,$AF,$D0,$78,$8F,$9F,$E1,$7F,$3E,$AF,$5C,$79,$BE
	.dc.b	$4A,$46,$0F,$7B,$EA,$F9,$19,$18,$3E,$1B,$F6,$BE,$F3,$E6,$61,$FD
	.dc.b	$D1,$BA,$FE,$23,$E8,$7F,$C8,$97,$FE,$BE,$9C,$BF,$85,$F8,$CF,$F5
	.dc.b	$6D,$7E,$39,$B1,$9F,$6F,$F0,$F1,$8C,$3F,$24,$BD,$4C,$9A,$F1,$ED
	.dc.b	$FF,$B3,$3E,$F9,$EB,$7E,$41,$B1,$FE,$D0,$FA,$17,$46,$6B,$F2,$BF
	.dc.b	$6D,$8A,$FF,$6A,$B7,$3F,$F0,$8A,$F5,$37,$7F,$9C,$FC,$33,$F3,$57
	.dc.b	$FE,$DC,$6D,$FF,$86,$1F,$D1,$B7,$C0,$F9,$5E,$D4,$AF,$F7,$1F,$D1
	.dc.b	$7D,$7A,$F9,$93,$FA,$8F,$BF,$FE,$6D,$5F,$EE,$86,$CF,$F8,$91,$FC
	.dc.b	$94,$67,$7F,$D7,$AF,$26,$98,$F2,$EC,$7F,$BC,$BA,$3F,$CB,$4D,$F8
	.dc.b	$8F,$DB,$FB,$2F,$7B,$F7,$35,$AC,$FF,$C6,$7F,$71,$EC,$EA,$FE,$0E
	.dc.b	$DC,$7F,$81,$3F,$9F,$6F,$93,$FC,$0A,$17,$4C,$77,$5A,$FC,$47,$BF
	.dc.b	$F8,$F3,$F8,$4F,$D1,$BC,$1C,$4F,$F1,$5C,$BF,$8B,$F9,$EF,$4F,$F8
	.dc.b	$7B,$67,$47,$F1,$55,$E2,$6A,$7F,$8A,$FD,$C7,$F0,$5D,$FF,$C9,$ED
	.dc.b	$C3,$F9,$1B,$FD,$4F,$0E,$73,$FC,$96,$35,$1F,$C8,$7E,$8D,$A7,$F2
	.dc.b	$06,$C7,$0A,$7D,$1B,$E0,$75,$7E,$2F,$AC,$97,$BC,$5F,$5B,$8C,$7F
	.dc.b	$CC,$7E,$73,$E9,$BF,$96,$B6,$FB,$3F,$EF,$4D,$7F,$9D,$B9,$AF,$F3
	.dc.b	$5F,$D2,$3E,$A7,$F9,$AF,$C4,$7F,$B8,$70,$F3,$4E,$5F,$44,$66,$BF
	.dc.b	$CE,$6B,$E2,$F8,$9C,$BE,$72,$D7,$08,$FE,$97,$5F,$D1,$6B,$84,$FF
	.dc.b	$47,$8C,$4E,$9F,$EA,$13,$A7,$41,$7C,$5E,$E3,$0D,$8C,$D7,$09,$CB
	.dc.b	$63,$31,$EA,$ED,$F8,$35,$F0,$9F,$EA,$F8,$FF,$BA,$AF,$E0,$F2,$E3
	.dc.b	$E6,$7E,$3B,$23,$A3,$E1,$23,$DF,$C3,$F2,$11,$C1,$E3,$F3,$FE,$89
	.dc.b	$F4,$A2,$C7,$D0,$8F,$61,$1E,$07,$B7,$E9,$7E,$23,$74,$F8,$91,$8D
	.dc.b	$3E,$4B,$5F,$A0,$E7,$87,$FB,$FB,$FC,$45,$B3,$FD,$E2,$31,$5F,$DF
	.dc.b	$73,$FF,$6C,$3F,$9B,$87,$C1,$7C,$8F,$94,$E0,$C9,$F3,$3E,$42,$3E
	.dc.b	$9F,$8E,$F3,$91,$C1,$ED,$F3,$FC,$47,$8E,$2C,$78,$62,$E7,$FE,$C6
	.dc.b	$C1,$58,$CF,$FC,$37,$F5,$9F,$ED,$EC,$3F,$08,$DB,$FC,$6B,$5F,$CA
	.dc.b	$35,$FF,$0E,$3B,$9F,$8F,$F4,$5F,$E1,$C8,$C5,$7F,$88,$70,$4F,$F8
	.dc.b	$83,$E9,$BF,$C5,$53,$FC,$05,$B8,$7F,$B7,$2F,$F5,$98,$7F,$8C,$27
	.dc.b	$2D,$9F,$CF,$47,$07,$F5,$0D,$7F,$C6,$0D,$7F,$C7,$13,$FA,$99,$E9
	.dc.b	$E8,$79,$F9,$BC,$8C,$C3,$F4,$D8,$7F,$58,$C3,$AC,$F8,$09,$3E,$9F
	.dc.b	$82,$E1,$3F,$EF,$4A,$47,$C3,$F9,$2F,$81,$EE,$DB,$F8,$0D,$BE,$0D
	.dc.b	$CB,$E0,$3F,$1F,$C9,$95,$FC,$9F,$4F,$EA,$4F,$C9,$8C,$DF,$85,$FA
	.dc.b	$5D,$38,$C3,$E5,$74,$FD,$13,$EB,$7C,$5D,$C3,$E9,$45,$C3,$E9,$4D
	.dc.b	$AE,$35,$A3,$C5,$19,$7E,$7D,$5A,$F9,$F3,$56,$BF,$C2,$5F,$8D,$3E
	.dc.b	$2A,$E9,$18,$6D,$FE,$5A,$F9,$5F,$18,$F1,$D7,$9A,$B6,$78,$8A,$F8
	.dc.b	$9A,$F8,$AF,$C9,$93,$0E,$B1,$86,$D7,$09,$D3,$CB,$0F,$13,$DE,$FC
	.dc.b	$47,$A4,$98,$79,$8F,$54,$71,$C3,$B3,$4E,$D8,$78,$2B,$8C,$7E,$BA
	.dc.b	$31,$5C,$E3,$07,$19,$D9,$EF,$29,$EF,$7A,$9B,$8E,$51,$B6,$BB,$38
	.dc.b	$E1,$F0,$1E,$78,$7C,$07,$FD,$44,$D5,$1B,$9E,$73,$93,$51,$9A,$F9
	.dc.b	$13,$AC,$3F,$05,$7D,$9F,$DE,$92,$7F,$66,$D8,$FE,$C3,$7F,$8B,$7F
	.dc.b	$80,$C6,$5B,$3F,$C3,$7E,$9B,$F1,$24,$7E,$D3,$63,$DA,$DB,$F4,$B6
	.dc.b	$7C,$E7,$F0,$C7,$13,$F0,$AB,$F8,$E4,$62,$3E,$55,$7F,$19,$8C,$5F
	.dc.b	$C3,$6D,$70,$BF,$19,$2F,$C6,$DB,$F1,$B4,$F1,$C7,$D2,$AE,$71,$EF
	.dc.b	$35,$1E,$57,$0F,$42,$4F,$46,$C7,$47,$F6,$F3,$D1,$FE,$38,$D9,$F6
	.dc.b	$4F,$69,$F7,$BF,$14,$C5,$FB,$DB,$7E,$F9,$EE,$79,$4E,$B1,$A3,$CF
	.dc.b	$5F,$D5,$BA,$FF,$DC,$FF,$47,$FE,$48,$AF,$E8,$D1,$8F,$16,$5F,$F9
	.dc.b	$2E,$B5,$7C,$63,$13,$FE,$F0,$F4,$5F,$4D,$FE,$27,$6C,$FF,$89,$5B
	.dc.b	$8F,$F8,$B9,$AF,$F8,$D7,$E4,$DF,$5B,$FE,$FA,$6D,$7F,$96,$EB,$8E
	.dc.b	$28,$5C,$AD,$BF,$D7,$FC,$0F,$ED,$06,$D5,$55,$F4,$3F,$E6,$0B,$FE
	.dc.b	$85,$5C,$1F,$EF,$2F,$F9,$07,$87,$FD,$89,$18,$3F,$C8,$1F,$4F,$FD
	.dc.b	$22,$FF,$C9,$2D,$7F,$CA,$0D,$7C,$E1,$C3,$EA,$F6,$9F,$05,$FE,$4F
	.dc.b	$6D,$7F,$93,$5B,$3F,$E5,$66,$BF,$E5,$DF,$9C,$FB,$4D,$E0,$FF,$2C
	.dc.b	$46,$23,$C2,$D7,$E4,$FC,$F3,$9B,$FC,$BC,$DC,$3F,$CB,$8D,$9F,$F3
	.dc.b	$2B,$5F,$B1,$E6,$7F,$99,$78,$3F,$D0,$11,$8A,$FF,$32,$46,$23,$E0
	.dc.b	$FC,$5B,$EB,$BF,$CD,$8D,$AF,$F3,$53,$67,$FC,$E6,$D7,$FC,$F5,$F5
	.dc.b	$DE,$94,$D2,$69,$FE,$77,$FA,$87,$9D,$F3,$2B,$FC,$FB,$18,$E0,$FF
	.dc.b	$3C,$2F,$9F,$7A,$38,$BE,$17,$FE,$89,$FA,$6A,$DC,$3F,$D0,$0A,$32
	.dc.b	$FD,$E6,$BF,$96,$5F,$4F,$C1,$B1,$FE,$89,$6B,$FE,$92,$6B,$FE,$99
	.dc.b	$EA,$79,$DE,$06,$C7,$EB,$63,$07,$93,$E9,$BF,$D3,$10,$69,$38,$3D
	.dc.b	$3D,$6F,$5E,$92,$BF,$D3,$91,$97,$FD,$3D,$F4,$BF,$01,$34,$8F,$C4
	.dc.b	$FA,$67,$89,$49,$1F,$EA,$78,$CB,$FE,$A9,$FA,$EF,$8E,$9A,$4D,$FE
	.dc.b	$7F,$D3,$FB,$09,$9E,$0F,$A7,$D7,$FC,$4A,$FF,$33,$E2,$60,$8F,$E7
	.dc.b	$37,$D9,$1C,$1F,$92,$F6,$BE,$EF,$57,$FB,$92,$30,$43,$FD,$8F,$FA
	.dc.b	$0B,$C0,$FC,$1B,$63,$FD,$88,$D7,$71,$EA,$F2,$A4,$7F,$B2,$74,$FF
	.dc.b	$14,$36,$3F,$1B,$F5,$BF,$1A,$3F,$8C,$EB,$FC,$2B,$C3,$7C,$CF,$F0
	.dc.b	$CF,$23,$F8,$E4,$62,$BF,$92,$C7,$F1,$FB,$E4,$FF,$21,$FB,$87,$CA
	.dc.b	$F1,$C7,$F8,$AC,$7F,$BB,$EF,$8C,$63,$D5,$D4,$7F,$30,$9E,$2E,$8C
	.dc.b	$7B,$BA,$50,$E0,$C3,$63,$81,$A9,$E1,$F7,$EE,$EE,$5A,$BE,$3A,$9A
	.dc.b	$8E,$DF,$72,$F7,$6F,$BB,$32,$EB,$39,$7B,$FE,$20,$F8,$EF,$32,$F9
	.dc.b	$CA,$62,$3D,$5F,$80,$6E,$BC,$FA,$73,$7E,$99,$FC,$E7,$2D,$D7,$C1
	.dc.b	$F9,$F7,$D7,$5F,$AF,$97,$F0,$27,$2D,$F6,$F8,$7C,$74,$8C,$1F,$1B
	.dc.b	$F0,$07,$D9,$7C,$4F,$96,$93,$98,$FA,$1F,$78,$F9,$8B,$F1,$72,$BE
	.dc.b	$7C,$E5,$BE,$87,$0F,$1E,$46,$0F,$AB,$F8,$07,$E5,$1F,$78,$E3,$FE
	.dc.b	$74,$FC,$4F,$6F,$F0,$13,$BF,$0F,$FB,$77,$DE,$38,$3F,$08,$BE,$C6
	.dc.b	$98,$8F,$C4,$78,$6F,$AD,$BF,$5D,$33,$FC,$3C,$E5,$FC,$4F,$DE,$BE
	.dc.b	$2F,$F2,$1F,$05,$FD,$BA,$7C,$ED,$D3,$FC,$01,$5B,$9E,$7C,$1E,$0D
	.dc.b	$E4,$60,$FC,$AF,$DE,$BE,$AB,$F3,$8B,$E3,$A9,$88,$FD,$07,$DA,$7E
	.dc.b	$2E,$BF,$3B,$9F,$E2,$5F,$1F,$81,$AF,$0F,$F4,$DF,$69,$F0,$E7,$F4
	.dc.b	$6F,$C9,$F3,$BF,$C0,$ED,$8F,$F0,$FB,$B7,$E9,$AF,$03,$9F,$2C,$9E
	.dc.b	$71,$DB,$CE,$F8,$BF,$27,$27,$BC,$78,$BC,$ED,$46,$0F,$1B,$FB,$BF
	.dc.b	$B3,$FE,$19,$1F,$82,$FB,$4F,$F0,$43,$FB,$A6,$BF,$86,$8D,$F3,$7F
	.dc.b	$8B,$D7,$83,$CF,$D4,$95,$FC,$0A,$72,$FF,$04,$FB,$1F,$94,$FE,$43
	.dc.b	$EA,$59,$7F,$0D,$6C,$6F,$87,$F8,$85,$7C,$FC,$BF,$C7,$CF,$26,$C6
	.dc.b	$B8,$BF,$8B,$7D,$A3,$BB,$A7,$B1,$31,$FE,$2B,$39,$6F,$81,$C2,$A4
	.dc.b	$60,$DF,$97,$D2,$64,$60,$E7,$BF,$A7,$92,$46,$0E,$FC,$3E,$49,$91
	.dc.b	$83,$C7,$E4,$7F,$96,$9F,$CF,$0C,$BF,$80,$6D,$88,$C4,$ED,$FD,$24
	.dc.b	$E9,$E2,$FF,$2D,$FB,$4E,$0D,$8C,$FA,$A9,$CB,$63,$FC,$43,$F4,$9F
	.dc.b	$E0,$32,$38,$3C,$5F,$17,$FE,$71,$C7,$FC,$E1,$1F,$E2,$EE,$CF,$E8
	.dc.b	$2F,$F9,$EA,$BF,$CF,$7F,$68,$E5,$FD,$D6,$1C,$6F,$F7,$9C,$B7,$0F
	.dc.b	$F3,$5B,$5F,$F4,$1C,$D7,$F9,$E4,$F6,$E5,$C1,$BE,$67,$DA,$62,$7C
	.dc.b	$7D,$9C,$4C,$5F,$F2,$00,$F5,$F9,$C9,$3C,$0F,$5B,$E3,$D9,$CF,$4F
	.dc.b	$07,$95,$6B,$FB,$0C,$FF,$7A,$6F,$2E,$71,$FE,$5F,$39,$7F,$98,$FD
	.dc.b	$A7,$C9,$6E,$DE,$BE,$FF,$85,$39,$AF,$F4,$F3,$F0,$AB,$8F,$06,$AF
	.dc.b	$6F,$0F,$12,$65,$FD,$D2,$FF,$8F,$36,$33,$7F,$90,$9F,$E4,$93,$F8
	.dc.b	$F6,$C6,$67,$EF,$BF,$EB,$68,$7D,$46,$21,$18,$AE,$53,$98,$C4,$F7
	.dc.b	$F2,$F0,$4A,$F0,$3B,$FA,$A9,$1E,$36,$CF,$E9,$BE,$CF,$FA,$84,$FE
	.dc.b	$95,$B1,$9D,$FC,$91,$88,$FE,$95,$F1,$3F,$86,$AF,$D6,$36,$3F,$62
	.dc.b	$D7,$7C,$9E,$D8,$FF,$63,$C6,$BE,$6F,$04,$AF,$DB,$7D,$9B,$EC,$BE
	.dc.b	$23,$63,$31,$83,$E4,$7D,$93,$A6,$92,$7F,$80,$46,$5F,$E0,$5F,$65
	.dc.b	$F9,$86,$F5,$7D,$18,$C6,$5F,$C2,$9A,$FF,$69,$D7,$FB,$F5,$7C,$7F
	.dc.b	$5F,$E2,$91,$FE,$1D,$F6,$3F,$CD,$1B,$E5,$7D,$D8,$C5,$FF,$16,$6B
	.dc.b	$FD,$D6,$7F,$C7,$87,$F1,$72,$BF,$C7,$1A,$EF,$C2,$FC,$47,$1F,$F9
	.dc.b	$16,$7F,$8F,$36,$7F,$91,$B5,$FF,$01,$35,$FF,$78,$FD,$8F,$F9,$06
	.dc.b	$7F,$93,$36,$3F,$95,$B5,$FF,$05,$35,$E3,$3F,$99,$83,$49,$3F,$8D
	.dc.b	$F9,$CF,$C5,$91,$C1,$FC,$D7,$C4,$FE,$F0,$DF,$27,$F4,$91,$8C,$3F
	.dc.b	$9E,$B5,$FF,$0E,$7D,$6F,$F0,$CA,$FE,$7C,$D8,$FE,$86,$D7,$7E,$17
	.dc.b	$EB,$FE,$C3,$FD,$D0,$4C,$70,$7E,$CE,$BF,$D8,$57,$D3,$DD,$C1,$1F
	.dc.b	$E7,$7F,$41,$FB,$9F,$0B,$FD,$F9,$97,$F3,$08,$C1,$FB,$DF,$AC,$FF
	.dc.b	$78,$D7,$F5,$36,$CF,$F5,$86,$BF,$E3,$76,$A2,$2F,$77,$A1,$27,$A3
	.dc.b	$67,$FC,$AB,$3F,$E9,$C9,$FF,$46,$BC,$5B,$F1,$DF,$D1,$E1,$F5,$B7
	.dc.b	$EA,$70,$DA,$E3,$1E,$B2,$3E,$B7,$DD,$C3,$23,$DC,$DC,$3D,$DD,$BD
	.dc.b	$CF,$04,$59,$F2,$F7,$F8,$CF,$27,$45,$8C,$9F,$26,$3E,$61,$1C,$1E
	.dc.b	$EE,$1E,$18,$7C,$27,$FB,$27,$D6,$34,$BE,$AE,$1B,$7F,$EB,$29,$FF
	.dc.b	$11,$3F,$E0,$96,$DF,$F8,$39,$AF,$F9,$B5,$AF,$D7,$0F,$F8,$FA,$39
	.dc.b	$7A,$9F,$CF,$E3,$13,$CE,$31,$EA,$79,$8F,$3B,$6B,$C9,$F1,$5E,$A3
	.dc.b	$D7,$F1,$D6,$32,$79,$E3,$B1,$1C,$1E,$DF,$13,$86,$1F,$E9,$AC,$7F
	.dc.b	$93,$37,$EB,$3C,$C4,$AF,$84,$FA,$7E,$57,$E5,$9F,$31,$B1,$FE,$DA
	.dc.b	$FA,$BF,$CC,$37,$A7,$FC,$29,$18,$BF,$F5,$4B,$FD,$D6,$7E,$43,$FD
	.dc.b	$EE,$3F,$D3,$0D,$73,$3F,$D9,$E3,$11,$FA,$55,$EA,$76,$3B,$F8,$F6
	.dc.b	$FC,$91,$C4,$8F,$F9,$16,$78,$B6,$3C,$FE,$B7,$1B,$F3,$BE,$AF,$82
	.dc.b	$F5,$FA,$56,$FE,$03,$86,$9F,$0E,$31,$5F,$03,$C4,$F8,$67,$C4,$F4
	.dc.b	$2D,$7C,$98,$B3,$F1,$A6,$FA,$8A,$F8,$FF,$4F,$F3,$0F,$9B,$F3,$D6
	.dc.b	$3E,$94,$5B,$F9,$F3,$6B,$94,$7C,$99,$C5,$7D,$2A,$B3,$98,$E6,$7D
	.dc.b	$D8,$CC,$35,$56,$14,$D8,$C3,$6B,$9D,$79,$1B,$7E,$48,$E9,$5C,$27
	.dc.b	$27,$C0,$2C,$7E,$06,$BA,$3F,$8C,$9F,$87,$3A,$8E,$6D,$BF,$2B,$67
	.dc.b	$B4,$F1,$8C,$1E,$F2,$B9,$3D,$05,$BF,$45,$7C,$29,$ED,$7E,$49,$CD
	.dc.b	$F9,$2F,$85,$7A,$DA,$7A,$DE,$F3,$C1,$F1,$46,$8F,$59,$6F,$D6,$DA
	.dc.b	$F5,$D7,$18,$C5,$72,$8C,$11,$F1,$3E,$AB,$F4,$46,$0F,$C2,$1E,$43
	.dc.b	$13,$E7,$3A,$9F,$EB,$2B,$F8,$4D,$BF,$DC,$12,$7F,$60,$D8,$D4,$60
	.dc.b	$FE,$15,$F3,$7F,$8B,$23,$F6,$5B,$1F,$05,$B8,$7A,$E3,$F8,$24,$62
	.dc.b	$7C,$F1,$F9,$78,$C3,$F9,$D3,$F8,$9B,$87,$E6,$CF,$DC,$8C,$5F,$B2
	.dc.b	$31,$5E,$32,$57,$8D,$B1,$E3,$8F,$04,$60,$F1,$CA,$BA,$DF,$F2,$66
	.dc.b	$CF,$C2,$38,$57,$9E,$FC,$2F,$72,$3D,$EF,$A4,$F0,$9F,$5B,$FC,$BA
	.dc.b	$BA,$C7,$C0,$8D,$4F,$B5,$D1,$8B,$F2,$34,$F2,$4F,$81,$F5,$D7,$4B
	.dc.b	$F2,$9E,$37,$E6,$7D,$43,$E4,$7F,$DA,$CB,$E4,$AC,$61,$97,$FE,$21
	.dc.b	$BF,$E0,$95,$B8,$C1,$FD,$BD,$E8,$E3,$0F,$F1,$44,$6D,$4F,$2F,$17
	.dc.b	$FD,$61,$3F,$E2,$59,$FE,$F0,$70,$8C,$9A,$F5,$BC,$E7,$A1,$B7,$DE
	.dc.b	$54,$F8,$2F,$69,$EE,$F1,$AD,$6E,$74,$F8,$2B,$CF,$5E,$42,$46,$61
	.dc.b	$D3,$E7,$7C,$ED,$7F,$C2,$0D,$7F,$C9,$0D,$5F,$0F,$FE,$DA,$7C,$0D
	.dc.b	$7E,$0B,$5F,$F0,$D3,$5F,$97,$E9,$FC,$82,$57,$97,$0F,$F5,$2C,$FA
	.dc.b	$FD,$D7,$67,$FB,$75,$B5,$EB,$6C,$EE,$7F,$DC,$58,$3E,$42,$92,$FE
	.dc.b	$14,$65,$F8,$9F,$8A,$BD,$BF,$F7,$53,$63,$31,$83,$FD,$DD,$F5,$0F
	.dc.b	$A7,$E1,$97,$D1,$8C,$4F,$CF,$5F,$50,$F0,$B1,$7C,$2F,$F0,$FF,$70
	.dc.b	$EF,$FA,$D0,$8C,$3F,$65,$AF,$DF,$6B,$F9,$0F,$A8,$7D,$4F,$26,$C7
	.dc.b	$E0,$5A,$FE,$1D,$AF,$E6,$3E,$77,$FB,$7B,$F8,$76,$BF,$8D,$6B,$BF
	.dc.b	$03,$CF,$F5,$3F,$DB,$E3,$F4,$2D,$7F,$CB,$4D,$7F,$D2,$4D,$5F,$53
	.dc.b	$F9,$47,$F2,$CD,$7F,$38,$D7,$F5,$AD,$7F,$CC,$DF,$53,$F2,$13,$48
	.dc.b	$FC,$CF,$A9,$F4,$BF,$E6,$D6,$BF,$EA,$56,$A6,$C7,$D9,$8F,$D5,$35
	.dc.b	$FD,$83,$5F,$E0,$2D,$7F,$CF,$5F,$4B,$EF,$27,$1F,$DE,$7E,$17,$D4
	.dc.b	$33,$FC,$0B,$6B,$F7,$0D,$7F,$7C,$D7,$F8,$7A,$FA,$8E,$24,$8E,$2F
	.dc.b	$7F,$02,$47,$F0,$3B,$46,$23,$F8,$57,$D4,$7C,$D7,$FD,$90,$D4,$D3
	.dc.b	$12,$F0,$7D,$84,$9E,$CF,$E7,$3B,$5E,$FD,$24,$FE,$ED,$FD,$0F,$D4
	.dc.b	$7D,$22,$71,$60,$FD,$2F,$D4,$7A,$2B,$F8,$E3,$5F,$E4,$4D,$7F,$99
	.dc.b	$B5,$FF,$56,$2F,$73,$99,$23,$9B,$FB,$2F,$A8,$F9,$D1,$FE,$B3,$6B
	.dc.b	$FE,$EB,$6A,$6F,$D4,$3D,$7F,$DD,$B6,$7F,$99,$35,$FE,$70,$D7,$FA
	.dc.b	$4F,$D4,$3E,$57,$F8,$1B,$63,$F9,$D3,$5F,$E8,$0D,$7F,$A8,$FD,$45
	.dc.b	$7F,$D9,$4D,$41,$C3,$67,$F8,$8F,$C4,$FF,$6B,$8F,$E9,$0D,$7F,$A7
	.dc.b	$35,$FE,$BE,$D7,$FD,$B0,$BE,$0F,$99,$23,$9B,$FC,$7B,$EA,$3F,$9B
	.dc.b	$47,$F5,$56,$BF,$D6,$DA,$FF,$6C,$6B,$FE,$E5,$FA,$77,$C0,$FF,$28
	.dc.b	$6C,$7F,$5F,$6B,$FD,$95,$AF,$F7,$67,$F2,$6E,$F8,$B9,$11,$FE,$D1
	.dc.b	$F4,$FF,$DC,$DB,$E8,$FF,$36,$8C,$47,$F7,$16,$BF,$E0,$2F,$A7,$7D
	.dc.b	$0F,$F3,$C6,$C7,$F7,$36,$BF,$DE,$5A,$FF,$82,$BE,$2D,$F3,$FF,$D1
	.dc.b	$1B,$1F,$DE,$DA,$FF,$80,$5A,$FF,$84,$BE,$9D,$EF,$7F,$4C,$6C,$7F
	.dc.b	$80,$DA,$FF,$82,$5A,$F8,$7E,$9F,$C0,$47,$C0,$FF,$56,$FA,$67,$C1
	.dc.b	$52,$4F,$F8,$3E,$32,$FF,$84,$7E,$99,$F3,$FF,$D7,$9B,$19,$8C,$1F
	.dc.b	$D8,$BE,$99,$F3,$D4,$93,$FE,$1B,$8C,$B7,$9B,$81,$1C,$1B,$F0,$BA
	.dc.b	$11,$C1,$CF,$BF,$C0,$47,$07,$F8,$A7,$E9,$5F,$45,$FA,$A8,$FF,$21
	.dc.b	$F9,$97,$E9,$BF,$DB,$31,$FE,$3B,$6B,$A3,$F6,$35,$C9,$FF,$20,$2F
	.dc.b	$90,$E9,$F0,$9F,$ED,$98,$D4,$F1,$0E,$23,$53,$C2,$FC,$9F,$17,$F3
	.dc.b	$2F,$87,$C7,$FF,$1F,$7F,$C9,$CD,$7A,$46,$EF,$FD,$D0,$FF,$0D,$6F
	.dc.b	$07,$59,$FE,$13,$3B,$BE,$FE,$16,$65,$F4,$CF,$93,$C6,$5F,$A6,$CB
	.dc.b	$FD,$E8,$DC,$3E,$55,$6E,$FF,$CC,$4F,$E7,$2B,$FD,$F4,$E6,$3E,$13
	.dc.b	$63,$3D,$7E,$18,$78,$96,$25,$74,$8D,$70,$79,$38,$E4,$F8,$23,$96
	.dc.b	$FA,$8C,$1E,$47,$B7,$87,$FE,$82,$8F,$F3,$86,$3F,$C4,$B2,$FF,$37
	.dc.b	$36,$33,$3B,$7F,$D0,$B0,$F9,$D7,$D3,$70,$6C,$F4,$6B,$C1,$AF,$F9
	.dc.b	$FF,$E7,$7F,$C4,$04,$7F,$C9,$2F,$F9,$2B,$E0,$1B,$FF,$A3,$67,$FC
	.dc.b	$A4,$D5,$3F,$D1,$3E,$99,$C9,$FE,$9A,$BE,$35,$F9,$13,$F7,$37,$CF
	.dc.b	$97,$3A,$FF,$90,$6A,$CF,$FA,$8A,$6D,$7E,$9F,$96,$BF,$D4,$9B,$C8
	.dc.b	$F8,$CC,$CF,$F6,$1B,$F4,$7C,$A6,$0C,$DD,$8F,$67,$CA,$70,$31,$5E
	.dc.b	$77,$BE,$7F,$84,$9F,$F5,$73,$67,$F4,$BD,$DA,$E7,$DF,$34,$BD,$C5
	.dc.b	$AF,$D9,$7D,$2F,$CA,$6E,$FE,$AE,$EF,$6C,$E7,$2F,$93,$48,$F9,$2F
	.dc.b	$2E,$0D,$DF,$07,$8F,$85,$32,$FF,$62,$5F,$FA,$BF,$0E,$6F,$F3,$08
	.dc.b	$CC,$F8,$3E,$77,$86,$3E,$67,$C7,$F4,$DE,$4F,$1E,$DF,$47,$0F,$6F
	.dc.b	$97,$B7,$EE,$7D,$67,$B3,$EC,$2C,$F9,$1F,$B7,$F4,$9F,$CD,$A3,$FD
	.dc.b	$B5,$F6,$CF,$A9,$CF,$27,$E0,$23,$13,$FE,$DF,$F8,$FF,$C0,$37,$EC
	.dc.b	$3F,$0B,$18,$CF,$F9,$02,$8C,$FD,$0E,$BE,$86,$DF,$37,$F1,$BF,$54
	.dc.b	$FA,$3F,$C7,$FD,$B3,$ED,$F3,$18,$7F,$27,$F5,$8F,$0A,$C6,$1B,$8F
	.dc.b	$FB,$D7,$EB,$1E,$17,$E6,$97,$83,$63,$0F,$C4,$F9,$EF,$CF,$FD,$BF
	.dc.b	$F7,$AB,$FF,$7F,$73,$7F,$B7,$BB,$BF,$0B,$39,$E4,$FE,$7D,$F5,$BF
	.dc.b	$E5,$5B,$FF,$1B,$D7,$E2,$A1,$C7,$FA,$A7,$55,$C2,$3F,$5B,$5F,$C3
	.dc.b	$4E,$27,$F5,$71,$F6,$BA,$F2,$F6,$FE,$D3,$B3,$FD,$25,$3C,$2F,$FC
	.dc.b	$8F,$F5,$EF,$80,$D7,$C9,$7E,$EA,$31,$5B,$A7,$D7,$FE,$E5,$FE,$B0
	.dc.b	$FC,$88,$D3,$9D,$7F,$4A,$E6,$77,$8F,$F0,$68,$78,$16,$30,$FE,$43
	.dc.b	$AF,$F8,$5F,$DB,$FF,$87,$B1,$F9,$B1,$8F,$0B,$F8,$7F,$A1,$FE,$53
	.dc.b	$9F,$33,$F8,$56,$FA,$BA,$ED,$FE,$2B,$F5,$FF,$C1,$76,$FD,$CB,$FE
	.dc.b	$00,$AF,$73,$97,$FC,$04,$E7,$6F,$EE,$DF,$60,$CF,$F8,$FF,$DB,$BE
	.dc.b	$BF,$F9,$0C,$E7,$0F,$91,$39,$7A,$FC,$47,$F2,$6F,$3F,$FC,$EF,$18
	.dc.b	$3E,$73,$8C,$7F,$C0,$8B,$CC,$E1,$F4,$DC,$5B,$5F,$57,$EC,$1F,$47
	.dc.b	$FC,$C9,$7C,$45,$8C,$32,$78,$7C,$CD,$CB,$FC,$25,$7F,$EC,$89,$C9
	.dc.b	$23,$35,$D6,$BB,$7C,$BC,$AF,$17,$D8,$DB,$E7,$3F,$D1,$3E,$A1,$D7
	.dc.b	$9E,$5F,$E5,$31,$8C,$3D,$5C,$FE,$9E,$7F,$53,$63,$DD,$17,$1F,$3C
	.dc.b	$DB,$FE,$67,$AF,$EA,$AF,$E2,$78,$5F,$0F,$E1,$FE,$23,$23,$E4,$74
	.dc.b	$5C,$7E,$0B,$F2,$49,$1C,$8D,$9F,$A3,$FB,$1F,$F1,$13,$7E,$1B,$FB
	.dc.b	$0C,$31,$B1,$27,$F5,$5A,$FE,$81,$E0,$63,$BF,$8F,$C2,$F2,$F8,$5B
	.dc.b	$3D,$DC,$34,$F1,$46,$33,$FF,$25,$65,$FE,$D5,$9F,$37,$CB,$79,$3B
	.dc.b	$BC,$AC,$8F,$57,$A1,$6F,$8B,$EA,$3F,$AE,$BF,$E6,$3B,$FE,$55,$3F
	.dc.b	$E6,$58,$C4,$E9,$FF,$68,$CE,$9D,$05,$F5,$9F,$18,$F8,$FF,$57,$6F
	.dc.b	$E6,$47,$3A,$E0,$DA,$E3,$3D,$27,$2D,$FB,$1F,$EF,$11,$E5,$6B,$FE
	.dc.b	$7B,$6B,$FE,$E6,$6A,$D9,$F6,$3F,$97,$3F,$C3,$73,$FD,$FE,$BF,$D0
	.dc.b	$93,$A7,$F9,$24,$7F,$46,$7C,$5E,$97,$A6,$31,$7F,$09,$AF,$96,$3A
	.dc.b	$FC,$3F,$66,$D7,$48,$B3,$FE,$85,$AE,$7C,$FF,$E1,$07,$FC,$77,$5E
	.dc.b	$2F,$80,$F3,$9E,$8F,$AA,$B1,$93,$E3,$47,$94,$8F,$C9,$EC,$FF,$1C
	.dc.b	$FD,$B7,$F9,$32,$3D,$B1,$87,$E1,$B5,$C9,$FD,$16,$38,$63,$E7,$FB
	.dc.b	$2F,$92,$7C,$AD,$AD,$7C,$88,$B3,$ED,$9B,$9F,$B6,$6D,$7D,$99,$B5
	.dc.b	$F6,$66,$AC,$FE,$A3,$E1,$AE,$C4,$7B,$36,$FB,$34,$EC,$58,$EC,$D7
	.dc.b	$B5,$F8,$29,$76,$30,$DA,$E1,$3A,$6C,$E9,$A6,$A7,$85,$7A,$49,$7E
	.dc.b	$67,$D2,$62,$7F,$39,$3C,$27,$44,$8C,$9F,$96,$AF,$05,$71,$3F,$44
	.dc.b	$E2,$78,$46,$27,$C8,$79,$9E,$07,$CF,$AF,$04,$76,$8C,$DF,$BD,$F3
	.dc.b	$D7,$47,$84,$F3,$8C,$4E,$DB,$7E,$39,$E9,$1F,$9F,$8C,$57,$D4,$3A
	.dc.b	$CE,$8F,$13,$FC,$06,$FF,$64,$48,$FE,$0C,$D8,$FD,$A3,$5E,$11,$FE
	.dc.b	$1F,$FB,$CF,$ED,$2F,$E0,$5B,$1F,$09,$B7,$ED,$6C,$F8,$E4,$23,$2F
	.dc.b	$A6,$7F,$D1,$31,$88,$F8,$A2,$33,$E1,$FE,$7A,$75,$8F,$C7,$BE,$B7
	.dc.b	$D6,$B4,$48,$F4,$36,$3D,$0D,$BF,$43,$67,$D1,$3C,$DE,$24,$78,$B6
	.dc.b	$78,$B4,$E3,$7E,$6C,$3F,$D8,$6D,$8F,$99,$7D,$EF,$E1,$47,$AA,$75
	.dc.b	$1F,$06,$31,$3B,$8F,$8B,$7F,$12,$7C,$11,$C8,$F9,$F5,$CA,$75,$1E
	.dc.b	$82,$3E,$86,$CF,$A2,$BB,$61,$F3,$67,$9C,$63,$0F,$5B,$6B,$D6,$78
	.dc.b	$A7,$CA,$FA,$0F,$EF,$FB,$7F,$88,$4F,$F7,$2F,$D8,$BE,$CB,$13,$A2
	.dc.b	$57,$C1,$3F,$13,$3B,$8C,$E5,$FE,$D0,$8F,$F5,$54,$FF,$73,$6B,$F8
	.dc.b	$46,$BC,$BE,$37,$FB,$64,$7F,$AC,$67,$FD,$B3,$CB,$F9,$18,$C5,$FE
	.dc.b	$32,$31,$F0,$BC,$09,$1F,$DC,$67,$FD,$89,$87,$F7,$EA,$FF,$24,$37
	.dc.b	$C7,$FE,$F5,$18,$8F,$F0,$43,$5F,$ED,$7F,$75,$FD,$FF,$9C,$D2,$61
	.dc.b	$DF,$DD,$F4,$91,$C1,$FE,$03,$9F,$F7,$45,$7F,$84,$1A,$FF,$85,$DA
	.dc.b	$FF,$7A,$E6,$F0,$79,$92,$B9,$36,$7F,$C3,$3F,$B1,$7F,$9C,$DF,$F1
	.dc.b	$03,$5F,$D8,$B5,$FD,$FB,$57,$D9,$FF,$9D,$3E,$1F,$FC,$30,$D8,$FF
	.dc.b	$12,$AF,$B3,$7A,$3A,$F4,$17,$FE,$1F,$FB,$3F,$8B,$85,$DF,$FC,$44
	.dc.b	$A3,$2F,$F8,$C1,$AF,$A2,$F7,$97,$68,$FF,$15,$35,$F7,$C6,$BE,$3A
	.dc.b	$46,$9A,$FC,$4F,$92,$E6,$47,$9B,$E4,$FA,$F7,$C7,$D2,$4F,$F8,$E6
	.dc.b	$32,$F8,$BE,$1F,$CA,$47,$07,$9B,$C5,$F4,$91,$C1,$CF,$C2,$F0,$11
	.dc.b	$C1,$EE,$EE,$F8,$24,$70,$7F,$92,$FE,$AD,$F6,$BF,$31,$B1,$98,$C1
	.dc.b	$F2,$BE,$E6,$47,$CA,$6B,$FE,$60,$9F,$F5,$8B,$D7,$C2,$E8,$47,$E8
	.dc.b	$F1,$7D,$42,$38,$3C,$9E,$CF,$8C,$8E,$0F,$47,$B3,$FD,$0B,$C2,$75
	.dc.b	$FF,$44,$C5,$39,$8C,$36,$BF,$AA,$CF,$FC,$01,$7F,$DF,$DF,$28,$F7
	.dc.b	$35,$08,$C3,$A8,$CF,$3B,$CC,$DF,$D6,$BE,$B6,$92,$FF,$C8,$DA,$F4
	.dc.b	$FA,$D7,$8D,$49,$3F,$DA,$37,$F5,$BF,$DA,$A5,$E4,$F1,$F7,$24,$EF
	.dc.b	$87,$FB,$1D,$27,$B7,$B9,$EA,$FA,$D7,$C4,$D2,$61,$FE,$59,$F8,$AF
	.dc.b	$7F,$D6,$BD,$9A,$49,$EC,$7F,$7C,$87,$B3,$F1,$BE,$B5,$E2,$52,$57
	.dc.b	$01,$0F,$0B,$E6,$FD,$6F,$09,$D3,$5F,$E8,$41,$C7,$17,$C4,$87,$B0
	.dc.b	$FA,$AF,$AF,$F3,$BF,$D3,$67,$C0,$D7,$9B,$6B,$93,$FC,$46,$37,$F0
	.dc.b	$BC,$88,$F2,$F8,$6F,$C1,$FD,$6B,$D3,$A4,$9F,$F4,$14,$FE,$23,$EB
	.dc.b	$5F,$09,$49,$3F,$0E,$1F,$27,$BF,$0E,$4E,$5F,$C8,$FE,$23,$FE,$CF
	.dc.b	$D7,$E0,$7D,$77,$E5,$EB,$E2,$B6,$33,$D1,$F9,$8E,$1F,$CC,$57,$D1
	.dc.b	$FA,$EF,$8C,$DE,$2F,$CE,$C6,$23,$E7,$AE,$1F,$D0,$F9,$6F,$A7,$A4
	.dc.b	$AF,$B9,$0E,$4F,$D3,$FD,$73,$E5,$29,$2B,$FD,$4D,$E0,$74,$FA,$E9
	.dc.b	$1C,$9E,$FC,$5F,$B0,$FA,$E7,$BF,$49,$5F,$8B,$87,$C9,$7E,$DB,$EB
	.dc.b	$9E,$DD,$25,$7E,$47,$EB,$BE,$37,$95,$FE,$48,$5F,$0F,$FB,$E6,$CE
	.dc.b	$E3,$1C,$B9,$E4,$93,$CD,$E1,$DD,$C7,$D3,$93,$DD,$E9,$E5,$C8,$F1
	.dc.b	$35,$FC,$EE,$2E,$2F,$E1,$FF,$5A,$FA,$AF,$E2,$3F,$5D,$FE,$F4,$BF
	.dc.b	$D5,$35,$FD,$2C,$ED,$FD,$72,$F9,$3E,$75,$27,$31,$FC,$73,$DD,$F8
	.dc.b	$F1,$FB,$5F,$33,$F8,$43,$73,$D7,$16,$FC,$CB,$3F,$BB,$BE,$2F,$ED
	.dc.b	$9F,$DA,$F1,$FF,$9D,$3E,$B9,$F2,$DF,$CA,$B2,$FA,$37,$FE,$84,$6A
	.dc.b	$9D,$BF,$C1,$DF,$C8,$36,$7F,$99,$C6,$A3,$F8,$2B,$A6,$F7,$6B,$C0
	.dc.b	$91,$CE,$38,$ED,$E3,$E5,$93,$DE,$3A,$EF,$A8,$C1,$E3,$7C,$3E,$4F
	.dc.b	$FA,$56,$3F,$D2,$27,$FA,$7B,$1F,$DF,$F8,$B8,$8C,$4E,$DD,$CE,$9D
	.dc.b	$3F,$C0,$BE,$BB,$84,$7C,$3F,$25,$18,$9D,$4F,$09,$CB,$FE,$22,$F9
	.dc.b	$DF,$F1,$1A,$69,$1E,$BF,$0D,$FD,$33,$1F,$E5,$51,$FE,$30,$E1,$FE
	.dc.b	$6C,$FF,$AE,$2B,$FD,$73,$F5,$CE,$3F,$7E,$1C,$6F,$E0,$F2,$DC,$3F
	.dc.b	$D6,$6D,$7F,$D8,$33,$5F,$DA,$CF,$6E,$5C,$1B,$E5,$FF,$38,$CF,$FA
	.dc.b	$F2,$38,$7A,$32,$B8,$1E,$67,$A7,$C2,$C8,$C1,$EA,$7C,$5A,$FF,$15
	.dc.b	$8F,$EA,$4D,$BF,$E9,$B8,$7F,$9E,$A3,$FA,$61,$92,$F1,$F2,$39,$F2
	.dc.b	$DB,$FE,$B9,$F5,$DF,$27,$0F,$EA,$ED,$BF,$EA,$D8,$7F,$82,$6B,$7A
	.dc.b	$7C,$5A,$F8,$77,$C3,$5C,$4E,$5D,$78,$B2,$FB,$FB,$F6,$FB,$CF,$3F
	.dc.b	$17,$19,$FF,$65,$CB,$F9,$2E,$1E,$6F,$88,$F0,$79,$5E,$16,$C7,$AA
	.dc.b	$3C,$9F,$2D,$8C,$BD,$6E,$27,$DF,$F3,$56,$7D,$EF,$DF,$FA,$D7,$9B
	.dc.b	$4C,$79,$B6,$FF,$C0,$58,$FF,$89,$DB,$BF,$E1,$E3,$FD,$33,$AF,$F7
	.dc.b	$D7,$59,$7F,$80,$9F,$F0,$CD,$72,$F8,$2E,$1E,$EF,$17,$F2,$1F,$56
	.dc.b	$FA,$4B,$F3,$98,$6D,$FF,$84,$FE,$F9,$D7,$F9,$77,$F3,$D1,$98,$C7
	.dc.b	$76,$79,$24,$F1,$D2,$EB,$FE,$16,$FA,$8F,$C4,$4A,$24,$7F,$3B,$8F
	.dc.b	$F1,$13,$A8,$CF,$6F,$E9,$E1,$A5,$F9,$8C,$36,$FF,$C5,$5C,$FF,$E2
	.dc.b	$E6,$F8,$1F,$AF,$8F,$F1,$64,$60,$FF,$18,$6F,$F9,$D7,$C9,$97,$C5
	.dc.b	$F4,$FF,$6D,$3F,$E2,$99,$DC,$7E,$E7,$EF,$9F,$43,$4C,$79,$B6,$BF
	.dc.b	$C7,$92,$FA,$8E,$1F,$45,$C5,$B3,$FE,$41,$FA,$76,$9F,$C1,$57,$7D
	.dc.b	$8C,$3F,$C2,$7E,$99,$F3,$57,$DE,$C3,$70,$FF,$26,$4F,$B7,$83,$E8
	.dc.b	$35,$3E,$D7,$FC,$96,$E7,$E9,$BF,$CB,$D3,$FE,$55,$6C,$FF,$94,$1F
	.dc.b	$F2,$EC,$67,$E2,$FF,$8E,$7C,$8F,$EE,$3C,$1F,$F2,$93,$67,$E6,$BF
	.dc.b	$56,$FF,$CB,$2E,$5F,$C2,$3F,$9E,$8D,$F8,$FD,$3C,$8E,$AF,$F2,$9E
	.dc.b	$47,$C8,$D3,$F9,$67,$DE,$FE,$8B,$1F,$F3,$2C,$EB,$0F,$F3,$73,$63
	.dc.b	$3F,$2B,$C6,$72,$F5,$3F,$A0,$C7,$E5,$E7,$FD,$4F,$3F,$D4,$9B,$1F
	.dc.b	$D6,$DA,$FF,$60,$7F,$C8,$D1,$BF,$63,$9C,$FE,$21,$B3,$A8,$E5,$F0
	.dc.b	$1E,$D3,$DD,$36,$BD,$3E,$F7,$C1,$3E,$17,$B9,$63,$27,$96,$3A,$91
	.dc.b	$FE,$D3,$F4,$DF,$98,$6E,$3E,$48,$C6,$3F,$DD,$5A,$FF,$7A,$73,$C1
	.dc.b	$FE,$8D,$7F,$C3,$35,$FE,$99,$72,$FE,$A9,$CC,$FF,$A5,$9A,$F3,$EC
	.dc.b	$F0,$12,$7C,$BE,$E7,$8C,$8E,$0F,$4F,$B9,$FD,$A6,$FF,$D1,$31,$E1
	.dc.b	$9C,$4E,$79,$24,$F9,$7B,$6E,$3E,$39,$F8,$04,$9F,$8A,$FD,$6F,$BD
	.dc.b	$7D,$25,$24,$FF,$84,$A3,$2F,$CC,$F1,$3E,$51,$1C,$1F,$43,$C5,$F4
	.dc.b	$1E,$98,$B1,$FE,$C1,$FA,$6F,$E4,$8F,$F8,$79,$B8,$FF,$89,$DA,$FF
	.dc.b	$8B,$1A,$FF,$18,$77,$BF,$FB,$05,$FE,$DD,$39,$9F,$F0,$84,$61,$FE
	.dc.b	$64,$FF,$3E,$8F,$17,$99,$AF,$6B,$6C,$8F,$23,$F9,$A9,$E1,$5E,$58
	.dc.b	$F9,$1F,$63,$DB,$D3,$FF,$0F,$68,$8C,$61,$EF,$9E,$44,$7E,$03,$6F
	.dc.b	$FD,$BB,$87,$F0,$37,$CF,$1E,$AE,$DF,$5F,$9B,$EC,$F1,$6E,$1F,$2D
	.dc.b	$B7,$C9,$BA,$FC,$57,$E1,$FC,$F4,$AF,$A5,$E9,$DB,$F0,$CD,$BF,$99
	.dc.b	$36,$BF,$A3,$FD,$33,$E7,$FF,$C7,$8D,$8F,$F7,$E7,$C7,$3E,$A7,$35
	.dc.b	$27,$FE,$00,$56,$F2,$FF,$7D,$DE,$E7,$35,$B9,$FF,$7F,$D6,$F1,$FF
	.dc.b	$7F,$5E,$E7,$35,$B9,$5F,$9A,$3F,$CA,$15,$B9,$CD,$7F,$73,$BE,$97
	.dc.b	$B3,$C2,$7A,$B0,$E8,$E6,$7B,$B9,$36,$E0,$95,$E6,$9E,$31,$DF,$0A
	.dc.b	$B0,$F2,$5F,$5B,$F2,$CB,$0F,$2B,$C6,$7C,$0F,$67,$81,$1F,$F0,$75
	.dc.b	$74,$78,$1B,$2D,$79,$67,$D2,$F8,$2B,$BC,$F8,$6F,$E4,$E1,$E8,$9C
	.dc.b	$9A,$C3,$B5,$6F,$0E,$67,$8D,$F7,$57,$1B,$F8,$35,$C2,$F9,$C6,$27
	.dc.b	$57,$CA,$FD,$71,$87,$D4,$74,$8D,$3E,$13,$37,$D6,$B8,$61,$EF,$9E
	.dc.b	$38,$78,$2F,$E5,$CF,$C1,$AD,$C7,$F8,$92,$F9,$3D,$AB,$E0,$61,$AC
	.dc.b	$3E,$5C,$EE,$3C,$D1,$FE,$C2,$9F,$CF,$36,$FF,$B1,$C6,$23,$F1,$86
	.dc.b	$6B,$FD,$7F,$19,$8F,$B2,$64,$FD,$1F,$37,$D3,$24,$F9,$5B,$1E,$E6
	.dc.b	$E3,$ED,$9F,$D8,$C6,$1F,$AD,$5F,$CF,$2B,$84,$73,$23,$82,$57,$37
	.dc.b	$C7,$1A,$8C,$1F,$8B,$AE,$D3,$F1,$8F,$D8,$38,$7E,$C4,$6E,$79,$1E
	.dc.b	$49,$D5,$7E,$41,$B7,$FC,$41,$B3,$EC,$38,$DF,$73,$F5,$51,$A7,$CE
	.dc.b	$7A,$A3,$B1,$C1,$F1,$BF,$C7,$A3,$85,$F9,$67,$81,$D6,$FE,$44,$7C
	.dc.b	$0A,$E0,$73,$8F,$43,$6F,$D0,$F7,$FA,$67,$D0,$FF,$2A,$F9,$EF,$F7
	.dc.b	$34,$62,$7F,$93,$C6,$5A,$7F,$A2,$F0,$FE,$0F,$3B,$C7,$F9,$3D,$7F
	.dc.b	$34,$F1,$7F,$1E,$4B,$FC,$6B,$F9,$2F,$93,$FE,$72,$6E,$3F,$9A,$BF
	.dc.b	$EF,$7F,$9F,$72,$7F,$8E,$9B,$3F,$E8,$8C,$3F,$D1,$31,$87,$01,$C3
	.dc.b	$BC,$7F,$D7,$2D,$E5,$FF,$22,$C6,$23,$FD,$26,$D7,$F9,$B7,$D3,$FE
	.dc.b	$A0,$97,$FA,$67,$8F,$97,$23,$C3,$2A,$FF,$6F,$BE,$46,$BE,$9F,$90
	.dc.b	$FF,$56,$CF,$06,$C7,$A9,$AE,$FB,$BF,$CB,$7F,$76,$FA,$6A,$4B,$F0
	.dc.b	$C6,$5F,$1F,$D4,$7F,$A6,$26,$FA,$7C,$70,$F8,$F2,$FA,$87,$C4,$EA
	.dc.b	$FD,$7C,$0D,$BD,$B8,$7C,$D6,$C6,$63,$71,$AF,$BA,$7D,$DD,$24,$FC
	.dc.b	$68,$CB,$F2,$3E,$19,$DF,$D9,$B1,$98,$C1,$E2,$FB,$BF,$F3,$EC,$1A
	.dc.b	$4D,$FC,$FF,$76,$F2,$A9,$2B,$EB,$C6,$5F,$B3,$F7,$7E,$E4,$D2,$3F
	.dc.b	$0B,$EB,$3C,$AF,$E4,$1A,$FE,$55,$AF,$FB,$79,$AF,$FB,$CF,$EF,$3E
	.dc.b	$93,$F9,$86,$BF,$9D,$6B,$FE,$E6,$6B,$FE,$FB,$FA,$CF,$82,$4C,$7E
	.dc.b	$AB,$F1,$3E,$F1,$AF,$D8,$6D,$7E,$59,$AF,$EA,$1A,$FE,$B5,$7D,$E7
	.dc.b	$12,$47,$17,$BF,$83,$23,$F4,$36,$8C,$47,$ED,$3E,$F3,$E7,$37,$6F
	.dc.b	$C5,$C6,$23,$F7,$8D,$47,$BF,$C2,$F6,$91,$EE,$FE,$4B,$EA,$24,$7E
	.dc.b	$61,$AF,$F0,$4C,$1E,$E7,$C6,$24,$7A,$1F,$CD,$FC,$6C,$8F,$D0,$B5
	.dc.b	$FE,$1B,$CA,$F9,$7F,$2C,$3C,$7F,$F1,$4C,$E6,$3F,$93,$70,$EA,$31
	.dc.b	$1B,$8C,$7C,$77,$2A,$E3,$ED,$59,$E0,$DA,$F8,$76,$8C,$42,$1E,$D6
	.dc.b	$3B,$5E,$9D,$F4,$E5,$F9,$0F,$9B,$3A,$70,$73,$F6,$BA,$E0,$F6,$A9
	.dc.b	$88,$CF,$CB,$EB,$EE,$65,$F2,$B4,$F8,$FE,$26,$4F,$6C,$51,$E7,$EC
	.dc.b	$EF,$D3,$2B,$93,$E2,$FB,$CF,$1F,$CB,$49,$EC,$F9,$BE,$5B,$E0,$78
	.dc.b	$BF,$07,$DB,$95,$DE,$3E,$3E,$2F,$99,$BF,$27,$2B,$E0,$CE,$5B,$F0
	.dc.b	$5C,$3E,$0A,$46,$0F,$57,$DA,$7C,$F2,$3E,$DF,$17,$BF,$D9,$F8,$3D
	.dc.b	$99,$3D,$A3,$C7,$ED,$79,$3D,$09,$3E,$28,$FB,$BF,$B1,$FF,$9D,$3C
	.dc.b	$1F,$F3,$AB,$7C,$1F,$C8,$B6,$BF,$4E,$FF,$A9,$5F,$5F,$7F,$C0,$23
	.dc.b	$EE,$EF,$F8,$7D,$32,$3E,$0B,$F1,$B8,$7C,$3F,$01,$E2,$F5,$65,$7E
	.dc.b	$1B,$1F,$F7,$BB,$F8,$8F,$A7,$F9,$0F,$E7,$4F,$DC,$B8,$D3,$FD,$1A
	.dc.b	$DD,$3F,$53,$F4,$CF,$7F,$F3,$6B,$E5,$E9,$30,$FC,$AC,$3E,$9F,$7E
	.dc.b	$64,$9C,$BC,$BE,$3F,$9F,$C0,$49,$E2,$F6,$F8,$F9,$1D,$1A,$FA,$46
	.dc.b	$0F,$8F,$FD,$7F,$D3,$3C,$FF,$D8,$FD,$3B,$C4,$A6,$A3,$7E,$85,$C7
	.dc.b	$F5,$B7,$C5,$B1,$FB,$AF,$9D,$F8,$91,$FC,$83,$CC,$69,$FB,$E6,$C7
	.dc.b	$ED,$A7,$69,$78,$BC,$FC,$69,$3B,$8F,$E0,$FF,$4A,$F9,$2E,$9C,$D2
	.dc.b	$BF,$80,$CE,$5B,$F4,$FC,$2A,$46,$0D,$FA,$BD,$26,$46,$0E,$7B,$FA
	.dc.b	$79,$A4,$60,$EF,$BF,$92,$64,$60,$F1,$FA,$BF,$CB,$5F,$F1,$EE,$5F
	.dc.b	$C7,$BB,$3F,$C5,$ED,$8C,$CE,$DF,$F2,$34,$3E,$99,$F4,$FC,$1B,$3D
	.dc.b	$1A,$F0,$6B,$FE,$4C,$FA,$7F,$F0,$21,$1F,$F1,$1B,$FE,$24,$F8,$7F
	.dc.b	$F7,$EE,$3F,$EE,$A9,$FF,$15,$F7,$FF,$2C,$7F,$AD,$D7,$F5,$CF,$A7
	.dc.b	$FB,$11,$FD,$2A,$F8,$D7,$F9,$77,$92,$CF,$F4,$E6,$BF,$D9,$26,$BF
	.dc.b	$C6,$E7,$B7,$26,$0D,$F3,$3E,$D3,$13,$E3,$F4,$78,$98,$BF,$D0,$07
	.dc.b	$AF,$A3,$27,$81,$EB,$7C,$7A,$FF,$63,$8F,$E9,$2D,$AF,$D5,$61,$FB
	.dc.b	$B9,$CF,$37,$0E,$39,$18,$38,$BE,$EF,$A8,$F9,$37,$FB,$36,$E0,$AF
	.dc.b	$EE,$38,$3C,$1E,$1B,$71,$DE,$57,$F5,$CC,$38,$BF,$DE,$1F,$91,$8F
	.dc.b	$5F,$37,$C0,$78,$7D,$0B,$5C,$63,$C9,$E7,$E2,$BC,$8F,$9F,$DD,$F4
	.dc.b	$7A,$16,$FC,$0E,$1F,$17,$A1,$6F,$D8,$FB,$CF,$81,$E8,$59,$F4,$3F
	.dc.b	$80,$FA,$87,$AF,$4C,$7A,$F6,$BF,$BC,$E2,$D7,$F0,$ED,$8C,$EB,$89
	.dc.b	$CF,$A7,$AF,$5E,$D7,$27,$F1,$FF,$50,$FA,$6B,$F2,$38,$6D,$7F,$82
	.dc.b	$3E,$A1,$D9,$F9,$67,$F8,$A4,$66,$31,$E5,$E7,$92,$4F,$1D,$AF,$27
	.dc.b	$F3,$1F,$A8,$FC,$57,$67,$DF,$8F,$F2,$1C,$7F,$31,$AF,$F0,$BB,$A7
	.dc.b	$FC,$66,$E6,$39,$69,$FA,$88,$FF,$41,$4F,$08,$C7,$BD,$FA,$CE,$7F
	.dc.b	$E5,$33,$FE,$22,$6D,$FF,$43,$7F,$49,$19,$F0,$BF,$67,$BF,$E7,$6B
	.dc.b	$FA,$43,$72,$F4,$47,$C9,$F5,$77,$EB,$7E,$EB,$EA,$1F,$41,$4C,$7A
	.dc.b	$17,$3F,$F1,$9C,$BE,$A3,$87,$C2,$F1,$6C,$FF,$8E,$3E,$A1,$BF,$F0
	.dc.b	$67,$FC,$5B,$19,$8C,$78,$1F,$C2,$BE,$9D,$F2,$F7,$EA,$30,$DB,$FF
	.dc.b	$21,$CF,$AD,$FF,$2F,$B7,$83,$FC,$EA,$FF,$73,$8D,$EF,$FC,$5B,$EA
	.dc.b	$3F,$C9,$11,$FE,$4C,$6D,$7F,$6D,$7F,$C8,$B1,$9F,$8B,$FE,$3B,$F5
	.dc.b	$1F,$B4,$E4,$F9,$2D,$9F,$F3,$7B,$F5,$23,$3E,$4F,$0F,$13,$8F,$91
	.dc.b	$6F,$FC,$B7,$7F,$EF,$2B,$FA,$7C,$CF,$A8,$FE,$56,$DA,$FF,$30,$FD
	.dc.b	$47,$FA,$A1,$FF,$31,$C6,$75,$FF,$02,$2F,$91,$E2,$72,$BB,$5C,$23
	.dc.b	$F9,$BF,$CF,$7D,$48,$C1,$FE,$6E,$71,$7D,$3D,$CF,$1F,$C2,$F9,$1B
	.dc.b	$5E,$28,$B3,$B8,$C1,$E6,$F8,$4F,$59,$EC,$F8,$1B,$19,$3D,$7F,$47
	.dc.b	$8A,$F5,$BF,$D3,$3E,$A3,$F2,$73,$FE,$82,$7F,$C0,$CD,$D7,$5E,$04
	.dc.b	$8D,$3C,$63,$17,$FE,$29,$AE,$2F,$F8,$AF,$4F,$C9,$B9,$AF,$8B,$D7
	.dc.b	$E1,$F8,$1F,$17,$C0,$DC,$BC,$4E,$1A,$79,$63,$17,$DF,$A3,$D4,$7A
	.dc.b	$FC,$1B,$5E,$B8,$F7,$7B,$18,$AF,$33,$FD,$B3,$EA,$1F,$19,$F0,$FC
	.dc.b	$AC,$37,$0F,$89,$E5,$7C,$73,$18,$7B,$DC,$47,$FB,$12,$3F,$D8,$BC
	.dc.b	$1F,$C4,$1C,$07,$0E,$DA,$FF,$B3,$3E,$9F,$FD,$C3,$3F,$EB,$47,$FA
	.dc.b	$CC,$7F,$B9,$1C,$BF,$E5,$77,$29,$FE,$D2,$F8,$DA,$F5,$F6,$BF,$C2
	.dc.b	$CF,$36,$BC,$7D,$1B,$3F,$CF,$2B,$D3,$39,$6C,$66,$3D,$AD,$ED,$FE
	.dc.b	$42,$FF,$42,$8F,$35,$7F,$86,$A3,$13,$AF,$5B,$B9,$2F,$AB,$E2,$F0
	.dc.b	$78,$1C,$5B,$1F,$16,$2E,$1F,$0E,$6D,$78,$BC,$8F,$01,$E1,$F4,$2E
	.dc.b	$1F,$36,$3D,$E4,$7D,$AF,$C2,$F8,$8F,$47,$C8,$7A,$5B,$3F,$55,$B8
	.dc.b	$FD,$36,$E1,$FD,$0B,$E9,$BF,$1B,$5F,$EE,$E6,$D7,$FA,$3D,$7D,$37
	.dc.b	$F8,$F9,$CD,$71,$E2,$C1,$99,$B7,$CE,$2D,$F3,$9B,$5C,$6B,$47,$86
	.dc.b	$32,$A6,$5F,$82,$31,$08,$C3,$A9,$E2,$F0,$BA,$EE,$2C,$F0,$8E,$53
	.dc.b	$E9,$AF,$1C,$7C,$18,$DD,$70,$6C,$EA,$FD,$55,$B8,$F5,$19,$3C,$F7
	.dc.b	$DC,$97,$CE,$35,$3E,$96,$BE,$46,$CF,$D2,$6D,$F8,$A7,$C0,$4B,$F1
	.dc.b	$9F,$76,$FB,$46,$6B,$C7,$18,$9D,$C6,$0E,$A7,$82,$34,$FA,$AB,$E3
	.dc.b	$57,$33,$E7,$46,$CE,$2E,$4F,$A0,$E2,$BB,$C7,$BC,$D5,$FC,$5A,$EB
	.dc.b	$3A,$9F,$84,$6A,$FD,$6F,$F1,$BA,$FC,$BB,$6B,$F8,$3C,$62,$3D,$17
	.dc.b	$FC,$1A,$31,$1E,$83,$F9,$0B,$87,$BD,$FF,$89,$BB,$3F,$81,$D7,$29
	.dc.b	$FE,$91,$EB,$7E,$32,$73,$1C,$EB,$84,$E4,$95,$F7,$2B,$75,$DA,$3F
	.dc.b	$37,$18,$BF,$CA,$37,$0F,$EE,$F7,$CA,$FE,$64,$E6,$FF,$BC,$4E,$70
	.dc.b	$F9,$13,$9A,$F3,$D7,$86,$B4,$FA,$EB,$CE,$E4,$93,$DA,$76,$FB,$63
	.dc.b	$CF,$18,$8F,$51,$FB,$7A,$F0,$57,$48,$E5,$5D,$23,$CB,$5C,$70,$F7
	.dc.b	$B7,$0F,$7D,$6C,$F3,$1E,$03,$17,$E3,$3C,$13,$C6,$3E,$19,$A8,$F2
	.dc.b	$8C,$3C,$B1,$CE,$3C,$A4,$C3,$CB,$1C,$70,$F2,$BF,$C9,$7E,$0F,$F9
	.dc.b	$11,$FC,$8F,$E6,$AC,$64,$97,$FC,$A6,$32,$FF,$2B,$F8,$BF,$CA,$92
	.dc.b	$7F,$28,$FE,$63,$CE,$91,$F9,$76,$CF,$E5,$BC,$6F,$CE,$92,$7F,$38
	.dc.b	$FE,$83,$E9,$64,$7E,$7D,$B3,$FD,$FB,$47,$2F,$F6,$C6,$D7,$F6,$A6
	.dc.b	$CE,$E7,$FB,$8F,$C1,$BB,$29,$2F,$FB,$8C,$65,$E5,$F3,$5D,$48,$E0
	.dc.b	$FE,$EF,$F4,$9F,$E7,$EF,$A4,$BE,$A9,$CF,$CA,$BB,$F3,$1F,$DF,$BE
	.dc.b	$D5,$F2,$FF,$E0,$0D,$91,$87,$51,$98,$C4,$7F,$87,$BE,$6D,$F4,$BC
	.dc.b	$5B,$1F,$E0,$A6,$BB,$8E,$DF,$38,$F6,$69,$27,$FC,$23,$19,$6F,$C1
	.dc.b	$70,$23,$83,$FC,$2F,$F4,$4F,$85,$F5,$36,$33,$18,$3C,$3F,$21,$23
	.dc.b	$BB,$5F,$F1,$CD,$FF,$B7,$9F,$1F,$93,$E8,$23,$E0,$F1,$3B,$91,$C1
	.dc.b	$FE,$2A,$F9,$BF,$F5,$8B,$78,$FE,$74,$62,$3F,$C8,$ED,$7F,$C9,$9F
	.dc.b	$3F,$FE,$06,$7F,$C6,$6D,$7F,$C9,$AD,$77,$E3,$7C,$CF,$A7,$91,$F2
	.dc.b	$9A,$F0,$F8,$3F,$BD,$46,$92,$3E,$8F,$27,$D6,$23,$83,$C7,$EF,$72
	.dc.b	$23,$83,$CF,$EF,$7F,$83,$3C,$8F,$F7,$2E,$1F,$EA,$B9,$FF,$09,$46
	.dc.b	$27,$5D,$9F,$E1,$2A,$78,$B8,$BF,$F0,$C7,$0B,$BB,$10,$8C,$3D,$2F
	.dc.b	$FC,$37,$18,$86,$3E,$2F,$7B,$8E,$2F,$16,$C4,$AE,$47,$F8,$9E,$74
	.dc.b	$E0,$70,$FC,$97,$18,$7E,$3D,$EF,$8F,$8B,$9B,$CF,$5B,$9C,$9F,$E3
	.dc.b	$3E,$B7,$C9,$FB,$1B,$1D,$DA,$F0,$0B,$E4,$F8,$7B,$32,$78,$3C,$FD
	.dc.b	$AF,$8F,$9B,$D2,$BE,$94,$BF,$4C,$78,$7D,$AF,$17,$B5,$27,$C5,$1E
	.dc.b	$5F,$6B,$CD,$F4,$12,$7C,$D1,$DB,$DC,$91,$F1,$1F,$0F,$C8,$7B,$7D
	.dc.b	$C9,$3E,$D7,$E0,$7D,$07,$C1,$F9,$39,$3F,$05,$F8,$9F,$27,$EA,$F9
	.dc.b	$3F,$5F,$8D,$2B,$E8,$C7,$E1,$31,$FE,$11,$1F,$1D,$79,$59,$35,$E5
	.dc.b	$EF,$CA,$E1,$E5,$4B,$FB,$D6,$9F,$93,$EE,$C9,$FB,$EF,$CC,$FA,$0F
	.dc.b	$07,$91,$E1,$F7,$E5,$72,$8F,$27,$AF,$E5,$F8,$A9,$3D,$A3,$F0,$BE
	.dc.b	$1C,$8F,$C7,$78,$48,$FC,$4F,$9D,$ED,$EC,$95,$E6,$7E,$07,$6F,$C1
	.dc.b	$E9,$93,$EA,$7E,$27,$07,$E4,$BE,$03,$F2,$7E,$B4,$AE,$F1,$FA,$CF
	.dc.b	$A0,$F9,$3E,$5F,$86,$7F,$D1,$11,$8D,$3F,$D0,$ED,$8F,$F4,$67,$D0
	.dc.b	$7E,$E1,$FD,$32,$F7,$BF,$6E,$D8,$CC,$63,$E4,$B3,$E6,$C9,$E2,$F0
	.dc.b	$F3,$B8,$FB,$92,$7A,$BD,$3E,$2A,$47,$81,$AF,$E9,$F1,$7C,$5F,$F0
	.dc.b	$8F,$9F,$FD,$44,$FE,$EF,$E8,$1E,$1F,$F0,$C6,$C7,$EF,$E7,$6F,$EC
	.dc.b	$57,$9D,$CF,$CF,$93,$B8,$FE,$2B,$F0,$FF,$16,$1E,$5B,$4F,$E3,$2D
	.dc.b	$8F,$E1,$93,$B3,$F2,$EE,$3C,$5E,$67,$4F,$1B,$F9,$0F,$CF,$BE,$33
	.dc.b	$9F,$B1,$2F,$F8,$B4,$E5,$BF,$41,$C2,$A4,$60,$DF,$AD,$D2,$64,$60
	.dc.b	$E7,$B7,$A3,$9A,$46,$0E,$FB,$F9,$26,$46,$0F,$1F,$AB,$FC,$C9,$FF
	.dc.b	$67,$9F,$ED,$3C,$91,$98,$CC,$63,$B3,$71,$B9,$D3,$FE,$D8,$97,$CF
	.dc.b	$BE,$83,$72,$9C,$C6,$1B,$1D,$1A,$F0,$FA,$0F,$F1,$12,$69,$1E,$BF
	.dc.b	$0D,$FA,$5C,$7F,$4D,$1F,$EE,$D8,$C7,$81,$E8,$AF,$F7,$87,$D0,$39
	.dc.b	$7E,$0D,$6A,$FF,$DE,$3C,$B5,$FF,$7B,$35,$FF,$1C,$61,$FD,$45,$FF
	.dc.b	$14,$37,$97,$47,$1E,$67,$C6,$E2,$78,$7A,$3C,$CC,$5F,$F4,$B3,$F8
	.dc.b	$F4,$75,$F4,$BC,$06,$DC,$1F,$D8,$76,$FE,$AE,$DE,$7F,$07,$87,$E9
	.dc.b	$9C,$DF,$F5,$77,$FA,$B5,$71,$E5,$D5,$ED,$F8,$1F,$41,$EE,$BF,$EB
	.dc.b	$CD,$BF,$9B,$A3,$C2,$DF,$07,$0E,$09,$9F,$F7,$5C,$38,$2C,$64,$7C
	.dc.b	$87,$AF,$9D,$E0,$3C,$3E,$8D,$AE,$51,$E4,$F4,$31,$5E,$47,$CF,$EF
	.dc.b	$7A,$3C,$FB,$7E,$17,$0F,$8F,$CF,$B7,$EC,$7D,$FE,$7C,$9D,$1E,$97
	.dc.b	$F0,$55,$E9,$8F,$F0,$83,$EC,$AF,$63,$97,$FC,$08,$E5,$FE,$FD,$87
	.dc.b	$2F,$80,$E6,$DA,$E7,$E4,$76,$3B,$FA,$B6,$3F,$C3,$DF,$3E,$F6,$FF
	.dc.b	$28,$D8,$CF,$B7,$89,$FC,$C7,$CF,$BE,$1A,$98,$F6,$6D,$FF,$8A,$65
	.dc.b	$C1,$C3,$83,$8B,$67,$FC,$5D,$F3,$FF,$CB,$9B,$E4,$FE,$96,$3F,$83
	.dc.b	$C6,$0D,$E7,$93,$5E,$F5,$8F,$F1,$C5,$BC,$0F,$D7,$3F,$B3,$8C,$C6
	.dc.b	$3C,$4F,$D9,$7C,$FB,$CC,$BE,$AE,$1B,$7F,$E4,$3F,$9F,$79,$FF,$B9
	.dc.b	$7F,$43,$E1,$FF,$04,$73,$3B,$F1,$75,$F0,$72,$39,$78,$B7,$B7,$FC
	.dc.b	$7D,$F3,$CF,$77,$F8,$23,$6B,$FC,$A4,$BE,$07,$33,$FC,$27,$E7,$9F
	.dc.b	$2B,$4C,$79,$D7,$0F,$F2,$D4,$FB,$5B,$C3,$C7,$87,$93,$FC,$4F,$E7
	.dc.b	$9F,$0D,$79,$30,$DB,$FF,$32,$7C,$F7,$F0,$AD,$FF,$CA,$CF,$CF,$8C
	.dc.b	$FA,$3F,$C7,$63,$13,$AF,$1F,$06,$7C,$9B,$7F,$E6,$FF,$9E,$F7,$3F
	.dc.b	$E7,$1F,$9D,$F4,$5F,$F9,$8E,$72,$FF,$99,$63,$3D,$3F,$CD,$1B,$5F
	.dc.b	$C0,$E5,$1F,$E8,$18,$E9,$BF,$F3,$18,$C4,$F3,$DF,$06,$66,$D7,$8F
	.dc.b	$E0,$38,$1C,$7D,$EB,$19,$3D,$1F,$3D,$8C,$3C,$CF,$F3,$FF,$9D,$F1
	.dc.b	$CF,$FA,$35,$FE,$F8,$DC,$B5,$EF,$C8,$D3,$C6,$31,$7F,$E8,$5A,$FD
	.dc.b	$43,$7C,$6F,$E9,$9B,$3E,$2B,$C1,$13,$1F,$07,$C0,$5B,$E4,$F9,$49
	.dc.b	$1E,$59,$B3,$DF,$BF,$D4,$7A,$FE,$32,$D7,$AE,$3D,$DE,$06,$2B,$CC
	.dc.b	$FF,$5F,$F9,$E7,$C4,$FC,$3F,$9A,$C3,$70,$F8,$9E,$8F,$C7,$31,$87
	.dc.b	$BD,$FE,$D1,$F3,$DF,$C8,$FA,$3F,$B9,$B6,$BF,$01,$8B,$E5,$B7,$84
	.dc.b	$9F,$F5,$E5,$7F,$29,$38,$47,$F4,$97,$1E,$EF,$F7,$E8,$FC,$54,$7F
	.dc.b	$57,$7F,$C3,$6D,$73,$1B,$79,$79,$9E,$63,$CF,$DF,$6F,$C3,$1B,$95
	.dc.b	$FE,$99,$EB,$19,$8C,$47,$8B,$DB,$C1,$2B,$FC,$43,$28,$E1,$ED,$FC
	.dc.b	$63,$E3,$B6,$7E,$0F,$B7,$D4,$8F,$26,$DF,$FB,$46,$FF,$9C,$CF,$03
	.dc.b	$E4,$FA,$1F,$48,$8E,$67,$84,$7D,$6F,$6A,$4F,$D0,$F3,$2D,$75,$9F
	.dc.b	$71,$27,$D6,$FC,$5F,$9E,$7C,$7F,$F8,$9D,$FC,$4C,$FF,$BD,$5C,$B7
	.dc.b	$93,$53,$B6,$C6,$4E,$1E,$6F,$33,$FC,$65,$F3,$BF,$D6,$E3,$24,$C7
	.dc.b	$FD,$E5,$87,$F2,$E8,$FF,$07,$72,$FF,$90,$AB,$85,$6A,$FA,$57,$1A
	.dc.b	$D7,$B7,$8A,$D4,$E4,$FE,$2D,$7C,$AB,$FC,$9B,$87,$7A,$D9,$23,$9D
	.dc.b	$71,$8F,$F1,$04,$EE,$78,$E1,$EC,$C3,$C1,$18,$BE,$16,$C3,$C1,$1F
	.dc.b	$C9,$F0,$E4,$DC,$38,$BE,$6C,$2D,$F8,$26,$BC,$E2,$D7,$28,$D5,$7C
	.dc.b	$83,$91,$E2,$8D,$41,$B7,$E4,$8E,$27,$C1,$9F,$21,$CD,$FF,$20,$46
	.dc.b	$DE,$57,$E2,$3E,$14,$F7,$7C,$F3,$BA,$F0,$92,$37,$5D,$CF,$0E,$1F
	.dc.b	$25,$3F,$49,$D1,$E0,$BF,$9B,$7D,$6B,$C5,$1E,$79,$F8,$CF,$8C,$DE
	.dc.b	$1A,$8E,$75,$C6,$75,$1E,$07,$E4,$BE,$0C,$3D,$11,$D6,$78,$4E,$6F
	.dc.b	$D7,$87,$A4,$95,$F2,$A7,$A1,$F2,$B0,$F1,$3F,$EB,$8C,$3F,$24,$DB
	.dc.b	$FD,$A1,$2B,$FD,$63,$3A,$9E,$86,$E3,$33,$FB,$9F,$3F,$F0,$A4,$8F
	.dc.b	$C1,$36,$7D,$8D,$C3,$CF,$1F,$9E,$8C,$4F,$A6,$7F,$4F,$18,$9F,$AF
	.dc.b	$3F,$9B,$8C,$47,$D5,$3F,$76,$E2,$FF,$02,$DB,$F3,$57,$90,$95,$E4
	.dc.b	$6D,$F9,$23,$F3,$15,$C6,$7F,$8A,$36,$3B,$C7,$03,$D0,$79,$4C,$3E
	.dc.b	$83,$DC,$E1,$F4,$4F,$BA,$31,$7E,$86,$DF,$A2,$BC,$C4,$AF,$35,$7F
	.dc.b	$B1,$AB,$67,$98,$B3,$E6,$38,$3E,$67,$C6,$60,$F3,$1F,$01,$FE,$65
	.dc.b	$F1,$5F,$CD,$4F,$E6,$DF,$2D,$63,$24,$9F,$E5,$71,$97,$F2,$27,$F1
	.dc.b	$3E,$3F,$EC,$9A,$7F,$5B,$AF,$F7,$EB,$67,$FC,$78,$7F,$00,$9F,$E8
	.dc.b	$DE,$54,$AF,$C9,$B6,$76,$3E,$4B,$F2,$CF,$FB,$1D,$AE,$74,$FE,$E9
	.dc.b	$18,$F8,$47,$77,$F7,$76,$E1,$D2,$FF,$DA,$D3,$93,$D3,$E7,$49,$F4
	.dc.b	$B6,$7C,$7F,$4E,$F9,$DF,$F0,$13,$6B,$A3,$67,$73,$FE,$07,$F9,$EF
	.dc.b	$F0,$E3,$E1,$6C,$7B,$97,$CA,$BC,$0D,$78,$3D,$EB,$E1,$FC,$C3,$E7
	.dc.b	$7D,$50,$8C,$3E,$D6,$BC,$23,$7F,$32,$F8,$BA,$49,$F8,$91,$97,$E3
	.dc.b	$7D,$3F,$F8,$7D,$34,$9B,$78,$BE,$79,$EC,$52,$57,$CF,$8C,$BD,$BC
	.dc.b	$8F,$19,$1C,$1E,$2F,$23,$CE,$47,$07,$3F,$0F,$C0,$47,$07,$DD,$E7
	.dc.b	$FF,$25,$37,$7F,$8F,$18,$8F,$C6,$B5,$FC,$AF,$CE,$7F,$86,$1F,$C3
	.dc.b	$35,$FC,$9B,$5D,$F8,$9D,$3C,$29,$1E,$16,$BC,$3E,$7B,$EB,$D1,$A4
	.dc.b	$8F,$A3,$F1,$7F,$58,$8E,$0F,$1F,$CF,$72,$23,$83,$CF,$E1,$7F,$AA
	.dc.b	$7C,$57,$17,$FA,$AF,$14,$E6,$30,$DA,$09,$78,$99,$87,$2B,$BB,$0D
	.dc.b	$8F,$EC,$CD,$7F,$82,$D7,$FB,$D5,$FE,$D4,$BC,$2E,$8E,$63,$7E,$E6
	.dc.b	$27,$3E,$E7,$87,$89,$F5,$34,$CD,$F4,$9A,$F8,$1E,$BE,$E7,$1E,$09
	.dc.b	$3F,$DE,$ED,$1D,$76,$F7,$75,$FE,$17,$87,$E4,$DF,$C4,$8F,$B0,$FF
	.dc.b	$80,$DA,$FF,$22,$78,$70,$FF,$B6,$5F,$F7,$EC,$6F,$93,$A9,$DB,$C6
	.dc.b	$F6,$71,$C9,$F2,$3E,$FF,$93,$F1,$E5,$27,$C0,$F9,$A7,$CF,$32,$7C
	.dc.b	$8F,$AB,$E7,$7E,$41,$F2,$78,$A4,$FC,$18,$F7,$FD,$0F,$C0,$F2,$64
	.dc.b	$FB,$63,$E1,$FC,$07,$C4,$E3,$93,$F0,$63,$EA,$72,$C8,$F9,$8F,$D7
	.dc.b	$AF,$C4,$7D,$47,$F1,$2E,$2F,$97,$48,$C7,$17,$FB,$FD,$B1,$F9,$27
	.dc.b	$FE,$00,$BF,$E3,$15,$F0,$3D,$09,$19,$F8,$BF,$C3,$69,$F8,$7C,$E4
	.dc.b	$FD,$F8,$F1,$FB,$B2,$3A,$3E,$6F,$7B,$CF,$F3,$92,$7C,$0F,$AB,$E8
	.dc.b	$FF,$25,$C1,$F9,$3E,$B9,$5F,$8B,$8F,$7F,$BD,$F0,$3E,$1A,$4F,$A6
	.dc.b	$3E,$1F,$C3,$FC,$4F,$52,$4F,$B6,$3F,$5D,$FB,$27,$F3,$F8,$FE,$DA
	.dc.b	$DE,$4F,$92,$DE,$4F,$DB,$CA,$3E,$4A,$FA,$1F,$DC,$BF,$E5,$9F,$A2
	.dc.b	$FE,$78,$FF,$78,$6D,$FD,$49,$DC,$FF,$7C,$5D,$3F,$C0,$57,$5D,$31
	.dc.b	$19,$EB,$95,$FE,$67,$AF,$DE,$5F,$5E,$BE,$5C,$D2,$B4,$F5,$E7,$91
	.dc.b	$C5,$AF,$8F,$8B,$F8,$97,$D0,$BE,$73,$F8,$A7,$D1,$3C,$EA,$6A,$37
	.dc.b	$D7,$70,$FF,$3D,$5F,$16,$C7,$F1,$DF,$94,$F8,$7C,$9F,$33,$9D,$A7
	.dc.b	$F2,$16,$DF,$FA,$0E,$B6,$FF,$28,$5F,$3D,$CF,$D5,$93,$B8,$FE,$55
	.dc.b	$F4,$3F,$9A,$8F,$F4,$65,$FD,$0E,$2F,$D2,$B7,$2F,$BB,$5B,$AF,$F1
	.dc.b	$7A,$F0,$39,$F8,$32,$7F,$D2,$D3,$96,$FC,$AF,$82,$A4,$60,$DF,$07
	.dc.b	$B7,$9E,$46,$0E,$7C,$1E,$89,$91,$83,$BF,$85,$FE,$AC,$7F,$C6,$F8
	.dc.b	$A3,$11,$FE,$40,$8C,$6B,$FE,$31,$6C,$70,$F0,$9F,$45,$96,$C6,$63
	.dc.b	$07,$F8,$81,$7D,$15,$23,$FE,$24,$7F,$C4,$BE,$E7,$F3,$AC,$3F,$88
	.dc.b	$CF,$F8,$B7,$83,$FD,$78,$FF,$60,$AF,$EC,$3F,$44,$E5,$FF,$2B,$E1
	.dc.b	$C6,$FF,$CB,$3C,$B7,$0F,$D8,$B5,$FE,$D3,$35,$FF,$17,$4F,$6E,$5C
	.dc.b	$1B,$E5,$FF,$3E,$4F,$F9,$A8,$FC,$4C,$F9,$3D,$9E,$47,$07,$07,$A3
	.dc.b	$D9,$EC,$69,$C1,$FE,$D7,$D5,$CD,$E5,$6C,$7E,$43,$C1,$C4,$E7,$92
	.dc.b	$4F,$18,$B8,$FF,$9F,$BE,$8B,$E4,$37,$5F,$57,$7F,$C1,$9C,$E5,$FC
	.dc.b	$F9,$F8,$35,$C7,$7D,$5E,$DE,$1F,$19,$32,$FF,$77,$5F,$FA,$1F,$0E
	.dc.b	$53,$9C,$3C,$5F,$45,$67,$B3,$E1,$F7,$BC,$5F,$1B,$71,$E6,$E1,$EB
	.dc.b	$F1,$97,$0F,$63,$E9,$F9,$F9,$3A,$3C,$6F,$DD,$AF,$F1,$2C,$7F,$89
	.dc.b	$B1,$FE,$B7,$1F,$02,$3F,$99,$BA,$BF,$F5,$2B,$97,$FC,$5A,$B1,$E9
	.dc.b	$EE,$75,$6D,$75,$39,$F8,$9E,$13,$C5,$F4,$56,$35,$87,$F8,$E3,$E8
	.dc.b	$9F,$25,$F9,$36,$CE,$A5,$0F,$A0,$D5,$7E,$67,$E8,$9F,$21,$4C,$7C
	.dc.b	$1D,$C7,$94,$B8,$78,$70,$F1,$6C,$F7,$FA,$27,$8F,$FA,$56,$C6,$7C
	.dc.b	$7C,$4E,$74,$93,$C7,$E8,$AE,$FF,$EC,$EC,$5D,$FF,$AF,$7F,$6D,$5E
	.dc.b	$D7,$2D,$F0,$FF,$69,$EA,$BC,$AB,$E8,$E1,$B5,$F1,$7E,$8B,$F9,$63
	.dc.b	$7C,$2F,$DE,$46,$33,$FF,$6F,$BF,$AA,$6F,$CC,$68,$DF,$8F,$6F,$E8
	.dc.b	$7D,$13,$E8,$BF,$83,$36,$33,$F3,$18,$9F,$E1,$5F,$44,$F8,$4A,$63
	.dc.b	$E1,$6D,$FD,$B9,$F6,$37,$7E,$3B,$F2,$7F,$8A,$7D,$13,$E0,$2F,$16
	.dc.b	$1B,$7F,$85,$FA,$27,$A1,$FC,$6D,$F9,$7C,$5F,$EF,$E7,$33,$BE,$7D
	.dc.b	$78,$D2,$39,$79,$36,$DE,$4F,$F9,$03,$E8,$1C,$9F,$C9,$FE,$8B,$D3
	.dc.b	$5A,$C3,$FC,$37,$5A,$7F,$31,$3A,$BF,$F2,$4B,$FE,$89,$8C,$F0,$73
	.dc.b	$E1,$E8,$C8,$FF,$7F,$4E,$F0,$FF,$13,$9F,$E9,$E7,$0E,$1A,$FF,$8B
	.dc.b	$17,$BD,$FD,$22,$38,$35,$46,$A7,$BC,$E6,$31,$3E,$D3,$C1,$F0,$9F
	.dc.b	$06,$7E,$13,$FD,$27,$E8,$B9,$DF,$ED,$DF,$F2,$FB,$6F,$5E,$EC,$8F
	.dc.b	$5E,$1F,$EB,$8A,$DB,$86,$BF,$01,$AF,$F9,$21,$FD,$9B,$BF,$0F,$D9
	.dc.b	$18,$8E,$11,$8F,$13,$CC,$79,$DB,$5E,$78,$D3,$E4,$95,$7F,$80,$E3
	.dc.b	$0D,$8F,$37,$77,$80,$93,$A7,$C7,$DD,$23,$8B,$67,$DD,$F0,$BE,$82
	.dc.b	$4F,$67,$D7,$E5,$C8,$F0,$B6,$7E,$4F,$CF,$FF,$37,$7F,$0A,$D7,$FC
	.dc.b	$DA,$D7,$FD,$CA,$D4,$FF,$64,$78,$FA,$BF,$85,$8C,$47,$28,$FC,$47
	.dc.b	$CE,$FF,$95,$EF,$F9,$A3,$67,$F0,$C7,$49,$FF,$06,$46,$79,$3F,$D5
	.dc.b	$37,$FB,$4A,$F3,$F4,$F7,$26,$1F,$46,$7F,$73,$5C,$3C,$6F,$F0,$B4
	.dc.b	$62,$7D,$B1,$8F,$52,$47,$F4,$79,$FE,$E4,$FC,$28,$CE,$1F,$21,$F7
	.dc.b	$12,$3C,$1E,$A7,$6F,$27,$BB,$63,$A4,$7C,$52,$3E,$76,$D7,$C2,$FA
	.dc.b	$A4,$F1,$8C,$3E,$AF,$53,$D7,$EA,$58,$F4,$38,$7E,$13,$F4,$C8,$FE
	.dc.b	$57,$E7,$DF,$41,$F3,$B5,$FF,$1D,$46,$27,$EA,$C6,$5A,$7F,$6B,$8D
	.dc.b	$61,$F6,$38,$3F,$A9,$BF,$6A,$3F,$C7,$BB,$FF,$B8,$1B,$5F,$EB,$87
	.dc.b	$F9,$F3,$CF,$C9,$F0,$91,$ED,$E5,$5B,$E9,$36,$FA,$4D,$AE,$93,$68
	.dc.b	$2F,$BB,$CA,$7C,$04,$7D,$15,$C5,$E1,$56,$B8,$DF,$AE,$30,$7C,$8A
	.dc.b	$B7,$CE,$7C,$AD,$3C,$A5,$BF,$2D,$F4,$8C,$C7,$94,$93,$C9,$B1,$A8
	.dc.b	$F1,$3F,$18,$E9,$1E,$86,$CF,$58,$EF,$7E,$67,$DF,$3A,$8F,$75,$E6
	.dc.b	$FA,$D6,$8F,$09,$CA,$BA,$CE,$DF,$21,$F7,$5C,$46,$CE,$53,$D0,$93
	.dc.b	$C2,$3E,$1C,$F0,$7D,$47,$08,$D3,$89,$E2,$F0,$8F,$51,$E4,$2B,$E8
	.dc.b	$3F,$05,$3D,$6F,$F2,$71,$C4,$D3,$FE,$94,$7F,$B3,$B5,$FA,$A6,$2F
	.dc.b	$FB,$14,$66,$3D,$C6,$6B,$F5,$D3,$A3,$FD,$4D,$C8,$F9,$6F,$C7,$36
	.dc.b	$7E,$CB,$6B,$D0,$DC,$7D,$F1,$FE,$5E,$AE,$24,$AE,$2D,$8E,$2D,$9E
	.dc.b	$35,$CE,$BE,$5B,$6B,$EB,$4F,$F1,$07,$F5,$31,$97,$E8,$4F,$EB,$1F
	.dc.b	$13,$E0,$23,$E0,$6C,$F8,$2B,$C5,$1E,$06,$CF,$82,$3A,$57,$7C,$3E
	.dc.b	$AC,$F6,$3A,$12,$FA,$57,$63,$A1,$63,$A1,$EA,$BE,$EF,$A8,$C4,$6D
	.dc.b	$C1,$DE,$7A,$BD,$E3,$AB,$4E,$A6,$DC,$C7,$02,$38,$24,$70,$BF,$76
	.dc.b	$1F,$84,$BF,$E6,$3E,$EF,$E3,$09,$7F,$CD,$DF,$F1,$85,$7E,$EE,$FF
	.dc.b	$D3,$F9,$70,$AF,$F4,$F4,$E7,$27,$C7,$7E,$58,$93,$FD,$05,$FF,$1D
	.dc.b	$7C,$67,$F8,$C3,$0F,$F1,$5B,$5F,$F2,$3B,$5F,$C0,$35,$FE,$5B,$BB
	.dc.b	$E7,$BF,$C8,$8D,$BF,$F1,$13,$67,$FC,$6E,$D7,$FC,$A7,$BB,$C9,$FF
	.dc.b	$02,$92,$BF,$C5,$51,$97,$FC,$59,$F2,$FF,$E1,$12,$57,$F9,$4D,$FF
	.dc.b	$0A,$7C,$F3,$E6,$BC,$93,$FE,$46,$8C,$65,$FE,$35,$5F,$3C,$EE,$C5
	.dc.b	$F0,$BF,$F3,$3F,$D2,$BD,$EF,$F1,$E4,$23,$0F,$F8,$FD,$AF,$F9,$51
	.dc.b	$AF,$F9,$C3,$27,$C1,$F2,$6C,$7B,$23,$07,$6F,$5F,$FD,$0F,$1F,$E4
	.dc.b	$B6,$BF,$E6,$06,$BF,$E7,$C6,$BF,$B5,$F9,$D7,$A5,$E8,$6C,$7F,$95
	.dc.b	$5A,$FF,$9A,$9A,$FF,$A1,$BE,$57,$FD,$D6,$FF,$96,$DA,$FF,$9C,$1A
	.dc.b	$EF,$9F,$E2,$FC,$C3,$E7,$69,$27,$FC,$C9,$19,$7C,$5E,$AF,$94,$8E
	.dc.b	$0F,$F3,$57,$D5,$7D,$56,$F0,$FD,$28,$C4,$7F,$A0,$9A,$FF,$A7,$BE
	.dc.b	$7B,$FD,$2C,$9A,$4E,$6F,$B7,$F5,$3F,$ED,$59,$FF,$3C,$35,$FF,$46
	.dc.b	$B5,$FF,$54,$B5,$FE,$44,$BE,$9B,$99,$23,$9B,$F8,$8A,$FF,$6E,$47
	.dc.b	$FA,$15,$AF,$FA,$69,$AF,$A7,$BF,$F1,$FF,$80,$7A,$89,$86,$0E,$9F
	.dc.b	$3D,$2B,$A3,$5E,$9F,$3D,$E1,$23,$E1,$7C,$9E,$DC,$8F,$23,$5F,$27
	.dc.b	$B7,$E9,$23,$E9,$7D,$9E,$DC,$8F,$63,$5F,$67,$B7,$F0,$48,$FC,$17
	.dc.b	$F5,$7F,$40,$C6,$92,$7F,$D5,$51,$97,$F1,$3D,$DF,$D4,$7C,$BF,$EA
	.dc.b	$D0,$E5,$FE,$D8,$FF,$90,$23,$31,$88,$FF,$46,$BF,$C1,$5D,$6B,$FD
	.dc.b	$6E,$3E,$9C,$F0,$F2,$FB,$4E,$5B,$5F,$E4,$CA,$DC,$FF,$63,$C1,$B6
	.dc.b	$A9,$18,$7B,$9F,$E5,$47,$0F,$FA,$81,$79,$5C,$6F,$E0,$4F,$A3,$3C
	.dc.b	$19,$BD,$54,$9E,$A7,$C3,$79,$7C,$EF,$8F,$03,$78,$25,$70,$9C,$BC
	.dc.b	$B1,$FE,$92,$DF,$91,$F2,$1E,$5F,$2F,$CD,$F3,$5E,$7E,$29,$5C,$63
	.dc.b	$C1,$C7,$23,$B3,$E3,$AF,$8D,$DC,$FA,$2A,$4A,$F5,$C3,$E0,$37,$F0
	.dc.b	$12,$72,$F2,$F9,$BF,$76,$9E,$FC,$E5,$7B,$63,$BF,$C0,$C8,$EA,$F8
	.dc.b	$BE,$07,$C7,$F3,$52,$7C,$2F,$9B,$E8,$3E,$57,$07,$CB,$EC,$95,$F2
	.dc.b	$63,$D7,$F0,$3E,$CF,$87,$93,$EA,$8F,$7F,$C4,$7C,$0F,$83,$93,$EE
	.dc.b	$8F,$C4,$7B,$1F,$D4,$7C,$0F,$CE,$B7,$BF,$FB,$8F,$AB,$AE,$7F,$C7
	.dc.b	$7C,$FF,$E2,$AF,$F4,$7F,$41,$FD,$33,$2F,$F7,$5B,$63,$F4,$F3,$BC
	.dc.b	$3F,$C7,$4F,$E0,$2B,$FD,$E2,$FE,$49,$FC,$5B,$5C,$F7,$F8,$3B,$E4
	.dc.b	$69,$F1,$F6,$F9,$39,$E4,$F2,$7C,$FD,$12,$3A,$B5,$F0,$F1,$FE,$A7
	.dc.b	$E7,$DF,$35,$FA,$BF,$A0,$7C,$15,$35,$1B,$EC,$B8,$7E,$8A,$F8,$B6
	.dc.b	$3F,$69,$F0,$3F,$0F,$83,$FC,$B5,$D0,$D3,$F7,$0D,$BF,$D5,$D6,$DF
	.dc.b	$F6,$5A,$F9,$CE,$7E,$C4,$9D,$C7,$F0,$1F,$A0,$78,$5D,$3C,$99,$5F
	.dc.b	$B6,$9C,$B7,$C3,$E1,$52,$30,$6F,$E5,$3A,$4C,$8C,$1C,$F8,$3D,$3C
	.dc.b	$D2,$30,$77,$E0,$F2,$4C,$8C,$1E,$3F,$11,$D5,$FE,$7D,$8F,$F3,$F6
	.dc.b	$69,$CC,$64,$46,$67,$11,$89,$DB,$CE,$5E,$23,$E8,$37,$18,$6C,$66
	.dc.b	$7B,$4E,$5F,$F1,$8F,$D0,$7F,$8C,$93,$48,$F8,$BD,$6F,$F4,$D6,$3F
	.dc.b	$D1,$A3,$FC,$7D,$CD,$FC,$99,$1C,$87,$D0,$39,$7E,$15,$F2,$AF,$89
	.dc.b	$CB,$6F,$FD,$44,$D7,$9C,$D7,$FC,$7F,$3D,$F9,$70,$6F,$9B,$FD,$3B
	.dc.b	$3F,$8B,$9F,$27,$C9,$F1,$36,$E0,$F4,$7C,$7F,$53,$4E,$0F,$E8,$3A
	.dc.b	$39,$BC,$8D,$8F,$E8,$B8,$7F,$4E,$77,$3A,$E3,$91,$83,$9B,$ED,$FA
	.dc.b	$0F,$93,$97,$E1,$A7,$FD,$85,$18,$C3,$FB,$9E,$0E,$FE,$1A,$F1,$DA
	.dc.b	$61,$FD,$5B,$0E,$0F,$28,$ED,$F1,$F2,$35,$F4,$B7,$3F,$01,$F6,$2D
	.dc.b	$5E,$5F,$A1,$D6,$9F,$E0,$49,$47,$A7,$E0,$BD,$5F,$23,$72,$F1,$38
	.dc.b	$7C,$9F,$23,$72,$EE,$30,$47,$C3,$F9,$E9,$5C,$0F,$63,$F8,$3B,$FC
	.dc.b	$0D,$F2,$6E,$5F,$E1,$87,$F0,$11,$9D,$7F,$15,$E6,$7F,$76,$C3,$FB
	.dc.b	$DB,$67,$C6,$FD,$96,$F9,$3F,$91,$FB,$E7,$85,$4C,$78,$77,$0F,$F0
	.dc.b	$2F,$D1,$3E,$7B,$F3,$0D,$8C,$FD,$4E,$27,$A1,$23,$FC,$1B,$83,$E8
	.dc.b	$31,$9F,$C6,$FB,$EF,$A7,$5F,$E1,$26,$C6,$79,$FF,$4D,$18,$E1,$FF
	.dc.b	$1C,$FD,$17,$F0,$4B,$FF,$0C,$B6,$FE,$8A,$F6,$7F,$5C,$E6,$BF,$C8
	.dc.b	$35,$FC,$B8,$FF,$10,$96,$BE,$CB,$FE,$DD,$8D,$ED,$FB,$78,$7A,$D4
	.dc.b	$C7,$B1,$63,$FC,$57,$1F,$EE,$A6,$FB,$9F,$BF,$8F,$CF,$46,$0F,$F1
	.dc.b	$8F,$DE,$35,$FE,$08,$D8,$CF,$C8,$E2,$73,$F1,$12,$78,$FC,$CD,$D7
	.dc.b	$F2,$3F,$45,$FD,$82,$7F,$C7,$AD,$8C,$CF,$F1,$18,$C6,$DF,$E6,$4F
	.dc.b	$31,$EF,$FF,$15,$6E,$1F,$E4,$65,$EC,$E6,$7F,$8D,$7D,$E3,$EB,$29
	.dc.b	$8E,$5B,$87,$F9,$3B,$E8,$DF,$05,$FC,$85,$B1,$9F,$92,$C4,$FF,$25
	.dc.b	$F2,$9E,$7D,$31,$E8,$5B,$FF,$2C,$CB,$E5,$38,$1C,$7C,$8B,$3F,$E5
	.dc.b	$EF,$85,$FE,$F0,$DF,$77,$F9,$97,$D1,$7C,$28,$FE,$7B,$CF,$FE,$49
	.dc.b	$9F,$E6,$D5,$F0,$27,$97,$1E,$27,$2C,$8E,$DC,$76,$FF,$81,$DF,$CA
	.dc.b	$7F,$82,$FD,$17,$E3,$6B,$BB,$F9,$1E,$6F,$F4,$AB,$99,$FE,$9E,$E6
	.dc.b	$77,$E0,$7F,$49,$8C,$4E,$AF,$13,$9F,$76,$5F,$F9,$FB,$4F,$F6,$83
	.dc.b	$FE,$80,$C1,$3A,$C3,$CB,$EE,$DC,$BC,$6F,$7E,$7F,$07,$D1,$5C,$79
	.dc.b	$38,$69,$D6,$31,$7F,$02,$7D,$1E,$B6,$30,$F7,$BF,$D8,$FE,$B1,$F3
	.dc.b	$FE,$DE,$8C,$37,$2F,$47,$4F,$C1,$31,$97,$CC,$7F,$B6,$7D,$1F,$F7
	.dc.b	$69,$FF,$4E,$B6,$7F,$9A,$E0,$F9,$2D,$C4,$AD,$F4,$D8,$FE,$77,$7F
	.dc.b	$D4,$1F,$F0,$0D,$BE,$8B,$BF,$5F,$A5,$B8,$78,$3B,$3C,$66,$2F,$9B
	.dc.b	$88,$FF,$10,$E1,$F9,$E6,$C7,$F8,$46,$7F,$C1,$F3,$97,$FC,$55,$83
	.dc.b	$EA,$BD,$FE,$2F,$C9,$6E,$1F,$02,$39,$52,$FF,$86,$3F,$E3,$36,$BF
	.dc.b	$E3,$96,$BF,$A2,$5E,$B7,$26,$D7,$A2,$30,$47,$E7,$FA,$FF,$4A,$3E
	.dc.b	$9B,$F5,$3E,$6B,$91,$27,$E0,$BD,$7E,$27,$EC,$9B,$8F,$B2,$E1,$A7
	.dc.b	$DC,$8C,$61,$FD,$C3,$E8,$DF,$39,$FE,$2D,$6D,$7F,$B7,$BE,$93,$F6
	.dc.b	$5D,$08,$FD,$FC,$EA,$F9,$5F,$C8,$8C,$47,$F9,$75,$AF,$08,$F9,$BF
	.dc.b	$46,$FA,$9F,$07,$CE,$7F,$8F,$63,$18,$7F,$BB,$B0,$4E,$6B,$73,$FE
	.dc.b	$F2,$AD,$E1,$EB,$AF,$57,$CD,$4A,$FF,$7B,$F1,$FF,$13,$AF,$81,$39
	.dc.b	$6C,$EB,$C6,$C6,$9E,$F9,$FF,$2C,$E6,$BC,$DF,$F2,$0C,$3A,$5E,$ED
	.dc.b	$7B,$AF,$55,$F0,$9F,$D1,$E1,$5D,$C5,$C3,$9D,$7C,$E8,$D5,$F3,$AF
	.dc.b	$D9,$DE,$27,$83,$6B,$B9,$EB,$2C,$FB,$89,$87,$9F,$0E,$F1,$EE,$BE
	.dc.b	$4F,$6A,$E3,$5E,$DC,$3C,$4D,$AF,$E2,$58,$70,$8C,$57,$86,$FB,$BE
	.dc.b	$B3,$99,$F2,$67,$83,$98,$F0,$1C,$0F,$23,$C1,$F9,$2D,$7E,$4B,$70
	.dc.b	$F5,$C6,$0E,$45,$AF,$81,$5E,$08,$EA,$78,$23,$D2,$F0,$C3,$DF,$7E
	.dc.b	$73,$C0,$E2,$38,$CF,$49,$CE,$1F,$51,$A7,$96,$FC,$15,$E3,$8E,$8F
	.dc.b	$79,$E0,$D8,$FF,$5F,$DF,$E0,$63,$F5,$93,$FA,$F2,$4E,$9B,$1F,$EC
	.dc.b	$99,$E0,$FC,$DB,$FB,$5F,$1F,$F8,$C2,$5F,$B1,$B3,$F0,$5B,$3F,$10
	.dc.b	$E3,$3B,$24,$ED,$B1,$B7,$F8,$3C,$F2,$7C,$46,$0D,$98,$9F,$D7,$C6
	.dc.b	$DF,$9B,$1B,$7B,$12,$7B,$3C,$23,$9B,$C0,$C1,$B7,$B3,$FB,$69,$EC
	.dc.b	$FD,$33,$73,$D1,$D4,$F4,$74,$60,$E8,$77,$31,$1C,$69,$5D,$9F,$C4
	.dc.b	$4E,$AB,$C7,$1A,$7B,$3A,$31,$1D,$8F,$0B,$8A,$ED,$5D,$6F,$F9,$D3
	.dc.b	$6F,$F9,$53,$6F,$FA,$03,$FE,$EB,$FA,$1F,$F4,$4C,$7F,$8E,$83,$87
	.dc.b	$FC,$7A,$D7,$FD,$F6,$D5,$A7,$F6,$6C,$3F,$91,$36,$F8,$FC,$37,$F3
	.dc.b	$DE,$6F,$E7,$DF,$0D,$FD,$08,$93,$E1,$C0,$74,$8F,$EE,$7F,$35,$FD
	.dc.b	$0A,$75,$28,$C3,$C5,$AF,$AD,$AF,$F9,$5B,$E6,$3F,$D3,$0F,$BD,$AE
	.dc.b	$7C,$AF,$F0,$0C,$62,$3C,$7F,$58,$F8,$6F,$F0,$2B,$70,$F1,$B6,$7D
	.dc.b	$4D,$7E,$67,$D2,$FF,$8A,$6B,$D2,$D8,$F8,$0B,$E9,$5E,$1E,$BC,$4F
	.dc.b	$05,$7C,$5E,$2F,$F7,$7B,$F0,$9A,$E7,$6F,$8A,$FD,$A6,$FD,$63,$E7
	.dc.b	$29,$27,$E5,$C6,$5F,$9B,$F3,$DF,$E2,$24,$D2,$6D,$DB,$E9,$5F,$1F
	.dc.b	$49,$5F,$5A,$32,$FD,$8F,$A5,$F2,$A6,$91,$F4,$E7,$FE,$9A,$7D,$DE
	.dc.b	$46,$08,$F5,$F1,$3C,$44,$70,$7E,$1B,$9B,$FC,$B0,$DD,$FE,$2C,$62
	.dc.b	$3F,$11,$1F,$23,$E9,$3F,$C4,$71,$F8,$E6,$BF,$95,$6B,$FA,$76,$BF
	.dc.b	$EA,$29,$7D,$1F,$62,$47,$8D,$ED,$E5,$C8,$EC,$D7,$F4,$5D,$0F,$8F
	.dc.b	$FB,$6D,$8F,$D0,$B5,$DC,$7A,$7E,$B2,$47,$A5,$AF,$A7,$C1,$FF,$5B
	.dc.b	$F8,$1F,$E1,$7C,$FF,$C3,$93,$FB,$08,$C4,$EB,$8F,$FD,$7B,$5F,$C6
	.dc.b	$AB,$51,$FC,$13,$EA,$35,$1F,$C1,$1A,$FF,$0A,$6C,$BE,$7A,$FD,$0E
	.dc.b	$19,$5D,$0F,$E0,$32,$9E,$3E,$37,$4E,$0F,$F3,$D5,$6C,$FE,$05,$E8
	.dc.b	$69,$C5,$FF,$BC,$1B,$59,$5A,$7C,$1F,$77,$CF,$C0,$F9,$9A,$4A,$ED
	.dc.b	$E0,$3D,$EF,$6A,$F1,$6C,$49,$DC,$74,$F9,$9D,$F2,$C9,$F4,$62,$9F
	.dc.b	$8F,$F8,$1F,$BE,$7D,$9D,$AF,$5B,$E5,$B6,$3E,$04,$ED,$F5,$F3,$7F
	.dc.b	$BC,$BE,$F1,$F4,$FF,$49,$FB,$13,$A3,$F9,$6B,$8E,$1F,$F2,$C3,$C2
	.dc.b	$3E,$67,$DD,$FB,$5B,$BF,$DC,$8F,$85,$1A,$DF,$CB,$E9,$F9,$BE,$3A
	.dc.b	$4F,$AA,$38,$7C,$AF,$1F,$72,$4F,$68,$FC,$3F,$DD,$BD,$1B,$84,$AF
	.dc.b	$BD,$39,$7E,$2F,$9B,$F1,$3C,$09,$18,$3F,$09,$D3,$F8,$67,$E0,$1F
	.dc.b	$77,$C4,$FC,$AC,$7E,$DA,$37,$DD,$8E,$CC,$F6,$C9,$E7,$1F,$4B,$DB
	.dc.b	$FA,$7F,$11,$27,$EB,$C7,$E8,$3E,$E5,$F2,$B7,$E0,$E5,$7E,$5E,$72
	.dc.b	$FE,$67,$EE,$7F,$8F,$3E,$67,$EF,$37,$BF,$F5,$71,$AE,$BF,$F4,$77
	.dc.b	$DB,$BE,$07,$F0,$BE,$74,$C3,$F4,$93,$97,$F4,$DF,$71,$FA,$37,$E1
	.dc.b	$AF,$8D,$FD,$BB,$63,$7C,$3F,$BA,$F3,$DF,$21,$7E,$9E,$57,$EC,$67
	.dc.b	$2F,$EC,$FB,$BF,$D4,$5A,$FF,$A8,$DF,$E5,$2D,$F8,$4F,$E0,$B1,$AF
	.dc.b	$1B,$FB,$FD,$3E,$E3,$E0,$DF,$FA,$AD,$7D,$2F,$F0,$C6,$CF,$F0,$17
	.dc.b	$75,$C9,$74,$6F,$A2,$56,$E3,$F8,$AF,$D6,$7E,$86,$FC,$3F,$5A,$FA
	.dc.b	$5F,$E3,$2D,$BF,$E1,$D5,$B7,$FC,$2A,$BE,$97,$9F,$99,$27,$71,$DB
	.dc.b	$D6,$91,$C9,$F0,$FA,$DE,$2F,$A7,$93,$D9,$F2,$F4,$48,$F0,$B5,$FC
	.dc.b	$DE,$3F,$6F,$87,$FD,$A1,$F5,$9F,$C2,$E7,$F9,$27,$D6,$BB,$3F,$99
	.dc.b	$B6,$3F,$94,$4E,$DF,$D2,$AF,$96,$FE,$72,$E6,$FA,$FA,$B8,$9F,$E7
	.dc.b	$BF,$55,$F1,$6F,$F5,$BE,$27,$E2,$5B,$A6,$B8,$F7,$D1,$67,$F9,$C5
	.dc.b	$F1,$CB,$FD,$D0,$FD,$C8,$FD,$BF,$D6,$7F,$11,$7A,$FC,$6F,$F4,$CC
	.dc.b	$BE,$94,$27,$57,$FE,$F1,$7F,$B2,$37,$D3,$FE,$A7,$1A,$E8,$FC,$7B
	.dc.b	$A6,$F9,$7A,$F6,$A6,$1F,$D3,$6B,$FA,$23,$78,$3C,$5C,$F2,$30,$6F
	.dc.b	$87,$BC,$C8,$C1,$CF,$E6,$3F,$A8,$BF,$E5,$FB,$46,$23,$FC,$E5,$18
	.dc.b	$D3,$FC,$BE,$D8,$E1,$E1,$3E,$B7,$2D,$8C,$C6,$0F,$F1,$7F,$CF,$C8
	.dc.b	$FF,$22,$B5,$FF,$38,$7C,$5B,$7F,$ED,$91,$FE,$52,$6A,$9F,$F3,$AF
	.dc.b	$D6,$B9,$3F,$D1,$17,$C6,$BF,$97,$9F,$AB,$BE,$7C,$D9,$D3,$53,$FE
	.dc.b	$7E,$AC,$57,$FA,$46,$AC,$FF,$04,$E6,$AF,$FA,$11,$BC,$BF,$E2,$EA
	.dc.b	$F3,$C6,$A7,$FB,$44,$7A,$BC,$DC,$4E,$5D,$7C,$34,$9D,$1F,$01,$EF
	.dc.b	$97,$F8,$0E,$3F,$C1,$CD,$BF,$CF,$F8,$3A,$E7,$DF,$24,$BD,$C5,$AF
	.dc.b	$EA,$3F,$5B,$F2,$DB,$B7,$AB,$C0,$F6,$CE,$72,$F8,$AF,$C9,$AE,$3B
	.dc.b	$EA,$F6,$F0,$F9,$19,$97,$F8,$8E,$FF,$D6,$17,$FE,$AA,$53,$A8,$F1
	.dc.b	$FA,$F7,$1E,$EF,$8B,$E2,$FC,$7F,$2F,$71,$E8,$F9,$BC,$99,$18,$3C
	.dc.b	$0F,$AB,$C9,$FB,$B8,$FF,$3C,$CF,$F9,$9D,$7F,$AE,$1B,$1F,$EB,$F6
	.dc.b	$BF,$ED,$1B,$F8,$0D,$D3,$9B,$FE,$C4,$8C,$BF,$D2,$1A,$FF,$B2,$3E
	.dc.b	$47,$F9,$A5,$FF,$B1,$9B,$1F,$EC,$D6,$BF,$ED,$F6,$BF,$D4,$BE,$AB
	.dc.b	$FC,$E1,$3F,$EC,$E6,$C7,$FB,$55,$AF,$FB,$9D,$AF,$F5,$AF,$AD,$7D
	.dc.b	$67,$A2,$11,$8A,$FF,$6C,$B6,$3F,$DC,$0D,$7A,$4F,$1A,$C4,$79,$63
	.dc.b	$07,$5F,$AC,$FF,$79,$13,$87,$07,$87,$EB,$1E,$05,$26,$5F,$EE,$A8
	.dc.b	$CB,$FE,$EC,$3F,$DC,$18,$7F,$03,$F1,$BD,$51,$82,$7D,$5E,$08,$F1
	.dc.b	$7E,$00,$FA,$CA,$48,$FF,$7C,$C6,$5F,$F7,$DF,$D6,$7F,$90,$5B,$F0
	.dc.b	$BF,$26,$31,$87,$F8,$18,$07,$2E,$57,$D6,$71,$25,$76,$7E,$A7,$D5
	.dc.b	$FF,$65,$8E,$95,$D2,$30,$E1,$AF,$85,$AF,$F8,$63,$C6,$7C,$17,$DF
	.dc.b	$6D,$74,$9F,$E5,$13,$93,$F0,$9F,$56,$FA,$7A,$4A,$E1,$19,$79,$7D
	.dc.b	$5B,$E9,$FF,$18,$D8,$CC,$60,$FC,$87,$82,$F1,$A9,$27,$C9,$19,$6F
	.dc.b	$D5,$F0,$23,$83,$D5,$F2,$1F,$E4,$76,$FA,$BF,$9D,$8C,$6B,$F2,$9A
	.dc.b	$FD,$75,$F0,$5B,$25,$74,$6C,$FF,$9D,$A7,$E4,$B6,$3F,$A0,$C6,$3C
	.dc.b	$6F,$E9,$3F,$55,$FE,$B3,$BF,$F6,$13,$FE,$B3,$6C,$FF,$48,$8C,$5F
	.dc.b	$FA,$09,$FC,$E9,$3A,$3F,$6C,$E3,$D4,$A4,$BF,$B5,$1F,$E2,$56,$FA
	.dc.b	$9D,$4E,$D3,$63,$C7,$F5,$5E,$13,$C5,$C7,$63,$27,$48,$F0,$91,$EF
	.dc.b	$E8,$F9,$08,$E0,$F1,$F1,$7A,$08,$E0,$F3,$FC,$0F,$B0,$8E,$0F,$CA
	.dc.b	$7D,$4F,$EA,$9B,$AF,$F1,$78,$C3,$FA,$46,$BF,$B1,$7F,$AD,$37,$8F
	.dc.b	$81,$27,$AF,$A5,$CC,$8E,$0F,$0F,$2F,$72,$38,$25,$7F,$AE,$2F,$FC
	.dc.b	$1D,$1F,$EF,$F7,$F3,$11,$9F,$53,$FC,$0B,$1F,$C0,$A3,$FD,$16,$D7
	.dc.b	$F8,$3B,$53,$4F,$EC,$B3,$D7,$C3,$E7,$DF,$D1,$EB,$E4,$F6,$F3,$6D
	.dc.b	$78,$DC,$3F,$31,$F9,$44,$7E,$7F,$BB,$E6,$3C,$FF,$11,$63,$D9,$1F
	.dc.b	$44,$8F,$02,$65,$F0,$FE,$B3,$83,$63,$F9,$1F,$57,$F8,$B3,$E9,$DF
	.dc.b	$0D,$FE,$28,$5F,$57,$63,$0C,$AF,$E2,$D3,$A6,$C7,$F5,$98,$C5,$FF
	.dc.b	$26,$5F,$55,$D7,$E4,$7B,$36,$FC,$67,$FA,$F2,$7F,$C6,$D5,$FE,$F2
	.dc.b	$6E,$3E,$7A,$E1,$39,$F8,$2F,$41,$E9,$F9,$FB,$87,$49,$FE,$83,$82
	.dc.b	$B3,$1D,$6A,$E1,$F0,$E6,$DF,$8E,$6D,$72,$9A,$AD,$38,$7B,$57,$86
	.dc.b	$75,$5E,$58,$C4,$EA,$7D,$51,$89,$D4,$60,$97,$E5,$9E,$37,$DE,$BC
	.dc.b	$0D,$8E,$46,$8B,$5C,$9F,$9D,$3D,$AB,$A4,$E4,$F7,$4F,$17,$07,$A9
	.dc.b	$E7,$1F,$89,$AF,$04,$EA,$FE,$1D,$68,$EE,$58,$FA,$73,$C2,$BC,$27
	.dc.b	$39,$D4,$F7,$3A,$4E,$E3,$9C,$7D,$97,$75,$F2,$EB,$9C,$7D,$93,$37
	.dc.b	$F1,$9A,$70,$BE,$97,$F0,$0E,$87,$AE,$78,$36,$78,$1C,$9F,$2B,$5F
	.dc.b	$F0,$85,$FD,$D6,$CF,$E9,$49,$3F,$AA,$6C,$7E,$9D,$B7,$FB,$5F,$77
	.dc.b	$F1,$84,$AF,$7B,$6B,$E8,$36,$7E,$0C,$7E,$82,$31,$3F,$22,$7A,$3C
	.dc.b	$89,$3C,$9B,$1C,$A3,$F3,$D3,$C9,$F8,$D1,$C9,$E0,$47,$85,$A7,$84
	.dc.b	$FD,$B9,$E0,$6A,$33,$3F,$66,$BF,$81,$46,$2F,$F8,$A3,$63,$F5,$2D
	.dc.b	$BF,$E0,$CD,$BE,$B3,$EF,$39,$9E,$C3,$13,$F1,$5C,$3E,$C7,$6F,$AD
	.dc.b	$DB,$F0,$9D,$BE,$B7,$71,$EB,$73,$7E,$B9,$EC,$73,$3B,$18,$9E,$0F
	.dc.b	$F1,$FF,$87,$FE,$54,$4A,$FE,$5E,$FF,$2D,$F3,$FF,$9B,$1F,$CD,$DB
	.dc.b	$1F,$C3,$FD,$CF,$F1,$F5,$7F,$95,$9A,$FF,$98,$DA,$FF,$99,$03,$87
	.dc.b	$FC,$CF,$B3,$D7,$FF,$4A,$36,$BF,$CA,$CD,$9F,$F3,$1B,$5F,$F3,$87
	.dc.b	$CE,$7F,$88,$5F,$F3,$3B,$5C,$F8,$7F,$E9,$F8,$C4,$78,$FE,$9D,$F4
	.dc.b	$5E,$18,$FF,$37,$C6,$31,$FF,$33,$2F,$9F,$78,$38,$BE,$17,$FE,$81
	.dc.b	$FB,$17,$CD,$7F,$9B,$61,$18,$7F,$CE,$0D,$7F,$CF,$AD,$7F,$D1,$5F
	.dc.b	$4E,$FA,$8E,$4D,$8F,$F3,$C3,$5F,$F4,$2B,$5F,$F4,$97,$D3,$3E,$07
	.dc.b	$C4,$D8,$FF,$40,$35,$FF,$46,$B5,$FF,$4D,$7D,$3B,$E1,$7D,$4D,$8F
	.dc.b	$F4,$43,$5F,$F4,$AB,$5F,$F5,$17,$D7,$BC,$BF,$84,$D8,$FF,$48,$35
	.dc.b	$FF,$4E,$B5,$F4,$F9,$5E,$82,$3E,$87,$C7,$E2,$48,$F1,$B5,$FF,$53
	.dc.b	$FD,$2F,$D6,$7F,$D3,$CD,$73,$C3,$F5,$63,$11,$FE,$B9,$FA,$6F,$F7
	.dc.b	$22,$69,$3A,$7E,$E7,$D8,$BD,$9A,$4A,$FF,$56,$46,$5B,$EF,$F0,$23
	.dc.b	$83,$FD,$65,$7F,$E1,$86,$F8,$BF,$89,$8C,$47,$FB,$09,$AF,$FB,$3F
	.dc.b	$ED,$FE,$43,$FA,$D6,$BF,$ED,$26,$A6,$C7,$F8,$2A,$3C,$3E,$37,$72
	.dc.b	$3E,$4F,$1B,$C4,$47,$07,$A3,$E4,$BC,$C4,$70,$7B,$3C,$3F,$51,$1C
	.dc.b	$1F,$03,$DB,$F7,$11,$C1,$F1,$3D,$FF,$84,$47,$07,$FB,$67,$BD,$EE
	.dc.b	$7E,$BD,$B1,$98,$C1,$9F,$72,$47,$16,$BF,$C2,$3E,$3B,$F7,$1E,$29
	.dc.b	$A4,$8F,$F7,$47,$D4,$33,$FD,$F3,$63,$31,$83,$F8,$17,$D4,$7F,$55
	.dc.b	$8F,$F7,$83,$5F,$F7,$EB,$5F,$E4,$C1,$C3,$FC,$A7,$EA,$18,$FF,$0D
	.dc.b	$6C,$7F,$BE,$DA,$F2,$0E,$1E,$BD,$0F,$A8,$FD,$11,$23,$F4,$EF,$5F
	.dc.b	$A8,$91,$D5,$AF,$5F,$A8,$7B,$5F,$C7,$DB,$1E,$10,$E1,$E1,$1F,$B1
	.dc.b	$F9,$39,$1F,$B6,$6B,$E9,$F9,$47,$D3,$FF,$2B,$6C,$75,$6B,$B8,$D7
	.dc.b	$24,$8D,$35,$FE,$0B,$ED,$FF,$01,$23,$FC,$19,$FE,$71,$F4,$CF,$A7
	.dc.b	$A4,$9F,$64,$65,$F7,$7D,$3B,$C6,$FE,$80,$D8,$CC,$60,$FE,$89,$F4
	.dc.b	$EF,$1E,$92,$7E,$44,$65,$F9,$5F,$4E,$ED,$FE,$98,$D8,$CC,$60,$FE
	.dc.b	$A1,$F4,$CF,$A7,$A4,$9F,$A9,$19,$7E,$B7,$99,$FC,$80,$9E,$57,$F5
	.dc.b	$97,$F9,$1F,$D3,$BE,$23,$FA,$E3,$6B,$EF,$35,$FC,$83,$5F,$CD,$AF
	.dc.b	$9D,$E2,$48,$E2,$FF,$67,$FA,$5F,$F5,$44,$7E,$25,$AF,$E6,$DA,$FE
	.dc.b	$99,$AF,$F8,$35,$7D,$37,$32,$47,$37,$B7,$8F,$FE,$13,$F8,$16,$7C
	.dc.b	$63,$0D,$AF,$F5,$65,$FE,$F5,$57,$04,$23,$FC,$3A,$F0,$F8,$0E,$6E
	.dc.b	$A3,$2F,$2A,$F4,$7C,$96,$A3,$97,$05,$A4,$D3,$11,$BF,$A5,$FE,$2D
	.dc.b	$5F,$EC,$1F,$A6,$7C,$57,$36,$C7,$FB,$16,$76,$F8,$4F,$4A,$F8,$3E
	.dc.b	$9E,$5C,$9E,$11,$E5,$F8,$EF,$F0,$46,$DF,$06,$5F,$21,$E6,$7F,$DA
	.dc.b	$4D,$7B,$09,$E3,$B7,$F9,$0D,$7D,$17,$5F,$7E,$57,$FB,$6E,$72,$FF
	.dc.b	$B7,$7E,$97,$F8,$EE,$3E,$1C,$7F,$0A,$DF,$53,$E5,$4E,$B2,$FF,$28
	.dc.b	$2F,$A6,$CF,$9F,$31,$E9,$1C,$3C,$8E,$2C,$8F,$04,$74,$DF,$EB,$66
	.dc.b	$E7,$A6,$6B,$CD,$C1,$5F,$83,$97,$D0,$D7,$E4,$3F,$3E,$3D,$9F,$2D
	.dc.b	$F8,$C7,$F9,$C4,$23,$11,$FD,$12,$31,$3F,$CB,$5B,$1C,$3E,$7D,$F4
	.dc.b	$D9,$6C,$66,$30,$7F,$95,$57,$CE,$52,$3F,$E5,$97,$FC,$B5,$F0,$3F
	.dc.b	$9F,$C3,$F0,$13,$FE,$62,$E1,$FD,$0C,$62,$39,$8F,$A6,$FC,$03,$FE
	.dc.b	$81,$AE,$55,$FD,$7B,$96,$CF,$FA,$15,$AF,$EB,$26,$BF,$E5,$B9,$EE
	.dc.b	$7F,$66,$E6,$C9,$C3,$81,$F2,$58,$9D,$D7,$FA,$47,$C2,$E6,$65,$FC
	.dc.b	$A9,$FC,$76,$36,$FE,$F3,$C9,$F0,$9C,$23,$31,$FC,$FD,$DE,$8E,$5F
	.dc.b	$4B,$63,$F4,$7F,$25,$89,$CF,$34,$9E,$31,$6F,$F4,$FF,$4D,$F3,$5B
	.dc.b	$B7,$B7,$C2,$F8,$B3,$9C,$BF,$64,$FC,$5A,$E3,$C1,$AB,$DB,$C3,$E5
	.dc.b	$E6,$5E,$EB,$FE,$F2,$D8,$EB,$E9,$C8,$C1,$B7,$C3,$F2,$F2,$39,$7C
	.dc.b	$55,$CB,$D3,$E9,$F9,$8C,$69,$E6,$7D,$3E,$4F,$AB,$E2,$2E,$1E,$37
	.dc.b	$F0,$38,$FA,$EF,$F8,$73,$F5,$70,$FE,$16,$E6,$BF,$D9,$8E,$67,$F8
	.dc.b	$A7,$C9,$7F,$42,$9F,$B6,$FE,$DD,$BE,$5F,$E3,$A3,$17,$FC,$5B,$E9
	.dc.b	$7F,$83,$57,$E0,$9F,$B5,$19,$F0,$FF,$2B,$18,$C3,$F9,$17,$D3,$3E
	.dc.b	$9B,$F3,$3F,$6D,$FD,$13,$1F,$C5,$CE,$5F,$C6,$C6,$71,$F0,$57,$4F
	.dc.b	$B2,$EA,$DA,$FC,$9F,$D4,$39,$3F,$4A,$B9,$2C,$61,$FD,$4F,$D4,$3C
	.dc.b	$5B,$F4,$58,$6E,$1F,$9E,$3F,$C9,$65,$E0,$FD,$8B,$8D,$3F,$9C,$BF
	.dc.b	$11,$BF,$51,$AF,$A2,$DB,$67,$F5,$2B,$E7,$BF,$76,$FF,$5A,$8C,$C6
	.dc.b	$3D,$5F,$DF,$FD,$47,$E4,$B0,$FE,$8E,$FB,$EB,$F6,$6E,$79,$3F,$A8
	.dc.b	$39,$9D,$FC,$A7,$F0,$8F,$2F,$FB,$DD,$F0,$8C,$3F,$C3,$3E,$A3,$FB
	.dc.b	$2B,$FB,$E5,$19,$F5,$F4,$F3,$F8,$4F,$E2,$7F,$50,$EC,$C6,$1F,$C1
	.dc.b	$A3,$0D,$E3,$DF,$D3,$70,$6C,$7F,$0C,$FA,$87,$1F,$F1,$E5,$F1,$76
	.dc.b	$30,$FF,$22,$F2,$3F,$87,$70,$7F,$6A,$7F,$B5,$5F,$F1,$67,$3C,$5F
	.dc.b	$DC,$DC,$CE,$FC,$3E,$5F,$0B,$CD,$B7,$FC,$7B,$E4,$FE,$A5,$7F,$76
	.dc.b	$7F,$C9,$D1,$9E,$6F,$E6,$71,$89,$D7,$BF,$8E,$2C,$B6,$FA,$45,$9D
	.dc.b	$C6,$0F,$E5,$BF,$51,$F8,$DD,$3F,$C1,$4D,$8F,$F1,$5C,$A3,$FD,$7B
	.dc.b	$1C,$3D,$CF,$E8,$B1,$B9,$FF,$18,$FD,$7F,$F9,$36,$B4,$7F,$3A,$71
	.dc.b	$1F,$E1,$29,$FF,$0F,$4F,$8B,$DB,$E7,$EE,$74,$64,$64,$F3,$C6,$2B
	.dc.b	$97,$2F,$88,$F1,$F2,$DA,$F6,$C7,$9B,$D8,$C5,$78,$5C,$4F,$97,$86
	.dc.b	$CF,$F8,$9B,$34,$7B,$B9,$7D,$FF,$09,$6F,$E4,$B8,$7D,$DC,$97,$0E
	.dc.b	$4F,$F6,$AF,$A9,$FE,$1D,$C3,$FE,$37,$7F,$A4,$57,$CE,$72,$DE,$6E
	.dc.b	$07,$F7,$2C,$FF,$4F,$87,$F8,$EA,$B8,$BF,$43,$B3,$B7,$B3,$DF,$D9
	.dc.b	$B8,$FF,$6A,$BF,$CA,$BD,$E3,$C9,$F0,$3B,$8D,$38,$69,$C6,$31,$87
	.dc.b	$87,$E3,$BD,$9E,$57,$B5,$B5,$DE,$2C,$EE,$30,$78,$FD,$4F,$88,$7C
	.dc.b	$5F,$4E,$C6,$4F,$F0,$0F,$D8,$FF,$0A,$AF,$81,$5F,$E1,$48,$C7,$07
	.dc.b	$CB,$DB,$E7,$1F,$3F,$86,$E1,$F0,$63,$E9,$7D,$56,$31,$F9,$4E,$27
	.dc.b	$FD,$CD,$BA,$3F,$C5,$52,$8C,$4E,$A7,$3F,$52,$FA,$9A,$49,$FF,$78
	.dc.b	$46,$5F,$F7,$97,$1F,$F0,$4C,$FF,$CF,$0D,$8F,$D7,$CE,$F1,$FC,$97
	.dc.b	$17,$F8,$CA,$3F,$C5,$73,$FE,$24,$6C,$FF,$9F,$17,$CF,$EF,$E0,$65
	.dc.b	$78,$9F,$39,$1F,$2F,$C0,$F5,$3B,$7C,$3D,$8E,$D1,$70,$ED,$37,$3F
	.dc.b	$F4,$86,$2A,$C5,$FA,$AA,$DF,$FA,$D2,$B5,$3D,$0F,$54,$EA,$53,$A7
	.dc.b	$D9,$5E,$4A,$E1,$18,$6D,$F8,$AF,$95,$F1,$8F,$2D,$F1,$BE,$04,$8C
	.dc.b	$DF,$88,$E6,$7D,$29,$E7,$7D,$1B,$1A,$AF,$64,$61,$E9,$1C,$4D,$3E
	.dc.b	$C9,$E5,$18,$B4,$62,$78,$9F,$1D,$C5,$F9,$A3,$E8,$D7,$1B,$F0,$4E
	.dc.b	$49,$1E,$19,$D9,$D0,$F9,$B3,$C2,$35,$3C,$27,$D7,$1F,$0D,$E1,$7F
	.dc.b	$2E,$36,$7D,$D3,$C4,$7A,$EF,$CE,$EA,$7D,$4E,$9E,$71,$A7,$B9,$E6
	.dc.b	$AF,$13,$E6,$9F,$CA,$91,$FE,$A6,$70,$2D,$7E,$1D,$B7,$FB,$42,$4F
	.dc.b	$1A,$FF,$73,$74,$FE,$9A,$3E,$AB,$6B,$E3,$36,$7C,$6D,$9F,$64,$A3
	.dc.b	$0F,$AA,$7A,$E5,$F9,$B9,$CC,$FA,$CF,$F7,$8B,$8C,$3E,$64,$0E,$93
	.dc.b	$C4,$97,$E6,$6C,$79,$9B,$7E,$66,$CF,$9A,$FB,$61,$FE,$BA,$6C,$7F
	.dc.b	$AB,$1B,$7E,$4A,$E2,$FA,$E7,$FC,$03,$3A,$BF,$6C,$78,$C9,$3E,$36
	.dc.b	$C7,$8D,$A7,$8E,$BA,$CF,$C3,$9E,$6F,$12,$3C,$5B,$3C,$4E,$6F,$94
	.dc.b	$F9,$15,$E5,$8F,$3D,$76,$C3,$E6,$D7,$CC,$9E,$2F,$85,$D9,$88,$DD
	.dc.b	$73,$9E,$B5,$FC,$C7,$CF,$FE,$7E,$4B,$FE,$7A,$D9,$60,$9F,$F1,$4F
	.dc.b	$CD,$FF,$8A,$27,$F9,$B5,$7F,$BB,$23,$15,$B9,$FC,$16,$FF,$E5,$A7
	.dc.b	$F3,$E7,$F4,$DF,$33,$FA,$9C,$66,$3F,$46,$D7,$FC,$6F,$C0,$FA,$CF
	.dc.b	$EB,$2D,$CB,$FA,$8C,$60,$FE,$BB,$F5,$4F,$AA,$FF,$36,$5F,$82,$31
	.dc.b	$8F,$8A,$10,$F9,$F7,$81,$98,$DC,$FA,$35,$78,$FC,$61,$18,$7A,$35
	.dc.b	$E1,$1B,$8F,$F5,$44,$6B,$D7,$C1,$1F,$27,$D2,$BE,$AB,$B3,$63,$31
	.dc.b	$83,$A7,$CA,$48,$F0,$B5,$F9,$F2,$F0,$79,$92,$3B,$BE,$5F,$2A,$47
	.dc.b	$A9,$AF,$2F,$23,$C6,$47,$CE,$F9,$BC,$1F,$F3,$6F,$37,$E0,$65,$82
	.dc.b	$73,$C9,$FE,$72,$8C,$4F,$F0,$7A,$E1,$0F,$07,$73,$FC,$2F,$E7,$7E
	.dc.b	$AB,$BB,$FF,$3D,$C6,$23,$F2,$F1,$8E,$2E,$30,$FA,$77,$83,$8F,$A4
	.dc.b	$F3,$67,$E7,$6E,$1F,$E8,$66,$DF,$F1,$B5,$F5,$7D,$3E,$9F,$FC,$D7
	.dc.b	$3E,$33,$F9,$03,$87,$C6,$63,$7F,$1E,$4F,$A1,$E5,$ED,$CA,$E4,$2F
	.dc.b	$FD,$92,$E3,$17,$CD,$7A,$7C,$29,$39,$7A,$F3,$FC,$0F,$A7,$FE,$79
	.dc.b	$3D,$1A,$F9,$03,$8D,$FE,$17,$D7,$7C,$5F,$A8,$95,$E1,$7E,$47,$93
	.dc.b	$23,$C8,$D7,$A7,$C9,$3E,$2B,$E7,$7C,$CC,$9F,$53,$F4,$7E,$9E,$47
	.dc.b	$B5,$AF,$97,$8D,$F2,$5F,$5F,$DF,$93,$F0,$DF,$B5,$EF,$C8,$F8,$CD
	.dc.b	$7D,$DF,$5C,$DB,$F0,$7E,$43,$E0,$69,$2B,$E8,$C3,$E0,$76,$48,$CB
	.dc.b	$CB,$E2,$BF,$1B,$E4,$BE,$5E,$92,$BE,$DC,$3E,$07,$F2,$7F,$52,$E1
	.dc.b	$A4,$AF,$EB,$5A,$7E,$63,$EA,$7F,$CA,$51,$F8,$46,$BF,$65,$BD,$1F
	.dc.b	$AF,$5F,$2B,$FA,$0F,$A9,$F5,$57,$E3,$1A,$FE,$05,$B5,$FB,$75,$AF
	.dc.b	$EA,$3C,$8F,$CB,$57,$E5,$1A,$FE,$25,$B5,$FE,$12,$EA,$F0,$7C,$4C
	.dc.b	$AF,$03,$E3,$F9,$3F,$D9,$FD,$4B,$E1,$29,$2B,$F4,$30,$F8,$4F,$17
	.dc.b	$C9,$C9,$F1,$3F,$BC,$F7,$BD,$F1,$FA,$8F,$9B,$7D,$4F,$F0,$06,$C6
	.dc.b	$63,$1C,$79,$ED,$93,$C5,$FE,$0F,$F5,$1F,$D2,$23,$DD,$F1,$1F,$0C
	.dc.b	$8F,$F8,$3F,$9F,$7F,$53,$23,$F7,$34,$8E,$7F,$2D,$FC,$4B,$EA,$5F
	.dc.b	$19,$49,$7F,$C0,$23,$F8,$AB,$FC,$0B,$EA,$7F,$92,$BF,$B7,$6F,$3F
	.dc.b	$F2,$D6,$FA,$DF,$E1,$CF,$2F,$F8,$FF,$D4,$BC,$8A,$4C,$3F,$86,$C3
	.dc.b	$83,$F9,$2F,$A2,$F7,$E9,$2B,$F8,$9C,$3D,$EF,$A1,$94,$9F,$F1,$83
	.dc.b	$F4,$F1,$FE,$5F,$F5,$2F,$90,$A4,$AF,$E3,$B1,$F0,$9B,$F4,$1C,$08
	.dc.b	$EB,$DB,$FE,$73,$F5,$3F,$E8,$C8,$FE,$4A,$D7,$F8,$D3,$78,$7F,$A8
	.dc.b	$3F,$A4,$CB,$FC,$A5,$28,$FC,$3B,$C3,$C7,$EA,$76,$F4,$FB,$F4,$D9
	.dc.b	$EF,$18,$7C,$3F,$2F,$E4,$E4,$91,$83,$CB,$BF,$A2,$64,$60,$FE,$7B
	.dc.b	$F5,$1F,$C1,$DF,$E7,$FF,$53,$FD,$7D,$BF,$07,$FD,$62,$75,$1F,$CD
	.dc.b	$D7,$CC,$67,$D2,$93,$D2,$3F,$AF,$FC,$1B,$E9,$7F,$B0,$FC,$CB,$3A
	.dc.b	$6A,$37,$F4,$D7,$0F,$EA,$37,$C5,$B1,$FD,$AB,$EA,$7F,$83,$F2,$7C
	.dc.b	$CC,$7E,$83,$7D,$0D,$7B,$1B,$F4,$EE,$3F,$D7,$30,$E2,$FF,$A1,$5F
	.dc.b	$E1,$EE,$7C,$BE,$0E,$E7,$C1,$1E,$1D,$FD,$7C,$B2,$BA,$47,$97,$83
	.dc.b	$51,$83,$C0,$FA,$7E,$37,$FC,$B7,$1F,$E5,$2C,$7F,$C7,$7D,$5F,$E4
	.dc.b	$36,$C6,$67,$6F,$FA,$DA,$1F,$1A,$FA,$9E,$0D,$9E,$8D,$78,$35,$FF
	.dc.b	$30,$FD,$0F,$F9,$25,$34,$8F,$F9,$57,$E3,$3F,$CD,$58,$FF,$8A,$E3
	.dc.b	$FC,$AB,$C3,$FE,$06,$7F,$D5,$D5,$FE,$AF,$FA,$9F,$E1,$31,$FE,$B1
	.dc.b	$BE,$35,$FE,$CE,$E6,$B3,$FA,$46,$BB,$8E,$91,$88,$72,$D3,$7C,$AF
	.dc.b	$9A,$C5,$F8,$FC,$4E,$06,$2F,$FC,$1C,$7F,$03,$8F,$47,$99,$D4,$DB
	.dc.b	$83,$FC,$43,$AB,$97,$CA,$D8,$FF,$12,$E3,$FE,$A2,$8C,$5F,$F8,$6D
	.dc.b	$FE,$E5,$3C,$B9,$38,$DF,$27,$DF,$F5,$3F,$2A,$FF,$C4,$4D,$BF,$F2
	.dc.b	$C6,$0F,$03,$7B,$F0,$DE,$67,$FE,$36,$C3,$83,$FC,$39,$FD,$83,$9F
	.dc.b	$5B,$B9,$E0,$F6,$2C,$71,$F5,$FC,$66,$34,$F1,$BE,$6F,$92,$93,$DF
	.dc.b	$E1,$6D,$7A,$3D,$6F,$59,$8D,$FF,$CE,$19,$A3,$DF,$E5,$FC,$0F,$2E
	.dc.b	$E3,$E9,$7F,$03,$97,$F5,$48,$FF,$28,$B7,$1F,$F2,$33,$FE,$68,$8C
	.dc.b	$F2,$70,$CE,$47,$09,$B5,$C3,$E3,$BF,$1B,$F8,$07,$FB,$26,$BA,$C6
	.dc.b	$1F,$C8,$FD,$47,$CA,$A0,$7E,$5B,$FE,$57,$74,$E7,$8B,$8F,$0F,$2E
	.dc.b	$2B,$7F,$E6,$AB,$FF,$05,$37,$5E,$E7,$83,$E5,$2C,$7F,$9C,$3E,$9F
	.dc.b	$FC,$6C,$DE,$7F,$D2,$C6,$33,$FF,$34,$28,$CF,$C6,$7E,$AB,$8F,$D3
	.dc.b	$7C,$E3,$17,$9F,$93,$93,$FE,$7E,$AF,$F5,$F3,$63,$F6,$7E,$DF,$F7
	.dc.b	$98,$FF,$41,$B6,$BF,$CF,$0F,$F7,$98,$CF,$A1,$FB,$AF,$B7,$FD,$0D
	.dc.b	$7F,$9F,$5B,$D3,$FE,$B3,$7F,$A4,$D7,$FA,$39,$CB,$FD,$99,$FF,$0C
	.dc.b	$39,$BE,$9E,$EF,$57,$F8,$3F,$DB,$FD,$BC,$3F,$D1,$4F,$FA,$92,$BF
	.dc.b	$D3,$2E,$70,$FF,$61,$39,$9D,$FC,$0F,$2F,$81,$E6,$DB,$E3,$F3,$DD
	.dc.b	$8E,$FE,$0D,$8F,$F5,$2F,$D3,$3E,$1B,$F8,$CB,$63,$3C,$58,$9F,$E3
	.dc.b	$BF,$6F,$F9,$9C,$FF,$D4,$0D,$D7,$FD,$AC,$FF,$AC,$EF,$FD,$62,$E5
	.dc.b	$FF,$04,$BF,$EE,$97,$3C,$7D,$3B,$BA,$BF,$CA,$BE,$9F,$FD,$19,$3F
	.dc.b	$EB,$A6,$E1,$FE,$AF,$FB,$67,$D2,$EF,$E4,$F8,$7A,$F3,$0E,$11,$6F
	.dc.b	$FD,$6D,$87,$FA,$4E,$7F,$D8,$BF,$44,$F1,$F5,$B7,$FB,$BE,$31,$3E
	.dc.b	$1F,$5B,$C9,$EB,$79,$5B,$1E,$58,$B8,$F6,$9B,$7F,$EF,$9C,$7F,$40
	.dc.b	$DF,$92,$F7,$7C,$27,$BD,$98,$FB,$FC,$0B,$8F,$A1,$FE,$9B,$F6,$CF
	.dc.b	$A4,$F8,$DF,$53,$86,$DF,$C7,$8F,$95,$F1,$F8,$C3,$E0,$B8,$9F,$83
	.dc.b	$E3,$59,$FF,$6D,$70,$23,$FA,$DF,$D4,$7E,$CA,$7F,$DB,$CF,$F0,$83
	.dc.b	$1B,$2F,$FC,$8B,$19,$F4,$FF,$B1,$E0,$F8,$4E,$71,$87,$AF,$AF,$DB
	.dc.b	$D3,$B8,$7F,$BE,$30,$FE,$51,$8F,$5C,$BF,$8D,$C6,$27,$BF,$C1,$71
	.dc.b	$78,$3F,$1B,$C4,$F4,$FA,$16,$7A,$38,$7B,$7A,$17,$0F,$6B,$F0,$09
	.dc.b	$1C,$0D,$9E,$5F,$03,$E2,$F8,$1F,$19,$B1,$E9,$8B,$5F,$22,$6E,$7E
	.dc.b	$B8,$67,$F1,$6F,$E0,$6D,$F4,$38,$7E,$8B,$23,$E2,$77,$DB,$FA,$6F
	.dc.b	$F8,$43,$E9,$FF,$82,$63,$D5,$B5,$FE,$78,$FA,$9F,$E3,$D6,$AD,$7D
	.dc.b	$1F,$13,$97,$45,$BF,$E9,$33,$CE,$33,$1F,$E8,$2A,$FF,$41,$E1,$F9
	.dc.b	$38,$C4,$7F,$50,$8C,$38,$6B,$C5,$7D,$0F,$F8,$CE,$38,$35,$46,$A7
	.dc.b	$AC,$62,$75,$3F,$E2,$99,$F3,$9E,$4F,$77,$E2,$4F,$C5,$7E,$37,$C3
	.dc.b	$49,$DD,$F9,$FE,$7F,$0F,$C5,$8F,$6C,$F6,$23,$E2,$C1,$1E,$2F,$8B
	.dc.b	$FA,$47,$E8,$39,$6D,$76,$BF,$5C,$E1,$FD,$7C,$DD,$3E,$C4,$DA,$FF
	.dc.b	$57,$F5,$FF,$BF,$5E,$58,$A7,$58,$7C,$F7,$C1,$3A,$73,$1E,$08,$FE
	.dc.b	$3F,$86,$2B,$94,$6C,$C9,$67,$E1,$1D,$5C,$61,$FC,$96,$B9,$5F,$6C
	.dc.b	$3C,$D7,$C0,$F0,$17,$0E,$ED,$9F,$3B,$70,$EA,$7E,$32,$33,$87,$C4
	.dc.b	$8C,$CF,$8C,$F9,$2E,$1E,$F1,$B7,$27,$52,$39,$3D,$51,$C6,$BC,$B1
	.dc.b	$98,$FA,$38,$7B,$9E,$D8,$7B,$A7,$57,$E3,$6A,$8D,$4F,$AC,$ED,$87
	.dc.b	$B1,$C4,$F9,$1B,$5E,$48,$C1,$D6,$32,$79,$0E,$67,$92,$FC,$D1,$C0
	.dc.b	$F1,$38,$9F,$2D,$F9,$27,$A1,$0F,$93,$5D,$5E,$58,$7C,$48,$F0,$C7
	.dc.b	$F9,$32,$79,$3C,$0F,$A8,$7E,$F7,$0F,$D2,$4F,$E9,$DB,$1F,$C0,$DA
	.dc.b	$EE,$34,$7E,$7B,$CE,$FC,$B1,$1F,$AC,$D8,$F2,$B7,$0F,$A3,$82,$31
	.dc.b	$5E,$8A,$FE,$17,$18,$7E,$04,$EE,$7A,$92,$3A,$B6,$BA,$D7,$19,$F8
	.dc.b	$07,$49,$D1,$DE,$BA,$47,$22,$57,$29,$DC,$78,$AB,$A5,$7C,$F8,$FE
	.dc.b	$55,$18,$BF,$E7,$CD,$AF,$8C,$72,$9F,$1D,$78,$2F,$B4,$78,$27,$C9
	.dc.b	$5C,$63,$F6,$8D,$9F,$E9,$0D,$9F,$E7,$AD,$9F,$1C,$77,$AF,$34,$7C
	.dc.b	$13,$B1,$FE,$90,$F3,$9D,$FF,$CE,$DB,$5F,$CE,$9B,$3F,$D2,$1A,$FF
	.dc.b	$45,$F9,$1F,$E7,$A4,$9F,$E6,$0F,$F4,$5F,$9C,$FF,$38,$CF,$F5,$06
	.dc.b	$BF,$D5,$5A,$FF,$A8,$83,$87,$FD,$4B,$F4,$AF,$A7,$7D,$3F,$F4,$E8
	.dc.b	$C3,$7E,$81,$E2,$66,$B7,$7F,$D8,$3E,$C5,$F6,$5F,$D5,$A1,$18,$7F
	.dc.b	$AC,$B5,$E1,$1B,$FA,$97,$D3,$D2,$4F,$F6,$08,$CB,$FD,$8B,$47,$CD
	.dc.b	$77,$6C,$66,$30,$67,$CC,$91,$C5,$AF,$F7,$8A,$78,$9E,$32,$47,$85
	.dc.b	$F3,$7C,$14,$8F,$23,$5E,$5E,$27,$CF,$E8,$FF,$0F,$5F,$F9,$2E,$7F
	.dc.b	$B1,$35,$FE,$8C,$FF,$91,$5F,$EC,$DF,$51,$FE,$25,$9F,$F1,$37,$D3
	.dc.b	$7D,$59,$FE,$D4,$D7,$FA,$74,$EF,$9B,$EF,$36,$3F,$DC,$2D,$4D,$1E
	.dc.b	$1F,$9E,$7F,$0D,$F5,$2C,$F1,$7F,$EE,$A8,$C3,$FD,$ED,$AF,$F5,$FF
	.dc.b	$A8,$7C,$2F,$08,$4B,$E2,$70,$D8,$E5,$1B,$C9,$F3,$D7,$C7,$95,$F1
	.dc.b	$1E,$98,$FE,$0A,$3F,$B7,$7D,$4B,$D4,$F0,$36,$3F,$C2,$0D,$7E,$3F
	.dc.b	$C1,$F7,$F2,$64,$7D,$07,$C5,$E4,$F6,$FA,$A9,$3F,$E1,$9C,$51,$E7
	.dc.b	$F9,$99,$3F,$61,$AF,$D4,$F2,$7D,$84,$7E,$EB,$EA,$F8,$FF,$81,$F1
	.dc.b	$72,$7F,$02,$FC,$3F,$1E,$47,$E1,$1A,$FF,$83,$7A,$1A,$FC,$FF,$15
	.dc.b	$F0,$34,$95,$FE,$31,$87,$C1,$6C,$91,$97,$97,$CD,$7D,$9F,$0D,$F1
	.dc.b	$74,$95,$CF,$E0,$BC,$04,$70,$77,$F8,$3E,$5F,$17,$23,$FC,$85,$68
	.dc.b	$ED,$E8,$7E,$1F,$EE,$59,$52,$5F,$F9,$26,$1F,$2B,$BF,$95,$93,$97
	.dc.b	$96,$7E,$9E,$CF,$57,$D3,$CA,$F0,$C7,$7F,$33,$C1,$F0,$F2,$7B,$47
	.dc.b	$8F,$E1,$FC,$9E,$74,$9F,$14,$7C,$5F,$8C,$91,$EC,$7E,$47,$C6,$FA
	.dc.b	$F8,$24,$FA,$5F,$76,$5E,$FC,$E4,$FB,$5F,$85,$7F,$A9,$F3,$FF,$D1
	.dc.b	$DE,$77,$FA,$9D,$FE,$63,$5F,$E6,$B6,$BF,$E4,$F7,$7E,$6F,$EE,$BD
	.dc.b	$5F,$D7,$FC,$9C,$AF,$DC,$47,$EC,$BE,$5F,$F6,$7F,$15,$27,$F7,$91
	.dc.b	$E1,$A9,$1D,$DF,$24,$7E,$EB,$D0,$FD,$DF,$AB,$2B,$F8,$24,$7E,$FB
	.dc.b	$DF,$FD,$FF,$C3,$C9,$FE,$11,$1E,$DC,$64,$7A,$DF,$81,$F1,$5F,$C1
	.dc.b	$BC,$89,$3F,$C3,$DF,$E1,$3E,$C7,$F0,$AF,$88,$93,$FC,$4D,$FE,$1B
	.dc.b	$F1,$BF,$27,$E2,$FE,$56,$92,$BE,$44,$7F,$15,$FA,$1F,$F4,$CC,$7F
	.dc.b	$A5,$5A,$FF,$9E,$5B,$C9,$FE,$D7,$5F,$59,$C7,$E8,$A5,$71,$7A,$7A
	.dc.b	$3D,$7D,$1B,$3D,$63,$0F,$7F,$A0,$F1,$7C,$2C,$8C,$1E,$3F,$4F,$CD
	.dc.b	$32,$30,$7F,$AA,$FE,$85,$E3,$FF,$2F,$FA,$2F,$E9,$53,$FE,$91,$6E
	.dc.b	$DF,$EE,$FA,$DC,$F1,$F2,$3F,$9B,$46,$23,$FD,$30,$D9,$E1,$3F,$CE
	.dc.b	$F8,$1D,$9F,$CF,$7C,$66,$74,$D4,$6F,$B6,$E5,$FE,$C1,$BE,$2D,$8F
	.dc.b	$E8,$DF,$45,$FC,$52,$3F,$D8,$D8,$7D,$0F,$03,$FC,$30,$D8,$09,$E0
	.dc.b	$FF,$AB,$DF,$E1,$2F,$FB,$3D,$CE,$6A,$FF,$11,$46,$1B,$E6,$78,$BC
	.dc.b	$BF,$1B,$2F,$94,$79,$B8,$3E,$07,$1C,$9E,$D1,$EA,$DF,$51,$83,$C4
	.dc.b	$FB,$BC,$F6,$7F,$EF,$AC,$D1,$8A,$FF,$1B,$1E,$E8,$CC,$EE,$75,$3A
	.dc.b	$97,$D0,$BE,$8B,$A3,$67,$99,$F1,$DC,$3C,$5A,$EB,$E8,$BF,$C7,$09
	.dc.b	$A4,$79,$9F,$E4,$2D,$BF,$DF,$8D,$3F,$DC,$07,$D6,$8F,$AF,$F4,$4E
	.dc.b	$2F,$6F,$8F,$F4,$EB,$7C,$97,$5C,$47,$DE,$8C,$63,$C0,$EE,$E2,$3F
	.dc.b	$CB,$9C,$95,$FC,$2B,$79,$1F,$21,$99,$4D,$F1,$66,$1E,$A8,$B3,$EA
	.dc.b	$F1,$78,$18,$CB,$CB,$08,$F0,$47,$F5,$4A,$FC,$AB,$FE,$39,$E6,$F1
	.dc.b	$77,$E9,$DF,$34,$AD,$9F,$E4,$78,$C6,$1F,$86,$8F,$F3,$C7,$D0,$FC
	.dc.b	$DC,$3E,$2B,$63,$F4,$BD,$DF,$06,$31,$5F,$A6,$7F,$6F,$3D,$A3,$1B
	.dc.b	$EF,$83,$83,$C7,$C8,$95,$BA,$5F,$FB,$78,$4F,$7F,$A2,$95,$B9,$CB
	.dc.b	$63,$C9,$E3,$C8,$D7,$91,$70,$F7,$1E,$7F,$23,$D4,$67,$5F,$7B,$ED
	.dc.b	$F2,$3D,$DF,$11,$70,$F0,$3F,$81,$C7,$EA,$47,$A5,$B8,$76,$3F,$CF
	.dc.b	$BD,$1B,$E4,$E1,$9C,$BE,$17,$6F,$83,$FE,$59,$8F,$89,$F4,$5F,$C2
	.dc.b	$DE,$DF,$11,$F9,$18,$C5,$77,$FA,$2F,$A6,$9F,$A6,$E7,$8F,$F3,$11
	.dc.b	$8D,$FF,$42,$BC,$DC,$F1,$E9,$B9,$7D,$59,$FF,$3B,$37,$7E,$7E,$7F
	.dc.b	$47,$F4,$9F,$44,$F5,$AF,$7E,$1B,$7F,$7E,$5E,$B7,$EA,$F9,$BF,$97
	.dc.b	$CE,$63,$1B,$E7,$D1,$93,$F8,$7A,$FF,$5C,$B6,$3F,$69,$ED,$3C,$FA
	.dc.b	$63,$E0,$AD,$7E,$3F,$E8,$BF,$AD,$B7,$C1,$FD,$E4,$7F,$B7,$B9,$3F
	.dc.b	$82,$BA,$CF,$EE,$3F,$E5,$6A,$E5,$8F,$0F,$7B,$8B,$FC,$17,$E8,$BE
	.dc.b	$AE,$1F,$C3,$9F,$F5,$0D,$7E,$7D,$CB,$FE,$E7,$73,$87,$E2,$57,$CE
	.dc.b	$F3,$F8,$4E,$8D,$AE,$5F,$2F,$DC,$F0,$78,$B6,$3F,$59,$F4,$4F,$73
	.dc.b	$F8,$C3,$63,$3C,$98,$9F,$E3,$9F,$45,$FC,$A6,$7F,$68,$DD,$3F,$38
	.dc.b	$FF,$AC,$23,$3E,$27,$0F,$0B,$8F,$3D,$9E,$2F,$F2,$7F,$A1,$FF,$43
	.dc.b	$C7,$EF,$DB,$87,$E9,$FE,$89,$F3,$FB,$F8,$AE,$1F,$13,$30,$E1,$16
	.dc.b	$FF,$5D,$87,$FA,$3A,$7F,$85,$7C,$EB,$CB,$D7,$3F,$F3,$E8,$C4,$F8
	.dc.b	$7D,$AF,$27,$B5,$E5,$6C,$79,$62,$E3,$DA,$6D,$FF,$48,$C7,$F3,$D7
	.dc.b	$EB,$F0,$BD,$DF,$11,$EF,$64,$7B,$FC,$4B,$8F,$A1,$FE,$95,$F4,$4F
	.dc.b	$9F,$F8,$DF,$13,$86,$DF,$C7,$F8,$9F,$94,$63,$0F,$82,$E2,$3E,$0F
	.dc.b	$95,$67,$F8,$97,$02,$3F,$AC,$7D,$17,$EB,$A7,$F8,$B3,$FC,$20,$C6
	.dc.b	$FF,$D9,$1F,$E1,$74,$FA,$2F,$EC,$0D,$C3,$F9,$C4,$EA,$7A,$FA,$92
	.dc.b	$73,$ED,$DC,$7B,$CF,$8B,$B3,$19,$77,$7E,$1F,$87,$E6,$F4,$2E,$3D
	.dc.b	$9F,$4F,$A3,$23,$07,$89,$C6,$3F,$E1,$2D,$3F,$A2,$B6,$3F,$B5,$BF
	.dc.b	$E3,$C8,$CB,$FE,$22,$5D,$FF,$1B,$BF,$E3,$B6,$FD,$F1,$CA,$94,$8C
	.dc.b	$36,$7F,$C3,$2D,$9F,$26,$DA,$24,$E9,$E3,$DD,$23,$8B,$67,$FC,$5B
	.dc.b	$F4,$3F,$C1,$9F,$F1,$CB,$5F,$F2,$B8,$70,$FF,$96,$9A,$E7,$93,$D5
	.dc.b	$1F,$91,$AF,$F1,$7B,$67,$BC,$FB,$F8,$BF,$81,$70,$7E,$AF,$83,$FC
	.dc.b	$45,$7B,$8C,$61,$FD,$EE,$3F,$C4,$6D,$F7,$77,$F2,$FC,$19,$1E,$07
	.dc.b	$C6,$47,$C7,$F2,$BD,$BD,$EE,$ED,$8E,$D1,$EA,$23,$EA,$7D,$BE,$57
	.dc.b	$98,$F3,$FC,$55,$9F,$34,$5D,$BF,$BC,$64,$AC,$67,$FE,$46,$85,$62
	.dc.b	$FE,$7C,$6D,$E6,$4B,$E7,$3E,$08,$C5,$73,$8E,$53,$F5,$6E,$BF,$66
	.dc.b	$2D,$F5,$8E,$77,$C8,$F0,$CF,$58,$C9,$2F,$B4,$FD,$3B,$E7,$5D,$1B
	.dc.b	$1C,$EB,$C4,$D7,$C4,$DB,$F1,$5F,$49,$ED,$3E,$23,$B1,$BB,$EC,$D3
	.dc.b	$95,$F8,$27,$89,$E0,$29,$E0,$2D,$F8,$27,$B4,$62,$BE,$05,$F8,$C9
	.dc.b	$3A,$7E,$C5,$F7,$73,$5D,$DA,$F7,$6D,$F7,$8F,$D4,$5F,$1B,$FB,$F7
	.dc.b	$67,$31,$F0,$1F,$23,$EE,$3E,$03,$EA,$8F,$80,$FE,$E2,$7F,$2B,$3F
	.dc.b	$B0,$6C,$7E,$F9,$AE,$E3,$47,$F9,$9F,$E4,$7F,$B4,$E1,$EA,$6C,$7C
	.dc.b	$13,$F8,$1B,$89,$F3,$9F,$E5,$57,$0F,$C1,$8D,$F8,$BF,$92,$9C,$E3
	.dc.b	$FA,$09,$F0,$4E,$48,$FE,$1E,$77,$87,$FA,$95,$B1,$C0,$D4,$F0,$BF
	.dc.b	$BF,$3A,$7C,$91,$9C,$3D,$8D,$AF,$64,$FF,$BD,$2B,$85,$7F,$AE,$9B
	.dc.b	$3E,$A3,$B9,$27,$BD,$73,$7E,$1D,$71,$8C,$47,$C3,$3B,$38,$C3,$E1
	.dc.b	$47,$9A,$BA,$57,$99,$A7,$9A,$17,$E4,$79,$C6,$9F,$0B,$E7,$7C,$AE
	.dc.b	$5F,$11,$CA,$FC,$58,$7C,$D6,$CF,$CD,$9E,$24,$BF,$23,$FE,$93,$E0
	.dc.b	$7D,$07,$F5,$96,$D7,$FB,$B9,$B3,$FE,$FA,$6B,$FD,$30,$FF,$05,$7C
	.dc.b	$7F,$F6,$1E,$DF,$F3,$8C,$7F,$C0,$06,$BB,$6B,$FD,$53,$E8,$5F,$43
	.dc.b	$E2,$8E,$82,$33,$8F,$FC,$00,$5F,$2A,$F7,$B1,$7C,$6F,$C9,$BB,$CA
	.dc.b	$E7,$1F,$85,$6B,$F4,$9B,$1B,$8D,$FC,$FB,$4D,$B6,$33,$18,$39,$FC
	.dc.b	$CB,$E1,$A9,$27,$C3,$19,$7C,$7F,$01,$FE,$31,$6E,$FE,$48,$C4,$7C
	.dc.b	$26,$BF,$3F,$07,$C2,$79,$09,$1E,$87,$DB,$F6,$6F,$26,$92,$7E,$1C
	.dc.b	$65,$F7,$E1,$EE,$23,$83,$E1,$F8,$3F,$08,$8E,$0F,$9B,$F3,$DF,$E2
	.dc.b	$14,$D2,$73,$7D,$05,$F4,$14,$93,$C9,$F9,$7F,$07,$23,$E4,$B5,$FC
	.dc.b	$2C,$BE,$7F,$A1,$23,$C6,$FD,$4E,$19,$1F,$49,$AF,$2F,$77,$C8,$47
	.dc.b	$D6,$F9,$FE,$7B,$FC,$53,$D8,$CF,$FC,$57,$8A,$73,$18,$6D,$8F,$87
	.dc.b	$E1,$61,$72,$3E,$7B,$0D,$8E,$18,$F7,$79,$46,$54,$65,$F9,$FF,$29
	.dc.b	$D7,$2F,$F0,$B4,$72,$CB,$CF,$39,$35,$E1,$C9,$D1,$E7,$AE,$2F,$6A
	.dc.b	$F7,$CF,$9B,$97,$B7,$C3,$CA,$E2,$7E,$01,$F1,$7C,$3F,$BF,$E7,$9C
	.dc.b	$18,$AD,$E4,$AD,$4F,$C5,$F0,$1E,$ED,$F1,$66,$1D,$A7,$2D,$9E,$04
	.dc.b	$75,$F2,$DF,$3F,$C0,$7C,$3D,$24,$F3,$F7,$BC,$1E,$1F,$85,$91,$DE
	.dc.b	$39,$7B,$3C,$FE,$1E,$4F,$F6,$F8,$FB,$9F,$3C,$CA,$92,$7E,$2C,$3E
	.dc.b	$47,$7F,$17,$27,$2F,$2C,$BD,$1E,$2F,$A7,$E7,$65,$7F,$7D,$8E,$FF
	.dc.b	$3D,$E0,$F8,$39,$3D,$A3,$C7,$F0,$7E,$4F,$83,$93,$E2,$8F,$89,$F1
	.dc.b	$12,$3F,$C1,$0F,$C7,$F8,$9F,$5F,$04,$9F,$4B,$EE,$CB,$DF,$9C,$9F
	.dc.b	$6B,$F0,$AF,$F3,$DE,$5F,$F4,$5F,$0B,$F2,$CF,$F3,$7A,$FC,$3B,$5F
	.dc.b	$C7,$1F,$A7,$78,$78,$3F,$E9,$8F,$A0,$D5,$7E,$7D,$FE,$55,$5C,$03
	.dc.b	$88,$D7,$A5,$FA,$F8,$7C,$AD,$F8,$69,$7F,$98,$9C,$BE,$4F,$4F,$D2
	.dc.b	$47,$CF,$5C,$FC,$EE,$9E,$6C,$9E,$51,$DF,$E1,$3C,$1F,$09,$27,$B4
	.dc.b	$7C,$4C,$A4,$7C,$27,$E3,$FC,$37,$9F,$CA,$93,$E5,$7D,$5F,$0B,$EB
	.dc.b	$F8,$59,$3E,$97,$DD,$F1,$3F,$43,$E0,$FE,$8E,$92,$BE,$7C,$7F,$11
	.dc.b	$F9,$F7,$A9,$49,$3F,$BA,$87,$C1,$6F,$E7,$E4,$E5,$E5,$EA,$73,$F5
	.dc.b	$2C,$F1,$8C,$3C,$FE,$77,$C1,$F2,$12,$30,$77,$F5,$BC,$93,$23,$07
	.dc.b	$F0,$CF,$9E,$79,$3F,$CA,$7E,$7D,$EF,$53,$F9,$64,$EA,$B8,$79,$78
	.dc.b	$B9,$18,$3F,$8B,$F0,$FE,$1F,$C1,$FF,$72,$F9,$2C,$FF,$9B,$36,$B7
	.dc.b	$DF,$FC,$E6,$FF,$0F,$1F,$91,$79,$3F,$CF,$DA,$9D,$F8,$5F,$CD,$FE
	.dc.b	$7F,$F8,$CC,$FE,$66,$FE,$9B,$FC,$A9,$AF,$E7,$63,$73,$C7,$CF,$FE
	.dc.b	$95,$18,$CB,$FD,$FA,$DA,$FE,$A0,$FF,$13,$7F,$9A,$BF,$C4,$5B,$3C
	.dc.b	$8E,$7E,$77,$4D,$3E,$0F,$1C,$AE,$31,$E0,$DB,$51,$83,$AB,$E4,$F4
	.dc.b	$7F,$AA,$C7,$F8,$EB,$1F,$F1,$96,$7F,$E3,$26,$C6,$67,$6F,$F9,$2E
	.dc.b	$1F,$3C,$F9,$FE,$0D,$9E,$8D,$78,$35,$FF,$24,$FC,$FF,$F8,$F9,$34
	.dc.b	$8F,$F9,$13,$C9,$FE,$DD,$8F,$F6,$B8,$FF,$26,$6F,$FD,$71,$FE,$F5
	.dc.b	$5F,$DE,$FE,$7F,$EE,$47,$F7,$D3,$67,$F1,$33,$FC,$85,$3C,$78,$FA
	.dc.b	$BF,$E6,$7C,$2C,$F1,$9A,$FF,$81,$FC,$9A,$FF,$18,$AF,$C2,$7C,$6E
	.dc.b	$27,$3E,$3C,$9F,$34,$5A,$F1,$78,$FC,$8C,$69,$ED,$7B,$BF,$D4,$EB
	.dc.b	$FB,$CB,$67,$F2,$9E,$06,$B9,$77,$C9,$2F,$71,$6B,$F9,$97,$CF,$FC
	.dc.b	$2C,$3F,$C2,$8D,$AF,$D6,$F7,$FB,$A3,$15,$FE,$25,$7E,$3B,$CB,$6D
	.dc.b	$6F,$B7,$87,$95,$30,$FF,$07,$DB,$CA,$C6,$1E,$2F,$9F,$EC,$63,$3F
	.dc.b	$1B,$E1,$F2,$A6,$1C,$BC,$BB,$7E,$AF,$2F,$CC,$63,$0F,$5B,$E9,$F8
	.dc.b	$99,$7E,$2F,$32,$DF,$F5,$3C,$5E,$67,$DC,$7F,$D4,$78,$FF,$8D,$DC
	.dc.b	$DF,$F8,$A1,$CB,$F8,$B5,$8F,$1C,$39,$36,$7F,$C5,$FF,$3D,$FE,$6A
	.dc.b	$6F,$C4,$7E,$2A,$1F,$1D,$63,$F1,$BF,$3D,$F8,$BD,$3F,$B4,$37,$3F
	.dc.b	$F2,$52,$EB,$DF,$5F,$0E,$19,$9F,$F8,$FA,$BF,$CB,$0D,$8E,$BE,$57
	.dc.b	$6F,$94,$B3,$B7,$F3,$BF,$3D,$FE,$32,$8F,$F2,$3B,$73,$FE,$F4,$BD
	.dc.b	$FD,$FC,$07,$E9,$38,$7F,$0F,$5F,$13,$AF,$1D,$79,$75,$D3,$BE,$BB
	.dc.b	$9F,$F9,$53,$04,$7E,$BB,$E7,$BF,$DB,$7C,$7F,$E6,$95,$F4,$39,$F9
	.dc.b	$3D,$3B,$F2,$B8,$79,$17,$2F,$F2,$FD,$FF,$16,$6F,$BD,$D4,$ED,$EF
	.dc.b	$58,$FF,$32,$FC,$FF,$D4,$6F,$67,$F0,$08,$FF,$6D,$C6,$0D,$FC,$7E
	.dc.b	$4D,$7B,$76,$39,$FA,$BC,$7D,$8E,$4D,$8F,$F3,$9F,$CF,$7F,$A4,$5B
	.dc.b	$CB,$FC,$3A,$31,$CF,$FE,$29,$7E,$69,$F9,$27,$FD,$30,$FE,$B6,$79
	.dc.b	$FB,$BC,$79,$F9,$78,$92,$7B,$F8,$F7,$0F,$F1,$A7,$CF,$3E,$83,$F8
	.dc.b	$EF,$CF,$BD,$7F,$E3,$F3,$9C,$3F,$D1,$33,$96,$CF,$0E,$8E,$2D,$8D
	.dc.b	$C5,$C3,$FC,$85,$5C,$5B,$1E,$0F,$86,$95,$DB,$E2,$2D,$70,$F7,$3C
	.dc.b	$A6,$3B,$BF,$D5,$98,$A3,$D3,$D5,$30,$FF,$4F,$FC,$FF,$DB,$C3,$C3
	.dc.b	$3E,$EE,$B9,$5E,$C8,$B3,$FE,$57,$F9,$EF,$A3,$97,$FA,$E5,$B5,$FD
	.dc.b	$2F,$07,$CC,$FF,$44,$9C,$DF,$FB,$2D,$FD,$19,$1D,$73,$FF,$4A,$F9
	.dc.b	$EF,$F1,$2E,$1F,$EB,$46,$CF,$86,$75,$E5,$75,$F6,$65,$FF,$AE,$B0
	.dc.b	$FE,$04,$D8,$F1,$7C,$0C,$8C,$1B,$7C,$DD,$32,$3E,$2F,$CF,$DC,$BE
	.dc.b	$07,$8F,$EB,$31,$BF,$C1,$7D,$DE,$A4,$BF,$9B,$E1,$5B,$F9,$1E,$4F
	.dc.b	$C3,$31,$87,$C9,$71,$5F,$ED,$2F,$A3,$FF,$7B,$3E,$B8,$FE,$DB,$18
	.dc.b	$AF,$A5,$BF,$CD,$3E,$77,$C5,$5B,$F8,$51,$F4,$7D,$0C,$57,$D9,$7F
	.dc.b	$BB,$FC,$FF,$F3,$18,$FF,$49,$3F,$83,$CB,$FD,$E8,$E5,$BE,$4F,$03
	.dc.b	$EB,$70,$DC,$7E,$B3,$F6,$7D,$8E,$BF,$41,$67,$FD,$DB,$82,$33,$18
	.dc.b	$F8,$0B,$97,$89,$F2,$92,$38,$CD,$9F,$3F,$97,$EA,$3D,$7C,$36,$BB
	.dc.b	$C7,$BB,$E0,$31,$5E,$E7,$E0,$FC,$4F,$F8,$9E,$03,$24,$66,$30,$D8
	.dc.b	$CD,$FF,$8A,$67,$FC,$55,$0F,$A0,$F9,$A7,$69,$F8,$2F,$D0,$F9,$89
	.dc.b	$1E,$0E,$03,$FE,$33,$F8,$17,$83,$AC,$7E,$AD,$DB,$DC,$7A,$2B,$77
	.dc.b	$F3,$AB,$4F,$91,$FF,$1A,$C6,$A7,$9C,$EC,$8F,$FB,$4E,$35,$5F,$82
	.dc.b	$AE,$73,$F1,$B1,$47,$4C,$33,$3F,$26,$2D,$F2,$C3,$C4,$76,$C3,$A3
	.dc.b	$F0,$4F,$81,$7C,$B0,$EB,$1B,$32,$5C,$3C,$58,$76,$AE,$0D,$9D,$4F
	.dc.b	$8D,$B8,$78,$E3,$8C,$6A,$FB,$C7,$C1,$BE,$73,$C1,$B1,$F2,$0D,$47
	.dc.b	$1B,$F0,$46,$70,$F6,$B6,$7D,$2D,$3C,$25,$18,$79,$5F,$41,$9C,$3C
	.dc.b	$B5,$F8,$08,$DE,$1E,$48,$EE,$66,$B9,$57,$0B,$FA,$37,$DE,$73,$87
	.dc.b	$19,$F2,$DF,$27,$6D,$AF,$C7,$61,$B7,$D3,$1D,$E3,$11,$E8,$8C,$4F
	.dc.b	$C5,$3F,$4D,$1F,$00,$FD,$46,$1F,$41,$B7,$FA,$D2,$4F,$08,$F8,$66
	.dc.b	$9C,$4F,$B6,$3F,$73,$C5,$FE,$24,$9F,$49,$FE,$9D,$71,$1E,$88,$C7
	.dc.b	$AB,$F9,$89,$CC,$FA,$67,$27,$E2,$A3,$35,$DE,$7F,$CA,$B5,$C4,$95
	.dc.b	$B6,$C7,$DD,$AE,$37,$FE,$A6,$6D,$7E,$86,$7D,$35,$C3,$0F,$A1,$1D
	.dc.b	$A7,$A1,$1E,$93,$B8,$F7,$C7,$27,$53,$CA,$BB,$CA,$3F,$15,$18,$6D
	.dc.b	$73,$9F,$8C,$73,$3B,$D7,$43,$BB,$E5,$32,$49,$F0,$4F,$38,$F9,$87
	.dc.b	$38,$CC,$F3,$8E,$E7,$38,$CB,$CE,$3B,$D7,$9D,$FA,$B1,$93,$8C,$65
	.dc.b	$ED,$1C,$2B,$C5,$87,$D9,$AF,$BF,$CD,$FE,$45,$BF,$CA,$B5,$FE,$78
	.dc.b	$D7,$FD,$CC,$D7,$FA,$3F,$D0,$7F,$87,$AF,$F2,$CD,$8F,$CF,$AF,$97
	.dc.b	$78,$3A,$F0,$D7,$F9,$DC,$3F,$CF,$B8,$5E,$EF,$D0,$28,$CB,$E6,$8C
	.dc.b	$FD,$07,$FB,$26,$3F,$4C,$D7,$F5,$ED,$7F,$AB,$35,$FF,$80,$0B,$E0
	.dc.b	$FF,$B0,$C7,$E1,$76,$BF,$2F,$FE,$49,$8F,$CB,$07,$0E,$CF,$ED,$35
	.dc.b	$CD,$DB,$5D,$07,$1C,$1F,$C6,$DB,$5E,$4F,$07,$C7,$3F,$C7,$FE,$81
	.dc.b	$E4,$E2,$BC,$F1,$8C,$3C,$AD,$7B,$E2,$F3,$38,$47,$F7,$EF,$9E,$79
	.dc.b	$18,$84,$61,$F7,$B5,$F4,$E8,$F4,$F8,$3F,$56,$3D,$FC,$18,$33,$E2
	.dc.b	$F5,$F9,$DF,$E7,$1B,$FF,$39,$6E,$DF,$41,$B7,$FE,$12,$5F,$0D,$C4
	.dc.b	$91,$F4,$A3,$2F,$D4,$F9,$5F,$23,$7D,$6F,$64,$7F,$A6,$63,$07,$1F
	.dc.b	$A0,$F2,$11,$C1,$D7,$C6,$F8,$9B,$BD,$9A,$49,$FC,$1F,$DC,$7A,$16
	.dc.b	$4F,$9F,$F8,$04,$8F,$47,$93,$F0,$FC,$9F,$88,$C8,$F6,$47,$C7,$F9
	.dc.b	$3F,$0F,$D0,$C9,$F8,$11,$F2,$FC,$AD,$46,$0F,$88,$FC,$FF,$80,$FB
	.dc.b	$9F,$42,$F5,$29,$9A,$F9,$5E,$55,$7F,$2B,$AA,$FE,$C4,$F2,$D6,$D7
	.dc.b	$03,$F4,$78,$7B,$DF,$D2,$F8,$EF,$A0,$FC,$62,$FA,$1A,$62,$33,$F3
	.dc.b	$16,$B8,$C6,$1E,$7F,$01,$F9,$5F,$0D,$F1,$54,$EB,$1D,$BE,$06,$D7
	.dc.b	$03,$C6,$F8,$BE,$07,$F3,$DE,$13,$E3,$69,$9A,$F4,$FC,$0D,$7F,$6F
	.dc.b	$B2,$FD,$73,$CB,$9F,$9F,$99,$2F,$F8,$2E,$29,$FD,$67,$5F,$F2,$19
	.dc.b	$FC,$3F,$9F,$83,$39,$6B,$C0,$DF,$81,$2B,$F1,$B1,$E8,$F3,$64,$76
	.dc.b	$7D,$7E,$EF,$6E,$89,$3F,$94,$7C,$3E,$07,$8B,$C0,$93,$F9,$87,$CB
	.dc.b	$EC,$7C,$4F,$84,$F8,$BE,$64,$AF,$5C,$7A,$BC,$2F,$5F,$B3,$27,$F4
	.dc.b	$31,$EE,$F6,$7D,$FF,$0D,$27,$F4,$B1,$F4,$39,$64,$7C,$67,$E9,$F9
	.dc.b	$DF,$C4,$67,$FA,$1C,$FF,$2E,$6B,$FE,$73,$F5,$75,$F1,$5B,$F0,$E4
	.dc.b	$ED,$E5,$E1,$7D,$EF,$7F,$EF,$F9,$D2,$BE,$AC,$77,$F9,$49,$1D,$DF
	.dc.b	$17,$87,$E3,$F0,$E4,$F8,$DF,$37,$C5,$FF,$27,$FD,$97,$F8,$01,$FC
	.dc.b	$4A,$7F,$9E,$B7,$D4,$FF,$2B,$3F,$C1,$77,$FC,$54,$FB,$7C,$4F,$80
	.dc.b	$7C,$1F,$17,$E1,$78,$FF,$0F,$E3,$A5,$7C,$38,$F0,$7B,$D2,$3F,$A9
	.dc.b	$E2,$8F,$27,$89,$FC,$EF,$E7,$5D,$D7,$BE,$61,$FD,$6A,$72,$DF,$4B
	.dc.b	$87,$CE,$C8,$C1,$BE,$EE,$93,$23,$07,$3E,$EF,$07,$0C,$8C,$1D,$FD
	.dc.b	$1F,$24,$C8,$C1,$FD,$B3,$E7,$1D,$1F,$D4,$FE,$75,$E0,$53,$FA,$B4
	.dc.b	$EA,$B8,$75,$62,$E4,$60,$FE,$EF,$F0,$EF,$92,$FE,$BD,$D2,$CE,$9F
	.dc.b	$D8,$67,$55,$C3,$E5,$71,$72,$30,$7F,$80,$BE,$77,$F8,$F3,$FE,$04
	.dc.b	$C1,$C9,$D8,$FE,$D9,$3A,$9F,$F0,$0A,$F5,$73,$E1,$49,$E9,$1C,$3E
	.dc.b	$6B,$8B,$23,$C1,$1D,$35,$F5,$F2,$49,$F2,$47,$83,$7D,$46,$0F,$43
	.dc.b	$E4,$F0,$7F,$CC,$51,$FE,$94,$C9,$18,$8F,$F7,$E4,$63,$C0,$FF,$49
	.dc.b	$36,$38,$78,$2F,$9D,$CB,$63,$31,$83,$FC,$76,$BE,$76,$91,$FF,$1F
	.dc.b	$BF,$E4,$0F,$88,$FF,$7E,$E1,$FE,$BF,$9F,$F2,$4F,$C1,$7F,$8C,$91
	.dc.b	$CC,$7C,$EB,$9B,$FC,$77,$1C,$0F,$F1,$89,$F8,$39,$E5,$CB,$D9,$DE
	.dc.b	$16,$7F,$C8,$73,$5E,$FC,$D5,$39,$E6,$7C,$6E,$63,$5F,$37,$31,$F4
	.dc.b	$45,$BF,$1F,$C3,$F3,$31,$AF,$BD,$F0,$2E,$6A,$77,$F0,$EE,$1F,$E5
	.dc.b	$49,$FD,$4B,$79,$B3,$87,$F9,$42,$72,$FF,$94,$BE,$77,$E2,$47,$FB
	.dc.b	$AD,$B5,$FE,$62,$D7,$FD,$E6,$FF,$98,$23,$35,$F4,$1F,$94,$F3,$DB
	.dc.b	$E3,$46,$A0,$EA,$FB,$7C,$66,$8D,$FC,$95,$C3,$B7,$C0,$72,$31,$87
	.dc.b	$9D,$EB,$ED,$49,$F2,$7C,$05,$AF,$67,$B7,$E2,$31,$97,$B5,$F2,$FB
	.dc.b	$32,$FD,$9E,$CD,$BF,$F3,$AE,$2F,$91,$FC,$03,$FE,$41,$AF,$F4,$0B
	.dc.b	$96,$FC,$3E,$B1,$DE,$16,$76,$FE,$27,$E7,$7F,$CB,$71,$FE,$8A,$6E
	.dc.b	$3F,$E8,$47,$F9,$D4,$67,$E2,$FF,$21,$F3,$BF,$8E,$8F,$F4,$8B,$6B
	.dc.b	$FD,$16,$BA,$B7,$D5,$C3,$86,$5F,$0F,$85,$B8,$70,$F3,$3A,$FB,$3D
	.dc.b	$9B,$5F,$E9,$EF,$9D,$FF,$17,$B7,$D8,$FD,$14,$3D,$9B,$1F,$A5,$E1
	.dc.b	$FC,$01,$3A,$F1,$E1,$67,$AB,$5D,$77,$2F,$F5,$5D,$A3,$F5,$DF,$3B
	.dc.b	$FE,$C4,$9F,$F5,$7A,$F7,$B3,$EF,$E9,$DF,$99,$C3,$CB,$B7,$C1,$E7
	.dc.b	$F2,$B2,$39,$FC,$9D,$E4,$FF,$59,$FC,$EF,$D0,$6F,$BD,$FB,$F8,$FF
	.dc.b	$6C,$57,$FB,$01,$D7,$86,$8F,$F6,$3C,$3E,$17,$81,$C7,$D6,$B3,$C7
	.dc.b	$D7,$E9,$ED,$75,$6C,$7F,$B3,$3E,$77,$F8,$DB,$79,$BF,$88,$46,$34
	.dc.b	$FF,$64,$BF,$AD,$3F,$28,$E3,$9B,$7E,$2F,$07,$8F,$8B,$30,$ED,$E4
	.dc.b	$5B,$FF,$69,$FC,$EB,$E1,$3F,$8F,$7C,$EB,$D9,$FE,$41,$39,$C3,$FD
	.dc.b	$C3,$39,$6C,$F0,$E8,$E2,$D8,$DC,$5C,$3F,$DC,$35,$C5,$B1,$E0,$F8
	.dc.b	$59,$5D,$BE,$5A,$D7,$0F,$47,$CA,$63,$9F,$E9,$47,$F4,$2C,$53,$EA
	.dc.b	$EA,$98,$7F,$BE,$3E,$77,$EC,$BF,$EF,$EA,$4F,$C0,$EB,$93,$EE,$8B
	.dc.b	$87,$FB,$E3,$E7,$3E,$96,$5E,$08,$46,$2B,$FA,$9C,$31,$7C,$7F,$F4
	.dc.b	$9A,$D4,$FF,$58,$7F,$93,$11,$D7,$3F,$F4,$FF,$9D,$FF,$10,$E1,$FD
	.dc.b	$7B,$1F,$04,$62,$BC,$95,$BF,$27,$B7,$B3,$30,$E0,$79,$31,$FE,$0C
	.dc.b	$E2,$FC,$53,$E5,$F6,$E4,$E6,$7D,$B3,$87,$E1,$74,$FA,$CF,$EC,$9A
	.dc.b	$FF,$B3,$6B,$86,$09,$F8,$1E,$34,$9C,$C5,$BF,$99,$EA,$7C,$53,$9F
	.dc.b	$83,$6B,$9F,$91,$F2,$4C,$57,$D3,$71,$1E,$7F,$9D,$FD,$A3,$C0,$C1
	.dc.b	$F4,$76,$8F,$9B,$E3,$6D,$FD,$D9,$FA,$DE,$7E,$38,$FE,$F6,$C8,$FF
	.dc.b	$01,$7C,$EF,$F3,$89,$FC,$13,$F8,$CB,$FF,$09,$B9,$C7,$F0,$6E,$61
	.dc.b	$C9,$FE,$15,$9F,$9C,$D8,$57,$4F,$23,$F0,$5C,$9F,$83,$F6,$3C,$3F
	.dc.b	$25,$71,$FA,$D9,$23,$CB,$EF,$C9,$CC,$5C,$BA,$C7,$92,$7F,$C5,$91
	.dc.b	$E8,$6A,$AF,$F0,$DC,$FF,$8A,$A3,$4D,$75,$E5,$70,$9F,$F3,$0C,$E5
	.dc.b	$B3,$C5,$AF,$F8,$D1,$FF,$4A,$C6,$FD,$EE,$44,$79,$7C,$87,$D0,$3E
	.dc.b	$8F,$C5,$D8,$F8,$31,$E1,$23,$C0,$F3,$6B,$FD,$2F,$0F,$82,$7F,$92
	.dc.b	$1C,$6D,$F6,$BE,$5B,$FC,$7D,$18,$AF,$DD,$B6,$B8,$7A,$5F,$82,$AD
	.dc.b	$B2,$BF,$51,$5B,$AF,$95,$59,$7C,$27,$FA,$DE,$B1,$7E,$1B,$E3,$18
	.dc.b	$7C,$2F,$77,$24,$9E,$77,$DF,$36,$1C,$67,$CF,$5B,$C3,$B4,$70,$C3
	.dc.b	$4F,$8E,$7F,$17,$58,$C3,$D4,$75,$71,$5C,$30,$EF,$5D,$4F,$39,$6F
	.dc.b	$F4,$75,$C2,$31,$1F,$11,$B7,$DA,$FD,$D3,$E5,$9C,$C6,$30,$FE,$61
	.dc.b	$5B,$9E,$78,$78,$AF,$81,$2B,$77,$C6,$BC,$B3,$96,$CF,$C4,$36,$F8
	.dc.b	$0F,$86,$58,$E0,$7B,$1C,$47,$08,$FB,$B5,$E3,$8C,$12,$7B,$9F,$C1
	.dc.b	$30,$F8,$95,$D0,$F8,$85,$C3,$DC,$F4,$8D,$D7,$38,$F1,$1B,$C3,$F7
	.dc.b	$35,$F9,$F8,$FF,$50,$B5,$F8,$26,$23,$64,$BF,$D3,$B6,$3F,$2F,$CB
	.dc.b	$FE,$0E,$BF,$04,$23,$15,$EC,$8F,$F5,$9C,$61,$FC,$89,$CB,$CC,$FC
	.dc.b	$7C,$66,$3F,$20,$66,$7F,$0B,$19,$C3,$F5,$4D,$C3,$FD,$64,$D9,$FE
	.dc.b	$AC,$DA,$E4,$7C,$B7,$18,$7F,$77,$6C,$FF,$76,$6C,$FA,$30,$F2,$92
	.dc.b	$7C,$6D,$8E,$8D,$AF,$87,$5C,$A7,$E2,$4E,$6B,$E2,$46,$1E,$64,$79
	.dc.b	$B6,$B9,$CF,$19,$F9,$31,$E9,$8C,$47,$C8,$8C,$4F,$13,$BD,$78,$C9
	.dc.b	$5C,$30,$F9,$6D,$C3,$E5,$B6,$BE,$5B,$70,$F9,$78,$79,$27,$B4,$7F
	.dc.b	$BC,$3E,$77,$FC,$3B,$7F,$D0,$DB,$1F,$D1,$D7,$C8,$3C,$1D,$78,$4B
	.dc.b	$FE,$95,$F2,$3F,$E2,$6C,$2C,$FF,$4D,$51,$97,$FA,$93,$5F,$F7,$EF
	.dc.b	$CE,$34,$E0,$D8,$FE,$AC,$D7,$FA,$D3,$5F,$F1,$B8,$F9,$B7,$B3,$E1
	.dc.b	$6C,$7F,$5F,$6B,$FD,$8D,$AF,$8B,$E1,$3C,$24,$7C,$2F,$93,$C3,$91
	.dc.b	$E4,$6B,$FD,$B7,$E6,$1E,$6F,$C1,$6C,$7F,$6F,$6B,$B8,$F7,$FA,$72
	.dc.b	$3D,$ED,$7D,$FC,$BF,$E5,$4E,$4F,$F2,$C6,$9F,$B9,$9F,$F1,$F3,$5F
	.dc.b	$EA,$AB,$C3,$E1,$CB,$FE,$5C,$F9,$AF,$F3,$05,$FF,$98,$5B,$19,$37
	.dc.b	$61,$0F,$85,$CC,$3E,$71,$9E,$2B,$53,$FE,$51,$6B,$FD,$A2,$76,$FF
	.dc.b	$95,$5F,$F4,$C3,$7D,$6E,$B1,$88,$FE,$DF,$2F,$99,$ED,$E1,$F7,$64
	.dc.b	$F7,$8F,$2F,$CD,$7B,$23,$D5,$C9,$F8,$9D,$7F,$D0,$0D,$D7,$FC,$FF
	.dc.b	$5B,$AF,$7F,$99,$EB,$F9,$A9,$1E,$B8,$F7,$78,$BE,$F6,$47,$BE,$3E
	.dc.b	$17,$C9,$7C,$6C,$5F,$0D,$4C,$D7,$C5,$F1,$AB,$EE,$AF,$95,$1F,$2F
	.dc.b	$DF,$F9,$FF,$36,$F9,$CA,$66,$FE,$77,$8F,$5F,$8F,$5F,$4E,$38,$EB
	.dc.b	$67,$67,$FA,$16,$D1,$F7,$3C,$57,$CB,$5F,$9C,$97,$FE,$8B,$9C,$B7
	.dc.b	$E5,$F8,$54,$8C,$1B,$F8,$0F,$C3,$F8,$2F,$89,$A7,$78,$F0,$7C,$0D
	.dc.b	$AE,$07,$77,$C9,$F0,$3F,$90,$F0,$DF,$19,$4C,$D7,$97,$E0,$AB,$FE
	.dc.b	$36,$D5,$7E,$D9,$E5,$CD,$CF,$C8,$97,$FD,$4A,$3F,$13,$7F,$8A,$F3
	.dc.b	$24,$FF,$AA,$FB,$3F,$91,$B7,$C0,$FC,$7F,$81,$23,$07,$F5,$CF,$27
	.dc.b	$D0,$47,$FA,$FF,$B7,$F9,$6E,$79,$1D,$5F,$CC,$F8,$1F,$9A,$F0,$24
	.dc.b	$F8,$5F,$CE,$FA,$9F,$0F,$E1,$3E,$27,$91,$2B,$FB,$5C,$7E,$8B,$C1
	.dc.b	$FD,$1F,$AB,$27,$D5,$1F,$A6,$F9,$AF,$D3,$FC,$3C,$9F,$74,$7C,$FE
	.dc.b	$99,$1F,$DD,$9F,$A5,$E6,$FF,$02,$9F,$E8,$53,$FE,$CF,$6B,$FE,$66
	.dc.b	$6F,$93,$FE,$EF,$5F,$15,$C7,$C3,$95,$C5,$E9,$E1,$7D,$FF,$77,$F0
	.dc.b	$1E,$74,$AF,$F0,$3C,$78,$7E,$4E,$47,$85,$F2,$78,$7E,$5F,$0E,$4F
	.dc.b	$95,$F4,$7C,$5F,$F1,$AF,$5B,$F8,$01,$FC,$5B,$67,$9F,$FC,$86,$FF
	.dc.b	$AC,$FA,$5B,$8F,$81,$E3,$7C,$1F,$1A,$57,$C1,$8F,$89,$E4,$7C,$5F
	.dc.b	$8D,$93,$F1,$63,$C1,$ED,$C8,$FF,$7B,$66,$8F,$27,$8F,$FC,$BB,$E7
	.dc.b	$1D,$F7,$C0,$98,$7F,$BF,$67,$2D,$F4,$B8,$7C,$E4,$8C,$1B,$EF,$E9
	.dc.b	$32,$30,$73,$EF,$F0,$76,$C8,$C1,$DF,$D2,$F2,$4C,$8C,$1F,$D1,$F9
	.dc.b	$7E,$4B,$FD,$27,$E6,$DD,$5F,$D2,$3E,$73,$FA,$A4,$EF,$9F,$E7,$46
	.dc.b	$23,$FD,$5B,$5C,$1D,$1C,$FA,$FF,$A8,$C6,$63,$77,$89,$FE,$AB,$F0
	.dc.b	$7F,$DF,$B1,$F1,$D3,$A5,$9F,$F5,$A6,$C7,$61,$5C,$73,$FC,$12,$F9
	.dc.b	$3E,$BE,$EC,$AE,$13,$97,$97,$CE,$7F,$26,$7F,$D8,$F8,$39,$7C,$07
	.dc.b	$F6,$A9,$D6,$7E,$65,$EB,$67,$C3,$98,$74,$8E,$1F,$33,$C5,$91,$E0
	.dc.b	$8E,$9C,$1E,$DE,$59,$3E,$48,$F0,$70,$EA,$30,$7A,$1F,$27,$84,$CF
	.dc.b	$FC,$5B,$3F,$E0,$3C,$D3,$98,$CC,$62,$77,$08,$C3,$AF,$09,$F3,$9A
	.dc.b	$94,$62,$75,$39,$8C,$47,$4F,$9C,$FF,$26,$26,$91,$F0,$FC,$27,$E4
	.dc.b	$B0,$FC,$0C,$7F,$96,$79,$7E,$F3,$FE,$1D,$AF,$F0,$F7,$CE,$7E,$02
	.dc.b	$3F,$80,$5F,$1A,$FF,$26,$F2,$D9,$FE,$08,$D7,$FC,$4D,$35,$FF,$28
	.dc.b	$CF,$6E,$6C,$1B,$E5,$7C,$46,$27,$C7,$F3,$3C,$4C,$5F,$D1,$8F,$F2
	.dc.b	$EC,$75,$F9,$19,$1C,$0F,$5B,$E3,$D5,$CB,$4F,$07,$85,$6B,$F3,$D3
	.dc.b	$FB,$46,$F2,$E7,$1F,$01,$FB,$B7,$DD,$F3,$9F,$26,$32,$1C,$57,$EB
	.dc.b	$B4,$78,$1C,$37,$E3,$C1,$32,$FD,$76,$1C,$1F,$E5,$CF,$C8,$C7,$AA
	.dc.b	$F9,$1F,$09,$E2,$F9,$1B,$5C,$BE,$03,$CA,$62,$F8,$BE,$8F,$6A,$4F
	.dc.b	$83,$E0,$2D,$76,$F6,$FD,$A6,$37,$FE,$0F,$9A,$3E,$0F,$B3,$30,$F4
	.dc.b	$FB,$37,$0F,$87,$F1,$7F,$72,$7F,$85,$36,$33,$EE,$FE,$1E,$31,$97
	.dc.b	$F1,$45,$E5,$64,$D7,$91,$70,$FE,$29,$F6,$AF,$A2,$FC,$83,$63,$3E
	.dc.b	$0E,$27,$3D,$12,$78,$FA,$B7,$5F,$B3,$F6,$8E,$FF,$CD,$36,$BF,$91
	.dc.b	$3F,$C7,$DB,$1A,$E0,$91,$AE,$2B,$7F,$84,$FC,$03,$F9,$6C,$FF,$2B
	.dc.b	$6C,$67,$B3,$F5,$11,$8E,$2F,$E6,$B5,$FC,$08,$FE,$64,$5A,$FC,$7B
	.dc.b	$FC,$40,$CE,$DF,$B1,$FB,$5F,$F4,$B4,$7F,$39,$6C,$F0,$FA,$5D,$3B
	.dc.b	$F1,$25,$6F,$BA,$CF,$E7,$2F,$F0,$13,$CF,$4E,$DD,$7D,$DF,$E0,$1E
	.dc.b	$37,$FA,$4E,$7F,$A4,$36,$FF,$4A,$BE,$1B,$7F,$0B,$C3,$CD,$97,$C3
	.dc.b	$D5,$B8,$7E,$B3,$ED,$3F,$73,$3F,$D4,$DB,$19,$F4,$3F,$88,$46,$38
	.dc.b	$BF,$AE,$AF,$9B,$C9,$AF,$9B,$B8,$73,$E3,$E2,$72,$FA,$5B,$1F,$D8
	.dc.b	$7C,$9F,$F3,$6B,$7C,$7F,$E3,$D1,$FB,$38,$C1,$FD,$9F,$F0,$AF,$F6
	.dc.b	$23,$7E,$07,$F9,$24,$7F,$8A,$A3,$07,$F6,$DF,$9E,$FF,$72,$B7,$E5
	.dc.b	$BF,$95,$7C,$F6,$F8,$7F,$86,$D6,$9B,$E1,$7F,$2F,$8F,$F3,$DD,$F1
	.dc.b	$3A,$46,$7C,$39,$18,$3C,$0F,$1D,$E4,$7F,$7F,$C3,$FC,$60,$F9,$63
	.dc.b	$B7,$04,$8C,$1E,$97,$FA,$07,$CE,$FE,$3B,$8F,$FC,$1A,$D9,$FF,$24
	.dc.b	$5F,$E0,$5F,$F0,$64,$67,$3F,$F7,$63,$FC,$85,$BB,$F4,$7A,$BD,$BC
	.dc.b	$39,$5E,$7E,$8B,$5F,$CC,$AD,$E9,$F9,$0C,$E5,$E4,$7C,$FE,$E4,$C3
	.dc.b	$C7,$16,$F9,$7A,$1E,$C3,$E9,$FC,$A5,$AF,$4F,$A5,$F0,$0C,$57,$85
	.dc.b	$F8,$7E,$0C,$9F,$99,$F1,$D6,$BC,$DE,$17,$C8,$31,$A7,$C8,$7E,$5F
	.dc.b	$C6,$CB,$FA,$9E,$FD,$BF,$77,$C6,$FD,$03,$18,$7D,$07,$E9,$FC,$54
	.dc.b	$BF,$9F,$16,$BF,$AB,$FC,$DF,$E5,$9B,$DD,$FD,$E7,$E7,$9F,$2D,$FD
	.dc.b	$EE,$B5,$97,$F9,$21,$FC,$2D,$F2,$F8,$AD,$61,$B7,$87,$B1,$33,$FF
	.dc.b	$25,$7C,$F3,$E7,$B1,$87,$8B,$E0,$7B,$18,$CF,$B3,$E1,$E3,$98,$7E
	.dc.b	$57,$DA,$B7,$EA,$E2,$F3,$18,$C3,$CC,$FA,$7E,$02,$5F,$F9,$6F,$7F
	.dc.b	$F2,$83,$63,$DD,$F1,$52,$3C,$FD,$57,$1F,$8F,$F2,$7F,$0C,$C6,$5F
	.dc.b	$0D,$F8,$DC,$72,$FF,$D8,$1E,$25,$BF,$F0,$2F,$1B,$DE,$FF,$13,$36
	.dc.b	$BC,$97,$89,$FF,$15,$7D,$DB,$E4,$FF,$C6,$F8,$04,$66,$FF,$C9,$AD
	.dc.b	$8F,$F2,$AB,$5F,$F5,$C1,$C2,$7F,$C6,$DF,$85,$FF,$B4,$2B,$FC,$FD
	.dc.b	$5C,$74,$FF,$42,$CE,$6F,$FC,$3B,$2F,$8C,$F2,$9F,$E3,$FF,$91,$FE
	.dc.b	$7D,$7F,$E8,$C9,$D6,$1D,$2B,$D3,$7E,$86,$CE,$A3,$CF,$18,$F5,$25
	.dc.b	$7B,$1B,$3E,$C7,$C1,$D5,$F1,$9B,$3E,$AF,$88,$F9,$27,$97,$A6,$C6
	.dc.b	$49,$7F,$E3,$8C,$FF,$C2,$31,$FE,$DD,$CB,$FA,$53,$6F,$8D,$6E,$EE
	.dc.b	$7E,$F9,$B5,$FE,$E5,$C3,$9D,$FD,$7C,$39,$B6,$B6,$FD,$B9,$DB,$F7
	.dc.b	$23,$2D,$AF,$F7,$1E,$18,$BE,$73,$93,$C9,$3E,$CC,$3C,$8D,$3C,$95
	.dc.b	$C6,$FA,$E1,$E0,$C3,$99,$23,$E4,$57,$1C,$3D,$0D,$55,$C6,$7A,$4F
	.dc.b	$98,$D5,$73,$71,$5F,$8A,$9D,$BE,$83,$11,$C2,$3C,$31,$82,$61,$EC
	.dc.b	$78,$34,$93,$F5,$CE,$EE,$23,$D2,$72,$8F,$86,$77,$3F,$53,$3F,$94
	.dc.b	$23,$82,$47,$EB,$23,$2D,$8F,$D4,$FC,$47,$D5,$23,$F0,$9B,$1E,$13
	.dc.b	$F0,$8E,$2B,$CD,$1F,$BC,$8C,$4F,$AE,$3F,$63,$18,$7D,$73,$FA,$F8
	.dc.b	$C4,$7E,$0E,$31,$3E,$12,$47,$84,$EC,$E2,$3C,$37,$FE,$A1,$9D,$36
	.dc.b	$33,$5D,$A3,$F0,$ED,$BF,$E2,$ED,$BF,$86,$75,$3D,$55,$E4,$BE,$B3
	.dc.b	$B7,$DB,$3B,$7D,$AF,$EC,$E3,$53,$C4,$FE,$1D,$F0,$7F,$C9,$8F,$E4
	.dc.b	$FE,$FD,$AF,$E3,$71,$FC,$38,$8F,$F2,$67,$F8,$8F,$24,$8F,$E5,$2D
	.dc.b	$C3,$FD,$FB,$9B,$C7,$FE,$32,$DA,$FF,$7D,$B6,$77,$3A,$F9,$69,$5A
	.dc.b	$6C,$FF,$32,$1F,$41,$FE,$63,$E5,$C4,$63,$DD,$FE,$8C,$EE,$5F,$26
	.dc.b	$F4,$77,$1C,$67,$A7,$CF,$BE,$83,$FA,$34,$7F,$B5,$3C,$BF,$F6,$43
	.dc.b	$FD,$27,$89,$AE,$A7,$33,$FE,$DA,$BE,$4E,$E7,$9C,$3B,$FA,$DF,$2A
	.dc.b	$FE,$A9,$3C,$1E,$58,$6A,$F8,$DD,$9D,$FC,$6D,$5F,$38,$F6,$30,$D8
	.dc.b	$F1,$1C,$0A,$31,$7A,$1C,$D5,$D3,$11,$9F,$8D,$F0,$7C,$F7,$F9,$1A
	.dc.b	$F9,$B5,$F8,$51,$CF,$6E,$11,$E0,$C1,$39,$8C,$7A,$15,$F9,$55,$E3
	.dc.b	$8F,$77,$CF,$FC,$79,$70,$5F,$9F,$FF,$52,$47,$B1,$AE,$B8,$3E,$DD
	.dc.b	$71,$AE,$4B,$D1,$EB,$32,$BE,$2C,$E5,$BE,$CF,$CF,$FA,$07,$CE,$D3
	.dc.b	$CB,$1E,$6F,$1A,$D7,$03,$9B,$EA,$EC,$FB,$1F,$59,$FE,$AD,$9F,$A6
	.dc.b	$D7,$D0,$7F,$85,$F1,$4F,$C2,$F4,$F9,$4F,$9A,$30,$FA,$3D,$2A,$F2
	.dc.b	$14,$AF,$97,$3F,$33,$E2,$7E,$DF,$D0,$CB,$F8,$31,$F4,$3D,$2D,$46
	.dc.b	$0F,$8A,$FD,$4F,$57,$F2,$5C,$7F,$BF,$9F,$A5,$E2,$65,$F9,$7E,$8D
	.dc.b	$7C,$9A,$2F,$EE,$CF,$E6,$FC,$3F,$F1,$C5,$7E,$5B,$B9,$E5,$FE,$70
	.dc.b	$D7,$A3,$BF,$42,$D7,$E7,$2F,$8B,$F0,$5F,$F1,$83,$9F,$43,$83,$B9
	.dc.b	$F0,$47,$87,$E6,$BF,$53,$F3,$D2,$BA,$47,$97,$E6,$F5,$18,$3C,$0F
	.dc.b	$A7,$E1,$7F,$69,$E0,$BE,$26,$99,$AF,$37,$91,$5F,$35,$7C,$18,$F8
	.dc.b	$5E,$87,$EF,$7D,$0F,$B1,$5F,$B8,$6B,$ED,$8F,$91,$F0,$DC,$23,$E1
	.dc.b	$46,$1F,$89,$F5,$55,$F8,$93,$F3,$E3,$F8,$57,$DB,$3C,$1B,$E1,$4B
	.dc.b	$FE,$0F,$39,$6F,$CC,$F0,$F9,$69,$18,$3F,$86,$FD,$B3,$D8,$FE,$2C
	.dc.b	$B8,$A9,$88,$CF,$06,$BE,$36,$5F,$28,$FE,$3D,$F6,$AF,$97,$BF,$2F
	.dc.b	$2B,$F8,$DC,$E5,$BF,$13,$C3,$D5,$91,$83,$F9,$0F,$7B,$E0,$BF,$95
	.dc.b	$3F,$BF,$8F,$D4,$37,$D2,$FE,$20,$EE,$38,$F9,$3C,$3E,$56,$46,$0F
	.dc.b	$E5,$BF,$6B,$F9,$86,$F8,$3F,$CD,$E0,$DB,$C7,$16,$7C,$09,$7C,$A3
	.dc.b	$87,$CA,$F1,$F9,$59,$3D,$A3,$FA,$17,$DA,$FE,$3E,$3F,$9E,$BF,$D5
	.dc.b	$35,$FE,$50,$DF,$5B,$FA,$EB,$B9,$FE,$0D,$F6,$8D,$BF,$A6,$2F,$8A
	.dc.b	$A6,$23,$FA,$87,$77,$AA,$BF,$A5,$BF,$AE,$7F,$88,$37,$1F,$E7,$0E
	.dc.b	$E3,$3E,$17,$2F,$A6,$91,$CA,$3A,$EB,$DB,$C2,$93,$DA,$3F,$B0,$7D
	.dc.b	$A7,$AE,$3F,$D7,$8F,$F6,$D7,$FA,$EB,$5F,$E4,$4E,$D7,$D3,$FF,$43
	.dc.b	$DF,$FB,$FC,$FF,$3C,$9E,$3E,$D7,$F6,$DA,$D4,$7F,$B2,$5F,$88,$FF
	.dc.b	$6A,$5E,$BE,$CE,$1B,$F1,$E8,$E5,$C1,$2F,$84,$75,$F5,$B5,$18,$3A
	.dc.b	$3E,$1F,$17,$C5,$E2,$C9,$F0,$47,$97,$D3,$D4,$60,$F2,$3F,$E0,$5F
	.dc.b	$A6,$7C,$57,$F8,$1F,$E9,$DF,$01,$4D,$46,$FB,$6E,$1F,$E0,$6B,$E2
	.dc.b	$D8,$FF,$08,$7C,$BB,$C6,$FF,$09,$7A,$0D,$29,$A8,$DF,$91,$70,$FF
	.dc.b	$08,$DF,$16,$C7,$F8,$6F,$E9,$FF,$11,$D1,$FE,$32,$C7,$E6,$D7,$F8
	.dc.b	$61,$AF,$F5,$D9,$DB,$FE,$FD,$7F,$1D,$DD,$FE,$07,$73,$06,$9A,$AF
	.dc.b	$F1,$FB,$FB,$78,$E9,$E2,$78,$CF,$27,$89,$E5,$DF,$DF,$CB,$2B,$FC
	.dc.b	$57,$7F,$19,$F2,$6F,$B7,$35,$ED,$8F,$77,$89,$FE,$1A,$9F,$F2,$E5
	.dc.b	$7F,$25,$CF,$FC,$B6,$D8,$CC,$ED,$FF,$2A,$9E,$BF,$31,$F4,$FC,$1B
	.dc.b	$1D,$1A,$F0,$AF,$E6,$93,$FE,$68,$80,$FA,$0C,$34,$D3,$83,$87,$C6
	.dc.b	$7F,$96,$71,$FF,$1A,$4F,$F9,$B7,$7F,$F2,$6B,$D2,$BA,$FD,$3B,$8F
	.dc.b	$D3,$87,$1B,$F5,$F2,$5C,$3F,$CE,$2D,$7C,$B3,$5F,$C8,$4F,$6E,$4C
	.dc.b	$1B,$E4,$7C,$96,$27,$C7,$F3,$BC,$4C,$5F,$F8,$94,$3D,$7E,$72,$4F
	.dc.b	$03,$D6,$F8,$F6,$FF,$09,$C7,$F9,$A9,$B5,$FE,$2F,$C3,$F9,$44,$E7
	.dc.b	$9B,$87,$2C,$8C,$1C,$5F,$77,$D3,$FC,$9B,$FF,$1C,$37,$0F,$F3,$06
	.dc.b	$0E,$ED,$ED,$C3,$79,$97,$F9,$F3,$0E,$0F,$FA,$69,$F8,$F8,$F4,$F9
	.dc.b	$DE,$E7,$83,$E4,$2D,$71,$F7,$3C,$66,$2B,$C6,$F9,$BD,$39,$3D,$FE
	.dc.b	$42,$D7,$A3,$D5,$F5,$98,$DF,$FC,$F1,$9A,$3D,$FF,$1D,$30,$F4,$7D
	.dc.b	$2D,$C3,$FC,$B1,$F2,$BF,$6A,$7F,$D2,$ED,$8C,$FB,$DF,$85,$8C,$65
	.dc.b	$FE,$85,$7F,$C9,$95,$C3,$C8,$D7,$8B,$B7,$F1,$9F,$4E,$F9,$7A,$63
	.dc.b	$C3,$B7,$FE,$A6,$9F,$F5,$63,$7A,$38,$FB,$BC,$9F,$CB,$7D,$3F,$E2
	.dc.b	$A7,$FD,$5A,$DD,$BF,$CE,$4F,$F8,$66,$33,$DD,$C3,$82,$47,$0E,$2B
	.dc.b	$7F,$E7,$9F,$A7,$FE,$59,$3F,$EB,$86,$C6,$7B,$3F,$4D,$18,$E2,$FF
	.dc.b	$53,$57,$F8,$C4,$FF,$5F,$96,$BF,$D0,$8F,$F8,$CC,$CE,$DF,$AF,$FA
	.dc.b	$7F,$E0,$71,$FE,$C6,$6C,$F0,$F7,$74,$EF,$C3,$95,$BE,$EB,$3F,$E9
	.dc.b	$1B,$FC,$04,$F3,$D3,$B7,$5F,$77,$F7,$DE,$33,$DB,$A6,$3E,$6A,$DF
	.dc.b	$FB,$62,$5E,$6F,$03,$8F,$A7,$67,$FD,$BB,$F4,$FF,$B4,$E1,$FF,$64
	.dc.b	$36,$3F,$DF,$AF,$F8,$C9,$DF,$8F,$FC,$3E,$1F,$0D,$4C,$7B,$B6,$BA
	.dc.b	$F1,$EC,$E1,$EE,$58,$FF,$76,$F9,$3F,$E8,$16,$F8,$FF,$C6,$E1,$E4
	.dc.b	$D8,$FE,$3D,$F4,$FF,$ED,$3A,$FF,$7A,$B7,$9B,$FD,$74,$A3,$7F,$31
	.dc.b	$FC,$97,$E9,$DF,$0F,$4F,$E5,$1F,$4F,$BA,$FF,$7F,$CE,$9B,$E2,$71
	.dc.b	$38,$38,$37,$E2,$75,$34,$E0,$E7,$C1,$E1,$3B,$FC,$3D,$8E,$FC,$5E
	.dc.b	$53,$1A,$FF,$B4,$F1,$F9,$B3,$FD,$43,$97,$FD,$F9,$F4,$FF,$8E,$6D
	.dc.b	$7A,$BA,$BF,$A1,$4E,$6B,$FA,$BC,$A3,$F8,$FB,$FD,$69,$AE,$4E,$3C
	.dc.b	$7C,$BC,$59,$5E,$13,$9E,$FF,$91,$71,$5C,$E7,$C3,$E9,$49,$CC,$F8
	.dc.b	$67,$0F,$FB,$BF,$0F,$EB,$D3,$A6,$C6,$6B,$D7,$EA,$7B,$3D,$29,$7D
	.dc.b	$31,$7C,$AE,$30,$E9,$D1,$F0,$8C,$6D,$F2,$1F,$8B,$E1,$4C,$39,$7C
	.dc.b	$C5,$BF,$27,$87,$F2,$8C,$61,$F4,$1F,$9B,$F1,$F2,$FC,$5F,$07,$6F
	.dc.b	$D9,$F0,$1F,$48,$C6,$1F,$61,$FA,$BF,$4F,$2F,$E0,$45,$A5,$7E,$65
	.dc.b	$FF,$07,$FD,$27,$E7,$63,$10,$8C,$57,$DE,$FA,$77,$CC,$7F,$80,$E7
	.dc.b	$35,$CB,$AF,$06,$7C,$0B,$87,$D6,$AF,$F6,$E3,$7D,$8F,$F0,$67,$D3
	.dc.b	$BE,$9F,$A4,$75,$F7,$E5,$6C,$E4,$F8,$78,$E4,$7D,$7F,$66,$E9,$EE
	.dc.b	$8F,$CA,$E5,$FE,$84,$7C,$5C,$5A,$E0,$F7,$CE,$6B,$81,$F1,$3E,$27
	.dc.b	$DC,$7B,$EA,$DF,$AF,$E2,$7E,$11,$EA,$EB,$B1,$EF,$FA,$7F,$8C,$63
	.dc.b	$0E,$A7,$CD,$7E,$57,$1C,$9F,$CD,$FB,$0F,$1B,$17,$C3,$95,$EF,$7F
	.dc.b	$8C,$DB,$7F,$37,$0C,$4F,$F8,$DF,$E9,$9F,$37,$FE,$39,$FA,$77,$D4
	.dc.b	$66,$30,$CC,$3F,$CB,$79,$7F,$94,$AB,$71,$88,$46,$1D,$B5,$FF,$31
	.dc.b	$DB,$E7,$3F,$CA,$71,$FE,$5C,$8C,$57,$36,$C7,$81,$AF,$29,$FF,$28
	.dc.b	$7C,$DF,$D5,$E5,$FC,$AD,$70,$7F,$2F,$19,$BF,$E2,$73,$FD,$2E,$FE
	.dc.b	$7C,$FE,$41,$FA,$31,$9A,$E3,$D3,$D8,$97,$FC,$32,$74,$FF,$1E,$BE
	.dc.b	$55,$E2,$F8,$8F,$9C,$7C,$FF,$5E,$D7,$18,$B8,$F8,$26,$D7,$1A,$D1
	.dc.b	$DE,$32,$FA,$AA,$D7,$C9,$9A,$B0,$FF,$65,$D7,$28,$EC,$DB,$ED,$1F
	.dc.b	$02,$FC,$D3,$93,$B1,$63,$B3,$5E,$D1,$E0,$AF,$EF,$11,$F6,$E3,$07
	.dc.b	$47,$B4,$F1,$9F,$2C,$79,$0C,$92,$BC,$A7,$DC,$BE,$75,$C5,$B3,$B8
	.dc.b	$FB,$B1,$89,$E3,$18,$3C,$17,$E7,$9D,$47,$BA,$BC,$55,$C1,$F5,$47
	.dc.b	$73,$89,$5F,$01,$A3,$F2,$D5,$F9,$C7,$E2,$C7,$E6,$E3,$17,$F0,$27
	.dc.b	$F4,$71,$82,$4E,$CF,$EB,$38,$7F,$A5,$A7,$B8,$71,$7E,$78,$C7,$3F
	.dc.b	$E0,$A7,$35,$ED,$AF,$D1,$CE,$49,$1F,$6E,$77,$5E,$BA,$DC,$E6,$B7
	.dc.b	$3E,$2A,$DB,$EE,$3F,$57,$1A,$AF,$24,$EB,$0F,$BF,$1D,$2B,$95,$6E
	.dc.b	$FC,$C7,$EA,$DC,$47,$8E,$30,$78,$C8,$F2,$6C,$F8,$27,$8C,$F9,$63
	.dc.b	$C0,$66,$76,$78,$23,$FD,$DB,$7E,$7C,$3F,$86,$DF,$2C,$3D,$37,$E2
	.dc.b	$BF,$01,$30,$FE,$1F,$EC,$7F,$14,$3F,$97,$7C,$25,$AF,$B7,$1F,$85
	.dc.b	$C8,$D2,$47,$F2,$B8,$FF,$06,$B7,$C8,$E0,$47,$5E,$D7,$32,$38,$3F
	.dc.b	$9A,$7D,$43,$D1,$FE,$8B,$9F,$F3,$A8,$C7,$2E,$BE,$79,$E9,$70,$BD
	.dc.b	$DF,$F8,$E8,$FC,$87,$77,$F3,$F6,$9E,$77,$1A,$1D,$7D,$33,$EA,$3F
	.dc.b	$C8,$7C,$3F,$E0,$AF,$33,$FC,$DB,$87,$F8,$D9,$AF,$F6,$2A,$E0,$FF
	.dc.b	$8E,$97,$83,$CB,$7F,$F2,$6C,$6A,$78,$61,$94,$23,$8B,$CF,$E3,$34
	.dc.b	$BE,$85,$EC,$E1,$B1,$B3,$C5,$E8,$F0,$B5,$D3,$11,$9F,$8C,$EB,$F5
	.dc.b	$0F,$5E,$99,$C3,$D9,$B5,$7E,$05,$75,$8E,$DE,$95,$9D,$9F,$0D,$F4
	.dc.b	$FD,$1F,$F8,$D3,$87,$FC,$6B,$F4,$9F,$7E,$BF,$CC,$2D,$7F,$C1,$B3
	.dc.b	$BE,$1F,$F3,$AA,$F4,$F9,$C4,$9F,$F3,$4C,$E5,$BF,$0D,$F2,$3E,$A1
	.dc.b	$F3,$74,$F1,$C7,$93,$C6,$B5,$C0,$E6,$FA,$3E,$9F,$E8,$FD,$47,$FA
	.dc.b	$EA,$7F,$CE,$ED,$7C,$F1,$EE,$F5,$78,$47,$92,$30,$F9,$BD,$3A,$FB
	.dc.b	$A7,$E2,$C7,$C6,$F0,$FE,$9F,$D4,$4A,$F6,$C7,$CA,$EE,$D4,$60,$F8
	.dc.b	$2F,$CE,$F5,$BF,$0F,$C5,$FE,$80,$9F,$9B,$E3,$65,$F8,$FE,$8D,$7E
	.dc.b	$7D,$7D,$68,$FC,$7F,$8B,$FE,$30,$9F,$F4,$AF,$7B,$CD,$FC,$81,$AF
	.dc.b	$47,$7E,$7D,$AF,$F4,$DD,$F1,$7F,$C8,$EF,$F8,$AD,$CF,$CD,$F0,$77
	.dc.b	$3E,$08,$F0,$FC,$DF,$E7,$7E,$A2,$57,$48,$F2,$FC,$E6,$A3,$07,$81
	.dc.b	$F4,$FC,$47,$EA,$3C,$17,$C5,$D3,$35,$E6,$F2,$6B,$E6,$AF,$83,$1F
	.dc.b	$0B,$D9,$FD,$8F,$9A,$EA,$A6,$6F,$DF,$F1,$35,$F7,$D7,$CA,$8F,$97
	.dc.b	$F4,$36,$76,$7C,$67,$F7,$BF,$50,$F0,$2F,$83,$2B,$FD,$8B,$39,$6F
	.dc.b	$CD,$70,$F7,$A4,$60,$FF,$65,$FD,$43,$E5,$FF,$84,$AE,$3A,$62,$33
	.dc.b	$C3,$AF,$8F,$97,$CA,$3F,$88,$FD,$43,$E6,$2F,$CC,$4A,$FF,$6C,$CE
	.dc.b	$5B,$F1,$9C,$3D,$59,$18,$3F,$DB,$FE,$03,$E1,$7F,$8E,$BF,$B9,$8F
	.dc.b	$F4,$6B,$7D,$2F,$F5,$DB,$B8,$E3,$E4,$F0,$F7,$E4,$60,$FF,$75,$FD
	.dc.b	$47,$E5,$DB,$E1,$7F,$2A,$83,$6F,$1C,$59,$F0,$65,$F2,$8E,$1F,$2D
	.dc.b	$C7,$E5,$A4,$F6,$8F,$E6,$BF,$51,$F8,$E8,$FF,$7D,$BF,$D1,$B5,$FF
	.dc.b	$69,$B7,$D6,$FE,$7F,$28,$E0,$FF,$AB,$3E,$A3,$FB,$7B,$FD,$12,$BF
	.dc.b	$DB,$70,$F8,$7F,$E8,$6D,$9D,$5F,$48,$E3,$1F,$EB,$6E,$EF,$7B,$A3
	.dc.b	$FA,$7B,$FB,$37,$5E,$4F,$F4,$E8,$DC,$71,$F0,$F1,$F5,$12,$39,$09
	.dc.b	$FF,$5F,$38,$5A,$F3,$F0,$E4,$E6,$3F,$AD,$FD,$47,$C3,$8F,$EC,$2F
	.dc.b	$F6,$07,$FD,$9A,$DD,$7F,$DF,$6E,$E1,$F5,$1F,$D1,$5E,$3E,$CF,$43
	.dc.b	$FB,$3C,$66,$37,$BE,$3D,$79,$1E,$A9,$F9,$4F,$FB,$71,$74,$F2,$E9
	.dc.b	$93,$B8,$EB,$EB,$76,$64,$73,$8F,$0F,$8D,$E2,$F1,$A4,$F7,$8F,$2F
	.dc.b	$A9,$A8,$C1,$E3,$7F,$BF,$FD,$3B,$E3,$3F,$C0,$3F,$50,$F8,$0A,$6A
	.dc.b	$37,$DD,$70,$FA,$B7,$C5,$B1,$FE,$0A,$F9,$97,$8F,$FE,$0C,$F4,$9A
	.dc.b	$53,$51,$BF,$26,$E1,$F8,$0B,$E2,$D8,$FF,$0A,$FD,$47,$E2,$FA,$1E
	.dc.b	$07,$E7,$B2,$73,$77,$37,$DF,$C3,$C5,$B9,$7E,$33,$1E,$58,$7E,$99
	.dc.b	$FC,$9E,$3D,$27,$FC,$4F,$1F,$BD,$51,$95,$1C,$27,$95,$79,$BC,$5F
	.dc.b	$3F,$0F,$C1,$E7,$98,$78,$63,$D7,$C3,$A8,$C1,$E5,$7D,$FE,$2F,$82
	.dc.b	$3F,$CA,$07,$81,$C3,$FE,$53,$CB,$F9,$86,$7A,$8D,$36,$3F,$9B,$7D
	.dc.b	$45,$F3,$8D,$25,$7F,$96,$FE,$87,$FC,$B8,$9A,$47,$B7,$C6,$FC,$7C
	.dc.b	$3E,$24,$7F,$9A,$37,$FD,$CB,$FE,$50,$AF,$F2,$8F,$D4,$39,$7F,$CD
	.dc.b	$58,$71,$BF,$F3,$67,$35,$C3,$F8,$CB,$5F,$F2,$CC,$D7,$F1,$B3,$DB
	.dc.b	$9B,$06,$F9,$9F,$2F,$89,$F1,$FC,$FF,$13,$17,$F7,$23,$FC,$F9,$1D
	.dc.b	$7E,$2A,$47,$03,$D6,$F8,$F6,$FF,$7E,$D7,$E1,$39,$BC,$1E,$1E,$5D
	.dc.b	$73,$4C,$3F,$8D,$DF,$03,$FD,$53,$7F,$E7,$4F,$A8,$F9,$0F,$8C,$C6
	.dc.b	$DE,$BF,$07,$E1,$4E,$AB,$F2,$4F,$C2,$AE,$01,$C4,$6A,$B8,$F0,$72
	.dc.b	$F9,$E9,$8F,$F2,$EB,$FF,$45,$B6,$3B,$FB,$B2,$30,$6D,$F1,$FA,$72
	.dc.b	$39,$7C,$75,$CF,$C7,$83,$D5,$F4,$99,$D3,$D2,$FB,$3E,$36,$61,$E3
	.dc.b	$FA,$6B,$87,$B3,$E5,$3E,$DC,$FF,$44,$6C,$67,$DE,$FC,$34,$63,$1F
	.dc.b	$DB,$2F,$2B,$26,$BC,$7B,$87,$F4,$EF,$A8,$7C,$BF,$E3,$DB,$19,$F0
	.dc.b	$B1,$39,$ED,$93,$C7,$DD,$BA,$FD,$2F,$A8,$77,$7E,$65,$B5,$FD,$71
	.dc.b	$FF,$0F,$36,$35,$C1,$23,$5C,$56,$FE,$EF,$D4,$7F,$2A,$9F,$EC,$AD
	.dc.b	$8C,$F5,$FE,$9E,$31,$C5,$FC,$5E,$BF,$C6,$C7,$F6,$C2,$D7,$E2,$1F
	.dc.b	$F1,$C1,$9D,$BF,$61,$F5,$1F,$EC,$88,$FE,$E6,$D9,$E1,$EE,$E9,$DF
	.dc.b	$87,$2B,$7D,$B6,$7F,$2B,$7F,$83,$9E,$7A,$76,$EB,$EE,$FE,$FF,$C6
	.dc.b	$FF,$00,$4F,$F8,$09,$B7,$F9,$E7,$FD,$F5,$19,$F6,$78,$79,$92,$38
	.dc.b	$7A,$96,$FF,$4B,$F5,$1F,$B8,$27,$9F,$FC,$35,$C7,$C2,$D3,$1E,$ED
	.dc.b	$C3,$FC,$27,$2E,$1E,$07,$1F,$6E,$CF,$F8,$63,$C7,$FF,$41,$B7,$C6
	.dc.b	$FE,$35,$0F,$1E,$C7,$F1,$DF,$A8,$FF,$68,$57,$F8,$81,$BA,$FE,$F1
	.dc.b	$FE,$BF,$19,$F9,$6F,$E4,$9F,$50,$F8,$6A,$7F,$27,$FA,$7D,$E1,$FC
	.dc.b	$12,$75,$7F,$E2,$C5,$E1,$72,$38,$C6,$5E,$BE,$1C,$8C,$1C,$5F,$0F
	.dc.b	$0C,$8F,$F1,$BE,$1F,$E6,$F7,$B4,$79,$B8,$A4,$60,$F1,$3F,$CF,$7E
	.dc.b	$A3,$F1,$9D,$FF,$DB,$5B,$3F,$E5,$AB,$FB,$F9,$FF,$1E,$9C,$A0,$FF
	.dc.b	$25,$9F,$F3,$0B,$63,$4F,$07,$7C,$3E,$1F,$0E,$57,$AB,$A6,$D7,$7F
	.dc.b	$47,$CC,$63,$2F,$33,$E9,$F4,$E5,$FF,$96,$32,$4E,$2F,$D1,$F3,$BE
	.dc.b	$F3,$EB,$7C,$A5,$BE,$1C,$FF,$0C,$C5,$7B,$DF,$8D,$E0,$49,$F9,$FF
	.dc.b	$1B,$6B,$E2,$78,$3F,$2C,$C6,$BF,$2D,$F9,$DF,$1F,$2F,$EB,$FB,$D6
	.dc.b	$FE,$67,$C0,$7D,$33,$18,$7D,$37,$EB,$7D,$44,$BF,$74,$5A,$FE,$A3
	.dc.b	$F4,$BF,$98,$BF,$F1,$33,$6B,$FD,$27,$F5,$0F,$97,$FE,$F9,$39,$BF
	.dc.b	$B4,$FE,$25,$E7,$F1,$FA,$F0,$36,$F0,$F5,$E6,$3F,$E8,$6F,$A8,$7D
	.dc.b	$46,$30,$F1,$FC,$07,$63,$1A,$F9,$1F,$0F,$1C,$C3,$F3,$3E,$CD,$BF
	.dc.b	$5F,$17,$98,$C6,$1E,$C7,$D3,$F0,$12,$FF,$D2,$FB,$FF,$94,$9B,$1E
	.dc.b	$EF,$89,$91,$E7,$EA,$B8,$FC,$8F,$A8,$F8,$66,$32,$F9,$2F,$C6,$E3
	.dc.b	$97,$FE,$A4,$F5,$DE,$2E,$30,$FF,$00,$F2,$3D,$DF,$F1,$43,$6F,$CB
	.dc.b	$86,$27,$FC,$59,$F4,$CF,$89,$A7,$F8,$BF,$E5,$FF,$97,$EB,$FE,$5F
	.dc.b	$9D,$61,$FE,$B4,$6E,$5F,$E5,$46,$BF,$ED,$00,$E1,$FF,$69,$B5,$FD
	.dc.b	$C7,$D3,$BE,$6F,$FC,$93,$18,$6A,$8D,$57,$F9,$75,$B1,$FE,$64,$7C
	.dc.b	$77,$FE,$75,$6C,$FF,$B8,$9D,$C6,$71,$F8,$3F,$0F,$2B,$FD,$9F,$97
	.dc.b	$F8,$25,$F3,$C6,$70,$FF,$16,$E5,$FE,$40,$AD,$47,$F1,$18,$FF,$21
	.dc.b	$9F,$DB,$30,$F9,$1E,$EF,$CB,$E1,$F9,$EC,$C7,$E5,$39,$69,$2F,$BC
	.dc.b	$65,$B1,$9A,$EA,$FA,$2B,$A4,$E8,$93,$FE,$97,$C1,$3A,$8E,$33,$AA
	.dc.b	$FE,$54,$AB,$71,$DC,$FB,$37,$C5,$DC,$6A,$79,$1F,$0E,$74,$FF,$35
	.dc.b	$8D,$57,$80,$95,$D0,$F5,$1D,$E7,$93,$5F,$25,$F4,$7F,$1B,$76,$FA
	.dc.b	$D7,$39,$F1,$9A,$7A,$B8,$C3,$EB,$46,$F0,$F8,$D8,$75,$9C,$D7,$88
	.dc.b	$FF,$36,$DF,$67,$15,$EA,$6D,$FC,$4B,$E5,$87,$95,$B1,$EB,$BE,$55
	.dc.b	$E5,$3C,$95,$F7,$EF,$A4,$64,$FD,$66,$1F,$E3,$86,$CF,$E9,$A3,$11
	.dc.b	$F0,$4C,$DF,$FB,$0E,$33,$1E,$B3,$27,$E7,$3E,$07,$E0,$12,$7E,$84
	.dc.b	$FF,$98,$A7,$E4,$4E,$5B,$3A,$8C,$5F,$A2,$B8,$36,$33,$3F,$3C,$FE
	.dc.b	$06,$E2,$BE,$49,$FA,$B7,$0F,$C5,$9F,$E2,$51,$8A,$FA,$B3,$FC,$3A
	.dc.b	$31,$7F,$C7,$1B,$7F,$C5,$DB,$3E,$F8,$F4,$11,$F5,$92,$FD,$11,$96
	.dc.b	$C6,$63,$D4,$F9,$DD,$18,$7C,$8F,$F0,$CE,$2F,$E5,$87,$F2,$DF,$83
	.dc.b	$B1,$93,$FD,$F1,$92,$7F,$90,$B6,$33,$18,$F9,$F9,$1C,$1B,$7F,$CC
	.dc.b	$47,$D3,$BE,$9F,$FA,$24,$66,$31,$C7,$B8,$DF,$CF,$3D,$1C,$56,$A7
	.dc.b	$A6,$7F,$EA,$B7,$CF,$DD,$C5,$FE,$10,$D8,$D4,$70,$E7,$C1,$1F,$7B
	.dc.b	$C7,$E9,$9F,$4F,$FC,$4B,$85,$BF,$F1,$4F,$2B,$FC,$D1,$87,$F8,$FE
	.dc.b	$72,$FF,$90,$63,$2F,$FB,$8A,$F9,$BB,$AF,$E3,$B0,$ED,$ED,$18,$C2
	.dc.b	$D7,$F9,$2E,$B8,$BC,$EB,$2A,$1F,$19,$8F,$A1,$7B,$14,$DC,$EA,$BC
	.dc.b	$47,$5F,$3F,$37,$23,$27,$58,$E3,$F1,$9E,$0F,$A7,$FF,$25,$CE,$9A
	.dc.b	$FB,$E3,$BE,$BC,$23,$9E,$09,$CC,$63,$CF,$AF,$C7,$AF,$3C,$7B,$BE
	.dc.b	$9F,$E6,$CB,$82,$FD,$27,$E2,$E3,$CE,$D7,$5C,$3F,$D2,$AB,$8D,$72
	.dc.b	$5E,$87,$59,$95,$F0,$27,$2D,$F6,$7E,$7F,$D3,$BE,$6E,$9E,$58,$F3
	.dc.b	$7A,$76,$B8,$1C,$DF,$57,$7F,$D8,$FA,$77,$BF,$4C,$D7,$87,$D2,$AF
	.dc.b	$A8,$FF,$46,$66,$BF,$89,$3F,$17,$D2,$B5,$C4,$F5,$3F,$27,$E2,$BE
	.dc.b	$C7,$D3,$C9,$F7,$47,$CD,$F4,$B5,$18,$3E,$13,$F4,$7C,$1F,$C7,$F2
	.dc.b	$3D,$CA,$66,$BE,$47,$83,$5F,$40,$D7,$F6,$3C,$DF,$46,$7F,$33,$E2
	.dc.b	$FF,$8B,$EB,$F1,$FE,$03,$CE,$FC,$D1,$AF,$47,$7E,$85,$AF,$CA,$DF
	.dc.b	$17,$BB,$FE,$49,$73,$F1,$5C,$1D,$CF,$82,$3C,$3F,$35,$FA,$7F,$A7
	.dc.b	$95,$D2,$3C,$BF,$37,$A8,$C1,$E0,$7D,$3F,$0D,$FB,$2F,$09,$F1,$54
	.dc.b	$CD,$79,$BC,$9A,$F9,$AB,$E0,$C7,$C2,$F6,$7F,$77,$E6,$BB,$29,$9B
	.dc.b	$F7,$F7,$57,$DF,$5F,$2A,$3E,$5F,$D0,$59,$D9,$F1,$9F,$E0,$FF,$4E
	.dc.b	$F0,$AF,$87,$2B,$F7,$F3,$96,$FA,$BC,$3D,$59,$18,$3F,$82,$FD,$3B
	.dc.b	$D8,$FE,$26,$B8,$E9,$88,$CF,$0E,$BE,$32,$5F,$28,$FE,$37,$F4,$EF
	.dc.b	$96,$BF,$2D,$2B,$F8,$9C,$E5,$BF,$11,$C3,$D5,$91,$83,$F8,$CF,$80
	.dc.b	$F8,$1F,$E4,$EF,$EF,$23,$F2,$AD,$F1,$3F,$86,$BB,$8E,$3E,$57,$0F
	.dc.b	$94,$91,$83,$F9,$2F,$D3,$FE,$5D,$BE,$17,$F3,$48,$36,$F1,$C5,$9F
	.dc.b	$06,$5F,$28,$E1,$F2,$9C,$7E,$52,$4F,$68,$FE,$7F,$F4,$FF,$8E,$8F
	.dc.b	$E6,$AF,$F4,$ED,$7F,$92,$B7,$D6,$FF,$34,$BB,$9F,$D9,$FD,$3B,$6F
	.dc.b	$E9,$2B,$E2,$29,$88,$FE,$9B,$DD,$EA,$AF,$E8,$6F,$EB,$5F,$DE,$B7
	.dc.b	$1F,$E6,$AE,$E3,$3E,$17,$2F,$A7,$91,$CA,$3A,$EB,$DB,$C2,$93,$DA
	.dc.b	$3F,$AE,$FD,$3F,$AE,$01,$FE,$D0,$FF,$56,$6B,$FC,$41,$E0,$BE,$9F
	.dc.b	$FA,$0E,$FF,$E8,$C9,$FE,$75,$3C,$7D,$AF,$ED,$75,$A8,$EE,$FC,$57
	.dc.b	$FB,$1A,$F6,$36,$70,$DF,$8F,$47,$2E,$09,$7C,$23,$AF,$AF,$A8,$C1
	.dc.b	$D1,$F0,$F8,$BE,$2F,$16,$4F,$82,$3C,$BE,$BE,$A3,$07,$91,$FF,$01
	.dc.b	$FD,$33,$E8,$FF,$C0,$BF,$4E,$F8,$0A,$6A,$37,$DB,$70,$FE,$FF,$7C
	.dc.b	$5B,$1F,$E0,$EF,$98,$78,$DF,$E1,$0F,$45,$A5,$35,$1B,$F2,$2E,$1F
	.dc.b	$E0,$AB,$E2,$D8,$FF,$0C,$FD,$3F,$E2,$7A,$3F,$D7,$F8,$FC,$DA,$FF
	.dc.b	$08,$B5,$FE,$97,$3B,$7F,$06,$FE,$3F,$BB,$FC,$0A,$E6,$0D,$35,$5F
	.dc.b	$EC,$C7,$F7,$11,$D3,$C4,$F1,$9E,$4F,$13,$CB,$BF,$BF,$96,$57,$F8
	.dc.b	$92,$FE,$33,$E4,$DF,$6E,$6B,$DB,$1E,$EF,$13,$FC,$31,$3F,$E6,$3A
	.dc.b	$FE,$4D,$9F,$F9,$7D,$B1,$99,$DB,$FE,$E2,$3D,$7E,$63,$E9,$F8,$36
	.dc.b	$3A,$35,$E1,$5F,$CD,$67,$FC,$C9,$0F,$A0,$A4,$7F,$CD,$0F,$F9,$A7
	.dc.b	$E3,$3F,$CB,$78,$FF,$8C,$27,$FC,$E5,$BF,$F9,$11,$FD,$9D,$7E,$D3
	.dc.b	$E9,$DC,$7E,$7C,$38,$DF,$A7,$92,$E1,$FE,$FC,$6B,$FB,$C9,$AF,$E3
	.dc.b	$A7,$B7,$26,$0D,$F2,$3E,$57,$13,$E3,$F9,$DE,$26,$2F,$FC,$22,$1E
	.dc.b	$BF,$39,$27,$81,$EB,$7C,$7B,$7F,$84,$63,$FC,$C2,$DA,$FF,$0D,$E1
	.dc.b	$FC,$9E,$73,$CD,$C3,$96,$46,$0E,$2F,$BB,$E9,$FE,$4D,$FF,$89,$1B
	.dc.b	$87,$F9,$73,$0F,$F5,$64,$E7,$5E,$1B,$4C,$7F,$CE,$78,$70,$7F,$91
	.dc.b	$D2,$3E,$43,$D7,$E7,$7C,$07,$87,$E4,$2D,$72,$F7,$7C,$86,$2B,$C8
	.dc.b	$F9,$FD,$39,$3E,$0F,$90,$B5,$E9,$F5,$7D,$86,$36,$FF,$3A,$E6,$8F
	.dc.b	$81,$F1,$D3,$0F,$4F,$D2,$DC,$3F,$C8,$BF,$29,$F6,$E7,$FD,$1E,$D8
	.dc.b	$CF,$BD,$F8,$68,$C6,$5F,$E8,$25,$E3,$E4,$D7,$89,$70,$FF,$4C,$FD
	.dc.b	$3F,$FC,$00,$7E,$3B,$71,$FE,$96,$FF,$BE,$0C,$27,$FD,$53,$1C,$B9
	.dc.b	$7F,$2F,$0F,$72,$C7,$E6,$BE,$9F,$F1,$D5,$FE,$A8,$6D,$FF,$98,$9F
	.dc.b	$F0,$E4,$67,$AF,$86,$D2,$38,$70,$5B,$FF,$34,$FD,$3F,$F2,$E9,$FF
	.dc.b	$59,$B6,$33,$E6,$7E,$AA,$31,$C1,$FE,$A6,$AF,$F1,$A1,$FE,$BA,$2D
	.dc.b	$7F,$9D,$1F,$F1,$B1,$9F,$17,$F6,$7F,$4F,$FD,$BA,$3F,$D8,$2D,$9E
	.dc.b	$1E,$E6,$9D,$F8,$52,$B7,$DB,$67,$FD,$03,$7F,$83,$9E,$7A,$76,$EA
	.dc.b	$EE,$FF,$02,$F1,$5E,$D5,$31,$F3,$36,$FF,$DA,$72,$F3,$38,$1C,$7D
	.dc.b	$2B,$3F,$ED,$8F,$A7,$FD,$C7,$0F,$FB,$21,$B1,$FA,$87,$FC,$6A,$EF
	.dc.b	$C6,$FE,$27,$1F,$EB,$48,$E5,$F0,$B8,$F7,$32,$EB,$8E,$4F,$7F,$6E
	.dc.b	$F1,$7F,$A8,$3C,$8F,$F4,$1C,$FF,$BA,$9B,$19,$F2,$3F,$90,$C6,$34
	.dc.b	$FF,$6E,$FD,$3F,$FB,$52,$FF,$DE,$2D,$BF,$F5,$5A,$8F,$E5,$2E,$67
	.dc.b	$FD,$CB,$F4,$EF,$85,$FE,$5A,$DA,$FF,$7D,$FD,$3F,$17,$FD,$FB,$19
	.dc.b	$C3,$FD,$6E,$FE,$76,$33,$E1,$72,$BE,$6F,$4F,$0A,$4E,$8E,$6F,$83
	.dc.b	$82,$47,$83,$E1,$6E,$3D,$B8,$7C,$A6,$34,$F4,$BF,$D1,$3E,$9F,$F2
	.dc.b	$11,$FF,$00,$1B,$7F,$C8,$35,$74,$6F,$0E,$1C,$12,$BF,$C8,$F8,$71
	.dc.b	$9C,$1F,$ED,$4C,$3F,$6B,$1D,$7C,$3F,$EA,$78,$FE,$4A,$F8,$9F,$CB
	.dc.b	$AB,$6D,$F4,$3C,$C7,$9E,$AC,$70,$C3,$FC,$AF,$5F,$D9,$23,$10,$8C
	.dc.b	$3E,$3A,$F8,$1E,$97,$C1,$F4,$25,$FD,$8F,$94,$B7,$E9,$E6,$F8,$C6
	.dc.b	$34,$F8,$CF,$C9,$F0,$25,$FE,$03,$E2,$ED,$FC,$1F,$07,$E7,$18,$C3
	.dc.b	$E7,$3F,$47,$E3,$65,$FE,$23,$E0,$6D,$FC,$9F,$7B,$EB,$18,$C3,$EB
	.dc.b	$3F,$67,$E9,$E5,$FC,$0E,$1F,$F2,$93,$63,$FC,$03,$F4,$9F,$99,$8F
	.dc.b	$EB,$6D,$C3,$FC,$FF,$F4,$EF,$96,$DF,$CE,$F0,$EB,$98,$7C,$CC,$38
	.dc.b	$60,$7F,$D8,$11,$9F,$5B,$B9,$FE,$10,$FA,$77,$D3,$E2,$79,$7B,$DE
	.dc.b	$43,$18,$38,$3F,$60,$FF,$A4,$B1,$FE,$A7,$5E,$8E,$4F,$59,$F8,$AF
	.dc.b	$62,$C7,$97,$8F,$DE,$62,$FA,$47,$C0,$8F,$87,$F1,$12,$38,$1E,$D7
	.dc.b	$E3,$FC,$44,$8F,$C5,$EA,$EB,$C6,$1D,$0F,$37,$D3,$7C,$E3,$3A,$FA
	.dc.b	$A3,$E3,$47,$D2,$E3,$98,$7E,$5B,$D7,$78,$B8,$BE,$87,$FB,$AF,$DC
	.dc.b	$FF,$1C,$F2,$9D,$65,$D6,$31,$87,$E8,$3E,$99,$F3,$1F,$E4,$BC,$3F
	.dc.b	$D7,$71,$89,$FF,$2F,$46,$27,$4A,$74,$E9,$B7,$EB,$F9,$7F,$F5,$A5
	.dc.b	$F7,$BF,$F2,$6C,$62,$FD,$3F,$4F,$C2,$BF,$C5,$2D,$8F,$F6,$6A,$F9
	.dc.b	$BF,$F2,$AB,$98,$3E,$27,$FC,$E8,$D7,$9D,$FA,$23,$11,$EE,$F5,$3D
	.dc.b	$79,$7F,$92,$E3,$FC,$F8,$D7,$FC,$76,$D7,$86,$1F,$27,$E1,$BF,$CC
	.dc.b	$F3,$FD,$6A,$7E,$4C,$63,$E9,$E5,$7C,$FF,$7A,$D7,$D8,$BF,$F0,$E9
	.dc.b	$23,$3A,$FE,$0E,$33,$7D,$A7,$37,$E4,$8B,$7F,$62,$6B,$F5,$62,$D7
	.dc.b	$F8,$3A,$FA,$46,$0E,$97,$F8,$37,$0F,$9C,$C1,$2B,$D1,$1C,$1B,$19
	.dc.b	$9E,$AF,$18,$C9,$23,$2F,$C1,$BE,$EE,$67,$E7,$C6,$27,$67,$C3,$71
	.dc.b	$1F,$04,$ED,$1A,$71,$3C,$CF,$88,$F9,$4C,$C6,$E3,$26,$CF,$91,$1B
	.dc.b	$73,$1B,$72,$6E,$FE,$64,$F4,$24,$7E,$AA,$FE,$35,$71,$A4,$7E,$40
	.dc.b	$95,$FA,$EA,$EA,$7E,$4A,$32,$FE,$B5,$AE,$63,$FA,$CE,$AF,$63,$F0
	.dc.b	$2D,$8F,$54,$A3,$0F,$BD,$B8,$7B,$5B,$3E,$A9,$E1,$5D,$09,$1D,$27
	.dc.b	$73,$EF,$8F,$1C,$EA,$3B,$36,$BB,$36,$7B,$57,$2A,$F9,$B5,$FC,$26
	.dc.b	$B5,$1D,$4F,$E3,$F3,$BC,$3F,$D0,$AD,$C3,$F8,$6D,$70,$AE,$78,$7F
	.dc.b	$AD,$2B,$4F,$98,$F5,$4E,$EF,$9B,$6F,$9C,$E8,$97,$EC,$6E,$1E,$C6
	.dc.b	$9E,$CA,$FE,$0B,$E9,$C8,$FA,$FC,$76,$FF,$81,$4F,$F1,$22,$4F,$E0
	.dc.b	$F1,$FF,$21,$4F,$FB,$30,$7D,$3F,$F8,$BB,$8F,$F1,$0D,$8F,$E6,$EB
	.dc.b	$E7,$5E,$8E,$BD,$2F,$E8,$95,$FC,$F3,$77,$67,$E4,$63,$F0,$CD,$8C
	.dc.b	$C7,$E5,$5A,$F1,$E9,$D1,$1D,$3C,$FF,$19,$7A,$B4,$93,$F9,$F8,$CB
	.dc.b	$FA,$1F,$A6,$7D,$47,$FB,$AB,$93,$FB,$27,$9C,$F4,$71,$C7,$FE,$EE
	.dc.b	$8C,$61,$FE,$3A,$C3,$8B,$7C,$1D,$A1,$1A,$7F,$B4,$B7,$0F,$EE,$4F
	.dc.b	$F7,$98,$E9,$F1,$7C,$97,$D0,$FA,$1F,$EE,$2D,$AE,$33,$BA,$7A,$76
	.dc.b	$25,$F9,$E3,$5F,$17,$D7,$EA,$1E,$C5,$33,$5E,$DD,$2B,$E8,$AE,$D1
	.dc.b	$DF,$D5,$B3,$B3,$E2,$3E,$9F,$A3,$FF,$07,$69,$FE,$0F,$FA,$4F,$F3
	.dc.b	$BD,$7F,$98,$9A,$FF,$83,$27,$7C,$1F,$26,$10,$F5,$3A,$4C,$9F,$F3
	.dc.b	$5C,$E5,$BF,$0D,$F2,$7E,$A1,$F3,$74,$F2,$47,$97,$D7,$B5,$C0,$E6
	.dc.b	$FA,$7E,$9F,$E9,$7D,$43,$E0,$A9,$9A,$F0,$FA,$F5,$F5,$57,$C1,$8F
	.dc.b	$85,$EB,$59,$D9,$E8,$7E,$37,$C4,$FD,$3F,$A8,$93,$EC,$8F,$95,$E1
	.dc.b	$EA,$30,$7C,$07,$E7,$7B,$3F,$87,$E3,$7C,$0D,$33,$5F,$1B,$C4,$AF
	.dc.b	$CF,$AF,$AD,$1F,$8F,$F2,$3F,$C4,$13,$FE,$96,$F0,$1E,$97,$E4,$DB
	.dc.b	$1F,$E9,$A9,$DB,$FE,$44,$7F,$C7,$AE,$7D,$5E,$87,$5F,$57,$B7,$CD
	.dc.b	$D9,$E5,$18,$7A,$7C,$DF,$7F,$A8,$91,$83,$C1,$F3,$9E,$79,$91,$83
	.dc.b	$C9,$F1,$3F,$A9,$F0,$9F,$33,$4F,$74,$7B,$FC,$EB,$5C,$0F,$63,$F0
	.dc.b	$FD,$DF,$D9,$7A,$0E,$AA,$66,$BE,$17,$65,$7E,$15,$7C,$B8,$F9,$9F
	.dc.b	$43,$67,$67,$C9,$7F,$7D,$F5,$0F,$02,$F8,$32,$BF,$D8,$F3,$96,$FC
	.dc.b	$D7,$0F,$6A,$46,$0F,$F6,$6F,$D4,$3D,$EF,$E1,$4B,$92,$98,$8C,$F0
	.dc.b	$EB,$DC,$97,$CA,$3F,$89,$7D,$43,$E6,$2F,$CC,$4A,$FF,$6D,$CE,$5B
	.dc.b	$F1,$DC,$3D,$59,$18,$3F,$DC,$3E,$0B,$E1,$7F,$8F,$3F,$B9,$8F,$F4
	.dc.b	$6B,$7D,$2F,$F4,$B3,$B8,$E3,$E4,$F0,$F8,$29,$18,$3F,$DD,$BF,$51
	.dc.b	$F9,$67,$FD,$2E,$BC,$3F,$E5,$8D,$8D,$DE,$38,$B3,$E0,$C9,$E7,$1C
	.dc.b	$3E,$5B,$8F,$CB,$49,$EF,$1F,$CD,$FE,$A3,$F1,$F1,$FE,$FE,$7F,$A3
	.dc.b	$BF,$EA,$86,$EB,$FE,$B1,$77,$1F,$EA,$CF,$A8,$FF,$16,$3F,$D2,$67
	.dc.b	$FD,$6D,$0F,$8D,$FE,$88,$D9,$D7,$B1,$FC,$9E,$38,$C7,$FA,$DB,$BB
	.dc.b	$DC,$DF,$27,$FA,$64,$7E,$DA,$35,$C1,$8F,$0F,$3F,$51,$27,$A9,$FD
	.dc.b	$6E,$BF,$D8,$6E,$02,$DB,$AF,$87,$27,$53,$97,$87,$D4,$7C,$2D,$3F
	.dc.b	$9F,$3F,$D8,$63,$FD,$9A,$DD,$BF,$DA,$CE,$E3,$FB,$67,$D4,$7F,$46
	.dc.b	$7F,$B7,$47,$F3,$E7,$AF,$B3,$FD,$A6,$75,$07,$E4,$3F,$ED,$A5,$EB
	.dc.b	$EC,$E1,$C3,$C7,$A7,$97,$4C,$BF,$81,$69,$ED,$EB,$6E,$73,$18,$77
	.dc.b	$E3,$78,$FC,$69,$18,$39,$FA,$9E,$79,$91,$83,$E7,$7D,$3B,$E3,$BF
	.dc.b	$C0,$9F,$50,$F8,$0A,$7F,$81,$E7,$55,$C3,$BF,$17,$23,$07,$D9,$F9
	.dc.b	$97,$8F,$FE,$0F,$F4,$99,$D3,$FC,$25,$3A,$AE,$1E,$4E,$2E,$46,$0F
	.dc.b	$C2,$7D,$47,$E3,$7A,$1D,$3F,$DE,$F2,$F9,$F5,$F8,$96,$A9,$DB,$FE
	.dc.b	$29,$AE,$55,$FE,$03,$7F,$37,$3A,$94,$69,$E1,$E0,$7F,$8A,$63,$75
	.dc.b	$C7,$C5,$C7,$83,$2F,$CF,$1B,$DB,$E1,$72,$C9,$F5,$C7,$3E,$0D,$46
	.dc.b	$0F,$7B,$DF,$C5,$F5,$47,$F9,$A4,$F2,$B8,$7F,$CD,$99,$7F,$33,$D3
	.dc.b	$FD,$64,$D8,$E0,$FF,$39,$FA,$8C,$F9,$C6,$92,$7F,$CD,$5F,$43,$FE
	.dc.b	$6B,$4D,$23,$FE,$BF,$F8,$FF,$9B,$87,$C9,$8F,$F3,$BE,$FF,$BB,$7F
	.dc.b	$CB,$15,$FE,$59,$FA,$87,$2F,$FA,$0F,$0E,$37,$FE,$85,$E4,$B8,$7F
	.dc.b	$8D,$DA,$FF,$98,$66,$BF,$8E,$9E,$DC,$B8,$37,$CA,$F9,$9C,$4F,$8F
	.dc.b	$E7,$B8,$98,$BF,$C3,$47,$FA,$56,$3A,$FC,$54,$8E,$07,$AD,$F1,$ED
	.dc.b	$FE,$1A,$8F,$E3,$0D,$AE,$67,$F9,$DF,$0F,$E5,$D3,$AE,$6E,$3C,$B2
	.dc.b	$30,$72,$7D,$FF,$51,$F2,$AF,$98,$71,$87,$E5,$F0,$77,$F0,$DB,$8E
	.dc.b	$F3,$2F,$E5,$98,$70,$7F,$CC,$4F,$C9,$C7,$AB,$FE,$95,$8D,$FC,$87
	.dc.b	$8B,$E4,$3C,$6F,$93,$DE,$93,$A3,$AB,$E8,$F5,$24,$78,$3E,$3E,$E3
	.dc.b	$DB,$D6,$F6,$98,$DF,$F6,$19,$A3,$E0,$FC,$74,$C3,$D3,$F4,$D7,$0F
	.dc.b	$87,$F2,$DF,$76,$7F,$A4,$36,$33,$EF,$FE,$22,$31,$97,$EF,$57,$93
	.dc.b	$93,$5E,$35,$C3,$FA,$97,$D4,$3E,$67,$F2,$2D,$8C,$F8,$58,$9C,$F4
	.dc.b	$49,$E3,$EF,$5D,$7E,$CF,$D4,$3B,$BF,$36,$DA,$FE,$C0,$FF,$88,$DB
	.dc.b	$1A,$E0,$91,$AE,$2B,$7F,$84,$FA,$8F,$E5,$B3,$FD,$A5,$B1,$9E,$CF
	.dc.b	$D4,$C6,$38,$BF,$8F,$57,$F8,$EC,$FE,$E0,$5A,$FC,$7B,$FE,$3D,$33
	.dc.b	$B7,$EC,$BE,$A3,$FD,$A5,$1F,$DD,$DB,$3C,$3D,$ED,$3B,$F0,$E5,$6F
	.dc.b	$BA,$CF,$E7,$2F,$F0,$13,$CF,$4E,$DD,$7D,$DF,$E0,$3E,$37,$F8,$12
	.dc.b	$7F,$C0,$CD,$BF,$D2,$AF,$9A,$DF,$B3,$C3,$CD,$97,$C3,$D5,$B8,$7E
	.dc.b	$B3,$EA,$3F,$71,$C3,$FD,$05,$B5,$FE,$C7,$7F,$C7,$CF,$F8,$55,$CC
	.dc.b	$FF,$BC,$DF,$F5,$D4,$67,$E1,$FF,$8B,$C3,$DF,$B1,$8E,$29,$5C,$FD
	.dc.b	$EB,$87,$EF,$BC,$8F,$F4,$6C,$FF,$88,$DB,$19,$F1,$FF,$91,$C6,$33
	.dc.b	$FE,$B1,$F5,$1F,$ED,$BB,$FF,$15,$36,$FF,$84,$BF,$DA,$23,$3F,$31
	.dc.b	$FC,$B3,$EA,$1F,$11,$4F,$E5,$FF,$51,$BC,$3F,$88,$CE,$AF,$FC,$6C
	.dc.b	$BC,$1E,$47,$18,$CB,$D7,$C2,$91,$83,$8B,$E1,$E1,$91,$FE,$41,$C3
	.dc.b	$FC,$F4,$F6,$8F,$37,$0C,$8C,$1E,$27,$FA,$2F,$D4,$7E,$4B,$6F,$EF
	.dc.b	$4D,$9F,$F8,$01,$7F,$85,$7F,$C9,$D1,$9C,$C1,$FE,$4C,$E1,$AE,$4E
	.dc.b	$FB,$F8,$3C,$29,$5E,$9E,$7B,$5F,$CE,$2D,$E8,$F9,$8C,$E5,$E6,$7D
	.dc.b	$3E,$9C,$C3,$CB,$16,$F9,$7A,$1E,$E3,$EA,$FC,$AD,$AF,$5F,$37,$C2
	.dc.b	$31,$5E,$17,$E2,$F7,$C9,$F9,$DF,$1D,$6B,$CD,$E0,$7C,$A3,$1A,$7C
	.dc.b	$A7,$E6,$FC,$84,$BF,$AD,$F0,$96,$FD,$DF,$01,$F4,$8C,$61,$F4,$9F
	.dc.b	$AB,$F5,$12,$FE,$8C,$5A,$FE,$BB,$F4,$DF,$97,$BF,$F1,$93,$6B,$EA
	.dc.b	$CB,$EA,$1F,$31,$FE,$04,$AD,$4F,$DA,$7F,$12,$F4,$F9,$0D,$78,$1B
	.dc.b	$78,$7A,$D3,$1F,$F4,$87,$D3,$BE,$A3,$18,$79,$3D,$FE,$C6,$35,$F2
	.dc.b	$BE,$1E,$39,$87,$E6,$7D,$8B,$7E,$CE,$2F,$31,$8C,$3D,$AF,$A7,$E2
	.dc.b	$E5,$FF,$A8,$77,$FF,$44,$B6,$3D,$DF,$15,$23,$CF,$D5,$71,$F9,$3F
	.dc.b	$51,$F0,$CC,$65,$F2,$9F,$8D,$C7,$2F,$FD,$5D,$EB,$7F,$15,$7C,$11
	.dc.b	$FE,$2E,$E3,$7B,$94,$E1,$8F,$F8,$4A,$33,$5F,$EB,$7F,$A7,$FE,$4B
	.dc.b	$3F,$E5,$56,$BF,$60,$38,$7E,$DB,$5F,$D8,$BC,$B2,$FF,$39,$7C,$C3
	.dc.b	$9B,$FC,$83,$39,$BF,$C1,$CE,$5F,$17,$C2,$68,$FF,$23,$C7,$F8,$02
	.dc.b	$3B,$C6,$5F,$F3,$56,$1E,$48,$FF,$2C,$3E,$26,$B9,$8D,$D3,$E4,$B2
	.dc.b	$D8,$CC,$60,$8F,$FB,$6F,$EA,$3D,$15,$EE,$9F,$F2,$DC,$62,$3E,$2F
	.dc.b	$97,$EE,$3F,$CB,$FF,$B3,$7C,$E8,$FF,$20,$E1,$F5,$DB,$3B,$C3,$FD
	.dc.b	$D1,$5B,$D3,$FD,$35,$7B,$6D,$6E,$FF,$C9,$75,$BB,$FA,$73,$9A,$DC
	.dc.b	$74,$DB,$0E,$8F,$C0,$4F,$9A,$33,$7F,$6A,$2A,$BE,$CE,$6B,$89,$1F
	.dc.b	$1D,$70,$C3,$D1,$3D,$E7,$2E,$F0,$F2,$C7,$C2,$8F,$31,$93,$84,$68
	.dc.b	$FF,$5E,$5F,$8A,$7E,$7C,$ED,$F2,$1D,$8E,$84,$9E,$51,$FC,$4A,$79
	.dc.b	$57,$9A,$B8,$D6,$9F,$E9,$73,$AC,$3D,$87,$C7,$BF,$04,$F8,$8E,$CF
	.dc.b	$5C,$36,$EA,$B9,$3C,$30,$F0,$57,$CD,$AD,$BC,$8F,$0B,$87,$F6,$51
	.dc.b	$F8,$96,$DF,$E6,$89,$5F,$9E,$6C,$7E,$ED,$B8,$7E,$33,$CC,$FA,$C4
	.dc.b	$AF,$23,$6B,$C4,$7F,$00,$71,$3F,$16,$31,$3C,$08,$F0,$BF,$F1,$5B
	.dc.b	$6B,$8C,$FD,$C3,$F8,$4B,$88,$FA,$31,$83,$9D,$77,$AE,$6D,$AE,$15
	.dc.b	$F4,$63,$D9,$5C,$0E,$D1,$FC,$52,$77,$5F,$40,$E7,$3A,$9F,$59,$E7
	.dc.b	$79,$3E,$43,$93,$87,$C0,$F9,$DE,$04,$AF,$34,$6E,$7C,$15,$FC,$13
	.dc.b	$EA,$BF,$8A,$E4,$69,$2B,$F8,$EF,$D5,$7F,$98,$F3,$BE,$CF,$F3,$CC
	.dc.b	$FF,$9A,$09,$7C,$FB,$CF,$DD,$FF,$26,$FA,$A7,$CC,$7F,$28,$C1,$18
	.dc.b	$75,$19,$8C,$47,$6D,$1F,$41,$C5,$B1,$FC,$C9,$AE,$E3,$5F,$29,$23
	.dc.b	$4D,$7D,$34,$F6,$BC,$04,$8F,$43,$E7,$FA,$97,$D5,$7F,$82,$B9,$1C
	.dc.b	$3F,$E0,$CF,$35,$E8,$66,$30,$DC,$7F,$C8,$B8,$72,$7F,$C8,$F7,$C9
	.dc.b	$CF,$83,$B5,$3B,$7F,$DF,$2D,$9E,$11,$A9,$E1,$2F,$8E,$E5,$F4,$6F
	.dc.b	$6A,$9A,$8D,$FA,$77,$0F,$1E,$7F,$E2,$6C,$39,$46,$0F,$4C,$F6,$F8
	.dc.b	$EF,$37,$D5,$7F,$95,$27,$41,$C3,$E8,$8F,$2E,$DC,$63,$76,$9C,$C6
	.dc.b	$3D,$3A,$FC,$0A,$F6,$C7,$C5,$FA,$AF,$83,$BF,$C2,$FA,$6F,$F5,$C5
	.dc.b	$F8,$9A,$A3,$E3,$CF,$0D,$3F,$C8,$6B,$D4,$E9,$12,$BD,$73,$96,$FB
	.dc.b	$9F,$57,$EA,$9F,$3F,$4F,$24,$79,$7C,$AB,$5C,$0E,$6F,$A7,$EA,$7E
	.dc.b	$F7,$D5,$3C,$6A,$66,$BC,$3E,$B5,$7D,$5A,$7A,$2F,$E1,$4F,$C3,$F5
	.dc.b	$6D,$70,$3D,$2F,$C7,$F8,$AF,$AF,$F5,$52,$7D,$B1,$F2,$FD,$5D,$46
	.dc.b	$0F,$82,$FC,$FF,$63,$F2,$9C,$8F,$7E,$99,$AF,$8F,$EA,$D7,$CE,$7F
	.dc.b	$A9,$39,$17,$F6,$67,$F3,$DE,$5F,$EE,$39,$FF,$37,$E0,$3D,$3F,$CC
	.dc.b	$9A,$F2,$37,$E4,$5C,$FF,$1F,$87,$1E,$4F,$F3,$EA,$9E,$BF,$3F,$C2
	.dc.b	$B7,$18,$78,$FC,$FF,$E9,$BE,$AA,$46,$0E,$BF,$41,$E6,$99,$18,$3C
	.dc.b	$3F,$13,$FB,$7F,$09,$F3,$94,$F6,$C7,$BB,$C9,$B5,$C0,$F4,$BF,$0B
	.dc.b	$D7,$FE,$01,$ED,$3B,$69,$9A,$F7,$F7,$D7,$E6,$9F,$CC,$2B,$E5,$C7
	.dc.b	$CC,$F4,$AC,$F0,$3E,$3B,$FC,$37,$EA,$BF,$CC,$91,$FB,$85,$E2,$64
	.dc.b	$D7,$89,$BF,$9C,$E1,$EF,$4B,$E1,$1F,$C5,$FE,$A9,$F3,$17,$92,$57
	.dc.b	$F0,$49,$CB,$78,$78,$7C,$84,$8C,$1F,$C2,$BE,$A9,$F3,$3F,$C8,$D7
	.dc.b	$CC,$D3,$11,$9F,$8D,$D7,$AB,$2F,$94,$7F,$2B,$F0,$5F,$0B,$7D,$29
	.dc.b	$5F,$C6,$27,$2D,$F3,$38,$7C,$BC,$8C,$1F,$C7,$BE,$AB,$F2,$CD,$F0
	.dc.b	$FF,$9C,$41,$B7,$8E,$2C,$F8,$32,$F9,$47,$0F,$97,$E3,$EF,$49,$ED
	.dc.b	$1F,$D0,$FE,$AB,$F1,$B1,$FC,$B9,$FE,$9E,$DF,$7B,$5E,$27,$F4,$9F
	.dc.b	$AA,$6D,$7E,$32,$5F,$F3,$69,$CB,$FC,$E3,$BF,$D0,$DF,$27,$FA,$94
	.dc.b	$7E,$AE,$35,$C1,$8F,$13,$3F,$17,$27,$94,$70,$DF,$8F,$89,$27,$B4
	.dc.b	$7F,$5D,$FA,$AF,$4C,$7E,$69,$7C,$16,$4D,$70,$7F,$63,$FA,$AF,$E7
	.dc.b	$14,$15,$FC,$C9,$FE,$A2,$D7,$F4,$B1,$C1,$2D,$FA,$7B,$52,$7F,$AC
	.dc.b	$CE,$5B,$D3,$E1,$E2,$91,$83,$7E,$CF,$96,$64,$60,$E7,$E3,$7A,$7C
	.dc.b	$79,$18,$3B,$FA,$DE,$D9,$91,$83,$FB,$5F,$D4,$BE,$9B,$FC,$09,$F5
	.dc.b	$4F,$82,$A7,$F8,$1E,$75,$5C,$3B,$F1,$72,$30,$7F,$76,$F9,$9F,$D5
	.dc.b	$BF,$DE,$3D,$36,$5F,$E1,$16,$C6,$FC,$9F,$F0,$A5,$7E,$D2,$30,$43
	.dc.b	$F8,$9B,$97,$F8,$AF,$D5,$7E,$25,$FE,$2F,$4E,$3E,$C7,$F8,$7E,$75
	.dc.b	$ED,$FF,$87,$1F,$C9,$D7,$2E,$FD,$1B,$EE,$E1,$E2,$D9,$FF,$08,$D7
	.dc.b	$0B,$E5,$3D,$75,$F6,$72,$4B,$DC,$78,$77,$D4,$60,$E6,$F9,$7C,$5F
	.dc.b	$F0,$54,$7F,$98,$31,$46,$23,$F2,$87,$E5,$63,$38,$FF,$99,$A1,$18
	.dc.b	$79,$7A,$0F,$AA,$D4,$FF,$2A,$6C,$66,$77,$1F,$E6,$95,$F4,$54,$8F
	.dc.b	$F9,$B5,$FF,$37,$7C,$7F,$F9,$37,$0F,$F2,$3C,$FF,$9D,$7D,$CF,$F1
	.dc.b	$BB,$FA,$BA,$FD,$67,$D5,$39,$3D,$73,$B3,$FA,$79,$FC,$CA,$B8,$F2
	.dc.b	$F5,$77,$8D,$9F,$87,$35,$FD,$C7,$A3,$5F,$E7,$B5,$F9,$1F,$98,$C4
	.dc.b	$E7,$E7,$E4,$F9,$A2,$D7,$8B,$E7,$B9,$18,$D7,$DA,$F7,$7F,$C1,$55
	.dc.b	$FE,$54,$6C,$FE,$4B,$C3,$D7,$36,$F9,$65,$EE,$2D,$7F,$6E,$FA,$AF
	.dc.b	$85,$87,$F9,$C1,$B5,$F8,$0E,$B7,$83,$EF,$9C,$DF,$F1,$77,$E4,$3C
	.dc.b	$F7,$D7,$06,$DE,$1E,$34,$C7,$FC,$D7,$7F,$C9,$1B,$1D,$7C,$E9,$18
	.dc.b	$36,$F8,$7D,$59,$1C,$BE,$42,$E5,$EB,$F5,$BC,$C6,$36,$F6,$3E,$9F
	.dc.b	$8E,$97,$E2,$FA,$7B,$7F,$E1,$0F,$95,$74,$7D,$D6,$D7,$FA,$09,$7B
	.dc.b	$D9,$9D,$79,$52,$B5,$E3,$DC,$3F,$C3,$7F,$54,$FA,$AF,$C4,$36,$BF
	.dc.b	$D1,$AF,$F8,$39,$B1,$AE,$79,$1A,$F7,$ED,$FF,$89,$BE,$A9,$DF,$F9
	.dc.b	$36,$D7,$FA,$61,$FC,$9C,$EE,$75,$BC,$AD,$70,$DA,$FF,$18,$FD,$57
	.dc.b	$F2,$89,$FF,$50,$B6,$33,$D7,$FA,$18,$C7,$27,$FA,$E2,$BD,$55,$FE
	.dc.b	$3A,$6C,$67,$C7,$FD,$44,$60,$FE,$1B,$F5,$5F,$D8,$5F,$F5,$7B,$6B
	.dc.b	$97,$BB,$FA,$F8,$C5,$FF,$B1,$9F,$6B,$7C,$3D,$77,$6D,$E1,$A4,$C7
	.dc.b	$AF,$5D,$CB,$FC,$9F,$E3,$7F,$91,$A7,$FD,$7E,$D8,$CF,$CE,$7E,$FE
	.dc.b	$31,$97,$FB,$59,$79,$99,$35,$EA,$5C,$3F,$D9,$3F,$55,$FB,$6B,$FF
	.dc.b	$6E,$B6,$3F,$96,$BF,$E2,$D7,$7E,$2F,$F0,$D8,$FF,$55,$47,$2F,$87
	.dc.b	$C7,$B9,$97,$5C,$52,$7B,$FB,$77,$6F,$F3,$5F,$91,$FE,$7A,$9F,$F6
	.dc.b	$DB,$63,$3E,$3F,$F1,$B8,$C6,$9F,$EF,$2F,$AA,$FF,$65,$5F,$FB,$89
	.dc.b	$B7,$FE,$76,$51,$FC,$8D,$CC,$FF,$BE,$BE,$A9,$F0,$FF,$C9,$DB,$5F
	.dc.b	$EE,$CF,$AA,$E2,$FF,$BB,$A3,$38,$7F,$A0,$97,$81,$C4,$E5,$76,$B8
	.dc.b	$78,$3D,$4C,$5F,$17,$C1,$C1,$27,$C1,$F0,$F6,$BB,$70,$F9,$4C,$69
	.dc.b	$E9,$7F,$9F,$7D,$57,$E4,$2F,$FB,$1B,$6B,$FC,$71,$AB,$A3,$71,$C3
	.dc.b	$82,$5F,$F8,$F7,$1F,$8F,$3F,$E3,$FC,$39,$27,$F9,$6B,$7C,$2F,$09
	.dc.b	$FD,$3F,$1F,$C8,$DF,$8A,$3C,$BE,$8C,$8E,$C6,$4B,$5D,$E3,$D7,$E9
	.dc.b	$49,$CC,$F8,$67,$0F,$FA,$CB,$0F,$E6,$CD,$F8,$1F,$17,$CA,$FC,$47
	.dc.b	$E2,$F3,$4C,$39,$DF,$0D,$3D,$F8,$7B,$6B,$E5,$F7,$4B,$FC,$17,$C6
	.dc.b	$DA,$F0,$78,$1F,$40,$C6,$1C,$B1,$47,$D4,$F8,$F9,$87,$E2,$FD,$EB
	.dc.b	$87,$A7,$E0,$3E,$C9,$8C,$79,$3F,$73,$EA,$A5,$FA,$22,$D7,$FB,$4B
	.dc.b	$E9,$FF,$37,$3F,$E2,$26,$C7,$F9,$DF,$EA,$9F,$2F,$FE,$03,$9C,$C9
	.dc.b	$FF,$5A,$D7,$CE,$6C,$75,$F8,$FE,$1D,$7C,$7D,$69,$7F,$53,$EA,$9F
	.dc.b	$55,$8C,$3C,$9E,$C7,$73,$1B,$FC,$EC,$FF,$AB,$CF,$A7,$93,$C8,$7E
	.dc.b	$27,$D8,$B9,$78,$F8,$BD,$06,$2B,$E0,$3E,$BF,$7E,$4E,$CF,$53,$EF
	.dc.b	$F8,$99,$1E,$7E,$AB,$7F,$27,$EA,$BE,$21,$8D,$3C,$CF,$C7,$E3,$97
	.dc.b	$F9,$2F,$59,$E2,$E3,$06,$1F,$8A,$E4,$7B,$9F,$E3,$16,$E1,$F3,$71
	.dc.b	$C4,$FF,$8D,$7E,$A7,$F9,$66,$1C,$AB,$FC,$B7,$F2,$FF,$DE,$9F,$D0
	.dc.b	$CE,$B1,$FC,$BC,$E6,$B8,$1F,$EB,$57,$11,$AE,$7E,$DF,$13,$FE,$43
	.dc.b	$9C,$E1,$FE,$68,$C3,$F9,$F6,$1F,$EB,$D6,$A9,$FA,$F8,$7E,$9A,$72
	.dc.b	$4F,$84,$F5,$7A,$7E,$B7,$D9,$F2,$1F,$E5,$1F,$AA,$F4,$65,$DE,$31
	.dc.b	$9C,$8F,$F3,$B4,$FC,$2C,$FF,$CC,$2D,$7F,$CF,$AD,$7C,$DE,$6F,$C9
	.dc.b	$3E,$57,$0D,$AF,$8F,$16,$7E,$28,$CF,$EC,$CE,$89,$1A,$8F,$EB,$6D
	.dc.b	$AE,$5A,$7C,$1B,$B7,$F2,$66,$D7,$92,$39,$D7,$C7,$9C,$CF,$86,$AD
	.dc.b	$F8,$A3,$C1,$18,$3D,$77,$C8,$FA,$CF,$0A,$D4,$66,$7E,$C4,$61,$E7
	.dc.b	$1F,$88,$8C,$5F,$A6,$30,$71,$9F,$23,$6F,$C9,$3B,$9E,$64,$7D,$93
	.dc.b	$BB,$F2,$34,$E5,$7E,$07,$D2,$78,$2B,$93,$86,$9E,$A7,$93,$83,$EC
	.dc.b	$47,$B5,$FF,$52,$DF,$D6,$6D,$7F,$AB,$9F,$A4,$7F,$AA,$8F,$69,$63
	.dc.b	$83,$E1,$38,$D7,$FB,$1E,$3F,$03,$7F,$D1,$3E,$A3,$F3,$35,$D2,$31
	.dc.b	$97,$DC,$9C,$CF,$69,$CD,$FC,$D9,$CD,$F4,$8F,$F6,$BC,$63,$0F,$F5
	.dc.b	$B3,$63,$FA,$FB,$6F,$F3,$2D,$BE,$B1,$F1,$63,$13,$D3,$0F,$21,$1F
	.dc.b	$13,$67,$83,$6B,$EF,$1D,$0B,$3C,$CE,$24,$9E,$2D,$8D,$C0,$57,$91
	.dc.b	$C3,$5C,$D7,$86,$BF,$05,$5E,$16,$9E,$19,$EE,$48,$EE,$F2,$8F,$11
	.dc.b	$C1,$EF,$3D,$5F,$27,$D4,$FF,$88,$78,$7F,$09,$08,$C3,$F8,$75,$F3
	.dc.b	$8F,$3F,$71,$A9,$FF,$7A,$E4,$E1,$E7,$1F,$01,$B1,$98,$EE,$D7,$F2
	.dc.b	$9F,$52,$F9,$5F,$91,$1F,$C5,$A3,$0F,$F8,$81,$AF,$E0,$BE,$7B,$7B
	.dc.b	$7E,$55,$BB,$7E,$31,$B3,$F4,$1A,$FD,$4E,$4B,$8B,$E4,$A9,$27,$C1
	.dc.b	$ED,$70,$C8,$D2,$47,$97,$DB,$EB,$F4,$EF,$A9,$EF,$E7,$3D,$2A,$6A
	.dc.b	$37,$E1,$5C,$3F,$C7,$B7,$C5,$B1,$D3,$E3,$7D,$3F,$42,$F6,$E9,$9A
	.dc.b	$E3,$EA,$D7,$F9,$FE,$0B,$F2,$CF,$9B,$E3,$BE,$1F,$D4,$BD,$9A,$67
	.dc.b	$0F,$1E,$D5,$E7,$5E,$E8,$F7,$FA,$B6,$76,$7A,$9F,$9F,$F4,$7F,$E4
	.dc.b	$CE,$9F,$F2,$6F,$D2,$7F,$9C,$6B,$FC,$B6,$D7,$FA,$AC,$EF,$AB,$F0
	.dc.b	$6B,$D6,$E7,$12,$7F,$CC,$73,$96,$FC,$EF,$DE,$FA,$97,$D4,$D3,$C7
	.dc.b	$1E,$4F,$32,$D7,$03,$9B,$E8,$FA,$8F,$C3,$FD,$4B,$C9,$A6,$6B,$C3
	.dc.b	$EB,$D7,$D3,$5F,$02,$3E,$0F,$AD,$67,$67,$A1,$F8,$BF,$17,$F4,$BE
	.dc.b	$A6,$4F,$B2,$3E,$4F,$AF,$A8,$C1,$F0,$1F,$9B,$EC,$FE,$73,$8D,$F0
	.dc.b	$54,$CD,$7C,$6F,$5A,$BF,$3A,$BE,$AC,$7E,$93,$CF,$79,$7F,$8E,$F3
	.dc.b	$65,$FF,$A4,$27,$2D,$F2,$B8,$54,$8C,$1B,$F4,$7A,$4C,$8C,$1C,$FE
	.dc.b	$7B,$F3,$7F,$53,$23,$07,$7F,$93,$F2,$4C,$8C,$1E,$3F,$8B,$FD,$6F
	.dc.b	$CD,$48,$C1,$E7,$F2,$7D,$93,$23,$07,$AF,$D8,$FE,$07,$ED,$BA,$29
	.dc.b	$F0,$E3,$E2,$75,$5A,$E0,$7C,$37,$E4,$7A,$BA,$8C,$1F,$1D,$FE,$1F
	.dc.b	$F5,$2E,$EB,$DF,$2B,$FD,$77,$39,$6F,$CD,$F0,$F7,$E4,$60,$FF,$60
	.dc.b	$FD,$4B,$E8,$3F,$8D,$AE,$3A,$62,$33,$C1,$AF,$90,$97,$CA,$3F,$91
	.dc.b	$FD,$4B,$E5,$EF,$CC,$4A,$FF,$67,$CE,$5B,$F1,$DC,$3D,$49,$18,$3F
	.dc.b	$DA,$BD,$EF,$86,$FE,$5E,$FE,$D6,$3F,$CA,$2D,$F4,$3F,$D9,$EE,$E3
	.dc.b	$8F,$8F,$C3,$E5,$E4,$60,$FF,$70,$FD,$4F,$F4,$9B,$FF,$6B,$AF,$0B
	.dc.b	$F9,$EB,$67,$78,$63,$8B,$3E,$04,$9E,$71,$C3,$DE,$E3,$EF,$C9,$EF
	.dc.b	$1F,$D2,$3E,$A5,$E1,$DF,$80,$95,$FE,$F4,$9C,$BF,$EF,$6F,$A9,$6F
	.dc.b	$FD,$41,$7C,$65,$31,$1F,$D5,$3B,$FD,$15,$FE,$76,$7F,$AA,$3F,$F0
	.dc.b	$01,$AE,$78,$38,$78,$7C,$7E,$2E,$4E,$E3,$A6,$DD,$7C,$39,$3C,$E3
	.dc.b	$FB,$1F,$D4,$FF,$55,$8F,$F0,$C7,$2B,$D9,$FE,$E7,$2F,$83,$D9,$C3
	.dc.b	$87,$FB,$5F,$D4,$FF,$37,$C3,$FC,$41,$5F,$CC,$1E,$77,$CB,$D9,$DB
	.dc.b	$C3,$87,$8F,$AF,$2B,$67,$F8,$A7,$0F,$8E,$E0,$2E,$BF,$0F,$1C,$9D
	.dc.b	$4E,$5B,$EB,$79,$AA,$46,$0D,$F9,$1E,$AF,$22,$46,$0E,$7E,$A7,$BA
	.dc.b	$64,$60,$F2,$7D,$43,$E2,$7F,$C1,$BF,$52,$F8,$0A,$7F,$84,$27,$55
	.dc.b	$C3,$C1,$C5,$C8,$C1,$EF,$F9,$87,$93,$FE,$18,$F5,$19,$D3,$FC,$37
	.dc.b	$3A,$AE,$1E,$5E,$2E,$46,$0F,$95,$F5,$3F,$8A,$EB,$FB,$D1,$FE,$57
	.dc.b	$C5,$CB,$DC,$FF,$13,$D6,$F9,$FF,$C6,$AF,$E5,$AF,$8D,$A3,$4F,$F9
	.dc.b	$85,$AE,$7C,$0E,$7E,$3F,$46,$47,$92,$3B,$ED,$EF,$E5,$93,$E8,$8F
	.dc.b	$1F,$06,$A3,$07,$B1,$F3,$F8,$EC,$BF,$CB,$B9,$7F,$25,$6C,$66,$BF
	.dc.b	$D4,$93,$96,$F9,$EF,$A9,$E2,$D8,$CC,$60,$FF,$2F,$AF,$A1,$A4,$7F
	.dc.b	$CC,$6F,$F9,$93,$E4,$3E,$7E,$3F,$22,$7F,$CD,$9B,$FE,$75,$FF,$44
	.dc.b	$57,$FA,$27,$EA,$5C,$BF,$E6,$49,$D9,$FE,$FA,$3F,$0B,$5C,$79,$3A
	.dc.b	$BB,$C6,$CF,$F9,$B2,$6B,$FE,$95,$E5,$AF,$F4,$1A,$FD,$77,$2B,$E5
	.dc.b	$F3,$5A,$F9,$D9,$5E,$78,$B7,$E3,$F8,$BE,$66,$36,$F7,$3E,$07,$FD
	.dc.b	$0D,$5F,$BB,$6C,$FD,$78,$FF,$40,$E3,$FC,$99,$D7,$37,$1E,$59,$18
	.dc.b	$39,$3E,$CF,$A9,$F8,$B8,$7E,$51,$B3,$D3,$BF,$5B,$EF,$79,$3F,$C3
	.dc.b	$6B,$79,$FF,$AE,$5F,$9A,$F9,$4C,$78,$DD,$CF,$07,$8F,$6B,$97,$9F
	.dc.b	$E3,$31,$5C,$45,$FF,$4A,$9F,$07,$A7,$E9,$3D,$5F,$1D,$63,$C9,$EA
	.dc.b	$FB,$4C,$5F,$3B,$47,$C1,$F8,$D9,$5E,$BF,$A6,$B7,$E1,$F9,$6F,$B9
	.dc.b	$3F,$CA,$DB,$19,$F7,$BF,$0B,$18,$E2,$FD,$EA,$F2,$72,$6B,$C6,$B8
	.dc.b	$7F,$35,$FA,$97,$D4,$FE,$39,$B1,$9F,$0B,$13,$9E,$A9,$3C,$7D,$DB
	.dc.b	$AF,$C7,$FA,$97,$7F,$E6,$1B,$5F,$D0,$DF,$CC,$4E,$E7,$5C,$12,$B5
	.dc.b	$C5,$6B,$E9,$7D,$4F,$F2,$A9,$FE,$98,$D8,$CF,$67,$E9,$A3,$1C,$5F
	.dc.b	$C7,$AB,$FC,$60,$7F,$53,$2D,$7D,$D7,$FC,$64,$67,$6F,$D7,$FD,$4A
	.dc.b	$69,$FB,$29,$7B,$B8,$7F,$AF,$4F,$B2,$FF,$96,$B6,$3F,$DC,$AF,$E5
	.dc.b	$23,$3E,$17,$6C,$E5,$72,$EA,$B8,$FE,$47,$C6,$7B,$7F,$C0,$9B,$5F
	.dc.b	$DB,$57,$D3,$66,$75,$E6,$CA,$D7,$AB,$70,$FC,$E7,$D4,$FE,$EB,$5F
	.dc.b	$E8,$6D,$AF,$F7,$DB,$FE,$35,$7F,$BD,$39,$9F,$88,$FF,$AB,$63,$3F
	.dc.b	$0D,$FC,$5A,$1E,$ED,$8C,$70,$CA,$E7,$EE,$5C,$3F,$5D,$E4,$7E,$B5
	.dc.b	$BE,$2F,$F2,$08,$FF,$79,$57,$F8,$2D,$D6,$7F,$D6,$BE,$A7,$FD,$9D
	.dc.b	$3F,$E0,$F6,$DF,$EE,$5F,$F1,$24,$67,$E5,$BF,$95,$FD,$4B,$E1,$E9
	.dc.b	$FC,$BB,$EA,$77,$87,$F0,$19,$D5,$FF,$86,$D7,$81,$C8,$E3,$19,$7A
	.dc.b	$F8,$32,$30,$71,$7C,$3C,$32,$3F,$C4,$D8,$7F,$9C,$5E,$D1,$E6,$E1
	.dc.b	$91,$83,$C4,$FF,$44,$FA,$9F,$CB,$47,$F8,$BD,$B9,$7F,$19,$BF,$C2
	.dc.b	$CE,$63,$86,$F2,$30,$71,$7A,$78,$32,$3C,$BC,$F6,$BF,$92,$60,$F4
	.dc.b	$7C,$46,$78,$BF,$C1,$B1,$9B,$F2,$7A,$7E,$73,$6E,$0F,$47,$9D,$EB
	.dc.b	$3E,$97,$C9,$D8,$F6,$7A,$1E,$F3,$19,$78,$9F,$85,$DF,$27,$E5,$FC
	.dc.b	$6D,$AF,$3F,$83,$F1,$CC,$69,$E6,$7E,$57,$CD,$4B,$FA,$7E,$ED,$BF
	.dc.b	$7F,$BF,$F3,$CC,$61,$EE,$7E,$97,$D4,$CB,$FA,$51,$6B,$FA,$57,$D3
	.dc.b	$7E,$52,$FF,$C6,$2D,$AF,$E1,$D2,$FA,$97,$CB,$FF,$80,$2B,$53,$FC
	.dc.b	$4D,$FC,$33,$D3,$E3,$F5,$E0,$6D,$E1,$EB,$4C,$7F,$CE,$5F,$52,$FA
	.dc.b	$9C,$61,$E4,$F7,$FB,$18,$D7,$CA,$F8,$78,$E6,$1F,$96,$F6,$2D,$FB
	.dc.b	$38,$BC,$C6,$30,$F6,$BE,$9F,$8A,$97,$FE,$83,$DF,$FC,$AE,$D8,$F7
	.dc.b	$7C,$4C,$8F,$3F,$55,$C7,$E4,$FD,$4F,$C3,$31,$97,$CA,$7E,$37,$1C
	.dc.b	$BF,$F4,$87,$AD,$FC,$49,$F0,$47,$F8,$AB,$8D,$EE,$53,$86,$3F,$DF
	.dc.b	$E3,$35,$FE,$99,$FA,$8F,$E4,$12,$6F,$CB,$FF,$6F,$9F,$B3,$1A,$3F
	.dc.b	$D1,$8F,$1F,$83,$FF,$1C,$C6,$5B,$EC,$C9,$FF,$53,$53,$4C,$61,$FE
	.dc.b	$11,$C3,$FB,$9B,$7D,$DC,$ED,$FE,$7A,$9C,$B7,$CB,$EE,$7F,$92,$3E
	.dc.b	$9F,$D7,$7C,$E3,$C7,$E6,$4C,$FD,$D1,$67,$C1,$EA,$7F,$97,$DF,$EC
	.dc.b	$58,$FF,$60,$AF,$F2,$E4,$62,$74,$FC,$56,$CE,$E3,$FC,$55,$F5,$3F
	.dc.b	$16,$7F,$CB,$6F,$CC,$F6,$FF,$8B,$E7,$FE,$94,$6E,$1F,$A4,$CD,$F1
	.dc.b	$7F,$E6,$29,$CE,$3F,$E6,$33,$C2,$FF,$20,$7F,$CE,$0D,$7F,$5A,$D7
	.dc.b	$FC,$D5,$E7,$F0,$25,$FB,$21,$3F,$E7,$87,$9C,$FC,$BF,$8B,$F1,$36
	.dc.b	$BF,$DB,$D5,$FD,$F5,$EF,$F0,$FF,$47,$8F,$D0,$C8,$DB,$96,$93,$0F
	.dc.b	$F2,$85,$73,$7F,$C2,$59,$7F,$86,$33,$F8,$D7,$CE,$76,$F3,$6D,$F3
	.dc.b	$9D,$BF,$EF,$3A,$E4,$F9,$8D,$16,$7F,$CC,$11,$A7,$FD,$21,$5D,$EB
	.dc.b	$9C,$6B,$0C,$5F,$19,$C9,$FE,$CE,$BE,$55,$F3,$2F,$6F,$4A,$F2,$5E
	.dc.b	$23,$51,$E4,$7E,$0C,$67,$0F,$89,$1A,$3C,$53,$B7,$18,$78,$A3,$F1
	.dc.b	$55,$C6,$FF,$0E,$48,$FC,$05,$FF,$A2,$2F,$F1,$33,$97,$F3,$F1,$97
	.dc.b	$E3,$1A,$9F,$C0,$C7,$94,$8F,$8D,$FF,$21,$47,$BC,$FB,$AE,$9F,$AE
	.dc.b	$62,$38,$46,$6F,$D5,$3F,$EC,$18,$C4,$FC,$F3,$F3,$AE,$23,$B4,$FE
	.dc.b	$AE,$31,$1F,$12,$76,$76,$23,$E2,$9E,$11,$87,$C3,$1F,$A4,$9E,$67
	.dc.b	$D2,$3D,$AF,$53,$CD,$3F,$A4,$F3,$FF,$70,$7E,$E7,$D1,$B1,$93,$E8
	.dc.b	$C7,$EE,$88,$FD,$4F,$63,$F7,$A4,$70,$7E,$DF,$3F,$E4,$B7,$FE,$EF
	.dc.b	$6B,$FC,$27,$93,$F8,$AC,$62,$3F,$57,$EB,$E0,$95,$CD,$FE,$37,$F5
	.dc.b	$3F,$E6,$CE,$BF,$E0,$8D,$AF,$E4,$2B,$E7,$9C,$3A,$F7,$FF,$A6,$D7
	.dc.b	$F2,$6F,$A9,$FF,$23,$BF,$B1,$F6,$BF,$BF,$F1,$7E,$95,$BA,$7F,$80
	.dc.b	$AB,$75,$FE,$D1,$5E,$D6,$A3,$FD,$29,$1A,$B5,$71,$BD,$52,$B8,$BF
	.dc.b	$E9,$BA,$DB,$CB,$DD,$DE,$EF,$9E,$A7,$2B,$4D,$EB,$C3,$2F,$A6,$5E
	.dc.b	$99,$F0,$FD,$E3,$A3,$AE,$B2,$F6,$7F,$AB,$67,$4E,$03,$FC,$C9,$C3
	.dc.b	$A7,$FD,$E8,$DD,$3D,$71,$C3,$A3,$E0,$FD,$43,$EA,$BE,$1F,$AE,$F6
	.dc.b	$A9,$A8,$DF,$95,$70,$F1,$5F,$16,$C7,$4F,$8E,$F9,$BF,$55,$F3,$25
	.dc.b	$9F,$8A,$30,$F9,$2D,$5E,$4F,$73,$95,$F6,$8C,$3E,$0F,$8E,$FB,$3F
	.dc.b	$55,$FE,$6D,$8F,$88,$D7,$BF,$1F,$F9,$23,$05,$7C,$0D,$BE,$0B,$2B
	.dc.b	$D7,$1F,$13,$DC,$D4,$60,$F7,$BF,$88,$FA,$AF,$97,$A7,$CC,$FA,$6F
	.dc.b	$F6,$6D,$7D,$56,$BE,$23,$FD,$CD,$3C,$32,$FF,$2D,$AF,$77,$A4,$49
	.dc.b	$FB,$D3,$96,$FC,$27,$E6,$3E,$A9,$F3,$94,$F2,$47,$97,$E0,$2D,$70
	.dc.b	$39,$BE,$9F,$A9,$FD,$0F,$D5,$3D,$3A,$66,$BC,$3E,$FD,$7D,$43,$35
	.dc.b	$FC,$39,$F8,$9E,$F5,$AE,$27,$A9,$F9,$1F,$1D,$F6,$3E,$AA,$4F,$BA
	.dc.b	$3E,$67,$BD,$A8,$C1,$F0,$9F,$A1,$F0,$3F,$B8,$E3,$7C,$45,$33,$5F
	.dc.b	$23,$DF,$AF,$A3,$55,$FD,$99,$FE,$03,$E7,$3E,$AB,$F2,$FF,$33,$30
	.dc.b	$FD,$74,$E5,$BE,$67,$0A,$91,$83,$7F,$3B,$D2,$64,$60,$E7,$ED,$FE
	.dc.b	$93,$EA,$A4,$60,$EF,$EE,$79,$26,$46,$0F,$1F,$A5,$FB,$6F,$66,$46
	.dc.b	$0F,$3F,$9D,$EC,$99,$18,$3D,$7E,$FF,$F2,$0F,$81,$74,$D3,$E1,$C7
	.dc.b	$C4,$EB,$B5,$C0,$F8,$6F,$C8,$FA,$3D,$46,$0F,$8E,$FF,$2C,$FA,$A7
	.dc.b	$7D,$F0,$25,$7F,$19,$9C,$B7,$E7,$38,$7C,$3C,$8C,$1F,$C7,$FE,$A9
	.dc.b	$F0,$7F,$CE,$17,$1D,$31,$19,$E0,$D7,$C0,$4B,$E5,$1F,$D0,$3E,$AB
	.dc.b	$F9,$44,$7E,$71,$FE,$4D,$E4,$FF,$B3,$9B,$00,$F0,$F2,$FF,$95,$AF
	.dc.b	$91,$E9,$E9,$CA,$FE,$6D,$39,$7F,$9C,$77,$BE,$2B,$FA,$8B,$FB,$D8
	.dc.b	$FD,$6B,$70,$FE,$6E,$EE,$38,$F8,$FC,$3E,$62,$46,$0F,$E8,$9F,$54
	.dc.b	$AF,$EB,$AB,$C1,$A6,$23,$3C,$3A,$EF,$97,$CA,$38,$FB,$DC,$BD,$F9
	.dc.b	$3D,$A3,$FB,$4F,$D5,$3C,$2B,$EE,$CA,$FE,$AD,$39,$7F,$AC,$7D,$53
	.dc.b	$6F,$EE,$2B,$E3,$E9,$88,$FE,$EB,$DD,$E9,$AF,$EC,$0F,$F7,$16,$F8
	.dc.b	$FA,$E0,$DF,$87,$C3,$EA,$A5,$70,$8E,$7B,$F4,$F0,$E4,$F4,$8F,$F0
	.dc.b	$1F,$D5,$7F,$61,$8F,$ED,$CB,$E0,$32,$6B,$83,$FC,$13,$F5,$5F,$CE
	.dc.b	$EB,$F0,$D5,$FC,$D1,$E3,$EC,$6B,$BF,$7C,$3C,$3D,$69,$5F,$DE,$F0
	.dc.b	$F8,$4D,$EB,$EB,$C7,$23,$06,$FD,$5F,$0C,$C8,$C1,$CF,$C7,$F2,$F8
	.dc.b	$F2,$30,$77,$F4,$BD,$33,$23,$07,$F8,$37,$EA,$5F,$27,$FE,$26,$FA
	.dc.b	$A7,$BD,$4F,$F1,$4C,$EA,$B8,$78,$38,$B9,$18,$3F,$C3,$1F,$37,$FA
	.dc.b	$C7,$FC,$6F,$8F,$F8,$CB,$D3,$3F,$E3,$49,$D1,$FA,$FE,$3F,$D9,$52
	.dc.b	$33,$F1,$3B,$F8,$8F,$F1,$DF,$D5,$7E,$16,$BF,$AB,$E2,$E5,$EC,$6B
	.dc.b	$C0,$DF,$81,$6B,$FC,$59,$7C,$7C,$9F,$F1,$3B,$F9,$18,$FE,$C2,$FE
	.dc.b	$B5,$DC,$F2,$78,$F9,$1E,$2D,$FD,$9C,$D2,$7A,$C7,$9B,$87,$51,$83
	.dc.b	$C2,$FA,$BC,$8F,$F1,$3C,$7C,$03,$F8,$96,$3F,$C8,$32,$FF,$79,$36
	.dc.b	$34,$1C,$3C,$5F,$E5,$3F,$55,$C3,$D1,$34,$93,$FE,$FD,$1F,$47,$5A
	.dc.b	$65,$C1,$BF,$90,$FF,$33,$E1,$FE,$4E,$9F,$F3,$FF,$2F,$F9,$5D,$FE
	.dc.b	$55,$5F,$CA,$FE,$AB,$ED,$47,$DC,$BE,$35,$EF,$E5,$B3,$F8,$06,$BF
	.dc.b	$CC,$E6,$BF,$ED,$E3,$FD,$5A,$7F,$AB,$B9,$1F,$37,$AE,$6D,$FC,$BC
	.dc.b	$C7,$C9,$16,$FF,$C3,$A3,$E5,$FA,$99,$D7,$D8,$F8,$9F,$E5,$D5,$FE
	.dc.b	$73,$6C,$FE,$43,$C3,$D7,$36,$F9,$65,$EE,$2D,$7F,$8B,$3E,$AB,$E3
	.dc.b	$B7,$6F,$7E,$CF,$07,$E0,$D6,$B3,$FF,$3D,$BF,$EF,$E2,$56,$B8,$37
	.dc.b	$E3,$CC,$FF,$D1,$15,$FC,$6D,$B1,$D3,$D0,$91,$83,$6F,$83,$D5,$91
	.dc.b	$CB,$E3,$EE,$BE,$9F,$5F,$CA,$63,$4F,$2B,$E8,$F8,$D9,$7E,$2F,$A8
	.dc.b	$B7,$FE,$4C,$F9,$8F,$AF,$3F,$E9,$D6,$C6,$7D,$FF,$C0,$C6,$31,$FF
	.dc.b	$4D,$BF,$E3,$CA,$E1,$E5,$6B,$C7,$DB,$F8,$8F,$AA,$7C,$D5,$31,$E0
	.dc.b	$DB,$FF,$56,$4F,$FB,$29,$BC,$FC,$7D,$EE,$4F,$E4,$BE,$A9,$DF,$4C
	.dc.b	$77,$DB,$FF,$5B,$4B,$87,$81,$C7,$8E,$CF,$FA,$EF,$EA,$BF,$90,$B7
	.dc.b	$7F,$CF,$C7,$E0,$63,$07,$FB,$04,$FF,$11,$97,$5F,$D2,$BF,$E2,$43
	.dc.b	$07,$FB,$1F,$EA,$BF,$C3,$4D,$E4,$DF,$BD,$FA,$C9,$CC,$7F,$9F,$96
	.dc.b	$99,$35,$D9,$6F,$A6,$7C,$4E,$5D,$36,$3F,$DA,$DE,$2B,$DC,$FD,$CB
	.dc.b	$63,$3F,$39,$89,$CF,$99,$27,$8F,$AB,$75,$FF,$48,$FD,$57,$EB,$BB
	.dc.b	$3F,$DB,$6D,$AF,$E8,$AF,$F8,$7D,$FF,$73,$39,$9F,$F1,$D3,$FE,$9F
	.dc.b	$8C,$FC,$3F,$F0,$C8,$7B,$D6,$31,$C5,$2B,$9F,$BB,$70,$FF,$51,$F8
	.dc.b	$DF,$E2,$19,$FF,$7A,$36,$33,$E2,$FF,$18,$8C,$67,$FE,$F2,$FA,$AF
	.dc.b	$F6,$05,$FF,$BE,$DB,$7F,$EA,$E5,$1F,$C8,$1C,$CF,$FB,$EB,$EA,$9F
	.dc.b	$0F,$FC,$91,$B5,$FE,$57,$9F,$F7,$FF,$D5,$72,$8C,$1F,$DA,$A7,$78
	.dc.b	$7F,$AF,$17,$83,$CC,$E9,$85,$AE,$35,$FE,$63,$AF,$D3,$3C,$7C,$1D
	.dc.b	$36,$76,$A7,$E6,$C6,$1B,$1F,$EC,$CC,$3F,$CC,$93,$9E,$0F,$67,$04
	.dc.b	$8C,$1C,$5F,$E8,$7F,$55,$F9,$8D,$3E,$3E,$1F,$E7,$78,$C5,$7F,$82
	.dc.b	$30,$73,$70,$AE,$3B,$CA,$F3,$DF,$09,$E5,$3D,$BC,$29,$5E,$CC,$3F
	.dc.b	$35,$5F,$EE,$4B,$FE,$07,$3C,$BD,$0F,$29,$C2,$74,$FA,$3D,$A9,$3B
	.dc.b	$3C,$2F,$B3,$E0,$A4,$77,$F9,$3B,$7D,$F9,$7E,$09,$8D,$BE,$0C,$32
	.dc.b	$4F,$C6,$ED,$98,$7A,$7E,$36,$DF,$A7,$BF,$E5,$98,$C7,$9E,$3F,$E9
	.dc.b	$DB,$F8,$BF,$35,$F4,$0F,$EE,$7B,$FF,$8B,$63,$27,$CA,$F8,$0F,$AA
	.dc.b	$6D,$C1,$F3,$BE,$AB,$EC,$9A,$70,$7E,$23,$E9,$FF,$3E,$FF,$90,$1B
	.dc.b	$9F,$FA,$CF,$EA,$9F,$2D,$FE,$04,$9C,$C9,$FC,$5A,$D3,$E2,$C6,$1B
	.dc.b	$F3,$BC,$7E,$43,$97,$A9,$32,$FC,$CF,$D5,$3E,$AB,$19,$73,$9F,$27
	.dc.b	$BF,$E1,$33,$97,$67,$C9,$C9,$27,$F1,$1E,$B5,$BE,$47,$FB,$37,$5F
	.dc.b	$94,$E0,$F5,$7C,$3E,$F1,$F4,$BE,$FF,$80,$98,$7C,$78,$B5,$C6,$DF
	.dc.b	$15,$F1,$4F,$F1,$4E,$BF,$E0,$19,$D4,$7C,$9F,$AA,$98,$7D,$18,$B3
	.dc.b	$CF,$8B,$E7,$1F,$E3,$2F,$80,$FD,$CE,$9F,$32,$3F,$C6,$BC,$4F,$72
	.dc.b	$9C,$31,$FE,$07,$19,$7F,$82,$FD,$57,$F3,$5C,$4D,$F9,$8F,$EF,$D3
	.dc.b	$FE,$80,$8D,$1F,$E9,$47,$8F,$C2,$7F,$91,$23,$2D,$F6,$64,$FF,$12
	.dc.b	$A6,$D8,$C3,$E7,$61,$FE,$04,$AF,$1F,$BF,$9D,$7F,$57,$39,$AD,$F9
	.dc.b	$7D,$CF,$F2,$9F,$D4,$7A,$E3,$38,$7B,$3C,$CF,$21,$B7,$07,$C0,$F4
	.dc.b	$FF,$CB,$7F,$55,$EC,$6F,$C3,$49,$FF,$3D,$DA,$BF,$C9,$AF,$FA,$0D
	.dc.b	$AF,$FA,$91,$A8,$FE,$F2,$7F,$9C,$CC,$7B,$9F,$DF,$9F,$F4,$63,$5F
	.dc.b	$F5,$6B,$5D,$9E,$EF,$9B,$FF,$38,$C7,$F2,$58,$64,$9C,$FA,$1F,$13
	.dc.b	$E2,$EE,$3F,$D0,$AB,$FC,$29,$7F,$9D,$6E,$FF,$C6,$1B,$3E,$2B,$E4
	.dc.b	$FF,$4A,$C7,$FC,$26,$D7,$C7,$EF,$79,$19,$19,$38,$46,$30,$FD,$64
	.dc.b	$F2,$6B,$CA,$78,$46,$1B,$87,$F2,$C9,$57,$38,$C4,$59,$F9,$33,$6B
	.dc.b	$E7,$CD,$7F,$C5,$71,$63,$C5,$5C,$AB,$CE,$78,$0F,$A7,$1B,$9E,$0D
	.dc.b	$8D,$57,$66,$BD,$9B,$1D,$9E,$67,$68,$F7,$12,$BC,$4F,$C6,$AE,$F3
	.dc.b	$F9,$2A,$F9,$13,$F8,$87,$F1,$8F,$F9,$19,$A6,$9E,$E7,$FB,$51,$C3
	.dc.b	$E7,$30,$FE,$AB,$E5,$7E,$36,$26,$92,$BD,$B1,$96,$C6,$6B,$D5,$1F
	.dc.b	$A1,$7F,$2B,$19,$C3,$DA,$7E,$3D,$C4,$F9,$A3,$0F,$42,$4F,$43,$F5
	.dc.b	$95,$E1,$70,$76,$8D,$CF,$67,$C5,$3B,$8C,$4F,$2A,$F0,$BF,$0E,$3C
	.dc.b	$2F,$49,$E1,$5E,$53,$9D,$70,$9F,$D8,$F0,$7F,$12,$3F,$6B,$E9,$5A
	.dc.b	$FD,$84,$7D,$6B,$FF,$17,$D7,$C5,$38,$CF,$F0,$E8,$CC,$7A,$38,$F8
	.dc.b	$12,$74,$F3,$E2,$91,$C5,$B3,$D4,$7D,$57,$F8,$C3,$AB,$F8,$DB,$63
	.dc.b	$4B,$E7,$9C,$1A,$F5,$BF,$A5,$57,$3C,$DF,$2F,$FC,$92,$3F,$88,$B6
	.dc.b	$33,$1E,$B8,$D7,$D5,$7F,$97,$23,$F8,$97,$B9,$F8,$5E,$2F,$E2,$0D
	.dc.b	$DB,$F1,$15,$BA,$FE,$E4,$BD,$BD,$47,$F2,$D8,$D5,$FF,$8C,$1B,$5C
	.dc.b	$AB,$74,$97,$BD,$8C,$9F,$27,$4D,$CE,$AB,$FC,$D2,$BB,$33,$72,$73
	.dc.b	$1D,$BE,$AB,$F8,$AC,$7F,$2F,$C7,$B1,$9E,$9D,$74,$D9,$FF,$3A,$D7
	.dc.b	$07,$FC,$20,$FF,$48,$7F,$C2,$5F,$4E,$FA,$AF,$5F,$AF,$FE,$4C,$9F
	.dc.b	$F3,$EB,$5D,$C7,$C0,$97,$BB,$62,$4F,$FA,$12,$72,$DF,$8F,$F8,$FF
	.dc.b	$55,$F3,$E3,$FD,$12,$D7,$C7,$1D,$7D,$DE,$CC,$8E,$91,$E1,$F9,$0F
	.dc.b	$A3,$F5,$4F,$06,$99,$AF,$16,$95,$F7,$D7,$A6,$3D,$5E,$E5,$9D,$9E
	.dc.b	$97,$EE,$FD,$2F,$F9,$C3,$A7,$FC,$E3,$F4,$DF,$E8,$EA,$FF,$50,$B5
	.dc.b	$FF,$1E,$4E,$FA,$BF,$8B,$2F,$73,$9C,$49,$FF,$54,$CE,$5B,$F1,$9F
	.dc.b	$8C,$FA,$A7,$CF,$D3,$C7,$1E,$4F,$7E,$D7,$03,$9B,$E8,$FA,$9F,$CA
	.dc.b	$FD,$53,$C9,$A6,$6B,$C3,$EF,$D7,$D3,$5F,$02,$3E,$0F,$BB,$67,$67
	.dc.b	$A1,$F8,$BF,$21,$F4,$BE,$AA,$4F,$B2,$3E,$4F,$BB,$A8,$C1,$F0,$1F
	.dc.b	$9B,$F0,$3F,$AA,$E3,$7C,$55,$33,$5F,$1B,$DE,$AF,$CE,$AF,$AB,$1F
	.dc.b	$B2,$F3,$DF,$55,$F8,$EF,$66,$5F,$FB,$5A,$72,$DF,$33,$85,$48,$C1
	.dc.b	$BF,$A0,$E9,$32,$30,$73,$FA,$0F,$CD,$FB,$52,$30,$77,$F6,$BC,$93
	.dc.b	$23,$07,$8F,$D2,$FD,$7F,$CE,$48,$C1,$E7,$F3,$FD,$93,$23,$07,$AF
	.dc.b	$DA,$FE,$23,$EF,$BA,$29,$F0,$E3,$E2,$75,$5A,$E0,$7C,$37,$E4,$7A
	.dc.b	$DA,$8C,$1F,$1D,$FE,$3B,$F5,$4E,$EB,$DF,$2B,$FE,$00,$CE,$5B,$F3
	.dc.b	$5C,$3E,$16,$46,$0F,$F2,$C7,$8F,$FE,$DA,$FA,$A7,$C2,$7F,$2A,$5C
	.dc.b	$94,$C4,$67,$87,$5F,$2B,$30,$E7,$E4,$FF,$80,$F0,$FE,$6B,$1F,$E6
	.dc.b	$5F,$AA,$FE,$3D,$97,$FB,$A1,$FE,$44,$DF,$67,$F9,$CC,$6A,$FF,$C1
	.dc.b	$8B,$E4,$B3,$E9,$CC,$3B,$9D,$63,$FA,$0F,$80,$F8,$9B,$E7,$CA,$D8
	.dc.b	$9D,$38,$5E,$47,$2F,$99,$93,$98,$FE,$9B,$F5,$5C,$63,$9A,$F0,$F2
	.dc.b	$6A,$F7,$C9,$C3,$C1,$97,$C2,$39,$FB,$DD,$3D,$F9,$3D,$23,$FA,$DF
	.dc.b	$D5,$3C,$4B,$F2,$D2,$BD,$73,$97,$DB,$F5,$4D,$FF,$B1,$AF,$91,$A6
	.dc.b	$23,$FB,$3F,$81,$E9,$AF,$8A,$FF,$66,$6F,$93,$AE,$0D,$F8,$BC,$3E
	.dc.b	$2E,$57,$08,$E7,$BF,$4F,$16,$4F,$48,$FE,$EB,$F5,$5F,$D6,$7B,$9D
	.dc.b	$BF,$E4,$98,$FE,$AB,$5F,$EF,$36,$CE,$A3,$FD,$5D,$1C,$67,$FC,$A1
	.dc.b	$F5,$5F,$CF,$67,$3E,$C7,$63,$FC,$07,$19,$83,$F0,$DF,$C1,$AF,$5B
	.dc.b	$67,$0E,$1E,$3D,$7C,$B8,$A5,$F0,$8E,$BE,$AE,$A3,$07,$47,$C3,$E4
	.dc.b	$78,$BC,$89,$3E,$08,$F2,$FA,$7A,$8C,$1E,$47,$FC,$37,$F5,$2F,$92
	.dc.b	$FF,$0E,$FD,$53,$DE,$A6,$A3,$7E,$0D,$C3,$F3,$F7,$C5,$B1,$FE,$26
	.dc.b	$F9,$CF,$D8,$F1,$FF,$9E,$7D,$46,$9F,$E2,$A6,$DF,$EA,$2B,$6F,$FB
	.dc.b	$7D,$FD,$B3,$7C,$BE,$87,$5F,$8A,$FF,$19,$FD,$57,$E2,$E5,$D3,$F6
	.dc.b	$DF,$F1,$F7,$37,$73,$7D,$FC,$3B,$EC,$FE,$E2,$F9,$5F,$17,$F3,$11
	.dc.b	$D3,$5F,$F4,$94,$7E,$D6,$30,$7F,$BD,$5D,$CF,$1A,$F2,$F8,$FE,$6E
	.dc.b	$1F,$81,$CF,$30,$F0,$47,$AB,$87,$51,$83,$C8,$FB,$BC,$7F,$6C,$7F
	.dc.b	$9C,$F2,$FE,$53,$97,$F9,$D6,$11,$87,$53,$C1,$FF,$43,$47,$F3,$2F
	.dc.b	$AA,$E0,$D8,$E9,$E9,$27,$2D,$8F,$F3,$B7,$D1,$FF,$9D,$D3,$48,$F8
	.dc.b	$FE,$47,$F5,$98,$FD,$38,$FF,$41,$EF,$FC,$99,$1C,$87,$D5,$39,$7F
	.dc.b	$D3,$97,$CA,$BF,$D3,$DC,$B6,$FF,$D8,$8D,$79,$CD,$7F,$DF,$B3,$DF
	.dc.b	$97,$06,$F9,$9F,$37,$89,$F2,$7D,$0F,$13,$17,$F7,$8F,$3B,$D7,$E2
	.dc.b	$A4,$F0,$3D,$8F,$8F,$5F,$E8,$51,$FD,$11,$B5,$CF,$0F,$E5,$13,$9E
	.dc.b	$6E,$1C,$B2,$30,$71,$7D,$DF,$55,$F2,$AF,$C8,$DC,$3F,$41,$83,$C0
	.dc.b	$DE,$DC,$37,$99,$7F,$56,$C3,$83,$FE,$ED,$7E,$3E,$3D,$3C,$EE,$E7
	.dc.b	$83,$E4,$6D,$71,$F4,$3C,$66,$2B,$C6,$F9,$BD,$69,$3D,$FE,$3A,$D7
	.dc.b	$A3,$D7,$F5,$98,$DF,$F7,$19,$FF,$73,$BF,$37,$C6,$FC,$03,$E0,$FD
	.dc.b	$45,$8F,$ED,$FF,$31,$F6,$9B,$CF,$F8,$38,$FF,$0D,$46,$0D,$F9,$39
	.dc.b	$35,$E3,$58,$FE,$F3,$F5,$4F,$9A,$FC,$63,$63,$3E,$06,$27,$3D,$12
	.dc.b	$78,$FC,$05,$E1,$FC,$47,$D5,$3B,$BF,$2C,$DA,$FF,$03,$AE,$FC,$CE
	.dc.b	$B8,$25,$6B,$92,$E1,$F9,$3F,$AA,$FE,$57,$3F,$E1,$06,$C6,$7B,$7F
	.dc.b	$49,$18,$E6,$FE,$49,$5F,$E3,$23,$FC,$2E,$5A,$FC,$F3,$FE,$34,$33
	.dc.b	$BF,$EB,$7E,$AB,$F6,$F1,$FE,$1E,$6C,$F0,$F7,$74,$EF,$59,$5B,$EE
	.dc.b	$B3,$FA,$BB,$FB,$CD,$D3,$B7,$5F,$77,$F7,$9E,$2B,$DB,$A6,$3E,$72
	.dc.b	$E5,$FE,$30,$9F,$F1,$6B,$7C,$CE,$3E,$A7,$27,$F8,$37,$D5,$7E,$CB
	.dc.b	$87,$FA,$33,$6B,$F8,$9E,$88,$FF,$18,$47,$F8,$F1,$D4,$7F,$18,$7F
	.dc.b	$D5,$D1,$9F,$88,$EB,$EE,$F6,$7B,$F1,$49,$E5,$EE,$5C,$3F,$86,$79
	.dc.b	$1F,$E3,$19,$FF,$24,$B6,$33,$E4,$7F,$1D,$8C,$67,$FD,$77,$EA,$BF
	.dc.b	$D9,$D7,$FE,$50,$6D,$FF,$17,$7F,$C4,$31,$9F,$83,$FE,$4D,$F5,$4F
	.dc.b	$88,$A7,$F2,$9F,$A9,$DE,$1F,$C7,$E7,$57,$FE,$5C,$5E,$0F,$23,$8C
	.dc.b	$65,$EB,$E1,$48,$C1,$C5,$F0,$F0,$C8,$FF,$33,$E1,$FE,$6B,$7B,$47
	.dc.b	$9B,$86,$46,$0F,$13,$FC,$FF,$EA,$BF,$23,$B7,$F7,$F6,$CF,$DA,$B7
	.dc.b	$3E,$F4,$E1,$C1,$2B,$FC,$EF,$7C,$27,$FA,$9B,$FC,$91,$BE,$17,$73
	.dc.b	$E3,$74,$DA,$E3,$E8,$F8,$CC,$57,$F4,$7B,$47,$9F,$D2,$95,$C0,$F0
	.dc.b	$BE,$BF,$42,$47,$C4,$F9,$3B,$87,$3E,$7F,$80,$63,$7F,$80,$FC,$3E
	.dc.b	$F9,$7F,$33,$E3,$AD,$F8,$FC,$0F,$90,$63,$0F,$90,$FC,$BF,$9A,$97
	.dc.b	$F5,$3D,$DB,$7E,$BF,$7F,$E8,$18,$C3,$E8,$3F,$4F,$EA,$A5,$FC,$C8
	.dc.b	$B5,$FD,$A7,$EA,$3F,$2B,$7F,$E3,$56,$D7,$EC,$3E,$A9,$F2,$FF,$DE
	.dc.b	$E7,$37,$FE,$06,$7F,$0C,$F3,$F8,$FD,$78,$3B,$78,$7C,$9C,$C7,$FD
	.dc.b	$67,$F5,$2F,$AA,$C6,$1E,$3F,$67,$B1,$8D,$7C,$8F,$87,$8E,$61,$F9
	.dc.b	$6F,$62,$DF,$AF,$8B,$CC,$63,$0F,$63,$E9,$F8,$A9,$7F,$EC,$4D,$FF
	.dc.b	$D0,$4D,$8F,$77,$C5,$48,$F3,$F5,$5C,$7E,$47,$D5,$7C,$33,$19,$7C
	.dc.b	$97,$E3,$71,$CB,$FF,$68,$FA,$CF,$1B,$18,$7F,$86,$79,$1E,$E7,$F8
	.dc.b	$9D,$B7,$E5,$C3,$13,$FE,$2B,$FA,$AF,$E3,$F6,$6F,$CC,$7F,$6F,$8F
	.dc.b	$F1,$24,$68,$FF,$45,$5F,$FB,$89,$B1,$AF,$84,$E9,$F0,$9D,$7D,$A9
	.dc.b	$7F,$EE,$8C,$1B,$63,$0E,$BE,$F7,$8C,$C7,$07,$F9,$F7,$3F,$EE,$93
	.dc.b	$E2,$DF,$D0,$7F,$90,$FE,$A3,$DB,$19,$CB,$C9,$E7,$7B,$4D,$B8,$3F
	.dc.b	$DF,$3E,$E7,$F0,$39,$FE,$E8,$73,$F8,$AF,$F2,$74,$64,$FE,$C3,$1F
	.dc.b	$D2,$0D,$CF,$F9,$CE,$33,$1F,$D3,$1A,$FF,$98,$3E,$AB,$E2,$BC,$5A
	.dc.b	$FF,$52,$6B,$BE,$0F,$F3,$53,$C6,$FF,$CC,$12,$6C,$E1,$F3,$FB,$D3
	.dc.b	$0F,$F3,$7E,$BF,$EA,$5C,$FF,$C0,$0F,$FA,$9F,$2F,$F2,$0B,$6B,$86
	.dc.b	$5F,$E6,$1A,$DD,$FA,$E7,$E1,$F9,$F2,$3A,$9F,$DA,$A5,$EC,$66,$FF
	.dc.b	$CD,$58,$FF,$88,$5B,$7F,$5E,$75,$8F,$8E,$2E,$1C,$E6,$D7,$FA,$22
	.dc.b	$F9,$5E,$EB,$C3,$3F,$E8,$CC,$3C,$31,$F7,$30,$E3,$3F,$E2,$F7,$C1
	.dc.b	$19,$24,$79,$1E,$6A,$79,$B9,$25,$78,$5F,$8D,$87,$6C,$3C,$B7,$F8
	.dc.b	$1C,$3B,$D7,$08,$C1,$E0,$BF,$A9,$7C,$6B,$CF,$87,$82,$7B,$12,$B8
	.dc.b	$DF,$81,$E4,$62,$35,$1F,$0E,$FA,$4F,$E5,$6F,$F1,$73,$F0,$E3,$04
	.dc.b	$7F,$38,$64,$B8,$7D,$97,$F4,$0D,$7C,$11,$F9,$FF,$3B,$E1,$11,$F2
	.dc.b	$C7,$D9,$8C,$3F,$0C,$C4,$69,$FF,$32,$46,$B0,$F5,$C6,$27,$55,$FD
	.dc.b	$B9,$B5,$B7,$C7,$3D,$0E,$C7,$12,$CF,$16,$CF,$17,$F1,$D7,$FC,$02
	.dc.b	$31,$1F,$0E,$FF,$51,$18,$AF,$BF,$3D,$E7,$CF,$1D,$5B,$3F,$C5,$7D
	.dc.b	$99,$87,$E3,$BF,$67,$FE,$21,$9A,$73,$8F,$D2,$8C,$57,$FB,$FD,$B1
	.dc.b	$99,$FB,$71,$FE,$38,$8E,$1E,$CE,$09,$5F,$72,$1F,$57,$FE,$26,$EA
	.dc.b	$C3,$63,$F0,$2B,$E7,$DE,$B6,$BD,$7F,$E9,$75,$F8,$6D,$32,$F5,$6B
	.dc.b	$99,$F0,$28,$CB,$FC,$A5,$71,$69,$B1,$CE,$30,$72,$FA,$BF,$F2,$9C
	.dc.b	$7E,$DB,$5A,$FF,$17,$F6,$FF,$CF,$7C,$2D,$BF,$CF,$B7,$A8,$C5,$04
	.dc.b	$3D,$BE,$35,$C2,$ED,$6E,$B9,$4F,$4A,$CC,$F2,$E5,$B8,$BE,$46,$9A
	.dc.b	$C3,$F9,$2D,$7F,$0E,$7F,$00,$DA,$D1,$BE,$3E,$BF,$57,$F1,$27,$F9
	.dc.b	$A6,$3D,$8C,$EB,$AE,$6B,$3F,$C6,$AB,$86,$DF,$02,$11,$FD,$79,$46
	.dc.b	$7E,$A9,$F5,$7E,$CF,$5D,$ED,$53,$DF,$3A,$BF,$E4,$EB,$CA,$CD,$C9
	.dc.b	$CC,$70,$F8,$EF,$8F,$F5,$7F,$3E,$7F,$97,$B5,$F1,$F2,$7F,$AD,$30
	.dc.b	$57,$83,$DC,$E5,$5E,$08,$C3,$E2,$F8,$EF,$A7,$F5,$7F,$E7,$28,$FE
	.dc.b	$76,$D7,$A9,$F8,$1B,$4F,$B3,$5F,$6B,$27,$DB,$1F,$03,$DC,$D4,$60
	.dc.b	$F8,$2F,$E0,$BE,$AF,$E7,$70,$7C,$FF,$A7,$FF,$49,$D7,$F4,$C6,$BE
	.dc.b	$21,$3C,$35,$FF,$69,$AF,$77,$A4,$49,$FE,$AB,$39,$6F,$C2,$7E,$4B
	.dc.b	$EA,$DF,$3B,$4F,$24,$79,$7E,$02,$D7,$03,$9B,$E9,$FA,$AF,$CD,$FD
	.dc.b	$5B,$CF,$A6,$6B,$C3,$EF,$D7,$D5,$9A,$FE,$14,$FC,$3F,$7A,$D7,$03
	.dc.b	$D2,$FC,$7F,$8D,$FA,$FF,$57,$27,$DB,$1F,$2F,$DE,$D4,$60,$F8,$2F
	.dc.b	$CF,$F8,$1F,$D8,$F1,$BE,$22,$99,$AF,$8F,$EF,$57,$CF,$5F,$5E,$3F
	.dc.b	$73,$E7,$BE,$AF,$F2,$BE,$FC,$BF,$F0,$1C,$E5,$BE,$67,$0A,$91,$83
	.dc.b	$7F,$3D,$D2,$64,$60,$E7,$EC,$7E,$8B,$EA,$E4,$60,$EF,$F3,$FE,$49
	.dc.b	$91,$83,$C7,$E9,$7E,$D7,$E7,$24,$60,$F3,$F9,$FE,$C9,$91,$83,$D7
	.dc.b	$EF,$7F,$18,$F8,$27,$45,$3E,$1C,$7C,$4E,$BB,$5C,$0F,$86,$FC,$8F
	.dc.b	$A4,$D4,$60,$F8,$EF,$F2,$4F,$AB,$77,$DF,$02,$57,$F8,$BA,$72,$DF
	.dc.b	$9D,$E1,$F1,$12,$30,$7F,$8D,$3E,$AD,$F1,$1F,$CC,$17,$1D,$31,$19
	.dc.b	$E0,$D7,$C0,$CB,$E5,$1F,$CE,$3E,$AF,$F9,$17,$97,$FC,$95,$FE,$4F
	.dc.b	$E5,$7F,$45,$A4,$61,$FE,$90,$F0,$8D,$2F,$92,$E9,$E9,$CA,$FF,$25
	.dc.b	$CE,$5F,$F2,$6F,$7B,$E2,$FF,$A4,$BF,$BB,$8F,$E0,$8D,$C7,$F9,$C3
	.dc.b	$B8,$E3,$E3,$70,$F9,$99,$18,$3F,$CB,$3F,$56,$AF,$EA,$AB,$C1,$A6
	.dc.b	$23,$3C,$3A,$EF,$97,$CA,$38,$FB,$DC,$BD,$F9,$3D,$A3,$FB,$0F,$D5
	.dc.b	$BC,$2B,$EE,$CA,$FF,$35,$CE,$5F,$F3,$6F,$D5,$B6,$FE,$D2,$BE,$42
	.dc.b	$98,$8F,$ED,$BD,$DE,$9A,$FF,$3A,$BF,$DB,$5B,$E3,$EB,$83,$7E,$1F
	.dc.b	$0F,$AB,$95,$C2,$39,$EF,$D3,$C3,$93,$D2,$3F,$BD,$FD,$5F,$F6,$08
	.dc.b	$FF,$41,$BF,$D7,$5B,$F0,$1A,$E0,$FF,$00,$FD,$5F,$F3,$99,$FF,$06
	.dc.b	$57,$F3,$27,$8F,$B1,$AE,$FD,$F0,$F0,$F5,$65,$7F,$A4,$30,$F8,$4D
	.dc.b	$EB,$EB,$C7,$23,$06,$FD,$4F,$0C,$C8,$C1,$CF,$C7,$F2,$F8,$F2,$30
	.dc.b	$77,$F4,$BD,$33,$23,$07,$FA,$8B,$EA,$9F,$21,$FE,$1E,$FA,$B7,$BD
	.dc.b	$4F,$F1,$0C,$EA,$B8,$78,$38,$B9,$18,$3F,$D5,$BF,$39,$FA,$B7,$FC
	.dc.b	$61,$8F,$F8,$AB,$D3,$3F,$E2,$C9,$D1,$FA,$F9,$FF,$1C,$38,$F8,$9D
	.dc.b	$7C,$4F,$F8,$CB,$EA,$FF,$09,$5F,$DB,$70,$72,$75,$35,$E0,$6F,$C0
	.dc.b	$B5,$FE,$C0,$BE,$3E,$4F,$F8,$9D,$FC,$84,$7F,$77,$7F,$58,$EE,$79
	.dc.b	$3C,$7C,$8F,$16,$DE,$BE,$59,$3D,$63,$CD,$C1,$A8,$C1,$E1,$7D,$5E
	.dc.b	$47,$F8,$D2,$3F,$D6,$78,$FF,$1E,$CB,$FC,$F4,$D8,$CC,$ED,$FE,$30
	.dc.b	$23,$F9,$3F,$D5,$F0,$F4,$4D,$24,$FF,$9D,$BE,$93,$FC,$EE,$9A,$47
	.dc.b	$FD,$15,$F2,$1F,$E6,$AC,$7F,$C9,$91,$FE,$86,$EF,$FF,$74,$BF,$E6
	.dc.b	$DA,$FF,$37,$7D,$5B,$9B,$CB,$87,$1B,$F3,$F3,$5C,$3F,$98,$35,$FF
	.dc.b	$3B,$4D,$7F,$D0,$F3,$DB,$9B,$06,$F9,$9F,$39,$89,$F1,$FC,$BF,$13
	.dc.b	$17,$FE,$31,$3F,$D7,$27,$FA,$0F,$E5,$FB,$1B,$9F,$5B,$E3,$DB,$F9
	.dc.b	$64,$7F,$A2,$5F,$F1,$FF,$37,$87,$C3,$D3,$BE,$69,$5B,$3F,$D2,$B1
	.dc.b	$8C,$3F,$C8,$5F,$57,$F2,$5B,$B7,$AF,$C1,$F8,$53,$9C,$BF,$CF,$4F
	.dc.b	$C2,$AE,$3C,$1A,$BD,$BC,$3C,$79,$97,$86,$BF,$D4,$17,$FE,$77,$8C
	.dc.b	$42,$3F,$CF,$F1,$87,$F9,$E3,$7D,$0E,$73,$E8,$7C,$BE,$AC,$9E,$1F
	.dc.b	$1F,$6B,$DB,$EB,$FA,$8C,$69,$EA,$7D,$BF,$1B,$2F,$C1,$F5,$36,$FF
	.dc.b	$CC,$DF,$33,$F7,$27,$CE,$D8,$CF,$BF,$F8,$78,$C6,$3F,$E9,$F5,$E5
	.dc.b	$64,$D7,$8F,$70,$F8,$3F,$56,$F9,$DF,$C8,$36,$33,$E1,$62,$73,$DB
	.dc.b	$27,$8F,$BD,$75,$FF,$3E,$7D,$5F,$E3,$09,$E0,$7E,$6D,$C7,$05,$31
	.dc.b	$C5,$70,$FA,$7F,$57,$FC,$89,$BB,$7E,$8E,$1D,$76,$3F,$4D,$3F,$E2
	.dc.b	$43,$ED,$97,$2F,$F4,$73,$FE,$26,$33,$B7,$EB,$7E,$AF,$FD,$C5,$1F
	.dc.b	$82,$6C,$F0,$F7,$B4,$EF,$59,$5B,$EC,$B3,$FE,$9B,$BF,$F1,$14,$F3
	.dc.b	$D3,$B7,$4F,$77,$F7,$9E,$33,$DC,$A6,$3E,$76,$DF,$E5,$25,$E6,$F0
	.dc.b	$38,$FA,$B6,$7F,$33,$F5,$7F,$B4,$D3,$FD,$58,$DE,$CF,$F7,$03,$FE
	.dc.b	$21,$9D,$F8,$9F,$C3,$63,$FD,$3D,$1C,$BE,$1F,$1E,$F6,$5D,$71,$C9
	.dc.b	$EF,$EE,$DC,$BF,$D7,$5E,$3F,$E9,$9B,$E2,$FF,$19,$8F,$D5,$C6,$2B
	.dc.b	$F5,$FF,$57,$FE,$BD,$6F,$CB,$7F,$1F,$84,$58,$FE,$45,$F5,$6F,$88
	.dc.b	$A7,$F2,$5F,$AB,$DD,$7E,$EA,$74,$DF,$0B,$89,$C1,$C1,$BF,$0F,$A9
	.dc.b	$A7,$07,$3E,$0F,$09,$DF,$E2,$2C,$77,$E1,$F2,$98,$D7,$FD,$B9,$8F
	.dc.b	$CC,$7F,$DB,$DF,$57,$F8,$CE,$5F,$6C,$04,$66,$FF,$CE,$1A,$3A,$7F
	.dc.b	$A1,$4E,$6B,$83,$FC,$8E,$BF,$CE,$CD,$8C,$9C,$38,$78,$FA,$52,$BF
	.dc.b	$8F,$E1,$F9,$17,$F9,$0D,$FF,$42,$1E,$97,$80,$E1,$87,$A9,$F1,$FA
	.dc.b	$92,$B8,$C5,$9F,$7F,$C2,$7A,$0F,$AD,$F2,$76,$BB,$FA,$7E,$C3,$15
	.dc.b	$F1,$9F,$7F,$85,$27,$CF,$F1,$D6,$BE,$5F,$87,$F1,$0C,$70,$FC,$C8
	.dc.b	$F9,$71,$F2,$3E,$6A,$5F,$C0,$F8,$0B,$5F,$53,$E0,$BE,$69,$8C,$39
	.dc.b	$70,$7F,$8F,$AF,$E2,$FD,$5F,$D2,$36,$E0,$FE,$93,$F5,$1F,$9B,$C1
	.dc.b	$7E,$38,$C3,$E5,$FA,$B7,$CB,$FF,$7D,$9C,$DF,$FA,$BD,$FF,$52,$3D
	.dc.b	$3A,$F5,$E0,$6D,$E1,$EC,$49,$FE,$B3,$F5,$6F,$AB,$C6,$9E,$29,$E9
	.dc.b	$F0,$1D,$CC,$EB,$E5,$7C,$5C,$92,$7F,$0B,$ED,$5B,$E4,$7E,$5F,$5F
	.dc.b	$F2,$5C,$FC,$0E,$3D,$63,$EF,$7D,$9F,$01,$27,$47,$C6,$7E,$07,$C5
	.dc.b	$48,$F4,$F5,$DC,$7B,$8C,$5F,$57,$F1,$CD,$65,$F3,$5F,$95,$C7,$30
	.dc.b	$FE,$F7,$EC,$3C,$6C,$63,$D3,$91,$EF,$7F,$8B,$5B,$7F,$6B,$0C,$4F
	.dc.b	$F8,$C7,$EA,$FF,$95,$4F,$FB,$8D,$B7,$FB,$9F,$98,$FE,$DF,$39,$F4
	.dc.b	$B8,$7C,$24,$8F,$F0,$6D,$F0,$B3,$FE,$8C,$8C,$FB,$7D,$CF,$F2,$06
	.dc.b	$1F,$02,$32,$72,$F8,$0F,$21,$B7,$07,$6D,$7D,$07,$F9,$27,$EA,$3D
	.dc.b	$B8,$FC,$CC,$7E,$25,$79,$3C,$EF,$71,$C1,$C1,$FE,$24,$F7,$7F,$B1
	.dc.b	$D7,$F9,$B1,$AF,$F1,$90,$E1,$FE,$38,$D7,$27,$8F,$E6,$BF,$CB,$91
	.dc.b	$FC,$6E,$18,$27,$FC,$C1,$F5,$6F,$8C,$A6,$37,$B8,$FF,$8E,$27,$4F
	.dc.b	$FA,$03,$0F,$E5,$91,$87,$0D,$7F,$D0,$AD,$7D,$58,$FF,$92,$DF,$77
	.dc.b	$C0,$7F,$9F,$1F,$5B,$5F,$7B,$63,$FC,$A5,$CF,$FC,$45,$F8,$77,$F2
	.dc.b	$67,$DF,$1F,$03,$CD,$93,$F3,$7D,$0B,$87,$E8,$2B,$F9,$8C,$FD,$1A
	.dc.b	$F8,$F5,$AC,$3B,$45,$9F,$4C,$DB,$FA,$33,$5F,$AB,$16,$FA,$C7,$79
	.dc.b	$EB,$5D,$C9,$19,$7C,$C6,$2F,$C2,$D3,$C2,$58,$F0,$B5,$F0,$B6,$BC
	.dc.b	$33,$CA,$7E,$05,$F9,$6F,$CC,$DB,$E0,$D9,$F2,$1C,$23,$A4,$FD,$32
	.dc.b	$3F,$83,$8F,$CA,$C6,$1F,$C3,$5F,$C2,$7F,$39,$19,$6C,$FF,$A1,$35
	.dc.b	$FD,$3C,$78,$30,$FE,$9D,$18,$7D,$46,$38,$3E,$84,$66,$BF,$1B,$7F
	.dc.b	$B3,$9C,$92,$3E,$34,$EE,$FE,$49,$FC,$89,$C3,$EA,$8F,$24,$F1,$8F
	.dc.b	$24,$7E,$CE,$77,$87,$CE,$8C,$56,$F0,$F3,$C6,$0F,$0C,$65,$F3,$9D
	.dc.b	$8C,$92,$FB,$9F,$E1,$9B,$EF,$87,$B6,$75,$7D,$AB,$57,$FC,$03,$CF
	.dc.b	$95,$F3,$7D,$AB,$5F,$A1,$9F,$D6,$12,$7E,$D6,$5F,$EE,$97,$EE,$7D
	.dc.b	$5F,$F8,$97,$A2,$FA,$9F,$CF,$30,$FE,$05,$2F,$9E,$7A,$BA,$BF,$C2
	.dc.b	$6E,$F9,$6F,$C3,$61,$FC,$49,$B1,$98,$DC,$6B,$EA,$DC,$74,$93,$F8
	.dc.b	$F8,$CB,$E6,$F8,$5E,$B9,$9A,$48,$FC,$B7,$D5,$BD,$CF,$E5,$9C,$9F
	.dc.b	$8E,$6F,$07,$DD,$6D,$7F,$92,$6B,$83,$FE,$DA,$5E,$E6,$A1,$1A,$7F
	.dc.b	$B6,$B6,$FF,$C5,$0F,$08,$E5,$F0,$1C,$76,$7C,$9D,$3A,$52,$6F,$36
	.dc.b	$19,$7E,$38,$F1,$7D,$5F,$C4,$8F,$CE,$65,$D4,$CF,$06,$B9,$AC,$FF
	.dc.b	$BD,$2B,$86,$9F,$E3,$D7,$FA,$5C,$7F,$81,$BE,$A1,$F5,$7E,$FF,$6D
	.dc.b	$EF,$53,$E1,$4E,$AB,$FD,$FE,$BC,$EC,$DC,$9C,$C7,$0F,$90,$F9,$5F
	.dc.b	$57,$F1,$67,$FC,$2A,$DC,$FF,$CA,$18,$2B,$BF,$C0,$71,$BE,$F1,$87
	.dc.b	$C3,$F2,$3F,$57,$EA,$FF,$CD,$F1,$FE,$83,$C3,$FC,$B5,$18,$8F,$74
	.dc.b	$FA,$B5,$F5,$B2,$3D,$71,$EE,$F8,$1D,$46,$0F,$7B,$F8,$3F,$A9,$FF
	.dc.b	$33,$61,$FE,$67,$FA,$7F,$F4,$75,$78,$7A,$FF,$BE,$46,$23,$FC,$67
	.dc.b	$5C,$27,$FC,$00,$BE,$03,$A4,$49,$E4,$7F,$A7,$67,$4E,$03,$FE,$E1
	.dc.b	$71,$F5,$6F,$9F,$FC,$C3,$63,$84,$ED,$F1,$FC,$17,$AF,$E4,$64,$6A
	.dc.b	$3D,$DF,$55,$FA,$1F,$AB,$7A,$74,$CD,$74,$F8,$2A,$FB,$AB,$E3,$47
	.dc.b	$C7,$F8,$1B,$3B,$3C,$6F,$CB,$F9,$2F,$B5,$F5,$72,$7C,$F1,$F3,$FE
	.dc.b	$0B,$51,$83,$D6,$FD,$3F,$84,$FD,$B7,$1B,$E3,$29,$9A,$F8,$3F,$01
	.dc.b	$5F,$A7,$5F,$6E,$3F,$7D,$E9,$3E,$AF,$F2,$FF,$03,$2F,$E8,$4E,$5B
	.dc.b	$E6,$70,$A9,$18,$37,$F4,$1D,$26,$46,$0E,$7E,$EF,$E9,$3D,$E9,$18
	.dc.b	$3B,$FD,$07,$92,$64,$60,$F1,$FA,$5F,$B7,$F9,$49,$18,$3C,$FE,$7F
	.dc.b	$B2,$64,$60,$F5,$FB,$9F,$C7,$7E,$0D,$D1,$4F,$87,$1F,$13,$AA,$D7
	.dc.b	$03,$E1,$BF,$23,$D9,$D4,$60,$F8,$EF,$F2,$9F,$AB,$77,$5E,$F9,$5F
	.dc.b	$9B,$9C,$B7,$D8,$E1,$F0,$F2,$30,$7E,$87,$EA,$DF,$0F,$FC,$D5,$71
	.dc.b	$D3,$11,$9E,$0D,$7C,$AC,$BE,$51,$FC,$F7,$EA,$FF,$91,$F9,$7F,$EE
	.dc.b	$17,$F9,$3F,$95,$FE,$75,$6D,$03,$C2,$35,$F2,$9C,$FD,$29,$5F,$B7
	.dc.b	$A4,$FF,$4B,$EF,$7C,$5D,$F3,$A5,$FE,$F6,$72,$DF,$1F,$87,$CC,$C8
	.dc.b	$C1,$FC,$0F,$EA,$D5,$FD,$5D,$78,$54,$C4,$67,$8B,$5E,$04,$BE,$51
	.dc.b	$C7,$DE,$E5,$EF,$C9,$ED,$1F,$D8,$FE,$AD,$E1,$DF,$76,$57,$F1,$59
	.dc.b	$CB,$FC,$5F,$EA,$DB,$7F,$6B,$5F,$25,$4C,$47,$F6,$FE,$FF,$4D,$7F
	.dc.b	$1F,$7F,$B4,$B7,$C8,$D7,$06,$FC,$4E,$1F,$57,$2B,$84,$73,$DF,$A7
	.dc.b	$89,$27,$A4,$7F,$7D,$FA,$BF,$EB,$F1,$FC,$B5,$FE,$B8,$DF,$80,$D7
	.dc.b	$07,$F8,$0F,$EA,$FF,$9D,$4F,$E0,$EB,$F9,$73,$FE,$F0,$6F,$7B,$BF
	.dc.b	$FD,$7D,$5C,$28,$FC,$27,$F9,$E2,$F5,$7B,$1D,$F8,$BC,$1D,$7E,$1E
	.dc.b	$39,$7C,$23,$C9,$EA,$6A,$30,$74,$7D,$1E,$3F,$A7,$C7,$93,$E0,$8F
	.dc.b	$67,$A5,$A8,$C1,$E4,$7F,$C4,$5F,$54,$F9,$2F,$F1,$27,$D5,$BD,$EA
	.dc.b	$6A,$37,$E0,$DC,$3F,$AC,$5F,$16,$C7,$F8,$B7,$E4,$3F,$5B,$1F,$E3
	.dc.b	$0C,$BF,$C6,$1E,$9F,$F8,$FE,$3F,$B0,$4E,$8F,$D8,$3F,$B3,$71,$F1
	.dc.b	$3C,$FE,$27,$FC,$71,$F5,$7F,$88,$A7,$3F,$D3,$7F,$75,$87,$CF,$7F
	.dc.b	$B6,$35,$DF,$3F,$FB,$D6,$B9,$57,$19,$47,$E5,$9D,$9F,$C1,$5D,$CF
	.dc.b	$1F,$03,$C7,$E0,$79,$3C,$7B,$7F,$DE,$EF,$96,$1D,$5F,$DD,$C6,$78
	.dc.b	$3E,$1F,$34,$8C,$1C,$78,$3D,$F3,$23,$07,$5F,$1F,$E0,$BF,$EF,$DB
	.dc.b	$FE,$4F,$87,$FA,$F5,$B1,$99,$DB,$FC,$7C,$F6,$7A,$0F,$AB,$E0,$D8
	.dc.b	$E8,$D7,$81,$F1,$9C,$38,$1F,$49,$FE,$88,$69,$FE,$8A,$70,$7F,$A3
	.dc.b	$3E,$4B,$EC,$63,$F4,$67,$FD,$27,$C1,$FE,$23,$7F,$9A,$57,$F3,$5F
	.dc.b	$AB,$72,$F8,$B0,$E3,$7E,$4E,$5B,$87,$B5,$AF,$F3,$D9,$AF,$FA,$5A
	.dc.b	$7B,$72,$E0,$DF,$2B,$E7,$31,$3E,$3F,$97,$E2,$62,$FF,$0A,$7F,$49
	.dc.b	$7A,$FC,$BC,$9E,$07,$AD,$F1,$ED,$FC,$F2,$3F,$C8,$0D,$AE,$78,$7F
	.dc.b	$2A,$9C,$F3,$70,$E4,$91,$83,$8B,$EE,$FA,$BF,$93,$7E,$46,$E1,$F9
	.dc.b	$DC,$1D,$FB,$DB,$86,$F3,$2F,$F2,$96,$1C,$1F,$C3,$3F,$1F,$1E,$9E
	.dc.b	$37,$73,$C1,$E3,$DA,$E3,$E7,$F8,$CC,$57,$8D,$F3,$7A,$D2,$7B,$FC
	.dc.b	$75,$AF,$47,$AD,$EB,$31,$BF,$ED,$73,$FE,$DF,$7E,$6F,$8D,$F8,$07
	.dc.b	$C1,$FA,$9B,$1F,$E6,$BF,$98,$FB,$4D,$E7,$FC,$1C,$7F,$86,$23,$06
	.dc.b	$FC,$8C,$9A,$F1,$6C,$7F,$9D,$7E,$AD,$F3,$BF,$8C,$6C,$67,$C0,$C4
	.dc.b	$E7,$A2,$4F,$1F,$76,$F0,$FE,$23,$EA,$DD,$DF,$96,$6D,$7F,$A0,$D7
	.dc.b	$7E,$67,$5C,$12,$B5,$C9,$70,$FC,$9F,$D5,$FF,$BA,$E7,$FD,$18,$D8
	.dc.b	$CF,$6F,$E9,$23,$1C,$DF,$C8,$6B,$FC,$62,$7F,$A4,$CB,$5F,$9E,$7F
	.dc.b	$C6,$66,$77,$FD,$6F,$D5,$FF,$8A,$63,$FD,$34,$D9,$E1,$EE,$69,$DE
	.dc.b	$B2,$B7,$DD,$67,$F5,$77,$F7,$9B,$A7,$6E,$BE,$EF,$EF,$3C,$57,$B5
	.dc.b	$4C,$7C,$F5,$CB,$FD,$59,$2F,$33,$81,$C7,$D3,$B3,$FE,$B1,$FA,$BF
	.dc.b	$D9,$70,$FF,$43,$6C,$7F,$27,$7F,$C6,$2E,$FC,$7F,$E1,$B1,$FE,$AA
	.dc.b	$8E,$5F,$0F,$8F,$73,$2E,$B8,$E4,$F7,$F6,$EF,$0F,$F0,$AF,$27,$FC
	.dc.b	$63,$3F,$EC,$36,$C6,$7C,$9F,$E3,$71,$8D,$3F,$AC,$7D,$5F,$FB,$3E
	.dc.b	$FF,$D9,$4D,$BF,$E2,$AF,$F8,$5E,$33,$F0,$5F,$C9,$3E,$AD,$F0,$D4
	.dc.b	$FE,$4F,$F5,$7B,$C3,$F8,$EC,$EA,$FF,$DA,$CB,$C2,$E4,$71,$8C,$BD
	.dc.b	$7C,$39,$18,$38,$BE,$1E,$19,$1F,$EE,$1C,$3F,$CD,$2F,$68,$F3,$70
	.dc.b	$C8,$C1,$E2,$7F,$9E,$FD,$5F,$E3,$F6,$FE,$F4,$D9,$FE,$BD,$6E,$8D
	.dc.b	$E9,$C3,$82,$57,$FB,$C6,$F8,$4F,$FA,$0D,$FE,$44,$DF,$0F,$B9,$F1
	.dc.b	$BA,$AD,$71,$F4,$7C,$66,$2B,$FA,$2D,$A3,$CF,$E9,$4A,$E0,$78,$5F
	.dc.b	$5F,$A3,$23,$E2,$7C,$95,$C3,$9F,$3F,$C0,$31,$BF,$C0,$7E,$1F,$81
	.dc.b	$2F,$E6,$7C,$6D,$BF,$1F,$83,$F2,$0C,$61,$F2,$1F,$97,$F3,$52,$FE
	.dc.b	$A7,$C1,$5B,$F5,$FB,$FF,$40,$C6,$1F,$41,$FA,$7F,$57,$2F,$E6,$45
	.dc.b	$AF,$EC,$FF,$51,$F9,$5B,$FF,$18,$36,$BF,$C2,$9F,$56,$F9,$6F,$EF
	.dc.b	$33,$9B,$FF,$5B,$3F,$86,$79,$FC,$76,$BC,$1D,$BC,$3E,$4A,$63,$FE
	.dc.b	$BD,$CB,$FC,$7D,$F5,$6F,$AB,$CE,$3E,$4F,$6B,$B9,$8A,$F2,$BE,$2E
	.dc.b	$39,$87,$E5,$FD,$9B,$7E,$CE,$2F,$39,$8C,$3D,$AF,$AB,$E2,$A5,$F8
	.dc.b	$CF,$F1,$96,$FF,$E8,$57,$17,$EE,$9F,$83,$F1,$52,$73,$2D,$5D,$59
	.dc.b	$8F,$9B,$F5,$7F,$20,$C6,$7F,$39,$F9,$7C,$73,$0E,$7E,$BB,$C6,$C6
	.dc.b	$1F,$E1,$CE,$47,$BB,$FE,$2A,$6D,$F8,$B7,$FA,$CF,$92,$33,$5F,$E2
	.dc.b	$4F,$AB,$FE,$52,$FF,$BB,$5A,$BE,$5F,$FB,$AB,$BF,$4B,$FC,$6F,$3F
	.dc.b	$E9,$3B,$F7,$B6,$75,$18,$F8,$4D,$FB,$52,$3E,$2E,$0D,$F1,$7E,$3F
	.dc.b	$7B,$A9,$8E,$1F,$F4,$06,$5F,$DE,$A7,$D1,$BF,$88,$FF,$24,$FD,$47
	.dc.b	$B2,$B7,$1E,$6F,$3A,$67,$E6,$8B,$7F,$E4,$4F,$73,$FC,$66,$FF,$9A
	.dc.b	$1A,$FF,$9C,$5A,$8F,$EF,$FE,$6F,$F9,$8D,$C4,$7F,$12,$58,$FF,$72
	.dc.b	$7B,$4F,$9B,$E2,$3F,$CB,$F5,$FE,$37,$CB,$FD,$38,$DC,$7F,$D0,$18
	.dc.b	$7F,$7C,$9C,$FB,$BC,$0F,$06,$3F,$E1,$06,$DF,$F9,$06,$3D,$51,$8F
	.dc.b	$26,$61,$FE,$81,$9F,$F0,$BC,$61,$FF,$42,$52,$3F,$CE,$7F,$57,$E9
	.dc.b	$C3,$CB,$18,$CB,$D3,$F0,$F3,$1F,$CF,$FD,$5F,$F8,$89,$FD,$0E,$3F
	.dc.b	$D8,$5F,$95,$3B,$CB,$FC,$E7,$5B,$7E,$94,$67,$1F,$86,$FF,$A1,$27
	.dc.b	$51,$8A,$EB,$3B,$7F,$D4,$D3,$A7,$B9,$A2,$57,$82,$F8,$3F,$62,$AA
	.dc.b	$AE,$4E,$6B,$8D,$F8,$63,$94,$60,$EE,$FA,$9F,$87,$7D,$0F,$85,$1C
	.dc.b	$CE,$26,$C9,$3D,$4F,$F6,$F5,$78,$A3,$FC,$9B,$7C,$CF,$89,$56,$FC
	.dc.b	$53,$F9,$5A,$E1,$3E,$08,$E3,$3F,$81,$8F,$F0,$BC,$7E,$7E,$BF,$12
	.dc.b	$49,$D3,$63,$F0,$AD,$BF,$CB,$FC,$FF,$B4,$95,$E2,$AF,$F3,$1C,$61
	.dc.b	$F6,$C7,$EA,$E3,$13,$EF,$8C,$1C,$6D,$3A,$8F,$9B,$3E,$07,$51,$9A
	.dc.b	$F9,$93,$97,$B9,$FB,$48,$D3,$F0,$8F,$E0,$0E,$2B,$E3,$1E,$0A,$EA
	.dc.b	$49,$E6,$7B,$1C,$57,$A6,$3B,$47,$E8,$BD,$3F,$DA,$E3,$FE,$86,$9F
	.dc.b	$F7,$83,$5F,$F7,$AB,$5F,$F7,$F7,$D5,$FF,$88,$7B,$3F,$6C,$D8,$FF
	.dc.b	$7C,$2F,$9F,$71,$6B,$D6,$FE,$83,$5F,$EF,$DF,$AB,$7D,$07,$EF,$E1
	.dc.b	$18,$7F,$81,$35,$E1,$1B,$CB,$FC,$45,$1F,$C1,$DA,$FA,$BD,$6D,$1B
	.dc.b	$F5,$A4,$6D,$AF,$8F,$E4,$FC,$58,$1A,$48,$F1,$FB,$9F,$CB,$5F,$F7
	.dc.b	$BF,$4F,$F9,$65,$FE,$D6,$D7,$FD,$F7,$57,$89,$C5,$C6,$1E,$E7,$39
	.dc.b	$CD,$7F,$39,$9E,$15,$BC,$34,$F1,$BB,$5B,$F8,$0A,$FF,$3C,$CA,$E5
	.dc.b	$EB,$8F,$EF,$F8,$AB,$6B,$E0,$6C,$49,$DC,$74,$FA,$B7,$6F,$4C,$A5
	.dc.b	$7F,$82,$A7,$2D,$E8,$E1,$52,$30,$7F,$84,$3E,$A5,$F5,$7E,$8F,$6F
	.dc.b	$F9,$E4,$FF,$85,$1A,$B8,$7E,$2A,$AE,$2F,$3F,$3B,$E0,$C6,$23,$87
	.dc.b	$0F,$D7,$8C,$57,$27,$FC,$3F,$3E,$3F,$91,$F9,$3F,$57,$F5,$23,$FC
	.dc.b	$4A,$D7,$99,$FD,$B6,$53,$E0,$F7,$FC,$2C,$9F,$74,$79,$3E,$47,$EA
	.dc.b	$FD,$5F,$E5,$B1,$F5,$B7,$2F,$34,$FA,$B4,$E1,$87,$C7,$8C,$3F,$27
	.dc.b	$DF,$AF,$90,$F8,$93,$F0,$A3,$F0,$7F,$53,$F2,$31,$F9,$3F,$4F,$FE
	.dc.b	$99,$BF,$F2,$0B,$5E,$53,$BC,$7F,$3C,$BE,$03,$9C,$49,$FF,$24,$CE
	.dc.b	$5B,$F0,$DF,$92,$FA,$A7,$D5,$D3,$C7,$1E,$4F,$83,$B5,$C0,$E6,$FA
	.dc.b	$3E,$AB,$F3,$7F,$56,$F4,$A9,$9A,$F0,$FC,$1D,$7D,$35,$F0,$23,$E0
	.dc.b	$FC,$15,$9D,$9E,$87,$E2,$FC,$8F,$D4,$F4,$24,$FB,$23,$E4,$FC,$16
	.dc.b	$A3,$07,$C0,$7E,$6F,$C2,$7E,$C3,$8D,$F1,$54,$CD,$7C,$6F,$83,$AF
	.dc.b	$CE,$AF,$AB,$1F,$B8,$F4,$7F,$8F,$73,$FE,$7B,$BD,$F0,$1F,$94,$35
	.dc.b	$E8,$6F,$CC,$B9,$7F,$9E,$B0,$E3,$D5,$FC,$55,$FE,$01,$1F,$85,$5F
	.dc.b	$3D,$C6,$38,$57,$86,$3C,$5F,$3D,$FA,$2F,$AB,$95,$D6,$3C,$DE,$D6
	.dc.b	$A3,$07,$85,$F5,$7A,$3F,$BB,$F6,$A4,$F9,$63,$DD,$E7,$6A,$30,$7A
	.dc.b	$5F,$85,$EE,$FF,$1A,$F8,$17,$5D,$33,$5E,$FE,$DA,$FB,$6B,$E5,$47
	.dc.b	$CB,$FA,$4B,$3B,$3E,$33,$FC,$9B,$EA,$DE,$0D,$F0,$A5,$7F,$A9,$27
	.dc.b	$2D,$F9,$DE,$1F,$13,$23,$07,$FA,$AB,$EA,$DF,$0F,$FC,$C9,$71,$53
	.dc.b	$11,$9E,$1D,$7C,$AC,$BE,$51,$FC,$EB,$EA,$DE,$BD,$F5,$E5,$7F,$AE
	.dc.b	$27,$2D,$F9,$3E,$1E,$A4,$8C,$1F,$EB,$EF,$01,$F1,$9F,$D1,$9F,$DC
	.dc.b	$47,$F0,$96,$F8,$7F,$CB,$9D,$C7,$1F,$27,$87,$CC,$C8,$C1,$FE,$CA
	.dc.b	$FA,$BF,$CB,$B7,$C1,$FE,$A7,$1F,$90,$8D,$5E,$38,$B3,$E0,$49,$E5
	.dc.b	$1C,$3D,$DE,$3E,$F4,$9E,$D1,$FD,$7B,$EA,$DE,$1D,$F8,$A9,$5F,$ED
	.dc.b	$C9,$CB,$FE,$DE,$FA,$B6,$DF,$D9,$97,$C9,$D3,$11,$FD,$AB,$BF,$D1
	.dc.b	$5F,$CB,$5F,$D5,$F1,$7F,$52,$6F,$95,$FE,$03,$77,$19,$F1,$39,$7D
	.dc.b	$5C,$BF,$F7,$7D,$7F,$B9,$5B,$BF,$7F,$12,$46,$0F,$F7,$AF,$D5,$FF
	.dc.b	$62,$C7,$FB,$0B,$FD,$62,$3F,$A0,$36,$33,$BF,$F8,$0B,$EA,$FF,$9D
	.dc.b	$57,$FB,$0E,$BF,$99,$BD,$3D,$9D,$6B,$BD,$F8,$7A,$F2,$BF,$D9,$59
	.dc.b	$7F,$70,$C7,$E3,$C6,$2F,$FC,$45,$0E,$AF,$07,$0C,$AD,$1F,$ED,$39
	.dc.b	$D3,$80,$BD,$6F,$45,$49,$D4,$E5,$BE,$3F,$B7,$C7,$91,$83,$7E,$A7
	.dc.b	$C1,$99,$18,$3C,$1F,$54,$F9,$5F,$F1,$3F,$D5,$BE,$02,$9F,$E2,$A9
	.dc.b	$D5,$70,$EF,$C5,$C8,$C1,$EB,$F8,$BF,$F1,$0B,$FD,$CF,$D3,$65,$FE
	.dc.b	$34,$6C,$6F,$C9,$FF,$1B,$D7,$EE,$A3,$FB,$D3,$CB,$8F,$FC,$3E,$DE
	.dc.b	$3F,$F2,$83,$B8,$FF,$11,$FD,$5F,$E4,$AB,$FE,$00,$60,$E5,$F0,$1F
	.dc.b	$E4,$49,$D4,$9F,$CB,$E1,$FE,$5B,$72,$4F,$1B,$87,$8B,$C5,$95,$F6
	.dc.b	$67,$F7,$EF,$F8,$45,$F8,$EF,$C1,$E5,$F8,$07,$87,$6F,$16,$F6,$79
	.dc.b	$46,$1E,$9E,$2F,$F8,$FD,$FF,$37,$03,$D0,$E5,$CD,$FF,$2E,$D7,$71
	.dc.b	$B8,$C3,$87,$F9,$AF,$D5,$EA,$3E,$17,$9E,$8C,$4E,$A7,$84,$E5,$FF
	.dc.b	$4B,$7D,$27,$FA,$5D,$34,$8F,$5F,$92,$FF,$3D,$E1,$FE,$58,$8F,$F5
	.dc.b	$1E,$FF,$A1,$7D,$95,$ED,$FA,$B7,$1F,$C0,$C3,$8D,$FC,$2E,$4B,$87
	.dc.b	$FA,$45,$AF,$C5,$9A,$FF,$9D,$A7,$B7,$A1,$83,$7C,$AF,$9D,$C4,$F8
	.dc.b	$FE,$5F,$89,$8B,$FF,$26,$1F,$EC,$5D,$7D,$5F,$27,$D8,$DD,$7A,$DF
	.dc.b	$1C,$7F,$86,$23,$F7,$CD,$9F,$D2,$78,$5A,$E8,$DF,$2C,$BD,$C5,$AF
	.dc.b	$F2,$E7,$D5,$FC,$86,$ED,$EA,$F0,$3E,$0C,$E7,$2F,$F4,$2B,$F0,$6B
	.dc.b	$8F,$06,$AF,$6F,$0F,$1A,$65,$FC,$4E,$DF,$21,$8C,$3C,$3E,$7F,$63
	.dc.b	$19,$76,$7C,$3E,$9C,$C3,$97,$C7,$DB,$F4,$FA,$BE,$63,$18,$79,$9F
	.dc.b	$4F,$C7,$4B,$F1,$7D,$4D,$BF,$F3,$E7,$CC,$FD,$79,$FE,$50,$D8,$CF
	.dc.b	$BD,$F8,$28,$C6,$3F,$EA,$65,$E5,$64,$D7,$8F,$70,$FE,$65,$F5,$6F
	.dc.b	$9D,$FC,$5B,$63,$3E,$16,$27,$3D,$D2,$78,$FB,$B7,$5F,$F4,$97,$D5
	.dc.b	$FE,$1E,$7F,$DC,$CD,$8F,$DD,$3F,$E1,$C8,$FE,$84,$E6,$0B,$7F,$CE
	.dc.b	$B6,$77,$C3,$89,$FD,$0F,$D5,$FF,$2E,$9F,$E9,$8D,$BF,$F5,$22,$EB
	.dc.b	$DE,$DF,$A8,$AF,$F1,$B9,$FD,$4C,$B5,$FE,$AB,$7F,$C7,$26,$75,$FD
	.dc.b	$7F,$D5,$FF,$13,$8F,$EB,$4D,$9E,$1E,$EE,$9D,$F8,$92,$B7,$DB,$67
	.dc.b	$FD,$71,$6D,$7A,$9D,$BA,$6D,$FF,$67,$F1,$DE,$CF,$F0,$06,$C6,$7E
	.dc.b	$77,$13,$9F,$3A,$4F,$1F,$5A,$E7,$FE,$C7,$FA,$BF,$DA,$76,$7F,$BB
	.dc.b	$1B,$5F,$CB,$E5,$1F,$E8,$88,$FE,$F0,$EA,$3F,$9A,$3F,$EB,$58,$CF
	.dc.b	$C3,$7F,$14,$87,$BD,$63,$1C,$52,$B9,$FB,$B7,$0F,$F6,$D7,$8F,$FE
	.dc.b	$3C,$9F,$F0,$33,$63,$3E,$3F,$F2,$08,$C6,$6A,$F4,$7D,$5F,$FB,$5D
	.dc.b	$FF,$07,$36,$FF,$DC,$CE,$E3,$3F,$2F,$FC,$AB,$EA,$DF,$0F,$4F,$E5
	.dc.b	$BF,$57,$BC,$3F,$DD,$D3,$AB,$FF,$0D,$2F,$07,$91,$C6,$32,$F5,$F0
	.dc.b	$A4,$60,$E2,$F8,$78,$64,$7F,$89,$70,$FF,$3B,$BD,$A3,$CD,$C3,$23
	.dc.b	$07,$89,$FE,$87,$F5,$7F,$93,$DB,$E2,$67,$FD,$A6,$31,$5F,$77,$07
	.dc.b	$3F,$0A,$E3,$C1,$2B,$FC,$6D,$7C,$27,$94,$F6,$F4,$A5,$79,$FA,$2D
	.dc.b	$32,$F0,$AC,$1E,$97,$98,$DC,$EA,$33,$7E,$0F,$53,$D6,$6D,$C1,$E4
	.dc.b	$F3,$FD,$E7,$D6,$F9,$4B,$1E,$8E,$7F,$86,$63,$2F,$24,$7E,$77,$04
	.dc.b	$FC,$8E,$E9,$3F,$47,$E3,$6D,$F0,$3E,$57,$81,$F3,$8C,$E9,$E9,$7E
	.dc.b	$8F,$CD,$CB,$FB,$5E,$F5,$BE,$47,$C3,$F8,$0F,$AE,$67,$0F,$84,$FD
	.dc.b	$AF,$AB,$97,$F4,$A2,$D7,$2F,$A8,$FC,$E5,$FC,$E6,$D7,$EE,$7E,$AD
	.dc.b	$F3,$1F,$E0,$69,$CD,$FF,$8A,$DF,$C6,$BC,$FE,$3F,$5E,$0E,$DE,$1E
	.dc.b	$B4,$C7,$FC,$F1,$F5,$4F,$AB,$C6,$1E,$3F,$80,$EC,$63,$5F,$23,$E1
	.dc.b	$E3,$98,$7E,$77,$D8,$B7,$EB,$E2,$F3,$18,$C3,$D8,$FA,$7E,$02,$5F
	.dc.b	$FA,$27,$7F,$F2,$F3,$63,$DD,$F1,$52,$3C,$FD,$77,$1F,$91,$F5,$7F
	.dc.b	$0C,$C6,$5F,$25,$F8,$DC,$72,$FF,$D2,$DE,$B7,$F1,$A7,$C1,$1F,$E3
	.dc.b	$0E,$37,$B9,$4E,$18,$FD,$E8,$CD,$7F,$A7,$FE,$AF,$F9,$44,$9B,$F2
	.dc.b	$FF,$DE,$27,$F9,$84,$68,$FF,$4B,$3C,$7E,$13,$FC,$7F,$19,$6F,$B3
	.dc.b	$27,$FD,$59,$4D,$31,$87,$E4,$30,$FE,$F8,$DF,$7B,$3B,$7E,$DA,$72
	.dc.b	$DF,$33,$B9,$FE,$4E,$FA,$8F,$5D,$F3,$8F,$1F,$9B,$33,$F7,$45,$9F
	.dc.b	$CF,$FB,$7F,$E3,$59,$FE,$99,$7F,$DB,$27,$3E,$67,$5F,$89,$95,$FE
	.dc.b	$C5,$AD,$65,$FC,$2B,$0E,$58,$7F,$A0,$E7,$35,$FD,$5E,$7F,$C8,$15
	.dc.b	$BF,$73,$D1,$C1,$FE,$66,$9C,$D6,$E7,$11,$96,$C7,$FB,$56,$3F,$82
	.dc.b	$BF,$ED,$7F,$AB,$EF,$8F,$FA,$4E,$72,$FE,$F2,$32,$9F,$73,$FE,$9A
	.dc.b	$6B,$FD,$B5,$AF,$FA,$75,$AA,$1F,$41,$E9,$3D,$5F,$0D,$6F,$CD,$16
	.dc.b	$BE,$7C,$DC,$BE,$44,$DA,$F8,$D3,$6B,$D5,$35,$F0,$45,$BF,$0D,$76
	.dc.b	$3E,$24,$66,$B4,$D8,$D5,$78,$9B,$7E,$26,$9E,$22,$C7,$89,$AF,$89
	.dc.b	$B7,$E2,$8F,$C5,$DF,$C2,$9C,$BE,$26,$9C,$2F,$C9,$3F,$EC,$4A,$E4
	.dc.b	$FF,$B3,$1A,$FE,$26,$BF,$D6,$CF,$8C,$E0,$E1,$FA,$8D,$8E,$31,$BB
	.dc.b	$FE,$BF,$F5,$5F,$C9,$70,$F5,$B6,$7D,$31,$FE,$DB,$9D,$F8,$5F,$46
	.dc.b	$B5,$1F,$1E,$BF,$A3,$C6,$2B,$CC,$71,$3F,$C6,$CE,$63,$C5,$18,$AE
	.dc.b	$84,$7C,$6D,$8F,$19,$D9,$C5,$78,$4F,$2C,$F0,$24,$F0,$8C,$B6,$3C
	.dc.b	$06,$A3,$93,$88,$D1,$FE,$FB,$AF,$43,$F3,$23,$D8,$E6,$BC,$71,$F0
	.dc.b	$23,$CC,$E9,$7D,$5F,$F8,$6B,$A7,$FD,$F8,$D8,$FF,$2D,$03,$7F,$3E
	.dc.b	$E2,$DF,$AB,$FC,$EA,$76,$32,$7D,$4E,$23,$D0,$D7,$4D,$8D,$C6,$FE
	.dc.b	$AD,$F4,$5F,$E6,$BD,$BF,$C4,$71,$89,$F5,$35,$FB,$FE,$B6,$F6,$FB
	.dc.b	$AD,$8F,$90,$D7,$CC,$DC,$3F,$DB,$31,$F1,$57,$AD,$92,$47,$83,$E8
	.dc.b	$B9,$60,$69,$23,$CB,$F2,$9A,$F7,$7C,$1D,$5F,$E5,$F9,$FE,$6D,$19
	.dc.b	$7D,$7E,$ED,$7F,$C3,$F8,$6D,$B1,$E0,$F8,$1E,$78,$C9,$F8,$71,$E4
	.dc.b	$F8,$2D,$46,$0F,$8E,$FC,$1F,$AB,$76,$F9,$30,$95,$F8,$59,$CB,$7A
	.dc.b	$B8,$54,$8C,$1F,$8B,$FA,$BF,$B2,$FF,$8D,$BD,$9F,$F2,$2B,$F8,$F6
	.dc.b	$BF,$E3,$A8,$DF,$27,$F9,$79,$FA,$31,$9E,$6F,$AD,$1A,$E5,$FF,$56
	.dc.b	$3A,$AE,$1F,$21,$F6,$FE,$95,$AD,$38,$C7,$2F,$76,$E5,$E7,$8C,$3E
	.dc.b	$9F,$91,$FC,$27,$D5,$FF,$9B,$A3,$F4,$CD,$7D,$11,$E4,$D7,$CA,$C8
	.dc.b	$F8,$51,$E8,$F7,$75,$18,$3E,$33,$F9,$1F,$AB,$FF,$3E,$E9,$FE,$7E
	.dc.b	$FA,$8F,$F4,$5D,$7E,$D5,$AF,$F9,$82,$77,$BF,$FB,$35,$7B,$DC,$E2
	.dc.b	$4F,$EF,$67,$2D,$F8,$5F,$D0,$7D,$5B,$E7,$E9,$E3,$8F,$27,$C0,$DA
	.dc.b	$E0,$73,$7D,$1F,$55,$FA,$9F,$AB,$7A,$54,$CD,$78,$7E,$06,$BE,$9A
	.dc.b	$F8,$11,$F0,$7E,$02,$CE,$CF,$43,$F1,$7E,$47,$E9,$FA,$12,$7D,$91
	.dc.b	$F2,$7D,$FD,$46,$0F,$80,$FC,$DF,$82,$FD,$F7,$1B,$E2,$A9,$9A,$F8
	.dc.b	$DE,$FD,$7E,$75,$7D,$58,$FE,$0D,$E8,$FF,$0D,$E7,$FF,$59,$F7,$BE
	.dc.b	$6B,$F2,$06,$BD,$0D,$F9,$37,$2F,$E5,$D8,$71,$EA,$FE,$55,$08,$FE
	.dc.b	$10,$EB,$CD,$E2,$F0,$AF,$0C,$78,$BE,$7B,$F3,$DF,$57,$2B,$AC,$79
	.dc.b	$BD,$BD,$46,$0F,$0B,$EA,$F3,$FF,$90,$78,$4F,$66,$99,$AF,$3F,$91
	.dc.b	$5F,$F5,$7E,$9F,$C4,$78,$7F,$DA,$6F,$EB,$A3,$DB,$EE,$7F,$2B,$F8
	.dc.b	$07,$65,$3E,$4C,$7C,$AE,$EB,$5C,$0F,$8A,$FC,$EF,$A4,$D4,$60,$F9
	.dc.b	$2F,$F3,$7F,$AB,$78,$57,$C3,$95,$FD,$6E,$72,$DF,$99,$E1,$F0,$D2
	.dc.b	$30,$7F,$61,$FA,$B7,$C1,$7F,$43,$5C,$B4,$C4,$67,$97,$5F,$2B,$2F
	.dc.b	$94,$7F,$4B,$FA,$BF,$F2,$84,$7F,$6D,$7F,$94,$BF,$ED,$86,$F2,$7F
	.dc.b	$BF,$1D,$C6,$7E,$47,$97,$A7,$23,$94,$7F,$57,$F0,$1F,$15,$7D,$09
	.dc.b	$5F,$DE,$E7,$2D,$F2,$78,$7C,$CC,$8C,$1F,$E0,$1F,$AB,$FC,$D3,$7C
	.dc.b	$2F,$EC,$90,$6D,$E3,$87,$3E,$0C,$BE,$51,$C3,$DC,$E3,$EE,$C9,$ED
	.dc.b	$1F,$DB,$FE,$AF,$F2,$11,$FE,$11,$7F,$A1,$F1,$3C,$7F,$D2,$C6,$1B
	.dc.b	$E2,$7F,$77,$FA,$B6,$F7,$E4,$26,$3F,$E1,$B9,$CB,$FE,$1D,$EE,$F5
	.dc.b	$37,$C9,$FF,$00,$47,$EC,$21,$9F,$85,$FF,$0F,$47,$07,$39,$7E,$E2
	.dc.b	$1E,$1F,$2F,$AB,$95,$DE,$3A,$EF,$DB,$C3,$93,$E3,$8F,$F0,$87,$D5
	.dc.b	$FB,$63,$FC,$58,$FF,$5B,$C7,$F8,$2B,$63,$3F,$31,$FE,$17,$FA,$BF
	.dc.b	$E8,$31,$C3,$D9,$E6,$66,$F5,$C1,$BF,$5E,$57,$F8,$FE,$D3,$CF,$AF
	.dc.b	$A7,$1C,$AD,$47,$7F,$5B,$51,$83,$93,$E3,$F1,$FC,$9E,$3C,$9E,$D1
	.dc.b	$E7,$F5,$35,$18,$3C,$4F,$F8,$C7,$EA,$9F,$29,$FE,$33,$FA,$B7,$BF
	.dc.b	$4D,$46,$FC,$0B,$87,$F9,$66,$F8,$B6,$3F,$C7,$9F,$15,$FA,$EE,$2F
	.dc.b	$E5,$FE,$A3,$4F,$F2,$03,$6F,$FC,$C5,$5B,$7F,$CB,$CF,$ED,$3B,$3F
	.dc.b	$09,$1F,$E4,$2F,$86,$EA,$76,$F8,$7F,$F2,$57,$D5,$7E,$2E,$B7,$83
	.dc.b	$9F,$B9,$AF,$03,$7E,$3D,$CB,$FC,$E5,$87,$1D,$7F,$A2,$BF,$91,$C7
	.dc.b	$C2,$FE,$DA,$7F,$26,$7F,$9C,$DD,$4F,$0A,$F1,$F8,$FE,$4E,$1F,$7F
	.dc.b	$3C,$C3,$BC,$7A,$38,$75,$18,$3C,$6F,$B3,$C7,$FD,$2C,$7F,$9F,$32
	.dc.b	$FE,$4B,$9F,$F9,$ED,$B1,$99,$DB,$FE,$88,$8F,$E5,$BF,$57,$BF,$44
	.dc.b	$D2,$4F,$FA,$0F,$E9,$3F,$D0,$89,$A4,$7F,$D1,$5F,$23,$FB,$7C,$7E
	.dc.b	$AC,$7F,$A4,$F7,$FF,$4E,$3F,$E9,$7A,$FF,$4C,$7D,$5F,$F9,$EA,$3F
	.dc.b	$B8,$DF,$1A,$FF,$4E,$F2,$D9,$FE,$EA,$D7,$FD,$41,$35,$FF,$33,$CF
	.dc.b	$6E,$5C,$1B,$E5,$7C,$E6,$27,$C7,$F4,$3C,$4C,$5F,$DA,$3F,$D5,$8F
	.dc.b	$5F,$93,$93,$C0,$F5,$BE,$3D,$7F,$C2,$11,$FE,$B9,$6D,$73,$C3,$F7
	.dc.b	$B3,$9E,$5E,$1C,$92,$30,$71,$7D,$DF,$57,$F2,$6F,$C8,$DC,$3F,$37
	.dc.b	$E0,$6B,$6D,$EF,$31,$FF,$65,$5E,$DF,$F0,$EB,$F1,$B0,$E7,$E3,$76
	.dc.b	$3B,$FC,$ED,$AE,$3E,$7F,$88,$C5,$78,$9F,$2F,$AD,$27,$BF,$C7,$DA
	.dc.b	$F3,$FA,$FE,$A3,$1B,$7E,$E3,$34,$7B,$BE,$3A,$61,$E8,$FA,$9B,$87
	.dc.b	$D3,$F9,$9F,$B3,$3F,$EE,$56,$C6,$7E,$03,$F0,$91,$8C,$BF,$82,$AF
	.dc.b	$2B,$26,$BC,$7B,$87,$FB,$BF,$EA,$DF,$3D,$F8,$D6,$C6,$7C,$1C,$4E
	.dc.b	$7A,$A4,$F1,$F7,$EE,$BF,$89,$FA,$B7,$77,$E5,$DB,$5F,$EF,$D5,$E0
	.dc.b	$66,$75,$C3,$2B,$5C,$77,$0F,$CA,$7D,$5F,$F2,$79,$FF,$80,$1C,$5F
	.dc.b	$C8,$A3,$13,$AE,$DF,$D3,$46,$27,$F9,$3D,$7F,$8B,$0E,$27,$F9,$80
	.dc.b	$C5,$7E,$85,$FF,$17,$9A,$47,$F8,$0F,$7F,$D9,$7D,$5E,$27,$52,$8C
	.dc.b	$3C,$FD,$ED,$BC,$35,$95,$C2,$BF,$29,$87,$F3,$C7,$E2,$C6,$73,$EE
	.dc.b	$78,$3A,$AC,$78,$3C,$57,$B9,$FC,$15,$B1,$9F,$9D,$C4,$E7,$CC,$93
	.dc.b	$C7,$D5,$BA,$FF,$01,$FA,$BF,$DC,$EB,$FD,$39,$B5,$FE,$86,$7F,$C6
	.dc.b	$4F,$C0,$73,$3F,$19,$FF,$58,$C6,$7E,$23,$F8,$C4,$7F,$85,$A3,$07
	.dc.b	$1E,$0C,$9A,$F7,$2C,$7C,$CF,$1F,$FD,$08,$DF,$13,$F9,$14,$3C,$6B
	.dc.b	$1F,$C9,$7E,$AF,$FD,$A3,$5F,$51,$B9,$FF,$1D,$7F,$C7,$91,$9F,$84
	.dc.b	$FE,$59,$F5,$6F,$88,$A7,$F2,$FF,$AB,$DE,$1F,$C9,$67,$57,$F8,$05
	.dc.b	$E0,$72,$38,$C6,$5E,$BE,$0C,$8C,$1C,$5F,$0F,$C1,$C8,$FC,$5E,$1F
	.dc.b	$E6,$C7,$B4,$79,$B8,$A4,$60,$F1,$3F,$D1,$7E,$AF,$F2,$3C,$9F,$E0
	.dc.b	$66,$CF,$FA,$FA,$DD,$1B,$D3,$87,$14,$AF,$CD,$DF,$09,$FD,$0B,$FC
	.dc.b	$AD,$BE,$17,$73,$E3,$74,$DA,$E3,$E8,$F8,$CC,$57,$F4,$DB,$47,$9F
	.dc.b	$DB,$95,$C0,$F0,$BE,$BF,$3E,$47,$C4,$F9,$5B,$87,$3E,$8F,$80,$63
	.dc.b	$7F,$80,$FC,$3E,$F9,$7F,$33,$E3,$AD,$F8,$FC,$1F,$90,$63,$0F,$90
	.dc.b	$FC,$BF,$91,$97,$F5,$3D,$EB,$7E,$BF,$80,$FA,$06,$30,$FA,$0F,$D3
	.dc.b	$FA,$B9,$7F,$32,$2D,$7F,$6F,$FA,$9F,$CA,$DF,$F8,$F1,$B5,$FE,$F4
	.dc.b	$FA,$B7,$CC,$7F,$80,$67,$37,$FC,$51,$FC,$33,$CF,$E4,$35,$E0,$ED
	.dc.b	$E1,$EB,$4C,$7F,$8B,$FD,$5B,$EA,$31,$87,$8F,$DF,$EC,$63,$5F,$23
	.dc.b	$E1,$E3,$98,$7E,$5B,$D8,$B7,$EB,$E2,$F3,$18,$C3,$D8,$FA,$7E,$2A
	.dc.b	$5F,$F2,$8D,$FF,$CA,$8D,$8F,$77,$C5,$48,$F3,$F5,$5C,$7E,$47,$D5
	.dc.b	$FC,$33,$19,$7C,$97,$E3,$71,$CB,$FE,$6B,$EB,$7F,$11,$BF,$A1,$3F
	.dc.b	$E2,$BE,$37,$B9,$4E,$13,$FE,$23,$8C,$DF,$F3,$EF,$AB,$FE,$3F,$26
	.dc.b	$FC,$C7,$F7,$19,$FE,$8F,$39,$F8,$4F,$F1,$C5,$7F,$A3,$27,$97,$C2
	.dc.b	$62,$72,$EB,$DA,$93,$FD,$33,$06,$B8,$C7,$FC,$6D,$8F,$F7,$66,$FB
	.dc.b	$D9,$DF,$FD,$0B,$39,$6F,$9B,$E2,$3F,$C9,$5F,$51,$EC,$BE,$71,$E6
	.dc.b	$F3,$A6,$7E,$F8,$B3,$FE,$45,$F7,$3F,$AB,$CF,$B9,$B1,$FD,$9A,$FF
	.dc.b	$B5,$E1,$FE,$96,$9C,$CF,$F8,$01,$FF,$2D,$1F,$D7,$9E,$9F,$0B,$88
	.dc.b	$CF,$9B,$30,$FE,$DB,$83,$D7,$C6,$1E,$3F,$33,$99,$8C,$BC,$0F,$6F
	.dc.b	$5E,$61,$FD,$DB,$EA,$7C,$15,$E5,$9F,$1F,$D0,$4A,$FE,$F7,$A7,$F5
	.dc.b	$0C,$7E,$04,$FF,$9C,$7E,$AF,$FC,$0F,$1F,$78,$38,$7F,$02,$D7,$FC
	.dc.b	$12,$D7,$7C,$5C,$CE,$9C,$B7,$0F,$F0,$5D,$7D,$97,$9C,$E9,$CB,$FE
	.dc.b	$83,$CF,$FB,$14,$7B,$72,$C7,$9D,$96,$47,$9F,$1F,$F0,$DD,$7B,$5F
	.dc.b	$53,$5F,$CC,$B5,$DE,$5D,$CE,$6F,$FA,$1E,$11,$99,$ED,$29,$CC,$61
	.dc.b	$B1,$E3,$6B,$E2,$F8,$8F,$67,$9F,$ED,$6D,$7D,$98,$FF,$56,$CF,$A9
	.dc.b	$AF,$76,$D1,$BE,$7F,$C7,$39,$CF,$A5,$F9,$5F,$07,$C0,$97,$E7,$6C
	.dc.b	$FE,$87,$0D,$CF,$E1,$9F,$CD,$9E,$2D,$CD,$24,$7C,$48,$CB,$63,$33
	.dc.b	$F5,$67,$2D,$C3,$F4,$E7,$6A,$FC,$AB,$87,$A3,$E1,$23,$F3,$FE,$1F
	.dc.b	$E1,$91,$C1,$F4,$FD,$1F,$8E,$47,$07,$D7,$F4,$3E,$59,$1C,$1F,$6F
	.dc.b	$D1,$F9,$E4,$70,$7D,$FF,$93,$FA,$64,$70,$7E,$1B,$8D,$F3,$DF,$69
	.dc.b	$B1,$98,$C1,$F3,$76,$91,$F2,$DB,$A7,$E7,$7E,$5B,$FC,$6F,$3F,$B5
	.dc.b	$6C,$7F,$34,$6B,$FE,$7C,$6A,$FA,$BF,$F2,$3C,$FE,$F5,$B1,$FC,$F1
	.dc.b	$AF,$FA,$11,$AF,$1F,$6F,$FA,$5B,$FA,$F8,$FF,$17,$F8,$FF,$A4,$5F
	.dc.b	$3E,$F7,$31,$7C,$2F,$C1,$E5,$FF,$4A,$ED,$65,$E9,$F5,$F0,$DA,$FF
	.dc.b	$48,$E4,$BC,$C7,$97,$E3,$2B,$ED,$8B,$3B,$8E,$33,$AF,$AB,$7C,$AF
	.dc.b	$0E,$C7,$BB,$4D,$46,$FE,$56,$E1,$FE,$A2,$BE,$2D,$8E,$9E,$7F,$8B
	.dc.b	$8D,$E8,$53,$35,$C7,$D0,$AB,$87,$E0,$BF,$EB,$2C,$58,$79,$EB,$D1
	.dc.b	$F0,$7E,$EF,$2B,$FC,$E7,$7F,$D2,$23,$2F,$97,$E0,$EB,$D7,$8F,$FB
	.dc.b	$BE,$3C,$23,$07,$79,$F8,$9E,$5F,$5D,$E4,$FB,$E3,$E4,$7C,$3E,$A3
	.dc.b	$07,$C3,$7E,$8F,$D5,$BC,$FF,$36,$12,$BF,$D9,$33,$96,$F8,$DC,$2A
	.dc.b	$46,$0F,$F6,$7F,$D5,$3E,$AF,$EE,$F8,$DF,$EA,$39,$FF,$69,$B5,$F5
	.dc.b	$89,$E0,$F2,$F6,$3F,$0B,$18,$8D,$B7,$93,$E8,$C7,$19,$DF,$95,$F8
	.dc.b	$CF,$A5,$6B,$4C,$D7,$9B,$CB,$AF,$4A,$EF,$1E,$0F,$2F,$F2,$BF,$57
	.dc.b	$FE,$84,$AF,$F7,$5B,$5F,$1C,$79,$B4,$E1,$1F,$0A,$30,$FC,$4F,$32
	.dc.b	$BE,$A9,$F6,$C7,$E8,$BE,$AF,$D7,$AF,$B3,$EA,$3F,$D4,$77,$FE,$FA
	.dc.b	$6B,$F2,$27,$7D,$1F,$E3,$C5,$E2,$F3,$89,$23,$84,$77,$F8,$AF,$D8
	.dc.b	$FD,$5B,$CF,$A6,$6B,$86,$5F,$3B,$17,$C4,$66,$7C,$58,$7A,$63,$D5
	.dc.b	$F5,$5F,$BC,$FA,$B7,$BB,$4C,$DF,$4D,$7D,$1F,$0F,$88,$F6,$5F,$C3
	.dc.b	$8F,$89,$F0,$D6,$78,$1E,$BC,$D3,$A8,$C8,$87,$CA,$7D,$AF,$66,$4E
	.dc.b	$A7,$2D,$F8,$5F,$A5,$52,$30,$6F,$E2,$3F,$8B,$F2,$3E,$36,$9F,$66
	.dc.b	$3E,$D7,$C3,$DA,$E0,$77,$7F,$8F,$FA,$EF,$AB,$FC,$B7,$9D,$2B,$D3
	.dc.b	$39,$6F,$95,$C2,$A4,$60,$DF,$A3,$D2,$64,$60,$E7,$F4,$1F,$A3,$F7
	.dc.b	$24,$60,$EF,$F4,$1E,$49,$91,$83,$C7,$E8,$7E,$EB,$E0,$64,$60,$F3
	.dc.b	$F8,$FE,$C9,$91,$83,$D7,$F0,$FF,$CF,$BE,$2B,$EB,$C7,$FA,$27,$A3
	.dc.b	$E1,$CF,$C4,$EC,$B3,$C0,$F8,$6F,$C8,$F5,$F5,$18,$3E,$3B,$FD,$2B
	.dc.b	$EA,$FF,$2B,$1F,$82,$5E,$0E,$4D,$78,$3B,$F6,$F8,$7B,$B2,$F8,$47
	.dc.b	$F5,$4F,$AB,$7C,$45,$E3,$95,$F8,$E9,$CB,$78,$38,$7C,$B4,$8C,$1F
	.dc.b	$94,$FA,$B7,$AF,$FD,$81,$FE,$59,$1F,$C8,$9B,$E4,$FF,$1C,$77,$1C
	.dc.b	$7E,$23,$87,$A5,$23,$07,$E8,$3B,$FF,$AC,$B7,$CE,$FE,$DB,$1F,$C1
	.dc.b	$23,$5E,$FE,$3C,$6C,$FC,$D4,$9E,$51,$FD,$D3,$EA,$FF,$AD,$C7,$EB
	.dc.b	$17,$85,$93,$58,$6F,$8B,$87,$81,$2F,$84,$73,$F7,$7A,$7B,$D2,$7A
	.dc.b	$47,$F8,$0B,$EA,$DE,$1D,$F8,$C9,$5F,$BC,$9C,$BF,$BE,$FA,$B6,$FF
	.dc.b	$E0,$A5,$F2,$74,$C4,$7F,$83,$BB,$FD,$35,$FC,$19,$FF,$01,$BF,$D4
	.dc.b	$9B,$CF,$FD,$31,$DC,$67,$C3,$E5,$F5,$72,$39,$47,$5D,$BB,$78,$72
	.dc.b	$7B,$47,$F8,$7B,$EA,$FF,$B0,$63,$FD,$61,$7C,$06,$4D,$6F,$FE,$24
	.dc.b	$FA,$BF,$E7,$B8,$7F,$39,$AF,$E6,$CF,$F1,$E6,$BF,$DA,$A3,$69,$F8
	.dc.b	$6F,$F2,$45,$EB,$75,$3B,$6F,$DF,$A7,$C1,$C1,$2F,$84,$78,$FD,$4D
	.dc.b	$46,$0E,$8F,$9F,$C5,$F4,$78,$D2,$7C,$11,$EB,$F4,$B5,$18,$3C,$8F
	.dc.b	$F9,$0B,$EA,$9F,$13,$FE,$44,$FA,$B7,$BF,$4D,$46,$FB,$AE,$1F,$D0
	.dc.b	$2F,$8B,$63,$FC,$9B,$F1,$7F,$AC,$F7,$7F,$B1,$4F,$F8,$13,$3F,$F2
	.dc.b	$8F,$A6,$77,$F1,$1C,$3C,$BB,$5F,$D3,$30,$E4,$FF,$6B,$7F,$6A,$FF
	.dc.b	$86,$BE,$AF,$F1,$38,$F6,$C7,$E7,$CF,$F5,$46,$BB,$F5,$FF,$CC,$35
	.dc.b	$F8,$EE,$BF,$F0,$DB,$CA,$7F,$BC,$35,$3B,$BF,$5B,$FB,$68,$F1,$78
	.dc.b	$FD,$0E,$BE,$3F,$6E,$0F,$7F,$34,$AF,$EC,$F7,$F2,$1F,$57,$06,$DC
	.dc.b	$D7,$9A,$3C,$FE,$3F,$F9,$86,$7F,$DF,$14,$8C,$47,$CE,$3C,$EE,$1F
	.dc.b	$3B,$5F,$6F,$A2,$FA,$BF,$F7,$AD,$7F,$30,$6C,$66,$7B,$B5,$FF,$7D
	.dc.b	$FD,$2F,$FB,$F1,$FF,$0F,$35,$34,$D7,$C9,$7F,$9E,$37,$FF,$24,$CF
	.dc.b	$FA,$7F,$7F,$F0,$4B,$C8,$7D,$5F,$FA,$39,$FC,$4D,$71,$9F,$77,$2D
	.dc.b	$9F,$C6,$B5,$FF,$18,$4D,$7F,$CD,$D3,$DB,$97,$06,$F9,$1F,$3D,$89
	.dc.b	$F1,$FC,$DF,$13,$17,$FE,$68,$0F,$5F,$9B,$93,$C0,$F5,$BE,$3D,$3F
	.dc.b	$D2,$B1,$FE,$26,$6D,$7F,$9C,$B0,$FE,$53,$39,$E5,$E1,$C9,$23,$07
	.dc.b	$17,$DD,$F5,$7F,$26,$FF,$CF,$6D,$C3,$FC,$EB,$87,$FB,$06,$FF,$C6
	.dc.b	$B3,$97,$F6,$6F,$C4,$AE,$3B,$74,$DB,$AB,$DB,$C6,$97,$CB,$C7,$B7
	.dc.b	$E0,$F3,$FC,$66,$33,$FF,$42,$E5,$FD,$32,$FC,$3E,$A7,$A0,$F4,$FC
	.dc.b	$75,$8F,$2F,$AD,$EC,$31,$8F,$B1,$F7,$FC,$6C,$9F,$5F,$CC,$DA,$FF
	.dc.b	$4D,$FC,$CF,$DA,$9F,$F2,$9B,$63,$3E,$F7,$E1,$63,$1A,$7F,$A6,$97
	.dc.b	$91,$93,$5E,$2D,$C3,$FC,$BB,$F5,$6F,$9D,$FC,$73,$63,$3D,$F8,$9C
	.dc.b	$F3,$C9,$E3,$EF,$DD,$7F,$D6,$3F,$57,$F8,$A9,$FF,$34,$B6,$33,$DF
	.dc.b	$F9,$B8,$C6,$DF,$EB,$05,$C1,$93,$5C,$57,$0F,$F3,$9F,$D5,$FF,$2D
	.dc.b	$6E,$FF,$A5,$8F,$C2,$65,$FE,$C2,$74,$FF,$B1,$27,$FC,$62,$7F,$9F
	.dc.b	$0A,$E7,$C5,$FD,$74,$63,$2F,$F6,$1F,$D5,$FF,$A7,$E3,$FD,$06,$DA
	.dc.b	$E9,$EE,$FE,$D6,$31,$5C,$35,$C1,$9E,$DB,$87,$2D,$38,$1C,$7A,$6C
	.dc.b	$7F,$A4,$3C,$5F,$F2,$7B,$7D,$BF,$E0,$50,$F9,$EB,$18,$F2,$E5,$71
	.dc.b	$F5,$2E,$FF,$EE,$4F,$AB,$FD,$AE,$DF,$ED,$E6,$D7,$F5,$A7,$FC,$66
	.dc.b	$FF,$A8,$1C,$CF,$FA,$FD,$FF,$58,$C6,$7E,$1F,$F8,$A4,$3D,$DB,$18
	.dc.b	$E2,$95,$CF,$DC,$B8,$7F,$BD,$BC,$7F,$D5,$37,$C5,$FE,$3D,$0F,$1E
	.dc.b	$C7,$F2,$1F,$AB,$FF,$69,$E1,$FE,$B4,$6D,$FF,$C0,$05,$1B,$F8,$4F
	.dc.b	$E5,$1F,$56,$F8,$7A,$7F,$2B,$FA,$BD,$D7,$FA,$FA,$74,$DF,$07,$89
	.dc.b	$C1,$C1,$BF,$0B,$A9,$A7,$07,$3E,$1F,$09,$DF,$E1,$EC,$77,$E2,$F2
	.dc.b	$98,$D5,$C9,$EA,$7F,$DD,$D9,$7D,$07,$35,$FE,$10,$FA,$BF,$C7,$B6
	.dc.b	$3D,$9D,$3F,$D1,$E7,$31,$FE,$F7,$7F,$93,$37,$8F,$4E,$DE,$1E,$94
	.dc.b	$BF,$F7,$0E,$FF,$92,$AF,$09,$CE,$7B,$7A,$52,$BF,$DC,$F8,$7E,$F1
	.dc.b	$F3,$47,$8F,$DA,$91,$A8,$B7,$D0,$57,$F9,$6A,$79,$FA,$3F,$5F,$E4
	.dc.b	$BD,$8F,$B7,$A2,$4E,$8F,$0B,$F0,$7C,$09,$1E,$AF,$8D,$B8,$F7,$C5
	.dc.b	$E1,$7C,$73,$3C,$5F,$46,$7C,$71,$F2,$FE,$6A,$61,$F1,$3E,$12,$D7
	.dc.b	$3F,$80,$FA,$06,$30,$E5,$8F,$F9,$42,$FE,$2F,$D2,$FD,$53,$6E,$05
	.dc.b	$FC,$3F,$A8,$FC,$ED,$F8,$DB,$3F,$E5,$0F,$AB,$7C,$B7,$F8,$16,$73
	.dc.b	$7C,$97,$5E,$4D,$78,$17,$0E,$F8,$F5,$F5,$F9,$1F,$E0,$FF,$AB,$7D
	.dc.b	$5E,$2B,$67,$F9,$83,$1F,$E9,$ED,$F7,$F5,$9F,$A5,$F2,$72,$49,$FC
	.dc.b	$5F,$B3,$6B,$DD,$C7,$E9,$31,$A7,$BD,$F6,$7C,$54,$BF,$0D,$7B,$67
	.dc.b	$0F,$C7,$F8,$AF,$84,$7A,$7A,$AD,$70,$96,$8F,$AB,$F9,$06,$A3,$E8
	.dc.b	$3F,$2F,$8E,$57,$1F,$8B,$FE,$3C,$F3,$BF,$A2,$F5,$E4,$7B,$DF,$E3
	.dc.b	$76,$C6,$EF,$13,$FE,$3B,$FA,$BF,$E5,$B9,$1B,$F2,$FF,$32,$3F,$07
	.dc.b	$1A,$3F,$D3,$77,$EB,$6C,$7E,$A5,$D7,$C2,$75,$F8,$09,$5F,$06,$FE
	.dc.b	$03,$C6,$3C,$5F,$01,$23,$07,$57,$CD,$E6,$48,$F9,$5F,$51,$EC,$CB
	.dc.b	$C1,$3E,$BF,$36,$57,$96,$2C,$FE,$F7,$DB,$FF,$1A,$73,$FE,$B5,$B5
	.dc.b	$FE,$AA,$C3,$FB,$83,$FE,$AC,$8C,$D7,$E9,$11,$FE,$63,$2C,$7D,$D7
	.dc.b	$71,$A7,$5E,$54,$9F,$4F,$A9,$6B,$C9,$E5,$73,$31,$8F,$95,$ED,$EA
	.dc.b	$4B,$FC,$57,$D4,$F8,$2F,$D9,$3E,$3F,$9F,$95,$E3,$8B,$3F,$C7,$7E
	.dc.b	$AF,$FC,$58,$DE,$0F,$F3,$E4,$7C,$08,$FE,$02,$EB,$2F,$E2,$4F,$08
	.dc.b	$FF,$51,$35,$FF,$68,$B5,$E3,$3F,$E9,$26,$DF,$E8,$67,$FA,$EC,$78
	.dc.b	$63,$1E,$B7,$FA,$37,$26,$54,$C6,$56,$BE,$0C,$75,$B5,$70,$9F,$F5
	.dc.b	$B4,$7F,$1A,$8C,$38,$6B,$FE,$BB,$73,$97,$F2,$FF,$94,$7C,$F7,$7A
	.dc.b	$FE,$89,$18,$BF,$DD,$36,$3B,$B5,$FF,$63,$35,$FF,$7A,$B5,$F2,$79
	.dc.b	$DE,$F3,$E0,$79,$F6,$BE,$EC,$7B,$71,$FE,$8B,$1E,$CA,$F7,$CE,$49
	.dc.b	$1E,$4F,$57,$94,$FA,$1F,$CB,$7C,$4F,$C7,$E4,$34,$95,$D6,$18,$FE
	.dc.b	$74,$E3,$1F,$9D,$70,$ED,$E8,$47,$04,$BF,$A7,$3D,$23,$11,$FE,$EB
	.dc.b	$8F,$6B,$AC,$3C,$C4,$9F,$81,$C0,$F6,$7F,$00,$D8,$CC,$60,$F9,$1E
	.dc.b	$8C,$8F,$91,$A7,$F6,$1C,$3F,$05,$3F,$89,$D7,$FC,$19,$1F,$63,$DF
	.dc.b	$C1,$2F,$E8,$4B,$CE,$E0,$48,$CB,$E4,$F9,$D9,$1E,$86,$DF,$C2,$F8
	.dc.b	$7F,$A4,$49,$FA,$0F,$D5,$F8,$A9,$1F,$4D,$B3,$F2,$BD,$6F,$F1,$A4
	.dc.b	$FC,$96,$C7,$F1,$76,$BF,$C5,$DA,$BE,$AF,$FB,$9C,$FF,$A2,$63,$FB
	.dc.b	$FC,$61,$FF,$08,$1B,$F1,$F8,$C6,$67,$FC,$0B,$3C,$39,$7F,$A7,$BF
	.dc.b	$B0,$9F,$F4,$27,$8D,$F9,$77,$F5,$7E,$E5,$AD,$D7,$23,$C9,$AF,$4C
	.dc.b	$31,$7F,$3E,$7C,$C6,$8A,$E5,$F3,$AF,$2B,$CB,$D5,$FB,$BF,$56,$DF
	.dc.b	$EC,$9B,$7F,$02,$AB,$F1,$7E,$26,$E1,$CE,$3F,$BC,$35,$FF,$23,$28
	.dc.b	$E3,$F5,$6F,$94,$E7,$CE,$F7,$A9,$A8,$DF,$CA,$5C,$7F,$C0,$97,$C5
	.dc.b	$B1,$D3,$E4,$3C,$BD,$1F,$E2,$A9,$FE,$23,$C9,$97,$8F,$9D,$5F,$F2
	.dc.b	$D6,$0B,$F2,$CF,$9B,$B3,$DF,$E5,$7E,$AE,$FF,$B5,$C6,$5F,$1F,$C1
	.dc.b	$D7,$9C,$7B,$A3,$DF,$E5,$72,$D6,$57,$AA,$3E,$1F,$C2,$EA,$30,$7B
	.dc.b	$9F,$9F,$F5,$6F,$8B,$F1,$E1,$2B,$FC,$4D,$39,$6F,$8B,$C2,$A4,$60
	.dc.b	$FF,$16,$7D,$53,$EA,$FE,$DF,$81,$FE,$B9,$9F,$F1,$83,$5F,$F3,$F4
	.dc.b	$ED,$E3,$EC,$FE,$0A,$31,$1A,$AC,$4E,$7C,$AF,$C3,$FD,$5B,$4A,$66
	.dc.b	$FC,$7E,$5D,$7B,$D7,$58,$ED,$E2,$FE,$43,$EA,$DE,$95,$33,$7E,$CD
	.dc.b	$6B,$E8,$AF,$34,$79,$FC,$BB,$3B,$3E,$1B,$F9,$CF,$A7,$FF,$45,$78
	.dc.b	$BF,$E8,$BF,$A7,$FF,$7C,$57,$F9,$4D,$AF,$FA,$86,$77,$E3,$7E,$F9
	.dc.b	$78,$DC,$E2,$4F,$F9,$6E,$72,$DF,$37,$F5,$7F,$56,$FA,$BA,$78,$E3
	.dc.b	$C9,$E5,$5A,$E0,$73,$7D,$1F,$55,$FB,$4F,$AB,$7B,$74,$CD,$78,$7E
	.dc.b	$1A,$BE,$9A,$F8,$11,$F0,$7E,$12,$CE,$CF,$43,$F1,$7E,$53,$E9,$7B
	.dc.b	$32,$7D,$91,$F2,$7E,$17,$51,$83,$E0,$3F,$37,$E1,$FF,$84,$F1,$BE
	.dc.b	$3E,$99,$AF,$8D,$F0,$B5,$F9,$D5,$F5,$63,$F8,$8F,$BA,$FA,$BF,$C6
	.dc.b	$F9,$D2,$FF,$D1,$13,$96,$F9,$7C,$2A,$46,$0D,$FA,$5D,$26,$46,$0E
	.dc.b	$7F,$41,$F9,$AF,$76,$46,$0E,$FF,$33,$E4,$99,$18,$3C,$7E,$87,$EB
	.dc.b	$BE,$06,$46,$0F,$3F,$91,$EC,$99,$18,$3D,$7F,$0B,$FC,$C3,$E1,$DD
	.dc.b	$14,$F8,$71,$F1,$3A,$AD,$70,$3E,$1B,$F2,$3D,$8D,$46,$0F,$8E,$FF
	.dc.b	$3C,$FA,$BF,$E8,$DE,$2F,$FB,$AD,$78,$19,$35,$E0,$6F,$DC,$E1,$EF
	.dc.b	$4C,$7F,$D7,$17,$FC,$FD,$FF,$5D,$7D,$5B,$E2,$7F,$A4,$AE,$4A,$62
	.dc.b	$33,$C3,$AF,$87,$97,$CA,$3F,$A8,$FD,$5F,$FA,$32,$3F,$D8,$CF,$F2
	.dc.b	$15,$B3,$CA,$FE,$E3,$39,$8C,$4F,$FB,$E2,$39,$3C,$27,$7F,$11,$DB
	.dc.b	$D2,$93,$DA,$3F,$B0,$F7,$FF,$4D,$8E,$00,$FE,$95,$E6,$E8,$DF,$C0
	.dc.b	$70,$F1,$F8,$FC,$D4,$AF,$F6,$F5,$A7,$FB,$6F,$D5,$FF,$53,$9F,$F7
	.dc.b	$12,$F0,$B2,$6A,$B7,$C5,$C3,$C0,$97,$C2,$39,$FB,$DD,$3D,$F9,$3D
	.dc.b	$23,$FB,$DF,$D5,$BC,$3B,$F3,$12,$BF,$DE,$53,$97,$FD,$E9,$F5,$6D
	.dc.b	$FF,$C0,$6B,$E4,$E9,$88,$FF,$03,$F8,$1E,$9A,$FF,$7E,$3F,$DE,$5F
	.dc.b	$92,$7E,$3B,$C8,$DF,$07,$0F,$13,$8F,$D5,$CA,$E3,$1D,$37,$EB,$E2
	.dc.b	$49,$EB,$1F,$E1,$6F,$AB,$FE,$B5,$DB,$F8,$09,$43,$E0,$34,$6F,$87
	.dc.b	$FC,$3B,$F5,$7F,$CF,$2B,$EE,$57,$F3,$57,$EF,$5F,$38,$C4,$7A,$A7
	.dc.b	$82,$7E,$23,$C3,$BB,$FC,$87,$0F,$57,$B9,$E0,$E2,$F0,$F5,$F8,$B8
	.dc.b	$E6,$1C,$4F,$C7,$5F,$E1,$DC,$1D,$FD,$4E,$2E,$23,$4F,$96,$3D,$BE
	.dc.b	$47,$BB,$C8,$93,$B8,$F8,$3E,$96,$A3,$07,$37,$FC,$83,$F5,$4F,$89
	.dc.b	$FF,$21,$FD,$5B,$DE,$A6,$A3,$7E,$0D,$C3,$CF,$7C,$5B,$1F,$E4,$CF
	.dc.b	$8C,$FD,$B7,$17,$F9,$A3,$D3,$FE,$15,$5F,$86,$6F,$2F,$DB,$AD,$CF
	.dc.b	$EC,$5F,$DD,$C6,$7E,$23,$A1,$D7,$E2,$7F,$CB,$3F,$57,$F9,$19,$FC
	.dc.b	$26,$1F,$4F,$0F,$96,$D7,$F2,$B3,$BC,$7F,$CF,$6F,$E6,$E3,$3E,$07
	.dc.b	$43,$AF,$81,$DB,$C7,$B3,$F5,$AB,$86,$3F,$93,$7F,$7D,$1D,$B7,$FF
	.dc.b	$36,$E5,$F6,$2B,$CB,$D1,$E8,$9F,$4E,$F6,$FA,$46,$1E,$DE,$3F,$F2
	.dc.b	$27,$FD,$FB,$8F,$F2,$FC,$3F,$DC,$8D,$8C,$CE,$DF,$F4,$74,$7F,$37
	.dc.b	$FA,$BD,$FA,$26,$92,$7F,$DD,$C3,$E9,$6B,$4C,$B8,$37,$F2,$7F,$CC
	.dc.b	$B1,$FF,$54,$B7,$87,$FD,$3D,$5C,$7E,$AF,$FD,$08,$FF,$AA,$A7,$4F
	.dc.b	$FA,$B3,$92,$BF,$CC,$5A,$A2,$C6,$39,$25,$FD,$EF,$9D,$B7,$FC,$48
	.dc.b	$7C,$D7,$23,$39,$7F,$AF,$9E,$BF,$17,$2F,$F6,$35,$C1,$B1,$E4,$AF
	.dc.b	$F4,$64,$7E,$D9,$B3,$F9,$BF,$0B,$5C,$DB,$E3,$97,$B8,$B5,$FC,$9B
	.dc.b	$EA,$FD,$93,$FC,$01,$B1,$FA,$5F,$03,$E1,$C6,$31,$FE,$40,$FF,$A2
	.dc.b	$6B,$86,$DA,$DF,$6F,$0F,$1A,$4F,$5F,$1E,$E5,$E1,$F3,$FA,$98,$CB
	.dc.b	$AB,$E0,$F5,$65,$F9,$7E,$3A,$DF,$A7,$D6,$F2,$98,$C3,$CA,$FA,$3E
	.dc.b	$36,$5F,$B7,$E6,$6D,$FF,$46,$F9,$97,$3F,$E0,$1B,$5F,$C8,$97,$BF
	.dc.b	$99,$D7,$93,$2B,$5E,$55,$C3,$FA,$8F,$D5,$BE,$73,$F1,$6D,$AF,$E5
	.dc.b	$AF,$F8,$65,$B1,$AF,$1A,$46,$BD,$EB,$7F,$D7,$3E,$AF,$F0,$8D,$EF
	.dc.b	$FC,$BC,$3C,$0B,$18,$E0,$98,$7F,$3C,$A7,$16,$30,$FE,$CF,$F5,$7F
	.dc.b	$EE,$BA,$FE,$82,$D8,$CF,$6F,$E9,$23,$1C,$BF,$D8,$6B,$D5,$5F,$DC
	.dc.b	$1B,$19,$F1,$FF,$57,$18,$3F,$C3,$92,$FA,$BF,$F0,$F4,$7F,$50,$6C
	.dc.b	$73,$F7,$7F,$67,$18,$BF,$EE,$0F,$BE,$B8,$6D,$AE,$ED,$BC,$34,$93
	.dc.b	$D7,$AE,$E5,$FE,$06,$F1,$BF,$CA,$73,$FD,$81,$B1,$9F,$9E,$FE,$07
	.dc.b	$18,$CF,$FC,$00,$FF,$8B,$6B,$87,$99,$AF,$57,$6F,$F0,$AF,$AB,$FD
	.dc.b	$AD,$FF,$81,$DB,$5F,$09,$FF,$19,$3F,$DC,$1C,$E3,$FE,$41,$7F,$D6
	.dc.b	$31,$9F,$87,$E9,$EE,$75,$7B,$70,$C9,$E5,$ED,$DC,$3F,$C4,$7E,$3F
	.dc.b	$F8,$CE,$7F,$BE,$B6,$33,$E3,$FF,$20,$8C,$67,$FE,$1D,$FA,$BF,$F6
	.dc.b	$8D,$FF,$81,$1B,$7F,$E2,$E7,$FC,$C9,$19,$F8,$4F,$E5,$3F,$56,$F8
	.dc.b	$7A,$7F,$2C,$FA,$BD,$E1,$FE,$36,$9D,$5F,$F8,$41,$78,$1C,$8E,$31
	.dc.b	$97,$AF,$83,$23,$07,$17,$C3,$C3,$23,$FC,$37,$83,$E1,$F1,$8F,$87
	.dc.b	$8B,$CE,$63,$6F,$F1,$E6,$7F,$42,$7F,$C9,$7F,$57,$F9,$1C,$7F,$32
	.dc.b	$D8,$FF,$3B,$62,$E8,$FE,$93,$39,$93,$FC,$A6,$B8,$F0,$EA,$76,$F0
	.dc.b	$F0,$E4,$FA,$3E,$16,$D7,$8B,$D4,$EA,$63,$4F,$F2,$96,$28,$F0,$FB
	.dc.b	$93,$0F,$F1,$DC,$E5,$B1,$E6,$F4,$64,$7C,$6F,$92,$B8,$FC,$0E,$8F
	.dc.b	$59,$8C,$BB,$BE,$EE,$F9,$7F,$3B,$E3,$6D,$FC,$8F,$07,$E1,$98,$C3
	.dc.b	$CE,$FC,$6F,$9A,$97,$F5,$BE,$16,$DF,$D0,$F8,$2F,$96,$63,$0F,$7B
	.dc.b	$F3,$BE,$AE,$5F,$BA,$2D,$7F,$A0,$BE,$9F,$F2,$F7,$FE,$77,$6D,$7F
	.dc.b	$12,$FA,$B7,$CB,$7F,$80,$27,$37,$FE,$B1,$7F,$11,$5F,$E6,$46,$C7
	.dc.b	$5F,$07,$87,$6F,$1F,$5A,$5F,$F9,$AB,$EA,$DF,$57,$8C,$3C,$9E,$C7
	.dc.b	$73,$1A,$F9,$5F,$17,$1C,$C3,$F3,$3E,$C5,$BF,$67,$17,$9C,$C6,$1E
	.dc.b	$D7,$D5,$F0,$12,$FD,$11,$6B,$E2,$7C,$57,$BC,$FF,$0F,$E9,$FE,$99
	.dc.b	$AF,$74,$7C,$3F,$AB,$91,$83,$E1,$3F,$1F,$8A,$47,$FA,$23,$D6,$78
	.dc.b	$F8,$C7,$FD,$67,$C4,$F7,$3F,$C5,$AD,$BF,$99,$86,$27,$FC,$63,$F5
	.dc.b	$7F,$C8,$AC,$DF,$97,$FE,$EB,$1F,$EB,$E8,$D1,$FE,$89,$BF,$F4,$CB
	.dc.b	$63,$5F,$07,$D3,$E1,$3A,$FC,$3C,$BF,$F4,$FE,$0D,$B1,$87,$5F,$7F
	.dc.b	$C6,$63,$93,$FD,$81,$9F,$F7,$B9,$F1,$6F,$E8,$3F,$C9,$3F,$4F,$ED
	.dc.b	$8C,$E5,$E4,$F3,$FD,$A6,$DC,$1F,$EB,$0F,$6B,$FC,$59,$3F,$ED,$A6
	.dc.b	$E3,$FE,$CF,$BF,$ED,$52,$3F,$C0,$1E,$DF,$F9,$62,$BF,$B2,$47,$17
	.dc.b	$11,$97,$5E,$6C,$C3,$FD,$7F,$6F,$5B,$18,$78,$7C,$DE,$86,$32,$FF
	.dc.b	$70,$63,$FC,$2E,$7C,$DF,$0B,$E1,$3F,$CD,$3F,$53,$E2,$AD,$C7,$97
	.dc.b	$E7,$A6,$7E,$F8,$B7,$FE,$F8,$FA,$BF,$99,$7F,$EE,$D6,$D7,$F8,$21
	.dc.b	$F8,$32,$72,$FC,$36,$B9,$F6,$7F,$CF,$71,$83,$FA,$7C,$77,$8F,$F7
	.dc.b	$D3,$D5,$AF,$36,$BC,$E7,$FD,$16,$B1,$63,$A6,$CE,$9B,$3F,$E9,$26
	.dc.b	$BF,$E2,$23,$94,$FF,$A4,$3E,$51,$F3,$DF,$2A,$30,$CB,$F0,$CE,$9B
	.dc.b	$1E,$68,$C5,$7F,$A8,$1F,$D3,$E4,$AF,$F1,$8C,$61,$B5,$E1,$F1,$7E
	.dc.b	$CF,$9D,$EB,$24,$FA,$DB,$83,$0E,$8F,$E4,$F0,$FE,$C5,$5F,$95,$9C
	.dc.b	$CF,$C7,$AF,$77,$AF,$2B,$D6,$FA,$48,$FC,$ED,$DE,$87,$E7,$4F,$CF
	.dc.b	$7C,$55,$9F,$D4,$47,$D6,$9D,$C7,$18,$C1,$9B,$91,$D5,$B8,$7F,$B1
	.dc.b	$AB,$FC,$CB,$39,$24,$66,$7F,$07,$C1,$FE,$16,$9F,$C1,$B6,$7F,$D9
	.dc.b	$ED,$7E,$F3,$5F,$A3,$E9,$FD,$72,$4F,$DC,$7E,$D7,$AF,$23,$EF,$B6
	.dc.b	$7E,$E7,$9B,$F7,$C9,$3F,$84,$7F,$05,$F4,$12,$3F,$0E,$D9,$F7,$7C
	.dc.b	$47,$C7,$24,$FC,$F7,$E5,$7C,$5C,$8F,$A4,$D9,$F8,$DE,$B7,$E0,$27
	.dc.b	$F3,$5A,$A3,$13,$FE,$7E,$8C,$4F,$F9,$F2,$3F,$8C,$46,$1F,$D8,$AF
	.dc.b	$57,$8C,$69,$AF,$FA,$75,$B1,$B9,$E9,$2F,$AB,$FF,$19,$79,$3F,$B6
	.dc.b	$6C,$7E,$BC,$D7,$3F,$FB,$ED,$C4,$69,$FD,$A2,$DB,$FD,$2F,$78,$6C
	.dc.b	$FF,$A6,$A7,$4F,$89,$AF,$F0,$CE,$67,$A5,$C2,$3F,$4C,$D5,$1A,$7D
	.dc.b	$6D,$7F,$8A,$CB,$CD,$FF,$54,$F4,$7E,$B2,$9E,$A6,$32,$FD,$EB,$70
	.dc.b	$FE,$8D,$87,$17,$F6,$93,$E9,$F8,$CD,$47,$C9,$C6,$D6,$E3,$F7,$D5
	.dc.b	$B7,$89,$FD,$47,$EA,$DF,$29,$CB,$BD,$EE,$D3,$AC,$EA,$B8,$85,$F2
	.dc.b	$9A,$B9,$3A,$9C,$B7,$D1,$F3,$ED,$FE,$7B,$8E,$6D,$7C,$E7,$96,$3C
	.dc.b	$3E,$8F,$89,$91,$D6,$3C,$BF,$09,$F0,$BC,$9F,$F4,$8C,$FF,$64,$8C
	.dc.b	$BE,$3F,$84,$AF,$63,$D5,$1E,$E8,$F7,$F9,$5C,$F3,$95,$EB,$8F,$87
	.dc.b	$F0,$DA,$8C,$1E,$F7,$E9,$FD,$5B,$CE,$F2,$61,$2B,$E4,$CE,$5B,$D9
	.dc.b	$C2,$A4,$60,$F9,$FF,$56,$FA,$BF,$BF,$E0,$3E,$0E,$9F,$82,$9D,$57
	.dc.b	$0F,$84,$C5,$C8,$C1,$D3,$CB,$FC,$4F,$D5,$B5,$A7,$38,$E9,$E6,$5A
	.dc.b	$E0,$79,$1F,$07,$8D,$F9,$2F,$AB,$FF,$4D,$4F,$E1,$9A,$FC,$FC,$D3
	.dc.b	$E7,$D7,$8C,$FB,$A3,$0F,$C0,$F1,$EB,$C6,$7D,$D1,$FA,$0F,$AB,$FE
	.dc.b	$AB,$E2,$7F,$56,$FA,$8F,$F5,$D5,$FE,$59,$AF,$DF,$9D,$F8,$3F,$E3
	.dc.b	$18,$47,$EC,$9F,$CE,$BA,$7F,$C6,$CD,$77,$E2,$F8,$7C,$7F,$D8,$FD
	.dc.b	$5B,$EA,$E9,$9A,$E7,$E7,$57,$C5,$5E,$98,$F5,$79,$DF,$BA,$FA,$B7
	.dc.b	$B5,$4C,$DF,$93,$CD,$AF,$AE,$BE,$14,$7C,$3F,$85,$B3,$B3,$D6,$FC
	.dc.b	$7F,$92,$FA,$DE,$C4,$9F,$7C,$7C,$BF,$85,$D4,$60,$F8,$6F,$CF,$F8
	.dc.b	$8F,$E1,$FC,$6F,$5A,$99,$AF,$93,$F0,$B5,$FA,$15,$F5,$E3,$F8,$BF
	.dc.b	$AA,$FA,$BF,$C8,$F9,$D2,$FF,$86,$CE,$5B,$E5,$F0,$A9,$18,$37,$E8
	.dc.b	$F4,$99,$18,$39,$FC,$FF,$E7,$BE,$AE,$46,$0E,$FF,$33,$E4,$99,$18
	.dc.b	$3C,$7E,$87,$EC,$3E,$06,$46,$0F,$3F,$91,$EC,$99,$18,$3D,$7F,$09
	.dc.b	$FC,$DB,$E2,$3F,$A2,$78,$7F,$E2,$C6,$F8,$7F,$93,$E0,$FF,$58,$DF
	.dc.b	$C2,$EC,$F8,$F8,$48,$C1,$F1,$BD,$6F,$97,$32,$30,$7F,$36,$FA,$B7
	.dc.b	$81,$FD,$19,$FE,$BB,$9F,$F8,$F1,$B9,$7E,$8D,$DD,$71,$F7,$78,$7B
	.dc.b	$B2,$30,$7F,$42,$FA,$B7,$C5,$FF,$52,$5C,$94,$C4,$67,$87,$5F,$0D
	.dc.b	$2F,$94,$7F,$5A,$FA,$B7,$B1,$7D,$89,$5F,$D4,$27,$2D,$F8,$5E,$1E
	.dc.b	$A4,$8C,$1F,$D5,$BB,$FF,$A6,$71,$7F,$96,$9F,$DF,$47,$F4,$16,$C6
	.dc.b	$7E,$37,$FB,$5C,$3C,$6B,$F3,$52,$FA,$47,$F7,$0F,$AB,$55,$F0,$A5
	.dc.b	$7F,$66,$9C,$B7,$8B,$87,$81,$23,$06,$FD,$EE,$9E,$FC,$8C,$1F,$DC
	.dc.b	$3E,$AD,$E1,$FF,$80,$1F,$E8,$51,$9F,$7F,$13,$FE,$04,$FA,$B6,$F7
	.dc.b	$E4,$E5,$FF,$7A,$9C,$BF,$DF,$3D,$4F,$EE,$EF,$F6,$57,$FC,$00,$DF
	.dc.b	$27,$FC,$1F,$1A,$E4,$FF,$42,$BF,$98,$9E,$1F,$57,$A3,$7E,$37,$0D
	.dc.b	$B8,$F8,$D2,$BC,$11,$FE,$1A,$FA,$BF,$5C,$7F,$75,$5F,$07,$93,$5C
	.dc.b	$1F,$E2,$0F,$AB,$FE,$69,$3F,$CC,$6B,$FC,$2A,$D7,$FB,$F4,$7F,$39
	.dc.b	$1B,$F1,$F5,$64,$FF,$87,$27,$2D,$EA,$ED,$C5,$23,$06,$FD,$4F,$14
	.dc.b	$C8,$C1,$CF,$C7,$F3,$78,$F2,$30,$77,$F4,$BD,$53,$23,$07,$F8,$BB
	.dc.b	$EA,$9F,$11,$FE,$3E,$FA,$BF,$F3,$74,$FF,$8C,$9A,$FF,$86,$E7,$6F
	.dc.b	$1E,$DF,$F2,$2C,$62,$35,$58,$9F,$F2,$4F,$C5,$FF,$84,$E7,$FC,$49
	.dc.b	$E9,$B1,$FF,$26,$B6,$3F,$C8,$33,$BE,$AF,$F6,$2B,$FB,$48,$FF,$16
	.dc.b	$39,$F3,$FF,$B7,$36,$07,$F9,$3B,$CA,$FF,$2C,$FD,$5F,$E3,$AF,$FD
	.dc.b	$9F,$83,$93,$C2,$6B,$BB,$7E,$2D,$C7,$FC,$A7,$87,$1C,$FF,$BE,$3F
	.dc.b	$95,$6F,$8D,$C2,$77,$5E,$08,$F0,$EF,$EE,$E4,$95,$D2,$3C,$BC,$1A
	.dc.b	$8C,$1E,$07,$D3,$E2,$FF,$BA,$E3,$FD,$2D,$97,$F2,$2D,$BF,$D3,$10
	.dc.b	$8C,$3A,$9E,$0F,$F8,$4C,$D7,$9C,$FA,$BE,$35,$FC,$B9,$B1,$99,$EE
	.dc.b	$D7,$BF,$D2,$F8,$1F,$9E,$D4,$D3,$5F,$42,$DF,$FC,$FB,$28,$C3,$FE
	.dc.b	$67,$39,$8F,$AB,$FF,$43,$BF,$E2,$B8,$DE,$7C,$23,$EF,$FA,$18,$BF
	.dc.b	$73,$63,$84,$78,$23,$11,$FE,$CE,$8F,$1F,$2E,$39,$5F,$2F,$AA,$FE
	.dc.b	$8F,$5E,$8F,$99,$C1,$9B,$B1,$EC,$F9,$9E,$06,$2B,$D8,$F8,$32,$FE
	.dc.b	$77,$3F,$E9,$E6,$CF,$E9,$3C,$2D,$73,$6F,$96,$5E,$E2,$D7,$FA,$37
	.dc.b	$EA,$FE,$54,$FF,$AA,$1B,$1F,$AE,$F0,$3D,$F1,$8C,$7E,$43,$FE,$89
	.dc.b	$AE,$1B,$EB,$83,$6F,$0F,$16,$4F,$5F,$1A,$E5,$E1,$F3,$7A,$98,$CB
	.dc.b	$AB,$E0,$F4,$A5,$F9,$7E,$3E,$DF,$A7,$D4,$F2,$98,$C3,$CA,$FA,$3E
	.dc.b	$36,$5F,$B7,$E6,$2D,$FF,$AB,$7E,$63,$EB,$CF,$FB,$21,$B1,$9F,$77
	.dc.b	$EF,$C6,$31,$FC,$42,$F2,$B2,$6B,$C8,$B8,$7F,$B4,$BE,$AD,$F3,$BF
	.dc.b	$88,$6C,$67,$C1,$C4,$E7,$BE,$4F,$1F,$72,$EB,$FE,$C3,$FA,$B7,$81
	.dc.b	$F9,$26,$D7,$FB,$81,$FC,$7C,$EE,$75,$C1,$2B,$5C,$56,$BF,$D9,$DF
	.dc.b	$57,$FC,$AE,$7F,$DD,$2D,$8C,$F6,$7E,$82,$31,$C5,$FA,$9A,$FF,$16
	.dc.b	$9F,$EF,$02,$D7,$FB,$65,$FF,$17,$99,$DB,$F5,$3F,$57,$FC,$BA,$3F
	.dc.b	$DE,$ED,$9E,$1E,$E6,$9D,$F8,$72,$B7,$DB,$67,$FD,$CD,$6D,$3A,$9D
	.dc.b	$BA,$AD,$FF,$3C,$CF,$F8,$07,$8C,$F6,$BF,$76,$D9,$D1,$FE,$B1,$5F
	.dc.b	$3F,$A9,$DF,$9B,$2B,$79,$FF,$1A,$F5,$70,$FF,$BE,$7E,$AF,$F6,$B9
	.dc.b	$7F,$0F,$6D,$7F,$AF,$57,$A1,$FC,$31,$CD,$7F,$52,$7F,$D5,$F3,$DA
	.dc.b	$11,$87,$B7,$B7,$C7,$E1,$79,$71,$CB,$E1,$6F,$6F,$18,$32,$FE,$45
	.dc.b	$E4,$7F,$8E,$2F,$B3,$67,$51,$FC,$A7,$C8,$FE,$43,$39,$AF,$E5,$5F
	.dc.b	$56,$F9,$4F,$E4,$AD,$BF,$43,$FC,$E5,$B1,$FC,$AB,$EA,$DF,$0D,$4F
	.dc.b	$E5,$BF,$57,$BC,$FC,$27,$F7,$49,$DD,$FC,$35,$E1,$73,$39,$46,$5E
	.dc.b	$DE,$1C,$8C,$1C,$5F,$17,$14,$8F,$9D,$83,$E1,$B1,$8F,$88,$63,$F4
	.dc.b	$75,$FE,$94,$DB,$FF,$7C,$2C,$3D,$DC,$7F,$D2,$3E,$AF,$F3,$D7,$FD
	.dc.b	$45,$B7,$FE,$FF,$CB,$F1,$EF,$DF,$8C,$E3,$FE,$14,$7F,$99,$47,$F8
	.dc.b	$5A,$11,$87,$47,$7E,$0F,$07,$89,$2B,$DB,$D3,$6B,$D9,$6F,$53,$CC
	.dc.b	$67,$2F,$33,$E9,$F7,$66,$1E,$58,$B7,$CB,$D1,$F7,$1F,$6B,$E4,$2D
	.dc.b	$7A,$F9,$FE,$11,$8A,$F0,$BF,$17,$C0,$93,$F4,$BE,$4E,$D7,$9B,$C2
	.dc.b	$F9,$46,$34,$F9,$4F,$CD,$F9,$A9,$7F,$6B,$E0,$2D,$FB,$BE,$6B,$E9
	.dc.b	$18,$C3,$E9,$3F,$57,$EA,$E5,$FD,$18,$B5,$F8,$1F,$AA,$FC,$E3,$7B
	.dc.b	$BF,$C0,$7F,$56,$F9,$6F,$F0,$2D,$6B,$2F,$F0,$73,$F8,$BC,$7F,$74
	.dc.b	$DC,$39,$46,$3B,$78,$FA,$F2,$3F,$80,$7D,$53,$EA,$F1,$7E,$3F,$81
	.dc.b	$EE,$63,$2E,$EF,$8B,$8E,$61,$F9,$CF,$66,$DF,$AF,$8B,$CE,$63,$0F
	.dc.b	$3B,$EA,$F8,$19,$7E,$88,$B5,$F0,$FE,$2B,$DE,$7F,$89,$34,$FF,$51
	.dc.b	$D7,$BA,$3E,$1F,$D5,$C8,$C1,$F0,$9F,$8F,$C3,$23,$F8,$F7,$AF,$FA
	.dc.b	$97,$E3,$C7,$F8,$C3,$85,$EE,$D3,$86,$5F,$B5,$8C,$D7,$F2,$7F,$AB
	.dc.b	$FE,$57,$26,$FC,$C7,$F7,$B9,$FD,$54,$68,$FF,$4D,$BC,$7E,$17,$FC
	.dc.b	$7F,$19,$6F,$B5,$27,$F9,$A5,$34,$C6,$1F,$C1,$F0,$FE,$FE,$DF,$7B
	.dc.b	$3C,$5F,$E5,$19,$CB,$7C,$BE,$E7,$F9,$3B,$E9,$FD,$77,$CE,$3C,$7E
	.dc.b	$64,$CF,$DD,$16,$7F,$8B,$7B,$9F,$E3,$56,$FB,$9C,$F6,$FF,$2C,$CE
	.dc.b	$72,$FF,$31,$3F,$E0,$86,$BA,$9D,$BC,$3C,$C9,$8F,$F5,$3C,$FF,$A6
	.dc.b	$3F,$D5,$67,$FC,$27,$8F,$8B,$C9,$EE,$70,$71,$3B,$F8,$4F,$19,$FE
	.dc.b	$6C,$FA,$9F,$14,$7F,$1D,$BF,$E9,$CD,$F9,$EF,$49,$C1,$F0,$3C,$7C
	.dc.b	$AE,$57,$67,$FB,$45,$F2,$C3,$FD,$7A,$AF,$C0,$70,$9D,$B8,$7F,$D4
	.dc.b	$B1,$B6,$C7,$FA,$26,$7F,$99,$4E,$5F,$F5,$30,$C1,$F0,$DB,$8F,$F6
	.dc.b	$6C,$62,$FF,$D6,$2D,$8E,$2D,$7F,$D6,$0B,$CB,$FB,$27,$DA,$AB,$5F
	.dc.b	$46,$3B,$57,$85,$B5,$C6,$7B,$C6,$3C,$B9,$1E,$26,$D7,$AB,$07,$97
	.dc.b	$D1,$B5,$F1,$23,$06,$7C,$A9,$1F,$21,$B5,$F1,$3E,$4F,$FC,$5C,$DF
	.dc.b	$47,$CF,$18,$8F,$F6,$73,$5F,$9B,$F4,$3E,$12,$5F,$D4,$7D,$FF,$56
	.dc.b	$F9,$EA,$49,$FF,$6E,$C6,$5F,$9F,$F0,$DF,$10,$8E,$09,$7F,$4A,$32
	.dc.b	$D8,$FA,$FF,$15,$F3,$08,$E0,$FB,$7A,$7F,$B5,$1F,$B2,$D9,$FC,$E3
	.dc.b	$5F,$C1,$B5,$64,$F2,$30,$DA,$FF,$33,$48,$DC,$E8,$E5,$C2,$F4,$A9
	.dc.b	$2B,$8D,$7E,$56,$37,$AC,$9C,$D2,$31,$5F,$9F,$C3,$FC,$69,$5F,$58
	.dc.b	$F2,$90,$EB,$F5,$7F,$D9,$9F,$F0,$B4,$A3,$0F,$F8,$65,$AF,$AD,$AE
	.dc.b	$FC,$5F,$89,$1F,$D9,$E7,$9B,$67,$94,$EB,$E1,$7F,$A2,$C7,$EC,$A7
	.dc.b	$FC,$F3,$E3,$FE,$D2,$3C,$5F,$3E,$F6,$B1,$5A,$9E,$1E,$87,$F8,$B7
	.dc.b	$2E,$ED,$8F,$43,$5F,$43,$5F,$76,$F6,$BE,$D4,$7C,$48,$C1,$9F,$85
	.dc.b	$F2,$47,$C5,$6C,$9A,$63,$CD,$FF,$20,$78,$8C,$7F,$C8,$5B,$FF,$9C
	.dc.b	$70,$FF,$43,$35,$FF,$2C,$60,$BE,$35,$F4,$7E,$32,$D7,$D1,$9A,$FD
	.dc.b	$76,$D2,$73,$F5,$6F,$95,$DF,$6B,$DD,$A6,$A3,$7F,$11,$72,$FF,$4D
	.dc.b	$5F,$16,$C7,$4F,$90,$F0,$EB,$FE,$71,$9F,$5F,$36,$5E,$3E,$95,$7F
	.dc.b	$CE,$98,$2F,$CB,$3E,$6F,$84,$F5,$F9,$5F,$E8,$6B,$FE,$B1,$19,$7C
	.dc.b	$7F,$09,$5E,$71,$EE,$8F,$7F,$C3,$F2,$D6,$57,$AA,$3E,$1F,$C4,$EA
	.dc.b	$30,$7B,$9F,$97,$F5,$6F,$43,$C7,$84,$AF,$F5,$E4,$E5,$BE,$07,$0A
	.dc.b	$91,$83,$FD,$85,$F5,$0F,$AB,$FA,$FE,$23,$D0,$A7,$DA,$9D,$57,$0F
	.dc.b	$86,$C5,$C8,$C1,$D3,$CF,$FC,$0F,$D5,$B5,$A7,$38,$E9,$E8,$5A,$E0
	.dc.b	$79,$1F,$07,$95,$F8,$AF,$AB,$FF,$46,$4F,$FB,$65,$AF,$FA,$AF,$34
	.dc.b	$F9,$F5,$E3,$3E,$E8,$C3,$F0,$3D,$0A,$F1,$9F,$74,$7E,$63,$EA,$FF
	.dc.b	$49,$3F,$EE,$7F,$A8,$FF,$7C,$B7,$CC,$D7,$8B,$BF,$89,$B3,$FE,$ED
	.dc.b	$BE,$2F,$FA,$FD,$FD,$57,$8F,$85,$E9,$FE,$9B,$EA,$FF,$93,$57,$FB
	.dc.b	$D5,$AF,$18,$F2,$7C,$47,$95,$91,$DE,$3D,$1F,$55,$FB,$0F,$AB,$7B
	.dc.b	$F4,$CD,$79,$7D,$0A,$F9,$6B,$E0,$47,$C1,$F4,$2C,$EC,$F6,$3F,$17
	.dc.b	$E5,$3E,$A7,$B7,$27,$E0,$47,$C9,$F8,$6D,$46,$0F,$88,$FC,$DF,$89
	.dc.b	$FE,$0B,$C8,$F8,$EA,$66,$BE,$57,$C4,$57,$E5,$57,$D5,$8F,$E1,$BE
	.dc.b	$C3,$EA,$FF,$1F,$E7,$4B,$FE,$B9,$CD,$F4,$AB,$51,$8E,$8F,$F7,$04
	.dc.b	$7F,$09,$8F,$F7,$A3,$A6,$DF,$6F,$2F,$BF,$A5,$67,$91,$FD,$A6,$B9
	.dc.b	$61,$DE,$7C,$BF,$41,$FA,$2F,$80,$97,$C0,$5A,$7D,$7F,$41,$C2,$75
	.dc.b	$39,$6C,$7C,$0F,$43,$F6,$FF,$07,$27,$71,$F1,$3C,$8D,$46,$0E,$6F
	.dc.b	$C8,$F8,$7F,$E6,$BF,$16,$EA,$A6,$6B,$C3,$D9,$5F,$8B,$5F,$42,$3E
	.dc.b	$8F,$B1,$67,$67,$A1,$FE,$83,$F5,$6F,$02,$F8,$32,$BE,$14,$E5,$BE
	.dc.b	$E7,$0F,$7A,$46,$0F,$91,$F5,$6F,$8A,$FE,$9A,$B8,$A9,$88,$CF,$0E
	.dc.b	$BE,$5E,$5F,$28,$FE,$AB,$F5,$7F,$E6,$D8,$FA,$8F,$F2,$95,$D3,$FB
	.dc.b	$A8,$C3,$7D,$8E,$3F,$15,$CB,$D3,$97,$CA,$3F,$B0,$F8,$1F,$D4,$63
	.dc.b	$9F,$81,$FE,$2F,$7F,$A7,$79,$BA,$37,$F0,$1C,$3C,$8E,$3F,$09,$2B
	.dc.b	$F1,$18,$27,$FB,$6F,$D5,$FF,$57,$9F,$C6,$2F,$0F,$26,$AB,$7C,$7C
	.dc.b	$3C,$29,$7C,$23,$9F,$BD,$D3,$DF,$93,$D2,$3F,$BD,$FD,$5B,$C4,$BE
	.dc.b	$F4,$AF,$CE,$4E,$5F,$CF,$7D,$5F,$F8,$45,$BF,$29,$FE,$04,$83,$78
	.dc.b	$3F,$C0,$FE,$07,$A6,$BF,$4C,$FE,$B3,$93,$E4,$36,$3F,$CB,$0E,$DF
	.dc.b	$9A,$63,$C4,$E7,$F5,$72,$39,$C7,$6D,$BB,$F8,$92,$7B,$C7,$F8,$63
	.dc.b	$EA,$FF,$B1,$47,$ED,$D7,$C1,$64,$D6,$FF,$E1,$EF,$AB,$FE,$7F,$DD
	.dc.b	$FD,$02,$BF,$9C,$B7,$E3,$B5,$B6,$F8,$38,$7A,$F3,$3F,$E0,$98,$FC
	.dc.b	$46,$F5,$F5,$E2,$91,$83,$7E,$B7,$86,$64,$60,$E7,$E4,$79,$7C,$89
	.dc.b	$18,$3B,$FA,$9E,$99,$91,$83,$F8,$B7,$D5,$3E,$27,$FC,$77,$F5,$6F
	.dc.b	$80,$A7,$F8,$FA,$75,$5C,$3C,$0C,$5C,$8C,$1F,$C8,$7E,$33,$F5,$7C
	.dc.b	$5F,$A3,$F5,$3F,$C8,$93,$FC,$95,$AF,$DC,$3F,$C8,$9F,$13,$FE,$4C
	.dc.b	$AF,$DC,$46,$5E,$6F,$FA,$6D,$AF,$F9,$2F,$CB,$FF,$29,$FD,$5F,$E2
	.dc.b	$63,$FC,$29,$CB,$FE,$56,$C3,$FC,$DB,$1D,$E7,$47,$E3,$7A,$3F,$BD
	.dc.b	$B8,$F6,$79,$F9,$1D,$3C,$8B,$7F,$CF,$6B,$86,$BF,$BF,$7F,$6D,$1D
	.dc.b	$76,$F0,$73,$48,$E1,$1E,$6E,$0F,$3B,$23,$A4,$7A,$BC,$8F,$E2,$B1
	.dc.b	$FE,$A4,$17,$FC,$8F,$4F,$F4,$B3,$63,$4D,$7F,$D5,$6F,$F2,$BF,$AB
	.dc.b	$DF,$A4,$69,$27,$FD,$1B,$F4,$BF,$E8,$E4,$D2,$3F,$EA,$5F,$92,$FF
	.dc.b	$14,$E1,$FE,$22,$8F,$F5,$36,$BF,$D8,$5F,$F6,$85,$7F,$B4,$7E,$AF
	.dc.b	$FD,$01,$1F,$C9,$6F,$8D,$7F,$B0,$39,$AC,$FF,$29,$6B,$FE,$D9,$9A
	.dc.b	$FF,$99,$67,$B7,$2E,$0D,$F2,$3E,$7B,$13,$E3,$F9,$AE,$26,$2F,$F2
	.dc.b	$01,$EB,$F3,$52,$78,$1E,$B7,$C7,$AF,$FA,$26,$3F,$BF,$36,$BF,$35
	.dc.b	$87,$F2,$09,$CF,$2F,$0E,$49,$18,$38,$BE,$EF,$AB,$F9,$37,$FA,$36
	.dc.b	$E1,$FE,$4E,$C3,$FD,$69,$7F,$E0,$D9,$CB,$FD,$3D,$F8,$95,$C7,$6E
	.dc.b	$9B,$75,$7B,$78,$F2,$F9,$79,$16,$FC,$1E,$87,$8C,$C6,$7F,$E5,$EC
	.dc.b	$BF,$A2,$5F,$87,$D6,$F4,$1E,$9F,$90,$B1,$E5,$F5,$FD,$86,$31,$F6
	.dc.b	$3E,$FF,$8E,$93,$EB,$F9,$AB,$5F,$C0,$FE,$67,$ED,$4F,$F8,$AD,$B1
	.dc.b	$9F,$80,$FC,$2C,$63,$4F,$F3,$9A,$F2,$72,$6B,$C9,$B8,$7F,$8D,$7E
	.dc.b	$AD,$F3,$DF,$8E,$6C,$67,$C0,$C4,$E7,$9E,$4F,$1F,$7E,$EB,$FC,$67
	.dc.b	$EA,$FF,$15,$3F,$E4,$16,$C6,$7B,$FF,$37,$18,$DB,$FD,$12,$B8,$32
	.dc.b	$6B,$8A,$E1,$FE,$4B,$FA,$BF,$F7,$63,$77,$FD,$2C,$7E,$13,$2F,$E5
	.dc.b	$4E,$9F,$E5,$93,$FE,$2B,$3F,$CA,$85,$73,$E2,$FE,$BA,$31,$97,$FA
	.dc.b	$73,$EA,$D5,$FB,$26,$CF,$2F,$7B,$13,$9D,$64,$FF,$98,$6B,$F2,$77
	.dc.b	$CE,$79,$67,$2B,$5D,$17,$0F,$E8,$5E,$2F,$FA,$A2,$7F,$CD,$4D,$8C
	.dc.b	$FC,$A7,$F0,$48,$C6,$3F,$EB,$27,$FC,$51,$5C,$3C,$AD,$7A,$9B,$7F
	.dc.b	$85,$FD,$5F,$ED,$EF,$FD,$72,$DA,$FF,$1C,$3F,$E2,$D7,$FC,$F4,$E7
	.dc.b	$1F,$86,$FF,$AC,$23,$3F,$11,$D3,$DD,$EA,$F6,$E0,$93,$CB,$DC,$B8
	.dc.b	$7F,$5F,$F1,$BF,$56,$DF,$0F,$F8,$FC,$7F,$BD,$6B,$FD,$14,$EB,$3F
	.dc.b	$F6,$7F,$D5,$BE,$17,$F9,$2B,$6B,$FD,$20,$9C,$C7,$F2,$AF,$AB,$7C
	.dc.b	$45,$3F,$96,$FD,$5E,$EB,$FD,$31,$3A,$6F,$77,$13,$83,$83,$7D,$FD
	.dc.b	$4D,$38,$39,$F0,$F8,$4E,$FF,$11,$63,$BF,$17,$94,$C7,$07,$F8,$0F
	.dc.b	$1F,$9C,$FF,$81,$7E,$AF,$F2,$2D,$3D,$3D,$1F,$D1,$A3,$3C,$1F,$EF
	.dc.b	$47,$F9,$2C,$FE,$69,$A9,$DE,$1F,$E7,$77,$27,$ED,$5C,$7A,$3F,$1B
	.dc.b	$E1,$BB,$3D,$FD,$29,$3F,$EB,$FA,$E1,$7D,$E7,$C9,$ED,$CA,$D4,$59
	.dc.b	$F6,$79,$FE,$93,$E9,$FC,$9D,$AE,$9E,$8F,$B4,$C5,$7C,$37,$E0,$78
	.dc.b	$12,$7F,$DA,$1C,$BF,$EB,$F7,$C9,$1F,$17,$C1,$91,$83,$D0,$FC,$9F
	.dc.b	$9A,$91,$CB,$DD,$B8,$FD,$1F,$7F,$E7,$18,$E2,$F9,$CF,$D1,$FA,$49
	.dc.b	$7F,$0E,$2D,$7F,$8F,$7E,$AB,$F2,$CD,$ED,$FE,$FB,$F5,$6F,$97,$FE
	.dc.b	$FF,$5A,$CB,$F3,$57,$FC,$4A,$1D,$7A,$37,$85,$CB,$FD,$E1,$3F,$ED
	.dc.b	$36,$FA,$BF,$E0,$AF,$AA,$7D,$5F,$68,$EF,$EF,$4A,$D9,$C9,$F1,$FC
	.dc.b	$AC,$8F,$9B,$EC,$5D,$3D,$9C,$5E,$83,$1B,$7A,$21,$A2,$7D,$BF,$17
	.dc.b	$30,$F3,$C5,$AF,$8D,$F1,$5F,$08,$F4,$75,$DA,$F6,$FD,$5F,$C6,$31
	.dc.b	$5F,$35,$F9,$3C,$72,$7F,$9C,$E7,$E3,$F8,$B7,$8D,$9C,$3F,$CE,$1C
	.dc.b	$AF,$77,$FC,$5E,$DC,$3E,$5E,$38,$9F,$F1,$9F,$D5,$FF,$23,$B3,$7E
	.dc.b	$63,$FB,$95,$FF,$B8,$63,$47,$FA,$35,$F1,$99,$F8,$4E,$7F,$0B,$D1
	.dc.b	$EB,$F0,$12,$77,$87,$F5,$2C,$1B,$E6,$3F,$D1,$18,$FF,$78,$6F,$C0
	.dc.b	$67,$83,$E9,$4E,$5B,$BF,$A8,$FF,$26,$FD,$3F,$BE,$F9,$C7,$BB,$CE
	.dc.b	$99,$F8,$A7,$DF,$38,$7F,$D3,$1E,$DF,$F5,$E9,$FF,$80,$0D,$8F,$ED
	.dc.b	$37,$FD,$B2,$73,$F0,$DF,$E5,$CA,$14,$C3,$63,$C1,$1F,$C3,$2B,$7E
	.dc.b	$6F,$A3,$D7,$E4,$F3,$F3,$25,$F0,$8B,$5E,$6F,$5F,$C0,$7F,$9B,$3E
	.dc.b	$A7,$C3,$5D,$A3,$C9,$F3,$F2,$30,$78,$9E,$11,$FE,$7F,$AF,$F0,$2D
	.dc.b	$7F,$5A,$7F,$C0,$E6,$23,$4A,$34,$E9,$4E,$9B,$5F,$E9,$F9,$E1,$39
	.dc.b	$24,$7D,$3F,$94,$7C,$E7,$AE,$7F,$D0,$F1,$89,$F9,$3E,$77,$CF,$3E
	.dc.b	$87,$C4,$D8,$C9,$DF,$64,$FA,$1B,$19,$8C,$79,$F2,$38,$37,$2F,$F5
	.dc.b	$B7,$D5,$BD,$AF,$82,$DA,$F6,$36,$77,$3F,$07,$CF,$95,$F0,$76,$FF
	.dc.b	$79,$36,$3E,$5E,$9F,$E2,$3B,$F9,$E1,$C3,$F4,$9A,$FF,$8E,$9A,$EF
	.dc.b	$CB,$E6,$4A,$FA,$13,$FC,$A6,$7D,$7F,$3D,$EA,$24,$FA,$9F,$8D,$F1
	.dc.b	$12,$3E,$53,$67,$FD,$B1,$B3,$E2,$FE,$61,$2B,$E8,$36,$76,$D7,$DB
	.dc.b	$3C,$67,$EB,$7A,$9F,$D1,$1C,$B6,$7F,$62,$D7,$FC,$B3,$28,$C4,$7E
	.dc.b	$C3,$3B,$3F,$3A,$3F,$DB,$4D,$AF,$0C,$EB,$D4,$91,$FE,$12,$AE,$B1
	.dc.b	$87,$8F,$CB,$74,$24,$EE,$3F,$7F,$1E,$2F,$AB,$FF,$18,$47,$FB,$A5
	.dc.b	$B3,$F7,$1B,$1F,$AC,$6B,$FE,$74,$7F,$B5,$C6,$FD,$5E,$84,$7C,$11
	.dc.b	$FD,$A1,$BE,$57,$84,$8E,$BD,$DF,$E8,$57,$F8,$C1,$48,$CC,$79,$7A
	.dc.b	$5F,$05,$FB,$7A,$D7,$97,$7D,$EF,$E0,$78,$FA,$DB,$87,$AD,$AF,$AD
	.dc.b	$AF,$83,$D1,$B7,$FD,$A6,$3D,$71,$83,$3E,$97,$F0,$78,$F3,$B6,$4D
	.dc.b	$35,$D4,$F1,$A9,$2B,$F6,$AF,$1F,$43,$F8,$DF,$91,$FD,$EF,$AB,$FC
	.dc.b	$E1,$5F,$C4,$78,$B3,$AF,$D5,$BB,$3F,$57,$E2,$EB,$F2,$A2,$E8,$6C
	.dc.b	$EB,$EA,$DF,$2B,$C3,$BD,$F0,$14,$D4,$6F,$E5,$6E,$5F,$D5,$6F,$8B
	.dc.b	$63,$A7,$A9,$E2,$DB,$FC,$53,$3F,$CA,$7A,$72,$F1,$F5,$2A,$E1,$FF
	.dc.b	$0D,$62,$C3,$CD,$5E,$7F,$13,$DB,$E7,$7F,$1F,$C3,$FB,$33,$5E,$5C
	.dc.b	$5F,$D1,$30,$C6,$3D,$3E,$13,$95,$7A,$70,$F8,$51,$F0,$FC,$EE,$BB
	.dc.b	$CA,$F7,$47,$C7,$F8,$7D,$46,$0F,$84,$FD,$0F,$AB,$7A,$3E,$6C,$25
	.dc.b	$7F,$7B,$9C,$B7,$C5,$E1,$52,$30,$7F,$80,$7E,$A1,$F5,$7F,$73,$C8
	.dc.b	$7C,$2D,$3E,$FC,$EA,$B8,$7B,$98,$B9,$18,$3A,$7A,$3F,$86,$FA,$B6
	.dc.b	$B4,$E7,$1D,$3D,$2B,$5C,$0F,$23,$E0,$F3,$BF,$1F,$F5,$6F,$5E,$99
	.dc.b	$AF,$56,$D5,$FA,$A7,$FB,$FB,$5F,$E6,$CE,$8F,$81,$E8,$7B,$26,$46
	.dc.b	$0F,$F1,$27,$D5,$FF,$97,$FA,$3F,$CC,$1F,$51,$FC,$1E,$7F,$C5,$0D
	.dc.b	$7B,$89,$E1,$D7,$FE,$08,$7F,$5B,$1A,$72,$8E,$11,$FA,$C8,$DD,$7F
	.dc.b	$9B,$D7,$99,$FB,$0F,$AB,$7A,$54,$E1,$1C,$7D,$6B,$8F,$AA,$30,$FB
	.dc.b	$3E,$AB,$F7,$3F,$57,$FC,$1E,$3F,$C8,$4D,$7C,$D1,$E3,$F8,$7F,$23
	.dc.b	$23,$E2,$C7,$9F,$D2,$D4,$60,$F9,$2F,$AF,$E4,$FE,$CF,$B9,$27,$E6
	.dc.b	$C7,$BF,$E1,$B5,$18,$3E,$8B,$F0,$FE,$23,$F8,$7F,$13,$E3,$A9,$9A
	.dc.b	$FA,$FF,$11,$5F,$9D,$5F,$2E,$3F,$8B,$FB,$6F,$AB,$FC,$A7,$9D,$2F
	.dc.b	$FC,$C5,$39,$6F,$95,$C2,$A4,$60,$DF,$A3,$D2,$64,$60,$E7,$F4,$1F
	.dc.b	$A1,$F8,$19,$18,$3B,$FC,$CF,$92,$64,$60,$F1,$F9,$FF,$B3,$F8,$49
	.dc.b	$18,$3C,$FE,$3F,$B2,$64,$60,$F5,$FC,$37,$F3,$6F,$8A,$74,$53,$E1
	.dc.b	$C7,$C4,$F7,$AD,$70,$3E,$1B,$F2,$3D,$8D,$46,$0F,$8E,$FF,$42,$FA
	.dc.b	$B7,$7D,$EF,$95,$FE,$8D,$9C,$B7,$DC,$E1,$EE,$C8,$C1,$FE,$93,$FA
	.dc.b	$B7,$C6,$FF,$4E,$5C,$74,$C4,$67,$83,$5F,$2D,$2F,$94,$7F,$56,$FA
	.dc.b	$BF,$F4,$9C,$7F,$A7,$DF,$E4,$AF,$EE,$1B,$E4,$FF,$B7,$5D,$C6,$7E
	.dc.b	$2F,$97,$A5,$23,$94,$7F,$62,$EE,$FE,$A3,$1F,$C0,$9F,$DC,$F9,$9F
	.dc.b	$E8,$86,$C7,$EF,$DD,$E1,$FE,$EB,$5E,$2F,$3F,$9A,$95,$FE,$B5,$9C
	.dc.b	$BF,$EB,$7F,$AB,$FE,$B0,$DF,$03,$FB,$9C,$1B,$38,$E0,$CF,$74,$BE
	.dc.b	$51,$C3,$DD,$E3,$EF,$49,$ED,$1F,$DF,$BE,$AD,$E0,$DF,$72,$57,$FB
	.dc.b	$26,$72,$FF,$B2,$FE,$AD,$B7,$F8,$19,$FF,$0B,$C6,$7E,$2B,$13,$FE
	.dc.b	$0A,$ED,$F4,$4F,$FB,$4D,$FE,$FE,$FF,$2C,$6F,$13,$8F,$FD,$67,$1C
	.dc.b	$27,$5E,$0F,$3F,$AB,$91,$CE,$3B,$6F,$DF,$C1,$93,$DE,$3F,$C3,$7F
	.dc.b	$57,$FD,$82,$3F,$DC,$8B,$E0,$32,$6B,$83,$FC,$43,$F5,$7F,$CE,$A9
	.dc.b	$5F,$CD,$1E,$3E,$B6,$B5,$DF,$07,$0F,$52,$57,$FB,$C7,$0F,$8A,$FF
	.dc.b	$4A,$7E,$FB,$78,$7B,$9E,$0E,$BF,$0F,$A7,$67,$94,$61,$E9,$E2,$F9
	.dc.b	$BC,$59,$18,$3C,$1E,$8F,$AA,$64,$60,$FE,$A9,$CD,$F3,$DF,$EA,$DF
	.dc.b	$54,$F8,$8F,$F2,$0F,$D5,$BD,$DA,$7F,$91,$27,$55,$CB,$87,$FD,$D3
	.dc.b	$0F,$03,$58,$4A,$D1,$DA,$3F,$C9,$BF,$17,$FB,$4B,$FF,$78,$FA,$2D
	.dc.b	$3F,$CA,$2D,$BE,$27,$F9,$33,$E1,$B8,$FC,$3F,$26,$46,$B0,$FD,$EE
	.dc.b	$DE,$D8,$7D,$5F,$E3,$B2,$EF,$CD,$D8,$FF,$2F,$C6,$67,$FB,$C3,$F9
	.dc.b	$1A,$E4,$E6,$39,$FA,$DC,$3C,$8E,$3E,$45,$9F,$3D,$71,$C7,$FC,$05
	.dc.b	$1E,$29,$EF,$C1,$EE,$E6,$93,$C2,$3C,$7C,$3A,$8C,$1D,$1F,$3F,$91
	.dc.b	$FE,$5F,$8F,$F5,$06,$3F,$CA,$B4,$FF,$51,$47,$C0,$9D,$3F,$EF,$09
	.dc.b	$D3,$C3,$CF,$7D,$5F,$09,$FE,$6A,$D8,$CC,$EE,$3F,$D3,$0B,$E9,$69
	.dc.b	$1F,$F4,$E3,$FE,$9D,$F9,$4F,$C7,$63,$F4,$E7,$FD,$63,$B7,$DC,$7F
	.dc.b	$C4,$D5,$FE,$27,$FA,$BF,$F4,$4C,$7C,$8B,$E3,$5F,$2A,$3F,$CC,$AF
	.dc.b	$1E,$4C,$C6,$E3,$07,$5E,$5C,$C7,$FB,$B2,$3F,$17,$CC,$F9,$EC,$CA
	.dc.b	$AF,$CD,$4B,$E4,$79,$DD,$FC,$4C,$8C,$1E,$B7,$BE,$BF,$E9,$18,$FC
	.dc.b	$CB,$6F,$FC,$97,$8F,$FA,$FA,$73,$CB,$C3,$92,$46,$0E,$2F,$AB,$EA
	.dc.b	$FE,$4D,$FF,$95,$1B,$87,$E7,$F0,$FF,$5B,$5F,$EA,$27,$2F,$E4,$9F
	.dc.b	$93,$5C,$75,$E9,$B7,$57,$B7,$8F,$2F,$97,$91,$6F,$C1,$E8,$78,$CC
	.dc.b	$67,$FA,$FC,$BF,$A5,$DF,$87,$D4,$F4,$1E,$9F,$8D,$B1,$E5,$F5,$7D
	.dc.b	$86,$31,$F6,$3E,$FF,$8D,$93,$EB,$F9,$BB,$5F,$E7,$4F,$99,$FB,$53
	.dc.b	$FC,$19,$B1,$9F,$7B,$F0,$31,$8D,$3F,$82,$2F,$27,$26,$BC,$6B,$87
	.dc.b	$F1,$0F,$AB,$7C,$F7,$E2,$9B,$19,$F0,$31,$39,$E7,$93,$C7,$DD,$BA
	.dc.b	$FF,$A2,$BE,$AD,$DB,$F9,$46,$D7,$F1,$E7,$F2,$4D,$8D,$70,$48,$D7
	.dc.b	$15,$BF,$F4,$AF,$D5,$FF,$29,$9F,$E4,$ED,$8C,$F9,$FF,$A2,$8C,$71
	.dc.b	$7F,$25,$AF,$F1,$99,$FC,$B8,$B5,$FE,$A0,$7F,$C6,$A6,$76,$FD,$5F
	.dc.b	$D5,$FF,$A7,$63,$F9,$AB,$67,$87,$BD,$A7,$7E,$14,$AD,$F7,$59,$FF
	.dc.b	$56,$DB,$4E,$A7,$6E,$AB,$7F,$D0,$BC,$5F,$F5,$3B,$7D,$CF,$DE,$C3
	.dc.b	$E7,$EC,$63,$CC,$95,$C7,$D5,$B9,$FF,$AF,$3E,$AF,$F6,$9C,$3F,$D1
	.dc.b	$DB,$5F,$CE,$1F,$F1,$83,$FD,$49,$CC,$FF,$AF,$57,$C3,$7F,$0E,$6C
	.dc.b	$EF,$DE,$C4,$E7,$8A,$4F,$3F,$76,$E1,$FE,$CE,$F2,$3F,$52,$DF,$1B
	.dc.b	$F8,$CC,$7F,$BE,$2B,$FB,$1B,$AC,$FF,$AF,$7D,$5B,$E8,$3F,$8F,$B6
	.dc.b	$BF,$B4,$BF,$E2,$D6,$C7,$F2,$4F,$AB,$7C,$3D,$3F,$93,$FD,$56,$F3
	.dc.b	$FF,$70,$CE,$AF,$FB,$9A,$F0,$79,$1C,$63,$2F,$5F,$0A,$46,$0E,$2F
	.dc.b	$87,$86,$47,$F7,$EC,$1F,$41,$8C,$7C,$3C,$5E,$73,$1B,$7F,$7D,$CF
	.dc.b	$E7,$CF,$FB,$DB,$EA,$FF,$25,$3F,$E0,$16,$E3,$FD,$CB,$1F,$C3,$3F
	.dc.b	$E0,$C8,$CD,$7F,$BD,$DF,$E4,$ED,$E0,$FE,$93,$1A,$83,$A6,$F8,$7A
	.dc.b	$3E,$4F,$45,$BE,$DE,$97,$23,$15,$E7,$7A,$FB,$92,$76,$79,$5F,$0F
	.dc.b	$A1,$23,$E3,$7C,$9D,$BF,$81,$E9,$79,$8C,$6D,$F0,$5F,$4F,$81,$2F
	.dc.b	$E7,$7C,$75,$BF,$91,$E1,$7B,$8C,$61,$F2,$5F,$83,$F3,$52,$FE,$B7
	.dc.b	$BF,$6F,$E8,$7C,$0F,$C6,$31,$87,$D1,$7E,$4F,$D2,$CB,$FA,$11,$69
	.dc.b	$5F,$E3,$7F,$A8,$FC,$C4,$62,$BF,$C5,$31,$89,$FF,$1E,$7D,$5B,$E5
	.dc.b	$FF,$BE,$CE,$6B,$7D,$78,$33,$E0,$58,$FF,$25,$CF,$FB,$71,$BE,$BF
	.dc.b	$F8,$1B,$EA,$DF,$57,$D2,$3A,$FB,$F2,$B6,$72,$7C,$3C,$72,$3F,$35
	.dc.b	$EC,$DD,$FD,$40,$64,$E2,$F4,$9B,$E0,$F8,$4F,$B3,$E2,$E6,$1E,$88
	.dc.b	$B7,$D0,$E7,$F1,$5F,$08,$F5,$75,$5A,$F7,$FD,$5F,$C6,$31,$5C,$72
	.dc.b	$47,$CA,$E3,$95,$FE,$70,$F5,$DE,$36,$30,$F0,$72,$BD,$DF,$F1,$6B
	.dc.b	$6F,$E6,$E1,$89,$FF,$18,$FD,$5F,$F2,$8B,$37,$E6,$3F,$BA,$5F,$F9
	.dc.b	$9A,$34,$7F,$A4,$DF,$F4,$0B,$67,$E0,$3A,$F8,$3E,$BE,$D4,$AF,$F4
	.dc.b	$3D,$FB,$1E,$31,$E2,$F7,$A4,$60,$EA,$F9,$B4,$91,$FE,$90,$FA,$8F
	.dc.b	$66,$5E,$09,$F5,$F9,$B2,$BC,$B1,$67,$EA,$7B,$9F,$E2,$EE,$7F,$F4
	.dc.b	$93,$6B,$FD,$07,$87,$F6,$E9,$CF,$B9,$FE,$56,$AF,$F0,$0D,$F2,$71
	.dc.b	$19,$75,$E5,$CC,$3C,$DE,$AD,$BF,$0F,$97,$CC,$C6,$3E,$27,$B7,$AB
	.dc.b	$2F,$FD,$5B,$F5,$3E,$0A,$F3,$CF,$8F,$E7,$E5,$7F,$AC,$B3,$FE,$A0
	.dc.b	$FF,$A7,$14,$7F,$9B,$AE,$CF,$48,$C5,$65,$91,$FE,$BC,$F9,$47,$CF
	.dc.b	$63,$3F,$CE,$46,$27,$53,$FE,$89,$9F,$D5,$4E,$5F,$F4,$20,$70,$FF
	.dc.b	$A1,$DA,$F9,$FC,$DE,$4F,$F9,$FD,$B9,$1A,$63,$CD,$98,$7F,$AB,$27
	.dc.b	$FC,$2C,$D8,$E7,$F5,$7F,$D8,$E3,$FD,$62,$DB,$F2,$B5,$FF,$4D,$35
	.dc.b	$FF,$62,$E0,$F4,$3C,$AD,$AF,$F5,$DB,$67,$DE,$D7,$DD,$E9,$70,$24
	.dc.b	$F0,$79,$F9,$32,$39,$B6,$79,$FD,$0F,$80,$93,$E0,$7E,$2F,$C2,$48
	.dc.b	$F8,$4D,$9F,$A3,$A3,$E2,$3E,$59,$2B,$E4,$3F,$3B,$D3,$47,$F9,$1E
	.dc.b	$BF,$92,$52,$73,$1F,$EC,$78,$C4,$EE,$3F,$D6,$D3,$A8,$FF,$62,$4E
	.dc.b	$A7,$84,$E5,$FC,$B3,$5E,$ED,$7F,$D8,$1E,$95,$93,$49,$5E,$1D,$3F
	.dc.b	$D7,$15,$FE,$DB,$6C,$7E,$3F,$CB,$E9,$18,$9F,$D3,$7C,$B7,$52,$57
	.dc.b	$AF,$04,$78,$FE,$AF,$FC,$61,$3F,$ED,$F6,$D7,$F4,$90,$E1,$FE,$98
	.dc.b	$D7,$F4,$5F,$05,$E2,$24,$FA,$5E,$DE,$44,$8F,$99,$5F,$E3,$06,$C7
	.dc.b	$8F,$E1,$3F,$9F,$5F,$EB,$AB,$FC,$DD,$E4,$7E,$C2,$3C,$7D,$0F,$6F
	.dc.b	$15,$A9,$E1,$CB,$E9,$CB,$D8,$D8,$F6,$4A,$31,$1E,$C6,$BF,$03,$96
	.dc.b	$C9,$A4,$8F,$5F,$07,$95,$F8,$ED,$93,$4C,$6B,$27,$D2,$D7,$F5,$FF
	.dc.b	$09,$CF,$84,$D2,$47,$C0,$F3,$3E,$B7,$7F,$E6,$A3,$F8,$A7,$AF,$8D
	.dc.b	$7F,$7E,$D8,$FB,$35,$5F,$BB,$F1,$57,$44,$D3,$11,$BF,$AB,$7C,$AF
	.dc.b	$1F,$0D,$EF,$53,$51,$BF,$95,$B9,$7F,$84,$F5,$F3,$61,$CA,$31,$AF
	.dc.b	$E3,$E1,$5D,$FE,$43,$CD,$C1,$FE,$76,$BF,$E7,$5D,$39,$78,$9F,$E1
	.dc.b	$FB,$7A,$39,$8F,$F0,$C4,$2F,$D5,$3E,$BF,$85,$F8,$5E,$6F,$FA,$36
	.dc.b	$FF,$B9,$46,$5E,$A2,$DF,$0B,$98,$EB,$3F,$16,$3E,$37,$99,$D3,$79
	.dc.b	$5C,$AD,$3F,$2F,$E2,$37,$39,$8C,$3E,$0F,$AB,$7A,$1E,$6C,$64,$E6
	.dc.b	$3E,$D4,$BC,$3B,$12,$79,$47,$DF,$FA,$87,$D5,$FE,$07,$C7,$7C,$35
	.dc.b	$35,$1B,$F7,$6E,$1F,$0A,$F8,$B6,$3A,$7C,$97,$E2,$FE,$AF,$E2,$CF
	.dc.b	$C8,$6B,$FE,$49,$A4,$F8,$FD,$2E,$33,$D6,$30,$F7,$F3,$7F,$2D,$F5
	.dc.b	$6F,$5A,$9E,$B8,$F6,$6B,$6B,$81,$E7,$7E,$07,$9F,$A8,$C1,$EB,$7F
	.dc.b	$45,$F5,$7F,$A9,$F0,$3F,$55,$F5,$3F,$EF,$DA,$FC,$0B,$5F,$F3,$14
	.dc.b	$EF,$C1,$FE,$4C,$FF,$B8,$E3,$F0,$EE,$5F,$F3,$93,$53,$B7,$9F,$99
	.dc.b	$FB,$3F,$AB,$FE,$6B,$1F,$90,$6B,$A3,$AF,$9B,$E8,$99,$18,$3C,$3E
	.dc.b	$A7,$EF,$7E,$AD,$EE,$D3,$DF,$1F,$03,$E2,$2D,$70,$3D,$2F,$C4,$F8
	.dc.b	$7D,$46,$0F,$6B,$F2,$3D,$0F,$AB,$EE,$49,$F8,$31,$F3,$3E,$1F,$51
	.dc.b	$83,$E2,$BF,$43,$E2,$3F,$89,$70,$FF,$84,$E7,$E7,$FB,$99,$7E,$5F
	.dc.b	$C3,$57,$E8,$D7,$D8,$8F,$E3,$5E,$CB,$EA,$FF,$25,$E7,$4B,$FD,$CC
	.dc.b	$E5,$BE,$5F,$0A,$91,$83,$7E,$8F,$49,$91,$83,$9F,$CF,$FE,$7F,$EA
	.dc.b	$E4,$60,$EF,$F4,$1E,$49,$91,$83,$C7,$E8,$7E,$C7,$E1,$24,$60,$F3
	.dc.b	$F9,$DE,$C9,$91,$83,$D7,$F0,$DF,$CE,$7E,$25,$D1,$4F,$87,$1F,$13
	.dc.b	$AA,$D7,$03,$E1,$BF,$23,$D7,$D4,$60,$F8,$EF,$F4,$5F,$AB,$FE,$C1
	.dc.b	$1F,$C8,$9F,$ED,$0D,$EE,$D7,$7E,$FD,$BE,$1E,$E4,$AE,$11,$FD,$3F
	.dc.b	$EA,$DF,$17,$78,$E5,$7F,$2E,$9C,$B7,$83,$87,$CB,$C8,$C1,$FC,$D3
	.dc.b	$EA,$FF,$90,$37,$D7,$FE,$B9,$1F,$CA,$23,$5E,$BE,$3E,$2F,$3E,$9C
	.dc.b	$9E,$51,$FD,$8F,$BF,$FA,$74,$7F,$B7,$DF,$DB,$79,$BF,$DB,$5B,$E6
	.dc.b	$FF,$89,$5D,$C6,$7C,$7E,$5F,$09,$2F,$FA,$4D,$7F,$7A,$7F,$A5,$FD
	.dc.b	$5F,$F5,$36,$F8,$5F,$DC,$E3,$FB,$0C,$6A,$71,$C5,$9F,$02,$4F,$28
	.dc.b	$E1,$EE,$F1,$F7,$64,$F6,$8F,$EF,$DF,$56,$F0,$EF,$BB,$2B,$FA,$EC
	.dc.b	$E5,$FE,$BF,$F5,$6D,$BF,$C0,$CF,$F8,$6A,$33,$F1,$18,$9F,$F0,$57
	.dc.b	$7F,$A2,$5C,$5F,$E1,$38,$FE,$FD,$18,$E3,$FF,$26,$46,$27,$5C,$5C
	.dc.b	$BC,$3E,$7F,$57,$30,$FE,$E3,$69,$EF,$C1,$E0,$F0,$E5,$6A,$3F,$C3
	.dc.b	$9F,$57,$FD,$72,$3F,$BC,$AF,$80,$C9,$AE,$1F,$F1,$17,$D5,$FF,$35
	.dc.b	$9F,$E5,$FA,$FF,$8A,$DB,$B7,$F9,$8E,$7F,$9D,$1F,$0F,$0F,$60,$0B
	.dc.b	$D5,$EA,$76,$E1,$EF,$D5,$E0,$E2,$97,$FE,$0E,$BF,$C3,$3D,$0E,$7E
	.dc.b	$A7,$07,$55,$E8,$8F,$4F,$8B,$EA,$F1,$A5,$73,$8F,$6F,$A5,$A8,$C1
	.dc.b	$DD,$FF,$21,$7D,$57,$F8,$A2,$3F,$C3,$FF,$57,$FE,$94,$6F,$B9,$AF
	.dc.b	$88,$DF,$81,$67,$FC,$4B,$7C,$5F,$42,$8F,$F2,$77,$C5,$7E,$CE,$7D
	.dc.b	$9E,$93,$3F,$F2,$93,$63,$FC,$5F,$3B,$E5,$FF,$2E,$3F,$BE,$8F,$88
	.dc.b	$E7,$93,$FD,$2C,$D8,$FA,$4E,$E7,$FC,$C9,$F5,$7F,$8F,$AF,$F4,$E5
	.dc.b	$FD,$5B,$FF,$1E,$B5,$FA,$13,$B7,$EE,$3F,$9D,$AE,$7E,$AF,$F9,$A6
	.dc.b	$35,$27,$57,$FE,$A9,$7F,$7D,$1D,$3B,$F8,$1C,$7C,$5E,$5B,$7C,$2E
	.dc.b	$49,$5F,$E5,$0B,$F9,$99,$7A,$B6,$DC,$E6,$FC,$51,$E3,$F1,$7F,$63
	.dc.b	$3F,$EB,$93,$D2,$E1,$C5,$A3,$11,$E7,$6B,$F0,$3C,$F7,$D5,$FF,$AC
	.dc.b	$6B,$F9,$CB,$63,$33,$DC,$F8,$4E,$1F,$F5,$C7,$D2,$FF,$A9,$53,$48
	.dc.b	$FF,$AF,$7E,$47,$FD,$E7,$9F,$FA,$6E,$3F,$D7,$DB,$FF,$9C,$9F,$D7
	.dc.b	$57,$EB,$FE,$AF,$FD,$23,$1F,$ED,$7B,$E3,$5F,$07,$D0,$B3,$FE,$DB
	.dc.b	$6B,$FB,$79,$AF,$F9,$BA,$7B,$79,$F8,$37,$C8,$F9,$DC,$4F,$8F,$E6
	.dc.b	$78,$98,$BF,$DC,$07,$AF,$C4,$49,$E0,$7A,$DF,$1E,$BF,$E9,$58,$FF
	.dc.b	$47,$B6,$BF,$4F,$E1,$6B,$9B,$7C,$B2,$F7,$16,$BF,$84,$7D,$5F,$C8
	.dc.b	$6E,$DE,$AF,$03,$E0,$CE,$72,$FF,$64,$3F,$06,$B8,$F0,$6A,$F6,$F0
	.dc.b	$F1,$A6,$5F,$EA,$4B,$FE,$58,$D8,$EB,$E7,$C8,$C1,$B7,$C3,$E9,$C8
	.dc.b	$E5,$F1,$77,$2F,$4F,$A9,$E6,$31,$A7,$99,$F4,$FC,$54,$BF,$17,$CC
	.dc.b	$DB,$FE,$51,$F2,$FF,$5E,$7F,$D7,$2D,$8C,$FB,$9F,$82,$8C,$63,$FE
	.dc.b	$E9,$5E,$4E,$4D,$78,$D7,$0F,$F6,$1F,$D5,$BE,$73,$F1,$6D,$8C,$F8
	.dc.b	$18,$9C,$F6,$C9,$E3,$EE,$DD,$7F,$9F,$7D,$5F,$E1,$E7,$FD,$9E,$D8
	.dc.b	$CF,$81,$F9,$88,$C6,$DF,$EF,$F5,$C1,$93,$5C,$57,$0F,$F6,$CF,$D5
	.dc.b	$FF,$2F,$6E,$FF,$A2,$8F,$F7,$FC,$60,$FF,$6F,$9F,$E3,$62,$ED,$FA
	.dc.b	$87,$F8,$41,$83,$FD,$CB,$F5,$7F,$D8,$5B,$A6,$FD,$DF,$D7,$CE,$63
	.dc.b	$FA,$E3,$FE,$D7,$9E,$3A,$EB,$AF,$6F,$0C,$E4,$F6,$E8,$B8,$7F,$65
	.dc.b	$F1,$5E,$D7,$EF,$1B,$5F,$EF,$85,$F3,$B9,$9D,$79,$92,$B5,$EA,$DC
	.dc.b	$3F,$B8,$7D,$5F,$EC,$79,$55,$FD,$F6,$53,$98,$C2,$F3,$FF,$86,$B9
	.dc.b	$9F,$F0,$23,$FE,$B2,$9F,$F0,$35,$74,$9C,$FC,$2F,$0F,$77,$8B,$CB
	.dc.b	$8E,$5F,$03,$BF,$B9,$8B,$FF,$03,$78,$DF,$E3,$D9,$D4,$7F,$1D,$F1
	.dc.b	$3F,$8F,$CF,$FB,$FA,$73,$28,$E0,$FF,$83,$FE,$AD,$F3,$FF,$C9,$9B
	.dc.b	$5C,$1F,$86,$D8,$FE,$57,$F5,$6F,$86,$A7,$F2,$EF,$AB,$DE,$1F,$E1
	.dc.b	$B9,$D5,$F8,$97,$83,$C8,$E3,$19,$7A,$F8,$52,$30,$71,$7C,$3C,$12
	.dc.b	$3D,$B8,$3E,$7F,$18,$F8,$78,$7C,$E6,$38,$BE,$94,$33,$FA,$CF,$F8
	.dc.b	$C7,$EA,$FF,$31,$3A,$6E,$3F,$83,$C7,$F1,$AF,$CC,$8C,$D7,$F9,$0D
	.dc.b	$FE,$5D,$1F,$89,$6C,$64,$FE,$A6,$E1,$3F,$BE,$8E,$1E,$9F,$CB,$E9
	.dc.b	$DB,$C3,$D3,$93,$E1,$8B,$7E,$9F,$6F,$A9,$8C,$BD,$4F,$83,$E0,$E5
	.dc.b	$FC,$8F,$92,$B7,$F0,$7D,$1F,$29,$8C,$3E,$13,$E8,$F0,$65,$FD,$0F
	.dc.b	$8D,$B7,$F2,$7C,$3F,$69,$8C,$3E,$53,$F0,$3E,$66,$5F,$D8,$F8,$5B
	.dc.b	$7F,$47,$E0,$BE,$29,$8C,$3E,$93,$F2,$3E,$AE,$5F,$D1,$8B,$5F,$E6
	.dc.b	$9F,$A8,$FC,$CB,$7B,$7F,$C0,$3F,$56,$F9,$6F,$F0,$1D,$6B,$2F,$C1
	.dc.b	$BF,$89,$C7,$F4,$ED,$C3,$94,$63,$AF,$8F,$AF,$23,$F5,$BF,$56,$FA
	.dc.b	$BC,$5F,$8F,$E0,$7B,$98,$CB,$BB,$E2,$E3,$98,$7E,$6B,$D9,$B7,$EB
	.dc.b	$E2,$F3,$98,$C3,$CE,$FA,$BE,$06,$5F,$A2,$2D,$7C,$3F,$8A,$F7,$9F
	.dc.b	$E2,$2D,$3F,$BC,$C6,$6B,$DF,$F5,$7F,$10,$DB,$83,$E1,$F0,$FC,$83
	.dc.b	$FC,$57,$F0,$3F,$EB,$58,$C9,$FC,$3B,$85,$EF,$7F,$8C,$1B,$1A,$C3
	.dc.b	$FD,$3D,$18,$7F,$D4,$1F,$57,$FC,$A5,$37,$E5,$FF,$BB,$D7,$F3,$38
	.dc.b	$D0,$F8,$5F,$F1,$EB,$63,$F8,$FB,$A8,$CF,$B7,$27,$F9,$1D,$34,$C6
	.dc.b	$1E,$2F,$80,$E6,$63,$8B,$F8,$3E,$5F,$DF,$E7,$CF,$E5,$F8,$0F,$F2
	.dc.b	$6F,$D3,$FA,$EB,$71,$E4,$F3,$26,$7E,$48,$B7,$FE,$BF,$F6,$FF,$C6
	.dc.b	$57,$FC,$51,$B5,$FD,$47,$0F,$EE,$95,$BF,$73,$FC,$B7,$5F,$DA,$E3
	.dc.b	$93,$88,$CB,$AF,$26,$5F,$F4,$2B,$7A,$B8,$C3,$C5,$E5,$F4,$31,$97
	.dc.b	$8D,$EF,$EA,$CC,$3F,$A5,$FD,$47,$86,$BD,$13,$E4,$F9,$E9,$5F,$D4
	.dc.b	$34,$FE,$9F,$5E,$D9,$FF,$39,$D9,$FF,$40,$63,$FE,$EB,$8C,$5F,$FA
	.dc.b	$36,$11,$87,$FD,$20,$D7,$FD,$41,$F2,$6F,$9D,$E7,$08,$C5,$73,$6C
	.dc.b	$7F,$A6,$5A,$F2,$F0,$BF,$D0,$F1,$F9,$08,$FF,$7F,$3C,$31,$FE,$92
	.dc.b	$E6,$19,$3C,$BF,$0B,$70,$F5,$36,$7F,$D5,$0D,$7F,$D7,$1F,$57,$FD
	.dc.b	$92,$7E,$03,$63,$FD,$5C,$D7,$7E,$BF,$BB,$07,$A1,$49,$7E,$F7,$E1
	.dc.b	$DB,$D7,$A4,$9E,$EF,$3F,$26,$47,$C7,$6E,$3F,$1F,$E8,$3C,$04,$9F
	.dc.b	$98,$F8,$FC,$59,$1F,$3D,$B3,$F0,$3E,$27,$E2,$92,$7E,$2B,$F2,$3D
	.dc.b	$1F,$F1,$34,$7F,$B6,$F6,$46,$2B,$FC,$69,$18,$9F,$F2,$74,$0F,$A7
	.dc.b	$3B,$71,$2F,$43,$11,$F6,$1B,$19,$AE,$73,$96,$C6,$63,$FD,$AC,$D8
	.dc.b	$F3,$4F,$23,$AE,$CF,$2A,$92,$7A,$C6,$5B,$E0,$F0,$23,$83,$C7,$F5
	.dc.b	$5F,$E2,$C7,$FD,$FC,$DC,$7F,$CA,$CD,$7E,$13,$5F,$D1,$7C,$0F,$90
	.dc.b	$93,$E2,$7C,$FE,$4C,$8F,$2B,$67,$FD,$E9,$5F,$E6,$CF,$27,$F6,$8D
	.dc.b	$8F,$F3,$31,$E0,$E9,$FF,$27,$38,$11,$B7,$D3,$EC,$7F,$9B,$70,$B1
	.dc.b	$EE,$8C,$3E,$C6,$BE,$CE,$3A,$F4,$6D,$1A,$6B,$47,$9F,$49,$5F,$49
	.dc.b	$E3,$A7,$F0,$F8,$F3,$B6,$CD,$35,$CC,$F1,$69,$2B,$F7,$2F,$3C,$1F
	.dc.b	$0F,$49,$3F,$C4,$DE,$DB,$FF,$A3,$FB,$3F,$86,$FA,$97,$7F,$F4,$96
	.dc.b	$1F,$E7,$6C,$3F,$CF,$7A,$30,$D4,$7E,$03,$1A,$FD,$08,$B3,$B8,$E3
	.dc.b	$3A,$FA,$A7,$CA,$70,$F0,$5E,$DD,$35,$1B,$F8,$4B,$87,$FA,$32,$F8
	.dc.b	$B6,$3A,$7C,$7F,$8B,$4F,$F3,$A4,$FF,$A4,$9A,$B7,$FF,$39,$DA,$BC
	.dc.b	$9E,$8F,$2A,$ED,$18,$7C,$1E,$1F,$BB,$CA,$FF,$45,$C7,$F5,$D8,$F6
	.dc.b	$1E,$DF,$0E,$C7,$03,$D0,$FC,$1F,$2B,$96,$B2,$7D,$91,$F1,$7E,$13
	.dc.b	$51,$83,$E0,$3F,$3B,$EA,$9E,$77,$8F,$09,$5F,$EB,$09,$CB,$7C,$3E
	.dc.b	$15,$23,$07,$FA,$DB,$EA,$1F,$55,$F6,$BC,$67,$C0,$53,$EE,$CE,$AB
	.dc.b	$87,$C0,$E2,$E4,$60,$E9,$F2,$3F,$84,$FA,$AF,$E8,$51,$FE,$C4,$6B
	.dc.b	$F4,$FB,$BF,$DC,$18,$2B,$BF,$C1,$F8,$19,$5E,$78,$F1,$F9,$9F,$90
	.dc.b	$FA,$A7,$AB,$4C,$D7,$B7,$6A,$F7,$3F,$9C,$62,$BF,$6C,$FB,$BC,$EB
	.dc.b	$5C,$4F,$8C,$FE,$7B,$EA,$BF,$D7,$7D,$BF,$EB,$CF,$A8,$FF,$80,$35
	.dc.b	$FE,$E0,$6B,$DC,$4F,$0D,$BF,$DF,$CF,$EB,$63,$4E,$5F,$E9,$8D,$78
	.dc.b	$F8,$DE,$1F,$33,$F5,$FF,$54,$FA,$BA,$66,$BF,$DE,$37,$FC,$55,$B5
	.dc.b	$97,$BF,$9D,$EB,$F5,$3F,$73,$F5,$7F,$C3,$67,$FD,$F2,$D7,$CB,$1F
	.dc.b	$07,$D0,$E1,$1D,$A3,$0F,$83,$E1,$2B,$EC,$9F,$91,$1F,$27,$E4,$7E
	.dc.b	$C7,$B7,$2B,$CF,$1F,$37,$D8,$D4,$60,$F5,$BF,$47,$E1,$BF,$87,$F1
	.dc.b	$7F,$84,$A7,$E8,$7B,$59,$7E,$0F,$C2,$D7,$E6,$D7,$D9,$8F,$E2,$FE
	.dc.b	$C3,$EA,$FF,$2D,$E6,$CB,$FE,$F9,$D1,$FE,$F8,$AD,$46,$3A,$3F,$C0
	.dc.b	$0F,$F9,$07,$CB,$E7,$84,$8D,$1F,$E0,$5A,$FE,$2E,$FE,$0D,$7A,$3E
	.dc.b	$2A,$93,$C0,$FF,$26,$47,$9B,$E7,$FF,$55,$F5,$72,$75,$69,$F5,$FD
	.dc.b	$06,$E7,$31,$87,$7E,$87,$EF,$7E,$0A,$46,$0E,$7E,$7F,$C4,$99,$18
	.dc.b	$3B,$FC,$27,$F3,$EF,$87,$FE,$BB,$1F,$E9,$CE,$AF,$97,$3F,$33,$DD
	.dc.b	$B3,$C0,$F3,$BF,$43,$D8,$D4,$60,$F5,$BF,$D2,$BE,$AF,$F3,$11,$FA
	.dc.b	$95,$E0,$E4,$D7,$83,$BF,$73,$87,$BD,$2F,$E5,$5F,$F6,$27,$E6,$7D
	.dc.b	$5B,$E2,$BF,$AB,$AE,$3A,$62,$33,$C1,$AF,$94,$97,$CA,$3F,$AF,$FD
	.dc.b	$5F,$F9,$CA,$3E,$C3,$FC,$C1,$FE,$00,$DE,$C7,$67,$E2,$23,$84,$EB
	.dc.b	$E2,$39,$FA,$72,$39,$C7,$F6,$CE,$FF,$EB,$38,$72,$9F,$F2,$84,$7F
	.dc.b	$09,$8C,$47,$F5,$BF,$37,$87,$C6,$F1,$F1,$F9,$7C,$34,$AF,$C7,$60
	.dc.b	$9F,$EF,$3F,$57,$FD,$8E,$7F,$24,$FE,$59,$BE,$16,$AB,$7C,$5C,$3C
	.dc.b	$09,$5C,$23,$9F,$BF,$D3,$E0,$24,$F4,$8F,$F0,$3F,$D5,$BC,$3B,$EF
	.dc.b	$4A,$FD,$14,$E5,$FD,$27,$D5,$FF,$89,$1B,$F1,$9F,$E1,$08,$FF,$30
	.dc.b	$46,$B8,$3F,$C2,$7E,$07,$A6,$3F,$99,$3F,$B2,$E4,$F8,$ED,$E8,$FD
	.dc.b	$DB,$B8,$CF,$87,$CB,$EA,$E5,$FE,$D6,$93,$DB,$7E,$FE,$1C,$AD,$47
	.dc.b	$F8,$93,$EA,$FF,$B5,$C7,$EF,$57,$C1,$64,$D7,$07,$F8,$AB,$EA,$FF
	.dc.b	$A0,$4F,$F3,$DA,$E1,$EC,$E7,$6D,$70,$EF,$D6,$95,$FC,$27,$0F,$86
	.dc.b	$FF,$51,$7E,$F3,$9E,$1E,$C7,$7E,$BF,$07,$AB,$67,$94,$61,$E9,$E4
	.dc.b	$79,$7C,$89,$18,$3C,$1E,$9F,$A6,$64,$60,$FE,$39,$F5,$4F,$89,$FF
	.dc.b	$23,$FD,$5F,$FA,$96,$7F,$90,$35,$FE,$D1,$3B,$78,$F7,$FF,$93,$A3
	.dc.b	$11,$AA,$C4,$FF,$94,$BE,$33,$F6,$1C,$3F,$CD,$BD,$2F,$E0,$57,$FD
	.dc.b	$CD,$BD,$9F,$A8,$AD,$CF,$FA,$ED,$FD,$AC,$67,$E2,$3A,$1D,$7E,$27
	.dc.b	$FC,$BF,$F5,$7F,$89,$BF,$F6,$16,$0E,$5F,$F3,$1B,$63,$F9,$E4,$ED
	.dc.b	$3F,$8F,$CB,$FA,$5B,$92,$78,$DD,$BC,$6E,$EC,$AE,$71,$E2,$DB,$BF
	.dc.b	$24,$9E,$F1,$E6,$DF,$51,$83,$C6,$FA,$BC,$6F,$F3,$24,$7F,$AA,$71
	.dc.b	$FE,$49,$BF,$FA,$AE,$3C,$F3,$A7,$FD,$63,$3A,$78,$79,$EF,$AB,$E1
	.dc.b	$3F,$CB,$DB,$19,$9D,$C7,$FA,$85,$7D,$25,$23,$FE,$A5,$7F,$D4,$DF
	.dc.b	$45,$FC,$87,$1F,$E3,$F2,$8C,$6F,$FE,$C1,$AF,$F7,$A7,$D5,$FF,$A3
	.dc.b	$2B,$FB,$BD,$69,$FE,$F3,$C9,$5F,$F7,$D3,$5E,$97,$FD,$CD,$FF,$37
	.dc.b	$B7,$93,$47,$1E,$37,$CB,$E2,$78,$7C,$C7,$33,$17,$FE,$43,$0F,$7F
	.dc.b	$8C,$93,$C0,$F0,$3E,$5D,$7F,$D2,$D1,$FE,$05,$6D,$7E,$6F,$C1,$D7
	.dc.b	$36,$F9,$65,$EE,$2D,$7F,$94,$FE,$AF,$DA,$DD,$FD,$5D,$FF,$0E,$73
	.dc.b	$97,$F4,$C7,$E0,$D7,$1E,$0D,$5E,$DE,$1E,$2C,$CB,$FC,$35,$7F,$C7
	.dc.b	$9B,$1D,$7C,$E9,$18,$36,$F8,$7D,$49,$1C,$BE,$3E,$E5,$E9,$F5,$7C
	.dc.b	$C6,$34,$F3,$3E,$9F,$8D,$97,$E2,$FA,$9B,$7F,$E7,$3F,$99,$FA,$F3
	.dc.b	$FE,$2F,$6C,$67,$DE,$FC,$24,$63,$1F,$ED,$AB,$C8,$C9,$AF,$22,$E1
	.dc.b	$FE,$39,$FA,$B7,$CE,$FE,$35,$B1,$9F,$07,$13,$9E,$D9,$3C,$7D,$DB
	.dc.b	$AF,$FA,$27,$EA,$FF,$0F,$3F,$E4,$56,$C6,$7C,$0F,$CD,$46,$36,$FF
	.dc.b	$01,$AE,$0C,$9A,$E2,$B8,$7F,$93,$FE,$AD,$BF,$E8,$DB,$19,$EB,$C4
	.dc.b	$FE,$9E,$3F,$C5,$47,$F9,$5C,$B9,$FF,$A7,$DF,$F1,$61,$9D,$FF,$5D
	.dc.b	$F5,$69,$A7,$3F,$72,$D7,$F9,$86,$5B,$70,$38,$F6,$59,$D6,$9D,$0E
	.dc.b	$BD,$36,$3F,$CD,$5E,$2F,$F9,$21,$BE,$CF,$EF,$E1,$F5,$76,$31,$E6
	.dc.b	$4A,$E3,$E9,$DD,$BF,$D7,$5F,$57,$FB,$0E,$6F,$F1,$7B,$6B,$FD,$06
	.dc.b	$FF,$9F,$1F,$F3,$D3,$99,$F8,$0B,$E1,$BF,$88,$36,$77,$EE,$62,$73
	.dc.b	$C5,$27,$9F,$B7,$70,$FF,$66,$78,$DF,$E2,$99,$FF,$44,$36,$33,$E2
	.dc.b	$FF,$1D,$8C,$67,$FE,$46,$FA,$B7,$C1,$FF,$21,$6E,$1F,$E9,$17,$FC
	.dc.b	$AE,$D8,$FE,$4D,$F5,$6F,$86,$A7,$F2,$9F,$AB,$DE,$7F,$ED,$F9,$D5
	.dc.b	$FF,$A6,$D7,$83,$C8,$E3,$19,$7A,$F8,$52,$30,$71,$7C,$3C,$32,$3F
	.dc.b	$D4,$D8,$3E,$1B,$18,$F8,$78,$BC,$E6,$36,$FF,$32,$E7,$F3,$E7,$FD
	.dc.b	$E9,$F5,$7F,$93,$C7,$FD,$60,$D8,$FF,$7C,$63,$F8,$69,$CC,$7F,$47
	.dc.b	$9F,$E4,$B1,$FE,$C2,$6C,$7E,$75,$E5,$7F,$E7,$47,$30,$7F,$6D,$3C
	.dc.b	$7D,$2F,$95,$D1,$D5,$ED,$E9,$49,$F2,$C5,$BF,$4F,$B7,$E3,$31,$8F
	.dc.b	$77,$CD,$F0,$52,$FE,$3F,$C9,$5B,$F8,$3C,$FE,$B3,$18,$79,$DF,$77
	.dc.b	$7C,$BF,$9F,$F1,$96,$FE,$4F,$83,$F0,$CC,$61,$EF,$7E,$37,$CD,$4B
	.dc.b	$FA,$FE,$FD,$BF,$A3,$F3,$5F,$2C,$C6,$1F,$1D,$F9,$DF,$57,$2F,$EA
	.dc.b	$C5,$A5,$7F,$A8,$3E,$AB,$F3,$31,$8A,$FF,$7A,$C6,$27,$FD,$4B,$F5
	.dc.b	$6F,$97,$FF,$00,$4E,$6B,$7D,$58,$33,$DF,$63,$FD,$E9,$2F,$5B,$89
	.dc.b	$FE,$09,$FA,$A7,$D5,$E2,$B7,$F0,$1D,$8C,$6B,$E5,$3F,$91,$61,$FD
	.dc.b	$36,$7C,$FC,$7E,$43,$F5,$9E,$C5,$8F,$0F,$17,$A0,$C5,$F3,$17,$FE
	.dc.b	$51,$9F,$3F,$C0,$7B,$8E,$0E,$0F,$5F,$C4,$FC,$23,$D5,$D5,$63,$DF
	.dc.b	$F5,$5F,$18,$C6,$1D,$F0,$47,$CA,$E2,$95,$FC,$F7,$93,$FD,$A7,$EB
	.dc.b	$7F,$A9,$A3,$E4,$CF,$F8,$BF,$91,$EE,$53,$8E,$5E,$B8,$CD,$73,$3F
	.dc.b	$A4,$43,$EA,$FF,$96,$37,$E0,$BF,$C7,$5F,$2F,$F2,$E3,$FA,$74,$E8
	.dc.b	$FF,$48,$57,$38,$CF,$C2,$7F,$90,$67,$4D,$F6,$A5,$6E,$97,$F0,$23
	.dc.b	$17,$FD,$83,$5F,$F7,$8B,$FD,$FA,$BE,$A3,$63,$37,$DF,$DF,$F0,$79
	.dc.b	$93,$0E,$DF,$4F,$F0,$31,$E3,$3E,$5F,$36,$57,$82,$2C,$FE,$07,$DB
	.dc.b	$FF,$1A,$E4,$AF,$8F,$18,$7E,$4E,$1E,$77,$3E,$DF,$F9,$86,$7F,$B7
	.dc.b	$C7,$BD,$B7,$D6,$75,$1B,$F2,$A5,$7A,$3D,$5B,$7E,$3F,$33,$A1,$8C
	.dc.b	$BC,$91,$FE,$01,$C1,$3E,$1F,$84,$98,$7C,$FF,$A9,$F0,$E1,$C3,$0F
	.dc.b	$F0,$1D,$79,$FE,$7E,$5F,$9E,$2C,$FE,$8E,$DF,$0D,$FE,$7D,$5E,$6D
	.dc.b	$8C,$39,$F3,$66,$1E,$18,$B7,$CE,$7F,$D1,$EF,$F8,$66,$D1,$88,$46
	.dc.b	$2B,$FD,$50,$D8,$FC,$C9,$C7,$CD,$E1,$06,$92,$78,$F9,$9A,$23,$83
	.dc.b	$FD,$69,$F2,$7F,$E1,$DB,$EA,$1C,$3E,$06,$BF,$AA,$6B,$FE,$33,$7F
	.dc.b	$8E,$47,$4F,$2F,$D9,$18,$9E,$AD,$7D,$7F,$56,$F9,$DF,$84,$DB,$FF
	.dc.b	$64,$36,$7C,$AD,$7E,$1F,$17,$F9,$56,$7F,$D9,$CD,$8F,$5B,$5D,$FC
	.dc.b	$37,$CF,$87,$C4,$D2,$57,$27,$D3,$E5,$FE,$42,$3E,$13,$6F,$E1,$35
	.dc.b	$F8,$4D,$7E,$57,$97,$73,$FE,$59,$1F,$ED,$06,$D7,$CD,$9E,$18,$3D
	.dc.b	$2A,$49,$FF,$06,$5F,$E8,$63,$0F,$FB,$6A,$78,$4E,$A3,$FC,$C3,$3A
	.dc.b	$8F,$8F,$F2,$9D,$88,$EE,$91,$E9,$FA,$BF,$F1,$9C,$FE,$41,$B7,$FC
	.dc.b	$A9,$AF,$F9,$CD,$A9,$FE,$D7,$1C,$7D,$3E,$84,$7C,$D1,$FE,$31,$6F
	.dc.b	$8F,$E1,$23,$AF,$85,$FE,$7F,$7F,$96,$AF,$F3,$8F,$93,$F9,$88,$F0
	.dc.b	$7C,$F7,$FB,$ED,$D5,$70,$7F,$35,$E5,$FF,$42,$C2,$C7,$C7,$DF,$FB
	.dc.b	$E3,$67,$4F,$F4,$8F,$DA,$39,$57,$EC,$9B,$19,$C7,$6F,$FA,$56,$32
	.dc.b	$FC,$5F,$53,$2F,$C5,$6E,$7F,$F0,$02,$3E,$1C,$63,$D9,$91,$F1,$9A
	.dc.b	$FF,$1B,$14,$FA,$DF,$85,$18,$6C,$66,$3E,$33,$5F,$DF,$AF,$A2,$D9
	.dc.b	$23,$6F,$2F,$A2,$91,$C9,$AF,$D3,$F4,$BF,$B6,$F8,$BF,$CB,$F7,$FF
	.dc.b	$42,$CF,$F6,$16,$BF,$E0,$FD,$15,$C3,$6F,$C6,$7C,$6D,$AF,$C6,$4D
	.dc.b	$7F,$18,$DA,$FF,$67,$C7,$F6,$DF,$AD,$65,$BF,$09,$EF,$D3,$94,$EA
	.dc.b	$B8,$7C,$C6,$2E,$46,$0E,$9E,$B7,$87,$5F,$F4,$34,$7A,$FA,$39,$CF
	.dc.b	$4F,$5A,$CF,$03,$C8,$F8,$3E,$1B,$D5,$E8,$FF,$A6,$A7,$FB,$BC,$65
	.dc.b	$F5,$7C,$35,$7F,$C6,$9A,$6D,$B1,$EA,$F4,$39,$E9,$27,$E1,$47,$BB
	.dc.b	$E2,$B5,$18,$3E,$33,$F2,$FE,$B5,$EA,$79,$30,$95,$FE,$17,$9C,$B7
	.dc.b	$C3,$E1,$52,$30,$7F,$87,$7E,$A9,$F5,$BF,$5F,$CF,$7C,$35,$3E,$D4
	.dc.b	$EA,$B8,$7C,$06,$2E,$46,$0E,$9E,$A7,$E0,$7E,$B5,$AD,$39,$C7,$4F
	.dc.b	$56,$D7,$03,$C8,$F8,$3C,$FF,$C5,$7D,$6F,$F8,$22,$7F,$C6,$8D,$7F
	.dc.b	$CC,$59,$FF,$44,$7D,$9A,$F1,$7D,$D5,$EA,$8F,$5F,$AB,$67,$67,$C3
	.dc.b	$7F,$31,$F5,$9F,$E1,$1F,$13,$FC,$25,$F5,$4F,$37,$F3,$8D,$BF,$F2
	.dc.b	$35,$6F,$C6,$FF,$2B,$BF,$A8,$BF,$F0,$D3,$99,$07,$0D,$F3,$FC,$3F
	.dc.b	$17,$FA,$9F,$AD,$FE,$7B,$3F,$E5,$36,$BF,$E8,$1A,$4F,$A3,$D9,$E3
	.dc.b	$3E,$48,$C3,$E6,$FA,$CF,$DA,$7D,$6B,$E0,$A9,$F0,$23,$E0,$FC,$5D
	.dc.b	$AE,$07,$B9,$F8,$BE,$CE,$A3,$07,$C2,$7E,$4F,$CB,$FD,$8F,$81,$93
	.dc.b	$F1,$A3,$E6,$FC,$5E,$A3,$07,$CA,$7E,$8F,$BB,$F3,$BD,$79,$3F,$3A
	.dc.b	$3E,$AF,$B5,$A8,$C1,$F4,$9F,$E2,$1E,$DF,$F2,$58,$FF,$3D,$78,$0F
	.dc.b	$3B,$F2,$86,$BD,$7D,$F9,$76,$BF,$D0,$17,$C5,$FF,$57,$3F,$C4,$5B
	.dc.b	$E9,$70,$77,$3E,$08,$F0,$FD,$17,$E8,$7E,$06,$57,$48,$F2,$FD,$16
	.dc.b	$A3,$07,$81,$F4,$FA,$3F,$B3,$F8,$49,$3E,$48,$F6,$F9,$3A,$8C,$1E
	.dc.b	$87,$E0,$FC,$3F,$F3,$1F,$8B,$74,$53,$35,$EE,$EA,$AF,$BA,$BE,$4C
	.dc.b	$7C,$AF,$62,$CE,$CF,$8A,$FF,$3D,$FA,$DF,$F7,$24,$7F,$A9,$DF,$C9
	.dc.b	$B7,$BB,$5D,$FB,$F7,$B8,$7B,$D2,$B8,$47,$F4,$7F,$AD,$7C,$65,$E2
	.dc.b	$95,$FE,$B3,$9C,$B7,$87,$87,$CB,$C8,$C1,$FE,$B9,$FA,$D7,$B1,$FD
	.dc.b	$4D,$FE,$4B,$1F,$EE,$96,$F9,$5F,$E8,$F7,$71,$C7,$E2,$F8,$7A,$92
	.dc.b	$30,$7F,$B1,$7B,$FF,$A6,$BF,$EF,$37,$F8,$07,$9B,$FE,$D8,$6C,$7F
	.dc.b	$00,$77,$87,$FA,$75,$78,$DF,$D9,$D7,$CE,$53,$11,$FD,$AF,$EB,$7F
	.dc.b	$35,$7F,$ED,$55,$E0,$64,$D5,$EF,$87,$87,$7C,$BE,$11,$CF,$DD,$E9
	.dc.b	$EF,$49,$E9,$1F,$DE,$7E,$B5,$E1,$5F,$76,$57,$FB,$96,$72,$FF,$B9
	.dc.b	$FE,$B7,$D2,$DF,$96,$FF,$01,$41,$BC,$1F,$E0,$5E,$EF,$4D,$7F,$BB
	.dc.b	$DF,$D7,$71,$7F,$B0,$5B,$C7,$FC,$A2,$7F,$8F,$F0,$F1,$F0,$B9,$7D
	.dc.b	$6C,$9E,$51,$D7,$83,$B7,$85,$27,$B4,$7F,$85,$7E,$B7,$FA,$FC,$7F
	.dc.b	$BF,$D7,$C0,$E4,$D7,$0F,$F8,$6F,$EB,$7F,$9B,$CF,$F3,$2D,$BF,$DA
	.dc.b	$CD,$D7,$CD,$3F,$CF,$0F,$81,$87,$F4,$0A,$FF,$6E,$43,$D4,$E8,$75
	.dc.b	$E1,$ED,$D5,$DF,$8A,$61,$C7,$1E,$B3,$E3,$F4,$F7,$39,$3F,$A6,$E3
	.dc.b	$F8,$C7,$01,$47,$AF,$C4,$F6,$78,$B2,$B7,$5F,$E3,$65,$A7,$A3,$D2
	.dc.b	$E1,$1A,$BF,$89,$1F,$E3,$FF,$AC,$7C,$47,$F9,$07,$EB,$5E,$FD,$35
	.dc.b	$1B,$F0,$2E,$3E,$6B,$E2,$D8,$FF,$25,$7C,$5F,$ED,$B8,$3F,$05,$E8
	.dc.b	$F9,$75,$46,$35,$DF,$C3,$F0,$F2,$6E,$5F,$0B,$1E,$4F,$F7,$27,$F7
	.dc.b	$EF,$BA,$FF,$1D,$F5,$BF,$90,$8F,$EE,$F9,$7D,$77,$E5,$B5,$E0,$7F
	.dc.b	$7C,$8E,$09,$FC,$EB,$A3,$1E,$AF,$F9,$96,$36,$6B,$3F,$5C,$F3,$F1
	.dc.b	$F5,$E0,$C8,$FB,$75,$FC,$09,$E8,$FC,$87,$B6,$1F,$81,$6F,$3F,$7E
	.dc.b	$1F,$07,$05,$9E,$71,$87,$AF,$8F,$F8,$07,$F6,$07,$B5,$C3,$8C,$11
	.dc.b	$88,$F9,$2D,$7E,$27,$9E,$FA,$DF,$F6,$05,$7F,$3E,$6C,$66,$7B,$9F
	.dc.b	$25,$C3,$FE,$C7,$FA,$7F,$F5,$3A,$69,$1F,$F6,$67,$D3,$7C,$AD,$FF
	.dc.b	$D6,$0D,$E1,$EB,$5F,$E3,$2F,$AD,$FF,$47,$DF,$F0,$DA,$D3,$FC,$3F
	.dc.b	$96,$BF,$E3,$86,$A8,$B1,$8E,$59,$87,$E1,$BE,$56,$DF,$E0,$47,$CE
	.dc.b	$72,$33,$9F,$FB,$15,$EB,$F3,$92,$FF,$73,$5C,$1B,$1E,$4A,$FF,$0E
	.dc.b	$47,$EF,$9B,$3F,$82,$F0,$B5,$CF,$BE,$39,$7B,$8B,$5F,$95,$FA,$DF
	.dc.b	$64,$FF,$08,$6C,$7E,$33,$C1,$F8,$71,$8C,$7F,$2F,$28,$FF,$4A,$D7
	.dc.b	$1D,$B7,$BF,$07,$8F,$8D,$27,$B7,$8F,$70,$F1,$79,$FD,$8C,$65,$DB
	.dc.b	$2F,$EA,$77,$E5,$F4,$BC,$67,$93,$E3,$EC,$7A,$7D,$4F,$39,$8C,$7D
	.dc.b	$CF,$AB,$E3,$64,$FB,$FE,$72,$D7,$ED,$FE,$69,$CF,$F8,$26,$D7,$F2
	.dc.b	$C5,$EE,$66,$75,$E5,$4A,$D7,$8F,$70,$FE,$05,$F5,$AF,$9F,$FC,$6B
	.dc.b	$6B,$F9,$BB,$FE,$22,$6C,$6B,$C6,$91,$AF,$76,$DF,$F0,$DF,$AD,$FC
	.dc.b	$33,$7B,$FF,$33,$1F,$E2,$2A,$FE,$86,$EB,$97,$EC,$AE,$0D,$9C,$38
	.dc.b	$AD,$7F,$4A,$FA,$D6,$DF,$A4,$6C,$67,$B3,$13,$FA,$88,$F5,$67,$FC
	.dc.b	$85,$B9,$7E,$21,$FE,$12,$7F,$56,$32,$7F,$7F,$FA,$DF,$F5,$3B,$FD
	.dc.b	$69,$B3,$CF,$DD,$FD,$AC,$62,$BF,$26,$FB,$E7,$F9,$83,$63,$3D,$BC
	.dc.b	$3C,$3E,$3A,$4C,$7B,$75,$5C,$7F,$9C,$78,$BF,$EA,$E9,$FE,$D6,$D8
	.dc.b	$CF,$C9,$FF,$06,$8C,$65,$FA,$47,$FC,$7B,$5C,$3C,$CD,$7A,$BB,$7F
	.dc.b	$86,$FD,$6F,$EE,$2F,$F5,$6D,$AF,$F1,$03,$FE,$3E,$7F,$BD,$B9,$C7
	.dc.b	$FD,$96,$FF,$AE,$E3,$3F,$0B,$D3,$DC,$EA,$F6,$E1,$93,$CB,$DB,$B8
	.dc.b	$7F,$55,$F1,$FF,$5A,$DF,$17,$F9,$0C,$3C,$7B,$1F,$C9,$3E,$B5,$F2
	.dc.b	$94,$C6,$34,$FF,$08,$FD,$6B,$E1,$BF,$95,$B6,$3F,$C2,$BF,$5B,$FD
	.dc.b	$E9,$DC,$EB,$C1,$91,$83,$9B,$CB,$C2,$91,$83,$BB,$DB,$86,$47,$6F
	.dc.b	$9F,$BB,$FF,$73,$CB,$E4,$37,$8B,$3C,$9F,$C7,$27,$2F,$F1,$EF,$AD
	.dc.b	$FC,$76,$1F,$E5,$36,$DF,$FB,$F3,$57,$4F,$F4,$89,$CC,$9F,$E5,$15
	.dc.b	$C7,$8F,$53,$B7,$87,$89,$27,$E2,$FC,$2D,$AF,$F0,$55,$BD,$3E,$C6
	.dc.b	$73,$F3,$3E,$1F,$72,$61,$FE,$42,$AE,$0D,$8F,$37,$A3,$23,$E9,$FC
	.dc.b	$9D,$C7,$BF,$4F,$AC,$C6,$5F,$0D,$F7,$78,$12,$FE,$DF,$C6,$DB,$F3
	.dc.b	$F8,$5F,$0C,$C6,$1F,$2D,$F8,$DF,$35,$2F,$F0,$7F,$0B,$6F,$DF,$F0
	.dc.b	$5F,$2C,$C6,$1F,$4D,$F9,$DF,$4D,$2F,$DD,$16,$BF,$C6,$1F,$55,$F9
	.dc.b	$69,$FF,$31,$B6,$3F,$C6,$9F,$5A,$F9,$7F,$EF,$F3,$9C,$7F,$AE,$AE
	.dc.b	$CC,$9A,$F0,$6E,$3D,$3D,$6E,$27,$F8,$23,$EB,$5F,$5B,$89,$F0,$FB
	.dc.b	$1D,$8C,$6B,$E6,$7C,$3C,$72,$7F,$4B,$EC,$5A,$F5,$F1,$79,$8C,$6D
	.dc.b	$E6,$7D,$3E,$FC,$BF,$7C,$5A,$F8,$7F,$17,$EE,$3C,$DD,$56,$BE,$2F
	.dc.b	$D6,$FC,$23,$15,$F2,$9F,$8B,$C7,$27,$FD,$1B,$EB,$3C,$7C,$5F,$F9
	.dc.b	$7F,$91,$EE,$7F,$8A,$9B,$7F,$4B,$0C,$4F,$F8,$BB,$EB,$7F,$8E,$59
	.dc.b	$BF,$2F,$FD,$C2,$3F,$CD,$51,$A3,$FD,$1F,$7F,$E9,$E6,$C6,$BE,$0B
	.dc.b	$A7,$C1,$F5,$F8,$59,$7F,$EA,$4C,$1B,$63,$0E,$BF,$01,$E3,$31,$C9
	.dc.b	$FE,$1E,$CF,$FB,$B4,$F8,$B7,$F4,$1F,$E4,$6F,$A7,$F6,$C6,$72,$F2
	.dc.b	$79,$FE,$D3,$6E,$0F,$F5,$AF,$B7,$FE,$28,$9F,$F4,$33,$71,$FF,$16
	.dc.b	$DF,$F6,$49,$09,$D5,$7F,$B9,$5F,$EB,$CD,$7F,$CB,$26,$A3,$8B,$A6
	.dc.b	$F9,$9A,$3D,$5E,$B5,$AF,$2F,$99,$C8,$C5,$79,$9E,$BE,$B4,$9F,$F6
	.dc.b	$7F,$D4,$75,$AF,$34,$F8,$BE,$7A,$57,$C0,$8B,$3F,$EA,$5A,$79,$7F
	.dc.b	$E4,$C8,$FF,$80,$11,$8A,$FF,$42,$C2,$30,$FF,$A2,$1A,$FF,$A6,$17
	.dc.b	$97,$CE,$3E,$4C,$62,$79,$B6,$39,$4E,$DE,$CF,$FA,$0B,$E4,$FF,$9E
	.dc.b	$D7,$BA,$31,$E9,$FF,$A1,$AF,$FD,$BF,$3D,$3C,$BE,$51,$8C,$3C,$D1
	.dc.b	$D7,$0F,$F0,$54,$7F,$AB,$9A,$FD,$E0,$E1,$FC,$0B,$5C,$F9,$1C,$C9
	.dc.b	$7F,$1A,$7F,$CA,$AD,$F9,$FF,$01,$1C,$1F,$33,$87,$FC,$84,$DF,$84
	.dc.b	$F8,$51,$8C,$3F,$D6,$AD,$7F,$D8,$AB,$E1,$F6,$4A,$E8,$F2,$F4,$3F
	.dc.b	$DB,$B1,$C1,$B5,$F9,$96,$BF,$96,$6B,$F0,$BC,$6B,$91,$A4,$AF,$F2
	.dc.b	$2C,$FE,$7B,$D0,$E3,$89,$A4,$9D,$CF,$EA,$FE,$B3,$64,$8C,$D2,$31
	.dc.b	$3F,$95,$8C,$4E,$A7,$2F,$FB,$A3,$EA,$BF,$C3,$95,$F9,$B6,$C6,$4F
	.dc.b	$6B,$88,$FC,$06,$2F,$82,$F7,$92,$BC,$F1,$97,$D3,$5F,$D5,$5B,$E3
	.dc.b	$F4,$23,$AE,$3F,$E6,$78,$7F,$BB,$AB,$FC,$2F,$E3,$7E,$B1,$7C,$FF
	.dc.b	$FB,$76,$75,$5C,$1F,$67,$B3,$F9,$5C,$2C,$78,$A3,$0F,$E6,$6D,$18
	.dc.b	$8F,$E3,$3F,$57,$CE,$BF,$3D,$08,$C3,$FA,$16,$BC,$31,$E4,$AF,$FD
	.dc.b	$6A,$FC,$2F,$47,$31,$88,$FA,$11,$8F,$62,$47,$C8,$6B,$F5,$79,$B8
	.dc.b	$91,$F9,$AF,$83,$07,$D0,$FF,$12,$9F,$43,$63,$31,$FC,$59,$AF,$B8
	.dc.b	$F7,$7C,$A6,$C8,$ED,$E5,$E6,$C8,$E4,$D7,$EC,$79,$FF,$C1,$7C,$6F
	.dc.b	$F7,$8F,$1F,$F9,$CA,$7F,$A2,$C6,$3D,$7D,$70,$7D,$3B,$B5,$FD,$2B
	.dc.b	$7F,$F5,$DB,$C2,$74,$D3,$53,$C1,$FA,$1F,$15,$7E,$AD,$F2,$D9,$F0
	.dc.b	$9E,$F5,$35,$1B,$DE,$E1,$D4,$5F,$27,$0A,$3B,$6F,$E1,$DF,$FC,$65
	.dc.b	$5C,$5A,$B7,$6B,$FC,$A7,$05,$F9,$FD,$3E,$77,$DA,$30,$F8,$3E,$13
	.dc.b	$DD,$E6,$9A,$7B,$CF,$81,$F0,$96,$78,$1E,$87,$E2,$79,$BC,$B6,$93
	.dc.b	$EC,$8F,$91,$F0,$FA,$8C,$1F,$01,$F9,$DF,$56,$F8,$CF,$1E,$12,$BE
	.dc.b	$44,$E5,$BE,$27,$0A,$91,$83,$E7,$7D,$4B,$EA,$FE,$D7,$93,$FE,$BB
	.dc.b	$9F,$A4,$D7,$E5,$71,$7F,$51,$AE,$11,$CB,$DC,$FC,$14,$62,$36,$7F
	.dc.b	$85,$1C,$72,$7D,$28,$C4,$F0,$F9,$3F,$C5,$7D,$33,$6A,$66,$BC,$FE
	.dc.b	$95,$7A,$8C,$57,$E2,$9F,$1F,$9D,$F9,$5F,$AB,$FF,$52,$5F,$E3,$5A
	.dc.b	$F2,$BF,$EA,$CF,$C4,$D7,$8B,$F1,$AB,$DB,$1E,$EF,$4A,$CE,$CF,$96
	.dc.b	$FE,$8F,$EA,$7D,$D1,$E6,$FA,$BF,$F6,$0B,$F9,$D6,$BE,$A8,$DE,$FC
	.dc.b	$1F,$DC,$46,$78,$3A,$1D,$7C,$4E,$DE,$5F,$EC,$7E,$AF,$FA,$95,$7E
	.dc.b	$A5,$AF,$58,$F2,$7A,$BC,$23,$F5,$D8,$7F,$32,$6C,$7A,$7C,$AF,$DE
	.dc.b	$7D,$5B,$DC,$A6,$6B,$8F,$C3,$D7,$D5,$5F,$06,$3E,$17,$AF,$67,$67
	.dc.b	$81,$F8,$DF,$29,$F5,$7D,$C9,$3E,$48,$F9,$5F,$0D,$A8,$C1,$E8,$7E
	.dc.b	$77,$C4,$7F,$11,$E3,$7A,$74,$CD,$7B,$BE,$1E,$BF,$3E,$BE,$B4,$7F
	.dc.b	$19,$F6,$3F,$99,$F3,$7D,$8E,$FF,$E0,$B7,$FC,$4D,$AF,$E4,$27,$7C
	.dc.b	$FF,$E3,$37,$F8,$B4,$67,$D5,$E8,$75,$F2,$3B,$79,$F6,$7F,$8F,$57
	.dc.b	$0C,$7F,$0E,$FF,$1D,$8C,$FC,$FF,$E9,$3E,$AE,$46,$0E,$3F,$41,$E9
	.dc.b	$99,$18,$3A,$F9,$BF,$BC,$F8,$29,$18,$3C,$3E,$3F,$C1,$99,$18,$3C
	.dc.b	$BF,$0B,$FC,$F7,$E2,$5D,$54,$F9,$11,$F2,$7B,$2D,$70,$3D,$AF,$CD
	.dc.b	$F5,$75,$18,$3E,$0B,$FD,$27,$EA,$DE,$05,$F0,$A5,$7F,$43,$9C,$B7
	.dc.b	$D9,$E1,$ED,$48,$C1,$FD,$27,$EA,$DF,$19,$FD,$55,$71,$D3,$11,$9E
	.dc.b	$0D,$7C,$BC,$BE,$51,$FD,$77,$EA,$FF,$D5,$91,$FD,$55,$FE,$5C,$FF
	.dc.b	$12,$6F,$47,$F0,$07,$71,$9F,$89,$E5,$E8,$C8,$E5,$1F,$DA,$7D,$BF
	.dc.b	$EA,$F7,$FC,$21,$FE,$02,$DF,$37,$5F,$0B,$BF,$23,$87,$C3,$4C,$7F
	.dc.b	$B4,$60,$9F,$EE,$9F,$57,$F9,$8C,$7F,$89,$BF,$8D,$AF,$E5,$2D,$8C
	.dc.b	$D7,$0E,$0E,$3D,$D2,$FF,$BA,$5F,$CA,$6F,$B9,$DB,$DD,$91,$83,$FB
	.dc.b	$D7,$D5,$BC,$1F,$F0,$33,$FD,$2F,$3F,$E4,$AD,$F1,$FF,$D3,$8E,$EB
	.dc.b	$FC,$05,$F5,$7F,$E5,$F6,$FC,$97,$F8,$3E,$0D,$E0,$FF,$09,$76,$FA
	.dc.b	$F0,$FF,$53,$3F,$AD,$6F,$91,$AF,$23,$7E,$1F,$0F,$AB,$99,$FF,$84
	.dc.b	$70,$4F,$4D,$FA,$F8,$72,$B5,$1F,$E2,$1F,$AB,$F5,$47,$FB,$C9,$7B
	.dc.b	$D9,$35,$F3,$3F,$E2,$7F,$AB,$FE,$7F,$B7,$FA,$EE,$BF,$9B,$F1,$FF
	.dc.b	$48,$6C,$7F,$AF,$E7,$78,$7F,$BF,$9F,$82,$FF,$89,$DF,$F5,$03,$7D
	.dc.b	$6E,$DC,$5D,$FA,$FC,$1C,$72,$B8,$47,$8F,$D5,$D4,$60,$E8,$F9,$FC
	.dc.b	$7F,$47,$8F,$27,$C1,$1E,$BF,$4F,$51,$83,$C8,$FF,$91,$7E,$A9,$F0
	.dc.b	$FF,$E4,$7F,$AB,$7B,$D4,$D4,$6F,$C1,$B8,$7F,$90,$EF,$8B,$63,$FC
	.dc.b	$A1,$F1,$5F,$B0,$E2,$FF,$72,$7A,$7F,$E5,$8A,$FF,$02,$B7,$97,$FB
	.dc.b	$9D,$6E,$47,$F9,$53,$E1,$F9,$FC,$3C,$8F,$F2,$94,$FE,$EA,$BF,$BD
	.dc.b	$FD,$5F,$E2,$E7,$8E,$1F,$41,$EF,$CD,$FE,$63,$9D,$41,$FC,$A3,$E0
	.dc.b	$72,$4F,$03,$87,$91,$C5,$93,$FE,$63,$9F,$DD,$69,$FF,$00,$5F,$87
	.dc.b	$1E,$EE,$6F,$F3,$96,$3F,$C0,$63,$87,$07,$8F,$7F,$23,$23,$A4,$79
	.dc.b	$FC,$8F,$F3,$34,$7F,$A4,$F0,$FE,$4F,$AF,$FA,$56,$3D,$53,$A7,$FD
	.dc.b	$6B,$3A,$78,$7A,$0F,$AB,$E1,$3F,$CC,$9B,$19,$9D,$C7,$FA,$9D,$7D
	.dc.b	$2D,$23,$FE,$AB,$7F,$D5,$9F,$43,$FE,$49,$C7,$FC,$5B,$28,$C6,$DF
	.dc.b	$EA,$BA,$FB,$BF,$57,$FE,$90,$AF,$BE,$6E,$7F,$D1,$CF,$F9,$AD,$BC
	.dc.b	$9C,$A3,$77,$5F,$C3,$C5,$5C,$B4,$D7,$03,$E4,$F1,$87,$6F,$9A,$E0
	.dc.b	$62,$FF,$CB,$21,$E9,$F3,$52,$78,$1E,$67,$C5,$AF,$FA,$62,$3F,$D4
	.dc.b	$CD,$AF,$CF,$F8,$3A,$E7,$DF,$2C,$BD,$C5,$AF,$F3,$4F,$D5,$FC,$36
	.dc.b	$EF,$EA,$F0,$3E,$04,$E7,$2F,$F2,$DB,$F1,$EB,$8F,$06,$AF,$6F,$0F
	.dc.b	$16,$65,$FE,$BB,$BF,$E4,$6D,$8E,$BE,$7C,$8C,$1B,$7C,$3E,$AC,$8E
	.dc.b	$5F,$1F,$72,$F4,$FA,$DE,$63,$1A,$79,$9F,$4F,$C6,$CB,$F1,$7C,$DD
	.dc.b	$BF,$F4,$4F,$CC,$7D,$79,$FF,$68,$36,$33,$EF,$FE,$06,$31,$8F,$F9
	.dc.b	$F1,$79,$19,$35,$E4,$DC,$3F,$DB,$5F,$56,$F9,$EF,$C5,$36,$33,$E0
	.dc.b	$E2,$73,$DB,$27,$8F,$BD,$75,$FF,$4F,$FD,$5F,$E1,$E7,$FD,$CC,$D8
	.dc.b	$CF,$81,$F9,$78,$C6,$DF,$E9,$25,$C1,$93,$5C,$57,$0F,$F7,$87,$D5
	.dc.b	$B7,$FD,$03,$63,$3D,$78,$9F,$D2,$47,$F8,$B8,$FF,$7C,$17,$3F,$F5
	.dc.b	$AB,$FE,$30,$33,$BF,$EA,$FE,$AF,$F9,$8C,$7F,$BF,$DB,$3C,$3D,$DD
	.dc.b	$3B,$DA,$56,$FB,$2C,$FF,$B0,$AD,$A7,$53,$B7,$4D,$BF,$E7,$FC,$1F
	.dc.b	$EB,$4F,$15,$EC,$FE,$F5,$B3,$A3,$F8,$3A,$F9,$ED,$4E,$FC,$C9,$5B
	.dc.b	$CF,$FA,$5D,$BD,$4C,$C7,$FB,$5B,$EA,$FF,$6D,$97,$FB,$11,$B7,$FC
	.dc.b	$51,$79,$FF,$C3,$DC,$D7,$F5,$67,$FD,$65,$3D,$E1,$18,$7B,$7B,$7C
	.dc.b	$7E,$1F,$97,$1C,$BE,$16,$F6,$F1,$87,$FB,$AB,$C7,$FF,$1B,$57,$56
	.dc.b	$C6,$7C,$6F,$E4,$51,$8D,$3F,$DB,$BF,$56,$F8,$4F,$E4,$CD,$C3,$CC
	.dc.b	$9C,$C7,$F2,$BF,$AB,$7C,$3D,$3F,$97,$7D,$5E,$EB,$DD,$3A,$6F,$85
	.dc.b	$C4,$E0,$E0,$DF,$87,$D4,$D3,$83,$9F,$17,$84,$EF,$F0,$F6,$3B,$F1
	.dc.b	$F9,$4C,$70,$36,$FF,$7D,$E5,$F3,$DF,$F7,$EF,$D5,$FE,$61,$B3,$EA
	.dc.b	$EA,$FE,$91,$39,$C3,$FA,$8B,$FC,$AD,$BC,$3A,$AD,$BC,$3D,$39,$3F
	.dc.b	$7F,$7F,$CA,$D7,$8E,$39,$CF,$6F,$4E,$47,$E1,$70,$FE,$02,$F8,$23
	.dc.b	$C7,$ED,$C8,$D4,$5B,$E8,$7F,$89,$2B,$F9,$BB,$80,$E3,$3F,$B9,$1F
	.dc.b	$E2,$AC,$7F,$DE,$37,$E1,$F8,$2F,$83,$CD,$23,$07,$95,$F8,$BD,$F2
	.dc.b	$3D,$DF,$19,$70,$F0,$65,$FB,$C9,$F6,$78,$39,$CF,$E8,$BF,$3B,$E6
	.dc.b	$A4,$FC,$7F,$7A,$E1,$CB,$E0,$3E,$99,$8C,$3E,$CB,$F5,$BE,$96,$5F
	.dc.b	$C9,$8B,$5F,$07,$EA,$BF,$38,$DE,$DF,$F0,$27,$D5,$BE,$5B,$FC,$0D
	.dc.b	$5A,$CB,$E2,$DA,$1D,$7A,$37,$85,$CB,$F7,$93,$FE,$E2,$6F,$AD,$FE
	.dc.b	$11,$FA,$B7,$D5,$F6,$8E,$FF,$01,$2B,$67,$27,$C7,$C9,$23,$F1,$5E
	.dc.b	$C5,$D3,$D9,$C5,$E8,$31,$B7,$A3,$44,$7B,$3E,$02,$61,$E6,$8B,$7F
	.dc.b	$17,$E2,$3E,$09,$E7,$EA,$B5,$EC,$FA,$BF,$8A,$62,$BE,$63,$F2,$38
	.dc.b	$E4,$FF,$1D,$F5,$BF,$D6,$2F,$C4,$8F,$F1,$97,$23,$DC,$A7,$0C,$3F
	.dc.b	$2D,$19,$AF,$E4,$DF,$57,$FC,$AA,$4D,$F9,$6F,$EF,$73,$FE,$44,$8D
	.dc.b	$1F,$E9,$67,$8F,$C2,$7F,$90,$63,$2D,$F6,$64,$FF,$33,$A6,$98,$C3
	.dc.b	$F5,$18,$7F,$7F,$6F,$BD,$9E,$3F,$C2,$CE,$5B,$E5,$F7,$3F,$CA,$1F
	.dc.b	$4D,$EB,$BE,$71,$E3,$F3,$26,$7E,$E8,$B3,$FB,$7F,$6F,$FC,$6D,$87
	.dc.b	$E4,$9B,$5F,$E6,$2C,$3F,$BB,$4E,$7D,$BF,$F2,$F5,$7F,$97,$2B,$FA
	.dc.b	$73,$6F,$A4,$EA,$37,$E5,$CA,$FE,$A9,$87,$F4,$D7,$8C,$74,$F2,$A4
	.dc.b	$60,$EA,$F8,$3E,$0E,$47,$F5,$DF,$A8,$F0,$67,$E9,$9F,$2F,$CF,$CA
	.dc.b	$F2,$C5,$D3,$F8,$B5,$FF,$07,$6F,$93,$FE,$79,$87,$97,$62,$5F,$F6
	.dc.b	$BF,$91,$FE,$FC,$F8,$63,$FD,$01,$18,$BF,$E4,$4D,$7F,$D1,$61,$C3
	.dc.b	$FE,$8E,$6B,$FE,$CE,$6B,$BF,$7B,$FD,$32,$E5,$AE,$6B,$FD,$61,$3F
	.dc.b	$D0,$23,$BB,$FC,$15,$F0,$9E,$1F,$17,$8C,$66,$37,$18,$FA,$09,$5D
	.dc.b	$1B,$3C,$A9,$E0,$72,$25,$78,$9F,$2F,$13,$E1,$A9,$27,$FD,$8D,$19
	.dc.b	$6F,$9D,$FD,$19,$F8,$F0,$8C,$61,$FE,$B4,$8C,$4F,$F9,$0A,$3F,$DE
	.dc.b	$B1,$87,$FD,$7E,$BC,$DE,$31,$A6,$BF,$EB,$46,$C6,$E7,$A4,$FF,$AE
	.dc.b	$3C,$EC,$92,$33,$28,$E3,$F5,$72,$72,$DA,$FF,$68,$FD,$57,$F8,$5D
	.dc.b	$B3,$E2,$8C,$3F,$86,$6C,$7F,$B4,$9A,$83,$86,$C7,$33,$DB,$A7,$F9
	.dc.b	$35,$F5,$B6,$3F,$DC,$6D,$8F,$C3,$FC,$07,$0F,$1A,$47,$06,$D7,$AB
	.dc.b	$D7,$FE,$58,$FE,$33,$0F,$F3,$07,$8D,$FF,$00,$57,$CF,$7F,$BB,$E7
	.dc.b	$55,$C1,$F7,$72,$7F,$BB,$B1,$B1,$F4,$2F,$FC,$DE,$D9,$D3,$F0,$F9
	.dc.b	$2B,$F0,$E5,$18,$8F,$8A,$D8,$CC,$75,$F3,$F0,$47,$07,$87,$E8,$78
	.dc.b	$11,$C1,$E5,$E6,$E8,$47,$07,$E2,$3E,$AD,$E4,$F8,$DB,$19,$8C,$19
	.dc.b	$F8,$3F,$DA,$C7,$DD,$9F,$F3,$EB,$86,$BF,$BA,$6B,$FC,$31,$7C,$CF
	.dc.b	$28,$46,$1E,$51,$98,$C4,$73,$F9,$4C,$11,$F0,$3C,$F3,$71,$7E,$46
	.dc.b	$7F,$8E,$FA,$F7,$C2,$FF,$0C,$F3,$7F,$9F,$67,$FD,$21,$7C,$1B,$F1
	.dc.b	$3B,$E3,$FC,$E6,$36,$7E,$D3,$5F,$CE,$CD,$FA,$BF,$3E,$3F,$E2,$6F
	.dc.b	$09,$EF,$6D,$B7,$FE,$9F,$AD,$E3,$E7,$7E,$4D,$FF,$8B,$9C,$CF,$EA
	.dc.b	$DA,$9D,$CF,$1F,$91,$F2,$69,$FE,$34,$AF,$E8,$5E,$97,$96,$7C,$DC
	.dc.b	$76,$7C,$51,$87,$C9,$F0,$7E,$DF,$2C,$D3,$DC,$7B,$FE,$0E,$CF,$03
	.dc.b	$D8,$FC,$3F,$86,$E5,$CD,$27,$E0,$47,$C7,$F8,$8D,$46,$0F,$88,$FC
	.dc.b	$DF,$AB,$7A,$1E,$3C,$25,$7F,$B1,$E7,$2D,$F1,$B8,$54,$8C,$1F,$EC
	.dc.b	$DF,$A9,$7D,$5F,$D9,$F2,$9F,$0B,$4F,$B9,$3A,$AE,$1E,$E6,$2E,$46
	.dc.b	$0E,$9F,$27,$F8,$3F,$A6,$6B,$4E,$71,$D3,$D2,$B5,$C0,$F2,$3E,$0F
	.dc.b	$43,$F1,$9F,$56,$F5,$A9,$9A,$F5,$6D,$5F,$F4,$D6,$BF,$D6,$5C,$9E
	.dc.b	$FF,$4B,$D7,$32,$30,$7F,$BB,$FE,$AB,$FC,$F7,$DF,$FE,$7C,$FA,$97
	.dc.b	$9F,$F9,$D6,$D6,$FC,$0F,$D0,$5F,$EA,$63,$FD,$54,$F2,$8F,$D8,$36
	.dc.b	$3F,$A1,$3B,$BF,$F4,$4B,$FC,$A2,$3F,$D6,$5F,$56,$F5,$3F,$58,$D8
	.dc.b	$F3,$7A,$3E,$0F,$47,$C2,$C8,$F5,$47,$93,$CF,$FD,$A7,$D5,$BD,$DA
	.dc.b	$66,$BE,$07,$C4,$57,$D3,$5E,$C8,$F6,$FC,$2D,$9D,$9F,$1D,$F8,$3F
	.dc.b	$29,$F5,$BD,$E9,$3F,$2E,$3E,$2F,$B9,$A8,$C1,$F3,$DF,$93,$F1,$3F
	.dc.b	$37,$D8,$93,$F4,$E3,$E6,$FC,$3E,$A3,$07,$D7,$7F,$88,$7B,$6F,$AB
	.dc.b	$FC,$87,$9D,$2B,$FB,$87,$37,$FA,$DA,$B5,$18,$E7,$FE,$49,$1F,$C2
	.dc.b	$8C,$C7,$FB,$8D,$B3,$C7,$CC,$EF,$ED,$F8,$19,$5C,$CF,$EF,$D5,$FC
	.dc.b	$49,$BE,$D7,$E8,$FE,$AE,$46,$CF,$ED,$F1,$EA,$FA,$0F,$5B,$23,$54
	.dc.b	$9F,$7F,$A3,$FB,$7F,$83,$95,$A8,$F8,$7E,$4E,$A3,$07,$27,$E3,$FC
	.dc.b	$2F,$F3,$7F,$89,$74,$D3,$35,$E0,$EB,$AF,$C3,$AF,$9F,$1F,$43,$D8
	.dc.b	$B3,$B3,$CE,$FF,$43,$FA,$BF,$EF,$30,$E2,$FF,$73,$C7,$E7,$DB,$E0
	.dc.b	$6F,$C0,$E1,$EE,$F1,$F7,$A6,$1F,$1F,$04,$FF,$4F,$FA,$B7,$C6,$5E
	.dc.b	$39,$7F,$36,$72,$DE,$2E,$1F,$2D,$23,$07,$D3,$FA,$B7,$B3,$FD,$6D
	.dc.b	$FE,$57,$1E,$9E,$CF,$ED,$51,$8E,$0F,$F8,$02,$F0,$9E,$5F,$15,$C7
	.dc.b	$D4,$91,$83,$F0,$1D,$FF,$D4,$DE,$87,$F9,$15,$FE,$09,$87,$F7,$76
	.dc.b	$C6,$7E,$03,$FB,$74,$3C,$6B,$F1,$32,$BA,$C7,$F7,$3F,$AB,$55,$F0
	.dc.b	$E5,$7E,$46,$72,$DE,$2E,$1E,$0C,$8C,$1B,$F7,$FA,$7C,$04,$8C,$1F
	.dc.b	$9B,$FA,$B7,$87,$FE,$03,$7F,$A5,$C6,$7D,$FC,$4F,$F8,$1F,$EA,$DB
	.dc.b	$DF,$92,$97,$FA,$59,$CB,$FA,$7F,$03,$F5,$89,$FF,$00,$93,$7F,$F0
	.dc.b	$92,$F0,$EF,$D5,$CA,$E5,$1A,$D7,$7E,$1C,$9E,$D1,$FE,$1B,$FA,$BF
	.dc.b	$4C,$7D,$60,$BE,$0F,$46,$F7,$FF,$11,$7D,$5F,$F3,$FA,$DF,$B1,$CC
	.dc.b	$CF,$56,$B8,$37,$EA,$CA,$FE,$09,$69,$E7,$D5,$D3,$8E,$56,$A3,$BF
	.dc.b	$A9,$A8,$C1,$C9,$F1,$F8,$DE,$4F,$22,$4F,$68,$F3,$FA,$7A,$8C,$1E
	.dc.b	$27,$FC,$79,$F5,$5F,$E6,$28,$F8,$5F,$57,$FE,$B5,$7F,$8D,$35,$F8
	.dc.b	$D1,$BF,$15,$BF,$F9,$46,$1D,$DD,$3E,$2A,$5F,$F2,$39,$CB,$FC,$97
	.dc.b	$E3,$7F,$5D,$7F,$E5,$9F,$4B,$F8,$0C,$FF,$29,$6B,$F5,$67,$6F,$FA
	.dc.b	$BD,$FD,$A3,$9F,$89,$FF,$2B,$46,$A3,$FC,$A7,$E5,$7F,$96,$7E,$AF
	.dc.b	$FC,$97,$6E,$6F,$F2,$E3,$6B,$F9,$C4,$FF,$94,$0F,$C9,$97,$D7,$E5
	.dc.b	$E0,$F3,$F2,$2D,$70,$8C,$3C,$B6,$EB,$CD,$23,$07,$6E,$0F,$24,$C8
	.dc.b	$C1,$E2,$F1,$FF,$96,$3F,$E7,$D3,$FD,$9D,$97,$F4,$7E,$2F,$F4,$AB
	.dc.b	$63,$41,$C3,$C5,$FE,$57,$F5,$7C,$3D,$13,$49,$3F,$E9,$7F,$A4,$FF
	.dc.b	$4C,$26,$91,$FF,$52,$FD,$0F,$E0,$9F,$D5,$07,$18,$66,$3F,$D2,$D3
	.dc.b	$FE,$E7,$FA,$BF,$F4,$2C,$7F,$BA,$6B,$7C,$DE,$FE,$6B,$3F,$E9,$D6
	.dc.b	$BB,$8F,$ED,$CF,$F9,$A1,$BC,$BA,$8F,$E7,$F1,$F8,$BF,$96,$C4,$E7
	.dc.b	$E6,$64,$F0,$39,$BC,$7E,$2E,$46,$0E,$EF,$8B,$4F,$E6,$31,$FD,$FD
	.dc.b	$B8,$7F,$0B,$E7,$FD,$F4,$E7,$9B,$87,$1C,$8C,$1C,$5F,$6F,$D5,$FC
	.dc.b	$98,$FF,$05,$36,$FA,$77,$EB,$7D,$EF,$2F,$71,$6B,$F8,$E5,$FF,$A0
	.dc.b	$27,$9F,$8B,$DB,$C6,$EE,$F8,$3C,$E9,$3A,$3C,$2F,$93,$D3,$91,$E4
	.dc.b	$F8,$FB,$8F,$6F,$57,$D2,$63,$87,$F0,$19,$A3,$DB,$F1,$F3,$0F,$6F
	.dc.b	$D4,$DC,$3F,$99,$7C,$D3,$A3,$EF,$B6,$BF,$C5,$8B,$DE,$CC,$EB,$C8
	.dc.b	$95,$AF,$26,$E1,$FC,$F7,$EA,$DF,$3D,$F8,$A6,$D7,$F8,$DD,$FF,$07
	.dc.b	$B6,$35,$D5,$23,$5E,$F5,$BF,$E9,$1F,$56,$EE,$FC,$AB,$6B,$FC,$84
	.dc.b	$FE,$4A,$77,$3A,$DE,$56,$B8,$6D,$7F,$53,$FA,$BF,$F1,$1C,$FF,$92
	.dc.b	$DB,$19,$EC,$FD,$1C,$63,$97,$F5,$D5,$EA,$AF,$EB,$8D,$8C,$F8,$DF
	.dc.b	$AA,$8C,$09,$9F,$57,$63,$FC,$B4,$D8,$E7,$EE,$FE,$CA,$31,$7F,$BD
	.dc.b	$7D,$CD,$F0,$F5,$E0,$6D,$E1,$A4,$C7,$AF,$65,$CB,$FB,$97,$8B,$FE
	.dc.b	$01,$9F,$F3,$53,$63,$3F,$25,$FC,$0A,$31,$97,$F0,$D7,$FC,$51,$5C
	.dc.b	$3C,$BD,$7A,$7B,$7F,$84,$FD,$5F,$ED,$AF,$F8,$A3,$6B,$E1,$62,$87
	.dc.b	$9D,$C3,$C8,$E3,$F0,$D2,$FF,$CF,$D5,$FE,$13,$6C,$76,$E2,$91,$AF
	.dc.b	$6A,$E3,$FE,$0F,$F2,$7F,$C5,$73,$FE,$87,$6C,$67,$C8,$FE,$3F,$18
	.dc.b	$D3,$F9,$57,$D5,$BE,$83,$F9,$1B,$70,$FF,$48,$3F,$DD,$1B,$1F,$CA
	.dc.b	$3E,$AD,$F0,$F4,$FE,$57,$F5,$7B,$CF,$FC,$45,$3A,$BF,$F4,$D2,$F0
	.dc.b	$79,$1C,$63,$2F,$5F,$0A,$46,$0E,$2F,$87,$E0,$64,$7F,$A9,$70,$7C
	.dc.b	$3E,$31,$F0,$F1,$79,$CC,$6D,$FD,$27,$3F,$F6,$F6,$1F,$EA,$DF,$AB
	.dc.b	$FC,$8B,$F6,$1B,$7F,$9B,$C7,$FD,$FB,$3B,$8F,$E9,$33,$FC,$AE,$FF
	.dc.b	$AA,$36,$79,$4E,$B8,$37,$E2,$49,$F4,$7C,$2D,$AF,$F2,$45,$BD,$4E
	.dc.b	$A6,$71,$EA,$F8,$3D,$C9,$87,$A6,$2D,$F2,$F5,$3C,$A7,$D4,$F9,$2B
	.dc.b	$5E,$FE,$7F,$49,$8A,$F0,$BE,$CF,$06,$4F,$CD,$F8,$DB,$5E,$6F,$0F
	.dc.b	$E0,$98,$D3,$E0,$BF,$13,$E6,$65,$FD,$5F,$7E,$DF,$BB,$E6,$BE,$49
	.dc.b	$8C,$3E,$4B,$F3,$3E,$96,$5F,$D5,$8B,$5F,$E6,$DF,$A8,$FC,$B3,$7B
	.dc.b	$BF,$BD,$FD,$5B,$E5,$FF,$BE,$D6,$B2,$FF,$04,$3F,$86,$C7,$FD,$D8
	.dc.b	$DC,$39,$46,$3B,$38,$FB,$12,$3F,$DE,$5F,$56,$FA,$BC,$5F,$8F,$DA
	.dc.b	$EE,$63,$2E,$EF,$8B,$8E,$61,$F9,$7F,$6A,$DF,$AF,$8B,$CE,$63,$0F
	.dc.b	$3B,$EA,$F8,$A9,$7E,$88,$B5,$F0,$FE,$2B,$DE,$7F,$87,$34,$FF,$48
	.dc.b	$57,$BA,$3E,$1F,$D5,$C8,$C1,$F0,$9F,$8F,$C3,$23,$F9,$D6,$5F,$E2
	.dc.b	$FF,$63,$F8,$A6,$3F,$46,$BF,$C5,$3C,$2F,$7A,$9C,$67,$FD,$45,$19
	.dc.b	$C3,$99,$FC,$93,$EA,$FF,$93,$4F,$F4,$76,$AF,$98,$FE,$EC,$FF,$AA
	.dc.b	$E3,$23,$E1,$7F,$C7,$2D,$AE,$B3,$CF,$D2,$CD,$6B,$E2,$25,$6A,$BB
	.dc.b	$6D,$87,$B7,$C0,$74,$31,$BF,$F9,$3F,$2F,$EF,$B3,$E3,$F3,$7C,$27
	.dc.b	$F9,$2B,$E9,$FD,$D1,$9C,$7B,$57,$F0,$E8,$F5,$79,$DC,$27,$2F,$F9
	.dc.b	$4B,$DB,$FF,$28,$4F,$FB,$25,$B8,$FF,$97,$F0,$FE,$DB,$3F,$DB,$67
	.dc.b	$37,$FD,$39,$FE,$CE,$D7,$A7,$B9,$D5,$ED,$E6,$C9,$F8,$56,$F5,$71
	.dc.b	$87,$87,$CB,$F2,$18,$CB,$FC,$E1,$8F,$F1,$5C,$3E,$53,$FD,$2D,$F1
	.dc.b	$7C,$3F,$F9,$B3,$EA,$7C,$78,$79,$E3,$DD,$F3,$D2,$B8,$C5,$9F,$F7
	.dc.b	$45,$3E,$0F,$FC,$FD,$1F,$DA,$A3,$15,$FE,$94,$84,$61,$FF,$4B,$B5
	.dc.b	$FF,$4F,$60,$F8,$9E,$71,$F2,$5A,$A3,$55,$D5,$B1,$FE,$A2,$5E,$6F
	.dc.b	$D7,$3E,$C5,$5A,$FC,$65,$BE,$4B,$AB,$63,$D0,$DC,$3C,$F3,$E2,$FA
	.dc.b	$D7,$CF,$52,$5F,$BE,$32,$F9,$73,$F4,$E7,$EF,$C3,$F0,$51,$83,$3F
	.dc.b	$17,$2B,$D8,$D9,$FF,$5E,$FC,$B3,$C7,$FF,$58,$57,$FB,$16,$30,$FF
	.dc.b	$8D,$E7,$FC,$79,$18,$7E,$AB,$63,$87,$8D,$CA,$7F,$C4,$33,$FE,$E5
	.dc.b	$9E,$50,$8C,$3C,$A7,$83,$F9,$DF,$42,$BF,$55,$B9,$1A,$6B,$37,$CB
	.dc.b	$D2,$63,$E0,$8C,$BE,$2C,$7F,$C2,$CF,$BC,$EE,$E3,$0F,$4C,$78,$BE
	.dc.b	$AF,$FA,$7B,$FE,$E9,$6C,$7F,$8F,$9A,$FF,$98,$5A,$9A,$4F,$47,$A1
	.dc.b	$D7,$C5,$95,$F1,$A7,$FA,$A3,$DE,$3C,$5F,$11,$FC,$BE,$FF,$DE,$B4
	.dc.b	$8C,$47,$97,$E7,$FF,$DD,$0E,$A3,$2F,$D4,$F2,$7F,$CA,$B8,$7D,$F6
	.dc.b	$C6,$4F,$8C,$E1,$FF,$3E,$7D,$67,$69,$FF,$2E,$C2,$30,$FF,$98,$1A
	.dc.b	$F0,$F2,$B9,$3F,$0F,$D0,$B9,$7F,$AD,$63,$E0,$46,$0D,$D3,$E3,$29
	.dc.b	$27,$F1,$71,$96,$F9,$3C,$08,$E0,$F9,$9E,$8F,$80,$8E,$0F,$A1,$CB
	.dc.b	$E4,$23,$83,$BF,$93,$FC,$45,$F8,$0D,$BF,$F6,$1C,$7F,$15,$6B,$F1
	.dc.b	$87,$1F,$F2,$F7,$F7,$9E,$1F,$38,$C4,$6D,$AF,$E3,$FC,$FF,$06,$A6
	.dc.b	$92,$3F,$92,$43,$CF,$FF,$48,$F8,$8E,$1F,$D9,$F3,$61,$B7,$FD,$8E
	.dc.b	$F8,$37,$E4,$34,$FE,$6F,$0A,$FE,$6D,$B2,$9C,$FD,$63,$E5,$F7,$E1
	.dc.b	$BD,$FA,$6A,$37,$F1,$77,$2F,$EE,$97,$C5,$B1,$D3,$D3,$F0,$F7,$7F
	.dc.b	$9F,$A7,$FB,$D3,$56,$FE,$F7,$FD,$A5,$6A,$F2,$FA,$7C,$E7,$BC,$61
	.dc.b	$F0,$FC,$3F,$BB,$CE,$FF,$4B,$47,$F6,$68,$F6,$9E,$EF,$87,$B1,$C0
	.dc.b	$F4,$BF,$0B,$9B,$96,$B2,$7D,$B1,$F1,$BE,$2B,$51,$83,$E0,$BF,$3B
	.dc.b	$EB,$1E,$9F,$8F,$09,$5F,$E1,$99,$CB,$7C,$8E,$15,$23,$07,$F8,$7F
	.dc.b	$EA,$5F,$59,$F6,$BE,$09,$F0,$94,$FB,$B3,$AA,$E1,$F0,$D8,$B9,$18
	.dc.b	$3A,$7A,$7F,$84,$FA,$C6,$B4,$E7,$1D,$3D,$5B,$5C,$0F,$23,$E0,$F4
	.dc.b	$BF,$1B,$F5,$9F,$EA,$19,$FF,$1B,$35,$FB,$BE,$0F,$FB,$B3,$45,$7A
	.dc.b	$36,$E5,$5E,$F8,$C3,$F0,$7E,$1A,$BC,$80,$A5,$7C,$29,$FC,$FF,$D5
	.dc.b	$FE,$2E,$B8,$0F,$AC,$FE,$60,$FF,$92,$DA,$EA,$38,$76,$F1,$7F,$DD
	.dc.b	$D1,$FE,$51,$73,$97,$8A,$11,$87,$CA,$F0,$7A,$7C,$67,$EC,$BE,$B3
	.dc.b	$FC,$1D,$1F,$E5,$B6,$BA,$3B,$7C,$47,$AA,$64,$60,$F1,$7D,$5F,$EF
	.dc.b	$3E,$B1,$F0,$74,$F8,$31,$F0,$BD,$1B,$5C,$0F,$53,$F1,$BD,$1D,$46
	.dc.b	$0F,$73,$F2,$BE,$5B,$EE,$FC,$14,$9F,$85,$1F,$3B,$D0,$D4,$60,$F8
	.dc.b	$CF,$D2,$F7,$BF,$88,$F1,$3E,$3E,$99,$AF,$99,$ED,$57,$E9,$57,$DA
	.dc.b	$8F,$E3,$3E,$E3,$EB,$3F,$31,$E5,$CB,$FF,$42,$4E,$5B,$E4,$F0,$A9
	.dc.b	$18,$37,$E7,$F4,$99,$18,$39,$FD,$17,$E9,$7E,$0E,$46,$0E,$FE,$E7
	.dc.b	$92,$64,$60,$F1,$F9,$DF,$B7,$F8,$59,$18,$3C,$FE,$37,$B2,$64,$60
	.dc.b	$F5,$FC,$4F,$F3,$8F,$8B,$74,$53,$E1,$C7,$C4,$EA,$B5,$C0,$F8,$6F
	.dc.b	$C8,$F5,$75,$18,$3E,$3B,$FD,$13,$EB,$1D,$F7,$BE,$57,$FA,$BE,$72
	.dc.b	$DF,$67,$87,$B7,$23,$07,$FA,$D7,$EB,$1F,$03,$FD,$41,$FE,$FD,$1F
	.dc.b	$B2,$6F,$8F,$FB,$57,$71,$C7,$83,$87,$CC,$48,$C1,$FE,$C1,$FA,$CF
	.dc.b	$F4,$03,$7D,$5F,$EB,$B1,$FC,$B2,$3F,$7E,$E9,$BF,$2B,$AF,$46,$47
	.dc.b	$38,$FE,$CB,$DF,$FD,$4F,$3F,$E1,$0F,$EE,$5B,$F0,$3A,$F8,$8D,$F8
	.dc.b	$FC,$3E,$72,$63,$FE,$D5,$C1,$3F,$DC,$3E,$B3,$FA,$D4,$FF,$B6,$57
	.dc.b	$85,$93,$55,$BF,$03,$87,$81,$2F,$84,$73,$F7,$7A,$7B,$D2,$7A,$47
	.dc.b	$F7,$EF,$AC,$78,$77,$DD,$95,$FE,$E9,$9C,$BF,$EE,$BF,$AC,$6F,$FE
	.dc.b	$06,$5F,$2D,$4C,$47,$F8,$2B,$BF,$D3,$5F,$EF,$37,$FC,$00,$FF,$2E
	.dc.b	$6F,$27,$F3,$17,$71,$9F,$0F,$97,$C7,$48,$E5,$1D,$76,$ED,$E1,$C9
	.dc.b	$ED,$1F,$E1,$AF,$AC,$FE,$C3,$1F,$D0,$B1,$FE,$7D,$0F,$81,$D1,$BD
	.dc.b	$FF,$C4,$3F,$59,$FC,$D6,$D5,$FC,$C2,$F8,$FC,$9E,$BD,$BD,$F0,$70
	.dc.b	$F5,$E6,$1D,$E7,$FA,$06,$3F,$1A,$30,$7F,$54,$7F,$02,$E0,$3F,$39
	.dc.b	$C3,$5F,$EA,$AF,$06,$FA,$DE,$8E,$C9,$1C,$23,$D7,$E4,$7B,$3C,$89
	.dc.b	$3D,$23,$DF,$EA,$6A,$30,$78,$1F,$F2,$1F,$D5,$BE,$23,$FC,$8B,$F5
	.dc.b	$8F,$7E,$9A,$8D,$F8,$17,$0F,$65,$F1,$6C,$7F,$93,$BE,$2F,$F6,$5C
	.dc.b	$7F,$DC,$7D,$4F,$F2,$75,$7C,$36,$BF,$DD,$27,$7C,$3F,$DA,$DF,$DC
	.dc.b	$57,$F7,$97,$33,$FE,$00,$6B,$FE,$55,$F8,$8F,$F2,$EF,$D6,$7E,$42
	.dc.b	$FF,$BA,$60,$E4,$FF,$30,$B6,$FE,$A5,$6D,$FF,$05,$BF,$96,$C7,$FC
	.dc.b	$10,$E6,$0D,$35,$7F,$E0,$17,$F7,$B1,$FE,$0E,$72,$F4,$F1,$BC,$BE
	.dc.b	$37,$9B,$6F,$1F,$2C,$AF,$1C,$7A,$B6,$D4,$60,$F3,$BE,$EF,$1B,$FC
	.dc.b	$CF,$1F,$E9,$5C,$7F,$D5,$D9,$FF,$A5,$E3,$CF,$3A,$7F,$D7,$F3,$A7
	.dc.b	$87,$A2,$FA,$CE,$13,$FC,$E1,$B1,$99,$DC,$7F,$AA,$57,$D3,$52,$3F
	.dc.b	$EA,$D7,$FD,$5D,$F4,$5F,$E7,$BC,$7F,$C6,$52,$8C,$6F,$FE,$AD,$AF
	.dc.b	$F1,$DF,$D6,$7F,$A3,$EB,$FC,$59,$5A,$7F,$C5,$DE,$95,$7F,$C8,$2D
	.dc.b	$7A,$5F,$EB,$DF,$F3,$8B,$79,$34,$71,$E3,$7C,$B6,$27,$87,$CE,$73
	.dc.b	$31,$7F,$E4,$80,$F7,$F9,$C9,$3C,$0F,$03,$E5,$D7,$FD,$31,$1F,$C0
	.dc.b	$5B,$5F,$E5,$2C,$3F,$96,$CE,$79,$78,$72,$48,$C1,$C5,$F8,$1F,$59
	.dc.b	$EE,$BF,$F2,$DB,$70,$FD,$77,$7E,$B7,$DE,$F3,$1F,$E2,$57,$B7,$FC
	.dc.b	$B0,$FC,$6C,$39,$F8,$BD,$8E,$FE,$35,$AE,$3E,$7F,$88,$C5,$78,$9F
	.dc.b	$2F,$AB,$27,$BF,$C7,$DA,$F3,$FA,$DE,$A3,$1B,$7F,$9D,$73,$47,$BB
	.dc.b	$E3,$A6,$1E,$8F,$9D,$B8,$7F,$9E,$BE,$6F,$EC,$CF,$F2,$D6,$C6,$7D
	.dc.b	$FF,$C3,$46,$32,$FF,$41,$2F,$23,$26,$BC,$9B,$87,$F3,$7F,$AC,$7C
	.dc.b	$E7,$E3,$DB,$19,$F0,$71,$39,$EA,$93,$C7,$DF,$BA,$FF,$A4,$3E,$B3
	.dc.b	$F1,$33,$FD,$0D,$B1,$9F,$03,$F3,$91,$8D,$BF,$D3,$0B,$83,$26,$B8
	.dc.b	$AE,$1F,$D3,$7E,$B3,$FD,$F4,$DD,$FF,$4D,$0E,$CB,$1F,$A9,$9F,$F1
	.dc.b	$A9,$FD,$54,$B9,$FF,$A9,$97,$8D,$BE,$0F,$D8,$7D,$67,$FA,$C2,$7F
	.dc.b	$AE,$36,$B8,$7B,$BA,$77,$BC,$AD,$F7,$59,$FF,$59,$DF,$E0,$27,$9E
	.dc.b	$BD,$BA,$FB,$BF,$BF,$F1,$9E,$D5,$31,$F4,$16,$FF,$B7,$4F,$F8,$C1
	.dc.b	$BE,$67,$1F,$73,$93,$FC,$27,$EB,$3F,$67,$CF,$FE,$C5,$6D,$7F,$AE
	.dc.b	$DF,$F1,$AB,$FD,$E5,$CE,$BF,$ED,$07,$FD,$5F,$19,$F8,$7E,$9E,$EF
	.dc.b	$57,$B7,$14,$9E,$5E,$E5,$C3,$FD,$A7,$E4,$7F,$8D,$A7,$FC,$0C,$D8
	.dc.b	$CF,$91,$FC,$7E,$31,$9F,$FB,$6F,$EB,$1F,$09,$FC,$8D,$B8,$7F,$83
	.dc.b	$D3,$98,$FE,$51,$F5,$8F,$87,$A7,$F2,$BF,$AC,$DD,$7F,$85,$E7,$4D
	.dc.b	$F0,$B8,$9C,$1C,$1B,$F0,$FA,$9A,$70,$73,$E1,$F0,$9D,$FE,$1E,$C7
	.dc.b	$7E,$2F,$29,$8E,$0F,$F7,$BE,$3F,$39,$FF,$7C,$FD,$67,$E3,$F7,$FF
	.dc.b	$7E,$36,$FF,$DE,$5A,$3A,$3F,$A3,$4E,$6C,$1F,$E4,$4E,$1A,$E4,$E3
	.dc.b	$C3,$CB,$D3,$98,$7F,$8E,$F0,$FC,$9B,$FE,$3D,$BF,$DC,$3D,$AF,$B7
	.dc.b	$A7,$E2,$38,$BE,$A7,$CB,$ED,$C9,$E5,$16,$7D,$FE,$97,$A8,$FA,$DF
	.dc.b	$27,$6B,$C1,$CF,$EE,$31,$5F,$19,$F8,$3E,$04,$9F,$47,$C6,$DA,$F9
	.dc.b	$7E,$0F,$C6,$31,$C3,$F3,$1F,$93,$F3,$52,$FE,$07,$BD,$6F,$E9,$FC
	.dc.b	$07,$CE,$31,$87,$D4,$7E,$8F,$D3,$4B,$F8,$D1,$69,$5F,$96,$7E,$AF
	.dc.b	$D5,$FE,$69,$CD,$7C,$B8,$C4,$EE,$5F,$58,$F9,$7F,$F0,$25,$6A,$F8
	.dc.b	$75,$E0,$CF,$81,$63,$FC,$E9,$3F,$EE,$06,$FA,$FF,$E0,$DF,$AB,$7D
	.dc.b	$67,$48,$EB,$F0,$12,$B6,$72,$7C,$3C,$72,$3F,$0D,$EC,$DD,$BD,$67
	.dc.b	$BB,$8B,$CE,$67,$83,$D9,$A7,$F9,$5E,$FC,$BF,$15,$EC,$36,$E0,$F4
	.dc.b	$FC,$4F,$C0,$3D,$1D,$56,$3D,$BF,$59,$F1,$0C,$65,$E2,$18,$23,$E4
	.dc.b	$F1,$CA,$FF,$4C,$7A,$EF,$1B,$18,$71,$E5,$7B,$BF,$E3,$36,$DF,$CC
	.dc.b	$C3,$13,$FE,$37,$FA,$CF,$E5,$36,$6F,$CB,$FF,$75,$BF,$D4,$46,$8F
	.dc.b	$F4,$A3,$FE,$A9,$6C,$FA,$5D,$7C,$27,$5F,$66,$57,$FA,$BE,$FD,$8F
	.dc.b	$18,$F1,$7B,$F2,$30,$75,$7C,$DA,$48,$FF,$5C,$7D,$3F,$B3,$2F,$04
	.dc.b	$FA,$FC,$D9,$5E,$58,$B3,$F3,$7D,$BF,$F1,$67,$37,$E5,$DB,$5F,$C1
	.dc.b	$70,$FE,$DF,$39,$F6,$FF,$CB,$94,$29,$86,$D7,$FB,$2E,$3F,$87,$DF
	.dc.b	$F1,$C6,$AB,$7A,$9D,$3C,$DE,$BE,$54,$C3,$94,$5B,$F4,$7C,$2F,$88
	.dc.b	$FF,$37,$7D,$4F,$8E,$BC,$11,$E6,$F9,$F9,$18,$3C,$8F,$0F,$82,$F5
	.dc.b	$FC,$3D,$9F,$43,$EE,$F3,$64,$60,$F6,$38,$C7,$F9,$51,$ED,$7F,$D1
	.dc.b	$2D,$8F,$F4,$F6,$C9,$CC,$BE,$AD,$F3,$98,$6D,$FF,$A9,$5B,$3D,$8E
	.dc.b	$13,$CA,$FF,$C6,$15,$DB,$1C,$11,$F0,$FC,$46,$C8,$E0,$F2,$F9,$FF
	.dc.b	$C7,$5F,$5D,$23,$18,$7F,$47,$8C,$4F,$F8,$AE,$3E,$E4,$E9,$FE,$B3
	.dc.b	$F2,$BF,$C1,$5F,$F5,$5B,$67,$E3,$B6,$3F,$A9,$35,$FC,$37,$A1,$CD
	.dc.b	$FC,$73,$6C,$D3,$1F,$29,$2F,$C9,$5F,$95,$8C,$3E,$28,$E9,$7F,$E0
	.dc.b	$A8,$C4,$70,$8C,$3F,$73,$EA,$FF,$9F,$37,$D0,$F2,$46,$2B,$FB,$53
	.dc.b	$5F,$EF,$2B,$C3,$D9,$2B,$CB,$19,$6C,$66,$7F,$DA,$33,$96,$C7,$83
	.dc.b	$DA,$FE,$37,$1F,$83,$9F,$F0,$57,$87,$FE,$EB,$5F,$41,$FE,$D9,$9D
	.dc.b	$57,$07,$D9,$F0,$DF,$E0,$1C,$6C,$79,$23,$0F,$B1,$AF,$F8,$53,$EB
	.dc.b	$39,$4F,$E8,$1B,$1F,$E0,$B6,$BB,$C7,$8A,$C7,$FD,$B6,$FC,$2F,$47
	.dc.b	$31,$88,$F8,$31,$8F,$66,$47,$C8,$6B,$FE,$F6,$B7,$8B,$D9,$B1,$F8
	.dc.b	$26,$BB,$8E,$FE,$94,$8F,$A6,$D7,$E8,$FA,$3E,$42,$3F,$61,$F3,$FA
	.dc.b	$3F,$B3,$8F,$BE,$D7,$DC,$DC,$8E,$7A,$BF,$8F,$CF,$EF,$7C,$2C,$11
	.dc.b	$FB,$5F,$0B,$C4,$8E,$0F,$BD,$E9,$7F,$8E,$7C,$7F,$F0,$76,$28,$C7
	.dc.b	$0F,$E5,$1B,$5F,$94,$F4,$2B,$F4,$FD,$0B,$D3,$FE,$76,$8F,$F2,$8E
	.dc.b	$3F,$E1,$98,$C4,$7F,$73,$C7,$FD,$6C,$EB,$1F,$D0,$FC,$57,$49,$CC
	.dc.b	$EF,$0C,$4F,$83,$EB,$1F,$43,$E2,$F2,$1E,$F5,$35,$1B,$F9,$6B,$8F
	.dc.b	$73,$FC,$CB,$7C,$9C,$05,$1D,$FE,$4B,$DB,$DF,$2B,$74,$9F,$27,$A9
	.dc.b	$B9,$CC,$61,$DF,$C3,$7C,$8F,$40,$D3,$D8,$7B,$7E,$1A,$CF,$03,$BB
	.dc.b	$F0,$79,$F9,$69,$27,$C7,$1F,$17,$E2,$75,$18,$3C,$EF,$D8,$FA,$C7
	.dc.b	$A9,$E3,$C2,$57,$C2,$9C,$B7,$CA,$E1,$52,$30,$7C,$8F,$A9,$7D,$67
	.dc.b	$E1,$3C,$D7,$C3,$53,$F0,$F3,$AA,$E1,$EF,$E2,$E4,$60,$E9,$EA,$7E
	.dc.b	$3B,$EB,$1A,$D3,$9C,$74,$F5,$6D,$70,$3C,$8F,$83,$D3,$FC,$BF,$D6
	.dc.b	$3D,$AA,$66,$BD,$5B,$57,$FC,$91,$AA,$FD,$33,$EA,$F8,$7B,$5C,$0F
	.dc.b	$84,$FE,$93,$EB,$3F,$CF,$1E,$2F,$F9,$E7,$EA,$7F,$D6,$D5,$F8,$E6
	.dc.b	$BF,$E5,$59,$DF,$8B,$FE,$AB,$7F,$4F,$5F,$E8,$17,$33,$FE,$BC,$6A
	.dc.b	$77,$3C,$7D,$3F,$DA,$FD,$63,$D8,$A7,$96,$3C,$DF,$0F,$70,$FD,$0E
	.dc.b	$09,$CC,$63,$EA,$FF,$7F,$F5,$8F,$83,$A7,$BE,$3E,$07,$A9,$6B,$81
	.dc.b	$D1,$F8,$9E,$A6,$A3,$07,$81,$F9,$1E,$97,$D6,$F8,$09,$3E,$48,$F9
	.dc.b	$9E,$9E,$A3,$07,$A1,$FA,$1F,$13,$FC,$53,$8D,$EC,$D3,$35,$EE,$F8
	.dc.b	$8A,$FD,$1A,$FB,$11,$FC,$73,$DE,$7D,$67,$E4,$7C,$D9,$7F,$C1,$A7
	.dc.b	$2D,$F2,$F8,$54,$8C,$1B,$F4,$7A,$4C,$8C,$1C,$FE,$87,$F3,$DF,$09
	.dc.b	$23,$07,$7F,$9A,$F2,$4C,$8C,$1E,$3F,$43,$F6,$1F,$0D,$23,$07,$9F
	.dc.b	$C8,$F6,$4C,$8C,$1E,$BF,$88,$FE,$77,$F1,$4E,$8A,$7C,$38,$F8,$9D
	.dc.b	$56,$B8,$1F,$0D,$F9,$1E,$EE,$A3,$07,$C7,$7F,$A3,$FD,$63,$C0,$BD
	.dc.b	$F2,$BF,$98,$CE,$5B,$EE,$70,$F7,$64,$60,$FE,$6F,$F5,$9F,$DD,$3C
	.dc.b	$7F,$F7,$32,$E3,$FE,$AA,$D9,$DF,$C1,$E3,$87,$3F,$2F,$27,$9C,$7F
	.dc.b	$5C,$FA,$CF,$F4,$A4,$7F,$44,$7F,$91,$BF,$EC,$B6,$E3,$FE,$F5,$77
	.dc.b	$19,$F9,$4E,$5E,$94,$8E,$51,$FD,$A3,$BB,$FC,$79,$1F,$D4,$1F,$DC
	.dc.b	$A3,$FA,$67,$C5,$EF,$E0,$78,$78,$DC,$7E,$6E,$57,$18,$FE,$E7,$F5
	.dc.b	$9F,$D4,$E3,$FA,$DB,$FD,$69,$BE,$0E,$B1,$DF,$17,$0E,$F9,$5C,$23
	.dc.b	$9F,$BF,$D3,$E0,$24,$F4,$8F,$F0,$0F,$D6,$3C,$2B,$EF,$4A,$FE,$D7
	.dc.b	$39,$7F,$B6,$FD,$63,$7F,$F0,$4B,$FE,$1C,$8C,$FC,$5E,$27,$FC,$1B
	.dc.b	$DD,$E9,$9F,$F7,$D3,$FD,$DD,$BE,$3E,$B8,$37,$E1,$F0,$FA,$C9,$5F
	.dc.b	$DF,$2F,$FD,$F8,$DD,$FA,$F8,$72,$30,$7F,$80,$BE,$B3,$FA,$CB,$7E
	.dc.b	$0B,$FC,$3F,$06,$F0,$7F,$88,$BE,$B3,$F9,$9C,$FF,$2F,$AE,$5E,$C6
	.dc.b	$7D,$AD,$70,$EF,$D6,$95,$FE,$10,$C3,$E1,$2F,$05,$7F,$83,$84,$BB
	.dc.b	$3C,$1C,$92,$79,$47,$8F,$D5,$F2,$32,$3B,$47,$9F,$C8,$F4,$79,$12
	.dc.b	$7C,$51,$EB,$F4,$F5,$18,$3C,$CF,$F9,$03,$EA,$DF,$13,$FE,$42,$FA
	.dc.b	$C7,$BD,$4D,$46,$FC,$2B,$87,$F8,$BA,$F8,$B6,$3F,$C9,$7F,$19,$FA
	.dc.b	$CE,$0F,$81,$EA,$34,$FF,$27,$B6,$FF,$C6,$F5,$B7,$F5,$0F,$EC,$B8
	.dc.b	$BE,$4C,$63,$E2,$3A,$9D,$BE,$27,$FC,$B3,$F5,$9F,$8B,$AD,$D7,$F8
	.dc.b	$3B,$9B,$B9,$AF,$0B,$7E,$15,$9F,$F2,$3D,$F1,$D7,$EB,$3F,$97,$C3
	.dc.b	$9B,$FB,$39,$FB,$93,$AA,$E7,$1C,$BC,$8F,$1F,$0F,$BB,$9E,$4F,$68
	.dc.b	$F3,$F0,$EA,$30,$78,$9F,$5F,$91,$F8,$98,$FF,$7F,$65,$FC,$8F,$3F
	.dc.b	$F4,$73,$63,$33,$B7,$F1,$C2,$3F,$D4,$7F,$59,$C3,$D2,$34,$93,$FE
	.dc.b	$97,$FA,$6F,$F4,$C2,$69,$1F,$F4,$F7,$D1,$FC,$E7,$BB,$5F,$CF,$35
	.dc.b	$FE,$44,$D7,$FC,$AF,$BF,$FA,$97,$8F,$04,$7F,$D5,$1C,$7B,$23,$83
	.dc.b	$FD,$59,$F3,$7C,$C8,$E0,$FF,$58,$7C,$57,$72,$38,$3F,$D6,$DB,$7F
	.dc.b	$9F,$DB,$C5,$E7,$8C,$63,$F2,$8F,$DD,$46,$78,$65,$72,$6C,$EF,$EA
	.dc.b	$DA,$7C,$16,$D7,$FB,$15,$B3,$B9,$D6,$92,$B4,$D9,$E9,$E2,$F3,$24
	.dc.b	$F3,$7B,$79,$D2,$3B,$36,$7C,$9E,$A7,$8C,$93,$E3,$7C,$DE,$A4,$8F
	.dc.b	$33,$67,$D9,$F1,$5E,$B2,$4F,$AD,$FB,$DF,$32,$F9,$9A,$49,$FF,$72
	.dc.b	$46,$5B,$E2,$70,$23,$83,$FD,$D5,$F5,$6F,$9E,$FC,$53,$63,$31,$83
	.dc.b	$3E,$14,$8E,$2D,$CB,$ED,$7D,$5B,$9F,$F2,$8D,$AF,$F7,$C3,$67,$73
	.dc.b	$AF,$9F,$95,$A6,$CF,$E1,$7E,$AF,$F8,$E4,$FF,$C0,$16,$C6,$7C,$6F
	.dc.b	$D1,$46,$23,$F6,$B2,$EE,$FD,$33,$4F,$C8,$47,$F4,$88,$C9,$FA,$BF
	.dc.b	$AB,$FF,$06,$4E,$9B,$3F,$97,$6B,$FC,$09,$AF,$F5,$15,$DB,$CC,$93
	.dc.b	$CD,$ED,$C7,$23,$B3,$6B,$F4,$9D,$DF,$E8,$D9,$F3,$B6,$33,$EA,$FF
	.dc.b	$01,$8C,$47,$F1,$47,$FB,$04,$F0,$F1,$34,$47,$E0,$FD,$5F,$EC,$AB
	.dc.b	$F8,$DB,$5F,$ED,$4D,$7F,$C5,$CD,$4F,$FA,$76,$39,$FC,$2F,$F1,$38
	.dc.b	$C4,$FE,$F5,$AF,$6F,$73,$44,$AF,$1B,$FC,$6F,$BB,$FC,$4D,$1F,$C8
	.dc.b	$BB,$30,$47,$EC,$FD,$5B,$E0,$3F,$91,$36,$33,$18,$3F,$93,$7D,$5B
	.dc.b	$E1,$29,$27,$F0,$71,$96,$FC,$27,$02,$38,$37,$D5,$D0,$8E,$0E,$7E
	.dc.b	$FF,$80,$8E,$0E,$FB,$F9,$08,$E0,$FC,$AF,$D5,$7E,$33,$7F,$F2,$F3
	.dc.b	$53,$49,$B7,$F4,$37,$F9,$0C,$7F,$56,$6B,$FE,$25,$6B,$D3,$AF,$81
	.dc.b	$2B,$C2,$F3,$F2,$A4,$79,$1B,$3F,$CE,$69,$E9,$78,$49,$5E,$A7,$C9
	.dc.b	$EE,$C8,$F6,$B6,$B9,$78,$FE,$92,$4F,$C2,$7D,$9E,$3C,$8F,$8A,$D9
	.dc.b	$F1,$75,$7C,$12,$4F,$CA,$7E,$27,$CB,$48,$F9,$AD,$9F,$57,$AD,$F2
	.dc.b	$49,$3F,$49,$F9,$9F,$4B,$23,$EA,$B6,$7F,$AD,$7D,$3F,$E4,$5B,$CF
	.dc.b	$FD,$D6,$31,$1F,$E0,$86,$BF,$E5,$75,$F0,$FB,$25,$FF,$16,$8C,$B7
	.dc.b	$E2,$FA,$11,$C1,$BF,$33,$C0,$47,$07,$3F,$8C,$F2,$11,$C1,$DF,$DA
	.dc.b	$F4,$11,$C1,$E3,$F8,$1F,$61,$1C,$1E,$7F,$7F,$E0,$11,$C1,$EB,$FA
	.dc.b	$3F,$88,$47,$07,$BF,$E4,$BE,$41,$1C,$1F,$CE,$34,$7A,$5F,$E2,$16
	.dc.b	$C6,$63,$07,$F8,$97,$E9,$FF,$8C,$13,$D8,$C1,$9F,$62,$57,$F4,$49
	.dc.b	$FF,$3B,$B6,$39,$79,$D2,$34,$DE,$2F,$F0,$D5,$7F,$69,$6F,$A7,$E1
	.dc.b	$25,$75,$F0,$BC,$84,$70,$78,$7C,$2F,$41,$1C,$1F,$D6,$3C,$EF,$F1
	.dc.b	$ED,$FF,$8A,$5A,$FF,$95,$1A,$FF,$A8,$DA,$FF,$B9,$DF,$EA,$B1,$CD
	.dc.b	$FF,$25,$18,$8E,$8E,$3C,$29,$3E,$16,$CF,$87,$C2,$E2,$49,$F3,$3D
	.dc.b	$3C,$99,$1E,$96,$CF,$A7,$E5,$BC,$24,$9F,$73,$E4,$F9,$D9,$1F,$05
	.dc.b	$B3,$F0,$7C,$1F,$49,$27,$E3,$3E,$CF,$06,$47,$C9,$6C,$FC,$9F,$53
	.dc.b	$E0,$92,$7E,$73,$F1,$3D,$89,$1F,$45,$B3,$F4,$7C,$3F,$F5,$4C,$FD
	.dc.b	$16,$C7,$D1,$6B,$F4,$5A,$FF,$81,$BC,$3B,$BF,$FB,$12,$BF,$C1,$6D
	.dc.b	$9F,$F4,$13,$63,$EF,$78,$7C,$08,$F0,$79,$FC,$3C,$8E,$6D,$AF,$F4
	.dc.b	$67,$D2,$7F,$BA,$27,$FC,$32,$D8,$CC,$79,$A3,$0F,$FA,$81,$FE,$3C
	.dc.b	$DF,$17,$44,$79,$F6,$71,$23,$83,$BF,$CB,$7F,$9E,$AF,$FD,$05,$68
	.dc.b	$C4,$77,$6B,$FE,$FD,$8D,$79,$7F,$E3,$A5,$F4,$FF,$C1,$27,$9B,$5E
	.dc.b	$53,$B8,$C4,$73,$F5,$8F,$C2,$6B,$F1,$9B,$87,$C1,$6C,$78,$FC,$8A
	.dc.b	$F2,$C1,$18,$84,$61,$F9,$F1,$98,$FF,$3E,$47,$D2,$F2,$30,$47,$EA
	.dc.b	$FC,$C6,$C8,$E0,$FB,$3F,$35,$CC,$8E,$0F,$BB,$E5,$77,$23,$83,$F0
	.dc.b	$3E,$57,$8C,$8E,$0F,$C2,$F9,$5F,$E6,$87,$D6,$DC,$4D,$31,$F1,$72
	.dc.b	$7E,$B3,$5F,$BF,$EC,$71,$23,$F6,$DE,$9E,$64,$8F,$BC,$D7,$F0,$FE
	.dc.b	$67,$84,$8F,$E0,$DF,$27,$C5,$C8,$FC,$33,$5F,$C7,$FB,$BE,$92,$3F
	.dc.b	$8B,$7D,$DF,$48,$F8,$AA,$49,$FF,$44,$C6,$5B,$F1,$5C,$08,$E0,$DF
	.dc.b	$9F,$D0,$8E,$0E,$7E,$7F,$80,$8E,$0E,$FE,$C7,$90,$8E,$0F,$1F,$B1
	.dc.b	$E8,$23,$83,$CF,$F2,$1E,$C2,$38,$3D,$7E,$EF,$C0,$23,$83,$FD,$4F
	.dc.b	$F4,$6F,$86,$FC,$13,$63,$31,$83,$3E,$2C,$8E,$2D,$7F,$E0,$07,$D0
	.dc.b	$3E,$8F,$F1,$8D,$8F,$F5,$AB,$5D,$C6,$BD,$C9,$1A,$6B,$D3,$CF,$E6
	.dc.b	$47,$9B,$DB,$CF,$91,$D9,$AF,$93,$CD,$F1,$91,$F1,$BE,$6F,$36,$47
	.dc.b	$99,$AF,$B3,$CD,$F5,$91,$F5,$BF,$A9,$FA,$07,$8F,$49,$3F,$ED,$48
	.dc.b	$CB,$7C,$CE,$04,$70,$6F,$C8,$E8,$47,$07,$3F,$23,$C0,$47,$07,$7F
	.dc.b	$1B,$C8,$47,$07,$8F,$C6,$F4,$11,$C1,$E7,$F1,$BD,$84,$70,$7A,$FC
	.dc.b	$3F,$80,$47,$07,$BF,$C3,$F8,$84,$70,$7C,$3F,$89,$F9,$04,$70,$7C
	.dc.b	$7F,$73,$E6,$11,$C1,$FE,$FE,$F3,$5F,$3D,$FC,$65,$B1,$98,$C1,$9F
	.dc.b	$12,$47,$16,$BF,$D1,$0F,$E8,$F2,$F0,$FA,$12,$3C,$0F,$7F,$90,$91
	.dc.b	$E3,$6B,$CC,$FE,$C1,$F2,$1E,$52,$3E,$97,$D1,$E1,$C8,$F6,$35,$E5
	.dc.b	$6F,$0F,$DC,$48,$F8,$4F,$C1,$F6,$E4,$7C,$56,$BC,$BD,$BF,$8C,$47
	.dc.b	$E5,$3F,$27,$E5,$64,$7C,$D6,$BE,$BF,$9C,$FE,$BC,$DE,$6F,$E9,$11
	.dc.b	$88,$F9,$4D,$7F,$B7,$AF,$2F,$64,$8F,$8C,$23,$4E,$3E,$71,$F0,$3F
	.dc.b	$D5,$5B,$19,$8C,$19,$D6,$47,$16,$BF,$4F,$E7,$1E,$3F,$F5,$F6,$C7
	.dc.b	$89,$AE,$E3,$5E,$E4,$8D,$35,$FB,$DE,$77,$FA,$F9,$F5,$B5,$CF,$B3
	.dc.b	$FD,$B6,$31,$1F,$8F,$5D,$79,$24,$72,$7F,$BA,$7C,$E2,$69,$27,$E4
	.dc.b	$C6,$5B,$CB,$C0,$8E,$0D,$F9,$DD,$08,$E0,$FA,$7F,$38,$F8,$0F,$F0
	.dc.b	$23,$63,$31,$83,$FC,$11,$F3,$8F,$5E,$92,$7E,$EC,$65,$FB,$FE,$2F
	.dc.b	$EB,$5B,$DB,$FE,$11,$8C,$47,$EA,$5A,$FE,$D5,$75,$6C,$91,$D1,$E5
	.dc.b	$E9,$C8,$EE,$D7,$F6,$9F,$36,$F8,$9F,$F1,$03,$63,$F2,$6D,$77,$1F
	.dc.b	$E2,$5F,$9B,$FE,$73,$1F,$BD,$6B,$FC,$25,$AF,$F9,$39,$AF,$F9,$49
	.dc.b	$FF,$15,$BF,$A0,$6B,$CF,$DB,$EC,$77,$E3,$91,$B6,$BE,$1F,$1F,$C8
	.dc.b	$47,$A3,$E7,$ED,$91,$DD,$AF,$A7,$C7,$F6,$11,$F2,$3F,$E4,$3F,$9A
	.dc.b	$7B,$34,$93,$FB,$88,$CB,$7C,$6E,$04,$70,$7E,$FB,$E3,$3F,$C1,$8F
	.dc.b	$F2,$56,$BF,$CC,$5A,$FF,$9D,$9A,$FF,$9E,$9F,$DA,$BB,$ED,$FF,$2C
	.dc.b	$46,$23,$84,$7F,$96,$FE,$6B,$FC,$75,$1F,$CC,$DA,$FF,$3F,$6B,$FE
	.dc.b	$84,$6B,$FE,$88,$7F,$22,$E7,$93,$A1,$1F,$E3,$31,$FB,$17,$B7,$67
	.dc.b	$84,$8E,$B2,$F2,$11,$C1,$CB,$1F,$41,$1C,$1D,$BA,$DF,$33,$49,$1E
	.dc.b	$D8,$CB,$EF,$F0,$9F,$35,$E6,$6C,$66,$30,$7F,$A0,$57,$CA,$D2,$3F
	.dc.b	$E8,$57,$FD,$0D,$F0,$DF,$9C,$83,$49,$EA,$FF,$A2,$FD,$ED,$11,$C1
	.dc.b	$FE,$8F,$E0,$E4,$47,$07,$FA,$4F,$E3,$3B,$11,$C1,$FE,$97,$F8,$CF
	.dc.b	$11,$1C,$1F,$D3,$35,$FE,$50,$DE,$0F,$D9,$C6,$32,$FE,$EC,$D7,$FC
	.dc.b	$04,$FE,$82,$79,$70,$70,$23,$FD,$63,$E6,$BE,$0B,$74,$FD,$FC,$62
	.dc.b	$FF,$C0,$4D,$7F,$C1,$CF,$F9,$BE,$79,$69,$C0,$8F,$5E,$CE,$64,$70
	.dc.b	$78,$7C,$3E,$E4,$70,$79,$7D,$DF,$19,$1C,$1E,$9F,$27,$CE,$47,$07
	.dc.b	$B7,$DD,$F5,$91,$C1,$FD,$E3,$E4,$1F,$13,$FC,$79,$B1,$98,$C1,$9E
	.dc.b	$D9,$1C,$5B,$AF,$F8,$9F,$E6,$9F,$21,$FC,$A1,$B5,$FE,$06,$6C,$EE
	.dc.b	$75,$EB,$4A,$D3,$67,$FC,$65,$F3,$4E,$7F,$E6,$6D,$AF,$F0,$8B,$67
	.dc.b	$73,$AF,$90,$95,$A6,$CF,$F8,$F7,$E6,$BF,$C3,$93,$FE,$1A,$6C,$67
	.dc.b	$C4,$FE,$83,$18,$8F,$F2,$74,$FF,$87,$0F,$F1,$09,$5F,$F2,$43,$5D
	.dc.b	$F7,$7F,$4A,$F9,$A4,$52,$4F,$F8,$AA,$32,$DE,$DE,$04,$70,$6F,$8F
	.dc.b	$A1,$1C,$1F,$E3,$4E,$E7,$9B,$FD,$69,$B1,$98,$C1,$9F,$22,$47,$16
	.dc.b	$E3,$FE,$63,$F9,$AF,$D7,$3E,$A0,$E1,$F6,$B5,$CF,$81,$FD,$AA,$1E
	.dc.b	$DD,$25,$FF,$92,$A3,$2D,$F4,$F8,$11,$C1,$FE,$50,$F0,$3F,$C3,$CD
	.dc.b	$ED,$FE,$EF,$18,$C7,$FC,$F4,$D7,$FD,$0D,$F3,$4F,$84,$FE,$FA,$DB
	.dc.b	$FF,$2D,$B6,$7F,$D0,$4D,$7F,$D1,$DF,$32,$F9,$AF,$F0,$2B,$6B,$FC
	.dc.b	$C6,$D9,$FF,$45,$35,$E3,$ED,$F0,$24,$F0,$79,$F5,$C8,$E6,$D9,$E7
	.dc.b	$EA,$F8,$09,$3E,$07,$C7,$BC,$8F,$1B,$67,$FD,$39,$F3,$5F,$8E,$69
	.dc.b	$FE,$1E,$70,$FF,$AA,$DA,$FD,$F7,$F9,$13,$75,$E0,$4A,$FF,$41,$CF
	.dc.b	$EC,$E7,$97,$67,$52,$4E,$BC,$AF,$01,$1C,$1C,$BD,$0F,$21,$1C,$1D
	.dc.b	$BE,$33,$D0,$47,$07,$8B,$C8,$F6,$11,$C1,$E6,$F2,$3E,$01,$1C,$1E
	.dc.b	$AE,$BF,$88,$47,$07,$BB,$E2,$FE,$41,$1C,$1F,$0B,$E2,$FE,$61,$1C
	.dc.b	$1F,$1B,$E5,$FE,$81,$1C,$1F,$EA,$CF,$9A,$FC,$86,$DF,$AF,$6B,$FC
	.dc.b	$29,$A9,$A3,$E0,$3F,$CB,$2D,$9E,$91,$83,$3F,$09,$23,$C0,$DA,$FF
	.dc.b	$71,$57,$F4,$46,$F9,$9D,$09,$5E,$7F,$8B,$EE,$47,$07,$AF,$E2,$FC
	.dc.b	$64,$70,$7B,$FD,$CF,$39,$1C,$1F,$0F,$DC,$F5,$91,$C1,$F1,$FE,$53
	.dc.b	$DE,$47,$07,$CB,$F8,$CF,$86,$47,$07,$FB,$6B,$57,$A7,$FE,$80,$6C
	.dc.b	$66,$30,$7F,$A1,$3E,$6B,$F8,$F1,$3C,$7C,$19,$F6,$25,$7F,$B9,$A7
	.dc.b	$FC,$29,$BF,$F2,$93,$FA,$1B,$87,$0F,$F4,$97,$F9,$73,$5D,$F9,$DE
	.dc.b	$13,$C5,$E9,$CA,$EC,$D9,$EC,$2B,$FD,$63,$3E,$3F,$07,$D4,$47,$CD
	.dc.b	$E0,$FB,$48,$E0,$FF,$7F,$F9,$DF,$E2,$1B,$EA,$D7,$D8,$D7,$FA,$33
	.dc.b	$5F,$EC,$2F,$FB,$EA,$39,$BF,$EA,$E3,$11,$D1,$C7,$81,$27,$C2,$D9
	.dc.b	$F0,$F8,$1C,$49,$3E,$67,$A7,$93,$23,$D2,$D9,$F4,$FC,$D7,$84,$93
	.dc.b	$EE,$7C,$9F,$23,$23,$E0,$B6,$7E,$0F,$81,$E9,$24,$FC,$67,$D9,$E0
	.dc.b	$48,$F9,$2D,$9F,$93,$EA,$7C,$12,$4F,$CE,$7E,$27,$B1,$23,$E8,$B6
	.dc.b	$7E,$8F,$84,$9F,$E6,$FB,$23,$13,$FE,$DE,$8C,$4E,$A7,$2F,$D8,$F0
	.dc.b	$AD,$7D,$86,$BF,$61,$AA,$91,$85,$E1,$52,$57,$91,$D7,$CD,$48,$F3
	.dc.b	$B6,$FF,$DE,$7F,$31,$EB,$9F,$EA,$2D,$8C,$C7,$8A,$30,$FE,$AD,$FE
	.dc.b	$F0,$DF,$3F,$44,$79,$F6,$71,$23,$83,$BF,$CD,$7F,$02,$BF,$D4,$DA
	.dc.b	$31,$1D,$FE,$43,$FD,$52,$EA,$32,$FF,$89,$67,$87,$77,$EC,$A1,$18
	.dc.b	$78,$46,$63,$11,$EB,$F5,$7F,$C5,$D8,$58,$F8,$71,$87,$B3,$5F,$F2
	.dc.b	$6C,$BC,$8C,$47,$81,$B1,$98,$F9,$B1,$8F,$1E,$47,$D0,$6B,$F5,$3E
	.dc.b	$6B,$89,$1F,$AA,$F4,$F9,$49,$1F,$61,$AF,$DC,$F2,$7C,$24,$7E,$EB
	.dc.b	$E4,$F2,$64,$7E,$01,$AF,$E1,$3C,$9F,$94,$FC,$A6,$E4,$69,$8F,$8C
	.dc.b	$93,$F3,$5A,$FD,$CF,$63,$89,$1F,$A4,$F4,$F2,$E4,$7D,$56,$BF,$84
	.dc.b	$F2,$FC,$24,$7E,$D3,$E4,$F9,$A9,$1F,$75,$AF,$E3,$3D,$DF,$49,$1F
	.dc.b	$C1,$3E,$EF,$9A,$7C,$AD,$24,$FE,$2E,$32,$DF,$8A,$E0,$47,$06,$FC
	.dc.b	$FE,$84,$70,$73,$F3,$FC,$04,$70,$77,$F6,$3C,$84,$70,$78,$FD,$8F
	.dc.b	$41,$1C,$1E,$7F,$8E,$F6,$11,$C1,$EB,$F7,$7E,$01,$1C,$1F,$A9,$F9
	.dc.b	$A7,$C3,$7E,$09,$B1,$98,$C1,$9F,$16,$47,$16,$BF,$D9,$3E,$69,$F3
	.dc.b	$5F,$8C,$6C,$7E,$DD,$AE,$E3,$5E,$DC,$8D,$35,$E9,$E7,$F3,$23,$CD
	.dc.b	$ED,$E7,$C8,$EC,$D7,$C9,$E6,$F8,$C8,$F8,$DF,$37,$9B,$23,$CC,$D7
	.dc.b	$D9,$E6,$FA,$C8,$FA,$DF,$D4,$FC,$D7,$F4,$48,$FF,$02,$B5,$FF,$0A
	.dc.b	$35,$FF,$5F,$35,$FF,$6C,$BF,$EA,$97,$F8,$E3,$5E,$7E,$67,$63,$BF
	.dc.b	$8F,$23,$6D,$7C,$3E,$3F,$90,$8F,$47,$CF,$E2,$C8,$EE,$D7,$D3,$E2
	.dc.b	$FB,$08,$F9,$1F,$7F,$8B,$23,$CE,$D7,$E0,$F8,$5F,$10,$8F,$B1,$F8
	.dc.b	$FE,$14,$8F,$7B,$5F,$93,$F1,$3F,$30,$8F,$C4,$7E,$7F,$B7,$23,$E3
	.dc.b	$B5,$FF,$1A,$F9,$AF,$9A,$FE,$3A,$D8,$FE,$90,$D7,$71,$AF,$0E,$46
	.dc.b	$9A,$F4,$F0,$B9,$91,$E6,$F6,$F8,$F9,$1D,$9A,$F9,$3E,$3F,$C6,$47
	.dc.b	$C6,$F9,$BC,$29,$1E,$66,$BE,$CF,$0B,$D6,$47,$D6,$FB,$BD,$B9,$1E
	.dc.b	$E6,$BF,$13,$DB,$F8,$64,$7E,$1B,$F1,$BE,$6A,$47,$C6,$6B,$FE,$60
	.dc.b	$F9,$AF,$EB,$CD,$E6,$FE,$93,$18,$8F,$F3,$93,$51,$17,$95,$C0,$91
	.dc.b	$FD,$EE,$32,$FF,$7D,$F9,$A7,$C0,$FF,$56,$6C,$66,$30,$67,$49,$1C
	.dc.b	$5A,$FF,$9E,$3E,$6B,$FC,$5C,$FF,$82,$5A,$E7,$C6,$FE,$C7,$18,$8F
	.dc.b	$F4,$62,$F7,$32,$48,$E4,$FF,$6A,$F3,$7F,$D8,$D1,$FE,$15,$6B,$FE
	.dc.b	$88,$6B,$FE,$97,$6B,$F1,$82,$EA,$E8,$48,$E8,$FF,$78,$F9,$AD,$C7
	.dc.b	$F8,$8D,$AF,$FA,$59,$AF,$FA,$95,$AF,$75,$C7,$CC,$91,$CD,$ED,$E6
	.dc.b	$48,$EC,$D7,$FD,$47,$F3,$4F,$80,$FF,$05,$B6,$3F,$C6,$CD,$77,$1F
	.dc.b	$E1,$0F,$9A,$7A,$B4,$93,$FE,$3D,$8C,$BF,$E3,$FF,$11,$D7,$FE,$18
	.dc.b	$6C,$66,$30,$67,$A6,$47,$16,$BF,$EB,$A9,$7A,$3C,$C9,$1D,$DF,$F1
	.dc.b	$2F,$CD,$3E,$22,$92,$7F,$CA,$51,$97,$FC,$A9,$F3,$5F,$CF,$53,$49
	.dc.b	$EE,$FF,$8C,$1F,$F1,$74,$70,$F6,$B0,$47,$9F,$1E,$C8,$E0,$EF,$E3
	.dc.b	$F3,$23,$83,$C7,$DB,$DC,$8E,$0F,$3F,$8F,$E3,$23,$83,$FC,$DD,$F3
	.dc.b	$4F,$63,$FC,$8E,$D8,$CC,$60,$CF,$8B,$23,$8B,$5F,$F7,$4F,$C5,$FF
	.dc.b	$83,$9B,$EC,$7F,$94,$A3,$11,$FE,$FB,$6B,$FA,$57,$F6,$8D,$EF,$E0
	.dc.b	$47,$FD,$09,$F3,$5F,$E3,$A7,$FD,$EE,$D7,$F5,$C1,$C3,$FB,$16,$BF
	.dc.b	$C6,$1F,$C6,$BC,$B9,$BF,$CC,$F1,$88,$E6,$D7,$97,$76,$88,$F9,$1E
	.dc.b	$39,$48,$F3,$B5,$F3,$E5,$D8,$8F,$B1,$F0,$F6,$BE,$67,$DE,$D9,$F7
	.dc.b	$B5,$F2,$B5,$F7,$78,$6F,$9A,$F8,$ED,$8F,$8E,$D7,$DA,$D7,$E4,$8F
	.dc.b	$95,$FF,$44,$B4,$FF,$45,$B8,$3F,$D1,$BF,$23,$FE,$E8,$C3,$FC,$49
	.dc.b	$C1,$7A,$3E,$9C,$DB,$FE,$45,$35,$FE,$92,$DF,$31,$E7,$E6,$7F,$CB
	.dc.b	$37,$FD,$3B,$0F,$F1,$95,$6A,$30,$7E,$03,$8B,$87,$C6,$F1,$6C,$6A
	.dc.b	$FF,$99,$4F,$5F,$8D,$CC,$7C,$F9,$FC,$4C,$65,$E1,$0B,$FF,$6E,$46
	.dc.b	$21,$3E,$9F,$88,$F6,$63,$FC,$FA,$B8,$BF,$AD,$6F,$1E,$B8,$E4,$76
	.dc.b	$C5,$D5,$8C,$BC,$87,$F6,$E8,$CC,$F9,$3E,$69,$C1,$F2,$DF,$8F,$5C
	.dc.b	$03,$A9,$E1,$C1,$B3,$E8,$75,$58,$F1,$3C,$FC,$0E,$9E,$35,$9F,$03
	.dc.b	$87,$C5,$F1,$D7,$1F,$06,$28,$F2,$7B,$DE,$5F,$32,$DF,$A5,$C3,$EB
	.dc.b	$F8,$EB,$87,$27,$D9,$EF,$FB,$7D,$FB,$5F,$05,$FC,$57,$C5,$BE,$2E
	.dc.b	$FC,$5E,$1B,$5F,$6A,$5C,$1C,$3D,$CE,$2D,$9F,$BF,$F3,$4F,$91,$FC
	.dc.b	$B2,$F9,$5B,$18,$73,$CF,$AF,$9E,$B7,$C9,$FC,$F7,$CD,$BC,$0B,$E0
	.dc.b	$61,$B5,$F8,$E9,$71,$70,$E2,$E2,$D9,$FC,$9F,$CD,$FF,$8B,$9B,$B7
	.dc.b	$EA,$E3,$1C,$9F,$81,$7D,$F3,$F9,$A3,$FC,$5E,$5D,$BF,$64,$E2,$7F
	.dc.b	$10,$FC,$13,$F4,$3F,$37,$FE,$3E,$6F,$6F,$EE,$63,$13,$F8,$F7,$F1
	.dc.b	$CD,$F0,$F5,$AE,$DB,$7D,$3A,$39,$69,$CD,$B1,$FB,$0F,$15,$EA,$7F
	.dc.b	$07,$5E,$FD,$8C,$39,$F4,$75,$E1,$DB,$E4,$FF,$0F,$F9,$BF,$DA,$6B
	.dc.b	$FA,$57,$F6,$8D,$F1,$B5,$E3,$6F,$E0,$B8,$7C,$15,$C7,$F8,$1E,$08
	.dc.b	$E9,$EE,$FF,$1B,$E4,$F8,$7E,$57,$FB,$1E,$93,$9C,$F5,$E4,$6F,$C7
	.dc.b	$B9,$7F,$0E,$C7,$F5,$0F,$F1,$0F,$9B,$7C,$4F,$F2,$75,$F2,$B6,$30
	.dc.b	$FF,$2B,$F9,$BF,$CB,$57,$F1,$85,$F0,$99,$F1,$74,$C9,$D7,$8B,$75
	.dc.b	$FE,$11,$7F,$C7,$E7,$27,$F2,$2F,$A9,$78,$7D,$EF,$F9,$D4,$62,$7C
	.dc.b	$1F,$01,$E3,$E5,$F2,$36,$3C,$91,$C3,$CD,$C6,$5F,$C5,$ED,$1F,$D1
	.dc.b	$7E,$73,$E5,$DF,$F9,$A5,$FC,$6F,$37,$FB,$CD,$CD,$FF,$A4,$9C,$CE
	.dc.b	$F8,$BC,$BF,$1F,$CD,$B1,$FC,$F2,$7F,$68,$DF,$27,$C1,$E2,$78,$59
	.dc.b	$1B,$F7,$2E,$3F,$CB,$6D,$1E,$73,$E1,$FC,$D5,$AF,$90,$FA,$FE,$4B
	.dc.b	$D9,$EA,$D9,$F1,$38,$7E,$77,$CD,$5C,$7E,$23,$F0,$FD,$5F,$89,$E4
	.dc.b	$5A,$F5,$38,$7F,$AB,$FD,$51,$C5,$FF,$43,$8C,$47,$C1,$E2,$F9,$DC
	.dc.b	$3F,$3D,$B1,$F1,$63,$E9,$69,$8B,$F8,$0E,$27,$FB,$2F,$E0,$DF,$B6
	.dc.b	$7E,$64,$7F,$7D,$8C,$4F,$F5,$3F,$9E,$FC,$DD,$7F,$6C,$7F,$36,$DE
	.dc.b	$EF,$F0,$24,$62,$78,$77,$7E,$3B,$87,$2D,$BE,$5E,$4F,$0F,$5F,$8B
	.dc.b	$63,$B4,$7F,$83,$FA,$BC,$7D,$3F,$17,$1F,$F6,$CE,$5F,$D2,$6F,$FB
	.dc.b	$FD,$69,$FF,$5C,$43,$E4,$FC,$BF,$27,$E6,$6C,$FA,$E3,$D3,$F0,$F8
	.dc.b	$C3,$E0,$38,$9E,$B9,$D9,$FE,$ED,$8F,$F1,$C9,$F8,$9E,$67,$C2,$F9
	.dc.b	$CF,$86,$D8,$F9,$11,$FE,$29,$FA,$67,$CA,$E7,$0F,$F0,$04,$62,$BE
	.dc.b	$77,$B9,$FE,$2F,$E1,$7B,$BF,$3A,$30,$C9,$FF,$0F,$F4,$3E,$67,$18
	.dc.b	$FF,$82,$E3,$13,$FE,$24,$F9,$FF,$E5,$7C,$9F,$2E,$91,$FC,$B2,$30
	.dc.b	$EA,$7F,$DC,$4E,$5B,$E9,$7F,$90,$FE,$89,$C7,$FE,$45,$AD,$60,$13
	.dc.b	$AB,$F1,$7C,$36,$FD,$CE,$0D,$8F,$34,$75,$F9,$2C,$5F,$F8,$96,$FF
	.dc.b	$DA,$B5,$EB,$F2,$FC,$1C,$DE,$16,$C7,$BE,$3F,$CB,$1F,$4F,$FE,$D6
	.dc.b	$8C,$1F,$E4,$67,$17,$FE,$31,$F7,$7F,$C6,$7B,$FC,$54,$7F,$98,$4C
	.dc.b	$D7,$B6,$73,$19,$3B,$FD,$1D,$9E,$CF,$1F,$33,$97,$99,$67,$FC,$AF
	.dc.b	$5F,$E5,$46,$C7,$A3,$3D,$C7,$9D,$F1,$7A,$DE,$3F,$5A,$CF,$27,$0F
	.dc.b	$F9,$8B,$EA,$1F,$57,$8D,$3F,$C9,$D1,$89,$F0,$F9,$9E,$D3,$DD,$F3
	.dc.b	$F6,$3C,$BE,$6F,$E1,$EF,$E0,$6A,$8C,$47,$DF,$8C,$4F,$FA,$52,$3E
	.dc.b	$4C,$E9,$7D,$57,$F5,$E9,$D3,$63,$FD,$36,$D7,$FD,$8A,$D7,$C3,$EB
	.dc.b	$73,$BF,$EB,$51,$FE,$A2,$6B,$FE,$8D,$8D,$76,$7F,$09,$BF,$F4,$83
	.dc.b	$67,$D6,$D7,$D4,$D7,$F0,$3E,$75,$B3,$49,$5F,$E2,$C9,$FD,$07,$BB
	.dc.b	$F9,$F7,$E4,$36,$3F,$DB,$ED,$8F,$C5,$35,$F9,$5E,$8F,$72,$4F,$58
	.dc.b	$FC,$9C,$79,$7E,$97,$FA,$2C,$7D,$56,$CF,$FB,$B1,$B1,$F9,$76,$BE
	.dc.b	$6E,$BF,$75,$FE,$B2,$3C,$F1,$83,$B7,$89,$2F,$C5,$7F,$CF,$DE,$D1
	.dc.b	$E3,$F8,$DF,$E2,$F7,$FE,$D4,$A4,$62,$3C,$9F,$1D,$FE,$A2,$75,$19
	.dc.b	$7C,$DF,$07,$FA,$DC,$3D,$4D,$8C,$9E,$27,$0F,$F0,$FB,$FF,$28,$B9
	.dc.b	$21,$F1,$63,$6A,$3F,$CA,$AF,$B3,$CC,$CC,$62,$3E,$64,$70,$CF,$FA
	.dc.b	$64,$7D,$36,$BF,$7B,$CE,$D1,$CB,$E3,$24,$7C,$86,$BF,$5F,$E8,$3B
	.dc.b	$91,$F9,$AF,$9B,$EA,$DE,$4D,$24,$FE,$0E,$32,$FD,$4F,$47,$D2,$47
	.dc.b	$07,$D8,$F2,$BF,$68,$FC,$16,$E9,$FC,$EA,$3F,$70,$D7,$ED,$71,$68
	.dc.b	$8E,$9E,$7D,$0F,$1A,$92,$7F,$7A,$F6,$F4,$64,$76,$6B,$F8,$77,$F9
	.dc.b	$67,$8D,$FD,$57,$0F,$F7,$6C,$FF,$BF,$1A,$FF,$B9,$D7,$A3,$C3,$93
	.dc.b	$F9,$A6,$BF,$EC,$1A,$E1,$39,$7F,$0C,$D5,$1B,$7A,$7D,$36,$6C,$D3
	.dc.b	$1F,$31,$27,$C8,$D7,$FD,$F5,$5F,$E7,$A7,$FB,$16,$3F,$DB,$FE,$99
	.dc.b	$F4,$7E,$16,$CE,$B3,$FE,$99,$18,$85,$71,$F9,$9E,$04,$8E,$07,$F6
	.dc.b	$F9,$47,$B7,$07,$C1,$F7,$F8,$A9,$7C,$05,$A3,$C9,$CB,$27,$53,$F8
	.dc.b	$E7,$C7,$97,$9A,$7B,$7C,$97,$AF,$E1,$E5,$70,$7D,$D8,$C8,$E6,$D7
	.dc.b	$E3,$FA,$1F,$0F,$E7,$64,$78,$1F,$8D,$F1,$72,$3C,$6D,$7E,$7F,$A1
	.dc.b	$F6,$FE,$99,$F2,$F4,$93,$EF,$8F,$EF,$6D,$F2,$78,$11,$F8,$9C,$3F
	.dc.b	$9B,$7E,$37,$D3,$7F,$8B,$5B,$ED,$FE,$26,$31,$1F,$10,$FF,$10,$2F
	.dc.b	$88,$D1,$23,$9B,$F9,$1F,$AB,$F3,$C7,$6F,$47,$A1,$1F,$0F,$A5,$DC
	.dc.b	$8E,$0F,$B5,$F6,$FE,$77,$C4,$30,$FF,$60,$BE,$AF,$4B,$D0,$7A,$7D
	.dc.b	$29,$1E,$E6,$BC,$A7,$FA,$FB,$F0,$BD,$2F,$80,$47,$F1,$5F,$79,$FC
	.dc.b	$ED,$F0,$FD,$4F,$F9,$37,$BF,$FC,$9C,$DE,$CF,$F1,$43,$67,$FC,$8C
	.dc.b	$FF,$BA,$1F,$CA,$35,$E7,$F1,$1F,$B7,$8C,$57,$F9,$67,$EB,$7D,$B5
	.dc.b	$E0,$F4,$B4,$47,$9F,$A5,$C4,$8E,$0F,$D1,$71,$7D,$AB,$F3,$F9,$FD
	.dc.b	$C9,$3E,$4F,$47,$C4,$47,$07,$A3,$D0,$F3,$11,$C1,$FA,$FF,$25,$DD
	.dc.b	$EB,$F9,$59,$3F,$01,$F7,$79,$B2,$3E,$1B,$5F,$BB,$9B,$A7,$E2,$7C
	.dc.b	$D4,$9F,$92,$FC,$7F,$86,$91,$F3,$1A,$FE,$1B,$EB,$1B,$7F,$23,$F6
	.dc.b	$3F,$D2,$B3,$FC,$15,$AF,$E3,$1B,$C1,$FE,$89,$7F,$8B,$37,$CE,$E4
	.dc.b	$49,$E5,$E7,$75,$23,$83,$F8,$97,$C4,$7F,$0D,$AE,$9F,$43,$E3,$24
	.dc.b	$F8,$FE,$87,$CC,$47,$07,$2F,$2F,$D5,$E6,$C8,$F5,$3E,$DF,$36,$47
	.dc.b	$B5,$AF,$FA,$9F,$87,$E1,$79,$92,$3E,$13,$F1,$7D,$59,$1F,$15,$AF
	.dc.b	$C5,$F6,$BE,$51,$1F,$94,$FF,$33,$3F,$A8,$7D,$5F,$F8,$37,$5F,$F5
	.dc.b	$93,$7C,$2F,$F4,$63,$6B,$FD,$54,$BE,$3B,$8F,$AD,$27,$F9,$D4,$A3
	.dc.b	$AF,$47,$F5,$DF,$AB,$7C,$5D,$25,$FF,$41,$87,$C5,$EF,$86,$4E,$5E
	.dc.b	$5C,$3F,$D9,$FE,$AF,$F9,$4C,$FF,$0E,$6E,$DF,$EC,$B6,$ED,$FE,$A5
	.dc.b	$5F,$15,$C7,$E1,$65,$FF,$51,$9F,$F4,$53,$7E,$7B,$FB,$A7,$BA,$F8
	.dc.b	$5A,$49,$FE,$B1,$0E,$ED,$F8,$92,$72,$F2,$F1,$3F,$BF,$7D,$5F,$F5
	.dc.b	$D9,$FE,$C0,$D7,$F9,$3B,$76,$FF,$72,$3F,$93,$DB,$FD,$7C,$BB,$79
	.dc.b	$92,$B9,$F2,$76,$F1,$A4,$76,$7F,$C1,$DF,$57,$FE,$C7,$8F,$E6,$AB
	.dc.b	$E3,$32,$48,$FE,$E9,$0F,$8C,$FF,$0B,$7D,$5B,$E1,$E9,$2B,$FB,$D4
	.dc.b	$37,$FF,$0E,$7A,$1F,$E0,$29,$FE,$FC,$D7,$FA,$2B,$77,$6F,$FE,$DB
	.dc.b	$8F,$84,$DF,$0B,$99,$2B,$9F,$85,$DB,$E0,$64,$76,$7C,$3B,$FF,$8B
	.dc.b	$7E,$AD,$F3,$54,$95,$FE,$11,$8F,$84,$FF,$84,$FE,$AF,$FA,$22,$69
	.dc.b	$3D,$7F,$F1,$C3,$FE,$55,$D7,$D3,$23,$87,$C5,$E8,$DF,$0F,$0E,$09
	.dc.b	$3D,$DE,$7D,$12,$3C,$4D,$7F,$B5,$E1,$FE,$4E,$7C,$DE,$2F,$88,$8F
	.dc.b	$A7,$C5,$F2,$91,$C1,$ED,$F4,$7F,$CA,$1F,$57,$F8,$D8,$F4,$35,$34
	.dc.b	$79,$BC,$09,$1F,$E3,$98,$CB,$FE,$18,$E2,$FF,$2E,$7C,$CF,$F8,$32
	.dc.b	$3F,$C0,$6D,$DF,$E3,$C2,$31,$CD,$F2,$63,$1F,$0B,$C8,$8F,$F9,$1E
	.dc.b	$7F,$7A,$FC,$FF,$85,$FE,$B6,$FF,$92,$FE,$AF,$FA,$1D,$F3,$6C,$7D
	.dc.b	$56,$A7,$F2,$51,$BF,$4F,$FC,$E9,$18,$9E,$11,$8F,$12,$47,$76,$BF
	.dc.b	$E2,$0F,$DA,$7F,$5D,$9F,$BA,$FE,$61,$3E,$2F,$AE,$F2,$FD,$0D,$7F
	.dc.b	$C5,$33,$FB,$FA,$FC,$67,$D8,$7C,$79,$F5,$78,$BF,$E7,$47,$F7,$87
	.dc.b	$AD,$C3,$FB,$FA,$E4,$E9,$AE,$7C,$57,$D6,$F2,$8F,$89,$18,$78,$35
	.dc.b	$E1,$1F,$EB,$A5,$F5,$14,$8F,$FA,$F9,$FF,$5F,$FD,$1F,$E9,$B2,$FC
	.dc.b	$0C,$FF,$99,$B8,$3F,$D0,$4A,$75,$1F,$C5,$FE,$3A,$CF,$E1,$5A,$FF
	.dc.b	$A3,$3C,$BA,$F3,$3F,$CA,$3E,$8F,$F3,$5A,$EB,$CD,$AF,$47,$6D,$8F
	.dc.b	$0C,$72,$F9,$9C,$5F,$F9,$68,$39,$9F,$F4,$C5,$BE,$5B,$15,$E0,$7C
	.dc.b	$BA,$7F,$A6,$EB,$FD,$3C,$FF,$BA,$9B,$CD,$AE,$59,$1D,$3C,$0B,$87
	.dc.b	$A9,$F7,$FD,$6F,$C5,$9F,$F5,$43,$EF,$C3,$F5,$6E,$6B,$FC,$E2,$E6
	.dc.b	$7F,$58,$D8,$FE,$2D,$93,$BF,$AF,$07,$CC,$8C,$5F,$6F,$1F,$C3,$E7
	.dc.b	$F8,$9B,$1E,$28,$F2,$F9,$D8,$C7,$C2,$FA,$3E,$3B,$D3,$EA,$DB,$F4
	.dc.b	$B8,$7D,$7E,$A5,$BF,$4B,$F0,$3E,$37,$E0,$FC,$6D,$AF,$82,$FE,$0B
	.dc.b	$E7,$BF,$C2,$91,$FE,$CB,$5F,$31,$9F,$7B,$4E,$F8,$78,$79,$56,$F8
	.dc.b	$3F,$8C,$FA,$DF,$F0,$9C,$7F,$B5,$57,$D4,$67,$DE,$D3,$BE,$4E,$1F
	.dc.b	$2D,$6F,$83,$F9,$6F,$AD,$FF,$08,$C7,$FB,$7D,$FC,$5B,$7B,$F5,$DF
	.dc.b	$BE,$2E,$1C,$56,$B8,$3F,$A0,$FA,$DF,$F1,$74,$7F,$BA,$5F,$AF,$C9
	.dc.b	$FE,$B0,$72,$DE,$0F,$D3,$CF,$F8,$C8,$FF,$78,$1E,$E8,$FF,$5B,$B9
	.dc.b	$6E,$DF,$AE,$FA,$DF,$EB,$93,$FE,$F7,$7F,$1D,$DB,$FC,$E9,$CD,$7F
	.dc.b	$B0,$1C,$CF,$F3,$B5,$E1,$73,$D7,$A3,$67,$A7,$47,$7D,$7C,$0D,$8F
	.dc.b	$ED,$98,$FF,$48,$F1,$5E,$E7,$F0,$25,$F2,$16,$30,$E7,$D6,$D7,$87
	.dc.b	$70,$E6,$7F,$4F,$B4,$7F,$0C,$FA,$DF,$DC,$6F,$FE,$DA,$7F,$70,$DF
	.dc.b	$1B,$5E,$36,$FD,$DE,$1F,$0F,$72,$EB,$7F,$E0,$3C,$13,$D7,$DE,$FE
	.dc.b	$33,$E6,$7C,$8E,$0F,$EC,$75,$FE,$B4,$C3,$8E,$1F,$C8,$CD,$F8,$FC
	.dc.b	$3C,$7B,$3A,$C3,$FC,$6C,$F0,$FA,$D7,$BD,$FC,$99,$FF,$2D,$46,$63
	.dc.b	$1F,$09,$FC,$AB,$EB,$7F,$3B,$3E,$05,$F0,$F9,$F1,$74,$C9,$D7,$8B
	.dc.b	$75,$FF,$7F,$DA,$3A,$1F,$CD,$BE,$B5,$E1,$E2,$FD,$B1,$87,$C1,$F0
	.dc.b	$9E,$3E,$5F,$23,$63,$C9,$1C,$FC,$DC,$64,$E4,$FF,$05,$E0,$9F,$E8
	.dc.b	$BF,$5B,$F3,$70,$FF,$06,$3F,$94,$BF,$F1,$B3,$96,$F1,$70,$3D,$1F
	.dc.b	$33,$6B,$CE,$F4,$F0,$FA,$F9,$16,$7E,$BE,$1F,$BC,$C3,$C9,$1E,$09
	.dc.b	$F1,$FB,$FC,$23,$6E,$1F,$87,$F5,$96,$B8,$9F,$E4,$BD,$11,$EC,$F9
	.dc.b	$DF,$6F,$A9,$6F,$C2,$E1,$F9,$BF,$49,$70,$E6,$30,$47,$C6,$F5,$7E
	.dc.b	$3F,$91,$6F,$D6,$E1,$FC,$9F,$D6,$FF,$9D,$9C,$98,$3E,$7F,$17,$D0
	.dc.b	$E1,$B1,$F1,$1C,$3F,$1B,$5B,$9F,$93,$83,$DB,$39,$3F,$43,$F5,$BF
	.dc.b	$BE,$71,$3E,$A8,$C3,$EC,$FA,$DF,$CF,$CF,$EA,$1F,$CF,$C6,$7B,$BF
	.dc.b	$C0,$F1,$8E,$6F,$F2,$8B,$6B,$FC,$A1,$C1,$FE,$53,$79,$77,$7E,$67
	.dc.b	$93,$8F,$95,$CB,$D7,$B5,$DD,$C3,$FB,$8E,$AF,$2E,$5D,$A7,$FC,$2F
	.dc.b	$18,$9F,$F2,$F6,$4F,$9C,$F2,$FC,$E7,$99,$B3,$EB,$8F,$4F,$C6,$E3
	.dc.b	$7F,$80,$E2,$7C,$19,$D9,$FB,$F8,$FF,$24,$9F,$89,$E6,$7C,$2F,$A6
	.dc.b	$F8,$6D,$8F,$91,$1F,$E2,$CF,$98,$7D,$6E,$70,$FC,$4C,$62,$BE,$77
	.dc.b	$B9,$FE,$32,$E2,$7B,$DF,$3A,$30,$C9,$FE,$37,$D4,$FA,$4C,$63,$F9
	.dc.b	$48,$C4,$FF,$20,$FA,$DF,$E6,$3C,$BF,$E8,$67,$F9,$8B,$7D,$3F,$F2
	.dc.b	$0C,$62,$7F,$D1,$0D,$BF,$F5,$97,$D6,$B8,$F7,$EA,$7F,$92,$23,$17
	.dc.b	$DF,$E1,$F9,$FB,$9D,$1B,$1E,$38,$E1,$F2,$78,$C7,$F5,$77,$FE,$E3
	.dc.b	$AF,$47,$99,$E4,$E7,$F2,$B6,$3D,$91,$FE,$58,$FA,$D7,$C7,$E7,$0F
	.dc.b	$DA,$C6,$2B,$FA,$0F,$BD,$FE,$37,$9F,$ED,$2F,$F9,$80,$C1,$FD,$B2
	.dc.b	$36,$F5,$F9,$DD,$36,$39,$F9,$FC,$7C,$FE,$4D,$8E,$F1,$E5,$CF,$18
	.dc.b	$79,$1F,$07,$B1,$E1,$F6,$2D,$F9,$9C,$3F,$D6,$3E,$B1,$F5,$B8,$C3
	.dc.b	$F8,$8C,$62,$7D,$BE,$87,$FA,$0A,$7F,$DD,$1A,$23,$11,$FE,$08,$6C
	.dc.b	$66,$7F,$D2,$DF,$5B,$FD,$59,$BE,$CF,$08,$C5,$FF,$A9,$1A,$FF,$B3
	.dc.b	$4F,$EE,$7F,$13,$DB,$DF,$FF,$23,$DF,$FA,$85,$B3,$FE,$AC,$6B,$C7
	.dc.b	$0F,$E9,$D3,$FE,$20,$8F,$4F,$72,$31,$1F,$E2,$08,$C3,$FE,$A0,$9E
	.dc.b	$2D,$F4,$B3,$5D,$5B,$1F,$EE,$39,$46,$23,$E3,$B5,$FF,$06,$F6,$5C
	.dc.b	$3F,$3B,$1F,$EA,$E6,$DF,$C1,$6C,$7F,$BB,$D7,$8D,$C4,$93,$EA,$8C
	.dc.b	$BE,$CF,$AD,$FE,$85,$5F,$EB,$A6,$C7,$CD,$6C,$7F,$BE,$DA,$8F,$F1
	.dc.b	$1F,$5F,$9C,$93,$DD,$F5,$7D,$34,$8F,$13,$67,$C1,$F1,$DF,$C4,$1F
	.dc.b	$F7,$0D,$FF,$AC,$FC,$5F,$F6,$D2,$F8,$FF,$F1,$54,$F9,$6B,$83,$E7
	.dc.b	$F8,$6F,$23,$EB,$6E,$39,$8F,$81,$18,$7D,$9E,$6D,$7D,$98,$7F,$94
	.dc.b	$9F,$8D,$19,$51,$97,$E4,$46,$63,$15,$F2,$63,$07,$1D,$3F,$A3,$57
	.dc.b	$D3,$F3,$F0,$47,$F1,$FC,$8E,$BF,$03,$63,$31,$83,$C1,$F4,$92,$3E
	.dc.b	$E3,$5F,$F7,$AF,$CB,$BC,$7F,$53,$63,$F3,$AD,$77,$1E,$FF,$97,$F9
	.dc.b	$FA,$7E,$BD,$B3,$F3,$9A,$FE,$0D,$AF,$C6,$E5,$FD,$B3,$F8,$96,$BF
	.dc.b	$39,$AE,$CF,$BF,$C9,$8D,$CD,$24,$7F,$08,$F1,$78,$EA,$69,$23,$F8
	.dc.b	$17,$C4,$F6,$E0,$34,$91,$F8,$EF,$4B,$F1,$3E,$6F,$EF,$A3,$EF,$C6
	.dc.b	$35,$FD,$13,$63,$F0,$FD,$35,$FC,$2B,$E9,$23,$F9,$A7,$DF,$C1,$FE
	.dc.b	$D2,$8F,$86,$D7,$E3,$B5,$FE,$3C,$D7,$F6,$5D,$3D,$F6,$34,$C7,$CD
	.dc.b	$49,$FF,$48,$CA,$31,$1F,$E4,$AA,$FF,$42,$BF,$E9,$6F,$AD,$7D,$2F
	.dc.b	$AD,$B1,$98,$C7,$97,$23,$83,$5E,$9F,$0D,$C7,$D1,$91,$D5,$E9,$CF
	.dc.b	$23,$C0,$D7,$C9,$F2,$5E,$1F,$16,$47,$95,$F2,$5C,$8F,$43,$5F,$67
	.dc.b	$9B,$E9,$F9,$B9,$1E,$D7,$D9,$F1,$B2,$3E,$03,$5F,$89,$DD,$F6,$FE
	.dc.b	$B5,$F2,$D4,$93,$FE,$BC,$8F,$F7,$1B,$7C,$8E,$04,$7F,$D8,$3C,$5F
	.dc.b	$95,$7F,$D8,$7F,$59,$FE,$29,$6F,$B1,$F8,$98,$C4,$7F,$9F,$97,$C5
	.dc.b	$64,$91,$C9,$FC,$87,$D4,$F9,$A3,$AF,$D4,$73,$23,$E0,$F4,$3B,$11
	.dc.b	$C1,$FE,$D6,$FA,$DF,$33,$E1,$F8,$FF,$21,$1F,$4F,$A3,$E7,$23,$83
	.dc.b	$DB,$E8,$FA,$C8,$E0,$FF,$71,$FD,$6F,$F3,$AF,$13,$F5,$DF,$5B,$FA
	.dc.b	$F6,$F8,$BF,$AB,$8C,$3F,$CF,$1F,$F7,$1D,$7F,$BB,$1A,$FF,$A9,$1B
	.dc.b	$3D,$3E,$27,$F6,$9F,$5B,$FE,$D8,$AE,$1E,$97,$52,$3C,$FD,$2F,$01
	.dc.b	$1C,$1F,$EF,$AE,$1F,$AB,$87,$A3,$D1,$F3,$12,$7C,$9E,$97,$A4,$8E
	.dc.b	$0F,$47,$C3,$7B,$48,$E0,$FE,$D9,$5F,$C4,$7C,$A7,$77,$C3,$F3,$65
	.dc.b	$7C,$17,$E3,$79,$B2,$3E,$23,$5F,$F6,$3E,$6E,$8F,$99,$F3,$52,$7E
	.dc.b	$53,$F3,$FE,$12,$47,$CD,$6B,$FE,$D0,$FA,$D6,$FF,$C7,$FD,$8F,$F4
	.dc.b	$6C,$FA,$39,$BF,$B8,$C6,$23,$FD,$AE,$DD,$FF,$96,$3F,$C3,$23,$3E
	.dc.b	$6F,$32,$3C,$CF,$F0,$74,$7F,$9B,$9C,$1F,$E0,$3F,$37,$C4,$47,$3F
	.dc.b	$0F,$FD,$6E,$FB,$FD,$27,$9C,$93,$CA,$DF,$47,$EB,$24,$65,$FE,$79
	.dc.b	$6F,$2F,$E0,$79,$B2,$B9,$3F,$0F,$CD,$91,$D5,$AF,$FA,$97,$8B,$E4
	.dc.b	$79,$92,$3C,$4F,$CB,$F5,$A4,$79,$5A,$FC,$AF,$5B,$E8,$11,$F5,$3F
	.dc.b	$D4,$7C,$B3,$FD,$4F,$EB,$7F,$C2,$DC,$1F,$D6,$DB,$C7,$FE,$26,$6D
	.dc.b	$FF,$8E,$D7,$C6,$F1,$F5,$A4,$FD,$09,$FE,$94,$DE,$9F,$EC,$1F,$5A
	.dc.b	$F8,$AA,$49,$FA,$F0,$F8,$AD,$F1,$49,$CB,$CB,$8B,$FB,$57,$D6,$FF
	.dc.b	$2C,$96,$DF,$DC,$E3,$0D,$F4,$F7,$F1,$3C,$3E,$5A,$5F,$E1,$AD,$1D
	.dc.b	$3E,$7F,$FB,$B7,$B8,$F8,$FA,$4B,$FC,$6C,$7F,$93,$DB,$E3,$70,$23
	.dc.b	$AF,$1B,$FC,$01,$F5,$AA,$A4,$9F,$CB,$C7,$E4,$F9,$FF,$CA,$AF,$E4
	.dc.b	$23,$3D,$FC,$8E,$7C,$9D,$3C,$89,$3C,$9F,$F0,$77,$D6,$FF,$A8,$78
	.dc.b	$BF,$C2,$CB,$E3,$72,$4A,$FD,$3C,$3E,$37,$FC,$2D,$F5,$BF,$E2,$C9
	.dc.b	$FD,$5B,$5F,$9F,$A7,$F9,$AA,$31,$87,$F8,$8D,$FF,$14,$CF,$EC,$BD
	.dc.b	$1F,$DA,$3C,$0F,$F4,$23,$89,$FF,$15,$36,$BF,$CE,$AF,$C7,$9E,$5E
	.dc.b	$17,$F8,$A6,$3F,$CB,$D1,$CC,$38,$AE,$7E,$16,$FE,$02,$57,$8D,$E5
	.dc.b	$AF,$F8,$CF,$EB,$5F,$3B,$49,$5F,$C2,$A3,$E4,$3F,$C3,$3E,$B7,$FA
	.dc.b	$7B,$FE,$9D,$6A,$69,$FD,$3B,$4F,$F4,$7A,$F8,$BF,$F2,$13,$67,$A4
	.dc.b	$63,$E2,$F3,$D1,$27,$C0,$F0,$E8,$91,$E3,$6B,$F0,$EF,$FD,$7A,$F9
	.dc.b	$FC,$5E,$E4,$7D,$5E,$27,$88,$8E,$0F,$77,$A5,$FE,$57,$FA,$DF,$CA
	.dc.b	$47,$FA,$ED,$A9,$A3,$CF,$E0,$48,$FE,$67,$19,$7F,$C3,$FC,$3F,$E6
	.dc.b	$3F,$99,$FF,$0A,$47,$DB,$6F,$27,$F9,$B5,$BC,$1F,$EB,$87,$F7,$AD
	.dc.b	$F1,$39,$12,$BF,$C5,$7C,$9F,$E6,$FF,$AC,$FE,$95,$87,$F9,$DD,$B7
	.dc.b	$FE,$C2,$6D,$7F,$B5,$DF,$C9,$37,$D2,$E4,$47,$FA,$6C,$7E,$55,$FC
	.dc.b	$93,$6B,$B7,$83,$E1,$3F,$D0,$1F,$1F,$FB,$3C,$3C,$1F,$17,$E5,$26
	.dc.b	$1D,$37,$F4,$11,$C1,$E0,$F0,$BF,$8F,$AF,$AD,$6F,$EB,$9C,$CF,$FB
	.dc.b	$78,$69,$FD,$6B,$4F,$F5,$6B,$63,$FD,$62,$D7,$83,$FC,$DF,$EB,$77
	.dc.b	$E2,$1A,$49,$FF,$5A,$FD,$47,$FA,$D9,$34,$8F,$FA,$F3,$E9,$1B,$7F
	.dc.b	$76,$8F,$F3,$4B,$63,$31,$FB,$D3,$25,$F8,$EC,$E5,$FE,$B1,$F3,$BF
	.dc.b	$E0,$05,$F3,$9C,$BF,$EF,$E8,$CD,$73,$E5,$CA,$27,$9D,$87,$11,$54
	.dc.b	$53,$1F,$33,$6F,$D4,$E1,$FF,$08,$DB,$E5,$71,$7F,$C5,$E3,$13,$FE
	.dc.b	$16,$CB,$FA,$23,$79,$3C,$51,$8A,$FE,$3C,$FF,$BD,$67,$47,$6E,$FD
	.dc.b	$D7,$A6,$D1,$EB,$FA,$D6,$D7,$6C,$37,$0F,$F1,$3C,$FC,$EC,$7E,$67
	.dc.b	$76,$73,$F4,$3C,$FC,$5E,$9E,$65,$C3,$8B,$87,$9F,$97,$6F,$F9,$B6
	.dc.b	$3F,$D3,$67,$BF,$C6,$F9,$7D,$1F,$33,$63,$C7,$1E,$9F,$47,$17,$C9
	.dc.b	$F6,$7C,$67,$B7,$E3,$2D,$FA,$9F,$B5,$F2,$FF,$E1,$68,$FF,$21,$AF
	.dc.b	$98,$CF,$B7,$A7,$7C,$1C,$3C,$9B,$7C,$1F,$C2,$7D,$6F,$F8,$56,$3F
	.dc.b	$C9,$AB,$EA,$33,$ED,$E9,$DF,$4F,$0F,$96,$B7,$C1,$FC,$77,$D6,$FE
	.dc.b	$32,$3F,$CA,$EB,$BB,$3D,$FA,$77,$C5,$C3,$8A,$DF,$07,$F3,$1F,$5B
	.dc.b	$FE,$F9,$8F,$F3,$0A,$DF,$3E,$86,$9F,$CF,$4F,$F8,$CC,$FF,$33,$9E
	.dc.b	$DE,$5F,$ED,$8E,$5B,$BF,$E9,$7E,$B7,$F5,$13,$FE,$6E,$7F,$1A,$D9
	.dc.b	$D7,$76,$FC,$3E,$1B,$5A,$E0,$F3,$E9,$E9,$B5,$9E,$8F,$EC,$FC,$4F
	.dc.b	$EF,$B1,$FE,$7C,$5F,$1F,$9F,$67,$4E,$FD,$5E,$1E,$2D,$BE,$0F,$EF
	.dc.b	$7E,$B7,$F5,$9E,$2F,$F0,$ED,$D1,$FA,$D7,$FC,$18,$EA,$BF,$83,$46
	.dc.b	$BC,$7E,$5E,$D7,$3F,$84,$B5,$FE,$90,$BF,$F3,$6B,$7D,$5F,$E1,$9C
	.dc.b	$7F,$23,$1E,$B1,$87,$3E,$46,$BC,$8B,$7F,$E9,$AB,$FD,$45,$7F,$87
	.dc.b	$FE,$B5,$F2,$7F,$C5,$DF,$F2,$B5,$7F,$A8,$1C,$B7,$E0,$BF,$8E,$7D
	.dc.b	$6F,$E5,$E3,$FD,$4A,$BE,$17,$3E,$2E,$99,$3A,$F1,$6E,$5F,$E2,$FB
	.dc.b	$47,$43,$F9,$47,$D4,$3C,$3C,$5F,$FA,$CA,$30,$F8,$3E,$0B,$C7,$C9
	.dc.b	$E4,$6C,$79,$23,$9F,$9B,$8C,$BF,$C7,$D6,$8F,$E6,$DF,$55,$F2,$FA
	.dc.b	$3E,$D3,$F9,$2A,$FF,$22,$39,$C3,$EC,$B9,$9D,$F1,$7A,$3E,$67,$9B
	.dc.b	$6B,$FD,$93,$5F,$B5,$BE,$7E,$47,$83,$C2,$F0,$B2,$B7,$EC,$DA,$F0
	.dc.b	$BE,$63,$E2,$FD,$5D,$9F,$88,$FA,$BE,$73,$D7,$EA,$D9,$F0,$B8,$7E
	.dc.b	$57,$D6,$DC,$3E,$1B,$F0,$BD,$5F,$87,$E4,$5A,$F4,$B8,$7F,$DC,$7F
	.dc.b	$5B,$FD,$C1,$C9,$83,$E5,$F1,$7C,$CE,$1B,$1F,$09,$C3,$F1,$34,$B9
	.dc.b	$7C,$E7,$E9,$9F,$DA,$7E,$B7,$F6,$F1,$99,$FF,$79,$C6,$38,$FF,$61
	.dc.b	$F5,$BF,$96,$AF,$F3,$D3,$F9,$A6,$F7,$FF,$73,$8C,$4F,$0E,$EF,$C7
	.dc.b	$F1,$65,$B8,$FF,$C0,$0A,$FE,$0A,$DF,$5F,$97,$93,$CD,$91,$FE,$01
	.dc.b	$C3,$F8,$17,$57,$96,$3F,$7F,$7F,$D2,$73,$FF,$47,$4E,$5F,$F0,$44
	.dc.b	$3E,$73,$CD,$F3,$9E,$76,$DF,$8E,$3D,$5F,$17,$8B,$F7,$B8,$9F,$11
	.dc.b	$FC,$3B,$0F,$4C,$EA,$32,$FF,$85,$E3,$38,$FC,$AF,$2F,$E3,$7D,$37
	.dc.b	$C7,$6D,$FC,$38,$FF,$0E,$7C,$BB,$EB,$73,$8F,$FA,$A6,$31,$5F,$27
	.dc.b	$DC,$FF,$11,$70,$BD,$EF,$A5,$18,$64,$F9,$2F,$F6,$9F,$49,$87,$FD
	.dc.b	$69,$D2,$8F,$F1,$67,$D6,$FF,$2B,$9F,$F1,$83,$FC,$BF,$6F,$E6,$4E
	.dc.b	$60,$E6,$FF,$98,$B6,$3F,$C7,$1F,$5B,$FE,$AB,$9F,$14,$E6,$53,$97
	.dc.b	$FD,$8D,$1F,$D6,$EB,$BF,$B5,$E4,$F8,$6F,$2B,$30,$F6,$60,$F9,$1C
	.dc.b	$63,$FE,$CF,$8C,$4F,$83,$CA,$F6,$F2,$7B,$9B,$1E,$48,$FF,$26,$7D
	.dc.b	$6B,$E3,$F3,$87,$FB,$66,$31,$5F,$2F,$DD,$FF,$19,$3F,$D4,$11,$FE
	.dc.b	$55,$33,$3F,$E5,$67,$07,$6F,$9C,$B3,$F5,$69,$1C,$7C,$BE,$5E,$5D
	.dc.b	$AF,$03,$87,$86,$77,$1F,$33,$E1,$F5,$BC,$5E,$B5,$AF,$43,$87,$F0
	.dc.b	$9F,$58,$FA,$DC,$61,$FE,$F7,$8C,$4E,$BD,$3F,$EE,$53,$FD,$BA,$7F
	.dc.b	$A3,$BF,$E9,$90,$E1,$FF,$4E,$35,$E1,$E5,$72,$7F,$CF,$90,$8C,$47
	.dc.b	$26,$C6,$6B,$B4,$BD,$BF,$E3,$2F,$FA,$A5,$AF,$F8,$09,$AF,$FA,$25
	.dc.b	$AF,$1F,$0F,$FD,$33,$3C,$5B,$1C,$5A,$F1,$6B,$F7,$BC,$BB,$C4,$69
	.dc.b	$2B,$FC,$15,$3F,$E9,$6F,$3F,$64,$8C,$C7,$E2,$23,$AF,$D6,$7F,$41
	.dc.b	$8F,$9E,$D9,$FF,$63,$36,$3F,$C3,$ED,$7F,$D4,$CF,$F3,$78,$DF,$99
	.dc.b	$D0,$8F,$82,$3F,$9D,$3D,$3C,$2F,$09,$1D,$7C,$67,$F0,$DB,$FB,$D5
	.dc.b	$FE,$A7,$F0,$BE,$9A,$FA,$0F,$F4,$CC,$EA,$B8,$3F,$57,$C4,$FF,$1A
	.dc.b	$E1,$63,$E0,$EE,$9C,$C7,$D9,$6B,$FE,$C1,$F9,$FE,$75,$FE,$E7,$6C
	.dc.b	$7F,$90,$1A,$EF,$2E,$2B,$27,$9B,$86,$C7,$C3,$8C,$11,$C1,$0F,$A6
	.dc.b	$F1,$F8,$C9,$1F,$2D,$AF,$D8,$FA,$2E,$C4,$7E,$83,$E1,$F3,$A4,$7D
	.dc.b	$36,$BF,$80,$E4,$F3,$11,$FB,$0F,$A7,$CE,$FD,$1C,$7B,$5B,$89,$A6
	.dc.b	$BE,$55,$F2,$14,$95,$FA,$C8,$CB,$FA,$ED,$5F,$63,$D9,$B1,$98,$C1
	.dc.b	$9F,$3E,$47,$16,$BF,$C1,$67,$F9,$63,$D7,$C5,$E8,$47,$C5,$D5,$E7
	.dc.b	$23,$83,$CD,$E1,$FF,$9F,$BC,$BF,$F4,$26,$5F,$CC,$27,$FA,$F3,$54
	.dc.b	$6A,$78,$72,$7F,$A1,$A9,$18,$A0,$8C,$BD,$23,$4E,$7C,$4B,$BB,$E9
	.dc.b	$A9,$27,$FB,$84,$3E,$37,$64,$8C,$BD,$35,$FF,$3F,$47,$F7,$7F,$AD
	.dc.b	$7D,$2F,$81,$B1,$98,$C6,$F2,$38,$35,$F8,$3A,$FE,$97,$E0,$B9,$7C
	.dc.b	$D4,$9E,$CF,$5F,$8A,$91,$E1,$6B,$C8,$FD,$97,$C9,$F8,$FE,$1E,$47
	.dc.b	$9D,$F3,$5C,$8F,$53,$5E,$43,$4F,$EB,$EF,$BF,$E6,$FD,$C4,$7E,$17
	.dc.b	$C5,$FC,$12,$38,$3F,$C3,$92,$F3,$FE,$AF,$D6,$FE,$6A,$7B,$B7,$9F
	.dc.b	$FD,$94,$DD,$BF,$D8,$EF,$E1,$E3,$3D,$1C,$89,$5D,$BD,$3F,$C0,$FD
	.dc.b	$6F,$F8,$9E,$3F,$C5,$8D,$7D,$AD,$AF,$F6,$C2,$F8,$7E,$24,$9E,$2F
	.dc.b	$E3,$3E,$A7,$C9,$1F,$92,$97,$D4,$78,$09,$1E,$0F,$88,$F1,$91,$C1
	.dc.b	$FE,$3F,$FA,$D6,$5E,$8F,$52,$4F,$A1,$F5,$FC,$04,$8F,$5B,$5F,$5F
	.dc.b	$AB,$F0,$08,$FC,$07,$F4,$7F,$5B,$FD,$02,$3E,$EF,$D6,$FF,$BD,$F7
	.dc.b	$FF,$7C,$37,$8B,$FD,$EA,$D9,$FD,$7B,$FE,$E6,$7F,$CA,$CD,$79,$FA
	.dc.b	$9F,$B0,$8C,$52,$BF,$DF,$FF,$5B,$FE,$DB,$BE,$3F,$2D,$B2,$3D,$3E
	.dc.b	$27,$91,$1C,$1F,$E6,$5E,$3F,$B1,$7E,$6F,$3F,$C0,$49,$F2,$F9,$FE
	.dc.b	$32,$38,$3D,$3E,$7F,$9C,$8E,$0F,$F3,$97,$90,$ED,$F6,$7C,$B4,$9F
	.dc.b	$82,$FB,$FE,$26,$47,$C4,$6B,$FA,$AC,$FF,$0A,$FC,$8F,$9A,$F8,$C4
	.dc.b	$7E,$5F,$AD,$F2,$48,$E0,$FF,$42,$7D,$6B,$5F,$E3,$BE,$D7,$FA,$46
	.dc.b	$7F,$D1,$0D,$7F,$72,$DE,$3F,$A5,$1F,$CA,$23,$F8,$AB,$AF,$2F,$99
	.dc.b	$1E,$7E,$5F,$62,$38,$3F,$D2,$DF,$14,$F2,$FC,$7F,$49,$27,$C6,$F9
	.dc.b	$BE,$8E,$47,$99,$AF,$FA,$9F,$E1,$7D,$7E,$6C,$8F,$5B,$EE,$F3,$64
	.dc.b	$7B,$9A,$FF,$A9,$3C,$6F,$87,$E5,$C8,$F8,$6F,$C6,$F1,$64,$7C,$66
	.dc.b	$BF,$1B,$DD,$F9,$64,$7E,$5B,$FC,$FC,$FE,$97,$F5,$AE,$FA,$49,$FF
	.dc.b	$5C,$47,$F8,$59,$FE,$52,$FF,$4B,$73,$E9,$F2,$39,$F4,$7F,$56,$FA
	.dc.b	$D7,$C6,$52,$57,$FB,$0E,$1F,$19,$BD,$E4,$E5,$E5,$BF,$F6,$0F,$AD
	.dc.b	$FE,$4D,$E2,$7F,$66,$A4,$61,$FE,$D2,$D7,$3E,$D7,$1F,$95,$91,$FE
	.dc.b	$D3,$94,$75,$F9,$FF,$ED,$BE,$CB,$E1,$A9,$2F,$FD,$B5,$1F,$E4,$F6
	.dc.b	$F8,$BC,$08,$EB,$C5,$FE,$EF,$F5,$AA,$A4,$9F,$F7,$24,$7E,$3D,$BD
	.dc.b	$7C,$08,$EB,$8B,$9F,$8F,$23,$8B,$FE,$04,$FA,$DF,$89,$1F,$D7,$97
	.dc.b	$C7,$64,$91,$FE,$F3,$87,$C7,$7F,$83,$3E,$B7,$FD,$B5,$3F,$EF,$76
	.dc.b	$BF,$DA,$1B,$E3,$FE,$C2,$10,$DB,$FC,$2B,$E7,$FF,$8B,$A7,$FB,$7B
	.dc.b	$6B,$83,$79,$7E,$D3,$F1,$67,$FC,$43,$3A,$F0,$39,$92,$33,$5C,$3C
	.dc.b	$0E,$FE,$F4,$9C,$9F,$E2,$AC,$11,$E4,$DB,$FC,$53,$F5,$AF,$9B,$A4
	.dc.b	$BE,$22,$3E,$4B,$8F,$AD,$FE,$90,$9A,$4F,$5B,$FC,$6A,$FF,$98,$63
	.dc.b	$87,$C5,$60,$CF,$CA,$EB,$82,$4F,$57,$8F,$4C,$8F,$03,$5F,$F0,$CD
	.dc.b	$BE,$63,$BF,$8B,$27,$CC,$F8,$BC,$59,$1E,$96,$BC,$BD,$3F,$F2,$5F
	.dc.b	$D6,$FE,$46,$3F,$C4,$AD,$E6,$FB,$CD,$E5,$FD,$82,$F3,$78,$92,$BE
	.dc.b	$4C,$65,$FF,$12,$F2,$7F,$96,$3E,$6F,$FC,$1B,$06,$93,$E1,$34,$48
	.dc.b	$FA,$71,$FB,$BF,$86,$FE,$B1,$1F,$5B,$EB,$3F,$9F,$E7,$F9,$36,$E7
	.dc.b	$FC,$15,$B3,$FE,$6C,$7F,$20,$DF,$53,$FC,$D7,$18,$83,$4F,$CA,$3F
	.dc.b	$E4,$86,$D7,$7F,$13,$89,$FE,$74,$F8,$CF,$D8,$C7,$8F,$D4,$EA,$4B
	.dc.b	$F4,$6F,$E0,$23,$83,$D9,$E2,$FF,$A2,$17,$D6,$FC,$77,$D5,$19,$7F
	.dc.b	$7F,$92,$31,$1F,$05,$AF,$0C,$3F,$82,$BF,$0F,$C6,$7D,$6E,$23,$E0
	.dc.b	$C6,$1D,$DF,$F3,$A9,$D3,$63,$FD,$83,$F5,$1F,$EC,$24,$D2,$3D,$7E
	.dc.b	$8F,$FB,$BE,$1F,$C2,$63,$FD,$87,$C1,$FB,$05,$3A,$8F,$E3,$7F,$27
	.dc.b	$67,$FA,$03,$57,$A3,$FF,$00,$67,$94,$E6,$31,$3F,$E8,$01,$CB,$C5
	.dc.b	$FF,$58,$D7,$86,$31,$E8,$C8,$F2,$7C,$8D,$AE,$4F,$13,$B7,$CB,$D9
	.dc.b	$FE,$13,$82,$3C,$D9,$FF,$41,$9F,$E1,$AF,$F8,$35,$BC,$BA,$E4,$91
	.dc.b	$CB,$C0,$B9,$7F,$A6,$31,$47,$C1,$FA,$DF,$8F,$5F,$C6,$1F,$97,$5F
	.dc.b	$E9,$E7,$2D,$DB,$81,$F1,$7B,$AD,$7A,$1E,$9E,$3F,$5F,$3E,$CF,$57
	.dc.b	$0F,$7F,$3A,$E1,$DD,$F2,$7C,$6F,$97,$D4,$B5,$E5,$70,$F9,$FD,$4B
	.dc.b	$7E,$77,$D9,$F1,$9E,$DF,$8C,$B5,$ED,$7F,$01,$F3,$0F,$98,$BF,$31
	.dc.b	$86,$D7,$F3,$A9,$6F,$C3,$C9,$E2,$D9,$FE,$7F,$F5,$BF,$E1,$26,$FD
	.dc.b	$47,$E3,$63,$1A,$FF,$B2,$14,$67,$9B,$5F,$2D,$6F,$9B,$F9,$5F,$AD
	.dc.b	$FC,$6E,$1F,$ED,$05,$DB,$9E,$ED,$3B,$E2,$E1,$C5,$71,$FE,$A7,$68
	.dc.b	$FD,$07,$D6,$FF,$BE,$E7,$FA,$BB,$EE,$6F,$9F,$AD,$FF,$4D,$1F,$E3
	.dc.b	$23,$FA,$E9,$EC,$EF,$FE,$F4,$E7,$4F,$F7,$2B,$93,$FC,$5D,$F5,$A8
	.dc.b	$FD,$83,$F8,$C9,$FF,$01,$39,$9F,$ED,$2E,$5E,$3E,$17,$0D,$B8,$B6
	.dc.b	$37,$D1,$D3,$6E,$AD,$8F,$EE,$7E,$23,$D9,$FE,$00,$FF,$95,$E3,$31
	.dc.b	$8F,$85,$CF,$AB,$AF,$0E,$D7,$27,$F8,$47,$D6,$FE,$CF,$76,$FF,$E1
	.dc.b	$B8,$FD,$A4,$62,$BF,$C9,$D1,$B9,$D7,$8B,$CB,$DC,$E7,$F0,$F6,$FF
	.dc.b	$C1,$36,$8E,$FE,$EF,$F1,$7F,$2F,$E3,$65,$D4,$F9,$95,$FE,$B3,$BF
	.dc.b	$F0,$8C,$EB,$2F,$F1,$73,$FE,$3A,$8C,$F8,$BC,$FC,$6E,$8F,$F2,$3F
	.dc.b	$AD,$7B,$B7,$E0,$F0,$DB,$FF,$0E,$FD,$6F,$E6,$9B,$F0,$DF,$CA,$E3
	.dc.b	$1A,$F9,$42,$8D,$1B,$F0,$ED,$74,$70,$FF,$8A,$BE,$B7,$F3,$0E,$4C
	.dc.b	$78,$3D,$FE,$0F,$C1,$C9,$67,$CA,$E1,$E5,$E6,$5C,$FC,$98,$23,$FA
	.dc.b	$17,$D6,$FC,$7A,$FF,$42,$3F,$91,$AF,$CC,$39,$6F,$17,$03,$FA,$55
	.dc.b	$BE,$6B,$17,$E9,$7A,$F8,$3D,$BC,$7B,$3F,$E4,$6C,$7F,$76,$D8,$F1
	.dc.b	$FB,$DB,$8D,$38,$7E,$27,$D2,$5C,$FD,$6F,$AB,$E7,$BD,$7E,$9D,$AE
	.dc.b	$CE,$1F,$99,$F4,$96,$FE,$1B,$F0,$BD,$4F,$87,$E3,$5A,$F3,$38,$7F
	.dc.b	$CB,$BF,$5A,$71,$7F,$7E,$31,$1E,$DE,$2F,$9B,$C1,$F3,$9B,$1F,$06
	.dc.b	$3E,$8E,$98,$BF,$C4,$6E,$8C,$D7,$F9,$B3,$EB,$7F,$74,$FC,$98,$FE
	.dc.b	$F5,$18,$AF,$C8,$7D,$6F,$E6,$AB,$FC,$E6,$FF,$AC,$DB,$D7,$FE,$01
	.dc.b	$8C,$4F,$0E,$CF,$B7,$C5,$96,$DF,$2F,$1F,$87,$AD,$C5,$B1,$DA,$3F
	.dc.b	$C1,$9D,$3E,$3F,$0B,$FD,$3D,$97,$F4,$FA,$FF,$42,$CE,$5F,$F6,$94
	.dc.b	$3E,$77,$C9,$F3,$BE,$56,$CF,$AA,$3D,$1F,$19,$8D,$7D,$EE,$27,$AE
	.dc.b	$76,$7F,$65,$8F,$F1,$F7,$E2,$F9,$5F,$07,$E2,$BE,$13,$6F,$E3,$C7
	.dc.b	$F8,$97,$E5,$DF,$4D,$9B,$FD,$E4,$62,$BE,$6F,$B7,$FE,$2B,$E1,$7B
	.dc.b	$9F,$36,$30,$C9,$FF,$52,$F4,$BE,$B7,$18,$FF,$07,$8C,$4F,$FA,$AB
	.dc.b	$EB,$7F,$94,$3F,$EF,$77,$F9,$6F,$27,$FB,$25,$CD,$7F,$4D,$73,$3B
	.dc.b	$F4,$7F,$C7,$DF,$5A,$E3,$FF,$20,$56,$AF,$FA,$A4,$E6,$FC,$1F,$0F
	.dc.b	$AF,$97,$DB,$63,$C9,$1D,$3E,$4B,$17,$FC,$7E,$D1,$DB,$CB,$EF,$CD
	.dc.b	$70,$F6,$38,$7F,$D8,$FF,$5A,$F8,$FC,$63,$FC,$A6,$31,$3F,$EC,$CF
	.dc.b	$73,$FC,$6B,$C1,$FE,$E9,$7F,$CA,$71,$9F,$86,$FF,$2E,$46,$2F,$F9
	.dc.b	$93,$6B,$FD,$E1,$6F,$9C,$DC,$FF,$98,$E3,$18,$77,$F3,$B9,$F9,$DD
	.dc.b	$1B,$1E,$38,$E1,$AE,$32,$F4,$3E,$2F,$5F,$C7,$EB,$DB,$F5,$38,$7F
	.dc.b	$DC,$FF,$58,$FA,$DC,$61,$FD,$26,$31,$3A,$E9,$FF,$40,$4F,$F8,$2E
	.dc.b	$7F,$9D,$4F,$FA,$45,$B1,$FE,$96,$6B,$FE,$11,$3F,$C2,$5E,$E7,$F0
	.dc.b	$77,$07,$F6,$FF,$7B,$AB,$6B,$FD,$39,$3B,$8E,$4F,$EA,$DE,$2D,$73
	.dc.b	$E2,$72,$8C,$47,$F8,$87,$83,$F8,$E5,$7F,$7A,$8C,$79,$9F,$CA,$DF
	.dc.b	$F4,$E4,$E9,$B1,$99,$F3,$CB,$D2,$C5,$7B,$63,$13,$D6,$3F,$D2,$F3
	.dc.b	$A7,$C4,$D5,$F5,$9F,$CE,$1F,$F5,$63,$67,$E2,$B6,$3E,$53,$5F,$97
	.dc.b	$D3,$CF,$1F,$D0,$D7,$85,$B3,$FE,$B7,$6C,$74,$F0,$FC,$57,$FC,$CA
	.dc.b	$30,$7A,$5C,$57,$83,$E3,$7F,$86,$BF,$ED,$28,$FF,$53,$78,$7F,$ED
	.dc.b	$15,$F4,$3F,$EC,$29,$D5,$70,$7D,$7B,$FE,$0F,$0B,$1D,$E3,$0F,$B9
	.dc.b	$AF,$F9,$67,$E5,$79,$4F,$E2,$1B,$1F,$8D,$6B,$BC,$78,$BF,$3B,$CE
	.dc.b	$B9,$7F,$A4,$63,$DD,$18,$33,$F1,$B2,$3E,$13,$5F,$AA,$DC,$3E,$AB
	.dc.b	$63,$EA,$FD,$17,$62,$3F,$2D,$F1,$E2,$F1,$E9,$27,$F1,$11,$97,$E9
	.dc.b	$79,$DE,$82,$38,$3E,$B7,$91,$FF,$00,$5F,$C5,$B7,$3F,$E6,$11,$F5
	.dc.b	$63,$1C,$52,$3E,$C3,$5F,$D9,$8B,$7D,$6F,$56,$C6,$8D,$46,$4D,$79
	.dc.b	$B2,$35,$3F,$D6,$9B,$1C,$FC,$59,$1A,$6B,$E2,$97,$D1,$F8,$49,$1D
	.dc.b	$1F,$81,$F3,$0F,$0F,$F8,$77,$8D,$FB,$BA,$4B,$AF,$3C,$7F,$C4,$E3
	.dc.b	$13,$FE,$A5,$AE,$11,$CE,$31,$3B,$BE,$E4,$8F,$85,$3B,$7E,$4B,$FE
	.dc.b	$C6,$73,$E2,$79,$88,$FE,$9F,$E9,$BC,$6F,$FA,$D7,$85,$18,$8F,$E0
	.dc.b	$AD,$87,$C6,$D2,$4F,$98,$E4,$F1,$D7,$FD,$01,$19,$1F,$5B,$F3,$1C
	.dc.b	$35,$FF,$44,$37,$4F,$94,$FF,$65,$8D,$EF,$E5,$8C,$47,$08,$F0,$7C
	.dc.b	$26,$7E,$56,$4F,$67,$87,$3C,$8F,$0B,$5F,$F4,$C6,$2F,$92,$ED,$F0
	.dc.b	$F2,$7C,$EF,$87,$09,$1E,$A6,$BC,$BC,$DF,$37,$CD,$C8,$F7,$BE,$9F
	.dc.b	$8C,$91,$F0,$9A,$F8,$BC,$EF,$A5,$F5,$AF,$46,$92,$7E,$64,$7E,$29
	.dc.b	$FF,$58,$37,$87,$F2,$4D,$E8,$FE,$C4,$D9,$FE,$39,$E9,$7D,$FF,$AD
	.dc.b	$FF,$1A,$47,$D7,$6B,$FE,$BB,$6D,$FE,$75,$7C,$47,$12,$4F,$17,$F1
	.dc.b	$5F,$53,$E9,$8E,$9F,$51,$DC,$8F,$7F,$89,$F1,$11,$C1,$F8,$9F,$AD
	.dc.b	$65,$E7,$F8,$E9,$3E,$77,$D5,$EA,$C8,$F5,$35,$F5,$7C,$4F,$BC,$8F
	.dc.b	$BD,$FD,$07,$D6,$FF,$49,$8F,$F6,$D7,$D6,$FE,$EF,$7F,$DE,$37,$87
	.dc.b	$FC,$18,$D9,$07,$FD,$D6,$FE,$85,$AF,$4F,$89,$FD,$74,$62,$BF,$84
	.dc.b	$7D,$6B,$3C,$FA,$53,$0E,$4F,$0F,$66,$47,$56,$BF,$EF,$1E,$2F,$AF
	.dc.b	$9F,$9B,$D0,$EE,$49,$F2,$7C,$47,$88,$8E,$0F,$47,$A3,$E6,$23,$83
	.dc.b	$F7,$9E,$43,$B3,$D7,$F2,$B2,$7E,$03,$EE,$F8,$99,$1F,$0D,$AF,$9F
	.dc.b	$A7,$E1,$FC,$CC,$8F,$90,$FC,$6F,$88,$91,$F2,$DA,$B8,$FF,$98,$7D
	.dc.b	$6B,$6F,$E3,$5E,$DF,$F4,$1A,$D7,$07,$F9,$0E,$31,$7F,$CD,$DB,$19
	.dc.b	$F4,$F8,$92,$BF,$8D,$4F,$F1,$59,$E5,$E6,$76,$24,$E9,$FE,$55,$F1
	.dc.b	$5F,$DA,$E3,$C2,$7E,$1A,$9F,$49,$E6,$24,$76,$FA,$3F,$49,$1C,$1F
	.dc.b	$CA,$EB,$FC,$F4,$F9,$3C,$BF,$79,$1F,$3F,$97,$F0,$88,$E0,$FF,$59
	.dc.b	$F0,$FC,$6F,$26,$47,$B9,$F9,$3E,$1C,$8F,$82,$D7,$D6,$7F,$8C,$3E
	.dc.b	$03,$E7,$91,$F8,$EF,$F4,$33,$FA,$67,$D6,$BB,$69,$27,$FA,$34,$7F
	.dc.b	$89,$5E,$AF,$F5,$07,$3E,$9F,$23,$9F,$3F,$F5,$7F,$AD,$7C,$65,$25
	.dc.b	$7F,$51,$87,$C6,$6F,$79,$39,$79,$6F,$FD,$87,$EB,$7F,$93,$49,$A4
	.dc.b	$F6,$F5,$F2,$D2,$7F,$AF,$4A,$39,$7C,$F7,$F6,$BF,$65,$F0,$F4,$97
	.dc.b	$FD,$96,$3F,$CB,$2D,$F0,$B8,$11,$D7,$85,$FD,$D7,$EB,$55,$49,$3F
	.dc.b	$DB,$E3,$F2,$1E,$07,$F8,$4D,$75,$F1,$24,$EF,$8B,$A7,$89,$23,$93
	.dc.b	$FE,$03,$FA,$DF,$F5,$3E,$1F,$E1,$D5,$F1,$F9,$25,$7F,$7F,$87,$C7
	.dc.b	$FF,$82,$FE,$B7,$FD,$CB,$3F,$E0,$36,$BF,$8A,$6D,$FF,$8A,$56,$DF
	.dc.b	$E1,$3F,$37,$FB,$FC,$FF,$8B,$5B,$E2,$FF,$AF,$5B,$7F,$CC,$9F,$8A
	.dc.b	$FF,$84,$57,$81,$CC,$91,$9F,$03,$B7,$B7,$23,$83,$E1,$DB,$FC,$49
	.dc.b	$F5,$AF,$9C,$A4,$AF,$F0,$F4,$7C,$47,$FC,$41,$F5,$BF,$D1,$53,$49
	.dc.b	$EA,$FF,$8B,$DF,$F3,$3C,$7E,$B1,$BA,$FF,$92,$DB,$AF,$FB,$6D,$F8
	.dc.b	$ED,$DB,$91,$2B,$C3,$CF,$D4,$8E,$0F,$2F,$CC,$78,$7C,$39,$1E,$87
	.dc.b	$C9,$E1,$C8,$F5,$B5,$F3,$79,$FF,$E4,$7F,$AD,$FC,$7E,$5F,$EE,$F6
	.dc.b	$CF,$F6,$46,$A7,$F9,$0B,$FE,$45,$6B,$FC,$31,$BA,$7F,$9A,$DB,$3F
	.dc.b	$EA,$FE,$1F,$F2,$B7,$CD,$FF,$83,$6F,$F8,$9B,$6F,$FC,$E4,$DA,$FE
	.dc.b	$EB,$28,$FD,$E9,$3E,$2B,$99,$FE,$59,$E3,$FF,$31,$7D,$5F,$F3,$FA
	.dc.b	$FE,$FA,$D4,$D3,$F2,$2E,$3D,$0E,$24,$7F,$CC,$51,$F9,$47,$F9,$4B
	.dc.b	$71,$FF,$44,$37,$1F,$3B,$67,$FC,$1B,$F1,$7F,$B2,$9C,$FC,$47,$98
	.dc.b	$93,$DB,$D1,$F4,$91,$C1,$E2,$F0,$7F,$B4,$2F,$AD,$F8,$8F,$BA,$32
	.dc.b	$FC,$83,$DE,$E1,$F2,$B5,$E1,$97,$FA,$D7,$2F,$E6,$7F,$5B,$7C,$33
	.dc.b	$49,$5F,$EC,$9F,$A7,$FF,$65,$26,$91,$FF,$60,$7D,$23,$5F,$F4,$64
	.dc.b	$7F,$99,$5B,$19,$8F,$EA,$A6,$4B,$F1,$39,$CB,$FD,$6D,$E7,$3D,$7C
	.dc.b	$61,$EF,$AB,$7F,$E3,$CA,$D4,$F8,$39,$33,$3F,$E4,$0F,$3E,$BF,$EB
	.dc.b	$F9,$F2,$C6,$0C,$FC,$7D,$8F,$4B,$87,$FD,$4D,$6F,$97,$C5,$FF,$6E
	.dc.b	$8C,$4F,$FA,$AF,$3F,$E8,$2D,$E4,$F1,$C6,$2B,$FB,$AB,$FE,$FF,$9D
	.dc.b	$1D,$BB,$F7,$5E,$AB,$47,$B3,$EB,$5B,$DD,$B0,$DC,$3F,$D7,$93,$F3
	.dc.b	$71,$F9,$7D,$D9,$CF,$D0,$F3,$F1,$7A,$79,$97,$0E,$2E,$1E,$7E,$5D
	.dc.b	$BF,$F0,$6E,$3F,$D2,$A7,$BF,$C4,$F9,$7D,$2F,$33,$63,$C7,$1E,$9F
	.dc.b	$4B,$17,$C9,$F6,$7C,$1F,$B7,$E2,$2D,$FA,$9F,$B7,$F3,$0F,$98,$BF
	.dc.b	$31,$86,$D7,$FB,$6E,$5C,$1C,$3C,$9E,$2D,$9F,$F7,$07,$D6,$FF,$84
	.dc.b	$1B,$F5,$BF,$88,$8C,$6B,$FE,$2A,$51,$9E,$8D,$7C,$9D,$BE,$6F,$E4
	.dc.b	$3E,$B5,$DD,$7B,$B0,$DA,$FF,$78,$4B,$8B,$87,$17,$16,$CF,$FB,$D7
	.dc.b	$EB,$7F,$DF,$AD,$DB,$F3,$B1,$8D,$BF,$C7,$EF,$C0,$9F,$F7,$E1,$FE
	.dc.b	$30,$2F,$BF,$FA,$47,$13,$FE,$46,$7D,$FD,$7F,$B5,$87,$D6,$FF,$AA
	.dc.b	$27,$FD,$53,$39,$8F,$C8,$C6,$0F,$E7,$2E,$E7,$FC,$A2,$E9,$BE,$17
	.dc.b	$ED,$23,$E2,$CF,$19,$FF,$5A,$C7,$ED,$5E,$3A,$EF,$A3,$86,$96,$F8
	.dc.b	$57,$9A,$3F,$7F,$E2,$3D,$9B,$F0,$78,$6D,$F0,$9F,$F0,$CB,$7C,$3E
	.dc.b	$3E,$AF,$27,$F8,$4F,$D6,$FE,$DF,$7F,$88,$FE,$DF,$3F,$F3,$6B,$9B
	.dc.b	$FE,$B2,$E6,$77,$ED,$72,$F8,$5E,$6D,$AF,$4D,$7F,$AD,$5F,$5F,$8D
	.dc.b	$F2,$6F,$BF,$A7,$FC,$6A,$31,$3F,$D9,$9F,$F1,$EB,$C3,$C5,$D7,$8D
	.dc.b	$B7,$F9,$0F,$D6,$BD,$BB,$F0,$58,$6D,$FC,$AF,$AD,$FC,$EB,$7E,$17
	.dc.b	$F9,$4C,$63,$83,$FD,$16,$A3,$26,$BC,$2B,$5C,$DC,$3F,$5B,$EB,$7F
	.dc.b	$DD,$0E,$4C,$78,$1D,$BE,$07,$BF,$1D,$9F,$23,$87,$97,$97,$73,$FF
	.dc.b	$4E,$E0,$8F,$E7,$FF,$59,$F1,$A7,$FD,$FE,$FE,$47,$93,$F3,$8E,$60
	.dc.b	$E6,$FF,$C2,$2D,$80,$B7,$CB,$76,$E4,$FE,$9B,$18,$BF,$C9,$DF,$E7
	.dc.b	$2F,$FD,$62,$BC,$6F,$2F,$81,$E6,$65,$70,$F7,$6D,$F0,$7D,$67,$6F
	.dc.b	$A4,$B3,$F2,$DF,$7F,$CC,$FC,$0F,$4E,$CF,$89,$C3,$F4,$7E,$92,$E1
	.dc.b	$F2,$1F,$8F,$E9,$FC,$8F,$1A,$D7,$A9,$C3,$FA,$EF,$AD,$FE,$F8,$E4
	.dc.b	$C1,$F4,$38,$FE,$8F,$15,$8F,$86,$E1,$F8,$BA,$5C,$BE,$2B,$F6,$0F
	.dc.b	$EE,$FF,$5B,$FB,$99,$D9,$FB,$E7,$13,$FE,$DC,$FA,$DF,$CE,$4F,$F0
	.dc.b	$27,$FD,$72,$DE,$CF,$F0,$0C,$62,$78,$76,$FD,$8E,$1C,$B6,$F9,$79
	.dc.b	$1C,$3D,$5E,$2D,$8E,$D1,$FE,$0C,$EA,$F1,$F5,$7F,$27,$CB,$FA,$A5
	.dc.b	$7F,$14,$9C,$BF,$E5,$C8,$7C,$C7,$93,$E6,$3C,$AD,$9F,$54,$7A,$3E
	.dc.b	$9F,$1A,$FB,$DC,$4F,$5C,$EC,$FF,$BF,$B1,$FF,$7C,$4F,$C3,$F2,$7E
	.dc.b	$0F,$C5,$7C,$26,$C7,$C7,$8F,$F1,$2F,$CA,$BE,$97,$38,$36,$FE,$85
	.dc.b	$39,$BF,$9D,$ED,$7F,$8B,$38,$5E,$DF,$CE,$8C,$32,$7F,$9A,$F4,$3E
	.dc.b	$B7,$19,$75,$9F,$F3,$FC,$E5,$FE,$77,$F5,$BF,$CA,$E7,$F6,$2F,$F2
	.dc.b	$B8,$FF,$42,$B9,$83,$97,$7F,$03,$FE,$40,$FA,$D7,$1F,$F9,$0A,$B5
	.dc.b	$49,$CD,$F8,$3E,$1B,$5E,$D6,$DB,$1E,$48,$E9,$F5,$58,$BF,$51,$FC
	.dc.b	$1A,$D1,$DF,$CB,$F0,$73,$5C,$3D,$AE,$1F,$EB,$3F,$5A,$F8,$EC,$63
	.dc.b	$D0,$46,$67,$FA,$FF,$BB,$FE,$3B,$7F,$D4,$EF,$F6,$7E,$6F,$EF,$0E
	.dc.b	$6B,$F8,$CB,$99,$DB,$FE,$64,$A7,$D3,$FF,$99,$A3,$55,$FC,$82,$73
	.dc.b	$5E,$0F,$33,$5E,$66,$DB,$1E,$48,$E9,$A6,$2F,$D5,$7F,$EF,$CA,$F4
	.dc.b	$FA,$DD,$FD,$6F,$03,$63,$DB,$1F,$E7,$8F,$AC,$7D,$6E,$70,$F8,$91
	.dc.b	$8A,$FF,$00,$BF,$CC,$63,$FD,$44,$1C,$3F,$EA,$56,$BF,$EC,$E6,$BF
	.dc.b	$CF,$BD,$BF,$D4,$BC,$9A,$FF,$AB,$5A,$FF,$B5,$9A,$F1,$FA,$0F,$F1
	.dc.b	$43,$96,$B9,$AF,$2A,$94,$EA,$1B,$3D,$0F,$F5,$35,$FF,$AF,$63,$0F
	.dc.b	$F8,$B6,$7E,$EC,$E9,$FC,$1B,$5F,$EB,$3F,$59,$FC,$E1,$F2,$B6,$7F
	.dc.b	$D9,$0D,$8F,$F7,$8B,$5F,$89,$CB,$CD,$F4,$37,$1F,$CF,$D7,$8E,$72
	.dc.b	$D8,$7D,$1E,$C9,$19,$7A,$5F,$FA,$AF,$BF,$E2,$B6,$FE,$DB,$FE,$C1
	.dc.b	$F5,$F3,$F1,$BF,$C4,$27,$3F,$05,$FD,$D9,$F6,$B7,$1F,$6B,$63,$DA
	.dc.b	$D7,$E2,$F9,$35,$60,$8C,$36,$3D,$CD,$7E,$5F,$9B,$A2,$3A,$78,$FC
	.dc.b	$5C,$8E,$2D,$7E,$9F,$0F,$62,$3D,$9A,$E8,$C1,$0F,$C7,$FD,$5B,$C4
	.dc.b	$F5,$36,$3E,$A3,$5D,$C7,$BF,$E5,$7F,$D9,$31,$F6,$5A,$FE,$69,$BA
	.dc.b	$7C,$D8,$46,$1F,$C3,$F8,$BF,$9C,$7E,$4B,$73,$FE,$63,$1F,$72,$31
	.dc.b	$C7,$23,$EF,$B5,$FD,$CF,$D7,$3C,$2E,$8D,$8F,$D4,$B5,$DC,$78,$73
	.dc.b	$FE,$B5,$1F,$AE,$6B,$FC,$11,$AF,$AD,$AF,$E2,$5F,$E6,$4F,$1F,$0B
	.dc.b	$A1,$1E,$9F,$49,$DC,$8E,$0F,$07,$D0,$78,$C8,$E0,$FE,$03,$A7,$FB
	.dc.b	$11,$BE,$17,$C9,$8C,$47,$F1,$A6,$BF,$45,$CF,$89,$FE,$B5,$6E,$FF
	.dc.b	$A1,$6C,$FF,$96,$1F,$F2,$83,$FE,$A6,$5E,$A7,$E9,$EB,$97,$89,$6F
	.dc.b	$93,$6F,$95,$57,$F4,$BD,$77,$8B,$F7,$B1,$FE,$B2,$6B,$FD,$65,$7C
	.dc.b	$DE,$C9,$1B,$7A,$6F,$FE,$8B,$8F,$F5,$DF,$D6,$B7,$F0,$36,$33,$18
	.dc.b	$E0,$91,$C1,$AF,$F6,$9A,$7C,$1F,$2E,$19,$3D,$9E,$BC,$D2,$3C,$2D
	.dc.b	$79,$7C,$87,$8B,$E1,$64,$79,$9F,$2D,$C8,$F4,$B5,$F1,$79,$5E,$AF
	.dc.b	$9B,$91,$EE,$7D,$BF,$13,$23,$E0,$B5,$F5,$79,$9F,$47,$EB,$5F,$2D
	.dc.b	$49,$3F,$EE,$28,$FE,$92,$DF,$27,$81,$1F,$F7,$3E,$FF,$A9,$E9,$FF
	.dc.b	$7F,$7D,$6F,$F8,$D6,$7F,$DD,$6D,$7F,$C2,$4D,$8D,$FC,$3F,$E0,$E0
	.dc.b	$D2,$4F,$FB,$D3,$EA,$7C,$EF,$4F,$A8,$E0,$47,$BF,$C4,$73,$23,$83
	.dc.b	$FD,$FB,$F5,$BE,$7E,$FF,$E7,$D8,$BE,$37,$C4,$4A,$F4,$FB,$5E,$52
	.dc.b	$38,$3D,$BF,$11,$E9,$23,$83,$FD,$7F,$AB,$0F,$E9,$9F,$5B,$FC,$92
	.dc.b	$31,$3F,$EC,$4F,$AD,$AF,$F8,$DD,$B1,$86,$D7,$F5,$57,$FD,$E5,$3D
	.dc.b	$A3,$1F,$11,$FA,$D9,$CC,$F3,$8F,$D8,$7D,$6F,$BE,$38,$1E,$1F,$47
	.dc.b	$81,$1C,$8A,$FE,$AD,$87,$8F,$D1,$EC,$49,$FA,$DF,$01,$E1,$F4,$24
	.dc.b	$72,$BF,$F0,$AB,$7E,$23,$CC,$47,$06,$FE,$23,$D4,$47,$07,$93,$CA
	.dc.b	$77,$7B,$FE,$52,$4F,$77,$E1,$7C,$4C,$8F,$13,$5F,$F3,$27,$53,$AB
	.dc.b	$E4,$7C,$CC,$9F,$43,$F2,$FE,$1A,$47,$AD,$AF,$F9,$B7,$EB,$7D,$AF
	.dc.b	$F9,$BF,$E7,$FF,$A3,$F1,$FD,$C6,$F3,$FF,$83,$5B,$20,$FF,$15,$79
	.dc.b	$79,$BF,$C9,$E3,$15,$CE,$31,$E6,$CA,$FA,$F5,$FD,$41,$F1,$47,$F2
	.dc.b	$FF,$8A,$FE,$F1,$7E,$3F,$A4,$E8,$4A,$E3,$F4,$7D,$C8,$E0,$FC,$1D
	.dc.b	$7F,$9F,$9E,$FE,$6F,$94,$8F,$8B,$CD,$F4,$11,$C1,$FE,$BA,$E0,$F6
	.dc.b	$79,$72,$3D,$2F,$BF,$D8,$91,$EC,$6B,$F3,$7E,$07,$E2,$11,$F8,$2F
	.dc.b	$F4,$23,$FA,$77,$D6,$FE,$63,$87,$FC,$78,$D9,$FA,$0D,$4B,$DF,$E3
	.dc.b	$EB,$49,$FD,$1C,$FF,$52,$6F,$37,$F5,$BF,$AD,$7C,$65,$24,$FE,$AA
	.dc.b	$1F,$19,$BE,$09,$39,$79,$70,$7F,$65,$FA,$DF,$E6,$33,$FE,$B9,$6F
	.dc.b	$2F,$F0,$66,$ED,$FE,$57,$7F,$BB,$C9,$7C,$B7,$32,$47,$EF,$21,$F3
	.dc.b	$DF,$DC,$FD,$B7,$C8,$D2,$57,$F0,$28,$FC,$AB,$7C,$1E,$04,$75,$E0
	.dc.b	$FF,$7D,$FA,$D5,$52,$4F,$F0,$C8,$FC,$9E,$7F,$E6,$F5,$D7,$C4,$93
	.dc.b	$BE,$2E,$9E,$1C,$8E,$4F,$F8,$2F,$EB,$7F,$C9,$9C,$1F,$CA,$97,$C8
	.dc.b	$64,$95,$FC,$72,$1F,$21,$FE,$13,$FA,$DF,$F7,$7C,$FF,$20,$6B,$FE
	.dc.b	$E9,$6D,$FF,$36,$5B,$FF,$86,$FC,$FF,$D8,$4F,$F3,$A6,$ED,$FE,$8B
	.dc.b	$6D,$FE,$FD,$F8,$AF,$F2,$E5,$DF,$CC,$91,$9E,$FE,$DE,$F4,$8E,$0F
	.dc.b	$87,$6F,$F1,$67,$D6,$BE,$72,$92,$BF,$9D,$C7,$C4,$7F,$9E,$FD,$6F
	.dc.b	$F4,$C7,$F8,$93,$53,$4F,$E9,$7A,$7F,$56,$7F,$CE,$70,$8C,$7C,$6F
	.dc.b	$F8,$F6,$73,$1B,$F8,$DC,$6F,$2F,$C2,$EF,$A2,$47,$91,$AF,$5E,$0F
	.dc.b	$F5,$A6,$0F,$99,$ED,$E2,$4A,$F5,$BE,$1F,$0E,$47,$B9,$AF,$33,$DF
	.dc.b	$E7,$7F,$94,$FE,$B7,$F2,$71,$FC,$CD,$A9,$A3,$CB,$E0,$48,$FE,$CF
	.dc.b	$19,$7F,$C4,$BC,$7F,$E5,$FF,$8B,$FF,$0C,$C7,$84,$FF,$34,$FC,$36
	.dc.b	$4D,$7C,$2C,$8F,$EE,$33,$FB,$E7,$FA,$33,$57,$C2,$FF,$62,$7F,$BB
	.dc.b	$FD,$67,$F4,$6E,$0F,$F6,$D3,$63,$FA,$73,$53,$F9,$46,$FA,$3F,$E7
	.dc.b	$58,$C1,$3C,$4C,$12,$7B,$BF,$E7,$BF,$8E,$FD,$8D,$F8,$3E,$2F,$91
	.dc.b	$2B,$CB,$BF,$52,$38,$3D,$3E,$2F,$E2,$D7,$D6,$FF,$A3,$9F,$4C,$65
	.dc.b	$FF,$65,$65,$FC,$23,$AB,$FD,$99,$1A,$6B,$EE,$9E,$0F,$C4,$F1,$9F
	.dc.b	$5B,$AB,$FE,$72,$D8,$CC,$F4,$9C,$BF,$ED,$5F,$A7,$FF,$6B,$26,$91
	.dc.b	$F0,$FC,$B7,$F8,$FB,$1F,$F1,$EC,$7F,$9C,$35,$FF,$16,$A9,$D4,$7F
	.dc.b	$B9,$79,$2C,$FF,$81,$4E,$07,$E2,$7C,$FF,$F0,$3C,$F2,$9D,$CF,$FA
	.dc.b	$5B,$D8,$AB,$9A,$FC,$9E,$1B,$7E,$28,$DF,$C9,$E2,$BF,$2C,$1C,$CF
	.dc.b	$F9,$13,$17,$CB,$62,$BC,$8F,$8B,$4F,$E8,$35,$FE,$4A,$5C,$99,$E6
	.dc.b	$D3,$27,$A7,$83,$70,$F7,$3E,$CF,$AD,$F8,$D3,$FE,$54,$7E,$1E,$1F
	.dc.b	$E6,$87,$35,$FA,$D7,$33,$F2,$5B,$1F,$E2,$FC,$9D,$FD,$78,$3E,$4C
	.dc.b	$62,$FB,$78,$FE,$1F,$3F,$C4,$D8,$F1,$47,$97,$CE,$C6,$3E,$17,$D1
	.dc.b	$F1,$FE,$9F,$56,$DF,$A5,$C3,$EB,$F5,$2D,$FA,$5F,$81,$F1,$BF,$07
	.dc.b	$E3,$6D,$7C,$17,$EF,$FC,$C7,$F8,$5E,$3F,$CE,$EB,$E6,$33,$EF,$69
	.dc.b	$DF,$0F,$0F,$2A,$DF,$07,$F1,$3F,$5B,$FE,$16,$8F,$F4,$0A,$FA,$DC
	.dc.b	$FB,$DA,$77,$C9,$C3,$E5,$ED,$F0,$7F,$27,$F5,$AE,$FB,$DF,$86,$D7
	.dc.b	$FA,$3A,$7F,$C2,$6D,$E1,$E3,$C5,$C9,$FC,$EF,$D6,$BD,$1B,$C1,$86
	.dc.b	$DF,$FA,$60,$FF,$19,$97,$6F,$D2,$B8,$E5,$FE,$64,$FB,$DF,$E6,$9F
	.dc.b	$5B,$FE,$73,$8F,$F5,$0B,$F8,$CE,$EF,$F5,$AB,$99,$DF,$8B,$FB,$08
	.dc.b	$FD,$75,$70,$DB,$1E,$26,$5D,$75,$6F,$7B,$5D,$9F,$DC,$F8,$CF,$72
	.dc.b	$FC,$8E,$1B,$5F,$EB,$49,$FF,$08,$B7,$C4,$E3,$EB,$F2,$7F,$81,$FD
	.dc.b	$6F,$EC,$B7,$FE,$A0,$FE,$CA,$BF,$D7,$AE,$78,$3F,$DA,$2E,$67,$7E
	.dc.b	$EF,$2F,$87,$E6,$DB,$E3,$EF,$7F,$10,$E3,$F8,$98,$F2,$8C,$61,$9F
	.dc.b	$1B,$5E,$35,$C3,$FD,$9B,$87,$F8,$C9,$FF,$67,$FD,$6B,$E5,$7F,$8D
	.dc.b	$BF,$E5,$48,$CC,$63,$E1,$7F,$8F,$FD,$6F,$E5,$67,$FD,$AE,$BE,$23
	.dc.b	$3E,$26,$99,$3A,$F1,$2E,$BF,$DC,$ED,$1D,$0F,$E5,$7F,$5A,$F0,$B1
	.dc.b	$7F,$EE,$48,$C3,$E0,$F8,$5F,$1F,$1F,$91,$B1,$E4,$8E,$7E,$66,$32
	.dc.b	$FF,$01,$5A,$3F,$9C,$FD,$6F,$CA,$BF,$E8,$0F,$E4,$5B,$5A,$E1,$91
	.dc.b	$E3,$F9,$AB,$3D,$1E,$7E,$07,$4F,$22,$D7,$FB,$F3,$0F,$E7,$8D,$8F
	.dc.b	$0F,$C0,$6E,$34,$E1,$F8,$1F,$59,$71,$F4,$BE,$8F,$9E,$F4,$FA,$76
	.dc.b	$BB,$38,$7E,$47,$D2,$5B,$F8,$2F,$C0,$F5,$3E,$0F,$8F,$6B,$CC,$E1
	.dc.b	$FF,$73,$77,$9F,$EC,$9F,$5B,$5D,$9A,$32,$7C,$CE,$2F,$9B,$C3,$63
	.dc.b	$E1,$38,$7E,$26,$B7,$3F,$F1,$B7,$02,$3E,$A9,$FD,$BB,$EB,$7F,$6D
	.dc.b	$5C,$0F,$05,$7F,$BE,$E3,$2F,$F8,$FF,$EB,$7F,$39,$3A,$84,$7E,$75
	.dc.b	$D7,$77,$F7,$A8,$C6,$9E,$F9,$1C,$3B,$BF,$2B,$C7,$B6,$E1,$D6,$DE
	.dc.b	$4F,$3F,$5B,$A3,$67,$51,$FE,$08,$EA,$F3,$E5,$DF,$2F,$2C,$E7,$CA
	.dc.b	$F2,$7C,$DF,$97,$E6,$EE,$3D,$DC,$3E,$1F,$8E,$B8,$7C,$17,$D8,$77
	.dc.b	$CE,$CF,$C3,$CF,$CE,$FA,$7C,$BF,$85,$F1,$9F,$0D,$B1,$ED,$8F,$F0
	.dc.b	$FF,$CB,$BE,$B7,$39,$7F,$9B,$63,$15,$F0,$FD,$BF,$F1,$2F,$13,$DD
	.dc.b	$F9,$D1,$86,$4F,$D5,$EA,$7C,$FE,$31,$FF,$3C,$46,$27,$ED,$7D,$6F
	.dc.b	$F2,$97,$FC,$6C,$FF,$29,$E5,$F9,$4E,$6B,$EC,$B9,$9D,$FC,$17,$F8
	.dc.b	$E3,$EB,$5C,$7F,$E3,$AA,$D5,$FE,$02,$73,$7E,$0F,$88,$D7,$B7,$B6
	.dc.b	$C7,$92,$3A,$7C,$9E,$2F,$FD,$21,$68,$ED,$E6,$77,$E6,$B8,$7B,$1C
	.dc.b	$3F,$97,$FA,$D7,$C8,$63,$1F,$F4,$D4,$62,$7F,$39,$EE,$FF,$8B,$A7
	.dc.b	$FC,$A0,$FF,$95,$0C,$1F,$D9,$9F,$F5,$13,$78,$3F,$21,$4F,$A8,$E1
	.dc.b	$3F,$E5,$C8,$C5,$F6,$F3,$BA,$79,$DD,$5B,$1E,$28,$E1,$A6,$32,$F3
	.dc.b	$BE,$3F,$63,$C9,$EC,$5B,$F4,$B8,$7F,$6D,$F5,$8F,$AD,$C6,$1F,$EB
	.dc.b	$98,$C4,$EB,$CA,$FF,$3E,$CF,$ED,$A7,$F8,$44,$FF,$A4,$9B,$1F,$E9
	.dc.b	$66,$BF,$E7,$D1,$F5,$AF,$87,$E7,$18,$6C,$66,$BF,$D3,$AD,$9F,$F5
	.dc.b	$0E,$FF,$E2,$A6,$FC,$17,$28,$C4,$77,$6B,$FE,$8D,$F7,$FF,$95,$3F
	.dc.b	$E9,$66,$DF,$95,$B1,$B8,$FE,$28,$7A,$7E,$B7,$F8,$EC,$7F,$A7,$9B
	.dc.b	$19,$F3,$B7,$3F,$E9,$B9,$CB,$7C,$3C,$BF,$C7,$A3,$F9,$25,$75,$9C
	.dc.b	$CA,$72,$F3,$F8,$EF,$DE,$3F,$EC,$28,$FF,$4A,$F8,$7F,$EC,$25,$F1
	.dc.b	$FF,$EA,$F9,$D5,$70,$7D,$DE,$2F,$FA,$BF,$1B,$1F,$E3,$F3,$F9,$8E
	.dc.b	$69,$D4,$7C,$96,$BF,$CD,$BE,$5B,$A5,$7D,$D6,$C7,$FA,$ED,$AE,$F2
	.dc.b	$E2,$FB,$7C,$FB,$99,$A4,$8F,$20,$A7,$C9,$F2,$6C,$69,$AF,$FB,$49
	.dc.b	$B8,$79,$7C,$3E,$04,$9E,$0F,$8B,$CF,$91,$E0,$C5,$18,$8F,$C0,$36
	.dc.b	$BF,$08,$D8,$FC,$07,$9F,$EC,$23,$CD,$F7,$F8,$FF,$EF,$88,$FB,$CD
	.dc.b	$BF,$E4,$B1,$F6,$E3,$5F,$2C,$F9,$0A,$49,$FF,$80,$31,$97,$F5,$C7
	.dc.b	$D5,$FA,$EE,$DA,$1A,$48,$D9,$F3,$7C,$FD,$91,$CC,$FF,$51,$7F,$67
	.dc.b	$C5,$E9,$E9,$34,$91,$C3,$CA,$F0,$91,$C1,$FB,$CF,$98,$E4,$47,$07
	.dc.b	$E6,$FE,$2B,$E2,$11,$C1,$F0,$3C,$0F,$F0,$3A,$F2,$3F,$BE,$6B,$FC
	.dc.b	$6E,$3F,$C2,$4D,$75,$C7,$C2,$3F,$B0,$C6,$BD,$0F,$9E,$47,$E6,$BF
	.dc.b	$4B,$E0,$24,$7D,$06,$BF,$B3,$FA,$6F,$1E,$1F,$E0,$DE,$44,$63,$1F
	.dc.b	$E0,$8D,$AF,$E0,$5F,$1D,$5F,$E0,$0D,$AF,$F6,$14,$7F,$8A,$DA,$FA
	.dc.b	$8F,$D9,$7C,$47,$1F,$AE,$7D,$2D,$24,$FF,$8C,$E3,$FA,$EB,$7D,$0E
	.dc.b	$04,$7F,$C7,$1C,$BF,$E8,$67,$87,$CB,$F5,$23,$CF,$D6,$F0,$11,$C1
	.dc.b	$CB,$C9,$F2,$65,$23,$C4,$F9,$FE,$2A,$47,$95,$AF,$B4,$FE,$01,$E9
	.dc.b	$FB,$79,$E4,$7A,$DF,$81,$F1,$D2,$3D,$CD,$79,$66,$F4,$7E,$77,$D7
	.dc.b	$7E,$9E,$7B,$37,$97,$FC,$88,$DE,$5F,$C6,$AF,$37,$89,$2B,$FC,$BD
	.dc.b	$19,$7E,$4F,$AD,$F6,$FE,$B7,$F1,$B1,$EB,$6E,$1F,$E5,$16,$E1,$F9
	.dc.b	$A5,$D3,$C4,$95,$FE,$6B,$8C,$BF,$E6,$DF,$A9,$F3,$3F,$E6,$88,$7D
	.dc.b	$47,$84,$91,$C7,$E2,$BC,$84,$70,$7F,$9D,$FE,$B9,$9F,$A7,$E4,$24
	.dc.b	$F8,$5F,$67,$C0,$48,$F2,$35,$F7,$7C,$57,$C1,$23,$E9,$7F,$3D,$F5
	.dc.b	$DF,$D5,$63,$EF,$7D,$67,$F9,$CF,$83,$FC,$E8,$DE,$0F,$DE,$B6,$7F
	.dc.b	$D0,$AF,$FB,$A5,$FF,$47,$35,$E7,$EC,$7E,$AE,$31,$5F,$E7,$FF,$AE
	.dc.b	$FF,$6F,$57,$0F,$89,$D1,$1E,$7F,$13,$C4,$8E,$0F,$A1,$E3,$F5,$FA
	.dc.b	$E9,$1E,$27,$C1,$EA,$48,$F2,$B5,$F7,$FA,$9E,$52,$3E,$A7,$F8,$17
	.dc.b	$94,$EE,$F5,$79,$B2,$BD,$EF,$B7,$E2,$A4,$7C,$26,$BF,$A9,$EB,$FC
	.dc.b	$1B,$F1,$BE,$6F,$E2,$11,$F9,$3E,$DF,$C7,$23,$83,$FD,$75,$F5,$CD
	.dc.b	$7F,$8B,$7B,$DF,$D0,$E7,$F6,$CD,$E3,$FF,$56,$37,$9B,$F9,$AB,$FC
	.dc.b	$52,$7F,$D8,$8D,$78,$F9,$DD,$0E,$BE,$6C,$8D,$B5,$FF,$50,$79,$9E
	.dc.b	$2F,$A5,$91,$D1,$F2,$FB,$92,$3B,$B5,$FF,$54,$79,$7E,$AF,$36,$47
	.dc.b	$91,$F6,$F9,$B2,$3C,$ED,$7F,$D4,$BE,$37,$C2,$F2,$E4,$7B,$1F,$8B
	.dc.b	$E2,$C8,$F7,$B5,$F8,$FE,$B7,$CA,$23,$F1,$1F,$E8,$5E,$6F,$F6,$C8
	.dc.b	$FF,$75,$7D,$77,$FB,$79,$BE,$07,$F4,$78,$C4,$7F,$27,$7F,$C5,$4F
	.dc.b	$0F,$53,$44,$79,$F4,$FF,$4F,$FA,$E7,$BD,$49,$3F,$EF,$98,$7B,$3B
	.dc.b	$E0,$93,$97,$97,$07,$F5,$9F,$AE,$FE,$45,$3F,$CE,$5B,$D5,$FE,$EF
	.dc.b	$6F,$57,$F7,$85,$F1,$5C,$7E,$5A,$58,$E4,$F5,$F8,$6F,$EC,$DE,$CB
	.dc.b	$E1,$A9,$2B,$86,$5F,$EF,$D9,$FF,$31,$C6,$32,$FF,$03,$C3,$C5,$E6
	.dc.b	$49,$D1,$E1,$87,$8B,$FD,$D7,$EB,$91,$49,$3C,$04,$3B,$F8,$77,$49
	.dc.b	$D4,$FF,$95,$B2,$65,$FE,$1A,$8F,$A7,$3A,$F1,$FC,$04,$7B,$7D,$77
	.dc.b	$FA,$AE,$F8,$C7,$C7,$87,$C6,$7F,$82,$DB,$1C,$23,$1F,$19,$FE,$0F
	.dc.b	$FA,$EF,$F7,$34,$FA,$5A,$FF,$71,$6D,$F6,$84,$7C,$97,$E0,$7A,$1F
	.dc.b	$E0,$C7,$FB,$C3,$67,$83,$73,$FF,$1B,$BF,$2A,$78,$F8,$5F,$E2,$18
	.dc.b	$FE,$ED,$18,$3A,$F8,$59,$F7,$A4,$78,$1E,$1A,$FF,$8A,$7E,$B9,$F3
	.dc.b	$94,$95,$F5,$63,$E1,$BF,$E0,$DF,$AE,$FE,$98,$9A,$4F,$5F,$FC,$68
	.dc.b	$FF,$9D,$23,$87,$C5,$E0,$CF,$C9,$EB,$A6,$4F,$E0,$EB,$E3,$3D,$FA
	.dc.b	$79,$91,$D3,$DB,$E5,$FB,$F8,$D2,$79,$3E,$2F,$1A,$47,$56,$BE,$BF
	.dc.b	$4F,$FC,$95,$F5,$DF,$91,$8F,$CA,$35,$FF,$16,$37,$9B,$EE,$2F,$3F
	.dc.b	$89,$27,$8B,$FC,$53,$8F,$FC,$AD,$F3,$5F,$E1,$09,$34,$9F,$0D,$A2
	.dc.b	$47,$E9,$63,$F8,$0B,$FE,$C9,$E5,$FF,$30,$7D,$67,$F4,$3C,$BF,$1C
	.dc.b	$DE,$AF,$F3,$EB,$6B,$F8,$23,$F8,$B6,$FA,$7C,$88,$FE,$CE,$3F,$1C
	.dc.b	$FF,$92,$DB,$5D,$BC,$3F,$09,$FE,$71,$F8,$BF,$D6,$61,$E0,$F1,$3C
	.dc.b	$A4,$C3,$A6,$DE,$82,$38,$3C,$1E,$27,$FA,$4D,$7D,$77,$C6,$7D,$31
	.dc.b	$97,$FD,$32,$33,$FF,$6E,$E7,$FE,$B4,$6C,$7F,$AF,$1A,$F0,$7F,$9C
	.dc.b	$FD,$76,$FD,$03,$49,$3F,$EC,$FF,$A8,$FF,$68,$26,$91,$FF,$61,$FD
	.dc.b	$1F,$F0,$CC,$3F,$80,$47,$FB,$47,$5F,$E3,$CA,$75,$1E,$AF,$94,$B3
	.dc.b	$FE,$B4,$6A,$F4,$1E,$DE,$23,$A1,$FE,$7F,$8C,$D7,$6E,$6D,$7A,$1B
	.dc.b	$6C,$78,$A3,$97,$CD,$62,$FF,$D1,$01,$CC,$FF,$35,$B7,$CC,$62,$BC
	.dc.b	$0F,$9B,$4F,$E7,$B5,$FC,$ED,$FE,$74,$DE,$6D,$72,$C8,$E9,$E0,$5C
	.dc.b	$3D,$4F,$C0,$FA,$EF,$8D,$3F,$D1,$5F,$7E,$1F,$CA,$DC,$D7,$FA,$7D
	.dc.b	$CC,$FF,$B7,$1B,$1F,$D8,$F2,$77,$F5,$E0,$F9,$B1,$8B,$ED,$E4,$78
	.dc.b	$7C,$EF,$13,$63,$C5,$1E,$5F,$3B,$18,$F8,$5F,$47,$C6,$FA,$7D,$4B
	.dc.b	$7E,$97,$0F,$AF,$D4,$B7,$E9,$7E,$07,$C6,$7C,$1F,$8C,$B5,$F0,$5F
	.dc.b	$C1,$FC,$CB,$E6,$6F,$D7,$61,$B5,$FD,$AE,$7D,$98,$FF,$B0,$D7,$95
	.dc.b	$CB,$8B,$9B,$F8,$EF,$AE,$FF,$0B,$DF,$F7,$35,$F5,$39,$F7,$B4,$EF
	.dc.b	$93,$87,$CB,$5B,$E0,$FE,$63,$EB,$BF,$C2,$B1,$FD,$F5,$FC,$63,$7B
	.dc.b	$F5,$DF,$BE,$3E,$1C,$76,$B8,$3F,$A2,$FA,$E7,$A3,$78,$30,$DA,$FF
	.dc.b	$05,$1F,$E3,$42,$ED,$FA,$A7,$1C,$5F,$EE,$57,$DE,$FF,$B9,$FE,$BB
	.dc.b	$FC,$51,$1F,$E1,$37,$F1,$BD,$DF,$DE,$5C,$CE,$FC,$5F,$DA,$C7,$FB
	.dc.b	$76,$B8,$6D,$8F,$13,$2E,$BA,$B7,$BD,$AE,$CF,$EF,$BC,$67,$B7,$7E
	.dc.b	$17,$0D,$AF,$F1,$34,$FF,$8C,$5B,$E2,$71,$F5,$F9,$3F,$C2,$3E,$BB
	.dc.b	$F6,$7B,$B8,$3F,$C2,$51,$FB,$48,$C5,$FF,$91,$D4,$70,$9D,$F8,$FC
	.dc.b	$FD,$DE,$9F,$11,$70,$FF,$1C,$5A,$3C,$1E,$AF,$F1,$8E,$4F,$8F,$8F
	.dc.b	$82,$31,$96,$7C,$7D,$78,$F7,$2F,$F2,$16,$08,$FE,$43,$F5,$CF,$7A
	.dc.b	$FC,$26,$1B,$7F,$E4,$9F,$AE,$FC,$BB,$7E,$1F,$F9,$4C,$63,$3F,$79
	.dc.b	$FE,$65,$51,$A3,$7E,$25,$AE,$8E,$1F,$F2,$B7,$D7,$3C,$2C,$61,$D0
	.dc.b	$46,$67,$C9,$F0,$9E,$2E,$4F,$1B,$63,$D1,$1F,$CF,$A5,$E6,$E6,$FA
	.dc.b	$46,$2B,$FC,$CD,$F5,$9F,$31,$FB,$0F,$E5,$9B,$3F,$D2,$23,$0F,$FA
	.dc.b	$0D,$AF,$9B,$E6,$B5,$C5,$E1,$78,$78,$1C,$7C,$7B,$87,$81,$C3,$E2
	.dc.b	$F7,$EE,$1F,$07,$1F,$F0,$DC,$E4,$F8,$FF,$4D,$E3,$AF,$73,$E5,$F9
	.dc.b	$EF,$37,$A9,$6B,$DA,$E1,$F9,$FF,$5D,$71,$F8,$6F,$B7,$D5,$F7,$78
	.dc.b	$F6,$BE,$2B,$87,$FD,$19,$F5,$DF,$E8,$67,$26,$0F,$8D,$C5,$F1,$F8
	.dc.b	$2C,$7C,$E7,$0F,$D0,$D2,$E5,$EF,$7E,$71,$FD,$DB,$EB,$BF,$73,$3B
	.dc.b	$3F,$D3,$8E,$27,$F0,$DF,$5D,$F9,$B9,$FF,$50,$3F,$EB,$36,$F5,$FF
	.dc.b	$80,$23,$13,$C3,$B3,$F0,$DC,$39,$6D,$F2,$F2,$78,$7A,$FC,$5B,$1D
	.dc.b	$A3,$FC,$17,$D3,$E3,$F0,$FF,$D2,$99,$7F,$4B,$AF,$F5,$94,$E5,$FE
	.dc.b	$09,$0F,$9C,$F2,$7C,$E7,$95,$B3,$EA,$8F,$47,$C6,$E3,$5F,$7B,$89
	.dc.b	$EB,$9D,$9F,$D4,$62,$8F,$81,$E5,$7C,$1F,$8B,$B7,$F1,$9C,$3F,$EC
	.dc.b	$7F,$AE,$7D,$4E,$31,$FD,$94,$62,$7E,$6F,$B9,$FE,$2A,$E1,$7B,$BF
	.dc.b	$36,$30,$C9,$FF,$69,$74,$3E,$7F,$18,$FE,$F6,$31,$3F,$ED,$7F,$AE
	.dc.b	$FE,$4C,$FF,$29,$7F,$96,$72,$7F,$AF,$DC,$D7,$FB,$CD,$CC,$EF,$D1
	.dc.b	$FF,$1E,$FD,$73,$8F,$FC,$7F,$5A,$BF,$F7,$D4,$E6,$FC,$1F,$11,$AF
	.dc.b	$6B,$6D,$8F,$24,$74,$F9,$2C,$5F,$F1,$5B,$47,$6F,$2F,$BF,$3D,$C3
	.dc.b	$D8,$E1,$FF,$7A,$7D,$73,$E4,$31,$8F,$F1,$F8,$C4,$FF,$BE,$7D,$CF
	.dc.b	$F1,$7C,$9F,$EC,$8F,$F2,$57,$26,$38,$3F,$DC,$8D,$9F,$EA,$54,$F9
	.dc.b	$DF,$F3,$0C,$E6,$C2,$75,$5E,$1F,$37,$7E,$6F,$06,$C7,$96,$3F,$CD
	.dc.b	$72,$D3,$37,$EB,$7B,$FA,$DE,$0F,$5A,$E1,$ED,$70,$F8,$B4,$FF,$7E
	.dc.b	$7D,$73,$EB,$33,$8F,$F4,$38,$C5,$6C,$FE,$DC,$BC,$EF,$F4,$2C,$FB
	.dc.b	$23,$13,$FE,$9C,$8C,$4E,$94,$E9,$D6,$CF,$7F,$4D,$BE,$53,$F4,$E7
	.dc.b	$26,$BD,$59,$3A,$AF,$EB,$15,$D2,$7C,$1F,$59,$FC,$5E,$3F,$D3,$6D
	.dc.b	$AF,$F5,$8B,$63,$FC,$14,$D7,$EB,$BF,$C6,$A3,$7E,$01,$FF,$15,$CE
	.dc.b	$A3,$0F,$66,$C7,$6F,$8D,$FD,$CC,$FC,$5A,$FF,$49,$F8,$1E,$E7,$FD
	.dc.b	$4B,$EB,$5A,$DD,$79,$BE,$07,$FC,$3D,$85,$8F,$08,$BF,$F1,$8B,$86
	.dc.b	$BF,$EB,$AF,$9F,$E7,$3F,$ED,$06,$C7,$F8,$A5,$AF,$E5,$9A,$F5,$F3
	.dc.b	$7C,$2F,$C6,$6B,$F1,$9A,$EC,$F1,$7C,$5E,$2C,$D2,$47,$AA,$9E,$07
	.dc.b	$26,$C6,$63,$07,$3F,$36,$47,$BD,$AF,$DE,$F3,$7C,$04,$7E,$23,$5D
	.dc.b	$18,$21,$F8,$6F,$13,$F2,$2F,$CD,$6E,$3F,$C7,$E3,$EC,$46,$3C,$19
	.dc.b	$1F,$6D,$AF,$E8,$F1,$7D,$77,$46,$C7,$E7,$9A,$EE,$35,$E5,$C8,$D3
	.dc.b	$5E,$9E,$17,$32,$3C,$DE,$DE,$2C,$8E,$CD,$7F,$6D,$F5,$CF,$98,$F7
	.dc.b	$B6,$3F,$62,$D7,$71,$F0,$BE,$16,$47,$E6,$5A,$FE,$77,$D0,$F8,$E4
	.dc.b	$7F,$3C,$FC,$CF,$98,$FE,$AD,$1F,$BE,$6B,$FC,$25,$AF,$F9,$F4,$38
	.dc.b	$7F,$D0,$0B,$DD,$FF,$41,$78,$5F,$B2,$9D,$F9,$F8,$E8,$DB,$63,$F9
	.dc.b	$9D,$F1,$7F,$8F,$18,$BD,$3E,$06,$C6,$E3,$FC,$D0,$BD,$DD,$F2,$3E
	.dc.b	$9E,$92,$BA,$9F,$B7,$F8,$AD,$91,$CC,$FF,$B1,$9D,$F2,$7F,$9F,$9E
	.dc.b	$3F,$5C,$FA,$5F,$2B,$63,$31,$8E,$49,$1C,$1A,$FF,$9F,$E9,$F0,$9C
	.dc.b	$BE,$56,$4F,$67,$AF,$AB,$23,$C2,$D7,$97,$C8,$F8,$BE,$1E,$47,$99
	.dc.b	$F2,$DC,$8F,$4B,$5F,$17,$A9,$EA,$E8,$91,$EE,$7D,$BF,$15,$23,$E0
	.dc.b	$B5,$F5,$7A,$5F,$5B,$EB,$9F,$2B,$49,$3F,$42,$3F,$BE,$37,$CE,$E0
	.dc.b	$47,$E7,$FB,$1F,$80,$FA,$EF,$C9,$F6,$FE,$2D,$BD,$BF,$EC,$F6,$D7
	.dc.b	$F7,$C7,$F0,$93,$BE,$CE,$44,$9F,$C0,$FD,$57,$99,$FC,$1D,$3E,$A3
	.dc.b	$C0,$48,$CF,$C5,$F8,$C8,$E0,$FC,$5F,$D7,$33,$F4,$7A,$92,$7A,$BE
	.dc.b	$BF,$80,$91,$E0,$6B,$FE,$C1,$C5,$F1,$5F,$04,$91,$E6,$7F,$47,$F5
	.dc.b	$CF,$57,$F4,$BF,$5D,$FA,$FE,$0F,$D8,$37,$87,$FD,$E8,$DC,$3F,$C4
	.dc.b	$2F,$FB,$8D,$FD,$1B,$5F,$F6,$93,$67,$9F,$C5,$7E,$CB,$EB,$BF,$DB
	.dc.b	$15,$C3,$D7,$F0,$91,$E7,$F1,$3E,$42,$38,$3E,$8F,$97,$E8,$FA,$D9
	.dc.b	$1E,$27,$D7,$EB,$48,$F2,$B5,$F2,$FA,$DF,$00,$8F,$A9,$FE,$0F,$E6
	.dc.b	$3B,$FE,$2F,$A1,$2B,$DE,$FC,$8F,$42,$47,$C2,$6B,$FE,$F3,$EC,$FC
	.dc.b	$23,$F1,$BE,$67,$E7,$91,$F9,$3F,$13,$F4,$88,$E0,$FE,$1F,$F5,$CD
	.dc.b	$7F,$8D,$FD,$07,$FA,$42,$7F,$89,$B5,$72,$7F,$29,$8C,$37,$CF,$E2
	.dc.b	$49,$E2,$F4,$F3,$E4,$74,$6B,$FE,$9C,$F3,$7C,$3F,$4B,$23,$C2,$F9
	.dc.b	$3E,$92,$47,$91,$AF,$FA,$93,$CC,$F4,$F9,$D2,$3D,$2F,$B3,$CE,$91
	.dc.b	$EC,$6B,$FE,$A1,$F2,$3E,$0F,$9B,$23,$E0,$BF,$13,$C7,$91,$F1,$1A
	.dc.b	$FC,$4F,$07,$FC,$F2,$B8,$1F,$35,$F3,$09,$1F,$31,$FE,$93,$E9,$6A
	.dc.b	$3F,$A0,$FD,$77,$FB,$1D,$BE,$07,$F4,$E8,$C4,$77,$3F,$D0,$EB,$E0
	.dc.b	$75,$EB,$C9,$E6,$F1,$EA,$FE,$AF,$F5,$DF,$E2,$89,$E4,$1C,$74,$7F
	.dc.b	$65,$6E,$BE,$C5,$EE,$72,$E3,$97,$FD,$62,$7F,$A9,$3D,$1F,$E9,$2F
	.dc.b	$F5,$CF,$AE,$FE,$5A,$F8,$1B,$87,$F7,$36,$E1,$F2,$97,$C7,$FF,$6E
	.dc.b	$5F,$33,$49,$7D,$7E,$27,$FB,$9F,$B6,$F8,$9A,$49,$FE,$DB,$1F,$55
	.dc.b	$BE,$2F,$02,$3A,$F1,$7F,$BE,$FD,$72,$A9,$27,$FB,$BC,$3C,$0D,$F7
	.dc.b	$C9,$CB,$CB,$E1,$F9,$F9,$12,$78,$BF,$E0,$AF,$AE,$FF,$0D,$71,$7F
	.dc.b	$85,$57,$C8,$64,$95,$FE,$06,$87,$C8,$7F,$84,$BE,$BB,$FD,$C3,$3F
	.dc.b	$E0,$A6,$BF,$79,$B7,$FE,$21,$5B,$FF,$86,$BD,$1F,$F7,$15,$7F,$84
	.dc.b	$9A,$FE,$19,$B5,$FE,$29,$7E,$24,$6F,$C3,$E4,$47,$97,$87,$D7,$E0
	.dc.b	$64,$75,$7C,$1A,$FF,$8A,$BE,$BB,$FC,$4D,$3F,$E2,$26,$BF,$96,$6E
	.dc.b	$3F,$E3,$97,$E2,$C7,$F8,$A3,$EB,$BF,$A5,$26,$93,$D6,$FF,$1B,$BF
	.dc.b	$E7,$58,$FD,$0B,$6B,$FC,$88,$DE,$CF,$DD,$BF,$21,$BA,$F2,$25,$78
	.dc.b	$79,$FA,$91,$C1,$E5,$F9,$9F,$0F,$85,$23,$D0,$F9,$3C,$29,$1E,$B6
	.dc.b	$BE,$6F,$47,$FC,$9F,$F5,$DF,$8F,$8F,$F2,$3B,$5F,$DD,$B7,$4F,$F2
	.dc.b	$F2,$F3,$B8,$92,$78,$BF,$C4,$B8,$7F,$CB,$7F,$39,$FE,$11,$9F,$E0
	.dc.b	$AD,$BF,$F3,$43,$76,$FE,$3E,$FE,$E2,$41,$C3,$7E,$13,$A9,$FC,$83
	.dc.b	$8F,$FC,$D3,$F5,$BF,$E5,$59,$FE,$26,$DC,$3F,$CE,$CD,$C3,$F9,$6B
	.dc.b	$F9,$26,$FA,$5C,$89,$5F,$E6,$A9,$FC,$AB,$D3,$C1,$EE,$47,$FC,$DF
	.dc.b	$F1,$9F,$B0,$BE,$BE,$0F,$90,$93,$CF,$6F,$39,$1C,$1D,$FC,$1F,$EC
	.dc.b	$AB,$EB,$BE,$13,$F1,$32,$4E,$A3,$FD,$31,$18,$D3,$FD,$35,$1B,$9D
	.dc.b	$3F,$01,$AF,$3F,$09,$F5,$DC,$21,$7F,$EE,$68,$C3,$63,$AB,$5E,$1F
	.dc.b	$51,$FE,$DA,$4D,$23,$D7,$E0,$5A,$FF,$A5,$A3,$FC,$D6,$D8,$CC,$7F
	.dc.b	$4C,$32,$5F,$8B,$CE,$7F,$EB,$1F,$5F,$FB,$35,$F3,$9C,$BF,$DA,$A3
	.dc.b	$35,$CF,$93,$2F,$F6,$EF,$36,$BF,$90,$8A,$FF,$48,$AF,$34,$E3,$E7
	.dc.b	$32,$D8,$F5,$47,$2B,$7C,$BE,$6F,$FB,$2C,$62,$BF,$D6,$B9,$7F,$43
	.dc.b	$6F,$1F,$8A,$31,$5F,$DB,$1F,$E8,$93,$A3,$FD,$7B,$4E,$FE,$17,$E1
	.dc.b	$7D,$7F,$5D,$FE,$69,$AF,$F6,$1B,$F2,$AF,$FD,$B0,$E7,$0F,$EF,$2E
	.dc.b	$67,$FC,$22,$D8,$FD,$9E,$0E,$EE,$BB,$7B,$DE,$FF,$25,$E0,$F3,$2E
	.dc.b	$3E,$07,$0F,$8B,$CB,$B8,$F8,$1F,$3F,$C6,$FA,$3D,$1B,$5E,$87,$0F
	.dc.b	$AB,$D1,$B7,$E8,$7D,$FF,$19,$F0,$3E,$32,$D7,$C0,$7E,$EF,$CC,$FF
	.dc.b	$87,$63,$FD,$CA,$BE,$67,$3E,$D6,$9D,$EF,$C3,$C9,$B7,$C1,$FC,$3F
	.dc.b	$D7,$7F,$86,$A3,$FD,$DE,$BE,$BB,$3E,$E6,$9D,$F2,$F0,$F9,$4B,$7C
	.dc.b	$1F,$C8,$FD,$77,$E3,$F8,$7F,$C5,$6B,$B7,$3D,$DA,$77,$C7,$C3,$8E
	.dc.b	$E3,$FF,$00,$2D,$1F,$9C,$FA,$E7,$A1,$78,$70,$DB,$FE,$5F,$DB,$FB
	.dc.b	$0D,$3F,$6B,$E3,$7E,$92,$7D,$D1,$FE,$3E,$51,$9E,$1F,$D4,$FD,$77
	.dc.b	$F3,$89,$E6,$7F,$96,$9F,$C7,$5F,$57,$57,$FE,$48,$73,$3B,$F0,$F9
	.dc.b	$F0,$74,$6C,$74,$AF,$E8,$B5,$FE,$3A,$8C,$42,$1B,$F9,$3A,$BC,$AF
	.dc.b	$EF,$BC,$57,$B3,$7E,$0B,$0D,$C7,$9C,$FF,$87,$1B,$E2,$71,$F5,$79
	.dc.b	$3F,$C2,$3E,$BB,$F6,$FB,$FC,$97,$F6,$FA,$FF,$99,$1C,$DF,$F9,$F1
	.dc.b	$CC,$EF,$DA,$E5,$F0,$BC,$DB,$5E,$CA,$FF,$5F,$3E,$EF,$2F,$E4,$DF
	.dc.b	$7F,$53,$F8,$CC,$62,$7F,$D1,$2F,$F9,$01,$E1,$E3,$6B,$C7,$DB,$FC
	.dc.b	$83,$EB,$9E,$DD,$F8,$1C,$36,$FE,$6F,$D7,$7E,$75,$BF,$0B,$FC,$A2
	.dc.b	$31,$C1,$FE,$82,$51,$93,$5E,$25,$AE,$6E,$1F,$B3,$F5,$DF,$98,$72
	.dc.b	$63,$C2,$ED,$F0,$3D,$F8,$EC,$F9,$1C,$3C,$BC,$DB,$9F,$FA,$53,$04
	.dc.b	$7F,$3E,$FA,$EF,$8D,$3F,$E0,$E7,$F2,$2D,$35,$C5,$2F,$CB,$F3,$16
	.dc.b	$7A,$3C,$FC,$1E,$9E,$45,$AF,$C9,$E1,$FB,$96,$C7,$87,$DE,$DC,$69
	.dc.b	$C3,$F0,$BE,$B6,$E3,$E9,$7D,$1F,$39,$E9,$F5,$2D,$76,$70,$FC,$AF
	.dc.b	$A6,$B7,$F0,$5F,$81,$EA,$FC,$1F,$1A,$D7,$99,$C3,$FA,$AF,$AE,$FE
	.dc.b	$EA,$E4,$C1,$F2,$78,$7E,$57,$05,$8F,$7B,$87,$E0,$E9,$72,$FF,$62
	.dc.b	$EE,$8F,$A4,$7F,$70,$FA,$EF,$D9,$D7,$03,$F7,$6E,$2B,$FD,$9B,$F5
	.dc.b	$DF,$96,$9F,$DF,$BF,$EB,$56,$F5,$FF,$7A,$8C,$4F,$0E,$CF,$EF,$BC
	.dc.b	$DF,$E4,$BB,$E3,$18,$E3,$E1,$E4,$71,$F5,$ED,$76,$70,$FF,$0E,$E9
	.dc.b	$F2,$78,$7F,$C9,$F3,$FE,$99,$9F,$FB,$8A,$73,$5F,$EF,$A9,$7C,$D7
	.dc.b	$97,$E6,$BC,$CD,$9F,$5C,$7A,$7E,$A3,$18,$7C,$07,$13,$E1,$CE,$CF
	.dc.b	$77,$E0,$79,$3F,$07,$E2,$AD,$7C,$67,$0F,$F2,$6F,$AE,$7D,$3E,$30
	.dc.b	$FF,$7C,$C6,$27,$E6,$FB,$9F,$E2,$4E,$27,$BD,$F3,$63,$0C,$9F,$E6
	.dc.b	$3D,$4F,$9E,$C6,$2E,$3F,$E9,$93,$9A,$FE,$6F,$F5,$DF,$C9,$2B,$FC
	.dc.b	$FE,$FF,$24,$8F,$F4,$0C,$23,$4D,$F8,$3F,$F1,$BC,$6E,$3F,$D0,$FF
	.dc.b	$5C,$E2,$CD,$7F,$8F,$23,$17,$E0,$F6,$B8,$FB,$5C,$9B,$1E,$48,$E1
	.dc.b	$F5,$78,$C7,$D5,$3E,$4B,$4F,$8B,$CB,$F1,$F5,$5A,$F6,$B8,$7F,$AA
	.dc.b	$FD,$73,$E3,$F1,$8F,$43,$FD,$3F,$19,$9F,$EB,$7E,$EF,$F8,$C6,$4C
	.dc.b	$8C,$FC,$37,$F9,$62,$31,$3C,$CD,$BF,$E5,$CA,$7C,$EE,$E3,$0F,$1F
	.dc.b	$3F,$97,$9F,$71,$F0,$B8,$7D,$3A,$5C,$7B,$62,$8F,$17,$AF,$E3,$F5
	.dc.b	$ED,$FA,$9C,$3F,$DD,$BE,$B5,$F5,$D8,$C7,$D1,$18,$9F,$EF,$7F,$5A
	.dc.b	$F8,$7F,$F0,$A5,$7E,$12,$31,$3F,$E9,$76,$C7,$FA,$81,$AF,$E2,$8F
	.dc.b	$93,$EB,$F2,$7F,$CF,$CD,$73,$3F,$EA,$1E,$0F,$EA,$51,$FE,$AA,$7F
	.dc.b	$A4,$C7,$F8,$62,$3F,$D0,$DD,$7F,$E9,$68,$ED,$3A,$9F,$3B,$63,$C5
	.dc.b	$F5,$BF,$C5,$1B,$E0,$F1,$AF,$CF,$CE,$5B,$19,$7F,$DC,$71,$FE,$8C
	.dc.b	$F0,$3F,$D6,$CD,$8F,$F5,$FA,$FA,$07,$B3,$AF,$8E,$FD,$CD,$6B,$E1
	.dc.b	$3F,$5B,$97,$91,$B1,$E2,$6B,$E0,$6A,$C1,$E5,$61,$B1,$F1,$4D,$46
	.dc.b	$4D,$79,$D2,$35,$28,$C4,$7D,$E8,$FF,$3A,$3A,$F8,$DE,$C4,$78,$FD
	.dc.b	$1F,$84,$8E,$0E,$BE,$7F,$94,$8E,$0F,$0F,$9F,$E9,$23,$83,$CB,$E3
	.dc.b	$FF,$B8,$DF,$80,$DC,$3F,$87,$47,$D2,$8C,$11,$C1,$0F,$AB,$E1,$F1
	.dc.b	$23,$F6,$9E,$DC,$CF,$0E,$92,$7F,$3C,$F8,$FE,$87,$F9,$CC,$7F,$BD
	.dc.b	$DA,$FF,$C0,$06,$BF,$A1,$6B,$FC,$B1,$FE,$24,$F1,$F1,$3A,$11,$E9
	.dc.b	$F4,$7D,$C8,$E0,$FD,$87,$CC,$FA,$08,$E0,$FC,$AF,$C4,$7C,$72,$38
	.dc.b	$3F,$33,$E9,$FC,$B2,$38,$21,$FC,$6F,$5F,$F6,$F3,$FE,$7E,$6B,$9F
	.dc.b	$4F,$FA,$3F,$9F,$FE,$99,$F9,$87,$C1,$63,$A7,$FA,$5C,$7F,$55,$BF
	.dc.b	$F4,$3B,$5F,$E5,$0F,$07,$33,$F8,$18,$C4,$7F,$0B,$0E,$1F,$E1,$CD
	.dc.b	$7B,$FC,$17,$12,$3C,$4E,$2F,$E2,$FE,$9B,$CD,$1F,$BE,$6B,$FA,$F6
	.dc.b	$E5,$FE,$BA,$8F,$85,$F1,$D8,$23,$FE,$A2,$E1,$FF,$38,$BF,$EA,$4F
	.dc.b	$AD,$70,$F4,$6C,$66,$31,$C5,$23,$83,$5F,$E9,$94,$F8,$6E,$5F,$37
	.dc.b	$27,$B3,$D7,$D6,$91,$E1,$6B,$CB,$C9,$F1,$7C,$44,$8F,$33,$E5,$B9
	.dc.b	$1E,$96,$BE,$2F,$4F,$D5,$D1,$23,$DC,$FB,$7E,$2E,$47,$C1,$6B,$EA
	.dc.b	$F4,$7E,$6F,$D6,$BD,$6A,$49,$FF,$65,$C7,$F7,$06,$F9,$BC,$08,$FC
	.dc.b	$FE,$EF,$AF,$F5,$9F,$8B,$EC,$FF,$75,$37,$B3,$FC,$36,$DA,$07,$F1
	.dc.b	$B3,$C3,$B3,$99,$27,$FD,$BB,$F5,$3E,$77,$FD,$BF,$4F,$A8,$F0,$92
	.dc.b	$33,$F1,$9E,$42,$38,$3F,$DC,$FF,$5B,$FC,$3D,$E7,$EA,$7A,$88,$F6
	.dc.b	$F5,$3D,$A4,$70,$78,$BE,$2F,$E0,$91,$C1,$FE,$F4,$FA,$DF,$EA,$6F
	.dc.b	$FB,$DB,$EB,$5B,$FE,$79,$B1,$98,$C7,$C5,$48,$E0,$D7,$FC,$3D,$F5
	.dc.b	$AC,$B9,$7A,$92,$7B,$3D,$7D,$49,$1E,$16,$BF,$E2,$7E,$1F,$A8,$B2
	.dc.b	$7A,$5E,$52,$47,$A7,$D2,$F4,$11,$C1,$ED,$F4,$BD,$84,$70,$7E,$03
	.dc.b	$9B,$DE,$BC,$A7,$6F,$C3,$F3,$E4,$FC,$77,$E3,$79,$F2,$3E,$53,$5F
	.dc.b	$57,$47,$CB,$F9,$99,$1F,$3D,$F9,$DF,$0F,$23,$E9,$35,$FF,$24,$7D
	.dc.b	$6F,$C0,$7C,$DC,$DF,$E7,$2F,$6B,$FA,$6E,$BE,$86,$C6,$7E,$2B,$F8
	.dc.b	$BC,$7F,$0C,$AF,$F2,$93,$6B,$87,$9D,$A3,$7E,$74,$AE,$E7,$E9,$2B
	.dc.b	$FA,$AB,$CA,$3F,$91,$FC,$53,$CD,$EF,$F4,$92,$B8,$9F,$E2,$17,$C7
	.dc.b	$F4,$72,$33,$82,$31,$1F,$EA,$0F,$33,$D3,$E7,$48,$DB,$EC,$F3,$A4
	.dc.b	$72,$6B,$FE,$6F,$F1,$3F,$64,$F6,$F3,$7E,$19,$1F,$0F,$95,$F1,$88
	.dc.b	$E0,$F2,$FC,$CF,$CA,$23,$83,$F9,$E3,$FD,$13,$EB,$7F,$C5,$F1,$FE
	.dc.b	$7F,$6F,$17,$E3,$1B,$C9,$FE,$AA,$7F,$C5,$D3,$F1,$9F,$E9,$4F,$1F
	.dc.b	$57,$A1,$D7,$A3,$FA,$8F,$D6,$BE,$36,$92,$BE,$7C,$3E,$37,$7C,$12
	.dc.b	$72,$F2,$E0,$FE,$B7,$F5,$BF,$CA,$E7,$FD,$2C,$DD,$7F,$46,$DD,$7F
	.dc.b	$D8,$2B,$E3,$78,$FC,$C4,$BF,$BF,$28,$EB,$F3,$DF,$DA,$3D,$A7,$C4
	.dc.b	$D2,$5F,$E1,$A3,$FC,$CC,$DF,$13,$81,$1D,$78,$9F,$DC,$BE,$B5,$54
	.dc.b	$93,$F9,$08,$FC,$A3,$7A,$F8,$11,$D7,$1F,$3F,$1A,$47,$17,$FB,$FF
	.dc.b	$D6,$FF,$A5,$F8,$7F,$82,$AF,$8F,$C9,$2B,$F3,$F0,$F8,$FF,$F0,$3F
	.dc.b	$D6,$FF,$B5,$27,$F4,$6D,$7F,$D8,$8D,$BF,$E2,$2B,$6F,$F0,$7F,$A5
	.dc.b	$FB,$09,$FE,$2A,$DE,$5F,$F7,$9B,$6F,$FA,$CB,$F1,$1F,$D8,$BF,$EC
	.dc.b	$36,$F7,$F4,$3A,$F7,$F6,$F7,$E4,$F0,$7C,$3B,$7F,$88,$3E,$B5,$F0
	.dc.b	$B4,$95,$FB,$E8,$F8,$8F,$F0,$0F,$AD,$FE,$57,$A7,$F2,$C6,$E9,$FD
	.dc.b	$BA,$51,$87,$FB,$8A,$F6,$7F,$C5,$EF,$F9,$F2,$3F,$DD,$4D,$AE,$9F
	.dc.b	$17,$AF,$8B,$DF,$04,$BF,$13,$CB,$A6,$47,$95,$AF,$4F,$8A,$EF,$E2
	.dc.b	$C8,$F5,$3E,$2F,$16,$47,$B5,$AF,$93,$D2,$FF,$22,$FD,$6F,$E4,$23
	.dc.b	$FE,$00,$37,$4F,$E9,$AD,$DF,$E1,$2F,$37,$89,$2B,$F9,$4C,$65,$FF
	.dc.b	$25,$72,$7F,$94,$BE,$6B,$FC,$21,$0C,$3F,$AC,$C6,$1B,$F0,$FB,$F8
	.dc.b	$69,$5F,$CD,$2B,$F7,$8F,$F8,$81,$AB,$E2,$7F,$B0,$BF,$CE,$BE,$AF
	.dc.b	$FA,$46,$9F,$65,$B1,$FE,$28,$6A,$7F,$28,$DF,$57,$FC,$D3,$18,$27
	.dc.b	$8B,$82,$4F,$77,$FC,$DF,$F1,$BF,$B1,$BF,$07,$AB,$C8,$95,$E5,$E0
	.dc.b	$EA,$47,$07,$A7,$C6,$F8,$CB,$EB,$5A,$FA,$E7,$33,$F9,$53,$D4,$E1
	.dc.b	$D7,$07,$FA,$1F,$4F,$EB,$0D,$F1,$DF,$5B,$49,$5F,$E9,$5A,$FE,$71
	.dc.b	$1F,$ED,$98,$7D,$3D,$23,$FE,$DD,$7F,$DB,$DF,$2B,$FE,$32,$C3,$FC
	.dc.b	$2B,$3F,$E6,$5D,$BF,$B6,$A9,$D4,$7A,$79,$2C,$FE,$BD,$AF,$FA,$1F
	.dc.b	$D9,$AF,$33,$F1,$53,$FB,$5F,$49,$3A,$E6,$DF,$A5,$6B,$C2,$E1,$F1
	.dc.b	$FC,$D5,$C3,$99,$FB,$F7,$B9,$E2,$C1,$F2,$D8,$AF,$F0,$3C,$61,$FF
	.dc.b	$04,$E9,$FC,$F9,$BC,$BE,$A8,$C6,$1D,$8F,$D1,$46,$4E,$DE,$05,$AE
	.dc.b	$AF,$C2,$FA,$DF,$91,$1F,$E1,$67,$E1,$5F,$F9,$09,$CD,$F5,$73,$3F
	.dc.b	$C6,$5B,$1F,$E8,$8C,$1D,$FD,$77,$F7,$3D,$FC,$8F,$07,$A1,$71,$F0
	.dc.b	$38,$7C,$5E,$7D,$C7,$C0,$F9,$FE,$3B,$D1,$EA,$DA,$F4,$38,$7D,$5E
	.dc.b	$AD,$BF,$43,$EF,$F8,$AF,$81,$EE,$DA,$F8,$0F,$E1,$3E,$61,$F3,$17
	.dc.b	$E6,$30,$DA,$FF,$1D,$CF,$B9,$BE,$57,$1E,$0E,$4F,$E3,$BE,$B7,$FC
	.dc.b	$25,$3F,$E4,$25,$F5,$B9,$F7,$F4,$EF,$97,$87,$CB,$DB,$E0,$FE,$63
	.dc.b	$EB,$7F,$C2,$11,$FE,$4C,$5D,$B9,$EE,$D3,$BE,$2E,$1C,$56,$F8,$3F
	.dc.b	$A2,$FA,$D7,$A3,$78,$30,$DA,$FF,$2C,$9F,$E2,$F2,$ED,$FA,$A7,$1C
	.dc.b	$BF,$91,$7D,$EF,$E4,$FE,$B7,$FD,$97,$1F,$E6,$27,$F1,$7D,$DF,$E8
	.dc.b	$E7,$33,$BF,$13,$F6,$B0,$F0,$EE,$F8,$6E,$1C,$FA,$B7,$BF,$06,$C7
	.dc.b	$F9,$C3,$C5,$7B,$BF,$C0,$17,$C9,$D8,$C3,$9F,$5B,$5E,$2D,$BE,$4F
	.dc.b	$F0,$8F,$AD,$FD,$7F,$07,$FA,$99,$FF,$40,$4F,$F9,$F9,$CB,$FD,$F5
	.dc.b	$CF,$07,$ED,$57,$BD,$CF,$E2,$3A,$36,$B9,$7B,$FF,$C5,$B8,$FE,$2E
	.dc.b	$3D,$E3,$18,$E7,$C6,$D7,$8D,$70,$FF,$47,$E1,$FE,$31,$7F,$D2,$3F
	.dc.b	$5B,$FE,$0B,$6F,$C2,$FF,$22,$8C,$61,$FC,$25,$FF,$2A,$CF,$FA,$63
	.dc.b	$EB,$7F,$2E,$DF,$88,$FE,$53,$18,$AF,$E2,$0A,$32,$6B,$C2,$B5,$CD
	.dc.b	$C3,$FE,$A4,$FA,$8F,$F7,$33,$93,$1E,$07,$6F,$85,$EF,$C3,$67,$C8
	.dc.b	$E1,$E5,$E5,$DC,$FF,$91,$60,$8F,$E7,$FF,$57,$F2,$AB,$FC,$60,$FE
	.dc.b	$49,$B5,$AD,$E4,$79,$7E,$6E,$CF,$47,$9F,$7F,$4F,$16,$D7,$FA,$FB
	.dc.b	$0F,$D9,$B6,$3C,$3E,$CE,$E3,$4E,$1F,$85,$F5,$77,$1F,$4B,$E8,$F9
	.dc.b	$FF,$4F,$A5,$6B,$B3,$87,$E5,$7D,$6D,$BF,$82,$FC,$0F,$4F,$E0,$F8
	.dc.b	$96,$BC,$CE,$1F,$F6,$AF,$D6,$9C,$5F,$F4,$B8,$C4,$7B,$78,$7E,$5F
	.dc.b	$07,$CC,$6C,$7C,$18,$F9,$FA,$62,$FF,$AA,$6E,$8C,$D7,$FB,$93,$EB
	.dc.b	$7F,$6A,$FC,$98,$FE,$EB,$18,$AF,$EB,$9F,$5B,$F9,$6F,$03,$CD,$01
	.dc.b	$1F,$98,$77,$F1,$9F,$DF,$A3,$13,$C2,$77,$DD,$FE,$02,$E7,$FF,$10
	.dc.b	$E5,$FE,$F6,$BD,$BF,$E7,$58,$CD,$78,$BC,$6E,$9E,$B7,$56,$D7,$18
	.dc.b	$FF,$06,$F5,$7A,$72,$F0,$63,$FD,$32,$BD,$11,$E3,$9D,$47,$FA,$0E
	.dc.b	$5F,$39,$E9,$F9,$CF,$53,$67,$CD,$1E,$DF,$8D,$C6,$7F,$19,$C4,$F6
	.dc.b	$3F,$D1,$D8,$7F,$20,$7D,$D1,$E6,$CF,$E3,$79,$3F,$1F,$E3,$2D,$7C
	.dc.b	$37,$0F,$3C,$3E,$87,$CB,$BE,$67,$31,$FE,$11,$8C,$4F,$CB,$F6,$FF
	.dc.b	$C5,$FC,$4F,$73,$EA,$46,$19,$3D,$6F,$F7,$1F,$5B,$87,$FC,$39,$D4
	.dc.b	$8F,$F1,$CF,$D6,$FF,$2E,$9F,$C1,$3F,$CB,$B9,$3E,$F3,$98,$39,$BF
	.dc.b	$F5,$83,$63,$F1,$5F,$5A,$E1,$EB,$F0,$7F,$E4,$68,$C5,$F9,$6F,$FA
	.dc.b	$FE,$1D,$BD,$9F,$1F,$C4,$79,$19,$5E,$BB,$7C,$96,$33,$FF,$1B,$46
	.dc.b	$27,$A7,$97,$EC,$E7,$F6,$B6,$3C,$11,$FE,$58,$FA,$D7,$C8,$67,$0F
	.dc.b	$F2,$04,$62,$BE,$57,$B9,$FE,$34,$9F,$ED,$4F,$F9,$80,$C1,$FD,$B6
	.dc.b	$36,$F5,$F9,$CD,$36,$39,$F9,$DC,$7C,$EE,$4D,$8E,$F1,$E5,$CF,$18
	.dc.b	$79,$1F,$07,$AD,$E1,$F5,$AD,$F9,$9C,$3F,$81,$FA,$C7,$D6,$E3,$0F
	.dc.b	$F2,$E4,$62,$7D,$BE,$FF,$FA,$0A,$7F,$81,$68,$8C,$47,$F8,$25,$B1
	.dc.b	$99,$FF,$4E,$4F,$0E,$AF,$E1,$71,$FE,$9F,$9F,$E9,$0F,$F8,$52,$74
	.dc.b	$D7,$FC,$29,$3C,$1F,$F7,$D4,$BC,$CC,$47,$91,$AF,$F8,$6D,$B1,$BA
	.dc.b	$E9,$F5,$9F,$E8,$2E,$DF,$2D,$A3,$0F,$A1,$FF,$49,$FA,$BA,$73,$1F
	.dc.b	$EB,$77,$FC,$3E,$F4,$F7,$FF,$93,$3F,$EA,$88,$CB,$63,$37,$C7,$C6
	.dc.b	$B3,$C2,$BF,$C4,$EF,$9A,$32,$FF,$31,$8C,$BF,$EB,$4F,$77,$F8,$CB
	.dc.b	$F1,$DA,$E7,$C9,$E2,$A3,$37,$E9,$F8,$9C,$D9,$A4,$AF,$8F,$F4,$9A
	.dc.b	$23,$83,$EB,$7C,$B3,$C3,$EE,$D8,$CC,$60,$EF,$E6,$48,$FA,$6D,$7E
	.dc.b	$47,$8B,$F5,$1F,$A8,$DC,$7F,$84,$47,$D3,$8C,$78,$32,$3E,$B3,$5F
	.dc.b	$B4,$D7,$EC,$35,$FA,$FF,$5B,$FE,$46,$7F,$06,$D7,$F3,$2D,$7F,$DE
	.dc.b	$CD,$7E,$FE,$AF,$83,$F4,$36,$3F,$1A,$D7,$F4,$6D,$7F,$E0,$06,$7F
	.dc.b	$C3,$9F,$C9,$B5,$FD,$4B,$5D,$F8,$3F,$16,$1F,$45,$49,$3C,$9D,$7C
	.dc.b	$C4,$8E,$AD,$7F,$3D,$87,$CD,$23,$F9,$F7,$E8,$7A,$12,$3F,$44,$D7
	.dc.b	$F4,$CF,$E8,$3C,$9F,$F0,$06,$C8,$C7,$1F,$E7,$5B,$5F,$9B,$F4,$2B
	.dc.b	$FB,$17,$EF,$11,$FD,$83,$F8,$3F,$98,$7B,$D4,$93,$FB,$A8,$CB,$63
	.dc.b	$AC,$62,$3F,$DF,$0D,$C7,$FC,$2C,$BD,$ED,$F8,$9C,$19,$5F,$D9,$E5
	.dc.b	$3E,$8C,$DF,$4B,$49,$5F,$DB,$63,$FD,$7E,$DF,$8A,$E0,$47,$FB,$97
	.dc.b	$0F,$F9,$ED,$FE,$E9,$F5,$BF,$71,$BC,$5F,$1A,$31,$0D,$3F,$C5,$D1
	.dc.b	$F8,$76,$F1,$EC,$93,$DB,$E1,$39,$72,$48,$EE,$F5,$EA,$91,$E2,$6B
	.dc.b	$DF,$93,$F8,$EE,$6F,$8F,$F2,$7C,$34,$AF,$4B,$E7,$C6,$47,$B1,$AF
	.dc.b	$33,$FC,$99,$4F,$3B,$DD,$F3,$72,$7E,$1B,$F0,$7E,$2A,$47,$C6,$6B
	.dc.b	$CC,$53,$CE,$FC,$6F,$D6,$BE,$52,$92,$BF,$C4,$51,$FE,$E1,$6F,$95
	.dc.b	$C0,$8F,$D6,$F5,$3F,$2F,$F5,$BF,$E2,$58,$FF,$15,$35,$F1,$37,$93
	.dc.b	$FC,$C4,$FE,$3F,$87,$FA,$2B,$67,$FC,$D8,$D4,$DF,$AA,$F4,$47,$6F
	.dc.b	$A9,$F0,$91,$F0,$FC,$57,$90,$8E,$0F,$F1,$F7,$D6,$FF,$1B,$C7,$C7
	.dc.b	$F1,$DE,$A2,$4F,$AB,$D3,$F6,$91,$C1,$EE,$F4,$FE,$09,$1C,$1F,$E4
	.dc.b	$BF,$AD,$7A,$7F,$BC,$FA,$CF,$F4,$16,$3F,$E8,$36,$EF,$FD,$99,$B7
	.dc.b	$FE,$8D,$5F,$13,$C4,$91,$FE,$57,$8C,$BF,$E5,$9F,$AD,$69,$DF,$D6
	.dc.b	$93,$B7,$C5,$EB,$48,$E4,$D7,$F1,$1C,$5F,$5F,$2F,$27,$A9,$E9,$24
	.dc.b	$F8,$7D,$6F,$61,$1C,$1E,$5F,$57,$E0,$11,$C1,$FE,$6F,$F2,$9D,$DF
	.dc.b	$17,$D1,$93,$ED,$7E,$47,$A3,$23,$E0,$35,$F3,$F4,$FC,$DF,$97,$91
	.dc.b	$F1,$5F,$A1,$F0,$B2,$3E,$43,$5F,$D3,$7D,$6B,$5F,$E6,$3F,$3F,$FE
	.dc.b	$8D,$9F,$F4,$23,$5F,$D6,$37,$9F,$FD,$78,$FF,$12,$6F,$9D,$C8,$93
	.dc.b	$CB,$CD,$EA,$47,$07,$FA,$47,$E2,$7F,$C0,$CF,$6F,$A4,$F1,$91,$F1
	.dc.b	$FD,$1F,$98,$8E,$0E,$5E,$5F,$AB,$CD,$91,$EA,$7D,$BE,$6C,$8F,$6B
	.dc.b	$5F,$F4,$FF,$0F,$C2,$F2,$E4,$7C,$27,$E2,$F9,$12,$3E,$2B,$5F,$8B
	.dc.b	$EC,$FC,$A2,$3F,$29,$FE,$BB,$E6,$FF,$8C,$63,$FD,$61,$F5,$BF,$E3
	.dc.b	$AE,$1F,$F7,$33,$7B,$7F,$C9,$2D,$9F,$F7,$82,$F8,$FF,$EC,$EF,$F4
	.dc.b	$C8,$FE,$40,$DA,$E7,$EA,$EB,$A7,$FB,$6F,$D6,$BE,$2E,$93,$0F,$F6
	.dc.b	$14,$3E,$2F,$7C,$52,$72,$F2,$E2,$FE,$EF,$F5,$BF,$C9,$E7,$F9,$83
	.dc.b	$73,$FF,$80,$0D,$DB,$FC,$C6,$BE,$2B,$8F,$CB,$CB,$FF,$69,$4A,$3A
	.dc.b	$FC,$3F,$F8,$13,$DB,$7C,$85,$25,$FF,$B6,$61,$DF,$BF,$12,$4E,$5E
	.dc.b	$5E,$27,$F8,$3B,$EB,$57,$49,$5F,$EE,$38,$FC,$9E,$CD,$BF,$CF,$70
	.dc.b	$EE,$E4,$4A,$E1,$C9,$D7,$C6,$91,$CD,$FF,$0E,$FD,$6F,$FA,$12,$3F
	.dc.b	$AB,$3F,$E0,$A7,$FD,$E4,$D7,$3F,$03,$C3,$E3,$7F,$C4,$BF,$5B,$FE
	.dc.b	$DC,$9F,$F7,$C3,$5F,$EC,$0D,$D3,$D1,$01,$1F,$E5,$18,$CC,$3D,$0F
	.dc.b	$F0,$46,$BF,$E9,$86,$C7,$16,$BB,$DB,$FC,$6F,$1F,$DB,$63,$B5,$74
	.dc.b	$8C,$47,$F7,$36,$C6,$7C,$1E,$3E,$FC,$AE,$38,$78,$63,$AE,$BF,$E4
	.dc.b	$4F,$AD,$FF,$1A,$56,$8F,$63,$87,$FC,$00,$DC,$3D,$AF,$CB,$7D,$FF
	.dc.b	$5B,$FD,$3A,$4D,$27,$AF,$FE,$52,$7F,$D1,$B1,$FE,$0B,$6C,$F2,$F8
	.dc.b	$CD,$7C,$66,$F9,$E5,$F9,$4E,$AF,$3E,$79,$19,$C3,$FC,$C8,$D8,$F0
	.dc.b	$7C,$5F,$87,$C4,$93,$B7,$C9,$E2,$48,$E4,$D7,$CD,$E9,$7F,$9A,$BE
	.dc.b	$B7,$F3,$3C,$5F,$ED,$26,$CF,$E0,$DA,$97,$9D,$C4,$91,$EB,$8C,$BF
	.dc.b	$E5,$6E,$3F,$F3,$BF,$CF,$FF,$85,$23,$FC,$60,$DE,$1F,$C6,$B7,$0F
	.dc.b	$F7,$1B,$FC,$0E,$7F,$28,$D4,$D3,$1E,$5F,$F6,$68,$F9,$9F,$59,$FD
	.dc.b	$3E,$BF,$DD,$AD,$8F,$CF,$35,$3F,$98,$27,$A5,$FE,$8E,$71,$D5,$49
	.dc.b	$3D,$9F,$F4,$3B,$BF,$89,$8F,$89,$E5,$AC,$8F,$2B,$5F,$F2,$95,$7F
	.dc.b	$07,$CF,$F6,$5F,$5B,$EE,$9F,$3F,$85,$FE,$79,$7F,$82,$1E,$87,$0F
	.dc.b	$F0,$7A,$E4,$E9,$AE,$7C,$27,$D6,$F2,$8F,$83,$18,$78,$35,$E1,$1F
	.dc.b	$EE,$D5,$F4,$D4,$8F,$FB,$C5,$FF,$79,$7C,$17,$F0,$CC,$BF,$84,$CF
	.dc.b	$F9,$AF,$7F,$D0,$3F,$E4,$06,$FC,$5E,$DF,$E4,$0D,$8F,$E2,$9E,$77
	.dc.b	$F3,$89,$D9,$FA,$97,$11,$FE,$7B,$1F,$15,$C5,$FE,$39,$E7,$D7,$35
	.dc.b	$E2,$8C,$19,$F9,$7B,$1E,$67,$0F,$7F,$AD,$B5,$FE,$89,$C1,$1D,$F2
	.dc.b	$FE,$8F,$5F,$BD,$5C,$79,$E4,$D3,$27,$D5,$DD,$71,$E8,$FA,$7E,$B7
	.dc.b	$E3,$47,$FA,$5D,$F9,$4D,$DF,$5B,$CB,$F8,$9D,$B6,$BC,$CF,$3F,$93
	.dc.b	$E9,$E6,$5A,$FE,$25,$8F,$FB,$59,$B1,$E1,$F3,$37,$1A,$7C,$9F,$1B
	.dc.b	$E5,$F4,$EC,$F2,$70,$F4,$F4,$EE,$7E,$97,$D9,$F0,$FE,$DF,$8C,$B5
	.dc.b	$E2,$7E,$C7,$CB,$FF,$86,$63,$F9,$32,$F9,$7C,$FB,$BA,$77,$C1,$C3
	.dc.b	$C8,$B7,$C1,$FC,$17,$D6,$FF,$85,$E3,$F9,$82,$FA,$DC,$FB,$BA,$77
	.dc.b	$D9,$C3,$E5,$2D,$F0,$7F,$19,$F5,$BF,$E1,$58,$FE,$74,$BB,$F3,$E0
	.dc.b	$69,$DF,$17,$0E,$2B,$7C,$1F,$CB,$7D,$6B,$CF,$BC,$18,$6D,$7F,$47
	.dc.b	$3F,$C6,$85,$DB,$F3,$CE,$39,$7F,$DB,$8F,$BD,$FF,$6F,$7D,$6F,$F7
	.dc.b	$D8,$FE,$A0,$FE,$33,$C2,$FE,$FA,$E6,$77,$DF,$FA,$98,$FD,$3C,$72
	.dc.b	$DB,$1E,$26,$5D,$75,$6F,$7B,$97,$67,$F6,$5E,$2F,$F8,$26,$3F,$B0
	.dc.b	$2F,$90,$CF,$B7,$A7,$7E,$AF,$0F,$16,$DF,$07,$F7,$9F,$5B,$FA,$ED
	.dc.b	$7F,$DF,$4F,$EB,$A7,$FB,$63,$9E,$0F,$F0,$CB,$99,$DF,$B9,$CB,$E1
	.dc.b	$B9,$B6,$F8,$FB,$BF,$C2,$78,$FE,$26,$3C,$A3,$18,$67,$C6,$D7,$8D
	.dc.b	$70,$FE,$F9,$82,$3F,$89,$7D,$6F,$FB,$5A,$7F,$C0,$0F,$F9,$59,$BE
	.dc.b	$EE,$BE,$0F,$F8,$C7,$D6,$FE,$56,$7F,$C0,$CB,$E1,$B3,$E2,$69,$93
	.dc.b	$AF,$12,$EA,$DF,$FC,$73,$82,$7A,$9F,$C9,$7E,$A3,$FD,$C7,$7C,$4F
	.dc.b	$F0,$A3,$8B,$E5,$1F,$E4,$55,$4F,$8A,$F3,$F1,$7A,$1B,$5E,$88,$ED
	.dc.b	$E6,$62,$F9,$9F,$E4,$FC,$11,$FC,$DF,$EA,$FE,$7D,$F7,$7F,$2C,$DA
	.dc.b	$D7,$14,$8F,$3F,$CC,$D9,$EA,$30,$47,$5E,$FE,$DE,$3D,$BF,$F1,$96
	.dc.b	$3F,$B9,$6C,$78,$FD,$ED,$C6,$9C,$3F,$1B,$EA,$EE,$5E,$0C,$91,$EB
	.dc.b	$FA,$1F,$67,$A9,$6F,$BB,$87,$E7,$FD,$6D,$C3,$93,$F0,$FD,$4F,$89
	.dc.b	$E3,$5A,$F3,$B8,$7F,$C9,$5F,$5B,$FE,$91,$72,$60,$F9,$9C,$5F,$37
	.dc.b	$86,$C7,$C0,$70,$FC,$2D,$2E,$5F,$2B,$83,$D7,$39,$3F,$CB,$1F,$5B
	.dc.b	$FB,$B9,$D5,$7F,$6F,$8C,$74,$7F,$A2,$3E,$B7,$F4,$15,$F5,$1F,$D0
	.dc.b	$37,$B7,$FB,$B4,$62,$78,$76,$FE,$57,$93,$2D,$CB,$FC,$D1,$5F,$C2
	.dc.b	$5B,$EB,$72,$F1,$F9,$B2,$3F,$CD,$DD,$5E,$47,$F3,$55,$FD,$37,$3F
	.dc.b	$C3,$CE,$5F,$F5,$14,$3E,$6F,$CB,$F3,$7E,$66,$DF,$8A,$3D,$3F,$19
	.dc.b	$8B,$F7,$38,$9F,$0E,$56,$7B,$BF,$03,$C9,$F8,$3F,$15,$6B,$DA,$E1
	.dc.b	$FF,$43,$7C,$BB,$EB,$71,$87,$E7,$A3,$13,$F1,$3D,$BF,$F1,$07,$03
	.dc.b	$DD,$F9,$B1,$86,$4F,$FA,$47,$A5,$F5,$B8,$C7,$F5,$31,$89,$FF,$4B
	.dc.b	$7D,$6F,$F2,$B7,$F8,$13,$FC,$AF,$9B,$FD,$8A,$E6,$BF,$D9,$4E,$67
	.dc.b	$7F,$05,$FE,$33,$FA,$D7,$17,$F8,$D6,$B5,$7F,$ED,$19,$CD,$F8,$3E
	.dc.b	$1F,$5E,$D6,$DB,$1E,$48,$E9,$F2,$38,$BF,$E0,$36,$8E,$DE,$5F,$7E
	.dc.b	$AB,$87,$B1,$C3,$FE,$B3,$FA,$D7,$C7,$63,$1F,$E1,$71,$89,$FF,$5C
	.dc.b	$7B,$9F,$E3,$2E,$0F,$F7,$2B,$FD,$99,$BF,$0F,$FE,$51,$8C,$4F,$F2
	.dc.b	$96,$DF,$FB,$BE,$DF,$39,$B9,$FF,$2C,$46,$2B,$BF,$9F,$CF,$CF,$E8
	.dc.b	$D8,$F1,$C7,$0D,$71,$8F,$A1,$F1,$7A,$FE,$3F,$5E,$DF,$A9,$C3,$FE
	.dc.b	$D3,$FA,$C7,$D6,$E3,$0F,$E5,$71,$89,$FF,$6C,$7B,$7F,$C3,$E7,$FC
	.dc.b	$FA,$D8,$FE,$8D,$08,$C3,$FD,$25,$A8,$7D,$5F,$F0,$49,$E2,$D8,$FE
	.dc.b	$A0,$D7,$83,$5E,$3F,$1B,$F9,$F7,$FD,$4B,$1F,$E7,$4E,$EF,$F4,$F2
	.dc.b	$F9,$B7,$B1,$8B,$E1,$7E,$0F,$27,$FA,$F6,$36,$3F,$C4,$87,$82,$32
	.dc.b	$F9,$1A,$F9,$3C,$9A,$FB,$1B,$89,$A6,$3B,$FF,$D6,$53,$F0,$9B,$3F
	.dc.b	$A6,$8F,$4C,$63,$AE,$47,$B1,$AF,$FB,$13,$E5,$1F,$5D,$D1,$B1,$FE
	.dc.b	$BF,$6B,$B8,$D7,$91,$23,$4D,$7A,$76,$F3,$23,$CD,$ED,$E0,$48,$EC
	.dc.b	$D7,$FD,$AF,$F5,$CF,$97,$F7,$B6,$3F,$DA,$4D,$77,$1F,$0B,$E1,$24
	.dc.b	$7D,$A6,$BF,$79,$AF,$E2,$1A,$FE,$23,$CD,$F9,$64,$7F,$06,$D7,$FC
	.dc.b	$0F,$B2,$3F,$C3,$DE,$13,$8B,$95,$7E,$5E,$30,$FE,$4B,$CD,$DC,$FF
	.dc.b	$89,$7E,$BB,$FD,$A3,$3F,$EE,$D6,$BF,$EF,$86,$BF,$A6,$6B,$C7,$C0
	.dc.b	$E6,$FE,$96,$38,$DF,$82,$11,$8A,$E1,$1E,$56,$BD,$EB,$F1,$B2,$69
	.dc.b	$8E,$CE,$F3,$D1,$B3,$FC,$0A,$3A,$1E,$3F,$2F,$27,$42,$3E,$F7,$BF
	.dc.b	$BB,$A8,$E2,$7A,$23,$AC,$48,$F8,$CF,$06,$9F,$84,$F8,$4F,$C3,$91
	.dc.b	$C1,$FC,$38,$FE,$15,$F8,$2F,$F1,$0C,$FE,$37,$0A,$BD,$AD,$AF,$1D
	.dc.b	$6F,$4F,$F2,$E3,$63,$99,$F1,$1C,$27,$F8,$4B,$EC,$DD,$F4,$FF,$29
	.dc.b	$B1,$98,$C7,$C6,$48,$E0,$D7,$FC,$AF,$D1,$FE,$7D,$7E,$3F,$D7,$BE
	.dc.b	$9B,$EA,$B6,$33,$18,$E8,$91,$C1,$AF,$4F,$86,$E3,$F2,$F2,$3A,$BD
	.dc.b	$3D,$39,$1E,$06,$BE,$4F,$94,$F0,$FC,$4C,$8F,$2B,$E4,$B9,$1E,$86
	.dc.b	$BE,$CF,$43,$D3,$DB,$23,$DA,$FB,$39,$24,$7C,$06,$BF,$13,$CD,$FC
	.dc.b	$AF,$D7,$BD,$2A,$49,$FC,$6C,$7E,$8D,$FE,$98,$D4,$D3,$1C,$2F,$4F
	.dc.b	$F4,$3F,$5F,$FE,$22,$AF,$CB,$B5,$FF,$45,$B7,$AB,$F6,$4B,$AF,$89
	.dc.b	$27,$8B,$FA,$CF,$AB,$F4,$47,$4F,$A9,$EE,$47,$BF,$C4,$F8,$88,$E0
	.dc.b	$FD,$4F,$D7,$F9,$DF,$2F,$CC,$7A,$08,$FA,$3D,$DF,$59,$1C,$1E,$CF
	.dc.b	$89,$F7,$91,$C1,$FB,$6F,$AF,$FC,$EB,$FE,$AA,$FA,$FF,$E2,$1B,$7F
	.dc.b	$11,$6F,$87,$FE,$C3,$6C,$FF,$7B,$5F,$11,$FC,$31,$B1,$FC,$09,$AF
	.dc.b	$58,$FE,$23,$F5,$EC,$F8,$7C,$B4,$AE,$0F,$3F,$87,$91,$CD,$AF,$FB
	.dc.b	$03,$87,$EB,$65,$E5,$FA,$EF,$11,$27,$C5,$EB,$F9,$48,$E0,$F3,$7A
	.dc.b	$FE,$92,$38,$3F,$8D,$F9,$1F,$C2,$DF,$8D,$E9,$7B,$C8,$FB,$FD,$2F
	.dc.b	$84,$47,$07,$F2,$4C,$9C,$FF,$1F,$E6,$E4,$FC,$77,$E5,$7C,$3C,$8F
	.dc.b	$94,$D7,$FD,$BF,$F5,$ED,$7F,$9D,$FB,$BF,$E8,$79,$FE,$62,$D7,$FD
	.dc.b	$CA,$DE,$6F,$E9,$8F,$F0,$D8,$CF,$99,$C8,$8F,$2F,$33,$A9,$1C,$1F
	.dc.b	$E9,$6F,$2F,$C3,$F4,$D2,$3C,$2F,$93,$E9,$64,$79,$1A,$FF,$A6,$FC
	.dc.b	$AF,$4F,$99,$23,$D2,$FB,$3C,$C9,$1E,$C6,$BF,$E9,$8E,$1F,$83,$E5
	.dc.b	$48,$F8,$2F,$C4,$F1,$E4,$7C,$46,$BF,$13,$E0,$7E,$49,$1F,$92,$FF
	.dc.b	$66,$F2,$7F,$CB,$91,$FD,$73,$EB,$DD,$DF,$DA,$9B,$19,$8C,$7C,$7E
	.dc.b	$7D,$39,$3C,$5E,$1D,$3F,$DC,$BE,$BD,$F1,$94,$95,$FD,$AA,$1E,$F6
	.dc.b	$F8,$64,$E5,$E5,$C3,$FD,$F3,$EB,$FF,$91,$13,$E3,$31,$EF,$E7,$E5
	.dc.b	$E6,$1F,$DD,$A9,$1C,$7E,$83,$FC,$0B,$ED,$3E,$46,$92,$FF,$BE,$C7
	.dc.b	$F9,$B5,$BE,$37,$02,$3A,$F1,$BF,$C1,$FF,$5E,$AA,$49,$FF,$03,$C3
	.dc.b	$BF,$7D,$D2,$72,$F2,$E3,$E7,$ED,$49,$E2,$FF,$87,$3E,$BF,$FB,$C8
	.dc.b	$F8,$FC,$12,$7F,$C2,$B0,$F8,$FF,$F1,$1F,$D7,$FF,$B2,$67,$FC,$32
	.dc.b	$D5,$E4,$7F,$8C,$E3,$0D,$DF,$FC,$57,$E8,$FF,$B2,$AB,$FC,$42,$D7
	.dc.b	$8F,$A3,$FE,$A1,$51,$9A,$FF,$1F,$3F,$16,$38,$F8,$9D,$08,$F4,$F1
	.dc.b	$3B,$FB,$F2,$3B,$BE,$2D,$BF,$C7,$BF,$5E,$F9,$EA,$4A,$FF,$1A,$C7
	.dc.b	$C5,$7F,$C6,$FF,$5F,$FC,$F1,$34,$9E,$C7,$F9,$21,$FF,$4B,$47,$B0
	.dc.b	$FF,$5E,$38,$D3,$FC,$B0,$DD,$7E,$03,$F2,$1B,$B7,$32,$57,$8B,$A3
	.dc.b	$B1,$1C,$1E,$6F,$8B,$F1,$78,$B2,$3D,$2F,$97,$C5,$91,$EC,$6B,$E7
	.dc.b	$3D,$BE,$9F,$F9,$8B,$EB,$FF,$21,$1F,$E5,$86,$BB,$6E,$9F,$E7,$35
	.dc.b	$E7,$F1,$24,$F1,$7F,$84,$71,$7F,$9B,$BE,$6F,$FC,$07,$3E,$16,$DF
	.dc.b	$F9,$F5,$BB,$7E,$15,$FF,$0D,$13,$C5,$E4,$7F,$13,$E4,$FF,$3C,$FD
	.dc.b	$77,$F3,$9B,$FF,$43,$36,$FF,$1C,$DC,$3F,$DE,$0F,$E3,$E3,$3E,$97
	.dc.b	$22,$4F,$F9,$E2,$7F,$24,$FC,$76,$D7,$6F,$0B,$C2,$7F,$A2,$27,$C7
	.dc.b	$E0,$C7,$93,$E6,$D6,$47,$56,$BD,$7C,$1F,$D6,$AF,$AF,$6D,$EA,$9C
	.dc.b	$CF,$EA,$46,$48,$CC,$6F,$2F,$F4,$33,$A8,$CB,$F0,$BC,$27,$D7,$EA
	.dc.b	$FF,$98,$B6,$33,$3F,$E8,$F9,$CB,$FE,$F8,$FA,$9F,$F7,$CA,$69,$1F
	.dc.b	$0F,$CB,$FF,$B2,$31,$FF,$5D,$47,$F9,$6F,$83,$FD,$4A,$A7,$51,$ED
	.dc.b	$F8,$AB,$3F,$C1,$DA,$BC,$D7,$B3,$88,$E8,$7E,$76,$33,$5D,$B9,$75
	.dc.b	$E7,$6D,$B1,$E2,$8E,$5F,$35,$8B,$FD,$30,$73,$3F,$EB,$CB,$7C,$C6
	.dc.b	$2B,$C0,$F9,$B4,$FE,$7D,$5F,$EC,$27,$F9,$E3,$79,$B5,$CD,$23,$A7
	.dc.b	$7D,$C3,$D4,$FC,$0F,$AF,$F8,$D3,$FE,$CC,$7D,$F8,$7F,$B9,$5C,$D7
	.dc.b	$EF,$5C,$CF,$F2,$F6,$C0,$F9,$5D,$DD,$78,$3E,$6E,$88,$F0,$78,$BE
	.dc.b	$1F,$32,$E3,$E1,$70,$F8,$FC,$BB,$8F,$85,$F4,$7C,$57,$A7,$D1,$B5
	.dc.b	$E9,$70,$FA,$FD,$1B,$7E,$97,$E0,$7C,$27,$C1,$F8,$9B,$5F,$05,$FC
	.dc.b	$1F,$CD,$3D,$9B,$F5,$F8,$6D,$7F,$BC,$A7,$D8,$DF,$27,$8F,$07,$27
	.dc.b	$F1,$BF,$5F,$FE,$0E,$9F,$F7,$CA,$FA,$AC,$FB,$7A,$77,$C9,$C3,$E4
	.dc.b	$AD,$F0,$7F,$2F,$F5,$EE,$FB,$DF,$86,$D7,$D3,$EB,$3F,$D9,$69,$C7
	.dc.b	$CB,$8F,$9B,$67,$81,$FD,$AF,$EB,$DE,$87,$E9,$9F,$7F,$37,$F3,$F7
	.dc.b	$33,$C2,$3F,$05,$1A,$C7,$C3,$1F,$85,$9F,$F1,$99,$B7,$CB,$BF,$EC
	.dc.b	$A3,$31,$FD,$2D,$F9,$0F,$F4,$DF,$AF,$E4,$DF,$17,$F7,$31,$8B,$EF
	.dc.b	$28,$FC,$CC,$61,$78,$9C,$36,$E2,$DC,$3C,$93,$FE,$38,$6E,$9D,$BA
	.dc.b	$BB,$BF,$C2,$3C,$6F,$F0,$7C,$FA,$DF,$F2,$CB,$7D,$9D,$7C,$0E,$FD
	.dc.b	$5E,$1E,$2D,$AE,$0F,$F1,$2F,$AF,$FD,$EE,$FF,$DA,$DF,$DE,$B7,$C8
	.dc.b	$D7,$91,$BF,$6B,$87,$C2,$DC,$7E,$66,$08,$E9,$EA,$FF,$1E,$E5,$F8
	.dc.b	$D5,$FE,$27,$A4,$E7,$2D,$79,$1B,$F2,$2D,$FD,$7C,$3F,$C7,$8F,$D9
	.dc.b	$FA,$FF,$F6,$AB,$7E,$07,$F9,$5C,$63,$0F,$F0,$23,$FE,$5F,$9F,$C0
	.dc.b	$7D,$7F,$E8,$5B,$F0,$9F,$CC,$A3,$15,$FE,$0B,$51,$93,$5E,$1D,$AE
	.dc.b	$6E,$1F,$C6,$7D,$7F,$FB,$85,$C9,$8F,$07,$B7,$C0,$F7,$E2,$B3,$E4
	.dc.b	$70,$F2,$F3,$6E,$7F,$E1,$DC,$11,$FD,$23,$EB,$7E,$5C,$FF,$21,$7F
	.dc.b	$1F,$5F,$E5,$37,$30,$72,$EF,$7F,$EA,$36,$F9,$6E,$91,$87,$B7,$83
	.dc.b	$DF,$C7,$B8,$FE,$A7,$1F,$E9,$4D,$8F,$27,$B5,$B8,$D3,$87,$E2,$FD
	.dc.b	$6D,$CF,$D8,$FA,$FE,$67,$D9,$EA,$5A,$EC,$E1,$F9,$BF,$5F,$6F,$E2
	.dc.b	$3F,$0F,$D5,$F8,$9E,$35,$AF,$33,$87,$F8,$07,$D7,$FF,$14,$72,$60
	.dc.b	$F9,$9C,$5F,$37,$82,$C7,$BD,$C3,$F0,$74,$B9,$7F,$94,$77,$47,$D5
	.dc.b	$3F,$BE,$FD,$7F,$ED,$A7,$51,$8F,$1B,$FC,$03,$F5,$FF,$97,$F3,$3F
	.dc.b	$84,$BF,$97,$AF,$F2,$E3,$9C,$7F,$D1,$CE,$67,$7D,$DF,$6B,$8F,$9B
	.dc.b	$6B,$F8,$F5,$7F,$04,$BE,$7E,$BF,$83,$C9,$F0,$B2,$BF,$92,$75,$79
	.dc.b	$F2,$F1,$CF,$F8,$5A,$31,$3E,$2F,$99,$F4,$FC,$CF,$A9,$B1,$E0,$8F
	.dc.b	$6F,$D5,$63,$2F,$5B,$89,$EF,$95,$9F,$F3,$D6,$1E,$7C,$BE,$1F,$97
	.dc.b	$F1,$BE,$27,$E3,$B7,$0F,$6C,$7F,$8A,$BE,$BD,$F5,$39,$C7,$FD,$09
	.dc.b	$18,$AF,$87,$ED,$FF,$8C,$38,$5E,$EF,$D2,$8C,$32,$7F,$A3,$74,$BE
	.dc.b	$9F,$18,$FF,$A3,$E3,$13,$FD,$2F,$EB,$FF,$94,$ED,$FD,$09,$FE,$52
	.dc.b	$DF,$4B,$FC,$7F,$18,$9F,$F6,$43,$6F,$FC,$35,$F5,$EE,$3D,$FA,$7F
	.dc.b	$E4,$68,$C5,$F7,$F8,$7E,$7E,$D7,$46,$C7,$8E,$38,$7D,$6E,$31,$FF
	.dc.b	$53,$5A,$3C,$7E,$67,$93,$9E,$E1,$EB,$70,$FF,$67,$FA,$F7,$C7,$63
	.dc.b	$1F,$F5,$84,$62,$7F,$B6,$7B,$BF,$E3,$49,$FE,$BE,$FF,$97,$8C,$1F
	.dc.b	$D8,$A3,$6F,$F9,$8A,$5F,$39,$B8,$C3,$C7,$CE,$E5,$E7,$5B,$EE,$E1
	.dc.b	$F3,$67,$71,$F2,$BE,$1F,$5F,$C5,$EB,$DA,$F3,$B8,$7F,$C0,$9F,$5E
	.dc.b	$FA,$EC,$61,$FE,$CD,$8C,$4F,$F8,$27,$EA,$FF,$A7,$3F,$E9,$96,$CF
	.dc.b	$F9,$4C,$38,$7F,$CA,$CD,$73,$F1,$9F,$92,$7F,$D4,$46,$3B,$7F,$D3
	.dc.b	$F3,$FE,$84,$F5,$6D,$6E,$BC,$1E,$5F,$7C,$2C,$7F,$85,$8E,$F1,$97
	.dc.b	$C9,$68,$C4,$79,$3C,$7A,$FA,$1B,$26,$98,$EB,$FF,$59,$CF,$B9,$B3
	.dc.b	$F9,$98,$F4,$C6,$39,$E4,$7B,$1A,$FC,$0F,$AD,$74,$74,$6C,$7F,$B0
	.dc.b	$1A,$EE,$3C,$3F,$3F,$FC,$02,$3F,$D8,$AD,$7E,$5B,$5F,$F6,$73,$5F
	.dc.b	$AC,$FE,$71,$E3,$D3,$D0,$8F,$4F,$A1,$EE,$47,$07,$97,$E5,$BD,$04
	.dc.b	$70,$7D,$AF,$83,$F8,$A4,$70,$7D,$EF,$2F,$E4,$91,$C1,$0F,$C2,$EA
	.dc.b	$8D,$46,$1B,$E2,$7F,$94,$A7,$53,$F9,$79,$FC,$A4,$E7,$CB,$BC,$5F
	.dc.b	$84,$7E,$C9,$2B,$F2,$2F,$DE,$FA,$EF,$F4,$04,$7F,$9A,$5F,$F2,$DB
	.dc.b	$7B,$F9,$BF,$EB,$38,$DA,$BF,$F3,$2C,$62,$3F,$E0,$03,$5F,$D4,$35
	.dc.b	$E5,$DD,$E1,$3A,$A8,$EB,$3D,$27,$31,$FE,$DD,$9F,$37,$91,$E9,$23
	.dc.b	$FA,$E3,$93,$ED,$F6,$F8,$B2,$3D,$D1,$D0,$D3,$C6,$51,$88,$FD,$CC
	.dc.b	$61,$E2,$43,$E1,$65,$EE,$F6,$FD,$7F,$5D,$FC,$1E,$72,$7F,$B6,$F9
	.dc.b	$7F,$CC,$AF,$FA,$5D,$AE,$84,$71,$CF,$FD,$22,$FE,$D2,$32,$7E,$D9
	.dc.b	$C1,$1F,$09,$0F,$F5,$22,$EB,$F9,$FC,$0F,$A7,$A4,$AF,$E4,$5F,$17
	.dc.b	$A2,$38,$3F,$D5,$DC,$FF,$E6,$A7,$FD,$61,$F5,$CE,$7F,$B6,$D8,$CC
	.dc.b	$63,$A2,$47,$06,$BF,$06,$9F,$05,$CB,$CE,$93,$D9,$EB,$E6,$C8,$F0
	.dc.b	$B5,$E5,$F1,$DE,$2F,$84,$91,$E6,$7C,$B7,$23,$D2,$D7,$C5,$E5,$FA
	.dc.b	$BB,$A4,$7B,$9F,$6F,$C3,$C8,$F8,$2D,$7D,$5E,$5F,$E7,$3E,$B9,$F2
	.dc.b	$94,$93,$FE,$D7,$8F,$D7,$BF,$81,$6F,$3F,$FB,$41,$BD,$5F,$D4,$5B
	.dc.b	$21,$C1,$FC,$AA,$BF,$DA,$FF,$5D,$FE,$1D,$9F,$F7,$23,$5F,$C7,$36
	.dc.b	$37,$DB,$FB,$28,$FC,$F4,$EA,$30,$7E,$DB,$EA,$BD,$71,$D3,$EA,$38
	.dc.b	$11,$EF,$F0,$FC,$C8,$E0,$FF,$7B,$7D,$77,$9F,$4F,$ED,$98,$BE,$2F
	.dc.b	$C4,$4A,$F4,$FB,$5E,$52,$38,$3D,$BF,$0F,$E9,$23,$83,$E3,$6B,$FE
	.dc.b	$FA,$FA,$EF,$D1,$46,$31,$FE,$F9,$F5,$DF,$EF,$EA,$FF,$7F,$B6,$32
	.dc.b	$7F,$16,$5F,$49,$49,$5C,$CF,$A9,$1A,$70,$3E,$BB,$DC,$E2,$3F,$BC
	.dc.b	$37,$D3,$EC,$77,$F8,$49,$1B,$6B,$FC,$1F,$8B,$ED,$63,$EA,$FA,$DF
	.dc.b	$31,$27,$B7,$C3,$7A,$48,$E0,$F1,$7A,$DE,$D2,$38,$3D,$3E,$3F,$F1
	.dc.b	$6D,$BE,$2F,$A7,$F0,$C9,$3E,$BF,$4F,$E3,$11,$C1,$F0,$B3,$74,$7C
	.dc.b	$BF,$9C,$93,$F0,$DF,$9D,$F0,$92,$3E,$33,$5F,$E4,$DF,$5C,$DB,$FA
	.dc.b	$4F,$D0,$7F,$81,$74,$70,$7F,$8D,$E3,$0D,$F8,$9D,$F9,$93,$1F,$AB
	.dc.b	$5F,$C5,$DF,$E6,$4D,$BE,$71,$FA,$F8,$FF,$06,$3A,$F3,$3C,$24,$7E
	.dc.b	$FF,$C4,$BC,$CF,$2F,$D2,$C9,$EE,$FA,$3E,$92,$47,$89,$AF,$FA,$7B
	.dc.b	$CB,$F6,$F9,$B2,$3C,$EF,$C0,$F3,$64,$7A,$9A,$FF,$A6,$F9,$3E,$2F
	.dc.b	$97,$23,$DE,$FC,$8F,$72,$47,$C2,$6B,$F0,$BE,$67,$E6,$91,$F8,$EF
	.dc.b	$F7,$8F,$2C,$FF,$7A,$FA,$E7,$81,$49,$5F,$A2,$87,$C7,$6F,$D5,$93
	.dc.b	$97,$97,$67,$F8,$13,$EB,$9F,$1B,$49,$5F,$AC,$87,$C6,$EF,$86,$4E
	.dc.b	$5E,$5C,$3F,$E0,$EF,$AE,$FE,$53,$26,$93,$E0,$75,$F2,$F2,$7F,$73
	.dc.b	$28,$E5,$F3,$DF,$E1,$8F,$6D,$F2,$54,$97,$FC,$02,$1D,$DB,$F2,$24
	.dc.b	$E5,$E5,$E4,$7F,$88,$BE,$B9,$74,$95,$FC,$2A,$3F,$18,$DE,$DE,$04
	.dc.b	$75,$C5,$CF,$DB,$91,$C5,$FF,$17,$FD,$77,$F8,$37,$C7,$FF,$4E,$2F
	.dc.b	$90,$C9,$2B,$F8,$CC,$3E,$43,$FC,$6F,$F5,$DF,$EC,$F9,$FE,$3A,$D7
	.dc.b	$FC,$34,$DF,$23,$F5,$30,$86,$FF,$E4,$0F,$4B,$F5,$75,$A6,$E5,$FE
	.dc.b	$AF,$6D,$FF,$5F,$7E,$2B,$FC,$B1,$78,$7C,$C9,$19,$F0,$FB,$7C,$04
	.dc.b	$8E,$0F,$87,$6F,$F2,$8F,$D7,$3E,$22,$92,$BF,$9C,$C7,$C5,$7F,$9D
	.dc.b	$FD,$77,$F4,$57,$FB,$93,$53,$4F,$E8,$4F,$F8,$F5,$FF,$55,$37,$E3
	.dc.b	$FF,$CC,$31,$8D,$7E,$3B,$F2,$75,$FF,$68,$3F,$22,$BC,$3B,$70,$38
	.dc.b	$F3,$C8,$F2,$B5,$F0,$7C,$77,$6F,$12,$47,$A9,$F0,$F8,$92,$3D,$AD
	.dc.b	$7D,$1E,$A7,$F9,$DB,$EB,$BF,$23,$1F,$E5,$76,$E9,$F7,$9B,$A7,$FB
	.dc.b	$9D,$79,$DC,$49,$5F,$D9,$23,$2F,$F9,$33,$8B,$FD,$07,$F3,$5F,$E0
	.dc.b	$E8,$FF,$32,$37,$0F,$C5,$B7,$0F,$F7,$A3,$FB,$C2,$7C,$87,$23,$F8
	.dc.b	$B7,$27,$FA,$3B,$EB,$3F,$9E,$57,$FB,$ED,$B8,$7F,$8A,$DA,$9F,$C8
	.dc.b	$B7,$D4,$E4,$47,$FB,$EC,$7E,$51,$FF,$3D,$37,$1F,$D0,$B6,$BC,$1E
	.dc.b	$0F,$FA,$72,$BC,$9E,$9C,$79,$BE,$7D,$A4,$76,$6B,$DB,$C1,$FF,$59
	.dc.b	$2F,$AE,$6D,$ED,$9C,$CF,$FA,$4F,$24,$62,$35,$9F,$FA,$A6,$72,$DF
	.dc.b	$09,$F5,$D4,$95,$FE,$9D,$3E,$34,$69,$FF,$59,$0F,$A7,$FF,$4C,$11
	.dc.b	$CB,$AF,$90,$FE,$DB,$7F,$C1,$A7,$FC,$CF,$BF,$F8,$A1,$FF,$4C,$37
	.dc.b	$E3,$76,$FF,$AB,$5B,$1F,$E6,$3F,$3B,$F9,$7C,$EC,$FF,$18,$B8,$8F
	.dc.b	$F5,$70,$E4,$E2,$FF,$9A,$FC,$EA,$E6,$BC,$51,$83,$3F,$35,$63,$CC
	.dc.b	$E1,$EF,$F2,$F6,$BF,$D7,$98,$23,$BE,$5F,$CF,$AB,$FC,$8A,$FF,$81
	.dc.b	$DB,$CB,$AE,$59,$1E,$AE,$EB,$8F,$47,$D3,$F5,$DF,$1A,$3F,$D9,$4F
	.dc.b	$CA,$6E,$FA,$DE,$5F,$C4,$ED,$B5,$E6,$79,$F8,$9D,$3C,$BB,$5F,$E5
	.dc.b	$AC,$7F,$A3,$36,$3C,$3E,$56,$E3,$4F,$93,$E3,$BC,$BE,$8D,$9E,$4E
	.dc.b	$1E,$9E,$8D,$CF,$D2,$FB,$3E,$33,$DB,$F1,$96,$BC,$4F,$D8,$F9,$9F
	.dc.b	$F0,$5C,$7F,$9B,$97,$D7,$67,$DA,$D3,$BE,$0E,$1E,$45,$BE,$0F,$E0
	.dc.b	$BE,$BB,$FC,$17,$1F,$E7,$85,$F5,$D9,$F6,$F4,$EF,$B3,$87,$CA,$DB
	.dc.b	$E0,$FE,$33,$EB,$9E,$05,$F0,$30,$DA,$FF,$42,$CB,$8B,$87,$17,$16
	.dc.b	$CF,$FA,$2B,$EB,$9E,$7F,$E6,$17,$05,$8C,$3F,$9C,$9F,$F1,$49,$FE
	.dc.b	$92,$3D,$BC,$AF,$07,$FB,$AC,$6A,$30,$FF,$B3,$5F,$F3,$5F,$07,$E9
	.dc.b	$BE,$BB,$FD,$A1,$3F,$E9,$C7,$F2,$15,$FD,$F9,$CD,$72,$3F,$DA,$CE
	.dc.b	$A7,$A3,$FA,$D8,$D0,$7E,$3C,$75,$F1,$3C,$1B,$78,$7A,$7C,$5A,$DA
	.dc.b	$F1,$3F,$BA,$F1,$5E,$C5,$F8,$2C,$36,$BF,$D6,$73,$FE,$12,$6F,$89
	.dc.b	$C7,$D4,$E4,$FF,$04,$FA,$EF,$DB,$6D,$E6,$7F,$6C,$DF,$23,$5E,$46
	.dc.b	$FD,$9E,$1F,$09,$72,$FF,$60,$E1,$FE,$66,$6F,$A9,$FC,$43,$CB,$F8
	.dc.b	$FC,$BF,$E2,$6A,$4E,$73,$D7,$91,$BF,$22,$E3,$FE,$CC,$C7,$FC,$68
	.dc.b	$FF,$B3,$BE,$BB,$FD,$AA,$DF,$81,$FE,$3B,$18,$C3,$E7,$BF,$E5,$89
	.dc.b	$FF,$6A,$FD,$77,$E7,$9B,$F0,$9F,$C9,$23,$15,$F5,$94,$64,$D7,$89
	.dc.b	$6B,$9B,$87,$FD,$C3,$F5,$DF,$EF,$D7,$26,$3C,$2E,$DF,$01,$DF,$8E
	.dc.b	$CF,$91,$C3,$CB,$CD,$B9,$FE,$23,$04,$7F,$3A,$FA,$CF,$97,$5F,$99
	.dc.b	$7F,$21,$40,$E9,$C3,$A7,$3B,$FF,$44,$B7,$CC,$F5,$8C,$3D,$FC,$0F
	.dc.b	$07,$91,$71,$FF,$7F,$E2,$8C,$C6,$3D,$9B,$97,$E8,$2B,$FB,$BC,$E4
	.dc.b	$F8,$FF,$57,$EB,$AF,$13,$ED,$F9,$DF,$77,$AB,$6B,$BB,$87,$E7,$FD
	.dc.b	$75,$C7,$E3,$3F,$17,$D5,$F8,$DE,$3D,$AF,$3B,$87,$E3,$77,$7F,$9E
	.dc.b	$3E,$BB,$FD,$D7,$1A,$32,$7F,$05,$73,$87,$AF,$8F,$E9,$70,$FD,$36
	.dc.b	$D7,$C4,$8F,$AD,$AE,$2B,$E1,$38,$9E,$B1,$F5,$FE,$BB,$F7,$71,$F3
	.dc.b	$27,$FB,$BC,$62,$7F,$84,$FD,$77,$E7,$72,$FF,$47,$BF,$9D,$8C,$FC
	.dc.b	$F7,$F7,$E8,$C4,$EB,$BB,$EF,$71,$65,$B9,$78,$B0,$5F,$FB,$72,$30
	.dc.b	$FF,$50,$8F,$0F,$95,$D7,$D8,$95,$C7,$AB,$CD,$97,$92,$7F,$C1,$F1
	.dc.b	$89,$FE,$5D,$83,$E7,$3C,$DF,$39,$E7,$6C,$F8,$23,$D5,$F1,$58,$CF
	.dc.b	$E0,$B8,$9F,$06,$76,$7F,$96,$E3,$E9,$7E,$3F,$97,$F0,$FE,$27,$E2
	.dc.b	$36,$FD,$B1,$FE,$24,$F9,$67,$CC,$66,$FF,$9B,$C6,$2B,$E1,$FB,$BF
	.dc.b	$E2,$AE,$17,$BF,$F3,$E3,$0C,$9F,$99,$D0,$FA,$8C,$63,$FD,$0A,$31
	.dc.b	$3F,$47,$EB,$BF,$98,$F2,$FF,$B2,$9F,$E6,$2D,$F4,$BF,$C7,$31,$89
	.dc.b	$FE,$B0,$DB,$FE,$01,$F5,$CE,$3D,$FA,$7F,$E4,$08,$C5,$F7,$F8,$6E
	.dc.b	$7E,$DF,$46,$C7,$8E,$38,$7D,$5E,$31,$FE,$B3,$68,$F1,$F9,$9E,$4E
	.dc.b	$7B,$87,$AD,$C3,$F8,$DF,$AE,$7C,$76,$31,$FE,$C7,$18,$9F,$C9,$7B
	.dc.b	$BF,$E3,$BE,$1F,$F7,$5B,$FD,$A5,$BF,$11,$FE,$5A,$8C,$4F,$F7,$96
	.dc.b	$DF,$F2,$1B,$7C,$E6,$E7,$FC,$C3,$18,$AE,$FE,$7F,$3F,$3F,$A3,$63
	.dc.b	$C7,$1C,$35,$C6,$3E,$87,$C5,$EC,$78,$FD,$8B,$7E,$A7,$0F,$EB,$7E
	.dc.b	$B5,$F5,$D8,$C3,$FC,$07,$18,$9F,$D9,$7D,$5F,$F9,$A7,$B7,$FD,$21
	.dc.b	$48,$C3,$FE,$93,$5F,$34,$F5,$B7,$1A,$9F,$F0,$E1,$C7,$C6,$FF,$0F
	.dc.b	$61,$89,$46,$27,$FC,$45,$18,$7F,$C3,$A7,$F4,$B7,$0F,$FA,$67,$DF
	.dc.b	$FD,$D3,$A0,$E1,$D7,$93,$D5,$FE,$A9,$19,$BF,$17,$4F,$FA,$AA,$FF
	.dc.b	$19,$5E,$A8,$C1,$FE,$AE,$FA,$D7,$55,$24,$FC,$68,$CB,$F2,$30,$7C
	.dc.b	$0F,$FA,$D9,$B1,$98,$C1,$9F,$87,$91,$C5,$AF,$FB,$07,$E7,$DF,$43
	.dc.b	$FE,$C0,$6C,$7D,$66,$BB,$8D,$78,$B2,$34,$D7,$A7,$CB,$73,$23,$CD
	.dc.b	$F8,$9E,$74,$8F,$BA,$D7,$F0,$4F,$F9,$67,$C0,$70,$7D,$A8,$C3,$67
	.dc.b	$F1,$BE,$76,$1F,$B8,$FD,$12,$3F,$8A,$6B,$F8,$F6,$BF,$89,$7E,$17
	.dc.b	$C7,$73,$8F,$CB,$45,$E2,$39,$D3,$FD,$1B,$3F,$D3,$DA,$FC,$DA,$E6
	.dc.b	$FC,$B7,$FD,$DB,$F5,$DF,$E8,$C8,$FE,$AC,$D7,$FC,$D6,$DC,$BE,$0C
	.dc.b	$7F,$88,$63,$FC,$28,$F0,$EE,$E6,$47,$F5,$B1,$A7,$FD,$FC,$BD,$4F
	.dc.b	$63,$9A,$EB,$28,$C6,$7F,$E8,$97,$84,$79,$BD,$EF,$49,$1E,$ED,$7D
	.dc.b	$2F,$63,$BF,$BD,$23,$CA,$FC,$22,$3E,$8F,$03,$4F,$C2,$8F,$3C,$7D
	.dc.b	$3F,$AE,$FE,$03,$8F,$A1,$B8,$FF,$A7,$DB,$7F,$E4,$77,$F8,$53,$FE
	.dc.b	$03,$72,$9A,$4E,$CE,$C7,$F1,$68,$D3,$63,$9C,$7F,$12,$32,$7F,$16
	.dc.b	$FA,$7F,$C3,$11,$FE,$43,$CD,$F8,$92,$38,$3F,$11,$F0,$BF,$8F,$FA
	.dc.b	$EF,$D7,$E5,$F5,$1B,$97,$FA,$ED,$B5,$FE,$63,$7F,$A8,$BF,$E1,$F6
	.dc.b	$BF,$EB,$46,$CF,$3F,$2F,$F3,$1F,$0D,$DF,$A6,$5F,$17,$C5,$E8,$48
	.dc.b	$E8,$D7,$FD,$7F,$CC,$F9,$2F,$46,$92,$7C,$4F,$AF,$09,$1E,$56,$BC
	.dc.b	$BC,$DF,$81,$F3,$72,$3D,$4F,$C3,$F8,$B9,$1E,$D6,$BE,$2F,$33,$F6
	.dc.b	$DF,$5C,$F4,$69,$27,$FC,$85,$0F,$47,$64,$8C,$BF,$BB,$F1,$FF,$80
	.dc.b	$7D,$6F,$F8,$76,$7F,$C9,$4D,$7F,$DB,$ED,$E3,$FD,$4A,$EC,$E2,$49
	.dc.b	$E2,$FF,$0B,$FA,$AF,$34,$74,$FA,$8E,$E4,$7B,$FC,$4F,$88,$8E,$0F
	.dc.b	$F2,$E7,$D7,$7F,$44,$7C,$7F,$1D,$E8,$23,$E8,$F7,$7D,$64,$70,$7B
	.dc.b	$3E,$27,$DE,$47,$07,$F9,$A7,$EB,$BF,$3F,$AF,$F0,$6F,$AD,$FE,$0E
	.dc.b	$DE,$FF,$E4,$D1,$87,$FD,$6D,$48,$7C,$4E,$89,$1F,$E7,$58,$CB,$FE
	.dc.b	$77,$FA,$E6,$7D,$3E,$5A,$4E,$DE,$FE,$BC,$8E,$4D,$59,$7F,$1C,$E1
	.dc.b	$FA,$D9,$F9,$32,$7D,$6F,$A0,$97,$E3,$F6,$3D,$64,$70,$79,$FE,$2B
	.dc.b	$DE,$47,$07,$FA,$3F,$C8,$FF,$80,$2F,$B7,$D6,$F8,$A4,$7E,$07,$A9
	.dc.b	$F2,$08,$E0,$FF,$4C,$66,$E7,$F9,$BF,$35,$27,$E4,$3F,$43,$E2,$24
	.dc.b	$7C,$B6,$BF,$0B,$6F,$A7,$F5,$DF,$02,$3F,$D4,$9E,$EF,$FA,$11,$BF
	.dc.b	$19,$FD,$7E,$31,$1C,$4F,$F7,$83,$FC,$41,$FF,$1A,$FB,$7B,$23,$D3
	.dc.b	$CE,$E4,$47,$07,$FA,$CF,$E3,$1E,$77,$7F,$72,$4F,$91,$F1,$7D,$24
	.dc.b	$8F,$3B,$5F,$F4,$CF,$99,$E7,$F3,$A4,$7B,$1F,$57,$9D,$23,$DE,$D7
	.dc.b	$FD,$2D,$C5,$EF,$F3,$24,$7C,$47,$E1,$79,$32,$3E,$3B,$5F,$88,$7F
	.dc.b	$95,$BD,$EF,$90,$47,$E6,$BF,$E0,$53,$FC,$15,$F5,$DF,$E4,$EE,$FF
	.dc.b	$ED,$ED,$E5,$FD,$3B,$68,$17,$C8,$72,$F5,$A4,$FF,$B7,$65,$1D,$BA
	.dc.b	$7F,$C3,$1F,$5C,$F8,$CA,$4B,$FF,$72,$43,$E3,$37,$C3,$27,$2F,$2E
	.dc.b	$1F,$F1,$17,$D7,$7F,$29,$9F,$F7,$63,$5F,$86,$DD,$BF,$C1,$6F,$F8
	.dc.b	$67,$6F,$E0,$6B,$E5,$B9,$92,$B9,$FC,$F7,$F8,$BF,$DB,$7C,$9D,$24
	.dc.b	$FF,$BE,$E3,$FD,$0A,$DF,$13,$81,$1D,$78,$9F,$E3,$9F,$AE,$55,$24
	.dc.b	$FC,$ED,$3F,$C4,$33,$F9,$48,$C6,$9F,$C6,$A1,$D7,$CC,$93,$A3,$EB
	.dc.b	$C7,$CF,$70,$1F,$E0,$6F,$E2,$1B,$5E,$0F,$6B,$FC,$9D,$F5,$DF,$E1
	.dc.b	$41,$F0,$58,$25,$F4,$87,$C7,$FF,$95,$7E,$BB,$FD,$1B,$3F,$94,$6E
	.dc.b	$1F,$E3,$C6,$E7,$FC,$D1,$FF,$43,$CF,$E6,$BD,$1F,$F0,$2C,$FF,$38
	.dc.b	$6C,$7E,$49,$A9,$F9,$6F,$B9,$77,$FF,$99,$DB,$19,$8C,$77,$E7,$DF
	.dc.b	$93,$C5,$E1,$AF,$F9,$BF,$EB,$9F,$0D,$49,$5F,$32,$3E,$2B,$F3,$BE
	.dc.b	$BB,$FA,$83,$FA,$86,$A6,$9F,$D3,$9F,$DC,$AF,$96,$FF,$3F,$36,$3A
	.dc.b	$46,$3E,$43,$3C,$F2,$7C,$0F,$0E,$79,$1E,$36,$BD,$BE,$3B,$AF,$87
	.dc.b	$23,$D0,$F8,$3C,$39,$1E,$B6,$BE,$6F,$47,$FD,$25,$F5,$DF,$95,$8F
	.dc.b	$E2,$2D,$E6,$FF,$3C,$B7,$97,$FB,$8A,$F2,$F8,$92,$BF,$2B,$19,$7F
	.dc.b	$D3,$1C,$7F,$E9,$FF,$8D,$FF,$0A,$C7,$F1,$E6,$E1,$FE,$86,$6E,$1F
	.dc.b	$DF,$9F,$E0,$0D,$F8,$5E,$44,$AF,$F5,$27,$2F,$FA,$AF,$EB,$7F,$A5
	.dc.b	$13,$D2,$C7,$A5,$2B,$F5,$33,$F9,$02,$78,$5C,$49,$1A,$7F,$D3,$6F
	.dc.b	$5F,$46,$3C,$9F,$06,$F2,$3A,$B5,$FE,$77,$5F,$BE,$D7,$F9,$8F,$D7
	.dc.b	$7F,$A8,$25,$39,$7F,$D0,$0D,$7F,$C3,$E6,$BC,$4F,$F0,$8B,$A6,$BA
	.dc.b	$8F,$E6,$9F,$5D,$FE,$89,$8E,$BE,$72,$31,$3A,$9E,$F3,$97,$FD,$29
	.dc.b	$F4,$FF,$E9,$54,$D2,$3D,$7E,$43,$FD,$93,$97,$FA,$FA,$3F,$CC,$DC
	.dc.b	$5F,$C5,$94,$EA,$3D,$7F,$19,$67,$FC,$76,$D5,$E7,$3D,$8C,$47,$43
	.dc.b	$FB,$2C,$66,$BB,$72,$EB,$E6,$76,$D8,$F1,$47,$2F,$93,$C5,$FF,$6E
	.dc.b	$0E,$67,$F9,$6D,$BE,$BB,$15,$E0,$7C,$DA,$7F,$40,$AF,$E6,$6F,$F8
	.dc.b	$41,$BC,$BA,$E5,$91,$D3,$C0,$B8,$7A,$9F,$81,$F5,$DF,$1A,$7F,$9E
	.dc.b	$BE,$FC,$3F,$DD,$2E,$6B,$FC,$0A,$E6,$7F,$CC,$6D,$8F,$EF,$B9,$3B
	.dc.b	$FA,$F0,$7C,$D8,$C5,$F6,$F1,$FC,$3E,$77,$89,$B1,$E2,$8F,$2F,$9B
	.dc.b	$8C,$7C,$2F,$A3,$E3,$7D,$3E,$9D,$BF,$4B,$87,$D7,$E9,$DB,$F4,$BF
	.dc.b	$03,$E1,$FE,$0F,$C6,$5A,$F8,$2F,$E0,$FE,$67,$FC,$2B,$1F,$D8,$17
	.dc.b	$D7,$67,$DD,$D3,$BE,$1E,$1E,$55,$BE,$0F,$E3,$7E,$BB,$FC,$29,$1F
	.dc.b	$DA,$97,$CC,$E7,$DD,$D3,$BE,$4E,$1F,$2B,$6F,$83,$F9,$7F,$AE,$77
	.dc.b	$DE,$FC,$36,$BF,$BB,$CB,$8B,$87,$17,$16,$CF,$F7,$CF,$AE,$FF,$17
	.dc.b	$B7,$7F,$D2,$46,$39,$3F,$C8,$AF,$DA,$7F,$C8,$F3,$FE,$33,$3F,$C0
	.dc.b	$A7,$B9,$CF,$8B,$FA,$D8,$C7,$69,$FF,$55,$FD,$77,$F9,$11,$FF,$06
	.dc.b	$BF,$8F,$8D,$57,$F9,$22,$34,$FF,$95,$9D,$4F,$2F,$0F,$F7,$51,$F1
	.dc.b	$23,$9C,$63,$5C,$F4,$EB,$4B,$5E,$07,$F8,$17,$8D,$FE,$03,$8F,$F0
	.dc.b	$FA,$F9,$0C,$FB,$9A,$77,$EB,$70,$F1,$2D,$F0,$7F,$86,$7D,$77,$ED
	.dc.b	$F6,$FF,$36,$BF,$B7,$6F,$8F,$AF,$1F,$7E,$E7,$0F,$86,$B8,$FF,$8C
	.dc.b	$30,$47,$4F,$53,$F8,$C7,$9B,$F1,$39,$7E,$3E,$6B,$56,$A3,$7E,$47
	.dc.b	$0F,$22,$E5,$FE,$3C,$C7,$FC,$62,$FF,$8F,$BE,$BB,$FD,$BA,$DF,$83
	.dc.b	$FE,$4D,$18,$C3,$FD,$0E,$FF,$97,$27,$FC,$8B,$F5,$DF,$9C,$6F,$C3
	.dc.b	$7F,$2D,$8C,$57,$FA,$39,$46,$4D,$78,$76,$B9,$B8,$7F,$CA,$3F,$5D
	.dc.b	$FE,$EA,$72,$63,$C1,$ED,$F0,$7D,$F8,$EC,$F9,$1C,$3C,$BC,$DB,$9F
	.dc.b	$FA,$83,$04,$7F,$43,$FA,$EF,$99,$5F,$F0,$01,$FC,$7D,$7F,$A6,$9C
	.dc.b	$B7,$7E,$07,$F4,$BB,$7C,$CE,$2F,$D2,$F5,$F0,$3B,$78,$F6,$7F,$CD
	.dc.b	$B8,$FF,$44,$6C,$78,$FD,$DD,$C6,$9C,$3F,$13,$EB,$2E,$7E,$B7,$D5
	.dc.b	$F3,$9E,$BF,$52,$D7,$67,$0F,$CC,$FA,$EB,$7F,$0D,$F8,$5E,$AF,$C3
	.dc.b	$F1,$ED,$79,$9C,$3F,$E8,$4F,$AE,$FD,$E3,$93,$07,$CB,$E2,$F9,$9C
	.dc.b	$16,$3D,$EE,$1F,$83,$A5,$CB,$FD,$A7,$BA,$3E,$A1,$FD,$DF,$EB,$BF
	.dc.b	$6D,$3A,$8C,$78,$FF,$DE,$FE,$BB,$F2,$DE,$17,$E9,$1F,$CB,$57,$FB
	.dc.b	$79,$CE,$3F,$ED,$48,$46,$A7,$87,$77,$DB,$E3,$E8,$DB,$FF,$51,$D7
	.dc.b	$F0,$56,$FA,$FE,$1F,$27,$C4,$C8,$FF,$54,$F5,$7A,$31,$F2,$4F,$F8
	.dc.b	$46,$31,$3E,$3B,$7C,$D7,$AF,$E6,$BD,$8D,$9F,$0C,$7B,$FE,$2F,$1A
	.dc.b	$FB,$5C,$4F,$83,$3B,$3F,$EF,$EC,$7D,$0F,$C3,$F2,$FE,$47,$C5,$7C
	.dc.b	$96,$DF,$BA,$3F,$C4,$DF,$2C,$F9,$7C,$DB,$7F,$E2,$33,$9B,$F8,$BE
	.dc.b	$E7,$F8,$B7,$85,$EF,$7D,$58,$C3,$27,$FD,$9B,$D2,$FA,$EC,$65,$D6
	.dc.b	$7E,$B4,$E5,$FF,$69,$FD,$77,$F2,$DC,$FF,$90,$BF,$CB,$5B,$E9,$7F
	.dc.b	$8F,$63,$13,$F7,$DB,$61,$F5,$DF,$EA,$99,$EB,$E9,$FF,$91,$63,$13
	.dc.b	$E3,$38,$57,$F6,$17,$01,$FE,$BF,$5D,$FD,$9F,$17,$BF,$23,$FD,$D7
	.dc.b	$6F,$90,$C6,$1C,$A3,$13,$EE,$F3,$3D,$5C,$DE,$B6,$C7,$C2,$8F,$F2
	.dc.b	$D7,$D7,$3E,$37,$38,$79,$23,$15,$FE,$FA,$F7,$3F,$87,$4F,$A1,$BC
	.dc.b	$1F,$D2,$9F,$ED,$51,$9F,$33,$AE,$52,$F9,$45,$9E,$DE,$BF,$88,$FF
	.dc.b	$36,$7D,$6F,$F2,$68,$65,$FD,$52,$FC,$5E,$1F,$F9,$C7,$23,$C6,$32
	.dc.b	$7D,$5C,$FF,$AE,$E1,$E4,$C3,$FD,$2A,$D7,$F7,$4D,$7E,$E3,$57,$D6
	.dc.b	$7F,$9B,$7A,$3F,$D2,$CD,$8F,$F4,$F9,$AF,$8A,$FF,$44,$B8,$8D,$3F
	.dc.b	$EA,$45,$F4,$1F,$C2,$70,$FE,$E3,$EE,$FF,$BC,$DB,$F4,$99,$CF,$F8
	.dc.b	$75,$F0,$C3,$D9,$18,$C3,$51,$F9,$3C,$39,$4F,$F1,$56,$C7,$E5,$EB
	.dc.b	$93,$F9,$06,$E1,$E0,$8E,$A7,$3C,$3A,$BF,$0B,$0B,$5C,$30,$ED,$5A
	.dc.b	$C3,$BD,$EC,$E6,$53,$F3,$EE,$8D,$D7,$38,$F3,$5F,$3A,$91,$CA,$74
	.dc.b	$F3,$C3,$D0,$F4,$9F,$2D,$48,$F0,$DF,$79,$C4,$7C,$19,$EA,$F1,$3B
	.dc.b	$3A,$AE,$A4,$C3,$E8,$4E,$8E,$B5,$CA,$7F,$C5,$78,$79,$8F,$3E,$1E
	.dc.b	$1A,$E0,$78,$4F,$C5,$E1,$FE,$E1,$7E,$31,$F8,$77,$13,$F6,$23,$07
	.dc.b	$02,$1F,$8F,$8F,$46,$1E,$68,$FB,$DF,$2B,$87,$E6,$9E,$2C,$3C,$F1
	.dc.b	$98,$F9,$93,$C6,$73,$1B,$8C,$11,$FC,$64,$F1,$BF,$81,$7F,$86,$8C
	.dc.b	$39,$6C,$F3,$38,$94,$F0,$9E,$63,$D2,$66,$7A,$BF,$83,$8E,$6F,$CB
	.dc.b	$36,$5B,$FA,$75,$E1,$8E,$4D,$7F,$98,$FD,$5B,$E3,$3E,$93,$63,$F7
	.dc.b	$A1,$C3,$FC,$01,$AF,$EF,$7E,$1F,$F4,$F1,$AB,$FF,$36,$B7,$2F,$DD
	.dc.b	$CF,$EB,$3E,$AD,$DD,$7B,$E6,$3F,$A8,$9C,$B6,$78,$11,$FD,$6F,$D5
	.dc.b	$BE,$2F,$F6,$FF,$58,$F8,$4A,$4A,$E9,$F2,$DE,$13,$0F,$F5,$29,$E0
	.dc.b	$E8,$F1,$7C,$1F,$9C,$8F,$97,$E0,$FD,$44,$70,$79,$3E,$8F,$DD,$F3
	.dc.b	$32,$3D,$8F,$C1,$EF,$91,$EF,$6B,$FD,$93,$47,$83,$FC,$4B,$EB,$3E
	.dc.b	$54,$FF,$71,$6A,$68,$F0,$38,$12,$3F,$89,$C6,$5F,$93,$E2,$7F,$1E
	.dc.b	$FA,$CF,$F0,$54,$7F,$18,$6B,$FD,$D9,$BD,$1F,$EA,$85,$F0,$7C,$49
	.dc.b	$3C,$5F,$E5,$3F,$4F,$E6,$97,$D3,$77,$25,$77,$F8,$3F,$11,$1C,$1F
	.dc.b	$CA,$BE,$B1,$97,$9F,$E2,$A4,$F9,$DF,$57,$A3,$23,$D4,$D7,$D5,$F0
	.dc.b	$7E,$F2,$3E,$F7,$F9,$FF,$D6,$7F,$5A,$8F,$F0,$AF,$D5,$FF,$B1,$B7
	.dc.b	$FF,$63,$B7,$C1,$FF,$31,$36,$7F,$C4,$8F,$F4,$22,$79,$1F,$D2,$9F
	.dc.b	$E8,$BF,$58,$CF,$1F,$07,$2F,$FA,$3D,$23,$87,$BD,$27,$2D,$7F,$C5
	.dc.b	$3C,$7F,$53,$2F,$1F,$D6,$77,$24,$F4,$F8,$2F,$11,$1C,$1E,$0F,$82
	.dc.b	$F3,$11,$C1,$FD,$6B,$CB,$FF,$72,$BE,$6F,$93,$F6,$11,$F4,$FC,$17
	.dc.b	$BC,$8E,$0F,$EC,$79,$7E,$09,$F7,$F9,$7F,$14,$8F,$C2,$F7,$BE,$41
	.dc.b	$1C,$1F,$DB,$3E,$B1,$A7,$F7,$6F,$4F,$FC,$DD,$3F,$DC,$1A,$FF,$93
	.dc.b	$9B,$CB,$FF,$00,$17,$D1,$F1,$24,$F1,$7A,$7B,$52,$3A,$35,$FF,$2C
	.dc.b	$FC,$17,$F1,$67,$B7,$99,$E3,$23,$E3,$FA,$2F,$31,$1C,$1F,$E0,$69
	.dc.b	$79,$5E,$BF,$32,$4F,$AD,$F7,$79,$92,$3D,$CD,$7F,$CD,$7E,$07,$CD
	.dc.b	$7E,$07,$93,$F1,$48,$FC,$5F,$0B,$E4,$11,$C1,$F2,$7D,$7F,$98,$47
	.dc.b	$07,$F3,$07,$FC,$4B,$F5,$8E,$DA,$49,$FF,$10,$C3,$E2,$77,$EC,$49
	.dc.b	$CB,$CB,$A3,$FC,$5F,$F5,$8F,$83,$A4,$AF,$F1,$5C,$3E,$0F,$7C,$12
	.dc.b	$72,$F2,$E0,$FF,$1C,$FD,$67,$F1,$F9,$FF,$47,$B7,$A5,$E1,$FF,$B6
	.dc.b	$E3,$0D,$F9,$3E,$3F,$23,$2F,$FC,$77,$48,$EB,$F3,$7F,$E4,$6F,$75
	.dc.b	$F1,$54,$97,$FE,$42,$87,$7E,$FC,$59,$39,$79,$78,$BF,$E5,$0F,$AC
	.dc.b	$5D,$25,$7F,$93,$23,$F2,$1B,$7A,$A3,$FD,$69,$0E,$FE,$44,$8E,$1C
	.dc.b	$9D,$7D,$99,$1C,$DF,$F2,$FF,$D6,$7F,$BF,$63,$FD,$6A,$BE,$6F,$24
	.dc.b	$8F,$F2,$EC,$7F,$68,$7F,$D7,$7F,$59,$FE,$B5,$7F,$CC,$2D,$73,$F1
	.dc.b	$9F,$E6,$C8,$6D,$FE,$6E,$F3,$FF,$D6,$55,$FE,$68,$6B,$FE,$C6,$6E
	.dc.b	$BE,$F8,$FE,$4D,$B7,$1F,$0A,$47,$17,$A7,$85,$D7,$DE,$93,$D5,$F0
	.dc.b	$6D,$FE,$7E,$FA,$C7,$CC,$D2,$57,$F9,$EA,$3E,$13,$FE,$7C,$FA,$CF
	.dc.b	$E8,$09,$A4,$F5,$FF,$D1,$2B,$E5,$29,$27,$97,$0E,$B8,$24,$75,$78
	.dc.b	$F3,$C8,$F0,$35,$FF,$74,$DB,$E5,$3B,$F8,$B2,$7C,$CF,$8B,$C5,$91
	.dc.b	$E9,$6B,$CB,$D2,$FF,$4F,$FD,$67,$E3,$3C,$3F,$F7,$F4,$A3,$0F,$FC
	.dc.b	$01,$6B,$9F,$3F,$89,$1F,$F5,$0C,$23,$FD,$57,$F0,$8E,$4F,$F5,$6F
	.dc.b	$C7,$FF,$86,$89,$F1,$98,$F8,$69,$8F,$FA,$AA,$7F,$6A,$8F,$F7,$0C
	.dc.b	$74,$E7,$FF,$5B,$FD,$57,$F3,$C9,$FF,$58,$B5,$76,$7E,$26,$30,$DF
	.dc.b	$57,$89,$27,$8B,$D3,$B6,$47,$46,$BF,$E4,$93,$C3,$F1,$71,$F0,$BE
	.dc.b	$4D,$A4,$79,$1A,$F9,$38,$3E,$7E,$3F,$B9,$C7,$DD,$F5,$9F,$EF,$EA
	.dc.b	$F8,$13,$97,$FD,$E2,$D7,$F7,$0D,$79,$F8,$BF,$01,$DF,$8A,$FA,$CF
	.dc.b	$F3,$ED,$7F,$2F,$6C,$66,$77,$1F,$EB,$55,$F4,$94,$8F,$FA,$E5,$FF
	.dc.b	$5D,$7D,$0F,$F1,$3C,$BF,$6D,$3F,$E8,$DC,$FF,$DC,$6A,$75,$1F,$C3
	.dc.b	$F8,$EC,$FE,$41,$AB,$D0,$FE,$53,$3C,$A7,$31,$89,$F9,$47,$E6,$38
	.dc.b	$F8,$AF,$42,$C6,$19,$7E,$4F,$96,$B5,$C8,$FC,$73,$C8,$EF,$F5,$96
	.dc.b	$7F,$DF,$B8,$23,$CF,$9F,$F3,$89,$FF,$80,$2F,$FA,$79,$BC,$9A,$E4
	.dc.b	$91,$CB,$C0,$B9,$78,$71,$47,$C2,$FA,$CF,$91,$5F,$A1,$C7,$F9,$14
	.dc.b	$7C,$DB,$F4,$3A,$6E,$DC,$4F,$8F,$DD,$6F,$D2,$F5,$F1,$FB,$79,$B6
	.dc.b	$7B,$1F,$C0,$B0,$FE,$84,$E3,$0F,$D9,$46,$1F,$0F,$C8,$F2,$9D,$42
	.dc.b	$7F,$D6,$91,$85,$F1,$9E,$DF,$53,$DC,$D9,$D4,$7C,$1F,$53,$17,$EE
	.dc.b	$7E,$27,$C4,$7C,$5F,$7A,$DF,$57,$F1,$3F,$2D,$FE,$11,$8F,$1A,$FA
	.dc.b	$CC,$FB,$BA,$77,$C1,$C3,$C9,$B7,$C1,$FC,$9F,$D6,$7F,$84,$63,$DA
	.dc.b	$BE,$B3,$3E,$F6,$9D,$F4,$70,$F9,$6B,$7C,$1F,$CE,$FD,$67,$E5,$63
	.dc.b	$E3,$AE,$EC,$F7,$E9,$DF,$17,$0E,$2B,$7C,$1F,$D3,$FD,$63,$D0,$BC
	.dc.b	$18,$6D,$7D,$43,$FC,$5C,$5D,$BF,$5E,$E3,$97,$F4,$4F,$BD,$FD,$27
	.dc.b	$D6,$7F,$31,$8F,$BA,$FE,$3F,$BB,$FB,$A3,$99,$DF,$89,$FB,$A8,$FF
	.dc.b	$6A,$57,$0D,$B1,$E2,$65,$D7,$56,$F7,$B5,$D9,$FE,$07,$E2,$FF,$A9
	.dc.b	$63,$F1,$AF,$F9,$45,$BE,$DE,$BE,$17,$7E,$B7,$0F,$16,$D7,$07,$F8
	.dc.b	$6F,$D6,$7E,$DF,$7F,$E0,$0F,$ED,$DB,$E4,$6B,$C8,$DF,$B9,$C3,$E1
	.dc.b	$EE,$3F,$9E,$C1,$1D,$3D,$3F,$E3,$3C,$9F,$0F,$97,$FC,$31,$49,$CE
	.dc.b	$7A,$F2,$77,$E4,$DC,$BF,$53,$8F,$EA,$2B,$F8,$A7,$D6,$3D,$DF,$E4
	.dc.b	$AF,$F9,$42,$BF,$5E,$E5,$BF,$09,$FC,$AB,$EB,$3F,$2D,$1F,$B4,$5F
	.dc.b	$0F,$9F,$1B,$4C,$9D,$78,$D7,$2F,$E4,$97,$FC,$CE,$72,$7E,$FB,$EB
	.dc.b	$3F,$DB,$8E,$27,$F9,$54,$61,$E3,$F0,$7E,$3E,$4F,$23,$6B,$C9,$1F
	.dc.b	$D0,$29,$E7,$66,$F9,$BF,$D1,$3E,$AB,$E6,$5F,$F8,$59,$FC,$93,$67
	.dc.b	$5C,$12,$3C,$DF,$31,$67,$CA,$F3,$F0,$7A,$79,$36,$BF,$8C,$E1,$FB
	.dc.b	$36,$C7,$87,$DD,$DC,$69,$C3,$F0,$FE,$A6,$E3,$F0,$9F,$47,$CE,$FA
	.dc.b	$7D,$6B,$5D,$9C,$3F,$2F,$EB,$2D,$FC,$A7,$E0,$7A,$DF,$07,$C8,$B5
	.dc.b	$E6,$70,$FF,$30,$FA,$CF,$F4,$EB,$93,$07,$C9,$E1,$F9,$5C,$16,$3D
	.dc.b	$EE,$1F,$83,$A5,$CB,$FB,$26,$E8,$FA,$47,$F7,$6F,$AC,$FD,$A4,$EA
	.dc.b	$31,$E4,$7F,$7A,$FA,$CF,$CA,$4F,$F6,$D7,$F3,$0D,$EE,$D7,$6C,$CB
	.dc.b	$F1,$9C,$76,$BF,$16,$F3,$F2,$7A,$7B,$16,$BF,$A7,$E3,$FC,$FE,$FB
	.dc.b	$4F,$F8,$2B,$AB,$CB,$E2,$FF,$A2,$6B,$4F,$FA,$2E,$33,$7F,$E6,$09
	.dc.b	$7C,$DF,$A7,$E6,$FD,$4D,$9E,$F1,$ED,$F8,$CC,$65,$ED,$71,$3F,$D8
	.dc.b	$B0,$FE,$2D,$39,$8C,$67,$F1,$7C,$CF,$8D,$F1,$56,$7D,$6E,$1F,$ED
	.dc.b	$7F,$2C,$FA,$CC,$61,$FE,$17,$8C,$4F,$C2,$F7,$7F,$C5,$3C,$4F,$7F
	.dc.b	$E9,$46,$19,$3F,$DD,$BA,$9F,$59,$8C,$7F,$C4,$51,$89,$FE,$F7,$F5
	.dc.b	$9F,$C9,$F9,$3F,$D6,$4F,$F2,$76,$FA,$7F,$E3,$88,$C4,$FF,$A2,$9B
	.dc.b	$7F,$E7,$9F,$AC,$FF,$52,$4F,$4F,$53,$FC,$7F,$18,$9F,$F1,$95,$7F
	.dc.b	$53,$C3,$BF,$B5,$D7,$E1,$FB,$33,$1F,$F0,$7E,$0F,$91,$C6,$3F,$E3
	.dc.b	$B8,$C4,$FA,$BC,$DF,$37,$3F,$9D,$B1,$EE,$8F,$F2,$8F,$D6,$3E,$3B
	.dc.b	$38,$7F,$91,$63,$15,$FE,$1E,$F7,$7F,$83,$4F,$FB,$A4,$38,$7F,$DD
	.dc.b	$8D,$7F,$CA,$DE,$6F,$F9,$99,$FF,$4E,$53,$1C,$42,$31,$7D,$63,$13
	.dc.b	$A9,$CC,$78,$5F,$E9,$D3,$B6,$C6,$7D,$8E,$07,$6C,$1E,$15,$25,$F7
	.dc.b	$8C,$BE,$1F,$AB,$FF,$30,$73,$DF,$8A,$FC,$76,$3F,$06,$5F,$31,$FE
	.dc.b	$7F,$8C,$4E,$9F,$F6,$68,$FA,$CF,$C8,$BE,$D6,$BF,$2D,$AF,$06,$BF
	.dc.b	$ED,$6A,$FF,$10,$3F,$E5,$05,$F4,$5F,$ED,$AE,$66,$5B,$8C,$3C,$7E
	.dc.b	$12,$DF,$FB,$36,$F9,$BF,$E8,$37,$9B,$D6,$B6,$A3,$77,$DA,$39,$56
	.dc.b	$A7,$8C,$73,$AC,$4C,$8E,$84,$3C,$0E,$5E,$06,$6B,$B3,$E1,$8D,$46
	.dc.b	$2B,$BB,$6B,$37,$EF,$BE,$4F,$C3,$7E,$45,$E9,$F8,$53,$E9,$30,$F7
	.dc.b	$38,$9A,$38,$90,$F3,$92,$BC,$93,$CA,$FC,$B7,$E1,$AF,$79,$C6,$BD
	.dc.b	$0F,$63,$C0,$68,$FC,$75,$7E,$8D,$C9,$F8,$CA,$FB,$2D,$8D,$46,$23
	.dc.b	$E1,$1C,$8F,$CE,$47,$94,$FB,$BF,$29,$FE,$26,$8F,$25,$F0,$BF,$5C
	.dc.b	$7F,$A7,$A7,$4F,$96,$32,$D9,$32,$77,$3A,$24,$F5,$37,$29,$DC,$66
	.dc.b	$3F,$1A,$79,$0E,$9E,$DF,$A5,$C1,$C5,$C3,$E3,$36,$6B,$0F,$CC,$CF
	.dc.b	$58,$E0,$67,$0F,$C0,$DF,$8A,$3C,$27,$33,$EA,$E1,$F9,$48,$D4,$62
	.dc.b	$BA,$5F,$68,$F0,$9C,$5C,$3B,$33,$5F,$9B,$F8,$6F,$D7,$FD,$63,$E3
	.dc.b	$E9,$FA,$D8,$95,$FC,$55,$FB,$E4,$7F,$94,$7D,$63,$BF,$F1,$0D,$8C
	.dc.b	$C6,$08,$FF,$B6,$A3,$4B,$0F,$D8,$C6,$1F,$81,$F1,$BF,$C1,$21,$F2
	.dc.b	$BF,$C1,$BE,$B1,$F0,$B4,$D4,$4C,$3C,$AF,$90,$8F,$A3,$87,$A1,$1F
	.dc.b	$27,$C1,$F7,$23,$83,$F2,$9F,$43,$E3,$F9,$89,$1E,$B7,$CD,$D9,$23
	.dc.b	$DC,$D7,$DD,$C7,$F9,$2E,$4F,$EA,$91,$FE,$FC,$FA,$CF,$A4,$DE,$3F
	.dc.b	$E4,$71,$88,$E2,$7E,$81,$FE,$C6,$F1,$EC,$D9,$1F,$D5,$78,$5F,$CB
	.dc.b	$3E,$B3,$FA,$0F,$27,$F1,$E6,$E5,$FA,$46,$D7,$EB,$17,$C1,$F1,$24
	.dc.b	$7E,$C2,$B9,$47,$F3,$8F,$A7,$F3,$47,$5F,$AC,$F0,$12,$B8,$61,$DF
	.dc.b	$E0,$FC,$84,$8C,$9E,$17,$FA,$1F,$D6,$33,$F5,$7C,$AC,$AD,$8D,$11
	.dc.b	$EF,$F4,$24,$EA,$7F,$D4,$2D,$8F,$89,$F0,$72,$34,$D7,$E7,$8F,$AC
	.dc.b	$FE,$E0,$E3,$EA,$FF,$A3,$ED,$FD,$21,$BA,$FF,$0E,$6C,$FF,$16,$7F
	.dc.b	$D9,$CF,$A1,$AF,$3F,$81,$FE,$C3,$18,$AF,$EA,$7F,$59,$EC,$AE,$1F
	.dc.b	$07,$A2,$3C,$FD,$EE,$24,$70,$7C,$6E,$3F,$BA,$B0,$7D,$5F,$80,$91
	.dc.b	$E5,$F8,$2F,$19,$1C,$1E,$9F,$82,$F3,$91,$C1,$F5,$3C,$BF,$F7,$A3
	.dc.b	$EF,$F9,$5F,$69,$1F,$85,$F0,$5F,$00,$8E,$0F,$47,$4F,$C4,$F2,$A4
	.dc.b	$7C,$97,$E3,$FB,$D2,$3E,$63,$5F,$B3,$F5,$9E,$E7,$F0,$FE,$97,$F9
	.dc.b	$FD,$BF,$05,$FE,$10,$8C,$47,$E0,$17,$D6,$64,$91,$C9,$E1,$ED,$48
	.dc.b	$EA,$D7,$FD,$15,$E8,$75,$F3,$24,$78,$9F,$07,$D1,$48,$F2,$B5,$FF
	.dc.b	$49,$79,$7E,$5F,$36,$47,$A9,$F4,$79,$B2,$3D,$AD,$7F,$D1,$FC,$1E
	.dc.b	$DF,$6A,$47,$C2,$7E,$07,$8B,$23,$E2,$B5,$F8,$5F,$31,$F1,$48,$FC
	.dc.b	$A7,$FB,$D9,$FE,$38,$FA,$C7,$6D,$24,$FE,$DA,$1F,$15,$BF,$62,$4E
	.dc.b	$5E,$5C,$FF,$E4,$2F,$AC,$7C,$1D,$25,$7F,$00,$87,$C1,$EF,$82,$4E
	.dc.b	$5E,$5C,$1F,$E4,$CF,$AC,$FE,$4F,$3F,$BC,$6F,$8D,$FE,$40,$6F,$91
	.dc.b	$FE,$06,$5F,$29,$C7,$E4,$A5,$FF,$10,$94,$75,$F9,$DF,$F2,$CF,$C4
	.dc.b	$BE,$2E,$92,$FF,$8B,$C7,$F9,$E9,$BE,$27,$02,$3A,$F1,$3F,$CC,$7F
	.dc.b	$58,$AA,$49,$FE,$43,$1F,$D9,$F4,$FF,$0E,$AE,$DE,$24,$9D,$F1,$F4
	.dc.b	$F6,$E4,$72,$7F,$CE,$3F,$59,$FE,$8A,$E1,$FF,$31,$AF,$9D,$C9,$2B
	.dc.b	$F9,$8C,$7F,$72,$7F,$99,$FD,$67,$FB,$1D,$BF,$1B,$FE,$7C,$8C,$47
	.dc.b	$F2,$C5,$BF,$FA,$03,$D1,$FF,$62,$4F,$F3,$C6,$BF,$CC,$9B,$8F,$F9
	.dc.b	$D9,$F8,$91,$BF,$07,$91,$1E,$5E,$0F,$5F,$80,$91,$D5,$F0,$6B,$FE
	.dc.b	$90,$FA,$C7,$CD,$52,$57,$F4,$E8,$F8,$4F,$F5,$0F,$AC,$FE,$7A,$9A
	.dc.b	$4F,$5F,$FD,$36,$BE,$56,$92,$79,$70,$6B,$79,$1D,$5E,$3D,$12,$3C
	.dc.b	$0D,$7F,$AA,$DB,$E5,$7B,$F8,$92,$7C,$CF,$8B,$C4,$91,$E9,$6B,$CB
	.dc.b	$D2,$FF,$58,$FD,$67,$E2,$E3,$FC,$DA,$D4,$D3,$FC,$66,$FF,$72,$6B
	.dc.b	$FD,$A1,$BC,$BF,$EA,$C6,$F6,$7F,$88,$78,$FF,$D7,$FF,$21,$FE,$08
	.dc.b	$8F,$F3,$AB,$6F,$FC,$FA,$D7,$FD,$D7,$F0,$9C,$08,$FF,$80,$A3,$F7
	.dc.b	$2F,$FA,$0F,$97,$FD,$97,$F5,$7F,$CE,$A3,$FC,$0E,$D7,$FB,$EB,$72
	.dc.b	$FF,$60,$BF,$8E,$C3,$FD,$0E,$D9,$FF,$47,$B5,$34,$FC,$9B,$D3,$C0
	.dc.b	$F0,$91,$FF,$0B,$FC,$57,$EB,$6B,$AF,$A5,$E6,$24,$F9,$BD,$2F,$49
	.dc.b	$1C,$1E,$AF,$0B,$FC,$D0,$BE,$B1,$AF,$A2,$73,$3F,$EA,$03,$CE,$E1
	.dc.b	$D6,$99,$7D,$B1,$97,$DF,$E1,$BE,$B3,$51,$F0,$23,$0F,$1D,$BF,$9B
	.dc.b	$CE,$9B,$1F,$EC,$BF,$A6,$FF,$66,$26,$91,$EB,$F4,$3F,$EF,$FC,$7F
	.dc.b	$DE,$71,$FE,$95,$DB,$FC,$7E,$A7,$51,$ED,$F8,$CB,$3F,$EB,$E6,$AF
	.dc.b	$39,$ED,$62,$3A,$1F,$E3,$E8,$CD,$76,$E5,$D7,$9F,$B6,$C7,$8A,$39
	.dc.b	$7C,$BE,$2F,$FC,$8E,$1C,$CF,$F9,$5A,$DF,$2B,$8A,$F0,$3E,$6D,$3F
	.dc.b	$9F,$57,$F9,$71,$FF,$56,$37,$9F,$5C,$D2,$3A,$78,$17,$0F,$53,$F0
	.dc.b	$3E,$B3,$E3,$4F,$F9,$99,$F7,$E0,$8F,$F7,$1C,$6A,$BF,$CB,$EE,$A3
	.dc.b	$3B,$FC,$AC,$DD,$DD,$63,$0F,$7F,$1F,$C1,$E6,$DB,$F0,$38,$7C,$5F
	.dc.b	$23,$72,$EE,$F9,$FE,$3B,$D1,$E9,$DA,$F4,$38,$7D,$5E,$9D,$BF,$3B
	.dc.b	$EF,$F8,$DF,$81,$F1,$96,$BE,$03,$F8,$2F,$97,$FF,$08,$C7,$FA,$21
	.dc.b	$7D,$66,$7D,$CD,$3B,$DF,$87,$93,$6F,$83,$F8,$CF,$AC,$FF,$08,$C7
	.dc.b	$FA,$49,$7D,$3E,$7D,$DD,$3B,$E5,$E1,$F2,$F6,$F8,$3F,$96,$FA,$CF
	.dc.b	$F0,$7C,$7F,$A7,$1F,$C5,$37,$BF,$5D,$FB,$E2,$E1,$C5,$6B,$83,$FA
	.dc.b	$0F,$AC,$7A,$17,$83,$0D,$AF,$F5,$49,$FE,$2E,$2E,$DF,$A8,$71,$CB
	.dc.b	$FE,$A2,$7D,$EF,$FA,$93,$EB,$0F,$EB,$9F,$C5,$77,$7E,$36,$33,$1A
	.dc.b	$9F,$F5,$BB,$A7,$8C,$21,$E1,$F3,$DB,$A3,$6B,$8F,$4F,$7D,$BC,$0D
	.dc.b	$8F,$F6,$1F,$8C,$F6,$FF,$80,$BF,$E5,$48,$CC,$63,$E1,$33,$EB,$6B
	.dc.b	$C4,$B5,$C9,$FE,$13,$F5,$9F,$B4,$DB,$FD,$86,$FE,$D1,$BE,$3E,$BC
	.dc.b	$7D,$FB,$9C,$3E,$1A,$E3,$FE,$D6,$C1,$1D,$3D,$5F,$E2,$BE,$6F,$C5
	.dc.b	$CB,$AC,$61,$CF,$91,$AF,$22,$E3,$FE,$E0,$B4,$7F,$1E,$FA,$C7,$BB
	.dc.b	$7E,$0F,$0D,$BF,$F7,$3F,$D6,$7E,$59,$BF,$0D,$FC,$9A,$31,$9F,$FB
	.dc.b	$81,$46,$4D,$78,$B6,$B9,$B8,$7F,$DE,$7F,$59,$FE,$F2,$72,$63,$C3
	.dc.b	$ED,$F0,$7D,$F8,$EC,$F9,$1C,$3C,$BC,$EB,$9F,$FB,$CB,$04,$7F,$3C
	.dc.b	$FA,$AF,$91,$E1,$7E,$3E,$91,$F8,$D7,$F2,$AE,$AF,$F1,$EE,$63,$7C
	.dc.b	$5F,$D1,$B0,$7C,$CF,$58,$C3,$DF,$C2,$F0,$79,$17,$1F,$D0,$E5,$E1
	.dc.b	$CB,$F6,$F9,$23,$15,$E7,$F6,$F8,$DF,$03,$C6,$E5,$F9,$3F,$53,$6F
	.dc.b	$E0,$3E,$FF,$9D,$F8,$1E,$AD,$AE,$55,$EE,$8C,$C7,$D2,$FA,$CB,$3F
	.dc.b	$25,$F9,$1E,$B7,$C9,$F1,$ED,$72,$AF,$93,$19,$8C,$7D,$67,$F4,$77
	.dc.b	$5B,$7F,$A8,$1C,$9F,$4F,$8F,$EA,$70,$D8,$DB,$87,$8E,$B7,$1F,$8D
	.dc.b	$1F,$52,$3E,$E1,$FD,$E3,$EB,$3F,$75,$1C,$0F,$33,$89,$E2,$7F,$12
	.dc.b	$FA,$CF,$D0,$CF,$AD,$FF,$59,$37,$B3,$FC,$07,$18,$9E,$21,$C7,$1F
	.dc.b	$E2,$F9,$BF,$C5,$77,$CF,$B7,$F0,$DC,$9C,$BC,$8E,$7E,$C5,$AF,$03
	.dc.b	$87,$E6,$75,$79,$F1,$FC,$B6,$5F,$D4,$B3,$F0,$CE,$6B,$F9,$1C,$BE
	.dc.b	$6B,$CF,$F3,$5E,$86,$CF,$B6,$3D,$7F,$19,$8C,$3D,$77,$FE,$8F,$AF
	.dc.b	$85,$3F,$E2,$2C,$3F,$91,$56,$A3,$19,$FC,$4F,$2F,$E2,$FC,$55,$C3
	.dc.b	$E5,$38,$7F,$0D,$F5,$8F,$99,$C6,$3F,$2A,$31,$3F,$47,$DD,$FF,$18
	.dc.b	$F1,$3D,$FF,$A3,$18,$64,$FE,$47,$A5,$F5,$98,$C7,$EA,$C6,$27,$F2
	.dc.b	$DF,$59,$FC,$CF,$93,$F8,$03,$FC,$CD,$BE,$97,$F9,$02,$31,$3F,$D3
	.dc.b	$9B,$7F,$D6,$3E,$B3,$E3,$3F,$82,$8C,$A8,$CD,$F6,$F8,$9E,$9E,$DF
	.dc.b	$56,$E1,$E4,$8F,$F2,$7D,$3E,$47,$38,$7E,$3A,$31,$5E,$AF,$2F,$CD
	.dc.b	$CF,$E7,$6C,$7B,$A3,$FC,$B3,$F5,$8F,$9C,$CE,$1F,$98,$8C,$57,$EE
	.dc.b	$7D,$DF,$F7,$B3,$FE,$76,$6C,$FF,$9F,$5A,$FF,$A0,$DA,$FF,$72,$9F
	.dc.b	$EF,$AF,$FA,$1D,$AE,$73,$FF,$39,$BF,$E0,$08,$CB,$7E,$13,$2A,$3F
	.dc.b	$A5,$BC,$E3,$31,$88,$FF,$00,$FD,$5F,$F9,$93,$93,$FD,$1A,$D8,$FF
	.dc.b	$4A,$3F,$E7,$8F,$43,$3F,$15,$F8,$B9,$FF,$4C,$7D,$57,$E4,$DF,$F4
	.dc.b	$AB,$5C,$F5,$FF,$A6,$14,$65,$FF,$52,$3B,$E4,$FE,$94,$DD,$BF,$C2
	.dc.b	$AD,$9F,$F0,$FF,$D6,$7F,$9F,$5F,$E5,$11,$FE,$C9,$9F,$F0,$DC,$FF
	.dc.b	$AC,$A7,$F9,$63,$5F,$E1,$F3,$B7,$FC,$52,$D5,$1B,$AF,$EB,$8D,$AF
	.dc.b	$1F,$D0,$7F,$8A,$27,$CE,$D8,$F0,$F8,$D6,$FC,$ED,$7E,$0C,$DC,$3C
	.dc.b	$18,$75,$9D,$9E,$C9,$D1,$C6,$35,$1E,$6A,$CB,$B3,$DE,$EC,$CB,$F0
	.dc.b	$27,$0E,$8E,$06,$4C,$1F,$CD,$27,$BD,$7C,$53,$CF,$5C,$A7,$81,$30
	.dc.b	$F8,$F1,$97,$E2,$C7,$67,$2F,$67,$2F,$1C,$3E,$4B,$97,$E7,$BE,$63
	.dc.b	$F0,$F8,$7E,$2E,$CD,$24,$7E,$49,$AF,$E5,$9B,$5E,$02,$57,$E3,$B8
	.dc.b	$BE,$3C,$62,$BF,$0B,$5F,$5E,$73,$5E,$49,$F4,$1F,$8A,$8C,$BC,$0F
	.dc.b	$81,$1D,$5A,$64,$E2,$7E,$92,$38,$BF,$86,$36,$64,$B7,$F4,$EB,$C9
	.dc.b	$18,$78,$C7,$F1,$78,$C1,$C2,$73,$3A,$6C,$64,$D9,$E4,$9E,$4F,$EA
	.dc.b	$3D,$FF,$CC,$75,$48,$FD,$8C,$7E,$6C,$FA,$84,$7F,$81,$BF,$C0,$7E
	.dc.b	$B3,$FD,$1B,$1F,$E9,$03,$FD,$22,$E1,$C0,$70,$ED,$AF,$0E,$5F,$D6
	.dc.b	$47,$1C,$FF,$CA,$4D,$CF,$AF,$AD,$FA,$FF,$92,$FD,$97,$CA,$BE,$82
	.dc.b	$99,$CB,$F8,$5C,$FE,$34,$8F,$F1,$C7,$F2,$1E,$E7,$EE,$49,$1F,$C4
	.dc.b	$BD,$CF,$DE,$11,$C1,$FE,$3D,$FA,$C7,$D1,$7F,$01,$F9,$99,$3F,$C6
	.dc.b	$9F,$E0,$9F,$09,$23,$F8,$EB,$5F,$F1,$3D,$3C,$1F,$E1,$9F,$58,$F9
	.dc.b	$4A,$4A,$FF,$26,$47,$C3,$7F,$D8,$ED,$4D,$31,$4F,$0F,$F8,$C7,$D5
	.dc.b	$FF,$81,$2B,$FC,$AC,$D7,$FC,$72,$DE,$4F,$F4,$12,$F8,$1E,$24,$9E
	.dc.b	$2F,$F2,$3F,$A9,$F4,$47,$4E,$2E,$E4,$7B,$FC,$07,$88,$8E,$0F,$F3
	.dc.b	$47,$D6,$79,$DF,$2F,$C2,$FA,$08,$FA,$3C,$EF,$59,$1C,$1E,$CF,$7F
	.dc.b	$DE,$47,$07,$F9,$D7,$EB,$1E,$3F,$F3,$BF,$AB,$FF,$50,$6F,$FE,$A1
	.dc.b	$6F,$6F,$FB,$B1,$B7,$FE,$F6,$7F,$D8,$AF,$F9,$FD,$AF,$F9,$89,$B3
	.dc.b	$CF,$DF,$FE,$91,$F5,$8D,$3C,$1F,$21,$30,$E4,$F8,$FD,$B9,$1D,$5A
	.dc.b	$FF,$9B,$B8,$BE,$B6,$7E,$5F,$AC,$F5,$12,$7C,$9F,$01,$ED,$23,$83
	.dc.b	$D1,$F0,$1F,$04,$8E,$0F,$F4,$D7,$D6,$3B,$BE,$37,$C9,$49,$F8,$0F
	.dc.b	$C9,$F8,$09,$1F,$0D,$AF,$9F,$A7,$E7,$7C,$BC,$8F,$90,$FD,$1F,$7A
	.dc.b	$47,$CB,$6B,$FE,$8C,$FA,$C6,$BF,$DC,$7E,$7F,$FC,$15,$26,$93,$D2
	.dc.b	$D1,$23,$FD,$69,$1F,$E0,$53,$F9,$1F,$3F,$FB,$7D,$BC,$29,$91,$86
	.dc.b	$FA,$5F,$E0,$1F,$82,$7A,$3E,$3F,$A3,$95,$D9,$F3,$79,$D2,$3C,$2D
	.dc.b	$7F,$CF,$DE,$67,$AF,$CE,$91,$E6,$7D,$DE,$74,$8F,$4B,$5F,$F3,$DF
	.dc.b	$1F,$C3,$F3,$64,$7B,$9F,$8D,$E5,$48,$F8,$2D,$7F,$D6,$FB,$3E,$57
	.dc.b	$E6,$12,$3E,$3B,$FE,$1F,$F3,$35,$1F,$ED,$DF,$AC,$78,$1F,$E2,$36
	.dc.b	$C6,$63,$1F,$13,$9F,$9B,$93,$C5,$E1,$D7,$FE,$2B,$FA,$C7,$C2,$D2
	.dc.b	$57,$FB,$AE,$1F,$07,$BE,$29,$39,$79,$71,$7F,$8D,$7E,$B3,$F9,$21
	.dc.b	$3D,$AC,$7B,$39,$F9,$29,$87,$FB,$DE,$91,$C7,$E7,$3F,$C7,$DE,$EB
	.dc.b	$E2,$E9,$2F,$FD,$FD,$1F,$91,$6F,$8F,$C0,$8E,$BC,$7F,$F2,$47,$D6
	.dc.b	$2A,$92,$7F,$21,$E2,$3C,$4F,$C9,$42,$BF,$1E,$EA,$33,$0F,$03,$A9
	.dc.b	$27,$47,$E6,$E3,$E6,$B8,$3A,$79,$3E,$42,$39,$FA,$CF,$EF,$98,$78
	.dc.b	$E3,$F4,$90,$F9,$CF,$F3,$13,$63,$84,$63,$E7,$3F,$CC,$DF,$59,$FE
	.dc.b	$B9,$9F,$03,$55,$F5,$51,$98,$C2,$DB,$FC,$E1,$E9,$7E,$CE,$8D,$26
	.dc.b	$DA,$F1,$24,$FB,$E7,$FB,$73,$7C,$4E,$7F,$03,$23,$4F,$6D,$BF,$CF
	.dc.b	$DF,$58,$F9,$BA,$4A,$F9,$51,$F0,$5F,$5F,$1F,$D8,$FA,$CF,$E8,$F0
	.dc.b	$69,$3D,$9F,$F4,$52,$F9,$5A,$49,$FA,$D1,$F0,$EB,$A6,$FC,$09,$3A
	.dc.b	$E7,$E6,$47,$07,$2F,$95,$EF,$E2,$C8,$EA,$F8,$BC,$59,$1E,$06,$BE
	.dc.b	$C3,$F9,$26,$2F,$57,$FD,$45,$F5,$9F,$94,$9D,$07,$1C,$FF,$75,$BA
	.dc.b	$7E,$71,$7A,$1C,$89,$5F,$95,$8C,$BF,$E2,$FE,$3F,$F5,$87,$C5,$FF
	.dc.b	$83,$A3,$BB,$70,$FC,$53,$70,$FD,$4B,$FE,$25,$27,$89,$C8,$FE,$39
	.dc.b	$CB,$FE,$BB,$FA,$AF,$F6,$41,$3D,$6C,$7A,$D3,$2F,$D4,$4F,$E4,$49
	.dc.b	$D9,$C4,$91,$A7,$FC,$AA,$F5,$F5,$63,$C9,$F0,$6F,$23,$AB,$5F,$97
	.dc.b	$5F,$BD,$D7,$FA,$A7,$D6,$7F,$B8,$63,$13,$FE,$6E,$6B,$FE,$73,$6B
	.dc.b	$FC,$29,$AF,$F5,$D7,$F9,$7F,$D6,$7F,$9D,$A1,$18,$7A,$79,$E9,$D4
	.dc.b	$E6,$78,$7D,$2F,$17,$FB,$43,$53,$4D,$7C,$1F,$F0,$BC,$7F,$80,$4F
	.dc.b	$F9,$7F,$8B,$F8,$A3,$FE,$34,$6F,$C6,$6D,$FF,$47,$36,$3F,$BB,$79
	.dc.b	$FF,$E0,$19,$D9,$FC,$75,$C4,$7E,$50,$71,$F1,$7F,$BF,$79,$F5,$CD
	.dc.b	$78,$A3,$06,$7E,$8A,$C7,$99,$C3,$DF,$E4,$ED,$7E,$8B,$04,$77,$CB
	.dc.b	$FA,$05,$7F,$31,$7F,$C1,$2D,$E3,$D7,$1C,$8F,$57,$7D,$C7,$A3,$E9
	.dc.b	$FA,$CF,$8D,$1F,$B0,$7E,$53,$77,$D6,$F2,$FE,$27,$75,$AF,$33,$CF
	.dc.b	$C5,$E9,$E5,$DA,$FE,$8D,$8F,$F3,$F6,$C7,$87,$CB,$DC,$69,$F2,$7C
	.dc.b	$67,$97,$D2,$B3,$C9,$C3,$D3,$D2,$B9,$FA,$5F,$67,$C3,$7B,$7E,$2E
	.dc.b	$D7,$89,$FB,$1F,$2C,$F6,$AF,$D6,$61,$B5,$FD,$76,$5C,$1C,$3C,$8E
	.dc.b	$2D,$9F,$EC,$5F,$59,$FE,$0C,$6F,$CD,$7E,$16,$31,$AF,$F1,$C5,$19
	.dc.b	$EB,$D7,$D6,$5B,$E6,$FE,$37,$EB,$1D,$F7,$BF,$0D,$AF,$EE,$52,$E2
	.dc.b	$E1,$C5,$C5,$B3,$FD,$DF,$EB,$1E,$7F,$E6,$5F,$6C,$66,$31,$BF,$E7
	.dc.b	$63,$FC,$56,$7F,$7F,$3D,$7E,$07,$F6,$F7,$3C,$1F,$CE,$1C,$9F,$E1
	.dc.b	$0F,$AC,$FF,$31,$3F,$E0,$77,$F1,$F3,$FD,$DD,$CC,$FF,$40,$73,$3C
	.dc.b	$7C,$3F,$D7,$47,$C2,$8E,$51,$8D,$73,$D1,$AD,$2D,$77,$7F,$6D,$E2
	.dc.b	$7F,$7E,$8F,$F0,$C3,$FE,$50,$6F,$AF,$AF,$82,$DF,$A9,$C3,$C4,$B5
	.dc.b	$C1,$FE,$01,$F5,$9F,$B2,$DB,$FA,$BB,$FB,$26,$F8,$FA,$F1,$F7,$EC
	.dc.b	$F0,$F8,$6B,$8F,$F8,$A7,$0F,$F3,$43,$7D,$2F,$E1,$9E,$57,$C4,$E5
	.dc.b	$FF,$0E,$57,$FA,$A6,$BF,$C6,$13,$97,$FC,$7A,$FE,$B7,$3F,$ED,$2B
	.dc.b	$C6,$E7,$E3,$F4,$7F,$8C,$7D,$63,$DA,$BF,$03,$86,$E3,$FE,$3D,$FA
	.dc.b	$CF,$CB,$B7,$E1,$3F,$90,$46,$33,$FE,$EE,$A3,$26,$BC,$4B,$5C,$DC
	.dc.b	$3F,$E4,$9F,$A7,$FF,$6D,$39,$31,$E1,$76,$F8,$0E,$FC,$96,$7C,$8E
	.dc.b	$1E,$5E,$6D,$CF,$FC,$13,$82,$3F,$9B,$7D,$57,$C7,$9F,$F5,$A3,$F8
	.dc.b	$E6,$9A,$E2,$97,$E5,$F9,$8B,$3D,$1E,$7E,$0F,$4F,$22,$D7,$F9,$8F
	.dc.b	$04,$66,$31,$EE,$5C,$3F,$C3,$B4,$8F,$29,$FD,$2F,$37,$D5,$62,$FE
	.dc.b	$33,$EA,$F9,$BF,$5F,$AB,$67,$C0,$E1,$F9,$9F,$59,$72,$F8,$6F,$C2
	.dc.b	$F5,$7E,$1F,$91,$6B,$D0,$E1,$FF,$3E,$7D,$67,$FA,$25,$C9,$83,$E5
	.dc.b	$F1,$7C,$CE,$1B,$1F,$05,$C3,$F0,$F4,$B9,$7C,$07,$E9,$9F,$DA,$7E
	.dc.b	$B3,$F6,$91,$99,$FF,$46,$46,$3C,$0F,$F3,$E7,$D6,$7E,$5E,$7F,$D2
	.dc.b	$0F,$FA,$AD,$BD,$9F,$DC,$E3,$17,$FE,$4A,$6D,$7F,$A1,$78,$7E,$85
	.dc.b	$F2,$ED,$FA,$1C,$7C,$7C,$8E,$5E,$BD,$AE,$EE,$1F,$F5,$07,$57,$95
	.dc.b	$FF,$56,$E5,$FD,$2D,$CF,$95,$E3,$F9,$AF,$27,$CD,$59,$F4,$B8,$7D
	.dc.b	$7F,$13,$77,$F4,$3E,$B3,$FC,$1D,$8F,$F1,$78,$CC,$F8,$B3,$F8,$1E
	.dc.b	$67,$C1,$F8,$9B,$3F,$19,$C3,$FE,$B7,$F9,$77,$D3,$63,$0F,$F3,$7C
	.dc.b	$62,$7E,$6F,$B9,$FE,$1D,$E2,$7B,$BF,$36,$30,$C9,$FF,$61,$75,$3E
	.dc.b	$B3,$18,$FF,$9E,$A3,$13,$FE,$C7,$FA,$CF,$E5,$1D,$1F,$EB,$77,$F9
	.dc.b	$43,$7D,$3F,$F1,$64,$62,$7F,$DB,$2D,$B0,$FA,$BF,$F5,$04,$F5,$F5
	.dc.b	$3F,$C6,$91,$89,$FF,$45,$D7,$F5,$6C,$3C,$1F,$2B,$DB,$E2,$3B,$B3
	.dc.b	$1F,$F6,$E6,$0F,$AC,$C6,$3F,$E9,$48,$C4,$FA,$FC,$DF,$3F,$3F,$A1
	.dc.b	$B1,$EF,$8F,$F2,$37,$D6,$3E,$37,$38,$7F,$A7,$A3,$15,$FE,$EB,$7F
	.dc.b	$C0,$F1,$FE,$F8,$0E,$1F,$F7,$DB,$5F,$86,$D7,$75,$FE,$64,$72,$D7
	.dc.b	$33,$FE,$77,$94,$E7,$EB,$3F,$C9,$FC,$7F,$E7,$B6,$DF,$8C,$2F,$9C
	.dc.b	$7A,$3B,$F8,$BF,$C3,$D6,$F1,$7B,$9F,$E7,$D6,$C7,$86,$30,$7F,$A0
	.dc.b	$7E,$B3,$F9,$4C,$7F,$A2,$DA,$F8,$5A,$FC,$56,$BF,$84,$0F,$0E,$36
	.dc.b	$B9,$8C,$37,$D3,$E1,$3A,$AF,$C6,$D3,$D2,$CD,$F5,$FA,$CF,$B5,$5F
	.dc.b	$92,$BE,$0A,$1F,$25,$C2,$3F,$2F,$18,$78,$57,$3E,$2D,$47,$16,$CE
	.dc.b	$E3,$FD,$C6,$DB,$EE,$F8,$A3,$33,$C2,$E4,$76,$3C,$31,$D0,$CB,$D2
	.dc.b	$30,$EC,$F1,$16,$7C,$17,$F0,$67,$73,$E0,$34,$F4,$BF,$2D,$E9,$F3
	.dc.b	$C6,$6B,$E0,$57,$5A,$C3,$F1,$A7,$84,$EA,$3C,$C7,$63,$81,$0C,$92
	.dc.b	$BC,$2E,$5F,$0B,$C4,$F4,$11,$CB,$C6,$3C,$11,$CA,$3F,$41,$7F,$3D
	.dc.b	$FD,$A3,$5F,$31,$8B,$FF,$6D,$46,$63,$E8,$9F,$EE,$57,$0F,$CE,$3C
	.dc.b	$55,$E0,$3F,$35,$F3,$CF,$67,$E1,$CF,$23,$67,$F2,$28,$D3,$EF,$9E
	.dc.b	$4E,$F0,$FC,$AC,$EA,$38,$9E,$18,$F7,$1A,$AE,$F1,$9A,$E6,$AB,$95
	.dc.b	$70,$9C,$DF,$E4,$E3,$95,$78,$CF,$14,$F2,$AE,$2F,$0C,$3B,$C6,$27
	.dc.b	$64,$3C,$47,$BF,$0F,$2E,$1D,$EF,$A4,$62,$F9,$B6,$BC,$47,$A5,$FD
	.dc.b	$6F,$5C,$8F,$CF,$3F,$B2,$C1,$F1,$59,$9F,$D8,$CD,$CB,$E5,$CF,$EE
	.dc.b	$3E,$B1,$F0,$D4,$97,$FA,$B8,$FF,$5C,$B7,$DB,$FE,$54,$FF,$5A,$8C
	.dc.b	$BF,$CB,$5A,$FF,$41,$6B,$FC,$47,$E7,$3F,$1A,$47,$F1,$8D,$7F,$83
	.dc.b	$BF,$C2,$8F,$E1,$BF,$58,$F6,$BF,$86,$FB,$F2,$BF,$7C,$FF,$10,$DA
	.dc.b	$47,$F0,$16,$AD,$3F,$17,$F4,$5F,$C5,$BE,$62,$4F,$F0,$77,$F8,$CF
	.dc.b	$C3,$C8,$FE,$14,$D7,$97,$1F,$E7,$BC,$1F,$E4,$3F,$58,$F9,$5A,$4A
	.dc.b	$FE,$29,$1F,$39,$BC,$5C,$08,$FF,$19,$E5,$FE,$B6,$FF,$1B,$FA,$CF
	.dc.b	$F0,$A3,$7C,$CF,$E5,$F1,$88,$F1,$9F,$E0,$C7,$FA,$03,$C7,$AF,$64
	.dc.b	$7F,$93,$7D,$47,$9D,$ED,$F4,$BD,$08,$F8,$79,$3B,$91,$C1,$FC,$BF
	.dc.b	$EB,$3C,$EF,$88,$60,$F8,$8F,$39,$23,$D9,$E8,$7A,$88,$E0,$F8,$1F
	.dc.b	$07,$EE,$23,$83,$F9,$F7,$D6,$7F,$78,$7F,$A0,$7D,$67,$F5,$AE,$8F
	.dc.b	$EB,$6D,$E5,$FE,$D8,$D9,$FF,$18,$BF,$EC,$97,$8F,$C1,$FF,$5A,$8C
	.dc.b	$47,$AD,$B5,$FD,$9B,$EB,$3D,$93,$E0,$F8,$0E,$04,$7C,$7E,$EF,$32
	.dc.b	$38,$3E,$8F,$27,$7F,$AA,$91,$EA,$7C,$5F,$03,$23,$DA,$D7,$C9,$F0
	.dc.b	$3E,$72,$3F,$09,$FE,$E7,$F5,$8E,$EF,$67,$C9,$CA,$F8,$EF,$BF,$E0
	.dc.b	$A4,$7C,$A6,$BF,$6F,$A9,$D5,$F1,$7E,$5E,$4F,$D0,$7E,$47,$BF,$23
	.dc.b	$E9,$B5,$FC,$27,$D6,$36,$FF,$03,$FA,$FF,$E7,$A9,$FE,$F0,$D7,$F1
	.dc.b	$4D,$E4,$FF,$0A,$BF,$DE,$A3,$FC,$D2,$D7,$FC,$75,$E8,$F4,$3A,$FA
	.dc.b	$12,$3A,$B5,$FF,$43,$FA,$1E,$2F,$A3,$91,$E2,$7C,$BE,$74,$8F,$2B
	.dc.b	$5F,$F4,$7F,$97,$EA,$F3,$64,$7A,$9F,$6F,$9B,$23,$DA,$D7,$FD,$19
	.dc.b	$C3,$F0,$BC,$C9,$1F,$09,$F8,$BE,$DC,$8F,$8A,$D7,$E2,$FA,$DF,$28
	.dc.b	$8F,$CA,$7F,$A1,$1F,$E3,$4F,$AC,$77,$52,$4F,$F8,$9E,$1F,$13,$BF
	.dc.b	$9B,$93,$97,$97,$47,$F8,$FB,$EB,$1F,$07,$49,$5F,$E3,$18,$7C,$1E
	.dc.b	$F8,$24,$E5,$E5,$C1,$FE,$48,$FA,$CF,$E4,$93,$FC,$25,$BD,$3F,$E5
	.dc.b	$46,$F9,$3F,$E6,$57,$FA,$7C,$67,$E4,$B9,$12,$BF,$C8,$52,$F9,$CF
	.dc.b	$F2,$AF,$C4,$3E,$26,$92,$BF,$C9,$11,$FE,$83,$6F,$89,$C0,$8E,$BC
	.dc.b	$4F,$F2,$FF,$D6,$2A,$92,$7F,$CA,$50,$EE,$DF,$6C,$9C,$BC,$B8,$F9
	.dc.b	$F8,$D2,$78,$BF,$E6,$CF,$AC,$FF,$55,$70,$7F,$9D,$97,$CE,$64,$95
	.dc.b	$FE,$61,$87,$CE,$7F,$9D,$3E,$B3,$FD,$85,$3F,$E6,$56,$BF,$CD,$DB
	.dc.b	$7F,$E8,$25,$BF,$F9,$F3,$CF,$FF,$0B,$57,$FB,$61,$A9,$A7,$C3,$7F
	.dc.b	$CE,$4B,$C1,$E4,$48,$CF,$83,$D7,$DF,$91,$C1,$F0,$6B,$FE,$8C,$FA
	.dc.b	$C7,$CD,$D2,$57,$F9,$FE,$3E,$1B,$FE,$81,$FA,$CF,$E7,$C9,$A4,$F5
	.dc.b	$BF,$D2,$EB,$E5,$69,$27,$97,$C8,$EB,$69,$1D,$5E,$3D,$32,$3C,$0D
	.dc.b	$7F,$B1,$5B,$E5,$7B,$F8,$B2,$7C,$CF,$8B,$C5,$91,$E9,$6B,$CB,$D2
	.dc.b	$FF,$56,$FD,$67,$E2,$E3,$FD,$3C,$D7,$FB,$83,$79,$BF,$D7,$6B,$CE
	.dc.b	$E2,$49,$E2,$FF,$91,$F8,$FF,$D7,$3F,$31,$FE,$03,$9F,$EF,$4D,$BF
	.dc.b	$F6,$2B,$7C,$5F,$F6,$B3,$FB,$49,$07,$0D,$F8,$5E,$A7,$F9,$57,$9B
	.dc.b	$FD,$8F,$F5,$5F,$CE,$49,$E9,$E3,$D3,$97,$FE,$BB,$9F,$C7,$93,$C5
	.dc.b	$E2,$48,$D3,$FE,$46,$7A,$F8,$91,$E4,$F8,$36,$91,$D5,$AF,$F8,$66
	.dc.b	$9E,$27,$F9,$1E,$1F,$59,$ED,$7D,$F3,$AD,$FF,$CC,$6D,$8F,$F7,$D1
	.dc.b	$AA,$E4,$E2,$34,$DF,$15,$F5,$9C,$9B,$1C,$84,$FF,$30,$9D,$1F,$E7
	.dc.b	$28,$FA,$3F,$4B,$53,$48,$FF,$9E,$3E,$55,$C7,$FE,$E7,$8F,$F2,$C3
	.dc.b	$63,$31,$FE,$3B,$32,$5F,$8C,$CE,$7A,$8F,$F5,$F7,$B3,$8B,$F8,$35
	.dc.b	$67,$F0,$BE,$82,$B1,$1F,$88,$E5,$AB,$E3,$EC,$61,$98,$79,$FE,$4E
	.dc.b	$D7,$95,$E2,$74,$C1,$F2,$B8,$AF,$CA,$E7,$E7,$9C,$C6,$0F,$CC,$DF
	.dc.b	$FA,$17,$97,$CB,$5F,$CE,$67,$53,$CA,$73,$CB,$27,$33,$FE,$DF,$F0
	.dc.b	$70,$FF,$98,$F0,$47,$C0,$FA,$C7,$05,$E0,$C3,$70,$E7,$3F,$43,$3F
	.dc.b	$47,$7E,$72,$E8,$F3,$F2,$3A,$79,$F7,$0E,$2E,$1E,$7E,$75,$BF,$F3
	.dc.b	$B6,$3F,$D1,$A7,$BF,$C6,$F9,$7D,$5F,$33,$63,$C7,$1E,$9F,$53,$17
	.dc.b	$C9,$F6,$7C,$5F,$B7,$E2,$ED,$FA,$9F,$BF,$F3,$1F,$E0,$E8,$F8,$EB
	.dc.b	$EB,$33,$EE,$E9,$DF,$07,$0F,$2A,$DF,$07,$F1,$3F,$59,$FE,$0E,$8F
	.dc.b	$A2,$BE,$B3,$3E,$F6,$9D,$F4,$F0,$F9,$7B,$7C,$1F,$C9,$FD,$67,$E4
	.dc.b	$63,$ED,$AE,$FC,$F8,$1A,$77,$C5,$C3,$8A,$DF,$07,$F3,$BF,$58,$F4
	.dc.b	$6F,$06,$1B,$5F,$88,$3F,$C5,$65,$DB,$F4,$AE,$39,$7F,$D5,$8F,$BD
	.dc.b	$FF,$57,$7D,$67,$F3,$08,$FC,$8B,$F8,$C6,$C7,$EB,$A3,$0F,$0F,$13
	.dc.b	$1A,$E5,$B5,$CB,$A3,$86,$BC,$5B,$1F,$9E,$F1,$BF,$C0,$0D,$F8,$5F
	.dc.b	$DD,$C6,$31,$FF,$61,$3F,$E5,$09,$E1,$EB,$6B,$C4,$DB,$63,$F5,$3F
	.dc.b	$59,$FE,$2B,$6F,$8D,$FC,$1A,$31,$87,$FB,$35,$FD,$A3,$7D,$CD,$7C
	.dc.b	$3E,$DB,$7D,$3D,$4F,$E1,$FE,$47,$C7,$BE,$51,$87,$3E,$36,$BC,$6B
	.dc.b	$7F,$BC,$BF,$D3,$57,$FB,$77,$EB,$1E,$EF,$F1,$A7,$FC,$99,$5F,$C0
	.dc.b	$9C,$B7,$E1,$3F,$8F,$FD,$67,$E5,$63,$F8,$3A,$F8,$7C,$F8,$7A,$64
	.dc.b	$EB,$C3,$B9,$7F,$BB,$6D,$1D,$0F,$E5,$7F,$4F,$FE,$EF,$AE,$07,$F1
	.dc.b	$47,$15,$C7,$E0,$FC,$7C,$3E,$46,$D7,$92,$39,$F9,$78,$AF,$F7,$E5
	.dc.b	$A3,$F9,$CF,$D6,$7C,$BF,$43,$FA,$13,$F8,$E6,$D6,$B8,$66,$3E,$3F
	.dc.b	$98,$B3,$D1,$E7,$E0,$74,$F1,$ED,$7F,$2C,$C7,$F6,$4D,$8F,$0F,$BD
	.dc.b	$B8,$D3,$87,$E0,$7D,$55,$C7,$D2,$FA,$3E,$73,$D3,$EA,$5A,$EC,$E1
	.dc.b	$F9,$1F,$59,$6F,$E0,$BF,$03,$D4,$F8,$3E,$2D,$AF,$33,$87,$FA,$1F
	.dc.b	$D6,$7F,$6E,$72,$60,$F9,$3C,$5F,$2B,$82,$C7,$BD,$C3,$F0,$74,$B9
	.dc.b	$38,$BE,$7C,$7D,$23,$FB,$5F,$D6,$7E,$C6,$33,$5F,$D5,$23,$0F,$F5
	.dc.b	$EF,$AC,$FC,$AD,$F3,$8C,$76,$7F,$74,$9C,$DF,$F7,$D7,$F3,$B5,$B3
	.dc.b	$FC,$7E,$E2,$3F,$C0,$5C,$DF,$E4,$1B,$FE,$C7,$39,$7F,$B6,$C6,$6B
	.dc.b	$90,$9F,$E0,$CE,$1F,$E8,$CD,$F2,$3C,$9E,$B4,$BF,$ED,$FD,$3E,$CA
	.dc.b	$F1,$CF,$F8,$32,$31,$1D,$7E,$6B,$D9,$F3,$5E,$D6,$DF,$9A,$3E,$07
	.dc.b	$C6,$62,$BC,$4E,$27,$BE,$36,$7D,$38,$7A,$34,$F8,$DE,$5F,$C9,$F8
	.dc.b	$AF,$94,$DC,$3E,$1C,$7F,$88,$3E,$AD,$F5,$99,$C7,$E1,$C6,$2B,$E4
	.dc.b	$FB,$7F,$E2,$6E,$27,$BB,$F5,$63,$0C,$9F,$F0,$8F,$3B,$E7,$71,$8F
	.dc.b	$CE,$8C,$4F,$F8,$5B,$EB,$3F,$95,$ED,$FE,$39,$7F,$95,$B7,$E0,$BF
	.dc.b	$C6,$B1,$89,$FF,$12,$B6,$FF,$CF,$FF,$59,$FE,$A9,$9E,$9E,$9F,$F8
	.dc.b	$F2,$31,$3F,$7A,$BF,$AD,$61,$DF,$D9,$EB,$F1,$1D,$99,$8F,$F8,$AB
	.dc.b	$07,$C8,$E3,$1F,$C3,$C6,$27,$D5,$E5,$F9,$B9,$BC,$ED,$8F,$74,$7F
	.dc.b	$93,$BE,$AD,$F1,$D9,$C3,$F2,$31,$8A,$FF,$1C,$BF,$CD,$63,$FC,$E0
	.dc.b	$D7,$FC,$F6,$D7,$FC,$8E,$D7,$FD,$43,$F5,$9F,$E5,$0E,$3F,$F3,$C4
	.dc.b	$23,$0F,$F9,$ED,$FF,$31,$79,$DA,$73,$1F,$E8,$8F,$AC,$FF,$22,$BC
	.dc.b	$9A,$FF,$A0,$DA,$EE,$7F,$D0,$51,$FE,$B9,$8E,$5F,$13,$82,$3C,$FC
	.dc.b	$DD,$91,$C1,$0F,$DA,$CF,$FA,$7D,$B1,$FE,$66,$E2,$FF,$40,$7D,$63
	.dc.b	$E4,$B1,$AF,$F9,$AA,$1C,$F8,$C3,$FD,$8B,$EE,$3B,$31,$08,$C3,$FE
	.dc.b	$6A,$AE,$2D,$AD,$CE,$A3,$F8,$4B,$71,$F1,$CF,$4A,$E3,$5A,$BE,$26
	.dc.b	$66,$47,$F9,$E2,$F9,$61,$FE,$82,$6C,$7F,$B7,$2F,$9E,$1D,$A3,$86
	.dc.b	$1D,$1F,$31,$0F,$01,$A3,$11,$C5,$C9,$1C,$12,$FC,$53,$D6,$7F,$93
	.dc.b	$D7,$8A,$7E,$76,$19,$BF,$F5,$96,$1E,$8B,$5E,$DF,$55,$6E,$FA,$C7
	.dc.b	$91,$E6,$F9,$CF,$23,$87,$C3,$1C,$5F,$B7,$87,$CA,$72,$7E,$82,$BE
	.dc.b	$8B,$FE,$E2,$6B,$CC,$C4,$FF,$B6,$63,$31,$E6,$39,$46,$4E,$D8,$7C
	.dc.b	$58,$FA,$9D,$38,$C3,$CD,$39,$9F,$83,$39,$7A,$1E,$63,$E5,$B9,$7E
	.dc.b	$3D,$5B,$4D,$8C,$36,$7E,$41,$E2,$29,$C8,$B3,$C9,$B3,$C9,$A6,$A7
	.dc.b	$A3,$E5,$AF,$D6,$C6,$1F,$65,$73,$9D,$5F,$C0,$23,$E3,$3F,$B6,$FD
	.dc.b	$5B,$DB,$FD,$7B,$63,$FD,$AE,$D7,$F1,$BF,$27,$FA,$93,$73,$23,$F6
	.dc.b	$91,$FA,$DF,$92,$E7,$F3,$F2,$7F,$71,$19,$7F,$04,$FD,$D2,$3F,$C9
	.dc.b	$3E,$B3,$FD,$58,$DF,$5F,$F9,$3A,$8C,$BF,$EF,$40,$66,$3F,$11,$DF
	.dc.b	$FC,$2A,$77,$AB,$4F,$F7,$DC,$61,$FE,$29,$3F,$C1,$7D,$C9,$87,$F0
	.dc.b	$F7,$F8,$47,$B7,$23,$F8,$93,$5F,$F0,$BF,$D5,$BE,$8B,$F8,$77,$CC
	.dc.b	$49,$FE,$30,$FF,$11,$F8,$49,$1F,$C6,$DA,$F2,$F0,$BF,$90,$FD,$63
	.dc.b	$B6,$92,$7D,$FB,$FA,$DF,$83,$2F,$8A,$E2,$48,$DE,$FF,$19,$4E,$DC
	.dc.b	$9F,$2B,$47,$89,$FC,$CB,$EA,$FF,$C1,$13,$A0,$E1,$FF,$1B,$B7,$1F
	.dc.b	$24,$6B,$E0,$39,$11,$E5,$28,$FE,$7F,$F4,$FE,$F9,$CF,$3F,$84,$8F
	.dc.b	$0F,$7F,$C8,$47,$07,$97,$EB,$3D,$2F,$6F,$92,$F5,$11,$F0,$F9,$FE
	.dc.b	$D2,$38,$3C,$BE,$FF,$C1,$23,$83,$E1,$7D,$67,$F7,$0D,$7D,$FF,$59
	.dc.b	$F0,$1B,$C5,$FD,$6A,$30,$FD,$17,$FD,$8B,$5F,$2D,$AF,$F9,$81,$B3
	.dc.b	$D3,$E2,$FF,$B1,$FD,$63,$3E,$9F,$47,$30,$E4,$F7,$F7,$24,$75,$6B
	.dc.b	$FE,$6C,$E1,$FA,$D9,$F8,$FE,$AB,$CC,$49,$F2,$7C,$07,$A4,$8E,$0F
	.dc.b	$47,$C0,$7B,$48,$E0,$FC,$37,$D6,$3B,$BE,$17,$C9,$49,$F8,$0F,$C5
	.dc.b	$F8,$19,$1F,$0D,$AF,$9F,$A3,$E5,$7C,$BC,$8F,$90,$FC,$DF,$7E,$47
	.dc.b	$CB,$6B,$FE,$8A,$FA,$C6,$BF,$E0,$CF,$A0,$FF,$36,$CF,$E6,$DA,$FF
	.dc.b	$A4,$1B,$CD,$F9,$A7,$FC,$35,$1B,$F5,$79,$11,$E5,$EE,$75,$23,$83
	.dc.b	$F5,$1F,$05,$FC,$D6,$FA,$7A,$1E,$32,$4F,$8F,$D0,$F3,$11,$C1,$CB
	.dc.b	$CB,$F5,$79,$B2,$3D,$4F,$B7,$CD,$91,$ED,$6B,$FE,$7F,$E1,$F8,$5E
	.dc.b	$64,$8F,$84,$FC,$5F,$22,$47,$C5,$6B,$F1,$7D,$7F,$94,$47,$E5,$3F
	.dc.b	$E3,$8F,$2B,$FC,$71,$1F,$C1,$FE,$B1,$DF,$FE,$3C,$6C,$66,$31,$F0
	.dc.b	$F9,$F9,$A9,$3C,$5E,$1D,$3F,$E4,$4F,$AC,$7A,$74,$95,$FC,$56,$1F
	.dc.b	$09,$BE,$19,$39,$79,$70,$FF,$93,$BE,$B3,$F9,$14,$FF,$B4,$1B,$CD
	.dc.b	$FC,$69,$BE,$27,$F2,$95,$F2,$9C,$7E,$46,$5F,$F2,$59,$47,$5F,$9B
	.dc.b	$FF,$2D,$FB,$FF,$E3,$12,$7C,$06,$3C,$0C,$F8,$B3,$1F,$E5,$D4,$8E
	.dc.b	$3E,$2F,$F9,$93,$EB,$3F,$C7,$14,$69,$3E,$07,$5E,$04,$9F,$E7,$12
	.dc.b	$8E,$5C,$BC,$FC,$89,$5A,$7F,$CE,$3F,$59,$FD,$B7,$8F,$F9,$DA,$F9
	.dc.b	$BC,$92,$BF,$A2,$47,$F8,$1E,$7F,$A0,$FD,$67,$FA,$DA,$7F,$A4,$35
	.dc.b	$FF,$7E,$B6,$37,$AF,$FA,$03,$D0,$FF,$59,$D7,$F4,$F6,$A9,$FE,$9B
	.dc.b	$18,$6E,$FC,$7C,$39,$5C,$5E,$9E,$1F,$5F,$7E,$4F,$57,$C1,$AF,$FA
	.dc.b	$43,$EB,$1F,$33,$49,$5F,$D8,$63,$E1,$3F,$D8,$FE,$B3,$F9,$EB,$FE
	.dc.b	$12,$6A,$69,$FC,$F3,$B3,$FB,$92,$F9,$4F,$F4,$FB,$67,$A4,$63,$E4
	.dc.b	$33,$C1,$27,$C0,$F0,$E8,$91,$E3,$6B,$F1,$F0,$EF,$F2,$9D,$BC,$59
	.dc.b	$3E,$97,$C3,$E2,$C8,$F6,$35,$E4,$7B,$7D,$3F,$F5,$AF,$D6,$7F,$1B
	.dc.b	$C3,$FC,$18,$D9,$FF,$19,$B5,$2F,$3F,$89,$23,$FC,$19,$19,$7F,$C1
	.dc.b	$DE,$63,$8F,$FD,$87,$F1,$9F,$E0,$59,$F1,$87,$1D,$1F,$E1,$06,$E3
	.dc.b	$FE,$1F,$7F,$73,$3F,$E4,$66,$BC,$7E,$1B,$A9,$FC,$67,$9B,$FD,$A3
	.dc.b	$F5,$7F,$D0,$27,$BB,$70,$FF,$0E,$B7,$0F,$F1,$63,$F9,$36,$FA,$9C
	.dc.b	$89,$5F,$E2,$99,$FD,$C9,$3A,$FB,$9F,$E7,$28,$F0,$F8,$71,$E0,$F9
	.dc.b	$37,$91,$CD,$AF,$9B,$C2,$FD,$32,$FA,$CF,$F7,$B6,$1F,$C0,$73,$FF
	.dc.b	$34,$4F,$F9,$A2,$30,$E1,$AF,$F9,$B5,$AF,$BB,$C3,$7D,$67,$28,$4F
	.dc.b	$F3,$48,$C3,$63,$A4,$F0,$7F,$CF,$7F,$49,$FE,$7C,$4D,$23,$D7,$E4
	.dc.b	$BF,$CB,$D8,$7F,$93,$23,$FC,$C9,$BF,$F9,$4D,$4E,$A3,$FD,$09,$F1
	.dc.b	$96,$7F,$CC,$0D,$5E,$7B,$CA,$C4,$74,$3F,$1B,$19,$AE,$DC,$9A,$F3
	.dc.b	$F6,$D8,$F1,$47,$2F,$8E,$C5,$FE,$58,$39,$9F,$F3,$55,$BE,$B3,$15
	.dc.b	$E0,$7C,$DA,$39,$6F,$2E,$1B,$87,$F9,$CE,$7F,$DA,$6F,$4F,$03,$38
	.dc.b	$7A,$9F,$81,$F5,$9F,$1A,$7F,$CF,$4F,$BE,$BF,$CE,$EE,$6B,$F5,$AE
	.dc.b	$67,$FD,$10,$D8,$FF,$57,$E4,$EF,$EB,$C1,$F3,$63,$17,$DB,$C7,$F0
	.dc.b	$F9,$BE,$26,$C7,$8A,$3C,$BF,$23,$8C,$7C,$2F,$A3,$E3,$7D,$3E,$9D
	.dc.b	$BF,$4B,$87,$D7,$E9,$DB,$F4,$BF,$03,$E1,$BE,$0F,$C6,$5A,$F8,$2F
	.dc.b	$E0,$FE,$59,$ED,$DF,$AC,$C3,$6B,$FD,$41,$3E,$CC,$7F,$88,$AF,$2B
	.dc.b	$97,$17,$37,$F1,$DF,$59,$FE,$12,$BF,$F5,$42,$FA,$6C,$FB,$BA,$77
	.dc.b	$C9,$C3,$EB,$2D,$F0,$7F,$31,$F5,$9F,$E1,$08,$FF,$59,$3F,$8C,$6F
	.dc.b	$7E,$BB,$F7,$C7,$C3,$8E,$D7,$07,$F4,$5F,$58,$F4,$6F,$06,$1B,$5F
	.dc.b	$EB,$F3,$FC,$5E,$5D,$BF,$54,$E3,$8B,$F9,$9B,$EF,$7F,$9A,$FD,$67
	.dc.b	$FB,$D2,$3F,$D9,$0F,$E3,$7B,$BF,$D8,$AE,$67,$7E,$2F,$ED,$63,$FD
	.dc.b	$B5,$5C,$36,$C7,$89,$97,$5D,$5B,$DE,$D7,$67,$F7,$DE,$33,$DB,$BF
	.dc.b	$07,$86,$D7,$FB,$6E,$7F,$C5,$CD,$F1,$38,$FA,$DC,$9F,$E1,$1F,$59
	.dc.b	$FB,$39,$FF,$71,$3F,$B0,$DF,$FA,$9B,$96,$F9,$1C,$3D,$CE,$3F,$0D
	.dc.b	$6F,$8B,$D3,$D4,$FE,$29,$C7,$F0,$F9,$3F,$DD,$39,$7F,$AA,$6F,$FD
	.dc.b	$E1,$39,$7F,$DF,$4F,$F8,$CD,$CF,$8D,$CB,$C7,$E6,$FF,$1F,$FA,$C7
	.dc.b	$C9,$DF,$84,$C3,$6F,$FD,$F9,$F5,$9F,$95,$6F,$C3,$FF,$27,$8C,$70
	.dc.b	$FF,$70,$51,$93,$5E,$25,$AE,$6E,$15,$7F,$78,$FA,$7F,$F6,$D4,$68
	.dc.b	$CE,$E7,$72,$9D,$BE,$1F,$84,$F2,$71,$F9,$5B,$5E,$A8,$FE,$7D,$4F
	.dc.b	$37,$37,$E1,$7F,$A1,$7D,$67,$CB,$8F,$2B,$F9,$5B,$F0,$B9,$83,$97
	.dc.b	$7C,$5D,$BE,$67,$9B,$6B,$D6,$72,$EF,$EF,$E3,$F8,$1B,$1A,$BF,$0C
	.dc.b	$F9,$3D,$EC,$C7,$F8,$7B,$1F,$F0,$F5,$6E,$3C,$73,$EE,$FA,$AE,$13
	.dc.b	$9D,$11,$EE,$F9,$CF,$7F,$A9,$6B,$96,$3F,$E5,$E6,$C7,$C8,$FA,$CD
	.dc.b	$C7,$37,$E3,$FA,$BF,$23,$C6,$B3,$C9,$C3,$E5,$FA,$CF,$B4,$E4,$C1
	.dc.b	$F4,$38,$BE,$8F,$15,$8F,$1B,$87,$CB,$A5,$DF,$E1,$3F,$60,$FE,$F5
	.dc.b	$F5,$9F,$BD,$9D,$9F,$15,$C4,$FF,$91,$BE,$B3,$F4,$33,$F2,$57,$67
	.dc.b	$F8,$0D,$CC,$EE,$30,$7E,$2B,$8A,$D7,$CD,$78,$79,$1C,$7D,$7B,$3D
	.dc.b	$5C,$3F,$5F,$A3,$C7,$D5,$F9,$DC,$7F,$47,$97,$F5,$0A,$D7,$93,$E3
	.dc.b	$F9,$BF,$27,$CD,$D9,$F4,$38,$7D,$5F,$19,$70,$F7,$3E,$B3,$FC,$3D
	.dc.b	$97,$FB,$C2,$76,$78,$F4,$F8,$1E,$57,$C1,$F8,$BB,$3F,$15,$C3,$F8
	.dc.b	$FF,$97,$7D,$66,$32,$FF,$38,$C6,$27,$E6,$7B,$9F,$E2,$DE,$37,$BB
	.dc.b	$F3,$63,$0C,$9F,$CD,$74,$3E,$B3,$18,$FF,$9F,$23,$13,$F9,$FF,$AC
	.dc.b	$FE,$59,$CB,$F2,$69,$1F,$CC,$63,$0E,$A7,$F8,$4B,$96,$FA,$3F,$E4
	.dc.b	$1F,$AC,$FF,$58,$47,$58,$C7,$1F,$F9,$16,$BF,$AE,$5F,$4F,$63,$1F
	.dc.b	$0F,$96,$61,$FB,$3C,$5F,$23,$8C,$FF,$D2,$B1,$89,$F5,$79,$7D,$79
	.dc.b	$FB,$36,$3D,$D1,$FE,$55,$FA,$C7,$C7,$E7,$0F,$F4,$FC,$62,$BF,$81
	.dc.b	$7D,$67,$F9,$27,$8F,$FC,$E4,$DA,$FF,$3E,$BF,$E5,$DF,$3B,$3F,$15
	.dc.b	$F5,$27,$FD,$03,$83,$D9,$FF,$3D,$42,$30,$FF,$9F,$5A,$F0,$8D,$FD
	.dc.b	$67,$FA,$CE,$3F,$9B,$87,$0F,$F3,$A6,$BF,$D0,$9A,$B0,$FF,$09,$3E
	.dc.b	$AF,$80,$F1,$C6,$23,$D7,$1E,$6F,$AC,$FF,$5A,$47,$81,$AF,$F4,$86
	.dc.b	$BF,$D4,$1A,$9F,$F5,$C3,$9F,$A0,$FE,$9F,$CB,$F8,$D9,$CC,$EF,$7C
	.dc.b	$EF,$D7,$A2,$CF,$5F,$8D,$AF,$56,$DF,$FB,$84,$8E,$0C,$CC,$9F,$E7
	.dc.b	$93,$FE,$7F,$6C,$E5,$FF,$6F,$C7,$27,$27,$09,$D6,$1E,$9A,$ED,$39
	.dc.b	$21,$C9,$CB,$D2,$78,$DF,$C9,$9C,$D7,$C3,$9C,$D7,$BC,$F0,$9C,$4E
	.dc.b	$B1,$E1,$72,$F1,$73,$5D,$5C,$BF,$2D,$C9,$D4,$F3,$18,$3E,$C4,$F9
	.dc.b	$23,$94,$78,$48,$7E,$0E,$3F,$0C,$47,$FD,$AD,$68,$C4,$7F,$B8,$A3
	.dc.b	$0E,$A3,$35,$E9,$8E,$CF,$F1,$1F,$95,$7C,$4F,$C4,$9F,$29,$E7,$8C
	.dc.b	$BE,$9B,$FC,$15,$FE,$1A,$73,$3D,$30,$F8,$4D,$3F,$C4,$F1,$AC,$3C
	.dc.b	$44,$9D,$BE,$3C,$3C,$58,$76,$BF,$F0,$35,$F1,$BD,$92,$3F,$49,$87
	.dc.b	$8E,$F8,$BE,$BA,$E1,$3A,$3D,$51,$B6,$9E,$F9,$E2,$D3,$27,$AC,$4F
	.dc.b	$47,$F6,$5F,$29,$FA,$8F,$AC,$7B,$F4,$FD,$4C,$49,$FB,$2F,$F0,$12
	.dc.b	$3F,$73,$E5,$7F,$82,$11,$C1,$0F,$D4,$BF,$B4,$3C,$1E,$A7,$ED,$BB
	.dc.b	$BF,$73,$F5,$9F,$EF,$6A,$FF,$17,$35,$FD,$8C,$6D,$FC,$5B,$FC,$4C
	.dc.b	$8F,$E3,$BD,$EF,$E0,$24,$7F,$7B,$EF,$7F,$04,$23,$83,$EB,$FD,$17
	.dc.b	$F0,$8F,$98,$91,$FC,$11,$FE,$17,$F0,$D2,$3F,$83,$B5,$72,$7E,$57
	.dc.b	$C1,$FE,$27,$F5,$8F,$94,$A4,$AF,$F2,$54,$3E,$33,$64,$8C,$BF,$97
	.dc.b	$F1,$3F,$8E,$FD,$67,$F7,$79,$FF,$29,$35,$F0,$F4,$FF,$9E,$23,$19
	.dc.b	$7F,$77,$5F,$05,$C8,$95,$C9,$FE,$53,$F5,$1E,$78,$EB,$F4,$BE,$02
	.dc.b	$3E,$0E,$5F,$19,$1C,$1F,$E6,$5F,$AC,$67,$E8,$F9,$49,$3E,$87,$D7
	.dc.b	$E8,$48,$F5,$B5,$F5,$9F,$E9,$8C,$DF,$03,$F0,$89,$1F,$09,$FE,$83
	.dc.b	$F5,$8F,$33,$FA,$1F,$D6,$7F,$85,$F8,$3F,$C3,$0D,$E1,$FF,$4E,$B7
	.dc.b	$0F,$F5,$2B,$FE,$B6,$7F,$CF,$CD,$7C,$61,$C4,$F4,$F8,$DF,$EA,$3F
	.dc.b	$58,$D7,$C3,$F0,$13,$0E,$6F,$93,$DE,$91,$D9,$AF,$6E,$4F,$B3,$9F
	.dc.b	$9B,$EA,$BD,$64,$9F,$2F,$C0,$FB,$88,$E0,$F4,$FC,$0F,$C2,$23,$83
	.dc.b	$FD,$35,$F5,$8E,$EF,$8F,$F2,$92,$7E,$0B,$F2,$BE,$0E,$47,$C4,$6B
	.dc.b	$E7,$EB,$F9,$FF,$31,$23,$E4,$BF,$4B,$E0,$24,$7C,$C6,$BF,$33,$EB
	.dc.b	$1A,$FF,$7A,$FA,$0F,$F2,$FC,$FF,$AB,$DA,$FD,$26,$F3,$7F,$94,$DF
	.dc.b	$F0,$39,$3D,$5E,$47,$3F,$56,$57,$36,$BF,$E6,$AF,$4F,$C1,$E7,$C8
	.dc.b	$F0,$3E,$3F,$3E,$47,$8D,$AF,$F9,$CB,$CB,$F4,$79,$B2,$3D,$0F,$AF
	.dc.b	$CD,$91,$EB,$6B,$FE,$6F,$E1,$F8,$1E,$64,$8F,$80,$FC,$3F,$1A,$47
	.dc.b	$C3,$6B,$F0,$FD,$9F,$90,$47,$E4,$3F,$D7,$CF,$F1,$2F,$D6,$3B,$A9
	.dc.b	$27,$FD,$B9,$0F,$89,$DF,$B3,$27,$2F,$2E,$8F,$F1,$7F,$D6,$3D,$3A
	.dc.b	$4A,$FF,$73,$43,$E0,$B7,$C1,$27,$2F,$2E,$0F,$F1,$CF,$D6,$7F,$23
	.dc.b	$9F,$F7,$73,$5F,$D8,$B7,$A7,$FD,$24,$BE,$5B,$8F,$C8,$CA,$E2,$F4
	.dc.b	$F9,$CF,$F2,$2F,$BF,$FE,$D0,$9F,$F7,$D3,$5F,$DE,$B7,$0F,$F4,$F2
	.dc.b	$ED,$E3,$E1,$4A,$E2,$F4,$F0,$BF,$CA,$3F,$59,$FE,$49,$9F,$E0,$ED
	.dc.b	$C3,$FD,$50,$DF,$21,$32,$3F,$C4,$2F,$0E,$DE,$64,$77,$5C,$B8,$FB
	.dc.b	$F8,$B2,$73,$87,$45,$87,$EF,$5C,$9E,$0F,$AC,$FF,$43,$E1,$FE,$BB
	.dc.b	$5F,$39,$FE,$6A,$6C,$F0,$0E,$1F,$E4,$6F,$F8,$21,$D7,$D6,$7F,$AC
	.dc.b	$9B,$F1,$5F,$E7,$48,$C4,$7F,$2A,$5B,$7F,$9E,$3C,$EF,$F5,$7C,$F7
	.dc.b	$6B,$FC,$C1,$B8,$FF,$B4,$5F,$95,$1F,$29,$78,$3C,$C9,$1C,$FC,$1E
	.dc.b	$DE,$F4,$8E,$CF,$87,$7F,$F4,$4F,$D6,$3E,$66,$92,$BE,$14,$7C,$1C
	.dc.b	$BF,$DC,$5F,$59,$FD,$26,$4D,$27,$AB,$FE,$94,$5F,$1F,$49,$3F,$36
	.dc.b	$3E,$2D,$74,$DB,$81,$27,$5C,$FC,$C8,$E0,$E5,$F2,$9D,$FC,$49,$1D
	.dc.b	$5F,$17,$89,$23,$C0,$D7,$FA,$EE,$2F,$47,$FD,$57,$F5,$9F,$94,$CF
	.dc.b	$FE,$00,$37,$AB,$F1,$CD,$BF,$CD,$AF,$37,$89,$23,$F1,$51,$97,$FC
	.dc.b	$CB,$C7,$FE,$B8,$F9,$7F,$F0,$84,$7F,$6E,$6E,$0C,$3F,$39,$18,$6F
	.dc.b	$C1,$F1,$25,$7E,$5E,$7F,$83,$4F,$E8,$FE,$27,$F4,$53,$FD,$EB,$EA
	.dc.b	$BF,$D9,$A4,$F5,$3F,$D9,$2F,$E3,$AB,$F5,$4D,$9F,$DB,$B5,$34,$FC
	.dc.b	$9B,$8F,$13,$99,$1F,$D5,$7C,$57,$EC,$2B,$AF,$A9,$E0,$24,$FE,$BE
	.dc.b	$FE,$33,$E7,$DF,$CA,$47,$5E,$2F,$F9,$41,$7D,$63,$5F,$4C,$E6,$7F
	.dc.b	$82,$1E,$87,$0E,$B7,$CB,$EE,$8C,$BF,$03,$C6,$7D,$66,$A3,$E0,$C6
	.dc.b	$1E,$39,$7F,$43,$9D,$36,$38,$0F,$A5,$AD,$32,$E0,$DF,$C1,$7E,$06
	.dc.b	$3F,$C6,$27,$F9,$AF,$6F,$E2,$8F,$F8,$FD,$BF,$17,$BC,$F8,$4F,$F2
	.dc.b	$E8,$C4,$FF,$25,$F4,$3F,$9E,$3C,$0F,$E3,$CE,$27,$BF,$2F,$07,$F9
	.dc.b	$67,$9F,$5F,$E7,$31,$5F,$EA,$55,$E4,$9C,$7D,$16,$5B,$1E,$88,$E5
	.dc.b	$6F,$94,$CD,$FF,$93,$A3,$15,$FC,$C7,$27,$37,$85,$FF,$75,$46,$63
	.dc.b	$1C,$B2,$3D,$7E,$05,$C3,$A3,$E9,$FA,$CF,$91,$1F,$E5,$C7,$E6,$37
	.dc.b	$7D,$6F,$2F,$E2,$F7,$DA,$F3,$3C,$FC,$6E,$9E,$75,$AF,$E9,$18,$FF
	.dc.b	$4C,$6C,$78,$7E,$43,$71,$A7,$C9,$F1,$BE,$5F,$52,$CF,$27,$0F,$4F
	.dc.b	$52,$E7,$E9,$7D,$9F,$0D,$ED,$F8,$CB,$5E,$27,$EC,$7C,$B7,$F8,$76
	.dc.b	$3F,$AE,$2F,$AC,$CF,$BB,$A7,$7C,$1C,$3C,$9B,$7C,$1F,$C1,$7D,$67
	.dc.b	$F8,$76,$3F,$B3,$2F,$AC,$CF,$BD,$A7,$7D,$7C,$3E,$56,$DF,$07,$F1
	.dc.b	$9F,$59,$F8,$AE,$1F,$F4,$7A,$EF,$CF,$81,$A7,$7C,$7C,$38,$EE,$3F
	.dc.b	$DD,$EF,$FD,$24,$FF,$79,$FA,$C7,$A3,$F9,$95,$C3,$63,$0F,$E7,$67
	.dc.b	$FC,$78,$7F,$80,$4F,$76,$7F,$EA,$27,$2D,$DF,$F4,$9F,$59,$FE,$1C
	.dc.b	$9F,$F0,$43,$F8,$EF,$0B,$F6,$6E,$6B,$FD,$56,$E6,$7F,$C3,$0F,$FB
	.dc.b	$8D,$E3,$B7,$4F,$13,$AB,$DB,$A7,$BE,$F6,$BB,$BF,$B5,$F1,$5E,$DD
	.dc.b	$F8,$5C,$36,$BF,$C3,$93,$FE,$12,$6F,$89,$C7,$D6,$E4,$FE,$FF,$EB
	.dc.b	$3F,$63,$3F,$E2,$37,$F6,$3B,$7F,$B1,$1C,$B7,$C7,$E1,$EE,$71,$F8
	.dc.b	$7B,$7C,$5E,$9E,$EF,$F0,$BF,$2B,$E2,$72,$7F,$18,$CB,$FD,$69,$7F
	.dc.b	$E3,$19,$CB,$FE,$3D,$7F,$59,$97,$FB,$51,$78,$DC,$FC,$7E,$8F,$F1
	.dc.b	$7F,$AC,$7B,$B7,$E8,$B0,$DC,$7F,$C7,$DF,$59,$F9,$96,$FC,$47,$F1
	.dc.b	$F8,$C6,$7F,$EE,$25,$19,$35,$E2,$5A,$E6,$E1,$FF,$25,$7D,$67,$E5
	.dc.b	$2F,$94,$FF,$2B,$8C,$6F,$FC,$E3,$0F,$F1,$DD,$F9,$38,$BC,$5F,$07
	.dc.b	$E3,$64,$74,$F3,$6E,$5F,$EF,$6C,$51,$FC,$DF,$EB,$3E,$4C,$FF,$9C
	.dc.b	$1F,$C8,$34,$D7,$14,$BF,$3F,$CC,$D9,$E8,$F3,$EF,$E9,$E4,$5A,$FF
	.dc.b	$33,$61,$FB,$76,$C7,$87,$DA,$DC,$69,$C3,$F1,$3E,$AA,$E3,$E9,$7D
	.dc.b	$1F,$3B,$E9,$F5,$6D,$76,$70,$FC,$CF,$AC,$B7,$F0,$5F,$81,$EA,$FC
	.dc.b	$1F,$1A,$D7,$99,$C3,$FE,$7D,$FA,$CF,$F1,$73,$93,$07,$C9,$E1,$F9
	.dc.b	$5C,$16,$3D,$EE,$1F,$83,$A5,$C9,$CD,$FD,$CB,$81,$3F,$4C,$FE,$D5
	.dc.b	$F5,$9F,$B6,$BE,$27,$FA,$3D,$C5,$F2,$8F,$F4,$E7,$D6,$7E,$69,$FF
	.dc.b	$4A,$2E,$CF,$EE,$8E,$67,$81,$FE,$AE,$75,$3C,$27,$F2,$1C,$5C,$1B
	.dc.b	$1D,$3C,$8E,$7E,$BF,$46,$C7,$82,$3F,$C0,$5D,$3E,$6D,$7F,$C1,$F8
	.dc.b	$FF,$4D,$CB,$C2,$27,$57,$EA,$F9,$BF,$3F,$CD,$FA,$1B,$1E,$C8,$F5
	.dc.b	$FC,$66,$2F,$9E,$1F,$E8,$AA,$F8,$33,$E2,$CF,$71,$FE,$B7,$C7,$FD
	.dc.b	$FC,$FC,$6F,$27,$E2,$FC,$67,$C6,$6C,$7C,$A8,$FF,$0E,$7D,$5B,$EB
	.dc.b	$33,$9F,$AA,$31,$5F,$43,$DC,$FF,$11,$70,$BD,$EF,$A3,$18,$64,$FF
	.dc.b	$B2,$7A,$1F,$59,$8C,$7E,$2C,$62,$7F,$D9,$DF,$59,$FD,$3B,$93,$FC
	.dc.b	$72,$D8,$FF,$6E,$3F,$D4,$23,$3E,$BF,$F8,$CF,$EB,$1F,$29,$4F,$F1
	.dc.b	$B5,$71,$C7,$FD,$DF,$3A,$C3,$FD,$B8,$BE,$83,$91,$CF,$DC,$B3,$E4
	.dc.b	$F8,$1E,$66,$2F,$EE,$E1,$FC,$93,$3F,$F2,$7F,$D6,$3E,$BF,$FC,$A3
	.dc.b	$5F,$EF,$38,$C6,$1F,$E6,$96,$C7,$F9,$C1,$AF,$F9,$DB,$EB,$3F,$C8
	.dc.b	$9C,$3F,$E6,$F6,$C7,$F9,$D9,$7C,$DB,$CD,$D7,$C5,$FD,$3A,$FF,$3D
	.dc.b	$FD,$67,$F3,$67,$FD,$F5,$8D,$7F,$2A,$DD,$3F,$CC,$F3,$73,$54,$8C
	.dc.b	$38,$AA,$9B,$13,$D4,$E3,$3D,$9B,$7F,$19,$B7,$AA,$E1,$5E,$69,$CB
	.dc.b	$DD,$AA,$36,$F5,$38,$1E,$8A,$F5,$CF,$09,$F3,$DC,$8C,$12,$B8,$9C
	.dc.b	$CF,$DC,$CF,$6A,$F4,$46,$9E,$AF,$7B,$F5,$57,$9A,$FB,$4E,$4F,$95
	.dc.b	$1D,$DE,$D5,$D1,$EF,$19,$8C,$12,$BA,$CF,$33,$E1,$4F,$72,$39,$BF
	.dc.b	$44,$FE,$46,$73,$3D,$8D,$38,$AE,$D1,$E6,$32,$7B,$E7,$93,$83,$91
	.dc.b	$EA,$72,$E2,$3E,$2B,$93,$ED,$1C,$5C,$9E,$07,$F1,$53,$F0,$67,$DF
	.dc.b	$1F,$9A,$8F,$F6,$4D,$7E,$66,$3F,$DB,$6D,$76,$62,$36,$47,$FD,$B1
	.dc.b	$1F,$11,$F7,$1F,$7B,$D4,$F0,$38,$BF,$2D,$FE,$56,$73,$3F,$40,$F5
	.dc.b	$46,$5A,$71,$7D,$33,$FA,$C9,$E3,$5E,$1A,$FC,$D4,$62,$F2,$D8,$E6
	.dc.b	$68,$B3,$C4,$46,$E3,$E8,$5F,$8D,$A7,$8C,$87,$33,$27,$AC,$B3,$D1
	.dc.b	$EF,$5E,$46,$9E,$2A,$F0,$5F,$C6,$8F,$E0,$D1,$8B,$F8,$27,$8E,$75
	.dc.b	$3D,$DB,$7D,$E3,$C3,$7D,$DA,$77,$3F,$41,$F5,$8F,$1B,$F8,$6C,$48
	.dc.b	$C3,$83,$F8,$CB,$87,$0D,$7F,$13,$F1,$1F,$C3,$23,$58,$31,$FE,$33
	.dc.b	$18,$7F,$8B,$4F,$F1,$2F,$AC,$7C,$25,$F8,$D9,$8F,$ED,$67,$2D,$9E
	.dc.b	$04,$7F,$75,$F2,$FF,$E0,$57,$F7,$9F,$2C,$F4,$BF,$8F,$B6,$33,$1E
	.dc.b	$1F,$62,$46,$08,$7F,$84,$30,$47,$F2,$8F,$AB,$FE,$E3,$3F,$C2,$5A
	.dc.b	$FF,$85,$9B,$B7,$C1,$8F,$E8,$30,$F6,$39,$12,$39,$3F,$CD,$3E,$A1
	.dc.b	$AF,$7F,$A4,$95,$DD,$F1,$7B,$12,$3C,$4D,$7F,$C4,$FF,$59,$F0,$DF
	.dc.b	$27,$C8,$7A,$48,$FA,$7C,$7F,$61,$1C,$1E,$DF,$67,$E0,$11,$C1,$FC
	.dc.b	$A7,$EB,$3F,$B4,$3F,$CA,$FE,$B3,$E0,$3F,$E3,$86,$EF,$86,$F3,$FF
	.dc.b	$5C,$7F,$D5,$73,$C7,$E0,$FF,$AA,$C6,$23,$94,$7F,$58,$FA,$C6,$7A
	.dc.b	$F8,$E9,$5E,$17,$8F,$AD,$23,$C8,$D7,$FC,$97,$C5,$F5,$B2,$F1,$7D
	.dc.b	$5F,$80,$93,$EB,$F6,$FC,$64,$70,$7B,$FD,$BF,$39,$1C,$1F,$D3,$7C
	.dc.b	$E7,$C5,$FB,$3E,$3A,$4F,$C7,$7D,$FE,$D4,$8F,$94,$D7,$CF,$D1,$F1
	.dc.b	$3E,$4A,$47,$CF,$7E,$3F,$AF,$23,$E9,$35,$FF,$34,$FD,$63,$5F,$F0
	.dc.b	$0F,$9D,$FE,$55,$9F,$EC,$4D,$7F,$CD,$ED,$E7,$FB,$6B,$E8,$38,$92
	.dc.b	$78,$BD,$3E,$B2,$47,$46,$BF,$E6,$1F,$92,$F0,$FD,$0C,$8F,$0B,$E4
	.dc.b	$FA,$09,$1E,$46,$BF,$E6,$9F,$AC,$F4,$F9,$F2,$3D,$2F,$B3,$D1,$91
	.dc.b	$EC,$6B,$FE,$66,$E1,$F8,$3F,$1F,$23,$E0,$BF,$13,$E1,$24,$7C,$46
	.dc.b	$BF,$13,$E5,$7E,$49,$1F,$92,$FF,$8B,$3C,$F3,$FE,$2E,$FA,$C7,$7D
	.dc.b	$25,$7F,$84,$61,$F1,$3B,$F7,$E4,$E5,$E5,$D3,$FE,$38,$FA,$C7,$D1
	.dc.b	$D2,$57,$F8,$6E,$1F,$09,$BE,$19,$39,$79,$70,$FF,$90,$BE,$B3,$F9
	.dc.b	$6C,$FF,$AB,$9B,$D5,$FB,$A6,$F8,$9F,$E5,$27,$F9,$1C,$FF,$8A,$5F
	.dc.b	$E5,$8F,$1F,$57,$A1,$D7,$E6,$FF,$C9,$FF,$45,$FE,$CD,$9F,$F1,$83
	.dc.b	$5F,$F5,$EB,$70,$FE,$12,$FF,$9D,$23,$7E,$1F,$22,$3C,$BC,$3F,$F2
	.dc.b	$DF,$D6,$2A,$92,$7F,$C7,$D0,$EE,$DF,$6C,$9C,$BC,$B8,$B9,$F8,$B2
	.dc.b	$78,$BF,$E6,$8F,$AC,$FE,$CF,$A7,$F1,$E5,$F3,$99,$25,$7F,$93,$61
	.dc.b	$F3,$9F,$E7,$0F,$AC,$FF,$56,$CF,$F9,$45,$AF,$FB,$79,$B7,$FC,$A9
	.dc.b	$6F,$FE,$78,$F3,$9E,$2D,$25,$FF,$96,$E3,$E1,$37,$C2,$E0,$47,$5E
	.dc.b	$17,$3F,$A9,$91,$C5,$ED,$B7,$FA,$1B,$EB,$1F,$37,$49,$5F,$E6,$98
	.dc.b	$F8,$2F,$F9,$AF,$EB,$3F,$9E,$BF,$EA,$06,$A6,$9F,$CF,$76,$FE,$8E
	.dc.b	$BE,$5B,$FD,$2A,$D9,$E9,$18,$F9,$1C,$EF,$27,$C0,$F0,$E9,$91,$E3
	.dc.b	$6A,$C3,$BF,$CB,$76,$F1,$24,$FA,$5F,$0F,$89,$23,$D8,$D7,$97,$81
	.dc.b	$FE,$B4,$D1,$E8,$FF,$AB,$3E,$B3,$F1,$D5,$FE,$8D,$6B,$C0,$FF,$61
	.dc.b	$B8,$CB,$FB,$3A,$F3,$B9,$12,$79,$3F,$C3,$78,$FF,$D7,$3F,$13,$FE
	.dc.b	$06,$9F,$F4,$D3,$5E,$C7,$F8,$F7,$E0,$77,$F0,$F2,$36,$D7,$7E,$7F
	.dc.b	$F5,$67,$FD,$49,$F5,$5F,$CF,$DB,$E9,$FF,$B2,$23,$11,$D9,$FC,$9B
	.dc.b	$C3,$B7,$44,$79,$F8,$DC,$48,$E0,$FF,$1B,$9D,$7C,$58,$F8,$9F,$06
	.dc.b	$B2,$3C,$AD,$7C,$5E,$2F,$DD,$5F,$59,$FE,$57,$C5,$39,$9F,$F2,$FC
	.dc.b	$63,$87,$FC,$BD,$08,$C3,$FE,$61,$6B,$C3,$C6,$7D,$67,$09,$FE,$66
	.dc.b	$D8,$CC,$F1,$6B,$DB,$E9,$BB,$A0,$E1,$A6,$9D,$FC,$8F,$F9,$2F,$0F
	.dc.b	$F1,$6C,$FF,$91,$BA,$3F,$DB,$4A,$75,$1F,$E7,$CE,$3B,$3D,$CE,$07
	.dc.b	$E1,$BC,$E7,$B1,$A8,$F2,$4D,$9C,$D7,$6E,$5D,$57,$F9,$92,$30,$78
	.dc.b	$BE,$2B,$26,$BE,$8A,$C7,$99,$C3,$FE,$F1,$B7,$C9,$62,$FC,$8F,$8F
	.dc.b	$4F,$E7,$95,$FE,$F7,$7F,$9D,$37,$97,$5C,$B2,$3A,$78,$37,$0F,$73
	.dc.b	$ED,$FA,$CF,$8D,$3F,$F0,$05,$F8,$78,$7F,$96,$1C,$D7,$EA,$9C,$CF
	.dc.b	$C9,$6C,$7D,$9C,$9E,$07,$5D,$FE,$43,$DF,$C7,$F0,$79,$F7,$1F,$03
	.dc.b	$87,$C5,$E7,$5C,$7C,$0F,$9F,$E2,$FD,$1E,$9D,$AF,$43,$87,$D5,$E9
	.dc.b	$DB,$F4,$3E,$FF,$84,$F8,$1F,$15,$6B,$E0,$3F,$7F,$E5,$7F,$C2,$31
	.dc.b	$F9,$3E,$1F,$F3,$CC,$3E,$B3,$5E,$E6,$DE,$1C,$1C,$7C,$AB,$87,$13
	.dc.b	$F1,$D7,$FC,$91,$C6,$9F,$9F,$FA,$CF,$F0,$AC,$62,$10,$F9,$AF,$C9
	.dc.b	$3A,$8F,$E3,$6E,$67,$83,$FE,$F8,$6F,$CA,$70,$E6,$E2,$FE,$6F,$EB
	.dc.b	$3F,$2D,$7F,$C9,$17,$6E,$7B,$B4,$EF,$8F,$87,$1D,$CB,$CD,$7F,$E9
	.dc.b	$07,$D1,$F5,$8F,$47,$F5,$0F,$E0,$A3,$31,$8E,$1F,$D6,$47,$F8,$E0
	.dc.b	$F8,$07,$B7,$93,$FD,$4E,$E7,$4F,$E7,$2E,$4F,$DC,$7D,$67,$F0,$77
	.dc.b	$E4,$3F,$96,$9F,$F5,$7B,$99,$FE,$82,$E6,$78,$F8,$5F,$BD,$8F,$87
	.dc.b	$5C,$23,$1B,$67,$A3,$5B,$5C,$3B,$BF,$C1,$7C,$57,$B7,$7E,$13,$0D
	.dc.b	$AF,$B9,$3F,$E1,$86,$F8,$5C,$7D,$6E,$4F,$F0,$FF,$AC,$FD,$EE,$FF
	.dc.b	$D6,$1F,$DE,$B7,$C6,$D7,$8D,$BF,$73,$87,$C3,$5C,$BF,$15,$87,$F9
	.dc.b	$C1,$BE,$EF,$F1,$BF,$33,$E2,$72,$7F,$B5,$2B,$FD,$67,$5F,$91,$9C
	.dc.b	$BF,$C9,$5F,$F1,$DB,$9F,$1B,$97,$8F,$CD,$FE,$4D,$F5,$8F,$91,$BF
	.dc.b	$45,$86,$DF,$E7,$7E,$B3,$F3,$1C,$3F,$DD,$17,$C3,$FF,$2F,$73,$AF
	.dc.b	$FB,$A1,$CD,$71,$9D,$78,$9B,$6C,$75,$8F,$E7,$1F,$59,$F9,$5C,$3B
	.dc.b	$46,$3C,$3E,$DF,$09,$DF,$92,$E1,$E5,$70,$FE,$CF,$07,$9B,$8C,$7F
	.dc.b	$C1,$71,$89,$FD,$C7,$D5,$7C,$89,$FE,$B4,$FE,$3D,$A7,$F4,$B7,$09
	.dc.b	$B2,$F9,$8D,$72,$7A,$9E,$1E,$0F,$1F,$22,$E3,$E0,$70,$F8,$BD,$FB
	.dc.b	$97,$F8,$83,$24,$78,$4F,$8B,$F5,$56,$BD,$CF,$97,$E6,$FC,$DE,$A5
	.dc.b	$9F,$6B,$87,$E2,$7D,$65,$C7,$E1,$BE,$DF,$57,$DD,$E4,$5A,$F8,$AE
	.dc.b	$1F,$E3,$DF,$58,$71,$7F,$E3,$68,$C4,$7C,$CE,$2F,$91,$C1,$F2,$5B
	.dc.b	$1F,$42,$3E,$66,$98,$BF,$80,$E2,$7F,$96,$7D,$67,$ED,$9F,$AF,$1F
	.dc.b	$DF,$63,$13,$FE,$46,$FA,$CF,$CD,$57,$F3,$35,$DB,$FE,$03,$73,$3B
	.dc.b	$8C,$1F,$90,$E0,$B5,$F8,$97,$87,$95,$C7,$D8,$B3,$D5,$C3,$FD,$07
	.dc.b	$A3,$C7,$DD,$F1,$71,$FF,$12,$65,$FD,$51,$FF,$2D,$CE,$AB,$FC,$35
	.dc.b	$2F,$9A,$F3,$7C,$D7,$9D,$B3,$EB,$8F,$57,$C6,$E3,$0F,$80,$E2,$7C
	.dc.b	$59,$D9,$EE,$FC,$1F,$2F,$E1,$7C,$65,$AF,$8C,$E1,$FE,$B5,$F5,$6F
	.dc.b	$AC,$C6,$1F,$E7,$28,$C4,$FC,$DF,$77,$FC,$5D,$C4,$F7,$7E,$74,$61
	.dc.b	$93,$FD,$97,$9D,$F5,$98,$C7,$FC,$FB,$18,$9F,$ED,$5F,$57,$FD,$9E
	.dc.b	$7F,$C7,$C1,$C3,$FE,$41,$6B,$FE,$51,$6A,$FA,$C7,$C9,$7F,$93,$A3
	.dc.b	$FC,$AD,$18,$AF,$F3,$3B,$63,$B9,$C6,$78,$3C,$FE,$7E,$C9,$A4,$AF
	.dc.b	$F3,$8D,$FF,$26,$AF,$4B,$5F,$47,$C1,$74,$8C,$47,$F9,$A7,$EA,$FF
	.dc.b	$C9,$3A,$7F,$9E,$9B,$1E,$47,$FC,$BB,$E4,$67,$E1,$FE,$84,$FA,$3E
	.dc.b	$AF,$FD,$CE,$DF,$67,$FC,$EB,$18,$8F,$96,$D7,$FC,$D5,$F5,$6F,$43
	.dc.b	$FC,$D9,$DF,$F9,$99,$4F,$F3,$E7,$FC,$3C,$D7,$FD,$6A,$F1,$73,$3C
	.dc.b	$B6,$EB,$E8,$D9,$FF,$43,$57,$1B,$EC,$F1,$8E,$77,$B7,$C3,$56,$78
	.dc.b	$5F,$A2,$B0,$66,$65,$78,$88,$79,$9C,$BD,$8C,$DF,$8A,$32,$F9,$27
	.dc.b	$AC,$FC,$09,$CD,$78,$63,$8B,$EE,$74,$60,$F7,$CF,$07,$E0,$CF,$47
	.dc.b	$C4,$79,$4E,$26,$2B,$C4,$E1,$F3,$3D,$4F,$AB,$3D,$5F,$8E,$66,$FE
	.dc.b	$64,$F0,$8E,$C7,$5B,$F5,$38,$8D,$1E,$07,$47,$A0,$F1,$47,$08,$D1
	.dc.b	$C1,$F8,$51,$C0,$F2,$9C,$0E,$46,$67,$CB,$1C,$27,$94,$FC,$83,$EC
	.dc.b	$9F,$C0,$A7,$F0,$CD,$AF,$F6,$8C,$62,$3D,$66,$4F,$F7,$1B,$97,$A1
	.dc.b	$FB,$E7,$C2,$E2,$7D,$75,$FE,$CD,$F9,$37,$D0,$7D,$79,$CD,$6E,$B5
	.dc.b	$39,$7E,$21,$23,$FD,$C9,$5C,$9F,$A9,$86,$DC,$D5,$7E,$39,$F2,$67
	.dc.b	$83,$F1,$E7,$51,$8B,$F1,$CF,$06,$C6,$6F,$C5,$39,$6C,$66,$3D,$44
	.dc.b	$34,$79,$0F,$39,$F8,$F7,$83,$A2,$61,$F2,$94,$6A,$30,$4B,$FD,$E4
	.dc.b	$65,$F8,$11,$C6,$31,$3E,$01,$7F,$C4,$E3,$31,$F0,$CF,$F5,$83,$88
	.dc.b	$F8,$73,$CF,$0E,$B7,$F2,$1B,$7E,$B7,$F8,$67,$C3,$FF,$0F,$FA,$BF
	.dc.b	$F1,$7C,$FF,$AE,$DA,$FF,$1E,$8D,$BF,$8C,$7F,$22,$47,$FA,$48,$FA
	.dc.b	$A7,$C2,$62,$30,$D8,$CC,$70,$6B,$FD,$49,$B9,$7F,$33,$6E,$89,$91
	.dc.b	$87,$F1,$FE,$37,$F2,$98,$7A,$9F,$CA,$FE,$AD,$E7,$D3,$51,$2F,$C4
	.dc.b	$F9,$48,$FB,$3E,$AF,$FB,$CB,$FE,$DF,$6B,$F5,$B1,$F2,$28,$CE,$5F
	.dc.b	$D2,$D7,$A9,$FC,$F5,$B3,$CE,$30,$7F,$41,$FA,$77,$C7,$EF,$E9,$65
	.dc.b	$78,$5E,$5E,$AC,$8F,$23,$5F,$21,$F1,$FE,$AD,$C1,$E1,$F7,$64,$FA
	.dc.b	$DF,$27,$8B,$23,$DC,$D7,$99,$ED,$F5,$BD,$44,$7E,$23,$FD,$67,$EA
	.dc.b	$DE,$07,$F5,$BF,$AB,$FE,$D9,$C3,$FD,$B5,$BC,$5F,$21,$B8,$7D,$D7
	.dc.b	$FD,$68,$FD,$F9,$E1,$F0,$3C,$C9,$19,$AF,$A3,$F5,$6D,$3C,$1F,$19
	.dc.b	$2B,$47,$57,$C9,$E9,$C8,$C8,$C9,$19,$8F,$83,$C9,$F7,$31,$F4,$7D
	.dc.b	$57,$B4,$93,$CA,$FF,$CB,$8D,$F5,$FE,$11,$1D,$7A,$FF,$14,$8E,$0F
	.dc.b	$1F,$9A,$F8,$9F,$95,$F1,$92,$7B,$3F,$37,$D8,$91,$E1,$6B,$E8,$EA
	.dc.b	$FA,$5F,$21,$23,$CC,$FD,$5F,$56,$47,$A5,$AF,$E0,$3E,$AD,$B7,$F8
	.dc.b	$57,$E6,$3F,$CC,$D3,$F2,$1A,$FE,$19,$BC,$FF,$E2,$B5,$F4,$1C,$49
	.dc.b	$3C,$5E,$9F,$41,$23,$A3,$5F,$F3,$97,$C8,$F8,$7E,$82,$47,$85,$F2
	.dc.b	$7C,$FC,$8F,$23,$5F,$F3,$D7,$9D,$E9,$F4,$A4,$7A,$5F,$67,$A1,$23
	.dc.b	$D8,$D7,$FC,$EF,$C3,$F0,$7E,$36,$47,$C1,$7E,$27,$8B,$23,$E2,$35
	.dc.b	$F8,$9F,$27,$F2,$48,$FC,$97,$FA,$29,$FE,$41,$FA,$BF,$C8,$C7,$E5
	.dc.b	$1A,$FE,$C9,$BC,$3F,$E5,$75,$F0,$DC,$7E,$62,$57,$17,$A7,$3F,$F9
	.dc.b	$37,$EA,$DF,$3F,$49,$5F,$A2,$87,$D5,$EF,$82,$4E,$5E,$5C,$1F,$E5
	.dc.b	$7F,$AB,$FE,$65,$26,$93,$D3,$D7,$A9,$27,$F5,$F3,$FC,$D1,$BF,$37
	.dc.b	$FE,$61,$F7,$BF,$DB,$31,$FB,$46,$BF,$C4,$9B,$C3,$FE,$77,$5D,$9C
	.dc.b	$7C,$19,$5C,$5E,$9E,$0F,$F9,$B7,$EA,$FF,$D0,$F8,$7F,$9F,$DB,$E4
	.dc.b	$7F,$3A,$6D,$FF,$4F,$5D,$DC,$7B,$64,$FF,$08,$94,$75,$E4,$ED,$E2
	.dc.b	$4A,$D3,$FE,$7D,$FA,$BF,$F5,$EE,$5F,$E8,$F5,$F3,$79,$25,$7F,$14
	.dc.b	$87,$CD,$FF,$A1,$7E,$AF,$FD,$7B,$3F,$C6,$1A,$FF,$36,$6D,$FF,$A6
	.dc.b	$D6,$FF,$E8,$DF,$3D,$E2,$D2,$5F,$F2,$28,$F8,$29,$78,$5C,$49,$1B
	.dc.b	$F0,$BA,$7D,$44,$8E,$4F,$7D,$BF,$D3,$9F,$56,$F9,$9A,$4A,$FE,$63
	.dc.b	$1F,$09,$FE,$67,$F5,$7F,$D1,$13,$49,$EB,$7F,$A9,$D7,$CA,$52,$4F
	.dc.b	$2F,$8F,$D6,$F2,$3A,$BC,$7A,$A4,$78,$1A,$FF,$60,$B7,$CA,$77,$F1
	.dc.b	$E4,$F9,$9F,$17,$8F,$23,$D2,$D7,$97,$A3,$FE,$BD,$FA,$BF,$C5,$C7
	.dc.b	$F5,$06,$BF,$DB,$9B,$CF,$FE,$D3,$5E,$7F,$12,$4F,$17,$FC,$9B,$C7
	.dc.b	$FE,$C9,$F9,$7F,$F0,$4C,$FF,$5C,$6B,$FD,$E9,$B7,$FE,$DF,$7F,$72
	.dc.b	$4F,$87,$E4,$7F,$96,$39,$7F,$DA,$9F,$55,$FC,$EA,$FF,$B3,$B5,$FF
	.dc.b	$02,$B6,$FF,$DD,$4F,$E3,$A3,$7D,$5C,$88,$F2,$F1,$7A,$91,$C1,$FD
	.dc.b	$D3,$E1,$FF,$4F,$E2,$FF,$8B,$AF,$E2,$B7,$C5,$F2,$9E,$6D,$A5,$F9
	.dc.b	$9A,$F9,$BC,$4F,$F3,$6A,$FA,$BF,$F2,$FE,$1F,$BA,$D3,$FC,$B6,$D9
	.dc.b	$FF,$1E,$9F,$E5,$17,$0E,$1A,$F9,$7C,$57,$D5,$F2,$6C,$72,$13,$FC
	.dc.b	$CA,$74,$7F,$99,$E1,$F4,$B4,$8F,$F9,$AD,$FF,$36,$7C,$13,$5F,$F0
	.dc.b	$CC,$FF,$92,$1B,$19,$8F,$F1,$C9,$92,$FC,$46,$72,$D4,$3D,$0F,$E5
	.dc.b	$77,$D2,$B4,$FF,$97,$A3,$33,$D3,$93,$2F,$F9,$8B,$D0,$AF,$F9,$DE
	.dc.b	$2B,$FE,$74,$AF,$3C,$E3,$E8,$72,$D8,$F5,$C7,$2B,$7C,$96,$6F,$FC
	.dc.b	$A7,$18,$AF,$F1,$CE,$4E,$5F,$0B,$FC,$E2,$33,$18,$E4,$91,$FE,$40
	.dc.b	$A7,$81,$8C,$3A,$BE,$AF,$AB,$FF,$2D,$D7,$F9,$19,$F9,$58,$FC,$87
	.dc.b	$35,$FE,$66,$73,$3F,$E8,$46,$C7,$FA,$66,$DD,$DD,$77,$F7,$3D,$FC
	.dc.b	$7F,$07,$9D,$71,$F0,$38,$7C,$5E,$6D,$C7,$C0,$F9,$FE,$2F,$D1,$EA
	.dc.b	$5A,$F4,$38,$7D,$5E,$9D,$BF,$43,$EF,$F8,$6F,$81,$F1,$56,$BE,$03
	.dc.b	$F7,$3E,$57,$FC,$1F,$1F,$E6,$85,$F5,$79,$F7,$34,$EF,$7E,$1E,$4D
	.dc.b	$BE,$0F,$E1,$BE,$AF,$FC,$1F,$1F,$E7,$25,$F3,$39,$F7,$74,$EF,$97
	.dc.b	$87,$C9,$DB,$E0,$FE,$43,$EA,$FF,$1D,$C3,$FE,$98,$5D,$D9,$EF,$D3
	.dc.b	$BE,$3E,$1C,$77,$1F,$F4,$1D,$FF,$A2,$DF,$F4,$2F,$D5,$BD,$0F,$CE
	.dc.b	$BF,$6A,$33,$18,$E1,$FD,$0C,$7F,$8B,$8F,$F4,$69,$EE,$CF,$FD,$56
	.dc.b	$E5,$BB,$FE,$A3,$EA,$FF,$DB,$D3,$FE,$94,$7F,$1B,$DD,$FB,$67,$35
	.dc.b	$FE,$B4,$73,$3F,$ED,$27,$FD,$E2,$F1,$DB,$A7,$87,$D5,$ED,$D3,$DF
	.dc.b	$7B,$5D,$DF,$DC,$F8,$BF,$E0,$18,$FF,$52,$BF,$EA,$46,$FB,$7A,$F8
	.dc.b	$FD,$FA,$DC,$3C,$4B,$5C,$1F,$E0,$7F,$57,$FB,$58,$FF,$57,$BF,$B5
	.dc.b	$DB,$FD,$9A,$E5,$BE,$37,$0F,$73,$8F,$C3,$5B,$E2,$F4,$F5,$3F,$87
	.dc.b	$F9,$9F,$13,$93,$F8,$DE,$5F,$EA,$9B,$FF,$5E,$CE,$5F,$F7,$BB,$FE
	.dc.b	$30,$73,$E3,$72,$F1,$F9,$BF,$C6,$BE,$AD,$EE,$DF,$A1,$C3,$6F,$FD
	.dc.b	$93,$F5,$7F,$98,$6F,$C3,$FF,$22,$8C,$70,$FF,$BA,$94,$64,$D7,$89
	.dc.b	$6B,$9B,$87,$FD,$AB,$F5,$7F,$93,$BE,$53,$FC,$BA,$31,$BF,$F3,$AC
	.dc.b	$3F,$C6,$F7,$E4,$E2,$F1,$7C,$1F,$8D,$91,$D3,$CD,$B9,$7F,$BF,$B1
	.dc.b	$47,$F3,$AF,$AB,$F9,$12,$0F,$E4,$1C,$3A,$73,$C5,$C4,$F3,$7C,$C5
	.dc.b	$C3,$D6,$F5,$EF,$ED,$E4,$59,$FF,$79,$E1,$FB,$86,$C7,$8F,$DE,$DC
	.dc.b	$69,$C3,$F1,$BE,$A6,$E5,$EB,$7D,$5F,$37,$EB,$F5,$2D,$76,$70,$FC
	.dc.b	$EF,$AB,$B7,$F0,$DF,$85,$EA,$FC,$3F,$1A,$D7,$99,$C3,$F1,$79,$7F
	.dc.b	$B5,$FD,$5F,$F7,$58,$D1,$93,$E5,$60,$8D,$29,$FE,$E7,$B7,$D2,$AF
	.dc.b	$87,$C1,$F4,$F8,$BE,$A3,$23,$E3,$ED,$70,$E5,$1D,$AD,$3F,$6C,$FE
	.dc.b	$EB,$F5,$7F,$BD,$AD,$9F,$6A,$33,$E3,$7F,$7A,$FA,$BF,$D0,$E1,$E9
	.dc.b	$AF,$C1,$43,$B7,$5D,$BB,$65,$7E,$63,$92,$DF,$E3,$5E,$9E,$3F,$5F
	.dc.b	$5E,$D7,$5A,$63,$FD,$39,$F2,$98,$9F,$F0,$6F,$5F,$A7,$87,$FC,$33
	.dc.b	$5B,$7F,$C3,$71,$9B,$F9,$F2,$F9,$AF,$67,$CD,$7B,$5B,$3E,$19,$4F
	.dc.b	$C1,$F8,$CC,$E5,$E2,$9F,$F4,$7D,$6E,$7C,$79,$EE,$3D,$58,$FF,$27
	.dc.b	$BE,$1E,$57,$CB,$F8,$AF,$98,$DA,$F0,$47,$F8,$A3,$EA,$9F,$57,$9B
	.dc.b	$F8,$31,$8A,$F3,$7B,$9F,$E2,$EE,$27,$BD,$F5,$E3,$0C,$9F,$99,$D4
	.dc.b	$FA,$BC,$63,$F3,$23,$13,$F4,$7E,$AB,$F8,$4C,$FF,$95,$DA,$FF,$98
	.dc.b	$9A,$FE,$69,$AF,$F0,$5F,$AB,$FF,$01,$CF,$F9,$7E,$11,$87,$FC,$C4
	.dc.b	$D7,$FC,$DA,$D7,$F4,$AB,$E7,$7F,$CA,$B1,$86,$BD,$DA,$F6,$6F,$D5
	.dc.b	$FF,$92,$32,$F1,$36,$7F,$CE,$4B,$E6,$1E,$4E,$BE,$23,$E5,$D7,$F9
	.dc.b	$DF,$EA,$5E,$B7,$AE,$11,$87,$DC,$D7,$84,$6E,$31,$D9,$FE,$6E,$6E
	.dc.b	$BF,$C0,$9B,$5F,$CB,$7E,$AF,$F0,$2F,$E8,$E1,$F2,$7F,$C2,$2F,$FC
	.dc.b	$EF,$5F,$EA,$FB,$FE,$16,$DB,$FE,$6D,$7C,$1E,$57,$C6,$B7,$F4,$16
	.dc.b	$7C,$0D,$7D,$B3,$6B,$94,$7E,$CA,$FB,$26,$C6,$B0,$F7,$57,$39,$D9
	.dc.b	$A7,$A1,$9B,$97,$E6,$21,$DD,$CB,$FC,$0A,$FB,$61,$F2,$6B,$38,$7C
	.dc.b	$4B,$EB,$7C,$A3,$0F,$F0,$DC,$3C,$4E,$6F,$E1,$B9,$AE,$2F,$7B,$EB
	.dc.b	$3A,$8C,$1C,$4F,$92,$71,$30,$7C,$A3,$07,$AC,$CE,$1E,$A7,$89,$93
	.dc.b	$A9,$30,$F4,$B9,$7E,$6B,$DB,$0F,$9B,$87,$A2,$73,$3B,$BF,$44,$60
	.dc.b	$8E,$09,$1F,$88,$C3,$94,$62,$3E,$F5,$61,$D1,$FA,$09,$FC,$F1,$1C
	.dc.b	$10,$FD,$53,$96,$9F,$09,$FC,$77,$B3,$D6,$3E,$41,$F0,$6B,$F4,$D1
	.dc.b	$9C,$7E,$44,$EB,$0F,$D6,$4E,$6F,$4D,$9F,$9E,$D7,$A1,$FA,$67,$13
	.dc.b	$F1,$23,$07,$94,$E2,$53,$89,$FC,$0A,$38,$DF,$E4,$E3,$F7,$11,$87
	.dc.b	$E3,$9F,$99,$70,$FA,$63,$DA,$F5,$9F,$13,$D4,$C4,$75,$79,$18,$3A
	.dc.b	$C7,$C2,$70,$75,$3D,$67,$A2,$FE,$49,$2B,$C6,$D8,$FE,$E5,$F5,$2F
	.dc.b	$7F,$F8,$22,$F8,$BA,$62,$24,$FF,$3F,$9E,$0D,$8F,$E5,$11,$88,$FF
	.dc.b	$10,$9F,$E4,$BF,$AA,$7D,$7F,$F2,$D6,$C7,$F4,$96,$BC,$03,$87,$9B
	.dc.b	$AC,$99,$FF,$8B,$A3,$0D,$AF,$E3,$D1,$FD,$8E,$BD,$FE,$A7,$F2,$1F
	.dc.b	$96,$7C,$1D,$3E,$0C,$49,$F7,$3E,$C2,$3F,$D7,$FE,$AF,$A3,$FE,$18
	.dc.b	$6E,$5E,$48,$FF,$35,$46,$32,$FF,$21,$AF,$5F,$F9,$93,$67,$FC,$48
	.dc.b	$D7,$A4,$7F,$37,$FA,$8F,$A9,$1A,$FA,$4E,$24,$78,$FA,$FD,$08,$E0
	.dc.b	$FF,$18,$7D,$5B,$C9,$F0,$FC,$74,$9F,$0B,$E4,$F0,$E4,$79,$1A,$FB
	.dc.b	$FD,$7F,$49,$1F,$4B,$FD,$37,$EA,$FF,$B7,$C7,$F8,$FF,$EA,$BE,$23
	.dc.b	$FE,$3E,$6F,$06,$1B,$C1,$FE,$68,$7F,$D6,$73,$C7,$D5,$FE,$B1,$18
	.dc.b	$8E,$51,$FD,$6F,$EA,$D9,$EB,$D7,$95,$E1,$78,$FA,$32,$3C,$8D,$7F
	.dc.b	$CA,$3C,$5F,$5D,$FF,$51,$63,$FE,$15,$7D,$5E,$47,$88,$F1,$FA,$B2
	.dc.b	$3D,$CD,$7D,$DE,$AF,$A0,$8F,$C3,$7F,$B9,$7D,$5B,$C2,$F6,$FC,$5C
	.dc.b	$AF,$90,$FC,$0F,$56,$47,$CB,$6B,$E8,$E9,$F8,$BF,$1F,$23,$E8,$3F
	.dc.b	$23,$D2,$91,$F4,$DA,$FF,$9C,$3E,$AF,$DC,$FF,$9C,$BC,$EF,$F0,$AB
	.dc.b	$7D,$6F,$F0,$44,$63,$A3,$ED,$BF,$E0,$62,$79,$7A,$37,$F4,$12,$FF
	.dc.b	$D0,$12,$8C,$47,$F9,$83,$E3,$BA,$FD,$04,$8D,$BE,$0F,$A0,$91,$C9
	.dc.b	$AF,$F9,$A3,$EA,$FC,$BF,$1B,$23,$BB,$E8,$F8,$D9,$1E,$26,$BF,$E6
	.dc.b	$5E,$1F,$6F,$C6,$48,$F3,$BF,$03,$C5,$91,$EA,$6B,$F1,$FD,$5F,$8A
	.dc.b	$47,$DE,$FF,$8B,$FE,$B3,$FC,$83,$0F,$AC,$78,$1F,$E3,$36,$C7,$FA
	.dc.b	$95,$AF,$FA,$91,$7C,$26,$FE,$5E,$4E,$DE,$5D,$3F,$E3,$DF,$AC,$7D
	.dc.b	$1D,$25,$7F,$AC,$61,$F3,$3B,$E1,$93,$97,$97,$0F,$F9,$1F,$EB,$3F
	.dc.b	$9E,$75,$7E,$E9,$BE,$5F,$FB,$79,$B6,$0B,$E4,$B9,$7C,$84,$9F,$F6
	.dc.b	$14,$A3,$B7,$AD,$FE,$55,$FA,$2F,$F6,$85,$7F,$B1,$DA,$FF,$B1,$9B
	.dc.b	$8F,$F0,$C7,$FC,$FD,$1B,$F1,$39,$11,$E5,$E2,$7F,$98,$7E,$B1,$54
	.dc.b	$93,$FE,$D5,$87,$7E,$FB,$A4,$E5,$E5,$C7,$CF,$C6,$93,$C5,$FF,$37
	.dc.b	$7D,$67,$F6,$2D,$3F,$90,$AF,$9A,$C9,$2B,$FD,$C9,$0F,$9A,$FF,$3B
	.dc.b	$7D,$63,$EA,$E9,$2B,$FD,$D5,$0E,$0F,$F3,$E7,$A5,$FA,$D9,$FF,$77
	.dc.b	$35,$FF,$76,$B7,$2F,$E6,$6F,$C2,$8D,$F8,$7C,$88,$F2,$F0,$FA,$FC
	.dc.b	$2C,$8E,$AF,$83,$6F,$F4,$67,$D6,$3E,$6A,$92,$BF,$E0,$04,$7C,$17
	.dc.b	$F0,$D9,$7F,$44,$FA,$CF,$E8,$31,$F8,$96,$A6,$9F,$D0,$63,$87,$CA
	.dc.b	$7F,$A6,$E0,$D3,$1B,$E7,$69,$7E,$0E,$DF,$CB,$3D,$63,$97,$4C,$8D
	.dc.b	$1E,$37,$0B,$1F,$47,$CA,$78,$7C,$49,$3C,$0F,$43,$E5,$F1,$24,$67
	.dc.b	$44,$62,$3A,$65,$E4,$97,$A5,$FE,$B7,$FA,$CF,$C9,$D7,$56,$BC,$0F
	.dc.b	$D8,$38,$CB,$FB,$7A,$F3,$B9,$12,$79,$3F,$C5,$78,$FF,$D8,$9F,$2F
	.dc.b	$FE,$0E,$9E,$81,$C5,$FF,$7A,$6E,$5F,$29,$FE,$03,$3B,$F8,$AE,$64
	.dc.b	$7E,$27,$9B,$FD,$7D,$F8,$DF,$57,$FE,$CE,$27,$A9,$FE,$D5,$7F,$2D
	.dc.b	$87,$D6,$6C,$FF,$0D,$6A,$69,$F9,$B7,$D2,$DC,$7F,$C2,$2D,$9E,$9E
	.dc.b	$27,$F9,$0E,$BC,$1E,$1C,$7C,$AF,$8F,$59,$1E,$86,$BD,$BC,$3F,$C2
	.dc.b	$2F,$AC,$F9,$0F,$9E,$32,$FF,$99,$31,$46,$23,$DC,$D7,$86,$5F,$E6
	.dc.b	$37,$DD,$E2,$3E,$B3,$17,$FC,$F9,$B1,$99,$FF,$35,$4E,$5E,$43,$E9
	.dc.b	$2B,$4C,$B8,$37,$F2,$7F,$E5,$BC,$3F,$C7,$33,$FE,$85,$DB,$F3,$CA
	.dc.b	$75,$1F,$E8,$0F,$90,$B3,$FA,$E6,$AF,$39,$EC,$E2,$3A,$1F,$90,$8C
	.dc.b	$D7,$6E,$4D,$79,$FB,$6C,$78,$A3,$97,$C6,$62,$FF,$32,$1C,$CF,$ED
	.dc.b	$6D,$F2,$78,$AF,$03,$E6,$D3,$FA,$2D,$7E,$ED,$FE,$86,$DE,$5D,$72
	.dc.b	$C8,$E9,$E0,$5C,$3D,$4F,$C0,$FA,$CF,$8D,$3F,$C1,$5F,$7E,$1F,$E6
	.dc.b	$47,$35,$FB,$07,$33,$FC,$59,$B1,$FD,$B7,$2F,$E1,$33,$CF,$7F,$9B
	.dc.b	$18,$9E,$5E,$37,$87,$CE,$F1,$36,$FC,$51,$E5,$F3,$71,$5E,$17,$D1
	.dc.b	$F1,$9E,$9F,$4A,$DF,$A5,$C3,$EB,$F4,$AD,$FA,$5F,$81,$F0,$DF,$07
	.dc.b	$E2,$ED,$7C,$17,$F0,$7F,$2D,$FE,$1C,$8F,$E5,$0B,$EB,$33,$ED,$E9
	.dc.b	$DF,$95,$C3,$C9,$B7,$C1,$FC,$6F,$D6,$7F,$87,$23,$F9,$92,$FA,$CC
	.dc.b	$FB,$9A,$77,$C9,$C3,$EB,$2D,$F0,$7F,$2F,$F5,$9F,$E1,$98,$FE,$78
	.dc.b	$FE,$2D,$BD,$FA,$EF,$DF,$17,$0E,$2B,$5C,$1F,$D0,$FD,$63,$D1,$BC
	.dc.b	$18,$6D,$7F,$49,$3F,$C7,$25,$DB,$F5,$2E,$39,$7F,$9A,$BE,$F7,$F9
	.dc.b	$BF,$D6,$7F,$8C,$63,$FA,$93,$F8,$CE,$EF,$F5,$8B,$99,$DF,$8B,$FB
	.dc.b	$48,$78,$97,$7C,$37,$0E,$7D,$5B,$DF,$83,$63,$FB,$07,$8C,$F6,$FF
	.dc.b	$7E,$FF,$97,$A3,$31,$8F,$83,$CF,$AD,$AF,$16,$D7,$27,$F8,$3F,$D6
	.dc.b	$7E,$CB,$7F,$EA,$6F,$EC,$9B,$E4,$6B,$C8,$DF,$B9,$C3,$E1,$6E,$3F
	.dc.b	$DC,$F0,$47,$4F,$77,$F8,$9F,$27,$C3,$E5,$FF,$6A,$D7,$FA,$CE,$FF
	.dc.b	$BD,$4E,$5F,$F0,$3B,$FE,$39,$73,$E3,$F2,$F2,$39,$BF,$C7,$BE,$B1
	.dc.b	$F2,$77,$E8,$F0,$DB,$FF,$03,$7D,$67,$E5,$5B,$F0,$FF,$C9,$A3,$1C
	.dc.b	$5F,$DC,$94,$64,$D7,$8B,$6B,$9B,$87,$FC,$25,$F5,$9F,$EE,$47,$26
	.dc.b	$3C,$3E,$DF,$09,$DF,$92,$CF,$91,$C3,$CB,$CE,$B9,$FF,$81,$30,$47
	.dc.b	$F3,$CF,$AA,$F8,$D3,$FE,$81,$7F,$21,$5F,$E2,$E7,$30,$72,$EF,$8B
	.dc.b	$FA,$2D,$BE,$67,$A4,$61,$ED,$E1,$77,$F2,$2E,$3F,$E2,$FC,$7F,$69
	.dc.b	$8F,$F8,$61,$BC,$E9,$F1,$7B,$FC,$5E,$0F,$A4,$F0,$FD,$4D,$8F,$92
	.dc.b	$FB,$7E,$73,$DD,$EA,$D9,$F0,$38,$7E,$7F,$D6,$5C,$BE,$33,$F1,$7D
	.dc.b	$6F,$8D,$E3,$DA,$F4,$38,$7F,$C9,$1F,$59,$E7,$72,$60,$F9,$DC,$7F
	.dc.b	$3F,$86,$C7,$C1,$70,$FC,$3D,$2E,$5F,$0D,$FA,$C7,$F7,$2F,$AC,$FD
	.dc.b	$B4,$EC,$FF,$2C,$38,$9F,$F2,$0F,$D6,$7E,$6A,$7F,$CB,$8B,$B7,$FB
	.dc.b	$DB,$99,$DC,$60,$FB,$7C,$16,$BF,$26,$F0,$F2,$38,$FB,$16,$7A,$B8
	.dc.b	$7F,$CD,$5D,$3E,$3E,$4F,$B5,$8F,$FB,$37,$2F,$E9,$0F,$F9,$5A,$75
	.dc.b	$5F,$E7,$C9,$7C,$D7,$9B,$E6,$BC,$ED,$9F,$5C,$7A,$BE,$37,$18,$7C
	.dc.b	$07,$13,$E2,$CE,$CF,$77,$E0,$F9,$7F,$0B,$E3,$2D,$7C,$67,$0F,$FA
	.dc.b	$0F,$EB,$1F,$33,$8C,$3F,$CD,$B1,$89,$F9,$BE,$EF,$F8,$9B,$89,$EF
	.dc.b	$FC,$E8,$C3,$27,$FD,$1D,$CE,$FA,$CC,$63,$FE,$78,$8C,$4F,$FA,$4F
	.dc.b	$EA,$FF,$B6,$3F,$E4,$B6,$CF,$FA,$88,$FF,$7C,$38,$70,$D7,$FD,$4E
	.dc.b	$FF,$B3,$E3,$97,$A7,$FE,$44,$9E,$2D,$53,$B7,$B0,$FA,$BF,$F1,$8E
	.dc.b	$3F,$E5,$D6,$C6,$9F,$F2,$2F,$8B,$9F,$85,$F8,$73,$CA,$DC,$3F,$E5
	.dc.b	$C6,$C7,$B6,$30,$7F,$98,$3E,$AF,$E8,$C7,$F9,$AD,$AF,$91,$AF,$C3
	.dc.b	$6B,$FE,$C6,$77,$D3,$F9,$28,$46,$34,$D7,$17,$FB,$32,$B5,$3F,$EB
	.dc.b	$39,$FF,$67,$FD,$5F,$FA,$86,$FF,$D5,$AB,$E4,$78,$4E,$9B,$21,$3C
	.dc.b	$6F,$C4,$DA,$E1,$3F,$ED,$18,$E8,$DF,$57,$9B,$C9,$B1,$A8,$F1,$DE
	.dc.b	$26,$47,$38,$D9,$87,$D3,$18,$74,$77,$21,$D0,$C9,$C0,$CD,$FA,$A7
	.dc.b	$94,$64,$91,$97,$8B,$F0,$27,$C4,$E6,$3B,$1C,$2F,$E7,$CF,$49,$D1
	.dc.b	$C6,$7C,$33,$F2,$1E,$31,$97,$91,$F4,$6F,$E3,$47,$29,$EB,$3C,$8D
	.dc.b	$5F,$78,$CC,$72,$3E,$CC,$65,$E9,$7E,$A9,$F1,$57,$59,$C9,$D4,$E2
	.dc.b	$7C,$28,$EB,$1E,$73,$84,$68,$F0,$5F,$E1,$5F,$29,$C4,$EA,$7C,$F3
	.dc.b	$F8,$13,$F9,$56,$D7,$FB,$22,$31,$1E,$C3,$27,$FB,$55,$CB,$D0,$C9
	.dc.b	$E2,$8F,$C8,$FC,$8B,$CD,$F8,$13,$E2,$BF,$56,$1F,$9F,$85,$6D,$F8
	.dc.b	$58,$7A,$DF,$CD,$1F,$A5,$AF,$DA,$4E,$E3,$C0,$7C,$97,$11,$DA,$3C
	.dc.b	$15,$F2,$8F,$04,$E9,$C4,$68,$87,$92,$FD,$2F,$88,$FB,$D3,$C5,$DD
	.dc.b	$F9,$CF,$44,$66,$38,$5F,$8D,$B7,$E2,$BE,$B7,$C7,$0F,$93,$18,$7C
	.dc.b	$38,$73,$9D,$5F,$BE,$B4,$D3,$E4,$D7,$08,$CC,$7A,$1B,$87,$8E,$FA
	.dc.b	$56,$30,$F2,$5F,$23,$A0,$AF,$E4,$9F,$57,$FE,$8A,$8F,$A9,$3F,$6E
	.dc.b	$31,$1B,$8C,$7B,$DF,$84,$F0,$3F,$8A,$D6,$AF,$9C,$E0,$D5,$F4,$3F
	.dc.b	$92,$7D,$53,$EA,$FF,$8E,$B6,$34,$1C,$3E,$13,$8F,$4F,$03,$FA,$AB
	.dc.b	$82,$C7,$F6,$D6,$C6,$96,$0F,$3F,$0D,$8E,$6D,$7F,$B0,$B5,$FE,$E0
	.dc.b	$E7,$1F,$93,$3F,$08,$FE,$6F,$F5,$6F,$4F,$F9,$CF,$D5,$F8,$F0,$F5
	.dc.b	$07,$17,$FD,$55,$BB,$7E,$AD,$7A,$3C,$89,$5F,$1A,$32,$FC,$8F,$A8
	.dc.b	$7C,$77,$83,$E8,$A4,$ED,$F1,$FA,$52,$39,$35,$F2,$FD,$5B,$93,$D3
	.dc.b	$AC,$9F,$03,$EC,$F0,$E4,$78,$DA,$FB,$B3,$FF,$42,$3E,$7F,$47,$E1
	.dc.b	$91,$FB,$FF,$56,$F0,$BF,$AF,$FD,$57,$F7,$7E,$2F,$EF,$0D,$E2,$FE
	.dc.b	$1C,$DB,$FE,$DC,$FF,$AE,$1F,$C4,$B5,$F9,$ED,$9E,$7E,$F7,$F6,$CF
	.dc.b	$AB,$FA,$D5,$C3,$E1,$FC,$24,$79,$F9,$DE,$42,$38,$3E,$87,$57,$A3
	.dc.b	$EA,$E4,$78,$9F,$5F,$A5,$23,$CA,$D7,$CB,$E9,$7C,$02,$3E,$A7,$FC
	.dc.b	$03,$F5,$7F,$CA,$75,$79,$DF,$18,$95,$F0,$3C,$AF,$92,$47,$07,$E1
	.dc.b	$7A,$BE,$6F,$C7,$C8,$F8,$EF,$D0,$F4,$24,$7C,$A6,$BF,$98,$FA,$B6
	.dc.b	$BF,$E1,$4F,$98,$FF,$15,$4F,$F8,$7D,$A9,$A3,$E7,$F8,$12,$3F,$7F
	.dc.b	$19,$6F,$D5,$F5,$23,$83,$FC,$E9,$F1,$BE,$1F,$A0,$91,$C9,$F2,$7C
	.dc.b	$FC,$8E,$AD,$7F,$CF,$3F,$57,$E9,$F8,$A9,$1E,$27,$D9,$F1,$52,$3C
	.dc.b	$AD,$7F,$CE,$DC,$1F,$07,$D2,$91,$EA,$7E,$27,$8B,$23,$DA,$D7,$F8
	.dc.b	$0E,$AF,$90,$F9,$44,$8F,$86,$FF,$70,$3F,$C8,$3F,$56,$EC,$A4,$9F
	.dc.b	$E4,$F0,$F8,$2D,$FC,$BC,$9C,$BC,$B9,$FF,$C9,$7F,$56,$FA,$1A,$4A
	.dc.b	$FE,$63,$0F,$99,$DF,$0C,$9C,$BC,$B8,$7F,$CA,$BF,$57,$FD,$07,$C7
	.dc.b	$FF,$01,$36,$BF,$CC,$4D,$4B,$E4,$38,$FC,$6C,$9F,$E8,$32,$8E,$BF
	.dc.b	$33,$FE,$62,$F9,$8F,$F6,$D5,$7F,$46,$6B,$FC,$B1,$BE,$5F,$F9,$FD
	.dc.b	$FC,$94,$6F,$C3,$E4,$47,$97,$87,$FE,$6E,$FA,$B5,$52,$4F,$F5,$48
	.dc.b	$FF,$42,$3F,$E7,$D5,$DB,$C4,$91,$BE,$2E,$9E,$F4,$8E,$4F,$F9,$F3
	.dc.b	$EA,$FF,$D7,$59,$FF,$A5,$57,$CC,$E4,$95,$FD,$92,$1F,$33,$FE,$84
	.dc.b	$FA,$B7,$D5,$52,$57,$F6,$A8,$FE,$0E,$FF,$6C,$F4,$1E,$3F,$FA,$31
	.dc.b	$B1,$98,$C6,$F9,$F0,$E4,$F1,$78,$78,$7C,$7E,$1A,$4F,$57,$A6,$FF
	.dc.b	$E9,$8F,$AB,$7C,$C5,$25,$7F,$7E,$8F,$7B,$FE,$00,$FA,$BF,$E7,$89
	.dc.b	$A4,$F5,$BF,$D4,$6B,$E4,$E9,$27,$97,$06,$BA,$64,$75,$78,$F4,$C8
	.dc.b	$F0,$35,$FE,$EB,$6F,$93,$EF,$E2,$C9,$F3,$3E,$2F,$16,$47,$A5,$AF
	.dc.b	$2F,$4B,$FD,$73,$F5,$7F,$88,$83,$49,$E7,$E8,$91,$FE,$22,$84,$7F
	.dc.b	$9C,$B8,$BF,$D8,$7F,$2D,$FD,$FA,$BF,$C4,$ED,$7F,$C1,$AD,$ED,$FF
	.dc.b	$70,$3F,$B1,$27,$C5,$F2,$3F,$CF,$BC,$BF,$EC,$EF,$A9,$FE,$65,$7F
	.dc.b	$E1,$66,$DF,$FB,$A5,$BE,$3F,$F9,$FD,$FC,$4C,$EF,$AF,$91,$1F,$F1
	.dc.b	$DC,$7E,$B1,$BD,$7D,$C8,$FF,$90,$3E,$23,$F4,$77,$D7,$D4,$F2,$12
	.dc.b	$79,$ED,$E7,$23,$83,$BF,$89,$FE,$5D,$5F,$56,$DF,$D1,$39,$9F,$F4
	.dc.b	$C0,$CF,$F7,$7A,$FF,$93,$1B,$1A,$35,$3C,$5F,$85,$E2,$BE,$AF,$83
	.dc.b	$63,$85,$7F,$2F,$9C,$9F,$E5,$38,$7D,$1D,$23,$FE,$57,$7F,$CB,$1F
	.dc.b	$02,$DB,$FC,$CB,$3F,$E3,$E6,$C6,$63,$FC,$9A,$64,$BF,$13,$9C,$F5
	.dc.b	$0F,$3F,$FB,$B5,$F4,$AD,$3F,$EB,$B8,$CC,$F4,$E5,$CB,$FE,$BF,$F3
	.dc.b	$EA,$11,$87,$F9,$F5,$7A,$27,$1F,$19,$96,$C7,$B2,$39,$DB,$E5,$73
	.dc.b	$7F,$E6,$88,$C5,$7F,$A1,$B2,$FE,$6E,$DE,$4F,$1C,$62,$BF,$CD,$EA
	.dc.b	$32,$76,$F0,$2D,$7F,$A4,$2D,$1E,$CF,$AB,$FF,$2A,$4F,$FA,$51,$F9
	.dc.b	$97,$F2,$9C,$E1,$FE,$7C,$73,$3F,$EE,$36,$C7,$C5,$B7,$77,$5D,$BD
	.dc.b	$EF,$7F,$1F,$C1,$E7,$5C,$7C,$0E,$1F,$17,$9B,$71,$F0,$3E,$7F,$8C
	.dc.b	$F4,$7A,$56,$BD,$0E,$1F,$57,$A5,$6F,$D0,$FB,$FE,$2F,$E0,$7C,$5D
	.dc.b	$AF,$80,$FD,$EF,$98,$FF,$07,$47,$FA,$D9,$7D,$5E,$7D,$BD,$3B,$DF
	.dc.b	$87,$93,$6F,$83,$F8,$8F,$AB,$FF,$07,$47,$FB,$01,$7D,$5E,$7D,$CD
	.dc.b	$3B,$E5,$E1,$F5,$76,$F8,$3F,$92,$FA,$BF,$C7,$F0,$FF,$AB,$9F,$F5
	.dc.b	$F4,$FF,$B3,$1C,$F4,$FE,$89,$CC,$EF,$8B,$97,$17,$36,$DF,$FB,$53
	.dc.b	$EA,$DE,$87,$E8,$1F,$B7,$19,$8C,$70,$7E,$92,$3F,$C5,$C7,$FB,$70
	.dc.b	$F7,$E1,$FE,$C2,$72,$DD,$BF,$57,$F5,$7F,$AB,$9F,$F7,$1B,$F8,$ED
	.dc.b	$3F,$72,$E6,$BF,$D9,$6E,$67,$F6,$50,$8F,$D9,$C7,$2F,$07,$AF,$85
	.dc.b	$D9,$EF,$D1,$E0,$D6,$D7,$81,$FD,$EF,$88,$F6,$EF,$C1,$E1,$B5,$FE
	.dc.b	$F8,$9F,$F4,$AB,$7C,$2E,$3E,$B7,$27,$F8,$3F,$D5,$FE,$D7,$7F,$E2
	.dc.b	$6F,$ED,$6B,$FE,$00,$B9,$7D,$6E,$77,$FF,$74,$BF,$ED,$58,$DF,$C2
	.dc.b	$F4,$F7,$7A,$BD,$BD,$5F,$E2,$BE,$5F,$C7,$C7,$BC,$63,$1C,$F8,$BA
	.dc.b	$F1,$6E,$5F,$C6,$EF,$F9,$3E,$1F,$E3,$08,$C1,$F3,$BE,$AD,$EE,$FF
	.dc.b	$22,$5F,$45,$63,$0F,$F2,$6F,$AB,$FC,$ED,$72,$CF,$F9,$44,$3E,$1B
	.dc.b	$5E,$1E,$D9,$5B,$A5,$7F,$39,$8C,$3C,$5B,$E1,$F7,$3F,$9A,$FD,$5B
	.dc.b	$C2,$C4,$F3,$8C,$3C,$BE,$0F,$CD,$C9,$E7,$6C,$76,$8E,$7E,$5E,$32
	.dc.b	$70,$FF,$4C,$C1,$3F,$D1,$3E,$AF,$E8,$61,$F9,$47,$F2,$D7,$FD,$1D
	.dc.b	$CB,$78,$B8,$1E,$8F,$98,$B5,$E7,$7A,$78,$1D,$7D,$4B,$3F,$17,$0F
	.dc.b	$DD,$E7,$FD,$79,$B1,$E1,$F6,$78,$3B,$7C,$C7,$C6,$FA,$9B,$1F,$15
	.dc.b	$F5,$7C,$EF,$AF,$D3,$B3,$DD,$C3,$F2,$FE,$AE,$ED,$F1,$23,$F6,$7C
	.dc.b	$09,$F8,$9E,$A7,$C5,$F1,$ED,$7A,$5C,$3F,$7F,$EA,$FF,$C0,$0E,$4C
	.dc.b	$1F,$37,$8F,$E7,$71,$58,$F8,$4E,$1F,$89,$AD,$CF,$C8,$7D,$17,$EB
	.dc.b	$1F,$DE,$BE,$AF,$F7,$D3,$C0,$FC,$83,$89,$E2,$3E,$AF,$F4,$33,$F9
	.dc.b	$57,$F4,$2D,$EE,$FF,$02,$C6,$27,$8B,$78,$FF,$11,$C5,$AE,$EF,$CB
	.dc.b	$BC,$3C,$9E,$3E,$B5,$CB,$B3,$87,$F4,$9D,$5E,$5C,$3F,$C3,$B8,$FF
	.dc.b	$56,$CB,$CF,$39,$AF,$E4,$B2,$F9,$BF,$2F,$CD,$F9,$9B,$3E,$B8,$F4
	.dc.b	$FC,$66,$30,$F4,$B8,$9F,$0E,$76,$7B,$BF,$03,$CA,$F8,$3F,$15,$6B
	.dc.b	$E3,$38,$7F,$79,$F5,$6F,$A9,$C6,$1F,$3A,$31,$3F,$37,$DB,$FF,$17
	.dc.b	$F1,$3D,$DF,$9B,$18,$64,$FF,$07,$E8,$7C,$F6,$31,$FB,$31,$89,$FE
	.dc.b	$19,$F5,$4F,$9C,$FF,$22,$D7,$F4,$A8,$C4,$FF,$98,$61,$18,$7F,$CC
	.dc.b	$8D,$7F,$CD,$7F,$57,$FE,$39,$C7,$83,$63,$FC,$D4,$BE,$69,$E4,$6B
	.dc.b	$E1,$BD,$95,$D3,$DD,$F0,$3E,$06,$E3,$FE,$2A,$8F,$23,$5F,$1E,$DA
	.dc.b	$23,$A7,$9F,$D5,$FF,$A7,$A3,$FD,$02,$D7,$FB,$4B,$5F,$F2,$E3,$53
	.dc.b	$FE,$A2,$73,$F3,$DF,$DC,$3B,$3F,$CD,$B3,$99,$7D,$5B,$E4,$F5,$C1
	.dc.b	$FD,$DB,$EA,$3F,$D6,$CF,$83,$D2,$CD,$F8,$1A,$A7,$6F,$F9,$BD,$AF
	.dc.b	$92,$37,$87,$F9,$F6,$3F,$A4,$4E,$8D,$C6,$63,$F7,$6D,$C7,$FD,$0B
	.dc.b	$3B,$F8,$9E,$77,$2F,$9C,$79,$27,$9C,$E6,$7F,$82,$C7,$17,$FA,$C5
	.dc.b	$F5,$C3,$DD,$1C,$88,$78,$AB,$96,$1E,$2C,$3B,$18,$AE,$35,$DA,$32
	.dc.b	$F6,$AD,$3F,$16,$32,$F5,$34,$78,$2B,$B3,$83,$47,$53,$55,$E0,$8F
	.dc.b	$BB,$5D,$1E,$33,$C1,$CB,$C0,$F1,$1D,$5C,$BD,$48,$E0,$93,$F8,$5A
	.dc.b	$EC,$F8,$49,$3D,$0E,$2E,$1F,$C1,$1E,$13,$F1,$11,$F1,$1C,$9C,$0E
	.dc.b	$47,$FB,$8B,$04,$7E,$C1,$B1,$FC,$05,$AF,$FB,$8D,$AF,$18,$E0,$47
	.dc.b	$83,$EA,$7D,$15,$F8,$EF,$81,$C4,$79,$8F,$45,$7F,$B2,$9B,$5A,$8C
	.dc.b	$47,$A4,$E0,$57,$35,$F5,$09,$7F,$2E,$7F,$47,$7C,$EF,$A3,$63,$C2
	.dc.b	$D7,$A3,$6B,$9C,$7E,$C6,$31,$5D,$63,$F4,$31,$87,$DF,$1F,$BF,$8C
	.dc.b	$47,$A2,$7E,$13,$EE,$3C,$25,$3C,$25,$7C,$2F,$F0,$38,$E7,$7F,$BD
	.dc.b	$6D,$FE,$D5,$B3,$F0,$0F,$F2,$57,$C9,$3C,$BF,$E1,$AD,$8C,$C7,$EF
	.dc.b	$7D,$39,$18,$21,$FB,$E7,$F8,$17,$A7,$CB,$E0,$A4,$FF,$10,$8C,$BF
	.dc.b	$89,$7F,$18,$47,$FA,$67,$D4,$BD,$4F,$E7,$ED,$8C,$C6,$0F,$CA,$79
	.dc.b	$7F,$C9,$27,$53,$FE,$14,$6E,$DF,$C6,$A7,$F9,$4F,$D3,$FF,$62,$AF
	.dc.b	$F3,$53,$5F,$F0,$DB,$65,$C1,$FE,$AA,$8E,$6D,$C3,$99,$2B,$FC,$E9
	.dc.b	$F5,$3F,$CC,$A9,$F4,$5E,$02,$4F,$83,$D4,$F1,$91,$C1,$FE,$7D,$FA
	.dc.b	$97,$9F,$E8,$F6,$E4,$FA,$1F,$5F,$8B,$23,$D6,$D7,$D7,$EA,$FC,$02
	.dc.b	$3F,$01,$FE,$97,$F5,$3F,$DC,$23,$FD,$19,$F4,$FF,$01,$BB,$FF,$50
	.dc.b	$8C,$47,$F9,$01,$FF,$4F,$BC,$3D,$5D,$11,$FF,$4B,$7D,$4F,$FA,$3D
	.dc.b	$EB,$F1,$5C,$C8,$F8,$3D,$2E,$C4,$70,$7C,$CF,$37,$C5,$E6,$C8,$F3
	.dc.b	$BE,$5F,$56,$47,$A9,$AF,$9F,$D5,$F5,$11,$F7,$BF,$DB,$3E,$A5,$F0
	.dc.b	$FE,$FF,$8A,$95,$F1,$1F,$85,$EB,$C8,$F8,$ED,$7F,$CC,$7D,$5F,$80
	.dc.b	$7E,$5F,$9D,$F2,$48,$FC,$EF,$37,$E6,$11,$C1,$FE,$BD,$FA,$9F,$F1
	.dc.b	$93,$FE,$BF,$F4,$BF,$C8,$8D,$F5,$FF,$C0,$71,$88,$FF,$39,$BF,$E0
	.dc.b	$F2,$7A,$1A,$37,$F3,$D2,$7A,$35,$FF,$27,$7C,$BF,$4F,$9D,$91,$E1
	.dc.b	$7B,$FC,$E4,$8F,$23,$5F,$F2,$B7,$D4,$F9,$3E,$2A,$47,$A5,$F3,$FA
	.dc.b	$72,$3D,$8D,$7F,$CA,$7E,$37,$B3,$E2,$64,$7C,$17,$DF,$E2,$C8,$F8
	.dc.b	$8D,$7E,$0F,$C7,$FC,$42,$3F,$25,$FF,$14,$FD,$59,$FF,$15,$FD,$53
	.dc.b	$BE,$92,$BF,$DD,$D0,$F8,$5D,$FC,$B4,$9C,$BC,$BA,$3F,$C6,$BF,$54
	.dc.b	$FA,$0A,$4A,$FF,$7C,$43,$E6,$37,$C3,$27,$2F,$2E,$1F,$F1,$FF,$D5
	.dc.b	$7F,$5A,$F2,$95,$23,$0D,$F6,$F7,$EC,$F0,$F6,$A4,$F0,$F5,$BF,$73
	.dc.b	$7F,$CF,$63,$FD,$72,$BD,$AF,$F2,$6F,$D1,$BE,$1E,$92,$B9,$67,$E2
	.dc.b	$53,$FE,$79,$72,$79,$97,$8F,$D0,$91,$A3,$8F,$8F,$FE,$5C,$FA,$AE
	.dc.b	$71,$96,$BF,$EC,$E6,$E7,$FC,$49,$78,$3C,$7C,$09,$5C,$5E,$9C,$BD
	.dc.b	$7C,$99,$3D,$5F,$F3,$5F,$D5,$7A,$E3,$FD,$B6,$BE,$67,$24,$8F,$44
	.dc.b	$3E,$67,$FC,$E7,$F5,$4F,$A9,$A4,$AF,$74,$7C,$27,$E0,$7A,$8F,$27
	.dc.b	$FC,$F6,$D8,$CC,$63,$8B,$3E,$34,$9E,$2F,$0F,$1B,$8F,$C3,$C9,$EA
	.dc.b	$F4,$DB,$FD,$0F,$F5,$4F,$98,$A4,$AF,$A5,$1E,$F7,$EA,$7D,$57,$F3
	.dc.b	$EE,$6F,$E7,$ED,$E9,$FC,$03,$67,$F2,$6F,$F4,$25,$3C,$3E,$4F,$FD
	.dc.b	$2F,$26,$98,$E1,$CF,$54,$BF,$0B,$C3,$AA,$47,$91,$AF,$6C,$7C,$7F
	.dc.b	$27,$DB,$C7,$93,$EA,$7C,$3E,$3C,$8F,$6B,$5E,$47,$AB,$D4,$FF,$56
	.dc.b	$FD,$57,$E3,$23,$F2,$CD,$76,$1C,$6D,$FD,$91,$7A,$3C,$89,$3C,$9F
	.dc.b	$E2,$3C,$7F,$EB,$AF,$87,$FF,$5B,$93,$E2,$F1,$E4,$4B,$FD,$34,$FE
	.dc.b	$F9,$37,$D1,$FE,$AF,$1F,$AB,$FA,$9F,$F6,$71,$3D,$5F,$F6,$3B,$F8
	.dc.b	$D8,$F4,$B7,$4F,$EF,$6D,$E1,$FD,$E3,$67,$F8,$7B,$FB,$08,$CF,$67
	.dc.b	$42,$4F,$F9,$28,$EF,$E2,$C7,$C8,$F8,$B6,$91,$E7,$6B,$E4,$F1,$7F
	.dc.b	$12,$BE,$AB,$DE,$F8,$A3,$2F,$ED,$F1,$46,$23,$D2,$D7,$86,$5F,$BD
	.dc.b	$7D,$7E,$33,$EA,$B1,$0B,$FE,$71,$18,$6C,$71,$6B,$C3,$E8,$F8,$BF
	.dc.b	$CD,$1A,$9A,$6B,$E5,$7F,$C5,$98,$7F,$8A,$A7,$FD,$5B,$B7,$F2,$35
	.dc.b	$3A,$8F,$E2,$9C,$76,$7F,$98,$35,$7A,$0F,$6B,$11,$D0,$FC,$BC,$66
	.dc.b	$BB,$72,$EB,$D1,$DB,$63,$C5,$1C,$BE,$3F,$17,$FA,$00,$E6,$7F,$9D
	.dc.b	$5B,$E5,$71,$5E,$07,$CD,$A7,$F4,$6A,$FE,$80,$FF,$38,$6F,$2E,$B9
	.dc.b	$64,$74,$EF,$B8,$7A,$9F,$81,$F5,$5F,$1A,$7F,$A5,$3E,$FC,$3F,$CA
	.dc.b	$AE,$6B,$F6,$EE,$67,$FA,$C3,$63,$FB,$5E,$4E,$EE,$BC,$1F,$36,$31
	.dc.b	$7D,$BC,$6F,$0F,$9F,$E2,$6C,$78,$A3,$CB,$E7,$63,$1F,$0B,$E8,$F8
	.dc.b	$EF,$4F,$AB,$6F,$D2,$E1,$F5,$FA,$96,$FD,$2F,$C0,$F8,$DF,$83,$F1
	.dc.b	$B6,$BE,$0B,$F8,$3F,$98,$FF,$0B,$47,$F6,$D5,$F5,$59,$F7,$74,$EF
	.dc.b	$87,$87,$93,$6F,$83,$F8,$DF,$AA,$FF,$0B,$47,$F7,$75,$F4,$99,$F7
	.dc.b	$B4,$EF,$93,$87,$D5,$5B,$E0,$FE,$5F,$EA,$9D,$F7,$BF,$0D,$AF,$F0
	.dc.b	$24,$FF,$84,$DB,$C3,$C7,$8B,$93,$FA,$1F,$AA,$7A,$17,$83,$0D,$BF
	.dc.b	$F0,$71,$FE,$33,$2E,$DF,$A9,$71,$CB,$FC,$F1,$F7,$BF,$CF,$BE,$AB
	.dc.b	$FC,$5F,$1F,$E1,$77,$F1,$9D,$DF,$EA,$27,$33,$BF,$17,$F6,$90,$F1
	.dc.b	$2E,$F8,$6E,$1C,$FA,$B7,$BF,$06,$C7,$F8,$97,$C6,$7B,$9F,$BF,$7F
	.dc.b	$CA,$D1,$98,$C7,$C2,$E7,$D5,$D7,$8B,$6B,$93,$FC,$1F,$EA,$BF,$65
	.dc.b	$BF,$F5,$A7,$F6,$4D,$F2,$35,$E4,$6F,$DC,$E1,$F0,$F7,$1F,$F1,$C6
	.dc.b	$1F,$E6,$96,$FB,$BF,$C4,$F9,$3E,$1F,$2F,$FB,$1A,$BF,$D5,$75,$FE
	.dc.b	$3F,$9C,$BF,$E4,$97,$FC,$5C,$E7,$C7,$E5,$E4,$73,$7F,$8F,$7D,$53
	.dc.b	$E5,$2F,$C2,$E1,$B7,$FE,$4B,$FA,$AF,$CA,$B7,$E2,$3F,$93,$46,$38
	.dc.b	$BF,$BC,$A8,$C9,$AF,$16,$D7,$37,$0F,$F9,$5F,$EA,$5E,$1E,$30,$FF
	.dc.b	$01,$C6,$27,$C5,$F0,$9E,$0E,$4F,$0B,$63,$CD,$1F,$CE,$65,$E7,$66
	.dc.b	$F9,$BF,$CF,$7E,$AB,$E3,$C7,$FA,$01,$FC,$8D,$FF,$9C,$5C,$C1,$CB
	.dc.b	$BE,$2E,$DF,$35,$CD,$B5,$FE,$71,$AF,$D9,$B7,$C9,$F0,$78,$7E,$16
	.dc.b	$46,$FD,$FB,$8F,$F8,$8A,$FF,$C2,$73,$93,$E3,$7D,$47,$A6,$BD,$CF
	.dc.b	$AF,$E7,$BD,$9E,$AD,$AF,$13,$87,$E7,$7D,$55,$C7,$E6,$BF,$0F,$D6
	.dc.b	$F8,$9E,$3D,$AF,$53,$87,$FD,$19,$F5,$47,$17,$FE,$38,$8C,$47,$C1
	.dc.b	$E1,$F9,$DC,$3F,$3D,$B1,$F1,$63,$E9,$69,$8B,$F8,$0E,$27,$FD,$35
	.dc.b	$F5,$5F,$B6,$7E,$64,$7F,$75,$8C,$4F,$F9,$23,$EA,$BF,$35,$5F,$EA
	.dc.b	$17,$FD,$5C,$DE,$BF,$EF,$91,$89,$E1,$D9,$F8,$7E,$1C,$B6,$F9,$79
	.dc.b	$1C,$3D,$8E,$2D,$8E,$D1,$FE,$07,$E9,$F1,$F8,$7F,$17,$1F,$F6,$5E
	.dc.b	$5F,$D2,$1F,$F2,$EC,$EA,$FF,$D7,$D2,$F9,$BF,$2F,$CD,$F9,$9B,$3E
	.dc.b	$B8,$F4,$FC,$66,$30,$F8,$0E,$27,$FD,$81,$87,$F1,$A9,$CC,$63,$3F
	.dc.b	$83,$E5,$FC,$2F,$8B,$B3,$F1,$DC,$3F,$EC,$9F,$AA,$7C,$CE,$30,$FF
	.dc.b	$3A,$C6,$27,$E7,$7B,$BF,$E2,$7E,$17,$BF,$F3,$A3,$0C,$9F,$F6,$A7
	.dc.b	$43,$E7,$B1,$8F,$FA,$02,$31,$3F,$ED,$9F,$AA,$FF,$10,$5F,$F9,$29
	.dc.b	$B5,$FE,$4F,$7F,$C6,$9E,$26,$7E,$13,$C3,$3F,$E5,$3F,$AA,$70,$7F
	.dc.b	$92,$9B,$1F,$E5,$46,$BB,$8D,$69,$23,$4D,$7F,$CB,$DF,$55,$FE,$44
	.dc.b	$7F,$CB,$AD,$73,$E5,$FF,$96,$23,$11,$FE,$E1,$5F,$21,$92,$47,$26
	.dc.b	$BF,$EA,$28,$DB,$FE,$FB,$6B,$9F,$4C,$FF,$BF,$FE,$A7,$FD,$34,$3E
	.dc.b	$1F,$31,$FE,$AC,$8C,$3B,$F7,$73,$DD,$F9,$78,$4F,$2A,$CF,$17,$28
	.dc.b	$E3,$3C,$EF,$2F,$8A,$38,$3E,$3A,$CB,$D2,$6C,$EA,$3C,$93,$89,$91
	.dc.b	$C4,$87,$73,$D0,$62,$FD,$15,$E4,$6C,$F2,$9E,$A6,$EF,$D5,$19,$9E
	.dc.b	$37,$E4,$31,$3A,$AE,$07,$B6,$32,$FB,$EB,$83,$A3,$DC,$79,$8F,$09
	.dc.b	$2B,$CA,$E5,$F0,$BC,$6F,$EC,$CF,$48,$E1,$7E,$37,$E1,$18,$3C,$87
	.dc.b	$37,$55,$D2,$34,$79,$0E,$2E,$49,$1C,$09,$1D,$0E,$35,$E4,$7C,$D7
	.dc.b	$F0,$9E,$87,$E3,$8A,$79,$CC,$47,$F8,$DD,$FF,$61,$B5,$FF,$64,$35
	.dc.b	$DD,$FF,$B0,$23,$11,$E4,$3D,$57,$F2,$93,$F9,$9F,$91,$75,$F9,$6B
	.dc.b	$D1,$5A,$C7,$F4,$75,$A8,$F9,$07,$DB,$28,$74,$FC,$88,$F1,$4F,$A6
	.dc.b	$FC,$17,$DF,$0F,$89,$87,$09,$F3,$12,$FD,$51,$97,$9C,$F0,$AE,$58
	.dc.b	$78,$AB,$4F,$94,$F1,$16,$FC,$2F,$A3,$0F,$7C,$62,$36,$7C,$8C,$3C
	.dc.b	$37,$DA,$33,$7D,$9D,$1D,$67,$6F,$5C,$3C,$CD,$3C,$D5,$CE,$FD,$D1
	.dc.b	$FC,$52,$FA,$61,$E4,$C3,$9C,$F9,$23,$0E,$EF,$F8,$2F,$9D,$FC,$23
	.dc.b	$EA,$5E,$4D,$33,$5F,$C7,$E7,$F0,$C4,$7F,$1A,$FE,$27,$CF,$91,$F9
	.dc.b	$06,$BF,$90,$7F,$89,$91,$FE,$57,$9F,$F2,$3B,$FF,$29,$C7,$F2,$DF
	.dc.b	$17,$F8,$DF,$D4,$BC,$FA,$7F,$1C,$89,$3F,$9E,$7F,$34,$47,$F4,$C7
	.dc.b	$F3,$8F,$A9,$FE,$C4,$F8,$36,$FE,$6F,$18,$8F,$CC,$1B,$EE,$FE,$57
	.dc.b	$1E,$18,$E0,$73,$BC,$91,$C8,$FA,$3F,$E4,$D7,$DB,$E8,$79,$92,$77
	.dc.b	$6F,$43,$B9,$23,$2F,$F3,$CF,$A9,$FC,$44,$FC,$69,$F3,$87,$1B,$7F
	.dc.b	$4D,$6E,$1E,$EF,$0B,$D6,$4A,$F0,$BE,$EF,$42,$47,$91,$AF,$6F,$A9
	.dc.b	$78,$5F,$D3,$BE,$A7,$FA,$D6,$5F,$D6,$DB,$8F,$F6,$26,$DF,$F6,$87
	.dc.b	$FD,$40,$FC,$66,$BE,$A6,$CF,$3F,$43,$FA,$DF,$D4,$FA,$AB,$97,$C2
	.dc.b	$F8,$48,$F3,$F3,$7C,$84,$70,$7D,$4E,$2F,$BD,$87,$9F,$EA,$7D,$44
	.dc.b	$9F,$27,$A3,$ED,$23,$83,$D1,$E8,$FC,$12,$38,$3F,$03,$F5,$2F,$13
	.dc.b	$E3,$7C,$3C,$9F,$80,$FC,$9F,$2A,$47,$C3,$6B,$F7,$33,$74,$FC,$FF
	.dc.b	$8D,$93,$F2,$5F,$A5,$E5,$C8,$F9,$8D,$7F,$0B,$F5,$2D,$BF,$C0,$BF
	.dc.b	$2D,$FE,$4E,$9F,$CB,$B5,$FC,$5B,$78,$3F,$C2,$CB,$E7,$78,$92,$78
	.dc.b	$BD,$3E,$8A,$47,$46,$BF,$E5,$AF,$8B,$F0,$FC,$F4,$8F,$0B,$E4,$F9
	.dc.b	$D9,$1E,$46,$BF,$E6,$2F,$A9,$F4,$FC,$44,$8F,$4B,$EC,$F5,$24,$7B
	.dc.b	$1A,$FF,$97,$F8,$7E,$0F,$C3,$C8,$F8,$2F,$C4,$F3,$E4,$7C,$46,$BF
	.dc.b	$13,$E3,$BE,$49,$1F,$92,$FF,$8C,$3D,$23,$FE,$32,$FA,$97,$75,$25
	.dc.b	$7F,$06,$87,$C1,$6F,$E5,$64,$E5,$E5,$D3,$FE,$3C,$FA,$97,$D0,$52
	.dc.b	$57,$F1,$08,$7C,$BE,$F8,$64,$E5,$E5,$C3,$FE,$46,$FA,$9F,$ED,$9E
	.dc.b	$4F,$F9,$61,$B5,$FE,$5E,$6A,$5E,$BF,$1F,$62,$4F,$F2,$09,$FE,$92
	.dc.b	$DF,$97,$FF,$29,$FC,$B3,$E1,$29,$27,$F9,$3C,$3B,$F7,$E3,$49,$CB
	.dc.b	$CB,$C6,$FF,$2E,$FD,$4A,$E9,$2B,$F9,$8C,$3C,$0D,$F7,$C9,$CB,$CB
	.dc.b	$93,$9F,$91,$27,$8B,$FE,$6A,$FA,$9F,$F1,$77,$8B,$FE,$77,$5F,$2F
	.dc.b	$92,$57,$F4,$08,$7C,$BF,$F9,$CB,$EA,$5F,$4F,$49,$5F,$D1,$A1,$C1
	.dc.b	$FE,$78,$F4,$5E,$4D,$25,$7F,$4C,$8F,$80,$DF,$1B,$81,$1D,$78,$DC
	.dc.b	$FE,$1A,$47,$17,$B6,$FF,$E8,$6F,$A9,$7C,$AD,$25,$7F,$5A,$8F,$7B
	.dc.b	$FD,$73,$EA,$7F,$9D,$A6,$93,$D7,$FF,$48,$AF,$91,$A4,$9E,$5C,$1A
	.dc.b	$DE,$47,$57,$8F,$54,$8F,$03,$5F,$EB,$F6,$F9,$1E,$FE,$44,$9F,$33
	.dc.b	$E2,$F2,$24,$7A,$5A,$F2,$F5,$3F,$D4,$FF,$53,$F8,$78,$FE,$F2,$D7
	.dc.b	$FB,$6B,$7A,$BF,$D7,$0B,$D2,$E2,$49,$E2,$FF,$03,$E3,$FF,$59,$FC
	.dc.b	$2F,$F8,$0A,$7F,$D9,$2D,$4D,$3F,$5E,$4F,$8A,$E2,$7F,$0D,$E5,$FF
	.dc.b	$5E,$FD,$3F,$F3,$3A,$FF,$07,$B5,$FF,$01,$B7,$2F,$F6,$73,$F8,$C8
	.dc.b	$DF,$57,$22,$3C,$BC,$3E,$A4,$70,$7F,$87,$3E,$27,$F4,$7E,$5F,$FB
	.dc.b	$7A,$FE,$23,$7C,$4F,$29,$E6,$DA,$5F,$99,$AF,$9B,$C4,$FF,$45,$AF
	.dc.b	$A9,$F8,$6F,$FB,$BB,$04,$EA,$3F,$D4,$11,$8D,$3F,$C7,$B0,$8C,$3F
	.dc.b	$E4,$06,$BC,$3C,$57,$D4,$F0,$9F,$E5,$6D,$8C,$CF,$16,$BD,$BE,$87
	.dc.b	$BA,$0E,$1A,$69,$DF,$C1,$B5,$FF,$22,$CB,$E3,$F0,$FF,$8E,$8F,$F1
	.dc.b	$6C,$6A,$33,$96,$A7,$C1,$18,$9F,$14,$3D,$1F,$E5,$51,$C4,$FF,$29
	.dc.b	$B8,$8E,$DC,$9C,$5E,$3E,$8D,$7D,$91,$5F,$E7,$F5,$E8,$9C,$7C,$76
	.dc.b	$5B,$1E,$C8,$E5,$6F,$92,$CD,$FF,$95,$A3,$15,$FE,$66,$CB,$F9,$D3
	.dc.b	$78,$FC,$51,$8A,$FF,$2E,$BF,$CE,$E7,$47,$B3,$C0,$DD,$78,$1F,$57
	.dc.b	$D4,$FC,$AB,$FF,$32,$BF,$3E,$BF,$CE,$AE,$6F,$E6,$39,$9D,$EB,$EA
	.dc.b	$EE,$E6,$DB,$E3,$E3,$F6,$F3,$BB,$B6,$3A,$C7,$8B,$CD,$C6,$1E,$27
	.dc.b	$CB,$F1,$BE,$6F,$56,$DF,$91,$C3,$E6,$F5,$2D,$FA,$9F,$6F,$CF,$FB
	.dc.b	$BE,$32,$D7,$B1,$FB,$58,$7F,$A9,$E3,$FD,$26,$DB,$FF,$43,$BF,$D0
	.dc.b	$23,$3F,$1F,$C2,$F8,$DD,$9E,$2F,$E1,$7F,$17,$FA,$58,$7F,$A4,$1F
	.dc.b	$E4,$53,$FE,$A0,$73,$C3,$F9,$77,$33,$FE,$96,$FB,$A7,$3F,$E3,$DF
	.dc.b	$E4,$37,$FE,$A8,$72,$DF,$1B,$5F,$07,$23,$5A,$DC,$3F,$D4,$3E,$73
	.dc.b	$E6,$3F,$34,$DA,$FF,$59,$AF,$AF,$CC,$FE,$7F,$EE,$9E,$BD,$F5,$F0
	.dc.b	$DC,$3F,$D7,$7C,$AF,$07,$F4,$CD,$8C,$F8,$38,$9F,$D5,$7E,$2A,$F8
	.dc.b	$EA,$63,$D7,$B7,$FE,$C6,$FB,$B7,$CF,$7E,$C5,$B1,$9F,$8C,$C4,$E7
	.dc.b	$E1,$75,$F5,$B6,$B9,$3F,$B9,$FC,$21,$DD,$7B,$B0,$DA,$FF,$6A,$FD
	.dc.b	$E7,$EC,$BA,$3F,$D9,$2B,$E0,$7F,$81,$39,$F0,$3F,$88,$39,$AE,$3B
	.dc.b	$E8,$DF,$BF,$63,$FD,$C3,$F8,$9B,$DB,$FE,$18,$D8,$CF,$B7,$89,$FE
	.dc.b	$21,$F7,$BF,$B1,$CB,$F8,$F3,$6B,$F8,$EB,$F3,$F4,$FF,$6F,$B9,$7F
	.dc.b	$99,$3F,$9B,$73,$E4,$F4,$F2,$BA,$BF,$C7,$7B,$3F,$C5,$F7,$FE,$F6
	.dc.b	$5E,$76,$7C,$2D,$3F,$C8,$FE,$ED,$F0,$77,$E2,$70,$DC,$3F,$DF,$FF
	.dc.b	$59,$F8,$A6,$F8,$7F,$CA,$E3,$1A,$FF,$BD,$94,$64,$FE,$61,$F5,$8E
	.dc.b	$9C,$5F,$F4,$DE,$1F,$E8,$13,$98,$C1,$FD,$03,$EB,$3F,$A9,$38,$C7
	.dc.b	$FA,$A4,$7F,$1D,$8D,$47,$F3,$D8,$C3,$AF,$63,$C3,$F3,$D9,$6D,$73
	.dc.b	$95,$7F,$42,$58,$7F,$53,$8F,$DB,$C6,$27,$C9,$E3,$77,$F3,$E4,$73
	.dc.b	$AF,$DF,$D6,$CF,$3C,$F9,$7C,$EF,$A1,$F6,$9E,$7F,$4A,$E1,$C5,$C3
	.dc.b	$CF,$A2,$E1,$CC,$F3,$BE,$F3,$E6,$FD,$6D,$9F,$66,$E8,$F8,$9E,$87
	.dc.b	$C5,$F3,$6D,$79,$DC,$3F,$0F,$EB,$7F,$BD,$39,$30,$7C,$DE,$3F,$9D
	.dc.b	$C7,$63,$E0,$38,$7E,$17,$D6,$DD,$3D,$1C,$28,$FC,$87,$5F,$F7,$3C
	.dc.b	$5F,$13,$8C,$3C,$4F,$F7,$7F,$AC,$FC,$AC,$7D,$77,$FD,$5C,$DE,$AD
	.dc.b	$75,$FE,$3B,$96,$4E,$E2,$E9,$F2,$78,$9F,$6D,$D7,$E6,$7F,$2A,$DB
	.dc.b	$ED,$F3,$1E,$13,$17,$F4,$73,$FF,$6D,$CF,$8F,$D2,$F3,$1E,$7F,$A1
	.dc.b	$B1,$E7,$F2,$BF,$C3,$97,$FE,$1A,$BF,$F4,$44,$FF,$8F,$1B,$1F,$E1
	.dc.b	$E3,$FC,$60,$E1,$C7,$D0,$7F,$01,$6F,$57,$47,$FC,$6D,$19,$7F,$C4
	.dc.b	$ED,$79,$F8,$9B,$55,$FC,$12,$FC,$53,$98,$C4,$F8,$CD,$FC,$8E,$49
	.dc.b	$5E,$97,$A7,$D0,$FF,$02,$9F,$53,$5F,$F1,$CB,$5F,$F2,$63,$5F,$F7
	.dc.b	$47,$B5,$E5,$E2,$FE,$B9,$1E,$16,$BF,$15,$AF,$F9,$13,$AF,$FC,$00
	.dc.b	$FC,$96,$B9,$EF,$F8,$11,$88,$FF,$2E,$7D,$67,$F8,$E6,$FF,$CD,$AD
	.dc.b	$8F,$80,$FF,$BD,$FC,$5C,$FC,$3F,$F7,$19,$F8,$9F,$4C,$FA,$8F,$F2
	.dc.b	$AF,$91,$FE,$59,$CB,$FD,$9F,$5F,$C3,$5A,$FE,$E6,$B8,$3F,$C4,$5A
	.dc.b	$FE,$F6,$37,$3C,$F8,$FA,$F0,$7F,$98,$AF,$73,$FE,$67,$6C,$7C,$39
	.dc.b	$DD,$FF,$9E,$6F,$9C,$F1,$C3,$32,$BE,$2E,$EF,$37,$C2,$3F,$D0,$53
	.dc.b	$A7,$85,$6A,$A4,$7A,$E3,$AC,$EE,$B9,$4F,$F4,$1A,$CD,$48,$F8,$D1
	.dc.b	$C9,$F8,$D1,$97,$E2,$46,$70,$F8,$13,$97,$E3,$4F,$23,$C4,$43,$89
	.dc.b	$93,$A9,$97,$8B,$A3,$24,$72,$4C,$3E,$54,$71,$7E,$83,$D6,$3C,$AE
	.dc.b	$4F,$98,$66,$38,$BD,$5E,$2F,$86,$38,$B9,$38,$99,$8E,$31,$A3,$C9
	.dc.b	$1A,$71,$1A,$3A,$BA,$8F,$B2,$79,$4F,$A8,$79,$4F,$BF,$3F,$36,$78
	.dc.b	$92,$32,$71,$BF,$BA,$7D,$EC,$3F,$0C,$7E,$07,$0F,$B1,$5E,$3A,$E2
	.dc.b	$7A,$CF,$43,$97,$CA,$64,$F2,$9F,$0C,$C9,$FC,$9E,$7F,$52,$DB,$FF
	.dc.b	$6C,$C6,$23,$E7,$99,$3F,$DD,$6E,$5F,$19,$93,$B3,$EA,$D7,$E1,$47
	.dc.b	$A3,$1F,$9A,$FE,$92,$32,$FA,$23,$2F,$EB,$63,$33,$F1,$2B,$F4,$B1
	.dc.b	$87,$2D,$9F,$A9,$28,$C4,$79,$1B,$5D,$A7,$53,$C2,$B9,$B4,$E7,$5C
	.dc.b	$2F,$E7,$57,$09,$F7,$D7,$06,$9C,$0E,$47,$84,$B7,$E1,$BE,$91,$F1
	.dc.b	$6F,$F8,$B4,$62,$3E,$24,$61,$E2,$F8,$A3,$57,$F3,$E7,$27,$33,$E7
	.dc.b	$18,$39,$98,$3C,$E7,$94,$B5,$E5,$78,$47,$57,$F9,$97,$C0,$7E,$18
	.dc.b	$8F,$E1,$1A,$E8,$FE,$6C,$FF,$1C,$F1,$7F,$8E,$FA,$DF,$8E,$24,$FE
	.dc.b	$31,$FC,$8E,$92,$3F,$1E,$D7,$FB,$67,$D3,$7F,$73,$4D,$26,$7F,$DC
	.dc.b	$60,$69,$FE,$73,$06,$98,$F0,$3F,$9E,$FD,$37,$FA,$9A,$BF,$CB,$D5
	.dc.b	$FE,$7E,$6C,$EB,$0D,$F9,$32,$7F,$CC,$97,$FE,$49,$78,$FB,$1E,$0F
	.dc.b	$07,$C2,$C8,$EB,$1E,$4E,$EF,$2F,$D2,$C9,$F0,$C6,$5A,$43,$FC,$B9
	.dc.b	$9F,$FB,$2D,$F3,$F8,$FF,$D5,$BB,$DF,$0F,$4F,$83,$12,$76,$43,$9F
	.dc.b	$81,$F1,$88,$FC,$27,$A7,$1F,$CA,$F8,$99,$3F,$19,$F9,$BE,$44,$8F
	.dc.b	$92,$D7,$FC,$E5,$F4,$CD,$7F,$B5,$FC,$FF,$FB,$7A,$7F,$D1,$8D,$7F
	.dc.b	$CF,$0D,$E9,$FB,$11,$F7,$21,$F0,$5C,$89,$1C,$9E,$BF,$31,$FD,$E3
	.dc.b	$EB,$5F,$69,$49,$5E,$0F,$98,$F1,$91,$C1,$E4,$F9,$7F,$39,$1C,$1F
	.dc.b	$EA,$3A,$7D,$47,$B3,$E9,$24,$FB,$1F,$7F,$CA,$C8,$F7,$B5,$FF,$0D
	.dc.b	$71,$7C,$4F,$7E,$47,$C4,$7E,$3F,$BB,$23,$E3,$B5,$F8,$FF,$4F,$FE
	.dc.b	$16,$FC,$77,$FC,$C9,$C5,$F2,$25,$18,$F0,$FF,$4B,$18,$38,$79,$71
	.dc.b	$FF,$5F,$4E,$DF,$D5,$B5,$E7,$1F,$E2,$4F,$9D,$FF,$6B,$47,$08,$E4
	.dc.b	$FF,$AC,$63,$2F,$FA,$CF,$F0,$27,$17,$F8,$B1,$B1,$FE,$CD,$6B,$BE
	.dc.b	$2F,$CE,$79,$1F,$9D,$F7,$A4,$FF,$B4,$FC,$04,$F8,$FD,$EF,$27,$BB
	.dc.b	$2B,$FD,$B1,$9A,$7C,$1C,$9E,$1E,$89,$5E,$18,$FD,$57,$CB,$48,$EC
	.dc.b	$FE,$B7,$CA,$F8,$1E,$54,$9E,$AF,$F9,$1B,$F6,$AF,$A1,$CB,$F4,$78
	.dc.b	$5F,$11,$EF,$25,$FB,$FE,$27,$F7,$1E,$7F,$EE,$76,$93,$ED,$8E,$9E
	.dc.b	$7F,$5F,$5E,$4F,$FB,$CF,$3F,$E2,$CF,$FB,$7F,$57,$C2,$FF,$96,$17
	.dc.b	$C2,$D3,$11,$FC,$1F,$92,$57,$CB,$7F,$85,$70,$7A,$FC,$69,$3D,$DF
	.dc.b	$77,$AF,$D7,$D7,$ED,$F1,$12,$BF,$98,$F2,$7E,$47,$4F,$F2,$DF,$97
	.dc.b	$F1,$3C,$1F,$C8,$B6,$BF,$9B,$B5,$3E,$CA,$FC,$D6,$EF,$53,$E7,$12
	.dc.b	$BE,$17,$A9,$EF,$E9,$91,$E2,$C7,$F3,$F9,$27,$E2,$3F,$E7,$CC,$9C
	.dc.b	$77,$79,$87,$13,$CB,$1F,$77,$E0,$3E,$F7,$C3,$49,$FA,$51,$FE,$86
	.dc.b	$FA,$37,$C0,$D2,$4F,$31,$0F,$8C,$E1,$EB,$49,$D4,$A3,$A7,$AD,$27
	.dc.b	$2D,$5C,$7F,$BF,$A7,$C0,$F8,$FE,$8E,$57,$47,$CD,$EC,$48,$EE,$D7
	.dc.b	$9F,$1F,$F0,$6B,$7B,$1F,$EA,$2F,$A3,$7C,$ED,$25,$FB,$63,$F7,$39
	.dc.b	$FF,$11,$6E,$1F,$C4,$1B,$3F,$DD,$9A,$BE,$AF,$F4,$EF,$C7,$F8,$A7
	.dc.b	$CE,$FF,$AC,$9B,$19,$8C,$7C,$CC,$8E,$0D,$7D,$67,$F8,$12,$9E,$47
	.dc.b	$FA,$F3,$E8,$BF,$A5,$CF,$D5,$6B,$C0,$38,$CF,$F9,$62,$F7,$79,$12
	.dc.b	$79,$3D,$7E,$92,$47,$56,$BF,$C9,$4F,$16,$D1,$F1,$3E,$5F,$56,$47
	.dc.b	$95,$AF,$96,$DE,$AF,$E0,$A1,$F4,$8D,$FF,$9F,$D6,$B3,$FF,$13,$36
	.dc.b	$BF,$A1,$9F,$E2,$87,$0E,$1A,$F2,$FA,$37,$D2,$72,$6C,$72,$3E,$3B
	.dc.b	$87,$93,$5D,$7D,$27,$F8,$DD,$FF,$1B,$B5,$34,$D7,$B5,$FE,$02,$C3
	.dc.b	$FA,$E4,$FE,$13,$97,$F5,$8A,$75,$1E,$CE,$4B,$3F,$D8,$1A,$BD,$57
	.dc.b	$87,$88,$E8,$7E,$06,$33,$5D,$B9,$B5,$E7,$ED,$B1,$E2,$8E,$5E,$86
	.dc.b	$2F,$F1,$47,$F3,$29,$F3,$C7,$8A,$DE,$77,$09,$FE,$0D,$18,$7F,$84
	.dc.b	$68,$E6,$F4,$BF,$B2,$8C,$C6,$39,$A4,$72,$F0,$2E,$5E,$B7,$E0,$FD
	.dc.b	$27,$C7,$9F,$E2,$8F,$C0,$C3,$FC,$10,$E6,$BF,$44,$E6,$7F,$C1,$2D
	.dc.b	$8F,$F3,$7E,$2E,$FE,$BC,$1E,$D7,$BF,$9F,$E0,$F3,$EE,$3E,$07,$0F
	.dc.b	$8B,$E1,$AE,$3E,$07,$CF,$ED,$7A,$3D,$CB,$5E,$87,$0F,$AB,$CF,$B7
	.dc.b	$E8,$7D,$FE,$7F,$C0,$F5,$AD,$7C,$07,$F0,$78,$7F,$B2,$63,$F9,$B3
	.dc.b	$6F,$F8,$12,$CF,$7F,$17,$C3,$0E,$38,$5C,$38,$BF,$8E,$FA,$4F,$AD
	.dc.b	$8F,$F0,$B7,$FC,$43,$3F,$D0,$DC,$F1,$7F,$8D,$DC,$CF,$F1,$1F,$A4
	.dc.b	$FC,$43,$7C,$1F,$CC,$C6,$2B,$FC,$7C,$FE,$FE,$FF,$A7,$2F,$84,$D9
	.dc.b	$C3,$5B,$3F,$D4,$FC,$E7,$C8,$7E,$91,$B1,$9F,$A4,$C4,$FE,$A3,$E9
	.dc.b	$1F,$0F,$7E,$1F,$0D,$BF,$EB,$DC,$AF,$0B,$F5,$ED,$8C,$F8,$58,$9F
	.dc.b	$D9,$FD,$23,$E5,$A9,$8F,$5E,$DF,$F6,$9F,$A4,$7D,$27,$EE,$9B,$19
	.dc.b	$F8,$FC,$4E,$7E,$93,$5F,$49,$6B,$93,$FC,$0B,$E9,$1D,$B7,$B7,$0D
	.dc.b	$AF,$EF,$1F,$49,$FB,$3E,$DF,$E8,$8F,$F8,$4E,$33,$E5,$FF,$0C,$8C
	.dc.b	$4E,$B6,$C1,$9F,$7A,$DF,$F8,$0F,$E9,$3F,$C8,$AD,$F8,$BF,$E2,$B1
	.dc.b	$FD,$5A,$30,$7F,$82,$7E,$93,$F6,$18,$1A,$7C,$D7,$FA,$C3,$8F,$17
	.dc.b	$F8,$FA,$F2,$2C,$61,$FE,$47,$D5,$FE,$0F,$BF,$F0,$B2,$F3,$73,$E0
	.dc.b	$69,$FE,$53,$F4,$8F,$8C,$BF,$0D,$86,$E1,$FE,$1F,$FA,$4F,$C3,$B7
	.dc.b	$C1,$FE,$63,$18,$DB,$FB,$72,$8C,$9F,$CD,$BE,$89,$CF,$8B,$FF,$15
	.dc.b	$46,$1F,$F1,$67,$D2,$7F,$C0,$06,$FB,$1F,$CF,$A3,$18,$FF,$7B,$7F
	.dc.b	$8E,$79,$7F,$EA,$56,$D7,$FA,$92,$FF,$D0,$CF,$23,$BF,$C8,$73,$F1
	.dc.b	$7A,$7B,$16,$7C,$2E,$1F,$1F,$9D,$72,$F0,$3F,$23,$E2,$BC,$9E,$8D
	.dc.b	$AF,$4B,$87,$D7,$D1,$6F,$D0,$FA,$CF,$89,$F1,$F6,$7E,$1B,$EF,$F3
	.dc.b	$FE,$07,$99,$67,$E2,$B8,$7F,$CA,$5F,$48,$71,$87,$F8,$7E,$31,$1F
	.dc.b	$33,$8F,$E5,$6F,$F2,$DB,$1F,$43,$E4,$7E,$71,$8B,$F9,$CF,$F6,$EB
	.dc.b	$FF,$24,$CF,$83,$E2,$7F,$13,$D9,$97,$FB,$9F,$D1,$FF,$9C,$A7,$FC
	.dc.b	$CC,$DC,$3F,$C6,$4F,$E3,$E3,$3D,$7F,$85,$E2,$91,$83,$F0,$AF,$4F
	.dc.b	$90,$91,$F8,$8F,$A4,$B8,$74,$F9,$0F,$09,$8E,$AF,$F7,$46,$48,$F2
	.dc.b	$FA,$33,$0F,$2F,$CF,$DC,$3F,$C8,$DD,$CC,$FF,$C1,$AD,$AF,$5F,$93
	.dc.b	$89,$FC,$E7,$4C,$9F,$F6,$D7,$4D,$C3,$DF,$F2,$3F,$10,$F8,$BE,$25
	.dc.b	$AF,$8B,$1F,$E1,$AF,$9D,$FE,$43,$5C,$27,$FD,$1D,$18,$E0,$F5,$F2
	.dc.b	$7F,$8A,$28,$38,$7E,$AB,$FE,$39,$6B,$FE,$44,$6B,$C0,$FD,$9F,$D1
	.dc.b	$73,$6E,$1F,$E4,$16,$9F,$E4,$3E,$94,$EF,$E8,$FF,$C3,$FE,$D6,$DB
	.dc.b	$5F,$E4,$97,$FC,$67,$E2,$E7,$D0,$FE,$AB,$3F,$E4,$DF,$B1,$FF,$13
	.dc.b	$BD,$DA,$E6,$3F,$C9,$91,$88,$FF,$2E,$FD,$23,$8F,$D5,$BF,$E8,$67
	.dc.b	$E1,$CF,$99,$FF,$5B,$35,$FF,$41,$3C,$5C,$CF,$2D,$7E,$24,$EE,$BA
	.dc.b	$52,$31,$3A,$AE,$13,$E0,$6B,$BD,$2D,$70,$8D,$1E,$03,$CB,$1A,$37
	.dc.b	$32,$32,$43,$D2,$E5,$F5,$C7,$17,$A1,$A3,$85,$FA,$AB,$C5,$1E,$BA
	.dc.b	$E5,$39,$AE,$53,$C4,$91,$97,$9D,$FC,$17,$BC,$65,$F1,$98,$8D,$38
	.dc.b	$24,$78,$EB,$C3,$3F,$42,$BC,$33,$9A,$E4,$7D,$C8,$CC,$F4,$9D,$4F
	.dc.b	$48,$E4,$F0,$3D,$C7,$A0,$E4,$66,$FF,$04,$F1,$32,$7C,$08,$EC,$74
	.dc.b	$74,$74,$34,$70,$9F,$84,$79,$09,$1C,$5C,$BD,$23,$F0,$67,$49,$FA
	.dc.b	$A7,$92,$78,$C7,$59,$E4,$71,$7C,$51,$D1,$D1,$D1,$E2,$FE,$4C,$F8
	.dc.b	$47,$08,$E5,$3E,$0A,$FA,$73,$E4,$3F,$5F,$7F,$EB,$D9,$F8,$F1,$83
	.dc.b	$F8,$33,$87,$EF,$99,$3F,$D9,$4E,$5F,$25,$78,$27,$F4,$FF,$3A,$F8
	.dc.b	$5F,$0D,$79,$4D,$9F,$C7,$A3,$4F,$D9,$9D,$C6,$23,$E9,$9C,$48,$E7
	.dc.b	$1F,$87,$87,$CD,$C3,$E5,$4E,$5F,$25,$7E,$26,$30,$F8,$8E,$93,$C5
	.dc.b	$F2,$61,$E8,$8C,$47,$82,$31,$87,$99,$B1,$D5,$AF,$A2,$BB,$BD,$A7
	.dc.b	$83,$EF,$38,$14,$E1,$3C,$B0,$FB,$F5,$CA,$FC,$64,$BF,$9F,$5B,$C3
	.dc.b	$E0,$B7,$0F,$75,$F1,$AC,$42,$30,$F8,$70,$FE,$4B,$18,$8F,$A3,$87
	.dc.b	$C2,$8C,$4F,$D0,$AF,$0D,$71,$C3,$C2,$D3,$C3,$87,$F2,$8F,$80,$91
	.dc.b	$FC,$E9,$FE,$57,$E2,$C8,$FC,$8B,$F9,$52,$3F,$94,$E1,$FE,$63,$EB
	.dc.b	$48,$FE,$87,$1F,$CD,$08,$E0,$FE,$8F,$E2,$FF,$3A,$FA,$3F,$E8,$3A
	.dc.b	$7F,$42,$6E,$7F,$DA,$5B,$5F,$DC,$9F,$F3,$53,$FD,$D2,$78,$79,$FC
	.dc.b	$EA,$4E,$5E,$51,$FD,$23,$E9,$3F,$D3,$33,$93,$A4,$78,$CD,$65,$E5
	.dc.b	$F9,$BE,$1E,$5C,$AE,$03,$24,$FA,$FB,$F8,$4E,$A7,$2D,$8F,$81,$E1
	.dc.b	$7C,$1F,$A4,$93,$B8,$CB,$48,$35,$F8,$DE,$27,$C9,$F2,$7F,$7B,$F1
	.dc.b	$12,$BC,$11,$96,$90,$E5,$C1,$F3,$7C,$3F,$A4,$48,$F4,$3F,$C2,$39
	.dc.b	$7E,$B7,$C4,$C9,$F6,$3F,$67,$C2,$91,$EF,$6B,$D8,$FF,$14,$FC,$E3
	.dc.b	$6F,$EF,$3F,$1F,$FE,$D4,$9F,$96,$D7,$80,$71,$B7,$F8,$19,$7