#include "z64.h"
#include "vi.h"

/*
 * only for printing messages to emulator's status bar
 */
#include <commctrl.h>

onetime onetimewarnings;

UINT8* rdram_8;
UINT16* rdram_16;
UINT32 plim =
    0x003FFFFF;
UINT32 idxlim16 =
    0x001FFFFF;
UINT32 idxlim32 =
    0x000FFFFF;
UINT8 hidden_bits[0x400000];

UINT32 gamma_table[0x100];
UINT32 gamma_dither_table[0x4000];
INT32 vi_restore_table[0x400];
INT32 oldvstart = 1337;

GLuint screen[PRESCALE_HEIGHT][PRESCALE_WIDTH];
GLuint* PreScale;

const GLbitfield buffers_mask =
    GL_COLOR_BUFFER_BIT
  | GL_STENCIL_BUFFER_BIT
  | GL_DEPTH_BUFFER_BIT;

/*
 * Update the emulator window screen size?
 * Here, `sync' indicates whether there are changes in the queue.
 */
BOOL sync = 1;

static UINT32 tvfadeoutstate[625];
static UINT32 brightness = 0;
static UINT32 prevwasblank = 0;

static const char* GL_errors[8] = {
    "GL_NO_ERROR", /* There is no current error. */
    "GL_INVALID_ENUM", /* Invalid parameter. */
    "GL_INVALID_VALUE", /* Invalid enum parameter value. */
    "GL_INVALID_OPERATION", /* Illegal call. */
    "GL_STACK_OVERFLOW",
    "GL_STACK_UNDERFLOW",
    "GL_OUT_OF_MEMORY", /* Unable to allocate memory. */

    "GL_UNKNOWN_ERROR" /* ??? */
};

static void video_filter16(
    int* r, int* g, int* b, UINT32 fboffset, UINT32 num, UINT32 hres,
    UINT32 centercvg);
static void video_filter32(
    int* endr, int* endg, int* endb, UINT32 fboffset, UINT32 num, UINT32 hres,
    UINT32 centercvg);
static void divot_filter(
    CCVG* final, CCVG centercolor, CCVG leftcolor, CCVG rightcolor);
static void restore_filter16(
    int* r, int* g, int* b, UINT32 fboffset, UINT32 num, UINT32 hres);
static void restore_filter32(
    int* r, int* g, int* b, UINT32 fboffset, UINT32 num, UINT32 hres);
static void gamma_filters(unsigned char* argb, int gamma_and_dither);
static void adjust_brightness(unsigned char* argb, int brightcoeff);
static void vi_vl_lerp(CCVG* up, CCVG down, UINT32 frac);
static void video_max_optimized(UINT32* Pixels, UINT32* pen);

static void vi_fetch_filter16(
    CCVG* res, UINT32 fboffset, UINT32 cur_x, UINT32 fsaa, UINT32 dither_filter,
    UINT32 vres);
static void vi_fetch_filter32(
    CCVG* res, UINT32 fboffset, UINT32 cur_x, UINT32 fsaa, UINT32 dither_filter,
    UINT32 vres);

static void do_frame_buffer_proper(
    UINT32 prescale_ptr, int hres, int vres, int x_start, int vitype,
    int linecount);
static void do_frame_buffer_raw(
    UINT32 prescale_ptr, int hres, int vres, int x_start, int vitype,
    int linecount);
static void (*do_frame_buffer[2])(UINT32, int, int, int, int, int) = {
    do_frame_buffer_proper, do_frame_buffer_raw
};

static void (*vi_fetch_filter_ptr)(
    CCVG*, UINT32, UINT32, UINT32, UINT32, UINT32);
static void (*vi_fetch_filter_func[2])(
    CCVG*, UINT32, UINT32, UINT32, UINT32, UINT32) = {
    vi_fetch_filter16, vi_fetch_filter32
};

static const GLfloat SSE1_lower_left[2][4] = {
    {  0.f,  0.f,  0.f, +1.f },
    { -1.f, -1.f,  0.f, +1.f },
};
static const GLfloat SSE1_lower_right[2][4] = {
    { +1.f,  0.f,  0.f, +1.f },
    { +1.f, -1.f,  0.f, +1.f },
};
static const GLfloat SSE1_upper_left[2][4] = {
    {  0.f, +1.f,  0.f, +1.f },
    { -1.f, +1.f,  0.f, +1.f },
};
static const GLfloat SSE1_upper_right[2][4] = {
    { +1.f, +1.f,  0.f, +1.f },
    { +1.f, +1.f,  0.f, +1.f },
};

GLuint name;
static void draw_screen_hard(int is_PAL, int shift)
{
    GLenum error;
    GLsizei height;

    height = (unsigned)(is_PAL ? 576 : 480) >> shift;

    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);

 /* glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL); */
    glTexImage2D(
        GL_TEXTURE_2D,
        0,
        4, /* needs more benchmark testing ... should this be GL_RGBA8? */
        PRESCALE_WIDTH,
        height,
        0,
        GL_BGRA_EXT,
        GL_UNSIGNED_BYTE,
        screen);
    glBegin(GL_QUADS);

    glTexCoord4fv(SSE1_lower_left[0]);
    glVertex4fv(SSE1_upper_left[1]);

    glTexCoord4fv(SSE1_upper_left[0]);
    glVertex4fv(SSE1_lower_left[1]);

    glTexCoord4fv(SSE1_upper_right[0]);
    glVertex4fv(SSE1_lower_right[1]);

    glTexCoord4fv(SSE1_lower_right[0]);
    glVertex4fv(SSE1_upper_right[1]);

    glEnd();
    error = glGetError();
    if (error != GL_NO_ERROR)
        DisplayGLError("Problem scaling the frame drawing.", error);
    return;
}

static void draw_screen_soft(int is_PAL, int shift)
{
    GLenum error;
    GLfloat scale_x, scale_y;
    GLsizei height;

    glDisable(GL_TEXTURE_2D);
/*
 * fixes a bug in my GeForce 6150 LE 9.18.13.783 drivers when using PJ64 1.x
 *
 * Apparently leaving GL_TEXTURE_2D enabled breaks glDrawPixels only on that
 * particular emulator AND only if not locally enforcing the generation and
 * deletion of the binded texture quad local to function drawn_screen_hard().
 */

    height = (unsigned)(is_PAL ? 576 : 480) >> shift;
#ifndef _DEBUG
    glPixelStorei(GL_UNPACK_ALIGNMENT, 8);
#endif

    if (is_full_screen != 0)
    {
        int dimensions[2];

        get_screen_size(dimensions);
        scale_x = (GLfloat)(dimensions[0]) / +(GLfloat)(640);
        scale_y = (GLfloat)(dimensions[1]) / -(GLfloat)(height);
    }
    else
    {
        GLint viewport[4];

        glGetIntegerv(GL_VIEWPORT, viewport);
        scale_x = (GLfloat)(viewport[2]) / +(GLfloat)(640);
        scale_y = (GLfloat)(viewport[3]) / -(GLfloat)(height);
    }
    glPixelZoom(scale_x, scale_y);
    glDrawPixels(640, height, GL_BGRA_EXT, GL_UNSIGNED_BYTE, screen);

    glEnable(GL_TEXTURE_2D); /* need this re-enabled for draw_screen_hard() */
    error = glGetError();
    if (error != GL_NO_ERROR)
        DisplayGLError("Problem scaling the frame drawing.", error);
    return;
}

static void (*draw_screen[2])(int is_PAL, int shift) = {
    draw_screen_soft, draw_screen_hard
};

int hres_old, vres_old; /* for optional FB size change detection */
void rdp_update(void)
{
    UINT32 prescale_ptr;
    UINT32 pix;
    UINT8 cur_cvg;
    int hres, vres;
    int h_start, v_start;
    int x_start;
    int h_end;
    int two_lines, line_shifter, line_count;
    int hrightblank;
    int vactivelines;
    int validh;
    int serration_pulses;
    int lowerfield;
    register int i, j;
    const int x_add = *GET_GFX_INFO(VI_X_SCALE_REG) & 0x00000FFF;
    const int v_sync = *GET_GFX_INFO(VI_V_SYNC_REG) & 0x000003FF;
    const int ispal  = (v_sync > 550);
    const int x1 = (*GET_GFX_INFO(VI_H_START_REG) >> 16) & 0x03FF;
    const int y1 = (*GET_GFX_INFO(VI_V_START_REG) >> 16) & 0x03FF;
    const int x2 = (*GET_GFX_INFO(VI_H_START_REG) >>  0) & 0x03FF;
    const int y2 = (*GET_GFX_INFO(VI_V_START_REG) >>  0) & 0x03FF;
    const int delta_x = x2 - x1;
    const int delta_y = y2 - y1;
    const int vitype = *GET_GFX_INFO(VI_STATUS_REG) & 0x00000003;
    const int pixel_size = sizeof(INT32);

    serration_pulses  = *GET_GFX_INFO(VI_STATUS_REG) >> 6;
    serration_pulses &= (y1 != oldvstart);
    lowerfield = serration_pulses & (ispal ? y1 < oldvstart : y1 > oldvstart);
    two_lines = serration_pulses ^ 0;
    line_shifter = serration_pulses ^ 1;
    line_count = pitchindwords << serration_pulses;

    hres = delta_x;
    vres = delta_y;
    h_start = x1 - (ispal ? 128 : 108);
    v_start = y1 - (ispal ?  47 :  37);
    x_start = (*gfx.VI_X_SCALE_REG >> 16) & 0x00000FFF;

    if (h_start < 0)
    {
        x_start -= x_add * h_start;
        h_start  = 0;
    }
    oldvstart = y1;
    v_start >>= 1;
    v_start  &= -(v_start >= 0);
    vres >>= 1;

    if (hres > PRESCALE_WIDTH - h_start)
        hres = PRESCALE_WIDTH - h_start;
    if (vres > PRESCALE_HEIGHT - v_start)
        vres = PRESCALE_HEIGHT - v_start;
    h_end = hres + h_start;

    hrightblank = PRESCALE_WIDTH - h_end;
    vactivelines = v_sync - (ispal ? 47 : 37);
    if (vactivelines > PRESCALE_HEIGHT)
    {
        DisplayError("VI_V_SYNC_REG too big");
        return;
    }
    if (vactivelines < 0)
        return;
    vactivelines >>= line_shifter;
    validh = (hres >= 0 && h_start >= 0 && h_start < PRESCALE_WIDTH);
    pix = 0;
    cur_cvg = 0;
    if (hres <= 0 || vres <= 0 || (!(vitype & 2) && prevwasblank))
        return;

    PreScale = (GLuint *)&screen[0];
    sync |= (hres != hres_old) | (vres != vres_old);
    if (sync & is_full_screen)
    { /* GL context calls apparently fail when directly in ChangeWindow. */
        GLfloat aspect_ratio, aspect_ratio_PC;
        GLint viewport[4];
        int dimensions[2];

        get_screen_size(dimensions);
#if 0
        aspect_ratio = (GLfloat)(hres)/2.f / (GLfloat)(vres);
#else
        aspect_ratio = 4.f / 3.f;
#endif
        aspect_ratio_PC = (GLfloat)dimensions[0] / (GLfloat)dimensions[1];

        if (aspect_ratio_PC < 1.f)
        { /* narrow, tall picture */
            viewport[2] = dimensions[0];
            viewport[3] = (GLint)((GLfloat)dimensions[0] / aspect_ratio);
            viewport[0] = 0;
            viewport[1] = (dimensions[1] - viewport[3]) / 2;
        }
        else
        { /* wide, short picture */
            viewport[3] = dimensions[1];
            viewport[2] = (GLint)((GLfloat)dimensions[1] * aspect_ratio);
            viewport[1] = 0;
            viewport[0] = (dimensions[0] - viewport[2]) / 2;
        }
        glViewport(viewport[0], viewport[1], viewport[2], viewport[3]);
        glRasterPos2d(-1., +1.);
        sync = FALSE;
    }
    else if (sync) /* not in full-screen */
    {
        GLint viewport[4] = { 0 };
        const int x_scale = *GET_GFX_INFO(VI_X_SCALE_REG) & 0x00000FFF;
        const int y_scale = *GET_GFX_INFO(VI_Y_SCALE_REG) & 0x00000FFF;

        if (gfx.hStatusBar != NULL)
        {
            RECT status_bar;

            GetClientRect(gfx.hStatusBar, &status_bar);
            viewport[0] = status_bar.left;
            viewport[1] = status_bar.bottom;
        }
        switch (cfg[4])
        {
            case 0x00:
                viewport[2] = 640;
                viewport[3] = 480;
                break;
            case 0x01:
                viewport[2] = (GLsizei)(hres * x_scale/1024) - (hres & 1);
                viewport[3] = (GLsizei)(vres * y_scale/1024);
                break;
            case 0x02:
                viewport[2] = (cfg[0] << 8) | cfg[1];
                viewport[3] = (cfg[2] << 8) | cfg[3];
                viewport[2] = (viewport[2] & 0x0000FFFF) + 1;
                viewport[3] = (viewport[3] & 0x0000FFFF) + 1;
                break;
            default:
                DisplayError("Invalid resolution control setting.");
                break;
        }
        screen_resize(viewport[2], viewport[3]);
        glViewport(viewport[0], viewport[1], viewport[2], viewport[3]);
        glRasterPos2d(-1., +1.);
#if 0
        sync = FALSE;
#else
        sync <<= 16; /* timing fix when exiting full screen with hStatusBar */
#endif
    }
    hres_old = hres;
    vres_old = vres;

    if (vitype >> 1 == 0)
    {
        zerobuf(tvfadeoutstate, pixel_size*PRESCALE_HEIGHT);
        for (i = 0; i < PRESCALE_HEIGHT; i++)
            zerobuf(&PreScale[i * pitchindwords], pixel_size*PRESCALE_WIDTH);
        prevwasblank = 1;
        goto no_frame_buffer;
    }
#undef RENDER_CVG_BITS16
#undef RENDER_CVG_BITS32
#undef RENDER_MIN_CVG_ONLY
#undef RENDER_MAX_CVG_ONLY

#undef MONITOR_Z
#undef BW_ZBUFFER
#undef ZBUFF_AS_16B_IATEXTURE

#ifdef MONITOR_Z
    frame_buffer = zb_address;
#endif

    prevwasblank = 0;
    if (h_start > 0 && h_start < PRESCALE_WIDTH)
        for (i = 0; i < vactivelines; i++)
            zerobuf(&PreScale[i*pitchindwords], pixel_size*h_start);

    if (h_end >= 0 && h_end < PRESCALE_WIDTH)
        for (i = 0; i < vactivelines; i++)
            zerobuf(&PreScale[i*pitchindwords + h_end], pixel_size*hrightblank);

    for (i = 0; i < (v_start << two_lines) + lowerfield; i++)
    {
        tvfadeoutstate[i] >>= 1;
        if (tvfadeoutstate[i] & validh)
            zerobuf(&PreScale[i*pitchindwords + h_start], pixel_size*hres);
    }

    if (serration_pulses == 0)
        for (j = 0; j < vres; j++)
            tvfadeoutstate[i++] = 2;
    else
        for (j = 0; j < vres; j++)
        {
            tvfadeoutstate[i] = 2;
            ++i;
            tvfadeoutstate[i] >>= 1;
            if (~tvfadeoutstate[i] & validh)
                zerobuf(&PreScale[i*pitchindwords + h_start], pixel_size*hres);
            ++i;
        }

    while (i < vactivelines)
    {
        tvfadeoutstate[i] >>= 1;
        if (~tvfadeoutstate[i] & validh)
            zerobuf(&PreScale[i*pitchindwords + h_start], pixel_size*hres);
        ++i;
    }

    prescale_ptr =
        (v_start * line_count) + h_start + (lowerfield ? pitchindwords : 0);
    do_frame_buffer[cfg[23] & 0x01](
        prescale_ptr, hres, vres, x_start, vitype, line_count);

no_frame_buffer:
    if (dst.left < dst.right && dst.top < dst.bottom)
    {
#ifdef _DEBUG
        glClear(buffers_mask | GL_ACCUM_BUFFER_BIT);
#else
        glClear(buffers_mask);
#endif
        draw_screen[!!(cfg[23] & 0x02)](ispal, line_shifter);
        i = swap_buffers();
        if (i == 0)
            DisplayError("Failed to swap buffers.");
    }
    return;
}

static void do_frame_buffer_proper(
    UINT32 prescale_ptr, int hres, int vres, int x_start, int vitype,
    int linecount)
{
    GLubyte bgra[4];
    GLuint* scanline;
    CCVG viaa_array[2048];
    CCVG divot_array[2048];
    CCVG *viaa_cache, *viaa_cache_next, *divot_cache, *divot_cache_next;
    CCVG *tempccvgptr;
    CCVG color, nextcolor, scancolor, scannextcolor;
    UINT32 pixels = 0, nextpixels = 0;
    UINT32 prevy = 0;
    UINT32 y_start = (vi_y_scale >> 16) & 0x0FFF;
	UINT32 frame_buffer = vi_origin & 0x00FFFFFF;
    signed int cache_marker_init;
    int line_x = 0, next_line_x = 0, prev_line_x = 0, far_line_x = 0;
    int prev_scan_x = 0, scan_x = 0, next_scan_x = 0, far_scan_x = 0;
    int prev_x = 0, cur_x = 0, next_x = 0, far_x = 0;
    int cache_marker = 0, cache_next_marker = 0, divot_cache_marker = 0, divot_cache_next_marker = 0;
    int xfrac = 0, yfrac = 0;
    int slowbright;
    int lerping = 0;
    int vi_width_low = vi_width & 0xFFF;
    const int x_add = *GET_GFX_INFO(VI_X_SCALE_REG) & 0x00000FFF;
    UINT32 y_add = vi_y_scale & 0xfff;
    register int i, j;
    const int gamma_dither     = !!(*GET_GFX_INFO(VI_STATUS_REG) & 0x00000004);
    const int gamma            = !!(*GET_GFX_INFO(VI_STATUS_REG) & 0x00000008);
    const int divot            = !!(*GET_GFX_INFO(VI_STATUS_REG) & 0x00000010);
    const int clock_enable     = !!(*GET_GFX_INFO(VI_STATUS_REG) & 0x00000020);
    const int extralines       =  !(*GET_GFX_INFO(VI_STATUS_REG) & 0x00000100);
    const int fsaa             =  !(*GET_GFX_INFO(VI_STATUS_REG) & 0x00000200);
    const int dither_filter    = !!(*GET_GFX_INFO(VI_STATUS_REG) & 0x00010000);
    const int gamma_and_dither = (gamma << 1) | gamma_dither;
    const int lerp_en          = fsaa | extralines;

    if (frame_buffer == 0)
        return;

    if (clock_enable)
        DisplayError(
            "rdp_update: vbus_clock_enable bit set in VI_CONTROL_REG "\
            "register. Never run this code on your N64! It's rumored that "\
            "turning this bit on will result in permanent damage to the "\
            "hardware! Emulation will now continue.");

    viaa_cache = &viaa_array[0];
    viaa_cache_next = &viaa_array[1024];
    divot_cache = &divot_array[0];
    divot_cache_next = &divot_array[1024];

    cache_marker_init  = (x_start >> 10) - 2;
    cache_marker_init |= -(cache_marker_init < 0);

    slowbright = 0;
#ifdef WIN32
    if (GetAsyncKeyState(0x91))
        brightness = ++brightness & 0xF;
    slowbright = brightness >> 1;
#endif
    pixels = 0;

    for (j = 0; j < vres; j++)
    {
        x_start = (vi_x_scale >> 16) & 0x0FFF;

        if ((y_start >> 10) == (prevy + 1) && j)
        {
            cache_marker = cache_next_marker;
            cache_next_marker = cache_marker_init;

            tempccvgptr = viaa_cache;
            viaa_cache = viaa_cache_next;
            viaa_cache_next = tempccvgptr;
            if (divot == 0)
                {/* do nothing and branch */}
            else
            {
                divot_cache_marker = divot_cache_next_marker;
                divot_cache_next_marker = cache_marker_init;
                tempccvgptr = divot_cache;
                divot_cache = divot_cache_next;
                divot_cache_next = tempccvgptr;
            }
        }
        else if ((y_start >> 10) != prevy || !j)
        {
            cache_marker = cache_next_marker = cache_marker_init;
            if (divot == 0)
                {/* do nothing and branch */}
            else
                divot_cache_marker
              = divot_cache_next_marker
              = cache_marker_init;
        }

        scanline = &PreScale[prescale_ptr];
        prescale_ptr += linecount;

        prevy = y_start >> 10;
        yfrac = (y_start >> 5) & 0x1f;
        pixels = vi_width_low * prevy;
        nextpixels = pixels + vi_width_low;

        for (i = 0; i < hres; i++)
        {
            line_x = x_start >> 10;
            prev_line_x = line_x - 1;
            next_line_x = line_x + 1;
            far_line_x = line_x + 2;

            cur_x = pixels + line_x;
            prev_x = pixels + prev_line_x;
            next_x = pixels + next_line_x;
            far_x = pixels + far_line_x;

            scan_x = nextpixels + line_x;
            prev_scan_x = nextpixels + prev_line_x;
            next_scan_x = nextpixels + next_line_x;
            far_scan_x = nextpixels + far_line_x;

            xfrac = (x_start >> 5) & 0x1f;
            lerping = lerp_en & (xfrac || yfrac);

            if (prev_line_x > cache_marker)
            {
                vi_fetch_filter_func[vitype & 1](
                    &viaa_cache[prev_line_x], frame_buffer, prev_x, fsaa,
                    dither_filter, vres);
                vi_fetch_filter_func[vitype & 1](
                    &viaa_cache[line_x], frame_buffer, cur_x, fsaa,
                    dither_filter, vres);
                vi_fetch_filter_func[vitype & 1](
                    &viaa_cache[next_line_x], frame_buffer, next_x, fsaa,
                    dither_filter, vres);
                cache_marker = next_line_x;
            }
            else if (line_x > cache_marker)
            {
                vi_fetch_filter_func[vitype & 1](
                    &viaa_cache[line_x], frame_buffer, cur_x, fsaa,
                    dither_filter, vres);
                vi_fetch_filter_func[vitype & 1](
                    &viaa_cache[next_line_x], frame_buffer, next_x, fsaa,
                    dither_filter, vres);
                cache_marker = next_line_x;
            }
            else if (next_line_x > cache_marker)
            {
                vi_fetch_filter_func[vitype & 1](
                    &viaa_cache[next_line_x], frame_buffer, next_x, fsaa,
                    dither_filter, vres);
                cache_marker = next_line_x;
            }

            if (prev_line_x > cache_next_marker)
            {
                vi_fetch_filter_func[vitype & 1](
                    &viaa_cache_next[prev_line_x], frame_buffer, prev_scan_x,
                    fsaa, dither_filter, vres);
                vi_fetch_filter_func[vitype & 1](
                    &viaa_cache_next[line_x], frame_buffer, scan_x, fsaa,
                    dither_filter, vres);
                vi_fetch_filter_func[vitype & 1](
                    &viaa_cache_next[next_line_x], frame_buffer, next_scan_x,
                    fsaa, dither_filter, vres);
                cache_next_marker = next_line_x;
            }
            else if (line_x > cache_next_marker)
            {
                vi_fetch_filter_func[vitype & 1](
                    &viaa_cache_next[line_x], frame_buffer, scan_x, fsaa,
                    dither_filter, vres);
                vi_fetch_filter_func[vitype & 1](
                    &viaa_cache_next[next_line_x], frame_buffer, next_scan_x,
                    fsaa, dither_filter, vres);
                cache_next_marker = next_line_x;
            }
            else if (next_line_x > cache_next_marker)
            {
                vi_fetch_filter_func[vitype & 1](
                    &viaa_cache_next[next_line_x], frame_buffer, next_scan_x,
                    fsaa, dither_filter, vres);
                cache_next_marker = next_line_x;
            }

            if (divot == 0)
                color = viaa_cache[line_x];
            else
            {
                if (far_line_x > cache_marker)
                {
                    vi_fetch_filter_func[vitype & 1](
                        &viaa_cache[far_line_x], frame_buffer, far_x, fsaa,
                        dither_filter, vres);
                    cache_marker = far_line_x;
                }

                if (far_line_x > cache_next_marker)
                {
                    vi_fetch_filter_func[vitype & 1](
                        &viaa_cache_next[far_line_x], frame_buffer, far_scan_x,
                        fsaa, dither_filter, vres);
                    cache_next_marker = far_line_x;
                }

                if (line_x > divot_cache_marker)
                {
                    divot_filter(
                        &divot_cache[line_x], viaa_cache[line_x],
                        viaa_cache[prev_line_x], viaa_cache[next_line_x]);
                    divot_filter(
                        &divot_cache[next_line_x], viaa_cache[next_line_x],
                        viaa_cache[line_x], viaa_cache[far_line_x]);
                    divot_cache_marker = next_line_x;
                }
                else if (next_line_x > divot_cache_marker)
                {
                    divot_filter(
                        &divot_cache[next_line_x], viaa_cache[next_line_x],
                        viaa_cache[line_x], viaa_cache[far_line_x]);
                    divot_cache_marker = next_line_x;
                }

                if (line_x > divot_cache_next_marker)
                {
                    divot_filter(
                        &divot_cache_next[line_x], viaa_cache_next[line_x],
                        viaa_cache_next[prev_line_x],
                        viaa_cache_next[next_line_x]);
                    divot_filter(
                        &divot_cache_next[next_line_x],
                        viaa_cache_next[next_line_x], viaa_cache_next[line_x],
                        viaa_cache_next[far_line_x]);
                    divot_cache_next_marker = next_line_x;
                }
                else if (next_line_x > divot_cache_next_marker)
                {
                    divot_filter(
                        &divot_cache_next[next_line_x],
                        viaa_cache_next[next_line_x], viaa_cache_next[line_x],
                        viaa_cache_next[far_line_x]);
                    divot_cache_next_marker = next_line_x;
                }
                color = divot_cache[line_x];
            }

            if (lerping)
            {
                if (divot == 0)
                { /* branch unlikely */
                    nextcolor = viaa_cache[next_line_x];
                    scancolor = viaa_cache_next[line_x];
                    scannextcolor = viaa_cache_next[next_line_x];
                }
                else
                {
                    nextcolor = divot_cache[next_line_x];
                    scancolor = divot_cache_next[line_x];
                    scannextcolor = divot_cache_next[next_line_x];
                }
                if (yfrac == 0)
                    {}
                else
                {
                    vi_vl_lerp(&color, scancolor, yfrac);
                    vi_vl_lerp(&nextcolor, scannextcolor, yfrac);
                }
                if (xfrac == 0)
                    {}
                else
                    vi_vl_lerp(&color, nextcolor, xfrac);
            }
            bgra[2] = color.r;
            bgra[1] = color.g;
            bgra[0] = color.b;

            gamma_filters(bgra, gamma_and_dither);
#ifdef BW_ZBUFFER
            UINT32 tempz = RREADIDX16((frame_buffer >> 1) + cur_x);

            pix = tempz;
            bgra[0] = bgra[1] = bgra[2] = pix >> 8;
#endif
#ifdef ZBUFF_AS_16B_IATEXTURE
            bgra[0] = bgra[1] = bgra[2] =
                (unsigned char)(pix >> 8)*(unsigned char)(pix >> 0) >> 8;
#endif
#ifdef RENDER_CVG_BITS16
            bgra[0] = bgra[1] = bgra[2] = cur_cvg << 5;
#endif
#ifdef RENDER_CVG_BITS32
            bgra[0] = bgra[1] = bgra[2] = cur_cvg << 5;
#endif
#ifdef RENDER_MIN_CVG_ONLY
            if (!cur_cvg)
                bgra[0] = bgra[1] = bgra[2] = 0x00;
            else
                bgra[0] = bgra[1] = bgra[2] = 0xFF;
#endif
#ifdef RENDER_MAX_CVG_ONLY
            if (cur_cvg != 7)
                bgra[0] = bgra[1] = bgra[2] = 0x00;
            else
                bgra[0] = bgra[1] = bgra[2] = 0xFF;
#endif
            x_start += x_add;
            scanline[i] = *(GLint *)bgra;
            if (slowbright == 0)
                continue; /* branch very likely */
            adjust_brightness(bgra, slowbright);
            scanline[i] = *(GLint *)bgra;
        }
        y_start += y_add;
    }
}
static void do_frame_buffer_raw(
    UINT32 prescale_ptr, int hres, int vres, int x_start, int vitype,
    int linecount)
{
    GLubyte bgra[4];
    GLuint* scanline;
    int pixels;
    int prevy, y_start;
    int cur_x, line_x;
    register int i;
    const int frame_buffer = *GET_GFX_INFO(VI_ORIGIN_REG) & 0x00FFFFFF;
    const int VI_width = *GET_GFX_INFO(VI_WIDTH_REG) & 0x00000FFF;
    const int x_add = *GET_GFX_INFO(VI_X_SCALE_REG) & 0x00000FFF;
    const int y_add = *GET_GFX_INFO(VI_Y_SCALE_REG) & 0x00000FFF;

    if (frame_buffer == 0)
        return;
    bgra[3] = 0xFF;
    y_start = *GET_GFX_INFO(VI_Y_SCALE_REG)>>16 & 0x0FFF;

    if (vitype & 1) /* 32-bit RGBA (branch unlikely) */
    {
        while (--vres >= 0)
        {
            x_start = *GET_GFX_INFO(VI_X_SCALE_REG)>>16 & 0x0FFF;
            scanline = &PreScale[prescale_ptr];
            prescale_ptr += linecount;

            prevy = y_start >> 10;
            pixels = VI_width * prevy;

            for (i = 0; i < hres; i++)
            {
                GLuint pixel;
                unsigned long addr;

                line_x = x_start >> 10;
                cur_x = pixels + line_x;

                x_start += x_add;
                addr = frame_buffer + 4*cur_x;
                if (plim < addr)
                    continue;
                pixel = *(GLuint *)(DRAM + addr);
                bgra[2] = (pixel >> 24) & 0xFF;
                bgra[1] = (pixel >> 16) & 0xFF;
                bgra[0] = (pixel >>  8) & 0xFF;
             /* bgra[3] = (pixel >>  0) & 0xFF; */
                scanline[i] = *(GLuint *)(bgra);
            }
            y_start += y_add;
        }
    }
    else /* 16-bit RRRRR GGGGG BBBBB A */
    {
        while (--vres >= 0)
        {
            x_start = *GET_GFX_INFO(VI_X_SCALE_REG)>>16 & 0x0FFF;
            scanline = &PreScale[prescale_ptr];
            prescale_ptr += linecount;

            prevy = y_start >> 10;
            pixels = VI_width * prevy;

            for (i = 0; i < hres; i++)
            {
                GLushort pixel;
                unsigned long addr;

                line_x = x_start >> 10;
                cur_x = pixels + line_x;

                x_start += x_add;
                addr = frame_buffer + 2*cur_x;
                if (plim < addr)
                    continue;
                addr = addr ^ (WORD_ADDR_XOR << 1);
                pixel = *(GLushort *)(DRAM + addr);
                bgra[2] = (pixel & 0xF800) >> (11 - 3);
                bgra[1] = (pixel & 0x07C0) >> (6 - 3);
                bgra[0] = (pixel & 0x003E) << 2 >> (1 - 1);
             /* bgra[3] = (pixel & 0x0001) ? ~0x00 : 0x00; */
                scanline[i] = *(GLuint *)(bgra);
            }
            y_start += y_add;
        }
    }
    return;
}

static void vi_fetch_filter16(
    CCVG* res, UINT32 fboffset, UINT32 cur_x, UINT32 fsaa, UINT32 dither_filter,
    UINT32 vres)
{
    int r, g, b;
    UINT32 pix, hval;
    UINT32 cur_cvg;
    UINT32 idx = (fboffset >> 1) + cur_x;
    UINT32 fbw = vi_width & 0xfff;

    PAIRREAD16(pix, hval, idx);
    if (fsaa)
        cur_cvg = ((pix & 1) << 2) | hval;
    else
        cur_cvg = 7;
    r = GET_HI(pix);
    g = GET_MED(pix);
    b = GET_LOW(pix);

    if (cur_cvg == 7)
    {
        if (dither_filter)
            restore_filter16(&r, &g, &b, fboffset, cur_x, fbw);
    }
    else
    {
        video_filter16(&r, &g, &b, fboffset, cur_x, fbw, cur_cvg);
    }

    res -> r = r;
    res -> g = g;
    res -> b = b;
    res -> cvg = cur_cvg;
    return;
}

static void vi_fetch_filter32(
    CCVG* res, UINT32 fboffset, UINT32 cur_x, UINT32 fsaa, UINT32 dither_filter,
    UINT32 vres)
{
    int r, g, b;
    UINT32 cur_cvg;
    UINT32 pix = RREADIDX32((fboffset >> 2) + cur_x);
    UINT32 fbw = vi_width & 0xfff;

    if (fsaa)
        cur_cvg = (pix >> 5) & 7;
    else
        cur_cvg = 7;

    r = (pix >> 24) & 0xff;
    g = (pix >> 16) & 0xff;
    b = (pix >> 8) & 0xff;

    if (cur_cvg == 7)
    {
        if (dither_filter)
            restore_filter32(&r, &g, &b, fboffset, cur_x, fbw);
    }
    else
    {
        video_filter32(&r, &g, &b, fboffset, cur_x, fbw, cur_cvg);
    }

    res -> r = r;
    res -> g = g;
    res -> b = b;
    res -> cvg = cur_cvg;
    return;
}

static void video_filter16(
    int* endr, int* endg, int* endb, UINT32 fboffset, UINT32 num, UINT32 hres,
    UINT32 centercvg)
{
    UINT32 penumaxr, penumaxg, penumaxb, penuminr, penuming, penuminb;
    UINT16 pix;
    UINT32 numoffull = 1;
    UINT32 hidval;
    UINT32 r, g, b; 
    UINT32 backr[7], backg[7], backb[7];
    UINT32 invr[7], invg[7], invb[7];
    UINT32 colr, colg, colb;

    UINT32 idx = (fboffset >> 1) + num;
    UINT32 leftup = idx - hres - 1;
    UINT32 rightup = idx - hres + 1;
    UINT32 toleft = idx - 2;
    UINT32 toright = idx + 2;
    UINT32 leftdown = idx + hres - 1;
    UINT32 rightdown = idx + hres + 1;
    UINT32 coeff = 7 - centercvg;

    r = *endr;
    g = *endg;
    b = *endb;

    backr[0] = r;
    backg[0] = g;
    backb[0] = b;
    invr[0] = (~r) & 0xff;
    invg[0] = (~g) & 0xff;
    invb[0] = (~b) & 0xff;

    VI_ANDER(leftup);
    VI_ANDER(rightup);
    VI_ANDER(toleft);
    VI_ANDER(toright);
    VI_ANDER(leftdown);
    VI_ANDER(rightdown);

    video_max_optimized(&backr[0], &penumaxr);
    video_max_optimized(&backg[0], &penumaxg);
    video_max_optimized(&backb[0], &penumaxb);
    video_max_optimized(&invr[0], &penuminr);
    video_max_optimized(&invg[0], &penuming);
    video_max_optimized(&invb[0], &penuminb);

    penuminr = (~penuminr) & 0xFF;
    penuming = (~penuming) & 0xFF;
    penuminb = (~penuminb) & 0xFF;

    colr = penuminr + penumaxr - (r << 1);
    colg = penuming + penumaxg - (g << 1);
    colb = penuminb + penumaxb - (b << 1);

    colr = (((colr * coeff) + 4) >> 3) + r;
    colg = (((colg * coeff) + 4) >> 3) + g;
    colb = (((colb * coeff) + 4) >> 3) + b;

    *endr = colr & 0xFF;
    *endg = colg & 0xFF;
    *endb = colb & 0xFF;
    return;
}

static void video_filter32(
    int* endr, int* endg, int* endb, UINT32 fboffset, UINT32 num, UINT32 hres,
    UINT32 centercvg)
{
    UINT32 penumaxr, penumaxg, penumaxb, penuminr, penuming, penuminb;
    UINT32 numoffull = 1;
    UINT32 pix = 0, pixcvg = 0;
    UINT32 r, g, b; 
    UINT32 backr[7], backg[7], backb[7];
    UINT32 invr[7], invg[7], invb[7];
    UINT32 colr, colg, colb;

    UINT32 idx = (fboffset >> 2) + num;
    UINT32 leftup = idx - hres - 1;
    UINT32 rightup = idx - hres + 1;
    UINT32 toleft = idx - 2;
    UINT32 toright = idx + 2;
    UINT32 leftdown = idx + hres - 1;
    UINT32 rightdown = idx + hres + 1;
    UINT32 coeff = 7 - centercvg;

    r = *endr;
    g = *endg;
    b = *endb;

    backr[0] = r;
    backg[0] = g;
    backb[0] = b;
    invr[0] = (~r) & 0xff;
    invg[0] = (~g) & 0xff;
    invb[0] = (~b) & 0xff;

    VI_ANDER32(leftup);
    VI_ANDER32(rightup);
    VI_ANDER32(toleft);
    VI_ANDER32(toright);
    VI_ANDER32(leftdown);
    VI_ANDER32(rightdown);

    video_max_optimized(&backr[0], &penumaxr);
    video_max_optimized(&backg[0], &penumaxg);
    video_max_optimized(&backb[0], &penumaxb);
    video_max_optimized(&invr[0], &penuminr);
    video_max_optimized(&invg[0], &penuming);
    video_max_optimized(&invb[0], &penuminb);

    penuminr = (~penuminr) & 0xFF;
    penuming = (~penuming) & 0xFF;
    penuminb = (~penuminb) & 0xFF;

    colr = penuminr + penumaxr - (r << 1);
    colg = penuming + penumaxg - (g << 1);
    colb = penuminb + penumaxb - (b << 1);

    colr = (((colr * coeff) + 4) >> 3) + r;
    colg = (((colg * coeff) + 4) >> 3) + g;
    colb = (((colb * coeff) + 4) >> 3) + b;

    *endr = colr & 0xFF;
    *endg = colg & 0xFF;
    *endb = colb & 0xFF;
    return;
}

static void divot_filter(
    CCVG* final, CCVG centercolor, CCVG leftcolor, CCVG rightcolor)
{
    UINT32 leftr, leftg, leftb;
    UINT32 rightr, rightg, rightb;
    UINT32 centerr, centerg, centerb;

    *final = centercolor;
    if ((centercolor.cvg & leftcolor.cvg & rightcolor.cvg) == 7)
        return;

    leftr = leftcolor.r;    
    leftg = leftcolor.g;    
    leftb = leftcolor.b;
    rightr = rightcolor.r;    
    rightg = rightcolor.g;    
    rightb = rightcolor.b;
    centerr = centercolor.r;
    centerg = centercolor.g;
    centerb = centercolor.b;

    if ((leftr >= centerr && rightr >= leftr) || (leftr >= rightr && centerr >= leftr))
        final -> r = leftr;
    else if ((rightr >= centerr && leftr >= rightr) || (rightr >= leftr && centerr >= rightr))
        final -> r = rightr;

    if ((leftg >= centerg && rightg >= leftg) || (leftg >= rightg && centerg >= leftg))
        final -> g = leftg;
    else if ((rightg >= centerg && leftg >= rightg) || (rightg >= leftg && centerg >= rightg))
        final -> g = rightg;

    if ((leftb >= centerb && rightb >= leftb) || (leftb >= rightb && centerb >= leftb))
        final -> b = leftb;
    else if ((rightb >= centerb && leftb >= rightb) || (rightb >= leftb && centerb >= rightb))
        final -> b = rightb;
    return;
}

static void restore_filter16(
    int* r, int* g, int* b, UINT32 fboffset, UINT32 num, UINT32 hres)
{
    UINT32 tempr, tempg, tempb;
    UINT16 pix;

    UINT32 idx = (fboffset >> 1) + num;
    UINT32 leftuppix = idx - hres - 1;
    UINT32 leftdownpix = idx + hres - 1;
    UINT32 toleftpix = idx - 1;

    UINT32 rend = *r;
    UINT32 gend = *g;
    UINT32 bend = *b;
    UINT32 rcomp = (rend >> 3) & 31;
    UINT32 gcomp = (gend >> 3) & 31;
    UINT32 bcomp = (bend >> 3) & 31;

    VI_COMPARE(leftuppix);
    VI_COMPARE(leftuppix + 1);
    VI_COMPARE(leftuppix + 2);
    VI_COMPARE(leftdownpix);
    VI_COMPARE(leftdownpix + 1);
    VI_COMPARE(leftdownpix + 2);
    VI_COMPARE(toleftpix);
    VI_COMPARE(toleftpix + 2);

    *r = rend;
    *g = gend;
    *b = bend;
    return;
}

static void restore_filter32(
    int* r, int* g, int* b, UINT32 fboffset, UINT32 num, UINT32 hres)
{
    UINT32 tempr, tempg, tempb;
    UINT32 pix;

    UINT32 idx = (fboffset >> 2) + num;
    UINT32 leftuppix = idx - hres - 1;
    UINT32 leftdownpix = idx + hres - 1;
    UINT32 toleftpix = idx - 1;

    UINT32 rend = *r;
    UINT32 gend = *g;
    UINT32 bend = *b;
    UINT32 rcomp = (rend >> 3) & 31;
    UINT32 gcomp = (gend >> 3) & 31;
    UINT32 bcomp = (bend >> 3) & 31;

    VI_COMPARE32(leftuppix);
    VI_COMPARE32(leftuppix + 1);
    VI_COMPARE32(leftuppix + 2);
    VI_COMPARE32(leftdownpix);
    VI_COMPARE32(leftdownpix + 1);
    VI_COMPARE32(leftdownpix + 2);
    VI_COMPARE32(toleftpix);
    VI_COMPARE32(toleftpix + 2);

    *r = rend;
    *g = gend;
    *b = bend;
    return;
}

static void gamma_filters(unsigned char* bgra, int gamma_and_dither)
{
    int cdith, dith;
    int r, g, b;

    r = bgra[2];
    g = bgra[1];
    b = bgra[0];

    switch (gamma_and_dither)
    {
        case 0:
            return;
            break;
        case 1:
            cdith = irand();
            r += cdith & (r < 255);
            cdith = cdith >> 1;
            g += cdith & (g < 255);
            cdith = cdith >> 2;
            b += cdith & (b < 255);
            break;
        case 2:
            r = gamma_table[r];
            g = gamma_table[g];
            b = gamma_table[b];
            break;
        case 3:
            cdith = irand();
            dith = cdith & 0x3F;
            r = gamma_dither_table[(r << 6) | dith];
            dith = (cdith >> 6) & 0x3F;
            g = gamma_dither_table[(g << 6) | dith];
            dith = (cdith >> 9)&0x38 | (cdith & 0x07);
            b = gamma_dither_table[(b << 6) | dith];
            break;
    }
    bgra[2] = (unsigned char)(r);
    bgra[1] = (unsigned char)(g);
    bgra[0] = (unsigned char)(b);
    return;
}

static void adjust_brightness(unsigned char* bgra, int brightcoeff)
{
    int r, g, b;

    r = bgra[2];
    g = bgra[1];
    b = bgra[0];
    brightcoeff &= 7;
    switch (brightcoeff)
    {
        case 0:    
            break;
        case 1: 
        case 2:
        case 3:
            r += (r >> (4 - brightcoeff));
            g += (g >> (4 - brightcoeff));
            b += (b >> (4 - brightcoeff));
            if (r > 0xFF)
                r = 0xFF;
            if (g > 0xFF)
                g = 0xFF;
            if (b > 0xFF)
                b = 0xFF;
            break;
        case 4:
        case 5:
        case 6:
        case 7:
            r = (r + 1) << (brightcoeff - 3);
            g = (g + 1) << (brightcoeff - 3);
            b = (b + 1) << (brightcoeff - 3);
            if (r > 0xFF)
                r = 0xFF;
            if (g > 0xFF)
                g = 0xFF;
            if (b > 0xFF)
                b = 0xFF;
            break;
    }
    bgra[2] = (unsigned char)(r);
    bgra[1] = (unsigned char)(g);
    bgra[0] = (unsigned char)(b);
    return;
}

static void vi_vl_lerp(CCVG* up, CCVG down, UINT32 frac)
{
    UINT32 r0, g0, b0;

    if (frac == 0)
        return;

    r0 = up -> r;
    g0 = up -> g;
    b0 = up -> b;

    up -> r = (((frac*(down.r - r0) + 16) >> 5) + r0) & 0xFF;
    up -> g = (((frac*(down.g - g0) + 16) >> 5) + g0) & 0xFF;
    up -> b = (((frac*(down.b - b0) + 16) >> 5) + b0) & 0xFF;
    return;
}

static void video_max_optimized(UINT32* Pixels, UINT32* pen)
{
    int i;
    int pos;
    UINT32 max;
    UINT32 curpen = Pixels[0];

    pos = 0;
    for (i = 1; i < 7; i++)
    {
        if (Pixels[i] > Pixels[pos])
        {
            curpen = Pixels[pos];
            pos = i;            
        }
    }
    max = Pixels[pos];
    if (curpen != max)
    {
        for (i = pos + 1; i < 7; i++)
        {
            if (Pixels[i] > curpen)
            {
                curpen = Pixels[i];
            }
        }
    }
    *pen = curpen;
    return;
}

NOINLINE void DisplayError(char * error)
{
    const HWND hWnd = GET_GFX_INFO(hWnd);

    MessageBox(hWnd, error, NULL, MB_ICONERROR);
    return;
}
NOINLINE void DisplayWarning(char * warning)
{
    const HWND hWnd = GET_GFX_INFO(hWnd);

    MessageBox(hWnd, warning, "Warning", MB_ICONWARNING);
    return;
}

/*
 * idea picked up from the source to Nemu64 graphics plugin
 * thanks to Lemmy for the example
 */
NOINLINE void DisplayInStatusPanel(char * message)
{
    const HWND hStatusBar = GET_GFX_INFO(hStatusBar);

    if (hStatusBar == NULL)
        DisplayWarning(message);
    else
        SendMessage(hStatusBar, SB_SETTEXT, 0x0000, (LPARAM)message);
    return;
}

/*
 * specific tracing of any OpenGL errors
 * based on the illustrations in the OpenAL specifications
 */
NOINLINE void DisplayGLError(char * text, GLenum error)
{
    HWND hWnd;

    hWnd = GET_GFX_INFO(hWnd);
    if (error == GL_NO_ERROR)
    {
        MessageBoxA(hWnd, text, GL_errors[GL_NO_ERROR], MB_ICONINFORMATION);
        return;
    }
    error -= GL_INVALID_ENUM - 1;

    if (error < 0 || error > 07)
        error = 7;
    hWnd = NULL; /* kinda annoying when you can't close parent windows */
    MessageBoxA(hWnd, text, GL_errors[error], MB_ICONERROR);
    return;
}

NOINLINE void zerobuf(void * memory, size_t length)
{
    size_t count;
    unsigned char* bytes;

    bytes = (unsigned char *)(memory);
    for (count = 0; count < length; count++)
        bytes[count] = 0x00;
    return;
}

NOINLINE int file_in(char* path, void* data, unsigned long length)
{
    HANDLE file;
    BOOL success;
    DWORD read;

    file = CreateFile(
        path,
        GENERIC_READ,
        FILE_SHARE_READ, 
        NULL,
        OPEN_EXISTING,
        FILE_ATTRIBUTE_NORMAL,
        NULL);
    success  = (file != INVALID_HANDLE_VALUE);
    success &= ReadFile(file, data, length, &read, NULL);
    success &= CloseHandle(file);
    return (success);
}

NOINLINE void file_out(char* path, void* data, unsigned long length)
{
    HANDLE file;
    BOOL success;
    DWORD written;

    file = CreateFile(
        path,
        GENERIC_WRITE,
        FILE_SHARE_READ,
        NULL,
        CREATE_ALWAYS,
        FILE_FLAG_WRITE_THROUGH,
        NULL);
    success  = (file != INVALID_HANDLE_VALUE);
    success &= WriteFile(file, data, length, &written, NULL);
    success &= CloseHandle(file);
    if (success == FALSE)
        DisplayError("Failed to write file.");
    return;
}

NOINLINE void screen_resize(GLsizei width, GLsizei height)
{
    RECT bigrect, smallrect, statusrect;
    POINT p;
    int rightdiff;
    int bottomdiff;

    sync = TRUE;
    GetWindowRect(gfx.hWnd, &bigrect);
    GetClientRect(gfx.hWnd, &smallrect);
    rightdiff = width - smallrect.right;
    bottomdiff = height - smallrect.bottom;

    statusrect.left = 0;
    statusrect.bottom = 0;
    if (gfx.hStatusBar != NULL)
    {
        GetClientRect(gfx.hStatusBar, &statusrect);
        bottomdiff += statusrect.bottom;
    }

    MoveWindow(
        gfx.hWnd,
        bigrect.left,
        bigrect.top,
        bigrect.right - bigrect.left + rightdiff,
        bigrect.bottom - bigrect.top + bottomdiff,
        TRUE);
    src.top = src.left = 0;
    src.bottom = 0;
    src.right = PRESCALE_WIDTH;
    p.x = p.y = 0;
    GetClientRect(gfx.hWnd, &dst);
    ClientToScreen(gfx.hWnd, &p);
    OffsetRect(&dst, p.x, p.y);
    dst.bottom -= statusrect.bottom;
    return;
}

NOINLINE void get_screen_size(int* dimensions)
{
    static MONITORINFO info;
    HMONITOR monitor;
    int width, height;
    const HWND hWnd = GET_GFX_INFO(hWnd);

    info.cbSize = sizeof(MONITORINFO);

    width = GetDeviceCaps(device_context, HORZRES);
    height = GetDeviceCaps(device_context, VERTRES);
    *(dimensions + 0) = GetSystemMetrics(SM_CXSCREEN);
    *(dimensions + 1) = GetSystemMetrics(SM_CYSCREEN);
    if (dimensions[0] != width || dimensions[1] != height)
        MessageBox(
            hWnd,
            "Conflicting monitor resolution information.\n"\
            "Perhaps you have more than one output device?",
            NULL,
            MB_ICONERROR | MB_OK);

/*
 * optional, extra analysis
 */
    monitor = MonitorFromWindow(hWnd, MONITOR_DEFAULTTONEAREST);
    GetMonitorInfo(monitor, &info);
    width  = (int)info.rcMonitor.right - (int)info.rcMonitor.left;
    height = (int)info.rcMonitor.bottom - (int)info.rcMonitor.top;
    if (dimensions[0] != width || dimensions[1] != height)
        MessageBox(
            hWnd,
            "Conflicting monitor resolution information.\n"\
            "Perhaps you have more than one output device?",
            "Warning",
            MB_ICONWARNING | MB_OK);
    return;
}

NOINLINE int swap_buffers(void)
{
    GLenum error;
    BOOL pass;

    pass = TRUE;
    if (double_buffering != GL_FALSE)
    {
        Sleep(5);
#ifdef _DEBUG
        glFinish();
#endif
        pass &= SwapBuffers(device_context);
        Sleep(5);
    }
    else
    {
#ifdef _DEBUG
        glFinish();
#else
        glFlush();
#endif
    }
    error = glGetError();
    if (error != GL_NO_ERROR)
    {
        pass = FALSE;
        DisplayGLError("Failed to sync commands.", error);
    }
    return (pass);
}

#ifdef _DEBUG
void trace_VI_registers(void)
{
    FILE * stream;

    stream = fopen("rcp_vi.txt", "w");
    if (stream == NULL)
    {
        DisplayError("Could not trace VI register segments.");
        return;
    }

    fprintf(stream, "VI_STATUS_REG :  %08X\n", *gfx.VI_STATUS_REG);
    fprintf(stream, "VI_ORIGIN_REG :  %08X\n", *gfx.VI_ORIGIN_REG);
    fprintf(stream, "VI_WIDTH_REG  :  %08X\n", *gfx.VI_WIDTH_REG);
    fprintf(stream, "VI_INTR_REG   :  %08X\n", *gfx.VI_INTR_REG);
    fprintf(stream, "VI_CURRENT_REG:  %08X\n", *gfx.VI_V_CURRENT_LINE_REG);
    fprintf(stream, "VI_BURST_REG  :  %08X\n", *gfx.VI_TIMING_REG);
    fprintf(stream, "VI_V_SYNC_REG :  %08X\n", *gfx.VI_V_SYNC_REG);
    fprintf(stream, "VI_H_SYNC_REG :  %08X\n", *gfx.VI_H_SYNC_REG);
    fprintf(stream, "VI_LEAP_REG   :  %08X\n", *gfx.VI_LEAP_REG);
    fprintf(stream, "VI_H_VIDEO_REG:  %08X\n", *gfx.VI_H_START_REG);
    fprintf(stream, "VI_V_VIDEO_REG:  %08X\n", *gfx.VI_V_START_REG);
    fprintf(stream, "VI_V_BURST_REG:  %08X\n", *gfx.VI_V_BURST_REG);
    fprintf(stream, "VI_X_SCALE_REG:  %08X\n", *gfx.VI_X_SCALE_REG);
    fprintf(stream, "VI_Y_SCALE_REG:  %08X\n", *gfx.VI_Y_SCALE_REG);
    fclose(stream);
    return;
}
#endif
