/***********************************************************************************
  Snes9x - Portable Super Nintendo Entertainment System (TM) emulator.

  (c) Copyright 1996 - 2002  Gary Henderson (gary.henderson@ntlworld.com),
                             Jerremy Koot (jkoot@snes9x.com)

  (c) Copyright 2002 - 2004  Matthew Kendora

  (c) Copyright 2002 - 2005  Peter Bortas (peter@bortas.org)

  (c) Copyright 2004 - 2005  Joel Yliluoma (http://iki.fi/bisqwit/)

  (c) Copyright 2001 - 2006  John Weidman (jweidman@slip.net)

  (c) Copyright 2002 - 2006  funkyass (funkyass@spam.shaw.ca),
                             Kris Bleakley (codeviolation@hotmail.com)

  (c) Copyright 2002 - 2010  Brad Jorsch (anomie@users.sourceforge.net),
                             Nach (n-a-c-h@users.sourceforge.net),

  (c) Copyright 2002 - 2011  zones (kasumitokoduck@yahoo.com)

  (c) Copyright 2006 - 2007  nitsuja

  (c) Copyright 2009 - 2011  BearOso,
                             OV2


  BS-X C emulator code
  (c) Copyright 2005 - 2006  Dreamer Nom,
                             zones

  C4 x86 assembler and some C emulation code
  (c) Copyright 2000 - 2003  _Demo_ (_demo_@zsnes.com),
                             Nach,
                             zsKnight (zsknight@zsnes.com)

  C4 C++ code
  (c) Copyright 2003 - 2006  Brad Jorsch,
                             Nach

  DSP-1 emulator code
  (c) Copyright 1998 - 2006  _Demo_,
                             Andreas Naive (andreasnaive@gmail.com),
                             Gary Henderson,
                             Ivar (ivar@snes9x.com),
                             John Weidman,
                             Kris Bleakley,
                             Matthew Kendora,
                             Nach,
                             neviksti (neviksti@hotmail.com)

  DSP-2 emulator code
  (c) Copyright 2003         John Weidman,
                             Kris Bleakley,
                             Lord Nightmare (lord_nightmare@users.sourceforge.net),
                             Matthew Kendora,
                             neviksti

  DSP-3 emulator code
  (c) Copyright 2003 - 2006  John Weidman,
                             Kris Bleakley,
                             Lancer,
                             z80 gaiden

  DSP-4 emulator code
  (c) Copyright 2004 - 2006  Dreamer Nom,
                             John Weidman,
                             Kris Bleakley,
                             Nach,
                             z80 gaiden

  OBC1 emulator code
  (c) Copyright 2001 - 2004  zsKnight,
                             pagefault (pagefault@zsnes.com),
                             Kris Bleakley
                             Ported from x86 assembler to C by sanmaiwashi

  SPC7110 and RTC C++ emulator code used in 1.39-1.51
  (c) Copyright 2002         Matthew Kendora with research by
                             zsKnight,
                             John Weidman,
                             Dark Force

  SPC7110 and RTC C++ emulator code used in 1.52+
  (c) Copyright 2009         byuu,
                             neviksti

  S-DD1 C emulator code
  (c) Copyright 2003         Brad Jorsch with research by
                             Andreas Naive,
                             John Weidman

  S-RTC C emulator code
  (c) Copyright 2001 - 2006  byuu,
                             John Weidman

  ST010 C++ emulator code
  (c) Copyright 2003         Feather,
                             John Weidman,
                             Kris Bleakley,
                             Matthew Kendora

  Super FX x86 assembler emulator code
  (c) Copyright 1998 - 2003  _Demo_,
                             pagefault,
                             zsKnight

  Super FX C emulator code
  (c) Copyright 1997 - 1999  Ivar,
                             Gary Henderson,
                             John Weidman

  Sound emulator code used in 1.5-1.51
  (c) Copyright 1998 - 2003  Brad Martin
  (c) Copyright 1998 - 2006  Charles Bilyue'

  Sound emulator code used in 1.52+
  (c) Copyright 2004 - 2007  Shay Green (gblargg@gmail.com)

  SH assembler code partly based on x86 assembler code
  (c) Copyright 2002 - 2004  Marcus Comstedt (marcus@mc.pp.se)

  2xSaI filter
  (c) Copyright 1999 - 2001  Derek Liauw Kie Fa

  HQ2x, HQ3x, HQ4x filters
  (c) Copyright 2003         Maxim Stepin (maxim@hiend3d.com)

  NTSC filter
  (c) Copyright 2006 - 2007  Shay Green

  GTK+ GUI code
  (c) Copyright 2004 - 2011  BearOso

  Win32 GUI code
  (c) Copyright 2003 - 2006  blip,
                             funkyass,
                             Matthew Kendora,
                             Nach,
                             nitsuja
  (c) Copyright 2009 - 2011  OV2

  Mac OS GUI code
  (c) Copyright 1998 - 2001  John Stiles
  (c) Copyright 2001 - 2011  zones


  Specific ports contains the works of other authors. See headers in
  individual files.


  Snes9x homepage: http://www.snes9x.com/

  Permission to use, copy, modify and/or distribute Snes9x in both binary
  and source form, for non-commercial purposes, is hereby granted without
  fee, providing that this license information and copyright notice appear
  with all copies and any derived work.

  This software is provided 'as-is', without any express or implied
  warranty. In no event shall the authors be held liable for any damages
  arising from the use of this software or it's derivatives.

  Snes9x is freeware for PERSONAL USE only. Commercial users should
  seek permission of the copyright holders first. Commercial use includes,
  but is not limited to, charging money for Snes9x or software derived from
  Snes9x, including Snes9x or derivatives in commercial game bundles, and/or
  using Snes9x as a promotion for your commercial product.

  The copyright holders request that bug fixes and improvements to the code
  should be forwarded to them so everyone can benefit from the modifications
  in future versions.

  Super NES and Super Nintendo Entertainment System are trademarks of
  Nintendo Co., Limited and its subsidiary companies.
 ***********************************************************************************/

#pragma comment( lib, "d3d9" )
#pragma comment( lib, "d3dx9" )
#pragma comment( lib, "DxErr" )

#include "cdirect3d.h"
#include "win32_display.h"
#include "../snes9x.h"
#include "../gfx.h"
#include "../display.h"
#include "wsnes9x.h"
#include <Dxerr.h>
#include <commctrl.h>

#include "../filter/hq2x.h"
#include "../filter/2xsai.h"

#ifndef max
#define max(a, b) (((a) > (b)) ? (a) : (b))
#endif
#ifndef min
#define min(a, b) (((a) < (b)) ? (a) : (b))
#endif


bool bOculusInit = false;

Ptr<DeviceManager>	pManager;
Ptr<HMDDevice>		pHMD;
Ptr<SensorDevice>	pSensor;

HMDInfo			hmd;
SensorFusion	SFusion;
MagCalibration	MagCal;
StereoConfig	stereo;

Vector3f EyePos(0,0,0);
float EyeYaw,EyePitch,EyeRoll;
float renderScale;
Matrix4f View;

LPDIRECT3DPIXELSHADER9       pixelShader;
LPD3DXCONSTANTTABLE          constantTable;
LPD3DXBUFFER                 code;

LPDIRECT3DTEXTURE9 pRenderTexture	= NULL;
LPDIRECT3DSURFACE9 pRenderSurface	= NULL;
LPDIRECT3DSURFACE9 pRenderDepth		= NULL;

/* CDirect3D::CDirect3D()
sets default values for the variables
*/
CDirect3D::CDirect3D()
{
	init_done = false;
	pD3D = NULL;
	pDevice = NULL;
	drawSurface = NULL;
	vertexBuffer = NULL;
	afterRenderWidth = 0;
	afterRenderHeight = 0;
	quadTextureSize = 0;
	fullscreen = false;
	filterScale = 1;
	for(int i = 0; i < MAX_SHADER_TEXTURES; i++) {
		rubyLUT[i] = NULL;
	}
	effect=NULL;
	shader_type = D3D_SHADER_NONE;
	shaderTimer = 1.0f;
	shaderTimeStart = 0;
	shaderTimeElapsed = 0;
	cgContext = NULL;
	cgVertexProgram = cgFragmentProgram = NULL;
	cgAvailable = false;
}

/* CDirect3D::~CDirect3D()
releases allocated objects
*/
CDirect3D::~CDirect3D()
{
	DeInitialize();
	Oculus_DeInitialize();
}


/*  CDirect3D::Initialize
Initializes Direct3D (always in window mode)
IN:
hWnd	-	the HWND of the window in which we render/the focus window for fullscreen
-----
returns true if successful, false otherwise
*/
bool CDirect3D::Initialize(HWND hWnd)
{
	if(init_done)
		return true;

	pD3D = Direct3DCreate9(D3D_SDK_VERSION);
	if(pD3D == NULL) {
		DXTRACE_ERR_MSGBOX(TEXT("Error creating initial D3D9 object"), 0);
		return false;
	}

	ZeroMemory(&dPresentParams, sizeof(dPresentParams));
	dPresentParams.hDeviceWindow = hWnd;
    dPresentParams.Windowed = true;
	dPresentParams.BackBufferCount = GUI.DoubleBuffered?2:1;
    dPresentParams.SwapEffect = D3DSWAPEFFECT_DISCARD;
	dPresentParams.BackBufferFormat = D3DFMT_UNKNOWN;

	HRESULT hr = pD3D->CreateDevice(D3DADAPTER_DEFAULT,
                      D3DDEVTYPE_HAL,
                      hWnd,
                      D3DCREATE_MIXED_VERTEXPROCESSING,
					  &dPresentParams,
                      &pDevice);
	if(FAILED(hr)) {
		DXTRACE_ERR_MSGBOX(TEXT("Error creating D3D9 device"), hr);
		return false;
	}

	hr = pDevice->CreateVertexBuffer(sizeof(triangleStripVertices),D3DUSAGE_WRITEONLY,FVF_COORDS_TEX,D3DPOOL_MANAGED,&vertexBuffer,NULL);
	if(FAILED(hr)) {
		DXTRACE_ERR_MSGBOX(TEXT("Error creating vertex buffer"), hr);
		return false;
	}

	cgAvailable = loadCgFunctions();

	if(cgAvailable) {
		cgContext = cgCreateContext();
		hr = cgD3D9SetDevice(pDevice);
		if(FAILED(hr)) {
			DXTRACE_ERR_MSGBOX(TEXT("Error setting cg device"), hr);
		}
	}

	pDevice->SetRenderState(D3DRS_LIGHTING, FALSE);

	pDevice->Clear(0, NULL, D3DCLEAR_TARGET, D3DCOLOR_XRGB(0, 0, 0), 1.0f, 0);

	if (!Oculus_Initialize())
	{
		Settings.Oculus = FALSE;
		//Maybe Setup for a Fake Oculus?
	}

	init_done = true;

	SetViewport();

	ApplyDisplayChanges();

	return true;

}

void CDirect3D::DeInitialize()
{
	DestroyDrawSurface();
	SetShader(NULL);

	if(cgContext) {
		cgDestroyContext(cgContext);
		cgContext = NULL;
	}

	if(cgAvailable)
		cgD3D9SetDevice(NULL);

	if(vertexBuffer) {
		vertexBuffer->Release();
		vertexBuffer = NULL;
	}

	if( pDevice ) {
		pDevice->Release();
		pDevice = NULL;
	}

	if( pD3D ) {
		pD3D->Release();
		pD3D = NULL;
	}

	init_done = false;
	afterRenderWidth = 0;
	afterRenderHeight = 0;
	quadTextureSize = 0;
	fullscreen = false;
	filterScale = 0;
	if(cgAvailable)
		unloadCgLibrary();
	cgAvailable = false;
}

bool CDirect3D::SetShader(const TCHAR *file)
{
	SetShaderCG(NULL);
	SetShaderHLSL(NULL);
	shader_type = D3D_SHADER_NONE;
	if(file!=NULL && lstrlen(file)>3 && _tcsncicmp(&file[lstrlen(file)-3],TEXT(".cg"),3)==0) {
		return SetShaderCG(file);
	} else {
		return SetShaderHLSL(file);
	}
}

void CDirect3D::checkForCgError(const char *situation)
{
	char buffer[4096];
	CGerror error = cgGetError();
	const char *string = cgGetErrorString(error);

	if (error != CG_NO_ERROR) {
		sprintf(buffer,
			  "Situation: %s\n"
			  "Error: %s\n\n"
			  "Cg compiler output...\n", situation, string);
		MessageBoxA(0, buffer,
				  "Cg error", MB_OK|MB_ICONEXCLAMATION);
		if (error == CG_COMPILER_ERROR) {
			MessageBoxA(0, cgGetLastListing(cgContext),
					  "Cg compilation error", MB_OK|MB_ICONEXCLAMATION);
		}
	}
}

bool CDirect3D::SetShaderCG(const TCHAR *file)
{
	TCHAR errorMsg[MAX_PATH + 50];
	HRESULT hr;

	if(cgFragmentProgram) {
		cgDestroyProgram(cgFragmentProgram);
		cgFragmentProgram = NULL;
	}
	if(cgVertexProgram) {
		cgDestroyProgram(cgVertexProgram);
		cgVertexProgram = NULL;
	}

	if (file == NULL || *file==TEXT('\0'))
		return true;

	if(!cgAvailable) {
		MessageBox(NULL, TEXT("The CG runtime is unavailable, CG shaders will not run.\nConsult the snes9x readme for information on how to obtain the runtime."), TEXT("CG Error"),
			MB_OK|MB_ICONEXCLAMATION);
        return false;
    }

	CGprofile vertexProfile = cgD3D9GetLatestVertexProfile();
	CGprofile pixelProfile = cgD3D9GetLatestPixelProfile();

	const char** vertexOptions = cgD3D9GetOptimalOptions(vertexProfile);
	const char** pixelOptions = cgD3D9GetOptimalOptions(pixelProfile);

	char *fileContents = ReadShaderFileContents(file);
	if(!fileContents)
		return false;

	cgVertexProgram = cgCreateProgram( cgContext, CG_SOURCE, fileContents,
						vertexProfile, "main_vertex", vertexOptions);

	checkForCgError("Compiling vertex program");

	cgFragmentProgram = cgCreateProgram( cgContext, CG_SOURCE, fileContents,
						pixelProfile, "main_fragment", pixelOptions);

	checkForCgError("Compiling fragment program");

	delete [] fileContents;

	if(!cgVertexProgram || !cgFragmentProgram) {
		return false;
	}

	if(cgVertexProgram) {
		hr = cgD3D9LoadProgram(cgVertexProgram,true,0);
		hr = cgD3D9BindProgram(cgVertexProgram);

		D3DXMATRIX matWorld;
		D3DXMATRIX matView;
		D3DXMATRIX matProj;
		D3DXMATRIX mvp;

		pDevice->GetTransform(D3DTS_WORLD,&matWorld);
		pDevice->GetTransform(D3DTS_VIEW,&matView);
		pDevice->GetTransform(D3DTS_PROJECTION,&matProj);

		mvp = matWorld * matView * matProj;
		D3DXMatrixTranspose(&mvp,&mvp);

		CGparameter cgpModelViewProj = cgGetNamedParameter(cgVertexProgram, "modelViewProj");

		if(cgpModelViewProj)
			cgD3D9SetUniformMatrix(cgpModelViewProj,&mvp);
	}
	if(cgFragmentProgram) {
		hr = cgD3D9LoadProgram(cgFragmentProgram,false,0);
		hr = cgD3D9BindProgram(cgFragmentProgram);
	}

	shader_type = D3D_SHADER_CG;

	return true;
}

bool CDirect3D::SetShaderHLSL(const TCHAR *file)
{
	//MUDLORD: the guts
	//Compiles a shader from files on disc
	//Sets LUT textures to texture files in PNG format.

	TCHAR folder[MAX_PATH];
	TCHAR rubyLUTfileName[MAX_PATH];
	TCHAR *slash;
	char *shaderText = NULL;

	TCHAR errorMsg[MAX_PATH + 50];

	IXMLDOMDocument * pXMLDoc = NULL;
	IXMLDOMElement * pXDE = NULL;
	IXMLDOMNode * pXDN = NULL;
	BSTR queryString, nodeContent;

	HRESULT hr;

	shaderTimer = 1.0f;
	shaderTimeStart = 0;
	shaderTimeElapsed = 0;

	if(effect) {
		effect->Release();
		effect = NULL;
	}
	for(int i = 0; i < MAX_SHADER_TEXTURES; i++) {
		if (rubyLUT[i] != NULL) {
			rubyLUT[i]->Release();
			rubyLUT[i] = NULL;
		}
	}
	if (file == NULL || *file==TEXT('\0'))
		return true;

	hr = CoCreateInstance(CLSID_DOMDocument,NULL,CLSCTX_INPROC_SERVER,IID_PPV_ARGS(&pXMLDoc));

	if(FAILED(hr)) {
		MessageBox(NULL, TEXT("Error creating XML Parser"), TEXT("Shader Loading Error"),
			MB_OK|MB_ICONEXCLAMATION);
		return false;
	}

	VARIANT fileName;
	VARIANT_BOOL ret;
	fileName.vt = VT_BSTR;
#ifdef UNICODE
	fileName.bstrVal = SysAllocString(file);
#else
	wchar_t tempfilename[MAX_PATH];
	MultiByteToWideChar(CP_UTF8,0,file,-1,tempfilename,MAX_PATH);
	fileName.bstrVal = SysAllocString(tempfilename);
#endif
	hr = pXMLDoc->load(fileName,&ret);
	SysFreeString(fileName.bstrVal);

	if(FAILED(hr) || hr==S_FALSE) {
		_stprintf(errorMsg,TEXT("Error loading HLSL shader file:\n%s"),file);
		MessageBox(NULL, errorMsg, TEXT("Shader Loading Error"), MB_OK|MB_ICONEXCLAMATION);
		pXMLDoc->Release();
		return false;
	}

	VARIANT attributeValue;
	BSTR attributeName;

	hr = pXMLDoc->get_documentElement(&pXDE);
	if(FAILED(hr) || hr==S_FALSE) {
		_stprintf(errorMsg,TEXT("Error loading root element from file:\n%s"),file);
		MessageBox(NULL, errorMsg, TEXT("Shader Loading Error"), MB_OK|MB_ICONEXCLAMATION);
		pXMLDoc->Release();
		return false;
	}

	attributeName=SysAllocString(L"language");
	pXDE->getAttribute(attributeName,&attributeValue);
	SysFreeString(attributeName);
	pXDE->Release();

	if(attributeValue.vt!=VT_BSTR || lstrcmpiW(attributeValue.bstrVal,L"hlsl")) {
		_stprintf(errorMsg,TEXT("Shader language is <%s>, expected <HLSL> in file:\n%s"),attributeValue.bstrVal,file);
		MessageBox(NULL, errorMsg, TEXT("Shader Loading Error"), MB_OK|MB_ICONEXCLAMATION);
		if(attributeValue.vt==VT_BSTR) SysFreeString(attributeValue.bstrVal);
		pXMLDoc->Release();
		return false;
	}
	if(attributeValue.vt==VT_BSTR) SysFreeString(attributeValue.bstrVal);

	queryString=SysAllocString(L"/shader/source");
	hr = pXMLDoc->selectSingleNode(queryString,&pXDN);
	SysFreeString(queryString);

	if(hr == S_OK) {
		hr = pXDN->get_text(&nodeContent);
		if(hr == S_OK) {
			int requiredChars = WideCharToMultiByte(CP_ACP,0,nodeContent,-1,shaderText,0,NULL,NULL);
			shaderText = new char[requiredChars];
			WideCharToMultiByte(CP_UTF8,0,nodeContent,-1,shaderText,requiredChars,NULL,NULL);
		}
		SysFreeString(nodeContent);
		pXDN->Release();
		pXDN = NULL;
	}

	pXMLDoc->Release();

	if(!shaderText) {
		_stprintf(errorMsg,TEXT("No HLSL shader program in file:\n%s"),file);
		MessageBox(NULL, errorMsg, TEXT("Shader Loading Error"),
			MB_OK|MB_ICONEXCLAMATION);
		return false;
	}

	LPD3DXBUFFER pBufferErrors = NULL;
	hr = D3DXCreateEffect( pDevice,shaderText,strlen(shaderText),NULL, NULL,
		D3DXSHADER_ENABLE_BACKWARDS_COMPATIBILITY, NULL, &effect, 
		&pBufferErrors );
	delete[] shaderText;
	if( FAILED(hr) ) {
		_stprintf(errorMsg,TEXT("Error parsing HLSL shader file:\n%s"),file);
		MessageBox(NULL, errorMsg, TEXT("Shader Loading Error"), MB_OK|MB_ICONEXCLAMATION);
		if(pBufferErrors) {
			LPVOID pCompilErrors = pBufferErrors->GetBufferPointer();
			MessageBox(NULL, (const TCHAR*)pCompilErrors, TEXT("FX Compile Error"),
				MB_OK|MB_ICONEXCLAMATION);
		}
		return false;
	}

	lstrcpy(folder,file);
	slash = _tcsrchr(folder,TEXT('\\'));
	if(slash)
		*(slash+1)=TEXT('\0');
	else
		*folder=TEXT('\0');
	SetCurrentDirectory(S9xGetDirectoryT(DEFAULT_DIR));

	for(int i = 0; i < MAX_SHADER_TEXTURES; i++) {		
		_stprintf(rubyLUTfileName, TEXT("%srubyLUT%d.png"), folder, i);
		hr = D3DXCreateTextureFromFile(pDevice,rubyLUTfileName,&rubyLUT[i]);
		if FAILED(hr){
			rubyLUT[i] = NULL;
		}
	}

	D3DXHANDLE hTech;
	effect->FindNextValidTechnique(NULL,&hTech);
	effect->SetTechnique( hTech );
	shader_type = D3D_SHADER_HLSL;
	return true;
}

void CDirect3D::SetShaderVars()
{
	if(shader_type == D3D_SHADER_HLSL) {
		D3DXVECTOR4 rubyTextureSize;
		D3DXVECTOR4 rubyInputSize;
		D3DXVECTOR4 rubyOutputSize;
		D3DXHANDLE  rubyTimer;

		int shaderTime = GetTickCount();
		shaderTimeElapsed += shaderTime - shaderTimeStart;
		shaderTimeStart = shaderTime;
		if(shaderTimeElapsed > 100) {
			shaderTimeElapsed = 0;
			shaderTimer += 0.01f;
		}
		rubyTextureSize.x = rubyTextureSize.y = (float)quadTextureSize;
		rubyTextureSize.z = rubyTextureSize.w = 1.0 / quadTextureSize;
		rubyInputSize.x = (float)afterRenderWidth;
		rubyInputSize.y = (float)afterRenderHeight;
		rubyInputSize.z = 1.0 / rubyInputSize.y;
		rubyInputSize.w = 1.0 / rubyInputSize.z;
		rubyOutputSize.x  = GUI.Stretch?(float)dPresentParams.BackBufferWidth:(float)afterRenderWidth;
		rubyOutputSize.y  = GUI.Stretch?(float)dPresentParams.BackBufferHeight:(float)afterRenderHeight;
		rubyOutputSize.z = 1.0 / rubyOutputSize.y;
		rubyOutputSize.w = 1.0 / rubyOutputSize.x;
		rubyTimer = effect->GetParameterByName(0, "rubyTimer");

		effect->SetFloat(rubyTimer, shaderTimer);
		effect->SetVector("rubyTextureSize", &rubyTextureSize);
		effect->SetVector("rubyInputSize", &rubyInputSize);
		effect->SetVector("rubyOutputSize", &rubyOutputSize);

		effect->SetTexture("rubyTexture", drawSurface);
		for(int i = 0; i < MAX_SHADER_TEXTURES; i++) {
			char rubyLUTName[256];
			sprintf(rubyLUTName, "rubyLUT%d", i);
			if (rubyLUT[i] != NULL) {
				effect->SetTexture( rubyLUTName, rubyLUT[i] );
			}
		}
	} else if(shader_type == D3D_SHADER_CG) {

		D3DXVECTOR2 videoSize;
		D3DXVECTOR2 textureSize;
		D3DXVECTOR2 outputSize;
		videoSize.x = (float)afterRenderWidth;
		videoSize.y = (float)afterRenderHeight;
		textureSize.x = textureSize.y = (float)quadTextureSize;
		outputSize.x = GUI.Stretch?(float)dPresentParams.BackBufferWidth:(float)afterRenderWidth;
		outputSize.y = GUI.Stretch?(float)dPresentParams.BackBufferHeight:(float)afterRenderHeight;

#define setProgramUniform(program,varname,floats)\
{\
	CGparameter cgp = cgGetNamedParameter(program, varname);\
	if(cgp)\
		cgD3D9SetUniform(cgp,floats);\
}\

		setProgramUniform(cgFragmentProgram,"IN.video_size",&videoSize);
		setProgramUniform(cgFragmentProgram,"IN.texture_size",&textureSize);
		setProgramUniform(cgFragmentProgram,"IN.output_size",&outputSize);

		setProgramUniform(cgVertexProgram,"IN.video_size",&videoSize);
		setProgramUniform(cgVertexProgram,"IN.texture_size",&textureSize);
		setProgramUniform(cgVertexProgram,"IN.output_size",&outputSize);
	}
}

/*  CDirect3D::Render
does the actual rendering, changes the draw surface if necessary and recalculates
the vertex information if filter output size changes
IN:
Src		-	the input surface
*/
void CDirect3D::Render(SSurface Src)
{
	SSurface Dst;
	RECT dstRect;
	unsigned int newFilterScale;
	D3DLOCKED_RECT lr;
	D3DLOCKED_RECT lrConv;
	HRESULT hr;

	if(!init_done) return;

	//create a new draw surface if the filter scale changes
	//at least factor 2 so we can display unscaled hi-res images
	newFilterScale = max(2,max(GetFilterScale(GUI.ScaleHiRes),GetFilterScale(GUI.Scale)));
	if(newFilterScale!=filterScale) {
		ChangeDrawSurfaceSize(newFilterScale);
	}

	if(FAILED(hr = pDevice->TestCooperativeLevel())) {
		switch(hr) {
			case D3DERR_DEVICELOST:		//do no rendering until device is restored
				return;
			case D3DERR_DEVICENOTRESET: //we can reset now
				ResetDevice();
				return;
			default:
				DXTRACE_ERR_MSGBOX( TEXT("Internal driver error"), hr);
				return;
		}
	}

	//BlankTexture(drawSurface);
	if(FAILED(hr = drawSurface->LockRect(0, &lr, NULL, 0))) {
		DXTRACE_ERR_MSGBOX( TEXT("Unable to lock texture"), hr);
		return;
	} else {
		Dst.Surface = (unsigned char *)lr.pBits;
		Dst.Height = quadTextureSize;
		Dst.Width = quadTextureSize;
		Dst.Pitch = lr.Pitch;

		RenderMethod (Src, Dst, &dstRect);
		if(!Settings.AutoDisplayMessages) {
			WinSetCustomDisplaySurface((void *)Dst.Surface, Dst.Pitch/2, dstRect.right-dstRect.left, dstRect.bottom-dstRect.top, GetFilterScale(CurrentScale));
			S9xDisplayMessages ((uint16*)Dst.Surface, Dst.Pitch/2, dstRect.right-dstRect.left, dstRect.bottom-dstRect.top, GetFilterScale(CurrentScale));
		}

		drawSurface->UnlockRect(0);
	}

	//if the output size of the render method changes we need to update the viewport
	if(afterRenderHeight != dstRect.bottom || afterRenderWidth != dstRect.right) {
		afterRenderHeight = dstRect.bottom;
		afterRenderWidth = dstRect.right;
		SetViewport();
	}

	if(!GUI.Stretch||GUI.AspectRatio)
		pDevice->Clear(0, NULL, D3DCLEAR_TARGET, D3DCOLOR_XRGB(0, 0, 0), 1.0f, 0);

	pDevice->BeginScene();

	if (Settings.Oculus && bOculusInit)
	{
		Oculus_Render();
	}
	else
	{
		pDevice->SetTexture(0, drawSurface);
		pDevice->SetFVF(FVF_COORDS_TEX);
		pDevice->SetStreamSource(0,vertexBuffer,0,sizeof(VERTEX));

		SetShaderVars();

		if(shader_type == D3D_SHADER_CG) {
			cgD3D9BindProgram(cgFragmentProgram);
			cgD3D9BindProgram(cgVertexProgram);
		}

		if (shader_type == D3D_SHADER_HLSL) {
			UINT passes;

			hr = effect->Begin(&passes, 0);
			for(UINT pass = 0; pass < passes; pass++ ) {
				effect->BeginPass(pass);
				pDevice->DrawPrimitive(D3DPT_TRIANGLESTRIP,0,2);
				effect->EndPass();
			}
			effect->End();
		} else {
			pDevice->DrawPrimitive(D3DPT_TRIANGLESTRIP,0,2);
		}
	}

    pDevice->EndScene();

	pDevice->Present(NULL, NULL, NULL, NULL);

    return;
}

/*  CDirect3D::CreateDrawSurface
calculates the necessary texture size (multiples of 2)
and creates a new texture
*/
void CDirect3D::CreateDrawSurface()
{
	unsigned int neededSize;
	HRESULT hr;

	//we need at least 512 pixels (SNES_WIDTH * 2) so we can start with that value
	quadTextureSize = 512;
	neededSize = SNES_WIDTH * filterScale;
	while(quadTextureSize < neededSize)
		quadTextureSize *=2;

	//D3DFMT_R5G6B5
	if(!drawSurface) {
		hr = pDevice->CreateTexture(
			quadTextureSize, quadTextureSize,
			1, // 1 level, no mipmaps
			0, // dynamic textures can be locked
			D3DFMT_R5G6B5,
			D3DPOOL_MANAGED,
			&drawSurface,
			NULL );

		if(FAILED(hr)) {
			DXTRACE_ERR_MSGBOX(TEXT("Error while creating texture"), hr);
			return;
		}
	}

	if (bOculusInit)
	{
		int texw = (int)ceil(renderScale * dPresentParams.BackBufferWidth), texh = (int)ceil(renderScale * dPresentParams.BackBufferHeight);

		pDevice->CreateTexture(texw,texh,1,D3DUSAGE_RENDERTARGET,D3DFMT_R5G6B5,D3DPOOL_DEFAULT,&pRenderTexture,NULL);
		pRenderTexture->GetSurfaceLevel(0,&pRenderSurface);
		pDevice->CreateDepthStencilSurface(texw,texh,D3DFMT_D16,D3DMULTISAMPLE_NONE,0,FALSE,&pRenderDepth,NULL);
	}

}

/*  CDirect3D::DestroyDrawSurface
releases the old textures (if allocated)
*/
void CDirect3D::DestroyDrawSurface()
{
	if(drawSurface) {
		drawSurface->Release();
		drawSurface = NULL;
	}
	if (pRenderTexture) pRenderTexture->Release();
	if (pRenderSurface) pRenderSurface->Release();
	if (pRenderDepth) pRenderDepth->Release();

	pRenderTexture = NULL;
	pRenderSurface = NULL;
	pRenderDepth = NULL;
}

/*  CDirect3D::BlankTexture
clears a texture (fills it with zeroes)
IN:
texture		-	the texture to be blanked
-----
returns true if successful, false otherwise
*/
bool CDirect3D::BlankTexture(LPDIRECT3DTEXTURE9 texture)
{
	D3DLOCKED_RECT lr;
	HRESULT hr;

	if(FAILED(hr = texture->LockRect(0, &lr, NULL, 0))) {
		DXTRACE_ERR_MSGBOX( TEXT("Unable to lock texture"), hr);
		return false;
	} else {
		memset(lr.pBits, 0, lr.Pitch * quadTextureSize);
		texture->UnlockRect(0);
		return true;
	}
}

/*  CDirect3D::ChangeDrawSurfaceSize
changes the draw surface size: deletes the old textures, creates a new texture
and calculate new vertices
IN:
scale	-	the scale that has to fit into the textures
-----
returns true if successful, false otherwise
*/
bool CDirect3D::ChangeDrawSurfaceSize(unsigned int scale)
{
	filterScale = scale;

	if(pDevice) {
		DestroyDrawSurface();
		CreateDrawSurface();
		SetupVertices();
		return true;
	}
	return false;
}

/*  CDirect3D::SetupVertices
calculates the vertex coordinates
(respecting the stretch and aspect ratio settings)
*/
void CDirect3D::SetupVertices()
{
	void *pLockedVertexBuffer;

	float tX = (float)afterRenderWidth / (float)quadTextureSize;
	float tY = (float)afterRenderHeight / (float)quadTextureSize;

	triangleStripVertices[0] = VERTEX(0.0f,0.0f,0.0f,0.0f,tY);
	triangleStripVertices[1] = VERTEX(0.0f,1.0f,0.0f,0.0f,0.0f);
	triangleStripVertices[2] = VERTEX(1.0f,0.0f,0.0f,tX,tY);
	triangleStripVertices[3] = VERTEX(1.0f,1.0f,0.0f,tX,0.0f);

	HRESULT hr = vertexBuffer->Lock(0,0,&pLockedVertexBuffer,NULL);
	memcpy(pLockedVertexBuffer,triangleStripVertices,sizeof(triangleStripVertices));
	vertexBuffer->Unlock();
}

void CDirect3D::SetViewport()
{
	D3DXMATRIX matIdentity;
	D3DXMATRIX matProjection;

	D3DXMatrixOrthoOffCenterLH(&matProjection,0.0f,1.0f,0.0f,1.0f,0.0f,1.0f);
	D3DXMatrixIdentity(&matIdentity);
	pDevice->SetTransform(D3DTS_WORLD,&matIdentity);
	pDevice->SetTransform(D3DTS_VIEW,&matIdentity);
	pDevice->SetTransform(D3DTS_PROJECTION,&matProjection);

	RECT drawRect = CalculateDisplayRect(afterRenderWidth,afterRenderHeight,dPresentParams.BackBufferWidth,dPresentParams.BackBufferHeight);
	D3DVIEWPORT9 viewport;
	viewport.X = drawRect.left;
	viewport.Y = drawRect.top;
	viewport.Height = drawRect.bottom - drawRect.top;
	viewport.Width = drawRect.right - drawRect.left;
	viewport.MinZ = 0.0f;
	viewport.MaxZ = 1.0f;
	HRESULT hr = pDevice->SetViewport(&viewport);

	SetupVertices();
}

/*  CDirect3D::ChangeRenderSize
determines if we need to reset the device (if the size changed)
called with (0,0) whenever we want new settings to take effect
IN:
newWidth,newHeight	-	the new window size
-----
returns true if successful, false otherwise
*/
bool CDirect3D::ChangeRenderSize(unsigned int newWidth, unsigned int newHeight)
{
	if(!init_done)
		return false;

	//if we already have the desired size no change is necessary
	//during fullscreen no changes are allowed
	if(dPresentParams.BackBufferWidth == newWidth && dPresentParams.BackBufferHeight == newHeight)
		return true;

	if(!ResetDevice())
		return false;

	SetViewport();
	return true;
}

/*  CDirect3D::ResetDevice
resets the device
called if surface was lost or the settings/display size require a device reset
-----
returns true if successful, false otherwise
*/
bool CDirect3D::ResetDevice()
{
	if(!init_done) return false;

	HRESULT hr;

	//release prior to reset
	DestroyDrawSurface();
	
	if(cgAvailable) {
		cgD3D9SetDevice(NULL);
	}

	if(effect)
		effect->OnLostDevice();

	//zero or unknown values result in the current window size/display settings
	dPresentParams.BackBufferWidth = 0;
	dPresentParams.BackBufferHeight = 0;
	dPresentParams.BackBufferCount = GUI.DoubleBuffered?2:1;
	dPresentParams.BackBufferFormat = D3DFMT_UNKNOWN;
	dPresentParams.FullScreen_RefreshRateInHz = 0;
	dPresentParams.Windowed = true;
	dPresentParams.PresentationInterval = GUI.Vsync?D3DPRESENT_INTERVAL_ONE:D3DPRESENT_INTERVAL_IMMEDIATE;

	if(fullscreen) {
		dPresentParams.BackBufferWidth = GUI.FullscreenMode.width;
		dPresentParams.BackBufferHeight = GUI.FullscreenMode.height;
		dPresentParams.BackBufferCount = GUI.DoubleBuffered?2:1;
		dPresentParams.Windowed = false;
		if(GUI.FullscreenMode.depth == 32)
			dPresentParams.BackBufferFormat = D3DFMT_X8R8G8B8;
		else
			dPresentParams.BackBufferFormat = D3DFMT_R5G6B5;
		dPresentParams.FullScreen_RefreshRateInHz = GUI.FullscreenMode.rate;
	}

	if(FAILED(hr = pDevice->Reset(&dPresentParams))) {
		DXTRACE_ERR(TEXT("Unable to reset device"), hr);
		return false;
	}

	if(effect)
		effect->OnResetDevice();

	if(cgAvailable) {
		cgD3D9SetDevice(pDevice);
	}

	if(GUI.BilinearFilter) {
		pDevice->SetSamplerState(0, D3DSAMP_MAGFILTER, D3DTEXF_LINEAR);
		pDevice->SetSamplerState(0, D3DSAMP_MINFILTER, D3DTEXF_LINEAR);
	} else {
		pDevice->SetSamplerState(0, D3DSAMP_MAGFILTER, D3DTEXF_POINT);
		pDevice->SetSamplerState(0, D3DSAMP_MINFILTER, D3DTEXF_POINT);
	}

	pDevice->SetRenderState(D3DRS_LIGHTING, FALSE);

	pDevice->Clear(0, NULL, D3DCLEAR_TARGET, D3DCOLOR_XRGB(0, 0, 0), 1.0f, 0);
	
	//recreate the surface
	CreateDrawSurface();

	return true;
}

/*  CDirect3D::SetSnes9xColorFormat
sets the color format to 16bit (since the texture is always 16bit)
no depth conversion is necessary (done by D3D)
*/
void CDirect3D::SetSnes9xColorFormat()
{
	GUI.ScreenDepth = 16;
	GUI.BlueShift = 0;
	GUI.GreenShift = 6;
	GUI.RedShift = 11;
	S9xSetRenderPixelFormat (RGB565);
	S9xBlit2xSaIFilterInit();
	S9xBlitHQ2xFilterInit();
	GUI.NeedDepthConvert = FALSE;
	GUI.DepthConverted = TRUE;
	return;
}

/*  CDirect3D::SetFullscreen
enables/disables fullscreen mode
IN:
fullscreen	-	determines if fullscreen is enabled/disabled
-----
returns true if successful, false otherwise
*/
bool CDirect3D::SetFullscreen(bool fullscreen)
{
	if(!init_done)
		return false;

	if(this->fullscreen==fullscreen)
		return true;

	this->fullscreen = fullscreen;
	if(!ResetDevice())
		return false;

	//present here to get a fullscreen blank even if no rendering is done
	pDevice->Present(NULL,NULL,NULL,NULL);
	SetupVertices();
	return true;
}

/*  CDirect3D::EnumModes
enumerates possible display modes (only 16 and 32 bit) and fills the vector
IN:
modeVector	-	pointer to the mode vector
*/
void CDirect3D::EnumModes(std::vector<dMode> *modeVector)
{
	D3DDISPLAYMODE d3dMode;
	int modeCount,index;
	dMode mode;

	if(!init_done)
		return;

	//enumerate 32bit modes
	modeCount = pD3D->GetAdapterModeCount(D3DADAPTER_DEFAULT,D3DFMT_X8R8G8B8);
	for(int i=0;i<modeCount;i++) {
		if(pD3D->EnumAdapterModes(D3DADAPTER_DEFAULT,D3DFMT_X8R8G8B8,i,&d3dMode)==D3D_OK) {
			mode.width = d3dMode.Width;
			mode.height = d3dMode.Height;
			mode.rate = d3dMode.RefreshRate;
			mode.depth = 32;

			modeVector->push_back(mode);
		}

	}

	//enumerate 16bit modes
	modeCount = pD3D->GetAdapterModeCount(D3DADAPTER_DEFAULT,D3DFMT_R5G6B5);
	for(int i=0;i<modeCount;i++) {
		if(pD3D->EnumAdapterModes(D3DADAPTER_DEFAULT,D3DFMT_R5G6B5,i,&d3dMode)==D3D_OK) {
			mode.width = d3dMode.Width;
			mode.height = d3dMode.Height;
			mode.rate = d3dMode.RefreshRate;
			mode.depth = 16;

			modeVector->push_back(mode);
		}

	}
}

/*  CDirect3D::ApplyDisplayChanges
calls changerendersize to apply new display settings
-----
returns true if successful, false otherwise
*/
bool CDirect3D::ApplyDisplayChanges(void)
{
	if(GUI.shaderEnabled && GUI.D3DshaderFileName)
		SetShader(GUI.D3DshaderFileName);
	else
		SetShader(NULL);

	return ChangeRenderSize(0,0);
}


// OCULUS STUFF....

bool CDirect3D::Oculus_Initialize()
{
	HRESULT hr;
	System::Init(Log::ConfigureDefaultLog(LogMask_All));

	pManager = *DeviceManager::Create();
	if (!pManager) return false;

	pHMD = *pManager->EnumerateDevices<HMDDevice>().CreateDevice();
	if (!pHMD) return false;

	pHMD->GetDeviceInfo(&hmd);
	pSensor = *pHMD->GetSensor();
	if (!pSensor) return false;

	if (pSensor)SFusion.AttachToSensor(pSensor);
	MagCal.BeginAutoCalibration(SFusion);

	stereo.SetFullViewport(Viewport(0,0, 1280, 800));
	stereo.SetStereoMode(Stereo_LeftRight_Multipass);
	stereo.SetHMDInfo(hmd);
	stereo.SetDistortionFitPointVP(-1.0f, 0.0f);
	renderScale = stereo.GetDistortionScale();

	LPD3DXBUFFER pErrors = NULL;
	hr = D3DXCompileShaderFromFile("pixel.psh",NULL,NULL,"ps_main","ps_2_0",0,&code,&pErrors,&constantTable);
	if (hr != S_OK) return false;

	if (pErrors)char *ErrorMessage = (char *)pErrors->GetBufferPointer();

	pDevice->CreatePixelShader((DWORD*)code->GetBufferPointer(),&pixelShader);
	code->Release();

	bOculusInit = true;

	return true;
}

void CDirect3D::Oculus_DeInitialize()
{
	//TODO: Release stuff...too tired to add now
}

struct CUSTOMVERTEX_TEXTURE
{
	FLOAT x, y, z, rhw;
	FLOAT tu,tv;
};
#define D3DFVF_CUSTOMVERTEX_TEXTURE (D3DFVF_XYZRHW|D3DFVF_TEX1)

CUSTOMVERTEX_TEXTURE Verts[6] =
{
	{0		,0		,0,1, 0,0},
	{1280	,0		,0,1, 1,0},
	{0		,800	,0,1, 0,1},
	{1280	,0		,0,1, 1,0},
	{1280	,800	,0,1, 1,1},
	{0		,800	,0,1, 0,1},
};

void CDirect3D::Oculus_SetupFullScreenVertices()
{
	float w = dPresentParams.BackBufferWidth;
	float h = dPresentParams.BackBufferHeight;

	Verts[1].x = w;
	Verts[2].y = h;
	Verts[3].x = w;
	Verts[4].x = w;
	Verts[4].y = h;
	Verts[5].y = h;
}

struct CUSTOMVERTEX_SCREEN
{
	FLOAT x, y, z;
	FLOAT tu,tv;
};
#define D3DFVF_CUSTOMVERTEX_SCREEN (D3DFVF_XYZ|D3DFVF_TEX1)

CUSTOMVERTEX_SCREEN Screen[6] =
{
	{0		,0		,0, 0,0},
	{1280	,0		,0, 1,0},
	{0		,800	,0, 0,1},
	{1280	,0		,0, 1,0},
	{1280	,800	,0, 1,1},
	{0		,800	,0, 0,1},
};

void CDirect3D::Oculus_GameVertices()
{
	float tX = (float)afterRenderWidth / (float)quadTextureSize;
	float tY = (float)afterRenderHeight / (float)quadTextureSize;

	float w = SNES_WIDTH;
	float h = GUI.HeightExtend ? SNES_HEIGHT_EXTENDED : SNES_HEIGHT;

	//scale it down some...
	w /= 100;
	h /= 100;

	float hw = w/2;
	float hh = h/2;

	Screen[0].x = -hw;
	Screen[0].y = hh;

	Screen[1].x = hw;
	Screen[1].y = hh;
	Screen[1].tu = tX;

	Screen[2].x = -hw;
	Screen[2].y = -hh;
	Screen[2].tv = tY;

	Screen[3].x = hw;
	Screen[3].y = hh;
	Screen[3].tu = tX;

	Screen[4].x = hw;
	Screen[4].y = -hh;
	Screen[4].tu = tX;
	Screen[4].tv = tY;

	Screen[5].x = -hw;
	Screen[5].y = -hh;
	Screen[5].tv = tY;
}

void CDirect3D::_RenderEye(StereoEyeParams Eye)
{
	HRESULT hr;
	D3DVIEWPORT9 view_port;
	LPDIRECT3DSURFACE9 pBackBuffer		= NULL;
	LPDIRECT3DSURFACE9 pBackDepth		= NULL;

	pDevice->GetRenderTarget(0,&pBackBuffer);
	pDevice->GetDepthStencilSurface(&pBackDepth);

	pDevice->SetRenderTarget(0,pRenderSurface);
	pDevice->SetDepthStencilSurface(pRenderDepth);

	view_port.X			= ceil(renderScale *Eye.VP.x);
	view_port.Y			= ceil(renderScale *Eye.VP.y);
	view_port.Width		= ceil(renderScale *Eye.VP.w);
	view_port.Height	= ceil(renderScale *Eye.VP.h);
	view_port.MinZ		= 0.0f;
	view_port.MaxZ		= 1.0f;
	pDevice->SetViewport(&view_port);

	pDevice->SetTransform(D3DTS_PROJECTION, (D3DMATRIX*)&Eye.Projection.Transposed());

	Matrix4f EyeView = Eye.ViewAdjust * View;
	pDevice->SetTransform(D3DTS_VIEW,(D3DMATRIX*)&EyeView.Transposed());

	D3DXMATRIX world_matrix;
	D3DXMatrixTranslation(&world_matrix,0,0,-2);
	pDevice->SetTransform(D3DTS_WORLD,&world_matrix);

	pDevice->Clear(0, NULL, D3DCLEAR_TARGET|D3DCLEAR_ZBUFFER, D3DCOLOR_XRGB(64, 64, 64), 1, 0);

	pDevice->BeginScene();

	Oculus_GameVertices();

	pDevice->SetTexture(0,drawSurface);
	pDevice->SetPixelShader(NULL);
	pDevice->SetVertexShader(NULL);
	pDevice->SetFVF(D3DFVF_CUSTOMVERTEX_SCREEN);
	pDevice->SetRenderState( D3DRS_CULLMODE ,D3DCULL_NONE );

	hr = pDevice->DrawPrimitiveUP(D3DPT_TRIANGLELIST,2,Screen,sizeof(CUSTOMVERTEX_SCREEN));

	pDevice->EndScene();

	pDevice->SetRenderTarget(0,pBackBuffer);
	pDevice->SetDepthStencilSurface(pBackDepth);
	
	if (pBackBuffer) pBackBuffer->Release();
	if (pBackDepth) pBackDepth->Release();

	Oculus_SetupFullScreenVertices();

	float	w = float(Eye.VP.w) / float(dPresentParams.BackBufferWidth),
			h = float(Eye.VP.h) / float(dPresentParams.BackBufferHeight),
			x = float(Eye.VP.x) / float(dPresentParams.BackBufferWidth),
			y = float(Eye.VP.y) / float(dPresentParams.BackBufferHeight);

	float as = float(Eye.VP.w) / float(Eye.VP.h);

	float scaleFactor = 1.0f / Eye.pDistortion->Scale;

	float lc[2];
	if (Eye.VP.x == 0)
		lc[0] = x + (w + Eye.pDistortion->XCenterOffset * 0.5f)*0.5f;
	else
		lc[0] = x + (w - Eye.pDistortion->XCenterOffset * 0.5f)*0.5f;

	lc[1] = y + h*0.5f;
	constantTable->SetFloatArray(pDevice,"LensCenter",lc,2);

	float sc[2];
	sc[0] = x + w*0.5f;
	sc[1] = y + h*0.5f;
	constantTable->SetFloatArray(pDevice,"ScreenCenter",lc,2);

	float s[2];
	s[0] = (w/2) * scaleFactor;
	s[1] = (h/2) * scaleFactor * as;
	constantTable->SetFloatArray(pDevice,"Scale",s,2);

	float si[2];
	si[0] = (2/w);
	si[1] = (2/h) / as;
	constantTable->SetFloatArray(pDevice,"ScaleIn",si,2);

	constantTable->SetFloatArray(pDevice,"HmdWarpParam",hmd.DistortionK,4);

	view_port.X			= Eye.VP.x;
	view_port.Y			= Eye.VP.y;
	view_port.Width		= Eye.VP.w;
	view_port.Height	= Eye.VP.h;
	view_port.MinZ		= 0.0f;
	view_port.MaxZ		= 1.0f;
	pDevice->SetViewport(&view_port);
	pDevice->BeginScene();

	pDevice->SetPixelShader(pixelShader);

	pDevice->SetTexture(0,pRenderTexture);  
	pDevice->SetFVF(D3DFVF_CUSTOMVERTEX_TEXTURE);

	pDevice->SetRenderState(D3DRS_FILLMODE ,D3DFILL_SOLID);
	pDevice->DrawPrimitiveUP(D3DPT_TRIANGLELIST,2,Verts,sizeof(CUSTOMVERTEX_TEXTURE));

	pDevice->EndScene();

}

const Vector3f UpVector(0.0f, 1.0f, 0.0f);
const Vector3f ForwardVector(0.0f, 0.0f, -1.0f);
const Vector3f RightVector(1.0f, 0.0f, 0.0f);

void CDirect3D::Oculus_Render()
{
	stereo.SetFullViewport(Viewport(0,0, dPresentParams.BackBufferWidth, dPresentParams.BackBufferHeight));
	stereo.SetStereoMode(Stereo_LeftRight_Multipass);
	stereo.SetHMDInfo(hmd);
	stereo.SetDistortionFitPointVP(-1.0f, 0.0f);
	renderScale = stereo.GetDistortionScale();

	static DWORD LastTick = 0;
	DWORD Tick = GetTickCount();

	float dt = (Tick - LastTick)/1000.0f;
	LastTick = Tick;

	Vector3f localMoveVector(0,0,0);
	//Matrix4f yawRotate = Matrix4f::RotationY(EyeYaw);

	if (GetKeyState(VK_UP)&0x8000) localMoveVector = ForwardVector;
	if (GetKeyState(VK_DOWN)&0x8000) localMoveVector = -ForwardVector;
	if (GetKeyState(VK_LEFT)&0x8000) SFusion.Reset();
	if (GetKeyState(VK_RIGHT)&0x8000) MagCal.BeginAutoCalibration(SFusion);

	if (localMoveVector.x || localMoveVector.y || localMoveVector.z)
	{
		localMoveVector.Normalize();
		
		//Vector3f orientationVector = yawRotate.Transform(localMoveVector);
		//orientationVector *= 3 * dt;
		//EyePos += orientationVector;

		localMoveVector *= 3 * dt;//for now lets just
		EyePos += localMoveVector;//move forward/back
	}

	if (MagCal.IsAutoCalibrating()) MagCal.UpdateAutoCalibration(SFusion);
	if (MagCal.IsCalibrated() && SFusion.IsMagReady()) SFusion.SetYawCorrectionEnabled(true);

	static float LastSensorYaw = 0;

	Quatf hmdOrient = SFusion.GetOrientation();
	float yaw = 0.0f;
	hmdOrient.GetEulerAngles<Axis_Y, Axis_X, Axis_Z>(&yaw, &EyePitch, &EyeRoll);
	EyeYaw += (yaw - LastSensorYaw);
	LastSensorYaw = yaw;

	Matrix4f rollPitchYaw = Matrix4f::RotationY(EyeYaw) * Matrix4f::RotationX(EyePitch) * Matrix4f::RotationZ(EyeRoll);
	Vector3f up      = rollPitchYaw.Transform(UpVector);
	Vector3f forward = rollPitchYaw.Transform(ForwardVector);

	// Minimal head modelling.
	float headBaseToEyeHeight     = 0.15f;  // Vertical height of eye from base of head
	float headBaseToEyeProtrusion = 0.09f;  // Distance forward of eye from base of head

	Vector3f eyeCenterInHeadFrame(0.0f, headBaseToEyeHeight, -headBaseToEyeProtrusion);
	Vector3f shiftedEyePos = EyePos + rollPitchYaw.Transform(eyeCenterInHeadFrame);
	shiftedEyePos.y -= eyeCenterInHeadFrame.y; // Bring the head back down to original height

	View = Matrix4f::LookAtRH(shiftedEyePos, shiftedEyePos + forward, up); 

	_RenderEye(stereo.GetEyeRenderParams(StereoEye_Left));
	_RenderEye(stereo.GetEyeRenderParams(StereoEye_Right));

}
