﻿
'Copyright © 2016 Jean-Jacques STACINO
' author mail : jj.stac @ aliceadsl.fr


'This program is free software: you can redistribute it and/or modify
'    it under the terms of the GNU General Public License as published by
'    the Free Software Foundation, either version 3 of the License, or
'    (at your option) any later version.

'    This program is distributed in the hope that it will be useful,
'    but WITHOUT ANY WARRANTY; without even the implied warranty of
'    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
'    GNU General Public License for more details.

'    You should have received a copy of the GNU General Public License
'    along with this program.  If not, see <http://www.gnu.org/licenses/>.


' This file is part of VBHector.

'    VBHector is free software: you can redistribute it and/or modify
'    it under the terms of the GNU General Public License as published by
'    the Free Software Foundation, either version 3 of the License, or
'    (at your option) any later version.

'    Foobar is distributed in the hope that it will be useful,
'    but WITHOUT ANY WARRANTY; without even the implied warranty of
'    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
'    GNU General Public License for more details.

'    You should have received a copy of the GNU General Public License
'    along with VBHector.  If not, see <http://www.gnu.org/licenses/>.

' Several part such as modZ80, modAY8912 are from GPL project (VBSpec)
'   Tranlation from VB6 to VBNet and much more had been made.
'   Thanks' to them authors (Chris Cowley, Xavier) 
'   Note that the original files from Chris Cowley are joined to this project (Directory "Sources Licences" . 
' Emulation of the MEA8000 is from A.Mine, that I want to thank here for him's agreement.

Option Explicit On
Option Strict On

Module MEA8000
    '**********************************************************************

    '  Copyright (C) Antoine Mine' 2006

    '  Philips / Signetics MEA 8000 emulation.

    '  The MEA 8000 is a speech synthesis chip.
    '  The French company TMPI (Techni-musique & parole informatique) provided
    '  speech extensions for several 8-bit computers (Thomson, Amstrad, Oric).
    '  It was quite popular in France because of its ability to spell 'u'
    '  (unlike the more widespread SPO 296 chip).

    ' The synthesis is based on a 4-formant model.
    ' First, an initial sawtooth noise signal is generated.
    ' The signal passes through a cascade of 4 filters of increasing frequency.
    ' Each filter is a second order digital filter with a programmable
    ' frequency and bandwidth.
    ' All parameters, including filter parameters, are smoothly interpolated
    ' for the duration of a frame (8ms, 16ms, 32ms, or 64 ms).

    ' TODO:
    ' - REQ output pin
    ' - optimize mea8000_compute_sample
    ' - should we accept new frames in slow-stop mode ?
    '
    '**********************************************************************

    'Transformed in VBnet by yo_fr
    ' Started the 27/03/2014
    ' JJ Stacino 
    ' Get from MESS emulator


    '******************* internal chip data structure ******************

    '* finite machine state controling frames *
    Enum mea8000_state
        MEA8000_STOPPED    ' nothing to do, timer disabled
        MEA8000_WAIT_FIRST ' received pitch, wait for first full trame, timer disabled
        MEA8000_STARTED    ' playing a frame, timer on
        MEA8000_SLOWING    ' repating last frame with decreasing amplitude, timer on
    End Enum

    Public Strg_MEA As String

    Public Structure filter_t
        Dim fm, last_fm As Double          ' frequency, in Hz *
        Dim bw, last_bw As Double          ' band-width, in Hz *
        Dim output, last_output As Double  ' filter state *
    End Structure

    Public Structure mea8000_t
        ' ' configuration parameters *
        '	const mea8000_interface* iface;

        ' state 
        Dim state As mea8000_state ' current state *

        Dim buf() As Byte  'UINT8  store 4 consecutive data to form a frame info *
        Dim bufpos As Byte ' new byte to write in frame info buffer *

        Dim cont As Byte  ' if no data 0=stop 1=repeat last frame *
        Dim roe As Byte   ' enable req output, now unimplemented *

        Dim framelength As Integer  ' in samples *
        Dim framepos As Integer     ' in samples *
        Dim framelog As Integer     ' log2 of framelength *

        Dim lastsample, sample As Integer  ' output samples are interpolated *

        Dim phi As Int32  ' absolute phase for frequency / noise generator *

        Dim f() As filter_t ' filters *

        Dim last_ampl, ampl As Integer    ' amplitude * 1000 *
        Dim last_pitch, pitch As Integer ' pitch of sawtooth signal, in Hz *
        Dim noise As Byte

        Dim emu_timer As Timer

        '	devcb_resolved_write8 req_out;
        Dim req_out As Int16
    End Structure

    Dim mea8000 As mea8000_t


    '******************* utilitiy function and macros ********************

    Private Sub log(ByRef Phrase As String)
        If Demande_Trace_MEA Then
            Strg_MEA = Strg_MEA & Phrase & vbCrLf
        End If
    End Sub
    Private Sub log_point(ByRef Phrase As String)
        If Demande_Trace_MEA Then
            Strg_MEA = Strg_MEA & Phrase
        End If
    End Sub

    ' digital filters work at 8 kHz *
    Const F0 = 8096

    ' filtered output is supersampled x 8 *
    Const SUPERSAMPLING = 8

    ' actual output pediod : pour la fonction TIMER 
    ' qui dans VBHector n'as pas lieu d'être : on est à 44100hz 
    ' via la fonction mea8000_echantillon appelée par le core de VBHector dans run()
    '   Const SAMPLING = 1000000.0 / SUPERSAMPLING * F0  '  attotime::from_hz((SUPERSAMPLING*F0)) 

    ' VBHector fonctionne à 44100 Hz 
    ' Le ppcm entre 44100 et 8096 c'est 89258400
    ' Il va falloir faire une approximation
    ' 8096 * 5 = 40480 hz
    ' C'est un compromis !

    'Sinon on lance un 64Khz sur le sample et on n'en prend que lors que l'on en a besoin en 44Khz !
    ' C'est ce qui est réalisé ici !

    ' 
    '************************* quantization tables ***********************
    ' frequency, in Hz *

    Dim fm1_table() As Int16 = {
    150, 162, 174, 188, 202, 217, 233, 250,
    267, 286, 305, 325, 346, 368, 391, 415,
    440, 466, 494, 523, 554, 587, 622, 659,
    698, 740, 784, 830, 880, 932, 988, 1047}

    Dim fm2_table() As Int16 = {
        440, 466, 494, 523, 554, 587, 622, 659,
        698, 740, 784, 830, 880, 932, 988, 1047,
        1100, 1179, 1254, 1337, 1428, 1528, 1639, 1761,
        1897, 2047, 2214, 2400, 2609, 2842, 3105, 3400}

    Dim fm3_table() As Int16 = {
        1179, 1337, 1528, 1761, 2047, 2400, 2842, 3400}

    Dim fm4_table() As Int16 = {3500}

    ' bandwidth, in Hz *
    Dim bw_table() As Int16 = {726, 309, 125, 50}

    ' amplitude * 1000 *
    Dim ampl_table() As Int16 = {
        0, 8, 11, 16, 22, 31, 44, 62,
        88, 125, 177, 250, 354, 500, 707, 1000}

    ' pitch increment, in Hz / 8 ms *
    Dim pi_table() As Int16 = {
        0, 1, 2, 3, 4, 5, 6, 7,
        8, 9, 10, 11, 12, 13, 14, 15,
        0, -15, -14, -13, -12, -11, -10, -9,
        -8, -7, -6, -5, -4, -3, -2, -1}

    '***************************** REQ **********************************

    Private Function mea8000_accept_byte() As Short
        If mea8000.state = mea8000_state.MEA8000_STOPPED Or
           mea8000.state = mea8000_state.MEA8000_WAIT_FIRST Or
          (mea8000.state = mea8000_state.MEA8000_STARTED And mea8000.bufpos < 4) Then
            Return -1
        Else
            Return 0
        End If
    End Function

    Private Sub mea8000_update_req()

        '  actually, req pulses less than 3us for each new byte,
        '  it goes back up if there space left in the buffer, or stays low if the
        '  buffer contains a complete frame and the CPU nees to wait for the next
        '  frame end to compose a new frame.
        '
        mea8000.req_out = mea8000_accept_byte()
    End Sub



    '*********************** sound generation ***************************

    ' table amplitude [-QUANT,QUANT] *
    Const QUANT = 512

    ' filter coefficients from frequencies *
    Const TABLE_LEN = 3600
    Dim cos_table(TABLE_LEN) As Int16  '  fm => cos coefficient *
    Dim exp_table(TABLE_LEN) As Int16  '  bw => exp coefficient *
    Dim exp2_table(TABLE_LEN) As Int16 '  bw => 2*exp coefficient *

    ' noise generator table *
    Const NOISE_LEN = 8192
    Dim noise_table(NOISE_LEN) As Int16

    Const M_PI = 3.1415926535897931

    ' precompute tables *
    Public Sub mea8000_init_tables()

        Dim i As Int16
        Dim f As Double
        For i = 0 To TABLE_LEN - 1
            f = i / F0
            cos_table(i) = CShort(2.0 * Math.Cos(2.0 * M_PI * f) * QUANT)
            exp_table(i) = CShort(Math.Exp(-M_PI * f) * QUANT)
            exp2_table(i) = CShort(Math.Exp(-2 * M_PI * f) * QUANT)
        Next
        For i = 0 To NOISE_LEN - 1
            noise_table(i) = CShort((Rnd() Mod (2 * QUANT)) - QUANT) ' \mod
        Next i
    End Sub

    ' float version !

    ' linear interpolation
    Private Function mea8000_interp(ByRef org As Double, ByRef dst As Double) As Double
        If mea8000.framelength <> 0 Then
            Return org + ((dst - org) * mea8000.framepos) / mea8000.framelength
        Else
            Return 0
        End If
    End Function
    Private Function mea8000_interp_S(ByRef org As Integer, ByRef dst As Integer) As Short
        If mea8000.framelength <> 0 Then
            Return CShort(org + ((dst - org) * mea8000.framepos) / mea8000.framelength)
        Else
            Return 0
        End If
    End Function

    ' apply second order digital filter, sampling at F0 *
    Private Function mea8000_filter_step(ByRef I As Int16, ByRef input As Double) As Double
        Dim fm, bw, b, c, next_output As Double

        fm = mea8000_interp(mea8000.f(I).last_fm, mea8000.f(I).fm)
        bw = mea8000_interp(mea8000.f(I).last_bw, mea8000.f(I).bw)
        b = 2.0 * Math.Cos(2.0 * M_PI * fm / F0)
        c = -Math.Exp(-M_PI * bw / F0)
        next_output = input - c * (b * mea8000.f(I).output + c * mea8000.f(I).last_output)
        mea8000.f(I).last_output = mea8000.f(I).output
        mea8000.f(I).output = next_output
        Return next_output
    End Function

    ' noise, in [-1,1] *
    Private Function mea8000_noise_gen() As Double
        mea8000.phi = mea8000.phi + 1
        Return noise_table(mea8000.phi Mod NOISE_LEN) / QUANT
    End Function


    ' sawtooth waveform at F0, in [-1,1] *
    Private Function mea8000_freq_gen() As Double
        Dim pitch As Int16
        pitch = mea8000_interp_S(mea8000.last_pitch, mea8000.pitch)
        mea8000.phi += pitch
        Return (mea8000.phi Mod F0) / (F0 / 2.0) - 1.0
    End Function


    ' sample in [-32767,32767], at F0 *
    Private Function mea8000_compute_sample() As Int16
        Dim i As Int16
        Dim out As Double
        Dim ampl As Double

        ampl = mea8000_interp(8.0 * mea8000.last_ampl, 8.0 * mea8000.ampl)

        If (mea8000.noise = 1) Then
            ' Si le choix était 0 dans la pi table => c'est du bruit
            out = mea8000_noise_gen()
        Else
            ' si non génération de la fréquence voulue
            out = mea8000_freq_gen()
        End If

        out = out * ampl 'out *= ampl

        ' Mise en place des 3 filtres de fréquence
        For i = 0 To 3
            out = mea8000_filter_step(i, out)
        Next i

        'Bornage de la sortie
        If (out > 32767) Then
            out = 32767
        End If
        If (out < -32767) Then
            out = -32767
        End If
   
        Return CShort(out)
    End Function

    '*********************** frame management ***************************

    ' shift frame parameters from current to last *
    Private Sub mea8000_shift_frame()
        Dim i As Int16

        mea8000.last_pitch = mea8000.pitch
        For i = 0 To 3
            mea8000.f(i).last_bw = mea8000.f(i).bw
            mea8000.f(i).last_fm = mea8000.f(i).fm
        Next i
        mea8000.last_ampl = mea8000.ampl
    End Sub

    ' decode fields from buffer to current frame *
    Private Sub mea8000_decode_frame()

        Dim fd As Integer
        Dim pi As Int16

        fd = (mea8000.buf(3) >> 5) And 3  ' 0=8ms, 1=16ms, 2=32ms, 3=64ms *

        'Le pitch est le décalage de fréquence de la voix générée (0-2khz => 0 -> 64 ms)
        pi = pi_table(mea8000.buf(3) And &H1F) << fd ' décalage du temps trouvé dans la pi table de 8 / 16 / 32 ou 64 ms !

        If (mea8000.buf(3) And &H1F) = 16 Then ' 16 c'est le bruit demandé dans la table des pi
            mea8000.noise = 1
        Else
            mea8000.noise = 0
        End If

        mea8000.pitch = mea8000.last_pitch + pi
        mea8000.f(0).bw = bw_table(mea8000.buf(0) >> 6)
        mea8000.f(1).bw = bw_table((mea8000.buf(0) >> 4) And 3)
        mea8000.f(2).bw = bw_table((mea8000.buf(0) >> 2) And 3)
        mea8000.f(3).bw = bw_table(mea8000.buf(0) And 3)
        mea8000.f(3).fm = fm4_table(0)
        mea8000.f(2).fm = fm3_table(mea8000.buf(1) >> 5)
        mea8000.f(1).fm = fm2_table(mea8000.buf(1) And &H1F)
        mea8000.f(0).fm = fm1_table(mea8000.buf(2) >> 3)
        mea8000.ampl = ampl_table(((mea8000.buf(2) And 7) << 1) Or (mea8000.buf(3) >> 7))
        mea8000.framelog = fd + 6 + 3  ' 64 samples / ms *
        mea8000.framelength = 1 << mea8000.framelog
        mea8000.bufpos = 0
        Dim ASSS As String
        Dim LL As Int32

        LL = (8 << fd)

        ASSS = "mea8000decode_frame: pitch=" & mea8000.pitch.ToString & vbCrLf
        ASSS = ASSS & " noise=" & mea8000.noise.ToString & vbCrLf
        ASSS = ASSS & " fm1=" & mea8000.f(0).fm.ToString
        ASSS = ASSS & "Hz" & vbCrLf & "bw1=" & mea8000.f(0).bw.ToString
        ASSS = ASSS & "Hz" & vbCrLf & "fm2=" & mea8000.f(2).fm.ToString
        ASSS = ASSS & "Hz" & vbCrLf & "bw2=" & mea8000.f(2).bw.ToString
        ASSS = ASSS & "Hz" & vbCrLf & "fm3=" & mea8000.f(2).fm.ToString
        ASSS = ASSS & "Hz" & vbCrLf & "bw3=" & mea8000.f(2).bw.ToString
        ASSS = ASSS & "Hz" & vbCrLf & "fm4=" & mea8000.f(3).fm.ToString
        ASSS = ASSS & "Hz" & vbCrLf & "bw4=" & mea8000.f(3).bw.ToString
        ASSS = ASSS & "Hz" & vbCrLf & "ampl=" & (mea8000.ampl / 1000.0) & vbCrLf
        ASSS = ASSS & " fd=" & LL.ToString & "ms" & vbCrLf
        log(ASSS)
    End Sub

    Private Sub mea8000_start_frame()
        ' enter or stay in active mode *
        ''jj    mea8000.timer.reset(SAMPLING)
        mea8000.state = mea8000_state.MEA8000_STARTED ' ajouté 
        mea8000.framepos = 0
    End Sub

    Private Sub mea8000_stop_frame()
        ' enter stop mode *
        mea8000.state = mea8000_state.MEA8000_STOPPED
        End Sub

    ' next sample in frame, sampling at 64 kHz *
    Public Function mea8000_echantillon() As Short
        Dim pos As Integer
        Dim Sample As Integer = 0

        ' Selon l'état courant du mea, on effecture les taches demandées! 
        If (mea8000.state = mea8000_state.MEA8000_STOPPED) Then
            mea8000_echantillon = CShort(Sample)
            Exit Function
        End If

        pos = mea8000.framepos Mod SUPERSAMPLING   ' Modulo par rapport à l'interpolation

        If (pos = 0) Then
            ' sample is really computed only every 8 (5) -th time *
            mea8000.lastsample = mea8000.sample
            mea8000.sample = mea8000_compute_sample()
            Sample = mea8000.lastsample
            log_point("*")
        Else
            ' other samples are simply interpolated *
            Sample = CInt(mea8000.lastsample + ((pos * (mea8000.sample - mea8000.lastsample)) / SUPERSAMPLING))
        End If

        ' On incrémente la position courante
        mea8000.framepos += 1
        ' on controle la fin de la frame
        If (mea8000.framepos >= mea8000.framelength) Then
            mea8000_shift_frame()

            ' end of frame *
            If (mea8000.bufpos = 4) Then
                ' we have a successor *
                log(vbCrLf & " mea8000_timer_expire: new frame arrive !")
                mea8000_decode_frame()
                mea8000_start_frame()

            ElseIf (mea8000.cont <> 0) Then
                ' repeat mode *
                log(" mea8000_timer_expire: repeat frame")
                mea8000_start_frame()
            ElseIf mea8000.state = mea8000_state.MEA8000_STARTED Then
                ' slow stop *
                mea8000.ampl = 0
                log(" mea8000_timer_expire: fade frame")
                mea8000_start_frame()
                mea8000.state = mea8000_state.MEA8000_SLOWING

            ElseIf mea8000.state = mea8000_state.MEA8000_SLOWING Then
                log(" mea8000_timer_expire: stop frame")
                mea8000_stop_frame()
            End If
        End If
        ' Calcul du req_out !
        mea8000_update_req()

        'Mise en place de ce que l'on a calculé !
        mea8000_echantillon = CShort(Sample)

        ' continue frame, pas de relance à faire

    End Function

    '************************** CPU interface ****************************
    Public Function mea8000_r(ByRef Offset As Int16) As Byte
        ' Lecture du registre du mea8000
        ' => Status register !

        Dim ret As Byte = 0

        Select Case Offset
            Case 0 To 1 ' status register
                ' ready to accept next frame ?
                If (mea8000_accept_byte() << 7) = -128 Then
                    ret = 128
                End If
            Case Else
                ret = 0
        End Select
        Return ret

    End Function

    Public Sub mea8000_w(ByRef data As Byte, ByRef Offset As Int16)
        ' Ecriture des registres
        ' 0 => data
        ' 1 => controle
        Select Case Offset

            Case 0 ' data register *
                If (mea8000.state = mea8000_state.MEA8000_STOPPED) Then

                    ' got pitch byte before first frame *
                    mea8000.pitch = 2 * data
                    log(" mea8000_w pitch " & mea8000.pitch)
                    mea8000.state = mea8000_state.MEA8000_WAIT_FIRST
                    mea8000.bufpos = 0

                ElseIf (mea8000.bufpos = 4) Then

                    ' overflow *
                    log(" mea8000_w data overflow " & data)

                Else

                    ' enqueue frame byte *
                    log(" mea8000_w data " & data & " in frame pos " & mea8000.bufpos)
                    mea8000.buf(mea8000.bufpos) = data ' Mise en place de l'octet  reçu
                    mea8000.bufpos = CByte(mea8000.bufpos + 1) ' On incrémente le pointeur de données recues !
                    If (mea8000.bufpos = 4 And mea8000.state = mea8000_state.MEA8000_WAIT_FIRST) Then
                        ' ça y est on a reçu nos 4 paramétres !
                        ' fade-in first frame 
                        Dim old_pitch As Integer
                        old_pitch = mea8000.pitch
                        mea8000.last_pitch = old_pitch
                        mea8000_decode_frame() ' on decode la frame courante
                        mea8000_shift_frame()
                        mea8000.last_pitch = old_pitch
                        mea8000.ampl = 0
                        mea8000_start_frame()
                        mea8000.state = mea8000_state.MEA8000_STARTED ' et on prévient que c'est parti !
                    End If
                End If
                mea8000_update_req() ' Calcul du req_out 
                '    break()

            Case 1 ' command register *
                Dim stop0 As Integer
                stop0 = (data >> 4) And 1

                If (data And 8) <> 0 Then
                    mea8000.cont = CByte((data >> 2) And 1)
                End If
                If (data And 2) <> 0 Then
                    mea8000.roe = CByte(data And 1)
                End If

                If (stop0 = 0) Then
                    mea8000_stop_frame()
                End If
                log(" mea8000_w command " & data & " stop=" & stop0 & "  cont=" & mea8000.cont & "  roe= " & mea8000.roe)

                mea8000_update_req() ' Calcul du req_out 

            Case Else
                log(" mea8000_w invalid write offset " & Offset)
        End Select
    End Sub

    '************************ reset *****************************

    Public Sub DEVICE_RESET_MEA8000()

        Dim i As Int16
        log(("mea8000_reset !"))
        ''jj  mea8000.timer.reset()

        ' init des tableaux dans la strucutre
        ReDim mea8000.f(4)
        ReDim mea8000.buf(4)

        'Init des valeurs
        mea8000.phi = 0
        mea8000.cont = 0
        mea8000.roe = 0
        mea8000.state = mea8000_state.MEA8000_STOPPED
        mea8000_update_req() ' Calcul du req_out 
        For i = 0 To 3
            mea8000.f(i).last_output = 0
            mea8000.f(i).output = 0
        Next i

    End Sub

End Module
