﻿
'Copyright © 2016 Jean-Jacques STACINO
' author mail : jj.stac @ aliceadsl.fr


'This program is free software: you can redistribute it and/or modify
'    it under the terms of the GNU General Public License as published by
'    the Free Software Foundation, either version 3 of the License, or
'    (at your option) any later version.

'    This program is distributed in the hope that it will be useful,
'    but WITHOUT ANY WARRANTY; without even the implied warranty of
'    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
'    GNU General Public License for more details.

'    You should have received a copy of the GNU General Public License
'    along with this program.  If not, see <http://www.gnu.org/licenses/>.


' This file is part of VBHector.

'    VBHector is free software: you can redistribute it and/or modify
'    it under the terms of the GNU General Public License as published by
'    the Free Software Foundation, either version 3 of the License, or
'    (at your option) any later version.

'    Foobar is distributed in the hope that it will be useful,
'    but WITHOUT ANY WARRANTY; without even the implied warranty of
'    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
'    GNU General Public License for more details.

'    You should have received a copy of the GNU General Public License
'    along with VBHector.  If not, see <http://www.gnu.org/licenses/>.

' Several part such as modZ80, modAY8912 are from GPL project (VBSpec)
'   Tranlation from VB6 to VBNet and much more had been made.
'   Thanks' to them authors (Chris Cowley, Xavier) 
'   Note that the original files from Chris Cowley are joined to this project (Directory "Sources Licences" . 
' Emulation of the MEA8000 is from A.Mine, that I want to thank here for him's agreement.



Public Module SN76477

    ' Emulation du Texas Instruments SN76477
    ' inspirée du projet MESS
    ' Adapté au VBNet par Yo_Fr
    ' entamé le 2/1/2013, avec le son 1 bit fonctionnel !

    'Fonctions de conversion de valeurs
    Public Function RES_K(ByRef Resistance_K) As Double
        RES_K = Resistance_K * 1000.0
    End Function
    Public Function RES_M(ByRef Resistance_M) As Double
        RES_M = Resistance_M * 1000000.0 '1e6
    End Function
    Public Function CAP_U(ByRef Capacite_U) As Double
        CAP_U = Capacite_U * 0.000001 '1e-6
    End Function
    Public Function CAP_N(ByRef Capacite_N) As Double
        CAP_N = Capacite_N * 0.000000001 '1e-9
    End Function

    'Constantes du SN
    Const ONE_SHOT_CAP_VOLTAGE_MIN As Double = 0         ' the voltage at which the one-shot starts from (measured) 
    Const ONE_SHOT_CAP_VOLTAGE_MAX As Double = 2.5       ' the voltage at which the one-shot finishes (measured) 
    Const ONE_SHOT_CAP_VOLTAGE_RANGE As Double = (ONE_SHOT_CAP_VOLTAGE_MAX - ONE_SHOT_CAP_VOLTAGE_MIN)

    Const SLF_CAP_VOLTAGE_MIN As Double = 0.33 ' the voltage at the bottom peak of the SLF triangle wave (measured) 
    Const SLF_CAP_VOLTAGE_MAX As Double = 2.37 ' the voltage at the top peak of the SLF triangle wave (measured) 
    Const SLF_CAP_VOLTAGE_RANGE As Double = (SLF_CAP_VOLTAGE_MAX - SLF_CAP_VOLTAGE_MIN)

    Const VCO_MAX_EXT_VOLTAGE As Double = 2.35  ' the external voltage at which the VCO saturates and produces no output,
    '   also used as the voltage threshold for the SLF
    Const VCO_TO_SLF_VOLTAGE_DIFF As Double = 0.35 ' 11/02/2016 ex 0.35   2.35 quand c'etait bien
    Const VCO_CAP_VOLTAGE_MIN As Double = SLF_CAP_VOLTAGE_MIN   ' the voltage at the bottom peak of the VCO triangle wave 
    Const VCO_CAP_VOLTAGE_MAX As Double = SLF_CAP_VOLTAGE_MAX + VCO_TO_SLF_VOLTAGE_DIFF  ' the voltage at the bottom peak of the VCO triangle wave 
    Const VCO_CAP_VOLTAGE_RANGE As Double = VCO_CAP_VOLTAGE_MAX - VCO_CAP_VOLTAGE_MIN
    Const VCO_DUTY_CYCLE_50 As Double = 5.0  ' the high voltage that produces a 50% duty cycle 
    Const VCO_MIN_DUTY_CYCLE As Double = 18 ' the smallest possible duty cycle, in % 

    Const NOISE_MIN_CLOCK_RES As Double = 10000   ' the maximum resistor value that still produces a noise (measured) 
    Const NOISE_MAX_CLOCK_RES As Double = 3300000 ' the minimum resistor value that still produces a noise (measured) 
    Const NOISE_CAP_VOLTAGE_MIN As Double = 0     ' the minimum voltage that the noise filter cap can hold (measured) 
    Const NOISE_CAP_VOLTAGE_MAX As Double = 5.0   ' the maximum voltage that the noise filter cap can hold (measured) 
    Const NOISE_CAP_VOLTAGE_RANGE As Double = (NOISE_CAP_VOLTAGE_MAX - NOISE_CAP_VOLTAGE_MIN)
    Const NOISE_CAP_HIGH_THRESHOLD As Double = 3.35  ' the voltage at which the filtered noise bit goes to 0 (measured) 
    Const NOISE_CAP_LOW_THRESHOLD As Double = 0.74  ' the voltage at which the filtered noise bit goes to 1 (measured) 

    Const AD_CAP_VOLTAGE_MIN As Double = 0    ' the minimum voltage the attack/decay cap can hold (measured) 
    Const AD_CAP_VOLTAGE_MAX As Double = 4.44 ' the minimum voltage the attack/decay cap can hold (measured) 
    Const AD_CAP_VOLTAGE_RANGE As Double = (AD_CAP_VOLTAGE_MAX - AD_CAP_VOLTAGE_MIN)

    Const OUT_CENTER_LEVEL_VOLTAGE As Double = 2.57  ' the voltage that gets outputted when the volumne is 0 (measured) 
    Const OUT_HIGH_CLIP_THRESHOLD As Double = 3.51   ' the maximum voltage that can be put out (measured) 
    Const OUT_LOW_CLIP_THRESHOLD As Double = 0.715   ' the minimum voltage that can be put out (measured) 

    Const SN76477_EXTERNAL_VOLTAGE_DISCONNECT As Double = -1.0  ' indicates that the voltage is internally computed,
    '                                                             can be used in all the functions that take a
    '                                                             voltage on a capacitor 

    '  gain factors for OUT voltage in 0.1V increments (measured) 
    ' Pour les valeurs de tension suivantes :
    ' 0.0 => 0,9V
    ' 1.0 => 1,9V
    ' 2.0 => 2,9V
    ' 3.0 => 3,9V
    ' 4.0 => 4,4V

    Dim out_pos_gain() As Double = _
    {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.01, _
     0.03, 0.11, 0.15, 0.19, 0.21, 0.23, 0.26, 0.29, 0.31, 0.33, _
     0.36, 0.38, 0.41, 0.43, 0.46, 0.49, 0.52, 0.54, 0.57, 0.6, _
     0.62, 0.65, 0.68, 0.7, 0.73, 0.76, 0.8, 0.82, 0.84, 0.87, _
     0.9, 0.93, 0.96, 0.98, 1.0}

    Dim out_neg_gain() As Double = { _
         0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, -0.01, _
        -0.02, -0.09, -0.13, -0.15, -0.17, -0.19, -0.22, -0.24, -0.26, -0.28, _
        -0.3, -0.32, -0.34, -0.37, -0.39, -0.41, -0.44, -0.46, -0.48, -0.51, _
        -0.53, -0.56, -0.58, -0.6, -0.62, -0.65, -0.67, -0.69, -0.72, -0.74, _
        -0.76, -0.78, -0.81, -0.84, -0.85}

    Public DBG As Integer = 0
    Public Const DBG_NONE As Integer = 0
    Public Const DBG_VCO As Integer = 1
    Public Const DBG_SLF As Integer = 2
    Public Const DBG_NOISE As Integer = 4
    Public Const DBG_ONES As Integer = 8
    Public Const DBG_A_D As Integer = 16
    Public Const DBG_MIX As Integer = 32
    Public Const DBG_ENV As Integer = 64
    Public Const DBG_OUT As Integer = 128
    Public Const DBG_ALL As Integer = 255

    Public Const Update_NONE As Integer = 0
    Public Const Update_VCO As Integer = 1
    Public Const Update_SLF As Integer = 2
    Public Const Update_NOISE As Integer = 4
    Public Const Update_DIVERS As Integer = 8
    Public Const Update_ALL As Integer = 255
    Dim Need_Update As Integer = Update_ALL


    '*****************************************************************************
    '
    '  State structure
    '
    '****************************************************************************

    ' chip's external interface 
    Dim enable As UInt32
    Public envelope_mode As UInt32 'public pour modif en mode forcage dans modHector
    Dim vco_mode As UInt32
    Dim mixer_mode As UInt32

    Dim one_shot_res As Double
    Dim one_shot_cap As Double
    Dim one_shot_cap_voltage_ext As UInt32

    Dim slf_res As Double
    Dim slf_cap As Double
    Dim slf_cap_voltage_ext As UInt32

    Dim vco_voltage As Double
    Dim vco_res As Double
    Dim vco_cap As Double
    Dim vco_cap_voltage_ext As UInt32

    Dim noise_clock_res As Double
    Dim noise_clock_ext As UInt32
    Dim noise_clock As UInt32
    Dim noise_filter_res As Double
    Dim noise_filter_cap As Double
    Dim noise_filter_cap_voltage_ext As UInt32

    Dim attack_res As Double
    Dim decay_res As Double
    Dim attack_decay_cap As Double
    Dim attack_decay_cap_voltage_ext As Boolean

    Dim amplitude_res As Double
    Dim feedback_res As Double
    Dim pitch_voltage As Double

    ' chip's internal state 
    Dim one_shot_cap_voltage As Double      ' voltage on the one-shot cap 
    Dim one_shot_running_ff As UInt32       ' 1 = one-shot running, 0 = stopped 

    Dim slf_cap_voltage As Double           ' voltage on the SLF cap 
    Dim slf_out_ff As UInt32  '			    ' output of the SLF 

    Dim vco_cap_voltage As Double           ' voltage on the VCO cap 
    Dim vco_out_ff As UInt32                ' output of the VCO 
    Dim vco_alt_pos_edge_ff As UInt32       ' keeps track of the # of positive edges for VCO Alt envelope 

    Dim noise_filter_cap_voltage As Double  ' voltage on the noise filter cap 
    Dim real_noise_bit_ff As UInt32         ' the current noise bit before filtering 
    Dim filtered_noise_bit_ff As UInt32     ' the noise bit after filtering 
    Dim noise_gen_count As UInt32           ' noise freq emulation ''''''''''''''''''''''''''''''''''''''''''''''''''''ex integer

    Dim attack_decay_cap_voltage As Double  ' voltage on the attack/decay cap 

    Dim rng As UInt32                       'current value of the random number generator

    ' others 
    Public sample_rate As UInt32               ' from machine.sample_rate() 
    Dim init As Boolean                     ' Flag sn initialisé


    ' Données de pas pour les calculs de charges et décharges condensateur
    Dim one_shot_cap_charging_step As Double
    Dim one_shot_cap_discharging_step As Double
    Dim slf_cap_charging_step As Double
    Dim slf_cap_discharging_step As Double
    Dim vco_duty_cycle_multiplier As Double
    Dim vco_cap_charging_step As Double
    Dim vco_cap_discharging_step As Double
    Dim vco_cap_voltage_max2 As Double
    Dim noise_gen_freq As Int64  'UINT32
    Dim noise_filter_cap_charging_step As Double
    Dim noise_filter_cap_discharging_step As Double
    Dim attack_decay_cap_charging_step As Double
    Dim attack_decay_cap_discharging_step As Double
    Dim attack_decay_cap_charging As Integer 'int
    Dim voltage_out As Double
    Dim center_to_peak_voltage_out As Double



    ' Données de pas pour les calculs de charges et décharges condensateur


    ' 07/02/2016        Dim attack_decay_cap_charging As Integer ''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''oublie ?
    ' End Structure

    Function Min(ByRef A As Double, ByRef B As Double) As Double
        If A < B Then
            Min = A
        Else
            Min = B
        End If
    End Function
    Function Max(ByRef A As Double, ByRef B As Double) As Double
        If A > B Then
            Max = A
        Else
            Max = B
        End If
    End Function
    Function Compute_Freq(ByRef Rate As Double, ByRef VMin As Double, ByRef VMax As Double) As Integer
        'Retourne la frequence selon le rate....
        Dim Tmp As Double
        If (VMax - VMin) > 0 Then
            Tmp = (Rate / (VMax - VMin)) / 2
        Else
            Tmp = 999.9
        End If
        Return Tmp
    End Function
    Function Compute_Time(ByRef Rate As Double, ByRef VMin As Double, ByRef VMax As Double) As Integer
        'Retourne le temps selon le rate....
        Dim Tmp As Double
        If (Rate) > 0 Then
            Tmp = (VMax - VMin) / Rate
        Else
            Tmp = 0.999
        End If
        Return 1000 * Tmp
    End Function
    '*****************************************************************************
    '*
    '*  Functions for computing frequencies, voltages and similar values based
    '*  on the hardware itself.  Do NOT put anything emulation specific here,
    '*  such as calculations based on sample_rate.
    '*
    '****************************************************************************

    Function compute_one_shot_cap_charging_rate() As Double 'in V/sec
        Static Old_ret As Double

        ' this formula was derived using the data points below

        '   Res (kohms)  Cap (uF)   Time (millisec)
        '      47         0.33         11.84
        '      47         1.0          36.2
        '      47         1.5          52.1
        '      47         2.0          76.4
        '     100         0.33         24.4
        '     100         1.0          75.2
        '     100         1.5         108.5
        '     100         2.0         158.4

        Dim ret As Double = 0

        If ((one_shot_res > 0) And (one_shot_cap > 0)) Then
            ret = ONE_SHOT_CAP_VOLTAGE_RANGE / (0.8024 * one_shot_res * one_shot_cap + 0.002079)

        ElseIf (one_shot_cap > 0) Then

            ' if no resistor, there is no current to charge the cap,
            '  effectively making the one-shot time effectively infinite 
            ret = +1.0E-30

        ElseIf (one_shot_res > 0) Then

            ' if no cap, the voltage changes extremely fast,
            ' effectively making the one-shot time 0 
            ret = +1.0E+30
        End If

        If Old_ret <> ret Then
            If (DBG And DBG_ONES) <> 0 Then
                Strg_SN += " Temps ONS : " & Compute_Time(ret, ONE_SHOT_CAP_VOLTAGE_MIN, ONE_SHOT_CAP_VOLTAGE_MAX).ToString & vbCrLf
            End If
            Old_ret = ret
        End If

        Return ret
    End Function
    Function compute_one_shot_cap_discharging_rate() As Double  'in V/sec
        Static Old_ret As Double

        ' this formula was derived using the data points below

        '    Cap (uF)   Time (microsec)
        '      0.33           300
        '      1.0            850
        '      1.5           1300
        '      2.0           1900

        Dim ret As Double = 0

        If ((one_shot_res > 0) And (one_shot_cap > 0)) Then

            ret = ONE_SHOT_CAP_VOLTAGE_RANGE / (854.7 * one_shot_cap + 0.00001795)

        ElseIf (one_shot_res > 0) Then

            ' if no cap, the voltage changes extremely fast,
            '  effectively making the one-shot time 0 
            ret = +1.0E+30
        End If
        If Old_ret <> ret Then
            If (DBG And DBG_ONES) <> 0 Then
                Strg_SN += "One Shot temps : " & Compute_Time(ret, ONE_SHOT_CAP_VOLTAGE_MIN, ONE_SHOT_CAP_VOLTAGE_MAX).ToString & vbCrLf
            End If

            Old_ret = ret
        End If

        Return ret
    End Function
    Function compute_slf_cap_charging_rate() As Double  ' in V/sec
        ' this formula was derived using the data points below

        'Res (kohms)  Cap (uF)   Time (millisec)
        '         47        0.47          14.3
        '        120        0.47          35.6
        '        200        0.47          59.2
        '         47        1.00          28.6
        '        120        1.00          71.6
        '        200        1.00         119.0

        Dim Ret As Double = 0

        If ((slf_res > 0) And (slf_cap > 0)) Then
            Ret = SLF_CAP_VOLTAGE_RANGE / (0.5885 * slf_res * slf_cap + 0.0013)
        End If

        Return Ret
    End Function
    Function compute_slf_cap_discharging_rate() As Double ' in V/sec 
        Static Old_ret As Double

        ' this formula was derived using the data points below

        '    Res (kohms)  Cap (uF)   Time (millisec)
        '         47        0.47          13.32
        '        120        0.47          32.92
        '        200        0.47          54.4
        '         47        1.00          26.68
        '        120        1.00          66.2
        '        200        1.00         109.6
        Dim ret As Double = 0

        If ((slf_res > 0) And (slf_cap > 0)) Then
            ret = SLF_CAP_VOLTAGE_RANGE / (0.5413 * slf_res * slf_cap + 0.001343)
        End If

        If Old_ret <> ret Then
            If (DBG And DBG_SLF) <> 0 Then
                Strg_SN += "frequence SLF : " & Compute_Freq((slf_cap_charging_step * sample_rate) + ret, SLF_CAP_VOLTAGE_MIN, SLF_CAP_VOLTAGE_MAX).ToString & "Hz" & vbCrLf
            End If

            Old_ret = ret
        End If
        Return ret
    End Function
    Function compute_vco_cap_charging_discharging_rate() As Double ' in V/sec 

        Dim ret As Double = 0
        Static Old_ret As Double

        If ((vco_res > 0) And (vco_cap > 0)) Then
            ret = 0.64 * 2 * VCO_CAP_VOLTAGE_RANGE / (vco_res * vco_cap)
        End If
        If Old_ret <> ret Then
            If (DBG And DBG_VCO) <> 0 Then
                Strg_SN += "VCO frequence min " & Compute_Freq(ret * 2, VCO_CAP_VOLTAGE_MIN, vco_cap_voltage_max2).ToString & "Hz" & vbCrLf
            End If
            Old_ret = ret
        End If

        Return ret
    End Function

    Function compute_vco_duty_cycle() As Double ' no measure, just a number 
        Dim ret As Double = 0.5   ' 50% 
        Static Old_ret As Double
        If ((vco_voltage > 0) And (pitch_voltage <> VCO_DUTY_CYCLE_50)) Then

            ret = Max(0.5 * (pitch_voltage / vco_voltage), (VCO_MIN_DUTY_CYCLE / 100.0))

            ret = Min(ret, 1)
        End If
        If Old_ret <> ret Then
            If (DBG And DBG_VCO) <> 0 Then
                Strg_SN += "VCO duty cycle " & CUInt(ret * 100) / 100 & vbCrLf
            End If
            Old_ret = ret
        End If

        Return ret
    End Function
    Function compute_noise_gen_freq() As UInt32 ' in Hz 

        ' this formula was derived using the data points below

        '     Res (ohms)   Freq (Hz)
        '       10k          97493
        '       12k         83333
        '       15k      68493
        '       22k       49164
        '       27k       41166
        '       33k       34449
        '       36k       31969
        '       47k       25126
        '       56k       21322
        '       68k      17721.5
        '       82k       15089.2
        '       100k       12712.0
        '       150k       8746.4
        '       220k       6122.4
        '       270k       5101.5
        '       330k       4217.2
        '       390k       3614.5
        '       470k       3081.7
        '       680k       2132.7
        '       820k       1801.8
        '       1M         1459.9
        '       2.2M       705.13
        '        3.3M      487.59

        Dim RET As UInt32 = 0
        Static Old_Data As UInt32

        If ((noise_clock_res >= NOISE_MIN_CLOCK_RES) And _
            (noise_clock_res <= NOISE_MAX_CLOCK_RES)) Then
            RET = 339100000 * Math.Pow(noise_clock_res, -0.8849)
        End If
        If RET <> Old_Data Then
            If (DBG And DBG_NOISE) <> 0 Then
                Strg_SN += "Noise : Frequence =" & RET & "Hz" & vbCrLf
            End If
            Old_Data = RET
        End If
        Return RET
    End Function
    Function compute_noise_filter_cap_charging_rate() As Double ' in V/sec 
        ' this formula was derived using the data points below

        '     R*C        Time (sec)
        '    .000068     .0000184
        '    .0001496    .0000378
        '    .0002244    .0000548
        '    .0003196    .000077
        '    .0015       .000248
        '    .0033       .000540
        '    .00495      .000792
        '    .00705      .001096

        Dim ret As Double = 0
        Static Old_Data As Double

        If ((noise_filter_res > 0) And (noise_filter_cap > 0)) Then

            ret = NOISE_CAP_VOLTAGE_RANGE / (0.1571 * noise_filter_res * noise_filter_cap + 0.0000143)

        ElseIf (noise_filter_cap > 0) Then
            ' if no resistor, there is no current to charge the cap,
            '      effectively making the filter's output constants 
            ret = +1.0E-30

        ElseIf (noise_filter_res > 0) Then
            ' if no cap, the voltage changes extremely fast,
            '      effectively disabling the filter 
            ret = +1.0E+30
        End If

        If ret <> Old_Data Then
            If (DBG And DBG_NOISE) <> 0 Then
                Strg_SN += "Noise : Frequence filtre =" & Compute_Freq(ret + (noise_filter_cap_discharging_step * sample_rate), NOISE_CAP_LOW_THRESHOLD, NOISE_CAP_HIGH_THRESHOLD) & "Hz" & vbCrLf
            End If
            Old_Data = ret
        End If
        Return ret
    End Function

    Function compute_noise_filter_cap_discharging_rate() As Double ' in V/sec 
        ' this formula was derived using the data points below

        '     R*C        Time (sec)
        '    .000068     .000016
        '    .0001496    .0000322
        '    .0002244    .0000472
        '    .0003196    .0000654
        '    .0015       .000219
        '    .0033       .000468
        '    .00495      .000676
        '    .00705      .000948

        Dim ret As Double = 0
        Static Old_Data As Double

        If ((noise_filter_res > 0) And (noise_filter_cap > 0)) Then
            ret = NOISE_CAP_VOLTAGE_RANGE / (0.1331 * noise_filter_res * noise_filter_cap + 0.00001734)
        ElseIf (noise_filter_cap > 0) Then
            ' if no resistor, there is no current to charge the cap,
            '  effectively making the filter's output constants 
            ret = +1.0E-30
        ElseIf (noise_filter_res > 0) Then
            ' if no cap, the voltage changes extremely fast,
            '  effectively disabling the filter 
            ret = +1.0E+30
        End If

        If ret <> Old_Data Then
            If (DBG And DBG_NOISE) <> 0 Then
                Strg_SN += "Noise : Frequence filtre =" & Compute_Freq(ret + (noise_filter_cap_charging_step * sample_rate), NOISE_CAP_LOW_THRESHOLD, NOISE_CAP_HIGH_THRESHOLD) & "Hz" & vbCrLf
            End If
            Old_Data = ret
        End If

        Return ret
    End Function
    Function compute_attack_decay_cap_charging_rate() As Double  ' in V/sec 
        Static OData As Double
        Dim ret As Double = 0

        If ((attack_res > 0) And (attack_decay_cap > 0)) Then
            ret = AD_CAP_VOLTAGE_RANGE / (attack_res * attack_decay_cap)
        ElseIf (attack_decay_cap > 0) Then
            ' if no resistor, there is no current to charge the cap,
            '  effectively making the attack time infinite 
            ret = +1.0E-30
        ElseIf (attack_res > 0) Then
            ' if no cap, the voltage changes extremely fast,
            '   effectively making the attack time 0 
            ret = +1.0E+30
        End If
        If OData <> ret Then
            If (DBG And DBG_A_D) <> 0 Then
                Strg_SN = Strg_SN + "Compute Attack charging rate = " & Compute_Time(ret, AD_CAP_VOLTAGE_MIN, AD_CAP_VOLTAGE_MAX).ToString & vbCrLf
            End If
            OData = ret
        End If

        Return ret
    End Function
    Function compute_attack_decay_cap_discharging_rate() As Double  ' in V/sec 
        Static OData As Double

        Dim ret As Double = 0

        If ((decay_res > 0) And (attack_decay_cap > 0)) Then
            ret = AD_CAP_VOLTAGE_RANGE / (decay_res * attack_decay_cap)
        ElseIf (attack_decay_cap > 0) Then
            ' if no resistor, there is no current to charge the cap,
            '   effectively making the decay time infinite 
            ret = +1.0E-30
        ElseIf (attack_res > 0) Then
            ' if no cap, the voltage changes extremely fast,
            '   effectively making the decay time 0 
            ret = +1.0E+30
        End If
        If OData <> ret Then
            If (DBG And DBG_A_D) <> 0 Then
                Strg_SN = Strg_SN + "Compute Decay discharge rate= " & Compute_Time(ret, AD_CAP_VOLTAGE_MIN, AD_CAP_VOLTAGE_MAX).ToString & "ms" & vbCrLf
            End If
            OData = ret
        End If

        Return ret
    End Function
    Function compute_center_to_peak_voltage_out() As Double
        ' this formula was derived using the data points below

        '     Ra (kohms)  Rf (kohms)   Voltage
        '        150         47          1.28
        '        200         47          0.96
        '         47         22          1.8
        '        100         22          0.87
        '        150         22          0.6
        '        200         22          0.45
        '         47         10          0.81
        '        100         10          0.4
        '        150         10          0.27

        Dim ret As Double = 0
        Static Old_Ret As Double
        If (amplitude_res > 0) Then
            ret = 3.818 * (feedback_res / amplitude_res) + 0.03
        End If

        If Old_Ret <> ret Then
            If (DBG And DBG_OUT) <> 0 Then
                Strg_SN += "compute_center_to_peak_voltage_out " & ret & vbCrLf
            End If
            Old_Ret = ret
        End If

        Return ret
    End Function

    '*****************************************************************************
    ' *
    ' *  Noise generator
    ' *
    ' ****************************************************************************
    Sub intialize_noise()
        rng = 0
    End Sub
    Function generate_next_real_noise_bit() As UInt32
        Dim out As UInt32 = (((rng >> 28) And 1) Xor ((rng >> 0) And 1))

        ' if bits 0-4 and 28 are all zero then force the output to 1 
        If ((rng And &H1000001F) = 0) Then
            out = 1
        End If
        rng = (rng >> 1) Or (out << 30)

        Return out
    End Function
    '*****************************************************************************
    ' *
    ' *  Set enable input
    ' *
    ' ****************************************************************************
    Sub sn76477_enable_w(ByRef data As Double)
        Static Old_Data As Double

        ' if falling edge (we know that enable had change)
        If (enable = 0) And (data = 1) Then

            ' start the attack phase 
            attack_decay_cap_voltage = AD_CAP_VOLTAGE_MAX

            ' one-shot runs regardless of envelope mode 
            one_shot_running_ff = 1
        End If
        enable = data
        If data <> Old_Data Then
            Need_Update = Need_Update Or Update_DIVERS
            If (DBG And DBG_ONES) <> 0 Or (DBG And DBG_OUT) <> 0 Then
                Strg_SN = Strg_SN + "enable= " & data & vbCrLf
                Strg_SN = Strg_SN + "  =>One shot = " & one_shot_running_ff & vbCrLf
            End If
        End If
        Old_Data = data
    End Sub
    '*****************************************************************************
    ' *
    ' *  Set mixer select inputs
    ' *
    ' ****************************************************************************
    Sub sn76477_mixer_a_w(ByRef data As UInt32)
        Static Old_Data As UInteger
        mixer_mode = ((mixer_mode And Not (&H1)) Or data)
        If mixer_mode <> Old_Data Then
            Need_Update = Need_Update Or Update_DIVERS
            Old_Data = mixer_mode
        End If
    End Sub
    Sub sn76477_mixer_b_w(ByRef data As UInt32)
        Static Old_Data As UInteger
        mixer_mode = (mixer_mode And Not (&H2)) Or (data << 1)
        If mixer_mode <> Old_Data Then
            Need_Update = Need_Update Or Update_DIVERS
            Old_Data = mixer_mode
        End If
    End Sub
    Sub sn76477_mixer_c_w(ByRef data As UInt32)
        Static Old_Data As UInteger
        mixer_mode = (mixer_mode And Not (&H4)) Or (data << 2)
        If mixer_mode <> Old_Data Then
            Need_Update = Need_Update Or Update_DIVERS
            Old_Data = mixer_mode
        End If
    End Sub
    '*****************************************************************************
    ' *
    ' *  Set envelope select inputs
    ' *
    ' ****************************************************************************
    Sub sn76477_envelope_1_w(ByRef data As Double)
        Dim Octet As Byte
        Static Old_Data As UInteger

        Octet = CByte(data) And 1
        envelope_mode = ((envelope_mode And Not (&H1)) Or Octet)
        If envelope_mode <> Old_Data Then
            Need_Update = Need_Update Or Update_DIVERS
            Old_Data = envelope_mode
        End If
    End Sub
    Sub sn76477_envelope_2_w(ByRef data As Double)
        Dim Octet As Byte
        Static Old_data As UInteger

        Octet = CByte(data) And 1
        envelope_mode = (envelope_mode And Not (&H2)) Or (Octet << 1)
        If envelope_mode <> Old_data Then
            Need_Update = Need_Update Or Update_DIVERS
            Old_data = envelope_mode
        End If
    End Sub
    '*****************************************************************************
    ' *
    ' *  Set VCO select input
    ' *
    ' ****************************************************************************
    Sub sn76477_vco_w(ByRef data As UInt32)
        If vco_mode <> data Then
            Need_Update = Need_Update Or Update_DIVERS
            If (DBG And DBG_VCO) <> 0 Then
                Strg_SN = Strg_SN + "VCO mode= " & data & vbCrLf
            End If
        End If
        vco_mode = data
    End Sub
    '*****************************************************************************
    ' *
    ' *  Set one-shot resistor
    ' *
    ' ****************************************************************************
    '
    Sub sn76477_one_shot_res_w(ByRef data As Double)
        If data <> one_shot_res Then
            Need_Update = Need_Update Or Update_DIVERS
        End If
        one_shot_res = data
    End Sub
    '*****************************************************************************
    ' *
    ' *  Set one-shot capacitor
    ' *
    ' ****************************************************************************
    Sub sn76477_one_shot_cap_w(ByRef data As Double)
        If data <> one_shot_cap Then
            Need_Update = Need_Update Or Update_DIVERS
        End If
        one_shot_cap = data
    End Sub
    '*****************************************************************************
    ' *
    ' *  Set the voltage on the one-shot capacitor
    ' *
    ' ****************************************************************************
    Sub sn76477_one_shot_cap_voltage_w(ByRef data As Double) 'Maitenant appelé : c'est par fiation de la tension sur V CAPA ons qu'Hector agit! 
        Static Old_Data As Double
        '   Cette instruction permet de fixer par etat la tension au niveau de l'entrée CAPA du sn
        '   cela permet de piloter la tension de capa avec un fil et non par chage de la capa  !
        '   C'est utilse pour le oneshot !
        If (data = SN76477_EXTERNAL_VOLTAGE_DISCONNECT) Then
            ' switch to internal, if not already 
            If (one_shot_cap_voltage_ext) Then
                one_shot_cap_voltage_ext = 0
            End If

        Else
            ' set the voltage on the cap 
            If (Not (one_shot_cap_voltage_ext) Or (data <> one_shot_cap_voltage)) Then
                one_shot_cap_voltage_ext = 1
                one_shot_cap_voltage = data
            End If
        End If
        If Old_Data <> data Then
            Need_Update = Need_Update Or Update_VCO
            If (DBG And DBG_VCO) <> 0 Then
                Strg_SN = Strg_SN + "One Shot Cap voltage= " & data & "V" & vbCrLf
            End If
            Old_Data = data
        End If

    End Sub
    '*****************************************************************************
    ' *
    ' *  Set SLF resistor
    ' *
    ' ****************************************************************************
    Sub sn76477_slf_res_w(ByRef data As Double)
        If data <> slf_res Then
            Need_Update = Need_Update Or Update_SLF
        End If
        slf_res = data
    End Sub
    '*****************************************************************************
    ' *
    ' *  Set SLF capacitor
    ' *
    ' ****************************************************************************
    Sub sn76477_slf_cap_w(ByRef data As Double)
        If data <> slf_cap Then
            Need_Update = Need_Update Or Update_SLF
        End If
        slf_cap = data
    End Sub
    '*****************************************************************************
    ' *
    ' *  Set the voltage on the SLF capacitor
    ' *
    ' *  This is an alternate way of controlling the VCO as described in the book
    ' *
    ' ****************************************************************************
    Sub sn76477_slf_cap_voltage_w(ByRef data As Double)
        '   Cette instruction permet de fixer par etat la tension au niveau de l'entrée CAPA du sn
        '   cela permet de piloter la tension de capa avec un fil et non par chage de la capa  !
        '   C'est utilse pour le oneshot !
        Static Old_Data As Double

        If (data = SN76477_EXTERNAL_VOLTAGE_DISCONNECT) Then
            ' switch to internal, if not already 
            If (slf_cap_voltage_ext) Then
                slf_cap_voltage_ext = 0
            End If

        Else
            ' set the voltage on the cap 
            If (Not (slf_cap_voltage_ext) Or (data <> slf_cap_voltage)) Then
                slf_cap_voltage_ext = 1
                slf_cap_voltage = data
            End If
        End If
        If data <> Old_Data Then
            Need_Update = Need_Update Or Update_SLF
        End If
    End Sub
    '*****************************************************************************
    ' *
    ' *  Set VCO resistor
    ' *
    ' ****************************************************************************
    Sub sn76477_vco_res_w(ByRef data As Double)
        If data <> vco_res Then
            Need_Update = Need_Update Or Update_VCO
        End If
        vco_res = data
    End Sub
    '*****************************************************************************
    ' *
    ' *  Set VCO capacitor
    ' *
    ' ****************************************************************************
    Sub sn76477_vco_cap_w(ByRef data As Double)
        If data <> vco_cap Then
            Need_Update = Need_Update Or Update_VCO
        End If
        vco_cap = data
    End Sub
    '*****************************************************************************
    ' *
    ' *  Set the voltage on the VCO capacitor
    ' *
    ' ****************************************************************************
    Sub sn76477_vco_cap_voltage_w(ByRef data As Double)
        Static Old_Data As Double

        If (data = SN76477_EXTERNAL_VOLTAGE_DISCONNECT) Then
            ' switch to internal, if not already 
            If (vco_cap_voltage_ext) Then
                vco_cap_voltage_ext = 0
            End If
        Else

            ' set the voltage on the cap 
            If (Not (vco_cap_voltage_ext) Or (data <> vco_cap_voltage)) Then
                vco_cap_voltage_ext = 1
                vco_cap_voltage = data
            End If
        End If
        If Old_Data <> data Then
            Need_Update = Need_Update Or Update_VCO
            Old_Data = data
        End If
    End Sub
    '*****************************************************************************
    ' *
    ' *  Set VCO voltage
    ' *
    ' ****************************************************************************
    Sub sn76477_vco_voltage_w(ByRef data As Double)

        If vco_voltage <> data Then
            Need_Update = Need_Update Or Update_VCO
            If (DBG And DBG_VCO) <> 0 Then
                Strg_SN += "VCO Nouvelle valeur tension=" + data.ToString + "V" & vbCrLf
            End If
        End If
        vco_voltage = data
    End Sub
    '*****************************************************************************
    ' *
    ' *  Set pitch voltage
    ' *
    ' ****************************************************************************
    Sub sn76477_pitch_voltage_w(ByRef data As Double)
        If (DBG And DBG_VCO) <> 0 Then
            If pitch_voltage <> data Then
                Strg_SN += "VCO Nouvelle valeur tension=" + data.ToString + "V" & vbCrLf
            End If
        End If
        If pitch_voltage <> data Then
            Need_Update = Need_Update Or Update_DIVERS Or Update_VCO
        End If

        pitch_voltage = data
    End Sub
    '*****************************************************************************
    ' *
    ' *  Set noise external clock
    ' *
    ' ****************************************************************************
    Sub sn76477_noise_clock_w(ByRef data As UInt32)
        If (data <> noise_clock) Then
            If (DBG And DBG_NOISE) <> 0 Then
                If noise_clock <> data Then
                    Strg_SN += "NOISE Clock=" + data.ToString & vbCrLf
                End If
            End If

            noise_clock = data

            ' on the rising edge shift generate next value,
            '  if external control is enabled 
            If (noise_clock And noise_clock_ext) Then
                real_noise_bit_ff = generate_next_real_noise_bit()
            End If
        End If
    End Sub
    '*****************************************************************************
    ' *
    ' *  Set noise clock resistor
    ' *
    ' ****************************************************************************
    Sub _SN76477_noise_clock_res_w(ByRef data As Double)

        If (data = 0) Then
            noise_clock_ext = 1
        Else
            noise_clock_ext = 0

            noise_clock_res = data
        End If
        Need_Update = Need_Update Or Update_NOISE

    End Sub
    Sub sn76477_noise_clock_res_w(ByRef data As Double)
        If (((data = 0) And Not (noise_clock_ext)) Or ((data <> 0) And (data <> noise_clock_res))) Then
            _SN76477_noise_clock_res_w(data)
        End If
    End Sub
    '*****************************************************************************
    ' *
    ' *  Set noise filter resistor
    ' *
    ' ****************************************************************************
    Sub sn76477_noise_filter_res_w(ByRef data As Double)
        If noise_filter_res <> data Then
            Need_Update = Need_Update Or Update_NOISE
        End If
        noise_filter_res = data
    End Sub
    '*****************************************************************************
    ' *
    ' *  Set noise filter capacitor
    ' *
    ' ****************************************************************************
    Sub sn76477_noise_filter_cap_w(ByRef data As Double)
        If noise_filter_cap <> data Then
            Need_Update = Need_Update Or Update_NOISE
        End If
        noise_filter_cap = data
    End Sub
    '*****************************************************************************
    ' *
    ' *  Set the voltage on the noise filter capacitor
    ' *
    ' ****************************************************************************
    Sub sn76477_noise_filter_cap_voltage_w(ByRef data As Double)
        If (data = SN76477_EXTERNAL_VOLTAGE_DISCONNECT) Then
            ' switch to internal, if not already 
            If (noise_filter_cap_voltage_ext) Then
                noise_filter_cap_voltage_ext = 0
            End If
        Else
            ' set the voltage on the cap 
            If (Not (noise_filter_cap_voltage_ext) Or (data <> noise_filter_cap_voltage)) Then
                noise_filter_cap_voltage_ext = 1
                noise_filter_cap_voltage = data
                Need_Update = Need_Update Or Update_NOISE
            End If
        End If
    End Sub
    '*****************************************************************************
    ' *
    ' *  Set attack resistor
    ' *
    ' ****************************************************************************
    Sub sn76477_attack_res_w(ByRef data As Double)
        If attack_res <> data Then
            Need_Update = Need_Update Or Update_DIVERS
        End If
        attack_res = data
    End Sub
    '*****************************************************************************
    ' *
    ' *  Set decay resistor
    ' *
    ' ****************************************************************************
    Sub sn76477_decay_res_w(ByRef data As Double)
        If decay_res <> data Then
            Need_Update = Need_Update Or Update_DIVERS
        End If
        decay_res = data
    End Sub
    '*****************************************************************************
    ' *
    ' *  Set attack/decay capacitor
    ' *
    ' ****************************************************************************
    Sub sn76477_attack_decay_cap_w(ByRef data As Double)
        If attack_decay_cap <> data Then
            Need_Update = Need_Update Or Update_DIVERS
        End If
        attack_decay_cap = data
    End Sub
    '*****************************************************************************
    ' *
    ' *  Set the voltage on the attack/decay capacitor
    ' *
    ' ****************************************************************************
    Sub sn76477_attack_decay_cap_voltage_w(ByRef data As Double)
        If (data = SN76477_EXTERNAL_VOLTAGE_DISCONNECT) Then
            ' switch to internal, if not already 
            If (attack_decay_cap_voltage_ext) Then
                attack_decay_cap_voltage_ext = False
            End If
        Else
            ' set the voltage on the cap 
            If (Not (attack_decay_cap_voltage_ext) Or (data <> attack_decay_cap_voltage)) Then
                attack_decay_cap_voltage_ext = True
                attack_decay_cap_voltage = data
                Need_Update = Need_Update Or Update_DIVERS
            End If
        End If
    End Sub
    '*****************************************************************************
    ' *
    ' *  Set amplitude resistor
    ' *
    ' ****************************************************************************
    Sub sn76477_amplitude_res_w(ByRef data As Double)
        If amplitude_res <> data Then
            Need_Update = Need_Update Or Update_DIVERS
        End If
        amplitude_res = data
    End Sub
    '*****************************************************************************
    ' *
    ' *  Set feedback resistor
    ' *
    ' ****************************************************************************
    Sub sn76477_feedback_res_w(ByRef data As Double)
        If feedback_res <> data Then
            Need_Update = Need_Update Or Update_DIVERS
        End If
        feedback_res = data
    End Sub
    '*****************************************************************************
    ' *
    ' *  Sample generation
    ' *
    ' ****************************************************************************
    Function STREAM_UPDATE() As Short

        ' ********************************************************************************
        ' *  Calcul des valeurs de pas d'évolution selon la frequence à générer
        ' ********************************************************************************

        Static Old_Env As UInteger
        Static Old_Mixer As UInteger
        Dim out As UInt32 = 0

        If (Need_Update And Update_DIVERS) <> 0 Then
            ' compute charging values, doing it here ensures that we always use the latest values 
            one_shot_cap_charging_step = compute_one_shot_cap_charging_rate() / sample_rate
            one_shot_cap_discharging_step = compute_one_shot_cap_discharging_rate() / sample_rate

            attack_decay_cap_charging_step = compute_attack_decay_cap_charging_rate() / sample_rate
            attack_decay_cap_discharging_step = compute_attack_decay_cap_discharging_rate() / sample_rate

            center_to_peak_voltage_out = compute_center_to_peak_voltage_out()

        End If

        If (Need_Update And Update_SLF) <> 0 Then
            slf_cap_charging_step = compute_slf_cap_charging_rate() / sample_rate
            slf_cap_discharging_step = compute_slf_cap_discharging_rate() / sample_rate
        End If

        If (Need_Update And Update_VCO) <> 0 Then
            vco_duty_cycle_multiplier = (1 - compute_vco_duty_cycle()) * 2
            vco_cap_charging_step = compute_vco_cap_charging_discharging_rate() / vco_duty_cycle_multiplier / sample_rate
            vco_cap_discharging_step = compute_vco_cap_charging_discharging_rate() * vco_duty_cycle_multiplier / sample_rate
        End If

        If (Need_Update And Update_NOISE) <> 0 Then
            noise_filter_cap_charging_step = compute_noise_filter_cap_charging_rate() / sample_rate
            noise_filter_cap_discharging_step = compute_noise_filter_cap_discharging_rate() / sample_rate
            noise_gen_freq = compute_noise_gen_freq()
        End If


        Need_Update = Update_NONE

        '*********************************
        ' update the one-shot cap voltage 
        '*********************************
        If (one_shot_cap_voltage_ext = 0) Then
            If one_shot_running_ff = 1 Then
                ' charging
                If enable = 1 Then 'enable = 0 => pas de son !
                    one_shot_cap_voltage = Min(one_shot_cap_voltage + one_shot_cap_charging_step, ONE_SHOT_CAP_VOLTAGE_MAX)
                End If
            Else
                ' discharging
                one_shot_cap_voltage = Max(one_shot_cap_voltage - one_shot_cap_discharging_step, ONE_SHOT_CAP_VOLTAGE_MIN)
            End If
        End If

        If (one_shot_cap_voltage >= ONE_SHOT_CAP_VOLTAGE_MAX) Then
            one_shot_running_ff = 0
        End If

        '************************
        ' update the SLF (super low frequency oscillator) 
        '************************
        If (slf_cap_voltage_ext = 0) Then
            ' internal 
            If (slf_out_ff = 0) Then
                ' charging 
                slf_cap_voltage = Min(slf_cap_voltage + slf_cap_charging_step, SLF_CAP_VOLTAGE_MAX)
            Else
                ' discharging 
                slf_cap_voltage = Max(slf_cap_voltage - slf_cap_discharging_step, SLF_CAP_VOLTAGE_MIN)
            End If
        End If

        If (slf_cap_voltage >= SLF_CAP_VOLTAGE_MAX) Then
            slf_out_ff = 1
        ElseIf (slf_cap_voltage <= SLF_CAP_VOLTAGE_MIN) Then
            slf_out_ff = 0
        End If

        '************************
        ' update the VCO (voltage controlled oscillator) 
        '************************
        If (vco_mode = 1) Then
            ' VCO is controlled by SLF 
            vco_cap_voltage_max2 = slf_cap_voltage + VCO_TO_SLF_VOLTAGE_DIFF
        Else
            ' VCO is controlled by external voltage 
            vco_cap_voltage_max2 = vco_voltage + VCO_TO_SLF_VOLTAGE_DIFF
        End If

        If (vco_cap_voltage_ext = 0) Then
            If (vco_out_ff = 0) Then
                ' charging  (le max vient soit de l'externe soit du SLF)
                vco_cap_voltage = Min(vco_cap_voltage + vco_cap_charging_step, vco_cap_voltage_max2)
            Else
                ' discharging 
                vco_cap_voltage = Max(vco_cap_voltage - vco_cap_discharging_step, VCO_CAP_VOLTAGE_MIN)
            End If
        End If

        If (vco_cap_voltage >= vco_cap_voltage_max2) Then
            'fin de la charge
            If (vco_out_ff = 0) Then
                ' positive edge 
                If vco_alt_pos_edge_ff = 0 Then
                    vco_alt_pos_edge_ff = 1
                Else
                    vco_alt_pos_edge_ff = 0
                End If
            End If

            vco_out_ff = 1

        ElseIf (vco_cap_voltage <= VCO_CAP_VOLTAGE_MIN) Then
            'fin de la décharge
            vco_out_ff = 0
        End If


        '************************
        ' update the noise generator 
        '************************
        While ((noise_clock_ext = 0) And (noise_gen_count <= noise_gen_freq))
            noise_gen_count = noise_gen_count + sample_rate
            real_noise_bit_ff = generate_next_real_noise_bit()
        End While

        noise_gen_count = noise_gen_count - noise_gen_freq

        '************************
        ' update the noise filter 
        '************************
        If (noise_filter_cap_voltage_ext = 0) Then
            ' internal 
            If (real_noise_bit_ff = 1) Then
                ' charging 
                noise_filter_cap_voltage = Min(noise_filter_cap_voltage + noise_filter_cap_charging_step, NOISE_CAP_VOLTAGE_MAX)
            Else
                ' discharging 
                noise_filter_cap_voltage = Max(noise_filter_cap_voltage - noise_filter_cap_discharging_step, NOISE_CAP_VOLTAGE_MIN)
            End If
        End If

        ' check the thresholds 
        If (noise_filter_cap_voltage >= NOISE_CAP_HIGH_THRESHOLD) Then
            filtered_noise_bit_ff = 0
        ElseIf (noise_filter_cap_voltage <= NOISE_CAP_LOW_THRESHOLD) Then
            filtered_noise_bit_ff = 1
        End If

        '************************
        ' based on the envelope mode figure out the attack/decay phase we are in 
        '************************

        Dim VAL0, VAL1, VAL2, VAL3 As Integer

        VAL0 = 0
        VAL1 = 1
        VAL2 = 2
        VAL3 = 3

        If DBG = DBG_ENV Then
            If (Old_Env <> envelope_mode) Then
                Strg_SN += " Enveloppe ="
                Select Case envelope_mode
                    Case VAL0
                        Strg_SN += " 0-VCO"
                    Case VAL1
                        Strg_SN += " 1-one-shot"
                    Case VAL2
                        Strg_SN += " 2-mixer only "
                    Case VAL3
                        Strg_SN += " 3-VCO with alternating polarity "
                    Case Else
                        Strg_SN += " by default mixer only "
                End Select
                Strg_SN += vbCrLf
            End If
        End If

        Old_Env = envelope_mode

        Select Case envelope_mode
            Case VAL0 ' VCO 
                attack_decay_cap_charging = vco_out_ff

            Case VAL1 ' one-shot 
                attack_decay_cap_charging = one_shot_running_ff

            Case VAL2 ' mixer only 
                attack_decay_cap_charging = 1   ' never a decay phase 

            Case VAL3 ' VCO with alternating polarity 
                attack_decay_cap_charging = vco_alt_pos_edge_ff And vco_out_ff

            Case Else  ' by default mixer only 
                attack_decay_cap_charging = 1   ' never a decay phase 

        End Select

        '************************
        ' update a/d cap voltage 
        '************************
        If (attack_decay_cap_voltage_ext = 0) Then
            If attack_decay_cap_charging = 1 Then

                If (attack_decay_cap_charging_step > 0) Then
                    attack_decay_cap_voltage = Min(attack_decay_cap_voltage + attack_decay_cap_charging_step, AD_CAP_VOLTAGE_MAX)
                Else
                    ' no attack, voltage to max instantly 
                    attack_decay_cap_voltage = AD_CAP_VOLTAGE_MAX
                End If
            Else
                ' discharging 
                If (attack_decay_cap_discharging_step > 0) Then
                    attack_decay_cap_voltage = Max(attack_decay_cap_voltage - attack_decay_cap_discharging_step, AD_CAP_VOLTAGE_MIN)
                Else
                    ' no decay, voltage to min instantly 
                    attack_decay_cap_voltage = AD_CAP_VOLTAGE_MIN
                End If
            End If
        End If

        'Affichage tension A/D
        '        MainForm.Valeur = attack_decay_cap_voltage * 100
        MainForm.Valeur = vco_cap_voltage * 100

        Static MAXI As Double = 0
        Static MINI As Double = 100

        If slf_cap_voltage > MAXI Then MAXI = slf_cap_voltage
        If slf_cap_voltage < MINI Then MINI = slf_cap_voltage

        '************************
        ' mix the output, if enabled, or not saturated by the VCO
        '************************
        If ((enable = 1) And (vco_cap_voltage <= VCO_CAP_VOLTAGE_MAX)) Then

            ' enabled 
            Select Case mixer_mode
                Case 0 ' VCO 
                    out = vco_out_ff

                Case 1 ' SLF 
                    out = slf_out_ff

                Case 2 ' noise 
                    out = filtered_noise_bit_ff

                Case 3 ' VCO and noise 
                    out = vco_out_ff And filtered_noise_bit_ff

                Case 4 ' SLF and noise 
                    out = slf_out_ff And filtered_noise_bit_ff

                Case 5 ' VCO, SLF and noise 
                    out = vco_out_ff And slf_out_ff And filtered_noise_bit_ff

                Case 6 ' VCO and SLF 
                    out = vco_out_ff And slf_out_ff

                Case 7 ' inhibit 
                    out = 0
                Case Else
                    out = 0
            End Select

            If (DBG And DBG_MIX) <> 0 Then
                If (Old_Mixer <> mixer_mode) Then
                    Strg_SN += " Mixer =" & mixer_mode & vbCrLf
                    Old_Mixer = mixer_mode
                End If
            End If

            '************************
            ' determine the OUT voltage from the attack/delay cap voltage and clip it 
            '************************
            If out = 1 Then
                voltage_out = OUT_CENTER_LEVEL_VOLTAGE + center_to_peak_voltage_out * out_pos_gain((attack_decay_cap_voltage * 10))
                voltage_out = Min(voltage_out, OUT_HIGH_CLIP_THRESHOLD)
            Else
                voltage_out = OUT_CENTER_LEVEL_VOLTAGE + center_to_peak_voltage_out * out_neg_gain((attack_decay_cap_voltage * 10))
                voltage_out = Max(voltage_out, OUT_LOW_CLIP_THRESHOLD)
            End If

        Else

            ' disabled 
            voltage_out = OUT_CENTER_LEVEL_VOLTAGE
        End If

        ' convert it to a signed 16-bit sample,
        '     -32767 = OUT_LOW_CLIP_THRESHOLD
        '          0 = OUT_CENTER_LEVEL_VOLTAGE
        '      32767 = 2 * OUT_CENTER_LEVEL_VOLTAGE + OUT_LOW_CLIP_THRESHOLD
        '
        '          / Vout - Vmin    \
        'sample = |  ----------- - 1 | * 32767
        '          \ Vcen - Vmin    /

        STREAM_UPDATE = (((voltage_out - OUT_LOW_CLIP_THRESHOLD) / (OUT_CENTER_LEVEL_VOLTAGE - OUT_LOW_CLIP_THRESHOLD)) - 1) * 6000 ' un peu de place pour le mea ! => 16383 ' 32767
    End Function

End Module
