/**
 *                         Sidplay2 config file reader.
 *                         ----------------------------
 *  begin                : Sun Mar 25 2001
 *  copyright            : (C) 2000 by Simon White
 *  email                : s_a_white@email.com
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 * @author Ken Hndel
 *
 */
package sidplay;

import static libsidplay.common.ISID2Types.SID2_DEFAULT_OPTIMISATION;
import static libsidplay.common.ISID2Types.SID2_DEFAULT_PRECISION;
import static libsidplay.common.ISID2Types.SID2_DEFAULT_SAMPLING_FREQ;
import static libsidplay.common.ISID2Types.sid2_model_t.SID2_MOS8580;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;

import libsidplay.common.ISID2Types.sid2_clock_t;
import libsidplay.common.ISID2Types.sid2_model_t;
import libsidplay.common.ISID2Types.sid2_playback_t;
import libsidutils.IniReader;
import libsidutils.SidFilter;
import resid_builder.sid_filter_t;

public class IniConfig {
	public class sidplay2_section {
		int version;
		String database;
		int /* uint_least32_t */playLength;
		int /* uint_least32_t */recordLength;
	};

	public class console_section {
		// INI Section - [Console]
		boolean ansi;
		char topLeft;
		char topRight;
		char bottomLeft;
		char bottomRight;
		char vertical;
		char horizontal;
		char junctionLeft;
		char junctionRight;
	};

	public class audio_section {
		// INI Section - [Audio]
		long frequency;
		sid2_playback_t playback;
		int precision;
	};

	public class emulation_section {
		// INI Section - [Emulation]
		sid2_clock_t clockSpeed;
		boolean clockForced;
		sid2_model_t sidModel;
		boolean filter;
		String filter6581;
		String filter8580;
		byte /* uint_least8_t */optimiseLevel;
		boolean sidSamples;
	};

	protected static final String FILE_NAME = "sidplay2.ini";

	protected boolean status;
	protected sidplay2_section sidplay2_s = new sidplay2_section();
	protected console_section console_s = new console_section();
	protected audio_section audio_s = new audio_section();
	protected emulation_section emulation_s = new emulation_section();
	protected SidFilter filter6581 = new SidFilter();
	protected SidFilter filter8580 = new SidFilter();

	protected void clear() {
		sidplay2_s.version = 1; // INI File Version
		(sidplay2_s.database) = null;
		sidplay2_s.playLength = 0; // INFINITE
		sidplay2_s.recordLength = 3 * 60 + 30; // 3.5 minutes

		console_s.ansi = false;
		console_s.topLeft = '+';
		console_s.topRight = '+';
		console_s.bottomLeft = '+';
		console_s.bottomRight = '+';
		console_s.vertical = '|';
		console_s.horizontal = '-';
		console_s.junctionLeft = '+';
		console_s.junctionRight = '+';

		audio_s.frequency = SID2_DEFAULT_SAMPLING_FREQ;
		audio_s.playback = sid2_playback_t.sid2_mono;
		audio_s.precision = SID2_DEFAULT_PRECISION;

		emulation_s.clockSpeed = sid2_clock_t.SID2_CLOCK_CORRECT;
		emulation_s.clockForced = false;
		emulation_s.sidModel = sid2_model_t.SID2_MODEL_CORRECT;
		emulation_s.filter = true;
		emulation_s.optimiseLevel = SID2_DEFAULT_OPTIMISATION;
		emulation_s.sidSamples = true;

		(emulation_s.filter6581) = null;
		(emulation_s.filter8580) = null;
	}

	protected boolean readSidplay2(IniReader ini) {
		boolean ret = true;
		int time = 0, version = sidplay2_s.version;

		ret &= (version = ini.getPropertyInt("SIDPlay2", "Version", version)) > 0;
		if (version > 0)
			sidplay2_s.version = version;

		sidplay2_s.database = ini.getPropertyString("SIDPlay2",
				"Songlength Database", sidplay2_s.database);
		if ((time = ini
				.getPropertyTime("SIDPlay2", "Default Play Length", time)) > 0)
			sidplay2_s.playLength = (int /* uint_least32_t */) time;
		if ((time = ini.getPropertyTime("SIDPlay2", "Default Record Length",
				time)) > 0)
			sidplay2_s.recordLength = (int /* uint_least32_t */) time;

		return ret;
	}

	protected boolean readConsole(IniReader ini) {
		boolean ret = true;
		console_s.ansi = ini.getPropertyBool("Console", "Ansi", console_s.ansi);

		console_s.topLeft = getChar(ini.getPropertyString("Console",
				"Char Top Left", "'" + console_s.topLeft + "'"),
				console_s.topLeft);
		console_s.topRight = getChar(ini.getPropertyString("Console",
				"Char Top Right", "'" + console_s.topRight + "'"),
				console_s.topRight);
		console_s.bottomLeft = getChar(ini.getPropertyString("Console",
				"Char Bottom Left", "'" + console_s.bottomLeft + "'"),
				console_s.bottomLeft);
		console_s.bottomRight = getChar(ini.getPropertyString("Console",
				"Char Bottom Right", "'" + console_s.bottomRight + "'"),
				console_s.bottomRight);
		console_s.vertical = getChar(ini.getPropertyString("Console",
				"Char Vertical", "'" + console_s.vertical + "'"),
				console_s.vertical);
		console_s.horizontal = getChar(ini.getPropertyString("Console",
				"Char Horizontal", "'" + console_s.horizontal + "'"),
				console_s.horizontal);
		console_s.junctionLeft = getChar(ini.getPropertyString("Console",
				"Char Junction Left", "'" + console_s.junctionLeft + "'"),
				console_s.junctionLeft);
		console_s.junctionRight = getChar(ini.getPropertyString("Console",
				"Char Junction Right", "'" + console_s.junctionRight + "'"),
				console_s.junctionRight);
		return ret;
	}

	private char getChar(String str, char defaultChar) {
		char c = 0;
		// Check if we have an actual Character
		if (str.charAt(0) == '\'') {
			if (str.charAt(2) != '\'')
				status = false;
			else
				c = str.charAt(1);
		} // Nope is number
		else
			c = (char) Integer.parseInt(str);

		// Clip off special characters
		if ((int) c >= 32)
			return c;
		return defaultChar;
	}

	protected boolean readAudio(IniReader ini) {
		boolean ret = true;
		{
			int frequency = (int) audio_s.frequency;
			ret &= (frequency = ini.getPropertyInt("Audio", "Frequency",
					frequency)) > 0;
			audio_s.frequency = frequency;
		}

		{
			int channels = 0;
			ret &= (channels = ini
					.getPropertyInt("Audio", "Channels", channels)) > 0;
			if (channels > 0) {
				audio_s.playback = sid2_playback_t.sid2_mono;
				if (channels != 1)
					audio_s.playback = sid2_playback_t.sid2_stereo;
			}
		}

		ret &= (audio_s.precision = ini.getPropertyInt("Audio",
				"BitsPerSample", audio_s.precision)) > 0;
		return ret;
	}

	protected boolean readEmulation(IniReader ini) {
		boolean ret = true;
		{
			int clockSpeed = -1;
			ret &= (clockSpeed = ini.getPropertyInt("Emulation", "ClockSpeed",
					clockSpeed)) > 0;
			if (clockSpeed != -1) {
				if (clockSpeed < 0 || clockSpeed > 1)
					ret = false;
				emulation_s.clockSpeed = sid2_clock_t.SID2_CLOCK_PAL;
				if (clockSpeed > 0)
					emulation_s.clockSpeed = sid2_clock_t.SID2_CLOCK_NTSC;
			}
		}

		emulation_s.clockForced = ini.getPropertyInt("Emulation",
				"ForceSongSpeed", emulation_s.clockForced ? 1 : 0) > 0;

		{
			int mos8580 = -1;
			ret &= (mos8580 = ini.getPropertyInt("Emulation", "MOS8580",
					mos8580)) > 0;
			if (mos8580 != -1) {
				if (mos8580 < 0 || mos8580 > 1)
					ret = false;
				emulation_s.sidModel = sid2_model_t.SID2_MOS6581;
				if (mos8580 > 0)
					emulation_s.sidModel = sid2_model_t.SID2_MOS8580;
			}
		}

		emulation_s.filter = ini.getPropertyInt("Emulation", "UseFilter",
				emulation_s.filter ? 1 : 0) > 0;

		{
			int optimiseLevel = -1;
			ret &= (optimiseLevel = ini.getPropertyInt("Emulation",
					"OptimiseLevel", optimiseLevel)) > 0;
			if (optimiseLevel != -1)
				emulation_s.optimiseLevel = (byte) optimiseLevel;
		}

		emulation_s.filter6581 = ini.getPropertyString("Emulation",
				"Filter6581", emulation_s.filter6581);
		emulation_s.filter8580 = ini.getPropertyString("Emulation",
				"Filter8580", emulation_s.filter8580);
		emulation_s.sidSamples = ini.getPropertyInt("Emulation", "SidSamples",
				emulation_s.sidSamples ? 1 : 0) > 0;

		// These next two change the ini section!
		if (emulation_s.filter6581 != null && emulation_s.filter6581.length() > 0) {
			// Try to load the filter
			filter6581.read(ini, emulation_s.filter6581);
			if (!filter6581.bool()) {
				filter6581.read(emulation_s.filter6581);
				if (!filter6581.bool())
					ret = false;
			}
		}

		if (emulation_s.filter8580 != null && emulation_s.filter8580.length() > 0) {
			// Try to load the filter
			filter8580.read(ini, emulation_s.filter8580);
			if (!filter8580.bool()) {
				filter8580.read(emulation_s.filter8580);
				if (!filter8580.bool())
					ret = false;
			}
		}

		return ret;
	}

	public IniConfig() {
		status = (true);

		// Initialize everything else
		sidplay2_s.database = null;
		emulation_s.filter6581 = null;
		emulation_s.filter8580 = null;
		clear();
	}

	public void read() {
		String path = System.getProperty("user.home");
		IniReader ini = null;
		String configPath;

		if (path == null)
			path = "";

		configPath = String.format("%s/%s", path, FILE_NAME);

		// Opens an existing file or creates a new one
		try {
			File iniFile = new File(configPath.replace("\\", "/"));
			if (!iniFile.exists()) {
				InputStream is = getClass().getClassLoader()
						.getResourceAsStream("sidplay/ini/" + FILE_NAME);
				FileOutputStream os = new FileOutputStream(iniFile);
				while (is.available() > 0) {
					os.write(is.read());
				}
				os.close();
				is.close();
				System.out.println("INI file created: " + iniFile.getAbsolutePath());
			}
			ini = new IniReader(iniFile.getAbsolutePath());
		} catch (IOException e) {
			// Unable to open file?
			clear();
			status = false;
			return;
		}

		clear();

		// This may not exist here...
		status &= readSidplay2(ini);
		status &= readConsole(ini);
		status &= readAudio(ini);
		status &= readEmulation(ini);
		return;
	}

	public boolean bool() {
		return status;
	}

	// Sidplay2 Specific Section
	final sidplay2_section sidplay2() {
		return sidplay2_s;
	}

	final console_section console() {
		return console_s;
	}

	final audio_section audio() {
		return audio_s;
	}

	final emulation_section emulation() {
		return emulation_s;
	}

	final sid_filter_t filter(sid2_model_t model) {
		if (model == SID2_MOS8580)
			return filter8580.provide();
		return filter6581.provide();
	}
}
