// **********************************
// * VICE Default Javascript Script *
// *        (C) by F. Gerbig        *
// *           15.11.2022           *
// *                                *
// *            based on            *
// *  WinVICE Default GEMUS Script  *
// *        (c) The GB64 Team       *
// *          8 August 2007         *
// *             and on             *
// *           "vice.sh"            *
// *        (C) by F. Gerbig        *
// *           2006-2018            *
// **********************************
//
//  PLEASE SET YOUR EMULATOR PROPERTIES AS FOLLOWS:
//
//  Emulator File: x64sc
//  Emulator configuration file: vice.ini
//  File Types: d64;t64;g64;g41;tap;crt;p00;prg;lnx
//
//  This script was written for VICE C64 v3.6,
//  but may work with other versions too.
//
//  Emulator Website: http://www.viceteam.org
//
//  if you have a disk item that comes on multiple disks
//  and works with multiple drives, you can set a
//  key=value pair to auto insert all the disks into
//  VICE's drives.  Simply press F2 in the main jGB window
//  and enter: maxdrives=x
//  where x is either 2, 3 or 4
//
//  if VICE requires True Drive Emulation use the key=value pair: vtde=yes
//
//  To enable the REU ("RAM Expansion Unit"),
//  use the key=value pair: reu=x
//  where x is 128,256,512,1024,2048,4086,8192 or 16384


function enableTruedrive() {
	// look for floppy speeder roms (which always need true drive emulation)
	// put ROMs with correct names in database "ROMs" directory
	// findRom returns the complete path+filename of the rom found first

	// *JiffyDOS* ROMs
	var KERNAL_ROM_JIFFYDOS = new File(findRom( ["JiffyDOS_C64.bin", "kernal.jfd", "Kernal JiffyDOS V6.01 (1989)(Creative Micro Designs).rom", "Kernal JiffyDOS V6.01 (2015)(Creative Micro Designs)[h DolphinDOS LogicDeLuxe].rom"] ));
	var DISK_ROM_JIFFYDOS =  new File(findRom( ["JiffyDOS_1541-II.bin", "d1541ii.jfd", "1541-II JiffyDOS VX.XX (19xx)(Creative Micro Designs)[outdated].rom"]));
	
	// *DolphinDOS 2* ROMs
	var KERNAL_ROM_DOLPHINDOS2 = new File(findRom( ["sd_dd2_kernal_rom.bin", "dd2_kernal_rom.bin", "kernal.dd2", "Kernal Dolphin-DOS V2.0-1 (198x)(Dolphin Software).rom", "Kernal Dolphin-DOS V2.0-2 (198x)(Dolphin Software).rom", "Kernal Dolphin-DOS V2.0-3 (1986)(Dolphin Software).rom"] ));
	var DISK_ROM_DOLPHINDOS2 =   new File(findRom( ["2dosa_c.bin", "d1541.dd2", "1541 Dolphin-DOS V2.0 (198x)(Dolphin Software).rom", "1541 Dolphin-DOS V2.0-X (198x)(Dolphin Software).rom"] ));

	// *DolphinDOS 3* ROMs
	var KERNAL_ROM_DOLPHINDOS3 = new File(findRom( ["kernal.dd3", "Kernal Dolphin-DOS V3.0 (198x)(Dolphin Software).rom"] ));
	var DISK_ROM_DOLPHINDOS3 =   new File(findRom( ["3dosa_c_27c256_payload.bin", "d1541ii.dd3", "1541 Dolphin-DOS V3.0 (198x)(Dolphin Software).rom"] ));

	// *SpeedDOS Plus* ROMs
	var BASIC_ROM_SPEEDDOSPLUS = new File(findRom( ["basic.sdp"] ));
	var KERNAL_ROM_SPEEDDOSPLUS = new File(findRom( ["kernal_en_sd", "kernal.sdp", "Kernal SpeedDOS plus V2.0 (198x)(Elektronik Service).rom", "Kernal SpeedDOS plus+ V2.0 (198x)(Elektronik Service).rom", "Kernal SpeedDOS plus+ V2.7 (198x)(Elektronik Service).rom", "Kernal SpeedDOS plus V2.7 (xxxx)(Elektronik Service)[h Mr.Z].rom"] ));
	var DISK_ROM_SPEEDDOSPLUS = new File(findRom( ["1541both_sd", "d1541ii.sdp", "1541-II SpeedDOS plus V2.7 (xxxx)(Elektronik Service)[h Mr.Z].rom", "1541 SpeedDOS plus V2.7 (xxxx)(Elektronik Service)[h Mr.Z][0xE000].rom"] ));

	if (KERNAL_ROM_JIFFYDOS.exists() && DISK_ROM_JIFFYDOS.exists()) {
		// *JiffyDOS* parameters
		Add_CLP("-drive8type", "1542") // 1542: CBM 1541-II
		Add_CLP("-drive8truedrive")
		Add_CLP("-kernal", KERNAL_ROM_JIFFYDOS);
		Add_CLP("-dos1541II", DISK_ROM_JIFFYDOS);
	} else if (KERNAL_ROM_DOLPHINDOS2.exists() && DISK_ROM_DOLPHINDOS2.exists()) {
		// *DolphinDOS 2* parameters
		Add_CLP("-drive8type", "1541")
		Add_CLP("-drive8truedrive")
		Add_CLP("-drive8ram8000")
		Add_CLP("-kernal", KERNAL_ROM_DOLPHINDOS2);
		Add_CLP("-dos1541", DISK_ROM_DOLPHINDOS2);
		Add_CLP("-userportdevice", "20")
		Add_CLP("-parallel8", "1")
	} else if (KERNAL_ROM_DOLPHINDOS3.exists() && DISK_ROM_DOLPHINDOS3.exists()) {
		// *DolphinDOS 3* parameters
		Add_CLP("-drive8type", "1542") // 1542: CBM 1541-II
		Add_CLP("-drive8truedrive")
		Add_CLP("-drive8ram6000")
		Add_CLP("-kernal", KERNAL_ROM_DOLPHINDOS3);
		Add_CLP("-dos1541II", DISK_ROM_DOLPHINDOS3);
		Add_CLP("-userportdevice", "20")
		Add_CLP("-parallel8", "2")
	} else if (KERNAL_ROM_SPEEDDOSPLUS.exists() && DISK_ROM_SPEEDDOSPLUS.exists()) {
		// *SpeedDOS Plus* parameters
		Add_CLP("-drive8type", "1542") // 1542: CBM 1541-II
		Add_CLP("-drive8truedrive")
		if (BASIC_ROM_SPEEDDOSPLUS.exists()) {
			Add_CLP("-basic", BASIC_ROM_SPEEDDOSPLUS);
		}
		Add_CLP("-kernal", KERNAL_ROM_SPEEDDOSPLUS);
		Add_CLP("-dos1541II", DISK_ROM_SPEEDDOSPLUS);
		Add_CLP("-userportdevice", "20")
		Add_CLP("-parallel8", "1")
	} else {
		// *no floppy speeder*
		Show_Message("No floppy speeder ROMS found.\nPlease put the roms in the database 'ROMs' directory.");
		Add_CLP("-drive8truedrive");
		Add_CLP("-drive9truedrive");
		Add_CLP("-drive10truedrive");
		Add_CLP("-drive11truedrive");
	}
}

function disableTruedrive() {
	Add_CLP("-virtualdev8");
	Add_CLP("+drive8truedrive");
	Add_CLP("-virtualdev9");
	Add_CLP("+drive9truedrive");
	Add_CLP("-virtualdev10");
	Add_CLP("+drive10truedrive");
	Add_CLP("-virtualdev11");
	Add_CLP("+drive11truedrive");
}


// search for "x64sc"
command = findInPath("x64sc");
var iniSection = "C64SC";

// if command was not found, search for "x64"
if (!command) {
	command = findInPath("x64");
	var iniSection = "C64";
}

// if command was not found, fail with explanatory message
if (!command) {
	Show_Message("Could not find VICE C64 emulator (neither executable 'x64sc' nor 'x64') in operating system path.");
	throw ''; // end script
}

Add_CLP(command);
Add_CLP("-config", emulatorConfigFile);

// print all variables
Dump();

// These are the only known C64 roms with custom characters
// SWE - Swedish, TUR - Turkish, DAN - Danish, SPA - Spanish, GER - German, JAP - Japanese.
if (Value("rom").contains("SWE", "DAN", "TUR", "SPA", "GER", "JAP", "*")) {
	Show_Message("The game specifies a special ROM set '" + Value("rom") + "' to use.\nThe ROM files 'KERNAL.ROM' and 'CHAR.ROM' must be in the folder '" + dbPath + "/ROMs/" + Value("rom") + "' for the game to work.");
	Add_CLP("-kernal", dbPath + "/ROMs/" + Value("rom") + "/KERNAL.ROM");
	Add_CLP("-chargen", dbPath + "/ROMs/" + Value("rom") + "/CHAR.ROM");
}

if (itemType.contains("d64", "t64", "g64", "g41", "tap", "crt", "p00", "prg", "lnx")) {

	// option to turn on/off Warp Mode ("speed increase") at startup
	// Set_INI_Value(emulatorConfigFile, iniSection, "WarpMode", "1");

	// option to start in fullscreen mode ("no border=640x400, border=600x800 or larger");
	// Set_INI_Value(emulatorConfigFile, iniSection, "FullscreenEnabled", "1");
	// Set_INI_Value(emulatorConfigFile, iniSection, "FullscreenWidth", "640");
	// Set_INI_Value(emulatorConfigFile, iniSection, "FullscreenHeight", "400");
	// Set_INI_Value(emulatorConfigFile, iniSection, "FullscreenBitdepth", "32");

	// VICII settings
	// Set_INI_Value(emulatorConfigFile, iniSection, "VICIIDoubleSize", "1");
	// Set_INI_Value(emulatorConfigFile, iniSection, "VICIIScale2x", "1");

	// set path and type of floppy disk drives
	Set_INI_Value(emulatorConfigFile, iniSection, "FSDevice8Dir", itemPath);
	Set_INI_Value(emulatorConfigFile, iniSection, "Drive9Type", "0");
	Set_INI_Value(emulatorConfigFile, iniSection, "Drive10Type", "0");
	Set_INI_Value(emulatorConfigFile, iniSection, "Drive11Type", "0");

	// set to extend tracks on access
	//Set_INI_Value(emulatorConfigFile, iniSection, "Drive8ExtendImagePolicy", "2");
	//Set_INI_Value(emulatorConfigFile, iniSection, "Drive9ExtendImagePolicy", "2");
	//Set_INI_Value(emulatorConfigFile, iniSection, "Drive10ExtendImagePolicy", "2");
	//Set_INI_Value(emulatorConfigFile, iniSection, "Drive11ExtendImagePolicy", "2");

	// autostart or 'manual load' the item image
	if (itemType.contains("crt")) {
		// autostart the cart image
		Add_CLP("-cartcrt", itemPathAndFile);
	} else {
		// insert multiple disks into drives 9, 10 and 11?
		if (itemType.contains("d64", "g64", "g41")) {
			if (Value("maxdrives").contains("2", "3", "4")) {
				if (itemFiles.length > 1) {
					Add_CLP("-9", itemPathsAndFiles[1]);
					Set_INI_Value(emulatorConfigFile, iniSection, "FSDevice9Dir", itemPaths[1]);
					Set_INI_Value(emulatorConfigFile, iniSection, "Drive9Type", "1541");
				}
			}
			if (Value("maxdrives").contains("3", "4")) {
				if (itemFiles.length > 2) {
					Add_CLP("-10", itemPathsAndFiles[2]);
					Set_INI_Value(emulatorConfigFile, iniSection, "FSDevice10Dir", itemPaths[2]);
					Set_INI_Value(emulatorConfigFile, iniSection, "Drive10Type", "1541");
				}
			}
			if (Value("maxdrives").contains("4")) {
				if (itemFiles.length > 3) {
					Add_CLP("-11", itemPathsAndFiles[3]);
					Set_INI_Value(emulatorConfigFile, iniSection, "FSDevice11Dir", itemPaths[3]);
					Set_INI_Value(emulatorConfigFile, iniSection, "Drive11Type", "1541");
				}
			}
		}

		// autostart or 'manual load' the item image
		if (imageName.contains("*")) {
			if (itemVersionComment.contains("*load manually*")) {
				if (itemType.contains("d64", "g64", "g41")) {
					Add_CLP("-autoload", itemPathAndFile + ":" + imageNameNative.replaceAll("\"", "?") + "*");
				} else {
					Add_CLP("-autoload", itemPathAndFile + ":" + imageNameNative.replaceAll("\"", "?"));
				}
				if (Value("keybuf").contains("*")) {
					Add_CLP("-keybuf", Value("keybuf"));
				}
			} else {
				if (itemType.contains("d64", "g64", "g41")) {
					Add_CLP("-autostart", itemPathAndFile + ":" + imageNameNative.replaceAll("\"", "?") + "*");
				} else {
					Add_CLP("-autostart", itemPathAndFile + ":" + imageNameNative.replaceAll("\"", "?"));
				}
			}
		} else {
			if (itemVersionComment.contains("*load manually*")) {
				Add_CLP("-autoload", itemPathAndFile);
			} else {
				Add_CLP("-autostart", itemPathAndFile);
			}
		}
	}

	// PAL or NTSC setup?
	if (isRunsOnPal) {
		Add_CLP("-pal");
	} else {
		Add_CLP("-ntsc");
	}

	// floppy true drive emulation
	if (itemType.contains("d64", "g64", "g41")) {
		// true drive emulation ("vtde=yes" is for VICE only, not CCS64);
		// set GEMUS key "vtde" to overrule database isRunsOnTrueDrive settings
		if (isRunsOnTrueDrive) {
			if (Value("vtde").contains("no")) {
				disableTruedrive();
			} else {
				enableTruedrive();
			}
		} else {
			if (Value("vtde").contains("yes")) {
				enableTruedrive();
			} else {
				disableTruedrive();
			}
		}
	}

	// set the item controls
	// These settings are designed to automatically set the controls for player 1
	// to the same controller, regardless of which PORT the item uses for player 1.
	//
	// 0=None, 1=numpad, 2=custom keyset 1, 3=custom keyset 2, 4=joystick 1, 5=joystick 2
	//
	// Replace * with <DISABLED> to disable the automatic settings for the JoyPorts
	if (itemFile.contains("*")) {
		if (itemControl === JOYPORT2) {

			// set joystick 2
			if (useHardwareJoystick) {
				Add_CLP("-joydev2", "4"); // joystick 1
			} else {
				Add_CLP("-joydev2", "2"); // custom keyset 1
			}

			// set joystick 1
			if (itemPlayersMax > 1) {
				if (useHardwareJoystick) {
					Add_CLP("-joydev1", "5"); // joystick 2
				} else {
					Add_CLP("-joydev1", "3"); // custom keyset 2
				}
			} else {
				Add_CLP("-joydev1", "0"); // no joystick
			}

			Set_INI_Value(emulatorConfigFile, iniSection, "Mouse", "0"); // no mouse
		} else if (itemControl === JOYPORT1) {

			// set joystick 1
			if (useHardwareJoystick) {
				Add_CLP("-joydev1", "4"); // joystick 1
			} else {
				Add_CLP("-joydev1", "2"); // custom keyset 1
			}

			// set joystick 2
			if (itemPlayersMax > 1) {
				if (useHardwareJoystick) {
					Add_CLP("-joydev2", "5"); // joystick 2
				} else {
					Add_CLP("-joydev2", "3"); // custom keyset 2
				}
			} else {
				Add_CLP("-joydev2", "0"); // no joystick
			}

			Set_INI_Value(emulatorConfigFile, iniSection, "Mouse", "0"); // no mouse
		} else if (itemControl === MOUSE) {
			// 1351 Mouse emulation ("the Port cannot be changed or determined via the GUI");
			Set_INI_Value(emulatorConfigFile, iniSection, "Mouse", "1");
			Set_INI_Value(emulatorConfigFile, iniSection, "Mouseport", "1");
			Add_CLP("-joydev1", "0"); // no joystick
			Add_CLP("-joydev2", "0"); // no joystick
		} else if (itemControl === PADDLEPORT1) {
			// Mouse PORT1 setting may give partial PaddlePort1 control
			Set_INI_Value(emulatorConfigFile, iniSection, "Mouse", "1");
			Set_INI_Value(emulatorConfigFile, iniSection, "Mouseport", "1");
			Add_CLP("-joydev1", "0"); // no joystick
			Add_CLP("-joydev2", "0"); // no joystick
		} else if (itemControl === PADDLEPORT2) {
			// Mouse PORT2 setting may give partial PaddlePort2 control
			Set_INI_Value(emulatorConfigFile, iniSection, "Mouse", "1");
			Set_INI_Value(emulatorConfigFile, iniSection, "Mouseport", "2");
			Add_CLP("-joydev1", "0"); // no joystick
			Add_CLP("-joydev2", "0"); // no joystick
		} else { // keyboard
			// Disable Joysticks and Mouse
			Add_CLP("-joydev1", "0");
			Add_CLP("-joydev2", "0");
			Set_INI_Value(emulatorConfigFile, iniSection, "Mouse", "0"); // no mouse
		}
	}

	// REU ("Ram Expansion Unit") = 128,256,512,1024,2048,4086,8192 or 16384 kB
	if (Value("reu").contains("*")) {
		// enable REU and set to reu value
		if (Value("reu").contains("128", "256", "512", "1024", "2048", "4096", "8192", "16384")) {
			Set_INI_Value(emulatorConfigFile, iniSection, "REUsize", Value("reu"));
			Set_INI_Value(emulatorConfigFile, iniSection, "REU", "1");
		} else if (Value("reu").contains(false)) {
			// disable REU if set to off
			Set_INI_Value(emulatorConfigFile, iniSection, "REU", "0");
		} else {
			// set to 512 kB for all other REU values ("on, yes, 500, etc.");
			Set_INI_Value(emulatorConfigFile, iniSection, "REUsize", "512");
			Set_INI_Value(emulatorConfigFile, iniSection, "REU", "1");
		}
	} else {
		// disable REU
		Set_INI_Value(emulatorConfigFile, iniSection, "REU", "0");
	}

	// unsupported item controls
	if (itemControl === LIGHTPEN) {
		Show_Message("This emulator does not have LightPen emulation.\n\nThis item may be uncontrollable/unplayable.");
	} else if (itemControl === KOALAPAD) {
		Show_Message("This emulator does not have KoalaPad emulation.\n\nThis item may be uncontrollable/unplayable.");
	} else if (itemControl === LIGHTGUN) {
		Show_Message("This emulator does not have LightGun emulation.\n\nThis item may be uncontrollable/unplayable.");
	} else if (itemControl === PADDLEPORT1) {
		Show_Message("This emulator does not have Paddle emulation.\n\nMouse Port1 setting may allow partial control via the mouse.\n\nThis item may still be uncontrollable/unplayable.");
	} else if (itemControl === PADDLEPORT2) {
		Show_Message("This emulator does not have Paddle emulation.\n\nMouse Port2 setting may allow partial control via the mouse.\n\nThis item may still be uncontrollable/unplayable.");
	}

	// give the user a warning message?
	if (itemVersionComment.contains("*use C64S*", "*use CCS*", "*use real C64*")) {
		Show_Message("This item may not work properly with this emulator.\n\nSee Version Comment for more info.");
	}
	if (itemVersionComment.contains("*not 100%*", "*not working*", "*doesn't work*")) {
		Show_Message("This item may not work properly.");
	}
	if (itemVersionComment.contains("*PET Emulator*")) {
		Show_Message("Use PET Emulator to run this item.");
	}
	if (itemVersionComment.contains("*ATTN:*", "*NOTE:*", "*ATTN!:*", "*NOTE!:*")) {
		// point out that critical info is in the version comment
		Show_Message("Read this item's Version Comments for very important information/instructions.");
	}
	if (itemVersionComment.contains("*VIC-20*", "*VIC20*", "*VC-20*", "*VC20*")) {
		if (itemVersionComment.contains("*emulator*")) {
			Show_Message("Use VIC-20 column Emulator to run this item properly.");
		}
	}

	// show message for the user
	if (Value("msg").contains("*")) {
		Show_Message(Value("msg"));
	}

	//println(itemPathsAndFiles);
	//println(itemPathsAndFiles.length);
	if (itemPathsAndFiles.length > 1) {
		// initialize fliplist
		var fliplist = File(dbPath + separator + "Scripts" + separator + "vice.vfl");

		// create new fliplist
		if (fliplist.exists()) {
			fliplist.remove();
		}
		fliplist.createNewFile();

		// write fliplist path and name to ini file
		Set_INI_Value(emulatorConfigFile, iniSection, "FliplistName", fliplist.getAbsolutePath());

		// write to fliplist
		Set_CFG(fliplist, "# Vice fliplist file" + lineSeparator);
		Set_CFG(fliplist, "UNIT 8");

		for (var i = 0; i < itemPathsAndFiles.length; i++) {
			Set_CFG(fliplist, itemPathsAndFiles[i] + "");
		}

		if (Value("maxdrives").contains("2", "3", "4")) {
			if (itemFiles.length > 1) {
				Set_CFG(fliplist, "UNIT 9");
				for (var i = 0; i < itemPathsAndFiles.length; i++) {
					Set_CFG(fliplist, itemPathsAndFiles[i] + "");
				}
			}
		}
		if (Value("maxdrives").contains("3", "4")) {
			if (itemFiles.length > 2) {
				Set_CFG(fliplist, "UNIT 10");
				for (var i = 0; i < itemPathsAndFiles.length; i++) {
					Set_CFG(fliplist, itemPathsAndFiles[i] + "");
				}
			}
		}
		if (Value("maxdrives").contains("4")) {
			if (itemFiles.length > 3) {
				Set_CFG(fliplist, "UNIT 11");
				for (var i = 0; i < itemPathsAndFiles.length; i++) {
					Set_CFG(fliplist, itemPathsAndFiles[i] + "");
				}
			}
		}

	}

	// run the emulator
	Run_Emulator();

} else {

	// invalid item file type
	Show_Message(NOT_SUPPORTED + "\n\nSupported types: D64, T64, G64/G41, TAP, CRT, P00, PRG, LNX");

}



