/**
 *                         filter type decoding support
 *                         ----------------------------
 *  begin                : Sun Mar 11 2001
 *  copyright            : (C) 2001 by Simon White
 *  email                : s_a_white@email.com
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 * @author Ken Hndel
 *
 */
package libsidutils;

import java.util.NoSuchElementException;
import java.util.StringTokenizer;

import resid_builder.sid_filter_t;

public class SidFilter {
	public boolean m_status;
	protected String m_errorString;
	public sid_filter_t m_filter = new sid_filter_t();

	protected void readType1(IniReader ini, String heading) {

		int propertyInt = ini.getPropertyInt(heading, "DistortionLowThreshold",
				-1);
		if (propertyInt != -1)
			m_filter.Lthreshold = propertyInt;
		propertyInt = ini.getPropertyInt(heading, "DistortionLowSteepness", -1);
		if (propertyInt != -1)
			m_filter.Lsteepness = propertyInt;
		propertyInt = ini.getPropertyInt(heading, "DistortionLowLp", -1);
		if (propertyInt != -1)
			m_filter.Llp = propertyInt;
		propertyInt = ini.getPropertyInt(heading, "DistortionLowBp", -1);
		if (propertyInt != -1)
			m_filter.Lbp = propertyInt;
		propertyInt = ini.getPropertyInt(heading, "DistortionLowHp", -1);
		if (propertyInt != -1)
			m_filter.Lhp = propertyInt;
		propertyInt = ini
				.getPropertyInt(heading, "DistortionHighThreshold", -1);
		if (propertyInt != -1)
			m_filter.Hthreshold = propertyInt;
		propertyInt = ini
				.getPropertyInt(heading, "DistortionHighSteepness", -1);
		if (propertyInt != -1)
			m_filter.Hsteepness = propertyInt;
		propertyInt = ini.getPropertyInt(heading, "DistortionHighLp", -1);
		if (propertyInt != -1)
			m_filter.Hlp = propertyInt;
		propertyInt = ini.getPropertyInt(heading, "DistortionHighBp", -1);
		if (propertyInt != -1)
			m_filter.Hbp = propertyInt;
		propertyInt = ini.getPropertyInt(heading, "DistortionHighHp", -1);
		if (propertyInt != -1)
			m_filter.Hhp = propertyInt;

		int points = ini.getPropertyInt(heading, "points", -1);

		if (points < 0) {
			clear();
			m_errorString = "SID Filter: Invalid Type 1 filter definition";
			m_status = false;
			return;
		}
		// Make sure there are enough filter points
		if ((points < 2) || (points > 0x800)) {
			clear();
			m_errorString = "SID Filter: Invalid Type 1 filter definition";
			m_status = false;
			return;
		}
		m_filter.points = points;

		{
			int reg = -1, fc = -1;
			for (int i = 0; i < m_filter.points; i++) {
				// First lets read the SID cutoff register value
				String val = ini.getPropertyString(heading, String.format(
						"point%d", i + 1), "0,0");
				StringTokenizer tok = new StringTokenizer(val, ",", false);
				try {
					reg = new Integer(tok.nextToken()).intValue();
					if (reg < 0) {
						throw new NoSuchElementException();
					}
				} catch (NoSuchElementException e) {
					clear();
					m_errorString = "SID Filter: Invalid Type 1 filter definition";
					m_status = false;
					return;
				}
				try {
					fc = new Integer(tok.nextToken()).intValue();
					if (fc < 0) {
						throw new NoSuchElementException();
					}
				} catch (NoSuchElementException e) {
					clear();
					m_errorString = "SID Filter: Invalid Type 1 filter definition";
					m_status = false;
					return;
				}

				// Got valid reg/fc
				m_filter.cutoff[i][0] = reg;
				m_filter.cutoff[i][1] = fc;
			}
		}
	}

	protected void readType2(IniReader ini, String heading) {
		double fs, fm, ft;

		// Read filter parameters
		fs = ini.getPropertyDouble(heading, "fs", -1);
		if (fs < 0) {
			clear();
			m_errorString = "SID Filter: Invalid Type 2 filter definition";
			m_status = false;
			return;
		}
		fm = ini.getPropertyDouble(heading, "fm", -1);
		if (fm < 0) {
			clear();
			m_errorString = "SID Filter: Invalid Type 2 filter definition";
			m_status = false;
			return;
		}
		ft = ini.getPropertyDouble(heading, "ft", -1);
		if (ft < 0) {
			clear();
			m_errorString = "SID Filter: Invalid Type 2 filter definition";
			m_status = false;
			return;
		}

		// Calculate the filter
		calcType2(fs, fm, ft);
	}

	protected void clear() {
		m_filter.points = 0;
		m_filter.Lthreshold = 0;
		m_status = false;
		m_errorString = "SID Filter: No filter loaded";
	}

	public SidFilter() {
	}

	public void read(final String filename) {
		try {
			IniReader ini = new IniReader(filename);
			read(ini, "Filter");
		} catch (Exception e) {
			// Illegal ini file
			m_status = false;
			m_errorString = "SID Filter: Unable to open filter file";

		}

	}

	public void read(IniReader ini, final String heading) {
		int type = 1;

		clear();
		m_status = true;

		type = ini.getPropertyInt(heading, "type", 1);
		switch (type) {
		case 1:
			readType1(ini, heading);
			break;

		case 2:
			readType2(ini, heading);
			break;

		default:
			m_status = false;
			m_errorString = "SID Filter: Invalid filter type";
			break;
		}
	}

	public void calcType2(double fs, double fm, double ft) {
		double fcMax = 1.0;
		double fcMin = 0.01;
		double fc;

		// Definition from reSID
		m_filter.points = 0x100;

		// Create filter
		for (int i = 0; i < 0x100; i++) {
			int rk = i << 3;
			m_filter.cutoff[i][0] = rk;
			fc = Math.exp((double) rk / 0x800 * Math.log(fs)) / fm + ft;
			if (fc < fcMin)
				fc = fcMin;
			if (fc > fcMax)
				fc = fcMax;
			m_filter.cutoff[i][1] = (int) (fc * 4100);
		}
	}

	public final String error() {
		return m_errorString;
	}

	public final sid_filter_t provide() {
		if (!m_status)
			return null;
		return m_filter;
	}

	public boolean bool() {
		return m_status;
	}

}
