package sidplay;

import static libsidplay.common.ISID2Types.sid2_model_t;
import static libsidplay.common.ISID2Types.SID2_DEFAULT_OPTIMISATION;
import static libsidplay.common.ISID2Types.SID2_DEFAULT_PRECISION;
import static libsidplay.common.ISID2Types.SID2_DEFAULT_SAMPLING_FREQ;
import static libsidplay.common.ISID2Types.SID2_MAX_OPTIMISATION;
import static libsidplay.common.ISID2Types.SID2_MAX_PRECISION;

import java.io.IOException;
import java.io.PrintStream;
import java.util.logging.Level;
import java.util.logging.Logger;

import libsidplay.SIDPlay2;
import libsidplay.common.Event;
import libsidplay.common.IEventContext;
import libsidplay.common.ISID2Types.sid2_clock_t;
import libsidplay.common.ISID2Types.sid2_config_t;
import libsidplay.common.ISID2Types.sid2_env_t;
import libsidplay.common.ISID2Types.sid2_info_t;
import libsidplay.common.ISID2Types.sid2_model_t;
import libsidplay.common.ISID2Types.sid2_playback_t;
import libsidplay.common.ISID2Types.sid2_player_t;
import libsidplay.components.sidtune.SidTune;
import libsidplay.components.sidtune.SidTuneInfo;
import libsidutils.SidDatabase;
import libsidutils.SidFilter;
import libsidutils.SidTuneMod;
import resid_builder.ReSIDBuilder;
import resid_builder.resid.SID;
import sidplay.audio.AudioBase;
import sidplay.audio.AudioConfig;
import sidplay.audio.AudioDriver;
import sidplay.audio.Audio_Null;
import sidplay.audio.WavFile;

public class ConsolePlayer extends Event {

	private static final Logger PLAYER = Logger.getLogger(ConsolePlayer.class
			.getName());

	// Previous song select timeout (3 secs)
	public static final int SID2_PREV_SONG_TIMEOUT = 4;

	enum player_colour_t {
		black, red, green, yellow, blue, magenta, cyan, white
	};

	enum player_table_t {
		tableStart, tableMiddle, tableSeperator, tableEnd
	};

	public static final int playerError = 0;
	public static final int playerRunning = 1;
	public static final int playerPaused = 2;
	public static final int playerStopped = 3;
	public static final int playerRestart = 4;
	public static final int playerExit = 5;
	public static final int playerFast = 128;
	public static final int playerFastRestart = playerRestart | playerFast;
	public static final int playerFastExit = playerExit | playerFast;

	enum SIDEMUS {
		/*
		 * Same as EMU_DEFAULT except no soundcard. Still allows wav generation
		 */
		EMU_NONE,
		/* The following require a soundcard */
		EMU_DEFAULT, EMU_RESID, EMU_SIDPLAY1,
		/* The following should disable the soundcard */
		EMU_HARDSID, EMU_SIDSTATION, EMU_COMMODORE, EMU_SIDSYN, EMU_END
	};

	enum OUTPUTS {
		/* Define possible output sources */
		OUT_NULL,
		/* Hardware */
		OUT_SOUNDCARD,
		/* File creation support */
		OUT_WAV, OUT_AU, OUT_END
	};

	/**
	 * Error and status message numbers.
	 */
	enum ERR {
		ERR_SYNTAX, ERR_NOT_ENOUGH_MEMORY, ERR_SIGHANDLER, ERR_FILE_OPEN
	};

	private final String m_name;

	private SIDPlay2 m_engine = new SIDPlay2();

	private sid2_config_t m_engCfg;

	SidTuneMod m_tune;

	private int m_state;

	public int state() {
		return m_state;
	}

	private String m_outfile;

	private IEventContext m_context;

	private String m_filename;

	IniConfig m_iniCfg = new IniConfig();
	SidDatabase m_database = new SidDatabase();

	//
	// Display parameters
	//

	private short /* uint_least8_t */m_quietLevel;

	private long /* uint_least32_t */m_crc;

	private boolean m_cpudebug;

	private class m_filter_t {
		SidFilter definition = new SidFilter();
		boolean enabled;
	};

	private m_filter_t m_filter = new m_filter_t();

	private class m_driver_t {
		OUTPUTS output = OUTPUTS.OUT_NULL; // Selected output type
		SIDEMUS sid = SIDEMUS.EMU_NONE; // Sid emulation
		boolean file; // File based driver
		AudioConfig cfg = new AudioConfig();
		public AudioBase selected; // Selected Output Driver
		public AudioBase device; // HW/File Driver
		Audio_Null nullAudio = new Audio_Null(); // Used for everything
	};

	private m_driver_t m_driver = new m_driver_t();

	private class m_timer_t { // secs
		long /* uint_least32_t */start;
		long /* uint_least32_t */current;
		long /* uint_least32_t */stop;
		long /* uint_least32_t */length;
		boolean valid;
	};

	private m_timer_t m_timer = new m_timer_t();

	private class m_track_t {
		int /* uint_least16_t */first;
		int /* uint_least16_t */selected;
		int /* uint_least16_t */songs;
		boolean loop;
		boolean single;
	};

	private m_track_t m_track = new m_track_t();

	public int getCurrentSong() {
		return m_track.selected;
	}

	public int getSongCount() {
		return m_track.songs;
	}

	private class m_speed_t {
		short /* uint_least8_t */current;
		short /* uint_least8_t */max;
	};

	private m_speed_t m_speed = new m_speed_t();

	private final String RESID_ID = "ReSID";

	private int m_verboseLevel;

	public ConsolePlayer(final String name) {
		super("External Timer\n");
		m_name = (name);
		m_tune = (null);
		m_state = (playerStopped);
		m_outfile = (null);
		m_context = (null);
		m_quietLevel = (0);
		m_verboseLevel = (0);
		m_crc = (0);
		m_cpudebug = (true);
		// Other defaults
		m_filename = "";
		m_filter.enabled = true;
		// m_driver.device = null;
		m_timer.start = 0;
		m_timer.length = 0; // FOREVER
		m_timer.valid = false;
		m_track.first = 0;
		m_track.selected = 0;
		m_track.loop = false;
		m_track.single = false;
		m_speed.current = 1;
		m_speed.max = 32;

		// Read default configuration
		m_iniCfg.read();
		m_engCfg = m_engine.config();

		{ // Load ini settings
			IniConfig.audio_section audio = m_iniCfg.audio();
			IniConfig.emulation_section emulation = m_iniCfg.emulation();

			// INI Configuration Settings
			m_engCfg.clockForced = emulation.clockForced;
			m_engCfg.clockSpeed = emulation.clockSpeed;
			m_engCfg.clockDefault = sid2_clock_t.SID2_CLOCK_PAL;
			m_engCfg.frequency = audio.frequency;
			m_engCfg.optimisation = emulation.optimiseLevel;
			m_engCfg.playback = audio.playback;
			m_engCfg.precision = audio.precision;
			m_engCfg.sidModel = emulation.sidModel;
			m_engCfg.sidDefault = sid2_model_t.SID2_MOS6581;
			m_engCfg.sidSamples = emulation.sidSamples;
			m_filter.enabled = emulation.filter;
		}

		// Copy default setting to audio configuration
		m_driver.cfg.channels = 1; // Mono
		if (m_engCfg.playback == sid2_playback_t.sid2_stereo)
			m_driver.cfg.channels = 2;
		m_driver.cfg.frequency = m_engCfg.frequency;
		m_driver.cfg.precision = m_engCfg.precision;

		createOutput(OUTPUTS.OUT_NULL, null);
		createSidEmu(SIDEMUS.EMU_NONE);
	}

	/**
	 * Create the output object to process sound buffer
	 * 
	 * @param driver
	 * @param tuneInfo
	 * @return
	 */
	public boolean createOutput(OUTPUTS driver, final SidTuneInfo tuneInfo) {
		StringBuffer name = null;
		String title = m_outfile;

		// Remove old audio driver
		m_driver.nullAudio.close();
		m_driver.selected = m_driver.nullAudio;
		if (m_driver.device != null) {
			if (m_driver.device != m_driver.nullAudio)
				m_driver.device.close();
		}

		// Create audio driver
		switch (driver) {
		case OUT_NULL:
			m_driver.device = m_driver.nullAudio;
			title = "";
			break;

		case OUT_SOUNDCARD:
			m_driver.device = new AudioDriver();

			break;

		case OUT_WAV:
			m_driver.device = new WavFile();
			break;

		default:
			break;
		}

		// Generate a name for the wav file
		if (title == null) {
			int /* size_t */length, i;
			title = tuneInfo.dataFileName;
			length = (title.length());
			i = length;
			while (i > 0) {
				if (title.charAt(--i) == '.')
					break;
			}
			if (i == 0)
				i = length;

			name = new StringBuffer();

			name.append(title.substring(0, i));
			// Prevent extension ".sid.wav"

			// Change name based on subtune
			if (tuneInfo.songs > 1)
				name.append(String.format("[%d]", tuneInfo.currentSong));
			name.append(m_driver.device.extension());
			title = name.toString();
		}

		// Configure with user settings
		m_driver.cfg.frequency = m_engCfg.frequency;
		m_driver.cfg.precision = m_engCfg.precision;
		m_driver.cfg.channels = 1; // Mono
		m_driver.cfg.bufSize = 0; // Recalculate
		if (m_engCfg.playback == sid2_playback_t.sid2_stereo)
			m_driver.cfg.channels = 2;

		{ // Open the hardware
			m_driver.device.open(m_driver.cfg, title);

			// Can't open the same driver twice
			if (driver != OUTPUTS.OUT_NULL) {
				m_driver.nullAudio.open(m_driver.cfg, title);
			}
		}

		// See what we got
		m_engCfg.frequency = m_driver.cfg.frequency;
		m_engCfg.precision = m_driver.cfg.precision;
		switch (m_driver.cfg.channels) {
		case 1:
			if (m_engCfg.playback == sid2_playback_t.sid2_stereo)
				m_engCfg.playback = sid2_playback_t.sid2_mono;
			break;
		case 2:
			if (m_engCfg.playback != sid2_playback_t.sid2_stereo)
				m_engCfg.playback = sid2_playback_t.sid2_stereo;
			break;
		default:
			System.err.println(m_name + ": ERROR: " + m_driver.cfg.channels
					+ " audio channels not supported");
			return false;
		}
		return true;
	}

	/**
	 * Create the sid emulation
	 * 
	 * @param emu
	 * @return
	 */
	public boolean createSidEmu(SIDEMUS emu) {
		// Remove old driver and emulation
		if (m_engCfg.sidEmulation != null) {
			m_engCfg.sidEmulation = null;
			m_engine.config(m_engCfg);
		}

		// Now setup the sid emulation
		switch (emu) {
		case EMU_RESID: {
			ReSIDBuilder rs = new ReSIDBuilder(RESID_ID);
			if (rs.bool()) {
				m_engCfg.sidEmulation = rs;
				if (!rs.bool()) {
					displayError(m_engCfg.sidEmulation.error());
					m_engCfg.sidEmulation = null;
					return false;
				}

				// Setup the emulation
				rs.create((m_engine.info()).maxsids);
				if (!rs.bool()) {
					displayError(m_engCfg.sidEmulation.error());
					m_engCfg.sidEmulation = null;
					return false;
				}
				rs.filter(m_filter.enabled);
				if (!rs.bool()) {
					displayError(m_engCfg.sidEmulation.error());
					m_engCfg.sidEmulation = null;
					return false;
				}
				rs.sampling(m_driver.cfg.frequency);
				if (!rs.bool()) {
					displayError(m_engCfg.sidEmulation.error());
					m_engCfg.sidEmulation = null;
					return false;
				}
				if (m_filter.enabled && m_filter.definition != null) {
					// Setup filter
					rs.filter(m_filter.definition.provide());
					if (!rs.bool()) {
						displayError(m_engCfg.sidEmulation.error());
						m_engCfg.sidEmulation = null;
						return false;
					}
				}
			}
			break;
		}

			// #ifdef HAVE_HARDSID_BUILDER
			// case EMU_HARDSID:
			// {
			// #ifdef HAVE_EXCEPTIONS
			// HardSIDBuilder *hs = new(std::nothrow) HardSIDBuilder( HARDSID_ID
			// );
			// #else
			// HardSIDBuilder *hs = new HardSIDBuilder( HARDSID_ID );
			// #endif
			// if (hs)
			// {
			// m_engCfg.sidEmulation = hs;
			// if (!*hs) goto createSidEmu_error;
			//
			// // Setup the emulation
			// hs->create ((m_engine.info ()).maxsids);
			// if (!*hs) goto createSidEmu_error;
			// hs->filter (m_filter.enabled);
			// if (!*hs) goto createSidEmu_error;
			// }
			// break;
			// }
			// #endif // HAVE_HARDSID_BUILDER

		default:
			// Emulation Not yet handled
			// This default case results in the default
			// emulation
			break;
		}

		if (m_engCfg.sidEmulation == null) {
			if (emu != SIDEMUS.EMU_NONE && emu != SIDEMUS.EMU_DEFAULT) {
				// No sid emulation?
				displayError(m_name, ERR.ERR_NOT_ENOUGH_MEMORY);
				return false;
			}
		}
		return true;
	}

	public boolean open() {
		final SidTuneInfo tuneInfo;

		if ((m_state & ~playerFast) == playerRestart) {
			if (m_quietLevel < 2)
				System.err.println();
			if ((m_state & playerFast) != 0)
				m_driver.selected.reset();
			m_state = playerStopped;
		}

		// Select the required song
		m_track.selected = m_tune.selectSong(m_track.selected);
		if (m_engine.load(m_tune) < 0) {
			displayError(m_engine.error());
			return false;
		}

		// Get tune details
		tuneInfo = (m_engine.info()).tuneInfo;
		if (!m_track.single)
			m_track.songs = tuneInfo.songs;
		if (!createOutput(m_driver.output, tuneInfo))
			return false;
		if (!createSidEmu(m_driver.sid))
			return false;

		// Configure engine with settings
		if (m_engine.config(m_engCfg) < 0) {
			// Config failed
			displayError(m_engine.error());
			return false;
		}

		// Start the player. Do this by fast
		// forwarding to the start position
		m_driver.selected = m_driver.nullAudio;
		m_speed.current = m_speed.max;
		m_engine.fastForward(100 * m_speed.current);

		// As yet we don't have a required songlength
		// so try the songlength database
		if (!m_timer.valid) {
			int /* int_least32_t */length = m_database.length(m_tune);
			if (length > 0)
				m_timer.length = length;
		}

		// Set up the play timer
		m_context = (m_engine.info()).eventContext;
		m_timer.stop = 0;
		m_timer.stop += m_timer.length;

		if (m_timer.valid) {
			// Length relative to start
			m_timer.stop += m_timer.start;
		} else {
			// Check to make start time dosen't exceed end
			if ((m_timer.stop & ((m_timer.start >= m_timer.stop) ? 1 : 0)) != 0) {
				displayError("ERROR: Start time exceeds song length!");
				return false;
			}
		}

		m_timer.current = ~0;
		m_state = playerRunning;

		// Update display
		menu();
		event();
		return true;
	}

	public void close() {
		m_engine.stop();
		if (m_state == playerExit) {
			// Natural finish
			emuflush();
			if (m_driver.file)
				System.err.print((char) 7);
			// Bell
		} else
			// Destroy buffers
			m_driver.selected.reset();

		// Shutdown drivers, etc
		createOutput(OUTPUTS.OUT_NULL, null);
		createSidEmu(SIDEMUS.EMU_NONE);
		m_engine.load(null);
		m_engine.config(m_engCfg);
	}

	/**
	 * Flush any hardware sid fifos so all music is played
	 */
	void emuflush() {
		// switch (m_driver.sid)
		// {
		// #ifdef HAVE_HARDSID_BUILDER
		// case EMU_HARDSID:
		// ((HardSIDBuilder *)m_engCfg.sidEmulation)->flush ();
		// break;
		// #endif // HAVE_HARDSID_BUILDER
		// default:
		// break;
		// }
	}

	/**
	 * Out play loop to be externally called
	 */
	public boolean play() {
		byte[] soundBuffer = m_driver.selected.buffer();
		int /* uint_least32_t */length = m_driver.cfg.bufSize;

		short[] buffer = new short[length];

		if (m_state == playerRunning) {
			// Fill buffer
			long /* uint_least32_t */ret;
			ret = m_engine.play(buffer, length);
			for (int i = 0; i < buffer.length; i++) {
				soundBuffer[i] = (byte) (buffer[i]);
			}

			if (PLAYER.isLoggable(Level.FINE)) {
				for (int i = 0; i < soundBuffer.length; i += 16) {
					StringBuffer result = new StringBuffer();
					for (int j = 0; j < 16 && i + j < soundBuffer.length; j++) {
						result.append(String.format("0x%02x ", soundBuffer[i
								+ j]));
					}
					result.append("\n");
					PLAYER.fine(result.toString());
				}
			}
			if (ret < length) {
				if (m_engine.state() != sid2_player_t.sid2_stopped) {
					m_state = playerError;
					return false;
				}
				return false;
			}
		}

		switch (m_state) {
		case playerRunning:
			m_driver.selected.write();
			// Deliberate run on
		case playerPaused:
			// Check for a keypress (approx 250ms rate, but really depends
			// on music buffer sizes). Don't do this for high quiet levels
			// as chances are we are under remote control.
			// wait for <Enter> to be pressed
			// then finish
			try {
				if ((m_quietLevel < 2) && System.in.available() != 0)
					decodeKeys();
			} catch (IOException e) {
				e.printStackTrace();
			}
			return true;
		default:
			if (m_quietLevel < 2)
				System.err.println();
			if (m_crc != 0) {
				// final SidTuneInfo tuneInfo = (m_engine.info()).tuneInfo;
				// System.err.println(m_engine.info().sid2crc + " : " +
				// m_filename
				// + " - song " + tuneInfo.currentSong + "/"
				// + tuneInfo.songs);
			}
			m_engine.stop();
			break;
		}
		return false;
	}

	private void decodeKeys() {
		try {
			int key = System.in.read();
			switch (key) {
			case 'h':
				m_state = playerFastRestart;
				m_track.selected = 1;
				break;

			case 'e':
				m_state = playerFastRestart;
				m_track.selected = m_track.songs;
				break;

			case '>':
				nextSong();
				break;

			case '<':
				previousSong();
				break;

			case '.':
				fastForward();
				break;

			case ',':
				normalSpeed();
				break;

			case 'p':
				pause();
				break;

			case 'q':
				quit();
				break;

			case 'd':
				SID.ANTTI_LANKILA_PATCH ^= true;
				System.out.println(String.format("SID chip distortion simulation: %s",
						(SID.ANTTI_LANKILA_PATCH ? "on" : "off")));
				break;

			default:
				break;
			}
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	public void pause() {
		if (m_state == playerPaused) {
			m_state = playerRunning;
		} else {
			m_state = playerPaused;
			m_driver.selected.pause();
		}
	}

	public void nextSong() {
		m_state = playerFastRestart;
		if (!m_track.single) {
			m_track.selected++;
			if (m_track.selected > m_track.songs)
				m_track.selected = 1;
		}
	}

	public void previousSong() {
		m_state = playerFastRestart;
		if (!m_track.single) { // Only select previous song if less
			// than timeout
			// else restart current song
			if ((m_engine.time() / m_engine.timebase()) < SID2_PREV_SONG_TIMEOUT) {
				m_track.selected--;
				if (m_track.selected < 1)
					m_track.selected = m_track.songs;
			}
		}
	}

	public void fastForward() {
		m_speed.current *= 2;
		if (m_speed.current > m_speed.max)
			m_speed.current = m_speed.max;
		m_engine.fastForward(100 * m_speed.current);
	}

	public void normalSpeed() {
		m_speed.current = 1;
		m_engine.fastForward(100);
	}

	public void quit() {
		m_state = playerFastExit;
	}

	/**
	 * External Timer Event
	 */
	public void event() {
		long /* uint_least32_t */seconds = m_engine.time()
				/ m_engine.timebase();
		if (m_quietLevel == 0) {
			// System.err.println(((seconds / 60) % 100) + ':' + (seconds %
			// 60));
		}

		if (seconds != m_timer.current) {
			m_timer.current = seconds;

			// Handle exiting on crc completion
			if (m_crc != 0 && (m_crc == (m_engine.info()).sid2crcCount))
				m_timer.stop = seconds;

			if (seconds == m_timer.start) {
				// Switch audio drivers.
				m_driver.selected = m_driver.device;
				for (int i = 0; i < m_driver.selected.buffer().length; i++) {
					m_driver.selected.buffer()[i] = 0;
				}
				normalSpeed();
				if (m_cpudebug)
					m_engine.debug(true);
			} else if (m_timer.stop != 0 && (seconds == m_timer.stop)) {
				m_state = playerExit;
				for (;;) {
					if (m_track.single)
						return;
					// Move to next track
					m_track.selected++;
					if (m_track.selected > m_track.songs)
						m_track.selected = 1;
					if (m_track.selected == m_track.first)
						return;
					m_state = playerRestart;
					break;
				}
				if (m_track.loop)
					m_state = playerRestart;
			}
		}

		// Units in C64 clock cycles
		m_context.schedule(this, 900000, event_phase_t.EVENT_CLOCK_PHI1);
	}

	public void displayError(final String error) {
		System.err.println(m_name + ": " + error);
	}

	public void displayError(final String arg0, ERR /* uint */num) {
		System.err.println(arg0 + ": ");

		if (num == ERR.ERR_SYNTAX) {
			System.err.println("command line syntax error" + "\n" + "Try `"
					+ arg0 + " --help' for more information.");
		} else

		if (num == ERR.ERR_NOT_ENOUGH_MEMORY) {
			System.err.println("ERROR: Not enough memory.");
		} else

		if (num == ERR.ERR_SIGHANDLER) {
			System.err.println("ERROR: Could not install signal handler.");
		} else

		if (num == ERR.ERR_FILE_OPEN) {
			System.err.println("ERROR: Could not open file for binary input.");
		}
	}

	/**
	 * Convert time from integer
	 * 
	 * @param str
	 * @param time
	 * @return
	 */
	long /* uint_least32_t */parseTime(final String str,
			long /* uint_least32_t & */time) {
		int sep;
		long /* uint_least32_t */_time;

		// Check for empty string
		if (str.length() == 0)
			return -1;

		sep = str.lastIndexOf(':');
		if (sep == -1) {
			// User gave seconds
			_time = new Integer(str);
		} else {
			// Read in MM:SS format
			int val;
			val = new Integer(str.substring(0, sep)).intValue();
			if (val < 0 || val > 99)
				return -1;
			_time = (long /* uint_least32_t */) val * 60;
			val = new Integer(str.substring(sep + 1)).intValue();
			if (val < 0 || val > 59)
				return -1;
			_time += (long /* uint_least32_t */) val;
		}

		return _time;
	}

	/**
	 * Parse command line arguments
	 * 
	 * @param argv
	 * @return
	 */
	public int args(String[] argv) {
		int infile = 0;
		int i = 0;
		boolean err = false;

		if (argv.length == 0) // at least one argument required
		{
			displayArgs(null);
			return -1;
		}

		// default arg options
		m_driver.output = OUTPUTS.OUT_SOUNDCARD;
		m_driver.file = false;
		m_driver.sid = SIDEMUS.EMU_RESID;

		// parse command line arguments
		while ((i < argv.length)) {
			if ((argv[i].charAt(0) == '-') && (argv[i].length() > 1)) {
				// help options
				if ((argv[i].charAt(1) == 'h') || argv[i].equals("--help")) {
					displayArgs(null);
					return 0;
				} else if (argv[i].equals("--help-debug")) {
					displayDebugArgs();
					return 0;
				}

				else if (argv[i].charAt(1) == 'b') {
					long time = parseTime(argv[i].substring(2), m_timer.start);
					if (time == -1)
						err = true;
					m_timer.start = time;
				} else if (argv[i].equals("-fd")) {
					// Override sidTune and enable the second sid
					m_engCfg.forceDualSids = true;
				} else if (argv[i].equals("-fs")) {
					// Force samples through soundcard instead of SID
					m_engCfg.sidSamples = false;
				} else if (argv[i].startsWith("-f")) {
					if (argv[i].length() == 2)
						err = true;
					m_engCfg.frequency = new Integer(argv[i].substring(2));
				}

				// Player Mode (Environment) Options ----------
				else if (argv[i].equals("-mb")) {
					// Bankswitching
					m_engCfg.environment = sid2_env_t.sid2_envBS;
				} else if (argv[i].equals("-mr")) {
					// Real C64
					m_engCfg.environment = sid2_env_t.sid2_envR;
				} else if (argv[i].equals("-mt")) {
					// Transparent Rom
					m_engCfg.environment = sid2_env_t.sid2_envTP;
				} else if (argv[i].equals("-m")) {
					// PlaySID
					m_engCfg.environment = sid2_env_t.sid2_envPS;
				}

				// New/No filter options
				else if (argv[i].startsWith("-nf")) {
					if (argv[i].length() == 3)
						m_filter.enabled = false;
					else {
						// New filter
						// This line will open an existing file
						m_filter.enabled = true;
						m_filter.definition.read(argv[i].substring(3));
						if (!m_filter.definition.bool()) {
							displayError(m_filter.definition.error());
							return -1;
						}
					}
				}

				// Newer sid (8580)
				else if (argv[i].startsWith("-ns")) {
					switch (argv[i].charAt(3)) {
					case '1':
						m_engCfg.sidModel = sid2_model_t.SID2_MOS8580;
						break;
					// No new sid so use old one (6581)
					case '0':
						m_engCfg.sidModel = sid2_model_t.SID2_MOS6581;
						break;
					default:
						err = true;
					}
				}

				// Track options
				else if (argv[i].startsWith("-nols")) {
					m_track.loop = true;
					m_track.single = true;
					m_track.first = new Integer(argv[i].substring(4));
				} else if (argv[i].startsWith("-ol")) {
					m_track.loop = true;
					m_track.first = new Integer(argv[i].substring(3));
				} else if (argv[i].startsWith("-os")) {
					m_track.single = true;
					m_track.first = new Integer(argv[i].substring(3));
				} else if (argv[i].startsWith("-o")) {
					// User forgot track number ?
					if (argv[i].length() == 2)
						err = true;
					m_track.first = new Integer(argv[i].substring(2));
				}

				else if (argv[i].startsWith("-O")) {
					if (argv[i].length() != 2)
						m_engCfg.optimisation = (byte) new Integer(argv[i]
								.substring(2)).intValue();
				}

				else if (argv[i].startsWith("-p")) {
					// User forgot precision
					if (argv[i].length() == 2)
						err = true;
					{
						short /* uint_least8_t */precision = new Short(argv[i]
								.substring(2)).shortValue();
						if (precision <= 8)
							precision = 8;
						else if (precision <= 16)
							precision = 16;
						else
							precision = 24;

						if (precision > SID2_MAX_PRECISION)
							precision = SID2_MAX_PRECISION;
						m_engCfg.precision = precision;
					}
				}

				else if (argv[i].startsWith("-q")) {
					if (argv[i].length() == 2)
						m_quietLevel = 1;
					else
						m_quietLevel = new Short(argv[i].substring(2))
								.shortValue();
				}

				// Stereo Options
				else if (argv[i].equals("-sl")) {
					// Left Channel
					m_engCfg.playback = sid2_playback_t.sid2_left;
				} else if (argv[i].equals("-sr")) {
					// Right Channel
					m_engCfg.playback = sid2_playback_t.sid2_right;
				} else if (argv[i].equals("-s")) {
					// Stereo Playback
					m_engCfg.playback = sid2_playback_t.sid2_stereo;
				}

				else if (argv[i].startsWith("-t")) {
					long time = parseTime(argv[i].substring(2), m_timer.length);
					if (time == -1)
						err = true;
					m_timer.length = time;
					m_timer.valid = true;
				}

				// Video/Verbose Options
				else if (argv[i].equals("-vnf")) {
					m_engCfg.clockForced = true;
					m_engCfg.clockSpeed = sid2_clock_t.SID2_CLOCK_NTSC;
				} else if (argv[i].equals("-vpf")) {
					m_engCfg.clockForced = true;
					m_engCfg.clockSpeed = sid2_clock_t.SID2_CLOCK_PAL;
				} else if (argv[i].equals("-vf")) {
					m_engCfg.clockForced = true;
				} else if (argv[i].equals("-vn")) {
					m_engCfg.clockSpeed = sid2_clock_t.SID2_CLOCK_NTSC;
				} else if (argv[i].equals("-vp")) {
					m_engCfg.clockSpeed = sid2_clock_t.SID2_CLOCK_PAL;
				} else if (argv[i].startsWith("-v")) {
					if (argv[i].length() == 2)
						m_verboseLevel = 1;
					else
						m_verboseLevel = new Integer(argv[i].substring(2));
				} else if (argv[i].startsWith("--crc")) {
					m_crc = ~0;
					if (argv[i].charAt(6) == '=')
						m_crc = new Long(argv[i].substring(7)).longValue();
				} else if (argv[i].startsWith("--delay=")) {
					m_engCfg.powerOnDelay = new Integer(argv[i].substring(8));
				}

				// File format conversions
				else if (argv[i].startsWith("--wav")) {
					m_driver.output = OUTPUTS.OUT_WAV;
					m_driver.file = true;
					if (argv[i].length() > 5)
						m_outfile = argv[i].substring(5);
				} else if (argv[i].startsWith("-w")) {
					m_driver.output = OUTPUTS.OUT_WAV;
					m_driver.file = true;
					if (argv[i].length() > 2)
						m_outfile = argv[i].substring(2);
				} else if (argv[i].startsWith("--au")) {
					m_driver.output = OUTPUTS.OUT_AU;
					m_driver.file = true;
					if (argv[i].length() > 4)
						m_outfile = argv[i].substring(4);
				}

				// Hardware selection
				else if (argv[i].startsWith("--hardsid")) {
					// m_driver.sid = SIDEMUS.EMU_HARDSID;
					// m_driver.output = OUTPUTS.OUT_NULL;
				}

				// These are for debug
				else if (argv[i].equals("--none")) {
					m_driver.sid = SIDEMUS.EMU_NONE;
					m_driver.output = OUTPUTS.OUT_NULL;
				} else if (argv[i].equals("--nosid")) {
					m_driver.sid = SIDEMUS.EMU_NONE;
				} else if (argv[i].equals("--cpu-debug")) {
					m_cpudebug = true;
				}

				else {
					err = true;
				}

			} else {
				// Reading file name
				if (infile == 0)
					infile = i;
				else
					err = true;
			}

			if (err) {
				displayArgs(argv[i]);
				return -1;
			}

			i++; // next index
		}

		// Load the tune
		m_filename = argv[infile];
		m_tune = new SidTuneMod(m_filename);
		// m_tune.load(m_filename);
		if (!m_tune.bool()) {
			displayError((m_tune.getInfo()).statusString);
			return -1;
		}

		// If filename specified we can only convert one song
		if (m_outfile != null)
			m_track.single = true;

		// Can only loop if not creating audio files
		if (m_driver.output == OUTPUTS.OUT_WAV
				|| m_driver.output == OUTPUTS.OUT_AU
				|| m_driver.output == OUTPUTS.OUT_END)
			m_track.loop = false;

		// Check to see if we are trying to generate an audio file
		// whilst using a hardware emulation
		if (m_driver.file
				&& (m_driver.sid == SIDEMUS.EMU_HARDSID
						|| m_driver.sid == SIDEMUS.EMU_SIDSTATION
						|| m_driver.sid == SIDEMUS.EMU_COMMODORE
						|| m_driver.sid == SIDEMUS.EMU_SIDSYN || m_driver.sid == SIDEMUS.EMU_END)) {
			displayError("ERROR: Cannot generate audio files using hardware emulations");
			return -1;
		}

		// Select the desired track
		m_track.first = m_tune.selectSong(m_track.first);
		m_track.selected = m_track.first;
		if (m_track.single)
			m_track.songs = 1;

		// CRC handling (remove random behaviour)
		if (m_crc != 0) {
			m_engCfg.powerOnDelay = 0;
			m_engCfg.sid2crcCount = m_crc;
		}

		// If user provided no time then load songlength database
		// and set default lengths incase it's not found in there.
		{
			if (m_driver.file && m_timer.valid && m_timer.length == 0) {
				// Time of 0 provided for wav generation
				displayError("ERROR: -t0 invalid in record mode");
				return -1;
			}
			if (!m_timer.valid) {
				final String database = (m_iniCfg.sidplay2()).database;
				m_timer.length = (m_iniCfg.sidplay2()).playLength;
				if (m_driver.file)
					m_timer.length = (m_iniCfg.sidplay2()).recordLength;
				if (database != null && database.length() > 0) {
					// Try loading the database
					// specified by the user
					if (m_database.open(database) < 0) {
						displayError(m_database.error());
						return -1;
					}
				}
			}
		}

		if (!m_filter.definition.bool()) {
			m_filter.definition.m_filter = m_iniCfg.filter(m_engCfg.sidModel);
			m_filter.definition.m_status = true;
		}
		// #if HAVE_TSID == 1
		// // Set TSIDs base directory
		// if (!m_tsid.setBaseDir(true))
		// {
		// displayError (m_tsid.getError ());
		// return -1;
		// }
		// #endif

		// Configure engine with settings
		if (m_engine.config(m_engCfg) < 0) { // Config failed
			displayError(m_engine.error());
			return -1;
		}
		return 1;
	}

	void displayArgs(String arg) {
		PrintStream out = (arg != null) ? System.err : System.out;

		if (arg != null)
			out.println("Option Error: " + arg);
		else
			out.println("Syntax: " + m_name + " [-<option>...] <datafile>");

		out
				.println("Options:"
						+ "\n"
						+ " --help|-h    display this screen"
						+ "\n"
						+ " --help-debug debug help menu"
						+ "\n"
						+ " -b<num>      set start time in [m:]s format (default 0)"
						+ "\n"

						+ " -f<num>      set frequency in Hz (default: "
						+ SID2_DEFAULT_SAMPLING_FREQ
						+ ")"
						+ "\n"
						+ " -fd          force dual sid environment"
						+ "\n"
						+ " -fs          force samples to a channel (default: uses sid)"
						+ "\n"

						+ " -nf[filter]  no/new SID filter emulation"
						+ "\n"
						+ " -ns[0|1]     (no) MOS 8580 waveforms (default: from tune or cfg)"
						+ "\n"

						+ " -o<l|s>      looping and/or single track"
						+ "\n"
						+ " -o<num>      start track (default: preset)"
						+ "\n"
						+ " -O<num>      optimisation level, max is "
						+ (int /* uint */) (SID2_MAX_OPTIMISATION - 1)
						+ " (default: "
						+ (int /* uint */) SID2_DEFAULT_OPTIMISATION
						+ ')'
						+ "\n"

						+ " -p<num>      set bit precision for samples. "
						+ "(default: "
						+ (int /* uint */) SID2_DEFAULT_PRECISION
						+ ")"
						+ "\n"

						// #if !defined(DISALLOW_STEREO_SOUND)
						+ " -s[l|r]      stereo sid support or [left/right] channel only"
						+ "\n"
						// #endif

						+ " -t<num>      set play length in [m:]s format (0 is endless)"
						+ "\n"

						+ " -<v[level]|q>       verbose (level=0,1,2) or quiet (no time display) output"
						+ "\n"
						+ " -v[p|n][f]   set VIC PAL/NTSC clock speed (default: defined by song)"
						+ "\n"
						+ "              Use 'f' to force the clock by preventing speed fixing"
						+ "\n"

						+ " -w[name]     create wav file (default: <datafile>[n].wav)"
						+ "\n");
		// #ifdef HAVE_HARDSID_BUILDER
		// {
		// HardSIDBuilder hs("");
		// if (hs.devices (false))
		// out << " --hardsid enable hardsid support" << endl;
		// }
		// #endif
		out.println("\n"
		// Changed to new homepage address
				+ "Home Page: http://jsidplay2.sourceforge.net/" + "\n");
		// << "Mail comments, bug reports, or contributions to
		// <sidplay2@email.com>." << endl;
		System.out.println("<press return>");
		try {
			System.in.read();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	void displayDebugArgs() {
		PrintStream out = System.out;

		out
				.println("Debug Options:"
						+ "\n"
						+ " --cpu-debug   display cpu register and assembly dumps"
						+ "\n"
						+ " --crc[=<num>] generate CRC for [<num>] sid writes (default: 0)"
						+ "\n"
						+ " --delay=<num> simulate c64 power on delay"
						+ "\n"

						+ " -m            PlaySID compatibility mode (read the docs!)"
						+ "\n" + " -mt           Sidplays Transparent Rom mode"
						+ "\n" + " -mb           Sidplays Bankswitching mode"
						+ "\n" + " -mr           Real C64 mode (default)>"
						+ "\n"

						+ " --wav<file>   wav file output device" + "\n"
						+ " --none        no audio output device" + "\n"
						+ " --nosid       no sid emulation" + "\n");
		System.out.println("<press return>");
		try {
			System.in.read();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	void menu() {
		final sid2_info_t info = m_engine.info();
		final SidTuneInfo tuneInfo = info.tuneInfo;
		if (m_quietLevel > 1)
			return;

		IniConfig.console_section console = m_iniCfg.console();

		System.out.println(String.format("%c%s%c", console.topLeft, setfill(
				console.horizontal, 54), console.topRight));
		System.out.println(String.format("%c%54s%c", console.vertical,
				" Java SIDPLAY - Music Player & C64 SID Chip Emulator  ",
				console.vertical));
		System.out.println(String.format("%c%54s%c", console.vertical,
				"          Sidplay V" + sidplay.Config.VERSION + ", "
						+ info.name + " V" + info.version + "           ",
				console.vertical));
		System.out.println(String.format("%c%s%c", console.junctionLeft,
				setfill(console.horizontal, 54), console.junctionRight));
		if (tuneInfo.numberOfInfoStrings > 0) {
			if (!tuneInfo.musPlayer && (tuneInfo.numberOfInfoStrings == 3)) {
				System.out.println(String.format("%c Title        : %37s %c",
						console.vertical, tuneInfo.infoString[0],
						console.vertical));
				System.out.println(String.format("%c Author       : %37s %c",
						console.vertical, tuneInfo.infoString[1],
						console.vertical));
				System.out.println(String.format("%c Released     : %37s %c",
						console.vertical, tuneInfo.infoString[2],
						console.vertical));
			} else {
				for (int i = 0; i < tuneInfo.numberOfInfoStrings; i++) {
					System.out.println(String.format(
							"%c Description  : %37s %c", console.vertical,
							tuneInfo.infoString[i], console.vertical));
				}
			}
			System.out.println(String.format("%c%s%c", console.junctionLeft,
					setfill(console.horizontal, 54), console.junctionRight));
		}
		if (m_verboseLevel != 0) {
			System.out.println(String.format("%c File format  : %37s %c",
					console.vertical, tuneInfo.formatString, console.vertical));
			System.out.println(String.format("%c Filename(s)  : %37s %c",
					console.vertical, tuneInfo.dataFileName, console.vertical));
			// Second file is only sometimes present
			if (tuneInfo.infoFileName != null) {
				System.out.println(String.format("%c              : %37s %c",
						console.vertical, tuneInfo.infoFileName,
						console.vertical));
			}
			System.out.println(String.format("%c Condition    : %37s %c",
					console.vertical, tuneInfo.statusString, console.vertical));
		}
		System.out.print(String.format("%c Playlist     : ", console.vertical));
		{ // This will be the format used for playlists
			int i = 1;
			if (!m_track.single) {
				i = m_track.selected;
				i -= (m_track.first - 1);
				if (i < 1)
					i += m_track.songs;
			}
			System.out.println(String.format("%37s %c", i + "/" + m_track.songs
					+ " (tune " + tuneInfo.currentSong + "/" + tuneInfo.songs
					+ "[" + tuneInfo.startSong + "])"
					+ (m_track.loop ? " [LOOPING]" : ""), console.vertical));
		}
		if (m_verboseLevel > 0) {
			System.out.println(String.format("%c%s%c", console.bottomLeft,
					setfill(console.horizontal, 54), console.bottomRight));
			System.out.println(String.format("%c Song Speed   : %37s %c",
					console.vertical, tuneInfo.speedString, console.vertical));
		}
		System.out.print(String.format("%c Song Length  : ", console.vertical));
		if (m_timer.stop != 0) {
			String time = String.format("%02d:%02d",
					((m_timer.stop / 60) % 100), (m_timer.stop % 60));
			System.out.print(String.format("%37s %c", "" + time,
					console.vertical));
		} else if (m_timer.valid)
			System.out.print(String.format("%37s %c", "FOREVER",
					console.vertical));
		else
			System.out.print(String.format("%37s %c", "UNKNOWN",
					console.vertical));
		System.out.println();
		if (m_verboseLevel > 0) {
			System.out.println(String.format("%c%s%c", console.bottomLeft,
					setfill(console.horizontal, 54), console.bottomRight));
			StringBuffer line = new StringBuffer();
			// Display PSID Driver location
			line.append("DRIVER = ");
			if (info.driverAddr == 0)
				line.append("NOT PRESENT");
			else {
				line.append(String.format("$%04x", info.driverAddr));
				line.append(String.format("-$%04x", info.driverAddr
						+ (info.driverLength - 1)));
			}
			if (tuneInfo.playAddr == 0xffff)
				line.append(String.format(", SYS = $%04x", tuneInfo.initAddr));
			else
				line.append(String.format(", INIT = $%04x", tuneInfo.initAddr));

			System.out.println(String.format("%c Addresses    : %37s %c",
					console.vertical, line.toString(), console.vertical));
			line = new StringBuffer();
			line.append(String.format("LOAD   = $%04x", tuneInfo.loadAddr));
			line.append(String.format("-$%04x", tuneInfo.loadAddr
					+ (tuneInfo.c64dataLen - 1)));
			if (tuneInfo.playAddr != 0xffff)
				line.append(String.format(", PLAY = $%04x", tuneInfo.playAddr));
			System.out.println(String.format("%c              : %37s %c",
					console.vertical, line.toString(), console.vertical));

			line = new StringBuffer();
			line.append(String.format("Filter = %s",
					((m_filter.enabled == true) ? "Yes" : "No")));
			line
					.append(String
							.format(
									", Model = %s",
									(info.tuneInfo.sidModel == SidTune.SIDTUNE_SIDMODEL_8580 ? "8580"
											: "6581")));
			System.out.println(String.format("%c SID Details  : %37s %c",
					console.vertical, line.toString(), console.vertical));

			line = new StringBuffer();
			switch (info.environment) {
			case sid2_envPS:
				line.append("PlaySID-specific rips");
				break;
			case sid2_envTP:
				line.append("Transparent ROM");
				break;
			case sid2_envBS:
				line.append("Bank Switching");
				break;
			case sid2_envR: // When it happens
				line.append("Real C64");
				break;
			case sid2_envTR:
				line.append("Sidusage Tracker Mode");
				break;
			default:
				line.append("Unknown");
			}

			if (m_engCfg.environment != info.environment)
				line.append(" (forced)");
			System.out.println(String.format("%c Environment  : %37s %c",
					console.vertical, line.toString(), console.vertical));

			if (m_verboseLevel > 1) {
				line = new StringBuffer();
				line.append(String.format("%d (cycles at poweron)",
						info.powerOnDelay));
				System.out.println(String.format("%c Delay        : %37s %c",
						console.vertical, line.toString(), console.vertical));
			}
		}

		System.out.println(String.format("%c%s%c", console.bottomLeft, setfill(
				console.horizontal, 54), console.bottomRight));
		System.out.println("keyboard control (press enter after command):");
		System.out.println("< > - play previous/next tune");
		System.out.println(", . - normal/faster speed");
		System.out.println("p   - pause/continue player");
		System.out.println("h e - play first/last tune");
		System.out.println("d   - SID chip distortion simulation on/off\n"
				+ "      (http://bel.fi/~alankila/c64-sw/)");
		System.out.println("q   - quit player");
	}

	private String setfill(char ch, int length) {
		StringBuffer ret = new StringBuffer();
		for (int i = 0; i < length; i++) {
			ret.append(ch);
		}
		return ret.toString();
	}

	public static void main(String[] args) {
		ConsolePlayer player = new ConsolePlayer("java -jar jsidplay2.jar");
		if (player.args(args) < 0) {
			System.exit(1);
		}
		if (player.m_tune == null) {
			System.exit(0);
		}
		main_restart: while (true) {
			if (!player.open()) {
				player.close();
				System.exit(0);
			}
			while (true) {
				if (!player.play())
					break;
			}
			if ((player.state() & ~playerFast) == playerRestart)
				continue main_restart;
			break;
		}
		player.close();
	}
}
