#include <stdio.h>
#include <stdlib.h>
#include <windows.h>
#include <winioctl.h>
#include <assert.h>
#include "krnpatch/krnpatch.h"
#include <commctrl.h>
#include "resource.h"

// names of resource entries for binary file extraction:
#define	KRNPATCH_SYS_RSRC_NAME	"KRNPATCH.SYS"
#define	BLEEM_NT_DLL_RSRC_NAME	"BLEEM_NT.DLL"
// names of dll's to search for/replace during .exe patching
#define	KERNEL32_DLL_NAME		"KERNEL32.DLL"
#define	BLEEM_NT_DLL_NAME		"bleem_nt.dll"
// name of patched .exe file
#define	BLEEM_NT_TMP_NAME		"bleem_nt.tmp"
// default pathnames for bleem!.exe and boot.ini files
#define	DEFAULT_BLEEM_EXE_NAME	"bleem!.exe"
#define	DEFAULT_BOOT_INI_NAME	"c:\\boot.ini"
// name of registry entry used for default values saving
#define	BLEEMRUN_REG_KEY_NAME	"Software\\bleem\\bleemrun"

#if _MSC_VER < 1200
typedef struct {
	DWORD dwOSVersionInfoSize;
	DWORD dwMajorVersion;
	DWORD dwMinorVersion;
	DWORD dwBuildNumber;
	DWORD dwPlatformId;
	CHAR  szCSDVersion[ 128 ];
	WORD wServicePackMajor;
	WORD wServicePackMinor;
	WORD wReserved[2];
} OSVERSIONINFOEX;
#endif

typedef struct {
	char	*os_boot,  *stripped_os_boot;
	char	*os_label, *stripped_os_label;
	char	*os_flags, *stripped_os_flags;
	BOOL	is_NT;
	BOOL	is_3gb;
	BOOL	is_mine;
	int		usr_mem;
	int		idx_lnk;
	char	*_alloc_data;	// allocated data (2 * original size)
	char	*_section_ptr;	// ptr to original 'os_section_buff[]' data
} OS_INFO;

// global variables, initialized once at program startup:
static HINSTANCE		hInstance = NULL;
static OSVERSIONINFOEX	os_version;
static BOOL				is_3gb_os;		// OS is /3gb capable now
static BOOL				bootini_has_single_NTOs_entry = FALSE;	// boot.ini contains a single NT O.S. entry
static BOOL				bootini_has_3gb_enabled_entry = FALSE;	// boot.ini contains a /3gb enabled entry
// other variables, used inside the program:
static HANDLE			hDriver = INVALID_HANDLE_VALUE;
static char	boot_ini_file[MAX_PATH] = DEFAULT_BOOT_INI_NAME;
static char	bleem_exe_file[MAX_PATH] = DEFAULT_BLEEM_EXE_NAME;
static char	bleem_cmdline_args[500] = "";
// boot.ini '[operating systems]' section content:
static char	os_section_buff[8192];
static OS_INFO	os_info[256];
static int		os_info_size = 0;

BOOL IsAValidBleemExeFile(const char *fname);
BOOL MakePEFileAsNTCompatible(const char *fname);
BOOL StartBleemExeFile(const char *fname);
BOOL test_3gb_os();
BOOL CheckIfSingleNTOsBootSystem(BOOL *is_single, BOOL *is_3gb);
BOOL CALLBACK BootIniDialogProc(HWND hwnd, UINT msg, WPARAM wPar, LPARAM lPar);
BOOL CALLBACK MainDialogProc(HWND hwnd, UINT msg, WPARAM wPar, LPARAM lPar);
BOOL ExecFromCmdLine(int argc, char *argv[]);
int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow);

//******************************************************************************

static BOOL CreateFileFromResource(LPCSTR rsrc_name, LPCSTR file_path)
{
	HRSRC	hResInfo = FindResource(hInstance, rsrc_name, "BINRES");
	if (hResInfo == NULL) return FALSE;
	HGLOBAL	hResData = LoadResource(hInstance, hResInfo);
	if (hResData == NULL) return FALSE;
	LPVOID	lpData = LockResource(hResData);
	DWORD	dwSize = SizeofResource(hInstance, hResInfo);
	if (lpData == NULL || dwSize == 0) return FALSE;
	HANDLE	hFile = CreateFile(file_path, GENERIC_WRITE, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
	if (hFile == INVALID_HANDLE_VALUE) return FALSE;
	DWORD	dwOutByteCount;
	WriteFile(hFile, lpData, dwSize, &dwOutByteCount, NULL);
	CloseHandle(hFile);
	if (dwOutByteCount != dwSize) return FALSE;
	return TRUE;
}

//******************************************************************************

BOOL IsAValidBleemExeFile(const char *fname)
{
	BOOL	res = FALSE;
	HANDLE	hFile;
	static const char BLEEM_COPYRIGHT_STRING[]	= "bleem! written by Randy Linden";
	static const int  BLEEM_COPYRIGHT_OFFSET	= 0x202;

	if ((hFile = CreateFile(fname, GENERIC_READ, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL)) != INVALID_HANDLE_VALUE) {
		char	buffer[sizeof(BLEEM_COPYRIGHT_STRING)];
		DWORD	dwOutSize;
		if (SetFilePointer(hFile, BLEEM_COPYRIGHT_OFFSET, NULL, FILE_BEGIN) != (DWORD)-1 &&
			ReadFile(hFile, buffer, sizeof(BLEEM_COPYRIGHT_STRING), &dwOutSize, NULL) &&
			dwOutSize == sizeof(BLEEM_COPYRIGHT_STRING) &&
			!memcmp(buffer, BLEEM_COPYRIGHT_STRING, sizeof(BLEEM_COPYRIGHT_STRING)))
			res = TRUE;
		CloseHandle(hFile);
	}
	return res;
}

BOOL MakePEFileAsNTCompatible(const char *fname)
{
	BOOL	res = FALSE;
	HANDLE					hFile;
	IMAGE_DOS_HEADER		dos_header;
	IMAGE_NT_HEADERS		nt_header;
	IMAGE_SECTION_HEADER	section_headers[8];
	char					import_section_data[8192];
	IMAGE_IMPORT_DESCRIPTOR	*import_dir;
	DWORD					dwOutSize, idx, section_offset, size, name_offset;

	for (;;) {	// not a real loop: just to exit with 'break' on error
		if ((hFile = CreateFile(fname, GENERIC_READ | GENERIC_WRITE, 0, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL)) == INVALID_HANDLE_VALUE)
			break;
		if (SetFilePointer(hFile, 0, NULL, FILE_BEGIN) == (DWORD)-1 ||
			!ReadFile(hFile, &dos_header, sizeof(IMAGE_DOS_HEADER), &dwOutSize, NULL) ||
			dwOutSize != sizeof(IMAGE_DOS_HEADER) ||
			dos_header.e_magic != IMAGE_DOS_SIGNATURE ||
			dos_header.e_lfarlc < sizeof(IMAGE_DOS_HEADER))
			break;
		if (SetFilePointer(hFile, dos_header.e_lfanew, NULL, FILE_BEGIN) == (DWORD)-1 ||
			!ReadFile(hFile, &nt_header, sizeof(IMAGE_NT_HEADERS), &dwOutSize, NULL) ||
			dwOutSize != sizeof(IMAGE_NT_HEADERS) ||
			nt_header.Signature != IMAGE_NT_SIGNATURE ||
			nt_header.FileHeader.SizeOfOptionalHeader != sizeof(IMAGE_OPTIONAL_HEADER) ||
			nt_header.OptionalHeader.NumberOfRvaAndSizes != IMAGE_NUMBEROF_DIRECTORY_ENTRIES)
			break;
		if (//SetFilePointer(hFile, dos_header.e_lfanew + sizeof(IMAGE_NT_HEADERS), NULL, FILE_BEGIN) == (DWORD)-1 ||
			nt_header.FileHeader.NumberOfSections > (sizeof(section_headers) / sizeof(section_headers[0])) ||
			!ReadFile(hFile, section_headers, sizeof(IMAGE_SECTION_HEADER) * nt_header.FileHeader.NumberOfSections, &dwOutSize, NULL) ||
			dwOutSize != sizeof(IMAGE_SECTION_HEADER) * nt_header.FileHeader.NumberOfSections)
			break;
		// make PE header WinNT compabile (drop empty section, fix header & image sizes)
		if (section_headers[nt_header.FileHeader.NumberOfSections - 1].Misc.VirtualSize == 0 &&
			section_headers[nt_header.FileHeader.NumberOfSections - 1].VirtualAddress == 0)
			nt_header.FileHeader.NumberOfSections--;
		nt_header.OptionalHeader.SizeOfHeaders = section_headers[0].PointerToRawData;
		// enable /3GB large address support
		nt_header.FileHeader.Characteristics |= 0x20;	//IMAGE_FILE_LARGE_ADDRESS_AWARE
		size = (section_headers[nt_header.FileHeader.NumberOfSections - 1].Misc.VirtualSize + 
				section_headers[nt_header.FileHeader.NumberOfSections - 1].VirtualAddress +
				(nt_header.OptionalHeader.SectionAlignment - 1)) &
				~(nt_header.OptionalHeader.SectionAlignment - 1);
		nt_header.OptionalHeader.SizeOfImage = size;
		// search & replace 'KERNEL32.DLL' with 'BLEEM_NT.DLL'
		for (idx = 0; idx < nt_header.FileHeader.NumberOfSections; idx++)
			if ((section_offset = nt_header.OptionalHeader.DataDirectory[IMAGE_DIRECTORY_ENTRY_IMPORT].VirtualAddress - section_headers[idx].VirtualAddress) >= 0 &&
				(section_offset + nt_header.OptionalHeader.DataDirectory[IMAGE_DIRECTORY_ENTRY_IMPORT].Size) <= section_headers[idx].Misc.VirtualSize)
				break;
		if (idx >= nt_header.FileHeader.NumberOfSections ||
			nt_header.OptionalHeader.DataDirectory[IMAGE_DIRECTORY_ENTRY_IMPORT].Size > sizeof(import_section_data) ||
			SetFilePointer(hFile, section_offset + section_headers[idx].PointerToRawData, NULL, FILE_BEGIN) == (DWORD)-1 ||
			!ReadFile(hFile, import_section_data, nt_header.OptionalHeader.DataDirectory[IMAGE_DIRECTORY_ENTRY_IMPORT].Size, &dwOutSize, NULL) ||
			dwOutSize != nt_header.OptionalHeader.DataDirectory[IMAGE_DIRECTORY_ENTRY_IMPORT].Size)
			break;
		for (import_dir = (IMAGE_IMPORT_DESCRIPTOR *)import_section_data, size = nt_header.OptionalHeader.DataDirectory[IMAGE_DIRECTORY_ENTRY_IMPORT].Size; ; import_dir++, size -= sizeof(IMAGE_IMPORT_DESCRIPTOR)) {
			if (size < sizeof(IMAGE_IMPORT_DESCRIPTOR) || import_dir->Name == NULL ||
				(name_offset = import_dir->Name - nt_header.OptionalHeader.DataDirectory[IMAGE_DIRECTORY_ENTRY_IMPORT].VirtualAddress) > nt_header.OptionalHeader.DataDirectory[IMAGE_DIRECTORY_ENTRY_IMPORT].Size)
				break;
			if (!stricmp((LPSTR)&import_section_data[name_offset], KERNEL32_DLL_NAME) ||
				!stricmp((LPSTR)&import_section_data[name_offset], BLEEM_NT_DLL_NAME))	// (input file already patched)
			{
				if (SetFilePointer(hFile, section_offset + section_headers[idx].PointerToRawData + name_offset, NULL, FILE_BEGIN) != (DWORD)-1 &&
					WriteFile(hFile, BLEEM_NT_DLL_NAME, sizeof(KERNEL32_DLL_NAME), &dwOutSize, NULL) &&
					dwOutSize == sizeof(KERNEL32_DLL_NAME) &&
					SetFilePointer(hFile, dos_header.e_lfanew, NULL, FILE_BEGIN) != (DWORD)-1 &&
					WriteFile(hFile, &nt_header, sizeof(IMAGE_NT_HEADERS), &dwOutSize, NULL) &&
					dwOutSize == sizeof(IMAGE_NT_HEADERS))
					res = TRUE;		// success !!!
				break;
			}
		}
		break;	// always break loop (see comment on 'for (;;)' below)
	}
	if (hFile != INVALID_HANDLE_VALUE)
		CloseHandle(hFile);
	return res;
}

BOOL StartBleemExeFile(const char *fname)
{
	char	path_name[MAX_PATH], *lpName;
	GetFullPathName(fname, sizeof(path_name), path_name, &lpName);

#ifndef	_DEBUG
	strcpy(lpName, BLEEM_NT_DLL_NAME);
	CreateFileFromResource(BLEEM_NT_DLL_RSRC_NAME, path_name);
#endif//_DEBUG

	strcpy(lpName, BLEEM_NT_TMP_NAME);
	if (!CopyFile(fname, path_name, FALSE)) {
		// error copying .exe to temporary file
		if (GetLastError() != ERROR_SHARING_VIOLATION)
			return FALSE;
		// another copy of the program is already patched & running: skip next step
	} else {
		// successful copied .exe to temporary file
		if (!MakePEFileAsNTCompatible(path_name))
			return FALSE;
	}

	char	appl_name[MAX_PATH], cmd_line[MAX_PATH+500];
	//strcpy(lpName, BLEEM_NT_TMP_NAME);
	strcpy(appl_name, path_name);
	sprintf(cmd_line, "\"%s\"%c%s", appl_name, (*bleem_cmdline_args) ? ' ' : '\0', bleem_cmdline_args);
	*lpName = '\0';	// leave the working directory on 'path_name'
	STARTUPINFO			si;
	PROCESS_INFORMATION	pi;
	ZeroMemory(&si, sizeof(si));
	si.cb = sizeof(si);
	if (!CreateProcess(appl_name, cmd_line, NULL, NULL, FALSE, 0, NULL, path_name, &si, &pi))
		return FALSE;
	CloseHandle(pi.hThread);
	CloseHandle(pi.hProcess);

	return TRUE;
}

static void LoadDefaultValuesFromReg(void)
{
	HKEY	hKey;
	DWORD	dwType, dwSize;

	if (RegOpenKeyEx(HKEY_CURRENT_USER, BLEEMRUN_REG_KEY_NAME, 0, KEY_ALL_ACCESS, &hKey) != ERROR_SUCCESS)
		return;
	RegQueryValueEx(hKey, "LastRunFile", 0, &dwType, (LPBYTE)bleem_exe_file, &(dwSize = sizeof(bleem_exe_file)));
	RegQueryValueEx(hKey, "CmdLineArgs", 0, &dwType, (LPBYTE)bleem_cmdline_args, &(dwSize = sizeof(bleem_cmdline_args)));
	RegCloseKey(hKey);
}

static void SaveDefaultValuesToReg(void)
{
	HKEY	hKey;
	DWORD	dwDisposition;

	if (RegCreateKeyEx(HKEY_CURRENT_USER, BLEEMRUN_REG_KEY_NAME, 0, "", REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &hKey, &dwDisposition) != ERROR_SUCCESS)
		return;
	RegSetValueEx(hKey, "LastRunFile", 0, REG_SZ, (LPBYTE)bleem_exe_file, strlen(bleem_exe_file));
	RegSetValueEx(hKey, "CmdLineArgs", 0, REG_SZ, (LPBYTE)bleem_cmdline_args, strlen(bleem_cmdline_args));
	RegCloseKey(hKey);
}

static char *SetupOFNInitialDir(char *base_dir, const char *file_name)
{
    char    *p;
    if ((p = strrchr(file_name, '\\')) != NULL)
    {
        if (p > file_name && *(p-1) == ':')
            p++;    // if 'X:\', keep the '\' at the end
        strncpy(base_dir, file_name, (int)(p - file_name));
        base_dir[(int)(p - file_name)] = '\0';
    }
    else
        strcpy(base_dir, ".\\");
	return base_dir;
}

//******************************************************************************

static BOOL LoadDeviceDriver(LPCSTR name, LPCSTR path)
{
	SC_HANDLE	schManager, schDriver;
	char	driverName[50];
	
	if ((schManager = OpenSCManager(NULL, NULL, SC_MANAGER_ALL_ACCESS)) != NULL) {
		if ((schDriver = CreateService(schManager, name, name, SERVICE_ALL_ACCESS, SERVICE_KERNEL_DRIVER, SERVICE_DEMAND_START, SERVICE_ERROR_NORMAL, path, NULL, NULL, NULL, NULL, NULL)) != NULL ||
			(schDriver = OpenService(schManager, name, SERVICE_ALL_ACCESS)) != NULL) {
			StartService(schDriver, 0, NULL);
			CloseServiceHandle(schDriver);
		}
		CloseServiceHandle(schManager);
	}
	sprintf(driverName, "\\\\.\\%s", name);
	if ((hDriver = CreateFile(driverName, GENERIC_READ | GENERIC_WRITE, 0, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL)) == INVALID_HANDLE_VALUE)
		return FALSE;
	return TRUE;
}

static BOOL Win2K_PatchNtOsKernel(void)
{
	union {
		KRNPATCH_FIND_DATA	find_data;
		KRNPATCH_READ_DATA	read_data;
		KRNPATCH_WRITE_DATA	write_data;
	} buffer;
	HKEY	hKey;
	DWORD	dwByteCount, found_offset;
	char	ntoskrnl_exe[MAX_PATH];

	static const BYTE	find_data_w2k[] = {
		0x8B, 0x45, 0x10,		// mov	eax,[ebp+10h]
		0xF6, 0x40, 0x1C, 0x20,	// test	byte ptr [eax+1Ch],20h
	};
	static const int	read_offset_w2k = -2;
	static const BYTE	read_data_w2k[] = {
		0x74, 0x09,				// jz	.....
	};
	static const BYTE	write_data_w2k[] = {
		0x90, 0x90				// nop (x2)
	};

	strcpy(ntoskrnl_exe, "ntoskrnl.exe");
	if (RegOpenKeyEx(HKEY_LOCAL_MACHINE, "SYSTEM\\CurrentControlSet\\Control", 0, KEY_ALL_ACCESS, &hKey) == ERROR_SUCCESS) {
		char	*data = NULL, *p;
		dwByteCount = 0;
		if (RegQueryValueEx(hKey, "SystemStartOptions", NULL, NULL, NULL, &dwByteCount) == ERROR_SUCCESS) {
			data = (char *)malloc(dwByteCount);
			RegQueryValueEx(hKey, "SystemStartOptions", NULL, NULL, (BYTE *)data, &dwByteCount);
		}
		RegCloseKey(hKey);
		if (data != NULL) {
			for (p = strtok(data, " "); p != NULL; p = strtok(NULL, " ")) {
				if (strnicmp(p, "KERNEL=", 7))
					continue;
				strcpy(ntoskrnl_exe, p + 7);
			}
			free(data);
		}
	}

	memset(&buffer, 0, sizeof(buffer));
	strcpy(buffer.find_data.module_name, ntoskrnl_exe);
	buffer.find_data.scn_cnt_type = IMAGE_SCN_CNT_CODE;	// scan 'code' sections only
	buffer.find_data.start_offset = 0;
	buffer.find_data.data_len = sizeof(find_data_w2k);
	memcpy(buffer.find_data.data, find_data_w2k, buffer.find_data.data_len);
	if (!DeviceIoControl(hDriver, IOCTL_KRNPATCH_FIND_DATA, &buffer, sizeof(buffer.find_data), &buffer, sizeof(buffer.find_data), &dwByteCount, NULL))
		return FALSE;
	found_offset = buffer.find_data.found_offset;

	buffer.find_data.start_offset = found_offset + 1;
	if (DeviceIoControl(hDriver, IOCTL_KRNPATCH_FIND_DATA, &buffer, sizeof(buffer), &buffer, sizeof(buffer), &dwByteCount, NULL))
		return FALSE;

	memset(&buffer, 0, sizeof(buffer));
	strcpy(buffer.read_data.module_name, ntoskrnl_exe);
	buffer.read_data.start_offset = found_offset + read_offset_w2k;
	buffer.read_data.data_len = sizeof(read_data_w2k);
	if (!DeviceIoControl(hDriver, IOCTL_KRNPATCH_READ_DATA, &buffer, sizeof(buffer.read_data), &buffer, sizeof(buffer.read_data), &dwByteCount, NULL))
		return FALSE;
	if (!memcmp(buffer.read_data.data, write_data_w2k, sizeof(write_data_w2k)))
		return TRUE;	// already patched...
	if (memcmp(buffer.read_data.data, read_data_w2k, sizeof(read_data_w2k)))
		return FALSE;	// not matching data...

	memset(&buffer, 0, sizeof(buffer));
	strcpy(buffer.write_data.module_name, ntoskrnl_exe);
	buffer.write_data.start_offset = found_offset + read_offset_w2k;
	buffer.write_data.data_len = sizeof(write_data_w2k);
	memcpy(buffer.write_data.old_data, read_data_w2k, buffer.write_data.data_len);
	memcpy(buffer.write_data.new_data, write_data_w2k, buffer.write_data.data_len);
	if (!DeviceIoControl(hDriver, IOCTL_KRNPATCH_WRITE_DATA, &buffer, sizeof(buffer.write_data), &buffer, sizeof(buffer.write_data), &dwByteCount, NULL))
		return FALSE;

	return TRUE;
}

static void UnloadDeviceDriver(LPCSTR name)
{
	SC_HANDLE	schManager, schDriver;
	SERVICE_STATUS	serviceStatus;

	if (hDriver != INVALID_HANDLE_VALUE) {
		CloseHandle(hDriver);
		hDriver = INVALID_HANDLE_VALUE;
	}
	if ((schManager = OpenSCManager(NULL, NULL, SC_MANAGER_ALL_ACCESS)) != NULL) {
		if ((schDriver = OpenService(schManager, name, SERVICE_ALL_ACCESS)) != NULL) {
			ControlService(schDriver, SERVICE_CONTROL_STOP, &serviceStatus);
			DeleteService(schDriver);
			CloseServiceHandle(schDriver);
		}
		CloseServiceHandle(schManager);
	}
}

BOOL test_3gb_os()
{
	BOOL	res = FALSE;
	SYSTEM_INFO		system_info;
	MEMORY_BASIC_INFORMATION	memory_info;

	if (os_version.dwPlatformId != VER_PLATFORM_WIN32_NT)
		return TRUE;
	// on NT-based platforms, ensure system was started with the /3GB option:
	GetSystemInfo(&system_info);
	if ((DWORD)system_info.lpMaximumApplicationAddress < 0xA4000000) {
		return FALSE;	// not started with '/3gb' option
	}
	// ensure than address range from 2Gb to 3Gb is available for the program
	VirtualQuery((LPVOID)0x80000000, &memory_info, sizeof(memory_info));
	if (memory_info.AllocationBase != 0) {	// top memory has been locked from kernel
		// Win2K only: if the memory area from 2Gb to 3Gb was locked by 'ntoskrnl.exe', patch it !!!
		if (os_version.dwMajorVersion == 5 &&
			os_version.dwMinorVersion == 0) {	// 5.0 = Win2K
			char	buff[MAX_PATH], *p;
			GetModuleFileName(NULL, buff, sizeof(buff));
			p = strrchr(buff, '\\');
			strcpy((p != NULL) ? p + 1 : buff, KRNPATCH_DRIVER_FILENAME);
#ifndef	_DEBUG
			CreateFileFromResource(KRNPATCH_SYS_RSRC_NAME, buff);
#endif//_DEBUG
			if ((res = LoadDeviceDriver(KRNPATCH_DOS_DEVICENAME, buff)))
				res = Win2K_PatchNtOsKernel();
			UnloadDeviceDriver(KRNPATCH_DOS_DEVICENAME);
#ifndef	_DEBUG
			DeleteFile(buff);
#endif//_DEBUG
		}
		return res;
	} else {	// 3Gb of user-mode address space avail for use
		res = TRUE;
	}
	return res;
}

//******************************************************************************

static int MsgBox(HWND hWnd, WORD wTextID, WORD wCaptionID, UINT uType)
{
	char	szText[2048];
	char	szCaption[256];
	int		iRes;

	LoadString(NULL, wTextID, szText, sizeof(szText));
	LoadString(NULL, wCaptionID, szCaption, sizeof(szCaption));

	iRes = MessageBox(hWnd, szText, szCaption, uType);
	return iRes;
}

//******************************************************************************

static BOOL BootIni_ConvertOsSectionToOsInfo(char **section_ptr, OS_INFO *info_ptr)
{
	info_ptr->_section_ptr = *section_ptr;
	info_ptr->_alloc_data = (char *)malloc(2 * (strlen(*section_ptr) + 1));
	strcpy(info_ptr->_alloc_data, *section_ptr);
	char *p = info_ptr->_alloc_data;

	do {
		info_ptr->os_boot = p;
		if ((p = strchr(p, '=')) == NULL)
			break;		// boot must be '=' terminated
		*p++ = '\0';
		if (*p++ != '"')
			break;		// label must be '"' enclosed
		info_ptr->os_label = p;
		if ((p = strchr(p, '"')) == NULL)
			break;		// label must be '"' enclosed
		*p++ = '\0';
		info_ptr->os_flags = p;
		if (*p != ' ' && *p != '\0')
			break;		// flags must be empty or starting with blank
		p += (strlen(p) + 1);
		*section_ptr += (p - info_ptr->_alloc_data);
		// make a 'working copy' of the data (to be stripped from 'BootIni_ParseOsInfo()')
		strcpy(info_ptr->stripped_os_boot = p, info_ptr->os_boot);
		p += strlen(p) + 1;
		strcpy(info_ptr->stripped_os_label = p, info_ptr->os_label);
		p += strlen(p) + 1;
		strcpy(info_ptr->stripped_os_flags = p, info_ptr->os_flags);
		//p += strlen(p) + 1;
		return TRUE;
	} while (0);
	// we can reach here in case of error parsing the line:
	free(info_ptr->_alloc_data);
	*section_ptr += (strlen(*section_ptr) + 1);
	return FALSE;
}

static void BootIni_ParseOsInfo(OS_INFO *info_ptr)
{
	// Note: this function modify the data on 'stripped_*' fields:
	// - The 'stripped_os_boot' value is returned lowercase.
	// - The 'stripped_os_label' value is returned stripped from ' [/3GB Enabled]'.
	// - The 'stripped_os_flags' value is returned lowercase and stripped from ' /3gb' and ' /userva='nnnn''.
	char	*p;

	info_ptr->is_NT = TRUE;
	info_ptr->is_3gb = FALSE;
	info_ptr->is_mine = FALSE;
	info_ptr->usr_mem = 0;
	strlwr(info_ptr->stripped_os_boot);
	strlwr(info_ptr->stripped_os_flags);

	if (!strstr(info_ptr->stripped_os_boot, ")disk(")) {
		info_ptr->is_NT = FALSE;	// not a Win2K/XP boot partition
	} else if ((p = strstr(info_ptr->stripped_os_flags, " /3gb")) != NULL &&
		(p[5] == ' ' || p[5] == '\0')) {
		strcpy(p, p + 5);			// skip ' /3gb' string
		if ((p = strstr(info_ptr->stripped_os_flags, " /userva=")) != NULL &&
			isdigit(p[9]) && isdigit(p[10]) && isdigit(p[11]) && isdigit(p[12]) &&
			(p[13] == ' ' || p[13] == '\0')) {
			info_ptr->usr_mem = atoi(p + 9);	// save '/userva=' option value
			strcpy(p, p + 13);		// skip ' /userva=nnnn' string
		}
		if ((p = strstr(info_ptr->stripped_os_label, " [/3GB Enabled]")) != NULL) {
			info_ptr->is_mine = TRUE;	// mark the entry as 'created from myself'
			strcpy(p, p + 15);		// skip additional label comment
		}
		info_ptr->is_NT = TRUE;
		info_ptr->is_3gb = TRUE;	// option '/3gb' present
	} else {
		info_ptr->is_NT = TRUE;
		info_ptr->is_3gb = FALSE;	// option '/3gb' NOT present
	}
}

static void BootIni_LinkOsEntries(OS_INFO os_info[], int os_info_size)
{
	for (int idx_cur = 0; idx_cur < os_info_size; idx_cur++) {
		OS_INFO *ptr_cur = &os_info[idx_cur];
		ptr_cur->idx_lnk = -1;		// init entry as not linked
		if (!ptr_cur->is_NT)
			continue;
		for (int idx_lnk = 0; idx_lnk < idx_cur; idx_lnk++) {
			OS_INFO *ptr_lnk = &os_info[idx_lnk];
			if (!ptr_lnk->is_NT || (ptr_cur->is_3gb == ptr_lnk->is_3gb) ||
				stricmp(ptr_cur->stripped_os_boot, ptr_lnk->stripped_os_boot) ||
				stricmp(ptr_cur->stripped_os_label, ptr_lnk->stripped_os_label) ||
				stricmp(ptr_cur->stripped_os_flags, ptr_lnk->stripped_os_flags))
				continue;
			ptr_cur->idx_lnk = idx_lnk;	// save linked entry idx
			ptr_lnk->idx_lnk = idx_cur;
		}
	}
}

static BOOL BootIni_LoadOsSectionBuff(void)
{
	// ensure boot.ini file exists and is accessible for write
	DWORD dwFileAttributes = GetFileAttributes(boot_ini_file);
	if (dwFileAttributes == (DWORD)-1)
		return FALSE;					// file not found
	if ((dwFileAttributes & FILE_ATTRIBUTE_READONLY) &&
		!SetFileAttributes(boot_ini_file, dwFileAttributes & ~FILE_ATTRIBUTE_READONLY))
		return FALSE;
	HANDLE hFile = CreateFile(boot_ini_file, GENERIC_READ | GENERIC_WRITE, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
	if (hFile != INVALID_HANDLE_VALUE)
		CloseHandle(hFile);

	// read O.S. section content (on Win9x, this work only if file is NOT readonly):
	if (hFile != INVALID_HANDLE_VALUE)
		GetPrivateProfileSection("operating systems", os_section_buff, sizeof(os_section_buff), boot_ini_file);

	SetFileAttributes(boot_ini_file, dwFileAttributes);
	if (hFile == INVALID_HANDLE_VALUE)
		return FALSE;

	return TRUE;
}

static BOOL BootIni_SaveOsSectionBuff(BOOL use_boot_delay)
{
	DWORD dwFileAttributes = GetFileAttributes(boot_ini_file);
	if (dwFileAttributes == (DWORD)-1)
		return FALSE;					// file not found
	if ((dwFileAttributes & FILE_ATTRIBUTE_READONLY) &&
		!SetFileAttributes(boot_ini_file, dwFileAttributes & ~FILE_ATTRIBUTE_READONLY))
		return FALSE;

	char	boot_delay[50];
	int delay = GetPrivateProfileInt("boot loader", "timeout", 0, boot_ini_file);
	if (!use_boot_delay && delay >  0) delay = 0;	// single O.S. boot: NO delay on boot
	if (use_boot_delay && delay <= 0) delay = 5;	// multiple O.S. boot: force delay on boot
	itoa(delay, boot_delay, 10);

	BOOL res = WritePrivateProfileSection("operating systems", os_section_buff, boot_ini_file);
	WritePrivateProfileString("boot loader", "timeout", boot_delay, boot_ini_file);
	WritePrivateProfileString(NULL, NULL, NULL, boot_ini_file);	// flush cache

	if (dwFileAttributes & FILE_ATTRIBUTE_READONLY)
		SetFileAttributes(boot_ini_file, dwFileAttributes);
	return res;
}

static void BootIni_UnloadOsInfoStruct(void)
{
	while (os_info_size > 0)
		free(os_info[os_info_size--]._alloc_data);
	os_info_size = 0;
	memset(&os_info, 0, sizeof(os_info));
}

static void BootIni_RebuildOsInfoStruct(void)
{
	BootIni_UnloadOsInfoStruct();	// free any previously allocated data on 'os_info' struct
	char	*p;
	for (p = os_section_buff, os_info_size = 0; *p; ) {
		if (!BootIni_ConvertOsSectionToOsInfo(&p, &os_info[os_info_size]))
			continue;
		BootIni_ParseOsInfo(&os_info[os_info_size]);
		os_info_size++;
	}
	BootIni_LinkOsEntries(os_info, os_info_size);
}

static BOOL BootIni_LoadOsInfoStruct(void)
{
	if (!BootIni_LoadOsSectionBuff())
		return FALSE;
	BootIni_RebuildOsInfoStruct();
	return TRUE;
}

static BOOL BootIni_SaveOsInfoStruct(void)
{
	if (!BootIni_SaveOsSectionBuff((os_info_size > 1) ? TRUE : FALSE))
		return FALSE;
	return TRUE;
}

static BOOL BootIni_EnableOsInfoEntry(int idx_entry)
{
	char	tmp_entry[512];
	int		tmp_entry_size;

	if (idx_entry < 0 || idx_entry >= os_info_size)
		return FALSE;
	OS_INFO *ptr_cur = &os_info[idx_entry];
	if (!ptr_cur->is_NT ||		// not an NT entry
		ptr_cur->is_3gb ||		// entry is already '/3gb' enabled
		ptr_cur->idx_lnk != -1)	// a linked '/3gb' enabled entry is already present
		return TRUE;
	// make room in 'os_section_buff' for the new data & copy data AFTER current entry
	sprintf(tmp_entry, "%s=\"%s%s\"%s%s", ptr_cur->os_boot,
		ptr_cur->os_label, " [/3GB Enabled]",
		ptr_cur->os_flags, " /3gb /userva=2944");
	tmp_entry_size = strlen(tmp_entry) + 1;
	assert((ptr_cur->_section_ptr >= &os_section_buff[0]) && (ptr_cur->_section_ptr <= &os_section_buff[sizeof(os_section_buff) - 1]));	// '_section_ptr' MUST BE inside 'os_section_buff' data area
	char	*p = ptr_cur->_section_ptr + strlen(ptr_cur->_section_ptr) + 1;
	memmove(p + tmp_entry_size, p, sizeof(os_section_buff) - ((p + tmp_entry_size) - os_section_buff));
	memcpy(p, tmp_entry, tmp_entry_size);
	// rebuild the updated 'os_info' struct from scratch
	BootIni_RebuildOsInfoStruct();
	return TRUE;
}

static BOOL BootIni_DisableOsInfoEntry(int idx_entry)
{
	if (idx_entry < 0 || idx_entry >= os_info_size)
		return FALSE;
	OS_INFO *ptr_cur = &os_info[idx_entry];
	if (!ptr_cur->is_NT ||		// not an NT entry
		(!ptr_cur->is_3gb &&		// entry is NOT '/3gb' enabled
		 ptr_cur->idx_lnk == -1))	// a linked '/3gb' enabled entry is NOT present
		return TRUE;
	if (!ptr_cur->is_3gb)		// if currently selected entry is not the '/3gb' enabled one, select the right one
		ptr_cur = &os_info[idx_entry = ptr_cur->idx_lnk];
	assert((ptr_cur->_section_ptr >= &os_section_buff[0]) && (ptr_cur->_section_ptr <= &os_section_buff[sizeof(os_section_buff) - 1]));	// '_section_ptr' MUST BE inside 'os_section_buff' data area
	char	*p = ptr_cur->_section_ptr + strlen(ptr_cur->_section_ptr) + 1;
	// copy back data from 'os_section_buff' OVER current entry
	memmove(ptr_cur->_section_ptr, p, sizeof(os_section_buff) - (p - os_section_buff));
	// rebuild the updated 'os_info' struct from scratch
	BootIni_RebuildOsInfoStruct();
	return TRUE;
}

//******************************************************************************

BOOL CheckIfSingleNTOsBootSystem(BOOL *is_single, BOOL *is_3gb)
{
	int		num_unique_NTOs_entries = 0;
	BOOL	has_3gb_enabled_entries = FALSE;

	if (!BootIni_LoadOsInfoStruct())
		return FALSE;

	for (int idx = 0; idx < os_info_size; idx++) {
		if (!os_info[idx].is_NT)
			continue;
		if (os_info[idx].is_3gb)
			has_3gb_enabled_entries = TRUE;
		if (os_info[idx].idx_lnk == -1 || os_info[idx].idx_lnk > idx)
			num_unique_NTOs_entries++;
	}
	BootIni_UnloadOsInfoStruct();

	if (is_single != NULL) *is_single = (num_unique_NTOs_entries != 1) ? FALSE : TRUE;
	if (is_3gb != NULL) *is_3gb = has_3gb_enabled_entries;
	return TRUE;
}

//******************************************************************************

static void BootIni_AllocListView(HWND hwnd)
{
	int		idx;
	static WORD	lv_icons[2] = { IDI_ICON_DISABLED, IDI_ICON_ENABLED };
	static int lv_column_widths[4] = { 35, 205, 205, 100 };
	static const char *lv_column_titles[4] = { "/3gb", "Boot Unit", "OS Label", "Boot Flags" };

	HWND hLV = GetDlgItem(hwnd, IDC_LIST_BOOTINI);
	ListView_DeleteAllItems(hLV);
	ListView_SetExtendedListViewStyle(hLV, LVS_EX_FULLROWSELECT);

	for (idx = 0; idx < 4; idx++)
	{
		LV_COLUMN lvc;
		memset(&lvc, 0, sizeof(lvc));
		lvc.mask = LVCF_FMT | LVCF_WIDTH | LVCF_TEXT | LVCF_SUBITEM;
		lvc.fmt = LVCFMT_LEFT;
		lvc.cx = lv_column_widths[idx];
		lvc.iSubItem = idx;
		lvc.pszText = (char *)lv_column_titles[idx];
		ListView_InsertColumn(hLV, idx, &lvc);
	}

	HIMAGELIST hImageListLarge = ImageList_Create(GetSystemMetrics(SM_CXICON), GetSystemMetrics(SM_CYICON), TRUE, 1, 1);
	HIMAGELIST hImageListSmall = ImageList_Create(GetSystemMetrics(SM_CXSMICON), GetSystemMetrics(SM_CYSMICON), TRUE, 1, 1);
	for (idx = 0; idx < (sizeof(lv_icons) / sizeof(lv_icons[0])); idx++) {
		HICON hIcon = LoadIcon(hInstance, MAKEINTRESOURCE(lv_icons[idx]));
		ImageList_AddIcon(hImageListLarge, hIcon);
		ImageList_AddIcon(hImageListSmall, hIcon);
	}
	ListView_SetImageList(hLV, hImageListLarge, LVSIL_NORMAL);
	ListView_SetImageList(hLV, hImageListSmall, LVSIL_SMALL);
}

static void BootIni_FreeListView(HWND hwnd)
{
	HWND hLV = GetDlgItem(hwnd, IDC_LIST_BOOTINI);
	HIMAGELIST hImageListLarge = ListView_GetImageList(hLV, LVSIL_NORMAL);
	HIMAGELIST hImageListSmall = ListView_GetImageList(hLV, LVSIL_SMALL);
	ImageList_Destroy(hImageListLarge);
	ImageList_Destroy(hImageListSmall);
}

static BOOL BootIni_LoadBootIniFileToListView(HWND hwnd)
{
	if (!BootIni_LoadOsInfoStruct())
		return FALSE;

	// parse section buffer & fill listview entries
	HWND hLV = GetDlgItem(hwnd, IDC_LIST_BOOTINI);
	for (int idx = 0; idx < os_info_size; idx++) {
		LV_ITEM	lvi;
		memset(&lvi, 0, sizeof(lvi));
		lvi.mask = LVIF_IMAGE;
		lvi.iItem = ListView_GetItemCount(hLV);
		lvi.iSubItem  = 0;
		lvi.iImage = -1;
		ListView_InsertItem(hLV, &lvi);
		ListView_SetItemText(hLV, lvi.iItem, 1, os_info[idx].os_boot);
		ListView_SetItemText(hLV, lvi.iItem, 2, os_info[idx].os_label);
		ListView_SetItemText(hLV, lvi.iItem, 3, os_info[idx].os_flags);
	}
	return TRUE;
}

static BOOL BootIni_SaveListViewToBootIniFile(HWND hwnd)
{
	return BootIni_SaveOsInfoStruct();
}

static void BootIni_UpdateListViewImages(HWND hwnd)
{
	int		idx, img;

	HWND	hLV = GetDlgItem(hwnd, IDC_LIST_BOOTINI);
	int		lvc = ListView_GetItemCount(hLV);
	for (idx = 0; idx < lvc; idx++) {
		LV_ITEM	lvi;
		memset(&lvi, 0, sizeof(lvi));
		lvi.mask = LVIF_IMAGE;
		lvi.iItem = idx;
		lvi.iSubItem = 0;
		ListView_GetItem(hLV, &lvi);
		assert(lvc == os_info_size);
		if (!os_info[idx].is_NT)
			img = -1;	// not a Win2K/XP boot partition (no image)
		else if (!os_info[idx].is_3gb && os_info[idx].idx_lnk == -1)
			img = 0;	// option '/3gb' not present and no linked entry (image 'KO')
		else
			img = 1;	// option '/3gb' present or linked entry (image 'OK')

		if (img == lvi.iImage)
			continue;
		lvi.iImage = img;
		ListView_SetItem(hLV, &lvi);
	}
}

static BOOL BootIni_EnableSelectedListViewEntry(HWND hwnd, BOOL bEnable)
{
	int		idx;

	HWND hLV = GetDlgItem(hwnd, IDC_LIST_BOOTINI);
	if ((idx = ListView_GetNextItem(hLV, -1, LVNI_SELECTED)) < 0)
	{
		MsgBox(hwnd, IDS_NO_ENTRY_SELECTED_TEXT,
			IDS_NO_ENTRY_SELECTED_CAPTION, MB_OK | MB_ICONWARNING);
		return FALSE;	// no entry selected
	}
	assert(idx < os_info_size);
	if (!os_info[idx].is_NT) {
		MsgBox(hwnd, IDS_NOT_AN_XP_BOOT_PARTITION_TEXT,
			IDS_NOT_AN_XP_BOOT_PARTITION_CAPTION, MB_OK | MB_ICONINFORMATION);
		return FALSE;	// not a Win2K/XP boot partition
	}
	if (bEnable && !os_info[idx].is_3gb && os_info[idx].idx_lnk == -1) {

		BootIni_EnableOsInfoEntry(idx);

		LV_ITEM	lvi;
		memset(&lvi, 0, sizeof(lvi));
		lvi.mask = LVIF_IMAGE;
		lvi.iItem = idx + 1;
		lvi.iSubItem  = 0;
		lvi.iImage = -1;
		ListView_InsertItem(hLV, &lvi);
		ListView_SetItemText(hLV, lvi.iItem, 1, os_info[idx + 1].os_boot);
		ListView_SetItemText(hLV, lvi.iItem, 2, os_info[idx + 1].os_label);
		ListView_SetItemText(hLV, lvi.iItem, 3, os_info[idx + 1].os_flags);
		BootIni_UpdateListViewImages(hwnd);
		MsgBox(hwnd, IDS_ENABLED_3GB_OPTION_TEXT,
			IDS_ENABLED_3GB_OPTION_CAPTION, MB_OK | MB_ICONINFORMATION);
	} else if (!bEnable && os_info[idx].idx_lnk != -1) {

		ListView_DeleteItem(hLV, (os_info[idx].is_3gb) ? idx : os_info[idx].idx_lnk);
		BootIni_DisableOsInfoEntry(idx);
		BootIni_UpdateListViewImages(hwnd);
		MsgBox(hwnd, IDS_DISABLED_3GB_OPTION_TEXT,
			IDS_DISABLED_3GB_OPTION_CAPTION, MB_OK | MB_ICONINFORMATION);
	} else {
		return FALSE;	// no changes have been done
	}
	return TRUE;		// options have changed
}

BOOL CALLBACK BootIniDialogProc(HWND hwnd, UINT msg, WPARAM wPar, LPARAM lPar)
{
	switch (msg) {
	case WM_INITDIALOG: {
		BootIni_AllocListView(hwnd);
		while (!BootIni_LoadBootIniFileToListView(hwnd)) {
			if (MsgBox(hwnd, IDS_ERROR_LOADING_BOOT_INI_TEXT,
					IDS_ERROR_LOADING_BOOT_INI_CAPTION, MB_OKCANCEL | MB_ICONWARNING) != IDOK) {
				EndDialog(hwnd, 0);
				return TRUE;
			}
			OPENFILENAME ofn;
			char	base_dir[MAX_PATH];
			memset(&ofn, 0, sizeof(ofn));
			ofn.lStructSize = sizeof(ofn);
			ofn.hwndOwner = hwnd;
			ofn.lpstrFilter = "boot.ini file\0boot.ini\0";
			ofn.lpstrFile = boot_ini_file;
			ofn.nMaxFile = sizeof(boot_ini_file);
            ofn.lpstrInitialDir = SetupOFNInitialDir(base_dir, ofn.lpstrFile);
			ofn.lpstrTitle = "Select the 'boot.ini' file to open:";
			ofn.Flags = OFN_FILEMUSTEXIST | OFN_NOCHANGEDIR | OFN_EXPLORER;
			if (!GetOpenFileName(&ofn)) {
				EndDialog(hwnd, 0);
				return TRUE;
			}
		}
		BootIni_UpdateListViewImages(hwnd);
		EnableWindow(GetDlgItem(hwnd, IDOK), FALSE);
		} return TRUE;
	case WM_DESTROY: {
		BootIni_FreeListView(hwnd);
		EndDialog(hwnd, 0);
		} return TRUE;
	case WM_COMMAND:
		switch (LOWORD(wPar)) {
		case IDC_BUTTON_ENABLE:
			if (BootIni_EnableSelectedListViewEntry(hwnd, TRUE))
				EnableWindow(GetDlgItem(hwnd, IDOK), TRUE);
			break;
		case IDC_BUTTON_DISABLE:
			if (BootIni_EnableSelectedListViewEntry(hwnd, FALSE))
				EnableWindow(GetDlgItem(hwnd, IDOK), TRUE);
			break;
		case IDOK:
			if (!BootIni_SaveListViewToBootIniFile(hwnd)) {
				MsgBox(hwnd, IDS_ERROR_SAVING_BOOT_INI_TEXT,
					IDS_ERROR_SAVING_BOOT_INI_CAPTION, MB_OK | MB_ICONWARNING);
				break;	// don't close the dialog
			}
			MsgBox(hwnd, IDS_SAVED_BOOT_INI_TEXT,
				IDS_SAVED_BOOT_INI_CAPTION, MB_OK | MB_ICONINFORMATION);
		case IDCANCEL:
			EndDialog(hwnd,0);
			break;
		}
		break;
	default:
		break;
	}
	return FALSE;
}

static BOOL BootIni_Toggle3GBOption_SingleNTOs(HWND hwnd)
{
	int idx, choice;
	BOOL is_3gb_enabled = FALSE;

	if (!BootIni_LoadOsInfoStruct())
		return FALSE;

	for (idx = 0; idx < os_info_size; idx++) {
		OS_INFO *ptr_cur = &os_info[idx];
		if (!ptr_cur->is_NT)	// not an NT entry
			continue;
		if (ptr_cur->is_3gb ||		// entry is already '/3gb' enabled
			ptr_cur->idx_lnk != -1)	// a linked '/3gb' enabled entry is already present
			is_3gb_enabled = TRUE;
		break;
	}
	assert(idx < os_info_size);

	if (!is_3gb_enabled) {
		choice = MsgBox(hwnd, IDS_ENABLE_3GB_OPTION_TEXT,
			IDS_ENABLE_3GB_OPTION_CAPTION, MB_YESNOCANCEL);
	} else {
		choice = MsgBox(hwnd, IDS_DISABLE_3GB_OPTION_TEXT,
			IDS_DISABLE_3GB_OPTION_CAPTION, MB_YESNOCANCEL);
	}

	if (choice == IDYES)
	{
		if (is_3gb_enabled)
			BootIni_DisableOsInfoEntry(idx);
		else
			BootIni_EnableOsInfoEntry(idx);

		if (!BootIni_SaveOsInfoStruct()) {
			MsgBox(hwnd, IDS_ERROR_SAVING_BOOT_INI_TEXT,
				IDS_ERROR_SAVING_BOOT_INI_CAPTION, MB_OK | MB_ICONWARNING);
		} else {
			MsgBox(hwnd, IDS_SAVED_BOOT_INI_TEXT,
				IDS_SAVED_BOOT_INI_CAPTION, MB_OK | MB_ICONINFORMATION);
			EndDialog(hwnd,0);
		}
	}
	BootIni_UnloadOsInfoStruct();
	return (choice != IDCANCEL) ? TRUE : FALSE;
}

//******************************************************************************

BOOL CALLBACK MainDialogProc(HWND hwnd, UINT msg, WPARAM wPar, LPARAM lPar)
{
	switch (msg) {
	case WM_INITDIALOG:
		if (!is_3gb_os) {
			if (bootini_has_3gb_enabled_entry) {
				MsgBox(NULL, IDS_NO_3GB_ENABLED_BOOT_TEXT,
					IDS_NO_3GB_ENABLED_BOOT_CAPTION, MB_OK | MB_ICONINFORMATION);
			} else {
				MsgBox(NULL, IDS_NO_3GB_ENABLED_OS_TEXT,
					IDS_NO_3GB_ENABLED_OS_CAPTION, MB_OK | MB_ICONINFORMATION);
			}
			EnableWindow(GetDlgItem(hwnd, IDOK), FALSE);
			EnableWindow(GetDlgItem(hwnd, IDC_EDIT_BLEEMEXE), FALSE);
			EnableWindow(GetDlgItem(hwnd, IDC_BUTTON_BROWSE), FALSE);
			EnableWindow(GetDlgItem(hwnd, IDC_STATIC_SELECTFILE), FALSE);
			EnableWindow(GetDlgItem(hwnd, IDC_STATIC_INFO1), FALSE);
			EnableWindow(GetDlgItem(hwnd, IDC_STATIC_INFO2), FALSE);
		}
		LoadDefaultValuesFromReg();		// reload last selected .exe file
		SetWindowText(GetDlgItem(hwnd, IDC_EDIT_BLEEMEXE), bleem_exe_file);
		return TRUE;

	case WM_DESTROY:
		EndDialog(hwnd, 0);
		return TRUE;

	case WM_COMMAND:
		switch (LOWORD(wPar)) {
		case IDC_BUTTON_BROWSE: {
			OPENFILENAME ofn;
			char	base_dir[MAX_PATH];
			GetWindowText(GetDlgItem(hwnd, IDC_EDIT_BLEEMEXE), bleem_exe_file, sizeof(bleem_exe_file));
			memset(&ofn, 0, sizeof(ofn));
			ofn.lStructSize = sizeof(ofn);
			ofn.hwndOwner = hwnd;
			ofn.lpstrFilter = "Executable Files (*.exe)\0*.exe\0";
			ofn.lpstrFile = bleem_exe_file;
			ofn.nMaxFile = sizeof(bleem_exe_file);
            ofn.lpstrInitialDir = SetupOFNInitialDir(base_dir, ofn.lpstrFile);
			ofn.lpstrTitle = "Select the 'Bleem!.exe' file you want to run:";
			ofn.Flags = OFN_FILEMUSTEXIST | OFN_NOCHANGEDIR | OFN_EXPLORER;
			if (!GetOpenFileName(&ofn))
				break;
			SetWindowText(GetDlgItem(hwnd, IDC_EDIT_BLEEMEXE), bleem_exe_file);
			} break;
		case IDC_BUTTON_SETUP3GB:
			if (os_version.dwPlatformId == VER_PLATFORM_WIN32_NT &&
				os_version.dwMajorVersion == 5 &&
				os_version.dwMinorVersion == 1 &&	// 5.1 = WinXP
				os_version.wServicePackMajor < 2) {	// SP < 2
				MsgBox(hwnd, IDS_XPSP1_NTLDR_BUG_TEXT,
					IDS_XPSP1_NTLDR_BUG_CAPTION, MB_OK | MB_ICONERROR);
				break;
			}
			if (!bootini_has_single_NTOs_entry ||
				!BootIni_Toggle3GBOption_SingleNTOs(hwnd))
				DialogBox(hInstance, MAKEINTRESOURCE(IDD_DIALOG_BOOTINI), hwnd, (DLGPROC)BootIniDialogProc);
			break;
		case IDOK:
			GetWindowText(GetDlgItem(hwnd, IDC_EDIT_BLEEMEXE), bleem_exe_file, sizeof(bleem_exe_file));
			if (!IsAValidBleemExeFile(bleem_exe_file)) {
				MsgBox(hwnd, IDS_ERROR_OPENING_BLEEM_EXE_TEXT,
					IDS_ERROR_OPENING_BLEEM_EXE_CAPTION, MB_OK | MB_ICONWARNING);
				break;	// don't close the dialog
			}
			SaveDefaultValuesToReg();	// save last selected .exe file
			StartBleemExeFile(bleem_exe_file);
		case IDCANCEL:
			EndDialog(hwnd,0);
			break;
		}
		break;

	default:
		break;
	}
	return FALSE;
}

//******************************************************************************

BOOL ExecFromCmdLine(int argc, char *argv[])
{
	int		idx;
	char	*p;

	// process cmdline options
	for (idx = 1; idx < argc; idx++) {
		p = argv[idx];
		if (*p != '-' && *p != '/')
			break;
		p++;
		if (*p == '?' || *p == 'h') {
			MsgBox(NULL, IDS_CMDLINE_HELP_TEXT,
				IDS_CMDLINE_HELP_CAPTION, MB_OK | MB_ICONINFORMATION);
			return TRUE;
		//} else if (...) {	// add here any other 'option' processing
		}
	}
	// process .exe filename path
	if (idx < argc)
		strcpy(bleem_exe_file, argv[idx++]);
	// process .exe command line args
	for (p = bleem_cmdline_args; idx < argc; idx++) {
		if (*bleem_cmdline_args)
			*p++ = ' ';
		p += sprintf(p, "\"%s\"", argv[idx]);
	}

	if (!is_3gb_os) {
		if (bootini_has_3gb_enabled_entry) {
			MsgBox(NULL, IDS_NO_3GB_ENABLED_BOOT_TEXT,
				IDS_NO_3GB_ENABLED_BOOT_CAPTION, MB_OK | MB_ICONINFORMATION);
		} else {
			MsgBox(NULL, IDS_NO_3GB_ENABLED_OS_TEXT,
				IDS_NO_3GB_ENABLED_OS_CAPTION, MB_OK | MB_ICONINFORMATION);
		}
		return FALSE;
	}

	if (!IsAValidBleemExeFile(bleem_exe_file)) {
		MsgBox(NULL, IDS_ERROR_OPENING_BLEEM_EXE_TEXT,
			IDS_ERROR_OPENING_BLEEM_EXE_CAPTION, MB_OK | MB_ICONERROR);
		return FALSE;
	}
	StartBleemExeFile(bleem_exe_file);
	return TRUE;
}

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
	::hInstance = hInstance;
	memset(&os_version, 0, sizeof(os_version));
	// read OS version (first extended, then standard if not supported)
	os_version.dwOSVersionInfoSize = sizeof(OSVERSIONINFOEX);
	if (GetVersionEx((OSVERSIONINFO *)&os_version) == FALSE) {
		os_version.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
		GetVersionEx((OSVERSIONINFO *)&os_version);
	}
	// check '/3gb enabled' OS capability
	is_3gb_os = test_3gb_os();
	// check boot.ini 'operating systems' section content
	CheckIfSingleNTOsBootSystem(&bootini_has_single_NTOs_entry, &bootini_has_3gb_enabled_entry);

	if (__argc <= 1) {
		InitCommonControls();
		DialogBox(hInstance, MAKEINTRESOURCE(IDD_DIALOG_MAIN), NULL, (DLGPROC)MainDialogProc);
	} else {
		ExecFromCmdLine(__argc, __argv);
	}
	return 0;
}
