
#include	"bleem_nt.h"

// SPTI interface
#include	"winioctl.h"
#include	"ntddscsi.h"
// SCSI commands & CDB structs
#include	"ntddcdrm.h"
#include	"scsi.h"
//// ASPI interface
//#include	"wnaspi32.h"
//#include	"scsidefs.h"
// MSCDEX interface
#undef LOCK_UNLOCK_DOOR	// (already defined in 'winioctl.h')
#include	"mscdex.h"
#include	"iso_fs.h"

// MSCDEX version:
#define	MSCDEX_VERSION_HI	2
#define	MSCDEX_VERSION_LO	23

// DEVICE DRIVER REQUEST status bits:
#define	REQUEST_STATUS_DONE		0x0100
#define	REQUEST_STATUS_BUSY		0x0200
#define	REQUEST_STATUS_ERROR	0x8000
// DEVICE DRIVER REQUEST errors are translated in range 0x00 - 0x0F from 'ERROR_WRITE_PROTECT' to 'ERROR_WRONG_DISK'
#define	STATUS_ERR(err)	(REQUEST_STATUS_ERROR | (BYTE)(err - ERROR_WRITE_PROTECT))

// MEDIA CHANGE values (see 'DRIVE_POINTER.media_change'):
#define	MEDIA_HAS_CHANGED	0xFF		// media has changed
#define	MEDIA_NOT_CHANGED	0x01		// media has not changed
#define	MEDIA_NOT_SURE		0x00		// not sure

// ADDRESSING MODE values (HSG / Red Book):
#define	ADDR_MODE_HSG		0x00		// HSG (logical sector nr)
#define	ADDR_MODE_RB		0x01		// Red Book (frame/second/minute/unused (logical sector nr = minute * 4500 + second * 75 + frame - 150))

// READING MODES values (COOKED / RAW):
#define	READ_MODE_COOKED	0x00		// cooked (2048 bytes/sector)
#define	READ_MODE_RAW		0x01		// raw (2352 bytes/sector)

// Other defines:
#define	NUM_DRIVES			26			// 'Z'-'A'+1
#define	COOKED_SECTOR_SIZE	2048
#define	RAW_SECTOR_SIZE		2352
#define	SCSI_CDROM_TIMEOUT	10
#define MSF_OFFSET			(2 * 75)	// offset of first cdrom sector (0:2:0)

//******************************************************************************

// ISO9660 / High Sierra volume descriptors:
typedef union {
	ISO_VOLUME_DESCRIPTOR	iso_vd;
	ISO_PRIMARY_DESCRIPTOR	iso_pd;
	ISO_SUPPLEMENTARY_DESCRIPTOR	iso_sd;
//	HS_VOLUME_DESCRIPTOR	hs_vd;
//	HS_PRIMARY_DESCRIPTOR	hs_pd;
//	HS_SUPPLEMENTARY_DESCRIPTOR		hs_sd;
	BYTE	raw_data[COOKED_SECTOR_SIZE];
} VOLUME_DESCRIPTOR_UNION;

// SCSI_PASS_THROUGH_DIRECT with SENSE_DATA buffer struct:
typedef struct {
	SCSI_PASS_THROUGH_DIRECT	sptd;
	ULONG	Filler;		// realign buffer to double word boundary
	SENSE_DATA	SenseBuf;
	UCHAR	ucAdditionalSenseBuf[32 - SENSE_BUFFER_SIZE];
} SCSI_PASS_THROUGH_DIRECT_WITH_BUFFER;

// DRIVE_POINTER struct:
typedef struct {
	HANDLE	drive_handle;
	BYTE	unit_nr;
	BYTE	subunit_nr;
	BYTE	media_changed;				// 0xFF (media has changed), 0x01 (media has not changed), 0x00 (not sure)
	BOOLEAN	playing;
	BOOLEAN	paused;
	BOOLEAN	trayopen;
	BOOLEAN	unlocked;
	BYTE	last_audio_status;
	BOOLEAN	readed_toc;
//	BOOLEAN	readed_vol_descr;
//	BOOLEAN	use_suppl_vol_descr;
//	BYTE	suppl_vol_descr_pref;
	BOOLEAN	ioctl_res;					// GetAudioStatus() ioctl result (see 'GetAudioStatus()')
	DWORD	error_code;
	DWORD	translated_error_code;
	DWORD	play_start;
	DWORD	play_end;
	SUB_Q_CURRENT_POSITION	subqpos;
	CDROM_TOC	toc;
//	VOL_DESCR_DATA	vol_descr;
} DRIVE_POINTER;

//******************************************************************************

static DRIVE_POINTER	*DrivePointers[NUM_DRIVES];	// ptrs to drives data struct
static int		FirstDrive		= -1;			// first CD-ROM drive (-1=not assigned, 0='A', ...)
static int		NumDrives		= 0;			// number of CD-ROM drives
static BOOL		need_scan_drives = TRUE;

#ifdef	USE_DUMP_BLI_FILE
HANDLE	dump_bli_file = INVALID_HANDLE_VALUE;
#endif//USE_DUMP_BLI_FILE

//******************************************************************************

// Function prototypes:
HANDLE	OpenPhysicalDrive(int drive);
void	ClosePhysicalDrive(DRIVE_POINTER *DrvInfo);
BOOL	GetAudioStatus(DRIVE_POINTER *DrvInfo);
BOOL	ReadTOC(DRIVE_POINTER *DrvInfo);
BOOL	ReadSectorCooked(DRIVE_POINTER *DrvInfo, BYTE *sector_data, DWORD sector_nr, DWORD num_read_sectors);
DWORD	ProcessError(DRIVE_POINTER *DrvInfo, BOOL close_drive);
DRIVE_POINTER	*GetDrivePointer(int drive, BOOL open_drive_handle);
int		MSF2INT(const MSF *msf);
void	INT2MSF(int i, MSF *msf);
int		RB2INT(const RB_ADDR *fsm);
void	INT2RB(int i, RB_ADDR *fsm);
//void	ApiGetNumberOfCDROMDrives(void);
//void	ApiGetDriveDeviceList(void);
//void	ApiGetCopyrightFileName(void);
void	ApiReadVTOC(void);
//void	ApiTurnDebuggingOn(void);
//void	ApiAbsoluteDiskRead(void);
//void	ApiAbsoluteDiskWrite(void);
//void	ApiReserved(void);
void	ApiDriveCheck(void);
void	ApiGetMSCDEXVersion(void);
//void	ApiGetCDROMDriveLetters(void);
//void	ApiGetSetVolumeDescriptorPreference(void);
//void	ApiGetDirectoryEntry(void);
void	ApiSendDeviceDriverRequest(void);
void	DevIoctlInput(DRIVE_POINTER *DrvInfo, DEVICE_REQUEST *VdmReq);
//void	DevInputFlush(DRIVE_POINTER *DrvInfo, DEVICE_REQUEST *VdmReq);
void	DevIoctlOutput(DRIVE_POINTER *DrvInfo, DEVICE_REQUEST *VdmReq);
//void	DevOpen(DRIVE_POINTER *DrvInfo, DEVICE_REQUEST *VdmReq);
void	DevReadLong(DRIVE_POINTER *DrvInfo, DEVICE_REQUEST *VdmReq);
//void	DevReadLongPrefetch(DRIVE_POINTER *DrvInfo, DEVICE_REQUEST *VdmReq);
//void	DevSeek(DRIVE_POINTER *DrvInfo, DEVICE_REQUEST *VdmReq);
void	DevPlayAudio(DRIVE_POINTER *DrvInfo, DEVICE_REQUEST *VdmReq);
void	DevStopAudio(DRIVE_POINTER *DrvInfo, DEVICE_REQUEST *VdmReq);
//void	DevWriteLong(DRIVE_POINTER *DrvInfo, DEVICE_REQUEST *VdmReq);
void	DevResumeAudio(DRIVE_POINTER *DrvInfo, DEVICE_REQUEST *VdmReq);

//******************************************************************************

void cdrom_init(void)
{
	int		old_error_mode, drive;
	HANDLE	hdrv;
	char	root_name[4] = "?:\\";

	if (!need_scan_drives)
		return;
	need_scan_drives = FALSE;

	old_error_mode = SetErrorMode(SEM_FAILCRITICALERRORS);
	for (drive = 0; drive < NUM_DRIVES; drive++) {
		DRIVE_POINTER	*DrvInfo;
		root_name[0] = drive + 'A';
		if (GetDriveType(root_name) != DRIVE_CDROM)
			continue;
		if ((hdrv = OpenPhysicalDrive(drive)) == INVALID_HANDLE_VALUE) {
			DrivePointers[drive] = NULL;
			continue;
		}
		if ((DrvInfo = (DRIVE_POINTER *)HeapAlloc(hProcessHeap, HEAP_ZERO_MEMORY, sizeof(DRIVE_POINTER))) == NULL) {
			CloseHandle(hdrv);
			cdrom_free();
			return;
		}
		DrivePointers[drive] = DrvInfo;
		DrvInfo->drive_handle = hdrv;
		DrvInfo->unit_nr = drive;
		DrvInfo->subunit_nr = NumDrives;
		DrvInfo->media_changed = MEDIA_HAS_CHANGED;
		DrvInfo->unlocked = TRUE;
		DrvInfo->last_audio_status = AUDIO_STATUS_NO_STATUS;
		DrvInfo->readed_toc = FALSE;
//		DrvInfo->readed_vol_descr = FALSE;
		DrvInfo->play_start = 0;
		DrvInfo->play_end = 0;
		GetAudioStatus(DrvInfo);	//init playing, paused, trayopen & ioctl_res
//		DrvInfo->use_suppl_vol_descr = FALSE;
//		DrvInfo->suppl_vol_descr_pref = 0;
		NumDrives++;
		if (FirstDrive == -1)
			FirstDrive = drive;
		ClosePhysicalDrive(DrvInfo);	// close 'hdrv' handle
	}
}

void cdrom_free(void)
{
	int		drive;

	for (drive = 0; drive < NUM_DRIVES; drive++) {
		if (DrivePointers[drive] == NULL)
			continue;
		ClosePhysicalDrive(DrivePointers[drive]);
		HeapFree(hProcessHeap, 0, DrivePointers[drive]);
		DrivePointers[drive] = NULL;
	}
	FirstDrive = -1;
	NumDrives = 0;
}

//******************************************************************************

HANDLE OpenPhysicalDrive(int drive)
{
	char	drive_name[7] = "\\\\.\\?:";
	HANDLE	hDrv;

	drive_name[4] = drive + 'A';
	// try to open drive twice: first RW, then RO (RW mode is required for IOCTL SCSI commands)
	if ((hDrv = CreateFile(drive_name, GENERIC_READ | GENERIC_WRITE, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL)) == INVALID_HANDLE_VALUE)
		hDrv = CreateFile(drive_name, GENERIC_READ, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
	return hDrv;
}

void ClosePhysicalDrive(DRIVE_POINTER *DrvInfo)
{
	if (DrvInfo->drive_handle == INVALID_HANDLE_VALUE)
		return;
	CloseHandle(DrvInfo->drive_handle);
	DrvInfo->drive_handle = INVALID_HANDLE_VALUE;
	// invalidate cached disk info
	DrvInfo->readed_toc = FALSE;
//	DrvInfo->readed_vol_descr = FALSE;
}

BOOL GetAudioStatus(DRIVE_POINTER *DrvInfo)
{
	DWORD	dwOutByteCount;
	static CDROM_SUB_Q_DATA_FORMAT	req = { IOCTL_CDROM_CURRENT_POSITION, 0 };
	SUB_Q_CHANNEL_DATA	data;

	DrvInfo->playing = DrvInfo->paused = DrvInfo->trayopen = FALSE;
	if (!(DrvInfo->ioctl_res = DeviceIoControl(DrvInfo->drive_handle, IOCTL_CDROM_READ_Q_CHANNEL, &req, sizeof(req), &data, sizeof(data), &dwOutByteCount, NULL))) {
		ProcessError(DrvInfo, FALSE);	// do NOT close drive !!!
		return FALSE;
	}
	DrvInfo->subqpos = data.CurrentPosition;
	switch (DrvInfo->subqpos.Header.AudioStatus) {
	case AUDIO_STATUS_IN_PROGRESS:
		DrvInfo->playing = TRUE;
		DrvInfo->paused = FALSE;
		DrvInfo->last_audio_status = AUDIO_STATUS_IN_PROGRESS;
		break;
	case AUDIO_STATUS_PAUSED:
		if (DrvInfo->last_audio_status != AUDIO_STATUS_IN_PROGRESS)
			break;
		DrvInfo->playing = FALSE;
		DrvInfo->paused = TRUE;
		break;
	case AUDIO_STATUS_PLAY_COMPLETE:
	case AUDIO_STATUS_PLAY_ERROR:
		DrvInfo->playing = DrvInfo->paused = FALSE;
		DrvInfo->last_audio_status = DrvInfo->subqpos.Header.AudioStatus;
		break;
	default:
		break;
	}
	return TRUE;
}

BOOL ReadTOC(DRIVE_POINTER *DrvInfo)
{
	DWORD	dwOutByteCount;

	if (DrvInfo->readed_toc)
		return TRUE;
	if (!DeviceIoControl(DrvInfo->drive_handle, IOCTL_CDROM_READ_TOC, NULL, 0, &DrvInfo->toc, sizeof(DrvInfo->toc), &dwOutByteCount, NULL)) {
		ProcessError(DrvInfo, TRUE);
		return FALSE;
	}
	DrvInfo->readed_toc = TRUE;
	DrvInfo->media_changed = MEDIA_NOT_CHANGED;
	return TRUE;
}

BOOL ReadSectorCooked(DRIVE_POINTER *DrvInfo, BYTE *sector_data, DWORD sector_nr, DWORD num_read_sectors)
{
	DWORD	dwReadByteCount;
	if (SetFilePointer(DrvInfo->drive_handle, sector_nr * COOKED_SECTOR_SIZE, NULL, FILE_BEGIN) == 0xFFFFFFFF ||
		ReadFile(DrvInfo->drive_handle, sector_data, num_read_sectors * COOKED_SECTOR_SIZE, &dwReadByteCount, NULL) == FALSE ||
		dwReadByteCount != (num_read_sectors * COOKED_SECTOR_SIZE)) {
		ProcessError(DrvInfo, TRUE);
		return FALSE;
	}
	return TRUE;
}

BOOL ExecScsiCmd(DRIVE_POINTER *DrvInfo, void *data_buf, DWORD data_len, const CDB *cdb, BYTE cdblen, int data_dir)
{
	SCSI_PASS_THROUGH_DIRECT_WITH_BUFFER	sb;
	SCSI_ADDRESS	sa;
	DWORD	dwOutByteCount, retries, error;
	BOOL	retry, delay;

	memset(&sa, 0, sizeof(sa));
	sa.Length = sizeof(sa);
	if (!DeviceIoControl(DrvInfo->drive_handle, IOCTL_SCSI_GET_ADDRESS, NULL, 0, &sa, sizeof(sa), &dwOutByteCount, NULL)) {
		ProcessError(DrvInfo, TRUE);
		return FALSE;
	}
	for (retries = 2; ; retries--) {
		memset(&sb, 0, sizeof(sb));
		sb.sptd.Length = sizeof(sb.sptd);
		sb.sptd.PathId = sa.PortNumber;//sa.PathId
		sb.sptd.TargetId = sa.TargetId;
		sb.sptd.Lun = sa.Lun;
		sb.sptd.DataIn = data_dir; // SCSI_IOCTL_DATA_IN / SCSI_IOCTL_DATA_OUT / SCSI_IOCTL_DATA_UNSPECIFIED
		sb.sptd.DataTransferLength = data_len;
		sb.sptd.DataBuffer = data_buf;
		memcpy(sb.sptd.Cdb, cdb, sb.sptd.CdbLength = cdblen);
		sb.sptd.TimeOutValue = SCSI_CDROM_TIMEOUT;
		sb.sptd.SenseInfoLength = SENSE_BUFFER_SIZE;
		sb.sptd.SenseInfoOffset = (int)((char *)&sb.SenseBuf - (char *)&sb);
		if (!DeviceIoControl(DrvInfo->drive_handle, IOCTL_SCSI_PASS_THROUGH_DIRECT, &sb, sizeof(sb), &sb, sizeof(sb), &dwOutByteCount, NULL)) {
			ProcessError(DrvInfo, TRUE);
			break;
		}
		retry = delay = FALSE;			// default: no retry & delay
		if (sb.sptd.ScsiStatus == 0) {	// status = no error
			error = NO_ERROR;
		} else if (sb.sptd.SenseInfoLength != 0) {	// status = check sense area
			switch (sb.SenseBuf.SenseKey) {
			case SCSI_SENSE_RECOVERED_ERROR:	// Recovered Error
				error = NO_ERROR;
				break;
			case SCSI_SENSE_NOT_READY:			// Not Ready
				error = ERROR_NOT_READY;
				retry = TRUE;
				if (sb.SenseBuf.AdditionalSenseCode == SCSI_ADSENSE_LUN_NOT_READY &&
					(sb.SenseBuf.AdditionalSenseCodeQualifier == SCSI_SENSEQ_BECOMING_READY || sb.SenseBuf.AdditionalSenseCodeQualifier == SCSI_SENSEQ_OPERATION_IN_PROGRESS))
					delay = TRUE;		// unit becoming ready: delay before retrying
				break;
			case SCSI_SENSE_ILLEGAL_REQUEST:	// Illegal Request
				error = ERROR_INVALID_FUNCTION;
				break;
			case SCSI_SENSE_UNIT_ATTENTION:		// Unit Attention
				error = ERROR_MEDIA_CHANGED;
				retry = TRUE;
				break;
			default:							// Other error conditions
				error = ERROR_IO_DEVICE;
				break;
			}
		} else {						// status = unspecified error conditions
			error = ERROR_IO_DEVICE;
		}
		if (error == NO_ERROR)
			return TRUE;
		SetLastError(error);			// set error code for 'ProcessError()'
		if (!retry || retries == 0)
			break;
		ProcessError(DrvInfo, FALSE);	// do NOT close drive !!!
		if (delay)
			Sleep(1000);
	}
	ProcessError(DrvInfo, TRUE);
	return FALSE;
}

DWORD ProcessError(DRIVE_POINTER *DrvInfo, BOOL close_drive)
{
	DrvInfo->error_code = GetLastError();
	switch (DrvInfo->error_code) {
	case ERROR_NO_MEDIA_IN_DRIVE:
	case ERROR_NOT_READY:
		DrvInfo->trayopen = TRUE;
		DrvInfo->unlocked = TRUE;
	case ERROR_MEDIA_CHANGED:
		DrvInfo->media_changed = MEDIA_HAS_CHANGED;
		DrvInfo->translated_error_code = ERROR_NOT_READY;
		break;
	default:
		DrvInfo->translated_error_code = ERROR_GEN_FAILURE;
		break;
	}
	DrvInfo->readed_toc = FALSE;
//	DrvInfo->readed_vol_descr = FALSE;
	if (close_drive)
		ClosePhysicalDrive(DrvInfo);
	// ret a 'STATUS_ERR()' translated error code (for ApiSendDeviceDriverRequest() subfunctions only)
	return DrvInfo->translated_error_code;
}

DRIVE_POINTER *GetDrivePointer(int drive, BOOL open_drive_handle)
{
	DRIVE_POINTER	*DrvInfo;

	if (drive >= NUM_DRIVES || (DrvInfo = DrivePointers[drive]) == NULL) {
		return NULL;
	}
	if (!open_drive_handle || DrvInfo->drive_handle != INVALID_HANDLE_VALUE ||
		(DrvInfo->drive_handle = OpenPhysicalDrive(DrvInfo->unit_nr)) != INVALID_HANDLE_VALUE) {
		return DrvInfo;
	}
	// failed opening drive, remove drive from drives list
	HeapFree(hProcessHeap, 0, DrvInfo);
	DrivePointers[drive] = NULL;
	NumDrives--;
	if (drive == FirstDrive) {
		FirstDrive = -1;
		while (++drive < NUM_DRIVES) {
			if (DrivePointers[drive] != NULL) {
				FirstDrive = drive;
				break;
			}
		}
	}
	return NULL;
}

int MSF2INT(const MSF *msf) {
	return (msf->minute * 60 + msf->second) * 75 + msf->frame - MSF_OFFSET;
}

void INT2MSF(int i, MSF *msf) {
	i += MSF_OFFSET;
	msf->frame = i % 75;
	i /= 75;
	msf->second = i % 60;
	msf->minute = i / 60;
}

int RB2INT(const RB_ADDR *fsm) {
	return (fsm->minute * 60 + fsm->second) * 75 + fsm->frame - MSF_OFFSET;
}

void INT2RB(int i, RB_ADDR *fsm) {
	i += MSF_OFFSET;
	fsm->frame = i % 75;
	i /= 75;
	fsm->second = i % 60;
	fsm->minute = i / 60;
	fsm->reserved = 0;
}

//******************************************************************************

#define	BUILD_READ_TOC_CDB(cdb, cdblen, toc_len)				\
	cdb.READ_TOC.OperationCode = SCSIOP_READ_TOC;				\
	cdb.READ_TOC.Msf = CDB_USE_MSF;								\
	cdb.READ_TOC.AllocationLength[0] = (BYTE)(toc_len >> 8);	\
	cdb.READ_TOC.AllocationLength[1] = (BYTE)toc_len;			\
	cdblen = sizeof(cdb.READ_TOC);

#define	BUILD_READ_SECTOR_COOKED_CDB(cdb, cdblen, loc, nsec)	\
	cdb.READ_CD.OperationCode = SCSIOP_READ_CD;					\
	cdb.READ_CD.StartingLBA[1] = (BYTE)(loc >> 16);				\
	cdb.READ_CD.StartingLBA[2] = (BYTE)(loc >> 8);				\
	cdb.READ_CD.StartingLBA[3] = (BYTE)(loc);					\
	cdb.READ_CD.TransferBlocks[2] = (BYTE)(nsec);				\
	cdb.READ_CD.IncludeUserData = 1;							\
	cdblen = sizeof(cdb.READ_CD);

#define	BUILD_READ_SECTOR_RAW_CDB(cdb, cdblen, loc, nsec)		\
	cdb.READ_CD.OperationCode = SCSIOP_READ_CD;					\
	cdb.READ_CD.StartingLBA[1] = (BYTE)(loc >> 16);				\
	cdb.READ_CD.StartingLBA[2] = (BYTE)(loc >> 8);				\
	cdb.READ_CD.StartingLBA[3] = (BYTE)(loc);					\
	cdb.READ_CD.TransferBlocks[2] = (BYTE)(nsec);				\
	cdb.READ_CD.IncludeUserData = 1;							\
	cdb.READ_CD.IncludeEDC = 1;									\
	cdb.READ_CD.HeaderCode = 3;									\
	cdb.READ_CD.IncludeSyncData = 1;							\
	cdblen = sizeof(cdb.READ_CD);

//******************************************************************************

static REAL_MODE_CALL_REGS	*mscdex_regs = NULL;
#define	getAX()		((WORD)mscdex_regs->RealMode_EAX)
#define	getBX()		((WORD)mscdex_regs->RealMode_EBX)
#define	getCX()		((WORD)mscdex_regs->RealMode_ECX)
#define	getDX()		((WORD)mscdex_regs->RealMode_EDX)
#define	getSI()		((WORD)mscdex_regs->RealMode_ESI)
#define	getDI()		((WORD)mscdex_regs->RealMode_EDI)
#define	getES()		(mscdex_regs->RealMode_ES)
#define	getDS()		(mscdex_regs->RealMode_DS)
#define setAX(w)	{ mscdex_regs->RealMode_EAX = (w); }
#define setBX(w)	{ mscdex_regs->RealMode_EBX = (w); }
#define setCX(w)	{ mscdex_regs->RealMode_ECX = (w); }
#define setDX(w)	{ mscdex_regs->RealMode_EDX = (w); }
#define setCF(b)	{ if (b) mscdex_regs->RealMode_Flags |= 0x0001; \
					  else	 mscdex_regs->RealMode_Flags &= ~0x0001; }

void WINAPI MSCDEXRequest(REAL_MODE_CALL_REGS *regs)
{
	// save DPMI interrupt registers for get/set macros
	mscdex_regs = regs;

#ifdef	_DEBUG
	if ((getAX() & 0xFF00) != 0x1500)
		__asm int 3
#endif//_DEBUG

	setCF(0);							// reset carry flag (no error)
	switch (getAX() & 0x00FF) {	// AX = 15xx
//	case API_GET_NUMBER_OF_DRIVES:		// GET NUMBER OF CD-ROM DRIVES
//		ApiGetNumberOfCDROMDrives();
//		break;
//	case API_GET_DRIVE_DEVICE_LIST:		// GET DRIVE DEVICE LIST
//		ApiGetDriveDeviceList();
//		break;
//	case API_GET_COPYRIGHT_NAME:		// GET COPYRIGHT FILE NAME
//	case API_GET_ABSTRACT_NAME:			// GET ABSTRACT FILE NAME
//	case API_GET_BIBLIOGRAPHIC_NAME:	// GET BIBLIOGRAPHIC DOC FILE NAME
//		ApiGetCopyrightFileName();
//		break;
	case API_READ_VTOC:					// READ VTOC
		ApiReadVTOC();
		break;
//	case API_TURN_DEBUGGING_ON:			// TURN DEBUGGING ON
//	case API_TURN_DEBUGGING_OFF:		// TURN DEBUGGING OFF
//		//ApiTurnDebuggingOn();	// do nothing
//		break;
//	case API_ABSOLUTE_DISK_READ:		// ABSOLUTE DISK READ
//		ApiAbsoluteDiskRead();
//		break;
//	case API_ABSOLUTE_DISK_WRITE:		// ABSOLUTE DISK WRITE
//		ApiAbsoluteDiskWrite();
//		break;
//	case API_RESERVED:					// RESERVED
//		//ApiReserved();	// do nothing
//		break;
	case API_DRIVE_CHECK:				// DRIVE CHECK
		ApiDriveCheck();
		break;
	case API_GET_MSCDEX_VERSION:		// GET MSCDEX.EXE VERSION
		ApiGetMSCDEXVersion();
		break;
//	case API_GET_DRIVE_LETTERS:			// GET CD-ROM DRIVE LETTERS
//		ApiGetCDROMDriveLetters();
//		break;
//	case API_GETSET_VOL_DESCR_PREFS:	// GET/SET VOLUME DESCRIPTOR PREFERENCE
//		ApiGetSetVolumeDescriptorPreference();
//		break;
//	case API_GET_DIRECTORY_ENTRY:		// GET DIRECTORY ENTRY
//		ApiGetDirectoryEntry();
//		break;
	case API_SEND_DEVICE_REQUEST:		// SEND DEVICE DRIVER REQUEST
		ApiSendDeviceDriverRequest();
		break;
	default:
#ifdef	_DEBUG
		__asm int 3
#endif//_DEBUG
		setAX(ERROR_INVALID_FUNCTION);
		setCF(1);						// set carry flag (error)
		break;
	}
}


//******************************************************************************



// Warning: The only 'valid' MSCDEX error codes are the following:
//	ERROR_WRITE_PROTECT			// 00 = write-protect violation
//	ERROR_BAD_UNIT				// 01 = unknown unit
//	ERROR_NOT_READY				// 02 = drive not ready
//	ERROR_BAD_COMMAND			// 03 = unknown command
//	ERROR_CRC					// 04 = CRC error
//	ERROR_BAD_LENGTH			// 05 = bad drive request structure length
//	ERROR_SEEK					// 06 = seek error
//	ERROR_NOT_DOS_DISK			// 07 = unknown media
//	ERROR_SECTOR_NOT_FOUND		// 08 = sector not found
//	ERROR_OUT_OF_PAPER			// 09 = printer out of paper
//	ERROR_WRITE_FAULT			// 0A = write fault
//	ERROR_READ_FAULT			// 0B = read fault
//	ERROR_GEN_FAILURE			// 0C = general failure
//	ERROR_SHARING_VIOLATION		// 0D = reserved
//	ERROR_LOCK_VIOLATION		// 0E = reserved
//	ERROR_WRONG_DISK			// 0F = invalid disk change




//INT 2F - CD-ROM - READ VTOC
//	AX = 1505h
//	ES:BX -> 2048-byte buffer
//	CX = drive number (0=A:)
//	DX = sector index (0=first volume descriptor,1=second,...)
//Return: CF set on error
//	    AX = error code (15=invalid drive,21=not ready)
//	CF clear if successful
//	    AX = volume descriptor type (1=standard,FFh=terminator,0=other)
void ApiReadVTOC(void)
{
	int		drive, sector_index;
	BYTE	type;
	DRIVE_POINTER	*DrvInfo;
	VOLUME_DESCRIPTOR_UNION	*p;

	drive = getCX();
	if ((DrvInfo = GetDrivePointer(drive, TRUE)) == NULL) {
		setAX(ERROR_INVALID_DRIVE);
		setCF(1);						// set carry flag (error)
		return;
	}
	sector_index = getDX();
	p = (VOLUME_DESCRIPTOR_UNION *)((nt_dos_translate_adr) ?
		(getES() << 16) + getBX() : (getES() << 4) + getBX());
	if (!ReadSectorCooked(DrvInfo, p->raw_data, 16 + sector_index, 1)) {
		setAX(ERROR_NOT_READY);
		setCF(1);						// set carry flag (error)
		return;
	}
	if (!strncmp((char *)p->iso_vd.id, ISO_STANDARD_ID, sizeof(p->iso_vd.id))) {
		type = ISO711(p->iso_vd.type);
//	} else if (!strncmp((char *)p->hs_vd.id, HS_STANDARD_ID, sizeof(p->hs_vd.id))) {
//		type = ISO711(p->hs_vd.type);
	} else {
		setAX(ERROR_BAD_FORMAT);
		setCF(1);						// set carry flag (error)
		return;
	}
	setAX((WORD)((type == ISO_VD_PRIMARY) ? 0x01 :
				 (type == ISO_VD_END) ? 0xFF : 0));
}

//INT 2F - CD-ROM v2.00+ - DRIVE CHECK
//	AX = 150Bh
//	CX = drive number (0=A:)
//Return: BX = ADADh if MSCDEX.EXE installed
//	    AX = support status
//		0000h if drive not supported
//		nonzero if supported
void ApiDriveCheck(void)
{
	int		drive;

	setBX(0xADAD);
	drive = getCX();
	setAX((WORD)(((GetDrivePointer(drive, FALSE)) == NULL) ? 0 : 1));
}

//INT 2F - CD-ROM v2.00+ - GET MSCDEX.EXE VERSION (GET VERSION)
//	AX = 150Ch
//	BX = 0000h
//Return: BH = major version
//	BL = minor version
void ApiGetMSCDEXVersion(void)
{
	setBX(MAKEWORD(MSCDEX_VERSION_LO, MSCDEX_VERSION_HI));
}

//INT 2F - CD-ROM v2.10+ - SEND DEVICE DRIVER REQUEST
//	AX = 1510h
//	CX = CD-ROM drive letter (0 = A, 1 = B, etc)
//	ES:BX -> CD-ROM device driver request header (see #02573 at AX=0802h)
//Return: CF clear if device driver has been called (check the request header's
//	      status word to determine whether an error has occurred)
//	    ES:BX request header updated
//	CF set if device driver has not been called
//	    AX = error code (000Fh = invalid drive, 0001h = invalid function)
//	    ES:BX request header unchanged
//Notes:	MSCDEX initializes the device driver request header's subunit field
//	  based on the drive number specified in CX
void ApiSendDeviceDriverRequest(void)
{
	int		drive;
	DRIVE_POINTER	*DrvInfo;
	DEVICE_REQUEST	*VdmReq;

	drive = getCX();
	if ((DrvInfo = GetDrivePointer(drive, TRUE)) == NULL) {
		setAX(ERROR_INVALID_DRIVE);
		setCF(1);						// set carry flag (error)
		return;
	}
	VdmReq = (DEVICE_REQUEST *)((nt_dos_translate_adr) ?
		(getES() << 16) + getBX() : (getES() << 4) + getBX());
	VdmReq->header.unit = DrvInfo->subunit_nr;
	*(WORD *)&VdmReq->header.status = 0;
	GetAudioStatus(DrvInfo);
	switch (VdmReq->header.cmd)
	{
	case DEV_IOCTL_INPUT:				// IOCTL INPUT
		DevIoctlInput(DrvInfo, VdmReq);
		break;
	case DEV_INPUT_FLUSH:				// INPUT FLUSH (character devices)
	case DEV_OUTPUT_FLUSH:				// OUTPUT FLUSH (character devices)
		//DevInputFlush(DrvInfo, VdmReq);	// do nothing
		break;
	case DEV_IOCTL_OUTPUT:				// IOCTL OUTPUT
		DevIoctlOutput(DrvInfo, VdmReq);
		break;
	case DEV_OPEN:						// DEVICE OPEN
	case DEV_CLOSE:						// DEVICE CLOSE
		//DevOpen(DrvInfo, VdmReq);	// do nothing
		break;
	case DEV_READ_LONG:					// (CD-ROM) READ LONG
		DevReadLong(DrvInfo, VdmReq);
		break;
	case DEV_READ_LONG_PREFETCH:		// (CD-ROM) READ LONG PREFETCH
		//DevReadLongPrefetch(DrvInfo, VdmReq);	// do nothing
		break;
//	case DEV_SEEK:						// (CD-ROM) SEEK
//		DevSeek(DrvInfo, VdmReq);
//		break;
	case DEV_PLAY_AUDIO:				// (CD-ROM) PLAY AUDIO
		DevPlayAudio(DrvInfo, VdmReq);
		break;
	case DEV_STOP_AUDIO:				// (CD-ROM) STOP AUDIO
		DevStopAudio(DrvInfo, VdmReq);
		break;
//	case DEV_WRITE_LONG:				// (CD-ROM) WRITE LONG
//	case DEV_WRITE_LONG_VERIFY:			// (CD-ROM) WRITE LONG VERIFY
//		DevWriteLong(DrvInfo, VdmReq);
//		break;
	case DEV_RESUME_AUDIO:				// (CD-ROM) RESUME AUDIO
		DevResumeAudio(DrvInfo, VdmReq);
		break;
	default:
#ifdef	_DEBUG
		__asm int 3
#endif//_DEBUG
		*(WORD *)&VdmReq->header.status = STATUS_ERR(ERROR_BAD_COMMAND);
		break;
	}
	if (!DrvInfo->ioctl_res)			// (see 'GetAudioStatus()')
		ClosePhysicalDrive(DrvInfo);
	if (DrvInfo->playing && !VdmReq->header.status.error)
		VdmReq->header.status.busy = 1;
	VdmReq->header.status.done = 1;
}

void DevIoctlInput(DRIVE_POINTER *DrvInfo, DEVICE_REQUEST *VdmReq)
{
	CONTROL_BLOCK	*cb;
//	DWORD	dwOutByteCount, error_code;
//	VOLUME_CONTROL	volume_control;
//	CDROM_SUB_Q_DATA_FORMAT	upc_req;
//	SUB_Q_MEDIA_CATALOG_NUMBER	media_catalog;
	TRACK_DATA	*track;
//	int		idx;

	cb = (CONTROL_BLOCK *)((nt_dos_translate_adr) ?
		(getES() << 16) + ((VdmReq->ioctl_input.addr >> 12) & 0xFFF0) + (VdmReq->ioctl_input.addr & 0x0F) :
		((VdmReq->ioctl_input.addr >> 12) & 0x000FFFF0) + (VdmReq->ioctl_input.addr & 0x0F));
#ifdef	USE_DUMP_BLI_FILE
	if (dump_bli_file != INVALID_HANDLE_VALUE) {
		CONTROL_BLOCK	tmp_cb;
		DWORD	nBytesRead;
		if (ReadFile(dump_bli_file, &tmp_cb, 0x07, &nBytesRead, NULL) && nBytesRead == 0x07 &&
			tmp_cb.header.cmd == cb->header.cmd) {
			memcpy(cb, &tmp_cb, 0x07);
			return;
		}
		CloseHandle(dump_bli_file);
		dump_bli_file = INVALID_HANDLE_VALUE;
#ifdef	_DEBUG
		Log("DEV_IOCTL_INPUT: closed 'dump.bli' file");
#endif//_DEBUG
	}
#endif//USE_DUMP_BLI_FILE
	switch (cb->header.cmd) {
//	case DEVICE_DRIVER_ADDRESS:
//		cb->device_driver_address.address = DeviceHeader;
//		break;
//	case HEAD_LOCATION:
//		if (!DrvInfo->ioctl_res) {		// (see 'GetAudioStatus()')
//			*(WORD *)&VdmReq->header.status = STATUS_ERR(DrvInfo->translated_error_code);
//			break;
//		}
//		idx = MSF2INT((MSF *)&DrvInfo->subqpos.AbsoluteAddress[1]);
//		if (cb->head_location.mode == ADDR_MODE_HSG) {
//			cb->head_location.loc = idx;
//		} else if (cb->head_location.mode == ADDR_MODE_RB) {
//			INT2RB(idx, (RB_ADDR *)&cb->head_location.loc);
//		} else {
//			*(WORD *)&VdmReq->header.status = STATUS_ERR(ERROR_SHARING_VIOLATION);
//			break;
//		}
//		break;
//	case AUDIO_CHANNEL_INFO:
//		if (!DeviceIoControl(DrvInfo->drive_handle, IOCTL_CDROM_GET_VOLUME, NULL, 0, &volume_control, sizeof(volume_control), &dwOutByteCount, NULL)) {
//			error_code = ProcessError(DrvInfo, TRUE);
//			*(WORD *)&VdmReq->header.status = STATUS_ERR(error_code);
//			break;
//		}
//		cb->audio_channel_info.channels[0].channel = 0;
//		cb->audio_channel_info.channels[0].volume = volume_control.PortVolume[0];
//		cb->audio_channel_info.channels[1].channel = 1;
//		cb->audio_channel_info.channels[1].volume = volume_control.PortVolume[1];
//		cb->audio_channel_info.channels[2].channel = 2;
//		cb->audio_channel_info.channels[2].volume = volume_control.PortVolume[2];
//		cb->audio_channel_info.channels[3].channel = 3;
//		cb->audio_channel_info.channels[3].volume = volume_control.PortVolume[3];
//		break;
	case DEVICE_STATUS:
		//*(DWORD *)&cb->device_status.parms = 0;
		//cb->device_status.parms.open = FALSE;
		//cb->device_status.parms.unlocked = FALSE;
		//cb->device_status.parms.rawread = TRUE;
		//cb->device_status.parms.cdrw = FALSE;
		//cb->device_status.parms.playback = TRUE;
		//cb->device_status.parms.interleaving = FALSE;
		//cb->device_status.parms.reserved0 = FALSE;
		//cb->device_status.parms.prefetch = TRUE;
		//cb->device_status.parms.audiochannel = TRUE;
		//cb->device_status.parms.redbookaddr = TRUE;
		//cb->device_status.parms.reserved1 = FALSE;
		//cb->device_status.parms.diskabsent = FALSE;
		//cb->device_status.parms.rwsubchannel = TRUE;	// (not really supported)
		*(DWORD *)&cb->device_status.parms = 0x1394;
		if (DrvInfo->trayopen)
			cb->device_status.parms.open = TRUE;
		if (DrvInfo->unlocked)
			cb->device_status.parms.unlocked = TRUE;
		if (!DrvInfo->ioctl_res &&		// (see 'GetAudioStatus()')
			(DrvInfo->error_code == ERROR_NO_MEDIA_IN_DRIVE || DrvInfo->error_code == ERROR_NOT_READY))
			cb->device_status.parms.diskabsent = TRUE;
		break;
//	case SECTOR_SIZE:
//		if (cb->sector_size.mode == READ_MODE_COOKED)
//			cb->sector_size.size = COOKED_SECTOR_SIZE;
//		else //if (cb->sector_size.mode == READ_MODE_RAW)
//			cb->sector_size.size = RAW_SECTOR_SIZE;
//		break;
//	case VOLUME_SIZE:
//		if (!ReadTOC(DrvInfo)) {
//			*(WORD *)&VdmReq->header.status = STATUS_ERR(DrvInfo->translated_error_code);
//			break;
//		}
//		track = &DrvInfo->toc.TrackData[DrvInfo->toc.LastTrack];
//		cb->volume_size.size = MSF2INT((MSF *)&track->Address[1]);
//		break;
//	case MEDIA_CHANGED:
//		if (DrvInfo->media_changed == MEDIA_NOT_CHANGED && 
//			!DeviceIoControl(DrvInfo->drive_handle, IOCTL_CDROM_CHECK_VERIFY, NULL, 0, NULL, 0, &dwOutByteCount, NULL)) {
//			ProcessError(DrvInfo, TRUE);
//			DrvInfo->media_changed = MEDIA_HAS_CHANGED;	// force 'media changed' status
//		}
//		cb->media_changed.state = DrvInfo->media_changed;
//		break;
	case AUDIO_DISK_INFO:
		if (!ReadTOC(DrvInfo)) {
			*(WORD *)&VdmReq->header.status = STATUS_ERR(DrvInfo->translated_error_code);
			break;
		}
		cb->audio_disk_info.first = DrvInfo->toc.FirstTrack;
		cb->audio_disk_info.last = DrvInfo->toc.LastTrack;
		track = &DrvInfo->toc.TrackData[DrvInfo->toc.LastTrack];
		cb->audio_disk_info.leadout.frame = track->Address[3];
		cb->audio_disk_info.leadout.second = track->Address[2];
		cb->audio_disk_info.leadout.minute = track->Address[1];
		cb->audio_disk_info.leadout.reserved = track->Address[0];
		break;
	case AUDIO_TRACK_INFO:
		if (!ReadTOC(DrvInfo)) {
			*(WORD *)&VdmReq->header.status = STATUS_ERR(DrvInfo->translated_error_code);
			break;
		}
		if (cb->audio_track_info.track < 1 || cb->audio_track_info.track > DrvInfo->toc.LastTrack) {
			*(WORD *)&VdmReq->header.status = STATUS_ERR(ERROR_GEN_FAILURE);
			break;
		}
		track = &DrvInfo->toc.TrackData[cb->audio_track_info.track - 1];
		cb->audio_track_info.loc.frame = track->Address[3];
		cb->audio_track_info.loc.second = track->Address[2];
		cb->audio_track_info.loc.minute = track->Address[1];
		cb->audio_track_info.loc.reserved = track->Address[0];
		cb->audio_track_info.ctrlinfo = (track->Control << 4) | track->Adr;
		break;
	case AUDIO_Q_CHANNEL_INFO:
		if (!DrvInfo->ioctl_res) {		// (see 'GetAudioStatus()')
			*(WORD *)&VdmReq->header.status = STATUS_ERR(DrvInfo->translated_error_code);
			break;
		}
		cb->audio_q_channel_info.ctrladr = (DrvInfo->subqpos.Control << 4) | DrvInfo->subqpos.ADR;
		cb->audio_q_channel_info.track = DrvInfo->subqpos.TrackNumber;
		cb->audio_q_channel_info.pointidx = DrvInfo->subqpos.IndexNumber;
		cb->audio_q_channel_info.runtrack.minute = DrvInfo->subqpos.TrackRelativeAddress[1];
		cb->audio_q_channel_info.runtrack.second = DrvInfo->subqpos.TrackRelativeAddress[2];
		cb->audio_q_channel_info.runtrack.frame = DrvInfo->subqpos.TrackRelativeAddress[3];
		cb->audio_q_channel_info.reserved = DrvInfo->subqpos.AbsoluteAddress[0];
		cb->audio_q_channel_info.rundisk.minute = DrvInfo->subqpos.AbsoluteAddress[1];
		cb->audio_q_channel_info.rundisk.second = DrvInfo->subqpos.AbsoluteAddress[2];
		cb->audio_q_channel_info.rundisk.frame = DrvInfo->subqpos.AbsoluteAddress[3];
		break;
//	case UPC_CODE:
//		 upc_req.Format = IOCTL_CDROM_MEDIA_CATALOG;
//		 upc_req.Track = 0;
//		if (!DeviceIoControl(DrvInfo->drive_handle, IOCTL_CDROM_READ_Q_CHANNEL, &upc_req, sizeof(upc_req), &media_catalog, sizeof(media_catalog), &dwOutByteCount, NULL)) {
//			error_code = ProcessError(DrvInfo, TRUE);
//			*(WORD *)&VdmReq->header.status = STATUS_ERR(error_code);
//			break;
//		}
//		if (!media_catalog.Mcval) {
//			*(WORD *)&VdmReq->header.status = STATUS_ERR(ERROR_SECTOR_NOT_FOUND);
//			break;
//		}
//		for (idx = 0; idx < sizeof(cb->upc_code.code); idx++)
//			cb->upc_code.code[idx] = (media_catalog.MediaCatalog[idx*2] << 4) | (media_catalog.MediaCatalog[idx*2+1] & 0x0F);
//		break;
	case AUDIO_STATUS_INFO:
		*(WORD *)&cb->audio_status_info.status = 0;
		if (DrvInfo->paused)
			*(WORD *)&cb->audio_status_info.status = 1;
		if (DrvInfo->playing || DrvInfo->paused) {
			*(DWORD *)&cb->audio_status_info.start = DrvInfo->play_start;
			*(DWORD *)&cb->audio_status_info.end = DrvInfo->play_end;
		} else {
			*(DWORD *)&cb->audio_status_info.start = 0;
			*(DWORD *)&cb->audio_status_info.end = 0;
		}
		break;
	//case RESERVED:
	//case ERROR_STATISTIC:
	//case DRIVE_BYTES:
	//case AUDIO_SUBCHANNEL_INFO:
	default:
#ifdef	_DEBUG
		__asm int 3
#endif//_DEBUG
		*(WORD *)&VdmReq->header.status = STATUS_ERR(ERROR_BAD_COMMAND);
		break;
	}
}

void DevIoctlOutput(DRIVE_POINTER *DrvInfo, DEVICE_REQUEST *VdmReq)
{
	CONTROL_BLOCK	*cb;
	DWORD	dwOutByteCount, error_code;
//	VOLUME_CONTROL	volume_control;
//	PREVENT_MEDIA_REMOVAL	media_removal;

	cb = (CONTROL_BLOCK *)((nt_dos_translate_adr) ?
		(getES() << 16) + ((VdmReq->ioctl_output.addr >> 12) & 0xFFF0) + (VdmReq->ioctl_output.addr & 0x0F) :
		((VdmReq->ioctl_output.addr >> 12) & 0x000FFFF0) + (VdmReq->ioctl_output.addr & 0x0F));
	switch (cb->header.cmd)
	{
	case EJECT_DISK:
		if (!DeviceIoControl(DrvInfo->drive_handle, IOCTL_CDROM_EJECT_MEDIA, NULL, 0, NULL, 0, &dwOutByteCount, NULL)) {
			error_code = ProcessError(DrvInfo, TRUE);
			*(WORD *)&VdmReq->header.status = STATUS_ERR(error_code);
			break;
		}
		//DrvInfo->trayopen = TRUE;
		break;
//	case LOCK_UNLOCK_DOOR:
//		media_removal.PreventMediaRemoval = cb->lock_unlock_door.lock;
//		if (!DeviceIoControl(DrvInfo->drive_handle, IOCTL_CDROM_MEDIA_REMOVAL, &media_removal, sizeof(media_removal), NULL, 0, &dwOutByteCount, NULL)) {
//			error_code = ProcessError(DrvInfo, TRUE);
//			*(WORD *)&VdmReq->header.status = STATUS_ERR(error_code);
//			break;
//		}
//		DrvInfo->unlocked = !cb->lock_unlock_door.lock;
//		break;
//	case RESET_DRIVE:
//		break;
//	case AUDIO_CHANNEL_CONTROL:
//		volume_control.PortVolume[0] = cb->audio_channel_control.channels[0].volume;
//		volume_control.PortVolume[1] = cb->audio_channel_control.channels[1].volume;
//		volume_control.PortVolume[2] = cb->audio_channel_control.channels[2].volume;
//		volume_control.PortVolume[3] = cb->audio_channel_control.channels[3].volume;
//		if (!DeviceIoControl(DrvInfo->drive_handle, IOCTL_CDROM_SET_VOLUME, &volume_control, sizeof(volume_control), NULL, 0, &dwOutByteCount, NULL)) {
//			error_code = ProcessError(DrvInfo, TRUE);
//			*(WORD *)&VdmReq->header.status = STATUS_ERR(error_code);
//			break;
//		}
//		break;
	case CLOSE_TRAY:
		if (!DeviceIoControl(DrvInfo->drive_handle, IOCTL_CDROM_LOAD_MEDIA, NULL, 0, NULL, 0, &dwOutByteCount, NULL)) {
			error_code = ProcessError(DrvInfo, TRUE);
			*(WORD *)&VdmReq->header.status = STATUS_ERR(error_code);
			break;
		}
		//DrvInfo->trayopen = FALSE;
		break;
	default:
#ifdef	_DEBUG
		__asm int 3
#endif//_DEBUG
		*(WORD *)&VdmReq->header.status = STATUS_ERR(ERROR_BAD_COMMAND);
		break;
	}
}

void DevReadLong(DRIVE_POINTER *DrvInfo, DEVICE_REQUEST *VdmReq)
{
	BYTE	*p;
	DWORD	read_loc, read_len;
	CDB		cdb;

	p = (BYTE *)((nt_dos_translate_adr) ?
		(getES() << 16) + ((VdmReq->read_long.addr >> 12) & 0xFFF0) + (VdmReq->read_long.addr & 0x0F) :
		((VdmReq->read_long.addr >> 12) & 0x000FFFF0) + (VdmReq->read_long.addr & 0x0F));
#ifdef	USE_DUMP_BLI_FILE
	if (dump_bli_file != INVALID_HANDLE_VALUE) {
		DWORD	nBytesRead;
		if (ReadFile(dump_bli_file, p, 0x0800, &nBytesRead, NULL) && nBytesRead == 0x0800)
			return;
		CloseHandle(dump_bli_file);
		dump_bli_file = INVALID_HANDLE_VALUE;
#ifdef	_DEBUG
		Log("DEV_READ_LONG: closed 'dump.bli' file");
#endif//_DEBUG
	}
#endif//USE_DUMP_BLI_FILE
	if (VdmReq->read_long.addr_mode == ADDR_MODE_HSG) {
		read_loc = VdmReq->read_long.loc;
	} else if (VdmReq->read_long.addr_mode == ADDR_MODE_RB) {
		read_loc = RB2INT((RB_ADDR *)&VdmReq->read_long.loc);
	} else {
		*(WORD *)&VdmReq->header.status = STATUS_ERR(ERROR_SHARING_VIOLATION);
		return;
	}
	memset(&cdb, 0, sizeof(cdb));
	cdb.READ_CD.OperationCode = SCSIOP_READ_CD;
	//cdb.READ_CD.ExpectedSectorType = CD_EXPECTED_SECTOR_ANY;
	//cdb.READ_CD.Lun = 0;				// RESERVED
	//cdb.READ_CD.StartingLBA[0] = 0;
	cdb.READ_CD.StartingLBA[1] = (BYTE)(read_loc >> 16);
	cdb.READ_CD.StartingLBA[2] = (BYTE)(read_loc >> 8);
	cdb.READ_CD.StartingLBA[3] = (BYTE)read_loc;
	//cdb.READ_CD.TransferBlocks[0] = 0;
	//cdb.READ_CD.TransferBlocks[1] = (BYTE)(VdmReq->read_long.nsec >> 8);
	cdb.READ_CD.TransferBlocks[2] = (BYTE)(VdmReq->read_long.nsec);
	if (VdmReq->read_long.rd_mode == READ_MODE_COOKED) {
		//cdb.READ_CD.IncludeEDC = 0;
		cdb.READ_CD.IncludeUserData = 1;
		//cdb.READ_CD.HeaderCode = 0;
		//cdb.READ_CD.IncludeSyncData = 0;
		read_len = COOKED_SECTOR_SIZE * VdmReq->read_long.nsec;
	} else { // if (VdmReq->read_long.rd_mode == READ_MODE_RAW)
		cdb.READ_CD.IncludeEDC = 1;
		cdb.READ_CD.IncludeUserData = 1;
		cdb.READ_CD.HeaderCode = 3;
		cdb.READ_CD.IncludeSyncData = 1;
		read_len = RAW_SECTOR_SIZE * VdmReq->read_long.nsec;
	}
	//cdb.READ_CD.SubChannelSelection = 0;
	if (!ExecScsiCmd(DrvInfo, p, read_len, &cdb, sizeof(cdb.READ_CD), SCSI_IOCTL_DATA_IN)) {
		*(WORD *)&VdmReq->header.status = STATUS_ERR(DrvInfo->translated_error_code);
		return;
	}
}

//void	DevReadLongPrefetch(DRIVE_POINTER *DrvInfo, DEVICE_REQUEST *VdmReq)
//{
//}

void DevPlayAudio(DRIVE_POINTER *DrvInfo, DEVICE_REQUEST *VdmReq)
{
	DWORD	dwOutByteCount, error_code, start_loc, end_loc;
	CDROM_PLAY_AUDIO_MSF	play_audio_msf;

	if (VdmReq->play_audio.addr_mode == ADDR_MODE_HSG) {
		start_loc = VdmReq->play_audio.loc;
	} else if (VdmReq->play_audio.addr_mode == ADDR_MODE_RB) {
		start_loc = RB2INT((RB_ADDR *)&VdmReq->play_audio.loc);
	} else {
		*(WORD *)&VdmReq->header.status = STATUS_ERR(ERROR_SHARING_VIOLATION);
		return;
	}
	INT2MSF(start_loc, (MSF *)&play_audio_msf.StartingM);
	end_loc = start_loc + VdmReq->play_audio.nsec;
	INT2MSF(end_loc, (MSF *)&play_audio_msf.EndingM);
	if (!DeviceIoControl(DrvInfo->drive_handle, IOCTL_CDROM_PLAY_AUDIO_MSF, &play_audio_msf, sizeof(play_audio_msf), NULL, 0, &dwOutByteCount, NULL)) {
		error_code = ProcessError(DrvInfo, TRUE);
		*(WORD *)&VdmReq->header.status = STATUS_ERR(error_code);
		return;
	}
	DrvInfo->playing = TRUE;
	DrvInfo->paused = FALSE;
	// save start/end play locations (see AUDIO_STATUS_INFO)
	INT2RB(start_loc, (RB_ADDR *)&DrvInfo->play_start);
	INT2RB(end_loc, (RB_ADDR *)&DrvInfo->play_end);
}

void DevStopAudio(DRIVE_POINTER *DrvInfo, DEVICE_REQUEST *VdmReq)
{
	DWORD	dwOutByteCount, error_code, paused_loc;

	if (DrvInfo->playing) {
		if (!DeviceIoControl(DrvInfo->drive_handle, IOCTL_CDROM_PAUSE_AUDIO, NULL, 0, NULL, 0, &dwOutByteCount, NULL)) {
			error_code = ProcessError(DrvInfo, TRUE);
			*(WORD *)&VdmReq->header.status = STATUS_ERR(error_code);
			return;
		}
		paused_loc = MSF2INT((MSF *)&DrvInfo->subqpos.AbsoluteAddress[1]);
		INT2RB(paused_loc, (RB_ADDR *)&DrvInfo->play_start);
		DrvInfo->playing = FALSE;
		DrvInfo->paused = TRUE;
	} else if (DrvInfo->paused) {
		DrvInfo->playing = FALSE;
		DrvInfo->paused = FALSE;
		DrvInfo->last_audio_status = AUDIO_STATUS_PLAY_COMPLETE;
		if (!DeviceIoControl(DrvInfo->drive_handle, IOCTL_CDROM_STOP_AUDIO, NULL, 0, NULL, 0, &dwOutByteCount, NULL)) {
			error_code = ProcessError(DrvInfo, TRUE);
			*(WORD *)&VdmReq->header.status = STATUS_ERR(error_code);
			return;
		}
	} // else (!playing && !paused), do nothing
}

void DevResumeAudio(DRIVE_POINTER *DrvInfo, DEVICE_REQUEST *VdmReq)
{
	DWORD	dwOutByteCount, error_code;

	if (!DrvInfo->paused) {
		*(WORD *)&VdmReq->header.status = STATUS_ERR(ERROR_GEN_FAILURE);
		return;
	}
	if (!DeviceIoControl(DrvInfo->drive_handle, IOCTL_CDROM_RESUME_AUDIO, NULL, 0, NULL, 0, &dwOutByteCount, NULL)) {
		error_code = ProcessError(DrvInfo, TRUE);
		*(WORD *)&VdmReq->header.status = STATUS_ERR(error_code);
		return;
	}
}

