//
// File: iso_fs.h
//
// ISO9660 / High Sierra filesystem structures & definitions
//
// Copyright (C) 2002  SaPu <spuccini@supereva.it>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#ifndef __ISO_FS_H__
#define	__ISO_FS_H__

/*
 * Iso9660 filesystem constants/structures
 */

/* This part borrowed from the bsd386 isofs */
#define	ISODCL(from, to)		(to - from + 1)

typedef struct iso_volume_descriptor {
	BYTE	type					[ISODCL (  1,   1)]; /* 711 */
	BYTE	id						[ISODCL (  2,   6)];
	BYTE	version					[ISODCL (  7,   7)];
	BYTE	data					[ISODCL (  8,2048)];
} ISO_VOLUME_DESCRIPTOR;

/* volume descriptor types */
#define	ISO_VD_PRIMARY			1
#define	ISO_VD_SUPPLEMENTARY	2
#define	ISO_VD_END				255

#define	ISO_STANDARD_ID			"CD001"

typedef struct iso_primary_descriptor {
	BYTE	type					[ISODCL (  1,   1)]; /* 711 */
	BYTE	id						[ISODCL (  2,   6)];
	BYTE	version					[ISODCL (  7,   7)]; /* 711 */
	BYTE	unused1					[ISODCL (  8,   8)];
	BYTE	system_id				[ISODCL (  9,  40)]; /* achars */
	BYTE	volume_id				[ISODCL ( 41,  77)]; /* dchars */
	BYTE	unused2					[ISODCL ( 78,  80)];
	BYTE	volume_space_size		[ISODCL ( 81,  88)]; /* 733 */
	BYTE	unused3					[ISODCL ( 89, 120)];
	BYTE	volume_set_size			[ISODCL (121, 124)]; /* 723 */
	BYTE	volume_sequence_number	[ISODCL (125, 128)]; /* 723 */
	BYTE	logical_block_size		[ISODCL (129, 132)]; /* 723 */
	BYTE	path_table_size			[ISODCL (133, 140)]; /* 733 */
	BYTE	type_l_path_table		[ISODCL (141, 144)]; /* 731 */
	BYTE	opt_type_l_path_table	[ISODCL (145, 148)]; /* 731 */
	BYTE	type_m_path_table		[ISODCL (149, 152)]; /* 732 */
	BYTE	opt_type_m_path_table	[ISODCL (153, 156)]; /* 732 */
	BYTE	root_directory_record	[ISODCL (157, 190)]; /* 9.1 */
	BYTE	volume_set_id			[ISODCL (191, 318)]; /* dchars */
	BYTE	publisher_id			[ISODCL (319, 446)]; /* achars */
	BYTE	preparer_id				[ISODCL (447, 574)]; /* achars */
	BYTE	application_id			[ISODCL (575, 702)]; /* achars */
	BYTE	copyright_file_id		[ISODCL (703, 739)]; /* 7.5 dchars */
	BYTE	abstract_file_id		[ISODCL (740, 776)]; /* 7.5 dchars */
	BYTE	bibliographic_file_id	[ISODCL (777, 813)]; /* 7.5 dchars */
	BYTE	creation_date			[ISODCL (814, 830)]; /* 8.4.26.1 */
	BYTE	modification_date		[ISODCL (831, 847)]; /* 8.4.26.1 */
	BYTE	expiration_date			[ISODCL (848, 864)]; /* 8.4.26.1 */
	BYTE	effective_date			[ISODCL (865, 881)]; /* 8.4.26.1 */
	BYTE	file_structure_version	[ISODCL (882, 882)]; /* 711 */
	BYTE	unused4					[ISODCL (883, 883)];
	BYTE	application_data		[ISODCL (884, 1395)];
	BYTE	unused5					[ISODCL (1396,2048)];
} ISO_PRIMARY_DESCRIPTOR;

/* Almost the same as the primary descriptor but two fields are specified */
typedef struct iso_supplementary_descriptor {
	BYTE	type					[ISODCL (  1,   1)]; /* 711 */
	BYTE	id						[ISODCL (  2,   6)];
	BYTE	version					[ISODCL (  7,   7)]; /* 711 */
	BYTE	flags					[ISODCL (  8,   8)]; /* 853 */
	BYTE	system_id				[ISODCL (  9,  40)]; /* achars */
	BYTE	volume_id				[ISODCL ( 41,  77)]; /* dchars */
	BYTE	unused2					[ISODCL ( 78,  80)];
	BYTE	volume_space_size		[ISODCL ( 81,  88)]; /* 733 */
	BYTE	escape					[ISODCL ( 89, 120)]; /* 856 */
	BYTE	volume_set_size			[ISODCL (121, 124)]; /* 723 */
	BYTE	volume_sequence_number	[ISODCL (125, 128)]; /* 723 */
	BYTE	logical_block_size		[ISODCL (129, 132)]; /* 723 */
	BYTE	path_table_size			[ISODCL (133, 140)]; /* 733 */
	BYTE	type_l_path_table		[ISODCL (141, 144)]; /* 731 */
	BYTE	opt_type_l_path_table	[ISODCL (145, 148)]; /* 731 */
	BYTE	type_m_path_table		[ISODCL (149, 152)]; /* 732 */
	BYTE	opt_type_m_path_table	[ISODCL (153, 156)]; /* 732 */
	BYTE	root_directory_record	[ISODCL (157, 190)]; /* 9.1 */
	BYTE	volume_set_id			[ISODCL (191, 318)]; /* dchars */
	BYTE	publisher_id			[ISODCL (319, 446)]; /* achars */
	BYTE	preparer_id				[ISODCL (447, 574)]; /* achars */
	BYTE	application_id			[ISODCL (575, 702)]; /* achars */
	BYTE	copyright_file_id		[ISODCL (703, 739)]; /* 7.5 dchars */
	BYTE	abstract_file_id		[ISODCL (740, 776)]; /* 7.5 dchars */
	BYTE	bibliographic_file_id	[ISODCL (777, 813)]; /* 7.5 dchars */
	BYTE	creation_date			[ISODCL (814, 830)]; /* 8.4.26.1 */
	BYTE	modification_date		[ISODCL (831, 847)]; /* 8.4.26.1 */
	BYTE	expiration_date			[ISODCL (848, 864)]; /* 8.4.26.1 */
	BYTE	effective_date			[ISODCL (865, 881)]; /* 8.4.26.1 */
	BYTE	file_structure_version	[ISODCL (882, 882)]; /* 711 */
	BYTE	unused4					[ISODCL (883, 883)];
	BYTE	application_data		[ISODCL (884, 1395)];
	BYTE	unused5					[ISODCL (1396,2048)];
} ISO_SUPPLEMENTARY_DESCRIPTOR;

#define	HS_STANDARD_ID			"CDROM"

typedef struct hs_volume_descriptor {
	BYTE	foo						[ISODCL (  1,   8)]; /* 733 */
	BYTE	type					[ISODCL (  9,   9)]; /* 711 */
	BYTE	id						[ISODCL ( 10,  14)];
	BYTE	version					[ISODCL ( 15,  15)]; /* 711 */
	BYTE	data					[ISODCL ( 16,2048)];
} HS_VOLUME_DESCRIPTOR;

typedef struct hs_primary_descriptor {
	BYTE	foo						[ISODCL (  1,   8)]; /* 733 */
	BYTE	type					[ISODCL (  9,   9)]; /* 711 */
	BYTE	id						[ISODCL ( 10,  14)];
	BYTE	version					[ISODCL ( 15,  15)]; /* 711 */
	BYTE	unused1					[ISODCL ( 16,  16)]; /* 711 */
	BYTE	system_id				[ISODCL ( 17,  48)]; /* achars */
	BYTE	volume_id				[ISODCL ( 49,  80)]; /* dchars */
	BYTE	unused2					[ISODCL ( 81,  88)]; /* 733 */
	BYTE	volume_space_size		[ISODCL ( 89,  96)]; /* 733 */
	BYTE	unused3					[ISODCL ( 97, 128)]; /* 733 */
	BYTE	volume_set_size			[ISODCL (129, 132)]; /* 723 */
	BYTE	volume_sequence_number	[ISODCL (133, 136)]; /* 723 */
	BYTE	logical_block_size		[ISODCL (137, 140)]; /* 723 */
	BYTE	path_table_size			[ISODCL (141, 148)]; /* 733 */
	BYTE	type_l_path_table		[ISODCL (149, 152)]; /* 731 */
	BYTE	opt_type_l_path_table	[ISODCL (153, 156)]; /* 731 */
	BYTE	unused4					[ISODCL (157, 180)]; /* 733 */
	BYTE	root_directory_record	[ISODCL (181, 214)]; /* 9.1 */
	BYTE	unused5					[ISODCL (215, 726)]; /* 733 */
	BYTE	copyright_file_id		[ISODCL (727, 758)]; /* 7.5 dchars */
	BYTE	abstract_file_id		[ISODCL (759, 790)]; /* 7.5 dchars */
	BYTE	unused6					[ISODCL (791,2048)];
} HS_PRIMARY_DESCRIPTOR;

/* Almost the same as the primary descriptor but two fields are specified */
typedef struct hs_supplementary_descriptor {
	BYTE	foo						[ISODCL (  1,   8)]; /* 733 */
	BYTE	type					[ISODCL (  9,   9)]; /* 711 */
	BYTE	id						[ISODCL ( 10,  14)];
	BYTE	version					[ISODCL ( 15,  15)]; /* 711 */
	BYTE	flags					[ISODCL ( 16,  16)]; /* 853 */
	BYTE	system_id				[ISODCL ( 17,  48)]; /* achars */
	BYTE	volume_id				[ISODCL ( 49,  80)]; /* dchars */
	BYTE	unused2					[ISODCL ( 81,  88)]; /* 733 */
	BYTE	volume_space_size		[ISODCL ( 89,  96)]; /* 733 */
	BYTE	escape					[ISODCL ( 97, 128)]; /* 856 */
	BYTE	volume_set_size			[ISODCL (129, 132)]; /* 723 */
	BYTE	volume_sequence_number	[ISODCL (133, 136)]; /* 723 */
	BYTE	logical_block_size		[ISODCL (137, 140)]; /* 723 */
	BYTE	path_table_size			[ISODCL (141, 148)]; /* 733 */
	BYTE	type_l_path_table		[ISODCL (149, 152)]; /* 731 */
	BYTE	opt_type_l_path_table	[ISODCL (153, 156)]; /* 731 */
	BYTE	unused4					[ISODCL (157, 180)]; /* 733 */
	BYTE	root_directory_record	[ISODCL (181, 214)]; /* 9.1 */
	BYTE	unused5					[ISODCL (215, 726)]; /* 733 */
	BYTE	copyright_file_id		[ISODCL (727, 758)]; /* 7.5 dchars */
	BYTE	abstract_file_id		[ISODCL (759, 790)]; /* 7.5 dchars */
	BYTE	unused6					[ISODCL (791,2048)];
} HS_SUPPLEMENTARY_DESCRIPTOR;

/* We use this to help us look up the parent inode numbers. */

typedef struct iso_path_table {
	BYTE	name_len				[2]; /* 721 */
	BYTE	extent					[4]; /* 731 */
	BYTE	parent					[2]; /* 721 */
	BYTE	name					[1];
} ISO_PATH_TABLE;

/* high sierra is identical to iso, except that the date is only 6 bytes, and
   there is an extra reserved byte after the flags */

typedef struct iso_directory_record {
	BYTE	length					[ISODCL ( 1,  1)]; /* 711 */
	BYTE	ext_attr_length			[ISODCL ( 2,  2)]; /* 711 */
	BYTE	extent					[ISODCL ( 3, 10)]; /* 733 */
	BYTE	size					[ISODCL (11, 18)]; /* 733 */
	BYTE	date					[ISODCL (19, 25)]; /* 7 by 711 */
	BYTE	flags					[ISODCL (26, 26)];
	BYTE	file_unit_size			[ISODCL (27, 27)]; /* 711 */
	BYTE	interleave				[ISODCL (28, 28)]; /* 711 */
	BYTE	volume_sequence_number	[ISODCL (29, 32)]; /* 723 */
	BYTE	name_len				[ISODCL (33, 33)]; /* 711 */
	BYTE	name					[1];
} ISO_DIRECTORY_RECORD;

/* Iso directory flags */
#define	ISO_FILE				0x00	/* Not really a flag... */
#define	ISO_EXISTENCE			0x01	/* Do not make existence known (hidden) */
#define	ISO_DIRECTORY			0x02	/* This file is a directory */
#define	ISO_ASSOCIATED			0x04	/* This file is an assiciated file */
#define	ISO_RECORD				0x08	/* Record format in extended attr. != 0 */
#define	ISO_PROTECTION			0x10	/* No read/execute perm. in ext. attr. */
#define	ISO_DRESERVED1			0x20	/* Reserved bit 5 */
#define	ISO_DRESERVED2			0x40	/* Reserved bit 6 */
#define	ISO_MULTIEXTENT			0x80	/* Not final entry of a mult. ext. file */

#define	ISOFS_BLOCK_BITS		11
#define	ISOFS_BLOCK_SIZE		2048

#define	ISO711(s)	(*(BYTE *)(s))
#define	ISO712(s)	((int)(*(char *)(s)))
#define	ISO721(s)	(*(WORD *)(s))
#define	ISO722(s)	((WORD)((*(BYTE *)(s)) << 8) | (*((BYTE *)(s)+1)))
#define	ISO723(s)	(*(WORD *)(s))
#define	ISO731(s)	(*(DWORD *)(s))
#define	ISO732(s)	((DWORD)((*(BYTE *)(s)) << 24) | ((*((BYTE *)(s)+1)) << 16) | ((*((BYTE *)(s)+2)) << 8) | (*((BYTE *)(s)+3)))
#define	ISO733(s)	(*(DWORD *)(s))

#endif//__ISO_FS_H__

