//******************************************************************************
//*
//* KRNPATCH.SYS: NT Kernel Patcher for WinNT/2K/XP operating systems
//*
//*		Copyright (c) 2002, by SaPu. All rights reserved.
//*
//*		This program is FREEWARE and cannot be used for any commercial purposes
//*		and/or for profit without an explicit consent of the author.
//*		The term FREEWARE refers only to the mode of distribution, the program
//*		and the relative source code are copyrighted.
//*		Reuse of this code in a commercial applications is not allowed.
//*		Author is not responsible for any damage derived from use of this program.
//*
//* Purpose:
//*		Allow an user-mode program to search, read, and modify code/data inside
//*		an already loaded kernel-mode driver.
//*
//******************************************************************************

#define	NONPAGED_ONLY	// if defined, access only segments marked as 'not-pageable'

#include "ntddk.h"
#include "ntimage.h"
#include "string.h"
#include "krnpatch.h"

typedef struct _KRNPATCH_DEVICE_EXTENSION {
	ULONG	Information;
} KRNPATCH_DEVICE_EXTENSION, *PKRNPATCH_DEVICE_EXTENSION;

NTSTATUS DriverEntry(IN PDRIVER_OBJECT DriverObject, IN PUNICODE_STRING RegistryPath);
VOID DrvUnloadDriver(IN PDRIVER_OBJECT DriverObject);
NTSTATUS DrvDispatch(IN PDEVICE_OBJECT DeviceObject, IN PIRP Irp);
NTSTATUS FindData(IN OUT PKRNPATCH_FIND_DATA OutputBuffer);
NTSTATUS ReadData(IN OUT PKRNPATCH_READ_DATA OutputBuffer);
NTSTATUS WriteData(IN OUT PKRNPATCH_WRITE_DATA OutputBuffer);
NTSTATUS FindModuleInfo(IN LPCSTR ModuleName, OUT PULONG pBaseAddress, OUT PULONG pImageSize);
BOOLEAN NextSectionInRange(IN ULONG BaseAddress, IN ULONG StartOffset, IN ULONG ScnCntType, OUT PULONG pSectionStart, OUT PULONG pSectionLen, OUT PULONG pSectionChr OPTIONAL);
NTSTATUS WritePhysicalMemory(IN PHYSICAL_ADDRESS PhysicalAddress, IN ULONG DataSize, IN PUCHAR Data, IN PUCHAR OldData OPTIONAL);

//******************************************************************************

#define SystemModuleInformation     11

typedef struct _SYSTEM_MODULE_INFORMATION {
    ULONG  Reserved[2];
    PVOID  Base;
    ULONG  Size;
    ULONG  Flags;
    USHORT Index;
    USHORT Unknown;
    USHORT LoadCount;
    USHORT ModuleNameOffset;
    CHAR   ImageName[256];
} SYSTEM_MODULE_INFORMATION, *PSYSTEM_MODULE_INFORMATION;

extern NTSTATUS NTAPI ZwQuerySystemInformation(IN ULONG SystemInformationClass, IN PVOID SystemInformation, IN ULONG SystemInformationLength, OUT PULONG ReturnLength);

//******************************************************************************

#undef	__TEXT
#define __TEXT(x)	L ## x
static WCHAR	deviceName[] = L"\\Device\\" KRNPATCH_DEVICENAME;
static WCHAR	deviceLink[] = L"\\DosDevices\\" KRNPATCH_DOS_DEVICENAME;

NTSTATUS DriverEntry(IN PDRIVER_OBJECT DriverObject, IN PUNICODE_STRING RegistryPath)
{
	UNICODE_STRING nameString, linkString;
	PDEVICE_OBJECT deviceObject;
	NTSTATUS status;
	PKRNPATCH_DEVICE_EXTENSION extension;

	RtlInitUnicodeString(&nameString, deviceName);
	status = IoCreateDevice(DriverObject, sizeof(KRNPATCH_DEVICE_EXTENSION), &nameString, FILE_DEVICE_UNKNOWN, 0, TRUE, &deviceObject);
	if (!NT_SUCCESS(status))
		return status;
	RtlInitUnicodeString(&linkString, deviceLink);
	status = IoCreateSymbolicLink(&linkString, &nameString);
	if (!NT_SUCCESS(status)) {
		IoDeleteDevice (DriverObject->DeviceObject);
		return status;
	}
	DriverObject->DriverUnload = DrvUnloadDriver;
	DriverObject->MajorFunction[IRP_MJ_CREATE] = DrvDispatch;
	DriverObject->MajorFunction[IRP_MJ_CLOSE] = DrvDispatch;
	DriverObject->MajorFunction[IRP_MJ_DEVICE_CONTROL] = DrvDispatch;

	extension = deviceObject->DeviceExtension;
	extension->Information = 0;

	return STATUS_SUCCESS;
}

VOID DrvUnloadDriver(IN PDRIVER_OBJECT DriverObject)
{
    UNICODE_STRING linkString;

    IoDeleteDevice (DriverObject->DeviceObject);
    RtlInitUnicodeString(&linkString, deviceLink);
    IoDeleteSymbolicLink (&linkString);
}

NTSTATUS DrvDispatch(IN PDEVICE_OBJECT DeviceObject, IN PIRP Irp)
{
	NTSTATUS status;
	PIO_STACK_LOCATION irpStack;
	PKRNPATCH_DEVICE_EXTENSION extension = DeviceObject->DeviceExtension;

	irpStack = IoGetCurrentIrpStackLocation(Irp);

	switch (irpStack->MajorFunction) {
	case IRP_MJ_CREATE:
		Irp->IoStatus.Status = STATUS_SUCCESS;
		Irp->IoStatus.Information = 0;
		break;

	case IRP_MJ_CLOSE:
		Irp->IoStatus.Status = STATUS_SUCCESS;
		Irp->IoStatus.Information = 0;
		break;

	case IRP_MJ_DEVICE_CONTROL:
		Irp->IoStatus.Status = STATUS_SUCCESS;
		Irp->IoStatus.Information = 0;
		switch (irpStack->Parameters.DeviceIoControl.IoControlCode) {
		case IOCTL_KRNPATCH_FIND_DATA: {
			PKRNPATCH_FIND_DATA OutputBuffer = (PKRNPATCH_FIND_DATA)Irp->AssociatedIrp.SystemBuffer;
			if (irpStack->Parameters.DeviceIoControl.InputBufferLength < sizeof(KRNPATCH_FIND_DATA)) {
				Irp->IoStatus.Status = STATUS_INVALID_PARAMETER;
				break;
			}
			if (irpStack->Parameters.DeviceIoControl.OutputBufferLength < sizeof(KRNPATCH_FIND_DATA)) {
				Irp->IoStatus.Status = STATUS_BUFFER_TOO_SMALL;
				break;
			}
			status = FindData(OutputBuffer);
			if (!NT_SUCCESS(status)) {
				Irp->IoStatus.Status = status;
				break;
			}
			Irp->IoStatus.Information = sizeof(KRNPATCH_FIND_DATA);
			break;
		} break;
		case IOCTL_KRNPATCH_READ_DATA: {
			PKRNPATCH_READ_DATA OutputBuffer = (PKRNPATCH_READ_DATA)Irp->AssociatedIrp.SystemBuffer;
			if (irpStack->Parameters.DeviceIoControl.InputBufferLength < sizeof(KRNPATCH_READ_DATA)) {
				Irp->IoStatus.Status = STATUS_INVALID_PARAMETER;
				break;
			}
			if (irpStack->Parameters.DeviceIoControl.OutputBufferLength < sizeof(KRNPATCH_READ_DATA)) {
				Irp->IoStatus.Status = STATUS_BUFFER_TOO_SMALL;
				break;
			}
			status = ReadData(OutputBuffer);
			if (!NT_SUCCESS(status)) {
				Irp->IoStatus.Status = status;
				break;
			}
			Irp->IoStatus.Information = sizeof(KRNPATCH_READ_DATA);
			break;
		} break;
		case IOCTL_KRNPATCH_WRITE_DATA: {
			PKRNPATCH_WRITE_DATA OutputBuffer = (PKRNPATCH_WRITE_DATA)Irp->AssociatedIrp.SystemBuffer;
			if (irpStack->Parameters.DeviceIoControl.InputBufferLength < sizeof(KRNPATCH_WRITE_DATA)) {
				Irp->IoStatus.Status = STATUS_INVALID_PARAMETER;
				break;
			}
			if (irpStack->Parameters.DeviceIoControl.OutputBufferLength < sizeof(KRNPATCH_WRITE_DATA)) {
				Irp->IoStatus.Status = STATUS_BUFFER_TOO_SMALL;
				break;
			}
			status = WriteData(OutputBuffer);
			if (!NT_SUCCESS(status)) {
				Irp->IoStatus.Status = status;
				break;
			}
			Irp->IoStatus.Information = sizeof(KRNPATCH_WRITE_DATA);
			break;
		} break;
		default:
			Irp->IoStatus.Status = STATUS_INVALID_DEVICE_REQUEST;
			break;
		}
	}

	status = Irp->IoStatus.Status;
	IoCompleteRequest(Irp, IO_NO_INCREMENT);
	return status;
}

NTSTATUS FindData(IN OUT PKRNPATCH_FIND_DATA OutputBuffer)
{
	NTSTATUS status;
	ULONG	base_address, image_size;
	ULONG	section_start, section_len;
	ULONG	start_offset, addr, end_addr;

	KdPrint(("Entering FindData()\n"));

	if (OutputBuffer->data_len == 0 || OutputBuffer->data_len > sizeof(OutputBuffer->data))
		return STATUS_INVALID_PARAMETER;

	status = FindModuleInfo(OutputBuffer->module_name, &base_address, &image_size);
	if (!NT_SUCCESS(status)) {
		return status;
	}
	KdPrint(("Module %s: base=%x size=%x\n", OutputBuffer->module_name, base_address, image_size));

	start_offset = OutputBuffer->start_offset;
	while (NextSectionInRange(base_address, start_offset, OutputBuffer->scn_cnt_type, &section_start, &section_len, NULL)) {
		if (start_offset < (section_start - base_address))
			start_offset = section_start - base_address;
		for (addr = base_address + start_offset, end_addr = section_start + section_len - OutputBuffer->data_len; addr < end_addr; addr++) {
			try {
				if (RtlCompareMemory(OutputBuffer->data, (PVOID)addr, OutputBuffer->data_len) != OutputBuffer->data_len)
					continue;
				KdPrint(("Found: addr=%x\n", addr));
				OutputBuffer->found_offset = addr - base_address;
				return STATUS_SUCCESS;
			} except (EXCEPTION_EXECUTE_HANDLER) {
				KdPrint(("Caught Exception: addr=%x\n", addr));
				return STATUS_INVALID_ADDRESS;
			}
		}
		start_offset = section_start + section_len - base_address;
	}
	KdPrint(("Not found\n"));
	return STATUS_NOT_FOUND;
}

NTSTATUS ReadData(IN OUT PKRNPATCH_READ_DATA OutputBuffer)
{
	NTSTATUS status;
	ULONG	base_address, image_size;
	ULONG	section_start, section_len;
	ULONG	addr;

	KdPrint(("Entering ReadData()\n"));

	if (OutputBuffer->data_len == 0 || OutputBuffer->data_len > sizeof(OutputBuffer->data))
		return STATUS_INVALID_PARAMETER;

	status = FindModuleInfo(OutputBuffer->module_name, &base_address, &image_size);
	if (!NT_SUCCESS(status)) {
		return status;
	}
	KdPrint(("Module %s: base=%x size=%x\n", OutputBuffer->module_name, base_address, image_size));

	addr = base_address + OutputBuffer->start_offset;
	if (!NextSectionInRange(base_address, OutputBuffer->start_offset, IMAGE_SCN_CNT_CODE | IMAGE_SCN_CNT_INITIALIZED_DATA | IMAGE_SCN_CNT_UNINITIALIZED_DATA, &section_start, &section_len, NULL) ||
		addr < section_start || addr >= (section_start + section_len - OutputBuffer->data_len))
		return STATUS_INVALID_ADDRESS;
	try {
		RtlCopyMemory(OutputBuffer->data, (PVOID)addr, OutputBuffer->data_len);
	} except (EXCEPTION_EXECUTE_HANDLER) {
		KdPrint(("Caught Exception\n"));
		return STATUS_INVALID_ADDRESS;
	}
	return STATUS_SUCCESS;
}

NTSTATUS WriteData(IN OUT PKRNPATCH_WRITE_DATA OutputBuffer)
{
	NTSTATUS status;
	ULONG	base_address, image_size;
	ULONG	section_start, section_len, section_chr;
	ULONG	addr, offset, page_block_len;
	PVOID	page_lock_handle;
	PHYSICAL_ADDRESS	physical_address;

	KdPrint(("Entering WriteData()\n"));

	if (OutputBuffer->data_len == 0 || OutputBuffer->data_len > sizeof(OutputBuffer->new_data))
		return STATUS_INVALID_PARAMETER;

	status = FindModuleInfo(OutputBuffer->module_name, &base_address, &image_size);
	if (!NT_SUCCESS(status)) {
		return status;
	}
	KdPrint(("Module %s: base=%x size=%x\n", OutputBuffer->module_name, base_address, image_size));

	addr = base_address + OutputBuffer->start_offset;
	if (!NextSectionInRange(base_address, OutputBuffer->start_offset, IMAGE_SCN_CNT_CODE | IMAGE_SCN_CNT_INITIALIZED_DATA | IMAGE_SCN_CNT_UNINITIALIZED_DATA, &section_start, &section_len, &section_chr) ||
		addr < section_start || addr >= (section_start + section_len - OutputBuffer->data_len))
		return STATUS_INVALID_ADDRESS;
	try {
		if (RtlCompareMemory(OutputBuffer->old_data, (PVOID)addr, OutputBuffer->data_len) != OutputBuffer->data_len) {
			KdPrint(("Verify Failed\n"));
			return STATUS_NOT_FOUND;
		}
	} except (EXCEPTION_EXECUTE_HANDLER) {
		KdPrint(("Caught Exception\n"));
		return STATUS_INVALID_ADDRESS;
	}

	if (!(section_chr & IMAGE_SCN_MEM_NOT_PAGED)) {
		if (section_chr & IMAGE_SCN_CNT_CODE)
			page_lock_handle = MmLockPagableCodeSection((PVOID)addr);
		else
			page_lock_handle = MmLockPagableDataSection((PVOID)addr);
		if (page_lock_handle == NULL)
			return STATUS_INVALID_ADDRESS;
	} else {
		page_lock_handle = NULL;
	}

	for (offset = 0; offset < OutputBuffer->data_len; offset += page_block_len) {
		page_block_len = PAGE_SIZE - BYTE_OFFSET(addr + offset);
		if (page_block_len > (OutputBuffer->data_len - offset))
			page_block_len = OutputBuffer->data_len - offset;
		physical_address = MmGetPhysicalAddress((PVOID)(addr + offset));
		KdPrint(("physical address is: %08x%08x\n", physical_address.HighPart, physical_address.LowPart));
		status = WritePhysicalMemory(physical_address, page_block_len, &OutputBuffer->new_data[offset], &OutputBuffer->old_data[offset]);
		if (!NT_SUCCESS(status))
			break;
	}

	if (page_lock_handle != NULL)
		MmUnlockPagableImageSection(page_lock_handle);

	return status;
}

NTSTATUS FindModuleInfo(IN LPCSTR ModuleName, OUT PULONG pBaseAddress, OUT PULONG pImageSize)
{
	NTSTATUS status;
	PSYSTEM_MODULE_INFORMATION info;
	ULONG	*data, data_len;

	status = ZwQuerySystemInformation(SystemModuleInformation, &data_len, 0, &data_len);
    if ((data = (ULONG *)ExAllocatePool(PagedPool, data_len)) == NULL)
        return STATUS_INSUFFICIENT_RESOURCES;
	status = ZwQuerySystemInformation(SystemModuleInformation, data, data_len, NULL);
	if (!NT_SUCCESS(status)) {
		ExFreePool(data);
		return status;
	}
	for (data_len = data[0], info = (PSYSTEM_MODULE_INFORMATION)&data[1]; data_len > 0; data_len--, info++) {
		CHAR *p = ((p = strrchr(info->ImageName, '\\')) != NULL) ? p + 1 : info->ImageName;
		if (!_stricmp(p, ModuleName)) {
			*pBaseAddress = (ULONG)info->Base;
			*pImageSize = info->Size;
			break;
		}
	}
	ExFreePool(data);
	if (data_len == 0)
		return STATUS_NO_SUCH_FILE;
	return STATUS_SUCCESS;
}

BOOLEAN NextSectionInRange(IN ULONG BaseAddress, IN ULONG StartOffset, IN ULONG ScnCntType, OUT PULONG pSectionStart, OUT PULONG pSectionLen, OUT PULONG pSectionChr OPTIONAL)
{
	ULONG	idx;
	IMAGE_DOS_HEADER	*dos_header;
	IMAGE_NT_HEADERS	*nt_header;
	IMAGE_SECTION_HEADER	*section_headers;

	try {
#define	BASE_ADDRESS(offset)	((ULONG)BaseAddress + (ULONG)(offset))
		dos_header = (IMAGE_DOS_HEADER *)BASE_ADDRESS(0);
		if (dos_header->e_magic != IMAGE_DOS_SIGNATURE ||
			dos_header->e_lfarlc < sizeof(IMAGE_DOS_HEADER)) {
			KdPrint(("Bad IMAGE_DOS_HEADER\n"));
			return FALSE;
		}
		nt_header = (IMAGE_NT_HEADERS *)BASE_ADDRESS(dos_header->e_lfanew);
		if (nt_header->Signature != IMAGE_NT_SIGNATURE ||
			nt_header->FileHeader.SizeOfOptionalHeader != sizeof(IMAGE_OPTIONAL_HEADER) ||
			nt_header->OptionalHeader.Magic != IMAGE_NT_OPTIONAL_HDR_MAGIC ||
			nt_header->OptionalHeader.NumberOfRvaAndSizes != IMAGE_NUMBEROF_DIRECTORY_ENTRIES) {
			KdPrint(("Bad IMAGE_NT_HEADERS\n"));
			return FALSE;
		}
		for (idx = 0, section_headers = IMAGE_FIRST_SECTION(nt_header); idx < nt_header->FileHeader.NumberOfSections; idx++, section_headers++) {
			if ((section_headers->Characteristics & IMAGE_SCN_MEM_DISCARDABLE) ||
				!(section_headers->Characteristics & IMAGE_SCN_MEM_READ))
				continue;	// skip discarded / not readable sections
#ifdef	NONPAGED_ONLY
			if (!(section_headers->Characteristics & IMAGE_SCN_MEM_NOT_PAGED))
				continue;	// skip pageable sections
#endif//NONPAGED_ONLY
			if (!(section_headers->Characteristics & (IMAGE_SCN_CNT_CODE | IMAGE_SCN_CNT_INITIALIZED_DATA | IMAGE_SCN_CNT_UNINITIALIZED_DATA) & ScnCntType))
				continue;
			if (StartOffset >= (section_headers->VirtualAddress + section_headers->Misc.VirtualSize))
				continue;
			KdPrint(("Section %02d: %-8.8s addr=%08x-%08x ch=%08x\n", idx + 1, section_headers->Name, BASE_ADDRESS(section_headers->VirtualAddress), BASE_ADDRESS(section_headers->VirtualAddress) + section_headers->Misc.VirtualSize, section_headers->Characteristics));
			*pSectionStart = BASE_ADDRESS(section_headers->VirtualAddress);
			*pSectionLen = section_headers->Misc.VirtualSize;
			if (pSectionChr != NULL) *pSectionChr = section_headers->Characteristics;
			return TRUE;
		}
#undef	BASE_ADDRESS
	} except (EXCEPTION_EXECUTE_HANDLER) {
		KdPrint(("Caught Exception\n"));
		return FALSE;
	}
	return FALSE;
}


NTSTATUS WritePhysicalMemory(IN PHYSICAL_ADDRESS PhysicalAddress, IN ULONG DataSize, IN PUCHAR Data, IN PUCHAR OldData OPTIONAL)
{
	NTSTATUS	status;
	UNICODE_STRING	physical_memory_unicode_string;
	OBJECT_ATTRIBUTES	object_attributes;
	HANDLE		physical_memory_handle = NULL;
	PHYSICAL_ADDRESS	view_base;
	ULONG		view_size;
	PVOID		virtual_address;

	RtlInitUnicodeString(&physical_memory_unicode_string, L"\\Device\\PhysicalMemory");
	InitializeObjectAttributes(&object_attributes,  &physical_memory_unicode_string, OBJ_CASE_INSENSITIVE, NULL, NULL);
	status = ZwOpenSection(&physical_memory_handle, SECTION_ALL_ACCESS, &object_attributes);
	if (!NT_SUCCESS(status)) {
		KdPrint(("ZwOpenSection failed\n"));
		return status;
	}
	view_base = PhysicalAddress;
	view_size = DataSize;
	virtual_address = NULL;
	status = ZwMapViewOfSection(physical_memory_handle, (HANDLE)-1, &virtual_address, 0L, DataSize, &view_base, &view_size, ViewShare, 0, PAGE_READWRITE);
	ZwClose(physical_memory_handle);
	if (!NT_SUCCESS(status)) {
		KdPrint(("ZwMapViewOfSection failed\n"));
		return status;
	}
	(ULONG)virtual_address += PhysicalAddress.LowPart - view_base.LowPart;
	KdPrint(("memory mapped at address: %x\n", (ULONG)virtual_address));

	if (OldData != NULL &&
		RtlCompareMemory(virtual_address, OldData, DataSize) != DataSize) {
		KdPrint(("Verify Failed\n"));
		ZwUnmapViewOfSection((HANDLE)-1, virtual_address);
		return STATUS_NOT_FOUND;
	}

	RtlCopyMemory(virtual_address, Data, DataSize);

	ZwUnmapViewOfSection((HANDLE)-1, virtual_address);
	return STATUS_SUCCESS;
}

//******************************************************************************

/*
//
// Note: The following sample code could be used from an user-mode application ...
//		 ... to load, call, and then unload the device driver.
//

#include <winioctl.h>
#include "krnpatch.h"

static HANDLE hDriver = INVALID_HANDLE_VALUE;

static BOOL LoadDeviceDriver(LPCSTR name, LPCSTR path)
{
	SC_HANDLE	schManager, schDriver;
	char	driverName[50];
	
	if ((schManager = OpenSCManager(NULL, NULL, SC_MANAGER_ALL_ACCESS)) != NULL) {
		if ((schDriver = CreateService(schManager, name, name, SERVICE_ALL_ACCESS, SERVICE_KERNEL_DRIVER, SERVICE_DEMAND_START, SERVICE_ERROR_NORMAL, path, NULL, NULL, NULL, NULL, NULL)) != NULL ||
			(schDriver = OpenService(schManager, name, SERVICE_ALL_ACCESS)) != NULL) {
			StartService(schDriver, 0, NULL);
			CloseServiceHandle(schDriver);
		}
		CloseServiceHandle(schManager);
	}
	sprintf(driverName, "\\\\.\\%s", name);
	if ((hDriver = CreateFile(driverName, GENERIC_READ | GENERIC_WRITE, 0, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL)) == INVALID_HANDLE_VALUE)
		return FALSE;
	return TRUE;
}

static BOOL IoctlDriver(void)
{
	union {
		KRNPATCH_FIND_DATA	find_data;
		KRNPATCH_READ_DATA	read_data;
		KRNPATCH_WRITE_DATA	write_data;
	} buffer;
	DWORD	dwByteCount;
	const char	module_name[] = .....;

	const BYTE	find_data[] = { ..... };
	memset(&buffer, 0, sizeof(buffer));
	strcpy(buffer.find_data.module_name, module_name);
	buffer.find_data.scn_cnt_type = IMAGE_SCN_CNT_CODE;
	//buffer.find_data.scn_cnt_type = IMAGE_SCN_CNT_CODE | IMAGE_SCN_CNT_INITIALIZED_DATA | IMAGE_SCN_CNT_UNINITIALIZED_DATA;
	buffer.find_data.start_offset = 0;
	buffer.find_data.data_len = sizeof(find_data);
	memcpy(buffer.find_data.data, find_data, buffer.find_data.data_len);
	if (!DeviceIoControl(hDriver, IOCTL_KRNPATCH_FIND_DATA, &buffer, sizeof(buffer.find_data), &buffer, sizeof(buffer.find_data), &dwByteCount, NULL))
		return FALSE;

	DWORD	read_offset = .....;
	BYTE	read_data_buff[.....];
	memset(&buffer, 0, sizeof(buffer));
	strcpy(buffer.read_data.module_name, module_name);
	buffer.read_data.start_offset = read_offset;
	buffer.read_data.data_len = sizeof(read_data_buff);
	if (!DeviceIoControl(hDriver, IOCTL_KRNPATCH_READ_DATA, &buffer, sizeof(buffer.read_data), &buffer, sizeof(buffer.read_data), &dwByteCount, NULL))
		return FALSE;
	memcpy(read_data_buff, buffer.read_data.data, buffer.read_data.data_len);

	DWORD	write_offset = .....;
	const BYTE	old_write_data[] = { ..... };
	const BYTE	new_write_data[] = { ..... };
	memset(&buffer, 0, sizeof(buffer));
	strcpy(buffer.write_data.module_name, module_name);
	buffer.write_data.start_offset = write_offset;
	buffer.write_data.data_len = sizeof(new_write_data);
	memcpy(buffer.write_data.old_data, old_write_data, buffer.write_data.data_len);
	memcpy(buffer.write_data.new_data, new_write_data, buffer.write_data.data_len);
	if (!DeviceIoControl(hDriver, IOCTL_KRNPATCH_WRITE_DATA, &buffer, sizeof(buffer.write_data), &buffer, sizeof(buffer.write_data), &dwByteCount, NULL))
		return FALSE;

	return TRUE;
}

static void UnloadDeviceDriver(LPCSTR name)
{
	SC_HANDLE	schManager, schDriver;
	SERVICE_STATUS	serviceStatus;

	if (hDriver != INVALID_HANDLE_VALUE) {
		CloseHandle(hDriver);
		hDriver = INVALID_HANDLE_VALUE;
	}
	if ((schManager = OpenSCManager(NULL, NULL, SC_MANAGER_ALL_ACCESS)) != NULL) {
		if ((schDriver = OpenService(schManager, name, SERVICE_ALL_ACCESS)) != NULL) {
			ControlService(schDriver, SERVICE_CONTROL_STOP, &serviceStatus);
			DeleteService(schDriver);
			CloseServiceHandle(schDriver);
		}
		CloseServiceHandle(schManager);
	}
}

void driver_sample()
{
	BOOL	res;
	char	buff[MAX_PATH], *p;

	GetModuleFileName(NULL, buff, sizeof(buff));
	p = strrchr(buff, '\\');
	strcpy((p != NULL) ? p + 1 : buff, KRNPATCH_DRIVER_FILENAME);
	if ((res = LoadDeviceDriver(KRNPATCH_DOS_DEVICENAME, buff)))
		res = IoctlDriver();
	UnloadDeviceDriver(KRNPATCH_DOS_DEVICENAME);
}

*/

//******************************************************************************

