
#include <stdio.h>
#include <windows.h>
#include "bleem_nt.h"

//******************************************************************************

static sreg		seg_prefix = NO_SEG;
static sreg		cur_seg_prefix = NO_SEG;
static EA_ADDR	ModRM_EA(0, NO_SEG);
static DWORD	dr_regs[8];
static CONTEXT	ctx_trap;
static LDT_DESCR	ldt_descr[16];
static BOOL		ldt_alloc_flg[16] = { 0, };

// Note: The LDT descriptors addresses starts decreasing from ...
// ... the max. avail address (0xFFF8) to avoid conflicts with ...
// ... the other REAL descriptors already allocated from the O.S.
#define	LDT_ADDR	0xFFF8	// address of first LDT descriptor to return
#define	LDT_SEG		0		// segment used for mapping LDT r/w access

// Number of bytes of the original program stack to reserve ...
// ... when executing the 'ExecX86ContextTrap()' callback
#define	STACK_RESERVE	0x40000

//******************************************************************************

BOOL	AllocLDTDescriptor(DWORD *ldt_addr);
BOOL	FreeLDTDescriptor(DWORD ldt_addr);
BOOL	TrackX86Opcode(LPCONTEXT context);
void	ExecX86ContextTrap(void (WINAPI *trap_fn)(LPCONTEXT context));

//******************************************************************************

BOOL AllocLDTDescriptor(DWORD *ldt_addr) {
	DWORD	idx;
	for (idx = 0; idx < (sizeof(ldt_descr) / sizeof(ldt_descr[0])); idx++) {
		if (ldt_alloc_flg[idx])
			continue;
		// ret the 0-based address of the 'ldt_descr' entry
		*ldt_addr = LDT_ADDR - ((DWORD)&ldt_descr[idx] - (DWORD)&ldt_descr[0]);
		ldt_alloc_flg[idx] = TRUE;
		return TRUE;
	}
	return FALSE;
}

BOOL FreeLDTDescriptor(DWORD ldt_addr) {
	DWORD	idx = (LDT_ADDR - ldt_addr) / sizeof(ldt_descr[0]);
	if (idx > (sizeof(ldt_descr) / sizeof(ldt_descr[0])) ||
		!ldt_alloc_flg[idx])
		return FALSE;
	ldt_alloc_flg[idx] = FALSE;
	return TRUE;
}

static LDT_DESCR *GetLDTDescrFromAddr(DWORD ldt_addr) {
	DWORD	idx = (LDT_ADDR - ldt_addr) / sizeof(ldt_descr[0]);
	if (idx > (sizeof(ldt_descr) / sizeof(ldt_descr[0])) ||
		!ldt_alloc_flg[idx])
		return NULL;
	return &ldt_descr[idx];
}

//******************************************************************************

__declspec(naked) void ExecX86ContextTrap(void (WINAPI *trap_fn)(LPCONTEXT context))
{
	__asm pushfd
	__asm pop	[ctx_trap.EFlags]
	__asm mov	[ctx_trap.Eax],eax
	__asm mov	[ctx_trap.Ecx],ecx
	__asm mov	[ctx_trap.Edx],edx
	__asm mov	[ctx_trap.Ebx],ebx
	__asm mov	[ctx_trap.Ebp],ebp
	__asm mov	[ctx_trap.Esi],esi
	__asm mov	[ctx_trap.Edi],edi
	__asm xor	eax,eax
	__asm mov	ax,es
	__asm mov	[ctx_trap.SegEs],eax
	__asm mov	ax,cs
	__asm mov	[ctx_trap.SegCs],eax
	__asm mov	ax,ss
	__asm mov	[ctx_trap.SegSs],eax
	__asm mov	ax,ds
	__asm mov	[ctx_trap.SegDs],eax
	__asm mov	ax,fs
	__asm mov	[ctx_trap.SegFs],eax
	__asm mov	ax,gs
	__asm mov	[ctx_trap.SegGs],eax
	__asm pop	[ctx_trap.Eip]
	__asm pop	eax
	__asm mov	[ctx_trap.Esp],esp
	__asm mov	ecx,STACK_RESERVE / PAGE_SIZE
	__asm stk_reserve:
	__asm sub	esp,PAGE_SIZE
	__asm test	[esp],eax
	__asm loop	stk_reserve
	__asm cld
	__asm push	offset ctx_trap
	__asm call	eax
	__asm xor	eax,eax
//	__asm mov	eax,[ctx_trap.SegEs]
//	__asm mov	es,ax
//	__asm mov	eax,[ctx_trap.SegDs]
//	__asm mov	ds,ax
	__asm mov	eax,[ctx_trap.Eax]
	__asm mov	ecx,[ctx_trap.Ecx]
	__asm mov	edx,[ctx_trap.Edx]
	__asm mov	ebx,[ctx_trap.Ebx]
	__asm mov	ebp,[ctx_trap.Ebp]
	__asm mov	esi,[ctx_trap.Esi]
	__asm mov	edi,[ctx_trap.Edi]
	__asm push	[ctx_trap.EFlags]
	__asm popfd
//	__asm mov	eax,[ctx_trap.SegSs]
//	__asm mov	ss,ax
	__asm mov	esp,[ctx_trap.Esp]
	__asm jmp	[ctx_trap.Eip]
}


//******************************************************************************

BYTE READ_BYTE(LPCONTEXT context, EA_ADDR EA_src) {
//	EA_src.seg;
	return *(BYTE *)EA_src.addr;
}
WORD READ_WORD(LPCONTEXT context, EA_ADDR EA_src) {
//	EA_src.seg;
	return *(WORD *)EA_src.addr;
}
DWORD READ_DWORD(LPCONTEXT context, EA_ADDR EA_src) {
//	EA_src.seg;
	if (context->SegDs == LDT_SEG)	// if SegDs == LDT_SEG, read from LDT descriptor table
		return *(DWORD *)((DWORD)GetLDTDescrFromAddr(EA_src.addr & ~(sizeof(LDT_DESCR) - 1)) + (EA_src.addr & (sizeof(LDT_DESCR) - 1)));
	return *(DWORD *)EA_src.addr;
}
DWORDLONG READ_QWORD(LPCONTEXT context, EA_ADDR EA_src) {
//	EA_src.seg;
	return *(DWORDLONG *)EA_src.addr;
}
void WRITE_BYTE(LPCONTEXT context, EA_ADDR EA_dest, BYTE value) {
//	EA_dest.seg;
	*(BYTE *)EA_dest.addr = value;
}
void WRITE_WORD(LPCONTEXT context, EA_ADDR EA_dest, WORD value) {
//	EA_dest.seg;
	*(WORD *)EA_dest.addr = value;
}
void WRITE_DWORD(LPCONTEXT context, EA_ADDR EA_dest, DWORD value) {
//	EA_dest.seg;
//	*(DWORD *)EA_dest.addr = value;
	if (context->SegDs == LDT_SEG)	// if SegDs == LDT_SEG, write into LDT descriptor table
		*(DWORD *)((DWORD)GetLDTDescrFromAddr(EA_dest.addr & ~(sizeof(LDT_DESCR) - 1)) + (EA_dest.addr & (sizeof(LDT_DESCR) - 1))) = value;
	else
		*(DWORD *)EA_dest.addr = value;
}

//******************************************************************************

static DWORD GET_REG32(LPCONTEXT context, reg32 iReg) {
	DWORD	value = 0;
	switch (iReg) {
	case EAX: value = context->Eax; break;
	case ECX: value = context->Ecx; break;
	case EDX: value = context->Edx; break;
	case EBX: value = context->Ebx; break;
	case ESP: value = context->Esp; break;
	case EBP: value = context->Ebp; break;
	case ESI: value = context->Esi; break;
	case EDI: value = context->Edi; break;
	default: __asm int 3; break;
	}
	return value;
}

static void SET_REG32(LPCONTEXT context, reg32 iReg, DWORD value) {
	switch (iReg) {
	case EAX: context->Eax = value; break;
	case ECX: context->Ecx = value; break;
	case EDX: context->Edx = value; break;
	case EBX: context->Ebx = value; break;
	case ESP: context->Esp = value; break;
	case EBP: context->Ebp = value; break;
	case ESI: context->Esi = value; break;
	case EDI: context->Edi = value; break;
	default: __asm int 3; break;
	}
}

static WORD GET_REG16(LPCONTEXT context, reg16 iReg) {
	return (WORD)GET_REG32(context, (reg32)iReg);
}

static void SET_REG16(LPCONTEXT context, reg16 iReg, WORD value) {
	DWORD	tmp_value = GET_REG32(context, (reg32)iReg);
	tmp_value = (tmp_value & 0xFFFF0000) | value;
	SET_REG32(context, (reg32)iReg, tmp_value);
}

static BYTE GET_REG8(LPCONTEXT context, reg8 iReg) {
	DWORD	value = GET_REG32(context, (reg32)(iReg & 3));
	if (iReg & 4) value >>= 8;	// iReg: 0-3 = al-bl, 4-7 = ah-bh
	return (BYTE)value;
}

static void SET_REG8(LPCONTEXT context, reg8 iReg, BYTE value) {
	DWORD	tmp_value = GET_REG32(context, (reg32)(iReg & 3));
	if (iReg & 4)					// iReg: 0-3 = al-bl, 4-7 = ah-bh
		tmp_value = (tmp_value & 0xFFFF00FF) | (value << 8);
	else
		tmp_value = (tmp_value & 0xFFFFFF00) | value;
	SET_REG32(context, (reg32)(iReg & 3), tmp_value);
}

static DWORD GET_SEG(LPCONTEXT context, sreg iSeg) {
	DWORD	value = 0;
	switch (iSeg) {
	case ES: value = context->SegEs; break;
	case CS: value = context->SegCs; break;
	case SS: value = context->SegSs; break;
	case DS: value = context->SegDs; break;
	case FS: value = context->SegFs; break;
	case GS: value = context->SegGs; break;
	default: __asm int 3; break;
	}
	return value;
}

static void SET_SEG(LPCONTEXT context, sreg iSeg, DWORD value) {
	switch (iSeg) {
	case ES: context->SegEs = value; break;
	case CS: context->SegCs = value; break;
	case SS: context->SegSs = value; break;
	case DS: context->SegDs = value; break;
	case FS: context->SegFs = value; break;
	case GS: context->SegGs = value; break;
	default: __asm int 3; break;
	}
}

static DWORD GET_FLAGS(LPCONTEXT context) {
	DWORD	value = context->EFlags;
	return value;
}

static void SET_FLAGS(LPCONTEXT context, DWORD value) {
	context->EFlags = value;
}

//******************************************************************************

static inline OPCODE	PEEK_OPCODE(LPCONTEXT context)	{ OPCODE tmp = *(OPCODE *)context->Eip; return tmp; }
static inline OPCODE	FETCH_OPCODE(LPCONTEXT context) { OPCODE tmp = *(OPCODE *)context->Eip++; return tmp; }
static inline BYTE		FETCH_BYTE(LPCONTEXT context)	{ BYTE tmp = *(BYTE *)context->Eip++; return tmp; }
static inline WORD		FETCH_WORD(LPCONTEXT context)	{ WORD tmp = *(WORD *)context->Eip; context->Eip += 2; return tmp; }
static inline DWORD		FETCH_DWORD(LPCONTEXT context)	{ DWORD tmp = *(DWORD *)context->Eip; context->Eip += 4; return tmp; }
static inline LONG		FETCH_OFFSET32(LPCONTEXT context) { LONG tmp = *(LONG *)context->Eip; context->Eip += 4; return tmp; }
static inline LONG		FETCH_OFFSET16(LPCONTEXT context) { LONG tmp = *(SHORT *)context->Eip; context->Eip += 2; return tmp; }
static inline LONG		FETCH_OFFSET8(LPCONTEXT context) { LONG tmp = *(CHAR *)context->Eip++; return tmp; }

static inline void		PUSH_DWORD(LPCONTEXT context, DWORD value) { context->Esp -= 4; *(DWORD *)context->Esp = value; }
static inline DWORD		POP_DWORD(LPCONTEXT context) { DWORD value = *(DWORD *)context->Esp; context->Esp += 4; return value; }

static inline void		SET_SEG_PREFIX(sreg iSeg) { seg_prefix = iSeg; }
static inline void		SET_CUR_SEG_PREFIX(sreg iSeg) { cur_seg_prefix = iSeg; }
static inline sreg		REG32_DEFAULT_SREG(reg32 reg) { return (reg == ESP || reg == EBP) ? SS : DS; }

static inline DWORD		GET_REG_DR(DWORD regno) { return dr_regs[regno]; }
static inline void		SET_REG_DR(DWORD regno, DWORD value) { dr_regs[regno] = value; }

static BOOL	test_cond_o(LPCONTEXT context) { return (context->EFlags & OF) ? TRUE : FALSE; }
static BOOL	test_cond_b(LPCONTEXT context) { return (context->EFlags & CF) ? TRUE : FALSE; }
static BOOL	test_cond_e(LPCONTEXT context) { return (context->EFlags & ZF) ? TRUE : FALSE; }
static BOOL	test_cond_be(LPCONTEXT context) { return (context->EFlags & (CF | ZF)) ? TRUE : FALSE; }
static BOOL	test_cond_s(LPCONTEXT context) { return (context->EFlags & SF) ? TRUE : FALSE; }
static BOOL	test_cond_p(LPCONTEXT context) { return (context->EFlags & PF) ? TRUE : FALSE; }
static BOOL	test_cond_l(LPCONTEXT context) { return ((context->EFlags & (SF | OF)) != 0) && ((context->EFlags & (SF | OF)) != (SF | OF)) ? TRUE : FALSE; }
static BOOL	test_cond_le(LPCONTEXT context) { return (context->EFlags & ZF) || (((context->EFlags & (SF | OF)) != 0) && ((context->EFlags & (SF | OF)) != (SF | OF))) ? TRUE : FALSE; }
static BOOL	test_cond_ecxz(LPCONTEXT context) { return (context->Ecx == 0) ? TRUE : FALSE; }

//******************************************************************************

static void ModRMGetEA(LPCONTEXT context, OPCODE ModRM) {
	if (ModRM.adr_op == 3)
		__asm int 3
	if (ModRM.mem_op == 4) {
		OPCODE SubRM = FETCH_OPCODE(context);
		if (SubRM.mem_op == 5 && ModRM.adr_op == 0) {
			// adr = abs32 + reg2 * N, reg2 = SubRM.reg_op, N = SubRM.adr_op
			ModRM_EA.addr = FETCH_DWORD(context);
			ModRM_EA.seg = DS;
		} else {
			// adr = reg1 + reg2 * N, reg1 = SubRM.mem_op, reg2 = SubRM.reg_op, N = SubRM.adr_op
			ModRM_EA.addr = GET_REG32(context, (reg32)SubRM.mem_op);
			ModRM_EA.seg = REG32_DEFAULT_SREG((reg32)SubRM.mem_op);
		}
		if ((reg32)SubRM.reg_op != ESP)	// never use ESP as 'reg2'
			ModRM_EA.addr += (GET_REG32(context, (reg32)SubRM.reg_op) << SubRM.adr_op);
	} else if (ModRM.mem_op == 5 && ModRM.adr_op == 0) {
		// adr = abs32
		ModRM_EA.addr = FETCH_DWORD(context);
		ModRM_EA.seg = DS;
	} else {
		// adr = reg1, reg1 = ModRM.mem_op
		ModRM_EA.addr = GET_REG32(context, (reg32)ModRM.mem_op);
		ModRM_EA.seg = REG32_DEFAULT_SREG((reg32)ModRM.mem_op);
	}
	switch (ModRM.adr_op) {
	case 0:		// [adr]
		break;
	case 1:		// [adr + offset8]
		ModRM_EA.addr += FETCH_OFFSET8(context);
		break;
	case 2:		// [adr + offset32]
		ModRM_EA.addr += FETCH_OFFSET32(context);
		break;
	}
}

static BYTE GetModRMRM8(LPCONTEXT context, OPCODE ModRM) {
	BYTE	value;
	if (ModRM.adr_op == 3) {	// reg8 (al-bl,ah-bh)
		value = GET_REG8(context, (reg8)ModRM.mem_op);
	} else {					// byte ptr [ModRM_EA]
		ModRMGetEA(context, ModRM);
		value = READ_BYTE(context, ModRM_EA);
	}
	return value;
}
static void SetModRMRM8(LPCONTEXT context, OPCODE ModRM, BYTE value) {
	if (ModRM.adr_op == 3) {	// reg8 (al-bl,ah-bh)
		SET_REG8(context, (reg8)ModRM.mem_op, value);
	} else {					// byte ptr [ModRM_EA]
		ModRMGetEA(context, ModRM);
		WRITE_BYTE(context, ModRM_EA, value);
	}
}
static void PutbackModRMRM8(LPCONTEXT context, OPCODE ModRM, BYTE value) {
	if (ModRM.adr_op == 3) {	// reg8 (al-bl,ah-bh)
		SET_REG8(context, (reg8)ModRM.mem_op, value);
	} else {					// byte ptr [ModRM_EA]
		WRITE_BYTE(context, ModRM_EA, value);
	}
}

static WORD GetModRMRM16(LPCONTEXT context, OPCODE ModRM) {
	WORD	value;
	if (ModRM.adr_op == 3) {	// reg16 (ax-di)
		value = GET_REG16(context, (reg16)ModRM.mem_op);
	} else {					// word ptr [ModRM_EA]
		ModRMGetEA(context, ModRM);
		value = READ_WORD(context, ModRM_EA);
	}
	return value;
}
static void SetModRMRM16(LPCONTEXT context, OPCODE ModRM, WORD value) {
	if (ModRM.adr_op == 3) {	// reg16 (ax-di)
		SET_REG16(context, (reg16)ModRM.mem_op, value);
	} else {					// word ptr [ModRM_EA]
		ModRMGetEA(context, ModRM);
		WRITE_WORD(context, ModRM_EA, value);
	}
}
static void PutbackModRMRM16(LPCONTEXT context, OPCODE ModRM, WORD value) {
	if (ModRM.adr_op == 3) {	// reg16 (ax-di)
		SET_REG16(context, (reg16)ModRM.mem_op, value);
	} else {					// word ptr [ModRM_EA]
		WRITE_WORD(context, ModRM_EA, value);
	}
}

static DWORD GetModRMRM32(LPCONTEXT context, OPCODE ModRM) {
	DWORD	value;
	if (ModRM.adr_op == 3) {	// reg32 (eax-edi)
		value = GET_REG32(context, (reg32)ModRM.mem_op);
	} else {					// dword ptr [ModRM_EA]
		ModRMGetEA(context, ModRM);
		value = READ_DWORD(context, ModRM_EA);
	}
	return value;
}
static void SetModRMRM32(LPCONTEXT context, OPCODE ModRM, DWORD value) {
	if (ModRM.adr_op == 3) {	// reg32 (eax-edi)
		SET_REG32(context, (reg32)ModRM.mem_op, value);
	} else {					// dword ptr [ModRM_EA]
		ModRMGetEA(context, ModRM);
		WRITE_DWORD(context, ModRM_EA, value);
	}
}
static void PutbackModRMRM32(LPCONTEXT context, OPCODE ModRM, DWORD value) {
	if (ModRM.adr_op == 3) {	// reg32 (eax-edi)
		SET_REG32(context, (reg32)ModRM.mem_op, value);
	} else {					// dword ptr [ModRM_EA]
		WRITE_DWORD(context, ModRM_EA, value);
	}
}

static ULARGE_INTEGER GetModRMRM64(LPCONTEXT context, OPCODE ModRM) {
	ULARGE_INTEGER	value;
	if (ModRM.adr_op == 3) {	// reg32 (eax-edi)
		__asm int 3		// invalid
	} else {					// qword ptr [ModRM_EA]
		ModRMGetEA(context, ModRM);
		value.QuadPart = READ_QWORD(context, ModRM_EA);
	}
	return value;
}

//******************************************************************************

#define	LOAD_FLAGS(context)	DWORD	flags = GET_FLAGS(context);
#define	SAVE_FLAGS(context)	SET_FLAGS(context, flags);

#define	GET_AH_FROM_SRCB()	{ __asm mov	ah,src }
#define	GET_CL_FROM_SRCB()	{ __asm mov	cl,src }
#define	GET_DL_FROM_SRCB()	{ __asm mov	dl,src }
#define	GET_EDX_FROM_SRCD()	{ __asm mov	edx,src }
#define	GET_AL_FROM_DESTB()	{ __asm mov	al,dest }
#define	GET_EAX_FROM_DESTD(){ __asm mov	eax,dest }
#define	GET_FLG_FROM_FLAGS(){ __asm pushfd __asm push flags __asm popfd }

#define	PUT_AH_INTO_SRCB()	{ __asm mov	src,ah }
#define	PUT_DL_INTO_SRCB()	{ __asm mov	src,dl }
#define	PUT_EDX_INTO_SRCD()	{ __asm mov	src,edx }
#define	PUT_AL_INTO_DESTB()	{ __asm mov	dest,al }
#define	PUT_EAX_INTO_DESTD(){ __asm mov	dest,eax }
#define	PUT_FLG_INTO_FLAGS(){ __asm pushfd __asm pop flags __asm popfd }

#define	CALL_OPC(opc)		{ __asm opc }
#define	CALL_OPC_AL(opc)	{ __asm opc	al }
#define	CALL_OPC_AH(opc)	{ __asm opc	ah }
#define	CALL_OPC_EAX(opc)	{ __asm opc	eax }
#define	CALL_OPC_AL_AH(opc)	{ __asm opc	al,ah }
#define	CALL_OPC_AL_CL(opc)	{ __asm opc	al,cl }
#define	CALL_OPC_AL_DL(opc)	{ __asm opc	al,dl }
#define	CALL_OPC_EAX_CL(opc){ __asm opc	eax,cl }
#define	CALL_OPC_EAX_EDX(opc)	{ __asm opc	eax,edx }

#define	GET_AH_AL_FLG_FROM_SRCB_DESTB_FLAGS()		\
	GET_AH_FROM_SRCB()								\
	GET_AL_FROM_DESTB()								\
	GET_FLG_FROM_FLAGS()
#define	GET_CL_AL_FLG_FROM_SRCB_DESTB_FLAGS()		\
	GET_CL_FROM_SRCB()								\
	GET_AL_FROM_DESTB()								\
	GET_FLG_FROM_FLAGS()
#define	GET_DL_AL_FLG_FROM_SRCB_DESTB_FLAGS()		\
	GET_DL_FROM_SRCB()								\
	GET_AL_FROM_DESTB()								\
	GET_FLG_FROM_FLAGS()
#define	GET_CL_EAX_FLG_FROM_SRCB_DESTD_FLAGS()		\
	GET_CL_FROM_SRCB()								\
	GET_EAX_FROM_DESTD()							\
	GET_FLG_FROM_FLAGS()
#define	GET_EDX_EAX_FLG_FROM_SRCD_DESTD_FLAGS()		\
	GET_EDX_FROM_SRCD()								\
	GET_EAX_FROM_DESTD()							\
	GET_FLG_FROM_FLAGS()
#define	GET_DL_AL_FROM_SRCB_DESTB()					\
	GET_DL_FROM_SRCB()								\
	GET_AL_FROM_DESTB()
#define	GET_EDX_EAX_FROM_SRCD_DESTD()				\
	GET_EDX_FROM_SRCD()								\
	GET_EAX_FROM_DESTD()
#define	GET_AL_FLG_FROM_DESTB_FLAGS()				\
	GET_AL_FROM_DESTB()								\
	GET_FLG_FROM_FLAGS()
#define	GET_EAX_FLG_FROM_DESTD_FLAGS()				\
	GET_EAX_FROM_DESTD()							\
	GET_FLG_FROM_FLAGS()

#define	PUT_AH_AL_FLG_INTO_SRCB_DESTB_FLAGS()		\
	PUT_AH_INTO_SRCB()								\
	PUT_AL_INTO_DESTB()								\
	PUT_FLG_INTO_FLAGS()
#define	PUT_DL_AL_FLG_INTO_SRCB_DESTB_FLAGS()		\
	PUT_DL_INTO_SRCB()								\
	PUT_AL_INTO_DESTB()								\
	PUT_FLG_INTO_FLAGS()
#define	PUT_EDX_EAX_FLG_INTO_SRCD_DESTD_FLAGS()		\
	PUT_EDX_INTO_SRCD()								\
	PUT_EAX_INTO_DESTD()							\
	PUT_FLG_INTO_FLAGS()
#define	PUT_DL_AL_INTO_SRCB_DESTB()					\
	PUT_DL_INTO_SRCB()								\
	PUT_AL_INTO_DESTB()
#define	PUT_EDX_EAX_INTO_SRCD_DESTD()				\
	PUT_EDX_INTO_SRCD()								\
	PUT_EAX_INTO_DESTD()
#define	PUT_AL_FLG_INTO_DESTB_FLAGS()				\
	PUT_AL_INTO_DESTB()								\
	PUT_FLG_INTO_FLAGS()
#define	PUT_EAX_FLG_INTO_DESTD_FLAGS()				\
	PUT_EAX_INTO_DESTD()							\
	PUT_FLG_INTO_FLAGS()

//******************************************************************************
// 'opc'			in=eax,efl		out=eax,efl
#define	OPCEAX_FN__TO_DEST_FLAGS(opc)				\
	LOAD_FLAGS(context)								\
	GET_EAX_FLG_FROM_DESTD_FLAGS()					\
	CALL_OPC(opc)									\
	PUT_EAX_FLG_INTO_DESTD_FLAGS()					\
	SAVE_FLAGS(context)
// 'opc'			in=eax,edx,efl	out=eax,edx,efl
#define	OPCEAXEDX_FN__TO_SRC_DEST_FLAGS(opc)		\
	LOAD_FLAGS(context)								\
	GET_EDX_EAX_FLG_FROM_SRCD_DESTD_FLAGS()			\
	CALL_OPC(opc)									\
	PUT_EDX_EAX_FLG_INTO_SRCD_DESTD_FLAGS()			\
	SAVE_FLAGS(context)
// 'opc al'			in=al,efl		out=al,efl
#define	OPC_AL_FN__TO_DEST_FLAGS(opc)				\
	LOAD_FLAGS(context)								\
	GET_AL_FLG_FROM_DESTB_FLAGS()					\
	CALL_OPC_AL(opc)								\
	PUT_AL_FLG_INTO_DESTB_FLAGS()					\
	SAVE_FLAGS(context)
// 'opc ah'			in=al,ah,efl	out=al,ah,efl
#define	OPCAL_AH_FN__TO_SRC_DEST_FLAGS(opc)		\
	LOAD_FLAGS(context)								\
	GET_AH_AL_FLG_FROM_SRCB_DESTB_FLAGS()			\
	CALL_OPC_AH(opc)								\
	PUT_AH_AL_FLG_INTO_SRCB_DESTB_FLAGS()			\
	SAVE_FLAGS(context)
// 'opc eax'		in=eax,efl		out=eax,efl
#define	OPC_EAX_FN__TO_DEST_FLAGS(opc)				\
	LOAD_FLAGS(context)								\
	GET_EAX_FLG_FROM_DESTD_FLAGS()					\
	CALL_OPC_EAX(opc)								\
	PUT_EAX_FLG_INTO_DESTD_FLAGS()					\
	SAVE_FLAGS(context)
// 'opc edx'		in=eax,edx,efl	out=eax,edx,efl
#define	OPCEAX_EDX_FN__TO_SRC_DEST_FLAGS(opc)		\
	LOAD_FLAGS(context)								\
	GET_EDX_EAX_FLG_FROM_SRCD_DESTD_FLAGS()			\
	CALL_OPC(opc edx)								\
	PUT_EDX_EAX_FLG_INTO_SRCD_DESTD_FLAGS()			\
	SAVE_FLAGS(context)
// 'opc al,ah'		in=al,ah,efl	out=al,ah,efl
#define	OPC_AL_AH_FN__TO_SRC_DEST_FLAGS(opc)		\
	LOAD_FLAGS(context)								\
	GET_AH_AL_FLG_FROM_SRCB_DESTB_FLAGS()			\
	CALL_OPC_AL_AH(opc)								\
	PUT_AH_AL_FLG_INTO_SRCB_DESTB_FLAGS()			\
	SAVE_FLAGS(context)
// 'opc al,cl'		in=al,cl,efl	out=al,efl
#define	OPC_AL_CL_FN__TO_DEST_FLAGS(opc)			\
	LOAD_FLAGS(context)								\
	GET_CL_AL_FLG_FROM_SRCB_DESTB_FLAGS()			\
	CALL_OPC_AL_CL(opc)								\
	PUT_AL_FLG_INTO_DESTB_FLAGS()					\
	SAVE_FLAGS(context)
// 'opc al,dl'		in=al,dl,efl	out=al,efl
#define	OPC_AL_DL_FN__TO_DEST_FLAGS(opc)			\
	LOAD_FLAGS(context)								\
	GET_DL_AL_FLG_FROM_SRCB_DESTB_FLAGS()			\
	CALL_OPC_AL_DL(opc)								\
	PUT_AL_FLG_INTO_DESTB_FLAGS()					\
	SAVE_FLAGS(context)
// 'opc al,dl'		in=al,dl,efl	out=efl
#define	OPC_AL_DL_FN__TO_FLAGS(opc)					\
	LOAD_FLAGS(context)								\
	GET_DL_AL_FLG_FROM_SRCB_DESTB_FLAGS()			\
	CALL_OPC_AL_DL(opc)								\
	PUT_FLG_INTO_FLAGS()							\
	SAVE_FLAGS(context)
// 'opc eax,cl'		in=eax,cl,efl	out=eax,efl
#define	OPC_EAX_CL_FN__TO_DEST_FLAGS(opc)			\
	LOAD_FLAGS(context)								\
	GET_CL_EAX_FLG_FROM_SRCB_DESTD_FLAGS()			\
	CALL_OPC_EAX_CL(opc)							\
	PUT_EAX_FLG_INTO_DESTD_FLAGS()					\
	SAVE_FLAGS(context)
// 'opc eax,edx'	in=eax,edx,efl	out=eax,edx,efl
#define	OPC_EAX_EDX_FN__TO_SRC_DEST_FLAGS(opc)		\
	LOAD_FLAGS(context)								\
	GET_EDX_EAX_FLG_FROM_SRCD_DESTD_FLAGS()			\
	CALL_OPC_EAX_EDX(opc)							\
	PUT_EDX_EAX_FLG_INTO_SRCD_DESTD_FLAGS()			\
	SAVE_FLAGS(context)
// 'opc eax,edx'	in=eax,edx,efl	out=eax,efl
#define	OPC_EAX_EDX_FN__TO_DEST_FLAGS(opc)			\
	LOAD_FLAGS(context)								\
	GET_EDX_EAX_FLG_FROM_SRCD_DESTD_FLAGS()			\
	CALL_OPC_EAX_EDX(opc)							\
	PUT_EAX_FLG_INTO_DESTD_FLAGS()					\
	SAVE_FLAGS(context)
// 'opc eax,edx'	in=eax,edx,efl	out=efl
#define	OPC_EAX_EDX_FN__TO_FLAGS(opc)				\
	LOAD_FLAGS(context)								\
	GET_EDX_EAX_FLG_FROM_SRCD_DESTD_FLAGS()			\
	CALL_OPC_EAX_EDX(opc)							\
	PUT_FLG_INTO_FLAGS()							\
	SAVE_FLAGS(context)


//******************************************************************************

//******************************************************************************

#define	DECLARE_I_Invalid(opc)									\
static inline BOOL i_## opc ## (LPCONTEXT context) {			\
	{ __asm int 3 }												\
	return FALSE;												\
}

#define	DECLARE_I_Trap(opc)										\
static inline BOOL i_## opc ## (LPCONTEXT context) {			\
	{ __asm int 3 }												\
	return FALSE;												\
}

//******************************************************************************

static BOOL i_add_b_r8(LPCONTEXT context) {		// 00
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = GET_REG8(context, (reg8)ModRM.reg_op);
	OPC_AL_DL_FN__TO_DEST_FLAGS(add);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_add_d_r32(LPCONTEXT context) {	// 01
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = GET_REG32(context, (reg32)ModRM.reg_op);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(add);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_add_r8_b(LPCONTEXT context) {		// 02
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GET_REG8(context, (reg8)ModRM.reg_op);
	BYTE	src = GetModRMRM8(context, ModRM);
	OPC_AL_DL_FN__TO_DEST_FLAGS(add);
	SET_REG8(context, (reg8)ModRM.reg_op, dest);
	return TRUE;
}

static BOOL i_add_r32_d(LPCONTEXT context) {	// 03
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GET_REG32(context, (reg32)ModRM.reg_op);
	DWORD	src = GetModRMRM32(context, ModRM);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(add);
	SET_REG32(context, (reg32)ModRM.reg_op, dest);
	return TRUE;
}

static BOOL i_add_al_i8(LPCONTEXT context) {	// 04
	BYTE	dest = GET_REG8(context, AL);
	BYTE	src = FETCH_BYTE(context);
	OPC_AL_DL_FN__TO_DEST_FLAGS(add);
	SET_REG8(context, AL, dest);
	return TRUE;
}

static BOOL i_add_eax_i32(LPCONTEXT context) {	// 05
	DWORD	dest = GET_REG32(context, EAX);
	DWORD	src = FETCH_DWORD(context);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(add);
	SET_REG32(context, EAX, dest);
	return TRUE;
}

static BOOL i_push_es(LPCONTEXT context) {		// 06
	DWORD	dest = GET_SEG(context, ES);
	PUSH_DWORD(context, dest);
	return TRUE;
}

static BOOL i_pop_es(LPCONTEXT context) {		// 07
	DWORD	dest = POP_DWORD(context);
	SET_SEG(context, ES, dest);
	return TRUE;
}

static BOOL i_or_b_r8(LPCONTEXT context) {		// 08
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = GET_REG8(context, (reg8)ModRM.reg_op);
	OPC_AL_DL_FN__TO_DEST_FLAGS(or);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_or_d_r32(LPCONTEXT context) {		// 09
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = GET_REG32(context, (reg32)ModRM.reg_op);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(or);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_or_r8_b(LPCONTEXT context) {		// 0A
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GET_REG8(context, (reg8)ModRM.reg_op);
	BYTE	src = GetModRMRM8(context, ModRM);
	OPC_AL_DL_FN__TO_DEST_FLAGS(or);
	SET_REG8(context, (reg8)ModRM.reg_op, dest);
	return TRUE;
}

static BOOL i_or_r32_d(LPCONTEXT context) {		// 0B
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GET_REG32(context, (reg32)ModRM.reg_op);
	DWORD	src = GetModRMRM32(context, ModRM);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(or);
	SET_REG32(context, (reg32)ModRM.reg_op, dest);
	return TRUE;
}

static BOOL i_or_al_i8(LPCONTEXT context) {		// 0C
	BYTE	dest = GET_REG8(context, AL);
	BYTE	src = FETCH_BYTE(context);
	OPC_AL_DL_FN__TO_DEST_FLAGS(or);
	SET_REG8(context, AL, dest);
	return TRUE;
}

static BOOL i_or_eax_i32(LPCONTEXT context) {	// 0D
	DWORD	dest = GET_REG32(context, EAX);
	DWORD	src = FETCH_DWORD(context);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(or);
	SET_REG32(context, EAX, dest);
	return TRUE;
}

static BOOL i_push_cs(LPCONTEXT context) {		// 0E
	DWORD	dest = GET_SEG(context, CS);
	PUSH_DWORD(context, dest);
	return TRUE;
}

static BOOL i_lgdt_f(LPCONTEXT context) {		// 0F 01 ?2?
	OPCODE	ModRM = FETCH_OPCODE(context);
	if (ModRM.adr_op == 3)	// 'lgdt reg' is invalid
		return FALSE;
	DWORD	dest = GetModRMRM32(context, ModRM);
	// do nothing
	return TRUE;
}

static BOOL i_lidt_f(LPCONTEXT context) {		// 0F 01 ?3?
	OPCODE	ModRM = FETCH_OPCODE(context);
	if (ModRM.adr_op == 3)	// 'lidt reg' is invalid
		return FALSE;
	DWORD	dest = GetModRMRM32(context, ModRM);
	// do nothing
	return TRUE;
}

static BOOL i_0f01pre(LPCONTEXT context) {		// 0F 01
	BOOL	res = FALSE;
	OPCODE	ModRM = PEEK_OPCODE(context);
	switch (ModRM.reg_op) {
//	case 0x00: res = i_sgdt_f(context); break;
//	case 0x01: res = i_sidt_f(context); break;
	case 0x02: res = i_lgdt_f(context); break;
	case 0x03: res = i_lidt_f(context); break;
//	case 0x04: res = i_smsw_w(context); break;
	case 0x05: break;
//	case 0x06: res = i_lmsw_w(context); break;
	default:   break;
	}
	return res;
}

static BOOL i_mov_r32_rdr(LPCONTEXT context) {	// 0F 21
	OPCODE	ModRM = FETCH_OPCODE(context);
	if (ModRM.adr_op != 3)
		return FALSE;
	SET_REG32(context, (reg32)ModRM.mem_op, GET_REG_DR(ModRM.reg_op));
	return TRUE;
}

static BOOL i_mov_rdr_r32(LPCONTEXT context) {	// 0F 23
	OPCODE	ModRM = FETCH_OPCODE(context);
	if (ModRM.adr_op != 3)
		return FALSE;
	SET_REG_DR(ModRM.reg_op, GET_REG32(context, (reg32)ModRM.mem_op));
	return TRUE;
}

static BOOL i_jo_o32(LPCONTEXT context) {		// 0F 80
	DWORD	offset = FETCH_OFFSET32(context);
	if (test_cond_o(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_jno_o32(LPCONTEXT context) {		// 0F 81
	DWORD	offset = FETCH_OFFSET32(context);
	if (!test_cond_o(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_jb_o32(LPCONTEXT context) {		// 0F 82
	DWORD	offset = FETCH_OFFSET32(context);
	if (test_cond_b(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_jae_o32(LPCONTEXT context) {		// 0F 83
	DWORD	offset = FETCH_OFFSET32(context);
	if (!test_cond_b(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_je_o32(LPCONTEXT context) {		// 0F 84
	DWORD	offset = FETCH_OFFSET32(context);
	if (test_cond_e(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_jne_o32(LPCONTEXT context) {		// 0F 85
	DWORD	offset = FETCH_OFFSET32(context);
	if (!test_cond_e(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_jbe_o32(LPCONTEXT context) {		// 0F 86
	DWORD	offset = FETCH_OFFSET32(context);
	if (test_cond_be(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_ja_o32(LPCONTEXT context) {		// 0F 87
	DWORD	offset = FETCH_OFFSET32(context);
	if (!test_cond_be(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_js_o32(LPCONTEXT context) {		// 0F 88
	DWORD	offset = FETCH_OFFSET32(context);
	if (test_cond_s(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_jns_o32(LPCONTEXT context) {		// 0F 89
	DWORD	offset = FETCH_OFFSET32(context);
	if (!test_cond_s(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_jp_o32(LPCONTEXT context) {		// 0F 8A
	DWORD	offset = FETCH_OFFSET32(context);
	if (test_cond_p(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_jnp_o32(LPCONTEXT context) {		// 0F 8B
	DWORD	offset = FETCH_OFFSET32(context);
	if (!test_cond_p(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_jl_o32(LPCONTEXT context) {		// 0F 8C
	DWORD	offset = FETCH_OFFSET32(context);
	if (test_cond_l(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_jge_o32(LPCONTEXT context) {		// 0F 8D
	DWORD	offset = FETCH_OFFSET32(context);
	if (!test_cond_l(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_jle_o32(LPCONTEXT context) {		// 0F 8E
	DWORD	offset = FETCH_OFFSET32(context);
	if (test_cond_le(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_jg_o32(LPCONTEXT context) {		// 0F 8F
	DWORD	offset = FETCH_OFFSET32(context);
	if (!test_cond_le(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_0fpre(LPCONTEXT context) {		// 0F
	BOOL	res = FALSE;
	OPCODE	ModRM = FETCH_OPCODE(context);
	switch (ModRM.byte) {
	//...
	case 0x01: res = i_0f01pre(context); break;
	//...
	case 0x21: res = i_mov_r32_rdr(context); break;
	//...
	case 0x23: res = i_mov_rdr_r32(context); break;
	//...
	case 0x80: res = i_jo_o32(context); break;
	case 0x81: res = i_jno_o32(context); break;
	case 0x82: res = i_jb_o32(context); break;
	case 0x83: res = i_jae_o32(context); break;
	case 0x84: res = i_je_o32(context); break;
	case 0x85: res = i_jne_o32(context); break;
	case 0x86: res = i_jbe_o32(context); break;
	case 0x87: res = i_ja_o32(context); break;
	case 0x88: res = i_js_o32(context); break;
	case 0x89: res = i_jns_o32(context); break;
	case 0x8a: res = i_jp_o32(context); break;
	case 0x8b: res = i_jnp_o32(context); break;
	case 0x8c: res = i_jl_o32(context); break;
	case 0x8d: res = i_jge_o32(context); break;
	case 0x8e: res = i_jle_o32(context); break;
	case 0x8f: res = i_jg_o32(context); break;
	//...
	default: break;
	}
	return res;
}

static BOOL i_adc_b_r8(LPCONTEXT context) {		// 10
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = GET_REG8(context, (reg8)ModRM.reg_op);
	OPC_AL_DL_FN__TO_DEST_FLAGS(adc);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_adc_d_r32(LPCONTEXT context) {	// 11
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = GET_REG32(context, (reg32)ModRM.reg_op);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(adc);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_adc_r8_b(LPCONTEXT context) {		// 12
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GET_REG8(context, (reg8)ModRM.reg_op);
	BYTE	src = GetModRMRM8(context, ModRM);
	OPC_AL_DL_FN__TO_DEST_FLAGS(adc);
	SET_REG8(context, (reg8)ModRM.reg_op, dest);
	return TRUE;
}

static BOOL i_adc_r32_d(LPCONTEXT context) {	// 13
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GET_REG32(context, (reg32)ModRM.reg_op);
	DWORD	src = GetModRMRM32(context, ModRM);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(adc);
	SET_REG32(context, (reg32)ModRM.reg_op, dest);
	return TRUE;
}

static BOOL i_adc_al_i8(LPCONTEXT context) {	// 14
	BYTE	dest = GET_REG8(context, AL);
	BYTE	src = FETCH_BYTE(context);
	OPC_AL_DL_FN__TO_DEST_FLAGS(adc);
	SET_REG8(context, AL, dest);
	return TRUE;
}

static BOOL i_adc_eax_i32(LPCONTEXT context) {	// 15
	DWORD	dest = GET_REG32(context, EAX);
	DWORD	src = FETCH_DWORD(context);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(adc);
	SET_REG32(context, EAX, dest);
	return TRUE;
}

static BOOL i_push_ss(LPCONTEXT context) {		// 16
	DWORD	dest = GET_SEG(context, SS);
	PUSH_DWORD(context, dest);
	return TRUE;
}

static BOOL i_pop_ss(LPCONTEXT context) {		// 17
	DWORD	dest = POP_DWORD(context);
	SET_SEG(context, SS, dest);
	return TRUE;
}

static BOOL i_sbb_b_r8(LPCONTEXT context) {		// 18
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = GET_REG8(context, (reg8)ModRM.reg_op);
	OPC_AL_DL_FN__TO_DEST_FLAGS(sbb);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_sbb_d_r32(LPCONTEXT context) {	// 19
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = GET_REG32(context, (reg32)ModRM.reg_op);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(sbb);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_sbb_r8_b(LPCONTEXT context) {		// 1A
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GET_REG8(context, (reg8)ModRM.reg_op);
	BYTE	src = GetModRMRM8(context, ModRM);
	OPC_AL_DL_FN__TO_DEST_FLAGS(sbb);
	SET_REG8(context, (reg8)ModRM.reg_op, dest);
	return TRUE;
}

static BOOL i_sbb_r32_d(LPCONTEXT context) {	// 1B
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GET_REG32(context, (reg32)ModRM.reg_op);
	DWORD	src = GetModRMRM32(context, ModRM);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(sbb);
	SET_REG32(context, (reg32)ModRM.reg_op, dest);
	return TRUE;
}

static BOOL i_sbb_al_i8(LPCONTEXT context) {	// 1C
	BYTE	dest = GET_REG8(context, AL);
	BYTE	src = FETCH_BYTE(context);
	OPC_AL_DL_FN__TO_DEST_FLAGS(sbb);
	SET_REG8(context, AL, dest);
	return TRUE;
}

static BOOL i_sbb_eax_i32(LPCONTEXT context) {	// 1D
	DWORD	dest = GET_REG32(context, EAX);
	DWORD	src = FETCH_DWORD(context);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(sbb);
	SET_REG32(context, EAX, dest);
	return TRUE;
}

static BOOL i_push_ds(LPCONTEXT context) {		// 1E
	DWORD	dest = GET_SEG(context, DS);
	PUSH_DWORD(context, dest);
	return TRUE;
}

static BOOL i_pop_ds(LPCONTEXT context) {		// 1F
	DWORD	dest = POP_DWORD(context);
	SET_SEG(context, DS, dest);
	return TRUE;
}

static BOOL i_and_b_r8(LPCONTEXT context) {		// 20
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = GET_REG8(context, (reg8)ModRM.reg_op);
	OPC_AL_DL_FN__TO_DEST_FLAGS(and);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_and_d_r32(LPCONTEXT context) {	// 21
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = GET_REG32(context, (reg32)ModRM.reg_op);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(and);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_and_r8_b(LPCONTEXT context) {		// 22
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GET_REG8(context, (reg8)ModRM.reg_op);
	BYTE	src = GetModRMRM8(context, ModRM);
	OPC_AL_DL_FN__TO_DEST_FLAGS(and);
	SET_REG8(context, (reg8)ModRM.reg_op, dest);
	return TRUE;
}

static BOOL i_and_r32_d(LPCONTEXT context) {	// 23
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GET_REG32(context, (reg32)ModRM.reg_op);
	DWORD	src = GetModRMRM32(context, ModRM);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(and);
	SET_REG32(context, (reg32)ModRM.reg_op, dest);
	return TRUE;
}

static BOOL i_and_al_i8(LPCONTEXT context) {	// 24
	BYTE	dest = GET_REG8(context, AL);
	BYTE	src = FETCH_BYTE(context);
	OPC_AL_DL_FN__TO_DEST_FLAGS(and);
	SET_REG8(context, AL, dest);
	return TRUE;
}

static BOOL i_and_eax_i32(LPCONTEXT context) {	// 25
	DWORD	dest = GET_REG32(context, EAX);
	DWORD	src = FETCH_DWORD(context);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(and);
	SET_REG32(context, EAX, dest);
	return TRUE;
}

static BOOL i_es(LPCONTEXT context) {			// 26
	SET_SEG_PREFIX(ES);
	return TRUE;
}

static BOOL i_daa(LPCONTEXT context) {			// 27
	DWORD	dest = GET_REG32(context, EAX);
	OPCEAX_FN__TO_DEST_FLAGS(daa);
	SET_REG32(context, EAX, dest);
	return TRUE;
}

static BOOL i_sub_b_r8(LPCONTEXT context) {		// 28
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = GET_REG8(context, (reg8)ModRM.reg_op);
	OPC_AL_DL_FN__TO_DEST_FLAGS(sub);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_sub_d_r32(LPCONTEXT context) {	// 29
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = GET_REG32(context, (reg32)ModRM.reg_op);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(sub);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_sub_r8_b(LPCONTEXT context) {		// 2A
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GET_REG8(context, (reg8)ModRM.reg_op);
	BYTE	src = GetModRMRM8(context, ModRM);
	OPC_AL_DL_FN__TO_DEST_FLAGS(sub);
	SET_REG8(context, (reg8)ModRM.reg_op, dest);
	return TRUE;
}

static BOOL i_sub_r32_d(LPCONTEXT context) {	// 2B
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GET_REG32(context, (reg32)ModRM.reg_op);
	DWORD	src = GetModRMRM32(context, ModRM);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(sub);
	SET_REG32(context, (reg32)ModRM.reg_op, dest);
	return TRUE;
}

static BOOL i_sub_al_i8(LPCONTEXT context) {	// 2C
	BYTE	dest = GET_REG8(context, AL);
	BYTE	src = FETCH_BYTE(context);
	OPC_AL_DL_FN__TO_DEST_FLAGS(sub);
	SET_REG8(context, AL, dest);
	return TRUE;
}

static BOOL i_sub_eax_i32(LPCONTEXT context) {	// 2D
	DWORD	dest = GET_REG32(context, EAX);
	DWORD	src = FETCH_DWORD(context);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(sub);
	SET_REG32(context, EAX, dest);
	return TRUE;
}

static BOOL i_cs(LPCONTEXT context) {			// 2E
	SET_SEG_PREFIX(CS);
	return TRUE;
}

static BOOL i_das(LPCONTEXT context) {			// 2F
	DWORD	dest = GET_REG32(context, EAX);
	OPCEAX_FN__TO_DEST_FLAGS(das);
	SET_REG32(context, EAX, dest);
	return TRUE;
}

static BOOL i_xor_b_r8(LPCONTEXT context) {		// 30
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = GET_REG8(context, (reg8)ModRM.reg_op);
	OPC_AL_DL_FN__TO_DEST_FLAGS(xor);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_xor_d_r32(LPCONTEXT context) {	// 31
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = GET_REG32(context, (reg32)ModRM.reg_op);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(xor);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_xor_r8_b(LPCONTEXT context) {		// 32
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GET_REG8(context, (reg8)ModRM.reg_op);
	BYTE	src = GetModRMRM8(context, ModRM);
	OPC_AL_DL_FN__TO_DEST_FLAGS(xor);
	SET_REG8(context, (reg8)ModRM.reg_op, dest);
	return TRUE;
}

static BOOL i_xor_r32_d(LPCONTEXT context) {	// 33
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GET_REG32(context, (reg32)ModRM.reg_op);
	DWORD	src = GetModRMRM32(context, ModRM);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(xor);
	SET_REG32(context, (reg32)ModRM.reg_op, dest);
	return TRUE;
}

static BOOL i_xor_al_i8(LPCONTEXT context) {	// 34
	BYTE	dest = GET_REG8(context, AL);
	BYTE	src = FETCH_BYTE(context);
	OPC_AL_DL_FN__TO_DEST_FLAGS(xor);
	SET_REG8(context, AL, dest);
	return TRUE;
}

static BOOL i_xor_eax_i32(LPCONTEXT context) {	// 35
	DWORD	dest = GET_REG32(context, EAX);
	DWORD	src = FETCH_DWORD(context);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(xor);
	SET_REG32(context, EAX, dest);
	return TRUE;
}

static BOOL i_ss(LPCONTEXT context) {			// 36
	SET_SEG_PREFIX(SS);
	return TRUE;
}

static BOOL i_aaa(LPCONTEXT context) {			// 37
	DWORD	dest = GET_REG32(context, EAX);
	OPCEAX_FN__TO_DEST_FLAGS(aaa);
	SET_REG32(context, EAX, dest);
	return TRUE;
}

static BOOL i_cmp_b_r8(LPCONTEXT context) {		// 38
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = GET_REG8(context, (reg8)ModRM.reg_op);
	OPC_AL_DL_FN__TO_FLAGS(cmp);
	return TRUE;
}

static BOOL i_cmp_d_r32(LPCONTEXT context) {	// 39
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = GET_REG32(context, (reg32)ModRM.reg_op);
	OPC_EAX_EDX_FN__TO_FLAGS(cmp);
	return TRUE;
}

static BOOL i_cmp_r8_b(LPCONTEXT context) {		// 3A
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GET_REG8(context, (reg8)ModRM.reg_op);
	BYTE	src = GetModRMRM8(context, ModRM);
	OPC_AL_DL_FN__TO_FLAGS(cmp);
	return TRUE;
}

static BOOL i_cmp_r32_d(LPCONTEXT context) {	// 3B
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GET_REG32(context, (reg32)ModRM.reg_op);
	DWORD	src = GetModRMRM32(context, ModRM);
	OPC_EAX_EDX_FN__TO_FLAGS(cmp);
	return TRUE;
}

static BOOL i_cmp_al_i8(LPCONTEXT context) {	// 3C
	BYTE	dest = GET_REG8(context, AL);
	BYTE	src = FETCH_BYTE(context);
	OPC_AL_DL_FN__TO_FLAGS(cmp);
	return TRUE;
}

static BOOL i_cmp_eax_i32(LPCONTEXT context) {	// 3D
	DWORD	dest = GET_REG32(context, EAX);
	DWORD	src = FETCH_DWORD(context);
	OPC_EAX_EDX_FN__TO_FLAGS(cmp);
	return TRUE;
}

static BOOL i_ds(LPCONTEXT context) {			// 3E
	SET_SEG_PREFIX(DS);
	return TRUE;
}

static BOOL i_aas(LPCONTEXT context) {			// 3F
	DWORD	dest = GET_REG32(context, EAX);
	OPCEAX_FN__TO_DEST_FLAGS(aas);
	SET_REG32(context, EAX, dest);
	return TRUE;
}

static BOOL i_inc_eax(LPCONTEXT context) {		// 40
	DWORD	dest = GET_REG32(context, EAX);
	OPC_EAX_FN__TO_DEST_FLAGS(inc);
	SET_REG32(context, EAX, dest);
	return TRUE;
}

static BOOL i_inc_ecx(LPCONTEXT context) {		// 41
	DWORD	dest = GET_REG32(context, ECX);
	OPC_EAX_FN__TO_DEST_FLAGS(inc);
	SET_REG32(context, ECX, dest);
	return TRUE;
}

static BOOL i_inc_edx(LPCONTEXT context) {		// 42
	DWORD	dest = GET_REG32(context, EDX);
	OPC_EAX_FN__TO_DEST_FLAGS(inc);
	SET_REG32(context, EDX, dest);
	return TRUE;
}

static BOOL i_inc_ebx(LPCONTEXT context) {		// 43
	DWORD	dest = GET_REG32(context, EBX);
	OPC_EAX_FN__TO_DEST_FLAGS(inc);
	SET_REG32(context, EBX, dest);
	return TRUE;
}

static BOOL i_inc_esp(LPCONTEXT context) {		// 44
	DWORD	dest = GET_REG32(context, ESP);
	OPC_EAX_FN__TO_DEST_FLAGS(inc);
	SET_REG32(context, ESP, dest);
	return TRUE;
}

static BOOL i_inc_ebp(LPCONTEXT context) {		// 45
	DWORD	dest = GET_REG32(context, EBP);
	OPC_EAX_FN__TO_DEST_FLAGS(inc);
	SET_REG32(context, EBP, dest);
	return TRUE;
}

static BOOL i_inc_esi(LPCONTEXT context) {		// 46
	DWORD	dest = GET_REG32(context, ESI);
	OPC_EAX_FN__TO_DEST_FLAGS(inc);
	SET_REG32(context, ESI, dest);
	return TRUE;
}

static BOOL i_inc_edi(LPCONTEXT context) {		// 47
	DWORD	dest = GET_REG32(context, EDI);
	OPC_EAX_FN__TO_DEST_FLAGS(inc);
	SET_REG32(context, EDI, dest);
	return TRUE;
}

static BOOL i_dec_eax(LPCONTEXT context) {		// 48
	DWORD	dest = GET_REG32(context, EAX);
	OPC_EAX_FN__TO_DEST_FLAGS(dec);
	SET_REG32(context, EAX, dest);
	return TRUE;
}

static BOOL i_dec_ecx(LPCONTEXT context) {		// 49
	DWORD	dest = GET_REG32(context, ECX);
	OPC_EAX_FN__TO_DEST_FLAGS(dec);
	SET_REG32(context, ECX, dest);
	return TRUE;
}

static BOOL i_dec_edx(LPCONTEXT context) {		// 4A
	DWORD	dest = GET_REG32(context, EDX);
	OPC_EAX_FN__TO_DEST_FLAGS(dec);
	SET_REG32(context, EDX, dest);
	return TRUE;
}

static BOOL i_dec_ebx(LPCONTEXT context) {		// 4B
	DWORD	dest = GET_REG32(context, EBX);
	OPC_EAX_FN__TO_DEST_FLAGS(dec);
	SET_REG32(context, EBX, dest);
	return TRUE;
}

static BOOL i_dec_esp(LPCONTEXT context) {		// 4C
	DWORD	dest = GET_REG32(context, ESP);
	OPC_EAX_FN__TO_DEST_FLAGS(dec);
	SET_REG32(context, ESP, dest);
	return TRUE;
}

static BOOL i_dec_ebp(LPCONTEXT context) {		// 4D
	DWORD	dest = GET_REG32(context, EBP);
	OPC_EAX_FN__TO_DEST_FLAGS(dec);
	SET_REG32(context, EBP, dest);
	return TRUE;
}

static BOOL i_dec_esi(LPCONTEXT context) {		// 4E
	DWORD	dest = GET_REG32(context, ESI);
	OPC_EAX_FN__TO_DEST_FLAGS(dec);
	SET_REG32(context, ESI, dest);
	return TRUE;
}

static BOOL i_dec_edi(LPCONTEXT context) {		// 4F
	DWORD	dest = GET_REG32(context, EDI);
	OPC_EAX_FN__TO_DEST_FLAGS(dec);
	SET_REG32(context, EDI, dest);
	return TRUE;
}

static BOOL i_push_eax(LPCONTEXT context) {		// 50
	DWORD	dest = GET_REG32(context, EAX);
	PUSH_DWORD(context, dest);
	return TRUE;
}

static BOOL i_push_ecx(LPCONTEXT context) {		// 51
	DWORD	dest = GET_REG32(context, ECX);
	PUSH_DWORD(context, dest);
	return TRUE;
}

static BOOL i_push_edx(LPCONTEXT context) {		// 52
	DWORD	dest = GET_REG32(context, EDX);
	PUSH_DWORD(context, dest);
	return TRUE;
}

static BOOL i_push_ebx(LPCONTEXT context) {		// 53
	DWORD	dest = GET_REG32(context, EBX);
	PUSH_DWORD(context, dest);
	return TRUE;
}

static BOOL i_push_esp(LPCONTEXT context) {		// 54
	DWORD	dest = GET_REG32(context, ESP);
	PUSH_DWORD(context, dest);
	return TRUE;
}

static BOOL i_push_ebp(LPCONTEXT context) {		// 55
	DWORD	dest = GET_REG32(context, EBP);
	PUSH_DWORD(context, dest);
	return TRUE;
}

static BOOL i_push_esi(LPCONTEXT context) {		// 56
	DWORD	dest = GET_REG32(context, ESI);
	PUSH_DWORD(context, dest);
	return TRUE;
}

static BOOL i_push_edi(LPCONTEXT context) {		// 57
	DWORD	dest = GET_REG32(context, EDI);
	PUSH_DWORD(context, dest);
	return TRUE;
}

static BOOL i_pop_eax(LPCONTEXT context) {		// 58
	DWORD	dest = POP_DWORD(context);
	SET_REG32(context, EAX, dest);
	return TRUE;
}

static BOOL i_pop_ecx(LPCONTEXT context) {		// 59
	DWORD	dest = POP_DWORD(context);
	SET_REG32(context, ECX, dest);
	return TRUE;
}

static BOOL i_pop_edx(LPCONTEXT context) {		// 5A
	DWORD	dest = POP_DWORD(context);
	SET_REG32(context, EDX, dest);
	return TRUE;
}

static BOOL i_pop_ebx(LPCONTEXT context) {		// 5B
	DWORD	dest = POP_DWORD(context);
	SET_REG32(context, EBX, dest);
	return TRUE;
}

static BOOL i_pop_esp(LPCONTEXT context) {		// 5C
	DWORD	dest = POP_DWORD(context);
	SET_REG32(context, ESP, dest);
	return TRUE;
}

static BOOL i_pop_ebp(LPCONTEXT context) {		// 5D
	DWORD	dest = POP_DWORD(context);
	SET_REG32(context, EBP, dest);
	return TRUE;
}

static BOOL i_pop_esi(LPCONTEXT context) {		// 5E
	DWORD	dest = POP_DWORD(context);
	SET_REG32(context, ESI, dest);
	return TRUE;
}

static BOOL i_pop_edi(LPCONTEXT context) {		// 5F
	DWORD	dest = POP_DWORD(context);
	SET_REG32(context, EDI, dest);
	return TRUE;
}

static BOOL i_pushad(LPCONTEXT context) {		// 60
	DWORD	tmp_esp = GET_REG32(context, ESP);
	PUSH_DWORD(context, GET_REG32(context, EAX));
	PUSH_DWORD(context, GET_REG32(context, ECX));
	PUSH_DWORD(context, GET_REG32(context, EDX));
	PUSH_DWORD(context, GET_REG32(context, EBX));
	PUSH_DWORD(context, tmp_esp);
	PUSH_DWORD(context, GET_REG32(context, EBP));
	PUSH_DWORD(context, GET_REG32(context, ESI));
	PUSH_DWORD(context, GET_REG32(context, EDI));
	return TRUE;
}

static BOOL i_popad(LPCONTEXT context) {		// 61
	SET_REG32(context, EDI, POP_DWORD(context));
	SET_REG32(context, ESI, POP_DWORD(context));
	SET_REG32(context, EBP, POP_DWORD(context));
	POP_DWORD(context);
	SET_REG32(context, EBX, POP_DWORD(context));
	SET_REG32(context, EDX, POP_DWORD(context));
	SET_REG32(context, ECX, POP_DWORD(context));
	SET_REG32(context, EAX, POP_DWORD(context));
	return TRUE;
}

DECLARE_I_Invalid(bound)					// 62
DECLARE_I_Invalid(arpl)						// 63
DECLARE_I_Invalid(64invalid)				// 64
DECLARE_I_Invalid(65invalid)				// 65
DECLARE_I_Invalid(66invalid)				// 66
DECLARE_I_Invalid(67invalid)				// 67

static BOOL i_push_i32(LPCONTEXT context) {		// 68
	DWORD	dest = FETCH_DWORD(context);
	PUSH_DWORD(context, dest);
	return TRUE;
}

static BOOL i_imul_r32_d_i32(LPCONTEXT context) {// 69
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = (DWORD)FETCH_OFFSET32(context);	// signed
	OPCEAX_EDX_FN__TO_SRC_DEST_FLAGS(imul);
	SET_REG32(context, (reg32)ModRM.reg_op, dest);
	return TRUE;
}

static BOOL i_push_o8(LPCONTEXT context) {		// 6A
	DWORD	dest = FETCH_OFFSET8(context);
	PUSH_DWORD(context, dest);
	return TRUE;
}

static BOOL i_imul_r32_d_i8(LPCONTEXT context) {// 6B
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = (DWORD)FETCH_OFFSET8(context);	// signed
	OPCEAX_EDX_FN__TO_SRC_DEST_FLAGS(imul);
	SET_REG32(context, (reg32)ModRM.reg_op, dest);
	return TRUE;
}

DECLARE_I_Invalid(insb)						// 6C
DECLARE_I_Invalid(insd)						// 6D
DECLARE_I_Invalid(outsb)					// 6E
DECLARE_I_Invalid(outsd)					// 6F

static BOOL i_jo_o8(LPCONTEXT context) {		// 70
	DWORD	offset = FETCH_OFFSET8(context);
	if (test_cond_o(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_jno_o8(LPCONTEXT context) {		// 71
	DWORD	offset = FETCH_OFFSET8(context);
	if (!test_cond_o(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_jb_o8(LPCONTEXT context) {		// 72
	DWORD	offset = FETCH_OFFSET8(context);
	if (test_cond_b(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_jae_o8(LPCONTEXT context) {		// 73
	DWORD	offset = FETCH_OFFSET8(context);
	if (!test_cond_b(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_je_o8(LPCONTEXT context) {		// 74
	DWORD	offset = FETCH_OFFSET8(context);
	if (test_cond_e(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_jne_o8(LPCONTEXT context) {		// 75
	DWORD	offset = FETCH_OFFSET8(context);
	if (!test_cond_e(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_jbe_o8(LPCONTEXT context) {		// 76
	DWORD	offset = FETCH_OFFSET8(context);
	if (test_cond_be(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_ja_o8(LPCONTEXT context) {		// 77
	DWORD	offset = FETCH_OFFSET8(context);
	if (!test_cond_be(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_js_o8(LPCONTEXT context) {		// 78
	DWORD	offset = FETCH_OFFSET8(context);
	if (test_cond_s(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_jns_o8(LPCONTEXT context) {		// 79
	DWORD	offset = FETCH_OFFSET8(context);
	if (!test_cond_s(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_jp_o8(LPCONTEXT context) {		// 7A
	DWORD	offset = FETCH_OFFSET8(context);
	if (test_cond_p(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_jnp_o8(LPCONTEXT context) {		// 7B
	DWORD	offset = FETCH_OFFSET8(context);
	if (!test_cond_p(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_jl_o8(LPCONTEXT context) {		// 7C
	DWORD	offset = FETCH_OFFSET8(context);
	if (test_cond_l(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_jge_o8(LPCONTEXT context) {		// 7D
	DWORD	offset = FETCH_OFFSET8(context);
	if (!test_cond_l(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_jle_o8(LPCONTEXT context) {		// 7E
	DWORD	offset = FETCH_OFFSET8(context);
	if (test_cond_le(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_jg_o8(LPCONTEXT context) {		// 7F
	DWORD	offset = FETCH_OFFSET8(context);
	if (!test_cond_le(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_add_b_i8(LPCONTEXT context) {		// 80 ?0?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = FETCH_BYTE(context);
	OPC_AL_DL_FN__TO_DEST_FLAGS(add);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_or_b_i8(LPCONTEXT context) {		// 80 ?1?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = FETCH_BYTE(context);
	OPC_AL_DL_FN__TO_DEST_FLAGS(or);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_adc_b_i8(LPCONTEXT context) {		// 80 ?2?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = FETCH_BYTE(context);
	OPC_AL_DL_FN__TO_DEST_FLAGS(adc);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_sbb_b_i8(LPCONTEXT context) {		// 80 ?3?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = FETCH_BYTE(context);
	OPC_AL_DL_FN__TO_DEST_FLAGS(sbb);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_and_b_i8(LPCONTEXT context) {		// 80 ?4?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = FETCH_BYTE(context);
	OPC_AL_DL_FN__TO_DEST_FLAGS(and);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_sub_b_i8(LPCONTEXT context) {		// 80 ?5?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = FETCH_BYTE(context);
	OPC_AL_DL_FN__TO_DEST_FLAGS(sub);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_xor_b_i8(LPCONTEXT context) {		// 80 ?6?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = FETCH_BYTE(context);
	OPC_AL_DL_FN__TO_DEST_FLAGS(xor);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_cmp_b_i8(LPCONTEXT context) {		// 80 ?7?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = FETCH_BYTE(context);
	OPC_AL_DL_FN__TO_FLAGS(cmp);
	return TRUE;
}

static BOOL i_80pre(LPCONTEXT context) {		// 80
	BOOL	res = FALSE;
	OPCODE	ModRM = PEEK_OPCODE(context);
	switch (ModRM.reg_op) {
	case 0x00: res = i_add_b_i8(context); break;
	case 0x01: res = i_or_b_i8(context);  break;
	case 0x02: res = i_adc_b_i8(context); break;
	case 0x03: res = i_sbb_b_i8(context); break;
	case 0x04: res = i_and_b_i8(context); break;
	case 0x05: res = i_sub_b_i8(context); break;
	case 0x06: res = i_xor_b_i8(context); break;
	default:   res = i_cmp_b_i8(context); break;
	}
	return res;
}

static BOOL i_add_d_i32(LPCONTEXT context) {	// 81 ?0?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = FETCH_DWORD(context);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(add);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_or_d_i32(LPCONTEXT context) {		// 81 ?1?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = FETCH_DWORD(context);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(or);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_adc_d_i32(LPCONTEXT context) {	// 81 ?2?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = FETCH_DWORD(context);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(adc);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_sbb_d_i32(LPCONTEXT context) {	// 81 ?3?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = FETCH_DWORD(context);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(sbb);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_and_d_i32(LPCONTEXT context) {	// 81 ?4?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = FETCH_DWORD(context);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(and);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_sub_d_i32(LPCONTEXT context) {	// 81 ?5?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = FETCH_DWORD(context);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(sub);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_xor_d_i32(LPCONTEXT context) {	// 81 ?6?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = FETCH_DWORD(context);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(xor);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_cmp_d_i32(LPCONTEXT context) {	// 81 ?7?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = FETCH_DWORD(context);
	OPC_EAX_EDX_FN__TO_FLAGS(cmp);
	return TRUE;
}

static BOOL i_81pre(LPCONTEXT context) {		// 81
	BOOL	res = FALSE;
	OPCODE	ModRM = PEEK_OPCODE(context);
	switch (ModRM.reg_op) {
	case 0x00: res = i_add_d_i32(context); break;
	case 0x01: res = i_or_d_i32(context);  break;
	case 0x02: res = i_adc_d_i32(context); break;
	case 0x03: res = i_sbb_d_i32(context); break;
	case 0x04: res = i_and_d_i32(context); break;
	case 0x05: res = i_sub_d_i32(context); break;
	case 0x06: res = i_xor_d_i32(context); break;
	default:   res = i_cmp_d_i32(context); break;
	}
	return res;
}

DECLARE_I_Invalid(82invalid)				// 82

static BOOL i_add_d_o8(LPCONTEXT context) {		// 83 ?0?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = FETCH_OFFSET8(context);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(add);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_or_d_o8(LPCONTEXT context) {		// 83 ?1?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = FETCH_OFFSET8(context);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(or);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_adc_d_o8(LPCONTEXT context) {		// 83 ?2?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = FETCH_OFFSET8(context);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(adc);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_sbb_d_o8(LPCONTEXT context) {		// 83 ?3?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = FETCH_OFFSET8(context);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(sbb);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_and_d_o8(LPCONTEXT context) {		// 83 ?4?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = FETCH_OFFSET8(context);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(and);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_sub_d_o8(LPCONTEXT context) {		// 83 ?5?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = FETCH_OFFSET8(context);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(sub);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_xor_d_o8(LPCONTEXT context) {		// 83 ?6?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = FETCH_OFFSET8(context);
	OPC_EAX_EDX_FN__TO_DEST_FLAGS(xor);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_cmp_d_o8(LPCONTEXT context) {		// 83 ?7?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = FETCH_OFFSET8(context);
	OPC_EAX_EDX_FN__TO_FLAGS(cmp);
	return TRUE;
}

static BOOL i_83pre(LPCONTEXT context) {		// 83
	BOOL	res = FALSE;
	OPCODE	ModRM = PEEK_OPCODE(context);
	switch (ModRM.reg_op) {
	case 0x00: res = i_add_d_o8(context); break;
	case 0x01: res = i_or_d_o8(context);  break;
	case 0x02: res = i_adc_d_o8(context); break;
	case 0x03: res = i_sbb_d_o8(context); break;
	case 0x04: res = i_and_d_o8(context); break;
	case 0x05: res = i_sub_d_o8(context); break;
	case 0x06: res = i_xor_d_o8(context); break;
	default:   res = i_cmp_d_o8(context); break;
	}
	return res;
}

static BOOL i_test_b_r8(LPCONTEXT context) {	// 84
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = GET_REG8(context, (reg8)ModRM.reg_op);
	OPC_AL_DL_FN__TO_FLAGS(test);
	return TRUE;
}

static BOOL i_test_d_r32(LPCONTEXT context) {	// 85
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = GET_REG32(context, (reg32)ModRM.reg_op);
	OPC_EAX_EDX_FN__TO_FLAGS(test);
	return TRUE;
}

static BOOL i_xchg_b_r8(LPCONTEXT context) {	// 86
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = GET_REG8(context, (reg8)ModRM.reg_op);
	PutbackModRMRM8(context, ModRM, src);
	SET_REG8(context, (reg8)ModRM.reg_op, dest);
	return TRUE;
}

static BOOL i_xchg_d_r32(LPCONTEXT context) {	// 87
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = GET_REG32(context, (reg32)ModRM.reg_op);
	PutbackModRMRM32(context, ModRM, src);
	SET_REG32(context, (reg32)ModRM.reg_op, dest);
	return TRUE;
}

static BOOL i_mov_b_r8(LPCONTEXT context) {		// 88
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	src = GET_REG8(context, (reg8)ModRM.reg_op);
	SetModRMRM8(context, ModRM, src);
	return TRUE;
}

static BOOL i_mov_d_r32(LPCONTEXT context) {	// 89
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	src = GET_REG32(context, (reg32)ModRM.reg_op);
	SetModRMRM32(context, ModRM, src);
	return TRUE;
}

static BOOL i_mov_r8_b(LPCONTEXT context) {		// 8A
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	src = GetModRMRM8(context, ModRM);
	SET_REG8(context, (reg8)ModRM.reg_op, src);
	return TRUE;
}

static BOOL i_mov_r32_d(LPCONTEXT context) {	// 8B
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	src = GetModRMRM32(context, ModRM);
	SET_REG32(context, (reg32)ModRM.reg_op, src);
	return TRUE;
}

static BOOL i_mov_w_seg(LPCONTEXT context) {	// 8C
	OPCODE	ModRM = FETCH_OPCODE(context);
	if ((sreg)ModRM.reg_op > GS)
		return FALSE;
	WORD	src = (WORD)GET_SEG(context, (sreg)ModRM.reg_op);
	SetModRMRM16(context, ModRM, src);
	return TRUE;
}

static BOOL i_lea_r32_d(LPCONTEXT context) {	// 8D
	OPCODE	ModRM = FETCH_OPCODE(context);
	if (ModRM.adr_op == 3)	// 'lea reg,reg' is invalid
		return FALSE;
	ModRMGetEA(context, ModRM);
	SET_REG32(context, (reg32)ModRM.reg_op, ModRM_EA.addr);
	return TRUE;
}

static BOOL i_mov_seg_w(LPCONTEXT context) {	// 8E
	OPCODE	ModRM = FETCH_OPCODE(context);
	if ((sreg)ModRM.reg_op > GS)
		return FALSE;
	WORD	src = GetModRMRM16(context, ModRM);
	SET_SEG(context, (sreg)ModRM.reg_op, (DWORD)src);
	return TRUE;
}

static BOOL i_pop_d(LPCONTEXT context) {		// 8F
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = POP_DWORD(context);
	SetModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_nop(LPCONTEXT context) {			// 90
	return TRUE;
}

static BOOL i_xchg_eax_ecx(LPCONTEXT context) {	// 91
	DWORD	dest = GET_REG32(context, EAX);
	DWORD	src = GET_REG32(context, ECX);
	SET_REG32(context, ECX, dest);
	SET_REG32(context, EAX, src);
	return TRUE;
}

static BOOL i_xchg_eax_edx(LPCONTEXT context) {	// 92
	DWORD	dest = GET_REG32(context, EAX);
	DWORD	src = GET_REG32(context, EDX);
	SET_REG32(context, EDX, dest);
	SET_REG32(context, EAX, src);
	return TRUE;
}

static BOOL i_xchg_eax_ebx(LPCONTEXT context) {	// 93
	DWORD	dest = GET_REG32(context, EAX);
	DWORD	src = GET_REG32(context, EBX);
	SET_REG32(context, EBX, dest);
	SET_REG32(context, EAX, src);
	return TRUE;
}

static BOOL i_xchg_eax_esp(LPCONTEXT context) {	// 94
	DWORD	dest = GET_REG32(context, EAX);
	DWORD	src = GET_REG32(context, ESP);
	SET_REG32(context, ESP, dest);
	SET_REG32(context, EAX, src);
	return TRUE;
}

static BOOL i_xchg_eax_ebp(LPCONTEXT context) {	// 95
	DWORD	dest = GET_REG32(context, EAX);
	DWORD	src = GET_REG32(context, EBP);
	SET_REG32(context, EBP, dest);
	SET_REG32(context, EAX, src);
	return TRUE;
}

static BOOL i_xchg_eax_esi(LPCONTEXT context) {	// 96
	DWORD	dest = GET_REG32(context, EAX);
	DWORD	src = GET_REG32(context, ESI);
	SET_REG32(context, ESI, dest);
	SET_REG32(context, EAX, src);
	return TRUE;
}

static BOOL i_xchg_eax_edi(LPCONTEXT context) {	// 97
	DWORD	dest = GET_REG32(context, EAX);
	DWORD	src = GET_REG32(context, EDI);
	SET_REG32(context, EDI, dest);
	SET_REG32(context, EAX, src);
	return TRUE;
}

static BOOL i_cwde(LPCONTEXT context) {			// 98
	DWORD	dest = GET_REG32(context, EAX);
	OPCEAX_FN__TO_DEST_FLAGS(cwde);
	SET_REG32(context, EAX, dest);
	return TRUE;
}

static BOOL i_cdq(LPCONTEXT context) {			// 99
	DWORD	dest = GET_REG32(context, EAX);
	DWORD	src;// = GET_REG32(context, EDX);
	OPCEAXEDX_FN__TO_SRC_DEST_FLAGS(cdq);
	SET_REG32(context, EAX, dest);
	SET_REG32(context, EDX, src);
	return TRUE;
}

DECLARE_I_Invalid(call_far)					// 9A
DECLARE_I_Invalid(wait)						// 9B

static BOOL i_pushfd(LPCONTEXT context) {		// 9C
	DWORD	dest = GET_FLAGS(context);
	PUSH_DWORD(context, dest);
	return TRUE;
}

static BOOL i_popfd(LPCONTEXT context) {		// 9D
	DWORD	dest = POP_DWORD(context);
	SET_FLAGS(context, dest);
	return TRUE;
}

static BOOL i_sahf(LPCONTEXT context) {			// 9E
	DWORD	dest = GET_REG32(context, EAX);
	OPCEAX_FN__TO_DEST_FLAGS(sahf);
	return TRUE;
}

static BOOL i_lahf(LPCONTEXT context) {			// 9F
	DWORD	dest = GET_REG32(context, EAX);
	OPCEAX_FN__TO_DEST_FLAGS(lahf);
	SET_REG32(context, EAX, dest);
	return TRUE;
}

static BOOL i_mov_al_m32(LPCONTEXT context) {	// A0
	EA_ADDR	EA_src(FETCH_DWORD(context), DS);
	BYTE	dest = READ_BYTE(context, EA_src);
	SET_REG8(context, AL, dest);
	return TRUE;
}

static BOOL i_mov_eax_m32(LPCONTEXT context) {	// A1
	EA_ADDR	EA_src(FETCH_DWORD(context), DS);
	DWORD	dest = READ_DWORD(context, EA_src);
	SET_REG32(context, EAX, dest);
	return TRUE;
}

static BOOL i_mov_m32_al(LPCONTEXT context) {	// A2
	EA_ADDR	EA_dest(FETCH_DWORD(context), DS);
	BYTE	src = GET_REG8(context, AL);
	WRITE_BYTE(context, EA_dest, src);
	return TRUE;
}

static BOOL i_mov_m32_eax(LPCONTEXT context) {	// A3
	EA_ADDR	EA_dest(FETCH_DWORD(context), DS);
	DWORD	src = GET_REG32(context, EAX);
	WRITE_DWORD(context, EA_dest, src);
	return TRUE;
}

static BOOL i_movsb(LPCONTEXT context) {		// A4
	EA_ADDR	EA_dest(GET_REG32(context, EDI), ES);
	EA_ADDR	EA_src(GET_REG32(context, ESI), DS);
	BYTE	src = READ_BYTE(context, EA_src);
	WRITE_BYTE(context, EA_dest, src);
	LONG	offset = (GET_FLAGS(context) & DF) ? -1 : 1;
	SET_REG32(context, EDI, GET_REG32(context, EDI) + offset);
	SET_REG32(context, ESI, GET_REG32(context, ESI) + offset);
	return TRUE;
}

static BOOL i_movsd(LPCONTEXT context) {		// A5
	EA_ADDR	EA_dest(GET_REG32(context, EDI), ES);
	EA_ADDR	EA_src(GET_REG32(context, ESI), DS);
	DWORD	src = READ_DWORD(context, EA_src);
	WRITE_DWORD(context, EA_dest, src);
	LONG	offset = (GET_FLAGS(context) & DF) ? -4 : 4;
	SET_REG32(context, EDI, GET_REG32(context, EDI) + offset);
	SET_REG32(context, ESI, GET_REG32(context, ESI) + offset);
	return TRUE;
}

DECLARE_I_Invalid(cmpsb)					// A6
DECLARE_I_Invalid(cmpsd)					// A7

static BOOL i_test_al_i8(LPCONTEXT context) {	// A8
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GET_REG8(context, AL);
	BYTE	src = FETCH_BYTE(context);
	OPC_AL_DL_FN__TO_FLAGS(test);
	return TRUE;
}

static BOOL i_test_eax_i32(LPCONTEXT context) {	// A9
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GET_REG32(context, EAX);
	DWORD	src = FETCH_DWORD(context);
	OPC_EAX_EDX_FN__TO_FLAGS(test);
	return TRUE;
}

static BOOL i_stosb(LPCONTEXT context) {		// AA
	EA_ADDR	EA_dest(GET_REG32(context, EDI), ES);
	BYTE	src = GET_REG8(context, AL);
	WRITE_BYTE(context, EA_dest, src);
	LONG	offset = (GET_FLAGS(context) & DF) ? -1 : 1;
	SET_REG32(context, EDI, GET_REG32(context, EDI) + offset);
	return TRUE;
}

static BOOL i_stosd(LPCONTEXT context) {		// AB
	EA_ADDR	EA_dest(GET_REG32(context, EDI), ES);
	DWORD	src = GET_REG32(context, EAX);
	WRITE_DWORD(context, EA_dest, src);
	LONG	offset = (GET_FLAGS(context) & DF) ? -4 : 4;
	SET_REG32(context, EDI, GET_REG32(context, EDI) + offset);
	return TRUE;
}

static BOOL i_lodsb(LPCONTEXT context) {		// AC
	EA_ADDR	EA_src(GET_REG32(context, ESI), DS);
	BYTE	src = READ_BYTE(context, EA_src);
	SET_REG8(context, AL, src);
	LONG	offset = (GET_FLAGS(context) & DF) ? -1 : 1;
	SET_REG32(context, ESI, GET_REG32(context, ESI) + offset);
	return TRUE;
}

static BOOL i_lodsd(LPCONTEXT context) {		// AD
	EA_ADDR	EA_src(GET_REG32(context, ESI), DS);
	DWORD	src = READ_DWORD(context, EA_src);
	SET_REG32(context, EAX, src);
	LONG	offset = (GET_FLAGS(context) & DF) ? -4 : 4;
	SET_REG32(context, ESI, GET_REG32(context, ESI) + offset);
	return TRUE;
}

DECLARE_I_Invalid(scasb)					// AE
DECLARE_I_Invalid(scasd)					// AF

static BOOL i_mov_al_i8(LPCONTEXT context) {	// B0
	BYTE	src = FETCH_BYTE(context);
	SET_REG8(context, AL, src);
	return TRUE;
}

static BOOL i_mov_cl_i8(LPCONTEXT context) {	// B1
	BYTE	src = FETCH_BYTE(context);
	SET_REG8(context, CL, src);
	return TRUE;
}

static BOOL i_mov_dl_i8(LPCONTEXT context) {	// B2
	BYTE	src = FETCH_BYTE(context);
	SET_REG8(context, DL, src);
	return TRUE;
}

static BOOL i_mov_bl_i8(LPCONTEXT context) {	// B3
	BYTE	src = FETCH_BYTE(context);
	SET_REG8(context, BL, src);
	return TRUE;
}

static BOOL i_mov_ah_i8(LPCONTEXT context) {	// B4
	BYTE	src = FETCH_BYTE(context);
	SET_REG8(context, AH, src);
	return TRUE;
}

static BOOL i_mov_ch_i8(LPCONTEXT context) {	// B5
	BYTE	src = FETCH_BYTE(context);
	SET_REG8(context, CH, src);
	return TRUE;
}

static BOOL i_mov_dh_i8(LPCONTEXT context) {	// B6
	BYTE	src = FETCH_BYTE(context);
	SET_REG8(context, DH, src);
	return TRUE;
}

static BOOL i_mov_bh_i8(LPCONTEXT context) {	// B7
	BYTE	src = FETCH_BYTE(context);
	SET_REG8(context, BH, src);
	return TRUE;
}

static BOOL i_mov_eax_i32(LPCONTEXT context) {	// B8
	DWORD	src = FETCH_DWORD(context);
	SET_REG32(context, EAX, src);
	return TRUE;
}

static BOOL i_mov_ecx_i32(LPCONTEXT context) {	// B9
	DWORD	src = FETCH_DWORD(context);
	SET_REG32(context, ECX, src);
	return TRUE;
}

static BOOL i_mov_edx_i32(LPCONTEXT context) {	// BA
	DWORD	src = FETCH_DWORD(context);
	SET_REG32(context, EDX, src);
	return TRUE;
}

static BOOL i_mov_ebx_i32(LPCONTEXT context) {	// BB
	DWORD	src = FETCH_DWORD(context);
	SET_REG32(context, EBX, src);
	return TRUE;
}

static BOOL i_mov_esp_i32(LPCONTEXT context) {	// BC
	DWORD	src = FETCH_DWORD(context);
	SET_REG32(context, ESP, src);
	return TRUE;
}

static BOOL i_mov_ebp_i32(LPCONTEXT context) {	// BD
	DWORD	src = FETCH_DWORD(context);
	SET_REG32(context, EBP, src);
	return TRUE;
}

static BOOL i_mov_esi_i32(LPCONTEXT context) {	// BE
	DWORD	src = FETCH_DWORD(context);
	SET_REG32(context, ESI, src);
	return TRUE;
}

static BOOL i_mov_edi_i32(LPCONTEXT context) {	// BF
	DWORD	src = FETCH_DWORD(context);
	SET_REG32(context, EDI, src);
	return TRUE;
}

static BOOL i_rol_b_i8(LPCONTEXT context) {		// C0 ?0?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = FETCH_BYTE(context);
	OPC_AL_CL_FN__TO_DEST_FLAGS(rol);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_ror_b_i8(LPCONTEXT context) {		// C0 ?1?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = FETCH_BYTE(context);
	OPC_AL_CL_FN__TO_DEST_FLAGS(ror);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_rcl_b_i8(LPCONTEXT context) {		// C0 ?2?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = FETCH_BYTE(context);
	OPC_AL_CL_FN__TO_DEST_FLAGS(rcl);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_rcr_b_i8(LPCONTEXT context) {		// C0 ?3?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = FETCH_BYTE(context);
	OPC_AL_CL_FN__TO_DEST_FLAGS(rcr);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_shl_b_i8(LPCONTEXT context) {		// C0 ?4?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = FETCH_BYTE(context);
	OPC_AL_CL_FN__TO_DEST_FLAGS(shl);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_shr_b_i8(LPCONTEXT context) {		// C0 ?5?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = FETCH_BYTE(context);
	OPC_AL_CL_FN__TO_DEST_FLAGS(shr);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_sal_b_i8(LPCONTEXT context) {		// C0 ?6?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = FETCH_BYTE(context);
	OPC_AL_CL_FN__TO_DEST_FLAGS(sal);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_sar_b_i8(LPCONTEXT context) {		// C0 ?7?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = FETCH_BYTE(context);
	OPC_AL_CL_FN__TO_DEST_FLAGS(sar);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_c0pre(LPCONTEXT context) {		// C0
	BOOL	res = FALSE;
	OPCODE	ModRM = PEEK_OPCODE(context);
	switch (ModRM.reg_op) {
	case 0x00: res = i_rol_b_i8(context); break;
	case 0x01: res = i_ror_b_i8(context); break;
	case 0x02: res = i_rcl_b_i8(context); break;
	case 0x03: res = i_rcr_b_i8(context); break;
	case 0x04: res = i_shl_b_i8(context); break;
	case 0x05: res = i_shr_b_i8(context); break;
	case 0x06: res = i_sal_b_i8(context); break;
	default:   res = i_sar_b_i8(context); break;
	}
	return res;
}

static BOOL i_rol_d_i8(LPCONTEXT context) {		// C1 ?0?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	BYTE	src = FETCH_BYTE(context);
	OPC_EAX_CL_FN__TO_DEST_FLAGS(rol);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_ror_d_i8(LPCONTEXT context) {		// C1 ?1?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	BYTE	src = FETCH_BYTE(context);
	OPC_EAX_CL_FN__TO_DEST_FLAGS(ror);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_rcl_d_i8(LPCONTEXT context) {		// C1 ?2?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	BYTE	src = FETCH_BYTE(context);
	OPC_EAX_CL_FN__TO_DEST_FLAGS(rcl);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_rcr_d_i8(LPCONTEXT context) {		// C1 ?3?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	BYTE	src = FETCH_BYTE(context);
	OPC_EAX_CL_FN__TO_DEST_FLAGS(rcr);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_shl_d_i8(LPCONTEXT context) {		// C1 ?4?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	BYTE	src = FETCH_BYTE(context);
	OPC_EAX_CL_FN__TO_DEST_FLAGS(shl);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_shr_d_i8(LPCONTEXT context) {		// C1 ?5?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	BYTE	src = FETCH_BYTE(context);
	OPC_EAX_CL_FN__TO_DEST_FLAGS(shr);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_sal_d_i8(LPCONTEXT context) {		// C1 ?6?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	BYTE	src = FETCH_BYTE(context);
	OPC_EAX_CL_FN__TO_DEST_FLAGS(sal);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_sar_d_i8(LPCONTEXT context) {		// C1 ?7?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	BYTE	src = FETCH_BYTE(context);
	OPC_EAX_CL_FN__TO_DEST_FLAGS(sar);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_c1pre(LPCONTEXT context) {		// C1
	BOOL	res = FALSE;
	OPCODE	ModRM = PEEK_OPCODE(context);
	switch (ModRM.reg_op) {
	case 0x00: res = i_rol_d_i8(context); break;
	case 0x01: res = i_ror_d_i8(context); break;
	case 0x02: res = i_rcl_d_i8(context); break;
	case 0x03: res = i_rcr_d_i8(context); break;
	case 0x04: res = i_shl_d_i8(context); break;
	case 0x05: res = i_shr_d_i8(context); break;
	case 0x06: res = i_sal_d_i8(context); break;
	default:   res = i_sar_d_i8(context); break;
	}
	return res;
}

static BOOL i_ret_i16(LPCONTEXT context) {	// C2
	WORD	offset = FETCH_WORD(context);
	DWORD	dest = POP_DWORD(context);
	SET_REG32(context, ESP, GET_REG32(context, ESP) + offset);
	context->Eip = dest;
	return TRUE;
}

static BOOL i_ret(LPCONTEXT context) {		// C3
	DWORD	dest = POP_DWORD(context);
	context->Eip = dest;
	return TRUE;
}

DECLARE_I_Invalid(les_fw)					// C4
DECLARE_I_Invalid(lds_fw)					// C5

static BOOL i_mov_b_i8(LPCONTEXT context) {		// C6
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = FETCH_BYTE(context);
	PutbackModRMRM8(context, ModRM, src);
	return TRUE;
}

static BOOL i_mov_d_i32(LPCONTEXT context) {	// C7
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = FETCH_DWORD(context);
	PutbackModRMRM32(context, ModRM, src);
	return TRUE;
}

DECLARE_I_Invalid(enter_i16i8)				// C8
DECLARE_I_Invalid(leave)					// C9
/*static BOOL i_leave(LPCONTEXT context) {		// C9
	SET_REG32(context, ESP, GET_REG32(context, EBP));
	SET_REG32(context, EBP, POP_DWORD(context));
	return TRUE;
}*/
DECLARE_I_Invalid(retf_d16)					// CA

static BOOL i_retf(LPCONTEXT context) {		// CB
	context->Eip = POP_DWORD(context);
	POP_DWORD(context);		// CS
	DWORD tmp_esp = POP_DWORD(context);
	POP_DWORD(context);		// SS
	SET_REG32(context, ESP, tmp_esp);
	return TRUE;
}

DECLARE_I_Invalid(int3)						// CC
DECLARE_I_Invalid(int_i8)					// CD
DECLARE_I_Invalid(into)						// CE
DECLARE_I_Invalid(iretd)					// CF

static BOOL i_rol_b_1(LPCONTEXT context) {		// D0 ?0?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = 1;
	OPC_AL_CL_FN__TO_DEST_FLAGS(rol);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_ror_b_1(LPCONTEXT context) {		// D0 ?1?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = 1;
	OPC_AL_CL_FN__TO_DEST_FLAGS(ror);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_rcl_b_1(LPCONTEXT context) {		// D0 ?2?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = 1;
	OPC_AL_CL_FN__TO_DEST_FLAGS(rcl);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_rcr_b_1(LPCONTEXT context) {		// D0 ?3?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = 1;
	OPC_AL_CL_FN__TO_DEST_FLAGS(rcr);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_shl_b_1(LPCONTEXT context) {		// D0 ?4?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = 1;
	OPC_AL_CL_FN__TO_DEST_FLAGS(shl);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_shr_b_1(LPCONTEXT context) {		// D0 ?5?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = 1;
	OPC_AL_CL_FN__TO_DEST_FLAGS(shr);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_sal_b_1(LPCONTEXT context) {		// D0 ?6?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = 1;
	OPC_AL_CL_FN__TO_DEST_FLAGS(sal);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_sar_b_1(LPCONTEXT context) {		// D0 ?7?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = 1;
	OPC_AL_CL_FN__TO_DEST_FLAGS(sar);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_d0pre(LPCONTEXT context) {		// D0
	BOOL	res = FALSE;
	OPCODE	ModRM = PEEK_OPCODE(context);
	switch (ModRM.reg_op) {
	case 0x00: res = i_rol_b_1(context); break;
	case 0x01: res = i_ror_b_1(context); break;
	case 0x02: res = i_rcl_b_1(context); break;
	case 0x03: res = i_rcr_b_1(context); break;
	case 0x04: res = i_shl_b_1(context); break;
	case 0x05: res = i_shr_b_1(context); break;
	case 0x06: res = i_sal_b_1(context); break;
	default:   res = i_sar_b_1(context); break;
	}
	return res;
}

static BOOL i_rol_d_1(LPCONTEXT context) {		// D1 ?0?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	BYTE	src = 1;
	OPC_EAX_CL_FN__TO_DEST_FLAGS(rol);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_ror_d_1(LPCONTEXT context) {		// D1 ?1?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	BYTE	src = 1;
	OPC_EAX_CL_FN__TO_DEST_FLAGS(ror);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_rcl_d_1(LPCONTEXT context) {		// D1 ?2?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	BYTE	src = 1;
	OPC_EAX_CL_FN__TO_DEST_FLAGS(rcl);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_rcr_d_1(LPCONTEXT context) {		// D1 ?3?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	BYTE	src = 1;
	OPC_EAX_CL_FN__TO_DEST_FLAGS(rcr);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_shl_d_1(LPCONTEXT context) {		// D1 ?4?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	BYTE	src = 1;
	OPC_EAX_CL_FN__TO_DEST_FLAGS(shl);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_shr_d_1(LPCONTEXT context) {		// D1 ?5?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	BYTE	src = 1;
	OPC_EAX_CL_FN__TO_DEST_FLAGS(shr);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_sal_d_1(LPCONTEXT context) {		// D1 ?6?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	BYTE	src = 1;
	OPC_EAX_CL_FN__TO_DEST_FLAGS(sal);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_sar_d_1(LPCONTEXT context) {		// D1 ?7?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	BYTE	src = 1;
	OPC_EAX_CL_FN__TO_DEST_FLAGS(sar);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_d1pre(LPCONTEXT context) {		// D1
	BOOL	res = FALSE;
	OPCODE	ModRM = PEEK_OPCODE(context);
	switch (ModRM.reg_op) {
	case 0x00: res = i_rol_d_1(context); break;
	case 0x01: res = i_ror_d_1(context); break;
	case 0x02: res = i_rcl_d_1(context); break;
	case 0x03: res = i_rcr_d_1(context); break;
	case 0x04: res = i_shl_d_1(context); break;
	case 0x05: res = i_shr_d_1(context); break;
	case 0x06: res = i_sal_d_1(context); break;
	default:   res = i_sar_d_1(context); break;
	}
	return res;
}

static BOOL i_rol_b_cl(LPCONTEXT context) {		// D2 ?0?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = GET_REG8(context, CL);
	OPC_AL_CL_FN__TO_DEST_FLAGS(rol);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_ror_b_cl(LPCONTEXT context) {		// D2 ?1?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = GET_REG8(context, CL);
	OPC_AL_CL_FN__TO_DEST_FLAGS(ror);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_rcl_b_cl(LPCONTEXT context) {		// D2 ?2?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = GET_REG8(context, CL);
	OPC_AL_CL_FN__TO_DEST_FLAGS(rcl);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_rcr_b_cl(LPCONTEXT context) {		// D2 ?3?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = GET_REG8(context, CL);
	OPC_AL_CL_FN__TO_DEST_FLAGS(rcr);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_shl_b_cl(LPCONTEXT context) {		// D2 ?4?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = GET_REG8(context, CL);
	OPC_AL_CL_FN__TO_DEST_FLAGS(shl);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_shr_b_cl(LPCONTEXT context) {		// D2 ?5?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = GET_REG8(context, CL);
	OPC_AL_CL_FN__TO_DEST_FLAGS(shr);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_sal_b_cl(LPCONTEXT context) {		// D2 ?6?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = GET_REG8(context, CL);
	OPC_AL_CL_FN__TO_DEST_FLAGS(sal);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_sar_b_cl(LPCONTEXT context) {		// D2 ?7?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = GET_REG8(context, CL);
	OPC_AL_CL_FN__TO_DEST_FLAGS(sar);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_d2pre(LPCONTEXT context) {		// D2
	BOOL	res = FALSE;
	OPCODE	ModRM = PEEK_OPCODE(context);
	switch (ModRM.reg_op) {
	case 0x00: res = i_rol_b_cl(context); break;
	case 0x01: res = i_ror_b_cl(context); break;
	case 0x02: res = i_rcl_b_cl(context); break;
	case 0x03: res = i_rcr_b_cl(context); break;
	case 0x04: res = i_shl_b_cl(context); break;
	case 0x05: res = i_shr_b_cl(context); break;
	case 0x06: res = i_sal_b_cl(context); break;
	default:   res = i_sar_b_cl(context); break;
	}
	return res;
}

static BOOL i_rol_d_cl(LPCONTEXT context) {		// D3 ?0?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	BYTE	src = GET_REG8(context, CL);
	OPC_EAX_CL_FN__TO_DEST_FLAGS(rol);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_ror_d_cl(LPCONTEXT context) {		// D3 ?1?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	BYTE	src = GET_REG8(context, CL);
	OPC_EAX_CL_FN__TO_DEST_FLAGS(ror);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_rcl_d_cl(LPCONTEXT context) {		// D3 ?2?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	BYTE	src = GET_REG8(context, CL);
	OPC_EAX_CL_FN__TO_DEST_FLAGS(rcl);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_rcr_d_cl(LPCONTEXT context) {		// D3 ?3?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	BYTE	src = GET_REG8(context, CL);
	OPC_EAX_CL_FN__TO_DEST_FLAGS(rcr);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_shl_d_cl(LPCONTEXT context) {		// D3 ?4?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	BYTE	src = GET_REG8(context, CL);
	OPC_EAX_CL_FN__TO_DEST_FLAGS(shl);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_shr_d_cl(LPCONTEXT context) {		// D3 ?5?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	BYTE	src = GET_REG8(context, CL);
	OPC_EAX_CL_FN__TO_DEST_FLAGS(shr);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_sal_d_cl(LPCONTEXT context) {		// D3 ?6?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	BYTE	src = GET_REG8(context, CL);
	OPC_EAX_CL_FN__TO_DEST_FLAGS(sal);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_sar_d_cl(LPCONTEXT context) {		// D3 ?7?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	BYTE	src = GET_REG8(context, CL);
	OPC_EAX_CL_FN__TO_DEST_FLAGS(sar);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_d3pre(LPCONTEXT context) {		// D3
	BOOL	res = FALSE;
	OPCODE	ModRM = PEEK_OPCODE(context);
	switch (ModRM.reg_op) {
	case 0x00: res = i_rol_d_cl(context); break;
	case 0x01: res = i_ror_d_cl(context); break;
	case 0x02: res = i_rcl_d_cl(context); break;
	case 0x03: res = i_rcr_d_cl(context); break;
	case 0x04: res = i_shl_d_cl(context); break;
	case 0x05: res = i_shr_d_cl(context); break;
	case 0x06: res = i_sal_d_cl(context); break;
	default:   res = i_sar_d_cl(context); break;
	}
	return res;
}

DECLARE_I_Invalid(d4invalid)				// D4
DECLARE_I_Invalid(d5invalid)				// D5
DECLARE_I_Invalid(d6invalid)				// D6
DECLARE_I_Invalid(xlat)						// D7
DECLARE_I_Invalid(d8escape)					// D8
DECLARE_I_Invalid(d9escape)					// D9
DECLARE_I_Invalid(daescape)					// DA
DECLARE_I_Invalid(dbescape)					// DB
DECLARE_I_Invalid(dcescape)					// DC
DECLARE_I_Invalid(ddescape)					// DD
DECLARE_I_Invalid(deescape)					// DE
DECLARE_I_Invalid(dfescape)					// DF

static BOOL i_loopne_o8(LPCONTEXT context) {	// E0
	DWORD	offset = FETCH_OFFSET8(context);
	SET_REG32(context, ECX, GET_REG32(context, ECX) - 1);
	if (!test_cond_ecxz(context) && !test_cond_e(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_loope_o8(LPCONTEXT context) {		// E1
	DWORD	offset = FETCH_OFFSET8(context);
	SET_REG32(context, ECX, GET_REG32(context, ECX) - 1);
	if (!test_cond_ecxz(context) && test_cond_e(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_loop_o8(LPCONTEXT context) {		// E2
	DWORD	offset = FETCH_OFFSET8(context);
	SET_REG32(context, ECX, GET_REG32(context, ECX) - 1);
	if (!test_cond_ecxz(context))
		context->Eip += offset;
	return TRUE;
}

static BOOL i_jecxz_o8(LPCONTEXT context) {		// E3
	DWORD	offset = FETCH_OFFSET8(context);
	if (test_cond_ecxz(context))
		context->Eip += offset;
	return TRUE;
}

DECLARE_I_Invalid(in_al_i8)					// E4
DECLARE_I_Invalid(in_eax_i8)				// E5
DECLARE_I_Invalid(out_al_i8)				// E6
DECLARE_I_Invalid(out_eax_i8)				// E7

static BOOL i_call_o32(LPCONTEXT context) {		// E8
	DWORD	offset = FETCH_OFFSET32(context);
	PUSH_DWORD(context, context->Eip);
	context->Eip += offset;
	return TRUE;
}

static BOOL i_jmp_o32(LPCONTEXT context) {		// E9
	DWORD	offset = FETCH_OFFSET32(context);
	context->Eip += offset;
	return TRUE;
}

DECLARE_I_Invalid(jmp_far)					// EA

static BOOL i_jmp_o8(LPCONTEXT context) {		// EB
	DWORD	offset = FETCH_OFFSET8(context);
	context->Eip += offset;
	return TRUE;
}

DECLARE_I_Invalid(in_al_dx)					// EC
DECLARE_I_Invalid(in_eax_dx)				// ED
DECLARE_I_Invalid(out_dx_al)				// EE
DECLARE_I_Invalid(out_dx_eax)				// EF
DECLARE_I_Invalid(lock)						// F0
DECLARE_I_Invalid(f1invalid)				// F1
/*
static BOOL i_rep(LPCONTEXT context, BOOL zr_cond) {
	seg		cur_seg_prefix = seg_prefix;
	DWORD	cur_Eip = context->Eip;
	DWORD	count = GET_REG32(ECX);
	for (;;)
	{
		OPCODE	ModRM = PEEK_OPCODE(context);
		switch (ModRM.byte) {
		case 0x26:	// es: FETCH_OPCODE(context); i_es(context); continue;
		case 0x2e: FETCH_OPCODE(context); i_cs(context); continue;
		case 0x36: FETCH_OPCODE(context); i_ss(context); continue;
		case 0x3e: FETCH_OPCODE(context); i_ds(context); continue;
@@@
	case 0xa4: res = i_movsb(context); continue;
	case 0xa5: res = i_movsd(context); continue;
	case 0xaa: res = i_stosb(context); continue;
	case 0xab: res = i_stosd(context); continue;
	case 0xac: res = i_lodsb(context); continue;
	case 0xad: res = i_lodsd(context); continue;

	case 0xa6: res = i_cmpsb(context); break;
	case 0xa7: res = i_cmpsd(context); break;
	case 0xae: res = i_scasb(context); break;
	case 0xaf: res = i_scasd(context); break;

	default:
		return TRUE;
		}
	}
}

static BOOL i_repne(LPCONTEXT context) {		// F2
	return i_rep(context, FALSE);
}

static BOOL i_repe(LPCONTEXT context) {			// F3
	return i_rep(context, TRUE);
}
*/
DECLARE_I_Invalid(repne)					// F2
DECLARE_I_Invalid(repe)						// F3

DECLARE_I_Invalid(hlt)						// F4

static BOOL i_cmc(LPCONTEXT context) {			// F5
	SET_FLAGS(context, GET_FLAGS(context) ^ CF);
	return TRUE;
}

static BOOL i_test_b_i8(LPCONTEXT context) {	// F6 ?0? / ?1?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	BYTE	src = FETCH_BYTE(context);
	OPC_AL_DL_FN__TO_FLAGS(test);
	return TRUE;
}

static BOOL i_not_b(LPCONTEXT context) {		// F6 ?2?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	OPC_AL_FN__TO_DEST_FLAGS(not);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_neg_b(LPCONTEXT context) {		// F6 ?3?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	OPC_AL_FN__TO_DEST_FLAGS(neg);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_mul_al_b(LPCONTEXT context) {		// F6 ?4?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GET_REG8(context, AL);
	BYTE	src = GetModRMRM8(context, ModRM);
	OPCAL_AH_FN__TO_SRC_DEST_FLAGS(mul);
	SET_REG8(context, AL, dest);
	SET_REG8(context, AH, src);
	return TRUE;
}

static BOOL i_imul_b(LPCONTEXT context) {		// F6 ?5?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GET_REG8(context, AL);
	BYTE	src = GetModRMRM8(context, ModRM);
	OPCAL_AH_FN__TO_SRC_DEST_FLAGS(imul);
	SET_REG8(context, AL, dest);
	SET_REG8(context, AH, src);
	return TRUE;
}

static BOOL i_div_al_b(LPCONTEXT context) {		// F6 ?6?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GET_REG8(context, AL);
	BYTE	src = GetModRMRM8(context, ModRM);
	OPCAL_AH_FN__TO_SRC_DEST_FLAGS(div);
	SET_REG8(context, AL, dest);
	SET_REG8(context, AH, src);
	return TRUE;
}

static BOOL i_idiv_b(LPCONTEXT context) {		// F6 ?7?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GET_REG8(context, AL);
	BYTE	src = GetModRMRM8(context, ModRM);
	OPCAL_AH_FN__TO_SRC_DEST_FLAGS(idiv);
	SET_REG8(context, AL, dest);
	SET_REG8(context, AH, src);
	return TRUE;
}

static BOOL i_f6pre(LPCONTEXT context) {		// F6
	BOOL	res = FALSE;
	OPCODE	ModRM = PEEK_OPCODE(context);
	switch (ModRM.reg_op) {
	case 0x00: res = i_test_b_i8(context); break;
	case 0x01: res = i_test_b_i8(context); break;
	case 0x02: res = i_not_b(context); break;
	case 0x03: res = i_neg_b(context); break;
	case 0x04: res = i_mul_al_b(context); break;
	case 0x05: res = i_imul_b(context); break;
	case 0x06: res = i_div_al_b(context); break;
	default:   res = i_idiv_b(context); break;
	}
	return res;
}

static BOOL i_test_d_i32(LPCONTEXT context) {	// F7 ?0? / ?1?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	DWORD	src = FETCH_DWORD(context);
	OPC_EAX_EDX_FN__TO_FLAGS(test);
	return TRUE;
}

static BOOL i_not_d(LPCONTEXT context) {		// F7 ?2?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	OPC_EAX_FN__TO_DEST_FLAGS(not);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_neg_d(LPCONTEXT context) {		// F7 ?3?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	OPC_EAX_FN__TO_DEST_FLAGS(neg);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_mul_eax_d(LPCONTEXT context) {	// F7 ?4?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GET_REG32(context, EAX);
	DWORD	src = GetModRMRM32(context, ModRM);
	OPCEAX_EDX_FN__TO_SRC_DEST_FLAGS(mul);
	SET_REG32(context, EAX, dest);
	SET_REG32(context, EDX, src);
	return TRUE;
}

static BOOL i_imul_d(LPCONTEXT context) {		// F7 ?5?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GET_REG32(context, EAX);
	DWORD	src = GetModRMRM32(context, ModRM);
	OPCEAX_EDX_FN__TO_SRC_DEST_FLAGS(imul);
	SET_REG32(context, EAX, dest);
	SET_REG32(context, EDX, src);
	return TRUE;
}

static BOOL i_div_eax_d(LPCONTEXT context) {	// F7 ?6?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GET_REG32(context, EAX);
	DWORD	src = GetModRMRM32(context, ModRM);
	OPCEAX_EDX_FN__TO_SRC_DEST_FLAGS(div);
	SET_REG32(context, EAX, dest);
	SET_REG32(context, EDX, src);
	return TRUE;
}

static BOOL i_idiv_d(LPCONTEXT context) {		// F7 ?7?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GET_REG32(context, EAX);
	DWORD	src = GetModRMRM32(context, ModRM);
	OPCEAX_EDX_FN__TO_SRC_DEST_FLAGS(idiv);
	SET_REG32(context, EAX, dest);
	SET_REG32(context, EDX, src);
	return TRUE;
}

static BOOL i_f7pre(LPCONTEXT context) {		// F7
	BOOL	res = FALSE;
	OPCODE	ModRM = PEEK_OPCODE(context);
	switch (ModRM.reg_op) {
	case 0x00: res = i_test_d_i32(context); break;
	case 0x01: res = i_test_d_i32(context); break;
	case 0x02: res = i_not_d(context); break;
	case 0x03: res = i_neg_d(context); break;
	case 0x04: res = i_mul_eax_d(context); break;
	case 0x05: res = i_imul_d(context); break;
	case 0x06: res = i_div_eax_d(context); break;
	default:   res = i_idiv_d(context); break;
	}
	return res;
}

static BOOL i_clc(LPCONTEXT context) {			// F8
	SET_FLAGS(context, GET_FLAGS(context) & ~CF);
	return TRUE;
}

static BOOL i_stc(LPCONTEXT context) {			// F9
	SET_FLAGS(context, GET_FLAGS(context) | CF);
	return TRUE;
}

DECLARE_I_Invalid(cli)						// FA
DECLARE_I_Invalid(sti)						// FB

static BOOL i_cld(LPCONTEXT context) {			// FC
	SET_FLAGS(context, GET_FLAGS(context) & ~DF);
	return TRUE;
}

static BOOL i_std(LPCONTEXT context) {			// FD
	SET_FLAGS(context, GET_FLAGS(context) | DF);
	return TRUE;
}

static BOOL i_inc_b(LPCONTEXT context) {		// FE ?0?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	OPC_AL_FN__TO_DEST_FLAGS(inc);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_dec_b(LPCONTEXT context) {		// FE ?1?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	BYTE	dest = GetModRMRM8(context, ModRM);
	OPC_AL_FN__TO_DEST_FLAGS(dec);
	PutbackModRMRM8(context, ModRM, dest);
	return TRUE;
}

static BOOL i_fepre(LPCONTEXT context) {		// FE
	BOOL	res = FALSE;
	OPCODE	ModRM = PEEK_OPCODE(context);
	switch (ModRM.reg_op) {
	case 0x00: res = i_inc_b(context); break;
	case 0x01: res = i_dec_b(context); break;
	default:   break;
	}
	return res;
}

static BOOL i_inc_d(LPCONTEXT context) {		// FF ?0?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	OPC_EAX_FN__TO_DEST_FLAGS(inc);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_dec_d(LPCONTEXT context) {		// FF ?1?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	OPC_EAX_FN__TO_DEST_FLAGS(dec);
	PutbackModRMRM32(context, ModRM, dest);
	return TRUE;
}

static BOOL i_call_d(LPCONTEXT context) {	// FF ?2?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	PUSH_DWORD(context, context->Eip);
	context->Eip = dest;
	return TRUE;
}

static BOOL i_call_f(LPCONTEXT context) {	// FF ?3?
	OPCODE	ModRM = FETCH_OPCODE(context);
	if (ModRM.adr_op == 3)	// 'call fword ptr reg' is invalid
		return FALSE;
	ULARGE_INTEGER	descr = GetModRMRM64(context, ModRM);
	LDT_DESCR	*seg = GetLDTDescrFromAddr(descr.LowPart);
	LDT_DESCR	*adr = GetLDTDescrFromAddr(descr.HighPart);
	if (//seg == NULL || !DESCR_ar_LDT_CODE_SEG(seg->ldt.ar) ||
		adr == NULL || !DESCR_ar_LDT_CALL_GATE(adr->ldt.ar))
		return FALSE;
	DWORD	tmp_esp = GET_REG32(context, ESP);
	PUSH_DWORD(context, GET_SEG(context, SS));
	PUSH_DWORD(context, tmp_esp);
	// Note: Save 0 instead of the CS segment to raise an exception ...
	// ... when the program tries to execute the 'retf' statement
	PUSH_DWORD(context, 0);	// GET_SEG(context, CS)
	PUSH_DWORD(context, context->Eip);
	context->Eip = (adr->call_gate.offset_high << 16) | adr->call_gate.offset_low;
	return TRUE;
}

static BOOL i_jmp_d(LPCONTEXT context) {		// FF ?4?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	context->Eip = dest;
	return TRUE;
}

DECLARE_I_Invalid(jmp_f)					// FF ?5?

static BOOL i_push_d(LPCONTEXT context) {		// FF ?6?
	// Note: ModRM.reg_op is not used
	OPCODE	ModRM = FETCH_OPCODE(context);
	DWORD	dest = GetModRMRM32(context, ModRM);
	PUSH_DWORD(context, dest);
	return TRUE;
}

static BOOL i_ffpre(LPCONTEXT context) {		// FF
	BOOL	res = FALSE;
	OPCODE	ModRM = PEEK_OPCODE(context);
	switch (ModRM.reg_op) {
	case 0x00: res = i_inc_d(context); break;
	case 0x01: res = i_dec_d(context); break;
	case 0x02: res = i_call_d(context); break;
	case 0x03: res = i_call_f(context); break;
	case 0x04: res = i_jmp_d(context); break;
	case 0x05: res = i_jmp_f(context); break;
	case 0x06: res = i_push_d(context); break;
	default:   break;
	}
	return res;
}

BOOL TrackX86Opcode(LPCONTEXT context)
{
	DWORD	save_Eip = context->Eip;
	DWORD	res = FALSE;
	__try {
		OPCODE	ModRM = FETCH_OPCODE(context);
		switch (ModRM.byte) {
		case 0x00: res = i_add_b_r8(context); break;
		case 0x01: res = i_add_d_r32(context); break;
		case 0x02: res = i_add_r8_b(context); break;
		case 0x03: res = i_add_r32_d(context); break;
		case 0x04: res = i_add_al_i8(context); break;
		case 0x05: res = i_add_eax_i32(context); break;
		case 0x06: res = i_push_es(context); break;
		case 0x07: res = i_pop_es(context); break;
		case 0x08: res = i_or_b_r8(context); break;
		case 0x09: res = i_or_d_r32(context); break;
		case 0x0a: res = i_or_r8_b(context); break;
		case 0x0b: res = i_or_r32_d(context); break;
		case 0x0c: res = i_or_al_i8(context); break;
		case 0x0d: res = i_or_eax_i32(context); break;
		case 0x0e: res = i_push_cs(context); break;
		case 0x0f: res = i_0fpre(context); break;
		case 0x10: res = i_adc_b_r8(context); break;
		case 0x11: res = i_adc_d_r32(context); break;
		case 0x12: res = i_adc_r8_b(context); break;
		case 0x13: res = i_adc_r32_d(context); break;
		case 0x14: res = i_adc_al_i8(context); break;
		case 0x15: res = i_adc_eax_i32(context); break;
		case 0x16: res = i_push_ss(context); break;
		case 0x17: res = i_pop_ss(context); break;
		case 0x18: res = i_sbb_b_r8(context); break;
		case 0x19: res = i_sbb_d_r32(context); break;
		case 0x1a: res = i_sbb_r8_b(context); break;
		case 0x1b: res = i_sbb_r32_d(context); break;
		case 0x1c: res = i_sbb_al_i8(context); break;
		case 0x1d: res = i_sbb_eax_i32(context); break;
		case 0x1e: res = i_push_ds(context); break;
		case 0x1f: res = i_pop_ds(context); break;
		case 0x20: res = i_and_b_r8(context); break;
		case 0x21: res = i_and_d_r32(context); break;
		case 0x22: res = i_and_r8_b(context); break;
		case 0x23: res = i_and_r32_d(context); break;
		case 0x24: res = i_and_al_i8(context); break;
		case 0x25: res = i_and_eax_i32(context); break;
		case 0x26: res = i_es(context); break;
		case 0x27: res = i_daa(context); break;
		case 0x28: res = i_sub_b_r8(context); break;
		case 0x29: res = i_sub_d_r32(context); break;
		case 0x2a: res = i_sub_r8_b(context); break;
		case 0x2b: res = i_sub_r32_d(context); break;
		case 0x2c: res = i_sub_al_i8(context); break;
		case 0x2d: res = i_sub_eax_i32(context); break;
		case 0x2e: res = i_cs(context); break;
		case 0x2f: res = i_das(context); break;
		case 0x30: res = i_xor_b_r8(context); break;
		case 0x31: res = i_xor_d_r32(context); break;
		case 0x32: res = i_xor_r8_b(context); break;
		case 0x33: res = i_xor_r32_d(context); break;
		case 0x34: res = i_xor_al_i8(context); break;
		case 0x35: res = i_xor_eax_i32(context); break;
		case 0x36: res = i_ss(context); break;
		case 0x37: res = i_aaa(context); break;
		case 0x38: res = i_cmp_b_r8(context); break;
		case 0x39: res = i_cmp_d_r32(context); break;
		case 0x3a: res = i_cmp_r8_b(context); break;
		case 0x3b: res = i_cmp_r32_d(context); break;
		case 0x3c: res = i_cmp_al_i8(context); break;
		case 0x3d: res = i_cmp_eax_i32(context); break;
		case 0x3e: res = i_ds(context); break;
		case 0x3f: res = i_aas(context); break;
		case 0x40: res = i_inc_eax(context); break;
		case 0x41: res = i_inc_ecx(context); break;
		case 0x42: res = i_inc_edx(context); break;
		case 0x43: res = i_inc_ebx(context); break;
		case 0x44: res = i_inc_esp(context); break;
		case 0x45: res = i_inc_ebp(context); break;
		case 0x46: res = i_inc_esi(context); break;
		case 0x47: res = i_inc_edi(context); break;
		case 0x48: res = i_dec_eax(context); break;
		case 0x49: res = i_dec_ecx(context); break;
		case 0x4a: res = i_dec_edx(context); break;
		case 0x4b: res = i_dec_ebx(context); break;
		case 0x4c: res = i_dec_esp(context); break;
		case 0x4d: res = i_dec_ebp(context); break;
		case 0x4e: res = i_dec_esi(context); break;
		case 0x4f: res = i_dec_edi(context); break;
		case 0x50: res = i_push_eax(context); break;
		case 0x51: res = i_push_ecx(context); break;
		case 0x52: res = i_push_edx(context); break;
		case 0x53: res = i_push_ebx(context); break;
		case 0x54: res = i_push_esp(context); break;
		case 0x55: res = i_push_ebp(context); break;
		case 0x56: res = i_push_esi(context); break;
		case 0x57: res = i_push_edi(context); break;
		case 0x58: res = i_pop_eax(context); break;
		case 0x59: res = i_pop_ecx(context); break;
		case 0x5a: res = i_pop_edx(context); break;
		case 0x5b: res = i_pop_ebx(context); break;
		case 0x5c: res = i_pop_esp(context); break;
		case 0x5d: res = i_pop_ebp(context); break;
		case 0x5e: res = i_pop_esi(context); break;
		case 0x5f: res = i_pop_edi(context); break;
		case 0x60: res = i_pushad(context); break;
		case 0x61: res = i_popad(context); break;
		case 0x62: res = i_bound(context); break;
		case 0x63: res = i_arpl(context); break;
		case 0x64: res = i_64invalid(context); break;
		case 0x65: res = i_65invalid(context); break;
		case 0x66: res = i_66invalid(context); break;
		case 0x67: res = i_67invalid(context); break;
		case 0x68: res = i_push_i32(context); break;
		case 0x69: res = i_imul_r32_d_i32(context); break;
		case 0x6a: res = i_push_o8(context); break;
		case 0x6b: res = i_imul_r32_d_i8(context); break;
		case 0x6c: res = i_insb(context); break;
		case 0x6d: res = i_insd(context); break;
		case 0x6e: res = i_outsb(context); break;
		case 0x6f: res = i_outsd(context); break;
		case 0x70: res = i_jo_o8(context); break;
		case 0x71: res = i_jno_o8(context); break;
		case 0x72: res = i_jb_o8(context); break;
		case 0x73: res = i_jae_o8(context); break;
		case 0x74: res = i_je_o8(context); break;
		case 0x75: res = i_jne_o8(context); break;
		case 0x76: res = i_jbe_o8(context); break;
		case 0x77: res = i_ja_o8(context); break;
		case 0x78: res = i_js_o8(context); break;
		case 0x79: res = i_jns_o8(context); break;
		case 0x7a: res = i_jp_o8(context); break;
		case 0x7b: res = i_jnp_o8(context); break;
		case 0x7c: res = i_jl_o8(context); break;
		case 0x7d: res = i_jge_o8(context); break;
		case 0x7e: res = i_jle_o8(context); break;
		case 0x7f: res = i_jg_o8(context); break;
		case 0x80: res = i_80pre(context); break;
		case 0x81: res = i_81pre(context); break;
		case 0x82: res = i_82invalid(context); break;
		case 0x83: res = i_83pre(context); break;
		case 0x84: res = i_test_b_r8(context); break;
		case 0x85: res = i_test_d_r32(context); break;
		case 0x86: res = i_xchg_b_r8(context); break;
		case 0x87: res = i_xchg_d_r32(context); break;
		case 0x88: res = i_mov_b_r8(context); break;
		case 0x89: res = i_mov_d_r32(context); break;
		case 0x8a: res = i_mov_r8_b(context); break;
		case 0x8b: res = i_mov_r32_d(context); break;
		case 0x8c: res = i_mov_w_seg(context); break;
		case 0x8d: res = i_lea_r32_d(context); break;
		case 0x8e: res = i_mov_seg_w(context); break;
		case 0x8f: res = i_pop_d(context); break;
		case 0x90: res = i_nop(context); break;
		case 0x91: res = i_xchg_eax_ecx(context); break;
		case 0x92: res = i_xchg_eax_edx(context); break;
		case 0x93: res = i_xchg_eax_ebx(context); break;
		case 0x94: res = i_xchg_eax_esp(context); break;
		case 0x95: res = i_xchg_eax_ebp(context); break;
		case 0x96: res = i_xchg_eax_esi(context); break;
		case 0x97: res = i_xchg_eax_edi(context); break;
		case 0x98: res = i_cwde(context); break;
		case 0x99: res = i_cdq(context); break;
		case 0x9a: res = i_call_far(context); break;
		case 0x9b: res = i_wait(context); break;
		case 0x9c: res = i_pushfd(context); break;
		case 0x9d: res = i_popfd(context); break;
		case 0x9e: res = i_sahf(context); break;
		case 0x9f: res = i_lahf(context); break;
		case 0xa0: res = i_mov_al_m32(context); break;
		case 0xa1: res = i_mov_eax_m32(context); break;
		case 0xa2: res = i_mov_m32_al(context); break;
		case 0xa3: res = i_mov_m32_eax(context); break;
		case 0xa4: res = i_movsb(context); break;
		case 0xa5: res = i_movsd(context); break;
		case 0xa6: res = i_cmpsb(context); break;
		case 0xa7: res = i_cmpsd(context); break;
		case 0xa8: res = i_test_al_i8(context); break;
		case 0xa9: res = i_test_eax_i32(context); break;
		case 0xaa: res = i_stosb(context); break;
		case 0xab: res = i_stosd(context); break;
		case 0xac: res = i_lodsb(context); break;
		case 0xad: res = i_lodsd(context); break;
		case 0xae: res = i_scasb(context); break;
		case 0xaf: res = i_scasd(context); break;
		case 0xb0: res = i_mov_al_i8(context); break;
		case 0xb1: res = i_mov_cl_i8(context); break;
		case 0xb2: res = i_mov_dl_i8(context); break;
		case 0xb3: res = i_mov_bl_i8(context); break;
		case 0xb4: res = i_mov_ah_i8(context); break;
		case 0xb5: res = i_mov_ch_i8(context); break;
		case 0xb6: res = i_mov_dh_i8(context); break;
		case 0xb7: res = i_mov_bh_i8(context); break;
		case 0xb8: res = i_mov_eax_i32(context); break;
		case 0xb9: res = i_mov_ecx_i32(context); break;
		case 0xba: res = i_mov_edx_i32(context); break;
		case 0xbb: res = i_mov_ebx_i32(context); break;
		case 0xbc: res = i_mov_esp_i32(context); break;
		case 0xbd: res = i_mov_ebp_i32(context); break;
		case 0xbe: res = i_mov_esi_i32(context); break;
		case 0xbf: res = i_mov_edi_i32(context); break;
		case 0xc0: res = i_c0pre(context); break;
		case 0xc1: res = i_c1pre(context); break;
		case 0xc2: res = i_ret_i16(context); break;
		case 0xc3: res = i_ret(context); break;
		case 0xc4: res = i_les_fw(context); break;
		case 0xc5: res = i_lds_fw(context); break;
		case 0xc6: res = i_mov_b_i8(context); break;
		case 0xc7: res = i_mov_d_i32(context); break;
		case 0xc8: res = i_enter_i16i8(context); break;
		case 0xc9: res = i_leave(context); break;
		case 0xca: res = i_retf_d16(context); break;
		case 0xcb: res = i_retf(context); break;
		case 0xcc: res = i_int3(context); break;
		case 0xcd: res = i_int_i8(context); break;
		case 0xce: res = i_into(context); break;
		case 0xcf: res = i_iretd(context); break;
		case 0xd0: res = i_d0pre(context); break;
		case 0xd1: res = i_d1pre(context); break;
		case 0xd2: res = i_d2pre(context); break;
		case 0xd3: res = i_d3pre(context); break;
		case 0xd4: res = i_d4invalid(context); break;
		case 0xd5: res = i_d5invalid(context); break;
		case 0xd6: res = i_d6invalid(context); break;
		case 0xd7: res = i_xlat(context); break;
		case 0xd8: res = i_d8escape(context); break;
		case 0xd9: res = i_d9escape(context); break;
		case 0xda: res = i_daescape(context); break;
		case 0xdb: res = i_dbescape(context); break;
		case 0xdc: res = i_dcescape(context); break;
		case 0xdd: res = i_ddescape(context); break;
		case 0xde: res = i_deescape(context); break;
		case 0xdf: res = i_dfescape(context); break;
		case 0xe0: res = i_loopne_o8(context); break;
		case 0xe1: res = i_loope_o8(context); break;
		case 0xe2: res = i_loop_o8(context); break;
		case 0xe3: res = i_jecxz_o8(context); break;
		case 0xe4: res = i_in_al_i8(context); break;
		case 0xe5: res = i_in_eax_i8(context); break;
		case 0xe6: res = i_out_al_i8(context); break;
		case 0xe7: res = i_out_eax_i8(context); break;
		case 0xe8: res = i_call_o32(context); break;
		case 0xe9: res = i_jmp_o32(context); break;
		case 0xea: res = i_jmp_far(context); break;
		case 0xeb: res = i_jmp_o8(context); break;
		case 0xec: res = i_in_al_dx(context); break;
		case 0xed: res = i_in_eax_dx(context); break;
		case 0xee: res = i_out_dx_al(context); break;
		case 0xef: res = i_out_dx_eax(context); break;
		case 0xf0: res = i_lock(context); break;
		case 0xf1: res = i_f1invalid(context); break;
		case 0xf2: res = i_repne(context); break;
		case 0xf3: res = i_repe(context); break;
		case 0xf4: res = i_hlt(context); break;
		case 0xf5: res = i_cmc(context); break;
		case 0xf6: res = i_f6pre(context); break;
		case 0xf7: res = i_f7pre(context); break;
		case 0xf8: res = i_clc(context); break;
		case 0xf9: res = i_stc(context); break;
		case 0xfa: res = i_cli(context); break;
		case 0xfb: res = i_sti(context); break;
		case 0xfc: res = i_cld(context); break;
		case 0xfd: res = i_std(context); break;
		case 0xfe: res = i_fepre(context); break;
		case 0xff: res = i_ffpre(context); break;
		default: break;
		}
	} __except(EXCEPTION_EXECUTE_HANDLER) {
		__asm int 3
		res = FALSE;
	}
	if (!res)
		context->Eip = save_Eip;
	return res;
}
