///////////////////////////////////////////////////////////////////////////////
//
//  File:    joy.cpp
//
//  Class:   Joystick
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      The Joystick class encapsulates joystick access.  This is an abstract
//      class that serves as a base for derived joystick classes for each
//      platform supported.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "joy.h"
#include "config.h"


///////////////////////////////////////////////////////////////////////////////
//  Static Data Initialization.
///////////////////////////////////////////////////////////////////////////////
//  The names of the joystick controls.  This is declared as an an array of 
//  indeterminate size instead of size JOY_COUNT so that we can check to make 
//  sure that all controls have names.  If we had specified the size of 
//  JOY_COUNT we could have left entries out of this list without the compiler 
//  complaining.  Instead, an ASSERT in the constructor will catch any size
//  mismatches.
const char* Joystick::sm_pstrJoyNames[ ] =
{
    "No Joy",

    "J1 Left",    "J1 Right",   "J1 Up",      "J1 Down",
    "J1 Btn1",    "J1 Btn2",    "J1 Btn3",    "J1 Btn4",

    "J2 Left",    "J2 Right",   "J2 Up",      "J2 Down",
    "J2 Btn1",    "J2 Btn2",    "J2 Btn3",    "J2 Btn4"
};



///////////////////////////////////////////////////////////////////////////////
//
//  Function: Joystick
//
//  Description:
//
//      This is the main constructor for a joystick object.
//      It is protected so that only derived classes have access to it.
//      Clients should use a build method to create a joystick object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Joystick::Joystick(
    const KString&  iName
)
:
    DigitalController ( iName )
{
    //  Check to make sure we have descriptions for each control.
    ASSERT(
        sizeof( sm_pstrJoyNames ) / sizeof( sm_pstrJoyNames[ 0 ] ) == JOY_COUNT
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: init
//
//  Description:
//
//      This is called to initialize the joystick object.  By using an init
//      member we get access to virtual functions that we wouldn't in the
//      constructor.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Joystick::init(
)
{
    //  Call the base class.
    DigitalController::init( );

    //  Is the joystick disabled?
    if( Configuration::s_instance( ).getParam( "-nojoy" ) )
    {
        m_bEnabled = FALSE;
    }
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~Joystick
//
//  Description:
//
//      This is the destructor for a joystick object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Joystick::~Joystick(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getNumSwitches
//
//  Description:
//
//      This returns the number of switches(controls) on the joystick.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The number of switches(controls).
//
///////////////////////////////////////////////////////////////////////////////
DWord
Joystick::getNumSwitches(
) const
{
    return( JOY_COUNT );
}





///////////////////////////////////////////////////////////////////////////////
//
//  Function: switchName
//
//  Description:
//
//      This returns the name of the specified switch(control).
//
//  Parameters:
//
//      dwSwitch (input)
//          The switch(control) of interest.
//
//  Returns:
//
//      The name of the switch(control).
//
///////////////////////////////////////////////////////////////////////////////
const
char*
Joystick::switchName(
    DWord dwSwitch
) const
{
    ASSERT( dwSwitch < getNumSwitches( ) );

    //  Return the switch(control) name.
    return( sm_pstrJoyNames[ dwSwitch ] );
}

