///////////////////////////////////////////////////////////////////////////////
//
//  File:       switch.h
//
//  Class:      Switch - Concrete
//  
//  Hierarchy:  Switch - RepBase 
//
//  Author:     Kevin Brisley
//
//  Description:
//
//      The Switch class encapsulates a single digital control on a digital
//      controller (e.g. key, button, joystick direction, etc.).  A switch
//      can take on one of two possible states (on, off).
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////
#ifndef _switch_h_
#define _switch_h_

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "reptypes.h"
#include "repbase.h"
#include "kstring.h"
#include "kplist.h"

//  Forward Class Declarations.
class DigitalController;
class ByteModifier;


///////////////////////////////////////////////////////////////////////////////
//  Switch Class Definition
///////////////////////////////////////////////////////////////////////////////
class Switch : public RepBase
{
    public:

        //  Creation/Deletion.
                 Switch ( const KString& iName, DigitalController* pOwner );
        virtual ~Switch ( );

        //  Base overrides.
        virtual const KString& getClassName ( ) const;

        //  Access.
        Byte isOn                 ( ) const;
        Byte isOnWait             ( );
        void waitUntilOff         ( ) const;
        Byte onPhysically         ( ) const;
        Byte onVirtually          ( ) const;
        void physicalOn           ( const Byte bOn ); 
        void virtualOn            ( const Byte bOn ); 
        void enableMappings       ( const Byte bEnable );
        void clearMappings        ( );
        void addMapping           ( Byte* pbLocation, const Byte bMask );
        void enableHardware       ( const Byte bEnable );

    protected:

        //  Utility.
        void set               ( );

        //  Data Members.
        DigitalController*     m_pOwner;
        Byte                   m_bOnPhysically;
        Byte                   m_bOnVirtually;
        Byte                   m_bWaiting;
        DWord                  m_dwWaitStart; 
        Byte                   m_bMappingsEnabled;
        KPtrList<ByteModifier> m_byteModifierList;
        Byte                   m_bHardwareEnabled;

        //  Static Data Members.
        static const DWord sm_dwWaitPeriod;

    private:

};


///////////////////////////////////////////////////////////////////////////////
//  Switch Inline Functions.
///////////////////////////////////////////////////////////////////////////////
inline
Byte
Switch::isOn(
) const
{
    return( m_bOnPhysically || m_bOnVirtually );
}

inline
Byte
Switch::onPhysically(
) const
{
    return( m_bOnPhysically );
}

inline
Byte
Switch::onVirtually(
) const
{
    return( m_bOnVirtually );
}

inline
void
Switch::physicalOn(
    const Byte bOn
)
{
    if( m_bHardwareEnabled )
    {
        m_bOnPhysically = bOn;
        set( );
    }
}
    
inline
void
Switch::virtualOn(
    const Byte bOn
)
{
    m_bOnVirtually = bOn;
    set( );
}
    
inline
void
Switch::enableMappings(
    const Byte bEnable
)
{
    m_bMappingsEnabled = bEnable;
}

inline
void
Switch::enableHardware(
    const Byte bEnable
)
{
    m_bHardwareEnabled = bEnable;
}

#endif
