///////////////////////////////////////////////////////////////////////////////
//
//  File:    6502mfc.cpp
//
//  Class:   CPU6502MFC
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class wraps Marat's 6502 emulation core.  This is a general 6502
//      core that can be used for any platform.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "reptypes.h"
#include "6502mfc.h"
#include "appfile.h"
#include "space.h"




///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_build
//
//  Description:
//
//      This is a factory method to create a 6502 CPU object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      pGame (input)
//          The game the CPU belongs to.
//
//      pSpace (input)
//          The address space that the CPU is primarily bound to.
//
//      ePurpose (input)
//          The purpose of the CPU.
//
//
//  Returns:
//
//      A pointer to the new object.
//
///////////////////////////////////////////////////////////////////////////////
CPU6502MFC*
CPU6502MFC::s_build(
    const KString&   iName,
    Game*            pGame, 
    AddrSpace*       pSpace,
    const CPUPurpose ePurpose /* = GAME */
)
{
    //  Create the new object.
    CPU6502MFC* pThis = new CPU6502MFC( iName, pGame, pSpace, ePurpose );

    //  Initialize the new object.
    pThis->init( );

    //  Send back the new CPU.
    return( pThis );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: CPU6502MFC
//
//  Description:
//
//      This is the main constructor for a 6502 CPU object.  It is protected
//      so that only derived classes have access to it.  Clients should
//      use the build method to create a CPU object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      pGame (input)
//          The game the CPU belongs to.
//
//      pSpace (input)
//          The address space that the CPU is primarily bound to.
//
//      ePurpose (input)
//          The purpose of the CPU.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
CPU6502MFC::CPU6502MFC(
    const KString&   iName,
    Game*            pGame, 
    AddrSpace*       pSpace,
    const CPUPurpose ePurpose /* = GAME */
)
:
    CPU6502  ( iName, pGame, pSpace, ePurpose )
{
    //  All initialization is done in init( ).
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: init
//
//  Description:
//
//      This is called to initialize the cpu object.  By using an init
//      member we get access to virtual functions that we wouldn't in the
//      constructor.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPU6502MFC::init(
)
{
    //  Call the base class.
    CPU6502::init( );

    //  Reset the CPU to initialize it.
    reset( ); 
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~CPU6502MFC
//
//  Description:
//
//      This is the destructor for the 6502 cpu object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
CPU6502MFC::~CPU6502MFC(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the 'C' based 6502 emulation object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
CPU6502MFC::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "CPU6502MFC" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: save
//
//  Description:
//
//      This member will save the CPU to a file for retrieval at a later
//      date.
//
//  Parameters:
//
//      pSaveFile
//          The file to save the CPU to.
//
//  Returns:
//
//      TRUE if there were errors during the save.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
CPU6502MFC::save(
    AppFile* pSaveFile
)
{
    ASSERT( pSaveFile != NULL );

    //  We always allow the base class to save itself first.
    CPU6502::save( pSaveFile );

    //  Save the member data that keeps the state.
    pSaveFile->write( ( Byte* )&m_context, sizeof( m_context ) );

    //  Was it saved successfully?
    return( pSaveFile->count( ) != sizeof( m_context ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: load
//
//  Description:
//
//      This member will load the CPU from a file that it was previously
//      saved to.
//
//  Parameters:
//
//      pLoadFile
//          The file to load the CPU from.
//
//  Returns:
//
//      TRUE if there were errors during the load.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
CPU6502MFC::load(
    AppFile* pLoadFile
)
{
    ASSERT( pLoadFile != NULL );

    //  We always allow the base class to load itself first.
    CPU6502::load( pLoadFile );

    //  Load the member data that keeps the state.
    pLoadFile->read( ( Byte* )&m_context, sizeof( m_context ) );

    //  Was it loaded successfully?
    return( pLoadFile->count( ) != sizeof( m_context ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: reset
//
//  Description:
//
//      This member is called to have the CPU reset itself.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPU6502MFC::reset(
)
{
    //  Let the base clase reset.
    CPU6502::reset( );

    //  Do the reset.
    activate( );
    Reset6502( &m_context );
    deactivate( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: run
//
//  Description:
//
//      This member is called to have the CPU execute for a time slice.
//
//  Parameters:
//
//      nCycles (input)
//          The number of cycles to execute for.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPU6502MFC::run(
    const int32 nCycles
)
{
    //  We should be active.
    ASSERT( sm_pCPU == this );
 
    //  Calculate the cycles to execute this slice.
    m_nCyclesThisRun = nCycles + m_nCyclesCarry;

    //  Let's rock.
    m_nCyclesCarry = Run6502( &m_context, m_nCyclesThisRun );

    //  Update the total cycles executed.
    m_nCyclesTotal += ( m_nCyclesThisRun - m_nCyclesCarry );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: interrupt
//
//  Description:
//
//      This member is called to interrupt the cpu.
//
//  Parameters:
//
//      nType (input)
//          The type of interrupt.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPU6502MFC::interrupt(
    const int32 nType
)
{
    //  Interrupt.
    Interrupt6502( &m_context, nType );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getReg
//
//  Description:
//
//      This member is called to retrieve the value of a CPU register.
//
//  Parameters:
//
//      nReg (input)
//          The register to retrieve.
//
//  Returns:
//
//      The value of the register.
//
///////////////////////////////////////////////////////////////////////////////
int32
CPU6502MFC::getReg(
    int32 nReg
)
{
    //  The result.
    int32 nResult = 0;

    //  Return the requested register.
    switch( nReg )
    {
        case REG_ICOUNT: nResult = m_context.ICount;                    break;
        case REG_PC:     nResult = m_context.PC.W;                      break;
        case REG_A:      nResult = m_context.A;                         break;
        case REG_X:      nResult = m_context.X;                         break;
        case REG_Y:      nResult = m_context.Y;                         break;
        case REG_SP:     nResult = m_context.S;                         break;
        case REG_SR:     nResult = m_context.P;                         break;
        default:     fatalError( "Unknown 6502 register %d", nReg ); break;
    }

    //  Return the result.
    return( nResult );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setReg
//
//  Description:
//
//      This member is called to set the value of a CPU register.
//
//  Parameters:
//
//      nReg (input)
//          The register to set.
//
//      nValue (input)
//          The value to set the register to.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPU6502MFC::setReg(
    int32 nReg,
    int32 nValue
)
{
    //  Set the requested register.
    switch( nReg )
    {
        case REG_ICOUNT: m_context.ICount = nValue;                    break;
        case REG_PC:     m_context.PC.W   = nValue;                    break;
        case REG_A:      m_context.A      = nValue;                    break;
        case REG_X:      m_context.X      = nValue;                    break;
        case REG_Y:      m_context.Y      = nValue;                    break;
        case REG_SP:     m_context.S      = nValue;                    break;
        case REG_SR:     m_context.P      = nValue;                    break;
        default:     fatalError( "Unknown 6502 register %d", nReg );   break;
    }
}
