///////////////////////////////////////////////////////////////////////////////
//
//  File:    cpu68000.cpp
//
//  Class:   CPU68000
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class serves as a base class for any 68000 CPU emulation cores.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#ifdef DEBUGGER
#include <stdio.h>
#endif

//  Application Headers.
#include "reptypes.h"
#include "cpu68000.h"
#ifdef DEBUGGER
#include "space.h"
#endif


///////////////////////////////////////////////////////////////////////////////
//  Static Member Data Initialization.
///////////////////////////////////////////////////////////////////////////////
#ifdef DEBUGGER
//  The following table is a listing of all the possible condition codes.
char* CPU68000::sm_astrCond[ 16 ] =
{
    "T ", "F ", "HI", "LS", "CC", "CS", "NE", "EQ", 
    "VC", "VS", "PL", "MI", "GE", "LT", "GT", "LE"
};
#endif



///////////////////////////////////////////////////////////////////////////////
//
//  Function: CPU68000
//
//  Description:
//
//      This is the main constructor for a 68000 CPU object.  It is protected
//      so that only derived classes have access to it.  Clients should
//      use the build method to create a CPU object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      pGame (input)
//          The game the CPU belongs to.
//
//      pSpace (input)
//          The address space that the CPU is primarily bound to.
//
//      ePurpose (input)
//          The purpose of the CPU.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
CPU68000::CPU68000(
    const KString&   iName,
    Game*            pGame,
    AddrSpace*       pSpace,
    const CPUPurpose ePurpose /* = GAME */
)
:
    CPU  ( iName, pGame, pSpace, ePurpose )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~CPU68000
//
//  Description:
//
//      This is the destructor for a 68000 CPU object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
CPU68000::~CPU68000(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getType
//
//  Description:
//
//      This member returns the type of CPU.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
CPU68000::getType(
) const
{
    //  The name of the class.
    static const KString cpuType( "68000" );

    return( cpuType );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getAddrSize
//
//  Description:
//
//      This member indicates the size of address space of the CPU.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      the size of the CPU's address space.
//
///////////////////////////////////////////////////////////////////////////////
DWord
CPU68000::getAddrSize(
) const
{
    return( 0x1000000 );
}





#ifdef DEBUGGER

///////////////////////////////////////////////////////////////////////////////
//  The following macros are helpful when disassembling.  They will retrieve
//  bytes from the current location to form words or double words, and
//  increment the PC appropriately.
///////////////////////////////////////////////////////////////////////////////
#define READ_OP(DEST)                                                         \
{                                                                             \
    Byte _b[2];                                                               \
    _b[0] = m_pSpace->getDecryptByte( rdwAddress + 0 );                       \
    _b[1] = m_pSpace->getDecryptByte( rdwAddress + 1 );                       \
    if( getClassName( ) == "CPU68000NCA" )                                    \
    {                                                                         \
        Byte _bTemp; _bTemp = _b[0] ; _b[0] = _b[1] ; _b[1] = _bTemp;         \
    }                                                                         \
    ( DEST ) = ( _b[0] << 8 ) + _b[1];                                        \
    rdwAddress += 2;                                                          \
    sprintf( strVals, "%02x %02x", _b[0], _b[1] );                            \
}
#define READ_WORD(DEST)                                                       \
{                                                                             \
    Byte _b[2];                                                               \
    _b[0] = m_pSpace->getByte( rdwAddress + 0 );                              \
    _b[1] = m_pSpace->getByte( rdwAddress + 1 );                              \
    if( getClassName( ) == "CPU68000NCA" )                                    \
    {                                                                         \
        Byte _bTemp; _bTemp = _b[0] ; _b[0] = _b[1] ; _b[1] = _bTemp;         \
    }                                                                         \
    ( DEST ) = ( _b[0] << 8 ) + _b[1];                                        \
    rdwAddress += 2;                                                          \
    sprintf( strVals, "%s %02x %02x", strVals, _b[0], _b[1] );                \
}
#define READ_DWORD(DEST)                                                      \
{                                                                             \
    Byte _b[4];                                                               \
    _b[0] = m_pSpace->getByte( rdwAddress + 0 );                              \
    _b[1] = m_pSpace->getByte( rdwAddress + 1 );                              \
    _b[2] = m_pSpace->getByte( rdwAddress + 2 );                              \
    _b[3] = m_pSpace->getByte( rdwAddress + 3 );                              \
    if( getClassName( ) == "CPU68000NCA" )                                    \
    {                                                                         \
        Byte _bTemp;                                                          \
        _bTemp = _b[0] ; _b[0] = _b[1] ; _b[1] = _bTemp;                      \
        _bTemp = _b[2] ; _b[2] = _b[3] ; _b[3] = _bTemp;                      \
    }                                                                         \
    ( DEST ) = ( _b[0] << 24 ) + ( _b[1] << 16 ) + ( _b[2] << 8 ) + _b[3];    \
    sprintf(                                                                  \
        strVals, "%s %02x %02x %02x %02x", strVals, _b[0], _b[1], _b[2], _b[3]\
    );                                                                        \
    rdwAddress += 4;                                                          \
}
#define MAKE_EA(VAL,SIZE) dbgMakeEA( VAL, rdwAddress, SIZE, strVals )
///////////////////////////////////////////////////////////////////////////////
//
//  Function: dbgMakeEA
//
//  Description:
//
//      This member is called to return the effective address based on 
//      the parameter specified.
//
//  Parameters:
//
//      nVal (input)
//          The value.
//
//      rdwAddress (input/output)
//          The address to consider.  It will be incremented as values
//          are read from memory.
//
//      nSize (input)
//          The size of the operation in bytes (1, 2, or 4).
//
//      strVals (input)
//          The buffer to keep track of the values read.
//
//  Returns:
//
//      A pointer to the string containing the effective address.
//
///////////////////////////////////////////////////////////////////////////////
char* 
CPU68000::dbgMakeEA( 
    int    nVal, 
    DWord& rdwAddress, 
    int    nSize,
    char*  strVals
)
{
    //  A double buffer and an index into the current one.
    static char astrBuffer[ 2 ][ 80 ];
    static int  nWhich = 0;

    //  A pointer to the current buffer.
    char* pstrBuf = astrBuffer[ nWhich ];

    //  The register.
    int nReg = nVal & 0x07;

    //  A parameter.
    DWord dwParam;

    //  A temporary variable.
    int nTemp;


    //  Change to the opposite buffer.
    nWhich ^= 1;

    //  Do it.
    switch( ( nVal >> 3 ) & 0x07 )
    {
        case 0:
            sprintf( pstrBuf, "D%d", nReg );
            break;
        case 1:
            sprintf( pstrBuf, "A%d", nReg );
            break;
        case 2:
            sprintf( pstrBuf, "(A%d)", nReg );
            break;
        case 3:
            sprintf( pstrBuf, "(A%d)+", nReg );
            break;
        case 4:
            sprintf( pstrBuf, "-(A%d)", nReg );
            break;
        case 5:
            READ_WORD( dwParam );
            if( dwParam & 0x8000 )
            {
                sprintf( 
                    pstrBuf, 
                    "(-$%04x,A%d)", 
                    -( signed short )dwParam & 0xffff, 
                    nReg 
                );
            }
            else
            {
                sprintf( pstrBuf, "($%04lx,A%d)", dwParam, nReg );
            }
            break;
        case 6:
            READ_WORD( dwParam );
            nTemp = dwParam & 0xff;
            if( nTemp & 0x80 )
            {
                sprintf(
                    pstrBuf,
                    "(-$%02x,A%d,D%ld.%c)",
                    -( signed char )nTemp & 0xff,
                    nReg,
                    ( dwParam >> 12 ) & 0x07,
                    ( dwParam & 800 ) ? 'L' : 'W'
                );
            }
            else
            {
                sprintf(
                    pstrBuf,
                    "($%02x,A%d,D%ld.%c)",
                    nTemp, 
                    nReg,
                    ( dwParam >> 12 ) & 0x07,
                    ( dwParam & 800 ) ? 'L' : 'W'
                );
            }
            break;
        case 7:
            switch( nReg )
            {
                case 0:
                    READ_WORD( dwParam );
                    sprintf( pstrBuf, "$%04lx", dwParam );
                    break;
                case 1:
                    READ_DWORD( dwParam );
                    sprintf( pstrBuf, "$%08lx", dwParam );
                    break;
                case 2:
                    READ_WORD( dwParam );
                    if( dwParam & 0x8000 )
                    {
                        sprintf( 
                            pstrBuf, 
                            "(-$%04x,PC)", 
                            -( signed short )dwParam & 0xffff 
                        );
                    }
                    else
                    {
                        sprintf( pstrBuf, "($%04lx,PC)", dwParam );
                    }
                    break;
                case 3:
                    READ_WORD( dwParam );
                    nTemp = dwParam & 0xff;
                    if( nTemp & 0x80 )
                    {
                        sprintf(
                            pstrBuf,
                            "(-$%02x,PC,D%ld.%c)",
                            -( signed char )nTemp & 0xff,
                            ( dwParam >> 12 ) & 0x07,
                            ( dwParam & 800 ) ? 'L' : 'W'
                        );
                    }
                    else
                    {
                        sprintf(
                            pstrBuf,
                            "($%02x,PC,D%ld.%c)",
                            nTemp,
                            ( dwParam >> 12 ) & 0x07,
                            ( dwParam & 800 ) ? 'L' : 'W'
                        );
                    }
                    break;
                case 4:
                    if( nSize == 1 )
                    {
                        READ_WORD( dwParam );
                        nTemp = dwParam & 0xff;
                        sprintf( pstrBuf, "#$%02x", nTemp );
                    }
                    else
                    if( nSize == 2 )
                    {
                        READ_WORD( dwParam );
                        sprintf( pstrBuf, "#$%04lx", dwParam );
                    }
                    else
                    {
                        READ_DWORD( dwParam );
                        sprintf( pstrBuf, "#$%08lx", dwParam );
                    }
                    break;
            }
            break;
    }

    return( pstrBuf );
}
           
          
              
///////////////////////////////////////////////////////////////////////////////
//
//  Function: dbgMakeRegList
//
//  Description:
//
//      This member is called to return a list of registers.
//
//  Parameters:
//
//      pstrBuf (input/output)
//          A pointer to the buffer to place the register list in.
//
//      wVal (input)
//          The value used to determine the register list.
//
//  Returns:
//
//      A pointer to the end of the buffer after the list has been stored.
//
///////////////////////////////////////////////////////////////////////////////
char* 
CPU68000::dbgMakeRegList( 
    char* pstrBuf, 
    Word  wVal 
)
{
    //  Work variables.
    int nStart = -1;
    int nSep   = 0;
    int nI;


    for( nI = 0 ; nI < 8 ; nI += 1, wVal >>= 1 )
    {
        if( ( wVal & 1 ) && ( nStart == -1 ) )
        {
            nStart = nI;
        }
        else
        if( !( wVal & 1 ) && ( nStart != -1 ) )
        {
            if( nSep++ )
            {
                pstrBuf += sprintf( pstrBuf, "/" );
            }
            if( nStart == nI - 1 )
            {
                pstrBuf += sprintf( pstrBuf, "D%d", nStart );
            }
            else
            {
                pstrBuf += sprintf( pstrBuf, "D%d-D%d", nStart, nI - 1 );
            }
            nStart = -1;
        }
    }
    if( nStart != -1 )
    {
        if( nSep++ )
        {
            pstrBuf += sprintf( pstrBuf, "/" );
        }
        if( nStart == 7 )
        {
            pstrBuf += sprintf( pstrBuf, "D7" );
        }
        else
        {
            pstrBuf += sprintf( pstrBuf, "D%d-D7", nStart );
        }
        nStart = -1;
    }

    for( nI = 0 ; nI < 8 ; nI += 1, wVal >>= 1 )
    {
        if( ( wVal & 1 ) && ( nStart == -1 ) )
        {
            nStart = nI;
        }
        else
        if( !( wVal & 1 ) && ( nStart != -1 ) )
        {
            if( nSep++ )
            {
                pstrBuf += sprintf( pstrBuf, "/" );
            }
            if( nStart == nI - 1 )
            {
                pstrBuf += sprintf( pstrBuf, "A%d", nStart );
            }
            else
            {
                pstrBuf += sprintf( pstrBuf, "A%d-A%d", nStart, nI - 1 );
            }
            nStart = -1;
        }
    }
    if( nStart != -1 )
    {
        if( nSep++ )
        {
            pstrBuf += sprintf( pstrBuf, "/" );
        }
        if( nStart == 7 )
        {
            pstrBuf += sprintf( pstrBuf, "A7" );
        }
        else
        {
            pstrBuf += sprintf( pstrBuf, "A%d-A7", nStart );
        }
        nStart = -1;
    }
    
    return( pstrBuf );
}
         

///////////////////////////////////////////////////////////////////////////////
//
//  Function: dbgMakeRevRegList
//
//  Description:
//
//      This member is called to return a reversed list of registers.
//
//  Parameters:
//
//      pstrBuf (input/output)
//          A pointer to the buffer to place the register list in.
//
//      wVal (input)
//          The value used to determine the register list.
//
//  Returns:
//
//      A pointer to the end of the buffer after the list has been stored.
//
///////////////////////////////////////////////////////////////////////////////
char* 
CPU68000::dbgMakeRevRegList( 
    char* pstrBuf, 
    Word  wVal 
)
{
    //  Work variables.
    int nStart = -1;
    int nSep   = 0;
    int nI;


    for( nI = 0 ; nI < 8 ; nI += 1, wVal <<= 1 )
    {
        if( ( wVal & 0x8000 ) && ( nStart == -1 ) )
        {
            nStart = nI;
        }
        else
        if( !( wVal & 0x8000 ) && ( nStart != -1 ) )
        {
            if( nSep++ )
            {
                pstrBuf += sprintf( pstrBuf, "/" );
            }
            if( nStart == nI - 1 )
            {
                pstrBuf += sprintf( pstrBuf, "D%d", nStart );
            }
            else
            {
                pstrBuf += sprintf( pstrBuf, "D%d-D%d", nStart, nI - 1 );
            }
            nStart = -1;
        }
    }
    if( nStart != -1 )
    {
        if( nSep++ )
        {
            pstrBuf += sprintf( pstrBuf, "/" );
        }
        if( nStart == 7 )
        {
            pstrBuf += sprintf( pstrBuf, "D7" );
        }
        else
        {
            pstrBuf += sprintf( pstrBuf, "D%d-D7", nStart );
        }
        nStart = -1;
    }

    for( nI = 0 ; nI < 8 ; nI += 1, wVal <<= 1 )
    {
        if( ( wVal & 0x8000 ) && ( nStart == -1 ) )
        {
            nStart = nI;
        }
        else
        if( !( wVal & 0x8000 ) && ( nStart != -1 ) )
        {
            if( nSep++ )
            {
                pstrBuf += sprintf( pstrBuf, "/" );
            }
            if( nStart == nI - 1 )
            {
                pstrBuf += sprintf( pstrBuf, "A%d", nStart );
            }
            else
            {
                pstrBuf += sprintf( pstrBuf, "A%d-A%d", nStart, nI - 1 );
            }
            nStart = -1;
        }
    }
    if( nStart != -1 )
    {
        if( nSep++ )
        {
            pstrBuf += sprintf( pstrBuf, "/" );
        }
        if( nStart == 7 )
        {
            pstrBuf += sprintf( pstrBuf, "A7" );
        }
        else
        {
            pstrBuf += sprintf( pstrBuf, "A%d-A7", nStart );
        }
        nStart = -1;
    }
    
    return( pstrBuf );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: dbgDisassemble
//
//  Description:
//
//      This member is called to disassemble the source at the specified
//      location and return it to the client.  The contents in the returned
//      buffer are valid until the next invocation of the function.
//
//      Thanks go to Aaron Giles for his disassembler from which this is 
//      based.  He claims it may not be the most efficient disassembler, but
//      hey, we really don't need efficiency here and it works so what the
//      hell.
//
//  Parameters:
//
//      rdwAddress (input/output)
//          The address to disassemble from.  It will contain the address of
//          the next instruction when the function completes.
//
//  Returns:
//
//      A pointer to the string containing the disassembled code.
//
///////////////////////////////////////////////////////////////////////////////
const
char*
CPU68000::dbgDisassemble(
    DWord& rdwAddress
)
{
    //  The following static buffer will contain the disassembled code line.
    static char strLine[ 128 ];

    //  Work buffers.
    char strAddr[ 20 ];
    char strVals[ 32 ];
    char strDasm[ 64 ];

    //  A couple of temporary pointers.
    char* pTemp1;
    char* pTemp2;

    //  An opcode.
    Word wOp;

    //  A parameter.
    DWord dwParam;

    //  Various helpers.
    Byte bLo;
    Byte bRegLo;
    Byte bRegHi;

    //  The original address passed in.
    DWord dwOrigAddress = rdwAddress;



    //  If the address is out of range then return nothing.
    if( rdwAddress > 0xffffff )
    {
        return( "" );
    }
    
    //  Initialize the buffers.
    sprintf( strAddr, "$%06lx", rdwAddress );
    strcpy( strVals, "" );
    strcpy( strDasm, "" );

    //  Read the opcode.
    READ_OP( wOp );

    //  Set the helper variables.
    bLo    = wOp & 0x3f;
    bRegHi = ( wOp >> 9 ) & 0x07;
    bRegLo = wOp & 0x07;
    

    //  Process the opcode.
    switch( wOp & 0xffc0 )
    {
        case 0x0000:
            READ_WORD( dwParam ); 
            if( bLo == 0x3c )
            {
                sprintf( strDasm, "ORI      #$%02lx,CCR", dwParam & 0xff );
            }
            else
            {
                sprintf(
                    strDasm,
                    "ORI.B    #$%02lx,%s",
                    dwParam & 0xff,
                    MAKE_EA( bLo, 1 )
                );
            }
            break;
        case 0x0040:
            READ_WORD( dwParam ); 
            if( bLo == 0x3c )
            {
                sprintf( strDasm, "ORI      #$%04lx,CCR", dwParam & 0xffff );
            }
            else
            {
                sprintf(
                    strDasm,
                    "ORI.W    #$%04lx,%s",
                    dwParam & 0xffff,
                    MAKE_EA( bLo, 2 )
                );
            }
            break;
        case 0x0080:
            READ_DWORD( dwParam ); 
            sprintf(
                strDasm, "ORI.L    #$%08lx,%s", dwParam, MAKE_EA( bLo, 4 )
            );
            break;
        case 0x0100: case 0x0300: case 0x0500: case 0x0700:
        case 0x0900: case 0x0b00: case 0x0d00: case 0x0f00:
            if( ( bLo & 0x38 ) == 0x08 )
            {
                READ_WORD( dwParam ); 
                sprintf(
                    strDasm, 
                    "MOVEP.W  ($%04lx,A%d),D%d", 
                    dwParam,
                    bRegLo,
                    bRegHi
                );
            }
            else
            {
                sprintf(
                    strDasm, 
                    "BTST     D%d,%s",
                    bRegHi,
                    MAKE_EA( bLo, 2 )
                );
            }
            break;
        case 0x0140: case 0x0340: case 0x0540: case 0x0740:
        case 0x0940: case 0x0b40: case 0x0d40: case 0x0f40:
            if( ( bLo & 0x38 ) == 0x08 )
            {
                READ_WORD( dwParam ); 
                sprintf(
                    strDasm, 
                    "MOVEP.L  ($%04lx,A%d),D%d", 
                    dwParam,
                    bRegLo,
                    bRegHi
                );
            }
            else
            {
                sprintf(
                    strDasm, 
                    "BCHG     D%d,%s",
                    bRegHi,
                    MAKE_EA( bLo, 2 )
                );
            }
            break;
        case 0x0180: case 0x0380: case 0x0580: case 0x0780: 
        case 0x0980: case 0x0b80: case 0x0d80: case 0x0f80: 
            if( ( bLo & 0x38 ) == 0x08 )
            {
                READ_WORD( dwParam ); 
                sprintf(
                    strDasm, 
                    "MOVEP.W  D%d,($%04lx,A%d)", 
                    bRegHi,
                    dwParam,
                    bRegLo
                );
            }
            else
            {
                sprintf(
                    strDasm, 
                    "BCLR     D%d,%s",
                    bRegHi,
                    MAKE_EA( bLo, 2 )
                );
            }
            break;
        case 0x01c0: case 0x03c0: case 0x05c0: case 0x07c0: 
        case 0x09c0: case 0x0bc0: case 0x0dc0: case 0x0fc0: 
            if( ( bLo & 0x38 ) == 0x08 )
            {
                READ_WORD( dwParam ); 
                sprintf(
                    strDasm, 
                    "MOVEP.L  D%d,($%04lx,A%d)", 
                    bRegHi,
                    dwParam,
                    bRegLo
                );
            }
            else
            {
                sprintf(
                    strDasm, 
                    "BSET     D%d,%s",
                    bRegHi,
                    MAKE_EA( bLo, 2 )
                );
            }
            break;
        case 0x0200:
            READ_WORD( dwParam );
            if( bLo == 0x3c )
            {
                sprintf( strDasm, "ANDI     #$%02lx,CCR", dwParam & 0xff );
            } 
            else
            {
                sprintf(
                    strDasm,
                    "ANDI.B   #$%02lx,%s",
                    dwParam & 0xff,
                    MAKE_EA( bLo, 1 )
                );
            }
            break;
        case 0x0240:
            READ_WORD( dwParam );
            if( bLo == 0x3c )
            {
                sprintf( strDasm, "ANDI     #$%04lx,SR", dwParam & 0xffff );
            } 
            else
            {
                sprintf(
                    strDasm,
                    "ANDI.W   #$%04lx,%s",
                    dwParam & 0xffff,
                    MAKE_EA( bLo, 2 )
                );
            }
            break;
        case 0x0280:
            READ_DWORD( dwParam );
            sprintf(
                strDasm, "ANDI.L   #$%08lx,%s", dwParam, MAKE_EA( bLo, 4 )
            );
            break;
        case 0x0400:
            READ_WORD( dwParam );
            sprintf(
                strDasm, 
                "SUBI.B   #$%02lx,%s", 
                dwParam & 0xff, 
                MAKE_EA( bLo, 1 )
            );
            break;
        case 0x0440:
            READ_WORD( dwParam );
            sprintf(
                strDasm, 
                "SUBI.W   #$%04lx,%s", 
                dwParam & 0xffff,
                MAKE_EA( bLo, 2 )
            );
            break;
        case 0x0480:
            READ_DWORD( dwParam );
            sprintf(
                strDasm, "SUBI.L   #$%08lx,%s", dwParam, MAKE_EA( bLo, 4 )
            );
            break;
        case 0x0600:
            READ_WORD( dwParam );
            sprintf(
                strDasm, 
                "ADDI.B   #$%02lx,%s", 
                dwParam & 0xff, 
                MAKE_EA( bLo, 1 )
            );
            break;
        case 0x0640:
            READ_WORD( dwParam );
            sprintf(
                strDasm, 
                "ADDI.W   #$%04lx,%s", 
                dwParam & 0xffff,
                MAKE_EA( bLo, 2 )
            );
            break;
        case 0x0680:
            READ_DWORD( dwParam );
            sprintf(
                strDasm, "ADDI.L   #$%08lx,%s", dwParam, MAKE_EA( bLo, 4 )
            );
            break;
        case 0x0800:
            READ_WORD( dwParam );
            sprintf(
                strDasm, 
                "BTST     #$%02lx,%s", 
                dwParam & 0xff, 
                MAKE_EA( bLo, 2 )
            );
            break;
        case 0x0840:
            READ_WORD( dwParam );
            sprintf(
                strDasm, 
                "BCHG     #$%02lx,%s", 
                dwParam & 0xff, 
                MAKE_EA( bLo, 2 )
            );
            break;
        case 0x0880:
            READ_WORD( dwParam );
            sprintf(
                strDasm, 
                "BCLR     #$%02lx,%s", 
                dwParam & 0xff, 
                MAKE_EA( bLo, 2 )
            );
            break;
        case 0x08c0:
            READ_WORD( dwParam );
            sprintf(
                strDasm, 
                "BSET     #$%02lx,%s", 
                dwParam & 0xff, 
                MAKE_EA( bLo, 2 )
            );
            break;
        case 0x0a00:
            READ_WORD( dwParam );
            if( bLo == 0x3c )
            {
                sprintf( strDasm, "EORI     #$%02lx,CCR", dwParam & 0xff );
            }
            else
            {
                sprintf(
                    strDasm, 
                    "EORI.B   #$%02lx,%s", 
                    dwParam & 0xff, 
                    MAKE_EA( bLo, 1 )
                );
            }
            break;
        case 0x0a40:
            READ_WORD( dwParam );
            if( bLo == 0x3c )
            {
                sprintf( strDasm, "EORI     #$%04lx,CCR", dwParam & 0xffff );
            }
            else
            {
                sprintf(
                    strDasm, 
                    "EORI.W   #$%04lx,%s", 
                    dwParam & 0xffff, 
                    MAKE_EA( bLo, 2 )
                );
            }
            break;
        case 0x0a80:
            READ_DWORD( dwParam );
            sprintf(
                strDasm, "EORI.L   #$%08lx,%s", dwParam, MAKE_EA( bLo, 4 )
            );
            break;
        case 0x0c00:
            READ_WORD( dwParam );
            sprintf(
                strDasm, 
                "CMPI.B   #$%02lx,%s", 
                dwParam & 0xff, 
                MAKE_EA( bLo, 1 )
            );
            break;
        case 0x0c40:
            READ_WORD( dwParam );
            sprintf(
                strDasm, 
                "CMPI.W   #$%04lx,%s", 
                dwParam & 0xffff, 
                MAKE_EA( bLo, 2 )
            );
            break;
        case 0x0c80:
            READ_DWORD( dwParam );
            sprintf(
                strDasm, "CMPI.L   #$%08lx,%s", dwParam, MAKE_EA( bLo, 4 )
            );
            break;
        case 0xe00:
            READ_WORD( dwParam );
            if( dwParam & 0x0800 )
            { 
                sprintf(
                    strDasm, 
                    "MOVES.B  %c%ld,%s",
                    dwParam & 0x8000 ? 'A' : 'D',
                    ( dwParam >> 12 ) & 0x07,
                    MAKE_EA( bLo, 1 )
                );
            }
            else
            {
                sprintf(
                    strDasm, 
                    "MOVES.B  %s,%c%ld",
                    MAKE_EA( bLo, 1 ),
                    dwParam & 0x8000 ? 'A' : 'D',
                    ( dwParam >> 12 ) & 0x07
                );
            }
            break;
        case 0x0e40:
            READ_WORD( dwParam );
            if( dwParam & 0x0800 )
            { 
                sprintf(
                    strDasm, 
                    "MOVES.W  %c%ld,%s",
                    dwParam & 0x8000 ? 'A' : 'D',
                    ( dwParam >> 12 ) & 0x07,
                    MAKE_EA( bLo, 2 )
                );
            }
            else
            {
                sprintf(
                    strDasm, 
                    "MOVES.W  %s,%c%ld",
                    MAKE_EA( bLo, 2 ),
                    dwParam & 0x8000 ? 'A' : 'D',
                    ( dwParam >> 12 ) & 0x07
                );
            }
            break;
        case 0x0e80:
            READ_WORD( dwParam );
            if( dwParam & 0x0800 )
            { 
                sprintf(
                    strDasm, 
                    "MOVES.L  %c%ld,%s",
                    dwParam & 0x8000 ? 'A' : 'D',
                    ( dwParam >> 12 ) & 0x07,
                    MAKE_EA( bLo, 4 )
                );
            }
            else
            {
                sprintf(
                    strDasm, 
                    "MOVES.L  %s,%c%ld",
                    MAKE_EA( bLo, 4 ),
                    dwParam & 0x8000 ? 'A' : 'D',
                    ( dwParam >> 12 ) & 0x07
                );
            }
            break;
        case 0x1000: case 0x1080: case 0x10c0:
        case 0x1100: case 0x1140: case 0x1180: case 0x11c0:
        case 0x1200: case 0x1280: case 0x12c0:
        case 0x1300: case 0x1340: case 0x1380: case 0x13c0:
        case 0x1400: case 0x1480: case 0x14c0:
        case 0x1500: case 0x1540: case 0x1580:
        case 0x1600: case 0x1680: case 0x16c0:
        case 0x1700: case 0x1740: case 0x1780:
        case 0x1800: case 0x1880: case 0x18c0:
        case 0x1900: case 0x1940: case 0x1980:
        case 0x1a00: case 0x1a80: case 0x1ac0:
        case 0x1b00: case 0x1b40: case 0x1b80:
        case 0x1c00: case 0x1c80: case 0x1cc0:
        case 0x1d00: case 0x1d40: case 0x1d80:
        case 0x1e00: case 0x1e80: case 0x1ec0:
        case 0x1f00: case 0x1f40: case 0x1f80:
            pTemp1 = MAKE_EA( bLo, 1 );
            pTemp2 = 
                MAKE_EA( ( ( wOp >> 9 ) & 0x07 ) + ( ( wOp >> 3 ) & 0x38 ), 1 );
            sprintf( strDasm, "MOVE.B   %s,%s", pTemp1, pTemp2 );
            break;
        case 0x2000: case 0x2080: case 0x20c0:
        case 0x2100: case 0x2140: case 0x2180: case 0x21c0:
        case 0x2200: case 0x2280: case 0x22c0:
        case 0x2300: case 0x2340: case 0x2380: case 0x23c0:
        case 0x2400: case 0x2480: case 0x24c0:
        case 0x2500: case 0x2540: case 0x2580:
        case 0x2600: case 0x2680: case 0x26c0:
        case 0x2700: case 0x2740: case 0x2780:
        case 0x2800: case 0x2880: case 0x28c0:
        case 0x2900: case 0x2940: case 0x2980:
        case 0x2a00: case 0x2a80: case 0x2ac0:
        case 0x2b00: case 0x2b40: case 0x2b80:
        case 0x2c00: case 0x2c80: case 0x2cc0:
        case 0x2d00: case 0x2d40: case 0x2d80:
        case 0x2e00: case 0x2e80: case 0x2ec0:
        case 0x2f00: case 0x2f40: case 0x2f80:
            pTemp1 = MAKE_EA( bLo, 4 );
            pTemp2 = 
                MAKE_EA( ( ( wOp >> 9 ) & 0x07 ) + ( ( wOp >> 3 ) & 0x38 ), 4 );
            sprintf( strDasm, "MOVE.L   %s,%s", pTemp1, pTemp2 );
            break;
        case 0x2040: case 0x2240: case 0x2440: case 0x2640:
        case 0x2840: case 0x2a40: case 0x2c40: case 0x2e40:
            sprintf( strDasm, "MOVEA.L  %s,A%d", MAKE_EA( bLo, 4 ), bRegHi );
            break;
        case 0x3000: case 0x3080: case 0x30c0:
        case 0x3100: case 0x3140: case 0x3180: case 0x31c0:
        case 0x3200: case 0x3280: case 0x32c0:
        case 0x3300: case 0x3340: case 0x3380: case 0x33c0:
        case 0x3400: case 0x3480: case 0x34c0:
        case 0x3500: case 0x3540: case 0x3580:
        case 0x3600: case 0x3680: case 0x36c0:
        case 0x3700: case 0x3740: case 0x3780:
        case 0x3800: case 0x3880: case 0x38c0:
        case 0x3900: case 0x3940: case 0x3980:
        case 0x3a00: case 0x3a80: case 0x3ac0:
        case 0x3b00: case 0x3b40: case 0x3b80:
        case 0x3c00: case 0x3c80: case 0x3cc0:
        case 0x3d00: case 0x3d40: case 0x3d80:
        case 0x3e00: case 0x3e80: case 0x3ec0:
        case 0x3f00: case 0x3f40: case 0x3f80:
            pTemp1 = MAKE_EA( bLo, 2 );
            pTemp2 =  
                MAKE_EA( ( ( wOp >> 9 ) & 0x07 ) + ( ( wOp >> 3 ) & 0x38 ), 2 );
            sprintf( strDasm, "MOVE.W   %s,%s", pTemp1, pTemp2 );
            break;
        case 0x3040: case 0x3240: case 0x3440: case 0x3640:
        case 0x3840: case 0x3a40: case 0x3c40: case 0x3e40:
            sprintf( strDasm, "MOVEA.W  %s,A%d", MAKE_EA( bLo, 2 ), bRegHi );
            break;
        case 0x4000:
            sprintf( strDasm, "NEGX.B   %s", MAKE_EA( bLo, 1 ) );
            break;
        case 0x4040:
            sprintf( strDasm, "NEGX.W   %s", MAKE_EA( bLo, 2 ) );
            break;
        case 0x4080:
            sprintf( strDasm, "NEGX.L   %s", MAKE_EA( bLo, 4 ) );
            break;
        case 0x4140: case 0x4380: case 0x4580: case 0x4780:
        case 0x4980: case 0x4b80: case 0x4d80: case 0x4f80:
            sprintf( strDasm, "CHK.W    %s,D%d", MAKE_EA( bLo, 2 ), bRegHi );
            break;
        case 0x41c0: case 0x43c0: case 0x45c0: case 0x47c0:
        case 0x49c0: case 0x4bc0: case 0x4dc0: case 0x4fc0:
            sprintf( strDasm, "LEA      %s,A%d", MAKE_EA( bLo, 4 ), bRegHi );
            break;
        case 0x4200:
            sprintf( strDasm, "CLR.B    %s", MAKE_EA( bLo, 1 ) );
            break;
        case 0x4240:
            sprintf( strDasm, "CLR.W    %s", MAKE_EA( bLo, 2 ) );
            break;
        case 0x4280:
            sprintf( strDasm, "CLR.L    %s", MAKE_EA( bLo, 4 ) );
            break;
        case 0x42c0:
            sprintf( strDasm, "MOVE     CCR,%s", MAKE_EA( bLo, 2 ) );
            break;
        case 0x4400:
            sprintf( strDasm, "NEG.B    %s", MAKE_EA( bLo, 1 ) );
            break;
        case 0x4440:
            sprintf( strDasm, "NEG.W    %s", MAKE_EA( bLo, 2 ) );
            break;
        case 0x4480:
            sprintf( strDasm, "NEG.L    %s", MAKE_EA( bLo, 4 ) );
            break;
        case 0x44c0:
            sprintf( strDasm, "MOVE     %s,CCR", MAKE_EA( bLo, 2 ) );
            break;
        case 0x4600:
            sprintf( strDasm, "NOT.B    %s", MAKE_EA( bLo, 1 ) );
            break;
        case 0x4640:
            sprintf( strDasm, "NOT.W    %s", MAKE_EA( bLo, 2 ) );
            break;
        case 0x4680:
            sprintf( strDasm, "NOT.L    %s", MAKE_EA( bLo, 4 ) );
            break;
        case 0x46c0:
            sprintf( strDasm, "MOVE     %s,SR", MAKE_EA( bLo, 2 ) );
            break;
        case 0x4800:
            sprintf( strDasm, "NBCD.B   %s", MAKE_EA( bLo, 1 ) );
            break;
        case 0x4840:
            if( ( bLo & 0x38 ) == 0x00 )
            {
                sprintf( strDasm, "SWAP     D%d", bRegLo );
            }
            else
            {
                sprintf( strDasm, "PEA      %s", MAKE_EA( bLo, 4 ) );
            }
            break;
        case 0x4880:
            if( ( bLo & 0x38 ) == 0x00 )
            {
                sprintf( strDasm, "EXT.W    D%d", bRegLo );
            }
            else
            {
                pTemp1 = &( strDasm[ 0 ] );
                READ_WORD( dwParam );
                pTemp1 += sprintf( pTemp1, "MOVEM.W  " );
                if( ( bLo & 0x38 ) != 0x20 )
                {
                    pTemp1 = dbgMakeRegList( pTemp1, dwParam );
                }
                else
                {
                    pTemp1 = dbgMakeRevRegList( pTemp1, dwParam );
                }
                sprintf( pTemp1, ",%s", MAKE_EA( bLo, 2 ) );
            }
            break;
        case 0x48c0: 
            if( ( bLo & 0x38 ) == 0x00 )
            {
                sprintf( strDasm, "EXT.L    D%d", bRegLo );
            }
            else
            {
                pTemp1 = &( strDasm[ 0 ] );
                READ_WORD( dwParam );
                pTemp1 += sprintf( pTemp1, "MOVEM.L  " );
                if( ( bLo & 0x38 ) != 0x20 )
                {
                    pTemp1 = dbgMakeRegList( pTemp1, dwParam );
                }
                else
                {
                    pTemp1 = dbgMakeRevRegList( pTemp1, dwParam );
                }
                sprintf( pTemp1, ",%s", MAKE_EA( bLo, 4 ) );
            }
            break;
        case 0x4a00:
            sprintf( strDasm, "TST.B    %s", MAKE_EA( bLo, 1 ) );
            break;
        case 0x4a40:
            sprintf( strDasm, "TST.W    %s", MAKE_EA( bLo, 2 ) );
            break;
        case 0x4a80:
            sprintf( strDasm, "TST.L    %s", MAKE_EA( bLo, 4 ) );
            break;
        case 0x4ac0:
            sprintf( strDasm, "TAS.B    %s", MAKE_EA( bLo, 1 ) );
            break;
        case 0x4c80: 
            pTemp1 = &( strDasm[ 0 ] );
            READ_WORD( dwParam );
            pTemp1 += sprintf( pTemp1, "MOVEM.W  %s,", MAKE_EA( bLo, 2 ) );
            dbgMakeRegList( pTemp1, dwParam );
            break;
        case 0x4cc0: 
            pTemp1 = &( strDasm[ 0 ] );
            READ_WORD( dwParam );
            pTemp1 += sprintf( pTemp1, "MOVEM.L  %s,", MAKE_EA( bLo, 4 ) );
            dbgMakeRegList( pTemp1, dwParam );
            break;
        case 0x4e40:
            if( ( bLo & 30 ) == 0x00 )
            {
                sprintf( strDasm, "TRAP     #$%02x", bLo & 0x0f );
            }
            else
            if( ( bLo & 0x38 ) == 0x10 )
            {
                READ_WORD( dwParam );
                sprintf( strDasm, "LINK     A%d,#$%02lx", bRegLo, dwParam );
            }
            else
            if( ( bLo & 0x38 ) == 0x18 )
            {
                READ_WORD( dwParam );
                sprintf( strDasm, "UNLK     A%d,#$%02lx", bRegLo, dwParam );
            }
            else
            if( ( bLo & 0x38 ) == 0x20 )
            {
                sprintf( strDasm, "MOVE     A%d,USP", bRegLo );
            }
            else
            if( ( bLo & 0x38 ) == 0x28 )
            {
                sprintf( strDasm, "MOVE     USP,A%d", bRegLo );
            }
            else
            if( bLo == 0x30 )
            {
                sprintf( strDasm, "RESET" );
            }
            else
            if( bLo == 0x31 )
            {
                sprintf( strDasm, "NOP" );
            }
            else
            if( bLo == 0x32 )
            {
                sprintf( strDasm, "STOP" );
            }
            else
            if( bLo == 0x33 )
            {
                sprintf( strDasm, "RTE" );
            }
            else
            if( bLo == 0x35 )
            {
                sprintf( strDasm, "RTS" );
            }
            else
            if( bLo == 0x36 )
            {
                sprintf( strDasm, "TRAPV" );
            }
            else
            if( bLo == 0x37 )
            {
                sprintf( strDasm, "RTR" );
            }
            else
            if( bLo == 0x3a )
            {
                READ_WORD( dwParam );
                switch( dwParam & 0xfff )
                {
                    case 0x000: pTemp1 = "SFC"; break;
                    case 0x001: pTemp1 = "DFC"; break;
                    case 0x800: pTemp1 = "USP"; break;
                    case 0x801: pTemp1 = "VBR"; break;
                    default:    pTemp1 = "???"; break;
                }
                sprintf(
                    strDasm,
                    "MOVEC    %s,%c%ld",
                    pTemp1,
                    dwParam & 0x8000 ? 'A' : 'D',
                    ( dwParam >> 12 ) & 0x07
                );
            }
            else
            if( bLo == 0x3b )
            {
                READ_WORD( dwParam );
                switch( dwParam & 0xfff )
                {
                    case 0x000: pTemp1 = "SFC"; break;
                    case 0x001: pTemp1 = "DFC"; break;
                    case 0x800: pTemp1 = "USP"; break;
                    case 0x801: pTemp1 = "VBR"; break;
                    default:    pTemp1 = "???"; break;
                }
                sprintf(
                    strDasm,
                    "MOVEC    %c%ld,%s",
                    dwParam & 0x8000 ? 'A' : 'D',
                    ( dwParam >> 12 ) & 0x07,
                    pTemp1
                );
            }
            else
            {
                sprintf( strDasm, "DC.W     $%04x", wOp );
            }
            break;
        case 0x4e80:
            sprintf( strDasm, "JSR      %s", MAKE_EA( bLo, 4 ) );
            break;
        case 0x4ec0:
            sprintf( strDasm, "JMP      %s", MAKE_EA( bLo, 4 ) );
            break;
        case 0x5000: case 0x5200: case 0x5400: case 0x5600:
        case 0x5800: case 0x5a00: case 0x5c00: case 0x5e00:
            sprintf( 
                strDasm, 
                "ADDQ.B   #%d,%s", 
                ( bRegHi == 0 ) ? 8 : bRegHi,
                MAKE_EA( bLo, 1 )
            );
            break;
        case 0x5040: case 0x5240: case 0x5440: case 0x5640:
        case 0x5840: case 0x5a40: case 0x5c40: case 0x5e40:
            sprintf( 
                strDasm, 
                "ADDQ.W   #%d,%s", 
                ( bRegHi == 0 ) ? 8 : bRegHi,
                MAKE_EA( bLo, 2 )
            );
            break;
        case 0x5080: case 0x5280: case 0x5480: case 0x5680:
        case 0x5880: case 0x5a80: case 0x5c80: case 0x5e80:
            sprintf( 
                strDasm, 
                "ADDQ.L   #%d,%s", 
                ( bRegHi == 0 ) ? 8 : bRegHi,
                MAKE_EA( bLo, 4 )
            );
            break;
        case 0x50c0: case 0x52c0: case 0x54c0: case 0x56c0:
        case 0x58c0: case 0x5ac0: case 0x5cc0: case 0x5ec0:
        case 0x51c0: case 0x53c0: case 0x55c0: case 0x57c0:
        case 0x59c0: case 0x5bc0: case 0x5dc0: case 0x5fc0:
            if( ( bLo & 0x38 ) == 0x08 )
            {
                READ_WORD( dwParam );
                if( dwParam & 0x8000 )
                {
                    sprintf( 
                        strDasm, 
                        "DB%s     D%d,*-$%04x [%08lx]",
                        sm_astrCond[ ( wOp >> 8 ) & 0x0f ],
                        bRegLo,
                        ( int )( -( signed short )dwParam ) - 2,
                        dwOrigAddress + ( signed short )dwParam + 2
                    );
                }
                else
                {
                    sprintf( 
                        strDasm, 
                        "DB%s     D%d,*+$%04lx [%08lx]",
                        sm_astrCond[ ( wOp >> 8 ) & 0x0f ],
                        bRegLo,
                        dwParam - 2,
                        dwOrigAddress + dwParam + 2
                    );
                }
            }
            else
            {
                sprintf( 
                    strDasm, 
                    "S%s.B    %s", 
                    sm_astrCond[ ( wOp >> 8 ) & 0x0f ],
                    MAKE_EA( bLo, 1 )
                );
            }
            break;
        case 0x5100: case 0x5300: case 0x5500: case 0x5700:
        case 0x5900: case 0x5b00: case 0x5d00: case 0x5f00:
            sprintf( 
                strDasm,
                "SUBQ.B   #%d,%s",
                ( bRegHi == 0 ) ? 8 : bRegHi,
                MAKE_EA( bLo, 1 )
            );
            break;
        case 0x5140: case 0x5340: case 0x5540: case 0x5740:
        case 0x5940: case 0x5b40: case 0x5d40: case 0x5f40:
            sprintf( 
                strDasm,
                "SUBQ.W   #%d,%s",
                ( bRegHi == 0 ) ? 8 : bRegHi,
                MAKE_EA( bLo, 2 )
            );
            break;
        case 0x5180: case 0x5380: case 0x5580: case 0x5780:
        case 0x5980: case 0x5b80: case 0x5d80: case 0x5f80:
            sprintf( 
                strDasm,
                "SUBQ.L   #%d,%s",
                ( bRegHi == 0 ) ? 8 : bRegHi,
                MAKE_EA( bLo, 4 )
            );
            break;
        case 0x6000: case 0x6040: case 0x6080: case 0x60c0:
            dwParam = wOp & 0xff;
            if( dwParam == 0 )
            {
                READ_WORD( dwParam )
                if( dwParam & 0x8000 )
                {
                    sprintf(
                        strDasm,
                        "BRA      *-$%04x [%08lx]",
                        ( int )( -( signed short )dwParam ) - 2,
                        dwOrigAddress + ( signed short )dwParam + 2
                    );
                }
                else
                {
                    sprintf(
                        strDasm,
                        "BRA      *+$%04lx [%08lx]",
                        dwParam + 2,
                        dwOrigAddress + dwParam + 2
                    );
                }
            }
            else
            {
                if( dwParam & 0x80 )
                {
                    sprintf(
                        strDasm,
                        "BRA.S    *-$%02x [%08lx]",
                        ( int )( -( signed char )dwParam ) - 2,
                        dwOrigAddress + ( signed char )dwParam + 2
                    );
                }
                else
                {
                    sprintf(
                        strDasm,
                        "BRA.S    *+$%02lx [%08lx]",
                        dwParam + 2,
                        dwOrigAddress + dwParam + 2
                    );
                }
            }
            break;
        case 0x6100: case 0x6140: case 0x6180: case 0x61c0:
            dwParam = wOp & 0xff;
            if( dwParam == 0 )
            {
                READ_WORD( dwParam )
                if( dwParam & 0x8000 )
                {
                    sprintf(
                        strDasm,
                        "BSR      *-$%04x [%08lx]",
                        ( int )( -( signed short )dwParam ) - 2,
                        dwOrigAddress + ( signed short )dwParam + 2
                    );
                }
                else
                {
                    sprintf(
                        strDasm,
                        "BSR      *+$%04lx [%08lx]",
                        dwParam + 2,
                        dwOrigAddress + dwParam + 2
                    );
                }
            }
            else
            {
                if( dwParam & 0x80 )
                {
                    sprintf(
                        strDasm,
                        "BSR.S    *-$%02x [%08lx]",
                        ( int )( -( signed char )dwParam ) - 2,
                        dwOrigAddress + ( signed char )dwParam + 2
                    );
                }
                else
                {
                    sprintf(
                        strDasm,
                        "BSR.S    *+$%02lx [%08lx]",
                        dwParam + 2,
                        dwOrigAddress + dwParam + 2
                    );
                }
            }
            break;
        case 0x6200: case 0x6240: case 0x6280: case 0x62c0:
        case 0x6300: case 0x6340: case 0x6380: case 0x63c0:
        case 0x6400: case 0x6440: case 0x6480: case 0x64c0:
        case 0x6500: case 0x6540: case 0x6580: case 0x65c0:
        case 0x6600: case 0x6640: case 0x6680: case 0x66c0:
        case 0x6700: case 0x6740: case 0x6780: case 0x67c0:
        case 0x6800: case 0x6840: case 0x6880: case 0x68c0:
        case 0x6900: case 0x6940: case 0x6980: case 0x69c0:
        case 0x6a00: case 0x6a40: case 0x6a80: case 0x6ac0:
        case 0x6b00: case 0x6b40: case 0x6b80: case 0x6bc0:
        case 0x6c00: case 0x6c40: case 0x6c80: case 0x6cc0:
        case 0x6d00: case 0x6d40: case 0x6d80: case 0x6dc0:
        case 0x6e00: case 0x6e40: case 0x6e80: case 0x6ec0:
        case 0x6f00: case 0x6f40: case 0x6f80: case 0x6fc0:
            dwParam = wOp & 0xff;
            if( dwParam == 0 )
            {
                READ_WORD( dwParam )
                if( dwParam & 0x8000 )
                {
                    sprintf(
                        strDasm,
                        "B%s      *-$%04x [%08lx]",
                        sm_astrCond[ ( wOp >> 8 ) & 0x0f ],
                        ( int )( -( signed short )dwParam ) - 2,
                        dwOrigAddress + ( signed short )dwParam + 2
                    );
                }
                else
                {
                    sprintf(
                        strDasm,
                        "B%s      *+$%04lx [%08lx]",
                        sm_astrCond[ ( wOp >> 8 ) & 0x0f ],
                        dwParam + 2,
                        dwOrigAddress + dwParam + 2
                    );
                }
            }
            else
            {
                if( dwParam & 0x80 )
                {
                    sprintf(
                        strDasm,
                        "B%s.S    *-$%02x [%08lx]",
                        sm_astrCond[ ( wOp >> 8 ) & 0x0f ],
                        ( int )( -( signed char )dwParam ) - 2,
                        dwOrigAddress + ( signed char )dwParam + 2
                    );
                }
                else
                {
                    sprintf(
                        strDasm,
                        "B%s.S    *+$%02lx [%08lx]",
                        sm_astrCond[ ( wOp >> 8 ) & 0x0f ],
                        dwParam + 2,
                        dwOrigAddress + dwParam + 2
                    );
                }
            }
            break;
        case 0x7000: case 0x7040: case 0x7080: case 0x70c0:
        case 0x7200: case 0x7240: case 0x7280: case 0x72c0:
        case 0x7400: case 0x7440: case 0x7480: case 0x74c0:
        case 0x7600: case 0x7640: case 0x7680: case 0x76c0:
        case 0x7800: case 0x7840: case 0x7880: case 0x78c0:
        case 0x7a00: case 0x7a40: case 0x7a80: case 0x7ac0:
        case 0x7c00: case 0x7c40: case 0x7c80: case 0x7cc0:
        case 0x7e00: case 0x7e40: case 0x7e80: case 0x7ec0:
            dwParam = wOp & 0xff;
            if( dwParam & 0x80 )
            {
                sprintf(
                    strDasm,
                    "MOVEQ    #$-%02x,D%d",
                    -( signed char )dwParam, 
                    bRegHi
                );
            }
            else
            {
                sprintf( strDasm, "MOVEQ    #$%02lx,D%d", dwParam, bRegHi );
            }
            break;
        case 0x8000: case 0x8200: case 0x8400: case 0x8600:
        case 0x8800: case 0x8a00: case 0x8c00: case 0x8e00:
            sprintf( strDasm, "OR.B     %s,D%d", MAKE_EA( bLo, 1 ), bRegHi );
            break;
        case 0x8040: case 0x8240: case 0x8440: case 0x8640:
        case 0x8840: case 0x8a40: case 0x8c40: case 0x8e40:
            sprintf( strDasm, "OR.W     %s,D%d", MAKE_EA( bLo, 2 ), bRegHi );
            break;
        case 0x8080: case 0x8280: case 0x8480: case 0x8680:
        case 0x8880: case 0x8a80: case 0x8c80: case 0x8e80:
            sprintf( strDasm, "OR.L     %s,D%d", MAKE_EA( bLo, 4 ), bRegHi );
            break;
        case 0x80c0: case 0x82c0: case 0x84c0: case 0x86c0:
        case 0x88c0: case 0x8ac0: case 0x8cc0: case 0x8ec0:
            sprintf( strDasm, "DIVU.W   %s,D%d", MAKE_EA( bLo, 2 ), bRegHi );
            break;
        case 0x8100: case 0x8300: case 0x8500: case 0x8700:
        case 0x8900: case 0x8b00: case 0x8d00: case 0x8f00:
            sprintf( strDasm, "OR.B     D%d,%s", bRegHi, MAKE_EA( bLo, 1 ) );
            break;
        case 0x8140: case 0x8340: case 0x8540: case 0x8740:
        case 0x8940: case 0x8b40: case 0x8d40: case 0x8f40:
            sprintf( strDasm, "OR.W     D%d,%s", bRegHi, MAKE_EA( bLo, 2 ) );
            break;
        case 0x8180: case 0x8380: case 0x8580: case 0x8780:
        case 0x8980: case 0x8b80: case 0x8d80: case 0x8f80:
            sprintf( strDasm, "OR.L     D%d,%s", bRegHi, MAKE_EA( bLo, 4 ) );
            break;
        case 0x81c0: case 0x83c0: case 0x85c0: case 0x87c0:
        case 0x89c0: case 0x8bc0: case 0x8dc0: case 0x8fc0:
            sprintf( strDasm, "DIVS.W   %s,D%d", MAKE_EA( bLo, 2 ), bRegHi );
            break;
        case 0x9000: case 0x9200: case 0x9400: case 0x9600:
        case 0x9800: case 0x9a00: case 0x9c00: case 0x9e00:
            sprintf( strDasm, "SUB.B    %s,D%d", MAKE_EA( bLo, 1 ), bRegHi );
            break;
        case 0x9040: case 0x9240: case 0x9440: case 0x9640:
        case 0x9840: case 0x9a40: case 0x9c40: case 0x9e40:
            sprintf( strDasm, "SUB.W    %s,D%d", MAKE_EA( bLo, 2 ), bRegHi );
            break;
        case 0x9080: case 0x9280: case 0x9480: case 0x9680:
        case 0x9880: case 0x9a80: case 0x9c80: case 0x9e80:
            sprintf( strDasm, "SUB.L    %s,D%d", MAKE_EA( bLo, 4 ), bRegHi );
            break;
        case 0x90c0: case 0x92c0: case 0x94c0: case 0x96c0:
        case 0x98c0: case 0x9ac0: case 0x9cc0: case 0x9ec0:
            sprintf( strDasm, "SUBA.W   %s,A%d", MAKE_EA( bLo, 2 ), bRegHi );
            break;
        case 0x9100: case 0x9300: case 0x9500: case 0x9700:
        case 0x9900: case 0x9b00: case 0x9d00: case 0x9f00:
            if( ( bLo & 0x30 ) == 0 )
            {
                if( bLo & 0x08 )
                {
                    sprintf( 
                        strDasm, "SUBX.B   -(A%d),-(A%d)", bRegLo, bRegHi 
                    );
                }
                else
                {
                    sprintf( strDasm, "SUBX.B   D%d,D%d", bRegLo, bRegHi );
                }
            }
            else
            {
                sprintf( 
                    strDasm, "SUB.B    D%d,%s", bRegHi, MAKE_EA( bLo, 1 ) 
                );
            }
            break;
        case 0x9140: case 0x9340: case 0x9540: case 0x9740:
        case 0x9940: case 0x9b40: case 0x9d40: case 0x9f40:
            if( ( bLo & 0x30 ) == 0 )
            {
                if( bLo & 0x08 )
                {
                    sprintf( 
                        strDasm, "SUBX.W   -(A%d),-(A%d)", bRegLo, bRegHi 
                    );
                }
                else
                {
                    sprintf( strDasm, "SUBX.W   D%d,D%d", bRegLo, bRegHi );
                }
            }
            else
            {
                sprintf( 
                    strDasm, "SUB.W    D%d,%s", bRegHi, MAKE_EA( bLo, 2 ) 
                );
            }
            break; 
        case 0x9180: case 0x9380: case 0x9580: case 0x9780:
        case 0x9980: case 0x9b80: case 0x9d80: case 0x9f80:
            if( ( bLo & 0x30 ) == 0 )
            {
                if( bLo & 0x08 )
                {
                    sprintf( 
                        strDasm, "SUBX.L   -(A%d),-(A%d)", bRegLo, bRegHi 
                    );
                }
                else
                {
                    sprintf( strDasm, "SUBX.L   D%d,D%d", bRegLo, bRegHi );
                }
            }
            else
            {
                sprintf( 
                    strDasm, "SUB.L    D%d,%s", bRegHi, MAKE_EA( bLo, 4 ) 
                );
            }
            break; 
        case 0x91c0: case 0x93c0: case 0x95c0: case 0x97c0:
        case 0x99c0: case 0x9bc0: case 0x9dc0: case 0x9fc0:
            sprintf( strDasm, "SUBA.L   %s,A%d", MAKE_EA( bLo, 4 ), bRegHi );
            break;
        case 0xb000: case 0xb200: case 0xb400: case 0xb600:
        case 0xb800: case 0xba00: case 0xbc00: case 0xbe00:
            sprintf( strDasm, "CMP.B    %s,D%d", MAKE_EA( bLo, 1 ), bRegHi );
            break;
        case 0xb040: case 0xb240: case 0xb440: case 0xb640:
        case 0xb840: case 0xba40: case 0xbc40: case 0xbe40:
            sprintf( strDasm, "CMP.W    %s,D%d", MAKE_EA( bLo, 2 ), bRegHi );
            break;
        case 0xb080: case 0xb280: case 0xb480: case 0xb680:
        case 0xb880: case 0xba80: case 0xbc80: case 0xbe80:
            sprintf( strDasm, "CMP.L    %s,D%d", MAKE_EA( bLo, 4 ), bRegHi );
            break;
        case 0xb0c0: case 0xb2c0: case 0xb4c0: case 0xb6c0:
        case 0xb8c0: case 0xbac0: case 0xbcc0: case 0xbec0:
            sprintf( strDasm, "CMPA.W   %s,A%d", MAKE_EA( bLo, 2 ), bRegHi );
            break;
        case 0xb100: case 0xb300: case 0xb500: case 0xb700:
        case 0xb900: case 0xbb00: case 0xbd00: case 0xbf00:
            if( ( bLo & 0x38 ) == 0x08 )
            {
                sprintf( strDasm, "CMPM.B   (A%d)+,(A%d)+", bRegLo, bRegHi );
            }
            else
            {
                sprintf( 
                    strDasm, "EOR.B    D%d,%s", bRegHi, MAKE_EA( bLo, 1 ) 
                );
            }
            break;
        case 0xb140: case 0xb340: case 0xb540: case 0xb740:
        case 0xb940: case 0xbb40: case 0xbd40: case 0xbf40:
            if( ( bLo & 0x38 ) == 0x08 )
            {
                sprintf( strDasm, "CMPM.W   (A%d)+,(A%d)+", bRegLo, bRegHi );
            }
            else
            {
                sprintf( 
                    strDasm, "EOR.W    D%d,%s", bRegHi, MAKE_EA( bLo, 2 ) 
                );
            }
            break;
        case 0xb180: case 0xb380: case 0xb580: case 0xb780:
        case 0xb980: case 0xbb80: case 0xbd80: case 0xbf80:
            if( ( bLo & 0x38 ) == 0x08 )
            {
                sprintf( strDasm, "CMPM.L   (A%d)+,(A%d)+", bRegLo, bRegHi );
            }
            else
            {
                sprintf( 
                    strDasm, "EOR.L    D%d,%s", bRegHi, MAKE_EA( bLo, 4 ) 
                );
            }
            break;
        case 0xb1c0: case 0xb3c0: case 0xb5c0: case 0xb7c0:
        case 0xb9c0: case 0xbbc0: case 0xbdc0: case 0xbfc0:
            sprintf( strDasm, "CMPA.L   %s,A%d", MAKE_EA( bLo, 4 ), bRegHi );
            break;
        case 0xc000: case 0xc200: case 0xc400: case 0xc600:
        case 0xc800: case 0xca00: case 0xcc00: case 0xce00:
            sprintf( strDasm, "AND.B    %s,D%d", MAKE_EA( bLo, 1 ), bRegHi );
            break;
        case 0xc040: case 0xc240: case 0xc440: case 0xc640:
        case 0xc840: case 0xca40: case 0xcc40: case 0xce40:
            sprintf( strDasm, "AND.W    %s,D%d", MAKE_EA( bLo, 2 ), bRegHi );
            break;
        case 0xc080: case 0xc280: case 0xc480: case 0xc680:
        case 0xc880: case 0xca80: case 0xcc80: case 0xce80:
            sprintf( strDasm, "AND.L    %s,D%d", MAKE_EA( bLo, 4 ), bRegHi );
            break;
        case 0xc0c0: case 0xc2c0: case 0xc4c0: case 0xc6c0:
        case 0xc8c0: case 0xcac0: case 0xccc0: case 0xcec0:
            sprintf( strDasm, "MULU.W   %s,D%d", MAKE_EA( bLo, 2 ), bRegHi );
            break;
        case 0xc100: case 0xc300: case 0xc500: case 0xc700:
        case 0xc900: case 0xcb00: case 0xcd00: case 0xcf00:
            if( ( bLo & 0x30 ) == 0 )
            {
                if( bLo & 0x08 )
                {
                    sprintf( 
                        strDasm, "ABCD.B   -(A%d),-(A%d)", bRegLo, bRegHi 
                    );
                }
                else
                {
                    sprintf( 
                        strDasm, "ABCD.B   D%d,D%d", bRegLo, bRegHi 
                    );
                }
            }
            else
            {
                sprintf( 
                    strDasm, "AND.B    D%d,%s", bRegHi, MAKE_EA( bLo, 1 ) 
                );
            }
            break;
        case 0xc140: case 0xc340: case 0xc540: case 0xc740:
        case 0xc940: case 0xcb40: case 0xcd40: case 0xcf40:
            if( ( bLo & 0x30 ) == 0 )
            {
                if( bLo & 0x08 )
                {
                    sprintf( 
                        strDasm, "EXG      A%d,A%d", bRegHi, bRegLo 
                    );
                }
                else
                {
                    sprintf( 
                        strDasm, "EXG      D%d,D%d", bRegHi, bRegLo 
                    );
                }
            }
            else
            {
                sprintf( 
                    strDasm, "AND.W    D%d,%s", bRegHi, MAKE_EA( bLo, 2 ) 
                );
            }
            break;
        case 0xc180: case 0xc380: case 0xc580: case 0xc780:
        case 0xc980: case 0xcb80: case 0xcd80: case 0xcf80:
            if( ( bLo & 0x30 ) == 0x08 )
            {
                sprintf( strDasm, "EXG      D%d,A%d", bRegHi, bRegLo );
            }
            else
            {
                sprintf( 
                    strDasm, "AND.L    D%d,%s", bRegHi, MAKE_EA( bLo, 4 ) 
                );
            }
            break;
        case 0xc1c0: case 0xc3c0: case 0xc5c0: case 0xc7c0:
        case 0xc9c0: case 0xcbc0: case 0xcdc0: case 0xcfc0:
            sprintf( strDasm, "MULS.W   %s,D%d", MAKE_EA( bLo, 2 ), bRegHi );
            break;
        case 0xd000: case 0xd200: case 0xd400: case 0xd600:
        case 0xd800: case 0xda00: case 0xdc00: case 0xde00:
            sprintf( strDasm, "ADD.B    %s,D%d", MAKE_EA( bLo, 1 ), bRegHi );
            break;
        case 0xd040: case 0xd240: case 0xd440: case 0xd640:
        case 0xd840: case 0xda40: case 0xdc40: case 0xde40:
            sprintf( strDasm, "ADD.W    %s,D%d", MAKE_EA( bLo, 2 ), bRegHi );
            break;
        case 0xd080: case 0xd280: case 0xd480: case 0xd680:
        case 0xd880: case 0xda80: case 0xdc80: case 0xde80:
            sprintf( strDasm, "ADD.L    %s,D%d", MAKE_EA( bLo, 4 ), bRegHi );
            break;
        case 0xd0c0: case 0xd2c0: case 0xd4c0: case 0xd6c0:
        case 0xd8c0: case 0xdac0: case 0xdcc0: case 0xdec0:
            sprintf( strDasm, "ADDA.W   %s,D%d", MAKE_EA( bLo, 2 ), bRegHi );
            break;
        case 0xd100: case 0xd300: case 0xd500: case 0xd700:
        case 0xd900: case 0xdb00: case 0xdd00: case 0xdf00:
            if( ( bLo & 0x30 ) == 0 )
            {
                if( bLo & 0x08 )
                {
                    sprintf( 
                        strDasm, "ADDX.B   -(A%d),-(A%d)", bRegLo, bRegHi 
                    );
                }
                else
                {
                    sprintf( strDasm, "ADDX.B   D%d,D%d", bRegLo, bRegHi );
                }
            }
            else
            {
                sprintf( 
                    strDasm, "ADD.B    D%d,%s", bRegHi, MAKE_EA( bLo, 1 ) 
                );
            }
            break;
        case 0xd140: case 0xd340: case 0xd540: case 0xd740:
        case 0xd940: case 0xdb40: case 0xdd40: case 0xdf40:
            if( ( bLo & 0x30 ) == 0 )
            {
                if( bLo & 0x08 )
                {
                    sprintf( 
                        strDasm, "ADDX.W   -(A%d),-(A%d)", bRegLo, bRegHi 
                    );
                }
                else
                {
                    sprintf( strDasm, "ADDX.W   D%d,D%d", bRegLo, bRegHi );
                }
            }
            else
            {
                sprintf( 
                    strDasm, "ADD.W    D%d,%s", bRegHi, MAKE_EA( bLo, 2 ) 
                );
            }
            break;
        case 0xd180: case 0xd380: case 0xd580: case 0xd780:
        case 0xd980: case 0xdb80: case 0xdd80: case 0xdf80:
            if( ( bLo & 0x30 ) == 0 )
            {
                if( bLo & 0x08 )
                {
                    sprintf( 
                        strDasm, "ADDX.L   -(A%d),-(A%d)", bRegLo, bRegHi 
                    );
                }
                else
                {
                    sprintf( strDasm, "ADDX.L   D%d,D%d", bRegLo, bRegHi );
                }
            }
            else
            {
                sprintf( 
                    strDasm, "ADD.L    D%d,%s", bRegHi, MAKE_EA( bLo, 4 ) 
                );
            }
            break;
        case 0xd1c0: case 0xd3c0: case 0xd5c0: case 0xd7c0:
        case 0xd9c0: case 0xdbc0: case 0xddc0: case 0xdfc0:
            sprintf( strDasm, "ADDA.L   %s,A%d", MAKE_EA( bLo, 4 ), bRegHi );
            break;
        case 0xe000: case 0xe200: case 0xe400: case 0xe600:
        case 0xe800: case 0xea00: case 0xec00: case 0xee00:
            switch( ( bLo >> 3 ) & 0x07 )
            {
                case 0:
                    sprintf( 
                        strDasm, 
                        "ASR.B    #%d,D%d", 
                        ( bRegHi == 0 ) ? 8 : bRegHi, 
                        bRegLo 
                    );
                    break;
                case 1:
                    sprintf( 
                        strDasm, 
                        "LSR.B    #%d,D%d", 
                        ( bRegHi == 0 ) ? 8 : bRegHi, 
                        bRegLo 
                    );
                    break;
                case 2:
                    sprintf( 
                        strDasm, 
                        "ROXR.B   #%d,D%d", 
                        ( bRegHi == 0 ) ? 8 : bRegHi, 
                        bRegLo 
                    );
                    break;
                case 3:
                    sprintf( 
                        strDasm, 
                        "ROR.B    #%d,D%d", 
                        ( bRegHi == 0 ) ? 8 : bRegHi, 
                        bRegLo 
                    );
                    break;
                case 4:
                    sprintf( strDasm, "ASR.B    D%d,D%d", bRegHi, bRegLo );
                    break;
                case 5:
                    sprintf( strDasm, "LSR.B    D%d,D%d", bRegHi, bRegLo );
                    break;
                case 6:
                    sprintf( strDasm, "ROXR.B   D%d,D%d", bRegHi, bRegLo );
                    break;
                case 7:
                    sprintf( strDasm, "ROR.B    D%d,D%d", bRegHi, bRegLo );
                    break;
            }
            break;
        case 0xe040: case 0xe240: case 0xe440: case 0xe640:
        case 0xe840: case 0xea40: case 0xec40: case 0xee40:
            switch( ( bLo >> 3 ) & 0x07 )
            {
                case 0:
                    sprintf( 
                        strDasm, 
                        "ASR.W    #%d,D%d", 
                        ( bRegHi == 0 ) ? 8 : bRegHi, 
                        bRegLo 
                    );
                    break;
                case 1:
                    sprintf( 
                        strDasm, 
                        "LSR.W    #%d,D%d", 
                        ( bRegHi == 0 ) ? 8 : bRegHi, 
                        bRegLo 
                    );
                    break;
                case 2:
                    sprintf( 
                        strDasm, 
                        "ROXR.W   #%d,D%d", 
                        ( bRegHi == 0 ) ? 8 : bRegHi, 
                        bRegLo 
                    );
                    break;
                case 3:
                    sprintf( 
                        strDasm, 
                        "ROR.W    #%d,D%d", 
                        ( bRegHi == 0 ) ? 8 : bRegHi, 
                        bRegLo 
                    );
                    break;
                case 4:
                    sprintf( strDasm, "ASR.W    D%d,D%d", bRegHi, bRegLo );
                    break;
                case 5:
                    sprintf( strDasm, "LSR.W    D%d,D%d", bRegHi, bRegLo );
                    break;
                case 6:
                    sprintf( strDasm, "ROXR.W   D%d,D%d", bRegHi, bRegLo );
                    break;
                case 7:
                    sprintf( strDasm, "ROR.W    D%d,D%d", bRegHi, bRegLo );
                    break;
            }
            break;
        case 0xe080: case 0xe280: case 0xe480: case 0xe680:
        case 0xe880: case 0xea80: case 0xec80: case 0xee80:
            switch( ( bLo >> 3 ) & 0x07 )
            {
                case 0:
                    sprintf( 
                        strDasm, 
                        "ASR.L    #%d,D%d", 
                        ( bRegHi == 0 ) ? 8 : bRegHi, 
                        bRegLo 
                    );
                    break;
                case 1:
                    sprintf( 
                        strDasm, 
                        "LSR.L    #%d,D%d", 
                        ( bRegHi == 0 ) ? 8 : bRegHi, 
                        bRegLo 
                    );
                    break;
                case 2:
                    sprintf( 
                        strDasm, 
                        "ROXR.L   #%d,D%d", 
                        ( bRegHi == 0 ) ? 8 : bRegHi, 
                        bRegLo 
                    );
                    break;
                case 3:
                    sprintf( 
                        strDasm, 
                        "ROR.L    #%d,D%d", 
                        ( bRegHi == 0 ) ? 8 : bRegHi, 
                        bRegLo 
                    );
                    break;
                case 4:
                    sprintf( strDasm, "ASR.L    D%d,D%d", bRegHi, bRegLo );
                    break;
                case 5:
                    sprintf( strDasm, "LSR.L    D%d,D%d", bRegHi, bRegLo );
                    break;
                case 6:
                    sprintf( strDasm, "ROXR.L   D%d,D%d", bRegHi, bRegLo );
                    break;
                case 7:
                    sprintf( strDasm, "ROR.L    D%d,D%d", bRegHi, bRegLo );
                    break;
            }
            break;
        case 0xe0c0: case 0xe2c0: case 0xe4c0: case 0xe6c0:
        case 0xe8c0: case 0xeac0: case 0xecc0: case 0xeec0:
            switch( ( wOp >> 8 ) & 0x07 )
            {
                case 0: 
                    sprintf( strDasm, "ASR.L    #1,%s", MAKE_EA( bLo, 4 ) );
                    break;
                case 1: 
                    sprintf( strDasm, "ASL.L    #1,%s", MAKE_EA( bLo, 4 ) );
                    break;
                case 2: 
                    sprintf( strDasm, "LSR.L    #1,%s", MAKE_EA( bLo, 4 ) );
                    break;
                case 3: 
                    sprintf( strDasm, "LSL.L    #1,%s", MAKE_EA( bLo, 4 ) );
                    break;
                case 4: 
                    sprintf( strDasm, "ROXR.L   #1,%s", MAKE_EA( bLo, 4 ) );
                    break;
                case 5: 
                    sprintf( strDasm, "ROXL.L   #1,%s", MAKE_EA( bLo, 4 ) );
                    break;
                case 6: 
                    sprintf( strDasm, "ROR.L    #1,%s", MAKE_EA( bLo, 4 ) );
                    break;
                case 7: 
                    sprintf( strDasm, "ROL.L    #1,%s", MAKE_EA( bLo, 4 ) );
                    break;
            }
            break;
        case 0xe100: case 0xe300: case 0xe500: case 0xe700:
        case 0xe900: case 0xeb00: case 0xed00: case 0xef00:
            switch( ( bLo >> 3 ) & 0x07 )
            {
                case 0:
                    sprintf( 
                        strDasm, 
                        "ASL.B    #%d,D%d", 
                        ( bRegHi == 0 ) ? 8 : bRegHi, 
                        bRegLo 
                    );
                    break;
                case 1:
                    sprintf( 
                        strDasm, 
                        "LSL.B    #%d,D%d", 
                        ( bRegHi == 0 ) ? 8 : bRegHi, 
                        bRegLo 
                    );
                    break;
                case 2:
                    sprintf( 
                        strDasm, 
                        "ROXL.B   #%d,D%d", 
                        ( bRegHi == 0 ) ? 8 : bRegHi, 
                        bRegLo 
                    );
                    break;
                case 3:
                    sprintf( 
                        strDasm, 
                        "ROL.B    #%d,D%d", 
                        ( bRegHi == 0 ) ? 8 : bRegHi, 
                        bRegLo 
                    );
                    break;
                case 4:
                    sprintf( strDasm, "ASL.B    D%d,D%d", bRegHi, bRegLo );
                    break;
                case 5:
                    sprintf( strDasm, "LSL.B    D%d,D%d", bRegHi, bRegLo );
                    break;
                case 6:
                    sprintf( strDasm, "ROXL.B   D%d,D%d", bRegHi, bRegLo );
                    break;
                case 7:
                    sprintf( strDasm, "ROL.B    D%d,D%d", bRegHi, bRegLo );
                    break;
            }
            break;
        case 0xe140: case 0xe340: case 0xe540: case 0xe740:
        case 0xe940: case 0xeb40: case 0xed40: case 0xef40:
            switch( ( bLo >> 3 ) & 0x07 )
            {
                case 0:
                    sprintf( 
                        strDasm, 
                        "ASL.W    #%d,D%d", 
                        ( bRegHi == 0 ) ? 8 : bRegHi, 
                        bRegLo 
                    );
                    break;
                case 1:
                    sprintf( 
                        strDasm, 
                        "LSL.W    #%d,D%d", 
                        ( bRegHi == 0 ) ? 8 : bRegHi, 
                        bRegLo 
                    );
                    break;
                case 2:
                    sprintf( 
                        strDasm, 
                        "ROXL.W   #%d,D%d", 
                        ( bRegHi == 0 ) ? 8 : bRegHi, 
                        bRegLo 
                    );
                    break;
                case 3:
                    sprintf( 
                        strDasm, 
                        "ROL.W    #%d,D%d", 
                        ( bRegHi == 0 ) ? 8 : bRegHi, 
                        bRegLo 
                    );
                    break;
                case 4:
                    sprintf( strDasm, "ASL.W    D%d,D%d", bRegHi, bRegLo );
                    break;
                case 5:
                    sprintf( strDasm, "LSL.W    D%d,D%d", bRegHi, bRegLo );
                    break;
                case 6:
                    sprintf( strDasm, "ROXL.W   D%d,D%d", bRegHi, bRegLo );
                    break;
                case 7:
                    sprintf( strDasm, "ROL.W    D%d,D%d", bRegHi, bRegLo );
                    break;
            }
            break;
        case 0xe180: case 0xe380: case 0xe580: case 0xe780:
        case 0xe980: case 0xeb80: case 0xed80: case 0xef80:
            switch( ( bLo >> 3 ) & 0x07 )
            {
                case 0:
                    sprintf( 
                        strDasm, 
                        "ASL.L    #%d,D%d", 
                        ( bRegHi == 0 ) ? 8 : bRegHi, 
                        bRegLo 
                    );
                    break;
                case 1:
                    sprintf( 
                        strDasm, 
                        "LSL.L    #%d,D%d", 
                        ( bRegHi == 0 ) ? 8 : bRegHi, 
                        bRegLo 
                    );
                    break;
                case 2:
                    sprintf( 
                        strDasm, 
                        "ROXL.L   #%d,D%d", 
                        ( bRegHi == 0 ) ? 8 : bRegHi, 
                        bRegLo 
                    );
                    break;
                case 3:
                    sprintf( 
                        strDasm, 
                        "ROL.L    #%d,D%d", 
                        ( bRegHi == 0 ) ? 8 : bRegHi, 
                        bRegLo 
                    );
                    break;
                case 4:
                    sprintf( strDasm, "ASL.L    D%d,D%d", bRegHi, bRegLo );
                    break;
                case 5:
                    sprintf( strDasm, "LSL.L    D%d,D%d", bRegHi, bRegLo );
                    break;
                case 6:
                    sprintf( strDasm, "ROXL.L   D%d,D%d", bRegHi, bRegLo );
                    break;
                case 7:
                    sprintf( strDasm, "ROL.L    D%d,D%d", bRegHi, bRegLo );
                    break;
            }
            break;
        default:
            sprintf( strDasm, "DC.W     $%04x", wOp );
            break;
    }

    //  OK, build the disassembled line and return it.
    sprintf( strLine, "%-7s %-31s %s", strAddr, strVals, strDasm );
    return( strLine );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: dbgRegister
//
//  Description:
//
//      This member is called to return a string describing the value of
//      a register to the client.
//
//  Parameters:
//
//      dwNum (input)
//          Which register to retrieve.
//
//  Returns:
//
//      A pointer to the string containing the register description.
//      NULL if the register is invalid.
//
///////////////////////////////////////////////////////////////////////////////
const
char*
CPU68000::dbgRegister(
    const DWord dwNum
)
{
    //  The following static buffer will contain the register.
    static char strLine[ 80 ];

    //  Which register?
    //  For convenience, we combine the AX/DX registers on one line since
    //  there are too many to fit in the window vertically and we want the
    //  user to see them all.
    switch( dwNum )
    {
        case REG_ICOUNT: 
            sprintf( strLine, "IC:  $%04lx", getReg( REG_ICOUNT ) );  break;
        case REG_PC: 
            sprintf( strLine, "PC:  $%06lx", getReg( REG_PC ) );      break;
        case REG_ASP:
            sprintf( strLine, "ASP: $%08lx", getReg( REG_ASP ) );     break;
        case REG_SR:
            sprintf( 
                strLine, 
                "SR:  %c%c%c%c%c", 
                ( ( getReg( REG_SR ) >> 16 ) & 0x01 ) ? 'X' : '-',
                ( ( getReg( REG_SR ) >>  7 ) & 0x01 ) ? 'N' : '-',
                ( ( getReg( REG_SR ) >>  6 ) & 0x01 ) ? 'Z' : '-',
                ( ( getReg( REG_SR ) >> 11 ) & 0x01 ) ? 'V' : '-',
                ( ( getReg( REG_SR ) >>  0 ) & 0x01 ) ? 'C' : '-'
            ); 
            break;
        case REG_A0:
            sprintf( 
                strLine, 
                "A0:  $%08lx  D0:  $%08lx", 
                getReg( REG_A0 ),
                getReg( REG_D0 )
            );      
            break;
        case REG_A1:
            sprintf( 
                strLine, 
                "A1:  $%08lx  D1:  $%08lx", 
                getReg( REG_A1 ),
                getReg( REG_D1 )
            );      
            break;
        case REG_A2:
            sprintf( 
                strLine, 
                "A2:  $%08lx  D2:  $%08lx", 
                getReg( REG_A2 ),
                getReg( REG_D2 )
            );      
            break;
        case REG_A3:
            sprintf( 
                strLine, 
                "A3:  $%08lx  D3:  $%08lx", 
                getReg( REG_A3 ),
                getReg( REG_D3 )
            );      
            break;
        case REG_A4:
            sprintf( 
                strLine, 
                "A4:  $%08lx  D4:  $%08lx", 
                getReg( REG_A4 ),
                getReg( REG_D4 )
            );      
            break;
        case REG_A5:
            sprintf( 
                strLine, 
                "A5:  $%08lx  D5:  $%08lx", 
                getReg( REG_A5 ),
                getReg( REG_D5 )
            );      
            break;
        case REG_A6:
            sprintf( 
                strLine, 
                "A6:  $%08lx  D6:  $%08lx", 
                getReg( REG_A6 ),
                getReg( REG_D6 )
            );      
            break;
        case REG_A7:
            sprintf( 
                strLine, 
                "A7:  $%08lx  D7:  $%08lx", 
                getReg( REG_A7 ),
                getReg( REG_D7 )
            );      
            break;
        default:     
            return( NULL );
    }

    return( strLine );
}

#endif
