/******************************************************************************
*
*  File: 6809.c
*
*  Description:
* 
*      This file implements an emulation of the Motorola 6809.
*
*  Author: Kevin Brisley
*
*  Copyright (c) 1996  Kevin Brisley
*  All rights reserved.
*
******************************************************************************/

/******************************************************************************
*  Header Files.
******************************************************************************/
/*
 *  System Headers.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/*
 *  Local Headers.
 */
#include "mc6809.h"
#include "space.h"
#ifdef DEBUGGER
#include "debugger.h"
#endif



/******************************************************************************
*  Types.
******************************************************************************/
/*
 *  The following type indicates the addressing modes available.
 */
typedef enum
{ 
    AM_DIRECT,
    AM_INHERENT,
    AM_RELATIVE,
    AM_IMMEDIATE,
    AM_INDEXED,
    AM_EXTENDED,

    AM_NA
}
MC6809AddressMode;

    
/******************************************************************************
*  Miscellaneous Operations.
******************************************************************************/
#define SWAP_BYTE(B1, B2)                                                     \
{                                                                             \
    Byte sb_bTemp;                                                            \
    sb_bTemp = B1;                                                            \
    B1 = B2;                                                                  \
    B2 = sb_bTemp;                                                            \
}

#define SWAP_WORD(W1, W2)                                                     \
{                                                                             \
    Word sw_wTemp;                                                            \
    sw_wTemp = W1;                                                            \
    W1 = W2;                                                                  \
    W2 = sw_wTemp;                                                            \
}

/******************************************************************************
*  Bit Operations.
******************************************************************************/
#define BTST0(THEBYTE) (THEBYTE & 0x01 ? 1 : 0)
#define BTST1(THEBYTE) (THEBYTE & 0x02 ? 1 : 0)
#define BTST2(THEBYTE) (THEBYTE & 0x04 ? 1 : 0)
#define BTST3(THEBYTE) (THEBYTE & 0x08 ? 1 : 0)
#define BTST4(THEBYTE) (THEBYTE & 0x10 ? 1 : 0)
#define BTST5(THEBYTE) (THEBYTE & 0x20 ? 1 : 0)
#define BTST6(THEBYTE) (THEBYTE & 0x40 ? 1 : 0)
#define BTST7(THEBYTE) (THEBYTE & 0x80 ? 1 : 0)
#define BTST8(THEBYTE) (THEBYTE & 0x100 ? 1 : 0)
#define BTST15(THEBYTE) (THEBYTE & 0x8000 ? 1 : 0)
#define BTST16(THEBYTE) (THEBYTE & 0x10000L ? 1 : 0)

#define BSET0(THEBYTE)  (THEBYTE |= 0x01)
#define BSET1(THEBYTE)  (THEBYTE |= 0x02)
#define BSET2(THEBYTE)  (THEBYTE |= 0x04)
#define BSET3(THEBYTE)  (THEBYTE |= 0x08)
#define BSET4(THEBYTE)  (THEBYTE |= 0x10)
#define BSET5(THEBYTE)  (THEBYTE |= 0x20)
#define BSET6(THEBYTE)  (THEBYTE |= 0x40)
#define BSET7(THEBYTE)  (THEBYTE |= 0x80)


/******************************************************************************
*  Memory Read Macros.
******************************************************************************/
/*
 *  The following macro reads a memory location without modifying the program
 *  counter.
 */
#define MEM_READ_BYTE_NC(DEST, LOCATION)                                      \
{                                                                             \
    DEST = (Byte)CPU::sm_pCPU->readMem(LOCATION);                             \
}

/*
 *  The following macro is used to read the next opcode and then
 *  increment the program counter.  
 */
#define MEM_READ_OP(DEST)                                                     \
{                                                                             \
    DEST = (CPU::sm_pSpace->getBuffer())[gpMC6809->pc];                       \
    gpMC6809->pc += 1;                                                        \
}

/*
 *  The following macro is used to read the byte at the current location and
 *  then increment the program counter.  Since this is used for reading 
 *  locations in ROM, we can use the quick read.
 */
#define MEM_READ_BYTE(DEST)                                                   \
{                                                                             \
    DEST = (CPU::sm_pSpace->getBuffer())[gpMC6809->pc];                       \
    gpMC6809->pc += 1;                                                        \
}

/*
 *  The following macro is used to read a word at the specified location
 *  without modifying the program counter.
 */
#define MEM_READ_WORD_NC(DEST, LOCATION)                                      \
{                                                                             \
    Word mrwnc_wTemp;                                                         \
    MEM_READ_BYTE_NC (DEST, LOCATION);                                        \
    DEST <<= 8;                                                               \
    MEM_READ_BYTE_NC (mrwnc_wTemp, LOCATION + 1);                             \
    DEST |= mrwnc_wTemp;                                                      \
}

/*
 *  The following macro is used to read a word.  It reads the next two bytes
 *  of memory and then increments the program counter by 2.  Since this is
 *  reading from the PC, and therefore ROM, we can use a quick read.
 */
#define MEM_READ_WORD(DEST)                                                   \
{                                                                             \
    Word mrw_wTemp;                                                           \
    DEST = (CPU::sm_pSpace->getBuffer())[gpMC6809->pc];                       \
    DEST <<= 8;                                                               \
    mrw_wTemp = (CPU::sm_pSpace->getBuffer())[gpMC6809->pc + 1];              \
    DEST |= mrw_wTemp;                                                        \
    gpMC6809->pc += 2;                                                        \
}

/*
 *  The following macro writes a byte to a memory location.
 */
#define MEM_WRITE_BYTE(LOCATION, VALUE)                                       \
{                                                                             \
    CPU::sm_pCPU->writeMem (LOCATION, VALUE);                                 \
}

/*
 *  The following macro writes a word to a memory location.
 */
#define MEM_WRITE_WORD(LOCATION, VALUE)                                       \
{                                                                             \
    MEM_WRITE_BYTE (LOCATION, (Byte)(VALUE >> 8));                            \
    MEM_WRITE_BYTE (LOCATION + 1, (Byte)(VALUE));                             \
}



/*
 *  The following macro determines which register to use based on the value
 *  passed in.
 *  KSB:  The extra cases are added in for performance reasons.  They should
 *        not occur.
 */
#define REFERENCED_REGISTER(PREG, PDV)                                        \
{                                                                             \
    switch ((Byte)(PDV & 0x60))                                               \
    {                                                                         \
        case 0x00:                                                            \
        {                                                                     \
            PREG = &(gpMC6809->x);                                            \
            break;                                                            \
        }                                                                     \
        case 0x20:                                                            \
        {                                                                     \
            PREG = &(gpMC6809->y);                                            \
            break;                                                            \
        }                                                                     \
        case 0x40:                                                            \
        {                                                                     \
            PREG = &(gpMC6809->u);                                            \
            break;                                                            \
        }                                                                     \
        case 0x60:                                                            \
        {                                                                     \
            PREG = &(gpMC6809->s);                                            \
            break;                                                            \
        }                                                                     \
        case 0x80:                                                            \
        {                                                                     \
            PREG = &(gpMC6809->s);                                            \
            break;                                                            \
        }                                                                     \
        default:                                                              \
        {                                                                     \
            PREG = &(gpMC6809->s);                                            \
            break;                                                            \
        }                                                                     \
    }                                                                         \
}

/*
 *  The following macro performs a predecrement on a register based on 
 *  the byte passed in.
 */
#define PREDECREMENT(PDV)                                                     \
{                                                                             \
    switch (PDV)                                                              \
    {                                                                         \
        case 0x82:                                                            \
            gpMC6809->x -= 1;                                                 \
            break;                                                            \
        case 0xa2:                                                            \
            gpMC6809->y -= 1;                                                 \
            break;                                                            \
        case 0xc2:                                                            \
            gpMC6809->u -= 1;                                                 \
            break;                                                            \
        case 0xe2:                                                            \
            gpMC6809->s -= 1;                                                 \
            break;                                                            \
        case 0x92:                                                            \
        case 0xb2:                                                            \
        case 0xd2:                                                            \
        case 0xf2:                                                            \
            printf ("Invalid Predecrement.\n");                               \
            fflush (stdout);                                                  \
            exit (0);                                                         \
            break;                                                            \
        case 0x93:                                                            \
        case 0x83:                                                            \
            gpMC6809->x -= 2;                                                 \
            break;                                                            \
        case 0xb3:                                                            \
        case 0xa3:                                                            \
            gpMC6809->y -= 2;                                                 \
            break;                                                            \
        case 0xd3:                                                            \
        case 0xc3:                                                            \
            gpMC6809->u -= 2;                                                 \
            break;                                                            \
        case 0xf3:                                                            \
        case 0xe3:                                                            \
            gpMC6809->s -= 2;                                                 \
            break;                                                            \
    }                                                                         \
}

/*
 *  The following macro performs a postincrement on a register based on 
 *  the byte passed in.
 */
#define POSTINCREMENT(PDV)                                                    \
{                                                                             \
    switch (PDV)                                                              \
    {                                                                         \
        case 0x80:                                                            \
            gpMC6809->x += 1;                                                 \
            break;                                                            \
        case 0xa0:                                                            \
            gpMC6809->y += 1;                                                 \
            break;                                                            \
        case 0xc0:                                                            \
            gpMC6809->u += 1;                                                 \
            break;                                                            \
        case 0xe0:                                                            \
            gpMC6809->s += 1;                                                 \
            break;                                                            \
        case 0x90:                                                            \
        case 0xb0:                                                            \
        case 0xd0:                                                            \
        case 0xf0:                                                            \
            printf ("Invalid PostIncrement.\n");                              \
            fflush (stdout);                                                  \
            exit (0);                                                         \
            break;                                                            \
        case 0x91:                                                            \
        case 0x81:                                                            \
            gpMC6809->x += 2;                                                 \
            break;                                                            \
        case 0xb1:                                                            \
        case 0xa1:                                                            \
            gpMC6809->y += 2;                                                 \
            break;                                                            \
        case 0xd1:                                                            \
        case 0xc1:                                                            \
            gpMC6809->u += 2;                                                 \
            break;                                                            \
        case 0xf1:                                                            \
        case 0xe1:                                                            \
            gpMC6809->s += 2;                                                 \
            break;                                                            \
    }                                                                         \
}

/*
 *  The following finds the effective address based on a byte and register
 *  specified.
 */
#define EA_GET_POST(DEST, PDV, REG)                                           \
{                                                                             \
    Byte eagp_bTemp;                                                          \
    Word eagp_wTemp;                                                          \
    DEST = 0x0000;                                                            \
    if ((PDV & 0x80) == 0x00)                                                 \
    {                                                                         \
        if (PDV & 0x10)                                                       \
        {                                                                     \
            DEST = (REG) + ((Word)(PDV & 0x1f) | 0xffe0);                     \
        }                                                                     \
        else                                                                  \
        {                                                                     \
            DEST = (REG) + ((Word)(PDV & 0x1f));                              \
        }                                                                     \
    }                                                                         \
    else                                                                      \
    {                                                                         \
        switch (PDV & 0x1f)                                                   \
        {                                                                     \
            case 0x00:                                                        \
            case 0x01:                                                        \
            case 0x02:                                                        \
            case 0x03:                                                        \
            case 0x04:                                                        \
            case 0x11:                                                        \
            case 0x13:                                                        \
            case 0x14:                                                        \
            {                                                                 \
                DEST = (REG);                                                 \
                break;                                                        \
            }                                                                 \
            case 0x05:                                                        \
            case 0x15:                                                        \
            {                                                                 \
                if (gpMC6809->acc.B.l & 0x80)                                 \
                {                                                             \
                    DEST = (REG) + ((Word)gpMC6809->acc.B.l | 0xff00);        \
                }                                                             \
                else                                                          \
                {                                                             \
                    DEST = (REG) + ((Word)gpMC6809->acc.B.l);                 \
                }                                                             \
                break;                                                        \
            }                                                                 \
            case 0x06:                                                        \
            case 0x16:                                                        \
            {                                                                 \
                if (gpMC6809->acc.B.h & 0x80)                                 \
                {                                                             \
                    DEST = (REG) + ((Word)gpMC6809->acc.B.h | 0xff00);        \
                }                                                             \
                else                                                          \
                {                                                             \
                    DEST = (REG) + ((Word)gpMC6809->acc.B.h);                 \
                }                                                             \
                break;                                                        \
            }                                                                 \
            case 0x08:                                                        \
            case 0x18:                                                        \
            {                                                                 \
                MEM_READ_BYTE (eagp_bTemp);                                   \
                if (eagp_bTemp & 0x80)                                        \
                {                                                             \
                    DEST = (REG) + ((Word)eagp_bTemp | 0xff00);               \
                }                                                             \
                else                                                          \
                {                                                             \
                    DEST = (REG) + ((Word)eagp_bTemp);                        \
                }                                                             \
                break;                                                        \
            }                                                                 \
            case 0x09:                                                        \
            case 0x19:                                                        \
            {                                                                 \
                MEM_READ_WORD (eagp_wTemp);                                   \
                DEST = (REG) + eagp_wTemp;                                    \
                break;                                                        \
            }                                                                 \
            case 0x0b:                                                        \
            case 0x1b:                                                        \
            {                                                                 \
                DEST = gpMC6809->acc.W + (REG);                               \
                break;                                                        \
            }                                                                 \
            case 0x0c:                                                        \
            case 0x1c:                                                        \
            {                                                                 \
                MEM_READ_BYTE (eagp_bTemp);                                   \
                if (eagp_bTemp & 0x80)                                        \
                {                                                             \
                    DEST = gpMC6809->pc + ((Word)eagp_bTemp | 0xff00);        \
                }                                                             \
                else                                                          \
                {                                                             \
                    DEST = gpMC6809->pc + ((Word)eagp_bTemp);                 \
                }                                                             \
                break;                                                        \
            }                                                                 \
            case 0x0d:                                                        \
            case 0x1d:                                                        \
            {                                                                 \
                MEM_READ_WORD (eagp_wTemp);                                   \
                DEST = gpMC6809->pc + eagp_wTemp;                             \
                break;                                                        \
            }                                                                 \
            case 0x1f:                                                        \
            {                                                                 \
                MEM_READ_WORD (DEST);                                         \
                break;                                                        \
            }                                                                 \
            default:                                                          \
            {                                                                 \
                printf ("Invalid indirect addressing postbyte.\n");           \
                fflush (stdout);                                              \
                exit (0);                                                     \
                break;                                                        \
            }                                                                 \
        }                                                                     \
        if (PDV & 0x10)                                                       \
        {                                                                     \
            eagp_wTemp = DEST;                                                \
            MEM_READ_WORD_NC (DEST, eagp_wTemp);                              \
        }                                                                     \
    }                                                                         \
}

/******************************************************************************
*  Opcode Macros.
******************************************************************************/
#define OP_ADC(ARG)                                                           \
{                                                                             \
    Byte adc_bO = MC6809ReadMemOperand ();                                    \
    Byte adc_bTemp;                                                           \
    Word adc_wTemp;                                                           \
    adc_bTemp = (ARG & 0x0f) + (adc_bO & 0x0f) + gpMC6809->cc.bit.c;          \
    gpMC6809->cc.bit.h = BTST4 (adc_bTemp);                                   \
    adc_bTemp = (ARG & 0x7f) + (adc_bO & 0x7f) + gpMC6809->cc.bit.c;          \
    gpMC6809->cc.bit.v = BTST7 (adc_bTemp);                                   \
    adc_wTemp = ARG + adc_bO + gpMC6809->cc.bit.c;                            \
    gpMC6809->cc.bit.c = BTST8 (adc_wTemp);                                   \
    ARG = adc_wTemp & 0xff;                                                   \
	gpMC6809->cc.bit.v ^= gpMC6809->cc.bit.c;                                 \
    gpMC6809->cc.bit.n = BTST7 (ARG);                                         \
    gpMC6809->cc.bit.z = !(ARG);                                              \
}

#define OP_ADC_I(ARG,CYC)                                                     \
{                                                                             \
    Byte adc_bO = MC6809ReadMemIndexedOperand (CYC);                          \
    Byte adc_bTemp;                                                           \
    Word adc_wTemp;                                                           \
    adc_bTemp = (ARG & 0x0f) + (adc_bO & 0x0f) + gpMC6809->cc.bit.c;          \
    gpMC6809->cc.bit.h = BTST4 (adc_bTemp);                                   \
    adc_bTemp = (ARG & 0x7f) + (adc_bO & 0x7f) + gpMC6809->cc.bit.c;          \
    gpMC6809->cc.bit.v = BTST7 (adc_bTemp);                                   \
    adc_wTemp = ARG + adc_bO + gpMC6809->cc.bit.c;                            \
    gpMC6809->cc.bit.c = BTST8 (adc_wTemp);                                   \
    ARG = adc_wTemp & 0xff;                                                   \
	gpMC6809->cc.bit.v ^= gpMC6809->cc.bit.c;                                 \
    gpMC6809->cc.bit.n = BTST7 (ARG);                                         \
    gpMC6809->cc.bit.z = !(ARG);                                              \
}

#define OP_ADD(ARG)                                                           \
{                                                                             \
    Byte add_bO = MC6809ReadMemOperand ();                                    \
    Byte add_bTemp;                                                           \
    Word add_wTemp;                                                           \
    add_bTemp = (ARG & 0x0f) + (add_bO & 0x0f);                               \
    gpMC6809->cc.bit.h = BTST4 (add_bTemp);                                   \
    add_bTemp = (ARG & 0x7f) + (add_bO & 0x7f);                               \
    gpMC6809->cc.bit.v = BTST7 (add_bTemp);                                   \
    add_wTemp = ARG + add_bO;                                                 \
    gpMC6809->cc.bit.c = BTST8 (add_wTemp);                                   \
    ARG = add_wTemp & 0xff;                                                   \
	gpMC6809->cc.bit.v ^= gpMC6809->cc.bit.c;                                 \
    gpMC6809->cc.bit.n = BTST7 (ARG);                                         \
    gpMC6809->cc.bit.z = !(ARG);                                              \
}

#define OP_ADD_I(ARG, CYC)                                                    \
{                                                                             \
    Byte add_bO = MC6809ReadMemIndexedOperand (CYC);                          \
    Byte add_bTemp;                                                           \
    Word add_wTemp;                                                           \
    add_bTemp = (ARG & 0x0f) + (add_bO & 0x0f);                               \
    gpMC6809->cc.bit.h = BTST4 (add_bTemp);                                   \
    add_bTemp = (ARG & 0x7f) + (add_bO & 0x7f);                               \
    gpMC6809->cc.bit.v = BTST7 (add_bTemp);                                   \
    add_wTemp = ARG + add_bO;                                                 \
    gpMC6809->cc.bit.c = BTST8 (add_wTemp);                                   \
    ARG = add_wTemp & 0xff;                                                   \
	gpMC6809->cc.bit.v ^= gpMC6809->cc.bit.c;                                 \
    gpMC6809->cc.bit.n = BTST7 (ARG);                                         \
    gpMC6809->cc.bit.z = !(ARG);                                              \
}

#define OP_AND(ARG)                                                           \
{                                                                             \
    Byte and_bO = MC6809ReadMemOperand ();                                    \
    ARG = ARG & and_bO;                                                       \
	gpMC6809->cc.bit.v = 0;                                                   \
    gpMC6809->cc.bit.n = BTST7 (ARG);                                         \
    gpMC6809->cc.bit.z = !(ARG);                                              \
}

#define OP_AND_I(ARG, CYC)                                                    \
{                                                                             \
    Byte and_bO = MC6809ReadMemIndexedOperand (CYC);                          \
    ARG = ARG & and_bO;                                                       \
	gpMC6809->cc.bit.v = 0;                                                   \
    gpMC6809->cc.bit.n = BTST7 (ARG);                                         \
    gpMC6809->cc.bit.z = !(ARG);                                              \
}

#define OP_ASR(ARG)                                                           \
{                                                                             \
    gpMC6809->cc.bit.c = BTST0 (ARG);                                         \
    ARG >>= 1;                                                                \
	if ((gpMC6809->cc.bit.n = BTST6 (ARG)) != 0)                              \
    {                                                                         \
        BSET7 (ARG);                                                          \
    }                                                                         \
    gpMC6809->cc.bit.z = !(ARG);                                              \
}

#define OP_BR(TEST)                                                           \
{                                                                             \
    Byte br_bOffset;                                                          \
    MEM_READ_BYTE (br_bOffset);                                               \
    if (TEST)                                                                 \
    {                                                                         \
        if (br_bOffset & 0x80)                                                \
        {                                                                     \
            gpMC6809->pc += (Word)br_bOffset | 0xff00;                        \
        }                                                                     \
        else                                                                  \
        {                                                                     \
            gpMC6809->pc += (Word)br_bOffset;                                 \
        }                                                                     \
    }                                                                         \
}
 
#define OP_LBR(TEST, CYC)                                                     \
{                                                                             \
    Word lbr_wOffset;                                                         \
    MEM_READ_WORD (lbr_wOffset);                                              \
    if (TEST)                                                                 \
    {                                                                         \
        gpMC6809->pc += lbr_wOffset;                                          \
        *(CYC) = 1;                                                           \
    }                                                                         \
    else                                                                      \
    {                                                                         \
        *(CYC) = 0;                                                           \
    }                                                                         \
}
 
#define OP_BIT(REG)                                                           \
{                                                                             \
    Byte bit_bO;                                                              \
    bit_bO = MC6809ReadMemOperand ();                                         \
    bit_bO &= REG;                                                            \
    gpMC6809->cc.bit.n = BTST7 (bit_bO);                                      \
    gpMC6809->cc.bit.v = 0;                                                   \
    gpMC6809->cc.bit.z = !bit_bO;                                             \
}
 
#define OP_BIT_I(REG, CYC)                                                    \
{                                                                             \
    Byte bit_bO;                                                              \
    bit_bO = MC6809ReadMemIndexedOperand (CYC);                               \
    bit_bO &= REG;                                                            \
    gpMC6809->cc.bit.n = BTST7 (bit_bO);                                      \
    gpMC6809->cc.bit.v = 0;                                                   \
    gpMC6809->cc.bit.z = !bit_bO;                                             \
}
 
#define OP_CLR(BYTE)                                                          \
{                                                                             \
    gpMC6809->cc.all &= 0xf0;                                                 \
    gpMC6809->cc.all |= 0x04;                                                 \
    BYTE = 0;                                                                 \
}
 
#define OP_CMP_BYTE(BYTE)                                                     \
{                                                                             \
    Byte cmpbyte_bO;                                                          \
    Byte cmpbyte_bT;                                                          \
    Byte cmpbyte_bR;                                                          \
    cmpbyte_bO = MC6809ReadMemOperand ();                                     \
    cmpbyte_bR = BYTE - cmpbyte_bO;                                           \
    cmpbyte_bT = (BYTE ^ cmpbyte_bO) & 0x80 ? 1 : 0;                          \
    gpMC6809->cc.bit.n = BTST7 (cmpbyte_bR);                                  \
    gpMC6809->cc.bit.z = cmpbyte_bR ? 0 : 1;                                  \
    gpMC6809->cc.bit.v =                                                      \
        cmpbyte_bT & (((cmpbyte_bO ^ cmpbyte_bR) & 0x80) ? 0 : 1);            \
    gpMC6809->cc.bit.c =                                                      \
        ((((~BYTE) & cmpbyte_bO) & 0x80) ? 1 : 0) |                           \
        (!cmpbyte_bT & gpMC6809->cc.bit.n);                                   \
}
 
#define OP_CMP_I_BYTE(BYTE, CYC)                                              \
{                                                                             \
    Byte cmpbyte_bO;                                                          \
    Byte cmpbyte_bT;                                                          \
    Byte cmpbyte_bR;                                                          \
    cmpbyte_bO = MC6809ReadMemIndexedOperand (CYC);                           \
    cmpbyte_bR = BYTE - cmpbyte_bO;                                           \
    cmpbyte_bT = (BYTE ^ cmpbyte_bO) & 0x80 ? 1 : 0;                          \
    gpMC6809->cc.bit.n = BTST7 (cmpbyte_bR);                                  \
    gpMC6809->cc.bit.z = cmpbyte_bR ? 0 : 1;                                  \
    gpMC6809->cc.bit.v =                                                      \
        cmpbyte_bT & (((cmpbyte_bO ^ cmpbyte_bR) & 0x80) ? 0 : 1);            \
    gpMC6809->cc.bit.c =                                                      \
        ((((~BYTE) & cmpbyte_bO) & 0x80) ? 1 : 0) |                           \
        (!cmpbyte_bT & gpMC6809->cc.bit.n);                                   \
}
 
#define OP_CMP_WORD(WORD)                                                     \
{                                                                             \
    Word  cmpword_wO;                                                         \
    Word  cmpword_wT;                                                         \
    Word  cmpword_wR;                                                         \
    cmpword_wO = MC6809ReadMemWordOperand ();                                 \
    cmpword_wR = WORD - cmpword_wO;                                           \
    cmpword_wT = (WORD ^ cmpword_wO) & 0x8000 ? 1 : 0;                        \
    gpMC6809->cc.bit.n = BTST15 (cmpword_wR);                                 \
    gpMC6809->cc.bit.z = cmpword_wR ? 0 : 1;                                  \
    gpMC6809->cc.bit.v =                                                      \
        cmpword_wT & (((cmpword_wO ^ cmpword_wR) & 0x8000) ? 0 : 1);          \
    gpMC6809->cc.bit.c =                                                      \
        ((((~WORD) & cmpword_wO) & 0x8000) ? 1 : 0) |                         \
        (!cmpword_wT & gpMC6809->cc.bit.n);                                   \
}
 
#define OP_COM(BYTE)                                                          \
{                                                                             \
    (BYTE) = ~(BYTE);                                                         \
    gpMC6809->cc.bit.c = 1;                                                   \
    gpMC6809->cc.bit.v = 0;                                                   \
    gpMC6809->cc.bit.n = BTST7 (BYTE);                                        \
    gpMC6809->cc.bit.z = !(BYTE);                                             \
}
 
#define OP_DEC(BYTE)                                                          \
{                                                                             \
    gpMC6809->cc.bit.v = (BYTE == 0x80);                                      \
    BYTE -= 1;                                                                \
    gpMC6809->cc.bit.n = BTST7 (BYTE);                                        \
    gpMC6809->cc.bit.z = !(BYTE);                                             \
}
 
#define OP_EOR(BYTE)                                                          \
{                                                                             \
    (BYTE) ^= MC6809ReadMemOperand ();                                        \
    gpMC6809->cc.bit.v = 0;                                                   \
    gpMC6809->cc.bit.n = BTST7 (BYTE);                                        \
    gpMC6809->cc.bit.z = !(BYTE);                                             \
}
 
#define OP_EOR_I(BYTE, CYC)                                                   \
{                                                                             \
    (BYTE) ^= MC6809ReadMemIndexedOperand (CYC);                              \
    gpMC6809->cc.bit.v = 0;                                                   \
    gpMC6809->cc.bit.n = BTST7 (BYTE);                                        \
    gpMC6809->cc.bit.z = !(BYTE);                                             \
}
 
#define OP_INC(BYTE)                                                          \
{                                                                             \
    gpMC6809->cc.bit.v = (BYTE == 0x7f);                                      \
    BYTE += 1;                                                                \
    gpMC6809->cc.bit.n = BTST7 (BYTE);                                        \
    gpMC6809->cc.bit.z = !(BYTE);                                             \
}
 
#define OP_LD_BYTE(BYTE)                                                      \
{                                                                             \
    BYTE = MC6809ReadMemOperand ();                                           \
    gpMC6809->cc.bit.n = BTST7 (BYTE);                                        \
    gpMC6809->cc.bit.v = 0;                                                   \
    gpMC6809->cc.bit.z = !(BYTE);                                             \
}
 
#define OP_LD_I_BYTE(BYTE, CYC)                                               \
{                                                                             \
    BYTE = MC6809ReadMemIndexedOperand (CYC);                                 \
    gpMC6809->cc.bit.n = BTST7 (BYTE);                                        \
    gpMC6809->cc.bit.v = 0;                                                   \
    gpMC6809->cc.bit.z = !(BYTE);                                             \
}
 
#define OP_LD_WORD(WORD)                                                      \
{                                                                             \
    WORD = MC6809ReadMemWordOperand ();                                       \
    gpMC6809->cc.bit.n = WORD & 0x8000 ? 1 : 0;                               \
    gpMC6809->cc.bit.v = 0;                                                   \
    gpMC6809->cc.bit.z = !(WORD);                                             \
}
 
#define OP_LD_I_WORD(WORD, CYC)                                               \
{                                                                             \
    WORD = MC6809ReadMemIndexedWordOperand (CYC);                             \
    gpMC6809->cc.bit.n = WORD & 0x8000 ? 1 : 0;                               \
    gpMC6809->cc.bit.v = 0;                                                   \
    gpMC6809->cc.bit.z = !(WORD);                                             \
}
 
#define OP_LSL(BYTE)                                                          \
{                                                                             \
    gpMC6809->cc.bit.c = BTST7 (BYTE);                                        \
    gpMC6809->cc.bit.v = BTST7 (BYTE) ^ BTST6 (BYTE);                         \
    BYTE <<= 1;                                                               \
    gpMC6809->cc.bit.n = BTST7 (BYTE);                                        \
    gpMC6809->cc.bit.z = !(BYTE);                                             \
}
 
#define OP_LSR(BYTE)                                                          \
{                                                                             \
    gpMC6809->cc.bit.c = BTST0 (BYTE);                                        \
    BYTE >>= 1;                                                               \
    gpMC6809->cc.bit.n = 0;                                                   \
    gpMC6809->cc.bit.z = !(BYTE);                                             \
}
 
#define OP_NEG(BYTE)                                                          \
{                                                                             \
    gpMC6809->cc.bit.v = (BYTE == 0x80);                                      \
    gpMC6809->cc.bit.c = (BYTE != 0);                                         \
    BYTE = ((~BYTE) + 1) & 0xff;                                              \
    gpMC6809->cc.bit.n = BTST7 (BYTE);                                        \
    gpMC6809->cc.bit.z = !(BYTE);                                             \
}
 
#define OP_OR(BYTE)                                                           \
{                                                                             \
    BYTE |= MC6809ReadMemOperand ();                                          \
    gpMC6809->cc.bit.v = 0;                                                   \
    gpMC6809->cc.bit.n = BTST7 (BYTE);                                        \
    gpMC6809->cc.bit.z = !(BYTE);                                             \
}
 
#define OP_OR_I(BYTE, CYC)                                                    \
{                                                                             \
    BYTE |= MC6809ReadMemIndexedOperand (&nCycles);                           \
    gpMC6809->cc.bit.v = 0;                                                   \
    gpMC6809->cc.bit.n = BTST7 (BYTE);                                        \
    gpMC6809->cc.bit.z = !(BYTE);                                             \
}
 
#define OP_PSH(BYTE, WORDS, WORDU)                                            \
{                                                                             \
    switch ((Byte)(BYTE & 0xf0))                                              \
    {                                                                         \
        case 0xf0:                                                            \
            WORDS -= 2;                                                       \
            MEM_WRITE_BYTE (WORDS, (Byte)(gpMC6809->pc >> 8));                \
            MEM_WRITE_BYTE (WORDS + 1, (Byte)(gpMC6809->pc));                 \
        case 0x70:                                                            \
            WORDS -= 2;                                                       \
            MEM_WRITE_BYTE (WORDS, (Byte)(WORDU >> 8));                       \
            MEM_WRITE_BYTE (WORDS + 1, (Byte)WORDU);                          \
        case 0x30:                                                            \
            WORDS -= 2;                                                       \
            MEM_WRITE_BYTE (WORDS, (Byte)(gpMC6809->y >> 8));                 \
            MEM_WRITE_BYTE (WORDS + 1, (Byte)(gpMC6809->y));                  \
        case 0x10:                                                            \
            WORDS -= 2;                                                       \
            MEM_WRITE_BYTE (WORDS, (Byte)(gpMC6809->x >> 8));                 \
            MEM_WRITE_BYTE (WORDS + 1, (Byte)(gpMC6809->x));                  \
        case 0x00:                                                            \
            break;                                                            \
        case 0xe0:                                                            \
            WORDS -= 2;                                                       \
            MEM_WRITE_BYTE (WORDS, (Byte)(gpMC6809->pc >> 8));                \
            MEM_WRITE_BYTE (WORDS + 1, (Byte)(gpMC6809->pc));                 \
        case 0x60:                                                            \
            WORDS -= 2;                                                       \
            MEM_WRITE_BYTE (WORDS, (Byte)(WORDU >> 8));                       \
            MEM_WRITE_BYTE (WORDS + 1, (Byte)WORDU);                          \
        case 0x20:                                                            \
            WORDS -= 2;                                                       \
            MEM_WRITE_BYTE (WORDS, (Byte)(gpMC6809->y >> 8));                 \
            MEM_WRITE_BYTE (WORDS + 1, (Byte)(gpMC6809->y));                  \
            break;                                                            \
        case 0xd0:                                                            \
            WORDS -= 2;                                                       \
            MEM_WRITE_BYTE (WORDS, (Byte)(gpMC6809->pc >> 8));                \
            MEM_WRITE_BYTE (WORDS + 1, (Byte)(gpMC6809->pc));                 \
        case 0x50:                                                            \
            WORDS -= 2;                                                       \
            MEM_WRITE_BYTE (WORDS, (Byte)(WORDU >> 8));                       \
            MEM_WRITE_BYTE (WORDS + 1, (Byte)WORDU);                          \
            WORDS -= 2;                                                       \
            MEM_WRITE_BYTE (WORDS, (Byte)(gpMC6809->x >> 8));                 \
            MEM_WRITE_BYTE (WORDS + 1, (Byte)(gpMC6809->x));                  \
            break;                                                            \
        case 0xc0:                                                            \
            WORDS -= 2;                                                       \
            MEM_WRITE_BYTE (WORDS, (Byte)(gpMC6809->pc >> 8));                \
            MEM_WRITE_BYTE (WORDS + 1, (Byte)(gpMC6809->pc));                 \
        case 0x40:                                                            \
            WORDS -= 2;                                                       \
            MEM_WRITE_BYTE (WORDS, (Byte)(WORDU >> 8));                       \
            MEM_WRITE_BYTE (WORDS + 1, (Byte)WORDU);                          \
            break;                                                            \
        case 0xb0:                                                            \
            WORDS -= 2;                                                       \
            MEM_WRITE_BYTE (WORDS, (Byte)(gpMC6809->pc >> 8));                \
            MEM_WRITE_BYTE (WORDS + 1, (Byte)(gpMC6809->pc));                 \
            WORDS -= 2;                                                       \
            MEM_WRITE_BYTE (WORDS, (Byte)(gpMC6809->y >> 8));                 \
            MEM_WRITE_BYTE (WORDS + 1, (Byte)(gpMC6809->y));                  \
            WORDS -= 2;                                                       \
            MEM_WRITE_BYTE (WORDS, (Byte)(gpMC6809->x >> 8));                 \
            MEM_WRITE_BYTE (WORDS + 1, (Byte)(gpMC6809->x));                  \
            break;                                                            \
        case 0xa0:                                                            \
            WORDS -= 2;                                                       \
            MEM_WRITE_BYTE (WORDS, (Byte)(gpMC6809->pc >> 8));                \
            MEM_WRITE_BYTE (WORDS + 1, (Byte)(gpMC6809->pc));                 \
            WORDS -= 2;                                                       \
            MEM_WRITE_BYTE (WORDS, (Byte)(gpMC6809->y >> 8));                 \
            MEM_WRITE_BYTE (WORDS + 1, (Byte)(gpMC6809->y));                  \
            break;                                                            \
        case 0x90:                                                            \
            WORDS -= 2;                                                       \
            MEM_WRITE_BYTE (WORDS, (Byte)(gpMC6809->pc >> 8));                \
            MEM_WRITE_BYTE (WORDS + 1, (Byte)(gpMC6809->pc));                 \
            WORDS -= 2;                                                       \
            MEM_WRITE_BYTE (WORDS, (Byte)(gpMC6809->x >> 8));                 \
            MEM_WRITE_BYTE (WORDS + 1, (Byte)(gpMC6809->x));                  \
            break;                                                            \
        case 0x80:                                                            \
            WORDS -= 2;                                                       \
            MEM_WRITE_BYTE (WORDS, (Byte)(gpMC6809->pc >> 8));                \
            MEM_WRITE_BYTE (WORDS + 1, (Byte)(gpMC6809->pc));                 \
            break;                                                            \
    }                                                                         \
    switch ((Byte)(BYTE & 0x0f))                                              \
    {                                                                         \
        case 0x0f:                                                            \
            WORDS -= 1;                                                       \
            MEM_WRITE_BYTE (WORDS, gpMC6809->dp);                             \
        case 0x07:                                                            \
            WORDS -= 1;                                                       \
            MEM_WRITE_BYTE (WORDS, gpMC6809->acc.B.l);                        \
        case 0x03:                                                            \
            WORDS -= 1;                                                       \
            MEM_WRITE_BYTE (WORDS, gpMC6809->acc.B.h);                        \
        case 0x01:                                                            \
            WORDS -= 1;                                                       \
            MEM_WRITE_BYTE (WORDS, gpMC6809->cc.all);                         \
        case 0x00:                                                            \
            break;                                                            \
        case 0x0e:                                                            \
            WORDS -= 1;                                                       \
            MEM_WRITE_BYTE (WORDS, gpMC6809->dp);                             \
        case 0x06:                                                            \
            WORDS -= 1;                                                       \
            MEM_WRITE_BYTE (WORDS, gpMC6809->acc.B.l);                        \
        case 0x02:                                                            \
            WORDS -= 1;                                                       \
            MEM_WRITE_BYTE (WORDS, gpMC6809->acc.B.h);                        \
            break;                                                            \
        case 0x0d:                                                            \
            WORDS -= 1;                                                       \
            MEM_WRITE_BYTE (WORDS, gpMC6809->dp);                             \
        case 0x05:                                                            \
            WORDS -= 1;                                                       \
            MEM_WRITE_BYTE (WORDS, gpMC6809->acc.B.l);                        \
            WORDS -= 1;                                                       \
            MEM_WRITE_BYTE (WORDS, gpMC6809->cc.all);                         \
            break;                                                            \
        case 0x0c:                                                            \
            WORDS -= 1;                                                       \
            MEM_WRITE_BYTE (WORDS, gpMC6809->dp);                             \
        case 0x04:                                                            \
            WORDS -= 1;                                                       \
            MEM_WRITE_BYTE (WORDS, gpMC6809->acc.B.l);                        \
            break;                                                            \
        case 0x0b:                                                            \
            WORDS -= 1;                                                       \
            MEM_WRITE_BYTE (WORDS, gpMC6809->dp);                             \
            WORDS -= 1;                                                       \
            MEM_WRITE_BYTE (WORDS, gpMC6809->acc.B.h);                        \
            WORDS -= 1;                                                       \
            MEM_WRITE_BYTE (WORDS, gpMC6809->cc.all);                         \
            break;                                                            \
        case 0x09:                                                            \
            WORDS -= 1;                                                       \
            MEM_WRITE_BYTE (WORDS, gpMC6809->dp);                             \
            WORDS -= 1;                                                       \
            MEM_WRITE_BYTE (WORDS, gpMC6809->cc.all);                         \
            break;                                                            \
        case 0x0a:                                                            \
            WORDS -= 1;                                                       \
            MEM_WRITE_BYTE (WORDS, gpMC6809->dp);                             \
            WORDS -= 1;                                                       \
            MEM_WRITE_BYTE (WORDS, gpMC6809->acc.B.h);                        \
            break;                                                            \
        case 0x08:                                                            \
            WORDS -= 1;                                                       \
            MEM_WRITE_BYTE (WORDS, gpMC6809->dp);                             \
            break;                                                            \
    }                                                                         \
}
 
#define OP_PUL(BYTE, WORDS, WORDU)                                            \
{                                                                             \
    switch ((Byte)(BYTE & 0x0f))                                              \
    {                                                                         \
        case 0x0f:                                                            \
            MEM_READ_BYTE_NC (gpMC6809->cc.all, WORDS);                       \
            WORDS += 1;                                                       \
        case 0x0e:                                                            \
            MEM_READ_BYTE_NC (gpMC6809->acc.B.h, WORDS);                      \
            WORDS += 1;                                                       \
        case 0x0c:                                                            \
            MEM_READ_BYTE_NC (gpMC6809->acc.B.l, WORDS);                      \
            WORDS += 1;                                                       \
        case 0x08:                                                            \
            MEM_READ_BYTE_NC (gpMC6809->dp, WORDS);                           \
            WORDS += 1;                                                       \
            gpMC6809->dpHi = (Word)(gpMC6809->dp << 8);                       \
        case 0x00:                                                            \
            break;                                                            \
        case 0x07:                                                            \
            MEM_READ_BYTE_NC (gpMC6809->cc.all, WORDS);                       \
            WORDS += 1;                                                       \
        case 0x06:                                                            \
            MEM_READ_BYTE_NC (gpMC6809->acc.B.h, WORDS);                      \
            WORDS += 1;                                                       \
        case 0x04:                                                            \
            MEM_READ_BYTE_NC (gpMC6809->acc.B.l, WORDS);                      \
            WORDS += 1;                                                       \
            break;                                                            \
        case 0x0b:                                                            \
            MEM_READ_BYTE_NC (gpMC6809->cc.all, WORDS);                       \
            WORDS += 1;                                                       \
        case 0x0a:                                                            \
            MEM_READ_BYTE_NC (gpMC6809->acc.B.h, WORDS);                      \
            WORDS += 1;                                                       \
            MEM_READ_BYTE_NC (gpMC6809->dp, WORDS);                           \
            WORDS += 1;                                                       \
            gpMC6809->dpHi = (Word)(gpMC6809->dp << 8);                       \
            break;                                                            \
        case 0x03:                                                            \
            MEM_READ_BYTE_NC (gpMC6809->cc.all, WORDS);                       \
            WORDS += 1;                                                       \
        case 0x02:                                                            \
            MEM_READ_BYTE_NC (gpMC6809->acc.B.h, WORDS);                      \
            WORDS += 1;                                                       \
            break;                                                            \
        case 0x0d:                                                            \
            MEM_READ_BYTE_NC (gpMC6809->cc.all, WORDS);                       \
            WORDS += 1;                                                       \
            MEM_READ_BYTE_NC (gpMC6809->acc.B.l, WORDS);                      \
            WORDS += 1;                                                       \
            MEM_READ_BYTE_NC (gpMC6809->dp, WORDS);                           \
            WORDS += 1;                                                       \
            gpMC6809->dpHi = (Word)(gpMC6809->dp << 8);                       \
            break;                                                            \
        case 0x09:                                                            \
            MEM_READ_BYTE_NC (gpMC6809->cc.all, WORDS);                       \
            WORDS += 1;                                                       \
            MEM_READ_BYTE_NC (gpMC6809->dp, WORDS);                           \
            WORDS += 1;                                                       \
            gpMC6809->dpHi = (Word)(gpMC6809->dp << 8);                       \
            break;                                                            \
        case 0x05:                                                            \
            MEM_READ_BYTE_NC (gpMC6809->cc.all, WORDS);                       \
            WORDS += 1;                                                       \
            MEM_READ_BYTE_NC (gpMC6809->acc.B.l, WORDS);                      \
            WORDS += 1;                                                       \
            break;                                                            \
        case 0x01:                                                            \
            MEM_READ_BYTE_NC (gpMC6809->cc.all, WORDS);                       \
            WORDS += 1;                                                       \
            break;                                                            \
    }                                                                         \
    switch ((Byte)(BYTE & 0xf0))                                              \
    {                                                                         \
        case 0xf0:                                                            \
            MEM_READ_WORD_NC (gpMC6809->x, WORDS);                            \
            WORDS += 2;                                                       \
        case 0xe0:                                                            \
            MEM_READ_WORD_NC (gpMC6809->y, WORDS);                            \
            WORDS += 2;                                                       \
        case 0xc0:                                                            \
            MEM_READ_WORD_NC (WORDU, WORDS);                                  \
            WORDS += 2;                                                       \
        case 0x80:                                                            \
            MEM_READ_WORD_NC (gpMC6809->pc, WORDS);                           \
            WORDS += 2;                                                       \
        case 0x00:                                                            \
            break;                                                            \
        case 0x70:                                                            \
            MEM_READ_WORD_NC (gpMC6809->x, WORDS);                            \
            WORDS += 2;                                                       \
        case 0x60:                                                            \
            MEM_READ_WORD_NC (gpMC6809->y, WORDS);                            \
            WORDS += 2;                                                       \
        case 0x40:                                                            \
            MEM_READ_WORD_NC (WORDU, WORDS);                                  \
            WORDS += 2;                                                       \
            break;                                                            \
        case 0xb0:                                                            \
            MEM_READ_WORD_NC (gpMC6809->x, WORDS);                            \
            WORDS += 2;                                                       \
        case 0xa0:                                                            \
            MEM_READ_WORD_NC (gpMC6809->y, WORDS);                            \
            WORDS += 2;                                                       \
            MEM_READ_WORD_NC (gpMC6809->pc, WORDS);                           \
            WORDS += 2;                                                       \
            break;                                                            \
        case 0x30:                                                            \
            MEM_READ_WORD_NC (gpMC6809->x, WORDS);                            \
            WORDS += 2;                                                       \
        case 0x20:                                                            \
            MEM_READ_WORD_NC (gpMC6809->y, WORDS);                            \
            WORDS += 2;                                                       \
            break;                                                            \
        case 0xd0:                                                            \
            MEM_READ_WORD_NC (gpMC6809->x, WORDS);                            \
            WORDS += 2;                                                       \
            MEM_READ_WORD_NC (WORDU, WORDS);                                  \
            WORDS += 2;                                                       \
            MEM_READ_WORD_NC (gpMC6809->pc, WORDS);                           \
            WORDS += 2;                                                       \
            break;                                                            \
        case 0x90:                                                            \
            MEM_READ_WORD_NC (gpMC6809->x, WORDS);                            \
            WORDS += 2;                                                       \
            MEM_READ_WORD_NC (gpMC6809->pc, WORDS);                           \
            WORDS += 2;                                                       \
            break;                                                            \
        case 0x50:                                                            \
            MEM_READ_WORD_NC (gpMC6809->x, WORDS);                            \
            WORDS += 2;                                                       \
            MEM_READ_WORD_NC (WORDU, WORDS);                                  \
            WORDS += 2;                                                       \
            break;                                                            \
        case 0x10:                                                            \
            MEM_READ_WORD_NC (gpMC6809->x, WORDS);                            \
            WORDS += 2;                                                       \
            break;                                                            \
    }                                                                         \
}
 
#define OP_ROL(BYTE)                                                          \
{                                                                             \
    int32 rol_nOC = gpMC6809->cc.bit.c;                                       \
    gpMC6809->cc.bit.c = BTST7 (BYTE);                                        \
    gpMC6809->cc.bit.n = BTST6 (BYTE);                                        \
    gpMC6809->cc.bit.v = gpMC6809->cc.bit.c ^ gpMC6809->cc.bit.n;             \
    BYTE <<= 1;                                                               \
    if (rol_nOC)                                                              \
    {                                                                         \
        BSET0 (BYTE);                                                         \
    }                                                                         \
    gpMC6809->cc.bit.z = !(BYTE);                                             \
}
 
#define OP_ROR(BYTE)                                                          \
{                                                                             \
    int32 ror_nOC = gpMC6809->cc.bit.c;                                       \
    gpMC6809->cc.bit.c = BTST0 (BYTE);                                        \
    BYTE >>= 1;                                                               \
    if (ror_nOC)                                                              \
    {                                                                         \
        BSET7 (BYTE);                                                         \
    }                                                                         \
    gpMC6809->cc.bit.n = BTST7 (BYTE);                                        \
    gpMC6809->cc.bit.z = !(BYTE);                                             \
}
 
#define OP_SBC(ARG)                                                           \
{                                                                             \
    Byte sbc_bO;                                                              \
    Byte sbc_bT;                                                              \
    Byte sbc_bR;                                                              \
    sbc_bO = MC6809ReadMemOperand ();                                         \
    sbc_bR = ARG - sbc_bO - (gpMC6809->cc.all & 0x01);                        \
    sbc_bT = (ARG ^ sbc_bO) & 0x80 ? 1 : 0;                                   \
    gpMC6809->cc.bit.n = (sbc_bR & 0x80) ? 1 : 0;                             \
    gpMC6809->cc.bit.z = sbc_bR ? 0 : 1;                                      \
    gpMC6809->cc.bit.v =                                                      \
        sbc_bT & (((sbc_bO ^ sbc_bR) & 0x80) ? 0 : 1);                        \
    gpMC6809->cc.bit.c =                                                      \
	    ((((~ARG) & sbc_bO) & 0x80) ? 1 : 0) |                                \
        (!sbc_bT & gpMC6809->cc.bit.n);                                       \
    ARG = sbc_bR;                                                             \
}

#define OP_SBC_I(ARG, CYC)                                                    \
{                                                                             \
    Byte sbc_bO;                                                              \
    Byte sbc_bT;                                                              \
    Byte sbc_bR;                                                              \
    sbc_bO = MC6809ReadMemIndexedOperand (CYC);                               \
    sbc_bR = ARG - sbc_bO - (gpMC6809->cc.all & 0x01);                        \
    sbc_bT = (ARG ^ sbc_bO) & 0x80 ? 1 : 0;                                   \
    gpMC6809->cc.bit.n = (sbc_bR & 0x80) ? 1 : 0;                             \
    gpMC6809->cc.bit.z = sbc_bR ? 0 : 1;                                      \
    gpMC6809->cc.bit.v =                                                      \
        sbc_bT & (((sbc_bO ^ sbc_bR) & 0x80) ? 0 : 1);                        \
    gpMC6809->cc.bit.c =                                                      \
	    ((((~ARG) & sbc_bO) & 0x80) ? 1 : 0) |                                \
        (!sbc_bT & gpMC6809->cc.bit.n);                                       \
    ARG = sbc_bR;                                                             \
}

#define OP_ST_BYTE(BYTE)                                                      \
{                                                                             \
    Word stbyte_wAddress;                                                     \
    stbyte_wAddress = MC6809ReadMemEA ();                                     \
    MEM_WRITE_BYTE (stbyte_wAddress, BYTE);                                   \
	gpMC6809->cc.bit.v = 0;                                                   \
    gpMC6809->cc.bit.n = BTST7 (BYTE);                                        \
    gpMC6809->cc.bit.z = !(BYTE);                                             \
}

#define OP_ST_I_BYTE(BYTE, CYC)                                               \
{                                                                             \
    Word stbyte_wAddress;                                                     \
    stbyte_wAddress = MC6809ReadMemIA (CYC);                                  \
    MEM_WRITE_BYTE (stbyte_wAddress, BYTE);                                   \
	gpMC6809->cc.bit.v = 0;                                                   \
    gpMC6809->cc.bit.n = BTST7 (BYTE);                                        \
    gpMC6809->cc.bit.z = !(BYTE);                                             \
}

#define OP_ST_WORD(WORD)                                                      \
{                                                                             \
    Word stword_wAddress;                                                     \
    stword_wAddress = MC6809ReadMemEA ();                                     \
    MEM_WRITE_WORD (stword_wAddress, WORD);                                   \
	gpMC6809->cc.bit.v = 0;                                                   \
    gpMC6809->cc.bit.n = BTST15 (WORD);                                       \
    gpMC6809->cc.bit.z = !(WORD);                                             \
}

#define OP_ST_I_WORD(WORD, CYC)                                               \
{                                                                             \
    Word stword_wAddress;                                                     \
    stword_wAddress = MC6809ReadMemIA (CYC);                                  \
    MEM_WRITE_WORD (stword_wAddress, WORD);                                   \
	gpMC6809->cc.bit.v = 0;                                                   \
    gpMC6809->cc.bit.n = WORD & 0x8000 ? 1 : 0;                               \
    gpMC6809->cc.bit.z = !(WORD);                                             \
}

#define OP_SUB(ARG)                                                           \
{                                                                             \
    Byte sub_bO;                                                              \
    Byte sub_bT;                                                              \
    Byte sub_bR;                                                              \
    sub_bO = MC6809ReadMemOperand ();                                         \
    sub_bR = ARG - sub_bO;                                                    \
    sub_bT = (ARG ^ sub_bO) & 0x80 ? 1 : 0;                                   \
    gpMC6809->cc.bit.n = (sub_bR & 0x80) ? 1 : 0;                             \
    gpMC6809->cc.bit.z = sub_bR ? 0 : 1;                                      \
    gpMC6809->cc.bit.v =                                                      \
        sub_bT & (((sub_bO ^ sub_bR) & 0x80) ? 0 : 1);                        \
	gpMC6809->cc.bit.c =                                                      \
        ((((~ARG) & sub_bO) & 0x80) ? 1 : 0) |                                \
        (!sub_bT & gpMC6809->cc.bit.n);                                       \
    ARG = sub_bR;                                                             \
}

#define OP_SUB_I(ARG, CYC)                                                    \
{                                                                             \
    Byte sub_bO;                                                              \
    Byte sub_bT;                                                              \
    Byte sub_bR;                                                              \
    sub_bO = MC6809ReadMemIndexedOperand (CYC);                               \
    sub_bR = ARG - sub_bO;                                                    \
    sub_bT = (ARG ^ sub_bO) & 0x80 ? 1 : 0;                                   \
    gpMC6809->cc.bit.n = (sub_bR & 0x80) ? 1 : 0;                             \
    gpMC6809->cc.bit.z = sub_bR ? 0 : 1;                                      \
    gpMC6809->cc.bit.v =                                                      \
        sub_bT & (((sub_bO ^ sub_bR) & 0x80) ? 0 : 1);                        \
	gpMC6809->cc.bit.c =                                                      \
        ((((~ARG) & sub_bO) & 0x80) ? 1 : 0) |                                \
        (!sub_bT & gpMC6809->cc.bit.n);                                       \
    ARG = sub_bR;                                                             \
}

#define OP_TST(BYTE)                                                          \
{                                                                             \
    gpMC6809->cc.bit.v = 0;                                                   \
    gpMC6809->cc.bit.n = BTST7 (BYTE);                                        \
    gpMC6809->cc.bit.z = !(BYTE);                                             \
}




/******************************************************************************
*  Function Prototypes.
******************************************************************************/
static
void
MC6809FillStaticTables (
    void
);

static
void
MC6809DoInterrupt (
    void
);

static
Byte
MC6809ReadMemOperand (
    void
);

static
Byte
MC6809ReadMemIndexedOperand (
    int32 *pnCycles
);

static
Word
MC6809ReadMemWordOperand (
    void
);

static
Word
MC6809ReadMemIndexedWordOperand (
    int32 *pnCycles
);

static
Word
MC6809ReadMemEA (
    void
);

static
Word
MC6809ReadMemIA (
    int32 *pnCycles
);

static void          mc6809err         (void);
static void          mc6809eerr        (void);
static void          mc6809ABX         (void);
static void          mc6809ADCA        (void);
static void          mc6809ADCB        (void);
static int32         mc6809ADCA_I      (void);
static int32         mc6809ADCB_I      (void);
static void          mc6809ADDA        (void);
static void          mc6809ADDB        (void);
static int32         mc6809ADDA_I      (void);
static int32         mc6809ADDB_I      (void);
static void          mc6809ADDD        (void);
static int32         mc6809ADDD_I      (void);
static void          mc6809ANDA        (void);
static void          mc6809ANDB        (void);
static void          mc6809ANDCC       (void);
static int32         mc6809ANDA_I      (void);
static int32         mc6809ANDB_I      (void);
static void          mc6809ASRA        (void);
static void          mc6809ASRB        (void);
static void          mc6809ASR         (void);
static int32         mc6809ASR_I       (void);
static void          mc6809BCC         (void);
static int32         mc6809LBCC        (void);
static void          mc6809BCS         (void);
static int32         mc6809LBCS        (void);
static void          mc6809BEQ         (void);
static int32         mc6809LBEQ        (void);
static void          mc6809BGE         (void);
static int32         mc6809LBGE        (void);
static void          mc6809BGT         (void);
static int32         mc6809LBGT        (void);
static void          mc6809BHI         (void);
static int32         mc6809LBHI        (void);
static void          mc6809BITA        (void);
static void          mc6809BITB        (void);
static int32         mc6809BITA_I      (void);
static int32         mc6809BITB_I      (void);
static void          mc6809BLE         (void);
static int32         mc6809LBLE        (void);
static void          mc6809BLS         (void);
static int32         mc6809LBLS        (void);
static void          mc6809BLT         (void);
static int32         mc6809LBLT        (void);
static void          mc6809BMI         (void);
static int32         mc6809LBMI        (void);
static void          mc6809BNE         (void);
static int32         mc6809LBNE        (void);
static void          mc6809BPL         (void);
static int32         mc6809LBPL        (void);
static void          mc6809BRA         (void);
static int32         mc6809LBRA        (void);
static void          mc6809BRN         (void);
static int32         mc6809LBRN        (void);
static void          mc6809BSR         (void);
static int32         mc6809LBSR        (void);
static void          mc6809BVC         (void);
static int32         mc6809LBVC        (void);
static void          mc6809BVS         (void);
static int32         mc6809LBVS        (void);
static void          mc6809CLRA        (void);
static void          mc6809CLRB        (void);
static void          mc6809CLR         (void);
static int32         mc6809CLR_I       (void);
static void          mc6809CMPA        (void);
static void          mc6809CMPB        (void);
static int32         mc6809CMPA_I      (void);
static int32         mc6809CMPB_I      (void);
static void          mc6809CMPD        (void);
static void          mc6809CMPX        (void);
static void          mc6809CMPY        (void);
static void          mc6809CMPU        (void);
static void          mc6809CMPS        (void);
static int32         mc6809CMPD_I      (void);
static int32         mc6809CMPX_I      (void);
static int32         mc6809CMPY_I      (void);
static int32         mc6809CMPU_I      (void);
static int32         mc6809CMPS_I      (void);
static void          mc6809COMA        (void);
static void          mc6809COMB        (void);
static void          mc6809COM         (void);
static int32         mc6809COM_I       (void);
static void          mc6809CWAI        (void);
static void          mc6809DAA         (void);
static void          mc6809DECA        (void);
static void          mc6809DECB        (void);
static void          mc6809DEC         (void);
static int32         mc6809DEC_I       (void);
static void          mc6809EORA        (void);
static void          mc6809EORB        (void);
static int32         mc6809EORA_I      (void);
static int32         mc6809EORB_I      (void);
static void          mc6809EXG         (void);
static void          mc6809INCA        (void);
static void          mc6809INCB        (void);
static void          mc6809INC         (void);
static int32         mc6809INC_I       (void);
static void          mc6809JMP         (void);
static int32         mc6809JMP_I       (void);
static void          mc6809JSR         (void);
static int32         mc6809JSR_I       (void);
static void          mc6809LDA         (void);
static void          mc6809LDB         (void);
static int32         mc6809LDA_I       (void);
static int32         mc6809LDB_I       (void);
static void          mc6809LDD         (void);
static void          mc6809LDX         (void);
static void          mc6809LDY         (void);
static void          mc6809LDS         (void);
static void          mc6809LDU         (void);
static int32         mc6809LDD_I       (void);
static int32         mc6809LDX_I       (void);
static int32         mc6809LDY_I       (void);
static int32         mc6809LDS_I       (void);
static int32         mc6809LDU_I       (void);
static int32         mc6809LEAX        (void);
static int32         mc6809LEAY        (void);
static int32         mc6809LEAS        (void);
static int32         mc6809LEAU        (void);
static void          mc6809LSLA        (void);
static void          mc6809LSLB        (void);
static void          mc6809LSL         (void);
static int32         mc6809LSL_I       (void);
static void          mc6809LSRA        (void);
static void          mc6809LSRB        (void);
static void          mc6809LSR         (void);
static int32         mc6809LSR_I       (void);
static void          mc6809MUL         (void);
static void          mc6809NEGA        (void);
static void          mc6809NEGB        (void);
static void          mc6809NEG         (void);
static int32         mc6809NEG_I       (void);
static void          mc6809NOP         (void);
static void          mc6809ORA         (void);
static void          mc6809ORB         (void);
static int32         mc6809ORA_I       (void);
static int32         mc6809ORB_I       (void);
static void          mc6809ORCC        (void);
static int32         mc6809PSHS        (void);
static int32         mc6809PSHU        (void);
static int32         mc6809PULS        (void);
static int32         mc6809PULU        (void);
static void          mc6809ROLA        (void);
static void          mc6809ROLB        (void);
static void          mc6809ROL         (void);
static int32         mc6809ROL_I       (void);
static void          mc6809RORA        (void);
static void          mc6809RORB        (void);
static void          mc6809ROR         (void);
static int32         mc6809ROR_I       (void);
static int32         mc6809RTI         (void);
static void          mc6809RTS         (void);
static void          mc6809SBCA        (void);
static void          mc6809SBCB        (void);
static int32         mc6809SBCA_I      (void);
static int32         mc6809SBCB_I      (void);
static void          mc6809SEX         (void);
static void          mc6809STA         (void);
static void          mc6809STB         (void);
static int32         mc6809STA_I       (void);
static int32         mc6809STB_I       (void);
static void          mc6809STD         (void);
static void          mc6809STX         (void);
static void          mc6809STY         (void);
static void          mc6809STS         (void);
static void          mc6809STU         (void);
static int32         mc6809STD_I       (void);
static int32         mc6809STX_I       (void);
static int32         mc6809STY_I       (void);
static int32         mc6809STS_I       (void);
static int32         mc6809STU_I       (void);
static void          mc6809SUBA        (void);
static void          mc6809SUBB        (void);
static int32         mc6809SUBA_I      (void);
static int32         mc6809SUBB_I      (void);
static void          mc6809SUBD        (void);
static int32         mc6809SUBD_I      (void);
static void          mc6809SWI         (void);
static void          mc6809SWI2        (void);
static void          mc6809SWI3        (void);
static void          mc6809SYNC        (void);
static void          mc6809TFR         (void);
static void          mc6809TSTA        (void);
static void          mc6809TSTB        (void);
static void          mc6809TST         (void);
static int32         mc6809TST_I       (void);


/******************************************************************************
*  Globals.
******************************************************************************/
/*
 *  To avoid passing things all around, we use a global structure to store
 *  the CPU information.
 */
MC6809 *gpMC6809;

/*
 *  The address mode of the current instruction.
 */
static MC6809AddressMode eAM;

/*
 *  The following are the two bytes used to read the opcodes as the 6809
 *  executes.
 */
static Byte op1;
static Byte op2;

/*
 *  The following tables hold the number of cycles for the indexed operands
 *  and the psh/pul operations.
 */
static Byte anIndexedCycles[0x100];
static Byte anPSHPULCycles[0x100];

    
/******************************************************************************
*
*  Function: MC6809FillStaticTables
*
*  Description:
*
*      This function is called to fill in the values for the static tables.
*
*  Parameters:
*
*      None.
*
*  Returns:
*      Nothing.
*
******************************************************************************/
static
void
MC6809FillStaticTables (
    void
)
{
    /*
     *  Indicates if the tables have been filled.
     */ 
    static Byte bTablesFilled = 0;

    /*
     *  Indices used for iterations.
     */
    int32 nI;
  
    /*
     *  The number of cycles taken by an instruction.
     */
    Byte nCycles;


    /* 
     *  If the tables have been filled then get out.
     */
    if (bTablesFilled)
    {
        return;
    }
    else
    {
        bTablesFilled = 1;
    }

    /*
     *  Set the number of cycles for the indexed instructions.
     */
    for (nI = 0 ; nI < 128 ; nI += 1)
    {
        anIndexedCycles[nI] = 1;
    }
    for (nI = 128 ; nI < 256 ; nI += 1)
    {
        switch ((Byte)(nI & 0x1f))
        {
            case 0x05:
            case 0x06:
            case 0x08:
            case 0x0c:
            {
                anIndexedCycles[nI] = 1;
                break;
            }
            case 0x00:
            case 0x02:
            {
                anIndexedCycles[nI] = 2;
                break;
            }
            case 0x01:
            case 0x03:
            case 0x14:
            {
                anIndexedCycles[nI] = 3;
                break;
            }
            case 0x09:
            case 0x0b:
            case 0x18:
            case 0x15:
            case 0x16:
            case 0x1c:
            {
                anIndexedCycles[nI] = 4;
                break;
            }
            case 0x0d:
            case 0x1f:
            {
                anIndexedCycles[nI] = 5;
                break;
            }
            case 0x11:
            case 0x13:
            {
                anIndexedCycles[nI] = 6;
                break;
            }
            case 0x19:
            case 0x1b:
            {
                anIndexedCycles[nI] = 7;
                break;
            }
            case 0x1d:
            {
                anIndexedCycles[nI] = 8;
                break;
            }
            default:
            {
                anIndexedCycles[nI] = 0;
                break;
            }
        }
    }

    /*
     *  Now fill in the table of cycles for the various push and pul operations.
     */
    for (nI = 0 ; nI < 256 ; nI += 1)
    {
        nCycles = 0;
        if (nI & 0x01) 
        {
            nCycles += 1;
        }
        if (nI & 0x02) 
        {
            nCycles += 1;
        }
        if (nI & 0x04) 
        {
            nCycles += 1;
        }
        if (nI & 0x08) 
        {
            nCycles += 1;
        }
        if (nI & 0x10) 
        {
            nCycles += 2;
        }
        if (nI & 0x20) 
        {
            nCycles += 2;
        }
        if (nI & 0x40) 
        {
            nCycles += 2;
        }
        if (nI & 0x80) 
        {
            nCycles += 2;
        }
        anPSHPULCycles[nI] = nCycles;
    }
}



/******************************************************************************
*
*  Function: MC6809Reset
*
*  Description:
*
*      This function resets a 6809 object.
*
*  Parameters:
*
*      pMC6809 (input/output)
*          The context of the CPU to reset.
*
*  Returns:
*
*      Nothing.
*
******************************************************************************/
void
MC6809Reset (
    MC6809 *pMC6809
)
{
    /*
     *  Fill the static tables.
     */
    MC6809FillStaticTables ();

    /*
     *  Clear the context.
     */
    memset ((void *)pMC6809, 0x00, sizeof (MC6809));

    /*
     *  The Reset vector (start address) is located at 0xFFFE/0xFFFF.
     */
    MEM_READ_WORD_NC (pMC6809->pc, 0xfffe);

    /*
     *  IRQ and FIRQ are initially disabled.
     */
    pMC6809->cc.bit.i = 1;
    pMC6809->cc.bit.f = 1;

    /*
     *  Set the dp convenience value.
     */
    pMC6809->dpHi = (Word)(pMC6809->dp << 8);
}



/******************************************************************************
*
*  Function: MC6809DoInterrupt
*
*  Description:
*
*      This function performs an interrupt for the specified CPU.
*
*  Parameters:
*
*      None.
*
*  Returns:
*
*      Nothing.
*
******************************************************************************/
static
void
MC6809DoInterrupt (
    void
)
{
    /*
     *  The normal everyday hardware interrupt.
     */
    if (gpMC6809->pendingNMI)
    { 
        gpMC6809->pendingSYNC = 0;
        gpMC6809->pendingCWAI = 0;
        gpMC6809->pendingNMI  = 0;
        gpMC6809->cc.bit.e = 1;
        OP_PSH (0xff, gpMC6809->s, gpMC6809->u);
        gpMC6809->cc.bit.f = 1;  
        gpMC6809->cc.bit.i = 1;
        MEM_READ_WORD_NC (gpMC6809->pc, 0xfffc);
    }
    else
    if (gpMC6809->pendingFIRQ && (gpMC6809->cc.bit.f == 0))
    {
        gpMC6809->pendingSYNC = 0;
        gpMC6809->pendingCWAI = 0;
        gpMC6809->pendingFIRQ = 0;
        gpMC6809->cc.bit.e = 0;
        OP_PSH (0x81, gpMC6809->s, gpMC6809->u);
        gpMC6809->cc.bit.f = 1;  
        gpMC6809->cc.bit.i = 1;
        MEM_READ_WORD_NC (gpMC6809->pc, 0xfff6);
    }
    if (gpMC6809->pendingIRQ && (gpMC6809->cc.bit.i == 0))
    {
        gpMC6809->pendingSYNC = 0;
        gpMC6809->pendingCWAI = 0;
        gpMC6809->pendingIRQ = 0;
        gpMC6809->cc.bit.e = 1;
        OP_PSH (0xff, gpMC6809->s, gpMC6809->u);
        /*
         *  There seems to be some contention on whether or not IRQ's
         *  preclude FIRQ's.  I'm going with no for now.
        gpMC6809->cc.bit.f = 1;  
         */
        gpMC6809->cc.bit.i = 1;
        MEM_READ_WORD_NC (gpMC6809->pc, 0xfff8);
    }
}
            

/******************************************************************************
*
*  Function: MC6809Interrupt
*
*  Description:
*
*      This function will create a pending interrupt of the specified type
*      for the specified cpu.
*
*  Parameters:
*
*      pMC6809 (input/output)
*          The CPU to interrupt.
*
*      nType (input)
*          The type of interrupt.
*
*  Returns:
*
*      Nothing.
*
******************************************************************************/
void
MC6809Interrupt (
    MC6809 *pMC6809,
    int32   nType
)
{
    /*
     *  What type.
     */
    switch (nType)
    {
        case CPU::INT_IRQ:    pMC6809->pendingIRQ  = 1; break;
        case CPU::INT_FIRQ:   pMC6809->pendingFIRQ = 1; break;
        case CPU::INT_NMI:    pMC6809->pendingNMI  = 1; break;
        default:                                        break;
    }
}



/******************************************************************************
*  The following macros are used in the MC6809Execute function.
******************************************************************************/
#define CASE_CONST_CYCLES(CVAL, AMODE, FUNC, CYCLES)                          \
    case CVAL:                                                                \
    {                                                                         \
        eAM = AMODE;                                                          \
        gpMC6809->ICount -= CYCLES;                                             \
        FUNC;                                                                 \
        break;                                                                \
    }

#define CASE_VAR_CYCLES(CVAL, AMODE, FUNC, CYCLES)                            \
    case CVAL:                                                                \
    {                                                                         \
        eAM = AMODE;                                                          \
        gpMC6809->ICount -= ( CYCLES + FUNC );                                  \
        break;                                                                \
    }
/******************************************************************************
*
*  Function: MC6809Execute
*
*  Description:
*
*      This function executes the specified amount of 6809 cycles.
*
*  Parameters:
*
*      pMC6809 (input)
*          The context of the CPU to execute.
*
*      nCycles (input)
*          The number of cycles to perform in this execution slice.
*
*  Returns:
*
*      The number of cycles that were not run (can be negative if we executed
*      more cycles than specified).
*
******************************************************************************/
int32
MC6809Execute (
    MC6809 *pMC6809,
    int32   nCycles
)
{
    /*
     *  Point the global CPU pointer to the object passed in.
     */
    gpMC6809 = pMC6809;

    /* 
     *  Check for an interrupt.
     */
    MC6809DoInterrupt ();

    /*
     *  Start the countdown with the number of cycles to execute unless
     *  we're waiting for a CWAI or SYNC instruction.
     */
    if( gpMC6809->pendingCWAI || gpMC6809->pendingSYNC )
    {
        gpMC6809->ICount = 0;
    }
    else
    {
        gpMC6809->ICount = nCycles;
    }

    /*
     *  Execute until the cycles have expired.
     */
    while (gpMC6809->ICount > 0)
    {
#ifdef DEBUGGER
        Debugger::s_instance( ).check( );
#endif

/*
printf ("0x%08x\n", gpMC6809->pc);
*/

#ifdef PC_PIPE
        static int32 nCount = 0;
        int32 nPC;
        char  cCR;
        nCount += 1;
        scanf( "%lx%c", &nPC, &cCR );
        if( nPC == gpMC6809->pc )
        {
            printf( "0x%08x\n", gpMC6809->pc );
        }
        else
        {
            printf( "0x%08x FAIL -> 0x%08x (%d)\n", gpMC6809->pc, nPC, nCount );
            fflush( stdout );
            exit( 0 );
        }
#endif
    
        /*
         *  Get the opcode of the next instruction and up the PC.
         */
        MEM_READ_OP (op1);
    
        /*
         *  Perform the instruction.
         */
        switch (op1)
        {
            CASE_CONST_CYCLES (0x00, AM_DIRECT,    mc6809NEG(),     6);
            CASE_CONST_CYCLES (0x03, AM_DIRECT,    mc6809COM(),     6);
            CASE_CONST_CYCLES (0x04, AM_DIRECT,    mc6809LSR(),     6);
            CASE_CONST_CYCLES (0x06, AM_DIRECT,    mc6809ROR(),     6);
            CASE_CONST_CYCLES (0x07, AM_DIRECT,    mc6809ASR(),     6);
            CASE_CONST_CYCLES (0x08, AM_DIRECT,    mc6809LSL(),     6);
            CASE_CONST_CYCLES (0x09, AM_DIRECT,    mc6809ROL(),     6);
            CASE_CONST_CYCLES (0x0a, AM_DIRECT,    mc6809DEC(),     6);
            CASE_CONST_CYCLES (0x0c, AM_DIRECT,    mc6809INC(),     6);
            CASE_CONST_CYCLES (0x0d, AM_DIRECT,    mc6809TST(),     6);
            CASE_CONST_CYCLES (0x0e, AM_DIRECT,    mc6809JMP(),     3);
            CASE_CONST_CYCLES (0x0f, AM_DIRECT,    mc6809CLR(),     3);
            
            /*
             *  Page 0x10 opcodes.
             */
            case 0x10:
            {
                /*
                 *  Get the page 0x10 opcode.
                 */
                MEM_READ_OP (op2);

                switch (op2)
                {
                    CASE_CONST_CYCLES (0x21, AM_NA,        mc6809LBRN(),    5);
                    CASE_VAR_CYCLES   (0x22, AM_NA,        mc6809LBHI(),    5);
                    CASE_VAR_CYCLES   (0x23, AM_NA,        mc6809LBLS(),    5);
                    CASE_VAR_CYCLES   (0x24, AM_NA,        mc6809LBCC(),    5);
                    CASE_VAR_CYCLES   (0x25, AM_NA,        mc6809LBCS(),    5);
                    CASE_VAR_CYCLES   (0x26, AM_NA,        mc6809LBNE(),    5);
                    CASE_VAR_CYCLES   (0x27, AM_NA,        mc6809LBEQ(),    5);
                    CASE_VAR_CYCLES   (0x28, AM_NA,        mc6809LBVC(),    5);
                    CASE_VAR_CYCLES   (0x29, AM_NA,        mc6809LBVS(),    5);
                    CASE_VAR_CYCLES   (0x2a, AM_NA,        mc6809LBPL(),    5);
                    CASE_VAR_CYCLES   (0x2b, AM_NA,        mc6809LBMI(),    5);
                    CASE_VAR_CYCLES   (0x2c, AM_NA,        mc6809LBGE(),    5);
                    CASE_VAR_CYCLES   (0x2d, AM_NA,        mc6809LBLT(),    5);
                    CASE_VAR_CYCLES   (0x2e, AM_NA,        mc6809LBGT(),    5);
                    CASE_VAR_CYCLES   (0x2f, AM_NA,        mc6809LBLE(),    5);

                    CASE_CONST_CYCLES (0x3f, AM_INHERENT,  mc6809SWI2(),   20);

                    CASE_CONST_CYCLES (0x83, AM_IMMEDIATE, mc6809CMPD(),    5);
                    CASE_CONST_CYCLES (0x8c, AM_IMMEDIATE, mc6809CMPY(),    5);
                    CASE_CONST_CYCLES (0x8e, AM_IMMEDIATE, mc6809LDY(),     4);

                    CASE_CONST_CYCLES (0x93, AM_DIRECT,    mc6809CMPD(),    7);
                    CASE_CONST_CYCLES (0x9c, AM_DIRECT,    mc6809CMPY(),    7);
                    CASE_CONST_CYCLES (0x9e, AM_DIRECT,    mc6809LDY(),     6);
                    CASE_CONST_CYCLES (0x9f, AM_DIRECT,    mc6809STY(),     6);

                    CASE_VAR_CYCLES   (0xa3, AM_INDEXED,   mc6809CMPD_I(),  7);
                    CASE_VAR_CYCLES   (0xac, AM_INDEXED,   mc6809CMPY_I(),  7);
                    CASE_VAR_CYCLES   (0xae, AM_INDEXED,   mc6809LDY_I(),   6);
                    CASE_VAR_CYCLES   (0xaf, AM_INDEXED,   mc6809STY_I(),   6);

                    CASE_CONST_CYCLES (0xb3, AM_EXTENDED,  mc6809CMPD(),    8);
                    CASE_CONST_CYCLES (0xbc, AM_EXTENDED,  mc6809CMPY(),    8);
                    CASE_CONST_CYCLES (0xbe, AM_EXTENDED,  mc6809LDY(),     7);
                    CASE_CONST_CYCLES (0xbf, AM_EXTENDED,  mc6809STY(),     7);

                    CASE_CONST_CYCLES (0xce, AM_IMMEDIATE, mc6809LDS(),     4);

                    CASE_CONST_CYCLES (0xde, AM_DIRECT,    mc6809LDS(),     6);
                    CASE_CONST_CYCLES (0xdf, AM_DIRECT,    mc6809STS(),     6);

                    CASE_VAR_CYCLES   (0xee, AM_INDEXED,   mc6809LDS_I(),   6);
                    CASE_VAR_CYCLES   (0xef, AM_INDEXED,   mc6809STS_I(),   6);

                    CASE_CONST_CYCLES (0xfe, AM_EXTENDED,  mc6809LDS(),     7);
                    CASE_CONST_CYCLES (0xff, AM_EXTENDED,  mc6809STS(),     7);

                    /*
                     *  Unknown page 0x10 opcode.
                     */
                    default:
                    {
                        mc6809eerr ();
                        break;
                    }
                }
                break;
            }
         
            /*
             *  Page 0x11 opcodes.
             */
            case 0x11:
            {
                /*
                 *  Get the page 0x11 opcode.
                 */
                MEM_READ_OP (op2);

                switch (op2)
                {
                    CASE_CONST_CYCLES (0x3f, AM_INHERENT,  mc6809SWI3(),   20);

                    CASE_CONST_CYCLES (0x83, AM_IMMEDIATE, mc6809CMPU(),    5);
                    CASE_CONST_CYCLES (0x8c, AM_IMMEDIATE, mc6809CMPS(),    5);

                    CASE_CONST_CYCLES (0x93, AM_DIRECT,    mc6809CMPU(),    7);
                    CASE_CONST_CYCLES (0x9c, AM_DIRECT,    mc6809CMPS(),    7);

                    CASE_VAR_CYCLES   (0xa3, AM_INDEXED,   mc6809CMPU_I(),  7);
                    CASE_VAR_CYCLES   (0xac, AM_INDEXED,   mc6809CMPS_I(),  7);

                    CASE_CONST_CYCLES (0xb3, AM_EXTENDED,  mc6809CMPU(),    8);
                    CASE_CONST_CYCLES (0xbc, AM_EXTENDED,  mc6809CMPS(),    8);

                    /*
                     *  Unknown page 0x11 opcode.
                     */
                    default:
                    {
                        mc6809eerr ();
                        break;
                    }
                }
                break;
            }

            CASE_CONST_CYCLES (0x12, AM_INHERENT,  mc6809NOP(),     2);
            CASE_CONST_CYCLES (0x13, AM_NA,        mc6809SYNC(),    0);
            CASE_CONST_CYCLES (0x16, AM_NA,        mc6809LBRA(),    5);
            CASE_CONST_CYCLES (0x17, AM_NA,        mc6809LBSR(),    9);
            CASE_CONST_CYCLES (0x19, AM_INHERENT,  mc6809DAA(),     2);
            CASE_CONST_CYCLES (0x1a, AM_IMMEDIATE, mc6809ORCC(),    3);
            CASE_CONST_CYCLES (0x1c, AM_IMMEDIATE, mc6809ANDCC(),   3);
            CASE_CONST_CYCLES (0x1d, AM_INHERENT,  mc6809SEX(),     2);
            CASE_CONST_CYCLES (0x1e, AM_INHERENT,  mc6809EXG(),     8);
            CASE_CONST_CYCLES (0x1f, AM_INHERENT,  mc6809TFR(),     6);

            
            CASE_CONST_CYCLES (0x20, AM_NA,        mc6809BRA(),     3);
            CASE_CONST_CYCLES (0x21, AM_NA,        mc6809BRN(),     3);
            CASE_CONST_CYCLES (0x22, AM_NA,        mc6809BHI(),     3);
            CASE_CONST_CYCLES (0x23, AM_NA,        mc6809BLS(),     3);
            CASE_CONST_CYCLES (0x24, AM_NA,        mc6809BCC(),     3);
            CASE_CONST_CYCLES (0x25, AM_NA,        mc6809BCS(),     3);
            CASE_CONST_CYCLES (0x26, AM_NA,        mc6809BNE(),     3);
            CASE_CONST_CYCLES (0x27, AM_NA,        mc6809BEQ(),     3);
            CASE_CONST_CYCLES (0x28, AM_NA,        mc6809BVC(),     3);
            CASE_CONST_CYCLES (0x29, AM_NA,        mc6809BVS(),     3);
            CASE_CONST_CYCLES (0x2a, AM_NA,        mc6809BPL(),     3);
            CASE_CONST_CYCLES (0x2b, AM_NA,        mc6809BMI(),     3);
            CASE_CONST_CYCLES (0x2c, AM_NA,        mc6809BGE(),     3);
            CASE_CONST_CYCLES (0x2d, AM_NA,        mc6809BLT(),     3);
            CASE_CONST_CYCLES (0x2e, AM_NA,        mc6809BGT(),     3);
            CASE_CONST_CYCLES (0x2f, AM_NA,        mc6809BLE(),     3);

            CASE_VAR_CYCLES   (0x30, AM_NA,        mc6809LEAX(),    4);
            CASE_VAR_CYCLES   (0x31, AM_NA,        mc6809LEAY(),    4);
            CASE_VAR_CYCLES   (0x32, AM_NA,        mc6809LEAS(),    4);
            CASE_VAR_CYCLES   (0x33, AM_NA,        mc6809LEAU(),    4);
            CASE_VAR_CYCLES   (0x34, AM_NA,        mc6809PSHS(),    5);
            CASE_VAR_CYCLES   (0x35, AM_NA,        mc6809PULS(),    5);
            CASE_VAR_CYCLES   (0x36, AM_INHERENT,  mc6809PSHU(),    5);
            CASE_VAR_CYCLES   (0x37, AM_INHERENT,  mc6809PULU(),    5);
            CASE_CONST_CYCLES (0x39, AM_INHERENT,  mc6809RTS(),     5);
            CASE_CONST_CYCLES (0x3a, AM_INHERENT,  mc6809ABX(),     3);
            CASE_VAR_CYCLES   (0x3b, AM_INHERENT,  mc6809RTI(),     0);
            CASE_CONST_CYCLES (0x3c, AM_NA,        mc6809CWAI(),    0);
            CASE_CONST_CYCLES (0x3d, AM_INHERENT,  mc6809MUL(),    11);
            CASE_CONST_CYCLES (0x3f, AM_INHERENT,  mc6809SWI(),    19);

            CASE_CONST_CYCLES (0x40, AM_INHERENT,  mc6809NEGA(),    2);
            CASE_CONST_CYCLES (0x43, AM_INHERENT,  mc6809COMA(),    2);
            CASE_CONST_CYCLES (0x44, AM_INHERENT,  mc6809LSRA(),    2);
            CASE_CONST_CYCLES (0x46, AM_INHERENT,  mc6809RORA(),    2);
            CASE_CONST_CYCLES (0x47, AM_INHERENT,  mc6809ASRA(),    2);
            CASE_CONST_CYCLES (0x48, AM_INHERENT,  mc6809LSLA(),    2);
            CASE_CONST_CYCLES (0x49, AM_INHERENT,  mc6809ROLA(),    2);
            CASE_CONST_CYCLES (0x4a, AM_INHERENT,  mc6809DECA(),    2);
            CASE_CONST_CYCLES (0x4c, AM_INHERENT,  mc6809INCA(),    2);
            CASE_CONST_CYCLES (0x4d, AM_INHERENT,  mc6809TSTA(),    2);
            CASE_CONST_CYCLES (0x4f, AM_INHERENT,  mc6809CLRA(),    2);

            CASE_CONST_CYCLES (0x50, AM_INHERENT,  mc6809NEGB(),    2);
            CASE_CONST_CYCLES (0x53, AM_INHERENT,  mc6809COMB(),    2);
            CASE_CONST_CYCLES (0x54, AM_INHERENT,  mc6809LSRB(),    2);
            CASE_CONST_CYCLES (0x56, AM_INHERENT,  mc6809RORB(),    2);
            CASE_CONST_CYCLES (0x57, AM_INHERENT,  mc6809ASRB(),    2);
            CASE_CONST_CYCLES (0x58, AM_INHERENT,  mc6809LSLB(),    2);
            CASE_CONST_CYCLES (0x59, AM_INHERENT,  mc6809ROLB(),    2);
            CASE_CONST_CYCLES (0x5a, AM_INHERENT,  mc6809DECB(),    2);
            CASE_CONST_CYCLES (0x5c, AM_INHERENT,  mc6809INCB(),    2);
            CASE_CONST_CYCLES (0x5d, AM_INHERENT,  mc6809TSTB(),    2);
            CASE_CONST_CYCLES (0x5f, AM_INHERENT,  mc6809CLRB(),    2);

            CASE_VAR_CYCLES   (0x60, AM_INDEXED,   mc6809NEG_I(),   6);
            CASE_VAR_CYCLES   (0x63, AM_INDEXED,   mc6809COM_I(),   6);
            CASE_VAR_CYCLES   (0x64, AM_INDEXED,   mc6809LSR_I(),   6);
            CASE_VAR_CYCLES   (0x66, AM_INDEXED,   mc6809ROR_I(),   6);
            CASE_VAR_CYCLES   (0x67, AM_INDEXED,   mc6809ASR_I(),   6);
            CASE_VAR_CYCLES   (0x68, AM_INDEXED,   mc6809LSL_I(),   6);
            CASE_VAR_CYCLES   (0x69, AM_INDEXED,   mc6809ROL_I(),   6);
            CASE_VAR_CYCLES   (0x6a, AM_INDEXED,   mc6809DEC_I(),   6);
            CASE_VAR_CYCLES   (0x6c, AM_INDEXED,   mc6809INC_I(),   6);
            CASE_VAR_CYCLES   (0x6d, AM_INDEXED,   mc6809TST_I(),   6);
            CASE_VAR_CYCLES   (0x6e, AM_INDEXED,   mc6809JMP_I(),   3);
            CASE_VAR_CYCLES   (0x6f, AM_INDEXED,   mc6809CLR_I(),   6);

            CASE_CONST_CYCLES (0x70, AM_EXTENDED,  mc6809NEG(),     7);
            CASE_CONST_CYCLES (0x73, AM_EXTENDED,  mc6809COM(),     7);
            CASE_CONST_CYCLES (0x74, AM_EXTENDED,  mc6809LSR(),     7);
            CASE_CONST_CYCLES (0x76, AM_EXTENDED,  mc6809ROR(),     7);
            CASE_CONST_CYCLES (0x77, AM_EXTENDED,  mc6809ASR(),     7);
            CASE_CONST_CYCLES (0x78, AM_EXTENDED,  mc6809LSL(),     7);
            CASE_CONST_CYCLES (0x79, AM_EXTENDED,  mc6809ROL(),     3);
            CASE_CONST_CYCLES (0x7a, AM_EXTENDED,  mc6809DEC(),     7);
            CASE_CONST_CYCLES (0x7c, AM_EXTENDED,  mc6809INC(),     7);
            CASE_CONST_CYCLES (0x7d, AM_EXTENDED,  mc6809TST(),     7);
            CASE_CONST_CYCLES (0x7e, AM_EXTENDED,  mc6809JMP(),     4);
            CASE_CONST_CYCLES (0x7f, AM_EXTENDED,  mc6809CLR(),     7);
            
            CASE_CONST_CYCLES (0x80, AM_IMMEDIATE, mc6809SUBA(),    2);
            CASE_CONST_CYCLES (0x81, AM_IMMEDIATE, mc6809CMPA(),    2);
            CASE_CONST_CYCLES (0x82, AM_IMMEDIATE, mc6809SBCA(),    2);
            CASE_CONST_CYCLES (0x83, AM_IMMEDIATE, mc6809SUBD(),    4);
            CASE_CONST_CYCLES (0x84, AM_IMMEDIATE, mc6809ANDA(),    2);
            CASE_CONST_CYCLES (0x85, AM_IMMEDIATE, mc6809BITA(),    2);
            CASE_CONST_CYCLES (0x86, AM_IMMEDIATE, mc6809LDA(),     2);
            CASE_CONST_CYCLES (0x88, AM_IMMEDIATE, mc6809EORA(),    2);
            CASE_CONST_CYCLES (0x89, AM_IMMEDIATE, mc6809ADCA(),    2);
            CASE_CONST_CYCLES (0x8a, AM_IMMEDIATE, mc6809ORA(),     2);
            CASE_CONST_CYCLES (0x8b, AM_IMMEDIATE, mc6809ADDA(),    2);
            CASE_CONST_CYCLES (0x8c, AM_IMMEDIATE, mc6809CMPX(),    4);
            CASE_CONST_CYCLES (0x8d, AM_NA,        mc6809BSR(),     7);
            CASE_CONST_CYCLES (0x8e, AM_IMMEDIATE, mc6809LDX(),     3);

            CASE_CONST_CYCLES (0x90, AM_DIRECT,    mc6809SUBA(),    4);
            CASE_CONST_CYCLES (0x91, AM_DIRECT,    mc6809CMPA(),    4);
            CASE_CONST_CYCLES (0x92, AM_DIRECT,    mc6809SBCA(),    4);
            CASE_CONST_CYCLES (0x93, AM_DIRECT,    mc6809SUBD(),    6);
            CASE_CONST_CYCLES (0x94, AM_DIRECT,    mc6809ANDA(),    4);
            CASE_CONST_CYCLES (0x95, AM_DIRECT,    mc6809BITA(),    4);
            CASE_CONST_CYCLES (0x96, AM_DIRECT,    mc6809LDA(),     4);
            CASE_CONST_CYCLES (0x97, AM_DIRECT,    mc6809STA(),     4);
            CASE_CONST_CYCLES (0x98, AM_DIRECT,    mc6809EORA(),    4);
            CASE_CONST_CYCLES (0x99, AM_DIRECT,    mc6809ADCA(),    4);
            CASE_CONST_CYCLES (0x9a, AM_DIRECT,    mc6809ORA(),     4);
            CASE_CONST_CYCLES (0x9b, AM_DIRECT,    mc6809ADDA(),    4);
            CASE_CONST_CYCLES (0x9c, AM_DIRECT,    mc6809CMPX(),    6);
            CASE_CONST_CYCLES (0x9d, AM_DIRECT,    mc6809JSR(),     7);
            CASE_CONST_CYCLES (0x9e, AM_DIRECT,    mc6809LDX(),     5);
            CASE_CONST_CYCLES (0x9f, AM_DIRECT,    mc6809STX(),     5);

            CASE_VAR_CYCLES   (0xa0, AM_INDEXED,   mc6809SUBA_I(),  4);
            CASE_VAR_CYCLES   (0xa1, AM_INDEXED,   mc6809CMPA_I(),  4);
            CASE_VAR_CYCLES   (0xa2, AM_INDEXED,   mc6809SBCA_I(),  4);
            CASE_VAR_CYCLES   (0xa3, AM_INDEXED,   mc6809SUBD_I(),  6);
            CASE_VAR_CYCLES   (0xa4, AM_INDEXED,   mc6809ANDA_I(),  4);
            CASE_VAR_CYCLES   (0xa5, AM_INDEXED,   mc6809BITA_I(),  4);
            CASE_VAR_CYCLES   (0xa6, AM_INDEXED,   mc6809LDA_I(),   4);
            CASE_VAR_CYCLES   (0xa7, AM_INDEXED,   mc6809STA_I(),   4);
            CASE_VAR_CYCLES   (0xa8, AM_INDEXED,   mc6809EORA_I(),  4);
            CASE_VAR_CYCLES   (0xa9, AM_INDEXED,   mc6809ADCA_I(),  4);
            CASE_VAR_CYCLES   (0xaa, AM_INDEXED,   mc6809ORA_I(),   4);
            CASE_VAR_CYCLES   (0xab, AM_INDEXED,   mc6809ADDA_I(),  4);
            CASE_VAR_CYCLES   (0xac, AM_INDEXED,   mc6809CMPX_I(),  6);
            CASE_VAR_CYCLES   (0xad, AM_INDEXED,   mc6809JSR_I(),   7);
            CASE_VAR_CYCLES   (0xae, AM_INDEXED,   mc6809LDX_I(),   5);
            CASE_VAR_CYCLES   (0xaf, AM_INDEXED,   mc6809STX_I(),   5);

            CASE_CONST_CYCLES (0xb0, AM_EXTENDED,  mc6809SUBA(),    5);
            CASE_CONST_CYCLES (0xb1, AM_EXTENDED,  mc6809CMPA(),    5);
            CASE_CONST_CYCLES (0xb2, AM_EXTENDED,  mc6809SBCA(),    5);
            CASE_CONST_CYCLES (0xb3, AM_EXTENDED,  mc6809SUBD(),    7);
            CASE_CONST_CYCLES (0xb4, AM_EXTENDED,  mc6809ANDA(),    5);
            CASE_CONST_CYCLES (0xb5, AM_EXTENDED,  mc6809BITA(),    5);
            CASE_CONST_CYCLES (0xb6, AM_EXTENDED,  mc6809LDA(),     5);
            CASE_CONST_CYCLES (0xb7, AM_EXTENDED,  mc6809STA(),     5);
            CASE_CONST_CYCLES (0xb8, AM_EXTENDED,  mc6809EORA(),    5);
            CASE_CONST_CYCLES (0xb9, AM_EXTENDED,  mc6809ADCA(),    5);
            CASE_CONST_CYCLES (0xba, AM_EXTENDED,  mc6809ORA(),     5);
            CASE_CONST_CYCLES (0xbb, AM_EXTENDED,  mc6809ADDA(),    5);
            CASE_CONST_CYCLES (0xbc, AM_EXTENDED,  mc6809CMPX(),    7);
            CASE_CONST_CYCLES (0xbd, AM_EXTENDED,  mc6809JSR(),     8);
            CASE_CONST_CYCLES (0xbe, AM_EXTENDED,  mc6809LDX(),     6);
            CASE_CONST_CYCLES (0xbf, AM_EXTENDED,  mc6809STX(),     6);

            CASE_CONST_CYCLES (0xc0, AM_IMMEDIATE, mc6809SUBB(),    2);
            CASE_CONST_CYCLES (0xc1, AM_IMMEDIATE, mc6809CMPB(),    2);
            CASE_CONST_CYCLES (0xc2, AM_IMMEDIATE, mc6809SBCB(),    2);
            CASE_CONST_CYCLES (0xc3, AM_IMMEDIATE, mc6809ADDD(),    4);
            CASE_CONST_CYCLES (0xc4, AM_IMMEDIATE, mc6809ANDB(),    2);
            CASE_CONST_CYCLES (0xc5, AM_IMMEDIATE, mc6809BITB(),    2);
            CASE_CONST_CYCLES (0xc6, AM_IMMEDIATE, mc6809LDB(),     2);
            CASE_CONST_CYCLES (0xc8, AM_IMMEDIATE, mc6809EORB(),    2);
            CASE_CONST_CYCLES (0xc9, AM_IMMEDIATE, mc6809ADCB(),    2);
            CASE_CONST_CYCLES (0xca, AM_IMMEDIATE, mc6809ORB(),     2);
            CASE_CONST_CYCLES (0xcb, AM_IMMEDIATE, mc6809ADDB(),    2);
            CASE_CONST_CYCLES (0xcc, AM_IMMEDIATE, mc6809LDD(),     3);
            CASE_CONST_CYCLES (0xce, AM_IMMEDIATE, mc6809LDU(),     3);

            CASE_CONST_CYCLES (0xd0, AM_DIRECT,    mc6809SUBB(),    4);
            CASE_CONST_CYCLES (0xd1, AM_DIRECT,    mc6809CMPB(),    4);
            CASE_CONST_CYCLES (0xd2, AM_DIRECT,    mc6809SBCB(),    4);
            CASE_CONST_CYCLES (0xd3, AM_DIRECT,    mc6809ADDD(),    6);
            CASE_CONST_CYCLES (0xd4, AM_DIRECT,    mc6809ANDB(),    4);
            CASE_CONST_CYCLES (0xd5, AM_DIRECT,    mc6809BITB(),    4);
            CASE_CONST_CYCLES (0xd6, AM_DIRECT,    mc6809LDB(),     4);
            CASE_CONST_CYCLES (0xd7, AM_DIRECT,    mc6809STB(),     4);
            CASE_CONST_CYCLES (0xd8, AM_DIRECT,    mc6809EORB(),    4);
            CASE_CONST_CYCLES (0xd9, AM_DIRECT,    mc6809ADCB(),    4);
            CASE_CONST_CYCLES (0xda, AM_DIRECT,    mc6809ORB(),     4);
            CASE_CONST_CYCLES (0xdb, AM_DIRECT,    mc6809ADDB(),    4);
            CASE_CONST_CYCLES (0xdc, AM_DIRECT,    mc6809LDD(),     5);
            CASE_CONST_CYCLES (0xdd, AM_DIRECT,    mc6809STD(),     5);
            CASE_CONST_CYCLES (0xde, AM_DIRECT,    mc6809LDU(),     5);
            CASE_CONST_CYCLES (0xdf, AM_DIRECT,    mc6809STU(),     5);

            CASE_VAR_CYCLES   (0xe0, AM_INDEXED,   mc6809SUBB_I(),  4);
            CASE_VAR_CYCLES   (0xe1, AM_INDEXED,   mc6809CMPB_I(),  4);
            CASE_VAR_CYCLES   (0xe2, AM_INDEXED,   mc6809SBCB_I(),  4);
            CASE_VAR_CYCLES   (0xe3, AM_INDEXED,   mc6809ADDD_I(),  6);
            CASE_VAR_CYCLES   (0xe4, AM_INDEXED,   mc6809ANDB_I(),  4);
            CASE_VAR_CYCLES   (0xe5, AM_INDEXED,   mc6809BITB_I(),  4);
            CASE_VAR_CYCLES   (0xe6, AM_INDEXED,   mc6809LDB_I(),   4);
            CASE_VAR_CYCLES   (0xe7, AM_INDEXED,   mc6809STB_I(),   4);
            CASE_VAR_CYCLES   (0xe8, AM_INDEXED,   mc6809EORB_I(),  4);
            CASE_VAR_CYCLES   (0xe9, AM_INDEXED,   mc6809ADCB_I(),  4);
            CASE_VAR_CYCLES   (0xea, AM_INDEXED,   mc6809ORB_I(),   4);
            CASE_VAR_CYCLES   (0xeb, AM_INDEXED,   mc6809ADDB_I(),  4);
            CASE_VAR_CYCLES   (0xec, AM_INDEXED,   mc6809LDD_I(),   5);
            CASE_VAR_CYCLES   (0xed, AM_INDEXED,   mc6809STD_I(),   5);
            CASE_VAR_CYCLES   (0xee, AM_INDEXED,   mc6809LDU_I(),   5);
            CASE_VAR_CYCLES   (0xef, AM_INDEXED,   mc6809STU_I(),   5);

            CASE_CONST_CYCLES (0xf0, AM_EXTENDED,  mc6809SUBB(),    5);
            CASE_CONST_CYCLES (0xf1, AM_EXTENDED,  mc6809CMPB(),    5);
            CASE_CONST_CYCLES (0xf2, AM_EXTENDED,  mc6809SBCB(),    5);
            CASE_CONST_CYCLES (0xf3, AM_EXTENDED,  mc6809ADDD(),    7);
            CASE_CONST_CYCLES (0xf4, AM_EXTENDED,  mc6809ANDB(),    5);
            CASE_CONST_CYCLES (0xf5, AM_EXTENDED,  mc6809BITB(),    5);
            CASE_CONST_CYCLES (0xf6, AM_EXTENDED,  mc6809LDB(),     5);
            CASE_CONST_CYCLES (0xf7, AM_EXTENDED,  mc6809STB(),     5);
            CASE_CONST_CYCLES (0xf8, AM_EXTENDED,  mc6809EORB(),    5);
            CASE_CONST_CYCLES (0xf9, AM_EXTENDED,  mc6809ADCB(),    5);
            CASE_CONST_CYCLES (0xfa, AM_EXTENDED,  mc6809ORB(),     5);
            CASE_CONST_CYCLES (0xfb, AM_EXTENDED,  mc6809ADDB(),    5);
            CASE_CONST_CYCLES (0xfc, AM_EXTENDED,  mc6809LDD(),     6);
            CASE_CONST_CYCLES (0xfd, AM_EXTENDED,  mc6809STD(),     6);
            CASE_CONST_CYCLES (0xfe, AM_EXTENDED,  mc6809LDU(),     6);
            CASE_CONST_CYCLES (0xff, AM_EXTENDED,  mc6809STU(),     6);

            /*
             *  Unknown opcode.
             */
            default:
            {
                mc6809err ();
                break;
            }
        }
    }
          
    /*
     *  Return the remaining instruction cycles.
     */
#ifdef PC_PIPE
    return (0);
#else
    return (gpMC6809->ICount);
#endif
}



/******************************************************************************
*
*  Function: MC6809ReadMemOperand
*
*  Description:
*
*      This function is called to read a byte from memory based on the current
*      addressing mode.
*
*  Parameters:
*
*      None.
*
*  Returns:
*      The byte read.
*
******************************************************************************/
static
Byte
MC6809ReadMemOperand (
    void
)
{
    /*
     *  The value read.
     */
    Byte bValue = 0x00;

    /*
     *  A temporary byte.
     */
    Byte bTemp;

    /*
     *  An address.
     */
    Word wAddress;


    /*
     *  Get the byte based on the current addressing mode.
     */
    switch (eAM)
    {
        case AM_IMMEDIATE:
        case AM_RELATIVE:
        {
            /*
             *  Read the next byte and increment the PC.
             */
            MEM_READ_BYTE (bValue);
            break;
        }

        case AM_EXTENDED:
        {
            /*
             *  Read the address from the next two bytes and get the byte
             *  at the resulting location.
             */
            MEM_READ_WORD (wAddress);
            MEM_READ_BYTE_NC (bValue, wAddress);
            break;
        }

        case AM_DIRECT:
        {
            /*
             *  Use the direct page register as the upper byte and the
             *  next byte as the lower byte to form the address to read
             *  from.
             */
            MEM_READ_BYTE (bTemp);
            wAddress = (Word)(gpMC6809->dpHi | bTemp);
            MEM_READ_BYTE_NC (bValue, wAddress);
            break;
        }

        default:
        {
            printf ("Invalid addressing mode.\n");
            fflush (stdout);
            exit (0);
            break;
        }
    }

    /*
     *  Return the value.
     */
    return (bValue);
}




/******************************************************************************
*
*  Function: MC6809ReadMemIndexedOperand
*
*  Description:
*
*      This function is called to read an indexed byte.
*
*  Parameters:
*
*      pnCycles
*          A buffer to store the cycles that the reading eats up.
*
*  Returns:
*      The indexed byte.
*
******************************************************************************/
static
Byte
MC6809ReadMemIndexedOperand (
    int32 *pnCycles
)
{
    /*
     *  The value of the operand.
     */
    Byte bValue = 0x00;

    /*
     *  An address.
     */
    Word wAddress = 0x0000;

    /*
     *  The post byte value.
     */
    Byte bPost;

    /*
     *  A pointer to the register involved in the indexed address.
     */
    Word *pwRefReg;


    /*
     *  Get the post byte.
     */
    MEM_READ_BYTE (bPost);

    /*
     *  Get the register that is referenced by the post byte.
     */
    REFERENCED_REGISTER (pwRefReg, bPost);

    /*
     *  If a predecrement is required then do it.
     */
    if ((bPost & 0x8e) == 0x82)
    {
        PREDECREMENT (bPost);
    }
 
    /*
     *  Find the effective address.
     */
    EA_GET_POST (wAddress, bPost, *pwRefReg);
    MEM_READ_BYTE_NC (bValue, wAddress);

    /*
     *  If a postincrement is required then do it.
     */
    if ((bPost & 0x8e) == 0x80)
    {
        POSTINCREMENT (bPost);
    }

    /*
     *  Record the cycles used.
     */
    *pnCycles = anIndexedCycles[bPost];

    /*
     *  Return the value at the address.
     */
    return (bValue);
}




/******************************************************************************
*
*  Function: MC6809ReadMemWordOperand
*
*  Description:
*
*      This function is called to read a word from memory based on the current
*      addressing mode.
*
*  Parameters:
*
*      None.
*
*  Returns:
*      The word read.
*
******************************************************************************/
static
Word
MC6809ReadMemWordOperand (
    void
)
{
    /*
     *  The value read.
     */
    Word wValue = 0x0000;

    /*
     *  A temporary byte.
     */
    Byte bTemp;

    /*
     *  An address.
     */
    Word wAddress;


    /*
     *  Get the word based on the current addressing mode.
     */
    switch (eAM)
    {
        case AM_IMMEDIATE:
        case AM_RELATIVE:
        {
            /*
             *  Read the next word and increment the PC.
             */
            MEM_READ_WORD (wValue);
            break;
        }

        case AM_EXTENDED:
        {
            /*
             *  Read the address from the next two bytes and get the word
             *  at the resulting location.
             */
            MEM_READ_WORD (wAddress);
            MEM_READ_WORD_NC (wValue, wAddress);
            break;
        }

        case AM_DIRECT:
        {
            /*
             *  Use the direct page register as the upper byte and the
             *  next byte as the lower byte to form the address to read
             *  from.
             */
            MEM_READ_BYTE (bTemp);
            wAddress = (Word)(gpMC6809->dpHi | bTemp);
            MEM_READ_WORD_NC (wValue, wAddress);
            break;
        }

        default:
        {
            printf ("Invalid addressing mode.\n");
            fflush (stdout);
            exit (0);
            break;
        }
    }

    /*
     *  Return the value.
     */
    return (wValue);
}




/******************************************************************************
*
*  Function: MC6809ReadMemIndexedWordOperand
*
*  Description:
*
*      This function is called to read an indexed word.
*
*  Parameters:
*
*      pnCycles
*          A buffer to store the cycles that the reading eats up.
*
*  Returns:
*      The indexed word.
*
******************************************************************************/
static
Word
MC6809ReadMemIndexedWordOperand (
    int32 *pnCycles
)
{
    /*
     *  The value of the operand.
     */
    Word wValue = 0x0000;

    /*
     *  An address.
     */
    Word wAddress = 0x0000;

    /*
     *  The post byte value.
     */
    Byte bPost;

    /*
     *  A pointer to the register involved in the indexed address.
     */
    Word *pwRefReg;


    /*
     *  Get the post byte.
     */
    MEM_READ_BYTE (bPost);

    /*
     *  Get the register that is referenced by the post byte.
     */
    REFERENCED_REGISTER (pwRefReg, bPost);

    /*
     *  If a predecrement is required then do it.
     */
    if ((bPost & 0x8e) == 0x82)
    {
        PREDECREMENT (bPost);
    }
 
    /*
     *  Find the effective address.
     */
    EA_GET_POST (wAddress, bPost, *pwRefReg);

    /*
     *  If a postincrement is required then do it.
     */
    if ((bPost & 0x8e) == 0x80)
    {
        POSTINCREMENT (bPost);
    }

    /*
     *  Record the cycles used.
     */
    *pnCycles = anIndexedCycles[bPost];

    /*
     *  Return the value at the address.
     */
    MEM_READ_WORD_NC (wValue, wAddress);
    return (wValue);
}




/******************************************************************************
*
*  Function: MC6809ReadMemEA
*
*  Description:
*
*      This function is called to read the effective address based on the 
*      addressing mode.
*
*  Parameters:
*
*      None.
*
*  Returns:
*      The effective address.
*
******************************************************************************/
static
Word
MC6809ReadMemEA (
    void
)
{
    /*
     *  The effective address.
     */
    Word wAddress = 0x0000;

    /*
     *  A temporary byte.
     */
    Byte bTemp;


    /*
     *  Get the address based on the current addressing mode.
     */
    switch (eAM)
    {
        case AM_EXTENDED:
        {
            /*
             *  Read the address from the next two bytes and get the word
             *  at the resulting location.
             */
            MEM_READ_WORD (wAddress);
            break;
        }

        case AM_DIRECT:
        {
            /*
             *  Use the direct page register as the upper byte and the
             *  next byte as the lower byte to form the address to read
             *  from.
             */
            MEM_READ_BYTE (bTemp);
            wAddress = (Word)(gpMC6809->dpHi | bTemp);
            break;
        }

        default:
        {
            printf ("Invalid addressing mode.\n");
            fflush (stdout);
            exit (0);
            break;
        }
    }

    /*
     *  Return the address.
     */
    return (wAddress);
}




/******************************************************************************
*
*  Function: MC6809ReadMemIA
*
*  Description:
*
*      This function is called to read the indexed address.
*
*  Parameters:
*
*      pnCycles
*          A buffer to store the cycles that the address eats up.
*
*  Returns:
*      The indexed address.
*
******************************************************************************/
static
Word
MC6809ReadMemIA (
    int32 *pnCycles
)
{
    /*
     *  The indexed address.
     */
    Word wAddress = 0x0000;

    /*
     *  The post byte value.
     */
    Byte bPost;

    /*
     *  A pointer to the register involved in the indexed address.
     */
    Word *pwRefReg;


    /*
     *  Get the post byte.
     */
    MEM_READ_BYTE (bPost);

    /*
     *  Get the register that is referenced by the post byte.
     */
    REFERENCED_REGISTER (pwRefReg, bPost);

    /*
     *  If a predecrement is required then do it.
     */
    if ((bPost & 0x8e) == 0x82)
    {
        PREDECREMENT (bPost);
    }
 
    /*
     *  Find the effective address.
     */
    EA_GET_POST (wAddress, bPost, *pwRefReg);

    /*
     *  If a postincrement is required then do it.
     */
    if ((bPost & 0x8e) == 0x80)
    {
        POSTINCREMENT (bPost);
    }

    /*
     *  Record the cycles used.
     */
    *pnCycles = anIndexedCycles[bPost];

    /*
     *  Return the address.
     */
    return (wAddress);
}



static
void
mc6809err (
    void
)
{
}

static
void
mc6809eerr (
    void
)
{
}

static
void
mc6809ABX (
    void
)
{
    gpMC6809->x += gpMC6809->acc.B.l;
}

static
void
mc6809ADCA (
    void
)
{
    OP_ADC (gpMC6809->acc.B.h);
}

static
void
mc6809ADCB (
    void
)
{
    OP_ADC (gpMC6809->acc.B.l);
}

static
int32
mc6809ADCA_I (
    void
)
{
    int32 nCycles;
    OP_ADC_I (gpMC6809->acc.B.h, &nCycles);
    return (nCycles);
}

static
int32
mc6809ADCB_I (
    void
)
{
    int32 nCycles;
    OP_ADC_I (gpMC6809->acc.B.l, &nCycles);
    return (nCycles);
}


static
void
mc6809ADDA (
    void
)
{
    OP_ADD (gpMC6809->acc.B.h);
}

static
void
mc6809ADDB (
    void
)
{
    OP_ADD (gpMC6809->acc.B.l);
}

static
int32
mc6809ADDA_I (
    void
)
{
    int32 nCycles;
    OP_ADD_I (gpMC6809->acc.B.h, &nCycles);
    return (nCycles);
}

static
int32
mc6809ADDB_I (
    void
)
{
    int32 nCycles;
    OP_ADD_I (gpMC6809->acc.B.l, &nCycles);
    return (nCycles);
}

static
void
mc6809ADDD (
    void
)
{
    Word  wO; 
    Word  wTemp;
    DWord dwTemp;

    wO = MC6809ReadMemWordOperand ();

    wTemp = (gpMC6809->acc.W & 0x7fff) + (wO & 0x7fff);
    gpMC6809->cc.bit.v = BTST15 (wTemp);

    dwTemp = (DWord)(gpMC6809->acc.W) + wO;
    gpMC6809->cc.bit.c = BTST16 (dwTemp);
    gpMC6809->acc.W = (Word)(dwTemp & 0xffff); 

    gpMC6809->cc.bit.v ^= gpMC6809->cc.bit.c;
    gpMC6809->cc.bit.n = BTST15 (gpMC6809->acc.W);
    gpMC6809->cc.bit.z = !(gpMC6809->acc.W);
}

static
int32
mc6809ADDD_I (
    void
)
{
    int32     nCycles;
    Word  wO; 
    Word  wTemp;
    DWord dwTemp;

    wO = MC6809ReadMemIndexedWordOperand (&nCycles);

    wTemp = (gpMC6809->acc.W & 0x7fff) + (wO & 0x7fff);
    gpMC6809->cc.bit.v = BTST15 (wTemp);

    dwTemp = (DWord)(gpMC6809->acc.W) + wO;
    gpMC6809->cc.bit.c = BTST16 (dwTemp);
    gpMC6809->acc.W = (Word)(dwTemp & 0xffff); 

    gpMC6809->cc.bit.v ^= gpMC6809->cc.bit.c;
    gpMC6809->cc.bit.n = BTST15 (gpMC6809->acc.W);
    gpMC6809->cc.bit.z = !(gpMC6809->acc.W);

    return (nCycles);
}

static
void
mc6809ANDA (
    void
)
{
    OP_AND (gpMC6809->acc.B.h);
}

static
void
mc6809ANDB (
    void
)
{
    OP_AND (gpMC6809->acc.B.l);
}

static
void
mc6809ANDCC (
    void
)
{
    Byte bO;

    MEM_READ_BYTE (bO);
    gpMC6809->cc.all &= bO;
}

static
int32
mc6809ANDA_I (
    void
)
{
    int32 nCycles;
    OP_AND_I (gpMC6809->acc.B.h, &nCycles);
    return (nCycles);
}

static
int32
mc6809ANDB_I (
    void
)
{
    int32 nCycles;
    OP_AND_I (gpMC6809->acc.B.l, &nCycles);
    return (nCycles);
}

static
void
mc6809ASRA  (
    void
)
{
    OP_ASR (gpMC6809->acc.B.h); 
}

static
void
mc6809ASRB  (
    void
)
{
    OP_ASR (gpMC6809->acc.B.l); 
}

static
void
mc6809ASR  (
    void
)
{
    Word wAddress;
    Byte bValue;

    wAddress = MC6809ReadMemEA ();
    MEM_READ_BYTE_NC (bValue, wAddress);

    OP_ASR (bValue);
    MEM_WRITE_BYTE (wAddress, bValue);
}

static
int32
mc6809ASR_I  (
    void
)
{
    int32 nCycles;
    Word        wAddress;
    Byte        bValue;

    wAddress = MC6809ReadMemIA (&nCycles);
    MEM_READ_BYTE_NC (bValue, wAddress);

    OP_ASR (bValue);
    MEM_WRITE_BYTE (wAddress, bValue);

    return (nCycles);
}

static 
void
mc6809BCC (
    void
)
{
    OP_BR (!(gpMC6809->cc.bit.c));
}

static 
int32
mc6809LBCC (
    void
)
{
    int32 nCycles;
    OP_LBR (!(gpMC6809->cc.bit.c), &nCycles);
    return (nCycles); 
}

static 
void
mc6809BCS (
    void
)
{
    OP_BR (gpMC6809->cc.bit.c);
}

static 
int32
mc6809LBCS (
    void
)
{
    int32 nCycles;
    OP_LBR (gpMC6809->cc.bit.c, &nCycles);
    return (nCycles);
}

static 
void
mc6809BEQ (
    void
)
{
    OP_BR (gpMC6809->cc.bit.z);
}

static 
int32
mc6809LBEQ (
    void
)
{
    int32 nCycles;
    OP_LBR (gpMC6809->cc.bit.z, &nCycles);
    return (nCycles);
}

static 
void
mc6809BGE (
    void
)
{
    OP_BR (!(gpMC6809->cc.bit.n ^ gpMC6809->cc.bit.v));
}

static 
int32
mc6809LBGE (
    void
)
{
    int32 nCycles;
    OP_LBR (!(gpMC6809->cc.bit.n ^ gpMC6809->cc.bit.v), &nCycles);
    return (nCycles);
}

static 
void
mc6809BGT (
    void
)
{
    OP_BR (!(gpMC6809->cc.bit.z | (gpMC6809->cc.bit.n ^ gpMC6809->cc.bit.v)));
}

static 
int32
mc6809LBGT (
    void
)
{
    int32 nCycles;
    OP_LBR (
        !(gpMC6809->cc.bit.z | (gpMC6809->cc.bit.n ^ gpMC6809->cc.bit.v)), &nCycles
    );
    return (nCycles);
}

static 
void
mc6809BHI (
    void
)
{
    OP_BR (!(gpMC6809->cc.bit.c | gpMC6809->cc.bit.z));
}

static 
int32
mc6809LBHI (
    void
)
{
    int32 nCycles;
    OP_LBR (!(gpMC6809->cc.bit.c | gpMC6809->cc.bit.z), &nCycles);
    return (nCycles);
}

static 
void
mc6809BITA (
    void
)
{
    OP_BIT (gpMC6809->acc.B.h);
}

static 
void
mc6809BITB (
    void
)
{
    OP_BIT (gpMC6809->acc.B.l);
}

static 
int32
mc6809BITA_I (
    void
)
{
    int32 nCycles;
    OP_BIT_I (gpMC6809->acc.B.h, &nCycles);
    return (nCycles);
}

static 
int32
mc6809BITB_I (
    void
)
{
    int32 nCycles;
    OP_BIT_I (gpMC6809->acc.B.l, &nCycles);
    return (nCycles);
}

static 
void
mc6809BLE (
    void
)
{
    OP_BR (gpMC6809->cc.bit.z | (gpMC6809->cc.bit.n ^ gpMC6809->cc.bit.v));
}

static 
int32
mc6809LBLE (
    void
)
{
    int32 nCycles;
    OP_LBR (gpMC6809->cc.bit.z | (gpMC6809->cc.bit.n ^ gpMC6809->cc.bit.v), &nCycles);
    return (nCycles);
}

static 
void
mc6809BLS (
    void
)
{
    OP_BR (gpMC6809->cc.bit.c | gpMC6809->cc.bit.z);
}

static 
int32
mc6809LBLS (
    void
)
{
    int32 nCycles;
    OP_LBR (gpMC6809->cc.bit.c | gpMC6809->cc.bit.z, &nCycles);
    return (nCycles);
}

static 
void
mc6809BLT (
    void
)
{
    OP_BR (gpMC6809->cc.bit.n ^ gpMC6809->cc.bit.v);
}

static 
int32
mc6809LBLT (
    void
)
{
    int32 nCycles;
    OP_LBR (gpMC6809->cc.bit.n ^ gpMC6809->cc.bit.v, &nCycles);
    return (nCycles);
}

static 
void
mc6809BMI (
    void
)
{
    OP_BR (gpMC6809->cc.bit.n);
}

static 
int32
mc6809LBMI (
    void
)
{
    int32 nCycles;
    OP_LBR (gpMC6809->cc.bit.n, &nCycles);
    return (nCycles);
}

static 
void
mc6809BNE (
    void
)
{
    OP_BR (!(gpMC6809->cc.bit.z));
}

static 
int32
mc6809LBNE (
    void
)
{
    int32 nCycles;
    OP_LBR (!(gpMC6809->cc.bit.z), &nCycles);
    return (nCycles);
}

static 
void
mc6809BPL (
    void
)
{
    OP_BR (!(gpMC6809->cc.bit.n));
}

static 
int32
mc6809LBPL (
    void
)
{
    int32 nCycles;
    OP_LBR (!(gpMC6809->cc.bit.n), &nCycles);
    return (nCycles);
}

static 
void
mc6809BRA (
    void
)
{
    Byte bOffset;

    /*
     *  We don't use OP_BR() to avoid an unecessary check since we know
     *  we are always going to branch.
     */
    MEM_READ_BYTE (bOffset);

    if (bOffset & 0x80)
    {
        gpMC6809->pc += (Word)bOffset | 0xff00;
    }
    else
    {
        gpMC6809->pc += (Word)bOffset;
    }
}

static 
int32
mc6809LBRA (
    void
)
{
    Word wOffset;

    /*
     *  We don't use OP_LBR() to avoid an unecessary check since we know
     *  we are always going to branch.
     */
    MEM_READ_WORD (wOffset);
    gpMC6809->pc += wOffset;

    return (0);
}

static 
void
mc6809BRN (
    void
)
{
    Byte bOffset;

    /*
     *  We don't use OP_BR() to avoid an unecessary check since we know
     *  we are never going to branch.
     */
    MEM_READ_BYTE (bOffset);
}

static 
int32
mc6809LBRN (
    void
)
{
    Word wOffset;

    /*
     *  We don't use OP_LBR() to avoid an unecessary check since we know
     *  we are never going to branch.
     */
    MEM_READ_WORD (wOffset);

    return (0);
}

static 
void
mc6809BSR (
    void
)
{
    Byte bO;

    MEM_READ_BYTE (bO);

    gpMC6809->s -= 1;
    MEM_WRITE_BYTE (gpMC6809->s, (Byte)(gpMC6809->pc));
    gpMC6809->s -= 1;
    MEM_WRITE_BYTE (gpMC6809->s, (Byte)(gpMC6809->pc >> 8));

    if (bO & 0x80) 
    {
        gpMC6809->pc += ((Word)bO | 0xff00);
    }
    else
    {
        gpMC6809->pc += (Word)bO; 
    }
}

static 
int32
mc6809LBSR (
    void
)
{
    Word wO;

    MEM_READ_WORD (wO);

    gpMC6809->s -= 1;
    MEM_WRITE_BYTE (gpMC6809->s, (Byte)(gpMC6809->pc));
    gpMC6809->s -= 1;
    MEM_WRITE_BYTE (gpMC6809->s, (Byte)(gpMC6809->pc >> 8));
    gpMC6809->pc += wO;

    return 0;
}

static 
void
mc6809BVC (
    void
)
{
    OP_BR (!(gpMC6809->cc.bit.v));
}

static 
int32
mc6809LBVC (
    void
)
{
    int32 nCycles;
    OP_LBR (!(gpMC6809->cc.bit.v), &nCycles);
    return (nCycles);
}

static 
void
mc6809BVS (
    void
)
{
    OP_BR (gpMC6809->cc.bit.v);
}

static 
int32
mc6809LBVS (
    void
)
{
    int32 nCycles;
    OP_LBR (gpMC6809->cc.bit.v, &nCycles);
    return (nCycles);
}

static 
void
mc6809CLRA (
    void
)
{
    OP_CLR (gpMC6809->acc.B.h);
}

static 
void
mc6809CLRB (
    void
)
{
    OP_CLR (gpMC6809->acc.B.l);
}

static 
void
mc6809CLR (
    void
)
{
    Word wAddress;
    Byte bO;

    wAddress = MC6809ReadMemEA ();
/*
    MEM_READ_BYTE_NC (bO, wAddress);
*/
    OP_CLR (bO);
    MEM_WRITE_BYTE (wAddress, bO);
}

static 
int32
mc6809CLR_I (
    void
)
{
    int32  nCycles;
    Word wAddress;
    Byte bO;

    wAddress = MC6809ReadMemIA (&nCycles);
/*
    MEM_READ_BYTE_NC (bO, wAddress);
*/
    OP_CLR (bO);
    MEM_WRITE_BYTE (wAddress, bO);

    return (nCycles);
}

static 
void
mc6809CMPA (
    void
)
{
    OP_CMP_BYTE (gpMC6809->acc.B.h);
}

static 
void
mc6809CMPB (
    void
)
{
    OP_CMP_BYTE (gpMC6809->acc.B.l);
}

static
int32
mc6809CMPA_I (
    void
)
{
    int32 nCycles;
    OP_CMP_I_BYTE (gpMC6809->acc.B.h, &nCycles);
    return (nCycles);
}

static
int32
mc6809CMPB_I (
    void
)
{
    int32 nCycles;
    OP_CMP_I_BYTE (gpMC6809->acc.B.l, &nCycles);
    return (nCycles);
}

static 
void
mc6809CMPD (
    void
)
{
    OP_CMP_WORD (gpMC6809->acc.W);
}

static 
void
mc6809CMPX (
    void
)
{
    OP_CMP_WORD (gpMC6809->x);
}

static 
void
mc6809CMPY (
    void
)
{
    OP_CMP_WORD (gpMC6809->y);
}

static 
void
mc6809CMPU (
    void
)
{
    OP_CMP_WORD (gpMC6809->u);
}

static 
void
mc6809CMPS (
    void
)
{
    OP_CMP_WORD (gpMC6809->s);
}

static
int32
mc6809CMPD_I (
    void
)
{
    int32  nCycles;
    Word wO;
    Word wT;
    Word wR;

    wO = MC6809ReadMemIndexedWordOperand (&nCycles);
    wR = gpMC6809->acc.W - wO;
    wT = (gpMC6809->acc.W ^ wO) & 0x8000 ? 1 : 0;
    gpMC6809->cc.bit.n = BTST15 (wR);
    gpMC6809->cc.bit.z = wR ? 0 : 1;
    gpMC6809->cc.bit.v = wT & (((wO ^ wR) & 0x8000) ? 0 : 1);
    gpMC6809->cc.bit.c = 
        ((((~(gpMC6809->acc.W)) & wO) & 0x8000) ? 1 : 0) | 
        (!wT & gpMC6809->cc.bit.n);

    return (nCycles); 
}

static
int32
mc6809CMPX_I (
    void
)
{
    int32  nCycles;
    Word wO;
    Word wT;
    Word wR;

    wO = MC6809ReadMemIndexedWordOperand (&nCycles);
    wR = gpMC6809->x - wO;
    wT = (gpMC6809->x ^ wO) & 0x8000 ? 1 : 0;
    gpMC6809->cc.bit.n = BTST15 (wR);
    gpMC6809->cc.bit.z = wR ? 0 : 1;
    gpMC6809->cc.bit.v = wT & (((wO ^ wR) & 0x8000) ? 0 : 1);
    gpMC6809->cc.bit.c = 
        ((((~(gpMC6809->x)) & wO) & 0x8000) ? 1 : 0) | (!wT & gpMC6809->cc.bit.n);

    return (nCycles); 
}

static
int32
mc6809CMPY_I (
    void
)
{
    int32  nCycles;
    Word wO;
    Word wT;
    Word wR;

    wO = MC6809ReadMemIndexedWordOperand (&nCycles);
    wR = gpMC6809->y - wO;
    wT = (gpMC6809->y ^ wO) & 0x8000 ? 1 : 0;
    gpMC6809->cc.bit.n = BTST15 (wR);
    gpMC6809->cc.bit.z = wR ? 0 : 1;
    gpMC6809->cc.bit.v = wT & (((wO ^ wR) & 0x8000) ? 0 : 1);
    gpMC6809->cc.bit.c = 
        ((((~(gpMC6809->y)) & wO) & 0x8000) ? 1 : 0) | (!wT & gpMC6809->cc.bit.n);

    return (nCycles); 
}

static
int32
mc6809CMPU_I (
    void
)
{
    int32  nCycles;
    Word wO;
    Word wT;
    Word wR;

    wO = MC6809ReadMemIndexedWordOperand (&nCycles);
    wR = gpMC6809->u - wO;
    wT = (gpMC6809->u ^ wO) & 0x8000 ? 1 : 0;
    gpMC6809->cc.bit.n = BTST15 (wR);
    gpMC6809->cc.bit.z = wR ? 0 : 1;
    gpMC6809->cc.bit.v = wT & (((wO ^ wR) & 0x8000) ? 0 : 1);
    gpMC6809->cc.bit.c = 
        ((((~(gpMC6809->u)) & wO) & 0x8000) ? 1 : 0) | (!wT & gpMC6809->cc.bit.n);

    return (nCycles); 
}

static
int32
mc6809CMPS_I (
    void
)
{
    int32  nCycles;
    Word wO;
    Word wT;
    Word wR;

    wO = MC6809ReadMemIndexedWordOperand (&nCycles);
    wR = gpMC6809->s - wO;
    wT = (gpMC6809->s ^ wO) & 0x8000 ? 1 : 0;
    gpMC6809->cc.bit.n = BTST15 (wR);
    gpMC6809->cc.bit.z = wR ? 0 : 1;
    gpMC6809->cc.bit.v = wT & (((wO ^ wR) & 0x8000) ? 0 : 1);
    gpMC6809->cc.bit.c = 
        ((((~(gpMC6809->s)) & wO) & 0x8000) ? 1 : 0) | (!wT & gpMC6809->cc.bit.n);

    return (nCycles); 
}

static 
void
mc6809COMA (
    void
)
{
    OP_COM (gpMC6809->acc.B.h);
}

static 
void
mc6809COMB (
    void
)
{
    OP_COM (gpMC6809->acc.B.l);
}

static 
void
mc6809COM (
    void
)
{
    Word wAddress;
    Byte bO;

    wAddress = MC6809ReadMemEA ();
    MEM_READ_BYTE_NC (bO, wAddress);

    OP_COM (bO);
    MEM_WRITE_BYTE (wAddress, bO);
}

static 
int32
mc6809COM_I (
    void
)
{
    int32  nCycles;
    Word wAddress;
    Byte bO;

    wAddress = MC6809ReadMemIA (&nCycles);
    MEM_READ_BYTE_NC (bO, wAddress);

    OP_COM (bO);
    MEM_WRITE_BYTE (wAddress, bO);

    return (nCycles);
}

static 
void
mc6809CWAI (
    void
)
{
    /*
     *  Modify the flags based on the next byte.
     */
    Byte bNext;
    MEM_READ_BYTE (bNext);
    gpMC6809->cc.all &= bNext;

    /*
     *  Force an interrupt.
     */
    gpMC6809->pendingCWAI = 1;
    gpMC6809->ICount = 0;
}
 
static
void
mc6809DAA (
    void
)
{
    Byte bC = 0;
    Byte bLSN = (gpMC6809->acc.B.h & 0x0f);
    Byte bMSN = (gpMC6809->acc.B.h & 0xf0) >> 4;
    Word wT;

    if (gpMC6809->cc.bit.h || (bLSN > 9))
    {
        bC |= 0x06;
    }

    if (gpMC6809->cc.bit.c || (bMSN > 9) || ((bMSN > 8) && (bLSN > 9)))
    {
        gpMC6809->cc.bit.c = 1;
        bC |= 0x60;
    }
 
    wT = (Word)(gpMC6809->acc.B.h) + bC;
/*
 *  KSB:  The following was the original set for the carry bit.  
 *  Unfortunately, this didn't seem to work in super pac-man so instead, 
 *  I added the line in the if statement above.
    gpMC6809->cc.bit.c = BTST8 (wT);
*/
    gpMC6809->acc.B.h = (Byte)wT;    

    gpMC6809->cc.bit.n = BTST7 (gpMC6809->acc.B.h);
    gpMC6809->cc.bit.z = !(gpMC6809->acc.B.h);
}

static 
void
mc6809DECA (
    void
)
{
    OP_DEC (gpMC6809->acc.B.h);
}

static 
void
mc6809DECB (
    void
)
{
    OP_DEC (gpMC6809->acc.B.l);
}

static
void
mc6809DEC (
    void
)
{
    Word wAddress;
    Byte bO;

    wAddress = MC6809ReadMemEA ();
    MEM_READ_BYTE_NC (bO, wAddress);
    
    OP_DEC (bO);
    MEM_WRITE_BYTE (wAddress, bO);
}

static
int32
mc6809DEC_I (
    void
)
{
    int32  nCycles;
    Word wAddress;
    Byte bO;

    wAddress = MC6809ReadMemIA (&nCycles);
    MEM_READ_BYTE_NC (bO, wAddress);
    
    OP_DEC (bO);
    MEM_WRITE_BYTE (wAddress, bO);

    return (nCycles);
}

static 
void
mc6809EORA (
    void
)
{
    OP_EOR (gpMC6809->acc.B.h);
}

static 
void
mc6809EORB (
    void
)
{
    OP_EOR (gpMC6809->acc.B.l);
}

static 
int32
mc6809EORA_I (
    void
)
{
    int32 nCycles;
    OP_EOR_I (gpMC6809->acc.B.h, &nCycles);
    return (nCycles);
}

static 
int32
mc6809EORB_I (
    void
)
{
    int32 nCycles;
    OP_EOR_I (gpMC6809->acc.B.l, &nCycles);
    return (nCycles);
}

static
void
mc6809EXG (
    void
)
{
    int32   nR1;
    int32   nR2;
    Byte  bO;

    MEM_READ_BYTE (bO);
    nR1 = (bO & 0xf0) >> 4;
    nR2 = (bO & 0x0f);
    if (nR1 <= 5)
    {
        if (nR2 > 5) 
        {
            printf ("Invalid exchange register.\n");
            fflush (stdout);
            exit (0);
        }
        switch (bO)
        {
            case 0x01: 
            case 0x10:
                SWAP_WORD (gpMC6809->acc.W, gpMC6809->x);
                break;
            case 0x02:
            case 0x20:
                SWAP_WORD (gpMC6809->acc.W, gpMC6809->y);
                break;
            case 0x03:
            case 0x30:
                SWAP_WORD (gpMC6809->acc.W, gpMC6809->u);
                break;
            case 0x04:
            case 0x40:
                SWAP_WORD (gpMC6809->acc.W, gpMC6809->s);
                break;
            case 0x05:
            case 0x50:
                SWAP_WORD (gpMC6809->acc.W, gpMC6809->pc);
                break;
            case 0x12:
            case 0x21:
                SWAP_WORD (gpMC6809->x, gpMC6809->y);
                break;
            case 0x13:
            case 0x31:
                SWAP_WORD (gpMC6809->x, gpMC6809->u);
                break;
            case 0x14:
            case 0x41:
                SWAP_WORD (gpMC6809->x, gpMC6809->s);
                break;
            case 0x15:
            case 0x51:
                SWAP_WORD (gpMC6809->x, gpMC6809->pc);
                break;
            case 0x23:
            case 0x32:
                SWAP_WORD (gpMC6809->y, gpMC6809->u);
                break;
            case 0x24:
            case 0x42:
                SWAP_WORD (gpMC6809->y, gpMC6809->s);
                break;
            case 0x25:
            case 0x52:
                SWAP_WORD (gpMC6809->y, gpMC6809->pc);
                break;
            case 0x34:
            case 0x43:
                SWAP_WORD (gpMC6809->u, gpMC6809->s);
                break;
            case 0x35:
            case 0x53:
                SWAP_WORD (gpMC6809->u, gpMC6809->pc);
                break;
            case 0x45:
            case 0x54:
                SWAP_WORD (gpMC6809->s, gpMC6809->pc);
                break;
        }
    }
    else 
    if ((nR1 >= 8) && (nR2 <= 11))
    {
        if ((nR2 < 8) || (nR2 > 11))
        {
            printf ("Invalid exchange register.\n");
            fflush (stdout);
            exit (0);
        }
        switch (bO) 
        {
            case 0x98:
            case 0x89:
                SWAP_BYTE (gpMC6809->acc.B.h, gpMC6809->acc.B.l);
                break; 
            case 0xa8:
            case 0x8a:
                SWAP_BYTE (gpMC6809->acc.B.h, gpMC6809->cc.all);
                break; 
            case 0xb8:
            case 0x8b:
                SWAP_BYTE (gpMC6809->acc.B.h, gpMC6809->dp);
                gpMC6809->dpHi = (Word)(gpMC6809->dp << 8);
                break; 
            case 0xa9:
            case 0x9a:
                SWAP_BYTE (gpMC6809->acc.B.l, gpMC6809->cc.all);
                break; 
            case 0xb9:
            case 0x9b:
                SWAP_BYTE (gpMC6809->acc.B.l, gpMC6809->dp);
                gpMC6809->dpHi = (Word)(gpMC6809->dp << 8);
                break; 
            case 0xba:
            case 0xab:
                SWAP_BYTE (gpMC6809->cc.all, gpMC6809->dp);
                gpMC6809->dpHi = (Word)(gpMC6809->dp << 8);
                break; 
        }
    }
    else
    {
        printf ("Invalid exchange register.\n");
        fflush (stdout);
        exit (0);
    }
}
       
static
void
mc6809INCA (
    void
)
{
    OP_INC (gpMC6809->acc.B.h);
}
    
static
void
mc6809INCB (
    void
)
{
    OP_INC (gpMC6809->acc.B.l);
}

static
void
mc6809INC (
    void
)
{
    Word wAddress;
    Byte bO;

    wAddress = MC6809ReadMemEA ();
    MEM_READ_BYTE_NC (bO, wAddress);

    OP_INC (bO);

    MEM_WRITE_BYTE (wAddress, bO);
}

static
int32
mc6809INC_I (
    void
)
{
    int32  nCycles; 
    Word wAddress;
    Byte bO;

    wAddress = MC6809ReadMemIA (&nCycles);
    MEM_READ_BYTE_NC (bO, wAddress);

    OP_INC (bO);

    MEM_WRITE_BYTE (wAddress, bO);

    return (nCycles);
}

static
void
mc6809JMP (
    void
)
{
    gpMC6809->pc = MC6809ReadMemEA ();
}

static
int32
mc6809JMP_I (
    void
)
{
    int32 nCycles;
    gpMC6809->pc = MC6809ReadMemIA (&nCycles);
    return (nCycles);
}

static
void 
mc6809JSR (
    void
)
{
    Word wAddress;

    wAddress = MC6809ReadMemEA ();

    gpMC6809->s -= 1;
    MEM_WRITE_BYTE (gpMC6809->s, gpMC6809->pc);
    gpMC6809->s -= 1;
    MEM_WRITE_BYTE (gpMC6809->s, (gpMC6809->pc >> 8));

    gpMC6809->pc = wAddress;
}

static
int32 
mc6809JSR_I (
    void
)
{
    int32  nCycles;
    Word wAddress;

    wAddress = MC6809ReadMemIA (&nCycles);

    gpMC6809->s -= 1;
    MEM_WRITE_BYTE (gpMC6809->s, gpMC6809->pc);
    gpMC6809->s -= 1;
    MEM_WRITE_BYTE (gpMC6809->s, (gpMC6809->pc >> 8));

    gpMC6809->pc = wAddress;

    return (nCycles);
}

static
void
mc6809LDA (
    void
)
{
    OP_LD_BYTE (gpMC6809->acc.B.h);
}

static
void
mc6809LDB (
    void
)
{
    OP_LD_BYTE (gpMC6809->acc.B.l);
}

static
int32
mc6809LDA_I (
    void
)
{
    int32 nCycles;
    OP_LD_I_BYTE (gpMC6809->acc.B.h, &nCycles);
    return (nCycles);
}

static
int32
mc6809LDB_I (
    void
)
{
    int32 nCycles;
    OP_LD_I_BYTE (gpMC6809->acc.B.l, &nCycles);
    return (nCycles);
}

static
void
mc6809LDD (
    void
)
{
    OP_LD_WORD (gpMC6809->acc.W);
}

static
void
mc6809LDX (
    void
)
{
    OP_LD_WORD (gpMC6809->x);
}

static
void
mc6809LDY (
    void
)
{
    OP_LD_WORD (gpMC6809->y);
}

static
void
mc6809LDS (
    void
)
{
    OP_LD_WORD (gpMC6809->s);
}

static
void
mc6809LDU (
    void
)
{
    OP_LD_WORD (gpMC6809->u);
}

static
int32
mc6809LDD_I (
    void
)
{
    int32 nCycles;
    OP_LD_I_WORD (gpMC6809->acc.W, &nCycles);
    return (nCycles);
}

static
int32
mc6809LDX_I (
    void
)
{
    int32 nCycles;
    OP_LD_I_WORD (gpMC6809->x, &nCycles);
    return (nCycles);
}

static
int32
mc6809LDY_I (
    void
)
{
    int32 nCycles;
    OP_LD_I_WORD (gpMC6809->y, &nCycles);
    return (nCycles);
}

static
int32
mc6809LDS_I (
    void
)
{
    int32 nCycles;
    OP_LD_I_WORD (gpMC6809->s, &nCycles);
    return (nCycles);
}

static
int32
mc6809LDU_I (
    void
)
{
    int32 nCycles;
    OP_LD_I_WORD (gpMC6809->u, &nCycles);
    return (nCycles);
}

static
int32
mc6809LEAX (
    void
)
{
    int32 nCycles;
    gpMC6809->x = MC6809ReadMemIA (&nCycles);
    gpMC6809->cc.bit.z = !(gpMC6809->x);
    return (nCycles);
}

static
int32
mc6809LEAY (
    void
)
{
    int32 nCycles;
    gpMC6809->y = MC6809ReadMemIA (&nCycles);
    gpMC6809->cc.bit.z = !(gpMC6809->y);
    return (nCycles);
}

static
int32
mc6809LEAS (
    void
)
{
    int32 nCycles;
    gpMC6809->s = MC6809ReadMemIA (&nCycles);
    return (nCycles);
}
    
static
int32
mc6809LEAU (
    void
)
{
    int32 nCycles;
    gpMC6809->u = MC6809ReadMemIA (&nCycles);
    return (nCycles);
}

static
void
mc6809LSLA (
    void
)
{
    OP_LSL (gpMC6809->acc.B.h);
}
    
static
void
mc6809LSLB (
    void
)
{
    OP_LSL (gpMC6809->acc.B.l);
}
    
static
void
mc6809LSL (
    void
)
{
    Word wAddress;
    Byte bO;

    wAddress = MC6809ReadMemEA ();
    MEM_READ_BYTE_NC (bO, wAddress);

    OP_LSL (bO);

    MEM_WRITE_BYTE (wAddress, bO);
}

static
int32
mc6809LSL_I (
    void
)
{
    int32  nCycles;
    Word wAddress;
    Byte bO;

    wAddress = MC6809ReadMemIA (&nCycles);
    MEM_READ_BYTE_NC (bO, wAddress);

    OP_LSL (bO);

    MEM_WRITE_BYTE (wAddress, bO);

    return (nCycles);
}

static
void
mc6809LSRA (
    void
)
{
    OP_LSR (gpMC6809->acc.B.h);
}
    
static
void
mc6809LSRB (
    void
)
{
    OP_LSR (gpMC6809->acc.B.l);
}
    
static
void
mc6809LSR (
    void
)
{
    Word wAddress;
    Byte bO;

    wAddress = MC6809ReadMemEA ();
    MEM_READ_BYTE_NC (bO, wAddress);

    OP_LSR (bO);

    MEM_WRITE_BYTE (wAddress, bO);
}

static
int32
mc6809LSR_I (
    void
)
{
    int32  nCycles;
    Word wAddress;
    Byte bO;

    wAddress = MC6809ReadMemIA (&nCycles);
    MEM_READ_BYTE_NC (bO, wAddress);

    OP_LSR (bO);

    MEM_WRITE_BYTE (wAddress, bO);

    return (nCycles);
}

static
void
mc6809MUL (
    void
)
{
    gpMC6809->acc.W = gpMC6809->acc.B.h * gpMC6809->acc.B.l;
    gpMC6809->cc.bit.c = BTST7 (gpMC6809->acc.B.l);
    gpMC6809->cc.bit.z = !(gpMC6809->acc.W);
}

static
void
mc6809NEGA (
    void
)
{
    OP_NEG (gpMC6809->acc.B.h);
}
    
static
void
mc6809NEGB (
    void
)
{
    OP_NEG (gpMC6809->acc.B.l);
}
    
static
void
mc6809NEG (
    void
)
{
    Word wAddress;
    Byte bO;

    wAddress = MC6809ReadMemEA ();
    MEM_READ_BYTE_NC (bO, wAddress);

    OP_NEG (bO);

    MEM_WRITE_BYTE (wAddress, bO);
}
    
static
int32
mc6809NEG_I (
    void
)
{
    int32  nCycles;
    Word wAddress;
    Byte bO;

    wAddress = MC6809ReadMemIA (&nCycles);
    MEM_READ_BYTE_NC (bO, wAddress);

    OP_NEG (bO);

    MEM_WRITE_BYTE (wAddress, bO);

    return (nCycles);
}
    
static
void
mc6809NOP (
    void
)
{
}

static
void
mc6809ORA (
    void
)
{
    OP_OR (gpMC6809->acc.B.h);
}

static
void
mc6809ORB (
    void
)
{
    OP_OR (gpMC6809->acc.B.l);
}

static
int32
mc6809ORA_I (
    void
)
{
    int32 nCycles;
    OP_OR_I (gpMC6809->acc.B.h, &nCycles);
    return (nCycles);
}

static
int32
mc6809ORB_I (
    void
)
{
    int32 nCycles;
    OP_OR_I (gpMC6809->acc.B.l, &nCycles);
    return (nCycles);
}

static
void
mc6809ORCC (
    void
)
{
    gpMC6809->cc.all |= MC6809ReadMemOperand ();
}

static
int32
mc6809PSHS (
    void
)
{
    Byte bO;

    MEM_READ_BYTE (bO);
    OP_PSH (bO, gpMC6809->s, gpMC6809->u);

    return (anPSHPULCycles[bO]);
}

static
int32
mc6809PSHU (
    void
)
{
    Byte bO;

    MEM_READ_BYTE (bO);
    OP_PSH (bO, gpMC6809->u, gpMC6809->s);

    return (anPSHPULCycles[bO]);
}

static
int32
mc6809PULS (
    void
)
{
    Byte bO;

    MEM_READ_BYTE (bO);
    OP_PUL (bO, gpMC6809->s, gpMC6809->u);

    return (anPSHPULCycles[bO]);
}

static
int32
mc6809PULU (
    void
)
{
    Byte bO;

    MEM_READ_BYTE (bO);
    OP_PUL (bO, gpMC6809->u, gpMC6809->s);

    return (anPSHPULCycles[bO]);
}

static
void
mc6809ROLA (
    void
)
{
    OP_ROL (gpMC6809->acc.B.h);
}
    
static
void
mc6809ROLB (
    void
)
{
    OP_ROL (gpMC6809->acc.B.l);
}
    
static
void
mc6809ROL (
    void
)
{
    Word wAddress;
    Byte bO;

    wAddress = MC6809ReadMemEA ();
    MEM_READ_BYTE_NC (bO, wAddress);

    OP_ROL (bO);

    MEM_WRITE_BYTE (wAddress, bO);
}

static
int32
mc6809ROL_I (
    void
)
{
    int32  nCycles;
    Word wAddress;
    Byte bO;

    wAddress = MC6809ReadMemIA (&nCycles);
    MEM_READ_BYTE_NC (bO, wAddress);

    OP_ROL (bO);

    MEM_WRITE_BYTE (wAddress, bO);

    return (nCycles);
}

static
void
mc6809RORA (
    void
)
{
    OP_ROR (gpMC6809->acc.B.h);
}
    
static
void
mc6809RORB (
    void
)
{
    OP_ROR (gpMC6809->acc.B.l);
}
    
static
void
mc6809ROR (
    void
)
{
    Word wAddress;
    Byte bO;

    wAddress = MC6809ReadMemEA ();
    MEM_READ_BYTE_NC (bO, wAddress);

    OP_ROR (bO);

    MEM_WRITE_BYTE (wAddress, bO);
}

static
int32
mc6809ROR_I (
    void
)
{
    int32  nCycles;
    Word wAddress;
    Byte bO;

    wAddress = MC6809ReadMemIA (&nCycles);
    MEM_READ_BYTE_NC (bO, wAddress);

    OP_ROR (bO);

    MEM_WRITE_BYTE (wAddress, bO);

    return (nCycles);
}

static
int32
mc6809RTI (
    void
)
{
    MEM_READ_BYTE_NC (gpMC6809->cc.all, gpMC6809->s);
    gpMC6809->s += 1;

    if (gpMC6809->cc.bit.e) 
    {
        OP_PUL (0xfe, gpMC6809->s, gpMC6809->u);
        return (15);
    }
    else
    {
        MEM_READ_WORD_NC (gpMC6809->pc, gpMC6809->s);
        gpMC6809->s += 2;
        return (6);
    }
}

static
void
mc6809RTS (
    void
)
{
    MEM_READ_WORD_NC (gpMC6809->pc, gpMC6809->s);
    gpMC6809->s += 2;
}

static
void
mc6809SBCA (
    void
)
{
    OP_SBC (gpMC6809->acc.B.h);
}

static
void
mc6809SBCB (
    void
)
{
    OP_SBC (gpMC6809->acc.B.l);
}

static
int32
mc6809SBCA_I (
    void
)
{
    int32 nCycles;
    OP_SBC_I (gpMC6809->acc.B.h, &nCycles);
    return (nCycles);
}

static
int32
mc6809SBCB_I (
    void
)
{
    int32 nCycles;
    OP_SBC_I (gpMC6809->acc.B.l, &nCycles);
    return (nCycles);
}

static
void
mc6809SEX (
    void
)
{
    gpMC6809->cc.bit.n = BTST7 (gpMC6809->acc.B.l);
    gpMC6809->cc.bit.z = !(gpMC6809->acc.B.l);
    gpMC6809->acc.B.h = gpMC6809->cc.bit.n ? 0xff : 0;
}

static
void
mc6809STA (
    void
)
{
    OP_ST_BYTE (gpMC6809->acc.B.h);
}
    
static
void
mc6809STB (
    void
)
{
    OP_ST_BYTE (gpMC6809->acc.B.l);
}
    
static
int32
mc6809STA_I (
    void
)
{
    int32 nCycles;
    OP_ST_I_BYTE (gpMC6809->acc.B.h, &nCycles);
    return (nCycles);
}
    
static
int32
mc6809STB_I (
    void
)
{
    int32 nCycles;
    OP_ST_I_BYTE (gpMC6809->acc.B.l, &nCycles);
    return (nCycles);
}
    
static
void
mc6809STD (
    void
)
{
    OP_ST_WORD (gpMC6809->acc.W);
}
    
static
void
mc6809STX (
    void
)
{
    OP_ST_WORD (gpMC6809->x);
}
    
static
void
mc6809STY (
    void
)
{
    OP_ST_WORD (gpMC6809->y);
}
    
static
void
mc6809STS (
    void
)
{
    OP_ST_WORD (gpMC6809->s);
}
    
static
void
mc6809STU (
    void
)
{
    OP_ST_WORD (gpMC6809->u);
}
    
static
int32
mc6809STD_I (
    void
)
{
    int32 nCycles;
    OP_ST_I_WORD (gpMC6809->acc.W, &nCycles);
    return (nCycles);
}
    
static
int32
mc6809STX_I (
    void
)
{
    int32 nCycles;
    OP_ST_I_WORD (gpMC6809->x, &nCycles);
    return (nCycles);
}
    
static
int32
mc6809STY_I (
    void
)
{
    int32 nCycles;
    OP_ST_I_WORD (gpMC6809->y, &nCycles);
    return (nCycles);
}
    
static
int32
mc6809STS_I (
    void
)
{
    int32 nCycles;
    OP_ST_I_WORD (gpMC6809->s, &nCycles);
    return (nCycles);
}
    
static
int32
mc6809STU_I (
    void
)
{
    int32 nCycles;
    OP_ST_I_WORD (gpMC6809->u, &nCycles);
    return (nCycles);
}
    
static
void
mc6809SUBA (
    void
)
{
    OP_SUB (gpMC6809->acc.B.h);
}

static
void
mc6809SUBB (
    void
)
{
    OP_SUB (gpMC6809->acc.B.l);
}

static
int32
mc6809SUBA_I (
    void
)
{
    int32 nCycles;
    OP_SUB_I (gpMC6809->acc.B.h, &nCycles);
    return (nCycles);
}

static
int32
mc6809SUBB_I (
    void
)
{
    int32 nCycles;
    OP_SUB_I (gpMC6809->acc.B.l, &nCycles);
    return (nCycles);
}

static
void
mc6809SUBD (
    void
)
{
    Word  wO;
    Word  wT;
    Word  wR;     

    wO = MC6809ReadMemWordOperand ();

    wR = gpMC6809->acc.W - wO;
    wT = (gpMC6809->acc.W ^ wO) & 0x8000 ? 1 : 0;
    gpMC6809->cc.bit.n = (wR & 0x8000) ? 1 : 0;
    gpMC6809->cc.bit.z = wR ? 0 : 1;
    gpMC6809->cc.bit.v = wT & (((wO ^ wR) & 0x8000) ? 0 : 1);
    gpMC6809->cc.bit.c = 
        ((((~(gpMC6809->acc.W)) & wO) & 0x8000) ? 1 : 0) | 
        (!wT & gpMC6809->cc.bit.n);
    gpMC6809->acc.W = wR;
}

static
int32
mc6809SUBD_I (
    void
)
{
    int32  nCycles;
    Word wO;
    Word wT;
    Word wR;     

    wO = MC6809ReadMemIndexedWordOperand (&nCycles);

    wR = gpMC6809->acc.W - wO;
    wT = (gpMC6809->acc.W ^ wO) & 0x8000 ? 1 : 0;
    gpMC6809->cc.bit.n = (wR & 0x8000) ? 1 : 0;
    gpMC6809->cc.bit.z = wR ? 0 : 1;
    gpMC6809->cc.bit.v = wT & (((wO ^ wR) & 0x8000) ? 0 : 1);
    gpMC6809->cc.bit.c = 
        ((((~(gpMC6809->acc.W)) & wO) & 0x8000) ? 1 : 0) | 
        (!wT & gpMC6809->cc.bit.n);
    gpMC6809->acc.W = wR;

    return (nCycles);
}

static
void
mc6809SWI (
    void
)
{
    gpMC6809->cc.bit.e = 1;
    OP_PSH (0xff, gpMC6809->s, gpMC6809->u);
    gpMC6809->cc.bit.f = 1;  
    gpMC6809->cc.bit.i = 1;
    MEM_READ_WORD_NC (gpMC6809->pc, 0xfffa);
}

static
void
mc6809SWI2 (
    void
)
{
    gpMC6809->cc.bit.e = 1;
    OP_PSH (0xff, gpMC6809->s, gpMC6809->u);
    MEM_READ_WORD_NC (gpMC6809->pc, 0xfff4);
}

static
void
mc6809SWI3 (
    void
)
{
    gpMC6809->cc.bit.e = 1;
    OP_PSH (0xff, gpMC6809->s, gpMC6809->u);
    MEM_READ_WORD_NC (gpMC6809->pc, 0xfff2);
}

static
void
mc6809SYNC (
    void
)
{
    /*
     *  Force an interrupt.
     */
    gpMC6809->pendingSYNC = 1;
    gpMC6809->ICount = 0;
}

static
void
mc6809TFR (
    void
)
{
    int32   nR1;
    int32   nR2;
    Byte  bO;

    MEM_READ_BYTE (bO);
    nR1 = (bO & 0xf0) >> 4;
    nR2 = (bO & 0x0f);
    if (nR1 <= 5)
    {
        if (nR2 > 5) 
        {
            printf ("Invalid transfer register.\n");
            fflush (stdout);
            exit (0);
        }
        switch (bO)
        {
            case 0x10:
                gpMC6809->acc.W = gpMC6809->x;
                break;
            case 0x20:
                gpMC6809->acc.W = gpMC6809->y;
                break;
            case 0x30:
                gpMC6809->acc.W = gpMC6809->u;
                break;
            case 0x40:
                gpMC6809->acc.W = gpMC6809->s;
                break;
            case 0x50:
                gpMC6809->acc.W = gpMC6809->pc;
                break;
            case 0x01:
                gpMC6809->x = gpMC6809->acc.W;
                break;
            case 0x21:
                gpMC6809->x = gpMC6809->y;
                break;
            case 0x31:
                gpMC6809->x = gpMC6809->u;
                break;
            case 0x41:
                gpMC6809->x = gpMC6809->s;
                break;
            case 0x51:
                gpMC6809->x = gpMC6809->pc;
                break;
            case 0x02:
                gpMC6809->y = gpMC6809->acc.W;
                break;
            case 0x12:
                gpMC6809->y = gpMC6809->x;
                break;
            case 0x32:
                gpMC6809->y = gpMC6809->u;
                break;
            case 0x42:
                gpMC6809->y = gpMC6809->s;
                break;
            case 0x52:
                gpMC6809->y = gpMC6809->pc;
                break;
            case 0x03:
                gpMC6809->u = gpMC6809->acc.W;
                break;
            case 0x13:
                gpMC6809->u = gpMC6809->x;
                break;
            case 0x23:
                gpMC6809->u = gpMC6809->y;
                break;
            case 0x43:
                gpMC6809->u = gpMC6809->s;
                break;
            case 0x53:
                gpMC6809->u = gpMC6809->pc;
                break;
            case 0x04:
                gpMC6809->s = gpMC6809->acc.W;
                break;
            case 0x14:
                gpMC6809->s = gpMC6809->x;
                break;
            case 0x24:
                gpMC6809->s = gpMC6809->y;
                break;
            case 0x34:
                gpMC6809->s = gpMC6809->u;
                break;
            case 0x54:
                gpMC6809->s = gpMC6809->pc;
                break;
            case 0x05:
                gpMC6809->pc = gpMC6809->acc.W;
                break;
            case 0x15:
                gpMC6809->pc = gpMC6809->x;
                break;
            case 0x25:
                gpMC6809->pc = gpMC6809->y;
                break;
            case 0x35:
                gpMC6809->pc = gpMC6809->u;
                break;
            case 0x45:
                gpMC6809->pc = gpMC6809->s;
                break;
        }
    }
    else 
    if ((nR1 >= 8) && (nR2 <= 11))
    {
        if ((nR2 < 8) || (nR2 > 11))
        {
            printf ("Invalid transfer register.\n");
            fflush (stdout);
            exit (0);
        }
        switch (bO)
        {
            case 0x98:
                gpMC6809->acc.B.h = gpMC6809->acc.B.l;
                break;
            case 0x89:
                gpMC6809->acc.B.l = gpMC6809->acc.B.h;
                break;
            case 0xa8:
                gpMC6809->acc.B.h = gpMC6809->cc.all;
                break;
            case 0x8a:
                gpMC6809->cc.all = gpMC6809->acc.B.h;
                break;
            case 0xb8:
                gpMC6809->acc.B.h = gpMC6809->dp;
                break;
            case 0x8b:
                gpMC6809->dp = gpMC6809->acc.B.h;
                gpMC6809->dpHi = (Word)(gpMC6809->dp << 8);
                break;
            case 0xa9:
                gpMC6809->acc.B.l = gpMC6809->cc.all;
                break;
            case 0x9a:
                gpMC6809->cc.all = gpMC6809->acc.B.l;
                break;
            case 0xb9:
                gpMC6809->acc.B.l = gpMC6809->dp;
                break;
            case 0x9b:
                gpMC6809->dp = gpMC6809->acc.B.l;
                gpMC6809->dpHi = (Word)(gpMC6809->dp << 8);
                break;
            case 0xba:
                gpMC6809->cc.all = gpMC6809->dp;
                break;
            case 0xab:
                gpMC6809->dp = gpMC6809->cc.all;
                gpMC6809->dpHi = (Word)(gpMC6809->dp << 8);
                break;
        }
    }
    else
    {
        printf ("Invalid transfer register.\n");
        fflush (stdout);
        exit (0);
    }
}

static
void
mc6809TSTA (
    void
)
{
    OP_TST (gpMC6809->acc.B.h);
}

static
void
mc6809TSTB (
    void
)
{
    OP_TST (gpMC6809->acc.B.l);
}

static
void
mc6809TST (
    void
)
{
    Word wAddress = MC6809ReadMemEA ();
    Byte bO;

    MEM_READ_BYTE_NC (bO, wAddress);
   
    OP_TST (bO);
}

static
int32
mc6809TST_I (
    void
)
{
    int32  nCycles;
    Word wAddress;
    Byte bO;

    wAddress = MC6809ReadMemIA (&nCycles);

    MEM_READ_BYTE_NC (bO, wAddress);
   
    OP_TST (bO);

    return (nCycles);
}
