///////////////////////////////////////////////////////////////////////////////
//
//  File:    cpu.cpp
//
//  Class:   CPU
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class serves as a base class for a CPU emulation.  
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>

//  Application Headers.
#include "reptypes.h"
#include "cpu.h"
#include "appfile.h"
#include "replay.h"
#include "sound.h"
#include "cpurh.h"
#include "cpuwh.h"
#include "space.h"
#include "input.h"
#ifdef DEBUGGER
#include "debugger.h"
#endif

//  Forward class declarations.
class Game;


///////////////////////////////////////////////////////////////////////////////
//  Static Data.
///////////////////////////////////////////////////////////////////////////////
//
//  The following pointers are used to allow easy (i.e.  fast) 
//  access to the currently running CPU object as well as its address space.
//
CPU*       CPU::sm_pCPU   = NULL;
AddrSpace* CPU::sm_pSpace = NULL;



///////////////////////////////////////////////////////////////////////////////
//
//  Function: CPU
//
//  Description:
//
//      This is the main constructor for a CPU object.  It is protected
//      so that only derived classes have access to it.  Clients should
//      use the build method to create a CPU object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      pGame (input)
//          The game that is running the CPU.
//
//      pSpace (input)
//          The address space that the CPU is primarily bound to.
//
//      ePurpose (input)
//          The purpose of the CPU.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
CPU::CPU(
    const KString&  iName,
    Game*           pGame,
    AddrSpace*      pSpace,
    CPUPurpose      ePurpose /* = GAME */
)
:
    RepBase                ( iName ),
    m_pGame                ( pGame ),
    m_pSpace               ( pSpace ),
    m_ePurpose             ( ePurpose ),
    m_readMemHandlerList   ( ),
    m_writeMemHandlerList  ( ),
    m_readPortHandlerList  ( ),
    m_writePortHandlerList ( ),
    m_nCyclesTotal         ( 0 ),
    m_nCyclesThisRun       ( 0 ),
    m_nCyclesCarry         ( 0 ),
    m_nClockSpeed          ( 1000000 ),
    m_nIntsPerFrame        ( 1 ),
    m_nIntsThisFrame       ( 0 ),
    m_bEnable              ( TRUE ),
    m_bDisable             ( FALSE ),
    m_bIEnable             ( TRUE ),
    m_pfInterrupt          ( ( IntFunc )NULL ),
    m_nIntVector           ( 0 )
{
    //  Check the parameters.
    ASSERT( pSpace != NULL );

    //  Initialization should be done in init( ).
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: init
//
//  Description:
//
//      This is called to initialize the CPU object.  By using an init
//      member we get access to virtual functions that we wouldn't in the
//      constructor.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPU::init(
)
{
    //  If the sound is turned off and this is a sound CPU then disable it
    //  permanently.
    if( 
        ( !( Replay::s_instance( ).getSound( )->isEnabled( ) ) ) &&
        ( m_ePurpose == SOUND )
    )
    {
        m_bEnable  = FALSE;
        m_bDisable = TRUE;
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~CPU
//
//  Description:
//
//      This is the destructor for the CPU object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
CPU::~CPU(
)
{
    //  Indices used for iterations.
    DWord dwI;


    //  If we're active then deactivate.
    if( this == sm_pCPU )
    {
        deactivate( );
    }

    //  Clean up the lists of handlers.
    for( dwI = 0 ; dwI < m_readMemHandlerList.entries( ) ; dwI += 1 )
    {
        delete m_readMemHandlerList[ dwI ];
    }
    m_readMemHandlerList.clear( );

    for( dwI = 0 ; dwI < m_writeMemHandlerList.entries( ) ; dwI += 1 )
    {
        delete m_writeMemHandlerList[ dwI ];
    }
    m_writeMemHandlerList.clear( );

    for( dwI = 0 ; dwI < m_readPortHandlerList.entries( ) ; dwI += 1 )
    {
        delete m_readPortHandlerList[ dwI ];
    }
    m_readPortHandlerList.clear( );

    for( dwI = 0 ; dwI < m_writePortHandlerList.entries( ) ; dwI += 1 )
    {
        delete m_writePortHandlerList[ dwI ];
    }
    m_writePortHandlerList.clear( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: save
//
//  Description:
//
//      This member will save the cpu to a file for retrieval at a later
//      date.
//
//  Parameters:
//
//      pSaveFile
//          The file to save the cpu to.
//
//  Returns:
//
//      TRUE if there were errors during the save.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
CPU::save(
    AppFile* pSaveFile
)
{
    ASSERT( pSaveFile != NULL );

    //  We always allow the base class to save itself first.
    RepBase::save( pSaveFile );

    //  Save the member data that keeps the state.  The remainder of the
    //  member data does not change and should not need to be saved.
    pSaveFile->clearTotal( );
    pSaveFile->write( &m_bEnable,  sizeof( m_bEnable ) );
    pSaveFile->write( &m_bIEnable, sizeof( m_bIEnable ) );

    //  Was it saved successfully?
    return(
        pSaveFile->total( ) != ( sizeof( m_bEnable ) + sizeof( m_bIEnable ) )
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: load
//
//  Description:
//
//      This member will load the cpu from a file that it was previously
//      saved to.
//
//  Parameters:
//
//      pLoadFile
//          The file to load the CPU from.
//
//  Returns:
//
//      TRUE if there were errors during the load.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
CPU::load(
    AppFile* pLoadFile
)
{
    ASSERT( pLoadFile != NULL );

    //  We always allow the base class to load itself first.
    RepBase::load( pLoadFile );

    //  Load the member data that keeps the state.
    pLoadFile->clearTotal( );
    pLoadFile->read( &m_bEnable,  sizeof( m_bEnable ) );
    pLoadFile->read( &m_bIEnable, sizeof( m_bIEnable ) );

    //  Was it loaded successfully?
    return(
        pLoadFile->total( ) != ( sizeof( m_bEnable ) + sizeof( m_bIEnable ) )
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: readMemHandler
//
//  Description:
//
//      This function is called from a game to add a memory handler for
//      CPU reads to the list of handlers.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the handler.
//
//      dwStart (input)
//          The start of the range the handler is to be used for.
//
//      dwEnd (input)
//          The end of the range the handler is to be used for.
//
//      pfHandler (input)
//          A pointer to the function that will handle the read.
//
//      pData (input)
//          A client data pointer.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPU::readMemHandler(
    const KString&          iName,
    DWord                   dwStart,
    DWord                   dwEnd,
    ReadHandler::ReadFunc   pfHandler,
    void*                   pData
)
{
    //  Check the parameters.
    ASSERT( dwStart <= dwEnd );
    ASSERT( pfHandler != NULL );

    //  If the handler exists then replace the information.
    for( DWord dwI = 0 ; dwI < m_readMemHandlerList.entries( ) ; dwI += 1 )
    {
        ReadHandler* pHandler = m_readMemHandlerList[ dwI ];

        if( 
            ( pHandler->getStart( ) == dwStart ) &&
            ( pHandler->getEnd( )   == dwEnd   )
        )
        {
            pHandler->setHandler( pfHandler );
            pHandler->setData( pData );
            return;
        }
    }
       
    //  Add the handler to the list.
    m_readMemHandlerList.add( 
        new ReadHandler( iName, dwStart, dwEnd, pfHandler, pData )
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: readMem
//
//  Description:
//
//      This function can be called from a CPU core when a value is to be
//      read from somewhere in the address space.  This will be bypassed if
//      the core being used supports direct calls of handler functions.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to read from.
//
//  Returns:
//
//      The value read.
//
///////////////////////////////////////////////////////////////////////////////
Byte
CPU::readMem(
    DWord dwAddress
)
{
    //  The number of entries in the read handler list.
    DWord dwNumEntries = m_readMemHandlerList.entries( );

    //  A convenience to the current handler.
    ReadHandler* pCurHandler;


#ifdef DEBUGGER
    Debugger::s_instance( ).checkRead( dwAddress );
#endif

    //  Plow through the handlers looking for the first one that matches
    //  the desired location.  
    for( DWord dwI = 0 ; dwI < dwNumEntries ; dwI += 1 )
    {
        //  Assign a convenience.
        pCurHandler = m_readMemHandlerList[ dwI ];

        //  Is this the one?
        if(
            ( dwAddress >= pCurHandler->getStart( ) ) &&
            ( dwAddress <= pCurHandler->getEnd( ) )
        )
        {
             //  Return the value from the handler.
             return(
                 ( *( pCurHandler->getHandler( ) ) )( dwAddress, pCurHandler )
             );
        }
    }

    //  No handler was found so just return the value from the address space
    //  of this CPU.
    return( m_pSpace->getByte( dwAddress ) );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: writeMemHandler
//
//  Description:
//
//      This function is called from a game to add a handler for writing
//      memory.  
//
//  Parameters:
//
//      iName (input)
//          The instance name of the handler.
//
//      dwStart (input)
//          The start of the range the handler is to be used for.
//
//      dwEnd (input)
//          The end of the range the handler is to be used for.
//
//      pfHandler (input)
//          A pointer to the function that will handle the write.
//
//      pData (input)
//          A client data pointer.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPU::writeMemHandler(
    const KString&          iName,
    DWord                   dwStart,
    DWord                   dwEnd,
    WriteHandler::WriteFunc pfHandler,
    void*                   pData
)
{
    //  Check the parameters.
    ASSERT( dwStart <= dwEnd );
    ASSERT( pfHandler != NULL );

    //  If the handler exists then replace the information.
    for( DWord dwI = 0 ; dwI < m_writeMemHandlerList.entries( ) ; dwI += 1 )
    {
        WriteHandler* pHandler = m_writeMemHandlerList[ dwI ];

        if( 
            ( pHandler->getStart( ) == dwStart ) &&
            ( pHandler->getEnd( )   == dwEnd   )
        )
        {
            pHandler->setHandler( pfHandler );
            pHandler->setData( pData );
            return;
        }
    }
       
    //  The handler doesn't exist so add it to the list.
    m_writeMemHandlerList.add( 
        new WriteHandler( iName, dwStart, dwEnd, pfHandler, pData )
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: writeMem
//
//  Description:
//
//      This function is called from a CPU core when a value is to be
//      written to somewhere in the address space.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to write to.
//
//      bValue (input)
//          The value to write.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPU::writeMem(
    DWord dwAddress,
    Byte  bValue
)
{
    //  The number of entries in the write handler list.
    DWord dwNumEntries = m_writeMemHandlerList.entries( );

    //  A convenience to the current handler.
    WriteHandler* pCurHandler;


#ifdef DEBUGGER
    Debugger::s_instance( ).checkWrite( dwAddress, bValue );
#endif

    //  Plow through the handlers looking for the first one that matches
    //  the desired location.  
    for( DWord dwI = 0 ; dwI < dwNumEntries ; dwI += 1 )
    {
        //  Assign a convenience.
        pCurHandler = m_writeMemHandlerList[ dwI ];

        //  Is this the one?
        if(
            ( dwAddress >= pCurHandler->getStart( ) ) &&
            ( dwAddress <= pCurHandler->getEnd( ) )
        )
        {
             //  Call the handler then return.
             ( *( pCurHandler->getHandler( ) ) )(
                     dwAddress, bValue, pCurHandler
             );
             return;
        }
    }

    //  No handler was found so just write value directly to the address 
    //  space of this CPU.
    m_pSpace->setByte( dwAddress, bValue );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: readPortHandler
//
//  Description:
//
//      This function is called from a game to add a port handler for
//      CPU reads to the list of handlers.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the handler.
//
//      dwStart (input)
//          The start of the range the handler is to be used for.
//
//      dwEnd (input)
//          The end of the range the handler is to be used for.
//
//      pfHandler (input)
//          A pointer to the function that will handle the read.
//
//      pData (input)
//          A client data pointer.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPU::readPortHandler(
    const KString&          iName,
    DWord                   dwStart,
    DWord                   dwEnd,
    ReadHandler::ReadFunc   pfHandler,
    void*                   pData
)
{
    //  Check the parameters.
    ASSERT( dwStart <= dwEnd );
    ASSERT( pfHandler != NULL );

    //  Add the handler to the list.
    m_readPortHandlerList.add( 
        new ReadHandler( iName, dwStart, dwEnd, pfHandler, pData )
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: readPort
//
//  Description:
//
//      This function can be called from a CPU core when a value is to be
//      read from a particular hardware.  This will be bypassed if
//      the core being used supports direct calls of handler functions.
//
//  Parameters:
//
//      dwPort (input)
//          The port to read from.
//
//  Returns:
//
//      The value read.
//
///////////////////////////////////////////////////////////////////////////////
Byte
CPU::readPort(
    DWord dwPort
)
{
    //  The number of entries in the read handler list.
    DWord dwNumEntries = m_readPortHandlerList.entries( );

    //  A convenience to the current handler.
    ReadHandler* pCurHandler;


    //  Plow through the handlers looking for the first one that matches
    //  the desired location.  
    for( DWord dwI = 0 ; dwI < dwNumEntries ; dwI += 1 )
    {
        //  Assign a convenience.
        pCurHandler = m_readPortHandlerList[ dwI ];

        //  Is this the one?
        if(
            ( dwPort >= pCurHandler->getStart( ) ) &&
            ( dwPort <= pCurHandler->getEnd( ) )
        )
        {
             //  Return the value from the handler.
             return(
                 ( *( pCurHandler->getHandler( ) ) )( dwPort, pCurHandler )
             );
        }
    }

    //  No handler was found so just return 0x00.
    return( 0x00 );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: writePortHandler
//
//  Description:
//
//      This function is called from a game to add a handler for writing
//      to a hardware port.  
//
//  Parameters:
//
//      iName (input)
//          The instance name of the handler.
//
//      dwStart (input)
//          The start of the range the handler is to be used for.
//
//      dwEnd (input)
//          The end of the range the handler is to be used for.
//
//      pfHandler (input)
//          A pointer to the function that will handle the write.
//
//      pData (input)
//          A client data pointer.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPU::writePortHandler(
    const KString&          iName,
    DWord                   dwStart,
    DWord                   dwEnd,
    WriteHandler::WriteFunc pfHandler,
    void*                   pData
)
{
    //  Check the parameters.
    ASSERT( dwStart <= dwEnd );
    ASSERT( pfHandler != NULL );

    //  Add the handler to the list.
    m_writePortHandlerList.add( 
        new WriteHandler( iName, dwStart, dwEnd, pfHandler, pData )
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: writePort
//
//  Description:
//
//      This function is called from a CPU core when a value is to be
//      written to a particular hardware port.
//
//  Parameters:
//
//      dwPort (input)
//          The port to write to.
//
//      bValue (input)
//          The value to write.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPU::writePort(
    DWord dwPort,
    Byte  bValue
)
{
    //  The number of entries in the write handler list.
    DWord dwNumEntries = m_writePortHandlerList.entries( );

    //  A convenience to the current handler.
    WriteHandler* pCurHandler;


    //  Plow through the handlers looking for the first one that matches
    //  the desired location.  
    for( DWord dwI = 0 ; dwI < dwNumEntries ; dwI += 1 )
    {
        //  Assign a convenience.
        pCurHandler = m_writePortHandlerList[ dwI ];

        //  Is this the one?
        if(
            ( dwPort >= pCurHandler->getStart( ) ) &&
            ( dwPort <= pCurHandler->getEnd( ) )
        )
        {
             //  Call the handler then return.
             ( *( pCurHandler->getHandler( ) ) )(
                     dwPort, bValue, pCurHandler
             );
             return;
        }
    }

    //  No handler was found so don't do anything.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: reset
//
//  Description:
//
//      This function is called to reset the CPU.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPU::reset(
)
{
    //  Reset the appropriate member data.
    m_nCyclesTotal   = 0;
    m_nCyclesThisRun = 0; 
    m_nCyclesCarry   = 0; 
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: activate
//
//  Description:
//
//      This function is called to activate a certain CPU.  The minimum
//      behaviour (provided by this base member) will assign the memory
//      regions and CPU pointer.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPU::activate(
)
{
    //  If this CPU is already active then nothing to do.
    if( this == sm_pCPU )
    {
        return;
    }

    //  If there is currently a different active CPU then deactivate it.
    if( sm_pCPU != NULL )
    {
        sm_pCPU->deactivate( );
    }

    //  Set the global pointers for use by the emulation core.
    sm_pCPU    = this;
    sm_pSpace  = m_pSpace;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: deactivate
//
//  Description:
//
//      This function is called to deactivate a certain CPU.  This resets the
//      global CPU pointers.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPU::deactivate(
)
{
    //  Reset the global pointers for use by the emulation core.
    sm_pCPU   = NULL;
    sm_pSpace = NULL;
}
