///////////////////////////////////////////////////////////////////////////////
//
//  File:       cpu.h
//
//  Class:      CPU - Abstract
//  
//  Hierarchy:  CPU - RepBase 
//
//  Author:     Kevin Brisley
//
//  Description:
//
//      This class serves as a base class for each of the CPU's supported
//      by Replay.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////
#ifndef _cpu_h_
#define _cpu_h_

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>

//  Application Headers.
#include "reptypes.h"
#include "repbase.h"
#include "kstring.h"
#include "cpurh.h"
#include "cpuwh.h"
#include "kplist.h"

//  Forward class declarations.
class Game;
class AddrSpace;
class CPU;
class Input;


///////////////////////////////////////////////////////////////////////////////
//  CPU Class Definition
///////////////////////////////////////////////////////////////////////////////
class CPU : public RepBase
{
    public:

        //  The following enum is list of CPU types.
        typedef enum
        {
            CPU_6502,
            CPU_6809,
            CPU_Z80, 
            CPU_68000,

            NUM_CPU_TYPES
        }
        CPUType;

        //  The possible CPU types.
        typedef enum
        {
            GAME,
            SOUND
        }
        CPUPurpose;

        //  All possible registers.
        enum
        {
            REG_ICOUNT = 0,
            REG_PC,

            REG_CPU_CUSTOM
        };
      
        //  Standard interrupt values.
        typedef enum
        {
            INT_NONE = -1,
            INT_NMI  = -2,
            INT_IRQ  = -3,
            INT_FIRQ = -4,
            INT_QUIT = -5
        }
        IntValue;

        //  The format of various handler functions.
        typedef int32 ( *IntFunc  )( CPU* , Game* );


        //  Deletion.
        virtual ~CPU ( );

        //  Identification.
        virtual const KString& getType     ( ) const = 0;
        virtual DWord          getAddrSize ( ) const = 0;

        //  Persistence.
        virtual Byte save ( AppFile* pSaveFile );
        virtual Byte load ( AppFile* pLoadFile );

        //  Setup interface.
        void  setClockSpeed    ( int32 nClockSpeed );
        int32 getClockSpeed    ( ) const;
        void  setIntsPerFrame  ( int32 nIntsPerFrame );
        int32 getIntsPerFrame  ( ) const;
        int32 getIntsThisFrame ( ) const;

        //  Handlers.
        void  interruptHandler ( IntFunc pfInterrupt );
        void  readMemHandler   ( 
            const KString&          iName,
            DWord                   dwStart,
            DWord                   dwEnd,
            ReadHandler::ReadFunc   pfHandler,
            void*                   pData
        );
        void  writeMemHandler  ( 
            const KString&          iName,
            DWord                   dwStart,
            DWord                   dwEnd,
            WriteHandler::WriteFunc pfHandler,
            void*                   pData
        );
        void  readPortHandler   ( 
            const KString&          iName,
            DWord                   dwStart,
            DWord                   dwEnd,
            ReadHandler::ReadFunc   pfHandler,
            void*                   pData
        );
        void  writePortHandler  ( 
            const KString&          iName,
            DWord                   dwStart,
            DWord                   dwEnd,
            WriteHandler::WriteFunc pfHandler,
            void*                   pData
        );

        //  Execution Interface.
        virtual void  reset        ( );
        virtual void  activate     ( );
        virtual void  deactivate   ( );
        virtual void  run          ( const int32 nCycles ) = 0;
        void          interrupt    ( );
        virtual void  interrupt    ( const int32 nType ) = 0;
        void          enable       ( const Byte bEnable );
        Byte          isEnabled    ( ) const;
        void          enableInts   ( const Byte bEnable );
        Byte          intsEnabled  ( ) const;
        void          setIntVector ( const int32 nIntVector );
        int32         getIntVector ( ) const;

        //  Core access.
        virtual Byte  readMem            ( DWord dwAddress );
        virtual void  writeMem           ( DWord dwAddress, Byte bValue );
        virtual Byte  readPort           ( DWord dwPort );
        virtual void  writePort          ( DWord dwPort, Byte bValue );
        virtual int32 getReg             ( const int32 nReg ) = 0;
        virtual void  setReg             ( 
            const int32 nReg, const int32 nValue 
        ) = 0;

        //  Cycle access.
        void          setCyclesLeft      ( const int32 nCycles );
        int32         getCyclesLeft      ( );
        int32         getCyclesExecuted  ( );
        int32         getCyclesTotal     ( );

#ifdef DEBUGGER
        virtual const char* dbgDisassemble ( DWord& rdwAddress ) = 0;
        virtual const char* dbgRegister    ( const DWord dwNum ) = 0;
#endif


        //  Static member data available for convenience & speed.
        static CPU*       sm_pCPU;
        static AddrSpace* sm_pSpace;

    protected:

        //  Creation is protected since a build method should be used.
        CPU ( 
            const KString&   iName, 
            Game*            pGame, 
            AddrSpace*       pSpace, 
            const CPUPurpose ePurpose = GAME 
        );

        //  Object initialization to be used by the build method.
        virtual void init ( );

        //  Member data.
        Game*                  m_pGame;
        AddrSpace*             m_pSpace;
        CPUPurpose             m_ePurpose;
        KPtrList<ReadHandler>  m_readMemHandlerList;
        KPtrList<WriteHandler> m_writeMemHandlerList;
        KPtrList<ReadHandler>  m_readPortHandlerList;
        KPtrList<WriteHandler> m_writePortHandlerList;
        int32                  m_nCyclesTotal;
        int32                  m_nCyclesThisRun; 
        int32                  m_nCyclesCarry; 
        int32                  m_nClockSpeed;
        int32                  m_nIntsPerFrame;
        int32                  m_nIntsThisFrame;
        Byte                   m_bEnable;
        Byte                   m_bDisable;
        Byte                   m_bIEnable;
        IntFunc                m_pfInterrupt;
        int32                  m_nIntVector;

    private:

};

///////////////////////////////////////////////////////////////////////////////
//  CPU Inline functions.
///////////////////////////////////////////////////////////////////////////////
inline
void
CPU::setClockSpeed(
    int32 nClockSpeed
)
{
    m_nClockSpeed = nClockSpeed;
}

inline
int32
CPU::getClockSpeed(
) const
{
    return( m_nClockSpeed );
}

inline
void
CPU::setIntsPerFrame(
    int32 nIntsPerFrame
)
{
    m_nIntsPerFrame  = nIntsPerFrame;
}

inline
int32
CPU::getIntsPerFrame(
) const
{
    return( m_nIntsPerFrame );
}

inline
int32
CPU::getIntsThisFrame(
) const
{
    return( m_nIntsThisFrame );
}

inline
void
CPU::interruptHandler(
    IntFunc pfInterrupt
)
{
    m_pfInterrupt = pfInterrupt;
}

inline
void
CPU::enable(
    const Byte bEnable
)
{
    if( !m_bDisable )
    {
        m_bEnable = bEnable;
    }
}

inline
Byte
CPU::isEnabled(
) const
{
    return( m_bEnable );
}

inline
void
CPU::enableInts(
    const Byte bEnable
)
{
    m_bIEnable = bEnable;
}

inline
Byte
CPU::intsEnabled(
) const
{
    return( m_bIEnable );
}

inline
void
CPU::setIntVector(
    const int32 nIntVector
)
{
    m_nIntVector = nIntVector;
}

inline
int32
CPU::getIntVector(
) const
{
    return( m_nIntVector );
}

inline
void
CPU::setCyclesLeft(
    const int32 nCycles
)
{
    setReg( REG_ICOUNT, nCycles );
}

inline
int32
CPU::getCyclesLeft(
)
{
    return( getReg( REG_ICOUNT ) );
}

inline
int32
CPU::getCyclesExecuted(
)
{
    return( m_nCyclesThisRun - getCyclesLeft( ) );
}

inline
int32
CPU::getCyclesTotal(
)
{
    return( m_nCyclesTotal + getCyclesExecuted( ) );
}

inline
void
CPU::interrupt(
)
{
    ASSERT( m_pfInterrupt != NULL );

    interrupt( ( *m_pfInterrupt )( this, m_pGame ) );
    m_nIntsThisFrame = ( m_nIntsThisFrame + 1 ) % m_nIntsPerFrame;
}

#endif
