///////////////////////////////////////////////////////////////////////////////
//
//  File:    brkpt.cpp
//
//  Class:   BreakPoint
//
//  Author:  Kevin Brisley
//
//  Description:
//
//
//      This class represents a debugger breakpoint.  A breakpoint is a
//      condition of a CPU for which the debugger should be invoked.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

#ifdef DEBUGGER

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "brkpt.h"


///////////////////////////////////////////////////////////////////////////////
//  Static Data Member Initialization.
///////////////////////////////////////////////////////////////////////////////
DWord BreakPoint::sm_dwID = 0;


///////////////////////////////////////////////////////////////////////////////
//
//  Function: BreakPoint
//
//  Description:
//
//      This is the main constructor for the breakpoint object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//      eType (input)
//          The type of breakpoint.
//
//      pCPU (input)
//          The CPU that the breakpoint is associated with.
//
//      dwStartAddr (input)
//          The start of the range that the breakpoint is valid for.
//
//      dwEndAddr (input)
//          The end of the range that the breakpoint is valid for.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
BreakPoint::BreakPoint(
    const KString& iName,
    const Type     eType,
    CPU*           pCPU,
    const DWord    dwStartAddr,
    const DWord    dwEndAddr
)
:
    RepBase              ( iName ),
    m_eType              ( eType ),
    m_pCPU               ( pCPU ),
    m_dwStartAddr        ( dwStartAddr ),
    m_dwEndAddr          ( dwEndAddr ),
    m_bEnabled           ( TRUE )
{
    //  Make sure a CPU was specified.
    ASSERT( m_pCPU != NULL );

    //  A type must be specified.
    ASSERT( m_eType != NONE );

    //  Assign the next ID to the break point.  We make no check that
    //  the ID is not already in use but this shouldn't be a problem
    //  since we're allowing 1000 ID's.  If it becomes a problem, you're
    //  doing way too much debugging.
    m_dwID = sm_dwID;
    sm_dwID = ( sm_dwID + 1 ) % 1000;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~BreakPoint
//
//  Description:
//
//      This is the destructor for a breakpoint.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
BreakPoint::~BreakPoint(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the breakpoint.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
BreakPoint::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "BreakPoint" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getTypeString
//
//  Description:
//
//      This member is used to return the name of the type of this breakpoint.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the breakpoint type.
//
///////////////////////////////////////////////////////////////////////////////
char*
BreakPoint::getTypeString(
) const
{
    switch( m_eType )
    {
        case PC:    return( "P" );                                    break;
        case READ:  return( "R" );                                    break;
        case WRITE: return( "W" );                                    break;
        default: fatalError( "Invalid breakpoint type %d", m_eType ); break;
    }

    //  Should never arrive here.
    return( "???" );
}


     
///////////////////////////////////////////////////////////////////////////////
//
//  Function: match
//
//  Description:
//
//      This member is called to see if this breakpoint matches the
//      parameter specified.
//
//  Parameters:
//
//      eType (input)
//          The type of breakpoint.
//
//      pCPU (input)
//          The CPU that the condition applies to.
//
//      dwAddr (input)
//          The affected memory address.
//
//  Returns:
//
//      TRUE  if the breakpoint matches the parameters.
//      FALSE otherwise.
//
///////////////////////////////////////////////////////////////////////////////
Byte
BreakPoint::match( 
    const Type  eType,
    const CPU*  pCPU,
    const DWord dwAddr
) const
{
    //  If everything is cool then inform the client that the breakpoint 
    //  matches.
    if( 
        ( pCPU == m_pCPU ) &&
        ( eType == m_eType ) &&
        ( dwAddr >= m_dwStartAddr ) &&
        ( dwAddr <= m_dwEndAddr )
    )
    {
        return( TRUE );
    }
   
    //  The breakpoint doesn't match.
    return( FALSE );
}

#endif
