///////////////////////////////////////////////////////////////////////////////
//
//  File:       brkpt.h
//
//  Class:      BreakPoint - Concrete
//  
//  Hierarchy:  BreakPoint - RepBase 
//
//  Author:     Kevin Brisley
//
//  Description:
//
//      This class represents a debugger breakpoint.  A breakpoint is a
//      condition of a CPU for which the debugger should be invoked.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

#ifdef DEBUGGER

#ifndef _brkpt_h_
#define _brkpt_h_

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>

//  Application Headers.
#include "reptypes.h"
#include "repbase.h"
#include "kstring.h"

//  Forward class declarations.
class CPU;


///////////////////////////////////////////////////////////////////////////////
//  BreakPoint Class Definition
///////////////////////////////////////////////////////////////////////////////
class BreakPoint : public RepBase
{
    public:

        //  The possible checks.
        typedef enum
        {
            NONE,

            PC,
            READ,
            WRITE
        }
        Type;
 
        //  Creation/Deletion.
        BreakPoint ( 
            const KString& iName, 
            const Type     eType,
            CPU*           pCPU,
            const DWord    dwStartAddr,
            const DWord    dwEndAddr
        );
        virtual ~BreakPoint ( );

        //  Base overrides.
        virtual const KString& getClassName ( ) const;

        //  Access.
        CPU*  getCPU        ( ) const;
        Type  getType       ( ) const;
        char* getTypeString ( ) const;
        DWord getStart      ( ) const;
        DWord getEnd        ( ) const;
        DWord getID         ( ) const;
        Byte  isEnabled     ( ) const;

        //  Interface.
        void enable ( Byte bEnable );
        Byte match  ( 
            const Type eType, const CPU* pCPU, const DWord dwAddr 
        ) const;

    protected:

        //  Member Data.
        Type  m_eType; 
        CPU*  m_pCPU;
        DWord m_dwStartAddr;
        DWord m_dwEndAddr;
        Byte  m_bEnabled;
        DWord m_dwID;

        //  Static Member Data.
        static DWord sm_dwID;

    private:

};

///////////////////////////////////////////////////////////////////////////////
//  BreakPoint Inline Functions.
///////////////////////////////////////////////////////////////////////////////
inline
CPU*
BreakPoint::getCPU(
) const
{
    ASSERT( m_pCPU != NULL );
    return( m_pCPU );
}

inline
BreakPoint::Type
BreakPoint::getType(
) const
{
    return( m_eType );
}

inline
DWord
BreakPoint::getStart(
) const
{
    return( m_dwStartAddr );
}

inline
DWord
BreakPoint::getEnd(
) const
{
    return( m_dwEndAddr );
}

inline
DWord
BreakPoint::getID(
) const
{
    return( m_dwID );
}

inline
void
BreakPoint::enable(
    Byte bEnable
)
{
    m_bEnabled = bEnable;
}

inline
Byte
BreakPoint::isEnabled(
) const
{
    return( m_bEnabled );
}


#endif

#endif
