///////////////////////////////////////////////////////////////////////////////
//
//  File:    dbgbps.cpp
//
//  Class:   DebugBreakPoints
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class represents a window in the Replay debugger that is
//      responsible for displaying all of the currently specified 
//      breakpoints.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

#ifdef DEBUGGER

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdio.h>

//  Application Headers.
#include "dbgbps.h"
#include "keyb.h"
#include "clip.h"
#include "brkpt.h"
#include "bitmap.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: DebugBreakPoints
//
//  Description:
//
//      This is the main constructor for the debug break point window object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//      pDebugger (input)
//          The debugger that the window belongs to.
//
//      dwX (input)
//          The X position of the window.
//
//      dwY (input)
//          The Y position of the window.
//
//      dwWidth (input)
//          The width of the window.
//
//      dwHeight (input)
//          The height of the window.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
DebugBreakPoints::DebugBreakPoints(
    const KString& iName,
    Debugger*      pDebugger,
    const DWord    dwX,
    const DWord    dwY,
    const DWord    dwWidth,
    const DWord    dwHeight
)
:
    DebugWindow ( iName, pDebugger, dwX, dwY, dwWidth, dwHeight )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~DebugBreakPoints
//
//  Description:
//
//      This is the destructor for the debugger break points window object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
DebugBreakPoints::~DebugBreakPoints(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the debugger break point window.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
DebugBreakPoints::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "DebugBreakPoints" );

    return( className );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: draw
//
//  Description:
//
//      This member is called when the window is to draw itself.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
DebugBreakPoints::draw(
)
{
    //  A temporary buffer.
    char strTemp[ 80 ];

    
    DebugWindow::draw( );


    //  Draw until we run out of breakpoints or room.
    for( 
        DWord dwI = m_pDebugger->m_dwBP ;
        dwI < m_pDebugger->m_bpList.entries( ) ; 
        dwI += 1 
    )
    {
        //  A convenience to the current breakpoint.
        BreakPoint* pBP = m_pDebugger->m_bpList[ dwI ];

        //  If we're out of room then quit.
        if( m_pCurClipping->m_nMinY > m_pCurClipping->m_nMaxY )
        {
            break;
        }

        //  If the CPU doesn't match then continue on to the next breakpoint.
        if( m_pDebugger->m_pCPU != pBP->getCPU( ) )
        {
            continue;
        }

        //  OK, draw the current breakpoint.
        sprintf(
            strTemp,
            "%03ld-%1s $%06lx-$%06lx (%s)       ",
            pBP->getID( ),
            pBP->getTypeString( ),
            pBP->getStart( ),
            pBP->getEnd( ),
            pBP->isEnabled( ) ? "E" : "D"
        );
        writeLine( strTemp );
    }

    //  Clear out the remainder of the window to get rid of any junk.
    m_pBitmap->clearArea( m_pCurClipping, m_pDebugger->sm_bColourBG );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: execute
//
//  Description:
//
//      This member is called when the window is to execute.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      One of the "leave" status codes.
//
///////////////////////////////////////////////////////////////////////////////
Debugger::Leave
DebugBreakPoints::execute(
)
{
    //  Allow basic movements.
    if( m_pKeyboard->switchOnWait( Keyboard::KEY__UP ) )
    {
        if( m_pDebugger->m_dwBP > 0 )
        {
            m_pDebugger->m_dwBP -= 1;
            m_pDebugger->m_bRedrawScreen = TRUE;
        }
    }
    else
    if( m_pKeyboard->switchOnWait( Keyboard::KEY__DOWN ) )
    {
        if( m_pDebugger->m_dwBP < m_pDebugger->m_bpList.entries( ) - 1 )
        {
            m_pDebugger->m_dwBP += 1;
            m_pDebugger->m_bRedrawScreen = TRUE;
        }
    }
    else
    if( m_pKeyboard->switchOnWait( Keyboard::KEY__PGUP ) )
    {
        if( m_pDebugger->m_dwBP >= m_dwPageLength )
        {
            m_pDebugger->m_dwBP -= m_dwPageLength;
        }
        else
        {
            m_pDebugger->m_dwBP = 0;
        }
        m_pDebugger->m_bRedrawScreen = TRUE;
    }
    else
    if( m_pKeyboard->switchOnWait( Keyboard::KEY__PGDN ) )
    {
        if( 
            ( m_pDebugger->m_dwBP + m_dwPageLength ) < 
            m_pDebugger->m_bpList.entries( )
        )
        {
            m_pDebugger->m_dwBP += m_dwPageLength;
        }
        else
        {
            m_pDebugger->m_dwBP = m_pDebugger->m_bpList.entries( ) - 1;
        }
        m_pDebugger->m_bRedrawScreen = TRUE;
    }
        
    return( Debugger::LEAVE_NO );
}



#endif
