///////////////////////////////////////////////////////////////////////////////
//
//  File:    dbgcmd.cpp
//
//  Class:   DebugCommand
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class represents a command line input window for the Replay+
//      debugger.  This window is where the user is able to type
//      instructions to the debugger.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

#ifdef DEBUGGER

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdio.h>
#include <string.h>

//  Application Headers.
#include "dbgcmd.h"
#include "game.h"
#include "cpu.h"
#include "brkpt.h"
#include "appfile.h"


///////////////////////////////////////////////////////////////////////////////
//  Static Member Data Initialization.
///////////////////////////////////////////////////////////////////////////////

//  This table lists the keys that are allowed in the command window (not
//  including editing keys).
Keyboard::KeyboardKey DebugCommand::sm_aeAllowed[ ] = 
{
    Keyboard::KEY__A, Keyboard::KEY__B, Keyboard::KEY__C, Keyboard::KEY__D,
    Keyboard::KEY__E, Keyboard::KEY__F, Keyboard::KEY__G, Keyboard::KEY__H,
    Keyboard::KEY__I, Keyboard::KEY__J, Keyboard::KEY__K, Keyboard::KEY__L,
    Keyboard::KEY__M, Keyboard::KEY__N, Keyboard::KEY__O, Keyboard::KEY__P,
    Keyboard::KEY__Q, Keyboard::KEY__R, Keyboard::KEY__S, Keyboard::KEY__T,
    Keyboard::KEY__U, Keyboard::KEY__V, Keyboard::KEY__W, Keyboard::KEY__X,
    Keyboard::KEY__Y, Keyboard::KEY__Z, Keyboard::KEY__0, Keyboard::KEY__1,
    Keyboard::KEY__2, Keyboard::KEY__3, Keyboard::KEY__4, Keyboard::KEY__5,
    Keyboard::KEY__6, Keyboard::KEY__7, Keyboard::KEY__8, Keyboard::KEY__9,
    Keyboard::KEY__6, Keyboard::KEY__7, Keyboard::KEY__8, Keyboard::KEY__9
};
DWord DebugCommand::sm_dwAllowed = 
    sizeof( sm_aeAllowed ) / sizeof( sm_aeAllowed[ 0 ] );

//  Indicates no parameter.
const DWord DebugCommand::sm_dwNone = ( DWord )-1;



///////////////////////////////////////////////////////////////////////////////
//
//  Function: DebugCommand
//
//  Description:
//
//      This is the main constructor for the debug command window object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//      pDebugger (input)
//          The debugger that the window belongs to.
//
//      dwX (input)
//          The X position of the window.
//
//      dwY (input)
//          The Y position of the window.
//
//      dwWidth (input)
//          The width of the window.
//
//      dwHeight (input)
//          The height of the window.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
DebugCommand::DebugCommand(
    const KString& iName,
    Debugger*      pDebugger,
    const DWord    dwX,
    const DWord    dwY,
    const DWord    dwWidth,
    const DWord    dwHeight
)
:
    DebugWindow         ( iName, pDebugger, dwX, dwY, dwWidth, dwHeight ),
    m_dwSearchStart     ( 0x0000 ),
    m_dwNumSearchValues ( 0 )
{
    //  Initialize the command line.
    strcpy( m_strCmd, "" );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~DebugCommand
//
//  Description:
//
//      This is the destructor for the debugger command window object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
DebugCommand::~DebugCommand(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the debugger command window.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
DebugCommand::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "DebugCommand" );

    return( className );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: draw
//
//  Description:
//
//      This member is called when the window is to draw itself.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
DebugCommand::draw(
)
{
    //  A buffer to contain the command line.
    char strLine[ 80 ];

    DebugWindow::draw( );

    //  Concatenate a cursor.
    strcat( m_strCmd, "<" );

    //  Draw the command as it currently stands.
    sprintf( strLine, " ->%-75s", m_strCmd );
    writeLine( strLine );

    //  Remove the cursor.
    m_strCmd[ strlen( m_strCmd ) - 1 ] = '\0';
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: execute
//
//  Description:
//
//      This member is called when the window is to execute.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      One of the "leave" status codes.
//
///////////////////////////////////////////////////////////////////////////////
Debugger::Leave
DebugCommand::execute(
)
{
    //  If there is room in the buffer then check for any allowed keys.
    if( strlen( m_strCmd ) < 75 )
    {
        for( DWord dwI = 0 ; dwI < sm_dwAllowed ; dwI += 1 )
        {
            if( m_pKeyboard->switchOn( sm_aeAllowed[ dwI ] ) )
            {
                m_pKeyboard->waitUntilOff( sm_aeAllowed[ dwI ] );
  
                //  Add it to the buffer.
                strcat( 
                    m_strCmd, m_pKeyboard->switchName( sm_aeAllowed[ dwI ] ) 
                );
                m_pDebugger->m_bRedrawWindow = TRUE;
                break;
            }
        }
    }
    
    //  We handle "space" separately since it's name is "SPACE" instead of " ".
    if( m_pKeyboard->switchOn( Keyboard::KEY__SPACE ) )
    {
        m_pKeyboard->waitUntilOff( Keyboard::KEY__SPACE );

        strcat( m_strCmd, " " );
        m_pDebugger->m_bRedrawWindow = TRUE;
    }

    //  If the backspace key has been pressed then delete a character from
    //  the command.
    if( 
        ( m_pKeyboard->switchOnWait( Keyboard::KEY__BS ) ) ||
        ( m_pKeyboard->switchOnWait( Keyboard::KEY__DEL ) ) ||
        ( m_pKeyboard->switchOnWait( Keyboard::KEY__LEFT ) )
    )
    {
        DWord dwLength = strlen( m_strCmd );
        if( dwLength > 0 )
        {
            m_strCmd[ dwLength - 1 ] = '\0';
            m_pDebugger->m_bRedrawWindow = TRUE;
        }
    }
                
    //  If the command has been confirmed then execute it.
    if( m_pKeyboard->switchOn( Keyboard::KEY__ENTER ) )
    {
        m_pKeyboard->waitUntilOff( Keyboard::KEY__ENTER );

        return( executeCmd( ) );
    }

    return( Debugger::LEAVE_NO );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: executeCmd
//
//  Description:
//
//      This member is called when the user has confirmed the current command.
//      It is responsible for parsing and executing the command.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      One of the "leave" status codes.
//
///////////////////////////////////////////////////////////////////////////////
Debugger::Leave
DebugCommand::executeCmd(
)
{
    //  The return code.
    Debugger::Leave eLeave = Debugger::LEAVE_NO;


    //  Execute based on the command.
    switch( getCommand( ) )
    {
        //  Continue execution of the program.
        case CMD_CONTINUE:   eLeave = cmdContinue( );                     break;
        case CMD_NEXT:       eLeave = cmdNext( );                         break;
        case CMD_MEMORY:     eLeave = cmdMemory( );                       break;
        case CMD_SOURCE:     eLeave = cmdSource( );                       break;
        case CMD_PC_BP:      eLeave = cmdBreakPoint( BreakPoint::PC );    break;
        case CMD_READ_BP:    eLeave = cmdBreakPoint( BreakPoint::READ );  break;
        case CMD_WRITE_BP:   eLeave = cmdBreakPoint( BreakPoint::WRITE ); break;
        case CMD_DISABLE_BP: eLeave = cmdEnableBreakPoint( FALSE );       break;
        case CMD_ENABLE_BP:  eLeave = cmdEnableBreakPoint( TRUE );        break;
        case CMD_DELETE_BP:  eLeave = cmdDeleteBreakPoint( );             break;
        case CMD_ASSIGN_MEM: eLeave = cmdAssignMemory( );                 break;
        case CMD_SEARCH_MEM: eLeave = cmdSearchMemory( );                 break;
        case CMD_CPU:        eLeave = cmdChangeCPU( );                    break;
        case CMD_TRACE:      eLeave = cmdTrace( );                        break;

        default: strcpy( m_pDebugger->m_strStatus, "Invalid command." );  break;
    }

    //  Toast the command.
    strcpy( m_strCmd, "" );

    //  Display a different source region.
    m_pDebugger->m_bRedrawScreen = TRUE;

    return( eLeave );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: cmdContinue
//
//  Description:
//
//      This member is called to process the continue command.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      One of the "leave" status codes.
//
///////////////////////////////////////////////////////////////////////////////
Debugger::Leave
DebugCommand::cmdContinue(
)
{
    //  Simply clear the status and indicate that we want to leave the
    //  debugger for a while.
    strcpy( m_pDebugger->m_strStatus, "" );
    return( Debugger::LEAVE_LONG );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: cmdNext
//
//  Description:
//
//      This member is called to process the next command.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      One of the "leave" status codes.
//
///////////////////////////////////////////////////////////////////////////////
Debugger::Leave
DebugCommand::cmdNext(
)
{
    //  Simply clear the status and indicate that we want to leave the
    //  debugger just for a single instruction and then return.
    strcpy( m_pDebugger->m_strStatus, "" );
    return( Debugger::LEAVE_SHORT );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: cmdMemory
//
//  Description:
//
//      This member is called to process the memory command.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      One of the "leave" status codes.
//
///////////////////////////////////////////////////////////////////////////////
Debugger::Leave
DebugCommand::cmdMemory(
)
{
    //  The first parameter is the address to display.
    m_pDebugger->m_dwMemAddr = 
        getParamHex( 0, 0, m_pDebugger->m_pCPU->getAddrSize( ) - 1 );

    //  If no memory address was specified then display memory at the
    //  current PC.
    if( m_pDebugger->m_dwMemAddr == sm_dwNone )
    {
        m_pDebugger->m_dwMemAddr = m_pDebugger->m_pCPU->getReg( CPU::REG_PC );
    } 

    return( Debugger::LEAVE_NO );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: cmdSource
//
//  Description:
//
//      This member is called to process the continue command.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      One of the "leave" status codes.
//
///////////////////////////////////////////////////////////////////////////////
Debugger::Leave
DebugCommand::cmdSource(
)
{
    //  The first parameter is the address to display source at.
    m_pDebugger->m_dwSrcAddr =
        getParamHex( 0, 0, m_pDebugger->m_pCPU->getAddrSize( ) - 1 );

    //  If no address is specified then display source at the PC.
    if( m_pDebugger->m_dwSrcAddr == sm_dwNone )
    {
        m_pDebugger->m_dwSrcAddr = m_pDebugger->m_pCPU->getReg( CPU::REG_PC );
    } 

    return( Debugger::LEAVE_NO );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: cmdBreakPoint
//
//  Description:
//
//      This member is called to process the breakpoint commands.
//
//  Parameters:
//
//      eType (input)
//          The type of breakpoint to create.
//
//  Returns:
//
//      One of the "leave" status codes.
//
///////////////////////////////////////////////////////////////////////////////
Debugger::Leave
DebugCommand::cmdBreakPoint(
    BreakPoint::Type eType
)
{
    //  The range the breakpoint is applicable to.
    DWord dwStart = 
                getParamHex( 0, 0, m_pDebugger->m_pCPU->getAddrSize( ) - 1 );
    DWord dwEnd   =
                getParamHex( 1, 0, m_pDebugger->m_pCPU->getAddrSize( ) - 1 );
           
    //  If the end of the range is not specified then the range is only
    //  one address long.
    if( dwEnd == sm_dwNone )
    {
        dwEnd = dwStart;
    }

    //  If no range was specified then we have a problem.
    if( ( dwStart == sm_dwNone ) || ( dwEnd == sm_dwNone ) )
    {
        sprintf( 
            m_pDebugger->m_strStatus, 
            "Usage: B <$000000-$%06lx> [<$000000-$%06lx>]",
            m_pDebugger->m_pCPU->getAddrSize( ) - 1,
            m_pDebugger->m_pCPU->getAddrSize( ) - 1 
        );
    }
    else
    {
        //  Create the new breakpoint.
        m_pDebugger->m_bpList.add( 
            new BreakPoint( 
                "BP", eType, m_pDebugger->m_pCPU, dwStart, dwEnd
            )
        ); 
        strcpy( m_pDebugger->m_strStatus, "PC BreakPoint added." );
    }

    return( Debugger::LEAVE_NO );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: cmdEnableBreakPoint
//
//  Description:
//
//      This member is called to enable or disable a breakpoint.
//
//  Parameters:
//
//      bEnable (input)
//          Indicates whether to enable or disable the breakpoint.
//
//  Returns:
//
//      One of the "leave" status codes.
//
///////////////////////////////////////////////////////////////////////////////
Debugger::Leave
DebugCommand::cmdEnableBreakPoint(
    Byte bEnable
)
{
    //  The specified breakpoint.
    DWord dwBP = getParamDec( 0 );

    //  Indicates the number of breakpoints disabled.
    DWord dwDisabled = 0;

    //  Look through the list of breakpoints.
    for( DWord dwI = 0 ; dwI < m_pDebugger->m_bpList.entries( ) ; dwI += 1 )
    {
        //  If the current breakpoint is not for this CPU then 
        //  continue on.
        if( m_pDebugger->m_bpList[ dwI ]->getCPU( ) != m_pDebugger->m_pCPU )
        {
            continue;
        }

        //  If there was no specific breakpoint specified or if the
        //  current breakpoint matches the one specified then mdoify it.
        if( 
            ( dwBP == sm_dwNone ) ||
            ( m_pDebugger->m_bpList[ dwI ]->getID( ) == dwBP )
        )
        {
            m_pDebugger->m_bpList[ dwI ]->enable( bEnable );
            dwDisabled += 1;
        }
        
        sprintf( 
            m_pDebugger->m_strStatus, "%ld Breakpoints disabled.", dwDisabled
        );
    }

    return( Debugger::LEAVE_NO );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: deleteBreakPoint
//
//  Description:
//
//      This member is called to delete a breakpoint.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      One of the "leave" status codes.
//
///////////////////////////////////////////////////////////////////////////////
Debugger::Leave
DebugCommand::cmdDeleteBreakPoint(
)
{
    //  The specified breakpoint.
    DWord dwBP = getParamDec( 0 );

    //  Indicates the number of breakpoints deleted.
    DWord dwDeleted = 0;

    //  Look through the list of breakpoints.
    for( DWord dwI = 0 ; dwI < m_pDebugger->m_bpList.entries( ) ; )
    {
        //  If the current breakpoint is not for this CPU then 
        //  continue on.
        if( m_pDebugger->m_bpList[ dwI ]->getCPU( ) != m_pDebugger->m_pCPU
        )
        {
            dwI += 1;
            continue;
        }

        //  If there was no specific breakpoint specified or if the
        //  current breakpoint matches the one specified then delete it.
        if( 
            ( dwBP == sm_dwNone ) ||
            ( m_pDebugger->m_bpList[ dwI ]->getID( ) == dwBP )
        )
        {
            delete m_pDebugger->m_bpList[ dwI ];
            m_pDebugger->m_bpList.remove( m_pDebugger->m_bpList[ dwI ] );
            dwDeleted += 1;
        }
        else
        {
            dwI += 1;
        }
    }

    sprintf( m_pDebugger->m_strStatus, "%ld Breakpoints deleted.", dwDeleted );

    return( Debugger::LEAVE_NO );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: cmdAssignMemory
//
//  Description:
//
//      This member is called to assign a value to a memory location.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      One of the "leave" status codes.
//
///////////////////////////////////////////////////////////////////////////////
Debugger::Leave
DebugCommand::cmdAssignMemory(
)
{
    //  The address and the value involved in the assignment.
    DWord dwAddr = getParamHex( 0, 0, m_pDebugger->m_pCPU->getAddrSize( ) - 1 );
    DWord dwValue = getParamHex( 1, 0, 0xff );

    //  If either the address or value is omitted then report an error.
    if( ( dwAddr == sm_dwNone ) || ( dwValue == sm_dwNone ) )
    {
        sprintf( 
            m_pDebugger->m_strStatus, 
            "Usage: A <$000000-$%06lx> <$00-$ff>" ,
            m_pDebugger->m_pCPU->getAddrSize( ) - 1
        );
    }
    else
    {
        //  Assign the memory using the CPU's memory handlers.
        m_pDebugger->m_pCPU->writeMem( dwAddr, dwValue );
        strcpy( m_pDebugger->m_strStatus, "Memory changed." );
    }

    return( Debugger::LEAVE_NO );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: cmdSearchMemory
//
//  Description:
//
//      This member is called to search memory for the specified pattern.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Debugger::Leave
DebugCommand::cmdSearchMemory(
)
{
    //  Indices used for iterations.
    DWord dwI;
   
    //  A parameter.
    DWord dwParam;


    //  If there are search values specified then record the values.
    if( getParamHex( 0, 0, 0xff ) != sm_dwNone )
    {
        m_dwNumSearchValues = 0;
        m_dwSearchStart     = 0x0000;

        //  Add each value.
        for( 
            dwI = 0, dwParam = getParamHex( dwI, 0, 0xff ) ;
            dwParam != sm_dwNone ;
            dwI += 1, dwParam = getParamHex( dwI, 0, 0xff )
        )
        {
            m_abSearchValues[ m_dwNumSearchValues ] = ( Byte )dwParam;
            m_dwNumSearchValues += 1;
        }
    }
    else
    {
        m_dwSearchStart += 1;
    }

    //  We must be searching on at least one item.
    if( m_dwNumSearchValues == 0 )
    {
        strcpy( m_pDebugger->m_strStatus, "Must specify search values" );
        return( Debugger::LEAVE_NO );
    }

    //  Do the search.
    while( 
        ( m_dwSearchStart + m_dwNumSearchValues ) < 
        m_pDebugger->m_pCPU->getAddrSize( )
    )
    {
        //  Do the compare.
        for( dwI = 0 ; dwI < m_dwNumSearchValues ; dwI += 1 )
        {
            if(
                m_pDebugger->m_pCPU->readMem( m_dwSearchStart + dwI ) != 
                m_abSearchValues[ dwI ]
            )
            {
                break;
            }
        }

        //  Did we match?
        if( dwI >= m_dwNumSearchValues )
        {
            break;
        }

        //  Move on.
        m_dwSearchStart += 1;
    }

    //  Did we match?
    if( 
        ( m_dwSearchStart + m_dwNumSearchValues ) > 
        m_pDebugger->m_pCPU->getAddrSize( ) 
    )
    {
        strcpy( m_pDebugger->m_strStatus, "Pattern not found." );
    }
    else
    {
        sprintf( 
            m_pDebugger->m_strStatus,
            "Pattern found at $%08lx.",
            m_dwSearchStart
        );

        m_pDebugger->m_dwMemAddr = m_dwSearchStart;
    }

    return( Debugger::LEAVE_NO );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: cmdChangeCPU
//
//  Description:
//
//      This member is called to change the currently displayed CPU.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      One of the "leave" status codes.
//
///////////////////////////////////////////////////////////////////////////////
Debugger::Leave
DebugCommand::cmdChangeCPU(
)
{
    ASSERT( m_pDebugger->m_pGame != NULL );

    //  The list of CPUs for the current game.
    const KPtrList<CPU>& cpuList = m_pDebugger->m_pGame->getCPUList( );

    //  Which CPU to change to.
    DWord dwCPU = getParamHex( 0, 0, cpuList.entries( ) - 1 );


    //  If no CPU was specified then use the currently executing CPU, 
    //  otherwise use the one specified.
    if( dwCPU == sm_dwNone )
    {
        m_pDebugger->m_pCPU = CPU::sm_pCPU;
        m_pDebugger->m_dwSrcAddr = cpuList[ dwCPU ]->getReg( CPU::REG_PC );

        strcpy( m_pDebugger->m_strStatus, "Changed to current CPU." );
    }
    else
    {
        m_pDebugger->m_pCPU = cpuList[ dwCPU ];
        m_pDebugger->m_dwSrcAddr = cpuList[ dwCPU ]->getReg( CPU::REG_PC );

        strcpy( m_pDebugger->m_strStatus, "Changed to new CPU." );
    }

    return( Debugger::LEAVE_NO );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: cmdTrace
//
//  Description:
//
//      This member is called to process the trace command.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      One of the "leave" status codes.
//
///////////////////////////////////////////////////////////////////////////////
Debugger::Leave
DebugCommand::cmdTrace(
)
{
    //  Get the type of tracing.
    const char* pstrType = getParam( 0 );

    //  If there is no type then turn tracing off, otherwise turn it on.
    if( pstrType == NULL )
    {
        if( m_pDebugger->m_eTrace != BreakPoint::NONE ) 
        {
            m_pDebugger->m_eTrace       = BreakPoint::NONE;
            m_pDebugger->m_dwTraceStart = 0x0000;
            m_pDebugger->m_dwTraceEnd   = 0x0000;
            delete m_pDebugger->m_pTraceFile;
            m_pDebugger->m_pTraceFile   = NULL;
            m_pDebugger->m_pTraceCPU    = NULL;
        }
        strcpy( m_pDebugger->m_strStatus, "Tracing off." );
    }
    else
    {
        //  If tracing is already on then report the fact to the user.
        if( m_pDebugger->m_eTrace != BreakPoint::NONE )
        {
            strcpy( m_pDebugger->m_strStatus, "Tracing already on." );
        }
        else
        {
            //  What type of tracing?
            if( *pstrType == 'P' )
            {
                m_pDebugger->m_eTrace = BreakPoint::PC;
            }
            else
            if( *pstrType == 'R' )
            {
                m_pDebugger->m_eTrace = BreakPoint::READ;
            }
            else
            if( *pstrType == 'W' )
            {
                m_pDebugger->m_eTrace = BreakPoint::WRITE;
            }
    
            //  Assign the start/end.
            m_pDebugger->m_dwTraceStart = 
                getParamHex( 1, 0, m_pDebugger->m_pCPU->getAddrSize( ) - 1 );
            m_pDebugger->m_dwTraceEnd = 
                getParamHex( 2, 0, m_pDebugger->m_pCPU->getAddrSize( ) - 1 );

            //  If there is no end specified then default it to the start.
            if( m_pDebugger->m_dwTraceEnd == sm_dwNone )
            {
                m_pDebugger->m_dwTraceEnd = m_pDebugger->m_dwTraceStart;
            }

            //  If no start then use the complete range.
            if( m_pDebugger->m_dwTraceStart == sm_dwNone )
            {
                m_pDebugger->m_dwTraceStart = 0x0000;
                m_pDebugger->m_dwTraceEnd   = 
                    m_pDebugger->m_pCPU->getAddrSize( ) - 1;
            }

            //  If there was no type specified then we have a problem.
            if( m_pDebugger->m_eTrace == BreakPoint::NONE )
            {
                m_pDebugger->m_eTrace = BreakPoint::NONE;
                sprintf( 
                    m_pDebugger->m_strStatus,
                    "Usage: T [ P|R|W "
                    "$000000-$%06lx [$000000-$%06lx] ]",
                    m_pDebugger->m_pCPU->getAddrSize( ) - 1,
                    m_pDebugger->m_pCPU->getAddrSize( ) - 1
                );
            }
            else
            {
                //  Open the trace file.
                m_pDebugger->m_pTraceFile = new AppFile( "trace.cpu", TRUE );
                if( m_pDebugger->m_pTraceFile->isOpen( ) )
                {
                    m_pDebugger->m_pTraceCPU = m_pDebugger->m_pCPU;
                    strcpy( m_pDebugger->m_strStatus, "Tracing On" );
                }
                else
                {              
                    strcpy( 
                        m_pDebugger->m_strStatus, "Couldn't open trace file."
                    );
                }
            }
        }
    }

    return( Debugger::LEAVE_NO );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: getCommand
//
//  Description:
//
//      This member is responsible for returning the command the user has
//      entered.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The command entered.
//
///////////////////////////////////////////////////////////////////////////////
DebugCommand::Command
DebugCommand::getCommand(
)
{
    //  The command character.
    char* pCmd;

 
    //  Loop until the command is found.
    for( 
        pCmd = &( m_strCmd[ 0 ] ) ; 
        ( *pCmd == ' ' ) && ( *pCmd != '\0' ) ;
        pCmd += 1
    )
    {
        //  All the work is done in the loop.
    }

    //  If we have found the command then return.
    switch( *pCmd )
    {
        case 'C': return( CMD_CONTINUE );
        case 'N': return( CMD_NEXT );
        case 'M': return( CMD_MEMORY );
        case 'S': return( CMD_SOURCE );
        case 'B': return( CMD_PC_BP );
        case 'R': return( CMD_READ_BP );
        case 'W': return( CMD_WRITE_BP );
        case 'D': return( CMD_DISABLE_BP );
        case 'E': return( CMD_ENABLE_BP );
        case 'X': return( CMD_DELETE_BP );
        case 'A': return( CMD_ASSIGN_MEM );
        case 'F': return( CMD_SEARCH_MEM );
        case 'P': return( CMD_CPU );
        case 'T': return( CMD_TRACE );
    }

    //  The command is invalid.
    return( CMD_INVALID );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: getParam
//
//  Description:
//
//      This member is responsible for returning the parameter specified.
//
//  Parameters:
//
//      dwIdx (input)
//          The parameter to return.  Numbering begins at 0 for the first
//          parameter (not including the command).
//
//  Returns:
//
//      A pointer to the parameter.
//
///////////////////////////////////////////////////////////////////////////////
const
char*
DebugCommand::getParam(
    const DWord dwIdx
)
{
    //  A temporary buffer.
    static char strCmd[ 80 ];

    //  A token.
    char* pToken = NULL;


    //  Make a copy of the command buffer.
    strcpy( strCmd, m_strCmd );

    //  The parameter is token dwIdx + 1 (for the command).
    for( DWord dwI = 0 ; dwI <= dwIdx + 1 ; dwI += 1 )
    {
        //  Get the next token.
        if( dwI == 0 )
        {
            pToken = strtok( strCmd, " " );
        }
        else
        {
            pToken = strtok( NULL, " " );
        }

        //  If the token isn't found quit.
        if( pToken == NULL )
        {
            break;
        }
    }
 
    //  Return the token.
    return( pToken );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: getParamDec
//
//  Description:
//
//      This member is responsible for returning the parameter specified.
//      It treats the parameter as a decimal value.
//
//  Parameters:
//
//      dwIdx (input)
//          The parameter to return.  Numbering begins at 0 for the first
//          parameter (not including the command).
//
//      dwStart (input)
//          The start of the range of valid values for the parameter.  -1
//          indicates that there is no lower limit.
//
//      dwEnd (input)
//          The end of the range of valid values for the parameter.  -1
//          indicates that there is no upper limit.
//
//  Returns:
//
//      The command entered.
//
///////////////////////////////////////////////////////////////////////////////
DWord
DebugCommand::getParamDec(
    const DWord dwIdx,
    const DWord dwStart /* = ( DWord )-1 */,
    const DWord dwEnd   /* = ( DWord )-1 */
)
{
    //  The parameter in string and integer form.
    const char* pstrParam;
    DWord       dwParam;


    //  Does the parameter exist?
    pstrParam = getParam( dwIdx );
    if( pstrParam != NULL )
    {
        dwParam = atol( pstrParam );
       
        //  Check bounds.
        if( 
            ( ( dwStart != sm_dwNone ) && ( dwParam < dwStart ) ) ||
            ( ( dwEnd   != sm_dwNone ) && ( dwParam > dwEnd ) )
        )
        {
            dwParam = sm_dwNone;
        }
    }
    else
    {
        dwParam = sm_dwNone;
    }

    //  return the value.
    return( dwParam );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: getParamHex
//
//  Description:
//
//      This member is responsible for returning the parameter specified.
//      It treats the parameter as a hex value.
//
//  Parameters:
//
//      dwIdx (input)
//          The parameter to return.  Numbering begins at 0 for the first
//          parameter (not including the command).
//
//      dwStart (input)
//          The start of the range of valid values for the parameter.  -1
//          indicates that there is no lower limit.
//
//      dwEnd (input)
//          The end of the range of valid values for the parameter.  -1
//          indicates that there is no upper limit.
//
//  Returns:
//
//      The command entered.
//
///////////////////////////////////////////////////////////////////////////////
DWord
DebugCommand::getParamHex(
    const DWord dwIdx,
    const DWord dwStart /* = ( DWord )-1 */,
    const DWord dwEnd   /* = ( DWord )-1 */
)
{
    //  The parameter in string and integer form.
    const char* pstrParam;
    DWord       dwParam;


    //  Does the parameter exist?
    pstrParam = getParam( dwIdx );
    if( pstrParam != NULL )
    {
        sscanf( pstrParam, "%lx", &dwParam );
       
        //  Check bounds.
        if( 
            ( ( dwStart != sm_dwNone ) && ( dwParam < dwStart ) ) ||
            ( ( dwEnd   != sm_dwNone ) && ( dwParam > dwEnd ) )
        )
        {
            dwParam = sm_dwNone;
        }
    }
    else
    {
        dwParam = sm_dwNone;
    }

    //  return the value.
    return( dwParam );
}


#endif
