///////////////////////////////////////////////////////////////////////////////
//
//  File:    dbgsrc.cpp
//
//  Class:   DebugSource
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class represents a window in the Replay+ debugger that is
//      used to browse the disassembled source code for the current
//      CPU being executed.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

#ifdef DEBUGGER

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdio.h>
#include <string.h>

//  Application Headers.
#include "dbgsrc.h"
#include "cpu.h"
#include "clip.h"
#include "brkpt.h"
#include "keyb.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: DebugSource
//
//  Description:
//
//      This is the main constructor for the debug source window object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//      pDebugger (input)
//          The debugger that the window belongs to.
//
//      dwX (input)
//          The X position of the window.
//
//      dwY (input)
//          The Y position of the window.
//
//      dwWidth (input)
//          The width of the window.
//
//      dwHeight (input)
//          The height of the window.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
DebugSource::DebugSource(
    const KString& iName,
    Debugger*      pDebugger,
    const DWord    dwX,
    const DWord    dwY,
    const DWord    dwWidth,
    const DWord    dwHeight
)
:
    DebugWindow  ( iName, pDebugger, dwX, dwY, dwWidth, dwHeight ),
    m_dwPrevAddr ( 0x0000 ),
    m_dwNextAddr ( 0x0000 )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~DebugSource
//
//  Description:
//
//      This is the destructor for the debugger source window object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
DebugSource::~DebugSource(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the debugger info window.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
DebugSource::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "DebugSource" );

    return( className );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: draw
//
//  Description:
//
//      This member is called when the window is to draw itself.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
DebugSource::draw(
)
{
    //  A temporary buffer.
    char strTemp[ 128 ];

    //  Addresses of interest.
    DWord dwAddr;
    DWord dwLastGoodAddr;
    DWord dwTraverseAddr;

    //  The number of lines from the top of the page to the source line.
    DWord dwLinesToSrc;


    DebugWindow::draw( );

    
    //  Since we'd like to see lines both above and below the current line
    //  for context, we need to search backwards to find an address that
    //  if we start disassembling at, we will disassemble the line at the
    //  current position and this line will appear somewhere around the 
    //  middle of the page.
    dwLastGoodAddr = m_pDebugger->m_dwSrcAddr;
    dwAddr         = m_pDebugger->m_dwSrcAddr;
    do
    {
        //  If we at the beginning of the source list then break out.
        if( dwAddr == 0x0000 )
        {
            break;
        }

        //  Back up one location.
        dwAddr         -= 1;
        dwTraverseAddr  = dwAddr;

        //  Find out whether disassembling from here would result in 
        //  disassembling our current source line.
        for( 
            dwLinesToSrc = 0 ; 
            dwTraverseAddr < m_pDebugger->m_dwSrcAddr ; 
            dwLinesToSrc += 1 
        )
        {
            m_pDebugger->m_pCPU->dbgDisassemble( dwTraverseAddr );
        }

        //  If we've landed on the source address then this is a good address
        //  to start at.
        if( dwTraverseAddr == m_pDebugger->m_dwSrcAddr )
        {
            dwLastGoodAddr = dwAddr;
        }
    }
    while( dwLinesToSrc < ( m_dwPageLength / 2 ) );

    //  Use the last good address.
    dwAddr = dwLastGoodAddr;

    //  Assign the initial previous and next addresses.
    m_dwPrevAddr = m_pDebugger->m_dwSrcAddr;
    m_dwNextAddr = m_pDebugger->m_dwSrcAddr;

    //  Draw lines until we run out of space.
    while( m_pCurClipping->m_nMinY < m_pCurClipping->m_nMaxY )
    {
        //  Indicates whether we should be updating the next address this
        //  time around.
        Byte bUpdateNextAddr = ( dwAddr == m_pDebugger->m_dwSrcAddr );

        //  Update the previous address.
        if( dwAddr < m_pDebugger->m_dwSrcAddr )
        {
            m_dwPrevAddr = dwAddr;
        }

        //  Important Note:  Do not place the indicator call in the sprintf
        //  since we are not guaranteed which call (the indicator() or the
        //  disassemble()) will get called first, and if it's the disassemble()
        //  then dwAddr will be modified.
        char strInd[ 4 ];
        Byte bHilight = 
            ( dwAddr == ( DWord )m_pDebugger->m_pCPU->getReg( CPU::REG_PC ) );
        strcpy( strInd, indicator( dwAddr ) );
        sprintf( 
            strTemp, 
            "%-3s%-75s", 
            strInd,
            m_pDebugger->m_pCPU->dbgDisassemble( dwAddr ) 
        );
        writeLine( strTemp, bHilight );

        //  Update the next address.
        if( bUpdateNextAddr )
        {
            m_dwNextAddr = dwAddr;
        }
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: execute
//
//  Description:
//
//      This member is called when the window is to execute.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      One of the "leave" status codes.
//
///////////////////////////////////////////////////////////////////////////////
Debugger::Leave
DebugSource::execute(
)
{
    //  Allow basic movements.
    if( m_pKeyboard->switchOnWait( Keyboard::KEY__UP ) )
    {
        if( m_pDebugger->m_dwSrcAddr > 0 )
        {
            m_pDebugger->m_dwSrcAddr = m_dwPrevAddr;
            m_pDebugger->m_bRedrawScreen = TRUE;
        }
    }
    else
    if( m_pKeyboard->switchOnWait( Keyboard::KEY__DOWN ) )
    {
        m_pDebugger->m_dwSrcAddr = m_dwNextAddr;
        m_pDebugger->m_bRedrawScreen = TRUE;
    }
    else
    if( m_pKeyboard->switchOnWait( Keyboard::KEY__PGUP ) )
    {
        if( m_pDebugger->m_dwSrcAddr >= m_dwPageLength )
        {
            m_pDebugger->m_dwSrcAddr -= m_dwPageLength;
        }
        else
        {
            m_pDebugger->m_dwSrcAddr = 0;
        }
        m_pDebugger->m_bRedrawScreen = TRUE;
    }
    else
    if( m_pKeyboard->switchOnWait( Keyboard::KEY__PGDN ) )
    {
        m_pDebugger->m_dwSrcAddr += m_dwPageLength;
        m_pDebugger->m_bRedrawScreen = TRUE;
    }

    return( Debugger::LEAVE_NO );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: indicator
//
//  Description:
//
//      This member returns the indicator symbols for the line starting
//      at the specified address.
//
//  Parameters:
//
//      dwAddr (input)
//          The address that the line starts at.
//
//  Returns:
//
//      A string containing the indicators.
//
///////////////////////////////////////////////////////////////////////////////
const
char*
DebugSource::indicator(
    const DWord dwAddr
)
{
    //  The buffer is valid until the next call to the function.
    static char strInd[ 4 ];

    //  The indicator is formed like Xxx where:
    //    X  will contain 'E' indicating an enabled breakpoint,
    //          'D' indicating a disabled breakpoint or ' ' if no breakpoint.
    //    xx will contain "->" if the line is where the program counter is or
    //       spaces if not.

    //  First of all, assume no breakpoint.
    strcpy( strInd, " " );
   
    //  No look for a breakpoint.
    for( DWord dwI = 0 ; dwI < m_pDebugger->m_bpList.entries( ) ; dwI += 1 )
    {
        //  If the breakpoint matches then indicate that in the buffer.
        if( 
            m_pDebugger->m_bpList[ dwI ]->match( 
                BreakPoint::PC, m_pDebugger->m_pCPU, dwAddr 
            ) 
        )
        {
            sprintf( 
                strInd, 
                "%s", 
                m_pDebugger->m_bpList[ dwI ]->isEnabled( ) ? "E" : "D"
            );
            break;
        }
    }

    //  No add the PC indicator if appropriate.
    if( dwAddr == ( DWord )m_pDebugger->m_pCPU->getReg( CPU::REG_PC ) )
    {
        strcat( strInd, "->" );
    }
    else
    {
        strcat( strInd, "  " );
    }

    return( strInd );
}


#endif
