///////////////////////////////////////////////////////////////////////////////
//
//  File:    dbgwin.cpp
//
//  Class:   DebugWindow
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This abstract class serves as a base for all of the various
//      windows that appear in the Replay+ debugger.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

#ifdef DEBUGGER

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "dbgwin.h"
#include "replay.h"
#include "debugger.h"
#include "bitmap.h"
#include "clip.h"
#include "gfxset.h"
#include "ctable.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: DebugWindow
//
//  Description:
//
//      This is the main constructor for the debug window base object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//      pDebugger (input)
//          The debugger that the window belongs to.
//
//      dwX (input)
//          The X position of the window.
//
//      dwY (input)
//          The Y position of the window.
//
//      dwWidth (input)
//          The width of the window.
//
//      dwHeight (input)
//          The height of the window.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
DebugWindow::DebugWindow(
    const KString& iName,
    Debugger*      pDebugger,
    const DWord    dwX,
    const DWord    dwY,
    const DWord    dwWidth,
    const DWord    dwHeight
)
:
    RepBase              ( iName ),
    m_pDebugger          ( pDebugger ),
    m_pCanvas            ( Replay::s_instance( ).getCanvas( ) ),
    m_pKeyboard          ( Replay::s_instance( ).getKeyboard( ) ),
    m_pBitmap            ( m_pDebugger->m_pScreenBitmap ),
    m_pFont              ( &( m_pCanvas->getFont( Canvas::FONT_SMALL ) ) ),
    m_nFontWidth         ( ( int32 )( *m_pFont )[ 0 ]->getWidth( ) ),
    m_nFontHeight        ( ( int32 )( *m_pFont )[ 0 ]->getHeight( ) ),
    m_pColourTable       ( m_pDebugger->m_pNormalColTable ),
    m_pFullClipping      ( NULL ),
    m_pCurClipping       ( NULL ),
    m_dwPageLength       ( 0 ),
    m_bActive            ( FALSE )
{
    //  Check the parameters.
    ASSERT( m_pDebugger       != NULL );
    ASSERT( m_pCanvas         != NULL );
    ASSERT( m_pBitmap         != NULL );
    ASSERT( m_pColourTable    != NULL );

    //  Create the clipping regions.
    m_pFullClipping = new Clipping( 
        dwX, dwX + dwWidth - 1, dwY, dwY + dwHeight - 1
    );
    m_pCurClipping = new Clipping( 0, 0, 0, 0 );
    resetCurClipping( );

    //  Calculate the length of the page in lines (excluding the border).
    m_dwPageLength = 
        ( m_pCurClipping->m_nMaxY - m_pCurClipping->m_nMinY + 1 ) / 
        m_nFontHeight;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~DebugWindow
//
//  Description:
//
//      This is the destructor for the debugger window base object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
DebugWindow::~DebugWindow(
)
{
    //  Free the clipping regions.
    delete m_pFullClipping;
    delete m_pCurClipping;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: activate
//
//  Description:
//
//      This member is called when the window is either activated or 
//      deactivated.
//
//  Parameters:
//
//      bActive (input)
//          Indicates whether to activate or deactivate the window.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
DebugWindow::activate(
    const Byte bActive
)
{
    m_bActive = bActive;

    //  Grab the appropriate colour table.
    if( m_bActive )
    {
        m_pColourTable = m_pDebugger->m_pSelectColTable;
    }
    else
    {
        m_pColourTable = m_pDebugger->m_pNormalColTable;
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: draw
//
//  Description:
//
//      This member is called when the window is to draw itself.  This base
//      member will simply draw a border around the window with the window
//      title contained in it.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
DebugWindow::draw(
)
{
    //  The current X, Y coordinate to draw at.
    int32 nX = m_pFullClipping->m_nMinX;
    int32 nY = m_pFullClipping->m_nMinY;
 

    //  First we draw a box around the window.  The box characters are as 
    //  follows in the font:
    //    3 - vertical bar.
    //    4 - horizontal bar.
    //    5 - upper left.
    //    6 - upper right.
    //    7 - lower left.
    //    8 - lower right.

    //  Draw the horizontal borders.
    for( 
        nX = m_pFullClipping->m_nMinX ; 
        nX < m_pFullClipping->m_nMaxX ; 
        nX += m_nFontWidth 
    )
    {
        m_pBitmap->blit(
            ( *m_pFont )[ 4 ],
            nX,
            m_pFullClipping->m_nMinY,
            *m_pColourTable,
            FALSE,
            FALSE,
            m_pBitmap->getFullClipping( ),
            Bitmap::TRANSPARENCY_NONE
        );
        m_pBitmap->blit(
            ( *m_pFont )[ 4 ],
            nX,
            m_pFullClipping->m_nMaxY - m_nFontHeight + 1,
            *m_pColourTable,
            FALSE,
            FALSE,
            m_pBitmap->getFullClipping( ),
            Bitmap::TRANSPARENCY_NONE
        );
    }

    //  Draw the vertical borders.
    for( 
        nY = m_pFullClipping->m_nMinY ; 
        nY < m_pFullClipping->m_nMaxY ; 
        nY += m_nFontHeight 
    )
    {
        m_pBitmap->blit(
            ( *m_pFont )[ 3 ],
            m_pFullClipping->m_nMinX,
            nY,
            *m_pColourTable,
            FALSE,
            FALSE,
            m_pBitmap->getFullClipping( ),
            Bitmap::TRANSPARENCY_NONE
        );
        m_pBitmap->blit(
            ( *m_pFont )[ 3 ],
            m_pFullClipping->m_nMaxX - m_nFontWidth + 1,
            nY,
            *m_pColourTable,
            FALSE,
            FALSE,
            m_pBitmap->getFullClipping( ),
            Bitmap::TRANSPARENCY_NONE
        );
    }

    //  Draw the four corners.
    m_pBitmap->blit(
        ( *m_pFont )[ 5 ],
        m_pFullClipping->m_nMinX,
        m_pFullClipping->m_nMinY,
        *m_pColourTable,
        FALSE,
        FALSE,
        m_pBitmap->getFullClipping( ),
        Bitmap::TRANSPARENCY_NONE
    );
    m_pBitmap->blit(
        ( *m_pFont )[ 6 ],
        m_pFullClipping->m_nMaxX - m_nFontWidth + 1,
        m_pFullClipping->m_nMinY,
        *m_pColourTable,
        FALSE,
        FALSE,
        m_pBitmap->getFullClipping( ),
        Bitmap::TRANSPARENCY_NONE
    );
    m_pBitmap->blit(
        ( *m_pFont )[ 7 ],
        m_pFullClipping->m_nMinX,
        m_pFullClipping->m_nMaxY - m_nFontHeight + 1,
        *m_pColourTable,
        FALSE,
        FALSE,
        m_pBitmap->getFullClipping( ),
        Bitmap::TRANSPARENCY_NONE
    );
    m_pBitmap->blit(
        ( *m_pFont )[ 8 ],
        m_pFullClipping->m_nMaxX - m_nFontWidth + 1,
        m_pFullClipping->m_nMaxY - m_nFontHeight + 1,
        *m_pColourTable,
        FALSE,
        FALSE,
        m_pBitmap->getFullClipping( ),
        Bitmap::TRANSPARENCY_NONE
    );

    //  Now draw the title of the window.
    m_pCanvas->drawText(
        getInstanceName( ),
        m_pBitmap,
        Canvas::FONT_SMALL,
        m_pFullClipping,
        *m_pColourTable,
        TRUE
    );

    //  Initialize the current clipping value.
    resetCurClipping( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: resetCurClipping
//
//  Description:
//
//      This member is called to reset the current clipping area to an area
//      that is the full clipping area less the border.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
DebugWindow::resetCurClipping(
)
{
    //  Reset the region.
    m_pCurClipping->m_nMinX = m_pFullClipping->m_nMinX + m_nFontWidth;
    m_pCurClipping->m_nMaxX = m_pFullClipping->m_nMaxX - m_nFontWidth;
    m_pCurClipping->m_nMinY = m_pFullClipping->m_nMinY + m_nFontHeight;
    m_pCurClipping->m_nMaxY = m_pFullClipping->m_nMaxY - m_nFontHeight;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: writeLine
//
//  Description:
//
//      This member is a utility function to display the specified string
//      in the window and update the clipping region to the next line.
//
//  Parameters:
//
//      pstrLine (input)
//          The line to display.
//
//      bHilight (input)
//          Indicates whether the line should be highlighted or not.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
DebugWindow::writeLine(
    const char* pstrLine,
    const Byte  bHilight /* = FALSE */
)
{
    ASSERT( pstrLine != NULL );

    //  Draw the line.
    m_pCanvas->drawText(
        pstrLine,
        m_pBitmap,
        Canvas::FONT_SMALL,
        m_pCurClipping,
        bHilight ? *( m_pDebugger->m_pHilightColTable ) : *m_pColourTable,
        FALSE
    );
    m_pCurClipping->m_nMinY += m_nFontHeight;
}




#endif
