///////////////////////////////////////////////////////////////////////////////
//
//  File:    debugger.cpp
//
//  Class:   Debugger
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class represents an integrated debugger that can be used
//      in Replay to write drivers.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

#ifdef DEBUGGER

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "debugger.h"
#include "dbgwin.h"
#include "dbgbps.h"
#include "dbgcmd.h"
#include "dbginfo.h"
#include "dbgmem.h"
#include "dbgreg.h"
#include "dbgsrc.h"
#include "dbgstat.h"
#include "replay.h"
#include "canvas.h"
#include "ctable.h"
#include "bitmap.h"
#include "keyb.h"
#include "clock.h"
#include "cpu.h"
#include "brkpt.h"
#include "appfile.h"


///////////////////////////////////////////////////////////////////////////////
//  File Constants.
///////////////////////////////////////////////////////////////////////////////
//  Some strings.
static const char* DEBUGGER_SCREEN_NAME  = "Debugger";

//  The page attributes.
static const Canvas::PageNumber PAGE   = Canvas::PAGE_ONE;
static const DWord PAGE_WIDTH          = 640;
static const DWord PAGE_HEIGHT         = 480;

//  The size of the windows.
static const DWord COL1_WIDTH          = PAGE_WIDTH / 2;
static const DWord COL2_WIDTH          = PAGE_WIDTH / 2;
static const DWord INFO_HEIGHT         = 80;
static const DWord BPS_HEIGHT          = 80;
static const DWord REG_HEIGHT          = 160;
static const DWord SRC_HEIGHT          = 184;
static const DWord MEM_HEIGHT          = 88;
static const DWord CMD_HEIGHT          = 24;
static const DWord STAT_HEIGHT         = 24;



///////////////////////////////////////////////////////////////////////////////
//  Static Member Initialization.
///////////////////////////////////////////////////////////////////////////////
const Byte Debugger::sm_bColourBG   = 242;
const Byte Debugger::sm_bColourNFG  = 243;
const Byte Debugger::sm_bColourHFG  = 244;
const Byte Debugger::sm_bColourSFG  = 245;



///////////////////////////////////////////////////////////////////////////////
//
//  Function: Debugger
//
//  Description:
//
//      This is the main constructor for the Debugger object.  It
//      is protected because it is a singleton and therefore cannot be
//      instantiated by anyone but itself.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Debugger::Debugger(
    const KString& iName
)
:
    RepBase              ( iName ),
    m_eWindow            ( WIN_COMMAND ),
    m_pDebugWindow       ( NULL ),
    m_pDebugInfo         ( NULL ),
    m_pDebugBreakPoints  ( NULL ),
    m_pDebugSource       ( NULL ),
    m_pDebugRegisters    ( NULL ),
    m_pDebugMemory       ( NULL ),
    m_pDebugCommand      ( NULL ),
    m_pDebugStatus       ( NULL ),
    m_pCanvas            ( Replay::s_instance( ).getCanvas( ) ),
    m_bRedrawScreen      ( FALSE ),
    m_bRedrawWindow      ( FALSE ),
    m_pScreenBitmap      ( NULL ),
    m_pFont              ( &( m_pCanvas->getFont( Canvas::FONT_SMALL ) ) ),
    m_pNormalColTable    ( NULL ),
    m_pHilightColTable   ( NULL ),
    m_pSelectColTable    ( NULL ),
    m_cpuList            ( 3 ),
    m_bInterrupt         ( FALSE ),
    m_pGame              ( NULL ),
    m_pCPU               ( NULL ),
    m_bpList             ( 5 ),
    m_dwBP               ( 0 ),
    m_dwSrcAddr          ( 0x0000 ),
    m_dwRegister         ( 0 ),
    m_dwMemAddr          ( 0x0000 ),
    m_pTraceFile         ( NULL ),
    m_pTraceCPU          ( NULL ),
    m_eTrace             ( BreakPoint::NONE ),
    m_dwTraceStart       ( 0x0000 ),
    m_dwTraceEnd         ( 0x0000 )
{
    //  Create that screen bitmap for the debugger.  This should be done
    //  before the windows are created since they may make use of it.
    m_pScreenBitmap = m_pCanvas->createBitmap( 
        DEBUGGER_SCREEN_NAME, PAGE_WIDTH, PAGE_HEIGHT, TRUE 
    );

    //  Create the colour tables.  These should be done before the windows
    //  are created since they may make use of them.
    m_pNormalColTable  = new ColourTable( "Normal CTable", 2 );
    m_pHilightColTable = new ColourTable( "Hilight CTable", 2 );
    m_pSelectColTable  = new ColourTable( "Select CTable", 2 );

    //  Create the debugger windows.
    m_pDebugInfo = new DebugInfo(
        "Information", this,
        0, 0, COL1_WIDTH, INFO_HEIGHT
    );
    m_pDebugBreakPoints = new DebugBreakPoints(
        "Break Points", this,
        0, INFO_HEIGHT, COL1_WIDTH , BPS_HEIGHT
    );
    m_pDebugRegisters = new DebugRegisters(
        "Registers", this,
        COL1_WIDTH, 0, COL2_WIDTH, REG_HEIGHT
    );
    m_pDebugSource = new DebugSource(
        "Source", this,
        0, REG_HEIGHT, PAGE_WIDTH, SRC_HEIGHT
    );
    m_pDebugMemory = new DebugMemory(
        "Memory", this,
        0, REG_HEIGHT + SRC_HEIGHT, PAGE_WIDTH, MEM_HEIGHT
    );
    m_pDebugCommand = new DebugCommand(
        "Command Entry", this,
        0, REG_HEIGHT + SRC_HEIGHT + MEM_HEIGHT, PAGE_WIDTH, CMD_HEIGHT
    );
    m_pDebugStatus = new DebugStatus(
        "Status", this,
        0, REG_HEIGHT + SRC_HEIGHT + MEM_HEIGHT + CMD_HEIGHT, 
        PAGE_WIDTH, STAT_HEIGHT
    );

    //  Select the starting window.
    setCurWindow( m_eWindow );

    //  Add a page for the debugger.
    m_pCanvas->addPage( DEBUGGER_SCREEN_NAME, PAGE, PAGE_WIDTH, PAGE_HEIGHT );

    //  Initialize the status message.
    strcpy( m_strStatus, "" );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~Debugger
//
//  Description:
//
//      This is the destructor for the debugger.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Debugger::~Debugger(
)
{
    //  Remove the debugger page from the canvas.
    m_pCanvas->removePage( PAGE );


    ASSERT( m_pDebugInfo         != NULL );
    ASSERT( m_pDebugBreakPoints  != NULL );
    ASSERT( m_pDebugSource       != NULL );
    ASSERT( m_pDebugRegisters    != NULL );
    ASSERT( m_pDebugMemory       != NULL );
    ASSERT( m_pDebugCommand      != NULL );
    ASSERT( m_pDebugStatus       != NULL );
    ASSERT( m_pScreenBitmap      != NULL );
    ASSERT( m_pNormalColTable    != NULL );
    ASSERT( m_pHilightColTable   != NULL );
    ASSERT( m_pSelectColTable    != NULL );

    //  Free the states.
    delete m_pDebugInfo;
    delete m_pDebugBreakPoints;
    delete m_pDebugSource;
    delete m_pDebugRegisters;
    delete m_pDebugMemory;
    delete m_pDebugCommand;
    delete m_pDebugStatus;

    //  Free the bitmaps.
    delete m_pScreenBitmap;

    //  Free the colour tables.
    delete m_pNormalColTable;
    delete m_pHilightColTable;
    delete m_pSelectColTable;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the debugger.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
Debugger::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "Debugger" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_instance
//
//  Description:
//
//      This member returns the singleton Debugger object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      A reference to the singleton Debugger object.
//
///////////////////////////////////////////////////////////////////////////////
Debugger&
Debugger::s_instance(
)
{
    //  The instance.
    static Debugger debugger( "Debugger" );

    return( debugger );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: setCurWindow
//
//  Description:
//
//      This member is called to set a new current window in the debugger.
//
//  Parameters:
//
//      eWindow (input)
//          The window that is to become current.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void  
Debugger::setCurWindow ( 
    DbgWindow eWindow
)
{
    //  Deactivate the old window if there was one.
    if( m_pDebugWindow != NULL )
    {
        m_pDebugWindow->activate( FALSE );
    }


    //  Set the new current window.
    m_eWindow = eWindow;
    switch( m_eWindow )
    {
        case WIN_INFO:         m_pDebugWindow = m_pDebugInfo;           break;
        case WIN_BREAK_POINTS: m_pDebugWindow = m_pDebugBreakPoints;    break;
        case WIN_SOURCE:       m_pDebugWindow = m_pDebugSource;         break;
        case WIN_REGISTERS:    m_pDebugWindow = m_pDebugRegisters;      break;
        case WIN_MEMORY:       m_pDebugWindow = m_pDebugMemory;         break;
        case WIN_COMMAND:      m_pDebugWindow = m_pDebugCommand;        break;
        case WIN_STATUS:       m_pDebugWindow = m_pDebugStatus;         break;
        default: fatalError( "Unknown debugger window %d.", eWindow );  break;
    }

    //  Tell the new window that it has been activated.
    m_pDebugWindow->activate( TRUE );

    //  The debugger screen should be redrawn to reflect the change.
    m_bRedrawScreen = TRUE;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: clear
//
//  Description:
//
//      This member should be called to clear out all debugger information.
//      Typically, this is done when a new game has been selected.
//
//  Parameters:
//
//      pGame (input)
//          The new game that the debugger is now associated with.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Debugger::clear(
    Game* pGame
) 
{
    ASSERT( pGame != NULL );

    //  Set back to the command window.
    setCurWindow( WIN_COMMAND );

    //  Reset the redraw items.
    m_bRedrawScreen = FALSE;
    m_bRedrawWindow = FALSE;

    //  Clear out the list of used CPUs.
    m_cpuList.clear( );

    //  No interrupt.
    m_bInterrupt = FALSE;

    //  Assign new game but no CPU yet.
    m_pGame = pGame;
    m_pCPU  = NULL;

    //  Clear the breakpoint list.
    for( DWord dwI = 0 ; dwI < m_bpList.entries( ) ; dwI += 1 )
    {
        delete m_bpList[ dwI ];
    }
    m_bpList.clear( );
    m_dwBP = 0;

    //  Point the addresses to the beginning of memory.
    m_dwSrcAddr = 0x0000;
    m_dwMemAddr = 0x0000;

    //  Turn off tracing.
    m_eTrace       = BreakPoint::NONE;
    m_dwTraceStart = 0x0000;
    m_dwTraceEnd   = 0x0000;
    m_pTraceCPU    = NULL;
    if( m_pTraceFile != NULL )
    {
        delete m_pTraceFile;
        m_pTraceFile = NULL;
    }

    //  Reset the register index.
    m_dwRegister = 0;

    //  No status message.
    strcpy( m_strStatus, "" );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: check
//
//  Description:
//
//      This member is called to check whether the debugger should be invoked.
//      A call to this member should be made by each emulation core just
//      before an instruction is executed.  Only those cores which do so
//      will be fully supported by the debugger.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Debugger::check(
) 
{
    //  Indices used for iterations.
    DWord dwI;


    ASSERT( CPU::sm_pCPU != NULL );


    //  Grab the PC of the current CPU.
    DWord dwPC = CPU::sm_pCPU->getReg( CPU::REG_PC );


    //  If PC tracing is on then dump to the trace file.
    if( 
        ( m_eTrace == BreakPoint::PC ) &&
        ( m_pTraceCPU == CPU::sm_pCPU ) &&
        ( dwPC >= m_dwTraceStart ) && 
        ( dwPC <= m_dwTraceEnd )
    )
    {
        char strLine[ 80 ];
        sprintf( 
            strLine, 
            "(%s) %s\n", 
            CPU::sm_pCPU->getInstanceName( ).data( ),
            CPU::sm_pCPU->dbgDisassemble( dwPC )
        );
        m_pTraceFile->write( strLine, strlen( strLine ) );
    }


    //  A CPU is always interrupted the first time it is hit.
    for( dwI = 0 ; dwI < m_cpuList.entries( ) ; dwI += 1 )
    {
        if( m_cpuList[ dwI ] == CPU::sm_pCPU )
        {
            break;
        }
    }
    if( dwI >= m_cpuList.entries( ) )
    {
        //  We haven't encountered this CPU before so add it to the list
        //  and interrupt.
        m_cpuList.add( CPU::sm_pCPU );
        m_bInterrupt = TRUE;
    }
        
    //  Check each breakpoint for a matching one.
    for( dwI = 0 ; dwI < m_bpList.entries( ) ; dwI += 1 )
    {
        if( 
            m_bpList[ dwI ]->isEnabled( ) &&
            m_bpList[ dwI ]->match( BreakPoint::PC, CPU::sm_pCPU, dwPC ) 
        )
        {
            m_bInterrupt = TRUE;
            break;
        }
    }


    //  If the run has been interrupted then execute the debugger.
    if( m_bInterrupt )
    {
        execute( );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: checkRead
//
//  Description:
//
//      This member is called to check whether the debugger should be invoked
//      based on a memory read that is occuring.  This should be called be
//      a CPU core each time a memory read is made.  Only cores that 
//      do this will be supported fully by the debugger.
//
//  Parameters:
//
//      dwAddr (input)
//          The address that is being read from.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Debugger::checkRead(
    const DWord dwAddr
) 
{
    ASSERT( CPU::sm_pCPU != NULL );


    //  If PC tracing is on then dump to the trace file.
    if(
        ( m_eTrace == BreakPoint::READ ) &&
        ( m_pTraceCPU == CPU::sm_pCPU ) &&
        ( dwAddr >= m_dwTraceStart ) && 
        ( dwAddr <= m_dwTraceEnd )
    )
    {
        char strLine[ 80 ];
        sprintf( 
            strLine, 
            "(%s:$%08lx)Read: $%08lx\n", 
            CPU::sm_pCPU->getInstanceName( ).data( ),
            CPU::sm_pCPU->getReg( CPU::REG_PC ),
            dwAddr
        );
        m_pTraceFile->write( strLine, strlen( strLine ) );
    }

    //  Check each breakpoint for a matching one.
    for( DWord dwI = 0 ; dwI < m_bpList.entries( ) ; dwI += 1 )
    {
        if( 
            m_bpList[ dwI ]->isEnabled( ) &&
            m_bpList[ dwI ]->match( BreakPoint::READ, CPU::sm_pCPU, dwAddr )
        )
        {
            m_bInterrupt = TRUE;
            break;
        }
    }


    //  The debugger will be executed during the next check( ).
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: checkWrite
//
//  Description:
//
//      This member is called to check whether the debugger should be invoked
//      based on a memory write that is occuring.  This should be called be
//      a CPU core each time a memory write is made.  Only cores that 
//      do this will be supported fully by the debugger.
//
//  Parameters:
//
//      dwAddr (input)
//          The address that is being written to.
//
//      bValue (input)
//          The value being written.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Debugger::checkWrite(
    const DWord dwAddr,
    const Byte  bValue
) 
{
    ASSERT( CPU::sm_pCPU != NULL );


    //  If PC tracing is on then dump to the trace file.
    if(
        ( m_eTrace == BreakPoint::WRITE ) &&
        ( m_pTraceCPU == CPU::sm_pCPU ) &&
        ( dwAddr >= m_dwTraceStart ) && 
        ( dwAddr <= m_dwTraceEnd )
    )
    {
        char strLine[ 80 ];
        sprintf( 
            strLine, 
            "(%s:$%08lx)Write: $%06lx <- $%02x\n", 
            CPU::sm_pCPU->getInstanceName( ).data( ),
            CPU::sm_pCPU->getReg( CPU::REG_PC ),
            dwAddr,
            bValue
        );
        m_pTraceFile->write( strLine, strlen( strLine ) );
    }

    //  Check each breakpoint for a matching one.
    for( DWord dwI = 0 ; dwI < m_bpList.entries( ) ; dwI += 1 )
    {
        if( 
            m_bpList[ dwI ]->isEnabled( ) &&
            m_bpList[ dwI ]->match( BreakPoint::WRITE, CPU::sm_pCPU, dwAddr ) 
        )
        {
            m_bInterrupt = TRUE;
            break;
        }
    }


    //  The debugger will be executed during the next check( ).
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: execute
//
//  Description:
//
//      This member is called to execute the debugger until the user
//      indicates that he/she is finished with the debugger.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Debugger::execute(
) 
{
    //  The following is a return code from the currently executing window
    //  and indicates if we have to leave the debugger execution.
    Leave eLeave;


    //  Turn to the debugger page and set the colours.
    m_pCanvas->turnPage( PAGE );
    setColours( );

    //  Disable key mappings while we're in the debugger so that
    //  key presses don't go to the game.
    Replay::s_instance( ).getKeyboard( )->enableMappings( FALSE );

    //  For the debugger we want throttling on.
    Replay::s_instance( ).getClock( )->setThrottle( TRUE );

    //  Always redraw the first time through.
    m_pScreenBitmap->clear( sm_bColourBG );
    m_bRedrawScreen = TRUE;


    //  Our CPU will initially be the current CPU until the user changes it.
    m_pCPU = CPU::sm_pCPU;

    //  The source window should start at the current PC.
    m_dwSrcAddr = m_pCPU->getReg( CPU::REG_PC );


    //  Loop indefinately until the user is finished with the debugger.
    for( ; ; )
    {
        //  Give the system a chance to catch up.
        Replay::s_instance( ).getClock( )->frameWait( );

        //  Check for a change in the active window.
        checkWindowSwitch( );

        //  If the screen or window needs to be redrawn then do it now.
        if( m_bRedrawScreen || m_bRedrawWindow )
        {
            //  All windows or just the current?
            if( m_bRedrawScreen )
            {
                //  Allow the windows to draw themselves.
                m_pDebugInfo->draw( );
                m_pDebugBreakPoints->draw( );
                m_pDebugSource->draw( );
                m_pDebugRegisters->draw( );
                m_pDebugMemory->draw( );
                m_pDebugCommand->draw( );
                m_pDebugStatus->draw( );
            }
            else
            {
                m_pDebugWindow->draw( );
            }

            //  Now draw the screen bitmap onto the canvas.
            m_pCanvas->draw( 
                m_pScreenBitmap, 
                0, 
                0, 
                m_pScreenBitmap->getWidth( ), 
                m_pScreenBitmap->getHeight( )
            );
            m_bRedrawScreen = FALSE;
            m_bRedrawWindow = FALSE;
        }

        //  Allow the current window to execute.
        eLeave = m_pDebugWindow->execute( );

        //  The ESC key acts like a continue.
        if( 
            Replay::s_instance( ).getKeyboard( )->switchOn( Keyboard::KEY__ESC )
        )
        {
            Replay::s_instance( ).getKeyboard( )->waitUntilOff(
                Keyboard::KEY__ESC
            );
            eLeave = LEAVE_LONG;
        }

        //  If we are leaving the debugger then we have to get out of the loop.
        if( eLeave != LEAVE_NO )
        {
            break;
        }

        //  Update the system and redraw if required.
        if( Replay::s_instance( ).update( ) )
        {
            m_bRedrawScreen = TRUE;
        }
    }

    //  If we are leaving for a long time (i.e. continuing) then reset the
    //  canvas to the game.
    if( eLeave == LEAVE_LONG )
    {
        //  Don't drop back in next time around.
        m_bInterrupt = FALSE;

        //  Re-enable the key mappings.
        Replay::s_instance( ).getKeyboard( )->enableMappings( TRUE );

        //  Free all colours to return the palette to what it was.
        m_pCanvas->freeColours( );
        m_pCanvas->turnPage( Canvas::PAGE_ZERO );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: checkWindowSwitch
//
//  Description:
//
//      This member is called to check if the user has switched windows.
//      If so, the new window is set to active.
//
//  Parameters:
//
//      Nothing.
//
//  Returns:
//
//      None.
//
///////////////////////////////////////////////////////////////////////////////
void
Debugger::checkWindowSwitch(
) 
{
    //  The keyboard object.
    Keyboard* pKeyboard = Replay::s_instance( ).getKeyboard( );


    //  Window switching is done through the <TAB> key.
    if( pKeyboard->switchOn( Keyboard::KEY__TAB ) )
    {
        pKeyboard->waitUntilOff( Keyboard::KEY__TAB );

        //  Now switch to the next window.
        if( 
            pKeyboard->switchOn( Keyboard::KEY__LSHIFT ) ||
            pKeyboard->switchOn( Keyboard::KEY__RSHIFT )
        )
        {
            setCurWindow( 
                ( DbgWindow )( ( m_eWindow + ( WIN_COUNT - 1 ) ) % WIN_COUNT ) 
            );
        }
        else
        {
            setCurWindow( ( DbgWindow )( ( m_eWindow + 1 ) % WIN_COUNT ) );
        }
    }
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: setColours
//
//  Description:
//
//      This member sets the colours to appropriate values for the debugger.
//
//  Parameters:
//
//      Nothing.
//
//  Returns:
//
//      None.
//
///////////////////////////////////////////////////////////////////////////////
void
Debugger::setColours(
)
{
    //  Only a very few colours are used in the debugger.  One for the
    //  background and two foreground colours; one for normal text and
    //  one for selected text.
    m_pCanvas->captureColour( sm_bColourBG,   000, 000, 000 );
    m_pCanvas->captureColour( sm_bColourNFG,  255, 255, 255 );
    m_pCanvas->captureColour( sm_bColourHFG,  000, 255, 000 );
    m_pCanvas->captureColour( sm_bColourSFG,  255, 000, 000 );

    //  Initialize the colour table for the two types of text.
    ( *m_pNormalColTable )[ 0 ] = sm_bColourBG;
    ( *m_pNormalColTable )[ 1 ] = sm_bColourNFG;
    ( *m_pHilightColTable )[ 0 ] = sm_bColourBG;
    ( *m_pHilightColTable )[ 1 ] = sm_bColourHFG;
    ( *m_pSelectColTable )[ 0 ] = sm_bColourBG;
    ( *m_pSelectColTable )[ 1 ] = sm_bColourSFG;
}

#endif
