///////////////////////////////////////////////////////////////////////////////
//
//  File:       debugger.h
//
//  Class:      Debugger - Concrete
//  
//  Hierarchy:  Debugger - RepBase 
//
//  Author:     Kevin Brisley
//
//  Description:
//
//      This class represents an integrated debugger that can be used
//      in Replay to write drivers.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

#ifdef DEBUGGER

#ifndef _debugger_h_
#define _debugger_h_

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>

//  Application Headers.
#include "reptypes.h"
#include "repbase.h"
#include "kstring.h"
#include "kplist.h"
#include "brkpt.h"

//  Forward class declarations.
class DebugWindow;
class DebugInfo;
class DebugBreakPoints;
class DebugSource;
class DebugRegisters;
class DebugMemory;
class DebugCommand;
class DebugStatus;
class Bitmap;
class ColourTable;
class Canvas;
class GfxSet;
class Game;
class CPU;
class AppFile;


///////////////////////////////////////////////////////////////////////////////
//  Debugger Class Definition
///////////////////////////////////////////////////////////////////////////////
class Debugger : public RepBase
{
    //  Since the windows are so closely tied with the debugger we make them
    //  friends of the debugger.
    friend class DebugWindow;
    friend class DebugInfo;
    friend class DebugBreakPoints;
    friend class DebugSource;
    friend class DebugRegisters;
    friend class DebugMemory;
    friend class DebugCommand;
    friend class DebugStatus;

    public:

        //  The possible reasons for leaving the debugger execution.
        typedef enum
        {
            LEAVE_NO,
            LEAVE_SHORT,
            LEAVE_LONG
        }
        Leave;
 
        //  Deletion.
        virtual ~Debugger ( );

        //  Base overrides.
        virtual const KString& getClassName ( ) const;

        //  Singleton Access.
        static Debugger& s_instance ( );

        //  Interface.
        void clear       ( Game* pGame );
        void interrupt   ( );
        void check       ( );
        void checkRead   ( const DWord dwAddr );
        void checkWrite  ( const DWord dwAddr, const Byte bValue );

    protected:

        //  The various debugger windows.  They should be placed in the
        //  order that you would like them traversed.
        typedef enum
        {
            WIN_INFO,
            WIN_BREAK_POINTS,
            WIN_REGISTERS,
            WIN_SOURCE,
            WIN_MEMORY,
            WIN_COMMAND,
            WIN_STATUS,

            WIN_COUNT
        }
        DbgWindow;
            
        //  Creation is protected since this is a singleton class.
        Debugger ( const KString& iName );

        //  Debugger execution.
        void execute ( );

        //  Utility.
        void setCurWindow      ( DbgWindow eWindow );
        void setColours        ( );
        void checkWindowSwitch ( );


        //  Member Data.
        DbgWindow                       m_eWindow;
        DebugWindow*                    m_pDebugWindow; 
        DebugInfo*                      m_pDebugInfo;
        DebugBreakPoints*               m_pDebugBreakPoints;
        DebugSource*                    m_pDebugSource;
        DebugRegisters*                 m_pDebugRegisters;
        DebugMemory*                    m_pDebugMemory;
        DebugCommand*                   m_pDebugCommand;
        DebugStatus*                    m_pDebugStatus;
        Canvas*                         m_pCanvas; 
        Byte                            m_bRedrawScreen;
        Byte                            m_bRedrawWindow;
        Bitmap*                         m_pScreenBitmap;
        GfxSet*                         m_pFont;
        ColourTable*                    m_pNormalColTable;
        ColourTable*                    m_pHilightColTable;
        ColourTable*                    m_pSelectColTable;
        KPtrList<CPU>                   m_cpuList;
        Byte                            m_bInterrupt;
        Game*                           m_pGame; 
        CPU*                            m_pCPU;
        KPtrList<BreakPoint>            m_bpList;
        DWord                           m_dwBP; 
        DWord                           m_dwSrcAddr;
        DWord                           m_dwRegister;
        DWord                           m_dwMemAddr;
        AppFile*                        m_pTraceFile;
        CPU*                            m_pTraceCPU;
        BreakPoint::Type                m_eTrace;
        DWord                           m_dwTraceStart;
        DWord                           m_dwTraceEnd;
        char                            m_strStatus[ 80 ];

        //  Static Member Data.
        static const Byte               sm_bColourBG;
        static const Byte               sm_bColourHFG;
        static const Byte               sm_bColourNFG;
        static const Byte               sm_bColourSFG;

    private:

};

///////////////////////////////////////////////////////////////////////////////
//  Debugger Inline Functions.
///////////////////////////////////////////////////////////////////////////////
inline
void
Debugger::interrupt(
)
{
    m_bInterrupt = TRUE;
}

#endif

#endif
